/**
 * vamas - Vamas file format parser
 * @version v0.3.0
 * @link https://github.com/cheminfo/vamas#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.Vamas = {}));
})(this, (function (exports) { 'use strict';

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */
    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }
      var i = 0;
      var len = buf.length;
      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        }
        // UTF8-2 = %xC2-DF UTF8-tail
        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        }
        // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail
        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )
        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )
        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }
        return false;
      }
      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */
    function ensureString(blob, options = {}) {
      if (typeof blob === 'string') {
        return blob;
      }
      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }
      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }
    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }
        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      }
      //@ts-expect-error an ArrayBuffer is also ok
      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    function parse(blob) {
      const text = ensureString(blob);
      const lines = text.split(/\r?\n/);
      let pointer = 0;
      let parsed = {
        header: {},
        blocks: [],
        info: {}
      };
      pointer = parseHeader(lines, parsed, pointer);
      for (let i = 0; i < parsed.info.nbBlocks; i++) {
        pointer = parseBlock(lines, parsed, pointer);
      }
      return parsed;
    }
    function parseHeader(lines, parsed, pointer) {
      const {
        header,
        info
      } = parsed;
      header['format identifier'] = lines[pointer++];
      header['institution identifier'] = lines[pointer++];
      header['instrument model identifier'] = lines[pointer++];
      header['operator identifier'] = lines[pointer++];
      header['experiment identifier'] = lines[pointer++];
      info.nbComments = Number(lines[pointer++]);
      header['number of lines in comment'] = info.nbComments;
      const comments = [];
      for (let i = 0; i < info.nbComments; i++) {
        comments.push(lines[pointer++]);
      }
      header['comment line'] = comments.join('\n');
      header['experiment mode'] = lines[pointer++];
      header['scan mode'] = lines[pointer++];
      if (['MAP', 'MAPD', 'NORM', 'SDP'].includes(header['experiment mode'])) {
        header['number of spectral regions'] = Number(lines[pointer++]);
      }
      if (['MAP', 'MAPD'].includes(header['experiment mode'])) {
        header['number of analysis positions'] = Number(lines[pointer++]);
        header['number of discrete x coordinates available in full map'] = Number(lines[pointer++]);
        header['number of discrete y coordinates available in full map'] = Number(lines[pointer++]);
      }
      info.nbExperimentVariables = Number(lines[pointer++]);
      const experimentVariables = [];
      header['number of experimental variables'] = info.nbExperimentVariables;
      for (let i = 0; i < info.nbExperimentVariables; i++) {
        experimentVariables.push({
          label: lines[pointer++],
          unit: lines[pointer++]
        });
      }
      header.experimentVariables = experimentVariables;

      /*
        If the values of any of the block parameters are the same in all of the
        blocks their values may be sent in the first block and then omitted
        from all subsequent blocks.
        - n > 0 : the parameters listed are to be included
        - n < 0 : the parameters listed are to be excluded
        - n = 0 : all parameters are to be given in all blocks
        A complete block contains 40 parts.
        */
      info.nbEntriesInclusionExclusion = Number(lines[pointer++]);
      header['number of entries in parameter inclusion or exclusion list'] = info.nbEntriesInclusionExclusion;
      info.blockParametersincludes = new Array(40).fill(info.nbEntriesInclusionExclusion <= 0);
      for (let i = 0; i < Math.abs(info.nbEntriesInclusionExclusion); i++) {
        info.blockParametersincludes[Number(lines[pointer++]) + 1] = info.nbEntriesInclusionExclusion > 0;
      }
      header['number of manually entered items in block'] = Number(lines[pointer++]);
      info.nbFutureUpgradeExperimentEntries = Number(lines[pointer++]);
      header['number of future upgrade experiment entries'] = info.nbFutureUpgradeExperimentEntries;
      const futureUpgradeExperimentEntries = [];
      for (let i = 0; i < info.nbFutureUpgradeExperimentEntries; i++) {
        futureUpgradeExperimentEntries.push({
          label: lines[pointer++],
          unit: lines[pointer++]
        });
      }
      header.futureUpgradeExperimentEntries = futureUpgradeExperimentEntries;
      if (info.nbFutureUpgradeExperimentEntries !== 0) {
        throw new Error('unsupported future upgrade experiment entries');
      }
      header['number of future upgrade block entries'] = Number(lines[pointer++]);
      if (header['number of future upgrade block entries'] !== 0) {
        throw new Error('unsupported future upgrade block entries');
      }
      info.nbBlocks = Number(lines[pointer++]);
      header['number of blocks'] = info.nbBlocks;
      return pointer;
    }
    function parseBlock(lines, parsed, pointer) {
      const {
        blocks,
        header,
        info
      } = parsed;
      const firstBlock = blocks[0];
      const includes = blocks.length === 0 ? new Array(40).fill(true) : info.blockParametersincludes;
      const block = {};
      block['block identifier'] = lines[pointer++];
      block['sample identifier'] = lines[pointer++];
      block['year in full'] = includes[0] ? Number(lines[pointer++]) : firstBlock['year in full'];
      block.month = includes[1] ? Number(lines[pointer++]) : firstBlock.month;
      block['day of month'] = includes[2] ? Number(lines[pointer++]) : firstBlock['day of month'];
      block.hours = includes[3] ? Number(lines[pointer++]) : firstBlock.hours;
      block.minutes = includes[4] ? Number(lines[pointer++]) : firstBlock.minutes;
      block.seconds = includes[5] ? Number(lines[pointer++]) : firstBlock.seconds;
      block['number of hours in advance of Greenwich Mean Time'] = includes[6] ? Number(lines[pointer++]) : firstBlock['number of hours in advance of Greenwich Mean Time'];
      if (includes[7]) {
        const nbComments = Number(lines[pointer++]);
        block['number of lines in block comment'] = nbComments;
        const comments = [];
        for (let i = 0; i < nbComments; i++) {
          comments.push(lines[pointer++]);
        }
        block.blockComment = comments.join('\n');
      } else {
        block['number of lines in block comment'] = firstBlock['number of lines in block comment'];
        block.blockComment = firstBlock.blockComment;
      }
      block.technique = includes[8] ? lines[pointer++] : firstBlock.technique;
      if (['MAP', 'MAPDP'].includes(header['experiment mode'])) {
        block['x coordinate'] = includes[9] ? Number(lines[pointer++]) : firstBlock['x coordinate'];
        block['y coordinate'] = includes[9] ? Number(lines[pointer++]) : firstBlock['y coordinate'];
      }
      if (includes[10]) {
        let values = [];
        for (let i = 0; i < header.experimentVariables.length; i++) {
          values.push(lines[pointer++]);
        }
        block['value of experimental variable'] = values;
      } else {
        block['value of experimental variable'] = firstBlock['value of experimental variable'];
      }
      block['analysis source label'] = includes[11] ? lines[pointer++] : firstBlock['analysis source label'];
      if (['MAPDP', 'MAPSVDP', 'SDP', 'SDPSV'].includes(header['experiment mode']) || ['SNMS energy spec', 'FABMS', 'FABMS energy spec', 'ISS', 'SIMS', 'SIMS energy spec', 'SNMS'].includes(block.technique)) {
        block['sputtering ion or atom atomic number'] = includes[12] ? Number(lines[pointer++]) : firstBlock['sputtering ion or atom atomic number'];
        block['number of atoms in sputtering ion or atom particle'] = includes[12] ? Number(lines[pointer++]) : firstBlock['number of atoms in sputtering ion or atom particle'];
        block['sputtering ion or atom charge sign and number'] = includes[12] ? lines[pointer++] : firstBlock['sputtering ion or atom charge sign and number'];
      }
      block['analysis source characteristic energy'] = includes[13] ? Number(lines[pointer++]) : firstBlock['analysis source characteristic energy'];
      block['analysis source strength'] = includes[14] ? Number(lines[pointer++]) : firstBlock['analysis source strength'];
      block['analysis source beam width x'] = includes[15] ? Number(lines[pointer++]) : firstBlock['analysis source beam width x'];
      block['analysis source beam width y'] = includes[15] ? Number(lines[pointer++]) : firstBlock['analysis source beam width y'];
      if (['MAP', 'MAPDP', 'MAPSV', 'MAPSVDP', 'SEM'].includes(header['experiment mode'])) {
        block['field of view x'] = includes[16] ? Number(lines[pointer++]) : firstBlock['field of view x'];
        block['field of view y'] = includes[16] ? Number(lines[pointer++]) : firstBlock['field of view y'];
      }
      if (['SEM', 'MAPSV', 'MAPSVDP'].includes(header['experiment mode'])) {
        block['first linescan start x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan start x coordinate'];
        block['first linescan start y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan start y coordinate'];
        block['first linescan finish x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan finish x coordinate'];
        block['first linescan finish y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan finish y coordinate'];
        block['last linescan start x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan last x coordinate'];
        block['last linescan start y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan last y coordinate'];
      }
      block['analysis source polar angle of incidence'] = includes[18] ? Number(lines[pointer++]) : firstBlock['analysis source polar angle of incidence'];
      block['analysis source azimuth'] = includes[19] ? Number(lines[pointer++]) : firstBlock['analysis source azimuth'];
      block['analyser mode'] = includes[20] ? lines[pointer++] : firstBlock['analyser mode'];
      block['analyser pass energy or retard ratio or mass resolution'] = includes[21] ? Number(lines[pointer++]) : firstBlock['analyser pass energy or retard ratio or mass resolution'];
      if (block.technique === 'AES diff') {
        block['differential width'] = includes[22] ? lines[pointer++] : firstBlock['differential width'];
      }
      block['magnification of analyser transfer lens'] = includes[23] ? Number(lines[pointer++]) : firstBlock['magnification of analyser transfer lens'];
      block['analyser work function or acceptance energy of atom or ion'] = includes[24] ? Number(lines[pointer++]) : firstBlock['analyser work function or acceptance energy of atom or ion'];
      block['target bias'] = includes[25] ? Number(lines[pointer++]) : firstBlock['target bias'];
      block['analysis width x'] = includes[26] ? Number(lines[pointer++]) : firstBlock['analysis width x'];
      block['analysis width y'] = includes[26] ? Number(lines[pointer++]) : firstBlock['analysis width y'];
      block['analyser axis take off polar angle'] = includes[27] ? Number(lines[pointer++]) : firstBlock['analyser axis take off polar angle'];
      block['analyser axis take off azimuth'] = includes[27] ? Number(lines[pointer++]) : firstBlock['analyser axis take off azimuth'];
      block['species label'] = includes[28] ? lines[pointer++] : firstBlock['species label'];
      block['transition or charge state label'] = includes[29] ? lines[pointer++] : firstBlock['transition or charge state label'];
      block['charge of detected particle'] = includes[29] ? Number(lines[pointer++]) : firstBlock['charge of detected particle'];
      if (header['scan mode'] !== 'REGULAR') {
        throw new Error('Only REGULAR scans are supported');
      }
      block['abscissa label'] = includes[30] ? lines[pointer++] : firstBlock['abscissa label'];
      block['abscissa units'] = includes[30] ? lines[pointer++] : firstBlock['abscissa units'];
      block['abscissa start'] = includes[30] ? Number(lines[pointer++]) : firstBlock['abscissa start'];
      block['abscissa increment'] = includes[30] ? Number(lines[pointer++]) : firstBlock['abscissa increment'];
      if (includes[31]) {
        const nbCorrespondingVariables = Number(lines[pointer++]);
        block['number of corresponding variables'] = nbCorrespondingVariables;
        const correspondingVariables = [];
        for (let i = 0; i < nbCorrespondingVariables; i++) {
          correspondingVariables.push({
            label: lines[pointer++],
            unit: lines[pointer++],
            array: []
          });
        }
        block.correspondingVariables = correspondingVariables;
      } else {
        block['number of corresponding variables'] = firstBlock['number of corresponding variables'];
        block.correspondingVariables = structuredClone(firstBlock.correspondingVariables);
        block.correspondingVariables.array = [];
      }
      block['signal mode'] = includes[32] ? lines[pointer++] : firstBlock['signal mode'];
      block['signal collection time'] = includes[33] ? Number(lines[pointer++]) : firstBlock['signal collection time'];
      block['number of scans to compile this block'] = includes[34] ? Number(lines[pointer++]) : firstBlock['number of scans to compile this block'];
      block['signal time correction'] = includes[35] ? Number(lines[pointer++]) : firstBlock['signal time correction'];
      if (['MAPDP', 'MAPSVDP', 'SDP', 'SDPSV'].includes(header['experiment mode']) && ['AES diff', 'AES dir', 'EDX', 'ELS', 'UPS', 'XPS', 'XRF'].includes(block.technique)) {
        block['sputtering source energy'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source energy'];
        block['sputtering source beam current'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source beam current'];
        block['sputtering source width x'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source width x'];
        block['sputtering source width y'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source width y'];
        block['sputtering source polar angle of incidence'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source polar angle of incidence'];
        block['sputtering source azimuth'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source azimuth'];
        block['sputtering mode'] = includes[36] ? lines[pointer++] : firstBlock['sputtering mode'];
      }
      block['sample normal polar angle of tilt'] = includes[37] ? Number(lines[pointer++]) : firstBlock['sample normal polar angle of tilt'];
      block['sample normal tilt azimuth'] = includes[37] ? Number(lines[pointer++]) : firstBlock['sample normal tilt azimuth'];
      block['sample rotation angle'] = includes[38] ? Number(lines[pointer++]) : firstBlock['sample rotation angle'];
      if (includes[39]) {
        const nbAdditionalNumericalParameters = Number(lines[pointer++]);
        block['number of additional numerical parameters'] = nbAdditionalNumericalParameters;
        const additionalNumericalParameters = [];
        for (let i = 0; i < nbAdditionalNumericalParameters; i++) {
          additionalNumericalParameters.push({
            label: lines[pointer++],
            unit: lines[pointer++],
            value: lines[pointer++]
          });
        }
        block.additionalNumericalParameters = additionalNumericalParameters;
      } else {
        block['number of additional numerical parameters'] = firstBlock['number of additional numerical parameters'];
        block.additionalNumericalParameters = firstBlock.additionalNumericalParameters;
      }
      block.nbOrdinateValues = Number(lines[pointer++]);
      for (let correspondingVariable of block.correspondingVariables) {
        correspondingVariable.minimumOrdinateValue = Number(lines[pointer++]);
        correspondingVariable.maximumOrdinateValue = Number(lines[pointer++]);
      }
      for (let i = 0; i < block.nbOrdinateValues / block.correspondingVariables.length; i++) {
        for (let correspondingVariable of block.correspondingVariables) {
          correspondingVariable.array.push(Number(lines[pointer++]));
        }
      }
      parsed.blocks.push(block);
      return pointer;
    }

    /**
     * Referencing: (energy calibration, typically performed on the binding energy scale)
     * " Calib M = 0 A = 3.111 ADD"
     * means that the block should be shifted in binding energy by adding the difference A - M = 3.111 eV. M is not necessarily 0.
     * If the line contains BE, it means that the operation should be done on the BE scale,
     * alternatively, simply change the sign of the difference, here it would be -3,111, to be added to the data in the binding energy scale.
     * @param {object} calibrations
     * @param {string} line
     */

    function appendCalibration(calibrations, line) {
      let calibration = {};
      // Calib M = 281.1700 A = 284.8 BE ADD
      // calibration may be present on many lines

      let fields = line.match(/Calib M = (?<measured>.*) A = (?<referenced>[^ ]*) (?<rest>.*)/);
      if (!fields) {
        throw new Error(`appendCalibration fails on: ${line}`);
      }
      if (!fields.groups.rest.includes('ADD')) {
        throw new Error(`appendCalibration fails because line does not contain ADD: ${line}`);
      }
      calibration.measured = Number(fields.groups.measured);
      calibration.referenced = Number(fields.groups.referenced);
      if (fields.groups.rest.includes('BE')) {
        calibration.bindingEnergyShift = calibration.referenced - calibration.measured;
        calibration.kind = 'bindingEnergy';
      } else {
        calibration.bindingEnergyShift = calibration.measured - calibration.referenced;
        calibration.kind = 'kineticEnergy';
      }
      calibration.kineticEnergyShift = -calibration.bindingEnergyShift;
      calibrations.push(calibration);
    }

    /**
     * Fitting / each 'component' is a fitted lineshape, and can be from a different element while on a given orbital, for instance Ruthenium has peaks in the C1s region. The component will often have overlapping lineshapes
     *
     * "CASA comp (*C 1s compA*) (*GL(30)*) Area 2447.211 1e-020 223027.03 -1 1 MFWHM 0.72109706 0.21 5.25 -1 1 Position 1203.2974 1198.6232 1207.3108 -1 1 RSF 0.278 MASS 12.011 INDEX -1 (*C 1s*) CONST (**) UNCORRECTEDRSF 0.278"
     * This relates to the fittings (comments on the constraints later on):
     *
     * componentID: C 1s compA
     * lineshape: GL(30)
     * area: 2447.211
     * areaConstraint lb: 1e-020
     * areaConstraint ub: 223027.03
     * fwhm: 0.72109706
     * fwhmConstraint lb: 0.21
     * fwhmConstraint ub: 5.25
     * position: 1203.2974 // in kinetic energy eV
     * positionConstraint lb: 1198.6232 // in kinetic energy eV
     * positionConstraint ub: 1207.3108 // in kinetic energy eV
     * relativeSensitivityFactor: 0.278
     * mass: 12.011
    index: -1	// index allows to group different components to plot their manifold
    uncorrectedRSF: 0.278
     
    Constraints: the constraints can either be absolute, in which case the value to constraint is directly followed by the constaints, e.g.:
    Position 1203.4193 1202.686 1207.3108
     
    or relative to another component, with this weird notation:
    "Position 1202.261 0 0 2 -1.05017"
    here the position is separated from the constaints by "0 0", then "2" indicates the component to which is is constrained, here the 2nd component (third line of the comp, i.e, starting from 0). And the last number "-1.05017" is the shift (in the kinetic energy scale)
     * @param components
     * @param line
     */

    function appendComponent(components, line) {
      // CASA comp (*Mo 3d MoS2 2H*) (*LA(1.53,243)*) Area 230.36971 1e-020 2327991 -1 1 MFWHM 0.88528218 0.2 2 -1 1 Position 1257.22 1257.02 1257.22 -1 1 RSF 10.804667 MASS 95.9219 INDEX -1 (*Mo 3d*) CONST (**) UNCORRECTEDRSF 9.5
      let component = {};
      const componentRegex = new RegExp([/CASA comp /, /\(\*(?<componentID>.*)\*\) /, /\((?<shape>[^ ]*)\) /, /(?<area>Area .*)/, /(?<fwhm>MFWHM .*)/, /(?<position>Position .*) /, /(?<rsf>RSF .*) /, /(?<mass>MASS .*) /, /(?<index>INDEX .*) /, /(?<const>CONST .*) /, /(?<uncorrectedRSF>UNCORRECTEDRSF.*)/].map(r => r.source).join(''));
      let fields = line.match(componentRegex);
      if (!fields) {
        throw new Error(`appendComponent fails on: ${line}`);
      }
      component = {
        componentID: fields.groups.componentID,
        shape: parseShape(fields.groups.shape),
        area: parseMultiplicationConstrain(fields.groups.area),
        fwhm: parseMultiplicationConstrain(fields.groups.fwhm),
        position: parseKEShiftConstrain(fields.groups.position),
        relativeSensitivityFactor: parseRSF(fields.groups.rsf),
        atomicMass: parseMass(fields.groups.mass),
        groupIndex: parseIndex(fields.groups.index),
        uncorrectedRSF: parseUncorrectedRSF(fields.groups.uncorrectedRSF)
      };
      components.push(component);
    }
    function parseShape(value) {
      let parts = value.replaceAll(/[()*,]/g, ' ').trim().split(/ +/);
      let options = {};
      let kind;
      switch (parts[0]) {
        case 'LA':
          // http://www.casaxps.com/help_manual/manual_updates/LA_Lineshape.pdf
          kind = 'lorentzianAsymmetric';
          options.alpha = Number(parts[1]);
          options.beta = Number(parts[2]);
          if (parts[3] !== undefined) {
            options.m = Number(parts[3]);
          }
          break;
        case 'GL':
          // http://www.casaxps.com/help_manual/line_shapes.htm
          kind = 'gaussianLorentzianProduct';
          options.mixingRatio = Number(parts[1]);
          break;
        case 'SGL':
          // https://docs.mantidproject.org/nightly/fitting/fitfunctions/PseudoVoigt.html
          kind = 'pseudoVoigt';
          options.mu = Number(parts[1]);
          break;
        case 'LF':
          kind = 'gaussianLorentzianSum';
          break;
        case 'LS':
          kind = 'gaussianLorentzianSum';
          break;
        default:
          throw new Error(`appendComponent: unknown shape: ${parts[0]}`);
      }
      return {
        kind,
        options
      };
    }
    function parseMultiplicationConstrain(value) {
      const parts = value.split(' ');
      const result = {
        value: Number(parts[1]),
        lowerBound: Number(parts[2]),
        upperBound: Number(parts[3])
      };
      const constrain = {
        linkedComponent: Number(parts[4]),
        // line number, no constrain if -1
        factor: Number(parts[5]) // multiplication factor of the value in the linked line
      };
      if (constrain.linkedComponent >= 0) {
        result.constrain = constrain;
      }
      return result;
    }
    function parseKEShiftConstrain(value) {
      const parts = value.split(' ');
      const result = {
        value: Number(parts[1]),
        lowerBound: Number(parts[2]),
        upperBound: Number(parts[3])
      };
      const constrain = {
        linkedComponent: Number(parts[4]),
        // line number, no constrain if -1
        shift: Number(parts[5]) // shift of the value in the linked line
      };
      if (constrain.linkedComponent >= 0) {
        result.constrain = constrain;
      }
      return result;
    }
    function parseRSF(value) {
      let parts = value.split(' ');
      return Number(parts[1]);
    }
    function parseMass(value) {
      let parts = value.split(' ');
      return Number(parts[1]);
    }
    function parseIndex(value) {
      let parts = value.split(' ');
      return Number(parts[1]);
    }
    function parseUncorrectedRSF(value) {
      let parts = value.split(' ');
      return Number(parts[1]);
    }

    /**
     * Regions are used to calculate the baseline.
     * There may be several overlapping regions.
     * We could just throw an error if it happens has this is likely just a bug from casaXPS to allow for this.
     * "CASA region (*C 1s*) (*Shirley*) 1202.686 1208.454 0.278 2 0 0 115.3918 -450 0 0 (*C 1s*) 12.011 0 0.278"
     *
     * THe last 3 numbers:  12.011 0 0.278, are the mass of the element (Carbon here), a separator
     * and the relative sensitivity factor of the element.
     *  Note that the relative sensitivity factor given within the background part ((*C 1s*) (*Shirley*) 1202.686 1208.454 0.278 2 0 0 115.3918 -450 0 0 (*C 1s*)), can be different than the last one.
     * @param regions
     * @param line
     */

    function appendRegion(regions, line) {
      // CASA region (*Mo 3d*) (*Shirley*) 1249.3343 1262.7065 10.804667 2 0 0 392.54541 -450 0 0 (*Mo 3d*) 95.9219 0 9.5
      let fields = line.match(/CASA region \(\*(?<regionID>.*)\*\) \(\*(?<backgroundType>.*)\*\) (?<backgroundOptions>.*) \(\*(?<regionBlockID>.*)\*\) (?<atomicMass>.*) (?<separator>.) (?<relativeSensitivityFactor>.*)/);
      if (!fields) {
        throw new Error(`appendRegion fails on: ${line}`);
      }
      let region = {
        regionID: fields.groups.regionID,
        block: {
          regionBlockID: fields.groups.regionBlockID,
          atomicMass: Number.parseFloat(fields.groups.atomicMass),
          relativeSensitivityFactor: Number.parseFloat(fields.groups.relativeSensitivityFactor)
        },
        background: {
          type: fields.groups.backgroundType,
          parameters: parseBackgroundParameters(fields.groups.backgroundType, fields.groups.backgroundOptions),
          rawParameters: fields.groups.backgroundOptions
        }
      };
      regions.push(region);
    }
    function parseBackgroundParameters(type, string) {
      const [kineticEnergyStart, kineticEnergyEnd, relativeSensitivityFactor, averageWidth, startOffset, endOffset, ...otherParameters] = string.split(/ +/).filter(field => field.trim() !== '').map(field => Number.parseFloat(field));

      // todo: strange we need to remove 1 from averageWidth
      return {
        kineticEnergyStart,
        kineticEnergyEnd,
        relativeSensitivityFactor,
        averageWidth: averageWidth - 1,
        startOffset,
        endOffset,
        crossSection: otherParameters
      };
    }

    function parseCASA(text) {
      const casa = {
        regions: [],
        components: [],
        calibrations: []
      };
      const lines = text.split(/\r?\n/);
      for (const line of lines) {
        if (line.startsWith('CASA comp ')) {
          appendComponent(casa.components, line);
        }
        if (line.startsWith('CASA region')) {
          appendRegion(casa.regions, line);
        }
        if (line.startsWith('Calib')) {
          appendCalibration(casa.calibrations, line);
        }
      }
      return casa;
    }

    /**
     * Instrument manufacturers store some key metadata in their own fields
     * in the vamas file. This does not follow the ISO specification.
     *
     * For this reason, there needs to be one parsing step that is specific to the manufacturer that extracts this data.
     *
     * Note that we only parse data that is not in ISO-specified block, in the case of Kratos, this is, for example, information about charge neutralization
     *
     * To avoid an additional for loop this function takes an object and a line and updates key/value pairs as appropriate
     * @param line
     */

    function splitTrim(line) {
      return line.replace(/\s/).split(':')[1];
    }
    function splitTrimValueUnit(line) {
      let intermediate = splitTrim(line);
      const unit = intermediate.match(/[A-Za-z]+/g)[0];
      const value = Number.parseFloat(intermediate.replace(unit, ''));
      return {
        value,
        unit
      };
    }
    function parseKratosMeta(text) {
      const meta = {};
      meta.chargeNeutraliser = {};
      meta.scanSettings = {};
      const lines = text.split(/\r?\n/);
      for (const line of lines) {
        if (line.startsWith('Charge Neutraliser :') && line.includes('On')) {
          meta.chargeNeutraliser.activated = true;
        }
        if (line.startsWith('Sweeps :')) {
          meta.scanSettings.numberSweeps = splitTrim(line);
        }
        if (line.startsWith('Dwell time :')) {
          meta.scanSettings.dwellTime = splitTrimValueUnit(line);
        }
        if (line.startsWith('Sweep time :')) {
          meta.scanSettings.sweepTime = splitTrimValueUnit(line);
        }
        if (line.startsWith('Charge Balance :')) {
          meta.chargeNeutraliser.chargeBalance = splitTrimValueUnit(line);
        }
      }
      return meta;
    }

    exports.parse = parse;
    exports.parseCASA = parseCASA;
    exports.parseKratosMeta = parseKratosMeta;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=vamas.js.map
