/**
 * xrd-analysis - xrd-analysis can convert output files from powder-xray diffraction into JCAMP-DX format and perform analysis (Scherrer equation, ...) on the diffractograms
 * @version v0.10.3
 * @link https://github.com/cheminfo/xrd-analysis#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.XRDAnalysis = {}));
}(this, (function (exports) { 'use strict';

  const toString$2 = Object.prototype.toString;
  function isAnyArray(object) {
    return toString$2.call(object).endsWith('Array]');
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function commonjsRequire (path) {
  	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
  }

  var medianQuickselect_min = {exports: {}};

  (function (module) {
    (function () {
      function a(d) {
        for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
          if (f <= e) return d[j];
          if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

          for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
            do h++; while (d[e] > d[h]);

            do i--; while (d[i] > d[e]);

            if (i < h) break;
            b(d, h, i);
          }

          b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
        }
      }

      var b = function b(d, e, f) {
        var _ref;

        return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
      },
          c = function c(d, e) {
        return ~~((d + e) / 2);
      };

      module.exports ? module.exports = a : window.median = a;
    })();
  })(medianQuickselect_min);

  var quickSelectMedian = medianQuickselect_min.exports;

  function median(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    return quickSelectMedian(input.slice());
  }

  /**
   * This function xAdd the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xAdd(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + array2[i];
      }
    }

    return array3;
  }

  /**
   * This function xMultiply the first array by the second array or a constant value to each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Float64Array}
   */

  function xMultiply(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Float64Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * array2[i];
      }
    }

    return array3;
  }

  /**
   * This function divide the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array<Number>|Number} array2
   * @return {Array}
   */

  function xDivide(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / array2[i];
      }
    }

    return array3;
  }

  /**
   * Returns the closest index of a `target` in an ordered array
   * @param {array<Number>} array
   * @param {number} target
   */
  function xFindClosestIndex(array, target) {
    let low = 0;
    let high = array.length - 1;
    let middle = 0;

    while (high - low > 1) {
      middle = low + (high - low >> 1);

      if (array[middle] < target) {
        low = middle;
      } else if (array[middle] > target) {
        high = middle;
      } else {
        return middle;
      }
    }

    if (low < array.length - 1) {
      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
        return low;
      } else {
        return low + 1;
      }
    } else {
      return low;
    }
  }

  /**
   * Returns an object with {fromIndex, toIndex} for a specific from / to
   * @param {array} x
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   */

  function xGetFromToIndex(x, options = {}) {
    let {
      fromIndex,
      toIndex,
      from,
      to
    } = options;

    if (fromIndex === undefined) {
      if (from !== undefined) {
        fromIndex = xFindClosestIndex(x, from);
      } else {
        fromIndex = 0;
      }
    }

    if (toIndex === undefined) {
      if (to !== undefined) {
        toIndex = xFindClosestIndex(x, to);
      } else {
        toIndex = x.length - 1;
      }
    }

    if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
    return {
      fromIndex,
      toIndex
    };
  }

  function _typeof(obj) {
    "@babel/helpers - typeof";

    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
      _typeof = function (obj) {
        return typeof obj;
      };
    } else {
      _typeof = function (obj) {
        return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
      };
    }

    return _typeof(obj);
  }
  /**
   * Fill an array with sequential numbers
   * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
   * @param {object} [options={}]
   * @param {number} [options.from=0] - first value in the array
   * @param {number} [options.to=10] - last value in the array
   * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
   * @param {number} [options.step] - if not provided calculated from size
   * @return {Array<number>}
   */


  function sequentialFill() {
    var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (_typeof(input) === 'object' && !isAnyArray(input)) {
      options = input;
      input = [];
    }

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    var _options = options,
        _options$from = _options.from,
        from = _options$from === void 0 ? 0 : _options$from,
        _options$to = _options.to,
        to = _options$to === void 0 ? 10 : _options$to,
        _options$size = _options.size,
        size = _options$size === void 0 ? input.length : _options$size,
        step = _options.step;

    if (size !== 0 && step) {
      throw new Error('step is defined by the array size');
    }

    if (!size) {
      if (step) {
        size = Math.floor((to - from) / step) + 1;
      } else {
        size = to - from + 1;
      }
    }

    if (!step && size) {
      step = (to - from) / (size - 1);
    }

    if (Array.isArray(input)) {
      // only works with normal array
      input.length = 0;

      for (var i = 0; i < size; i++) {
        input.push(from);
        from += step;
      }
    } else {
      if (input.length !== size) {
        throw new Error('sequentialFill typed array must have the correct length');
      }

      for (var _i = 0; _i < size; _i++) {
        input[_i] = from;
        from += step;
      }
    }

    return input;
  }

  /**
   * Returns true if x is monotone
   * @param {Array} array
   * @return {boolean}
   */
  function xIsMonotone(array) {
    if (array.length <= 2) {
      return true;
    }

    if (array[0] === array[1]) {
      // maybe a constant series
      for (let i = 1; i < array.length - 1; i++) {
        if (array[i] !== array[i + 1]) return false;
      }

      return true;
    }

    if (array[0] < array[array.length - 1]) {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] >= array[i + 1]) return false;
      }
    } else {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] <= array[i + 1]) return false;
      }
    }

    return true;
  }

  function sum(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += input[i];
    }

    return sumValue;
  }

  function mean(input) {
    return sum(input) / input.length;
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  /**
   * This function xSubtract the first array by the second array or a constant value from each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xSubtract(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - array2[i];
      }
    }

    return array3;
  }

  /**
   * Throw an error in no an object of x,y arrays
   * @param {DataXY} [data={}]
   */

  function xyCheck(data = {}) {
    if (!isAnyArray(data.x) || !isAnyArray(data.y)) {
      throw new Error('Data must be an object of x and y arrays');
    }

    if (data.x.length !== data.y.length) {
      throw new Error('The x and y arrays mush have the same length');
    }
  }

  /**
   * Filters x,y values to allow strictly growing values in x axis.
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {DataXY}
   */

  function xyEnsureGrowingX(data = {}) {
    xyCheck(data);
    const x = Array.from(data.x);
    const y = Array.from(data.y);
    let prevX = -Infinity;
    let ansX = [];
    let ansY = [];

    for (let index = 0; index < x.length; index++) {
      if (prevX < x[index]) {
        ansX.push(x[index]);
        ansY.push(y[index]);
        prevX = x[index];
      }
    }

    return {
      x: ansX,
      y: ansY
    };
  }

  /**
   * Filter out all the points for which x <= 0. Useful to display log scale data
   * @param {DataXY} [data={}]
   * @return {{x:[],y:[]}} An object with the filtered data
   */

  function xyFilterXPositive(data = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    const newX = [];
    const newY = [];

    for (let i = 0; i < x.length; i++) {
      if (x[i] > 0) {
        newX.push(x[i]);
        newY.push(y[i]);
      }
    }

    return {
      x: newX,
      y: newY
    };
  }

  /**
   * Calculate integration
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number} xyIntegration value on the specified range
   */

  function xyIntegration(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let currentxyIntegration = 0;

    for (let i = fromIndex; i < toIndex; i++) {
      currentxyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
    }

    return currentxyIntegration;
  }

  /**
   * Find the closest maximum going up hill
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @return {{x,y,xIndex}} An object with the x/y value
   */

  function xyMaxClosestYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      target,
      targetIndex
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        targetIndex = xFindClosestIndex(x, target);
      } else {
        targetIndex = 0;
      }
    }

    let previousIndex = Number.MIN_SAFE_INTEGER;
    let currentIndex = targetIndex;
    let xyMaxY = y[targetIndex];

    while (currentIndex !== previousIndex) {
      previousIndex = currentIndex;

      if (currentIndex > 0 && y[currentIndex - 1] > xyMaxY) {
        currentIndex--;
      } else if (currentIndex < x.length - 1 && y[currentIndex + 1] > xyMaxY) {
        currentIndex++;
      }

      xyMaxY = y[currentIndex];
    }

    return {
      x: x[currentIndex],
      y: y[currentIndex],
      index: currentIndex
    };
  }

  /**
   * Find the closest minimum going down hill
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @return {{x,y,xIndex}} An object with the x/y value
   */

  function xyMinClosestYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      target,
      targetIndex
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        targetIndex = xFindClosestIndex(x, target);
      } else {
        targetIndex = 0;
      }
    }

    let previousIndex = Number.MIN_SAFE_INTEGER;
    let currentIndex = targetIndex;
    let minY = y[targetIndex];

    while (currentIndex !== previousIndex) {
      previousIndex = currentIndex;

      if (currentIndex > 0 && y[currentIndex - 1] < minY) {
        currentIndex--;
      } else if (currentIndex < x.length - 1 && y[currentIndex + 1] < minY) {
        currentIndex++;
      }

      minY = y[currentIndex];
    }

    return {
      x: x[currentIndex],
      y: y[currentIndex],
      index: currentIndex
    };
  }

  const GAUSSIAN_EXP_FACTOR$1 = -4 * Math.LN2;
  const ROOT_PI_OVER_LN2$1 = Math.sqrt(Math.PI / Math.LN2);
  const ROOT_THREE$1 = Math.sqrt(3);
  const ROOT_2LN2$1 = Math.sqrt(2 * Math.LN2);
  const ROOT_2LN2_MINUS_ONE$1 = Math.sqrt(2 * Math.LN2) - 1;

  // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  // This code yields to a good approximation
  // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  function erfinv$1(x) {
    let a = 0.147;
    if (x === 0) return 0;
    let ln1MinusXSqrd = Math.log(1 - x * x);
    let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
    let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
    let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
    return secondSqrt * (x > 0 ? 1 : -1);
  }

  class Shape1D {}

  class Gaussian$1 extends Shape1D {
    constructor(options = {}) {
      super();
      const {
        fwhm = 500,
        sd,
        height
      } = options;
      this.fwhm = sd ? widthToFWHM$2(2 * sd) : fwhm;
      this.height = height === undefined ? Math.sqrt(-GAUSSIAN_EXP_FACTOR$1 / Math.PI) / this.fwhm : height;
    }

    fwhmToWidth(fwhm = this.fwhm) {
      return fwhmToWidth$2(fwhm);
    }

    widthToFWHM(width) {
      return widthToFWHM$2(width);
    }

    fct(x) {
      return fct$2(x, this.fwhm);
    }

    getArea() {
      return getArea$2({
        fwhm: this.fwhm,
        height: this.height
      });
    }

    getFactor(area) {
      return getFactor$2(area);
    }

    getData(options = {}) {
      const {
        length,
        factor
      } = options;
      return getData$2({
        fwhm: this.fwhm,
        height: this.height,
        factor,
        length
      });
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @returns - the y value of gaussian with the current parameters.
   */

  function fct$2(x, fwhm) {
    return Math.exp(GAUSSIAN_EXP_FACTOR$1 * Math.pow(x / fwhm, 2));
  }
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * for more information check the [mathworld page](https://mathworld.wolfram.com/GaussianFunction.html)
   * @returns fwhm
   */

  function widthToFWHM$2(width) {
    return width * ROOT_2LN2$1;
  }
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * for more information check the [mathworld page](https://mathworld.wolfram.com/GaussianFunction.html)
   * @param fwhm - Full Width at Half Maximum.
   * @returns width
   */

  function fwhmToWidth$2(fwhm) {
    return fwhm / ROOT_2LN2$1;
  }
  /**
   * Calculate the area of a specific shape.
   * @returns returns the area of the specific shape and parameters.
   */

  function getArea$2(options) {
    let {
      fwhm,
      sd,
      height = 1
    } = options;
    if (sd) fwhm = widthToFWHM$2(2 * sd);

    if (fwhm === undefined) {
      throw new Error('should pass fwhm or sd parameters');
    }

    return height * ROOT_PI_OVER_LN2$1 * fwhm / 2;
  }
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param [area=0.9999] Expected area to be covered.
   * @returns
   */

  function getFactor$2(area = 0.9999) {
    return Math.sqrt(2) * erfinv$1(area);
  }
  /**
   * Calculate intensity array of a gaussian shape.
   * @returns {Float64Array} Intensity values.
   */

  function getData$2(options = {}) {
    let {
      length,
      factor = getFactor$2(),
      fwhm = 500,
      sd,
      height
    } = options;
    if (sd) fwhm = widthToFWHM$2(2 * sd);

    if (!height) {
      height = Math.sqrt(-GAUSSIAN_EXP_FACTOR$1 / Math.PI) / fwhm;
    }

    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }

    const center = (length - 1) / 2;
    const data = new Float64Array(length);

    for (let i = 0; i <= center; i++) {
      data[i] = fct$2(i - center, fwhm) * height;
      data[length - 1 - i] = data[i];
    }

    return data;
  }

  class Lorentzian$1 extends Shape1D {
    constructor(options = {}) {
      super();
      const {
        fwhm = 500,
        height
      } = options;
      this.fwhm = fwhm;
      this.height = height === undefined ? 2 / Math.PI / fwhm : height;
    }

    fwhmToWidth(fwhm = this.fwhm) {
      return fwhmToWidth$1(fwhm);
    }

    widthToFWHM(width) {
      return widthToFWHM$1(width);
    }

    fct(x) {
      return fct$1(x, this.fwhm);
    }

    getArea() {
      return getArea$1({
        fwhm: this.fwhm,
        height: this.height
      });
    }

    getFactor(area) {
      return getFactor$1(area);
    }

    getData(options = {}) {
      const {
        length,
        factor
      } = options;
      return getData$1({
        fwhm: this.fwhm,
        height: this.height,
        factor,
        length
      });
    }

  }
  /**
   * Return a parameterized function of a lorentzian shape (see README for equation).
   * @param x - x value to calculate.
   * @param fwhm - full width half maximum
   * @returns - the y value of lorentzian with the current parameters.
   */

  function fct$1(x, fwhm) {
    return Math.pow(fwhm, 2) / (4 * Math.pow(x, 2) + Math.pow(fwhm, 2));
  }
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * for more information check the [mathworld page](https://mathworld.wolfram.com/LorentzianFunction.html)
   * @param width - Width between the inflection points
   * @returns fwhm
   */

  function widthToFWHM$1(width) {
    return width * ROOT_THREE$1;
  }
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * for more information check the [mathworld page](https://mathworld.wolfram.com/LorentzianFunction.html)
   * @param fwhm - Full Width at Half Maximum.
   * @returns width
   */

  function fwhmToWidth$1(fwhm) {
    return fwhm / ROOT_THREE$1;
  }
  /**
   * Calculate the area of a specific shape.
   * @returns returns the area of the specific shape and parameters.
   */

  function getArea$1(options) {
    const {
      fwhm,
      height = 1
    } = options;

    if (fwhm === undefined) {
      throw new Error('should pass fwhm or sd parameters');
    }

    return height * Math.PI * fwhm / 2;
  }
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param [area=0.9999] Expected area to be covered.
   * @returns
   */

  function getFactor$1(area = 0.9999) {
    return 2 * Math.tan(Math.PI * (area - 0.5));
  }
  /**
   * Calculate intensity array of a lorentzian shape.
   * @returns {Float64Array} y values
   */

  function getData$1(options = {}) {
    let {
      length,
      factor = getFactor$1(),
      fwhm = 500,
      height
    } = options;

    if (!height) {
      height = 2 / Math.PI / fwhm;
    }

    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }

    const center = (length - 1) / 2;
    const data = new Float64Array(length);

    for (let i = 0; i <= center; i++) {
      data[i] = fct$1(i - center, fwhm) * height;
      data[length - 1 - i] = data[i];
    }

    return data;
  }

  class PseudoVoigt$1 extends Shape1D {
    constructor(options = {}) {
      super();
      const {
        fwhm = 500,
        height,
        mu = 0.5
      } = options;
      this.mu = mu;
      this.fwhm = fwhm;
      this.height = height === undefined ? 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR$1 / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2) : height;
    }

    fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
      return fwhmToWidth(fwhm, mu);
    }

    widthToFWHM(width, mu = this.mu) {
      return widthToFWHM(width, mu);
    }

    fct(x) {
      return fct(x, this.fwhm, this.mu);
    }

    getArea() {
      return getArea({
        fwhm: this.fwhm,
        height: this.height,
        mu: this.mu
      });
    }

    getFactor(area) {
      return getFactor(area);
    }

    getData(options = {}) {
      const {
        length,
        factor
      } = options;
      return getData({
        fwhm: this.fwhm,
        height: this.height,
        mu: this.mu,
        factor,
        length
      });
    }

  }
  /**
   * Return a parameterized function of a pseudo voigt shape (see README for equation).
   * @param x - x value to calculate.
   * @param fwhm - full width half maximum
   * @returns - the y value of pseudo voigt with the current parameters.
   */

  function fct(x, fwhm, mu) {
    return (1 - mu) * fct$1(x, fwhm) + mu * fct$2(x, fwhm);
  }
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * @param width - Width between the inflection points
   * @param [mu=0.5] Ratio of gaussian contribution in the shape
   * @returns fwhm
   */

  function widthToFWHM(width, mu = 0.5) {
    return width * (mu * ROOT_2LN2_MINUS_ONE$1 + 1);
  }
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * @param fwhm - Full Width at Half Maximum.
   * @param [mu=0.5] Ratio of gaussian contribution in the shape
   * @returns width
   */

  function fwhmToWidth(fwhm, mu = 0.5) {
    return fwhm / (mu * ROOT_2LN2_MINUS_ONE$1 + 1);
  }
  /**
   * Calculate the area of a specific shape.
   * @returns returns the area of the specific shape and parameters.
   */

  function getArea(options) {
    const {
      fwhm,
      height = 1,
      mu = 0.5
    } = options;

    if (fwhm === undefined) {
      throw new Error('should pass fwhm or sd parameters');
    }

    return fwhm * height * (mu * ROOT_PI_OVER_LN2$1 + (1 - mu) * Math.PI) / 2;
  }
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param [area=0.9999] Expected area to be covered.
   * @returns
   */

  function getFactor(area = 0.9999, mu = 0.5) {
    return mu < 1 ? getFactor$1(area) : getFactor$2(area);
  }
  /**
   * Calculate intensity array of a pseudo voigt shape.
   * @returns {Float64Array} y values
   */

  function getData(options = {}) {
    let {
      length,
      factor = getFactor(),
      fwhm = 500,
      height,
      mu = 0.5
    } = options;

    if (!height) {
      height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR$1 / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
    }

    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }

    const center = (length - 1) / 2;
    const data = new Float64Array(length);

    for (let i = 0; i <= center; i++) {
      data[i] = fct(i - center, fwhm, mu) * height;
      data[length - 1 - i] = data[i];
    }

    return data;
  }

  /**
   * Generate a instance of a specific kind of shape.
   */

  function getShape1D(kind, shapeOptions = {}) {
    switch (kind) {
      case 'gaussian':
        return new Gaussian$1(shapeOptions);

      case 'lorentzian':
        return new Lorentzian$1(shapeOptions);

      case 'pseudoVoigt':
        return new PseudoVoigt$1(shapeOptions);

      default:
        {
          const unHandled = kind; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

          throw Error(`Unknown distribution ${unHandled}`);
        }
    }
  }

  /**
   * Apply Savitzky Golay algorithm
   * @param {array} [ys] Array of y values
   * @param {array|number} [xs] Array of X or deltaX
   * @param {object} [options={}]
   * @param {number} [options.windowSize=9]
   * @param {number} [options.derivative=0]
   * @param {number} [options.polynomial=3]
   * @return {array} Array containing the new ys (same length)
   */
  function SavitzkyGolay(ys, xs, options = {}) {
    let {
      windowSize = 9,
      derivative = 0,
      polynomial = 3
    } = options;

    if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
      throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
    }

    if (windowSize > ys.length) {
      throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
    }

    if (derivative < 0 || !Number.isInteger(derivative)) {
      throw new RangeError('Derivative should be a positive integer');
    }

    if (polynomial < 1 || !Number.isInteger(polynomial)) {
      throw new RangeError('Polynomial should be a positive integer');
    }

    if (polynomial >= 6) {
      // eslint-disable-next-line no-console
      console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
    }

    let half = Math.floor(windowSize / 2);
    let np = ys.length;
    let ans = new Array(np);
    let weights = fullWeights(windowSize, polynomial, derivative);
    let hs = 0;
    let constantH = true;

    if (Array.isArray(xs)) {
      constantH = false;
    } else {
      hs = Math.pow(xs, derivative);
    } //For the borders


    for (let i = 0; i < half; i++) {
      let wg1 = weights[half - i - 1];
      let wg2 = weights[half + i + 1];
      let d1 = 0;
      let d2 = 0;

      for (let l = 0; l < windowSize; l++) {
        d1 += wg1[l] * ys[l];
        d2 += wg2[l] * ys[np - windowSize + l];
      }

      if (constantH) {
        ans[half - i - 1] = d1 / hs;
        ans[np - half + i] = d2 / hs;
      } else {
        hs = getHs(xs, half - i - 1, half, derivative);
        ans[half - i - 1] = d1 / hs;
        hs = getHs(xs, np - half + i, half, derivative);
        ans[np - half + i] = d2 / hs;
      }
    } //For the internal points


    let wg = weights[half];

    for (let i = windowSize; i <= np; i++) {
      let d = 0;

      for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];

      if (!constantH) hs = getHs(xs, i - half - 1, half, derivative);
      ans[i - half - 1] = d / hs;
    }

    return ans;
  }

  function getHs(h, center, half, derivative) {
    let hs = 0;
    let count = 0;

    for (let i = center - half; i < center + half; i++) {
      if (i >= 0 && i < h.length - 1) {
        hs += h[i + 1] - h[i];
        count++;
      }
    }

    return Math.pow(hs / count, derivative);
  }

  function GramPoly(i, m, k, s) {
    let Grampoly = 0;

    if (k > 0) {
      Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * GramPoly(i, m, k - 1, s) + s * GramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * GramPoly(i, m, k - 2, s);
    } else {
      if (k === 0 && s === 0) {
        Grampoly = 1;
      } else {
        Grampoly = 0;
      }
    }

    return Grampoly;
  }

  function GenFact(a, b) {
    let gf = 1;

    if (a >= b) {
      for (let j = a - b + 1; j <= a; j++) {
        gf *= j;
      }
    }

    return gf;
  }

  function Weight(i, t, m, n, s) {
    let sum = 0;

    for (let k = 0; k <= n; k++) {
      //console.log(k);
      sum += (2 * k + 1) * (GenFact(2 * m, k) / GenFact(2 * m + k + 1, k + 1)) * GramPoly(i, m, k, 0) * GramPoly(t, m, k, s);
    }

    return sum;
  }
  /**
   *
   * @param m  Number of points
   * @param n  Polynomial grade
   * @param s  Derivative
   */


  function fullWeights(m, n, s) {
    let weights = new Array(m);
    let np = Math.floor(m / 2);

    for (let t = -np; t <= np; t++) {
      weights[t + np] = new Array(m);

      for (let j = -np; j <= np; j++) {
        weights[t + np][j + np] = Weight(j, t, np, n, s);
      }
    }

    return weights;
  }
  /*function entropy(data,h,options){
      var trend = SavitzkyGolay(data,h,trendOptions);
      var copy = new Array(data.length);
      var sum = 0;
      var max = 0;
      for(var i=0;i<data.length;i++){
          copy[i] = data[i]-trend[i];
      }

      sum/=data.length;
      console.log(sum+" "+max);
      console.log(stat.array.standardDeviation(copy));
      console.log(Math.abs(stat.array.mean(copy))/stat.array.standardDeviation(copy));
      return sum;

  }



  function guessWindowSize(data, h){
      console.log("entropy "+entropy(data,h,trendOptions));
      return 5;
  }
  */

  /**
   * Global spectra deconvolution
   * @param {object} data - Object data with x and y arrays
   * @param {Array<number>} [data.x] - Independent variable
   * @param {Array<number>} [data.y] - Dependent variable
   * @param {object} [options={}] - Options object
   * @param {object} [options.shape={}] - Object that specified the kind of shape to calculate the FWHM instead of width between inflection points. see https://mljs.github.io/peak-shape-generator/#inflectionpointswidthtofwhm
   * @param {object} [options.shape.kind='gaussian']
   * @param {object} [options.shape.options={}]
   * @param {object} [options.sgOptions] - Options object for Savitzky-Golay filter. See https://github.com/mljs/savitzky-golay-generalized#options
   * @param {number} [options.sgOptions.windowSize = 9] - points to use in the approximations
   * @param {number} [options.sgOptions.polynomial = 3] - degree of the polynomial to use in the approximations
   * @param {number} [options.minMaxRatio = 0.00025] - Threshold to determine if a given peak should be considered as a noise
   * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
   * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.
   * @param {number} [options.noiseLevel = 0] - Noise threshold in spectrum units
   * @param {boolean} [options.maxCriteria = true] - Peaks are local maximum(true) or minimum(false)
   * @param {boolean} [options.smoothY = true] - Select the peak intensities from a smoothed version of the independent variables
   * @param {boolean} [options.realTopDetection = false] - Use a quadratic optimizations with the peak and its 3 closest neighbors
   * to determine the true x,y values of the peak?
   * @param {number} [options.heightFactor = 0] - Factor to multiply the calculated height (usually 2)
   * @param {number} [options.derivativeThreshold = -1] - Filters based on the amplitude of the first derivative
   * @return {Array<object>}
   */

  function gsd(data, options = {}) {
    let {
      noiseLevel,
      sgOptions = {
        windowSize: 9,
        polynomial: 3
      },
      shape = {},
      smoothY = true,
      heightFactor = 0,
      broadRatio = 0.0,
      maxCriteria = true,
      minMaxRatio = 0.00025,
      derivativeThreshold = -1,
      realTopDetection = false
    } = options;
    let {
      y: yIn,
      x
    } = data;
    const y = yIn.slice();
    let equalSpaced = isEqualSpaced(x);

    if (maxCriteria === false) {
      for (let i = 0; i < y.length; i++) {
        y[i] *= -1;
      }
    }

    if (noiseLevel === undefined) {
      noiseLevel = equalSpaced ? getNoiseLevel(y) : 0;
    }

    for (let i = 0; i < y.length; i++) {
      y[i] -= noiseLevel;
    }

    for (let i = 0; i < y.length; i++) {
      if (y[i] < 0) {
        y[i] = 0;
      }
    } // If the max difference between delta x is less than 5%, then,
    // we can assume it to be equally spaced variable


    let yData = y;
    let dY, ddY;
    const {
      windowSize,
      polynomial
    } = sgOptions;

    if (equalSpaced) {
      if (smoothY) {
        yData = SavitzkyGolay(y, x[1] - x[0], {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 2
      });
    } else {
      if (smoothY) {
        yData = SavitzkyGolay(y, x, {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay(y, x, {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay(y, x, {
        windowSize,
        polynomial,
        derivative: 2
      });
    }

    const xData = x;
    const dX = x[1] - x[0];
    let maxDdy = 0;
    let maxY = 0;

    for (let i = 0; i < yData.length; i++) {
      if (Math.abs(ddY[i]) > maxDdy) {
        maxDdy = Math.abs(ddY[i]);
      }

      if (Math.abs(yData[i]) > maxY) {
        maxY = Math.abs(yData[i]);
      }
    }

    let lastMax = null;
    let lastMin = null;
    let minddY = [];
    let intervalL = [];
    let intervalR = [];
    let broadMask = []; // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum

    for (let i = 1; i < yData.length - 1; ++i) {
      // filter based on derivativeThreshold
      // console.log('pasa', y[i], dY[i], ddY[i]);
      if (Math.abs(dY[i]) > derivativeThreshold) {
        // Minimum in first derivative
        if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
          lastMin = {
            x: xData[i],
            index: i
          };

          if (dX > 0 && lastMax !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        } // Maximum in first derivative


        if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
          lastMax = {
            x: xData[i],
            index: i
          };

          if (dX < 0 && lastMin !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
      } // Minimum in second derivative


      if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
        minddY.push(i);
        broadMask.push(Math.abs(ddY[i]) <= broadRatio * maxDdy);
      }
    }

    let widthProcessor = shape.kind ? getShape1D(shape.kind, shape.options).widthToFWHM : x => x;
    let signals = [];
    let lastK = -1;
    let possible, frequency, distanceJ, minDistance, gettingCloser;

    for (let j = 0; j < minddY.length; ++j) {
      frequency = xData[minddY[j]];
      possible = -1;
      let k = lastK + 1;
      minDistance = Number.MAX_VALUE;
      distanceJ = 0;
      gettingCloser = true;

      while (possible === -1 && k < intervalL.length && gettingCloser) {
        distanceJ = Math.abs(frequency - (intervalL[k].x + intervalR[k].x) / 2); // Still getting closer?

        if (distanceJ < minDistance) {
          minDistance = distanceJ;
        } else {
          gettingCloser = false;
        }

        if (distanceJ < Math.abs(intervalL[k].x - intervalR[k].x) / 2) {
          possible = k;
          lastK = k;
        }

        ++k;
      }

      if (possible !== -1) {
        if (Math.abs(yData[minddY[j]]) > minMaxRatio * maxY) {
          let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
          signals.push({
            index: minddY[j],
            x: frequency,
            y: maxCriteria ? yData[minddY[j]] + noiseLevel : -yData[minddY[j]] - noiseLevel,
            width: widthProcessor(width),
            soft: broadMask[j]
          });
          signals[signals.length - 1].left = intervalL[possible];
          signals[signals.length - 1].right = intervalR[possible];

          if (heightFactor) {
            let yLeft = yData[intervalL[possible].index];
            let yRight = yData[intervalR[possible].index];
            signals[signals.length - 1].height = heightFactor * (signals[signals.length - 1].y - (yLeft + yRight) / 2);
          }
        }
      }
    }

    if (realTopDetection) {
      determineRealTop(signals, xData, yData);
    } // Correct the values to fit the original spectra data


    for (let j = 0; j < signals.length; j++) {
      signals[j].base = noiseLevel;
    }

    signals.sort(function (a, b) {
      return a.x - b.x;
    });
    return signals;
  }

  const isEqualSpaced = x => {
    let tmp;
    let maxDx = 0;
    let minDx = Number.MAX_SAFE_INTEGER;

    for (let i = 0; i < x.length - 1; ++i) {
      tmp = Math.abs(x[i + 1] - x[i]);

      if (tmp < minDx) {
        minDx = tmp;
      }

      if (tmp > maxDx) {
        maxDx = tmp;
      }
    }

    return (maxDx - minDx) / maxDx < 0.05;
  };

  const getNoiseLevel = y => {
    let mean = 0;
    let stddev = 0;
    let length = y.length;

    for (let i = 0; i < length; ++i) {
      mean += y[i];
    }

    mean /= length;
    let averageDeviations = new Array(length);

    for (let i = 0; i < length; ++i) {
      averageDeviations[i] = Math.abs(y[i] - mean);
    }

    averageDeviations.sort((a, b) => a - b);

    if (length % 2 === 1) {
      stddev = averageDeviations[(length - 1) / 2] / 0.6745;
    } else {
      stddev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
    }

    return stddev;
  };

  const determineRealTop = (peakList, x, y) => {
    let alpha, beta, gamma, p, currentPoint;

    for (let j = 0; j < peakList.length; j++) {
      currentPoint = peakList[j].index; // peakList[j][2];
      // The detected peak could be moved 1 or 2 units to left or right.

      if (y[currentPoint - 1] >= y[currentPoint - 2] && y[currentPoint - 1] >= y[currentPoint]) {
        currentPoint--;
      } else {
        if (y[currentPoint + 1] >= y[currentPoint] && y[currentPoint + 1] >= y[currentPoint + 2]) {
          currentPoint++;
        } else {
          if (y[currentPoint - 2] >= y[currentPoint - 3] && y[currentPoint - 2] >= y[currentPoint - 1]) {
            currentPoint -= 2;
          } else {
            if (y[currentPoint + 2] >= y[currentPoint + 1] && y[currentPoint + 2] >= y[currentPoint + 3]) {
              currentPoint += 2;
            }
          }
        }
      } // interpolation to a sin() function


      if (y[currentPoint - 1] > 0 && y[currentPoint + 1] > 0 && y[currentPoint] >= y[currentPoint - 1] && y[currentPoint] >= y[currentPoint + 1] && (y[currentPoint] !== y[currentPoint - 1] || y[currentPoint] !== y[currentPoint + 1])) {
        alpha = 20 * Math.log10(y[currentPoint - 1]);
        beta = 20 * Math.log10(y[currentPoint]);
        gamma = 20 * Math.log10(y[currentPoint + 1]);
        p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma); // console.log(alpha, beta, gamma, `p: ${p}`);
        // console.log(x[currentPoint]+" "+tmp+" "+currentPoint);

        peakList[j].x = x[currentPoint] + (x[currentPoint] - x[currentPoint - 1]) * p;
        peakList[j].y = y[currentPoint] - 0.25 * (y[currentPoint - 1] - y[currentPoint + 1]) * p;
      }
    }
  };

  var assignDeep$1 = {exports: {}};

  /*!
   * assign-symbols <https://github.com/jonschlinkert/assign-symbols>
   *
   * Copyright (c) 2015-present, Jon Schlinkert.
   * Licensed under the MIT License.
   */

  const toString$1 = Object.prototype.toString;
  const isEnumerable = Object.prototype.propertyIsEnumerable;
  const getSymbols = Object.getOwnPropertySymbols;

  var assignSymbols$1 = (target, ...args) => {
    if (!isObject$1(target)) {
      throw new TypeError('expected the first argument to be an object');
    }

    if (args.length === 0 || typeof Symbol !== 'function' || typeof getSymbols !== 'function') {
      return target;
    }

    for (let arg of args) {
      let names = getSymbols(arg);

      for (let key of names) {
        if (isEnumerable.call(arg, key)) {
          target[key] = arg[key];
        }
      }
    }

    return target;
  };

  function isObject$1(val) {
    return typeof val === 'function' || toString$1.call(val) === '[object Object]' || Array.isArray(val);
  }

  /*!
   * assign-deep <https://github.com/jonschlinkert/assign-deep>
   *
   * Copyright (c) 2017-present, Jon Schlinkert.
   * Released under the MIT License.
   */

  const toString = Object.prototype.toString;
  const assignSymbols = assignSymbols$1;

  const isValidKey = key => {
    return key !== '__proto__' && key !== 'constructor' && key !== 'prototype';
  };

  const assign = assignDeep$1.exports = (target, ...args) => {
    let i = 0;
    if (isPrimitive(target)) target = args[i++];
    if (!target) target = {};

    for (; i < args.length; i++) {
      if (isObject(args[i])) {
        for (const key of Object.keys(args[i])) {
          if (isValidKey(key)) {
            if (isObject(target[key]) && isObject(args[i][key])) {
              assign(target[key], args[i][key]);
            } else {
              target[key] = args[i][key];
            }
          }
        }

        assignSymbols(target, args[i]);
      }
    }

    return target;
  };

  function isObject(val) {
    return typeof val === 'function' || toString.call(val) === '[object Object]';
  }

  function isPrimitive(val) {
    return typeof val === 'object' ? val === null : typeof val !== 'function';
  }

  var assignDeep = assignDeep$1.exports;

  const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
  const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
  const ROOT_THREE = Math.sqrt(3);
  const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
  const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

  // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  // This code yields to a good approximation
  // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  function erfinv(x) {
    let a = 0.147;
    if (x === 0) return 0;
    let ln1MinusXSqrd = Math.log(1 - x * x);
    let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
    let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
    let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
    return secondSqrt * (x > 0 ? 1 : -1);
  }

  class Gaussian {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=4*LN2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.sd ? Gaussian.widthToFWHM(2 * options.sd) : options.fwhm ? options.fwhm : 500;
      this.height = options.height === undefined ? Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) / this.fwhm : options.height;
    }
    /**
     * Calculate a gaussian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = 6] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a gaussian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of gaussian with the current parameters.
     */


    fct(x) {
      return Gaussian.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Gaussian.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Gaussian.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} width - Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      //https://mathworld.wolfram.com/GaussianFunction.html
      return Gaussian.widthToFWHM(width);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} fwhm - Full Width at Half Maximum.
     * @returns {number} width
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Gaussian.fwhmToWidth(fwhm);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Gaussian.fct = function fct(x, fwhm = 500) {
    return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} width - Width between the inflection points
   * @returns {number} fwhm
   */


  Gaussian.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_2LN2;
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} fwhm - Full Width at Half Maximum.
   * @returns {number} width
   */


  Gaussian.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {object} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Gaussian.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * ROOT_PI_OVER_LN2 * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Gaussian.getFactor = function getFactor(area = 0.9999) {
    return Math.sqrt(2) * erfinv(area);
  };

  class Lorentzian {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 2 / Math.PI / this.fwhm : options.height;
    }
    /**
     * Calculate a lorentzian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a lorentzian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of lorentzian with the current parameters.
     */


    fct(x) {
      return Lorentzian.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Lorentzian.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Lorentzian.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [fwhm] - Full Width at Half Maximum.
     * @returns {number} width between the inflection points
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Lorentzian.fwhmToWidth(fwhm);
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [width] Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      return Lorentzian.widthToFWHM(width);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Lorentzian.fct = function fct(x, fwhm) {
    const squareFWHM = fwhm * fwhm;
    return squareFWHM / (4 * Math.pow(x, 2) + squareFWHM);
  };
  /**
   * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [fwhm] - Full Width at Half Maximum.
   * @returns {number} width between the inflection points
   */


  Lorentzian.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_THREE;
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [width] Width between the inflection points
   * @returns {number} fwhm
   */


  Lorentzian.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_THREE;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Lorentzian.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * Math.PI * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Lorentzian.getFactor = function getFactor(area = 0.9999) {
    return 2 * Math.tan(Math.PI * (area - 0.5));
  };

  class PseudoVoigt {
    /**
     * @param {object} [options={}]
     * @param {number} [options.height=1/(mu*FWHM/sqrt(4*LN2/PI)+(1-mu)*fwhm*PI*0.5)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm=500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.mu=0.5] - ratio of gaussian contribution.
     */
    constructor(options = {}) {
      this.mu = options.mu === undefined ? 0.5 : options.mu;
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 1 / (this.mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * this.fwhm + (1 - this.mu) * this.fwhm * Math.PI / 2) : options.height;
    }
    /**
     * Calculate a linear combination of gaussian and lorentzian function width an same full width at half maximum
     * @param { object } [options = {}]
     * @param { number } [options.factor = 2 * Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm in the calculation of the length.Default covers 99.99 % of area.
     * @param { number } [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return { object } - { fwhm, data<Float64Array>} - An with the number of points at half maximum and the array of y values covering the 99.99 % of the area.
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.ceil(this.fwhm * factor);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      let data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a linear combination of Gaussian and Lorentzian shapes where the full width at half maximum are the same for both kind of shapes (see README for equation).
     * @param {number} [x] x value to calculate.
     * @returns {number} - the y value of a pseudo voigt with the current parameters.
     */


    fct(x) {
      return PseudoVoigt.fct(x, this.fwhm, this.mu);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999] - required area to be coverage
     * @param {number} [mu=this.mu] - ratio of gaussian contribution.
     * @returns {number}
     */


    getFactor(area = 0.9999, mu = this.mu) {
      return PseudoVoigt.getFactor(area, mu);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return PseudoVoigt.getArea(this.fwhm, {
        height: this.height,
        mu: this.mu
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
     * @param {number} width - width between the inflection points
     * @param {number} [mu = 0.5] - ratio of gaussian contribution.
     * @returns {number} Full Width at Half Maximum (FMHM).
     */


    widthToFWHM(width, mu) {
      return PseudoVoigt.widthToFWHM(width, mu);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * @param {number} fwhm - Full Width at Half Maximum.
     * @param {number} [mu] - ratio of gaussian contribution.
     * @returns {number} width between the inflection points.
     */


    fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
      return PseudoVoigt.fwhmToWidth(fwhm, mu);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }
    /**
     * set a new mu
     * @param {number} mu - ratio of gaussian contribution.
     */


    setMu(mu) {
      this.mu = mu;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @param {number} [mu=0.5] - ratio of gaussian contribution.
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  PseudoVoigt.fct = function fct(x, fwhm, mu = 0.5) {
    return (1 - mu) * Lorentzian.fct(x, fwhm) + mu * Gaussian.fct(x, fwhm);
  };
  /**
   * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
   * @param {number} width - width between the inflection points
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} Full Width at Half Maximum (FMHM).
   */


  PseudoVoigt.widthToFWHM = function widthToFWHM(width, mu = 0.5) {
    return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * @param {number} fwhm - Full Width at Half Maximum.
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} width between the inflection points.
   */


  PseudoVoigt.fwhmToWidth = function fwhmToWidth(fwhm, mu = 0.5) {
    return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @param {number} [options.mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  PseudoVoigt.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1,
      mu = 0.5
    } = options;
    return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999] - required area to be coverage
   * @param {number} [mu=this.mu] - ratio of gaussian contribution.
   * @returns {number}
   */


  PseudoVoigt.getFactor = function getFactor(area = 0.9999, mu = 0.5) {
    return mu < 1 ? Lorentzian.getFactor(area) : Gaussian.getFactor(area);
  };

  /**
   * This function calculates the spectrum as a sum of linear combination of gaussian and lorentzian functions. The pseudo voigt
   * parameters are divided in 4 batches. 1st: centers; 2nd: heights; 3th: widths; 4th: mu's ;
   * @param t Ordinate value
   * @param p Lorentzian parameters
   * @returns {*}
   */

  function sumOfGaussianLorentzians(p) {
    return function (t) {
      let nL = p.length / 4;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * PseudoVoigt.fct(t - p[i], p[i + nL * 2], p[i + nL * 3]);
      }

      return result;
    };
  }

  /**
   * This function calculates the spectrum as a sum of gaussian functions. The Gaussian
   * parameters are divided in 3 batches. 1st: centers; 2nd: height; 3th: widths;
   * @param t Ordinate values
   * @param p Gaussian parameters
   * @returns {*}
   */

  function sumOfGaussians(p) {
    return function (t) {
      let nL = p.length / 3;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * Gaussian.fct(t - p[i], p[i + nL * 2]);
      }

      return result;
    };
  }

  /**
   * This function calculates the spectrum as a sum of lorentzian functions. The Lorentzian
   * parameters are divided in 3 batches. 1st: centers; 2nd: heights; 3th: widths;
   * @param t Ordinate values
   * @param p Lorentzian parameters
   * @returns {*}
   */

  function sumOfLorentzians(p) {
    return function (t) {
      let nL = p.length / 3;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * Lorentzian.fct(t - p[i], p[i + nL * 2]);
      }

      return result;
    };
  }

  function checkInput(data, peaks, options) {
    let {
      shape = {
        kind: 'gaussian'
      },
      optimization = {
        kind: 'lm'
      }
    } = options;

    if (typeof shape.kind !== 'string') {
      throw new Error('kind should be a string');
    }

    let kind = shape.kind.toLowerCase().replace(/[^a-z]/g, '');
    let paramsFunc;
    let defaultParameters;

    switch (kind) {
      case 'gaussian':
        paramsFunc = sumOfGaussians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          }
        };
        break;

      case 'lorentzian':
        paramsFunc = sumOfLorentzians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          }
        };
        break;

      case 'pseudovoigt':
        paramsFunc = sumOfGaussianLorentzians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          },
          mu: {
            init: peak => peak.mu !== undefined ? peak.mu : 0.5,
            min: () => 0,
            max: () => 1,
            gradientDifference: () => 0.01
          }
        };
        break;

      default:
        throw new Error('kind of shape is not supported');
    }

    let x = data.x;
    let maxY = max(data.y);
    let y = new Array(x.length);

    for (let i = 0; i < x.length; i++) {
      y[i] = data.y[i] / maxY;
    }

    for (let i = 0; i < peaks.length; i++) {
      peaks[i].y /= maxY;
    }

    let parameters = assignDeep({}, optimization.parameters, defaultParameters);

    for (let key in parameters) {
      for (let par in parameters[key]) {
        if (!Array.isArray(parameters[key][par])) {
          parameters[key][par] = [parameters[key][par]];
        }

        if (parameters[key][par].length !== 1 && parameters[key][par].length !== peaks.length) {
          throw new Error(`The length of ${key}-${par} is not correct`);
        }

        for (let index = 0; index < parameters[key][par].length; index++) {
          if (typeof parameters[key][par][index] === 'number') {
            let value = parameters[key][par][index];

            parameters[key][par][index] = () => value;
          }
        }
      }
    }

    optimization.parameters = parameters;
    return {
      y,
      x,
      maxY,
      peaks,
      paramsFunc,
      optimization
    };
  }

  function checkOptions$1(data, parameterizedFunction, options) {
    let {
      timeout,
      minValues,
      maxValues,
      initialValues,
      weights = 1,
      damping = 1e-2,
      dampingStepUp = 11,
      dampingStepDown = 9,
      maxIterations = 100,
      errorTolerance = 1e-7,
      centralDifference = false,
      gradientDifference = 10e-2,
      improvementThreshold = 1e-3
    } = options;

    if (damping <= 0) {
      throw new Error('The damping option must be a positive number');
    } else if (!data.x || !data.y) {
      throw new Error('The data parameter must have x and y elements');
    } else if (!isAnyArray(data.x) || data.x.length < 2 || !isAnyArray(data.y) || data.y.length < 2) {
      throw new Error('The data parameter elements must be an array with more than 2 points');
    } else if (data.x.length !== data.y.length) {
      throw new Error('The data parameter elements must have the same size');
    }

    let parameters = initialValues || new Array(parameterizedFunction.length).fill(1);
    let nbPoints = data.y.length;
    let parLen = parameters.length;
    maxValues = maxValues || new Array(parLen).fill(Number.MAX_SAFE_INTEGER);
    minValues = minValues || new Array(parLen).fill(Number.MIN_SAFE_INTEGER);

    if (maxValues.length !== minValues.length) {
      throw new Error('minValues and maxValues must be the same size');
    }

    if (!isAnyArray(parameters)) {
      throw new Error('initialValues must be an array');
    }

    if (typeof gradientDifference === 'number') {
      gradientDifference = new Array(parameters.length).fill(gradientDifference);
    } else if (isAnyArray(gradientDifference)) {
      if (gradientDifference.length !== parLen) {
        gradientDifference = new Array(parLen).fill(gradientDifference[0]);
      }
    } else {
      throw new Error('gradientDifference should be a number or array with length equal to the number of parameters');
    }

    let filler;

    if (typeof weights === 'number') {
      let value = 1 / weights ** 2;

      filler = () => value;
    } else if (isAnyArray(weights)) {
      if (weights.length < data.x.length) {
        let value = 1 / weights[0] ** 2;

        filler = () => value;
      } else {
        filler = i => 1 / weights[i] ** 2;
      }
    } else {
      throw new Error('weights should be a number or array with length equal to the number of data points');
    }

    let checkTimeout;

    if (timeout !== undefined) {
      if (typeof timeout !== 'number') {
        throw new Error('timeout should be a number');
      }

      let endTime = Date.now() + timeout * 1000;

      checkTimeout = () => Date.now() > endTime;
    } else {
      checkTimeout = () => false;
    }

    let weightSquare = new Array(data.x.length);

    for (let i = 0; i < nbPoints; i++) {
      weightSquare[i] = filler(i);
    }

    return {
      checkTimeout,
      minValues,
      maxValues,
      parameters,
      weightSquare,
      damping,
      dampingStepUp,
      dampingStepDown,
      maxIterations,
      errorTolerance,
      centralDifference,
      gradientDifference,
      improvementThreshold
    };
  }

  /**
   * the sum of the weighted squares of the errors (or weighted residuals) between the data.y
   * and the curve-fit function.
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} parameters - Array of current parameter values
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @param {Array} weightSquare - Square of weights
   * @return {number}
   */
  function errorCalculation(data, parameters, parameterizedFunction, weightSquare) {
    let error = 0;
    const func = parameterizedFunction(parameters);

    for (let i = 0; i < data.x.length; i++) {
      error += Math.pow(data.y[i] - func(data.x[i]), 2) / weightSquare[i];
    }

    return error;
  }

  function rescale(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    var currentMin = min(input);
    var currentMax = max(input);

    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }

    var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }

    var factor = (maxValue - minValue) / (currentMax - currentMin);

    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }

    return output;
  }

  const indent = ' '.repeat(2);
  const indentData = ' '.repeat(4);
  function inspectMatrix() {
    return inspectMatrixWithOptions(this);
  }
  function inspectMatrixWithOptions(matrix, options = {}) {
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8
    } = options;
    return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
  }

  function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];

    for (let i = 0; i < maxI; i++) {
      let line = [];

      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber(matrix.get(i, j), maxNumSize));
      }

      result.push(`${line.join(' ')}`);
    }

    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }

    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }

    return result.join(`\n${indentData}`);
  }

  function formatNumber(num, maxNumSize) {
    const numStr = String(num);

    if (numStr.length <= maxNumSize) {
      return numStr.padEnd(maxNumSize, ' ');
    }

    const precise = num.toPrecision(maxNumSize - 2);

    if (precise.length <= maxNumSize) {
      return precise;
    }

    const exponential = num.toExponential(maxNumSize - 2);
    const eIndex = exponential.indexOf('e');
    const e = exponential.slice(eIndex);
    return exponential.slice(0, maxNumSize - e.length) + e;
  }

  function installMathOperations(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };

    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };

    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };

    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };

    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;

    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };

    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };

    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;

    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };

    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };

    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;

    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };

    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };

    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;

    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };

    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };

    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };

    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };

    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };

    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };

    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };

    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };

    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };

    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };

    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };

    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };

    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };

    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };

    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };

    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };

    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };

    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };

    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };

    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };

    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };

    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };

    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };

    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };

    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };

    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };

    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };

    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };

    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };

    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };

    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };

    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };

    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };

    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };

    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };

    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };

    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };

    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };

    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };

    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };

    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };

    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }

      return this;
    };

    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }

      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }
  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */

  function checkColumnIndex(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkRowVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }

    return vector;
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkColumnVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }

    return vector;
  }
  function checkIndices(matrix, rowIndices, columnIndices) {
    return {
      row: checkRowIndices(matrix, rowIndices),
      column: checkColumnIndices(matrix, columnIndices)
    };
  }
  function checkRowIndices(matrix, rowIndices) {
    if (typeof rowIndices !== 'object') {
      throw new TypeError('unexpected type for row indices');
    }

    let rowOut = rowIndices.some(r => {
      return r < 0 || r >= matrix.rows;
    });

    if (rowOut) {
      throw new RangeError('row indices are out of range');
    }

    if (!Array.isArray(rowIndices)) rowIndices = Array.from(rowIndices);
    return rowIndices;
  }
  function checkColumnIndices(matrix, columnIndices) {
    if (typeof columnIndices !== 'object') {
      throw new TypeError('unexpected type for column indices');
    }

    let columnOut = columnIndices.some(c => {
      return c < 0 || c >= matrix.columns;
    });

    if (columnOut) {
      throw new RangeError('column indices are out of range');
    }

    if (!Array.isArray(columnIndices)) columnIndices = Array.from(columnIndices);
    return columnIndices;
  }
  function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }

    checkNumber('startRow', startRow);
    checkNumber('endRow', endRow);
    checkNumber('startColumn', startColumn);
    checkNumber('endColumn', endColumn);

    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray(length, value = 0) {
    let array = [];

    for (let i = 0; i < length; i++) {
      array.push(value);
    }

    return array;
  }

  function checkNumber(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }

  function checkNonEmpty(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow(matrix) {
    let sum = newArray(matrix.rows);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumByColumn(matrix) {
    let sum = newArray(matrix.columns);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumAll(matrix) {
    let v = 0;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }

    return v;
  }
  function productByRow(matrix) {
    let sum = newArray(matrix.rows, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productByColumn(matrix) {
    let sum = newArray(matrix.columns, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productAll(matrix) {
    let v = 1;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }

    return v;
  }
  function varianceByRow(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }

    return variance;
  }
  function varianceByColumn(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }

    return variance;
  }
  function varianceAll(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;

    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }

    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow(matrix) {
    const scale = [];

    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;

      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByRow(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn(matrix) {
    const scale = [];

    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;

      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByColumn(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;

    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }

    return Math.sqrt(sum);
  }
  function scaleAll(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;

      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }

      let newMatrix = new Matrix(newRows, newColumns);

      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }

      return newMatrix;
    }

    static rowVector(newData) {
      let vector = new Matrix(1, newData.length);

      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }

      return vector;
    }

    static columnVector(newData) {
      let vector = new Matrix(newData.length, 1);

      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }

      return vector;
    }

    static zeros(rows, columns) {
      return new Matrix(rows, columns);
    }

    static ones(rows, columns) {
      return new Matrix(rows, columns).fill(1);
    }

    static rand(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        random = Math.random
      } = options;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }

      return matrix;
    }

    static randInt(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }

      return matrix;
    }

    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }

      return matrix;
    }

    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }

      return matrix;
    }

    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static checkMatrix(value) {
      return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
    }

    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }

    get size() {
      return this.rows * this.columns;
    }

    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }

      return this;
    }

    to1DArray() {
      let array = [];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }

      return array;
    }

    to2DArray() {
      let copy = [];

      for (let i = 0; i < this.rows; i++) {
        copy.push([]);

        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }

      return copy;
    }

    toJSON() {
      return this.to2DArray();
    }

    isRowVector() {
      return this.rows === 1;
    }

    isColumnVector() {
      return this.columns === 1;
    }

    isVector() {
      return this.rows === 1 || this.columns === 1;
    }

    isSquare() {
      return this.rows === this.columns;
    }

    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }

    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }

        return true;
      }

      return false;
    }

    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;

      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }

        i++;
      }

      return isEchelonForm;
    }

    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;

      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }

        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }

        i++;
      }

      return isReducedEchelonForm;
    }

    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;

      while (h < result.rows && k < result.columns) {
        let iMax = h;

        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }

        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);

          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }

          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);

            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }

          h++;
          k++;
        }
      }

      return result;
    }

    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;

      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;

          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }

          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);

            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }

          h--;
        }
      }

      return result;
    }

    set() {
      throw new Error('set method is unimplemented');
    }

    get() {
      throw new Error('get method is unimplemented');
    }

    repeat(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        rows = 1,
        columns = 1
      } = options;

      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }

      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }

      let matrix = new Matrix(this.rows * rows, this.columns * columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }

      return matrix;
    }

    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }

      return this;
    }

    neg() {
      return this.mulS(-1);
    }

    getRow(index) {
      checkRowIndex(this, index);
      let row = [];

      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }

      return row;
    }

    getRowVector(index) {
      return Matrix.rowVector(this.getRow(index));
    }

    setRow(index, array) {
      checkRowIndex(this, index);
      array = checkRowVector(this, array);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }

      return this;
    }

    swapRows(row1, row2) {
      checkRowIndex(this, row1);
      checkRowIndex(this, row2);

      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }

      return this;
    }

    getColumn(index) {
      checkColumnIndex(this, index);
      let column = [];

      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }

      return column;
    }

    getColumnVector(index) {
      return Matrix.columnVector(this.getColumn(index));
    }

    setColumn(index, array) {
      checkColumnIndex(this, index);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }

      return this;
    }

    swapColumns(column1, column2) {
      checkColumnIndex(this, column1);
      checkColumnIndex(this, column2);

      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }

      return this;
    }

    addRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }

      return this;
    }

    subRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }

      return this;
    }

    mulRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }

      return this;
    }

    divRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }

      return this;
    }

    addColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }

      return this;
    }

    subColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }

      return this;
    }

    mulColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }

      return this;
    }

    divColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }

      return this;
    }

    mulRow(index, value) {
      checkRowIndex(this, index);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }

      return this;
    }

    mulColumn(index, value) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }

      return this;
    }

    max() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    maxIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    min() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    minIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    maxRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    maxRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    minRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    minRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    maxColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    maxColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    minColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    minColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];

      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }

      return diag;
    }

    norm(type = 'frobenius') {
      let result = 0;

      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }

        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }

    cumulativeSum() {
      let sum = 0;

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }

      return this;
    }

    dot(vector2) {
      if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();

      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }

      let dot = 0;

      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }

      return dot;
    }

    mmul(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix(m, p);
      let Bcolj = new Float64Array(n);

      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }

        for (let i = 0; i < m; i++) {
          let s = 0;

          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }

          result.set(i, j, s);
        }
      }

      return result;
    }

    strassen2x2(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1); // Compute intermediate values.

      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }

    strassen3x3(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }

    mmulStrassen(y) {
      y = Matrix.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;

      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      } // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.


      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;

        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      } // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.


      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c); // Our recursive multiplication function.

      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        } // Apply dynamic padding.


        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }

        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

        let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

        let c11 = AbstractMatrix.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix.add(m3, m5);
        let c21 = AbstractMatrix.add(m2, m4);
        let c22 = AbstractMatrix.sub(m1, m2);
        c22.add(m3);
        c22.add(m6); // Crop output to the desired size (undo dynamic padding).

        let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }

      return blockMult(x, y, r, c);
    }

    scaleRows(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);

        if (row.length > 0) {
          rescale(row, {
            min,
            max,
            output: row
          });
        }

        newMatrix.setRow(i, row);
      }

      return newMatrix;
    }

    scaleColumns(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);

        if (column.length) {
          rescale(column, {
            min: min,
            max: max,
            output: column
          });
        }

        newMatrix.setColumn(i, column);
      }

      return newMatrix;
    }

    flipRows() {
      const middle = Math.ceil(this.columns / 2);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }

      return this;
    }

    flipColumns() {
      const middle = Math.ceil(this.rows / 2);

      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }

      return this;
    }

    kroneckerProduct(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix(m * p, n * q);

      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }

      return result;
    }

    kroneckerSum(other) {
      other = Matrix.checkMatrix(other);

      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }

      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix.eye(n, n));
      let IxB = Matrix.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }

    transpose() {
      let result = new Matrix(this.columns, this.rows);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }

      return result;
    }

    sortRows(compareFunction = compareNumbers) {
      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }

      return this;
    }

    sortColumns(compareFunction = compareNumbers) {
      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }

      return this;
    }

    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }

      return newMatrix;
    }

    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;

      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }

          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }

      return newMatrix;
    }

    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;

      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }

          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }

      return newMatrix;
    }

    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix.checkMatrix(matrix);

      if (matrix.isEmpty()) {
        return this;
      }

      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange(this, startRow, endRow, startColumn, endColumn);

      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }

      return this;
    }

    selection(rowIndices, columnIndices) {
      let indices = checkIndices(this, rowIndices, columnIndices);
      let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

      for (let i = 0; i < indices.row.length; i++) {
        let rowIndex = indices.row[i];

        for (let j = 0; j < indices.column.length; j++) {
          let columnIndex = indices.column[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }

      return newMatrix;
    }

    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;

      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }

      return trace;
    }

    clone() {
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }

      return newMatrix;
    }

    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow(this);

        case 'column':
          return sumByColumn(this);

        case undefined:
          return sumAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    product(by) {
      switch (by) {
        case 'row':
          return productByRow(this);

        case 'column':
          return productByColumn(this);

        case undefined:
          return productAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    mean(by) {
      const sum = this.sum(by);

      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }

            return sum;
          }

        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }

            return sum;
          }

        case undefined:
          return sum / this.size;

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    variance(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;

      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByRow(this, unbiased, mean);
          }

        case 'column':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByColumn(this, unbiased, mean);
          }

        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }

            return varianceAll(this, unbiased, mean);
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      const variance = this.variance(by, options);

      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }

        return variance;
      }
    }

    center(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        center = this.mean(by)
      } = options;

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByRow(this, center);
            return this;
          }

        case 'column':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByColumn(this, center);
            return this;
          }

        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }

            centerAll(this, center);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    scale(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      let scale = options.scale;

      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByRow(this, scale);
            return this;
          }

        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByColumn(this, scale);
            return this;
          }

        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }

            scaleAll(this, scale);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    toString(options) {
      return inspectMatrixWithOptions(this, options);
    }

  }
  AbstractMatrix.prototype.klass = 'Matrix';

  if (typeof Symbol !== 'undefined') {
    AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
  }

  function compareNumbers(a, b) {
    return a - b;
  } // Synonyms


  AbstractMatrix.random = AbstractMatrix.rand;
  AbstractMatrix.randomInt = AbstractMatrix.randInt;
  AbstractMatrix.diagonal = AbstractMatrix.diag;
  AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
  AbstractMatrix.identity = AbstractMatrix.eye;
  AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
  AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
  class Matrix extends AbstractMatrix {
    constructor(nRows, nColumns) {
      super();

      if (Matrix.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];

        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (Array.isArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;

        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }

        this.data = [];

        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }

          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }

      this.rows = nRows;
      this.columns = nColumns;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

    removeRow(index) {
      checkRowIndex(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }

    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }

      checkRowIndex(this, index, true);
      array = Float64Array.from(checkRowVector(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }

    removeColumn(index) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);

        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }

        this.data[i] = newRow;
      }

      this.columns -= 1;
      return this;
    }

    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }

      checkColumnIndex(this, index, true);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;

        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        newRow[j++] = array[i];

        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }

        this.data[i] = newRow;
      }

      this.columns += 1;
      return this;
    }

  }
  installMathOperations(AbstractMatrix, Matrix);

  class BaseView extends AbstractMatrix {
    constructor(matrix, rows, columns) {
      super();
      this.matrix = matrix;
      this.rows = rows;
      this.columns = columns;
    }

  }

  class MatrixTransposeView extends BaseView {
    constructor(matrix) {
      super(matrix, matrix.columns, matrix.rows);
    }

    set(rowIndex, columnIndex, value) {
      this.matrix.set(columnIndex, rowIndex, value);
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.matrix.get(columnIndex, rowIndex);
    }

  }

  class WrapperMatrix2D extends AbstractMatrix {
    constructor(data) {
      super();
      this.data = data;
      this.rows = data.length;
      this.columns = data[0].length;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

  }

  class LuDecomposition {
    constructor(matrix) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      let lu = matrix.clone();
      let rows = lu.rows;
      let columns = lu.columns;
      let pivotVector = new Float64Array(rows);
      let pivotSign = 1;
      let i, j, k, p, s, t, v;
      let LUcolj, kmax;

      for (i = 0; i < rows; i++) {
        pivotVector[i] = i;
      }

      LUcolj = new Float64Array(rows);

      for (j = 0; j < columns; j++) {
        for (i = 0; i < rows; i++) {
          LUcolj[i] = lu.get(i, j);
        }

        for (i = 0; i < rows; i++) {
          kmax = Math.min(i, j);
          s = 0;

          for (k = 0; k < kmax; k++) {
            s += lu.get(i, k) * LUcolj[k];
          }

          LUcolj[i] -= s;
          lu.set(i, j, LUcolj[i]);
        }

        p = j;

        for (i = j + 1; i < rows; i++) {
          if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
            p = i;
          }
        }

        if (p !== j) {
          for (k = 0; k < columns; k++) {
            t = lu.get(p, k);
            lu.set(p, k, lu.get(j, k));
            lu.set(j, k, t);
          }

          v = pivotVector[p];
          pivotVector[p] = pivotVector[j];
          pivotVector[j] = v;
          pivotSign = -pivotSign;
        }

        if (j < rows && lu.get(j, j) !== 0) {
          for (i = j + 1; i < rows; i++) {
            lu.set(i, j, lu.get(i, j) / lu.get(j, j));
          }
        }
      }

      this.LU = lu;
      this.pivotVector = pivotVector;
      this.pivotSign = pivotSign;
    }

    isSingular() {
      let data = this.LU;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        if (data.get(j, j) === 0) {
          return true;
        }
      }

      return false;
    }

    solve(value) {
      value = Matrix.checkMatrix(value);
      let lu = this.LU;
      let rows = lu.rows;

      if (rows !== value.rows) {
        throw new Error('Invalid matrix dimensions');
      }

      if (this.isSingular()) {
        throw new Error('LU matrix is singular');
      }

      let count = value.columns;
      let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
      let columns = lu.columns;
      let i, j, k;

      for (k = 0; k < columns; k++) {
        for (i = k + 1; i < columns; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      for (k = columns - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / lu.get(k, k));
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      return X;
    }

    get determinant() {
      let data = this.LU;

      if (!data.isSquare()) {
        throw new Error('Matrix must be square');
      }

      let determinant = this.pivotSign;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        determinant *= data.get(j, j);
      }

      return determinant;
    }

    get lowerTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i > j) {
            X.set(i, j, data.get(i, j));
          } else if (i === j) {
            X.set(i, j, 1);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get upperTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i <= j) {
            X.set(i, j, data.get(i, j));
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get pivotPermutationVector() {
      return Array.from(this.pivotVector);
    }

  }

  function hypotenuse(a, b) {
    let r = 0;

    if (Math.abs(a) > Math.abs(b)) {
      r = b / a;
      return Math.abs(a) * Math.sqrt(1 + r * r);
    }

    if (b !== 0) {
      r = a / b;
      return Math.abs(b) * Math.sqrt(1 + r * r);
    }

    return 0;
  }

  class QrDecomposition {
    constructor(value) {
      value = WrapperMatrix2D.checkMatrix(value);
      let qr = value.clone();
      let m = value.rows;
      let n = value.columns;
      let rdiag = new Float64Array(n);
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        let nrm = 0;

        for (i = k; i < m; i++) {
          nrm = hypotenuse(nrm, qr.get(i, k));
        }

        if (nrm !== 0) {
          if (qr.get(k, k) < 0) {
            nrm = -nrm;
          }

          for (i = k; i < m; i++) {
            qr.set(i, k, qr.get(i, k) / nrm);
          }

          qr.set(k, k, qr.get(k, k) + 1);

          for (j = k + 1; j < n; j++) {
            s = 0;

            for (i = k; i < m; i++) {
              s += qr.get(i, k) * qr.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < m; i++) {
              qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
            }
          }
        }

        rdiag[k] = -nrm;
      }

      this.QR = qr;
      this.Rdiag = rdiag;
    }

    solve(value) {
      value = Matrix.checkMatrix(value);
      let qr = this.QR;
      let m = qr.rows;

      if (value.rows !== m) {
        throw new Error('Matrix row dimensions must agree');
      }

      if (!this.isFullRank()) {
        throw new Error('Matrix is rank deficient');
      }

      let count = value.columns;
      let X = value.clone();
      let n = qr.columns;
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        for (j = 0; j < count; j++) {
          s = 0;

          for (i = k; i < m; i++) {
            s += qr.get(i, k) * X.get(i, j);
          }

          s = -s / qr.get(k, k);

          for (i = k; i < m; i++) {
            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
          }
        }
      }

      for (k = n - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / this.Rdiag[k]);
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
          }
        }
      }

      return X.subMatrix(0, n - 1, 0, count - 1);
    }

    isFullRank() {
      let columns = this.QR.columns;

      for (let i = 0; i < columns; i++) {
        if (this.Rdiag[i] === 0) {
          return false;
        }
      }

      return true;
    }

    get upperTriangularMatrix() {
      let qr = this.QR;
      let n = qr.columns;
      let X = new Matrix(n, n);
      let i, j;

      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          if (i < j) {
            X.set(i, j, qr.get(i, j));
          } else if (i === j) {
            X.set(i, j, this.Rdiag[i]);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get orthogonalMatrix() {
      let qr = this.QR;
      let rows = qr.rows;
      let columns = qr.columns;
      let X = new Matrix(rows, columns);
      let i, j, k, s;

      for (k = columns - 1; k >= 0; k--) {
        for (i = 0; i < rows; i++) {
          X.set(i, k, 0);
        }

        X.set(k, k, 1);

        for (j = k; j < columns; j++) {
          if (qr.get(k, k) !== 0) {
            s = 0;

            for (i = k; i < rows; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < rows; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
      }

      return X;
    }

  }

  class SingularValueDecomposition {
    constructor(value, options = {}) {
      value = WrapperMatrix2D.checkMatrix(value);

      if (value.isEmpty()) {
        throw new Error('Matrix must be non-empty');
      }

      let m = value.rows;
      let n = value.columns;
      const {
        computeLeftSingularVectors = true,
        computeRightSingularVectors = true,
        autoTranspose = false
      } = options;
      let wantu = Boolean(computeLeftSingularVectors);
      let wantv = Boolean(computeRightSingularVectors);
      let swapped = false;
      let a;

      if (m < n) {
        if (!autoTranspose) {
          a = value.clone(); // eslint-disable-next-line no-console

          console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
        } else {
          a = value.transpose();
          m = a.rows;
          n = a.columns;
          swapped = true;
          let aux = wantu;
          wantu = wantv;
          wantv = aux;
        }
      } else {
        a = value.clone();
      }

      let nu = Math.min(m, n);
      let ni = Math.min(m + 1, n);
      let s = new Float64Array(ni);
      let U = new Matrix(m, nu);
      let V = new Matrix(n, n);
      let e = new Float64Array(n);
      let work = new Float64Array(m);
      let si = new Float64Array(ni);

      for (let i = 0; i < ni; i++) si[i] = i;

      let nct = Math.min(m - 1, n);
      let nrt = Math.max(0, Math.min(n - 2, m));
      let mrc = Math.max(nct, nrt);

      for (let k = 0; k < mrc; k++) {
        if (k < nct) {
          s[k] = 0;

          for (let i = k; i < m; i++) {
            s[k] = hypotenuse(s[k], a.get(i, k));
          }

          if (s[k] !== 0) {
            if (a.get(k, k) < 0) {
              s[k] = -s[k];
            }

            for (let i = k; i < m; i++) {
              a.set(i, k, a.get(i, k) / s[k]);
            }

            a.set(k, k, a.get(k, k) + 1);
          }

          s[k] = -s[k];
        }

        for (let j = k + 1; j < n; j++) {
          if (k < nct && s[k] !== 0) {
            let t = 0;

            for (let i = k; i < m; i++) {
              t += a.get(i, k) * a.get(i, j);
            }

            t = -t / a.get(k, k);

            for (let i = k; i < m; i++) {
              a.set(i, j, a.get(i, j) + t * a.get(i, k));
            }
          }

          e[j] = a.get(k, j);
        }

        if (wantu && k < nct) {
          for (let i = k; i < m; i++) {
            U.set(i, k, a.get(i, k));
          }
        }

        if (k < nrt) {
          e[k] = 0;

          for (let i = k + 1; i < n; i++) {
            e[k] = hypotenuse(e[k], e[i]);
          }

          if (e[k] !== 0) {
            if (e[k + 1] < 0) {
              e[k] = 0 - e[k];
            }

            for (let i = k + 1; i < n; i++) {
              e[i] /= e[k];
            }

            e[k + 1] += 1;
          }

          e[k] = -e[k];

          if (k + 1 < m && e[k] !== 0) {
            for (let i = k + 1; i < m; i++) {
              work[i] = 0;
            }

            for (let i = k + 1; i < m; i++) {
              for (let j = k + 1; j < n; j++) {
                work[i] += e[j] * a.get(i, j);
              }
            }

            for (let j = k + 1; j < n; j++) {
              let t = -e[j] / e[k + 1];

              for (let i = k + 1; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * work[i]);
              }
            }
          }

          if (wantv) {
            for (let i = k + 1; i < n; i++) {
              V.set(i, k, e[i]);
            }
          }
        }
      }

      let p = Math.min(n, m + 1);

      if (nct < n) {
        s[nct] = a.get(nct, nct);
      }

      if (m < p) {
        s[p - 1] = 0;
      }

      if (nrt + 1 < p) {
        e[nrt] = a.get(nrt, p - 1);
      }

      e[p - 1] = 0;

      if (wantu) {
        for (let j = nct; j < nu; j++) {
          for (let i = 0; i < m; i++) {
            U.set(i, j, 0);
          }

          U.set(j, j, 1);
        }

        for (let k = nct - 1; k >= 0; k--) {
          if (s[k] !== 0) {
            for (let j = k + 1; j < nu; j++) {
              let t = 0;

              for (let i = k; i < m; i++) {
                t += U.get(i, k) * U.get(i, j);
              }

              t = -t / U.get(k, k);

              for (let i = k; i < m; i++) {
                U.set(i, j, U.get(i, j) + t * U.get(i, k));
              }
            }

            for (let i = k; i < m; i++) {
              U.set(i, k, -U.get(i, k));
            }

            U.set(k, k, 1 + U.get(k, k));

            for (let i = 0; i < k - 1; i++) {
              U.set(i, k, 0);
            }
          } else {
            for (let i = 0; i < m; i++) {
              U.set(i, k, 0);
            }

            U.set(k, k, 1);
          }
        }
      }

      if (wantv) {
        for (let k = n - 1; k >= 0; k--) {
          if (k < nrt && e[k] !== 0) {
            for (let j = k + 1; j < n; j++) {
              let t = 0;

              for (let i = k + 1; i < n; i++) {
                t += V.get(i, k) * V.get(i, j);
              }

              t = -t / V.get(k + 1, k);

              for (let i = k + 1; i < n; i++) {
                V.set(i, j, V.get(i, j) + t * V.get(i, k));
              }
            }
          }

          for (let i = 0; i < n; i++) {
            V.set(i, k, 0);
          }

          V.set(k, k, 1);
        }
      }

      let pp = p - 1;
      let eps = Number.EPSILON;

      while (p > 0) {
        let k, kase;

        for (k = p - 2; k >= -1; k--) {
          if (k === -1) {
            break;
          }

          const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));

          if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
            e[k] = 0;
            break;
          }
        }

        if (k === p - 2) {
          kase = 4;
        } else {
          let ks;

          for (ks = p - 1; ks >= k; ks--) {
            if (ks === k) {
              break;
            }

            let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);

            if (Math.abs(s[ks]) <= eps * t) {
              s[ks] = 0;
              break;
            }
          }

          if (ks === k) {
            kase = 3;
          } else if (ks === p - 1) {
            kase = 1;
          } else {
            kase = 2;
            k = ks;
          }
        }

        k++;

        switch (kase) {
          case 1:
            {
              let f = e[p - 2];
              e[p - 2] = 0;

              for (let j = p - 2; j >= k; j--) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;

                if (j !== k) {
                  f = -sn * e[j - 1];
                  e[j - 1] = cs * e[j - 1];
                }

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                    V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                    V.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 2:
            {
              let f = e[k - 1];
              e[k - 1] = 0;

              for (let j = k; j < p; j++) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;
                f = -sn * e[j];
                e[j] = cs * e[j];

                if (wantu) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                    U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                    U.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 3:
            {
              const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
              const sp = s[p - 1] / scale;
              const spm1 = s[p - 2] / scale;
              const epm1 = e[p - 2] / scale;
              const sk = s[k] / scale;
              const ek = e[k] / scale;
              const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
              const c = sp * epm1 * (sp * epm1);
              let shift = 0;

              if (b !== 0 || c !== 0) {
                if (b < 0) {
                  shift = 0 - Math.sqrt(b * b + c);
                } else {
                  shift = Math.sqrt(b * b + c);
                }

                shift = c / (b + shift);
              }

              let f = (sk + sp) * (sk - sp) + shift;
              let g = sk * ek;

              for (let j = k; j < p - 1; j++) {
                let t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                let cs = f / t;
                let sn = g / t;

                if (j !== k) {
                  e[j - 1] = t;
                }

                f = cs * s[j] + sn * e[j];
                e[j] = cs * e[j] - sn * s[j];
                g = sn * s[j + 1];
                s[j + 1] = cs * s[j + 1];

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                    V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                    V.set(i, j, t);
                  }
                }

                t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                cs = f / t;
                sn = g / t;
                s[j] = t;
                f = cs * e[j] + sn * s[j + 1];
                s[j + 1] = -sn * e[j] + cs * s[j + 1];
                g = sn * e[j + 1];
                e[j + 1] = cs * e[j + 1];

                if (wantu && j < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                    U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                    U.set(i, j, t);
                  }
                }
              }

              e[p - 2] = f;
              break;
            }

          case 4:
            {
              if (s[k] <= 0) {
                s[k] = s[k] < 0 ? -s[k] : 0;

                if (wantv) {
                  for (let i = 0; i <= pp; i++) {
                    V.set(i, k, -V.get(i, k));
                  }
                }
              }

              while (k < pp) {
                if (s[k] >= s[k + 1]) {
                  break;
                }

                let t = s[k];
                s[k] = s[k + 1];
                s[k + 1] = t;

                if (wantv && k < n - 1) {
                  for (let i = 0; i < n; i++) {
                    t = V.get(i, k + 1);
                    V.set(i, k + 1, V.get(i, k));
                    V.set(i, k, t);
                  }
                }

                if (wantu && k < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = U.get(i, k + 1);
                    U.set(i, k + 1, U.get(i, k));
                    U.set(i, k, t);
                  }
                }

                k++;
              }
              p--;
              break;
            }
          // no default
        }
      }

      if (swapped) {
        let tmp = V;
        V = U;
        U = tmp;
      }

      this.m = m;
      this.n = n;
      this.s = s;
      this.U = U;
      this.V = V;
    }

    solve(value) {
      let Y = value;
      let e = this.threshold;
      let scols = this.s.length;
      let Ls = Matrix.zeros(scols, scols);

      for (let i = 0; i < scols; i++) {
        if (Math.abs(this.s[i]) <= e) {
          Ls.set(i, i, 0);
        } else {
          Ls.set(i, i, 1 / this.s[i]);
        }
      }

      let U = this.U;
      let V = this.rightSingularVectors;
      let VL = V.mmul(Ls);
      let vrows = V.rows;
      let urows = U.rows;
      let VLU = Matrix.zeros(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < scols; k++) {
            sum += VL.get(i, k) * U.get(j, k);
          }

          VLU.set(i, j, sum);
        }
      }

      return VLU.mmul(Y);
    }

    solveForDiagonal(value) {
      return this.solve(Matrix.diag(value));
    }

    inverse() {
      let V = this.V;
      let e = this.threshold;
      let vrows = V.rows;
      let vcols = V.columns;
      let X = new Matrix(vrows, this.s.length);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < vcols; j++) {
          if (Math.abs(this.s[j]) > e) {
            X.set(i, j, V.get(i, j) / this.s[j]);
          }
        }
      }

      let U = this.U;
      let urows = U.rows;
      let ucols = U.columns;
      let Y = new Matrix(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < ucols; k++) {
            sum += X.get(i, k) * U.get(j, k);
          }

          Y.set(i, j, sum);
        }
      }

      return Y;
    }

    get condition() {
      return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
    }

    get norm2() {
      return this.s[0];
    }

    get rank() {
      let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
      let r = 0;
      let s = this.s;

      for (let i = 0, ii = s.length; i < ii; i++) {
        if (s[i] > tol) {
          r++;
        }
      }

      return r;
    }

    get diagonal() {
      return Array.from(this.s);
    }

    get threshold() {
      return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
    }

    get leftSingularVectors() {
      return this.U;
    }

    get rightSingularVectors() {
      return this.V;
    }

    get diagonalMatrix() {
      return Matrix.diag(this.s);
    }

  }

  function inverse(matrix, useSVD = false) {
    matrix = WrapperMatrix2D.checkMatrix(matrix);

    if (useSVD) {
      return new SingularValueDecomposition(matrix).inverse();
    } else {
      return solve(matrix, Matrix.eye(matrix.rows));
    }
  }
  function solve(leftHandSide, rightHandSide, useSVD = false) {
    leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
    rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);

    if (useSVD) {
      return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
    } else {
      return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
    }
  }

  /**
   * Difference of the matrix function over the parameters
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} evaluatedData - Array of previous evaluated function values
   * @param {Array<number>} params - Array of previous parameter values
   * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
   * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {function} paramFunction - The parameters and returns a function with the independent variable as a parameter
   * @return {Matrix}
   */

  function gradientFunction(data, evaluatedData, params, gradientDifference, paramFunction, centralDifference) {
    const nbParams = params.length;
    const nbPoints = data.x.length;
    let ans = Matrix.zeros(nbParams, nbPoints);
    let rowIndex = 0;

    for (let param = 0; param < nbParams; param++) {
      if (gradientDifference[param] === 0) continue;
      let delta = gradientDifference[param];
      let auxParams = params.slice();
      auxParams[param] += delta;
      let funcParam = paramFunction(auxParams);

      if (!centralDifference) {
        for (let point = 0; point < nbPoints; point++) {
          ans.set(rowIndex, point, (evaluatedData[point] - funcParam(data.x[point])) / delta);
        }
      } else {
        auxParams = params.slice();
        auxParams[param] -= delta;
        delta *= 2;
        let funcParam2 = paramFunction(auxParams);

        for (let point = 0; point < nbPoints; point++) {
          ans.set(rowIndex, point, (funcParam2(data.x[point]) - funcParam(data.x[point])) / delta);
        }
      }

      rowIndex++;
    }

    return ans;
  }

  /**
   * Matrix function over the samples
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} evaluatedData - Array of previous evaluated function values
   * @return {Matrix}
   */

  function matrixFunction(data, evaluatedData) {
    const m = data.x.length;
    let ans = new Matrix(m, 1);

    for (let point = 0; point < m; point++) {
      ans.set(point, 0, data.y[point] - evaluatedData[point]);
    }

    return ans;
  }
  /**
   * Iteration for Levenberg-Marquardt
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} params - Array of previous parameter values
   * @param {number} damping - Levenberg-Marquardt parameter
   * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
   * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @return {Array<number>}
   */


  function step(data, params, damping, gradientDifference, parameterizedFunction, centralDifference, weights) {
    let value = damping;
    let identity = Matrix.eye(params.length, params.length, value);
    const func = parameterizedFunction(params);
    let evaluatedData = new Float64Array(data.x.length);

    for (let i = 0; i < data.x.length; i++) {
      evaluatedData[i] = func(data.x[i]);
    }

    let gradientFunc = gradientFunction(data, evaluatedData, params, gradientDifference, parameterizedFunction, centralDifference);
    let residualError = matrixFunction(data, evaluatedData);
    let inverseMatrix = inverse(identity.add(gradientFunc.mmul(gradientFunc.transpose().scale('row', {
      scale: weights
    }))));
    let jacobianWeigthResidualError = gradientFunc.mmul(residualError.scale('row', {
      scale: weights
    }));
    let perturbations = inverseMatrix.mmul(jacobianWeigthResidualError);
    return {
      perturbations,
      jacobianWeigthResidualError
    };
  }

  /**
   * Curve fitting algorithm
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @param {object} [options] - Options object
   * @param {number|array} [options.weights = 1] - weighting vector, if the length does not match with the number of data points, the vector is reconstructed with first value.
   * @param {number} [options.damping = 1e-2] - Levenberg-Marquardt parameter, small values of the damping parameter λ result in a Gauss-Newton update and large
  values of λ result in a gradient descent update
   * @param {number} [options.dampingStepDown = 9] - factor to reduce the damping (Levenberg-Marquardt parameter) when there is not an improvement when updating parameters.
   * @param {number} [options.dampingStepUp = 11] - factor to increase the damping (Levenberg-Marquardt parameter) when there is an improvement when updating parameters.
   * @param {number} [options.improvementThreshold = 1e-3] - the threshold to define an improvement through an update of parameters
   * @param {number|array} [options.gradientDifference = 10e-2] - The step size to approximate the jacobian matrix
   * @param {boolean} [options.centralDifference = false] - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {Array<number>} [options.minValues] - Minimum allowed values for parameters
   * @param {Array<number>} [options.maxValues] - Maximum allowed values for parameters
   * @param {Array<number>} [options.initialValues] - Array of initial parameter values
   * @param {number} [options.maxIterations = 100] - Maximum of allowed iterations
   * @param {number} [options.errorTolerance = 10e-3] - Minimum uncertainty allowed for each point.
   * @param {number} [options.timeout] - maximum time running before throw in seconds.
   * @return {{parameterValues: Array<number>, parameterError: number, iterations: number}}
   */

  function levenbergMarquardt(data, parameterizedFunction, options = {}) {
    let {
      checkTimeout,
      minValues,
      maxValues,
      parameters,
      weightSquare,
      damping,
      dampingStepUp,
      dampingStepDown,
      maxIterations,
      errorTolerance,
      centralDifference,
      gradientDifference,
      improvementThreshold
    } = checkOptions$1(data, parameterizedFunction, options);
    let error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
    let converged = error <= errorTolerance;
    let iteration = 0;

    for (; iteration < maxIterations && !converged; iteration++) {
      let previousError = error;
      let {
        perturbations,
        jacobianWeigthResidualError
      } = step(data, parameters, damping, gradientDifference, parameterizedFunction, centralDifference, weightSquare);

      for (let k = 0; k < parameters.length; k++) {
        parameters[k] = Math.min(Math.max(minValues[k], parameters[k] - perturbations.get(k, 0)), maxValues[k]);
      }

      error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
      if (isNaN(error)) break;
      let improvementMetric = (previousError - error) / perturbations.transpose().mmul(perturbations.mulS(damping).add(jacobianWeigthResidualError)).get(0, 0);

      if (improvementMetric > improvementThreshold) {
        damping = Math.max(damping / dampingStepDown, 1e-7);
      } else {
        error = previousError;
        damping = Math.min(damping * dampingStepUp, 1e7);
      }

      if (checkTimeout()) {
        throw new Error(`The execution time is over to ${options.timeout} seconds`);
      }

      converged = error <= errorTolerance;
    }

    return {
      parameterValues: parameters,
      parameterError: error,
      iterations: iteration
    };
  }

  const LEVENBERG_MARQUARDT = 1;
  function selectMethod(optimizationOptions = {}) {
    let {
      kind,
      options
    } = optimizationOptions;
    kind = getKind(kind);

    switch (kind) {
      case LEVENBERG_MARQUARDT:
        return {
          algorithm: levenbergMarquardt,
          optimizationOptions: checkOptions(kind, options)
        };

      default:
        throw new Error(`Unknown kind algorithm`);
    }
  }

  function checkOptions(kind, options = {}) {
    // eslint-disable-next-line default-case
    switch (kind) {
      case LEVENBERG_MARQUARDT:
        return Object.assign({}, lmOptions, options);
    }
  }

  function getKind(kind) {
    if (typeof kind !== 'string') return kind;

    switch (kind.toLowerCase().replace(/[^a-z]/g, '')) {
      case 'lm':
      case 'levenbergmarquardt':
        return LEVENBERG_MARQUARDT;

      default:
        throw new Error(`Unknown kind algorithm`);
    }
  }

  const lmOptions = {
    damping: 1.5,
    maxIterations: 100,
    errorTolerance: 1e-8
  };

  // const STATE_MIN = 1;
  // const STATE_MAX = 2;
  // const STATE_GRADIENT_DIFFERENCE = 3;
  // const X = 0;
  // const Y = 1;
  // const WIDTH = 2;
  // const MU = 3;
  // const keys = ['x', 'y', 'width', 'mu'];

  /**
   * Fits a set of points to the sum of a set of bell functions.
   * @param {object} data - An object containing the x and y data to be fitted.
   * @param {array} peaks - A list of initial parameters to be optimized. e.g. coming from a peak picking [{x, y, width}].
   * @param {object} [options = {}]
   * @param {object} [options.shape={}] - it's specify the kind of shape used to fitting.
   * @param {string} [options.shape.kind = 'gaussian'] - kind of shape; lorentzian, gaussian and pseudovoigt are supported.
   * @param {object} [options.optimization = {}] - it's specify the kind and options of the algorithm use to optimize parameters.
   * @param {object} [options.optimization.kind = 'lm'] - kind of algorithm. By default it's levenberg-marquardt.
   * @param {object} [options.optimization.parameters] - options of each parameter to be optimized e.g. For a gaussian shape
   *  it could have x, y and with properties, each of which could contain init, min, max and gradientDifference, those options will define the guess,
   *  the min and max value of the parameter (search space) and the step size to approximate the jacobian matrix respectively. Those options could be a number,
   *  array of numbers, callback, or array of callbacks. Each kind of shape has default parameters so it could be undefined.
   * @param {object} [options.optimization.parameters.x] - options for x parameter.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.init] - definition of the starting point of the parameter (the guess),
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the guess of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.min] - definition of the lower limit of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the min of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.max] - definition of the upper limit of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the max of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.gradientDifference] - definition of  the step size to approximate the jacobian matrix of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the gradientDifference of the first peak and so on.
   * @param {object} [options.optimization.options = {}] - options for the specific kind of algorithm.
   * @param {number} [options.optimization.options.timeout] - maximum time running before break in seconds.
   * @param {number} [options.optimization.options.damping=1.5]
   * @param {number} [options.optimization.options.maxIterations=100]
   * @param {number} [options.optimization.options.errorTolerance=1e-8]
   * @returns {object} - A object with fitting error and the list of optimized parameters { parameters: [ {x, y, width} ], error } if the kind of shape is pseudoVoigt mu parameter is optimized.
   */

  function optimize(data, peakList, options = {}) {
    const {
      y,
      x,
      maxY,
      peaks,
      paramsFunc,
      optimization
    } = checkInput(data, peakList, options);
    let parameters = optimization.parameters;
    let nbShapes = peaks.length;
    let parameterKey = Object.keys(parameters);
    let nbParams = nbShapes * parameterKey.length;
    let pMin = new Float64Array(nbParams);
    let pMax = new Float64Array(nbParams);
    let pInit = new Float64Array(nbParams);
    let gradientDifference = new Float64Array(nbParams);

    for (let i = 0; i < nbShapes; i++) {
      let peak = peaks[i];

      for (let k = 0; k < parameterKey.length; k++) {
        let key = parameterKey[k];
        let init = parameters[key].init;
        let min = parameters[key].min;
        let max = parameters[key].max;
        let gradientDifferenceValue = parameters[key].gradientDifference;
        pInit[i + k * nbShapes] = init[i % init.length](peak);
        pMin[i + k * nbShapes] = min[i % min.length](peak);
        pMax[i + k * nbShapes] = max[i % max.length](peak);
        gradientDifference[i + k * nbShapes] = gradientDifferenceValue[i % gradientDifferenceValue.length](peak);
      }
    }

    let {
      algorithm,
      optimizationOptions
    } = selectMethod(optimization);
    optimizationOptions.minValues = pMin;
    optimizationOptions.maxValues = pMax;
    optimizationOptions.initialValues = pInit;
    optimizationOptions.gradientDifference = gradientDifference;
    let pFit = algorithm({
      x,
      y
    }, paramsFunc, optimizationOptions);
    let {
      parameterError: error,
      iterations
    } = pFit;
    let result = {
      error,
      iterations,
      peaks
    };

    for (let i = 0; i < nbShapes; i++) {
      pFit.parameterValues[i + nbShapes] *= maxY;

      for (let k = 0; k < parameterKey.length; k++) {
        // we modify the optimized parameters
        peaks[i][parameterKey[k]] = pFit.parameterValues[i + k * nbShapes];
      }
    }

    return result;
  }

  function addStyle(serie, spectrum, options = {}) {
    let {
      color = '#A9A9A9',
      opacity = 1,
      lineWidth = 1
    } = options; // eslint-disable-next-line @typescript-eslint/prefer-regexp-exec

    if (color.match(/#[0-9A-F]{6}$/i)) {
      color = (color + (opacity * 255 >> 0).toString(16)).toUpperCase();
    } else {
      color = color.replace(/rgb ?\((.*)\)/, `rgba($1,${opacity})`);
    }

    serie.style = [{
      name: 'unselected',
      style: {
        line: {
          color,
          width: lineWidth,
          dash: 1
        }
      }
    }, {
      name: 'selected',
      style: {
        line: {
          color,
          width: lineWidth + 2,
          dash: 1
        }
      }
    }];
    serie.name = spectrum.label || spectrum.id;
  }

  const COLORS = ['#FFB300', '#803E75', '#FF6800', '#A6BDD7', '#C10020', '#CEA262', '#817066', '#007D34', '#F6768E', '#00538A', '#FF7A5C', '#53377A', '#FF8E00', '#B32851', '#F4C800', '#7F180D', '#93AA00', '#593315', '#F13A13', '#232C16'];

  /**
   * Generate a jsgraph chart format from an array of Analysis
   */

  function getJSGraph(analyses, options = {}) {
    const {
      colors = COLORS,
      opacities = [1],
      linesWidth = [1],
      selector = {},
      normalization,
      xAxis = {},
      yAxis = {}
    } = options;
    let series = [];
    let xLabel = '';
    let yLabel = '';

    for (let i = 0; i < analyses.length; i++) {
      const analysis = analyses[i];
      let serie = {};
      let currentData = analysis.getNormalizedSpectrum({
        selector,
        normalization
      });
      if (!currentData) continue;
      if (!xLabel) xLabel = currentData.variables.x.label;
      if (!yLabel) yLabel = currentData.variables.y.label;
      addStyle(serie, analysis, {
        color: colors[i % colors.length],
        opacity: opacities[i % opacities.length],
        lineWidth: linesWidth[i % linesWidth.length]
      });
      serie.data = {
        x: currentData.variables.x.data,
        y: currentData.variables.y.data
      };

      if (xAxis.logScale) {
        serie.data = xyFilterXPositive(serie.data);
      }

      series.push(serie);
    }

    return {
      axes: {
        x: {
          label: xLabel,
          unit: '',
          flipped: false,
          display: true,
          ...xAxis
        },
        y: {
          label: yLabel,
          unit: '',
          flipped: false,
          display: true,
          ...yAxis
        }
      },
      series
    };
  }

  function getNormalizationAnnotations(filter = {}, boundary = {
    y: {
      min: '0px',
      max: '2000px'
    }
  }) {
    let {
      exclusions = []
    } = filter;
    let annotations = [];
    exclusions = exclusions.filter(exclusion => !exclusion.ignore);
    annotations = exclusions.map(exclusion => {
      let annotation = {
        type: 'rect',
        position: [{
          x: exclusion.from,
          y: boundary.y.min
        }, {
          x: exclusion.to,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      };
      return annotation;
    });

    if (filter.from !== undefined) {
      annotations.push({
        type: 'rect',
        position: [{
          x: Number.MIN_SAFE_INTEGER,
          y: boundary.y.min
        }, {
          x: filter.from,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      });
    }

    if (filter.to !== undefined) {
      annotations.push({
        type: 'rect',
        position: [{
          x: filter.to,
          y: boundary.y.min
        }, {
          x: Number.MAX_SAFE_INTEGER,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      });
    }

    return annotations;
  }

  function appendDistinctParameter(values, key, value) {
    if (!values[key]) {
      values[key] = {
        key,
        values: [],
        count: 0
      };
    }

    if (!values[key].values.includes(value)) {
      values[key].values.push(value);
    }

    values[key].count++;
  }

  function appendDistinctValue(values, key) {
    if (!values[key]) {
      values[key] = {
        key,
        count: 0
      };
    }

    values[key].count++;
  }

  class AnalysesManager {
    constructor() {
      this.analyses = [];
    }

    addAnalysis(analysis) {
      let index = this.getAnalysisIndex(analysis.id);

      if (index === undefined) {
        this.analyses.push(analysis);
      } else {
        this.analyses[index] = analysis;
      }
    }

    getAnalyses(options = {}) {
      const {
        ids
      } = options;
      let analyses = [];

      for (const analysis of this.analyses) {
        if (!ids || ids.includes(analysis.id)) {
          analyses.push(analysis);
        }
      }

      return analyses;
    }

    getSpectra() {
      const spectra = [];

      for (const analysis of this.analyses) {
        spectra.push(...analysis.spectra);
      }

      return spectra;
    }
    /**
     * Get an array of objects (key + count) of all the titles
     */


    getDistinctTitles() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.title) {
          appendDistinctValue(values, spectrum.title);
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the units
     */


    getDistinctUnits() {
      var _a;

      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.variables) {
          for (let key in spectrum.variables) {
            const units = (_a = spectrum.variables[key].units) === null || _a === void 0 ? void 0 : _a.replace(/\s+\[.*/, '');

            if (units) {
              appendDistinctValue(values, units);
            }
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the labels
     */


    getDistinctLabels() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.variables) {
          for (let key in spectrum.variables) {
            appendDistinctValue(values, spectrum.variables[key].label.replace(/\s+\[.*/, ''));
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the dataTypes
     */


    getDistinctDataTypes() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.dataType) {
          appendDistinctValue(values, spectrum.dataType);
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the meta
     */


    getDistinctMeta() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.meta) {
          for (let key in spectrum.meta) {
            appendDistinctParameter(values, key, spectrum.meta[key]);
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }

    removeAllAnalyses() {
      this.analyses.splice(0);
    }
    /**
     * Remove the analysis from the AnalysesManager for the specified id
     */


    removeAnalysis(id) {
      let index = this.getAnalysisIndex(id);
      if (index === undefined) return undefined;
      return this.analyses.splice(index, 1);
    }
    /**
     * Returns the index of the analysis in the analyses array
     */


    getAnalysisIndex(id) {
      if (!id) return undefined;

      for (let i = 0; i < this.analyses.length; i++) {
        let analysis = this.analyses[i];
        if (analysis.id === id) return i;
      }

      return undefined;
    }
    /**
     * Checks if the ID of an analysis exists in the AnalysesManager
     */


    includes(id) {
      const index = this.getAnalysisIndex(id);
      return index === undefined ? false : !isNaN(index);
    }

  }

  // Based on https://github.com/scijs/cholesky-solve

  /*
  The MIT License (MIT)

  Copyright (c) 2013 Eric Arnebäck

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
  */
  function ldlSymbolic(n
  /* A and L are n-by-n, where n >= 0 */
  , Ap
  /* input of size n + 1, not modified */
  , Ai
  /* input of size nz=Ap[n], not modified */
  , Lp
  /* output of size n + 1, not defined on input */
  , Parent
  /* output of size n, not defined on input */
  , Lnz
  /* output of size n, not defined on input */
  , Flag
  /* workspace of size n, not defn. on input or output */
  ) {
    let i, k, p, kk, p2;

    for (k = 0; k < n; k++) {
      /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
      Parent[k] = -1;
      /* parent of k is not yet known */

      Flag[k] = k;
      /* mark node k as visited */

      Lnz[k] = 0;
      /* count of nonzeros in column k of L */

      kk = k;
      /* kth original, or permuted, column */

      p2 = Ap[kk + 1];

      for (p = Ap[kk]; p < p2; p++) {
        /* A (i,k) is nonzero (original or permuted A) */
        i = Ai[p];

        if (i < k) {
          /* follow path from i to root of etree, stop at flagged node */
          for (; Flag[i] !== k; i = Parent[i]) {
            /* find parent of i if not yet determined */
            if (Parent[i] === -1) Parent[i] = k;
            Lnz[i]++;
            /* L (k,i) is nonzero */

            Flag[i] = k;
            /* mark i as visited */
          }
        }
      }
    }
    /* construct Lp index array from Lnz column counts */


    Lp[0] = 0;

    for (k = 0; k < n; k++) {
      Lp[k + 1] = Lp[k] + Lnz[k];
    }
  }

  function ldlNumeric(n
  /* A and L are n-by-n, where n >= 0 */
  , Ap
  /* input of size n+1, not modified */
  , Ai
  /* input of size nz=Ap[n], not modified */
  , Ax
  /* input of size nz=Ap[n], not modified */
  , Lp
  /* input of size n+1, not modified */
  , Parent
  /* input of size n, not modified */
  , Lnz
  /* output of size n, not defn. on input */
  , Li
  /* output of size lnz=Lp[n], not defined on input */
  , Lx
  /* output of size lnz=Lp[n], not defined on input */
  , D
  /* output of size n, not defined on input */
  , Y
  /* workspace of size n, not defn. on input or output */
  , Pattern
  /* workspace of size n, not defn. on input or output */
  , Flag
  /* workspace of size n, not defn. on input or output */
  ) {
    let yi, lKi;
    let i, k, p, kk, p2, len, top;

    for (k = 0; k < n; k++) {
      /* compute nonzero Pattern of kth row of L, in topological order */
      Y[k] = 0.0;
      /* Y(0:k) is now all zero */

      top = n;
      /* stack for pattern is empty */

      Flag[k] = k;
      /* mark node k as visited */

      Lnz[k] = 0;
      /* count of nonzeros in column k of L */

      kk = k;
      /* kth original, or permuted, column */

      p2 = Ap[kk + 1];

      for (p = Ap[kk]; p < p2; p++) {
        i = Ai[p];
        /* get A(i,k) */

        if (i <= k) {
          Y[i] += Ax[p];
          /* scatter A(i,k) into Y (sum duplicates) */

          for (len = 0; Flag[i] !== k; i = Parent[i]) {
            Pattern[len++] = i;
            /* L(k,i) is nonzero */

            Flag[i] = k;
            /* mark i as visited */
          }

          while (len > 0) Pattern[--top] = Pattern[--len];
        }
      }
      /* compute numerical values kth row of L (a sparse triangular solve) */


      D[k] = Y[k];
      /* get D(k,k) and clear Y(k) */

      Y[k] = 0.0;

      for (; top < n; top++) {
        i = Pattern[top];
        /* Pattern[top:n-1] is pattern of L(:,k) */

        yi = Y[i];
        /* get and clear Y(i) */

        Y[i] = 0.0;
        p2 = Lp[i] + Lnz[i];

        for (p = Lp[i]; p < p2; p++) {
          Y[Li[p]] -= Lx[p] * yi;
        }

        lKi = yi / D[i];
        /* the nonzero entry L(k,i) */

        D[k] -= lKi * yi;
        Li[p] = k;
        /* store L(k,i) in column form of L */

        Lx[p] = lKi;
        Lnz[i]++;
        /* increment count of nonzeros in col i */
      }

      if (D[k] === 0.0) return k;
      /* failure, D(k,k) is zero */
    }

    return n;
    /* success, diagonal of D is all nonzero */
  }

  function ldlLsolve(n
  /* L is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , Lp
  /* input of size n+1, not modified */
  , Li
  /* input of size lnz=Lp[n], not modified */
  , Lx
  /* input of size lnz=Lp[n], not modified */
  ) {
    let j, p, p2;

    for (j = 0; j < n; j++) {
      p2 = Lp[j + 1];

      for (p = Lp[j]; p < p2; p++) {
        X[Li[p]] -= Lx[p] * X[j];
      }
    }
  }

  function ldlDsolve(n
  /* D is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , D
  /* input of size n, not modified */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[j] /= D[j];
    }
  }

  function ldlLTsolve(n
  /* L is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , Lp
  /* input of size n+1, not modified */
  , Li
  /* input of size lnz=Lp[n], not modified */
  , Lx
  /* input of size lnz=Lp[n], not modified */
  ) {
    let j, p, p2;

    for (j = n - 1; j >= 0; j--) {
      p2 = Lp[j + 1];

      for (p = Lp[j]; p < p2; p++) {
        X[j] -= Lx[p] * X[Li[p]];
      }
    }
  }

  function ldlPerm(n
  /* size of X, B, and P */
  , X
  /* output of size n. */
  , B
  /* input of size n. */
  , P
  /* input permutation array of size n. */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[j] = B[P[j]];
    }
  }

  function ldlPermt(n
  /* size of X, B, and P */
  , X
  /* output of size n. */
  , B
  /* input of size n. */
  , P
  /* input permutation array of size n. */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[P[j]] = B[j];
    }
  }

  function prepare(M, n, P) {
    // if a permutation was specified, apply it.
    if (P) {
      let Pinv = new Array(n);

      for (let k = 0; k < n; k++) {
        Pinv[P[k]] = k;
      }

      let Mt = []; // scratch memory
      // Apply permutation. We make M into P*M*P^T

      for (let a = 0; a < M.length; ++a) {
        let ar = Pinv[M[a][0]];
        let ac = Pinv[M[a][1]]; // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
        // if permuted element is below diagonal, we simply transpose it.

        if (ac < ar) {
          let t = ac;
          ac = ar;
          ar = t;
        }

        Mt[a] = [];
        Mt[a][0] = ar;
        Mt[a][1] = ac;
        Mt[a][2] = M[a][2];
      }

      M = Mt; // copy scratch memory.
    } else {
      // if P argument is null, we just use an identity permutation.
      P = [];

      for (let i = 0; i < n; ++i) {
        P[i] = i;
      }
    } // The sparse matrix we are decomposing is A.
    // Now we shall create A from M.


    let Ap = new Array(n + 1);
    let Ai = new Array(M.length);
    let Ax = new Array(M.length); // count number of non-zero elements in columns.

    let LNZ = [];

    for (let i = 0; i < n; ++i) {
      LNZ[i] = 0;
    }

    for (let a = 0; a < M.length; ++a) {
      LNZ[M[a][1]]++;
    }

    Ap[0] = 0;

    for (let i = 0; i < n; ++i) {
      Ap[i + 1] = Ap[i] + LNZ[i];
    }

    let coloffset = [];

    for (let a = 0; a < n; ++a) {
      coloffset[a] = 0;
    } // go through all elements in M, and add them to sparse matrix A.


    for (let i = 0; i < M.length; ++i) {
      let e = M[i];
      let col = e[1];
      let adr = Ap[col] + coloffset[col];
      Ai[adr] = e[0];
      Ax[adr] = e[2];
      coloffset[col]++;
    }

    let D = new Array(n);
    let Y = new Array(n);
    let Lp = new Array(n + 1);
    let Parent = new Array(n);
    let Lnz = new Array(n);
    let Flag = new Array(n);
    let Pattern = new Array(n);
    let bp1 = new Array(n);
    let x = new Array(n);
    let d;
    ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
    let Lx = new Array(Lp[n]);
    let Li = new Array(Lp[n]);
    d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);

    if (d === n) {
      return function (b) {
        ldlPerm(n, bp1, b, P);
        ldlLsolve(n, bp1, Lp, Li, Lx);
        ldlDsolve(n, bp1, D);
        ldlLTsolve(n, bp1, Lp, Li, Lx);
        ldlPermt(n, x, bp1, P);
        return x;
      };
    } else {
      return null;
    }
  }

  var cuthillMckee_1 = cuthillMckee;

  function compareNum(a, b) {
    return a - b;
  }

  function cuthillMckee(list, n) {
    var adj = new Array(n);
    var visited = new Array(n);

    for (var i = 0; i < n; ++i) {
      adj[i] = [];
      visited[i] = false;
    }

    for (var i = 0; i < list.length; ++i) {
      var l = list[i];
      adj[l[0]].push(l[1]);
    }

    var toVisit = new Array(n);
    var eol = 0;
    var ptr = 0;

    for (var i = 0; i < n; ++i) {
      if (visited[i]) {
        continue;
      }

      toVisit[eol++] = i;
      visited[i] = true;

      while (ptr < eol) {
        var v = toVisit[ptr++];
        var nbhd = adj[v];
        nbhd.sort(compareNum);

        for (var j = 0; j < nbhd.length; ++j) {
          var u = nbhd[j];

          if (visited[u]) {
            continue;
          }

          visited[u] = true;
          toVisit[eol++] = u;
        }
      }
    }

    var result = new Array(n);

    for (var i = 0; i < n; ++i) {
      result[toVisit[i]] = i;
    }

    return result;
  }

  var cuthillMckee$1 = cuthillMckee_1;

  const getClosestNumber = (array = [], goal = 0) => {
    const closest = array.reduce((prev, curr) => {
      return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
    });
    return closest;
  };

  const getCloseIndex = (array = [], goal = 0) => {
    const closest = getClosestNumber(array, goal);
    return array.indexOf(closest);
  };

  const updateSystem = (matrix, y, weights) => {
    let nbPoints = y.length;
    let l = nbPoints - 1;
    let newMatrix = new Array(matrix.length);
    let newVector = new Float64Array(nbPoints);

    for (let i = 0; i < l; i++) {
      let w = weights[i];
      let diag = i * 2;
      let next = diag + 1;
      newMatrix[diag] = matrix[diag].slice();
      newMatrix[next] = matrix[next].slice();

      if (w === 0) {
        newVector[i] = 0;
      } else {
        newVector[i] = y[i] * w;
        newMatrix[diag][2] += w;
      }
    }

    newVector[l] = y[l] * weights[l];
    newMatrix[l * 2] = matrix[l * 2].slice();
    newMatrix[l * 2][2] += weights[l];
    return [newMatrix, newVector];
  };

  const getDeltaMatrix = (nbPoints, lambda) => {
    let matrix = [];
    let last = nbPoints - 1;

    for (let i = 0; i < last; i++) {
      matrix.push([i, i, lambda * 2]);
      matrix.push([i + 1, i, -1 * lambda]);
    }

    matrix[0][2] = lambda;
    matrix.push([last, last, lambda]);
    return {
      lowerTriangularNonZeros: matrix,
      permutationEncodedArray: cuthillMckee$1(matrix, nbPoints)
    };
  };

  /**
   * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
   * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/master/airPLS_manuscript.pdf)
   * @param {Array<number>} x - x axis data useful when control points or zones are submitted
   * @param {Array<number>} y - Original data
   * @param {object} [options={}] - Options object
   * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
   * @param {number} [options.factorCriterion = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
   * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
   * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
   * @param {Array<number>} [options.controlPoints = []] - Array of x axis values to force that baseline cross those points.
   * @param {Array<number>} [options.baseLineZones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
   * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
   */

  function airPLS(x, y, options = {}) {
    let {
      maxIterations = 100,
      lambda = 100,
      factorCriterion = 0.001,
      weights = new Array(y.length).fill(1),
      controlPoints = [],
      baseLineZones = []
    } = options;

    if (controlPoints.length > 0) {
      controlPoints.forEach((e, i, arr) => arr[i] = getCloseIndex(x, e));
    }

    if (baseLineZones.length > 0) {
      baseLineZones.forEach(range => {
        let indexFrom = getCloseIndex(x, range.from);
        let indexTo = getCloseIndex(x, range.to);
        if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];

        for (let i = indexFrom; i < indexTo; i++) {
          controlPoints.push(i);
        }
      });
    }

    let baseline, iteration;
    let nbPoints = y.length;
    let l = nbPoints - 1;
    let sumNegDifferences = Number.MAX_SAFE_INTEGER;
    let stopCriterion = factorCriterion * y.reduce((sum, e) => Math.abs(e) + sum, 0);
    let {
      lowerTriangularNonZeros,
      permutationEncodedArray
    } = getDeltaMatrix(nbPoints, lambda);

    for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
      let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
      let cho = prepare(leftHandSide, nbPoints, permutationEncodedArray);
      baseline = cho(rightHandSide);
      sumNegDifferences = 0;
      let difference = y.map(calculateError);
      let maxNegativeDiff = -1 * Number.MAX_SAFE_INTEGER;

      for (let i = 1; i < l; i++) {
        let diff = difference[i];

        if (diff >= 0) {
          weights[i] = 0;
        } else {
          weights[i] = Math.exp(iteration * diff / sumNegDifferences);
          if (maxNegativeDiff < diff) maxNegativeDiff = diff;
        }
      }

      let value = Math.exp(iteration * maxNegativeDiff / sumNegDifferences);
      weights[0] = value;
      weights[l] = value;
      controlPoints.forEach(i => weights[i] = value);
    }

    return {
      corrected: y.map((e, i) => e - baseline[i]),
      baseline,
      iteration,
      error: sumNegDifferences
    };

    function calculateError(e, i) {
      let diff = e - baseline[i];
      if (diff < 0) sumNegDifferences += diff;
      return diff;
    }
  }

  /**
   * Adaptive iteratively reweighted penalized least squares [1]
   *
   * This function calls ml-airpls
   *
   * References:
   * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
   * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
   * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
   * @export
   * @param {Array<number>} ys
   * @param {object} [options] - Options object
   * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
   * @param {object} [options.regression] - Options for the regression
   * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
   * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
   * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
   * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
   * @returns {BaselineOutput}
   */

  function airPLSBaseline(ys, options = {}) {
    const numberPoints = ys.length;
    let {
      x,
      regressionOptions
    } = options;

    if (!x) {
      x = sequentialFill({
        from: 0,
        to: numberPoints - 1,
        size: numberPoints
      });
    }

    let output = airPLS(x, ys, regressionOptions);
    return {
      baseline: output.baseline,
      correctedSpectrum: output.corrected
    };
  }

  function maybeToPrecision(value, digits) {
    if (value < 0) {
      value = 0 - value;

      if (typeof digits === 'number') {
        return `- ${value.toPrecision(digits)}`;
      } else {
        return `- ${value.toString()}`;
      }
    } else {
      if (typeof digits === 'number') {
        return value.toPrecision(digits);
      } else {
        return value.toString();
      }
    }
  }

  function checkArraySize(x, y) {
    if (!isAnyArray(x) || !isAnyArray(y)) {
      throw new TypeError('x and y must be arrays');
    }

    if (x.length !== y.length) {
      throw new RangeError('x and y arrays must have the same length');
    }
  }

  class BaseRegression {
    constructor() {
      if (new.target === BaseRegression) {
        throw new Error('BaseRegression must be subclassed');
      }
    }

    predict(x) {
      if (typeof x === 'number') {
        return this._predict(x);
      } else if (isAnyArray(x)) {
        const y = [];

        for (let i = 0; i < x.length; i++) {
          y.push(this._predict(x[i]));
        }

        return y;
      } else {
        throw new TypeError('x must be a number or array');
      }
    }

    _predict() {
      throw new Error('_predict must be implemented');
    }

    train() {// Do nothing for this package
    }

    toString() {
      return '';
    }

    toLaTeX() {
      return '';
    }
    /**
     * Return the correlation coefficient of determination (r) and chi-square.
     * @param {Array<number>} x
     * @param {Array<number>} y
     * @return {object}
     */


    score(x, y) {
      if (!isAnyArray(x) || !isAnyArray(y) || x.length !== y.length) {
        throw new Error('x and y must be arrays of the same length');
      }

      const n = x.length;
      const y2 = new Array(n);

      for (let i = 0; i < n; i++) {
        y2[i] = this._predict(x[i]);
      }

      let xSum = 0;
      let ySum = 0;
      let chi2 = 0;
      let rmsd = 0;
      let xSquared = 0;
      let ySquared = 0;
      let xY = 0;

      for (let i = 0; i < n; i++) {
        xSum += y2[i];
        ySum += y[i];
        xSquared += y2[i] * y2[i];
        ySquared += y[i] * y[i];
        xY += y2[i] * y[i];

        if (y[i] !== 0) {
          chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
        }

        rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
      }

      const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
      return {
        r: r,
        r2: r * r,
        chi2: chi2,
        rmsd: Math.sqrt(rmsd / n)
      };
    }

  }

  class PolynomialRegression extends BaseRegression {
    constructor(x, y, degree) {
      super();

      if (x === true) {
        this.degree = y.degree;
        this.powers = y.powers;
        this.coefficients = y.coefficients;
      } else {
        checkArraySize(x, y);
        regress(this, x, y, degree);
      }
    }

    _predict(x) {
      let y = 0;

      for (let k = 0; k < this.powers.length; k++) {
        y += this.coefficients[k] * Math.pow(x, this.powers[k]);
      }

      return y;
    }

    toJSON() {
      return {
        name: 'polynomialRegression',
        degree: this.degree,
        powers: this.powers,
        coefficients: this.coefficients
      };
    }

    toString(precision) {
      return this._toFormula(precision, false);
    }

    toLaTeX(precision) {
      return this._toFormula(precision, true);
    }

    _toFormula(precision, isLaTeX) {
      let sup = '^';
      let closeSup = '';
      let times = ' * ';

      if (isLaTeX) {
        sup = '^{';
        closeSup = '}';
        times = '';
      }

      let fn = '';
      let str = '';

      for (let k = 0; k < this.coefficients.length; k++) {
        str = '';

        if (this.coefficients[k] !== 0) {
          if (this.powers[k] === 0) {
            str = maybeToPrecision(this.coefficients[k], precision);
          } else {
            if (this.powers[k] === 1) {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
            } else {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
            }
          }

          if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
            str = ` + ${str}`;
          } else if (k !== this.coefficients.length - 1) {
            str = ` ${str}`;
          }
        }

        fn = str + fn;
      }

      if (fn.charAt(0) === '+') {
        fn = fn.slice(1);
      }

      return `f(x) = ${fn}`;
    }

    static load(json) {
      if (json.name !== 'polynomialRegression') {
        throw new TypeError('not a polynomial regression model');
      }

      return new PolynomialRegression(true, json);
    }

  }

  function regress(pr, x, y, degree) {
    const n = x.length;
    let powers;

    if (Array.isArray(degree)) {
      powers = degree;
      degree = powers.length;
    } else {
      degree++;
      powers = new Array(degree);

      for (let k = 0; k < degree; k++) {
        powers[k] = k;
      }
    }

    const F = new Matrix(n, degree);
    const Y = new Matrix([y]);

    for (let k = 0; k < degree; k++) {
      for (let i = 0; i < n; i++) {
        if (powers[k] === 0) {
          F.set(i, k, 1);
        } else {
          F.set(i, k, Math.pow(x[i], powers[k]));
        }
      }
    }

    const FT = new MatrixTransposeView(F);
    const A = FT.mmul(F);
    const B = FT.mmul(new MatrixTransposeView(Y));
    pr.degree = degree - 1;
    pr.powers = powers;
    pr.coefficients = solve(A, B).to1DArray();
  }

  /**
   * Iterative regression-based baseline correction
   * @param {Array<number>} x - Independent axis variable
   * @param {Array<number>} y - Dependent axis variable
   * @param {object} [options] - Options object
   * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
   * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
   * @param {*} [options.regressionOptions] - Options for regressionFunction
   * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
   * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
   */

  function baselineCorrectionRegression(x, y, options = {}) {
    let {
      maxIterations = 100,
      Regression = PolynomialRegression,
      regressionOptions,
      tolerance = 0.001
    } = options;

    if (!regressionOptions && Regression === PolynomialRegression) {
      regressionOptions = 3;
    }

    let baseline = y.slice();
    let fitting = y.slice();
    let oldFitting = y;
    let iteration = 0;
    let delta;
    let regression;

    while (iteration < maxIterations) {
      // Calculate the fitting result
      regression = new Regression(x, baseline, regressionOptions);
      delta = 0;

      for (let i = 0; i < baseline.length; i++) {
        fitting[i] = regression.predict(x[i]);

        if (baseline[i] > fitting[i]) {
          baseline[i] = fitting[i];
        }

        delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
      } // Stop criterion


      if (delta < tolerance) {
        break;
      } else {
        oldFitting = fitting.slice();
        iteration++;
      }
    } // removes baseline


    let corrected = new Array(baseline.length);

    for (let j = 0; j < baseline.length; j++) {
      corrected[j] = y[j] - baseline[j];
    }

    return {
      corrected,
      delta,
      iteration,
      baseline,
      regression: regression
    };
  }

  /**
   * Iterative polynomial fitting [1]
   *
   * Implementation based on ml-baseline-correction-regression
   *
   * References:
   * [1] Gan, F.; Ruan, G.; Mo, J.
   * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
   *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
   * https://doi.org/10.1016/j.chemolab.2005.08.009.
   * @export
   * @param {Array<number>} ys
   * @param {object} [options] - Options object
   * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
   * @param {Object} [options.regression]
   * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
   * @param {Object} [options.regression]
   * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
   * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
   * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
   * @returns {BaselineOutput}
   */

  function iterativePolynomialBaseline(ys, options = {}) {
    const numberPoints = ys.length;
    let {
      x,
      regressionOptions
    } = options;

    if (!x) {
      x = sequentialFill({
        from: 0,
        to: numberPoints - 1,
        size: numberPoints
      });
    }

    let output = baselineCorrectionRegression(x, ys, regressionOptions);
    return {
      baseline: output.baseline,
      correctedSpectrum: output.corrected
    };
  }

  /**
   * Checks if input is valdi
   * @param {Array<number>} input

   */

  function xCheck(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
  }

  /**
   * This function pads an array
   * @param {Array} array - the array that will be padded
   * @param {object} [options={}]
   * @param {string} [options.algorithm=''] '', value, circular, duplicate
   * @param {number} [options.size=0] padding size before first element and after last element
   * @param {number} [options.value=0] value to use for padding (if algorithm='value')
   * @return {Array}
   */

  function xPadding(array, options = {}) {
    const {
      size = 0,
      value = 0,
      algorithm = ''
    } = options;
    xCheck(array);

    if (!algorithm) {
      if (array instanceof Float64Array) {
        return array.slice();
      } else {
        return Float64Array.from(array);
      }
    }

    let result = new Float64Array(array.length + size * 2);

    for (let i = 0; i < array.length; i++) {
      result[i + size] = array[i];
    }

    let fromEnd = size + array.length;
    let toEnd = 2 * size + array.length;

    switch (algorithm.toLowerCase()) {
      case 'value':
        for (let i = 0; i < size; i++) {
          result[i] = value;
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = value;
        }

        break;

      case 'duplicate':
        for (let i = 0; i < size; i++) {
          result[i] = array[0];
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = array[array.length - 1];
        }

        break;

      case 'circular':
        for (let i = 0; i < size; i++) {
          result[i] = array[(array.length - size % array.length + i) % array.length];
        }

        for (let i = 0; i < size; i++) {
          result[i + fromEnd] = array[i % array.length];
        }

        break;

      default:
        throw Error('xPadding: unknown algorithm');
    }

    return result;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {function} fct callback function that from an array returns a value.
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=0] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='value'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRolling(array, fct, options = {}) {
    xCheck(array);
    if (typeof fct !== 'function') throw Error('fct has to be a function');
    const {
      window = 5,
      padding = {}
    } = options;
    const {
      size = window - 1,
      algorithm,
      value
    } = padding;
    array = xPadding(array, {
      size,
      algorithm,
      value
    }); // ensure we get a copy and it is float64

    const newArray = [];

    for (let i = 0; i < array.length - window + 1; i++) {
      let subArray = new Float64Array(array.buffer, i * 8, window); // we will send a view to the original buffer

      newArray.push(fct(subArray));
    }

    return newArray;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingAverage(array, options = {}) {
    return xRolling(array, mean, options);
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingMedian(array, options = {}) {
    return xRolling(array, median, options);
  }

  /**

   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @returns {BaselineOutput}
   */

  function rollingAverageBaseline(ys, options = {}) {
    let window = Math.max(Math.round(ys.length * 0.1), 2);
    let defaults = {
      window: window,
      padding: {
        size: window - 1,
        algorithm: 'duplicate',
        value: 0
      }
    };
    let actualOptions = Object.assign({}, defaults, options);
    let baseline = xRollingAverage(ys, actualOptions);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  /**
   * Rolling ball baseline correction algorithm.
   * From the abstract of (1):
   * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
   * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
   *
   * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
   * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
   *
   * Reference:
   * (1) Kneen, M. A.; Annegarn, H. J.
   *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
   *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
   *     https://doi.org/10.1016/0168-583X(95)00908-6.
   * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
   *     https://cran.r-project.org/web/packages/baseline/index.html
   * @export
   * @param {Array} spectrum
   * @param {Object} [options={}]
   * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
   * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
   */

  function rollingBall(spectrum, options = {}) {
    if (!isAnyArray(spectrum)) {
      throw new Error('Spectrum must be an array');
    }

    if (spectrum.length === 0) {
      throw new TypeError('Spectrum must not be empty');
    }

    const numberPoints = spectrum.length;
    const maxima = new Float64Array(numberPoints);
    const minima = new Float64Array(numberPoints);
    const baseline = new Float64Array(numberPoints); // windowM 4 percent of spectrum length
    // windowS 8 percent of spectrum length

    const {
      windowM = Math.round(numberPoints * 0.04),
      windowS = Math.round(numberPoints * 0.08)
    } = options; // fi(1) in original paper

    for (let i = 0; i < spectrum.length; i++) {
      let windowLeft = max([0, i - windowM]);
      let windowRight = min([i + windowM + 1, spectrum.length]);
      minima[i] = min(spectrum.slice(windowLeft, windowRight));
    } // fi in original paper


    for (let i = 0; i < minima.length; i++) {
      let windowLeft = max([0, i - windowM]);
      let windowRight = min([i + windowM + 1, minima.length]);
      maxima[i] = max(minima.slice(windowLeft, windowRight));
    }

    for (let i = 0; i < minima.length; i++) {
      let windowLeft = max([0, i - windowS]);
      let windowRight = min([i + windowS + 1, maxima.length]);
      baseline[i] = mean(maxima.slice(windowLeft, windowRight));
    }

    return baseline;
  }

  /**
   * Rolling ball baseline correction algorithm.
   * From the abstract of (1):
   * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
   * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
   *
   * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
   * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
   *
   * Algorithm described in (1), but in the implementation here the window width does not change.
   *
   * Reference:
   * (1) Kneen, M. A.; Annegarn, H. J.
   *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
   *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
   *     https://doi.org/10.1016/0168-583X(95)00908-6.
   * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
   *     https://cran.r-project.org/web/packages/baseline/index.html
   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
   * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
   * @returns {BaselineOutput}
   */

  function rollingBallBaseline(ys, options = {}) {
    const baseline = rollingBall(ys, options);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  /**

   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @returns {BaselineOutput}
   */

  function rollingMedianBaseline(ys, options = {}) {
    let window = Math.max(Math.round(ys.length * 0.1), 2);
    let defaults = {
      window: window,
      padding: {
        size: window - 1,
        algorithm: 'duplicate',
        value: 0
      }
    };
    let actualOptions = Object.assign({}, defaults, options);
    let baseline = xRollingMedian(ys, actualOptions);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  function norm(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var _options$algorithm = options.algorithm,
        algorithm = _options$algorithm === void 0 ? 'absolute' : _options$algorithm,
        _options$sumValue = options.sumValue,
        sumValue = _options$sumValue === void 0 ? 1 : _options$sumValue,
        _options$maxValue = options.maxValue,
        maxValue = _options$maxValue === void 0 ? 1 : _options$maxValue;

    if (!isAnyArray(input)) {
      throw new Error('input must be an array');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    if (input.length === 0) {
      throw new Error('input must not be empty');
    }

    switch (algorithm.toLowerCase()) {
      case 'absolute':
        {
          var absoluteSumValue = absoluteSum(input) / sumValue;
          if (absoluteSumValue === 0) return input.slice(0);

          for (var i = 0; i < input.length; i++) {
            output[i] = input[i] / absoluteSumValue;
          }

          return output;
        }

      case 'max':
        {
          var currentMaxValue = max(input);
          if (currentMaxValue === 0) return input.slice(0);
          var factor = maxValue / currentMaxValue;

          for (var _i = 0; _i < input.length; _i++) {
            output[_i] = input[_i] * factor;
          }

          return output;
        }

      case 'sum':
        {
          var sumFactor = sum(input) / sumValue;
          if (sumFactor === 0) return input.slice(0);

          for (var _i2 = 0; _i2 < input.length; _i2++) {
            output[_i2] = input[_i2] / sumFactor;
          }

          return output;
        }

      default:
        throw new Error("norm: unknown algorithm: ".concat(algorithm));
    }
  }

  function absoluteSum(input) {
    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += Math.abs(input[i]);
    }

    return sumValue;
  }

  /**
   * Normalize an array of zones:
   * - ensure than from < to
   * - merge overlapping zones
   *
   * The method will always check if from if lower than to and will swap if required.
   * @param {Array} [zones=[]]
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
   */
  function normalize$1(zones = [], options = {}) {
    if (zones.length === 0) return [];
    let {
      from = Number.NEGATIVE_INFINITY,
      to = Number.POSITIVE_INFINITY
    } = options;
    if (from > to) [from, to] = [to, from];
    zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
      from: zone.to,
      to: zone.from
    } : zone);
    zones = zones.sort((a, b) => {
      if (a.from !== b.from) return a.from - b.from;
      return a.to - b.to;
    });
    zones.forEach(zone => {
      if (from > zone.from) zone.from = from;
      if (to < zone.to) zone.to = to;
    });
    zones = zones.filter(zone => zone.from <= zone.to);
    if (zones.length === 0) return [];
    let currentZone = zones[0];
    let result = [currentZone];

    for (let i = 1; i < zones.length; i++) {
      let zone = zones[i];

      if (zone.from <= currentZone.to) {
        currentZone.to = zone.to;
      } else {
        currentZone = zone;
        result.push(currentZone);
      }
    }

    return result;
  }

  /**
   * Convert an array of exclusions and keep only from / to
   *
   * The method will always check if from if lower than to and will swap if required.
   * @param {Array} [exclusions=[]]
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of zones (after inversion)
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of zones (after inversion)
   */

  function invert(exclusions = [], options = {}) {
    let {
      from = Number.NEGATIVE_INFINITY,
      to = Number.POSITIVE_INFINITY
    } = options;
    if (from > to) [from, to] = [to, from];
    exclusions = normalize$1(exclusions, {
      from,
      to
    });
    if (exclusions.length === 0) return [{
      from,
      to
    }];
    let zones = [];

    for (let i = 0; i < exclusions.length; i++) {
      let exclusion = exclusions[i];
      let nextExclusion = exclusions[i + 1];

      if (i === 0) {
        if (exclusion.from > from) {
          zones.push({
            from,
            to: exclusion.from
          });
        }
      }

      if (i === exclusions.length - 1) {
        if (exclusion.to < to) {
          zones.push({
            from: exclusion.to,
            to
          });
        }
      } else {
        zones.push({
          from: exclusion.to,
          to: nextExclusion.from
        });
      }
    }

    return zones;
  }

  /**
   * Add the number of points per zone to reach a specified total
   * @param {Array} [zones=[]]
   * @param {number} [numberOfPoints] Total number of points to distribute between zones
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
   */

  function zonesWithPoints(zones, numberOfPoints, options = {}) {
    if (zones.length === 0) return zones;
    zones = normalize$1(zones, options);
    const totalSize = zones.reduce((previous, current) => {
      return previous + (current.to - current.from);
    }, 0);
    let unitsPerPoint = totalSize / numberOfPoints;
    let currentTotal = 0;

    for (let i = 0; i < zones.length - 1; i++) {
      let zone = zones[i];
      zone.numberOfPoints = Math.min(Math.round((zone.to - zone.from) / unitsPerPoint), numberOfPoints - currentTotal);
      currentTotal += zone.numberOfPoints;
    }

    zones[zones.length - 1].numberOfPoints = numberOfPoints - currentTotal;
    return zones;
  }

  /**
   * function that retrieves the getEquallySpacedData with the variant "slot"
   *
   * @param {Array<number>} x
   * @param {Array<number>} y
   * @param {number} from - Initial point
   * @param {number} to - Final point
   * @param {number} numberOfPoints
   * @return {Array} - Array of y's equally spaced with the variant "slot"
   */
  function equallySpacedSlot(x, y, from, to, numberOfPoints) {
    let xLength = x.length;
    let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
    let halfStep = step / 2;
    let lastStep = x[x.length - 1] - x[x.length - 2];
    let start = from - halfStep;
    let output = new Array(numberOfPoints); // Init main variables

    let min = start;
    let max = start + step;
    let previousX = -Number.MAX_VALUE;
    let previousY = 0;
    let nextX = x[0];
    let nextY = y[0];
    let frontOutsideSpectra = 0;
    let backOutsideSpectra = true;
    let currentValue = 0; // for slot algorithm

    let currentPoints = 0;
    let i = 1; // index of input

    let j = 0; // index of output

    main: while (true) {
      if (previousX >= nextX) throw new Error('x must be an increasing serie');

      while (previousX - max > 0) {
        // no overlap with original point, just consume current value
        if (backOutsideSpectra) {
          currentPoints++;
          backOutsideSpectra = false;
        }

        output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
        j++;

        if (j === numberOfPoints) {
          break main;
        }

        min = max;
        max += step;
        currentValue = 0;
        currentPoints = 0;
      }

      if (previousX > min) {
        currentValue += previousY;
        currentPoints++;
      }

      if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
        currentPoints--;
      }

      previousX = nextX;
      previousY = nextY;

      if (i < xLength) {
        nextX = x[i];
        nextY = y[i];
        i++;
      } else {
        nextX += lastStep;
        nextY = 0;
        frontOutsideSpectra++;
      }
    }

    return output;
  }

  /**
   * Function that calculates the integral of the line between two
   * x-coordinates, given the slope and intercept of the line.
   * @param {number} x0
   * @param {number} x1
   * @param {number} slope
   * @param {number} intercept
   * @return {number} integral value.
   */
  function integral(x0, x1, slope, intercept) {
    return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
  }

  /**
   * function that retrieves the getEquallySpacedData with the variant "smooth"
   *
   * @param {Array<number>} x
   * @param {Array<number>} y
   * @param {number} from - Initial point
   * @param {number} to - Final point
   * @param {number} numberOfPoints
   * @return {Array} - Array of y's equally spaced with the variant "smooth"
   */

  function equallySpacedSmooth(x, y, from, to, numberOfPoints) {
    let xLength = x.length;
    let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
    let halfStep = step / 2;
    let output = new Array(numberOfPoints);
    let initialOriginalStep = x[1] - x[0];
    let lastOriginalStep = x[xLength - 1] - x[xLength - 2]; // Init main variables

    let min = from - halfStep;
    let max = from + halfStep;
    let previousX = Number.MIN_VALUE;
    let previousY = 0;
    let nextX = x[0] - initialOriginalStep;
    let nextY = 0;
    let currentValue = 0;
    let slope = 0;
    let intercept = 0;
    let sumAtMin = 0;
    let sumAtMax = 0;
    let i = 0; // index of input

    let j = 0; // index of output

    function getSlope(x0, y0, x1, y1) {
      return (y1 - y0) / (x1 - x0);
    }

    let add = 0;

    main: while (true) {
      if (previousX <= min && min <= nextX) {
        add = integral(0, min - previousX, slope, previousY);
        sumAtMin = currentValue + add;
      }

      while (nextX - max >= 0) {
        // no overlap with original point, just consume current value
        add = integral(0, max - previousX, slope, previousY);
        sumAtMax = currentValue + add;
        output[j++] = (sumAtMax - sumAtMin) / step;

        if (j === numberOfPoints) {
          break main;
        }

        min = max;
        max += step;
        sumAtMin = sumAtMax;
      }

      currentValue += integral(previousX, nextX, slope, intercept);
      previousX = nextX;
      previousY = nextY;

      if (i < xLength) {
        nextX = x[i];
        nextY = y[i];
        i++;
      } else if (i === xLength) {
        nextX += lastOriginalStep;
        nextY = 0;
      }

      slope = getSlope(previousX, previousY, nextX, nextY);
      intercept = -slope * previousX + previousY;
    }

    return output;
  }

  /**
   * Function that returns a Number array of equally spaced numberOfPoints
   * containing a representation of intensities of the spectra arguments x
   * and y.
   *
   * The options parameter contains an object in the following form:
   * from: starting point
   * to: last point
   * numberOfPoints: number of points between from and to
   * variant: "slot" or "smooth" - smooth is the default option
   *
   * The slot variant consist that each point in the new array is calculated
   * averaging the existing points between the slot that belongs to the current
   * value. The smooth variant is the same but takes the integral of the range
   * of the slot and divide by the step size between two points in the new array.
   *
   * If exclusions zone are present, zones are ignored !
   * @param {object} [arrayXY={}] - object containing 2 properties x and y (both an array)
   * @param {object} [options={}]
   * @param {number} [options.from=x[0]]
   * @param {number} [options.to=x[x.length-1]]
   * @param {string} [options.variant='smooth']
   * @param {number} [options.numberOfPoints=100]
   * @param {Array} [options.exclusions=[]] array of from / to that should be skipped for the generation of the points
   * @param {Array} [options.zones=[]] array of from / to that should be kept
   * @return {object<x: Array, y:Array>} new object with x / y array with the equally spaced data.
   */

  function equallySpaced(arrayXY = {}, options = {}) {
    let {
      x,
      y
    } = arrayXY;
    let xLength = x.length;
    let reverse = false;

    if (x.length > 1 && x[0] > x[1]) {
      x = x.slice().reverse();
      y = y.slice().reverse();
      reverse = true;
    }

    let {
      from = x[0],
      to = x[xLength - 1],
      variant = 'smooth',
      numberOfPoints = 100,
      exclusions = [],
      zones = []
    } = options;

    if (xLength !== y.length) {
      throw new RangeError("the x and y vector doesn't have the same size.");
    }

    if (typeof from !== 'number' || isNaN(from)) {
      throw new RangeError("'from' option must be a number");
    }

    if (typeof to !== 'number' || isNaN(to)) {
      throw new RangeError("'to' option must be a number");
    }

    if (typeof numberOfPoints !== 'number' || isNaN(numberOfPoints)) {
      throw new RangeError("'numberOfPoints' option must be a number");
    }

    if (numberOfPoints < 2) {
      throw new RangeError("'numberOfPoints' option must be greater than 1");
    }

    if (zones.length === 0) {
      zones = invert(exclusions, {
        from,
        to
      });
    }

    zones = zonesWithPoints(zones, numberOfPoints, {
      from,
      to
    });
    let xResult = [];
    let yResult = [];

    for (let zone of zones) {
      let zoneResult = processZone(x, y, zone.from, zone.to, zone.numberOfPoints, variant);
      xResult = xResult.concat(zoneResult.x);
      yResult = yResult.concat(zoneResult.y);
    }

    if (reverse) {
      if (from < to) {
        return {
          x: xResult.reverse(),
          y: yResult.reverse()
        };
      } else {
        return {
          x: xResult,
          y: yResult
        };
      }
    } else {
      if (from < to) {
        return {
          x: xResult,
          y: yResult
        };
      } else {
        return {
          x: xResult.reverse(),
          y: yResult.reverse()
        };
      }
    }
  }

  function processZone(x, y, from, to, numberOfPoints, variant) {
    if (numberOfPoints < 1) {
      throw new RangeError('the number of points must be at least 1');
    }

    let output = variant === 'slot' ? equallySpacedSlot(x, y, from, to, numberOfPoints) : equallySpacedSmooth(x, y, from, to, numberOfPoints);
    return {
      x: sequentialFill({
        from,
        to,
        size: numberOfPoints
      }),
      y: output
    };
  }

  function getZones(from, to, exclusions = []) {
    if (from > to) {
      [from, to] = [to, from];
    } // in exclusions from and to have to be defined


    exclusions = exclusions.filter(exclusion => exclusion.from !== undefined && exclusion.to !== undefined);
    exclusions = JSON.parse(JSON.stringify(exclusions)); // we ensure that from before to

    exclusions.forEach(exclusion => {
      if (exclusion.from > exclusion.to) {
        [exclusion.to, exclusion.from] = [exclusion.from, exclusion.to];
      }
    });
    exclusions.sort((a, b) => a.from - b.from); // we will rework the exclusions in order to remove overlap and outside range (from / to)

    exclusions.forEach(exclusion => {
      if (exclusion.from < from) exclusion.from = from;
      if (exclusion.to > to) exclusion.to = to;
    });

    for (let i = 0; i < exclusions.length - 1; i++) {
      if (exclusions[i].to > exclusions[i + 1].from) {
        exclusions[i].to = exclusions[i + 1].from;
      }
    }

    exclusions = exclusions.filter(exclusion => exclusion.from < exclusion.to);

    if (!exclusions || exclusions.length === 0) {
      return [{
        from,
        to
      }];
    }

    let zones = [];
    let currentFrom = from;

    for (let exclusion of exclusions) {
      if (currentFrom < exclusion.from) {
        zones.push({
          from: currentFrom,
          to: exclusion.from
        });
      }

      currentFrom = exclusion.to;
    }

    if (currentFrom < to) {
      zones.push({
        from: currentFrom,
        to: to
      });
    }

    return zones;
  }

  /**
   * Filter an array x/y based on various criteria
   * x points are expected to be sorted
   *
   * @param {object} points
   * @param {object} [options={}]
   * @param {array} [options.from]
   * @param {array} [options.to]
   * @param {array} [options.exclusions=[]]
   * @return {{x: Array<number>, y: Array<number>}}
   */

  function filterX(points, options = {}) {
    const {
      x,
      y
    } = points;
    const {
      from = x[0],
      to = x[x.length - 1],
      exclusions = []
    } = options;
    let zones = getZones(from, to, exclusions);
    let currentZoneIndex = 0;
    let newX = [];
    let newY = [];
    let position = 0;

    while (position < x.length) {
      if (x[position] <= zones[currentZoneIndex].to && x[position] >= zones[currentZoneIndex].from) {
        newX.push(x[position]);
        newY.push(y[position]);
      } else {
        if (x[position] > zones[currentZoneIndex].to) {
          currentZoneIndex++;
          if (!zones[currentZoneIndex]) break;
        }
      }

      position++;
    }

    return {
      x: newX,
      y: newY
    };
  }

  var defaultOptions$4 = {
    size: 1,
    value: 0
  };
  /**
   * Case when the entry is an array
   * @param data
   * @param options
   * @returns {Array}
   */

  function arrayCase(data, options) {
    var len = data.length;

    if (typeof options.size === 'number') {
      options.size = [options.size, options.size];
    }

    var cond = len + options.size[0] + options.size[1];
    var output;

    if (options.output) {
      if (options.output.length !== cond) {
        throw new RangeError('Wrong output size');
      }

      output = options.output;
    } else {
      output = new Array(cond);
    }

    var i;

    if (options.value === 'circular') {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) {
          output[i] = data[(len - options.size[0] % len + i) % len];
        } else if (i < options.size[0] + len) {
          output[i] = data[i - options.size[0]];
        } else {
          output[i] = data[(i - options.size[0]) % len];
        }
      }
    } else if (options.value === 'replicate') {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = data[0];else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = data[len - 1];
      }
    } else if (options.value === 'symmetric') {
      if (options.size[0] > len || options.size[1] > len) {
        throw new RangeError('expanded value should not be bigger than the data length');
      }

      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = data[options.size[0] - 1 - i];else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = data[2 * len + options.size[0] - i - 1];
      }
    } else {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = options.value;else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = options.value;
      }
    }

    return output;
  }
  /**
   * Case when the entry is a matrix
   * @param data
   * @param options
   * @returns {Array}
   */


  function matrixCase(data, options) {
    // var row = data.length;
    // var col = data[0].length;
    if (options.size[0] === undefined) {
      options.size = [options.size, options.size, options.size, options.size];
    }

    throw new Error('matrix not supported yet, sorry');
  }
  /**
   * Pads and array
   * @param {Array <number>} data
   * @param {object} options
   */


  function padArray(data, options) {
    options = Object.assign({}, defaultOptions$4, options);

    if (Array.isArray(data)) {
      if (Array.isArray(data[0])) return matrixCase(data, options);else return arrayCase(data, options);
    } else {
      throw new TypeError('data should be an array');
    }
  }

  var src = padArray;
  var padArray$1 = src;

  function factorial(n) {
    let r = 1;

    while (n > 0) r *= n--;

    return r;
  }

  function savitzkyGolay(data, h, options = {}) {
    const {
      windowSize = 5,
      derivative = 1,
      polynomial = 2,
      pad = 'none',
      padValue = 'replicate'
    } = options;

    if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
      throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
    }

    if (derivative < 0 || !Number.isInteger(derivative)) {
      throw new RangeError('Derivative should be a positive integer');
    }

    if (polynomial < 1 || !Number.isInteger(polynomial)) {
      throw new RangeError('Polynomial should be a positive integer');
    }

    let C, norm;
    let step = Math.floor(windowSize / 2);

    if (pad === 'pre') {
      data = padArray$1(data, {
        size: step,
        value: padValue
      });
    }

    let ans = new Array(data.length - 2 * step);

    if (windowSize === 5 && polynomial === 2 && (derivative === 1 || derivative === 2)) {
      if (derivative === 1) {
        C = [-2, -1, 0, 1, 2];
        norm = 10;
      } else {
        C = [2, -1, -2, -1, 2];
        norm = 7;
      }
    } else {
      let J = Matrix.ones(windowSize, polynomial + 1);
      let inic = -(windowSize - 1) / 2;

      for (let i = 0; i < J.rows; i++) {
        for (let j = 0; j < J.columns; j++) {
          if (inic + 1 !== 0 || j !== 0) J.set(i, j, Math.pow(inic + i, j));
        }
      }

      let Jtranspose = new MatrixTransposeView(J);
      let Jinv = inverse(Jtranspose.mmul(J));
      C = Jinv.mmul(Jtranspose);
      C = C.getRow(derivative);
      norm = 1 / factorial(derivative);
    }

    let det = norm * Math.pow(h, derivative);

    for (let k = 0; k < data.length - 2 * step; k++) {
      let d = 0;

      for (let l = 0; l < C.length; l++) d += C[l] * data[l + k] / det;

      ans[k] = d;
    }

    if (pad === 'post') {
      ans = padArray$1(ans, {
        size: step,
        value: padValue
      });
    }

    return ans;
  }

  var array = {};

  (function (exports) {

    function compareNumbers(a, b) {
      return a - b;
    }
    /**
     * Computes the sum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.sum = function sum(values) {
      var sum = 0;

      for (var i = 0; i < values.length; i++) {
        sum += values[i];
      }

      return sum;
    };
    /**
     * Computes the maximum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.max = function max(values) {
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] > max) max = values[i];
      }

      return max;
    };
    /**
     * Computes the minimum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.min = function min(values) {
      var min = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
      }

      return min;
    };
    /**
     * Computes the min and max of the given values
     * @param {Array} values
     * @returns {{min: number, max: number}}
     */


    exports.minMax = function minMax(values) {
      var min = values[0];
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
        if (values[i] > max) max = values[i];
      }

      return {
        min: min,
        max: max
      };
    };
    /**
     * Computes the arithmetic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.arithmeticMean = function arithmeticMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        sum += values[i];
      }

      return sum / l;
    };
    /**
     * {@link arithmeticMean}
     */


    exports.mean = exports.arithmeticMean;
    /**
     * Computes the geometric mean of the given values
     * @param {Array} values
     * @returns {number}
     */

    exports.geometricMean = function geometricMean(values) {
      var mul = 1;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        mul *= values[i];
      }

      return Math.pow(mul, 1 / l);
    };
    /**
     * Computes the mean of the log of the given values
     * If the return value is exponentiated, it gives the same result as the
     * geometric mean.
     * @param {Array} values
     * @returns {number}
     */


    exports.logMean = function logMean(values) {
      var lnsum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        lnsum += Math.log(values[i]);
      }

      return lnsum / l;
    };
    /**
     * Computes the weighted grand mean for a list of means and sample sizes
     * @param {Array} means - Mean values for each set of samples
     * @param {Array} samples - Number of original values for each set of samples
     * @returns {number}
     */


    exports.grandMean = function grandMean(means, samples) {
      var sum = 0;
      var n = 0;
      var l = means.length;

      for (var i = 0; i < l; i++) {
        sum += samples[i] * means[i];
        n += samples[i];
      }

      return sum / n;
    };
    /**
     * Computes the truncated mean of the given values using a given percentage
     * @param {Array} values
     * @param {number} percent - The percentage of values to keep (range: [0,1])
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.truncatedMean = function truncatedMean(values, percent, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var k = Math.floor(l * percent);
      var sum = 0;

      for (var i = k; i < l - k; i++) {
        sum += values[i];
      }

      return sum / (l - 2 * k);
    };
    /**
     * Computes the harmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.harmonicMean = function harmonicMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        if (values[i] === 0) {
          throw new RangeError('value at index ' + i + 'is zero');
        }

        sum += 1 / values[i];
      }

      return l / sum;
    };
    /**
     * Computes the contraharmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.contraHarmonicMean = function contraHarmonicMean(values) {
      var r1 = 0;
      var r2 = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        r1 += values[i] * values[i];
        r2 += values[i];
      }

      if (r2 < 0) {
        throw new RangeError('sum of values is negative');
      }

      return r1 / r2;
    };
    /**
     * Computes the median of the given values
     * @param {Array} values
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.median = function median(values, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var half = Math.floor(l / 2);

      if (l % 2 === 0) {
        return (values[half - 1] + values[half]) * 0.5;
      } else {
        return values[half];
      }
    };
    /**
     * Computes the variance of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.variance = function variance(values, unbiased) {
      if (unbiased === undefined) unbiased = true;
      var theMean = exports.mean(values);
      var theVariance = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        var x = values[i] - theMean;
        theVariance += x * x;
      }

      if (unbiased) {
        return theVariance / (l - 1);
      } else {
        return theVariance / l;
      }
    };
    /**
     * Computes the standard deviation of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.standardDeviation = function standardDeviation(values, unbiased) {
      return Math.sqrt(exports.variance(values, unbiased));
    };

    exports.standardError = function standardError(values) {
      return exports.standardDeviation(values) / Math.sqrt(values.length);
    };
    /**
     * IEEE Transactions on biomedical engineering, vol. 52, no. 1, january 2005, p. 76-
     * Calculate the standard deviation via the Median of the absolute deviation
     *  The formula for the standard deviation only holds for Gaussian random variables.
     * @returns {{mean: number, stdev: number}}
     */


    exports.robustMeanAndStdev = function robustMeanAndStdev(y) {
      var mean = 0,
          stdev = 0;
      var length = y.length,
          i = 0;

      for (i = 0; i < length; i++) {
        mean += y[i];
      }

      mean /= length;
      var averageDeviations = new Array(length);

      for (i = 0; i < length; i++) averageDeviations[i] = Math.abs(y[i] - mean);

      averageDeviations.sort(compareNumbers);

      if (length % 2 === 1) {
        stdev = averageDeviations[(length - 1) / 2] / 0.6745;
      } else {
        stdev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
      }

      return {
        mean: mean,
        stdev: stdev
      };
    };

    exports.quartiles = function quartiles(values, alreadySorted) {
      if (typeof alreadySorted === 'undefined') alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var quart = values.length / 4;
      var q1 = values[Math.ceil(quart) - 1];
      var q2 = exports.median(values, true);
      var q3 = values[Math.ceil(quart * 3) - 1];
      return {
        q1: q1,
        q2: q2,
        q3: q3
      };
    };

    exports.pooledStandardDeviation = function pooledStandardDeviation(samples, unbiased) {
      return Math.sqrt(exports.pooledVariance(samples, unbiased));
    };

    exports.pooledVariance = function pooledVariance(samples, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var sum = 0;
      var length = 0,
          l = samples.length;

      for (var i = 0; i < l; i++) {
        var values = samples[i];
        var vari = exports.variance(values);
        sum += (values.length - 1) * vari;
        if (unbiased) length += values.length - 1;else length += values.length;
      }

      return sum / length;
    };

    exports.mode = function mode(values) {
      var l = values.length,
          itemCount = new Array(l),
          i;

      for (i = 0; i < l; i++) {
        itemCount[i] = 0;
      }

      var itemArray = new Array(l);
      var count = 0;

      for (i = 0; i < l; i++) {
        var index = itemArray.indexOf(values[i]);
        if (index >= 0) itemCount[index]++;else {
          itemArray[count] = values[i];
          itemCount[count] = 1;
          count++;
        }
      }

      var maxValue = 0,
          maxIndex = 0;

      for (i = 0; i < count; i++) {
        if (itemCount[i] > maxValue) {
          maxValue = itemCount[i];
          maxIndex = i;
        }
      }

      return itemArray[maxIndex];
    };

    exports.covariance = function covariance(vector1, vector2, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var mean1 = exports.mean(vector1);
      var mean2 = exports.mean(vector2);
      if (vector1.length !== vector2.length) throw 'Vectors do not have the same dimensions';
      var cov = 0,
          l = vector1.length;

      for (var i = 0; i < l; i++) {
        var x = vector1[i] - mean1;
        var y = vector2[i] - mean2;
        cov += x * y;
      }

      if (unbiased) return cov / (l - 1);else return cov / l;
    };

    exports.skewness = function skewness(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var s2 = 0,
          s3 = 0,
          l = values.length;

      for (var i = 0; i < l; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s3 += dev * dev * dev;
      }

      var m2 = s2 / l;
      var m3 = s3 / l;
      var g = m3 / Math.pow(m2, 3 / 2.0);

      if (unbiased) {
        var a = Math.sqrt(l * (l - 1));
        var b = l - 2;
        return a / b * g;
      } else {
        return g;
      }
    };

    exports.kurtosis = function kurtosis(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var n = values.length,
          s2 = 0,
          s4 = 0;

      for (var i = 0; i < n; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s4 += dev * dev * dev * dev;
      }

      var m2 = s2 / n;
      var m4 = s4 / n;

      if (unbiased) {
        var v = s2 / (n - 1);
        var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
        var b = s4 / (v * v);
        var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
        return a * b - 3 * c;
      } else {
        return m4 / (m2 * m2) - 3;
      }
    };

    exports.entropy = function entropy(values, eps) {
      if (typeof eps === 'undefined') eps = 0;
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * Math.log(values[i] + eps);

      return -sum;
    };

    exports.weightedMean = function weightedMean(values, weights) {
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * weights[i];

      return sum;
    };

    exports.weightedStandardDeviation = function weightedStandardDeviation(values, weights) {
      return Math.sqrt(exports.weightedVariance(values, weights));
    };

    exports.weightedVariance = function weightedVariance(values, weights) {
      var theMean = exports.weightedMean(values, weights);
      var vari = 0,
          l = values.length;
      var a = 0,
          b = 0;

      for (var i = 0; i < l; i++) {
        var z = values[i] - theMean;
        var w = weights[i];
        vari += w * (z * z);
        b += w;
        a += w * w;
      }

      return vari * (b / (b * b - a));
    };

    exports.center = function center(values, inPlace) {
      if (typeof inPlace === 'undefined') inPlace = false;
      var result = values;
      if (!inPlace) result = [].concat(values);
      var theMean = exports.mean(result),
          l = result.length;

      for (var i = 0; i < l; i++) result[i] -= theMean;
    };

    exports.standardize = function standardize(values, standardDev, inPlace) {
      if (typeof standardDev === 'undefined') standardDev = exports.standardDeviation(values);
      if (typeof inPlace === 'undefined') inPlace = false;
      var l = values.length;
      var result = inPlace ? values : new Array(l);

      for (var i = 0; i < l; i++) result[i] = values[i] / standardDev;

      return result;
    };

    exports.cumulativeSum = function cumulativeSum(array) {
      var l = array.length;
      var result = new Array(l);
      result[0] = array[0];

      for (var i = 1; i < l; i++) result[i] = result[i - 1] + array[i];

      return result;
    };
  })(array);

  var Stat = array;

  function getNormalizedSpectrum(spectrum, options = {}) {
    var _a;

    let data = {
      x: spectrum.variables.x.data,
      y: spectrum.variables.y.data
    };
    let newSpectrum = {
      variables: {
        x: {
          data: spectrum.variables.x.data,
          units: spectrum.variables.x.units,
          label: spectrum.variables.x.label
        },
        y: {
          data: spectrum.variables.y.data,
          units: spectrum.variables.y.units,
          label: spectrum.variables.y.label
        }
      }
    };
    if (spectrum.title) newSpectrum.title = spectrum.title;
    if (spectrum.dataType) newSpectrum.dataType = spectrum.dataType;
    if (spectrum.meta) newSpectrum.meta = spectrum.meta;
    let {
      from = spectrum.variables.x.min,
      to = spectrum.variables.x.max,
      numberOfPoints,
      filters = [],
      exclusions = [],
      processing = '',
      keepYUnits = false
    } = options;
    let {
      x,
      y
    } = filterX(data, {
      from,
      to
    });

    switch (processing) {
      case 'firstDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `1st derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 1,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;

      case 'secondDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `2nd derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 2,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;

      case 'thirdDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `3rd derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 3,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;
    }

    if (!keepYUnits && filters.length) {
      // filters change the y axis, we get rid of the units
      newSpectrum.variables.y.units = '';
      newSpectrum.variables.y.label = (_a = newSpectrum.variables.y.label) === null || _a === void 0 ? void 0 : _a.replace(/\s*\[.*\]/, '');
    }

    for (let filter of filters) {
      let filterOptions = filter.options || {};

      switch (filter.name.toLowerCase()) {
        case 'centermean':
          {
            let mean = Stat.mean(y);
            y = xSubtract(y, mean);
            break;
          }

        case 'dividebysd':
          {
            let std = Stat.standardDeviation(y);
            y = xDivide(y, std);
            break;
          }

        case 'normalize':
          {
            y = norm(y, {
              sumValue: filterOptions.value ? Number(filterOptions.value) : 1,
              algorithm: 'absolute'
            });
            break;
          }

        case 'rescale':
          {
            y = rescale(y, {
              min: filterOptions.min ? Number(filterOptions.min) : 0,
              max: filterOptions.max ? Number(filterOptions.max) : 1
            });
            break;
          }

        case 'dividebymax':
          {
            let maxValue = max(y);
            y = xDivide(y, maxValue);
            break;
          }

        case 'multiply':
          {
            y = xMultiply(y, filterOptions.value ? Number(filterOptions.value) : 1);
            break;
          }

        case 'add':
          {
            y = xAdd(y, filterOptions.value ? Number(filterOptions.value) : 0);
            break;
          }

        case 'airplsbaseline':
          {
            y = airPLSBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'rollingaveragebaseline':
          {
            y = rollingAverageBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'iterativepolynomialbaseline':
          {
            y = iterativePolynomialBaseline(y, undefined).correctedSpectrum;
            break;
          }

        case 'rollingballbaseline':
          {
            y = rollingBallBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'rollingmedianbaseline':
          {
            y = rollingMedianBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'ensuregrowing':
          {
            const ans = xyEnsureGrowingX({
              x,
              y
            });
            x = ans.x;
            y = ans.y;
            break;
          }

        case '':
        case undefined:
          break;

        default:
          throw new Error(`Unknown process kind: ${filter.name}`);
      }
    }

    if (!numberOfPoints) {
      data = filterX({
        x,
        y
      }, {
        from,
        to,
        exclusions
      });
    } else {
      data = equallySpaced({
        x,
        y
      }, {
        from,
        to,
        numberOfPoints,
        exclusions
      });
    }

    newSpectrum.variables.x.data = x;
    newSpectrum.variables.x.min = min(x);
    newSpectrum.variables.x.max = max(x);
    newSpectrum.variables.x.isMonotone = xIsMonotone(x);
    newSpectrum.variables.y.data = y;
    newSpectrum.variables.y.min = min(y);
    newSpectrum.variables.y.max = max(y);
    newSpectrum.variables.y.isMonotone = xIsMonotone(y);
    return newSpectrum;
  }

  var quantities = {exports: {}};

  /*
  The MIT License (MIT)
  Copyright © 2006-2007 Kevin C. Olbrich
  Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

  Permission is hereby granted, free of charge, to any person obtaining a copy of
  this software and associated documentation files (the "Software"), to deal in
  the Software without restriction, including without limitation the rights to
  use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
  of the Software, and to permit persons to whom the Software is furnished to do
  so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
  */

  (function (module, exports) {
    (function (global, factory) {
      module.exports = factory() ;
    })(commonjsGlobal, function () {
      /**
       * Tests if a value is a string
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a string, false otherwise
       */

      function isString(value) {
        return typeof value === "string" || value instanceof String;
      }
      /*
       * Prefer stricter Number.isFinite if currently supported.
       * To be dropped when ES6 is finalized. Obsolete browsers will
       * have to use ES6 polyfills.
       */


      var isFiniteImpl = Number.isFinite || window.isFinite;
      /**
       * Tests if a value is a number
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a number, false otherwise
       */

      function isNumber(value) {
        // Number.isFinite allows not to consider NaN or '1' as numbers
        return isFiniteImpl(value);
      }
      /*
       * Identity function
       */


      function identity(value) {
        return value;
      }
      /**
       * Returns unique strings from list
       *
       * @param {string[]} strings - array of strings
       *
       *
       * @returns {string[]} a new array of strings without duplicates
       */


      function uniq(strings) {
        var seen = {};
        return strings.filter(function (item) {
          return seen.hasOwnProperty(item) ? false : seen[item] = true;
        });
      }

      function compareArray(array1, array2) {
        if (array2.length !== array1.length) {
          return false;
        }

        for (var i = 0; i < array1.length; i++) {
          if (array2[i].compareArray) {
            if (!array2[i].compareArray(array1[i])) {
              return false;
            }
          }

          if (array2[i] !== array1[i]) {
            return false;
          }
        }

        return true;
      }

      function assign(target, properties) {
        Object.keys(properties).forEach(function (key) {
          target[key] = properties[key];
        });
      }
      /**
       * Safely multiplies numbers while avoiding floating errors
       * like 0.1 * 0.1 => 0.010000000000000002
       *
       * @param {...number} numbers - numbers to multiply
       *
       * @returns {number} result
       */


      function mulSafe() {
        var result = 1,
            decimals = 0;

        for (var i = 0; i < arguments.length; i++) {
          var arg = arguments[i];
          decimals = decimals + getFractional(arg);
          result *= arg;
        }

        return decimals !== 0 ? round(result, decimals) : result;
      }
      /**
       * Safely divides two numbers while avoiding floating errors
       * like 0.3 / 0.05 => 5.999999999999999
       *
       * @returns {number} result
       * @param {number} num Numerator
       * @param {number} den Denominator
       */


      function divSafe(num, den) {
        if (den === 0) {
          throw new Error("Divide by zero");
        }

        var factor = Math.pow(10, getFractional(den));
        var invDen = factor / (factor * den);
        return mulSafe(num, invDen);
      }
      /**
       * Rounds value at the specified number of decimals
       *
       * @param {number} val - value to round
       * @param {number} decimals - number of decimals
       *
       * @returns {number} rounded number
       */


      function round(val, decimals) {
        return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
      }

      function getFractional(num) {
        // Check for NaNs or Infinities
        if (!isFinite(num)) {
          return 0;
        } // Faster than parsing strings
        // http://jsperf.com/count-decimals/2


        var count = 0;

        while (num % 1 !== 0) {
          num *= 10;
          count++;
        }

        return count;
      }
      /**
       * Custom error type definition
       * @constructor
       */


      function QtyError() {
        var err;

        if (!this) {
          // Allows to instantiate QtyError without new()
          err = Object.create(QtyError.prototype);
          QtyError.apply(err, arguments);
          return err;
        }

        err = Error.apply(this, arguments);
        this.name = "QtyError";
        this.message = err.message;
        this.stack = err.stack;
      }

      QtyError.prototype = Object.create(Error.prototype, {
        constructor: {
          value: QtyError
        }
      });
      /*
       * Throws incompatible units error
       * @param {string} left - units
       * @param {string} right - units incompatible with first argument
       * @throws "Incompatible units" error
       */

      function throwIncompatibleUnits(left, right) {
        throw new QtyError("Incompatible units: " + left + " and " + right);
      }

      var UNITS = {
        /* prefixes */
        "<googol>": [["googol"], 1e100, "prefix"],
        "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
        "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
        "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
        "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
        "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
        "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
        "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
        "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
        "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
        "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
        "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
        "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
        "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
        "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
        "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
        "<kilo>": [["k", "kilo"], 1e3, "prefix"],
        "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
        "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
        "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
        "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
        "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
        "<micro>": [["u", "\u03BC"
        /*µ as greek letter*/
        , "\u00B5"
        /*µ as micro sign*/
        , "Micro", "mc", "micro"], 1e-6, "prefix"],
        "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
        "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
        "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
        "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
        "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
        "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
        "<1>": [["1", "<1>"], 1, ""],

        /* length units */
        "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
        "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
        "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
        "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
        "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
        "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
        "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
        "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
        "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
        "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
        "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
        "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
        "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
        "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
        "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
        "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
        "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
        "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
        "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
        "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
        "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],

        /* mass */
        "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
        "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
        "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
        "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
        "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
        "<metric-ton>": [["tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
        "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
        "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
        "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
        "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
        "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
        "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
        "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],

        /* area */
        "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
        "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
        "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],

        /* volume */
        "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],

        /* speed */
        "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
        "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
        "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
        "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],

        /* acceleration */
        "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
        "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],

        /* temperature_difference */
        "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
        "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
        "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
        "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
        "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
        "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
        "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
        "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],

        /* time */
        "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
        "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
        "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
        "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
        "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
        "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
        "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
        "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
        "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],

        /* pressure */
        "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],

        /* viscosity */
        "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
        "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],

        /* substance */
        "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],

        /* concentration */
        "<molar>": [["M", "molar"], 1000, "concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
        "<wtpercent>": [["wt%", "wtpercent"], 10, "concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],

        /* activity */
        "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
        "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],

        /* capacitance */
        "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],

        /* charge */
        "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
        "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],

        /* current */
        "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],

        /* conductance */
        "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],

        /* inductance */
        "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],

        /* potential */
        "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],

        /* resistance */
        "<ohm>": [["Ohm", "ohm", "\u03A9"
        /*Ω as greek letter*/
        , "\u2126"
        /*Ω as ohm sign*/
        ], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],

        /* magnetism */
        "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],

        /* energy */
        "<joule>": [["J", "joule", "Joule", "joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],

        /* force */
        "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
        "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
        "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],

        /* frequency */
        "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],

        /* angle */
        "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
        "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
        "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
        "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],

        /* rotation */
        "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
        "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],

        /* information */
        "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
        "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],

        /* information rate */
        "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
        "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],

        /* currency */
        "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
        "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],

        /* luminosity */
        "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
        "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
        "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],

        /* power */
        "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],

        /* radiation */
        "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
        "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],

        /* rate */
        "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
        "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
        "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],

        /* resolution / typography */
        "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
        "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
        "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
        "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],

        /* other */
        "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
        "<each>": [["each"], 1.0, "counting", ["<each>"]],
        "<count>": [["count"], 1.0, "counting", ["<each>"]],
        "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
        "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
        "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
        "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
        "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
        "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
        "<ppt>": [["ppt"], 1e-9, "prefix_only", ["<1>"]],
        "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
        "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
      };
      var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
      var UNITY = "<1>";
      var UNITY_ARRAY = [UNITY]; // Setup

      /**
       * Asserts unit definition is valid
       *
       * @param {string} unitDef - Name of unit to test
       * @param {Object} definition - Definition of unit to test
       *
       * @returns {void}
       * @throws {QtyError} if unit definition is not valid
       */

      function validateUnitDefinition(unitDef, definition) {
        var scalar = definition[1];
        var numerator = definition[3] || [];
        var denominator = definition[4] || [];

        if (!isNumber(scalar)) {
          throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
        }

        numerator.forEach(function (unit) {
          if (UNITS[unit] === undefined) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
          }
        });
        denominator.forEach(function (unit) {
          if (UNITS[unit] === undefined) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
          }
        });
      }

      var PREFIX_VALUES = {};
      var PREFIX_MAP = {};
      var UNIT_VALUES = {};
      var UNIT_MAP = {};
      var OUTPUT_MAP = {};

      for (var unitDef in UNITS) {
        if (UNITS.hasOwnProperty(unitDef)) {
          var definition = UNITS[unitDef];

          if (definition[2] === "prefix") {
            PREFIX_VALUES[unitDef] = definition[1];

            for (var i = 0; i < definition[0].length; i++) {
              PREFIX_MAP[definition[0][i]] = unitDef;
            }
          } else {
            validateUnitDefinition(unitDef, definition);
            UNIT_VALUES[unitDef] = {
              scalar: definition[1],
              numerator: definition[3],
              denominator: definition[4]
            };

            for (var j = 0; j < definition[0].length; j++) {
              UNIT_MAP[definition[0][j]] = unitDef;
            }
          }

          OUTPUT_MAP[unitDef] = definition[0][0];
        }
      }
      /**
       * Returns a list of available units of kind
       *
       * @param {string} [kind] - kind of units
       * @returns {array} names of units
       * @throws {QtyError} if kind is unknown
       */


      function getUnits(kind) {
        var i;
        var units = [];
        var unitKeys = Object.keys(UNITS);

        if (typeof kind === "undefined") {
          for (i = 0; i < unitKeys.length; i++) {
            if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
              units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
            }
          }
        } else if (this.getKinds().indexOf(kind) === -1) {
          throw new QtyError("Kind not recognized");
        } else {
          for (i = 0; i < unitKeys.length; i++) {
            if (UNITS[unitKeys[i]][2] === kind) {
              units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
            }
          }
        }

        return units.sort(function (a, b) {
          if (a.toLowerCase() < b.toLowerCase()) {
            return -1;
          }

          if (a.toLowerCase() > b.toLowerCase()) {
            return 1;
          }

          return 0;
        });
      }
      /**
       * Returns a list of alternative names for a unit
       *
       * @param {string} unitName - name of unit
       * @returns {string[]} aliases for unit
       * @throws {QtyError} if unit is unknown
       */


      function getAliases(unitName) {
        if (!UNIT_MAP[unitName]) {
          throw new QtyError("Unit not recognized");
        }

        return UNITS[UNIT_MAP[unitName]][0];
      }

      var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];
      /*
      calculates the unit signature id for use in comparing compatible units and simplification
      the signature is based on a simple classification of units and is based on the following publication
       Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
      21(8), Aug 1995, pp.651-661
      doi://10.1109/32.403789
      http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
      */

      function unitSignature() {
        if (this.signature) {
          return this.signature;
        }

        var vector = unitSignatureVector.call(this);

        for (var i = 0; i < vector.length; i++) {
          vector[i] *= Math.pow(20, i);
        }

        return vector.reduce(function (previous, current) {
          return previous + current;
        }, 0);
      } // calculates the unit signature vector used by unit_signature


      function unitSignatureVector() {
        if (!this.isBase()) {
          return unitSignatureVector.call(this.toBase());
        }

        var vector = new Array(SIGNATURE_VECTOR.length);

        for (var i = 0; i < vector.length; i++) {
          vector[i] = 0;
        }

        var r, n;

        for (var j = 0; j < this.numerator.length; j++) {
          if (r = UNITS[this.numerator[j]]) {
            n = SIGNATURE_VECTOR.indexOf(r[2]);

            if (n >= 0) {
              vector[n] = vector[n] + 1;
            }
          }
        }

        for (var k = 0; k < this.denominator.length; k++) {
          if (r = UNITS[this.denominator[k]]) {
            n = SIGNATURE_VECTOR.indexOf(r[2]);

            if (n >= 0) {
              vector[n] = vector[n] - 1;
            }
          }
        }

        return vector;
      }

      var SIGN = "[+-]";
      var INTEGER = "\\d+";
      var SIGNED_INTEGER = SIGN + "?" + INTEGER;
      var FRACTION = "\\." + INTEGER;
      var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
      var EXPONENT = "[Ee]" + SIGNED_INTEGER;
      var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
      var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
      var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
      var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
      var POWER_OP = "\\^|\\*{2}"; // Allow unit powers representing scalar, length, area, volume; 4 is for some
      // special case representations in SI base units.

      var SAFE_POWER = "[01234]";
      var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
      var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");
      /* parse a string into a unit object.
       * Typical formats like :
       * "5.6 kg*m/s^2"
       * "5.6 kg*m*s^-2"
       * "5.6 kilogram*meter*second^-2"
       * "2.2 kPa"
       * "37 degC"
       * "1"  -- creates a unitless constant with value 1
       * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
       * 6'4"  -- recognized as 6 feet + 4 inches
       * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
       */

      function parse(val) {
        if (!isString(val)) {
          val = val.toString();
        }

        val = val.trim();
        var result = QTY_STRING_REGEX.exec(val);

        if (!result) {
          throw new QtyError(val + ": Quantity not recognized");
        }

        var scalarMatch = result[1];

        if (scalarMatch) {
          // Allow whitespaces between sign and scalar for loose parsing
          scalarMatch = scalarMatch.replace(/\s/g, "");
          this.scalar = parseFloat(scalarMatch);
        } else {
          this.scalar = 1;
        }

        var top = result[2];
        var bottom = result[3];
        var n, x, nx; // TODO DRY me

        while (result = TOP_REGEX.exec(top)) {
          n = parseFloat(result[2]);

          if (isNaN(n)) {
            // Prevents infinite loops
            throw new QtyError("Unit exponent is not a number");
          } // Disallow unrecognized unit even if exponent is 0


          if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
            throw new QtyError("Unit not recognized");
          }

          x = result[1] + " ";
          nx = "";

          for (var i = 0; i < Math.abs(n); i++) {
            nx += x;
          }

          if (n >= 0) {
            top = top.replace(result[0], nx);
          } else {
            bottom = bottom ? bottom + nx : nx;
            top = top.replace(result[0], "");
          }
        }

        while (result = BOTTOM_REGEX.exec(bottom)) {
          n = parseFloat(result[2]);

          if (isNaN(n)) {
            // Prevents infinite loops
            throw new QtyError("Unit exponent is not a number");
          } // Disallow unrecognized unit even if exponent is 0


          if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
            throw new QtyError("Unit not recognized");
          }

          x = result[1] + " ";
          nx = "";

          for (var j = 0; j < n; j++) {
            nx += x;
          }

          bottom = bottom.replace(result[0], nx);
        }

        if (top) {
          this.numerator = parseUnits(top.trim());
        }

        if (bottom) {
          this.denominator = parseUnits(bottom.trim());
        }
      }

      var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
        return b.length - a.length;
      }).join("|");
      var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
        return b.length - a.length;
      }).join("|");
      /*
       * Minimal boundary regex to support units with Unicode characters
       * \b only works for ASCII
       */

      var BOUNDARY_REGEX = "\\b|$";
      var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
      var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
      var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences

      var parsedUnitsCache = {};
      /**
       * Parses and converts units string to normalized unit array.
       * Result is cached to speed up next calls.
       *
       * @param {string} units Units string
       * @returns {string[]} Array of normalized units
       *
       * @example
       * // Returns ["<second>", "<meter>", "<second>"]
       * parseUnits("s m s");
       *
       */

      function parseUnits(units) {
        var cached = parsedUnitsCache[units];

        if (cached) {
          return cached;
        }

        var unitMatch,
            normalizedUnits = []; // Scan

        if (!UNIT_TEST_REGEX.test(units)) {
          throw new QtyError("Unit not recognized");
        }

        while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
          normalizedUnits.push(unitMatch.slice(1));
        }

        normalizedUnits = normalizedUnits.map(function (item) {
          return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
        }); // Flatten and remove null elements

        normalizedUnits = normalizedUnits.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        normalizedUnits = normalizedUnits.filter(function (item) {
          return item;
        });
        parsedUnitsCache[units] = normalizedUnits;
        return normalizedUnits;
      }
      /**
       * Parses a string as a quantity
       * @param {string} value - quantity as text
       * @throws if value is not a string
       * @returns {Qty|null} Parsed quantity or null if unrecognized
       */


      function globalParse(value) {
        if (!isString(value)) {
          throw new QtyError("Argument should be a string");
        }

        try {
          return this(value);
        } catch (e) {
          return null;
        }
      }
      /**
       * Tests if a value is a Qty instance
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a Qty instance, false otherwise
       */


      function isQty(value) {
        return value instanceof Qty;
      }

      function Qty(initValue, initUnits) {
        assertValidConstructorArgs.apply(null, arguments);

        if (!isQty(this)) {
          return new Qty(initValue, initUnits);
        }

        this.scalar = null;
        this.baseScalar = null;
        this.signature = null;
        this._conversionCache = {};
        this.numerator = UNITY_ARRAY;
        this.denominator = UNITY_ARRAY;

        if (isDefinitionObject(initValue)) {
          this.scalar = initValue.scalar;
          this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
          this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
        } else if (initUnits) {
          parse.call(this, initUnits);
          this.scalar = initValue;
        } else {
          parse.call(this, initValue);
        } // math with temperatures is very limited


        if (this.denominator.join("*").indexOf("temp") >= 0) {
          throw new QtyError("Cannot divide with temperatures");
        }

        if (this.numerator.join("*").indexOf("temp") >= 0) {
          if (this.numerator.length > 1) {
            throw new QtyError("Cannot multiply by temperatures");
          }

          if (!compareArray(this.denominator, UNITY_ARRAY)) {
            throw new QtyError("Cannot divide with temperatures");
          }
        }

        this.initValue = initValue;
        updateBaseScalar.call(this);

        if (this.isTemperature() && this.baseScalar < 0) {
          throw new QtyError("Temperatures must not be less than absolute zero");
        }
      }

      Qty.prototype = {
        // Properly set up constructor
        constructor: Qty
      };
      /**
       * Asserts constructor arguments are valid
       *
       * @param {*} value - Value to test
       * @param {string} [units] - Optional units when value is passed as a number
       *
       * @returns {void}
       * @throws {QtyError} if constructor arguments are invalid
       */

      function assertValidConstructorArgs(value, units) {
        if (units) {
          if (!(isNumber(value) && isString(units))) {
            throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
          }
        } else {
          if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
            throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
          }
        }
      }
      /**
       * Tests if a value is a Qty definition object
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a definition object, false otherwise
       */


      function isDefinitionObject(value) {
        return value && typeof value === "object" && value.hasOwnProperty("scalar");
      }

      function updateBaseScalar() {
        if (this.baseScalar) {
          return this.baseScalar;
        }

        if (this.isBase()) {
          this.baseScalar = this.scalar;
          this.signature = unitSignature.call(this);
        } else {
          var base = this.toBase();
          this.baseScalar = base.scalar;
          this.signature = base.signature;
        }
      }

      var KINDS = {
        "-312078": "elastance",
        "-312058": "resistance",
        "-312038": "inductance",
        "-152058": "potential",
        "-152040": "magnetism",
        "-152038": "magnetism",
        "-7997": "specific_volume",
        "-79": "snap",
        "-59": "jolt",
        "-39": "acceleration",
        "-38": "radiation",
        "-20": "frequency",
        "-19": "speed",
        "-18": "viscosity",
        "-17": "volumetric_flow",
        "-1": "wavenumber",
        "0": "unitless",
        "1": "length",
        "2": "area",
        "3": "volume",
        "20": "time",
        "400": "temperature",
        "7941": "yank",
        "7942": "power",
        "7959": "pressure",
        "7961": "force",
        "7962": "energy",
        "7979": "viscosity",
        "7981": "momentum",
        "7982": "angular_momentum",
        "7997": "density",
        "7998": "area_density",
        "8000": "mass",
        "152020": "radiation_exposure",
        "159999": "magnetism",
        "160000": "current",
        "160020": "charge",
        "312058": "conductance",
        "312078": "capacitance",
        "3199980": "activity",
        "3199997": "molar_concentration",
        "3200000": "substance",
        "63999998": "illuminance",
        "64000000": "luminous_power",
        "1280000000": "currency",
        "25599999980": "information_rate",
        "25600000000": "information",
        "511999999980": "angular_velocity",
        "512000000000": "angle"
      };
      /**
       * Returns the list of available well-known kinds of units, e.g.
       * "radiation" or "length".
       *
       * @returns {string[]} names of kinds of units
       */

      function getKinds() {
        return uniq(Object.keys(KINDS).map(function (knownSignature) {
          return KINDS[knownSignature];
        }));
      }

      Qty.prototype.kind = function () {
        return KINDS[this.signature.toString()];
      };

      assign(Qty.prototype, {
        isDegrees: function () {
          // signature may not have been calculated yet
          return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
        },
        isTemperature: function () {
          return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
        }
      });

      function subtractTemperatures(lhs, rhs) {
        var lhsUnits = lhs.units();
        var rhsConverted = rhs.to(lhsUnits);
        var dstDegrees = Qty(getDegreeUnits(lhsUnits));
        return Qty({
          "scalar": lhs.scalar - rhsConverted.scalar,
          "numerator": dstDegrees.numerator,
          "denominator": dstDegrees.denominator
        });
      }

      function subtractTempDegrees(temp, deg) {
        var tempDegrees = deg.to(getDegreeUnits(temp.units()));
        return Qty({
          "scalar": temp.scalar - tempDegrees.scalar,
          "numerator": temp.numerator,
          "denominator": temp.denominator
        });
      }

      function addTempDegrees(temp, deg) {
        var tempDegrees = deg.to(getDegreeUnits(temp.units()));
        return Qty({
          "scalar": temp.scalar + tempDegrees.scalar,
          "numerator": temp.numerator,
          "denominator": temp.denominator
        });
      }

      function getDegreeUnits(units) {
        if (units === "tempK") {
          return "degK";
        } else if (units === "tempC") {
          return "degC";
        } else if (units === "tempF") {
          return "degF";
        } else if (units === "tempR") {
          return "degR";
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }
      }

      function toDegrees(src, dst) {
        var srcDegK = toDegK(src);
        var dstUnits = dst.units();
        var dstScalar;

        if (dstUnits === "degK") {
          dstScalar = srcDegK.scalar;
        } else if (dstUnits === "degC") {
          dstScalar = srcDegK.scalar;
        } else if (dstUnits === "degF") {
          dstScalar = srcDegK.scalar * 9 / 5;
        } else if (dstUnits === "degR") {
          dstScalar = srcDegK.scalar * 9 / 5;
        } else {
          throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
        }

        return Qty({
          "scalar": dstScalar,
          "numerator": dst.numerator,
          "denominator": dst.denominator
        });
      }

      function toDegK(qty) {
        var units = qty.units();
        var q;

        if (units.match(/(deg)[CFRK]/)) {
          q = qty.baseScalar;
        } else if (units === "tempK") {
          q = qty.scalar;
        } else if (units === "tempC") {
          q = qty.scalar;
        } else if (units === "tempF") {
          q = qty.scalar * 5 / 9;
        } else if (units === "tempR") {
          q = qty.scalar * 5 / 9;
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }

        return Qty({
          "scalar": q,
          "numerator": ["<kelvin>"],
          "denominator": UNITY_ARRAY
        });
      }

      function toTemp(src, dst) {
        var dstUnits = dst.units();
        var dstScalar;

        if (dstUnits === "tempK") {
          dstScalar = src.baseScalar;
        } else if (dstUnits === "tempC") {
          dstScalar = src.baseScalar - 273.15;
        } else if (dstUnits === "tempF") {
          dstScalar = src.baseScalar * 9 / 5 - 459.67;
        } else if (dstUnits === "tempR") {
          dstScalar = src.baseScalar * 9 / 5;
        } else {
          throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
        }

        return Qty({
          "scalar": dstScalar,
          "numerator": dst.numerator,
          "denominator": dst.denominator
        });
      }

      function toTempK(qty) {
        var units = qty.units();
        var q;

        if (units.match(/(deg)[CFRK]/)) {
          q = qty.baseScalar;
        } else if (units === "tempK") {
          q = qty.scalar;
        } else if (units === "tempC") {
          q = qty.scalar + 273.15;
        } else if (units === "tempF") {
          q = (qty.scalar + 459.67) * 5 / 9;
        } else if (units === "tempR") {
          q = qty.scalar * 5 / 9;
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }

        return Qty({
          "scalar": q,
          "numerator": ["<temp-K>"],
          "denominator": UNITY_ARRAY
        });
      }

      assign(Qty.prototype, {
        /**
         * Converts to other compatible units.
         * Instance's converted quantities are cached for faster subsequent calls.
         *
         * @param {(string|Qty)} other - Target units as string or retrieved from
         *                               other Qty instance (scalar is ignored)
         *
         * @returns {Qty} New converted Qty instance with target units
         *
         * @throws {QtyError} if target units are incompatible
         *
         * @example
         * var weight = Qty("25 kg");
         * weight.to("lb"); // => Qty("55.11556554621939 lbs");
         * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
         */
        to: function (other) {
          var cached, target;

          if (other === undefined || other === null) {
            return this;
          }

          if (!isString(other)) {
            return this.to(other.units());
          }

          cached = this._conversionCache[other];

          if (cached) {
            return cached;
          } // Instantiating target to normalize units


          target = Qty(other);

          if (target.units() === this.units()) {
            return this;
          }

          if (!this.isCompatible(target)) {
            if (this.isInverse(target)) {
              target = this.inverse().to(other);
            } else {
              throwIncompatibleUnits(this.units(), target.units());
            }
          } else {
            if (target.isTemperature()) {
              target = toTemp(this, target);
            } else if (target.isDegrees()) {
              target = toDegrees(this, target);
            } else {
              var q = divSafe(this.baseScalar, target.baseScalar);
              target = Qty({
                "scalar": q,
                "numerator": target.numerator,
                "denominator": target.denominator
              });
            }
          }

          this._conversionCache[other] = target;
          return target;
        },
        // convert to base SI units
        // results of the conversion are cached so subsequent calls to this will be fast
        toBase: function () {
          if (this.isBase()) {
            return this;
          }

          if (this.isTemperature()) {
            return toTempK(this);
          }

          var cached = baseUnitCache[this.units()];

          if (!cached) {
            cached = toBaseUnits(this.numerator, this.denominator);
            baseUnitCache[this.units()] = cached;
          }

          return cached.mul(this.scalar);
        },
        // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
        toFloat: function () {
          if (this.isUnitless()) {
            return this.scalar;
          }

          throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
        },

        /**
         * Returns the nearest multiple of quantity passed as
         * precision
         *
         * @param {(Qty|string|number)} precQuantity - Quantity, string formated
         *   quantity or number as expected precision
         *
         * @returns {Qty} Nearest multiple of precQuantity
         *
         * @example
         * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
         * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
         * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
         * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
         *
         */
        toPrec: function (precQuantity) {
          if (isString(precQuantity)) {
            precQuantity = Qty(precQuantity);
          }

          if (isNumber(precQuantity)) {
            precQuantity = Qty(precQuantity + " " + this.units());
          }

          if (!this.isUnitless()) {
            precQuantity = precQuantity.to(this.units());
          } else if (!precQuantity.isUnitless()) {
            throwIncompatibleUnits(this.units(), precQuantity.units());
          }

          if (precQuantity.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
          return Qty(precRoundedResult + this.units());
        }
      });
      /**
       * Configures and returns a fast function to convert
       * Number values from units to others.
       * Useful to efficiently convert large array of values
       * with same units into others with iterative methods.
       * Does not take care of rounding issues.
       *
       * @param {string} srcUnits Units of values to convert
       * @param {string} dstUnits Units to convert to
       *
       * @returns {Function} Converting function accepting Number value
       *   and returning converted value
       *
       * @throws "Incompatible units" if units are incompatible
       *
       * @example
       * // Converting large array of numbers with the same units
       * // into other units
       * var converter = Qty.swiftConverter("m/h", "ft/s");
       * var convertedSerie = largeSerie.map(converter);
       *
       */

      function swiftConverter(srcUnits, dstUnits) {
        var srcQty = Qty(srcUnits);
        var dstQty = Qty(dstUnits);

        if (srcQty.eq(dstQty)) {
          return identity;
        }

        var convert;

        if (!srcQty.isTemperature()) {
          convert = function (value) {
            return value * srcQty.baseScalar / dstQty.baseScalar;
          };
        } else {
          convert = function (value) {
            // TODO Not optimized
            return srcQty.mul(value).to(dstQty).scalar;
          };
        }

        return function converter(value) {
          var i, length, result;

          if (!Array.isArray(value)) {
            return convert(value);
          } else {
            length = value.length;
            result = [];

            for (i = 0; i < length; i++) {
              result.push(convert(value[i]));
            }

            return result;
          }
        };
      }

      var baseUnitCache = {};

      function toBaseUnits(numerator, denominator) {
        var num = [];
        var den = [];
        var q = 1;
        var unit;

        for (var i = 0; i < numerator.length; i++) {
          unit = numerator[i];

          if (PREFIX_VALUES[unit]) {
            // workaround to fix
            // 0.1 * 0.1 => 0.010000000000000002
            q = mulSafe(q, PREFIX_VALUES[unit]);
          } else {
            if (UNIT_VALUES[unit]) {
              q *= UNIT_VALUES[unit].scalar;

              if (UNIT_VALUES[unit].numerator) {
                num.push(UNIT_VALUES[unit].numerator);
              }

              if (UNIT_VALUES[unit].denominator) {
                den.push(UNIT_VALUES[unit].denominator);
              }
            }
          }
        }

        for (var j = 0; j < denominator.length; j++) {
          unit = denominator[j];

          if (PREFIX_VALUES[unit]) {
            q /= PREFIX_VALUES[unit];
          } else {
            if (UNIT_VALUES[unit]) {
              q /= UNIT_VALUES[unit].scalar;

              if (UNIT_VALUES[unit].numerator) {
                den.push(UNIT_VALUES[unit].numerator);
              }

              if (UNIT_VALUES[unit].denominator) {
                num.push(UNIT_VALUES[unit].denominator);
              }
            }
          }
        } // Flatten


        num = num.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        den = den.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        return Qty({
          "scalar": q,
          "numerator": num,
          "denominator": den
        });
      }

      Qty.parse = globalParse;
      Qty.getUnits = getUnits;
      Qty.getAliases = getAliases;
      Qty.mulSafe = mulSafe;
      Qty.divSafe = divSafe;
      Qty.getKinds = getKinds;
      Qty.swiftConverter = swiftConverter;
      Qty.Error = QtyError;
      assign(Qty.prototype, {
        // Returns new instance with units of this
        add: function (other) {
          if (isString(other)) {
            other = Qty(other);
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.isTemperature() && other.isTemperature()) {
            throw new QtyError("Cannot add two temperatures");
          } else if (this.isTemperature()) {
            return addTempDegrees(this, other);
          } else if (other.isTemperature()) {
            return addTempDegrees(other, this);
          }

          return Qty({
            "scalar": this.scalar + other.to(this).scalar,
            "numerator": this.numerator,
            "denominator": this.denominator
          });
        },
        sub: function (other) {
          if (isString(other)) {
            other = Qty(other);
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.isTemperature() && other.isTemperature()) {
            return subtractTemperatures(this, other);
          } else if (this.isTemperature()) {
            return subtractTempDegrees(this, other);
          } else if (other.isTemperature()) {
            throw new QtyError("Cannot subtract a temperature from a differential degree unit");
          }

          return Qty({
            "scalar": this.scalar - other.to(this).scalar,
            "numerator": this.numerator,
            "denominator": this.denominator
          });
        },
        mul: function (other) {
          if (isNumber(other)) {
            return Qty({
              "scalar": mulSafe(this.scalar, other),
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          } else if (isString(other)) {
            other = Qty(other);
          }

          if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
            throw new QtyError("Cannot multiply by temperatures");
          } // Quantities should be multiplied with same units if compatible, with base units else


          var op1 = this;
          var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
          // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

          if (op1.isCompatible(op2) && op1.signature !== 400) {
            op2 = op2.to(op1);
          }

          var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
          return Qty({
            "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
            "numerator": numdenscale[0],
            "denominator": numdenscale[1]
          });
        },
        div: function (other) {
          if (isNumber(other)) {
            if (other === 0) {
              throw new QtyError("Divide by zero");
            }

            return Qty({
              "scalar": this.scalar / other,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          } else if (isString(other)) {
            other = Qty(other);
          }

          if (other.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          if (other.isTemperature()) {
            throw new QtyError("Cannot divide with temperatures");
          } else if (this.isTemperature() && !other.isUnitless()) {
            throw new QtyError("Cannot divide with temperatures");
          } // Quantities should be multiplied with same units if compatible, with base units else


          var op1 = this;
          var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
          // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

          if (op1.isCompatible(op2) && op1.signature !== 400) {
            op2 = op2.to(op1);
          }

          var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
          return Qty({
            "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
            "numerator": numdenscale[0],
            "denominator": numdenscale[1]
          });
        },
        // Returns a Qty that is the inverse of this Qty,
        inverse: function () {
          if (this.isTemperature()) {
            throw new QtyError("Cannot divide with temperatures");
          }

          if (this.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          return Qty({
            "scalar": 1 / this.scalar,
            "numerator": this.denominator,
            "denominator": this.numerator
          });
        }
      });

      function cleanTerms(num1, den1, num2, den2) {
        function notUnity(val) {
          return val !== UNITY;
        }

        num1 = num1.filter(notUnity);
        num2 = num2.filter(notUnity);
        den1 = den1.filter(notUnity);
        den2 = den2.filter(notUnity);
        var combined = {};

        function combineTerms(terms, direction) {
          var k;
          var prefix;
          var prefixValue;

          for (var i = 0; i < terms.length; i++) {
            if (PREFIX_VALUES[terms[i]]) {
              k = terms[i + 1];
              prefix = terms[i];
              prefixValue = PREFIX_VALUES[prefix];
              i++;
            } else {
              k = terms[i];
              prefix = null;
              prefixValue = 1;
            }

            if (k && k !== UNITY) {
              if (combined[k]) {
                combined[k][0] += direction;
                var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
                combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
              } else {
                combined[k] = [direction, k, prefix, 1, 1];
              }
            }
          }
        }

        combineTerms(num1, 1);
        combineTerms(den1, -1);
        combineTerms(num2, 1);
        combineTerms(den2, -1);
        var num = [];
        var den = [];
        var scale = 1;

        for (var prop in combined) {
          if (combined.hasOwnProperty(prop)) {
            var item = combined[prop];
            var n;

            if (item[0] > 0) {
              for (n = 0; n < item[0]; n++) {
                num.push(item[2] === null ? item[1] : [item[2], item[1]]);
              }
            } else if (item[0] < 0) {
              for (n = 0; n < -item[0]; n++) {
                den.push(item[2] === null ? item[1] : [item[2], item[1]]);
              }
            }

            scale *= divSafe(item[3], item[4]);
          }
        }

        if (num.length === 0) {
          num = UNITY_ARRAY;
        }

        if (den.length === 0) {
          den = UNITY_ARRAY;
        } // Flatten


        num = num.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        den = den.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        return [num, den, scale];
      }

      assign(Qty.prototype, {
        eq: function (other) {
          return this.compareTo(other) === 0;
        },
        lt: function (other) {
          return this.compareTo(other) === -1;
        },
        lte: function (other) {
          return this.eq(other) || this.lt(other);
        },
        gt: function (other) {
          return this.compareTo(other) === 1;
        },
        gte: function (other) {
          return this.eq(other) || this.gt(other);
        },
        // Compare two Qty objects. Throws an exception if they are not of compatible types.
        // Comparisons are done based on the value of the quantity in base SI units.
        //
        // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
        //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
        //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
        //
        //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
        //     Qty("10S").inverse().compareTo("10ohm") == -1
        //     Qty("10ohm").inverse().compareTo("10S") == -1
        //
        //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
        compareTo: function (other) {
          if (isString(other)) {
            return this.compareTo(Qty(other));
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.baseScalar < other.baseScalar) {
            return -1;
          } else if (this.baseScalar === other.baseScalar) {
            return 0;
          } else if (this.baseScalar > other.baseScalar) {
            return 1;
          }
        },
        // Return true if quantities and units match
        // Unit("100 cm").same(Unit("100 cm"))  # => true
        // Unit("100 cm").same(Unit("1 m"))     # => false
        same: function (other) {
          return this.scalar === other.scalar && this.units() === other.units();
        }
      });
      assign(Qty.prototype, {
        // returns true if no associated units
        // false, even if the units are "unitless" like 'radians, each, etc'
        isUnitless: function () {
          return [this.numerator, this.denominator].every(function (item) {
            return compareArray(item, UNITY_ARRAY);
          });
        },

        /*
        check to see if units are compatible, but not the scalar part
        this check is done by comparing signatures for performance reasons
        if passed a string, it will create a unit object with the string and then do the comparison
        this permits a syntax like:
        unit =~ "mm"
        if you want to do a regexp on the unit string do this ...
        unit.units =~ /regexp/
        */
        isCompatible: function (other) {
          if (isString(other)) {
            return this.isCompatible(Qty(other));
          }

          if (!isQty(other)) {
            return false;
          }

          if (other.signature !== undefined) {
            return this.signature === other.signature;
          } else {
            return false;
          }
        },

        /*
        check to see if units are inverse of each other, but not the scalar part
        this check is done by comparing signatures for performance reasons
        if passed a string, it will create a unit object with the string and then do the comparison
        this permits a syntax like:
        unit =~ "mm"
        if you want to do a regexp on the unit string do this ...
        unit.units =~ /regexp/
        */
        isInverse: function (other) {
          return this.inverse().isCompatible(other);
        },
        // Returns 'true' if the Unit is represented in base units
        isBase: function () {
          if (this._isBase !== undefined) {
            return this._isBase;
          }

          if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
            this._isBase = true;
            return this._isBase;
          }

          this.numerator.concat(this.denominator).forEach(function (item) {
            if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
              this._isBase = false;
            }
          }, this);

          if (this._isBase === false) {
            return this._isBase;
          }

          this._isBase = true;
          return this._isBase;
        }
      });

      function NestedMap() {}

      NestedMap.prototype.get = function (keys) {
        // Allows to pass key1, key2, ... instead of [key1, key2, ...]
        if (arguments.length > 1) {
          // Slower with Firefox but faster with Chrome than
          // Array.prototype.slice.call(arguments)
          // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
          keys = Array.apply(null, arguments);
        }

        return keys.reduce(function (map, key, index) {
          if (map) {
            var childMap = map[key];

            if (index === keys.length - 1) {
              return childMap ? childMap.data : undefined;
            } else {
              return childMap;
            }
          }
        }, this);
      };

      NestedMap.prototype.set = function (keys, value) {
        if (arguments.length > 2) {
          keys = Array.prototype.slice.call(arguments, 0, -1);
          value = arguments[arguments.length - 1];
        }

        return keys.reduce(function (map, key, index) {
          var childMap = map[key];

          if (childMap === undefined) {
            childMap = map[key] = {};
          }

          if (index === keys.length - 1) {
            childMap.data = value;
            return value;
          } else {
            return childMap;
          }
        }, this);
      };
      /**
       * Default formatter
       *
       * @param {number} scalar - scalar value
       * @param {string} units - units as string
       *
       * @returns {string} formatted result
       */


      function defaultFormatter(scalar, units) {
        return (scalar + " " + units).trim();
      }
      /**
       *
       * Configurable Qty default formatter
       *
       * @type {function}
       *
       * @param {number} scalar
       * @param {string} units
       *
       * @returns {string} formatted result
       */


      Qty.formatter = defaultFormatter;
      assign(Qty.prototype, {
        // returns the 'unit' part of the Unit object without the scalar
        units: function () {
          if (this._units !== undefined) {
            return this._units;
          }

          var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
          var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);

          if (numIsUnity && denIsUnity) {
            this._units = "";
            return this._units;
          }

          var numUnits = stringifyUnits(this.numerator);
          var denUnits = stringifyUnits(this.denominator);
          this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
          return this._units;
        },

        /**
         * Stringifies the quantity
         * Deprecation notice: only units parameter is supported.
         *
         * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
         *                              target units if string,
         *                              max number of decimals if number,
         *                              passed to #toPrec before converting if Qty
         *
         * @param {number=} maxDecimals - Maximum number of decimals of
         *                                formatted output
         *
         * @returns {string} reparseable quantity as string
         */
        toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
          var targetUnits;

          if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
            targetUnits = this.units();
            maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
          } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
            targetUnits = targetUnitsOrMaxDecimalsOrPrec;
          } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
            return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
          }

          var out = this.to(targetUnits);
          var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
          out = (outScalar + " " + out.units()).trim();
          return out;
        },

        /**
         * Format the quantity according to optional passed target units
         * and formatter
         *
         * @param {string} [targetUnits=current units] -
         *                 optional units to convert to before formatting
         *
         * @param {function} [formatter=Qty.formatter] -
         *                   delegates formatting to formatter callback.
         *                   formatter is called back with two parameters (scalar, units)
         *                   and should return formatted result.
         *                   If unspecified, formatting is delegated to default formatter
         *                   set to Qty.formatter
         *
         * @example
         * var roundingAndLocalizingFormatter = function(scalar, units) {
         *   // localize or limit scalar to n max decimals for instance
         *   // return formatted result
         * };
         * var qty = Qty('1.1234 m');
         * qty.format(); // same units, default formatter => "1.234 m"
         * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
         * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
         * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
         *
         * @returns {string} quantity as string
         */
        format: function (targetUnits, formatter) {
          if (arguments.length === 1) {
            if (typeof targetUnits === "function") {
              formatter = targetUnits;
              targetUnits = undefined;
            }
          }

          formatter = formatter || Qty.formatter;
          var targetQty = this.to(targetUnits);
          return formatter.call(this, targetQty.scalar, targetQty.units());
        }
      });
      var stringifiedUnitsCache = new NestedMap();
      /**
       * Returns a string representing a normalized unit array
       *
       * @param {string[]} units Normalized unit array
       * @returns {string} String representing passed normalized unit array and
       *   suitable for output
       *
       */

      function stringifyUnits(units) {
        var stringified = stringifiedUnitsCache.get(units);

        if (stringified) {
          return stringified;
        }

        var isUnity = compareArray(units, UNITY_ARRAY);

        if (isUnity) {
          stringified = "1";
        } else {
          stringified = simplify(getOutputNames(units)).join("*");
        } // Cache result


        stringifiedUnitsCache.set(units, stringified);
        return stringified;
      }

      function getOutputNames(units) {
        var unitNames = [],
            token,
            tokenNext;

        for (var i = 0; i < units.length; i++) {
          token = units[i];
          tokenNext = units[i + 1];

          if (PREFIX_VALUES[token]) {
            unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
            i++;
          } else {
            unitNames.push(OUTPUT_MAP[token]);
          }
        }

        return unitNames;
      }

      function simplify(units) {
        // this turns ['s','m','s'] into ['s2','m']
        var unitCounts = units.reduce(function (acc, unit) {
          var unitCounter = acc[unit];

          if (!unitCounter) {
            acc.push(unitCounter = acc[unit] = [unit, 0]);
          }

          unitCounter[1]++;
          return acc;
        }, []);
        return unitCounts.map(function (unitCount) {
          return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
        });
      }

      Qty.version = "1.7.6";
      return Qty;
    });
  })(quantities);

  var Qty = quantities.exports;

  function convertUnit(array, fromUnit, toUnit) {
    fromUnit = normalize(fromUnit);
    toUnit = normalize(toUnit);
    if (fromUnit === toUnit) return array;

    try {
      const convert = Qty.swiftConverter(fromUnit, toUnit); // Configures converter

      return convert(array);
    } catch (e) {
      return undefined;
    }
  }

  function normalize(unit) {
    unit = unit.replace(/°C/g, 'tempC');
    unit = unit.replace(/°F/g, 'tempF');
    unit = unit.replace(/(^|\W)K(\W|$)/g, '$1tempK$2');
    return unit;
  }

  const testRegExp = /^\/((?:\\\/|[^/])+)\/([migyu]{0,5})?$/;
  function ensureRegexp(string) {
    const parts = testRegExp.exec(string);

    if (parts) {
      try {
        return new RegExp(parts[1], parts[2]);
      } catch (err) {
        return stringToRegexp(string);
      }
    } else {
      return stringToRegexp(string);
    }
  }

  function stringToRegexp(string, flags = 'i') {
    return new RegExp(string.replace(/[[\]\\{}()+*?.$^|]/g, function (match) {
      return `\\${match}`;
    }), flags);
  }

  function getConvertedVariable(variable, newUnits) {
    const data = variable.units !== undefined && variable.units !== newUnits // would be nice if convertUnit would allow typedArray
    ? convertUnit(Array.from(variable.data), variable.units, newUnits) : variable.data;
    return {
      units: newUnits,
      label: variable.label.replace(`[${variable.units || ''}]`, `[${newUnits}]`),
      data: data || [],
      min: data ? min(data) : undefined,
      max: data ? max(data) : undefined,
      isMonotone: xIsMonotone(data)
    };
  }

  /**
   * Retrieve the spectrum with only X/Y data that match all the selectors
   * If more than one variable match the selector the 'x' or 'y' variable will be
   * taken
   */

  function getXYSpectrum(spectra = [], selector = {}) {
    if (spectra.length < 1) return;

    for (let spectrum of spectra) {
      let variableNames = Object.keys(spectrum.variables);
      if (!(variableNames.length > 1)) continue;
      let {
        dataType,
        title,
        xUnits,
        yUnits,
        variables,
        xVariable = 'x',
        yVariable = 'y',
        units,
        labels,
        xLabel,
        yLabel,
        meta
      } = selector; // we filter on general spectrum information

      if (dataType) {
        dataType = ensureRegexp(dataType);
        if (!spectrum.dataType || !spectrum.dataType.match(dataType)) continue;
      }

      if (title) {
        title = ensureRegexp(title);
        if (!spectrum.title || !spectrum.title.match(title)) continue;
      }

      if (meta && typeof meta === 'object') {
        if (!spectrum.meta) continue;

        for (let key in spectrum.meta) {
          if (!spectrum.meta[key]) continue;
          let value = ensureRegexp(spectrum.meta[key]);
          if (!spectrum.meta[key].match(value)) continue;
        }
      }

      if (units && !xUnits && !yUnits) [yUnits, xUnits] = units.split(/\s*vs\s*/);

      if (labels && !xLabel && !yLabel) {
        [yLabel, xLabel] = labels.split(/\s*vs\s*/);
      }

      if (variables) [yVariable, xVariable] = variables.split(/\s*vs\s*/);
      if (xLabel) xLabel = ensureRegexp(xLabel);
      if (yLabel) yLabel = ensureRegexp(yLabel);
      let x = getPossibleVariable(spectrum.variables, {
        units: xUnits,
        label: xLabel,
        variableName: xVariable
      });
      let y = getPossibleVariable(spectrum.variables, {
        units: yUnits,
        label: yLabel,
        variableName: yVariable
      });

      if (x && y) {
        return {
          title: spectrum.title,
          dataType: spectrum.dataType,
          meta: spectrum.meta,
          variables: {
            x,
            y
          }
        };
      }
    }

    return;
  }

  function getPossibleVariable(variables, selector = {}) {
    const {
      units,
      label,
      variableName
    } = selector;
    let possible = { ...variables
    };

    if (units !== undefined) {
      for (let key in possible) {
        let converted = convertUnit(1, variables[key].units || '', units);

        if (converted) {
          possible[key] = getConvertedVariable(variables[key], units);
        } else {
          possible[key] = undefined;
        }
      }
    }

    if (label !== undefined) {
      for (let key in possible) {
        if (!variables[key].label.match(label)) {
          possible[key] = undefined;
        }
      }
    }

    if (variableName !== undefined) {
      if (possible[variableName]) return possible[variableName];

      if (possible[variableName.toUpperCase()]) {
        return possible[variableName.toUpperCase()];
      }

      if (possible[variableName.toLowerCase()]) {
        return possible[variableName.toLowerCase()];
      }
    }

    const possibleFiltered = Object.values(possible).filter(val => val !== undefined);

    if (possibleFiltered.length > 0) {
      return possibleFiltered[0];
    }
  }

  /**
   * Class allowing to store and manipulate an analysis.
   * An analysis may contain one or more spectra that can be selected
   * based on their units
   */

  class Analysis {
    constructor(options = {}) {
      this.id = options.id || Math.random().toString(36).substring(2, 10);
      this.label = options.label || this.id;
      this.spectrumCallback = options.spectrumCallback;
      this.spectra = [];
      this.cache = {};
    }
    /**
     * Add a spectrum in the internal spectra variable
     */


    pushSpectrum(variables, options = {}) {
      this.spectra.push(standardizeData(variables, options, {
        spectrumCallback: this.spectrumCallback
      }));
      this.cache = {};
    }
    /**
     * Retrieve a Spectrum based on x/y units
     * @param selector.units Units separated by vs like for example "g vs °C"
     * @param selector.xUnits if undefined takes the first variable
     * @param selector.yUnits if undefined takes the second variable
     */


    getXYSpectrum(selector = {}) {
      let id = JSON.stringify(selector);

      if (!this.cache[id]) {
        this.cache[id] = getXYSpectrum(this.spectra, selector);
      }

      return this.cache[id];
    }
    /**
     * Retrieve a xy object
     * @param selector.units Units separated by vs like for example "g vs °C"
     * @param selector.xUnits if undefined takes the first variable
     * @param selector.yUnits if undefined takes the second variable
     */


    getXY(selector = {}) {
      let spectrum = this.getXYSpectrum(selector);
      if (!spectrum) return undefined;
      return {
        x: spectrum.variables.x.data,
        y: spectrum.variables.y.data
      };
    }
    /**
     * Return the data object for specific x/y units with possibly some
     * normalization options
     * @param options.selector.xUnits // if undefined takes the first variable
     * @param options.selector.yUnits // if undefined takes the second variable
     */


    getNormalizedSpectrum(options = {}) {
      const {
        normalization,
        selector
      } = options;
      const spectrum = this.getXYSpectrum(selector);
      if (!spectrum) return undefined;
      return getNormalizedSpectrum(spectrum, normalization);
    }
    /**
     * Returns the first spectrum. This method could be improved in the future
     * @returns
     */


    getSpectrum() {
      return this.spectra[0];
    }
    /**
     * Returns the xLabel
     * @param selector.xUnits // if undefined takes the first variable
     * @param selector.yUnits // if undefined takes the second variable
     */


    getXLabel(selector) {
      var _a;

      return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.x.label;
    }
    /**
     * Returns the yLabel
     * @param selector.xUnits // if undefined takes the first variable
     * @param selector.yUnits // if undefined takes the second variable
     */


    getYLabel(selector) {
      var _a;

      return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.y.label;
    }

  }
  /**
   * Internal function that ensure the order of x / y array
   */

  function standardizeData(variables, options, analysisOptions) {
    let {
      meta = {},
      tmp = {},
      dataType = '',
      title = ''
    } = options;
    const {
      spectrumCallback
    } = analysisOptions;

    if (spectrumCallback) {
      spectrumCallback(variables);
    }

    let xVariable = variables.x;
    let yVariable = variables.y;

    if (!xVariable || !yVariable) {
      throw Error('A spectrum must contain at least x and y variables');
    }

    if (!isAnyArray(xVariable.data) || !isAnyArray(yVariable.data)) {
      throw Error('x and y variables must contain an array data');
    }

    let x = xVariable.data;
    let reverse = x && x.length > 1 && x[0] > x[x.length - 1];

    for (let key in variables) {
      let variable = variables[key];
      if (reverse) variable.data = variable.data.reverse();
      variable.label = variable.label || key;

      if (!variable.units && variable.label.includes('[')) {
        variable.units = variable.label.replace(/^.*[([](?<units>.*)[)\]].*$/, '$<units>');
      }

      variable.min = min(variable.data);
      variable.max = max(variable.data);
      variable.isMonotone = xIsMonotone(variable.data);
    }

    return {
      variables,
      title,
      dataType,
      meta,
      tmp
    };
  }

  /**
   * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
   * @param {string|ArrayBuffer|ArrayBufferView} blob
   * @param {object} [options={}]
   * @param {string} [options.encoding] - Defaults to `utf-8` or automatic detection of `utf-16`.
   * @returns {string}
   */
  function ensureString(blob, options = {}) {
    if (typeof blob === 'string') {
      return blob;
    }

    if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
      const {
        encoding = guessEncoding(blob)
      } = options;
      const decoder = new TextDecoder(encoding);
      return decoder.decode(blob);
    }

    throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
  }

  function guessEncoding(blob) {
    const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

    if (uint8.length >= 2) {
      if (uint8[0] === 0xfe && uint8[1] === 0xff) {
        return 'utf-16be';
      }

      if (uint8[0] === 0xff && uint8[1] === 0xfe) {
        return 'utf-16le';
      }
    }

    return 'utf-8';
  }

  /**
   * Dynamically type a string
   * @param {string} value String to dynamically type
   * @returns {boolean|string|number}
   */
  function parseString(value) {
    if (value.length === 4 || value.length === 5) {
      let lowercase = value.toLowerCase();
      if (lowercase === 'true') return true;
      if (lowercase === 'false') return false;
    }

    let number = Number(value);

    if (number === 0 && !value.includes('0')) {
      return value;
    }

    if (!Number.isNaN(number)) return number;
    return value;
  }

  const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
  function complexChromatogram(result) {
    let spectra = result.spectra;
    let length = spectra.length;
    let chromatogram = {
      times: new Array(length),
      series: {
        ms: {
          dimension: 2,
          data: new Array(length)
        }
      }
    };
    let existingGCMSFields = [];

    for (let i = 0; i < GC_MS_FIELDS.length; i++) {
      let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

      if (spectra[0][label]) {
        existingGCMSFields.push(label);
        chromatogram.series[label] = {
          dimension: 1,
          data: new Array(length)
        };
      }
    }

    for (let i = 0; i < length; i++) {
      let spectrum = spectra[i];
      chromatogram.times[i] = spectrum.pageValue;

      for (let j = 0; j < existingGCMSFields.length; j++) {
        chromatogram.series[existingGCMSFields[j]].data[i] = parseFloat(spectrum[existingGCMSFields[j]]);
      }

      if (spectrum.data) {
        chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
      }
    }

    result.chromatogram = chromatogram;
  }
  function isMSField(canonicDataLabel) {
    return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
  }
  function convertMSFieldToLabel(value) {
    return value.toLowerCase().replace(/[^a-z0-9]/g, '');
  }

  function convertToFloatArray(stringArray) {
    let floatArray = [];

    for (let i = 0; i < stringArray.length; i++) {
      floatArray.push(parseFloat(stringArray[i]));
    }

    return floatArray;
  }

  function fastParseXYData(spectrum, value) {
    // TODO need to deal with result
    //  console.log(value);
    // we check if deltaX is defined otherwise we calculate it
    let yFactor = spectrum.yFactor;
    let deltaX = spectrum.deltaX;
    spectrum.isXYdata = true;
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData;
    let currentX = spectrum.firstX;
    let currentY = spectrum.firstY; // we skip the first line
    //

    let endLine = false;
    let ascii;
    let i = 0;

    for (; i < value.length; i++) {
      ascii = value.charCodeAt(i);

      if (ascii === 13 || ascii === 10) {
        endLine = true;
      } else {
        if (endLine) break;
      }
    } // we proceed taking the i after the first line


    let newLine = true;
    let isDifference = false;
    let isLastDifference = false;
    let lastDifference = 0;
    let isDuplicate = false;
    let inComment = false;
    let currentValue = 0; // can be a difference or a duplicate

    let lastValue = 0; // must be the real last value

    let isNegative = false;
    let inValue = false;
    let skipFirstValue = false;
    let decimalPosition = 0;

    for (; i <= value.length; i++) {
      if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

      if (inComment) {
        // we should ignore the text if we are after $$
        if (ascii === 13 || ascii === 10) {
          newLine = true;
          inComment = false;
        }
      } else {
        // when is it a new value ?
        // when it is not a digit, . or comma
        // it is a number that is either new or we continue
        if (ascii <= 57 && ascii >= 48) {
          // a number
          inValue = true;

          if (decimalPosition > 0) {
            currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
          } else {
            currentValue *= 10;
            currentValue += ascii - 48;
          }
        } else if (ascii === 44 || ascii === 46) {
          // a "," or "."
          inValue = true;
          decimalPosition++;
        } else {
          if (inValue) {
            // need to process the previous value
            if (newLine) {
              newLine = false; // we don't check the X value
              // console.log("NEW LINE",isDifference, lastDifference);
              // if new line and lastDifference, the first value is just a check !
              // that we don't check ...

              if (isLastDifference) skipFirstValue = true;
            } else {
              // need to deal with duplicate and differences
              if (skipFirstValue) {
                skipFirstValue = false;
              } else {
                if (isDifference) {
                  lastDifference = isNegative ? 0 - currentValue : currentValue;
                  isLastDifference = true;
                  isDifference = false;
                } else if (!isDuplicate) {
                  lastValue = isNegative ? 0 - currentValue : currentValue;
                }

                let duplicate = isDuplicate ? currentValue - 1 : 1;

                for (let j = 0; j < duplicate; j++) {
                  if (isLastDifference) {
                    currentY += lastDifference;
                  } else {
                    currentY = lastValue;
                  }

                  currentData.x.push(currentX);
                  currentData.y.push(currentY * yFactor);
                  currentX += deltaX;
                }
              }
            }

            isNegative = false;
            currentValue = 0;
            decimalPosition = 0;
            inValue = false;
            isDuplicate = false;
          } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


          if (ascii < 74 && ascii > 63) {
            inValue = true;
            isLastDifference = false;
            currentValue = ascii - 64;
          } else if (ascii > 96 && ascii < 106) {
            // negative SQZ digits a b c d e f g h i (ascii 97-105)
            inValue = true;
            isLastDifference = false;
            currentValue = ascii - 96;
            isNegative = true;
          } else if (ascii === 115) {
            // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
            inValue = true;
            isDuplicate = true;
            currentValue = 9;
          } else if (ascii > 82 && ascii < 91) {
            inValue = true;
            isDuplicate = true;
            currentValue = ascii - 82;
          } else if (ascii > 73 && ascii < 83) {
            // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
            inValue = true;
            isDifference = true;
            currentValue = ascii - 73;
          } else if (ascii > 105 && ascii < 115) {
            // negative DIF digits j k l m n o p q r (ascii 106-114)
            inValue = true;
            isDifference = true;
            currentValue = ascii - 105;
            isNegative = true;
          } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
            // $ sign, we need to check the next one
            inValue = true;
            inComment = true;
          } else if (ascii === 37) {
            // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
            inValue = true;
            isDifference = true;
            currentValue = 0;
            isNegative = false;
          } else if (ascii === 45) {
            // a "-"
            // check if after there is a number, decimal or comma
            let ascii2 = value.charCodeAt(i + 1);

            if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
              inValue = true;
              if (!newLine) isLastDifference = false;
              isNegative = true;
            }
          } else if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          } // and now analyse the details ... space or tabulation
          // if "+" we just don't care

        }
      }
    }
  }

  const removeCommentRegExp = /\$\$.*/;
  const peakTableSplitRegExp = /[,\t ]+/;
  function parsePeakTable(spectrum, value, result) {
    spectrum.isPeaktable = true;

    if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
      parseXY(spectrum, value, result);
    } else {
      parseXYZ(spectrum, value, result);
    } // we will add the data in the variables


    if (spectrum.variables) {
      for (let key in spectrum.variables) {
        spectrum.variables[key].data = spectrum.data[key];
      }
    }
  }

  function parseXY(spectrum, value, result) {
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData; // counts for around 20% of the time

    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

      if (values.length % 2 === 0) {
        for (let j = 0; j < values.length; j = j + 2) {
          // takes around 40% of the time to add and parse the 2 values nearly exclusively because of parseFloat
          currentData.x.push(parseFloat(values[j]) * spectrum.xFactor);
          currentData.y.push(parseFloat(values[j + 1]) * spectrum.yFactor);
        }
      } else {
        result.logs.push(`Format error: ${values}`);
      }
    }
  }

  function parseXYZ(spectrum, value, result) {
    let currentData = {};
    let variables = Object.keys(spectrum.variables);
    let numberOfVariables = variables.length;
    variables.forEach(variable => currentData[variable] = []);
    spectrum.data = currentData; // counts for around 20% of the time

    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

      if (values.length % numberOfVariables === 0) {
        for (let j = 0; j < values.length; j++) {
          // todo should try to find a xFactor (y, ...)
          currentData[variables[j % numberOfVariables]].push(parseFloat(values[j]));
        }
      } else {
        result.logs.push(`Format error: ${values}`);
      }
    }
  }

  function parseXYA(spectrum, value) {
    let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
    spectrum.isXYAdata = true;
    let values;
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData;
    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
      currentData.x.push(parseFloat(values[0]));
      currentData.y.push(parseFloat(values[1]));
    }
  }

  function convertTo3DZ(spectra) {
    let minZ = spectra[0].data.y[0];
    let maxZ = minZ;
    let ySize = spectra.length;
    let xSize = spectra[0].data.x.length;
    let z = new Array(ySize);

    for (let i = 0; i < ySize; i++) {
      z[i] = spectra[i].data.y;

      for (let j = 0; j < xSize; j++) {
        let value = z[i][j];
        if (value < minZ) minZ = value;
        if (value > maxZ) maxZ = value;
      }
    }

    const firstX = spectra[0].data.x[0];
    const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

    const firstY = spectra[0].pageValue;
    const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
    // min and max we need to invert the array

    if (firstX > lastX) {
      for (let spectrum of z) {
        spectrum.reverse();
      }
    }

    if (firstY > lastY) {
      z.reverse();
    }

    const medians = [];

    for (let i = 0; i < z.length; i++) {
      const row = Float64Array.from(z[i]);

      for (let i = 0; i < row.length; i++) {
        if (row[i] < 0) row[i] = -row[i];
      }

      medians.push(median(row));
    }

    const median$1 = median(medians);
    return {
      z: z,
      minX: Math.min(firstX, lastX),
      maxX: Math.max(firstX, lastX),
      minY: Math.min(firstY, lastY),
      maxY: Math.max(firstY, lastY),
      minZ: minZ,
      maxZ: maxZ,
      noise: median$1
    };
  }

  function generateContourLines(zData, options) {
    let noise = zData.noise;
    let z = zData.z;
    let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
    let isOver0, isOver1, isOver2, isOver3;
    let nbSubSpectra = z.length;
    let nbPovars = z[0].length;
    let pAx, pAy, pBx, pBy;
    let x0 = zData.minX;
    let xN = zData.maxX;
    let dx = (xN - x0) / (nbPovars - 1);
    let y0 = zData.minY;
    let yN = zData.maxY;
    let dy = (yN - y0) / (nbSubSpectra - 1);
    let minZ = zData.minZ;
    let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
    // -------------------------
    // Povars attribution
    //
    // 0----1
    // |  / |
    // | /  |
    // 2----3
    //
    // ---------------------d------

    let iter = options.nbContourLevels * 2;
    let contourLevels = new Array(iter);
    let lineZValue;

    for (let level = 0; level < iter; level++) {
      // multiply by 2 for positif and negatif
      let contourLevel = {};
      contourLevels[level] = contourLevel;
      let side = level % 2;
      let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

      if (side === 0) {
        lineZValue = factor + options.noiseMultiplier * noise;
      } else {
        lineZValue = 0 - factor - options.noiseMultiplier * noise;
      }

      let lines = [];
      contourLevel.zValue = lineZValue;
      contourLevel.lines = lines;
      if (lineZValue <= minZ || lineZValue >= maxZ) continue;

      for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
        let subSpectra = z[iSubSpectra];
        let subSpectraAfter = z[iSubSpectra + 1];

        for (let povar = 0; povar < nbPovars - 1; povar++) {
          povarHeight0 = subSpectra[povar];
          povarHeight1 = subSpectra[povar + 1];
          povarHeight2 = subSpectraAfter[povar];
          povarHeight3 = subSpectraAfter[povar + 1];
          isOver0 = povarHeight0 > lineZValue;
          isOver1 = povarHeight1 > lineZValue;
          isOver2 = povarHeight2 > lineZValue;
          isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
          // povar2 are below, we find the varersections and add
          // the segment

          if (isOver0 !== isOver1 && isOver0 !== isOver2) {
            pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
            pAy = iSubSpectra;
            pBx = povar;
            pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
            lines.push(pAx * dx + x0);
            lines.push(pAy * dy + y0);
            lines.push(pBx * dx + x0);
            lines.push(pBy * dy + y0);
          } // remove push does not help !!!!


          if (isOver3 !== isOver1 && isOver3 !== isOver2) {
            pAx = povar + 1;
            pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
            pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
            pBy = iSubSpectra + 1;
            lines.push(pAx * dx + x0);
            lines.push(pAy * dy + y0);
            lines.push(pBx * dx + x0);
            lines.push(pBy * dy + y0);
          } // test around the diagonal


          if (isOver1 !== isOver2) {
            pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
            pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

            if (isOver1 !== isOver0) {
              pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
              pBy = iSubSpectra;
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver2 !== isOver0) {
              pBx = povar;
              pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver1 !== isOver3) {
              pBx = povar + 1;
              pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver2 !== isOver3) {
              pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
              pBy = iSubSpectra + 1;
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }
          }
        }
      }
    }

    return {
      minX: zData.minX,
      maxX: zData.maxX,
      minY: zData.minY,
      maxY: zData.maxY,
      segments: contourLevels
    };
  }

  function add2D(result, options) {
    let zData = convertTo3DZ(result.spectra);

    if (!options.noContour) {
      result.contourLines = generateContourLines(zData, options);
      delete zData.z;
    }

    result.minMax = zData;
  }

  function addBaseline(data, baselineFct) {
    if (!baselineFct) return data;
    let xs = data.x;
    let ys = data.y;

    for (let i = 0; i < xs.length; i++) {
      ys[i] += baselineFct(xs[i]);
    }

    return data;
  }

  var defaultSource = Math.random;

  var randomUniform = (function sourceRandomUniform(source) {
    function randomUniform(min, max) {
      min = min == null ? 0 : +min;
      max = max == null ? 1 : +max;
      if (arguments.length === 1) max = min, min = 0;else max -= min;
      return function () {
        return source() * max + min;
      };
    }

    randomUniform.source = sourceRandomUniform;
    return randomUniform;
  })(defaultSource);

  var randomNormal = (function sourceRandomNormal(source) {
    function randomNormal(mu, sigma) {
      var x, r;
      mu = mu == null ? 0 : +mu;
      sigma = sigma == null ? 1 : +sigma;
      return function () {
        var y; // If available, use the second previously-generated uniform random.

        if (x != null) y = x, x = null; // Otherwise, generate a new x and y.
        else do {
          x = source() * 2 - 1;
          y = source() * 2 - 1;
          r = x * x + y * y;
        } while (!r || r > 1);
        return mu + sigma * y * Math.sqrt(-2 * Math.log(r) / r);
      };
    }

    randomNormal.source = sourceRandomNormal;
    return randomNormal;
  })(defaultSource);

  const LOOP = 8;
  const FLOAT_MUL = 1 / 16777216;
  const sh1 = 15;
  const sh2 = 18;
  const sh3 = 11;

  function multiply_uint32(n, m) {
    n >>>= 0;
    m >>>= 0;
    const nlo = n & 0xffff;
    const nhi = n - nlo;
    return (nhi * m >>> 0) + nlo * m >>> 0;
  }

  class XSadd {
    constructor(seed = Date.now()) {
      this.state = new Uint32Array(4);
      this.init(seed);
      this.random = this.getFloat.bind(this);
    }
    /**
     * Returns a 32-bit integer r (0 <= r < 2^32)
     */


    getUint32() {
      this.nextState();
      return this.state[3] + this.state[2] >>> 0;
    }
    /**
     * Returns a floating point number r (0.0 <= r < 1.0)
     */


    getFloat() {
      return (this.getUint32() >>> 8) * FLOAT_MUL;
    }

    init(seed) {
      if (!Number.isInteger(seed)) {
        throw new TypeError('seed must be an integer');
      }

      this.state[0] = seed;
      this.state[1] = 0;
      this.state[2] = 0;
      this.state[3] = 0;

      for (let i = 1; i < LOOP; i++) {
        this.state[i & 3] ^= i + multiply_uint32(1812433253, this.state[i - 1 & 3] ^ this.state[i - 1 & 3] >>> 30 >>> 0) >>> 0;
      }

      this.periodCertification();

      for (let i = 0; i < LOOP; i++) {
        this.nextState();
      }
    }

    periodCertification() {
      if (this.state[0] === 0 && this.state[1] === 0 && this.state[2] === 0 && this.state[3] === 0) {
        this.state[0] = 88; // X

        this.state[1] = 83; // S

        this.state[2] = 65; // A

        this.state[3] = 68; // D
      }
    }

    nextState() {
      let t = this.state[0];
      t ^= t << sh1;
      t ^= t >>> sh2;
      t ^= this.state[3] << sh3;
      this.state[0] = this.state[1];
      this.state[1] = this.state[2];
      this.state[2] = this.state[3];
      this.state[3] = t;
    }

  }

  function addNoise(data, percent = 0, options = {}) {
    const {
      seed
    } = options;
    const distribution = options.distribution || 'uniform';
    let generateRandomNumber;

    switch (distribution) {
      case 'uniform':
        {
          generateRandomNumber = getRandom(randomUniform, seed, -0.5, 0.5);
          break;
        }

      case 'normal':
        {
          generateRandomNumber = getRandom(randomNormal, seed);
          break;
        }

      default:
        {
          const unHandled = distribution; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

          throw Error(`Unknown distribution ${unHandled}`);
        }
    }

    if (!percent) return data;
    let ys = data.y;
    let factor = percent * findMax(ys) / 100;

    for (let i = 0; i < ys.length; i++) {
      ys[i] += generateRandomNumber() * factor;
    }

    return data;
  }

  function getRandom(func, seed, ...args) {
    return typeof seed === 'number' ? func.source(new XSadd(seed).random)(...args) : func(...args);
  }

  function findMax(array) {
    let max = Number.MIN_VALUE;

    for (let item of array) {
      if (item > max) max = item;
    }

    return max;
  }

  class SpectrumGenerator {
    constructor(options = {}) {
      const {
        from = 0,
        to = 1000,
        nbPoints = 10001,
        peakWidthFct = () => 5,
        shape = {
          kind: 'gaussian'
        }
      } = options;
      this.from = from;
      this.to = to;
      this.nbPoints = nbPoints;
      this.interval = (this.to - this.from) / (this.nbPoints - 1);
      this.peakWidthFct = peakWidthFct;
      this.maxPeakHeight = Number.MIN_SAFE_INTEGER;
      this.data = {
        x: new Float64Array(this.nbPoints),
        y: new Float64Array(this.nbPoints)
      };
      const kind = shape.kind;
      const {
        options: shapeOptions = {}
      } = shape;
      let shapeGenerator = getShape1D(kind, shapeOptions);
      this.shape = shapeGenerator;
      assertNumber(this.from, 'from');
      assertNumber(this.to, 'to');
      assertInteger(this.nbPoints, 'nbPoints');

      if (this.to <= this.from) {
        throw new RangeError('to option must be larger than from');
      }

      if (typeof this.peakWidthFct !== 'function') {
        throw new TypeError('peakWidthFct option must be a function');
      }

      this.reset();
    }
    /**
     * Add a series of peaks to the spectrum.
     * @param peaks - Peaks to add.
     */


    addPeaks(peaks, options) {
      if (!Array.isArray(peaks) && (typeof peaks !== 'object' || peaks.x === undefined || peaks.y === undefined || !Array.isArray(peaks.x) || !Array.isArray(peaks.y) || peaks.x.length !== peaks.y.length)) {
        throw new TypeError('peaks must be an array or an object containing x[] and y[]');
      }

      if (Array.isArray(peaks)) {
        for (const peak of peaks) {
          this.addPeak(peak, options);
        }
      } else {
        for (let i = 0; i < peaks.x.length; i++) {
          this.addPeak([peaks.x[i], peaks.y[i]], options);
        }
      }

      return this;
    }
    /**
     * Add a single peak to the spectrum.
     * @param peak
     * @param options
     */


    addPeak(peak, options = {}) {
      if (Array.isArray(peak) && peak.length < 2) {
        throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
      }

      if (!Array.isArray(peak) && (peak.x === undefined || peak.y === undefined)) {
        throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
      }

      let xPosition;
      let intensity;
      let peakWidth;
      let peakShapeOptions;

      if (Array.isArray(peak)) {
        [xPosition, intensity, peakWidth, peakShapeOptions] = peak;
      } else {
        xPosition = peak.x;
        intensity = peak.y;
        peakWidth = peak.width;
        peakShapeOptions = peak.shape;
      }

      if (intensity > this.maxPeakHeight) this.maxPeakHeight = intensity;
      let {
        width = peakWidth === undefined ? this.peakWidthFct(xPosition) : peakWidth,
        widthLeft,
        widthRight,
        shape: shapeOptions
      } = options;

      if (peakShapeOptions) {
        shapeOptions = shapeOptions ? { ...shapeOptions,
          ...peakShapeOptions
        } : peakShapeOptions;
      }

      if (shapeOptions) {
        const kind = shapeOptions.kind;
        const {
          options: shapeParameters = {}
        } = shapeOptions;
        this.shape = getShape1D(kind, shapeParameters);
      }

      if (!widthLeft) widthLeft = width;
      if (!widthRight) widthRight = width;
      let factor = options.factor === undefined ? this.shape.getFactor() : options.factor;
      const firstValue = xPosition - widthLeft / 2 * factor;
      const lastValue = xPosition + widthRight / 2 * factor;
      const firstPoint = Math.max(0, Math.floor((firstValue - this.from) / this.interval));
      const lastPoint = Math.min(this.nbPoints - 1, Math.ceil((lastValue - this.from) / this.interval));
      const middlePoint = Math.round((xPosition - this.from) / this.interval); // PEAK SHAPE MAY BE ASYMMETRC (widthLeft and widthRight) !
      // we calculate the left part of the shape

      this.shape.fwhm = widthLeft;

      for (let index = firstPoint; index < Math.max(middlePoint, 0); index++) {
        this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
      } // we calculate the right part of the gaussian


      this.shape.fwhm = widthRight;

      for (let index = Math.min(middlePoint, lastPoint); index <= lastPoint; index++) {
        this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
      }

      return this;
    }
    /**
     * Add a baseline to the spectrum.
     * @param baselineFct - Mathematical function producing the baseline you want.
     */


    addBaseline(baselineFct) {
      addBaseline(this.data, baselineFct);
      return this;
    }
    /**
     * Add noise to the spectrum.
     * @param percent - Noise's amplitude in percents of the spectrum max value. Default: 0.
     */


    addNoise(percent, options) {
      addNoise(this.data, percent, options);
      return this;
    }
    /**
     * Get the generated spectrum.
     */


    getSpectrum(options = {}) {
      if (typeof options === 'boolean') {
        options = {
          copy: options
        };
      }

      const {
        copy = true,
        threshold = 0
      } = options;

      if (threshold) {
        let minPeakHeight = this.maxPeakHeight * threshold;
        let x = [];
        let y = [];

        for (let i = 0; i < this.data.x.length; i++) {
          if (this.data.y[i] >= minPeakHeight) {
            x.push(this.data.x[i]);
            y.push(this.data.y[i]);
          }
        }

        return {
          x,
          y
        };
      }

      if (copy) {
        return {
          x: this.data.x.slice(),
          y: this.data.y.slice()
        };
      } else {
        return this.data;
      }
    }
    /**
     * Resets the generator with an empty spectrum.
     */


    reset() {
      const spectrum = this.data;

      for (let i = 0; i < this.nbPoints; i++) {
        spectrum.x[i] = this.from + i * this.interval;
      }

      return this;
    }

  }

  function assertInteger(value, name) {
    if (!Number.isInteger(value)) {
      throw new TypeError(`${name} option must be an integer`);
    }
  }

  function assertNumber(value, name) {
    if (!Number.isFinite(value)) {
      throw new TypeError(`${name} option must be a number`);
    }
  }

  // sources:
  // https://en.wikipedia.org/wiki/Gyromagnetic_ratio
  // TODO: #13 can we have a better source and more digits ? @jwist
  const gyromagneticRatio = {
    '1H': 267.52218744e6,
    '2H': 41.065e6,
    '3H': 285.3508e6,
    '3He': -203.789e6,
    '7Li': 103.962e6,
    '13C': 67.28284e6,
    '14N': 19.331e6,
    '15N': -27.116e6,
    '17O': -36.264e6,
    '19F': 251.662e6,
    '23Na': 70.761e6,
    '27Al': 69.763e6,
    '29Si': -53.19e6,
    '31P': 108.291e6,
    '57Fe': 8.681e6,
    '63Cu': 71.118e6,
    '67Zn': 16.767e6,
    '129Xe': -73.997e6
  };

  function postProcessingNMR(entriesFlat) {
    // specific NMR functions
    for (let entry of entriesFlat) {
      let observeFrequency = 0;
      let shiftOffsetVal = 0;

      for (let spectrum of entry.spectra) {
        if (entry.ntuples && entry.ntuples.symbol) {
          if (!observeFrequency && spectrum.observeFrequency) {
            observeFrequency = spectrum.observeFrequency;
          }

          if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }
        } else {
          observeFrequency = spectrum.observeFrequency;
          shiftOffsetVal = spectrum.shiftOffsetVal;
        }

        if (observeFrequency) {
          if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
            spectrum.xUnits = 'PPM';
            spectrum.xFactor = spectrum.xFactor / observeFrequency;
            spectrum.firstX = spectrum.firstX / observeFrequency;
            spectrum.lastX = spectrum.lastX / observeFrequency;
            spectrum.deltaX = spectrum.deltaX / observeFrequency;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] /= observeFrequency;
            }
          }
        }

        if (shiftOffsetVal) {
          let shift = spectrum.firstX - shiftOffsetVal;
          spectrum.firstX = spectrum.firstX - shift;
          spectrum.lastX = spectrum.lastX - shift;

          for (let i = 0; i < spectrum.data.x.length; i++) {
            spectrum.data.x[i] -= shift;
          }
        } // we will check if some nucleus are missing ...


        if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
          for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
            let symbol = entry.ntuples.symbol[i];
            let nucleus = entry.ntuples.nucleus[i];

            if (symbol.startsWith('F') && !nucleus) {
              if (symbol === 'F1') {
                // if F1 is defined we will use F2
                if (entry.tmp.$NUC2) {
                  entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                } else {
                  let f2index = entry.ntuples.symbol.indexOf('F2');

                  if (f2index && entry.ntuples.nucleus[f2index]) {
                    entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                  }
                }
              }

              if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
            }

            if (symbol === 'F2') {
              entry.yType = entry.ntuples.nucleus[0];
            }
          }
        }

        if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
          let unit = '';
          let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

          if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
            unit = entry.ntuples.units[pageSymbolIndex];
          }

          if (unit !== 'PPM') {
            if (pageSymbolIndex !== 0) {
              throw Error('Not sure about this ntuples format');
            }

            let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
            let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

            if (!ratio0 || !ratio1) {
              throw Error('Problem with determination of gyromagnetic ratio');
            }

            let ratio = ratio0 / ratio1 * observeFrequency;
            spectrum.pageValue /= ratio;
          }
        }
      }
    }
  }

  function profiling(result, action, options) {
    if (result.profiling) {
      result.profiling.push({
        action,
        time: Date.now() - options.start
      });
    }
  }

  function simpleChromatogram(result) {
    let data = result.spectra[0].data;
    result.chromatogram = {
      times: data.x.slice(),
      series: {
        intensity: {
          dimension: 1,
          data: data.y.slice()
        }
      }
    };
  }

  function postProcessing(entriesFlat, result, options) {
    // converting Hz to ppm
    postProcessingNMR(entriesFlat);

    for (let entry of entriesFlat) {
      if (Object.keys(entry.ntuples).length > 0) {
        let newNtuples = [];
        let keys = Object.keys(entry.ntuples);

        for (let i = 0; i < keys.length; i++) {
          let key = keys[i];
          let values = entry.ntuples[key];

          for (let j = 0; j < values.length; j++) {
            if (!newNtuples[j]) newNtuples[j] = {};
            newNtuples[j][key] = values[j];
          }
        }

        entry.ntuples = newNtuples;
      }

      if (entry.twoD && options.wantXY) {
        add2D(entry, options);
        profiling(result, 'Finished countour plot calculation', options);

        if (!options.keepSpectra) {
          delete entry.spectra;
        }
      } // maybe it is a GC (HPLC) / MS. In this case we add a new format


      if (options.chromatogram) {
        if (entry.spectra.length > 1) {
          complexChromatogram(entry);
        } else {
          simpleChromatogram(entry);
        }

        profiling(result, 'Finished chromatogram calculation', options);
      }

      delete entry.tmp;
    }
  }

  function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
    let xIndex = -1;
    let yIndex = -1;
    let firstVariable = '';
    let secondVariable = '';

    if (kind.indexOf('++') > 0) {
      firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
      secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
    } else {
      kind = kind.replace(/[^a-zA-Z]/g, '');
      firstVariable = kind.charAt(0);
      secondVariable = kind.charAt(1);
      spectrum.variables = {};

      for (let symbol of kind) {
        let lowerCaseSymbol = symbol.toLowerCase();
        let index = currentEntry.ntuples.symbol.indexOf(symbol);
        if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
        spectrum.variables[lowerCaseSymbol] = {};

        for (let key in currentEntry.ntuples) {
          if (currentEntry.ntuples[key][index]) {
            spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
          }
        }
      }
    }

    xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
    yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
    if (xIndex === -1) xIndex = 0;
    if (yIndex === -1) yIndex = 0;

    if (currentEntry.ntuples.first) {
      if (currentEntry.ntuples.first.length > xIndex) {
        spectrum.firstX = currentEntry.ntuples.first[xIndex];
      }

      if (currentEntry.ntuples.first.length > yIndex) {
        spectrum.firstY = currentEntry.ntuples.first[yIndex];
      }
    }

    if (currentEntry.ntuples.last) {
      if (currentEntry.ntuples.last.length > xIndex) {
        spectrum.lastX = currentEntry.ntuples.last[xIndex];
      }

      if (currentEntry.ntuples.last.length > yIndex) {
        spectrum.lastY = currentEntry.ntuples.last[yIndex];
      }
    }

    if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
      spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
    }

    if (currentEntry.ntuples.factor) {
      if (currentEntry.ntuples.factor.length > xIndex) {
        spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
      }

      if (currentEntry.ntuples.factor.length > yIndex) {
        spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
      }
    }

    if (currentEntry.ntuples.units) {
      if (currentEntry.ntuples.units.length > xIndex) {
        if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
          spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
        } else {
          spectrum.xUnits = currentEntry.ntuples.units[xIndex];
        }
      }

      if (currentEntry.ntuples.units.length > yIndex) {
        if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
          spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
        } else {
          spectrum.yUnits = currentEntry.ntuples.units[yIndex];
        }
      }
    }
  }

  function prepareSpectrum(spectrum) {
    if (!spectrum.xFactor) spectrum.xFactor = 1;
    if (!spectrum.yFactor) spectrum.yFactor = 1;
  }

  const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;

  class Spectrum {}

  const defaultOptions$3 = {
    keepRecordsRegExp: /^$/,
    canonicDataLabels: true,
    canonicMetadataLabels: false,
    dynamicTyping: true,
    withoutXY: false,
    chromatogram: false,
    keepSpectra: false,
    noContour: false,
    nbContourLevels: 7,
    noiseMultiplier: 5,
    profiling: false
  };
  /**
   *
   * @param {string|ArrayBuffer} jcamp
   * @param {object} [options]
   * @param {number} [options.keepRecordsRegExp=/^$/] By default we don't keep meta information
   * @param {number} [options.canonicDataLabels=true] Canonize the Labels (uppercase without symbol)
   * @param {number} [options.canonicMetadataLabels=false] Canonize the metadata Labels (uppercase without symbol)
   * @param {number} [options.dynamicTyping=false] Convert numbers to Number
   * @param {number} [options.withoutXY=false] Remove the XY data
   * @param {number} [options.chromatogram=false] Special post-processing for GC / HPLC / MS
   * @param {number} [options.keepSpectra=false] Force to keep the spectra in case of 2D
   * @param {number} [options.noContour=false] Don't calculate countour in case of 2D
   * @param {number} [options.nbContourLevels=7] Number of positive / negative contour levels to calculate
   * @param {number} [options.noiseMultiplier=5] Define for 2D the level as 5 times the median as default
   * @param {number} [options.profiling=false] Add profiling information
   */

  function convert(jcamp, options = {}) {
    jcamp = ensureString(jcamp);
    options = { ...defaultOptions$3,
      ...options
    };
    options.wantXY = !options.withoutXY;
    options.start = Date.now();
    let entriesFlat = [];
    let result = {
      profiling: options.profiling ? [] : false,
      logs: [],
      entries: []
    };
    let tmpResult = {
      children: []
    };
    let currentEntry = tmpResult;
    let parentsStack = [];
    let spectrum = new Spectrum();

    if (typeof jcamp !== 'string') {
      throw new TypeError('the JCAMP should be a string');
    }

    profiling(result, 'Before split to LDRS', options);
    let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
    profiling(result, 'Split to LDRS', options);
    if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

    for (let ldr of ldrs) {
      // This is a new LDR
      let position = ldr.indexOf('=');
      let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
      let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
      let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

      if (canonicDataLabel === 'DATATABLE') {
        let endLine = dataValue.indexOf('\n');
        if (endLine === -1) endLine = dataValue.indexOf('\r');

        if (endLine > 0) {
          // ##DATA TABLE= (X++(I..I)), XYDATA
          // We need to find the variables
          let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
          prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
          spectrum.datatable = infos[0];

          if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
            canonicDataLabel = 'PEAKTABLE';
          } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
            canonicDataLabel = 'XYDATA';
            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
          }
        }
      }

      if (canonicDataLabel === 'XYDATA') {
        if (options.wantXY) {
          prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

          if (dataValue.match(/.*\+\+.*/)) {
            // ex: (X++(Y..Y))
            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
            fastParseXYData(spectrum, dataValue);
          } else {
            parsePeakTable(spectrum, dataValue, result);
          }

          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      } else if (canonicDataLabel === 'PEAKTABLE') {
        if (options.wantXY) {
          prepareSpectrum(spectrum);
          parsePeakTable(spectrum, dataValue, result);
          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      }

      if (canonicDataLabel === 'PEAKASSIGNMENTS') {
        if (options.wantXY) {
          if (dataValue.match(/.*(XYA).*/)) {
            // ex: (XYA)
            parseXYA(spectrum, dataValue);
          }

          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      }

      if (canonicDataLabel === 'TITLE') {
        let parentEntry = currentEntry;

        if (!parentEntry.children) {
          parentEntry.children = [];
        }

        currentEntry = {
          spectra: [],
          ntuples: {},
          info: {},
          meta: {},
          tmp: {} // tmp information we need to keep for postprocessing

        };
        parentEntry.children.push(currentEntry);
        parentsStack.push(parentEntry);
        entriesFlat.push(currentEntry);
        currentEntry.title = dataValue;
      } else if (canonicDataLabel === 'DATATYPE') {
        currentEntry.dataType = dataValue;

        if (dataValue.match(/(^nd|\snd\s)/i)) {
          currentEntry.twoD = true;
        }
      } else if (canonicDataLabel === 'NTUPLES') {
        if (dataValue.match(/(^nd|\snd\s)/i)) {
          currentEntry.twoD = true;
        }
      } else if (canonicDataLabel === 'DATACLASS') {
        currentEntry.dataClass = dataValue;
      } else if (canonicDataLabel === 'XUNITS') {
        spectrum.xUnits = dataValue;
      } else if (canonicDataLabel === 'YUNITS') {
        spectrum.yUnits = dataValue;
      } else if (canonicDataLabel === 'FIRSTX') {
        spectrum.firstX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'LASTX') {
        spectrum.lastX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'FIRSTY') {
        spectrum.firstY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'LASTY') {
        spectrum.lastY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'NPOINTS') {
        spectrum.nbPoints = parseFloat(dataValue);
      } else if (canonicDataLabel === 'XFACTOR') {
        spectrum.xFactor = parseFloat(dataValue);
      } else if (canonicDataLabel === 'YFACTOR') {
        spectrum.yFactor = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MAXX') {
        spectrum.maxX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MINX') {
        spectrum.minX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MAXY') {
        spectrum.maxY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MINY') {
        spectrum.minY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'DELTAX') {
        spectrum.deltaX = parseFloat(dataValue);
      } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
        if (!spectrum.observeFrequency) {
          spectrum.observeFrequency = parseFloat(dataValue);
        }
      } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
        if (!spectrum.xType) {
          currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
        }
      } else if (canonicDataLabel === '$OFFSET') {
        // OFFSET for Bruker spectra
        currentEntry.shiftOffsetNum = 0;

        if (!spectrum.shiftOffsetVal) {
          spectrum.shiftOffsetVal = parseFloat(dataValue);
        }
      } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
        currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'SYMBOL') {
        currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARTYPE') {
        currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARFORM') {
        currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARDIM') {
        currentEntry.ntuples.vardim = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'UNITS') {
        currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'FACTOR') {
        currentEntry.ntuples.factor = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'FIRST') {
        currentEntry.ntuples.first = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'LAST') {
        currentEntry.ntuples.last = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'MIN') {
        currentEntry.ntuples.min = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'MAX') {
        currentEntry.ntuples.max = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === '.NUCLEUS') {
        if (currentEntry.ntuples) {
          currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
        }
      } else if (canonicDataLabel === 'PAGE') {
        spectrum.page = dataValue.trim();
        spectrum.pageValue = parseFloat(dataValue.replace(/^.*=/, ''));
        spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
      } else if (canonicDataLabel === 'RETENTIONTIME') {
        spectrum.pageValue = parseFloat(dataValue);
      } else if (isMSField(canonicDataLabel)) {
        spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
      } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
        spectrum.sampleDescription = dataValue;
      } else if (canonicDataLabel.startsWith('$NUC')) {
        if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
          currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
        }
      } else if (canonicDataLabel === 'END') {
        currentEntry = parentsStack.pop();
      }

      if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
        let value = dataValue.trim();
        let target, label;

        if (dataLabel.startsWith('$')) {
          label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
          target = currentEntry.meta;
        } else {
          label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
          target = currentEntry.info;
        }

        if (options.dynamicTyping) {
          value = parseString(value);
        }

        if (target[label]) {
          if (!Array.isArray(target[label])) {
            target[label] = [target[label]];
          }

          target[label].push(value);
        } else {
          target[label] = value;
        }
      }
    }

    profiling(result, 'Finished parsing', options);
    postProcessing(entriesFlat, result, options);
    profiling(result, 'Total time', options);
    /*
    if (result.children && result.children.length>0) {
      result = { ...result, ...result.children[0] };
    }
    */

    result.entries = tmpResult.children;
    result.flatten = entriesFlat;
    return result;
  }

  /**
   * Creates a new Analysis from a JCAMP string
   * @param {string} jcamp - String containing the JCAMP data
   * @param {object} [options={}]
   * @param {object} [options.id=Math.random()]
   * @param {string} [options.label=options.id] human redeable label
   * @param {string} [options.spectrumCallback] a callback to apply on variables when creating spectrum
   * @return {Analysis} - New class element with the given data
   */

  function fromJcamp(jcamp, options = {}) {
    let analysis = new Analysis(options);
    addJcamp(analysis, jcamp);
    return analysis;
  }

  function addJcamp(analysis, jcamp) {
    let converted = convert(jcamp, {
      keepRecordsRegExp: /.*/
    });

    for (let entry of converted.flatten) {
      if (!entry.spectra || !entry.spectra[0]) continue;
      let currentSpectrum = entry.spectra[0]; // we ensure variables

      if (!currentSpectrum.variables) {
        const variables = {};
        currentSpectrum.variables = variables;
        variables.x = {
          label: currentSpectrum.xUnits,
          symbol: 'X',
          data: currentSpectrum.data.x || currentSpectrum.data.X
        };
        variables.y = {
          label: currentSpectrum.yUnits,
          symbol: 'Y',
          data: currentSpectrum.data.y || currentSpectrum.data.Y
        };
      } else {
        for (let key in currentSpectrum.variables) {
          const variable = currentSpectrum.variables[key];
          if (variable.label) continue;
          variable.label = variable.name || variable.symbol || key;

          if (variable.units && !variable.label.includes(variable.units)) {
            variable.label += ` [${variable.units}]`;
          }
        }
      }

      analysis.pushSpectrum(currentSpectrum.variables, {
        dataType: entry.dataType,
        title: entry.title,
        meta: entry.meta
      });
    }
  }

  /**
   * Parse from a xyxy data array
   * @param {Array<Array<number>>} variables
   * @param {object} [meta] - same metadata object format that the fromText
   * @return {string} JCAMP of the input
   */

  function creatorNtuples(variables, options) {
    const {
      meta = {},
      info = {}
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = ''
    } = info;
    const symbol = [];
    const varName = [];
    const varType = [];
    const varDim = [];
    const units = [];
    const first = [];
    const last = [];
    const min$1 = [];
    const max$1 = [];
    const keys = Object.keys(variables);

    for (let i = 0; i < keys.length; i++) {
      const key = keys[i];
      let variable = variables[key];
      let name = variable.label && variable.label.replace(/ *\[.*/, '');
      let unit = variable.label && variable.label.replace(/.*\[(.*)\].*/, '$1');
      symbol.push(variable.symbol || key);
      varName.push(variable.name || name || key);
      varDim.push(variables[key].data.length);
      varType.push(variable.type ? variable.type.toUpperCase() : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
      units.push(variable.units || unit || '');
      first.push(variables[key][0]);
      last.push(variables[key][variables[key].length - 1]);
      min$1.push(min(variables[key].data));
      max$1.push(max(variables[key].data));
    }

    let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}\n`;

    for (const key in meta) {
      header += typeof meta[key] === 'object' ? `##$${key}=${JSON.stringify(meta[key])}\n` : `##$${key}=${meta[key]}\n`;
    }

    header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##PAGE= N=1\n`;
    header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;

    for (let i = 0; i < variables[keys[0]].data.length; i++) {
      let point = [];

      for (let key of keys) {
        point.push(variables[key].data[i]);
      }

      header += `${point.join('\t')}\n`;
    }

    header += '##END';
    return header;
  }

  function getNumber(number, factor) {
    if (factor !== 1) number /= factor;
    const rounded = Math.round(number);

    if (rounded !== number && Math.abs(rounded - number) <= Number.EPSILON) {
      return rounded;
    }

    return number;
  }

  function peakTableCreator(data, options = {}) {
    const {
      xFactor = 1,
      yFactor = 1
    } = options;
    let firstX = Number.POSITIVE_INFINITY;
    let lastX = Number.NEGATIVE_INFINITY;
    let firstY = Number.POSITIVE_INFINITY;
    let lastY = Number.NEGATIVE_INFINITY;
    let lines = [];

    for (let i = 0; i < data.x.length; i++) {
      let x = data.x[i];
      let y = data.y[i];

      if (firstX > x) {
        firstX = x;
      }

      if (lastX < x) {
        lastX = x;
      }

      if (firstY > y) {
        firstY = y;
      }

      if (lastY < y) {
        lastY = y;
      }
    }

    lines.push(`##FIRSTX=${firstX}`);
    lines.push(`##LASTX=${lastX}`);
    lines.push(`##FIRSTY=${firstY}`);
    lines.push(`##LASTY=${lastY}`);
    lines.push(`##XFACTOR=${xFactor}`);
    lines.push(`##YFACTOR=${yFactor}`);
    lines.push('##PEAK TABLE=(XY..XY)');

    for (let i = 0; i < data.x.length; i++) {
      lines.push(`${getNumber(data.x[i], xFactor)} ${getNumber(data.y[i], yFactor)}`);
    }

    return lines;
  }

  function xyDataCreator(data, options = {}) {
    const {
      xFactor = 1,
      yFactor = 1
    } = options;
    let firstX = data.x[0];
    let lastX = data.x[data.x.length - 1];
    let firstY = data.y[0];
    let lastY = data.y[data.y.length - 1];
    let nbPoints = data.x.length;
    let deltaX = (lastX - firstX) / (nbPoints - 1);
    let lines = [];
    lines.push(`##FIRSTX=${firstX}`);
    lines.push(`##LASTX=${lastX}`);
    lines.push(`##FIRSTY=${firstY}`);
    lines.push(`##LASTY=${lastY}`);
    lines.push(`##DELTAX=${deltaX}`);
    lines.push(`##XFACTOR=${xFactor}`);
    lines.push(`##YFACTOR=${yFactor}`);
    lines.push(`##DELTAX=${xFactor}`);
    lines.push('##XYDATA=(X++(Y..Y))');
    let line = data.x[0];

    for (let i = 0; i < data.x.length; i++) {
      line += ` ${getNumber(data.y[i], yFactor)}`;

      if (line.length > 70) {
        lines.push(line);

        if (i < data.x.length - 1) {
          line = getNumber(data.x[0] + i * deltaX, xFactor);
        } else {
          line = '';
        }
      }
    }

    if (line) lines.push(line);
    return lines;
  }

  /**
   * Create a jcamp
   * @param {object} data object of array
   * @param {object} [options={}] metadata object
   * @param {string} [options.info={}] metadata of the file
   * @param {string} [options.info.title=''] title of the file
   * @param {string} [options.info.owner=''] owner of the file
   * @param {string} [options.info.origin=''] origin of the file
   * @param {string} [options.info.dataType=''] type of data
   * @param {string} [options.info.xUnits=''] units for the x axis for variables===undefined
   * @param {string} [options.info.yUnits=''] units for the y axis for variables===undefined
   * @param {number} [options.info.xFactor=1] factor to multiply X values
   * @param {number} [options.info.yFactor=1] factor to multiply Y values
   * @param {object} [options.meta={}] comments to add to the file
   * @param {boolean} [options.xydata=false] Use XYDATA format. Will use first / last X and equidistant Xs values if true
   * @return {string} JCAMP of the input
   */

  function fromJSON(data, options = {}) {
    const {
      meta = {},
      info = {},
      xydata = false
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = '',
      xUnits = '',
      yUnits = '',
      xFactor = 1,
      yFactor = 1
    } = info;
    data = {
      x: data.x,
      y: data.y
    };
    let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}\n`;

    for (const key in meta) {
      header += key.toLowerCase() === 'cheminfo' ? `##$${key}=${JSON.stringify(meta[key])}\n` : `##$${key}=${meta[key]}\n`;
    } // we leave the header and utf8 fonts ${header.replace(/[^\t\r\n\x20-\x7F]/g, '')


    return `${header}##NPOINTS=${data.x.length}
${(xydata ? xyDataCreator(data, {
    xFactor,
    yFactor
  }) : peakTableCreator(data, {
    xFactor,
    yFactor
  })).join('\n')}
##END`;
  }

  /**
   * Create a jcamp from variables
   * @param {Array<Variable} [variables={}] - object of variables
   * @param {string} [options.info={}] - metadata of the file
   * @param {string} [options.info.title = ''] - title of the file
   * @param {string} [options.info.owner = ''] - owner of the file
   * @param {string} [options.info.origin = ''] - origin of the file
   * @param {string} [options.info.dataType = ''] - type of data
   * @param {object} [options.meta = {}] - comments to add to the file
   * @param {object} [options.forceNtuples = false] - force the ntuples format even if there is only x and y variables
   */

  function fromVariables(variables = {}, options = {}) {
    const {
      info,
      meta,
      forceNtuples
    } = options;
    let jcampOptions = {
      info,
      meta
    };
    let keys = Object.keys(variables).map(key => key.toLowerCase());

    if (keys.length === 2 && keys.includes('x') && keys.includes('y') && !forceNtuples) {
      let x = variables.x;
      let xLabel = x.label || x.name || 'x';
      jcampOptions.info.xUnits = xLabel.includes(variables.x.units) ? xLabel : `${xLabel} [${variables.x.units}]`;
      let y = variables.y;
      let yLabel = y.label || y.name || 'y';
      jcampOptions.info.yUnits = yLabel.includes(variables.y.units) ? yLabel : `${yLabel} [${variables.y.units}]`;
      return fromJSON({
        x: variables.x.data,
        y: variables.y.data
      }, jcampOptions);
    } else {
      return creatorNtuples(variables, options);
    }
  }

  function toJcamps(analysis, options = {}) {
    let jcamps = [];

    for (let spectrum of analysis.spectra) {
      jcamps.push(getJcamp(spectrum, options));
    }

    return jcamps;
  }

  function getJcamp(spectrum, options) {
    const {
      info = {},
      meta = {}
    } = options;
    let jcampOptions = {
      options: {},
      info: {
        title: spectrum.title,
        dataType: spectrum.dataType,
        ...info
      },
      meta: { ...spectrum.meta,
        ...meta
      }
    };
    return fromVariables(spectrum.variables, jcampOptions);
  }

  function toJcamp(analysis, options = {}) {
    return toJcamps(analysis, options).join('\n');
  }

  /** Based on a x value we will return a peak
   * if you set optimize=True the returned positions will be
   * the closest actual datapoints to the fitted peak location.
   * the x/y of the fitted peak will be in xOptimized and yOptimized
   */

  function peakPicking(spectrum,
  /** value to search (on x axis) */
  target, options = {}) {
    var _a, _b, _c;

    const {
      xVariable = 'x',
      yVariable = 'y',
      optimize: optimize$1 = false,
      expectedWidth = 1,
      max: isMax = true,
      shapeOptions = {}
    } = options;
    const x = (_a = spectrum.variables[xVariable]) === null || _a === void 0 ? void 0 : _a.data;
    let y;

    if (!isMax) {
      y = (_b = spectrum.variables[yVariable]) === null || _b === void 0 ? void 0 : _b.data.slice(); // do deep copy as we maybe need to flip sign
    } else {
      y = (_c = spectrum.variables[yVariable]) === null || _c === void 0 ? void 0 : _c.data;
    }

    if (!x || !y) return;
    let targetIndex;
    targetIndex = xFindClosestIndex(x, target);
    let optimizedPeak;
    let optimizedIndex;
    const result = {};

    if (optimize$1) {
      if (isMax === false) {
        let maximumY = max(y);

        for (let i = 0; i < y.length; i++) {
          y[i] *= -1;
          y[i] += maximumY; // This makes it somewhat more robust
        }
      }

      optimizedPeak = optimize({
        x,
        y
      }, [{
        x: x[targetIndex],
        y: y[targetIndex],
        width: expectedWidth
      }], shapeOptions);
      optimizedIndex = xFindClosestIndex(x, optimizedPeak.peaks[0].x);

      for (let key in spectrum.variables) {
        result[key] = spectrum.variables[key].data[optimizedIndex];
      }

      result.optimized = optimizedPeak.peaks[0];
    } else {
      for (let key in spectrum.variables) {
        result[key] = spectrum.variables[key].data[targetIndex];
      }
    }

    return result;
  }

  // @ts-ignore
  /** Based on a x value we will return a peak*/

  function autoPeakPicking(spectrum, options = {}) {
    var _a, _b;

    const {
      xVariable = 'x',
      yVariable = 'y',
      normalizationOptions,
      minPeakWidth
    } = options;
    let x = (_a = spectrum.variables[xVariable]) === null || _a === void 0 ? void 0 : _a.data;
    let y = (_b = spectrum.variables[yVariable]) === null || _b === void 0 ? void 0 : _b.data;

    if (normalizationOptions) {
      const tempSpectrum = {
        variables: {
          x: {
            data: x,
            label: ''
          },
          y: {
            data: y,
            label: ''
          }
        }
      };
      const normalizedSpectrum = getNormalizedSpectrum(tempSpectrum, normalizationOptions);
      x = normalizedSpectrum.variables.x.data;
      y = normalizedSpectrum.variables.y.data;
    }

    if (!x || !y) return;
    let {
      from,
      to
    } = options;
    let peaks = gsd({
      x,
      y
    }, options);

    if (normalizationOptions) {
      // we need to recalculate the real count
      const xyClosestYPoint = options.maxCriteria === undefined || options.maxCriteria ? xyMaxClosestYPoint : xyMinClosestYPoint;

      for (let peak of peaks) {
        const closest = xyClosestYPoint({
          x: spectrum.variables.x.data,
          y: spectrum.variables.y.data
        }, {
          target: peak.x
        });
        peak.x = closest.x;
        peak.y = closest.y;
      }
    }

    if (from !== undefined) {
      peaks = peaks.filter(peak => peak.x >= from);
    }

    if (to !== undefined) {
      peaks = peaks.filter(peak => peak.x <= to);
    }

    if (minPeakWidth) {
      peaks = peaks.filter(peak => peak.width >= minPeakWidth);
    }

    return peaks.map(peak => {
      const result = {};

      for (let key in spectrum.variables) {
        result[key] = spectrum.variables[key].data[peak.index];
      }

      result.width = peak.width;
      return result;
    });
  }

  const JSGraph$1 = {
    getJSGraph,
    getNormalizationAnnotations
  };

  function getAnnotations(peaks, options = {}) {
    const {
      fillColor = 'green',
      strokeColor = 'red',
      creationFct
    } = options;
    if (!peaks) return [];
    let annotations = peaks.map(peak => {
      let annotation = {
        line: 1,
        type: 'rect',
        strokeColor: strokeColor,
        strokeWidth: 0,
        fillColor: fillColor
      };

      if (creationFct) {
        creationFct(annotation, peak);
      }

      createAnnotation(annotation, peak, options);
      return annotation;
    });
    return annotations;
  }

  function createAnnotation(annotation, peak, options = {}) {
    const {
      showAssignment = true,
      assignmentAngle = -45
    } = options;
    let labels = [];
    let line = 0;

    if (showAssignment) {
      labels.push({
        text: peak.assignment,
        size: '18px',
        angle: assignmentAngle,
        anchor: 'left',
        color: 'darkred',
        position: {
          x: peak.x,
          y: peak.y,
          dy: `${-15 - line * 14}px`
        }
      });
      line++;
    }

    annotation.labels = labels;
    annotation.position = [{
      x: peak.x,
      y: peak.y,
      dy: '-10px',
      dx: '-1px'
    }, {
      x: peak.x,
      y: peak.y,
      dy: '-5px',
      dx: '1px'
    }];
  }

  var parser$1 = {};

  var node2json = {};

  var util$4 = {};

  (function (exports) {

    const nameStartChar = ':A-Za-z_\\u00C0-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u02FF\\u0370-\\u037D\\u037F-\\u1FFF\\u200C-\\u200D\\u2070-\\u218F\\u2C00-\\u2FEF\\u3001-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFFD';
    const nameChar = nameStartChar + '\\-.\\d\\u00B7\\u0300-\\u036F\\u203F-\\u2040';
    const nameRegexp = '[' + nameStartChar + '][' + nameChar + ']*';
    const regexName = new RegExp('^' + nameRegexp + '$');

    const getAllMatches = function (string, regex) {
      const matches = [];
      let match = regex.exec(string);

      while (match) {
        const allmatches = [];
        const len = match.length;

        for (let index = 0; index < len; index++) {
          allmatches.push(match[index]);
        }

        matches.push(allmatches);
        match = regex.exec(string);
      }

      return matches;
    };

    const isName = function (string) {
      const match = regexName.exec(string);
      return !(match === null || typeof match === 'undefined');
    };

    exports.isExist = function (v) {
      return typeof v !== 'undefined';
    };

    exports.isEmptyObject = function (obj) {
      return Object.keys(obj).length === 0;
    };
    /**
     * Copy all the properties of a into b.
     * @param {*} target
     * @param {*} a
     */


    exports.merge = function (target, a, arrayMode) {
      if (a) {
        const keys = Object.keys(a); // will return an array of own properties

        const len = keys.length; //don't make it inline

        for (let i = 0; i < len; i++) {
          if (arrayMode === 'strict') {
            target[keys[i]] = [a[keys[i]]];
          } else {
            target[keys[i]] = a[keys[i]];
          }
        }
      }
    };
    /* exports.merge =function (b,a){
      return Object.assign(b,a);
    } */


    exports.getValue = function (v) {
      if (exports.isExist(v)) {
        return v;
      } else {
        return '';
      }
    }; // const fakeCall = function(a) {return a;};
    // const fakeCallNoReturn = function() {};


    exports.buildOptions = function (options, defaultOptions, props) {
      var newOptions = {};

      if (!options) {
        return defaultOptions; //if there are not options
      }

      for (let i = 0; i < props.length; i++) {
        if (options[props[i]] !== undefined) {
          newOptions[props[i]] = options[props[i]];
        } else {
          newOptions[props[i]] = defaultOptions[props[i]];
        }
      }

      return newOptions;
    };
    /**
     * Check if a tag name should be treated as array
     *
     * @param tagName the node tagname
     * @param arrayMode the array mode option
     * @param parentTagName the parent tag name
     * @returns {boolean} true if node should be parsed as array
     */


    exports.isTagNameInArrayMode = function (tagName, arrayMode, parentTagName) {
      if (arrayMode === false) {
        return false;
      } else if (arrayMode instanceof RegExp) {
        return arrayMode.test(tagName);
      } else if (typeof arrayMode === 'function') {
        return !!arrayMode(tagName, parentTagName);
      }

      return arrayMode === "strict";
    };

    exports.isName = isName;
    exports.getAllMatches = getAllMatches;
    exports.nameRegexp = nameRegexp;
  })(util$4);

  const util$3 = util$4;

  const convertToJson = function (node, options, parentTagName) {
    const jObj = {}; // when no child node or attr is present

    if ((!node.child || util$3.isEmptyObject(node.child)) && (!node.attrsMap || util$3.isEmptyObject(node.attrsMap))) {
      return util$3.isExist(node.val) ? node.val : '';
    } // otherwise create a textnode if node has some text


    if (util$3.isExist(node.val) && !(typeof node.val === 'string' && (node.val === '' || node.val === options.cdataPositionChar))) {
      const asArray = util$3.isTagNameInArrayMode(node.tagname, options.arrayMode, parentTagName);
      jObj[options.textNodeName] = asArray ? [node.val] : node.val;
    }

    util$3.merge(jObj, node.attrsMap, options.arrayMode);
    const keys = Object.keys(node.child);

    for (let index = 0; index < keys.length; index++) {
      const tagName = keys[index];

      if (node.child[tagName] && node.child[tagName].length > 1) {
        jObj[tagName] = [];

        for (let tag in node.child[tagName]) {
          if (node.child[tagName].hasOwnProperty(tag)) {
            jObj[tagName].push(convertToJson(node.child[tagName][tag], options, tagName));
          }
        }
      } else {
        const result = convertToJson(node.child[tagName][0], options, tagName);
        const asArray = options.arrayMode === true && typeof result === 'object' || util$3.isTagNameInArrayMode(tagName, options.arrayMode, parentTagName);
        jObj[tagName] = asArray ? [result] : result;
      }
    } //add value


    return jObj;
  };

  node2json.convertToJson = convertToJson;

  var xmlstr2xmlnode = {};

  var xmlNode$1 = function (tagname, parent, val) {
    this.tagname = tagname;
    this.parent = parent;
    this.child = {}; //child tags

    this.attrsMap = {}; //attributes map

    this.val = val; //text only

    this.addChild = function (child) {
      if (Array.isArray(this.child[child.tagname])) {
        //already presents
        this.child[child.tagname].push(child);
      } else {
        this.child[child.tagname] = [child];
      }
    };
  };

  const util$2 = util$4;
  const buildOptions$3 = util$4.buildOptions;
  const xmlNode = xmlNode$1;
  '<((!\\[CDATA\\[([\\s\\S]*?)(]]>))|((NAME:)?(NAME))([^>]*)>|((\\/)(NAME)\\s*>))([^<]*)'.replace(/NAME/g, util$2.nameRegexp); //const tagsRegx = new RegExp("<(\\/?[\\w:\\-\._]+)([^>]*)>(\\s*"+cdataRegx+")*([^<]+)?","g");
  //const tagsRegx = new RegExp("<(\\/?)((\\w*:)?([\\w:\\-\._]+))([^>]*)>([^<]*)("+cdataRegx+"([^<]*))*([^<]+)?","g");
  //polyfill

  if (!Number.parseInt && window.parseInt) {
    Number.parseInt = window.parseInt;
  }

  if (!Number.parseFloat && window.parseFloat) {
    Number.parseFloat = window.parseFloat;
  }

  const defaultOptions$2 = {
    attributeNamePrefix: '@_',
    attrNodeName: false,
    textNodeName: '#text',
    ignoreAttributes: true,
    ignoreNameSpace: false,
    allowBooleanAttributes: false,
    //a tag can have attributes without any value
    //ignoreRootElement : false,
    parseNodeValue: true,
    parseAttributeValue: false,
    arrayMode: false,
    trimValues: true,
    //Trim string values of tag and attributes
    cdataTagName: false,
    cdataPositionChar: '\\c',
    tagValueProcessor: function (a, tagName) {
      return a;
    },
    attrValueProcessor: function (a, attrName) {
      return a;
    },
    stopNodes: [] //decodeStrict: false,

  };
  xmlstr2xmlnode.defaultOptions = defaultOptions$2;
  const props$2 = ['attributeNamePrefix', 'attrNodeName', 'textNodeName', 'ignoreAttributes', 'ignoreNameSpace', 'allowBooleanAttributes', 'parseNodeValue', 'parseAttributeValue', 'arrayMode', 'trimValues', 'cdataTagName', 'cdataPositionChar', 'tagValueProcessor', 'attrValueProcessor', 'parseTrueNumberOnly', 'stopNodes'];
  xmlstr2xmlnode.props = props$2;
  /**
   * Trim -> valueProcessor -> parse value
   * @param {string} tagName
   * @param {string} val
   * @param {object} options
   */

  function processTagValue(tagName, val, options) {
    if (val) {
      if (options.trimValues) {
        val = val.trim();
      }

      val = options.tagValueProcessor(val, tagName);
      val = parseValue(val, options.parseNodeValue, options.parseTrueNumberOnly);
    }

    return val;
  }

  function resolveNameSpace(tagname, options) {
    if (options.ignoreNameSpace) {
      const tags = tagname.split(':');
      const prefix = tagname.charAt(0) === '/' ? '/' : '';

      if (tags[0] === 'xmlns') {
        return '';
      }

      if (tags.length === 2) {
        tagname = prefix + tags[1];
      }
    }

    return tagname;
  }

  function parseValue(val, shouldParse, parseTrueNumberOnly) {
    if (shouldParse && typeof val === 'string') {
      let parsed;

      if (val.trim() === '' || isNaN(val)) {
        parsed = val === 'true' ? true : val === 'false' ? false : val;
      } else {
        if (val.indexOf('0x') !== -1) {
          //support hexa decimal
          parsed = Number.parseInt(val, 16);
        } else if (val.indexOf('.') !== -1) {
          parsed = Number.parseFloat(val);
          val = val.replace(/\.?0+$/, "");
        } else {
          parsed = Number.parseInt(val, 10);
        }

        if (parseTrueNumberOnly) {
          parsed = String(parsed) === val ? parsed : val;
        }
      }

      return parsed;
    } else {
      if (util$2.isExist(val)) {
        return val;
      } else {
        return '';
      }
    }
  } //TODO: change regex to capture NS
  //const attrsRegx = new RegExp("([\\w\\-\\.\\:]+)\\s*=\\s*(['\"])((.|\n)*?)\\2","gm");


  const attrsRegx = new RegExp('([^\\s=]+)\\s*(=\\s*([\'"])(.*?)\\3)?', 'g');

  function buildAttributesMap(attrStr, options) {
    if (!options.ignoreAttributes && typeof attrStr === 'string') {
      attrStr = attrStr.replace(/\r?\n/g, ' '); //attrStr = attrStr || attrStr.trim();

      const matches = util$2.getAllMatches(attrStr, attrsRegx);
      const len = matches.length; //don't make it inline

      const attrs = {};

      for (let i = 0; i < len; i++) {
        const attrName = resolveNameSpace(matches[i][1], options);

        if (attrName.length) {
          if (matches[i][4] !== undefined) {
            if (options.trimValues) {
              matches[i][4] = matches[i][4].trim();
            }

            matches[i][4] = options.attrValueProcessor(matches[i][4], attrName);
            attrs[options.attributeNamePrefix + attrName] = parseValue(matches[i][4], options.parseAttributeValue, options.parseTrueNumberOnly);
          } else if (options.allowBooleanAttributes) {
            attrs[options.attributeNamePrefix + attrName] = true;
          }
        }
      }

      if (!Object.keys(attrs).length) {
        return;
      }

      if (options.attrNodeName) {
        const attrCollection = {};
        attrCollection[options.attrNodeName] = attrs;
        return attrCollection;
      }

      return attrs;
    }
  }

  const getTraversalObj = function (xmlData, options) {
    xmlData = xmlData.replace(/\r\n?/g, "\n");
    options = buildOptions$3(options, defaultOptions$2, props$2);
    const xmlObj = new xmlNode('!xml');
    let currentNode = xmlObj;
    let textData = ""; //function match(xmlData){

    for (let i = 0; i < xmlData.length; i++) {
      const ch = xmlData[i];

      if (ch === '<') {
        if (xmlData[i + 1] === '/') {
          //Closing Tag
          const closeIndex = findClosingIndex(xmlData, ">", i, "Closing Tag is not closed.");
          let tagName = xmlData.substring(i + 2, closeIndex).trim();

          if (options.ignoreNameSpace) {
            const colonIndex = tagName.indexOf(":");

            if (colonIndex !== -1) {
              tagName = tagName.substr(colonIndex + 1);
            }
          }
          /* if (currentNode.parent) {
            currentNode.parent.val = util.getValue(currentNode.parent.val) + '' + processTagValue2(tagName, textData , options);
          } */


          if (currentNode) {
            if (currentNode.val) {
              currentNode.val = util$2.getValue(currentNode.val) + '' + processTagValue(tagName, textData, options);
            } else {
              currentNode.val = processTagValue(tagName, textData, options);
            }
          }

          if (options.stopNodes.length && options.stopNodes.includes(currentNode.tagname)) {
            currentNode.child = [];

            if (currentNode.attrsMap == undefined) {
              currentNode.attrsMap = {};
            }

            currentNode.val = xmlData.substr(currentNode.startIndex + 1, i - currentNode.startIndex - 1);
          }

          currentNode = currentNode.parent;
          textData = "";
          i = closeIndex;
        } else if (xmlData[i + 1] === '?') {
          i = findClosingIndex(xmlData, "?>", i, "Pi Tag is not closed.");
        } else if (xmlData.substr(i + 1, 3) === '!--') {
          i = findClosingIndex(xmlData, "-->", i, "Comment is not closed.");
        } else if (xmlData.substr(i + 1, 2) === '!D') {
          const closeIndex = findClosingIndex(xmlData, ">", i, "DOCTYPE is not closed.");
          const tagExp = xmlData.substring(i, closeIndex);

          if (tagExp.indexOf("[") >= 0) {
            i = xmlData.indexOf("]>", i) + 1;
          } else {
            i = closeIndex;
          }
        } else if (xmlData.substr(i + 1, 2) === '![') {
          const closeIndex = findClosingIndex(xmlData, "]]>", i, "CDATA is not closed.") - 2;
          const tagExp = xmlData.substring(i + 9, closeIndex); //considerations
          //1. CDATA will always have parent node
          //2. A tag with CDATA is not a leaf node so it's value would be string type.

          if (textData) {
            currentNode.val = util$2.getValue(currentNode.val) + '' + processTagValue(currentNode.tagname, textData, options);
            textData = "";
          }

          if (options.cdataTagName) {
            //add cdata node
            const childNode = new xmlNode(options.cdataTagName, currentNode, tagExp);
            currentNode.addChild(childNode); //for backtracking

            currentNode.val = util$2.getValue(currentNode.val) + options.cdataPositionChar; //add rest value to parent node

            if (tagExp) {
              childNode.val = tagExp;
            }
          } else {
            currentNode.val = (currentNode.val || '') + (tagExp || '');
          }

          i = closeIndex + 2;
        } else {
          //Opening tag
          const result = closingIndexForOpeningTag(xmlData, i + 1);
          let tagExp = result.data;
          const closeIndex = result.index;
          const separatorIndex = tagExp.indexOf(" ");
          let tagName = tagExp;
          let shouldBuildAttributesMap = true;

          if (separatorIndex !== -1) {
            tagName = tagExp.substr(0, separatorIndex).replace(/\s\s*$/, '');
            tagExp = tagExp.substr(separatorIndex + 1);
          }

          if (options.ignoreNameSpace) {
            const colonIndex = tagName.indexOf(":");

            if (colonIndex !== -1) {
              tagName = tagName.substr(colonIndex + 1);
              shouldBuildAttributesMap = tagName !== result.data.substr(colonIndex + 1);
            }
          } //save text to parent node


          if (currentNode && textData) {
            if (currentNode.tagname !== '!xml') {
              currentNode.val = util$2.getValue(currentNode.val) + '' + processTagValue(currentNode.tagname, textData, options);
            }
          }

          if (tagExp.length > 0 && tagExp.lastIndexOf("/") === tagExp.length - 1) {
            //selfClosing tag
            if (tagName[tagName.length - 1] === "/") {
              //remove trailing '/'
              tagName = tagName.substr(0, tagName.length - 1);
              tagExp = tagName;
            } else {
              tagExp = tagExp.substr(0, tagExp.length - 1);
            }

            const childNode = new xmlNode(tagName, currentNode, '');

            if (tagName !== tagExp) {
              childNode.attrsMap = buildAttributesMap(tagExp, options);
            }

            currentNode.addChild(childNode);
          } else {
            //opening tag
            const childNode = new xmlNode(tagName, currentNode);

            if (options.stopNodes.length && options.stopNodes.includes(childNode.tagname)) {
              childNode.startIndex = closeIndex;
            }

            if (tagName !== tagExp && shouldBuildAttributesMap) {
              childNode.attrsMap = buildAttributesMap(tagExp, options);
            }

            currentNode.addChild(childNode);
            currentNode = childNode;
          }

          textData = "";
          i = closeIndex;
        }
      } else {
        textData += xmlData[i];
      }
    }

    return xmlObj;
  };

  function closingIndexForOpeningTag(data, i) {
    let attrBoundary;
    let tagExp = "";

    for (let index = i; index < data.length; index++) {
      let ch = data[index];

      if (attrBoundary) {
        if (ch === attrBoundary) attrBoundary = ""; //reset
      } else if (ch === '"' || ch === "'") {
        attrBoundary = ch;
      } else if (ch === '>') {
        return {
          data: tagExp,
          index: index
        };
      } else if (ch === '\t') {
        ch = " ";
      }

      tagExp += ch;
    }
  }

  function findClosingIndex(xmlData, str, i, errMsg) {
    const closingIndex = xmlData.indexOf(str, i);

    if (closingIndex === -1) {
      throw new Error(errMsg);
    } else {
      return closingIndex + str.length - 1;
    }
  }

  xmlstr2xmlnode.getTraversalObj = getTraversalObj;

  var validator = {};

  const util$1 = util$4;
  const defaultOptions$1 = {
    allowBooleanAttributes: false //A tag can have attributes without any value

  };
  const props$1 = ['allowBooleanAttributes']; //const tagsPattern = new RegExp("<\\/?([\\w:\\-_\.]+)\\s*\/?>","g");

  validator.validate = function (xmlData, options) {
    options = util$1.buildOptions(options, defaultOptions$1, props$1); //xmlData = xmlData.replace(/(\r\n|\n|\r)/gm,"");//make it single line
    //xmlData = xmlData.replace(/(^\s*<\?xml.*?\?>)/g,"");//Remove XML starting tag
    //xmlData = xmlData.replace(/(<!DOCTYPE[\s\w\"\.\/\-\:]+(\[.*\])*\s*>)/g,"");//Remove DOCTYPE

    const tags = [];
    let tagFound = false; //indicates that the root tag has been closed (aka. depth 0 has been reached)

    let reachedRoot = false;

    if (xmlData[0] === '\ufeff') {
      // check for byte order mark (BOM)
      xmlData = xmlData.substr(1);
    }

    for (let i = 0; i < xmlData.length; i++) {
      if (xmlData[i] === '<' && xmlData[i + 1] === '?') {
        i += 2;
        i = readPI(xmlData, i);
        if (i.err) return i;
      } else if (xmlData[i] === '<') {
        //starting of tag
        //read until you reach to '>' avoiding any '>' in attribute value
        i++;

        if (xmlData[i] === '!') {
          i = readCommentAndCDATA(xmlData, i);
          continue;
        } else {
          let closingTag = false;

          if (xmlData[i] === '/') {
            //closing tag
            closingTag = true;
            i++;
          } //read tagname


          let tagName = '';

          for (; i < xmlData.length && xmlData[i] !== '>' && xmlData[i] !== ' ' && xmlData[i] !== '\t' && xmlData[i] !== '\n' && xmlData[i] !== '\r'; i++) {
            tagName += xmlData[i];
          }

          tagName = tagName.trim(); //console.log(tagName);

          if (tagName[tagName.length - 1] === '/') {
            //self closing tag without attributes
            tagName = tagName.substring(0, tagName.length - 1); //continue;

            i--;
          }

          if (!validateTagName(tagName)) {
            let msg;

            if (tagName.trim().length === 0) {
              msg = "There is an unnecessary space between tag name and backward slash '</ ..'.";
            } else {
              msg = "Tag '" + tagName + "' is an invalid name.";
            }

            return getErrorObject('InvalidTag', msg, getLineNumberForPosition(xmlData, i));
          }

          const result = readAttributeStr(xmlData, i);

          if (result === false) {
            return getErrorObject('InvalidAttr', "Attributes for '" + tagName + "' have open quote.", getLineNumberForPosition(xmlData, i));
          }

          let attrStr = result.value;
          i = result.index;

          if (attrStr[attrStr.length - 1] === '/') {
            //self closing tag
            attrStr = attrStr.substring(0, attrStr.length - 1);
            const isValid = validateAttributeString(attrStr, options);

            if (isValid === true) {
              tagFound = true; //continue; //text may presents after self closing tag
            } else {
              //the result from the nested function returns the position of the error within the attribute
              //in order to get the 'true' error line, we need to calculate the position where the attribute begins (i - attrStr.length) and then add the position within the attribute
              //this gives us the absolute index in the entire xml, which we can use to find the line at last
              return getErrorObject(isValid.err.code, isValid.err.msg, getLineNumberForPosition(xmlData, i - attrStr.length + isValid.err.line));
            }
          } else if (closingTag) {
            if (!result.tagClosed) {
              return getErrorObject('InvalidTag', "Closing tag '" + tagName + "' doesn't have proper closing.", getLineNumberForPosition(xmlData, i));
            } else if (attrStr.trim().length > 0) {
              return getErrorObject('InvalidTag', "Closing tag '" + tagName + "' can't have attributes or invalid starting.", getLineNumberForPosition(xmlData, i));
            } else {
              const otg = tags.pop();

              if (tagName !== otg) {
                return getErrorObject('InvalidTag', "Closing tag '" + otg + "' is expected inplace of '" + tagName + "'.", getLineNumberForPosition(xmlData, i));
              } //when there are no more tags, we reached the root level.


              if (tags.length == 0) {
                reachedRoot = true;
              }
            }
          } else {
            const isValid = validateAttributeString(attrStr, options);

            if (isValid !== true) {
              //the result from the nested function returns the position of the error within the attribute
              //in order to get the 'true' error line, we need to calculate the position where the attribute begins (i - attrStr.length) and then add the position within the attribute
              //this gives us the absolute index in the entire xml, which we can use to find the line at last
              return getErrorObject(isValid.err.code, isValid.err.msg, getLineNumberForPosition(xmlData, i - attrStr.length + isValid.err.line));
            } //if the root level has been reached before ...


            if (reachedRoot === true) {
              return getErrorObject('InvalidXml', 'Multiple possible root nodes found.', getLineNumberForPosition(xmlData, i));
            } else {
              tags.push(tagName);
            }

            tagFound = true;
          } //skip tag text value
          //It may include comments and CDATA value


          for (i++; i < xmlData.length; i++) {
            if (xmlData[i] === '<') {
              if (xmlData[i + 1] === '!') {
                //comment or CADATA
                i++;
                i = readCommentAndCDATA(xmlData, i);
                continue;
              } else if (xmlData[i + 1] === '?') {
                i = readPI(xmlData, ++i);
                if (i.err) return i;
              } else {
                break;
              }
            } else if (xmlData[i] === '&') {
              const afterAmp = validateAmpersand(xmlData, i);
              if (afterAmp == -1) return getErrorObject('InvalidChar', "char '&' is not expected.", getLineNumberForPosition(xmlData, i));
              i = afterAmp;
            }
          } //end of reading tag text value


          if (xmlData[i] === '<') {
            i--;
          }
        }
      } else {
        if (xmlData[i] === ' ' || xmlData[i] === '\t' || xmlData[i] === '\n' || xmlData[i] === '\r') {
          continue;
        }

        return getErrorObject('InvalidChar', "char '" + xmlData[i] + "' is not expected.", getLineNumberForPosition(xmlData, i));
      }
    }

    if (!tagFound) {
      return getErrorObject('InvalidXml', 'Start tag expected.', 1);
    } else if (tags.length > 0) {
      return getErrorObject('InvalidXml', "Invalid '" + JSON.stringify(tags, null, 4).replace(/\r?\n/g, '') + "' found.", 1);
    }

    return true;
  };
  /**
   * Read Processing insstructions and skip
   * @param {*} xmlData
   * @param {*} i
   */


  function readPI(xmlData, i) {
    var start = i;

    for (; i < xmlData.length; i++) {
      if (xmlData[i] == '?' || xmlData[i] == ' ') {
        //tagname
        var tagname = xmlData.substr(start, i - start);

        if (i > 5 && tagname === 'xml') {
          return getErrorObject('InvalidXml', 'XML declaration allowed only at the start of the document.', getLineNumberForPosition(xmlData, i));
        } else if (xmlData[i] == '?' && xmlData[i + 1] == '>') {
          //check if valid attribut string
          i++;
          break;
        } else {
          continue;
        }
      }
    }

    return i;
  }

  function readCommentAndCDATA(xmlData, i) {
    if (xmlData.length > i + 5 && xmlData[i + 1] === '-' && xmlData[i + 2] === '-') {
      //comment
      for (i += 3; i < xmlData.length; i++) {
        if (xmlData[i] === '-' && xmlData[i + 1] === '-' && xmlData[i + 2] === '>') {
          i += 2;
          break;
        }
      }
    } else if (xmlData.length > i + 8 && xmlData[i + 1] === 'D' && xmlData[i + 2] === 'O' && xmlData[i + 3] === 'C' && xmlData[i + 4] === 'T' && xmlData[i + 5] === 'Y' && xmlData[i + 6] === 'P' && xmlData[i + 7] === 'E') {
      let angleBracketsCount = 1;

      for (i += 8; i < xmlData.length; i++) {
        if (xmlData[i] === '<') {
          angleBracketsCount++;
        } else if (xmlData[i] === '>') {
          angleBracketsCount--;

          if (angleBracketsCount === 0) {
            break;
          }
        }
      }
    } else if (xmlData.length > i + 9 && xmlData[i + 1] === '[' && xmlData[i + 2] === 'C' && xmlData[i + 3] === 'D' && xmlData[i + 4] === 'A' && xmlData[i + 5] === 'T' && xmlData[i + 6] === 'A' && xmlData[i + 7] === '[') {
      for (i += 8; i < xmlData.length; i++) {
        if (xmlData[i] === ']' && xmlData[i + 1] === ']' && xmlData[i + 2] === '>') {
          i += 2;
          break;
        }
      }
    }

    return i;
  }

  var doubleQuote = '"';
  var singleQuote = "'";
  /**
   * Keep reading xmlData until '<' is found outside the attribute value.
   * @param {string} xmlData
   * @param {number} i
   */

  function readAttributeStr(xmlData, i) {
    let attrStr = '';
    let startChar = '';
    let tagClosed = false;

    for (; i < xmlData.length; i++) {
      if (xmlData[i] === doubleQuote || xmlData[i] === singleQuote) {
        if (startChar === '') {
          startChar = xmlData[i];
        } else if (startChar !== xmlData[i]) {
          //if vaue is enclosed with double quote then single quotes are allowed inside the value and vice versa
          continue;
        } else {
          startChar = '';
        }
      } else if (xmlData[i] === '>') {
        if (startChar === '') {
          tagClosed = true;
          break;
        }
      }

      attrStr += xmlData[i];
    }

    if (startChar !== '') {
      return false;
    }

    return {
      value: attrStr,
      index: i,
      tagClosed: tagClosed
    };
  }
  /**
   * Select all the attributes whether valid or invalid.
   */


  const validAttrStrRegxp = new RegExp('(\\s*)([^\\s=]+)(\\s*=)?(\\s*([\'"])(([\\s\\S])*?)\\5)?', 'g'); //attr, ="sd", a="amit's", a="sd"b="saf", ab  cd=""

  function validateAttributeString(attrStr, options) {
    //console.log("start:"+attrStr+":end");
    //if(attrStr.trim().length === 0) return true; //empty string
    const matches = util$1.getAllMatches(attrStr, validAttrStrRegxp);
    const attrNames = {};

    for (let i = 0; i < matches.length; i++) {
      if (matches[i][1].length === 0) {
        //nospace before attribute name: a="sd"b="saf"
        return getErrorObject('InvalidAttr', "Attribute '" + matches[i][2] + "' has no space in starting.", getPositionFromMatch(attrStr, matches[i][0]));
      } else if (matches[i][3] === undefined && !options.allowBooleanAttributes) {
        //independent attribute: ab
        return getErrorObject('InvalidAttr', "boolean attribute '" + matches[i][2] + "' is not allowed.", getPositionFromMatch(attrStr, matches[i][0]));
      }
      /* else if(matches[i][6] === undefined){//attribute without value: ab=
                      return { err: { code:"InvalidAttr",msg:"attribute " + matches[i][2] + " has no value assigned."}};
                  } */


      const attrName = matches[i][2];

      if (!validateAttrName(attrName)) {
        return getErrorObject('InvalidAttr', "Attribute '" + attrName + "' is an invalid name.", getPositionFromMatch(attrStr, matches[i][0]));
      }

      if (!attrNames.hasOwnProperty(attrName)) {
        //check for duplicate attribute.
        attrNames[attrName] = 1;
      } else {
        return getErrorObject('InvalidAttr', "Attribute '" + attrName + "' is repeated.", getPositionFromMatch(attrStr, matches[i][0]));
      }
    }

    return true;
  }

  function validateNumberAmpersand(xmlData, i) {
    let re = /\d/;

    if (xmlData[i] === 'x') {
      i++;
      re = /[\da-fA-F]/;
    }

    for (; i < xmlData.length; i++) {
      if (xmlData[i] === ';') return i;
      if (!xmlData[i].match(re)) break;
    }

    return -1;
  }

  function validateAmpersand(xmlData, i) {
    // https://www.w3.org/TR/xml/#dt-charref
    i++;
    if (xmlData[i] === ';') return -1;

    if (xmlData[i] === '#') {
      i++;
      return validateNumberAmpersand(xmlData, i);
    }

    let count = 0;

    for (; i < xmlData.length; i++, count++) {
      if (xmlData[i].match(/\w/) && count < 20) continue;
      if (xmlData[i] === ';') break;
      return -1;
    }

    return i;
  }

  function getErrorObject(code, message, lineNumber) {
    return {
      err: {
        code: code,
        msg: message,
        line: lineNumber
      }
    };
  }

  function validateAttrName(attrName) {
    return util$1.isName(attrName);
  } // const startsWithXML = /^xml/i;


  function validateTagName(tagname) {
    return util$1.isName(tagname)
    /* && !tagname.match(startsWithXML) */
    ;
  } //this function returns the line number for the character at the given index


  function getLineNumberForPosition(xmlData, index) {
    var lines = xmlData.substring(0, index).split(/\r?\n/);
    return lines.length;
  } //this function returns the position of the last character of match within attrStr


  function getPositionFromMatch(attrStr, match) {
    return attrStr.indexOf(match) + match.length;
  }

  var nimndata = {};

  const char = function (a) {
    return String.fromCharCode(a);
  };

  const chars = {
    nilChar: char(176),
    missingChar: char(201),
    nilPremitive: char(175),
    missingPremitive: char(200),
    emptyChar: char(178),
    emptyValue: char(177),
    //empty Premitive
    boundryChar: char(179),
    objStart: char(198),
    arrStart: char(204),
    arrayEnd: char(185)
  };
  const charsArr = [chars.nilChar, chars.nilPremitive, chars.missingChar, chars.missingPremitive, chars.boundryChar, chars.emptyChar, chars.emptyValue, chars.arrayEnd, chars.objStart, chars.arrStart];

  const _e = function (node, e_schema, options) {
    if (typeof e_schema === 'string') {
      //premitive
      if (node && node[0] && node[0].val !== undefined) {
        return getValue(node[0].val);
      } else {
        return getValue(node);
      }
    } else {
      const hasValidData = hasData(node);

      if (hasValidData === true) {
        let str = '';

        if (Array.isArray(e_schema)) {
          //attributes can't be repeated. hence check in children tags only
          str += chars.arrStart;
          const itemSchema = e_schema[0]; //var itemSchemaType = itemSchema;

          const arr_len = node.length;

          if (typeof itemSchema === 'string') {
            for (let arr_i = 0; arr_i < arr_len; arr_i++) {
              const r = getValue(node[arr_i].val);
              str = processValue(str, r);
            }
          } else {
            for (let arr_i = 0; arr_i < arr_len; arr_i++) {
              const r = _e(node[arr_i], itemSchema, options);

              str = processValue(str, r);
            }
          }

          str += chars.arrayEnd; //indicates that next item is not array item
        } else {
          //object
          str += chars.objStart;
          const keys = Object.keys(e_schema);

          if (Array.isArray(node)) {
            node = node[0];
          }

          for (let i in keys) {
            const key = keys[i]; //a property defined in schema can be present either in attrsMap or children tags
            //options.textNodeName will not present in both maps, take it's value from val
            //options.attrNodeName will be present in attrsMap

            let r;

            if (!options.ignoreAttributes && node.attrsMap && node.attrsMap[key]) {
              r = _e(node.attrsMap[key], e_schema[key], options);
            } else if (key === options.textNodeName) {
              r = _e(node.val, e_schema[key], options);
            } else {
              r = _e(node.child[key], e_schema[key], options);
            }

            str = processValue(str, r);
          }
        }

        return str;
      } else {
        return hasValidData;
      }
    }
  };

  const getValue = function (a
  /*, type*/
  ) {
    switch (a) {
      case undefined:
        return chars.missingPremitive;

      case null:
        return chars.nilPremitive;

      case '':
        return chars.emptyValue;

      default:
        return a;
    }
  };

  const processValue = function (str, r) {
    if (!isAppChar(r[0]) && !isAppChar(str[str.length - 1])) {
      str += chars.boundryChar;
    }

    return str + r;
  };

  const isAppChar = function (ch) {
    return charsArr.indexOf(ch) !== -1;
  };

  function hasData(jObj) {
    if (jObj === undefined) {
      return chars.missingChar;
    } else if (jObj === null) {
      return chars.nilChar;
    } else if (jObj.child && Object.keys(jObj.child).length === 0 && (!jObj.attrsMap || Object.keys(jObj.attrsMap).length === 0)) {
      return chars.emptyChar;
    } else {
      return true;
    }
  }

  const x2j$1 = xmlstr2xmlnode;
  const buildOptions$2 = util$4.buildOptions;

  const convert2nimn = function (node, e_schema, options) {
    options = buildOptions$2(options, x2j$1.defaultOptions, x2j$1.props);
    return _e(node, e_schema, options);
  };

  nimndata.convert2nimn = convert2nimn;

  var node2json_str = {};

  const util = util$4;
  const buildOptions$1 = util$4.buildOptions;
  const x2j = xmlstr2xmlnode; //TODO: do it later

  const convertToJsonString = function (node, options) {
    options = buildOptions$1(options, x2j.defaultOptions, x2j.props);
    options.indentBy = options.indentBy || '';
    return _cToJsonStr(node, options);
  };

  const _cToJsonStr = function (node, options, level) {
    let jObj = '{'; //traver through all the children

    const keys = Object.keys(node.child);

    for (let index = 0; index < keys.length; index++) {
      var tagname = keys[index];

      if (node.child[tagname] && node.child[tagname].length > 1) {
        jObj += '"' + tagname + '" : [ ';

        for (var tag in node.child[tagname]) {
          jObj += _cToJsonStr(node.child[tagname][tag], options) + ' , ';
        }

        jObj = jObj.substr(0, jObj.length - 1) + ' ] '; //remove extra comma in last
      } else {
        jObj += '"' + tagname + '" : ' + _cToJsonStr(node.child[tagname][0], options) + ' ,';
      }
    }

    util.merge(jObj, node.attrsMap); //add attrsMap as new children

    if (util.isEmptyObject(jObj)) {
      return util.isExist(node.val) ? node.val : '';
    } else {
      if (util.isExist(node.val)) {
        if (!(typeof node.val === 'string' && (node.val === '' || node.val === options.cdataPositionChar))) {
          jObj += '"' + options.textNodeName + '" : ' + stringval(node.val);
        }
      }
    } //add value


    if (jObj[jObj.length - 1] === ',') {
      jObj = jObj.substr(0, jObj.length - 2);
    }

    return jObj + '}';
  };

  function stringval(v) {
    if (v === true || v === false || !isNaN(v)) {
      return v;
    } else {
      return '"' + v + '"';
    }
  }

  node2json_str.convertToJsonString = convertToJsonString;

  const buildOptions = util$4.buildOptions;
  const defaultOptions = {
    attributeNamePrefix: '@_',
    attrNodeName: false,
    textNodeName: '#text',
    ignoreAttributes: true,
    cdataTagName: false,
    cdataPositionChar: '\\c',
    format: false,
    indentBy: '  ',
    supressEmptyNode: false,
    tagValueProcessor: function (a) {
      return a;
    },
    attrValueProcessor: function (a) {
      return a;
    }
  };
  const props = ['attributeNamePrefix', 'attrNodeName', 'textNodeName', 'ignoreAttributes', 'cdataTagName', 'cdataPositionChar', 'format', 'indentBy', 'supressEmptyNode', 'tagValueProcessor', 'attrValueProcessor'];

  function Parser(options) {
    this.options = buildOptions(options, defaultOptions, props);

    if (this.options.ignoreAttributes || this.options.attrNodeName) {
      this.isAttribute = function () {
        return false;
      };
    } else {
      this.attrPrefixLen = this.options.attributeNamePrefix.length;
      this.isAttribute = isAttribute;
    }

    if (this.options.cdataTagName) {
      this.isCDATA = isCDATA;
    } else {
      this.isCDATA = function () {
        return false;
      };
    }

    this.replaceCDATAstr = replaceCDATAstr;
    this.replaceCDATAarr = replaceCDATAarr;

    if (this.options.format) {
      this.indentate = indentate;
      this.tagEndChar = '>\n';
      this.newLine = '\n';
    } else {
      this.indentate = function () {
        return '';
      };

      this.tagEndChar = '>';
      this.newLine = '';
    }

    if (this.options.supressEmptyNode) {
      this.buildTextNode = buildEmptyTextNode;
      this.buildObjNode = buildEmptyObjNode;
    } else {
      this.buildTextNode = buildTextValNode;
      this.buildObjNode = buildObjectNode;
    }

    this.buildTextValNode = buildTextValNode;
    this.buildObjectNode = buildObjectNode;
  }

  Parser.prototype.parse = function (jObj) {
    return this.j2x(jObj, 0).val;
  };

  Parser.prototype.j2x = function (jObj, level) {
    let attrStr = '';
    let val = '';
    const keys = Object.keys(jObj);
    const len = keys.length;

    for (let i = 0; i < len; i++) {
      const key = keys[i];

      if (typeof jObj[key] === 'undefined') ; else if (jObj[key] === null) {
        val += this.indentate(level) + '<' + key + '/' + this.tagEndChar;
      } else if (jObj[key] instanceof Date) {
        val += this.buildTextNode(jObj[key], key, '', level);
      } else if (typeof jObj[key] !== 'object') {
        //premitive type
        const attr = this.isAttribute(key);

        if (attr) {
          attrStr += ' ' + attr + '="' + this.options.attrValueProcessor('' + jObj[key]) + '"';
        } else if (this.isCDATA(key)) {
          if (jObj[this.options.textNodeName]) {
            val += this.replaceCDATAstr(jObj[this.options.textNodeName], jObj[key]);
          } else {
            val += this.replaceCDATAstr('', jObj[key]);
          }
        } else {
          //tag value
          if (key === this.options.textNodeName) {
            if (jObj[this.options.cdataTagName]) ; else {
              val += this.options.tagValueProcessor('' + jObj[key]);
            }
          } else {
            val += this.buildTextNode(jObj[key], key, '', level);
          }
        }
      } else if (Array.isArray(jObj[key])) {
        //repeated nodes
        if (this.isCDATA(key)) {
          val += this.indentate(level);

          if (jObj[this.options.textNodeName]) {
            val += this.replaceCDATAarr(jObj[this.options.textNodeName], jObj[key]);
          } else {
            val += this.replaceCDATAarr('', jObj[key]);
          }
        } else {
          //nested nodes
          const arrLen = jObj[key].length;

          for (let j = 0; j < arrLen; j++) {
            const item = jObj[key][j];

            if (typeof item === 'undefined') ; else if (item === null) {
              val += this.indentate(level) + '<' + key + '/' + this.tagEndChar;
            } else if (typeof item === 'object') {
              const result = this.j2x(item, level + 1);
              val += this.buildObjNode(result.val, key, result.attrStr, level);
            } else {
              val += this.buildTextNode(item, key, '', level);
            }
          }
        }
      } else {
        //nested node
        if (this.options.attrNodeName && key === this.options.attrNodeName) {
          const Ks = Object.keys(jObj[key]);
          const L = Ks.length;

          for (let j = 0; j < L; j++) {
            attrStr += ' ' + Ks[j] + '="' + this.options.attrValueProcessor('' + jObj[key][Ks[j]]) + '"';
          }
        } else {
          const result = this.j2x(jObj[key], level + 1);
          val += this.buildObjNode(result.val, key, result.attrStr, level);
        }
      }
    }

    return {
      attrStr: attrStr,
      val: val
    };
  };

  function replaceCDATAstr(str, cdata) {
    str = this.options.tagValueProcessor('' + str);

    if (this.options.cdataPositionChar === '' || str === '') {
      return str + '<![CDATA[' + cdata + ']]' + this.tagEndChar;
    } else {
      return str.replace(this.options.cdataPositionChar, '<![CDATA[' + cdata + ']]' + this.tagEndChar);
    }
  }

  function replaceCDATAarr(str, cdata) {
    str = this.options.tagValueProcessor('' + str);

    if (this.options.cdataPositionChar === '' || str === '') {
      return str + '<![CDATA[' + cdata.join(']]><![CDATA[') + ']]' + this.tagEndChar;
    } else {
      for (let v in cdata) {
        str = str.replace(this.options.cdataPositionChar, '<![CDATA[' + cdata[v] + ']]>');
      }

      return str + this.newLine;
    }
  }

  function buildObjectNode(val, key, attrStr, level) {
    if (attrStr && !val.includes('<')) {
      return this.indentate(level) + '<' + key + attrStr + '>' + val + //+ this.newLine
      // + this.indentate(level)
      '</' + key + this.tagEndChar;
    } else {
      return this.indentate(level) + '<' + key + attrStr + this.tagEndChar + val + //+ this.newLine
      this.indentate(level) + '</' + key + this.tagEndChar;
    }
  }

  function buildEmptyObjNode(val, key, attrStr, level) {
    if (val !== '') {
      return this.buildObjectNode(val, key, attrStr, level);
    } else {
      return this.indentate(level) + '<' + key + attrStr + '/' + this.tagEndChar; //+ this.newLine
    }
  }

  function buildTextValNode(val, key, attrStr, level) {
    return this.indentate(level) + '<' + key + attrStr + '>' + this.options.tagValueProcessor(val) + '</' + key + this.tagEndChar;
  }

  function buildEmptyTextNode(val, key, attrStr, level) {
    if (val !== '') {
      return this.buildTextValNode(val, key, attrStr, level);
    } else {
      return this.indentate(level) + '<' + key + attrStr + '/' + this.tagEndChar;
    }
  }

  function indentate(level) {
    return this.options.indentBy.repeat(level);
  }

  function isAttribute(name
  /*, options*/
  ) {
    if (name.startsWith(this.options.attributeNamePrefix)) {
      return name.substr(this.attrPrefixLen);
    } else {
      return false;
    }
  }

  function isCDATA(name) {
    return name === this.options.cdataTagName;
  } //formatting
  //indentation
  //\n after each closing or self closing tag


  var json2xml = Parser;

  (function (exports) {

    const nodeToJson = node2json;
    const xmlToNodeobj = xmlstr2xmlnode;
    const x2xmlnode = xmlstr2xmlnode;
    const buildOptions = util$4.buildOptions;
    const validator$1 = validator;

    exports.parse = function (xmlData, options, validationOption) {
      if (validationOption) {
        if (validationOption === true) validationOption = {};
        const result = validator$1.validate(xmlData, validationOption);

        if (result !== true) {
          throw Error(result.err.msg);
        }
      }

      options = buildOptions(options, x2xmlnode.defaultOptions, x2xmlnode.props);
      const traversableObj = xmlToNodeobj.getTraversalObj(xmlData, options); //print(traversableObj, "  ");

      return nodeToJson.convertToJson(traversableObj, options);
    };

    exports.convertTonimn = nimndata.convert2nimn;
    exports.getTraversalObj = xmlToNodeobj.getTraversalObj;
    exports.convertToJson = nodeToJson.convertToJson;
    exports.convertToJsonString = node2json_str.convertToJsonString;
    exports.validate = validator$1.validate;
    exports.j2xParser = json2xml;

    exports.parseToNimn = function (xmlData, schema, options) {
      return exports.convertTonimn(exports.getTraversalObj(xmlData, options), schema, options);
    };
  })(parser$1);

  var parser = parser$1;

  var jszip_min = {exports: {}};

  /*!

  JSZip v3.7.1 - A JavaScript class for generating and reading zip files
  <http://stuartk.com/jszip>

  (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
  Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

  JSZip uses the library pako released under the MIT license :
  https://github.com/nodeca/pako/blob/master/LICENSE
  */

  (function (module, exports) {
    !function (t) {
      module.exports = t();
    }(function () {
      return function s(a, o, h) {
        function u(r, t) {
          if (!o[r]) {
            if (!a[r]) {
              var e = "function" == typeof commonjsRequire && commonjsRequire;
              if (!t && e) return e(r, !0);
              if (l) return l(r, !0);
              var i = new Error("Cannot find module '" + r + "'");
              throw i.code = "MODULE_NOT_FOUND", i;
            }

            var n = o[r] = {
              exports: {}
            };
            a[r][0].call(n.exports, function (t) {
              var e = a[r][1][t];
              return u(e || t);
            }, n, n.exports, s, a, o, h);
          }

          return o[r].exports;
        }

        for (var l = "function" == typeof commonjsRequire && commonjsRequire, t = 0; t < h.length; t++) u(h[t]);

        return u;
      }({
        1: [function (t, e, r) {

          var c = t("./utils"),
              d = t("./support"),
              p = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
          r.encode = function (t) {
            for (var e, r, i, n, s, a, o, h = [], u = 0, l = t.length, f = l, d = "string" !== c.getTypeOf(t); u < t.length;) f = l - u, i = d ? (e = t[u++], r = u < l ? t[u++] : 0, u < l ? t[u++] : 0) : (e = t.charCodeAt(u++), r = u < l ? t.charCodeAt(u++) : 0, u < l ? t.charCodeAt(u++) : 0), n = e >> 2, s = (3 & e) << 4 | r >> 4, a = 1 < f ? (15 & r) << 2 | i >> 6 : 64, o = 2 < f ? 63 & i : 64, h.push(p.charAt(n) + p.charAt(s) + p.charAt(a) + p.charAt(o));

            return h.join("");
          }, r.decode = function (t) {
            var e,
                r,
                i,
                n,
                s,
                a,
                o = 0,
                h = 0,
                u = "data:";
            if (t.substr(0, u.length) === u) throw new Error("Invalid base64 input, it looks like a data url.");
            var l,
                f = 3 * (t = t.replace(/[^A-Za-z0-9\+\/\=]/g, "")).length / 4;
            if (t.charAt(t.length - 1) === p.charAt(64) && f--, t.charAt(t.length - 2) === p.charAt(64) && f--, f % 1 != 0) throw new Error("Invalid base64 input, bad content length.");

            for (l = d.uint8array ? new Uint8Array(0 | f) : new Array(0 | f); o < t.length;) e = p.indexOf(t.charAt(o++)) << 2 | (n = p.indexOf(t.charAt(o++))) >> 4, r = (15 & n) << 4 | (s = p.indexOf(t.charAt(o++))) >> 2, i = (3 & s) << 6 | (a = p.indexOf(t.charAt(o++))), l[h++] = e, 64 !== s && (l[h++] = r), 64 !== a && (l[h++] = i);

            return l;
          };
        }, {
          "./support": 30,
          "./utils": 32
        }],
        2: [function (t, e, r) {

          var i = t("./external"),
              n = t("./stream/DataWorker"),
              s = t("./stream/Crc32Probe"),
              a = t("./stream/DataLengthProbe");

          function o(t, e, r, i, n) {
            this.compressedSize = t, this.uncompressedSize = e, this.crc32 = r, this.compression = i, this.compressedContent = n;
          }

          o.prototype = {
            getContentWorker: function () {
              var t = new n(i.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),
                  e = this;
              return t.on("end", function () {
                if (this.streamInfo.data_length !== e.uncompressedSize) throw new Error("Bug : uncompressed data size mismatch");
              }), t;
            },
            getCompressedWorker: function () {
              return new n(i.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize", this.compressedSize).withStreamInfo("uncompressedSize", this.uncompressedSize).withStreamInfo("crc32", this.crc32).withStreamInfo("compression", this.compression);
            }
          }, o.createWorkerFrom = function (t, e, r) {
            return t.pipe(new s()).pipe(new a("uncompressedSize")).pipe(e.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression", e);
          }, e.exports = o;
        }, {
          "./external": 6,
          "./stream/Crc32Probe": 25,
          "./stream/DataLengthProbe": 26,
          "./stream/DataWorker": 27
        }],
        3: [function (t, e, r) {

          var i = t("./stream/GenericWorker");
          r.STORE = {
            magic: "\0\0",
            compressWorker: function (t) {
              return new i("STORE compression");
            },
            uncompressWorker: function () {
              return new i("STORE decompression");
            }
          }, r.DEFLATE = t("./flate");
        }, {
          "./flate": 7,
          "./stream/GenericWorker": 28
        }],
        4: [function (t, e, r) {

          var i = t("./utils");

          var o = function () {
            for (var t, e = [], r = 0; r < 256; r++) {
              t = r;

              for (var i = 0; i < 8; i++) t = 1 & t ? 3988292384 ^ t >>> 1 : t >>> 1;

              e[r] = t;
            }

            return e;
          }();

          e.exports = function (t, e) {
            return void 0 !== t && t.length ? "string" !== i.getTypeOf(t) ? function (t, e, r, i) {
              var n = o,
                  s = i + r;
              t ^= -1;

              for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e[a])];

              return -1 ^ t;
            }(0 | e, t, t.length, 0) : function (t, e, r, i) {
              var n = o,
                  s = i + r;
              t ^= -1;

              for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e.charCodeAt(a))];

              return -1 ^ t;
            }(0 | e, t, t.length, 0) : 0;
          };
        }, {
          "./utils": 32
        }],
        5: [function (t, e, r) {

          r.base64 = !1, r.binary = !1, r.dir = !1, r.createFolders = !0, r.date = null, r.compression = null, r.compressionOptions = null, r.comment = null, r.unixPermissions = null, r.dosPermissions = null;
        }, {}],
        6: [function (t, e, r) {

          var i = null;
          i = "undefined" != typeof Promise ? Promise : t("lie"), e.exports = {
            Promise: i
          };
        }, {
          lie: 37
        }],
        7: [function (t, e, r) {

          var i = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Uint32Array,
              n = t("pako"),
              s = t("./utils"),
              a = t("./stream/GenericWorker"),
              o = i ? "uint8array" : "array";

          function h(t, e) {
            a.call(this, "FlateWorker/" + t), this._pako = null, this._pakoAction = t, this._pakoOptions = e, this.meta = {};
          }

          r.magic = "\b\0", s.inherits(h, a), h.prototype.processChunk = function (t) {
            this.meta = t.meta, null === this._pako && this._createPako(), this._pako.push(s.transformTo(o, t.data), !1);
          }, h.prototype.flush = function () {
            a.prototype.flush.call(this), null === this._pako && this._createPako(), this._pako.push([], !0);
          }, h.prototype.cleanUp = function () {
            a.prototype.cleanUp.call(this), this._pako = null;
          }, h.prototype._createPako = function () {
            this._pako = new n[this._pakoAction]({
              raw: !0,
              level: this._pakoOptions.level || -1
            });
            var e = this;

            this._pako.onData = function (t) {
              e.push({
                data: t,
                meta: e.meta
              });
            };
          }, r.compressWorker = function (t) {
            return new h("Deflate", t);
          }, r.uncompressWorker = function () {
            return new h("Inflate", {});
          };
        }, {
          "./stream/GenericWorker": 28,
          "./utils": 32,
          pako: 38
        }],
        8: [function (t, e, r) {

          function A(t, e) {
            var r,
                i = "";

            for (r = 0; r < e; r++) i += String.fromCharCode(255 & t), t >>>= 8;

            return i;
          }

          function i(t, e, r, i, n, s) {
            var a,
                o,
                h = t.file,
                u = t.compression,
                l = s !== O.utf8encode,
                f = I.transformTo("string", s(h.name)),
                d = I.transformTo("string", O.utf8encode(h.name)),
                c = h.comment,
                p = I.transformTo("string", s(c)),
                m = I.transformTo("string", O.utf8encode(c)),
                _ = d.length !== h.name.length,
                g = m.length !== c.length,
                b = "",
                v = "",
                y = "",
                w = h.dir,
                k = h.date,
                x = {
              crc32: 0,
              compressedSize: 0,
              uncompressedSize: 0
            };

            e && !r || (x.crc32 = t.crc32, x.compressedSize = t.compressedSize, x.uncompressedSize = t.uncompressedSize);
            var S = 0;
            e && (S |= 8), l || !_ && !g || (S |= 2048);
            var z = 0,
                C = 0;
            w && (z |= 16), "UNIX" === n ? (C = 798, z |= function (t, e) {
              var r = t;
              return t || (r = e ? 16893 : 33204), (65535 & r) << 16;
            }(h.unixPermissions, w)) : (C = 20, z |= function (t) {
              return 63 & (t || 0);
            }(h.dosPermissions)), a = k.getUTCHours(), a <<= 6, a |= k.getUTCMinutes(), a <<= 5, a |= k.getUTCSeconds() / 2, o = k.getUTCFullYear() - 1980, o <<= 4, o |= k.getUTCMonth() + 1, o <<= 5, o |= k.getUTCDate(), _ && (v = A(1, 1) + A(B(f), 4) + d, b += "up" + A(v.length, 2) + v), g && (y = A(1, 1) + A(B(p), 4) + m, b += "uc" + A(y.length, 2) + y);
            var E = "";
            return E += "\n\0", E += A(S, 2), E += u.magic, E += A(a, 2), E += A(o, 2), E += A(x.crc32, 4), E += A(x.compressedSize, 4), E += A(x.uncompressedSize, 4), E += A(f.length, 2), E += A(b.length, 2), {
              fileRecord: R.LOCAL_FILE_HEADER + E + f + b,
              dirRecord: R.CENTRAL_FILE_HEADER + A(C, 2) + E + A(p.length, 2) + "\0\0\0\0" + A(z, 4) + A(i, 4) + f + b + p
            };
          }

          var I = t("../utils"),
              n = t("../stream/GenericWorker"),
              O = t("../utf8"),
              B = t("../crc32"),
              R = t("../signature");

          function s(t, e, r, i) {
            n.call(this, "ZipFileWorker"), this.bytesWritten = 0, this.zipComment = e, this.zipPlatform = r, this.encodeFileName = i, this.streamFiles = t, this.accumulate = !1, this.contentBuffer = [], this.dirRecords = [], this.currentSourceOffset = 0, this.entriesCount = 0, this.currentFile = null, this._sources = [];
          }

          I.inherits(s, n), s.prototype.push = function (t) {
            var e = t.meta.percent || 0,
                r = this.entriesCount,
                i = this._sources.length;
            this.accumulate ? this.contentBuffer.push(t) : (this.bytesWritten += t.data.length, n.prototype.push.call(this, {
              data: t.data,
              meta: {
                currentFile: this.currentFile,
                percent: r ? (e + 100 * (r - i - 1)) / r : 100
              }
            }));
          }, s.prototype.openedSource = function (t) {
            this.currentSourceOffset = this.bytesWritten, this.currentFile = t.file.name;
            var e = this.streamFiles && !t.file.dir;

            if (e) {
              var r = i(t, e, !1, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
              this.push({
                data: r.fileRecord,
                meta: {
                  percent: 0
                }
              });
            } else this.accumulate = !0;
          }, s.prototype.closedSource = function (t) {
            this.accumulate = !1;
            var e = this.streamFiles && !t.file.dir,
                r = i(t, e, !0, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
            if (this.dirRecords.push(r.dirRecord), e) this.push({
              data: function (t) {
                return R.DATA_DESCRIPTOR + A(t.crc32, 4) + A(t.compressedSize, 4) + A(t.uncompressedSize, 4);
              }(t),
              meta: {
                percent: 100
              }
            });else for (this.push({
              data: r.fileRecord,
              meta: {
                percent: 0
              }
            }); this.contentBuffer.length;) this.push(this.contentBuffer.shift());
            this.currentFile = null;
          }, s.prototype.flush = function () {
            for (var t = this.bytesWritten, e = 0; e < this.dirRecords.length; e++) this.push({
              data: this.dirRecords[e],
              meta: {
                percent: 100
              }
            });

            var r = this.bytesWritten - t,
                i = function (t, e, r, i, n) {
              var s = I.transformTo("string", n(i));
              return R.CENTRAL_DIRECTORY_END + "\0\0\0\0" + A(t, 2) + A(t, 2) + A(e, 4) + A(r, 4) + A(s.length, 2) + s;
            }(this.dirRecords.length, r, t, this.zipComment, this.encodeFileName);

            this.push({
              data: i,
              meta: {
                percent: 100
              }
            });
          }, s.prototype.prepareNextSource = function () {
            this.previous = this._sources.shift(), this.openedSource(this.previous.streamInfo), this.isPaused ? this.previous.pause() : this.previous.resume();
          }, s.prototype.registerPrevious = function (t) {
            this._sources.push(t);

            var e = this;
            return t.on("data", function (t) {
              e.processChunk(t);
            }), t.on("end", function () {
              e.closedSource(e.previous.streamInfo), e._sources.length ? e.prepareNextSource() : e.end();
            }), t.on("error", function (t) {
              e.error(t);
            }), this;
          }, s.prototype.resume = function () {
            return !!n.prototype.resume.call(this) && (!this.previous && this._sources.length ? (this.prepareNextSource(), !0) : this.previous || this._sources.length || this.generatedError ? void 0 : (this.end(), !0));
          }, s.prototype.error = function (t) {
            var e = this._sources;
            if (!n.prototype.error.call(this, t)) return !1;

            for (var r = 0; r < e.length; r++) try {
              e[r].error(t);
            } catch (t) {}

            return !0;
          }, s.prototype.lock = function () {
            n.prototype.lock.call(this);

            for (var t = this._sources, e = 0; e < t.length; e++) t[e].lock();
          }, e.exports = s;
        }, {
          "../crc32": 4,
          "../signature": 23,
          "../stream/GenericWorker": 28,
          "../utf8": 31,
          "../utils": 32
        }],
        9: [function (t, e, r) {

          var u = t("../compressions"),
              i = t("./ZipFileWorker");

          r.generateWorker = function (t, a, e) {
            var o = new i(a.streamFiles, e, a.platform, a.encodeFileName),
                h = 0;

            try {
              t.forEach(function (t, e) {
                h++;

                var r = function (t, e) {
                  var r = t || e,
                      i = u[r];
                  if (!i) throw new Error(r + " is not a valid compression method !");
                  return i;
                }(e.options.compression, a.compression),
                    i = e.options.compressionOptions || a.compressionOptions || {},
                    n = e.dir,
                    s = e.date;

                e._compressWorker(r, i).withStreamInfo("file", {
                  name: t,
                  dir: n,
                  date: s,
                  comment: e.comment || "",
                  unixPermissions: e.unixPermissions,
                  dosPermissions: e.dosPermissions
                }).pipe(o);
              }), o.entriesCount = h;
            } catch (t) {
              o.error(t);
            }

            return o;
          };
        }, {
          "../compressions": 3,
          "./ZipFileWorker": 8
        }],
        10: [function (t, e, r) {

          function i() {
            if (!(this instanceof i)) return new i();
            if (arguments.length) throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");
            this.files = Object.create(null), this.comment = null, this.root = "", this.clone = function () {
              var t = new i();

              for (var e in this) "function" != typeof this[e] && (t[e] = this[e]);

              return t;
            };
          }

          (i.prototype = t("./object")).loadAsync = t("./load"), i.support = t("./support"), i.defaults = t("./defaults"), i.version = "3.7.1", i.loadAsync = function (t, e) {
            return new i().loadAsync(t, e);
          }, i.external = t("./external"), e.exports = i;
        }, {
          "./defaults": 5,
          "./external": 6,
          "./load": 11,
          "./object": 15,
          "./support": 30
        }],
        11: [function (t, e, r) {

          var i = t("./utils"),
              n = t("./external"),
              o = t("./utf8"),
              h = t("./zipEntries"),
              s = t("./stream/Crc32Probe"),
              u = t("./nodejsUtils");

          function l(i) {
            return new n.Promise(function (t, e) {
              var r = i.decompressed.getContentWorker().pipe(new s());
              r.on("error", function (t) {
                e(t);
              }).on("end", function () {
                r.streamInfo.crc32 !== i.decompressed.crc32 ? e(new Error("Corrupted zip : CRC32 mismatch")) : t();
              }).resume();
            });
          }

          e.exports = function (t, s) {
            var a = this;
            return s = i.extend(s || {}, {
              base64: !1,
              checkCRC32: !1,
              optimizedBinaryString: !1,
              createFolders: !1,
              decodeFileName: o.utf8decode
            }), u.isNode && u.isStream(t) ? n.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")) : i.prepareContent("the loaded zip file", t, !0, s.optimizedBinaryString, s.base64).then(function (t) {
              var e = new h(s);
              return e.load(t), e;
            }).then(function (t) {
              var e = [n.Promise.resolve(t)],
                  r = t.files;
              if (s.checkCRC32) for (var i = 0; i < r.length; i++) e.push(l(r[i]));
              return n.Promise.all(e);
            }).then(function (t) {
              for (var e = t.shift(), r = e.files, i = 0; i < r.length; i++) {
                var n = r[i];
                a.file(n.fileNameStr, n.decompressed, {
                  binary: !0,
                  optimizedBinaryString: !0,
                  date: n.date,
                  dir: n.dir,
                  comment: n.fileCommentStr.length ? n.fileCommentStr : null,
                  unixPermissions: n.unixPermissions,
                  dosPermissions: n.dosPermissions,
                  createFolders: s.createFolders
                });
              }

              return e.zipComment.length && (a.comment = e.zipComment), a;
            });
          };
        }, {
          "./external": 6,
          "./nodejsUtils": 14,
          "./stream/Crc32Probe": 25,
          "./utf8": 31,
          "./utils": 32,
          "./zipEntries": 33
        }],
        12: [function (t, e, r) {

          var i = t("../utils"),
              n = t("../stream/GenericWorker");

          function s(t, e) {
            n.call(this, "Nodejs stream input adapter for " + t), this._upstreamEnded = !1, this._bindStream(e);
          }

          i.inherits(s, n), s.prototype._bindStream = function (t) {
            var e = this;
            (this._stream = t).pause(), t.on("data", function (t) {
              e.push({
                data: t,
                meta: {
                  percent: 0
                }
              });
            }).on("error", function (t) {
              e.isPaused ? this.generatedError = t : e.error(t);
            }).on("end", function () {
              e.isPaused ? e._upstreamEnded = !0 : e.end();
            });
          }, s.prototype.pause = function () {
            return !!n.prototype.pause.call(this) && (this._stream.pause(), !0);
          }, s.prototype.resume = function () {
            return !!n.prototype.resume.call(this) && (this._upstreamEnded ? this.end() : this._stream.resume(), !0);
          }, e.exports = s;
        }, {
          "../stream/GenericWorker": 28,
          "../utils": 32
        }],
        13: [function (t, e, r) {

          var n = t("readable-stream").Readable;

          function i(t, e, r) {
            n.call(this, e), this._helper = t;
            var i = this;
            t.on("data", function (t, e) {
              i.push(t) || i._helper.pause(), r && r(e);
            }).on("error", function (t) {
              i.emit("error", t);
            }).on("end", function () {
              i.push(null);
            });
          }

          t("../utils").inherits(i, n), i.prototype._read = function () {
            this._helper.resume();
          }, e.exports = i;
        }, {
          "../utils": 32,
          "readable-stream": 16
        }],
        14: [function (t, e, r) {

          e.exports = {
            isNode: "undefined" != typeof Buffer,
            newBufferFrom: function (t, e) {
              if (Buffer.from && Buffer.from !== Uint8Array.from) return Buffer.from(t, e);
              if ("number" == typeof t) throw new Error('The "data" argument must not be a number');
              return new Buffer(t, e);
            },
            allocBuffer: function (t) {
              if (Buffer.alloc) return Buffer.alloc(t);
              var e = new Buffer(t);
              return e.fill(0), e;
            },
            isBuffer: function (t) {
              return Buffer.isBuffer(t);
            },
            isStream: function (t) {
              return t && "function" == typeof t.on && "function" == typeof t.pause && "function" == typeof t.resume;
            }
          };
        }, {}],
        15: [function (t, e, r) {

          function s(t, e, r) {
            var i,
                n = u.getTypeOf(e),
                s = u.extend(r || {}, f);
            s.date = s.date || new Date(), null !== s.compression && (s.compression = s.compression.toUpperCase()), "string" == typeof s.unixPermissions && (s.unixPermissions = parseInt(s.unixPermissions, 8)), s.unixPermissions && 16384 & s.unixPermissions && (s.dir = !0), s.dosPermissions && 16 & s.dosPermissions && (s.dir = !0), s.dir && (t = g(t)), s.createFolders && (i = _(t)) && b.call(this, i, !0);
            var a = "string" === n && !1 === s.binary && !1 === s.base64;
            r && void 0 !== r.binary || (s.binary = !a), (e instanceof d && 0 === e.uncompressedSize || s.dir || !e || 0 === e.length) && (s.base64 = !1, s.binary = !0, e = "", s.compression = "STORE", n = "string");
            var o = null;
            o = e instanceof d || e instanceof l ? e : p.isNode && p.isStream(e) ? new m(t, e) : u.prepareContent(t, e, s.binary, s.optimizedBinaryString, s.base64);
            var h = new c(t, o, s);
            this.files[t] = h;
          }

          var n = t("./utf8"),
              u = t("./utils"),
              l = t("./stream/GenericWorker"),
              a = t("./stream/StreamHelper"),
              f = t("./defaults"),
              d = t("./compressedObject"),
              c = t("./zipObject"),
              o = t("./generate"),
              p = t("./nodejsUtils"),
              m = t("./nodejs/NodejsStreamInputAdapter"),
              _ = function (t) {
            "/" === t.slice(-1) && (t = t.substring(0, t.length - 1));
            var e = t.lastIndexOf("/");
            return 0 < e ? t.substring(0, e) : "";
          },
              g = function (t) {
            return "/" !== t.slice(-1) && (t += "/"), t;
          },
              b = function (t, e) {
            return e = void 0 !== e ? e : f.createFolders, t = g(t), this.files[t] || s.call(this, t, null, {
              dir: !0,
              createFolders: e
            }), this.files[t];
          };

          function h(t) {
            return "[object RegExp]" === Object.prototype.toString.call(t);
          }

          var i = {
            load: function () {
              throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
            },
            forEach: function (t) {
              var e, r, i;

              for (e in this.files) i = this.files[e], (r = e.slice(this.root.length, e.length)) && e.slice(0, this.root.length) === this.root && t(r, i);
            },
            filter: function (r) {
              var i = [];
              return this.forEach(function (t, e) {
                r(t, e) && i.push(e);
              }), i;
            },
            file: function (t, e, r) {
              if (1 !== arguments.length) return t = this.root + t, s.call(this, t, e, r), this;

              if (h(t)) {
                var i = t;
                return this.filter(function (t, e) {
                  return !e.dir && i.test(t);
                });
              }

              var n = this.files[this.root + t];
              return n && !n.dir ? n : null;
            },
            folder: function (r) {
              if (!r) return this;
              if (h(r)) return this.filter(function (t, e) {
                return e.dir && r.test(t);
              });
              var t = this.root + r,
                  e = b.call(this, t),
                  i = this.clone();
              return i.root = e.name, i;
            },
            remove: function (r) {
              r = this.root + r;
              var t = this.files[r];
              if (t || ("/" !== r.slice(-1) && (r += "/"), t = this.files[r]), t && !t.dir) delete this.files[r];else for (var e = this.filter(function (t, e) {
                return e.name.slice(0, r.length) === r;
              }), i = 0; i < e.length; i++) delete this.files[e[i].name];
              return this;
            },
            generate: function (t) {
              throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
            },
            generateInternalStream: function (t) {
              var e,
                  r = {};

              try {
                if ((r = u.extend(t || {}, {
                  streamFiles: !1,
                  compression: "STORE",
                  compressionOptions: null,
                  type: "",
                  platform: "DOS",
                  comment: null,
                  mimeType: "application/zip",
                  encodeFileName: n.utf8encode
                })).type = r.type.toLowerCase(), r.compression = r.compression.toUpperCase(), "binarystring" === r.type && (r.type = "string"), !r.type) throw new Error("No output type specified.");
                u.checkSupport(r.type), "darwin" !== r.platform && "freebsd" !== r.platform && "linux" !== r.platform && "sunos" !== r.platform || (r.platform = "UNIX"), "win32" === r.platform && (r.platform = "DOS");
                var i = r.comment || this.comment || "";
                e = o.generateWorker(this, r, i);
              } catch (t) {
                (e = new l("error")).error(t);
              }

              return new a(e, r.type || "string", r.mimeType);
            },
            generateAsync: function (t, e) {
              return this.generateInternalStream(t).accumulate(e);
            },
            generateNodeStream: function (t, e) {
              return (t = t || {}).type || (t.type = "nodebuffer"), this.generateInternalStream(t).toNodejsStream(e);
            }
          };
          e.exports = i;
        }, {
          "./compressedObject": 2,
          "./defaults": 5,
          "./generate": 9,
          "./nodejs/NodejsStreamInputAdapter": 12,
          "./nodejsUtils": 14,
          "./stream/GenericWorker": 28,
          "./stream/StreamHelper": 29,
          "./utf8": 31,
          "./utils": 32,
          "./zipObject": 35
        }],
        16: [function (t, e, r) {
          e.exports = t("stream");
        }, {
          stream: void 0
        }],
        17: [function (t, e, r) {

          var i = t("./DataReader");

          function n(t) {
            i.call(this, t);

            for (var e = 0; e < this.data.length; e++) t[e] = 255 & t[e];
          }

          t("../utils").inherits(n, i), n.prototype.byteAt = function (t) {
            return this.data[this.zero + t];
          }, n.prototype.lastIndexOfSignature = function (t) {
            for (var e = t.charCodeAt(0), r = t.charCodeAt(1), i = t.charCodeAt(2), n = t.charCodeAt(3), s = this.length - 4; 0 <= s; --s) if (this.data[s] === e && this.data[s + 1] === r && this.data[s + 2] === i && this.data[s + 3] === n) return s - this.zero;

            return -1;
          }, n.prototype.readAndCheckSignature = function (t) {
            var e = t.charCodeAt(0),
                r = t.charCodeAt(1),
                i = t.charCodeAt(2),
                n = t.charCodeAt(3),
                s = this.readData(4);
            return e === s[0] && r === s[1] && i === s[2] && n === s[3];
          }, n.prototype.readData = function (t) {
            if (this.checkOffset(t), 0 === t) return [];
            var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
            return this.index += t, e;
          }, e.exports = n;
        }, {
          "../utils": 32,
          "./DataReader": 18
        }],
        18: [function (t, e, r) {

          var i = t("../utils");

          function n(t) {
            this.data = t, this.length = t.length, this.index = 0, this.zero = 0;
          }

          n.prototype = {
            checkOffset: function (t) {
              this.checkIndex(this.index + t);
            },
            checkIndex: function (t) {
              if (this.length < this.zero + t || t < 0) throw new Error("End of data reached (data length = " + this.length + ", asked index = " + t + "). Corrupted zip ?");
            },
            setIndex: function (t) {
              this.checkIndex(t), this.index = t;
            },
            skip: function (t) {
              this.setIndex(this.index + t);
            },
            byteAt: function (t) {},
            readInt: function (t) {
              var e,
                  r = 0;

              for (this.checkOffset(t), e = this.index + t - 1; e >= this.index; e--) r = (r << 8) + this.byteAt(e);

              return this.index += t, r;
            },
            readString: function (t) {
              return i.transformTo("string", this.readData(t));
            },
            readData: function (t) {},
            lastIndexOfSignature: function (t) {},
            readAndCheckSignature: function (t) {},
            readDate: function () {
              var t = this.readInt(4);
              return new Date(Date.UTC(1980 + (t >> 25 & 127), (t >> 21 & 15) - 1, t >> 16 & 31, t >> 11 & 31, t >> 5 & 63, (31 & t) << 1));
            }
          }, e.exports = n;
        }, {
          "../utils": 32
        }],
        19: [function (t, e, r) {

          var i = t("./Uint8ArrayReader");

          function n(t) {
            i.call(this, t);
          }

          t("../utils").inherits(n, i), n.prototype.readData = function (t) {
            this.checkOffset(t);
            var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
            return this.index += t, e;
          }, e.exports = n;
        }, {
          "../utils": 32,
          "./Uint8ArrayReader": 21
        }],
        20: [function (t, e, r) {

          var i = t("./DataReader");

          function n(t) {
            i.call(this, t);
          }

          t("../utils").inherits(n, i), n.prototype.byteAt = function (t) {
            return this.data.charCodeAt(this.zero + t);
          }, n.prototype.lastIndexOfSignature = function (t) {
            return this.data.lastIndexOf(t) - this.zero;
          }, n.prototype.readAndCheckSignature = function (t) {
            return t === this.readData(4);
          }, n.prototype.readData = function (t) {
            this.checkOffset(t);
            var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
            return this.index += t, e;
          }, e.exports = n;
        }, {
          "../utils": 32,
          "./DataReader": 18
        }],
        21: [function (t, e, r) {

          var i = t("./ArrayReader");

          function n(t) {
            i.call(this, t);
          }

          t("../utils").inherits(n, i), n.prototype.readData = function (t) {
            if (this.checkOffset(t), 0 === t) return new Uint8Array(0);
            var e = this.data.subarray(this.zero + this.index, this.zero + this.index + t);
            return this.index += t, e;
          }, e.exports = n;
        }, {
          "../utils": 32,
          "./ArrayReader": 17
        }],
        22: [function (t, e, r) {

          var i = t("../utils"),
              n = t("../support"),
              s = t("./ArrayReader"),
              a = t("./StringReader"),
              o = t("./NodeBufferReader"),
              h = t("./Uint8ArrayReader");

          e.exports = function (t) {
            var e = i.getTypeOf(t);
            return i.checkSupport(e), "string" !== e || n.uint8array ? "nodebuffer" === e ? new o(t) : n.uint8array ? new h(i.transformTo("uint8array", t)) : new s(i.transformTo("array", t)) : new a(t);
          };
        }, {
          "../support": 30,
          "../utils": 32,
          "./ArrayReader": 17,
          "./NodeBufferReader": 19,
          "./StringReader": 20,
          "./Uint8ArrayReader": 21
        }],
        23: [function (t, e, r) {

          r.LOCAL_FILE_HEADER = "PK", r.CENTRAL_FILE_HEADER = "PK", r.CENTRAL_DIRECTORY_END = "PK", r.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK", r.ZIP64_CENTRAL_DIRECTORY_END = "PK", r.DATA_DESCRIPTOR = "PK\b";
        }, {}],
        24: [function (t, e, r) {

          var i = t("./GenericWorker"),
              n = t("../utils");

          function s(t) {
            i.call(this, "ConvertWorker to " + t), this.destType = t;
          }

          n.inherits(s, i), s.prototype.processChunk = function (t) {
            this.push({
              data: n.transformTo(this.destType, t.data),
              meta: t.meta
            });
          }, e.exports = s;
        }, {
          "../utils": 32,
          "./GenericWorker": 28
        }],
        25: [function (t, e, r) {

          var i = t("./GenericWorker"),
              n = t("../crc32");

          function s() {
            i.call(this, "Crc32Probe"), this.withStreamInfo("crc32", 0);
          }

          t("../utils").inherits(s, i), s.prototype.processChunk = function (t) {
            this.streamInfo.crc32 = n(t.data, this.streamInfo.crc32 || 0), this.push(t);
          }, e.exports = s;
        }, {
          "../crc32": 4,
          "../utils": 32,
          "./GenericWorker": 28
        }],
        26: [function (t, e, r) {

          var i = t("../utils"),
              n = t("./GenericWorker");

          function s(t) {
            n.call(this, "DataLengthProbe for " + t), this.propName = t, this.withStreamInfo(t, 0);
          }

          i.inherits(s, n), s.prototype.processChunk = function (t) {
            if (t) {
              var e = this.streamInfo[this.propName] || 0;
              this.streamInfo[this.propName] = e + t.data.length;
            }

            n.prototype.processChunk.call(this, t);
          }, e.exports = s;
        }, {
          "../utils": 32,
          "./GenericWorker": 28
        }],
        27: [function (t, e, r) {

          var i = t("../utils"),
              n = t("./GenericWorker");

          function s(t) {
            n.call(this, "DataWorker");
            var e = this;
            this.dataIsReady = !1, this.index = 0, this.max = 0, this.data = null, this.type = "", this._tickScheduled = !1, t.then(function (t) {
              e.dataIsReady = !0, e.data = t, e.max = t && t.length || 0, e.type = i.getTypeOf(t), e.isPaused || e._tickAndRepeat();
            }, function (t) {
              e.error(t);
            });
          }

          i.inherits(s, n), s.prototype.cleanUp = function () {
            n.prototype.cleanUp.call(this), this.data = null;
          }, s.prototype.resume = function () {
            return !!n.prototype.resume.call(this) && (!this._tickScheduled && this.dataIsReady && (this._tickScheduled = !0, i.delay(this._tickAndRepeat, [], this)), !0);
          }, s.prototype._tickAndRepeat = function () {
            this._tickScheduled = !1, this.isPaused || this.isFinished || (this._tick(), this.isFinished || (i.delay(this._tickAndRepeat, [], this), this._tickScheduled = !0));
          }, s.prototype._tick = function () {
            if (this.isPaused || this.isFinished) return !1;
            var t = null,
                e = Math.min(this.max, this.index + 16384);
            if (this.index >= this.max) return this.end();

            switch (this.type) {
              case "string":
                t = this.data.substring(this.index, e);
                break;

              case "uint8array":
                t = this.data.subarray(this.index, e);
                break;

              case "array":
              case "nodebuffer":
                t = this.data.slice(this.index, e);
            }

            return this.index = e, this.push({
              data: t,
              meta: {
                percent: this.max ? this.index / this.max * 100 : 0
              }
            });
          }, e.exports = s;
        }, {
          "../utils": 32,
          "./GenericWorker": 28
        }],
        28: [function (t, e, r) {

          function i(t) {
            this.name = t || "default", this.streamInfo = {}, this.generatedError = null, this.extraStreamInfo = {}, this.isPaused = !0, this.isFinished = !1, this.isLocked = !1, this._listeners = {
              data: [],
              end: [],
              error: []
            }, this.previous = null;
          }

          i.prototype = {
            push: function (t) {
              this.emit("data", t);
            },
            end: function () {
              if (this.isFinished) return !1;
              this.flush();

              try {
                this.emit("end"), this.cleanUp(), this.isFinished = !0;
              } catch (t) {
                this.emit("error", t);
              }

              return !0;
            },
            error: function (t) {
              return !this.isFinished && (this.isPaused ? this.generatedError = t : (this.isFinished = !0, this.emit("error", t), this.previous && this.previous.error(t), this.cleanUp()), !0);
            },
            on: function (t, e) {
              return this._listeners[t].push(e), this;
            },
            cleanUp: function () {
              this.streamInfo = this.generatedError = this.extraStreamInfo = null, this._listeners = [];
            },
            emit: function (t, e) {
              if (this._listeners[t]) for (var r = 0; r < this._listeners[t].length; r++) this._listeners[t][r].call(this, e);
            },
            pipe: function (t) {
              return t.registerPrevious(this);
            },
            registerPrevious: function (t) {
              if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
              this.streamInfo = t.streamInfo, this.mergeStreamInfo(), this.previous = t;
              var e = this;
              return t.on("data", function (t) {
                e.processChunk(t);
              }), t.on("end", function () {
                e.end();
              }), t.on("error", function (t) {
                e.error(t);
              }), this;
            },
            pause: function () {
              return !this.isPaused && !this.isFinished && (this.isPaused = !0, this.previous && this.previous.pause(), !0);
            },
            resume: function () {
              if (!this.isPaused || this.isFinished) return !1;
              var t = this.isPaused = !1;
              return this.generatedError && (this.error(this.generatedError), t = !0), this.previous && this.previous.resume(), !t;
            },
            flush: function () {},
            processChunk: function (t) {
              this.push(t);
            },
            withStreamInfo: function (t, e) {
              return this.extraStreamInfo[t] = e, this.mergeStreamInfo(), this;
            },
            mergeStreamInfo: function () {
              for (var t in this.extraStreamInfo) this.extraStreamInfo.hasOwnProperty(t) && (this.streamInfo[t] = this.extraStreamInfo[t]);
            },
            lock: function () {
              if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
              this.isLocked = !0, this.previous && this.previous.lock();
            },
            toString: function () {
              var t = "Worker " + this.name;
              return this.previous ? this.previous + " -> " + t : t;
            }
          }, e.exports = i;
        }, {}],
        29: [function (t, e, r) {

          var h = t("../utils"),
              n = t("./ConvertWorker"),
              s = t("./GenericWorker"),
              u = t("../base64"),
              i = t("../support"),
              a = t("../external"),
              o = null;
          if (i.nodestream) try {
            o = t("../nodejs/NodejsStreamOutputAdapter");
          } catch (t) {}

          function l(t, o) {
            return new a.Promise(function (e, r) {
              var i = [],
                  n = t._internalType,
                  s = t._outputType,
                  a = t._mimeType;
              t.on("data", function (t, e) {
                i.push(t), o && o(e);
              }).on("error", function (t) {
                i = [], r(t);
              }).on("end", function () {
                try {
                  var t = function (t, e, r) {
                    switch (t) {
                      case "blob":
                        return h.newBlob(h.transformTo("arraybuffer", e), r);

                      case "base64":
                        return u.encode(e);

                      default:
                        return h.transformTo(t, e);
                    }
                  }(s, function (t, e) {
                    var r,
                        i = 0,
                        n = null,
                        s = 0;

                    for (r = 0; r < e.length; r++) s += e[r].length;

                    switch (t) {
                      case "string":
                        return e.join("");

                      case "array":
                        return Array.prototype.concat.apply([], e);

                      case "uint8array":
                        for (n = new Uint8Array(s), r = 0; r < e.length; r++) n.set(e[r], i), i += e[r].length;

                        return n;

                      case "nodebuffer":
                        return Buffer.concat(e);

                      default:
                        throw new Error("concat : unsupported type '" + t + "'");
                    }
                  }(n, i), a);

                  e(t);
                } catch (t) {
                  r(t);
                }

                i = [];
              }).resume();
            });
          }

          function f(t, e, r) {
            var i = e;

            switch (e) {
              case "blob":
              case "arraybuffer":
                i = "uint8array";
                break;

              case "base64":
                i = "string";
            }

            try {
              this._internalType = i, this._outputType = e, this._mimeType = r, h.checkSupport(i), this._worker = t.pipe(new n(i)), t.lock();
            } catch (t) {
              this._worker = new s("error"), this._worker.error(t);
            }
          }

          f.prototype = {
            accumulate: function (t) {
              return l(this, t);
            },
            on: function (t, e) {
              var r = this;
              return "data" === t ? this._worker.on(t, function (t) {
                e.call(r, t.data, t.meta);
              }) : this._worker.on(t, function () {
                h.delay(e, arguments, r);
              }), this;
            },
            resume: function () {
              return h.delay(this._worker.resume, [], this._worker), this;
            },
            pause: function () {
              return this._worker.pause(), this;
            },
            toNodejsStream: function (t) {
              if (h.checkSupport("nodestream"), "nodebuffer" !== this._outputType) throw new Error(this._outputType + " is not supported by this method");
              return new o(this, {
                objectMode: "nodebuffer" !== this._outputType
              }, t);
            }
          }, e.exports = f;
        }, {
          "../base64": 1,
          "../external": 6,
          "../nodejs/NodejsStreamOutputAdapter": 13,
          "../support": 30,
          "../utils": 32,
          "./ConvertWorker": 24,
          "./GenericWorker": 28
        }],
        30: [function (t, e, r) {

          if (r.base64 = !0, r.array = !0, r.string = !0, r.arraybuffer = "undefined" != typeof ArrayBuffer && "undefined" != typeof Uint8Array, r.nodebuffer = "undefined" != typeof Buffer, r.uint8array = "undefined" != typeof Uint8Array, "undefined" == typeof ArrayBuffer) r.blob = !1;else {
            var i = new ArrayBuffer(0);

            try {
              r.blob = 0 === new Blob([i], {
                type: "application/zip"
              }).size;
            } catch (t) {
              try {
                var n = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                n.append(i), r.blob = 0 === n.getBlob("application/zip").size;
              } catch (t) {
                r.blob = !1;
              }
            }
          }

          try {
            r.nodestream = !!t("readable-stream").Readable;
          } catch (t) {
            r.nodestream = !1;
          }
        }, {
          "readable-stream": 16
        }],
        31: [function (t, e, s) {

          for (var o = t("./utils"), h = t("./support"), r = t("./nodejsUtils"), i = t("./stream/GenericWorker"), u = new Array(256), n = 0; n < 256; n++) u[n] = 252 <= n ? 6 : 248 <= n ? 5 : 240 <= n ? 4 : 224 <= n ? 3 : 192 <= n ? 2 : 1;

          u[254] = u[254] = 1;

          function a() {
            i.call(this, "utf-8 decode"), this.leftOver = null;
          }

          function l() {
            i.call(this, "utf-8 encode");
          }

          s.utf8encode = function (t) {
            return h.nodebuffer ? r.newBufferFrom(t, "utf-8") : function (t) {
              var e,
                  r,
                  i,
                  n,
                  s,
                  a = t.length,
                  o = 0;

              for (n = 0; n < a; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

              for (e = h.uint8array ? new Uint8Array(o) : new Array(o), n = s = 0; s < o; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), r < 128 ? e[s++] = r : (r < 2048 ? e[s++] = 192 | r >>> 6 : (r < 65536 ? e[s++] = 224 | r >>> 12 : (e[s++] = 240 | r >>> 18, e[s++] = 128 | r >>> 12 & 63), e[s++] = 128 | r >>> 6 & 63), e[s++] = 128 | 63 & r);

              return e;
            }(t);
          }, s.utf8decode = function (t) {
            return h.nodebuffer ? o.transformTo("nodebuffer", t).toString("utf-8") : function (t) {
              var e,
                  r,
                  i,
                  n,
                  s = t.length,
                  a = new Array(2 * s);

              for (e = r = 0; e < s;) if ((i = t[e++]) < 128) a[r++] = i;else if (4 < (n = u[i])) a[r++] = 65533, e += n - 1;else {
                for (i &= 2 === n ? 31 : 3 === n ? 15 : 7; 1 < n && e < s;) i = i << 6 | 63 & t[e++], n--;

                1 < n ? a[r++] = 65533 : i < 65536 ? a[r++] = i : (i -= 65536, a[r++] = 55296 | i >> 10 & 1023, a[r++] = 56320 | 1023 & i);
              }

              return a.length !== r && (a.subarray ? a = a.subarray(0, r) : a.length = r), o.applyFromCharCode(a);
            }(t = o.transformTo(h.uint8array ? "uint8array" : "array", t));
          }, o.inherits(a, i), a.prototype.processChunk = function (t) {
            var e = o.transformTo(h.uint8array ? "uint8array" : "array", t.data);

            if (this.leftOver && this.leftOver.length) {
              if (h.uint8array) {
                var r = e;
                (e = new Uint8Array(r.length + this.leftOver.length)).set(this.leftOver, 0), e.set(r, this.leftOver.length);
              } else e = this.leftOver.concat(e);

              this.leftOver = null;
            }

            var i = function (t, e) {
              var r;

              for ((e = e || t.length) > t.length && (e = t.length), r = e - 1; 0 <= r && 128 == (192 & t[r]);) r--;

              return r < 0 ? e : 0 === r ? e : r + u[t[r]] > e ? r : e;
            }(e),
                n = e;

            i !== e.length && (h.uint8array ? (n = e.subarray(0, i), this.leftOver = e.subarray(i, e.length)) : (n = e.slice(0, i), this.leftOver = e.slice(i, e.length))), this.push({
              data: s.utf8decode(n),
              meta: t.meta
            });
          }, a.prototype.flush = function () {
            this.leftOver && this.leftOver.length && (this.push({
              data: s.utf8decode(this.leftOver),
              meta: {}
            }), this.leftOver = null);
          }, s.Utf8DecodeWorker = a, o.inherits(l, i), l.prototype.processChunk = function (t) {
            this.push({
              data: s.utf8encode(t.data),
              meta: t.meta
            });
          }, s.Utf8EncodeWorker = l;
        }, {
          "./nodejsUtils": 14,
          "./stream/GenericWorker": 28,
          "./support": 30,
          "./utils": 32
        }],
        32: [function (t, e, a) {

          var o = t("./support"),
              h = t("./base64"),
              r = t("./nodejsUtils"),
              i = t("set-immediate-shim"),
              u = t("./external");

          function n(t) {
            return t;
          }

          function l(t, e) {
            for (var r = 0; r < t.length; ++r) e[r] = 255 & t.charCodeAt(r);

            return e;
          }

          a.newBlob = function (e, r) {
            a.checkSupport("blob");

            try {
              return new Blob([e], {
                type: r
              });
            } catch (t) {
              try {
                var i = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                return i.append(e), i.getBlob(r);
              } catch (t) {
                throw new Error("Bug : can't construct the Blob.");
              }
            }
          };

          var s = {
            stringifyByChunk: function (t, e, r) {
              var i = [],
                  n = 0,
                  s = t.length;
              if (s <= r) return String.fromCharCode.apply(null, t);

              for (; n < s;) "array" === e || "nodebuffer" === e ? i.push(String.fromCharCode.apply(null, t.slice(n, Math.min(n + r, s)))) : i.push(String.fromCharCode.apply(null, t.subarray(n, Math.min(n + r, s)))), n += r;

              return i.join("");
            },
            stringifyByChar: function (t) {
              for (var e = "", r = 0; r < t.length; r++) e += String.fromCharCode(t[r]);

              return e;
            },
            applyCanBeUsed: {
              uint8array: function () {
                try {
                  return o.uint8array && 1 === String.fromCharCode.apply(null, new Uint8Array(1)).length;
                } catch (t) {
                  return !1;
                }
              }(),
              nodebuffer: function () {
                try {
                  return o.nodebuffer && 1 === String.fromCharCode.apply(null, r.allocBuffer(1)).length;
                } catch (t) {
                  return !1;
                }
              }()
            }
          };

          function f(t) {
            var e = 65536,
                r = a.getTypeOf(t),
                i = !0;
            if ("uint8array" === r ? i = s.applyCanBeUsed.uint8array : "nodebuffer" === r && (i = s.applyCanBeUsed.nodebuffer), i) for (; 1 < e;) try {
              return s.stringifyByChunk(t, r, e);
            } catch (t) {
              e = Math.floor(e / 2);
            }
            return s.stringifyByChar(t);
          }

          function d(t, e) {
            for (var r = 0; r < t.length; r++) e[r] = t[r];

            return e;
          }

          a.applyFromCharCode = f;
          var c = {};
          c.string = {
            string: n,
            array: function (t) {
              return l(t, new Array(t.length));
            },
            arraybuffer: function (t) {
              return c.string.uint8array(t).buffer;
            },
            uint8array: function (t) {
              return l(t, new Uint8Array(t.length));
            },
            nodebuffer: function (t) {
              return l(t, r.allocBuffer(t.length));
            }
          }, c.array = {
            string: f,
            array: n,
            arraybuffer: function (t) {
              return new Uint8Array(t).buffer;
            },
            uint8array: function (t) {
              return new Uint8Array(t);
            },
            nodebuffer: function (t) {
              return r.newBufferFrom(t);
            }
          }, c.arraybuffer = {
            string: function (t) {
              return f(new Uint8Array(t));
            },
            array: function (t) {
              return d(new Uint8Array(t), new Array(t.byteLength));
            },
            arraybuffer: n,
            uint8array: function (t) {
              return new Uint8Array(t);
            },
            nodebuffer: function (t) {
              return r.newBufferFrom(new Uint8Array(t));
            }
          }, c.uint8array = {
            string: f,
            array: function (t) {
              return d(t, new Array(t.length));
            },
            arraybuffer: function (t) {
              return t.buffer;
            },
            uint8array: n,
            nodebuffer: function (t) {
              return r.newBufferFrom(t);
            }
          }, c.nodebuffer = {
            string: f,
            array: function (t) {
              return d(t, new Array(t.length));
            },
            arraybuffer: function (t) {
              return c.nodebuffer.uint8array(t).buffer;
            },
            uint8array: function (t) {
              return d(t, new Uint8Array(t.length));
            },
            nodebuffer: n
          }, a.transformTo = function (t, e) {
            if (e = e || "", !t) return e;
            a.checkSupport(t);
            var r = a.getTypeOf(e);
            return c[r][t](e);
          }, a.getTypeOf = function (t) {
            return "string" == typeof t ? "string" : "[object Array]" === Object.prototype.toString.call(t) ? "array" : o.nodebuffer && r.isBuffer(t) ? "nodebuffer" : o.uint8array && t instanceof Uint8Array ? "uint8array" : o.arraybuffer && t instanceof ArrayBuffer ? "arraybuffer" : void 0;
          }, a.checkSupport = function (t) {
            if (!o[t.toLowerCase()]) throw new Error(t + " is not supported by this platform");
          }, a.MAX_VALUE_16BITS = 65535, a.MAX_VALUE_32BITS = -1, a.pretty = function (t) {
            var e,
                r,
                i = "";

            for (r = 0; r < (t || "").length; r++) i += "\\x" + ((e = t.charCodeAt(r)) < 16 ? "0" : "") + e.toString(16).toUpperCase();

            return i;
          }, a.delay = function (t, e, r) {
            i(function () {
              t.apply(r || null, e || []);
            });
          }, a.inherits = function (t, e) {
            function r() {}

            r.prototype = e.prototype, t.prototype = new r();
          }, a.extend = function () {
            var t,
                e,
                r = {};

            for (t = 0; t < arguments.length; t++) for (e in arguments[t]) arguments[t].hasOwnProperty(e) && void 0 === r[e] && (r[e] = arguments[t][e]);

            return r;
          }, a.prepareContent = function (r, t, i, n, s) {
            return u.Promise.resolve(t).then(function (i) {
              return o.blob && (i instanceof Blob || -1 !== ["[object File]", "[object Blob]"].indexOf(Object.prototype.toString.call(i))) && "undefined" != typeof FileReader ? new u.Promise(function (e, r) {
                var t = new FileReader();
                t.onload = function (t) {
                  e(t.target.result);
                }, t.onerror = function (t) {
                  r(t.target.error);
                }, t.readAsArrayBuffer(i);
              }) : i;
            }).then(function (t) {
              var e = a.getTypeOf(t);
              return e ? ("arraybuffer" === e ? t = a.transformTo("uint8array", t) : "string" === e && (s ? t = h.decode(t) : i && !0 !== n && (t = function (t) {
                return l(t, o.uint8array ? new Uint8Array(t.length) : new Array(t.length));
              }(t))), t) : u.Promise.reject(new Error("Can't read the data of '" + r + "'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"));
            });
          };
        }, {
          "./base64": 1,
          "./external": 6,
          "./nodejsUtils": 14,
          "./support": 30,
          "set-immediate-shim": 54
        }],
        33: [function (t, e, r) {

          var i = t("./reader/readerFor"),
              n = t("./utils"),
              s = t("./signature"),
              a = t("./zipEntry"),
              o = (t("./utf8"), t("./support"));

          function h(t) {
            this.files = [], this.loadOptions = t;
          }

          h.prototype = {
            checkSignature: function (t) {
              if (!this.reader.readAndCheckSignature(t)) {
                this.reader.index -= 4;
                var e = this.reader.readString(4);
                throw new Error("Corrupted zip or bug: unexpected signature (" + n.pretty(e) + ", expected " + n.pretty(t) + ")");
              }
            },
            isSignature: function (t, e) {
              var r = this.reader.index;
              this.reader.setIndex(t);
              var i = this.reader.readString(4) === e;
              return this.reader.setIndex(r), i;
            },
            readBlockEndOfCentral: function () {
              this.diskNumber = this.reader.readInt(2), this.diskWithCentralDirStart = this.reader.readInt(2), this.centralDirRecordsOnThisDisk = this.reader.readInt(2), this.centralDirRecords = this.reader.readInt(2), this.centralDirSize = this.reader.readInt(4), this.centralDirOffset = this.reader.readInt(4), this.zipCommentLength = this.reader.readInt(2);
              var t = this.reader.readData(this.zipCommentLength),
                  e = o.uint8array ? "uint8array" : "array",
                  r = n.transformTo(e, t);
              this.zipComment = this.loadOptions.decodeFileName(r);
            },
            readBlockZip64EndOfCentral: function () {
              this.zip64EndOfCentralSize = this.reader.readInt(8), this.reader.skip(4), this.diskNumber = this.reader.readInt(4), this.diskWithCentralDirStart = this.reader.readInt(4), this.centralDirRecordsOnThisDisk = this.reader.readInt(8), this.centralDirRecords = this.reader.readInt(8), this.centralDirSize = this.reader.readInt(8), this.centralDirOffset = this.reader.readInt(8), this.zip64ExtensibleData = {};

              for (var t, e, r, i = this.zip64EndOfCentralSize - 44; 0 < i;) t = this.reader.readInt(2), e = this.reader.readInt(4), r = this.reader.readData(e), this.zip64ExtensibleData[t] = {
                id: t,
                length: e,
                value: r
              };
            },
            readBlockZip64EndOfCentralLocator: function () {
              if (this.diskWithZip64CentralDirStart = this.reader.readInt(4), this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8), this.disksCount = this.reader.readInt(4), 1 < this.disksCount) throw new Error("Multi-volumes zip are not supported");
            },
            readLocalFiles: function () {
              var t, e;

              for (t = 0; t < this.files.length; t++) e = this.files[t], this.reader.setIndex(e.localHeaderOffset), this.checkSignature(s.LOCAL_FILE_HEADER), e.readLocalPart(this.reader), e.handleUTF8(), e.processAttributes();
            },
            readCentralDir: function () {
              var t;

              for (this.reader.setIndex(this.centralDirOffset); this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);) (t = new a({
                zip64: this.zip64
              }, this.loadOptions)).readCentralPart(this.reader), this.files.push(t);

              if (this.centralDirRecords !== this.files.length && 0 !== this.centralDirRecords && 0 === this.files.length) throw new Error("Corrupted zip or bug: expected " + this.centralDirRecords + " records in central dir, got " + this.files.length);
            },
            readEndOfCentral: function () {
              var t = this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);
              if (t < 0) throw !this.isSignature(0, s.LOCAL_FILE_HEADER) ? new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html") : new Error("Corrupted zip: can't find end of central directory");
              this.reader.setIndex(t);
              var e = t;

              if (this.checkSignature(s.CENTRAL_DIRECTORY_END), this.readBlockEndOfCentral(), this.diskNumber === n.MAX_VALUE_16BITS || this.diskWithCentralDirStart === n.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === n.MAX_VALUE_16BITS || this.centralDirRecords === n.MAX_VALUE_16BITS || this.centralDirSize === n.MAX_VALUE_32BITS || this.centralDirOffset === n.MAX_VALUE_32BITS) {
                if (this.zip64 = !0, (t = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR)) < 0) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");
                if (this.reader.setIndex(t), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR), this.readBlockZip64EndOfCentralLocator(), !this.isSignature(this.relativeOffsetEndOfZip64CentralDir, s.ZIP64_CENTRAL_DIRECTORY_END) && (this.relativeOffsetEndOfZip64CentralDir = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.relativeOffsetEndOfZip64CentralDir < 0)) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");
                this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.readBlockZip64EndOfCentral();
              }

              var r = this.centralDirOffset + this.centralDirSize;
              this.zip64 && (r += 20, r += 12 + this.zip64EndOfCentralSize);
              var i = e - r;
              if (0 < i) this.isSignature(e, s.CENTRAL_FILE_HEADER) || (this.reader.zero = i);else if (i < 0) throw new Error("Corrupted zip: missing " + Math.abs(i) + " bytes.");
            },
            prepareReader: function (t) {
              this.reader = i(t);
            },
            load: function (t) {
              this.prepareReader(t), this.readEndOfCentral(), this.readCentralDir(), this.readLocalFiles();
            }
          }, e.exports = h;
        }, {
          "./reader/readerFor": 22,
          "./signature": 23,
          "./support": 30,
          "./utf8": 31,
          "./utils": 32,
          "./zipEntry": 34
        }],
        34: [function (t, e, r) {

          var i = t("./reader/readerFor"),
              s = t("./utils"),
              n = t("./compressedObject"),
              a = t("./crc32"),
              o = t("./utf8"),
              h = t("./compressions"),
              u = t("./support");

          function l(t, e) {
            this.options = t, this.loadOptions = e;
          }

          l.prototype = {
            isEncrypted: function () {
              return 1 == (1 & this.bitFlag);
            },
            useUTF8: function () {
              return 2048 == (2048 & this.bitFlag);
            },
            readLocalPart: function (t) {
              var e, r;
              if (t.skip(22), this.fileNameLength = t.readInt(2), r = t.readInt(2), this.fileName = t.readData(this.fileNameLength), t.skip(r), -1 === this.compressedSize || -1 === this.uncompressedSize) throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");
              if (null === (e = function (t) {
                for (var e in h) if (h.hasOwnProperty(e) && h[e].magic === t) return h[e];

                return null;
              }(this.compressionMethod))) throw new Error("Corrupted zip : compression " + s.pretty(this.compressionMethod) + " unknown (inner file : " + s.transformTo("string", this.fileName) + ")");
              this.decompressed = new n(this.compressedSize, this.uncompressedSize, this.crc32, e, t.readData(this.compressedSize));
            },
            readCentralPart: function (t) {
              this.versionMadeBy = t.readInt(2), t.skip(2), this.bitFlag = t.readInt(2), this.compressionMethod = t.readString(2), this.date = t.readDate(), this.crc32 = t.readInt(4), this.compressedSize = t.readInt(4), this.uncompressedSize = t.readInt(4);
              var e = t.readInt(2);
              if (this.extraFieldsLength = t.readInt(2), this.fileCommentLength = t.readInt(2), this.diskNumberStart = t.readInt(2), this.internalFileAttributes = t.readInt(2), this.externalFileAttributes = t.readInt(4), this.localHeaderOffset = t.readInt(4), this.isEncrypted()) throw new Error("Encrypted zip are not supported");
              t.skip(e), this.readExtraFields(t), this.parseZIP64ExtraField(t), this.fileComment = t.readData(this.fileCommentLength);
            },
            processAttributes: function () {
              this.unixPermissions = null, this.dosPermissions = null;
              var t = this.versionMadeBy >> 8;
              this.dir = !!(16 & this.externalFileAttributes), 0 == t && (this.dosPermissions = 63 & this.externalFileAttributes), 3 == t && (this.unixPermissions = this.externalFileAttributes >> 16 & 65535), this.dir || "/" !== this.fileNameStr.slice(-1) || (this.dir = !0);
            },
            parseZIP64ExtraField: function (t) {
              if (this.extraFields[1]) {
                var e = i(this.extraFields[1].value);
                this.uncompressedSize === s.MAX_VALUE_32BITS && (this.uncompressedSize = e.readInt(8)), this.compressedSize === s.MAX_VALUE_32BITS && (this.compressedSize = e.readInt(8)), this.localHeaderOffset === s.MAX_VALUE_32BITS && (this.localHeaderOffset = e.readInt(8)), this.diskNumberStart === s.MAX_VALUE_32BITS && (this.diskNumberStart = e.readInt(4));
              }
            },
            readExtraFields: function (t) {
              var e,
                  r,
                  i,
                  n = t.index + this.extraFieldsLength;

              for (this.extraFields || (this.extraFields = {}); t.index + 4 < n;) e = t.readInt(2), r = t.readInt(2), i = t.readData(r), this.extraFields[e] = {
                id: e,
                length: r,
                value: i
              };

              t.setIndex(n);
            },
            handleUTF8: function () {
              var t = u.uint8array ? "uint8array" : "array";
              if (this.useUTF8()) this.fileNameStr = o.utf8decode(this.fileName), this.fileCommentStr = o.utf8decode(this.fileComment);else {
                var e = this.findExtraFieldUnicodePath();
                if (null !== e) this.fileNameStr = e;else {
                  var r = s.transformTo(t, this.fileName);
                  this.fileNameStr = this.loadOptions.decodeFileName(r);
                }
                var i = this.findExtraFieldUnicodeComment();
                if (null !== i) this.fileCommentStr = i;else {
                  var n = s.transformTo(t, this.fileComment);
                  this.fileCommentStr = this.loadOptions.decodeFileName(n);
                }
              }
            },
            findExtraFieldUnicodePath: function () {
              var t = this.extraFields[28789];

              if (t) {
                var e = i(t.value);
                return 1 !== e.readInt(1) ? null : a(this.fileName) !== e.readInt(4) ? null : o.utf8decode(e.readData(t.length - 5));
              }

              return null;
            },
            findExtraFieldUnicodeComment: function () {
              var t = this.extraFields[25461];

              if (t) {
                var e = i(t.value);
                return 1 !== e.readInt(1) ? null : a(this.fileComment) !== e.readInt(4) ? null : o.utf8decode(e.readData(t.length - 5));
              }

              return null;
            }
          }, e.exports = l;
        }, {
          "./compressedObject": 2,
          "./compressions": 3,
          "./crc32": 4,
          "./reader/readerFor": 22,
          "./support": 30,
          "./utf8": 31,
          "./utils": 32
        }],
        35: [function (t, e, r) {

          function i(t, e, r) {
            this.name = t, this.dir = r.dir, this.date = r.date, this.comment = r.comment, this.unixPermissions = r.unixPermissions, this.dosPermissions = r.dosPermissions, this._data = e, this._dataBinary = r.binary, this.options = {
              compression: r.compression,
              compressionOptions: r.compressionOptions
            };
          }

          var s = t("./stream/StreamHelper"),
              n = t("./stream/DataWorker"),
              a = t("./utf8"),
              o = t("./compressedObject"),
              h = t("./stream/GenericWorker");
          i.prototype = {
            internalStream: function (t) {
              var e = null,
                  r = "string";

              try {
                if (!t) throw new Error("No output type specified.");
                var i = "string" === (r = t.toLowerCase()) || "text" === r;
                "binarystring" !== r && "text" !== r || (r = "string"), e = this._decompressWorker();
                var n = !this._dataBinary;
                n && !i && (e = e.pipe(new a.Utf8EncodeWorker())), !n && i && (e = e.pipe(new a.Utf8DecodeWorker()));
              } catch (t) {
                (e = new h("error")).error(t);
              }

              return new s(e, r, "");
            },
            async: function (t, e) {
              return this.internalStream(t).accumulate(e);
            },
            nodeStream: function (t, e) {
              return this.internalStream(t || "nodebuffer").toNodejsStream(e);
            },
            _compressWorker: function (t, e) {
              if (this._data instanceof o && this._data.compression.magic === t.magic) return this._data.getCompressedWorker();

              var r = this._decompressWorker();

              return this._dataBinary || (r = r.pipe(new a.Utf8EncodeWorker())), o.createWorkerFrom(r, t, e);
            },
            _decompressWorker: function () {
              return this._data instanceof o ? this._data.getContentWorker() : this._data instanceof h ? this._data : new n(this._data);
            }
          };

          for (var u = ["asText", "asBinary", "asNodeBuffer", "asUint8Array", "asArrayBuffer"], l = function () {
            throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
          }, f = 0; f < u.length; f++) i.prototype[u[f]] = l;

          e.exports = i;
        }, {
          "./compressedObject": 2,
          "./stream/DataWorker": 27,
          "./stream/GenericWorker": 28,
          "./stream/StreamHelper": 29,
          "./utf8": 31
        }],
        36: [function (t, l, e) {
          (function (e) {

            var r,
                i,
                t = e.MutationObserver || e.WebKitMutationObserver;

            if (t) {
              var n = 0,
                  s = new t(u),
                  a = e.document.createTextNode("");
              s.observe(a, {
                characterData: !0
              }), r = function () {
                a.data = n = ++n % 2;
              };
            } else if (e.setImmediate || void 0 === e.MessageChannel) r = "document" in e && "onreadystatechange" in e.document.createElement("script") ? function () {
              var t = e.document.createElement("script");
              t.onreadystatechange = function () {
                u(), t.onreadystatechange = null, t.parentNode.removeChild(t), t = null;
              }, e.document.documentElement.appendChild(t);
            } : function () {
              setTimeout(u, 0);
            };else {
              var o = new e.MessageChannel();
              o.port1.onmessage = u, r = function () {
                o.port2.postMessage(0);
              };
            }

            var h = [];

            function u() {
              var t, e;
              i = !0;

              for (var r = h.length; r;) {
                for (e = h, h = [], t = -1; ++t < r;) e[t]();

                r = h.length;
              }

              i = !1;
            }

            l.exports = function (t) {
              1 !== h.push(t) || i || r();
            };
          }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
        }, {}],
        37: [function (t, e, r) {

          var n = t("immediate");

          function u() {}

          var l = {},
              s = ["REJECTED"],
              a = ["FULFILLED"],
              i = ["PENDING"];

          function o(t) {
            if ("function" != typeof t) throw new TypeError("resolver must be a function");
            this.state = i, this.queue = [], this.outcome = void 0, t !== u && c(this, t);
          }

          function h(t, e, r) {
            this.promise = t, "function" == typeof e && (this.onFulfilled = e, this.callFulfilled = this.otherCallFulfilled), "function" == typeof r && (this.onRejected = r, this.callRejected = this.otherCallRejected);
          }

          function f(e, r, i) {
            n(function () {
              var t;

              try {
                t = r(i);
              } catch (t) {
                return l.reject(e, t);
              }

              t === e ? l.reject(e, new TypeError("Cannot resolve promise with itself")) : l.resolve(e, t);
            });
          }

          function d(t) {
            var e = t && t.then;
            if (t && ("object" == typeof t || "function" == typeof t) && "function" == typeof e) return function () {
              e.apply(t, arguments);
            };
          }

          function c(e, t) {
            var r = !1;

            function i(t) {
              r || (r = !0, l.reject(e, t));
            }

            function n(t) {
              r || (r = !0, l.resolve(e, t));
            }

            var s = p(function () {
              t(n, i);
            });
            "error" === s.status && i(s.value);
          }

          function p(t, e) {
            var r = {};

            try {
              r.value = t(e), r.status = "success";
            } catch (t) {
              r.status = "error", r.value = t;
            }

            return r;
          }

          (e.exports = o).prototype.finally = function (e) {
            if ("function" != typeof e) return this;
            var r = this.constructor;
            return this.then(function (t) {
              return r.resolve(e()).then(function () {
                return t;
              });
            }, function (t) {
              return r.resolve(e()).then(function () {
                throw t;
              });
            });
          }, o.prototype.catch = function (t) {
            return this.then(null, t);
          }, o.prototype.then = function (t, e) {
            if ("function" != typeof t && this.state === a || "function" != typeof e && this.state === s) return this;
            var r = new this.constructor(u);
            this.state !== i ? f(r, this.state === a ? t : e, this.outcome) : this.queue.push(new h(r, t, e));
            return r;
          }, h.prototype.callFulfilled = function (t) {
            l.resolve(this.promise, t);
          }, h.prototype.otherCallFulfilled = function (t) {
            f(this.promise, this.onFulfilled, t);
          }, h.prototype.callRejected = function (t) {
            l.reject(this.promise, t);
          }, h.prototype.otherCallRejected = function (t) {
            f(this.promise, this.onRejected, t);
          }, l.resolve = function (t, e) {
            var r = p(d, e);
            if ("error" === r.status) return l.reject(t, r.value);
            var i = r.value;
            if (i) c(t, i);else {
              t.state = a, t.outcome = e;

              for (var n = -1, s = t.queue.length; ++n < s;) t.queue[n].callFulfilled(e);
            }
            return t;
          }, l.reject = function (t, e) {
            t.state = s, t.outcome = e;

            for (var r = -1, i = t.queue.length; ++r < i;) t.queue[r].callRejected(e);

            return t;
          }, o.resolve = function (t) {
            if (t instanceof this) return t;
            return l.resolve(new this(u), t);
          }, o.reject = function (t) {
            var e = new this(u);
            return l.reject(e, t);
          }, o.all = function (t) {
            var r = this;
            if ("[object Array]" !== Object.prototype.toString.call(t)) return this.reject(new TypeError("must be an array"));
            var i = t.length,
                n = !1;
            if (!i) return this.resolve([]);
            var s = new Array(i),
                a = 0,
                e = -1,
                o = new this(u);

            for (; ++e < i;) h(t[e], e);

            return o;

            function h(t, e) {
              r.resolve(t).then(function (t) {
                s[e] = t, ++a !== i || n || (n = !0, l.resolve(o, s));
              }, function (t) {
                n || (n = !0, l.reject(o, t));
              });
            }
          }, o.race = function (t) {
            var e = this;
            if ("[object Array]" !== Object.prototype.toString.call(t)) return this.reject(new TypeError("must be an array"));
            var r = t.length,
                i = !1;
            if (!r) return this.resolve([]);
            var n = -1,
                s = new this(u);

            for (; ++n < r;) a = t[n], e.resolve(a).then(function (t) {
              i || (i = !0, l.resolve(s, t));
            }, function (t) {
              i || (i = !0, l.reject(s, t));
            });

            var a;
            return s;
          };
        }, {
          immediate: 36
        }],
        38: [function (t, e, r) {

          var i = {};
          (0, t("./lib/utils/common").assign)(i, t("./lib/deflate"), t("./lib/inflate"), t("./lib/zlib/constants")), e.exports = i;
        }, {
          "./lib/deflate": 39,
          "./lib/inflate": 40,
          "./lib/utils/common": 41,
          "./lib/zlib/constants": 44
        }],
        39: [function (t, e, r) {

          var a = t("./zlib/deflate"),
              o = t("./utils/common"),
              h = t("./utils/strings"),
              n = t("./zlib/messages"),
              s = t("./zlib/zstream"),
              u = Object.prototype.toString,
              l = 0,
              f = -1,
              d = 0,
              c = 8;

          function p(t) {
            if (!(this instanceof p)) return new p(t);
            this.options = o.assign({
              level: f,
              method: c,
              chunkSize: 16384,
              windowBits: 15,
              memLevel: 8,
              strategy: d,
              to: ""
            }, t || {});
            var e = this.options;
            e.raw && 0 < e.windowBits ? e.windowBits = -e.windowBits : e.gzip && 0 < e.windowBits && e.windowBits < 16 && (e.windowBits += 16), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new s(), this.strm.avail_out = 0;
            var r = a.deflateInit2(this.strm, e.level, e.method, e.windowBits, e.memLevel, e.strategy);
            if (r !== l) throw new Error(n[r]);

            if (e.header && a.deflateSetHeader(this.strm, e.header), e.dictionary) {
              var i;
              if (i = "string" == typeof e.dictionary ? h.string2buf(e.dictionary) : "[object ArrayBuffer]" === u.call(e.dictionary) ? new Uint8Array(e.dictionary) : e.dictionary, (r = a.deflateSetDictionary(this.strm, i)) !== l) throw new Error(n[r]);
              this._dict_set = !0;
            }
          }

          function i(t, e) {
            var r = new p(e);
            if (r.push(t, !0), r.err) throw r.msg || n[r.err];
            return r.result;
          }

          p.prototype.push = function (t, e) {
            var r,
                i,
                n = this.strm,
                s = this.options.chunkSize;
            if (this.ended) return !1;
            i = e === ~~e ? e : !0 === e ? 4 : 0, "string" == typeof t ? n.input = h.string2buf(t) : "[object ArrayBuffer]" === u.call(t) ? n.input = new Uint8Array(t) : n.input = t, n.next_in = 0, n.avail_in = n.input.length;

            do {
              if (0 === n.avail_out && (n.output = new o.Buf8(s), n.next_out = 0, n.avail_out = s), 1 !== (r = a.deflate(n, i)) && r !== l) return this.onEnd(r), !(this.ended = !0);
              0 !== n.avail_out && (0 !== n.avail_in || 4 !== i && 2 !== i) || ("string" === this.options.to ? this.onData(h.buf2binstring(o.shrinkBuf(n.output, n.next_out))) : this.onData(o.shrinkBuf(n.output, n.next_out)));
            } while ((0 < n.avail_in || 0 === n.avail_out) && 1 !== r);

            return 4 === i ? (r = a.deflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === l) : 2 !== i || (this.onEnd(l), !(n.avail_out = 0));
          }, p.prototype.onData = function (t) {
            this.chunks.push(t);
          }, p.prototype.onEnd = function (t) {
            t === l && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = o.flattenChunks(this.chunks)), this.chunks = [], this.err = t, this.msg = this.strm.msg;
          }, r.Deflate = p, r.deflate = i, r.deflateRaw = function (t, e) {
            return (e = e || {}).raw = !0, i(t, e);
          }, r.gzip = function (t, e) {
            return (e = e || {}).gzip = !0, i(t, e);
          };
        }, {
          "./utils/common": 41,
          "./utils/strings": 42,
          "./zlib/deflate": 46,
          "./zlib/messages": 51,
          "./zlib/zstream": 53
        }],
        40: [function (t, e, r) {

          var d = t("./zlib/inflate"),
              c = t("./utils/common"),
              p = t("./utils/strings"),
              m = t("./zlib/constants"),
              i = t("./zlib/messages"),
              n = t("./zlib/zstream"),
              s = t("./zlib/gzheader"),
              _ = Object.prototype.toString;

          function a(t) {
            if (!(this instanceof a)) return new a(t);
            this.options = c.assign({
              chunkSize: 16384,
              windowBits: 0,
              to: ""
            }, t || {});
            var e = this.options;
            e.raw && 0 <= e.windowBits && e.windowBits < 16 && (e.windowBits = -e.windowBits, 0 === e.windowBits && (e.windowBits = -15)), !(0 <= e.windowBits && e.windowBits < 16) || t && t.windowBits || (e.windowBits += 32), 15 < e.windowBits && e.windowBits < 48 && 0 == (15 & e.windowBits) && (e.windowBits |= 15), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new n(), this.strm.avail_out = 0;
            var r = d.inflateInit2(this.strm, e.windowBits);
            if (r !== m.Z_OK) throw new Error(i[r]);
            this.header = new s(), d.inflateGetHeader(this.strm, this.header);
          }

          function o(t, e) {
            var r = new a(e);
            if (r.push(t, !0), r.err) throw r.msg || i[r.err];
            return r.result;
          }

          a.prototype.push = function (t, e) {
            var r,
                i,
                n,
                s,
                a,
                o,
                h = this.strm,
                u = this.options.chunkSize,
                l = this.options.dictionary,
                f = !1;
            if (this.ended) return !1;
            i = e === ~~e ? e : !0 === e ? m.Z_FINISH : m.Z_NO_FLUSH, "string" == typeof t ? h.input = p.binstring2buf(t) : "[object ArrayBuffer]" === _.call(t) ? h.input = new Uint8Array(t) : h.input = t, h.next_in = 0, h.avail_in = h.input.length;

            do {
              if (0 === h.avail_out && (h.output = new c.Buf8(u), h.next_out = 0, h.avail_out = u), (r = d.inflate(h, m.Z_NO_FLUSH)) === m.Z_NEED_DICT && l && (o = "string" == typeof l ? p.string2buf(l) : "[object ArrayBuffer]" === _.call(l) ? new Uint8Array(l) : l, r = d.inflateSetDictionary(this.strm, o)), r === m.Z_BUF_ERROR && !0 === f && (r = m.Z_OK, f = !1), r !== m.Z_STREAM_END && r !== m.Z_OK) return this.onEnd(r), !(this.ended = !0);
              h.next_out && (0 !== h.avail_out && r !== m.Z_STREAM_END && (0 !== h.avail_in || i !== m.Z_FINISH && i !== m.Z_SYNC_FLUSH) || ("string" === this.options.to ? (n = p.utf8border(h.output, h.next_out), s = h.next_out - n, a = p.buf2string(h.output, n), h.next_out = s, h.avail_out = u - s, s && c.arraySet(h.output, h.output, n, s, 0), this.onData(a)) : this.onData(c.shrinkBuf(h.output, h.next_out)))), 0 === h.avail_in && 0 === h.avail_out && (f = !0);
            } while ((0 < h.avail_in || 0 === h.avail_out) && r !== m.Z_STREAM_END);

            return r === m.Z_STREAM_END && (i = m.Z_FINISH), i === m.Z_FINISH ? (r = d.inflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === m.Z_OK) : i !== m.Z_SYNC_FLUSH || (this.onEnd(m.Z_OK), !(h.avail_out = 0));
          }, a.prototype.onData = function (t) {
            this.chunks.push(t);
          }, a.prototype.onEnd = function (t) {
            t === m.Z_OK && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = c.flattenChunks(this.chunks)), this.chunks = [], this.err = t, this.msg = this.strm.msg;
          }, r.Inflate = a, r.inflate = o, r.inflateRaw = function (t, e) {
            return (e = e || {}).raw = !0, o(t, e);
          }, r.ungzip = o;
        }, {
          "./utils/common": 41,
          "./utils/strings": 42,
          "./zlib/constants": 44,
          "./zlib/gzheader": 47,
          "./zlib/inflate": 49,
          "./zlib/messages": 51,
          "./zlib/zstream": 53
        }],
        41: [function (t, e, r) {

          var i = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Int32Array;
          r.assign = function (t) {
            for (var e = Array.prototype.slice.call(arguments, 1); e.length;) {
              var r = e.shift();

              if (r) {
                if ("object" != typeof r) throw new TypeError(r + "must be non-object");

                for (var i in r) r.hasOwnProperty(i) && (t[i] = r[i]);
              }
            }

            return t;
          }, r.shrinkBuf = function (t, e) {
            return t.length === e ? t : t.subarray ? t.subarray(0, e) : (t.length = e, t);
          };
          var n = {
            arraySet: function (t, e, r, i, n) {
              if (e.subarray && t.subarray) t.set(e.subarray(r, r + i), n);else for (var s = 0; s < i; s++) t[n + s] = e[r + s];
            },
            flattenChunks: function (t) {
              var e, r, i, n, s, a;

              for (e = i = 0, r = t.length; e < r; e++) i += t[e].length;

              for (a = new Uint8Array(i), e = n = 0, r = t.length; e < r; e++) s = t[e], a.set(s, n), n += s.length;

              return a;
            }
          },
              s = {
            arraySet: function (t, e, r, i, n) {
              for (var s = 0; s < i; s++) t[n + s] = e[r + s];
            },
            flattenChunks: function (t) {
              return [].concat.apply([], t);
            }
          };
          r.setTyped = function (t) {
            t ? (r.Buf8 = Uint8Array, r.Buf16 = Uint16Array, r.Buf32 = Int32Array, r.assign(r, n)) : (r.Buf8 = Array, r.Buf16 = Array, r.Buf32 = Array, r.assign(r, s));
          }, r.setTyped(i);
        }, {}],
        42: [function (t, e, r) {

          var h = t("./common"),
              n = !0,
              s = !0;

          try {
            String.fromCharCode.apply(null, [0]);
          } catch (t) {
            n = !1;
          }

          try {
            String.fromCharCode.apply(null, new Uint8Array(1));
          } catch (t) {
            s = !1;
          }

          for (var u = new h.Buf8(256), i = 0; i < 256; i++) u[i] = 252 <= i ? 6 : 248 <= i ? 5 : 240 <= i ? 4 : 224 <= i ? 3 : 192 <= i ? 2 : 1;

          function l(t, e) {
            if (e < 65537 && (t.subarray && s || !t.subarray && n)) return String.fromCharCode.apply(null, h.shrinkBuf(t, e));

            for (var r = "", i = 0; i < e; i++) r += String.fromCharCode(t[i]);

            return r;
          }

          u[254] = u[254] = 1, r.string2buf = function (t) {
            var e,
                r,
                i,
                n,
                s,
                a = t.length,
                o = 0;

            for (n = 0; n < a; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

            for (e = new h.Buf8(o), n = s = 0; s < o; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), r < 128 ? e[s++] = r : (r < 2048 ? e[s++] = 192 | r >>> 6 : (r < 65536 ? e[s++] = 224 | r >>> 12 : (e[s++] = 240 | r >>> 18, e[s++] = 128 | r >>> 12 & 63), e[s++] = 128 | r >>> 6 & 63), e[s++] = 128 | 63 & r);

            return e;
          }, r.buf2binstring = function (t) {
            return l(t, t.length);
          }, r.binstring2buf = function (t) {
            for (var e = new h.Buf8(t.length), r = 0, i = e.length; r < i; r++) e[r] = t.charCodeAt(r);

            return e;
          }, r.buf2string = function (t, e) {
            var r,
                i,
                n,
                s,
                a = e || t.length,
                o = new Array(2 * a);

            for (r = i = 0; r < a;) if ((n = t[r++]) < 128) o[i++] = n;else if (4 < (s = u[n])) o[i++] = 65533, r += s - 1;else {
              for (n &= 2 === s ? 31 : 3 === s ? 15 : 7; 1 < s && r < a;) n = n << 6 | 63 & t[r++], s--;

              1 < s ? o[i++] = 65533 : n < 65536 ? o[i++] = n : (n -= 65536, o[i++] = 55296 | n >> 10 & 1023, o[i++] = 56320 | 1023 & n);
            }

            return l(o, i);
          }, r.utf8border = function (t, e) {
            var r;

            for ((e = e || t.length) > t.length && (e = t.length), r = e - 1; 0 <= r && 128 == (192 & t[r]);) r--;

            return r < 0 ? e : 0 === r ? e : r + u[t[r]] > e ? r : e;
          };
        }, {
          "./common": 41
        }],
        43: [function (t, e, r) {

          e.exports = function (t, e, r, i) {
            for (var n = 65535 & t | 0, s = t >>> 16 & 65535 | 0, a = 0; 0 !== r;) {
              for (r -= a = 2e3 < r ? 2e3 : r; s = s + (n = n + e[i++] | 0) | 0, --a;);

              n %= 65521, s %= 65521;
            }

            return n | s << 16 | 0;
          };
        }, {}],
        44: [function (t, e, r) {

          e.exports = {
            Z_NO_FLUSH: 0,
            Z_PARTIAL_FLUSH: 1,
            Z_SYNC_FLUSH: 2,
            Z_FULL_FLUSH: 3,
            Z_FINISH: 4,
            Z_BLOCK: 5,
            Z_TREES: 6,
            Z_OK: 0,
            Z_STREAM_END: 1,
            Z_NEED_DICT: 2,
            Z_ERRNO: -1,
            Z_STREAM_ERROR: -2,
            Z_DATA_ERROR: -3,
            Z_BUF_ERROR: -5,
            Z_NO_COMPRESSION: 0,
            Z_BEST_SPEED: 1,
            Z_BEST_COMPRESSION: 9,
            Z_DEFAULT_COMPRESSION: -1,
            Z_FILTERED: 1,
            Z_HUFFMAN_ONLY: 2,
            Z_RLE: 3,
            Z_FIXED: 4,
            Z_DEFAULT_STRATEGY: 0,
            Z_BINARY: 0,
            Z_TEXT: 1,
            Z_UNKNOWN: 2,
            Z_DEFLATED: 8
          };
        }, {}],
        45: [function (t, e, r) {

          var o = function () {
            for (var t, e = [], r = 0; r < 256; r++) {
              t = r;

              for (var i = 0; i < 8; i++) t = 1 & t ? 3988292384 ^ t >>> 1 : t >>> 1;

              e[r] = t;
            }

            return e;
          }();

          e.exports = function (t, e, r, i) {
            var n = o,
                s = i + r;
            t ^= -1;

            for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e[a])];

            return -1 ^ t;
          };
        }, {}],
        46: [function (t, e, r) {

          var h,
              d = t("../utils/common"),
              u = t("./trees"),
              c = t("./adler32"),
              p = t("./crc32"),
              i = t("./messages"),
              l = 0,
              f = 4,
              m = 0,
              _ = -2,
              g = -1,
              b = 4,
              n = 2,
              v = 8,
              y = 9,
              s = 286,
              a = 30,
              o = 19,
              w = 2 * s + 1,
              k = 15,
              x = 3,
              S = 258,
              z = S + x + 1,
              C = 42,
              E = 113,
              A = 1,
              I = 2,
              O = 3,
              B = 4;

          function R(t, e) {
            return t.msg = i[e], e;
          }

          function T(t) {
            return (t << 1) - (4 < t ? 9 : 0);
          }

          function D(t) {
            for (var e = t.length; 0 <= --e;) t[e] = 0;
          }

          function F(t) {
            var e = t.state,
                r = e.pending;
            r > t.avail_out && (r = t.avail_out), 0 !== r && (d.arraySet(t.output, e.pending_buf, e.pending_out, r, t.next_out), t.next_out += r, e.pending_out += r, t.total_out += r, t.avail_out -= r, e.pending -= r, 0 === e.pending && (e.pending_out = 0));
          }

          function N(t, e) {
            u._tr_flush_block(t, 0 <= t.block_start ? t.block_start : -1, t.strstart - t.block_start, e), t.block_start = t.strstart, F(t.strm);
          }

          function U(t, e) {
            t.pending_buf[t.pending++] = e;
          }

          function P(t, e) {
            t.pending_buf[t.pending++] = e >>> 8 & 255, t.pending_buf[t.pending++] = 255 & e;
          }

          function L(t, e) {
            var r,
                i,
                n = t.max_chain_length,
                s = t.strstart,
                a = t.prev_length,
                o = t.nice_match,
                h = t.strstart > t.w_size - z ? t.strstart - (t.w_size - z) : 0,
                u = t.window,
                l = t.w_mask,
                f = t.prev,
                d = t.strstart + S,
                c = u[s + a - 1],
                p = u[s + a];
            t.prev_length >= t.good_match && (n >>= 2), o > t.lookahead && (o = t.lookahead);

            do {
              if (u[(r = e) + a] === p && u[r + a - 1] === c && u[r] === u[s] && u[++r] === u[s + 1]) {
                s += 2, r++;

                do {} while (u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && s < d);

                if (i = S - (d - s), s = d - S, a < i) {
                  if (t.match_start = e, o <= (a = i)) break;
                  c = u[s + a - 1], p = u[s + a];
                }
              }
            } while ((e = f[e & l]) > h && 0 != --n);

            return a <= t.lookahead ? a : t.lookahead;
          }

          function j(t) {
            var e,
                r,
                i,
                n,
                s,
                a,
                o,
                h,
                u,
                l,
                f = t.w_size;

            do {
              if (n = t.window_size - t.lookahead - t.strstart, t.strstart >= f + (f - z)) {
                for (d.arraySet(t.window, t.window, f, f, 0), t.match_start -= f, t.strstart -= f, t.block_start -= f, e = r = t.hash_size; i = t.head[--e], t.head[e] = f <= i ? i - f : 0, --r;);

                for (e = r = f; i = t.prev[--e], t.prev[e] = f <= i ? i - f : 0, --r;);

                n += f;
              }

              if (0 === t.strm.avail_in) break;
              if (a = t.strm, o = t.window, h = t.strstart + t.lookahead, u = n, l = void 0, l = a.avail_in, u < l && (l = u), r = 0 === l ? 0 : (a.avail_in -= l, d.arraySet(o, a.input, a.next_in, l, h), 1 === a.state.wrap ? a.adler = c(a.adler, o, l, h) : 2 === a.state.wrap && (a.adler = p(a.adler, o, l, h)), a.next_in += l, a.total_in += l, l), t.lookahead += r, t.lookahead + t.insert >= x) for (s = t.strstart - t.insert, t.ins_h = t.window[s], t.ins_h = (t.ins_h << t.hash_shift ^ t.window[s + 1]) & t.hash_mask; t.insert && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[s + x - 1]) & t.hash_mask, t.prev[s & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = s, s++, t.insert--, !(t.lookahead + t.insert < x)););
            } while (t.lookahead < z && 0 !== t.strm.avail_in);
          }

          function Z(t, e) {
            for (var r, i;;) {
              if (t.lookahead < z) {
                if (j(t), t.lookahead < z && e === l) return A;
                if (0 === t.lookahead) break;
              }

              if (r = 0, t.lookahead >= x && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), 0 !== r && t.strstart - r <= t.w_size - z && (t.match_length = L(t, r)), t.match_length >= x) {
                if (i = u._tr_tally(t, t.strstart - t.match_start, t.match_length - x), t.lookahead -= t.match_length, t.match_length <= t.max_lazy_match && t.lookahead >= x) {
                  for (t.match_length--; t.strstart++, t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart, 0 != --t.match_length;);

                  t.strstart++;
                } else t.strstart += t.match_length, t.match_length = 0, t.ins_h = t.window[t.strstart], t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + 1]) & t.hash_mask;
              } else i = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++;
              if (i && (N(t, !1), 0 === t.strm.avail_out)) return A;
            }

            return t.insert = t.strstart < x - 1 ? t.strstart : x - 1, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
          }

          function W(t, e) {
            for (var r, i, n;;) {
              if (t.lookahead < z) {
                if (j(t), t.lookahead < z && e === l) return A;
                if (0 === t.lookahead) break;
              }

              if (r = 0, t.lookahead >= x && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), t.prev_length = t.match_length, t.prev_match = t.match_start, t.match_length = x - 1, 0 !== r && t.prev_length < t.max_lazy_match && t.strstart - r <= t.w_size - z && (t.match_length = L(t, r), t.match_length <= 5 && (1 === t.strategy || t.match_length === x && 4096 < t.strstart - t.match_start) && (t.match_length = x - 1)), t.prev_length >= x && t.match_length <= t.prev_length) {
                for (n = t.strstart + t.lookahead - x, i = u._tr_tally(t, t.strstart - 1 - t.prev_match, t.prev_length - x), t.lookahead -= t.prev_length - 1, t.prev_length -= 2; ++t.strstart <= n && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), 0 != --t.prev_length;);

                if (t.match_available = 0, t.match_length = x - 1, t.strstart++, i && (N(t, !1), 0 === t.strm.avail_out)) return A;
              } else if (t.match_available) {
                if ((i = u._tr_tally(t, 0, t.window[t.strstart - 1])) && N(t, !1), t.strstart++, t.lookahead--, 0 === t.strm.avail_out) return A;
              } else t.match_available = 1, t.strstart++, t.lookahead--;
            }

            return t.match_available && (i = u._tr_tally(t, 0, t.window[t.strstart - 1]), t.match_available = 0), t.insert = t.strstart < x - 1 ? t.strstart : x - 1, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
          }

          function M(t, e, r, i, n) {
            this.good_length = t, this.max_lazy = e, this.nice_length = r, this.max_chain = i, this.func = n;
          }

          function H() {
            this.strm = null, this.status = 0, this.pending_buf = null, this.pending_buf_size = 0, this.pending_out = 0, this.pending = 0, this.wrap = 0, this.gzhead = null, this.gzindex = 0, this.method = v, this.last_flush = -1, this.w_size = 0, this.w_bits = 0, this.w_mask = 0, this.window = null, this.window_size = 0, this.prev = null, this.head = null, this.ins_h = 0, this.hash_size = 0, this.hash_bits = 0, this.hash_mask = 0, this.hash_shift = 0, this.block_start = 0, this.match_length = 0, this.prev_match = 0, this.match_available = 0, this.strstart = 0, this.match_start = 0, this.lookahead = 0, this.prev_length = 0, this.max_chain_length = 0, this.max_lazy_match = 0, this.level = 0, this.strategy = 0, this.good_match = 0, this.nice_match = 0, this.dyn_ltree = new d.Buf16(2 * w), this.dyn_dtree = new d.Buf16(2 * (2 * a + 1)), this.bl_tree = new d.Buf16(2 * (2 * o + 1)), D(this.dyn_ltree), D(this.dyn_dtree), D(this.bl_tree), this.l_desc = null, this.d_desc = null, this.bl_desc = null, this.bl_count = new d.Buf16(k + 1), this.heap = new d.Buf16(2 * s + 1), D(this.heap), this.heap_len = 0, this.heap_max = 0, this.depth = new d.Buf16(2 * s + 1), D(this.depth), this.l_buf = 0, this.lit_bufsize = 0, this.last_lit = 0, this.d_buf = 0, this.opt_len = 0, this.static_len = 0, this.matches = 0, this.insert = 0, this.bi_buf = 0, this.bi_valid = 0;
          }

          function G(t) {
            var e;
            return t && t.state ? (t.total_in = t.total_out = 0, t.data_type = n, (e = t.state).pending = 0, e.pending_out = 0, e.wrap < 0 && (e.wrap = -e.wrap), e.status = e.wrap ? C : E, t.adler = 2 === e.wrap ? 0 : 1, e.last_flush = l, u._tr_init(e), m) : R(t, _);
          }

          function K(t) {
            var e = G(t);
            return e === m && function (t) {
              t.window_size = 2 * t.w_size, D(t.head), t.max_lazy_match = h[t.level].max_lazy, t.good_match = h[t.level].good_length, t.nice_match = h[t.level].nice_length, t.max_chain_length = h[t.level].max_chain, t.strstart = 0, t.block_start = 0, t.lookahead = 0, t.insert = 0, t.match_length = t.prev_length = x - 1, t.match_available = 0, t.ins_h = 0;
            }(t.state), e;
          }

          function Y(t, e, r, i, n, s) {
            if (!t) return _;
            var a = 1;
            if (e === g && (e = 6), i < 0 ? (a = 0, i = -i) : 15 < i && (a = 2, i -= 16), n < 1 || y < n || r !== v || i < 8 || 15 < i || e < 0 || 9 < e || s < 0 || b < s) return R(t, _);
            8 === i && (i = 9);
            var o = new H();
            return (t.state = o).strm = t, o.wrap = a, o.gzhead = null, o.w_bits = i, o.w_size = 1 << o.w_bits, o.w_mask = o.w_size - 1, o.hash_bits = n + 7, o.hash_size = 1 << o.hash_bits, o.hash_mask = o.hash_size - 1, o.hash_shift = ~~((o.hash_bits + x - 1) / x), o.window = new d.Buf8(2 * o.w_size), o.head = new d.Buf16(o.hash_size), o.prev = new d.Buf16(o.w_size), o.lit_bufsize = 1 << n + 6, o.pending_buf_size = 4 * o.lit_bufsize, o.pending_buf = new d.Buf8(o.pending_buf_size), o.d_buf = 1 * o.lit_bufsize, o.l_buf = 3 * o.lit_bufsize, o.level = e, o.strategy = s, o.method = r, K(t);
          }

          h = [new M(0, 0, 0, 0, function (t, e) {
            var r = 65535;

            for (r > t.pending_buf_size - 5 && (r = t.pending_buf_size - 5);;) {
              if (t.lookahead <= 1) {
                if (j(t), 0 === t.lookahead && e === l) return A;
                if (0 === t.lookahead) break;
              }

              t.strstart += t.lookahead, t.lookahead = 0;
              var i = t.block_start + r;
              if ((0 === t.strstart || t.strstart >= i) && (t.lookahead = t.strstart - i, t.strstart = i, N(t, !1), 0 === t.strm.avail_out)) return A;
              if (t.strstart - t.block_start >= t.w_size - z && (N(t, !1), 0 === t.strm.avail_out)) return A;
            }

            return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : (t.strstart > t.block_start && (N(t, !1), t.strm.avail_out), A);
          }), new M(4, 4, 8, 4, Z), new M(4, 5, 16, 8, Z), new M(4, 6, 32, 32, Z), new M(4, 4, 16, 16, W), new M(8, 16, 32, 32, W), new M(8, 16, 128, 128, W), new M(8, 32, 128, 256, W), new M(32, 128, 258, 1024, W), new M(32, 258, 258, 4096, W)], r.deflateInit = function (t, e) {
            return Y(t, e, v, 15, 8, 0);
          }, r.deflateInit2 = Y, r.deflateReset = K, r.deflateResetKeep = G, r.deflateSetHeader = function (t, e) {
            return t && t.state ? 2 !== t.state.wrap ? _ : (t.state.gzhead = e, m) : _;
          }, r.deflate = function (t, e) {
            var r, i, n, s;
            if (!t || !t.state || 5 < e || e < 0) return t ? R(t, _) : _;
            if (i = t.state, !t.output || !t.input && 0 !== t.avail_in || 666 === i.status && e !== f) return R(t, 0 === t.avail_out ? -5 : _);
            if (i.strm = t, r = i.last_flush, i.last_flush = e, i.status === C) if (2 === i.wrap) t.adler = 0, U(i, 31), U(i, 139), U(i, 8), i.gzhead ? (U(i, (i.gzhead.text ? 1 : 0) + (i.gzhead.hcrc ? 2 : 0) + (i.gzhead.extra ? 4 : 0) + (i.gzhead.name ? 8 : 0) + (i.gzhead.comment ? 16 : 0)), U(i, 255 & i.gzhead.time), U(i, i.gzhead.time >> 8 & 255), U(i, i.gzhead.time >> 16 & 255), U(i, i.gzhead.time >> 24 & 255), U(i, 9 === i.level ? 2 : 2 <= i.strategy || i.level < 2 ? 4 : 0), U(i, 255 & i.gzhead.os), i.gzhead.extra && i.gzhead.extra.length && (U(i, 255 & i.gzhead.extra.length), U(i, i.gzhead.extra.length >> 8 & 255)), i.gzhead.hcrc && (t.adler = p(t.adler, i.pending_buf, i.pending, 0)), i.gzindex = 0, i.status = 69) : (U(i, 0), U(i, 0), U(i, 0), U(i, 0), U(i, 0), U(i, 9 === i.level ? 2 : 2 <= i.strategy || i.level < 2 ? 4 : 0), U(i, 3), i.status = E);else {
              var a = v + (i.w_bits - 8 << 4) << 8;
              a |= (2 <= i.strategy || i.level < 2 ? 0 : i.level < 6 ? 1 : 6 === i.level ? 2 : 3) << 6, 0 !== i.strstart && (a |= 32), a += 31 - a % 31, i.status = E, P(i, a), 0 !== i.strstart && (P(i, t.adler >>> 16), P(i, 65535 & t.adler)), t.adler = 1;
            }
            if (69 === i.status) if (i.gzhead.extra) {
              for (n = i.pending; i.gzindex < (65535 & i.gzhead.extra.length) && (i.pending !== i.pending_buf_size || (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending !== i.pending_buf_size));) U(i, 255 & i.gzhead.extra[i.gzindex]), i.gzindex++;

              i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), i.gzindex === i.gzhead.extra.length && (i.gzindex = 0, i.status = 73);
            } else i.status = 73;
            if (73 === i.status) if (i.gzhead.name) {
              n = i.pending;

              do {
                if (i.pending === i.pending_buf_size && (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending === i.pending_buf_size)) {
                  s = 1;
                  break;
                }

                s = i.gzindex < i.gzhead.name.length ? 255 & i.gzhead.name.charCodeAt(i.gzindex++) : 0, U(i, s);
              } while (0 !== s);

              i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), 0 === s && (i.gzindex = 0, i.status = 91);
            } else i.status = 91;
            if (91 === i.status) if (i.gzhead.comment) {
              n = i.pending;

              do {
                if (i.pending === i.pending_buf_size && (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending === i.pending_buf_size)) {
                  s = 1;
                  break;
                }

                s = i.gzindex < i.gzhead.comment.length ? 255 & i.gzhead.comment.charCodeAt(i.gzindex++) : 0, U(i, s);
              } while (0 !== s);

              i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), 0 === s && (i.status = 103);
            } else i.status = 103;

            if (103 === i.status && (i.gzhead.hcrc ? (i.pending + 2 > i.pending_buf_size && F(t), i.pending + 2 <= i.pending_buf_size && (U(i, 255 & t.adler), U(i, t.adler >> 8 & 255), t.adler = 0, i.status = E)) : i.status = E), 0 !== i.pending) {
              if (F(t), 0 === t.avail_out) return i.last_flush = -1, m;
            } else if (0 === t.avail_in && T(e) <= T(r) && e !== f) return R(t, -5);

            if (666 === i.status && 0 !== t.avail_in) return R(t, -5);

            if (0 !== t.avail_in || 0 !== i.lookahead || e !== l && 666 !== i.status) {
              var o = 2 === i.strategy ? function (t, e) {
                for (var r;;) {
                  if (0 === t.lookahead && (j(t), 0 === t.lookahead)) {
                    if (e === l) return A;
                    break;
                  }

                  if (t.match_length = 0, r = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++, r && (N(t, !1), 0 === t.strm.avail_out)) return A;
                }

                return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
              }(i, e) : 3 === i.strategy ? function (t, e) {
                for (var r, i, n, s, a = t.window;;) {
                  if (t.lookahead <= S) {
                    if (j(t), t.lookahead <= S && e === l) return A;
                    if (0 === t.lookahead) break;
                  }

                  if (t.match_length = 0, t.lookahead >= x && 0 < t.strstart && (i = a[n = t.strstart - 1]) === a[++n] && i === a[++n] && i === a[++n]) {
                    s = t.strstart + S;

                    do {} while (i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && n < s);

                    t.match_length = S - (s - n), t.match_length > t.lookahead && (t.match_length = t.lookahead);
                  }

                  if (t.match_length >= x ? (r = u._tr_tally(t, 1, t.match_length - x), t.lookahead -= t.match_length, t.strstart += t.match_length, t.match_length = 0) : (r = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++), r && (N(t, !1), 0 === t.strm.avail_out)) return A;
                }

                return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
              }(i, e) : h[i.level].func(i, e);
              if (o !== O && o !== B || (i.status = 666), o === A || o === O) return 0 === t.avail_out && (i.last_flush = -1), m;
              if (o === I && (1 === e ? u._tr_align(i) : 5 !== e && (u._tr_stored_block(i, 0, 0, !1), 3 === e && (D(i.head), 0 === i.lookahead && (i.strstart = 0, i.block_start = 0, i.insert = 0))), F(t), 0 === t.avail_out)) return i.last_flush = -1, m;
            }

            return e !== f ? m : i.wrap <= 0 ? 1 : (2 === i.wrap ? (U(i, 255 & t.adler), U(i, t.adler >> 8 & 255), U(i, t.adler >> 16 & 255), U(i, t.adler >> 24 & 255), U(i, 255 & t.total_in), U(i, t.total_in >> 8 & 255), U(i, t.total_in >> 16 & 255), U(i, t.total_in >> 24 & 255)) : (P(i, t.adler >>> 16), P(i, 65535 & t.adler)), F(t), 0 < i.wrap && (i.wrap = -i.wrap), 0 !== i.pending ? m : 1);
          }, r.deflateEnd = function (t) {
            var e;
            return t && t.state ? (e = t.state.status) !== C && 69 !== e && 73 !== e && 91 !== e && 103 !== e && e !== E && 666 !== e ? R(t, _) : (t.state = null, e === E ? R(t, -3) : m) : _;
          }, r.deflateSetDictionary = function (t, e) {
            var r,
                i,
                n,
                s,
                a,
                o,
                h,
                u,
                l = e.length;
            if (!t || !t.state) return _;
            if (2 === (s = (r = t.state).wrap) || 1 === s && r.status !== C || r.lookahead) return _;

            for (1 === s && (t.adler = c(t.adler, e, l, 0)), r.wrap = 0, l >= r.w_size && (0 === s && (D(r.head), r.strstart = 0, r.block_start = 0, r.insert = 0), u = new d.Buf8(r.w_size), d.arraySet(u, e, l - r.w_size, r.w_size, 0), e = u, l = r.w_size), a = t.avail_in, o = t.next_in, h = t.input, t.avail_in = l, t.next_in = 0, t.input = e, j(r); r.lookahead >= x;) {
              for (i = r.strstart, n = r.lookahead - (x - 1); r.ins_h = (r.ins_h << r.hash_shift ^ r.window[i + x - 1]) & r.hash_mask, r.prev[i & r.w_mask] = r.head[r.ins_h], r.head[r.ins_h] = i, i++, --n;);

              r.strstart = i, r.lookahead = x - 1, j(r);
            }

            return r.strstart += r.lookahead, r.block_start = r.strstart, r.insert = r.lookahead, r.lookahead = 0, r.match_length = r.prev_length = x - 1, r.match_available = 0, t.next_in = o, t.input = h, t.avail_in = a, r.wrap = s, m;
          }, r.deflateInfo = "pako deflate (from Nodeca project)";
        }, {
          "../utils/common": 41,
          "./adler32": 43,
          "./crc32": 45,
          "./messages": 51,
          "./trees": 52
        }],
        47: [function (t, e, r) {

          e.exports = function () {
            this.text = 0, this.time = 0, this.xflags = 0, this.os = 0, this.extra = null, this.extra_len = 0, this.name = "", this.comment = "", this.hcrc = 0, this.done = !1;
          };
        }, {}],
        48: [function (t, e, r) {

          e.exports = function (t, e) {
            var r, i, n, s, a, o, h, u, l, f, d, c, p, m, _, g, b, v, y, w, k, x, S, z, C;

            r = t.state, i = t.next_in, z = t.input, n = i + (t.avail_in - 5), s = t.next_out, C = t.output, a = s - (e - t.avail_out), o = s + (t.avail_out - 257), h = r.dmax, u = r.wsize, l = r.whave, f = r.wnext, d = r.window, c = r.hold, p = r.bits, m = r.lencode, _ = r.distcode, g = (1 << r.lenbits) - 1, b = (1 << r.distbits) - 1;

            t: do {
              p < 15 && (c += z[i++] << p, p += 8, c += z[i++] << p, p += 8), v = m[c & g];

              e: for (;;) {
                if (c >>>= y = v >>> 24, p -= y, 0 === (y = v >>> 16 & 255)) C[s++] = 65535 & v;else {
                  if (!(16 & y)) {
                    if (0 == (64 & y)) {
                      v = m[(65535 & v) + (c & (1 << y) - 1)];
                      continue e;
                    }

                    if (32 & y) {
                      r.mode = 12;
                      break t;
                    }

                    t.msg = "invalid literal/length code", r.mode = 30;
                    break t;
                  }

                  w = 65535 & v, (y &= 15) && (p < y && (c += z[i++] << p, p += 8), w += c & (1 << y) - 1, c >>>= y, p -= y), p < 15 && (c += z[i++] << p, p += 8, c += z[i++] << p, p += 8), v = _[c & b];

                  r: for (;;) {
                    if (c >>>= y = v >>> 24, p -= y, !(16 & (y = v >>> 16 & 255))) {
                      if (0 == (64 & y)) {
                        v = _[(65535 & v) + (c & (1 << y) - 1)];
                        continue r;
                      }

                      t.msg = "invalid distance code", r.mode = 30;
                      break t;
                    }

                    if (k = 65535 & v, p < (y &= 15) && (c += z[i++] << p, (p += 8) < y && (c += z[i++] << p, p += 8)), h < (k += c & (1 << y) - 1)) {
                      t.msg = "invalid distance too far back", r.mode = 30;
                      break t;
                    }

                    if (c >>>= y, p -= y, (y = s - a) < k) {
                      if (l < (y = k - y) && r.sane) {
                        t.msg = "invalid distance too far back", r.mode = 30;
                        break t;
                      }

                      if (S = d, (x = 0) === f) {
                        if (x += u - y, y < w) {
                          for (w -= y; C[s++] = d[x++], --y;);

                          x = s - k, S = C;
                        }
                      } else if (f < y) {
                        if (x += u + f - y, (y -= f) < w) {
                          for (w -= y; C[s++] = d[x++], --y;);

                          if (x = 0, f < w) {
                            for (w -= y = f; C[s++] = d[x++], --y;);

                            x = s - k, S = C;
                          }
                        }
                      } else if (x += f - y, y < w) {
                        for (w -= y; C[s++] = d[x++], --y;);

                        x = s - k, S = C;
                      }

                      for (; 2 < w;) C[s++] = S[x++], C[s++] = S[x++], C[s++] = S[x++], w -= 3;

                      w && (C[s++] = S[x++], 1 < w && (C[s++] = S[x++]));
                    } else {
                      for (x = s - k; C[s++] = C[x++], C[s++] = C[x++], C[s++] = C[x++], 2 < (w -= 3););

                      w && (C[s++] = C[x++], 1 < w && (C[s++] = C[x++]));
                    }

                    break;
                  }
                }
                break;
              }
            } while (i < n && s < o);

            i -= w = p >> 3, c &= (1 << (p -= w << 3)) - 1, t.next_in = i, t.next_out = s, t.avail_in = i < n ? n - i + 5 : 5 - (i - n), t.avail_out = s < o ? o - s + 257 : 257 - (s - o), r.hold = c, r.bits = p;
          };
        }, {}],
        49: [function (t, e, r) {

          var I = t("../utils/common"),
              O = t("./adler32"),
              B = t("./crc32"),
              R = t("./inffast"),
              T = t("./inftrees"),
              D = 1,
              F = 2,
              N = 0,
              U = -2,
              P = 1,
              i = 852,
              n = 592;

          function L(t) {
            return (t >>> 24 & 255) + (t >>> 8 & 65280) + ((65280 & t) << 8) + ((255 & t) << 24);
          }

          function s() {
            this.mode = 0, this.last = !1, this.wrap = 0, this.havedict = !1, this.flags = 0, this.dmax = 0, this.check = 0, this.total = 0, this.head = null, this.wbits = 0, this.wsize = 0, this.whave = 0, this.wnext = 0, this.window = null, this.hold = 0, this.bits = 0, this.length = 0, this.offset = 0, this.extra = 0, this.lencode = null, this.distcode = null, this.lenbits = 0, this.distbits = 0, this.ncode = 0, this.nlen = 0, this.ndist = 0, this.have = 0, this.next = null, this.lens = new I.Buf16(320), this.work = new I.Buf16(288), this.lendyn = null, this.distdyn = null, this.sane = 0, this.back = 0, this.was = 0;
          }

          function a(t) {
            var e;
            return t && t.state ? (e = t.state, t.total_in = t.total_out = e.total = 0, t.msg = "", e.wrap && (t.adler = 1 & e.wrap), e.mode = P, e.last = 0, e.havedict = 0, e.dmax = 32768, e.head = null, e.hold = 0, e.bits = 0, e.lencode = e.lendyn = new I.Buf32(i), e.distcode = e.distdyn = new I.Buf32(n), e.sane = 1, e.back = -1, N) : U;
          }

          function o(t) {
            var e;
            return t && t.state ? ((e = t.state).wsize = 0, e.whave = 0, e.wnext = 0, a(t)) : U;
          }

          function h(t, e) {
            var r, i;
            return t && t.state ? (i = t.state, e < 0 ? (r = 0, e = -e) : (r = 1 + (e >> 4), e < 48 && (e &= 15)), e && (e < 8 || 15 < e) ? U : (null !== i.window && i.wbits !== e && (i.window = null), i.wrap = r, i.wbits = e, o(t))) : U;
          }

          function u(t, e) {
            var r, i;
            return t ? (i = new s(), (t.state = i).window = null, (r = h(t, e)) !== N && (t.state = null), r) : U;
          }

          var l,
              f,
              d = !0;

          function j(t) {
            if (d) {
              var e;

              for (l = new I.Buf32(512), f = new I.Buf32(32), e = 0; e < 144;) t.lens[e++] = 8;

              for (; e < 256;) t.lens[e++] = 9;

              for (; e < 280;) t.lens[e++] = 7;

              for (; e < 288;) t.lens[e++] = 8;

              for (T(D, t.lens, 0, 288, l, 0, t.work, {
                bits: 9
              }), e = 0; e < 32;) t.lens[e++] = 5;

              T(F, t.lens, 0, 32, f, 0, t.work, {
                bits: 5
              }), d = !1;
            }

            t.lencode = l, t.lenbits = 9, t.distcode = f, t.distbits = 5;
          }

          function Z(t, e, r, i) {
            var n,
                s = t.state;
            return null === s.window && (s.wsize = 1 << s.wbits, s.wnext = 0, s.whave = 0, s.window = new I.Buf8(s.wsize)), i >= s.wsize ? (I.arraySet(s.window, e, r - s.wsize, s.wsize, 0), s.wnext = 0, s.whave = s.wsize) : (i < (n = s.wsize - s.wnext) && (n = i), I.arraySet(s.window, e, r - i, n, s.wnext), (i -= n) ? (I.arraySet(s.window, e, r - i, i, 0), s.wnext = i, s.whave = s.wsize) : (s.wnext += n, s.wnext === s.wsize && (s.wnext = 0), s.whave < s.wsize && (s.whave += n))), 0;
          }

          r.inflateReset = o, r.inflateReset2 = h, r.inflateResetKeep = a, r.inflateInit = function (t) {
            return u(t, 15);
          }, r.inflateInit2 = u, r.inflate = function (t, e) {
            var r,
                i,
                n,
                s,
                a,
                o,
                h,
                u,
                l,
                f,
                d,
                c,
                p,
                m,
                _,
                g,
                b,
                v,
                y,
                w,
                k,
                x,
                S,
                z,
                C = 0,
                E = new I.Buf8(4),
                A = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];

            if (!t || !t.state || !t.output || !t.input && 0 !== t.avail_in) return U;
            12 === (r = t.state).mode && (r.mode = 13), a = t.next_out, n = t.output, h = t.avail_out, s = t.next_in, i = t.input, o = t.avail_in, u = r.hold, l = r.bits, f = o, d = h, x = N;

            t: for (;;) switch (r.mode) {
              case P:
                if (0 === r.wrap) {
                  r.mode = 13;
                  break;
                }

                for (; l < 16;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if (2 & r.wrap && 35615 === u) {
                  E[r.check = 0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0), l = u = 0, r.mode = 2;
                  break;
                }

                if (r.flags = 0, r.head && (r.head.done = !1), !(1 & r.wrap) || (((255 & u) << 8) + (u >> 8)) % 31) {
                  t.msg = "incorrect header check", r.mode = 30;
                  break;
                }

                if (8 != (15 & u)) {
                  t.msg = "unknown compression method", r.mode = 30;
                  break;
                }

                if (l -= 4, k = 8 + (15 & (u >>>= 4)), 0 === r.wbits) r.wbits = k;else if (k > r.wbits) {
                  t.msg = "invalid window size", r.mode = 30;
                  break;
                }
                r.dmax = 1 << k, t.adler = r.check = 1, r.mode = 512 & u ? 10 : 12, l = u = 0;
                break;

              case 2:
                for (; l < 16;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if (r.flags = u, 8 != (255 & r.flags)) {
                  t.msg = "unknown compression method", r.mode = 30;
                  break;
                }

                if (57344 & r.flags) {
                  t.msg = "unknown header flags set", r.mode = 30;
                  break;
                }

                r.head && (r.head.text = u >> 8 & 1), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 3;

              case 3:
                for (; l < 32;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                r.head && (r.head.time = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, E[2] = u >>> 16 & 255, E[3] = u >>> 24 & 255, r.check = B(r.check, E, 4, 0)), l = u = 0, r.mode = 4;

              case 4:
                for (; l < 16;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                r.head && (r.head.xflags = 255 & u, r.head.os = u >> 8), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 5;

              case 5:
                if (1024 & r.flags) {
                  for (; l < 16;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.length = u, r.head && (r.head.extra_len = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0;
                } else r.head && (r.head.extra = null);

                r.mode = 6;

              case 6:
                if (1024 & r.flags && (o < (c = r.length) && (c = o), c && (r.head && (k = r.head.extra_len - r.length, r.head.extra || (r.head.extra = new Array(r.head.extra_len)), I.arraySet(r.head.extra, i, s, c, k)), 512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, r.length -= c), r.length)) break t;
                r.length = 0, r.mode = 7;

              case 7:
                if (2048 & r.flags) {
                  if (0 === o) break t;

                  for (c = 0; k = i[s + c++], r.head && k && r.length < 65536 && (r.head.name += String.fromCharCode(k)), k && c < o;);

                  if (512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, k) break t;
                } else r.head && (r.head.name = null);

                r.length = 0, r.mode = 8;

              case 8:
                if (4096 & r.flags) {
                  if (0 === o) break t;

                  for (c = 0; k = i[s + c++], r.head && k && r.length < 65536 && (r.head.comment += String.fromCharCode(k)), k && c < o;);

                  if (512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, k) break t;
                } else r.head && (r.head.comment = null);

                r.mode = 9;

              case 9:
                if (512 & r.flags) {
                  for (; l < 16;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (u !== (65535 & r.check)) {
                    t.msg = "header crc mismatch", r.mode = 30;
                    break;
                  }

                  l = u = 0;
                }

                r.head && (r.head.hcrc = r.flags >> 9 & 1, r.head.done = !0), t.adler = r.check = 0, r.mode = 12;
                break;

              case 10:
                for (; l < 32;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                t.adler = r.check = L(u), l = u = 0, r.mode = 11;

              case 11:
                if (0 === r.havedict) return t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, 2;
                t.adler = r.check = 1, r.mode = 12;

              case 12:
                if (5 === e || 6 === e) break t;

              case 13:
                if (r.last) {
                  u >>>= 7 & l, l -= 7 & l, r.mode = 27;
                  break;
                }

                for (; l < 3;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                switch (r.last = 1 & u, l -= 1, 3 & (u >>>= 1)) {
                  case 0:
                    r.mode = 14;
                    break;

                  case 1:
                    if (j(r), r.mode = 20, 6 !== e) break;
                    u >>>= 2, l -= 2;
                    break t;

                  case 2:
                    r.mode = 17;
                    break;

                  case 3:
                    t.msg = "invalid block type", r.mode = 30;
                }

                u >>>= 2, l -= 2;
                break;

              case 14:
                for (u >>>= 7 & l, l -= 7 & l; l < 32;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if ((65535 & u) != (u >>> 16 ^ 65535)) {
                  t.msg = "invalid stored block lengths", r.mode = 30;
                  break;
                }

                if (r.length = 65535 & u, l = u = 0, r.mode = 15, 6 === e) break t;

              case 15:
                r.mode = 16;

              case 16:
                if (c = r.length) {
                  if (o < c && (c = o), h < c && (c = h), 0 === c) break t;
                  I.arraySet(n, i, s, c, a), o -= c, s += c, h -= c, a += c, r.length -= c;
                  break;
                }

                r.mode = 12;
                break;

              case 17:
                for (; l < 14;) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if (r.nlen = 257 + (31 & u), u >>>= 5, l -= 5, r.ndist = 1 + (31 & u), u >>>= 5, l -= 5, r.ncode = 4 + (15 & u), u >>>= 4, l -= 4, 286 < r.nlen || 30 < r.ndist) {
                  t.msg = "too many length or distance symbols", r.mode = 30;
                  break;
                }

                r.have = 0, r.mode = 18;

              case 18:
                for (; r.have < r.ncode;) {
                  for (; l < 3;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.lens[A[r.have++]] = 7 & u, u >>>= 3, l -= 3;
                }

                for (; r.have < 19;) r.lens[A[r.have++]] = 0;

                if (r.lencode = r.lendyn, r.lenbits = 7, S = {
                  bits: r.lenbits
                }, x = T(0, r.lens, 0, 19, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                  t.msg = "invalid code lengths set", r.mode = 30;
                  break;
                }

                r.have = 0, r.mode = 19;

              case 19:
                for (; r.have < r.nlen + r.ndist;) {
                  for (; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (b < 16) u >>>= _, l -= _, r.lens[r.have++] = b;else {
                    if (16 === b) {
                      for (z = _ + 2; l < z;) {
                        if (0 === o) break t;
                        o--, u += i[s++] << l, l += 8;
                      }

                      if (u >>>= _, l -= _, 0 === r.have) {
                        t.msg = "invalid bit length repeat", r.mode = 30;
                        break;
                      }

                      k = r.lens[r.have - 1], c = 3 + (3 & u), u >>>= 2, l -= 2;
                    } else if (17 === b) {
                      for (z = _ + 3; l < z;) {
                        if (0 === o) break t;
                        o--, u += i[s++] << l, l += 8;
                      }

                      l -= _, k = 0, c = 3 + (7 & (u >>>= _)), u >>>= 3, l -= 3;
                    } else {
                      for (z = _ + 7; l < z;) {
                        if (0 === o) break t;
                        o--, u += i[s++] << l, l += 8;
                      }

                      l -= _, k = 0, c = 11 + (127 & (u >>>= _)), u >>>= 7, l -= 7;
                    }

                    if (r.have + c > r.nlen + r.ndist) {
                      t.msg = "invalid bit length repeat", r.mode = 30;
                      break;
                    }

                    for (; c--;) r.lens[r.have++] = k;
                  }
                }

                if (30 === r.mode) break;

                if (0 === r.lens[256]) {
                  t.msg = "invalid code -- missing end-of-block", r.mode = 30;
                  break;
                }

                if (r.lenbits = 9, S = {
                  bits: r.lenbits
                }, x = T(D, r.lens, 0, r.nlen, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                  t.msg = "invalid literal/lengths set", r.mode = 30;
                  break;
                }

                if (r.distbits = 6, r.distcode = r.distdyn, S = {
                  bits: r.distbits
                }, x = T(F, r.lens, r.nlen, r.ndist, r.distcode, 0, r.work, S), r.distbits = S.bits, x) {
                  t.msg = "invalid distances set", r.mode = 30;
                  break;
                }

                if (r.mode = 20, 6 === e) break t;

              case 20:
                r.mode = 21;

              case 21:
                if (6 <= o && 258 <= h) {
                  t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, R(t, d), a = t.next_out, n = t.output, h = t.avail_out, s = t.next_in, i = t.input, o = t.avail_in, u = r.hold, l = r.bits, 12 === r.mode && (r.back = -1);
                  break;
                }

                for (r.back = 0; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if (g && 0 == (240 & g)) {
                  for (v = _, y = g, w = b; g = (C = r.lencode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  u >>>= v, l -= v, r.back += v;
                }

                if (u >>>= _, l -= _, r.back += _, r.length = b, 0 === g) {
                  r.mode = 26;
                  break;
                }

                if (32 & g) {
                  r.back = -1, r.mode = 12;
                  break;
                }

                if (64 & g) {
                  t.msg = "invalid literal/length code", r.mode = 30;
                  break;
                }

                r.extra = 15 & g, r.mode = 22;

              case 22:
                if (r.extra) {
                  for (z = r.extra; l < z;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.length += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                }

                r.was = r.length, r.mode = 23;

              case 23:
                for (; g = (C = r.distcode[u & (1 << r.distbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                  if (0 === o) break t;
                  o--, u += i[s++] << l, l += 8;
                }

                if (0 == (240 & g)) {
                  for (v = _, y = g, w = b; g = (C = r.distcode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  u >>>= v, l -= v, r.back += v;
                }

                if (u >>>= _, l -= _, r.back += _, 64 & g) {
                  t.msg = "invalid distance code", r.mode = 30;
                  break;
                }

                r.offset = b, r.extra = 15 & g, r.mode = 24;

              case 24:
                if (r.extra) {
                  for (z = r.extra; l < z;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.offset += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                }

                if (r.offset > r.dmax) {
                  t.msg = "invalid distance too far back", r.mode = 30;
                  break;
                }

                r.mode = 25;

              case 25:
                if (0 === h) break t;

                if (c = d - h, r.offset > c) {
                  if ((c = r.offset - c) > r.whave && r.sane) {
                    t.msg = "invalid distance too far back", r.mode = 30;
                    break;
                  }

                  p = c > r.wnext ? (c -= r.wnext, r.wsize - c) : r.wnext - c, c > r.length && (c = r.length), m = r.window;
                } else m = n, p = a - r.offset, c = r.length;

                for (h < c && (c = h), h -= c, r.length -= c; n[a++] = m[p++], --c;);

                0 === r.length && (r.mode = 21);
                break;

              case 26:
                if (0 === h) break t;
                n[a++] = r.length, h--, r.mode = 21;
                break;

              case 27:
                if (r.wrap) {
                  for (; l < 32;) {
                    if (0 === o) break t;
                    o--, u |= i[s++] << l, l += 8;
                  }

                  if (d -= h, t.total_out += d, r.total += d, d && (t.adler = r.check = r.flags ? B(r.check, n, d, a - d) : O(r.check, n, d, a - d)), d = h, (r.flags ? u : L(u)) !== r.check) {
                    t.msg = "incorrect data check", r.mode = 30;
                    break;
                  }

                  l = u = 0;
                }

                r.mode = 28;

              case 28:
                if (r.wrap && r.flags) {
                  for (; l < 32;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (u !== (4294967295 & r.total)) {
                    t.msg = "incorrect length check", r.mode = 30;
                    break;
                  }

                  l = u = 0;
                }

                r.mode = 29;

              case 29:
                x = 1;
                break t;

              case 30:
                x = -3;
                break t;

              case 31:
                return -4;

              case 32:
              default:
                return U;
            }

            return t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, (r.wsize || d !== t.avail_out && r.mode < 30 && (r.mode < 27 || 4 !== e)) && Z(t, t.output, t.next_out, d - t.avail_out) ? (r.mode = 31, -4) : (f -= t.avail_in, d -= t.avail_out, t.total_in += f, t.total_out += d, r.total += d, r.wrap && d && (t.adler = r.check = r.flags ? B(r.check, n, d, t.next_out - d) : O(r.check, n, d, t.next_out - d)), t.data_type = r.bits + (r.last ? 64 : 0) + (12 === r.mode ? 128 : 0) + (20 === r.mode || 15 === r.mode ? 256 : 0), (0 == f && 0 === d || 4 === e) && x === N && (x = -5), x);
          }, r.inflateEnd = function (t) {
            if (!t || !t.state) return U;
            var e = t.state;
            return e.window && (e.window = null), t.state = null, N;
          }, r.inflateGetHeader = function (t, e) {
            var r;
            return t && t.state ? 0 == (2 & (r = t.state).wrap) ? U : ((r.head = e).done = !1, N) : U;
          }, r.inflateSetDictionary = function (t, e) {
            var r,
                i = e.length;
            return t && t.state ? 0 !== (r = t.state).wrap && 11 !== r.mode ? U : 11 === r.mode && O(1, e, i, 0) !== r.check ? -3 : Z(t, e, i, i) ? (r.mode = 31, -4) : (r.havedict = 1, N) : U;
          }, r.inflateInfo = "pako inflate (from Nodeca project)";
        }, {
          "../utils/common": 41,
          "./adler32": 43,
          "./crc32": 45,
          "./inffast": 48,
          "./inftrees": 50
        }],
        50: [function (t, e, r) {

          var D = t("../utils/common"),
              F = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0],
              N = [16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78],
              U = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0],
              P = [16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];

          e.exports = function (t, e, r, i, n, s, a, o) {
            var h,
                u,
                l,
                f,
                d,
                c,
                p,
                m,
                _,
                g = o.bits,
                b = 0,
                v = 0,
                y = 0,
                w = 0,
                k = 0,
                x = 0,
                S = 0,
                z = 0,
                C = 0,
                E = 0,
                A = null,
                I = 0,
                O = new D.Buf16(16),
                B = new D.Buf16(16),
                R = null,
                T = 0;

            for (b = 0; b <= 15; b++) O[b] = 0;

            for (v = 0; v < i; v++) O[e[r + v]]++;

            for (k = g, w = 15; 1 <= w && 0 === O[w]; w--);

            if (w < k && (k = w), 0 === w) return n[s++] = 20971520, n[s++] = 20971520, o.bits = 1, 0;

            for (y = 1; y < w && 0 === O[y]; y++);

            for (k < y && (k = y), b = z = 1; b <= 15; b++) if (z <<= 1, (z -= O[b]) < 0) return -1;

            if (0 < z && (0 === t || 1 !== w)) return -1;

            for (B[1] = 0, b = 1; b < 15; b++) B[b + 1] = B[b] + O[b];

            for (v = 0; v < i; v++) 0 !== e[r + v] && (a[B[e[r + v]]++] = v);

            if (c = 0 === t ? (A = R = a, 19) : 1 === t ? (A = F, I -= 257, R = N, T -= 257, 256) : (A = U, R = P, -1), b = y, d = s, S = v = E = 0, l = -1, f = (C = 1 << (x = k)) - 1, 1 === t && 852 < C || 2 === t && 592 < C) return 1;

            for (;;) {
              for (p = b - S, _ = a[v] < c ? (m = 0, a[v]) : a[v] > c ? (m = R[T + a[v]], A[I + a[v]]) : (m = 96, 0), h = 1 << b - S, y = u = 1 << x; n[d + (E >> S) + (u -= h)] = p << 24 | m << 16 | _ | 0, 0 !== u;);

              for (h = 1 << b - 1; E & h;) h >>= 1;

              if (0 !== h ? (E &= h - 1, E += h) : E = 0, v++, 0 == --O[b]) {
                if (b === w) break;
                b = e[r + a[v]];
              }

              if (k < b && (E & f) !== l) {
                for (0 === S && (S = k), d += y, z = 1 << (x = b - S); x + S < w && !((z -= O[x + S]) <= 0);) x++, z <<= 1;

                if (C += 1 << x, 1 === t && 852 < C || 2 === t && 592 < C) return 1;
                n[l = E & f] = k << 24 | x << 16 | d - s | 0;
              }
            }

            return 0 !== E && (n[d + E] = b - S << 24 | 64 << 16 | 0), o.bits = k, 0;
          };
        }, {
          "../utils/common": 41
        }],
        51: [function (t, e, r) {

          e.exports = {
            2: "need dictionary",
            1: "stream end",
            0: "",
            "-1": "file error",
            "-2": "stream error",
            "-3": "data error",
            "-4": "insufficient memory",
            "-5": "buffer error",
            "-6": "incompatible version"
          };
        }, {}],
        52: [function (t, e, r) {

          var n = t("../utils/common"),
              o = 0,
              h = 1;

          function i(t) {
            for (var e = t.length; 0 <= --e;) t[e] = 0;
          }

          var s = 0,
              a = 29,
              u = 256,
              l = u + 1 + a,
              f = 30,
              d = 19,
              _ = 2 * l + 1,
              g = 15,
              c = 16,
              p = 7,
              m = 256,
              b = 16,
              v = 17,
              y = 18,
              w = [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0],
              k = [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13],
              x = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7],
              S = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15],
              z = new Array(2 * (l + 2));

          i(z);
          var C = new Array(2 * f);
          i(C);
          var E = new Array(512);
          i(E);
          var A = new Array(256);
          i(A);
          var I = new Array(a);
          i(I);
          var O,
              B,
              R,
              T = new Array(f);

          function D(t, e, r, i, n) {
            this.static_tree = t, this.extra_bits = e, this.extra_base = r, this.elems = i, this.max_length = n, this.has_stree = t && t.length;
          }

          function F(t, e) {
            this.dyn_tree = t, this.max_code = 0, this.stat_desc = e;
          }

          function N(t) {
            return t < 256 ? E[t] : E[256 + (t >>> 7)];
          }

          function U(t, e) {
            t.pending_buf[t.pending++] = 255 & e, t.pending_buf[t.pending++] = e >>> 8 & 255;
          }

          function P(t, e, r) {
            t.bi_valid > c - r ? (t.bi_buf |= e << t.bi_valid & 65535, U(t, t.bi_buf), t.bi_buf = e >> c - t.bi_valid, t.bi_valid += r - c) : (t.bi_buf |= e << t.bi_valid & 65535, t.bi_valid += r);
          }

          function L(t, e, r) {
            P(t, r[2 * e], r[2 * e + 1]);
          }

          function j(t, e) {
            for (var r = 0; r |= 1 & t, t >>>= 1, r <<= 1, 0 < --e;);

            return r >>> 1;
          }

          function Z(t, e, r) {
            var i,
                n,
                s = new Array(g + 1),
                a = 0;

            for (i = 1; i <= g; i++) s[i] = a = a + r[i - 1] << 1;

            for (n = 0; n <= e; n++) {
              var o = t[2 * n + 1];
              0 !== o && (t[2 * n] = j(s[o]++, o));
            }
          }

          function W(t) {
            var e;

            for (e = 0; e < l; e++) t.dyn_ltree[2 * e] = 0;

            for (e = 0; e < f; e++) t.dyn_dtree[2 * e] = 0;

            for (e = 0; e < d; e++) t.bl_tree[2 * e] = 0;

            t.dyn_ltree[2 * m] = 1, t.opt_len = t.static_len = 0, t.last_lit = t.matches = 0;
          }

          function M(t) {
            8 < t.bi_valid ? U(t, t.bi_buf) : 0 < t.bi_valid && (t.pending_buf[t.pending++] = t.bi_buf), t.bi_buf = 0, t.bi_valid = 0;
          }

          function H(t, e, r, i) {
            var n = 2 * e,
                s = 2 * r;
            return t[n] < t[s] || t[n] === t[s] && i[e] <= i[r];
          }

          function G(t, e, r) {
            for (var i = t.heap[r], n = r << 1; n <= t.heap_len && (n < t.heap_len && H(e, t.heap[n + 1], t.heap[n], t.depth) && n++, !H(e, i, t.heap[n], t.depth));) t.heap[r] = t.heap[n], r = n, n <<= 1;

            t.heap[r] = i;
          }

          function K(t, e, r) {
            var i,
                n,
                s,
                a,
                o = 0;
            if (0 !== t.last_lit) for (; i = t.pending_buf[t.d_buf + 2 * o] << 8 | t.pending_buf[t.d_buf + 2 * o + 1], n = t.pending_buf[t.l_buf + o], o++, 0 === i ? L(t, n, e) : (L(t, (s = A[n]) + u + 1, e), 0 !== (a = w[s]) && P(t, n -= I[s], a), L(t, s = N(--i), r), 0 !== (a = k[s]) && P(t, i -= T[s], a)), o < t.last_lit;);
            L(t, m, e);
          }

          function Y(t, e) {
            var r,
                i,
                n,
                s = e.dyn_tree,
                a = e.stat_desc.static_tree,
                o = e.stat_desc.has_stree,
                h = e.stat_desc.elems,
                u = -1;

            for (t.heap_len = 0, t.heap_max = _, r = 0; r < h; r++) 0 !== s[2 * r] ? (t.heap[++t.heap_len] = u = r, t.depth[r] = 0) : s[2 * r + 1] = 0;

            for (; t.heap_len < 2;) s[2 * (n = t.heap[++t.heap_len] = u < 2 ? ++u : 0)] = 1, t.depth[n] = 0, t.opt_len--, o && (t.static_len -= a[2 * n + 1]);

            for (e.max_code = u, r = t.heap_len >> 1; 1 <= r; r--) G(t, s, r);

            for (n = h; r = t.heap[1], t.heap[1] = t.heap[t.heap_len--], G(t, s, 1), i = t.heap[1], t.heap[--t.heap_max] = r, t.heap[--t.heap_max] = i, s[2 * n] = s[2 * r] + s[2 * i], t.depth[n] = (t.depth[r] >= t.depth[i] ? t.depth[r] : t.depth[i]) + 1, s[2 * r + 1] = s[2 * i + 1] = n, t.heap[1] = n++, G(t, s, 1), 2 <= t.heap_len;);

            t.heap[--t.heap_max] = t.heap[1], function (t, e) {
              var r,
                  i,
                  n,
                  s,
                  a,
                  o,
                  h = e.dyn_tree,
                  u = e.max_code,
                  l = e.stat_desc.static_tree,
                  f = e.stat_desc.has_stree,
                  d = e.stat_desc.extra_bits,
                  c = e.stat_desc.extra_base,
                  p = e.stat_desc.max_length,
                  m = 0;

              for (s = 0; s <= g; s++) t.bl_count[s] = 0;

              for (h[2 * t.heap[t.heap_max] + 1] = 0, r = t.heap_max + 1; r < _; r++) p < (s = h[2 * h[2 * (i = t.heap[r]) + 1] + 1] + 1) && (s = p, m++), h[2 * i + 1] = s, u < i || (t.bl_count[s]++, a = 0, c <= i && (a = d[i - c]), o = h[2 * i], t.opt_len += o * (s + a), f && (t.static_len += o * (l[2 * i + 1] + a)));

              if (0 !== m) {
                do {
                  for (s = p - 1; 0 === t.bl_count[s];) s--;

                  t.bl_count[s]--, t.bl_count[s + 1] += 2, t.bl_count[p]--, m -= 2;
                } while (0 < m);

                for (s = p; 0 !== s; s--) for (i = t.bl_count[s]; 0 !== i;) u < (n = t.heap[--r]) || (h[2 * n + 1] !== s && (t.opt_len += (s - h[2 * n + 1]) * h[2 * n], h[2 * n + 1] = s), i--);
              }
            }(t, e), Z(s, u, t.bl_count);
          }

          function X(t, e, r) {
            var i,
                n,
                s = -1,
                a = e[1],
                o = 0,
                h = 7,
                u = 4;

            for (0 === a && (h = 138, u = 3), e[2 * (r + 1) + 1] = 65535, i = 0; i <= r; i++) n = a, a = e[2 * (i + 1) + 1], ++o < h && n === a || (o < u ? t.bl_tree[2 * n] += o : 0 !== n ? (n !== s && t.bl_tree[2 * n]++, t.bl_tree[2 * b]++) : o <= 10 ? t.bl_tree[2 * v]++ : t.bl_tree[2 * y]++, s = n, u = (o = 0) === a ? (h = 138, 3) : n === a ? (h = 6, 3) : (h = 7, 4));
          }

          function V(t, e, r) {
            var i,
                n,
                s = -1,
                a = e[1],
                o = 0,
                h = 7,
                u = 4;

            for (0 === a && (h = 138, u = 3), i = 0; i <= r; i++) if (n = a, a = e[2 * (i + 1) + 1], !(++o < h && n === a)) {
              if (o < u) for (; L(t, n, t.bl_tree), 0 != --o;);else 0 !== n ? (n !== s && (L(t, n, t.bl_tree), o--), L(t, b, t.bl_tree), P(t, o - 3, 2)) : o <= 10 ? (L(t, v, t.bl_tree), P(t, o - 3, 3)) : (L(t, y, t.bl_tree), P(t, o - 11, 7));
              s = n, u = (o = 0) === a ? (h = 138, 3) : n === a ? (h = 6, 3) : (h = 7, 4);
            }
          }

          i(T);
          var q = !1;

          function J(t, e, r, i) {
            P(t, (s << 1) + (i ? 1 : 0), 3), function (t, e, r, i) {
              M(t), i && (U(t, r), U(t, ~r)), n.arraySet(t.pending_buf, t.window, e, r, t.pending), t.pending += r;
            }(t, e, r, !0);
          }

          r._tr_init = function (t) {
            q || (function () {
              var t,
                  e,
                  r,
                  i,
                  n,
                  s = new Array(g + 1);

              for (i = r = 0; i < a - 1; i++) for (I[i] = r, t = 0; t < 1 << w[i]; t++) A[r++] = i;

              for (A[r - 1] = i, i = n = 0; i < 16; i++) for (T[i] = n, t = 0; t < 1 << k[i]; t++) E[n++] = i;

              for (n >>= 7; i < f; i++) for (T[i] = n << 7, t = 0; t < 1 << k[i] - 7; t++) E[256 + n++] = i;

              for (e = 0; e <= g; e++) s[e] = 0;

              for (t = 0; t <= 143;) z[2 * t + 1] = 8, t++, s[8]++;

              for (; t <= 255;) z[2 * t + 1] = 9, t++, s[9]++;

              for (; t <= 279;) z[2 * t + 1] = 7, t++, s[7]++;

              for (; t <= 287;) z[2 * t + 1] = 8, t++, s[8]++;

              for (Z(z, l + 1, s), t = 0; t < f; t++) C[2 * t + 1] = 5, C[2 * t] = j(t, 5);

              O = new D(z, w, u + 1, l, g), B = new D(C, k, 0, f, g), R = new D(new Array(0), x, 0, d, p);
            }(), q = !0), t.l_desc = new F(t.dyn_ltree, O), t.d_desc = new F(t.dyn_dtree, B), t.bl_desc = new F(t.bl_tree, R), t.bi_buf = 0, t.bi_valid = 0, W(t);
          }, r._tr_stored_block = J, r._tr_flush_block = function (t, e, r, i) {
            var n,
                s,
                a = 0;
            0 < t.level ? (2 === t.strm.data_type && (t.strm.data_type = function (t) {
              var e,
                  r = 4093624447;

              for (e = 0; e <= 31; e++, r >>>= 1) if (1 & r && 0 !== t.dyn_ltree[2 * e]) return o;

              if (0 !== t.dyn_ltree[18] || 0 !== t.dyn_ltree[20] || 0 !== t.dyn_ltree[26]) return h;

              for (e = 32; e < u; e++) if (0 !== t.dyn_ltree[2 * e]) return h;

              return o;
            }(t)), Y(t, t.l_desc), Y(t, t.d_desc), a = function (t) {
              var e;

              for (X(t, t.dyn_ltree, t.l_desc.max_code), X(t, t.dyn_dtree, t.d_desc.max_code), Y(t, t.bl_desc), e = d - 1; 3 <= e && 0 === t.bl_tree[2 * S[e] + 1]; e--);

              return t.opt_len += 3 * (e + 1) + 5 + 5 + 4, e;
            }(t), n = t.opt_len + 3 + 7 >>> 3, (s = t.static_len + 3 + 7 >>> 3) <= n && (n = s)) : n = s = r + 5, r + 4 <= n && -1 !== e ? J(t, e, r, i) : 4 === t.strategy || s === n ? (P(t, 2 + (i ? 1 : 0), 3), K(t, z, C)) : (P(t, 4 + (i ? 1 : 0), 3), function (t, e, r, i) {
              var n;

              for (P(t, e - 257, 5), P(t, r - 1, 5), P(t, i - 4, 4), n = 0; n < i; n++) P(t, t.bl_tree[2 * S[n] + 1], 3);

              V(t, t.dyn_ltree, e - 1), V(t, t.dyn_dtree, r - 1);
            }(t, t.l_desc.max_code + 1, t.d_desc.max_code + 1, a + 1), K(t, t.dyn_ltree, t.dyn_dtree)), W(t), i && M(t);
          }, r._tr_tally = function (t, e, r) {
            return t.pending_buf[t.d_buf + 2 * t.last_lit] = e >>> 8 & 255, t.pending_buf[t.d_buf + 2 * t.last_lit + 1] = 255 & e, t.pending_buf[t.l_buf + t.last_lit] = 255 & r, t.last_lit++, 0 === e ? t.dyn_ltree[2 * r]++ : (t.matches++, e--, t.dyn_ltree[2 * (A[r] + u + 1)]++, t.dyn_dtree[2 * N(e)]++), t.last_lit === t.lit_bufsize - 1;
          }, r._tr_align = function (t) {
            P(t, 2, 3), L(t, m, z), function (t) {
              16 === t.bi_valid ? (U(t, t.bi_buf), t.bi_buf = 0, t.bi_valid = 0) : 8 <= t.bi_valid && (t.pending_buf[t.pending++] = 255 & t.bi_buf, t.bi_buf >>= 8, t.bi_valid -= 8);
            }(t);
          };
        }, {
          "../utils/common": 41
        }],
        53: [function (t, e, r) {

          e.exports = function () {
            this.input = null, this.next_in = 0, this.avail_in = 0, this.total_in = 0, this.output = null, this.next_out = 0, this.avail_out = 0, this.total_out = 0, this.msg = "", this.state = null, this.data_type = 2, this.adler = 0;
          };
        }, {}],
        54: [function (t, e, r) {

          e.exports = "function" == typeof setImmediate ? setImmediate : function () {
            var t = [].slice.apply(arguments);
            t.splice(1, 0, 0), setTimeout.apply(null, t);
          };
        }, {}]
      }, {}, [10])(10);
    });
  })(jszip_min);

  var JSZip = jszip_min.exports;

  /* eslint-disable no-unused-vars */
  /**
   * Parse metadata from xml file that is contained in BRML
   * @export
   * @param {String} file
   * @returns {Object} containing data (x: 2theta, y: counts), info and metadata
   */

  function parseDiffractogram(file) {
    let json = parser.parse(file, {
      ignoreAttributes: false,
      attributeNamePrefix: '__'
    });
    const data = json.RawData;
    let axes = data.DataRoutes.DataRoute.ScanInformation.ScanAxes.ScanAxisInfo.map(element => ({
      id: element.__AxisId,
      name: element.__AxisName,
      unitBase: element.Unit.__Base,
      unitPrefix: element.Unit.__Prefix,
      reference: element.reference,
      start: element.start,
      stop: element.stop,
      increment: element.increment
    }));
    let adddata = {
      startTime: data.TimeStampStarted,
      endTime: data.TimeStampFinished,
      measurmentPoints: data.DataRoutes.DataRoute.ScanInformation.MeasurementPoints,
      timePerStep: data.DataRoutes.DataRoute.ScanInformation.TimePerStep,
      timePerStepEffective: data.DataRoutes.DataRoute.ScanInformation.TimePerStepEffective,
      scanMode: data.DataRoutes.DataRoute.ScanInformation.ScanMode,
      scanModeVisibleName: data.DataRoutes.DataRoute.ScanInformation.ScanModeVisibleName,
      userName: data.Identifier.__UserName,
      machineName: data.Identifier.__MachineName,
      guid: data.Identifier.Guid,
      axes: axes,
      goniometerType: data.FixedInformation.Instrument.GoniometerType,
      anode: data.FixedInformation.Instrument.PrimaryTracks.TrackInfoData.MountedOptics.InfoData[0].Tube.TubeMaterial,
      anodeVoltage: data.FixedInformation.Instrument.PrimaryTracks.TrackInfoData.MountedOptics.InfoData[0].Tube.Generator.Voltage.__Value,
      anodeVoltageUnit: data.FixedInformation.Instrument.PrimaryTracks.TrackInfoData.MountedOptics.InfoData[0].Tube.Generator.Voltage.__Unit,
      anodeSerialNumber: data.FixedInformation.Instrument.PrimaryTracks.TrackInfoData.MountedOptics.InfoData[0].Tube.SerialNumber.__Value
    };
    const diffractogram = getXYDiffractogram(data.DataRoutes.DataRoute.Datum);
    diffractogram.meta = { ...adddata,
      ...diffractogram.meta
    };
    let label = adddata.axes[0].name.replace(/two/i, '2').replace(/theta/i, 'ϴ');
    let unit = adddata.axes[0].unitBase.replace(/degree/i, '°');
    diffractogram.info.xUnits = `${label} [${unit}]`;
    return diffractogram;
  }
  /**
   * @param  {array} data array of strings of the measured points
   */

  function getXYDiffractogram(data) {
    let axis1 = [];
    let axis2 = [];
    let measuredTimePerStep = [];
    let plannedTimePerStep = [];
    let counts = [];
    data.forEach(element => {
      const factors = element.split(',');
      measuredTimePerStep.push(parseFloat(factors[0]));
      plannedTimePerStep.push(parseFloat(factors[1]));
      axis1.push(parseFloat(factors[2]));
      axis2.push(parseFloat(factors[3]));
      counts.push(parseFloat(factors[4]));
    });
    const diffractogram = {
      data: {
        x: axis1,
        y: counts
      },
      info: {
        xUnits: '2ϴ [°]',
        yUnits: 'counts',
        dataType: 'XRD pattern',
        origin: 'Data converted from BRML using convert-to-jcamp'
      },
      meta: {
        axis2: axis2,
        measuredTimePerStep: measuredTimePerStep,
        plannedTimePerStep: plannedTimePerStep
      }
    };
    return diffractogram;
  }
  /**
   * Read a BRML file (produced by Bruker instruments, a zip file that contains XMLs)
   * @export
   * @param {String/Array of bytes/ArrayBuffer/Uint8Array/Buffer/Blob/Promise} binary BRML file
   * @returns {Object} containing data (x: 2theta, y: counts), info and metadata
   */


  async function readBRML(binary) {
    let zip = new JSZip();
    const txt = await zip.loadAsync(binary).then(zipFiles => {
      return zipFiles.file('Experiment0/RawData0.xml').async('text');
    });
    const diffractogram = parseDiffractogram(txt);
    return diffractogram;
  }

  /**
   * Creates a new Chromatogram element based in a JCAMP string
   * @param {UInt8Array} blob - String containing the JCAMP data
   * @return {Analysis} - New class element with the given data
   */

  async function fromBRML(blob) {
    let parsed = await readBRML(blob);
    let analysis = new Analysis();
    const variables = {
      x: {
        data: parsed.data.x,
        label: parsed.info.xUnits
      },
      y: {
        data: parsed.data.y,
        label: parsed.info.yUnits
      }
    };
    analysis.pushSpectrum(variables, {
      title: parsed.info.title,
      meta: parsed.meta,
      dataType: 'XRD pattern'
    });
    return analysis;
  }

  /**
   * Parse diffractograms saved in xy files that are generated with PowDLL
   * @export
   * @param {String} [text] Text containing the data
   * @returns {Object} containing data (x: 2theta, y: counts), info and metadata
   */
  function parsePowDLLXY(text) {
    let lines = text.split(/\r?\n/).filter(line => !line.match(/^\s*$/));
    const header = lines[0];
    lines.splice(0, 1); // remove header line

    let data = {
      x: [],
      y: []
    };

    for (const line of lines) {
      let tmp = line.split(/\s+/);
      data.x.push(parseFloat(tmp[0].trim()));
      data.y.push(parseFloat(tmp[1].trim()));
    }

    let headerLines = header.split('" '); // try to make metadata consistent with Bruker

    let meta = {};
    meta.id = trimReplace(headerLines[0]);
    meta.comment = trimReplace(headerLines[1]);
    meta.userName = trimReplace(headerLines[2]);
    meta.anode = trimReplace(headerLines[3]);
    meta.scanType = trimReplace(headerLines[4]); // eslint-disable-next-line radix

    meta.timePerStep = parseInt(trimReplace(headerLines[5]));
    const diffractogram = {
      data: {
        x: data.x,
        y: data.y
      },
      info: {
        xUnits: '2ϴ [°]',
        yUnits: 'counts',
        dataType: 'XRD pattern',
        origin: 'Data converted from xy using convert-to-jcamp'
      },
      meta: meta
    };
    return diffractogram;
  }

  function trimReplace(string) {
    return string.split(':')[1].replace('"', '').replace("'", '').trim();
  }

  /**
   * Creates a new Chromatogram element based in a JCAMP string
   * @param {UInt8Array} blob - String containing the JCAMP data
   * @return {Analysis} - New class element with the given data
   */

  function fromPowDLLXY(blob) {
    let parsed = parsePowDLLXY(blob);
    let analysis = new Analysis();
    const variables = {
      x: {
        data: parsed.data.x,
        label: parsed.info.xUnits
      },
      y: {
        data: parsed.data.y,
        label: parsed.info.yUnits
      }
    };
    analysis.pushSpectrum(variables, {
      title: parsed.info.title,
      meta: parsed.meta,
      dataType: 'XRD pattern'
    });
    return analysis;
  }

  // source: http://gisaxs.com/index.php/X-ray_energy
  const kAlpha1Angstrom = {
    CuKa: 1.54184,
    CuKa2: 1.54439,
    CuKa1: 1.54056,
    CuKb1: 1.39222,
    MoKa: 0.71073,
    MoKa2: 0.71359,
    MoKa1: 0.7093,
    MoKb1: 0.63229,
    CrKa: 2.291,
    CrKa2: 2.29361,
    CrKa1: 2.2897,
    CrKb1: 2.08487,
    FeKa: 1.93735,
    FeKa2: 1.93998,
    FeKa1: 1.93604,
    FeKb1: 1.75661,
    CoKa: 1.79026,
    CoKa2: 1.79285,
    CoKa1: 1.78896,
    CoKb1: 1.63079,
    AgKa: 0.560885,
    AgKa2: 0.563813,
    AgKa1: 0.559421,
    AgKb1: 0.497082
  };

  function toRadians(angle) {
    return angle * (Math.PI / 180);
  }
  function toDegrees(radians) {
    return radians * 180 / Math.PI;
  }
  function getLamba(anode) {
    if (!(anode in kAlpha1Angstrom)) {
      throw new Error('The wavelength for the anode metal in the metadata is not defined');
    }

    return kAlpha1Angstrom[anode];
  }

  /**
   * d = Order of Reflection (n) × Wavelength (λ) / 2 × Sinθ
   *
   */

  /**
   * Calculates the interplanar spacing d according to Bragg's equation
   *
   * @export
   * @param {Array<number>} twoThetas. Note that this is 2θ *not* θ.
   * @param {Object} options.
   * @param {Number} options.orderOfReflection. Order of relection n. Defaults to 1.
   * @param {String} options.anode. Anode type. Defaults to "CuKa" Available options in constants.wavelengths.kAlpha1Angstrom
   * @returns {Array<number>} Interplanar spacings in Angstrom.
   */

  function calculateInterplanarSpacing(twoThetas, options = {}) {
    let {
      orderOfReflection = 1,
      anode = 'CuKa'
    } = options;
    const wavelength = getLamba(anode);
    return twoThetas.map(theta => {
      return orderOfReflection * wavelength / (2 * Math.sin(toRadians(theta)));
    });
  }
  /**
   * Calculates the reflex angles based on the interplanar spacings.
   *
   * @export
   * @param {Array<number>} ds spacings. In Angstrom.
   * @param {Object} options.
   * @param {Number} options.orderOfReflection. Order of relection n. Defaults to 1.
   * @param {String} options.anode. Anode type. Defaults to "CuKa" Available options in constants.wavelengths.kAlpha1Angstrom
   * @returns {Array<number>} twoThetas.
   */

  function calculateReflexes(ds, options = {}) {
    let {
      orderOfReflection = 1,
      anode = 'CuKa'
    } = options;
    const wavelength = getLamba(anode);
    return ds.map(d => {
      return toDegrees(Math.asin(orderOfReflection * wavelength / (2 * d)));
    });
  }

  /* eslint-disable no-empty-function */

  /**
   * Calculate the crystallite size according to the Scherrer equation.
   * Please check the strong assumptions (e.g., grains smaller than 0.1 to 0.2 μm)
   * for this analysis (https://en.wikipedia.org/wiki/Scherrer_equation).
   * The Scherrer equation only provides a lower bound, there also might be instrument broadening
   * (http://prism.mit.edu/XRAY/oldsite/CrystalSizeAnalysis.pdf)
   * @export
   * @param {Number} k  shape factor. The shape factor has a typical value of about 0.9, but varies with the actual shape of the crystallite. For a discussion see J. Appl. Cryst. 11 (1978) p102-113
   * @param {Number} lambda X-ray wavelength (output will be of the same unit)
   * @param {Number} beta FWHM line broadening in degree
   * @param {Number} theta  Bragg angle in degree, i.e. theta and not 2 theta
   * @returns mean size of the ordered (crystalline) domains in the unit of lambda
   */

  function scherrer(k, lambda, beta, theta) {
    return k * lambda / (toRadians(beta) * Math.cos(toRadians(theta)));
  }
  /**
   * Computes the broadening accoding to the Scherrer equation for every reflex
   * in the diffractogram. Uses the anode metal deposited in the metadata to
   * get the wavelength and the peaks previosuly picked
   * @export
   * @param {*} spectrum -with anode metal as metadata and picked peaks with FWHM, all in 2 theta units
   * @param {Number} k shape factor
   * @returns {Array<Object>} peaks with crystalliteSize attribute
   */

  function scherrerForSpectrum(spectrum, k = 0.94) {
    let newPeaks = spectrum.peaks;

    if (!('peaks' in spectrum)) {
      throw new Error('There must be peaks to calculate the scherrer broadening');
    }

    if (!('anode' in spectrum.meta)) {
      throw new Error('The anode metal must be available in the metadata of the spectrum');
    }

    const lambda = getLamba(spectrum.meta.anode);

    for (let peak of newPeaks) {
      peak.crystalliteSize = scherrer(k, lambda, peak.width, peak.x / 2) / 100;
    }

    return newPeaks;
  }

  function computeCrystallinity(spectrum, options = {}) {
    let {
      shape = {
        kind: 'gaussian'
      }
    } = options;
    const totalArea = xyIntegration({
      x: spectrum.variables.x.data,
      y: spectrum.variables.y.data
    });
    let peakArea = 0;
    spectrum.peaks.forEach(peak => {
      peakArea += getShape1D(shape.kind, {
        fwhm: peak.width,
        height: peak.y
      }).getArea();
    });
    return peakArea / totalArea;
  }

  function generatePattern(spectrum, options = {}) {
    let {
      from = Math.min(...spectrum.variables.x.data),
      to = Math.max(...spectrum.variables.x.data),
      nbPoints = 999
    } = options;
    const generator = new SpectrumGenerator({
      from: from,
      to: to,
      nbPoints: nbPoints
    });
    generator.addPeaks(spectrum.peaks);
    return generator.getSpectrum();
  }

  const JSGraph = { ...JSGraph$1,
    getAnnotations
  };

  exports.AnalysesManager = AnalysesManager;
  exports.Analysis = Analysis;
  exports.JSGraph = JSGraph;
  exports.autoPeakPicking = autoPeakPicking;
  exports.calculateInterplanarSpacing = calculateInterplanarSpacing;
  exports.calculateReflexes = calculateReflexes;
  exports.computeCrystallinity = computeCrystallinity;
  exports.fromBRML = fromBRML;
  exports.fromJcamp = fromJcamp;
  exports.fromPowDLLXY = fromPowDLLXY;
  exports.generatePattern = generatePattern;
  exports.kAlpha1Angstrom = kAlpha1Angstrom;
  exports.peakPicking = peakPicking;
  exports.scherrerForSpectrum = scherrerForSpectrum;
  exports.toJcamp = toJcamp;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=xrd-analysis.js.map
