/**
 * isotherm-analysis - isotherm-analysis allows to parse and analyze isotherms. It converts from multiple formats to JCAMP-DX and provides utilities for basic analysis.
 * @version v1.2.0
 * @link https://github.com/cheminfo/isotherm-analysis#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.IsothermAnalysis = {}));
})(this, (function (exports) { 'use strict';

	var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

	/**
	 * Create an array with numbers between "from" and "to" of length "length"
	 *
	 * @param options - options
	 * @return - array of distributed numbers between "from" and "to"
	 */
	function createFromToArray() {
	  let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let {
	    from = 0,
	    to = 1,
	    length = 1000,
	    includeFrom = true,
	    includeTo = true,
	    distribution = 'uniform'
	  } = options;
	  const array = new Float64Array(length);
	  let div = length;

	  if (includeFrom && includeTo) {
	    div = length - 1;
	  } else if (!includeFrom && includeTo || includeFrom && !includeTo) {
	    div = length;
	  } else if (!includeFrom && !includeTo) {
	    div = length + 1;
	  }

	  let delta = (to - from) / div;

	  if (distribution === 'uniform') {
	    if (includeFrom) {
	      let index = 0;

	      while (index < length) {
	        array[index] = from + delta * index;
	        index++;
	      }
	    } else {
	      let index = 0;

	      while (index < length) {
	        array[index] = from + delta * (index + 1);
	        index++;
	      }
	    }
	  } else if (distribution === 'log') {
	    let base = (to / from) ** (1 / div);
	    let firstExponent = Math.log(from) / Math.log(base);

	    if (includeFrom) {
	      let index = 0;

	      while (index < length) {
	        array[index] = base ** (firstExponent + index);
	        index++;
	      }
	    } else {
	      let index = 0;

	      while (index < length) {
	        array[index] = base ** (firstExponent + index + 1);
	        index++;
	      }
	    }
	  } else {
	    throw new Error('Please choose for the distribution either uniform or log. By default the distribution chosen is uniform.');
	  }

	  return array;
	}

	const toString = Object.prototype.toString;
	/**
	 * Checks if an object is an instance of an Array (array or typed array).
	 *
	 * @param {any} value - Object to check.
	 * @returns {boolean} True if the object is an array.
	 */

	function isAnyArray(value) {
	  return toString.call(value).endsWith('Array]');
	}

	var medianQuickselect_min = {exports: {}};

	(function (module) {
	  (function () {
	    function a(d) {
	      for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
	        if (f <= e) return d[j];
	        if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

	        for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
	          do h++; while (d[e] > d[h]);

	          do i--; while (d[i] > d[e]);

	          if (i < h) break;
	          b(d, h, i);
	        }

	        b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
	      }
	    }

	    var b = function b(d, e, f) {
	      var _ref;

	      return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
	    },
	        c = function c(d, e) {
	      return ~~((d + e) / 2);
	    };

	    module.exports ? module.exports = a : window.median = a;
	  })();
	})(medianQuickselect_min);

	var quickSelectMedian = medianQuickselect_min.exports;

	/**
	 * Calculates the median of an array
	 *
	 * @param input - Array containing values
	 * @returns - median
	 */

	function xMedian(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  } // we need to slice because the order of elements is changed in the quickselect
	  // https://github.com/mad-gooze/median-quickselect


	  return quickSelectMedian(input.slice());
	}

	/**
	 * This function
	 * @param output - undefined or a new array
	 * @param length - length of the output array
	 * @returns
	 */

	function getOutputArray(output, length) {
	  if (output !== undefined) {
	    if (!isAnyArray(output)) {
	      throw new TypeError('output option must be an array if specified');
	    }

	    if (output.length !== length) {
	      throw new TypeError('the output array does not have the correct length');
	    }

	    return output;
	  } else {
	    return new Float64Array(length);
	  }
	}

	/**
	 * Checks if input is of type array
	 *
	 * @param input - input
	 */

	function xCheck$1(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	}

	/**
	 * This function divide the first array by the second array or a constant value to each element of the first array
	 *
	 * @param array1 - first array
	 * @param array2 - second array or number
	 * @param options - options
	 */

	function xDivide(array1, array2) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let isConstant = false;
	  let constant = 0;

	  if (isAnyArray(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xDivide: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = Number(array2);
	  }

	  let array3 = getOutputArray(options.output, array1.length);

	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] / constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] / array2[i];
	    }
	  }

	  return array3;
	}

	/**
	 * Returns a copy of the data as Float64
	 *
	 * @param array - array of numbers
	 */

	function xEnsureFloat64(array) {
	  xCheck$1(array);

	  if (array instanceof Float64Array) {
	    return array.slice(0);
	  }

	  return Float64Array.from(array);
	}

	/**
	 * Returns the closest index of a `target`
	 *
	 * @param array - array of numbers
	 * @param target - target
	 * @returns - closest index
	 */
	function xFindClosestIndex(array, target) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  const {
	    sorted = true
	  } = options;

	  if (sorted) {
	    let low = 0;
	    let high = array.length - 1;
	    let middle = 0;

	    while (high - low > 1) {
	      middle = low + (high - low >> 1);

	      if (array[middle] < target) {
	        low = middle;
	      } else if (array[middle] > target) {
	        high = middle;
	      } else {
	        return middle;
	      }
	    }

	    if (low < array.length - 1) {
	      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
	        return low;
	      } else {
	        return low + 1;
	      }
	    } else {
	      return low;
	    }
	  } else {
	    let index = 0;
	    let diff = Number.POSITIVE_INFINITY;

	    for (let i = 0; i < array.length; i++) {
	      const currentDiff = Math.abs(array[i] - target);

	      if (currentDiff < diff) {
	        diff = currentDiff;
	        index = i;
	      }
	    }

	    return index;
	  }
	}

	/**
	 * Returns an object with {fromIndex, toIndex} for a specific from / to
	 *
	 * @param x - array of numbers
	 * @param options - Options
	 */

	function xGetFromToIndex(x) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fromIndex,
	    toIndex,
	    from,
	    to
	  } = options;

	  if (fromIndex === undefined) {
	    if (from !== undefined) {
	      fromIndex = xFindClosestIndex(x, from);
	    } else {
	      fromIndex = 0;
	    }
	  }

	  if (toIndex === undefined) {
	    if (to !== undefined) {
	      toIndex = xFindClosestIndex(x, to);
	    } else {
	      toIndex = x.length - 1;
	    }
	  }

	  if (fromIndex < 0) fromIndex = 0;
	  if (toIndex < 0) toIndex = 0;
	  if (fromIndex >= x.length) fromIndex = x.length - 1;
	  if (toIndex >= x.length) toIndex = x.length - 1;
	  if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
	  return {
	    fromIndex,
	    toIndex
	  };
	}

	/**
	 * Computes the maximal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMaxValue(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck$1(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let maxValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] > maxValue) {
	      maxValue = array[i];
	    }
	  }

	  return maxValue;
	}

	/**
	 * Computes the minimal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMinValue(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck$1(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let minValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] < minValue) {
	      minValue = array[i];
	    }
	  }

	  return minValue;
	}

	/**
	 * Check if the values are separated always by the same difference
	 *
	 * @param array - Monotone growing array of number
	 */
	function xIsEquallySpaced(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (array.length < 3) return true;
	  const {
	    tolerance = 0.05
	  } = options;
	  let maxDx = 0;
	  let minDx = Number.MAX_SAFE_INTEGER;

	  for (let i = 0; i < array.length - 1; ++i) {
	    let absoluteDifference = array[i + 1] - array[i];

	    if (absoluteDifference < minDx) {
	      minDx = absoluteDifference;
	    }

	    if (absoluteDifference > maxDx) {
	      maxDx = absoluteDifference;
	    }
	  }

	  return (maxDx - minDx) / maxDx < tolerance;
	}

	/**
	 * Returns true if x is monotone
	 *
	 * @param array - array of numbers
	 */
	function xIsMonotone(array) {
	  if (array.length <= 2) {
	    return true;
	  }

	  if (array[0] === array[1]) {
	    // maybe a constant series
	    for (let i = 1; i < array.length - 1; i++) {
	      if (array[i] !== array[i + 1]) return false;
	    }

	    return true;
	  }

	  if (array[0] < array[array.length - 1]) {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] >= array[i + 1]) return false;
	    }
	  } else {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] <= array[i + 1]) return false;
	    }
	  }

	  return true;
	}

	/**
	 * Returns true if x is monotone
	 *
	 * @param array - array of numbers
	 */
	function xIsMonotoneIncreasing(array) {
	  if (array.length < 2) {
	    return true;
	  }

	  for (let i = 0; i < array.length - 1; i++) {
	    if (array[i] >= array[i + 1]) return false;
	  }

	  return true;
	}

	/**
	 * Computes the mean value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMean(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck$1(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let sumValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }

	  return sumValue / (toIndex - fromIndex + 1);
	}

	/**
	 * This function calculates the median absolute deviation (MAD)
	 * https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * @param array
	 */

	function xMedianAbsoluteDeviation(array) {
	  const median = xMedian(array);
	  const averageDeviations = new Float64Array(array.length);

	  for (let i = 0; i < array.length; i++) {
	    averageDeviations[i] = Math.abs(array[i] - median);
	  }

	  return {
	    median,
	    mad: xMedian(averageDeviations)
	  };
	}

	/**
	 * Return min and max values of an array
	 *
	 * @param array - array of number
	 * @returns - Object with 2 properties, min and max
	 */

	function xMinMaxValues(array) {
	  xCheck$1(array);
	  let min = array[0];
	  let max = array[0];

	  for (let value of array) {
	    if (value < min) min = value;
	    if (value > max) max = value;
	  }

	  return {
	    min,
	    max
	  };
	}

	/**
	 * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
	 * This assumes a gaussian distribution of the noise
	 * @param array
	 * @returns noise level corresponding to one standard deviation
	 */

	function xNoiseStandardDeviation(array) {
	  const {
	    mad,
	    median
	  } = xMedianAbsoluteDeviation(array);
	  return {
	    sd: mad / 0.6744897501960817,
	    mad,
	    median
	  };
	}

	/**
	 * Calculate the sum of the values
	 *
	 * @param array - Object that contains property x (an ordered increasing array) and y (an array).
	 * @param options - Options.
	 * @returns XSum value on the specified range.
	 */

	function xSum(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck$1(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let sumValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }

	  return sumValue;
	}

	/**
	 * Divides the data with either the sum, the absolute sum or the maximum of the data
	 * @param array - Array containing values
	 * @param options - options
	 * @returns - normalized data
	 */

	function xNormed(input) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    algorithm = 'absolute',
	    value = 1
	  } = options;
	  xCheck$1(input);
	  const output = getOutputArray(options.output, input.length);

	  if (input.length === 0) {
	    throw new Error('input must not be empty');
	  }

	  switch (algorithm.toLowerCase()) {
	    case 'absolute':
	      {
	        let absoluteSumValue = absoluteSum(input) / value;

	        if (absoluteSumValue === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }

	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] / absoluteSumValue;
	        }

	        return output;
	      }

	    case 'max':
	      {
	        let currentMaxValue = xMaxValue(input);

	        if (currentMaxValue === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }

	        const factor = value / currentMaxValue;

	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] * factor;
	        }

	        return output;
	      }

	    case 'sum':
	      {
	        let sumFactor = xSum(input) / value;

	        if (sumFactor === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }

	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] / sumFactor;
	        }

	        return output;
	      }

	    default:
	      throw new Error(`norm: unknown algorithm: ${algorithm}`);
	  }
	}

	function absoluteSum(input) {
	  let sumValue = 0;

	  for (let i = 0; i < input.length; i++) {
	    sumValue += Math.abs(input[i]);
	  }

	  return sumValue;
	}

	/** Finds the variance of the data
	 *
	 * @param values - the values of the array
	 * @param options - options
	 * @returns variance
	 */

	function xVariance(values) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(values)) {
	    throw new TypeError('input must be an array');
	  }

	  const {
	    unbiased = true,
	    mean = xMean(values)
	  } = options;
	  let sqrError = 0;

	  for (let i = 0; i < values.length; i++) {
	    let x = values[i] - mean;
	    sqrError += x * x;
	  }

	  if (unbiased) {
	    return sqrError / (values.length - 1);
	  } else {
	    return sqrError / values.length;
	  }
	}

	/** Finds the standard deviation for the data at hand
	 *
	 * @param values - values in the data
	 * @param options - options
	 * @returns standard deviation
	 */

	function xStandardDeviation(values) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return Math.sqrt(xVariance(values, options));
	}

	/** Function used to rescale data
	 *
	 * @param input - input for the rescale
	 * @param options - options
	 * @returns rescaled data
	 */

	function xRescale(input) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck$1(input);
	  const output = getOutputArray(options.output, input.length);
	  const currentMin = xMinValue(input);
	  const currentMax = xMaxValue(input);

	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }

	  const {
	    min = 0,
	    max = 1
	  } = options;

	  if (min >= max) {
	    throw new RangeError('min option must be smaller than max option');
	  }

	  const factor = (max - min) / (currentMax - currentMin);

	  for (let i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + min;
	  }

	  return output;
	}

	/**
	 * Throw an error in no an object of x,y arrays
	 *
	 * @param data - array of points {x,y,z}
	 */

	function xyCheck(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    minLength
	  } = options;

	  if (typeof data !== 'object' || !isAnyArray(data.x) || !isAnyArray(data.y)) {
	    throw new Error('Data must be an object of x and y arrays');
	  }

	  if (data.x.length !== data.y.length) {
	    throw new Error('The x and y arrays must have the same length');
	  }

	  if (minLength) {
	    if (data.x.length < minLength) {
	      throw new Error(`data.x must have a length of at least ${minLength}`);
	    }
	  }
	}

	/**
	 * Filters x,y values to allow strictly growing values in x axis.
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array).
	 */

	function xyEnsureGrowingX(data) {
	  xyCheck(data);
	  if (xIsMonotoneIncreasing(data.x)) return data;
	  const x = Array.from(data.x);
	  const y = Array.from(data.y);
	  let prevX = Number.NEGATIVE_INFINITY;
	  let currentIndex = 0;

	  for (let index = 0; index < x.length; index++) {
	    if (prevX < x[index]) {
	      if (currentIndex < index) {
	        x[currentIndex] = x[index];
	        y[currentIndex] = y[index];
	      }

	      currentIndex++;
	      prevX = x[index];
	    }
	  }

	  x.length = currentIndex;
	  y.length = currentIndex;
	  return {
	    x,
	    y
	  };
	}

	/**
	 * Normalize an array of zones:
	 * - ensure than from < to
	 * - merge overlapping zones
	 * - deal with exclusions zones
	 * - if no zones is specified add one between -Infinity and +Infinity
	 * @param zones - array of zones
	 * @param options - options
	 * @returns array of zones
	 */
	function zonesNormalize() {
	  let zones = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    from = Number.NEGATIVE_INFINITY,
	    to = Number.POSITIVE_INFINITY,
	    exclusions = []
	  } = options;
	  if (from > to) [from, to] = [to, from];
	  zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
	    from: zone.to,
	    to: zone.from
	  } : zone);
	  zones = zones.sort((a, b) => {
	    if (a.from !== b.from) return a.from - b.from;
	    return a.to - b.to;
	  });

	  if (zones.length === 0) {
	    zones.push({
	      from,
	      to
	    });
	  }

	  zones.forEach(zone => {
	    if (from > zone.from) zone.from = from;
	    if (to < zone.to) zone.to = to;
	  });
	  zones = zones.filter(zone => zone.from <= zone.to);
	  if (zones.length === 0) return [];
	  let currentZone = zones[0];
	  let beforeExclusionsZones = [currentZone];

	  for (let i = 1; i < zones.length; i++) {
	    let zone = zones[i];

	    if (zone.from <= currentZone.to) {
	      if (currentZone.to < zone.to) {
	        currentZone.to = zone.to;
	      }
	    } else {
	      currentZone = zone;
	      beforeExclusionsZones.push(currentZone);
	    }
	  }

	  if (exclusions.length === 0) return beforeExclusionsZones;
	  const normalizedExclusions = zonesNormalize(exclusions);
	  let currentExclusionIndex = 0;
	  const results = [];
	  let counter = 0;

	  for (let zoneIndex = 0; zoneIndex < beforeExclusionsZones.length; zoneIndex++) {
	    if (counter++ > 5) break;
	    const zone = beforeExclusionsZones[zoneIndex];

	    if (currentExclusionIndex === normalizedExclusions.length) {
	      // we analysed all the exclusion zones
	      results.push(zone);
	      continue;
	    }

	    while (currentExclusionIndex < normalizedExclusions.length && normalizedExclusions[currentExclusionIndex].to <= zone.from) {
	      currentExclusionIndex++;
	    }

	    if (currentExclusionIndex === normalizedExclusions.length) {
	      // we analysed all the exclusion zones
	      results.push(zone);
	      continue;
	    }

	    if (zone.to < normalizedExclusions[currentExclusionIndex].from) {
	      // no problems, not yet in exclusion
	      results.push(zone);
	      continue;
	    }

	    if (normalizedExclusions[currentExclusionIndex].to >= zone.to) {
	      // could be totally excluded
	      if (normalizedExclusions[currentExclusionIndex].from <= zone.from) {
	        continue;
	      }

	      results.push({
	        from: normalizedExclusions[currentExclusionIndex].to,
	        to: zone.to
	      });
	    } // we cut in the middle, we need to create more zones, annoying !


	    if (normalizedExclusions[currentExclusionIndex].from > zone.from) {
	      results.push({
	        from: zone.from,
	        to: normalizedExclusions[currentExclusionIndex].from
	      });
	    }

	    zone.from = normalizedExclusions[currentExclusionIndex].to;
	    zoneIndex--;
	  }

	  return results;
	}

	/**
	 * Add the number of points per zone to reach a specified total
	 *
	 * @param zones - array of zones
	 * @param numberOfPoints - total number of points to distribute between zones
	 * @param options - options
	 * @returns array of zones with points
	 */

	function zonesWithPoints() {
	  let zones = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let numberOfPoints = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 10;
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  if (zones.length === 0) return zones;
	  let returnZones = zonesNormalize(zones, options);
	  const totalSize = returnZones.reduce((previous, current) => {
	    return previous + (current.to - current.from);
	  }, 0);
	  let unitsPerPoint = totalSize / numberOfPoints;
	  let currentTotal = 0;

	  for (let i = 0; i < returnZones.length - 1; i++) {
	    let zone = returnZones[i];
	    zone.numberOfPoints = Math.min(Math.round((zone.to - zone.from) / unitsPerPoint), numberOfPoints - currentTotal);
	    currentTotal += zone.numberOfPoints;
	  }

	  let zone = returnZones[returnZones.length - 1];
	  zone.numberOfPoints = numberOfPoints - currentTotal;
	  return returnZones;
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "slot"
	 *
	 * @param x
	 * @param y
	 * @param from
	 * @param to
	 * @param numberOfPoints
	 * @return Array of y's equally spaced with the variant "slot"
	 */
	function equallySpacedSlot(
	/** x coordinates */
	x,
	/** y coordinates */
	y,
	/** from value */
	from,
	/** to value */
	to,
	/** number of points */
	numberOfPoints) {
	  let xLength = x.length;
	  let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  let halfStep = step / 2;
	  let lastStep = x[x.length - 1] - x[x.length - 2];
	  let start = from - halfStep; // Changed Array to Float64Array

	  let output = new Float64Array(numberOfPoints); // Init main variables

	  let min = start;
	  let max = start + step;
	  let previousX = -Number.MAX_VALUE;
	  let previousY = 0;
	  let nextX = x[0];
	  let nextY = y[0];
	  let frontOutsideSpectra = 0;
	  let backOutsideSpectra = true;
	  let currentValue = 0; // for slot algorithm

	  let currentPoints = 0;
	  let i = 1; // index of input

	  let j = 0; // index of output

	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');

	    while (previousX - max > 0) {
	      // no overlap with original point, just consume current value
	      if (backOutsideSpectra) {
	        currentPoints++;
	        backOutsideSpectra = false;
	      }

	      output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
	      j++;

	      if (j === numberOfPoints) {
	        break main;
	      }

	      min = max;
	      max += step;
	      currentValue = 0;
	      currentPoints = 0;
	    }

	    if (previousX > min) {
	      currentValue += previousY;
	      currentPoints++;
	    }

	    if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
	      currentPoints--;
	    }

	    previousX = nextX;
	    previousY = nextY;

	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else {
	      nextX += lastStep;
	      nextY = 0;
	      frontOutsideSpectra++;
	    }
	  }

	  return output;
	}

	/**
	 * Function that calculates the integral of the line between two
	 * x-coordinates, given the slope and intercept of the line.
	 * @param x0
	 * @param x1
	 * @param slope
	 * @param intercept
	 * @return integral value.
	 */
	function integral(
	/** first coordinate of point */
	x0,
	/** second coordinate of point */
	x1,
	/** slope of the line */
	slope,
	/** intercept of the line on the y axis */
	intercept) {
	  return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "smooth"
	 *
	 * @param x
	 * @param y
	 * @param from
	 * @param to
	 * @param numberOfPoints
	 * @return - Array of y's equally spaced with the variant "smooth"
	 */

	function equallySpacedSmooth(
	/** x coordinates */
	x,
	/** y coordinates */
	y,
	/** from value */
	from,
	/** to value */
	to,
	/** number of points */
	numberOfPoints) {
	  let xLength = x.length;
	  let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  let halfStep = step / 2; // Changed Array to Float64Array

	  let output = new Float64Array(numberOfPoints);
	  let initialOriginalStep = x[1] - x[0];
	  let lastOriginalStep = x[xLength - 1] - x[xLength - 2]; // Init main variables

	  let min = from - halfStep;
	  let max = from + halfStep;
	  let previousX = Number.MIN_SAFE_INTEGER;
	  let previousY = 0;
	  let nextX = x[0] - initialOriginalStep;
	  let nextY = 0;
	  let currentValue = 0;
	  let slope = 0;
	  let intercept = 0;
	  let sumAtMin = 0;
	  let sumAtMax = 0;
	  let i = 0; // index of input

	  let j = 0; // index of output

	  function getSlope(x0, y0, x1, y1) {
	    return (y1 - y0) / (x1 - x0);
	  }

	  let add = 0;

	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');

	    if (previousX <= min && min <= nextX) {
	      add = integral(0, min - previousX, slope, previousY);
	      sumAtMin = currentValue + add;
	    }

	    while (nextX - max >= 0) {
	      // no overlap with original point, just consume current value
	      add = integral(0, max - previousX, slope, previousY);
	      sumAtMax = currentValue + add;
	      output[j++] = (sumAtMax - sumAtMin) / step;

	      if (j === numberOfPoints) {
	        break main;
	      }

	      min = max;
	      max += step;
	      sumAtMin = sumAtMax;
	    }

	    currentValue += integral(previousX, nextX, slope, intercept);
	    previousX = nextX;
	    previousY = nextY;

	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else if (i === xLength) {
	      nextX += lastOriginalStep;
	      nextY = 0;
	    }

	    slope = getSlope(previousX, previousY, nextX, nextY);
	    intercept = -slope * previousX + previousY;
	  }

	  return output;
	}

	/**
	 * Function that returns a Number array of equally spaced numberOfPoints
	 * containing a representation of intensities of the spectra arguments x
	 * and y.
	 *
	 * The options parameter contains an object in the following form:
	 * from: starting point
	 * to: last point
	 * numberOfPoints: number of points between from and to
	 * variant: "slot" or "smooth" - smooth is the default option
	 *
	 * The slot variant consist that each point in an array is calculated
	 * averaging the existing points between the slot that belongs to the current
	 * value. The smooth variant is the same but takes the integral of the range
	 * of the slot and divide by the step size between two points in an array.
	 *
	 * If exclusions zone are present, zones are ignored !
	 *
	 * @param data - object containing 2 properties x and y
	 * @param options - options
	 * @return new object with x / y array with the equally spaced data.
	 */

	function xyEquallySpaced(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    x,
	    y
	  } = data;
	  let xLength = x.length;
	  const {
	    from = x[0],
	    to = x[xLength - 1],
	    variant = 'smooth',
	    numberOfPoints = 100,
	    exclusions = [],
	    zones = [{
	      from,
	      to
	    }]
	  } = options;

	  if (from > to) {
	    throw new RangeError('from should be larger than to');
	  }

	  xyCheck(data);

	  if (numberOfPoints < 2) {
	    throw new RangeError("'numberOfPoints' option must be greater than 1");
	  }

	  const normalizedZones = zonesNormalize(zones, {
	    from,
	    to,
	    exclusions
	  });
	  const zonesWithPointsRes = zonesWithPoints(normalizedZones, numberOfPoints, {
	    from,
	    to
	  });
	  let xResult = [];
	  let yResult = [];

	  for (let zone of zonesWithPointsRes) {
	    if (!zone.numberOfPoints) {
	      zone.numberOfPoints = 0;
	    }

	    let zoneResult = processZone(Array.from(x), Array.from(y), zone.from, zone.to, zone.numberOfPoints, variant);
	    xResult = xResult.concat(zoneResult.x);
	    yResult = yResult.concat(zoneResult.y);
	  }

	  return {
	    x: xResult,
	    y: yResult
	  };
	}

	function processZone(x, y, from, to, numberOfPoints, variant) {
	  if (numberOfPoints < 1) {
	    throw new RangeError('the number of points must be at least 1');
	  }

	  let output = variant === 'slot' ? Array.from(equallySpacedSlot(x, y, from, to, numberOfPoints)) : Array.from(equallySpacedSmooth(x, y, from, to, numberOfPoints));
	  return {
	    x: Array.from(createFromToArray({
	      from,
	      to,
	      length: numberOfPoints
	    })),
	    y: output
	  };
	}

	/** Filter an array x/y based on various criteria x points are expected to be sorted
	 *
	 * @param data - points
	 * @param options - options
	 * @return filtered array
	 */

	function xyFilterX(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = data;
	  const {
	    from = x[0],
	    to = x[x.length - 1],
	    zones = [{
	      from,
	      to
	    }],
	    exclusions = []
	  } = options;
	  let normalizedZones = zonesNormalize(zones, {
	    from,
	    to,
	    exclusions
	  });
	  let currentZoneIndex = 0;
	  let newX = [];
	  let newY = [];
	  let position = 0;

	  while (position < x.length) {
	    if (x[position] <= normalizedZones[currentZoneIndex].to && x[position] >= normalizedZones[currentZoneIndex].from) {
	      newX.push(x[position]);
	      newY.push(y[position]);
	    } else if (x[position] > normalizedZones[currentZoneIndex].to) {
	      currentZoneIndex++;
	      if (!normalizedZones[currentZoneIndex]) break;
	    }

	    position++;
	  }

	  return {
	    x: newX,
	    y: newY
	  };
	}

	/**
	 * Filter out all the points for which x <= 0. Useful to display log scale data
	 *
	 * @param data - data
	 * @returns - An object with the filtered data
	 */

	function xyFilterXPositive(data) {
	  xyCheck(data);
	  const {
	    x,
	    y
	  } = data;
	  const newX = [];
	  const newY = [];
	  if (x === undefined || y === undefined) return {
	    x: newX,
	    y: newY
	  };

	  for (let i = 0; i < x.length; i++) {
	    if (x[i] > 0) {
	      newX.push(x[i]);
	      newY.push(y[i]);
	    }
	  }

	  return {
	    x: newX,
	    y: newY
	  };
	}

	/**
	 * Find the closest maximum going up hill
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
	 * @param options - options
	 * @returns - An object with the x/y value
	 */

	function xyMaxClosestYPoint(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xyCheck(data);
	  const {
	    x,
	    y
	  } = data;
	  let {
	    target,
	    targetIndex
	  } = options;

	  if (targetIndex === undefined) {
	    if (target !== undefined) {
	      targetIndex = xFindClosestIndex(x, target);
	    } else {
	      targetIndex = 0;
	    }
	  }

	  let previousIndex = Number.MIN_SAFE_INTEGER;
	  let currentIndex = targetIndex;
	  let xyMaxY = y[targetIndex];

	  while (currentIndex !== previousIndex) {
	    previousIndex = currentIndex;

	    if (currentIndex > 0 && y[currentIndex - 1] > xyMaxY) {
	      currentIndex--;
	    } else if (currentIndex < x.length - 1 && y[currentIndex + 1] > xyMaxY) {
	      currentIndex++;
	    }

	    xyMaxY = y[currentIndex];
	  }

	  return {
	    x: x[currentIndex],
	    y: y[currentIndex],
	    index: currentIndex
	  };
	}

	/**
	 * Find the closest minimum going down hill
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
	 * @param options - Options
	 * @returns - An object with the x/y value
	 */

	function xyMinClosestYPoint(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xyCheck(data);
	  const {
	    x,
	    y
	  } = data;
	  let {
	    target,
	    targetIndex
	  } = options;

	  if (targetIndex === undefined) {
	    if (target !== undefined) {
	      targetIndex = xFindClosestIndex(x, target);
	    } else {
	      targetIndex = 0;
	    }
	  }

	  let previousIndex = Number.MIN_SAFE_INTEGER;
	  let currentIndex = targetIndex;
	  let minY = y[targetIndex];

	  while (currentIndex !== previousIndex) {
	    previousIndex = currentIndex;

	    if (currentIndex > 0 && y[currentIndex - 1] < minY) {
	      currentIndex--;
	    } else if (currentIndex < x.length - 1 && y[currentIndex + 1] < minY) {
	      currentIndex++;
	    }

	    minY = y[currentIndex];
	  }

	  return {
	    x: x[currentIndex],
	    y: y[currentIndex],
	    index: currentIndex
	  };
	}

	function matrixCheck(data) {
	  if (data.length === 0 || data[0].length === 0) {
	    throw new RangeError('matrix should contain data');
	  }

	  const firstLength = data[0].length;

	  for (let i = 1; i < data.length; i++) {
	    if (data[i].length !== firstLength) {
	      throw new RangeError('All rows should has the same length');
	    }
	  }
	}

	/**
	 * Get min and max Z
	 *
	 * @param matrix - matrix [rows][cols].
	 */

	function matrixMinMaxZ(matrix) {
	  matrixCheck(matrix);
	  const nbRows = matrix.length;
	  const nbColumns = matrix[0].length;
	  let min = matrix[0][0];
	  let max = matrix[0][0];

	  for (let column = 0; column < nbColumns; column++) {
	    for (let row = 0; row < nbRows; row++) {
	      if (matrix[row][column] < min) min = matrix[row][column];
	      if (matrix[row][column] > max) max = matrix[row][column];
	    }
	  }

	  return {
	    min,
	    max
	  };
	}

	function max(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var _options$fromIndex = options.fromIndex,
	      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	      _options$toIndex = options.toIndex,
	      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }

	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }

	  var maxValue = input[fromIndex];

	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] > maxValue) maxValue = input[i];
	  }

	  return maxValue;
	}

	function min(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var _options$fromIndex = options.fromIndex,
	      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	      _options$toIndex = options.toIndex,
	      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }

	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }

	  var minValue = input[fromIndex];

	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] < minValue) minValue = input[i];
	  }

	  return minValue;
	}

	function rescale$1(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  } else if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var output;

	  if (options.output !== undefined) {
	    if (!isAnyArray(options.output)) {
	      throw new TypeError('output option must be an array if specified');
	    }

	    output = options.output;
	  } else {
	    output = new Array(input.length);
	  }

	  var currentMin = min(input);
	  var currentMax = max(input);

	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }

	  var _options$min = options.min,
	      minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
	      _options$max = options.max,
	      maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

	  if (minValue >= maxValue) {
	    throw new RangeError('min option must be smaller than max option');
	  }

	  var factor = (maxValue - minValue) / (currentMax - currentMin);

	  for (var i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + minValue;
	  }

	  return output;
	}

	const indent = ' '.repeat(2);
	const indentData = ' '.repeat(4);
	function inspectMatrix() {
	  return inspectMatrixWithOptions(this);
	}
	function inspectMatrixWithOptions(matrix) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    maxRows = 15,
	    maxColumns = 10,
	    maxNumSize = 8
	  } = options;
	  return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
	}

	function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
	  const {
	    rows,
	    columns
	  } = matrix;
	  const maxI = Math.min(rows, maxRows);
	  const maxJ = Math.min(columns, maxColumns);
	  const result = [];

	  for (let i = 0; i < maxI; i++) {
	    let line = [];

	    for (let j = 0; j < maxJ; j++) {
	      line.push(formatNumber(matrix.get(i, j), maxNumSize));
	    }

	    result.push(`${line.join(' ')}`);
	  }

	  if (maxJ !== columns) {
	    result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
	  }

	  if (maxI !== rows) {
	    result.push(`... ${rows - maxRows} more rows`);
	  }

	  return result.join(`\n${indentData}`);
	}

	function formatNumber(num, maxNumSize) {
	  const numStr = String(num);

	  if (numStr.length <= maxNumSize) {
	    return numStr.padEnd(maxNumSize, ' ');
	  }

	  const precise = num.toPrecision(maxNumSize - 2);

	  if (precise.length <= maxNumSize) {
	    return precise;
	  }

	  const exponential = num.toExponential(maxNumSize - 2);
	  const eIndex = exponential.indexOf('e');
	  const e = exponential.slice(eIndex);
	  return exponential.slice(0, maxNumSize - e.length) + e;
	}

	function installMathOperations(AbstractMatrix, Matrix) {
	  AbstractMatrix.prototype.add = function add(value) {
	    if (typeof value === 'number') return this.addS(value);
	    return this.addM(value);
	  };

	  AbstractMatrix.prototype.addS = function addS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.addM = function addM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.add = function add(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.add(value);
	  };

	  AbstractMatrix.prototype.sub = function sub(value) {
	    if (typeof value === 'number') return this.subS(value);
	    return this.subM(value);
	  };

	  AbstractMatrix.prototype.subS = function subS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.subM = function subM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sub = function sub(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sub(value);
	  };

	  AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
	  AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
	  AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
	  AbstractMatrix.subtract = AbstractMatrix.sub;

	  AbstractMatrix.prototype.mul = function mul(value) {
	    if (typeof value === 'number') return this.mulS(value);
	    return this.mulM(value);
	  };

	  AbstractMatrix.prototype.mulS = function mulS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.mulM = function mulM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.mul = function mul(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mul(value);
	  };

	  AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
	  AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
	  AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
	  AbstractMatrix.multiply = AbstractMatrix.mul;

	  AbstractMatrix.prototype.div = function div(value) {
	    if (typeof value === 'number') return this.divS(value);
	    return this.divM(value);
	  };

	  AbstractMatrix.prototype.divS = function divS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.divM = function divM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.div = function div(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.div(value);
	  };

	  AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
	  AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
	  AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
	  AbstractMatrix.divide = AbstractMatrix.div;

	  AbstractMatrix.prototype.mod = function mod(value) {
	    if (typeof value === 'number') return this.modS(value);
	    return this.modM(value);
	  };

	  AbstractMatrix.prototype.modS = function modS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.modM = function modM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.mod = function mod(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mod(value);
	  };

	  AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
	  AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
	  AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
	  AbstractMatrix.modulus = AbstractMatrix.mod;

	  AbstractMatrix.prototype.and = function and(value) {
	    if (typeof value === 'number') return this.andS(value);
	    return this.andM(value);
	  };

	  AbstractMatrix.prototype.andS = function andS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.andM = function andM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.and = function and(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.and(value);
	  };

	  AbstractMatrix.prototype.or = function or(value) {
	    if (typeof value === 'number') return this.orS(value);
	    return this.orM(value);
	  };

	  AbstractMatrix.prototype.orS = function orS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.orM = function orM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.or = function or(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.or(value);
	  };

	  AbstractMatrix.prototype.xor = function xor(value) {
	    if (typeof value === 'number') return this.xorS(value);
	    return this.xorM(value);
	  };

	  AbstractMatrix.prototype.xorS = function xorS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.xorM = function xorM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.xor = function xor(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.xor(value);
	  };

	  AbstractMatrix.prototype.leftShift = function leftShift(value) {
	    if (typeof value === 'number') return this.leftShiftS(value);
	    return this.leftShiftM(value);
	  };

	  AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.leftShift = function leftShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.leftShift(value);
	  };

	  AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
	    if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
	    return this.signPropagatingRightShiftM(value);
	  };

	  AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.signPropagatingRightShift(value);
	  };

	  AbstractMatrix.prototype.rightShift = function rightShift(value) {
	    if (typeof value === 'number') return this.rightShiftS(value);
	    return this.rightShiftM(value);
	  };

	  AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.rightShift = function rightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.rightShift(value);
	  };

	  AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
	  AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
	  AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
	  AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

	  AbstractMatrix.prototype.not = function not() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, ~this.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.not = function not(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.not();
	  };

	  AbstractMatrix.prototype.abs = function abs() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.abs(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.abs = function abs(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.abs();
	  };

	  AbstractMatrix.prototype.acos = function acos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acos(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.acos = function acos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acos();
	  };

	  AbstractMatrix.prototype.acosh = function acosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acosh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.acosh = function acosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acosh();
	  };

	  AbstractMatrix.prototype.asin = function asin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asin(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.asin = function asin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asin();
	  };

	  AbstractMatrix.prototype.asinh = function asinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asinh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.asinh = function asinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asinh();
	  };

	  AbstractMatrix.prototype.atan = function atan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atan(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.atan = function atan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atan();
	  };

	  AbstractMatrix.prototype.atanh = function atanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atanh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.atanh = function atanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atanh();
	  };

	  AbstractMatrix.prototype.cbrt = function cbrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cbrt(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cbrt = function cbrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cbrt();
	  };

	  AbstractMatrix.prototype.ceil = function ceil() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.ceil(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.ceil = function ceil(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.ceil();
	  };

	  AbstractMatrix.prototype.clz32 = function clz32() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.clz32(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.clz32 = function clz32(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.clz32();
	  };

	  AbstractMatrix.prototype.cos = function cos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cos(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cos = function cos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cos();
	  };

	  AbstractMatrix.prototype.cosh = function cosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cosh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cosh = function cosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cosh();
	  };

	  AbstractMatrix.prototype.exp = function exp() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.exp(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.exp = function exp(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.exp();
	  };

	  AbstractMatrix.prototype.expm1 = function expm1() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.expm1(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.expm1 = function expm1(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.expm1();
	  };

	  AbstractMatrix.prototype.floor = function floor() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.floor(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.floor = function floor(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.floor();
	  };

	  AbstractMatrix.prototype.fround = function fround() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.fround(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.fround = function fround(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.fround();
	  };

	  AbstractMatrix.prototype.log = function log() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log = function log(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log();
	  };

	  AbstractMatrix.prototype.log1p = function log1p() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log1p(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log1p = function log1p(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log1p();
	  };

	  AbstractMatrix.prototype.log10 = function log10() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log10(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log10 = function log10(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log10();
	  };

	  AbstractMatrix.prototype.log2 = function log2() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log2(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log2 = function log2(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log2();
	  };

	  AbstractMatrix.prototype.round = function round() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.round(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.round = function round(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.round();
	  };

	  AbstractMatrix.prototype.sign = function sign() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sign(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sign = function sign(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sign();
	  };

	  AbstractMatrix.prototype.sin = function sin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sin(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sin = function sin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sin();
	  };

	  AbstractMatrix.prototype.sinh = function sinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sinh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sinh = function sinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sinh();
	  };

	  AbstractMatrix.prototype.sqrt = function sqrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sqrt(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sqrt = function sqrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sqrt();
	  };

	  AbstractMatrix.prototype.tan = function tan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tan(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.tan = function tan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tan();
	  };

	  AbstractMatrix.prototype.tanh = function tanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tanh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.tanh = function tanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tanh();
	  };

	  AbstractMatrix.prototype.trunc = function trunc() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.trunc(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.trunc = function trunc(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.trunc();
	  };

	  AbstractMatrix.pow = function pow(matrix, arg0) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.pow(arg0);
	  };

	  AbstractMatrix.prototype.pow = function pow(value) {
	    if (typeof value === 'number') return this.powS(value);
	    return this.powM(value);
	  };

	  AbstractMatrix.prototype.powS = function powS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), value));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.powM = function powM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
	      }
	    }

	    return this;
	  };
	}

	/**
	 * @private
	 * Check that a row index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */

	function checkRowIndex(matrix, index, outer) {
	  let max = outer ? matrix.rows : matrix.rows - 1;

	  if (index < 0 || index > max) {
	    throw new RangeError('Row index out of range');
	  }
	}
	/**
	 * @private
	 * Check that a column index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */

	function checkColumnIndex(matrix, index, outer) {
	  let max = outer ? matrix.columns : matrix.columns - 1;

	  if (index < 0 || index > max) {
	    throw new RangeError('Column index out of range');
	  }
	}
	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */

	function checkRowVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }

	  if (vector.length !== matrix.columns) {
	    throw new RangeError('vector size must be the same as the number of columns');
	  }

	  return vector;
	}
	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */

	function checkColumnVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }

	  if (vector.length !== matrix.rows) {
	    throw new RangeError('vector size must be the same as the number of rows');
	  }

	  return vector;
	}
	function checkRowIndices(matrix, rowIndices) {
	  if (!isAnyArray(rowIndices)) {
	    throw new TypeError('row indices must be an array');
	  }

	  for (let i = 0; i < rowIndices.length; i++) {
	    if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
	      throw new RangeError('row indices are out of range');
	    }
	  }
	}
	function checkColumnIndices(matrix, columnIndices) {
	  if (!isAnyArray(columnIndices)) {
	    throw new TypeError('column indices must be an array');
	  }

	  for (let i = 0; i < columnIndices.length; i++) {
	    if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
	      throw new RangeError('column indices are out of range');
	    }
	  }
	}
	function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
	  if (arguments.length !== 5) {
	    throw new RangeError('expected 4 arguments');
	  }

	  checkNumber('startRow', startRow);
	  checkNumber('endRow', endRow);
	  checkNumber('startColumn', startColumn);
	  checkNumber('endColumn', endColumn);

	  if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
	    throw new RangeError('Submatrix indices are out of range');
	  }
	}
	function newArray(length) {
	  let value = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
	  let array = [];

	  for (let i = 0; i < length; i++) {
	    array.push(value);
	  }

	  return array;
	}

	function checkNumber(name, value) {
	  if (typeof value !== 'number') {
	    throw new TypeError(`${name} must be a number`);
	  }
	}

	function checkNonEmpty(matrix) {
	  if (matrix.isEmpty()) {
	    throw new Error('Empty matrix has no elements to index');
	  }
	}

	function sumByRow(matrix) {
	  let sum = newArray(matrix.rows);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] += matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function sumByColumn(matrix) {
	  let sum = newArray(matrix.columns);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] += matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function sumAll(matrix) {
	  let v = 0;

	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v += matrix.get(i, j);
	    }
	  }

	  return v;
	}
	function productByRow(matrix) {
	  let sum = newArray(matrix.rows, 1);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] *= matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function productByColumn(matrix) {
	  let sum = newArray(matrix.columns, 1);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] *= matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function productAll(matrix) {
	  let v = 1;

	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v *= matrix.get(i, j);
	    }
	  }

	  return v;
	}
	function varianceByRow(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];

	  for (let i = 0; i < rows; i++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;

	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean[i];
	      sum1 += x;
	      sum2 += x * x;
	    }

	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / cols) / cols);
	    }
	  }

	  return variance;
	}
	function varianceByColumn(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];

	  for (let j = 0; j < cols; j++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;

	    for (let i = 0; i < rows; i++) {
	      x = matrix.get(i, j) - mean[j];
	      sum1 += x;
	      sum2 += x * x;
	    }

	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / rows) / rows);
	    }
	  }

	  return variance;
	}
	function varianceAll(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const size = rows * cols;
	  let sum1 = 0;
	  let sum2 = 0;
	  let x = 0;

	  for (let i = 0; i < rows; i++) {
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean;
	      sum1 += x;
	      sum2 += x * x;
	    }
	  }

	  if (unbiased) {
	    return (sum2 - sum1 * sum1 / size) / (size - 1);
	  } else {
	    return (sum2 - sum1 * sum1 / size) / size;
	  }
	}
	function centerByRow(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[i]);
	    }
	  }
	}
	function centerByColumn(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[j]);
	    }
	  }
	}
	function centerAll(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean);
	    }
	  }
	}
	function getScaleByRow(matrix) {
	  const scale = [];

	  for (let i = 0; i < matrix.rows; i++) {
	    let sum = 0;

	    for (let j = 0; j < matrix.columns; j++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
	    }

	    scale.push(Math.sqrt(sum));
	  }

	  return scale;
	}
	function scaleByRow(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[i]);
	    }
	  }
	}
	function getScaleByColumn(matrix) {
	  const scale = [];

	  for (let j = 0; j < matrix.columns; j++) {
	    let sum = 0;

	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
	    }

	    scale.push(Math.sqrt(sum));
	  }

	  return scale;
	}
	function scaleByColumn(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[j]);
	    }
	  }
	}
	function getScaleAll(matrix) {
	  const divider = matrix.size - 1;
	  let sum = 0;

	  for (let j = 0; j < matrix.columns; j++) {
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / divider;
	    }
	  }

	  return Math.sqrt(sum);
	}
	function scaleAll(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale);
	    }
	  }
	}

	class AbstractMatrix {
	  static from1DArray(newRows, newColumns, newData) {
	    let length = newRows * newColumns;

	    if (length !== newData.length) {
	      throw new RangeError('data length does not match given dimensions');
	    }

	    let newMatrix = new Matrix(newRows, newColumns);

	    for (let row = 0; row < newRows; row++) {
	      for (let column = 0; column < newColumns; column++) {
	        newMatrix.set(row, column, newData[row * newColumns + column]);
	      }
	    }

	    return newMatrix;
	  }

	  static rowVector(newData) {
	    let vector = new Matrix(1, newData.length);

	    for (let i = 0; i < newData.length; i++) {
	      vector.set(0, i, newData[i]);
	    }

	    return vector;
	  }

	  static columnVector(newData) {
	    let vector = new Matrix(newData.length, 1);

	    for (let i = 0; i < newData.length; i++) {
	      vector.set(i, 0, newData[i]);
	    }

	    return vector;
	  }

	  static zeros(rows, columns) {
	    return new Matrix(rows, columns);
	  }

	  static ones(rows, columns) {
	    return new Matrix(rows, columns).fill(1);
	  }

	  static rand(rows, columns) {
	    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      random = Math.random
	    } = options;
	    let matrix = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.set(i, j, random());
	      }
	    }

	    return matrix;
	  }

	  static randInt(rows, columns) {
	    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1000,
	      random = Math.random
	    } = options;
	    if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
	    if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let interval = max - min;
	    let matrix = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        let value = min + Math.round(random() * interval);
	        matrix.set(i, j, value);
	      }
	    }

	    return matrix;
	  }

	  static eye(rows, columns, value) {
	    if (columns === undefined) columns = rows;
	    if (value === undefined) value = 1;
	    let min = Math.min(rows, columns);
	    let matrix = this.zeros(rows, columns);

	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, value);
	    }

	    return matrix;
	  }

	  static diag(data, rows, columns) {
	    let l = data.length;
	    if (rows === undefined) rows = l;
	    if (columns === undefined) columns = rows;
	    let min = Math.min(l, rows, columns);
	    let matrix = this.zeros(rows, columns);

	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, data[i]);
	    }

	    return matrix;
	  }

	  static min(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }

	    return result;
	  }

	  static max(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new this(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }

	    return result;
	  }

	  static checkMatrix(value) {
	    return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
	  }

	  static isMatrix(value) {
	    return value != null && value.klass === 'Matrix';
	  }

	  get size() {
	    return this.rows * this.columns;
	  }

	  apply(callback) {
	    if (typeof callback !== 'function') {
	      throw new TypeError('callback must be a function');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        callback.call(this, i, j);
	      }
	    }

	    return this;
	  }

	  to1DArray() {
	    let array = [];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        array.push(this.get(i, j));
	      }
	    }

	    return array;
	  }

	  to2DArray() {
	    let copy = [];

	    for (let i = 0; i < this.rows; i++) {
	      copy.push([]);

	      for (let j = 0; j < this.columns; j++) {
	        copy[i].push(this.get(i, j));
	      }
	    }

	    return copy;
	  }

	  toJSON() {
	    return this.to2DArray();
	  }

	  isRowVector() {
	    return this.rows === 1;
	  }

	  isColumnVector() {
	    return this.columns === 1;
	  }

	  isVector() {
	    return this.rows === 1 || this.columns === 1;
	  }

	  isSquare() {
	    return this.rows === this.columns;
	  }

	  isEmpty() {
	    return this.rows === 0 || this.columns === 0;
	  }

	  isSymmetric() {
	    if (this.isSquare()) {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j <= i; j++) {
	          if (this.get(i, j) !== this.get(j, i)) {
	            return false;
	          }
	        }
	      }

	      return true;
	    }

	    return false;
	  }

	  isEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isEchelonForm = true;
	    let checked = false;

	    while (i < this.rows && isEchelonForm) {
	      j = 0;
	      checked = false;

	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isEchelonForm = false;
	          checked = true;
	        }
	      }

	      i++;
	    }

	    return isEchelonForm;
	  }

	  isReducedEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isReducedEchelonForm = true;
	    let checked = false;

	    while (i < this.rows && isReducedEchelonForm) {
	      j = 0;
	      checked = false;

	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isReducedEchelonForm = false;
	          checked = true;
	        }
	      }

	      for (let k = j + 1; k < this.rows; k++) {
	        if (this.get(i, k) !== 0) {
	          isReducedEchelonForm = false;
	        }
	      }

	      i++;
	    }

	    return isReducedEchelonForm;
	  }

	  echelonForm() {
	    let result = this.clone();
	    let h = 0;
	    let k = 0;

	    while (h < result.rows && k < result.columns) {
	      let iMax = h;

	      for (let i = h; i < result.rows; i++) {
	        if (result.get(i, k) > result.get(iMax, k)) {
	          iMax = i;
	        }
	      }

	      if (result.get(iMax, k) === 0) {
	        k++;
	      } else {
	        result.swapRows(h, iMax);
	        let tmp = result.get(h, k);

	        for (let j = k; j < result.columns; j++) {
	          result.set(h, j, result.get(h, j) / tmp);
	        }

	        for (let i = h + 1; i < result.rows; i++) {
	          let factor = result.get(i, k) / result.get(h, k);
	          result.set(i, k, 0);

	          for (let j = k + 1; j < result.columns; j++) {
	            result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
	          }
	        }

	        h++;
	        k++;
	      }
	    }

	    return result;
	  }

	  reducedEchelonForm() {
	    let result = this.echelonForm();
	    let m = result.columns;
	    let n = result.rows;
	    let h = n - 1;

	    while (h >= 0) {
	      if (result.maxRow(h) === 0) {
	        h--;
	      } else {
	        let p = 0;
	        let pivot = false;

	        while (p < n && pivot === false) {
	          if (result.get(h, p) === 1) {
	            pivot = true;
	          } else {
	            p++;
	          }
	        }

	        for (let i = 0; i < h; i++) {
	          let factor = result.get(i, p);

	          for (let j = p; j < m; j++) {
	            let tmp = result.get(i, j) - factor * result.get(h, j);
	            result.set(i, j, tmp);
	          }
	        }

	        h--;
	      }
	    }

	    return result;
	  }

	  set() {
	    throw new Error('set method is unimplemented');
	  }

	  get() {
	    throw new Error('get method is unimplemented');
	  }

	  repeat() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      rows = 1,
	      columns = 1
	    } = options;

	    if (!Number.isInteger(rows) || rows <= 0) {
	      throw new TypeError('rows must be a positive integer');
	    }

	    if (!Number.isInteger(columns) || columns <= 0) {
	      throw new TypeError('columns must be a positive integer');
	    }

	    let matrix = new Matrix(this.rows * rows, this.columns * columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.setSubMatrix(this, this.rows * i, this.columns * j);
	      }
	    }

	    return matrix;
	  }

	  fill(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, value);
	      }
	    }

	    return this;
	  }

	  neg() {
	    return this.mulS(-1);
	  }

	  getRow(index) {
	    checkRowIndex(this, index);
	    let row = [];

	    for (let i = 0; i < this.columns; i++) {
	      row.push(this.get(index, i));
	    }

	    return row;
	  }

	  getRowVector(index) {
	    return Matrix.rowVector(this.getRow(index));
	  }

	  setRow(index, array) {
	    checkRowIndex(this, index);
	    array = checkRowVector(this, array);

	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, array[i]);
	    }

	    return this;
	  }

	  swapRows(row1, row2) {
	    checkRowIndex(this, row1);
	    checkRowIndex(this, row2);

	    for (let i = 0; i < this.columns; i++) {
	      let temp = this.get(row1, i);
	      this.set(row1, i, this.get(row2, i));
	      this.set(row2, i, temp);
	    }

	    return this;
	  }

	  getColumn(index) {
	    checkColumnIndex(this, index);
	    let column = [];

	    for (let i = 0; i < this.rows; i++) {
	      column.push(this.get(i, index));
	    }

	    return column;
	  }

	  getColumnVector(index) {
	    return Matrix.columnVector(this.getColumn(index));
	  }

	  setColumn(index, array) {
	    checkColumnIndex(this, index);
	    array = checkColumnVector(this, array);

	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, array[i]);
	    }

	    return this;
	  }

	  swapColumns(column1, column2) {
	    checkColumnIndex(this, column1);
	    checkColumnIndex(this, column2);

	    for (let i = 0; i < this.rows; i++) {
	      let temp = this.get(i, column1);
	      this.set(i, column1, this.get(i, column2));
	      this.set(i, column2, temp);
	    }

	    return this;
	  }

	  addRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[j]);
	      }
	    }

	    return this;
	  }

	  subRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[j]);
	      }
	    }

	    return this;
	  }

	  mulRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[j]);
	      }
	    }

	    return this;
	  }

	  divRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[j]);
	      }
	    }

	    return this;
	  }

	  addColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[i]);
	      }
	    }

	    return this;
	  }

	  subColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[i]);
	      }
	    }

	    return this;
	  }

	  mulColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[i]);
	      }
	    }

	    return this;
	  }

	  divColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[i]);
	      }
	    }

	    return this;
	  }

	  mulRow(index, value) {
	    checkRowIndex(this, index);

	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, this.get(index, i) * value);
	    }

	    return this;
	  }

	  mulColumn(index, value) {
	    checkColumnIndex(this, index);

	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, this.get(i, index) * value);
	    }

	    return this;
	  }

	  max(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }

	    switch (by) {
	      case 'row':
	        {
	          const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[row]) {
	                max[row] = this.get(row, column);
	              }
	            }
	          }

	          return max;
	        }

	      case 'column':
	        {
	          const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[column]) {
	                max[column] = this.get(row, column);
	              }
	            }
	          }

	          return max;
	        }

	      case undefined:
	        {
	          let max = this.get(0, 0);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max) {
	                max = this.get(row, column);
	              }
	            }
	          }

	          return max;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  maxIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) > v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }

	    return idx;
	  }

	  min(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }

	    switch (by) {
	      case 'row':
	        {
	          const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[row]) {
	                min[row] = this.get(row, column);
	              }
	            }
	          }

	          return min;
	        }

	      case 'column':
	        {
	          const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[column]) {
	                min[column] = this.get(row, column);
	              }
	            }
	          }

	          return min;
	        }

	      case undefined:
	        {
	          let min = this.get(0, 0);

	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min) {
	                min = this.get(row, column);
	              }
	            }
	          }

	          return min;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  minIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) < v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }

	    return idx;
	  }

	  maxRow(row) {
	    checkRowIndex(this, row);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(row, 0);

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	      }
	    }

	    return v;
	  }

	  maxRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }

	    return idx;
	  }

	  minRow(row) {
	    checkRowIndex(this, row);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(row, 0);

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	      }
	    }

	    return v;
	  }

	  minRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }

	    return idx;
	  }

	  maxColumn(column) {
	    checkColumnIndex(this, column);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, column);

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	      }
	    }

	    return v;
	  }

	  maxColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }

	    return idx;
	  }

	  minColumn(column) {
	    checkColumnIndex(this, column);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, column);

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	      }
	    }

	    return v;
	  }

	  minColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }

	    return idx;
	  }

	  diag() {
	    let min = Math.min(this.rows, this.columns);
	    let diag = [];

	    for (let i = 0; i < min; i++) {
	      diag.push(this.get(i, i));
	    }

	    return diag;
	  }

	  norm() {
	    let type = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 'frobenius';
	    let result = 0;

	    if (type === 'max') {
	      return this.max();
	    } else if (type === 'frobenius') {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j < this.columns; j++) {
	          result = result + this.get(i, j) * this.get(i, j);
	        }
	      }

	      return Math.sqrt(result);
	    } else {
	      throw new RangeError(`unknown norm type: ${type}`);
	    }
	  }

	  cumulativeSum() {
	    let sum = 0;

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        sum += this.get(i, j);
	        this.set(i, j, sum);
	      }
	    }

	    return this;
	  }

	  dot(vector2) {
	    if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
	    let vector1 = this.to1DArray();

	    if (vector1.length !== vector2.length) {
	      throw new RangeError('vectors do not have the same size');
	    }

	    let dot = 0;

	    for (let i = 0; i < vector1.length; i++) {
	      dot += vector1[i] * vector2[i];
	    }

	    return dot;
	  }

	  mmul(other) {
	    other = Matrix.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.columns;
	    let result = new Matrix(m, p);
	    let Bcolj = new Float64Array(n);

	    for (let j = 0; j < p; j++) {
	      for (let k = 0; k < n; k++) {
	        Bcolj[k] = other.get(k, j);
	      }

	      for (let i = 0; i < m; i++) {
	        let s = 0;

	        for (let k = 0; k < n; k++) {
	          s += this.get(i, k) * Bcolj[k];
	        }

	        result.set(i, j, s);
	      }
	    }

	    return result;
	  }

	  strassen2x2(other) {
	    other = Matrix.checkMatrix(other);
	    let result = new Matrix(2, 2);
	    const a11 = this.get(0, 0);
	    const b11 = other.get(0, 0);
	    const a12 = this.get(0, 1);
	    const b12 = other.get(0, 1);
	    const a21 = this.get(1, 0);
	    const b21 = other.get(1, 0);
	    const a22 = this.get(1, 1);
	    const b22 = other.get(1, 1); // Compute intermediate values.

	    const m1 = (a11 + a22) * (b11 + b22);
	    const m2 = (a21 + a22) * b11;
	    const m3 = a11 * (b12 - b22);
	    const m4 = a22 * (b21 - b11);
	    const m5 = (a11 + a12) * b22;
	    const m6 = (a21 - a11) * (b11 + b12);
	    const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

	    const c00 = m1 + m4 - m5 + m7;
	    const c01 = m3 + m5;
	    const c10 = m2 + m4;
	    const c11 = m1 - m2 + m3 + m6;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    return result;
	  }

	  strassen3x3(other) {
	    other = Matrix.checkMatrix(other);
	    let result = new Matrix(3, 3);
	    const a00 = this.get(0, 0);
	    const a01 = this.get(0, 1);
	    const a02 = this.get(0, 2);
	    const a10 = this.get(1, 0);
	    const a11 = this.get(1, 1);
	    const a12 = this.get(1, 2);
	    const a20 = this.get(2, 0);
	    const a21 = this.get(2, 1);
	    const a22 = this.get(2, 2);
	    const b00 = other.get(0, 0);
	    const b01 = other.get(0, 1);
	    const b02 = other.get(0, 2);
	    const b10 = other.get(1, 0);
	    const b11 = other.get(1, 1);
	    const b12 = other.get(1, 2);
	    const b20 = other.get(2, 0);
	    const b21 = other.get(2, 1);
	    const b22 = other.get(2, 2);
	    const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
	    const m2 = (a00 - a10) * (-b01 + b11);
	    const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
	    const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
	    const m5 = (a10 + a11) * (-b00 + b01);
	    const m6 = a00 * b00;
	    const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
	    const m8 = (-a00 + a20) * (b02 - b12);
	    const m9 = (a20 + a21) * (-b00 + b02);
	    const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
	    const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
	    const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
	    const m13 = (a02 - a22) * (b11 - b21);
	    const m14 = a02 * b20;
	    const m15 = (a21 + a22) * (-b20 + b21);
	    const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
	    const m17 = (a02 - a12) * (b12 - b22);
	    const m18 = (a11 + a12) * (-b20 + b22);
	    const m19 = a01 * b10;
	    const m20 = a12 * b21;
	    const m21 = a10 * b02;
	    const m22 = a20 * b01;
	    const m23 = a22 * b22;
	    const c00 = m6 + m14 + m19;
	    const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
	    const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
	    const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
	    const c11 = m2 + m4 + m5 + m6 + m20;
	    const c12 = m14 + m16 + m17 + m18 + m21;
	    const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
	    const c21 = m12 + m13 + m14 + m15 + m22;
	    const c22 = m6 + m7 + m8 + m9 + m23;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(0, 2, c02);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    result.set(1, 2, c12);
	    result.set(2, 0, c20);
	    result.set(2, 1, c21);
	    result.set(2, 2, c22);
	    return result;
	  }

	  mmulStrassen(y) {
	    y = Matrix.checkMatrix(y);
	    let x = this.clone();
	    let r1 = x.rows;
	    let c1 = x.columns;
	    let r2 = y.rows;
	    let c2 = y.columns;

	    if (c1 !== r2) {
	      // eslint-disable-next-line no-console
	      console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
	    } // Put a matrix into the top left of a matrix of zeros.
	    // `rows` and `cols` are the dimensions of the output matrix.


	    function embed(mat, rows, cols) {
	      let r = mat.rows;
	      let c = mat.columns;

	      if (r === rows && c === cols) {
	        return mat;
	      } else {
	        let resultat = AbstractMatrix.zeros(rows, cols);
	        resultat = resultat.setSubMatrix(mat, 0, 0);
	        return resultat;
	      }
	    } // Make sure both matrices are the same size.
	    // This is exclusively for simplicity:
	    // this algorithm can be implemented with matrices of different sizes.


	    let r = Math.max(r1, r2);
	    let c = Math.max(c1, c2);
	    x = embed(x, r, c);
	    y = embed(y, r, c); // Our recursive multiplication function.

	    function blockMult(a, b, rows, cols) {
	      // For small matrices, resort to naive multiplication.
	      if (rows <= 512 || cols <= 512) {
	        return a.mmul(b); // a is equivalent to this
	      } // Apply dynamic padding.


	      if (rows % 2 === 1 && cols % 2 === 1) {
	        a = embed(a, rows + 1, cols + 1);
	        b = embed(b, rows + 1, cols + 1);
	      } else if (rows % 2 === 1) {
	        a = embed(a, rows + 1, cols);
	        b = embed(b, rows + 1, cols);
	      } else if (cols % 2 === 1) {
	        a = embed(a, rows, cols + 1);
	        b = embed(b, rows, cols + 1);
	      }

	      let halfRows = parseInt(a.rows / 2, 10);
	      let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

	      let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
	      let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
	      let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
	      let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
	      let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
	      let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

	      let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
	      let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
	      let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
	      let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
	      let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
	      let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
	      let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

	      let c11 = AbstractMatrix.add(m1, m4);
	      c11.sub(m5);
	      c11.add(m7);
	      let c12 = AbstractMatrix.add(m3, m5);
	      let c21 = AbstractMatrix.add(m2, m4);
	      let c22 = AbstractMatrix.sub(m1, m2);
	      c22.add(m3);
	      c22.add(m6); // Crop output to the desired size (undo dynamic padding).

	      let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
	      resultat = resultat.setSubMatrix(c11, 0, 0);
	      resultat = resultat.setSubMatrix(c12, c11.rows, 0);
	      resultat = resultat.setSubMatrix(c21, 0, c11.columns);
	      resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
	      return resultat.subMatrix(0, rows - 1, 0, cols - 1);
	    }

	    return blockMult(x, y, r, c);
	  }

	  scaleRows() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let i = 0; i < this.rows; i++) {
	      const row = this.getRow(i);

	      if (row.length > 0) {
	        rescale$1(row, {
	          min,
	          max,
	          output: row
	        });
	      }

	      newMatrix.setRow(i, row);
	    }

	    return newMatrix;
	  }

	  scaleColumns() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let i = 0; i < this.columns; i++) {
	      const column = this.getColumn(i);

	      if (column.length) {
	        rescale$1(column, {
	          min: min,
	          max: max,
	          output: column
	        });
	      }

	      newMatrix.setColumn(i, column);
	    }

	    return newMatrix;
	  }

	  flipRows() {
	    const middle = Math.ceil(this.columns / 2);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < middle; j++) {
	        let first = this.get(i, j);
	        let last = this.get(i, this.columns - 1 - j);
	        this.set(i, j, last);
	        this.set(i, this.columns - 1 - j, first);
	      }
	    }

	    return this;
	  }

	  flipColumns() {
	    const middle = Math.ceil(this.rows / 2);

	    for (let j = 0; j < this.columns; j++) {
	      for (let i = 0; i < middle; i++) {
	        let first = this.get(i, j);
	        let last = this.get(this.rows - 1 - i, j);
	        this.set(i, j, last);
	        this.set(this.rows - 1 - i, j, first);
	      }
	    }

	    return this;
	  }

	  kroneckerProduct(other) {
	    other = Matrix.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.rows;
	    let q = other.columns;
	    let result = new Matrix(m * p, n * q);

	    for (let i = 0; i < m; i++) {
	      for (let j = 0; j < n; j++) {
	        for (let k = 0; k < p; k++) {
	          for (let l = 0; l < q; l++) {
	            result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
	          }
	        }
	      }
	    }

	    return result;
	  }

	  kroneckerSum(other) {
	    other = Matrix.checkMatrix(other);

	    if (!this.isSquare() || !other.isSquare()) {
	      throw new Error('Kronecker Sum needs two Square Matrices');
	    }

	    let m = this.rows;
	    let n = other.rows;
	    let AxI = this.kroneckerProduct(Matrix.eye(n, n));
	    let IxB = Matrix.eye(m, m).kroneckerProduct(other);
	    return AxI.add(IxB);
	  }

	  transpose() {
	    let result = new Matrix(this.columns, this.rows);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        result.set(j, i, this.get(i, j));
	      }
	    }

	    return result;
	  }

	  sortRows() {
	    let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

	    for (let i = 0; i < this.rows; i++) {
	      this.setRow(i, this.getRow(i).sort(compareFunction));
	    }

	    return this;
	  }

	  sortColumns() {
	    let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

	    for (let i = 0; i < this.columns; i++) {
	      this.setColumn(i, this.getColumn(i).sort(compareFunction));
	    }

	    return this;
	  }

	  subMatrix(startRow, endRow, startColumn, endColumn) {
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

	    for (let i = startRow; i <= endRow; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
	      }
	    }

	    return newMatrix;
	  }

	  subMatrixRow(indices, startColumn, endColumn) {
	    if (startColumn === undefined) startColumn = 0;
	    if (endColumn === undefined) endColumn = this.columns - 1;

	    if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
	      throw new RangeError('Argument out of range');
	    }

	    let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        if (indices[i] < 0 || indices[i] >= this.rows) {
	          throw new RangeError(`Row index out of range: ${indices[i]}`);
	        }

	        newMatrix.set(i, j - startColumn, this.get(indices[i], j));
	      }
	    }

	    return newMatrix;
	  }

	  subMatrixColumn(indices, startRow, endRow) {
	    if (startRow === undefined) startRow = 0;
	    if (endRow === undefined) endRow = this.rows - 1;

	    if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
	      throw new RangeError('Argument out of range');
	    }

	    let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startRow; j <= endRow; j++) {
	        if (indices[i] < 0 || indices[i] >= this.columns) {
	          throw new RangeError(`Column index out of range: ${indices[i]}`);
	        }

	        newMatrix.set(j - startRow, i, this.get(j, indices[i]));
	      }
	    }

	    return newMatrix;
	  }

	  setSubMatrix(matrix, startRow, startColumn) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (matrix.isEmpty()) {
	      return this;
	    }

	    let endRow = startRow + matrix.rows - 1;
	    let endColumn = startColumn + matrix.columns - 1;
	    checkRange(this, startRow, endRow, startColumn, endColumn);

	    for (let i = 0; i < matrix.rows; i++) {
	      for (let j = 0; j < matrix.columns; j++) {
	        this.set(startRow + i, startColumn + j, matrix.get(i, j));
	      }
	    }

	    return this;
	  }

	  selection(rowIndices, columnIndices) {
	    checkRowIndices(this, rowIndices);
	    checkColumnIndices(this, columnIndices);
	    let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

	    for (let i = 0; i < rowIndices.length; i++) {
	      let rowIndex = rowIndices[i];

	      for (let j = 0; j < columnIndices.length; j++) {
	        let columnIndex = columnIndices[j];
	        newMatrix.set(i, j, this.get(rowIndex, columnIndex));
	      }
	    }

	    return newMatrix;
	  }

	  trace() {
	    let min = Math.min(this.rows, this.columns);
	    let trace = 0;

	    for (let i = 0; i < min; i++) {
	      trace += this.get(i, i);
	    }

	    return trace;
	  }

	  clone() {
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let row = 0; row < this.rows; row++) {
	      for (let column = 0; column < this.columns; column++) {
	        newMatrix.set(row, column, this.get(row, column));
	      }
	    }

	    return newMatrix;
	  }

	  sum(by) {
	    switch (by) {
	      case 'row':
	        return sumByRow(this);

	      case 'column':
	        return sumByColumn(this);

	      case undefined:
	        return sumAll(this);

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  product(by) {
	    switch (by) {
	      case 'row':
	        return productByRow(this);

	      case 'column':
	        return productByColumn(this);

	      case undefined:
	        return productAll(this);

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  mean(by) {
	    const sum = this.sum(by);

	    switch (by) {
	      case 'row':
	        {
	          for (let i = 0; i < this.rows; i++) {
	            sum[i] /= this.columns;
	          }

	          return sum;
	        }

	      case 'column':
	        {
	          for (let i = 0; i < this.columns; i++) {
	            sum[i] /= this.rows;
	          }

	          return sum;
	        }

	      case undefined:
	        return sum / this.size;

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  variance(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      unbiased = true,
	      mean = this.mean(by)
	    } = options;

	    if (typeof unbiased !== 'boolean') {
	      throw new TypeError('unbiased must be a boolean');
	    }

	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }

	          return varianceByRow(this, unbiased, mean);
	        }

	      case 'column':
	        {
	          if (!isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }

	          return varianceByColumn(this, unbiased, mean);
	        }

	      case undefined:
	        {
	          if (typeof mean !== 'number') {
	            throw new TypeError('mean must be a number');
	          }

	          return varianceAll(this, unbiased, mean);
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  standardDeviation(by, options) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    const variance = this.variance(by, options);

	    if (by === undefined) {
	      return Math.sqrt(variance);
	    } else {
	      for (let i = 0; i < variance.length; i++) {
	        variance[i] = Math.sqrt(variance[i]);
	      }

	      return variance;
	    }
	  }

	  center(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      center = this.mean(by)
	    } = options;

	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }

	          centerByRow(this, center);
	          return this;
	        }

	      case 'column':
	        {
	          if (!isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }

	          centerByColumn(this, center);
	          return this;
	        }

	      case undefined:
	        {
	          if (typeof center !== 'number') {
	            throw new TypeError('center must be a number');
	          }

	          centerAll(this, center);
	          return this;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  scale(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    let scale = options.scale;

	    switch (by) {
	      case 'row':
	        {
	          if (scale === undefined) {
	            scale = getScaleByRow(this);
	          } else if (!isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }

	          scaleByRow(this, scale);
	          return this;
	        }

	      case 'column':
	        {
	          if (scale === undefined) {
	            scale = getScaleByColumn(this);
	          } else if (!isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }

	          scaleByColumn(this, scale);
	          return this;
	        }

	      case undefined:
	        {
	          if (scale === undefined) {
	            scale = getScaleAll(this);
	          } else if (typeof scale !== 'number') {
	            throw new TypeError('scale must be a number');
	          }

	          scaleAll(this, scale);
	          return this;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  toString(options) {
	    return inspectMatrixWithOptions(this, options);
	  }

	}
	AbstractMatrix.prototype.klass = 'Matrix';

	if (typeof Symbol !== 'undefined') {
	  AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
	}

	function compareNumbers(a, b) {
	  return a - b;
	}

	function isArrayOfNumbers(array) {
	  return array.every(element => {
	    return typeof element === 'number';
	  });
	} // Synonyms


	AbstractMatrix.random = AbstractMatrix.rand;
	AbstractMatrix.randomInt = AbstractMatrix.randInt;
	AbstractMatrix.diagonal = AbstractMatrix.diag;
	AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
	AbstractMatrix.identity = AbstractMatrix.eye;
	AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
	AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
	class Matrix extends AbstractMatrix {
	  constructor(nRows, nColumns) {
	    super();

	    if (Matrix.isMatrix(nRows)) {
	      // eslint-disable-next-line no-constructor-return
	      return nRows.clone();
	    } else if (Number.isInteger(nRows) && nRows >= 0) {
	      // Create an empty matrix
	      this.data = [];

	      if (Number.isInteger(nColumns) && nColumns >= 0) {
	        for (let i = 0; i < nRows; i++) {
	          this.data.push(new Float64Array(nColumns));
	        }
	      } else {
	        throw new TypeError('nColumns must be a positive integer');
	      }
	    } else if (isAnyArray(nRows)) {
	      // Copy the values from the 2D array
	      const arrayData = nRows;
	      nRows = arrayData.length;
	      nColumns = nRows ? arrayData[0].length : 0;

	      if (typeof nColumns !== 'number') {
	        throw new TypeError('Data must be a 2D array with at least one element');
	      }

	      this.data = [];

	      for (let i = 0; i < nRows; i++) {
	        if (arrayData[i].length !== nColumns) {
	          throw new RangeError('Inconsistent array dimensions');
	        }

	        if (!isArrayOfNumbers(arrayData[i])) {
	          throw new TypeError('Input data contains non-numeric values');
	        }

	        this.data.push(Float64Array.from(arrayData[i]));
	      }
	    } else {
	      throw new TypeError('First argument must be a positive number or an array');
	    }

	    this.rows = nRows;
	    this.columns = nColumns;
	  }

	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }

	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }

	  removeRow(index) {
	    checkRowIndex(this, index);
	    this.data.splice(index, 1);
	    this.rows -= 1;
	    return this;
	  }

	  addRow(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.rows;
	    }

	    checkRowIndex(this, index, true);
	    array = Float64Array.from(checkRowVector(this, array));
	    this.data.splice(index, 0, array);
	    this.rows += 1;
	    return this;
	  }

	  removeColumn(index) {
	    checkColumnIndex(this, index);

	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns - 1);

	      for (let j = 0; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }

	      for (let j = index + 1; j < this.columns; j++) {
	        newRow[j - 1] = this.data[i][j];
	      }

	      this.data[i] = newRow;
	    }

	    this.columns -= 1;
	    return this;
	  }

	  addColumn(index, array) {
	    if (typeof array === 'undefined') {
	      array = index;
	      index = this.columns;
	    }

	    checkColumnIndex(this, index, true);
	    array = checkColumnVector(this, array);

	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns + 1);
	      let j = 0;

	      for (; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }

	      newRow[j++] = array[i];

	      for (; j < this.columns + 1; j++) {
	        newRow[j] = this.data[i][j - 1];
	      }

	      this.data[i] = newRow;
	    }

	    this.columns += 1;
	    return this;
	  }

	}
	installMathOperations(AbstractMatrix, Matrix);

	class BaseView extends AbstractMatrix {
	  constructor(matrix, rows, columns) {
	    super();
	    this.matrix = matrix;
	    this.rows = rows;
	    this.columns = columns;
	  }

	}

	class MatrixTransposeView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.columns, matrix.rows);
	  }

	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(columnIndex, rowIndex, value);
	    return this;
	  }

	  get(rowIndex, columnIndex) {
	    return this.matrix.get(columnIndex, rowIndex);
	  }

	}

	class WrapperMatrix2D extends AbstractMatrix {
	  constructor(data) {
	    super();
	    this.data = data;
	    this.rows = data.length;
	    this.columns = data[0].length;
	  }

	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }

	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }

	}

	class LuDecomposition {
	  constructor(matrix) {
	    matrix = WrapperMatrix2D.checkMatrix(matrix);
	    let lu = matrix.clone();
	    let rows = lu.rows;
	    let columns = lu.columns;
	    let pivotVector = new Float64Array(rows);
	    let pivotSign = 1;
	    let i, j, k, p, s, t, v;
	    let LUcolj, kmax;

	    for (i = 0; i < rows; i++) {
	      pivotVector[i] = i;
	    }

	    LUcolj = new Float64Array(rows);

	    for (j = 0; j < columns; j++) {
	      for (i = 0; i < rows; i++) {
	        LUcolj[i] = lu.get(i, j);
	      }

	      for (i = 0; i < rows; i++) {
	        kmax = Math.min(i, j);
	        s = 0;

	        for (k = 0; k < kmax; k++) {
	          s += lu.get(i, k) * LUcolj[k];
	        }

	        LUcolj[i] -= s;
	        lu.set(i, j, LUcolj[i]);
	      }

	      p = j;

	      for (i = j + 1; i < rows; i++) {
	        if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
	          p = i;
	        }
	      }

	      if (p !== j) {
	        for (k = 0; k < columns; k++) {
	          t = lu.get(p, k);
	          lu.set(p, k, lu.get(j, k));
	          lu.set(j, k, t);
	        }

	        v = pivotVector[p];
	        pivotVector[p] = pivotVector[j];
	        pivotVector[j] = v;
	        pivotSign = -pivotSign;
	      }

	      if (j < rows && lu.get(j, j) !== 0) {
	        for (i = j + 1; i < rows; i++) {
	          lu.set(i, j, lu.get(i, j) / lu.get(j, j));
	        }
	      }
	    }

	    this.LU = lu;
	    this.pivotVector = pivotVector;
	    this.pivotSign = pivotSign;
	  }

	  isSingular() {
	    let data = this.LU;
	    let col = data.columns;

	    for (let j = 0; j < col; j++) {
	      if (data.get(j, j) === 0) {
	        return true;
	      }
	    }

	    return false;
	  }

	  solve(value) {
	    value = Matrix.checkMatrix(value);
	    let lu = this.LU;
	    let rows = lu.rows;

	    if (rows !== value.rows) {
	      throw new Error('Invalid matrix dimensions');
	    }

	    if (this.isSingular()) {
	      throw new Error('LU matrix is singular');
	    }

	    let count = value.columns;
	    let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
	    let columns = lu.columns;
	    let i, j, k;

	    for (k = 0; k < columns; k++) {
	      for (i = k + 1; i < columns; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }

	    for (k = columns - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / lu.get(k, k));
	      }

	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }

	    return X;
	  }

	  get determinant() {
	    let data = this.LU;

	    if (!data.isSquare()) {
	      throw new Error('Matrix must be square');
	    }

	    let determinant = this.pivotSign;
	    let col = data.columns;

	    for (let j = 0; j < col; j++) {
	      determinant *= data.get(j, j);
	    }

	    return determinant;
	  }

	  get lowerTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i > j) {
	          X.set(i, j, data.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, 1);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }

	    return X;
	  }

	  get upperTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i <= j) {
	          X.set(i, j, data.get(i, j));
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }

	    return X;
	  }

	  get pivotPermutationVector() {
	    return Array.from(this.pivotVector);
	  }

	}

	function hypotenuse(a, b) {
	  let r = 0;

	  if (Math.abs(a) > Math.abs(b)) {
	    r = b / a;
	    return Math.abs(a) * Math.sqrt(1 + r * r);
	  }

	  if (b !== 0) {
	    r = a / b;
	    return Math.abs(b) * Math.sqrt(1 + r * r);
	  }

	  return 0;
	}

	class QrDecomposition {
	  constructor(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    let qr = value.clone();
	    let m = value.rows;
	    let n = value.columns;
	    let rdiag = new Float64Array(n);
	    let i, j, k, s;

	    for (k = 0; k < n; k++) {
	      let nrm = 0;

	      for (i = k; i < m; i++) {
	        nrm = hypotenuse(nrm, qr.get(i, k));
	      }

	      if (nrm !== 0) {
	        if (qr.get(k, k) < 0) {
	          nrm = -nrm;
	        }

	        for (i = k; i < m; i++) {
	          qr.set(i, k, qr.get(i, k) / nrm);
	        }

	        qr.set(k, k, qr.get(k, k) + 1);

	        for (j = k + 1; j < n; j++) {
	          s = 0;

	          for (i = k; i < m; i++) {
	            s += qr.get(i, k) * qr.get(i, j);
	          }

	          s = -s / qr.get(k, k);

	          for (i = k; i < m; i++) {
	            qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }

	      rdiag[k] = -nrm;
	    }

	    this.QR = qr;
	    this.Rdiag = rdiag;
	  }

	  solve(value) {
	    value = Matrix.checkMatrix(value);
	    let qr = this.QR;
	    let m = qr.rows;

	    if (value.rows !== m) {
	      throw new Error('Matrix row dimensions must agree');
	    }

	    if (!this.isFullRank()) {
	      throw new Error('Matrix is rank deficient');
	    }

	    let count = value.columns;
	    let X = value.clone();
	    let n = qr.columns;
	    let i, j, k, s;

	    for (k = 0; k < n; k++) {
	      for (j = 0; j < count; j++) {
	        s = 0;

	        for (i = k; i < m; i++) {
	          s += qr.get(i, k) * X.get(i, j);
	        }

	        s = -s / qr.get(k, k);

	        for (i = k; i < m; i++) {
	          X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	        }
	      }
	    }

	    for (k = n - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / this.Rdiag[k]);
	      }

	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
	        }
	      }
	    }

	    return X.subMatrix(0, n - 1, 0, count - 1);
	  }

	  isFullRank() {
	    let columns = this.QR.columns;

	    for (let i = 0; i < columns; i++) {
	      if (this.Rdiag[i] === 0) {
	        return false;
	      }
	    }

	    return true;
	  }

	  get upperTriangularMatrix() {
	    let qr = this.QR;
	    let n = qr.columns;
	    let X = new Matrix(n, n);
	    let i, j;

	    for (i = 0; i < n; i++) {
	      for (j = 0; j < n; j++) {
	        if (i < j) {
	          X.set(i, j, qr.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, this.Rdiag[i]);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }

	    return X;
	  }

	  get orthogonalMatrix() {
	    let qr = this.QR;
	    let rows = qr.rows;
	    let columns = qr.columns;
	    let X = new Matrix(rows, columns);
	    let i, j, k, s;

	    for (k = columns - 1; k >= 0; k--) {
	      for (i = 0; i < rows; i++) {
	        X.set(i, k, 0);
	      }

	      X.set(k, k, 1);

	      for (j = k; j < columns; j++) {
	        if (qr.get(k, k) !== 0) {
	          s = 0;

	          for (i = k; i < rows; i++) {
	            s += qr.get(i, k) * X.get(i, j);
	          }

	          s = -s / qr.get(k, k);

	          for (i = k; i < rows; i++) {
	            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }
	    }

	    return X;
	  }

	}

	class SingularValueDecomposition {
	  constructor(value) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	    value = WrapperMatrix2D.checkMatrix(value);

	    if (value.isEmpty()) {
	      throw new Error('Matrix must be non-empty');
	    }

	    let m = value.rows;
	    let n = value.columns;
	    const {
	      computeLeftSingularVectors = true,
	      computeRightSingularVectors = true,
	      autoTranspose = false
	    } = options;
	    let wantu = Boolean(computeLeftSingularVectors);
	    let wantv = Boolean(computeRightSingularVectors);
	    let swapped = false;
	    let a;

	    if (m < n) {
	      if (!autoTranspose) {
	        a = value.clone(); // eslint-disable-next-line no-console

	        console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
	      } else {
	        a = value.transpose();
	        m = a.rows;
	        n = a.columns;
	        swapped = true;
	        let aux = wantu;
	        wantu = wantv;
	        wantv = aux;
	      }
	    } else {
	      a = value.clone();
	    }

	    let nu = Math.min(m, n);
	    let ni = Math.min(m + 1, n);
	    let s = new Float64Array(ni);
	    let U = new Matrix(m, nu);
	    let V = new Matrix(n, n);
	    let e = new Float64Array(n);
	    let work = new Float64Array(m);
	    let si = new Float64Array(ni);

	    for (let i = 0; i < ni; i++) si[i] = i;

	    let nct = Math.min(m - 1, n);
	    let nrt = Math.max(0, Math.min(n - 2, m));
	    let mrc = Math.max(nct, nrt);

	    for (let k = 0; k < mrc; k++) {
	      if (k < nct) {
	        s[k] = 0;

	        for (let i = k; i < m; i++) {
	          s[k] = hypotenuse(s[k], a.get(i, k));
	        }

	        if (s[k] !== 0) {
	          if (a.get(k, k) < 0) {
	            s[k] = -s[k];
	          }

	          for (let i = k; i < m; i++) {
	            a.set(i, k, a.get(i, k) / s[k]);
	          }

	          a.set(k, k, a.get(k, k) + 1);
	        }

	        s[k] = -s[k];
	      }

	      for (let j = k + 1; j < n; j++) {
	        if (k < nct && s[k] !== 0) {
	          let t = 0;

	          for (let i = k; i < m; i++) {
	            t += a.get(i, k) * a.get(i, j);
	          }

	          t = -t / a.get(k, k);

	          for (let i = k; i < m; i++) {
	            a.set(i, j, a.get(i, j) + t * a.get(i, k));
	          }
	        }

	        e[j] = a.get(k, j);
	      }

	      if (wantu && k < nct) {
	        for (let i = k; i < m; i++) {
	          U.set(i, k, a.get(i, k));
	        }
	      }

	      if (k < nrt) {
	        e[k] = 0;

	        for (let i = k + 1; i < n; i++) {
	          e[k] = hypotenuse(e[k], e[i]);
	        }

	        if (e[k] !== 0) {
	          if (e[k + 1] < 0) {
	            e[k] = 0 - e[k];
	          }

	          for (let i = k + 1; i < n; i++) {
	            e[i] /= e[k];
	          }

	          e[k + 1] += 1;
	        }

	        e[k] = -e[k];

	        if (k + 1 < m && e[k] !== 0) {
	          for (let i = k + 1; i < m; i++) {
	            work[i] = 0;
	          }

	          for (let i = k + 1; i < m; i++) {
	            for (let j = k + 1; j < n; j++) {
	              work[i] += e[j] * a.get(i, j);
	            }
	          }

	          for (let j = k + 1; j < n; j++) {
	            let t = -e[j] / e[k + 1];

	            for (let i = k + 1; i < m; i++) {
	              a.set(i, j, a.get(i, j) + t * work[i]);
	            }
	          }
	        }

	        if (wantv) {
	          for (let i = k + 1; i < n; i++) {
	            V.set(i, k, e[i]);
	          }
	        }
	      }
	    }

	    let p = Math.min(n, m + 1);

	    if (nct < n) {
	      s[nct] = a.get(nct, nct);
	    }

	    if (m < p) {
	      s[p - 1] = 0;
	    }

	    if (nrt + 1 < p) {
	      e[nrt] = a.get(nrt, p - 1);
	    }

	    e[p - 1] = 0;

	    if (wantu) {
	      for (let j = nct; j < nu; j++) {
	        for (let i = 0; i < m; i++) {
	          U.set(i, j, 0);
	        }

	        U.set(j, j, 1);
	      }

	      for (let k = nct - 1; k >= 0; k--) {
	        if (s[k] !== 0) {
	          for (let j = k + 1; j < nu; j++) {
	            let t = 0;

	            for (let i = k; i < m; i++) {
	              t += U.get(i, k) * U.get(i, j);
	            }

	            t = -t / U.get(k, k);

	            for (let i = k; i < m; i++) {
	              U.set(i, j, U.get(i, j) + t * U.get(i, k));
	            }
	          }

	          for (let i = k; i < m; i++) {
	            U.set(i, k, -U.get(i, k));
	          }

	          U.set(k, k, 1 + U.get(k, k));

	          for (let i = 0; i < k - 1; i++) {
	            U.set(i, k, 0);
	          }
	        } else {
	          for (let i = 0; i < m; i++) {
	            U.set(i, k, 0);
	          }

	          U.set(k, k, 1);
	        }
	      }
	    }

	    if (wantv) {
	      for (let k = n - 1; k >= 0; k--) {
	        if (k < nrt && e[k] !== 0) {
	          for (let j = k + 1; j < n; j++) {
	            let t = 0;

	            for (let i = k + 1; i < n; i++) {
	              t += V.get(i, k) * V.get(i, j);
	            }

	            t = -t / V.get(k + 1, k);

	            for (let i = k + 1; i < n; i++) {
	              V.set(i, j, V.get(i, j) + t * V.get(i, k));
	            }
	          }
	        }

	        for (let i = 0; i < n; i++) {
	          V.set(i, k, 0);
	        }

	        V.set(k, k, 1);
	      }
	    }

	    let pp = p - 1;
	    let eps = Number.EPSILON;

	    while (p > 0) {
	      let k, kase;

	      for (k = p - 2; k >= -1; k--) {
	        if (k === -1) {
	          break;
	        }

	        const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));

	        if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
	          e[k] = 0;
	          break;
	        }
	      }

	      if (k === p - 2) {
	        kase = 4;
	      } else {
	        let ks;

	        for (ks = p - 1; ks >= k; ks--) {
	          if (ks === k) {
	            break;
	          }

	          let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);

	          if (Math.abs(s[ks]) <= eps * t) {
	            s[ks] = 0;
	            break;
	          }
	        }

	        if (ks === k) {
	          kase = 3;
	        } else if (ks === p - 1) {
	          kase = 1;
	        } else {
	          kase = 2;
	          k = ks;
	        }
	      }

	      k++;

	      switch (kase) {
	        case 1:
	          {
	            let f = e[p - 2];
	            e[p - 2] = 0;

	            for (let j = p - 2; j >= k; j--) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;

	              if (j !== k) {
	                f = -sn * e[j - 1];
	                e[j - 1] = cs * e[j - 1];
	              }

	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, p - 1);
	                  V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
	                  V.set(i, j, t);
	                }
	              }
	            }

	            break;
	          }

	        case 2:
	          {
	            let f = e[k - 1];
	            e[k - 1] = 0;

	            for (let j = k; j < p; j++) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;
	              f = -sn * e[j];
	              e[j] = cs * e[j];

	              if (wantu) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, k - 1);
	                  U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
	                  U.set(i, j, t);
	                }
	              }
	            }

	            break;
	          }

	        case 3:
	          {
	            const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
	            const sp = s[p - 1] / scale;
	            const spm1 = s[p - 2] / scale;
	            const epm1 = e[p - 2] / scale;
	            const sk = s[k] / scale;
	            const ek = e[k] / scale;
	            const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
	            const c = sp * epm1 * (sp * epm1);
	            let shift = 0;

	            if (b !== 0 || c !== 0) {
	              if (b < 0) {
	                shift = 0 - Math.sqrt(b * b + c);
	              } else {
	                shift = Math.sqrt(b * b + c);
	              }

	              shift = c / (b + shift);
	            }

	            let f = (sk + sp) * (sk - sp) + shift;
	            let g = sk * ek;

	            for (let j = k; j < p - 1; j++) {
	              let t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              let cs = f / t;
	              let sn = g / t;

	              if (j !== k) {
	                e[j - 1] = t;
	              }

	              f = cs * s[j] + sn * e[j];
	              e[j] = cs * e[j] - sn * s[j];
	              g = sn * s[j + 1];
	              s[j + 1] = cs * s[j + 1];

	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, j + 1);
	                  V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
	                  V.set(i, j, t);
	                }
	              }

	              t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              cs = f / t;
	              sn = g / t;
	              s[j] = t;
	              f = cs * e[j] + sn * s[j + 1];
	              s[j + 1] = -sn * e[j] + cs * s[j + 1];
	              g = sn * e[j + 1];
	              e[j + 1] = cs * e[j + 1];

	              if (wantu && j < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, j + 1);
	                  U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
	                  U.set(i, j, t);
	                }
	              }
	            }

	            e[p - 2] = f;
	            break;
	          }

	        case 4:
	          {
	            if (s[k] <= 0) {
	              s[k] = s[k] < 0 ? -s[k] : 0;

	              if (wantv) {
	                for (let i = 0; i <= pp; i++) {
	                  V.set(i, k, -V.get(i, k));
	                }
	              }
	            }

	            while (k < pp) {
	              if (s[k] >= s[k + 1]) {
	                break;
	              }

	              let t = s[k];
	              s[k] = s[k + 1];
	              s[k + 1] = t;

	              if (wantv && k < n - 1) {
	                for (let i = 0; i < n; i++) {
	                  t = V.get(i, k + 1);
	                  V.set(i, k + 1, V.get(i, k));
	                  V.set(i, k, t);
	                }
	              }

	              if (wantu && k < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = U.get(i, k + 1);
	                  U.set(i, k + 1, U.get(i, k));
	                  U.set(i, k, t);
	                }
	              }

	              k++;
	            }
	            p--;
	            break;
	          }
	        // no default
	      }
	    }

	    if (swapped) {
	      let tmp = V;
	      V = U;
	      U = tmp;
	    }

	    this.m = m;
	    this.n = n;
	    this.s = s;
	    this.U = U;
	    this.V = V;
	  }

	  solve(value) {
	    let Y = value;
	    let e = this.threshold;
	    let scols = this.s.length;
	    let Ls = Matrix.zeros(scols, scols);

	    for (let i = 0; i < scols; i++) {
	      if (Math.abs(this.s[i]) <= e) {
	        Ls.set(i, i, 0);
	      } else {
	        Ls.set(i, i, 1 / this.s[i]);
	      }
	    }

	    let U = this.U;
	    let V = this.rightSingularVectors;
	    let VL = V.mmul(Ls);
	    let vrows = V.rows;
	    let urows = U.rows;
	    let VLU = Matrix.zeros(vrows, urows);

	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;

	        for (let k = 0; k < scols; k++) {
	          sum += VL.get(i, k) * U.get(j, k);
	        }

	        VLU.set(i, j, sum);
	      }
	    }

	    return VLU.mmul(Y);
	  }

	  solveForDiagonal(value) {
	    return this.solve(Matrix.diag(value));
	  }

	  inverse() {
	    let V = this.V;
	    let e = this.threshold;
	    let vrows = V.rows;
	    let vcols = V.columns;
	    let X = new Matrix(vrows, this.s.length);

	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < vcols; j++) {
	        if (Math.abs(this.s[j]) > e) {
	          X.set(i, j, V.get(i, j) / this.s[j]);
	        }
	      }
	    }

	    let U = this.U;
	    let urows = U.rows;
	    let ucols = U.columns;
	    let Y = new Matrix(vrows, urows);

	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;

	        for (let k = 0; k < ucols; k++) {
	          sum += X.get(i, k) * U.get(j, k);
	        }

	        Y.set(i, j, sum);
	      }
	    }

	    return Y;
	  }

	  get condition() {
	    return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
	  }

	  get norm2() {
	    return this.s[0];
	  }

	  get rank() {
	    let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
	    let r = 0;
	    let s = this.s;

	    for (let i = 0, ii = s.length; i < ii; i++) {
	      if (s[i] > tol) {
	        r++;
	      }
	    }

	    return r;
	  }

	  get diagonal() {
	    return Array.from(this.s);
	  }

	  get threshold() {
	    return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
	  }

	  get leftSingularVectors() {
	    return this.U;
	  }

	  get rightSingularVectors() {
	    return this.V;
	  }

	  get diagonalMatrix() {
	    return Matrix.diag(this.s);
	  }

	}

	function inverse(matrix) {
	  let useSVD = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
	  matrix = WrapperMatrix2D.checkMatrix(matrix);

	  if (useSVD) {
	    return new SingularValueDecomposition(matrix).inverse();
	  } else {
	    return solve(matrix, Matrix.eye(matrix.rows));
	  }
	}
	function solve(leftHandSide, rightHandSide) {
	  let useSVD = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
	  leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
	  rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);

	  if (useSVD) {
	    return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
	  } else {
	    return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
	  }
	}

	function addStyle(serie, spectrum) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    color = '#A9A9A9',
	    opacity = 1,
	    lineWidth = 1
	  } = options; // eslint-disable-next-line @typescript-eslint/prefer-regexp-exec

	  if (color.match(/#[0-9A-F]{6}$/i)) {
	    color = (color + (opacity * 255 >> 0).toString(16)).toUpperCase();
	  } else {
	    color = color.replace(/rgb ?\((.*)\)/, `rgba($1,${opacity})`);
	  }

	  serie.style = [{
	    name: 'unselected',
	    style: {
	      line: {
	        color,
	        width: lineWidth,
	        dash: 1
	      }
	    }
	  }, {
	    name: 'selected',
	    style: {
	      line: {
	        color,
	        width: lineWidth + 2,
	        dash: 1
	      }
	    }
	  }];
	  serie.name = spectrum.label || spectrum.id;
	}

	const COLORS = ['#FFB300', '#803E75', '#FF6800', '#A6BDD7', '#C10020', '#CEA262', '#817066', '#007D34', '#F6768E', '#00538A', '#FF7A5C', '#53377A', '#FF8E00', '#B32851', '#F4C800', '#7F180D', '#93AA00', '#593315', '#F13A13', '#232C16'];

	/**
	 * Generate a jsgraph chart format from an array of Analysis
	 */

	function getJSGraph(analyses) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    colors = COLORS,
	    opacities = [1],
	    linesWidth = [1],
	    selector = {},
	    normalization,
	    xAxis = {},
	    yAxis = {}
	  } = options;
	  let series = [];
	  let xLabel = xAxis.label;
	  let yLabel = yAxis.label;
	  let xUnits = xAxis.units;
	  let yUnits = yAxis.units;

	  for (let i = 0; i < analyses.length; i++) {
	    const analysis = analyses[i];
	    let serie = {};
	    let currentData = analysis.getNormalizedSpectrum({
	      selector,
	      normalization
	    });
	    if (!currentData) continue;
	    if (!xLabel) xLabel = currentData.variables.x.label;
	    if (!yLabel) yLabel = currentData.variables.y.label;
	    if (!xUnits) xUnits = currentData.variables.x.units;
	    if (!yUnits) yUnits = currentData.variables.y.units;
	    addStyle(serie, analysis, {
	      color: colors[i % colors.length],
	      opacity: opacities[i % opacities.length],
	      lineWidth: linesWidth[i % linesWidth.length]
	    });
	    serie.data = {
	      x: currentData.variables.x.data,
	      y: currentData.variables.y.data
	    };

	    if (xAxis.logScale) {
	      serie.data = xyFilterXPositive(serie.data);
	    }

	    series.push(serie);
	  }

	  return {
	    axes: {
	      x: {
	        label: xLabel,
	        unit: xUnits,
	        unitWrapperBefore: '(',
	        unitWrapperAfter: ')',
	        flipped: false,
	        display: true,
	        ...xAxis
	      },
	      y: {
	        label: yLabel,
	        unit: yUnits,
	        unitWrapperBefore: '(',
	        unitWrapperAfter: ')',
	        flipped: false,
	        display: true,
	        ...yAxis
	      }
	    },
	    series
	  };
	}

	function getNormalizationAnnotations() {
	  let filter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let boundary = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {
	    y: {
	      min: '0px',
	      max: '2000px'
	    }
	  };
	  let {
	    exclusions = []
	  } = filter;
	  let annotations = [];
	  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
	  annotations = exclusions.map(exclusion => {
	    let annotation = {
	      type: 'rect',
	      position: [{
	        x: exclusion.from,
	        y: boundary.y.min
	      }, {
	        x: exclusion.to,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    };
	    return annotation;
	  });

	  if (filter.from !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: Number.MIN_SAFE_INTEGER,
	        y: boundary.y.min
	      }, {
	        x: filter.from,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  if (filter.to !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: filter.to,
	        y: boundary.y.min
	      }, {
	        x: Number.MAX_SAFE_INTEGER,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  return annotations;
	}

	function appendDistinctParameter(values, key, value) {
	  if (!values[key]) {
	    values[key] = {
	      key,
	      values: [],
	      count: 0
	    };
	  }

	  if (!values[key].values.includes(value)) {
	    values[key].values.push(value);
	  }

	  values[key].count++;
	}

	function appendDistinctValue(values, key) {
	  if (!values[key]) {
	    values[key] = {
	      key,
	      count: 0
	    };
	  }

	  values[key].count++;
	}

	class AnalysesManager {
	  constructor() {
	    this.analyses = [];
	  }

	  addAnalysis(analysis) {
	    let index = this.getAnalysisIndex(analysis.id);

	    if (index === undefined) {
	      this.analyses.push(analysis);
	    } else {
	      this.analyses[index] = analysis;
	    }
	  }

	  getAnalyses() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      ids
	    } = options;
	    let analyses = [];

	    for (const analysis of this.analyses) {
	      if (!ids || ids.includes(analysis.id)) {
	        analyses.push(analysis);
	      }
	    }

	    return analyses;
	  }

	  getSpectra() {
	    const spectra = [];

	    for (const analysis of this.analyses) {
	      spectra.push(...analysis.spectra);
	    }

	    return spectra;
	  }
	  /**
	   * Get an array of objects (key + count) of all the titles
	   */


	  getDistinctTitles() {
	    let values = {};

	    for (let spectrum of this.getSpectra()) {
	      if (spectrum.title) {
	        appendDistinctValue(values, spectrum.title);
	      }
	    }

	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the units
	   */


	  getDistinctUnits() {
	    var _a;

	    let values = {};

	    for (let spectrum of this.getSpectra()) {
	      if (spectrum.variables) {
	        for (let [, variable] of Object.entries(spectrum.variables)) {
	          const units = (_a = variable.units) === null || _a === void 0 ? void 0 : _a.replace(/\s+\[.*/, '');

	          if (units) {
	            appendDistinctValue(values, units);
	          }
	        }
	      }
	    }

	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the labels
	   */


	  getDistinctLabels() {
	    let values = {};

	    for (let spectrum of this.getSpectra()) {
	      if (spectrum.variables) {
	        for (let [, variable] of Object.entries(spectrum.variables)) {
	          appendDistinctValue(values, variable.label.replace(/\s+\[.*/, ''));
	        }
	      }
	    }

	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the dataTypes
	   */


	  getDistinctDataTypes() {
	    let values = {};

	    for (let spectrum of this.getSpectra()) {
	      if (spectrum.dataType) {
	        appendDistinctValue(values, spectrum.dataType);
	      }
	    }

	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the meta
	   */


	  getDistinctMeta() {
	    let values = {};

	    for (let spectrum of this.getSpectra()) {
	      if (spectrum.meta) {
	        for (let key in spectrum.meta) {
	          appendDistinctParameter(values, key, spectrum.meta[key]);
	        }
	      }
	    }

	    return Object.keys(values).map(key => values[key]);
	  }

	  removeAllAnalyses() {
	    this.analyses.splice(0);
	  }
	  /**
	   * Remove the analysis from the AnalysesManager for the specified id
	   */


	  removeAnalysis(id) {
	    let index = this.getAnalysisIndex(id);
	    if (index === undefined) return undefined;
	    return this.analyses.splice(index, 1);
	  }
	  /**
	   * Returns the index of the analysis in the analyses array
	   */


	  getAnalysisIndex(id) {
	    if (!id) return undefined;

	    for (let i = 0; i < this.analyses.length; i++) {
	      let analysis = this.analyses[i];
	      if (analysis.id === id) return i;
	    }

	    return undefined;
	  }
	  /**
	   * Checks if the ID of an analysis exists in the AnalysesManager
	   */


	  includes(id) {
	    const index = this.getAnalysisIndex(id);
	    return index === undefined ? false : !isNaN(index);
	  }

	}

	/**
	 * Center the mean
	 * @param data
	 */

	function centerMean(data) {
	  const {
	    y
	  } = data;
	  const mean = xMean(y);

	  for (let i = 0; i < y.length; i++) {
	    y[i] -= mean;
	  }

	  return {
	    data
	  };
	}

	/**
	 * Center the median
	 * @param data
	 */

	function centerMedian(data) {
	  const {
	    y
	  } = data;
	  const median = xMedian(y);

	  for (let i = 0; i < y.length; i++) {
	    y[i] -= median;
	  }

	  return {
	    data
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */

	function fromTo(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(data.x, options);
	  return {
	    data: {
	      x: data.x.subarray(fromIndex, toIndex + 1),
	      y: data.y.subarray(fromIndex, toIndex + 1)
	    }
	  };
	}

	/**
	 * Norm the Y values
	 * @param data
	 */

	function normed(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xNormed(data.y, { ...options,
	    output: data.y
	  });
	  return {
	    data
	  };
	}

	/**
	 * Center the mean
	 * @param data
	 */

	function divideBySD(data) {
	  const {
	    y
	  } = data;
	  const sd = xStandardDeviation(y);

	  for (let i = 0; i < y.length; i++) {
	    y[i] /= sd;
	  }

	  return {
	    data
	  };
	}

	/**
	 * Center the mean
	 * @param data
	 */

	function rescale(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xRescale(data.y, { ...options,
	    output: data.y
	  });
	  return {
	    data
	  };
	}

	// Based on https://github.com/scijs/cholesky-solve

	/*
	The MIT License (MIT)

	Copyright (c) 2013 Eric Arnebäck

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
	THE SOFTWARE.
	*/
	function ldlSymbolic(n
	/* A and L are n-by-n, where n >= 0 */
	, Ap
	/* input of size n + 1, not modified */
	, Ai
	/* input of size nz=Ap[n], not modified */
	, Lp
	/* output of size n + 1, not defined on input */
	, Parent
	/* output of size n, not defined on input */
	, Lnz
	/* output of size n, not defined on input */
	, Flag
	/* workspace of size n, not defn. on input or output */
	) {
	  let i, k, p, kk, p2;

	  for (k = 0; k < n; k++) {
	    /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
	    Parent[k] = -1;
	    /* parent of k is not yet known */

	    Flag[k] = k;
	    /* mark node k as visited */

	    Lnz[k] = 0;
	    /* count of nonzeros in column k of L */

	    kk = k;
	    /* kth original, or permuted, column */

	    p2 = Ap[kk + 1];

	    for (p = Ap[kk]; p < p2; p++) {
	      /* A (i,k) is nonzero (original or permuted A) */
	      i = Ai[p];

	      if (i < k) {
	        /* follow path from i to root of etree, stop at flagged node */
	        for (; Flag[i] !== k; i = Parent[i]) {
	          /* find parent of i if not yet determined */
	          if (Parent[i] === -1) Parent[i] = k;
	          Lnz[i]++;
	          /* L (k,i) is nonzero */

	          Flag[i] = k;
	          /* mark i as visited */
	        }
	      }
	    }
	  }
	  /* construct Lp index array from Lnz column counts */


	  Lp[0] = 0;

	  for (k = 0; k < n; k++) {
	    Lp[k + 1] = Lp[k] + Lnz[k];
	  }
	}

	function ldlNumeric(n
	/* A and L are n-by-n, where n >= 0 */
	, Ap
	/* input of size n+1, not modified */
	, Ai
	/* input of size nz=Ap[n], not modified */
	, Ax
	/* input of size nz=Ap[n], not modified */
	, Lp
	/* input of size n+1, not modified */
	, Parent
	/* input of size n, not modified */
	, Lnz
	/* output of size n, not defn. on input */
	, Li
	/* output of size lnz=Lp[n], not defined on input */
	, Lx
	/* output of size lnz=Lp[n], not defined on input */
	, D
	/* output of size n, not defined on input */
	, Y
	/* workspace of size n, not defn. on input or output */
	, Pattern
	/* workspace of size n, not defn. on input or output */
	, Flag
	/* workspace of size n, not defn. on input or output */
	) {
	  let yi, lKi;
	  let i, k, p, kk, p2, len, top;

	  for (k = 0; k < n; k++) {
	    /* compute nonzero Pattern of kth row of L, in topological order */
	    Y[k] = 0.0;
	    /* Y(0:k) is now all zero */

	    top = n;
	    /* stack for pattern is empty */

	    Flag[k] = k;
	    /* mark node k as visited */

	    Lnz[k] = 0;
	    /* count of nonzeros in column k of L */

	    kk = k;
	    /* kth original, or permuted, column */

	    p2 = Ap[kk + 1];

	    for (p = Ap[kk]; p < p2; p++) {
	      i = Ai[p];
	      /* get A(i,k) */

	      if (i <= k) {
	        Y[i] += Ax[p];
	        /* scatter A(i,k) into Y (sum duplicates) */

	        for (len = 0; Flag[i] !== k; i = Parent[i]) {
	          Pattern[len++] = i;
	          /* L(k,i) is nonzero */

	          Flag[i] = k;
	          /* mark i as visited */
	        }

	        while (len > 0) Pattern[--top] = Pattern[--len];
	      }
	    }
	    /* compute numerical values kth row of L (a sparse triangular solve) */


	    D[k] = Y[k];
	    /* get D(k,k) and clear Y(k) */

	    Y[k] = 0.0;

	    for (; top < n; top++) {
	      i = Pattern[top];
	      /* Pattern[top:n-1] is pattern of L(:,k) */

	      yi = Y[i];
	      /* get and clear Y(i) */

	      Y[i] = 0.0;
	      p2 = Lp[i] + Lnz[i];

	      for (p = Lp[i]; p < p2; p++) {
	        Y[Li[p]] -= Lx[p] * yi;
	      }

	      lKi = yi / D[i];
	      /* the nonzero entry L(k,i) */

	      D[k] -= lKi * yi;
	      Li[p] = k;
	      /* store L(k,i) in column form of L */

	      Lx[p] = lKi;
	      Lnz[i]++;
	      /* increment count of nonzeros in col i */
	    }

	    if (D[k] === 0.0) return k;
	    /* failure, D(k,k) is zero */
	  }

	  return n;
	  /* success, diagonal of D is all nonzero */
	}

	function ldlLsolve(n
	/* L is n-by-n, where n >= 0 */
	, X
	/* size n. right-hand-side on input, soln. on output */
	, Lp
	/* input of size n+1, not modified */
	, Li
	/* input of size lnz=Lp[n], not modified */
	, Lx
	/* input of size lnz=Lp[n], not modified */
	) {
	  let j, p, p2;

	  for (j = 0; j < n; j++) {
	    p2 = Lp[j + 1];

	    for (p = Lp[j]; p < p2; p++) {
	      X[Li[p]] -= Lx[p] * X[j];
	    }
	  }
	}

	function ldlDsolve(n
	/* D is n-by-n, where n >= 0 */
	, X
	/* size n. right-hand-side on input, soln. on output */
	, D
	/* input of size n, not modified */
	) {
	  let j;

	  for (j = 0; j < n; j++) {
	    X[j] /= D[j];
	  }
	}

	function ldlLTsolve(n
	/* L is n-by-n, where n >= 0 */
	, X
	/* size n. right-hand-side on input, soln. on output */
	, Lp
	/* input of size n+1, not modified */
	, Li
	/* input of size lnz=Lp[n], not modified */
	, Lx
	/* input of size lnz=Lp[n], not modified */
	) {
	  let j, p, p2;

	  for (j = n - 1; j >= 0; j--) {
	    p2 = Lp[j + 1];

	    for (p = Lp[j]; p < p2; p++) {
	      X[j] -= Lx[p] * X[Li[p]];
	    }
	  }
	}

	function ldlPerm(n
	/* size of X, B, and P */
	, X
	/* output of size n. */
	, B
	/* input of size n. */
	, P
	/* input permutation array of size n. */
	) {
	  let j;

	  for (j = 0; j < n; j++) {
	    X[j] = B[P[j]];
	  }
	}

	function ldlPermt(n
	/* size of X, B, and P */
	, X
	/* output of size n. */
	, B
	/* input of size n. */
	, P
	/* input permutation array of size n. */
	) {
	  let j;

	  for (j = 0; j < n; j++) {
	    X[P[j]] = B[j];
	  }
	}

	function prepare(M, n, P) {
	  // if a permutation was specified, apply it.
	  if (P) {
	    let Pinv = new Array(n);

	    for (let k = 0; k < n; k++) {
	      Pinv[P[k]] = k;
	    }

	    let Mt = []; // scratch memory
	    // Apply permutation. We make M into P*M*P^T

	    for (let a = 0; a < M.length; ++a) {
	      let ar = Pinv[M[a][0]];
	      let ac = Pinv[M[a][1]]; // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
	      // if permuted element is below diagonal, we simply transpose it.

	      if (ac < ar) {
	        let t = ac;
	        ac = ar;
	        ar = t;
	      }

	      Mt[a] = [];
	      Mt[a][0] = ar;
	      Mt[a][1] = ac;
	      Mt[a][2] = M[a][2];
	    }

	    M = Mt; // copy scratch memory.
	  } else {
	    // if P argument is null, we just use an identity permutation.
	    P = [];

	    for (let i = 0; i < n; ++i) {
	      P[i] = i;
	    }
	  } // The sparse matrix we are decomposing is A.
	  // Now we shall create A from M.


	  let Ap = new Array(n + 1);
	  let Ai = new Array(M.length);
	  let Ax = new Array(M.length); // count number of non-zero elements in columns.

	  let LNZ = [];

	  for (let i = 0; i < n; ++i) {
	    LNZ[i] = 0;
	  }

	  for (let a = 0; a < M.length; ++a) {
	    LNZ[M[a][1]]++;
	  }

	  Ap[0] = 0;

	  for (let i = 0; i < n; ++i) {
	    Ap[i + 1] = Ap[i] + LNZ[i];
	  }

	  let coloffset = [];

	  for (let a = 0; a < n; ++a) {
	    coloffset[a] = 0;
	  } // go through all elements in M, and add them to sparse matrix A.


	  for (let i = 0; i < M.length; ++i) {
	    let e = M[i];
	    let col = e[1];
	    let adr = Ap[col] + coloffset[col];
	    Ai[adr] = e[0];
	    Ax[adr] = e[2];
	    coloffset[col]++;
	  }

	  let D = new Array(n);
	  let Y = new Array(n);
	  let Lp = new Array(n + 1);
	  let Parent = new Array(n);
	  let Lnz = new Array(n);
	  let Flag = new Array(n);
	  let Pattern = new Array(n);
	  let bp1 = new Array(n);
	  let x = new Array(n);
	  let d;
	  ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
	  let Lx = new Array(Lp[n]);
	  let Li = new Array(Lp[n]);
	  d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);

	  if (d === n) {
	    return function (b) {
	      ldlPerm(n, bp1, b, P);
	      ldlLsolve(n, bp1, Lp, Li, Lx);
	      ldlDsolve(n, bp1, D);
	      ldlLTsolve(n, bp1, Lp, Li, Lx);
	      ldlPermt(n, x, bp1, P);
	      return x;
	    };
	  } else {
	    return null;
	  }
	}

	var cuthillMckee_1 = cuthillMckee;

	function compareNum(a, b) {
	  return a - b;
	}

	function cuthillMckee(list, n) {
	  var adj = new Array(n);
	  var visited = new Array(n);

	  for (var i = 0; i < n; ++i) {
	    adj[i] = [];
	    visited[i] = false;
	  }

	  for (var i = 0; i < list.length; ++i) {
	    var l = list[i];
	    adj[l[0]].push(l[1]);
	  }

	  var toVisit = new Array(n);
	  var eol = 0;
	  var ptr = 0;

	  for (var i = 0; i < n; ++i) {
	    if (visited[i]) {
	      continue;
	    }

	    toVisit[eol++] = i;
	    visited[i] = true;

	    while (ptr < eol) {
	      var v = toVisit[ptr++];
	      var nbhd = adj[v];
	      nbhd.sort(compareNum);

	      for (var j = 0; j < nbhd.length; ++j) {
	        var u = nbhd[j];

	        if (visited[u]) {
	          continue;
	        }

	        visited[u] = true;
	        toVisit[eol++] = u;
	      }
	    }
	  }

	  var result = new Array(n);

	  for (var i = 0; i < n; ++i) {
	    result[toVisit[i]] = i;
	  }

	  return result;
	}

	var cuthillMckee$1 = cuthillMckee_1;

	const getClosestNumber = function () {
	  let array = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let goal = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
	  const closest = array.reduce((prev, curr) => {
	    return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
	  });
	  return closest;
	};

	const getCloseIndex = function () {
	  let array = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let goal = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
	  const closest = getClosestNumber(array, goal);
	  return array.indexOf(closest);
	};

	const updateSystem = (matrix, y, weights) => {
	  let nbPoints = y.length;
	  let l = nbPoints - 1;
	  let newMatrix = new Array(matrix.length);
	  let newVector = new Float64Array(nbPoints);

	  for (let i = 0; i < l; i++) {
	    let w = weights[i];
	    let diag = i * 2;
	    let next = diag + 1;
	    newMatrix[diag] = matrix[diag].slice();
	    newMatrix[next] = matrix[next].slice();

	    if (w === 0) {
	      newVector[i] = 0;
	    } else {
	      newVector[i] = y[i] * w;
	      newMatrix[diag][2] += w;
	    }
	  }

	  newVector[l] = y[l] * weights[l];
	  newMatrix[l * 2] = matrix[l * 2].slice();
	  newMatrix[l * 2][2] += weights[l];
	  return [newMatrix, newVector];
	};

	const getDeltaMatrix = (nbPoints, lambda) => {
	  let matrix = [];
	  let last = nbPoints - 1;

	  for (let i = 0; i < last; i++) {
	    matrix.push([i, i, lambda * 2]);
	    matrix.push([i + 1, i, -1 * lambda]);
	  }

	  matrix[0][2] = lambda;
	  matrix.push([last, last, lambda]);
	  return {
	    lowerTriangularNonZeros: matrix,
	    permutationEncodedArray: cuthillMckee$1(matrix, nbPoints)
	  };
	};

	/**
	 * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
	 * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/master/airPLS_manuscript.pdf)
	 * @param {Array<number>} x - x axis data useful when control points or zones are submitted
	 * @param {Array<number>} y - Original data
	 * @param {object} [options={}] - Options object
	 * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
	 * @param {number} [options.factorCriterion = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
	 * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
	 * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
	 * @param {Array<number>} [options.controlPoints = []] - Array of x axis values to force that baseline cross those points.
	 * @param {Array<number>} [options.baseLineZones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
	 * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
	 */

	function airPLS(x, y) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    maxIterations = 100,
	    lambda = 100,
	    factorCriterion = 0.001,
	    weights = new Array(y.length).fill(1),
	    controlPoints = [],
	    baseLineZones = []
	  } = options;

	  if (controlPoints.length > 0) {
	    controlPoints.forEach((e, i, arr) => arr[i] = getCloseIndex(x, e));
	  }

	  if (baseLineZones.length > 0) {
	    baseLineZones.forEach(range => {
	      let indexFrom = getCloseIndex(x, range.from);
	      let indexTo = getCloseIndex(x, range.to);
	      if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];

	      for (let i = indexFrom; i < indexTo; i++) {
	        controlPoints.push(i);
	      }
	    });
	  }

	  let baseline, iteration;
	  let nbPoints = y.length;
	  let l = nbPoints - 1;
	  let sumNegDifferences = Number.MAX_SAFE_INTEGER;
	  let stopCriterion = factorCriterion * y.reduce((sum, e) => Math.abs(e) + sum, 0);
	  let {
	    lowerTriangularNonZeros,
	    permutationEncodedArray
	  } = getDeltaMatrix(nbPoints, lambda);

	  for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
	    let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
	    let cho = prepare(leftHandSide, nbPoints, permutationEncodedArray);
	    baseline = cho(rightHandSide);
	    sumNegDifferences = 0;
	    let difference = y.map(calculateError);
	    let maxNegativeDiff = -1 * Number.MAX_SAFE_INTEGER;

	    for (let i = 1; i < l; i++) {
	      let diff = difference[i];

	      if (diff >= 0) {
	        weights[i] = 0;
	      } else {
	        weights[i] = Math.exp(iteration * diff / sumNegDifferences);
	        if (maxNegativeDiff < diff) maxNegativeDiff = diff;
	      }
	    }

	    let value = Math.exp(iteration * maxNegativeDiff / sumNegDifferences);
	    weights[0] = value;
	    weights[l] = value;
	    controlPoints.forEach(i => weights[i] = value);
	  }

	  return {
	    corrected: y.map((e, i) => e - baseline[i]),
	    baseline,
	    iteration,
	    error: sumNegDifferences
	  };

	  function calculateError(e, i) {
	    let diff = e - baseline[i];
	    if (diff < 0) sumNegDifferences += diff;
	    return diff;
	  }
	}

	function _typeof(obj) {
	  "@babel/helpers - typeof";

	  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
	    _typeof = function (obj) {
	      return typeof obj;
	    };
	  } else {
	    _typeof = function (obj) {
	      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
	    };
	  }

	  return _typeof(obj);
	}
	/**
	 * Fill an array with sequential numbers
	 * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
	 * @param {object} [options={}]
	 * @param {number} [options.from=0] - first value in the array
	 * @param {number} [options.to=10] - last value in the array
	 * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
	 * @param {number} [options.step] - if not provided calculated from size
	 * @return {Array<number>}
	 */


	function sequentialFill() {
	  var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (_typeof(input) === 'object' && !isAnyArray(input)) {
	    options = input;
	    input = [];
	  }

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  var _options = options,
	      _options$from = _options.from,
	      from = _options$from === void 0 ? 0 : _options$from,
	      _options$to = _options.to,
	      to = _options$to === void 0 ? 10 : _options$to,
	      _options$size = _options.size,
	      size = _options$size === void 0 ? input.length : _options$size,
	      step = _options.step;

	  if (size !== 0 && step) {
	    throw new Error('step is defined by the array size');
	  }

	  if (!size) {
	    if (step) {
	      size = Math.floor((to - from) / step) + 1;
	    } else {
	      size = to - from + 1;
	    }
	  }

	  if (!step && size) {
	    step = (to - from) / (size - 1);
	  }

	  if (Array.isArray(input)) {
	    // only works with normal array
	    input.length = 0;

	    for (var i = 0; i < size; i++) {
	      input.push(from);
	      from += step;
	    }
	  } else {
	    if (input.length !== size) {
	      throw new Error('sequentialFill typed array must have the correct length');
	    }

	    for (var _i = 0; _i < size; _i++) {
	      input[_i] = from;
	      from += step;
	    }
	  }

	  return input;
	}

	/**
	 * Adaptive iteratively reweighted penalized least squares [1]
	 *
	 * This function calls ml-airpls
	 *
	 * References:
	 * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
	 * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
	 * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
	 * @export
	 * @param {Array<number>} ys
	 * @param {object} [options] - Options object
	 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
	 * @param {object} [options.regression] - Options for the regression
	 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
	 * @returns {BaselineOutput}
	 */

	function airPLSBaseline$1(ys) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const numberPoints = ys.length;
	  let {
	    x,
	    regressionOptions
	  } = options;

	  if (!x) {
	    x = sequentialFill({
	      from: 0,
	      to: numberPoints - 1,
	      size: numberPoints
	    });
	  }

	  let output = airPLS(x, ys, regressionOptions);
	  return {
	    baseline: output.baseline,
	    correctedSpectrum: output.corrected
	  };
	}

	function maybeToPrecision(value, digits) {
	  if (value < 0) {
	    value = 0 - value;

	    if (typeof digits === 'number') {
	      return `- ${value.toPrecision(digits)}`;
	    } else {
	      return `- ${value.toString()}`;
	    }
	  } else {
	    if (typeof digits === 'number') {
	      return value.toPrecision(digits);
	    } else {
	      return value.toString();
	    }
	  }
	}

	function checkArraySize(x, y) {
	  if (!isAnyArray(x) || !isAnyArray(y)) {
	    throw new TypeError('x and y must be arrays');
	  }

	  if (x.length !== y.length) {
	    throw new RangeError('x and y arrays must have the same length');
	  }
	}

	class BaseRegression {
	  constructor() {
	    if (new.target === BaseRegression) {
	      throw new Error('BaseRegression must be subclassed');
	    }
	  }

	  predict(x) {
	    if (typeof x === 'number') {
	      return this._predict(x);
	    } else if (isAnyArray(x)) {
	      const y = [];

	      for (let i = 0; i < x.length; i++) {
	        y.push(this._predict(x[i]));
	      }

	      return y;
	    } else {
	      throw new TypeError('x must be a number or array');
	    }
	  }

	  _predict() {
	    throw new Error('_predict must be implemented');
	  }

	  train() {// Do nothing for this package
	  }

	  toString() {
	    return '';
	  }

	  toLaTeX() {
	    return '';
	  }
	  /**
	   * Return the correlation coefficient of determination (r) and chi-square.
	   * @param {Array<number>} x
	   * @param {Array<number>} y
	   * @return {object}
	   */


	  score(x, y) {
	    if (!isAnyArray(x) || !isAnyArray(y) || x.length !== y.length) {
	      throw new Error('x and y must be arrays of the same length');
	    }

	    const n = x.length;
	    const y2 = new Array(n);

	    for (let i = 0; i < n; i++) {
	      y2[i] = this._predict(x[i]);
	    }

	    let xSum = 0;
	    let ySum = 0;
	    let chi2 = 0;
	    let rmsd = 0;
	    let xSquared = 0;
	    let ySquared = 0;
	    let xY = 0;

	    for (let i = 0; i < n; i++) {
	      xSum += y2[i];
	      ySum += y[i];
	      xSquared += y2[i] * y2[i];
	      ySquared += y[i] * y[i];
	      xY += y2[i] * y[i];

	      if (y[i] !== 0) {
	        chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
	      }

	      rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
	    }

	    const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
	    return {
	      r: r,
	      r2: r * r,
	      chi2: chi2,
	      rmsd: Math.sqrt(rmsd / n)
	    };
	  }

	}

	class PolynomialRegression extends BaseRegression {
	  constructor(x, y, degree) {
	    super();

	    if (x === true) {
	      this.degree = y.degree;
	      this.powers = y.powers;
	      this.coefficients = y.coefficients;
	    } else {
	      checkArraySize(x, y);
	      regress(this, x, y, degree);
	    }
	  }

	  _predict(x) {
	    let y = 0;

	    for (let k = 0; k < this.powers.length; k++) {
	      y += this.coefficients[k] * Math.pow(x, this.powers[k]);
	    }

	    return y;
	  }

	  toJSON() {
	    return {
	      name: 'polynomialRegression',
	      degree: this.degree,
	      powers: this.powers,
	      coefficients: this.coefficients
	    };
	  }

	  toString(precision) {
	    return this._toFormula(precision, false);
	  }

	  toLaTeX(precision) {
	    return this._toFormula(precision, true);
	  }

	  _toFormula(precision, isLaTeX) {
	    let sup = '^';
	    let closeSup = '';
	    let times = ' * ';

	    if (isLaTeX) {
	      sup = '^{';
	      closeSup = '}';
	      times = '';
	    }

	    let fn = '';
	    let str = '';

	    for (let k = 0; k < this.coefficients.length; k++) {
	      str = '';

	      if (this.coefficients[k] !== 0) {
	        if (this.powers[k] === 0) {
	          str = maybeToPrecision(this.coefficients[k], precision);
	        } else {
	          if (this.powers[k] === 1) {
	            str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
	          } else {
	            str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
	          }
	        }

	        if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
	          str = ` + ${str}`;
	        } else if (k !== this.coefficients.length - 1) {
	          str = ` ${str}`;
	        }
	      }

	      fn = str + fn;
	    }

	    if (fn.charAt(0) === '+') {
	      fn = fn.slice(1);
	    }

	    return `f(x) = ${fn}`;
	  }

	  static load(json) {
	    if (json.name !== 'polynomialRegression') {
	      throw new TypeError('not a polynomial regression model');
	    }

	    return new PolynomialRegression(true, json);
	  }

	}

	function regress(pr, x, y, degree) {
	  const n = x.length;
	  let powers;

	  if (Array.isArray(degree)) {
	    powers = degree;
	    degree = powers.length;
	  } else {
	    degree++;
	    powers = new Array(degree);

	    for (let k = 0; k < degree; k++) {
	      powers[k] = k;
	    }
	  }

	  const F = new Matrix(n, degree);
	  const Y = new Matrix([y]);

	  for (let k = 0; k < degree; k++) {
	    for (let i = 0; i < n; i++) {
	      if (powers[k] === 0) {
	        F.set(i, k, 1);
	      } else {
	        F.set(i, k, Math.pow(x[i], powers[k]));
	      }
	    }
	  }

	  const FT = new MatrixTransposeView(F);
	  const A = FT.mmul(F);
	  const B = FT.mmul(new MatrixTransposeView(Y));
	  pr.degree = degree - 1;
	  pr.powers = powers;
	  pr.coefficients = solve(A, B).to1DArray();
	}

	/**
	 * Iterative regression-based baseline correction
	 * @param {Array<number>} x - Independent axis variable
	 * @param {Array<number>} y - Dependent axis variable
	 * @param {object} [options] - Options object
	 * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {*} [options.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
	 * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
	 */

	function baselineCorrectionRegression(x, y) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    maxIterations = 100,
	    Regression = PolynomialRegression,
	    regressionOptions,
	    tolerance = 0.001
	  } = options;

	  if (!regressionOptions && Regression === PolynomialRegression) {
	    regressionOptions = 3;
	  }

	  let baseline = y.slice();
	  let fitting = y.slice();
	  let oldFitting = y;
	  let iteration = 0;
	  let delta;
	  let regression;

	  while (iteration < maxIterations) {
	    // Calculate the fitting result
	    regression = new Regression(x, baseline, regressionOptions);
	    delta = 0;

	    for (let i = 0; i < baseline.length; i++) {
	      fitting[i] = regression.predict(x[i]);

	      if (baseline[i] > fitting[i]) {
	        baseline[i] = fitting[i];
	      }

	      delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
	    } // Stop criterion


	    if (delta < tolerance) {
	      break;
	    } else {
	      oldFitting = fitting.slice();
	      iteration++;
	    }
	  } // removes baseline


	  let corrected = new Array(baseline.length);

	  for (let j = 0; j < baseline.length; j++) {
	    corrected[j] = y[j] - baseline[j];
	  }

	  return {
	    corrected,
	    delta,
	    iteration,
	    baseline,
	    regression: regression
	  };
	}

	/**
	 * Iterative polynomial fitting [1]
	 *
	 * Implementation based on ml-baseline-correction-regression
	 *
	 * References:
	 * [1] Gan, F.; Ruan, G.; Mo, J.
	 * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
	 *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
	 * https://doi.org/10.1016/j.chemolab.2005.08.009.
	 * @export
	 * @param {Array<number>} ys
	 * @param {object} [options] - Options object
	 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
	 * @param {Object} [options.regression]
	 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {Object} [options.regression]
	 * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
	 * @returns {BaselineOutput}
	 */

	function iterativePolynomialBaseline$1(ys) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const numberPoints = ys.length;
	  let {
	    x,
	    regressionOptions
	  } = options;

	  if (!x) {
	    x = sequentialFill({
	      from: 0,
	      to: numberPoints - 1,
	      size: numberPoints
	    });
	  }

	  let output = baselineCorrectionRegression(x, ys, regressionOptions);
	  return {
	    baseline: output.baseline,
	    correctedSpectrum: output.corrected
	  };
	}

	function median(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  return quickSelectMedian(input.slice());
	}

	/**
	 * Checks if input is valdi
	 *
	 * @param input - input
	 */

	function xCheck(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	}

	function sum(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var sumValue = 0;

	  for (var i = 0; i < input.length; i++) {
	    sumValue += input[i];
	  }

	  return sumValue;
	}

	function mean(input) {
	  return sum(input) / input.length;
	}

	/**
	 * This function pads an array
	 *
	 *
	 * @param array - the array that will be padded
	 * @param options - options
	 */

	function xPadding(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    size = 0,
	    value = 0,
	    algorithm = ''
	  } = options;
	  xCheck(array);

	  if (!algorithm) {
	    if (array instanceof Float64Array) {
	      return array.slice();
	    } else {
	      return Float64Array.from(array);
	    }
	  }

	  let result = new Float64Array(array.length + size * 2);

	  for (let i = 0; i < array.length; i++) {
	    result[i + size] = array[i];
	  }

	  let fromEnd = size + array.length;
	  let toEnd = 2 * size + array.length;

	  switch (algorithm.toLowerCase()) {
	    case 'value':
	      for (let i = 0; i < size; i++) {
	        result[i] = value;
	      }

	      for (let i = fromEnd; i < toEnd; i++) {
	        result[i] = value;
	      }

	      break;

	    case 'duplicate':
	      for (let i = 0; i < size; i++) {
	        result[i] = array[0];
	      }

	      for (let i = fromEnd; i < toEnd; i++) {
	        result[i] = array[array.length - 1];
	      }

	      break;

	    case 'circular':
	      for (let i = 0; i < size; i++) {
	        result[i] = array[(array.length - size % array.length + i) % array.length];
	      }

	      for (let i = 0; i < size; i++) {
	        result[i + fromEnd] = array[i % array.length];
	      }

	      break;

	    default:
	      throw Error('xPadding: unknown algorithm');
	  }

	  return result;
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - the array that will be rotated
	 * @param fct - callback function that from an array returns a value.
	 * @param options - options
	 */

	function xRolling(array, fct) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  xCheck(array);
	  if (typeof fct !== 'function') throw Error('fct has to be a function');
	  const {
	    window = 5,
	    padding = {}
	  } = options;
	  const {
	    size = window - 1,
	    algorithm,
	    value
	  } = padding;
	  array = xPadding(array, {
	    size,
	    algorithm,
	    value
	  }); // ensure we get a copy and it is float64

	  const newArray = [];

	  for (let i = 0; i < array.length - window + 1; i++) {
	    let subArray = new Float64Array(array.buffer, i * 8, window); // we will send a view to the original buffer

	    newArray.push(fct(subArray));
	  }

	  return newArray;
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - the array that will be rotated
	 * @param options - option
	 */

	function xRollingAverage(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return xRolling(array, mean, options);
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - the array that will be rotated
	 * @param options - options
	 */

	function xRollingMedian(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return xRolling(array, median, options);
	}

	/**

	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
	 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
	 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
	 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
	 * @returns {BaselineOutput}
	 */

	function rollingAverageBaseline$1(ys) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let window = Math.max(Math.round(ys.length * 0.1), 2);
	  let defaults = {
	    window: window,
	    padding: {
	      size: window - 1,
	      algorithm: 'duplicate',
	      value: 0
	    }
	  };
	  let actualOptions = Object.assign({}, defaults, options);
	  let baseline = xRollingAverage(ys, actualOptions);
	  let corrected = new Float64Array(ys.length);

	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }

	  return {
	    baseline: baseline,
	    correctedSpectrum: corrected
	  };
	}

	/**
	 * Rolling ball baseline correction algorithm.
	 * From the abstract of (1):
	 * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
	 * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
	 *
	 * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
	 * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
	 *
	 * Reference:
	 * (1) Kneen, M. A.; Annegarn, H. J.
	 *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
	 *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
	 *     https://doi.org/10.1016/0168-583X(95)00908-6.
	 * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
	 *     https://cran.r-project.org/web/packages/baseline/index.html
	 * @export
	 * @param {Array} spectrum
	 * @param {Object} [options={}]
	 * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
	 * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the spectrum length
	 */

	function rollingBall(spectrum) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(spectrum)) {
	    throw new Error('Spectrum must be an array');
	  }

	  if (spectrum.length === 0) {
	    throw new TypeError('Spectrum must not be empty');
	  }

	  const numberPoints = spectrum.length;
	  const maxima = new Float64Array(numberPoints);
	  const minima = new Float64Array(numberPoints);
	  const baseline = new Float64Array(numberPoints); // windowM 4 percent of spectrum length
	  // windowS 8 percent of spectrum length

	  const {
	    windowM = Math.round(numberPoints * 0.04),
	    windowS = Math.round(numberPoints * 0.08)
	  } = options; // fi(1) in original paper

	  for (let i = 0; i < spectrum.length; i++) {
	    let windowLeft = Math.max(0, i - windowM);
	    let windowRight = Math.min(i + windowM + 1, spectrum.length);
	    minima[i] = xMinValue(spectrum, {
	      fromIndex: windowLeft,
	      toIndex: windowRight
	    });
	  } // fi in original paper


	  for (let i = 0; i < minima.length; i++) {
	    let windowLeft = Math.max(0, i - windowM);
	    let windowRight = Math.min(i + windowM + 1, minima.length);
	    maxima[i] = xMaxValue(minima, {
	      fromIndex: windowLeft,
	      toIndex: windowRight
	    });
	  }

	  for (let i = 0; i < minima.length; i++) {
	    let windowLeft = Math.max(0, i - windowS);
	    let windowRight = Math.min(i + windowS + 1, maxima.length);
	    baseline[i] = xMean(maxima.subarray(windowLeft, windowRight));
	  }

	  return baseline;
	}

	/**
	 * Rolling ball baseline correction algorithm.
	 * From the abstract of (1):
	 * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
	 * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
	 *
	 * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
	 * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
	 *
	 * Algorithm described in (1), but in the implementation here the window width does not change.
	 *
	 * Reference:
	 * (1) Kneen, M. A.; Annegarn, H. J.
	 *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
	 *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
	 *     https://doi.org/10.1016/0168-583X(95)00908-6.
	 * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
	 *     https://cran.r-project.org/web/packages/baseline/index.html
	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
	 * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
	 * @returns {BaselineOutput}
	 */

	function rollingBallBaseline$1(ys) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const baseline = rollingBall(ys, options);
	  let corrected = new Float64Array(ys.length);

	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }

	  return {
	    baseline: baseline,
	    correctedSpectrum: corrected
	  };
	}

	/**

	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
	 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
	 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
	 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
	 * @returns {BaselineOutput}
	 */

	function rollingMedianBaseline$1(ys) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let window = Math.max(Math.round(ys.length * 0.1), 2);
	  let defaults = {
	    window: window,
	    padding: {
	      size: window - 1,
	      algorithm: 'duplicate',
	      value: 0
	    }
	  };
	  let actualOptions = Object.assign({}, defaults, options);
	  let baseline = xRollingMedian(ys, actualOptions);
	  let corrected = new Float64Array(ys.length);

	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }

	  return {
	    baseline: baseline,
	    correctedSpectrum: corrected
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */

	function airPLSBaseline(data) {
	  data.y = airPLSBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */

	function iterativePolynomialBaseline(data) {
	  data.y = iterativePolynomialBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */

	function rollingAverageBaseline(data) {
	  data.y = rollingAverageBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */

	function rollingBallBaseline(data) {
	  data.y = rollingBallBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */

	function rollingMedianBaseline(data) {
	  data.y = rollingMedianBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	/**
	 * Apply Savitzky Golay algorithm
	 * @param [ys] Array of y values
	 * @param [xs] Array of X or deltaX
	 * @return  Array containing the new ys (same length)
	 */

	function sgg(ys, xs) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    windowSize = 9,
	    derivative = 0,
	    polynomial = 3
	  } = options;

	  if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
	    throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
	  }

	  if (!isAnyArray(ys)) {
	    throw new TypeError('Y values must be an array');
	  }

	  if (typeof xs === 'undefined') {
	    throw new TypeError('X must be defined');
	  }

	  if (windowSize > ys.length) {
	    throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
	  }

	  if (derivative < 0 || !Number.isInteger(derivative)) {
	    throw new RangeError('Derivative should be a positive integer');
	  }

	  if (polynomial < 1 || !Number.isInteger(polynomial)) {
	    throw new RangeError('Polynomial should be a positive integer');
	  }

	  if (polynomial >= 6) {
	    // eslint-disable-next-line no-console
	    console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
	  }

	  let half = Math.floor(windowSize / 2);
	  let np = ys.length;
	  let ans = new Float64Array(np);
	  let weights = fullWeights(windowSize, polynomial, derivative);
	  let hs = 0;
	  let constantH = true;

	  if (isAnyArray(xs)) {
	    constantH = false;
	  } else {
	    hs = Math.pow(xs, derivative);
	  } //For the borders


	  for (let i = 0; i < half; i++) {
	    let wg1 = weights[half - i - 1];
	    let wg2 = weights[half + i + 1];
	    let d1 = 0;
	    let d2 = 0;

	    for (let l = 0; l < windowSize; l++) {
	      d1 += wg1[l] * ys[l];
	      d2 += wg2[l] * ys[np - windowSize + l];
	    }

	    if (constantH) {
	      ans[half - i - 1] = d1 / hs;
	      ans[np - half + i] = d2 / hs;
	    } else {
	      hs = getHs(xs, half - i - 1, half, derivative);
	      ans[half - i - 1] = d1 / hs;
	      hs = getHs(xs, np - half + i, half, derivative);
	      ans[np - half + i] = d2 / hs;
	    }
	  } //For the internal points


	  let wg = weights[half];

	  for (let i = windowSize; i <= np; i++) {
	    let d = 0;

	    for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];

	    if (!constantH) {
	      hs = getHs(xs, i - half - 1, half, derivative);
	    }

	    ans[i - half - 1] = d / hs;
	  }

	  return ans;
	}

	function getHs(h, center, half, derivative) {
	  let hs = 0;
	  let count = 0;

	  for (let i = center - half; i < center + half; i++) {
	    if (i >= 0 && i < h.length - 1) {
	      hs += h[i + 1] - h[i];
	      count++;
	    }
	  }

	  return Math.pow(hs / count, derivative);
	}

	function gramPoly(i, m, k, s) {
	  let Grampoly = 0;

	  if (k > 0) {
	    Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
	  } else {
	    if (k === 0 && s === 0) {
	      Grampoly = 1;
	    } else {
	      Grampoly = 0;
	    }
	  }

	  return Grampoly;
	}

	function genFact(a, b) {
	  let gf = 1;

	  if (a >= b) {
	    for (let j = a - b + 1; j <= a; j++) {
	      gf *= j;
	    }
	  }

	  return gf;
	}

	function weight(i, t, m, n, s) {
	  let sum = 0;

	  for (let k = 0; k <= n; k++) {
	    sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
	  }

	  return sum;
	}
	/**
	 * @private
	 * @param m  Number of points
	 * @param n  Polynomial grade
	 * @param s  Derivative
	 */


	function fullWeights(m, n, s) {
	  let weights = new Array(m);
	  let np = Math.floor(m / 2);

	  for (let t = -np; t <= np; t++) {
	    weights[t + np] = new Float64Array(m);

	    for (let j = -np; j <= np; j++) {
	      weights[t + np][j + np] = weight(j, t, np, n, s);
	    }
	  }

	  return weights;
	}

	/**
	 * Calculate the first derivative using Savitzky–Golay filter.
	 * @param data
	 */

	function firstDerivative(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, { ...options,
	        derivative: 1
	      })
	    }
	  };
	}

	/**
	 * Calculate the second derivative using Savitzky–Golay filter.
	 * @param data
	 */

	function secondDerivative(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, { ...options,
	        derivative: 2
	      })
	    }
	  };
	}

	/**
	 * Calculate the third derivative using Savitzky–Golay filter.
	 * @param data
	 */

	function thirdDerivative(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, { ...options,
	        derivative: 3
	      })
	    }
	  };
	}

	/**
	 * Apply the Savitzky Golay Generalized Filter
	 * @param data
	 */

	function savitzkyGolay(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, options)
	    }
	  };
	}

	/**
	 * Ensure X values are strictly monotonic increasing
	 * http://www-groups.mcs.st-andrews.ac.uk/~john/analysis/Lectures/L8.html
	 * @param data
	 */

	function ensureGrowing(data) {
	  return {
	    data: xyEnsureGrowingX(data)
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */

	function equallySpaced(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return {
	    data: xyEquallySpaced(data, options)
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */

	function filterX(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return {
	    data: xyFilterX(data, options)
	  };
	}

	var Filters = /*#__PURE__*/Object.freeze({
		__proto__: null,
		centerMean: centerMean,
		centerMedian: centerMedian,
		fromTo: fromTo,
		normed: normed,
		divideBySD: divideBySD,
		rescale: rescale,
		airPLSBaseline: airPLSBaseline,
		iterativePolynomialBaseline: iterativePolynomialBaseline,
		rollingAverageBaseline: rollingAverageBaseline,
		rollingBallBaseline: rollingBallBaseline,
		rollingMedianBaseline: rollingMedianBaseline,
		firstDerivative: firstDerivative,
		secondDerivative: secondDerivative,
		thirdDerivative: thirdDerivative,
		savitzkyGolay: savitzkyGolay,
		ensureGrowing: ensureGrowing,
		equallySpaced: equallySpaced,
		filterX: filterX
	});

	/**
	 * Apply filters on {x:[], y:[]}
	 * @returns A very important number
	 */

	function filterXY(data, filters) {
	  let result = {
	    data: {
	      x: xEnsureFloat64(data.x),
	      y: xEnsureFloat64(data.y)
	    }
	  };
	  const logs = [];

	  for (let filter of filters) {
	    const start = Date.now(); // eslint-disable-next-line import/namespace

	    const filterFct = Filters[filter.name];

	    if (!filterFct) {
	      throw new Error(`Unknown filter: ${filter.name}`);
	    } // @ts-expect-error some method have options and some other ones don't have any options


	    result = filterFct(result.data, filter.options);
	    logs.push({
	      name: filter.name,
	      time: Date.now() - start
	    });
	  }

	  return {
	    logs,
	    data: result.data
	  };
	}

	function getNormalizedSpectrum(spectrum) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  var _a;

	  let data = {
	    x: spectrum.variables.x.data,
	    y: spectrum.variables.y.data
	  };
	  let newSpectrum = {
	    variables: {
	      x: {
	        data: spectrum.variables.x.data,
	        units: spectrum.variables.x.units,
	        label: spectrum.variables.x.label
	      },
	      y: {
	        data: spectrum.variables.y.data,
	        units: spectrum.variables.y.units,
	        label: spectrum.variables.y.label
	      }
	    }
	  };
	  if (spectrum.title) newSpectrum.title = spectrum.title;
	  if (spectrum.dataType) newSpectrum.dataType = spectrum.dataType;
	  if (spectrum.meta) newSpectrum.meta = spectrum.meta;
	  let {
	    from = spectrum.variables.x.min,
	    to = spectrum.variables.x.max,
	    numberOfPoints,
	    filters = [],
	    exclusions = [],
	    zones = []
	  } = options;
	  filters = JSON.parse(JSON.stringify(filters));

	  if (numberOfPoints) {
	    filters.push({
	      name: 'equallySpaced',
	      options: {
	        from,
	        to,
	        exclusions,
	        zones,
	        numberOfPoints
	      }
	    });
	  } else {
	    filters.push({
	      name: 'filterX',
	      options: {
	        from,
	        to,
	        exclusions,
	        zones
	      }
	    });
	  }

	  let {
	    x,
	    y
	  } = filterXY(data, filters).data; // filters change the y axis, we get rid of the units
	  // TODO we should deal correctly with this problem

	  if (filters.length > 1) {
	    newSpectrum.variables.y.units = '';
	    newSpectrum.variables.y.label = (_a = newSpectrum.variables.y.label) === null || _a === void 0 ? void 0 : _a.replace(/\s*\[.*\]/, '');
	  }

	  newSpectrum.variables.x.data = x;
	  newSpectrum.variables.x.min = min(x);
	  newSpectrum.variables.x.max = max(x);
	  newSpectrum.variables.x.isMonotone = xIsMonotone(x);
	  newSpectrum.variables.y.data = y;
	  newSpectrum.variables.y.min = min(y);
	  newSpectrum.variables.y.max = max(y);
	  newSpectrum.variables.y.isMonotone = xIsMonotone(y);
	  return newSpectrum;
	}

	var quantities = {exports: {}};

	/*
	The MIT License (MIT)
	Copyright © 2006-2007 Kevin C. Olbrich
	Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

	Permission is hereby granted, free of charge, to any person obtaining a copy of
	this software and associated documentation files (the "Software"), to deal in
	the Software without restriction, including without limitation the rights to
	use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is furnished to do
	so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
	*/

	(function (module, exports) {
	  (function (global, factory) {
	    module.exports = factory() ;
	  })(commonjsGlobal, function () {
	    /**
	     * Tests if a value is a string
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a string, false otherwise
	     */

	    function isString(value) {
	      return typeof value === "string" || value instanceof String;
	    }
	    /*
	     * Prefer stricter Number.isFinite if currently supported.
	     * To be dropped when ES6 is finalized. Obsolete browsers will
	     * have to use ES6 polyfills.
	     */


	    var isFiniteImpl = Number.isFinite || window.isFinite;
	    /**
	     * Tests if a value is a number
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a number, false otherwise
	     */

	    function isNumber(value) {
	      // Number.isFinite allows not to consider NaN or '1' as numbers
	      return isFiniteImpl(value);
	    }
	    /*
	     * Identity function
	     */


	    function identity(value) {
	      return value;
	    }
	    /**
	     * Returns unique strings from list
	     *
	     * @param {string[]} strings - array of strings
	     *
	     *
	     * @returns {string[]} a new array of strings without duplicates
	     */


	    function uniq(strings) {
	      var seen = {};
	      return strings.filter(function (item) {
	        return seen.hasOwnProperty(item) ? false : seen[item] = true;
	      });
	    }

	    function compareArray(array1, array2) {
	      if (array2.length !== array1.length) {
	        return false;
	      }

	      for (var i = 0; i < array1.length; i++) {
	        if (array2[i].compareArray) {
	          if (!array2[i].compareArray(array1[i])) {
	            return false;
	          }
	        }

	        if (array2[i] !== array1[i]) {
	          return false;
	        }
	      }

	      return true;
	    }

	    function assign(target, properties) {
	      Object.keys(properties).forEach(function (key) {
	        target[key] = properties[key];
	      });
	    }
	    /**
	     * Safely multiplies numbers while avoiding floating errors
	     * like 0.1 * 0.1 => 0.010000000000000002
	     *
	     * @param {...number} numbers - numbers to multiply
	     *
	     * @returns {number} result
	     */


	    function mulSafe() {
	      var result = 1,
	          decimals = 0;

	      for (var i = 0; i < arguments.length; i++) {
	        var arg = arguments[i];
	        decimals = decimals + getFractional(arg);
	        result *= arg;
	      }

	      return decimals !== 0 ? round(result, decimals) : result;
	    }
	    /**
	     * Safely divides two numbers while avoiding floating errors
	     * like 0.3 / 0.05 => 5.999999999999999
	     *
	     * @returns {number} result
	     * @param {number} num Numerator
	     * @param {number} den Denominator
	     */


	    function divSafe(num, den) {
	      if (den === 0) {
	        throw new Error("Divide by zero");
	      }

	      var factor = Math.pow(10, getFractional(den));
	      var invDen = factor / (factor * den);
	      return mulSafe(num, invDen);
	    }
	    /**
	     * Rounds value at the specified number of decimals
	     *
	     * @param {number} val - value to round
	     * @param {number} decimals - number of decimals
	     *
	     * @returns {number} rounded number
	     */


	    function round(val, decimals) {
	      return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
	    }

	    function getFractional(num) {
	      // Check for NaNs or Infinities
	      if (!isFinite(num)) {
	        return 0;
	      } // Faster than parsing strings
	      // http://jsperf.com/count-decimals/2


	      var count = 0;

	      while (num % 1 !== 0) {
	        num *= 10;
	        count++;
	      }

	      return count;
	    }
	    /**
	     * Custom error type definition
	     * @constructor
	     */


	    function QtyError() {
	      var err;

	      if (!this) {
	        // Allows to instantiate QtyError without new()
	        err = Object.create(QtyError.prototype);
	        QtyError.apply(err, arguments);
	        return err;
	      }

	      err = Error.apply(this, arguments);
	      this.name = "QtyError";
	      this.message = err.message;
	      this.stack = err.stack;
	    }

	    QtyError.prototype = Object.create(Error.prototype, {
	      constructor: {
	        value: QtyError
	      }
	    });
	    /*
	     * Throws incompatible units error
	     * @param {string} left - units
	     * @param {string} right - units incompatible with first argument
	     * @throws "Incompatible units" error
	     */

	    function throwIncompatibleUnits(left, right) {
	      throw new QtyError("Incompatible units: " + left + " and " + right);
	    }

	    var UNITS = {
	      /* prefixes */
	      "<googol>": [["googol"], 1e100, "prefix"],
	      "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
	      "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
	      "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
	      "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
	      "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
	      "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
	      "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
	      "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
	      "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
	      "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
	      "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
	      "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
	      "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
	      "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
	      "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
	      "<kilo>": [["k", "kilo"], 1e3, "prefix"],
	      "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
	      "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
	      "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
	      "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
	      "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
	      "<micro>": [["u", "\u03BC"
	      /*µ as greek letter*/
	      , "\u00B5"
	      /*µ as micro sign*/
	      , "Micro", "mc", "micro"], 1e-6, "prefix"],
	      "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
	      "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
	      "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
	      "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
	      "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
	      "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
	      "<1>": [["1", "<1>"], 1, ""],

	      /* length units */
	      "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
	      "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
	      "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
	      "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
	      "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
	      "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
	      "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
	      "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
	      "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
	      "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
	      "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
	      "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
	      "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
	      "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
	      "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
	      "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
	      "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
	      "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
	      "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
	      "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
	      "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],

	      /* mass */
	      "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
	      "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
	      "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
	      "<metric-ton>": [["tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
	      "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
	      "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
	      "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
	      "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
	      "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
	      "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
	      "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],

	      /* area */
	      "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
	      "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
	      "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],

	      /* volume */
	      "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],

	      /* speed */
	      "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
	      "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
	      "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
	      "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],

	      /* acceleration */
	      "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
	      "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],

	      /* temperature_difference */
	      "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
	      "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
	      "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
	      "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],

	      /* time */
	      "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
	      "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
	      "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
	      "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
	      "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
	      "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
	      "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
	      "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
	      "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],

	      /* pressure */
	      "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],

	      /* viscosity */
	      "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
	      "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],

	      /* substance */
	      "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],

	      /* concentration */
	      "<molar>": [["M", "molar"], 1000, "concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
	      "<wtpercent>": [["wt%", "wtpercent"], 10, "concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],

	      /* activity */
	      "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
	      "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],

	      /* capacitance */
	      "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],

	      /* charge */
	      "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
	      "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],

	      /* current */
	      "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],

	      /* conductance */
	      "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],

	      /* inductance */
	      "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],

	      /* potential */
	      "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],

	      /* resistance */
	      "<ohm>": [["Ohm", "ohm", "\u03A9"
	      /*Ω as greek letter*/
	      , "\u2126"
	      /*Ω as ohm sign*/
	      ], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],

	      /* magnetism */
	      "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],

	      /* energy */
	      "<joule>": [["J", "joule", "Joule", "joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],

	      /* force */
	      "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],

	      /* frequency */
	      "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],

	      /* angle */
	      "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
	      "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
	      "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
	      "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],

	      /* rotation */
	      "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
	      "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],

	      /* information */
	      "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
	      "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],

	      /* information rate */
	      "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
	      "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],

	      /* currency */
	      "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
	      "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],

	      /* luminosity */
	      "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
	      "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
	      "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],

	      /* power */
	      "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],

	      /* radiation */
	      "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
	      "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],

	      /* rate */
	      "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],

	      /* resolution / typography */
	      "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
	      "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
	      "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
	      "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],

	      /* other */
	      "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
	      "<each>": [["each"], 1.0, "counting", ["<each>"]],
	      "<count>": [["count"], 1.0, "counting", ["<each>"]],
	      "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
	      "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
	      "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
	      "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
	      "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
	      "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
	      "<ppt>": [["ppt"], 1e-9, "prefix_only", ["<1>"]],
	      "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
	      "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
	    };
	    var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
	    var UNITY = "<1>";
	    var UNITY_ARRAY = [UNITY]; // Setup

	    /**
	     * Asserts unit definition is valid
	     *
	     * @param {string} unitDef - Name of unit to test
	     * @param {Object} definition - Definition of unit to test
	     *
	     * @returns {void}
	     * @throws {QtyError} if unit definition is not valid
	     */

	    function validateUnitDefinition(unitDef, definition) {
	      var scalar = definition[1];
	      var numerator = definition[3] || [];
	      var denominator = definition[4] || [];

	      if (!isNumber(scalar)) {
	        throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
	      }

	      numerator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
	        }
	      });
	      denominator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
	        }
	      });
	    }

	    var PREFIX_VALUES = {};
	    var PREFIX_MAP = {};
	    var UNIT_VALUES = {};
	    var UNIT_MAP = {};
	    var OUTPUT_MAP = {};

	    for (var unitDef in UNITS) {
	      if (UNITS.hasOwnProperty(unitDef)) {
	        var definition = UNITS[unitDef];

	        if (definition[2] === "prefix") {
	          PREFIX_VALUES[unitDef] = definition[1];

	          for (var i = 0; i < definition[0].length; i++) {
	            PREFIX_MAP[definition[0][i]] = unitDef;
	          }
	        } else {
	          validateUnitDefinition(unitDef, definition);
	          UNIT_VALUES[unitDef] = {
	            scalar: definition[1],
	            numerator: definition[3],
	            denominator: definition[4]
	          };

	          for (var j = 0; j < definition[0].length; j++) {
	            UNIT_MAP[definition[0][j]] = unitDef;
	          }
	        }

	        OUTPUT_MAP[unitDef] = definition[0][0];
	      }
	    }
	    /**
	     * Returns a list of available units of kind
	     *
	     * @param {string} [kind] - kind of units
	     * @returns {array} names of units
	     * @throws {QtyError} if kind is unknown
	     */


	    function getUnits(kind) {
	      var i;
	      var units = [];
	      var unitKeys = Object.keys(UNITS);

	      if (typeof kind === "undefined") {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      } else if (this.getKinds().indexOf(kind) === -1) {
	        throw new QtyError("Kind not recognized");
	      } else {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (UNITS[unitKeys[i]][2] === kind) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      }

	      return units.sort(function (a, b) {
	        if (a.toLowerCase() < b.toLowerCase()) {
	          return -1;
	        }

	        if (a.toLowerCase() > b.toLowerCase()) {
	          return 1;
	        }

	        return 0;
	      });
	    }
	    /**
	     * Returns a list of alternative names for a unit
	     *
	     * @param {string} unitName - name of unit
	     * @returns {string[]} aliases for unit
	     * @throws {QtyError} if unit is unknown
	     */


	    function getAliases(unitName) {
	      if (!UNIT_MAP[unitName]) {
	        throw new QtyError("Unit not recognized");
	      }

	      return UNITS[UNIT_MAP[unitName]][0];
	    }

	    var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];
	    /*
	    calculates the unit signature id for use in comparing compatible units and simplification
	    the signature is based on a simple classification of units and is based on the following publication
	     Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
	    21(8), Aug 1995, pp.651-661
	    doi://10.1109/32.403789
	    http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
	    */

	    function unitSignature() {
	      if (this.signature) {
	        return this.signature;
	      }

	      var vector = unitSignatureVector.call(this);

	      for (var i = 0; i < vector.length; i++) {
	        vector[i] *= Math.pow(20, i);
	      }

	      return vector.reduce(function (previous, current) {
	        return previous + current;
	      }, 0);
	    } // calculates the unit signature vector used by unit_signature


	    function unitSignatureVector() {
	      if (!this.isBase()) {
	        return unitSignatureVector.call(this.toBase());
	      }

	      var vector = new Array(SIGNATURE_VECTOR.length);

	      for (var i = 0; i < vector.length; i++) {
	        vector[i] = 0;
	      }

	      var r, n;

	      for (var j = 0; j < this.numerator.length; j++) {
	        if (r = UNITS[this.numerator[j]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);

	          if (n >= 0) {
	            vector[n] = vector[n] + 1;
	          }
	        }
	      }

	      for (var k = 0; k < this.denominator.length; k++) {
	        if (r = UNITS[this.denominator[k]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);

	          if (n >= 0) {
	            vector[n] = vector[n] - 1;
	          }
	        }
	      }

	      return vector;
	    }

	    var SIGN = "[+-]";
	    var INTEGER = "\\d+";
	    var SIGNED_INTEGER = SIGN + "?" + INTEGER;
	    var FRACTION = "\\." + INTEGER;
	    var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
	    var EXPONENT = "[Ee]" + SIGNED_INTEGER;
	    var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
	    var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
	    var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
	    var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
	    var POWER_OP = "\\^|\\*{2}"; // Allow unit powers representing scalar, length, area, volume; 4 is for some
	    // special case representations in SI base units.

	    var SAFE_POWER = "[01234]";
	    var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
	    var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");
	    /* parse a string into a unit object.
	     * Typical formats like :
	     * "5.6 kg*m/s^2"
	     * "5.6 kg*m*s^-2"
	     * "5.6 kilogram*meter*second^-2"
	     * "2.2 kPa"
	     * "37 degC"
	     * "1"  -- creates a unitless constant with value 1
	     * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
	     * 6'4"  -- recognized as 6 feet + 4 inches
	     * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
	     */

	    function parse(val) {
	      if (!isString(val)) {
	        val = val.toString();
	      }

	      val = val.trim();
	      var result = QTY_STRING_REGEX.exec(val);

	      if (!result) {
	        throw new QtyError(val + ": Quantity not recognized");
	      }

	      var scalarMatch = result[1];

	      if (scalarMatch) {
	        // Allow whitespaces between sign and scalar for loose parsing
	        scalarMatch = scalarMatch.replace(/\s/g, "");
	        this.scalar = parseFloat(scalarMatch);
	      } else {
	        this.scalar = 1;
	      }

	      var top = result[2];
	      var bottom = result[3];
	      var n, x, nx; // TODO DRY me

	      while (result = TOP_REGEX.exec(top)) {
	        n = parseFloat(result[2]);

	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        } // Disallow unrecognized unit even if exponent is 0


	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }

	        x = result[1] + " ";
	        nx = "";

	        for (var i = 0; i < Math.abs(n); i++) {
	          nx += x;
	        }

	        if (n >= 0) {
	          top = top.replace(result[0], nx);
	        } else {
	          bottom = bottom ? bottom + nx : nx;
	          top = top.replace(result[0], "");
	        }
	      }

	      while (result = BOTTOM_REGEX.exec(bottom)) {
	        n = parseFloat(result[2]);

	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        } // Disallow unrecognized unit even if exponent is 0


	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }

	        x = result[1] + " ";
	        nx = "";

	        for (var j = 0; j < n; j++) {
	          nx += x;
	        }

	        bottom = bottom.replace(result[0], nx);
	      }

	      if (top) {
	        this.numerator = parseUnits(top.trim());
	      }

	      if (bottom) {
	        this.denominator = parseUnits(bottom.trim());
	      }
	    }

	    var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    /*
	     * Minimal boundary regex to support units with Unicode characters
	     * \b only works for ASCII
	     */

	    var BOUNDARY_REGEX = "\\b|$";
	    var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
	    var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
	    var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences

	    var parsedUnitsCache = {};
	    /**
	     * Parses and converts units string to normalized unit array.
	     * Result is cached to speed up next calls.
	     *
	     * @param {string} units Units string
	     * @returns {string[]} Array of normalized units
	     *
	     * @example
	     * // Returns ["<second>", "<meter>", "<second>"]
	     * parseUnits("s m s");
	     *
	     */

	    function parseUnits(units) {
	      var cached = parsedUnitsCache[units];

	      if (cached) {
	        return cached;
	      }

	      var unitMatch,
	          normalizedUnits = []; // Scan

	      if (!UNIT_TEST_REGEX.test(units)) {
	        throw new QtyError("Unit not recognized");
	      }

	      while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
	        normalizedUnits.push(unitMatch.slice(1));
	      }

	      normalizedUnits = normalizedUnits.map(function (item) {
	        return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
	      }); // Flatten and remove null elements

	      normalizedUnits = normalizedUnits.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      normalizedUnits = normalizedUnits.filter(function (item) {
	        return item;
	      });
	      parsedUnitsCache[units] = normalizedUnits;
	      return normalizedUnits;
	    }
	    /**
	     * Parses a string as a quantity
	     * @param {string} value - quantity as text
	     * @throws if value is not a string
	     * @returns {Qty|null} Parsed quantity or null if unrecognized
	     */


	    function globalParse(value) {
	      if (!isString(value)) {
	        throw new QtyError("Argument should be a string");
	      }

	      try {
	        return this(value);
	      } catch (e) {
	        return null;
	      }
	    }
	    /**
	     * Tests if a value is a Qty instance
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a Qty instance, false otherwise
	     */


	    function isQty(value) {
	      return value instanceof Qty;
	    }

	    function Qty(initValue, initUnits) {
	      assertValidConstructorArgs.apply(null, arguments);

	      if (!isQty(this)) {
	        return new Qty(initValue, initUnits);
	      }

	      this.scalar = null;
	      this.baseScalar = null;
	      this.signature = null;
	      this._conversionCache = {};
	      this.numerator = UNITY_ARRAY;
	      this.denominator = UNITY_ARRAY;

	      if (isDefinitionObject(initValue)) {
	        this.scalar = initValue.scalar;
	        this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
	        this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
	      } else if (initUnits) {
	        parse.call(this, initUnits);
	        this.scalar = initValue;
	      } else {
	        parse.call(this, initValue);
	      } // math with temperatures is very limited


	      if (this.denominator.join("*").indexOf("temp") >= 0) {
	        throw new QtyError("Cannot divide with temperatures");
	      }

	      if (this.numerator.join("*").indexOf("temp") >= 0) {
	        if (this.numerator.length > 1) {
	          throw new QtyError("Cannot multiply by temperatures");
	        }

	        if (!compareArray(this.denominator, UNITY_ARRAY)) {
	          throw new QtyError("Cannot divide with temperatures");
	        }
	      }

	      this.initValue = initValue;
	      updateBaseScalar.call(this);

	      if (this.isTemperature() && this.baseScalar < 0) {
	        throw new QtyError("Temperatures must not be less than absolute zero");
	      }
	    }

	    Qty.prototype = {
	      // Properly set up constructor
	      constructor: Qty
	    };
	    /**
	     * Asserts constructor arguments are valid
	     *
	     * @param {*} value - Value to test
	     * @param {string} [units] - Optional units when value is passed as a number
	     *
	     * @returns {void}
	     * @throws {QtyError} if constructor arguments are invalid
	     */

	    function assertValidConstructorArgs(value, units) {
	      if (units) {
	        if (!(isNumber(value) && isString(units))) {
	          throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
	        }
	      } else {
	        if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
	          throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
	        }
	      }
	    }
	    /**
	     * Tests if a value is a Qty definition object
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a definition object, false otherwise
	     */


	    function isDefinitionObject(value) {
	      return value && typeof value === "object" && value.hasOwnProperty("scalar");
	    }

	    function updateBaseScalar() {
	      if (this.baseScalar) {
	        return this.baseScalar;
	      }

	      if (this.isBase()) {
	        this.baseScalar = this.scalar;
	        this.signature = unitSignature.call(this);
	      } else {
	        var base = this.toBase();
	        this.baseScalar = base.scalar;
	        this.signature = base.signature;
	      }
	    }

	    var KINDS = {
	      "-312078": "elastance",
	      "-312058": "resistance",
	      "-312038": "inductance",
	      "-152058": "potential",
	      "-152040": "magnetism",
	      "-152038": "magnetism",
	      "-7997": "specific_volume",
	      "-79": "snap",
	      "-59": "jolt",
	      "-39": "acceleration",
	      "-38": "radiation",
	      "-20": "frequency",
	      "-19": "speed",
	      "-18": "viscosity",
	      "-17": "volumetric_flow",
	      "-1": "wavenumber",
	      "0": "unitless",
	      "1": "length",
	      "2": "area",
	      "3": "volume",
	      "20": "time",
	      "400": "temperature",
	      "7941": "yank",
	      "7942": "power",
	      "7959": "pressure",
	      "7961": "force",
	      "7962": "energy",
	      "7979": "viscosity",
	      "7981": "momentum",
	      "7982": "angular_momentum",
	      "7997": "density",
	      "7998": "area_density",
	      "8000": "mass",
	      "152020": "radiation_exposure",
	      "159999": "magnetism",
	      "160000": "current",
	      "160020": "charge",
	      "312058": "conductance",
	      "312078": "capacitance",
	      "3199980": "activity",
	      "3199997": "molar_concentration",
	      "3200000": "substance",
	      "63999998": "illuminance",
	      "64000000": "luminous_power",
	      "1280000000": "currency",
	      "25599999980": "information_rate",
	      "25600000000": "information",
	      "511999999980": "angular_velocity",
	      "512000000000": "angle"
	    };
	    /**
	     * Returns the list of available well-known kinds of units, e.g.
	     * "radiation" or "length".
	     *
	     * @returns {string[]} names of kinds of units
	     */

	    function getKinds() {
	      return uniq(Object.keys(KINDS).map(function (knownSignature) {
	        return KINDS[knownSignature];
	      }));
	    }

	    Qty.prototype.kind = function () {
	      return KINDS[this.signature.toString()];
	    };

	    assign(Qty.prototype, {
	      isDegrees: function () {
	        // signature may not have been calculated yet
	        return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
	      },
	      isTemperature: function () {
	        return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
	      }
	    });

	    function subtractTemperatures(lhs, rhs) {
	      var lhsUnits = lhs.units();
	      var rhsConverted = rhs.to(lhsUnits);
	      var dstDegrees = Qty(getDegreeUnits(lhsUnits));
	      return Qty({
	        "scalar": lhs.scalar - rhsConverted.scalar,
	        "numerator": dstDegrees.numerator,
	        "denominator": dstDegrees.denominator
	      });
	    }

	    function subtractTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar - tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }

	    function addTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar + tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }

	    function getDegreeUnits(units) {
	      if (units === "tempK") {
	        return "degK";
	      } else if (units === "tempC") {
	        return "degC";
	      } else if (units === "tempF") {
	        return "degF";
	      } else if (units === "tempR") {
	        return "degR";
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	    }

	    function toDegrees(src, dst) {
	      var srcDegK = toDegK(src);
	      var dstUnits = dst.units();
	      var dstScalar;

	      if (dstUnits === "degK") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degC") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degF") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else if (dstUnits === "degR") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
	      }

	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }

	    function toDegK(qty) {
	      var units = qty.units();
	      var q;

	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar;
	      } else if (units === "tempF") {
	        q = qty.scalar * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }

	      return Qty({
	        "scalar": q,
	        "numerator": ["<kelvin>"],
	        "denominator": UNITY_ARRAY
	      });
	    }

	    function toTemp(src, dst) {
	      var dstUnits = dst.units();
	      var dstScalar;

	      if (dstUnits === "tempK") {
	        dstScalar = src.baseScalar;
	      } else if (dstUnits === "tempC") {
	        dstScalar = src.baseScalar - 273.15;
	      } else if (dstUnits === "tempF") {
	        dstScalar = src.baseScalar * 9 / 5 - 459.67;
	      } else if (dstUnits === "tempR") {
	        dstScalar = src.baseScalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
	      }

	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }

	    function toTempK(qty) {
	      var units = qty.units();
	      var q;

	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar + 273.15;
	      } else if (units === "tempF") {
	        q = (qty.scalar + 459.67) * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }

	      return Qty({
	        "scalar": q,
	        "numerator": ["<temp-K>"],
	        "denominator": UNITY_ARRAY
	      });
	    }

	    assign(Qty.prototype, {
	      /**
	       * Converts to other compatible units.
	       * Instance's converted quantities are cached for faster subsequent calls.
	       *
	       * @param {(string|Qty)} other - Target units as string or retrieved from
	       *                               other Qty instance (scalar is ignored)
	       *
	       * @returns {Qty} New converted Qty instance with target units
	       *
	       * @throws {QtyError} if target units are incompatible
	       *
	       * @example
	       * var weight = Qty("25 kg");
	       * weight.to("lb"); // => Qty("55.11556554621939 lbs");
	       * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
	       */
	      to: function (other) {
	        var cached, target;

	        if (other === undefined || other === null) {
	          return this;
	        }

	        if (!isString(other)) {
	          return this.to(other.units());
	        }

	        cached = this._conversionCache[other];

	        if (cached) {
	          return cached;
	        } // Instantiating target to normalize units


	        target = Qty(other);

	        if (target.units() === this.units()) {
	          return this;
	        }

	        if (!this.isCompatible(target)) {
	          if (this.isInverse(target)) {
	            target = this.inverse().to(other);
	          } else {
	            throwIncompatibleUnits(this.units(), target.units());
	          }
	        } else {
	          if (target.isTemperature()) {
	            target = toTemp(this, target);
	          } else if (target.isDegrees()) {
	            target = toDegrees(this, target);
	          } else {
	            var q = divSafe(this.baseScalar, target.baseScalar);
	            target = Qty({
	              "scalar": q,
	              "numerator": target.numerator,
	              "denominator": target.denominator
	            });
	          }
	        }

	        this._conversionCache[other] = target;
	        return target;
	      },
	      // convert to base SI units
	      // results of the conversion are cached so subsequent calls to this will be fast
	      toBase: function () {
	        if (this.isBase()) {
	          return this;
	        }

	        if (this.isTemperature()) {
	          return toTempK(this);
	        }

	        var cached = baseUnitCache[this.units()];

	        if (!cached) {
	          cached = toBaseUnits(this.numerator, this.denominator);
	          baseUnitCache[this.units()] = cached;
	        }

	        return cached.mul(this.scalar);
	      },
	      // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
	      toFloat: function () {
	        if (this.isUnitless()) {
	          return this.scalar;
	        }

	        throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
	      },

	      /**
	       * Returns the nearest multiple of quantity passed as
	       * precision
	       *
	       * @param {(Qty|string|number)} precQuantity - Quantity, string formated
	       *   quantity or number as expected precision
	       *
	       * @returns {Qty} Nearest multiple of precQuantity
	       *
	       * @example
	       * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
	       * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
	       * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
	       * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
	       *
	       */
	      toPrec: function (precQuantity) {
	        if (isString(precQuantity)) {
	          precQuantity = Qty(precQuantity);
	        }

	        if (isNumber(precQuantity)) {
	          precQuantity = Qty(precQuantity + " " + this.units());
	        }

	        if (!this.isUnitless()) {
	          precQuantity = precQuantity.to(this.units());
	        } else if (!precQuantity.isUnitless()) {
	          throwIncompatibleUnits(this.units(), precQuantity.units());
	        }

	        if (precQuantity.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }

	        var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
	        return Qty(precRoundedResult + this.units());
	      }
	    });
	    /**
	     * Configures and returns a fast function to convert
	     * Number values from units to others.
	     * Useful to efficiently convert large array of values
	     * with same units into others with iterative methods.
	     * Does not take care of rounding issues.
	     *
	     * @param {string} srcUnits Units of values to convert
	     * @param {string} dstUnits Units to convert to
	     *
	     * @returns {Function} Converting function accepting Number value
	     *   and returning converted value
	     *
	     * @throws "Incompatible units" if units are incompatible
	     *
	     * @example
	     * // Converting large array of numbers with the same units
	     * // into other units
	     * var converter = Qty.swiftConverter("m/h", "ft/s");
	     * var convertedSerie = largeSerie.map(converter);
	     *
	     */

	    function swiftConverter(srcUnits, dstUnits) {
	      var srcQty = Qty(srcUnits);
	      var dstQty = Qty(dstUnits);

	      if (srcQty.eq(dstQty)) {
	        return identity;
	      }

	      var convert;

	      if (!srcQty.isTemperature()) {
	        convert = function (value) {
	          return value * srcQty.baseScalar / dstQty.baseScalar;
	        };
	      } else {
	        convert = function (value) {
	          // TODO Not optimized
	          return srcQty.mul(value).to(dstQty).scalar;
	        };
	      }

	      return function converter(value) {
	        var i, length, result;

	        if (!Array.isArray(value)) {
	          return convert(value);
	        } else {
	          length = value.length;
	          result = [];

	          for (i = 0; i < length; i++) {
	            result.push(convert(value[i]));
	          }

	          return result;
	        }
	      };
	    }

	    var baseUnitCache = {};

	    function toBaseUnits(numerator, denominator) {
	      var num = [];
	      var den = [];
	      var q = 1;
	      var unit;

	      for (var i = 0; i < numerator.length; i++) {
	        unit = numerator[i];

	        if (PREFIX_VALUES[unit]) {
	          // workaround to fix
	          // 0.1 * 0.1 => 0.010000000000000002
	          q = mulSafe(q, PREFIX_VALUES[unit]);
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q *= UNIT_VALUES[unit].scalar;

	            if (UNIT_VALUES[unit].numerator) {
	              num.push(UNIT_VALUES[unit].numerator);
	            }

	            if (UNIT_VALUES[unit].denominator) {
	              den.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      }

	      for (var j = 0; j < denominator.length; j++) {
	        unit = denominator[j];

	        if (PREFIX_VALUES[unit]) {
	          q /= PREFIX_VALUES[unit];
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q /= UNIT_VALUES[unit].scalar;

	            if (UNIT_VALUES[unit].numerator) {
	              den.push(UNIT_VALUES[unit].numerator);
	            }

	            if (UNIT_VALUES[unit].denominator) {
	              num.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      } // Flatten


	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return Qty({
	        "scalar": q,
	        "numerator": num,
	        "denominator": den
	      });
	    }

	    Qty.parse = globalParse;
	    Qty.getUnits = getUnits;
	    Qty.getAliases = getAliases;
	    Qty.mulSafe = mulSafe;
	    Qty.divSafe = divSafe;
	    Qty.getKinds = getKinds;
	    Qty.swiftConverter = swiftConverter;
	    Qty.Error = QtyError;
	    assign(Qty.prototype, {
	      // Returns new instance with units of this
	      add: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }

	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }

	        if (this.isTemperature() && other.isTemperature()) {
	          throw new QtyError("Cannot add two temperatures");
	        } else if (this.isTemperature()) {
	          return addTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          return addTempDegrees(other, this);
	        }

	        return Qty({
	          "scalar": this.scalar + other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      sub: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }

	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }

	        if (this.isTemperature() && other.isTemperature()) {
	          return subtractTemperatures(this, other);
	        } else if (this.isTemperature()) {
	          return subtractTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          throw new QtyError("Cannot subtract a temperature from a differential degree unit");
	        }

	        return Qty({
	          "scalar": this.scalar - other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      mul: function (other) {
	        if (isNumber(other)) {
	          return Qty({
	            "scalar": mulSafe(this.scalar, other),
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }

	        if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
	          throw new QtyError("Cannot multiply by temperatures");
	        } // Quantities should be multiplied with same units if compatible, with base units else


	        var op1 = this;
	        var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }

	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      div: function (other) {
	        if (isNumber(other)) {
	          if (other === 0) {
	            throw new QtyError("Divide by zero");
	          }

	          return Qty({
	            "scalar": this.scalar / other,
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }

	        if (other.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }

	        if (other.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        } else if (this.isTemperature() && !other.isUnitless()) {
	          throw new QtyError("Cannot divide with temperatures");
	        } // Quantities should be multiplied with same units if compatible, with base units else


	        var op1 = this;
	        var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }

	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      // Returns a Qty that is the inverse of this Qty,
	      inverse: function () {
	        if (this.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        }

	        if (this.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }

	        return Qty({
	          "scalar": 1 / this.scalar,
	          "numerator": this.denominator,
	          "denominator": this.numerator
	        });
	      }
	    });

	    function cleanTerms(num1, den1, num2, den2) {
	      function notUnity(val) {
	        return val !== UNITY;
	      }

	      num1 = num1.filter(notUnity);
	      num2 = num2.filter(notUnity);
	      den1 = den1.filter(notUnity);
	      den2 = den2.filter(notUnity);
	      var combined = {};

	      function combineTerms(terms, direction) {
	        var k;
	        var prefix;
	        var prefixValue;

	        for (var i = 0; i < terms.length; i++) {
	          if (PREFIX_VALUES[terms[i]]) {
	            k = terms[i + 1];
	            prefix = terms[i];
	            prefixValue = PREFIX_VALUES[prefix];
	            i++;
	          } else {
	            k = terms[i];
	            prefix = null;
	            prefixValue = 1;
	          }

	          if (k && k !== UNITY) {
	            if (combined[k]) {
	              combined[k][0] += direction;
	              var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
	              combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
	            } else {
	              combined[k] = [direction, k, prefix, 1, 1];
	            }
	          }
	        }
	      }

	      combineTerms(num1, 1);
	      combineTerms(den1, -1);
	      combineTerms(num2, 1);
	      combineTerms(den2, -1);
	      var num = [];
	      var den = [];
	      var scale = 1;

	      for (var prop in combined) {
	        if (combined.hasOwnProperty(prop)) {
	          var item = combined[prop];
	          var n;

	          if (item[0] > 0) {
	            for (n = 0; n < item[0]; n++) {
	              num.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          } else if (item[0] < 0) {
	            for (n = 0; n < -item[0]; n++) {
	              den.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          }

	          scale *= divSafe(item[3], item[4]);
	        }
	      }

	      if (num.length === 0) {
	        num = UNITY_ARRAY;
	      }

	      if (den.length === 0) {
	        den = UNITY_ARRAY;
	      } // Flatten


	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return [num, den, scale];
	    }

	    assign(Qty.prototype, {
	      eq: function (other) {
	        return this.compareTo(other) === 0;
	      },
	      lt: function (other) {
	        return this.compareTo(other) === -1;
	      },
	      lte: function (other) {
	        return this.eq(other) || this.lt(other);
	      },
	      gt: function (other) {
	        return this.compareTo(other) === 1;
	      },
	      gte: function (other) {
	        return this.eq(other) || this.gt(other);
	      },
	      // Compare two Qty objects. Throws an exception if they are not of compatible types.
	      // Comparisons are done based on the value of the quantity in base SI units.
	      //
	      // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
	      //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
	      //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
	      //
	      //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
	      //     Qty("10S").inverse().compareTo("10ohm") == -1
	      //     Qty("10ohm").inverse().compareTo("10S") == -1
	      //
	      //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
	      compareTo: function (other) {
	        if (isString(other)) {
	          return this.compareTo(Qty(other));
	        }

	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }

	        if (this.baseScalar < other.baseScalar) {
	          return -1;
	        } else if (this.baseScalar === other.baseScalar) {
	          return 0;
	        } else if (this.baseScalar > other.baseScalar) {
	          return 1;
	        }
	      },
	      // Return true if quantities and units match
	      // Unit("100 cm").same(Unit("100 cm"))  # => true
	      // Unit("100 cm").same(Unit("1 m"))     # => false
	      same: function (other) {
	        return this.scalar === other.scalar && this.units() === other.units();
	      }
	    });
	    assign(Qty.prototype, {
	      // returns true if no associated units
	      // false, even if the units are "unitless" like 'radians, each, etc'
	      isUnitless: function () {
	        return [this.numerator, this.denominator].every(function (item) {
	          return compareArray(item, UNITY_ARRAY);
	        });
	      },

	      /*
	      check to see if units are compatible, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isCompatible: function (other) {
	        if (isString(other)) {
	          return this.isCompatible(Qty(other));
	        }

	        if (!isQty(other)) {
	          return false;
	        }

	        if (other.signature !== undefined) {
	          return this.signature === other.signature;
	        } else {
	          return false;
	        }
	      },

	      /*
	      check to see if units are inverse of each other, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isInverse: function (other) {
	        return this.inverse().isCompatible(other);
	      },
	      // Returns 'true' if the Unit is represented in base units
	      isBase: function () {
	        if (this._isBase !== undefined) {
	          return this._isBase;
	        }

	        if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
	          this._isBase = true;
	          return this._isBase;
	        }

	        this.numerator.concat(this.denominator).forEach(function (item) {
	          if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
	            this._isBase = false;
	          }
	        }, this);

	        if (this._isBase === false) {
	          return this._isBase;
	        }

	        this._isBase = true;
	        return this._isBase;
	      }
	    });

	    function NestedMap() {}

	    NestedMap.prototype.get = function (keys) {
	      // Allows to pass key1, key2, ... instead of [key1, key2, ...]
	      if (arguments.length > 1) {
	        // Slower with Firefox but faster with Chrome than
	        // Array.prototype.slice.call(arguments)
	        // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
	        keys = Array.apply(null, arguments);
	      }

	      return keys.reduce(function (map, key, index) {
	        if (map) {
	          var childMap = map[key];

	          if (index === keys.length - 1) {
	            return childMap ? childMap.data : undefined;
	          } else {
	            return childMap;
	          }
	        }
	      }, this);
	    };

	    NestedMap.prototype.set = function (keys, value) {
	      if (arguments.length > 2) {
	        keys = Array.prototype.slice.call(arguments, 0, -1);
	        value = arguments[arguments.length - 1];
	      }

	      return keys.reduce(function (map, key, index) {
	        var childMap = map[key];

	        if (childMap === undefined) {
	          childMap = map[key] = {};
	        }

	        if (index === keys.length - 1) {
	          childMap.data = value;
	          return value;
	        } else {
	          return childMap;
	        }
	      }, this);
	    };
	    /**
	     * Default formatter
	     *
	     * @param {number} scalar - scalar value
	     * @param {string} units - units as string
	     *
	     * @returns {string} formatted result
	     */


	    function defaultFormatter(scalar, units) {
	      return (scalar + " " + units).trim();
	    }
	    /**
	     *
	     * Configurable Qty default formatter
	     *
	     * @type {function}
	     *
	     * @param {number} scalar
	     * @param {string} units
	     *
	     * @returns {string} formatted result
	     */


	    Qty.formatter = defaultFormatter;
	    assign(Qty.prototype, {
	      // returns the 'unit' part of the Unit object without the scalar
	      units: function () {
	        if (this._units !== undefined) {
	          return this._units;
	        }

	        var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
	        var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);

	        if (numIsUnity && denIsUnity) {
	          this._units = "";
	          return this._units;
	        }

	        var numUnits = stringifyUnits(this.numerator);
	        var denUnits = stringifyUnits(this.denominator);
	        this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
	        return this._units;
	      },

	      /**
	       * Stringifies the quantity
	       * Deprecation notice: only units parameter is supported.
	       *
	       * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
	       *                              target units if string,
	       *                              max number of decimals if number,
	       *                              passed to #toPrec before converting if Qty
	       *
	       * @param {number=} maxDecimals - Maximum number of decimals of
	       *                                formatted output
	       *
	       * @returns {string} reparseable quantity as string
	       */
	      toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
	        var targetUnits;

	        if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = this.units();
	          maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
	          return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
	        }

	        var out = this.to(targetUnits);
	        var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
	        out = (outScalar + " " + out.units()).trim();
	        return out;
	      },

	      /**
	       * Format the quantity according to optional passed target units
	       * and formatter
	       *
	       * @param {string} [targetUnits=current units] -
	       *                 optional units to convert to before formatting
	       *
	       * @param {function} [formatter=Qty.formatter] -
	       *                   delegates formatting to formatter callback.
	       *                   formatter is called back with two parameters (scalar, units)
	       *                   and should return formatted result.
	       *                   If unspecified, formatting is delegated to default formatter
	       *                   set to Qty.formatter
	       *
	       * @example
	       * var roundingAndLocalizingFormatter = function(scalar, units) {
	       *   // localize or limit scalar to n max decimals for instance
	       *   // return formatted result
	       * };
	       * var qty = Qty('1.1234 m');
	       * qty.format(); // same units, default formatter => "1.234 m"
	       * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
	       * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
	       * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
	       *
	       * @returns {string} quantity as string
	       */
	      format: function (targetUnits, formatter) {
	        if (arguments.length === 1) {
	          if (typeof targetUnits === "function") {
	            formatter = targetUnits;
	            targetUnits = undefined;
	          }
	        }

	        formatter = formatter || Qty.formatter;
	        var targetQty = this.to(targetUnits);
	        return formatter.call(this, targetQty.scalar, targetQty.units());
	      }
	    });
	    var stringifiedUnitsCache = new NestedMap();
	    /**
	     * Returns a string representing a normalized unit array
	     *
	     * @param {string[]} units Normalized unit array
	     * @returns {string} String representing passed normalized unit array and
	     *   suitable for output
	     *
	     */

	    function stringifyUnits(units) {
	      var stringified = stringifiedUnitsCache.get(units);

	      if (stringified) {
	        return stringified;
	      }

	      var isUnity = compareArray(units, UNITY_ARRAY);

	      if (isUnity) {
	        stringified = "1";
	      } else {
	        stringified = simplify(getOutputNames(units)).join("*");
	      } // Cache result


	      stringifiedUnitsCache.set(units, stringified);
	      return stringified;
	    }

	    function getOutputNames(units) {
	      var unitNames = [],
	          token,
	          tokenNext;

	      for (var i = 0; i < units.length; i++) {
	        token = units[i];
	        tokenNext = units[i + 1];

	        if (PREFIX_VALUES[token]) {
	          unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
	          i++;
	        } else {
	          unitNames.push(OUTPUT_MAP[token]);
	        }
	      }

	      return unitNames;
	    }

	    function simplify(units) {
	      // this turns ['s','m','s'] into ['s2','m']
	      var unitCounts = units.reduce(function (acc, unit) {
	        var unitCounter = acc[unit];

	        if (!unitCounter) {
	          acc.push(unitCounter = acc[unit] = [unit, 0]);
	        }

	        unitCounter[1]++;
	        return acc;
	      }, []);
	      return unitCounts.map(function (unitCount) {
	        return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
	      });
	    }

	    Qty.version = "1.7.6";
	    return Qty;
	  });
	})(quantities);

	var Qty = quantities.exports;

	function convertUnit(array, fromUnit, toUnit) {
	  fromUnit = normalize(fromUnit);
	  toUnit = normalize(toUnit);
	  if (fromUnit === toUnit) return array;
	  const convert = Qty.swiftConverter(fromUnit, toUnit); // Configures converter
	  //@ts-expect-error convert does not allowed typed array but it works

	  return convert(array);
	}

	function normalize(unit) {
	  unit = unit.replace(/°C/g, 'tempC');
	  unit = unit.replace(/°F/g, 'tempF');
	  unit = unit.replace(/(^|\W)K(\W|$)/g, '$1tempK$2');
	  return unit;
	}

	const testRegExp = /^\/((?:\\\/|[^/])+)\/([migyu]{0,5})?$/;
	function ensureRegexp(string) {
	  if (typeof string !== 'string') return string;
	  const parts = testRegExp.exec(string);

	  if (parts) {
	    try {
	      return new RegExp(parts[1], parts[2]);
	    } catch (err) {
	      return stringToRegexp(string);
	    }
	  } else {
	    return stringToRegexp(string);
	  }
	}

	function stringToRegexp(string) {
	  let flags = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'i';
	  return new RegExp(string.replace(/[[\]\\{}()+*?.$^|]/g, match => `\\${match}`), flags);
	}

	function getConvertedVariable(variable, newUnits) {
	  const data = variable.units !== undefined && variable.units !== newUnits // would be nice if convertUnit would allow typedArray
	  ? convertUnit(Array.from(variable.data), variable.units, newUnits) : variable.data;
	  return {
	    units: newUnits,
	    label: variable.label.replace(`[${variable.units || ''}]`, `[${newUnits}]`),
	    data: data || [],
	    min: data ? min(data) : undefined,
	    max: data ? max(data) : undefined,
	    isMonotone: xIsMonotone(data)
	  };
	}

	/* eslint-disable @typescript-eslint/no-dynamic-delete */
	/**
	 * Retrieve the spectrum with only X/Y data that match all the selectors
	 * If more than one variable match the selector the 'x' or 'y' variable will be
	 * taken
	 */

	function getXYSpectrum() {
	  let spectra = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let selector = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (spectra.length < 1) return;
	  let {
	    dataType,
	    title,
	    xUnits,
	    yUnits,
	    variables,
	    xVariable = 'x',
	    yVariable = 'y',
	    units,
	    labels,
	    xLabel,
	    yLabel,
	    meta,
	    index
	  } = selector;

	  if (index !== undefined) {
	    return spectra[index];
	  }

	  if (dataType) {
	    dataType = ensureRegexp(dataType);
	  }

	  if (title) {
	    title = ensureRegexp(title);
	  }

	  if (units && !xUnits && !yUnits) [yUnits, xUnits] = units.split(/\s*vs\s*/);

	  if (labels && !xLabel && !yLabel) {
	    [yLabel, xLabel] = labels.split(/\s*vs\s*/);
	  }

	  if (variables) {
	    const parts = variables.split(/\s*vs\s*/);

	    if (parts.length === 2) {
	      xVariable = parts[1];
	      yVariable = parts[0];
	    }
	  }

	  if (xLabel) xLabel = ensureRegexp(xLabel);
	  if (yLabel) yLabel = ensureRegexp(yLabel);

	  for (let spectrum of spectra) {
	    let variableNames = Object.keys(spectrum.variables);
	    if (!(variableNames.length > 1)) continue; // we filter on general spectrum information

	    if (dataType) {
	      if (!spectrum.dataType || !dataType.exec(spectrum.dataType)) {
	        continue;
	      }
	    }

	    if (title) {
	      if (!spectrum.title || !title.exec(spectrum.title)) {
	        continue;
	      }
	    }

	    if (meta && typeof meta === 'object') {
	      if (!spectrum.meta) continue;

	      for (let key in spectrum.meta) {
	        if (!spectrum.meta[key]) continue;
	        let value = ensureRegexp(spectrum.meta[key]);
	        if (!value.exec(spectrum.meta[key])) continue;
	      }
	    }

	    let x = getPossibleVariable(spectrum.variables, {
	      units: xUnits,
	      label: xLabel,
	      variableName: xVariable
	    });
	    let y = getPossibleVariable(spectrum.variables, {
	      units: yUnits,
	      label: yLabel,
	      variableName: yVariable
	    });

	    if (x && y) {
	      return {
	        title: spectrum.title,
	        dataType: spectrum.dataType,
	        meta: spectrum.meta,
	        variables: {
	          x,
	          y
	        }
	      };
	    }
	  }

	  return;
	}

	function getPossibleVariable(variables) {
	  let selector = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  var _a, _b;

	  const {
	    units,
	    label,
	    variableName
	  } = selector;
	  let possible = { ...variables
	  };
	  let key;

	  if (units !== undefined) {
	    for (key in possible) {
	      const variable = variables[key];
	      let convertibleUnits = true;

	      try {
	        convertUnit(1, (variable === null || variable === void 0 ? void 0 : variable.units) || '', units);
	      } catch (e) {
	        convertibleUnits = false;
	      }

	      if (convertibleUnits && variable) {
	        possible[key] = getConvertedVariable(variable, units);
	      } else {
	        delete possible[key];
	      }
	    }
	  }

	  if (label !== undefined) {
	    const regexpLabel = ensureRegexp(label);

	    for (key in possible) {
	      if (!regexpLabel.exec((_b = (_a = variables[key]) === null || _a === void 0 ? void 0 : _a.label) !== null && _b !== void 0 ? _b : '')) {
	        delete possible[key];
	      }
	    }
	  }

	  if (variableName !== undefined) {
	    if (possible[variableName]) return possible[variableName];
	    const upper = variableName.toUpperCase();

	    if (Object.prototype.hasOwnProperty.call(possible, upper)) {
	      return possible[upper];
	    }

	    const lower = variableName.toLowerCase();

	    if (Object.prototype.hasOwnProperty.call(possible, lower)) {
	      return possible[lower];
	    }
	  }

	  const possibleFiltered = Object.values(possible).filter(val => val !== undefined);

	  if (possibleFiltered.length > 0) {
	    return possibleFiltered[0];
	  }
	}

	/**
	 * Class allowing to store and manipulate an analysis.
	 * An analysis may contain one or more spectra that can be selected
	 * based on their units
	 */

	class Analysis {
	  constructor() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    this.id = options.id || Math.random().toString(36).substring(2, 10);
	    this.label = options.label || this.id;
	    this.spectrumCallback = options.spectrumCallback;
	    this.spectra = [];
	    this.cache = {};
	  }
	  /**
	   * Add a spectrum in the internal spectra variable
	   */


	  pushSpectrum(variables) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	    this.spectra.push(standardizeData(variables, options, {
	      spectrumCallback: this.spectrumCallback
	    }));
	    this.cache = {};
	  }
	  /**
	   * Retrieve a Spectrum based on x/y units
	   */


	  getXYSpectrum() {
	    let selector = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    let id = JSON.stringify(selector);

	    if (!this.cache[id]) {
	      this.cache[id] = getXYSpectrum(this.spectra, selector);
	    }

	    return this.cache[id];
	  }
	  /**
	   * Retrieve a xy object
	   * @param selector.units Units separated by vs like for example "g vs °C"
	   * @param selector.xUnits if undefined takes the first variable
	   * @param selector.yUnits if undefined takes the second variable
	   */


	  getXY() {
	    let selector = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    let spectrum = this.getXYSpectrum(selector);
	    if (!spectrum) return undefined;
	    return {
	      x: spectrum.variables.x.data,
	      y: spectrum.variables.y.data
	    };
	  }
	  /**
	   * Return the data object for specific x/y units with possibly some
	   * normalization options
	   * @param options.selector.xUnits // if undefined takes the first variable
	   * @param options.selector.yUnits // if undefined takes the second variable
	   */


	  getNormalizedSpectrum() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      normalization,
	      selector
	    } = options;
	    const spectrum = this.getXYSpectrum(selector);
	    if (!spectrum) return undefined;
	    return getNormalizedSpectrum(spectrum, normalization);
	  }
	  /**
	   * Returns the first spectrum. This method could be improved in the future
	   * @returns
	   */


	  getSpectrum() {
	    return this.spectra[0];
	  }
	  /**
	   * Returns the xLabel
	   * @param selector.xUnits // if undefined takes the first variable
	   * @param selector.yUnits // if undefined takes the second variable
	   */


	  getXLabel(selector) {
	    var _a;

	    return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.x.label;
	  }
	  /**
	   * Returns the yLabel
	   * @param selector.xUnits // if undefined takes the first variable
	   * @param selector.yUnits // if undefined takes the second variable
	   */


	  getYLabel(selector) {
	    var _a;

	    return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.y.label;
	  }

	}
	/**
	 * Internal function that ensure the order of x / y array
	 */

	function standardizeData(variables, options, analysisOptions) {
	  let {
	    meta = {},
	    dataType = '',
	    title = ''
	  } = options;
	  const {
	    spectrumCallback
	  } = analysisOptions;

	  if (spectrumCallback) {
	    spectrumCallback(variables);
	  }

	  let xVariable = variables.x;
	  let yVariable = variables.y;

	  if (!xVariable || !yVariable) {
	    throw Error('A spectrum must contain at least x and y variables');
	  }

	  if (!isAnyArray(xVariable.data) || !isAnyArray(yVariable.data)) {
	    throw Error('x and y variables must contain an array data');
	  }

	  let x = xVariable.data;
	  let reverse = x && x.length > 1 && x[0] > x[x.length - 1];

	  for (let [key, variable] of Object.entries(variables)) {
	    if (reverse) variable.data = variable.data.slice().reverse();
	    variable.label = variable.label || key;

	    if (variable.label.match(/^.*[([](?<units>.*)[)\]].*$/)) {
	      const units = variable.label.replace(/^.*[([](?<units>.*)[)\]].*$/, '$<units>');

	      if (!variable.units || variable.units === units) {
	        variable.units = units;
	        variable.label = variable.label.replace(/[([].*[)\]]/, '').trim();
	      }
	    }

	    variable.min = min(variable.data);
	    variable.max = max(variable.data);
	    variable.isMonotone = xIsMonotone(variable.data);
	  }

	  return {
	    variables,
	    title,
	    dataType,
	    meta
	  };
	}

	/*
	    https://tools.ietf.org/html/rfc3629

	    UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

	    UTF8-1    = %x00-7F

	    UTF8-2    = %xC2-DF UTF8-tail

	    UTF8-3    = %xE0 %xA0-BF UTF8-tail
	                %xE1-EC 2( UTF8-tail )
	                %xED %x80-9F UTF8-tail
	                %xEE-EF 2( UTF8-tail )

	    UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
	                %xF1-F3 3( UTF8-tail )
	                %xF4 %x80-8F 2( UTF8-tail )

	    UTF8-tail = %x80-BF
	*/

	/**
	 * Check if a Node.js Buffer or Uint8Array is UTF-8.
	 */
	function isUtf8(buf) {
	  if (!buf) {
	    return false;
	  }

	  var i = 0;
	  var len = buf.length;

	  while (i < len) {
	    // UTF8-1 = %x00-7F
	    if (buf[i] <= 0x7F) {
	      i++;
	      continue;
	    } // UTF8-2 = %xC2-DF UTF8-tail


	    if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
	      // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
	      if (buf[i + 1] >> 6 === 2) {
	        i += 2;
	        continue;
	      } else {
	        return false;
	      }
	    } // UTF8-3 = %xE0 %xA0-BF UTF8-tail
	    // UTF8-3 = %xED %x80-9F UTF8-tail


	    if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    } // UTF8-3 = %xE1-EC 2( UTF8-tail )
	    // UTF8-3 = %xEE-EF 2( UTF8-tail )


	    if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    } // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
	    //          %xF1-F3 3( UTF8-tail )
	    //          %xF4 %x80-8F 2( UTF8-tail )


	    if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
	      i += 4;
	      continue;
	    }

	    return false;
	  }

	  return true;
	}

	/**
	 * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
	 * @param blob
	 * @param options
	 * @returns
	 */

	function ensureString(blob) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (typeof blob === 'string') {
	    return blob;
	  }

	  if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
	    const {
	      encoding = guessEncoding(blob)
	    } = options;
	    const decoder = new TextDecoder(encoding);
	    return decoder.decode(blob);
	  }

	  throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
	}

	function guessEncoding(blob) {
	  const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

	  if (uint8.length >= 2) {
	    if (uint8[0] === 0xfe && uint8[1] === 0xff) {
	      return 'utf-16be';
	    }

	    if (uint8[0] === 0xff && uint8[1] === 0xfe) {
	      return 'utf-16le';
	    }
	  } //@ts-expect-error an ArrayBuffer is also ok


	  if (!isUtf8(blob)) return 'latin1';
	  return 'utf-8';
	}

	/**
	 * Dynamically type a string
	 * @param {string} value String to dynamically type
	 * @returns {boolean|string|number}
	 */
	function parseString(value) {
	  if (value.length === 4 || value.length === 5) {
	    let lowercase = value.toLowerCase();
	    if (lowercase === 'true') return true;
	    if (lowercase === 'false') return false;
	  }

	  let number = Number(value);

	  if (number === 0 && !value.includes('0')) {
	    return value;
	  }

	  if (!Number.isNaN(number)) return number;
	  return value;
	}

	const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
	function complexChromatogram(result) {
	  let spectra = result.spectra;
	  let length = spectra.length;
	  let chromatogram = {
	    times: new Array(length),
	    series: {
	      ms: {
	        dimension: 2,
	        data: new Array(length)
	      }
	    }
	  };
	  let existingGCMSFields = [];

	  for (let i = 0; i < GC_MS_FIELDS.length; i++) {
	    let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

	    if (spectra[0][label]) {
	      existingGCMSFields.push(label);
	      chromatogram.series[label] = {
	        dimension: 1,
	        data: new Array(length)
	      };
	    }
	  }

	  for (let i = 0; i < length; i++) {
	    let spectrum = spectra[i];
	    chromatogram.times[i] = spectrum.pageValue;

	    for (let j = 0; j < existingGCMSFields.length; j++) {
	      chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
	    }

	    if (spectrum.data) {
	      chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
	    }
	  }

	  result.chromatogram = chromatogram;
	}
	function isMSField(canonicDataLabel) {
	  return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
	}
	function convertMSFieldToLabel(value) {
	  return value.toLowerCase().replace(/[^a-z0-9]/g, '');
	}

	function convertToFloatArray(stringArray) {
	  let floatArray = [];

	  for (let i = 0; i < stringArray.length; i++) {
	    floatArray.push(Number(stringArray[i]));
	  }

	  return floatArray;
	}

	function fastParseXYData(spectrum, value) {
	  // TODO need to deal with result
	  //  console.log(value);
	  // we check if deltaX is defined otherwise we calculate it
	  let yFactor = spectrum.yFactor;
	  let deltaX = spectrum.deltaX;
	  spectrum.isXYdata = true;
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData;
	  let currentX = spectrum.firstX;
	  let currentY = spectrum.firstY; // we skip the first line
	  //

	  let endLine = false;
	  let ascii;
	  let i = 0;

	  for (; i < value.length; i++) {
	    ascii = value.charCodeAt(i);

	    if (ascii === 13 || ascii === 10) {
	      endLine = true;
	    } else {
	      if (endLine) break;
	    }
	  } // we proceed taking the i after the first line


	  let newLine = true;
	  let isDifference = false;
	  let isLastDifference = false;
	  let lastDifference = 0;
	  let isDuplicate = false;
	  let inComment = false;
	  let currentValue = 0; // can be a difference or a duplicate

	  let lastValue = 0; // must be the real last value

	  let isNegative = false;
	  let inValue = false;
	  let skipFirstValue = false;
	  let decimalPosition = 0;

	  for (; i <= value.length; i++) {
	    if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

	    if (inComment) {
	      // we should ignore the text if we are after $$
	      if (ascii === 13 || ascii === 10) {
	        newLine = true;
	        inComment = false;
	      }
	    } else {
	      // when is it a new value ?
	      // when it is not a digit, . or comma
	      // it is a number that is either new or we continue
	      if (ascii <= 57 && ascii >= 48) {
	        // a number
	        inValue = true;

	        if (decimalPosition > 0) {
	          currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
	        } else {
	          currentValue *= 10;
	          currentValue += ascii - 48;
	        }
	      } else if (ascii === 44 || ascii === 46) {
	        // a "," or "."
	        inValue = true;
	        decimalPosition++;
	      } else {
	        if (inValue) {
	          // need to process the previous value
	          if (newLine) {
	            newLine = false; // we don't check the X value
	            // console.log("NEW LINE",isDifference, lastDifference);
	            // if new line and lastDifference, the first value is just a check !
	            // that we don't check ...

	            if (isLastDifference) skipFirstValue = true;
	          } else {
	            // need to deal with duplicate and differences
	            if (skipFirstValue) {
	              skipFirstValue = false;
	            } else {
	              if (isDifference) {
	                lastDifference = isNegative ? 0 - currentValue : currentValue;
	                isLastDifference = true;
	                isDifference = false;
	              } else if (!isDuplicate) {
	                lastValue = isNegative ? 0 - currentValue : currentValue;
	              }

	              let duplicate = isDuplicate ? currentValue - 1 : 1;

	              for (let j = 0; j < duplicate; j++) {
	                if (isLastDifference) {
	                  currentY += lastDifference;
	                } else {
	                  currentY = lastValue;
	                }

	                currentData.x.push(currentX);
	                currentData.y.push(currentY * yFactor);
	                currentX += deltaX;
	              }
	            }
	          }

	          isNegative = false;
	          currentValue = 0;
	          decimalPosition = 0;
	          inValue = false;
	          isDuplicate = false;
	        } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


	        if (ascii < 74 && ascii > 63) {
	          inValue = true;
	          isLastDifference = false;
	          currentValue = ascii - 64;
	        } else if (ascii > 96 && ascii < 106) {
	          // negative SQZ digits a b c d e f g h i (ascii 97-105)
	          inValue = true;
	          isLastDifference = false;
	          currentValue = ascii - 96;
	          isNegative = true;
	        } else if (ascii === 115) {
	          // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
	          inValue = true;
	          isDuplicate = true;
	          currentValue = 9;
	        } else if (ascii > 82 && ascii < 91) {
	          inValue = true;
	          isDuplicate = true;
	          currentValue = ascii - 82;
	        } else if (ascii > 73 && ascii < 83) {
	          // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
	          inValue = true;
	          isDifference = true;
	          currentValue = ascii - 73;
	        } else if (ascii > 105 && ascii < 115) {
	          // negative DIF digits j k l m n o p q r (ascii 106-114)
	          inValue = true;
	          isDifference = true;
	          currentValue = ascii - 105;
	          isNegative = true;
	        } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
	          // $ sign, we need to check the next one
	          inValue = true;
	          inComment = true;
	        } else if (ascii === 37) {
	          // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
	          inValue = true;
	          isDifference = true;
	          currentValue = 0;
	          isNegative = false;
	        } else if (ascii === 45) {
	          // a "-"
	          // check if after there is a number, decimal or comma
	          let ascii2 = value.charCodeAt(i + 1);

	          if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
	            inValue = true;
	            if (!newLine) isLastDifference = false;
	            isNegative = true;
	          }
	        } else if (ascii === 13 || ascii === 10) {
	          newLine = true;
	          inComment = false;
	        } // and now analyse the details ... space or tabulation
	        // if "+" we just don't care

	      }
	    }
	  }
	}

	const removeCommentRegExp = /\$\$.*/;
	const peakTableSplitRegExp = /[,\t ]+/;
	function parsePeakTable(spectrum, value, result) {
	  spectrum.isPeaktable = true;

	  if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
	    parseXY(spectrum, value, result);
	  } else {
	    parseXYZ(spectrum, value, result);
	  } // we will add the data in the variables


	  if (spectrum.variables) {
	    for (let key in spectrum.variables) {
	      spectrum.variables[key].data = spectrum.data[key];
	    }
	  }
	}

	function parseXY(spectrum, value, result) {
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData; // counts for around 20% of the time

	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

	    if (values.length % 2 === 0) {
	      for (let j = 0; j < values.length; j = j + 2) {
	        // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
	        currentData.x.push(Number(values[j]) * spectrum.xFactor);
	        currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
	      }
	    } else {
	      result.logs.push(`Format error: ${values}`);
	    }
	  }
	}

	function parseXYZ(spectrum, value, result) {
	  let currentData = {};
	  let variables = Object.keys(spectrum.variables);
	  let numberOfVariables = variables.length;
	  variables.forEach(variable => currentData[variable] = []);
	  spectrum.data = currentData; // counts for around 20% of the time

	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

	    if (values.length % numberOfVariables === 0) {
	      for (let j = 0; j < values.length; j++) {
	        // todo should try to find a xFactor (y, ...)
	        currentData[variables[j % numberOfVariables]].push(Number(values[j]));
	      }
	    } else {
	      result.logs.push(`Format error: ${values}`);
	    }
	  }
	}

	function parseXYA(spectrum, value) {
	  let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
	  spectrum.isXYAdata = true;
	  let values;
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData;
	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
	    currentData.x.push(Number(values[0]));
	    currentData.y.push(Number(values[1]));
	  }
	}

	function convertTo3DZ(spectra) {
	  let minZ = spectra[0].data.y[0];
	  let maxZ = minZ;
	  let ySize = spectra.length;
	  let xSize = spectra[0].data.x.length;
	  let z = new Array(ySize);

	  for (let i = 0; i < ySize; i++) {
	    z[i] = spectra[i].data.y;

	    for (let j = 0; j < xSize; j++) {
	      let value = z[i][j];
	      if (value < minZ) minZ = value;
	      if (value > maxZ) maxZ = value;
	    }
	  }

	  const firstX = spectra[0].data.x[0];
	  const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

	  const firstY = spectra[0].pageValue;
	  const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
	  // min and max we need to invert the array

	  if (firstX > lastX) {
	    for (let spectrum of z) {
	      spectrum.reverse();
	    }
	  }

	  if (firstY > lastY) {
	    z.reverse();
	  }

	  const medians = [];

	  for (let i = 0; i < z.length; i++) {
	    const row = Float64Array.from(z[i]);

	    for (let i = 0; i < row.length; i++) {
	      if (row[i] < 0) row[i] = -row[i];
	    }

	    medians.push(median(row));
	  }

	  const median$1 = median(medians);
	  return {
	    z: z,
	    minX: Math.min(firstX, lastX),
	    maxX: Math.max(firstX, lastX),
	    minY: Math.min(firstY, lastY),
	    maxY: Math.max(firstY, lastY),
	    minZ: minZ,
	    maxZ: maxZ,
	    noise: median$1
	  };
	}

	function generateContourLines(zData, options) {
	  let noise = zData.noise;
	  let z = zData.z;
	  let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
	  let isOver0, isOver1, isOver2, isOver3;
	  let nbSubSpectra = z.length;
	  let nbPovars = z[0].length;
	  let pAx, pAy, pBx, pBy;
	  let x0 = zData.minX;
	  let xN = zData.maxX;
	  let dx = (xN - x0) / (nbPovars - 1);
	  let y0 = zData.minY;
	  let yN = zData.maxY;
	  let dy = (yN - y0) / (nbSubSpectra - 1);
	  let minZ = zData.minZ;
	  let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
	  // -------------------------
	  // Povars attribution
	  //
	  // 0----1
	  // |  / |
	  // | /  |
	  // 2----3
	  //
	  // ---------------------d------

	  let iter = options.nbContourLevels * 2;
	  let contourLevels = new Array(iter);
	  let lineZValue;

	  for (let level = 0; level < iter; level++) {
	    // multiply by 2 for positif and negatif
	    let contourLevel = {};
	    contourLevels[level] = contourLevel;
	    let side = level % 2;
	    let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

	    if (side === 0) {
	      lineZValue = factor + options.noiseMultiplier * noise;
	    } else {
	      lineZValue = 0 - factor - options.noiseMultiplier * noise;
	    }

	    let lines = [];
	    contourLevel.zValue = lineZValue;
	    contourLevel.lines = lines;
	    if (lineZValue <= minZ || lineZValue >= maxZ) continue;

	    for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
	      let subSpectra = z[iSubSpectra];
	      let subSpectraAfter = z[iSubSpectra + 1];

	      for (let povar = 0; povar < nbPovars - 1; povar++) {
	        povarHeight0 = subSpectra[povar];
	        povarHeight1 = subSpectra[povar + 1];
	        povarHeight2 = subSpectraAfter[povar];
	        povarHeight3 = subSpectraAfter[povar + 1];
	        isOver0 = povarHeight0 > lineZValue;
	        isOver1 = povarHeight1 > lineZValue;
	        isOver2 = povarHeight2 > lineZValue;
	        isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
	        // povar2 are below, we find the varersections and add
	        // the segment

	        if (isOver0 !== isOver1 && isOver0 !== isOver2) {
	          pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
	          pAy = iSubSpectra;
	          pBx = povar;
	          pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
	          lines.push(pAx * dx + x0);
	          lines.push(pAy * dy + y0);
	          lines.push(pBx * dx + x0);
	          lines.push(pBy * dy + y0);
	        } // remove push does not help !!!!


	        if (isOver3 !== isOver1 && isOver3 !== isOver2) {
	          pAx = povar + 1;
	          pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
	          pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
	          pBy = iSubSpectra + 1;
	          lines.push(pAx * dx + x0);
	          lines.push(pAy * dy + y0);
	          lines.push(pBx * dx + x0);
	          lines.push(pBy * dy + y0);
	        } // test around the diagonal


	        if (isOver1 !== isOver2) {
	          pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
	          pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

	          if (isOver1 !== isOver0) {
	            pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
	            pBy = iSubSpectra;
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver2 !== isOver0) {
	            pBx = povar;
	            pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver1 !== isOver3) {
	            pBx = povar + 1;
	            pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver2 !== isOver3) {
	            pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
	            pBy = iSubSpectra + 1;
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }
	        }
	      }
	    }
	  }

	  return {
	    minX: zData.minX,
	    maxX: zData.maxX,
	    minY: zData.minY,
	    maxY: zData.maxY,
	    segments: contourLevels
	  };
	}

	function add2D(result, options) {
	  let zData = convertTo3DZ(result.spectra);

	  if (!options.noContour) {
	    result.contourLines = generateContourLines(zData, options);
	    delete zData.z;
	  }

	  result.minMax = zData;
	}

	const gyromagneticRatio = {
	  '1H': 267.52218744e6,
	  '2H': 41.065e6,
	  '3H': 285.3508e6,
	  '3He': -203.789e6,
	  '7Li': 103.962e6,
	  '13C': 67.28284e6,
	  '14N': 19.331e6,
	  '15N': -27.116e6,
	  '17O': -36.264e6,
	  '19F': 251.662e6,
	  '23Na': 70.761e6,
	  '27Al': 69.763e6,
	  '29Si': -53.19e6,
	  '31P': 108.291e6,
	  '57Fe': 8.681e6,
	  '63Cu': 71.118e6,
	  '67Zn': 16.767e6,
	  '129Xe': -73.997e6
	};

	function postProcessingNMR(entriesFlat) {
	  // specific NMR functions
	  for (let entry of entriesFlat) {
	    let observeFrequency = 0;
	    let shiftOffsetVal = 0;

	    for (let spectrum of entry.spectra) {
	      if (entry.ntuples && entry.ntuples.symbol) {
	        if (!observeFrequency && spectrum.observeFrequency) {
	          observeFrequency = spectrum.observeFrequency;
	        }

	        if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
	          shiftOffsetVal = spectrum.shiftOffsetVal;
	        }
	      } else {
	        observeFrequency = spectrum.observeFrequency;
	        shiftOffsetVal = spectrum.shiftOffsetVal;
	      }

	      if (observeFrequency) {
	        if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
	          spectrum.xUnits = 'PPM';
	          spectrum.xFactor = spectrum.xFactor / observeFrequency;
	          spectrum.firstX = spectrum.firstX / observeFrequency;
	          spectrum.lastX = spectrum.lastX / observeFrequency;
	          spectrum.deltaX = spectrum.deltaX / observeFrequency;

	          for (let i = 0; i < spectrum.data.x.length; i++) {
	            spectrum.data.x[i] /= observeFrequency;
	          }
	        }
	      }

	      if (shiftOffsetVal) {
	        let shift = spectrum.firstX - shiftOffsetVal;
	        spectrum.firstX = spectrum.firstX - shift;
	        spectrum.lastX = spectrum.lastX - shift;

	        for (let i = 0; i < spectrum.data.x.length; i++) {
	          spectrum.data.x[i] -= shift;
	        }
	      } // we will check if some nucleus are missing ...


	      if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
	        for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
	          let symbol = entry.ntuples.symbol[i];
	          let nucleus = entry.ntuples.nucleus[i];

	          if (symbol.startsWith('F') && !nucleus) {
	            if (symbol === 'F1') {
	              // if F1 is defined we will use F2
	              if (entry.tmp.$NUC2) {
	                entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
	              } else {
	                let f2index = entry.ntuples.symbol.indexOf('F2');

	                if (f2index && entry.ntuples.nucleus[f2index]) {
	                  entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
	                }
	              }
	            }

	            if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
	          }

	          if (symbol === 'F2') {
	            entry.yType = entry.ntuples.nucleus[0];
	          }
	        }
	      }

	      if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
	        let unit = '';
	        let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

	        if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
	          unit = entry.ntuples.units[pageSymbolIndex];
	        }

	        if (unit !== 'PPM') {
	          if (pageSymbolIndex !== 0) {
	            throw Error('Not sure about this ntuples format');
	          }

	          let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
	          let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

	          if (!ratio0 || !ratio1) {
	            throw Error('Problem with determination of gyromagnetic ratio');
	          }

	          let ratio = ratio0 / ratio1 * observeFrequency;
	          spectrum.pageValue /= ratio;
	        }
	      }
	    }
	  }
	}

	function profiling(result, action, options) {
	  if (result.profiling) {
	    result.profiling.push({
	      action,
	      time: Date.now() - options.start
	    });
	  }
	}

	function simpleChromatogram(result) {
	  let data = result.spectra[0].data;
	  result.chromatogram = {
	    times: data.x.slice(),
	    series: {
	      intensity: {
	        dimension: 1,
	        data: data.y.slice()
	      }
	    }
	  };
	}

	function postProcessing(entriesFlat, result, options) {
	  // converting Hz to ppm
	  postProcessingNMR(entriesFlat);

	  for (let entry of entriesFlat) {
	    if (Object.keys(entry.ntuples).length > 0) {
	      let newNtuples = [];
	      let keys = Object.keys(entry.ntuples);

	      for (let i = 0; i < keys.length; i++) {
	        let key = keys[i];
	        let values = entry.ntuples[key];

	        for (let j = 0; j < values.length; j++) {
	          if (!newNtuples[j]) newNtuples[j] = {};
	          newNtuples[j][key] = values[j];
	        }
	      }

	      entry.ntuples = newNtuples;
	    }

	    if (entry.twoD && options.wantXY) {
	      add2D(entry, options);
	      profiling(result, 'Finished countour plot calculation', options);

	      if (!options.keepSpectra) {
	        delete entry.spectra;
	      }
	    } // maybe it is a GC (HPLC) / MS. In this case we add a new format


	    if (options.chromatogram) {
	      if (entry.spectra.length > 1) {
	        complexChromatogram(entry);
	      } else {
	        simpleChromatogram(entry);
	      }

	      profiling(result, 'Finished chromatogram calculation', options);
	    }

	    delete entry.tmp;
	  }
	}

	function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
	  let xIndex = -1;
	  let yIndex = -1;
	  let firstVariable = '';
	  let secondVariable = '';

	  if (kind.indexOf('++') > 0) {
	    firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
	    secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
	  } else {
	    kind = kind.replace(/[^a-zA-Z]/g, '');
	    firstVariable = kind.charAt(0);
	    secondVariable = kind.charAt(1);
	    spectrum.variables = {};

	    for (let symbol of kind) {
	      let lowerCaseSymbol = symbol.toLowerCase();
	      let index = currentEntry.ntuples.symbol.indexOf(symbol);
	      if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
	      spectrum.variables[lowerCaseSymbol] = {};

	      for (let key in currentEntry.ntuples) {
	        if (currentEntry.ntuples[key][index]) {
	          spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
	        }
	      }
	    }
	  }

	  xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
	  yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
	  if (xIndex === -1) xIndex = 0;
	  if (yIndex === -1) yIndex = 0;

	  if (currentEntry.ntuples.first) {
	    if (currentEntry.ntuples.first.length > xIndex) {
	      spectrum.firstX = currentEntry.ntuples.first[xIndex];
	    }

	    if (currentEntry.ntuples.first.length > yIndex) {
	      spectrum.firstY = currentEntry.ntuples.first[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.last) {
	    if (currentEntry.ntuples.last.length > xIndex) {
	      spectrum.lastX = currentEntry.ntuples.last[xIndex];
	    }

	    if (currentEntry.ntuples.last.length > yIndex) {
	      spectrum.lastY = currentEntry.ntuples.last[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
	    spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
	  }

	  if (currentEntry.ntuples.factor) {
	    if (currentEntry.ntuples.factor.length > xIndex) {
	      spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
	    }

	    if (currentEntry.ntuples.factor.length > yIndex) {
	      spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.units) {
	    if (currentEntry.ntuples.units.length > xIndex) {
	      if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
	        spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
	      } else {
	        spectrum.xUnits = currentEntry.ntuples.units[xIndex];
	      }
	    }

	    if (currentEntry.ntuples.units.length > yIndex) {
	      if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
	        spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
	      } else {
	        spectrum.yUnits = currentEntry.ntuples.units[yIndex];
	      }
	    }
	  }
	}

	function prepareSpectrum(spectrum) {
	  if (!spectrum.xFactor) spectrum.xFactor = 1;
	  if (!spectrum.yFactor) spectrum.yFactor = 1;
	}

	const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
	const defaultOptions = {
	  keepRecordsRegExp: /^$/,
	  canonicDataLabels: true,
	  canonicMetadataLabels: false,
	  dynamicTyping: true,
	  withoutXY: false,
	  chromatogram: false,
	  keepSpectra: false,
	  noContour: false,
	  nbContourLevels: 7,
	  noiseMultiplier: 5,
	  profiling: false
	};
	/**
	 *
	 * @typedef {object} ConvertOptions
	 * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
	 * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
	 * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
	 * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
	 * @property {boolean} [options.withoutXY=false] - Remove the XY data.
	 * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
	 * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
	 * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
	 * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
	 * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
	 * @property {boolean} [options.profiling=false] - Add profiling information.
	 */

	/**
	 *
	 * @typedef {object} Ntuples
	 * @property {string[]} [varname]
	 * @property {string[]} [symbol]
	 * @property {string[]} [vartype]
	 * @property {string[]} [varform]
	 * @property {number[]} [vardim]
	 * @property {string[]} [units]
	 * @property {number[]} [factor]
	 * @property {number[]} [first]
	 * @property {number[]} [last]
	 * @property {number[]} [min]
	 * @property {number[]} [max]
	 * @property {string[]} [nucleus]
	 */

	/**
	 * @typedef { Record<string, any> } Spectrum
	 * @property {Record<string, number[]>} [data]
	 * @property {number} [firstX]
	 * @property {number} [lastX]
	 * @property {number} [deltaX]
	 * @property {number} [yFactor]
	 * @property {number} [xFactor]
	 * @property {number} [nbPoints]
	 */

	/**
	 *
	 * @typedef {object} Entry
	 * @property {Spectrum[]} spectra
	 * @property {Ntuples} ntuples
	 * @property {object} meta
	 * @property {object} tmp
	 * @property {string} [title]
	 * @property {string} [dataType]
	 * @property {string} [dataClass]
	 * @property {boolean} [twoD]
	 */

	/**
	 *
	 * @typedef { object } ConvertResult
	 * @property { object[] | boolean } profiling
	 * @property { string[] } logs
	 * @property { object[] } entries
	 * @property { Entry[] } flatten
	 */

	/**
	 * Parse a jcamp.
	 *
	 * @param {string|ArrayBuffer|Uint8Array} jcamp
	 * @param {ConvertOptions} [options]
	 * @returns {ConvertResult}
	 */

	function convert(jcamp) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  jcamp = ensureString(jcamp);
	  options = { ...defaultOptions,
	    ...options
	  };
	  options.wantXY = !options.withoutXY;
	  options.start = Date.now();
	  let entriesFlat = [];
	  let result = {
	    profiling: options.profiling ? [] : false,
	    logs: [],
	    entries: []
	  };
	  let tmpResult = {
	    children: []
	  };
	  let currentEntry = tmpResult;
	  let parentsStack = [];
	  let spectrum = {};

	  if (typeof jcamp !== 'string') {
	    throw new TypeError('the JCAMP should be a string');
	  }

	  profiling(result, 'Before split to LDRS', options);
	  let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
	  profiling(result, 'Split to LDRS', options);
	  if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

	  for (let ldr of ldrs) {
	    // This is a new LDR
	    let position = ldr.indexOf('=');
	    let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
	    let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
	    let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

	    if (canonicDataLabel === 'DATATABLE') {
	      let endLine = dataValue.indexOf('\n');
	      if (endLine === -1) endLine = dataValue.indexOf('\r');

	      if (endLine > 0) {
	        // ##DATA TABLE= (X++(I..I)), XYDATA
	        // We need to find the variables
	        let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
	        prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
	        spectrum.datatable = infos[0];

	        if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
	          canonicDataLabel = 'PEAKTABLE';
	        } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
	          canonicDataLabel = 'XYDATA';

	          if (spectrum.nbPoints) {
	            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
	          }
	        }
	      }
	    }

	    if (canonicDataLabel === 'XYDATA') {
	      if (options.wantXY) {
	        prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

	        if (dataValue.match(/.*\+\+.*/)) {
	          // ex: (X++(Y..Y))
	          if (spectrum.nbPoints) {
	            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
	          }

	          fastParseXYData(spectrum, dataValue);
	        } else {
	          parsePeakTable(spectrum, dataValue, result);
	        }

	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    } else if (canonicDataLabel === 'PEAKTABLE') {
	      if (options.wantXY) {
	        prepareSpectrum(spectrum);
	        parsePeakTable(spectrum, dataValue, result);
	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    }

	    if (canonicDataLabel === 'PEAKASSIGNMENTS') {
	      if (options.wantXY) {
	        if (dataValue.match(/.*(XYA).*/)) {
	          // ex: (XYA)
	          parseXYA(spectrum, dataValue);
	        }

	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    }

	    if (canonicDataLabel === 'TITLE') {
	      let parentEntry = currentEntry;

	      if (!parentEntry.children) {
	        parentEntry.children = [];
	      }

	      currentEntry = {
	        spectra: [],
	        ntuples: {},
	        info: {},
	        meta: {},
	        tmp: {} // tmp information we need to keep for postprocessing

	      };
	      parentEntry.children.push(currentEntry);
	      parentsStack.push(parentEntry);
	      entriesFlat.push(currentEntry);
	      currentEntry.title = dataValue;
	    } else if (canonicDataLabel === 'DATATYPE') {
	      currentEntry.dataType = dataValue;

	      if (dataValue.match(/(^nd|\snd\s)/i)) {
	        currentEntry.twoD = true;
	      }
	    } else if (canonicDataLabel === 'NTUPLES') {
	      if (dataValue.match(/(^nd|\snd\s)/i)) {
	        currentEntry.twoD = true;
	      }
	    } else if (canonicDataLabel === 'DATACLASS') {
	      currentEntry.dataClass = dataValue;
	    } else if (canonicDataLabel === 'XUNITS') {
	      spectrum.xUnits = dataValue;
	    } else if (canonicDataLabel === 'YUNITS') {
	      spectrum.yUnits = dataValue;
	    } else if (canonicDataLabel === 'FIRSTX') {
	      spectrum.firstX = Number(dataValue);
	    } else if (canonicDataLabel === 'LASTX') {
	      spectrum.lastX = Number(dataValue);
	    } else if (canonicDataLabel === 'FIRSTY') {
	      spectrum.firstY = Number(dataValue);
	    } else if (canonicDataLabel === 'LASTY') {
	      spectrum.lastY = Number(dataValue);
	    } else if (canonicDataLabel === 'NPOINTS') {
	      spectrum.nbPoints = Number(dataValue);
	    } else if (canonicDataLabel === 'XFACTOR') {
	      spectrum.xFactor = Number(dataValue);
	    } else if (canonicDataLabel === 'YFACTOR') {
	      spectrum.yFactor = Number(dataValue);
	    } else if (canonicDataLabel === 'MAXX') {
	      spectrum.maxX = Number(dataValue);
	    } else if (canonicDataLabel === 'MINX') {
	      spectrum.minX = Number(dataValue);
	    } else if (canonicDataLabel === 'MAXY') {
	      spectrum.maxY = Number(dataValue);
	    } else if (canonicDataLabel === 'MINY') {
	      spectrum.minY = Number(dataValue);
	    } else if (canonicDataLabel === 'DELTAX') {
	      spectrum.deltaX = Number(dataValue);
	    } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
	      if (!spectrum.observeFrequency) {
	        spectrum.observeFrequency = Number(dataValue);
	      }
	    } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
	      if (!spectrum.xType) {
	        currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
	      }
	    } else if (canonicDataLabel === '$OFFSET') {
	      // OFFSET for Bruker spectra
	      currentEntry.shiftOffsetNum = 0;

	      if (!spectrum.shiftOffsetVal) {
	        spectrum.shiftOffsetVal = Number(dataValue);
	      }
	    } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
	      currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'SYMBOL') {
	      currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARTYPE') {
	      currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARFORM') {
	      currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARDIM') {
	      currentEntry.ntuples.vardim = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'UNITS') {
	      currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'FACTOR') {
	      currentEntry.ntuples.factor = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'FIRST') {
	      currentEntry.ntuples.first = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'LAST') {
	      currentEntry.ntuples.last = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'MIN') {
	      currentEntry.ntuples.min = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'MAX') {
	      currentEntry.ntuples.max = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === '.NUCLEUS') {
	      if (currentEntry.ntuples) {
	        currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
	      }
	    } else if (canonicDataLabel === 'PAGE') {
	      spectrum.page = dataValue.trim();
	      spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
	      spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
	    } else if (canonicDataLabel === 'RETENTIONTIME') {
	      spectrum.pageValue = Number(dataValue);
	    } else if (isMSField(canonicDataLabel)) {
	      spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
	    } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
	      spectrum.sampleDescription = dataValue;
	    } else if (canonicDataLabel.startsWith('$NUC')) {
	      if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
	        currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
	      }
	    } else if (canonicDataLabel === 'END') {
	      currentEntry = parentsStack.pop();
	    }

	    if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
	      let value = dataValue.trim();
	      let target, label;

	      if (dataLabel.startsWith('$')) {
	        label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
	        target = currentEntry.meta;
	      } else {
	        label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
	        target = currentEntry.info;
	      }

	      if (options.dynamicTyping) {
	        value = parseString(value);
	      }

	      if (target[label]) {
	        if (!Array.isArray(target[label])) {
	          target[label] = [target[label]];
	        }

	        target[label].push(value);
	      } else {
	        target[label] = value;
	      }
	    }
	  }

	  profiling(result, 'Finished parsing', options);
	  postProcessing(entriesFlat, result, options);
	  profiling(result, 'Total time', options);
	  /*
	  if (result.children && result.children.length>0) {
	    result = { ...result, ...result.children[0] };
	  }
	  */

	  result.entries = tmpResult.children;
	  result.flatten = entriesFlat;
	  return result;
	}

	/**
	 * Creates a new Analysis from a JCAMP string
	 * @param {string} jcamp - String containing the JCAMP data
	 * @param {object} [options={}]
	 * @param {object} [options.id=Math.random()]
	 * @param {string} [options.label=options.id] human redeable label
	 * @param {string} [options.spectrumCallback] a callback to apply on variables when creating spectrum
	 * @return {Analysis} - New class element with the given data
	 */

	function fromJcamp(jcamp) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let analysis = new Analysis(options);
	  addJcamp(analysis, jcamp);
	  return analysis;
	}

	function addJcamp(analysis, jcamp) {
	  let converted = convert(jcamp, {
	    keepRecordsRegExp: /.*/
	  });

	  for (let entry of converted.flatten) {
	    if (!entry.spectra || !entry.spectra[0]) continue;
	    let currentSpectrum = entry.spectra[0]; // we ensure variables

	    if (!currentSpectrum.variables) {
	      const variables = {};
	      currentSpectrum.variables = variables;
	      variables.x = {
	        label: currentSpectrum.xUnits,
	        symbol: 'X',
	        data: currentSpectrum.data.x || currentSpectrum.data.X
	      };
	      variables.y = {
	        label: currentSpectrum.yUnits,
	        symbol: 'Y',
	        data: currentSpectrum.data.y || currentSpectrum.data.Y
	      };
	    } else {
	      for (let key in currentSpectrum.variables) {
	        const variable = currentSpectrum.variables[key];
	        if (variable.label) continue;
	        variable.label = variable.name || variable.symbol || key;

	        if (variable.units && !variable.label.includes(variable.units)) {
	          variable.label += ` [${variable.units}]`;
	        }
	      }
	    }

	    analysis.pushSpectrum(currentSpectrum.variables, {
	      dataType: entry.dataType,
	      title: entry.title,
	      meta: entry.meta
	    });
	  }
	}

	const addInfoData = function (data) {
	  let keys = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : Object.keys(data);
	  let prefix = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '##$';
	  let header = '';

	  for (const key of keys) {
	    header += typeof data[key] === 'object' ? `${prefix}${key}=${JSON.stringify(data[key])}\n` : `${prefix}${key}=${data[key]}\n`;
	  }

	  return header;
	};

	function checkNumberOrArray(data) {
	  if (!isAnyArray(data) || isAnyArray(data[0])) {
	    throw new Error(`x and y data should be an array of numbers`);
	  }
	}

	function checkMatrix(data) {
	  if (!isAnyArray(data) || !isAnyArray(data[0])) {
	    throw new Error(`2D data should be a matrix`);
	  }
	}

	function getExtremeValues(data) {
	  if (isAnyArray(data[0])) {
	    checkMatrix(data);
	    const firstRow = data[0];
	    return {
	      firstLast: {
	        first: firstRow[0],
	        last: data[data.length - 1][data[0].length - 1]
	      },
	      minMax: matrixMinMaxZ(data)
	    };
	  }

	  checkNumberOrArray(data);
	  return {
	    firstLast: {
	      first: data[0],
	      last: data[data.length - 1]
	    },
	    minMax: xMinMaxValues(data)
	  };
	}

	/**
	 * class encodes a integer vector as a String in order to store it in a text file.
	 * The algorithms used to encode the data are describe in:
	 *            http://www.iupac.org/publications/pac/pdf/2001/pdf/7311x1765.pdf
	 */
	const newLine = '\n';
	const pseudoDigits = [['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], ['@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'], ['@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i'], ['%', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R'], ['%', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r'], [' ', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 's']];
	const SQZ_P = 1;
	const SQZ_N = 2;
	const DIF_P = 3;
	const DIF_N = 4;
	const DUP = 5;
	const maxLinelength = 100;
	/**
	 * This function encodes the given vector. The xyEncoding format is specified by the
	 * xyEncoding option
	 * @param xyEncoding: ('FIX','SQZ','DIF','DIFDUP','CVS','PAC') Default 'DIFDUP'
	 * @return {string}
	 */

	function vectorEncoder(data, firstX, intervalX, xyEncoding) {
	  switch (xyEncoding) {
	    case 'FIX':
	      return fixEncoding(data, firstX, intervalX);

	    case 'SQZ':
	      return squeezedEncoding(data, firstX, intervalX);

	    case 'DIF':
	      return differenceEncoding(data, firstX, intervalX);

	    case 'DIFDUP':
	      return differenceDuplicateEncoding(data, firstX, intervalX);

	    case 'CSV':
	      return commaSeparatedValuesEncoding(data, firstX, intervalX);

	    case 'PAC':
	      return packedEncoding(data, firstX, intervalX);

	    default:
	      return differenceEncoding(data, firstX, intervalX);
	  }
	}
	/**
	 * @private
	 * No data compression used. The data is separated by a comma(',').
	 */

	function commaSeparatedValuesEncoding(data, firstX, intervalX) {
	  return fixEncoding(data, firstX, intervalX, ',');
	}
	/**
	 * @private
	 * No data compression used. The data is separated by the specified separator.
	 */

	function fixEncoding(data, firstX, intervalX) {
	  let separator = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : ' ';
	  let outputData = '';
	  let j = 0;
	  let dataLength = data.length;

	  while (j < dataLength - 7) {
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = 0; i < 8; i++) {
	      outputData += separator + data[j++];
	    }

	    outputData += newLine;
	  }

	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = j; i < dataLength; i++) {
	      outputData += separator + data[i];
	    }
	  }

	  return outputData;
	}
	/**
	 * @private
	 * No data compression used. The data is separated by the sign of the number.
	 */

	function packedEncoding(data, firstX, intervalX) {
	  let outputData = '';
	  let j = 0;
	  let dataLength = data.length;

	  while (j < dataLength - 7) {
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = 0; i < 8; i++) {
	      outputData += data[j] < 0 ? data[j++] : `+${data[j++]}`;
	    }

	    outputData += newLine;
	  }

	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = j; i < dataLength; i++) {
	      outputData += data[i] < 0 ? data[i] : `+${data[i]}`;
	    }
	  }

	  return outputData;
	}
	/**
	 * @private
	 * Data compression is possible using the squeezed form (SQZ) in which the delimiter, the leading digit,
	 * and sign are replaced by a pseudo-digit from Table 1. For example, the Y-values 30, 32 would be
	 * represented as C0C2.
	 */

	function squeezedEncoding(data, firstX, intervalX) {
	  let outputData = ''; // String outputData = new String();

	  let j = 0;
	  let dataLength = data.length;

	  while (j < dataLength - 10) {
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = 0; i < 10; i++) {
	      outputData += squeezedDigit(data[j++].toString());
	    }

	    outputData += newLine;
	  }

	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);

	    for (let i = j; i < dataLength; i++) {
	      outputData += squeezedDigit(data[i].toString());
	    }
	  }

	  return outputData;
	}
	/**
	 * @private
	 * Duplicate suppression xyEncoding
	 */

	function differenceDuplicateEncoding(data, firstX, intervalX) {
	  let mult = 0;
	  let index = 0;
	  let charCount = 0; // We built a string where we store the encoded data.

	  let encodedData = '';
	  let encodedNumber = '';
	  let temp = ''; // We calculate the differences vector

	  let diffData = new Array(data.length - 1);

	  for (let i = 0; i < diffData.length; i++) {
	    diffData[i] = data[i + 1] - data[i];
	  } // We simulate a line carry


	  let numDiff = diffData.length;

	  while (index < numDiff) {
	    if (charCount === 0) {
	      // Start line
	      encodedNumber = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + differenceDigit(diffData[index].toString());
	      encodedData += encodedNumber;
	      charCount += encodedNumber.length;
	    } else {
	      // Try to insert next difference
	      if (diffData[index - 1] === diffData[index]) {
	        mult++;
	      } else {
	        if (mult > 0) {
	          // Now we know that it can be in line
	          mult++;
	          encodedNumber = duplicateDigit(mult.toString());
	          encodedData += encodedNumber;
	          charCount += encodedNumber.length;
	          mult = 0;
	          index--;
	        } else {
	          // Mirar si cabe, en caso contrario iniciar una nueva linea
	          encodedNumber = differenceDigit(diffData[index].toString());

	          if (encodedNumber.length + charCount < maxLinelength) {
	            encodedData += encodedNumber;
	            charCount += encodedNumber.length;
	          } else {
	            // Iniciar nueva linea
	            encodedData += newLine;
	            temp = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + encodedNumber;
	            encodedData += temp; // Each line start with first index number.

	            charCount = temp.length;
	          }
	        }
	      }
	    }

	    index++;
	  }

	  if (mult > 0) {
	    encodedData += duplicateDigit((mult + 1).toString());
	  } // We insert the last data from fid. It is done to control of data
	  // The last line start with the number of datas in the fid.


	  encodedData += newLine + Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString());
	  return encodedData;
	}
	/**
	 * @private
	 * Differential xyEncoding
	 */

	function differenceEncoding(data, firstX, intervalX) {
	  let index = 0;
	  let charCount = 0;
	  let i;
	  let encodedData = '';
	  let encodedNumber = '';
	  let temp = ''; // We calculate the differences vector

	  let diffData = new Array(data.length - 1);

	  for (i = 0; i < diffData.length; i++) {
	    diffData[i] = data[i + 1] - data[i];
	  }

	  let numDiff = diffData.length;

	  while (index < numDiff) {
	    if (charCount === 0) {
	      // We convert the first number.
	      encodedNumber = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + differenceDigit(diffData[index].toString());
	      encodedData += encodedNumber;
	      charCount += encodedNumber.length;
	    } else {
	      encodedNumber = differenceDigit(diffData[index].toString());

	      if (encodedNumber.length + charCount < maxLinelength) {
	        encodedData += encodedNumber;
	        charCount += encodedNumber.length;
	      } else {
	        encodedData += newLine;
	        temp = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + encodedNumber;
	        encodedData += temp; // Each line start with first index number.

	        charCount = temp.length;
	      }
	    }

	    index++;
	  } // We insert the last number from data. It is done to control of data


	  encodedData += newLine + Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString());
	  return encodedData;
	}
	/**
	 * @private
	 * Convert number to the ZQZ format, using pseudo digits.
	 */

	function squeezedDigit(num) {
	  let sqzDigits = '';

	  if (num.startsWith('-')) {
	    sqzDigits += pseudoDigits[SQZ_N][num.charCodeAt(1) - 48];

	    if (num.length > 2) {
	      sqzDigits += num.substring(2);
	    }
	  } else {
	    sqzDigits += pseudoDigits[SQZ_P][num.charCodeAt(0) - 48];

	    if (num.length > 1) {
	      sqzDigits += num.substring(1);
	    }
	  }

	  return sqzDigits;
	}
	/**
	 * Convert number to the DIF format, using pseudo digits.
	 */


	function differenceDigit(num) {
	  let diffDigits = '';

	  if (num.startsWith('-')) {
	    diffDigits += pseudoDigits[DIF_N][num.charCodeAt(1) - 48];

	    if (num.length > 2) {
	      diffDigits += num.substring(2);
	    }
	  } else {
	    diffDigits += pseudoDigits[DIF_P][num.charCodeAt(0) - 48];

	    if (num.length > 1) {
	      diffDigits += num.substring(1);
	    }
	  }

	  return diffDigits;
	}
	/**
	 * Convert number to the DUP format, using pseudo digits.
	 */


	function duplicateDigit(num) {
	  let dupDigits = '';
	  dupDigits += pseudoDigits[DUP][num.charCodeAt(0) - 48];

	  if (num.length > 1) {
	    dupDigits += num.substring(1);
	  }

	  return dupDigits;
	}

	/**
	 * Parse from a xyxy data array
	 * @param variables - Variables to convert to jcamp
	 * @param [options={}] - options that allows to add meta data in the jcamp
	 * @return JCAMP-DX text file corresponding to the variables
	 */

	function creatorNtuples(variables, options) {
	  const {
	    meta = {},
	    info = {},
	    xyEncoding = ''
	  } = options;
	  const {
	    title = '',
	    owner = '',
	    origin = '',
	    dataType = '',
	    xFactor = 1,
	    yFactor = 1
	  } = info;
	  const symbol = [];
	  const varName = [];
	  const varType = [];
	  const varDim = [];
	  const units = [];
	  const first = [];
	  const last = [];
	  const min = [];
	  const max = [];
	  const keys = Object.keys(variables);

	  for (let i = 0; i < keys.length; i++) {
	    const key = keys[i];
	    let variable = variables[key];
	    if (!variable) continue;
	    let name = variable?.label.replace(/ *\[.*/, '');
	    let unit = variable?.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
	    const {
	      firstLast,
	      minMax
	    } = getExtremeValues(variable.data);
	    symbol.push(variable.symbol || key);
	    varName.push(name || key);
	    varDim.push(variable.data.length);
	    first.push(firstLast.first);
	    last.push(firstLast.last);
	    max.push(minMax.max);
	    min.push(minMax.min);

	    if (variable.isDependent !== undefined) {
	      varType.push(variable.isDependent ? 'DEPENDENT' : 'INDEPENDENT');
	    } else {
	      varType.push(variable.isDependent !== undefined ? !variable.isDependent : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
	    }

	    units.push(variable.units || unit || '');
	  }

	  let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##DATA CLASS= NTUPLES
##ORIGIN=${origin}
##OWNER=${owner}\n`;
	  const infoKeys = Object.keys(info).filter(e => !['title', 'owner', 'origin', 'dataType'].includes(e));
	  header += addInfoData(info, infoKeys, '##');
	  header += addInfoData(meta);
	  header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##FIRST=     ${first.join()}
##LAST=      ${last.join()}
##MIN=       ${min.join()}
##MAX=       ${max.join()}\n`;

	  if (options.isNMR) {
	    let xData = variables.x.data;
	    let yData = variables.y.data;
	    checkNumberOrArray(xData);
	    checkNumberOrArray(yData);

	    if (options.isPeakData) {
	      header += `##DATA TABLE= (XY..XY), PEAKS\n`;

	      for (let point = 0; point < varDim[0]; point++) {
	        header += `${xData[point]}, ${yData[point]}\n`;
	      }
	    } else if (options.isXYData) {
	      const firstX = xData[0];
	      const lastX = xData[xData.length - 1];
	      const deltaX = (lastX - firstX) / xData.length;

	      for (const key of ['r', 'i']) {
	        const variable = variables[key];

	        if (variable) {
	          checkNumberOrArray(variable.data);
	          header += `##PAGE= ${key === 'r' ? 1 : 2}\n`;
	          header += `##DATA TABLE= (X++(${key === 'r' ? 'R..R' : 'I..I'})), XYDATA\n`;
	          header += vectorEncoder(xDivide(variable.data, yFactor, {
	            output: variable.data
	          }), firstX / xFactor, deltaX / xFactor, xyEncoding);
	        }
	      }
	    }
	  } else {
	    header += `##PAGE= N=1\n`;
	    header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;

	    for (let i = 0; i < variables.x.data.length; i++) {
	      let point = [];

	      for (let key of keys) {
	        let variable = variables[key];
	        if (!variable) continue;
	        point.push(variable.data[i]);
	      }

	      header += `${point.join('\t')}\n`;
	    }
	  }

	  header += `##NTUPLES= ${dataType}\n`;
	  header += '##END=';
	  return header;
	}

	function getFactorNumber(minMax) {
	  let maxValue = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 2 ** 31 - 1;
	  let factor;

	  if (minMax.min < 0) {
	    if (minMax.max > 0) {
	      factor = Math.max(-minMax.min, minMax.max) / maxValue;
	    } else {
	      factor = -minMax.min / maxValue;
	    }
	  } else {
	    factor = minMax.max / maxValue;
	  }

	  return factor;
	}

	function getBestFactor(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    maxValue,
	    factor,
	    minMax
	  } = options;

	  if (factor !== undefined) {
	    return factor;
	  } // is there non integer number ?


	  let onlyInteger = true;

	  for (let y of array) {
	    if (Math.round(y) !== y) {
	      onlyInteger = false;
	      break;
	    }
	  }

	  if (onlyInteger) {
	    return 1;
	  } // we need to rescale the values
	  // need to find the max and min values


	  const extremeValues = minMax || xMinMaxValues(array);
	  return getFactorNumber(extremeValues, maxValue);
	}

	/**
	 * Reconvert number to original value
	 * @param number Number used for computation
	 * @param factor Multiplying factor
	 * @returns Original value
	 */
	function getNumber(number, factor) {
	  if (factor !== 1) number /= factor;
	  const rounded = Math.round(number);

	  if (rounded !== number && Math.abs(rounded - number) <= Number.EPSILON) {
	    return rounded;
	  }

	  return number;
	}

	function peakTableCreator(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    xFactor = 1,
	    yFactor = 1
	  } = options.info || {};
	  let firstX = Number.POSITIVE_INFINITY;
	  let lastX = Number.NEGATIVE_INFINITY;
	  let firstY = Number.POSITIVE_INFINITY;
	  let lastY = Number.NEGATIVE_INFINITY;
	  let lines = [];

	  for (let i = 0; i < data.x.length; i++) {
	    let x = data.x[i];
	    let y = data.y[i];

	    if (firstX > x) {
	      firstX = x;
	    }

	    if (lastX < x) {
	      lastX = x;
	    }

	    if (firstY > y) {
	      firstY = y;
	    }

	    if (lastY < y) {
	      lastY = y;
	    }
	  }

	  lines.push(`##FIRSTX=${firstX}`);
	  lines.push(`##LASTX=${lastX}`);
	  lines.push(`##FIRSTY=${firstY}`);
	  lines.push(`##LASTY=${lastY}`);
	  lines.push(`##XFACTOR=${xFactor}`);
	  lines.push(`##YFACTOR=${yFactor}`);
	  lines.push('##PEAK TABLE=(XY..XY)');

	  for (let i = 0; i < data.x.length; i++) {
	    lines.push(`${getNumber(data.x[i], xFactor)} ${getNumber(data.y[i], yFactor)}`);
	  }

	  return lines;
	}

	function rescaleAndEnsureInteger(data, factor) {
	  if (factor === 1) return data.map(value => Math.round(value));
	  return xDivide(data, factor);
	}

	function xyDataCreator(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    xyEncoding = 'DIF'
	  } = options;
	  const {
	    xFactor = 1,
	    yFactor = 1
	  } = options.info || {};
	  let firstX = data.x[0];
	  let lastX = data.x[data.x.length - 1];
	  let firstY = data.y[0];
	  let lastY = data.y[data.y.length - 1];
	  let nbPoints = data.x.length;
	  let deltaX = (lastX - firstX) / (nbPoints - 1);
	  let lines = [];
	  lines.push(`##FIRSTX=${firstX}`);
	  lines.push(`##LASTX=${lastX}`);
	  lines.push(`##FIRSTY=${firstY}`);
	  lines.push(`##LASTY=${lastY}`);
	  lines.push(`##DELTAX=${deltaX}`);
	  lines.push(`##XFACTOR=${xFactor}`);
	  lines.push(`##YFACTOR=${yFactor}`);
	  lines.push('##XYDATA=(X++(Y..Y))');
	  let line = vectorEncoder(rescaleAndEnsureInteger(data.y, yFactor), firstX / xFactor, deltaX / xFactor, xyEncoding);
	  if (line) lines.push(line);
	  return lines;
	}

	const infoDefaultKeys = ['title', 'owner', 'origin', 'dataType', 'xUnits', 'yUnits', 'xFactor', 'yFactor'];
	/**
	 * Create a jcamp
	 * @param data object of array
	 * @param [options={meta:{},info:{}} - metadata object
	 * @returns JCAMP of the input
	 */

	function fromJSON(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    meta = {},
	    info = {},
	    xyEncoding
	  } = options;
	  let {
	    title = '',
	    owner = '',
	    origin = '',
	    dataType = '',
	    xUnits = '',
	    yUnits = '',
	    xFactor,
	    yFactor
	  } = info;
	  data = {
	    x: data.x,
	    y: data.y
	  };
	  let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}\n`;
	  const infoKeys = Object.keys(info).filter(keys => !infoDefaultKeys.includes(keys));
	  header += addInfoData(info, infoKeys, '##');
	  header += addInfoData(meta); // we leave the header and utf8 fonts ${header.replace(/[^\t\n\x20-\x7F]/g, '')

	  if (xyEncoding) {
	    xFactor = getBestFactor(data.x, {
	      factor: xFactor
	    });
	    yFactor = getBestFactor(data.y, {
	      factor: yFactor
	    });
	    return `${header}##NPOINTS=${data.x.length}
${xyDataCreator(data, {
      info: {
        xFactor,
        yFactor
      },
      xyEncoding
    }).join('\n')}
##END=`;
	  } else {
	    if (xFactor === undefined) xFactor = 1;
	    if (yFactor === undefined) yFactor = 1;

	    if (xFactor !== 1) {
	      //@ts-expect-error xFactor is always defined
	      data.x = data.x.map(value => value / xFactor);
	    }

	    if (yFactor !== 1) {
	      //@ts-expect-error yFactor is always defined
	      data.y = data.y.map(value => value / yFactor);
	    }

	    return `${header}##NPOINTS=${data.x.length}
${peakTableCreator(data, {
      info: {
        xFactor,
        yFactor
      }
    }).join('\n')}
##END=`;
	  }
	}

	/**
	 * Create a jcamp from variables
	 */

	function fromVariables(
	/** object of variables */
	variables) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    info = {},
	    meta = {},
	    forceNtuples = false
	  } = options;
	  let jcampOptions = {
	    info,
	    meta
	  };
	  let keys = Object.keys(variables).map(key => key.toLowerCase());

	  if (keys.length === 2 && keys.includes('x') && keys.includes('y') && !forceNtuples) {
	    let x = variables.x;
	    let xLabel = x.label || 'x';

	    if (variables.x.units) {
	      if (xLabel.includes(variables.x.units)) {
	        jcampOptions.info.xUnits = xLabel;
	      } else {
	        jcampOptions.info.xUnits = `${xLabel} (${variables.x.units})`;
	      }
	    } else {
	      jcampOptions.info.xUnits = xLabel;
	    }

	    let y = variables.y;
	    let yLabel = y.label || 'y';

	    if (variables.y.units) {
	      if (yLabel.includes(variables.y.units)) {
	        jcampOptions.info.xUnits = yLabel;
	      } else {
	        jcampOptions.info.yUnits = `${yLabel} (${variables.y.units})`;
	      }
	    } else {
	      jcampOptions.info.yUnits = yLabel;
	    }

	    const xData = variables.x.data;
	    const yData = variables.y.data;
	    checkNumberOrArray(xData);
	    checkNumberOrArray(yData);
	    return fromJSON({
	      x: xData,
	      y: yData
	    }, jcampOptions);
	  } else {
	    return creatorNtuples(variables, options);
	  }
	}

	function toJcamps(analysis) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let jcamps = [];

	  for (let spectrum of analysis.spectra) {
	    jcamps.push(getJcamp(spectrum, options));
	  }

	  return jcamps;
	}

	function getJcamp(spectrum, options) {
	  const {
	    info = {},
	    meta = {}
	  } = options;
	  let jcampOptions = {
	    options: {},
	    info: {
	      title: spectrum.title,
	      dataType: spectrum.dataType,
	      ...info
	    },
	    meta: { ...spectrum.meta,
	      ...meta
	    }
	  };
	  return fromVariables(spectrum.variables, jcampOptions);
	}

	function toJcamp(analysis) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return toJcamps(analysis, options).join('\n');
	}

	const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
	const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
	const ROOT_THREE = Math.sqrt(3);
	const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
	const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

	// https://en.wikipedia.org/wiki/Error_function#Inverse_functions
	// This code yields to a good approximation
	// If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
	function erfinv(x) {
	  let a = 0.147;
	  if (x === 0) return 0;
	  let ln1MinusXSqrd = Math.log(1 - x * x);
	  let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
	  let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
	  let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
	  return secondSqrt * (x > 0 ? 1 : -1);
	}

	class Gaussian {
	  constructor() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      fwhm = 500,
	      sd
	    } = options;
	    this.fwhm = sd ? gaussianWidthToFWHM(2 * sd) : fwhm;
	  }

	  fwhmToWidth() {
	    let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
	    return gaussianFwhmToWidth(fwhm);
	  }

	  widthToFWHM(width) {
	    return gaussianWidthToFWHM(width);
	  }

	  fct(x) {
	    return gaussianFct(x, this.fwhm);
	  }

	  getArea() {
	    let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : calculateGaussianHeight({
	      fwhm: this.fwhm
	    });
	    return getGaussianArea({
	      fwhm: this.fwhm,
	      height
	    });
	  }

	  getFactor(area) {
	    return getGaussianFactor(area);
	  }

	  getData() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    return getGaussianData(this, options);
	  }

	  calculateHeight() {
	    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
	    return calculateGaussianHeight({
	      fwhm: this.fwhm,
	      area
	    });
	  }

	  getParameters() {
	    return ['fwhm'];
	  }

	}
	function calculateGaussianHeight(options) {
	  let {
	    fwhm = 500,
	    area = 1,
	    sd
	  } = options;
	  if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
	  return 2 * area / ROOT_PI_OVER_LN2 / fwhm;
	}
	function gaussianFct(x, fwhm) {
	  return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
	}
	function gaussianWidthToFWHM(width) {
	  return width * ROOT_2LN2;
	}
	function gaussianFwhmToWidth(fwhm) {
	  return fwhm / ROOT_2LN2;
	}
	function getGaussianArea(options) {
	  let {
	    fwhm = 500,
	    sd,
	    height = 1
	  } = options;
	  if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
	  return height * ROOT_PI_OVER_LN2 * fwhm / 2;
	}
	function getGaussianFactor() {
	  let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
	  return Math.sqrt(2) * erfinv(area);
	}
	function getGaussianData() {
	  let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fwhm = 500,
	    sd
	  } = shape;
	  if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
	  let {
	    length,
	    factor = getGaussianFactor(),
	    height = calculateGaussianHeight({
	      fwhm
	    })
	  } = options;

	  if (!length) {
	    length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
	    if (length % 2 === 0) length++;
	  }

	  const center = (length - 1) / 2;
	  const data = new Float64Array(length);

	  for (let i = 0; i <= center; i++) {
	    data[i] = gaussianFct(i - center, fwhm) * height;
	    data[length - 1 - i] = data[i];
	  }

	  return data;
	}

	class Lorentzian {
	  constructor() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      fwhm = 500
	    } = options;
	    this.fwhm = fwhm;
	  }

	  fwhmToWidth() {
	    let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
	    return lorentzianFwhmToWidth(fwhm);
	  }

	  widthToFWHM(width) {
	    return lorentzianWidthToFWHM(width);
	  }

	  fct(x) {
	    return lorentzianFct(x, this.fwhm);
	  }

	  getArea() {
	    let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
	    return getLorentzianArea({
	      fwhm: this.fwhm,
	      height
	    });
	  }

	  getFactor(area) {
	    return getLorentzianFactor(area);
	  }

	  getData() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    return getLorentzianData(this, options);
	  }

	  calculateHeight() {
	    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
	    return calculateLorentzianHeight({
	      fwhm: this.fwhm,
	      area
	    });
	  }

	  getParameters() {
	    return ['fwhm'];
	  }

	}
	const calculateLorentzianHeight = _ref => {
	  let {
	    fwhm = 1,
	    area = 1
	  } = _ref;
	  return 2 * area / Math.PI / fwhm;
	};
	const lorentzianFct = (x, fwhm) => {
	  return Math.pow(fwhm, 2) / (4 * Math.pow(x, 2) + Math.pow(fwhm, 2));
	};
	const lorentzianWidthToFWHM = width => {
	  return width * ROOT_THREE;
	};
	const lorentzianFwhmToWidth = fwhm => {
	  return fwhm / ROOT_THREE;
	};
	const getLorentzianArea = options => {
	  const {
	    fwhm = 500,
	    height = 1
	  } = options;
	  return height * Math.PI * fwhm / 2;
	};
	const getLorentzianFactor = function () {
	  let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
	  return 2 * Math.tan(Math.PI * (area - 0.5));
	};
	const getLorentzianData = function () {
	  let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fwhm = 500
	  } = shape;
	  let {
	    length,
	    factor = getLorentzianFactor(),
	    height = calculateLorentzianHeight({
	      fwhm,
	      area: 1
	    })
	  } = options;

	  if (!length) {
	    length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
	    if (length % 2 === 0) length++;
	  }

	  const center = (length - 1) / 2;
	  const data = new Float64Array(length);

	  for (let i = 0; i <= center; i++) {
	    data[i] = lorentzianFct(i - center, fwhm) * height;
	    data[length - 1 - i] = data[i];
	  }

	  return data;
	};

	class PseudoVoigt {
	  constructor() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      fwhm = 500,
	      mu = 0.5
	    } = options;
	    this.mu = mu;
	    this.fwhm = fwhm;
	  }

	  fwhmToWidth() {
	    let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
	    let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
	    return pseudoVoigtFwhmToWidth(fwhm, mu);
	  }

	  widthToFWHM(width) {
	    let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
	    return pseudoVoigtWidthToFWHM(width, mu);
	  }

	  fct(x) {
	    return pseudoVoigtFct(x, this.fwhm, this.mu);
	  }

	  getArea() {
	    let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
	    return getPseudoVoigtArea({
	      fwhm: this.fwhm,
	      height,
	      mu: this.mu
	    });
	  }

	  getFactor(area) {
	    return getPseudoVoigtFactor(area);
	  }

	  getData() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      length,
	      factor,
	      height = calculatePseudoVoigtHeight({
	        fwhm: this.fwhm,
	        mu: this.mu,
	        area: 1
	      })
	    } = options;
	    return getPseudoVoigtData(this, {
	      factor,
	      length,
	      height
	    });
	  }

	  calculateHeight() {
	    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
	    return calculatePseudoVoigtHeight({
	      fwhm: this.fwhm,
	      mu: this.mu,
	      area
	    });
	  }

	  getParameters() {
	    return ['fwhm', 'mu'];
	  }

	}
	const calculatePseudoVoigtHeight = function () {
	  let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let {
	    fwhm = 1,
	    mu = 0.5,
	    area = 1
	  } = options;
	  return 2 * area / (fwhm * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI));
	};
	const pseudoVoigtFct = (x, fwhm, mu) => {
	  return (1 - mu) * lorentzianFct(x, fwhm) + mu * gaussianFct(x, fwhm);
	};
	const pseudoVoigtWidthToFWHM = function (width) {
	  let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
	  return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
	};
	const pseudoVoigtFwhmToWidth = function (fwhm) {
	  let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
	  return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
	};
	const getPseudoVoigtArea = options => {
	  const {
	    fwhm = 500,
	    height = 1,
	    mu = 0.5
	  } = options;
	  return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
	};
	const getPseudoVoigtFactor = function () {
	  let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
	  let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
	  return mu < 1 ? getLorentzianFactor(area) : getGaussianFactor(area);
	};
	const getPseudoVoigtData = function () {
	  let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fwhm = 500,
	    mu = 0.5
	  } = shape;
	  let {
	    length,
	    factor = getPseudoVoigtFactor(0.999, mu),
	    height = calculatePseudoVoigtHeight({
	      fwhm,
	      mu,
	      area: 1
	    })
	  } = options;

	  if (!height) {
	    height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
	  }

	  if (!length) {
	    length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
	    if (length % 2 === 0) length++;
	  }

	  const center = (length - 1) / 2;
	  const data = new Float64Array(length);

	  for (let i = 0; i <= center; i++) {
	    data[i] = pseudoVoigtFct(i - center, fwhm, mu) * height;
	    data[length - 1 - i] = data[i];
	  }

	  return data;
	};

	/**
	 * This function calculates the spectrum as a sum of gaussian functions. The Gaussian
	 * parameters are divided in 3 batches. 1st: centers; 2nd: height; 3th: widths;
	 * @param parameters - Gaussian parameters
	 */

	function sumOfGaussians(parameters) {
	  const nL = parameters.length / 3;
	  const gaussian = new Gaussian();
	  return x => {
	    let y = 0;

	    for (let i = 0; i < nL; i++) {
	      gaussian.fwhm = parameters[i + nL * 2];
	      y += parameters[i + nL] * gaussian.fct(x - parameters[i]);
	    }

	    return y;
	  };
	}

	/**
	 * This function calculates the spectrum as a sum of lorentzian functions. The Lorentzian
	 * parameters are divided in 3 batches. 1st: centers; 2nd: heights; 3th: widths;
	 * @param parameters - Lorentzian parameters
	 */

	function sumOfLorentzians(parameters) {
	  const lorentzian = new Lorentzian();
	  return x => {
	    let nL = parameters.length / 3;
	    let y = 0;

	    for (let i = 0; i < nL; i++) {
	      lorentzian.fwhm = parameters[i + nL * 2];
	      y += parameters[i + nL] * lorentzian.fct(x - parameters[i]);
	    }

	    return y;
	  };
	}

	/**
	 * This function calculates the spectrum as a sum of linear combination of gaussian and lorentzian functions. The pseudo voigt
	 * parameters are divided in 4 batches. 1st: centers; 2nd: heights; 3th: widths; 4th: mu's ;
	 * @param parameters Lorentzian parameters
	 */

	function sumOfPseudoVoigts(parameters) {
	  const pseudoVoigt = new PseudoVoigt();
	  return x => {
	    let nL = parameters.length / 4;
	    let y = 0;

	    for (let i = 0; i < nL; i++) {
	      pseudoVoigt.fwhm = parameters[i + nL * 2];
	      pseudoVoigt.mu = parameters[i + nL * 3];
	      y += parameters[i + nL] * pseudoVoigt.fct(x - parameters[i]);
	    }

	    return y;
	  };
	}

	const isValidKey = key => {
	  return key !== '__proto__' && key !== 'constructor' && key !== 'prototype';
	};

	const isObject = val => {
	  return typeof val === 'object';
	};

	const isPrimitive = val => {
	  return typeof val === 'object' ? val === null : typeof val !== 'function';
	};
	/** Algorithm to assign deep
	 * @param target
	 */


	function assignDeep(target) {
	  let index = 0;

	  for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
	    args[_key - 1] = arguments[_key];
	  }

	  if (isPrimitive(target)) target = args[index++];
	  if (!target) target = {};

	  for (; index < args.length; index++) {
	    if (!isObject(args[index])) continue;

	    for (const key in args[index]) {
	      if (!isValidKey(key)) continue;

	      if (isObject(target[key]) && isObject(args[index][key])) {
	        assignDeep(target[key], args[index][key]);
	      } else {
	        target[key] = args[index][key];
	      }
	    }
	  }

	  return target;
	}

	let xObject = {
	  init: peak => peak.x,
	  max: peak => peak.x + peak.fwhm * 2,
	  min: peak => peak.x - peak.fwhm * 2,
	  gradientDifference: peak => peak.fwhm * 2e-3
	};
	let yObject = {
	  init: peak => peak.y,
	  max: () => 1.5,
	  min: () => 0,
	  gradientDifference: () => 1e-3
	};
	let fwhmObject = {
	  init: peak => peak.fwhm,
	  max: peak => peak.fwhm * 4,
	  min: peak => peak.fwhm * 0.25,
	  gradientDifference: peak => peak.fwhm * 2e-3
	};
	let muObject = {
	  init: peak => peak.shape && peak.shape.mu !== undefined ? peak.shape.mu : 0.5,
	  min: () => 0,
	  max: () => 1,
	  gradientDifference: () => 0.01
	};
	/** Algorithm to check the input
	 * @param data - Data to check
	 * @param peakList - List of peaks
	 * @param options - Options for optimization
	 */

	function checkInput(data, peakList, options) {
	  let {
	    shape = {
	      kind: 'gaussian'
	    },
	    optimization = {
	      kind: 'lm'
	    }
	  } = options;
	  let peaks = JSON.parse(JSON.stringify(peakList));
	  let kind = shape.kind.toLowerCase().replace(/[^a-z]/g, '');
	  let paramsFunc;
	  let defaultParameters;

	  switch (kind) {
	    case 'gaussian':
	      paramsFunc = sumOfGaussians;
	      defaultParameters = {
	        x: xObject,
	        y: yObject,
	        fwhm: fwhmObject
	      };
	      break;

	    case 'lorentzian':
	      paramsFunc = sumOfLorentzians;
	      defaultParameters = {
	        x: xObject,
	        y: yObject,
	        fwhm: fwhmObject
	      };
	      break;

	    case 'pseudovoigt':
	      paramsFunc = sumOfPseudoVoigts;
	      defaultParameters = {
	        x: xObject,
	        y: yObject,
	        fwhm: fwhmObject,
	        mu: muObject
	      };
	      break;

	    default:
	      throw new Error('kind of shape is not supported');
	  }

	  let x = data.x;
	  let maxY = max(data.y);
	  let y = new Array(x.length);
	  let minY = Number.MAX_VALUE; // we need to move the data down to the baseline 0

	  for (let i = 0; i < x.length; i++) {
	    y[i] = data.y[i];

	    if (data.y[i] < minY) {
	      minY = data.y[i];
	    }
	  } // removing minY from each y, dividing by max


	  for (let i = 0; i < x.length; i++) {
	    y[i] = (y[i] - minY) / maxY;
	  }

	  peaks.forEach(peak => {
	    peak.y /= maxY;
	    peak.shape = {
	      kind: shape.kind,
	      ...peak.shape
	    };
	  });
	  let parameters = assignDeep({}, defaultParameters, optimization.parameters);

	  for (let key in parameters) {
	    for (let par in parameters[key]) {
	      if (!Array.isArray(parameters[key][par])) {
	        parameters[key][par] = [parameters[key][par]];
	      }

	      if (![peaks.length, 1].includes(parameters[key][par].length)) {
	        throw new Error(`The length of ${key}-${par} is not correct`);
	      }

	      for (let index = 0; index < parameters[key][par].length; index++) {
	        if (typeof parameters[key][par][index] === 'number') {
	          let value = parameters[key][par][index];

	          parameters[key][par][index] = () => value;
	        }
	      }
	    }
	  }

	  optimization.parameters = parameters;
	  return {
	    y,
	    x,
	    maxY,
	    minY,
	    peaks,
	    paramsFunc,
	    optimization
	  };
	}

	function checkOptions$1(data, parameterizedFunction, options) {
	  let {
	    timeout,
	    minValues,
	    maxValues,
	    initialValues,
	    weights = 1,
	    damping = 1e-2,
	    dampingStepUp = 11,
	    dampingStepDown = 9,
	    maxIterations = 100,
	    errorTolerance = 1e-7,
	    centralDifference = false,
	    gradientDifference = 10e-2,
	    improvementThreshold = 1e-3
	  } = options;

	  if (damping <= 0) {
	    throw new Error('The damping option must be a positive number');
	  } else if (!data.x || !data.y) {
	    throw new Error('The data parameter must have x and y elements');
	  } else if (!isAnyArray(data.x) || data.x.length < 2 || !isAnyArray(data.y) || data.y.length < 2) {
	    throw new Error('The data parameter elements must be an array with more than 2 points');
	  } else if (data.x.length !== data.y.length) {
	    throw new Error('The data parameter elements must have the same size');
	  }

	  let parameters = initialValues || new Array(parameterizedFunction.length).fill(1);
	  let nbPoints = data.y.length;
	  let parLen = parameters.length;
	  maxValues = maxValues || new Array(parLen).fill(Number.MAX_SAFE_INTEGER);
	  minValues = minValues || new Array(parLen).fill(Number.MIN_SAFE_INTEGER);

	  if (maxValues.length !== minValues.length) {
	    throw new Error('minValues and maxValues must be the same size');
	  }

	  if (!isAnyArray(parameters)) {
	    throw new Error('initialValues must be an array');
	  }

	  if (typeof gradientDifference === 'number') {
	    gradientDifference = new Array(parameters.length).fill(gradientDifference);
	  } else if (isAnyArray(gradientDifference)) {
	    if (gradientDifference.length !== parLen) {
	      gradientDifference = new Array(parLen).fill(gradientDifference[0]);
	    }
	  } else {
	    throw new Error('gradientDifference should be a number or array with length equal to the number of parameters');
	  }

	  let filler;

	  if (typeof weights === 'number') {
	    let value = 1 / weights ** 2;

	    filler = () => value;
	  } else if (isAnyArray(weights)) {
	    if (weights.length < data.x.length) {
	      let value = 1 / weights[0] ** 2;

	      filler = () => value;
	    } else {
	      filler = i => 1 / weights[i] ** 2;
	    }
	  } else {
	    throw new Error('weights should be a number or array with length equal to the number of data points');
	  }

	  let checkTimeout;

	  if (timeout !== undefined) {
	    if (typeof timeout !== 'number') {
	      throw new Error('timeout should be a number');
	    }

	    let endTime = Date.now() + timeout * 1000;

	    checkTimeout = () => Date.now() > endTime;
	  } else {
	    checkTimeout = () => false;
	  }

	  let weightSquare = new Array(data.x.length);

	  for (let i = 0; i < nbPoints; i++) {
	    weightSquare[i] = filler(i);
	  }

	  return {
	    checkTimeout,
	    minValues,
	    maxValues,
	    parameters,
	    weightSquare,
	    damping,
	    dampingStepUp,
	    dampingStepDown,
	    maxIterations,
	    errorTolerance,
	    centralDifference,
	    gradientDifference,
	    improvementThreshold
	  };
	}

	/**
	 * the sum of the weighted squares of the errors (or weighted residuals) between the data.y
	 * and the curve-fit function.
	 * @ignore
	 * @param {{x:ArrayLike<number>, y:ArrayLike<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
	 * @param {ArrayLike<number>} parameters - Array of current parameter values
	 * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
	 * @param {ArrayLike<number>} weightSquare - Square of weights
	 * @return {number}
	 */
	function errorCalculation(data, parameters, parameterizedFunction, weightSquare) {
	  let error = 0;
	  const func = parameterizedFunction(parameters);

	  for (let i = 0; i < data.x.length; i++) {
	    error += Math.pow(data.y[i] - func(data.x[i]), 2) / weightSquare[i];
	  }

	  return error;
	}

	/**
	 * Difference of the matrix function over the parameters
	 * @ignore
	 * @param {{x:ArrayLike<number>, y:ArrayLike<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
	 * @param {ArrayLike<number>} evaluatedData - Array of previous evaluated function values
	 * @param {Array<number>} params - Array of previous parameter values
	 * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
	 * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
	 * @param {function} paramFunction - The parameters and returns a function with the independent variable as a parameter
	 * @return {Matrix}
	 */

	function gradientFunction(data, evaluatedData, params, gradientDifference, paramFunction, centralDifference) {
	  const nbParams = params.length;
	  const nbPoints = data.x.length;
	  let ans = Matrix.zeros(nbParams, nbPoints);
	  let rowIndex = 0;

	  for (let param = 0; param < nbParams; param++) {
	    if (gradientDifference[param] === 0) continue;
	    let delta = gradientDifference[param];
	    let auxParams = params.slice();
	    auxParams[param] += delta;
	    let funcParam = paramFunction(auxParams);

	    if (!centralDifference) {
	      for (let point = 0; point < nbPoints; point++) {
	        ans.set(rowIndex, point, (evaluatedData[point] - funcParam(data.x[point])) / delta);
	      }
	    } else {
	      auxParams = params.slice();
	      auxParams[param] -= delta;
	      delta *= 2;
	      let funcParam2 = paramFunction(auxParams);

	      for (let point = 0; point < nbPoints; point++) {
	        ans.set(rowIndex, point, (funcParam2(data.x[point]) - funcParam(data.x[point])) / delta);
	      }
	    }

	    rowIndex++;
	  }

	  return ans;
	}

	/**
	 * Matrix function over the samples
	 * @ignore
	 * @param {{x:ArrayLike<number>, y:ArrayLike<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
	 * @param {ArrayLike<number>} evaluatedData - Array of previous evaluated function values
	 * @return {Matrix}
	 */

	function matrixFunction(data, evaluatedData) {
	  const m = data.x.length;
	  let ans = new Matrix(m, 1);

	  for (let point = 0; point < m; point++) {
	    ans.set(point, 0, data.y[point] - evaluatedData[point]);
	  }

	  return ans;
	}
	/**
	 * Iteration for Levenberg-Marquardt
	 * @ignore
	 * @param {{x:ArrayLike<number>, y:ArrayLike<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
	 * @param {Array<number>} params - Array of previous parameter values
	 * @param {number} damping - Levenberg-Marquardt parameter
	 * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
	 * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
	 * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
	 */


	function step(data, params, damping, gradientDifference, parameterizedFunction, centralDifference, weights) {
	  let value = damping;
	  let identity = Matrix.eye(params.length, params.length, value);
	  const func = parameterizedFunction(params);
	  let evaluatedData = new Float64Array(data.x.length);

	  for (let i = 0; i < data.x.length; i++) {
	    evaluatedData[i] = func(data.x[i]);
	  }

	  let gradientFunc = gradientFunction(data, evaluatedData, params, gradientDifference, parameterizedFunction, centralDifference);
	  let residualError = matrixFunction(data, evaluatedData);
	  let inverseMatrix = inverse(identity.add(gradientFunc.mmul(gradientFunc.transpose().scale('row', {
	    scale: weights
	  }))));
	  let jacobianWeightResidualError = gradientFunc.mmul(residualError.scale('row', {
	    scale: weights
	  }));
	  let perturbations = inverseMatrix.mmul(jacobianWeightResidualError);
	  return {
	    perturbations,
	    jacobianWeightResidualError
	  };
	}

	/**
	 * Curve fitting algorithm
	 * @param {{x:ArrayLike<number>, y:ArrayLike<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
	 * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
	 * @param {object} [options] - Options object
	 * @param {number|ArrayLike<number>} [options.weights = 1] - weighting vector, if the length does not match with the number of data points, the vector is reconstructed with first value.
	 * @param {number} [options.damping = 1e-2] - Levenberg-Marquardt parameter, small values of the damping parameter λ result in a Gauss-Newton update and large
	values of λ result in a gradient descent update
	 * @param {number} [options.dampingStepDown = 9] - factor to reduce the damping (Levenberg-Marquardt parameter) when there is not an improvement when updating parameters.
	 * @param {number} [options.dampingStepUp = 11] - factor to increase the damping (Levenberg-Marquardt parameter) when there is an improvement when updating parameters.
	 * @param {number} [options.improvementThreshold = 1e-3] - the threshold to define an improvement through an update of parameters
	 * @param {number|ArrayLike<number>} [options.gradientDifference = 10e-2] - The step size to approximate the jacobian matrix
	 * @param {boolean} [options.centralDifference = false] - If true the jacobian matrix is approximated by central differences otherwise by forward differences
	 * @param {ArrayLike<number>} [options.minValues] - Minimum allowed values for parameters
	 * @param {ArrayLike<number>} [options.maxValues] - Maximum allowed values for parameters
	 * @param {ArrayLike<number>} [options.initialValues] - Array of initial parameter values
	 * @param {number} [options.maxIterations = 100] - Maximum of allowed iterations
	 * @param {number} [options.errorTolerance = 10e-3] - Minimum uncertainty allowed for each point.
	 * @param {number} [options.timeout] - maximum time running before throw in seconds.
	 * @return {{parameterValues: Array<number>, parameterError: number, iterations: number}}
	 */

	function levenbergMarquardt(data, parameterizedFunction) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    checkTimeout,
	    minValues,
	    maxValues,
	    parameters,
	    weightSquare,
	    damping,
	    dampingStepUp,
	    dampingStepDown,
	    maxIterations,
	    errorTolerance,
	    centralDifference,
	    gradientDifference,
	    improvementThreshold
	  } = checkOptions$1(data, parameterizedFunction, options);
	  let error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
	  let optimalError = error;
	  let optimalParameters = parameters.slice();
	  let converged = error <= errorTolerance;
	  let iteration = 0;

	  for (; iteration < maxIterations && !converged; iteration++) {
	    let previousError = error;
	    let {
	      perturbations,
	      jacobianWeightResidualError
	    } = step(data, parameters, damping, gradientDifference, parameterizedFunction, centralDifference, weightSquare);

	    for (let k = 0; k < parameters.length; k++) {
	      parameters[k] = Math.min(Math.max(minValues[k], parameters[k] - perturbations.get(k, 0)), maxValues[k]);
	    }

	    error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
	    if (isNaN(error)) break;

	    if (error < optimalError - errorTolerance) {
	      optimalError = error;
	      optimalParameters = parameters.slice();
	    }

	    let improvementMetric = (previousError - error) / perturbations.transpose().mmul(perturbations.mul(damping).add(jacobianWeightResidualError)).get(0, 0);

	    if (improvementMetric > improvementThreshold) {
	      damping = Math.max(damping / dampingStepDown, 1e-7);
	    } else {
	      damping = Math.min(damping * dampingStepUp, 1e7);
	    }

	    if (checkTimeout()) {
	      throw new Error(`The execution time is over to ${options.timeout} seconds`);
	    }

	    converged = error <= errorTolerance;
	  }

	  return {
	    parameterValues: optimalParameters,
	    parameterError: optimalError,
	    iterations: iteration
	  };
	}

	const LEVENBERG_MARQUARDT = 1;
	/** Algorithm to select the method.
	 * @param optimizationOptions - Optimization options
	 * @returns - The algorithm and optimization options
	 */

	function selectMethod() {
	  let optimizationOptions = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let {
	    kind,
	    options
	  } = optimizationOptions;
	  kind = getKind(kind);

	  switch (kind) {
	    case LEVENBERG_MARQUARDT:
	      return {
	        algorithm: levenbergMarquardt,
	        optimizationOptions: checkOptions(kind, options)
	      };

	    default:
	      throw new Error(`Unknown kind algorithm`);
	  }
	}

	function checkOptions(kind) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  switch (kind) {
	    case LEVENBERG_MARQUARDT:
	      return Object.assign({}, lmOptions, options);

	    default:
	      throw new Error(`unknown kind: ${kind}`);
	  }
	}

	function getKind(kind) {
	  if (typeof kind !== 'string') return kind;

	  switch (kind.toLowerCase().replace(/[^a-z]/g, '')) {
	    case 'lm':
	    case 'levenbergmarquardt':
	      return LEVENBERG_MARQUARDT;

	    default:
	      throw new Error(`Unknown kind algorithm`);
	  }
	}

	const lmOptions = {
	  damping: 1.5,
	  maxIterations: 100,
	  errorTolerance: 1e-8
	};

	/**
	 * Fits a set of points to the sum of a set of bell functions.
	 *
	 * @param data - An object containing the x and y data to be fitted.
	 * @param peakList - A list of initial parameters to be optimized. e.g. coming from a peak picking [{x, y, width}].
	 * @param options - Options.
	 * @returns - An object with fitting error and the list of optimized parameters { parameters: [ {x, y, width} ], error } if the kind of shape is pseudoVoigt mu parameter is optimized.
	 */

	function optimize(data, peakList) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	  if (!options.shape) {
	    options = { ...options,
	      ...{
	        shape: {
	          kind: 'gaussian'
	        }
	      }
	    };
	  }

	  const {
	    y,
	    x,
	    maxY,
	    minY,
	    peaks,
	    paramsFunc,
	    optimization
	  } = checkInput(data, peakList, options);
	  let parameters = optimization.parameters;
	  let nbShapes = peaks.length;
	  let parameterKey = Object.keys(parameters);
	  let nbParams = nbShapes * parameterKey.length;
	  let pMin = new Float64Array(nbParams);
	  let pMax = new Float64Array(nbParams);
	  let pInit = new Float64Array(nbParams);
	  let gradientDifference = new Float64Array(nbParams);

	  for (let i = 0; i < nbShapes; i++) {
	    let peak = peaks[i];

	    for (let k = 0; k < parameterKey.length; k++) {
	      let key = parameterKey[k];
	      let init = parameters[key].init;
	      let min = parameters[key].min;
	      let max = parameters[key].max;
	      let gradientDifferenceValue = parameters[key].gradientDifference;
	      pInit[i + k * nbShapes] = init[i % init.length](peak);
	      pMin[i + k * nbShapes] = min[i % min.length](peak);
	      pMax[i + k * nbShapes] = max[i % max.length](peak);
	      gradientDifference[i + k * nbShapes] = gradientDifferenceValue[i % gradientDifferenceValue.length](peak);
	    }
	  }

	  let {
	    algorithm,
	    optimizationOptions
	  } = selectMethod(optimization);
	  optimizationOptions.minValues = pMin;
	  optimizationOptions.maxValues = pMax;
	  optimizationOptions.initialValues = pInit;
	  optimizationOptions.gradientDifference = gradientDifference;
	  let pFit = algorithm({
	    x,
	    y
	  }, paramsFunc, optimizationOptions);
	  let {
	    parameterError: error,
	    iterations
	  } = pFit;
	  let result = {
	    error,
	    iterations,
	    peaks
	  };

	  for (let i = 0; i < nbShapes; i++) {
	    for (let k = 0; k < parameterKey.length; k++) {
	      const key = parameterKey[k];
	      const value = pFit.parameterValues[i + k * nbShapes];

	      if (key === 'x' || key === 'fwhm') {
	        peaks[i][key] = value;
	      } else if (key === 'y') {
	        peaks[i][key] = value * maxY + minY;
	      } else {
	        peaks[i].shape[key] = value;
	      }
	    }
	  }

	  return result;
	}

	/** Based on a x value we will return a peak
	 * if you set optimize=True the returned positions will be
	 * the closest actual datapoints to the fitted peak location.
	 * the x/y of the fitted peak will be in xOptimized and yOptimized
	 */

	function peakPicking(spectrum,
	/** value to search (on x axis) */
	target) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	  var _a, _b, _c;

	  const {
	    xVariable = 'x',
	    yVariable = 'y',
	    optimize: optimize$1 = false,
	    expectedFWHM = 1,
	    max: isMax = true,
	    shapeOptions = {}
	  } = options;
	  const x = (_a = spectrum.variables[xVariable]) === null || _a === void 0 ? void 0 : _a.data;
	  let y;

	  if (!isMax) {
	    y = (_b = spectrum.variables[yVariable]) === null || _b === void 0 ? void 0 : _b.data.slice(); // do deep copy as we maybe need to flip sign
	  } else {
	    y = (_c = spectrum.variables[yVariable]) === null || _c === void 0 ? void 0 : _c.data;
	  }

	  if (!x || !y) return;
	  let targetIndex;
	  targetIndex = xFindClosestIndex(x, target);
	  let optimizedPeak;
	  let optimizedIndex;
	  const result = {};

	  if (optimize$1) {
	    if (isMax === false) {
	      let maximumY = max(y);

	      for (let i = 0; i < y.length; i++) {
	        y[i] *= -1;
	        y[i] += maximumY; // This makes it somewhat more robust
	      }
	    }

	    optimizedPeak = optimize({
	      x,
	      y
	    }, [{
	      x: x[targetIndex],
	      y: y[targetIndex],
	      fwhm: expectedFWHM
	    }], shapeOptions);
	    optimizedIndex = xFindClosestIndex(x, optimizedPeak.peaks[0].x);

	    for (let [key, variable] of Object.entries(spectrum.variables)) {
	      result[key] = variable.data[optimizedIndex];
	    }

	    result.optimized = optimizedPeak.peaks[0];
	  } else {
	    for (let [key, variable] of Object.entries(spectrum.variables)) {
	      result[key] = variable.data[targetIndex];
	    }
	  }

	  return result;
	}

	/**
	 * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
	 * This process is done in place and is very fast.
	 * @param data
	 * @param peaks
	 */
	function optimizeTop(data, peaks) {
	  const {
	    x,
	    y
	  } = data;

	  for (const peak of peaks) {
	    let currentIndex = peak.index; // The detected peak could be moved 1 or 2 units to left or right.

	    if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
	      currentIndex--;
	    } else {
	      if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
	        currentIndex++;
	      } else {
	        if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
	          currentIndex -= 2;
	        } else {
	          if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
	            currentIndex += 2;
	          }
	        }
	      }
	    } // interpolation to a sin() function


	    if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
	      let alpha = 20 * Math.log10(y[currentIndex - 1]);
	      let beta = 20 * Math.log10(y[currentIndex]);
	      let gamma = 20 * Math.log10(y[currentIndex + 1]);
	      let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
	      peak.x = x[currentIndex] + (x[currentIndex] - x[currentIndex - 1]) * p;
	      peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
	    }
	  }
	}

	/**
	 * Global spectra deconvolution
	 * @param  data - Object data with x and y arrays. Values in x has to be growing
	 * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
	 * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.

	 */

	function gsd(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    sgOptions = {
	      windowSize: 9,
	      polynomial: 3
	    },
	    noiseLevel,
	    smoothY = false,
	    maxCriteria = true,
	    minMaxRatio = 0.00025,
	    realTopDetection = false
	  } = options;
	  let {
	    x,
	    y
	  } = data;

	  if (!xIsMonotoneIncreasing(x)) {
	    throw new Error('GSD only accepts monotone increasing x values');
	  } //rescale;


	  y = y.slice(); // If the max difference between delta x is less than 5%, then,
	  // we can assume it to be equally spaced variable

	  let equallySpaced = xIsEquallySpaced(x);

	  if (noiseLevel === undefined) {
	    if (equallySpaced) {
	      const noiseInfo = xNoiseStandardDeviation(y);

	      if (maxCriteria) {
	        noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
	      } else {
	        noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
	      }
	    } else {
	      noiseLevel = 0;
	    }
	  } else {
	    if (maxCriteria === false) {
	      noiseLevel *= -1;
	    }
	  }

	  if (maxCriteria === false) {
	    for (let i = 0; i < y.length; i++) {
	      y[i] *= -1;
	    }
	  }

	  if (noiseLevel !== undefined) {
	    for (let i = 0; i < y.length; i++) {
	      if (y[i] < noiseLevel) {
	        y[i] = noiseLevel;
	      }
	    }
	  }

	  let yData = y;
	  let dY, ddY;
	  const {
	    windowSize,
	    polynomial
	  } = sgOptions;

	  if (equallySpaced) {
	    if (smoothY) {
	      yData = sgg(y, x[1] - x[0], {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }

	    dY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  } else {
	    if (smoothY) {
	      yData = sgg(y, x, {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }

	    dY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  }

	  const minY = xMinValue(yData);
	  const maxY = xMaxValue(yData);
	  if (minY > maxY || minY === maxY) return [];
	  const yThreshold = minY + (maxY - minY) * minMaxRatio;
	  const dX = x[1] - x[0];
	  let lastMax = null;
	  let lastMin = null;
	  let minddY = [];
	  let intervalL = [];
	  let intervalR = []; // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum

	  for (let i = 1; i < yData.length - 1; ++i) {
	    if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
	      lastMin = {
	        x: x[i],
	        index: i
	      };

	      if (dX > 0 && lastMax !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    } // Maximum in first derivative


	    if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
	      lastMax = {
	        x: x[i],
	        index: i
	      };

	      if (dX < 0 && lastMin !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    } // Minimum in second derivative


	    if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
	      minddY.push(i);
	    }
	  }

	  let lastK = -1;
	  const peaks = [];

	  for (const minddYIndex of minddY) {
	    let deltaX = x[minddYIndex];
	    let possible = -1;
	    let k = lastK + 1;
	    let minDistance = Number.POSITIVE_INFINITY;
	    let currentDistance = 0;

	    while (possible === -1 && k < intervalL.length) {
	      currentDistance = Math.abs(deltaX - (intervalL[k].x + intervalR[k].x) / 2);

	      if (currentDistance < (intervalR[k].x - intervalL[k].x) / 2) {
	        possible = k;
	        lastK = k;
	      }

	      ++k; // Not getting closer?

	      if (currentDistance >= minDistance) {
	        break;
	      }

	      minDistance = currentDistance;
	    }

	    if (possible !== -1) {
	      if (yData[minddYIndex] > yThreshold) {
	        let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
	        peaks.push({
	          x: deltaX,
	          y: yData[minddYIndex],
	          width: width,
	          index: minddYIndex,
	          ddY: ddY[minddYIndex],
	          inflectionPoints: {
	            from: intervalL[possible],
	            to: intervalR[possible]
	          }
	        });
	      }
	    }
	  }

	  if (realTopDetection) {
	    optimizeTop({
	      x,
	      y: yData
	    }, peaks);
	  }

	  peaks.forEach(peak => {
	    if (!maxCriteria) {
	      peak.y *= -1;
	      peak.ddY = peak.ddY * -1;
	    }
	  });
	  peaks.sort((a, b) => {
	    return a.x - b.x;
	  });
	  return peaks;
	}

	/** Based on a x value we will return a peak*/

	function autoPeakPicking(spectrum) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  var _a, _b;

	  const {
	    xVariable = 'x',
	    yVariable = 'y',
	    normalizationOptions,
	    minPeakWidth
	  } = options;
	  let x = (_a = spectrum.variables[xVariable]) === null || _a === void 0 ? void 0 : _a.data;
	  let y = (_b = spectrum.variables[yVariable]) === null || _b === void 0 ? void 0 : _b.data;
	  if (!x || !y) return [];

	  if (normalizationOptions) {
	    const tempSpectrum = {
	      variables: {
	        x: {
	          data: x,
	          label: ''
	        },
	        y: {
	          data: y,
	          label: ''
	        }
	      }
	    };
	    const normalizedSpectrum = getNormalizedSpectrum(tempSpectrum, normalizationOptions);
	    x = normalizedSpectrum.variables.x.data;
	    y = normalizedSpectrum.variables.y.data;
	  }

	  if (!x || !y) return;
	  let {
	    from,
	    to
	  } = options;
	  let peaks = gsd({
	    x,
	    y
	  }, options);

	  if (normalizationOptions) {
	    // we need to recalculate the real count
	    const xyClosestYPoint = options.maxCriteria === undefined || options.maxCriteria ? xyMaxClosestYPoint : xyMinClosestYPoint;

	    for (let peak of peaks) {
	      const closest = xyClosestYPoint({
	        x: spectrum.variables.x.data,
	        y: spectrum.variables.y.data
	      }, {
	        target: peak.x
	      });
	      peak.x = closest.x;
	      peak.y = closest.y;
	    }
	  }

	  if (from !== undefined) {
	    peaks = peaks.filter(peak => peak.x >= from);
	  }

	  if (to !== undefined) {
	    peaks = peaks.filter(peak => peak.x <= to);
	  }

	  if (minPeakWidth) {
	    peaks = peaks.filter(peak => peak.width >= minPeakWidth);
	  }

	  return peaks.map(peak => {
	    const result = {};

	    for (const [key, variable] of Object.entries(spectrum.variables)) {
	      result[key] = variable.data[peak.index];
	    }

	    result.width = peak.width;
	    return result;
	  });
	}

	const JSGraph = {
	  getJSGraph,
	  getNormalizationAnnotations
	};

	function lineSplitTrim(line) {
	  try {
	    return lineSplit(line)[1].trim();
	  } catch (e) {
	    return '';
	  }
	}
	function lineSplit(line) {
	  return line.split(/\s{4,}/);
	}

	/**
	 * Parses and standardizes the metadata
	 *
	 * @param {string[]} lines
	 * @returns {object}
	 */

	function parseIGAMeasurmentHeader(lines) {
	  let metaData = {}; // Let's use some cleaner names

	  metaData.systemType = lineSplitTrim(lines[0]);
	  metaData.systemOwner = lineSplitTrim(lines[1]);
	  metaData.systemReference = lineSplitTrim(lines[2]);
	  metaData.systemSerialNumber = lineSplitTrim(lines[3]);
	  metaData.sampleNumber = lineSplitTrim(lines[4]);
	  metaData.experimentType = lineSplitTrim(lines[5]); // eslint-disable-next-line radix

	  metaData.experimentNumber = parseInt(lineSplitTrim(lines[6]));
	  metaData.title = lineSplitTrim(lines[7]);
	  metaData.comment = lineSplitTrim(lines[8]);
	  metaData.source = lineSplitTrim(lines[9]);
	  metaData.batch = lineSplitTrim(lines[10]);
	  metaData.id = lineSplitTrim(lines[11]);
	  metaData.experimentTitle = lineSplitTrim(lines[12]);
	  let tmp = lineSplitTrim(lines[14]).split(/\s/);
	  metaData.sampleWeight = parseFloat(tmp[0]);
	  metaData.sampleWeightunits = tmp[1];
	  tmp = lineSplitTrim(lines[15]).split(/\s/);
	  metaData.sampleWeightDry = parseFloat(tmp[0]);
	  metaData.sampleWeightDryunits = tmp[1];
	  tmp = lineSplitTrim(lines[18]).split(/\s/);
	  metaData.balanceTrimV = parseFloat(tmp[0]);
	  metaData.balanceTrimVunits = tmp[1];
	  metaData.balanceTrimT = lineSplitTrim(lines[19]);
	  tmp = lineSplitTrim(lines[20]).split(/\s/);
	  metaData.counterWeightM = parseFloat(tmp[0]);
	  metaData.counterWeightMunits = tmp[1];
	  tmp = lineSplitTrim(lines[21]).split(/\s/);
	  metaData.counterWeightRho = parseFloat(tmp[0]);
	  metaData.counterWeightRhounits = tmp[1];
	  metaData.counterWeightT = lineSplitTrim(lines[22]);
	  tmp = lineSplitTrim(lines[23]).split(/\s/);
	  metaData.tungstenM = parseFloat(tmp[0]);
	  metaData.tungstenMunits = tmp[1];
	  tmp = lineSplitTrim(lines[24]).split(/\s/);
	  metaData.tungstenRho = parseFloat(tmp[0]);
	  metaData.tungstenRhounits = tmp[1];
	  metaData.counterWeightT = lineSplitTrim(lines[25]);
	  tmp = lineSplitTrim(lines[26]).split(/\s/);
	  metaData.chainExcessM = parseFloat(tmp[0]);
	  metaData.chainExcessMunits = tmp[1];
	  tmp = lineSplitTrim(lines[27]).split(/\s/);
	  metaData.chainExcessRho = parseFloat(tmp[0]);
	  metaData.chainExcessRhounits = tmp[1];
	  metaData.chainExcessT = lineSplitTrim(lines[28]);
	  metaData.applicationCode = lineSplitTrim(lines[44]);
	  metaData.mode = lineSplitTrim(lines[44]);
	  metaData.source = lineSplitTrim(lines[45]);
	  metaData.gasSource = lineSplitTrim(lines[46]);
	  metaData.vessel = lineSplitTrim(lines[47]);
	  metaData.reactor = lineSplitTrim(lines[48]);
	  metaData.pressureSensor = lineSplitTrim(lines[49]);
	  metaData.thermostat = lineSplitTrim(lines[50]);
	  metaData.nitrogenEOS = lineSplitTrim(lines[51]);
	  metaData.nitrogenSVP = lineSplitTrim(lines[52]);
	  metaData.seriesType = lineSplitTrim(lines[59]);
	  metaData.scan = parseInt(lineSplitTrim(lines[61]), 10);
	  metaData.course = lineSplitTrim(lines[62]);
	  metaData.referenceStateDevice = lineSplitTrim(lines[63]);
	  metaData.scanTitle = lineSplitTrim(lines[64]);
	  metaData.scanStart = lineSplitTrim(lines[65]);
	  return metaData;
	}
	/**
	 * Find the start and end line numbers of the measurement
	 *
	 * @param {string[]} lines
	 * @returns {Array<number>} Array of length 2 [startIndex, endIndex]
	 */


	function getLineNumbersOfMeasurement(lines) {
	  let starts = [];
	  let ends = [];

	  for (let [index, line] of lines.entries()) {
	    if (line.match('S E R I E S   D A T A   R E C O R D')) {
	      starts.push(index);
	    } else if (line.match('Scan Ends ')) {
	      ends.push(index + 2);
	    }
	  }

	  return [starts, ends];
	}

	function parseIGADataBlock(lines) {
	  let dataBlock = {
	    pressure: [],
	    gasDensity: [],
	    pp0: [],
	    totalWeight: [],
	    bet: [],
	    excessAdsorptionPercentage: [],
	    excessAdsorption: [],
	    sampleT: [],
	    wtPercent: []
	  };

	  for (let line of lines) {
	    let tmp = line.split(/\s{3,}/);
	    dataBlock.pressure.push(parseFloat(tmp[0]) * 0.1);
	    dataBlock.gasDensity.push(parseFloat(tmp[1]));
	    dataBlock.pp0.push(parseFloat(tmp[2]));
	    dataBlock.totalWeight.push(parseFloat(tmp[3]));
	    dataBlock.excessAdsorptionPercentage.push(parseFloat(tmp[4]));
	    dataBlock.excessAdsorption.push(parseFloat(tmp[5]));
	    dataBlock.bet.push(parseFloat(tmp[6]));
	    dataBlock.sampleT.push(parseFloat(tmp[7]));
	    dataBlock.wtPercent.push(parseFloat(tmp[8]));
	  }

	  return dataBlock;
	}

	function parseNonBETData(lines, sampleWeight) {
	  let dataBlock = {
	    pressure: [],
	    gasDensity: [],
	    totalWeight: [],
	    sampleT: [],
	    wtPercent: [],
	    excessAdsorption: []
	  };

	  for (let line of lines) {
	    let tmp = line.split(/\s{3,}/);
	    dataBlock.pressure.push(parseFloat(tmp[0]) * 0.1);
	    dataBlock.gasDensity.push(parseFloat(tmp[1]));
	    dataBlock.totalWeight.push(parseFloat(tmp[2]));
	    dataBlock.sampleT.push(parseFloat(tmp[3]));
	    let wtPercent = parseFloat(tmp[4]);
	    dataBlock.wtPercent.push(wtPercent);
	    dataBlock.excessAdsorption.push(wtPercent * sampleWeight);
	  }

	  return dataBlock;
	}

	function parseOneIGA(lines) {
	  const measLen = lines.length;
	  let meta = parseIGAMeasurmentHeader(lines.slice(2, 68));
	  let dataBlock;

	  if (lines[68].match('BET')) {
	    dataBlock = parseIGADataBlock(lines.slice(72, measLen - 3));
	  } else {
	    dataBlock = parseNonBETData(lines.slice(72, measLen - 3), meta.sampleWeightDry);
	  }

	  meta.scanEnd = lineSplitTrim(lines[measLen - 2]);
	  meta.scanReferenceState = lineSplitTrim(lines[measLen - 1]);
	  return {
	    meta,
	    data: dataBlock
	  };
	}
	/**
	 * Orchestrates the parsing of a IGA file. Creates a new Analysis element
	 *
	 * @param {string} text - String containing the IGA analysis data, maybe contain multiple measurements
	 * @return {Analysis} - New class element with the given data
	 */


	function fromIGA(text) {
	  const lines = text.split(/[\r\n]+/);
	  const lineNumbers = getLineNumbersOfMeasurement(lines);
	  let analysis = new Analysis();

	  for (let i = 0; i < lineNumbers[0].length; i++) {
	    let meas = parseOneIGA(lines.slice(lineNumbers[0][i], lineNumbers[1][i]));
	    meas.meta.adsorptionT = mean(meas.data.sampleT);
	    meas.meta.adsorptionTunits = '°C';
	    let spectrum;

	    if ('pp0' in meas.data) {
	      spectrum = {
	        x: {
	          data: meas.data.pressure,
	          label: 'Pressure',
	          isDependent: false,
	          units: 'kPa'
	        },
	        y: {
	          data: meas.data.excessAdsorption,
	          label: 'Excess Adsorption [mmol/g]',
	          isDependent: true,
	          units: 'mmol/g'
	        },
	        p: {
	          data: meas.data.pp0,
	          label: 'relative pressure',
	          isDependent: false,
	          units: ''
	        },
	        r: {
	          data: meas.data.excessAdsorptionPercentage,
	          label: 'Excess Adsorption',
	          isDependent: true,
	          units: '%'
	        },
	        t: {
	          data: meas.data.sampleT,
	          label: 'Sample Temperature',
	          isDependent: false,
	          units: '°C'
	        }
	      };
	    } else {
	      spectrum = {
	        x: {
	          data: meas.data.pressure,
	          label: 'Pressure',
	          isDependent: false,
	          units: 'kPa'
	        },
	        y: {
	          data: meas.data.excessAdsorption,
	          label: 'Excess Adsorption',
	          isDependent: true,
	          units: 'g/g'
	        },
	        r: {
	          data: meas.data.wtPercent,
	          label: 'Excess Adsorption',
	          isDependent: true,
	          units: '%'
	        },
	        t: {
	          data: meas.data.sampleT,
	          label: 'Sample Temperature',
	          isDependent: false,
	          units: '°C'
	        }
	      };
	    }

	    analysis.pushSpectrum(spectrum, {
	      dataType: 'Adsorption Isotherm',
	      title: meas.meta.experimentTitle,
	      meta: meas.meta
	    });
	  }

	  return analysis;
	}

	/**
	 * Parses some relevant fields in the meta data
	 * Tries to use naming that is consistent with the other
	 * isotherm parsers
	 *
	 * @param {string[]} lines
	 * @param {number} dataStartIndex
	 * @returns {object}
	 */

	function parseMetaBlock(lines, dataStartIndex) {
	  let meta = {};

	  for (let i = dataStartIndex - 13; i < dataStartIndex; i++) {
	    if (lines[i].match('Sample: ')) {
	      meta.sample = lineSplit(lines[i])[2].trim();
	    }

	    if (lines[i].match('Operator:  ')) {
	      meta.operator = lineSplit(lines[i])[1].trim();
	    }

	    if (lines[i].match('Submitter: ')) {
	      meta.submitter = lineSplit(lines[i])[1].trim();
	    }

	    if (lines[i].match('File: ')) {
	      meta.file = lineSplit(lines[i])[2].trim();
	    }

	    if (lines[i].match('Started: ')) {
	      meta.started = lineSplit(lines[i])[2].trim();
	    }

	    if (lines[i].match('Analysis adsorptive: ')) {
	      meta.adsorptive = lineSplit(lines[i])[4].trim();
	    }

	    if (lines[i].match('Completed: ')) {
	      meta.completed = lineSplit(lines[i])[2].trim();
	    }

	    if (lines[i].match('Equilibration time: ')) {
	      let time = lineSplit(lines[i])[4].trim().split(' ');
	      meta.equilibrationTime = parseFloat(time[0]);
	      meta.equilibrationTimeUnit = time[1];
	    }

	    if (lines[i].match('Report time: ')) {
	      meta.reportTime = lineSplit(lines[i])[2].trim();
	    }

	    if (lines[i].match('Sample mass: ')) {
	      let mass = lineSplit(lines[i])[2].trim().split(' ');
	      meta.sampleWeight = parseFloat(mass[0]);
	      meta.sampleWeightUnit = mass[1];
	    }

	    if (lines[i].match('Gemini model: ')) {
	      meta.model = lineSplit(lines[i])[4].trim();
	    }

	    if (lines[i].match('Sample density: ')) {
	      let density = lineSplit(lines[i])[3].split(' ');
	      meta.sampleDensity = parseFloat(density[0]);
	      meta.sampleDensityUnit = density[1];
	    }

	    if (lines[i].match('Evac. rate:  ')) {
	      let evacRate = lineSplit(lines[i])[2].split(' ');
	      meta.evacRate = parseFloat(evacRate[0]);
	      meta.evacRateUnit = evacRate[1];
	    }
	  }

	  return meta;
	}
	/**
	 * Find the line numbers of start and end of the
	 * isotherm data
	 *
	 * @param {string[]} lines
	 * @returns {Array<number>} array of length two [startIndex, endIndex]
	 */


	function findDataBlocks(lines) {
	  let isothermTableStarts = [];
	  let isothermTableEnds = [];
	  let isIsothermTable = false;

	  for (let i = 0; i < lines.length; i++) {
	    if (lines[i].match('Isotherm Tabular Report')) {
	      isIsothermTable = true;
	      isothermTableStarts.push(i);
	    }

	    if (isIsothermTable && lines[i].match('Micromeritics')) {
	      isothermTableEnds.push(i);
	      isIsothermTable = false;
	    }
	  }

	  return [isothermTableStarts, isothermTableEnds];
	}
	/**
	 * Parses the relevant fields from the isotherm table,
	 * converts the pressure to kPa and creates floats
	 *
	 * @param {Array<string>} lines
	 * @returns {Object}
	 */


	function parseIsothermTable(lines) {
	  let pSat = parseFloat(lines[5].trim()) * 0.13332; // convert mmHg to kPa

	  let data = {
	    x: [],
	    y: [],
	    p: []
	  };
	  data.pSat = pSat;
	  let parts;

	  for (let i = 6; i < lines.length; i++) {
	    parts = lineSplit(lines[i]);
	    data.x.push(parseFloat(parts[0]));
	    data.y.push(parseFloat(parts[2]));
	    data.p.push(parseFloat(parts[1]) * 0.13332); // convert mmHg to kPa
	  }

	  return data;
	}
	/**
	 * Orchestrates the parsing of a micrometrics TXT file.
	 * Takes the text and returns an Analysis object.
	 * Also parses relevant metadata and converts some units
	 *
	 * @export
	 * @param {string} text
	 * @returns {Analysis}
	 */


	function fromMicrometricsTXT(text) {
	  const lines = text.split(/\r?\n/).filter(line => !line.match(/^\s*$/));
	  let startsAndEnds = findDataBlocks(lines);
	  let analysis = new Analysis();
	  let data;
	  let type = 'Adsorption Isotherm';

	  for (let i = 0; i < startsAndEnds[0].length; i++) {
	    data = parseIsothermTable(lines.slice(startsAndEnds[0][i], startsAndEnds[1][i]));
	    let meta = parseMetaBlock(lines, startsAndEnds[0][i]);
	    meta.pSat = data.pSat;

	    if (data.x[1] > data.x[data.x.length - 1]) {
	      type = 'Desorption Isotherm';
	    }

	    analysis.pushSpectrum({
	      x: {
	        data: data.x,
	        label: 'relative pressure',
	        isDependent: false,
	        units: ''
	      },
	      y: {
	        data: data.y,
	        label: 'Excess Adsorption',
	        isDependent: true,
	        units: 'mmol/g'
	      },
	      p: {
	        data: data.p,
	        label: 'Pressure [kPa]',
	        isDependent: false,
	        units: 'kPa'
	      }
	    }, {
	      dataType: type,
	      title: meta.sample,
	      meta
	    });
	  }

	  return analysis;
	}

	var papaparse_min = {exports: {}};

	/* @license
	Papa Parse
	v5.3.2
	https://github.com/mholt/PapaParse
	License: MIT
	*/

	(function (module, exports) {
	  !function (e, t) {
	    module.exports = t() ;
	  }(commonjsGlobal, function s() {

	    var f = "undefined" != typeof self ? self : "undefined" != typeof window ? window : void 0 !== f ? f : {};
	    var n = !f.document && !!f.postMessage,
	        o = n && /blob:/i.test((f.location || {}).protocol),
	        a = {},
	        h = 0,
	        b = {
	      parse: function (e, t) {
	        var i = (t = t || {}).dynamicTyping || !1;
	        M(i) && (t.dynamicTypingFunction = i, i = {});

	        if (t.dynamicTyping = i, t.transform = !!M(t.transform) && t.transform, t.worker && b.WORKERS_SUPPORTED) {
	          var r = function () {
	            if (!b.WORKERS_SUPPORTED) return !1;
	            var e = (i = f.URL || f.webkitURL || null, r = s.toString(), b.BLOB_URL || (b.BLOB_URL = i.createObjectURL(new Blob(["(", r, ")();"], {
	              type: "text/javascript"
	            })))),
	                t = new f.Worker(e);
	            var i, r;
	            return t.onmessage = _, t.id = h++, a[t.id] = t;
	          }();

	          return r.userStep = t.step, r.userChunk = t.chunk, r.userComplete = t.complete, r.userError = t.error, t.step = M(t.step), t.chunk = M(t.chunk), t.complete = M(t.complete), t.error = M(t.error), delete t.worker, void r.postMessage({
	            input: e,
	            config: t,
	            workerId: r.id
	          });
	        }

	        var n = null;
	        b.NODE_STREAM_INPUT, "string" == typeof e ? n = t.download ? new l(t) : new p(t) : !0 === e.readable && M(e.read) && M(e.on) ? n = new g(t) : (f.File && e instanceof File || e instanceof Object) && (n = new c(t));
	        return n.stream(e);
	      },
	      unparse: function (e, t) {
	        var n = !1,
	            _ = !0,
	            m = ",",
	            y = "\r\n",
	            s = '"',
	            a = s + s,
	            i = !1,
	            r = null,
	            o = !1;

	        !function () {
	          if ("object" != typeof t) return;
	          "string" != typeof t.delimiter || b.BAD_DELIMITERS.filter(function (e) {
	            return -1 !== t.delimiter.indexOf(e);
	          }).length || (m = t.delimiter);
	          ("boolean" == typeof t.quotes || "function" == typeof t.quotes || Array.isArray(t.quotes)) && (n = t.quotes);
	          "boolean" != typeof t.skipEmptyLines && "string" != typeof t.skipEmptyLines || (i = t.skipEmptyLines);
	          "string" == typeof t.newline && (y = t.newline);
	          "string" == typeof t.quoteChar && (s = t.quoteChar);
	          "boolean" == typeof t.header && (_ = t.header);

	          if (Array.isArray(t.columns)) {
	            if (0 === t.columns.length) throw new Error("Option columns is empty");
	            r = t.columns;
	          }

	          void 0 !== t.escapeChar && (a = t.escapeChar + s);
	          ("boolean" == typeof t.escapeFormulae || t.escapeFormulae instanceof RegExp) && (o = t.escapeFormulae instanceof RegExp ? t.escapeFormulae : /^[=+\-@\t\r].*$/);
	        }();
	        var h = new RegExp(j(s), "g");
	        "string" == typeof e && (e = JSON.parse(e));

	        if (Array.isArray(e)) {
	          if (!e.length || Array.isArray(e[0])) return u(null, e, i);
	          if ("object" == typeof e[0]) return u(r || Object.keys(e[0]), e, i);
	        } else if ("object" == typeof e) return "string" == typeof e.data && (e.data = JSON.parse(e.data)), Array.isArray(e.data) && (e.fields || (e.fields = e.meta && e.meta.fields || r), e.fields || (e.fields = Array.isArray(e.data[0]) ? e.fields : "object" == typeof e.data[0] ? Object.keys(e.data[0]) : []), Array.isArray(e.data[0]) || "object" == typeof e.data[0] || (e.data = [e.data])), u(e.fields || [], e.data || [], i);

	        throw new Error("Unable to serialize unrecognized input");

	        function u(e, t, i) {
	          var r = "";
	          "string" == typeof e && (e = JSON.parse(e)), "string" == typeof t && (t = JSON.parse(t));
	          var n = Array.isArray(e) && 0 < e.length,
	              s = !Array.isArray(t[0]);

	          if (n && _) {
	            for (var a = 0; a < e.length; a++) 0 < a && (r += m), r += v(e[a], a);

	            0 < t.length && (r += y);
	          }

	          for (var o = 0; o < t.length; o++) {
	            var h = n ? e.length : t[o].length,
	                u = !1,
	                f = n ? 0 === Object.keys(t[o]).length : 0 === t[o].length;

	            if (i && !n && (u = "greedy" === i ? "" === t[o].join("").trim() : 1 === t[o].length && 0 === t[o][0].length), "greedy" === i && n) {
	              for (var d = [], l = 0; l < h; l++) {
	                var c = s ? e[l] : l;
	                d.push(t[o][c]);
	              }

	              u = "" === d.join("").trim();
	            }

	            if (!u) {
	              for (var p = 0; p < h; p++) {
	                0 < p && !f && (r += m);
	                var g = n && s ? e[p] : p;
	                r += v(t[o][g], p);
	              }

	              o < t.length - 1 && (!i || 0 < h && !f) && (r += y);
	            }
	          }

	          return r;
	        }

	        function v(e, t) {
	          if (null == e) return "";
	          if (e.constructor === Date) return JSON.stringify(e).slice(1, 25);
	          var i = !1;
	          o && "string" == typeof e && o.test(e) && (e = "'" + e, i = !0);
	          var r = e.toString().replace(h, a);
	          return (i = i || !0 === n || "function" == typeof n && n(e, t) || Array.isArray(n) && n[t] || function (e, t) {
	            for (var i = 0; i < t.length; i++) if (-1 < e.indexOf(t[i])) return !0;

	            return !1;
	          }(r, b.BAD_DELIMITERS) || -1 < r.indexOf(m) || " " === r.charAt(0) || " " === r.charAt(r.length - 1)) ? s + r + s : r;
	        }
	      }
	    };

	    if (b.RECORD_SEP = String.fromCharCode(30), b.UNIT_SEP = String.fromCharCode(31), b.BYTE_ORDER_MARK = "\ufeff", b.BAD_DELIMITERS = ["\r", "\n", '"', b.BYTE_ORDER_MARK], b.WORKERS_SUPPORTED = !n && !!f.Worker, b.NODE_STREAM_INPUT = 1, b.LocalChunkSize = 10485760, b.RemoteChunkSize = 5242880, b.DefaultDelimiter = ",", b.Parser = E, b.ParserHandle = i, b.NetworkStreamer = l, b.FileStreamer = c, b.StringStreamer = p, b.ReadableStreamStreamer = g, f.jQuery) {
	      var d = f.jQuery;

	      d.fn.parse = function (o) {
	        var i = o.config || {},
	            h = [];
	        return this.each(function (e) {
	          if (!("INPUT" === d(this).prop("tagName").toUpperCase() && "file" === d(this).attr("type").toLowerCase() && f.FileReader) || !this.files || 0 === this.files.length) return !0;

	          for (var t = 0; t < this.files.length; t++) h.push({
	            file: this.files[t],
	            inputElem: this,
	            instanceConfig: d.extend({}, i)
	          });
	        }), e(), this;

	        function e() {
	          if (0 !== h.length) {
	            var e,
	                t,
	                i,
	                r,
	                n = h[0];

	            if (M(o.before)) {
	              var s = o.before(n.file, n.inputElem);

	              if ("object" == typeof s) {
	                if ("abort" === s.action) return e = "AbortError", t = n.file, i = n.inputElem, r = s.reason, void (M(o.error) && o.error({
	                  name: e
	                }, t, i, r));
	                if ("skip" === s.action) return void u();
	                "object" == typeof s.config && (n.instanceConfig = d.extend(n.instanceConfig, s.config));
	              } else if ("skip" === s) return void u();
	            }

	            var a = n.instanceConfig.complete;
	            n.instanceConfig.complete = function (e) {
	              M(a) && a(e, n.file, n.inputElem), u();
	            }, b.parse(n.file, n.instanceConfig);
	          } else M(o.complete) && o.complete();
	        }

	        function u() {
	          h.splice(0, 1), e();
	        }
	      };
	    }

	    function u(e) {
	      this._handle = null, this._finished = !1, this._completed = !1, this._halted = !1, this._input = null, this._baseIndex = 0, this._partialLine = "", this._rowCount = 0, this._start = 0, this._nextChunk = null, this.isFirstChunk = !0, this._completeResults = {
	        data: [],
	        errors: [],
	        meta: {}
	      }, function (e) {
	        var t = w(e);
	        t.chunkSize = parseInt(t.chunkSize), e.step || e.chunk || (t.chunkSize = null);
	        this._handle = new i(t), (this._handle.streamer = this)._config = t;
	      }.call(this, e), this.parseChunk = function (e, t) {
	        if (this.isFirstChunk && M(this._config.beforeFirstChunk)) {
	          var i = this._config.beforeFirstChunk(e);

	          void 0 !== i && (e = i);
	        }

	        this.isFirstChunk = !1, this._halted = !1;
	        var r = this._partialLine + e;
	        this._partialLine = "";

	        var n = this._handle.parse(r, this._baseIndex, !this._finished);

	        if (!this._handle.paused() && !this._handle.aborted()) {
	          var s = n.meta.cursor;
	          this._finished || (this._partialLine = r.substring(s - this._baseIndex), this._baseIndex = s), n && n.data && (this._rowCount += n.data.length);
	          var a = this._finished || this._config.preview && this._rowCount >= this._config.preview;
	          if (o) f.postMessage({
	            results: n,
	            workerId: b.WORKER_ID,
	            finished: a
	          });else if (M(this._config.chunk) && !t) {
	            if (this._config.chunk(n, this._handle), this._handle.paused() || this._handle.aborted()) return void (this._halted = !0);
	            n = void 0, this._completeResults = void 0;
	          }
	          return this._config.step || this._config.chunk || (this._completeResults.data = this._completeResults.data.concat(n.data), this._completeResults.errors = this._completeResults.errors.concat(n.errors), this._completeResults.meta = n.meta), this._completed || !a || !M(this._config.complete) || n && n.meta.aborted || (this._config.complete(this._completeResults, this._input), this._completed = !0), a || n && n.meta.paused || this._nextChunk(), n;
	        }

	        this._halted = !0;
	      }, this._sendError = function (e) {
	        M(this._config.error) ? this._config.error(e) : o && this._config.error && f.postMessage({
	          workerId: b.WORKER_ID,
	          error: e,
	          finished: !1
	        });
	      };
	    }

	    function l(e) {
	      var r;
	      (e = e || {}).chunkSize || (e.chunkSize = b.RemoteChunkSize), u.call(this, e), this._nextChunk = n ? function () {
	        this._readChunk(), this._chunkLoaded();
	      } : function () {
	        this._readChunk();
	      }, this.stream = function (e) {
	        this._input = e, this._nextChunk();
	      }, this._readChunk = function () {
	        if (this._finished) this._chunkLoaded();else {
	          if (r = new XMLHttpRequest(), this._config.withCredentials && (r.withCredentials = this._config.withCredentials), n || (r.onload = v(this._chunkLoaded, this), r.onerror = v(this._chunkError, this)), r.open(this._config.downloadRequestBody ? "POST" : "GET", this._input, !n), this._config.downloadRequestHeaders) {
	            var e = this._config.downloadRequestHeaders;

	            for (var t in e) r.setRequestHeader(t, e[t]);
	          }

	          if (this._config.chunkSize) {
	            var i = this._start + this._config.chunkSize - 1;
	            r.setRequestHeader("Range", "bytes=" + this._start + "-" + i);
	          }

	          try {
	            r.send(this._config.downloadRequestBody);
	          } catch (e) {
	            this._chunkError(e.message);
	          }

	          n && 0 === r.status && this._chunkError();
	        }
	      }, this._chunkLoaded = function () {
	        4 === r.readyState && (r.status < 200 || 400 <= r.status ? this._chunkError() : (this._start += this._config.chunkSize ? this._config.chunkSize : r.responseText.length, this._finished = !this._config.chunkSize || this._start >= function (e) {
	          var t = e.getResponseHeader("Content-Range");
	          if (null === t) return -1;
	          return parseInt(t.substring(t.lastIndexOf("/") + 1));
	        }(r), this.parseChunk(r.responseText)));
	      }, this._chunkError = function (e) {
	        var t = r.statusText || e;

	        this._sendError(new Error(t));
	      };
	    }

	    function c(e) {
	      var r, n;
	      (e = e || {}).chunkSize || (e.chunkSize = b.LocalChunkSize), u.call(this, e);
	      var s = "undefined" != typeof FileReader;
	      this.stream = function (e) {
	        this._input = e, n = e.slice || e.webkitSlice || e.mozSlice, s ? ((r = new FileReader()).onload = v(this._chunkLoaded, this), r.onerror = v(this._chunkError, this)) : r = new FileReaderSync(), this._nextChunk();
	      }, this._nextChunk = function () {
	        this._finished || this._config.preview && !(this._rowCount < this._config.preview) || this._readChunk();
	      }, this._readChunk = function () {
	        var e = this._input;

	        if (this._config.chunkSize) {
	          var t = Math.min(this._start + this._config.chunkSize, this._input.size);
	          e = n.call(e, this._start, t);
	        }

	        var i = r.readAsText(e, this._config.encoding);
	        s || this._chunkLoaded({
	          target: {
	            result: i
	          }
	        });
	      }, this._chunkLoaded = function (e) {
	        this._start += this._config.chunkSize, this._finished = !this._config.chunkSize || this._start >= this._input.size, this.parseChunk(e.target.result);
	      }, this._chunkError = function () {
	        this._sendError(r.error);
	      };
	    }

	    function p(e) {
	      var i;
	      u.call(this, e = e || {}), this.stream = function (e) {
	        return i = e, this._nextChunk();
	      }, this._nextChunk = function () {
	        if (!this._finished) {
	          var e,
	              t = this._config.chunkSize;
	          return t ? (e = i.substring(0, t), i = i.substring(t)) : (e = i, i = ""), this._finished = !i, this.parseChunk(e);
	        }
	      };
	    }

	    function g(e) {
	      u.call(this, e = e || {});
	      var t = [],
	          i = !0,
	          r = !1;
	      this.pause = function () {
	        u.prototype.pause.apply(this, arguments), this._input.pause();
	      }, this.resume = function () {
	        u.prototype.resume.apply(this, arguments), this._input.resume();
	      }, this.stream = function (e) {
	        this._input = e, this._input.on("data", this._streamData), this._input.on("end", this._streamEnd), this._input.on("error", this._streamError);
	      }, this._checkIsFinished = function () {
	        r && 1 === t.length && (this._finished = !0);
	      }, this._nextChunk = function () {
	        this._checkIsFinished(), t.length ? this.parseChunk(t.shift()) : i = !0;
	      }, this._streamData = v(function (e) {
	        try {
	          t.push("string" == typeof e ? e : e.toString(this._config.encoding)), i && (i = !1, this._checkIsFinished(), this.parseChunk(t.shift()));
	        } catch (e) {
	          this._streamError(e);
	        }
	      }, this), this._streamError = v(function (e) {
	        this._streamCleanUp(), this._sendError(e);
	      }, this), this._streamEnd = v(function () {
	        this._streamCleanUp(), r = !0, this._streamData("");
	      }, this), this._streamCleanUp = v(function () {
	        this._input.removeListener("data", this._streamData), this._input.removeListener("end", this._streamEnd), this._input.removeListener("error", this._streamError);
	      }, this);
	    }

	    function i(m) {
	      var a,
	          o,
	          h,
	          r = Math.pow(2, 53),
	          n = -r,
	          s = /^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,
	          u = /^(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))$/,
	          t = this,
	          i = 0,
	          f = 0,
	          d = !1,
	          e = !1,
	          l = [],
	          c = {
	        data: [],
	        errors: [],
	        meta: {}
	      };

	      if (M(m.step)) {
	        var p = m.step;

	        m.step = function (e) {
	          if (c = e, _()) g();else {
	            if (g(), 0 === c.data.length) return;
	            i += e.data.length, m.preview && i > m.preview ? o.abort() : (c.data = c.data[0], p(c, t));
	          }
	        };
	      }

	      function y(e) {
	        return "greedy" === m.skipEmptyLines ? "" === e.join("").trim() : 1 === e.length && 0 === e[0].length;
	      }

	      function g() {
	        return c && h && (k("Delimiter", "UndetectableDelimiter", "Unable to auto-detect delimiting character; defaulted to '" + b.DefaultDelimiter + "'"), h = !1), m.skipEmptyLines && (c.data = c.data.filter(function (e) {
	          return !y(e);
	        })), _() && function () {
	          if (!c) return;

	          function e(e, t) {
	            M(m.transformHeader) && (e = m.transformHeader(e, t)), l.push(e);
	          }

	          if (Array.isArray(c.data[0])) {
	            for (var t = 0; _() && t < c.data.length; t++) c.data[t].forEach(e);

	            c.data.splice(0, 1);
	          } else c.data.forEach(e);
	        }(), function () {
	          if (!c || !m.header && !m.dynamicTyping && !m.transform) return c;

	          function e(e, t) {
	            var i,
	                r = m.header ? {} : [];

	            for (i = 0; i < e.length; i++) {
	              var n = i,
	                  s = e[i];
	              m.header && (n = i >= l.length ? "__parsed_extra" : l[i]), m.transform && (s = m.transform(s, n)), s = v(n, s), "__parsed_extra" === n ? (r[n] = r[n] || [], r[n].push(s)) : r[n] = s;
	            }

	            return m.header && (i > l.length ? k("FieldMismatch", "TooManyFields", "Too many fields: expected " + l.length + " fields but parsed " + i, f + t) : i < l.length && k("FieldMismatch", "TooFewFields", "Too few fields: expected " + l.length + " fields but parsed " + i, f + t)), r;
	          }

	          var t = 1;
	          !c.data.length || Array.isArray(c.data[0]) ? (c.data = c.data.map(e), t = c.data.length) : c.data = e(c.data, 0);
	          m.header && c.meta && (c.meta.fields = l);
	          return f += t, c;
	        }();
	      }

	      function _() {
	        return m.header && 0 === l.length;
	      }

	      function v(e, t) {
	        return i = e, m.dynamicTypingFunction && void 0 === m.dynamicTyping[i] && (m.dynamicTyping[i] = m.dynamicTypingFunction(i)), !0 === (m.dynamicTyping[i] || m.dynamicTyping) ? "true" === t || "TRUE" === t || "false" !== t && "FALSE" !== t && (function (e) {
	          if (s.test(e)) {
	            var t = parseFloat(e);
	            if (n < t && t < r) return !0;
	          }

	          return !1;
	        }(t) ? parseFloat(t) : u.test(t) ? new Date(t) : "" === t ? null : t) : t;
	        var i;
	      }

	      function k(e, t, i, r) {
	        var n = {
	          type: e,
	          code: t,
	          message: i
	        };
	        void 0 !== r && (n.row = r), c.errors.push(n);
	      }

	      this.parse = function (e, t, i) {
	        var r = m.quoteChar || '"';
	        if (m.newline || (m.newline = function (e, t) {
	          e = e.substring(0, 1048576);
	          var i = new RegExp(j(t) + "([^]*?)" + j(t), "gm"),
	              r = (e = e.replace(i, "")).split("\r"),
	              n = e.split("\n"),
	              s = 1 < n.length && n[0].length < r[0].length;
	          if (1 === r.length || s) return "\n";

	          for (var a = 0, o = 0; o < r.length; o++) "\n" === r[o][0] && a++;

	          return a >= r.length / 2 ? "\r\n" : "\r";
	        }(e, r)), h = !1, m.delimiter) M(m.delimiter) && (m.delimiter = m.delimiter(e), c.meta.delimiter = m.delimiter);else {
	          var n = function (e, t, i, r, n) {
	            var s, a, o, h;
	            n = n || [",", "\t", "|", ";", b.RECORD_SEP, b.UNIT_SEP];

	            for (var u = 0; u < n.length; u++) {
	              var f = n[u],
	                  d = 0,
	                  l = 0,
	                  c = 0;
	              o = void 0;

	              for (var p = new E({
	                comments: r,
	                delimiter: f,
	                newline: t,
	                preview: 10
	              }).parse(e), g = 0; g < p.data.length; g++) if (i && y(p.data[g])) c++;else {
	                var _ = p.data[g].length;
	                l += _, void 0 !== o ? 0 < _ && (d += Math.abs(_ - o), o = _) : o = _;
	              }

	              0 < p.data.length && (l /= p.data.length - c), (void 0 === a || d <= a) && (void 0 === h || h < l) && 1.99 < l && (a = d, s = f, h = l);
	            }

	            return {
	              successful: !!(m.delimiter = s),
	              bestDelimiter: s
	            };
	          }(e, m.newline, m.skipEmptyLines, m.comments, m.delimitersToGuess);

	          n.successful ? m.delimiter = n.bestDelimiter : (h = !0, m.delimiter = b.DefaultDelimiter), c.meta.delimiter = m.delimiter;
	        }
	        var s = w(m);
	        return m.preview && m.header && s.preview++, a = e, o = new E(s), c = o.parse(a, t, i), g(), d ? {
	          meta: {
	            paused: !0
	          }
	        } : c || {
	          meta: {
	            paused: !1
	          }
	        };
	      }, this.paused = function () {
	        return d;
	      }, this.pause = function () {
	        d = !0, o.abort(), a = M(m.chunk) ? "" : a.substring(o.getCharIndex());
	      }, this.resume = function () {
	        t.streamer._halted ? (d = !1, t.streamer.parseChunk(a, !0)) : setTimeout(t.resume, 3);
	      }, this.aborted = function () {
	        return e;
	      }, this.abort = function () {
	        e = !0, o.abort(), c.meta.aborted = !0, M(m.complete) && m.complete(c), a = "";
	      };
	    }

	    function j(e) {
	      return e.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
	    }

	    function E(e) {
	      var S,
	          O = (e = e || {}).delimiter,
	          x = e.newline,
	          I = e.comments,
	          T = e.step,
	          D = e.preview,
	          A = e.fastMode,
	          L = S = void 0 === e.quoteChar || null === e.quoteChar ? '"' : e.quoteChar;
	      if (void 0 !== e.escapeChar && (L = e.escapeChar), ("string" != typeof O || -1 < b.BAD_DELIMITERS.indexOf(O)) && (O = ","), I === O) throw new Error("Comment character same as delimiter");
	      !0 === I ? I = "#" : ("string" != typeof I || -1 < b.BAD_DELIMITERS.indexOf(I)) && (I = !1), "\n" !== x && "\r" !== x && "\r\n" !== x && (x = "\n");
	      var F = 0,
	          z = !1;
	      this.parse = function (r, t, i) {
	        if ("string" != typeof r) throw new Error("Input must be a string");
	        var n = r.length,
	            e = O.length,
	            s = x.length,
	            a = I.length,
	            o = M(T),
	            h = [],
	            u = [],
	            f = [],
	            d = F = 0;
	        if (!r) return C();

	        if (A || !1 !== A && -1 === r.indexOf(S)) {
	          for (var l = r.split(x), c = 0; c < l.length; c++) {
	            if (f = l[c], F += f.length, c !== l.length - 1) F += x.length;else if (i) return C();

	            if (!I || f.substring(0, a) !== I) {
	              if (o) {
	                if (h = [], k(f.split(O)), R(), z) return C();
	              } else k(f.split(O));

	              if (D && D <= c) return h = h.slice(0, D), C(!0);
	            }
	          }

	          return C();
	        }

	        for (var p = r.indexOf(O, F), g = r.indexOf(x, F), _ = new RegExp(j(L) + j(S), "g"), m = r.indexOf(S, F);;) if (r[F] !== S) {
	          if (I && 0 === f.length && r.substring(F, F + a) === I) {
	            if (-1 === g) return C();
	            F = g + s, g = r.indexOf(x, F), p = r.indexOf(O, F);
	          } else if (-1 !== p && (p < g || -1 === g)) f.push(r.substring(F, p)), F = p + e, p = r.indexOf(O, F);else {
	            if (-1 === g) break;
	            if (f.push(r.substring(F, g)), w(g + s), o && (R(), z)) return C();
	            if (D && h.length >= D) return C(!0);
	          }
	        } else for (m = F, F++;;) {
	          if (-1 === (m = r.indexOf(S, m + 1))) return i || u.push({
	            type: "Quotes",
	            code: "MissingQuotes",
	            message: "Quoted field unterminated",
	            row: h.length,
	            index: F
	          }), E();
	          if (m === n - 1) return E(r.substring(F, m).replace(_, S));

	          if (S !== L || r[m + 1] !== L) {
	            if (S === L || 0 === m || r[m - 1] !== L) {
	              -1 !== p && p < m + 1 && (p = r.indexOf(O, m + 1)), -1 !== g && g < m + 1 && (g = r.indexOf(x, m + 1));
	              var y = b(-1 === g ? p : Math.min(p, g));

	              if (r.substr(m + 1 + y, e) === O) {
	                f.push(r.substring(F, m).replace(_, S)), r[F = m + 1 + y + e] !== S && (m = r.indexOf(S, F)), p = r.indexOf(O, F), g = r.indexOf(x, F);
	                break;
	              }

	              var v = b(g);

	              if (r.substring(m + 1 + v, m + 1 + v + s) === x) {
	                if (f.push(r.substring(F, m).replace(_, S)), w(m + 1 + v + s), p = r.indexOf(O, F), m = r.indexOf(S, F), o && (R(), z)) return C();
	                if (D && h.length >= D) return C(!0);
	                break;
	              }

	              u.push({
	                type: "Quotes",
	                code: "InvalidQuotes",
	                message: "Trailing quote on quoted field is malformed",
	                row: h.length,
	                index: F
	              }), m++;
	            }
	          } else m++;
	        }

	        return E();

	        function k(e) {
	          h.push(e), d = F;
	        }

	        function b(e) {
	          var t = 0;

	          if (-1 !== e) {
	            var i = r.substring(m + 1, e);
	            i && "" === i.trim() && (t = i.length);
	          }

	          return t;
	        }

	        function E(e) {
	          return i || (void 0 === e && (e = r.substring(F)), f.push(e), F = n, k(f), o && R()), C();
	        }

	        function w(e) {
	          F = e, k(f), f = [], g = r.indexOf(x, F);
	        }

	        function C(e) {
	          return {
	            data: h,
	            errors: u,
	            meta: {
	              delimiter: O,
	              linebreak: x,
	              aborted: z,
	              truncated: !!e,
	              cursor: d + (t || 0)
	            }
	          };
	        }

	        function R() {
	          T(C()), h = [], u = [];
	        }
	      }, this.abort = function () {
	        z = !0;
	      }, this.getCharIndex = function () {
	        return F;
	      };
	    }

	    function _(e) {
	      var t = e.data,
	          i = a[t.workerId],
	          r = !1;
	      if (t.error) i.userError(t.error, t.file);else if (t.results && t.results.data) {
	        var n = {
	          abort: function () {
	            r = !0, m(t.workerId, {
	              data: [],
	              errors: [],
	              meta: {
	                aborted: !0
	              }
	            });
	          },
	          pause: y,
	          resume: y
	        };

	        if (M(i.userStep)) {
	          for (var s = 0; s < t.results.data.length && (i.userStep({
	            data: t.results.data[s],
	            errors: t.results.errors,
	            meta: t.results.meta
	          }, n), !r); s++);

	          delete t.results;
	        } else M(i.userChunk) && (i.userChunk(t.results, n, t.file), delete t.results);
	      }
	      t.finished && !r && m(t.workerId, t.results);
	    }

	    function m(e, t) {
	      var i = a[e];
	      M(i.userComplete) && i.userComplete(t), i.terminate(), delete a[e];
	    }

	    function y() {
	      throw new Error("Not implemented.");
	    }

	    function w(e) {
	      if ("object" != typeof e || null === e) return e;
	      var t = Array.isArray(e) ? [] : {};

	      for (var i in e) t[i] = w(e[i]);

	      return t;
	    }

	    function v(e, t) {
	      return function () {
	        e.apply(t, arguments);
	      };
	    }

	    function M(e) {
	      return "function" == typeof e;
	    }

	    return o && (f.onmessage = function (e) {
	      var t = e.data;
	      void 0 === b.WORKER_ID && t && (b.WORKER_ID = t.workerId);
	      if ("string" == typeof t.input) f.postMessage({
	        workerId: b.WORKER_ID,
	        results: b.parse(t.input, t.config),
	        finished: !0
	      });else if (f.File && t.input instanceof File || t.input instanceof Object) {
	        var i = b.parse(t.input, t.config);
	        i && f.postMessage({
	          workerId: b.WORKER_ID,
	          results: i,
	          finished: !0
	        });
	      }
	    }), (l.prototype = Object.create(u.prototype)).constructor = l, (c.prototype = Object.create(u.prototype)).constructor = c, (p.prototype = Object.create(p.prototype)).constructor = p, (g.prototype = Object.create(u.prototype)).constructor = g, b;
	  });
	})(papaparse_min);

	var Papa = papaparse_min.exports;

	const idealGasConstant = 22.413969545014; // mol / cm3 at STP

	/**
	 * Create an Analysis object from a micrometrics CSV file
	 *
	 * @export
	 * @param {string} text
	 * @returns {Analysis}
	 */

	function fromMicrometricsCSV(text) {
	  text = text.replace(/,/g, '.');
	  let parsed = Papa.parse(text, {
	    delimiter: ';',
	    dynamicTyping: false,
	    skipEmptyLines: true
	  }).data;

	  const arrayColumn = (arr, n) => arr.map(x => x[n]);

	  const headerRow = parsed.shift();
	  let analysis = new Analysis();

	  try {
	    analysis.pushSpectrum({
	      //FixMe: this should be p
	      x: {
	        data: arrayColumn(parsed, 0).filter(value => {
	          return value !== '';
	        }).map(x => {
	          return parseFloat(x);
	        }),
	        label: 'relative pressure p/p0',
	        isDependent: false,
	        units: ''
	      },
	      y: {
	        data: arrayColumn(parsed, 1).filter(value => {
	          return value !== '';
	        }).map(x => {
	          return parseFloat(x) / idealGasConstant * 1000;
	        }),
	        label: 'Excess adsorption',
	        isDependent: false,
	        units: 'mmol/g'
	      }
	    }, {
	      dataType: 'Adsorption Isotherm',
	      title: 'Adsorption',
	      meta: {
	        header: headerRow
	      }
	    });
	    analysis.pushSpectrum({
	      x: {
	        data: arrayColumn(parsed, 2).filter(value => {
	          return value !== '';
	        }).map(x => {
	          return parseFloat(x);
	        }),
	        label: 'relative pressure p/p0',
	        isDependent: false,
	        units: ''
	      },
	      y: {
	        data: arrayColumn(parsed, 3).filter(value => {
	          return value !== '';
	        }).map(x => {
	          return parseFloat(x) / idealGasConstant * 1000;
	        }),
	        label: 'Excess adsorption',
	        isDependent: true,
	        units: 'mmol/g'
	      }
	    }, {
	      dataType: 'Desorption Isotherm',
	      title: 'Desorption',
	      meta: {}
	    });
	  } catch (err) {
	    throw Error(`Could not parse desorption section due to ${err}. Please report an issue with an example file!`);
	  }

	  return analysis;
	}

	var current_ansi=1252;var VALID_ANSI=[874,932,936,949,950,1250,1251,1252,1253,1254,1255,1256,1257,1258,10000];/* ECMA-376 Part I 18.4.1 charset to codepage mapping */var CS2CP={/*::[*/0/*::]*/:1252,/* ANSI */ /*::[*/1/*::]*/:65001,/* DEFAULT */ /*::[*/2/*::]*/:65001,/* SYMBOL */ /*::[*/77/*::]*/:10000,/* MAC */ /*::[*/128/*::]*/:932,/* SHIFTJIS */ /*::[*/129/*::]*/:949,/* HANGUL */ /*::[*/130/*::]*/:1361,/* JOHAB */ /*::[*/134/*::]*/:936,/* GB2312 */ /*::[*/136/*::]*/:950,/* CHINESEBIG5 */ /*::[*/161/*::]*/:1253,/* GREEK */ /*::[*/162/*::]*/:1254,/* TURKISH */ /*::[*/163/*::]*/:1258,/* VIETNAMESE */ /*::[*/177/*::]*/:1255,/* HEBREW */ /*::[*/178/*::]*/:1256,/* ARABIC */ /*::[*/186/*::]*/:1257,/* BALTIC */ /*::[*/204/*::]*/:1251,/* RUSSIAN */ /*::[*/222/*::]*/:874,/* THAI */ /*::[*/238/*::]*/:1250,/* EASTEUROPE */ /*::[*/255/*::]*/:1252,/* OEM */ /*::[*/69/*::]*/:6969/* MISC */}/*:any*/;var set_ansi=function(cp/*:number*/){if(VALID_ANSI.indexOf(cp)==-1)return;current_ansi=CS2CP[0]=cp;};function reset_ansi(){set_ansi(1252);}var set_cp=function(cp/*:number*/){set_ansi(cp);};function reset_cp(){set_cp(1200);reset_ansi();}function char_codes(data/*:string*/)/*:Array<number>*/{var o/*:Array<number>*/=[];for(var i=0,len=data.length;i<len;++i)o[i]=data.charCodeAt(i);return o;}function utf16leread(data/*:string*/)/*:string*/{var o/*:Array<string>*/=[];for(var i=0;i<data.length>>1;++i)o[i]=String.fromCharCode(data.charCodeAt(2*i)+(data.charCodeAt(2*i+1)<<8));return o.join("");}function utf16beread(data/*:string*/)/*:string*/{var o/*:Array<string>*/=[];for(var i=0;i<data.length>>1;++i)o[i]=String.fromCharCode(data.charCodeAt(2*i+1)+(data.charCodeAt(2*i)<<8));return o.join("");}var debom=function(data/*:string*/)/*:string*/{var c1=data.charCodeAt(0),c2=data.charCodeAt(1);if(c1==0xFF&&c2==0xFE)return utf16leread(data.slice(2));if(c1==0xFE&&c2==0xFF)return utf16beread(data.slice(2));if(c1==0xFEFF)return data.slice(1);return data;};var _getchar=function _gc1(x/*:number*/)/*:string*/{return String.fromCharCode(x);};var _getansi=function _ga1(x/*:number*/)/*:string*/{return String.fromCharCode(x);};var $cptable;var Base64_map="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";function Base64_encode(input){var o="";var c1=0,c2=0,c3=0,e1=0,e2=0,e3=0,e4=0;for(var i=0;i<input.length;){c1=input.charCodeAt(i++);e1=c1>>2;c2=input.charCodeAt(i++);e2=(c1&3)<<4|c2>>4;c3=input.charCodeAt(i++);e3=(c2&15)<<2|c3>>6;e4=c3&63;if(isNaN(c2)){e3=e4=64;}else if(isNaN(c3)){e4=64;}o+=Base64_map.charAt(e1)+Base64_map.charAt(e2)+Base64_map.charAt(e3)+Base64_map.charAt(e4);}return o;}function Base64_decode(input){var o="";var c1=0,c2=0,c3=0,e1=0,e2=0,e3=0,e4=0;input=input.replace(/[^\w\+\/\=]/g,"");for(var i=0;i<input.length;){e1=Base64_map.indexOf(input.charAt(i++));e2=Base64_map.indexOf(input.charAt(i++));c1=e1<<2|e2>>4;o+=String.fromCharCode(c1);e3=Base64_map.indexOf(input.charAt(i++));c2=(e2&15)<<4|e3>>2;if(e3!==64){o+=String.fromCharCode(c2);}e4=Base64_map.indexOf(input.charAt(i++));c3=(e3&3)<<6|e4;if(e4!==64){o+=String.fromCharCode(c3);}}return o;}var has_buf=/*#__PURE__*/function(){return typeof Buffer!=='undefined'&&typeof process!=='undefined'&&typeof process.versions!=='undefined'&&!!process.versions.node;}();var Buffer_from=/*#__PURE__*/function(){if(typeof Buffer!=='undefined'){var nbfs=!Buffer.from;if(!nbfs)try{Buffer.from("foo","utf8");}catch(e){nbfs=true;}return nbfs?function(buf,enc){return enc?new Buffer(buf,enc):new Buffer(buf);}:Buffer.from.bind(Buffer);}return function(){};}();function new_raw_buf(len/*:number*/){/* jshint -W056 */if(has_buf)return Buffer.alloc?Buffer.alloc(len):new Buffer(len);return typeof Uint8Array!="undefined"?new Uint8Array(len):new Array(len);/* jshint +W056 */}function new_unsafe_buf(len/*:number*/){/* jshint -W056 */if(has_buf)return Buffer.allocUnsafe?Buffer.allocUnsafe(len):new Buffer(len);return typeof Uint8Array!="undefined"?new Uint8Array(len):new Array(len);/* jshint +W056 */}var s2a=function s2a(s/*:string*/)/*:any*/{if(has_buf)return Buffer_from(s,"binary");return s.split("").map(function(x/*:string*/)/*:number*/{return x.charCodeAt(0)&0xff;});};function a2s(data/*:any*/)/*:string*/{if(Array.isArray(data))return data.map(function(c){return String.fromCharCode(c);}).join("");var o/*:Array<string>*/=[];for(var i=0;i<data.length;++i)o[i]=String.fromCharCode(data[i]);return o.join("");}function ab2a(data/*:ArrayBuffer|Uint8Array*/)/*:Array<number>*/{if(typeof ArrayBuffer=='undefined')throw new Error("Unsupported");if(data instanceof ArrayBuffer)return ab2a(new Uint8Array(data));/*:: if(data instanceof ArrayBuffer) throw new Error("unreachable"); */var o=new Array(data.length);for(var i=0;i<data.length;++i)o[i]=data[i];return o;}var bconcat=has_buf?function(bufs){return Buffer.concat(bufs.map(function(buf){return Buffer.isBuffer(buf)?buf:Buffer_from(buf);}));}:function(bufs){if(typeof Uint8Array!=="undefined"){var i=0,maxlen=0;for(i=0;i<bufs.length;++i)maxlen+=bufs[i].length;var o=new Uint8Array(maxlen);var len=0;for(i=0,maxlen=0;i<bufs.length;maxlen+=len,++i){len=bufs[i].length;if(bufs[i]instanceof Uint8Array)o.set(bufs[i],maxlen);else if(typeof bufs[i]=="string"){throw "wtf";}else o.set(new Uint8Array(bufs[i]),maxlen);}return o;}return [].concat.apply([],bufs.map(function(buf){return Array.isArray(buf)?buf:[].slice.call(buf);}));};function utf8decode(content/*:string*/){var out=[],widx=0,L=content.length+250;var o=new_raw_buf(content.length+255);for(var ridx=0;ridx<content.length;++ridx){var c=content.charCodeAt(ridx);if(c<0x80)o[widx++]=c;else if(c<0x800){o[widx++]=192|c>>6&31;o[widx++]=128|c&63;}else if(c>=0xD800&&c<0xE000){c=(c&1023)+64;var d=content.charCodeAt(++ridx)&1023;o[widx++]=240|c>>8&7;o[widx++]=128|c>>2&63;o[widx++]=128|d>>6&15|(c&3)<<4;o[widx++]=128|d&63;}else {o[widx++]=224|c>>12&15;o[widx++]=128|c>>6&63;o[widx++]=128|c&63;}if(widx>L){out.push(o.slice(0,widx));widx=0;o=new_raw_buf(65535);L=65530;}}out.push(o.slice(0,widx));return bconcat(out);}var chr0=/\u0000/g,chr1=/[\u0001-\u0006]/g;/*::
	declare type Block = any;
	declare type BufArray = {
		newblk(sz:number):Block;
		next(sz:number):Block;
		end():any;
		push(buf:Block):void;
	};

	type RecordHopperCB = {(d:any, Rn:string, RT:number):?boolean;};

	type EvertType = {[string]:string};
	type EvertNumType = {[string]:number};
	type EvertArrType = {[string]:Array<string>};

	type StringConv = {(string):string};

	*/ /* ssf.js (C) 2013-present SheetJS -- http://sheetjs.com */ /*jshint -W041 */function _strrev(x/*:string*/)/*:string*/{var o="",i=x.length-1;while(i>=0)o+=x.charAt(i--);return o;}function pad0(v/*:any*/,d/*:number*/)/*:string*/{var t=""+v;return t.length>=d?t:fill('0',d-t.length)+t;}function pad_(v/*:any*/,d/*:number*/)/*:string*/{var t=""+v;return t.length>=d?t:fill(' ',d-t.length)+t;}function rpad_(v/*:any*/,d/*:number*/)/*:string*/{var t=""+v;return t.length>=d?t:t+fill(' ',d-t.length);}function pad0r1(v/*:any*/,d/*:number*/)/*:string*/{var t=""+Math.round(v);return t.length>=d?t:fill('0',d-t.length)+t;}function pad0r2(v/*:any*/,d/*:number*/)/*:string*/{var t=""+v;return t.length>=d?t:fill('0',d-t.length)+t;}var p2_32=/*#__PURE__*/Math.pow(2,32);function pad0r(v/*:any*/,d/*:number*/)/*:string*/{if(v>p2_32||v<-p2_32)return pad0r1(v,d);var i=Math.round(v);return pad0r2(i,d);}/* yes, in 2022 this is still faster than string compare */function SSF_isgeneral(s/*:string*/,i/*:?number*/)/*:boolean*/{i=i||0;return s.length>=7+i&&(s.charCodeAt(i)|32)===103&&(s.charCodeAt(i+1)|32)===101&&(s.charCodeAt(i+2)|32)===110&&(s.charCodeAt(i+3)|32)===101&&(s.charCodeAt(i+4)|32)===114&&(s.charCodeAt(i+5)|32)===97&&(s.charCodeAt(i+6)|32)===108;}var days/*:Array<Array<string> >*/=[['Sun','Sunday'],['Mon','Monday'],['Tue','Tuesday'],['Wed','Wednesday'],['Thu','Thursday'],['Fri','Friday'],['Sat','Saturday']];var months/*:Array<Array<string> >*/=[['J','Jan','January'],['F','Feb','February'],['M','Mar','March'],['A','Apr','April'],['M','May','May'],['J','Jun','June'],['J','Jul','July'],['A','Aug','August'],['S','Sep','September'],['O','Oct','October'],['N','Nov','November'],['D','Dec','December']];function SSF_init_table(t/*:any*/){if(!t)t={};t[0]='General';t[1]='0';t[2]='0.00';t[3]='#,##0';t[4]='#,##0.00';t[9]='0%';t[10]='0.00%';t[11]='0.00E+00';t[12]='# ?/?';t[13]='# ??/??';t[14]='m/d/yy';t[15]='d-mmm-yy';t[16]='d-mmm';t[17]='mmm-yy';t[18]='h:mm AM/PM';t[19]='h:mm:ss AM/PM';t[20]='h:mm';t[21]='h:mm:ss';t[22]='m/d/yy h:mm';t[37]='#,##0 ;(#,##0)';t[38]='#,##0 ;[Red](#,##0)';t[39]='#,##0.00;(#,##0.00)';t[40]='#,##0.00;[Red](#,##0.00)';t[45]='mm:ss';t[46]='[h]:mm:ss';t[47]='mmss.0';t[48]='##0.0E+0';t[49]='@';t[56]='"上午/下午 "hh"時"mm"分"ss"秒 "';return t;}/* repeated to satiate webpack */var table_fmt={0:'General',1:'0',2:'0.00',3:'#,##0',4:'#,##0.00',9:'0%',10:'0.00%',11:'0.00E+00',12:'# ?/?',13:'# ??/??',14:'m/d/yy',15:'d-mmm-yy',16:'d-mmm',17:'mmm-yy',18:'h:mm AM/PM',19:'h:mm:ss AM/PM',20:'h:mm',21:'h:mm:ss',22:'m/d/yy h:mm',37:'#,##0 ;(#,##0)',38:'#,##0 ;[Red](#,##0)',39:'#,##0.00;(#,##0.00)',40:'#,##0.00;[Red](#,##0.00)',45:'mm:ss',46:'[h]:mm:ss',47:'mmss.0',48:'##0.0E+0',49:'@',56:'"上午/下午 "hh"時"mm"分"ss"秒 "'};/* Defaults determined by systematically testing in Excel 2019 */ /* These formats appear to default to other formats in the table */var SSF_default_map={5:37,6:38,7:39,8:40,//  5 -> 37 ...  8 -> 40
	23:0,24:0,25:0,26:0,// 23 ->  0 ... 26 ->  0
	27:14,28:14,29:14,30:14,31:14,// 27 -> 14 ... 31 -> 14
	50:14,51:14,52:14,53:14,54:14,// 50 -> 14 ... 58 -> 14
	55:14,56:14,57:14,58:14,59:1,60:2,61:3,62:4,// 59 ->  1 ... 62 ->  4
	67:9,68:10,// 67 ->  9 ... 68 -> 10
	69:12,70:13,71:14,// 69 -> 12 ... 71 -> 14
	72:14,73:15,74:16,75:17,// 72 -> 14 ... 75 -> 17
	76:20,77:21,78:22,// 76 -> 20 ... 78 -> 22
	79:45,80:46,81:47,// 79 -> 45 ... 81 -> 47
	82:0// 82 ->  0 ... 65536 -> 0 (omitted)
	};/* These formats technically refer to Accounting formats with no equivalent */var SSF_default_str={//  5 -- Currency,   0 decimal, black negative
	5:'"$"#,##0_);\\("$"#,##0\\)',63:'"$"#,##0_);\\("$"#,##0\\)',//  6 -- Currency,   0 decimal, red   negative
	6:'"$"#,##0_);[Red]\\("$"#,##0\\)',64:'"$"#,##0_);[Red]\\("$"#,##0\\)',//  7 -- Currency,   2 decimal, black negative
	7:'"$"#,##0.00_);\\("$"#,##0.00\\)',65:'"$"#,##0.00_);\\("$"#,##0.00\\)',//  8 -- Currency,   2 decimal, red   negative
	8:'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',66:'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',// 41 -- Accounting, 0 decimal, No Symbol
	41:'_(* #,##0_);_(* \\(#,##0\\);_(* "-"_);_(@_)',// 42 -- Accounting, 0 decimal, $  Symbol
	42:'_("$"* #,##0_);_("$"* \\(#,##0\\);_("$"* "-"_);_(@_)',// 43 -- Accounting, 2 decimal, No Symbol
	43:'_(* #,##0.00_);_(* \\(#,##0.00\\);_(* "-"??_);_(@_)',// 44 -- Accounting, 2 decimal, $  Symbol
	44:'_("$"* #,##0.00_);_("$"* \\(#,##0.00\\);_("$"* "-"??_);_(@_)'};function SSF_frac(x/*:number*/,D/*:number*/,mixed/*:?boolean*/)/*:Array<number>*/{var sgn=x<0?-1:1;var B=x*sgn;var P_2=0,P_1=1,P=0;var Q_2=1,Q_1=0,Q=0;var A=Math.floor(B);while(Q_1<D){A=Math.floor(B);P=A*P_1+P_2;Q=A*Q_1+Q_2;if(B-A<0.00000005)break;B=1/(B-A);P_2=P_1;P_1=P;Q_2=Q_1;Q_1=Q;}if(Q>D){if(Q_1>D){Q=Q_2;P=P_2;}else {Q=Q_1;P=P_1;}}if(!mixed)return [0,sgn*P,Q];var q=Math.floor(sgn*P/Q);return [q,sgn*P-q*Q,Q];}function SSF_parse_date_code(v/*:number*/,opts/*:?any*/,b2/*:?boolean*/){if(v>2958465||v<0)return null;var date=v|0,time=Math.floor(86400*(v-date)),dow=0;var dout=[];var out={D:date,T:time,u:86400*(v-date)-time,y:0,m:0,d:0,H:0,M:0,S:0,q:0};if(Math.abs(out.u)<1e-6)out.u=0;if(opts&&opts.date1904)date+=1462;if(out.u>0.9999){out.u=0;if(++time==86400){out.T=time=0;++date;++out.D;}}if(date===60){dout=b2?[1317,10,29]:[1900,2,29];dow=3;}else if(date===0){dout=b2?[1317,8,29]:[1900,1,0];dow=6;}else {if(date>60)--date;/* 1 = Jan 1 1900 in Gregorian */var d=new Date(1900,0,1);d.setDate(d.getDate()+date-1);dout=[d.getFullYear(),d.getMonth()+1,d.getDate()];dow=d.getDay();if(date<60)dow=(dow+6)%7;if(b2)dow=SSF_fix_hijri(d,dout);}out.y=dout[0];out.m=dout[1];out.d=dout[2];out.S=time%60;time=Math.floor(time/60);out.M=time%60;time=Math.floor(time/60);out.H=time;out.q=dow;return out;}var SSFbasedate=/*#__PURE__*/new Date(1899,11,31,0,0,0);var SSFdnthresh=/*#__PURE__*/SSFbasedate.getTime();var SSFbase1904=/*#__PURE__*/new Date(1900,2,1,0,0,0);function datenum_local(v/*:Date*/,date1904/*:?boolean*/)/*:number*/{var epoch=/*#__PURE__*/v.getTime();if(date1904)epoch-=1461*24*60*60*1000;else if(v>=SSFbase1904)epoch+=24*60*60*1000;return (epoch-(SSFdnthresh+(/*#__PURE__*/v.getTimezoneOffset()-/*#__PURE__*/SSFbasedate.getTimezoneOffset())*60000))/(24*60*60*1000);}/* ECMA-376 18.8.30 numFmt*/ /* Note: `toPrecision` uses standard form when prec > E and E >= -6 */ /* exponent >= -9 and <= 9 */function SSF_strip_decimal(o/*:string*/)/*:string*/{return o.indexOf(".")==-1?o:o.replace(/(?:\.0*|(\.\d*[1-9])0+)$/,"$1");}/* General Exponential always shows 2 digits exp and trims the mantissa */function SSF_normalize_exp(o/*:string*/)/*:string*/{if(o.indexOf("E")==-1)return o;return o.replace(/(?:\.0*|(\.\d*[1-9])0+)[Ee]/,"$1E").replace(/(E[+-])(\d)$/,"$10$2");}/* exponent >= -9 and <= 9 */function SSF_small_exp(v/*:number*/)/*:string*/{var w=v<0?12:11;var o=SSF_strip_decimal(v.toFixed(12));if(o.length<=w)return o;o=v.toPrecision(10);if(o.length<=w)return o;return v.toExponential(5);}/* exponent >= 11 or <= -10 likely exponential */function SSF_large_exp(v/*:number*/)/*:string*/{var o=SSF_strip_decimal(v.toFixed(11));return o.length>(v<0?12:11)||o==="0"||o==="-0"?v.toPrecision(6):o;}function SSF_general_num(v/*:number*/)/*:string*/{var V=Math.floor(Math.log(Math.abs(v))*Math.LOG10E),o;if(V>=-4&&V<=-1)o=v.toPrecision(10+V);else if(Math.abs(V)<=9)o=SSF_small_exp(v);else if(V===10)o=v.toFixed(10).substr(0,12);else o=SSF_large_exp(v);return SSF_strip_decimal(SSF_normalize_exp(o.toUpperCase()));}/*
		"General" rules:
		- text is passed through ("@")
		- booleans are rendered as TRUE/FALSE
		- "up to 11 characters" displayed for numbers
		- Default date format (code 14) used for Dates

		The longest 32-bit integer text is "-2147483648", exactly 11 chars
		TODO: technically the display depends on the width of the cell
	*/function SSF_general(v/*:any*/,opts/*:any*/){switch(typeof v){case'string':return v;case'boolean':return v?"TRUE":"FALSE";case'number':return (v|0)===v?v.toString(10):SSF_general_num(v);case'undefined':return "";case'object':if(v==null)return "";if(v instanceof Date)return SSF_format(14,datenum_local(v,opts&&opts.date1904),opts);}throw new Error("unsupported value in General format: "+v);}function SSF_fix_hijri(date/*:Date*/,o/*:[number, number, number]*/){/* TODO: properly adjust y/m/d and  */o[0]-=581;var dow=date.getDay();if(date<60)dow=(dow+6)%7;return dow;}//var THAI_DIGITS = "\u0E50\u0E51\u0E52\u0E53\u0E54\u0E55\u0E56\u0E57\u0E58\u0E59".split("");
	function SSF_write_date(type/*:number*/,fmt/*:string*/,val,ss0/*:?number*/)/*:string*/{var o="",ss=0,tt=0,y=val.y,out,outl=0;switch(type){case 98:/* 'b' buddhist year */y=val.y+543;/* falls through */case 121:/* 'y' year */switch(fmt.length){case 1:case 2:out=y%100;outl=2;break;default:out=y%10000;outl=4;break;}break;case 109:/* 'm' month */switch(fmt.length){case 1:case 2:out=val.m;outl=fmt.length;break;case 3:return months[val.m-1][1];case 5:return months[val.m-1][0];default:return months[val.m-1][2];}break;case 100:/* 'd' day */switch(fmt.length){case 1:case 2:out=val.d;outl=fmt.length;break;case 3:return days[val.q][0];default:return days[val.q][1];}break;case 104:/* 'h' 12-hour */switch(fmt.length){case 1:case 2:out=1+(val.H+11)%12;outl=fmt.length;break;default:throw 'bad hour format: '+fmt;}break;case 72:/* 'H' 24-hour */switch(fmt.length){case 1:case 2:out=val.H;outl=fmt.length;break;default:throw 'bad hour format: '+fmt;}break;case 77:/* 'M' minutes */switch(fmt.length){case 1:case 2:out=val.M;outl=fmt.length;break;default:throw 'bad minute format: '+fmt;}break;case 115:/* 's' seconds */if(fmt!='s'&&fmt!='ss'&&fmt!='.0'&&fmt!='.00'&&fmt!='.000')throw 'bad second format: '+fmt;if(val.u===0&&(fmt=="s"||fmt=="ss"))return pad0(val.S,fmt.length);/*::if(!ss0) ss0 = 0; */if(ss0>=2)tt=ss0===3?1000:100;else tt=ss0===1?10:1;ss=Math.round(tt*(val.S+val.u));if(ss>=60*tt)ss=0;if(fmt==='s')return ss===0?"0":""+ss/tt;o=pad0(ss,2+ss0);if(fmt==='ss')return o.substr(0,2);return "."+o.substr(2,fmt.length-1);case 90:/* 'Z' absolute time */switch(fmt){case'[h]':case'[hh]':out=val.D*24+val.H;break;case'[m]':case'[mm]':out=(val.D*24+val.H)*60+val.M;break;case'[s]':case'[ss]':out=((val.D*24+val.H)*60+val.M)*60+Math.round(val.S+val.u);break;default:throw 'bad abstime format: '+fmt;}outl=fmt.length===3?1:2;break;case 101:/* 'e' era */out=y;outl=1;break;}var outstr=outl>0?pad0(out,outl):"";return outstr;}/*jshint -W086 */ /*jshint +W086 */function commaify(s/*:string*/)/*:string*/{var w=3;if(s.length<=w)return s;var j=s.length%w,o=s.substr(0,j);for(;j!=s.length;j+=w)o+=(o.length>0?",":"")+s.substr(j,w);return o;}var pct1=/%/g;function write_num_pct(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{var sfmt=fmt.replace(pct1,""),mul=fmt.length-sfmt.length;return write_num(type,sfmt,val*Math.pow(10,2*mul))+fill("%",mul);}function write_num_cm(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{var idx=fmt.length-1;while(fmt.charCodeAt(idx-1)===44)--idx;return write_num(type,fmt.substr(0,idx),val/Math.pow(10,3*(fmt.length-idx)));}function write_num_exp(fmt/*:string*/,val/*:number*/)/*:string*/{var o/*:string*/;var idx=fmt.indexOf("E")-fmt.indexOf(".")-1;if(fmt.match(/^#+0.0E\+0$/)){if(val==0)return "0.0E+0";else if(val<0)return "-"+write_num_exp(fmt,-val);var period=fmt.indexOf(".");if(period===-1)period=fmt.indexOf('E');var ee=Math.floor(Math.log(val)*Math.LOG10E)%period;if(ee<0)ee+=period;o=(val/Math.pow(10,ee)).toPrecision(idx+1+(period+ee)%period);if(o.indexOf("e")===-1){var fakee=Math.floor(Math.log(val)*Math.LOG10E);if(o.indexOf(".")===-1)o=o.charAt(0)+"."+o.substr(1)+"E+"+(fakee-o.length+ee);else o+="E+"+(fakee-ee);while(o.substr(0,2)==="0."){o=o.charAt(0)+o.substr(2,period)+"."+o.substr(2+period);o=o.replace(/^0+([1-9])/,"$1").replace(/^0+\./,"0.");}o=o.replace(/\+-/,"-");}o=o.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/,function($$,$1,$2,$3){return $1+$2+$3.substr(0,(period+ee)%period)+"."+$3.substr(ee)+"E";});}else o=val.toExponential(idx);if(fmt.match(/E\+00$/)&&o.match(/e[+-]\d$/))o=o.substr(0,o.length-1)+"0"+o.charAt(o.length-1);if(fmt.match(/E\-/)&&o.match(/e\+/))o=o.replace(/e\+/,"e");return o.replace("e","E");}var frac1=/# (\?+)( ?)\/( ?)(\d+)/;function write_num_f1(r/*:Array<string>*/,aval/*:number*/,sign/*:string*/)/*:string*/{var den=parseInt(r[4],10),rr=Math.round(aval*den),base=Math.floor(rr/den);var myn=rr-base*den,myd=den;return sign+(base===0?"":""+base)+" "+(myn===0?fill(" ",r[1].length+1+r[4].length):pad_(myn,r[1].length)+r[2]+"/"+r[3]+pad0(myd,r[4].length));}function write_num_f2(r/*:Array<string>*/,aval/*:number*/,sign/*:string*/)/*:string*/{return sign+(aval===0?"":""+aval)+fill(" ",r[1].length+2+r[4].length);}var dec1=/^#*0*\.([0#]+)/;var closeparen=/\).*[0#]/;var phone=/\(###\) ###\\?-####/;function hashq(str/*:string*/)/*:string*/{var o="",cc;for(var i=0;i!=str.length;++i)switch(cc=str.charCodeAt(i)){case 35:break;case 63:o+=" ";break;case 48:o+="0";break;default:o+=String.fromCharCode(cc);}return o;}function rnd(val/*:number*/,d/*:number*/)/*:string*/{var dd=Math.pow(10,d);return ""+Math.round(val*dd)/dd;}function dec(val/*:number*/,d/*:number*/)/*:number*/{var _frac=val-Math.floor(val),dd=Math.pow(10,d);if(d<(''+Math.round(_frac*dd)).length)return 0;return Math.round(_frac*dd);}function carry(val/*:number*/,d/*:number*/)/*:number*/{if(d<(''+Math.round((val-Math.floor(val))*Math.pow(10,d))).length){return 1;}return 0;}function flr(val/*:number*/)/*:string*/{if(val<2147483647&&val>-2147483648)return ""+(val>=0?val|0:val-1|0);return ""+Math.floor(val);}function write_num_flt(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{if(type.charCodeAt(0)===40&&!fmt.match(closeparen)){var ffmt=fmt.replace(/\( */,"").replace(/ \)/,"").replace(/\)/,"");if(val>=0)return write_num_flt('n',ffmt,val);return '('+write_num_flt('n',ffmt,-val)+')';}if(fmt.charCodeAt(fmt.length-1)===44)return write_num_cm(type,fmt,val);if(fmt.indexOf('%')!==-1)return write_num_pct(type,fmt,val);if(fmt.indexOf('E')!==-1)return write_num_exp(fmt,val);if(fmt.charCodeAt(0)===36)return "$"+write_num_flt(type,fmt.substr(fmt.charAt(1)==' '?2:1),val);var o;var r/*:?Array<string>*/,ri,ff,aval=Math.abs(val),sign=val<0?"-":"";if(fmt.match(/^00+$/))return sign+pad0r(aval,fmt.length);if(fmt.match(/^[#?]+$/)){o=pad0r(val,0);if(o==="0")o="";return o.length>fmt.length?o:hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(frac1))return write_num_f1(r,aval,sign);if(fmt.match(/^#+0+$/))return sign+pad0r(aval,fmt.length-fmt.indexOf("0"));if(r=fmt.match(dec1)){o=rnd(val,r[1].length).replace(/^([^\.]+)$/,"$1."+hashq(r[1])).replace(/\.$/,"."+hashq(r[1])).replace(/\.(\d*)$/,function($$,$1){return "."+$1+fill("0",hashq(/*::(*/r/*::||[""])*/[1]).length-$1.length);});return fmt.indexOf("0.")!==-1?o:o.replace(/^0\./,".");}fmt=fmt.replace(/^#+([0.])/,"$1");if(r=fmt.match(/^(0*)\.(#*)$/)){return sign+rnd(aval,r[2].length).replace(/\.(\d*[1-9])0*$/,".$1").replace(/^(-?\d*)$/,"$1.").replace(/^0\./,r[1].length?"0.":".");}if(r=fmt.match(/^#{1,3},##0(\.?)$/))return sign+commaify(pad0r(aval,0));if(r=fmt.match(/^#,##0\.([#0]*0)$/)){return val<0?"-"+write_num_flt(type,fmt,-val):commaify(""+(Math.floor(val)+carry(val,r[1].length)))+"."+pad0(dec(val,r[1].length),r[1].length);}if(r=fmt.match(/^#,#*,#0/))return write_num_flt(type,fmt.replace(/^#,#*,/,""),val);if(r=fmt.match(/^([0#]+)(\\?-([0#]+))+$/)){o=_strrev(write_num_flt(type,fmt.replace(/[\\-]/g,""),val));ri=0;return _strrev(_strrev(fmt.replace(/\\/g,"")).replace(/[0#]/g,function(x){return ri<o.length?o.charAt(ri++):x==='0'?'0':"";}));}if(fmt.match(phone)){o=write_num_flt(type,"##########",val);return "("+o.substr(0,3)+") "+o.substr(3,3)+"-"+o.substr(6);}var oa="";if(r=fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(/*::String(*/r[4]/*::)*/.length,7);ff=SSF_frac(aval,Math.pow(10,ri)-1,false);o=""+sign;oa=write_num("n",/*::String(*/r[1]/*::)*/,ff[1]);if(oa.charAt(oa.length-1)==" ")oa=oa.substr(0,oa.length-1)+"0";o+=oa+/*::String(*/r[2]/*::)*/+"/"+/*::String(*/r[3]/*::)*/;oa=rpad_(ff[2],ri);if(oa.length<r[4].length)oa=hashq(r[4].substr(r[4].length-oa.length))+oa;o+=oa;return o;}if(r=fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(Math.max(r[1].length,r[4].length),7);ff=SSF_frac(aval,Math.pow(10,ri)-1,true);return sign+(ff[0]||(ff[1]?"":"0"))+" "+(ff[1]?pad_(ff[1],ri)+r[2]+"/"+r[3]+rpad_(ff[2],ri):fill(" ",2*ri+1+r[2].length+r[3].length));}if(r=fmt.match(/^[#0?]+$/)){o=pad0r(val,0);if(fmt.length<=o.length)return o;return hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(/^([#0?]+)\.([#0]+)$/)){o=""+val.toFixed(Math.min(r[2].length,10)).replace(/([^0])0+$/,"$1");ri=o.indexOf(".");var lres=fmt.indexOf(".")-ri,rres=fmt.length-o.length-lres;return hashq(fmt.substr(0,lres)+o+fmt.substr(fmt.length-rres));}if(r=fmt.match(/^00,000\.([#0]*0)$/)){ri=dec(val,r[1].length);return val<0?"-"+write_num_flt(type,fmt,-val):commaify(flr(val)).replace(/^\d,\d{3}$/,"0$&").replace(/^\d*$/,function($$){return "00,"+($$.length<3?pad0(0,3-$$.length):"")+$$;})+"."+pad0(ri,r[1].length);}switch(fmt){case"###,##0.00":return write_num_flt(type,"#,##0.00",val);case"###,###":case"##,###":case"#,###":var x=commaify(pad0r(aval,0));return x!=="0"?sign+x:"";case"###,###.00":return write_num_flt(type,"###,##0.00",val).replace(/^0\./,".");case"#,###.00":return write_num_flt(type,"#,##0.00",val).replace(/^0\./,".");}throw new Error("unsupported format |"+fmt+"|");}function write_num_cm2(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{var idx=fmt.length-1;while(fmt.charCodeAt(idx-1)===44)--idx;return write_num(type,fmt.substr(0,idx),val/Math.pow(10,3*(fmt.length-idx)));}function write_num_pct2(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{var sfmt=fmt.replace(pct1,""),mul=fmt.length-sfmt.length;return write_num(type,sfmt,val*Math.pow(10,2*mul))+fill("%",mul);}function write_num_exp2(fmt/*:string*/,val/*:number*/)/*:string*/{var o/*:string*/;var idx=fmt.indexOf("E")-fmt.indexOf(".")-1;if(fmt.match(/^#+0.0E\+0$/)){if(val==0)return "0.0E+0";else if(val<0)return "-"+write_num_exp2(fmt,-val);var period=fmt.indexOf(".");if(period===-1)period=fmt.indexOf('E');var ee=Math.floor(Math.log(val)*Math.LOG10E)%period;if(ee<0)ee+=period;o=(val/Math.pow(10,ee)).toPrecision(idx+1+(period+ee)%period);if(!o.match(/[Ee]/)){var fakee=Math.floor(Math.log(val)*Math.LOG10E);if(o.indexOf(".")===-1)o=o.charAt(0)+"."+o.substr(1)+"E+"+(fakee-o.length+ee);else o+="E+"+(fakee-ee);o=o.replace(/\+-/,"-");}o=o.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/,function($$,$1,$2,$3){return $1+$2+$3.substr(0,(period+ee)%period)+"."+$3.substr(ee)+"E";});}else o=val.toExponential(idx);if(fmt.match(/E\+00$/)&&o.match(/e[+-]\d$/))o=o.substr(0,o.length-1)+"0"+o.charAt(o.length-1);if(fmt.match(/E\-/)&&o.match(/e\+/))o=o.replace(/e\+/,"e");return o.replace("e","E");}function write_num_int(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{if(type.charCodeAt(0)===40&&!fmt.match(closeparen)){var ffmt=fmt.replace(/\( */,"").replace(/ \)/,"").replace(/\)/,"");if(val>=0)return write_num_int('n',ffmt,val);return '('+write_num_int('n',ffmt,-val)+')';}if(fmt.charCodeAt(fmt.length-1)===44)return write_num_cm2(type,fmt,val);if(fmt.indexOf('%')!==-1)return write_num_pct2(type,fmt,val);if(fmt.indexOf('E')!==-1)return write_num_exp2(fmt,val);if(fmt.charCodeAt(0)===36)return "$"+write_num_int(type,fmt.substr(fmt.charAt(1)==' '?2:1),val);var o;var r/*:?Array<string>*/,ri,ff,aval=Math.abs(val),sign=val<0?"-":"";if(fmt.match(/^00+$/))return sign+pad0(aval,fmt.length);if(fmt.match(/^[#?]+$/)){o=""+val;if(val===0)o="";return o.length>fmt.length?o:hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(frac1))return write_num_f2(r,aval,sign);if(fmt.match(/^#+0+$/))return sign+pad0(aval,fmt.length-fmt.indexOf("0"));if(r=fmt.match(dec1)){/*:: if(!Array.isArray(r)) throw new Error("unreachable"); */o=(""+val).replace(/^([^\.]+)$/,"$1."+hashq(r[1])).replace(/\.$/,"."+hashq(r[1]));o=o.replace(/\.(\d*)$/,function($$,$1){/*:: if(!Array.isArray(r)) throw new Error("unreachable"); */return "."+$1+fill("0",hashq(r[1]).length-$1.length);});return fmt.indexOf("0.")!==-1?o:o.replace(/^0\./,".");}fmt=fmt.replace(/^#+([0.])/,"$1");if(r=fmt.match(/^(0*)\.(#*)$/)){return sign+(""+aval).replace(/\.(\d*[1-9])0*$/,".$1").replace(/^(-?\d*)$/,"$1.").replace(/^0\./,r[1].length?"0.":".");}if(r=fmt.match(/^#{1,3},##0(\.?)$/))return sign+commaify(""+aval);if(r=fmt.match(/^#,##0\.([#0]*0)$/)){return val<0?"-"+write_num_int(type,fmt,-val):commaify(""+val)+"."+fill('0',r[1].length);}if(r=fmt.match(/^#,#*,#0/))return write_num_int(type,fmt.replace(/^#,#*,/,""),val);if(r=fmt.match(/^([0#]+)(\\?-([0#]+))+$/)){o=_strrev(write_num_int(type,fmt.replace(/[\\-]/g,""),val));ri=0;return _strrev(_strrev(fmt.replace(/\\/g,"")).replace(/[0#]/g,function(x){return ri<o.length?o.charAt(ri++):x==='0'?'0':"";}));}if(fmt.match(phone)){o=write_num_int(type,"##########",val);return "("+o.substr(0,3)+") "+o.substr(3,3)+"-"+o.substr(6);}var oa="";if(r=fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(/*::String(*/r[4]/*::)*/.length,7);ff=SSF_frac(aval,Math.pow(10,ri)-1,false);o=""+sign;oa=write_num("n",/*::String(*/r[1]/*::)*/,ff[1]);if(oa.charAt(oa.length-1)==" ")oa=oa.substr(0,oa.length-1)+"0";o+=oa+/*::String(*/r[2]/*::)*/+"/"+/*::String(*/r[3]/*::)*/;oa=rpad_(ff[2],ri);if(oa.length<r[4].length)oa=hashq(r[4].substr(r[4].length-oa.length))+oa;o+=oa;return o;}if(r=fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(Math.max(r[1].length,r[4].length),7);ff=SSF_frac(aval,Math.pow(10,ri)-1,true);return sign+(ff[0]||(ff[1]?"":"0"))+" "+(ff[1]?pad_(ff[1],ri)+r[2]+"/"+r[3]+rpad_(ff[2],ri):fill(" ",2*ri+1+r[2].length+r[3].length));}if(r=fmt.match(/^[#0?]+$/)){o=""+val;if(fmt.length<=o.length)return o;return hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(/^([#0]+)\.([#0]+)$/)){o=""+val.toFixed(Math.min(r[2].length,10)).replace(/([^0])0+$/,"$1");ri=o.indexOf(".");var lres=fmt.indexOf(".")-ri,rres=fmt.length-o.length-lres;return hashq(fmt.substr(0,lres)+o+fmt.substr(fmt.length-rres));}if(r=fmt.match(/^00,000\.([#0]*0)$/)){return val<0?"-"+write_num_int(type,fmt,-val):commaify(""+val).replace(/^\d,\d{3}$/,"0$&").replace(/^\d*$/,function($$){return "00,"+($$.length<3?pad0(0,3-$$.length):"")+$$;})+"."+pad0(0,r[1].length);}switch(fmt){case"###,###":case"##,###":case"#,###":var x=commaify(""+aval);return x!=="0"?sign+x:"";default:if(fmt.match(/\.[0#?]*$/))return write_num_int(type,fmt.slice(0,fmt.lastIndexOf(".")),val)+hashq(fmt.slice(fmt.lastIndexOf(".")));}throw new Error("unsupported format |"+fmt+"|");}function write_num(type/*:string*/,fmt/*:string*/,val/*:number*/)/*:string*/{return (val|0)===val?write_num_int(type,fmt,val):write_num_flt(type,fmt,val);}function SSF_split_fmt(fmt/*:string*/)/*:Array<string>*/{var out/*:Array<string>*/=[];var in_str=false/*, cc*/;for(var i=0,j=0;i<fmt.length;++i)switch(/*cc=*/fmt.charCodeAt(i)){case 34:/* '"' */in_str=!in_str;break;case 95:case 42:case 92:/* '_' '*' '\\' */++i;break;case 59:/* ';' */out[out.length]=fmt.substr(j,i-j);j=i+1;}out[out.length]=fmt.substr(j);if(in_str===true)throw new Error("Format |"+fmt+"| unterminated string ");return out;}var SSF_abstime=/\[[HhMmSs\u0E0A\u0E19\u0E17]*\]/;function fmt_is_date(fmt/*:string*/)/*:boolean*/{var i=0,/*cc = 0,*/c="",o="";while(i<fmt.length){switch(c=fmt.charAt(i)){case'G':if(SSF_isgeneral(fmt,i))i+=6;i++;break;case'"':for(;/*cc=*/fmt.charCodeAt(++i)!==34&&i<fmt.length;){/*empty*/}++i;break;case'\\':i+=2;break;case'_':i+=2;break;case'@':++i;break;case'B':case'b':if(fmt.charAt(i+1)==="1"||fmt.charAt(i+1)==="2")return true;/* falls through */case'M':case'D':case'Y':case'H':case'S':case'E':/* falls through */case'm':case'd':case'y':case'h':case's':case'e':case'g':return true;case'A':case'a':case'上':if(fmt.substr(i,3).toUpperCase()==="A/P")return true;if(fmt.substr(i,5).toUpperCase()==="AM/PM")return true;if(fmt.substr(i,5).toUpperCase()==="上午/下午")return true;++i;break;case'[':o=c;while(fmt.charAt(i++)!==']'&&i<fmt.length)o+=fmt.charAt(i);if(o.match(SSF_abstime))return true;break;case'.':/* falls through */case'0':case'#':while(i<fmt.length&&("0#?.,E+-%".indexOf(c=fmt.charAt(++i))>-1||c=='\\'&&fmt.charAt(i+1)=="-"&&"0#".indexOf(fmt.charAt(i+2))>-1)){/* empty */}break;case'?':while(fmt.charAt(++i)===c){/* empty */}break;case'*':++i;if(fmt.charAt(i)==' '||fmt.charAt(i)=='*')++i;break;case'(':case')':++i;break;case'1':case'2':case'3':case'4':case'5':case'6':case'7':case'8':case'9':while(i<fmt.length&&"0123456789".indexOf(fmt.charAt(++i))>-1){/* empty */}break;case' ':++i;break;default:++i;break;}}return false;}function eval_fmt(fmt/*:string*/,v/*:any*/,opts/*:any*/,flen/*:number*/){var out=[],o="",i=0,c="",lst='t',dt,j,cc;var hr='H';/* Tokenize */while(i<fmt.length){switch(c=fmt.charAt(i)){case'G':/* General */if(!SSF_isgeneral(fmt,i))throw new Error('unrecognized character '+c+' in '+fmt);out[out.length]={t:'G',v:'General'};i+=7;break;case'"':/* Literal text */for(o="";(cc=fmt.charCodeAt(++i))!==34&&i<fmt.length;)o+=String.fromCharCode(cc);out[out.length]={t:'t',v:o};++i;break;case'\\':var w=fmt.charAt(++i),t=w==="("||w===")"?w:'t';out[out.length]={t:t,v:w};++i;break;case'_':out[out.length]={t:'t',v:" "};i+=2;break;case'@':/* Text Placeholder */out[out.length]={t:'T',v:v};++i;break;case'B':case'b':if(fmt.charAt(i+1)==="1"||fmt.charAt(i+1)==="2"){if(dt==null){dt=SSF_parse_date_code(v,opts,fmt.charAt(i+1)==="2");if(dt==null)return "";}out[out.length]={t:'X',v:fmt.substr(i,2)};lst=c;i+=2;break;}/* falls through */case'M':case'D':case'Y':case'H':case'S':case'E':c=c.toLowerCase();/* falls through */case'm':case'd':case'y':case'h':case's':case'e':case'g':if(v<0)return "";if(dt==null){dt=SSF_parse_date_code(v,opts);if(dt==null)return "";}o=c;while(++i<fmt.length&&fmt.charAt(i).toLowerCase()===c)o+=c;if(c==='m'&&lst.toLowerCase()==='h')c='M';if(c==='h')c=hr;out[out.length]={t:c,v:o};lst=c;break;case'A':case'a':case'上':var q={t:c,v:c};if(dt==null)dt=SSF_parse_date_code(v,opts);if(fmt.substr(i,3).toUpperCase()==="A/P"){if(dt!=null)q.v=dt.H>=12?"P":"A";q.t='T';hr='h';i+=3;}else if(fmt.substr(i,5).toUpperCase()==="AM/PM"){if(dt!=null)q.v=dt.H>=12?"PM":"AM";q.t='T';i+=5;hr='h';}else if(fmt.substr(i,5).toUpperCase()==="上午/下午"){if(dt!=null)q.v=dt.H>=12?"下午":"上午";q.t='T';i+=5;hr='h';}else {q.t="t";++i;}if(dt==null&&q.t==='T')return "";out[out.length]=q;lst=c;break;case'[':o=c;while(fmt.charAt(i++)!==']'&&i<fmt.length)o+=fmt.charAt(i);if(o.slice(-1)!==']')throw 'unterminated "[" block: |'+o+'|';if(o.match(SSF_abstime)){if(dt==null){dt=SSF_parse_date_code(v,opts);if(dt==null)return "";}out[out.length]={t:'Z',v:o.toLowerCase()};lst=o.charAt(1);}else if(o.indexOf("$")>-1){o=(o.match(/\$([^-\[\]]*)/)||[])[1]||"$";if(!fmt_is_date(fmt))out[out.length]={t:'t',v:o};}break;/* Numbers */case'.':if(dt!=null){o=c;while(++i<fmt.length&&(c=fmt.charAt(i))==="0")o+=c;out[out.length]={t:'s',v:o};break;}/* falls through */case'0':case'#':o=c;while(++i<fmt.length&&"0#?.,E+-%".indexOf(c=fmt.charAt(i))>-1)o+=c;out[out.length]={t:'n',v:o};break;case'?':o=c;while(fmt.charAt(++i)===c)o+=c;out[out.length]={t:c,v:o};lst=c;break;case'*':++i;if(fmt.charAt(i)==' '||fmt.charAt(i)=='*')++i;break;// **
	case'(':case')':out[out.length]={t:flen===1?'t':c,v:c};++i;break;case'1':case'2':case'3':case'4':case'5':case'6':case'7':case'8':case'9':o=c;while(i<fmt.length&&"0123456789".indexOf(fmt.charAt(++i))>-1)o+=fmt.charAt(i);out[out.length]={t:'D',v:o};break;case' ':out[out.length]={t:c,v:c};++i;break;case'$':out[out.length]={t:'t',v:'$'};++i;break;default:if(",$-+/():!^&'~{}<>=€acfijklopqrtuvwxzP".indexOf(c)===-1)throw new Error('unrecognized character '+c+' in '+fmt);out[out.length]={t:'t',v:c};++i;break;}}/* Scan for date/time parts */var bt=0,ss0=0,ssm;for(i=out.length-1,lst='t';i>=0;--i){switch(out[i].t){case'h':case'H':out[i].t=hr;lst='h';if(bt<1)bt=1;break;case's':if(ssm=out[i].v.match(/\.0+$/))ss0=Math.max(ss0,ssm[0].length-1);if(bt<3)bt=3;/* falls through */case'd':case'y':case'M':case'e':lst=out[i].t;break;case'm':if(lst==='s'){out[i].t='M';if(bt<2)bt=2;}break;case'X':/*if(out[i].v === "B2");*/break;case'Z':if(bt<1&&out[i].v.match(/[Hh]/))bt=1;if(bt<2&&out[i].v.match(/[Mm]/))bt=2;if(bt<3&&out[i].v.match(/[Ss]/))bt=3;}}/* time rounding depends on presence of minute / second / usec fields */switch(bt){case 0:break;case 1:/*::if(!dt) break;*/if(dt.u>=0.5){dt.u=0;++dt.S;}if(dt.S>=60){dt.S=0;++dt.M;}if(dt.M>=60){dt.M=0;++dt.H;}break;case 2:/*::if(!dt) break;*/if(dt.u>=0.5){dt.u=0;++dt.S;}if(dt.S>=60){dt.S=0;++dt.M;}break;}/* replace fields */var nstr="",jj;for(i=0;i<out.length;++i){switch(out[i].t){case't':case'T':case' ':case'D':break;case'X':out[i].v="";out[i].t=";";break;case'd':case'm':case'y':case'h':case'H':case'M':case's':case'e':case'b':case'Z':/*::if(!dt) throw "unreachable"; */out[i].v=SSF_write_date(out[i].t.charCodeAt(0),out[i].v,dt,ss0);out[i].t='t';break;case'n':case'?':jj=i+1;while(out[jj]!=null&&((c=out[jj].t)==="?"||c==="D"||(c===" "||c==="t")&&out[jj+1]!=null&&(out[jj+1].t==='?'||out[jj+1].t==="t"&&out[jj+1].v==='/')||out[i].t==='('&&(c===' '||c==='n'||c===')')||c==='t'&&(out[jj].v==='/'||out[jj].v===' '&&out[jj+1]!=null&&out[jj+1].t=='?'))){out[i].v+=out[jj].v;out[jj]={v:"",t:";"};++jj;}nstr+=out[i].v;i=jj-1;break;case'G':out[i].t='t';out[i].v=SSF_general(v,opts);break;}}var vv="",myv,ostr;if(nstr.length>0){if(nstr.charCodeAt(0)==40)/* '(' */{myv=v<0&&nstr.charCodeAt(0)===45?-v:v;ostr=write_num('n',nstr,myv);}else {myv=v<0&&flen>1?-v:v;ostr=write_num('n',nstr,myv);if(myv<0&&out[0]&&out[0].t=='t'){ostr=ostr.substr(1);out[0].v="-"+out[0].v;}}jj=ostr.length-1;var decpt=out.length;for(i=0;i<out.length;++i)if(out[i]!=null&&out[i].t!='t'&&out[i].v.indexOf(".")>-1){decpt=i;break;}var lasti=out.length;if(decpt===out.length&&ostr.indexOf("E")===-1){for(i=out.length-1;i>=0;--i){if(out[i]==null||'n?'.indexOf(out[i].t)===-1)continue;if(jj>=out[i].v.length-1){jj-=out[i].v.length;out[i].v=ostr.substr(jj+1,out[i].v.length);}else if(jj<0)out[i].v="";else {out[i].v=ostr.substr(0,jj+1);jj=-1;}out[i].t='t';lasti=i;}if(jj>=0&&lasti<out.length)out[lasti].v=ostr.substr(0,jj+1)+out[lasti].v;}else if(decpt!==out.length&&ostr.indexOf("E")===-1){jj=ostr.indexOf(".")-1;for(i=decpt;i>=0;--i){if(out[i]==null||'n?'.indexOf(out[i].t)===-1)continue;j=out[i].v.indexOf(".")>-1&&i===decpt?out[i].v.indexOf(".")-1:out[i].v.length-1;vv=out[i].v.substr(j+1);for(;j>=0;--j){if(jj>=0&&(out[i].v.charAt(j)==="0"||out[i].v.charAt(j)==="#"))vv=ostr.charAt(jj--)+vv;}out[i].v=vv;out[i].t='t';lasti=i;}if(jj>=0&&lasti<out.length)out[lasti].v=ostr.substr(0,jj+1)+out[lasti].v;jj=ostr.indexOf(".")+1;for(i=decpt;i<out.length;++i){if(out[i]==null||'n?('.indexOf(out[i].t)===-1&&i!==decpt)continue;j=out[i].v.indexOf(".")>-1&&i===decpt?out[i].v.indexOf(".")+1:0;vv=out[i].v.substr(0,j);for(;j<out[i].v.length;++j){if(jj<ostr.length)vv+=ostr.charAt(jj++);}out[i].v=vv;out[i].t='t';lasti=i;}}}for(i=0;i<out.length;++i)if(out[i]!=null&&'n?'.indexOf(out[i].t)>-1){myv=flen>1&&v<0&&i>0&&out[i-1].v==="-"?-v:v;out[i].v=write_num(out[i].t,out[i].v,myv);out[i].t='t';}var retval="";for(i=0;i!==out.length;++i)if(out[i]!=null)retval+=out[i].v;return retval;}var cfregex2=/\[(=|>[=]?|<[>=]?)(-?\d+(?:\.\d*)?)\]/;function chkcond(v,rr){if(rr==null)return false;var thresh=parseFloat(rr[2]);switch(rr[1]){case"=":if(v==thresh)return true;break;case">":if(v>thresh)return true;break;case"<":if(v<thresh)return true;break;case"<>":if(v!=thresh)return true;break;case">=":if(v>=thresh)return true;break;case"<=":if(v<=thresh)return true;break;}return false;}function choose_fmt(f/*:string*/,v/*:any*/){var fmt=SSF_split_fmt(f);var l=fmt.length,lat=fmt[l-1].indexOf("@");if(l<4&&lat>-1)--l;if(fmt.length>4)throw new Error("cannot find right format for |"+fmt.join("|")+"|");if(typeof v!=="number")return [4,fmt.length===4||lat>-1?fmt[fmt.length-1]:"@"];switch(fmt.length){case 1:fmt=lat>-1?["General","General","General",fmt[0]]:[fmt[0],fmt[0],fmt[0],"@"];break;case 2:fmt=lat>-1?[fmt[0],fmt[0],fmt[0],fmt[1]]:[fmt[0],fmt[1],fmt[0],"@"];break;case 3:fmt=lat>-1?[fmt[0],fmt[1],fmt[0],fmt[2]]:[fmt[0],fmt[1],fmt[2],"@"];break;}var ff=v>0?fmt[0]:v<0?fmt[1]:fmt[2];if(fmt[0].indexOf("[")===-1&&fmt[1].indexOf("[")===-1)return [l,ff];if(fmt[0].match(/\[[=<>]/)!=null||fmt[1].match(/\[[=<>]/)!=null){var m1=fmt[0].match(cfregex2);var m2=fmt[1].match(cfregex2);return chkcond(v,m1)?[l,fmt[0]]:chkcond(v,m2)?[l,fmt[1]]:[l,fmt[m1!=null&&m2!=null?2:1]];}return [l,ff];}function SSF_format(fmt/*:string|number*/,v/*:any*/,o/*:?any*/){if(o==null)o={};var sfmt="";switch(typeof fmt){case"string":if(fmt=="m/d/yy"&&o.dateNF)sfmt=o.dateNF;else sfmt=fmt;break;case"number":if(fmt==14&&o.dateNF)sfmt=o.dateNF;else sfmt=(o.table!=null?o.table/*:any*/:table_fmt)[fmt];if(sfmt==null)sfmt=o.table&&o.table[SSF_default_map[fmt]]||table_fmt[SSF_default_map[fmt]];if(sfmt==null)sfmt=SSF_default_str[fmt]||"General";break;}if(SSF_isgeneral(sfmt,0))return SSF_general(v,o);if(v instanceof Date)v=datenum_local(v,o.date1904);var f=choose_fmt(sfmt,v);if(SSF_isgeneral(f[1]))return SSF_general(v,o);if(v===true)v="TRUE";else if(v===false)v="FALSE";else if(v===""||v==null)return "";return eval_fmt(f[1],v,o,f[0]);}function SSF_load(fmt/*:string*/,idx/*:?number*/)/*:number*/{if(typeof idx!='number'){idx=+idx||-1;/*::if(typeof idx != 'number') return 0x188; */for(var i=0;i<0x0188;++i){/*::if(typeof idx != 'number') return 0x188; */if(table_fmt[i]==undefined){if(idx<0)idx=i;continue;}if(table_fmt[i]==fmt){idx=i;break;}}/*::if(typeof idx != 'number') return 0x188; */if(idx<0)idx=0x187;}/*::if(typeof idx != 'number') return 0x188; */table_fmt[idx]=fmt;return idx;}function make_ssf(){table_fmt=SSF_init_table();}var SSFImplicit/*{[number]:string}*/={"5":'"$"#,##0_);\\("$"#,##0\\)',"6":'"$"#,##0_);[Red]\\("$"#,##0\\)',"7":'"$"#,##0.00_);\\("$"#,##0.00\\)',"8":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"23":'General',"24":'General',"25":'General',"26":'General',"27":'m/d/yy',"28":'m/d/yy',"29":'m/d/yy',"30":'m/d/yy',"31":'m/d/yy',"32":'h:mm:ss',"33":'h:mm:ss',"34":'h:mm:ss',"35":'h:mm:ss',"36":'m/d/yy',"41":'_(* #,##0_);_(* \(#,##0\);_(* "-"_);_(@_)',"42":'_("$"* #,##0_);_("$"* \(#,##0\);_("$"* "-"_);_(@_)',"43":'_(* #,##0.00_);_(* \(#,##0.00\);_(* "-"??_);_(@_)',"44":'_("$"* #,##0.00_);_("$"* \(#,##0.00\);_("$"* "-"??_);_(@_)',"50":'m/d/yy',"51":'m/d/yy',"52":'m/d/yy',"53":'m/d/yy',"54":'m/d/yy',"55":'m/d/yy',"56":'m/d/yy',"57":'m/d/yy',"58":'m/d/yy',"59":'0',"60":'0.00',"61":'#,##0',"62":'#,##0.00',"63":'"$"#,##0_);\\("$"#,##0\\)',"64":'"$"#,##0_);[Red]\\("$"#,##0\\)',"65":'"$"#,##0.00_);\\("$"#,##0.00\\)',"66":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"67":'0%',"68":'0.00%',"69":'# ?/?',"70":'# ??/??',"71":'m/d/yy',"72":'m/d/yy',"73":'d-mmm-yy',"74":'d-mmm',"75":'mmm-yy',"76":'h:mm',"77":'h:mm:ss',"78":'m/d/yy h:mm',"79":'mm:ss',"80":'[h]:mm:ss',"81":'mmss.0'}/*:any*/;/* dateNF parse TODO: move to SSF */var dateNFregex=/[dD]+|[mM]+|[yYeE]+|[Hh]+|[Ss]+/g;function dateNF_regex(dateNF/*:string|number*/)/*:RegExp*/{var fmt=typeof dateNF=="number"?table_fmt[dateNF]:dateNF;fmt=fmt.replace(dateNFregex,"(\\d+)");return new RegExp("^"+fmt+"$");}function dateNF_fix(str/*:string*/,dateNF/*:string*/,match/*:Array<string>*/)/*:string*/{var Y=-1,m=-1,d=-1,H=-1,M=-1,S=-1;(dateNF.match(dateNFregex)||[]).forEach(function(n,i){var v=parseInt(match[i+1],10);switch(n.toLowerCase().charAt(0)){case'y':Y=v;break;case'd':d=v;break;case'h':H=v;break;case's':S=v;break;case'm':if(H>=0)M=v;else m=v;break;}});if(S>=0&&M==-1&&m>=0){M=m;m=-1;}var datestr=(""+(Y>=0?Y:new Date().getFullYear())).slice(-4)+"-"+("00"+(m>=1?m:1)).slice(-2)+"-"+("00"+(d>=1?d:1)).slice(-2);if(datestr.length==7)datestr="0"+datestr;if(datestr.length==8)datestr="20"+datestr;var timestr=("00"+(H>=0?H:0)).slice(-2)+":"+("00"+(M>=0?M:0)).slice(-2)+":"+("00"+(S>=0?S:0)).slice(-2);if(H==-1&&M==-1&&S==-1)return datestr;if(Y==-1&&m==-1&&d==-1)return timestr;return datestr+"T"+timestr;}/*::
	declare var ReadShift:any;
	declare var CheckField:any;
	declare var prep_blob:any;
	declare var __readUInt32LE:any;
	declare var __readInt32LE:any;
	declare var __toBuffer:any;
	declare var __utf16le:any;
	declare var bconcat:any;
	declare var s2a:any;
	declare var chr0:any;
	declare var chr1:any;
	declare var has_buf:boolean;
	declare var new_buf:any;
	declare var new_raw_buf:any;
	declare var new_unsafe_buf:any;
	declare var Buffer_from:any;
	*/ /* cfb.js (C) 2013-present SheetJS -- http://sheetjs.com */ /* vim: set ts=2: */ /*jshint eqnull:true */ /*exported CFB */ /*global Uint8Array:false, Uint16Array:false */ /*::
	type SectorEntry = {
		name?:string;
		nodes?:Array<number>;
		data:RawBytes;
	};
	type SectorList = {
		[k:string|number]:SectorEntry;
		name:?string;
		fat_addrs:Array<number>;
		ssz:number;
	}
	type CFBFiles = {[n:string]:CFBEntry};
	*/ /* crc32.js (C) 2014-present SheetJS -- http://sheetjs.com */ /* vim: set ts=2: */ /*exported CRC32 */var CRC32=/*#__PURE__*/function(){var CRC32={};CRC32.version='1.2.0';/* see perf/crc32table.js */ /*global Int32Array */function signed_crc_table()/*:any*/{var c=0,table/*:Array<number>*/=new Array(256);for(var n=0;n!=256;++n){c=n;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;c=c&1?-306674912^c>>>1:c>>>1;table[n]=c;}return typeof Int32Array!=='undefined'?new Int32Array(table):table;}var T0=signed_crc_table();function slice_by_16_tables(T){var c=0,v=0,n=0,table/*:Array<number>*/=typeof Int32Array!=='undefined'?new Int32Array(4096):new Array(4096);for(n=0;n!=256;++n)table[n]=T[n];for(n=0;n!=256;++n){v=T[n];for(c=256+n;c<4096;c+=256)v=table[c]=v>>>8^T[v&0xFF];}var out=[];for(n=1;n!=16;++n)out[n-1]=typeof Int32Array!=='undefined'?table.subarray(n*256,n*256+256):table.slice(n*256,n*256+256);return out;}var TT=slice_by_16_tables(T0);var T1=TT[0],T2=TT[1],T3=TT[2],T4=TT[3],T5=TT[4];var T6=TT[5],T7=TT[6],T8=TT[7],T9=TT[8],Ta=TT[9];var Tb=TT[10],Tc=TT[11],Td=TT[12],Te=TT[13],Tf=TT[14];function crc32_bstr(bstr/*:string*/,seed/*:number*/)/*:number*/{var C=seed/*:: ? 0 : 0 */^-1;for(var i=0,L=bstr.length;i<L;)C=C>>>8^T0[(C^bstr.charCodeAt(i++))&0xFF];return ~C;}function crc32_buf(B/*:Uint8Array|Array<number>*/,seed/*:number*/)/*:number*/{var C=seed/*:: ? 0 : 0 */^-1,L=B.length-15,i=0;for(;i<L;)C=Tf[B[i++]^C&255]^Te[B[i++]^C>>8&255]^Td[B[i++]^C>>16&255]^Tc[B[i++]^C>>>24]^Tb[B[i++]]^Ta[B[i++]]^T9[B[i++]]^T8[B[i++]]^T7[B[i++]]^T6[B[i++]]^T5[B[i++]]^T4[B[i++]]^T3[B[i++]]^T2[B[i++]]^T1[B[i++]]^T0[B[i++]];L+=15;while(i<L)C=C>>>8^T0[(C^B[i++])&0xFF];return ~C;}function crc32_str(str/*:string*/,seed/*:number*/)/*:number*/{var C=seed^-1;for(var i=0,L=str.length,c=0,d=0;i<L;){c=str.charCodeAt(i++);if(c<0x80){C=C>>>8^T0[(C^c)&0xFF];}else if(c<0x800){C=C>>>8^T0[(C^(192|c>>6&31))&0xFF];C=C>>>8^T0[(C^(128|c&63))&0xFF];}else if(c>=0xD800&&c<0xE000){c=(c&1023)+64;d=str.charCodeAt(i++)&1023;C=C>>>8^T0[(C^(240|c>>8&7))&0xFF];C=C>>>8^T0[(C^(128|c>>2&63))&0xFF];C=C>>>8^T0[(C^(128|d>>6&15|(c&3)<<4))&0xFF];C=C>>>8^T0[(C^(128|d&63))&0xFF];}else {C=C>>>8^T0[(C^(224|c>>12&15))&0xFF];C=C>>>8^T0[(C^(128|c>>6&63))&0xFF];C=C>>>8^T0[(C^(128|c&63))&0xFF];}}return ~C;}CRC32.table=T0;CRC32.bstr=crc32_bstr;CRC32.buf=crc32_buf;CRC32.str=crc32_str;return CRC32;}();/* [MS-CFB] v20171201 */var CFB=/*#__PURE__*/function _CFB(){var exports={};exports.version='1.2.1';/* [MS-CFB] 2.6.4 */function namecmp(l/*:string*/,r/*:string*/)/*:number*/{var L=l.split("/"),R=r.split("/");for(var i=0,c=0,Z=Math.min(L.length,R.length);i<Z;++i){if(c=L[i].length-R[i].length)return c;if(L[i]!=R[i])return L[i]<R[i]?-1:1;}return L.length-R.length;}function dirname(p/*:string*/)/*:string*/{if(p.charAt(p.length-1)=="/")return p.slice(0,-1).indexOf("/")===-1?p:dirname(p.slice(0,-1));var c=p.lastIndexOf("/");return c===-1?p:p.slice(0,c+1);}function filename(p/*:string*/)/*:string*/{if(p.charAt(p.length-1)=="/")return filename(p.slice(0,-1));var c=p.lastIndexOf("/");return c===-1?p:p.slice(c+1);}/* -------------------------------------------------------------------------- */ /* DOS Date format:
	   high|YYYYYYYm.mmmddddd.HHHHHMMM.MMMSSSSS|low
	   add 1980 to stored year
	   stored second should be doubled
	*/ /* write JS date to buf as a DOS date */function write_dos_date(buf/*:CFBlob*/,date/*:Date|string*/){if(typeof date==="string")date=new Date(date);var hms/*:number*/=date.getHours();hms=hms<<6|date.getMinutes();hms=hms<<5|date.getSeconds()>>>1;buf.write_shift(2,hms);var ymd/*:number*/=date.getFullYear()-1980;ymd=ymd<<4|date.getMonth()+1;ymd=ymd<<5|date.getDate();buf.write_shift(2,ymd);}/* read four bytes from buf and interpret as a DOS date */function parse_dos_date(buf/*:CFBlob*/)/*:Date*/{var hms=buf.read_shift(2)&0xFFFF;var ymd=buf.read_shift(2)&0xFFFF;var val=new Date();var d=ymd&0x1F;ymd>>>=5;var m=ymd&0x0F;ymd>>>=4;val.setMilliseconds(0);val.setFullYear(ymd+1980);val.setMonth(m-1);val.setDate(d);var S=hms&0x1F;hms>>>=5;var M=hms&0x3F;hms>>>=6;val.setHours(hms);val.setMinutes(M);val.setSeconds(S<<1);return val;}function parse_extra_field(blob/*:CFBlob*/)/*:any*/{prep_blob(blob,0);var o=/*::(*/{}/*:: :any)*/;var flags=0;while(blob.l<=blob.length-4){var type=blob.read_shift(2);var sz=blob.read_shift(2),tgt=blob.l+sz;var p={};switch(type){/* UNIX-style Timestamps */case 0x5455:{flags=blob.read_shift(1);if(flags&1)p.mtime=blob.read_shift(4);/* for some reason, CD flag corresponds to LFH */if(sz>5){if(flags&2)p.atime=blob.read_shift(4);if(flags&4)p.ctime=blob.read_shift(4);}if(p.mtime)p.mt=new Date(p.mtime*1000);}break;}blob.l=tgt;o[type]=p;}return o;}var fs/*:: = require('fs'); */;function get_fs(){return fs||(fs={});}function parse(file/*:RawBytes*/,options/*:CFBReadOpts*/)/*:CFBContainer*/{if(file[0]==0x50&&file[1]==0x4b)return parse_zip(file,options);if((file[0]|0x20)==0x6d&&(file[1]|0x20)==0x69)return parse_mad(file,options);if(file.length<512)throw new Error("CFB file size "+file.length+" < 512");var mver=3;var ssz=512;var nmfs=0;// number of mini FAT sectors
	var difat_sec_cnt=0;var dir_start=0;var minifat_start=0;var difat_start=0;var fat_addrs/*:Array<number>*/=[];// locations of FAT sectors
	/* [MS-CFB] 2.2 Compound File Header */var blob/*:CFBlob*/=/*::(*/file.slice(0,512)/*:: :any)*/;prep_blob(blob,0);/* major version */var mv=check_get_mver(blob);mver=mv[0];switch(mver){case 3:ssz=512;break;case 4:ssz=4096;break;case 0:if(mv[1]==0)return parse_zip(file,options);/* falls through */default:throw new Error("Major Version: Expected 3 or 4 saw "+mver);}/* reprocess header */if(ssz!==512){blob=/*::(*/file.slice(0,ssz)/*:: :any)*/;prep_blob(blob,28/* blob.l */);}/* Save header for final object */var header/*:RawBytes*/=file.slice(0,ssz);check_shifts(blob,mver);// Number of Directory Sectors
	var dir_cnt/*:number*/=blob.read_shift(4,'i');if(mver===3&&dir_cnt!==0)throw new Error('# Directory Sectors: Expected 0 saw '+dir_cnt);// Number of FAT Sectors
	blob.l+=4;// First Directory Sector Location
	dir_start=blob.read_shift(4,'i');// Transaction Signature
	blob.l+=4;// Mini Stream Cutoff Size
	blob.chk('00100000','Mini Stream Cutoff Size: ');// First Mini FAT Sector Location
	minifat_start=blob.read_shift(4,'i');// Number of Mini FAT Sectors
	nmfs=blob.read_shift(4,'i');// First DIFAT sector location
	difat_start=blob.read_shift(4,'i');// Number of DIFAT Sectors
	difat_sec_cnt=blob.read_shift(4,'i');// Grab FAT Sector Locations
	for(var q=-1,j=0;j<109;++j){/* 109 = (512 - blob.l)>>>2; */q=blob.read_shift(4,'i');if(q<0)break;fat_addrs[j]=q;}/** Break the file up into sectors */var sectors/*:Array<RawBytes>*/=sectorify(file,ssz);sleuth_fat(difat_start,difat_sec_cnt,sectors,ssz,fat_addrs);/** Chains */var sector_list/*:SectorList*/=make_sector_list(sectors,dir_start,fat_addrs,ssz);sector_list[dir_start].name="!Directory";if(nmfs>0&&minifat_start!==ENDOFCHAIN)sector_list[minifat_start].name="!MiniFAT";sector_list[fat_addrs[0]].name="!FAT";sector_list.fat_addrs=fat_addrs;sector_list.ssz=ssz;/* [MS-CFB] 2.6.1 Compound File Directory Entry */var files/*:CFBFiles*/={},Paths/*:Array<string>*/=[],FileIndex/*:CFBFileIndex*/=[],FullPaths/*:Array<string>*/=[];read_directory(dir_start,sector_list,sectors,Paths,nmfs,files,FileIndex,minifat_start);build_full_paths(FileIndex,FullPaths,Paths);Paths.shift();var o={FileIndex:FileIndex,FullPaths:FullPaths};// $FlowIgnore
	if(options&&options.raw)o.raw={header:header,sectors:sectors};return o;}// parse
	/* [MS-CFB] 2.2 Compound File Header -- read up to major version */function check_get_mver(blob/*:CFBlob*/)/*:[number, number]*/{if(blob[blob.l]==0x50&&blob[blob.l+1]==0x4b)return [0,0];// header signature 8
	blob.chk(HEADER_SIGNATURE,'Header Signature: ');// clsid 16
	//blob.chk(HEADER_CLSID, 'CLSID: ');
	blob.l+=16;// minor version 2
	var mver/*:number*/=blob.read_shift(2,'u');return [blob.read_shift(2,'u'),mver];}function check_shifts(blob/*:CFBlob*/,mver/*:number*/)/*:void*/{var shift=0x09;// Byte Order
	//blob.chk('feff', 'Byte Order: '); // note: some writers put 0xffff
	blob.l+=2;// Sector Shift
	switch(shift=blob.read_shift(2)){case 0x09:if(mver!=3)throw new Error('Sector Shift: Expected 9 saw '+shift);break;case 0x0c:if(mver!=4)throw new Error('Sector Shift: Expected 12 saw '+shift);break;default:throw new Error('Sector Shift: Expected 9 or 12 saw '+shift);}// Mini Sector Shift
	blob.chk('0600','Mini Sector Shift: ');// Reserved
	blob.chk('000000000000','Reserved: ');}/** Break the file up into sectors */function sectorify(file/*:RawBytes*/,ssz/*:number*/)/*:Array<RawBytes>*/{var nsectors=Math.ceil(file.length/ssz)-1;var sectors/*:Array<RawBytes>*/=[];for(var i=1;i<nsectors;++i)sectors[i-1]=file.slice(i*ssz,(i+1)*ssz);sectors[nsectors-1]=file.slice(nsectors*ssz);return sectors;}/* [MS-CFB] 2.6.4 Red-Black Tree */function build_full_paths(FI/*:CFBFileIndex*/,FP/*:Array<string>*/,Paths/*:Array<string>*/)/*:void*/{var i=0,L=0,R=0,C=0,j=0,pl=Paths.length;var dad/*:Array<number>*/=[],q/*:Array<number>*/=[];for(;i<pl;++i){dad[i]=q[i]=i;FP[i]=Paths[i];}for(;j<q.length;++j){i=q[j];L=FI[i].L;R=FI[i].R;C=FI[i].C;if(dad[i]===i){if(L!==-1/*NOSTREAM*/&&dad[L]!==L)dad[i]=dad[L];if(R!==-1&&dad[R]!==R)dad[i]=dad[R];}if(C!==-1/*NOSTREAM*/)dad[C]=i;if(L!==-1&&i!=dad[i]){dad[L]=dad[i];if(q.lastIndexOf(L)<j)q.push(L);}if(R!==-1&&i!=dad[i]){dad[R]=dad[i];if(q.lastIndexOf(R)<j)q.push(R);}}for(i=1;i<pl;++i)if(dad[i]===i){if(R!==-1/*NOSTREAM*/&&dad[R]!==R)dad[i]=dad[R];else if(L!==-1&&dad[L]!==L)dad[i]=dad[L];}for(i=1;i<pl;++i){if(FI[i].type===0/* unknown */)continue;j=i;if(j!=dad[j])do{j=dad[j];FP[i]=FP[j]+"/"+FP[i];}while(j!==0&&-1!==dad[j]&&j!=dad[j]);dad[i]=-1;}FP[0]+="/";for(i=1;i<pl;++i){if(FI[i].type!==2/* stream */)FP[i]+="/";}}function get_mfat_entry(entry/*:CFBEntry*/,payload/*:RawBytes*/,mini/*:?RawBytes*/)/*:CFBlob*/{var start=entry.start,size=entry.size;//return (payload.slice(start*MSSZ, start*MSSZ + size)/*:any*/);
	var o=[];var idx=start;while(mini&&size>0&&idx>=0){o.push(payload.slice(idx*MSSZ,idx*MSSZ+MSSZ));size-=MSSZ;idx=__readInt32LE(mini,idx*4);}if(o.length===0)return new_buf(0)/*:any*/;return bconcat(o).slice(0,entry.size)/*:any*/;}/** Chase down the rest of the DIFAT chain to build a comprehensive list
	    DIFAT chains by storing the next sector number as the last 32 bits */function sleuth_fat(idx/*:number*/,cnt/*:number*/,sectors/*:Array<RawBytes>*/,ssz/*:number*/,fat_addrs)/*:void*/{var q/*:number*/=ENDOFCHAIN;if(idx===ENDOFCHAIN){if(cnt!==0)throw new Error("DIFAT chain shorter than expected");}else if(idx!==-1/*FREESECT*/){var sector=sectors[idx],m=(ssz>>>2)-1;if(!sector)return;for(var i=0;i<m;++i){if((q=__readInt32LE(sector,i*4))===ENDOFCHAIN)break;fat_addrs.push(q);}sleuth_fat(__readInt32LE(sector,ssz-4),cnt-1,sectors,ssz,fat_addrs);}}/** Follow the linked list of sectors for a given starting point */function get_sector_list(sectors/*:Array<RawBytes>*/,start/*:number*/,fat_addrs/*:Array<number>*/,ssz/*:number*/,chkd/*:?Array<boolean>*/)/*:SectorEntry*/{var buf/*:Array<number>*/=[],buf_chain/*:Array<any>*/=[];if(!chkd)chkd=[];var modulus=ssz-1,j=0,jj=0;for(j=start;j>=0;){chkd[j]=true;buf[buf.length]=j;buf_chain.push(sectors[j]);var addr=fat_addrs[Math.floor(j*4/ssz)];jj=j*4&modulus;if(ssz<4+jj)throw new Error("FAT boundary crossed: "+j+" 4 "+ssz);if(!sectors[addr])break;j=__readInt32LE(sectors[addr],jj);}return {nodes:buf,data:__toBuffer([buf_chain])};}/** Chase down the sector linked lists */function make_sector_list(sectors/*:Array<RawBytes>*/,dir_start/*:number*/,fat_addrs/*:Array<number>*/,ssz/*:number*/)/*:SectorList*/{var sl=sectors.length,sector_list/*:SectorList*/=[]/*:any*/;var chkd/*:Array<boolean>*/=[],buf/*:Array<number>*/=[],buf_chain/*:Array<RawBytes>*/=[];var modulus=ssz-1,i=0,j=0,k=0,jj=0;for(i=0;i<sl;++i){buf=[]/*:Array<number>*/;k=i+dir_start;if(k>=sl)k-=sl;if(chkd[k])continue;buf_chain=[];var seen=[];for(j=k;j>=0;){seen[j]=true;chkd[j]=true;buf[buf.length]=j;buf_chain.push(sectors[j]);var addr/*:number*/=fat_addrs[Math.floor(j*4/ssz)];jj=j*4&modulus;if(ssz<4+jj)throw new Error("FAT boundary crossed: "+j+" 4 "+ssz);if(!sectors[addr])break;j=__readInt32LE(sectors[addr],jj);if(seen[j])break;}sector_list[k]={nodes:buf,data:__toBuffer([buf_chain])}/*:SectorEntry*/;}return sector_list;}/* [MS-CFB] 2.6.1 Compound File Directory Entry */function read_directory(dir_start/*:number*/,sector_list/*:SectorList*/,sectors/*:Array<RawBytes>*/,Paths/*:Array<string>*/,nmfs,files,FileIndex,mini){var minifat_store=0,pl=Paths.length?2:0;var sector=sector_list[dir_start].data;var i=0,namelen=0,name;for(;i<sector.length;i+=128){var blob/*:CFBlob*/=/*::(*/sector.slice(i,i+128)/*:: :any)*/;prep_blob(blob,64);namelen=blob.read_shift(2);name=__utf16le(blob,0,namelen-pl);Paths.push(name);var o/*:CFBEntry*/={name:name,type:blob.read_shift(1),color:blob.read_shift(1),L:blob.read_shift(4,'i'),R:blob.read_shift(4,'i'),C:blob.read_shift(4,'i'),clsid:blob.read_shift(16),state:blob.read_shift(4,'i'),start:0,size:0};var ctime/*:number*/=blob.read_shift(2)+blob.read_shift(2)+blob.read_shift(2)+blob.read_shift(2);if(ctime!==0)o.ct=read_date(blob,blob.l-8);var mtime/*:number*/=blob.read_shift(2)+blob.read_shift(2)+blob.read_shift(2)+blob.read_shift(2);if(mtime!==0)o.mt=read_date(blob,blob.l-8);o.start=blob.read_shift(4,'i');o.size=blob.read_shift(4,'i');if(o.size<0&&o.start<0){o.size=o.type=0;o.start=ENDOFCHAIN;o.name="";}if(o.type===5){/* root */minifat_store=o.start;if(nmfs>0&&minifat_store!==ENDOFCHAIN)sector_list[minifat_store].name="!StreamData";/*minifat_size = o.size;*/}else if(o.size>=4096/* MSCSZ */){o.storage='fat';if(sector_list[o.start]===undefined)sector_list[o.start]=get_sector_list(sectors,o.start,sector_list.fat_addrs,sector_list.ssz);sector_list[o.start].name=o.name;o.content=sector_list[o.start].data.slice(0,o.size)/*:any*/;}else {o.storage='minifat';if(o.size<0)o.size=0;else if(minifat_store!==ENDOFCHAIN&&o.start!==ENDOFCHAIN&&sector_list[minifat_store]){o.content=get_mfat_entry(o,sector_list[minifat_store].data,(sector_list[mini]||{}).data);}}if(o.content)prep_blob(o.content,0);files[name]=o;FileIndex.push(o);}}function read_date(blob/*:RawBytes|CFBlob*/,offset/*:number*/)/*:Date*/{return new Date((__readUInt32LE(blob,offset+4)/1e7*Math.pow(2,32)+__readUInt32LE(blob,offset)/1e7-11644473600)*1000);}function read_file(filename/*:string*/,options/*:CFBReadOpts*/){get_fs();return parse(fs.readFileSync(filename),options);}function read(blob/*:RawBytes|string*/,options/*:CFBReadOpts*/){var type=options&&options.type;if(!type){if(has_buf&&Buffer.isBuffer(blob))type="buffer";}switch(type||"base64"){case"file":/*:: if(typeof blob !== 'string') throw "Must pass a filename when type='file'"; */return read_file(blob,options);case"base64":/*:: if(typeof blob !== 'string') throw "Must pass a base64-encoded binary string when type='file'"; */return parse(s2a(Base64_decode(blob)),options);case"binary":/*:: if(typeof blob !== 'string') throw "Must pass a binary string when type='file'"; */return parse(s2a(blob),options);}return parse(/*::typeof blob == 'string' ? new Buffer(blob, 'utf-8') : */blob,options);}function init_cfb(cfb/*:CFBContainer*/,opts/*:?any*/)/*:void*/{var o=opts||{},root=o.root||"Root Entry";if(!cfb.FullPaths)cfb.FullPaths=[];if(!cfb.FileIndex)cfb.FileIndex=[];if(cfb.FullPaths.length!==cfb.FileIndex.length)throw new Error("inconsistent CFB structure");if(cfb.FullPaths.length===0){cfb.FullPaths[0]=root+"/";cfb.FileIndex[0]={name:root,type:5}/*:any*/;}if(o.CLSID)cfb.FileIndex[0].clsid=o.CLSID;seed_cfb(cfb);}function seed_cfb(cfb/*:CFBContainer*/)/*:void*/{var nm="\u0001Sh33tJ5";if(CFB.find(cfb,"/"+nm))return;var p=new_buf(4);p[0]=55;p[1]=p[3]=50;p[2]=54;cfb.FileIndex.push({name:nm,type:2,content:p,size:4,L:69,R:69,C:69}/*:any*/);cfb.FullPaths.push(cfb.FullPaths[0]+nm);rebuild_cfb(cfb);}function rebuild_cfb(cfb/*:CFBContainer*/,f/*:?boolean*/)/*:void*/{init_cfb(cfb);var gc=false,s=false;for(var i=cfb.FullPaths.length-1;i>=0;--i){var _file=cfb.FileIndex[i];switch(_file.type){case 0:if(s)gc=true;else {cfb.FileIndex.pop();cfb.FullPaths.pop();}break;case 1:case 2:case 5:s=true;if(isNaN(_file.R*_file.L*_file.C))gc=true;if(_file.R>-1&&_file.L>-1&&_file.R==_file.L)gc=true;break;default:gc=true;break;}}if(!gc&&!f)return;var now=new Date(1987,1,19),j=0;// Track which names exist
	var fullPaths=Object.create?Object.create(null):{};var data/*:Array<[string, CFBEntry]>*/=[];for(i=0;i<cfb.FullPaths.length;++i){fullPaths[cfb.FullPaths[i]]=true;if(cfb.FileIndex[i].type===0)continue;data.push([cfb.FullPaths[i],cfb.FileIndex[i]]);}for(i=0;i<data.length;++i){var dad=dirname(data[i][0]);s=fullPaths[dad];if(!s){data.push([dad,{name:filename(dad).replace("/",""),type:1,clsid:HEADER_CLSID,ct:now,mt:now,content:null}/*:any*/]);// Add name to set
	fullPaths[dad]=true;}}data.sort(function(x,y){return namecmp(x[0],y[0]);});cfb.FullPaths=[];cfb.FileIndex=[];for(i=0;i<data.length;++i){cfb.FullPaths[i]=data[i][0];cfb.FileIndex[i]=data[i][1];}for(i=0;i<data.length;++i){var elt=cfb.FileIndex[i];var nm=cfb.FullPaths[i];elt.name=filename(nm).replace("/","");elt.L=elt.R=elt.C=-(elt.color=1);elt.size=elt.content?elt.content.length:0;elt.start=0;elt.clsid=elt.clsid||HEADER_CLSID;if(i===0){elt.C=data.length>1?1:-1;elt.size=0;elt.type=5;}else if(nm.slice(-1)=="/"){for(j=i+1;j<data.length;++j)if(dirname(cfb.FullPaths[j])==nm)break;elt.C=j>=data.length?-1:j;for(j=i+1;j<data.length;++j)if(dirname(cfb.FullPaths[j])==dirname(nm))break;elt.R=j>=data.length?-1:j;elt.type=1;}else {if(dirname(cfb.FullPaths[i+1]||"")==dirname(nm))elt.R=i+1;elt.type=2;}}}function _write(cfb/*:CFBContainer*/,options/*:CFBWriteOpts*/)/*:RawBytes|string*/{var _opts=options||{};/* MAD is order-sensitive, skip rebuild and sort */if(_opts.fileType=='mad')return write_mad(cfb,_opts);rebuild_cfb(cfb);switch(_opts.fileType){case'zip':return write_zip(cfb,_opts);//case 'mad': return write_mad(cfb, _opts);
	}var L=function(cfb/*:CFBContainer*/)/*:Array<number>*/{var mini_size=0,fat_size=0;for(var i=0;i<cfb.FileIndex.length;++i){var file=cfb.FileIndex[i];if(!file.content)continue;/*:: if(file.content == null) throw new Error("unreachable"); */var flen=file.content.length;if(flen>0){if(flen<0x1000)mini_size+=flen+0x3F>>6;else fat_size+=flen+0x01FF>>9;}}var dir_cnt=cfb.FullPaths.length+3>>2;var mini_cnt=mini_size+7>>3;var mfat_cnt=mini_size+0x7F>>7;var fat_base=mini_cnt+fat_size+dir_cnt+mfat_cnt;var fat_cnt=fat_base+0x7F>>7;var difat_cnt=fat_cnt<=109?0:Math.ceil((fat_cnt-109)/0x7F);while(fat_base+fat_cnt+difat_cnt+0x7F>>7>fat_cnt)difat_cnt=++fat_cnt<=109?0:Math.ceil((fat_cnt-109)/0x7F);var L=[1,difat_cnt,fat_cnt,mfat_cnt,dir_cnt,fat_size,mini_size,0];cfb.FileIndex[0].size=mini_size<<6;L[7]=(cfb.FileIndex[0].start=L[0]+L[1]+L[2]+L[3]+L[4]+L[5])+(L[6]+7>>3);return L;}(cfb);var o=new_buf(L[7]<<9);var i=0,T=0;{for(i=0;i<8;++i)o.write_shift(1,HEADER_SIG[i]);for(i=0;i<8;++i)o.write_shift(2,0);o.write_shift(2,0x003E);o.write_shift(2,0x0003);o.write_shift(2,0xFFFE);o.write_shift(2,0x0009);o.write_shift(2,0x0006);for(i=0;i<3;++i)o.write_shift(2,0);o.write_shift(4,0);o.write_shift(4,L[2]);o.write_shift(4,L[0]+L[1]+L[2]+L[3]-1);o.write_shift(4,0);o.write_shift(4,1<<12);o.write_shift(4,L[3]?L[0]+L[1]+L[2]-1:ENDOFCHAIN);o.write_shift(4,L[3]);o.write_shift(-4,L[1]?L[0]-1:ENDOFCHAIN);o.write_shift(4,L[1]);for(i=0;i<109;++i)o.write_shift(-4,i<L[2]?L[1]+i:-1);}if(L[1]){for(T=0;T<L[1];++T){for(;i<236+T*127;++i)o.write_shift(-4,i<L[2]?L[1]+i:-1);o.write_shift(-4,T===L[1]-1?ENDOFCHAIN:T+1);}}var chainit=function(w/*:number*/)/*:void*/{for(T+=w;i<T-1;++i)o.write_shift(-4,i+1);if(w){++i;o.write_shift(-4,ENDOFCHAIN);}};T=i=0;for(T+=L[1];i<T;++i)o.write_shift(-4,consts.DIFSECT);for(T+=L[2];i<T;++i)o.write_shift(-4,consts.FATSECT);chainit(L[3]);chainit(L[4]);var j/*:number*/=0,flen/*:number*/=0;var file/*:CFBEntry*/=cfb.FileIndex[0];for(;j<cfb.FileIndex.length;++j){file=cfb.FileIndex[j];if(!file.content)continue;/*:: if(file.content == null) throw new Error("unreachable"); */flen=file.content.length;if(flen<0x1000)continue;file.start=T;chainit(flen+0x01FF>>9);}chainit(L[6]+7>>3);while(o.l&0x1FF)o.write_shift(-4,consts.ENDOFCHAIN);T=i=0;for(j=0;j<cfb.FileIndex.length;++j){file=cfb.FileIndex[j];if(!file.content)continue;/*:: if(file.content == null) throw new Error("unreachable"); */flen=file.content.length;if(!flen||flen>=0x1000)continue;file.start=T;chainit(flen+0x3F>>6);}while(o.l&0x1FF)o.write_shift(-4,consts.ENDOFCHAIN);for(i=0;i<L[4]<<2;++i){var nm=cfb.FullPaths[i];if(!nm||nm.length===0){for(j=0;j<17;++j)o.write_shift(4,0);for(j=0;j<3;++j)o.write_shift(4,-1);for(j=0;j<12;++j)o.write_shift(4,0);continue;}file=cfb.FileIndex[i];if(i===0)file.start=file.size?file.start-1:ENDOFCHAIN;var _nm/*:string*/=i===0&&_opts.root||file.name;flen=2*(_nm.length+1);o.write_shift(64,_nm,"utf16le");o.write_shift(2,flen);o.write_shift(1,file.type);o.write_shift(1,file.color);o.write_shift(-4,file.L);o.write_shift(-4,file.R);o.write_shift(-4,file.C);if(!file.clsid)for(j=0;j<4;++j)o.write_shift(4,0);else o.write_shift(16,file.clsid,"hex");o.write_shift(4,file.state||0);o.write_shift(4,0);o.write_shift(4,0);o.write_shift(4,0);o.write_shift(4,0);o.write_shift(4,file.start);o.write_shift(4,file.size);o.write_shift(4,0);}for(i=1;i<cfb.FileIndex.length;++i){file=cfb.FileIndex[i];/*:: if(!file.content) throw new Error("unreachable"); */if(file.size>=0x1000){o.l=file.start+1<<9;if(has_buf&&Buffer.isBuffer(file.content)){file.content.copy(o,o.l,0,file.size);// o is a 0-filled Buffer so just set next offset
	o.l+=file.size+511&-512;}else {for(j=0;j<file.size;++j)o.write_shift(1,file.content[j]);for(;j&0x1FF;++j)o.write_shift(1,0);}}}for(i=1;i<cfb.FileIndex.length;++i){file=cfb.FileIndex[i];/*:: if(!file.content) throw new Error("unreachable"); */if(file.size>0&&file.size<0x1000){if(has_buf&&Buffer.isBuffer(file.content)){file.content.copy(o,o.l,0,file.size);// o is a 0-filled Buffer so just set next offset
	o.l+=file.size+63&-64;}else {for(j=0;j<file.size;++j)o.write_shift(1,file.content[j]);for(;j&0x3F;++j)o.write_shift(1,0);}}}if(has_buf){o.l=o.length;}else {// When using Buffer, already 0-filled
	while(o.l<o.length)o.write_shift(1,0);}return o;}/* [MS-CFB] 2.6.4 (Unicode 3.0.1 case conversion) */function find(cfb/*:CFBContainer*/,path/*:string*/)/*:?CFBEntry*/{var UCFullPaths/*:Array<string>*/=cfb.FullPaths.map(function(x){return x.toUpperCase();});var UCPaths/*:Array<string>*/=UCFullPaths.map(function(x){var y=x.split("/");return y[y.length-(x.slice(-1)=="/"?2:1)];});var k/*:boolean*/=false;if(path.charCodeAt(0)===47/* "/" */){k=true;path=UCFullPaths[0].slice(0,-1)+path;}else k=path.indexOf("/")!==-1;var UCPath/*:string*/=path.toUpperCase();var w/*:number*/=k===true?UCFullPaths.indexOf(UCPath):UCPaths.indexOf(UCPath);if(w!==-1)return cfb.FileIndex[w];var m=!UCPath.match(chr1);UCPath=UCPath.replace(chr0,'');if(m)UCPath=UCPath.replace(chr1,'!');for(w=0;w<UCFullPaths.length;++w){if((m?UCFullPaths[w].replace(chr1,'!'):UCFullPaths[w]).replace(chr0,'')==UCPath)return cfb.FileIndex[w];if((m?UCPaths[w].replace(chr1,'!'):UCPaths[w]).replace(chr0,'')==UCPath)return cfb.FileIndex[w];}return null;}/** CFB Constants */var MSSZ=64;/* Mini Sector Size = 1<<6 */ //var MSCSZ = 4096; /* Mini Stream Cutoff Size */
	/* 2.1 Compound File Sector Numbers and Types */var ENDOFCHAIN=-2;/* 2.2 Compound File Header */var HEADER_SIGNATURE='d0cf11e0a1b11ae1';var HEADER_SIG=[0xD0,0xCF,0x11,0xE0,0xA1,0xB1,0x1A,0xE1];var HEADER_CLSID='00000000000000000000000000000000';var consts={/* 2.1 Compund File Sector Numbers and Types */MAXREGSECT:-6,DIFSECT:-4,FATSECT:-3,ENDOFCHAIN:ENDOFCHAIN,FREESECT:-1,/* 2.2 Compound File Header */HEADER_SIGNATURE:HEADER_SIGNATURE,HEADER_MINOR_VERSION:'3e00',MAXREGSID:-6,NOSTREAM:-1,HEADER_CLSID:HEADER_CLSID,/* 2.6.1 Compound File Directory Entry */EntryTypes:['unknown','storage','stream','lockbytes','property','root']};function write_file(cfb/*:CFBContainer*/,filename/*:string*/,options/*:CFBWriteOpts*/)/*:void*/{get_fs();var o=_write(cfb,options);/*:: if(typeof Buffer == 'undefined' || !Buffer.isBuffer(o) || !(o instanceof Buffer)) throw new Error("unreachable"); */fs.writeFileSync(filename,o);}function a2s(o/*:RawBytes*/)/*:string*/{var out=new Array(o.length);for(var i=0;i<o.length;++i)out[i]=String.fromCharCode(o[i]);return out.join("");}function write(cfb/*:CFBContainer*/,options/*:CFBWriteOpts*/)/*:RawBytes|string*/{var o=_write(cfb,options);switch(options&&options.type||"buffer"){case"file":get_fs();fs.writeFileSync(options.filename,o/*:any*/);return o;case"binary":return typeof o=="string"?o:a2s(o);case"base64":return Base64_encode(typeof o=="string"?o:a2s(o));case"buffer":if(has_buf)return Buffer.isBuffer(o)?o:Buffer_from(o);/* falls through */case"array":return typeof o=="string"?s2a(o):o;}return o;}/* node < 8.1 zlib does not expose bytesRead, so default to pure JS */var _zlib;function use_zlib(zlib){try{var InflateRaw=zlib.InflateRaw;var InflRaw=new InflateRaw();InflRaw._processChunk(new Uint8Array([3,0]),InflRaw._finishFlushFlag);if(InflRaw.bytesRead)_zlib=zlib;else throw new Error("zlib does not expose bytesRead");}catch(e){console.error("cannot use native zlib: "+(e.message||e));}}function _inflateRawSync(payload,usz){if(!_zlib)return _inflate(payload,usz);var InflateRaw=_zlib.InflateRaw;var InflRaw=new InflateRaw();var out=InflRaw._processChunk(payload.slice(payload.l),InflRaw._finishFlushFlag);payload.l+=InflRaw.bytesRead;return out;}function _deflateRawSync(payload){return _zlib?_zlib.deflateRawSync(payload):_deflate(payload);}var CLEN_ORDER=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];/*  LEN_ID = [ 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285 ]; */var LEN_LN=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258];/*  DST_ID = [  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13,  14,  15,  16,  17,  18,  19,   20,   21,   22,   23,   24,   25,   26,    27,    28,    29 ]; */var DST_LN=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577];function bit_swap_8(n){var t=(n<<1|n<<11)&0x22110|(n<<5|n<<15)&0x88440;return (t>>16|t>>8|t)&0xFF;}var use_typed_arrays=typeof Uint8Array!=='undefined';var bitswap8=use_typed_arrays?new Uint8Array(1<<8):[];for(var q=0;q<1<<8;++q)bitswap8[q]=bit_swap_8(q);function bit_swap_n(n,b){var rev=bitswap8[n&0xFF];if(b<=8)return rev>>>8-b;rev=rev<<8|bitswap8[n>>8&0xFF];if(b<=16)return rev>>>16-b;rev=rev<<8|bitswap8[n>>16&0xFF];return rev>>>24-b;}/* helpers for unaligned bit reads */function read_bits_2(buf,bl){var w=bl&7,h=bl>>>3;return (buf[h]|(w<=6?0:buf[h+1]<<8))>>>w&0x03;}function read_bits_3(buf,bl){var w=bl&7,h=bl>>>3;return (buf[h]|(w<=5?0:buf[h+1]<<8))>>>w&0x07;}function read_bits_4(buf,bl){var w=bl&7,h=bl>>>3;return (buf[h]|(w<=4?0:buf[h+1]<<8))>>>w&0x0F;}function read_bits_5(buf,bl){var w=bl&7,h=bl>>>3;return (buf[h]|(w<=3?0:buf[h+1]<<8))>>>w&0x1F;}function read_bits_7(buf,bl){var w=bl&7,h=bl>>>3;return (buf[h]|(w<=1?0:buf[h+1]<<8))>>>w&0x7F;}/* works up to n = 3 * 8 + 1 = 25 */function read_bits_n(buf,bl,n){var w=bl&7,h=bl>>>3,f=(1<<n)-1;var v=buf[h]>>>w;if(n<8-w)return v&f;v|=buf[h+1]<<8-w;if(n<16-w)return v&f;v|=buf[h+2]<<16-w;if(n<24-w)return v&f;v|=buf[h+3]<<24-w;return v&f;}/* helpers for unaligned bit writes */function write_bits_3(buf,bl,v){var w=bl&7,h=bl>>>3;if(w<=5)buf[h]|=(v&7)<<w;else {buf[h]|=v<<w&0xFF;buf[h+1]=(v&7)>>8-w;}return bl+3;}function write_bits_1(buf,bl,v){var w=bl&7,h=bl>>>3;v=(v&1)<<w;buf[h]|=v;return bl+1;}function write_bits_8(buf,bl,v){var w=bl&7,h=bl>>>3;v<<=w;buf[h]|=v&0xFF;v>>>=8;buf[h+1]=v;return bl+8;}function write_bits_16(buf,bl,v){var w=bl&7,h=bl>>>3;v<<=w;buf[h]|=v&0xFF;v>>>=8;buf[h+1]=v&0xFF;buf[h+2]=v>>>8;return bl+16;}/* until ArrayBuffer#realloc is a thing, fake a realloc */function realloc(b,sz/*:number*/){var L=b.length,M=2*L>sz?2*L:sz+5,i=0;if(L>=sz)return b;if(has_buf){var o=new_unsafe_buf(M);// $FlowIgnore
	if(b.copy)b.copy(o);else for(;i<b.length;++i)o[i]=b[i];return o;}else if(use_typed_arrays){var a=new Uint8Array(M);if(a.set)a.set(b);else for(;i<L;++i)a[i]=b[i];return a;}b.length=M;return b;}/* zero-filled arrays for older browsers */function zero_fill_array(n){var o=new Array(n);for(var i=0;i<n;++i)o[i]=0;return o;}/* build tree (used for literals and lengths) */function build_tree(clens,cmap,MAX/*:number*/)/*:number*/{var maxlen=1,w=0,i=0,j=0,ccode=0,L=clens.length;var bl_count=use_typed_arrays?new Uint16Array(32):zero_fill_array(32);for(i=0;i<32;++i)bl_count[i]=0;for(i=L;i<MAX;++i)clens[i]=0;L=clens.length;var ctree=use_typed_arrays?new Uint16Array(L):zero_fill_array(L);// []
	/* build code tree */for(i=0;i<L;++i){bl_count[w=clens[i]]++;if(maxlen<w)maxlen=w;ctree[i]=0;}bl_count[0]=0;for(i=1;i<=maxlen;++i)bl_count[i+16]=ccode=ccode+bl_count[i-1]<<1;for(i=0;i<L;++i){ccode=clens[i];if(ccode!=0)ctree[i]=bl_count[ccode+16]++;}/* cmap[maxlen + 4 bits] = (off&15) + (lit<<4) reverse mapping */var cleni=0;for(i=0;i<L;++i){cleni=clens[i];if(cleni!=0){ccode=bit_swap_n(ctree[i],maxlen)>>maxlen-cleni;for(j=(1<<maxlen+4-cleni)-1;j>=0;--j)cmap[ccode|j<<cleni]=cleni&15|i<<4;}}return maxlen;}/* Fixed Huffman */var fix_lmap=use_typed_arrays?new Uint16Array(512):zero_fill_array(512);var fix_dmap=use_typed_arrays?new Uint16Array(32):zero_fill_array(32);if(!use_typed_arrays){for(var i=0;i<512;++i)fix_lmap[i]=0;for(i=0;i<32;++i)fix_dmap[i]=0;}(function(){var dlens/*:Array<number>*/=[];var i=0;for(;i<32;i++)dlens.push(5);build_tree(dlens,fix_dmap,32);var clens/*:Array<number>*/=[];i=0;for(;i<=143;i++)clens.push(8);for(;i<=255;i++)clens.push(9);for(;i<=279;i++)clens.push(7);for(;i<=287;i++)clens.push(8);build_tree(clens,fix_lmap,288);})();var _deflateRaw=/*#__PURE__*/function _deflateRawIIFE(){var DST_LN_RE=use_typed_arrays?new Uint8Array(0x8000):[];var j=0,k=0;for(;j<DST_LN.length-1;++j){for(;k<DST_LN[j+1];++k)DST_LN_RE[k]=j;}for(;k<32768;++k)DST_LN_RE[k]=29;var LEN_LN_RE=use_typed_arrays?new Uint8Array(0x103):[];for(j=0,k=0;j<LEN_LN.length-1;++j){for(;k<LEN_LN[j+1];++k)LEN_LN_RE[k]=j;}function write_stored(data,out){var boff=0;while(boff<data.length){var L=Math.min(0xFFFF,data.length-boff);var h=boff+L==data.length;out.write_shift(1,+h);out.write_shift(2,L);out.write_shift(2,~L&0xFFFF);while(L-->0)out[out.l++]=data[boff++];}return out.l;}/* Fixed Huffman */function write_huff_fixed(data,out){var bl=0;var boff=0;var addrs=use_typed_arrays?new Uint16Array(0x8000):[];while(boff<data.length){var L=/* data.length - boff; */Math.min(0xFFFF,data.length-boff);/* write a stored block for short data */if(L<10){bl=write_bits_3(out,bl,+!!(boff+L==data.length));// jshint ignore:line
	if(bl&7)bl+=8-(bl&7);out.l=bl/8|0;out.write_shift(2,L);out.write_shift(2,~L&0xFFFF);while(L-->0)out[out.l++]=data[boff++];bl=out.l*8;continue;}bl=write_bits_3(out,bl,+!!(boff+L==data.length)+2);// jshint ignore:line
	var hash=0;while(L-->0){var d=data[boff];hash=(hash<<5^d)&0x7FFF;var match=-1,mlen=0;if(match=addrs[hash]){match|=boff&~0x7FFF;if(match>boff)match-=0x8000;if(match<boff)while(data[match+mlen]==data[boff+mlen]&&mlen<250)++mlen;}if(mlen>2){/* Copy Token  */d=LEN_LN_RE[mlen];if(d<=22)bl=write_bits_8(out,bl,bitswap8[d+1]>>1)-1;else {write_bits_8(out,bl,3);bl+=5;write_bits_8(out,bl,bitswap8[d-23]>>5);bl+=3;}var len_eb=d<8?0:d-4>>2;if(len_eb>0){write_bits_16(out,bl,mlen-LEN_LN[d]);bl+=len_eb;}d=DST_LN_RE[boff-match];bl=write_bits_8(out,bl,bitswap8[d]>>3);bl-=3;var dst_eb=d<4?0:d-2>>1;if(dst_eb>0){write_bits_16(out,bl,boff-match-DST_LN[d]);bl+=dst_eb;}for(var q=0;q<mlen;++q){addrs[hash]=boff&0x7FFF;hash=(hash<<5^data[boff])&0x7FFF;++boff;}L-=mlen-1;}else {/* Literal Token */if(d<=143)d=d+48;else bl=write_bits_1(out,bl,1);bl=write_bits_8(out,bl,bitswap8[d]);addrs[hash]=boff&0x7FFF;++boff;}}bl=write_bits_8(out,bl,0)-1;}out.l=(bl+7)/8|0;return out.l;}return function _deflateRaw(data,out){if(data.length<8)return write_stored(data,out);return write_huff_fixed(data,out);};}();function _deflate(data){var buf=new_buf(50+Math.floor(data.length*1.1));var off=_deflateRaw(data,buf);return buf.slice(0,off);}/* modified inflate function also moves original read head */var dyn_lmap=use_typed_arrays?new Uint16Array(32768):zero_fill_array(32768);var dyn_dmap=use_typed_arrays?new Uint16Array(32768):zero_fill_array(32768);var dyn_cmap=use_typed_arrays?new Uint16Array(128):zero_fill_array(128);var dyn_len_1=1,dyn_len_2=1;/* 5.5.3 Expanding Huffman Codes */function dyn(data,boff/*:number*/){/* nomenclature from RFC1951 refers to bit values; these are offset by the implicit constant */var _HLIT=read_bits_5(data,boff)+257;boff+=5;var _HDIST=read_bits_5(data,boff)+1;boff+=5;var _HCLEN=read_bits_4(data,boff)+4;boff+=4;var w=0;/* grab and store code lengths */var clens=use_typed_arrays?new Uint8Array(19):zero_fill_array(19);var ctree=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];var maxlen=1;var bl_count=use_typed_arrays?new Uint8Array(8):zero_fill_array(8);var next_code=use_typed_arrays?new Uint8Array(8):zero_fill_array(8);var L=clens.length;/* 19 */for(var i=0;i<_HCLEN;++i){clens[CLEN_ORDER[i]]=w=read_bits_3(data,boff);if(maxlen<w)maxlen=w;bl_count[w]++;boff+=3;}/* build code tree */var ccode=0;bl_count[0]=0;for(i=1;i<=maxlen;++i)next_code[i]=ccode=ccode+bl_count[i-1]<<1;for(i=0;i<L;++i)if((ccode=clens[i])!=0)ctree[i]=next_code[ccode]++;/* cmap[7 bits from stream] = (off&7) + (lit<<3) */var cleni=0;for(i=0;i<L;++i){cleni=clens[i];if(cleni!=0){ccode=bitswap8[ctree[i]]>>8-cleni;for(var j=(1<<7-cleni)-1;j>=0;--j)dyn_cmap[ccode|j<<cleni]=cleni&7|i<<3;}}/* read literal and dist codes at once */var hcodes/*:Array<number>*/=[];maxlen=1;for(;hcodes.length<_HLIT+_HDIST;){ccode=dyn_cmap[read_bits_7(data,boff)];boff+=ccode&7;switch(ccode>>>=3){case 16:w=3+read_bits_2(data,boff);boff+=2;ccode=hcodes[hcodes.length-1];while(w-->0)hcodes.push(ccode);break;case 17:w=3+read_bits_3(data,boff);boff+=3;while(w-->0)hcodes.push(0);break;case 18:w=11+read_bits_7(data,boff);boff+=7;while(w-->0)hcodes.push(0);break;default:hcodes.push(ccode);if(maxlen<ccode)maxlen=ccode;break;}}/* build literal / length trees */var h1=hcodes.slice(0,_HLIT),h2=hcodes.slice(_HLIT);for(i=_HLIT;i<286;++i)h1[i]=0;for(i=_HDIST;i<30;++i)h2[i]=0;dyn_len_1=build_tree(h1,dyn_lmap,286);dyn_len_2=build_tree(h2,dyn_dmap,30);return boff;}/* return [ data, bytesRead ] */function inflate(data,usz/*:number*/){/* shortcircuit for empty buffer [0x03, 0x00] */if(data[0]==3&&!(data[1]&0x3)){return [new_raw_buf(usz),2];}/* bit offset */var boff=0;/* header includes final bit and type bits */var header=0;var outbuf=new_unsafe_buf(usz?usz:1<<18);var woff=0;var OL=outbuf.length>>>0;var max_len_1=0,max_len_2=0;while((header&1)==0){header=read_bits_3(data,boff);boff+=3;if(header>>>1==0){/* Stored block */if(boff&7)boff+=8-(boff&7);/* 2 bytes sz, 2 bytes bit inverse */var sz=data[boff>>>3]|data[(boff>>>3)+1]<<8;boff+=32;/* push sz bytes */if(sz>0){if(!usz&&OL<woff+sz){outbuf=realloc(outbuf,woff+sz);OL=outbuf.length;}while(sz-->0){outbuf[woff++]=data[boff>>>3];boff+=8;}}continue;}else if(header>>1==1){/* Fixed Huffman */max_len_1=9;max_len_2=5;}else {/* Dynamic Huffman */boff=dyn(data,boff);max_len_1=dyn_len_1;max_len_2=dyn_len_2;}for(;;){// while(true) is apparently out of vogue in modern JS circles
	if(!usz&&OL<woff+32767){outbuf=realloc(outbuf,woff+32767);OL=outbuf.length;}/* ingest code and move read head */var bits=read_bits_n(data,boff,max_len_1);var code=header>>>1==1?fix_lmap[bits]:dyn_lmap[bits];boff+=code&15;code>>>=4;/* 0-255 are literals, 256 is end of block token, 257+ are copy tokens */if((code>>>8&0xFF)===0)outbuf[woff++]=code;else if(code==256)break;else {code-=257;var len_eb=code<8?0:code-4>>2;if(len_eb>5)len_eb=0;var tgt=woff+LEN_LN[code];/* length extra bits */if(len_eb>0){tgt+=read_bits_n(data,boff,len_eb);boff+=len_eb;}/* dist code */bits=read_bits_n(data,boff,max_len_2);code=header>>>1==1?fix_dmap[bits]:dyn_dmap[bits];boff+=code&15;code>>>=4;var dst_eb=code<4?0:code-2>>1;var dst=DST_LN[code];/* dist extra bits */if(dst_eb>0){dst+=read_bits_n(data,boff,dst_eb);boff+=dst_eb;}/* in the common case, manual byte copy is faster than TA set / Buffer copy */if(!usz&&OL<tgt){outbuf=realloc(outbuf,tgt+100);OL=outbuf.length;}while(woff<tgt){outbuf[woff]=outbuf[woff-dst];++woff;}}}}if(usz)return [outbuf,boff+7>>>3];return [outbuf.slice(0,woff),boff+7>>>3];}function _inflate(payload,usz){var data=payload.slice(payload.l||0);var out=inflate(data,usz);payload.l+=out[1];return out[0];}function warn_or_throw(wrn,msg){if(wrn){if(typeof console!=='undefined')console.error(msg);}else throw new Error(msg);}function parse_zip(file/*:RawBytes*/,options/*:CFBReadOpts*/)/*:CFBContainer*/{var blob/*:CFBlob*/=/*::(*/file/*:: :any)*/;prep_blob(blob,0);var FileIndex/*:CFBFileIndex*/=[],FullPaths/*:Array<string>*/=[];var o={FileIndex:FileIndex,FullPaths:FullPaths};init_cfb(o,{root:options.root});/* find end of central directory, start just after signature */var i=blob.length-4;while((blob[i]!=0x50||blob[i+1]!=0x4b||blob[i+2]!=0x05||blob[i+3]!=0x06)&&i>=0)--i;blob.l=i+4;/* parse end of central directory */blob.l+=4;var fcnt=blob.read_shift(2);blob.l+=6;var start_cd=blob.read_shift(4);/* parse central directory */blob.l=start_cd;for(i=0;i<fcnt;++i){/* trust local file header instead of CD entry */blob.l+=20;var csz=blob.read_shift(4);var usz=blob.read_shift(4);var namelen=blob.read_shift(2);var efsz=blob.read_shift(2);var fcsz=blob.read_shift(2);blob.l+=8;var offset=blob.read_shift(4);var EF=parse_extra_field(/*::(*/blob.slice(blob.l+namelen,blob.l+namelen+efsz)/*:: :any)*/);blob.l+=namelen+efsz+fcsz;var L=blob.l;blob.l=offset+4;parse_local_file(blob,csz,usz,o,EF);blob.l=L;}return o;}/* head starts just after local file header signature */function parse_local_file(blob/*:CFBlob*/,csz/*:number*/,usz/*:number*/,o/*:CFBContainer*/,EF){/* [local file header] */blob.l+=2;var flags=blob.read_shift(2);var meth=blob.read_shift(2);var date=parse_dos_date(blob);if(flags&0x2041)throw new Error("Unsupported ZIP encryption");var crc32=blob.read_shift(4);var _csz=blob.read_shift(4);var _usz=blob.read_shift(4);var namelen=blob.read_shift(2);var efsz=blob.read_shift(2);// TODO: flags & (1<<11) // UTF8
	var name="";for(var i=0;i<namelen;++i)name+=String.fromCharCode(blob[blob.l++]);if(efsz){var ef=parse_extra_field(/*::(*/blob.slice(blob.l,blob.l+efsz)/*:: :any)*/);if((ef[0x5455]||{}).mt)date=ef[0x5455].mt;if(((EF||{})[0x5455]||{}).mt)date=EF[0x5455].mt;}blob.l+=efsz;/* [encryption header] */ /* [file data] */var data=blob.slice(blob.l,blob.l+_csz);switch(meth){case 8:data=_inflateRawSync(blob,_usz);break;case 0:break;// TODO: scan for magic number
	default:throw new Error("Unsupported ZIP Compression method "+meth);}/* [data descriptor] */var wrn=false;if(flags&8){crc32=blob.read_shift(4);if(crc32==0x08074b50){crc32=blob.read_shift(4);wrn=true;}_csz=blob.read_shift(4);_usz=blob.read_shift(4);}if(_csz!=csz)warn_or_throw(wrn,"Bad compressed size: "+csz+" != "+_csz);if(_usz!=usz)warn_or_throw(wrn,"Bad uncompressed size: "+usz+" != "+_usz);//var _crc32 = CRC32.buf(data, 0);
	//if((crc32>>0) != (_crc32>>0)) warn_or_throw(wrn, "Bad CRC32 checksum: " + crc32 + " != " + _crc32);
	cfb_add(o,name,data,{unsafe:true,mt:date});}function write_zip(cfb/*:CFBContainer*/,options/*:CFBWriteOpts*/)/*:RawBytes*/{var _opts=options||{};var out=[],cdirs=[];var o/*:CFBlob*/=new_buf(1);var method=_opts.compression?8:0,flags=0;var i=0,j=0;var start_cd=0,fcnt=0;var root=cfb.FullPaths[0],fp=root,fi=cfb.FileIndex[0];var crcs=[];var sz_cd=0;for(i=1;i<cfb.FullPaths.length;++i){fp=cfb.FullPaths[i].slice(root.length);fi=cfb.FileIndex[i];if(!fi.size||!fi.content||fp=="\u0001Sh33tJ5")continue;var start=start_cd;/* TODO: CP437 filename */var namebuf=new_buf(fp.length);for(j=0;j<fp.length;++j)namebuf.write_shift(1,fp.charCodeAt(j)&0x7F);namebuf=namebuf.slice(0,namebuf.l);crcs[fcnt]=CRC32.buf(/*::((*/fi.content/*::||[]):any)*/,0);var outbuf=fi.content/*::||[]*/;if(method==8)outbuf=_deflateRawSync(outbuf);/* local file header */o=new_buf(30);o.write_shift(4,0x04034b50);o.write_shift(2,20);o.write_shift(2,flags);o.write_shift(2,method);/* TODO: last mod file time/date */if(fi.mt)write_dos_date(o,fi.mt);else o.write_shift(4,0);o.write_shift(-4,crcs[fcnt]);o.write_shift(4,outbuf.length);o.write_shift(4,/*::(*/fi.content/*::||[])*/.length);o.write_shift(2,namebuf.length);o.write_shift(2,0);start_cd+=o.length;out.push(o);start_cd+=namebuf.length;out.push(namebuf);/* TODO: extra fields? */ /* TODO: encryption header ? */start_cd+=outbuf.length;out.push(outbuf);/* central directory */o=new_buf(46);o.write_shift(4,0x02014b50);o.write_shift(2,0);o.write_shift(2,20);o.write_shift(2,flags);o.write_shift(2,method);o.write_shift(4,0);/* TODO: last mod file time/date */o.write_shift(-4,crcs[fcnt]);o.write_shift(4,outbuf.length);o.write_shift(4,/*::(*/fi.content/*::||[])*/.length);o.write_shift(2,namebuf.length);o.write_shift(2,0);o.write_shift(2,0);o.write_shift(2,0);o.write_shift(2,0);o.write_shift(4,0);o.write_shift(4,start);sz_cd+=o.l;cdirs.push(o);sz_cd+=namebuf.length;cdirs.push(namebuf);++fcnt;}/* end of central directory */o=new_buf(22);o.write_shift(4,0x06054b50);o.write_shift(2,0);o.write_shift(2,0);o.write_shift(2,fcnt);o.write_shift(2,fcnt);o.write_shift(4,sz_cd);o.write_shift(4,start_cd);o.write_shift(2,0);return bconcat([bconcat(out/*:any*/),bconcat(cdirs),o]/*:any*/);}var ContentTypeMap={"htm":"text/html","xml":"text/xml","gif":"image/gif","jpg":"image/jpeg","png":"image/png","mso":"application/x-mso","thmx":"application/vnd.ms-officetheme","sh33tj5":"application/octet-stream"}/*:any*/;function get_content_type(fi/*:CFBEntry*/,fp/*:string*/)/*:string*/{if(fi.ctype)return fi.ctype;var ext=fi.name||"",m=ext.match(/\.([^\.]+)$/);if(m&&ContentTypeMap[m[1]])return ContentTypeMap[m[1]];if(fp){m=(ext=fp).match(/[\.\\]([^\.\\])+$/);if(m&&ContentTypeMap[m[1]])return ContentTypeMap[m[1]];}return "application/octet-stream";}/* 76 character chunks TODO: intertwine encoding */function write_base64_76(bstr/*:string*/)/*:string*/{var data=Base64_encode(bstr);var o=[];for(var i=0;i<data.length;i+=76)o.push(data.slice(i,i+76));return o.join("\r\n")+"\r\n";}/*
	Rules for QP:
		- escape =## applies for all non-display characters and literal "="
		- space or tab at end of line must be encoded
		- \r\n newlines can be preserved, but bare \r and \n must be escaped
		- lines must not exceed 76 characters, use soft breaks =\r\n

	TODO: Some files from word appear to write line extensions with bare equals:

	```
	<table class=3DMsoTableGrid border=3D1 cellspacing=3D0 cellpadding=3D0 width=
	="70%"
	```
	*/function write_quoted_printable(text/*:string*/)/*:string*/{var encoded=text.replace(/[\x00-\x08\x0B\x0C\x0E-\x1F\x7E-\xFF=]/g,function(c){var w=c.charCodeAt(0).toString(16).toUpperCase();return "="+(w.length==1?"0"+w:w);});encoded=encoded.replace(/ $/mg,"=20").replace(/\t$/mg,"=09");if(encoded.charAt(0)=="\n")encoded="=0D"+encoded.slice(1);encoded=encoded.replace(/\r(?!\n)/mg,"=0D").replace(/\n\n/mg,"\n=0A").replace(/([^\r\n])\n/mg,"$1=0A");var o/*:Array<string>*/=[],split=encoded.split("\r\n");for(var si=0;si<split.length;++si){var str=split[si];if(str.length==0){o.push("");continue;}for(var i=0;i<str.length;){var end=76;var tmp=str.slice(i,i+end);if(tmp.charAt(end-1)=="=")end--;else if(tmp.charAt(end-2)=="=")end-=2;else if(tmp.charAt(end-3)=="=")end-=3;tmp=str.slice(i,i+end);i+=end;if(i<str.length)tmp+="=";o.push(tmp);}}return o.join("\r\n");}function parse_quoted_printable(data/*:Array<string>*/)/*:RawBytes*/{var o=[];/* unify long lines */for(var di=0;di<data.length;++di){var line=data[di];while(di<=data.length&&line.charAt(line.length-1)=="=")line=line.slice(0,line.length-1)+data[++di];o.push(line);}/* decode */for(var oi=0;oi<o.length;++oi)o[oi]=o[oi].replace(/[=][0-9A-Fa-f]{2}/g,function($$){return String.fromCharCode(parseInt($$.slice(1),16));});return s2a(o.join("\r\n"));}function parse_mime(cfb/*:CFBContainer*/,data/*:Array<string>*/,root/*:string*/)/*:void*/{var fname="",cte="",ctype="",fdata;var di=0;for(;di<10;++di){var line=data[di];if(!line||line.match(/^\s*$/))break;var m=line.match(/^(.*?):\s*([^\s].*)$/);if(m)switch(m[1].toLowerCase()){case"content-location":fname=m[2].trim();break;case"content-type":ctype=m[2].trim();break;case"content-transfer-encoding":cte=m[2].trim();break;}}++di;switch(cte.toLowerCase()){case'base64':fdata=s2a(Base64_decode(data.slice(di).join("")));break;case'quoted-printable':fdata=parse_quoted_printable(data.slice(di));break;default:throw new Error("Unsupported Content-Transfer-Encoding "+cte);}var file=cfb_add(cfb,fname.slice(root.length),fdata,{unsafe:true});if(ctype)file.ctype=ctype;}function parse_mad(file/*:RawBytes*/,options/*:CFBReadOpts*/)/*:CFBContainer*/{if(a2s(file.slice(0,13)).toLowerCase()!="mime-version:")throw new Error("Unsupported MAD header");var root=options&&options.root||"";// $FlowIgnore
	var data=(has_buf&&Buffer.isBuffer(file)?file.toString("binary"):a2s(file)).split("\r\n");var di=0,row="";/* if root is not specified, scan for the common prefix */for(di=0;di<data.length;++di){row=data[di];if(!/^Content-Location:/i.test(row))continue;row=row.slice(row.indexOf("file"));if(!root)root=row.slice(0,row.lastIndexOf("/")+1);if(row.slice(0,root.length)==root)continue;while(root.length>0){root=root.slice(0,root.length-1);root=root.slice(0,root.lastIndexOf("/")+1);if(row.slice(0,root.length)==root)break;}}var mboundary=(data[1]||"").match(/boundary="(.*?)"/);if(!mboundary)throw new Error("MAD cannot find boundary");var boundary="--"+(mboundary[1]||"");var FileIndex/*:CFBFileIndex*/=[],FullPaths/*:Array<string>*/=[];var o={FileIndex:FileIndex,FullPaths:FullPaths};init_cfb(o);var start_di,fcnt=0;for(di=0;di<data.length;++di){var line=data[di];if(line!==boundary&&line!==boundary+"--")continue;if(fcnt++)parse_mime(o,data.slice(start_di,di),root);start_di=di;}return o;}function write_mad(cfb/*:CFBContainer*/,options/*:CFBWriteOpts*/)/*:string*/{var opts=options||{};var boundary=opts.boundary||"SheetJS";boundary='------='+boundary;var out=['MIME-Version: 1.0','Content-Type: multipart/related; boundary="'+boundary.slice(2)+'"','','',''];var root=cfb.FullPaths[0],fp=root,fi=cfb.FileIndex[0];for(var i=1;i<cfb.FullPaths.length;++i){fp=cfb.FullPaths[i].slice(root.length);fi=cfb.FileIndex[i];if(!fi.size||!fi.content||fp=="\u0001Sh33tJ5")continue;/* Normalize filename */fp=fp.replace(/[\x00-\x08\x0B\x0C\x0E-\x1F\x7E-\xFF]/g,function(c){return "_x"+c.charCodeAt(0).toString(16)+"_";}).replace(/[\u0080-\uFFFF]/g,function(u){return "_u"+u.charCodeAt(0).toString(16)+"_";});/* Extract content as binary string */var ca=fi.content;// $FlowIgnore
	var cstr=has_buf&&Buffer.isBuffer(ca)?ca.toString("binary"):a2s(ca);/* 4/5 of first 1024 chars ascii -> quoted printable, else base64 */var dispcnt=0,L=Math.min(1024,cstr.length),cc=0;for(var csl=0;csl<=L;++csl)if((cc=cstr.charCodeAt(csl))>=0x20&&cc<0x80)++dispcnt;var qp=dispcnt>=L*4/5;out.push(boundary);out.push('Content-Location: '+(opts.root||'file:///C:/SheetJS/')+fp);out.push('Content-Transfer-Encoding: '+(qp?'quoted-printable':'base64'));out.push('Content-Type: '+get_content_type(fi,fp));out.push('');out.push(qp?write_quoted_printable(cstr):write_base64_76(cstr));}out.push(boundary+'--\r\n');return out.join("\r\n");}function cfb_new(opts/*:?any*/)/*:CFBContainer*/{var o/*:CFBContainer*/={}/*:any*/;init_cfb(o,opts);return o;}function cfb_add(cfb/*:CFBContainer*/,name/*:string*/,content/*:?RawBytes*/,opts/*:?any*/)/*:CFBEntry*/{var unsafe=opts&&opts.unsafe;if(!unsafe)init_cfb(cfb);var file=!unsafe&&CFB.find(cfb,name);if(!file){var fpath/*:string*/=cfb.FullPaths[0];if(name.slice(0,fpath.length)==fpath)fpath=name;else {if(fpath.slice(-1)!="/")fpath+="/";fpath=(fpath+name).replace("//","/");}file={name:filename(name),type:2}/*:any*/;cfb.FileIndex.push(file);cfb.FullPaths.push(fpath);if(!unsafe)CFB.utils.cfb_gc(cfb);}/*:: if(!file) throw new Error("unreachable"); */file.content=content/*:any*/;file.size=content?content.length:0;if(opts){if(opts.CLSID)file.clsid=opts.CLSID;if(opts.mt)file.mt=opts.mt;if(opts.ct)file.ct=opts.ct;}return file;}function cfb_del(cfb/*:CFBContainer*/,name/*:string*/)/*:boolean*/{init_cfb(cfb);var file=CFB.find(cfb,name);if(file)for(var j=0;j<cfb.FileIndex.length;++j)if(cfb.FileIndex[j]==file){cfb.FileIndex.splice(j,1);cfb.FullPaths.splice(j,1);return true;}return false;}function cfb_mov(cfb/*:CFBContainer*/,old_name/*:string*/,new_name/*:string*/)/*:boolean*/{init_cfb(cfb);var file=CFB.find(cfb,old_name);if(file)for(var j=0;j<cfb.FileIndex.length;++j)if(cfb.FileIndex[j]==file){cfb.FileIndex[j].name=filename(new_name);cfb.FullPaths[j]=new_name;return true;}return false;}function cfb_gc(cfb/*:CFBContainer*/)/*:void*/{rebuild_cfb(cfb,true);}exports.find=find;exports.read=read;exports.parse=parse;exports.write=write;exports.writeFile=write_file;exports.utils={cfb_new:cfb_new,cfb_add:cfb_add,cfb_del:cfb_del,cfb_mov:cfb_mov,cfb_gc:cfb_gc,ReadShift:ReadShift,CheckField:CheckField,prep_blob:prep_blob,bconcat:bconcat,use_zlib:use_zlib,_deflateRaw:_deflate,_inflateRaw:_inflate,consts:consts};return exports;}();/* read binary data from file */function read_binary(path/*:string*/){if(typeof Deno!=='undefined')return Deno.readFileSync(path);// $FlowIgnore
	if(typeof $!=='undefined'&&typeof File!=='undefined'&&typeof Folder!=='undefined')try{// extendscript
	// $FlowIgnore
	var infile=File(path);infile.open("r");infile.encoding="binary";var data=infile.read();infile.close();return data;}catch(e){if(!e.message||!e.message.match(/onstruct/))throw e;}throw new Error("Cannot access file "+path);}function keys(o/*:any*/)/*:Array<any>*/{var ks=Object.keys(o),o2=[];for(var i=0;i<ks.length;++i)if(Object.prototype.hasOwnProperty.call(o,ks[i]))o2.push(ks[i]);return o2;}function evert(obj/*:any*/)/*:EvertType*/{var o=[]/*:any*/,K=keys(obj);for(var i=0;i!==K.length;++i)o[obj[K[i]]]=K[i];return o;}var basedate=/*#__PURE__*/new Date(1899,11,30,0,0,0);// 2209161600000
	function datenum(v/*:Date*/,date1904/*:?boolean*/)/*:number*/{var epoch=/*#__PURE__*/v.getTime();if(date1904)epoch-=1462*24*60*60*1000;var dnthresh=/*#__PURE__*/basedate.getTime()+(/*#__PURE__*/v.getTimezoneOffset()-/*#__PURE__*/basedate.getTimezoneOffset())*60000;return (epoch-dnthresh)/(24*60*60*1000);}var refdate=/*#__PURE__*/new Date();var dnthresh=/*#__PURE__*/basedate.getTime()+(/*#__PURE__*/refdate.getTimezoneOffset()-/*#__PURE__*/basedate.getTimezoneOffset())*60000;var refoffset=/*#__PURE__*/refdate.getTimezoneOffset();function numdate(v/*:number*/)/*:Date*/{var out=new Date();out.setTime(v*24*60*60*1000+dnthresh);if(out.getTimezoneOffset()!==refoffset){out.setTime(out.getTime()+(out.getTimezoneOffset()-refoffset)*60000);}return out;}/* ISO 8601 Duration */function parse_isodur(s){var sec=0,mt=0,time=false;var m=s.match(/P([0-9\.]+Y)?([0-9\.]+M)?([0-9\.]+D)?T([0-9\.]+H)?([0-9\.]+M)?([0-9\.]+S)?/);if(!m)throw new Error("|"+s+"| is not an ISO8601 Duration");for(var i=1;i!=m.length;++i){if(!m[i])continue;mt=1;if(i>3)time=true;switch(m[i].slice(m[i].length-1)){case'Y':throw new Error("Unsupported ISO Duration Field: "+m[i].slice(m[i].length-1));case'D':mt*=24;/* falls through */case'H':mt*=60;/* falls through */case'M':if(!time)throw new Error("Unsupported ISO Duration Field: M");else mt*=60;}sec+=mt*parseInt(m[i],10);}return sec;}var good_pd_date_1=/*#__PURE__*/new Date('2017-02-19T19:06:09.000Z');var good_pd_date=/*#__PURE__*/isNaN(/*#__PURE__*/good_pd_date_1.getFullYear())?/*#__PURE__*/new Date('2/19/17'):good_pd_date_1;var good_pd=/*#__PURE__*/good_pd_date.getFullYear()==2017;/* parses a date as a local date */function parseDate(str/*:string|Date*/,fixdate/*:?number*/)/*:Date*/{var d=new Date(str);if(good_pd){/*:: if(fixdate == null) fixdate = 0; */if(fixdate>0)d.setTime(d.getTime()+d.getTimezoneOffset()*60*1000);else if(fixdate<0)d.setTime(d.getTime()-d.getTimezoneOffset()*60*1000);return d;}if(str instanceof Date)return str;if(good_pd_date.getFullYear()==1917&&!isNaN(d.getFullYear())){var s=d.getFullYear();if(str.indexOf(""+s)>-1)return d;d.setFullYear(d.getFullYear()+100);return d;}var n=str.match(/\d+/g)||["2017","2","19","0","0","0"];var out=new Date(+n[0],+n[1]-1,+n[2],+n[3]||0,+n[4]||0,+n[5]||0);if(str.indexOf("Z")>-1)out=new Date(out.getTime()-out.getTimezoneOffset()*60*1000);return out;}function cc2str(arr/*:Array<number>*/,debomit)/*:string*/{if(has_buf&&Buffer.isBuffer(arr)){if(debomit){if(arr[0]==0xFF&&arr[1]==0xFE)return utf8write(arr.slice(2).toString("utf16le"));if(arr[1]==0xFE&&arr[2]==0xFF)return utf8write(utf16beread(arr.slice(2).toString("binary")));}return arr.toString("binary");}if(typeof TextDecoder!=="undefined")try{if(debomit){if(arr[0]==0xFF&&arr[1]==0xFE)return utf8write(new TextDecoder("utf-16le").decode(arr.slice(2)));if(arr[0]==0xFE&&arr[1]==0xFF)return utf8write(new TextDecoder("utf-16be").decode(arr.slice(2)));}var rev={"\u20ac":"\x80","\u201a":"\x82","\u0192":"\x83","\u201e":"\x84","\u2026":"\x85","\u2020":"\x86","\u2021":"\x87","\u02c6":"\x88","\u2030":"\x89","\u0160":"\x8a","\u2039":"\x8b","\u0152":"\x8c","\u017d":"\x8e","\u2018":"\x91","\u2019":"\x92","\u201c":"\x93","\u201d":"\x94","\u2022":"\x95","\u2013":"\x96","\u2014":"\x97","\u02dc":"\x98","\u2122":"\x99","\u0161":"\x9a","\u203a":"\x9b","\u0153":"\x9c","\u017e":"\x9e","\u0178":"\x9f"};if(Array.isArray(arr))arr=new Uint8Array(arr);return new TextDecoder("latin1").decode(arr).replace(/[€‚ƒ„…†‡ˆ‰Š‹ŒŽ‘’“”•–—˜™š›œžŸ]/g,function(c){return rev[c]||c;});}catch(e){}var o=[];for(var i=0;i!=arr.length;++i)o.push(String.fromCharCode(arr[i]));return o.join("");}function dup(o/*:any*/)/*:any*/{if(typeof JSON!='undefined'&&!Array.isArray(o))return JSON.parse(JSON.stringify(o));if(typeof o!='object'||o==null)return o;if(o instanceof Date)return new Date(o.getTime());var out={};for(var k in o)if(Object.prototype.hasOwnProperty.call(o,k))out[k]=dup(o[k]);return out;}function fill(c/*:string*/,l/*:number*/)/*:string*/{var o="";while(o.length<l)o+=c;return o;}/* TODO: stress test */function fuzzynum(s/*:string*/)/*:number*/{var v/*:number*/=Number(s);if(!isNaN(v))return isFinite(v)?v:NaN;if(!/\d/.test(s))return v;var wt=1;var ss=s.replace(/([\d]),([\d])/g,"$1$2").replace(/[$]/g,"").replace(/[%]/g,function(){wt*=100;return "";});if(!isNaN(v=Number(ss)))return v/wt;ss=ss.replace(/[(](.*)[)]/,function($$,$1){wt=-wt;return $1;});if(!isNaN(v=Number(ss)))return v/wt;return v;}var lower_months=['january','february','march','april','may','june','july','august','september','october','november','december'];function fuzzydate(s/*:string*/)/*:Date*/{var o=new Date(s),n=new Date(NaN);var y=o.getYear(),m=o.getMonth(),d=o.getDate();if(isNaN(d))return n;var lower=s.toLowerCase();if(lower.match(/jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec/)){lower=lower.replace(/[^a-z]/g,"").replace(/([^a-z]|^)[ap]m?([^a-z]|$)/,"");if(lower.length>3&&lower_months.indexOf(lower)==-1)return n;}else if(lower.match(/[a-z]/))return n;if(y<0||y>8099)return n;if((m>0||d>1)&&y!=101)return o;if(s.match(/[^-0-9:,\/\\]/))return n;return o;}var split_regex=/*#__PURE__*/function(){var safe_split_regex="abacaba".split(/(:?b)/i).length==5;return function split_regex(str/*:string*/,re,def/*:string*/)/*:Array<string>*/{if(safe_split_regex||typeof re=="string")return str.split(re);var p=str.split(re),o=[p[0]];for(var i=1;i<p.length;++i){o.push(def);o.push(p[i]);}return o;};}();function getdatastr(data)/*:?string*/{if(!data)return null;if(data.content&&data.type)return cc2str(data.content,true);if(data.data)return debom(data.data);if(data.asNodeBuffer&&has_buf)return debom(data.asNodeBuffer().toString('binary'));if(data.asBinary)return debom(data.asBinary());if(data._data&&data._data.getContent)return debom(cc2str(Array.prototype.slice.call(data._data.getContent(),0)));return null;}function getdatabin(data){if(!data)return null;if(data.data)return char_codes(data.data);if(data.asNodeBuffer&&has_buf)return data.asNodeBuffer();if(data._data&&data._data.getContent){var o=data._data.getContent();if(typeof o=="string")return char_codes(o);return Array.prototype.slice.call(o);}if(data.content&&data.type)return data.content;return null;}function getdata(data){return data&&data.name.slice(-4)===".bin"?getdatabin(data):getdatastr(data);}/* Part 2 Section 10.1.2 "Mapping Content Types" Names are case-insensitive */ /* OASIS does not comment on filename case sensitivity */function safegetzipfile(zip,file/*:string*/){var k=zip.FullPaths||keys(zip.files);var f=file.toLowerCase().replace(/[\/]/g,'\\'),g=f.replace(/\\/g,'\/');for(var i=0;i<k.length;++i){var n=k[i].replace(/^Root Entry[\/]/,"").toLowerCase();if(f==n||g==n)return zip.files?zip.files[k[i]]:zip.FileIndex[i];}return null;}function getzipfile(zip,file/*:string*/){var o=safegetzipfile(zip,file);if(o==null)throw new Error("Cannot find file "+file+" in zip");return o;}function getzipdata(zip,file/*:string*/,safe/*:?boolean*/)/*:any*/{if(!safe)return getdata(getzipfile(zip,file));if(!file)return null;try{return getzipdata(zip,file);}catch(e){return null;}}function getzipstr(zip,file/*:string*/,safe/*:?boolean*/)/*:?string*/{if(!safe)return getdatastr(getzipfile(zip,file));if(!file)return null;try{return getzipstr(zip,file);}catch(e){return null;}}function getzipbin(zip,file/*:string*/,safe/*:?boolean*/)/*:any*/{if(!safe)return getdatabin(getzipfile(zip,file));if(!file)return null;try{return getzipbin(zip,file);}catch(e){return null;}}function zipentries(zip){var k=zip.FullPaths||keys(zip.files),o=[];for(var i=0;i<k.length;++i)if(k[i].slice(-1)!='/')o.push(k[i].replace(/^Root Entry[\/]/,""));return o.sort();}function zip_add_file(zip,path,content){if(zip.FullPaths){if(typeof content=="string"){var res;if(has_buf)res=Buffer_from(content);/* TODO: investigate performance in Edge 13 */ //else if(typeof TextEncoder !== "undefined") res = new TextEncoder().encode(content);
	else res=utf8decode(content);return CFB.utils.cfb_add(zip,path,res);}CFB.utils.cfb_add(zip,path,content);}else zip.file(path,content);}function zip_read(d,o){switch(o.type){case"base64":return CFB.read(d,{type:"base64"});case"binary":return CFB.read(d,{type:"binary"});case"buffer":case"array":return CFB.read(d,{type:"buffer"});}throw new Error("Unrecognized type "+o.type);}function resolve_path(path/*:string*/,base/*:string*/)/*:string*/{if(path.charAt(0)=="/")return path.slice(1);var result=base.split('/');if(base.slice(-1)!="/")result.pop();// folder path
	var target=path.split('/');while(target.length!==0){var step=target.shift();if(step==='..')result.pop();else if(step!=='.')result.push(step);}return result.join('/');}var XML_HEADER='<?xml version="1.0" encoding="UTF-8" standalone="yes"?>\r\n';var attregexg=/([^"\s?>\/]+)\s*=\s*((?:")([^"]*)(?:")|(?:')([^']*)(?:')|([^'">\s]+))/g;var tagregex1=/<[\/\?]?[a-zA-Z0-9:_-]+(?:\s+[^"\s?>\/]+\s*=\s*(?:"[^"]*"|'[^']*'|[^'">\s=]+))*\s*[\/\?]?>/mg,tagregex2=/<[^>]*>/g;var tagregex=/*#__PURE__*/XML_HEADER.match(tagregex1)?tagregex1:tagregex2;var nsregex=/<\w*:/,nsregex2=/<(\/?)\w+:/;function parsexmltag(tag/*:string*/,skip_root/*:?boolean*/,skip_LC/*:?boolean*/)/*:any*/{var z={}/*:any*/;var eq=0,c=0;for(;eq!==tag.length;++eq)if((c=tag.charCodeAt(eq))===32||c===10||c===13)break;if(!skip_root)z[0]=tag.slice(0,eq);if(eq===tag.length)return z;var m=tag.match(attregexg),j=0,v="",i=0,q="",cc="",quot=1;if(m)for(i=0;i!=m.length;++i){cc=m[i];for(c=0;c!=cc.length;++c)if(cc.charCodeAt(c)===61)break;q=cc.slice(0,c).trim();while(cc.charCodeAt(c+1)==32)++c;quot=(eq=cc.charCodeAt(c+1))==34||eq==39?1:0;v=cc.slice(c+1+quot,cc.length-quot);for(j=0;j!=q.length;++j)if(q.charCodeAt(j)===58)break;if(j===q.length){if(q.indexOf("_")>0)q=q.slice(0,q.indexOf("_"));// from ods
	z[q]=v;if(!skip_LC)z[q.toLowerCase()]=v;}else {var k=(j===5&&q.slice(0,5)==="xmlns"?"xmlns":"")+q.slice(j+1);if(z[k]&&q.slice(j-3,j)=="ext")continue;// from ods
	z[k]=v;if(!skip_LC)z[k.toLowerCase()]=v;}}return z;}function strip_ns(x/*:string*/)/*:string*/{return x.replace(nsregex2,"<$1");}var encodings={'&quot;':'"','&apos;':"'",'&gt;':'>','&lt;':'<','&amp;':'&'};var rencoding=/*#__PURE__*/evert(encodings);//var rencstr = "&<>'\"".split("");
	// TODO: CP remap (need to read file version to determine OS)
	var unescapexml/*:StringConv*/=/*#__PURE__*/function(){/* 22.4.2.4 bstr (Basic String) */var encregex=/&(?:quot|apos|gt|lt|amp|#x?([\da-fA-F]+));/ig,coderegex=/_x([\da-fA-F]{4})_/ig;return function unescapexml(text/*:string*/)/*:string*/{var s=text+'',i=s.indexOf("<![CDATA[");if(i==-1)return s.replace(encregex,function($$,$1){return encodings[$$]||String.fromCharCode(parseInt($1,$$.indexOf("x")>-1?16:10))||$$;}).replace(coderegex,function(m,c){return String.fromCharCode(parseInt(c,16));});var j=s.indexOf("]]>");return unescapexml(s.slice(0,i))+s.slice(i+9,j)+unescapexml(s.slice(j+3));};}();var decregex=/[&<>'"]/g;var htmlcharegex=/[\u0000-\u001f]/g;function escapehtml(text/*:string*/)/*:string*/{var s=text+'';return s.replace(decregex,function(y){return rencoding[y];}).replace(/\n/g,"<br/>").replace(htmlcharegex,function(s){return "&#x"+("000"+s.charCodeAt(0).toString(16)).slice(-4)+";";});}/* TODO: handle codepages */var xlml_fixstr/*:StringConv*/=/*#__PURE__*/function(){var entregex=/&#(\d+);/g;function entrepl($$/*:string*/,$1/*:string*/)/*:string*/{return String.fromCharCode(parseInt($1,10));}return function xlml_fixstr(str/*:string*/)/*:string*/{return str.replace(entregex,entrepl);};}();function parsexmlbool(value/*:any*/)/*:boolean*/{switch(value){case 1:case true:case'1':case'true':case'TRUE':return true;/* case '0': case 'false': case 'FALSE':*/default:return false;}}function utf8reada(orig/*:string*/)/*:string*/{var out="",i=0,c=0,d=0,e=0,f=0,w=0;while(i<orig.length){c=orig.charCodeAt(i++);if(c<128){out+=String.fromCharCode(c);continue;}d=orig.charCodeAt(i++);if(c>191&&c<224){f=(c&31)<<6;f|=d&63;out+=String.fromCharCode(f);continue;}e=orig.charCodeAt(i++);if(c<240){out+=String.fromCharCode((c&15)<<12|(d&63)<<6|e&63);continue;}f=orig.charCodeAt(i++);w=((c&7)<<18|(d&63)<<12|(e&63)<<6|f&63)-65536;out+=String.fromCharCode(0xD800+(w>>>10&1023));out+=String.fromCharCode(0xDC00+(w&1023));}return out;}function utf8readb(data){var out=new_raw_buf(2*data.length),w,i,j=1,k=0,ww=0,c;for(i=0;i<data.length;i+=j){j=1;if((c=data.charCodeAt(i))<128)w=c;else if(c<224){w=(c&31)*64+(data.charCodeAt(i+1)&63);j=2;}else if(c<240){w=(c&15)*4096+(data.charCodeAt(i+1)&63)*64+(data.charCodeAt(i+2)&63);j=3;}else {j=4;w=(c&7)*262144+(data.charCodeAt(i+1)&63)*4096+(data.charCodeAt(i+2)&63)*64+(data.charCodeAt(i+3)&63);w-=65536;ww=0xD800+(w>>>10&1023);w=0xDC00+(w&1023);}if(ww!==0){out[k++]=ww&255;out[k++]=ww>>>8;ww=0;}out[k++]=w%256;out[k++]=w>>>8;}return out.slice(0,k).toString('ucs2');}function utf8readc(data){return Buffer_from(data,'binary').toString('utf8');}var utf8corpus="foo bar baz\u00e2\u0098\u0083\u00f0\u009f\u008d\u00a3";var utf8read=has_buf&&(/*#__PURE__*/utf8readc(utf8corpus)==/*#__PURE__*/utf8reada(utf8corpus)&&utf8readc||/*#__PURE__*/utf8readb(utf8corpus)==/*#__PURE__*/utf8reada(utf8corpus)&&utf8readb)||utf8reada;var utf8write/*:StringConv*/=has_buf?function(data){return Buffer_from(data,'utf8').toString("binary");}:function(orig/*:string*/)/*:string*/{var out/*:Array<string>*/=[],i=0,c=0,d=0;while(i<orig.length){c=orig.charCodeAt(i++);switch(true){case c<128:out.push(String.fromCharCode(c));break;case c<2048:out.push(String.fromCharCode(192+(c>>6)));out.push(String.fromCharCode(128+(c&63)));break;case c>=55296&&c<57344:c-=55296;d=orig.charCodeAt(i++)-56320+(c<<10);out.push(String.fromCharCode(240+(d>>18&7)));out.push(String.fromCharCode(144+(d>>12&63)));out.push(String.fromCharCode(128+(d>>6&63)));out.push(String.fromCharCode(128+(d&63)));break;default:out.push(String.fromCharCode(224+(c>>12)));out.push(String.fromCharCode(128+(c>>6&63)));out.push(String.fromCharCode(128+(c&63)));}}return out.join("");};// matches <foo>...</foo> extracts content
	var matchtag=/*#__PURE__*/function(){var mtcache/*:{[k:string]:RegExp}*/={}/*:any*/;return function matchtag(f/*:string*/,g/*:?string*/)/*:RegExp*/{var t=f+"|"+(g||"");if(mtcache[t])return mtcache[t];return mtcache[t]=new RegExp('<(?:\\w+:)?'+f+'(?: xml:space="preserve")?(?:[^>]*)>([\\s\\S]*?)</(?:\\w+:)?'+f+'>',g||""/*:any*/);};}();var htmldecode/*:{(s:string):string}*/=/*#__PURE__*/function(){var entities/*:Array<[RegExp, string]>*/=[['nbsp',' '],['middot','·'],['quot','"'],['apos',"'"],['gt','>'],['lt','<'],['amp','&']].map(function(x/*:[string, string]*/){return [new RegExp('&'+x[0]+';',"ig"),x[1]];});return function htmldecode(str/*:string*/)/*:string*/{var o=str// Remove new lines and spaces from start of content
	.replace(/^[\t\n\r ]+/,"")// Remove new lines and spaces from end of content
	.replace(/[\t\n\r ]+$/,"")// Added line which removes any white space characters after and before html tags
	.replace(/>\s+/g,">").replace(/\s+</g,"<")// Replace remaining new lines and spaces with space
	.replace(/[\t\n\r ]+/g," ")// Replace <br> tags with new lines
	.replace(/<\s*[bB][rR]\s*\/?>/g,"\n")// Strip HTML elements
	.replace(/<[^>]*>/g,"");for(var i=0;i<entities.length;++i)o=o.replace(entities[i][0],entities[i][1]);return o;};}();var vtregex=/*#__PURE__*/function(){var vt_cache={};return function vt_regex(bt){if(vt_cache[bt]!==undefined)return vt_cache[bt];return vt_cache[bt]=new RegExp("<(?:vt:)?"+bt+">([\\s\\S]*?)</(?:vt:)?"+bt+">",'g');};}();var vtvregex=/<\/?(?:vt:)?variant>/g,vtmregex=/<(?:vt:)([^>]*)>([\s\S]*)</;function parseVector(data/*:string*/,opts)/*:Array<{v:string,t:string}>*/{var h=parsexmltag(data);var matches/*:Array<string>*/=data.match(vtregex(h.baseType))||[];var res/*:Array<any>*/=[];if(matches.length!=h.size){if(opts.WTF)throw new Error("unexpected vector length "+matches.length+" != "+h.size);return res;}matches.forEach(function(x/*:string*/){var v=x.replace(vtvregex,"").match(vtmregex);if(v)res.push({v:utf8read(v[2]),t:v[1]});});return res;}var wtregex=/(^\s|\s$|\n)/;function wxt_helper(h)/*:string*/{return keys(h).map(function(k){return " "+k+'="'+h[k]+'"';}).join("");}function writextag(f/*:string*/,g/*:?string*/,h){return '<'+f+(h!=null?wxt_helper(h):"")+(g!=null?(g.match(wtregex)?' xml:space="preserve"':"")+'>'+g+'</'+f:"/")+'>';}function xlml_normalize(d)/*:string*/{if(has_buf&&/*::typeof Buffer !== "undefined" && d != null && d instanceof Buffer &&*/Buffer.isBuffer(d))return d.toString('utf8');if(typeof d==='string')return d;/* duktape */if(typeof Uint8Array!=='undefined'&&d instanceof Uint8Array)return utf8read(a2s(ab2a(d)));throw new Error("Bad input format: expected Buffer or string");}/* UOS uses CJK in tags */var xlmlregex=/<(\/?)([^\s?><!\/:]*:|)([^\s?<>:\/]+)(?:[\s?:\/][^>]*)?>/mg;//var xlmlregex = /<(\/?)([a-z0-9]*:|)(\w+)[^>]*>/mg;
	var XMLNS={CORE_PROPS:'http://schemas.openxmlformats.org/package/2006/metadata/core-properties',CUST_PROPS:"http://schemas.openxmlformats.org/officeDocument/2006/custom-properties",EXT_PROPS:"http://schemas.openxmlformats.org/officeDocument/2006/extended-properties",CT:'http://schemas.openxmlformats.org/package/2006/content-types',RELS:'http://schemas.openxmlformats.org/package/2006/relationships',TCMNT:'http://schemas.microsoft.com/office/spreadsheetml/2018/threadedcomments','dc':'http://purl.org/dc/elements/1.1/','dcterms':'http://purl.org/dc/terms/','dcmitype':'http://purl.org/dc/dcmitype/','mx':'http://schemas.microsoft.com/office/mac/excel/2008/main','r':'http://schemas.openxmlformats.org/officeDocument/2006/relationships','sjs':'http://schemas.openxmlformats.org/package/2006/sheetjs/core-properties','vt':'http://schemas.openxmlformats.org/officeDocument/2006/docPropsVTypes','xsi':'http://www.w3.org/2001/XMLSchema-instance','xsd':'http://www.w3.org/2001/XMLSchema'}/*:any*/;var XMLNS_main=['http://schemas.openxmlformats.org/spreadsheetml/2006/main','http://purl.oclc.org/ooxml/spreadsheetml/main','http://schemas.microsoft.com/office/excel/2006/main','http://schemas.microsoft.com/office/excel/2006/2'];function read_double_le(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{var s=1-2*(b[idx+7]>>>7);var e=((b[idx+7]&0x7f)<<4)+(b[idx+6]>>>4&0x0f);var m=b[idx+6]&0x0f;for(var i=5;i>=0;--i)m=m*256+b[idx+i];if(e==0x7ff)return m==0?s*Infinity:NaN;if(e==0)e=-1022;else {e-=1023;m+=Math.pow(2,52);}return s*Math.pow(2,e-52)*m;}function write_double_le(b/*:RawBytes|CFBlob*/,v/*:number*/,idx/*:number*/){var bs=(v<0||1/v==-Infinity?1:0)<<7,e=0,m=0;var av=bs?-v:v;if(!isFinite(av)){e=0x7ff;m=isNaN(v)?0x6969:0;}else if(av==0)e=m=0;else {e=Math.floor(Math.log(av)/Math.LN2);m=av*Math.pow(2,52-e);if(e<=-1023&&(!isFinite(m)||m<Math.pow(2,52))){e=-1022;}else {m-=Math.pow(2,52);e+=1023;}}for(var i=0;i<=5;++i,m/=256)b[idx+i]=m&0xff;b[idx+6]=(e&0x0f)<<4|m&0xf;b[idx+7]=e>>4|bs;}var ___toBuffer=function(bufs/*:Array<Array<RawBytes> >*/)/*:RawBytes*/{var x=[],w=10240;for(var i=0;i<bufs[0].length;++i)if(bufs[0][i])for(var j=0,L=bufs[0][i].length;j<L;j+=w)x.push.apply(x,bufs[0][i].slice(j,j+w));return x;};var __toBuffer=has_buf?function(bufs){return bufs[0].length>0&&Buffer.isBuffer(bufs[0][0])?Buffer.concat(bufs[0].map(function(x){return Buffer.isBuffer(x)?x:Buffer_from(x);})):___toBuffer(bufs);}:___toBuffer;var ___utf16le=function(b/*:RawBytes|CFBlob*/,s/*:number*/,e/*:number*/)/*:string*/{var ss/*:Array<string>*/=[];for(var i=s;i<e;i+=2)ss.push(String.fromCharCode(__readUInt16LE(b,i)));return ss.join("").replace(chr0,'');};var __utf16le=has_buf?function(b/*:RawBytes|CFBlob*/,s/*:number*/,e/*:number*/)/*:string*/{if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___utf16le(b,s,e);return b.toString('utf16le',s,e).replace(chr0,'')/*.replace(chr1,'!')*/;}:___utf16le;var ___hexlify=function(b/*:RawBytes|CFBlob*/,s/*:number*/,l/*:number*/)/*:string*/{var ss/*:Array<string>*/=[];for(var i=s;i<s+l;++i)ss.push(("0"+b[i].toString(16)).slice(-2));return ss.join("");};var __hexlify=has_buf?function(b/*:RawBytes|CFBlob*/,s/*:number*/,l/*:number*/)/*:string*/{return Buffer.isBuffer(b)/*:: && b instanceof Buffer*/?b.toString('hex',s,s+l):___hexlify(b,s,l);}:___hexlify;var ___utf8=function(b/*:RawBytes|CFBlob*/,s/*:number*/,e/*:number*/){var ss=[];for(var i=s;i<e;i++)ss.push(String.fromCharCode(__readUInt8(b,i)));return ss.join("");};var __utf8=has_buf?function utf8_b(b/*:RawBytes|CFBlob*/,s/*:number*/,e/*:number*/){return Buffer.isBuffer(b)/*:: && (b instanceof Buffer)*/?b.toString('utf8',s,e):___utf8(b,s,e);}:___utf8;var ___lpstr=function(b/*:RawBytes|CFBlob*/,i/*:number*/){var len=__readUInt32LE(b,i);return len>0?__utf8(b,i+4,i+4+len-1):"";};var __lpstr=___lpstr;var ___cpstr=function(b/*:RawBytes|CFBlob*/,i/*:number*/){var len=__readUInt32LE(b,i);return len>0?__utf8(b,i+4,i+4+len-1):"";};var __cpstr=___cpstr;var ___lpwstr=function(b/*:RawBytes|CFBlob*/,i/*:number*/){var len=2*__readUInt32LE(b,i);return len>0?__utf8(b,i+4,i+4+len-1):"";};var __lpwstr=___lpwstr;var ___lpp4=function lpp4_(b/*:RawBytes|CFBlob*/,i/*:number*/){var len=__readUInt32LE(b,i);return len>0?__utf16le(b,i+4,i+4+len):"";};var __lpp4=___lpp4;var ___8lpp4=function(b/*:RawBytes|CFBlob*/,i/*:number*/){var len=__readUInt32LE(b,i);return len>0?__utf8(b,i+4,i+4+len):"";};var __8lpp4=___8lpp4;var ___double=function(b/*:RawBytes|CFBlob*/,idx/*:number*/){return read_double_le(b,idx);};var __double=___double;var is_buf=function is_buf_a(a){return Array.isArray(a)||typeof Uint8Array!=="undefined"&&a instanceof Uint8Array;};if(has_buf/*:: && typeof Buffer !== 'undefined'*/){__lpstr=function lpstr_b(b/*:RawBytes|CFBlob*/,i/*:number*/){if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___lpstr(b,i);var len=b.readUInt32LE(i);return len>0?b.toString('utf8',i+4,i+4+len-1):"";};__cpstr=function cpstr_b(b/*:RawBytes|CFBlob*/,i/*:number*/){if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___cpstr(b,i);var len=b.readUInt32LE(i);return len>0?b.toString('utf8',i+4,i+4+len-1):"";};__lpwstr=function lpwstr_b(b/*:RawBytes|CFBlob*/,i/*:number*/){if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___lpwstr(b,i);var len=2*b.readUInt32LE(i);return b.toString('utf16le',i+4,i+4+len-1);};__lpp4=function lpp4_b(b/*:RawBytes|CFBlob*/,i/*:number*/){if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___lpp4(b,i);var len=b.readUInt32LE(i);return b.toString('utf16le',i+4,i+4+len);};__8lpp4=function lpp4_8b(b/*:RawBytes|CFBlob*/,i/*:number*/){if(!Buffer.isBuffer(b)/*:: || !(b instanceof Buffer)*/)return ___8lpp4(b,i);var len=b.readUInt32LE(i);return b.toString('utf8',i+4,i+4+len);};__double=function double_(b/*:RawBytes|CFBlob*/,i/*:number*/){if(Buffer.isBuffer(b)/*::&& b instanceof Buffer*/)return b.readDoubleLE(i);return ___double(b,i);};is_buf=function is_buf_b(a){return Buffer.isBuffer(a)||Array.isArray(a)||typeof Uint8Array!=="undefined"&&a instanceof Uint8Array;};}var __readUInt8=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{return b[idx];};var __readUInt16LE=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{return b[idx+1]*(1<<8)+b[idx];};var __readInt16LE=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{var u=b[idx+1]*(1<<8)+b[idx];return u<0x8000?u:(0xffff-u+1)*-1;};var __readUInt32LE=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{return b[idx+3]*(1<<24)+(b[idx+2]<<16)+(b[idx+1]<<8)+b[idx];};var __readInt32LE=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{return b[idx+3]<<24|b[idx+2]<<16|b[idx+1]<<8|b[idx];};var __readInt32BE=function(b/*:RawBytes|CFBlob*/,idx/*:number*/)/*:number*/{return b[idx]<<24|b[idx+1]<<16|b[idx+2]<<8|b[idx+3];};function ReadShift(size/*:number*/,t/*:?string*/)/*:number|string*/{var o="",oI/*:: :number = 0*/,oR,oo=[],w,vv,i,loc;switch(t){case'dbcs':loc=this.l;if(has_buf&&Buffer.isBuffer(this))o=this.slice(this.l,this.l+2*size).toString("utf16le");else for(i=0;i<size;++i){o+=String.fromCharCode(__readUInt16LE(this,loc));loc+=2;}size*=2;break;case'utf8':o=__utf8(this,this.l,this.l+size);break;case'utf16le':size*=2;o=__utf16le(this,this.l,this.l+size);break;case'wstr':return ReadShift.call(this,size,'dbcs');/* [MS-OLEDS] 2.1.4 LengthPrefixedAnsiString */case'lpstr-ansi':o=__lpstr(this,this.l);size=4+__readUInt32LE(this,this.l);break;case'lpstr-cp':o=__cpstr(this,this.l);size=4+__readUInt32LE(this,this.l);break;/* [MS-OLEDS] 2.1.5 LengthPrefixedUnicodeString */case'lpwstr':o=__lpwstr(this,this.l);size=4+2*__readUInt32LE(this,this.l);break;/* [MS-OFFCRYPTO] 2.1.2 Length-Prefixed Padded Unicode String (UNICODE-LP-P4) */case'lpp4':size=4+__readUInt32LE(this,this.l);o=__lpp4(this,this.l);if(size&0x02)size+=2;break;/* [MS-OFFCRYPTO] 2.1.3 Length-Prefixed UTF-8 String (UTF-8-LP-P4) */case'8lpp4':size=4+__readUInt32LE(this,this.l);o=__8lpp4(this,this.l);if(size&0x03)size+=4-(size&0x03);break;case'cstr':size=0;o="";while((w=__readUInt8(this,this.l+size++))!==0)oo.push(_getchar(w));o=oo.join("");break;case'_wstr':size=0;o="";while((w=__readUInt16LE(this,this.l+size))!==0){oo.push(_getchar(w));size+=2;}size+=2;o=oo.join("");break;/* sbcs and dbcs support continue records in the SST way TODO codepages */case'dbcs-cont':o="";loc=this.l;for(i=0;i<size;++i){if(this.lens&&this.lens.indexOf(loc)!==-1){w=__readUInt8(this,loc);this.l=loc+1;vv=ReadShift.call(this,size-i,w?'dbcs-cont':'sbcs-cont');return oo.join("")+vv;}oo.push(_getchar(__readUInt16LE(this,loc)));loc+=2;}o=oo.join("");size*=2;break;case'cpstr':/* falls through */case'sbcs-cont':o="";loc=this.l;for(i=0;i!=size;++i){if(this.lens&&this.lens.indexOf(loc)!==-1){w=__readUInt8(this,loc);this.l=loc+1;vv=ReadShift.call(this,size-i,w?'dbcs-cont':'sbcs-cont');return oo.join("")+vv;}oo.push(_getchar(__readUInt8(this,loc)));loc+=1;}o=oo.join("");break;default:switch(size){case 1:oI=__readUInt8(this,this.l);this.l++;return oI;case 2:oI=(t==='i'?__readInt16LE:__readUInt16LE)(this,this.l);this.l+=2;return oI;case 4:case-4:if(t==='i'||(this[this.l+3]&0x80)===0){oI=(size>0?__readInt32LE:__readInt32BE)(this,this.l);this.l+=4;return oI;}else {oR=__readUInt32LE(this,this.l);this.l+=4;}return oR;case 8:case-8:if(t==='f'){if(size==8)oR=__double(this,this.l);else oR=__double([this[this.l+7],this[this.l+6],this[this.l+5],this[this.l+4],this[this.l+3],this[this.l+2],this[this.l+1],this[this.l+0]],0);this.l+=8;return oR;}else size=8;/* falls through */case 16:o=__hexlify(this,this.l,size);break;}}this.l+=size;return o;}var __writeUInt32LE=function(b/*:RawBytes|CFBlob*/,val/*:number*/,idx/*:number*/)/*:void*/{b[idx]=val&0xFF;b[idx+1]=val>>>8&0xFF;b[idx+2]=val>>>16&0xFF;b[idx+3]=val>>>24&0xFF;};var __writeInt32LE=function(b/*:RawBytes|CFBlob*/,val/*:number*/,idx/*:number*/)/*:void*/{b[idx]=val&0xFF;b[idx+1]=val>>8&0xFF;b[idx+2]=val>>16&0xFF;b[idx+3]=val>>24&0xFF;};var __writeUInt16LE=function(b/*:RawBytes|CFBlob*/,val/*:number*/,idx/*:number*/)/*:void*/{b[idx]=val&0xFF;b[idx+1]=val>>>8&0xFF;};function WriteShift(t/*:number*/,val/*:string|number*/,f/*:?string*/)/*:any*/{var size=0,i=0;if(f==='dbcs'){/*:: if(typeof val !== 'string') throw new Error("unreachable"); */for(i=0;i!=val.length;++i)__writeUInt16LE(this,val.charCodeAt(i),this.l+2*i);size=2*val.length;}else if(f==='sbcs'){{/*:: if(typeof val !== 'string') throw new Error("unreachable"); */val=val.replace(/[^\x00-\x7F]/g,"_");/*:: if(typeof val !== 'string') throw new Error("unreachable"); */for(i=0;i!=val.length;++i)this[this.l+i]=val.charCodeAt(i)&0xFF;}size=val.length;}else if(f==='hex'){for(;i<t;++i){/*:: if(typeof val !== "string") throw new Error("unreachable"); */this[this.l++]=parseInt(val.slice(2*i,2*i+2),16)||0;}return this;}else if(f==='utf16le'){/*:: if(typeof val !== "string") throw new Error("unreachable"); */var end/*:number*/=Math.min(this.l+t,this.length);for(i=0;i<Math.min(val.length,t);++i){var cc=val.charCodeAt(i);this[this.l++]=cc&0xff;this[this.l++]=cc>>8;}while(this.l<end)this[this.l++]=0;return this;}else/*:: if(typeof val === 'number') */switch(t){case 1:size=1;this[this.l]=val&0xFF;break;case 2:size=2;this[this.l]=val&0xFF;val>>>=8;this[this.l+1]=val&0xFF;break;case 3:size=3;this[this.l]=val&0xFF;val>>>=8;this[this.l+1]=val&0xFF;val>>>=8;this[this.l+2]=val&0xFF;break;case 4:size=4;__writeUInt32LE(this,val,this.l);break;case 8:size=8;if(f==='f'){write_double_le(this,val,this.l);break;}/* falls through */case 16:break;case-4:size=4;__writeInt32LE(this,val,this.l);break;}this.l+=size;return this;}function CheckField(hexstr/*:string*/,fld/*:string*/)/*:void*/{var m=__hexlify(this,this.l,hexstr.length>>1);if(m!==hexstr)throw new Error(fld+'Expected '+hexstr+' saw '+m);this.l+=hexstr.length>>1;}function prep_blob(blob,pos/*:number*/)/*:void*/{blob.l=pos;blob.read_shift=/*::(*/ReadShift/*:: :any)*/;blob.chk=CheckField;blob.write_shift=WriteShift;}function parsenoop(blob,length/*:: :number, opts?:any */){blob.l+=length;}function new_buf(sz/*:number*/)/*:Block*/{var o=new_raw_buf(sz);prep_blob(o,0);return o;}/* [MS-XLSB] 2.1.4 Record */function recordhopper(data,cb/*:RecordHopperCB*/,opts/*:?any*/){if(!data)return;var tmpbyte,cntbyte,length;prep_blob(data,data.l||0);var L=data.length,RT=0,tgt=0;while(data.l<L){RT=data.read_shift(1);if(RT&0x80)RT=(RT&0x7F)+((data.read_shift(1)&0x7F)<<7);var R=XLSBRecordEnum[RT]||XLSBRecordEnum[0xFFFF];tmpbyte=data.read_shift(1);length=tmpbyte&0x7F;for(cntbyte=1;cntbyte<4&&tmpbyte&0x80;++cntbyte)length+=((tmpbyte=data.read_shift(1))&0x7F)<<7*cntbyte;tgt=data.l+length;var d=R.f&&R.f(data,length,opts);data.l=tgt;if(cb(d,R,RT))return;}}/* control buffer usage for fixed-length buffers */function buf_array()/*:BufArray*/{var bufs/*:Array<Block>*/=[],blksz=has_buf?256:2048;var newblk=function ba_newblk(sz/*:number*/)/*:Block*/{var o/*:Block*/=new_buf(sz)/*:any*/;prep_blob(o,0);return o;};var curbuf/*:Block*/=newblk(blksz);var endbuf=function ba_endbuf(){if(!curbuf)return;if(curbuf.length>curbuf.l){curbuf=curbuf.slice(0,curbuf.l);curbuf.l=curbuf.length;}if(curbuf.length>0)bufs.push(curbuf);curbuf=null;};var next=function ba_next(sz/*:number*/)/*:Block*/{if(curbuf&&sz<curbuf.length-curbuf.l)return curbuf;endbuf();return curbuf=newblk(Math.max(sz+1,blksz));};var end=function ba_end(){endbuf();return bconcat(bufs);};var push=function ba_push(buf){endbuf();curbuf=buf;if(curbuf.l==null)curbuf.l=curbuf.length;next(blksz);};return {next:next,push:push,end:end,_bufs:bufs}/*:any*/;}/* XLS ranges enforced */function shift_cell_xls(cell/*:CellAddress*/,tgt/*:any*/,opts/*:?any*/)/*:CellAddress*/{var out=dup(cell);if(tgt.s){if(out.cRel)out.c+=tgt.s.c;if(out.rRel)out.r+=tgt.s.r;}else {if(out.cRel)out.c+=tgt.c;if(out.rRel)out.r+=tgt.r;}if(!opts||opts.biff<12){while(out.c>=0x100)out.c-=0x100;while(out.r>=0x10000)out.r-=0x10000;}return out;}function shift_range_xls(cell,range,opts){var out=dup(cell);out.s=shift_cell_xls(out.s,range.s,opts);out.e=shift_cell_xls(out.e,range.s,opts);return out;}function encode_cell_xls(c/*:CellAddress*/,biff/*:number*/)/*:string*/{if(c.cRel&&c.c<0){c=dup(c);while(c.c<0)c.c+=biff>8?0x4000:0x100;}if(c.rRel&&c.r<0){c=dup(c);while(c.r<0)c.r+=biff>8?0x100000:biff>5?0x10000:0x4000;}var s=encode_cell(c);if(!c.cRel&&c.cRel!=null)s=fix_col(s);if(!c.rRel&&c.rRel!=null)s=fix_row(s);return s;}function encode_range_xls(r,opts)/*:string*/{if(r.s.r==0&&!r.s.rRel){if(r.e.r==(opts.biff>=12?0xFFFFF:opts.biff>=8?0x10000:0x4000)&&!r.e.rRel){return (r.s.cRel?"":"$")+encode_col(r.s.c)+":"+(r.e.cRel?"":"$")+encode_col(r.e.c);}}if(r.s.c==0&&!r.s.cRel){if(r.e.c==(opts.biff>=12?0x3FFF:0xFF)&&!r.e.cRel){return (r.s.rRel?"":"$")+encode_row(r.s.r)+":"+(r.e.rRel?"":"$")+encode_row(r.e.r);}}return encode_cell_xls(r.s,opts.biff)+":"+encode_cell_xls(r.e,opts.biff);}function decode_row(rowstr/*:string*/)/*:number*/{return parseInt(unfix_row(rowstr),10)-1;}function encode_row(row/*:number*/)/*:string*/{return ""+(row+1);}function fix_row(cstr/*:string*/)/*:string*/{return cstr.replace(/([A-Z]|^)(\d+)$/,"$1$$$2");}function unfix_row(cstr/*:string*/)/*:string*/{return cstr.replace(/\$(\d+)$/,"$1");}function decode_col(colstr/*:string*/)/*:number*/{var c=unfix_col(colstr),d=0,i=0;for(;i!==c.length;++i)d=26*d+c.charCodeAt(i)-64;return d-1;}function encode_col(col/*:number*/)/*:string*/{if(col<0)throw new Error("invalid column "+col);var s="";for(++col;col;col=Math.floor((col-1)/26))s=String.fromCharCode((col-1)%26+65)+s;return s;}function fix_col(cstr/*:string*/)/*:string*/{return cstr.replace(/^([A-Z])/,"$$$1");}function unfix_col(cstr/*:string*/)/*:string*/{return cstr.replace(/^\$([A-Z])/,"$1");}function split_cell(cstr/*:string*/)/*:Array<string>*/{return cstr.replace(/(\$?[A-Z]*)(\$?\d*)/,"$1,$2").split(",");}//function decode_cell(cstr/*:string*/)/*:CellAddress*/ { var splt = split_cell(cstr); return { c:decode_col(splt[0]), r:decode_row(splt[1]) }; }
	function decode_cell(cstr/*:string*/)/*:CellAddress*/{var R=0,C=0;for(var i=0;i<cstr.length;++i){var cc=cstr.charCodeAt(i);if(cc>=48&&cc<=57)R=10*R+(cc-48);else if(cc>=65&&cc<=90)C=26*C+(cc-64);}return {c:C-1,r:R-1};}//function encode_cell(cell/*:CellAddress*/)/*:string*/ { return encode_col(cell.c) + encode_row(cell.r); }
	function encode_cell(cell/*:CellAddress*/)/*:string*/{var col=cell.c+1;var s="";for(;col;col=(col-1)/26|0)s=String.fromCharCode((col-1)%26+65)+s;return s+(cell.r+1);}function decode_range(range/*:string*/)/*:Range*/{var idx=range.indexOf(":");if(idx==-1)return {s:decode_cell(range),e:decode_cell(range)};return {s:decode_cell(range.slice(0,idx)),e:decode_cell(range.slice(idx+1))};}/*# if only one arg, it is assumed to be a Range.  If 2 args, both are cell addresses */function encode_range(cs/*:CellAddrSpec|Range*/,ce/*:?CellAddrSpec*/)/*:string*/{if(typeof ce==='undefined'||typeof ce==='number'){/*:: if(!(cs instanceof Range)) throw "unreachable"; */return encode_range(cs.s,cs.e);}/*:: if((cs instanceof Range)) throw "unreachable"; */if(typeof cs!=='string')cs=encode_cell(cs/*:any*/);if(typeof ce!=='string')ce=encode_cell(ce/*:any*/);/*:: if(typeof cs !== 'string') throw "unreachable"; */ /*:: if(typeof ce !== 'string') throw "unreachable"; */return cs==ce?cs:cs+":"+ce;}function safe_decode_range(range/*:string*/)/*:Range*/{var o={s:{c:0,r:0},e:{c:0,r:0}};var idx=0,i=0,cc=0;var len=range.length;for(idx=0;i<len;++i){if((cc=range.charCodeAt(i)-64)<1||cc>26)break;idx=26*idx+cc;}o.s.c=--idx;for(idx=0;i<len;++i){if((cc=range.charCodeAt(i)-48)<0||cc>9)break;idx=10*idx+cc;}o.s.r=--idx;if(i===len||cc!=10){o.e.c=o.s.c;o.e.r=o.s.r;return o;}++i;for(idx=0;i!=len;++i){if((cc=range.charCodeAt(i)-64)<1||cc>26)break;idx=26*idx+cc;}o.e.c=--idx;for(idx=0;i!=len;++i){if((cc=range.charCodeAt(i)-48)<0||cc>9)break;idx=10*idx+cc;}o.e.r=--idx;return o;}function safe_format_cell(cell/*:Cell*/,v/*:any*/){var q=cell.t=='d'&&v instanceof Date;if(cell.z!=null)try{return cell.w=SSF_format(cell.z,q?datenum(v):v);}catch(e){}try{return cell.w=SSF_format((cell.XF||{}).numFmtId||(q?14:0),q?datenum(v):v);}catch(e){return ''+v;}}function format_cell(cell/*:Cell*/,v/*:any*/,o/*:any*/){if(cell==null||cell.t==null||cell.t=='z')return "";if(cell.w!==undefined)return cell.w;if(cell.t=='d'&&!cell.z&&o&&o.dateNF)cell.z=o.dateNF;if(cell.t=="e")return BErr[cell.v]||cell.v;if(v==undefined)return safe_format_cell(cell,cell.v);return safe_format_cell(cell,v);}function sheet_to_workbook(sheet/*:Worksheet*/,opts)/*:Workbook*/{var n=opts&&opts.sheet?opts.sheet:"Sheet1";var sheets={};sheets[n]=sheet;return {SheetNames:[n],Sheets:sheets};}function sheet_add_aoa(_ws/*:?Worksheet*/,data/*:AOA*/,opts/*:?any*/)/*:Worksheet*/{var o=opts||{};var dense=_ws?Array.isArray(_ws):o.dense;var ws/*:Worksheet*/=_ws||(dense?[]/*:any*/:{}/*:any*/);var _R=0,_C=0;if(ws&&o.origin!=null){if(typeof o.origin=='number')_R=o.origin;else {var _origin/*:CellAddress*/=typeof o.origin=="string"?decode_cell(o.origin):o.origin;_R=_origin.r;_C=_origin.c;}if(!ws["!ref"])ws["!ref"]="A1:A1";}var range/*:Range*/={s:{c:10000000,r:10000000},e:{c:0,r:0}}/*:any*/;if(ws['!ref']){var _range=safe_decode_range(ws['!ref']);range.s.c=_range.s.c;range.s.r=_range.s.r;range.e.c=Math.max(range.e.c,_range.e.c);range.e.r=Math.max(range.e.r,_range.e.r);if(_R==-1)range.e.r=_R=_range.e.r+1;}for(var R=0;R!=data.length;++R){if(!data[R])continue;if(!Array.isArray(data[R]))throw new Error("aoa_to_sheet expects an array of arrays");for(var C=0;C!=data[R].length;++C){if(typeof data[R][C]==='undefined')continue;var cell/*:Cell*/={v:data[R][C]}/*:any*/;var __R=_R+R,__C=_C+C;if(range.s.r>__R)range.s.r=__R;if(range.s.c>__C)range.s.c=__C;if(range.e.r<__R)range.e.r=__R;if(range.e.c<__C)range.e.c=__C;if(data[R][C]&&typeof data[R][C]==='object'&&!Array.isArray(data[R][C])&&!(data[R][C]instanceof Date))cell=data[R][C];else {if(Array.isArray(cell.v)){cell.f=data[R][C][1];cell.v=cell.v[0];}if(cell.v===null){if(cell.f)cell.t='n';else if(o.nullError){cell.t='e';cell.v=0;}else if(!o.sheetStubs)continue;else cell.t='z';}else if(typeof cell.v==='number')cell.t='n';else if(typeof cell.v==='boolean')cell.t='b';else if(cell.v instanceof Date){cell.z=o.dateNF||table_fmt[14];if(o.cellDates){cell.t='d';cell.w=SSF_format(cell.z,datenum(cell.v));}else {cell.t='n';cell.v=datenum(cell.v);cell.w=SSF_format(cell.z,cell.v);}}else cell.t='s';}if(dense){if(!ws[__R])ws[__R]=[];if(ws[__R][__C]&&ws[__R][__C].z)cell.z=ws[__R][__C].z;ws[__R][__C]=cell;}else {var cell_ref=encode_cell({c:__C,r:__R}/*:any*/);if(ws[cell_ref]&&ws[cell_ref].z)cell.z=ws[cell_ref].z;ws[cell_ref]=cell;}}}if(range.s.c<10000000)ws['!ref']=encode_range(range);return ws;}function aoa_to_sheet(data/*:AOA*/,opts/*:?any*/)/*:Worksheet*/{return sheet_add_aoa(null,data,opts);}function parse_Int32LE(data){return data.read_shift(4,'i');}/* [MS-XLSB] 2.5.168 */function parse_XLWideString(data/*::, length*/)/*:string*/{var cchCharacters=data.read_shift(4);return cchCharacters===0?"":data.read_shift(cchCharacters,'dbcs');}//	var cchCharacters = data.read_shift(2);
	//	return cchCharacters === 0 ? "" : data.read_shift(cchCharacters, "utf16le");
	//}
	/* [MS-XLSB] 2.5.143 */function parse_StrRun(data){return {ich:data.read_shift(2),ifnt:data.read_shift(2)};}/* [MS-XLSB] 2.5.121 */function parse_RichStr(data,length/*:number*/)/*:XLString*/{var start=data.l;var flags=data.read_shift(1);var str=parse_XLWideString(data);var rgsStrRun=[];var z={t:str,h:str}/*:any*/;if((flags&1)!==0){/* fRichStr */ /* TODO: formatted string */var dwSizeStrRun=data.read_shift(4);for(var i=0;i!=dwSizeStrRun;++i)rgsStrRun.push(parse_StrRun(data));z.r=rgsStrRun;}else z.r=[{ich:0,ifnt:0}];//if((flags & 2) !== 0) { /* fExtStr */
	//	/* TODO: phonetic string */
	//}
	data.l=start+length;return z;}/* [MS-XLSB] 2.4.328 BrtCommentText (RichStr w/1 run) */var parse_BrtCommentText=parse_RichStr;/* [MS-XLSB] 2.5.9 */function parse_XLSBCell(data)/*:any*/{var col=data.read_shift(4);var iStyleRef=data.read_shift(2);iStyleRef+=data.read_shift(1)<<16;data.l++;//var fPhShow = data.read_shift(1);
	return {c:col,iStyleRef:iStyleRef};}/* Short XLSB Cell does not include column */function parse_XLSBShortCell(data)/*:any*/{var iStyleRef=data.read_shift(2);iStyleRef+=data.read_shift(1)<<16;data.l++;//var fPhShow = data.read_shift(1);
	return {c:-1,iStyleRef:iStyleRef};}/* [MS-XLSB] 2.5.21 */var parse_XLSBCodeName=parse_XLWideString;/* [MS-XLSB] 2.5.166 */function parse_XLNullableWideString(data/*::, length*/)/*:string*/{var cchCharacters=data.read_shift(4);return cchCharacters===0||cchCharacters===0xFFFFFFFF?"":data.read_shift(cchCharacters,'dbcs');}/* [MS-XLSB] 2.5.165 */var parse_XLNameWideString=parse_XLWideString;//var write_XLNameWideString = write_XLWideString;
	/* [MS-XLSB] 2.5.114 */var parse_RelID=parse_XLNullableWideString;/* [MS-XLS] 2.5.217 ; [MS-XLSB] 2.5.122 */function parse_RkNumber(data)/*:number*/{var b=data.slice(data.l,data.l+4);var fX100=b[0]&1,fInt=b[0]&2;data.l+=4;var RK=fInt===0?__double([0,0,0,0,b[0]&0xFC,b[1],b[2],b[3]],0):__readInt32LE(b,0)>>2;return fX100?RK/100:RK;}/* [MS-XLSB] 2.5.117 RfX */function parse_RfX(data/*::, length*/)/*:Range*/{var cell/*:Range*/={s:{},e:{}}/*:any*/;cell.s.r=data.read_shift(4);cell.e.r=data.read_shift(4);cell.s.c=data.read_shift(4);cell.e.c=data.read_shift(4);return cell;}/* [MS-XLSB] 2.5.153 UncheckedRfX */var parse_UncheckedRfX=parse_RfX;//	var cnt = data.read_shift(4);
	//	var out = [];
	//	for(var i = 0; i < cnt; ++i) {
	//		var rng = parse_UncheckedRfX(data);
	//		out.push(encode_range(rng));
	//	}
	//	return out.join(",");
	//}
	//function write_UncheckedSqRfX(sqrfx/*:string*/) {
	//	var parts = sqrfx.split(/\s*,\s*/);
	//	var o = new_buf(4); o.write_shift(4, parts.length);
	//	var out = [o];
	//	parts.forEach(function(rng) {
	//		out.push(write_UncheckedRfX(safe_decode_range(rng)));
	//	});
	//	return bconcat(out);
	//}
	/* [MS-XLS] 2.5.342 ; [MS-XLSB] 2.5.171 */ /* TODO: error checking, NaN and Infinity values are not valid Xnum */function parse_Xnum(data/*::, length*/){if(data.length-data.l<8)throw "XLS Xnum Buffer underflow";return data.read_shift(8,'f');}/* [MS-XLSB] 2.4.324 BrtColor */function parse_BrtColor(data/*::, length*/){var out={};var d=data.read_shift(1);//var fValidRGB = d & 1;
	var xColorType=d>>>1;var index=data.read_shift(1);var nTS=data.read_shift(2,'i');var bR=data.read_shift(1);var bG=data.read_shift(1);var bB=data.read_shift(1);data.l++;//var bAlpha = data.read_shift(1);
	switch(xColorType){case 0:out.auto=1;break;case 1:out.index=index;var icv=XLSIcv[index];/* automatic pseudo index 81 */if(icv)out.rgb=rgb2Hex(icv);break;case 2:/* if(!fValidRGB) throw new Error("invalid"); */out.rgb=rgb2Hex([bR,bG,bB]);break;case 3:out.theme=index;break;}if(nTS!=0)out.tint=nTS>0?nTS/32767:nTS/32768;return out;}/* [MS-XLSB] 2.5.52 */function parse_FontFlags(data/*::, length, opts*/){var d=data.read_shift(1);data.l++;var out={fBold:d&0x01,fItalic:d&0x02,fUnderline:d&0x04,fStrikeout:d&0x08,fOutline:d&0x10,fShadow:d&0x20,fCondense:d&0x40,fExtend:d&0x80};return out;}/* [MS-OLEDS] 2.3.1 and 2.3.2 */function parse_ClipboardFormatOrString(o,w/*:number*/)/*:string*/{// $FlowIgnore
	var ClipFmt={2:"BITMAP",3:"METAFILEPICT",8:"DIB",14:"ENHMETAFILE"};var m/*:number*/=o.read_shift(4);switch(m){case 0x00000000:return "";case 0xffffffff:case 0xfffffffe:return ClipFmt[o.read_shift(4)]||"";}if(m>0x190)throw new Error("Unsupported Clipboard: "+m.toString(16));o.l-=4;return o.read_shift(0,w==1?"lpstr":"lpwstr");}function parse_ClipboardFormatOrAnsiString(o){return parse_ClipboardFormatOrString(o,1);}function parse_ClipboardFormatOrUnicodeString(o){return parse_ClipboardFormatOrString(o,2);}/* [MS-OLEPS] 2.2 PropertyType */ // Note: some tree shakers cannot handle VT_VECTOR | $CONST, hence extra vars
	//var VT_EMPTY    = 0x0000;
	//var VT_NULL     = 0x0001;
	var VT_I2=0x0002;var VT_I4=0x0003;//var VT_R4       = 0x0004;
	//var VT_R8       = 0x0005;
	//var VT_CY       = 0x0006;
	//var VT_DATE     = 0x0007;
	//var VT_BSTR     = 0x0008;
	//var VT_ERROR    = 0x000A;
	var VT_BOOL=0x000B;var VT_VARIANT=0x000C;//var VT_DECIMAL  = 0x000E;
	//var VT_I1       = 0x0010;
	//var VT_UI1      = 0x0011;
	//var VT_UI2      = 0x0012;
	var VT_UI4=0x0013;//var VT_I8       = 0x0014;
	var VT_FILETIME=0x0040;var VT_BLOB=0x0041;//var VT_STREAM   = 0x0042;
	//var VT_STORAGE  = 0x0043;
	//var VT_STREAMED_Object  = 0x0044;
	//var VT_STORED_Object    = 0x0045;
	//var VT_BLOB_Object      = 0x0046;
	var VT_CF=0x0047;//var VT_CLSID    = 0x0048;
	var VT_VECTOR_VARIANT=0x100C;var VT_VECTOR_LPSTR=0x101E;//var VT_ARRAY    = 0x2000;
	var VT_STRING=0x0050;// 2.3.3.1.11 VtString
	var VT_USTR=0x0051;// 2.3.3.1.12 VtUnalignedString
	var VT_CUSTOM=[VT_STRING,VT_USTR];/* [MS-OSHARED] 2.3.3.2.2.1 Document Summary Information PIDDSI */var DocSummaryPIDDSI={/*::[*/0x01/*::]*/:{n:'CodePage',t:VT_I2},/*::[*/0x02/*::]*/:{n:'Category',t:VT_STRING},/*::[*/0x03/*::]*/:{n:'PresentationFormat',t:VT_STRING},/*::[*/0x04/*::]*/:{n:'ByteCount',t:VT_I4},/*::[*/0x05/*::]*/:{n:'LineCount',t:VT_I4},/*::[*/0x06/*::]*/:{n:'ParagraphCount',t:VT_I4},/*::[*/0x07/*::]*/:{n:'SlideCount',t:VT_I4},/*::[*/0x08/*::]*/:{n:'NoteCount',t:VT_I4},/*::[*/0x09/*::]*/:{n:'HiddenCount',t:VT_I4},/*::[*/0x0a/*::]*/:{n:'MultimediaClipCount',t:VT_I4},/*::[*/0x0b/*::]*/:{n:'ScaleCrop',t:VT_BOOL},/*::[*/0x0c/*::]*/:{n:'HeadingPairs',t:VT_VECTOR_VARIANT/* VT_VECTOR | VT_VARIANT */},/*::[*/0x0d/*::]*/:{n:'TitlesOfParts',t:VT_VECTOR_LPSTR/* VT_VECTOR | VT_LPSTR */},/*::[*/0x0e/*::]*/:{n:'Manager',t:VT_STRING},/*::[*/0x0f/*::]*/:{n:'Company',t:VT_STRING},/*::[*/0x10/*::]*/:{n:'LinksUpToDate',t:VT_BOOL},/*::[*/0x11/*::]*/:{n:'CharacterCount',t:VT_I4},/*::[*/0x13/*::]*/:{n:'SharedDoc',t:VT_BOOL},/*::[*/0x16/*::]*/:{n:'HyperlinksChanged',t:VT_BOOL},/*::[*/0x17/*::]*/:{n:'AppVersion',t:VT_I4,p:'version'},/*::[*/0x18/*::]*/:{n:'DigSig',t:VT_BLOB},/*::[*/0x1A/*::]*/:{n:'ContentType',t:VT_STRING},/*::[*/0x1B/*::]*/:{n:'ContentStatus',t:VT_STRING},/*::[*/0x1C/*::]*/:{n:'Language',t:VT_STRING},/*::[*/0x1D/*::]*/:{n:'Version',t:VT_STRING},/*::[*/0xFF/*::]*/:{},/* [MS-OLEPS] 2.18 */ /*::[*/0x80000000/*::]*/:{n:'Locale',t:VT_UI4},/*::[*/0x80000003/*::]*/:{n:'Behavior',t:VT_UI4},/*::[*/0x72627262/*::]*/:{}};/* [MS-OSHARED] 2.3.3.2.1.1 Summary Information Property Set PIDSI */var SummaryPIDSI={/*::[*/0x01/*::]*/:{n:'CodePage',t:VT_I2},/*::[*/0x02/*::]*/:{n:'Title',t:VT_STRING},/*::[*/0x03/*::]*/:{n:'Subject',t:VT_STRING},/*::[*/0x04/*::]*/:{n:'Author',t:VT_STRING},/*::[*/0x05/*::]*/:{n:'Keywords',t:VT_STRING},/*::[*/0x06/*::]*/:{n:'Comments',t:VT_STRING},/*::[*/0x07/*::]*/:{n:'Template',t:VT_STRING},/*::[*/0x08/*::]*/:{n:'LastAuthor',t:VT_STRING},/*::[*/0x09/*::]*/:{n:'RevNumber',t:VT_STRING},/*::[*/0x0A/*::]*/:{n:'EditTime',t:VT_FILETIME},/*::[*/0x0B/*::]*/:{n:'LastPrinted',t:VT_FILETIME},/*::[*/0x0C/*::]*/:{n:'CreatedDate',t:VT_FILETIME},/*::[*/0x0D/*::]*/:{n:'ModifiedDate',t:VT_FILETIME},/*::[*/0x0E/*::]*/:{n:'PageCount',t:VT_I4},/*::[*/0x0F/*::]*/:{n:'WordCount',t:VT_I4},/*::[*/0x10/*::]*/:{n:'CharCount',t:VT_I4},/*::[*/0x11/*::]*/:{n:'Thumbnail',t:VT_CF},/*::[*/0x12/*::]*/:{n:'Application',t:VT_STRING},/*::[*/0x13/*::]*/:{n:'DocSecurity',t:VT_I4},/*::[*/0xFF/*::]*/:{},/* [MS-OLEPS] 2.18 */ /*::[*/0x80000000/*::]*/:{n:'Locale',t:VT_UI4},/*::[*/0x80000003/*::]*/:{n:'Behavior',t:VT_UI4},/*::[*/0x72627262/*::]*/:{}};/* [MS-XLS] 2.4.63 Country/Region codes */var CountryEnum={/*::[*/0x0001/*::]*/:"US",// United States
	/*::[*/0x0002/*::]*/:"CA",// Canada
	/*::[*/0x0003/*::]*/:"",// Latin America (except Brazil)
	/*::[*/0x0007/*::]*/:"RU",// Russia
	/*::[*/0x0014/*::]*/:"EG",// Egypt
	/*::[*/0x001E/*::]*/:"GR",// Greece
	/*::[*/0x001F/*::]*/:"NL",// Netherlands
	/*::[*/0x0020/*::]*/:"BE",// Belgium
	/*::[*/0x0021/*::]*/:"FR",// France
	/*::[*/0x0022/*::]*/:"ES",// Spain
	/*::[*/0x0024/*::]*/:"HU",// Hungary
	/*::[*/0x0027/*::]*/:"IT",// Italy
	/*::[*/0x0029/*::]*/:"CH",// Switzerland
	/*::[*/0x002B/*::]*/:"AT",// Austria
	/*::[*/0x002C/*::]*/:"GB",// United Kingdom
	/*::[*/0x002D/*::]*/:"DK",// Denmark
	/*::[*/0x002E/*::]*/:"SE",// Sweden
	/*::[*/0x002F/*::]*/:"NO",// Norway
	/*::[*/0x0030/*::]*/:"PL",// Poland
	/*::[*/0x0031/*::]*/:"DE",// Germany
	/*::[*/0x0034/*::]*/:"MX",// Mexico
	/*::[*/0x0037/*::]*/:"BR",// Brazil
	/*::[*/0x003d/*::]*/:"AU",// Australia
	/*::[*/0x0040/*::]*/:"NZ",// New Zealand
	/*::[*/0x0042/*::]*/:"TH",// Thailand
	/*::[*/0x0051/*::]*/:"JP",// Japan
	/*::[*/0x0052/*::]*/:"KR",// Korea
	/*::[*/0x0054/*::]*/:"VN",// Viet Nam
	/*::[*/0x0056/*::]*/:"CN",// China
	/*::[*/0x005A/*::]*/:"TR",// Turkey
	/*::[*/0x0069/*::]*/:"JS",// Ramastan
	/*::[*/0x00D5/*::]*/:"DZ",// Algeria
	/*::[*/0x00D8/*::]*/:"MA",// Morocco
	/*::[*/0x00DA/*::]*/:"LY",// Libya
	/*::[*/0x015F/*::]*/:"PT",// Portugal
	/*::[*/0x0162/*::]*/:"IS",// Iceland
	/*::[*/0x0166/*::]*/:"FI",// Finland
	/*::[*/0x01A4/*::]*/:"CZ",// Czech Republic
	/*::[*/0x0376/*::]*/:"TW",// Taiwan
	/*::[*/0x03C1/*::]*/:"LB",// Lebanon
	/*::[*/0x03C2/*::]*/:"JO",// Jordan
	/*::[*/0x03C3/*::]*/:"SY",// Syria
	/*::[*/0x03C4/*::]*/:"IQ",// Iraq
	/*::[*/0x03C5/*::]*/:"KW",// Kuwait
	/*::[*/0x03C6/*::]*/:"SA",// Saudi Arabia
	/*::[*/0x03CB/*::]*/:"AE",// United Arab Emirates
	/*::[*/0x03CC/*::]*/:"IL",// Israel
	/*::[*/0x03CE/*::]*/:"QA",// Qatar
	/*::[*/0x03D5/*::]*/:"IR",// Iran
	/*::[*/0xFFFF/*::]*/:"US"// United States
	};/* [MS-XLS] 2.5.127 */var XLSFillPattern=[null,'solid','mediumGray','darkGray','lightGray','darkHorizontal','darkVertical','darkDown','darkUp','darkGrid','darkTrellis','lightHorizontal','lightVertical','lightDown','lightUp','lightGrid','lightTrellis','gray125','gray0625'];function rgbify(arr/*:Array<number>*/)/*:Array<[number, number, number]>*/{return arr.map(function(x){return [x>>16&255,x>>8&255,x&255];});}/* [MS-XLS] 2.5.161 */ /* [MS-XLSB] 2.5.75 Icv */var _XLSIcv=/*#__PURE__*/rgbify([/* Color Constants */0x000000,0xFFFFFF,0xFF0000,0x00FF00,0x0000FF,0xFFFF00,0xFF00FF,0x00FFFF,/* Overridable Defaults */0x000000,0xFFFFFF,0xFF0000,0x00FF00,0x0000FF,0xFFFF00,0xFF00FF,0x00FFFF,0x800000,0x008000,0x000080,0x808000,0x800080,0x008080,0xC0C0C0,0x808080,0x9999FF,0x993366,0xFFFFCC,0xCCFFFF,0x660066,0xFF8080,0x0066CC,0xCCCCFF,0x000080,0xFF00FF,0xFFFF00,0x00FFFF,0x800080,0x800000,0x008080,0x0000FF,0x00CCFF,0xCCFFFF,0xCCFFCC,0xFFFF99,0x99CCFF,0xFF99CC,0xCC99FF,0xFFCC99,0x3366FF,0x33CCCC,0x99CC00,0xFFCC00,0xFF9900,0xFF6600,0x666699,0x969696,0x003366,0x339966,0x003300,0x333300,0x993300,0x993366,0x333399,0x333333,/* Other entries to appease BIFF8/12 */0xFFFFFF,/* 0x40 icvForeground ?? */0x000000,/* 0x41 icvBackground ?? */0x000000,/* 0x42 icvFrame ?? */0x000000,/* 0x43 icv3D ?? */0x000000,/* 0x44 icv3DText ?? */0x000000,/* 0x45 icv3DHilite ?? */0x000000,/* 0x46 icv3DShadow ?? */0x000000,/* 0x47 icvHilite ?? */0x000000,/* 0x48 icvCtlText ?? */0x000000,/* 0x49 icvCtlScrl ?? */0x000000,/* 0x4A icvCtlInv ?? */0x000000,/* 0x4B icvCtlBody ?? */0x000000,/* 0x4C icvCtlFrame ?? */0x000000,/* 0x4D icvCtlFore ?? */0x000000,/* 0x4E icvCtlBack ?? */0x000000,/* 0x4F icvCtlNeutral */0x000000,/* 0x50 icvInfoBk ?? */0x000000/* 0x51 icvInfoText ?? */]);var XLSIcv=/*#__PURE__*/dup(_XLSIcv);/* [MS-XLSB] 2.5.97.2 */var BErr={/*::[*/0x00/*::]*/:"#NULL!",/*::[*/0x07/*::]*/:"#DIV/0!",/*::[*/0x0F/*::]*/:"#VALUE!",/*::[*/0x17/*::]*/:"#REF!",/*::[*/0x1D/*::]*/:"#NAME?",/*::[*/0x24/*::]*/:"#NUM!",/*::[*/0x2A/*::]*/:"#N/A",/*::[*/0x2B/*::]*/:"#GETTING_DATA",/*::[*/0xFF/*::]*/:"#WTF?"};//var RBErr = evert_num(BErr);
	var RBErr={"#NULL!":0x00,"#DIV/0!":0x07,"#VALUE!":0x0F,"#REF!":0x17,"#NAME?":0x1D,"#NUM!":0x24,"#N/A":0x2A,"#GETTING_DATA":0x2B,"#WTF?":0xFF};/* Parts enumerated in OPC spec, MS-XLSB and MS-XLSX */ /* 12.3 Part Summary <SpreadsheetML> */ /* 14.2 Part Summary <DrawingML> */ /* [MS-XLSX] 2.1 Part Enumerations ; [MS-XLSB] 2.1.7 Part Enumeration */var ct2type/*{[string]:string}*/={/* Workbook */"application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml":"workbooks","application/vnd.ms-excel.sheet.macroEnabled.main+xml":"workbooks","application/vnd.ms-excel.sheet.binary.macroEnabled.main":"workbooks","application/vnd.ms-excel.addin.macroEnabled.main+xml":"workbooks","application/vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml":"workbooks",/* Worksheet */"application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml":"sheets","application/vnd.ms-excel.worksheet":"sheets","application/vnd.ms-excel.binIndexWs":"TODO",/* Binary Index */ /* Chartsheet */"application/vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml":"charts","application/vnd.ms-excel.chartsheet":"charts",/* Macrosheet */"application/vnd.ms-excel.macrosheet+xml":"macros","application/vnd.ms-excel.macrosheet":"macros","application/vnd.ms-excel.intlmacrosheet":"TODO","application/vnd.ms-excel.binIndexMs":"TODO",/* Binary Index */ /* Dialogsheet */"application/vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml":"dialogs","application/vnd.ms-excel.dialogsheet":"dialogs",/* Shared Strings */"application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml":"strs","application/vnd.ms-excel.sharedStrings":"strs",/* Styles */"application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml":"styles","application/vnd.ms-excel.styles":"styles",/* File Properties */"application/vnd.openxmlformats-package.core-properties+xml":"coreprops","application/vnd.openxmlformats-officedocument.custom-properties+xml":"custprops","application/vnd.openxmlformats-officedocument.extended-properties+xml":"extprops",/* Custom Data Properties */"application/vnd.openxmlformats-officedocument.customXmlProperties+xml":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.customProperty":"TODO",/* Comments */"application/vnd.openxmlformats-officedocument.spreadsheetml.comments+xml":"comments","application/vnd.ms-excel.comments":"comments","application/vnd.ms-excel.threadedcomments+xml":"threadedcomments","application/vnd.ms-excel.person+xml":"people",/* Metadata (Stock/Geography and Dynamic Array) */"application/vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml":"metadata","application/vnd.ms-excel.sheetMetadata":"metadata",/* PivotTable */"application/vnd.ms-excel.pivotTable":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.pivotTable+xml":"TODO",/* Chart Objects */"application/vnd.openxmlformats-officedocument.drawingml.chart+xml":"TODO",/* Chart Colors */"application/vnd.ms-office.chartcolorstyle+xml":"TODO",/* Chart Style */"application/vnd.ms-office.chartstyle+xml":"TODO",/* Chart Advanced */"application/vnd.ms-office.chartex+xml":"TODO",/* Calculation Chain */"application/vnd.ms-excel.calcChain":"calcchains","application/vnd.openxmlformats-officedocument.spreadsheetml.calcChain+xml":"calcchains",/* Printer Settings */"application/vnd.openxmlformats-officedocument.spreadsheetml.printerSettings":"TODO",/* ActiveX */"application/vnd.ms-office.activeX":"TODO","application/vnd.ms-office.activeX+xml":"TODO",/* Custom Toolbars */"application/vnd.ms-excel.attachedToolbars":"TODO",/* External Data Connections */"application/vnd.ms-excel.connections":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.connections+xml":"TODO",/* External Links */"application/vnd.ms-excel.externalLink":"links","application/vnd.openxmlformats-officedocument.spreadsheetml.externalLink+xml":"links",/* PivotCache */"application/vnd.ms-excel.pivotCacheDefinition":"TODO","application/vnd.ms-excel.pivotCacheRecords":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheDefinition+xml":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheRecords+xml":"TODO",/* Query Table */"application/vnd.ms-excel.queryTable":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.queryTable+xml":"TODO",/* Shared Workbook */"application/vnd.ms-excel.userNames":"TODO","application/vnd.ms-excel.revisionHeaders":"TODO","application/vnd.ms-excel.revisionLog":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.revisionHeaders+xml":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.revisionLog+xml":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.userNames+xml":"TODO",/* Single Cell Table */"application/vnd.ms-excel.tableSingleCells":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.tableSingleCells+xml":"TODO",/* Slicer */"application/vnd.ms-excel.slicer":"TODO","application/vnd.ms-excel.slicerCache":"TODO","application/vnd.ms-excel.slicer+xml":"TODO","application/vnd.ms-excel.slicerCache+xml":"TODO",/* Sort Map */"application/vnd.ms-excel.wsSortMap":"TODO",/* Table */"application/vnd.ms-excel.table":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.table+xml":"TODO",/* Themes */"application/vnd.openxmlformats-officedocument.theme+xml":"themes",/* Theme Override */"application/vnd.openxmlformats-officedocument.themeOverride+xml":"TODO",/* Timeline */"application/vnd.ms-excel.Timeline+xml":"TODO",/* verify */"application/vnd.ms-excel.TimelineCache+xml":"TODO",/* verify */ /* VBA */"application/vnd.ms-office.vbaProject":"vba","application/vnd.ms-office.vbaProjectSignature":"TODO",/* Volatile Dependencies */"application/vnd.ms-office.volatileDependencies":"TODO","application/vnd.openxmlformats-officedocument.spreadsheetml.volatileDependencies+xml":"TODO",/* Control Properties */"application/vnd.ms-excel.controlproperties+xml":"TODO",/* Data Model */"application/vnd.openxmlformats-officedocument.model+data":"TODO",/* Survey */"application/vnd.ms-excel.Survey+xml":"TODO",/* Drawing */"application/vnd.openxmlformats-officedocument.drawing+xml":"drawings","application/vnd.openxmlformats-officedocument.drawingml.chartshapes+xml":"TODO","application/vnd.openxmlformats-officedocument.drawingml.diagramColors+xml":"TODO","application/vnd.openxmlformats-officedocument.drawingml.diagramData+xml":"TODO","application/vnd.openxmlformats-officedocument.drawingml.diagramLayout+xml":"TODO","application/vnd.openxmlformats-officedocument.drawingml.diagramStyle+xml":"TODO",/* VML */"application/vnd.openxmlformats-officedocument.vmlDrawing":"TODO","application/vnd.openxmlformats-package.relationships+xml":"rels","application/vnd.openxmlformats-officedocument.oleObject":"TODO",/* Image */"image/png":"TODO","sheet":"js"}/*:any*/;function new_ct()/*:any*/{return {workbooks:[],sheets:[],charts:[],dialogs:[],macros:[],rels:[],strs:[],comments:[],threadedcomments:[],links:[],coreprops:[],extprops:[],custprops:[],themes:[],styles:[],calcchains:[],vba:[],drawings:[],metadata:[],people:[],TODO:[],xmlns:""}/*:any*/;}function parse_ct(data/*:?string*/){var ct=new_ct();if(!data||!data.match)return ct;var ctext={};(data.match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(y[0].replace(nsregex,"<")){case'<?xml':break;case'<Types':ct.xmlns=y['xmlns'+(y[0].match(/<(\w+):/)||["",""])[1]];break;case'<Default':ctext[y.Extension]=y.ContentType;break;case'<Override':if(ct[ct2type[y.ContentType]]!==undefined)ct[ct2type[y.ContentType]].push(y.PartName);break;}});if(ct.xmlns!==XMLNS.CT)throw new Error("Unknown Namespace: "+ct.xmlns);ct.calcchain=ct.calcchains.length>0?ct.calcchains[0]:"";ct.sst=ct.strs.length>0?ct.strs[0]:"";ct.style=ct.styles.length>0?ct.styles[0]:"";ct.defaults=ctext;delete ct.calcchains;return ct;}/* 9.3 Relationships */var RELS={WB:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument",SHEET:"http://sheetjs.openxmlformats.org/officeDocument/2006/relationships/officeDocument",HLINK:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/hyperlink",VML:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/vmlDrawing",XPATH:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/externalLinkPath",XMISS:"http://schemas.microsoft.com/office/2006/relationships/xlExternalLinkPath/xlPathMissing",XLINK:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/externalLink",CXML:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/customXml",CXMLP:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/customXmlProps",CMNT:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/comments",CORE_PROPS:"http://schemas.openxmlformats.org/package/2006/relationships/metadata/core-properties",EXT_PROPS:'http://schemas.openxmlformats.org/officeDocument/2006/relationships/extended-properties',CUST_PROPS:'http://schemas.openxmlformats.org/officeDocument/2006/relationships/custom-properties',SST:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/sharedStrings",STY:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/styles",THEME:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/theme",CHART:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/chart",CHARTEX:"http://schemas.microsoft.com/office/2014/relationships/chartEx",CS:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/chartsheet",WS:["http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet","http://purl.oclc.org/ooxml/officeDocument/relationships/worksheet"],DS:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/dialogsheet",MS:"http://schemas.microsoft.com/office/2006/relationships/xlMacrosheet",IMG:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/image",DRAW:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/drawing",XLMETA:"http://schemas.openxmlformats.org/officeDocument/2006/relationships/sheetMetadata",TCMNT:"http://schemas.microsoft.com/office/2017/10/relationships/threadedComment",PEOPLE:"http://schemas.microsoft.com/office/2017/10/relationships/person",VBA:"http://schemas.microsoft.com/office/2006/relationships/vbaProject"}/*:any*/;/* 9.3.3 Representing Relationships */function get_rels_path(file/*:string*/)/*:string*/{var n=file.lastIndexOf("/");return file.slice(0,n+1)+'_rels/'+file.slice(n+1)+".rels";}function parse_rels(data/*:?string*/,currentFilePath/*:string*/){var rels={"!id":{}};if(!data)return rels;if(currentFilePath.charAt(0)!=='/'){currentFilePath='/'+currentFilePath;}var hash={};(data.match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);/* 9.3.2.2 OPC_Relationships */if(y[0]==='<Relationship'){var rel={};rel.Type=y.Type;rel.Target=y.Target;rel.Id=y.Id;if(y.TargetMode)rel.TargetMode=y.TargetMode;var canonictarget=y.TargetMode==='External'?y.Target:resolve_path(y.Target,currentFilePath);rels[canonictarget]=rel;hash[y.Id]=rel;}});rels["!id"]=hash;return rels;}/* Open Document Format for Office Applications (OpenDocument) Version 1.2 */ /* Part 3 Section 4 Manifest File */var CT_ODS="application/vnd.oasis.opendocument.spreadsheet";function parse_manifest(d,opts){var str=xlml_normalize(d);var Rn;var FEtag;while(Rn=xlmlregex.exec(str))switch(Rn[3]){case'manifest':break;// 4.2 <manifest:manifest>
	case'file-entry':// 4.3 <manifest:file-entry>
	FEtag=parsexmltag(Rn[0],false);if(FEtag.path=='/'&&FEtag.type!==CT_ODS)throw new Error("This OpenDocument is not a spreadsheet");break;case'encryption-data':// 4.4 <manifest:encryption-data>
	case'algorithm':// 4.5 <manifest:algorithm>
	case'start-key-generation':// 4.6 <manifest:start-key-generation>
	case'key-derivation':// 4.7 <manifest:key-derivation>
	throw new Error("Unsupported ODS Encryption");default:if(opts&&opts.WTF)throw Rn;}}/* ECMA-376 Part II 11.1 Core Properties Part */ /* [MS-OSHARED] 2.3.3.2.[1-2].1 (PIDSI/PIDDSI) */var CORE_PROPS/*:Array<Array<string> >*/=[["cp:category","Category"],["cp:contentStatus","ContentStatus"],["cp:keywords","Keywords"],["cp:lastModifiedBy","LastAuthor"],["cp:lastPrinted","LastPrinted"],["cp:revision","RevNumber"],["cp:version","Version"],["dc:creator","Author"],["dc:description","Comments"],["dc:identifier","Identifier"],["dc:language","Language"],["dc:subject","Subject"],["dc:title","Title"],["dcterms:created","CreatedDate",'date'],["dcterms:modified","ModifiedDate",'date']];var CORE_PROPS_REGEX/*:Array<RegExp>*/=/*#__PURE__*/function(){var r=new Array(CORE_PROPS.length);for(var i=0;i<CORE_PROPS.length;++i){var f=CORE_PROPS[i];var g="(?:"+f[0].slice(0,f[0].indexOf(":"))+":)"+f[0].slice(f[0].indexOf(":")+1);r[i]=new RegExp("<"+g+"[^>]*>([\\s\\S]*?)<\/"+g+">");}return r;}();function parse_core_props(data){var p={};data=utf8read(data);for(var i=0;i<CORE_PROPS.length;++i){var f=CORE_PROPS[i],cur=data.match(CORE_PROPS_REGEX[i]);if(cur!=null&&cur.length>0)p[f[1]]=unescapexml(cur[1]);if(f[2]==='date'&&p[f[1]])p[f[1]]=parseDate(p[f[1]]);}return p;}/* 15.2.12.3 Extended File Properties Part */ /* [MS-OSHARED] 2.3.3.2.[1-2].1 (PIDSI/PIDDSI) */var EXT_PROPS/*:Array<Array<string> >*/=[["Application","Application","string"],["AppVersion","AppVersion","string"],["Company","Company","string"],["DocSecurity","DocSecurity","string"],["Manager","Manager","string"],["HyperlinksChanged","HyperlinksChanged","bool"],["SharedDoc","SharedDoc","bool"],["LinksUpToDate","LinksUpToDate","bool"],["ScaleCrop","ScaleCrop","bool"],["HeadingPairs","HeadingPairs","raw"],["TitlesOfParts","TitlesOfParts","raw"]];function load_props_pairs(HP/*:string|Array<Array<any>>*/,TOP,props,opts){var v=[];if(typeof HP=="string")v=parseVector(HP,opts);else for(var j=0;j<HP.length;++j)v=v.concat(HP[j].map(function(hp){return {v:hp};}));var parts=typeof TOP=="string"?parseVector(TOP,opts).map(function(x){return x.v;}):TOP;var idx=0,len=0;if(parts.length>0)for(var i=0;i!==v.length;i+=2){len=+v[i+1].v;switch(v[i].v){case"Worksheets":case"工作表":case"Листы":case"أوراق العمل":case"ワークシート":case"גליונות עבודה":case"Arbeitsblätter":case"Çalışma Sayfaları":case"Feuilles de calcul":case"Fogli di lavoro":case"Folhas de cálculo":case"Planilhas":case"Regneark":case"Hojas de cálculo":case"Werkbladen":props.Worksheets=len;props.SheetNames=parts.slice(idx,idx+len);break;case"Named Ranges":case"Rangos con nombre":case"名前付き一覧":case"Benannte Bereiche":case"Navngivne områder":props.NamedRanges=len;props.DefinedNames=parts.slice(idx,idx+len);break;case"Charts":case"Diagramme":props.Chartsheets=len;props.ChartNames=parts.slice(idx,idx+len);break;}idx+=len;}}function parse_ext_props(data,p,opts){var q={};if(!p)p={};data=utf8read(data);EXT_PROPS.forEach(function(f){var xml=(data.match(matchtag(f[0]))||[])[1];switch(f[2]){case"string":if(xml)p[f[1]]=unescapexml(xml);break;case"bool":p[f[1]]=xml==="true";break;case"raw":var cur=data.match(new RegExp("<"+f[0]+"[^>]*>([\\s\\S]*?)<\/"+f[0]+">"));if(cur&&cur.length>0)q[f[1]]=cur[1];break;}});if(q.HeadingPairs&&q.TitlesOfParts)load_props_pairs(q.HeadingPairs,q.TitlesOfParts,p,opts);return p;}/* 15.2.12.2 Custom File Properties Part */var custregex=/<[^>]+>[^<]*/g;function parse_cust_props(data/*:string*/,opts){var p={},name="";var m=data.match(custregex);if(m)for(var i=0;i!=m.length;++i){var x=m[i],y=parsexmltag(x);switch(y[0]){case'<?xml':break;case'<Properties':break;case'<property':name=unescapexml(y.name);break;case'</property>':name=null;break;default:if(x.indexOf('<vt:')===0){var toks=x.split('>');var type=toks[0].slice(4),text=toks[1];/* 22.4.2.32 (CT_Variant). Omit the binary types from 22.4 (Variant Types) */switch(type){case'lpstr':case'bstr':case'lpwstr':p[name]=unescapexml(text);break;case'bool':p[name]=parsexmlbool(text);break;case'i1':case'i2':case'i4':case'i8':case'int':case'uint':p[name]=parseInt(text,10);break;case'r4':case'r8':case'decimal':p[name]=parseFloat(text);break;case'filetime':case'date':p[name]=parseDate(text);break;case'cy':case'error':p[name]=unescapexml(text);break;default:if(type.slice(-1)=='/')break;if(opts.WTF&&typeof console!=='undefined')console.warn('Unexpected',x,type,toks);}}else if(x.slice(0,2)==="</");else if(opts.WTF)throw new Error(x);}}return p;}/* Common Name -> XLML Name */var XLMLDocPropsMap={Title:'Title',Subject:'Subject',Author:'Author',Keywords:'Keywords',Comments:'Description',LastAuthor:'LastAuthor',RevNumber:'Revision',Application:'AppName',/* TotalTime: 'TotalTime', */LastPrinted:'LastPrinted',CreatedDate:'Created',ModifiedDate:'LastSaved',/* Pages */ /* Words */ /* Characters */Category:'Category',/* PresentationFormat */Manager:'Manager',Company:'Company',/* Guid */ /* HyperlinkBase */ /* Bytes */ /* Lines */ /* Paragraphs */ /* CharactersWithSpaces */AppVersion:'Version',ContentStatus:'ContentStatus',/* NOTE: missing from schema */Identifier:'Identifier',/* NOTE: missing from schema */Language:'Language'/* NOTE: missing from schema */};var evert_XLMLDPM;function xlml_set_prop(Props,tag/*:string*/,val){if(!evert_XLMLDPM)evert_XLMLDPM=evert(XLMLDocPropsMap);tag=evert_XLMLDPM[tag]||tag;Props[tag]=val;}/* [MS-DTYP] 2.3.3 FILETIME */ /* [MS-OLEDS] 2.1.3 FILETIME (Packet Version) */ /* [MS-OLEPS] 2.8 FILETIME (Packet Version) */function parse_FILETIME(blob){var dwLowDateTime=blob.read_shift(4),dwHighDateTime=blob.read_shift(4);return new Date((dwHighDateTime/1e7*Math.pow(2,32)+dwLowDateTime/1e7-11644473600)*1000).toISOString().replace(/\.000/,"");}/* [MS-OSHARED] 2.3.3.1.4 Lpstr */function parse_lpstr(blob,type,pad/*:?number*/){var start=blob.l;var str=blob.read_shift(0,'lpstr-cp');if(pad)while(blob.l-start&3)++blob.l;return str;}/* [MS-OSHARED] 2.3.3.1.6 Lpwstr */function parse_lpwstr(blob,type,pad){var str=blob.read_shift(0,'lpwstr');if(pad)blob.l+=4-(str.length+1&3)&3;return str;}/* [MS-OSHARED] 2.3.3.1.11 VtString */ /* [MS-OSHARED] 2.3.3.1.12 VtUnalignedString */function parse_VtStringBase(blob,stringType,pad){if(stringType===0x1F/*VT_LPWSTR*/)return parse_lpwstr(blob);return parse_lpstr(blob,stringType,pad);}function parse_VtString(blob,t/*:number*/,pad/*:?boolean*/){return parse_VtStringBase(blob,t,pad===false?0:4);}function parse_VtUnalignedString(blob,t/*:number*/){if(!t)throw new Error("VtUnalignedString must have positive length");return parse_VtStringBase(blob,t,0);}/* [MS-OSHARED] 2.3.3.1.7 VtVecLpwstrValue */function parse_VtVecLpwstrValue(blob)/*:Array<string>*/{var length=blob.read_shift(4);var ret/*:Array<string>*/=[];for(var i=0;i!=length;++i){var start=blob.l;ret[i]=blob.read_shift(0,'lpwstr').replace(chr0,'');if(blob.l-start&0x02)blob.l+=2;}return ret;}/* [MS-OSHARED] 2.3.3.1.9 VtVecUnalignedLpstrValue */function parse_VtVecUnalignedLpstrValue(blob)/*:Array<string>*/{var length=blob.read_shift(4);var ret/*:Array<string>*/=[];for(var i=0;i!=length;++i)ret[i]=blob.read_shift(0,'lpstr-cp').replace(chr0,'');return ret;}/* [MS-OSHARED] 2.3.3.1.13 VtHeadingPair */function parse_VtHeadingPair(blob){var start=blob.l;var headingString=parse_TypedPropertyValue(blob,VT_USTR);if(blob[blob.l]==0x00&&blob[blob.l+1]==0x00&&blob.l-start&0x02)blob.l+=2;var headerParts=parse_TypedPropertyValue(blob,VT_I4);return [headingString,headerParts];}/* [MS-OSHARED] 2.3.3.1.14 VtVecHeadingPairValue */function parse_VtVecHeadingPairValue(blob){var cElements=blob.read_shift(4);var out=[];for(var i=0;i<cElements/2;++i)out.push(parse_VtHeadingPair(blob));return out;}/* [MS-OLEPS] 2.18.1 Dictionary (uses 2.17, 2.16) */function parse_dictionary(blob,CodePage){var cnt=blob.read_shift(4);var dict/*:{[number]:string}*/={}/*:any*/;for(var j=0;j!=cnt;++j){var pid=blob.read_shift(4);var len=blob.read_shift(4);dict[pid]=blob.read_shift(len,CodePage===0x4B0?'utf16le':'utf8').replace(chr0,'').replace(chr1,'!');if(CodePage===0x4B0&&len%2)blob.l+=2;}if(blob.l&3)blob.l=blob.l>>2+1<<2;return dict;}/* [MS-OLEPS] 2.9 BLOB */function parse_BLOB(blob){var size=blob.read_shift(4);var bytes=blob.slice(blob.l,blob.l+size);blob.l+=size;if((size&3)>0)blob.l+=4-(size&3)&3;return bytes;}/* [MS-OLEPS] 2.11 ClipboardData */function parse_ClipboardData(blob){// TODO
	var o={};o.Size=blob.read_shift(4);//o.Format = blob.read_shift(4);
	blob.l+=o.Size+3-(o.Size-1)%4;return o;}/* [MS-OLEPS] 2.15 TypedPropertyValue */function parse_TypedPropertyValue(blob,type/*:number*/,_opts)/*:any*/{var t=blob.read_shift(2),ret,opts=_opts||{};blob.l+=2;if(type!==VT_VARIANT)if(t!==type&&VT_CUSTOM.indexOf(type)===-1&&!((type&0xFFFE)==0x101E&&(t&0xFFFE)==0x101E))throw new Error('Expected type '+type+' saw '+t);switch(type===VT_VARIANT?t:type){case 0x02/*VT_I2*/:ret=blob.read_shift(2,'i');if(!opts.raw)blob.l+=2;return ret;case 0x03/*VT_I4*/:ret=blob.read_shift(4,'i');return ret;case 0x0B/*VT_BOOL*/:return blob.read_shift(4)!==0x0;case 0x13/*VT_UI4*/:ret=blob.read_shift(4);return ret;case 0x1E/*VT_LPSTR*/:return parse_lpstr(blob,t,4).replace(chr0,'');case 0x1F/*VT_LPWSTR*/:return parse_lpwstr(blob);case 0x40/*VT_FILETIME*/:return parse_FILETIME(blob);case 0x41/*VT_BLOB*/:return parse_BLOB(blob);case 0x47/*VT_CF*/:return parse_ClipboardData(blob);case 0x50/*VT_STRING*/:return parse_VtString(blob,t,!opts.raw).replace(chr0,'');case 0x51/*VT_USTR*/:return parse_VtUnalignedString(blob,t/*, 4*/).replace(chr0,'');case 0x100C/*VT_VECTOR|VT_VARIANT*/:return parse_VtVecHeadingPairValue(blob);case 0x101E/*VT_VECTOR|VT_LPSTR*/:case 0x101F/*VT_VECTOR|VT_LPWSTR*/:return t==0x101F?parse_VtVecLpwstrValue(blob):parse_VtVecUnalignedLpstrValue(blob);default:throw new Error("TypedPropertyValue unrecognized type "+type+" "+t);}}/* [MS-OLEPS] 2.20 PropertySet */function parse_PropertySet(blob,PIDSI){var start_addr=blob.l;var size=blob.read_shift(4);var NumProps=blob.read_shift(4);var Props=[],i=0;var CodePage=0;var Dictionary=-1,DictObj/*:{[number]:string}*/={}/*:any*/;for(i=0;i!=NumProps;++i){var PropID=blob.read_shift(4);var Offset=blob.read_shift(4);Props[i]=[PropID,Offset+start_addr];}Props.sort(function(x,y){return x[1]-y[1];});var PropH={};for(i=0;i!=NumProps;++i){if(blob.l!==Props[i][1]){var fail=true;if(i>0&&PIDSI)switch(PIDSI[Props[i-1][0]].t){case 0x02/*VT_I2*/:if(blob.l+2===Props[i][1]){blob.l+=2;fail=false;}break;case 0x50/*VT_STRING*/:if(blob.l<=Props[i][1]){blob.l=Props[i][1];fail=false;}break;case 0x100C/*VT_VECTOR|VT_VARIANT*/:if(blob.l<=Props[i][1]){blob.l=Props[i][1];fail=false;}break;}if((!PIDSI||i==0)&&blob.l<=Props[i][1]){fail=false;blob.l=Props[i][1];}if(fail)throw new Error("Read Error: Expected address "+Props[i][1]+' at '+blob.l+' :'+i);}if(PIDSI){var piddsi=PIDSI[Props[i][0]];PropH[piddsi.n]=parse_TypedPropertyValue(blob,piddsi.t,{raw:true});if(piddsi.p==='version')PropH[piddsi.n]=String(PropH[piddsi.n]>>16)+"."+("0000"+String(PropH[piddsi.n]&0xFFFF)).slice(-4);if(piddsi.n=="CodePage")switch(PropH[piddsi.n]){case 0:PropH[piddsi.n]=1252;/* falls through */case 874:case 932:case 936:case 949:case 950:case 1250:case 1251:case 1253:case 1254:case 1255:case 1256:case 1257:case 1258:case 10000:case 1200:case 1201:case 1252:case 65000:case-536:case 65001:case-535:set_cp(CodePage=PropH[piddsi.n]>>>0&0xFFFF);break;default:throw new Error("Unsupported CodePage: "+PropH[piddsi.n]);}}else {if(Props[i][0]===0x1){CodePage=PropH.CodePage=parse_TypedPropertyValue(blob,VT_I2)/*:number*/;set_cp(CodePage);if(Dictionary!==-1){var oldpos=blob.l;blob.l=Props[Dictionary][1];DictObj=parse_dictionary(blob,CodePage);blob.l=oldpos;}}else if(Props[i][0]===0){if(CodePage===0){Dictionary=i;blob.l=Props[i+1][1];continue;}DictObj=parse_dictionary(blob,CodePage);}else {var name=DictObj[Props[i][0]];var val;/* [MS-OSHARED] 2.3.3.2.3.1.2 + PROPVARIANT */switch(blob[blob.l]){case 0x41/*VT_BLOB*/:blob.l+=4;val=parse_BLOB(blob);break;case 0x1E/*VT_LPSTR*/:blob.l+=4;val=parse_VtString(blob,blob[blob.l-4]).replace(/\u0000+$/,"");break;case 0x1F/*VT_LPWSTR*/:blob.l+=4;val=parse_VtString(blob,blob[blob.l-4]).replace(/\u0000+$/,"");break;case 0x03/*VT_I4*/:blob.l+=4;val=blob.read_shift(4,'i');break;case 0x13/*VT_UI4*/:blob.l+=4;val=blob.read_shift(4);break;case 0x05/*VT_R8*/:blob.l+=4;val=blob.read_shift(8,'f');break;case 0x0B/*VT_BOOL*/:blob.l+=4;val=parsebool(blob,4);break;case 0x40/*VT_FILETIME*/:blob.l+=4;val=parseDate(parse_FILETIME(blob));break;default:throw new Error("unparsed value: "+blob[blob.l]);}PropH[name]=val;}}}blob.l=start_addr+size;/* step ahead to skip padding */return PropH;}/* [MS-OLEPS] 2.21 PropertySetStream */function parse_PropertySetStream(file,PIDSI,clsid){var blob=file.content;if(!blob)return {}/*:any*/;prep_blob(blob,0);var NumSets,FMTID0,FMTID1,Offset0,Offset1=0;blob.chk('feff','Byte Order: ');/*var vers = */blob.read_shift(2);// TODO: check version
	var SystemIdentifier=blob.read_shift(4);var CLSID=blob.read_shift(16);if(CLSID!==CFB.utils.consts.HEADER_CLSID&&CLSID!==clsid)throw new Error("Bad PropertySet CLSID "+CLSID);NumSets=blob.read_shift(4);if(NumSets!==1&&NumSets!==2)throw new Error("Unrecognized #Sets: "+NumSets);FMTID0=blob.read_shift(16);Offset0=blob.read_shift(4);if(NumSets===1&&Offset0!==blob.l)throw new Error("Length mismatch: "+Offset0+" !== "+blob.l);else if(NumSets===2){FMTID1=blob.read_shift(16);Offset1=blob.read_shift(4);}var PSet0=parse_PropertySet(blob,PIDSI);var rval={SystemIdentifier:SystemIdentifier}/*:any*/;for(var y in PSet0)rval[y]=PSet0[y];//rval.blob = blob;
	rval.FMTID=FMTID0;//rval.PSet0 = PSet0;
	if(NumSets===1)return rval;if(Offset1-blob.l==2)blob.l+=2;if(blob.l!==Offset1)throw new Error("Length mismatch 2: "+blob.l+" !== "+Offset1);var PSet1;try{PSet1=parse_PropertySet(blob,null);}catch(e){/* empty */}for(y in PSet1)rval[y]=PSet1[y];rval.FMTID=[FMTID0,FMTID1];// TODO: verify FMTID0/1
	return rval;}function parsenoop2(blob,length){blob.read_shift(length);return null;}function parslurp(blob,length,cb){var arr=[],target=blob.l+length;while(blob.l<target)arr.push(cb(blob,target-blob.l));if(target!==blob.l)throw new Error("Slurp error");return arr;}function parsebool(blob,length/*:number*/){return blob.read_shift(length)===0x1;}function parseuint16(blob/*::, length:?number, opts:?any*/){return blob.read_shift(2,'u');}function parseuint16a(blob,length/*:: :?number, opts:?any*/){return parslurp(blob,length,parseuint16);}/* --- 2.5 Structures --- */ /* [MS-XLS] 2.5.10 Bes (boolean or error) */function parse_Bes(blob/*::, length*/){var v=blob.read_shift(1),t=blob.read_shift(1);return t===0x01?v:v===0x01;}/* [MS-XLS] 2.5.240 ShortXLUnicodeString */function parse_ShortXLUnicodeString(blob,length,opts){var cch=blob.read_shift(opts&&opts.biff>=12?2:1);var encoding='sbcs-cont';if(opts&&opts.biff>=8);if(!opts||opts.biff==8){var fHighByte=blob.read_shift(1);if(fHighByte){encoding='dbcs-cont';}}else if(opts.biff==12){encoding='wstr';}if(opts.biff>=2&&opts.biff<=5)encoding='cpstr';var o=cch?blob.read_shift(cch,encoding):"";return o;}/* 2.5.293 XLUnicodeRichExtendedString */function parse_XLUnicodeRichExtendedString(blob){var cch=blob.read_shift(2),flags=blob.read_shift(1);var/*fHighByte = flags & 0x1,*/fExtSt=flags&0x4,fRichSt=flags&0x8;var width=1+(flags&0x1);// 0x0 -> utf8, 0x1 -> dbcs
	var cRun=0,cbExtRst;var z={};if(fRichSt)cRun=blob.read_shift(2);if(fExtSt)cbExtRst=blob.read_shift(4);var encoding=width==2?'dbcs-cont':'sbcs-cont';var msg=cch===0?"":blob.read_shift(cch,encoding);if(fRichSt)blob.l+=4*cRun;//TODO: parse this
	if(fExtSt)blob.l+=cbExtRst;//TODO: parse this
	z.t=msg;if(!fRichSt){z.raw="<t>"+z.t+"</t>";z.r=z.t;}return z;}/* 2.5.296 XLUnicodeStringNoCch */function parse_XLUnicodeStringNoCch(blob,cch,opts){var retval;if(opts){if(opts.biff>=2&&opts.biff<=5)return blob.read_shift(cch,'cpstr');if(opts.biff>=12)return blob.read_shift(cch,'dbcs-cont');}var fHighByte=blob.read_shift(1);if(fHighByte===0){retval=blob.read_shift(cch,'sbcs-cont');}else {retval=blob.read_shift(cch,'dbcs-cont');}return retval;}/* 2.5.294 XLUnicodeString */function parse_XLUnicodeString(blob,length,opts){var cch=blob.read_shift(opts&&opts.biff==2?1:2);if(cch===0){blob.l++;return "";}return parse_XLUnicodeStringNoCch(blob,cch,opts);}/* BIFF5 override */function parse_XLUnicodeString2(blob,length,opts){if(opts.biff>5)return parse_XLUnicodeString(blob,length,opts);var cch=blob.read_shift(1);if(cch===0){blob.l++;return "";}return blob.read_shift(cch,opts.biff<=4||!blob.lens?'cpstr':'sbcs-cont');}/* [MS-XLS] 2.5.61 ControlInfo */function parse_ControlInfo(blob/*::, length, opts*/){var flags=blob.read_shift(1);blob.l++;var accel=blob.read_shift(2);blob.l+=2;return [flags,accel];}/* [MS-OSHARED] 2.3.7.6 URLMoniker TODO: flags */function parse_URLMoniker(blob/*::, length, opts*/){var len=blob.read_shift(4),start=blob.l;var extra=false;if(len>24){/* look ahead */blob.l+=len-24;if(blob.read_shift(16)==="795881f43b1d7f48af2c825dc4852763")extra=true;blob.l=start;}var url=blob.read_shift((extra?len-24:len)>>1,'utf16le').replace(chr0,"");if(extra)blob.l+=24;return url;}/* [MS-OSHARED] 2.3.7.8 FileMoniker TODO: all fields */function parse_FileMoniker(blob/*::, length*/){var cAnti=blob.read_shift(2);var preamble="";while(cAnti-->0)preamble+="../";var ansiPath=blob.read_shift(0,'lpstr-ansi');blob.l+=2;//var endServer = blob.read_shift(2);
	if(blob.read_shift(2)!=0xDEAD)throw new Error("Bad FileMoniker");var sz=blob.read_shift(4);if(sz===0)return preamble+ansiPath.replace(/\\/g,"/");var bytes=blob.read_shift(4);if(blob.read_shift(2)!=3)throw new Error("Bad FileMoniker");var unicodePath=blob.read_shift(bytes>>1,'utf16le').replace(chr0,"");return preamble+unicodePath;}/* [MS-OSHARED] 2.3.7.2 HyperlinkMoniker TODO: all the monikers */function parse_HyperlinkMoniker(blob,length){var clsid=blob.read_shift(16);switch(clsid){case"e0c9ea79f9bace118c8200aa004ba90b":return parse_URLMoniker(blob);case"0303000000000000c000000000000046":return parse_FileMoniker(blob);default:throw new Error("Unsupported Moniker "+clsid);}}/* [MS-OSHARED] 2.3.7.9 HyperlinkString */function parse_HyperlinkString(blob/*::, length*/){var len=blob.read_shift(4);var o=len>0?blob.read_shift(len,'utf16le').replace(chr0,""):"";return o;}/* [MS-OSHARED] 2.3.7.1 Hyperlink Object */function parse_Hyperlink(blob,length)/*:Hyperlink*/{var end=blob.l+length;var sVer=blob.read_shift(4);if(sVer!==2)throw new Error("Unrecognized streamVersion: "+sVer);var flags=blob.read_shift(2);blob.l+=2;var displayName,targetFrameName,moniker,oleMoniker,Loc="",guid,fileTime;if(flags&0x0010)displayName=parse_HyperlinkString(blob,end-blob.l);if(flags&0x0080)targetFrameName=parse_HyperlinkString(blob,end-blob.l);if((flags&0x0101)===0x0101)moniker=parse_HyperlinkString(blob,end-blob.l);if((flags&0x0101)===0x0001)oleMoniker=parse_HyperlinkMoniker(blob,end-blob.l);if(flags&0x0008)Loc=parse_HyperlinkString(blob,end-blob.l);if(flags&0x0020)guid=blob.read_shift(16);if(flags&0x0040)fileTime=parse_FILETIME(blob/*, 8*/);blob.l=end;var target=targetFrameName||moniker||oleMoniker||"";if(target&&Loc)target+="#"+Loc;if(!target)target="#"+Loc;if(flags&0x0002&&target.charAt(0)=="/"&&target.charAt(1)!="/")target="file://"+target;var out={Target:target}/*:any*/;if(guid)out.guid=guid;if(fileTime)out.time=fileTime;if(displayName)out.Tooltip=displayName;return out;}/* 2.5.178 LongRGBA */function parse_LongRGBA(blob/*::, length*/){var r=blob.read_shift(1),g=blob.read_shift(1),b=blob.read_shift(1),a=blob.read_shift(1);return [r,g,b,a];}/* 2.5.177 LongRGB */function parse_LongRGB(blob,length){var x=parse_LongRGBA(blob);x[3]=0;return x;}/* [MS-XLS] 2.5.19 */function parse_XLSCell(blob/*::, length*/)/*:Cell*/{var rw=blob.read_shift(2);// 0-indexed
	var col=blob.read_shift(2);var ixfe=blob.read_shift(2);return {r:rw,c:col,ixfe:ixfe}/*:any*/;}/* [MS-XLS] 2.5.134 */function parse_frtHeader(blob){var rt=blob.read_shift(2);var flags=blob.read_shift(2);// TODO: parse these flags
	blob.l+=8;return {type:rt,flags:flags};}function parse_OptXLUnicodeString(blob,length,opts){return length===0?"":parse_XLUnicodeString2(blob,length,opts);}/* [MS-XLS] 2.5.344 */function parse_XTI(blob,length,opts){var w=opts.biff>8?4:2;var iSupBook=blob.read_shift(w),itabFirst=blob.read_shift(w,'i'),itabLast=blob.read_shift(w,'i');return [iSupBook,itabFirst,itabLast];}/* [MS-XLS] 2.5.218 */function parse_RkRec(blob){var ixfe=blob.read_shift(2);var RK=parse_RkNumber(blob);return [ixfe,RK];}/* [MS-XLS] 2.5.1 */function parse_AddinUdf(blob,length,opts){blob.l+=4;length-=4;var l=blob.l+length;var udfName=parse_ShortXLUnicodeString(blob,length,opts);var cb=blob.read_shift(2);l-=blob.l;if(cb!==l)throw new Error("Malformed AddinUdf: padding = "+l+" != "+cb);blob.l+=cb;return udfName;}/* [MS-XLS] 2.5.209 TODO: Check sizes */function parse_Ref8U(blob/*::, length*/){var rwFirst=blob.read_shift(2);var rwLast=blob.read_shift(2);var colFirst=blob.read_shift(2);var colLast=blob.read_shift(2);return {s:{c:colFirst,r:rwFirst},e:{c:colLast,r:rwLast}};}/* [MS-XLS] 2.5.211 */function parse_RefU(blob/*::, length*/){var rwFirst=blob.read_shift(2);var rwLast=blob.read_shift(2);var colFirst=blob.read_shift(1);var colLast=blob.read_shift(1);return {s:{c:colFirst,r:rwFirst},e:{c:colLast,r:rwLast}};}/* [MS-XLS] 2.5.207 */var parse_Ref=parse_RefU;/* [MS-XLS] 2.5.143 */function parse_FtCmo(blob/*::, length*/){blob.l+=4;var ot=blob.read_shift(2);var id=blob.read_shift(2);var flags=blob.read_shift(2);blob.l+=12;return [id,ot,flags];}/* [MS-XLS] 2.5.149 */function parse_FtNts(blob){var out={};blob.l+=4;blob.l+=16;// GUID TODO
	out.fSharedNote=blob.read_shift(2);blob.l+=4;return out;}/* [MS-XLS] 2.5.142 */function parse_FtCf(blob){var out={};blob.l+=4;blob.cf=blob.read_shift(2);return out;}/* [MS-XLS] 2.5.140 - 2.5.154 and friends */function parse_FtSkip(blob){blob.l+=2;blob.l+=blob.read_shift(2);}var FtTab={/*::[*/0x00/*::]*/:parse_FtSkip,/* FtEnd */ /*::[*/0x04/*::]*/:parse_FtSkip,/* FtMacro */ /*::[*/0x05/*::]*/:parse_FtSkip,/* FtButton */ /*::[*/0x06/*::]*/:parse_FtSkip,/* FtGmo */ /*::[*/0x07/*::]*/:parse_FtCf,/* FtCf */ /*::[*/0x08/*::]*/:parse_FtSkip,/* FtPioGrbit */ /*::[*/0x09/*::]*/:parse_FtSkip,/* FtPictFmla */ /*::[*/0x0A/*::]*/:parse_FtSkip,/* FtCbls */ /*::[*/0x0B/*::]*/:parse_FtSkip,/* FtRbo */ /*::[*/0x0C/*::]*/:parse_FtSkip,/* FtSbs */ /*::[*/0x0D/*::]*/:parse_FtNts,/* FtNts */ /*::[*/0x0E/*::]*/:parse_FtSkip,/* FtSbsFmla */ /*::[*/0x0F/*::]*/:parse_FtSkip,/* FtGboData */ /*::[*/0x10/*::]*/:parse_FtSkip,/* FtEdoData */ /*::[*/0x11/*::]*/:parse_FtSkip,/* FtRboData */ /*::[*/0x12/*::]*/:parse_FtSkip,/* FtCblsData */ /*::[*/0x13/*::]*/:parse_FtSkip,/* FtLbsData */ /*::[*/0x14/*::]*/:parse_FtSkip,/* FtCblsFmla */ /*::[*/0x15/*::]*/:parse_FtCmo};function parse_FtArray(blob,length/*::, ot*/){var tgt=blob.l+length;var fts=[];while(blob.l<tgt){var ft=blob.read_shift(2);blob.l-=2;try{fts.push(FtTab[ft](blob,tgt-blob.l));}catch(e){blob.l=tgt;return fts;}}if(blob.l!=tgt)blob.l=tgt;//throw new Error("bad Object Ft-sequence");
	return fts;}/* --- 2.4 Records --- */ /* [MS-XLS] 2.4.21 */function parse_BOF(blob,length){var o={BIFFVer:0,dt:0};o.BIFFVer=blob.read_shift(2);length-=2;if(length>=2){o.dt=blob.read_shift(2);blob.l-=2;}switch(o.BIFFVer){case 0x0600:/* BIFF8 */case 0x0500:/* BIFF5 */case 0x0400:/* BIFF4 */case 0x0300:/* BIFF3 */case 0x0200:/* BIFF2 */case 0x0002:case 0x0007:/* BIFF2 */break;default:if(length>6)throw new Error("Unexpected BIFF Ver "+o.BIFFVer);}blob.read_shift(length);return o;}/* [MS-XLS] 2.4.146 */function parse_InterfaceHdr(blob,length){if(length===0)return 0x04b0;if(blob.read_shift(2)!==0x04b0);return 0x04b0;}/* [MS-XLS] 2.4.349 */function parse_WriteAccess(blob,length,opts){if(opts.enc){blob.l+=length;return "";}var l=blob.l;// TODO: make sure XLUnicodeString doesnt overrun
	var UserName=parse_XLUnicodeString2(blob,0,opts);blob.read_shift(length+l-blob.l);return UserName;}/* [MS-XLS] 2.4.351 */function parse_WsBool(blob,length,opts){var flags=opts&&opts.biff==8||length==2?blob.read_shift(2):(blob.l+=length,0);return {fDialog:flags&0x10,fBelow:flags&0x40,fRight:flags&0x80};}/* [MS-XLS] 2.4.28 */function parse_BoundSheet8(blob,length,opts){var pos=blob.read_shift(4);var hidden=blob.read_shift(1)&0x03;var dt=blob.read_shift(1);switch(dt){case 0:dt='Worksheet';break;case 1:dt='Macrosheet';break;case 2:dt='Chartsheet';break;case 6:dt='VBAModule';break;}var name=parse_ShortXLUnicodeString(blob,0,opts);if(name.length===0)name="Sheet1";return {pos:pos,hs:hidden,dt:dt,name:name};}/* [MS-XLS] 2.4.265 TODO */function parse_SST(blob,length)/*:SST*/{var end=blob.l+length;var cnt=blob.read_shift(4);var ucnt=blob.read_shift(4);var strs/*:SST*/=[]/*:any*/;for(var i=0;i!=ucnt&&blob.l<end;++i){strs.push(parse_XLUnicodeRichExtendedString(blob));}strs.Count=cnt;strs.Unique=ucnt;return strs;}/* [MS-XLS] 2.4.107 */function parse_ExtSST(blob,length){var extsst={};extsst.dsst=blob.read_shift(2);blob.l+=length-2;return extsst;}/* [MS-XLS] 2.4.221 TODO: check BIFF2-4 */function parse_Row(blob){var z={}/*:any*/;z.r=blob.read_shift(2);z.c=blob.read_shift(2);z.cnt=blob.read_shift(2)-z.c;var miyRw=blob.read_shift(2);blob.l+=4;// reserved(2), unused(2)
	var flags=blob.read_shift(1);// various flags
	blob.l+=3;// reserved(8), ixfe(12), flags(4)
	if(flags&0x07)z.level=flags&0x07;// collapsed: flags & 0x10
	if(flags&0x20)z.hidden=true;if(flags&0x40)z.hpt=miyRw/20;return z;}/* [MS-XLS] 2.4.125 */function parse_ForceFullCalculation(blob){var header=parse_frtHeader(blob);if(header.type!=0x08A3)throw new Error("Invalid Future Record "+header.type);var fullcalc=blob.read_shift(4);return fullcalc!==0x0;}/* [MS-XLS] 2.4.215 rt */function parse_RecalcId(blob){blob.read_shift(2);return blob.read_shift(4);}/* [MS-XLS] 2.4.87 */function parse_DefaultRowHeight(blob,length,opts){var f=0;if(!(opts&&opts.biff==2)){f=blob.read_shift(2);}var miyRw=blob.read_shift(2);if(opts&&opts.biff==2){f=1-(miyRw>>15);miyRw&=0x7fff;}var fl={Unsynced:f&1,DyZero:(f&2)>>1,ExAsc:(f&4)>>2,ExDsc:(f&8)>>3};return [fl,miyRw];}/* [MS-XLS] 2.4.345 TODO */function parse_Window1(blob){var xWn=blob.read_shift(2),yWn=blob.read_shift(2),dxWn=blob.read_shift(2),dyWn=blob.read_shift(2);var flags=blob.read_shift(2),iTabCur=blob.read_shift(2),iTabFirst=blob.read_shift(2);var ctabSel=blob.read_shift(2),wTabRatio=blob.read_shift(2);return {Pos:[xWn,yWn],Dim:[dxWn,dyWn],Flags:flags,CurTab:iTabCur,FirstTab:iTabFirst,Selected:ctabSel,TabRatio:wTabRatio};}/* [MS-XLS] 2.4.346 TODO */function parse_Window2(blob,length,opts){if(opts&&opts.biff>=2&&opts.biff<5)return {};var f=blob.read_shift(2);return {RTL:f&0x40};}/* [MS-XLS] 2.4.189 TODO */function/*blob, length, opts*/parse_Pane(){}/* [MS-XLS] 2.4.122 TODO */function parse_Font(blob,length,opts){var o/*:any*/={dyHeight:blob.read_shift(2),fl:blob.read_shift(2)};switch(opts&&opts.biff||8){case 2:break;case 3:case 4:blob.l+=2;break;default:blob.l+=10;break;}o.name=parse_ShortXLUnicodeString(blob,0,opts);return o;}/* [MS-XLS] 2.4.149 */function parse_LabelSst(blob){var cell=parse_XLSCell(blob);cell.isst=blob.read_shift(4);return cell;}/* [MS-XLS] 2.4.148 */function parse_Label(blob,length,opts){if(opts.biffguess&&opts.biff==2)opts.biff=5;var target=blob.l+length;var cell=parse_XLSCell(blob);if(opts.biff==2)blob.l++;var str=parse_XLUnicodeString(blob,target-blob.l,opts);cell.val=str;return cell;}/* [MS-XLS] 2.4.126 Number Formats */function parse_Format(blob,length,opts){var numFmtId=blob.read_shift(2);var fmtstr=parse_XLUnicodeString2(blob,0,opts);return [numFmtId,fmtstr];}var parse_BIFF2Format=parse_XLUnicodeString2;/* [MS-XLS] 2.4.90 */function parse_Dimensions(blob,length,opts){var end=blob.l+length;var w=opts.biff==8||!opts.biff?4:2;var r=blob.read_shift(w),R=blob.read_shift(w);var c=blob.read_shift(2),C=blob.read_shift(2);blob.l=end;return {s:{r:r,c:c},e:{r:R,c:C}};}/* [MS-XLS] 2.4.220 */function parse_RK(blob){var rw=blob.read_shift(2),col=blob.read_shift(2);var rkrec=parse_RkRec(blob);return {r:rw,c:col,ixfe:rkrec[0],rknum:rkrec[1]};}/* [MS-XLS] 2.4.175 */function parse_MulRk(blob,length){var target=blob.l+length-2;var rw=blob.read_shift(2),col=blob.read_shift(2);var rkrecs=[];while(blob.l<target)rkrecs.push(parse_RkRec(blob));if(blob.l!==target)throw new Error("MulRK read error");var lastcol=blob.read_shift(2);if(rkrecs.length!=lastcol-col+1)throw new Error("MulRK length mismatch");return {r:rw,c:col,C:lastcol,rkrec:rkrecs};}/* [MS-XLS] 2.4.174 */function parse_MulBlank(blob,length){var target=blob.l+length-2;var rw=blob.read_shift(2),col=blob.read_shift(2);var ixfes=[];while(blob.l<target)ixfes.push(blob.read_shift(2));if(blob.l!==target)throw new Error("MulBlank read error");var lastcol=blob.read_shift(2);if(ixfes.length!=lastcol-col+1)throw new Error("MulBlank length mismatch");return {r:rw,c:col,C:lastcol,ixfe:ixfes};}/* [MS-XLS] 2.5.20 2.5.249 TODO: interpret values here */function parse_CellStyleXF(blob,length,style,opts){var o={};var a=blob.read_shift(4),b=blob.read_shift(4);var c=blob.read_shift(4),d=blob.read_shift(2);o.patternType=XLSFillPattern[c>>26];if(!opts.cellStyles)return o;o.alc=a&0x07;o.fWrap=a>>3&0x01;o.alcV=a>>4&0x07;o.fJustLast=a>>7&0x01;o.trot=a>>8&0xFF;o.cIndent=a>>16&0x0F;o.fShrinkToFit=a>>20&0x01;o.iReadOrder=a>>22&0x02;o.fAtrNum=a>>26&0x01;o.fAtrFnt=a>>27&0x01;o.fAtrAlc=a>>28&0x01;o.fAtrBdr=a>>29&0x01;o.fAtrPat=a>>30&0x01;o.fAtrProt=a>>31&0x01;o.dgLeft=b&0x0F;o.dgRight=b>>4&0x0F;o.dgTop=b>>8&0x0F;o.dgBottom=b>>12&0x0F;o.icvLeft=b>>16&0x7F;o.icvRight=b>>23&0x7F;o.grbitDiag=b>>30&0x03;o.icvTop=c&0x7F;o.icvBottom=c>>7&0x7F;o.icvDiag=c>>14&0x7F;o.dgDiag=c>>21&0x0F;o.icvFore=d&0x7F;o.icvBack=d>>7&0x7F;o.fsxButton=d>>14&0x01;return o;}//function parse_CellXF(blob, length, opts) {return parse_CellStyleXF(blob,length,0, opts);}
	//function parse_StyleXF(blob, length, opts) {return parse_CellStyleXF(blob,length,1, opts);}
	/* [MS-XLS] 2.4.353 TODO: actually do this right */function parse_XF(blob,length,opts){var o={};o.ifnt=blob.read_shift(2);o.numFmtId=blob.read_shift(2);o.flags=blob.read_shift(2);o.fStyle=o.flags>>2&0x01;length-=6;o.data=parse_CellStyleXF(blob,length,o.fStyle,opts);return o;}/* [MS-XLS] 2.4.134 */function parse_Guts(blob){blob.l+=4;var out=[blob.read_shift(2),blob.read_shift(2)];if(out[0]!==0)out[0]--;if(out[1]!==0)out[1]--;if(out[0]>7||out[1]>7)throw new Error("Bad Gutters: "+out.join("|"));return out;}/* [MS-XLS] 2.4.24 */function parse_BoolErr(blob,length,opts){var cell=parse_XLSCell(blob);if(opts.biff==2||length==9)++blob.l;var val=parse_Bes(blob);cell.val=val;cell.t=val===true||val===false?'b':'e';return cell;}/* [MS-XLS] 2.4.180 Number */function parse_Number(blob,length,opts){if(opts.biffguess&&opts.biff==2)opts.biff=5;var cell=parse_XLSCell(blob);var xnum=parse_Xnum(blob);cell.val=xnum;return cell;}var parse_XLHeaderFooter=parse_OptXLUnicodeString;// TODO: parse 2.4.136
	/* [MS-XLS] 2.4.271 */function parse_SupBook(blob,length,opts){var end=blob.l+length;var ctab=blob.read_shift(2);var cch=blob.read_shift(2);opts.sbcch=cch;if(cch==0x0401||cch==0x3A01)return [cch,ctab];if(cch<0x01||cch>0xff)throw new Error("Unexpected SupBook type: "+cch);var virtPath=parse_XLUnicodeStringNoCch(blob,cch);/* TODO: 2.5.277 Virtual Path */var rgst=[];while(end>blob.l)rgst.push(parse_XLUnicodeString(blob));return [cch,ctab,virtPath,rgst];}/* [MS-XLS] 2.4.105 TODO */function parse_ExternName(blob,length,opts){var flags=blob.read_shift(2);var body;var o={fBuiltIn:flags&0x01,fWantAdvise:flags>>>1&0x01,fWantPict:flags>>>2&0x01,fOle:flags>>>3&0x01,fOleLink:flags>>>4&0x01,cf:flags>>>5&0x3FF,fIcon:flags>>>15&0x01}/*:any*/;if(opts.sbcch===0x3A01)body=parse_AddinUdf(blob,length-2,opts);//else throw new Error("unsupported SupBook cch: " + opts.sbcch);
	o.body=body||blob.read_shift(length-2);if(typeof body==="string")o.Name=body;return o;}/* [MS-XLS] 2.4.150 TODO */var XLSLblBuiltIn=["_xlnm.Consolidate_Area","_xlnm.Auto_Open","_xlnm.Auto_Close","_xlnm.Extract","_xlnm.Database","_xlnm.Criteria","_xlnm.Print_Area","_xlnm.Print_Titles","_xlnm.Recorder","_xlnm.Data_Form","_xlnm.Auto_Activate","_xlnm.Auto_Deactivate","_xlnm.Sheet_Title","_xlnm._FilterDatabase"];function parse_Lbl(blob,length,opts){var target=blob.l+length;var flags=blob.read_shift(2);var chKey=blob.read_shift(1);var cch=blob.read_shift(1);var cce=blob.read_shift(opts&&opts.biff==2?1:2);var itab=0;if(!opts||opts.biff>=5){if(opts.biff!=5)blob.l+=2;itab=blob.read_shift(2);if(opts.biff==5)blob.l+=2;blob.l+=4;}var name=parse_XLUnicodeStringNoCch(blob,cch,opts);if(flags&0x20)name=XLSLblBuiltIn[name.charCodeAt(0)];var npflen=target-blob.l;if(opts&&opts.biff==2)--npflen;/*jshint -W018 */var rgce=target==blob.l||cce===0||!(npflen>0)?[]:parse_NameParsedFormula(blob,npflen,opts,cce);/*jshint +W018 */return {chKey:chKey,Name:name,itab:itab,rgce:rgce};}/* [MS-XLS] 2.4.106 TODO: verify filename encoding */function parse_ExternSheet(blob,length,opts){if(opts.biff<8)return parse_BIFF5ExternSheet(blob,length,opts);var o=[],target=blob.l+length,len=blob.read_shift(opts.biff>8?4:2);while(len--!==0)o.push(parse_XTI(blob,opts.biff>8?12:6,opts));// [iSupBook, itabFirst, itabLast];
	if(blob.l!=target)throw new Error("Bad ExternSheet: "+blob.l+" != "+target);return o;}function parse_BIFF5ExternSheet(blob,length,opts){if(blob[blob.l+1]==0x03)blob[blob.l]++;var o=parse_ShortXLUnicodeString(blob,length,opts);return o.charCodeAt(0)==0x03?o.slice(1):o;}/* [MS-XLS] 2.4.176 TODO: check older biff */function parse_NameCmt(blob,length,opts){if(opts.biff<8){blob.l+=length;return;}var cchName=blob.read_shift(2);var cchComment=blob.read_shift(2);var name=parse_XLUnicodeStringNoCch(blob,cchName,opts);var comment=parse_XLUnicodeStringNoCch(blob,cchComment,opts);return [name,comment];}/* [MS-XLS] 2.4.260 */function parse_ShrFmla(blob,length,opts){var ref=parse_RefU(blob);blob.l++;var cUse=blob.read_shift(1);length-=8;return [parse_SharedParsedFormula(blob,length,opts),cUse,ref];}/* [MS-XLS] 2.4.4 TODO */function parse_Array(blob,length,opts){var ref=parse_Ref(blob);/* TODO: fAlwaysCalc */switch(opts.biff){case 2:blob.l++;length-=7;break;case 3:case 4:blob.l+=2;length-=8;break;default:blob.l+=6;length-=12;}return [ref,parse_ArrayParsedFormula(blob,length,opts)];}/* [MS-XLS] 2.4.173 */function parse_MTRSettings(blob){var fMTREnabled=blob.read_shift(4)!==0x00;var fUserSetThreadCount=blob.read_shift(4)!==0x00;var cUserThreadCount=blob.read_shift(4);return [fMTREnabled,fUserSetThreadCount,cUserThreadCount];}/* [MS-XLS] 2.5.186 TODO: BIFF5 */function parse_NoteSh(blob,length,opts){if(opts.biff<8)return;var row=blob.read_shift(2),col=blob.read_shift(2);var flags=blob.read_shift(2),idObj=blob.read_shift(2);var stAuthor=parse_XLUnicodeString2(blob,0,opts);if(opts.biff<8)blob.read_shift(1);return [{r:row,c:col},stAuthor,idObj,flags];}/* [MS-XLS] 2.4.179 */function parse_Note(blob,length,opts){/* TODO: Support revisions */return parse_NoteSh(blob,length,opts);}/* [MS-XLS] 2.4.168 */function parse_MergeCells(blob,length)/*:Array<Range>*/{var merges/*:Array<Range>*/=[];var cmcs=blob.read_shift(2);while(cmcs--)merges.push(parse_Ref8U(blob));return merges;}/* [MS-XLS] 2.4.181 TODO: parse all the things! */function parse_Obj(blob,length,opts){if(opts&&opts.biff<8)return parse_BIFF5Obj(blob,length,opts);var cmo=parse_FtCmo(blob);// id, ot, flags
	var fts=parse_FtArray(blob,length-22,cmo[1]);return {cmo:cmo,ft:fts};}/* from older spec */var parse_BIFF5OT={0x08:function(blob,length){var tgt=blob.l+length;blob.l+=10;// todo
	var cf=blob.read_shift(2);blob.l+=4;blob.l+=2;//var cbPictFmla = blob.read_shift(2);
	blob.l+=2;blob.l+=2;//var grbit = blob.read_shift(2);
	blob.l+=4;var cchName=blob.read_shift(1);blob.l+=cchName;// TODO: stName
	blob.l=tgt;// TODO: fmla
	return {fmt:cf};}};function parse_BIFF5Obj(blob,length,opts){blob.l+=4;//var cnt = blob.read_shift(4);
	var ot=blob.read_shift(2);var id=blob.read_shift(2);var grbit=blob.read_shift(2);blob.l+=2;//var colL = blob.read_shift(2);
	blob.l+=2;//var dxL = blob.read_shift(2);
	blob.l+=2;//var rwT = blob.read_shift(2);
	blob.l+=2;//var dyT = blob.read_shift(2);
	blob.l+=2;//var colR = blob.read_shift(2);
	blob.l+=2;//var dxR = blob.read_shift(2);
	blob.l+=2;//var rwB = blob.read_shift(2);
	blob.l+=2;//var dyB = blob.read_shift(2);
	blob.l+=2;//var cbMacro = blob.read_shift(2);
	blob.l+=6;length-=36;var fts=[];fts.push((parse_BIFF5OT[ot]||parsenoop)(blob,length,opts));return {cmo:[id,ot,grbit],ft:fts};}/* [MS-XLS] 2.4.329 TODO: parse properly */function parse_TxO(blob,length,opts){var s=blob.l;var texts="";try{blob.l+=4;var ot=(opts.lastobj||{cmo:[0,0]}).cmo[1];var controlInfo;// eslint-disable-line no-unused-vars
	if([0,5,7,11,12,14].indexOf(ot)==-1)blob.l+=6;else controlInfo=parse_ControlInfo(blob,6,opts);// eslint-disable-line no-unused-vars
	var cchText=blob.read_shift(2);/*var cbRuns = */blob.read_shift(2);/*var ifntEmpty = */parseuint16(blob,2);var len=blob.read_shift(2);blob.l+=len;//var fmla = parse_ObjFmla(blob, s + length - blob.l);
	for(var i=1;i<blob.lens.length-1;++i){if(blob.l-s!=blob.lens[i])throw new Error("TxO: bad continue record");var hdr=blob[blob.l];var t=parse_XLUnicodeStringNoCch(blob,blob.lens[i+1]-blob.lens[i]-1);texts+=t;if(texts.length>=(hdr?cchText:2*cchText))break;}if(texts.length!==cchText&&texts.length!==cchText*2){throw new Error("cchText: "+cchText+" != "+texts.length);}blob.l=s+length;/* [MS-XLS] 2.5.272 TxORuns */ //	var rgTxoRuns = [];
	//	for(var j = 0; j != cbRuns/8-1; ++j) blob.l += 8;
	//	var cchText2 = blob.read_shift(2);
	//	if(cchText2 !== cchText) throw new Error("TxOLastRun mismatch: " + cchText2 + " " + cchText);
	//	blob.l += 6;
	//	if(s + length != blob.l) throw new Error("TxO " + (s + length) + ", at " + blob.l);
	return {t:texts};}catch(e){blob.l=s+length;return {t:texts};}}/* [MS-XLS] 2.4.140 */function parse_HLink(blob,length){var ref=parse_Ref8U(blob);blob.l+=16;/* CLSID */var hlink=parse_Hyperlink(blob,length-24);return [ref,hlink];}/* [MS-XLS] 2.4.141 */function parse_HLinkTooltip(blob,length){blob.read_shift(2);var ref=parse_Ref8U(blob);var wzTooltip=blob.read_shift((length-10)/2,'dbcs-cont');wzTooltip=wzTooltip.replace(chr0,"");return [ref,wzTooltip];}/* [MS-XLS] 2.4.63 */function parse_Country(blob)/*:[string|number, string|number]*/{var o=[0,0],d;d=blob.read_shift(2);o[0]=CountryEnum[d]||d;d=blob.read_shift(2);o[1]=CountryEnum[d]||d;return o;}/* [MS-XLS] 2.4.50 ClrtClient */function parse_ClrtClient(blob){var ccv=blob.read_shift(2);var o=[];while(ccv-->0)o.push(parse_LongRGB(blob));return o;}/* [MS-XLS] 2.4.188 */function parse_Palette(blob){var ccv=blob.read_shift(2);var o=[];while(ccv-->0)o.push(parse_LongRGB(blob));return o;}/* [MS-XLS] 2.4.354 */function parse_XFCRC(blob){blob.l+=2;var o={cxfs:0,crc:0};o.cxfs=blob.read_shift(2);o.crc=blob.read_shift(4);return o;}/* [MS-XLS] 2.4.53 TODO: parse flags */ /* [MS-XLSB] 2.4.323 TODO: parse flags */function parse_ColInfo(blob,length,opts){if(!opts.cellStyles)return parsenoop(blob,length);var w=opts&&opts.biff>=12?4:2;var colFirst=blob.read_shift(w);var colLast=blob.read_shift(w);var coldx=blob.read_shift(w);var ixfe=blob.read_shift(w);var flags=blob.read_shift(2);if(w==2)blob.l+=2;var o={s:colFirst,e:colLast,w:coldx,ixfe:ixfe,flags:flags}/*:any*/;if(opts.biff>=5||!opts.biff)o.level=flags>>8&0x7;return o;}/* [MS-XLS] 2.4.257 */function parse_Setup(blob,length){var o={};if(length<32)return o;blob.l+=16;o.header=parse_Xnum(blob);o.footer=parse_Xnum(blob);blob.l+=2;return o;}/* [MS-XLS] 2.4.261 */function parse_ShtProps(blob,length,opts){var def={area:false};if(opts.biff!=5){blob.l+=length;return def;}var d=blob.read_shift(1);blob.l+=3;if(d&0x10)def.area=true;return def;}var parse_Blank=parse_XLSCell;/* [MS-XLS] 2.4.20 Just the cell */var parse_Scl=parseuint16a;/* [MS-XLS] 2.4.247 num, den */var parse_String=parse_XLUnicodeString;/* [MS-XLS] 2.4.268 */ /* --- Specific to versions before BIFF8 --- */function parse_ImData(blob){var cf=blob.read_shift(2);var env=blob.read_shift(2);var lcb=blob.read_shift(4);var o={fmt:cf,env:env,len:lcb,data:blob.slice(blob.l,blob.l+lcb)};blob.l+=lcb;return o;}/* BIFF2_??? where ??? is the name from [XLS] */function parse_BIFF2STR(blob,length,opts){if(opts.biffguess&&opts.biff==5)opts.biff=2;var cell=parse_XLSCell(blob);++blob.l;var str=parse_XLUnicodeString2(blob,length-7,opts);cell.t='str';cell.val=str;return cell;}function parse_BIFF2NUM(blob/*::, length*/){var cell=parse_XLSCell(blob);++blob.l;var num=parse_Xnum(blob);cell.t='n';cell.val=num;return cell;}function parse_BIFF2INT(blob){var cell=parse_XLSCell(blob);++blob.l;var num=blob.read_shift(2);cell.t='n';cell.val=num;return cell;}function parse_BIFF2STRING(blob){var cch=blob.read_shift(1);if(cch===0){blob.l++;return "";}return blob.read_shift(cch,'sbcs-cont');}/* TODO: convert to BIFF8 font struct */function parse_BIFF2FONTXTRA(blob,length){blob.l+=6;// unknown
	blob.l+=2;// font weight "bls"
	blob.l+=1;// charset
	blob.l+=3;// unknown
	blob.l+=1;// font family
	blob.l+=length-13;}/* TODO: parse rich text runs */function parse_RString(blob,length,opts){var end=blob.l+length;var cell=parse_XLSCell(blob);var cch=blob.read_shift(2);var str=parse_XLUnicodeStringNoCch(blob,cch,opts);blob.l=end;cell.t='str';cell.val=str;return cell;}/* from js-harb (C) 2014-present  SheetJS */var DBF_SUPPORTED_VERSIONS=[0x02,0x03,0x30,0x31,0x83,0x8B,0x8C,0xF5];var DBF=/*#__PURE__*/function(){var dbf_codepage_map={/* Code Pages Supported by Visual FoxPro */ /*::[*/0x01/*::]*/:437,/*::[*/0x02/*::]*/:850,/*::[*/0x03/*::]*/:1252,/*::[*/0x04/*::]*/:10000,/*::[*/0x64/*::]*/:852,/*::[*/0x65/*::]*/:866,/*::[*/0x66/*::]*/:865,/*::[*/0x67/*::]*/:861,/*::[*/0x68/*::]*/:895,/*::[*/0x69/*::]*/:620,/*::[*/0x6A/*::]*/:737,/*::[*/0x6B/*::]*/:857,/*::[*/0x78/*::]*/:950,/*::[*/0x79/*::]*/:949,/*::[*/0x7A/*::]*/:936,/*::[*/0x7B/*::]*/:932,/*::[*/0x7C/*::]*/:874,/*::[*/0x7D/*::]*/:1255,/*::[*/0x7E/*::]*/:1256,/*::[*/0x96/*::]*/:10007,/*::[*/0x97/*::]*/:10029,/*::[*/0x98/*::]*/:10006,/*::[*/0xC8/*::]*/:1250,/*::[*/0xC9/*::]*/:1251,/*::[*/0xCA/*::]*/:1254,/*::[*/0xCB/*::]*/:1253,/* shapefile DBF extension */ /*::[*/0x00/*::]*/:20127,/*::[*/0x08/*::]*/:865,/*::[*/0x09/*::]*/:437,/*::[*/0x0A/*::]*/:850,/*::[*/0x0B/*::]*/:437,/*::[*/0x0D/*::]*/:437,/*::[*/0x0E/*::]*/:850,/*::[*/0x0F/*::]*/:437,/*::[*/0x10/*::]*/:850,/*::[*/0x11/*::]*/:437,/*::[*/0x12/*::]*/:850,/*::[*/0x13/*::]*/:932,/*::[*/0x14/*::]*/:850,/*::[*/0x15/*::]*/:437,/*::[*/0x16/*::]*/:850,/*::[*/0x17/*::]*/:865,/*::[*/0x18/*::]*/:437,/*::[*/0x19/*::]*/:437,/*::[*/0x1A/*::]*/:850,/*::[*/0x1B/*::]*/:437,/*::[*/0x1C/*::]*/:863,/*::[*/0x1D/*::]*/:850,/*::[*/0x1F/*::]*/:852,/*::[*/0x22/*::]*/:852,/*::[*/0x23/*::]*/:852,/*::[*/0x24/*::]*/:860,/*::[*/0x25/*::]*/:850,/*::[*/0x26/*::]*/:866,/*::[*/0x37/*::]*/:850,/*::[*/0x40/*::]*/:852,/*::[*/0x4D/*::]*/:936,/*::[*/0x4E/*::]*/:949,/*::[*/0x4F/*::]*/:950,/*::[*/0x50/*::]*/:874,/*::[*/0x57/*::]*/:1252,/*::[*/0x58/*::]*/:1252,/*::[*/0x59/*::]*/:1252,/*::[*/0x6C/*::]*/:863,/*::[*/0x86/*::]*/:737,/*::[*/0x87/*::]*/:852,/*::[*/0x88/*::]*/:857,/*::[*/0xCC/*::]*/:1257,/*::[*/0xFF/*::]*/:16969};var dbf_reverse_map=evert({/*::[*/0x01/*::]*/:437,/*::[*/0x02/*::]*/:850,/*::[*/0x03/*::]*/:1252,/*::[*/0x04/*::]*/:10000,/*::[*/0x64/*::]*/:852,/*::[*/0x65/*::]*/:866,/*::[*/0x66/*::]*/:865,/*::[*/0x67/*::]*/:861,/*::[*/0x68/*::]*/:895,/*::[*/0x69/*::]*/:620,/*::[*/0x6A/*::]*/:737,/*::[*/0x6B/*::]*/:857,/*::[*/0x78/*::]*/:950,/*::[*/0x79/*::]*/:949,/*::[*/0x7A/*::]*/:936,/*::[*/0x7B/*::]*/:932,/*::[*/0x7C/*::]*/:874,/*::[*/0x7D/*::]*/:1255,/*::[*/0x7E/*::]*/:1256,/*::[*/0x96/*::]*/:10007,/*::[*/0x97/*::]*/:10029,/*::[*/0x98/*::]*/:10006,/*::[*/0xC8/*::]*/:1250,/*::[*/0xC9/*::]*/:1251,/*::[*/0xCA/*::]*/:1254,/*::[*/0xCB/*::]*/:1253,/*::[*/0x00/*::]*/:20127});/* TODO: find an actual specification */function dbf_to_aoa(buf,opts)/*:AOA*/{var out/*:AOA*/=[];var d/*:Block*/=new_raw_buf(1)/*:any*/;switch(opts.type){case'base64':d=s2a(Base64_decode(buf));break;case'binary':d=s2a(buf);break;case'buffer':case'array':d=buf;break;}prep_blob(d,0);/* header */var ft=d.read_shift(1);var memo=!!(ft&0x88);var vfp=false,l7=false;switch(ft){case 0x02:break;// dBASE II
	case 0x03:break;// dBASE III
	case 0x30:vfp=true;memo=true;break;// VFP
	case 0x31:vfp=true;memo=true;break;// VFP with autoincrement
	// 0x43 dBASE IV SQL table files
	// 0x63 dBASE IV SQL system files
	case 0x83:break;// dBASE III with memo
	case 0x8B:break;// dBASE IV with memo
	case 0x8C:l7=true;break;// dBASE Level 7 with memo
	// case 0xCB dBASE IV SQL table files with memo
	case 0xF5:break;// FoxPro 2.x with memo
	// case 0xFB FoxBASE
	default:throw new Error("DBF Unsupported Version: "+ft.toString(16));}var nrow=0,fpos=0x0209;if(ft==0x02)nrow=d.read_shift(2);d.l+=3;// dBASE II stores DDMMYY date, others use YYMMDD
	if(ft!=0x02)nrow=d.read_shift(4);if(nrow>1048576)nrow=1e6;if(ft!=0x02)fpos=d.read_shift(2);// header length
	var rlen=d.read_shift(2);// record length
	var/*flags = 0,*/current_cp=opts.codepage||1252;if(ft!=0x02){// 20 reserved bytes
	d.l+=16;/*flags = */d.read_shift(1);//if(memo && ((flags & 0x02) === 0)) throw new Error("DBF Flags " + flags.toString(16) + " ft " + ft.toString(16));
	/* codepage present in FoxPro and dBASE Level 7 */if(d[d.l]!==0)current_cp=dbf_codepage_map[d[d.l]];d.l+=1;d.l+=2;}if(l7)d.l+=36;// Level 7: 32 byte "Language driver name", 4 byte reserved
	/*:: type DBFField = { name:string; len:number; type:string; } */var fields/*:Array<DBFField>*/=[],field/*:DBFField*/={}/*:any*/;var hend=Math.min(d.length,ft==0x02?0x209:fpos-10-(vfp?264:0));var ww=l7?32:11;while(d.l<hend&&d[d.l]!=0x0d){field={}/*:any*/;field.name=$cptable.utils.decode(current_cp,d.slice(d.l,d.l+ww)).replace(/[\u0000\r\n].*$/g,"");d.l+=ww;field.type=String.fromCharCode(d.read_shift(1));if(ft!=0x02&&!l7)field.offset=d.read_shift(4);field.len=d.read_shift(1);if(ft==0x02)field.offset=d.read_shift(2);field.dec=d.read_shift(1);if(field.name.length)fields.push(field);if(ft!=0x02)d.l+=l7?13:14;switch(field.type){case'B':// Double (VFP) / Binary (dBASE L7)
	if((!vfp||field.len!=8)&&opts.WTF)console.log('Skipping '+field.name+':'+field.type);break;case'G':// General (FoxPro and dBASE L7)
	case'P':// Picture (FoxPro and dBASE L7)
	if(opts.WTF)console.log('Skipping '+field.name+':'+field.type);break;case'+':// Autoincrement (dBASE L7 only)
	case'0':// _NullFlags (VFP only)
	case'@':// Timestamp (dBASE L7 only)
	case'C':// Character (dBASE II)
	case'D':// Date (dBASE III)
	case'F':// Float (dBASE IV)
	case'I':// Long (VFP and dBASE L7)
	case'L':// Logical (dBASE II)
	case'M':// Memo (dBASE III)
	case'N':// Number (dBASE II)
	case'O':// Double (dBASE L7 only)
	case'T':// Datetime (VFP only)
	case'Y':// Currency (VFP only)
	break;default:throw new Error('Unknown Field Type: '+field.type);}}if(d[d.l]!==0x0D)d.l=fpos-1;if(d.read_shift(1)!==0x0D)throw new Error("DBF Terminator not found "+d.l+" "+d[d.l]);d.l=fpos;/* data */var R=0,C=0;out[0]=[];for(C=0;C!=fields.length;++C)out[0][C]=fields[C].name;while(nrow-->0){if(d[d.l]===0x2A){// TODO: record marked as deleted -- create a hidden row?
	d.l+=rlen;continue;}++d.l;out[++R]=[];C=0;for(C=0;C!=fields.length;++C){var dd=d.slice(d.l,d.l+fields[C].len);d.l+=fields[C].len;prep_blob(dd,0);var s=$cptable.utils.decode(current_cp,dd);switch(fields[C].type){case'C':// NOTE: it is conventional to write '  /  /  ' for empty dates
	if(s.trim().length)out[R][C]=s.replace(/\s+$/,"");break;case'D':if(s.length===8)out[R][C]=new Date(+s.slice(0,4),+s.slice(4,6)-1,+s.slice(6,8));else out[R][C]=s;break;case'F':out[R][C]=parseFloat(s.trim());break;case'+':case'I':out[R][C]=l7?dd.read_shift(-4,'i')^0x80000000:dd.read_shift(4,'i');break;case'L':switch(s.trim().toUpperCase()){case'Y':case'T':out[R][C]=true;break;case'N':case'F':out[R][C]=false;break;case'':case'?':break;default:throw new Error("DBF Unrecognized L:|"+s+"|");}break;case'M':/* TODO: handle memo files */if(!memo)throw new Error("DBF Unexpected MEMO for type "+ft.toString(16));out[R][C]="##MEMO##"+(l7?parseInt(s.trim(),10):dd.read_shift(4));break;case'N':s=s.replace(/\u0000/g,"").trim();// NOTE: dBASE II interprets "  .  " as 0
	if(s&&s!=".")out[R][C]=+s||0;break;case'@':// NOTE: dBASE specs appear to be incorrect
	out[R][C]=new Date(dd.read_shift(-8,'f')-0x388317533400);break;case'T':out[R][C]=new Date((dd.read_shift(4)-0x253D8C)*0x5265C00+dd.read_shift(4));break;case'Y':out[R][C]=dd.read_shift(4,'i')/1e4+dd.read_shift(4,'i')/1e4*Math.pow(2,32);break;case'O':out[R][C]=-dd.read_shift(-8,'f');break;case'B':if(vfp&&fields[C].len==8){out[R][C]=dd.read_shift(8,'f');break;}/* falls through */case'G':case'P':dd.l+=fields[C].len;break;case'0':if(fields[C].name==='_NullFlags')break;/* falls through */default:throw new Error("DBF Unsupported data type "+fields[C].type);}}}if(ft!=0x02)if(d.l<d.length&&d[d.l++]!=0x1A)throw new Error("DBF EOF Marker missing "+(d.l-1)+" of "+d.length+" "+d[d.l-1].toString(16));if(opts&&opts.sheetRows)out=out.slice(0,opts.sheetRows);opts.DBF=fields;return out;}function dbf_to_sheet(buf,opts)/*:Worksheet*/{var o=opts||{};if(!o.dateNF)o.dateNF="yyyymmdd";var ws=aoa_to_sheet(dbf_to_aoa(buf,o),o);ws["!cols"]=o.DBF.map(function(field){return {wch:field.len,DBF:field};});delete o.DBF;return ws;}function dbf_to_workbook(buf,opts)/*:Workbook*/{try{return sheet_to_workbook(dbf_to_sheet(buf,opts),opts);}catch(e){if(opts&&opts.WTF)throw e;}return {SheetNames:[],Sheets:{}};}var _RLEN={'B':8,'C':250,'L':1,'D':8,'?':0,'':0};function sheet_to_dbf(ws/*:Worksheet*/,opts/*:WriteOpts*/){var o=opts||{};if(+o.codepage>=0)set_cp(+o.codepage);if(o.type=="string")throw new Error("Cannot write DBF to JS string");var ba=buf_array();var aoa/*:AOA*/=sheet_to_json(ws,{header:1,raw:true,cellDates:true});var headers=aoa[0],data=aoa.slice(1),cols=ws["!cols"]||[];var i=0,j=0,hcnt=0,rlen=1;for(i=0;i<headers.length;++i){if(((cols[i]||{}).DBF||{}).name){headers[i]=cols[i].DBF.name;++hcnt;continue;}if(headers[i]==null)continue;++hcnt;if(typeof headers[i]==='number')headers[i]=headers[i].toString(10);if(typeof headers[i]!=='string')throw new Error("DBF Invalid column name "+headers[i]+" |"+typeof headers[i]+"|");if(headers.indexOf(headers[i])!==i)for(j=0;j<1024;++j)if(headers.indexOf(headers[i]+"_"+j)==-1){headers[i]+="_"+j;break;}}var range=safe_decode_range(ws['!ref']);var coltypes/*:Array<string>*/=[];var colwidths/*:Array<number>*/=[];var coldecimals/*:Array<number>*/=[];for(i=0;i<=range.e.c-range.s.c;++i){var guess='',_guess='',maxlen=0;var col/*:Array<any>*/=[];for(j=0;j<data.length;++j){if(data[j][i]!=null)col.push(data[j][i]);}if(col.length==0||headers[i]==null){coltypes[i]='?';continue;}for(j=0;j<col.length;++j){switch(typeof col[j]){/* TODO: check if L2 compat is desired */case'number':_guess='B';break;case'string':_guess='C';break;case'boolean':_guess='L';break;case'object':_guess=col[j]instanceof Date?'D':'C';break;default:_guess='C';}maxlen=Math.max(maxlen,String(col[j]).length);guess=guess&&guess!=_guess?'C':_guess;//if(guess == 'C') break;
	}if(maxlen>250)maxlen=250;_guess=((cols[i]||{}).DBF||{}).type;/* TODO: more fine grained control over DBF type resolution */if(_guess=='C'){if(cols[i].DBF.len>maxlen)maxlen=cols[i].DBF.len;}if(guess=='B'&&_guess=='N'){guess='N';coldecimals[i]=cols[i].DBF.dec;maxlen=cols[i].DBF.len;}colwidths[i]=guess=='C'||_guess=='N'?maxlen:_RLEN[guess]||0;rlen+=colwidths[i];coltypes[i]=guess;}var h=ba.next(32);h.write_shift(4,0x13021130);h.write_shift(4,data.length);h.write_shift(2,296+32*hcnt);h.write_shift(2,rlen);for(i=0;i<4;++i)h.write_shift(4,0);h.write_shift(4,0x00000000|(+dbf_reverse_map[/*::String(*/current_ansi/*::)*/]||0x03)<<8);for(i=0,j=0;i<headers.length;++i){if(headers[i]==null)continue;var hf=ba.next(32);var _f=(headers[i].slice(-10)+"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00").slice(0,11);hf.write_shift(1,_f,"sbcs");hf.write_shift(1,coltypes[i]=='?'?'C':coltypes[i],"sbcs");hf.write_shift(4,j);hf.write_shift(1,colwidths[i]||_RLEN[coltypes[i]]||0);hf.write_shift(1,coldecimals[i]||0);hf.write_shift(1,0x02);hf.write_shift(4,0);hf.write_shift(1,0);hf.write_shift(4,0);hf.write_shift(4,0);j+=colwidths[i]||_RLEN[coltypes[i]]||0;}var hb=ba.next(264);hb.write_shift(4,0x0000000D);for(i=0;i<65;++i)hb.write_shift(4,0x00000000);for(i=0;i<data.length;++i){var rout=ba.next(rlen);rout.write_shift(1,0);for(j=0;j<headers.length;++j){if(headers[j]==null)continue;switch(coltypes[j]){case'L':rout.write_shift(1,data[i][j]==null?0x3F:data[i][j]?0x54:0x46);break;case'B':rout.write_shift(8,data[i][j]||0,'f');break;case'N':var _n="0";if(typeof data[i][j]=="number")_n=data[i][j].toFixed(coldecimals[j]||0);for(hcnt=0;hcnt<colwidths[j]-_n.length;++hcnt)rout.write_shift(1,0x20);rout.write_shift(1,_n,"sbcs");break;case'D':if(!data[i][j])rout.write_shift(8,"00000000","sbcs");else {rout.write_shift(4,("0000"+data[i][j].getFullYear()).slice(-4),"sbcs");rout.write_shift(2,("00"+(data[i][j].getMonth()+1)).slice(-2),"sbcs");rout.write_shift(2,("00"+data[i][j].getDate()).slice(-2),"sbcs");}break;case'C':var _s=String(data[i][j]!=null?data[i][j]:"").slice(0,colwidths[j]);rout.write_shift(1,_s,"sbcs");for(hcnt=0;hcnt<colwidths[j]-_s.length;++hcnt)rout.write_shift(1,0x20);break;}}// data
	}ba.next(1).write_shift(1,0x1A);return ba.end();}return {to_workbook:dbf_to_workbook,to_sheet:dbf_to_sheet,from_sheet:sheet_to_dbf};}();var SYLK=/*#__PURE__*/function(){/* TODO: stress test sequences */var sylk_escapes={AA:'À',BA:'Á',CA:'Â',DA:195,HA:'Ä',JA:197,AE:'È',BE:'É',CE:'Ê',HE:'Ë',AI:'Ì',BI:'Í',CI:'Î',HI:'Ï',AO:'Ò',BO:'Ó',CO:'Ô',DO:213,HO:'Ö',AU:'Ù',BU:'Ú',CU:'Û',HU:'Ü',Aa:'à',Ba:'á',Ca:'â',Da:227,Ha:'ä',Ja:229,Ae:'è',Be:'é',Ce:'ê',He:'ë',Ai:'ì',Bi:'í',Ci:'î',Hi:'ï',Ao:'ò',Bo:'ó',Co:'ô',Do:245,Ho:'ö',Au:'ù',Bu:'ú',Cu:'û',Hu:'ü',KC:'Ç',Kc:'ç',q:'æ',z:'œ',a:'Æ',j:'Œ',DN:209,Dn:241,Hy:255,S:169,c:170,R:174,"B ":180,/*::[*/0/*::]*/:176,/*::[*/1/*::]*/:177,/*::[*/2/*::]*/:178,/*::[*/3/*::]*/:179,/*::[*/5/*::]*/:181,/*::[*/6/*::]*/:182,/*::[*/7/*::]*/:183,Q:185,k:186,b:208,i:216,l:222,s:240,y:248,"!":161,'"':162,"#":163,"(":164,"%":165,"'":167,"H ":168,"+":171,";":187,"<":188,"=":189,">":190,"?":191,"{":223}/*:any*/;var sylk_char_regex=new RegExp("\u001BN("+keys(sylk_escapes).join("|").replace(/\|\|\|/,"|\\||").replace(/([?()+])/g,"\\$1")+"|\\|)","gm");var sylk_char_fn=function(_,$1){var o=sylk_escapes[$1];return typeof o=="number"?_getansi(o):o;};var decode_sylk_char=function($$,$1,$2){var newcc=$1.charCodeAt(0)-0x20<<4|$2.charCodeAt(0)-0x30;return newcc==59?$$:_getansi(newcc);};sylk_escapes["|"]=254;/* TODO: find an actual specification */function sylk_to_aoa(d/*:RawData*/,opts)/*:[AOA, Worksheet]*/{switch(opts.type){case'base64':return sylk_to_aoa_str(Base64_decode(d),opts);case'binary':return sylk_to_aoa_str(d,opts);case'buffer':return sylk_to_aoa_str(has_buf&&Buffer.isBuffer(d)?d.toString('binary'):a2s(d),opts);case'array':return sylk_to_aoa_str(cc2str(d),opts);}throw new Error("Unrecognized type "+opts.type);}function sylk_to_aoa_str(str/*:string*/,opts)/*:[AOA, Worksheet]*/{var records=str.split(/[\n\r]+/),R=-1,C=-1,ri=0,rj=0,arr/*:AOA*/=[];var formats/*:Array<string>*/=[];var next_cell_format/*:string|null*/=null;var sht={},rowinfo/*:Array<RowInfo>*/=[],colinfo/*:Array<ColInfo>*/=[],cw/*:Array<string>*/=[];var Mval=0,j;if(+opts.codepage>=0)set_cp(+opts.codepage);for(;ri!==records.length;++ri){Mval=0;var rstr=records[ri].trim().replace(/\x1B([\x20-\x2F])([\x30-\x3F])/g,decode_sylk_char).replace(sylk_char_regex,sylk_char_fn);var record=rstr.replace(/;;/g,"\u0000").split(";").map(function(x){return x.replace(/\u0000/g,";");});var RT=record[0],val;if(rstr.length>0)switch(RT){case'ID':break;/* header */case'E':break;/* EOF */case'B':break;/* dimensions */case'O':break;/* options? */case'W':break;/* window? */case'P':if(record[1].charAt(0)=='P')formats.push(rstr.slice(3).replace(/;;/g,";"));break;case'C':var C_seen_K=false,C_seen_X=false,C_seen_S=false,C_seen_E=false,_R=-1,_C=-1;for(rj=1;rj<record.length;++rj)switch(record[rj].charAt(0)){case'A':break;// TODO: comment
	case'X':C=parseInt(record[rj].slice(1))-1;C_seen_X=true;break;case'Y':R=parseInt(record[rj].slice(1))-1;if(!C_seen_X)C=0;for(j=arr.length;j<=R;++j)arr[j]=[];break;case'K':val=record[rj].slice(1);if(val.charAt(0)==='"')val=val.slice(1,val.length-1);else if(val==='TRUE')val=true;else if(val==='FALSE')val=false;else if(!isNaN(fuzzynum(val))){val=fuzzynum(val);if(next_cell_format!==null&&fmt_is_date(next_cell_format))val=numdate(val);}else if(!isNaN(fuzzydate(val).getDate())){val=parseDate(val);}C_seen_K=true;break;case'E':C_seen_E=true;var formula=rc_to_a1(record[rj].slice(1),{r:R,c:C});arr[R][C]=[arr[R][C],formula];break;case'S':C_seen_S=true;arr[R][C]=[arr[R][C],"S5S"];break;case'G':break;// unknown
	case'R':_R=parseInt(record[rj].slice(1))-1;break;case'C':_C=parseInt(record[rj].slice(1))-1;break;default:if(opts&&opts.WTF)throw new Error("SYLK bad record "+rstr);}if(C_seen_K){if(arr[R][C]&&arr[R][C].length==2)arr[R][C][0]=val;else arr[R][C]=val;next_cell_format=null;}if(C_seen_S){if(C_seen_E)throw new Error("SYLK shared formula cannot have own formula");var shrbase=_R>-1&&arr[_R][_C];if(!shrbase||!shrbase[1])throw new Error("SYLK shared formula cannot find base");arr[R][C][1]=shift_formula_str(shrbase[1],{r:R-_R,c:C-_C});}break;case'F':var F_seen=0;for(rj=1;rj<record.length;++rj)switch(record[rj].charAt(0)){case'X':C=parseInt(record[rj].slice(1))-1;++F_seen;break;case'Y':R=parseInt(record[rj].slice(1))-1;/*C = 0;*/for(j=arr.length;j<=R;++j)arr[j]=[];break;case'M':Mval=parseInt(record[rj].slice(1))/20;break;case'F':break;/* ??? */case'G':break;/* hide grid */case'P':next_cell_format=formats[parseInt(record[rj].slice(1))];break;case'S':break;/* cell style */case'D':break;/* column */case'N':break;/* font */case'W':cw=record[rj].slice(1).split(" ");for(j=parseInt(cw[0],10);j<=parseInt(cw[1],10);++j){Mval=parseInt(cw[2],10);colinfo[j-1]=Mval===0?{hidden:true}:{wch:Mval};process_col(colinfo[j-1]);}break;case'C':/* default column format */C=parseInt(record[rj].slice(1))-1;if(!colinfo[C])colinfo[C]={};break;case'R':/* row properties */R=parseInt(record[rj].slice(1))-1;if(!rowinfo[R])rowinfo[R]={};if(Mval>0){rowinfo[R].hpt=Mval;rowinfo[R].hpx=pt2px(Mval);}else if(Mval===0)rowinfo[R].hidden=true;break;default:if(opts&&opts.WTF)throw new Error("SYLK bad record "+rstr);}if(F_seen<1)next_cell_format=null;break;default:if(opts&&opts.WTF)throw new Error("SYLK bad record "+rstr);}}if(rowinfo.length>0)sht['!rows']=rowinfo;if(colinfo.length>0)sht['!cols']=colinfo;if(opts&&opts.sheetRows)arr=arr.slice(0,opts.sheetRows);return [arr,sht];}function sylk_to_sheet(d/*:RawData*/,opts)/*:Worksheet*/{var aoasht=sylk_to_aoa(d,opts);var aoa=aoasht[0],ws=aoasht[1];var o=aoa_to_sheet(aoa,opts);keys(ws).forEach(function(k){o[k]=ws[k];});return o;}function sylk_to_workbook(d/*:RawData*/,opts)/*:Workbook*/{return sheet_to_workbook(sylk_to_sheet(d,opts),opts);}function write_ws_cell_sylk(cell/*:Cell*/,ws/*:Worksheet*/,R/*:number*/,C/*:number*/ /*::, opts*/)/*:string*/{var o="C;Y"+(R+1)+";X"+(C+1)+";K";switch(cell.t){case'n':o+=cell.v||0;if(cell.f&&!cell.F)o+=";E"+a1_to_rc(cell.f,{r:R,c:C});break;case'b':o+=cell.v?"TRUE":"FALSE";break;case'e':o+=cell.w||cell.v;break;case'd':o+='"'+(cell.w||cell.v)+'"';break;case's':o+='"'+cell.v.replace(/"/g,"").replace(/;/g,";;")+'"';break;}return o;}function write_ws_cols_sylk(out,cols){cols.forEach(function(col,i){var rec="F;W"+(i+1)+" "+(i+1)+" ";if(col.hidden)rec+="0";else {if(typeof col.width=='number'&&!col.wpx)col.wpx=width2px(col.width);if(typeof col.wpx=='number'&&!col.wch)col.wch=px2char(col.wpx);if(typeof col.wch=='number')rec+=Math.round(col.wch);}if(rec.charAt(rec.length-1)!=" ")out.push(rec);});}function write_ws_rows_sylk(out/*:Array<string>*/,rows/*:Array<RowInfo>*/){rows.forEach(function(row,i){var rec="F;";if(row.hidden)rec+="M0;";else if(row.hpt)rec+="M"+20*row.hpt+";";else if(row.hpx)rec+="M"+20*px2pt(row.hpx)+";";if(rec.length>2)out.push(rec+"R"+(i+1));});}function sheet_to_sylk(ws/*:Worksheet*/,opts/*:?any*/)/*:string*/{var preamble/*:Array<string>*/=["ID;PWXL;N;E"],o/*:Array<string>*/=[];var r=safe_decode_range(ws['!ref']),cell/*:Cell*/;var dense=Array.isArray(ws);var RS="\r\n";preamble.push("P;PGeneral");preamble.push("F;P0;DG0G8;M255");if(ws['!cols'])write_ws_cols_sylk(preamble,ws['!cols']);if(ws['!rows'])write_ws_rows_sylk(preamble,ws['!rows']);preamble.push("B;Y"+(r.e.r-r.s.r+1)+";X"+(r.e.c-r.s.c+1)+";D"+[r.s.c,r.s.r,r.e.c,r.e.r].join(" "));for(var R=r.s.r;R<=r.e.r;++R){for(var C=r.s.c;C<=r.e.c;++C){var coord=encode_cell({r:R,c:C});cell=dense?(ws[R]||[])[C]:ws[coord];if(!cell||cell.v==null&&(!cell.f||cell.F))continue;o.push(write_ws_cell_sylk(cell,ws,R,C));}}return preamble.join(RS)+RS+o.join(RS)+RS+"E"+RS;}return {to_workbook:sylk_to_workbook,to_sheet:sylk_to_sheet,from_sheet:sheet_to_sylk};}();var DIF=/*#__PURE__*/function(){function dif_to_aoa(d/*:RawData*/,opts)/*:AOA*/{switch(opts.type){case'base64':return dif_to_aoa_str(Base64_decode(d),opts);case'binary':return dif_to_aoa_str(d,opts);case'buffer':return dif_to_aoa_str(has_buf&&Buffer.isBuffer(d)?d.toString('binary'):a2s(d),opts);case'array':return dif_to_aoa_str(cc2str(d),opts);}throw new Error("Unrecognized type "+opts.type);}function dif_to_aoa_str(str/*:string*/,opts)/*:AOA*/{var records=str.split('\n'),R=-1,C=-1,ri=0,arr/*:AOA*/=[];for(;ri!==records.length;++ri){if(records[ri].trim()==='BOT'){arr[++R]=[];C=0;continue;}if(R<0)continue;var metadata=records[ri].trim().split(",");var type=metadata[0],value=metadata[1];++ri;var data=records[ri]||"";while((data.match(/["]/g)||[]).length&1&&ri<records.length-1)data+="\n"+records[++ri];data=data.trim();switch(+type){case-1:if(data==='BOT'){arr[++R]=[];C=0;continue;}else if(data!=='EOD')throw new Error("Unrecognized DIF special command "+data);break;case 0:if(data==='TRUE')arr[R][C]=true;else if(data==='FALSE')arr[R][C]=false;else if(!isNaN(fuzzynum(value)))arr[R][C]=fuzzynum(value);else if(!isNaN(fuzzydate(value).getDate()))arr[R][C]=parseDate(value);else arr[R][C]=value;++C;break;case 1:data=data.slice(1,data.length-1);data=data.replace(/""/g,'"');if(data&&data.match(/^=".*"$/))data=data.slice(2,-1);arr[R][C++]=data!==''?data:null;break;}if(data==='EOD')break;}if(opts&&opts.sheetRows)arr=arr.slice(0,opts.sheetRows);return arr;}function dif_to_sheet(str/*:string*/,opts)/*:Worksheet*/{return aoa_to_sheet(dif_to_aoa(str,opts),opts);}function dif_to_workbook(str/*:string*/,opts)/*:Workbook*/{return sheet_to_workbook(dif_to_sheet(str,opts),opts);}var sheet_to_dif=/*#__PURE__*/function(){var push_field=function pf(o/*:Array<string>*/,topic/*:string*/,v/*:number*/,n/*:number*/,s/*:string*/){o.push(topic);o.push(v+","+n);o.push('"'+s.replace(/"/g,'""')+'"');};var push_value=function po(o/*:Array<string>*/,type/*:number*/,v/*:any*/,s/*:string*/){o.push(type+","+v);o.push(type==1?'"'+s.replace(/"/g,'""')+'"':s);};return function sheet_to_dif(ws/*:Worksheet*/ /*::, opts:?any*/)/*:string*/{var o/*:Array<string>*/=[];var r=safe_decode_range(ws['!ref']),cell/*:Cell*/;var dense=Array.isArray(ws);push_field(o,"TABLE",0,1,"sheetjs");push_field(o,"VECTORS",0,r.e.r-r.s.r+1,"");push_field(o,"TUPLES",0,r.e.c-r.s.c+1,"");push_field(o,"DATA",0,0,"");for(var R=r.s.r;R<=r.e.r;++R){push_value(o,-1,0,"BOT");for(var C=r.s.c;C<=r.e.c;++C){var coord=encode_cell({r:R,c:C});cell=dense?(ws[R]||[])[C]:ws[coord];if(!cell){push_value(o,1,0,"");continue;}switch(cell.t){case'n':var val=cell.w;if(!val&&cell.v!=null)val=cell.v;if(val==null){if(cell.f&&!cell.F)push_value(o,1,0,"="+cell.f);else push_value(o,1,0,"");}else push_value(o,0,val,"V");break;case'b':push_value(o,0,cell.v?1:0,cell.v?"TRUE":"FALSE");break;case's':push_value(o,1,0,isNaN(cell.v)?cell.v:'="'+cell.v+'"');break;case'd':if(!cell.w)cell.w=SSF_format(cell.z||table_fmt[14],datenum(parseDate(cell.v)));push_value(o,0,cell.w,"V");break;default:push_value(o,1,0,"");}}}push_value(o,-1,0,"EOD");var RS="\r\n";var oo=o.join(RS);//while((oo.length & 0x7F) != 0) oo += "\0";
	return oo;};}();return {to_workbook:dif_to_workbook,to_sheet:dif_to_sheet,from_sheet:sheet_to_dif};}();var ETH=/*#__PURE__*/function(){function decode(s/*:string*/)/*:string*/{return s.replace(/\\b/g,"\\").replace(/\\c/g,":").replace(/\\n/g,"\n");}function encode(s/*:string*/)/*:string*/{return s.replace(/\\/g,"\\b").replace(/:/g,"\\c").replace(/\n/g,"\\n");}function eth_to_aoa(str/*:string*/,opts)/*:AOA*/{var records=str.split('\n'),R=-1,C=-1,ri=0,arr/*:AOA*/=[];for(;ri!==records.length;++ri){var record=records[ri].trim().split(":");if(record[0]!=='cell')continue;var addr=decode_cell(record[1]);if(arr.length<=addr.r)for(R=arr.length;R<=addr.r;++R)if(!arr[R])arr[R]=[];R=addr.r;C=addr.c;switch(record[2]){case't':arr[R][C]=decode(record[3]);break;case'v':arr[R][C]=+record[3];break;case'vtf':var _f=record[record.length-1];/* falls through */case'vtc':switch(record[3]){case'nl':arr[R][C]=+record[4]?true:false;break;default:arr[R][C]=+record[4];break;}if(record[2]=='vtf')arr[R][C]=[arr[R][C],_f];}}if(opts&&opts.sheetRows)arr=arr.slice(0,opts.sheetRows);return arr;}function eth_to_sheet(d/*:string*/,opts)/*:Worksheet*/{return aoa_to_sheet(eth_to_aoa(d,opts),opts);}function eth_to_workbook(d/*:string*/,opts)/*:Workbook*/{return sheet_to_workbook(eth_to_sheet(d,opts),opts);}var header=["socialcalc:version:1.5","MIME-Version: 1.0","Content-Type: multipart/mixed; boundary=SocialCalcSpreadsheetControlSave"].join("\n");var sep=["--SocialCalcSpreadsheetControlSave","Content-type: text/plain; charset=UTF-8"].join("\n")+"\n";/* TODO: the other parts */var meta=["# SocialCalc Spreadsheet Control Save","part:sheet"].join("\n");var end="--SocialCalcSpreadsheetControlSave--";function sheet_to_eth_data(ws/*:Worksheet*/)/*:string*/{if(!ws||!ws['!ref'])return "";var o/*:Array<string>*/=[],oo/*:Array<string>*/=[],cell,coord="";var r=decode_range(ws['!ref']);var dense=Array.isArray(ws);for(var R=r.s.r;R<=r.e.r;++R){for(var C=r.s.c;C<=r.e.c;++C){coord=encode_cell({r:R,c:C});cell=dense?(ws[R]||[])[C]:ws[coord];if(!cell||cell.v==null||cell.t==='z')continue;oo=["cell",coord,'t'];switch(cell.t){case's':case'str':oo.push(encode(cell.v));break;case'n':if(!cell.f){oo[2]='v';oo[3]=cell.v;}else {oo[2]='vtf';oo[3]='n';oo[4]=cell.v;oo[5]=encode(cell.f);}break;case'b':oo[2]='vt'+(cell.f?'f':'c');oo[3]='nl';oo[4]=cell.v?"1":"0";oo[5]=encode(cell.f||(cell.v?'TRUE':'FALSE'));break;case'd':var t=datenum(parseDate(cell.v));oo[2]='vtc';oo[3]='nd';oo[4]=""+t;oo[5]=cell.w||SSF_format(cell.z||table_fmt[14],t);break;case'e':continue;}o.push(oo.join(":"));}}o.push("sheet:c:"+(r.e.c-r.s.c+1)+":r:"+(r.e.r-r.s.r+1)+":tvf:1");o.push("valueformat:1:text-wiki");//o.push("copiedfrom:" + ws['!ref']); // clipboard only
	return o.join("\n");}function sheet_to_eth(ws/*:Worksheet*/ /*::, opts:?any*/)/*:string*/{return [header,sep,meta,sep,sheet_to_eth_data(ws),end].join("\n");// return ["version:1.5", sheet_to_eth_data(ws)].join("\n"); // clipboard form
	}return {to_workbook:eth_to_workbook,to_sheet:eth_to_sheet,from_sheet:sheet_to_eth};}();var PRN=/*#__PURE__*/function(){function set_text_arr(data/*:string*/,arr/*:AOA*/,R/*:number*/,C/*:number*/,o/*:any*/){if(o.raw)arr[R][C]=data;else if(data==="");else if(data==='TRUE')arr[R][C]=true;else if(data==='FALSE')arr[R][C]=false;else if(!isNaN(fuzzynum(data)))arr[R][C]=fuzzynum(data);else if(!isNaN(fuzzydate(data).getDate()))arr[R][C]=parseDate(data);else arr[R][C]=data;}function prn_to_aoa_str(f/*:string*/,opts)/*:AOA*/{var o=opts||{};var arr/*:AOA*/=[]/*:any*/;if(!f||f.length===0)return arr;var lines=f.split(/[\r\n]/);var L=lines.length-1;while(L>=0&&lines[L].length===0)--L;var start=10,idx=0;var R=0;for(;R<=L;++R){idx=lines[R].indexOf(" ");if(idx==-1)idx=lines[R].length;else idx++;start=Math.max(start,idx);}for(R=0;R<=L;++R){arr[R]=[];/* TODO: confirm that widths are always 10 */var C=0;set_text_arr(lines[R].slice(0,start).trim(),arr,R,C,o);for(C=1;C<=(lines[R].length-start)/10+1;++C)set_text_arr(lines[R].slice(start+(C-1)*10,start+C*10).trim(),arr,R,C,o);}if(o.sheetRows)arr=arr.slice(0,o.sheetRows);return arr;}// List of accepted CSV separators
	var guess_seps={/*::[*/0x2C/*::]*/:',',/*::[*/0x09/*::]*/:"\t",/*::[*/0x3B/*::]*/:';',/*::[*/0x7C/*::]*/:'|'};// CSV separator weights to be used in case of equal numbers
	var guess_sep_weights={/*::[*/0x2C/*::]*/:3,/*::[*/0x09/*::]*/:2,/*::[*/0x3B/*::]*/:1,/*::[*/0x7C/*::]*/:0};function guess_sep(str){var cnt={},instr=false,end=0,cc=0;for(;end<str.length;++end){if((cc=str.charCodeAt(end))==0x22)instr=!instr;else if(!instr&&cc in guess_seps)cnt[cc]=(cnt[cc]||0)+1;}cc=[];for(end in cnt)if(Object.prototype.hasOwnProperty.call(cnt,end)){cc.push([cnt[end],end]);}if(!cc.length){cnt=guess_sep_weights;for(end in cnt)if(Object.prototype.hasOwnProperty.call(cnt,end)){cc.push([cnt[end],end]);}}cc.sort(function(a,b){return a[0]-b[0]||guess_sep_weights[a[1]]-guess_sep_weights[b[1]];});return guess_seps[cc.pop()[1]]||0x2C;}function dsv_to_sheet_str(str/*:string*/,opts)/*:Worksheet*/{var o=opts||{};var sep="";var ws/*:Worksheet*/=o.dense?[]/*:any*/:{}/*:any*/;var range/*:Range*/={s:{c:0,r:0},e:{c:0,r:0}}/*:any*/;if(str.slice(0,4)=="sep="){// If the line ends in \r\n
	if(str.charCodeAt(5)==13&&str.charCodeAt(6)==10){sep=str.charAt(4);str=str.slice(7);}// If line ends in \r OR \n
	else if(str.charCodeAt(5)==13||str.charCodeAt(5)==10){sep=str.charAt(4);str=str.slice(6);}else sep=guess_sep(str.slice(0,1024));}else if(o&&o.FS)sep=o.FS;else sep=guess_sep(str.slice(0,1024));var R=0,C=0,v=0;var start=0,end=0,sepcc=sep.charCodeAt(0),instr=false,cc=0,startcc=str.charCodeAt(0);str=str.replace(/\r\n/mg,"\n");var _re/*:?RegExp*/=o.dateNF!=null?dateNF_regex(o.dateNF):null;function finish_cell(){var s=str.slice(start,end);var cell={}/*:any*/;if(s.charAt(0)=='"'&&s.charAt(s.length-1)=='"')s=s.slice(1,-1).replace(/""/g,'"');if(s.length===0)cell.t='z';else if(o.raw){cell.t='s';cell.v=s;}else if(s.trim().length===0){cell.t='s';cell.v=s;}else if(s.charCodeAt(0)==0x3D){if(s.charCodeAt(1)==0x22&&s.charCodeAt(s.length-1)==0x22){cell.t='s';cell.v=s.slice(2,-1).replace(/""/g,'"');}else if(fuzzyfmla(s)){cell.t='n';cell.f=s.slice(1);}else {cell.t='s';cell.v=s;}}else if(s=="TRUE"){cell.t='b';cell.v=true;}else if(s=="FALSE"){cell.t='b';cell.v=false;}else if(!isNaN(v=fuzzynum(s))){cell.t='n';if(o.cellText!==false)cell.w=s;cell.v=v;}else if(!isNaN(fuzzydate(s).getDate())||_re&&s.match(_re)){cell.z=o.dateNF||table_fmt[14];var k=0;if(_re&&s.match(_re)){s=dateNF_fix(s,o.dateNF,s.match(_re)||[]);k=1;}if(o.cellDates){cell.t='d';cell.v=parseDate(s,k);}else {cell.t='n';cell.v=datenum(parseDate(s,k));}if(o.cellText!==false)cell.w=SSF_format(cell.z,cell.v instanceof Date?datenum(cell.v):cell.v);if(!o.cellNF)delete cell.z;}else {cell.t='s';cell.v=s;}if(cell.t=='z');else if(o.dense){if(!ws[R])ws[R]=[];ws[R][C]=cell;}else ws[encode_cell({c:C,r:R})]=cell;start=end+1;startcc=str.charCodeAt(start);if(range.e.c<C)range.e.c=C;if(range.e.r<R)range.e.r=R;if(cc==sepcc)++C;else {C=0;++R;if(o.sheetRows&&o.sheetRows<=R)return true;}}outer:for(;end<str.length;++end)switch(cc=str.charCodeAt(end)){case 0x22:if(startcc===0x22)instr=!instr;break;case sepcc:case 0x0a:case 0x0d:if(!instr&&finish_cell())break outer;break;}if(end-start>0)finish_cell();ws['!ref']=encode_range(range);return ws;}function prn_to_sheet_str(str/*:string*/,opts)/*:Worksheet*/{if(!(opts&&opts.PRN))return dsv_to_sheet_str(str,opts);if(opts.FS)return dsv_to_sheet_str(str,opts);if(str.slice(0,4)=="sep=")return dsv_to_sheet_str(str,opts);if(str.indexOf("\t")>=0||str.indexOf(",")>=0||str.indexOf(";")>=0)return dsv_to_sheet_str(str,opts);return aoa_to_sheet(prn_to_aoa_str(str,opts),opts);}function prn_to_sheet(d/*:RawData*/,opts)/*:Worksheet*/{var str="",bytes=opts.type=='string'?[0,0,0,0]:firstbyte(d,opts);switch(opts.type){case'base64':str=Base64_decode(d);break;case'binary':str=d;break;case'buffer':if(opts.codepage==65001)str=d.toString('utf8');// TODO: test if buf
	else if(opts.codepage&&typeof $cptable!=='undefined')str=$cptable.utils.decode(opts.codepage,d);else str=has_buf&&Buffer.isBuffer(d)?d.toString('binary'):a2s(d);break;case'array':str=cc2str(d);break;case'string':str=d;break;default:throw new Error("Unrecognized type "+opts.type);}if(bytes[0]==0xEF&&bytes[1]==0xBB&&bytes[2]==0xBF)str=utf8read(str.slice(3));else if(opts.type!='string'&&opts.type!='buffer'&&opts.codepage==65001)str=utf8read(str);else if(opts.type=='binary'&&typeof $cptable!=='undefined'&&opts.codepage)str=$cptable.utils.decode(opts.codepage,$cptable.utils.encode(28591,str));if(str.slice(0,19)=="socialcalc:version:")return ETH.to_sheet(opts.type=='string'?str:utf8read(str),opts);return prn_to_sheet_str(str,opts);}function prn_to_workbook(d/*:RawData*/,opts)/*:Workbook*/{return sheet_to_workbook(prn_to_sheet(d,opts),opts);}function sheet_to_prn(ws/*:Worksheet*/ /*::, opts:?any*/)/*:string*/{var o/*:Array<string>*/=[];var r=safe_decode_range(ws['!ref']),cell/*:Cell*/;var dense=Array.isArray(ws);for(var R=r.s.r;R<=r.e.r;++R){var oo/*:Array<string>*/=[];for(var C=r.s.c;C<=r.e.c;++C){var coord=encode_cell({r:R,c:C});cell=dense?(ws[R]||[])[C]:ws[coord];if(!cell||cell.v==null){oo.push("          ");continue;}var w=(cell.w||(format_cell(cell),cell.w)||"").slice(0,10);while(w.length<10)w+=" ";oo.push(w+(C===0?" ":""));}o.push(oo.join(""));}return o.join("\n");}return {to_workbook:prn_to_workbook,to_sheet:prn_to_sheet,from_sheet:sheet_to_prn};}();/* Excel defaults to SYLK but warns if data is not valid */function read_wb_ID(d,opts){var o=opts||{},OLD_WTF=!!o.WTF;o.WTF=true;try{var out=SYLK.to_workbook(d,o);o.WTF=OLD_WTF;return out;}catch(e){o.WTF=OLD_WTF;if(!e.message.match(/SYLK bad record ID/)&&OLD_WTF)throw e;return PRN.to_workbook(d,opts);}}var WK_=/*#__PURE__*/function(){function lotushopper(data,cb/*:RecordHopperCB*/,opts/*:any*/){if(!data)return;prep_blob(data,data.l||0);var Enum=opts.Enum||WK1Enum;while(data.l<data.length){var RT=data.read_shift(2);var R=Enum[RT]||Enum[0xFFFF];var length=data.read_shift(2);var tgt=data.l+length;var d=R.f&&R.f(data,length,opts);data.l=tgt;if(cb(d,R,RT))return;}}function lotus_to_workbook(d/*:RawData*/,opts){switch(opts.type){case'base64':return lotus_to_workbook_buf(s2a(Base64_decode(d)),opts);case'binary':return lotus_to_workbook_buf(s2a(d),opts);case'buffer':case'array':return lotus_to_workbook_buf(d,opts);}throw "Unsupported type "+opts.type;}function lotus_to_workbook_buf(d,opts)/*:Workbook*/{if(!d)return d;var o=opts||{};var s/*:Worksheet*/=o.dense?[]:{}/*:any*/,n="Sheet1",next_n="",sidx=0;var sheets={},snames=[],realnames=[];var refguess={s:{r:0,c:0},e:{r:0,c:0}};var sheetRows=o.sheetRows||0;if(d[2]==0x00){if(d[3]==0x08||d[3]==0x09){if(d.length>=16&&d[14]==0x05&&d[15]===0x6c)throw new Error("Unsupported Works 3 for Mac file");}}if(d[2]==0x02){o.Enum=WK1Enum;lotushopper(d,function(val,R,RT){switch(RT){case 0x00:/* BOF */o.vers=val;if(val>=0x1000)o.qpro=true;break;case 0x06:refguess=val;break;/* RANGE */case 0xCC:if(val)next_n=val;break;/* SHEETNAMECS */case 0xDE:next_n=val;break;/* SHEETNAMELP */case 0x0F:/* LABEL */case 0x33:/* STRING */if(!o.qpro)val[1].v=val[1].v.slice(1);/* falls through */case 0x0D:/* INTEGER */case 0x0E:/* NUMBER */case 0x10:/* FORMULA */ /* TODO: actual translation of the format code */if(RT==0x0E&&(val[2]&0x70)==0x70&&(val[2]&0x0F)>1&&(val[2]&0x0F)<15){val[1].z=o.dateNF||table_fmt[14];if(o.cellDates){val[1].t='d';val[1].v=numdate(val[1].v);}}if(o.qpro){if(val[3]>sidx){s["!ref"]=encode_range(refguess);sheets[n]=s;snames.push(n);s=o.dense?[]:{};refguess={s:{r:0,c:0},e:{r:0,c:0}};sidx=val[3];n=next_n||"Sheet"+(sidx+1);next_n="";}}var tmpcell=o.dense?(s[val[0].r]||[])[val[0].c]:s[encode_cell(val[0])];if(tmpcell){tmpcell.t=val[1].t;tmpcell.v=val[1].v;if(val[1].z!=null)tmpcell.z=val[1].z;if(val[1].f!=null)tmpcell.f=val[1].f;break;}if(o.dense){if(!s[val[0].r])s[val[0].r]=[];s[val[0].r][val[0].c]=val[1];}else s[encode_cell(val[0])]=val[1];break;}},o);}else if(d[2]==0x1A||d[2]==0x0E){o.Enum=WK3Enum;if(d[2]==0x0E){o.qpro=true;d.l=0;}lotushopper(d,function(val,R,RT){switch(RT){case 0xCC:n=val;break;/* SHEETNAMECS */case 0x16:/* LABEL16 */val[1].v=val[1].v.slice(1);/* falls through */case 0x17:/* NUMBER17 */case 0x18:/* NUMBER18 */case 0x19:/* FORMULA19 */case 0x25:/* NUMBER25 */case 0x27:/* NUMBER27 */case 0x28:/* FORMULA28 */if(val[3]>sidx){s["!ref"]=encode_range(refguess);sheets[n]=s;snames.push(n);s=o.dense?[]:{};refguess={s:{r:0,c:0},e:{r:0,c:0}};sidx=val[3];n="Sheet"+(sidx+1);}if(sheetRows>0&&val[0].r>=sheetRows)break;if(o.dense){if(!s[val[0].r])s[val[0].r]=[];s[val[0].r][val[0].c]=val[1];}else s[encode_cell(val[0])]=val[1];if(refguess.e.c<val[0].c)refguess.e.c=val[0].c;if(refguess.e.r<val[0].r)refguess.e.r=val[0].r;break;case 0x1B:/* XFORMAT */if(val[0x36b0])realnames[val[0x36b0][0]]=val[0x36b0][1];break;case 0x0601:/* SHEETINFOQP */realnames[val[0]]=val[1];if(val[0]==sidx)n=val[1];break;}},o);}else throw new Error("Unrecognized LOTUS BOF "+d[2]);s["!ref"]=encode_range(refguess);sheets[next_n||n]=s;snames.push(next_n||n);if(!realnames.length)return {SheetNames:snames,Sheets:sheets};var osheets={},rnames=[];/* TODO: verify no collisions */for(var i=0;i<realnames.length;++i)if(sheets[snames[i]]){rnames.push(realnames[i]||snames[i]);osheets[realnames[i]]=sheets[realnames[i]]||sheets[snames[i]];}else {rnames.push(realnames[i]);osheets[realnames[i]]={"!ref":"A1"};}return {SheetNames:rnames,Sheets:osheets};}function sheet_to_wk1(ws/*:Worksheet*/,opts/*:WriteOpts*/){var o=opts||{};if(+o.codepage>=0)set_cp(+o.codepage);if(o.type=="string")throw new Error("Cannot write WK1 to JS string");var ba=buf_array();var range=safe_decode_range(ws["!ref"]);var dense=Array.isArray(ws);var cols=[];write_biff_rec(ba,0x00,write_BOF_WK1(0x0406));write_biff_rec(ba,0x06,write_RANGE(range));var max_R=Math.min(range.e.r,8191);for(var R=range.s.r;R<=max_R;++R){var rr=encode_row(R);for(var C=range.s.c;C<=range.e.c;++C){if(R===range.s.r)cols[C]=encode_col(C);var ref=cols[C]+rr;var cell=dense?(ws[R]||[])[C]:ws[ref];if(!cell||cell.t=="z")continue;/* TODO: formula records */if(cell.t=="n"){if((cell.v|0)==cell.v&&cell.v>=-32768&&cell.v<=32767)write_biff_rec(ba,0x0d,write_INTEGER(R,C,cell.v));else write_biff_rec(ba,0x0e,write_NUMBER(R,C,cell.v));}else {var str=format_cell(cell);write_biff_rec(ba,0x0F,write_LABEL(R,C,str.slice(0,239)));}}}write_biff_rec(ba,0x01);return ba.end();}function book_to_wk3(wb/*:Workbook*/,opts/*:WriteOpts*/){var o=opts||{};if(+o.codepage>=0)set_cp(+o.codepage);if(o.type=="string")throw new Error("Cannot write WK3 to JS string");var ba=buf_array();write_biff_rec(ba,0x00,write_BOF_WK3(wb));for(var i=0,cnt=0;i<wb.SheetNames.length;++i)if((wb.Sheets[wb.SheetNames[i]]||{})["!ref"])write_biff_rec(ba,0x1b,write_XFORMAT_SHEETNAME(wb.SheetNames[i],cnt++));var wsidx=0;for(i=0;i<wb.SheetNames.length;++i){var ws=wb.Sheets[wb.SheetNames[i]];if(!ws||!ws["!ref"])continue;var range=safe_decode_range(ws["!ref"]);var dense=Array.isArray(ws);var cols=[];var max_R=Math.min(range.e.r,8191);for(var R=range.s.r;R<=max_R;++R){var rr=encode_row(R);for(var C=range.s.c;C<=range.e.c;++C){if(R===range.s.r)cols[C]=encode_col(C);var ref=cols[C]+rr;var cell=dense?(ws[R]||[])[C]:ws[ref];if(!cell||cell.t=="z")continue;/* TODO: FORMULA19 NUMBER18 records */if(cell.t=="n"){write_biff_rec(ba,0x17,write_NUMBER_17(R,C,wsidx,cell.v));}else {var str=format_cell(cell);/* TODO: max len? */write_biff_rec(ba,0x16,write_LABEL_16(R,C,wsidx,str.slice(0,239)));}}}++wsidx;}write_biff_rec(ba,0x01);return ba.end();}function write_BOF_WK1(v/*:number*/){var out=new_buf(2);out.write_shift(2,v);return out;}function write_BOF_WK3(wb/*:Workbook*/){var out=new_buf(26);out.write_shift(2,0x1000);out.write_shift(2,0x0004);out.write_shift(4,0x0000);var rows=0,cols=0,wscnt=0;for(var i=0;i<wb.SheetNames.length;++i){var name=wb.SheetNames[i];var ws=wb.Sheets[name];if(!ws||!ws["!ref"])continue;++wscnt;var range=decode_range(ws["!ref"]);if(rows<range.e.r)rows=range.e.r;if(cols<range.e.c)cols=range.e.c;}if(rows>8191)rows=8191;out.write_shift(2,rows);out.write_shift(1,wscnt);out.write_shift(1,cols);out.write_shift(2,0x00);out.write_shift(2,0x00);out.write_shift(1,0x01);out.write_shift(1,0x02);out.write_shift(4,0);out.write_shift(4,0);return out;}function parse_RANGE(blob,length,opts){var o={s:{c:0,r:0},e:{c:0,r:0}};if(length==8&&opts.qpro){o.s.c=blob.read_shift(1);blob.l++;o.s.r=blob.read_shift(2);o.e.c=blob.read_shift(1);blob.l++;o.e.r=blob.read_shift(2);return o;}o.s.c=blob.read_shift(2);o.s.r=blob.read_shift(2);if(length==12&&opts.qpro)blob.l+=2;o.e.c=blob.read_shift(2);o.e.r=blob.read_shift(2);if(length==12&&opts.qpro)blob.l+=2;if(o.s.c==0xFFFF)o.s.c=o.e.c=o.s.r=o.e.r=0;return o;}function write_RANGE(range){var out=new_buf(8);out.write_shift(2,range.s.c);out.write_shift(2,range.s.r);out.write_shift(2,range.e.c);out.write_shift(2,range.e.r);return out;}function parse_cell(blob,length,opts){var o=[{c:0,r:0},{t:'n',v:0},0,0];if(opts.qpro&&opts.vers!=0x5120){o[0].c=blob.read_shift(1);o[3]=blob.read_shift(1);o[0].r=blob.read_shift(2);blob.l+=2;}else {o[2]=blob.read_shift(1);o[0].c=blob.read_shift(2);o[0].r=blob.read_shift(2);}return o;}function parse_LABEL(blob,length,opts){var tgt=blob.l+length;var o=parse_cell(blob,length,opts);o[1].t='s';if(opts.vers==0x5120){blob.l++;var len=blob.read_shift(1);o[1].v=blob.read_shift(len,'utf8');return o;}if(opts.qpro)blob.l++;o[1].v=blob.read_shift(tgt-blob.l,'cstr');return o;}function write_LABEL(R,C,s){/* TODO: encoding */var o=new_buf(7+s.length);o.write_shift(1,0xFF);o.write_shift(2,C);o.write_shift(2,R);o.write_shift(1,0x27);// ??
	for(var i=0;i<o.length;++i){var cc=s.charCodeAt(i);o.write_shift(1,cc>=0x80?0x5F:cc);}o.write_shift(1,0);return o;}function parse_INTEGER(blob,length,opts){var o=parse_cell(blob,length,opts);o[1].v=blob.read_shift(2,'i');return o;}function write_INTEGER(R,C,v){var o=new_buf(7);o.write_shift(1,0xFF);o.write_shift(2,C);o.write_shift(2,R);o.write_shift(2,v,'i');return o;}function parse_NUMBER(blob,length,opts){var o=parse_cell(blob,length,opts);o[1].v=blob.read_shift(8,'f');return o;}function write_NUMBER(R,C,v){var o=new_buf(13);o.write_shift(1,0xFF);o.write_shift(2,C);o.write_shift(2,R);o.write_shift(8,v,'f');return o;}function parse_FORMULA(blob,length,opts){var tgt=blob.l+length;var o=parse_cell(blob,length,opts);/* TODO: formula */o[1].v=blob.read_shift(8,'f');if(opts.qpro)blob.l=tgt;else {var flen=blob.read_shift(2);wk1_fmla_to_csf(blob.slice(blob.l,blob.l+flen),o);blob.l+=flen;}return o;}function wk1_parse_rc(B,V,col){var rel=V&0x8000;V&=~0x8000;V=(rel?B:0)+(V>=0x2000?V-0x4000:V);return (rel?"":"$")+(col?encode_col(V):encode_row(V));}/* var oprec = [
			8, 8, 8, 8, 8, 8, 8, 8, 6, 4, 4, 5, 5, 7, 3, 3,
			3, 3, 3, 3, 1, 1, 2, 6, 8, 8, 8, 8, 8, 8, 8, 8
		]; */ /* TODO: flesh out */var FuncTab={0x33:["FALSE",0],0x34:["TRUE",0],0x46:["LEN",1],0x50:["SUM",69],0x51:["AVERAGEA",69],0x52:["COUNTA",69],0x53:["MINA",69],0x54:["MAXA",69],0x6F:["T",1]};var BinOpTab=["","","","","","","","",// eslint-disable-line no-mixed-spaces-and-tabs
	"","+","-","*","/","^","=","<>",// eslint-disable-line no-mixed-spaces-and-tabs
	"<=",">=","<",">","","","","",// eslint-disable-line no-mixed-spaces-and-tabs
	"&","","","","","","",""// eslint-disable-line no-mixed-spaces-and-tabs
	];function wk1_fmla_to_csf(blob,o){prep_blob(blob,0);var out=[],argc=0,R="",C="",argL="",argR="";while(blob.l<blob.length){var cc=blob[blob.l++];switch(cc){case 0x00:out.push(blob.read_shift(8,'f'));break;case 0x01:{C=wk1_parse_rc(o[0].c,blob.read_shift(2),true);R=wk1_parse_rc(o[0].r,blob.read_shift(2),false);out.push(C+R);}break;case 0x02:{var c=wk1_parse_rc(o[0].c,blob.read_shift(2),true);var r=wk1_parse_rc(o[0].r,blob.read_shift(2),false);C=wk1_parse_rc(o[0].c,blob.read_shift(2),true);R=wk1_parse_rc(o[0].r,blob.read_shift(2),false);out.push(c+r+":"+C+R);}break;case 0x03:if(blob.l<blob.length){console.error("WK1 premature formula end");return;}break;case 0x04:out.push("("+out.pop()+")");break;case 0x05:out.push(blob.read_shift(2));break;case 0x06:{/* TODO: text encoding */var Z="";while(cc=blob[blob.l++])Z+=String.fromCharCode(cc);out.push('"'+Z.replace(/"/g,'""')+'"');}break;case 0x08:out.push("-"+out.pop());break;case 0x17:out.push("+"+out.pop());break;case 0x16:out.push("NOT("+out.pop()+")");break;case 0x14:case 0x15:{argR=out.pop();argL=out.pop();out.push(["AND","OR"][cc-0x14]+"("+argL+","+argR+")");}break;default:if(cc<0x20&&BinOpTab[cc]){argR=out.pop();argL=out.pop();out.push(argL+BinOpTab[cc]+argR);}else if(FuncTab[cc]){argc=FuncTab[cc][1];if(argc==69)argc=blob[blob.l++];if(argc>out.length){console.error("WK1 bad formula parse 0x"+cc.toString(16)+":|"+out.join("|")+"|");return;}var args=out.slice(-argc);out.length-=argc;out.push(FuncTab[cc][0]+"("+args.join(",")+")");}else if(cc<=0x07)return console.error("WK1 invalid opcode "+cc.toString(16));else if(cc<=0x18)return console.error("WK1 unsupported op "+cc.toString(16));else if(cc<=0x1E)return console.error("WK1 invalid opcode "+cc.toString(16));else if(cc<=0x73)return console.error("WK1 unsupported function opcode "+cc.toString(16));// possible future functions ??
	else return console.error("WK1 unrecognized opcode "+cc.toString(16));}}if(out.length==1)o[1].f=""+out[0];else console.error("WK1 bad formula parse |"+out.join("|")+"|");}function parse_cell_3(blob/*::, length*/){var o=[{c:0,r:0},{t:'n',v:0},0];o[0].r=blob.read_shift(2);o[3]=blob[blob.l++];o[0].c=blob[blob.l++];return o;}function parse_LABEL_16(blob,length){var o=parse_cell_3(blob);o[1].t='s';o[1].v=blob.read_shift(length-4,'cstr');return o;}function write_LABEL_16(R,C,wsidx,s){/* TODO: encoding */var o=new_buf(6+s.length);o.write_shift(2,R);o.write_shift(1,wsidx);o.write_shift(1,C);o.write_shift(1,0x27);for(var i=0;i<s.length;++i){var cc=s.charCodeAt(i);o.write_shift(1,cc>=0x80?0x5F:cc);}o.write_shift(1,0);return o;}function parse_NUMBER_18(blob,length){var o=parse_cell_3(blob);o[1].v=blob.read_shift(2);var v=o[1].v>>1;if(o[1].v&0x1){switch(v&0x07){case 0:v=(v>>3)*5000;break;case 1:v=(v>>3)*500;break;case 2:v=(v>>3)/20;break;case 3:v=(v>>3)/200;break;case 4:v=(v>>3)/2000;break;case 5:v=(v>>3)/20000;break;case 6:v=(v>>3)/16;break;case 7:v=(v>>3)/64;break;}}o[1].v=v;return o;}function parse_NUMBER_17(blob,length){var o=parse_cell_3(blob);var v1=blob.read_shift(4);var v2=blob.read_shift(4);var e=blob.read_shift(2);if(e==0xFFFF){if(v1===0&&v2===0xC0000000){o[1].t="e";o[1].v=0x0F;}// ERR -> #VALUE!
	else if(v1===0&&v2===0xD0000000){o[1].t="e";o[1].v=0x2A;}// NA -> #N/A
	else o[1].v=0;return o;}var s=e&0x8000;e=(e&0x7FFF)-16446;o[1].v=(1-s*2)*(v2*Math.pow(2,e+32)+v1*Math.pow(2,e));return o;}function write_NUMBER_17(R,C,wsidx,v){var o=new_buf(14);o.write_shift(2,R);o.write_shift(1,wsidx);o.write_shift(1,C);if(v==0){o.write_shift(4,0);o.write_shift(4,0);o.write_shift(2,0xFFFF);return o;}var s=0,e=0,v1=0,v2=0;if(v<0){s=1;v=-v;}e=Math.log2(v)|0;v/=Math.pow(2,e-31);v2=v>>>0;if((v2&0x80000000)==0){v/=2;++e;v2=v>>>0;}v-=v2;v2|=0x80000000;v2>>>=0;v*=Math.pow(2,32);v1=v>>>0;o.write_shift(4,v1);o.write_shift(4,v2);e+=0x3FFF+(s?0x8000:0);o.write_shift(2,e);return o;}function parse_FORMULA_19(blob,length){var o=parse_NUMBER_17(blob);blob.l+=length-14;/* TODO: WK3 formula */return o;}function parse_NUMBER_25(blob,length){var o=parse_cell_3(blob);var v1=blob.read_shift(4);o[1].v=v1>>6;return o;}function parse_NUMBER_27(blob,length){var o=parse_cell_3(blob);var v1=blob.read_shift(8,'f');o[1].v=v1;return o;}function parse_FORMULA_28(blob,length){var o=parse_NUMBER_27(blob);blob.l+=length-10;/* TODO: formula */return o;}function parse_SHEETNAMECS(blob,length){return blob[blob.l+length-1]==0?blob.read_shift(length,'cstr'):"";}function parse_SHEETNAMELP(blob,length){var len=blob[blob.l++];if(len>length-1)len=length-1;var o="";while(o.length<len)o+=String.fromCharCode(blob[blob.l++]);return o;}function parse_SHEETINFOQP(blob,length,opts){if(!opts.qpro||length<21)return;var id=blob.read_shift(1);blob.l+=17;blob.l+=1;//var len = blob.read_shift(1);
	blob.l+=2;var nm=blob.read_shift(length-21,'cstr');return [id,nm];}function parse_XFORMAT(blob,length){var o={},tgt=blob.l+length;while(blob.l<tgt){var dt=blob.read_shift(2);if(dt==0x36b0){o[dt]=[0,""];o[dt][0]=blob.read_shift(2);while(blob[blob.l]){o[dt][1]+=String.fromCharCode(blob[blob.l]);blob.l++;}blob.l++;}// TODO: 0x3a99 ??
	}return o;}function write_XFORMAT_SHEETNAME(name,wsidx){var out=new_buf(5+name.length);out.write_shift(2,0x36b0);out.write_shift(2,wsidx);for(var i=0;i<name.length;++i){var cc=name.charCodeAt(i);out[out.l++]=cc>0x7F?0x5F:cc;}out[out.l++]=0;return out;}var WK1Enum={/*::[*/0x0000/*::]*/:{n:"BOF",f:parseuint16},/*::[*/0x0001/*::]*/:{n:"EOF"},/*::[*/0x0002/*::]*/:{n:"CALCMODE"},/*::[*/0x0003/*::]*/:{n:"CALCORDER"},/*::[*/0x0004/*::]*/:{n:"SPLIT"},/*::[*/0x0005/*::]*/:{n:"SYNC"},/*::[*/0x0006/*::]*/:{n:"RANGE",f:parse_RANGE},/*::[*/0x0007/*::]*/:{n:"WINDOW1"},/*::[*/0x0008/*::]*/:{n:"COLW1"},/*::[*/0x0009/*::]*/:{n:"WINTWO"},/*::[*/0x000A/*::]*/:{n:"COLW2"},/*::[*/0x000B/*::]*/:{n:"NAME"},/*::[*/0x000C/*::]*/:{n:"BLANK"},/*::[*/0x000D/*::]*/:{n:"INTEGER",f:parse_INTEGER},/*::[*/0x000E/*::]*/:{n:"NUMBER",f:parse_NUMBER},/*::[*/0x000F/*::]*/:{n:"LABEL",f:parse_LABEL},/*::[*/0x0010/*::]*/:{n:"FORMULA",f:parse_FORMULA},/*::[*/0x0018/*::]*/:{n:"TABLE"},/*::[*/0x0019/*::]*/:{n:"ORANGE"},/*::[*/0x001A/*::]*/:{n:"PRANGE"},/*::[*/0x001B/*::]*/:{n:"SRANGE"},/*::[*/0x001C/*::]*/:{n:"FRANGE"},/*::[*/0x001D/*::]*/:{n:"KRANGE1"},/*::[*/0x0020/*::]*/:{n:"HRANGE"},/*::[*/0x0023/*::]*/:{n:"KRANGE2"},/*::[*/0x0024/*::]*/:{n:"PROTEC"},/*::[*/0x0025/*::]*/:{n:"FOOTER"},/*::[*/0x0026/*::]*/:{n:"HEADER"},/*::[*/0x0027/*::]*/:{n:"SETUP"},/*::[*/0x0028/*::]*/:{n:"MARGINS"},/*::[*/0x0029/*::]*/:{n:"LABELFMT"},/*::[*/0x002A/*::]*/:{n:"TITLES"},/*::[*/0x002B/*::]*/:{n:"SHEETJS"},/*::[*/0x002D/*::]*/:{n:"GRAPH"},/*::[*/0x002E/*::]*/:{n:"NGRAPH"},/*::[*/0x002F/*::]*/:{n:"CALCCOUNT"},/*::[*/0x0030/*::]*/:{n:"UNFORMATTED"},/*::[*/0x0031/*::]*/:{n:"CURSORW12"},/*::[*/0x0032/*::]*/:{n:"WINDOW"},/*::[*/0x0033/*::]*/:{n:"STRING",f:parse_LABEL},/*::[*/0x0037/*::]*/:{n:"PASSWORD"},/*::[*/0x0038/*::]*/:{n:"LOCKED"},/*::[*/0x003C/*::]*/:{n:"QUERY"},/*::[*/0x003D/*::]*/:{n:"QUERYNAME"},/*::[*/0x003E/*::]*/:{n:"PRINT"},/*::[*/0x003F/*::]*/:{n:"PRINTNAME"},/*::[*/0x0040/*::]*/:{n:"GRAPH2"},/*::[*/0x0041/*::]*/:{n:"GRAPHNAME"},/*::[*/0x0042/*::]*/:{n:"ZOOM"},/*::[*/0x0043/*::]*/:{n:"SYMSPLIT"},/*::[*/0x0044/*::]*/:{n:"NSROWS"},/*::[*/0x0045/*::]*/:{n:"NSCOLS"},/*::[*/0x0046/*::]*/:{n:"RULER"},/*::[*/0x0047/*::]*/:{n:"NNAME"},/*::[*/0x0048/*::]*/:{n:"ACOMM"},/*::[*/0x0049/*::]*/:{n:"AMACRO"},/*::[*/0x004A/*::]*/:{n:"PARSE"},/*::[*/0x0066/*::]*/:{n:"PRANGES??"},/*::[*/0x0067/*::]*/:{n:"RRANGES??"},/*::[*/0x0068/*::]*/:{n:"FNAME??"},/*::[*/0x0069/*::]*/:{n:"MRANGES??"},/*::[*/0x00CC/*::]*/:{n:"SHEETNAMECS",f:parse_SHEETNAMECS},/*::[*/0x00DE/*::]*/:{n:"SHEETNAMELP",f:parse_SHEETNAMELP},/*::[*/0xFFFF/*::]*/:{n:""}};var WK3Enum={/*::[*/0x0000/*::]*/:{n:"BOF"},/*::[*/0x0001/*::]*/:{n:"EOF"},/*::[*/0x0002/*::]*/:{n:"PASSWORD"},/*::[*/0x0003/*::]*/:{n:"CALCSET"},/*::[*/0x0004/*::]*/:{n:"WINDOWSET"},/*::[*/0x0005/*::]*/:{n:"SHEETCELLPTR"},/*::[*/0x0006/*::]*/:{n:"SHEETLAYOUT"},/*::[*/0x0007/*::]*/:{n:"COLUMNWIDTH"},/*::[*/0x0008/*::]*/:{n:"HIDDENCOLUMN"},/*::[*/0x0009/*::]*/:{n:"USERRANGE"},/*::[*/0x000A/*::]*/:{n:"SYSTEMRANGE"},/*::[*/0x000B/*::]*/:{n:"ZEROFORCE"},/*::[*/0x000C/*::]*/:{n:"SORTKEYDIR"},/*::[*/0x000D/*::]*/:{n:"FILESEAL"},/*::[*/0x000E/*::]*/:{n:"DATAFILLNUMS"},/*::[*/0x000F/*::]*/:{n:"PRINTMAIN"},/*::[*/0x0010/*::]*/:{n:"PRINTSTRING"},/*::[*/0x0011/*::]*/:{n:"GRAPHMAIN"},/*::[*/0x0012/*::]*/:{n:"GRAPHSTRING"},/*::[*/0x0013/*::]*/:{n:"??"},/*::[*/0x0014/*::]*/:{n:"ERRCELL"},/*::[*/0x0015/*::]*/:{n:"NACELL"},/*::[*/0x0016/*::]*/:{n:"LABEL16",f:parse_LABEL_16},/*::[*/0x0017/*::]*/:{n:"NUMBER17",f:parse_NUMBER_17},/*::[*/0x0018/*::]*/:{n:"NUMBER18",f:parse_NUMBER_18},/*::[*/0x0019/*::]*/:{n:"FORMULA19",f:parse_FORMULA_19},/*::[*/0x001A/*::]*/:{n:"FORMULA1A"},/*::[*/0x001B/*::]*/:{n:"XFORMAT",f:parse_XFORMAT},/*::[*/0x001C/*::]*/:{n:"DTLABELMISC"},/*::[*/0x001D/*::]*/:{n:"DTLABELCELL"},/*::[*/0x001E/*::]*/:{n:"GRAPHWINDOW"},/*::[*/0x001F/*::]*/:{n:"CPA"},/*::[*/0x0020/*::]*/:{n:"LPLAUTO"},/*::[*/0x0021/*::]*/:{n:"QUERY"},/*::[*/0x0022/*::]*/:{n:"HIDDENSHEET"},/*::[*/0x0023/*::]*/:{n:"??"},/*::[*/0x0025/*::]*/:{n:"NUMBER25",f:parse_NUMBER_25},/*::[*/0x0026/*::]*/:{n:"??"},/*::[*/0x0027/*::]*/:{n:"NUMBER27",f:parse_NUMBER_27},/*::[*/0x0028/*::]*/:{n:"FORMULA28",f:parse_FORMULA_28},/*::[*/0x008E/*::]*/:{n:"??"},/*::[*/0x0093/*::]*/:{n:"??"},/*::[*/0x0096/*::]*/:{n:"??"},/*::[*/0x0097/*::]*/:{n:"??"},/*::[*/0x0098/*::]*/:{n:"??"},/*::[*/0x0099/*::]*/:{n:"??"},/*::[*/0x009A/*::]*/:{n:"??"},/*::[*/0x009B/*::]*/:{n:"??"},/*::[*/0x009C/*::]*/:{n:"??"},/*::[*/0x00A3/*::]*/:{n:"??"},/*::[*/0x00AE/*::]*/:{n:"??"},/*::[*/0x00AF/*::]*/:{n:"??"},/*::[*/0x00B0/*::]*/:{n:"??"},/*::[*/0x00B1/*::]*/:{n:"??"},/*::[*/0x00B8/*::]*/:{n:"??"},/*::[*/0x00B9/*::]*/:{n:"??"},/*::[*/0x00BA/*::]*/:{n:"??"},/*::[*/0x00BB/*::]*/:{n:"??"},/*::[*/0x00BC/*::]*/:{n:"??"},/*::[*/0x00C3/*::]*/:{n:"??"},/*::[*/0x00C9/*::]*/:{n:"??"},/*::[*/0x00CC/*::]*/:{n:"SHEETNAMECS",f:parse_SHEETNAMECS},/*::[*/0x00CD/*::]*/:{n:"??"},/*::[*/0x00CE/*::]*/:{n:"??"},/*::[*/0x00CF/*::]*/:{n:"??"},/*::[*/0x00D0/*::]*/:{n:"??"},/*::[*/0x0100/*::]*/:{n:"??"},/*::[*/0x0103/*::]*/:{n:"??"},/*::[*/0x0104/*::]*/:{n:"??"},/*::[*/0x0105/*::]*/:{n:"??"},/*::[*/0x0106/*::]*/:{n:"??"},/*::[*/0x0107/*::]*/:{n:"??"},/*::[*/0x0109/*::]*/:{n:"??"},/*::[*/0x010A/*::]*/:{n:"??"},/*::[*/0x010B/*::]*/:{n:"??"},/*::[*/0x010C/*::]*/:{n:"??"},/*::[*/0x010E/*::]*/:{n:"??"},/*::[*/0x010F/*::]*/:{n:"??"},/*::[*/0x0180/*::]*/:{n:"??"},/*::[*/0x0185/*::]*/:{n:"??"},/*::[*/0x0186/*::]*/:{n:"??"},/*::[*/0x0189/*::]*/:{n:"??"},/*::[*/0x018C/*::]*/:{n:"??"},/*::[*/0x0200/*::]*/:{n:"??"},/*::[*/0x0202/*::]*/:{n:"??"},/*::[*/0x0201/*::]*/:{n:"??"},/*::[*/0x0204/*::]*/:{n:"??"},/*::[*/0x0205/*::]*/:{n:"??"},/*::[*/0x0280/*::]*/:{n:"??"},/*::[*/0x0281/*::]*/:{n:"??"},/*::[*/0x0282/*::]*/:{n:"??"},/*::[*/0x0283/*::]*/:{n:"??"},/*::[*/0x0284/*::]*/:{n:"??"},/*::[*/0x0285/*::]*/:{n:"??"},/*::[*/0x0286/*::]*/:{n:"??"},/*::[*/0x0287/*::]*/:{n:"??"},/*::[*/0x0288/*::]*/:{n:"??"},/*::[*/0x0292/*::]*/:{n:"??"},/*::[*/0x0293/*::]*/:{n:"??"},/*::[*/0x0294/*::]*/:{n:"??"},/*::[*/0x0295/*::]*/:{n:"??"},/*::[*/0x0296/*::]*/:{n:"??"},/*::[*/0x0299/*::]*/:{n:"??"},/*::[*/0x029A/*::]*/:{n:"??"},/*::[*/0x0300/*::]*/:{n:"??"},/*::[*/0x0304/*::]*/:{n:"??"},/*::[*/0x0601/*::]*/:{n:"SHEETINFOQP",f:parse_SHEETINFOQP},/*::[*/0x0640/*::]*/:{n:"??"},/*::[*/0x0642/*::]*/:{n:"??"},/*::[*/0x0701/*::]*/:{n:"??"},/*::[*/0x0702/*::]*/:{n:"??"},/*::[*/0x0703/*::]*/:{n:"??"},/*::[*/0x0704/*::]*/:{n:"??"},/*::[*/0x0780/*::]*/:{n:"??"},/*::[*/0x0800/*::]*/:{n:"??"},/*::[*/0x0801/*::]*/:{n:"??"},/*::[*/0x0804/*::]*/:{n:"??"},/*::[*/0x0A80/*::]*/:{n:"??"},/*::[*/0x2AF6/*::]*/:{n:"??"},/*::[*/0x3231/*::]*/:{n:"??"},/*::[*/0x6E49/*::]*/:{n:"??"},/*::[*/0x6F44/*::]*/:{n:"??"},/*::[*/0xFFFF/*::]*/:{n:""}};return {sheet_to_wk1:sheet_to_wk1,book_to_wk3:book_to_wk3,to_workbook:lotus_to_workbook};}();/* 18.4.7 rPr CT_RPrElt */function parse_rpr(rpr){var font={},m=rpr.match(tagregex),i=0;var pass=false;if(m)for(;i!=m.length;++i){var y=parsexmltag(m[i]);switch(y[0].replace(/\w*:/g,"")){/* 18.8.12 condense CT_BooleanProperty */ /* ** not required . */case'<condense':break;/* 18.8.17 extend CT_BooleanProperty */ /* ** not required . */case'<extend':break;/* 18.8.36 shadow CT_BooleanProperty */ /* ** not required . */case'<shadow':if(!y.val)break;/* falls through */case'<shadow>':case'<shadow/>':font.shadow=1;break;case'</shadow>':break;/* 18.4.1 charset CT_IntProperty TODO */case'<charset':if(y.val=='1')break;font.cp=CS2CP[parseInt(y.val,10)];break;/* 18.4.2 outline CT_BooleanProperty TODO */case'<outline':if(!y.val)break;/* falls through */case'<outline>':case'<outline/>':font.outline=1;break;case'</outline>':break;/* 18.4.5 rFont CT_FontName */case'<rFont':font.name=y.val;break;/* 18.4.11 sz CT_FontSize */case'<sz':font.sz=y.val;break;/* 18.4.10 strike CT_BooleanProperty */case'<strike':if(!y.val)break;/* falls through */case'<strike>':case'<strike/>':font.strike=1;break;case'</strike>':break;/* 18.4.13 u CT_UnderlineProperty */case'<u':if(!y.val)break;switch(y.val){case'double':font.uval="double";break;case'singleAccounting':font.uval="single-accounting";break;case'doubleAccounting':font.uval="double-accounting";break;}/* falls through */case'<u>':case'<u/>':font.u=1;break;case'</u>':break;/* 18.8.2 b */case'<b':if(y.val=='0')break;/* falls through */case'<b>':case'<b/>':font.b=1;break;case'</b>':break;/* 18.8.26 i */case'<i':if(y.val=='0')break;/* falls through */case'<i>':case'<i/>':font.i=1;break;case'</i>':break;/* 18.3.1.15 color CT_Color TODO: tint, theme, auto, indexed */case'<color':if(y.rgb)font.color=y.rgb.slice(2,8);break;case'<color>':case'<color/>':case'</color>':break;/* 18.8.18 family ST_FontFamily */case'<family':font.family=y.val;break;case'<family>':case'<family/>':case'</family>':break;/* 18.4.14 vertAlign CT_VerticalAlignFontProperty TODO */case'<vertAlign':font.valign=y.val;break;case'<vertAlign>':case'<vertAlign/>':case'</vertAlign>':break;/* 18.8.35 scheme CT_FontScheme TODO */case'<scheme':break;case'<scheme>':case'<scheme/>':case'</scheme>':break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':break;case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(y[0].charCodeAt(1)!==47&&!pass)throw new Error('Unrecognized rich format '+y[0]);}}return font;}var parse_rs=/*#__PURE__*/function(){var tregex=matchtag("t"),rpregex=matchtag("rPr");/* 18.4.4 r CT_RElt */function parse_r(r){/* 18.4.12 t ST_Xstring */var t=r.match(tregex)/*, cp = 65001*/;if(!t)return {t:"s",v:""};var o/*:Cell*/={t:'s',v:unescapexml(t[1])}/*:any*/;var rpr=r.match(rpregex);if(rpr)o.s=parse_rpr(rpr[1]);return o;}var rregex=/<(?:\w+:)?r>/g,rend=/<\/(?:\w+:)?r>/;return function parse_rs(rs){return rs.replace(rregex,"").split(rend).map(parse_r).filter(function(r){return r.v;});};}();/* Parse a list of <r> tags */var rs_to_html=/*#__PURE__*/function parse_rs_factory(){var nlregex=/(\r\n|\n)/g;function parse_rpr2(font,intro,outro){var style/*:Array<string>*/=[];if(font.u)style.push("text-decoration: underline;");if(font.uval)style.push("text-underline-style:"+font.uval+";");if(font.sz)style.push("font-size:"+font.sz+"pt;");if(font.outline)style.push("text-effect: outline;");if(font.shadow)style.push("text-shadow: auto;");intro.push('<span style="'+style.join("")+'">');if(font.b){intro.push("<b>");outro.push("</b>");}if(font.i){intro.push("<i>");outro.push("</i>");}if(font.strike){intro.push("<s>");outro.push("</s>");}var align=font.valign||"";if(align=="superscript"||align=="super")align="sup";else if(align=="subscript")align="sub";if(align!=""){intro.push("<"+align+">");outro.push("</"+align+">");}outro.push("</span>");return font;}/* 18.4.4 r CT_RElt */function r_to_html(r){var terms/*:[Array<string>, string, Array<string>]*/=[[],r.v,[]];if(!r.v)return "";if(r.s)parse_rpr2(r.s,terms[0],terms[2]);return terms[0].join("")+terms[1].replace(nlregex,'<br/>')+terms[2].join("");}return function parse_rs(rs){return rs.map(r_to_html).join("");};}();/* 18.4.8 si CT_Rst */var sitregex=/<(?:\w+:)?t[^>]*>([^<]*)<\/(?:\w+:)?t>/g,sirregex=/<(?:\w+:)?r>/;var sirphregex=/<(?:\w+:)?rPh.*?>([\s\S]*?)<\/(?:\w+:)?rPh>/g;function parse_si(x,opts){var html=opts?opts.cellHTML:true;var z={};if(!x)return {t:""};//var y;
	/* 18.4.12 t ST_Xstring (Plaintext String) */ // TODO: is whitespace actually valid here?
	if(x.match(/^\s*<(?:\w+:)?t[^>]*>/)){z.t=unescapexml(utf8read(x.slice(x.indexOf(">")+1).split(/<\/(?:\w+:)?t>/)[0]||""));z.r=utf8read(x);if(html)z.h=escapehtml(z.t);}/* 18.4.4 r CT_RElt (Rich Text Run) */else if(/*y = */x.match(sirregex)){z.r=utf8read(x);z.t=unescapexml(utf8read((x.replace(sirphregex,'').match(sitregex)||[]).join("").replace(tagregex,"")));if(html)z.h=rs_to_html(parse_rs(z.r));}/* 18.4.3 phoneticPr CT_PhoneticPr (TODO: needed for Asian support) */ /* 18.4.6 rPh CT_PhoneticRun (TODO: needed for Asian support) */return z;}/* 18.4 Shared String Table */var sstr0=/<(?:\w+:)?sst([^>]*)>([\s\S]*)<\/(?:\w+:)?sst>/;var sstr1=/<(?:\w+:)?(?:si|sstItem)>/g;var sstr2=/<\/(?:\w+:)?(?:si|sstItem)>/;function parse_sst_xml(data/*:string*/,opts)/*:SST*/{var s/*:SST*/=[]/*:any*/,ss="";if(!data)return s;/* 18.4.9 sst CT_Sst */var sst=data.match(sstr0);if(sst){ss=sst[2].replace(sstr1,"").split(sstr2);for(var i=0;i!=ss.length;++i){var o=parse_si(ss[i].trim(),opts);if(o!=null)s[s.length]=o;}sst=parsexmltag(sst[1]);s.Count=sst.count;s.Unique=sst.uniqueCount;}return s;}/* [MS-XLSB] 2.4.221 BrtBeginSst */function parse_BrtBeginSst(data){return [data.read_shift(4),data.read_shift(4)];}/* [MS-XLSB] 2.1.7.45 Shared Strings */function parse_sst_bin(data,opts)/*:SST*/{var s/*:SST*/=[]/*:any*/;var pass=false;recordhopper(data,function hopper_sst(val,R,RT){switch(RT){case 0x009F:/* BrtBeginSst */s.Count=val[0];s.Unique=val[1];break;case 0x0013:/* BrtSSTItem */s.push(val);break;case 0x00A0:/* BrtEndSst */return true;case 0x0023:/* BrtFRTBegin */pass=true;break;case 0x0024:/* BrtFRTEnd */pass=false;break;default:if(R.T);if(!pass||opts.WTF)throw new Error("Unexpected record 0x"+RT.toString(16));}});return s;}function _JS2ANSI(str/*:string*/)/*:Array<number>*/{var o/*:Array<number>*/=[],oo=str.split("");for(var i=0;i<oo.length;++i)o[i]=oo[i].charCodeAt(0);return o;}/* [MS-OFFCRYPTO] 2.1.4 Version */function parse_CRYPTOVersion(blob,length/*:?number*/){var o/*:any*/={};o.Major=blob.read_shift(2);o.Minor=blob.read_shift(2);/*:: if(length == null) return o; */if(length>=4)blob.l+=length-4;return o;}/* [MS-OFFCRYPTO] 2.1.5 DataSpaceVersionInfo */function parse_DataSpaceVersionInfo(blob){var o={};o.id=blob.read_shift(0,'lpp4');o.R=parse_CRYPTOVersion(blob,4);o.U=parse_CRYPTOVersion(blob,4);o.W=parse_CRYPTOVersion(blob,4);return o;}/* [MS-OFFCRYPTO] 2.1.6.1 DataSpaceMapEntry Structure */function parse_DataSpaceMapEntry(blob){var len=blob.read_shift(4);var end=blob.l+len-4;var o={};var cnt=blob.read_shift(4);var comps/*:Array<{t:number, v:string}>*/=[];/* [MS-OFFCRYPTO] 2.1.6.2 DataSpaceReferenceComponent Structure */while(cnt-->0)comps.push({t:blob.read_shift(4),v:blob.read_shift(0,'lpp4')});o.name=blob.read_shift(0,'lpp4');o.comps=comps;if(blob.l!=end)throw new Error("Bad DataSpaceMapEntry: "+blob.l+" != "+end);return o;}/* [MS-OFFCRYPTO] 2.1.6 DataSpaceMap */function parse_DataSpaceMap(blob){var o=[];blob.l+=4;// must be 0x8
	var cnt=blob.read_shift(4);while(cnt-->0)o.push(parse_DataSpaceMapEntry(blob));return o;}/* [MS-OFFCRYPTO] 2.1.7 DataSpaceDefinition */function parse_DataSpaceDefinition(blob)/*:Array<string>*/{var o/*:Array<string>*/=[];blob.l+=4;// must be 0x8
	var cnt=blob.read_shift(4);while(cnt-->0)o.push(blob.read_shift(0,'lpp4'));return o;}/* [MS-OFFCRYPTO] 2.1.8 DataSpaceDefinition */function parse_TransformInfoHeader(blob){var o={};/*var len = */blob.read_shift(4);blob.l+=4;// must be 0x1
	o.id=blob.read_shift(0,'lpp4');o.name=blob.read_shift(0,'lpp4');o.R=parse_CRYPTOVersion(blob,4);o.U=parse_CRYPTOVersion(blob,4);o.W=parse_CRYPTOVersion(blob,4);return o;}function parse_Primary(blob){/* [MS-OFFCRYPTO] 2.2.6 IRMDSTransformInfo */var hdr=parse_TransformInfoHeader(blob);/* [MS-OFFCRYPTO] 2.1.9 EncryptionTransformInfo */hdr.ename=blob.read_shift(0,'8lpp4');hdr.blksz=blob.read_shift(4);hdr.cmode=blob.read_shift(4);if(blob.read_shift(4)!=0x04)throw new Error("Bad !Primary record");return hdr;}/* [MS-OFFCRYPTO] 2.3.2 Encryption Header */function parse_EncryptionHeader(blob,length/*:number*/){var tgt=blob.l+length;var o={};o.Flags=blob.read_shift(4)&0x3F;blob.l+=4;o.AlgID=blob.read_shift(4);var valid=false;switch(o.AlgID){case 0x660E:case 0x660F:case 0x6610:valid=o.Flags==0x24;break;case 0x6801:valid=o.Flags==0x04;break;case 0:valid=o.Flags==0x10||o.Flags==0x04||o.Flags==0x24;break;default:throw 'Unrecognized encryption algorithm: '+o.AlgID;}if(!valid)throw new Error("Encryption Flags/AlgID mismatch");o.AlgIDHash=blob.read_shift(4);o.KeySize=blob.read_shift(4);o.ProviderType=blob.read_shift(4);blob.l+=8;o.CSPName=blob.read_shift(tgt-blob.l>>1,'utf16le');blob.l=tgt;return o;}/* [MS-OFFCRYPTO] 2.3.3 Encryption Verifier */function parse_EncryptionVerifier(blob,length/*:number*/){var o={},tgt=blob.l+length;blob.l+=4;// SaltSize must be 0x10
	o.Salt=blob.slice(blob.l,blob.l+16);blob.l+=16;o.Verifier=blob.slice(blob.l,blob.l+16);blob.l+=16;/*var sz = */blob.read_shift(4);o.VerifierHash=blob.slice(blob.l,tgt);blob.l=tgt;return o;}/* [MS-OFFCRYPTO] 2.3.4.* EncryptionInfo Stream */function parse_EncryptionInfo(blob){var vers=parse_CRYPTOVersion(blob);switch(vers.Minor){case 0x02:return [vers.Minor,parse_EncInfoStd(blob)];case 0x03:return [vers.Minor,parse_EncInfoExt()];case 0x04:return [vers.Minor,parse_EncInfoAgl(blob)];}throw new Error("ECMA-376 Encrypted file unrecognized Version: "+vers.Minor);}/* [MS-OFFCRYPTO] 2.3.4.5  EncryptionInfo Stream (Standard Encryption) */function parse_EncInfoStd(blob/*::, vers*/){var flags=blob.read_shift(4);if((flags&0x3F)!=0x24)throw new Error("EncryptionInfo mismatch");var sz=blob.read_shift(4);//var tgt = blob.l + sz;
	var hdr=parse_EncryptionHeader(blob,sz);var verifier=parse_EncryptionVerifier(blob,blob.length-blob.l);return {t:"Std",h:hdr,v:verifier};}/* [MS-OFFCRYPTO] 2.3.4.6  EncryptionInfo Stream (Extensible Encryption) */function/*::blob, vers*/parse_EncInfoExt(){throw new Error("File is password-protected: ECMA-376 Extensible");}/* [MS-OFFCRYPTO] 2.3.4.10 EncryptionInfo Stream (Agile Encryption) */function parse_EncInfoAgl(blob/*::, vers*/){var KeyData=["saltSize","blockSize","keyBits","hashSize","cipherAlgorithm","cipherChaining","hashAlgorithm","saltValue"];blob.l+=4;var xml=blob.read_shift(blob.length-blob.l,'utf8');var o={};xml.replace(tagregex,function xml_agile(x){var y/*:any*/=parsexmltag(x);switch(strip_ns(y[0])){case'<?xml':break;case'<encryption':case'</encryption>':break;case'<keyData':KeyData.forEach(function(k){o[k]=y[k];});break;case'<dataIntegrity':o.encryptedHmacKey=y.encryptedHmacKey;o.encryptedHmacValue=y.encryptedHmacValue;break;case'<keyEncryptors>':case'<keyEncryptors':o.encs=[];break;case'</keyEncryptors>':break;case'<keyEncryptor':o.uri=y.uri;break;case'</keyEncryptor>':break;case'<encryptedKey':o.encs.push(y);break;default:throw y[0];}});return o;}/* [MS-OFFCRYPTO] 2.3.5.1 RC4 CryptoAPI Encryption Header */function parse_RC4CryptoHeader(blob,length/*:number*/){var o={};var vers=o.EncryptionVersionInfo=parse_CRYPTOVersion(blob,4);length-=4;if(vers.Minor!=2)throw new Error('unrecognized minor version code: '+vers.Minor);if(vers.Major>4||vers.Major<2)throw new Error('unrecognized major version code: '+vers.Major);o.Flags=blob.read_shift(4);length-=4;var sz=blob.read_shift(4);length-=4;o.EncryptionHeader=parse_EncryptionHeader(blob,sz);length-=sz;o.EncryptionVerifier=parse_EncryptionVerifier(blob,length);return o;}/* [MS-OFFCRYPTO] 2.3.6.1 RC4 Encryption Header */function parse_RC4Header(blob/*::, length*/){var o={};var vers=o.EncryptionVersionInfo=parse_CRYPTOVersion(blob,4);if(vers.Major!=1||vers.Minor!=1)throw 'unrecognized version code '+vers.Major+' : '+vers.Minor;o.Salt=blob.read_shift(16);o.EncryptedVerifier=blob.read_shift(16);o.EncryptedVerifierHash=blob.read_shift(16);return o;}/* [MS-OFFCRYPTO] 2.3.7.1 Binary Document Password Verifier Derivation */function crypto_CreatePasswordVerifier_Method1(Password/*:string*/){var Verifier=0x0000,PasswordArray;var PasswordDecoded=_JS2ANSI(Password);var len=PasswordDecoded.length+1,i,PasswordByte;var Intermediate1,Intermediate2,Intermediate3;PasswordArray=new_raw_buf(len);PasswordArray[0]=PasswordDecoded.length;for(i=1;i!=len;++i)PasswordArray[i]=PasswordDecoded[i-1];for(i=len-1;i>=0;--i){PasswordByte=PasswordArray[i];Intermediate1=(Verifier&0x4000)===0x0000?0:1;Intermediate2=Verifier<<1&0x7FFF;Intermediate3=Intermediate1|Intermediate2;Verifier=Intermediate3^PasswordByte;}return Verifier^0xCE4B;}/* [MS-OFFCRYPTO] 2.3.7.2 Binary Document XOR Array Initialization */var crypto_CreateXorArray_Method1=/*#__PURE__*/function(){var PadArray=[0xBB,0xFF,0xFF,0xBA,0xFF,0xFF,0xB9,0x80,0x00,0xBE,0x0F,0x00,0xBF,0x0F,0x00];var InitialCode=[0xE1F0,0x1D0F,0xCC9C,0x84C0,0x110C,0x0E10,0xF1CE,0x313E,0x1872,0xE139,0xD40F,0x84F9,0x280C,0xA96A,0x4EC3];var XorMatrix=[0xAEFC,0x4DD9,0x9BB2,0x2745,0x4E8A,0x9D14,0x2A09,0x7B61,0xF6C2,0xFDA5,0xEB6B,0xC6F7,0x9DCF,0x2BBF,0x4563,0x8AC6,0x05AD,0x0B5A,0x16B4,0x2D68,0x5AD0,0x0375,0x06EA,0x0DD4,0x1BA8,0x3750,0x6EA0,0xDD40,0xD849,0xA0B3,0x5147,0xA28E,0x553D,0xAA7A,0x44D5,0x6F45,0xDE8A,0xAD35,0x4A4B,0x9496,0x390D,0x721A,0xEB23,0xC667,0x9CEF,0x29FF,0x53FE,0xA7FC,0x5FD9,0x47D3,0x8FA6,0x0F6D,0x1EDA,0x3DB4,0x7B68,0xF6D0,0xB861,0x60E3,0xC1C6,0x93AD,0x377B,0x6EF6,0xDDEC,0x45A0,0x8B40,0x06A1,0x0D42,0x1A84,0x3508,0x6A10,0xAA51,0x4483,0x8906,0x022D,0x045A,0x08B4,0x1168,0x76B4,0xED68,0xCAF1,0x85C3,0x1BA7,0x374E,0x6E9C,0x3730,0x6E60,0xDCC0,0xA9A1,0x4363,0x86C6,0x1DAD,0x3331,0x6662,0xCCC4,0x89A9,0x0373,0x06E6,0x0DCC,0x1021,0x2042,0x4084,0x8108,0x1231,0x2462,0x48C4];var Ror=function(Byte){return (Byte/2|Byte*128)&0xFF;};var XorRor=function(byte1,byte2){return Ror(byte1^byte2);};var CreateXorKey_Method1=function(Password){var XorKey=InitialCode[Password.length-1];var CurrentElement=0x68;for(var i=Password.length-1;i>=0;--i){var Char=Password[i];for(var j=0;j!=7;++j){if(Char&0x40)XorKey^=XorMatrix[CurrentElement];Char*=2;--CurrentElement;}}return XorKey;};return function(password/*:string*/){var Password=_JS2ANSI(password);var XorKey=CreateXorKey_Method1(Password);var Index=Password.length;var ObfuscationArray=new_raw_buf(16);for(var i=0;i!=16;++i)ObfuscationArray[i]=0x00;var Temp,PasswordLastChar,PadIndex;if((Index&1)===1){Temp=XorKey>>8;ObfuscationArray[Index]=XorRor(PadArray[0],Temp);--Index;Temp=XorKey&0xFF;PasswordLastChar=Password[Password.length-1];ObfuscationArray[Index]=XorRor(PasswordLastChar,Temp);}while(Index>0){--Index;Temp=XorKey>>8;ObfuscationArray[Index]=XorRor(Password[Index],Temp);--Index;Temp=XorKey&0xFF;ObfuscationArray[Index]=XorRor(Password[Index],Temp);}Index=15;PadIndex=15-Password.length;while(PadIndex>0){Temp=XorKey>>8;ObfuscationArray[Index]=XorRor(PadArray[PadIndex],Temp);--Index;--PadIndex;Temp=XorKey&0xFF;ObfuscationArray[Index]=XorRor(Password[Index],Temp);--Index;--PadIndex;}return ObfuscationArray;};}();/* [MS-OFFCRYPTO] 2.3.7.3 Binary Document XOR Data Transformation Method 1 */var crypto_DecryptData_Method1=function(password/*:string*/,Data,XorArrayIndex,XorArray,O){/* If XorArray is set, use it; if O is not set, make changes in-place */if(!O)O=Data;if(!XorArray)XorArray=crypto_CreateXorArray_Method1(password);var Index,Value;for(Index=0;Index!=Data.length;++Index){Value=Data[Index];Value^=XorArray[XorArrayIndex];Value=(Value>>5|Value<<3)&0xFF;O[Index]=Value;++XorArrayIndex;}return [O,XorArrayIndex,XorArray];};var crypto_MakeXorDecryptor=function(password/*:string*/){var XorArrayIndex=0,XorArray=crypto_CreateXorArray_Method1(password);return function(Data){var O=crypto_DecryptData_Method1("",Data,XorArrayIndex,XorArray);XorArrayIndex=O[1];return O[0];};};/* 2.5.343 */function parse_XORObfuscation(blob,length,opts,out){var o={key:parseuint16(blob),verificationBytes:parseuint16(blob)}/*:any*/;if(opts.password)o.verifier=crypto_CreatePasswordVerifier_Method1(opts.password);out.valid=o.verificationBytes===o.verifier;if(out.valid)out.insitu=crypto_MakeXorDecryptor(opts.password);return o;}/* 2.4.117 */function parse_FilePassHeader(blob,length/*:number*/,oo){var o=oo||{};o.Info=blob.read_shift(2);blob.l-=2;if(o.Info===1)o.Data=parse_RC4Header(blob);else o.Data=parse_RC4CryptoHeader(blob,length);return o;}function parse_FilePass(blob,length/*:number*/,opts){var o={Type:opts.biff>=8?blob.read_shift(2):0}/*:any*/;/* wEncryptionType */if(o.Type)parse_FilePassHeader(blob,length-2,o);else parse_XORObfuscation(blob,opts.biff>=8?length:length-2,opts,o);return o;}var RTF=/*#__PURE__*/function(){function rtf_to_sheet(d/*:RawData*/,opts)/*:Worksheet*/{switch(opts.type){case'base64':return rtf_to_sheet_str(Base64_decode(d),opts);case'binary':return rtf_to_sheet_str(d,opts);case'buffer':return rtf_to_sheet_str(has_buf&&Buffer.isBuffer(d)?d.toString('binary'):a2s(d),opts);case'array':return rtf_to_sheet_str(cc2str(d),opts);}throw new Error("Unrecognized type "+opts.type);}/* TODO: this is a stub */function rtf_to_sheet_str(str/*:string*/,opts)/*:Worksheet*/{var o=opts||{};var ws/*:Worksheet*/=o.dense?[]/*:any*/:{}/*:any*/;var rows=str.match(/\\trowd.*?\\row\b/g);if(!rows.length)throw new Error("RTF missing table");var range/*:Range*/={s:{c:0,r:0},e:{c:0,r:rows.length-1}}/*:any*/;rows.forEach(function(rowtf,R){if(Array.isArray(ws))ws[R]=[];var rtfre=/\\\w+\b/g;var last_index=0;var res;var C=-1;while(res=rtfre.exec(rowtf)){switch(res[0]){case"\\cell":var data=rowtf.slice(last_index,rtfre.lastIndex-res[0].length);if(data[0]==" ")data=data.slice(1);++C;if(data.length){// TODO: value parsing, including codepage adjustments
	var cell={v:data,t:"s"};if(Array.isArray(ws))ws[R][C]=cell;else ws[encode_cell({r:R,c:C})]=cell;}break;}last_index=rtfre.lastIndex;}if(C>range.e.c)range.e.c=C;});ws['!ref']=encode_range(range);return ws;}function rtf_to_workbook(d/*:RawData*/,opts)/*:Workbook*/{return sheet_to_workbook(rtf_to_sheet(d,opts),opts);}/* TODO: this is a stub */function sheet_to_rtf(ws/*:Worksheet*/ /*::, opts*/)/*:string*/{var o=["{\\rtf1\\ansi"];var r=safe_decode_range(ws['!ref']),cell/*:Cell*/;var dense=Array.isArray(ws);for(var R=r.s.r;R<=r.e.r;++R){o.push("\\trowd\\trautofit1");for(var C=r.s.c;C<=r.e.c;++C)o.push("\\cellx"+(C+1));o.push("\\pard\\intbl");for(C=r.s.c;C<=r.e.c;++C){var coord=encode_cell({r:R,c:C});cell=dense?(ws[R]||[])[C]:ws[coord];if(!cell||cell.v==null&&(!cell.f||cell.F))continue;o.push(" "+(cell.w||(format_cell(cell),cell.w)));o.push("\\cell");}o.push("\\pard\\intbl\\row");}return o.join("")+"}";}return {to_workbook:rtf_to_workbook,to_sheet:rtf_to_sheet,from_sheet:sheet_to_rtf};}();function hex2RGB(h){var o=h.slice(h[0]==="#"?1:0).slice(0,6);return [parseInt(o.slice(0,2),16),parseInt(o.slice(2,4),16),parseInt(o.slice(4,6),16)];}function rgb2Hex(rgb){for(var i=0,o=1;i!=3;++i)o=o*256+(rgb[i]>255?255:rgb[i]<0?0:rgb[i]);return o.toString(16).toUpperCase().slice(1);}function rgb2HSL(rgb){var R=rgb[0]/255,G=rgb[1]/255,B=rgb[2]/255;var M=Math.max(R,G,B),m=Math.min(R,G,B),C=M-m;if(C===0)return [0,0,R];var H6=0,S=0,L2=M+m;S=C/(L2>1?2-L2:L2);switch(M){case R:H6=((G-B)/C+6)%6;break;case G:H6=(B-R)/C+2;break;case B:H6=(R-G)/C+4;break;}return [H6/6,S,L2/2];}function hsl2RGB(hsl){var H=hsl[0],S=hsl[1],L=hsl[2];var C=S*2*(L<0.5?L:1-L),m=L-C/2;var rgb=[m,m,m],h6=6*H;var X;if(S!==0)switch(h6|0){case 0:case 6:X=C*h6;rgb[0]+=C;rgb[1]+=X;break;case 1:X=C*(2-h6);rgb[0]+=X;rgb[1]+=C;break;case 2:X=C*(h6-2);rgb[1]+=C;rgb[2]+=X;break;case 3:X=C*(4-h6);rgb[1]+=X;rgb[2]+=C;break;case 4:X=C*(h6-4);rgb[2]+=C;rgb[0]+=X;break;case 5:X=C*(6-h6);rgb[2]+=X;rgb[0]+=C;break;}for(var i=0;i!=3;++i)rgb[i]=Math.round(rgb[i]*255);return rgb;}/* 18.8.3 bgColor tint algorithm */function rgb_tint(hex,tint){if(tint===0)return hex;var hsl=rgb2HSL(hex2RGB(hex));if(tint<0)hsl[2]=hsl[2]*(1+tint);else hsl[2]=1-(1-hsl[2])*(1-tint);return rgb2Hex(hsl2RGB(hsl));}/* 18.3.1.13 width calculations */ /* [MS-OI29500] 2.1.595 Column Width & Formatting */var DEF_MDW=6,MAX_MDW=15,MIN_MDW=1,MDW=DEF_MDW;function width2px(width){return Math.floor((width+Math.round(128/MDW)/256)*MDW);}function px2char(px){return Math.floor((px-5)/MDW*100+0.5)/100;}function char2width(chr){return Math.round((chr*MDW+5)/MDW*256)/256;}//function px2char_(px) { return (((px - 5)/MDW * 100 + 0.5))/100; }
	//function char2width_(chr) { return (((chr * MDW + 5)/MDW*256))/256; }
	function cycle_width(collw){return char2width(px2char(width2px(collw)));}/* XLSX/XLSB/XLS specify width in units of MDW */function find_mdw_colw(collw){var delta=Math.abs(collw-cycle_width(collw)),_MDW=MDW;if(delta>0.005)for(MDW=MIN_MDW;MDW<MAX_MDW;++MDW)if(Math.abs(collw-cycle_width(collw))<=delta){delta=Math.abs(collw-cycle_width(collw));_MDW=MDW;}MDW=_MDW;}/* XLML specifies width in terms of pixels */ /*function find_mdw_wpx(wpx) {
		var delta = Infinity, guess = 0, _MDW = MIN_MDW;
		for(MDW=MIN_MDW; MDW<MAX_MDW; ++MDW) {
			guess = char2width_(px2char_(wpx))*256;
			guess = (guess) % 1;
			if(guess > 0.5) guess--;
			if(Math.abs(guess) < delta) { delta = Math.abs(guess); _MDW = MDW; }
		}
		MDW = _MDW;
	}*/function process_col(coll/*:ColInfo*/){if(coll.width){coll.wpx=width2px(coll.width);coll.wch=px2char(coll.wpx);coll.MDW=MDW;}else if(coll.wpx){coll.wch=px2char(coll.wpx);coll.width=char2width(coll.wch);coll.MDW=MDW;}else if(typeof coll.wch=='number'){coll.width=char2width(coll.wch);coll.wpx=width2px(coll.width);coll.MDW=MDW;}if(coll.customWidth)delete coll.customWidth;}var DEF_PPI=96,PPI=DEF_PPI;function px2pt(px){return px*96/PPI;}function pt2px(pt){return pt*PPI/96;}/* [MS-EXSPXML3] 2.4.54 ST_enmPattern */var XLMLPatternTypeMap={"None":"none","Solid":"solid","Gray50":"mediumGray","Gray75":"darkGray","Gray25":"lightGray","HorzStripe":"darkHorizontal","VertStripe":"darkVertical","ReverseDiagStripe":"darkDown","DiagStripe":"darkUp","DiagCross":"darkGrid","ThickDiagCross":"darkTrellis","ThinHorzStripe":"lightHorizontal","ThinVertStripe":"lightVertical","ThinReverseDiagStripe":"lightDown","ThinHorzCross":"lightGrid"};/* 18.8.5 borders CT_Borders */function parse_borders(t,styles,themes,opts){styles.Borders=[];var border={};var pass=false;(t[0].match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(strip_ns(y[0])){case'<borders':case'<borders>':case'</borders>':break;/* 18.8.4 border CT_Border */case'<border':case'<border>':case'<border/>':border=/*::(*/{}/*:: :any)*/;if(y.diagonalUp)border.diagonalUp=parsexmlbool(y.diagonalUp);if(y.diagonalDown)border.diagonalDown=parsexmlbool(y.diagonalDown);styles.Borders.push(border);break;case'</border>':break;/* note: not in spec, appears to be CT_BorderPr */case'<left/>':break;case'<left':case'<left>':break;case'</left>':break;/* note: not in spec, appears to be CT_BorderPr */case'<right/>':break;case'<right':case'<right>':break;case'</right>':break;/* 18.8.43 top CT_BorderPr */case'<top/>':break;case'<top':case'<top>':break;case'</top>':break;/* 18.8.6 bottom CT_BorderPr */case'<bottom/>':break;case'<bottom':case'<bottom>':break;case'</bottom>':break;/* 18.8.13 diagonal CT_BorderPr */case'<diagonal':case'<diagonal>':case'<diagonal/>':break;case'</diagonal>':break;/* 18.8.25 horizontal CT_BorderPr */case'<horizontal':case'<horizontal>':case'<horizontal/>':break;case'</horizontal>':break;/* 18.8.44 vertical CT_BorderPr */case'<vertical':case'<vertical>':case'<vertical/>':break;case'</vertical>':break;/* 18.8.37 start CT_BorderPr */case'<start':case'<start>':case'<start/>':break;case'</start>':break;/* 18.8.16 end CT_BorderPr */case'<end':case'<end>':case'<end/>':break;case'</end>':break;/* 18.8.? color CT_Color */case'<color':case'<color>':break;case'<color/>':case'</color>':break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':break;case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(opts&&opts.WTF){if(!pass)throw new Error('unrecognized '+y[0]+' in borders');}}});}/* 18.8.21 fills CT_Fills */function parse_fills(t,styles,themes,opts){styles.Fills=[];var fill={};var pass=false;(t[0].match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(strip_ns(y[0])){case'<fills':case'<fills>':case'</fills>':break;/* 18.8.20 fill CT_Fill */case'<fill>':case'<fill':case'<fill/>':fill={};styles.Fills.push(fill);break;case'</fill>':break;/* 18.8.24 gradientFill CT_GradientFill */case'<gradientFill>':break;case'<gradientFill':case'</gradientFill>':styles.Fills.push(fill);fill={};break;/* 18.8.32 patternFill CT_PatternFill */case'<patternFill':case'<patternFill>':if(y.patternType)fill.patternType=y.patternType;break;case'<patternFill/>':case'</patternFill>':break;/* 18.8.3 bgColor CT_Color */case'<bgColor':if(!fill.bgColor)fill.bgColor={};if(y.indexed)fill.bgColor.indexed=parseInt(y.indexed,10);if(y.theme)fill.bgColor.theme=parseInt(y.theme,10);if(y.tint)fill.bgColor.tint=parseFloat(y.tint);/* Excel uses ARGB strings */if(y.rgb)fill.bgColor.rgb=y.rgb.slice(-6);break;case'<bgColor/>':case'</bgColor>':break;/* 18.8.19 fgColor CT_Color */case'<fgColor':if(!fill.fgColor)fill.fgColor={};if(y.theme)fill.fgColor.theme=parseInt(y.theme,10);if(y.tint)fill.fgColor.tint=parseFloat(y.tint);/* Excel uses ARGB strings */if(y.rgb!=null)fill.fgColor.rgb=y.rgb.slice(-6);break;case'<fgColor/>':case'</fgColor>':break;/* 18.8.38 stop CT_GradientStop */case'<stop':case'<stop/>':break;case'</stop>':break;/* 18.8.? color CT_Color */case'<color':case'<color/>':break;case'</color>':break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':break;case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(opts&&opts.WTF){if(!pass)throw new Error('unrecognized '+y[0]+' in fills');}}});}/* 18.8.23 fonts CT_Fonts */function parse_fonts(t,styles,themes,opts){styles.Fonts=[];var font={};var pass=false;(t[0].match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(strip_ns(y[0])){case'<fonts':case'<fonts>':case'</fonts>':break;/* 18.8.22 font CT_Font */case'<font':case'<font>':break;case'</font>':case'<font/>':styles.Fonts.push(font);font={};break;/* 18.8.29 name CT_FontName */case'<name':if(y.val)font.name=utf8read(y.val);break;case'<name/>':case'</name>':break;/* 18.8.2  b CT_BooleanProperty */case'<b':font.bold=y.val?parsexmlbool(y.val):1;break;case'<b/>':font.bold=1;break;/* 18.8.26 i CT_BooleanProperty */case'<i':font.italic=y.val?parsexmlbool(y.val):1;break;case'<i/>':font.italic=1;break;/* 18.4.13 u CT_UnderlineProperty */case'<u':switch(y.val){case"none":font.underline=0x00;break;case"single":font.underline=0x01;break;case"double":font.underline=0x02;break;case"singleAccounting":font.underline=0x21;break;case"doubleAccounting":font.underline=0x22;break;}break;case'<u/>':font.underline=1;break;/* 18.4.10 strike CT_BooleanProperty */case'<strike':font.strike=y.val?parsexmlbool(y.val):1;break;case'<strike/>':font.strike=1;break;/* 18.4.2  outline CT_BooleanProperty */case'<outline':font.outline=y.val?parsexmlbool(y.val):1;break;case'<outline/>':font.outline=1;break;/* 18.8.36 shadow CT_BooleanProperty */case'<shadow':font.shadow=y.val?parsexmlbool(y.val):1;break;case'<shadow/>':font.shadow=1;break;/* 18.8.12 condense CT_BooleanProperty */case'<condense':font.condense=y.val?parsexmlbool(y.val):1;break;case'<condense/>':font.condense=1;break;/* 18.8.17 extend CT_BooleanProperty */case'<extend':font.extend=y.val?parsexmlbool(y.val):1;break;case'<extend/>':font.extend=1;break;/* 18.4.11 sz CT_FontSize */case'<sz':if(y.val)font.sz=+y.val;break;case'<sz/>':case'</sz>':break;/* 18.4.14 vertAlign CT_VerticalAlignFontProperty */case'<vertAlign':if(y.val)font.vertAlign=y.val;break;case'<vertAlign/>':case'</vertAlign>':break;/* 18.8.18 family CT_FontFamily */case'<family':if(y.val)font.family=parseInt(y.val,10);break;case'<family/>':case'</family>':break;/* 18.8.35 scheme CT_FontScheme */case'<scheme':if(y.val)font.scheme=y.val;break;case'<scheme/>':case'</scheme>':break;/* 18.4.1 charset CT_IntProperty */case'<charset':if(y.val=='1')break;y.codepage=CS2CP[parseInt(y.val,10)];break;/* 18.?.? color CT_Color */case'<color':if(!font.color)font.color={};if(y.auto)font.color.auto=parsexmlbool(y.auto);if(y.rgb)font.color.rgb=y.rgb.slice(-6);else if(y.indexed){font.color.index=parseInt(y.indexed,10);var icv=XLSIcv[font.color.index];if(font.color.index==81)icv=XLSIcv[1];if(!icv)icv=XLSIcv[1];//throw new Error(x); // note: 206 is valid
	font.color.rgb=icv[0].toString(16)+icv[1].toString(16)+icv[2].toString(16);}else if(y.theme){font.color.theme=parseInt(y.theme,10);if(y.tint)font.color.tint=parseFloat(y.tint);if(y.theme&&themes.themeElements&&themes.themeElements.clrScheme){font.color.rgb=rgb_tint(themes.themeElements.clrScheme[font.color.theme].rgb,font.color.tint||0);}}break;case'<color/>':case'</color>':break;/* note: sometimes mc:AlternateContent appears bare */case'<AlternateContent':pass=true;break;case'</AlternateContent>':pass=false;break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':break;case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(opts&&opts.WTF){if(!pass)throw new Error('unrecognized '+y[0]+' in fonts');}}});}/* 18.8.31 numFmts CT_NumFmts */function parse_numFmts(t,styles,opts){styles.NumberFmt=[];var k/*Array<number>*/=keys(table_fmt)/*:any*/;for(var i=0;i<k.length;++i)styles.NumberFmt[k[i]]=table_fmt[k[i]];var m=t[0].match(tagregex);if(!m)return;for(i=0;i<m.length;++i){var y=parsexmltag(m[i]);switch(strip_ns(y[0])){case'<numFmts':case'</numFmts>':case'<numFmts/>':case'<numFmts>':break;case'<numFmt':{var f=unescapexml(utf8read(y.formatCode)),j=parseInt(y.numFmtId,10);styles.NumberFmt[j]=f;if(j>0){if(j>0x188){for(j=0x188;j>0x3c;--j)if(styles.NumberFmt[j]==null)break;styles.NumberFmt[j]=f;}SSF_load(f,j);}}break;case'</numFmt>':break;default:if(opts.WTF)throw new Error('unrecognized '+y[0]+' in numFmts');}}}/* 18.8.10 cellXfs CT_CellXfs */var cellXF_uint=["numFmtId","fillId","fontId","borderId","xfId"];var cellXF_bool=["applyAlignment","applyBorder","applyFill","applyFont","applyNumberFormat","applyProtection","pivotButton","quotePrefix"];function parse_cellXfs(t,styles,opts){styles.CellXf=[];var xf;var pass=false;(t[0].match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x),i=0;switch(strip_ns(y[0])){case'<cellXfs':case'<cellXfs>':case'<cellXfs/>':case'</cellXfs>':break;/* 18.8.45 xf CT_Xf */case'<xf':case'<xf/>':xf=y;delete xf[0];for(i=0;i<cellXF_uint.length;++i)if(xf[cellXF_uint[i]])xf[cellXF_uint[i]]=parseInt(xf[cellXF_uint[i]],10);for(i=0;i<cellXF_bool.length;++i)if(xf[cellXF_bool[i]])xf[cellXF_bool[i]]=parsexmlbool(xf[cellXF_bool[i]]);if(styles.NumberFmt&&xf.numFmtId>0x188){for(i=0x188;i>0x3c;--i)if(styles.NumberFmt[xf.numFmtId]==styles.NumberFmt[i]){xf.numFmtId=i;break;}}styles.CellXf.push(xf);break;case'</xf>':break;/* 18.8.1 alignment CT_CellAlignment */case'<alignment':case'<alignment/>':var alignment={};if(y.vertical)alignment.vertical=y.vertical;if(y.horizontal)alignment.horizontal=y.horizontal;if(y.textRotation!=null)alignment.textRotation=y.textRotation;if(y.indent)alignment.indent=y.indent;if(y.wrapText)alignment.wrapText=parsexmlbool(y.wrapText);xf.alignment=alignment;break;case'</alignment>':break;/* 18.8.33 protection CT_CellProtection */case'<protection':break;case'</protection>':case'<protection/>':break;/* note: sometimes mc:AlternateContent appears bare */case'<AlternateContent':pass=true;break;case'</AlternateContent>':pass=false;break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':break;case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(opts&&opts.WTF){if(!pass)throw new Error('unrecognized '+y[0]+' in cellXfs');}}});}/* 18.8 Styles CT_Stylesheet*/var parse_sty_xml=/*#__PURE__*/function make_pstyx(){var numFmtRegex=/<(?:\w+:)?numFmts([^>]*)>[\S\s]*?<\/(?:\w+:)?numFmts>/;var cellXfRegex=/<(?:\w+:)?cellXfs([^>]*)>[\S\s]*?<\/(?:\w+:)?cellXfs>/;var fillsRegex=/<(?:\w+:)?fills([^>]*)>[\S\s]*?<\/(?:\w+:)?fills>/;var fontsRegex=/<(?:\w+:)?fonts([^>]*)>[\S\s]*?<\/(?:\w+:)?fonts>/;var bordersRegex=/<(?:\w+:)?borders([^>]*)>[\S\s]*?<\/(?:\w+:)?borders>/;return function parse_sty_xml(data,themes,opts){var styles={};if(!data)return styles;data=data.replace(/<!--([\s\S]*?)-->/mg,"").replace(/<!DOCTYPE[^\[]*\[[^\]]*\]>/gm,"");/* 18.8.39 styleSheet CT_Stylesheet */var t;/* 18.8.31 numFmts CT_NumFmts ? */if(t=data.match(numFmtRegex))parse_numFmts(t,styles,opts);/* 18.8.23 fonts CT_Fonts ? */if(t=data.match(fontsRegex))parse_fonts(t,styles,themes,opts);/* 18.8.21 fills CT_Fills ? */if(t=data.match(fillsRegex))parse_fills(t,styles,themes,opts);/* 18.8.5  borders CT_Borders ? */if(t=data.match(bordersRegex))parse_borders(t,styles,themes,opts);/* 18.8.9  cellStyleXfs CT_CellStyleXfs ? */ /* 18.8.8  cellStyles CT_CellStyles ? */ /* 18.8.10 cellXfs CT_CellXfs ? */if(t=data.match(cellXfRegex))parse_cellXfs(t,styles,opts);/* 18.8.15 dxfs CT_Dxfs ? */ /* 18.8.42 tableStyles CT_TableStyles ? */ /* 18.8.11 colors CT_Colors ? */ /* 18.2.10 extLst CT_ExtensionList ? */return styles;};}();/* [MS-XLSB] 2.4.657 BrtFmt */function parse_BrtFmt(data,length/*:number*/){var numFmtId=data.read_shift(2);var stFmtCode=parse_XLWideString(data);return [numFmtId,stFmtCode];}/* [MS-XLSB] 2.4.659 BrtFont TODO */function parse_BrtFont(data,length/*:number*/,opts){var out={}/*:any*/;out.sz=data.read_shift(2)/20;var grbit=parse_FontFlags(data);if(grbit.fItalic)out.italic=1;if(grbit.fCondense)out.condense=1;if(grbit.fExtend)out.extend=1;if(grbit.fShadow)out.shadow=1;if(grbit.fOutline)out.outline=1;if(grbit.fStrikeout)out.strike=1;var bls=data.read_shift(2);if(bls===0x02BC)out.bold=1;switch(data.read_shift(2)){/* case 0: out.vertAlign = "baseline"; break; */case 1:out.vertAlign="superscript";break;case 2:out.vertAlign="subscript";break;}var underline=data.read_shift(1);if(underline!=0)out.underline=underline;var family=data.read_shift(1);if(family>0)out.family=family;var bCharSet=data.read_shift(1);if(bCharSet>0)out.charset=bCharSet;data.l++;out.color=parse_BrtColor(data);switch(data.read_shift(1)){/* case 0: out.scheme = "none": break; */case 1:out.scheme="major";break;case 2:out.scheme="minor";break;}out.name=parse_XLWideString(data);return out;}/* TODO: gradient fill representation */var parse_BrtFill=parsenoop;/* [MS-XLSB] 2.4.824 BrtXF */function parse_BrtXF(data,length/*:number*/){var tgt=data.l+length;var ixfeParent=data.read_shift(2);var ifmt=data.read_shift(2);data.l=tgt;return {ixfe:ixfeParent,numFmtId:ifmt};}/* [MS-XLSB] 2.4.302 BrtBorder TODO */var parse_BrtBorder=parsenoop;/* [MS-XLSB] 2.1.7.50 Styles */function parse_sty_bin(data,themes,opts){var styles={};styles.NumberFmt=[]/*:any*/;for(var y in table_fmt)styles.NumberFmt[y]=table_fmt[y];styles.CellXf=[];styles.Fonts=[];var state/*:Array<string>*/=[];var pass=false;recordhopper(data,function hopper_sty(val,R,RT){switch(RT){case 0x002C:/* BrtFmt */styles.NumberFmt[val[0]]=val[1];SSF_load(val[1],val[0]);break;case 0x002B:/* BrtFont */styles.Fonts.push(val);if(val.color.theme!=null&&themes&&themes.themeElements&&themes.themeElements.clrScheme){val.color.rgb=rgb_tint(themes.themeElements.clrScheme[val.color.theme].rgb,val.color.tint||0);}break;case 0x0401:/* BrtKnownFonts */break;case 0x002D:/* BrtFill */break;case 0x002E:/* BrtBorder */break;case 0x002F:/* BrtXF */if(state[state.length-1]==0x0269/* BrtBeginCellXFs */){styles.CellXf.push(val);}break;case 0x0030:/* BrtStyle */case 0x01FB:/* BrtDXF */case 0x023C:/* BrtMRUColor */case 0x01DB:/* BrtIndexedColor */break;case 0x0493:/* BrtDXF14 */case 0x0836:/* BrtDXF15 */case 0x046A:/* BrtSlicerStyleElement */case 0x0200:/* BrtTableStyleElement */case 0x082F:/* BrtTimelineStyleElement */case 0x0C00:/* BrtUid */break;case 0x0023:/* BrtFRTBegin */pass=true;break;case 0x0024:/* BrtFRTEnd */pass=false;break;case 0x0025:/* BrtACBegin */state.push(RT);pass=true;break;case 0x0026:/* BrtACEnd */state.pop();pass=false;break;default:if(R.T>0)state.push(RT);else if(R.T<0)state.pop();else if(!pass||opts.WTF&&state[state.length-1]!=0x0025/* BrtACBegin */)throw new Error("Unexpected record 0x"+RT.toString(16));}});return styles;}/* Even though theme layout is dk1 lt1 dk2 lt2, true order is lt1 dk1 lt2 dk2 */var XLSXThemeClrScheme=['</a:lt1>','</a:dk1>','</a:lt2>','</a:dk2>','</a:accent1>','</a:accent2>','</a:accent3>','</a:accent4>','</a:accent5>','</a:accent6>','</a:hlink>','</a:folHlink>'];/* 20.1.6.2 clrScheme CT_ColorScheme */function parse_clrScheme(t,themes,opts){themes.themeElements.clrScheme=[];var color={};(t[0].match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(y[0]){/* 20.1.6.2 clrScheme (Color Scheme) CT_ColorScheme */case'<a:clrScheme':case'</a:clrScheme>':break;/* 20.1.2.3.32 srgbClr CT_SRgbColor */case'<a:srgbClr':color.rgb=y.val;break;/* 20.1.2.3.33 sysClr CT_SystemColor */case'<a:sysClr':color.rgb=y.lastClr;break;/* 20.1.4.1.1 accent1 (Accent 1) */ /* 20.1.4.1.2 accent2 (Accent 2) */ /* 20.1.4.1.3 accent3 (Accent 3) */ /* 20.1.4.1.4 accent4 (Accent 4) */ /* 20.1.4.1.5 accent5 (Accent 5) */ /* 20.1.4.1.6 accent6 (Accent 6) */ /* 20.1.4.1.9 dk1 (Dark 1) */ /* 20.1.4.1.10 dk2 (Dark 2) */ /* 20.1.4.1.15 folHlink (Followed Hyperlink) */ /* 20.1.4.1.19 hlink (Hyperlink) */ /* 20.1.4.1.22 lt1 (Light 1) */ /* 20.1.4.1.23 lt2 (Light 2) */case'<a:dk1>':case'</a:dk1>':case'<a:lt1>':case'</a:lt1>':case'<a:dk2>':case'</a:dk2>':case'<a:lt2>':case'</a:lt2>':case'<a:accent1>':case'</a:accent1>':case'<a:accent2>':case'</a:accent2>':case'<a:accent3>':case'</a:accent3>':case'<a:accent4>':case'</a:accent4>':case'<a:accent5>':case'</a:accent5>':case'<a:accent6>':case'</a:accent6>':case'<a:hlink>':case'</a:hlink>':case'<a:folHlink>':case'</a:folHlink>':if(y[0].charAt(1)==='/'){themes.themeElements.clrScheme[XLSXThemeClrScheme.indexOf(y[0])]=color;color={};}else {color.name=y[0].slice(3,y[0].length-1);}break;default:if(opts&&opts.WTF)throw new Error('Unrecognized '+y[0]+' in clrScheme');}});}/* 20.1.4.1.18 fontScheme CT_FontScheme */function/*::t, themes, opts*/parse_fontScheme(){}/* 20.1.4.1.15 fmtScheme CT_StyleMatrix */function/*::t, themes, opts*/parse_fmtScheme(){}var clrsregex=/<a:clrScheme([^>]*)>[\s\S]*<\/a:clrScheme>/;var fntsregex=/<a:fontScheme([^>]*)>[\s\S]*<\/a:fontScheme>/;var fmtsregex=/<a:fmtScheme([^>]*)>[\s\S]*<\/a:fmtScheme>/;/* 20.1.6.10 themeElements CT_BaseStyles */function parse_themeElements(data,themes,opts){themes.themeElements={};var t;[/* clrScheme CT_ColorScheme */['clrScheme',clrsregex,parse_clrScheme],/* fontScheme CT_FontScheme */['fontScheme',fntsregex,parse_fontScheme],/* fmtScheme CT_StyleMatrix */['fmtScheme',fmtsregex,parse_fmtScheme]].forEach(function(m){if(!(t=data.match(m[1])))throw new Error(m[0]+' not found in themeElements');m[2](t,themes,opts);});}var themeltregex=/<a:themeElements([^>]*)>[\s\S]*<\/a:themeElements>/;/* 14.2.7 Theme Part */function parse_theme_xml(data/*:string*/,opts){/* 20.1.6.9 theme CT_OfficeStyleSheet */if(!data||data.length===0)data=write_theme();var t;var themes={};/* themeElements CT_BaseStyles */if(!(t=data.match(themeltregex)))throw new Error('themeElements not found in theme');parse_themeElements(t[0],themes,opts);themes.raw=data;return themes;}function write_theme(Themes,opts)/*:string*/{if(opts&&opts.themeXLSX)return opts.themeXLSX;if(Themes&&typeof Themes.raw=="string")return Themes.raw;var o=[XML_HEADER];o[o.length]='<a:theme xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main" name="Office Theme">';o[o.length]='<a:themeElements>';o[o.length]='<a:clrScheme name="Office">';o[o.length]='<a:dk1><a:sysClr val="windowText" lastClr="000000"/></a:dk1>';o[o.length]='<a:lt1><a:sysClr val="window" lastClr="FFFFFF"/></a:lt1>';o[o.length]='<a:dk2><a:srgbClr val="1F497D"/></a:dk2>';o[o.length]='<a:lt2><a:srgbClr val="EEECE1"/></a:lt2>';o[o.length]='<a:accent1><a:srgbClr val="4F81BD"/></a:accent1>';o[o.length]='<a:accent2><a:srgbClr val="C0504D"/></a:accent2>';o[o.length]='<a:accent3><a:srgbClr val="9BBB59"/></a:accent3>';o[o.length]='<a:accent4><a:srgbClr val="8064A2"/></a:accent4>';o[o.length]='<a:accent5><a:srgbClr val="4BACC6"/></a:accent5>';o[o.length]='<a:accent6><a:srgbClr val="F79646"/></a:accent6>';o[o.length]='<a:hlink><a:srgbClr val="0000FF"/></a:hlink>';o[o.length]='<a:folHlink><a:srgbClr val="800080"/></a:folHlink>';o[o.length]='</a:clrScheme>';o[o.length]='<a:fontScheme name="Office">';o[o.length]='<a:majorFont>';o[o.length]='<a:latin typeface="Cambria"/>';o[o.length]='<a:ea typeface=""/>';o[o.length]='<a:cs typeface=""/>';o[o.length]='<a:font script="Jpan" typeface="ＭＳ Ｐゴシック"/>';o[o.length]='<a:font script="Hang" typeface="맑은 고딕"/>';o[o.length]='<a:font script="Hans" typeface="宋体"/>';o[o.length]='<a:font script="Hant" typeface="新細明體"/>';o[o.length]='<a:font script="Arab" typeface="Times New Roman"/>';o[o.length]='<a:font script="Hebr" typeface="Times New Roman"/>';o[o.length]='<a:font script="Thai" typeface="Tahoma"/>';o[o.length]='<a:font script="Ethi" typeface="Nyala"/>';o[o.length]='<a:font script="Beng" typeface="Vrinda"/>';o[o.length]='<a:font script="Gujr" typeface="Shruti"/>';o[o.length]='<a:font script="Khmr" typeface="MoolBoran"/>';o[o.length]='<a:font script="Knda" typeface="Tunga"/>';o[o.length]='<a:font script="Guru" typeface="Raavi"/>';o[o.length]='<a:font script="Cans" typeface="Euphemia"/>';o[o.length]='<a:font script="Cher" typeface="Plantagenet Cherokee"/>';o[o.length]='<a:font script="Yiii" typeface="Microsoft Yi Baiti"/>';o[o.length]='<a:font script="Tibt" typeface="Microsoft Himalaya"/>';o[o.length]='<a:font script="Thaa" typeface="MV Boli"/>';o[o.length]='<a:font script="Deva" typeface="Mangal"/>';o[o.length]='<a:font script="Telu" typeface="Gautami"/>';o[o.length]='<a:font script="Taml" typeface="Latha"/>';o[o.length]='<a:font script="Syrc" typeface="Estrangelo Edessa"/>';o[o.length]='<a:font script="Orya" typeface="Kalinga"/>';o[o.length]='<a:font script="Mlym" typeface="Kartika"/>';o[o.length]='<a:font script="Laoo" typeface="DokChampa"/>';o[o.length]='<a:font script="Sinh" typeface="Iskoola Pota"/>';o[o.length]='<a:font script="Mong" typeface="Mongolian Baiti"/>';o[o.length]='<a:font script="Viet" typeface="Times New Roman"/>';o[o.length]='<a:font script="Uigh" typeface="Microsoft Uighur"/>';o[o.length]='<a:font script="Geor" typeface="Sylfaen"/>';o[o.length]='</a:majorFont>';o[o.length]='<a:minorFont>';o[o.length]='<a:latin typeface="Calibri"/>';o[o.length]='<a:ea typeface=""/>';o[o.length]='<a:cs typeface=""/>';o[o.length]='<a:font script="Jpan" typeface="ＭＳ Ｐゴシック"/>';o[o.length]='<a:font script="Hang" typeface="맑은 고딕"/>';o[o.length]='<a:font script="Hans" typeface="宋体"/>';o[o.length]='<a:font script="Hant" typeface="新細明體"/>';o[o.length]='<a:font script="Arab" typeface="Arial"/>';o[o.length]='<a:font script="Hebr" typeface="Arial"/>';o[o.length]='<a:font script="Thai" typeface="Tahoma"/>';o[o.length]='<a:font script="Ethi" typeface="Nyala"/>';o[o.length]='<a:font script="Beng" typeface="Vrinda"/>';o[o.length]='<a:font script="Gujr" typeface="Shruti"/>';o[o.length]='<a:font script="Khmr" typeface="DaunPenh"/>';o[o.length]='<a:font script="Knda" typeface="Tunga"/>';o[o.length]='<a:font script="Guru" typeface="Raavi"/>';o[o.length]='<a:font script="Cans" typeface="Euphemia"/>';o[o.length]='<a:font script="Cher" typeface="Plantagenet Cherokee"/>';o[o.length]='<a:font script="Yiii" typeface="Microsoft Yi Baiti"/>';o[o.length]='<a:font script="Tibt" typeface="Microsoft Himalaya"/>';o[o.length]='<a:font script="Thaa" typeface="MV Boli"/>';o[o.length]='<a:font script="Deva" typeface="Mangal"/>';o[o.length]='<a:font script="Telu" typeface="Gautami"/>';o[o.length]='<a:font script="Taml" typeface="Latha"/>';o[o.length]='<a:font script="Syrc" typeface="Estrangelo Edessa"/>';o[o.length]='<a:font script="Orya" typeface="Kalinga"/>';o[o.length]='<a:font script="Mlym" typeface="Kartika"/>';o[o.length]='<a:font script="Laoo" typeface="DokChampa"/>';o[o.length]='<a:font script="Sinh" typeface="Iskoola Pota"/>';o[o.length]='<a:font script="Mong" typeface="Mongolian Baiti"/>';o[o.length]='<a:font script="Viet" typeface="Arial"/>';o[o.length]='<a:font script="Uigh" typeface="Microsoft Uighur"/>';o[o.length]='<a:font script="Geor" typeface="Sylfaen"/>';o[o.length]='</a:minorFont>';o[o.length]='</a:fontScheme>';o[o.length]='<a:fmtScheme name="Office">';o[o.length]='<a:fillStyleLst>';o[o.length]='<a:solidFill><a:schemeClr val="phClr"/></a:solidFill>';o[o.length]='<a:gradFill rotWithShape="1">';o[o.length]='<a:gsLst>';o[o.length]='<a:gs pos="0"><a:schemeClr val="phClr"><a:tint val="50000"/><a:satMod val="300000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="35000"><a:schemeClr val="phClr"><a:tint val="37000"/><a:satMod val="300000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="100000"><a:schemeClr val="phClr"><a:tint val="15000"/><a:satMod val="350000"/></a:schemeClr></a:gs>';o[o.length]='</a:gsLst>';o[o.length]='<a:lin ang="16200000" scaled="1"/>';o[o.length]='</a:gradFill>';o[o.length]='<a:gradFill rotWithShape="1">';o[o.length]='<a:gsLst>';o[o.length]='<a:gs pos="0"><a:schemeClr val="phClr"><a:tint val="100000"/><a:shade val="100000"/><a:satMod val="130000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="100000"><a:schemeClr val="phClr"><a:tint val="50000"/><a:shade val="100000"/><a:satMod val="350000"/></a:schemeClr></a:gs>';o[o.length]='</a:gsLst>';o[o.length]='<a:lin ang="16200000" scaled="0"/>';o[o.length]='</a:gradFill>';o[o.length]='</a:fillStyleLst>';o[o.length]='<a:lnStyleLst>';o[o.length]='<a:ln w="9525" cap="flat" cmpd="sng" algn="ctr"><a:solidFill><a:schemeClr val="phClr"><a:shade val="95000"/><a:satMod val="105000"/></a:schemeClr></a:solidFill><a:prstDash val="solid"/></a:ln>';o[o.length]='<a:ln w="25400" cap="flat" cmpd="sng" algn="ctr"><a:solidFill><a:schemeClr val="phClr"/></a:solidFill><a:prstDash val="solid"/></a:ln>';o[o.length]='<a:ln w="38100" cap="flat" cmpd="sng" algn="ctr"><a:solidFill><a:schemeClr val="phClr"/></a:solidFill><a:prstDash val="solid"/></a:ln>';o[o.length]='</a:lnStyleLst>';o[o.length]='<a:effectStyleLst>';o[o.length]='<a:effectStyle>';o[o.length]='<a:effectLst>';o[o.length]='<a:outerShdw blurRad="40000" dist="20000" dir="5400000" rotWithShape="0"><a:srgbClr val="000000"><a:alpha val="38000"/></a:srgbClr></a:outerShdw>';o[o.length]='</a:effectLst>';o[o.length]='</a:effectStyle>';o[o.length]='<a:effectStyle>';o[o.length]='<a:effectLst>';o[o.length]='<a:outerShdw blurRad="40000" dist="23000" dir="5400000" rotWithShape="0"><a:srgbClr val="000000"><a:alpha val="35000"/></a:srgbClr></a:outerShdw>';o[o.length]='</a:effectLst>';o[o.length]='</a:effectStyle>';o[o.length]='<a:effectStyle>';o[o.length]='<a:effectLst>';o[o.length]='<a:outerShdw blurRad="40000" dist="23000" dir="5400000" rotWithShape="0"><a:srgbClr val="000000"><a:alpha val="35000"/></a:srgbClr></a:outerShdw>';o[o.length]='</a:effectLst>';o[o.length]='<a:scene3d><a:camera prst="orthographicFront"><a:rot lat="0" lon="0" rev="0"/></a:camera><a:lightRig rig="threePt" dir="t"><a:rot lat="0" lon="0" rev="1200000"/></a:lightRig></a:scene3d>';o[o.length]='<a:sp3d><a:bevelT w="63500" h="25400"/></a:sp3d>';o[o.length]='</a:effectStyle>';o[o.length]='</a:effectStyleLst>';o[o.length]='<a:bgFillStyleLst>';o[o.length]='<a:solidFill><a:schemeClr val="phClr"/></a:solidFill>';o[o.length]='<a:gradFill rotWithShape="1">';o[o.length]='<a:gsLst>';o[o.length]='<a:gs pos="0"><a:schemeClr val="phClr"><a:tint val="40000"/><a:satMod val="350000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="40000"><a:schemeClr val="phClr"><a:tint val="45000"/><a:shade val="99000"/><a:satMod val="350000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="100000"><a:schemeClr val="phClr"><a:shade val="20000"/><a:satMod val="255000"/></a:schemeClr></a:gs>';o[o.length]='</a:gsLst>';o[o.length]='<a:path path="circle"><a:fillToRect l="50000" t="-80000" r="50000" b="180000"/></a:path>';o[o.length]='</a:gradFill>';o[o.length]='<a:gradFill rotWithShape="1">';o[o.length]='<a:gsLst>';o[o.length]='<a:gs pos="0"><a:schemeClr val="phClr"><a:tint val="80000"/><a:satMod val="300000"/></a:schemeClr></a:gs>';o[o.length]='<a:gs pos="100000"><a:schemeClr val="phClr"><a:shade val="30000"/><a:satMod val="200000"/></a:schemeClr></a:gs>';o[o.length]='</a:gsLst>';o[o.length]='<a:path path="circle"><a:fillToRect l="50000" t="50000" r="50000" b="50000"/></a:path>';o[o.length]='</a:gradFill>';o[o.length]='</a:bgFillStyleLst>';o[o.length]='</a:fmtScheme>';o[o.length]='</a:themeElements>';o[o.length]='<a:objectDefaults>';o[o.length]='<a:spDef>';o[o.length]='<a:spPr/><a:bodyPr/><a:lstStyle/><a:style><a:lnRef idx="1"><a:schemeClr val="accent1"/></a:lnRef><a:fillRef idx="3"><a:schemeClr val="accent1"/></a:fillRef><a:effectRef idx="2"><a:schemeClr val="accent1"/></a:effectRef><a:fontRef idx="minor"><a:schemeClr val="lt1"/></a:fontRef></a:style>';o[o.length]='</a:spDef>';o[o.length]='<a:lnDef>';o[o.length]='<a:spPr/><a:bodyPr/><a:lstStyle/><a:style><a:lnRef idx="2"><a:schemeClr val="accent1"/></a:lnRef><a:fillRef idx="0"><a:schemeClr val="accent1"/></a:fillRef><a:effectRef idx="1"><a:schemeClr val="accent1"/></a:effectRef><a:fontRef idx="minor"><a:schemeClr val="tx1"/></a:fontRef></a:style>';o[o.length]='</a:lnDef>';o[o.length]='</a:objectDefaults>';o[o.length]='<a:extraClrSchemeLst/>';o[o.length]='</a:theme>';return o.join("");}/* [MS-XLS] 2.4.326 TODO: payload is a zip file */function parse_Theme(blob,length,opts){var end=blob.l+length;var dwThemeVersion=blob.read_shift(4);if(dwThemeVersion===124226)return;if(!opts.cellStyles){blob.l=end;return;}var data=blob.slice(blob.l);blob.l=end;var zip;try{zip=zip_read(data,{type:"array"});}catch(e){return;}var themeXML=getzipstr(zip,"theme/theme/theme1.xml",true);if(!themeXML)return;return parse_theme_xml(themeXML,opts);}/* 2.5.49 */function parse_ColorTheme(blob/*::, length*/){return blob.read_shift(4);}/* 2.5.155 */function parse_FullColorExt(blob/*::, length*/){var o={};o.xclrType=blob.read_shift(2);o.nTintShade=blob.read_shift(2);switch(o.xclrType){case 0:blob.l+=4;break;case 1:o.xclrValue=parse_IcvXF(blob,4);break;case 2:o.xclrValue=parse_LongRGBA(blob);break;case 3:o.xclrValue=parse_ColorTheme(blob);break;case 4:blob.l+=4;break;}blob.l+=8;return o;}/* 2.5.164 TODO: read 7 bits*/function parse_IcvXF(blob,length){return parsenoop(blob,length);}/* 2.5.280 */function parse_XFExtGradient(blob,length){return parsenoop(blob,length);}/* [MS-XLS] 2.5.108 */function parse_ExtProp(blob/*::, length*/)/*:Array<any>*/{var extType=blob.read_shift(2);var cb=blob.read_shift(2)-4;var o=[extType];switch(extType){case 0x04:case 0x05:case 0x07:case 0x08:case 0x09:case 0x0A:case 0x0B:case 0x0D:o[1]=parse_FullColorExt(blob);break;case 0x06:o[1]=parse_XFExtGradient(blob,cb);break;case 0x0E:case 0x0F:o[1]=blob.read_shift(cb===1?1:2);break;default:throw new Error("Unrecognized ExtProp type: "+extType+" "+cb);}return o;}/* 2.4.355 */function parse_XFExt(blob,length){var end=blob.l+length;blob.l+=2;var ixfe=blob.read_shift(2);blob.l+=2;var cexts=blob.read_shift(2);var ext/*:AOA*/=[];while(cexts-->0)ext.push(parse_ExtProp(blob,end-blob.l));return {ixfe:ixfe,ext:ext};}/* xf is an XF, see parse_XFExt for xfext */function update_xfext(xf,xfext){xfext.forEach(function(xfe){switch(xfe[0]){/* 2.5.108 extPropData *//* indentation level */}});}function parse_BrtMdtinfo(data,length){return {flags:data.read_shift(4),version:data.read_shift(4),name:parse_XLWideString(data)};}function parse_BrtMdb(data){var out=[];var cnt=data.read_shift(4);while(cnt-->0)out.push([data.read_shift(4),data.read_shift(4)]);return out;}function parse_BrtBeginEsmdb(data){data.l+=4;return data.read_shift(4)!=0;}function parse_xlmeta_bin(data,name,_opts){var out={Types:[],Cell:[],Value:[]};var opts=_opts||{};var state=[];var pass=false;var metatype=2;recordhopper(data,function(val,R,RT){switch(RT){case 335:out.Types.push({name:val.name});break;case 51:val.forEach(function(r){if(metatype==1)out.Cell.push({type:out.Types[r[0]-1].name,index:r[1]});else if(metatype==0)out.Value.push({type:out.Types[r[0]-1].name,index:r[1]});});break;case 337:metatype=val?1:0;break;case 338:metatype=2;break;case 35:state.push(RT);pass=true;break;case 36:state.pop();pass=false;break;default:if(R.T);else if(!pass||opts.WTF&&state[state.length-1]!=35)throw new Error("Unexpected record 0x"+RT.toString(16));}});return out;}function parse_xlmeta_xml(data,name,opts){var out={Types:[],Cell:[],Value:[]};if(!data)return out;var pass=false;var metatype=2;var lastmeta;data.replace(tagregex,function(x){var y=parsexmltag(x);switch(strip_ns(y[0])){case"<?xml":break;case"<metadata":case"</metadata>":break;case"<metadataTypes":case"</metadataTypes>":break;case"<metadataType":out.Types.push({name:y.name});break;case"</metadataType>":break;case"<futureMetadata":for(var j=0;j<out.Types.length;++j)if(out.Types[j].name==y.name)lastmeta=out.Types[j];break;case"</futureMetadata>":break;case"<bk>":break;case"</bk>":break;case"<rc":if(metatype==1)out.Cell.push({type:out.Types[y.t-1].name,index:+y.v});else if(metatype==0)out.Value.push({type:out.Types[y.t-1].name,index:+y.v});break;case"</rc>":break;case"<cellMetadata":metatype=1;break;case"</cellMetadata>":metatype=2;break;case"<valueMetadata":metatype=0;break;case"</valueMetadata>":metatype=2;break;case"<extLst":case"<extLst>":case"</extLst>":case"<extLst/>":break;case"<ext":pass=true;break;case"</ext>":pass=false;break;case"<rvb":if(!lastmeta)break;if(!lastmeta.offsets)lastmeta.offsets=[];lastmeta.offsets.push(+y.i);break;default:if(!pass&&opts.WTF)throw new Error("unrecognized "+y[0]+" in metadata");}return x;});return out;}/* 18.6 Calculation Chain */function parse_cc_xml(data/*::, name, opts*/)/*:Array<any>*/{var d=[];if(!data)return d;var i=1;(data.match(tagregex)||[]).forEach(function(x){var y=parsexmltag(x);switch(y[0]){case'<?xml':break;/* 18.6.2  calcChain CT_CalcChain 1 */case'<calcChain':case'<calcChain>':case'</calcChain>':break;/* 18.6.1  c CT_CalcCell 1 */case'<c':delete y[0];if(y.i)i=y.i;else y.i=i;d.push(y);break;}});return d;}//function write_cc_xml(data, opts) { }
	/* [MS-XLSB] 2.6.4.1 */function parse_BrtCalcChainItem$(data){var out={};out.i=data.read_shift(4);var cell={};cell.r=data.read_shift(4);cell.c=data.read_shift(4);out.r=encode_cell(cell);var flags=data.read_shift(1);if(flags&0x2)out.l='1';if(flags&0x8)out.a='1';return out;}/* 18.6 Calculation Chain */function parse_cc_bin(data,name,opts){var out=[];recordhopper(data,function hopper_cc(val,R,RT){switch(RT){case 0x003F:/* 'BrtCalcChainItem$' */out.push(val);break;default:if(R.T);else throw new Error("Unexpected record 0x"+RT.toString(16));}});return out;}//function write_cc_bin(data, opts) { }
	/* 18.14 Supplementary Workbook Data */function/*::data, rel, name:string, _opts*/parse_xlink_xml(){//var opts = _opts || {};
	//if(opts.WTF) throw "XLSX External Link";
	}/* [MS-XLSB] 2.1.7.25 External Link */function parse_xlink_bin(data,rel,name/*:string*/,_opts){if(!data)return data;var opts=_opts||{};var pass=false;recordhopper(data,function xlink_parse(val,R,RT){switch(RT){case 0x0167:/* 'BrtSupTabs' */case 0x016B:/* 'BrtExternTableStart' */case 0x016C:/* 'BrtExternTableEnd' */case 0x016E:/* 'BrtExternRowHdr' */case 0x016F:/* 'BrtExternCellBlank' */case 0x0170:/* 'BrtExternCellReal' */case 0x0171:/* 'BrtExternCellBool' */case 0x0172:/* 'BrtExternCellError' */case 0x0173:/* 'BrtExternCellString' */case 0x01D8:/* 'BrtExternValueMeta' */case 0x0241:/* 'BrtSupNameStart' */case 0x0242:/* 'BrtSupNameValueStart' */case 0x0243:/* 'BrtSupNameValueEnd' */case 0x0244:/* 'BrtSupNameNum' */case 0x0245:/* 'BrtSupNameErr' */case 0x0246:/* 'BrtSupNameSt' */case 0x0247:/* 'BrtSupNameNil' */case 0x0248:/* 'BrtSupNameBool' */case 0x0249:/* 'BrtSupNameFmla' */case 0x024A:/* 'BrtSupNameBits' */case 0x024B:/* 'BrtSupNameEnd' */break;case 0x0023:/* 'BrtFRTBegin' */pass=true;break;case 0x0024:/* 'BrtFRTEnd' */pass=false;break;default:if(R.T);else if(!pass||opts.WTF)throw new Error("Unexpected record 0x"+RT.toString(16));}},opts);}/* 20.5 DrawingML - SpreadsheetML Drawing */ /* 20.5.2.35 wsDr CT_Drawing */function parse_drawing(data,rels/*:any*/){if(!data)return "??";/*
		  Chartsheet Drawing:
		   - 20.5.2.35 wsDr CT_Drawing
		    - 20.5.2.1  absoluteAnchor CT_AbsoluteAnchor
		     - 20.5.2.16 graphicFrame CT_GraphicalObjectFrame
		      - 20.1.2.2.16 graphic CT_GraphicalObject
		       - 20.1.2.2.17 graphicData CT_GraphicalObjectData
	          - chart reference
		   the actual type is based on the URI of the graphicData
			TODO: handle embedded charts and other types of graphics
		*/var id=(data.match(/<c:chart [^>]*r:id="([^"]*)"/)||["",""])[1];return rels['!id'][id].Target;}function sheet_insert_comments(sheet,comments/*:Array<RawComment>*/,threaded/*:boolean*/,people/*:?Array<any>*/){var dense=Array.isArray(sheet);var cell/*:Cell*/;comments.forEach(function(comment){var r=decode_cell(comment.ref);if(dense){if(!sheet[r.r])sheet[r.r]=[];cell=sheet[r.r][r.c];}else cell=sheet[comment.ref];if(!cell){cell={t:"z"}/*:any*/;if(dense)sheet[r.r][r.c]=cell;else sheet[comment.ref]=cell;var range=safe_decode_range(sheet["!ref"]||"BDWGO1000001:A1");if(range.s.r>r.r)range.s.r=r.r;if(range.e.r<r.r)range.e.r=r.r;if(range.s.c>r.c)range.s.c=r.c;if(range.e.c<r.c)range.e.c=r.c;var encoded=encode_range(range);if(encoded!==sheet["!ref"])sheet["!ref"]=encoded;}if(!cell.c)cell.c=[];var o/*:Comment*/={a:comment.author,t:comment.t,r:comment.r,T:threaded};if(comment.h)o.h=comment.h;/* threaded comments always override */for(var i=cell.c.length-1;i>=0;--i){if(!threaded&&cell.c[i].T)return;if(threaded&&!cell.c[i].T)cell.c.splice(i,1);}if(threaded&&people)for(i=0;i<people.length;++i){if(o.a==people[i].id){o.a=people[i].name||o.a;break;}}cell.c.push(o);});}/* 18.7 Comments */function parse_comments_xml(data/*:string*/,opts)/*:Array<RawComment>*/{/* 18.7.6 CT_Comments */if(data.match(/<(?:\w+:)?comments *\/>/))return [];var authors/*:Array<string>*/=[];var commentList/*:Array<RawComment>*/=[];var authtag=data.match(/<(?:\w+:)?authors>([\s\S]*)<\/(?:\w+:)?authors>/);if(authtag&&authtag[1])authtag[1].split(/<\/\w*:?author>/).forEach(function(x){if(x===""||x.trim()==="")return;var a=x.match(/<(?:\w+:)?author[^>]*>(.*)/);if(a)authors.push(a[1]);});var cmnttag=data.match(/<(?:\w+:)?commentList>([\s\S]*)<\/(?:\w+:)?commentList>/);if(cmnttag&&cmnttag[1])cmnttag[1].split(/<\/\w*:?comment>/).forEach(function(x){if(x===""||x.trim()==="")return;var cm=x.match(/<(?:\w+:)?comment[^>]*>/);if(!cm)return;var y=parsexmltag(cm[0]);var comment/*:RawComment*/={author:y.authorId&&authors[y.authorId]||"sheetjsghost",ref:y.ref,guid:y.guid}/*:any*/;var cell=decode_cell(y.ref);if(opts.sheetRows&&opts.sheetRows<=cell.r)return;var textMatch=x.match(/<(?:\w+:)?text>([\s\S]*)<\/(?:\w+:)?text>/);var rt=!!textMatch&&!!textMatch[1]&&parse_si(textMatch[1])||{r:"",t:"",h:""};comment.r=rt.r;if(rt.r=="<t></t>")rt.t=rt.h="";comment.t=(rt.t||"").replace(/\r\n/g,"\n").replace(/\r/g,"\n");if(opts.cellHTML)comment.h=rt.h;commentList.push(comment);});return commentList;}/* [MS-XLSX] 2.1.17 */function parse_tcmnt_xml(data/*:string*/,opts)/*:Array<RawComment>*/{var out=[];var pass=false,comment={},tidx=0;data.replace(tagregex,function xml_tcmnt(x,idx){var y/*:any*/=parsexmltag(x);switch(strip_ns(y[0])){case'<?xml':break;/* 2.6.207 ThreadedComments CT_ThreadedComments */case'<ThreadedComments':break;case'</ThreadedComments>':break;/* 2.6.205 threadedComment CT_ThreadedComment */case'<threadedComment':comment={author:y.personId,guid:y.id,ref:y.ref,T:1};break;case'</threadedComment>':if(comment.t!=null)out.push(comment);break;case'<text>':case'<text':tidx=idx+x.length;break;case'</text>':comment.t=data.slice(tidx,idx).replace(/\r\n/g,"\n").replace(/\r/g,"\n");break;/* 2.6.206 mentions CT_ThreadedCommentMentions TODO */case'<mentions':case'<mentions>':pass=true;break;case'</mentions>':pass=false;break;/* 2.6.202 mention CT_Mention TODO */ /* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':case'<extLst/>':break;/* 18.2.7  ext CT_Extension + */case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(!pass&&opts.WTF)throw new Error('unrecognized '+y[0]+' in threaded comments');}return x;});return out;}/* [MS-XLSX] 2.1.18 */function parse_people_xml(data/*:string*/,opts){var out=[];var pass=false;data.replace(tagregex,function xml_tcmnt(x){var y/*:any*/=parsexmltag(x);switch(strip_ns(y[0])){case'<?xml':break;/* 2.4.85 personList CT_PersonList */case'<personList':break;case'</personList>':break;/* 2.6.203 person CT_Person TODO: providers */case'<person':out.push({name:y.displayname,id:y.id});break;case'</person>':break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':case'<extLst/>':break;/* 18.2.7  ext CT_Extension + */case'<ext':pass=true;break;case'</ext>':pass=false;break;default:if(!pass&&opts.WTF)throw new Error('unrecognized '+y[0]+' in threaded comments');}return x;});return out;}/* [MS-XLSB] 2.4.28 BrtBeginComment */function parse_BrtBeginComment(data){var out={};out.iauthor=data.read_shift(4);var rfx=parse_UncheckedRfX(data);out.rfx=rfx.s;out.ref=encode_cell(rfx.s);data.l+=16;/*var guid = parse_GUID(data); */return out;}/* [MS-XLSB] 2.4.327 BrtCommentAuthor */var parse_BrtCommentAuthor=parse_XLWideString;/* [MS-XLSB] 2.1.7.8 Comments */function parse_comments_bin(data,opts)/*:Array<RawComment>*/{var out/*:Array<RawComment>*/=[];var authors/*:Array<string>*/=[];var c={};var pass=false;recordhopper(data,function hopper_cmnt(val,R,RT){switch(RT){case 0x0278:/* 'BrtCommentAuthor' */authors.push(val);break;case 0x027B:/* 'BrtBeginComment' */c=val;break;case 0x027D:/* 'BrtCommentText' */c.t=val.t;c.h=val.h;c.r=val.r;break;case 0x027C:/* 'BrtEndComment' */c.author=authors[c.iauthor];delete c/*:any*/.iauthor;if(opts.sheetRows&&c.rfx&&opts.sheetRows<=c.rfx.r)break;if(!c.t)c.t="";delete c.rfx;out.push(c);break;case 0x0C00:/* 'BrtUid' */break;case 0x0023:/* 'BrtFRTBegin' */pass=true;break;case 0x0024:/* 'BrtFRTEnd' */pass=false;break;case 0x0025:/* 'BrtACBegin' */break;case 0x0026:/* 'BrtACEnd' */break;default:if(R.T);else if(!pass||opts.WTF)throw new Error("Unexpected record 0x"+RT.toString(16));}});return out;}var CT_VBA="application/vnd.ms-office.vbaProject";function make_vba_xls(cfb){var newcfb=CFB.utils.cfb_new({root:"R"});cfb.FullPaths.forEach(function(p,i){if(p.slice(-1)==="/"||!p.match(/_VBA_PROJECT_CUR/))return;var newpath=p.replace(/^[^\/]*/,"R").replace(/\/_VBA_PROJECT_CUR\u0000*/,"");CFB.utils.cfb_add(newcfb,newpath,cfb.FileIndex[i].content);});return CFB.write(newcfb);}/* macro and dialog sheet stubs */function/*::data:any, opts, idx:number, rels, wb, themes, styles*/parse_ds_bin()/*:Worksheet*/{return {'!type':'dialog'};}function/*::data:any, opts, idx:number, rels, wb, themes, styles*/parse_ds_xml()/*:Worksheet*/{return {'!type':'dialog'};}function/*::data:any, opts, idx:number, rels, wb, themes, styles*/parse_ms_bin()/*:Worksheet*/{return {'!type':'macro'};}function/*::data:any, opts, idx:number, rels, wb, themes, styles*/parse_ms_xml()/*:Worksheet*/{return {'!type':'macro'};}/* TODO: it will be useful to parse the function str */var rc_to_a1=/*#__PURE__*/function(){var rcregex=/(^|[^A-Za-z_])R(\[?-?\d+\]|[1-9]\d*|)C(\[?-?\d+\]|[1-9]\d*|)(?![A-Za-z0-9_])/g;var rcbase/*:Cell*/={r:0,c:0}/*:any*/;function rcfunc($$,$1,$2,$3){var cRel=false,rRel=false;if($2.length==0)rRel=true;else if($2.charAt(0)=="["){rRel=true;$2=$2.slice(1,-1);}if($3.length==0)cRel=true;else if($3.charAt(0)=="["){cRel=true;$3=$3.slice(1,-1);}var R=$2.length>0?parseInt($2,10)|0:0,C=$3.length>0?parseInt($3,10)|0:0;if(cRel)C+=rcbase.c;else --C;if(rRel)R+=rcbase.r;else --R;return $1+(cRel?"":"$")+encode_col(C)+(rRel?"":"$")+encode_row(R);}return function rc_to_a1(fstr/*:string*/,base/*:Cell*/)/*:string*/{rcbase=base;return fstr.replace(rcregex,rcfunc);};}();var crefregex=/(^|[^._A-Z0-9])([$]?)([A-Z]{1,2}|[A-W][A-Z]{2}|X[A-E][A-Z]|XF[A-D])([$]?)(10[0-3]\d{4}|104[0-7]\d{3}|1048[0-4]\d{2}|10485[0-6]\d|104857[0-6]|[1-9]\d{0,5})(?![_.\(A-Za-z0-9])/g;var a1_to_rc=/*#__PURE__*/function(){return function a1_to_rc(fstr/*:string*/,base/*:CellAddress*/){return fstr.replace(crefregex,function($0,$1,$2,$3,$4,$5){var c=decode_col($3)-($2?0:base.c);var r=decode_row($5)-($4?0:base.r);var R=r==0?"":!$4?"["+r+"]":r+1;var C=c==0?"":!$2?"["+c+"]":c+1;return $1+"R"+R+"C"+C;});};}();/* no defined name can collide with a valid cell address A1:XFD1048576 ... except LOG10! */function shift_formula_str(f/*:string*/,delta/*:Cell*/)/*:string*/{return f.replace(crefregex,function($0,$1,$2,$3,$4,$5){return $1+($2=="$"?$2+$3:encode_col(decode_col($3)+delta.c))+($4=="$"?$4+$5:encode_row(decode_row($5)+delta.r));});}function shift_formula_xlsx(f/*:string*/,range/*:string*/,cell/*:string*/)/*:string*/{var r=decode_range(range),s=r.s,c=decode_cell(cell);var delta={r:c.r-s.r,c:c.c-s.c};return shift_formula_str(f,delta);}/* TODO: parse formula */function fuzzyfmla(f/*:string*/)/*:boolean*/{if(f.length==1)return false;return true;}function _xlfn(f/*:string*/)/*:string*/{return f.replace(/_xlfn\./g,"");}function parseread1(blob){blob.l+=1;return;}/* [MS-XLS] 2.5.51 */function parse_ColRelU(blob,length){var c=blob.read_shift(length==1?1:2);return [c&0x3FFF,c>>14&1,c>>15&1];}/* [MS-XLS] 2.5.198.105 ; [MS-XLSB] 2.5.97.89 */function parse_RgceArea(blob,length,opts){var w=2;if(opts){if(opts.biff>=2&&opts.biff<=5)return parse_RgceArea_BIFF2(blob);else if(opts.biff==12)w=4;}var r=blob.read_shift(w),R=blob.read_shift(w);var c=parse_ColRelU(blob,2);var C=parse_ColRelU(blob,2);return {s:{r:r,c:c[0],cRel:c[1],rRel:c[2]},e:{r:R,c:C[0],cRel:C[1],rRel:C[2]}};}/* BIFF 2-5 encodes flags in the row field */function parse_RgceArea_BIFF2(blob/*::, length, opts*/){var r=parse_ColRelU(blob,2),R=parse_ColRelU(blob,2);var c=blob.read_shift(1);var C=blob.read_shift(1);return {s:{r:r[0],c:c,cRel:r[1],rRel:r[2]},e:{r:R[0],c:C,cRel:R[1],rRel:R[2]}};}/* [MS-XLS] 2.5.198.105 ; [MS-XLSB] 2.5.97.90 */function parse_RgceAreaRel(blob,length,opts){if(opts.biff<8)return parse_RgceArea_BIFF2(blob);var r=blob.read_shift(opts.biff==12?4:2),R=blob.read_shift(opts.biff==12?4:2);var c=parse_ColRelU(blob,2);var C=parse_ColRelU(blob,2);return {s:{r:r,c:c[0],cRel:c[1],rRel:c[2]},e:{r:R,c:C[0],cRel:C[1],rRel:C[2]}};}/* [MS-XLS] 2.5.198.109 ; [MS-XLSB] 2.5.97.91 */function parse_RgceLoc(blob,length,opts){if(opts&&opts.biff>=2&&opts.biff<=5)return parse_RgceLoc_BIFF2(blob);var r=blob.read_shift(opts&&opts.biff==12?4:2);var c=parse_ColRelU(blob,2);return {r:r,c:c[0],cRel:c[1],rRel:c[2]};}function parse_RgceLoc_BIFF2(blob/*::, length, opts*/){var r=parse_ColRelU(blob,2);var c=blob.read_shift(1);return {r:r[0],c:c,cRel:r[1],rRel:r[2]};}/* [MS-XLS] 2.5.198.107, 2.5.47 */function parse_RgceElfLoc(blob/*::, length, opts*/){var r=blob.read_shift(2);var c=blob.read_shift(2);return {r:r,c:c&0xFF,fQuoted:!!(c&0x4000),cRel:c>>15,rRel:c>>15};}/* [MS-XLS] 2.5.198.111 ; [MS-XLSB] 2.5.97.92 TODO */function parse_RgceLocRel(blob,length,opts){var biff=opts&&opts.biff?opts.biff:8;if(biff>=2&&biff<=5)return parse_RgceLocRel_BIFF2(blob);var r=blob.read_shift(biff>=12?4:2);var cl=blob.read_shift(2);var cRel=(cl&0x4000)>>14,rRel=(cl&0x8000)>>15;cl&=0x3FFF;if(rRel==1)while(r>0x7FFFF)r-=0x100000;if(cRel==1)while(cl>0x1FFF)cl=cl-0x4000;return {r:r,c:cl,cRel:cRel,rRel:rRel};}function parse_RgceLocRel_BIFF2(blob/*::, length:number, opts*/){var rl=blob.read_shift(2);var c=blob.read_shift(1);var rRel=(rl&0x8000)>>15,cRel=(rl&0x4000)>>14;rl&=0x3FFF;if(rRel==1&&rl>=0x2000)rl=rl-0x4000;if(cRel==1&&c>=0x80)c=c-0x100;return {r:rl,c:c,cRel:cRel,rRel:rRel};}/* [MS-XLS] 2.5.198.27 ; [MS-XLSB] 2.5.97.18 */function parse_PtgArea(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;var area=parse_RgceArea(blob,opts.biff>=2&&opts.biff<=5?6:8,opts);return [type,area];}/* [MS-XLS] 2.5.198.28 ; [MS-XLSB] 2.5.97.19 */function parse_PtgArea3d(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;var ixti=blob.read_shift(2,'i');var w=8;if(opts)switch(opts.biff){case 5:blob.l+=12;w=6;break;case 12:w=12;break;}var area=parse_RgceArea(blob,w,opts);return [type,ixti,area];}/* [MS-XLS] 2.5.198.29 ; [MS-XLSB] 2.5.97.20 */function parse_PtgAreaErr(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;blob.l+=opts&&opts.biff>8?12:opts.biff<8?6:8;return [type];}/* [MS-XLS] 2.5.198.30 ; [MS-XLSB] 2.5.97.21 */function parse_PtgAreaErr3d(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;var ixti=blob.read_shift(2);var w=8;if(opts)switch(opts.biff){case 5:blob.l+=12;w=6;break;case 12:w=12;break;}blob.l+=w;return [type,ixti];}/* [MS-XLS] 2.5.198.31 ; [MS-XLSB] 2.5.97.22 */function parse_PtgAreaN(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;var area=parse_RgceAreaRel(blob,length-1,opts);return [type,area];}/* [MS-XLS] 2.5.198.32 ; [MS-XLSB] 2.5.97.23 */function parse_PtgArray(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;blob.l+=opts.biff==2?6:opts.biff==12?14:7;return [type];}/* [MS-XLS] 2.5.198.33 ; [MS-XLSB] 2.5.97.24 */function parse_PtgAttrBaxcel(blob){var bitSemi=blob[blob.l+1]&0x01;/* 1 = volatile */var bitBaxcel=1;blob.l+=4;return [bitSemi,bitBaxcel];}/* [MS-XLS] 2.5.198.34 ; [MS-XLSB] 2.5.97.25 */function parse_PtgAttrChoose(blob,length,opts)/*:Array<number>*/{blob.l+=2;var offset=blob.read_shift(opts&&opts.biff==2?1:2);var o/*:Array<number>*/=[];/* offset is 1 less than the number of elements */for(var i=0;i<=offset;++i)o.push(blob.read_shift(opts&&opts.biff==2?1:2));return o;}/* [MS-XLS] 2.5.198.35 ; [MS-XLSB] 2.5.97.26 */function parse_PtgAttrGoto(blob,length,opts){var bitGoto=blob[blob.l+1]&0xFF?1:0;blob.l+=2;return [bitGoto,blob.read_shift(opts&&opts.biff==2?1:2)];}/* [MS-XLS] 2.5.198.36 ; [MS-XLSB] 2.5.97.27 */function parse_PtgAttrIf(blob,length,opts){var bitIf=blob[blob.l+1]&0xFF?1:0;blob.l+=2;return [bitIf,blob.read_shift(opts&&opts.biff==2?1:2)];}/* [MS-XLSB] 2.5.97.28 */function parse_PtgAttrIfError(blob){var bitIf=blob[blob.l+1]&0xFF?1:0;blob.l+=2;return [bitIf,blob.read_shift(2)];}/* [MS-XLS] 2.5.198.37 ; [MS-XLSB] 2.5.97.29 */function parse_PtgAttrSemi(blob,length,opts){var bitSemi=blob[blob.l+1]&0xFF?1:0;blob.l+=opts&&opts.biff==2?3:4;return [bitSemi];}/* [MS-XLS] 2.5.198.40 ; [MS-XLSB] 2.5.97.32 */function parse_PtgAttrSpaceType(blob/*::, length*/){var type=blob.read_shift(1),cch=blob.read_shift(1);return [type,cch];}/* [MS-XLS] 2.5.198.38 ; [MS-XLSB] 2.5.97.30 */function parse_PtgAttrSpace(blob){blob.read_shift(2);return parse_PtgAttrSpaceType(blob);}/* [MS-XLS] 2.5.198.39 ; [MS-XLSB] 2.5.97.31 */function parse_PtgAttrSpaceSemi(blob){blob.read_shift(2);return parse_PtgAttrSpaceType(blob);}/* [MS-XLS] 2.5.198.84 ; [MS-XLSB] 2.5.97.68 TODO */function parse_PtgRef(blob,length,opts){//var ptg = blob[blob.l] & 0x1F;
	var type=(blob[blob.l]&0x60)>>5;blob.l+=1;var loc=parse_RgceLoc(blob,0,opts);return [type,loc];}/* [MS-XLS] 2.5.198.88 ; [MS-XLSB] 2.5.97.72 TODO */function parse_PtgRefN(blob,length,opts){var type=(blob[blob.l]&0x60)>>5;blob.l+=1;var loc=parse_RgceLocRel(blob,0,opts);return [type,loc];}/* [MS-XLS] 2.5.198.85 ; [MS-XLSB] 2.5.97.69 TODO */function parse_PtgRef3d(blob,length,opts){var type=(blob[blob.l]&0x60)>>5;blob.l+=1;var ixti=blob.read_shift(2);// XtiIndex
	if(opts&&opts.biff==5)blob.l+=12;var loc=parse_RgceLoc(blob,0,opts);// TODO: or RgceLocRel
	return [type,ixti,loc];}/* [MS-XLS] 2.5.198.62 ; [MS-XLSB] 2.5.97.45 TODO */function parse_PtgFunc(blob,length,opts){//var ptg = blob[blob.l] & 0x1F;
	var type=(blob[blob.l]&0x60)>>5;blob.l+=1;var iftab=blob.read_shift(opts&&opts.biff<=3?1:2);return [FtabArgc[iftab],Ftab[iftab],type];}/* [MS-XLS] 2.5.198.63 ; [MS-XLSB] 2.5.97.46 TODO */function parse_PtgFuncVar(blob,length,opts){var type=blob[blob.l++];var cparams=blob.read_shift(1),tab=opts&&opts.biff<=3?[type==0x58?-1:0,blob.read_shift(1)]:parsetab(blob);return [cparams,(tab[0]===0?Ftab:Cetab)[tab[1]]];}function parsetab(blob){return [blob[blob.l+1]>>7,blob.read_shift(2)&0x7FFF];}/* [MS-XLS] 2.5.198.41 ; [MS-XLSB] 2.5.97.33 */function parse_PtgAttrSum(blob,length,opts){blob.l+=opts&&opts.biff==2?3:4;return;}/* [MS-XLS] 2.5.198.58 ; [MS-XLSB] 2.5.97.40 */function parse_PtgExp(blob,length,opts){blob.l++;if(opts&&opts.biff==12)return [blob.read_shift(4,'i'),0];var row=blob.read_shift(2);var col=blob.read_shift(opts&&opts.biff==2?1:2);return [row,col];}/* [MS-XLS] 2.5.198.57 ; [MS-XLSB] 2.5.97.39 */function parse_PtgErr(blob){blob.l++;return BErr[blob.read_shift(1)];}/* [MS-XLS] 2.5.198.66 ; [MS-XLSB] 2.5.97.49 */function parse_PtgInt(blob){blob.l++;return blob.read_shift(2);}/* [MS-XLS] 2.5.198.42 ; [MS-XLSB] 2.5.97.34 */function parse_PtgBool(blob){blob.l++;return blob.read_shift(1)!==0;}/* [MS-XLS] 2.5.198.79 ; [MS-XLSB] 2.5.97.63 */function parse_PtgNum(blob){blob.l++;return parse_Xnum(blob);}/* [MS-XLS] 2.5.198.89 ; [MS-XLSB] 2.5.97.74 */function parse_PtgStr(blob,length,opts){blob.l++;return parse_ShortXLUnicodeString(blob,length-1,opts);}/* [MS-XLS] 2.5.192.112 + 2.5.192.11{3,4,5,6,7} */ /* [MS-XLSB] 2.5.97.93 + 2.5.97.9{4,5,6,7} */function parse_SerAr(blob,biff/*:number*/){var val=[blob.read_shift(1)];if(biff==12)switch(val[0]){case 0x02:val[0]=0x04;break;/* SerBool */case 0x04:val[0]=0x10;break;/* SerErr */case 0x00:val[0]=0x01;break;/* SerNum */case 0x01:val[0]=0x02;break;/* SerStr */}switch(val[0]){case 0x04:/* SerBool -- boolean */val[1]=parsebool(blob,1)?'TRUE':'FALSE';if(biff!=12)blob.l+=7;break;case 0x25:/* appears to be an alias */case 0x10:/* SerErr -- error */val[1]=BErr[blob[blob.l]];blob.l+=biff==12?4:8;break;case 0x00:/* SerNil -- honestly, I'm not sure how to reproduce this */blob.l+=8;break;case 0x01:/* SerNum -- Xnum */val[1]=parse_Xnum(blob);break;case 0x02:/* SerStr -- XLUnicodeString (<256 chars) */val[1]=parse_XLUnicodeString2(blob,0,{biff:biff>0&&biff<8?2:biff});break;default:throw new Error("Bad SerAr: "+val[0]);/* Unreachable */}return val;}/* [MS-XLS] 2.5.198.61 ; [MS-XLSB] 2.5.97.44 */function parse_PtgExtraMem(blob,cce,opts){var count=blob.read_shift(opts.biff==12?4:2);var out/*:Array<Range>*/=[];for(var i=0;i!=count;++i)out.push((opts.biff==12?parse_UncheckedRfX:parse_Ref8U)(blob));return out;}/* [MS-XLS] 2.5.198.59 ; [MS-XLSB] 2.5.97.41 */function parse_PtgExtraArray(blob,length,opts){var rows=0,cols=0;if(opts.biff==12){rows=blob.read_shift(4);// DRw
	cols=blob.read_shift(4);// DCol
	}else {cols=1+blob.read_shift(1);//DColByteU
	rows=1+blob.read_shift(2);//DRw
	}if(opts.biff>=2&&opts.biff<8){--rows;if(--cols==0)cols=0x100;}// $FlowIgnore
	for(var i=0,o/*:Array<Array<any>>*/=[];i!=rows&&(o[i]=[]);++i)for(var j=0;j!=cols;++j)o[i][j]=parse_SerAr(blob,opts.biff);return o;}/* [MS-XLS] 2.5.198.76 ; [MS-XLSB] 2.5.97.60 */function parse_PtgName(blob,length,opts){var type=blob.read_shift(1)>>>5&0x03;var w=!opts||opts.biff>=8?4:2;var nameindex=blob.read_shift(w);switch(opts.biff){case 2:blob.l+=5;break;case 3:case 4:blob.l+=8;break;case 5:blob.l+=12;break;}return [type,0,nameindex];}/* [MS-XLS] 2.5.198.77 ; [MS-XLSB] 2.5.97.61 */function parse_PtgNameX(blob,length,opts){if(opts.biff==5)return parse_PtgNameX_BIFF5(blob);var type=blob.read_shift(1)>>>5&0x03;var ixti=blob.read_shift(2);// XtiIndex
	var nameindex=blob.read_shift(4);return [type,ixti,nameindex];}function parse_PtgNameX_BIFF5(blob/*::, length, opts*/){var type=blob.read_shift(1)>>>5&0x03;var ixti=blob.read_shift(2,'i');// XtiIndex
	blob.l+=8;var nameindex=blob.read_shift(2);blob.l+=12;return [type,ixti,nameindex];}/* [MS-XLS] 2.5.198.70 ; [MS-XLSB] 2.5.97.54 */function parse_PtgMemArea(blob,length,opts){var type=blob.read_shift(1)>>>5&0x03;blob.l+=opts&&opts.biff==2?3:4;var cce=blob.read_shift(opts&&opts.biff==2?1:2);return [type,cce];}/* [MS-XLS] 2.5.198.72 ; [MS-XLSB] 2.5.97.56 */function parse_PtgMemFunc(blob,length,opts){var type=blob.read_shift(1)>>>5&0x03;var cce=blob.read_shift(opts&&opts.biff==2?1:2);return [type,cce];}/* [MS-XLS] 2.5.198.86 ; [MS-XLSB] 2.5.97.69 */function parse_PtgRefErr(blob,length,opts){var type=blob.read_shift(1)>>>5&0x03;blob.l+=4;if(opts.biff<8)blob.l--;if(opts.biff==12)blob.l+=2;return [type];}/* [MS-XLS] 2.5.198.87 ; [MS-XLSB] 2.5.97.71 */function parse_PtgRefErr3d(blob,length,opts){var type=(blob[blob.l++]&0x60)>>5;var ixti=blob.read_shift(2);var w=4;if(opts)switch(opts.biff){case 5:w=15;break;case 12:w=6;break;}blob.l+=w;return [type,ixti];}/* [MS-XLS] 2.5.198.71 ; [MS-XLSB] 2.5.97.55 */var parse_PtgMemErr=parsenoop;/* [MS-XLS] 2.5.198.73  ; [MS-XLSB] 2.5.97.57 */var parse_PtgMemNoMem=parsenoop;/* [MS-XLS] 2.5.198.92 */var parse_PtgTbl=parsenoop;function parse_PtgElfLoc(blob,length,opts){blob.l+=2;return [parse_RgceElfLoc(blob)];}function parse_PtgElfNoop(blob/*::, length, opts*/){blob.l+=6;return [];}/* [MS-XLS] 2.5.198.46 */var parse_PtgElfCol=parse_PtgElfLoc;/* [MS-XLS] 2.5.198.47 */var parse_PtgElfColS=parse_PtgElfNoop;/* [MS-XLS] 2.5.198.48 */var parse_PtgElfColSV=parse_PtgElfNoop;/* [MS-XLS] 2.5.198.49 */var parse_PtgElfColV=parse_PtgElfLoc;/* [MS-XLS] 2.5.198.50 */function parse_PtgElfLel(blob/*::, length, opts*/){blob.l+=2;return [parseuint16(blob),blob.read_shift(2)&0x01];}/* [MS-XLS] 2.5.198.51 */var parse_PtgElfRadical=parse_PtgElfLoc;/* [MS-XLS] 2.5.198.52 */var parse_PtgElfRadicalLel=parse_PtgElfLel;/* [MS-XLS] 2.5.198.53 */var parse_PtgElfRadicalS=parse_PtgElfNoop;/* [MS-XLS] 2.5.198.54 */var parse_PtgElfRw=parse_PtgElfLoc;/* [MS-XLS] 2.5.198.55 */var parse_PtgElfRwV=parse_PtgElfLoc;/* [MS-XLSB] 2.5.97.52 TODO */var PtgListRT=["Data","All","Headers","??","?Data2","??","?DataHeaders","??","Totals","??","??","??","?DataTotals","??","??","??","?Current"];function parse_PtgList(blob/*::, length, opts*/){blob.l+=2;var ixti=blob.read_shift(2);var flags=blob.read_shift(2);var idx=blob.read_shift(4);var c=blob.read_shift(2);var C=blob.read_shift(2);var rt=PtgListRT[flags>>2&0x1F];return {ixti:ixti,coltype:flags&0x3,rt:rt,idx:idx,c:c,C:C};}/* [MS-XLS] 2.5.198.91 ; [MS-XLSB] 2.5.97.76 */function parse_PtgSxName(blob/*::, length, opts*/){blob.l+=2;return [blob.read_shift(4)];}/* [XLS] old spec */function parse_PtgSheet(blob,length,opts){blob.l+=5;blob.l+=2;blob.l+=opts.biff==2?1:4;return ["PTGSHEET"];}function parse_PtgEndSheet(blob,length,opts){blob.l+=opts.biff==2?4:5;return ["PTGENDSHEET"];}function parse_PtgMemAreaN(blob/*::, length, opts*/){var type=blob.read_shift(1)>>>5&0x03;var cce=blob.read_shift(2);return [type,cce];}function parse_PtgMemNoMemN(blob/*::, length, opts*/){var type=blob.read_shift(1)>>>5&0x03;var cce=blob.read_shift(2);return [type,cce];}function parse_PtgAttrNoop(blob/*::, length, opts*/){blob.l+=4;return [0,0];}/* [MS-XLS] 2.5.198.25 ; [MS-XLSB] 2.5.97.16 */var PtgTypes={/*::[*/0x01/*::]*/:{n:'PtgExp',f:parse_PtgExp},/*::[*/0x02/*::]*/:{n:'PtgTbl',f:parse_PtgTbl},/*::[*/0x03/*::]*/:{n:'PtgAdd',f:parseread1},/*::[*/0x04/*::]*/:{n:'PtgSub',f:parseread1},/*::[*/0x05/*::]*/:{n:'PtgMul',f:parseread1},/*::[*/0x06/*::]*/:{n:'PtgDiv',f:parseread1},/*::[*/0x07/*::]*/:{n:'PtgPower',f:parseread1},/*::[*/0x08/*::]*/:{n:'PtgConcat',f:parseread1},/*::[*/0x09/*::]*/:{n:'PtgLt',f:parseread1},/*::[*/0x0A/*::]*/:{n:'PtgLe',f:parseread1},/*::[*/0x0B/*::]*/:{n:'PtgEq',f:parseread1},/*::[*/0x0C/*::]*/:{n:'PtgGe',f:parseread1},/*::[*/0x0D/*::]*/:{n:'PtgGt',f:parseread1},/*::[*/0x0E/*::]*/:{n:'PtgNe',f:parseread1},/*::[*/0x0F/*::]*/:{n:'PtgIsect',f:parseread1},/*::[*/0x10/*::]*/:{n:'PtgUnion',f:parseread1},/*::[*/0x11/*::]*/:{n:'PtgRange',f:parseread1},/*::[*/0x12/*::]*/:{n:'PtgUplus',f:parseread1},/*::[*/0x13/*::]*/:{n:'PtgUminus',f:parseread1},/*::[*/0x14/*::]*/:{n:'PtgPercent',f:parseread1},/*::[*/0x15/*::]*/:{n:'PtgParen',f:parseread1},/*::[*/0x16/*::]*/:{n:'PtgMissArg',f:parseread1},/*::[*/0x17/*::]*/:{n:'PtgStr',f:parse_PtgStr},/*::[*/0x1A/*::]*/:{n:'PtgSheet',f:parse_PtgSheet},/*::[*/0x1B/*::]*/:{n:'PtgEndSheet',f:parse_PtgEndSheet},/*::[*/0x1C/*::]*/:{n:'PtgErr',f:parse_PtgErr},/*::[*/0x1D/*::]*/:{n:'PtgBool',f:parse_PtgBool},/*::[*/0x1E/*::]*/:{n:'PtgInt',f:parse_PtgInt},/*::[*/0x1F/*::]*/:{n:'PtgNum',f:parse_PtgNum},/*::[*/0x20/*::]*/:{n:'PtgArray',f:parse_PtgArray},/*::[*/0x21/*::]*/:{n:'PtgFunc',f:parse_PtgFunc},/*::[*/0x22/*::]*/:{n:'PtgFuncVar',f:parse_PtgFuncVar},/*::[*/0x23/*::]*/:{n:'PtgName',f:parse_PtgName},/*::[*/0x24/*::]*/:{n:'PtgRef',f:parse_PtgRef},/*::[*/0x25/*::]*/:{n:'PtgArea',f:parse_PtgArea},/*::[*/0x26/*::]*/:{n:'PtgMemArea',f:parse_PtgMemArea},/*::[*/0x27/*::]*/:{n:'PtgMemErr',f:parse_PtgMemErr},/*::[*/0x28/*::]*/:{n:'PtgMemNoMem',f:parse_PtgMemNoMem},/*::[*/0x29/*::]*/:{n:'PtgMemFunc',f:parse_PtgMemFunc},/*::[*/0x2A/*::]*/:{n:'PtgRefErr',f:parse_PtgRefErr},/*::[*/0x2B/*::]*/:{n:'PtgAreaErr',f:parse_PtgAreaErr},/*::[*/0x2C/*::]*/:{n:'PtgRefN',f:parse_PtgRefN},/*::[*/0x2D/*::]*/:{n:'PtgAreaN',f:parse_PtgAreaN},/*::[*/0x2E/*::]*/:{n:'PtgMemAreaN',f:parse_PtgMemAreaN},/*::[*/0x2F/*::]*/:{n:'PtgMemNoMemN',f:parse_PtgMemNoMemN},/*::[*/0x39/*::]*/:{n:'PtgNameX',f:parse_PtgNameX},/*::[*/0x3A/*::]*/:{n:'PtgRef3d',f:parse_PtgRef3d},/*::[*/0x3B/*::]*/:{n:'PtgArea3d',f:parse_PtgArea3d},/*::[*/0x3C/*::]*/:{n:'PtgRefErr3d',f:parse_PtgRefErr3d},/*::[*/0x3D/*::]*/:{n:'PtgAreaErr3d',f:parse_PtgAreaErr3d},/*::[*/0xFF/*::]*/:{}};/* These are duplicated in the PtgTypes table */var PtgDupes={/*::[*/0x40/*::]*/:0x20,/*::[*/0x60/*::]*/:0x20,/*::[*/0x41/*::]*/:0x21,/*::[*/0x61/*::]*/:0x21,/*::[*/0x42/*::]*/:0x22,/*::[*/0x62/*::]*/:0x22,/*::[*/0x43/*::]*/:0x23,/*::[*/0x63/*::]*/:0x23,/*::[*/0x44/*::]*/:0x24,/*::[*/0x64/*::]*/:0x24,/*::[*/0x45/*::]*/:0x25,/*::[*/0x65/*::]*/:0x25,/*::[*/0x46/*::]*/:0x26,/*::[*/0x66/*::]*/:0x26,/*::[*/0x47/*::]*/:0x27,/*::[*/0x67/*::]*/:0x27,/*::[*/0x48/*::]*/:0x28,/*::[*/0x68/*::]*/:0x28,/*::[*/0x49/*::]*/:0x29,/*::[*/0x69/*::]*/:0x29,/*::[*/0x4A/*::]*/:0x2A,/*::[*/0x6A/*::]*/:0x2A,/*::[*/0x4B/*::]*/:0x2B,/*::[*/0x6B/*::]*/:0x2B,/*::[*/0x4C/*::]*/:0x2C,/*::[*/0x6C/*::]*/:0x2C,/*::[*/0x4D/*::]*/:0x2D,/*::[*/0x6D/*::]*/:0x2D,/*::[*/0x4E/*::]*/:0x2E,/*::[*/0x6E/*::]*/:0x2E,/*::[*/0x4F/*::]*/:0x2F,/*::[*/0x6F/*::]*/:0x2F,/*::[*/0x58/*::]*/:0x22,/*::[*/0x78/*::]*/:0x22,/*::[*/0x59/*::]*/:0x39,/*::[*/0x79/*::]*/:0x39,/*::[*/0x5A/*::]*/:0x3A,/*::[*/0x7A/*::]*/:0x3A,/*::[*/0x5B/*::]*/:0x3B,/*::[*/0x7B/*::]*/:0x3B,/*::[*/0x5C/*::]*/:0x3C,/*::[*/0x7C/*::]*/:0x3C,/*::[*/0x5D/*::]*/:0x3D,/*::[*/0x7D/*::]*/:0x3D};var Ptg18={/*::[*/0x01/*::]*/:{n:'PtgElfLel',f:parse_PtgElfLel},/*::[*/0x02/*::]*/:{n:'PtgElfRw',f:parse_PtgElfRw},/*::[*/0x03/*::]*/:{n:'PtgElfCol',f:parse_PtgElfCol},/*::[*/0x06/*::]*/:{n:'PtgElfRwV',f:parse_PtgElfRwV},/*::[*/0x07/*::]*/:{n:'PtgElfColV',f:parse_PtgElfColV},/*::[*/0x0A/*::]*/:{n:'PtgElfRadical',f:parse_PtgElfRadical},/*::[*/0x0B/*::]*/:{n:'PtgElfRadicalS',f:parse_PtgElfRadicalS},/*::[*/0x0D/*::]*/:{n:'PtgElfColS',f:parse_PtgElfColS},/*::[*/0x0F/*::]*/:{n:'PtgElfColSV',f:parse_PtgElfColSV},/*::[*/0x10/*::]*/:{n:'PtgElfRadicalLel',f:parse_PtgElfRadicalLel},/*::[*/0x19/*::]*/:{n:'PtgList',f:parse_PtgList},/*::[*/0x1D/*::]*/:{n:'PtgSxName',f:parse_PtgSxName},/*::[*/0xFF/*::]*/:{}};var Ptg19={/*::[*/0x00/*::]*/:{n:'PtgAttrNoop',f:parse_PtgAttrNoop},/*::[*/0x01/*::]*/:{n:'PtgAttrSemi',f:parse_PtgAttrSemi},/*::[*/0x02/*::]*/:{n:'PtgAttrIf',f:parse_PtgAttrIf},/*::[*/0x04/*::]*/:{n:'PtgAttrChoose',f:parse_PtgAttrChoose},/*::[*/0x08/*::]*/:{n:'PtgAttrGoto',f:parse_PtgAttrGoto},/*::[*/0x10/*::]*/:{n:'PtgAttrSum',f:parse_PtgAttrSum},/*::[*/0x20/*::]*/:{n:'PtgAttrBaxcel',f:parse_PtgAttrBaxcel},/*::[*/0x21/*::]*/:{n:'PtgAttrBaxcel',f:parse_PtgAttrBaxcel},/*::[*/0x40/*::]*/:{n:'PtgAttrSpace',f:parse_PtgAttrSpace},/*::[*/0x41/*::]*/:{n:'PtgAttrSpaceSemi',f:parse_PtgAttrSpaceSemi},/*::[*/0x80/*::]*/:{n:'PtgAttrIfError',f:parse_PtgAttrIfError},/*::[*/0xFF/*::]*/:{}};/* [MS-XLS] 2.5.198.103 ; [MS-XLSB] 2.5.97.87 */function parse_RgbExtra(blob,length,rgce,opts){if(opts.biff<8)return parsenoop(blob,length);var target=blob.l+length;var o=[];for(var i=0;i!==rgce.length;++i){switch(rgce[i][0]){case'PtgArray':/* PtgArray -> PtgExtraArray */rgce[i][1]=parse_PtgExtraArray(blob,0,opts);o.push(rgce[i][1]);break;case'PtgMemArea':/* PtgMemArea -> PtgExtraMem */rgce[i][2]=parse_PtgExtraMem(blob,rgce[i][1],opts);o.push(rgce[i][2]);break;case'PtgExp':/* PtgExp -> PtgExtraCol */if(opts&&opts.biff==12){rgce[i][1][1]=blob.read_shift(4);o.push(rgce[i][1]);}break;case'PtgList':/* TODO: PtgList -> PtgExtraList */case'PtgElfRadicalS':/* TODO: PtgElfRadicalS -> PtgExtraElf */case'PtgElfColS':/* TODO: PtgElfColS -> PtgExtraElf */case'PtgElfColSV':/* TODO: PtgElfColSV -> PtgExtraElf */throw "Unsupported "+rgce[i][0];}}length=target-blob.l;/* note: this is technically an error but Excel disregards */ //if(target !== blob.l && blob.l !== target - length) throw new Error(target + " != " + blob.l);
	if(length!==0)o.push(parsenoop(blob,length));return o;}/* [MS-XLS] 2.5.198.104 ; [MS-XLSB] 2.5.97.88 */function parse_Rgce(blob,length,opts){var target=blob.l+length;var R,id,ptgs=[];while(target!=blob.l){length=target-blob.l;id=blob[blob.l];R=PtgTypes[id]||PtgTypes[PtgDupes[id]];if(id===0x18||id===0x19)R=(id===0x18?Ptg18:Ptg19)[blob[blob.l+1]];if(!R||!R.f){/*ptgs.push*/parsenoop(blob,length);}else {ptgs.push([R.n,R.f(blob,length,opts)]);}}return ptgs;}function stringify_array(f/*:Array<Array<string>>*/)/*:string*/{var o/*:Array<string>*/=[];for(var i=0;i<f.length;++i){var x=f[i],r/*:Array<string>*/=[];for(var j=0;j<x.length;++j){var y=x[j];if(y)switch(y[0]){// TODO: handle embedded quotes
	case 0x02:/*:: if(typeof y[1] != 'string') throw "unreachable"; */r.push('"'+y[1].replace(/"/g,'""')+'"');break;default:r.push(y[1]);}else r.push("");}o.push(r.join(","));}return o.join(";");}/* [MS-XLS] 2.2.2 ; [MS-XLSB] 2.2.2 TODO */var PtgBinOp={PtgAdd:"+",PtgConcat:"&",PtgDiv:"/",PtgEq:"=",PtgGe:">=",PtgGt:">",PtgLe:"<=",PtgLt:"<",PtgMul:"*",PtgNe:"<>",PtgPower:"^",PtgSub:"-"};// List of invalid characters needs to be tested further
	function formula_quote_sheet_name(sname/*:string*/,opts)/*:string*/{if(!sname&&!(opts&&opts.biff<=5&&opts.biff>=2))throw new Error("empty sheet name");if(/[^\w\u4E00-\u9FFF\u3040-\u30FF]/.test(sname))return "'"+sname+"'";return sname;}function get_ixti_raw(supbooks,ixti/*:number*/,opts)/*:string*/{if(!supbooks)return "SH33TJSERR0";if(opts.biff>8&&(!supbooks.XTI||!supbooks.XTI[ixti]))return supbooks.SheetNames[ixti];if(!supbooks.XTI)return "SH33TJSERR6";var XTI=supbooks.XTI[ixti];if(opts.biff<8){if(ixti>10000)ixti-=65536;if(ixti<0)ixti=-ixti;return ixti==0?"":supbooks.XTI[ixti-1];}if(!XTI)return "SH33TJSERR1";var o="";if(opts.biff>8)switch(supbooks[XTI[0]][0]){case 0x0165:/* 'BrtSupSelf' */o=XTI[1]==-1?"#REF":supbooks.SheetNames[XTI[1]];return XTI[1]==XTI[2]?o:o+":"+supbooks.SheetNames[XTI[2]];case 0x0166:/* 'BrtSupSame' */if(opts.SID!=null)return supbooks.SheetNames[opts.SID];return "SH33TJSSAME"+supbooks[XTI[0]][0];case 0x0163:/* 'BrtSupBookSrc' */ /* falls through */default:return "SH33TJSSRC"+supbooks[XTI[0]][0];}switch(supbooks[XTI[0]][0][0]){case 0x0401:o=XTI[1]==-1?"#REF":supbooks.SheetNames[XTI[1]]||"SH33TJSERR3";return XTI[1]==XTI[2]?o:o+":"+supbooks.SheetNames[XTI[2]];case 0x3A01:return supbooks[XTI[0]].slice(1).map(function(name){return name.Name;}).join(";;");//return "SH33TJSERR8";
	default:if(!supbooks[XTI[0]][0][3])return "SH33TJSERR2";o=XTI[1]==-1?"#REF":supbooks[XTI[0]][0][3][XTI[1]]||"SH33TJSERR4";return XTI[1]==XTI[2]?o:o+":"+supbooks[XTI[0]][0][3][XTI[2]];}}function get_ixti(supbooks,ixti/*:number*/,opts)/*:string*/{var ixtiraw=get_ixti_raw(supbooks,ixti,opts);return ixtiraw=="#REF"?ixtiraw:formula_quote_sheet_name(ixtiraw,opts);}function stringify_formula(formula/*Array<any>*/,range,cell/*:any*/,supbooks,opts)/*:string*/{var biff=opts&&opts.biff||8;var _range=/*range != null ? range :*/{s:{c:0,r:0},e:{c:0,r:0}};var stack/*:Array<string>*/=[],e1,e2,/*::type,*/c/*:CellAddress*/,ixti=0,nameidx=0,r,sname="";if(!formula[0]||!formula[0][0])return "";var last_sp=-1,sp="";for(var ff=0,fflen=formula[0].length;ff<fflen;++ff){var f=formula[0][ff];switch(f[0]){case'PtgUminus':/* [MS-XLS] 2.5.198.93 */stack.push("-"+stack.pop());break;case'PtgUplus':/* [MS-XLS] 2.5.198.95 */stack.push("+"+stack.pop());break;case'PtgPercent':/* [MS-XLS] 2.5.198.81 */stack.push(stack.pop()+"%");break;case'PtgAdd':/* [MS-XLS] 2.5.198.26 */case'PtgConcat':/* [MS-XLS] 2.5.198.43 */case'PtgDiv':/* [MS-XLS] 2.5.198.45 */case'PtgEq':/* [MS-XLS] 2.5.198.56 */case'PtgGe':/* [MS-XLS] 2.5.198.64 */case'PtgGt':/* [MS-XLS] 2.5.198.65 */case'PtgLe':/* [MS-XLS] 2.5.198.68 */case'PtgLt':/* [MS-XLS] 2.5.198.69 */case'PtgMul':/* [MS-XLS] 2.5.198.75 */case'PtgNe':/* [MS-XLS] 2.5.198.78 */case'PtgPower':/* [MS-XLS] 2.5.198.82 */case'PtgSub':/* [MS-XLS] 2.5.198.90 */e1=stack.pop();e2=stack.pop();if(last_sp>=0){switch(formula[0][last_sp][1][0]){case 0:// $FlowIgnore
	sp=fill(" ",formula[0][last_sp][1][1]);break;case 1:// $FlowIgnore
	sp=fill("\r",formula[0][last_sp][1][1]);break;default:sp="";// $FlowIgnore
	if(opts.WTF)throw new Error("Unexpected PtgAttrSpaceType "+formula[0][last_sp][1][0]);}e2=e2+sp;last_sp=-1;}stack.push(e2+PtgBinOp[f[0]]+e1);break;case'PtgIsect':/* [MS-XLS] 2.5.198.67 */e1=stack.pop();e2=stack.pop();stack.push(e2+" "+e1);break;case'PtgUnion':/* [MS-XLS] 2.5.198.94 */e1=stack.pop();e2=stack.pop();stack.push(e2+","+e1);break;case'PtgRange':/* [MS-XLS] 2.5.198.83 */e1=stack.pop();e2=stack.pop();stack.push(e2+":"+e1);break;case'PtgAttrChoose':/* [MS-XLS] 2.5.198.34 */break;case'PtgAttrGoto':/* [MS-XLS] 2.5.198.35 */break;case'PtgAttrIf':/* [MS-XLS] 2.5.198.36 */break;case'PtgAttrIfError':/* [MS-XLSB] 2.5.97.28 */break;case'PtgRef':/* [MS-XLS] 2.5.198.84 */ /*::type = f[1][0]; */c=shift_cell_xls(f[1][1]/*:any*/,_range,opts);stack.push(encode_cell_xls(c,biff));break;case'PtgRefN':/* [MS-XLS] 2.5.198.88 */ /*::type = f[1][0]; */c=cell?shift_cell_xls(f[1][1]/*:any*/,cell,opts):f[1][1]/*:any*/;stack.push(encode_cell_xls(c,biff));break;case'PtgRef3d':/* [MS-XLS] 2.5.198.85 */ /*::type = f[1][0]; */ixti=/*::Number(*/f[1][1]/*::)*/;c=shift_cell_xls(f[1][2]/*:any*/,_range,opts);sname=get_ixti(supbooks,ixti,opts);stack.push(sname+"!"+encode_cell_xls(c,biff));break;case'PtgFunc':/* [MS-XLS] 2.5.198.62 */case'PtgFuncVar':/* [MS-XLS] 2.5.198.63 */ /* f[1] = [argc, func, type] */var argc/*:number*/=f[1][0]/*:any*/,func/*:string*/=f[1][1]/*:any*/;if(!argc)argc=0;argc&=0x7F;var args=argc==0?[]:stack.slice(-argc);stack.length-=argc;if(func==='User')func=args.shift();stack.push(func+"("+args.join(",")+")");break;case'PtgBool':/* [MS-XLS] 2.5.198.42 */stack.push(f[1]?"TRUE":"FALSE");break;case'PtgInt':/* [MS-XLS] 2.5.198.66 */stack.push(/*::String(*/f[1]/*::)*/);break;case'PtgNum':/* [MS-XLS] 2.5.198.79 TODO: precision? */stack.push(String(f[1]));break;case'PtgStr':/* [MS-XLS] 2.5.198.89 */ // $FlowIgnore
	stack.push('"'+f[1].replace(/"/g,'""')+'"');break;case'PtgErr':/* [MS-XLS] 2.5.198.57 */stack.push(/*::String(*/f[1]/*::)*/);break;case'PtgAreaN':/* [MS-XLS] 2.5.198.31 TODO */ /*::type = f[1][0]; */r=shift_range_xls(f[1][1],cell?{s:cell}:_range,opts);stack.push(encode_range_xls(r/*:any*/,opts));break;case'PtgArea':/* [MS-XLS] 2.5.198.27 TODO: fixed points */ /*::type = f[1][0]; */r=shift_range_xls(f[1][1],_range,opts);stack.push(encode_range_xls(r/*:any*/,opts));break;case'PtgArea3d':/* [MS-XLS] 2.5.198.28 TODO */ /*::type = f[1][0]; */ixti=/*::Number(*/f[1][1]/*::)*/;r=f[1][2];sname=get_ixti(supbooks,ixti,opts);stack.push(sname+"!"+encode_range_xls(r/*:any*/,opts));break;case'PtgAttrSum':/* [MS-XLS] 2.5.198.41 */stack.push("SUM("+stack.pop()+")");break;case'PtgAttrBaxcel':/* [MS-XLS] 2.5.198.33 */case'PtgAttrSemi':/* [MS-XLS] 2.5.198.37 */break;case'PtgName':/* [MS-XLS] 2.5.198.76 ; [MS-XLSB] 2.5.97.60 TODO: revisions */ /* f[1] = type, 0, nameindex */nameidx=f[1][2]/*:any*/;var lbl=(supbooks.names||[])[nameidx-1]||(supbooks[0]||[])[nameidx];var name=lbl?lbl.Name:"SH33TJSNAME"+String(nameidx);/* [MS-XLSB] 2.5.97.10 Ftab -- last verified 20220204 */if(name&&name.slice(0,6)=="_xlfn."&&!opts.xlfn)name=name.slice(6);stack.push(name);break;case'PtgNameX':/* [MS-XLS] 2.5.198.77 ; [MS-XLSB] 2.5.97.61 TODO: revisions */ /* f[1] = type, ixti, nameindex */var bookidx/*:number*/=f[1][1]/*:any*/;nameidx=f[1][2]/*:any*/;var externbook;/* TODO: Properly handle missing values -- this should be using get_ixti_raw primarily */if(opts.biff<=5){if(bookidx<0)bookidx=-bookidx;if(supbooks[bookidx])externbook=supbooks[bookidx][nameidx];}else {var o="";if(((supbooks[bookidx]||[])[0]||[])[0]==0x3A01);else if(((supbooks[bookidx]||[])[0]||[])[0]==0x0401){if(supbooks[bookidx][nameidx]&&supbooks[bookidx][nameidx].itab>0){o=supbooks.SheetNames[supbooks[bookidx][nameidx].itab-1]+"!";}}else o=supbooks.SheetNames[nameidx-1]+"!";if(supbooks[bookidx]&&supbooks[bookidx][nameidx])o+=supbooks[bookidx][nameidx].Name;else if(supbooks[0]&&supbooks[0][nameidx])o+=supbooks[0][nameidx].Name;else {var ixtidata=(get_ixti_raw(supbooks,bookidx,opts)||"").split(";;");if(ixtidata[nameidx-1])o=ixtidata[nameidx-1];// TODO: confirm this is correct
	else o+="SH33TJSERRX";}stack.push(o);break;}if(!externbook)externbook={Name:"SH33TJSERRY"};stack.push(externbook.Name);break;case'PtgParen':/* [MS-XLS] 2.5.198.80 */var lp='(',rp=')';if(last_sp>=0){sp="";switch(formula[0][last_sp][1][0]){// $FlowIgnore
	case 2:lp=fill(" ",formula[0][last_sp][1][1])+lp;break;// $FlowIgnore
	case 3:lp=fill("\r",formula[0][last_sp][1][1])+lp;break;// $FlowIgnore
	case 4:rp=fill(" ",formula[0][last_sp][1][1])+rp;break;// $FlowIgnore
	case 5:rp=fill("\r",formula[0][last_sp][1][1])+rp;break;default:// $FlowIgnore
	if(opts.WTF)throw new Error("Unexpected PtgAttrSpaceType "+formula[0][last_sp][1][0]);}last_sp=-1;}stack.push(lp+stack.pop()+rp);break;case'PtgRefErr':/* [MS-XLS] 2.5.198.86 */stack.push('#REF!');break;case'PtgRefErr3d':/* [MS-XLS] 2.5.198.87 */stack.push('#REF!');break;case'PtgExp':/* [MS-XLS] 2.5.198.58 TODO */c={c:f[1][1]/*:any*/,r:f[1][0]/*:any*/};var q={c:cell.c,r:cell.r}/*:any*/;if(supbooks.sharedf[encode_cell(c)]){var parsedf=supbooks.sharedf[encode_cell(c)];stack.push(stringify_formula(parsedf,_range,q,supbooks,opts));}else {var fnd=false;for(e1=0;e1!=supbooks.arrayf.length;++e1){/* TODO: should be something like range_has */e2=supbooks.arrayf[e1];if(c.c<e2[0].s.c||c.c>e2[0].e.c)continue;if(c.r<e2[0].s.r||c.r>e2[0].e.r)continue;stack.push(stringify_formula(e2[1],_range,q,supbooks,opts));fnd=true;break;}if(!fnd)stack.push(/*::String(*/f[1]/*::)*/);}break;case'PtgArray':/* [MS-XLS] 2.5.198.32 TODO */stack.push("{"+stringify_array(/*::(*/f[1]/*:: :any)*/)+"}");break;case'PtgMemArea':/* [MS-XLS] 2.5.198.70 TODO: confirm this is a non-display */ //stack.push("(" + f[2].map(encode_range).join(",") + ")");
	break;case'PtgAttrSpace':/* [MS-XLS] 2.5.198.38 */case'PtgAttrSpaceSemi':/* [MS-XLS] 2.5.198.39 */last_sp=ff;break;case'PtgTbl':/* [MS-XLS] 2.5.198.92 TODO */break;case'PtgMemErr':/* [MS-XLS] 2.5.198.71 */break;case'PtgMissArg':/* [MS-XLS] 2.5.198.74 */stack.push("");break;case'PtgAreaErr':/* [MS-XLS] 2.5.198.29 */stack.push("#REF!");break;case'PtgAreaErr3d':/* [MS-XLS] 2.5.198.30 */stack.push("#REF!");break;case'PtgList':/* [MS-XLSB] 2.5.97.52 */ // $FlowIgnore
	stack.push("Table"+f[1].idx+"[#"+f[1].rt+"]");break;case'PtgMemAreaN':case'PtgMemNoMemN':case'PtgAttrNoop':case'PtgSheet':case'PtgEndSheet':break;case'PtgMemFunc':/* [MS-XLS] 2.5.198.72 TODO */break;case'PtgMemNoMem':/* [MS-XLS] 2.5.198.73 TODO */break;case'PtgElfCol':/* [MS-XLS] 2.5.198.46 */case'PtgElfColS':/* [MS-XLS] 2.5.198.47 */case'PtgElfColSV':/* [MS-XLS] 2.5.198.48 */case'PtgElfColV':/* [MS-XLS] 2.5.198.49 */case'PtgElfLel':/* [MS-XLS] 2.5.198.50 */case'PtgElfRadical':/* [MS-XLS] 2.5.198.51 */case'PtgElfRadicalLel':/* [MS-XLS] 2.5.198.52 */case'PtgElfRadicalS':/* [MS-XLS] 2.5.198.53 */case'PtgElfRw':/* [MS-XLS] 2.5.198.54 */case'PtgElfRwV':/* [MS-XLS] 2.5.198.55 */throw new Error("Unsupported ELFs");case'PtgSxName':/* [MS-XLS] 2.5.198.91 TODO -- find a test case */throw new Error('Unrecognized Formula Token: '+String(f));default:throw new Error('Unrecognized Formula Token: '+String(f));}var PtgNonDisp=['PtgAttrSpace','PtgAttrSpaceSemi','PtgAttrGoto'];if(opts.biff!=3)if(last_sp>=0&&PtgNonDisp.indexOf(formula[0][ff][0])==-1){f=formula[0][last_sp];var _left=true;switch(f[1][0]){/* note: some bad XLSB files omit the PtgParen */case 4:_left=false;/* falls through */case 0:// $FlowIgnore
	sp=fill(" ",f[1][1]);break;case 5:_left=false;/* falls through */case 1:// $FlowIgnore
	sp=fill("\r",f[1][1]);break;default:sp="";// $FlowIgnore
	if(opts.WTF)throw new Error("Unexpected PtgAttrSpaceType "+f[1][0]);}stack.push((_left?sp:"")+stack.pop()+(_left?"":sp));last_sp=-1;}}if(stack.length>1&&opts.WTF)throw new Error("bad formula stack");return stack[0];}/* [MS-XLS] 2.5.198.1 TODO */function parse_ArrayParsedFormula(blob,length,opts/*::, ref*/){var target=blob.l+length,len=opts.biff==2?1:2;var rgcb,cce=blob.read_shift(len);// length of rgce
	if(cce==0xFFFF)return [[],parsenoop(blob,length-2)];var rgce=parse_Rgce(blob,cce,opts);if(length!==cce+len)rgcb=parse_RgbExtra(blob,length-cce-len,rgce,opts);blob.l=target;return [rgce,rgcb];}/* [MS-XLS] 2.5.198.3 TODO */function parse_XLSCellParsedFormula(blob,length,opts){var target=blob.l+length,len=opts.biff==2?1:2;var rgcb,cce=blob.read_shift(len);// length of rgce
	if(cce==0xFFFF)return [[],parsenoop(blob,length-2)];var rgce=parse_Rgce(blob,cce,opts);if(length!==cce+len)rgcb=parse_RgbExtra(blob,length-cce-len,rgce,opts);blob.l=target;return [rgce,rgcb];}/* [MS-XLS] 2.5.198.21 */function parse_NameParsedFormula(blob,length,opts,cce){var target=blob.l+length;var rgce=parse_Rgce(blob,cce,opts);var rgcb;if(target!==blob.l)rgcb=parse_RgbExtra(blob,target-blob.l,rgce,opts);return [rgce,rgcb];}/* [MS-XLS] 2.5.198.118 TODO */function parse_SharedParsedFormula(blob,length,opts){var target=blob.l+length;var rgcb,cce=blob.read_shift(2);// length of rgce
	var rgce=parse_Rgce(blob,cce,opts);if(cce==0xFFFF)return [[],parsenoop(blob,length-2)];if(length!==cce+2)rgcb=parse_RgbExtra(blob,target-cce-2,rgce,opts);return [rgce,rgcb];}/* [MS-XLS] 2.5.133 TODO: how to emit empty strings? */function parse_FormulaValue(blob/*::, length*/){var b;if(__readUInt16LE(blob,blob.l+6)!==0xFFFF)return [parse_Xnum(blob),'n'];switch(blob[blob.l]){case 0x00:blob.l+=8;return ["String",'s'];case 0x01:b=blob[blob.l+2]===0x1;blob.l+=8;return [b,'b'];case 0x02:b=blob[blob.l+2];blob.l+=8;return [b,'e'];case 0x03:blob.l+=8;return ["",'s'];}return [];}/* [MS-XLS] 2.4.127 TODO */function parse_Formula(blob,length,opts){var end=blob.l+length;var cell=parse_XLSCell(blob);if(opts.biff==2)++blob.l;var val=parse_FormulaValue(blob);var flags=blob.read_shift(1);if(opts.biff!=2){blob.read_shift(1);if(opts.biff>=5){/*var chn = */blob.read_shift(4);}}var cbf=parse_XLSCellParsedFormula(blob,end-blob.l,opts);return {cell:cell,val:val[0],formula:cbf,shared:flags>>3&1,tt:val[1]};}/* XLSB Parsed Formula records have the same shape */function parse_XLSBParsedFormula(data,length,opts){var cce=data.read_shift(4);var rgce=parse_Rgce(data,cce,opts);var cb=data.read_shift(4);var rgcb=cb>0?parse_RgbExtra(data,cb,rgce,opts):null;return [rgce,rgcb];}/* [MS-XLSB] 2.5.97.1 ArrayParsedFormula */var parse_XLSBArrayParsedFormula=parse_XLSBParsedFormula;/* [MS-XLSB] 2.5.97.4 CellParsedFormula */var parse_XLSBCellParsedFormula=parse_XLSBParsedFormula;/* [MS-XLSB] 2.5.97.8 DVParsedFormula */ //var parse_XLSBDVParsedFormula = parse_XLSBParsedFormula;
	/* [MS-XLSB] 2.5.97.9 FRTParsedFormula */ //var parse_XLSBFRTParsedFormula = parse_XLSBParsedFormula2;
	/* [MS-XLSB] 2.5.97.12 NameParsedFormula */var parse_XLSBNameParsedFormula=parse_XLSBParsedFormula;/* [MS-XLSB] 2.5.97.98 SharedParsedFormula */var parse_XLSBSharedParsedFormula=parse_XLSBParsedFormula;var Cetab={0:"BEEP",1:"OPEN",2:"OPEN.LINKS",3:"CLOSE.ALL",4:"SAVE",5:"SAVE.AS",6:"FILE.DELETE",7:"PAGE.SETUP",8:"PRINT",9:"PRINTER.SETUP",10:"QUIT",11:"NEW.WINDOW",12:"ARRANGE.ALL",13:"WINDOW.SIZE",14:"WINDOW.MOVE",15:"FULL",16:"CLOSE",17:"RUN",22:"SET.PRINT.AREA",23:"SET.PRINT.TITLES",24:"SET.PAGE.BREAK",25:"REMOVE.PAGE.BREAK",26:"FONT",27:"DISPLAY",28:"PROTECT.DOCUMENT",29:"PRECISION",30:"A1.R1C1",31:"CALCULATE.NOW",32:"CALCULATION",34:"DATA.FIND",35:"EXTRACT",36:"DATA.DELETE",37:"SET.DATABASE",38:"SET.CRITERIA",39:"SORT",40:"DATA.SERIES",41:"TABLE",42:"FORMAT.NUMBER",43:"ALIGNMENT",44:"STYLE",45:"BORDER",46:"CELL.PROTECTION",47:"COLUMN.WIDTH",48:"UNDO",49:"CUT",50:"COPY",51:"PASTE",52:"CLEAR",53:"PASTE.SPECIAL",54:"EDIT.DELETE",55:"INSERT",56:"FILL.RIGHT",57:"FILL.DOWN",61:"DEFINE.NAME",62:"CREATE.NAMES",63:"FORMULA.GOTO",64:"FORMULA.FIND",65:"SELECT.LAST.CELL",66:"SHOW.ACTIVE.CELL",67:"GALLERY.AREA",68:"GALLERY.BAR",69:"GALLERY.COLUMN",70:"GALLERY.LINE",71:"GALLERY.PIE",72:"GALLERY.SCATTER",73:"COMBINATION",74:"PREFERRED",75:"ADD.OVERLAY",76:"GRIDLINES",77:"SET.PREFERRED",78:"AXES",79:"LEGEND",80:"ATTACH.TEXT",81:"ADD.ARROW",82:"SELECT.CHART",83:"SELECT.PLOT.AREA",84:"PATTERNS",85:"MAIN.CHART",86:"OVERLAY",87:"SCALE",88:"FORMAT.LEGEND",89:"FORMAT.TEXT",90:"EDIT.REPEAT",91:"PARSE",92:"JUSTIFY",93:"HIDE",94:"UNHIDE",95:"WORKSPACE",96:"FORMULA",97:"FORMULA.FILL",98:"FORMULA.ARRAY",99:"DATA.FIND.NEXT",100:"DATA.FIND.PREV",101:"FORMULA.FIND.NEXT",102:"FORMULA.FIND.PREV",103:"ACTIVATE",104:"ACTIVATE.NEXT",105:"ACTIVATE.PREV",106:"UNLOCKED.NEXT",107:"UNLOCKED.PREV",108:"COPY.PICTURE",109:"SELECT",110:"DELETE.NAME",111:"DELETE.FORMAT",112:"VLINE",113:"HLINE",114:"VPAGE",115:"HPAGE",116:"VSCROLL",117:"HSCROLL",118:"ALERT",119:"NEW",120:"CANCEL.COPY",121:"SHOW.CLIPBOARD",122:"MESSAGE",124:"PASTE.LINK",125:"APP.ACTIVATE",126:"DELETE.ARROW",127:"ROW.HEIGHT",128:"FORMAT.MOVE",129:"FORMAT.SIZE",130:"FORMULA.REPLACE",131:"SEND.KEYS",132:"SELECT.SPECIAL",133:"APPLY.NAMES",134:"REPLACE.FONT",135:"FREEZE.PANES",136:"SHOW.INFO",137:"SPLIT",138:"ON.WINDOW",139:"ON.DATA",140:"DISABLE.INPUT",142:"OUTLINE",143:"LIST.NAMES",144:"FILE.CLOSE",145:"SAVE.WORKBOOK",146:"DATA.FORM",147:"COPY.CHART",148:"ON.TIME",149:"WAIT",150:"FORMAT.FONT",151:"FILL.UP",152:"FILL.LEFT",153:"DELETE.OVERLAY",155:"SHORT.MENUS",159:"SET.UPDATE.STATUS",161:"COLOR.PALETTE",162:"DELETE.STYLE",163:"WINDOW.RESTORE",164:"WINDOW.MAXIMIZE",166:"CHANGE.LINK",167:"CALCULATE.DOCUMENT",168:"ON.KEY",169:"APP.RESTORE",170:"APP.MOVE",171:"APP.SIZE",172:"APP.MINIMIZE",173:"APP.MAXIMIZE",174:"BRING.TO.FRONT",175:"SEND.TO.BACK",185:"MAIN.CHART.TYPE",186:"OVERLAY.CHART.TYPE",187:"SELECT.END",188:"OPEN.MAIL",189:"SEND.MAIL",190:"STANDARD.FONT",191:"CONSOLIDATE",192:"SORT.SPECIAL",193:"GALLERY.3D.AREA",194:"GALLERY.3D.COLUMN",195:"GALLERY.3D.LINE",196:"GALLERY.3D.PIE",197:"VIEW.3D",198:"GOAL.SEEK",199:"WORKGROUP",200:"FILL.GROUP",201:"UPDATE.LINK",202:"PROMOTE",203:"DEMOTE",204:"SHOW.DETAIL",206:"UNGROUP",207:"OBJECT.PROPERTIES",208:"SAVE.NEW.OBJECT",209:"SHARE",210:"SHARE.NAME",211:"DUPLICATE",212:"APPLY.STYLE",213:"ASSIGN.TO.OBJECT",214:"OBJECT.PROTECTION",215:"HIDE.OBJECT",216:"SET.EXTRACT",217:"CREATE.PUBLISHER",218:"SUBSCRIBE.TO",219:"ATTRIBUTES",220:"SHOW.TOOLBAR",222:"PRINT.PREVIEW",223:"EDIT.COLOR",224:"SHOW.LEVELS",225:"FORMAT.MAIN",226:"FORMAT.OVERLAY",227:"ON.RECALC",228:"EDIT.SERIES",229:"DEFINE.STYLE",240:"LINE.PRINT",243:"ENTER.DATA",249:"GALLERY.RADAR",250:"MERGE.STYLES",251:"EDITION.OPTIONS",252:"PASTE.PICTURE",253:"PASTE.PICTURE.LINK",254:"SPELLING",256:"ZOOM",259:"INSERT.OBJECT",260:"WINDOW.MINIMIZE",265:"SOUND.NOTE",266:"SOUND.PLAY",267:"FORMAT.SHAPE",268:"EXTEND.POLYGON",269:"FORMAT.AUTO",272:"GALLERY.3D.BAR",273:"GALLERY.3D.SURFACE",274:"FILL.AUTO",276:"CUSTOMIZE.TOOLBAR",277:"ADD.TOOL",278:"EDIT.OBJECT",279:"ON.DOUBLECLICK",280:"ON.ENTRY",281:"WORKBOOK.ADD",282:"WORKBOOK.MOVE",283:"WORKBOOK.COPY",284:"WORKBOOK.OPTIONS",285:"SAVE.WORKSPACE",288:"CHART.WIZARD",289:"DELETE.TOOL",290:"MOVE.TOOL",291:"WORKBOOK.SELECT",292:"WORKBOOK.ACTIVATE",293:"ASSIGN.TO.TOOL",295:"COPY.TOOL",296:"RESET.TOOL",297:"CONSTRAIN.NUMERIC",298:"PASTE.TOOL",302:"WORKBOOK.NEW",305:"SCENARIO.CELLS",306:"SCENARIO.DELETE",307:"SCENARIO.ADD",308:"SCENARIO.EDIT",309:"SCENARIO.SHOW",310:"SCENARIO.SHOW.NEXT",311:"SCENARIO.SUMMARY",312:"PIVOT.TABLE.WIZARD",313:"PIVOT.FIELD.PROPERTIES",314:"PIVOT.FIELD",315:"PIVOT.ITEM",316:"PIVOT.ADD.FIELDS",318:"OPTIONS.CALCULATION",319:"OPTIONS.EDIT",320:"OPTIONS.VIEW",321:"ADDIN.MANAGER",322:"MENU.EDITOR",323:"ATTACH.TOOLBARS",324:"VBAActivate",325:"OPTIONS.CHART",328:"VBA.INSERT.FILE",330:"VBA.PROCEDURE.DEFINITION",336:"ROUTING.SLIP",338:"ROUTE.DOCUMENT",339:"MAIL.LOGON",342:"INSERT.PICTURE",343:"EDIT.TOOL",344:"GALLERY.DOUGHNUT",350:"CHART.TREND",352:"PIVOT.ITEM.PROPERTIES",354:"WORKBOOK.INSERT",355:"OPTIONS.TRANSITION",356:"OPTIONS.GENERAL",370:"FILTER.ADVANCED",373:"MAIL.ADD.MAILER",374:"MAIL.DELETE.MAILER",375:"MAIL.REPLY",376:"MAIL.REPLY.ALL",377:"MAIL.FORWARD",378:"MAIL.NEXT.LETTER",379:"DATA.LABEL",380:"INSERT.TITLE",381:"FONT.PROPERTIES",382:"MACRO.OPTIONS",383:"WORKBOOK.HIDE",384:"WORKBOOK.UNHIDE",385:"WORKBOOK.DELETE",386:"WORKBOOK.NAME",388:"GALLERY.CUSTOM",390:"ADD.CHART.AUTOFORMAT",391:"DELETE.CHART.AUTOFORMAT",392:"CHART.ADD.DATA",393:"AUTO.OUTLINE",394:"TAB.ORDER",395:"SHOW.DIALOG",396:"SELECT.ALL",397:"UNGROUP.SHEETS",398:"SUBTOTAL.CREATE",399:"SUBTOTAL.REMOVE",400:"RENAME.OBJECT",412:"WORKBOOK.SCROLL",413:"WORKBOOK.NEXT",414:"WORKBOOK.PREV",415:"WORKBOOK.TAB.SPLIT",416:"FULL.SCREEN",417:"WORKBOOK.PROTECT",420:"SCROLLBAR.PROPERTIES",421:"PIVOT.SHOW.PAGES",422:"TEXT.TO.COLUMNS",423:"FORMAT.CHARTTYPE",424:"LINK.FORMAT",425:"TRACER.DISPLAY",430:"TRACER.NAVIGATE",431:"TRACER.CLEAR",432:"TRACER.ERROR",433:"PIVOT.FIELD.GROUP",434:"PIVOT.FIELD.UNGROUP",435:"CHECKBOX.PROPERTIES",436:"LABEL.PROPERTIES",437:"LISTBOX.PROPERTIES",438:"EDITBOX.PROPERTIES",439:"PIVOT.REFRESH",440:"LINK.COMBO",441:"OPEN.TEXT",442:"HIDE.DIALOG",443:"SET.DIALOG.FOCUS",444:"ENABLE.OBJECT",445:"PUSHBUTTON.PROPERTIES",446:"SET.DIALOG.DEFAULT",447:"FILTER",448:"FILTER.SHOW.ALL",449:"CLEAR.OUTLINE",450:"FUNCTION.WIZARD",451:"ADD.LIST.ITEM",452:"SET.LIST.ITEM",453:"REMOVE.LIST.ITEM",454:"SELECT.LIST.ITEM",455:"SET.CONTROL.VALUE",456:"SAVE.COPY.AS",458:"OPTIONS.LISTS.ADD",459:"OPTIONS.LISTS.DELETE",460:"SERIES.AXES",461:"SERIES.X",462:"SERIES.Y",463:"ERRORBAR.X",464:"ERRORBAR.Y",465:"FORMAT.CHART",466:"SERIES.ORDER",467:"MAIL.LOGOFF",468:"CLEAR.ROUTING.SLIP",469:"APP.ACTIVATE.MICROSOFT",470:"MAIL.EDIT.MAILER",471:"ON.SHEET",472:"STANDARD.WIDTH",473:"SCENARIO.MERGE",474:"SUMMARY.INFO",475:"FIND.FILE",476:"ACTIVE.CELL.FONT",477:"ENABLE.TIPWIZARD",478:"VBA.MAKE.ADDIN",480:"INSERTDATATABLE",481:"WORKGROUP.OPTIONS",482:"MAIL.SEND.MAILER",485:"AUTOCORRECT",489:"POST.DOCUMENT",491:"PICKLIST",493:"VIEW.SHOW",494:"VIEW.DEFINE",495:"VIEW.DELETE",509:"SHEET.BACKGROUND",510:"INSERT.MAP.OBJECT",511:"OPTIONS.MENONO",517:"MSOCHECKS",518:"NORMAL",519:"LAYOUT",520:"RM.PRINT.AREA",521:"CLEAR.PRINT.AREA",522:"ADD.PRINT.AREA",523:"MOVE.BRK",545:"HIDECURR.NOTE",546:"HIDEALL.NOTES",547:"DELETE.NOTE",548:"TRAVERSE.NOTES",549:"ACTIVATE.NOTES",620:"PROTECT.REVISIONS",621:"UNPROTECT.REVISIONS",647:"OPTIONS.ME",653:"WEB.PUBLISH",667:"NEWWEBQUERY",673:"PIVOT.TABLE.CHART",753:"OPTIONS.SAVE",755:"OPTIONS.SPELL",808:"HIDEALL.INKANNOTS"};var Ftab={0:"COUNT",1:"IF",2:"ISNA",3:"ISERROR",4:"SUM",5:"AVERAGE",6:"MIN",7:"MAX",8:"ROW",9:"COLUMN",10:"NA",11:"NPV",12:"STDEV",13:"DOLLAR",14:"FIXED",15:"SIN",16:"COS",17:"TAN",18:"ATAN",19:"PI",20:"SQRT",21:"EXP",22:"LN",23:"LOG10",24:"ABS",25:"INT",26:"SIGN",27:"ROUND",28:"LOOKUP",29:"INDEX",30:"REPT",31:"MID",32:"LEN",33:"VALUE",34:"TRUE",35:"FALSE",36:"AND",37:"OR",38:"NOT",39:"MOD",40:"DCOUNT",41:"DSUM",42:"DAVERAGE",43:"DMIN",44:"DMAX",45:"DSTDEV",46:"VAR",47:"DVAR",48:"TEXT",49:"LINEST",50:"TREND",51:"LOGEST",52:"GROWTH",53:"GOTO",54:"HALT",55:"RETURN",56:"PV",57:"FV",58:"NPER",59:"PMT",60:"RATE",61:"MIRR",62:"IRR",63:"RAND",64:"MATCH",65:"DATE",66:"TIME",67:"DAY",68:"MONTH",69:"YEAR",70:"WEEKDAY",71:"HOUR",72:"MINUTE",73:"SECOND",74:"NOW",75:"AREAS",76:"ROWS",77:"COLUMNS",78:"OFFSET",79:"ABSREF",80:"RELREF",81:"ARGUMENT",82:"SEARCH",83:"TRANSPOSE",84:"ERROR",85:"STEP",86:"TYPE",87:"ECHO",88:"SET.NAME",89:"CALLER",90:"DEREF",91:"WINDOWS",92:"SERIES",93:"DOCUMENTS",94:"ACTIVE.CELL",95:"SELECTION",96:"RESULT",97:"ATAN2",98:"ASIN",99:"ACOS",100:"CHOOSE",101:"HLOOKUP",102:"VLOOKUP",103:"LINKS",104:"INPUT",105:"ISREF",106:"GET.FORMULA",107:"GET.NAME",108:"SET.VALUE",109:"LOG",110:"EXEC",111:"CHAR",112:"LOWER",113:"UPPER",114:"PROPER",115:"LEFT",116:"RIGHT",117:"EXACT",118:"TRIM",119:"REPLACE",120:"SUBSTITUTE",121:"CODE",122:"NAMES",123:"DIRECTORY",124:"FIND",125:"CELL",126:"ISERR",127:"ISTEXT",128:"ISNUMBER",129:"ISBLANK",130:"T",131:"N",132:"FOPEN",133:"FCLOSE",134:"FSIZE",135:"FREADLN",136:"FREAD",137:"FWRITELN",138:"FWRITE",139:"FPOS",140:"DATEVALUE",141:"TIMEVALUE",142:"SLN",143:"SYD",144:"DDB",145:"GET.DEF",146:"REFTEXT",147:"TEXTREF",148:"INDIRECT",149:"REGISTER",150:"CALL",151:"ADD.BAR",152:"ADD.MENU",153:"ADD.COMMAND",154:"ENABLE.COMMAND",155:"CHECK.COMMAND",156:"RENAME.COMMAND",157:"SHOW.BAR",158:"DELETE.MENU",159:"DELETE.COMMAND",160:"GET.CHART.ITEM",161:"DIALOG.BOX",162:"CLEAN",163:"MDETERM",164:"MINVERSE",165:"MMULT",166:"FILES",167:"IPMT",168:"PPMT",169:"COUNTA",170:"CANCEL.KEY",171:"FOR",172:"WHILE",173:"BREAK",174:"NEXT",175:"INITIATE",176:"REQUEST",177:"POKE",178:"EXECUTE",179:"TERMINATE",180:"RESTART",181:"HELP",182:"GET.BAR",183:"PRODUCT",184:"FACT",185:"GET.CELL",186:"GET.WORKSPACE",187:"GET.WINDOW",188:"GET.DOCUMENT",189:"DPRODUCT",190:"ISNONTEXT",191:"GET.NOTE",192:"NOTE",193:"STDEVP",194:"VARP",195:"DSTDEVP",196:"DVARP",197:"TRUNC",198:"ISLOGICAL",199:"DCOUNTA",200:"DELETE.BAR",201:"UNREGISTER",204:"USDOLLAR",205:"FINDB",206:"SEARCHB",207:"REPLACEB",208:"LEFTB",209:"RIGHTB",210:"MIDB",211:"LENB",212:"ROUNDUP",213:"ROUNDDOWN",214:"ASC",215:"DBCS",216:"RANK",219:"ADDRESS",220:"DAYS360",221:"TODAY",222:"VDB",223:"ELSE",224:"ELSE.IF",225:"END.IF",226:"FOR.CELL",227:"MEDIAN",228:"SUMPRODUCT",229:"SINH",230:"COSH",231:"TANH",232:"ASINH",233:"ACOSH",234:"ATANH",235:"DGET",236:"CREATE.OBJECT",237:"VOLATILE",238:"LAST.ERROR",239:"CUSTOM.UNDO",240:"CUSTOM.REPEAT",241:"FORMULA.CONVERT",242:"GET.LINK.INFO",243:"TEXT.BOX",244:"INFO",245:"GROUP",246:"GET.OBJECT",247:"DB",248:"PAUSE",251:"RESUME",252:"FREQUENCY",253:"ADD.TOOLBAR",254:"DELETE.TOOLBAR",255:"User",256:"RESET.TOOLBAR",257:"EVALUATE",258:"GET.TOOLBAR",259:"GET.TOOL",260:"SPELLING.CHECK",261:"ERROR.TYPE",262:"APP.TITLE",263:"WINDOW.TITLE",264:"SAVE.TOOLBAR",265:"ENABLE.TOOL",266:"PRESS.TOOL",267:"REGISTER.ID",268:"GET.WORKBOOK",269:"AVEDEV",270:"BETADIST",271:"GAMMALN",272:"BETAINV",273:"BINOMDIST",274:"CHIDIST",275:"CHIINV",276:"COMBIN",277:"CONFIDENCE",278:"CRITBINOM",279:"EVEN",280:"EXPONDIST",281:"FDIST",282:"FINV",283:"FISHER",284:"FISHERINV",285:"FLOOR",286:"GAMMADIST",287:"GAMMAINV",288:"CEILING",289:"HYPGEOMDIST",290:"LOGNORMDIST",291:"LOGINV",292:"NEGBINOMDIST",293:"NORMDIST",294:"NORMSDIST",295:"NORMINV",296:"NORMSINV",297:"STANDARDIZE",298:"ODD",299:"PERMUT",300:"POISSON",301:"TDIST",302:"WEIBULL",303:"SUMXMY2",304:"SUMX2MY2",305:"SUMX2PY2",306:"CHITEST",307:"CORREL",308:"COVAR",309:"FORECAST",310:"FTEST",311:"INTERCEPT",312:"PEARSON",313:"RSQ",314:"STEYX",315:"SLOPE",316:"TTEST",317:"PROB",318:"DEVSQ",319:"GEOMEAN",320:"HARMEAN",321:"SUMSQ",322:"KURT",323:"SKEW",324:"ZTEST",325:"LARGE",326:"SMALL",327:"QUARTILE",328:"PERCENTILE",329:"PERCENTRANK",330:"MODE",331:"TRIMMEAN",332:"TINV",334:"MOVIE.COMMAND",335:"GET.MOVIE",336:"CONCATENATE",337:"POWER",338:"PIVOT.ADD.DATA",339:"GET.PIVOT.TABLE",340:"GET.PIVOT.FIELD",341:"GET.PIVOT.ITEM",342:"RADIANS",343:"DEGREES",344:"SUBTOTAL",345:"SUMIF",346:"COUNTIF",347:"COUNTBLANK",348:"SCENARIO.GET",349:"OPTIONS.LISTS.GET",350:"ISPMT",351:"DATEDIF",352:"DATESTRING",353:"NUMBERSTRING",354:"ROMAN",355:"OPEN.DIALOG",356:"SAVE.DIALOG",357:"VIEW.GET",358:"GETPIVOTDATA",359:"HYPERLINK",360:"PHONETIC",361:"AVERAGEA",362:"MAXA",363:"MINA",364:"STDEVPA",365:"VARPA",366:"STDEVA",367:"VARA",368:"BAHTTEXT",369:"THAIDAYOFWEEK",370:"THAIDIGIT",371:"THAIMONTHOFYEAR",372:"THAINUMSOUND",373:"THAINUMSTRING",374:"THAISTRINGLENGTH",375:"ISTHAIDIGIT",376:"ROUNDBAHTDOWN",377:"ROUNDBAHTUP",378:"THAIYEAR",379:"RTD",380:"CUBEVALUE",381:"CUBEMEMBER",382:"CUBEMEMBERPROPERTY",383:"CUBERANKEDMEMBER",384:"HEX2BIN",385:"HEX2DEC",386:"HEX2OCT",387:"DEC2BIN",388:"DEC2HEX",389:"DEC2OCT",390:"OCT2BIN",391:"OCT2HEX",392:"OCT2DEC",393:"BIN2DEC",394:"BIN2OCT",395:"BIN2HEX",396:"IMSUB",397:"IMDIV",398:"IMPOWER",399:"IMABS",400:"IMSQRT",401:"IMLN",402:"IMLOG2",403:"IMLOG10",404:"IMSIN",405:"IMCOS",406:"IMEXP",407:"IMARGUMENT",408:"IMCONJUGATE",409:"IMAGINARY",410:"IMREAL",411:"COMPLEX",412:"IMSUM",413:"IMPRODUCT",414:"SERIESSUM",415:"FACTDOUBLE",416:"SQRTPI",417:"QUOTIENT",418:"DELTA",419:"GESTEP",420:"ISEVEN",421:"ISODD",422:"MROUND",423:"ERF",424:"ERFC",425:"BESSELJ",426:"BESSELK",427:"BESSELY",428:"BESSELI",429:"XIRR",430:"XNPV",431:"PRICEMAT",432:"YIELDMAT",433:"INTRATE",434:"RECEIVED",435:"DISC",436:"PRICEDISC",437:"YIELDDISC",438:"TBILLEQ",439:"TBILLPRICE",440:"TBILLYIELD",441:"PRICE",442:"YIELD",443:"DOLLARDE",444:"DOLLARFR",445:"NOMINAL",446:"EFFECT",447:"CUMPRINC",448:"CUMIPMT",449:"EDATE",450:"EOMONTH",451:"YEARFRAC",452:"COUPDAYBS",453:"COUPDAYS",454:"COUPDAYSNC",455:"COUPNCD",456:"COUPNUM",457:"COUPPCD",458:"DURATION",459:"MDURATION",460:"ODDLPRICE",461:"ODDLYIELD",462:"ODDFPRICE",463:"ODDFYIELD",464:"RANDBETWEEN",465:"WEEKNUM",466:"AMORDEGRC",467:"AMORLINC",468:"CONVERT",724:"SHEETJS",469:"ACCRINT",470:"ACCRINTM",471:"WORKDAY",472:"NETWORKDAYS",473:"GCD",474:"MULTINOMIAL",475:"LCM",476:"FVSCHEDULE",477:"CUBEKPIMEMBER",478:"CUBESET",479:"CUBESETCOUNT",480:"IFERROR",481:"COUNTIFS",482:"SUMIFS",483:"AVERAGEIF",484:"AVERAGEIFS"};var FtabArgc={2:1,3:1,10:0,15:1,16:1,17:1,18:1,19:0,20:1,21:1,22:1,23:1,24:1,25:1,26:1,27:2,30:2,31:3,32:1,33:1,34:0,35:0,38:1,39:2,40:3,41:3,42:3,43:3,44:3,45:3,47:3,48:2,53:1,61:3,63:0,65:3,66:3,67:1,68:1,69:1,70:1,71:1,72:1,73:1,74:0,75:1,76:1,77:1,79:2,80:2,83:1,85:0,86:1,89:0,90:1,94:0,95:0,97:2,98:1,99:1,101:3,102:3,105:1,106:1,108:2,111:1,112:1,113:1,114:1,117:2,118:1,119:4,121:1,126:1,127:1,128:1,129:1,130:1,131:1,133:1,134:1,135:1,136:2,137:2,138:2,140:1,141:1,142:3,143:4,144:4,161:1,162:1,163:1,164:1,165:2,172:1,175:2,176:2,177:3,178:2,179:1,184:1,186:1,189:3,190:1,195:3,196:3,197:1,198:1,199:3,201:1,207:4,210:3,211:1,212:2,213:2,214:1,215:1,225:0,229:1,230:1,231:1,232:1,233:1,234:1,235:3,244:1,247:4,252:2,257:1,261:1,271:1,273:4,274:2,275:2,276:2,277:3,278:3,279:1,280:3,281:3,282:3,283:1,284:1,285:2,286:4,287:3,288:2,289:4,290:3,291:3,292:3,293:4,294:1,295:3,296:1,297:3,298:1,299:2,300:3,301:3,302:4,303:2,304:2,305:2,306:2,307:2,308:2,309:3,310:2,311:2,312:2,313:2,314:2,315:2,316:4,325:2,326:2,327:2,328:2,331:2,332:2,337:2,342:1,343:1,346:2,347:1,350:4,351:3,352:1,353:2,360:1,368:1,369:1,370:1,371:1,372:1,373:1,374:1,375:1,376:1,377:1,378:1,382:3,385:1,392:1,393:1,396:2,397:2,398:2,399:1,400:1,401:1,402:1,403:1,404:1,405:1,406:1,407:1,408:1,409:1,410:1,414:4,415:1,416:1,417:2,420:1,421:1,422:2,424:1,425:2,426:2,427:2,428:2,430:3,438:3,439:3,440:3,443:2,444:2,445:2,446:2,447:6,448:6,449:2,450:2,464:2,468:3,476:2,479:1,480:2,65535:0};/* Part 3 TODO: actually parse formulae */function ods_to_csf_formula(f/*:string*/)/*:string*/{if(f.slice(0,3)=="of:")f=f.slice(3);/* 5.2 Basic Expressions */if(f.charCodeAt(0)==61){f=f.slice(1);if(f.charCodeAt(0)==61)f=f.slice(1);}f=f.replace(/COM\.MICROSOFT\./g,"");/* Part 3 Section 5.8 References */f=f.replace(/\[((?:\.[A-Z]+[0-9]+)(?::\.[A-Z]+[0-9]+)?)\]/g,function($$,$1){return $1.replace(/\./g,"");});/* TODO: something other than this */f=f.replace(/\[.(#[A-Z]*[?!])\]/g,"$1");return f.replace(/[;~]/g,",").replace(/\|/g,";");}function ods_to_csf_3D(r/*:string*/)/*:[string, string]*/{var a=r.split(":");var s=a[0].split(".")[0];return [s,a[0].split(".")[1]+(a.length>1?":"+(a[1].split(".")[1]||a[1].split(".")[0]):"")];}var strs={};// shared strings
	var _ssfopts={};// spreadsheet formatting options
	function default_margins(margins/*:Margins*/,mode/*:?string*/){if(!margins)return;var defs=[0.7,0.7,0.75,0.75,0.3,0.3];if(mode=='xlml')defs=[1,1,1,1,0.5,0.5];if(margins.left==null)margins.left=defs[0];if(margins.right==null)margins.right=defs[1];if(margins.top==null)margins.top=defs[2];if(margins.bottom==null)margins.bottom=defs[3];if(margins.header==null)margins.header=defs[4];if(margins.footer==null)margins.footer=defs[5];}function safe_format(p/*:Cell*/,fmtid/*:number*/,fillid/*:?number*/,opts,themes,styles){try{if(opts.cellNF)p.z=table_fmt[fmtid];}catch(e){if(opts.WTF)throw e;}if(p.t==='z'&&!opts.cellStyles)return;if(p.t==='d'&&typeof p.v==='string')p.v=parseDate(p.v);if((!opts||opts.cellText!==false)&&p.t!=='z')try{if(table_fmt[fmtid]==null)SSF_load(SSFImplicit[fmtid]||"General",fmtid);if(p.t==='e')p.w=p.w||BErr[p.v];else if(fmtid===0){if(p.t==='n'){if((p.v|0)===p.v)p.w=p.v.toString(10);else p.w=SSF_general_num(p.v);}else if(p.t==='d'){var dd=datenum(p.v);if((dd|0)===dd)p.w=dd.toString(10);else p.w=SSF_general_num(dd);}else if(p.v===undefined)return "";else p.w=SSF_general(p.v,_ssfopts);}else if(p.t==='d')p.w=SSF_format(fmtid,datenum(p.v),_ssfopts);else p.w=SSF_format(fmtid,p.v,_ssfopts);}catch(e){if(opts.WTF)throw e;}if(!opts.cellStyles)return;if(fillid!=null)try{p.s=styles.Fills[fillid];if(p.s.fgColor&&p.s.fgColor.theme&&!p.s.fgColor.rgb){p.s.fgColor.rgb=rgb_tint(themes.themeElements.clrScheme[p.s.fgColor.theme].rgb,p.s.fgColor.tint||0);if(opts.WTF)p.s.fgColor.raw_rgb=themes.themeElements.clrScheme[p.s.fgColor.theme].rgb;}if(p.s.bgColor&&p.s.bgColor.theme){p.s.bgColor.rgb=rgb_tint(themes.themeElements.clrScheme[p.s.bgColor.theme].rgb,p.s.bgColor.tint||0);if(opts.WTF)p.s.bgColor.raw_rgb=themes.themeElements.clrScheme[p.s.bgColor.theme].rgb;}}catch(e){if(opts.WTF&&styles.Fills)throw e;}}function parse_ws_xml_dim(ws/*:Worksheet*/,s/*:string*/){var d=safe_decode_range(s);if(d.s.r<=d.e.r&&d.s.c<=d.e.c&&d.s.r>=0&&d.s.c>=0)ws["!ref"]=encode_range(d);}var mergecregex=/<(?:\w:)?mergeCell ref="[A-Z0-9:]+"\s*[\/]?>/g;var sheetdataregex=/<(?:\w+:)?sheetData[^>]*>([\s\S]*)<\/(?:\w+:)?sheetData>/;var hlinkregex=/<(?:\w:)?hyperlink [^>]*>/mg;var dimregex=/"(\w*:\w*)"/;var colregex=/<(?:\w:)?col\b[^>]*[\/]?>/g;var afregex=/<(?:\w:)?autoFilter[^>]*([\/]|>([\s\S]*)<\/(?:\w:)?autoFilter)>/g;var marginregex=/<(?:\w:)?pageMargins[^>]*\/>/g;var sheetprregex=/<(?:\w:)?sheetPr\b(?:[^>a-z][^>]*)?\/>/;var sheetprregex2=/<(?:\w:)?sheetPr[^>]*(?:[\/]|>([\s\S]*)<\/(?:\w:)?sheetPr)>/;var svsregex=/<(?:\w:)?sheetViews[^>]*(?:[\/]|>([\s\S]*)<\/(?:\w:)?sheetViews)>/;/* 18.3 Worksheets */function parse_ws_xml(data/*:?string*/,opts,idx/*:number*/,rels,wb/*:WBWBProps*/,themes,styles)/*:Worksheet*/{if(!data)return data;if(!rels)rels={'!id':{}};/* 18.3.1.99 worksheet CT_Worksheet */var s=opts.dense?[]/*:any*/:{}/*:any*/;var refguess/*:Range*/={s:{r:2000000,c:2000000},e:{r:0,c:0}}/*:any*/;var data1="",data2="";var mtch/*:?any*/=data.match(sheetdataregex);if(mtch){data1=data.slice(0,mtch.index);data2=data.slice(mtch.index+mtch[0].length);}else data1=data2=data;/* 18.3.1.82 sheetPr CT_SheetPr */var sheetPr=data1.match(sheetprregex);if(sheetPr)parse_ws_xml_sheetpr(sheetPr[0],s,wb,idx);else if(sheetPr=data1.match(sheetprregex2))parse_ws_xml_sheetpr2(sheetPr[0],sheetPr[1]||"",s,wb,idx);/* 18.3.1.35 dimension CT_SheetDimension */var ridx=(data1.match(/<(?:\w*:)?dimension/)||{index:-1}).index;if(ridx>0){var ref=data1.slice(ridx,ridx+50).match(dimregex);if(ref)parse_ws_xml_dim(s,ref[1]);}/* 18.3.1.88 sheetViews CT_SheetViews */var svs=data1.match(svsregex);if(svs&&svs[1])parse_ws_xml_sheetviews(svs[1],wb);/* 18.3.1.17 cols CT_Cols */var columns/*:Array<ColInfo>*/=[];if(opts.cellStyles){/* 18.3.1.13 col CT_Col */var cols=data1.match(colregex);if(cols)parse_ws_xml_cols(columns,cols);}/* 18.3.1.80 sheetData CT_SheetData ? */if(mtch)parse_ws_xml_data(mtch[1],s,opts,refguess,themes,styles);/* 18.3.1.2  autoFilter CT_AutoFilter */var afilter=data2.match(afregex);if(afilter)s['!autofilter']=parse_ws_xml_autofilter(afilter[0]);/* 18.3.1.55 mergeCells CT_MergeCells */var merges/*:Array<Range>*/=[];var _merge=data2.match(mergecregex);if(_merge)for(ridx=0;ridx!=_merge.length;++ridx)merges[ridx]=safe_decode_range(_merge[ridx].slice(_merge[ridx].indexOf("\"")+1));/* 18.3.1.48 hyperlinks CT_Hyperlinks */var hlink=data2.match(hlinkregex);if(hlink)parse_ws_xml_hlinks(s,hlink,rels);/* 18.3.1.62 pageMargins CT_PageMargins */var margins=data2.match(marginregex);if(margins)s['!margins']=parse_ws_xml_margins(parsexmltag(margins[0]));if(!s["!ref"]&&refguess.e.c>=refguess.s.c&&refguess.e.r>=refguess.s.r)s["!ref"]=encode_range(refguess);if(opts.sheetRows>0&&s["!ref"]){var tmpref=safe_decode_range(s["!ref"]);if(opts.sheetRows<=+tmpref.e.r){tmpref.e.r=opts.sheetRows-1;if(tmpref.e.r>refguess.e.r)tmpref.e.r=refguess.e.r;if(tmpref.e.r<tmpref.s.r)tmpref.s.r=tmpref.e.r;if(tmpref.e.c>refguess.e.c)tmpref.e.c=refguess.e.c;if(tmpref.e.c<tmpref.s.c)tmpref.s.c=tmpref.e.c;s["!fullref"]=s["!ref"];s["!ref"]=encode_range(tmpref);}}if(columns.length>0)s["!cols"]=columns;if(merges.length>0)s["!merges"]=merges;return s;}/* 18.3.1.82-3 sheetPr CT_ChartsheetPr / CT_SheetPr */function parse_ws_xml_sheetpr(sheetPr/*:string*/,s,wb/*:WBWBProps*/,idx/*:number*/){var data=parsexmltag(sheetPr);if(!wb.Sheets[idx])wb.Sheets[idx]={};if(data.codeName)wb.Sheets[idx].CodeName=unescapexml(utf8read(data.codeName));}function parse_ws_xml_sheetpr2(sheetPr/*:string*/,body/*:string*/,s,wb/*:WBWBProps*/,idx/*:number*/){parse_ws_xml_sheetpr(sheetPr.slice(0,sheetPr.indexOf(">")),s,wb,idx);}function parse_ws_xml_hlinks(s,data/*:Array<string>*/,rels){var dense=Array.isArray(s);for(var i=0;i!=data.length;++i){var val=parsexmltag(utf8read(data[i]),true);if(!val.ref)return;var rel=((rels||{})['!id']||[])[val.id];if(rel){val.Target=rel.Target;if(val.location)val.Target+="#"+unescapexml(val.location);}else {val.Target="#"+unescapexml(val.location);rel={Target:val.Target,TargetMode:'Internal'};}val.Rel=rel;if(val.tooltip){val.Tooltip=val.tooltip;delete val.tooltip;}var rng=safe_decode_range(val.ref);for(var R=rng.s.r;R<=rng.e.r;++R)for(var C=rng.s.c;C<=rng.e.c;++C){var addr=encode_cell({c:C,r:R});if(dense){if(!s[R])s[R]=[];if(!s[R][C])s[R][C]={t:"z",v:undefined};s[R][C].l=val;}else {if(!s[addr])s[addr]={t:"z",v:undefined};s[addr].l=val;}}}}function parse_ws_xml_margins(margin){var o={};["left","right","top","bottom","header","footer"].forEach(function(k){if(margin[k])o[k]=parseFloat(margin[k]);});return o;}function parse_ws_xml_cols(columns,cols){var seencol=false;for(var coli=0;coli!=cols.length;++coli){var coll=parsexmltag(cols[coli],true);if(coll.hidden)coll.hidden=parsexmlbool(coll.hidden);var colm=parseInt(coll.min,10)-1,colM=parseInt(coll.max,10)-1;if(coll.outlineLevel)coll.level=+coll.outlineLevel||0;delete coll.min;delete coll.max;coll.width=+coll.width;if(!seencol&&coll.width){seencol=true;find_mdw_colw(coll.width);}process_col(coll);while(colm<=colM)columns[colm++]=dup(coll);}}function parse_ws_xml_autofilter(data/*:string*/){var o={ref:(data.match(/ref="([^"]*)"/)||[])[1]};return o;}/* 18.3.1.88 sheetViews CT_SheetViews */ /* 18.3.1.87 sheetView CT_SheetView */var sviewregex=/<(?:\w:)?sheetView(?:[^>a-z][^>]*)?\/?>/;function parse_ws_xml_sheetviews(data,wb/*:WBWBProps*/){if(!wb.Views)wb.Views=[{}];(data.match(sviewregex)||[]).forEach(function(r/*:string*/,i/*:number*/){var tag=parsexmltag(r);// $FlowIgnore
	if(!wb.Views[i])wb.Views[i]={};// $FlowIgnore
	if(+tag.zoomScale)wb.Views[i].zoom=+tag.zoomScale;// $FlowIgnore
	if(parsexmlbool(tag.rightToLeft))wb.Views[i].RTL=true;});}var parse_ws_xml_data=/*#__PURE__*/function(){var cellregex=/<(?:\w+:)?c[ \/>]/,rowregex=/<\/(?:\w+:)?row>/;var rregex=/r=["']([^"']*)["']/,isregex=/<(?:\w+:)?is>([\S\s]*?)<\/(?:\w+:)?is>/;var refregex=/ref=["']([^"']*)["']/;var match_v=matchtag("v"),match_f=matchtag("f");return function parse_ws_xml_data(sdata/*:string*/,s,opts,guess/*:Range*/,themes,styles){var ri=0,x="",cells/*:Array<string>*/=[],cref/*:?Array<string>*/=[],idx=0,i=0,cc=0,d="",p/*:any*/;var tag,tagr=0,tagc=0;var sstr,ftag;var fmtid=0,fillid=0;var do_format=Array.isArray(styles.CellXf),cf;var arrayf/*:Array<[Range, string]>*/=[];var sharedf=[];var dense=Array.isArray(s);var rows/*:Array<RowInfo>*/=[],rowobj={},rowrite=false;var sheetStubs=!!opts.sheetStubs;for(var marr=sdata.split(rowregex),mt=0,marrlen=marr.length;mt!=marrlen;++mt){x=marr[mt].trim();var xlen=x.length;if(xlen===0)continue;/* 18.3.1.73 row CT_Row */var rstarti=0;outa:for(ri=0;ri<xlen;++ri)switch(/*x.charCodeAt(ri)*/x[ri]){case">"/*62*/:if(/*x.charCodeAt(ri-1) != 47*/x[ri-1]!="/"){++ri;break outa;}if(opts&&opts.cellStyles){// TODO: avoid duplication
	tag=parsexmltag(x.slice(rstarti,ri),true);tagr=tag.r!=null?parseInt(tag.r,10):tagr+1;tagc=-1;if(opts.sheetRows&&opts.sheetRows<tagr)continue;rowobj={};rowrite=false;if(tag.ht){rowrite=true;rowobj.hpt=parseFloat(tag.ht);rowobj.hpx=pt2px(rowobj.hpt);}if(tag.hidden=="1"){rowrite=true;rowobj.hidden=true;}if(tag.outlineLevel!=null){rowrite=true;rowobj.level=+tag.outlineLevel;}if(rowrite)rows[tagr-1]=rowobj;}break;case"<"/*60*/:rstarti=ri;break;}if(rstarti>=ri)break;tag=parsexmltag(x.slice(rstarti,ri),true);tagr=tag.r!=null?parseInt(tag.r,10):tagr+1;tagc=-1;if(opts.sheetRows&&opts.sheetRows<tagr)continue;if(guess.s.r>tagr-1)guess.s.r=tagr-1;if(guess.e.r<tagr-1)guess.e.r=tagr-1;if(opts&&opts.cellStyles){rowobj={};rowrite=false;if(tag.ht){rowrite=true;rowobj.hpt=parseFloat(tag.ht);rowobj.hpx=pt2px(rowobj.hpt);}if(tag.hidden=="1"){rowrite=true;rowobj.hidden=true;}if(tag.outlineLevel!=null){rowrite=true;rowobj.level=+tag.outlineLevel;}if(rowrite)rows[tagr-1]=rowobj;}/* 18.3.1.4 c CT_Cell */cells=x.slice(ri).split(cellregex);for(var rslice=0;rslice!=cells.length;++rslice)if(cells[rslice].trim().charAt(0)!="<")break;cells=cells.slice(rslice);for(ri=0;ri!=cells.length;++ri){x=cells[ri].trim();if(x.length===0)continue;cref=x.match(rregex);idx=ri;i=0;cc=0;x="<c "+(x.slice(0,1)=="<"?">":"")+x;if(cref!=null&&cref.length===2){idx=0;d=cref[1];for(i=0;i!=d.length;++i){if((cc=d.charCodeAt(i)-64)<1||cc>26)break;idx=26*idx+cc;}--idx;tagc=idx;}else ++tagc;for(i=0;i!=x.length;++i)if(x.charCodeAt(i)===62)break;++i;tag=parsexmltag(x.slice(0,i),true);if(!tag.r)tag.r=encode_cell({r:tagr-1,c:tagc});d=x.slice(i);p={t:""}/*:any*/;if((cref=d.match(match_v))!=null&&/*::cref != null && */cref[1]!=='')p.v=unescapexml(cref[1]);if(opts.cellFormula){if((cref=d.match(match_f))!=null&&/*::cref != null && */cref[1]!==''){/* TODO: match against XLSXFutureFunctions */p.f=unescapexml(utf8read(cref[1])).replace(/\r\n/g,"\n");if(!opts.xlfn)p.f=_xlfn(p.f);if(/*::cref != null && cref[0] != null && */cref[0].indexOf('t="array"')>-1){p.F=(d.match(refregex)||[])[1];if(p.F.indexOf(":")>-1)arrayf.push([safe_decode_range(p.F),p.F]);}else if(/*::cref != null && cref[0] != null && */cref[0].indexOf('t="shared"')>-1){// TODO: parse formula
	ftag=parsexmltag(cref[0]);var ___f=unescapexml(utf8read(cref[1]));if(!opts.xlfn)___f=_xlfn(___f);sharedf[parseInt(ftag.si,10)]=[ftag,___f,tag.r];}}else if(cref=d.match(/<f[^>]*\/>/)){ftag=parsexmltag(cref[0]);if(sharedf[ftag.si])p.f=shift_formula_xlsx(sharedf[ftag.si][1],sharedf[ftag.si][2]/*[0].ref*/,tag.r);}/* TODO: factor out contains logic */var _tag=decode_cell(tag.r);for(i=0;i<arrayf.length;++i)if(_tag.r>=arrayf[i][0].s.r&&_tag.r<=arrayf[i][0].e.r)if(_tag.c>=arrayf[i][0].s.c&&_tag.c<=arrayf[i][0].e.c)p.F=arrayf[i][1];}if(tag.t==null&&p.v===undefined){if(p.f||p.F){p.v=0;p.t="n";}else if(!sheetStubs)continue;else p.t="z";}else p.t=tag.t||"n";if(guess.s.c>tagc)guess.s.c=tagc;if(guess.e.c<tagc)guess.e.c=tagc;/* 18.18.11 t ST_CellType */switch(p.t){case'n':if(p.v==""||p.v==null){if(!sheetStubs)continue;p.t='z';}else p.v=parseFloat(p.v);break;case's':if(typeof p.v=='undefined'){if(!sheetStubs)continue;p.t='z';}else {sstr=strs[parseInt(p.v,10)];p.v=sstr.t;p.r=sstr.r;if(opts.cellHTML)p.h=sstr.h;}break;case'str':p.t="s";p.v=p.v!=null?utf8read(p.v):'';if(opts.cellHTML)p.h=escapehtml(p.v);break;case'inlineStr':cref=d.match(isregex);p.t='s';if(cref!=null&&(sstr=parse_si(cref[1]))){p.v=sstr.t;if(opts.cellHTML)p.h=sstr.h;}else p.v="";break;case'b':p.v=parsexmlbool(p.v);break;case'd':if(opts.cellDates)p.v=parseDate(p.v,1);else {p.v=datenum(parseDate(p.v,1));p.t='n';}break;/* error string in .w, number in .v */case'e':if(!opts||opts.cellText!==false)p.w=p.v;p.v=RBErr[p.v];break;}/* formatting */fmtid=fillid=0;cf=null;if(do_format&&tag.s!==undefined){cf=styles.CellXf[tag.s];if(cf!=null){if(cf.numFmtId!=null)fmtid=cf.numFmtId;if(opts.cellStyles){if(cf.fillId!=null)fillid=cf.fillId;}}}safe_format(p,fmtid,fillid,opts,themes,styles);if(opts.cellDates&&do_format&&p.t=='n'&&fmt_is_date(table_fmt[fmtid])){p.t='d';p.v=numdate(p.v);}if(tag.cm&&opts.xlmeta){var cm=(opts.xlmeta.Cell||[])[+tag.cm-1];if(cm&&cm.type=='XLDAPR')p.D=true;}if(dense){var _r=decode_cell(tag.r);if(!s[_r.r])s[_r.r]=[];s[_r.r][_r.c]=p;}else s[tag.r]=p;}}if(rows.length>0)s['!rows']=rows;};}();/* [MS-XLSB] 2.4.726 BrtRowHdr */function parse_BrtRowHdr(data,length){var z={}/*:any*/;var tgt=data.l+length;z.r=data.read_shift(4);data.l+=4;// TODO: ixfe
	var miyRw=data.read_shift(2);data.l+=1;// TODO: top/bot padding
	var flags=data.read_shift(1);data.l=tgt;if(flags&0x07)z.level=flags&0x07;if(flags&0x10)z.hidden=true;if(flags&0x20)z.hpt=miyRw/20;return z;}/* [MS-XLSB] 2.4.820 BrtWsDim */var parse_BrtWsDim=parse_UncheckedRfX;/* [MS-XLSB] 2.4.821 BrtWsFmtInfo */function/*::data, length*/parse_BrtWsFmtInfo(){}//function write_BrtWsFmtInfo(ws, o) { }
	/* [MS-XLSB] 2.4.823 BrtWsProp */function parse_BrtWsProp(data,length){var z={};var f=data[data.l];++data.l;z.above=!(f&0x40);z.left=!(f&0x80);/* TODO: pull flags */data.l+=18;z.name=parse_XLSBCodeName(data);return z;}/* [MS-XLSB] 2.4.306 BrtCellBlank */function parse_BrtCellBlank(data){var cell=parse_XLSBCell(data);return [cell];}function parse_BrtShortBlank(data){var cell=parse_XLSBShortCell(data);return [cell];}/* [MS-XLSB] 2.4.307 BrtCellBool */function parse_BrtCellBool(data){var cell=parse_XLSBCell(data);var fBool=data.read_shift(1);return [cell,fBool,'b'];}function parse_BrtShortBool(data){var cell=parse_XLSBShortCell(data);var fBool=data.read_shift(1);return [cell,fBool,'b'];}/* [MS-XLSB] 2.4.308 BrtCellError */function parse_BrtCellError(data){var cell=parse_XLSBCell(data);var bError=data.read_shift(1);return [cell,bError,'e'];}function parse_BrtShortError(data){var cell=parse_XLSBShortCell(data);var bError=data.read_shift(1);return [cell,bError,'e'];}/* [MS-XLSB] 2.4.311 BrtCellIsst */function parse_BrtCellIsst(data){var cell=parse_XLSBCell(data);var isst=data.read_shift(4);return [cell,isst,'s'];}function parse_BrtShortIsst(data){var cell=parse_XLSBShortCell(data);var isst=data.read_shift(4);return [cell,isst,'s'];}/* [MS-XLSB] 2.4.313 BrtCellReal */function parse_BrtCellReal(data){var cell=parse_XLSBCell(data);var value=parse_Xnum(data);return [cell,value,'n'];}function parse_BrtShortReal(data){var cell=parse_XLSBShortCell(data);var value=parse_Xnum(data);return [cell,value,'n'];}/* [MS-XLSB] 2.4.314 BrtCellRk */function parse_BrtCellRk(data){var cell=parse_XLSBCell(data);var value=parse_RkNumber(data);return [cell,value,'n'];}function parse_BrtShortRk(data){var cell=parse_XLSBShortCell(data);var value=parse_RkNumber(data);return [cell,value,'n'];}/* [MS-XLSB] 2.4.323 BrtCellRString */function parse_BrtCellRString(data){var cell=parse_XLSBCell(data);var value=parse_RichStr(data);return [cell,value,'is'];}/* [MS-XLSB] 2.4.317 BrtCellSt */function parse_BrtCellSt(data){var cell=parse_XLSBCell(data);var value=parse_XLWideString(data);return [cell,value,'str'];}function parse_BrtShortSt(data){var cell=parse_XLSBShortCell(data);var value=parse_XLWideString(data);return [cell,value,'str'];}/* [MS-XLSB] 2.4.653 BrtFmlaBool */function parse_BrtFmlaBool(data,length,opts){var end=data.l+length;var cell=parse_XLSBCell(data);cell.r=opts['!row'];var value=data.read_shift(1);var o=[cell,value,'b'];if(opts.cellFormula){data.l+=2;var formula=parse_XLSBCellParsedFormula(data,end-data.l,opts);o[3]=stringify_formula(formula,null/*range*/,cell,opts.supbooks,opts);/* TODO */}else data.l=end;return o;}/* [MS-XLSB] 2.4.654 BrtFmlaError */function parse_BrtFmlaError(data,length,opts){var end=data.l+length;var cell=parse_XLSBCell(data);cell.r=opts['!row'];var value=data.read_shift(1);var o=[cell,value,'e'];if(opts.cellFormula){data.l+=2;var formula=parse_XLSBCellParsedFormula(data,end-data.l,opts);o[3]=stringify_formula(formula,null/*range*/,cell,opts.supbooks,opts);/* TODO */}else data.l=end;return o;}/* [MS-XLSB] 2.4.655 BrtFmlaNum */function parse_BrtFmlaNum(data,length,opts){var end=data.l+length;var cell=parse_XLSBCell(data);cell.r=opts['!row'];var value=parse_Xnum(data);var o=[cell,value,'n'];if(opts.cellFormula){data.l+=2;var formula=parse_XLSBCellParsedFormula(data,end-data.l,opts);o[3]=stringify_formula(formula,null/*range*/,cell,opts.supbooks,opts);/* TODO */}else data.l=end;return o;}/* [MS-XLSB] 2.4.656 BrtFmlaString */function parse_BrtFmlaString(data,length,opts){var end=data.l+length;var cell=parse_XLSBCell(data);cell.r=opts['!row'];var value=parse_XLWideString(data);var o=[cell,value,'str'];if(opts.cellFormula){data.l+=2;var formula=parse_XLSBCellParsedFormula(data,end-data.l,opts);o[3]=stringify_formula(formula,null/*range*/,cell,opts.supbooks,opts);/* TODO */}else data.l=end;return o;}/* [MS-XLSB] 2.4.682 BrtMergeCell */var parse_BrtMergeCell=parse_UncheckedRfX;/* [MS-XLSB] 2.4.662 BrtHLink */function parse_BrtHLink(data,length/*::, opts*/){var end=data.l+length;var rfx=parse_UncheckedRfX(data);var relId=parse_XLNullableWideString(data);var loc=parse_XLWideString(data);var tooltip=parse_XLWideString(data);var display=parse_XLWideString(data);data.l=end;var o={rfx:rfx,relId:relId,loc:loc,display:display}/*:any*/;if(tooltip)o.Tooltip=tooltip;return o;}/* [MS-XLSB] 2.4.692 BrtPane */function/*data, length, opts*/parse_BrtPane(){}/* [MS-XLSB] 2.4.6 BrtArrFmla */function parse_BrtArrFmla(data,length,opts){var end=data.l+length;var rfx=parse_RfX(data);var fAlwaysCalc=data.read_shift(1);var o=[rfx];o[2]=fAlwaysCalc;if(opts.cellFormula){var formula=parse_XLSBArrayParsedFormula(data,end-data.l,opts);o[1]=formula;}else data.l=end;return o;}/* [MS-XLSB] 2.4.750 BrtShrFmla */function parse_BrtShrFmla(data,length,opts){var end=data.l+length;var rfx=parse_UncheckedRfX(data);var o=[rfx];if(opts.cellFormula){var formula=parse_XLSBSharedParsedFormula(data,end-data.l,opts);o[1]=formula;data.l=end;}else data.l=end;return o;}/* [MS-XLSB] 2.4.678 BrtMargins */var BrtMarginKeys=["left","right","top","bottom","header","footer"];function parse_BrtMargins(data/*::, length, opts*/)/*:Margins*/{var margins={}/*:any*/;BrtMarginKeys.forEach(function(k){margins[k]=parse_Xnum(data);});return margins;}/* [MS-XLSB] 2.4.299 BrtBeginWsView */function parse_BrtBeginWsView(data/*::, length, opts*/){var f=data.read_shift(2);data.l+=28;return {RTL:f&0x20};}function/*data, length, opts*/parse_BrtDVal(){}function/*data, length, opts*/parse_BrtDVal14(){}/* [MS-XLSB] 2.1.7.61 Worksheet */function parse_ws_bin(data,_opts,idx,rels,wb/*:WBWBProps*/,themes,styles)/*:Worksheet*/{if(!data)return data;var opts=_opts||{};if(!rels)rels={'!id':{}};var s/*:Worksheet*/=opts.dense?[]:{};var ref;var refguess={s:{r:2000000,c:2000000},e:{r:0,c:0}};var pass=false,end=false;var row,p,cf,R,C,addr,sstr,rr,cell/*:Cell*/;var merges/*:Array<Range>*/=[];opts.biff=12;opts['!row']=0;var ai=0,af=false;var arrayf/*:Array<[Range, string]>*/=[];var sharedf={};var supbooks=opts.supbooks||/*::(*/wb/*:: :any)*/.supbooks||[[]]/*:any*/;supbooks.sharedf=sharedf;supbooks.arrayf=arrayf;supbooks.SheetNames=wb.SheetNames||wb.Sheets.map(function(x){return x.name;});if(!opts.supbooks){opts.supbooks=supbooks;if(wb.Names)for(var i=0;i<wb.Names.length;++i)supbooks[0][i+1]=wb.Names[i];}var colinfo/*:Array<ColInfo>*/=[],rowinfo/*:Array<RowInfo>*/=[];var seencol=false;XLSBRecordEnum[0x0010]={n:"BrtShortReal",f:parse_BrtShortReal};var cm;recordhopper(data,function ws_parse(val,RR,RT){if(end)return;switch(RT){case 0x0094:/* 'BrtWsDim' */ref=val;break;case 0x0000:/* 'BrtRowHdr' */row=val;if(opts.sheetRows&&opts.sheetRows<=row.r)end=true;rr=encode_row(R=row.r);opts['!row']=row.r;if(val.hidden||val.hpt||val.level!=null){if(val.hpt)val.hpx=pt2px(val.hpt);rowinfo[val.r]=val;}break;case 0x0002:/* 'BrtCellRk' */case 0x0003:/* 'BrtCellError' */case 0x0004:/* 'BrtCellBool' */case 0x0005:/* 'BrtCellReal' */case 0x0006:/* 'BrtCellSt' */case 0x0007:/* 'BrtCellIsst' */case 0x0008:/* 'BrtFmlaString' */case 0x0009:/* 'BrtFmlaNum' */case 0x000A:/* 'BrtFmlaBool' */case 0x000B:/* 'BrtFmlaError' */case 0x000D:/* 'BrtShortRk' */case 0x000E:/* 'BrtShortError' */case 0x000F:/* 'BrtShortBool' */case 0x0010:/* 'BrtShortReal' */case 0x0011:/* 'BrtShortSt' */case 0x0012:/* 'BrtShortIsst' */case 0x003E:/* 'BrtCellRString' */p={t:val[2]}/*:any*/;switch(val[2]){case'n':p.v=val[1];break;case's':sstr=strs[val[1]];p.v=sstr.t;p.r=sstr.r;break;case'b':p.v=val[1]?true:false;break;case'e':p.v=val[1];if(opts.cellText!==false)p.w=BErr[p.v];break;case'str':p.t='s';p.v=val[1];break;case'is':p.t='s';p.v=val[1].t;break;}if(cf=styles.CellXf[val[0].iStyleRef])safe_format(p,cf.numFmtId,null,opts,themes,styles);C=val[0].c==-1?C+1:val[0].c;if(opts.dense){if(!s[R])s[R]=[];s[R][C]=p;}else s[encode_col(C)+rr]=p;if(opts.cellFormula){af=false;for(ai=0;ai<arrayf.length;++ai){var aii=arrayf[ai];if(row.r>=aii[0].s.r&&row.r<=aii[0].e.r)if(C>=aii[0].s.c&&C<=aii[0].e.c){p.F=encode_range(aii[0]);af=true;}}if(!af&&val.length>3)p.f=val[3];}if(refguess.s.r>row.r)refguess.s.r=row.r;if(refguess.s.c>C)refguess.s.c=C;if(refguess.e.r<row.r)refguess.e.r=row.r;if(refguess.e.c<C)refguess.e.c=C;if(opts.cellDates&&cf&&p.t=='n'&&fmt_is_date(table_fmt[cf.numFmtId])){var _d=SSF_parse_date_code(p.v);if(_d){p.t='d';p.v=new Date(_d.y,_d.m-1,_d.d,_d.H,_d.M,_d.S,_d.u);}}if(cm){if(cm.type=='XLDAPR')p.D=true;cm=void 0;}break;case 0x0001:/* 'BrtCellBlank' */case 0x000C:/* 'BrtShortBlank' */if(!opts.sheetStubs||pass)break;p={t:'z',v:void 0}/*:any*/;C=val[0].c==-1?C+1:val[0].c;if(opts.dense){if(!s[R])s[R]=[];s[R][C]=p;}else s[encode_col(C)+rr]=p;if(refguess.s.r>row.r)refguess.s.r=row.r;if(refguess.s.c>C)refguess.s.c=C;if(refguess.e.r<row.r)refguess.e.r=row.r;if(refguess.e.c<C)refguess.e.c=C;if(cm){if(cm.type=='XLDAPR')p.D=true;cm=void 0;}break;case 0x00B0:/* 'BrtMergeCell' */merges.push(val);break;case 0x0031:{/* 'BrtCellMeta' */cm=((opts.xlmeta||{}).Cell||[])[val-1];}break;case 0x01EE:/* 'BrtHLink' */var rel=rels['!id'][val.relId];if(rel){val.Target=rel.Target;if(val.loc)val.Target+="#"+val.loc;val.Rel=rel;}else if(val.relId==''){val.Target="#"+val.loc;}for(R=val.rfx.s.r;R<=val.rfx.e.r;++R)for(C=val.rfx.s.c;C<=val.rfx.e.c;++C){if(opts.dense){if(!s[R])s[R]=[];if(!s[R][C])s[R][C]={t:'z',v:undefined};s[R][C].l=val;}else {addr=encode_cell({c:C,r:R});if(!s[addr])s[addr]={t:'z',v:undefined};s[addr].l=val;}}break;case 0x01AA:/* 'BrtArrFmla' */if(!opts.cellFormula)break;arrayf.push(val);cell=opts.dense?s[R][C]:s[encode_col(C)+rr]/*:any*/;cell.f=stringify_formula(val[1],refguess,{r:row.r,c:C},supbooks,opts);cell.F=encode_range(val[0]);break;case 0x01AB:/* 'BrtShrFmla' */if(!opts.cellFormula)break;sharedf[encode_cell(val[0].s)]=val[1];cell=opts.dense?s[R][C]:s[encode_col(C)+rr];cell.f=stringify_formula(val[1],refguess,{r:row.r,c:C},supbooks,opts);break;/* identical to 'ColInfo' in XLS */case 0x003C:/* 'BrtColInfo' */if(!opts.cellStyles)break;while(val.e>=val.s){colinfo[val.e--]={width:val.w/256,hidden:!!(val.flags&0x01),level:val.level};if(!seencol){seencol=true;find_mdw_colw(val.w/256);}process_col(colinfo[val.e+1]);}break;case 0x00A1:/* 'BrtBeginAFilter' */s['!autofilter']={ref:encode_range(val)};break;case 0x01DC:/* 'BrtMargins' */s['!margins']=val;break;case 0x0093:/* 'BrtWsProp' */if(!wb.Sheets[idx])wb.Sheets[idx]={};if(val.name)wb.Sheets[idx].CodeName=val.name;if(val.above||val.left)s['!outline']={above:val.above,left:val.left};break;case 0x0089:/* 'BrtBeginWsView' */if(!wb.Views)wb.Views=[{}];if(!wb.Views[0])wb.Views[0]={};if(val.RTL)wb.Views[0].RTL=true;break;case 0x01E5:/* 'BrtWsFmtInfo' */break;case 0x0040:/* 'BrtDVal' */case 0x041D:/* 'BrtDVal14' */break;case 0x0097:/* 'BrtPane' */break;case 0x0098:/* 'BrtSel' */case 0x00AF:/* 'BrtAFilterDateGroupItem' */case 0x0284:/* 'BrtActiveX' */case 0x0271:/* 'BrtBigName' */case 0x0232:/* 'BrtBkHim' */case 0x018C:/* 'BrtBrk' */case 0x0458:/* 'BrtCFIcon' */case 0x047A:/* 'BrtCFRuleExt' */case 0x01D7:/* 'BrtCFVO' */case 0x041A:/* 'BrtCFVO14' */case 0x0289:/* 'BrtCellIgnoreEC' */case 0x0451:/* 'BrtCellIgnoreEC14' */case 0x024D:/* 'BrtCellSmartTagProperty' */case 0x025F:/* 'BrtCellWatch' */case 0x0234:/* 'BrtColor' */case 0x041F:/* 'BrtColor14' */case 0x00A8:/* 'BrtColorFilter' */case 0x00AE:/* 'BrtCustomFilter' */case 0x049C:/* 'BrtCustomFilter14' */case 0x01F3:/* 'BrtDRef' */case 0x01FB:/* 'BrtDXF' */case 0x0226:/* 'BrtDrawing' */case 0x00AB:/* 'BrtDynamicFilter' */case 0x00A7:/* 'BrtFilter' */case 0x0499:/* 'BrtFilter14' */case 0x00A9:/* 'BrtIconFilter' */case 0x049D:/* 'BrtIconFilter14' */case 0x0227:/* 'BrtLegacyDrawing' */case 0x0228:/* 'BrtLegacyDrawingHF' */case 0x0295:/* 'BrtListPart' */case 0x027F:/* 'BrtOleObject' */case 0x01DE:/* 'BrtPageSetup' */case 0x0219:/* 'BrtPhoneticInfo' */case 0x01DD:/* 'BrtPrintOptions' */case 0x0218:/* 'BrtRangeProtection' */case 0x044F:/* 'BrtRangeProtection14' */case 0x02A8:/* 'BrtRangeProtectionIso' */case 0x0450:/* 'BrtRangeProtectionIso14' */case 0x0400:/* 'BrtRwDescent' */case 0x0297:/* 'BrtSheetCalcProp' */case 0x0217:/* 'BrtSheetProtection' */case 0x02A6:/* 'BrtSheetProtectionIso' */case 0x01F8:/* 'BrtSlc' */case 0x0413:/* 'BrtSparkline' */case 0x01AC:/* 'BrtTable' */case 0x00AA:/* 'BrtTop10Filter' */case 0x0C00:/* 'BrtUid' */case 0x0032:/* 'BrtValueMeta' */case 0x0816:/* 'BrtWebExtension' */case 0x0415:/* 'BrtWsFmtInfoEx14' */break;case 0x0023:/* 'BrtFRTBegin' */pass=true;break;case 0x0024:/* 'BrtFRTEnd' */pass=false;break;case 0x0025:pass=true;break;case 0x0026:pass=false;break;default:if(RR.T);else if(!pass||opts.WTF)throw new Error("Unexpected record 0x"+RT.toString(16));}},opts);delete opts.supbooks;delete opts['!row'];if(!s["!ref"]&&(refguess.s.r<2000000||ref&&(ref.e.r>0||ref.e.c>0||ref.s.r>0||ref.s.c>0)))s["!ref"]=encode_range(ref||refguess);if(opts.sheetRows&&s["!ref"]){var tmpref=safe_decode_range(s["!ref"]);if(opts.sheetRows<=+tmpref.e.r){tmpref.e.r=opts.sheetRows-1;if(tmpref.e.r>refguess.e.r)tmpref.e.r=refguess.e.r;if(tmpref.e.r<tmpref.s.r)tmpref.s.r=tmpref.e.r;if(tmpref.e.c>refguess.e.c)tmpref.e.c=refguess.e.c;if(tmpref.e.c<tmpref.s.c)tmpref.s.c=tmpref.e.c;s["!fullref"]=s["!ref"];s["!ref"]=encode_range(tmpref);}}if(merges.length>0)s["!merges"]=merges;if(colinfo.length>0)s["!cols"]=colinfo;if(rowinfo.length>0)s["!rows"]=rowinfo;return s;}function parse_Cache(data/*:string*/)/*:[Array<number|string>, string, ?string]*/{var col/*:Array<number|string>*/=[];var num=data.match(/^<c:numCache>/);var f;/* 21.2.2.150 pt CT_NumVal */(data.match(/<c:pt idx="(\d*)">(.*?)<\/c:pt>/mg)||[]).forEach(function(pt){var q=pt.match(/<c:pt idx="(\d*?)"><c:v>(.*)<\/c:v><\/c:pt>/);if(!q)return;col[+q[1]]=num?+q[2]:q[2];});/* 21.2.2.71 formatCode CT_Xstring */var nf=unescapexml((data.match(/<c:formatCode>([\s\S]*?)<\/c:formatCode>/)||["","General"])[1]);(data.match(/<c:f>(.*?)<\/c:f>/mg)||[]).forEach(function(F){f=F.replace(/<.*?>/g,"");});return [col,nf,f];}/* 21.2 DrawingML - Charts */function parse_chart(data/*:?string*/,name/*:string*/,opts,rels,wb,csheet){var cs/*:Worksheet*/=csheet||{"!type":"chart"}/*:any*/;if(!data)return csheet;/* 21.2.2.27 chart CT_Chart */var C=0,R=0,col="A";var refguess={s:{r:2000000,c:2000000},e:{r:0,c:0}};/* 21.2.2.120 numCache CT_NumData */(data.match(/<c:numCache>[\s\S]*?<\/c:numCache>/gm)||[]).forEach(function(nc){var cache=parse_Cache(nc);refguess.s.r=refguess.s.c=0;refguess.e.c=C;col=encode_col(C);cache[0].forEach(function(n,i){cs[col+encode_row(i)]={t:'n',v:n,z:cache[1]};R=i;});if(refguess.e.r<R)refguess.e.r=R;++C;});if(C>0)cs["!ref"]=encode_range(refguess);return cs;}/* 18.3 Worksheets also covers Chartsheets */function parse_cs_xml(data/*:?string*/,opts,idx/*:number*/,rels,wb/*::, themes, styles*/)/*:Worksheet*/{if(!data)return data;/* 18.3.1.12 chartsheet CT_ChartSheet */if(!rels)rels={'!id':{}};var s={'!type':"chart",'!drawel':null,'!rel':""}/*:any*/;var m;/* 18.3.1.83 sheetPr CT_ChartsheetPr */var sheetPr=data.match(sheetprregex);if(sheetPr)parse_ws_xml_sheetpr(sheetPr[0],s,wb,idx);/* 18.3.1.36 drawing CT_Drawing */if(m=data.match(/drawing r:id="(.*?)"/))s['!rel']=m[1];if(rels['!id'][s['!rel']])s['!drawel']=rels['!id'][s['!rel']];return s;}/* [MS-XLSB] 2.4.331 BrtCsProp */function parse_BrtCsProp(data,length/*:number*/){data.l+=10;var name=parse_XLWideString(data);return {name:name};}/* [MS-XLSB] 2.1.7.7 Chart Sheet */function parse_cs_bin(data,opts,idx/*:number*/,rels,wb/*::, themes, styles*/)/*:Worksheet*/{if(!data)return data;if(!rels)rels={'!id':{}};var s={'!type':"chart",'!drawel':null,'!rel':""};var pass=false;recordhopper(data,function cs_parse(val,R,RT){switch(RT){case 0x0226:/* 'BrtDrawing' */s['!rel']=val;break;case 0x028B:/* 'BrtCsProp' */if(!wb.Sheets[idx])wb.Sheets[idx]={};if(val.name)wb.Sheets[idx].CodeName=val.name;break;case 0x0232:/* 'BrtBkHim' */case 0x028C:/* 'BrtCsPageSetup' */case 0x029D:/* 'BrtCsProtection' */case 0x02A7:/* 'BrtCsProtectionIso' */case 0x0227:/* 'BrtLegacyDrawing' */case 0x0228:/* 'BrtLegacyDrawingHF' */case 0x01DC:/* 'BrtMargins' */case 0x0C00:/* 'BrtUid' */break;case 0x0023:/* 'BrtFRTBegin' */pass=true;break;case 0x0024:/* 'BrtFRTEnd' */pass=false;break;case 0x0025:break;case 0x0026:break;default:if(R.T>0);else if(R.T<0);else if(!pass||opts.WTF)throw new Error("Unexpected record 0x"+RT.toString(16));}},opts);if(rels['!id'][s['!rel']])s['!drawel']=rels['!id'][s['!rel']];return s;}/* 18.2.28 (CT_WorkbookProtection) Defaults */var WBPropsDef=[['allowRefreshQuery',false,"bool"],['autoCompressPictures',true,"bool"],['backupFile',false,"bool"],['checkCompatibility',false,"bool"],['CodeName',''],['date1904',false,"bool"],['defaultThemeVersion',0,"int"],['filterPrivacy',false,"bool"],['hidePivotFieldList',false,"bool"],['promptedSolutions',false,"bool"],['publishItems',false,"bool"],['refreshAllConnections',false,"bool"],['saveExternalLinkValues',true,"bool"],['showBorderUnselectedTables',true,"bool"],['showInkAnnotation',true,"bool"],['showObjects','all'],['showPivotChartFilter',false,"bool"],['updateLinks','userSet']];/* 18.2.30 (CT_BookView) Defaults */var WBViewDef=[['activeTab',0,"int"],['autoFilterDateGrouping',true,"bool"],['firstSheet',0,"int"],['minimized',false,"bool"],['showHorizontalScroll',true,"bool"],['showSheetTabs',true,"bool"],['showVerticalScroll',true,"bool"],['tabRatio',600,"int"],['visibility','visible']//window{Height,Width}, {x,y}Window
	];/* 18.2.19 (CT_Sheet) Defaults */var SheetDef=[//['state', 'visible']
	];/* 18.2.2  (CT_CalcPr) Defaults */var CalcPrDef=[['calcCompleted','true'],['calcMode','auto'],['calcOnSave','true'],['concurrentCalc','true'],['fullCalcOnLoad','false'],['fullPrecision','true'],['iterate','false'],['iterateCount','100'],['iterateDelta','0.001'],['refMode','A1']];/* 18.2.3 (CT_CustomWorkbookView) Defaults */ /*var CustomWBViewDef = [
		['autoUpdate', 'false'],
		['changesSavedWin', 'false'],
		['includeHiddenRowCol', 'true'],
		['includePrintSettings', 'true'],
		['maximized', 'false'],
		['minimized', 'false'],
		['onlySync', 'false'],
		['personalView', 'false'],
		['showComments', 'commIndicator'],
		['showFormulaBar', 'true'],
		['showHorizontalScroll', 'true'],
		['showObjects', 'all'],
		['showSheetTabs', 'true'],
		['showStatusbar', 'true'],
		['showVerticalScroll', 'true'],
		['tabRatio', '600'],
		['xWindow', '0'],
		['yWindow', '0']
	];*/function push_defaults_array(target,defaults){for(var j=0;j!=target.length;++j){var w=target[j];for(var i=0;i!=defaults.length;++i){var z=defaults[i];if(w[z[0]]==null)w[z[0]]=z[1];else switch(z[2]){case"bool":if(typeof w[z[0]]=="string")w[z[0]]=parsexmlbool(w[z[0]]);break;case"int":if(typeof w[z[0]]=="string")w[z[0]]=parseInt(w[z[0]],10);break;}}}}function push_defaults(target,defaults){for(var i=0;i!=defaults.length;++i){var z=defaults[i];if(target[z[0]]==null)target[z[0]]=z[1];else switch(z[2]){case"bool":if(typeof target[z[0]]=="string")target[z[0]]=parsexmlbool(target[z[0]]);break;case"int":if(typeof target[z[0]]=="string")target[z[0]]=parseInt(target[z[0]],10);break;}}}function parse_wb_defaults(wb){push_defaults(wb.WBProps,WBPropsDef);push_defaults(wb.CalcPr,CalcPrDef);push_defaults_array(wb.WBView,WBViewDef);push_defaults_array(wb.Sheets,SheetDef);_ssfopts.date1904=parsexmlbool(wb.WBProps.date1904);}var badchars=/*#__PURE__*/"][*?\/\\".split("");function check_ws_name(n/*:string*/,safe/*:?boolean*/)/*:boolean*/{if(n.length>31){if(safe)return false;throw new Error("Sheet names cannot exceed 31 chars");}var _good=true;badchars.forEach(function(c){if(n.indexOf(c)==-1)return;if(!safe)throw new Error("Sheet name cannot contain : \\ / ? * [ ]");_good=false;});return _good;}/* 18.2 Workbook */var wbnsregex=/<\w+:workbook/;function parse_wb_xml(data,opts)/*:WorkbookFile*/{if(!data)throw new Error("Could not find file");var wb=/*::(*/{AppVersion:{},WBProps:{},WBView:[],Sheets:[],CalcPr:{},Names:[],xmlns:""}/*::)*/;var pass=false,xmlns="xmlns";var dname={},dnstart=0;data.replace(tagregex,function xml_wb(x,idx){var y/*:any*/=parsexmltag(x);switch(strip_ns(y[0])){case'<?xml':break;/* 18.2.27 workbook CT_Workbook 1 */case'<workbook':if(x.match(wbnsregex))xmlns="xmlns"+x.match(/<(\w+):/)[1];wb.xmlns=y[xmlns];break;case'</workbook>':break;/* 18.2.13 fileVersion CT_FileVersion ? */case'<fileVersion':delete y[0];wb.AppVersion=y;break;case'<fileVersion/>':case'</fileVersion>':break;/* 18.2.12 fileSharing CT_FileSharing ? */case'<fileSharing':break;case'<fileSharing/>':break;/* 18.2.28 workbookPr CT_WorkbookPr ? */case'<workbookPr':case'<workbookPr/>':WBPropsDef.forEach(function(w){if(y[w[0]]==null)return;switch(w[2]){case"bool":wb.WBProps[w[0]]=parsexmlbool(y[w[0]]);break;case"int":wb.WBProps[w[0]]=parseInt(y[w[0]],10);break;default:wb.WBProps[w[0]]=y[w[0]];}});if(y.codeName)wb.WBProps.CodeName=utf8read(y.codeName);break;case'</workbookPr>':break;/* 18.2.29 workbookProtection CT_WorkbookProtection ? */case'<workbookProtection':break;case'<workbookProtection/>':break;/* 18.2.1  bookViews CT_BookViews ? */case'<bookViews':case'<bookViews>':case'</bookViews>':break;/* 18.2.30   workbookView CT_BookView + */case'<workbookView':case'<workbookView/>':delete y[0];wb.WBView.push(y);break;case'</workbookView>':break;/* 18.2.20 sheets CT_Sheets 1 */case'<sheets':case'<sheets>':case'</sheets>':break;// aggregate sheet
	/* 18.2.19   sheet CT_Sheet + */case'<sheet':switch(y.state){case"hidden":y.Hidden=1;break;case"veryHidden":y.Hidden=2;break;default:y.Hidden=0;}delete y.state;y.name=unescapexml(utf8read(y.name));delete y[0];wb.Sheets.push(y);break;case'</sheet>':break;/* 18.2.15 functionGroups CT_FunctionGroups ? */case'<functionGroups':case'<functionGroups/>':break;/* 18.2.14   functionGroup CT_FunctionGroup + */case'<functionGroup':break;/* 18.2.9  externalReferences CT_ExternalReferences ? */case'<externalReferences':case'</externalReferences>':case'<externalReferences>':break;/* 18.2.8    externalReference CT_ExternalReference + */case'<externalReference':break;/* 18.2.6  definedNames CT_DefinedNames ? */case'<definedNames/>':break;case'<definedNames>':case'<definedNames':pass=true;break;case'</definedNames>':pass=false;break;/* 18.2.5    definedName CT_DefinedName + */case'<definedName':{dname={};dname.Name=utf8read(y.name);if(y.comment)dname.Comment=y.comment;if(y.localSheetId)dname.Sheet=+y.localSheetId;if(parsexmlbool(y.hidden||"0"))dname.Hidden=true;dnstart=idx+x.length;}break;case'</definedName>':{dname.Ref=unescapexml(utf8read(data.slice(dnstart,idx)));wb.Names.push(dname);}break;case'<definedName/>':break;/* 18.2.2  calcPr CT_CalcPr ? */case'<calcPr':delete y[0];wb.CalcPr=y;break;case'<calcPr/>':delete y[0];wb.CalcPr=y;break;case'</calcPr>':break;/* 18.2.16 oleSize CT_OleSize ? (ref required) */case'<oleSize':break;/* 18.2.4  customWorkbookViews CT_CustomWorkbookViews ? */case'<customWorkbookViews>':case'</customWorkbookViews>':case'<customWorkbookViews':break;/* 18.2.3  customWorkbookView CT_CustomWorkbookView + */case'<customWorkbookView':case'</customWorkbookView>':break;/* 18.2.18 pivotCaches CT_PivotCaches ? */case'<pivotCaches>':case'</pivotCaches>':case'<pivotCaches':break;/* 18.2.17 pivotCache CT_PivotCache ? */case'<pivotCache':break;/* 18.2.21 smartTagPr CT_SmartTagPr ? */case'<smartTagPr':case'<smartTagPr/>':break;/* 18.2.23 smartTagTypes CT_SmartTagTypes ? */case'<smartTagTypes':case'<smartTagTypes>':case'</smartTagTypes>':break;/* 18.2.22 smartTagType CT_SmartTagType ? */case'<smartTagType':break;/* 18.2.24 webPublishing CT_WebPublishing ? */case'<webPublishing':case'<webPublishing/>':break;/* 18.2.11 fileRecoveryPr CT_FileRecoveryPr ? */case'<fileRecoveryPr':case'<fileRecoveryPr/>':break;/* 18.2.26 webPublishObjects CT_WebPublishObjects ? */case'<webPublishObjects>':case'<webPublishObjects':case'</webPublishObjects>':break;/* 18.2.25 webPublishObject CT_WebPublishObject ? */case'<webPublishObject':break;/* 18.2.10 extLst CT_ExtensionList ? */case'<extLst':case'<extLst>':case'</extLst>':case'<extLst/>':break;/* 18.2.7  ext CT_Extension + */case'<ext':pass=true;break;//TODO: check with versions of excel
	case'</ext>':pass=false;break;/* Others */case'<ArchID':break;case'<AlternateContent':case'<AlternateContent>':pass=true;break;case'</AlternateContent>':pass=false;break;/* TODO */case'<revisionPtr':break;default:if(!pass&&opts.WTF)throw new Error('unrecognized '+y[0]+' in workbook');}return x;});if(XMLNS_main.indexOf(wb.xmlns)===-1)throw new Error("Unknown Namespace: "+wb.xmlns);parse_wb_defaults(wb);return wb;}/* [MS-XLSB] 2.4.304 BrtBundleSh */function parse_BrtBundleSh(data,length/*:number*/){var z={};z.Hidden=data.read_shift(4);//hsState ST_SheetState
	z.iTabID=data.read_shift(4);z.strRelID=parse_RelID(data);z.name=parse_XLWideString(data);return z;}/* [MS-XLSB] 2.4.815 BrtWbProp */function parse_BrtWbProp(data,length)/*:WBProps*/{var o/*:WBProps*/={}/*:any*/;var flags=data.read_shift(4);o.defaultThemeVersion=data.read_shift(4);var strName=length>8?parse_XLWideString(data):"";if(strName.length>0)o.CodeName=strName;o.autoCompressPictures=!!(flags&0x10000);o.backupFile=!!(flags&0x40);o.checkCompatibility=!!(flags&0x1000);o.date1904=!!(flags&0x01);o.filterPrivacy=!!(flags&0x08);o.hidePivotFieldList=!!(flags&0x400);o.promptedSolutions=!!(flags&0x10);o.publishItems=!!(flags&0x800);o.refreshAllConnections=!!(flags&0x40000);o.saveExternalLinkValues=!!(flags&0x80);o.showBorderUnselectedTables=!!(flags&0x04);o.showInkAnnotation=!!(flags&0x20);o.showObjects=["all","placeholders","none"][flags>>13&0x03];o.showPivotChartFilter=!!(flags&0x8000);o.updateLinks=["userSet","never","always"][flags>>8&0x03];return o;}function parse_BrtFRTArchID$(data,length){var o={};data.read_shift(4);o.ArchID=data.read_shift(4);data.l+=length-8;return o;}/* [MS-XLSB] 2.4.687 BrtName */function parse_BrtName(data,length,opts){var end=data.l+length;data.l+=4;//var flags = data.read_shift(4);
	data.l+=1;//var chKey = data.read_shift(1);
	var itab=data.read_shift(4);var name=parse_XLNameWideString(data);var formula=parse_XLSBNameParsedFormula(data,0,opts);var comment=parse_XLNullableWideString(data);//if(0 /* fProc */) {
	// unusedstring1: XLNullableWideString
	// description: XLNullableWideString
	// helpTopic: XLNullableWideString
	// unusedstring2: XLNullableWideString
	//}
	data.l=end;var out={Name:name,Ptg:formula}/*:any*/;if(itab<0xFFFFFFF)out.Sheet=itab;if(comment)out.Comment=comment;return out;}/* [MS-XLSB] 2.1.7.61 Workbook */function parse_wb_bin(data,opts)/*:WorkbookFile*/{var wb={AppVersion:{},WBProps:{},WBView:[],Sheets:[],CalcPr:{},xmlns:""};var state/*:Array<string>*/=[];var pass=false;if(!opts)opts={};opts.biff=12;var Names=[];var supbooks=[[]]/*:any*/;supbooks.SheetNames=[];supbooks.XTI=[];XLSBRecordEnum[0x0010]={n:"BrtFRTArchID$",f:parse_BrtFRTArchID$};recordhopper(data,function hopper_wb(val,R,RT){switch(RT){case 0x009C:/* 'BrtBundleSh' */supbooks.SheetNames.push(val.name);wb.Sheets.push(val);break;case 0x0099:/* 'BrtWbProp' */wb.WBProps=val;break;case 0x0027:/* 'BrtName' */if(val.Sheet!=null)opts.SID=val.Sheet;val.Ref=stringify_formula(val.Ptg,null,null,supbooks,opts);delete opts.SID;delete val.Ptg;Names.push(val);break;case 0x040C:/* 'BrtNameExt' */break;case 0x0165:/* 'BrtSupSelf' */case 0x0166:/* 'BrtSupSame' */case 0x0163:/* 'BrtSupBookSrc' */case 0x029B:/* 'BrtSupAddin' */if(!supbooks[0].length)supbooks[0]=[RT,val];else supbooks.push([RT,val]);supbooks[supbooks.length-1].XTI=[];break;case 0x016A:/* 'BrtExternSheet' */if(supbooks.length===0){supbooks[0]=[];supbooks[0].XTI=[];}supbooks[supbooks.length-1].XTI=supbooks[supbooks.length-1].XTI.concat(val);supbooks.XTI=supbooks.XTI.concat(val);break;case 0x0169:/* 'BrtPlaceholderName' */break;case 0x0817:/* 'BrtAbsPath15' */case 0x009E:/* 'BrtBookView' */case 0x008F:/* 'BrtBeginBundleShs' */case 0x0298:/* 'BrtBeginFnGroup' */case 0x0161:/* 'BrtBeginExternals' */break;/* case 'BrtModelTimeGroupingCalcCol' */case 0x0C00:/* 'BrtUid' */case 0x0C01:/* 'BrtRevisionPtr' */case 0x0216:/* 'BrtBookProtection' */case 0x02A5:/* 'BrtBookProtectionIso' */case 0x009D:/* 'BrtCalcProp' */case 0x0262:/* 'BrtCrashRecErr' */case 0x0802:/* 'BrtDecoupledPivotCacheID' */case 0x009B:/* 'BrtFileRecover' */case 0x0224:/* 'BrtFileSharing' */case 0x02A4:/* 'BrtFileSharingIso' */case 0x0080:/* 'BrtFileVersion' */case 0x0299:/* 'BrtFnGroup' */case 0x0850:/* 'BrtModelRelationship' */case 0x084D:/* 'BrtModelTable' */case 0x0225:/* 'BrtOleSize' */case 0x0805:/* 'BrtPivotTableRef' */case 0x0254:/* 'BrtSmartTagType' */case 0x081C:/* 'BrtTableSlicerCacheID' */case 0x081B:/* 'BrtTableSlicerCacheIDs' */case 0x0822:/* 'BrtTimelineCachePivotCacheID' */case 0x018D:/* 'BrtUserBookView' */case 0x009A:/* 'BrtWbFactoid' */case 0x045D:/* 'BrtWbProp14' */case 0x0229:/* 'BrtWebOpt' */case 0x082B:/* 'BrtWorkBookPr15' */break;case 0x0023:/* 'BrtFRTBegin' */state.push(RT);pass=true;break;case 0x0024:/* 'BrtFRTEnd' */state.pop();pass=false;break;case 0x0025:/* 'BrtACBegin' */state.push(RT);pass=true;break;case 0x0026:/* 'BrtACEnd' */state.pop();pass=false;break;case 0x0010:/* 'BrtFRTArchID$' */break;default:if(R.T);else if(!pass||opts.WTF&&state[state.length-1]!=0x0025/* BrtACBegin */&&state[state.length-1]!=0x0023/* BrtFRTBegin */)throw new Error("Unexpected record 0x"+RT.toString(16));}},opts);parse_wb_defaults(wb);// $FlowIgnore
	wb.Names=Names;wb/*:any*/.supbooks=supbooks;return wb;}function parse_wb(data,name/*:string*/,opts)/*:WorkbookFile*/{if(name.slice(-4)===".bin")return parse_wb_bin(data/*:any*/,opts);return parse_wb_xml(data/*:any*/,opts);}function parse_ws(data,name/*:string*/,idx/*:number*/,opts,rels,wb,themes,styles)/*:Worksheet*/{if(name.slice(-4)===".bin")return parse_ws_bin(data/*:any*/,opts,idx,rels,wb,themes,styles);return parse_ws_xml(data/*:any*/,opts,idx,rels,wb,themes,styles);}function parse_cs(data,name/*:string*/,idx/*:number*/,opts,rels,wb,themes,styles)/*:Worksheet*/{if(name.slice(-4)===".bin")return parse_cs_bin(data/*:any*/,opts,idx,rels,wb);return parse_cs_xml(data/*:any*/,opts,idx,rels,wb);}function parse_ms(data,name/*:string*/,idx/*:number*/,opts,rels,wb,themes,styles)/*:Worksheet*/{if(name.slice(-4)===".bin")return parse_ms_bin();return parse_ms_xml();}function parse_ds(data,name/*:string*/,idx/*:number*/,opts,rels,wb,themes,styles)/*:Worksheet*/{if(name.slice(-4)===".bin")return parse_ds_bin();return parse_ds_xml();}function parse_sty(data,name/*:string*/,themes,opts){if(name.slice(-4)===".bin")return parse_sty_bin(data/*:any*/,themes,opts);return parse_sty_xml(data/*:any*/,themes,opts);}function parse_theme(data/*:string*/,name/*:string*/,opts){return parse_theme_xml(data,opts);}function parse_sst(data,name/*:string*/,opts)/*:SST*/{if(name.slice(-4)===".bin")return parse_sst_bin(data/*:any*/,opts);return parse_sst_xml(data/*:any*/,opts);}function parse_cmnt(data,name/*:string*/,opts)/*:Array<RawComment>*/{if(name.slice(-4)===".bin")return parse_comments_bin(data/*:any*/,opts);return parse_comments_xml(data/*:any*/,opts);}function parse_cc(data,name/*:string*/,opts){if(name.slice(-4)===".bin")return parse_cc_bin(data/*:any*/);return parse_cc_xml(data/*:any*/);}function parse_xlink(data,rel,name/*:string*/,opts){if(name.slice(-4)===".bin")return parse_xlink_bin(data/*:any*/,rel,name,opts);return parse_xlink_xml();}function parse_xlmeta(data,name/*:string*/,opts){if(name.slice(-4)===".bin")return parse_xlmeta_bin(data/*:any*/,name,opts);return parse_xlmeta_xml(data/*:any*/,name,opts);}var attregexg2=/([\w:]+)=((?:")([^"]*)(?:")|(?:')([^']*)(?:'))/g;var attregex2=/([\w:]+)=((?:")(?:[^"]*)(?:")|(?:')(?:[^']*)(?:'))/;function xlml_parsexmltag(tag/*:string*/,skip_root/*:?boolean*/){var words=tag.split(/\s+/);var z/*:any*/=[]/*:any*/;if(!skip_root)z[0]=words[0];if(words.length===1)return z;var m=tag.match(attregexg2),y,j,w,i;if(m)for(i=0;i!=m.length;++i){y=m[i].match(attregex2);/*:: if(!y || !y[2]) continue; */if((j=y[1].indexOf(":"))===-1)z[y[1]]=y[2].slice(1,y[2].length-1);else {if(y[1].slice(0,6)==="xmlns:")w="xmlns"+y[1].slice(6);else w=y[1].slice(j+1);z[w]=y[2].slice(1,y[2].length-1);}}return z;}function xlml_parsexmltagobj(tag/*:string*/){var words=tag.split(/\s+/);var z={};if(words.length===1)return z;var m=tag.match(attregexg2),y,j,w,i;if(m)for(i=0;i!=m.length;++i){y=m[i].match(attregex2);/*:: if(!y || !y[2]) continue; */if((j=y[1].indexOf(":"))===-1)z[y[1]]=y[2].slice(1,y[2].length-1);else {if(y[1].slice(0,6)==="xmlns:")w="xmlns"+y[1].slice(6);else w=y[1].slice(j+1);z[w]=y[2].slice(1,y[2].length-1);}}return z;}// ----
	/* map from xlml named formats to SSF TODO: localize */var XLMLFormatMap/*: {[string]:string}*/;function xlml_format(format,value)/*:string*/{var fmt=XLMLFormatMap[format]||unescapexml(format);if(fmt==="General")return SSF_general(value);return SSF_format(fmt,value);}function xlml_set_custprop(Custprops,key,cp,val/*:string*/){var oval/*:any*/=val;switch((cp[0].match(/dt:dt="([\w.]+)"/)||["",""])[1]){case"boolean":oval=parsexmlbool(val);break;case"i2":case"int":oval=parseInt(val,10);break;case"r4":case"float":oval=parseFloat(val);break;case"date":case"dateTime.tz":oval=parseDate(val);break;case"i8":case"string":case"fixed":case"uuid":case"bin.base64":break;default:throw new Error("bad custprop:"+cp[0]);}Custprops[unescapexml(key)]=oval;}function safe_format_xlml(cell/*:Cell*/,nf,o){if(cell.t==='z')return;if(!o||o.cellText!==false)try{if(cell.t==='e'){cell.w=cell.w||BErr[cell.v];}else if(nf==="General"){if(cell.t==='n'){if((cell.v|0)===cell.v)cell.w=cell.v.toString(10);else cell.w=SSF_general_num(cell.v);}else cell.w=SSF_general(cell.v);}else cell.w=xlml_format(nf||"General",cell.v);}catch(e){if(o.WTF)throw e;}try{var z=XLMLFormatMap[nf]||nf||"General";if(o.cellNF)cell.z=z;if(o.cellDates&&cell.t=='n'&&fmt_is_date(z)){var _d=SSF_parse_date_code(cell.v);if(_d){cell.t='d';cell.v=new Date(_d.y,_d.m-1,_d.d,_d.H,_d.M,_d.S,_d.u);}}}catch(e){if(o.WTF)throw e;}}function process_style_xlml(styles,stag,opts){if(opts.cellStyles){if(stag.Interior){var I=stag.Interior;if(I.Pattern)I.patternType=XLMLPatternTypeMap[I.Pattern]||I.Pattern;}}styles[stag.ID]=stag;}/* TODO: there must exist some form of OSP-blessed spec */function parse_xlml_data(xml,ss,data,cell/*:any*/,base,styles,csty,row,arrayf,o){var nf="General",sid=cell.StyleID,S={};o=o||{};var interiors=[];var i=0;if(sid===undefined&&row)sid=row.StyleID;if(sid===undefined&&csty)sid=csty.StyleID;while(styles[sid]!==undefined){if(styles[sid].nf)nf=styles[sid].nf;if(styles[sid].Interior)interiors.push(styles[sid].Interior);if(!styles[sid].Parent)break;sid=styles[sid].Parent;}switch(data.Type){case'Boolean':cell.t='b';cell.v=parsexmlbool(xml);break;case'String':cell.t='s';cell.r=xlml_fixstr(unescapexml(xml));cell.v=xml.indexOf("<")>-1?unescapexml(ss||xml).replace(/<.*?>/g,""):cell.r;// todo: BR etc
	break;case'DateTime':if(xml.slice(-1)!="Z")xml+="Z";cell.v=(parseDate(xml)-new Date(Date.UTC(1899,11,30)))/(24*60*60*1000);if(cell.v!==cell.v)cell.v=unescapexml(xml);else if(cell.v<60)cell.v=cell.v-1;if(!nf||nf=="General")nf="yyyy-mm-dd";/* falls through */case'Number':if(cell.v===undefined)cell.v=+xml;if(!cell.t)cell.t='n';break;case'Error':cell.t='e';cell.v=RBErr[xml];if(o.cellText!==false)cell.w=xml;break;default:if(xml==""&&ss==""){cell.t='z';}else {cell.t='s';cell.v=xlml_fixstr(ss||xml);}break;}safe_format_xlml(cell,nf,o);if(o.cellFormula!==false){if(cell.Formula){var fstr=unescapexml(cell.Formula);/* strictly speaking, the leading = is required but some writers omit */if(fstr.charCodeAt(0)==61/* = */)fstr=fstr.slice(1);cell.f=rc_to_a1(fstr,base);delete cell.Formula;if(cell.ArrayRange=="RC")cell.F=rc_to_a1("RC:RC",base);else if(cell.ArrayRange){cell.F=rc_to_a1(cell.ArrayRange,base);arrayf.push([safe_decode_range(cell.F),cell.F]);}}else {for(i=0;i<arrayf.length;++i)if(base.r>=arrayf[i][0].s.r&&base.r<=arrayf[i][0].e.r)if(base.c>=arrayf[i][0].s.c&&base.c<=arrayf[i][0].e.c)cell.F=arrayf[i][1];}}if(o.cellStyles){interiors.forEach(function(x){if(!S.patternType&&x.patternType)S.patternType=x.patternType;});cell.s=S;}if(cell.StyleID!==undefined)cell.ixfe=cell.StyleID;}function xlml_clean_comment(comment/*:any*/){comment.t=comment.v||"";comment.t=comment.t.replace(/\r\n/g,"\n").replace(/\r/g,"\n");comment.v=comment.w=comment.ixfe=undefined;}/* TODO: Everything */function parse_xlml_xml(d,_opts)/*:Workbook*/{var opts=_opts||{};make_ssf();var str=debom(xlml_normalize(d));if(opts.type=='binary'||opts.type=='array'||opts.type=='base64'){str=utf8read(str);}var opening=str.slice(0,1024).toLowerCase(),ishtml=false;opening=opening.replace(/".*?"/g,"");if((opening.indexOf(">")&1023)>Math.min(opening.indexOf(",")&1023,opening.indexOf(";")&1023)){var _o=dup(opts);_o.type="string";return PRN.to_workbook(str,_o);}if(opening.indexOf("<?xml")==-1)["html","table","head","meta","script","style","div"].forEach(function(tag){if(opening.indexOf("<"+tag)>=0)ishtml=true;});if(ishtml)return html_to_workbook(str,opts);XLMLFormatMap={"General Number":"General","General Date":table_fmt[22],"Long Date":"dddd, mmmm dd, yyyy","Medium Date":table_fmt[15],"Short Date":table_fmt[14],"Long Time":table_fmt[19],"Medium Time":table_fmt[18],"Short Time":table_fmt[20],"Currency":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"Fixed":table_fmt[2],"Standard":table_fmt[4],"Percent":table_fmt[10],"Scientific":table_fmt[11],"Yes/No":'"Yes";"Yes";"No";@',"True/False":'"True";"True";"False";@',"On/Off":'"Yes";"Yes";"No";@'}/*:any*/;var Rn;var state=[],tmp;var sheets={},sheetnames/*:Array<string>*/=[],cursheet/*:Worksheet*/=opts.dense?[]:{},sheetname="";var cell={}/*:any*/,row={};// eslint-disable-line no-unused-vars
	var dtag=xlml_parsexmltag('<Data ss:Type="String">'),didx=0;var c=0,r=0;var refguess/*:Range*/={s:{r:2000000,c:2000000},e:{r:0,c:0}};var styles={},stag={};var ss="",fidx=0;var merges/*:Array<Range>*/=[];var Props={},Custprops={},pidx=0,cp=[];var comments/*:Array<Comment>*/=[],comment/*:Comment*/={}/*:any*/;var cstys=[],csty,seencol=false;var arrayf/*:Array<[Range, string]>*/=[];var rowinfo/*:Array<RowInfo>*/=[],rowobj={},cc=0,rr=0;var Workbook/*:WBWBProps*/={Sheets:[],WBProps:{date1904:false}}/*:any*/,wsprops={};xlmlregex.lastIndex=0;str=str.replace(/<!--([\s\S]*?)-->/mg,"");var raw_Rn3="";while(Rn=xlmlregex.exec(str))switch(Rn[3]=(raw_Rn3=Rn[3]).toLowerCase()){case'data'/*case 'Data'*/:if(raw_Rn3=="data"){if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else if(Rn[0].charAt(Rn[0].length-2)!=='/')state.push([Rn[3],true]);break;}if(state[state.length-1][1])break;if(Rn[1]==='/')parse_xlml_data(str.slice(didx,Rn.index),ss,dtag,state[state.length-1][0]==/*"Comment"*/"comment"?comment:cell,{c:c,r:r},styles,cstys[c],row,arrayf,opts);else {ss="";dtag=xlml_parsexmltag(Rn[0]);didx=Rn.index+Rn[0].length;}break;case'cell'/*case 'Cell'*/:if(Rn[1]==='/'){if(comments.length>0)cell.c=comments;if((!opts.sheetRows||opts.sheetRows>r)&&cell.v!==undefined){if(opts.dense){if(!cursheet[r])cursheet[r]=[];cursheet[r][c]=cell;}else cursheet[encode_col(c)+encode_row(r)]=cell;}if(cell.HRef){cell.l={Target:unescapexml(cell.HRef)}/*:any*/;if(cell.HRefScreenTip)cell.l.Tooltip=cell.HRefScreenTip;delete cell.HRef;delete cell.HRefScreenTip;}if(cell.MergeAcross||cell.MergeDown){cc=c+(parseInt(cell.MergeAcross,10)|0);rr=r+(parseInt(cell.MergeDown,10)|0);merges.push({s:{c:c,r:r},e:{c:cc,r:rr}});}if(!opts.sheetStubs){if(cell.MergeAcross)c=cc+1;else ++c;}else if(cell.MergeAcross||cell.MergeDown){/*:: if(!cc) cc = 0; if(!rr) rr = 0; */for(var cma=c;cma<=cc;++cma){for(var cmd=r;cmd<=rr;++cmd){if(cma>c||cmd>r){if(opts.dense){if(!cursheet[cmd])cursheet[cmd]=[];cursheet[cmd][cma]={t:'z'};}else cursheet[encode_col(cma)+encode_row(cmd)]={t:'z'};}}}c=cc+1;}else ++c;}else {cell=xlml_parsexmltagobj(Rn[0]);if(cell.Index)c=+cell.Index-1;if(c<refguess.s.c)refguess.s.c=c;if(c>refguess.e.c)refguess.e.c=c;if(Rn[0].slice(-2)==="/>")++c;comments=[];}break;case'row'/*case 'Row'*/:if(Rn[1]==='/'||Rn[0].slice(-2)==="/>"){if(r<refguess.s.r)refguess.s.r=r;if(r>refguess.e.r)refguess.e.r=r;if(Rn[0].slice(-2)==="/>"){row=xlml_parsexmltag(Rn[0]);if(row.Index)r=+row.Index-1;}c=0;++r;}else {row=xlml_parsexmltag(Rn[0]);if(row.Index)r=+row.Index-1;rowobj={};if(row.AutoFitHeight=="0"||row.Height){rowobj.hpx=parseInt(row.Height,10);rowobj.hpt=px2pt(rowobj.hpx);rowinfo[r]=rowobj;}if(row.Hidden=="1"){rowobj.hidden=true;rowinfo[r]=rowobj;}}break;case'worksheet'/*case 'Worksheet'*/:/* TODO: read range from FullRows/FullColumns */if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));sheetnames.push(sheetname);if(refguess.s.r<=refguess.e.r&&refguess.s.c<=refguess.e.c){cursheet["!ref"]=encode_range(refguess);if(opts.sheetRows&&opts.sheetRows<=refguess.e.r){cursheet["!fullref"]=cursheet["!ref"];refguess.e.r=opts.sheetRows-1;cursheet["!ref"]=encode_range(refguess);}}if(merges.length)cursheet["!merges"]=merges;if(cstys.length>0)cursheet["!cols"]=cstys;if(rowinfo.length>0)cursheet["!rows"]=rowinfo;sheets[sheetname]=cursheet;}else {refguess={s:{r:2000000,c:2000000},e:{r:0,c:0}};r=c=0;state.push([Rn[3],false]);tmp=xlml_parsexmltag(Rn[0]);sheetname=unescapexml(tmp.Name);cursheet=opts.dense?[]:{};merges=[];arrayf=[];rowinfo=[];wsprops={name:sheetname,Hidden:0};Workbook.Sheets.push(wsprops);}break;case'table'/*case 'Table'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else if(Rn[0].slice(-2)=="/>")break;else {state.push([Rn[3],false]);cstys=[];seencol=false;}break;case'style'/*case 'Style'*/:if(Rn[1]==='/')process_style_xlml(styles,stag,opts);else stag=xlml_parsexmltag(Rn[0]);break;case'numberformat'/*case 'NumberFormat'*/:stag.nf=unescapexml(xlml_parsexmltag(Rn[0]).Format||"General");if(XLMLFormatMap[stag.nf])stag.nf=XLMLFormatMap[stag.nf];for(var ssfidx=0;ssfidx!=0x188;++ssfidx)if(table_fmt[ssfidx]==stag.nf)break;if(ssfidx==0x188)for(ssfidx=0x39;ssfidx!=0x188;++ssfidx)if(table_fmt[ssfidx]==null){SSF_load(stag.nf,ssfidx);break;}break;case'column'/*case 'Column'*/:if(state[state.length-1][0]!==/*'Table'*/'table')break;csty=xlml_parsexmltag(Rn[0]);if(csty.Hidden){csty.hidden=true;delete csty.Hidden;}if(csty.Width)csty.wpx=parseInt(csty.Width,10);if(!seencol&&csty.wpx>10){seencol=true;MDW=DEF_MDW;//find_mdw_wpx(csty.wpx);
	for(var _col=0;_col<cstys.length;++_col)if(cstys[_col])process_col(cstys[_col]);}if(seencol)process_col(csty);cstys[csty.Index-1||cstys.length]=csty;for(var i=0;i<+csty.Span;++i)cstys[cstys.length]=dup(csty);break;case'namedrange'/*case 'NamedRange'*/:if(Rn[1]==='/')break;if(!Workbook.Names)Workbook.Names=[];var _NamedRange=parsexmltag(Rn[0]);var _DefinedName/*:DefinedName*/={Name:_NamedRange.Name,Ref:rc_to_a1(_NamedRange.RefersTo.slice(1),{r:0,c:0})}/*:any*/;if(Workbook.Sheets.length>0)_DefinedName.Sheet=Workbook.Sheets.length-1;/*:: if(Workbook.Names) */Workbook.Names.push(_DefinedName);break;case'namedcell'/*case 'NamedCell'*/:break;case'b'/*case 'B'*/:break;case'i'/*case 'I'*/:break;case'u'/*case 'U'*/:break;case's'/*case 'S'*/:break;case'em'/*case 'EM'*/:break;case'h2'/*case 'H2'*/:break;case'h3'/*case 'H3'*/:break;case'sub'/*case 'Sub'*/:break;case'sup'/*case 'Sup'*/:break;case'span'/*case 'Span'*/:break;case'alignment'/*case 'Alignment'*/:break;case'borders'/*case 'Borders'*/:break;case'border'/*case 'Border'*/:break;case'font'/*case 'Font'*/:if(Rn[0].slice(-2)==="/>")break;else if(Rn[1]==="/")ss+=str.slice(fidx,Rn.index);else fidx=Rn.index+Rn[0].length;break;case'interior'/*case 'Interior'*/:if(!opts.cellStyles)break;stag.Interior=xlml_parsexmltag(Rn[0]);break;case'protection'/*case 'Protection'*/:break;case'author'/*case 'Author'*/:case'title'/*case 'Title'*/:case'description'/*case 'Description'*/:case'created'/*case 'Created'*/:case'keywords'/*case 'Keywords'*/:case'subject'/*case 'Subject'*/:case'category'/*case 'Category'*/:case'company'/*case 'Company'*/:case'lastauthor'/*case 'LastAuthor'*/:case'lastsaved'/*case 'LastSaved'*/:case'lastprinted'/*case 'LastPrinted'*/:case'version'/*case 'Version'*/:case'revision'/*case 'Revision'*/:case'totaltime'/*case 'TotalTime'*/:case'hyperlinkbase'/*case 'HyperlinkBase'*/:case'manager'/*case 'Manager'*/:case'contentstatus'/*case 'ContentStatus'*/:case'identifier'/*case 'Identifier'*/:case'language'/*case 'Language'*/:case'appname'/*case 'AppName'*/:if(Rn[0].slice(-2)==="/>")break;else if(Rn[1]==="/")xlml_set_prop(Props,raw_Rn3,str.slice(pidx,Rn.index));else pidx=Rn.index+Rn[0].length;break;case'paragraphs'/*case 'Paragraphs'*/:break;case'styles'/*case 'Styles'*/:case'workbook'/*case 'Workbook'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else state.push([Rn[3],false]);break;case'comment'/*case 'Comment'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));xlml_clean_comment(comment);comments.push(comment);}else {state.push([Rn[3],false]);tmp=xlml_parsexmltag(Rn[0]);comment={a:tmp.Author}/*:any*/;}break;case'autofilter'/*case 'AutoFilter'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else if(Rn[0].charAt(Rn[0].length-2)!=='/'){var AutoFilter=xlml_parsexmltag(Rn[0]);cursheet['!autofilter']={ref:rc_to_a1(AutoFilter.Range).replace(/\$/g,"")};state.push([Rn[3],true]);}break;case'name'/*case 'Name'*/:break;case'datavalidation'/*case 'DataValidation'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else {if(Rn[0].charAt(Rn[0].length-2)!=='/')state.push([Rn[3],true]);}break;case'pixelsperinch'/*case 'PixelsPerInch'*/:break;case'componentoptions'/*case 'ComponentOptions'*/:case'documentproperties'/*case 'DocumentProperties'*/:case'customdocumentproperties'/*case 'CustomDocumentProperties'*/:case'officedocumentsettings'/*case 'OfficeDocumentSettings'*/:case'pivottable'/*case 'PivotTable'*/:case'pivotcache'/*case 'PivotCache'*/:case'names'/*case 'Names'*/:case'mapinfo'/*case 'MapInfo'*/:case'pagebreaks'/*case 'PageBreaks'*/:case'querytable'/*case 'QueryTable'*/:case'sorting'/*case 'Sorting'*/:case'schema'/*case 'Schema'*/://case 'data' /*case 'data'*/:
	case'conditionalformatting'/*case 'ConditionalFormatting'*/:case'smarttagtype'/*case 'SmartTagType'*/:case'smarttags'/*case 'SmartTags'*/:case'excelworkbook'/*case 'ExcelWorkbook'*/:case'workbookoptions'/*case 'WorkbookOptions'*/:case'worksheetoptions'/*case 'WorksheetOptions'*/:if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw new Error("Bad state: "+tmp.join("|"));}else if(Rn[0].charAt(Rn[0].length-2)!=='/')state.push([Rn[3],true]);break;case'null'/*case 'Null'*/:break;default:/* FODS file root is <office:document> */if(state.length==0&&Rn[3]=="document")return parse_fods(str,opts);/* UOS file root is <uof:UOF> */if(state.length==0&&Rn[3]=="uof"/*"UOF"*/)return parse_fods(str,opts);var seen=true;switch(state[state.length-1][0]){/* OfficeDocumentSettings */case'officedocumentsettings'/*case 'OfficeDocumentSettings'*/:switch(Rn[3]){case'allowpng'/*case 'AllowPNG'*/:break;case'removepersonalinformation'/*case 'RemovePersonalInformation'*/:break;case'downloadcomponents'/*case 'DownloadComponents'*/:break;case'locationofcomponents'/*case 'LocationOfComponents'*/:break;case'colors'/*case 'Colors'*/:break;case'color'/*case 'Color'*/:break;case'index'/*case 'Index'*/:break;case'rgb'/*case 'RGB'*/:break;case'targetscreensize'/*case 'TargetScreenSize'*/:break;case'readonlyrecommended'/*case 'ReadOnlyRecommended'*/:break;default:seen=false;}break;/* ComponentOptions */case'componentoptions'/*case 'ComponentOptions'*/:switch(Rn[3]){case'toolbar'/*case 'Toolbar'*/:break;case'hideofficelogo'/*case 'HideOfficeLogo'*/:break;case'spreadsheetautofit'/*case 'SpreadsheetAutoFit'*/:break;case'label'/*case 'Label'*/:break;case'caption'/*case 'Caption'*/:break;case'maxheight'/*case 'MaxHeight'*/:break;case'maxwidth'/*case 'MaxWidth'*/:break;case'nextsheetnumber'/*case 'NextSheetNumber'*/:break;default:seen=false;}break;/* ExcelWorkbook */case'excelworkbook'/*case 'ExcelWorkbook'*/:switch(Rn[3]){case'date1904'/*case 'Date1904'*/:/*:: if(!Workbook.WBProps) Workbook.WBProps = {}; */Workbook.WBProps.date1904=true;break;case'windowheight'/*case 'WindowHeight'*/:break;case'windowwidth'/*case 'WindowWidth'*/:break;case'windowtopx'/*case 'WindowTopX'*/:break;case'windowtopy'/*case 'WindowTopY'*/:break;case'tabratio'/*case 'TabRatio'*/:break;case'protectstructure'/*case 'ProtectStructure'*/:break;case'protectwindow'/*case 'ProtectWindow'*/:break;case'protectwindows'/*case 'ProtectWindows'*/:break;case'activesheet'/*case 'ActiveSheet'*/:break;case'displayinknotes'/*case 'DisplayInkNotes'*/:break;case'firstvisiblesheet'/*case 'FirstVisibleSheet'*/:break;case'supbook'/*case 'SupBook'*/:break;case'sheetname'/*case 'SheetName'*/:break;case'sheetindex'/*case 'SheetIndex'*/:break;case'sheetindexfirst'/*case 'SheetIndexFirst'*/:break;case'sheetindexlast'/*case 'SheetIndexLast'*/:break;case'dll'/*case 'Dll'*/:break;case'acceptlabelsinformulas'/*case 'AcceptLabelsInFormulas'*/:break;case'donotsavelinkvalues'/*case 'DoNotSaveLinkValues'*/:break;case'iteration'/*case 'Iteration'*/:break;case'maxiterations'/*case 'MaxIterations'*/:break;case'maxchange'/*case 'MaxChange'*/:break;case'path'/*case 'Path'*/:break;case'xct'/*case 'Xct'*/:break;case'count'/*case 'Count'*/:break;case'selectedsheets'/*case 'SelectedSheets'*/:break;case'calculation'/*case 'Calculation'*/:break;case'uncalced'/*case 'Uncalced'*/:break;case'startupprompt'/*case 'StartupPrompt'*/:break;case'crn'/*case 'Crn'*/:break;case'externname'/*case 'ExternName'*/:break;case'formula'/*case 'Formula'*/:break;case'colfirst'/*case 'ColFirst'*/:break;case'collast'/*case 'ColLast'*/:break;case'wantadvise'/*case 'WantAdvise'*/:break;case'boolean'/*case 'Boolean'*/:break;case'error'/*case 'Error'*/:break;case'text'/*case 'Text'*/:break;case'ole'/*case 'OLE'*/:break;case'noautorecover'/*case 'NoAutoRecover'*/:break;case'publishobjects'/*case 'PublishObjects'*/:break;case'donotcalculatebeforesave'/*case 'DoNotCalculateBeforeSave'*/:break;case'number'/*case 'Number'*/:break;case'refmoder1c1'/*case 'RefModeR1C1'*/:break;case'embedsavesmarttags'/*case 'EmbedSaveSmartTags'*/:break;default:seen=false;}break;/* WorkbookOptions */case'workbookoptions'/*case 'WorkbookOptions'*/:switch(Rn[3]){case'owcversion'/*case 'OWCVersion'*/:break;case'height'/*case 'Height'*/:break;case'width'/*case 'Width'*/:break;default:seen=false;}break;/* WorksheetOptions */case'worksheetoptions'/*case 'WorksheetOptions'*/:switch(Rn[3]){case'visible'/*case 'Visible'*/:if(Rn[0].slice(-2)==="/>");else if(Rn[1]==="/")switch(str.slice(pidx,Rn.index)){case"SheetHidden":wsprops.Hidden=1;break;case"SheetVeryHidden":wsprops.Hidden=2;break;}else pidx=Rn.index+Rn[0].length;break;case'header'/*case 'Header'*/:if(!cursheet['!margins'])default_margins(cursheet['!margins']={},'xlml');if(!isNaN(+parsexmltag(Rn[0]).Margin))cursheet['!margins'].header=+parsexmltag(Rn[0]).Margin;break;case'footer'/*case 'Footer'*/:if(!cursheet['!margins'])default_margins(cursheet['!margins']={},'xlml');if(!isNaN(+parsexmltag(Rn[0]).Margin))cursheet['!margins'].footer=+parsexmltag(Rn[0]).Margin;break;case'pagemargins'/*case 'PageMargins'*/:var pagemargins=parsexmltag(Rn[0]);if(!cursheet['!margins'])default_margins(cursheet['!margins']={},'xlml');if(!isNaN(+pagemargins.Top))cursheet['!margins'].top=+pagemargins.Top;if(!isNaN(+pagemargins.Left))cursheet['!margins'].left=+pagemargins.Left;if(!isNaN(+pagemargins.Right))cursheet['!margins'].right=+pagemargins.Right;if(!isNaN(+pagemargins.Bottom))cursheet['!margins'].bottom=+pagemargins.Bottom;break;case'displayrighttoleft'/*case 'DisplayRightToLeft'*/:if(!Workbook.Views)Workbook.Views=[];if(!Workbook.Views[0])Workbook.Views[0]={};Workbook.Views[0].RTL=true;break;case'freezepanes'/*case 'FreezePanes'*/:break;case'frozennosplit'/*case 'FrozenNoSplit'*/:break;case'splithorizontal'/*case 'SplitHorizontal'*/:case'splitvertical'/*case 'SplitVertical'*/:break;case'donotdisplaygridlines'/*case 'DoNotDisplayGridlines'*/:break;case'activerow'/*case 'ActiveRow'*/:break;case'activecol'/*case 'ActiveCol'*/:break;case'toprowbottompane'/*case 'TopRowBottomPane'*/:break;case'leftcolumnrightpane'/*case 'LeftColumnRightPane'*/:break;case'unsynced'/*case 'Unsynced'*/:break;case'print'/*case 'Print'*/:break;case'printerrors'/*case 'PrintErrors'*/:break;case'panes'/*case 'Panes'*/:break;case'scale'/*case 'Scale'*/:break;case'pane'/*case 'Pane'*/:break;case'number'/*case 'Number'*/:break;case'layout'/*case 'Layout'*/:break;case'pagesetup'/*case 'PageSetup'*/:break;case'selected'/*case 'Selected'*/:break;case'protectobjects'/*case 'ProtectObjects'*/:break;case'enableselection'/*case 'EnableSelection'*/:break;case'protectscenarios'/*case 'ProtectScenarios'*/:break;case'validprinterinfo'/*case 'ValidPrinterInfo'*/:break;case'horizontalresolution'/*case 'HorizontalResolution'*/:break;case'verticalresolution'/*case 'VerticalResolution'*/:break;case'numberofcopies'/*case 'NumberofCopies'*/:break;case'activepane'/*case 'ActivePane'*/:break;case'toprowvisible'/*case 'TopRowVisible'*/:break;case'leftcolumnvisible'/*case 'LeftColumnVisible'*/:break;case'fittopage'/*case 'FitToPage'*/:break;case'rangeselection'/*case 'RangeSelection'*/:break;case'papersizeindex'/*case 'PaperSizeIndex'*/:break;case'pagelayoutzoom'/*case 'PageLayoutZoom'*/:break;case'pagebreakzoom'/*case 'PageBreakZoom'*/:break;case'filteron'/*case 'FilterOn'*/:break;case'fitwidth'/*case 'FitWidth'*/:break;case'fitheight'/*case 'FitHeight'*/:break;case'commentslayout'/*case 'CommentsLayout'*/:break;case'zoom'/*case 'Zoom'*/:break;case'lefttoright'/*case 'LeftToRight'*/:break;case'gridlines'/*case 'Gridlines'*/:break;case'allowsort'/*case 'AllowSort'*/:break;case'allowfilter'/*case 'AllowFilter'*/:break;case'allowinsertrows'/*case 'AllowInsertRows'*/:break;case'allowdeleterows'/*case 'AllowDeleteRows'*/:break;case'allowinsertcols'/*case 'AllowInsertCols'*/:break;case'allowdeletecols'/*case 'AllowDeleteCols'*/:break;case'allowinserthyperlinks'/*case 'AllowInsertHyperlinks'*/:break;case'allowformatcells'/*case 'AllowFormatCells'*/:break;case'allowsizecols'/*case 'AllowSizeCols'*/:break;case'allowsizerows'/*case 'AllowSizeRows'*/:break;case'nosummaryrowsbelowdetail'/*case 'NoSummaryRowsBelowDetail'*/:if(!cursheet["!outline"])cursheet["!outline"]={};cursheet["!outline"].above=true;break;case'tabcolorindex'/*case 'TabColorIndex'*/:break;case'donotdisplayheadings'/*case 'DoNotDisplayHeadings'*/:break;case'showpagelayoutzoom'/*case 'ShowPageLayoutZoom'*/:break;case'nosummarycolumnsrightdetail'/*case 'NoSummaryColumnsRightDetail'*/:if(!cursheet["!outline"])cursheet["!outline"]={};cursheet["!outline"].left=true;break;case'blackandwhite'/*case 'BlackAndWhite'*/:break;case'donotdisplayzeros'/*case 'DoNotDisplayZeros'*/:break;case'displaypagebreak'/*case 'DisplayPageBreak'*/:break;case'rowcolheadings'/*case 'RowColHeadings'*/:break;case'donotdisplayoutline'/*case 'DoNotDisplayOutline'*/:break;case'noorientation'/*case 'NoOrientation'*/:break;case'allowusepivottables'/*case 'AllowUsePivotTables'*/:break;case'zeroheight'/*case 'ZeroHeight'*/:break;case'viewablerange'/*case 'ViewableRange'*/:break;case'selection'/*case 'Selection'*/:break;case'protectcontents'/*case 'ProtectContents'*/:break;default:seen=false;}break;/* PivotTable */case'pivottable'/*case 'PivotTable'*/:case'pivotcache'/*case 'PivotCache'*/:switch(Rn[3]){case'immediateitemsondrop'/*case 'ImmediateItemsOnDrop'*/:break;case'showpagemultipleitemlabel'/*case 'ShowPageMultipleItemLabel'*/:break;case'compactrowindent'/*case 'CompactRowIndent'*/:break;case'location'/*case 'Location'*/:break;case'pivotfield'/*case 'PivotField'*/:break;case'orientation'/*case 'Orientation'*/:break;case'layoutform'/*case 'LayoutForm'*/:break;case'layoutsubtotallocation'/*case 'LayoutSubtotalLocation'*/:break;case'layoutcompactrow'/*case 'LayoutCompactRow'*/:break;case'position'/*case 'Position'*/:break;case'pivotitem'/*case 'PivotItem'*/:break;case'datatype'/*case 'DataType'*/:break;case'datafield'/*case 'DataField'*/:break;case'sourcename'/*case 'SourceName'*/:break;case'parentfield'/*case 'ParentField'*/:break;case'ptlineitems'/*case 'PTLineItems'*/:break;case'ptlineitem'/*case 'PTLineItem'*/:break;case'countofsameitems'/*case 'CountOfSameItems'*/:break;case'item'/*case 'Item'*/:break;case'itemtype'/*case 'ItemType'*/:break;case'ptsource'/*case 'PTSource'*/:break;case'cacheindex'/*case 'CacheIndex'*/:break;case'consolidationreference'/*case 'ConsolidationReference'*/:break;case'filename'/*case 'FileName'*/:break;case'reference'/*case 'Reference'*/:break;case'nocolumngrand'/*case 'NoColumnGrand'*/:break;case'norowgrand'/*case 'NoRowGrand'*/:break;case'blanklineafteritems'/*case 'BlankLineAfterItems'*/:break;case'hidden'/*case 'Hidden'*/:break;case'subtotal'/*case 'Subtotal'*/:break;case'basefield'/*case 'BaseField'*/:break;case'mapchilditems'/*case 'MapChildItems'*/:break;case'function'/*case 'Function'*/:break;case'refreshonfileopen'/*case 'RefreshOnFileOpen'*/:break;case'printsettitles'/*case 'PrintSetTitles'*/:break;case'mergelabels'/*case 'MergeLabels'*/:break;case'defaultversion'/*case 'DefaultVersion'*/:break;case'refreshname'/*case 'RefreshName'*/:break;case'refreshdate'/*case 'RefreshDate'*/:break;case'refreshdatecopy'/*case 'RefreshDateCopy'*/:break;case'versionlastrefresh'/*case 'VersionLastRefresh'*/:break;case'versionlastupdate'/*case 'VersionLastUpdate'*/:break;case'versionupdateablemin'/*case 'VersionUpdateableMin'*/:break;case'versionrefreshablemin'/*case 'VersionRefreshableMin'*/:break;case'calculation'/*case 'Calculation'*/:break;default:seen=false;}break;/* PageBreaks */case'pagebreaks'/*case 'PageBreaks'*/:switch(Rn[3]){case'colbreaks'/*case 'ColBreaks'*/:break;case'colbreak'/*case 'ColBreak'*/:break;case'rowbreaks'/*case 'RowBreaks'*/:break;case'rowbreak'/*case 'RowBreak'*/:break;case'colstart'/*case 'ColStart'*/:break;case'colend'/*case 'ColEnd'*/:break;case'rowend'/*case 'RowEnd'*/:break;default:seen=false;}break;/* AutoFilter */case'autofilter'/*case 'AutoFilter'*/:switch(Rn[3]){case'autofiltercolumn'/*case 'AutoFilterColumn'*/:break;case'autofiltercondition'/*case 'AutoFilterCondition'*/:break;case'autofilterand'/*case 'AutoFilterAnd'*/:break;case'autofilteror'/*case 'AutoFilterOr'*/:break;default:seen=false;}break;/* QueryTable */case'querytable'/*case 'QueryTable'*/:switch(Rn[3]){case'id'/*case 'Id'*/:break;case'autoformatfont'/*case 'AutoFormatFont'*/:break;case'autoformatpattern'/*case 'AutoFormatPattern'*/:break;case'querysource'/*case 'QuerySource'*/:break;case'querytype'/*case 'QueryType'*/:break;case'enableredirections'/*case 'EnableRedirections'*/:break;case'refreshedinxl9'/*case 'RefreshedInXl9'*/:break;case'urlstring'/*case 'URLString'*/:break;case'htmltables'/*case 'HTMLTables'*/:break;case'connection'/*case 'Connection'*/:break;case'commandtext'/*case 'CommandText'*/:break;case'refreshinfo'/*case 'RefreshInfo'*/:break;case'notitles'/*case 'NoTitles'*/:break;case'nextid'/*case 'NextId'*/:break;case'columninfo'/*case 'ColumnInfo'*/:break;case'overwritecells'/*case 'OverwriteCells'*/:break;case'donotpromptforfile'/*case 'DoNotPromptForFile'*/:break;case'textwizardsettings'/*case 'TextWizardSettings'*/:break;case'source'/*case 'Source'*/:break;case'number'/*case 'Number'*/:break;case'decimal'/*case 'Decimal'*/:break;case'thousandseparator'/*case 'ThousandSeparator'*/:break;case'trailingminusnumbers'/*case 'TrailingMinusNumbers'*/:break;case'formatsettings'/*case 'FormatSettings'*/:break;case'fieldtype'/*case 'FieldType'*/:break;case'delimiters'/*case 'Delimiters'*/:break;case'tab'/*case 'Tab'*/:break;case'comma'/*case 'Comma'*/:break;case'autoformatname'/*case 'AutoFormatName'*/:break;case'versionlastedit'/*case 'VersionLastEdit'*/:break;case'versionlastrefresh'/*case 'VersionLastRefresh'*/:break;default:seen=false;}break;case'datavalidation'/*case 'DataValidation'*/:switch(Rn[3]){case'range'/*case 'Range'*/:break;case'type'/*case 'Type'*/:break;case'min'/*case 'Min'*/:break;case'max'/*case 'Max'*/:break;case'sort'/*case 'Sort'*/:break;case'descending'/*case 'Descending'*/:break;case'order'/*case 'Order'*/:break;case'casesensitive'/*case 'CaseSensitive'*/:break;case'value'/*case 'Value'*/:break;case'errorstyle'/*case 'ErrorStyle'*/:break;case'errormessage'/*case 'ErrorMessage'*/:break;case'errortitle'/*case 'ErrorTitle'*/:break;case'inputmessage'/*case 'InputMessage'*/:break;case'inputtitle'/*case 'InputTitle'*/:break;case'combohide'/*case 'ComboHide'*/:break;case'inputhide'/*case 'InputHide'*/:break;case'condition'/*case 'Condition'*/:break;case'qualifier'/*case 'Qualifier'*/:break;case'useblank'/*case 'UseBlank'*/:break;case'value1'/*case 'Value1'*/:break;case'value2'/*case 'Value2'*/:break;case'format'/*case 'Format'*/:break;case'cellrangelist'/*case 'CellRangeList'*/:break;default:seen=false;}break;case'sorting'/*case 'Sorting'*/:case'conditionalformatting'/*case 'ConditionalFormatting'*/:switch(Rn[3]){case'range'/*case 'Range'*/:break;case'type'/*case 'Type'*/:break;case'min'/*case 'Min'*/:break;case'max'/*case 'Max'*/:break;case'sort'/*case 'Sort'*/:break;case'descending'/*case 'Descending'*/:break;case'order'/*case 'Order'*/:break;case'casesensitive'/*case 'CaseSensitive'*/:break;case'value'/*case 'Value'*/:break;case'errorstyle'/*case 'ErrorStyle'*/:break;case'errormessage'/*case 'ErrorMessage'*/:break;case'errortitle'/*case 'ErrorTitle'*/:break;case'cellrangelist'/*case 'CellRangeList'*/:break;case'inputmessage'/*case 'InputMessage'*/:break;case'inputtitle'/*case 'InputTitle'*/:break;case'combohide'/*case 'ComboHide'*/:break;case'inputhide'/*case 'InputHide'*/:break;case'condition'/*case 'Condition'*/:break;case'qualifier'/*case 'Qualifier'*/:break;case'useblank'/*case 'UseBlank'*/:break;case'value1'/*case 'Value1'*/:break;case'value2'/*case 'Value2'*/:break;case'format'/*case 'Format'*/:break;default:seen=false;}break;/* MapInfo (schema) */case'mapinfo'/*case 'MapInfo'*/:case'schema'/*case 'Schema'*/:case'data'/*case 'data'*/:switch(Rn[3]){case'map'/*case 'Map'*/:break;case'entry'/*case 'Entry'*/:break;case'range'/*case 'Range'*/:break;case'xpath'/*case 'XPath'*/:break;case'field'/*case 'Field'*/:break;case'xsdtype'/*case 'XSDType'*/:break;case'filteron'/*case 'FilterOn'*/:break;case'aggregate'/*case 'Aggregate'*/:break;case'elementtype'/*case 'ElementType'*/:break;case'attributetype'/*case 'AttributeType'*/:break;/* These are from xsd (XML Schema Definition) */case'schema'/*case 'schema'*/:case'element'/*case 'element'*/:case'complextype'/*case 'complexType'*/:case'datatype'/*case 'datatype'*/:case'all'/*case 'all'*/:case'attribute'/*case 'attribute'*/:case'extends'/*case 'extends'*/:break;case'row'/*case 'row'*/:break;default:seen=false;}break;/* SmartTags (can be anything) */case'smarttags'/*case 'SmartTags'*/:break;default:seen=false;break;}if(seen)break;/* CustomDocumentProperties */if(Rn[3].match(/!\[CDATA/))break;if(!state[state.length-1][1])throw 'Unrecognized tag: '+Rn[3]+"|"+state.join("|");if(state[state.length-1][0]===/*'CustomDocumentProperties'*/'customdocumentproperties'){if(Rn[0].slice(-2)==="/>")break;else if(Rn[1]==="/")xlml_set_custprop(Custprops,raw_Rn3,cp,str.slice(pidx,Rn.index));else {cp=Rn;pidx=Rn.index+Rn[0].length;}break;}if(opts.WTF)throw 'Unrecognized tag: '+Rn[3]+"|"+state.join("|");}var out={}/*:any*/;if(!opts.bookSheets&&!opts.bookProps)out.Sheets=sheets;out.SheetNames=sheetnames;out.Workbook=Workbook;out.SSF=dup(table_fmt);out.Props=Props;out.Custprops=Custprops;return out;}function parse_xlml(data/*:RawBytes|string*/,opts)/*:Workbook*/{fix_read_opts(opts=opts||{});switch(opts.type||"base64"){case"base64":return parse_xlml_xml(Base64_decode(data),opts);case"binary":case"buffer":case"file":return parse_xlml_xml(data,opts);case"array":return parse_xlml_xml(a2s(data),opts);}/*:: throw new Error("unsupported type " + opts.type); */}/* [MS-OLEDS] 2.3.8 CompObjStream */function parse_compobj(obj/*:CFBEntry*/){var v={};var o=obj.content;/*:: if(o == null) return; */ /* [MS-OLEDS] 2.3.7 CompObjHeader -- All fields MUST be ignored */o.l=28;v.AnsiUserType=o.read_shift(0,"lpstr-ansi");v.AnsiClipboardFormat=parse_ClipboardFormatOrAnsiString(o);if(o.length-o.l<=4)return v;var m/*:number*/=o.read_shift(4);if(m==0||m>40)return v;o.l-=4;v.Reserved1=o.read_shift(0,"lpstr-ansi");if(o.length-o.l<=4)return v;m=o.read_shift(4);if(m!==0x71b239f4)return v;v.UnicodeClipboardFormat=parse_ClipboardFormatOrUnicodeString(o);m=o.read_shift(4);if(m==0||m>40)return v;o.l-=4;v.Reserved2=o.read_shift(0,"lpwstr");}/*
		Continue logic for:
		- 2.4.58 Continue          0x003c
		- 2.4.59 ContinueBigName   0x043c
		- 2.4.60 ContinueFrt       0x0812
		- 2.4.61 ContinueFrt11     0x0875
		- 2.4.62 ContinueFrt12     0x087f
	*/var CONTINUE_RT=[0x003c,0x043c,0x0812,0x0875,0x087f];function slurp(RecordType,R,blob,length/*:number*/,opts)/*:any*/{var l=length;var bufs=[];var d=blob.slice(blob.l,blob.l+l);if(opts&&opts.enc&&opts.enc.insitu&&d.length>0)switch(RecordType){case 0x0009:case 0x0209:case 0x0409:case 0x0809/* BOF */:case 0x002f/* FilePass */:case 0x0195/* FileLock */:case 0x00e1/* InterfaceHdr */:case 0x0196/* RRDInfo */:case 0x0138/* RRDHead */:case 0x0194/* UsrExcl */:case 0x000a/* EOF */:break;case 0x0085/* BoundSheet8 */:break;default:opts.enc.insitu(d);}bufs.push(d);blob.l+=l;var nextrt=__readUInt16LE(blob,blob.l),next=XLSRecordEnum[nextrt];var start=0;while(next!=null&&CONTINUE_RT.indexOf(nextrt)>-1){l=__readUInt16LE(blob,blob.l+2);start=blob.l+4;if(nextrt==0x0812/* ContinueFrt */)start+=4;else if(nextrt==0x0875||nextrt==0x087f){start+=12;}d=blob.slice(start,blob.l+4+l);bufs.push(d);blob.l+=4+l;next=XLSRecordEnum[nextrt=__readUInt16LE(blob,blob.l)];}var b=bconcat(bufs)/*:any*/;prep_blob(b,0);var ll=0;b.lens=[];for(var j=0;j<bufs.length;++j){b.lens.push(ll);ll+=bufs[j].length;}if(b.length<length)throw "XLS Record 0x"+RecordType.toString(16)+" Truncated: "+b.length+" < "+length;return R.f(b,b.length,opts);}function safe_format_xf(p/*:any*/,opts/*:ParseOpts*/,date1904/*:?boolean*/){if(p.t==='z')return;if(!p.XF)return;var fmtid=0;try{fmtid=p.z||p.XF.numFmtId||0;if(opts.cellNF)p.z=table_fmt[fmtid];}catch(e){if(opts.WTF)throw e;}if(!opts||opts.cellText!==false)try{if(p.t==='e'){p.w=p.w||BErr[p.v];}else if(fmtid===0||fmtid=="General"){if(p.t==='n'){if((p.v|0)===p.v)p.w=p.v.toString(10);else p.w=SSF_general_num(p.v);}else p.w=SSF_general(p.v);}else p.w=SSF_format(fmtid,p.v,{date1904:!!date1904,dateNF:opts&&opts.dateNF});}catch(e){if(opts.WTF)throw e;}if(opts.cellDates&&fmtid&&p.t=='n'&&fmt_is_date(table_fmt[fmtid]||String(fmtid))){var _d=SSF_parse_date_code(p.v);if(_d){p.t='d';p.v=new Date(_d.y,_d.m-1,_d.d,_d.H,_d.M,_d.S,_d.u);}}}function make_cell(val,ixfe,t)/*:Cell*/{return {v:val,ixfe:ixfe,t:t}/*:any*/;}// 2.3.2
	function parse_workbook(blob,options/*:ParseOpts*/)/*:Workbook*/{var wb={opts:{}}/*:any*/;var Sheets={};var out/*:Worksheet*/=options.dense?[]:{}/*:any*/;var Directory={};var range/*:Range*/={}/*:any*/;var last_formula=null;var sst/*:SST*/=[]/*:any*/;var cur_sheet="";var Preamble={};var lastcell,last_cell="",cc/*:Cell*/,cmnt,rngC,rngR;var sharedf={};var arrayf/*:Array<[Range, string]>*/=[];var temp_val/*:Cell*/;var country;var XFs=[];/* XF records */var palette/*:Array<[number, number, number]>*/=[];var Workbook/*:WBWBProps*/={Sheets:[],WBProps:{date1904:false},Views:[{}]}/*:any*/,wsprops={};var get_rgb=function getrgb(icv/*:number*/)/*:[number, number, number]*/{if(icv<8)return XLSIcv[icv];if(icv<64)return palette[icv-8]||XLSIcv[icv];return XLSIcv[icv];};var process_cell_style=function pcs(cell,line/*:any*/,options){var xfd=line.XF.data;if(!xfd||!xfd.patternType||!options||!options.cellStyles)return;line.s={}/*:any*/;line.s.patternType=xfd.patternType;var t;if(t=rgb2Hex(get_rgb(xfd.icvFore))){line.s.fgColor={rgb:t};}if(t=rgb2Hex(get_rgb(xfd.icvBack))){line.s.bgColor={rgb:t};}};var addcell=function addcell(cell/*:any*/,line/*:any*/,options/*:any*/){if(file_depth>1)return;if(options.sheetRows&&cell.r>=options.sheetRows)return;if(options.cellStyles&&line.XF&&line.XF.data)process_cell_style(cell,line,options);delete line.ixfe;delete line.XF;lastcell=cell;last_cell=encode_cell(cell);if(!range||!range.s||!range.e)range={s:{r:0,c:0},e:{r:0,c:0}};if(cell.r<range.s.r)range.s.r=cell.r;if(cell.c<range.s.c)range.s.c=cell.c;if(cell.r+1>range.e.r)range.e.r=cell.r+1;if(cell.c+1>range.e.c)range.e.c=cell.c+1;if(options.cellFormula&&line.f){for(var afi=0;afi<arrayf.length;++afi){if(arrayf[afi][0].s.c>cell.c||arrayf[afi][0].s.r>cell.r)continue;if(arrayf[afi][0].e.c<cell.c||arrayf[afi][0].e.r<cell.r)continue;line.F=encode_range(arrayf[afi][0]);if(arrayf[afi][0].s.c!=cell.c||arrayf[afi][0].s.r!=cell.r)delete line.f;if(line.f)line.f=""+stringify_formula(arrayf[afi][1],range,cell,supbooks,opts);break;}}{if(options.dense){if(!out[cell.r])out[cell.r]=[];out[cell.r][cell.c]=line;}else out[last_cell]=line;}};var opts={enc:false,// encrypted
	sbcch:0,// cch in the preceding SupBook
	snames:[],// sheetnames
	sharedf:sharedf,// shared formulae by address
	arrayf:arrayf,// array formulae array
	rrtabid:[],// RRTabId
	lastuser:"",// Last User from WriteAccess
	biff:8,// BIFF version
	codepage:0,// CP from CodePage record
	winlocked:0,// fLockWn from WinProtect
	cellStyles:!!options&&!!options.cellStyles,WTF:!!options&&!!options.wtf}/*:any*/;if(options.password)opts.password=options.password;var themes;var merges/*:Array<Range>*/=[];var objects=[];var colinfo/*:Array<ColInfo>*/=[],rowinfo/*:Array<RowInfo>*/=[];var seencol=false;var supbooks=[]/*:any*/;// 1-indexed, will hold extern names
	supbooks.SheetNames=opts.snames;supbooks.sharedf=opts.sharedf;supbooks.arrayf=opts.arrayf;supbooks.names=[];supbooks.XTI=[];var last_RT=0;var file_depth=0;/* TODO: make a real stack */var BIFF2Fmt=0,BIFF2FmtTable/*:Array<string>*/=[];var FilterDatabases=[];/* TODO: sort out supbooks and process elsewhere */var last_lbl/*:?DefinedName*/;/* explicit override for some broken writers */opts.codepage=1200;set_cp(1200);var seen_codepage=false;while(blob.l<blob.length-1){var s=blob.l;var RecordType=blob.read_shift(2);if(RecordType===0&&last_RT===0x000a/* EOF */)break;var length=blob.l===blob.length?0:blob.read_shift(2);var R=XLSRecordEnum[RecordType];//console.log(RecordType.toString(16), RecordType, R, blob.l, length, blob.length);
	//if(!R) console.log(blob.slice(blob.l, blob.l + length));
	if(R&&R.f){if(options.bookSheets){if(last_RT===0x0085/* BoundSheet8 */&&RecordType!==0x0085/* R.n !== 'BoundSheet8' */)break;}last_RT=RecordType;if(R.r===2||R.r==12){var rt=blob.read_shift(2);length-=2;if(!opts.enc&&rt!==RecordType&&((rt&0xFF)<<8|rt>>8)!==RecordType)throw new Error("rt mismatch: "+rt+"!="+RecordType);if(R.r==12){blob.l+=10;length-=10;}// skip FRT
	}//console.error(R,blob.l,length,blob.length);
	var val/*:any*/={}/*:any*/;if(RecordType===0x000a/* EOF */)val=/*::(*/R.f(blob,length,opts)/*:: :any)*/;else val=/*::(*/slurp(RecordType,R,blob,length,opts)/*:: :any)*/;/*:: val = (val:any); */if(file_depth==0&&[0x0009,0x0209,0x0409,0x0809].indexOf(last_RT)===-1/* 'BOF' */)continue;switch(RecordType){case 0x0022/* Date1904 */:/*:: if(!Workbook.WBProps) Workbook.WBProps = {}; */wb.opts.Date1904=Workbook.WBProps.date1904=val;break;case 0x0086/* WriteProtect */:wb.opts.WriteProtect=true;break;case 0x002f/* FilePass */:if(!opts.enc)blob.l=0;opts.enc=val;if(!options.password)throw new Error("File is password-protected");if(val.valid==null)throw new Error("Encryption scheme unsupported");if(!val.valid)throw new Error("Password is incorrect");break;case 0x005c/* WriteAccess */:opts.lastuser=val;break;case 0x0042/* CodePage */:var cpval=Number(val);/* overrides based on test cases */switch(cpval){case 0x5212:cpval=1200;break;case 0x8000:cpval=10000;break;case 0x8001:cpval=1252;break;}set_cp(opts.codepage=cpval);seen_codepage=true;break;case 0x013d/* RRTabId */:opts.rrtabid=val;break;case 0x0019/* WinProtect */:opts.winlocked=val;break;case 0x01b7/* RefreshAll */:wb.opts["RefreshAll"]=val;break;case 0x000c/* CalcCount */:wb.opts["CalcCount"]=val;break;case 0x0010/* CalcDelta */:wb.opts["CalcDelta"]=val;break;case 0x0011/* CalcIter */:wb.opts["CalcIter"]=val;break;case 0x000d/* CalcMode */:wb.opts["CalcMode"]=val;break;case 0x000e/* CalcPrecision */:wb.opts["CalcPrecision"]=val;break;case 0x005f/* CalcSaveRecalc */:wb.opts["CalcSaveRecalc"]=val;break;case 0x000f/* CalcRefMode */:opts.CalcRefMode=val;break;// TODO: implement R1C1
	case 0x08a3/* ForceFullCalculation */:wb.opts.FullCalc=val;break;case 0x0081/* WsBool */:if(val.fDialog)out["!type"]="dialog";if(!val.fBelow)(out["!outline"]||(out["!outline"]={})).above=true;if(!val.fRight)(out["!outline"]||(out["!outline"]={})).left=true;break;// TODO
	case 0x00e0/* XF */:XFs.push(val);break;case 0x01ae/* SupBook */:supbooks.push([val]);supbooks[supbooks.length-1].XTI=[];break;case 0x0023:case 0x0223/* ExternName */:supbooks[supbooks.length-1].push(val);break;case 0x0018:case 0x0218/* Lbl */:last_lbl={Name:val.Name,Ref:stringify_formula(val.rgce,range,null,supbooks,opts)}/*:DefinedName*/;if(val.itab>0)last_lbl.Sheet=val.itab-1;supbooks.names.push(last_lbl);if(!supbooks[0]){supbooks[0]=[];supbooks[0].XTI=[];}supbooks[supbooks.length-1].push(val);if(val.Name=="_xlnm._FilterDatabase"&&val.itab>0)if(val.rgce&&val.rgce[0]&&val.rgce[0][0]&&val.rgce[0][0][0]=='PtgArea3d')FilterDatabases[val.itab-1]={ref:encode_range(val.rgce[0][0][1][2])};break;case 0x0016/* ExternCount */:opts.ExternCount=val;break;case 0x0017/* ExternSheet */:if(supbooks.length==0){supbooks[0]=[];supbooks[0].XTI=[];}supbooks[supbooks.length-1].XTI=supbooks[supbooks.length-1].XTI.concat(val);supbooks.XTI=supbooks.XTI.concat(val);break;case 0x0894/* NameCmt */:/* TODO: search for correct name */if(opts.biff<8)break;if(last_lbl!=null)last_lbl.Comment=val[1];break;case 0x0012/* Protect */:out["!protect"]=val;break;/* for sheet or book */case 0x0013/* Password */:if(val!==0&&opts.WTF)console.error("Password verifier: "+val);break;case 0x0085/* BoundSheet8 */:{Directory[val.pos]=val;opts.snames.push(val.name);}break;case 0x000a/* EOF */:{if(--file_depth)break;if(range.e){if(range.e.r>0&&range.e.c>0){range.e.r--;range.e.c--;out["!ref"]=encode_range(range);if(options.sheetRows&&options.sheetRows<=range.e.r){var tmpri=range.e.r;range.e.r=options.sheetRows-1;out["!fullref"]=out["!ref"];out["!ref"]=encode_range(range);range.e.r=tmpri;}range.e.r++;range.e.c++;}if(merges.length>0)out["!merges"]=merges;if(objects.length>0)out["!objects"]=objects;if(colinfo.length>0)out["!cols"]=colinfo;if(rowinfo.length>0)out["!rows"]=rowinfo;Workbook.Sheets.push(wsprops);}if(cur_sheet==="")Preamble=out;else Sheets[cur_sheet]=out;out=options.dense?[]:{}/*:any*/;}break;case 0x0009:case 0x0209:case 0x0409:case 0x0809/* BOF */:{if(opts.biff===8)opts.biff={/*::[*/0x0009/*::]*/:2,/*::[*/0x0209/*::]*/:3,/*::[*/0x0409/*::]*/:4}[RecordType]||{/*::[*/0x0200/*::]*/:2,/*::[*/0x0300/*::]*/:3,/*::[*/0x0400/*::]*/:4,/*::[*/0x0500/*::]*/:5,/*::[*/0x0600/*::]*/:8,/*::[*/0x0002/*::]*/:2,/*::[*/0x0007/*::]*/:2}[val.BIFFVer]||8;opts.biffguess=val.BIFFVer==0;if(val.BIFFVer==0&&val.dt==0x1000){opts.biff=5;seen_codepage=true;set_cp(opts.codepage=28591);}if(opts.biff==8&&val.BIFFVer==0&&val.dt==16)opts.biff=2;if(file_depth++)break;out=options.dense?[]:{}/*:any*/;if(opts.biff<8&&!seen_codepage){seen_codepage=true;set_cp(opts.codepage=options.codepage||1252);}if(opts.biff<5||val.BIFFVer==0&&val.dt==0x1000){if(cur_sheet==="")cur_sheet="Sheet1";range={s:{r:0,c:0},e:{r:0,c:0}};/* fake BoundSheet8 */var fakebs8={pos:blob.l-length,name:cur_sheet};Directory[fakebs8.pos]=fakebs8;opts.snames.push(cur_sheet);}else cur_sheet=(Directory[s]||{name:""}).name;if(val.dt==0x20)out["!type"]="chart";if(val.dt==0x40)out["!type"]="macro";merges=[];objects=[];opts.arrayf=arrayf=[];colinfo=[];rowinfo=[];seencol=false;wsprops={Hidden:(Directory[s]||{hs:0}).hs,name:cur_sheet};}break;case 0x0203/* Number */:case 0x0003/* BIFF2NUM */:case 0x0002/* BIFF2INT */:{if(out["!type"]=="chart")if(options.dense?(out[val.r]||[])[val.c]:out[encode_cell({c:val.c,r:val.r})])++val.c;temp_val={ixfe:val.ixfe,XF:XFs[val.ixfe]||{},v:val.val,t:'n'}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);}break;case 0x0005:case 0x0205/* BoolErr */:{temp_val={ixfe:val.ixfe,XF:XFs[val.ixfe],v:val.val,t:val.t}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);}break;case 0x027e/* RK */:{temp_val={ixfe:val.ixfe,XF:XFs[val.ixfe],v:val.rknum,t:'n'}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);}break;case 0x00bd/* MulRk */:{for(var j=val.c;j<=val.C;++j){var ixfe=val.rkrec[j-val.c][0];temp_val={ixfe:ixfe,XF:XFs[ixfe],v:val.rkrec[j-val.c][1],t:'n'}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:j,r:val.r},temp_val,options);}}break;case 0x0006:case 0x0206:case 0x0406/* Formula */:{if(val.val=='String'){last_formula=val;break;}temp_val=make_cell(val.val,val.cell.ixfe,val.tt);temp_val.XF=XFs[temp_val.ixfe];if(options.cellFormula){var _f=val.formula;if(_f&&_f[0]&&_f[0][0]&&_f[0][0][0]=='PtgExp'){var _fr=_f[0][0][1][0],_fc=_f[0][0][1][1];var _fe=encode_cell({r:_fr,c:_fc});if(sharedf[_fe])temp_val.f=""+stringify_formula(val.formula,range,val.cell,supbooks,opts);else temp_val.F=((options.dense?(out[_fr]||[])[_fc]:out[_fe])||{}).F;}else temp_val.f=""+stringify_formula(val.formula,range,val.cell,supbooks,opts);}if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell(val.cell,temp_val,options);last_formula=val;}break;case 0x0007:case 0x0207/* String */:{if(last_formula){/* technically always true */last_formula.val=val;temp_val=make_cell(val,last_formula.cell.ixfe,'s');temp_val.XF=XFs[temp_val.ixfe];if(options.cellFormula){temp_val.f=""+stringify_formula(last_formula.formula,range,last_formula.cell,supbooks,opts);}if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell(last_formula.cell,temp_val,options);last_formula=null;}else throw new Error("String record expects Formula");}break;case 0x0021:case 0x0221/* Array */:{arrayf.push(val);var _arraystart=encode_cell(val[0].s);cc=options.dense?(out[val[0].s.r]||[])[val[0].s.c]:out[_arraystart];if(options.cellFormula&&cc){if(!last_formula)break;/* technically unreachable */if(!_arraystart||!cc)break;cc.f=""+stringify_formula(val[1],range,val[0],supbooks,opts);cc.F=encode_range(val[0]);}}break;case 0x04bc/* ShrFmla */:{if(!options.cellFormula)break;if(last_cell){/* TODO: capture range */if(!last_formula)break;/* technically unreachable */sharedf[encode_cell(last_formula.cell)]=val[0];cc=options.dense?(out[last_formula.cell.r]||[])[last_formula.cell.c]:out[encode_cell(last_formula.cell)];(cc||{}).f=""+stringify_formula(val[0],range,lastcell,supbooks,opts);}}break;case 0x00fd/* LabelSst */:temp_val=make_cell(sst[val.isst].t,val.ixfe,'s');if(sst[val.isst].h)temp_val.h=sst[val.isst].h;temp_val.XF=XFs[temp_val.ixfe];if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);break;case 0x0201/* Blank */:if(options.sheetStubs){temp_val={ixfe:val.ixfe,XF:XFs[val.ixfe],t:'z'}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);}break;case 0x00be/* MulBlank */:if(options.sheetStubs){for(var _j=val.c;_j<=val.C;++_j){var _ixfe=val.ixfe[_j-val.c];temp_val={ixfe:_ixfe,XF:XFs[_ixfe],t:'z'}/*:any*/;if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:_j,r:val.r},temp_val,options);}}break;case 0x00d6/* RString */:case 0x0204/* Label */:case 0x0004/* BIFF2STR */:temp_val=make_cell(val.val,val.ixfe,'s');temp_val.XF=XFs[temp_val.ixfe];if(BIFF2Fmt>0)temp_val.z=BIFF2FmtTable[temp_val.ixfe>>8&0x3F];safe_format_xf(temp_val,options,wb.opts.Date1904);addcell({c:val.c,r:val.r},temp_val,options);break;case 0x0000:case 0x0200/* Dimensions */:{if(file_depth===1)range=val;/* TODO: stack */}break;case 0x00fc/* SST */:{sst=val;}break;case 0x041e/* Format */:{/* val = [id, fmt] */if(opts.biff==4){BIFF2FmtTable[BIFF2Fmt++]=val[1];for(var b4idx=0;b4idx<BIFF2Fmt+163;++b4idx)if(table_fmt[b4idx]==val[1])break;if(b4idx>=163)SSF_load(val[1],BIFF2Fmt+163);}else SSF_load(val[1],val[0]);}break;case 0x001e/* BIFF2FORMAT */:{BIFF2FmtTable[BIFF2Fmt++]=val;for(var b2idx=0;b2idx<BIFF2Fmt+163;++b2idx)if(table_fmt[b2idx]==val)break;if(b2idx>=163)SSF_load(val,BIFF2Fmt+163);}break;case 0x00e5/* MergeCells */:merges=merges.concat(val);break;case 0x005d/* Obj */:objects[val.cmo[0]]=opts.lastobj=val;break;case 0x01b6/* TxO */:opts.lastobj.TxO=val;break;case 0x007f/* ImData */:opts.lastobj.ImData=val;break;case 0x01b8/* HLink */:{for(rngR=val[0].s.r;rngR<=val[0].e.r;++rngR)for(rngC=val[0].s.c;rngC<=val[0].e.c;++rngC){cc=options.dense?(out[rngR]||[])[rngC]:out[encode_cell({c:rngC,r:rngR})];if(cc)cc.l=val[1];}}break;case 0x0800/* HLinkTooltip */:{for(rngR=val[0].s.r;rngR<=val[0].e.r;++rngR)for(rngC=val[0].s.c;rngC<=val[0].e.c;++rngC){cc=options.dense?(out[rngR]||[])[rngC]:out[encode_cell({c:rngC,r:rngR})];if(cc&&cc.l)cc.l.Tooltip=val[1];}}break;case 0x001c/* Note */:{if(opts.biff<=5&&opts.biff>=2)break;/* TODO: BIFF5 */cc=options.dense?(out[val[0].r]||[])[val[0].c]:out[encode_cell(val[0])];var noteobj=objects[val[2]];if(!cc){if(options.dense){if(!out[val[0].r])out[val[0].r]=[];cc=out[val[0].r][val[0].c]={t:"z"}/*:any*/;}else {cc=out[encode_cell(val[0])]={t:"z"}/*:any*/;}range.e.r=Math.max(range.e.r,val[0].r);range.s.r=Math.min(range.s.r,val[0].r);range.e.c=Math.max(range.e.c,val[0].c);range.s.c=Math.min(range.s.c,val[0].c);}if(!cc.c)cc.c=[];cmnt={a:val[1],t:noteobj.TxO.t};cc.c.push(cmnt);}break;case 0x087d/* XFExt */:update_xfext(XFs[val.ixfe],val.ext);break;case 0x007d/* ColInfo */:{if(!opts.cellStyles)break;while(val.e>=val.s){colinfo[val.e--]={width:val.w/256,level:val.level||0,hidden:!!(val.flags&1)};if(!seencol){seencol=true;find_mdw_colw(val.w/256);}process_col(colinfo[val.e+1]);}}break;case 0x0208/* Row */:{var rowobj={};if(val.level!=null){rowinfo[val.r]=rowobj;rowobj.level=val.level;}if(val.hidden){rowinfo[val.r]=rowobj;rowobj.hidden=true;}if(val.hpt){rowinfo[val.r]=rowobj;rowobj.hpt=val.hpt;rowobj.hpx=pt2px(val.hpt);}}break;case 0x0026/* LeftMargin */:case 0x0027/* RightMargin */:case 0x0028/* TopMargin */:case 0x0029/* BottomMargin */:if(!out['!margins'])default_margins(out['!margins']={});out['!margins'][{0x26:"left",0x27:"right",0x28:"top",0x29:"bottom"}[RecordType]]=val;break;case 0x00a1/* Setup */:// TODO
	if(!out['!margins'])default_margins(out['!margins']={});out['!margins'].header=val.header;out['!margins'].footer=val.footer;break;case 0x023e/* Window2 */:// TODO
	// $FlowIgnore
	if(val.RTL)Workbook.Views[0].RTL=true;break;case 0x0092/* Palette */:palette=val;break;case 0x0896/* Theme */:themes=val;break;case 0x008c/* Country */:country=val;break;case 0x01ba/* CodeName */:{/*:: if(!Workbook.WBProps) Workbook.WBProps = {}; */if(!cur_sheet)Workbook.WBProps.CodeName=val||"ThisWorkbook";else wsprops.CodeName=val||wsprops.name;}break;}}else {if(!R)console.error("Missing Info for XLS Record 0x"+RecordType.toString(16));blob.l+=length;}}wb.SheetNames=keys(Directory).sort(function(a,b){return Number(a)-Number(b);}).map(function(x){return Directory[x].name;});if(!options.bookSheets)wb.Sheets=Sheets;if(!wb.SheetNames.length&&Preamble["!ref"]){wb.SheetNames.push("Sheet1");/*jshint -W069 */if(wb.Sheets)wb.Sheets["Sheet1"]=Preamble;/*jshint +W069 */}else wb.Preamble=Preamble;if(wb.Sheets)FilterDatabases.forEach(function(r,i){wb.Sheets[wb.SheetNames[i]]['!autofilter']=r;});wb.Strings=sst;wb.SSF=dup(table_fmt);if(opts.enc)wb.Encryption=opts.enc;if(themes)wb.Themes=themes;wb.Metadata={};if(country!==undefined)wb.Metadata.Country=country;if(supbooks.names.length>0)Workbook.Names=supbooks.names;wb.Workbook=Workbook;return wb;}/* TODO: split props*/var PSCLSID={SI:"e0859ff2f94f6810ab9108002b27b3d9",DSI:"02d5cdd59c2e1b10939708002b2cf9ae",UDI:"05d5cdd59c2e1b10939708002b2cf9ae"};function parse_xls_props(cfb/*:CFBContainer*/,props,o){/* [MS-OSHARED] 2.3.3.2.2 Document Summary Information Property Set */var DSI=CFB.find(cfb,'/!DocumentSummaryInformation');if(DSI&&DSI.size>0)try{var DocSummary=parse_PropertySetStream(DSI,DocSummaryPIDDSI,PSCLSID.DSI);for(var d in DocSummary)props[d]=DocSummary[d];}catch(e){if(o.WTF)throw e;/* empty */}/* [MS-OSHARED] 2.3.3.2.1 Summary Information Property Set*/var SI=CFB.find(cfb,'/!SummaryInformation');if(SI&&SI.size>0)try{var Summary=parse_PropertySetStream(SI,SummaryPIDSI,PSCLSID.SI);for(var s in Summary)if(props[s]==null)props[s]=Summary[s];}catch(e){if(o.WTF)throw e;/* empty */}if(props.HeadingPairs&&props.TitlesOfParts){load_props_pairs(props.HeadingPairs,props.TitlesOfParts,props,o);delete props.HeadingPairs;delete props.TitlesOfParts;}}function parse_xlscfb(cfb/*:any*/,options/*:?ParseOpts*/)/*:Workbook*/{if(!options)options={};fix_read_opts(options);reset_cp();if(options.codepage)set_ansi(options.codepage);var CompObj/*:?CFBEntry*/,WB/*:?any*/;if(cfb.FullPaths){if(CFB.find(cfb,'/encryption'))throw new Error("File is password-protected");CompObj=CFB.find(cfb,'!CompObj');WB=CFB.find(cfb,'/Workbook')||CFB.find(cfb,'/Book');}else {switch(options.type){case'base64':cfb=s2a(Base64_decode(cfb));break;case'binary':cfb=s2a(cfb);break;case'buffer':break;case'array':if(!Array.isArray(cfb))cfb=Array.prototype.slice.call(cfb);break;}prep_blob(cfb,0);WB={content:cfb}/*:any*/;}var/*::CompObjP, */WorkbookP/*:: :Workbook = XLSX.utils.book_new(); */;var _data/*:?any*/;if(CompObj)/*::CompObjP = */parse_compobj(CompObj);if(options.bookProps&&!options.bookSheets)WorkbookP={}/*:any*/;else/*:: if(cfb instanceof CFBContainer) */{var T=has_buf?'buffer':'array';if(WB&&WB.content)WorkbookP=parse_workbook(WB.content,options);/* Quattro Pro 7-8 */else if((_data=CFB.find(cfb,'PerfectOffice_MAIN'))&&_data.content)WorkbookP=WK_.to_workbook(_data.content,(options.type=T,options));/* Quattro Pro 9 */else if((_data=CFB.find(cfb,'NativeContent_MAIN'))&&_data.content)WorkbookP=WK_.to_workbook(_data.content,(options.type=T,options));/* Works 4 for Mac */else if((_data=CFB.find(cfb,'MN0'))&&_data.content)throw new Error("Unsupported Works 4 for Mac file");else throw new Error("Cannot find Workbook stream");if(options.bookVBA&&cfb.FullPaths&&CFB.find(cfb,'/_VBA_PROJECT_CUR/VBA/dir'))WorkbookP.vbaraw=make_vba_xls(cfb);}var props={};if(cfb.FullPaths)parse_xls_props(/*::((*/cfb/*:: :any):CFBContainer)*/,props,options);WorkbookP.Props=WorkbookP.Custprops=props;/* TODO: split up properties */if(options.bookFiles)WorkbookP.cfb=cfb;/*WorkbookP.CompObjP = CompObjP; // TODO: storage? */return WorkbookP;}/* [MS-XLSB] 2.3 Record Enumeration */var XLSBRecordEnum={/*::[*/0x0000/*::]*/:{/* n:"BrtRowHdr", */f:parse_BrtRowHdr},/*::[*/0x0001/*::]*/:{/* n:"BrtCellBlank", */f:parse_BrtCellBlank},/*::[*/0x0002/*::]*/:{/* n:"BrtCellRk", */f:parse_BrtCellRk},/*::[*/0x0003/*::]*/:{/* n:"BrtCellError", */f:parse_BrtCellError},/*::[*/0x0004/*::]*/:{/* n:"BrtCellBool", */f:parse_BrtCellBool},/*::[*/0x0005/*::]*/:{/* n:"BrtCellReal", */f:parse_BrtCellReal},/*::[*/0x0006/*::]*/:{/* n:"BrtCellSt", */f:parse_BrtCellSt},/*::[*/0x0007/*::]*/:{/* n:"BrtCellIsst", */f:parse_BrtCellIsst},/*::[*/0x0008/*::]*/:{/* n:"BrtFmlaString", */f:parse_BrtFmlaString},/*::[*/0x0009/*::]*/:{/* n:"BrtFmlaNum", */f:parse_BrtFmlaNum},/*::[*/0x000A/*::]*/:{/* n:"BrtFmlaBool", */f:parse_BrtFmlaBool},/*::[*/0x000B/*::]*/:{/* n:"BrtFmlaError", */f:parse_BrtFmlaError},/*::[*/0x000C/*::]*/:{/* n:"BrtShortBlank", */f:parse_BrtShortBlank},/*::[*/0x000D/*::]*/:{/* n:"BrtShortRk", */f:parse_BrtShortRk},/*::[*/0x000E/*::]*/:{/* n:"BrtShortError", */f:parse_BrtShortError},/*::[*/0x000F/*::]*/:{/* n:"BrtShortBool", */f:parse_BrtShortBool},/*::[*/0x0010/*::]*/:{/* n:"BrtShortReal", */f:parse_BrtShortReal},/*::[*/0x0011/*::]*/:{/* n:"BrtShortSt", */f:parse_BrtShortSt},/*::[*/0x0012/*::]*/:{/* n:"BrtShortIsst", */f:parse_BrtShortIsst},/*::[*/0x0013/*::]*/:{/* n:"BrtSSTItem", */f:parse_RichStr},/*::[*/0x0014/*::]*/:{/* n:"BrtPCDIMissing" */},/*::[*/0x0015/*::]*/:{/* n:"BrtPCDINumber" */},/*::[*/0x0016/*::]*/:{/* n:"BrtPCDIBoolean" */},/*::[*/0x0017/*::]*/:{/* n:"BrtPCDIError" */},/*::[*/0x0018/*::]*/:{/* n:"BrtPCDIString" */},/*::[*/0x0019/*::]*/:{/* n:"BrtPCDIDatetime" */},/*::[*/0x001A/*::]*/:{/* n:"BrtPCDIIndex" */},/*::[*/0x001B/*::]*/:{/* n:"BrtPCDIAMissing" */},/*::[*/0x001C/*::]*/:{/* n:"BrtPCDIANumber" */},/*::[*/0x001D/*::]*/:{/* n:"BrtPCDIABoolean" */},/*::[*/0x001E/*::]*/:{/* n:"BrtPCDIAError" */},/*::[*/0x001F/*::]*/:{/* n:"BrtPCDIAString" */},/*::[*/0x0020/*::]*/:{/* n:"BrtPCDIADatetime" */},/*::[*/0x0021/*::]*/:{/* n:"BrtPCRRecord" */},/*::[*/0x0022/*::]*/:{/* n:"BrtPCRRecordDt" */},/*::[*/0x0023/*::]*/:{/* n:"BrtFRTBegin", */T:1},/*::[*/0x0024/*::]*/:{/* n:"BrtFRTEnd", */T:-1},/*::[*/0x0025/*::]*/:{/* n:"BrtACBegin", */T:1},/*::[*/0x0026/*::]*/:{/* n:"BrtACEnd", */T:-1},/*::[*/0x0027/*::]*/:{/* n:"BrtName", */f:parse_BrtName},/*::[*/0x0028/*::]*/:{/* n:"BrtIndexRowBlock" */},/*::[*/0x002A/*::]*/:{/* n:"BrtIndexBlock" */},/*::[*/0x002B/*::]*/:{/* n:"BrtFont", */f:parse_BrtFont},/*::[*/0x002C/*::]*/:{/* n:"BrtFmt", */f:parse_BrtFmt},/*::[*/0x002D/*::]*/:{/* n:"BrtFill", */f:parse_BrtFill},/*::[*/0x002E/*::]*/:{/* n:"BrtBorder", */f:parse_BrtBorder},/*::[*/0x002F/*::]*/:{/* n:"BrtXF", */f:parse_BrtXF},/*::[*/0x0030/*::]*/:{/* n:"BrtStyle" */},/*::[*/0x0031/*::]*/:{/* n:"BrtCellMeta", */f:parse_Int32LE},/*::[*/0x0032/*::]*/:{/* n:"BrtValueMeta" */},/*::[*/0x0033/*::]*/:{/* n:"BrtMdb" */f:parse_BrtMdb},/*::[*/0x0034/*::]*/:{/* n:"BrtBeginFmd", */T:1},/*::[*/0x0035/*::]*/:{/* n:"BrtEndFmd", */T:-1},/*::[*/0x0036/*::]*/:{/* n:"BrtBeginMdx", */T:1},/*::[*/0x0037/*::]*/:{/* n:"BrtEndMdx", */T:-1},/*::[*/0x0038/*::]*/:{/* n:"BrtBeginMdxTuple", */T:1},/*::[*/0x0039/*::]*/:{/* n:"BrtEndMdxTuple", */T:-1},/*::[*/0x003A/*::]*/:{/* n:"BrtMdxMbrIstr" */},/*::[*/0x003B/*::]*/:{/* n:"BrtStr" */},/*::[*/0x003C/*::]*/:{/* n:"BrtColInfo", */f:parse_ColInfo},/*::[*/0x003E/*::]*/:{/* n:"BrtCellRString", */f:parse_BrtCellRString},/*::[*/0x003F/*::]*/:{/* n:"BrtCalcChainItem$", */f:parse_BrtCalcChainItem$},/*::[*/0x0040/*::]*/:{/* n:"BrtDVal", */f:parse_BrtDVal},/*::[*/0x0041/*::]*/:{/* n:"BrtSxvcellNum" */},/*::[*/0x0042/*::]*/:{/* n:"BrtSxvcellStr" */},/*::[*/0x0043/*::]*/:{/* n:"BrtSxvcellBool" */},/*::[*/0x0044/*::]*/:{/* n:"BrtSxvcellErr" */},/*::[*/0x0045/*::]*/:{/* n:"BrtSxvcellDate" */},/*::[*/0x0046/*::]*/:{/* n:"BrtSxvcellNil" */},/*::[*/0x0080/*::]*/:{/* n:"BrtFileVersion" */},/*::[*/0x0081/*::]*/:{/* n:"BrtBeginSheet", */T:1},/*::[*/0x0082/*::]*/:{/* n:"BrtEndSheet", */T:-1},/*::[*/0x0083/*::]*/:{/* n:"BrtBeginBook", */T:1,f:parsenoop,p:0},/*::[*/0x0084/*::]*/:{/* n:"BrtEndBook", */T:-1},/*::[*/0x0085/*::]*/:{/* n:"BrtBeginWsViews", */T:1},/*::[*/0x0086/*::]*/:{/* n:"BrtEndWsViews", */T:-1},/*::[*/0x0087/*::]*/:{/* n:"BrtBeginBookViews", */T:1},/*::[*/0x0088/*::]*/:{/* n:"BrtEndBookViews", */T:-1},/*::[*/0x0089/*::]*/:{/* n:"BrtBeginWsView", */T:1,f:parse_BrtBeginWsView},/*::[*/0x008A/*::]*/:{/* n:"BrtEndWsView", */T:-1},/*::[*/0x008B/*::]*/:{/* n:"BrtBeginCsViews", */T:1},/*::[*/0x008C/*::]*/:{/* n:"BrtEndCsViews", */T:-1},/*::[*/0x008D/*::]*/:{/* n:"BrtBeginCsView", */T:1},/*::[*/0x008E/*::]*/:{/* n:"BrtEndCsView", */T:-1},/*::[*/0x008F/*::]*/:{/* n:"BrtBeginBundleShs", */T:1},/*::[*/0x0090/*::]*/:{/* n:"BrtEndBundleShs", */T:-1},/*::[*/0x0091/*::]*/:{/* n:"BrtBeginSheetData", */T:1},/*::[*/0x0092/*::]*/:{/* n:"BrtEndSheetData", */T:-1},/*::[*/0x0093/*::]*/:{/* n:"BrtWsProp", */f:parse_BrtWsProp},/*::[*/0x0094/*::]*/:{/* n:"BrtWsDim", */f:parse_BrtWsDim,p:16},/*::[*/0x0097/*::]*/:{/* n:"BrtPane", */f:parse_BrtPane},/*::[*/0x0098/*::]*/:{/* n:"BrtSel" */},/*::[*/0x0099/*::]*/:{/* n:"BrtWbProp", */f:parse_BrtWbProp},/*::[*/0x009A/*::]*/:{/* n:"BrtWbFactoid" */},/*::[*/0x009B/*::]*/:{/* n:"BrtFileRecover" */},/*::[*/0x009C/*::]*/:{/* n:"BrtBundleSh", */f:parse_BrtBundleSh},/*::[*/0x009D/*::]*/:{/* n:"BrtCalcProp" */},/*::[*/0x009E/*::]*/:{/* n:"BrtBookView" */},/*::[*/0x009F/*::]*/:{/* n:"BrtBeginSst", */T:1,f:parse_BrtBeginSst},/*::[*/0x00A0/*::]*/:{/* n:"BrtEndSst", */T:-1},/*::[*/0x00A1/*::]*/:{/* n:"BrtBeginAFilter", */T:1,f:parse_UncheckedRfX},/*::[*/0x00A2/*::]*/:{/* n:"BrtEndAFilter", */T:-1},/*::[*/0x00A3/*::]*/:{/* n:"BrtBeginFilterColumn", */T:1},/*::[*/0x00A4/*::]*/:{/* n:"BrtEndFilterColumn", */T:-1},/*::[*/0x00A5/*::]*/:{/* n:"BrtBeginFilters", */T:1},/*::[*/0x00A6/*::]*/:{/* n:"BrtEndFilters", */T:-1},/*::[*/0x00A7/*::]*/:{/* n:"BrtFilter" */},/*::[*/0x00A8/*::]*/:{/* n:"BrtColorFilter" */},/*::[*/0x00A9/*::]*/:{/* n:"BrtIconFilter" */},/*::[*/0x00AA/*::]*/:{/* n:"BrtTop10Filter" */},/*::[*/0x00AB/*::]*/:{/* n:"BrtDynamicFilter" */},/*::[*/0x00AC/*::]*/:{/* n:"BrtBeginCustomFilters", */T:1},/*::[*/0x00AD/*::]*/:{/* n:"BrtEndCustomFilters", */T:-1},/*::[*/0x00AE/*::]*/:{/* n:"BrtCustomFilter" */},/*::[*/0x00AF/*::]*/:{/* n:"BrtAFilterDateGroupItem" */},/*::[*/0x00B0/*::]*/:{/* n:"BrtMergeCell", */f:parse_BrtMergeCell},/*::[*/0x00B1/*::]*/:{/* n:"BrtBeginMergeCells", */T:1},/*::[*/0x00B2/*::]*/:{/* n:"BrtEndMergeCells", */T:-1},/*::[*/0x00B3/*::]*/:{/* n:"BrtBeginPivotCacheDef", */T:1},/*::[*/0x00B4/*::]*/:{/* n:"BrtEndPivotCacheDef", */T:-1},/*::[*/0x00B5/*::]*/:{/* n:"BrtBeginPCDFields", */T:1},/*::[*/0x00B6/*::]*/:{/* n:"BrtEndPCDFields", */T:-1},/*::[*/0x00B7/*::]*/:{/* n:"BrtBeginPCDField", */T:1},/*::[*/0x00B8/*::]*/:{/* n:"BrtEndPCDField", */T:-1},/*::[*/0x00B9/*::]*/:{/* n:"BrtBeginPCDSource", */T:1},/*::[*/0x00BA/*::]*/:{/* n:"BrtEndPCDSource", */T:-1},/*::[*/0x00BB/*::]*/:{/* n:"BrtBeginPCDSRange", */T:1},/*::[*/0x00BC/*::]*/:{/* n:"BrtEndPCDSRange", */T:-1},/*::[*/0x00BD/*::]*/:{/* n:"BrtBeginPCDFAtbl", */T:1},/*::[*/0x00BE/*::]*/:{/* n:"BrtEndPCDFAtbl", */T:-1},/*::[*/0x00BF/*::]*/:{/* n:"BrtBeginPCDIRun", */T:1},/*::[*/0x00C0/*::]*/:{/* n:"BrtEndPCDIRun", */T:-1},/*::[*/0x00C1/*::]*/:{/* n:"BrtBeginPivotCacheRecords", */T:1},/*::[*/0x00C2/*::]*/:{/* n:"BrtEndPivotCacheRecords", */T:-1},/*::[*/0x00C3/*::]*/:{/* n:"BrtBeginPCDHierarchies", */T:1},/*::[*/0x00C4/*::]*/:{/* n:"BrtEndPCDHierarchies", */T:-1},/*::[*/0x00C5/*::]*/:{/* n:"BrtBeginPCDHierarchy", */T:1},/*::[*/0x00C6/*::]*/:{/* n:"BrtEndPCDHierarchy", */T:-1},/*::[*/0x00C7/*::]*/:{/* n:"BrtBeginPCDHFieldsUsage", */T:1},/*::[*/0x00C8/*::]*/:{/* n:"BrtEndPCDHFieldsUsage", */T:-1},/*::[*/0x00C9/*::]*/:{/* n:"BrtBeginExtConnection", */T:1},/*::[*/0x00CA/*::]*/:{/* n:"BrtEndExtConnection", */T:-1},/*::[*/0x00CB/*::]*/:{/* n:"BrtBeginECDbProps", */T:1},/*::[*/0x00CC/*::]*/:{/* n:"BrtEndECDbProps", */T:-1},/*::[*/0x00CD/*::]*/:{/* n:"BrtBeginECOlapProps", */T:1},/*::[*/0x00CE/*::]*/:{/* n:"BrtEndECOlapProps", */T:-1},/*::[*/0x00CF/*::]*/:{/* n:"BrtBeginPCDSConsol", */T:1},/*::[*/0x00D0/*::]*/:{/* n:"BrtEndPCDSConsol", */T:-1},/*::[*/0x00D1/*::]*/:{/* n:"BrtBeginPCDSCPages", */T:1},/*::[*/0x00D2/*::]*/:{/* n:"BrtEndPCDSCPages", */T:-1},/*::[*/0x00D3/*::]*/:{/* n:"BrtBeginPCDSCPage", */T:1},/*::[*/0x00D4/*::]*/:{/* n:"BrtEndPCDSCPage", */T:-1},/*::[*/0x00D5/*::]*/:{/* n:"BrtBeginPCDSCPItem", */T:1},/*::[*/0x00D6/*::]*/:{/* n:"BrtEndPCDSCPItem", */T:-1},/*::[*/0x00D7/*::]*/:{/* n:"BrtBeginPCDSCSets", */T:1},/*::[*/0x00D8/*::]*/:{/* n:"BrtEndPCDSCSets", */T:-1},/*::[*/0x00D9/*::]*/:{/* n:"BrtBeginPCDSCSet", */T:1},/*::[*/0x00DA/*::]*/:{/* n:"BrtEndPCDSCSet", */T:-1},/*::[*/0x00DB/*::]*/:{/* n:"BrtBeginPCDFGroup", */T:1},/*::[*/0x00DC/*::]*/:{/* n:"BrtEndPCDFGroup", */T:-1},/*::[*/0x00DD/*::]*/:{/* n:"BrtBeginPCDFGItems", */T:1},/*::[*/0x00DE/*::]*/:{/* n:"BrtEndPCDFGItems", */T:-1},/*::[*/0x00DF/*::]*/:{/* n:"BrtBeginPCDFGRange", */T:1},/*::[*/0x00E0/*::]*/:{/* n:"BrtEndPCDFGRange", */T:-1},/*::[*/0x00E1/*::]*/:{/* n:"BrtBeginPCDFGDiscrete", */T:1},/*::[*/0x00E2/*::]*/:{/* n:"BrtEndPCDFGDiscrete", */T:-1},/*::[*/0x00E3/*::]*/:{/* n:"BrtBeginPCDSDTupleCache", */T:1},/*::[*/0x00E4/*::]*/:{/* n:"BrtEndPCDSDTupleCache", */T:-1},/*::[*/0x00E5/*::]*/:{/* n:"BrtBeginPCDSDTCEntries", */T:1},/*::[*/0x00E6/*::]*/:{/* n:"BrtEndPCDSDTCEntries", */T:-1},/*::[*/0x00E7/*::]*/:{/* n:"BrtBeginPCDSDTCEMembers", */T:1},/*::[*/0x00E8/*::]*/:{/* n:"BrtEndPCDSDTCEMembers", */T:-1},/*::[*/0x00E9/*::]*/:{/* n:"BrtBeginPCDSDTCEMember", */T:1},/*::[*/0x00EA/*::]*/:{/* n:"BrtEndPCDSDTCEMember", */T:-1},/*::[*/0x00EB/*::]*/:{/* n:"BrtBeginPCDSDTCQueries", */T:1},/*::[*/0x00EC/*::]*/:{/* n:"BrtEndPCDSDTCQueries", */T:-1},/*::[*/0x00ED/*::]*/:{/* n:"BrtBeginPCDSDTCQuery", */T:1},/*::[*/0x00EE/*::]*/:{/* n:"BrtEndPCDSDTCQuery", */T:-1},/*::[*/0x00EF/*::]*/:{/* n:"BrtBeginPCDSDTCSets", */T:1},/*::[*/0x00F0/*::]*/:{/* n:"BrtEndPCDSDTCSets", */T:-1},/*::[*/0x00F1/*::]*/:{/* n:"BrtBeginPCDSDTCSet", */T:1},/*::[*/0x00F2/*::]*/:{/* n:"BrtEndPCDSDTCSet", */T:-1},/*::[*/0x00F3/*::]*/:{/* n:"BrtBeginPCDCalcItems", */T:1},/*::[*/0x00F4/*::]*/:{/* n:"BrtEndPCDCalcItems", */T:-1},/*::[*/0x00F5/*::]*/:{/* n:"BrtBeginPCDCalcItem", */T:1},/*::[*/0x00F6/*::]*/:{/* n:"BrtEndPCDCalcItem", */T:-1},/*::[*/0x00F7/*::]*/:{/* n:"BrtBeginPRule", */T:1},/*::[*/0x00F8/*::]*/:{/* n:"BrtEndPRule", */T:-1},/*::[*/0x00F9/*::]*/:{/* n:"BrtBeginPRFilters", */T:1},/*::[*/0x00FA/*::]*/:{/* n:"BrtEndPRFilters", */T:-1},/*::[*/0x00FB/*::]*/:{/* n:"BrtBeginPRFilter", */T:1},/*::[*/0x00FC/*::]*/:{/* n:"BrtEndPRFilter", */T:-1},/*::[*/0x00FD/*::]*/:{/* n:"BrtBeginPNames", */T:1},/*::[*/0x00FE/*::]*/:{/* n:"BrtEndPNames", */T:-1},/*::[*/0x00FF/*::]*/:{/* n:"BrtBeginPName", */T:1},/*::[*/0x0100/*::]*/:{/* n:"BrtEndPName", */T:-1},/*::[*/0x0101/*::]*/:{/* n:"BrtBeginPNPairs", */T:1},/*::[*/0x0102/*::]*/:{/* n:"BrtEndPNPairs", */T:-1},/*::[*/0x0103/*::]*/:{/* n:"BrtBeginPNPair", */T:1},/*::[*/0x0104/*::]*/:{/* n:"BrtEndPNPair", */T:-1},/*::[*/0x0105/*::]*/:{/* n:"BrtBeginECWebProps", */T:1},/*::[*/0x0106/*::]*/:{/* n:"BrtEndECWebProps", */T:-1},/*::[*/0x0107/*::]*/:{/* n:"BrtBeginEcWpTables", */T:1},/*::[*/0x0108/*::]*/:{/* n:"BrtEndECWPTables", */T:-1},/*::[*/0x0109/*::]*/:{/* n:"BrtBeginECParams", */T:1},/*::[*/0x010A/*::]*/:{/* n:"BrtEndECParams", */T:-1},/*::[*/0x010B/*::]*/:{/* n:"BrtBeginECParam", */T:1},/*::[*/0x010C/*::]*/:{/* n:"BrtEndECParam", */T:-1},/*::[*/0x010D/*::]*/:{/* n:"BrtBeginPCDKPIs", */T:1},/*::[*/0x010E/*::]*/:{/* n:"BrtEndPCDKPIs", */T:-1},/*::[*/0x010F/*::]*/:{/* n:"BrtBeginPCDKPI", */T:1},/*::[*/0x0110/*::]*/:{/* n:"BrtEndPCDKPI", */T:-1},/*::[*/0x0111/*::]*/:{/* n:"BrtBeginDims", */T:1},/*::[*/0x0112/*::]*/:{/* n:"BrtEndDims", */T:-1},/*::[*/0x0113/*::]*/:{/* n:"BrtBeginDim", */T:1},/*::[*/0x0114/*::]*/:{/* n:"BrtEndDim", */T:-1},/*::[*/0x0115/*::]*/:{/* n:"BrtIndexPartEnd" */},/*::[*/0x0116/*::]*/:{/* n:"BrtBeginStyleSheet", */T:1},/*::[*/0x0117/*::]*/:{/* n:"BrtEndStyleSheet", */T:-1},/*::[*/0x0118/*::]*/:{/* n:"BrtBeginSXView", */T:1},/*::[*/0x0119/*::]*/:{/* n:"BrtEndSXVI", */T:-1},/*::[*/0x011A/*::]*/:{/* n:"BrtBeginSXVI", */T:1},/*::[*/0x011B/*::]*/:{/* n:"BrtBeginSXVIs", */T:1},/*::[*/0x011C/*::]*/:{/* n:"BrtEndSXVIs", */T:-1},/*::[*/0x011D/*::]*/:{/* n:"BrtBeginSXVD", */T:1},/*::[*/0x011E/*::]*/:{/* n:"BrtEndSXVD", */T:-1},/*::[*/0x011F/*::]*/:{/* n:"BrtBeginSXVDs", */T:1},/*::[*/0x0120/*::]*/:{/* n:"BrtEndSXVDs", */T:-1},/*::[*/0x0121/*::]*/:{/* n:"BrtBeginSXPI", */T:1},/*::[*/0x0122/*::]*/:{/* n:"BrtEndSXPI", */T:-1},/*::[*/0x0123/*::]*/:{/* n:"BrtBeginSXPIs", */T:1},/*::[*/0x0124/*::]*/:{/* n:"BrtEndSXPIs", */T:-1},/*::[*/0x0125/*::]*/:{/* n:"BrtBeginSXDI", */T:1},/*::[*/0x0126/*::]*/:{/* n:"BrtEndSXDI", */T:-1},/*::[*/0x0127/*::]*/:{/* n:"BrtBeginSXDIs", */T:1},/*::[*/0x0128/*::]*/:{/* n:"BrtEndSXDIs", */T:-1},/*::[*/0x0129/*::]*/:{/* n:"BrtBeginSXLI", */T:1},/*::[*/0x012A/*::]*/:{/* n:"BrtEndSXLI", */T:-1},/*::[*/0x012B/*::]*/:{/* n:"BrtBeginSXLIRws", */T:1},/*::[*/0x012C/*::]*/:{/* n:"BrtEndSXLIRws", */T:-1},/*::[*/0x012D/*::]*/:{/* n:"BrtBeginSXLICols", */T:1},/*::[*/0x012E/*::]*/:{/* n:"BrtEndSXLICols", */T:-1},/*::[*/0x012F/*::]*/:{/* n:"BrtBeginSXFormat", */T:1},/*::[*/0x0130/*::]*/:{/* n:"BrtEndSXFormat", */T:-1},/*::[*/0x0131/*::]*/:{/* n:"BrtBeginSXFormats", */T:1},/*::[*/0x0132/*::]*/:{/* n:"BrtEndSxFormats", */T:-1},/*::[*/0x0133/*::]*/:{/* n:"BrtBeginSxSelect", */T:1},/*::[*/0x0134/*::]*/:{/* n:"BrtEndSxSelect", */T:-1},/*::[*/0x0135/*::]*/:{/* n:"BrtBeginISXVDRws", */T:1},/*::[*/0x0136/*::]*/:{/* n:"BrtEndISXVDRws", */T:-1},/*::[*/0x0137/*::]*/:{/* n:"BrtBeginISXVDCols", */T:1},/*::[*/0x0138/*::]*/:{/* n:"BrtEndISXVDCols", */T:-1},/*::[*/0x0139/*::]*/:{/* n:"BrtEndSXLocation", */T:-1},/*::[*/0x013A/*::]*/:{/* n:"BrtBeginSXLocation", */T:1},/*::[*/0x013B/*::]*/:{/* n:"BrtEndSXView", */T:-1},/*::[*/0x013C/*::]*/:{/* n:"BrtBeginSXTHs", */T:1},/*::[*/0x013D/*::]*/:{/* n:"BrtEndSXTHs", */T:-1},/*::[*/0x013E/*::]*/:{/* n:"BrtBeginSXTH", */T:1},/*::[*/0x013F/*::]*/:{/* n:"BrtEndSXTH", */T:-1},/*::[*/0x0140/*::]*/:{/* n:"BrtBeginISXTHRws", */T:1},/*::[*/0x0141/*::]*/:{/* n:"BrtEndISXTHRws", */T:-1},/*::[*/0x0142/*::]*/:{/* n:"BrtBeginISXTHCols", */T:1},/*::[*/0x0143/*::]*/:{/* n:"BrtEndISXTHCols", */T:-1},/*::[*/0x0144/*::]*/:{/* n:"BrtBeginSXTDMPS", */T:1},/*::[*/0x0145/*::]*/:{/* n:"BrtEndSXTDMPs", */T:-1},/*::[*/0x0146/*::]*/:{/* n:"BrtBeginSXTDMP", */T:1},/*::[*/0x0147/*::]*/:{/* n:"BrtEndSXTDMP", */T:-1},/*::[*/0x0148/*::]*/:{/* n:"BrtBeginSXTHItems", */T:1},/*::[*/0x0149/*::]*/:{/* n:"BrtEndSXTHItems", */T:-1},/*::[*/0x014A/*::]*/:{/* n:"BrtBeginSXTHItem", */T:1},/*::[*/0x014B/*::]*/:{/* n:"BrtEndSXTHItem", */T:-1},/*::[*/0x014C/*::]*/:{/* n:"BrtBeginMetadata", */T:1},/*::[*/0x014D/*::]*/:{/* n:"BrtEndMetadata", */T:-1},/*::[*/0x014E/*::]*/:{/* n:"BrtBeginEsmdtinfo", */T:1},/*::[*/0x014F/*::]*/:{/* n:"BrtMdtinfo", */f:parse_BrtMdtinfo},/*::[*/0x0150/*::]*/:{/* n:"BrtEndEsmdtinfo", */T:-1},/*::[*/0x0151/*::]*/:{/* n:"BrtBeginEsmdb", */f:parse_BrtBeginEsmdb,T:1},/*::[*/0x0152/*::]*/:{/* n:"BrtEndEsmdb", */T:-1},/*::[*/0x0153/*::]*/:{/* n:"BrtBeginEsfmd", */T:1},/*::[*/0x0154/*::]*/:{/* n:"BrtEndEsfmd", */T:-1},/*::[*/0x0155/*::]*/:{/* n:"BrtBeginSingleCells", */T:1},/*::[*/0x0156/*::]*/:{/* n:"BrtEndSingleCells", */T:-1},/*::[*/0x0157/*::]*/:{/* n:"BrtBeginList", */T:1},/*::[*/0x0158/*::]*/:{/* n:"BrtEndList", */T:-1},/*::[*/0x0159/*::]*/:{/* n:"BrtBeginListCols", */T:1},/*::[*/0x015A/*::]*/:{/* n:"BrtEndListCols", */T:-1},/*::[*/0x015B/*::]*/:{/* n:"BrtBeginListCol", */T:1},/*::[*/0x015C/*::]*/:{/* n:"BrtEndListCol", */T:-1},/*::[*/0x015D/*::]*/:{/* n:"BrtBeginListXmlCPr", */T:1},/*::[*/0x015E/*::]*/:{/* n:"BrtEndListXmlCPr", */T:-1},/*::[*/0x015F/*::]*/:{/* n:"BrtListCCFmla" */},/*::[*/0x0160/*::]*/:{/* n:"BrtListTrFmla" */},/*::[*/0x0161/*::]*/:{/* n:"BrtBeginExternals", */T:1},/*::[*/0x0162/*::]*/:{/* n:"BrtEndExternals", */T:-1},/*::[*/0x0163/*::]*/:{/* n:"BrtSupBookSrc", */f:parse_RelID},/*::[*/0x0165/*::]*/:{/* n:"BrtSupSelf" */},/*::[*/0x0166/*::]*/:{/* n:"BrtSupSame" */},/*::[*/0x0167/*::]*/:{/* n:"BrtSupTabs" */},/*::[*/0x0168/*::]*/:{/* n:"BrtBeginSupBook", */T:1},/*::[*/0x0169/*::]*/:{/* n:"BrtPlaceholderName" */},/*::[*/0x016A/*::]*/:{/* n:"BrtExternSheet", */f:parse_ExternSheet},/*::[*/0x016B/*::]*/:{/* n:"BrtExternTableStart" */},/*::[*/0x016C/*::]*/:{/* n:"BrtExternTableEnd" */},/*::[*/0x016E/*::]*/:{/* n:"BrtExternRowHdr" */},/*::[*/0x016F/*::]*/:{/* n:"BrtExternCellBlank" */},/*::[*/0x0170/*::]*/:{/* n:"BrtExternCellReal" */},/*::[*/0x0171/*::]*/:{/* n:"BrtExternCellBool" */},/*::[*/0x0172/*::]*/:{/* n:"BrtExternCellError" */},/*::[*/0x0173/*::]*/:{/* n:"BrtExternCellString" */},/*::[*/0x0174/*::]*/:{/* n:"BrtBeginEsmdx", */T:1},/*::[*/0x0175/*::]*/:{/* n:"BrtEndEsmdx", */T:-1},/*::[*/0x0176/*::]*/:{/* n:"BrtBeginMdxSet", */T:1},/*::[*/0x0177/*::]*/:{/* n:"BrtEndMdxSet", */T:-1},/*::[*/0x0178/*::]*/:{/* n:"BrtBeginMdxMbrProp", */T:1},/*::[*/0x0179/*::]*/:{/* n:"BrtEndMdxMbrProp", */T:-1},/*::[*/0x017A/*::]*/:{/* n:"BrtBeginMdxKPI", */T:1},/*::[*/0x017B/*::]*/:{/* n:"BrtEndMdxKPI", */T:-1},/*::[*/0x017C/*::]*/:{/* n:"BrtBeginEsstr", */T:1},/*::[*/0x017D/*::]*/:{/* n:"BrtEndEsstr", */T:-1},/*::[*/0x017E/*::]*/:{/* n:"BrtBeginPRFItem", */T:1},/*::[*/0x017F/*::]*/:{/* n:"BrtEndPRFItem", */T:-1},/*::[*/0x0180/*::]*/:{/* n:"BrtBeginPivotCacheIDs", */T:1},/*::[*/0x0181/*::]*/:{/* n:"BrtEndPivotCacheIDs", */T:-1},/*::[*/0x0182/*::]*/:{/* n:"BrtBeginPivotCacheID", */T:1},/*::[*/0x0183/*::]*/:{/* n:"BrtEndPivotCacheID", */T:-1},/*::[*/0x0184/*::]*/:{/* n:"BrtBeginISXVIs", */T:1},/*::[*/0x0185/*::]*/:{/* n:"BrtEndISXVIs", */T:-1},/*::[*/0x0186/*::]*/:{/* n:"BrtBeginColInfos", */T:1},/*::[*/0x0187/*::]*/:{/* n:"BrtEndColInfos", */T:-1},/*::[*/0x0188/*::]*/:{/* n:"BrtBeginRwBrk", */T:1},/*::[*/0x0189/*::]*/:{/* n:"BrtEndRwBrk", */T:-1},/*::[*/0x018A/*::]*/:{/* n:"BrtBeginColBrk", */T:1},/*::[*/0x018B/*::]*/:{/* n:"BrtEndColBrk", */T:-1},/*::[*/0x018C/*::]*/:{/* n:"BrtBrk" */},/*::[*/0x018D/*::]*/:{/* n:"BrtUserBookView" */},/*::[*/0x018E/*::]*/:{/* n:"BrtInfo" */},/*::[*/0x018F/*::]*/:{/* n:"BrtCUsr" */},/*::[*/0x0190/*::]*/:{/* n:"BrtUsr" */},/*::[*/0x0191/*::]*/:{/* n:"BrtBeginUsers", */T:1},/*::[*/0x0193/*::]*/:{/* n:"BrtEOF" */},/*::[*/0x0194/*::]*/:{/* n:"BrtUCR" */},/*::[*/0x0195/*::]*/:{/* n:"BrtRRInsDel" */},/*::[*/0x0196/*::]*/:{/* n:"BrtRREndInsDel" */},/*::[*/0x0197/*::]*/:{/* n:"BrtRRMove" */},/*::[*/0x0198/*::]*/:{/* n:"BrtRREndMove" */},/*::[*/0x0199/*::]*/:{/* n:"BrtRRChgCell" */},/*::[*/0x019A/*::]*/:{/* n:"BrtRREndChgCell" */},/*::[*/0x019B/*::]*/:{/* n:"BrtRRHeader" */},/*::[*/0x019C/*::]*/:{/* n:"BrtRRUserView" */},/*::[*/0x019D/*::]*/:{/* n:"BrtRRRenSheet" */},/*::[*/0x019E/*::]*/:{/* n:"BrtRRInsertSh" */},/*::[*/0x019F/*::]*/:{/* n:"BrtRRDefName" */},/*::[*/0x01A0/*::]*/:{/* n:"BrtRRNote" */},/*::[*/0x01A1/*::]*/:{/* n:"BrtRRConflict" */},/*::[*/0x01A2/*::]*/:{/* n:"BrtRRTQSIF" */},/*::[*/0x01A3/*::]*/:{/* n:"BrtRRFormat" */},/*::[*/0x01A4/*::]*/:{/* n:"BrtRREndFormat" */},/*::[*/0x01A5/*::]*/:{/* n:"BrtRRAutoFmt" */},/*::[*/0x01A6/*::]*/:{/* n:"BrtBeginUserShViews", */T:1},/*::[*/0x01A7/*::]*/:{/* n:"BrtBeginUserShView", */T:1},/*::[*/0x01A8/*::]*/:{/* n:"BrtEndUserShView", */T:-1},/*::[*/0x01A9/*::]*/:{/* n:"BrtEndUserShViews", */T:-1},/*::[*/0x01AA/*::]*/:{/* n:"BrtArrFmla", */f:parse_BrtArrFmla},/*::[*/0x01AB/*::]*/:{/* n:"BrtShrFmla", */f:parse_BrtShrFmla},/*::[*/0x01AC/*::]*/:{/* n:"BrtTable" */},/*::[*/0x01AD/*::]*/:{/* n:"BrtBeginExtConnections", */T:1},/*::[*/0x01AE/*::]*/:{/* n:"BrtEndExtConnections", */T:-1},/*::[*/0x01AF/*::]*/:{/* n:"BrtBeginPCDCalcMems", */T:1},/*::[*/0x01B0/*::]*/:{/* n:"BrtEndPCDCalcMems", */T:-1},/*::[*/0x01B1/*::]*/:{/* n:"BrtBeginPCDCalcMem", */T:1},/*::[*/0x01B2/*::]*/:{/* n:"BrtEndPCDCalcMem", */T:-1},/*::[*/0x01B3/*::]*/:{/* n:"BrtBeginPCDHGLevels", */T:1},/*::[*/0x01B4/*::]*/:{/* n:"BrtEndPCDHGLevels", */T:-1},/*::[*/0x01B5/*::]*/:{/* n:"BrtBeginPCDHGLevel", */T:1},/*::[*/0x01B6/*::]*/:{/* n:"BrtEndPCDHGLevel", */T:-1},/*::[*/0x01B7/*::]*/:{/* n:"BrtBeginPCDHGLGroups", */T:1},/*::[*/0x01B8/*::]*/:{/* n:"BrtEndPCDHGLGroups", */T:-1},/*::[*/0x01B9/*::]*/:{/* n:"BrtBeginPCDHGLGroup", */T:1},/*::[*/0x01BA/*::]*/:{/* n:"BrtEndPCDHGLGroup", */T:-1},/*::[*/0x01BB/*::]*/:{/* n:"BrtBeginPCDHGLGMembers", */T:1},/*::[*/0x01BC/*::]*/:{/* n:"BrtEndPCDHGLGMembers", */T:-1},/*::[*/0x01BD/*::]*/:{/* n:"BrtBeginPCDHGLGMember", */T:1},/*::[*/0x01BE/*::]*/:{/* n:"BrtEndPCDHGLGMember", */T:-1},/*::[*/0x01BF/*::]*/:{/* n:"BrtBeginQSI", */T:1},/*::[*/0x01C0/*::]*/:{/* n:"BrtEndQSI", */T:-1},/*::[*/0x01C1/*::]*/:{/* n:"BrtBeginQSIR", */T:1},/*::[*/0x01C2/*::]*/:{/* n:"BrtEndQSIR", */T:-1},/*::[*/0x01C3/*::]*/:{/* n:"BrtBeginDeletedNames", */T:1},/*::[*/0x01C4/*::]*/:{/* n:"BrtEndDeletedNames", */T:-1},/*::[*/0x01C5/*::]*/:{/* n:"BrtBeginDeletedName", */T:1},/*::[*/0x01C6/*::]*/:{/* n:"BrtEndDeletedName", */T:-1},/*::[*/0x01C7/*::]*/:{/* n:"BrtBeginQSIFs", */T:1},/*::[*/0x01C8/*::]*/:{/* n:"BrtEndQSIFs", */T:-1},/*::[*/0x01C9/*::]*/:{/* n:"BrtBeginQSIF", */T:1},/*::[*/0x01CA/*::]*/:{/* n:"BrtEndQSIF", */T:-1},/*::[*/0x01CB/*::]*/:{/* n:"BrtBeginAutoSortScope", */T:1},/*::[*/0x01CC/*::]*/:{/* n:"BrtEndAutoSortScope", */T:-1},/*::[*/0x01CD/*::]*/:{/* n:"BrtBeginConditionalFormatting", */T:1},/*::[*/0x01CE/*::]*/:{/* n:"BrtEndConditionalFormatting", */T:-1},/*::[*/0x01CF/*::]*/:{/* n:"BrtBeginCFRule", */T:1},/*::[*/0x01D0/*::]*/:{/* n:"BrtEndCFRule", */T:-1},/*::[*/0x01D1/*::]*/:{/* n:"BrtBeginIconSet", */T:1},/*::[*/0x01D2/*::]*/:{/* n:"BrtEndIconSet", */T:-1},/*::[*/0x01D3/*::]*/:{/* n:"BrtBeginDatabar", */T:1},/*::[*/0x01D4/*::]*/:{/* n:"BrtEndDatabar", */T:-1},/*::[*/0x01D5/*::]*/:{/* n:"BrtBeginColorScale", */T:1},/*::[*/0x01D6/*::]*/:{/* n:"BrtEndColorScale", */T:-1},/*::[*/0x01D7/*::]*/:{/* n:"BrtCFVO" */},/*::[*/0x01D8/*::]*/:{/* n:"BrtExternValueMeta" */},/*::[*/0x01D9/*::]*/:{/* n:"BrtBeginColorPalette", */T:1},/*::[*/0x01DA/*::]*/:{/* n:"BrtEndColorPalette", */T:-1},/*::[*/0x01DB/*::]*/:{/* n:"BrtIndexedColor" */},/*::[*/0x01DC/*::]*/:{/* n:"BrtMargins", */f:parse_BrtMargins},/*::[*/0x01DD/*::]*/:{/* n:"BrtPrintOptions" */},/*::[*/0x01DE/*::]*/:{/* n:"BrtPageSetup" */},/*::[*/0x01DF/*::]*/:{/* n:"BrtBeginHeaderFooter", */T:1},/*::[*/0x01E0/*::]*/:{/* n:"BrtEndHeaderFooter", */T:-1},/*::[*/0x01E1/*::]*/:{/* n:"BrtBeginSXCrtFormat", */T:1},/*::[*/0x01E2/*::]*/:{/* n:"BrtEndSXCrtFormat", */T:-1},/*::[*/0x01E3/*::]*/:{/* n:"BrtBeginSXCrtFormats", */T:1},/*::[*/0x01E4/*::]*/:{/* n:"BrtEndSXCrtFormats", */T:-1},/*::[*/0x01E5/*::]*/:{/* n:"BrtWsFmtInfo", */f:parse_BrtWsFmtInfo},/*::[*/0x01E6/*::]*/:{/* n:"BrtBeginMgs", */T:1},/*::[*/0x01E7/*::]*/:{/* n:"BrtEndMGs", */T:-1},/*::[*/0x01E8/*::]*/:{/* n:"BrtBeginMGMaps", */T:1},/*::[*/0x01E9/*::]*/:{/* n:"BrtEndMGMaps", */T:-1},/*::[*/0x01EA/*::]*/:{/* n:"BrtBeginMG", */T:1},/*::[*/0x01EB/*::]*/:{/* n:"BrtEndMG", */T:-1},/*::[*/0x01EC/*::]*/:{/* n:"BrtBeginMap", */T:1},/*::[*/0x01ED/*::]*/:{/* n:"BrtEndMap", */T:-1},/*::[*/0x01EE/*::]*/:{/* n:"BrtHLink", */f:parse_BrtHLink},/*::[*/0x01EF/*::]*/:{/* n:"BrtBeginDCon", */T:1},/*::[*/0x01F0/*::]*/:{/* n:"BrtEndDCon", */T:-1},/*::[*/0x01F1/*::]*/:{/* n:"BrtBeginDRefs", */T:1},/*::[*/0x01F2/*::]*/:{/* n:"BrtEndDRefs", */T:-1},/*::[*/0x01F3/*::]*/:{/* n:"BrtDRef" */},/*::[*/0x01F4/*::]*/:{/* n:"BrtBeginScenMan", */T:1},/*::[*/0x01F5/*::]*/:{/* n:"BrtEndScenMan", */T:-1},/*::[*/0x01F6/*::]*/:{/* n:"BrtBeginSct", */T:1},/*::[*/0x01F7/*::]*/:{/* n:"BrtEndSct", */T:-1},/*::[*/0x01F8/*::]*/:{/* n:"BrtSlc" */},/*::[*/0x01F9/*::]*/:{/* n:"BrtBeginDXFs", */T:1},/*::[*/0x01FA/*::]*/:{/* n:"BrtEndDXFs", */T:-1},/*::[*/0x01FB/*::]*/:{/* n:"BrtDXF" */},/*::[*/0x01FC/*::]*/:{/* n:"BrtBeginTableStyles", */T:1},/*::[*/0x01FD/*::]*/:{/* n:"BrtEndTableStyles", */T:-1},/*::[*/0x01FE/*::]*/:{/* n:"BrtBeginTableStyle", */T:1},/*::[*/0x01FF/*::]*/:{/* n:"BrtEndTableStyle", */T:-1},/*::[*/0x0200/*::]*/:{/* n:"BrtTableStyleElement" */},/*::[*/0x0201/*::]*/:{/* n:"BrtTableStyleClient" */},/*::[*/0x0202/*::]*/:{/* n:"BrtBeginVolDeps", */T:1},/*::[*/0x0203/*::]*/:{/* n:"BrtEndVolDeps", */T:-1},/*::[*/0x0204/*::]*/:{/* n:"BrtBeginVolType", */T:1},/*::[*/0x0205/*::]*/:{/* n:"BrtEndVolType", */T:-1},/*::[*/0x0206/*::]*/:{/* n:"BrtBeginVolMain", */T:1},/*::[*/0x0207/*::]*/:{/* n:"BrtEndVolMain", */T:-1},/*::[*/0x0208/*::]*/:{/* n:"BrtBeginVolTopic", */T:1},/*::[*/0x0209/*::]*/:{/* n:"BrtEndVolTopic", */T:-1},/*::[*/0x020A/*::]*/:{/* n:"BrtVolSubtopic" */},/*::[*/0x020B/*::]*/:{/* n:"BrtVolRef" */},/*::[*/0x020C/*::]*/:{/* n:"BrtVolNum" */},/*::[*/0x020D/*::]*/:{/* n:"BrtVolErr" */},/*::[*/0x020E/*::]*/:{/* n:"BrtVolStr" */},/*::[*/0x020F/*::]*/:{/* n:"BrtVolBool" */},/*::[*/0x0210/*::]*/:{/* n:"BrtBeginCalcChain$", */T:1},/*::[*/0x0211/*::]*/:{/* n:"BrtEndCalcChain$", */T:-1},/*::[*/0x0212/*::]*/:{/* n:"BrtBeginSortState", */T:1},/*::[*/0x0213/*::]*/:{/* n:"BrtEndSortState", */T:-1},/*::[*/0x0214/*::]*/:{/* n:"BrtBeginSortCond", */T:1},/*::[*/0x0215/*::]*/:{/* n:"BrtEndSortCond", */T:-1},/*::[*/0x0216/*::]*/:{/* n:"BrtBookProtection" */},/*::[*/0x0217/*::]*/:{/* n:"BrtSheetProtection" */},/*::[*/0x0218/*::]*/:{/* n:"BrtRangeProtection" */},/*::[*/0x0219/*::]*/:{/* n:"BrtPhoneticInfo" */},/*::[*/0x021A/*::]*/:{/* n:"BrtBeginECTxtWiz", */T:1},/*::[*/0x021B/*::]*/:{/* n:"BrtEndECTxtWiz", */T:-1},/*::[*/0x021C/*::]*/:{/* n:"BrtBeginECTWFldInfoLst", */T:1},/*::[*/0x021D/*::]*/:{/* n:"BrtEndECTWFldInfoLst", */T:-1},/*::[*/0x021E/*::]*/:{/* n:"BrtBeginECTwFldInfo", */T:1},/*::[*/0x0224/*::]*/:{/* n:"BrtFileSharing" */},/*::[*/0x0225/*::]*/:{/* n:"BrtOleSize" */},/*::[*/0x0226/*::]*/:{/* n:"BrtDrawing", */f:parse_RelID},/*::[*/0x0227/*::]*/:{/* n:"BrtLegacyDrawing" */},/*::[*/0x0228/*::]*/:{/* n:"BrtLegacyDrawingHF" */},/*::[*/0x0229/*::]*/:{/* n:"BrtWebOpt" */},/*::[*/0x022A/*::]*/:{/* n:"BrtBeginWebPubItems", */T:1},/*::[*/0x022B/*::]*/:{/* n:"BrtEndWebPubItems", */T:-1},/*::[*/0x022C/*::]*/:{/* n:"BrtBeginWebPubItem", */T:1},/*::[*/0x022D/*::]*/:{/* n:"BrtEndWebPubItem", */T:-1},/*::[*/0x022E/*::]*/:{/* n:"BrtBeginSXCondFmt", */T:1},/*::[*/0x022F/*::]*/:{/* n:"BrtEndSXCondFmt", */T:-1},/*::[*/0x0230/*::]*/:{/* n:"BrtBeginSXCondFmts", */T:1},/*::[*/0x0231/*::]*/:{/* n:"BrtEndSXCondFmts", */T:-1},/*::[*/0x0232/*::]*/:{/* n:"BrtBkHim" */},/*::[*/0x0234/*::]*/:{/* n:"BrtColor" */},/*::[*/0x0235/*::]*/:{/* n:"BrtBeginIndexedColors", */T:1},/*::[*/0x0236/*::]*/:{/* n:"BrtEndIndexedColors", */T:-1},/*::[*/0x0239/*::]*/:{/* n:"BrtBeginMRUColors", */T:1},/*::[*/0x023A/*::]*/:{/* n:"BrtEndMRUColors", */T:-1},/*::[*/0x023C/*::]*/:{/* n:"BrtMRUColor" */},/*::[*/0x023D/*::]*/:{/* n:"BrtBeginDVals", */T:1},/*::[*/0x023E/*::]*/:{/* n:"BrtEndDVals", */T:-1},/*::[*/0x0241/*::]*/:{/* n:"BrtSupNameStart" */},/*::[*/0x0242/*::]*/:{/* n:"BrtSupNameValueStart" */},/*::[*/0x0243/*::]*/:{/* n:"BrtSupNameValueEnd" */},/*::[*/0x0244/*::]*/:{/* n:"BrtSupNameNum" */},/*::[*/0x0245/*::]*/:{/* n:"BrtSupNameErr" */},/*::[*/0x0246/*::]*/:{/* n:"BrtSupNameSt" */},/*::[*/0x0247/*::]*/:{/* n:"BrtSupNameNil" */},/*::[*/0x0248/*::]*/:{/* n:"BrtSupNameBool" */},/*::[*/0x0249/*::]*/:{/* n:"BrtSupNameFmla" */},/*::[*/0x024A/*::]*/:{/* n:"BrtSupNameBits" */},/*::[*/0x024B/*::]*/:{/* n:"BrtSupNameEnd" */},/*::[*/0x024C/*::]*/:{/* n:"BrtEndSupBook", */T:-1},/*::[*/0x024D/*::]*/:{/* n:"BrtCellSmartTagProperty" */},/*::[*/0x024E/*::]*/:{/* n:"BrtBeginCellSmartTag", */T:1},/*::[*/0x024F/*::]*/:{/* n:"BrtEndCellSmartTag", */T:-1},/*::[*/0x0250/*::]*/:{/* n:"BrtBeginCellSmartTags", */T:1},/*::[*/0x0251/*::]*/:{/* n:"BrtEndCellSmartTags", */T:-1},/*::[*/0x0252/*::]*/:{/* n:"BrtBeginSmartTags", */T:1},/*::[*/0x0253/*::]*/:{/* n:"BrtEndSmartTags", */T:-1},/*::[*/0x0254/*::]*/:{/* n:"BrtSmartTagType" */},/*::[*/0x0255/*::]*/:{/* n:"BrtBeginSmartTagTypes", */T:1},/*::[*/0x0256/*::]*/:{/* n:"BrtEndSmartTagTypes", */T:-1},/*::[*/0x0257/*::]*/:{/* n:"BrtBeginSXFilters", */T:1},/*::[*/0x0258/*::]*/:{/* n:"BrtEndSXFilters", */T:-1},/*::[*/0x0259/*::]*/:{/* n:"BrtBeginSXFILTER", */T:1},/*::[*/0x025A/*::]*/:{/* n:"BrtEndSXFilter", */T:-1},/*::[*/0x025B/*::]*/:{/* n:"BrtBeginFills", */T:1},/*::[*/0x025C/*::]*/:{/* n:"BrtEndFills", */T:-1},/*::[*/0x025D/*::]*/:{/* n:"BrtBeginCellWatches", */T:1},/*::[*/0x025E/*::]*/:{/* n:"BrtEndCellWatches", */T:-1},/*::[*/0x025F/*::]*/:{/* n:"BrtCellWatch" */},/*::[*/0x0260/*::]*/:{/* n:"BrtBeginCRErrs", */T:1},/*::[*/0x0261/*::]*/:{/* n:"BrtEndCRErrs", */T:-1},/*::[*/0x0262/*::]*/:{/* n:"BrtCrashRecErr" */},/*::[*/0x0263/*::]*/:{/* n:"BrtBeginFonts", */T:1},/*::[*/0x0264/*::]*/:{/* n:"BrtEndFonts", */T:-1},/*::[*/0x0265/*::]*/:{/* n:"BrtBeginBorders", */T:1},/*::[*/0x0266/*::]*/:{/* n:"BrtEndBorders", */T:-1},/*::[*/0x0267/*::]*/:{/* n:"BrtBeginFmts", */T:1},/*::[*/0x0268/*::]*/:{/* n:"BrtEndFmts", */T:-1},/*::[*/0x0269/*::]*/:{/* n:"BrtBeginCellXFs", */T:1},/*::[*/0x026A/*::]*/:{/* n:"BrtEndCellXFs", */T:-1},/*::[*/0x026B/*::]*/:{/* n:"BrtBeginStyles", */T:1},/*::[*/0x026C/*::]*/:{/* n:"BrtEndStyles", */T:-1},/*::[*/0x0271/*::]*/:{/* n:"BrtBigName" */},/*::[*/0x0272/*::]*/:{/* n:"BrtBeginCellStyleXFs", */T:1},/*::[*/0x0273/*::]*/:{/* n:"BrtEndCellStyleXFs", */T:-1},/*::[*/0x0274/*::]*/:{/* n:"BrtBeginComments", */T:1},/*::[*/0x0275/*::]*/:{/* n:"BrtEndComments", */T:-1},/*::[*/0x0276/*::]*/:{/* n:"BrtBeginCommentAuthors", */T:1},/*::[*/0x0277/*::]*/:{/* n:"BrtEndCommentAuthors", */T:-1},/*::[*/0x0278/*::]*/:{/* n:"BrtCommentAuthor", */f:parse_BrtCommentAuthor},/*::[*/0x0279/*::]*/:{/* n:"BrtBeginCommentList", */T:1},/*::[*/0x027A/*::]*/:{/* n:"BrtEndCommentList", */T:-1},/*::[*/0x027B/*::]*/:{/* n:"BrtBeginComment", */T:1,f:parse_BrtBeginComment},/*::[*/0x027C/*::]*/:{/* n:"BrtEndComment", */T:-1},/*::[*/0x027D/*::]*/:{/* n:"BrtCommentText", */f:parse_BrtCommentText},/*::[*/0x027E/*::]*/:{/* n:"BrtBeginOleObjects", */T:1},/*::[*/0x027F/*::]*/:{/* n:"BrtOleObject" */},/*::[*/0x0280/*::]*/:{/* n:"BrtEndOleObjects", */T:-1},/*::[*/0x0281/*::]*/:{/* n:"BrtBeginSxrules", */T:1},/*::[*/0x0282/*::]*/:{/* n:"BrtEndSxRules", */T:-1},/*::[*/0x0283/*::]*/:{/* n:"BrtBeginActiveXControls", */T:1},/*::[*/0x0284/*::]*/:{/* n:"BrtActiveX" */},/*::[*/0x0285/*::]*/:{/* n:"BrtEndActiveXControls", */T:-1},/*::[*/0x0286/*::]*/:{/* n:"BrtBeginPCDSDTCEMembersSortBy", */T:1},/*::[*/0x0288/*::]*/:{/* n:"BrtBeginCellIgnoreECs", */T:1},/*::[*/0x0289/*::]*/:{/* n:"BrtCellIgnoreEC" */},/*::[*/0x028A/*::]*/:{/* n:"BrtEndCellIgnoreECs", */T:-1},/*::[*/0x028B/*::]*/:{/* n:"BrtCsProp", */f:parse_BrtCsProp},/*::[*/0x028C/*::]*/:{/* n:"BrtCsPageSetup" */},/*::[*/0x028D/*::]*/:{/* n:"BrtBeginUserCsViews", */T:1},/*::[*/0x028E/*::]*/:{/* n:"BrtEndUserCsViews", */T:-1},/*::[*/0x028F/*::]*/:{/* n:"BrtBeginUserCsView", */T:1},/*::[*/0x0290/*::]*/:{/* n:"BrtEndUserCsView", */T:-1},/*::[*/0x0291/*::]*/:{/* n:"BrtBeginPcdSFCIEntries", */T:1},/*::[*/0x0292/*::]*/:{/* n:"BrtEndPCDSFCIEntries", */T:-1},/*::[*/0x0293/*::]*/:{/* n:"BrtPCDSFCIEntry" */},/*::[*/0x0294/*::]*/:{/* n:"BrtBeginListParts", */T:1},/*::[*/0x0295/*::]*/:{/* n:"BrtListPart" */},/*::[*/0x0296/*::]*/:{/* n:"BrtEndListParts", */T:-1},/*::[*/0x0297/*::]*/:{/* n:"BrtSheetCalcProp" */},/*::[*/0x0298/*::]*/:{/* n:"BrtBeginFnGroup", */T:1},/*::[*/0x0299/*::]*/:{/* n:"BrtFnGroup" */},/*::[*/0x029A/*::]*/:{/* n:"BrtEndFnGroup", */T:-1},/*::[*/0x029B/*::]*/:{/* n:"BrtSupAddin" */},/*::[*/0x029C/*::]*/:{/* n:"BrtSXTDMPOrder" */},/*::[*/0x029D/*::]*/:{/* n:"BrtCsProtection" */},/*::[*/0x029F/*::]*/:{/* n:"BrtBeginWsSortMap", */T:1},/*::[*/0x02A0/*::]*/:{/* n:"BrtEndWsSortMap", */T:-1},/*::[*/0x02A1/*::]*/:{/* n:"BrtBeginRRSort", */T:1},/*::[*/0x02A2/*::]*/:{/* n:"BrtEndRRSort", */T:-1},/*::[*/0x02A3/*::]*/:{/* n:"BrtRRSortItem" */},/*::[*/0x02A4/*::]*/:{/* n:"BrtFileSharingIso" */},/*::[*/0x02A5/*::]*/:{/* n:"BrtBookProtectionIso" */},/*::[*/0x02A6/*::]*/:{/* n:"BrtSheetProtectionIso" */},/*::[*/0x02A7/*::]*/:{/* n:"BrtCsProtectionIso" */},/*::[*/0x02A8/*::]*/:{/* n:"BrtRangeProtectionIso" */},/*::[*/0x02A9/*::]*/:{/* n:"BrtDValList" */},/*::[*/0x0400/*::]*/:{/* n:"BrtRwDescent" */},/*::[*/0x0401/*::]*/:{/* n:"BrtKnownFonts" */},/*::[*/0x0402/*::]*/:{/* n:"BrtBeginSXTupleSet", */T:1},/*::[*/0x0403/*::]*/:{/* n:"BrtEndSXTupleSet", */T:-1},/*::[*/0x0404/*::]*/:{/* n:"BrtBeginSXTupleSetHeader", */T:1},/*::[*/0x0405/*::]*/:{/* n:"BrtEndSXTupleSetHeader", */T:-1},/*::[*/0x0406/*::]*/:{/* n:"BrtSXTupleSetHeaderItem" */},/*::[*/0x0407/*::]*/:{/* n:"BrtBeginSXTupleSetData", */T:1},/*::[*/0x0408/*::]*/:{/* n:"BrtEndSXTupleSetData", */T:-1},/*::[*/0x0409/*::]*/:{/* n:"BrtBeginSXTupleSetRow", */T:1},/*::[*/0x040A/*::]*/:{/* n:"BrtEndSXTupleSetRow", */T:-1},/*::[*/0x040B/*::]*/:{/* n:"BrtSXTupleSetRowItem" */},/*::[*/0x040C/*::]*/:{/* n:"BrtNameExt" */},/*::[*/0x040D/*::]*/:{/* n:"BrtPCDH14" */},/*::[*/0x040E/*::]*/:{/* n:"BrtBeginPCDCalcMem14", */T:1},/*::[*/0x040F/*::]*/:{/* n:"BrtEndPCDCalcMem14", */T:-1},/*::[*/0x0410/*::]*/:{/* n:"BrtSXTH14" */},/*::[*/0x0411/*::]*/:{/* n:"BrtBeginSparklineGroup", */T:1},/*::[*/0x0412/*::]*/:{/* n:"BrtEndSparklineGroup", */T:-1},/*::[*/0x0413/*::]*/:{/* n:"BrtSparkline" */},/*::[*/0x0414/*::]*/:{/* n:"BrtSXDI14" */},/*::[*/0x0415/*::]*/:{/* n:"BrtWsFmtInfoEx14" */},/*::[*/0x0416/*::]*/:{/* n:"BrtBeginConditionalFormatting14", */T:1},/*::[*/0x0417/*::]*/:{/* n:"BrtEndConditionalFormatting14", */T:-1},/*::[*/0x0418/*::]*/:{/* n:"BrtBeginCFRule14", */T:1},/*::[*/0x0419/*::]*/:{/* n:"BrtEndCFRule14", */T:-1},/*::[*/0x041A/*::]*/:{/* n:"BrtCFVO14" */},/*::[*/0x041B/*::]*/:{/* n:"BrtBeginDatabar14", */T:1},/*::[*/0x041C/*::]*/:{/* n:"BrtBeginIconSet14", */T:1},/*::[*/0x041D/*::]*/:{/* n:"BrtDVal14", */f:parse_BrtDVal14},/*::[*/0x041E/*::]*/:{/* n:"BrtBeginDVals14", */T:1},/*::[*/0x041F/*::]*/:{/* n:"BrtColor14" */},/*::[*/0x0420/*::]*/:{/* n:"BrtBeginSparklines", */T:1},/*::[*/0x0421/*::]*/:{/* n:"BrtEndSparklines", */T:-1},/*::[*/0x0422/*::]*/:{/* n:"BrtBeginSparklineGroups", */T:1},/*::[*/0x0423/*::]*/:{/* n:"BrtEndSparklineGroups", */T:-1},/*::[*/0x0425/*::]*/:{/* n:"BrtSXVD14" */},/*::[*/0x0426/*::]*/:{/* n:"BrtBeginSXView14", */T:1},/*::[*/0x0427/*::]*/:{/* n:"BrtEndSXView14", */T:-1},/*::[*/0x0428/*::]*/:{/* n:"BrtBeginSXView16", */T:1},/*::[*/0x0429/*::]*/:{/* n:"BrtEndSXView16", */T:-1},/*::[*/0x042A/*::]*/:{/* n:"BrtBeginPCD14", */T:1},/*::[*/0x042B/*::]*/:{/* n:"BrtEndPCD14", */T:-1},/*::[*/0x042C/*::]*/:{/* n:"BrtBeginExtConn14", */T:1},/*::[*/0x042D/*::]*/:{/* n:"BrtEndExtConn14", */T:-1},/*::[*/0x042E/*::]*/:{/* n:"BrtBeginSlicerCacheIDs", */T:1},/*::[*/0x042F/*::]*/:{/* n:"BrtEndSlicerCacheIDs", */T:-1},/*::[*/0x0430/*::]*/:{/* n:"BrtBeginSlicerCacheID", */T:1},/*::[*/0x0431/*::]*/:{/* n:"BrtEndSlicerCacheID", */T:-1},/*::[*/0x0433/*::]*/:{/* n:"BrtBeginSlicerCache", */T:1},/*::[*/0x0434/*::]*/:{/* n:"BrtEndSlicerCache", */T:-1},/*::[*/0x0435/*::]*/:{/* n:"BrtBeginSlicerCacheDef", */T:1},/*::[*/0x0436/*::]*/:{/* n:"BrtEndSlicerCacheDef", */T:-1},/*::[*/0x0437/*::]*/:{/* n:"BrtBeginSlicersEx", */T:1},/*::[*/0x0438/*::]*/:{/* n:"BrtEndSlicersEx", */T:-1},/*::[*/0x0439/*::]*/:{/* n:"BrtBeginSlicerEx", */T:1},/*::[*/0x043A/*::]*/:{/* n:"BrtEndSlicerEx", */T:-1},/*::[*/0x043B/*::]*/:{/* n:"BrtBeginSlicer", */T:1},/*::[*/0x043C/*::]*/:{/* n:"BrtEndSlicer", */T:-1},/*::[*/0x043D/*::]*/:{/* n:"BrtSlicerCachePivotTables" */},/*::[*/0x043E/*::]*/:{/* n:"BrtBeginSlicerCacheOlapImpl", */T:1},/*::[*/0x043F/*::]*/:{/* n:"BrtEndSlicerCacheOlapImpl", */T:-1},/*::[*/0x0440/*::]*/:{/* n:"BrtBeginSlicerCacheLevelsData", */T:1},/*::[*/0x0441/*::]*/:{/* n:"BrtEndSlicerCacheLevelsData", */T:-1},/*::[*/0x0442/*::]*/:{/* n:"BrtBeginSlicerCacheLevelData", */T:1},/*::[*/0x0443/*::]*/:{/* n:"BrtEndSlicerCacheLevelData", */T:-1},/*::[*/0x0444/*::]*/:{/* n:"BrtBeginSlicerCacheSiRanges", */T:1},/*::[*/0x0445/*::]*/:{/* n:"BrtEndSlicerCacheSiRanges", */T:-1},/*::[*/0x0446/*::]*/:{/* n:"BrtBeginSlicerCacheSiRange", */T:1},/*::[*/0x0447/*::]*/:{/* n:"BrtEndSlicerCacheSiRange", */T:-1},/*::[*/0x0448/*::]*/:{/* n:"BrtSlicerCacheOlapItem" */},/*::[*/0x0449/*::]*/:{/* n:"BrtBeginSlicerCacheSelections", */T:1},/*::[*/0x044A/*::]*/:{/* n:"BrtSlicerCacheSelection" */},/*::[*/0x044B/*::]*/:{/* n:"BrtEndSlicerCacheSelections", */T:-1},/*::[*/0x044C/*::]*/:{/* n:"BrtBeginSlicerCacheNative", */T:1},/*::[*/0x044D/*::]*/:{/* n:"BrtEndSlicerCacheNative", */T:-1},/*::[*/0x044E/*::]*/:{/* n:"BrtSlicerCacheNativeItem" */},/*::[*/0x044F/*::]*/:{/* n:"BrtRangeProtection14" */},/*::[*/0x0450/*::]*/:{/* n:"BrtRangeProtectionIso14" */},/*::[*/0x0451/*::]*/:{/* n:"BrtCellIgnoreEC14" */},/*::[*/0x0457/*::]*/:{/* n:"BrtList14" */},/*::[*/0x0458/*::]*/:{/* n:"BrtCFIcon" */},/*::[*/0x0459/*::]*/:{/* n:"BrtBeginSlicerCachesPivotCacheIDs", */T:1},/*::[*/0x045A/*::]*/:{/* n:"BrtEndSlicerCachesPivotCacheIDs", */T:-1},/*::[*/0x045B/*::]*/:{/* n:"BrtBeginSlicers", */T:1},/*::[*/0x045C/*::]*/:{/* n:"BrtEndSlicers", */T:-1},/*::[*/0x045D/*::]*/:{/* n:"BrtWbProp14" */},/*::[*/0x045E/*::]*/:{/* n:"BrtBeginSXEdit", */T:1},/*::[*/0x045F/*::]*/:{/* n:"BrtEndSXEdit", */T:-1},/*::[*/0x0460/*::]*/:{/* n:"BrtBeginSXEdits", */T:1},/*::[*/0x0461/*::]*/:{/* n:"BrtEndSXEdits", */T:-1},/*::[*/0x0462/*::]*/:{/* n:"BrtBeginSXChange", */T:1},/*::[*/0x0463/*::]*/:{/* n:"BrtEndSXChange", */T:-1},/*::[*/0x0464/*::]*/:{/* n:"BrtBeginSXChanges", */T:1},/*::[*/0x0465/*::]*/:{/* n:"BrtEndSXChanges", */T:-1},/*::[*/0x0466/*::]*/:{/* n:"BrtSXTupleItems" */},/*::[*/0x0468/*::]*/:{/* n:"BrtBeginSlicerStyle", */T:1},/*::[*/0x0469/*::]*/:{/* n:"BrtEndSlicerStyle", */T:-1},/*::[*/0x046A/*::]*/:{/* n:"BrtSlicerStyleElement" */},/*::[*/0x046B/*::]*/:{/* n:"BrtBeginStyleSheetExt14", */T:1},/*::[*/0x046C/*::]*/:{/* n:"BrtEndStyleSheetExt14", */T:-1},/*::[*/0x046D/*::]*/:{/* n:"BrtBeginSlicerCachesPivotCacheID", */T:1},/*::[*/0x046E/*::]*/:{/* n:"BrtEndSlicerCachesPivotCacheID", */T:-1},/*::[*/0x046F/*::]*/:{/* n:"BrtBeginConditionalFormattings", */T:1},/*::[*/0x0470/*::]*/:{/* n:"BrtEndConditionalFormattings", */T:-1},/*::[*/0x0471/*::]*/:{/* n:"BrtBeginPCDCalcMemExt", */T:1},/*::[*/0x0472/*::]*/:{/* n:"BrtEndPCDCalcMemExt", */T:-1},/*::[*/0x0473/*::]*/:{/* n:"BrtBeginPCDCalcMemsExt", */T:1},/*::[*/0x0474/*::]*/:{/* n:"BrtEndPCDCalcMemsExt", */T:-1},/*::[*/0x0475/*::]*/:{/* n:"BrtPCDField14" */},/*::[*/0x0476/*::]*/:{/* n:"BrtBeginSlicerStyles", */T:1},/*::[*/0x0477/*::]*/:{/* n:"BrtEndSlicerStyles", */T:-1},/*::[*/0x0478/*::]*/:{/* n:"BrtBeginSlicerStyleElements", */T:1},/*::[*/0x0479/*::]*/:{/* n:"BrtEndSlicerStyleElements", */T:-1},/*::[*/0x047A/*::]*/:{/* n:"BrtCFRuleExt" */},/*::[*/0x047B/*::]*/:{/* n:"BrtBeginSXCondFmt14", */T:1},/*::[*/0x047C/*::]*/:{/* n:"BrtEndSXCondFmt14", */T:-1},/*::[*/0x047D/*::]*/:{/* n:"BrtBeginSXCondFmts14", */T:1},/*::[*/0x047E/*::]*/:{/* n:"BrtEndSXCondFmts14", */T:-1},/*::[*/0x0480/*::]*/:{/* n:"BrtBeginSortCond14", */T:1},/*::[*/0x0481/*::]*/:{/* n:"BrtEndSortCond14", */T:-1},/*::[*/0x0482/*::]*/:{/* n:"BrtEndDVals14", */T:-1},/*::[*/0x0483/*::]*/:{/* n:"BrtEndIconSet14", */T:-1},/*::[*/0x0484/*::]*/:{/* n:"BrtEndDatabar14", */T:-1},/*::[*/0x0485/*::]*/:{/* n:"BrtBeginColorScale14", */T:1},/*::[*/0x0486/*::]*/:{/* n:"BrtEndColorScale14", */T:-1},/*::[*/0x0487/*::]*/:{/* n:"BrtBeginSxrules14", */T:1},/*::[*/0x0488/*::]*/:{/* n:"BrtEndSxrules14", */T:-1},/*::[*/0x0489/*::]*/:{/* n:"BrtBeginPRule14", */T:1},/*::[*/0x048A/*::]*/:{/* n:"BrtEndPRule14", */T:-1},/*::[*/0x048B/*::]*/:{/* n:"BrtBeginPRFilters14", */T:1},/*::[*/0x048C/*::]*/:{/* n:"BrtEndPRFilters14", */T:-1},/*::[*/0x048D/*::]*/:{/* n:"BrtBeginPRFilter14", */T:1},/*::[*/0x048E/*::]*/:{/* n:"BrtEndPRFilter14", */T:-1},/*::[*/0x048F/*::]*/:{/* n:"BrtBeginPRFItem14", */T:1},/*::[*/0x0490/*::]*/:{/* n:"BrtEndPRFItem14", */T:-1},/*::[*/0x0491/*::]*/:{/* n:"BrtBeginCellIgnoreECs14", */T:1},/*::[*/0x0492/*::]*/:{/* n:"BrtEndCellIgnoreECs14", */T:-1},/*::[*/0x0493/*::]*/:{/* n:"BrtDxf14" */},/*::[*/0x0494/*::]*/:{/* n:"BrtBeginDxF14s", */T:1},/*::[*/0x0495/*::]*/:{/* n:"BrtEndDxf14s", */T:-1},/*::[*/0x0499/*::]*/:{/* n:"BrtFilter14" */},/*::[*/0x049A/*::]*/:{/* n:"BrtBeginCustomFilters14", */T:1},/*::[*/0x049C/*::]*/:{/* n:"BrtCustomFilter14" */},/*::[*/0x049D/*::]*/:{/* n:"BrtIconFilter14" */},/*::[*/0x049E/*::]*/:{/* n:"BrtPivotCacheConnectionName" */},/*::[*/0x0800/*::]*/:{/* n:"BrtBeginDecoupledPivotCacheIDs", */T:1},/*::[*/0x0801/*::]*/:{/* n:"BrtEndDecoupledPivotCacheIDs", */T:-1},/*::[*/0x0802/*::]*/:{/* n:"BrtDecoupledPivotCacheID" */},/*::[*/0x0803/*::]*/:{/* n:"BrtBeginPivotTableRefs", */T:1},/*::[*/0x0804/*::]*/:{/* n:"BrtEndPivotTableRefs", */T:-1},/*::[*/0x0805/*::]*/:{/* n:"BrtPivotTableRef" */},/*::[*/0x0806/*::]*/:{/* n:"BrtSlicerCacheBookPivotTables" */},/*::[*/0x0807/*::]*/:{/* n:"BrtBeginSxvcells", */T:1},/*::[*/0x0808/*::]*/:{/* n:"BrtEndSxvcells", */T:-1},/*::[*/0x0809/*::]*/:{/* n:"BrtBeginSxRow", */T:1},/*::[*/0x080A/*::]*/:{/* n:"BrtEndSxRow", */T:-1},/*::[*/0x080C/*::]*/:{/* n:"BrtPcdCalcMem15" */},/*::[*/0x0813/*::]*/:{/* n:"BrtQsi15" */},/*::[*/0x0814/*::]*/:{/* n:"BrtBeginWebExtensions", */T:1},/*::[*/0x0815/*::]*/:{/* n:"BrtEndWebExtensions", */T:-1},/*::[*/0x0816/*::]*/:{/* n:"BrtWebExtension" */},/*::[*/0x0817/*::]*/:{/* n:"BrtAbsPath15" */},/*::[*/0x0818/*::]*/:{/* n:"BrtBeginPivotTableUISettings", */T:1},/*::[*/0x0819/*::]*/:{/* n:"BrtEndPivotTableUISettings", */T:-1},/*::[*/0x081B/*::]*/:{/* n:"BrtTableSlicerCacheIDs" */},/*::[*/0x081C/*::]*/:{/* n:"BrtTableSlicerCacheID" */},/*::[*/0x081D/*::]*/:{/* n:"BrtBeginTableSlicerCache", */T:1},/*::[*/0x081E/*::]*/:{/* n:"BrtEndTableSlicerCache", */T:-1},/*::[*/0x081F/*::]*/:{/* n:"BrtSxFilter15" */},/*::[*/0x0820/*::]*/:{/* n:"BrtBeginTimelineCachePivotCacheIDs", */T:1},/*::[*/0x0821/*::]*/:{/* n:"BrtEndTimelineCachePivotCacheIDs", */T:-1},/*::[*/0x0822/*::]*/:{/* n:"BrtTimelineCachePivotCacheID" */},/*::[*/0x0823/*::]*/:{/* n:"BrtBeginTimelineCacheIDs", */T:1},/*::[*/0x0824/*::]*/:{/* n:"BrtEndTimelineCacheIDs", */T:-1},/*::[*/0x0825/*::]*/:{/* n:"BrtBeginTimelineCacheID", */T:1},/*::[*/0x0826/*::]*/:{/* n:"BrtEndTimelineCacheID", */T:-1},/*::[*/0x0827/*::]*/:{/* n:"BrtBeginTimelinesEx", */T:1},/*::[*/0x0828/*::]*/:{/* n:"BrtEndTimelinesEx", */T:-1},/*::[*/0x0829/*::]*/:{/* n:"BrtBeginTimelineEx", */T:1},/*::[*/0x082A/*::]*/:{/* n:"BrtEndTimelineEx", */T:-1},/*::[*/0x082B/*::]*/:{/* n:"BrtWorkBookPr15" */},/*::[*/0x082C/*::]*/:{/* n:"BrtPCDH15" */},/*::[*/0x082D/*::]*/:{/* n:"BrtBeginTimelineStyle", */T:1},/*::[*/0x082E/*::]*/:{/* n:"BrtEndTimelineStyle", */T:-1},/*::[*/0x082F/*::]*/:{/* n:"BrtTimelineStyleElement" */},/*::[*/0x0830/*::]*/:{/* n:"BrtBeginTimelineStylesheetExt15", */T:1},/*::[*/0x0831/*::]*/:{/* n:"BrtEndTimelineStylesheetExt15", */T:-1},/*::[*/0x0832/*::]*/:{/* n:"BrtBeginTimelineStyles", */T:1},/*::[*/0x0833/*::]*/:{/* n:"BrtEndTimelineStyles", */T:-1},/*::[*/0x0834/*::]*/:{/* n:"BrtBeginTimelineStyleElements", */T:1},/*::[*/0x0835/*::]*/:{/* n:"BrtEndTimelineStyleElements", */T:-1},/*::[*/0x0836/*::]*/:{/* n:"BrtDxf15" */},/*::[*/0x0837/*::]*/:{/* n:"BrtBeginDxfs15", */T:1},/*::[*/0x0838/*::]*/:{/* n:"BrtEndDxfs15", */T:-1},/*::[*/0x0839/*::]*/:{/* n:"BrtSlicerCacheHideItemsWithNoData" */},/*::[*/0x083A/*::]*/:{/* n:"BrtBeginItemUniqueNames", */T:1},/*::[*/0x083B/*::]*/:{/* n:"BrtEndItemUniqueNames", */T:-1},/*::[*/0x083C/*::]*/:{/* n:"BrtItemUniqueName" */},/*::[*/0x083D/*::]*/:{/* n:"BrtBeginExtConn15", */T:1},/*::[*/0x083E/*::]*/:{/* n:"BrtEndExtConn15", */T:-1},/*::[*/0x083F/*::]*/:{/* n:"BrtBeginOledbPr15", */T:1},/*::[*/0x0840/*::]*/:{/* n:"BrtEndOledbPr15", */T:-1},/*::[*/0x0841/*::]*/:{/* n:"BrtBeginDataFeedPr15", */T:1},/*::[*/0x0842/*::]*/:{/* n:"BrtEndDataFeedPr15", */T:-1},/*::[*/0x0843/*::]*/:{/* n:"BrtTextPr15" */},/*::[*/0x0844/*::]*/:{/* n:"BrtRangePr15" */},/*::[*/0x0845/*::]*/:{/* n:"BrtDbCommand15" */},/*::[*/0x0846/*::]*/:{/* n:"BrtBeginDbTables15", */T:1},/*::[*/0x0847/*::]*/:{/* n:"BrtEndDbTables15", */T:-1},/*::[*/0x0848/*::]*/:{/* n:"BrtDbTable15" */},/*::[*/0x0849/*::]*/:{/* n:"BrtBeginDataModel", */T:1},/*::[*/0x084A/*::]*/:{/* n:"BrtEndDataModel", */T:-1},/*::[*/0x084B/*::]*/:{/* n:"BrtBeginModelTables", */T:1},/*::[*/0x084C/*::]*/:{/* n:"BrtEndModelTables", */T:-1},/*::[*/0x084D/*::]*/:{/* n:"BrtModelTable" */},/*::[*/0x084E/*::]*/:{/* n:"BrtBeginModelRelationships", */T:1},/*::[*/0x084F/*::]*/:{/* n:"BrtEndModelRelationships", */T:-1},/*::[*/0x0850/*::]*/:{/* n:"BrtModelRelationship" */},/*::[*/0x0851/*::]*/:{/* n:"BrtBeginECTxtWiz15", */T:1},/*::[*/0x0852/*::]*/:{/* n:"BrtEndECTxtWiz15", */T:-1},/*::[*/0x0853/*::]*/:{/* n:"BrtBeginECTWFldInfoLst15", */T:1},/*::[*/0x0854/*::]*/:{/* n:"BrtEndECTWFldInfoLst15", */T:-1},/*::[*/0x0855/*::]*/:{/* n:"BrtBeginECTWFldInfo15", */T:1},/*::[*/0x0856/*::]*/:{/* n:"BrtFieldListActiveItem" */},/*::[*/0x0857/*::]*/:{/* n:"BrtPivotCacheIdVersion" */},/*::[*/0x0858/*::]*/:{/* n:"BrtSXDI15" */},/*::[*/0x0859/*::]*/:{/* n:"BrtBeginModelTimeGroupings", */T:1},/*::[*/0x085A/*::]*/:{/* n:"BrtEndModelTimeGroupings", */T:-1},/*::[*/0x085B/*::]*/:{/* n:"BrtBeginModelTimeGrouping", */T:1},/*::[*/0x085C/*::]*/:{/* n:"BrtEndModelTimeGrouping", */T:-1},/*::[*/0x085D/*::]*/:{/* n:"BrtModelTimeGroupingCalcCol" */},/*::[*/0x0C00/*::]*/:{/* n:"BrtUid" */},/*::[*/0x0C01/*::]*/:{/* n:"BrtRevisionPtr" */},/*::[*/0x1000/*::]*/:{/* n:"BrtBeginDynamicArrayPr", */T:1},/*::[*/0x1001/*::]*/:{/* n:"BrtEndDynamicArrayPr", */T:-1},/*::[*/0x138A/*::]*/:{/* n:"BrtBeginRichValueBlock", */T:1},/*::[*/0x138B/*::]*/:{/* n:"BrtEndRichValueBlock", */T:-1},/*::[*/0x13D9/*::]*/:{/* n:"BrtBeginRichFilters", */T:1},/*::[*/0x13DA/*::]*/:{/* n:"BrtEndRichFilters", */T:-1},/*::[*/0x13DB/*::]*/:{/* n:"BrtRichFilter" */},/*::[*/0x13DC/*::]*/:{/* n:"BrtBeginRichFilterColumn", */T:1},/*::[*/0x13DD/*::]*/:{/* n:"BrtEndRichFilterColumn", */T:-1},/*::[*/0x13DE/*::]*/:{/* n:"BrtBeginCustomRichFilters", */T:1},/*::[*/0x13DF/*::]*/:{/* n:"BrtEndCustomRichFilters", */T:-1},/*::[*/0x13E0/*::]*/:{/* n:"BrtCustomRichFilter" */},/*::[*/0x13E1/*::]*/:{/* n:"BrtTop10RichFilter" */},/*::[*/0x13E2/*::]*/:{/* n:"BrtDynamicRichFilter" */},/*::[*/0x13E4/*::]*/:{/* n:"BrtBeginRichSortCondition", */T:1},/*::[*/0x13E5/*::]*/:{/* n:"BrtEndRichSortCondition", */T:-1},/*::[*/0x13E6/*::]*/:{/* n:"BrtRichFilterDateGroupItem" */},/*::[*/0x13E7/*::]*/:{/* n:"BrtBeginCalcFeatures", */T:1},/*::[*/0x13E8/*::]*/:{/* n:"BrtEndCalcFeatures", */T:-1},/*::[*/0x13E9/*::]*/:{/* n:"BrtCalcFeature" */},/*::[*/0x13EB/*::]*/:{/* n:"BrtExternalLinksPr" */},/*::[*/0xFFFF/*::]*/:{n:""}};/* [MS-XLS] 2.3 Record Enumeration (and other sources) */var XLSRecordEnum={/* [MS-XLS] 2.3 Record Enumeration 2021-08-17 */ /*::[*/0x0006/*::]*/:{/* n:"Formula", */f:parse_Formula},/*::[*/0x000a/*::]*/:{/* n:"EOF", */f:parsenoop2},/*::[*/0x000c/*::]*/:{/* n:"CalcCount", */f:parseuint16},//
	/*::[*/0x000d/*::]*/:{/* n:"CalcMode", */f:parseuint16},//
	/*::[*/0x000e/*::]*/:{/* n:"CalcPrecision", */f:parsebool},//
	/*::[*/0x000f/*::]*/:{/* n:"CalcRefMode", */f:parsebool},//
	/*::[*/0x0010/*::]*/:{/* n:"CalcDelta", */f:parse_Xnum},//
	/*::[*/0x0011/*::]*/:{/* n:"CalcIter", */f:parsebool},//
	/*::[*/0x0012/*::]*/:{/* n:"Protect", */f:parsebool},/*::[*/0x0013/*::]*/:{/* n:"Password", */f:parseuint16},/*::[*/0x0014/*::]*/:{/* n:"Header", */f:parse_XLHeaderFooter},/*::[*/0x0015/*::]*/:{/* n:"Footer", */f:parse_XLHeaderFooter},/*::[*/0x0017/*::]*/:{/* n:"ExternSheet", */f:parse_ExternSheet},/*::[*/0x0018/*::]*/:{/* n:"Lbl", */f:parse_Lbl},/*::[*/0x0019/*::]*/:{/* n:"WinProtect", */f:parsebool},/*::[*/0x001a/*::]*/:{/* n:"VerticalPageBreaks", */},/*::[*/0x001b/*::]*/:{/* n:"HorizontalPageBreaks", */},/*::[*/0x001c/*::]*/:{/* n:"Note", */f:parse_Note},/*::[*/0x001d/*::]*/:{/* n:"Selection", */},/*::[*/0x0022/*::]*/:{/* n:"Date1904", */f:parsebool},/*::[*/0x0023/*::]*/:{/* n:"ExternName", */f:parse_ExternName},/*::[*/0x0026/*::]*/:{/* n:"LeftMargin", */f:parse_Xnum},// *
	/*::[*/0x0027/*::]*/:{/* n:"RightMargin", */f:parse_Xnum},// *
	/*::[*/0x0028/*::]*/:{/* n:"TopMargin", */f:parse_Xnum},// *
	/*::[*/0x0029/*::]*/:{/* n:"BottomMargin", */f:parse_Xnum},// *
	/*::[*/0x002a/*::]*/:{/* n:"PrintRowCol", */f:parsebool},/*::[*/0x002b/*::]*/:{/* n:"PrintGrid", */f:parsebool},/*::[*/0x002f/*::]*/:{/* n:"FilePass", */f:parse_FilePass},/*::[*/0x0031/*::]*/:{/* n:"Font", */f:parse_Font},/*::[*/0x0033/*::]*/:{/* n:"PrintSize", */f:parseuint16},/*::[*/0x003c/*::]*/:{/* n:"Continue", */},/*::[*/0x003d/*::]*/:{/* n:"Window1", */f:parse_Window1},/*::[*/0x0040/*::]*/:{/* n:"Backup", */f:parsebool},/*::[*/0x0041/*::]*/:{/* n:"Pane", */f:parse_Pane},/*::[*/0x0042/*::]*/:{/* n:"CodePage", */f:parseuint16},/*::[*/0x004d/*::]*/:{/* n:"Pls", */},/*::[*/0x0050/*::]*/:{/* n:"DCon", */},/*::[*/0x0051/*::]*/:{/* n:"DConRef", */},/*::[*/0x0052/*::]*/:{/* n:"DConName", */},/*::[*/0x0055/*::]*/:{/* n:"DefColWidth", */f:parseuint16},/*::[*/0x0059/*::]*/:{/* n:"XCT", */},/*::[*/0x005a/*::]*/:{/* n:"CRN", */},/*::[*/0x005b/*::]*/:{/* n:"FileSharing", */},/*::[*/0x005c/*::]*/:{/* n:"WriteAccess", */f:parse_WriteAccess},/*::[*/0x005d/*::]*/:{/* n:"Obj", */f:parse_Obj},/*::[*/0x005e/*::]*/:{/* n:"Uncalced", */},/*::[*/0x005f/*::]*/:{/* n:"CalcSaveRecalc", */f:parsebool},//
	/*::[*/0x0060/*::]*/:{/* n:"Template", */},/*::[*/0x0061/*::]*/:{/* n:"Intl", */},/*::[*/0x0063/*::]*/:{/* n:"ObjProtect", */f:parsebool},/*::[*/0x007d/*::]*/:{/* n:"ColInfo", */f:parse_ColInfo},/*::[*/0x0080/*::]*/:{/* n:"Guts", */f:parse_Guts},/*::[*/0x0081/*::]*/:{/* n:"WsBool", */f:parse_WsBool},/*::[*/0x0082/*::]*/:{/* n:"GridSet", */f:parseuint16},/*::[*/0x0083/*::]*/:{/* n:"HCenter", */f:parsebool},/*::[*/0x0084/*::]*/:{/* n:"VCenter", */f:parsebool},/*::[*/0x0085/*::]*/:{/* n:"BoundSheet8", */f:parse_BoundSheet8},/*::[*/0x0086/*::]*/:{/* n:"WriteProtect", */},/*::[*/0x008c/*::]*/:{/* n:"Country", */f:parse_Country},/*::[*/0x008d/*::]*/:{/* n:"HideObj", */f:parseuint16},/*::[*/0x0090/*::]*/:{/* n:"Sort", */},/*::[*/0x0092/*::]*/:{/* n:"Palette", */f:parse_Palette},/*::[*/0x0097/*::]*/:{/* n:"Sync", */},/*::[*/0x0098/*::]*/:{/* n:"LPr", */},/*::[*/0x0099/*::]*/:{/* n:"DxGCol", */},/*::[*/0x009a/*::]*/:{/* n:"FnGroupName", */},/*::[*/0x009b/*::]*/:{/* n:"FilterMode", */},/*::[*/0x009c/*::]*/:{/* n:"BuiltInFnGroupCount", */f:parseuint16},/*::[*/0x009d/*::]*/:{/* n:"AutoFilterInfo", */},/*::[*/0x009e/*::]*/:{/* n:"AutoFilter", */},/*::[*/0x00a0/*::]*/:{/* n:"Scl", */f:parse_Scl},/*::[*/0x00a1/*::]*/:{/* n:"Setup", */f:parse_Setup},/*::[*/0x00ae/*::]*/:{/* n:"ScenMan", */},/*::[*/0x00af/*::]*/:{/* n:"SCENARIO", */},/*::[*/0x00b0/*::]*/:{/* n:"SxView", */},/*::[*/0x00b1/*::]*/:{/* n:"Sxvd", */},/*::[*/0x00b2/*::]*/:{/* n:"SXVI", */},/*::[*/0x00b4/*::]*/:{/* n:"SxIvd", */},/*::[*/0x00b5/*::]*/:{/* n:"SXLI", */},/*::[*/0x00b6/*::]*/:{/* n:"SXPI", */},/*::[*/0x00b8/*::]*/:{/* n:"DocRoute", */},/*::[*/0x00b9/*::]*/:{/* n:"RecipName", */},/*::[*/0x00bd/*::]*/:{/* n:"MulRk", */f:parse_MulRk},/*::[*/0x00be/*::]*/:{/* n:"MulBlank", */f:parse_MulBlank},/*::[*/0x00c1/*::]*/:{/* n:"Mms", */f:parsenoop2},/*::[*/0x00c5/*::]*/:{/* n:"SXDI", */},/*::[*/0x00c6/*::]*/:{/* n:"SXDB", */},/*::[*/0x00c7/*::]*/:{/* n:"SXFDB", */},/*::[*/0x00c8/*::]*/:{/* n:"SXDBB", */},/*::[*/0x00c9/*::]*/:{/* n:"SXNum", */},/*::[*/0x00ca/*::]*/:{/* n:"SxBool", */f:parsebool},/*::[*/0x00cb/*::]*/:{/* n:"SxErr", */},/*::[*/0x00cc/*::]*/:{/* n:"SXInt", */},/*::[*/0x00cd/*::]*/:{/* n:"SXString", */},/*::[*/0x00ce/*::]*/:{/* n:"SXDtr", */},/*::[*/0x00cf/*::]*/:{/* n:"SxNil", */},/*::[*/0x00d0/*::]*/:{/* n:"SXTbl", */},/*::[*/0x00d1/*::]*/:{/* n:"SXTBRGIITM", */},/*::[*/0x00d2/*::]*/:{/* n:"SxTbpg", */},/*::[*/0x00d3/*::]*/:{/* n:"ObProj", */},/*::[*/0x00d5/*::]*/:{/* n:"SXStreamID", */},/*::[*/0x00d7/*::]*/:{/* n:"DBCell", */},/*::[*/0x00d8/*::]*/:{/* n:"SXRng", */},/*::[*/0x00d9/*::]*/:{/* n:"SxIsxoper", */},/*::[*/0x00da/*::]*/:{/* n:"BookBool", */f:parseuint16},/*::[*/0x00dc/*::]*/:{/* n:"DbOrParamQry", */},/*::[*/0x00dd/*::]*/:{/* n:"ScenarioProtect", */f:parsebool},/*::[*/0x00de/*::]*/:{/* n:"OleObjectSize", */},/*::[*/0x00e0/*::]*/:{/* n:"XF", */f:parse_XF},/*::[*/0x00e1/*::]*/:{/* n:"InterfaceHdr", */f:parse_InterfaceHdr},/*::[*/0x00e2/*::]*/:{/* n:"InterfaceEnd", */f:parsenoop2},/*::[*/0x00e3/*::]*/:{/* n:"SXVS", */},/*::[*/0x00e5/*::]*/:{/* n:"MergeCells", */f:parse_MergeCells},/*::[*/0x00e9/*::]*/:{/* n:"BkHim", */},/*::[*/0x00eb/*::]*/:{/* n:"MsoDrawingGroup", */},/*::[*/0x00ec/*::]*/:{/* n:"MsoDrawing", */},/*::[*/0x00ed/*::]*/:{/* n:"MsoDrawingSelection", */},/*::[*/0x00ef/*::]*/:{/* n:"PhoneticInfo", */},/*::[*/0x00f0/*::]*/:{/* n:"SxRule", */},/*::[*/0x00f1/*::]*/:{/* n:"SXEx", */},/*::[*/0x00f2/*::]*/:{/* n:"SxFilt", */},/*::[*/0x00f4/*::]*/:{/* n:"SxDXF", */},/*::[*/0x00f5/*::]*/:{/* n:"SxItm", */},/*::[*/0x00f6/*::]*/:{/* n:"SxName", */},/*::[*/0x00f7/*::]*/:{/* n:"SxSelect", */},/*::[*/0x00f8/*::]*/:{/* n:"SXPair", */},/*::[*/0x00f9/*::]*/:{/* n:"SxFmla", */},/*::[*/0x00fb/*::]*/:{/* n:"SxFormat", */},/*::[*/0x00fc/*::]*/:{/* n:"SST", */f:parse_SST},/*::[*/0x00fd/*::]*/:{/* n:"LabelSst", */f:parse_LabelSst},/*::[*/0x00ff/*::]*/:{/* n:"ExtSST", */f:parse_ExtSST},/*::[*/0x0100/*::]*/:{/* n:"SXVDEx", */},/*::[*/0x0103/*::]*/:{/* n:"SXFormula", */},/*::[*/0x0122/*::]*/:{/* n:"SXDBEx", */},/*::[*/0x0137/*::]*/:{/* n:"RRDInsDel", */},/*::[*/0x0138/*::]*/:{/* n:"RRDHead", */},/*::[*/0x013b/*::]*/:{/* n:"RRDChgCell", */},/*::[*/0x013d/*::]*/:{/* n:"RRTabId", */f:parseuint16a},/*::[*/0x013e/*::]*/:{/* n:"RRDRenSheet", */},/*::[*/0x013f/*::]*/:{/* n:"RRSort", */},/*::[*/0x0140/*::]*/:{/* n:"RRDMove", */},/*::[*/0x014a/*::]*/:{/* n:"RRFormat", */},/*::[*/0x014b/*::]*/:{/* n:"RRAutoFmt", */},/*::[*/0x014d/*::]*/:{/* n:"RRInsertSh", */},/*::[*/0x014e/*::]*/:{/* n:"RRDMoveBegin", */},/*::[*/0x014f/*::]*/:{/* n:"RRDMoveEnd", */},/*::[*/0x0150/*::]*/:{/* n:"RRDInsDelBegin", */},/*::[*/0x0151/*::]*/:{/* n:"RRDInsDelEnd", */},/*::[*/0x0152/*::]*/:{/* n:"RRDConflict", */},/*::[*/0x0153/*::]*/:{/* n:"RRDDefName", */},/*::[*/0x0154/*::]*/:{/* n:"RRDRstEtxp", */},/*::[*/0x015f/*::]*/:{/* n:"LRng", */},/*::[*/0x0160/*::]*/:{/* n:"UsesELFs", */f:parsebool},/*::[*/0x0161/*::]*/:{/* n:"DSF", */f:parsenoop2},/*::[*/0x0191/*::]*/:{/* n:"CUsr", */},/*::[*/0x0192/*::]*/:{/* n:"CbUsr", */},/*::[*/0x0193/*::]*/:{/* n:"UsrInfo", */},/*::[*/0x0194/*::]*/:{/* n:"UsrExcl", */},/*::[*/0x0195/*::]*/:{/* n:"FileLock", */},/*::[*/0x0196/*::]*/:{/* n:"RRDInfo", */},/*::[*/0x0197/*::]*/:{/* n:"BCUsrs", */},/*::[*/0x0198/*::]*/:{/* n:"UsrChk", */},/*::[*/0x01a9/*::]*/:{/* n:"UserBView", */},/*::[*/0x01aa/*::]*/:{/* n:"UserSViewBegin", */},/*::[*/0x01ab/*::]*/:{/* n:"UserSViewEnd", */},/*::[*/0x01ac/*::]*/:{/* n:"RRDUserView", */},/*::[*/0x01ad/*::]*/:{/* n:"Qsi", */},/*::[*/0x01ae/*::]*/:{/* n:"SupBook", */f:parse_SupBook},/*::[*/0x01af/*::]*/:{/* n:"Prot4Rev", */f:parsebool},/*::[*/0x01b0/*::]*/:{/* n:"CondFmt", */},/*::[*/0x01b1/*::]*/:{/* n:"CF", */},/*::[*/0x01b2/*::]*/:{/* n:"DVal", */},/*::[*/0x01b5/*::]*/:{/* n:"DConBin", */},/*::[*/0x01b6/*::]*/:{/* n:"TxO", */f:parse_TxO},/*::[*/0x01b7/*::]*/:{/* n:"RefreshAll", */f:parsebool},//
	/*::[*/0x01b8/*::]*/:{/* n:"HLink", */f:parse_HLink},/*::[*/0x01b9/*::]*/:{/* n:"Lel", */},/*::[*/0x01ba/*::]*/:{/* n:"CodeName", */f:parse_XLUnicodeString},/*::[*/0x01bb/*::]*/:{/* n:"SXFDBType", */},/*::[*/0x01bc/*::]*/:{/* n:"Prot4RevPass", */f:parseuint16},/*::[*/0x01bd/*::]*/:{/* n:"ObNoMacros", */},/*::[*/0x01be/*::]*/:{/* n:"Dv", */},/*::[*/0x01c0/*::]*/:{/* n:"Excel9File", */f:parsenoop2},/*::[*/0x01c1/*::]*/:{/* n:"RecalcId", */f:parse_RecalcId,r:2},/*::[*/0x01c2/*::]*/:{/* n:"EntExU2", */f:parsenoop2},/*::[*/0x0200/*::]*/:{/* n:"Dimensions", */f:parse_Dimensions},/*::[*/0x0201/*::]*/:{/* n:"Blank", */f:parse_Blank},/*::[*/0x0203/*::]*/:{/* n:"Number", */f:parse_Number},/*::[*/0x0204/*::]*/:{/* n:"Label", */f:parse_Label},/*::[*/0x0205/*::]*/:{/* n:"BoolErr", */f:parse_BoolErr},/*::[*/0x0207/*::]*/:{/* n:"String", */f:parse_String},/*::[*/0x0208/*::]*/:{/* n:"Row", */f:parse_Row},/*::[*/0x020b/*::]*/:{/* n:"Index", */},/*::[*/0x0221/*::]*/:{/* n:"Array", */f:parse_Array},/*::[*/0x0225/*::]*/:{/* n:"DefaultRowHeight", */f:parse_DefaultRowHeight},/*::[*/0x0236/*::]*/:{/* n:"Table", */},/*::[*/0x023e/*::]*/:{/* n:"Window2", */f:parse_Window2},/*::[*/0x027e/*::]*/:{/* n:"RK", */f:parse_RK},/*::[*/0x0293/*::]*/:{/* n:"Style", */},/*::[*/0x0418/*::]*/:{/* n:"BigName", */},/*::[*/0x041e/*::]*/:{/* n:"Format", */f:parse_Format},/*::[*/0x043c/*::]*/:{/* n:"ContinueBigName", */},/*::[*/0x04bc/*::]*/:{/* n:"ShrFmla", */f:parse_ShrFmla},/*::[*/0x0800/*::]*/:{/* n:"HLinkTooltip", */f:parse_HLinkTooltip},/*::[*/0x0801/*::]*/:{/* n:"WebPub", */},/*::[*/0x0802/*::]*/:{/* n:"QsiSXTag", */},/*::[*/0x0803/*::]*/:{/* n:"DBQueryExt", */},/*::[*/0x0804/*::]*/:{/* n:"ExtString", */},/*::[*/0x0805/*::]*/:{/* n:"TxtQry", */},/*::[*/0x0806/*::]*/:{/* n:"Qsir", */},/*::[*/0x0807/*::]*/:{/* n:"Qsif", */},/*::[*/0x0808/*::]*/:{/* n:"RRDTQSIF", */},/*::[*/0x0809/*::]*/:{/* n:"BOF", */f:parse_BOF},/*::[*/0x080a/*::]*/:{/* n:"OleDbConn", */},/*::[*/0x080b/*::]*/:{/* n:"WOpt", */},/*::[*/0x080c/*::]*/:{/* n:"SXViewEx", */},/*::[*/0x080d/*::]*/:{/* n:"SXTH", */},/*::[*/0x080e/*::]*/:{/* n:"SXPIEx", */},/*::[*/0x080f/*::]*/:{/* n:"SXVDTEx", */},/*::[*/0x0810/*::]*/:{/* n:"SXViewEx9", */},/*::[*/0x0812/*::]*/:{/* n:"ContinueFrt", */},/*::[*/0x0813/*::]*/:{/* n:"RealTimeData", */},/*::[*/0x0850/*::]*/:{/* n:"ChartFrtInfo", */},/*::[*/0x0851/*::]*/:{/* n:"FrtWrapper", */},/*::[*/0x0852/*::]*/:{/* n:"StartBlock", */},/*::[*/0x0853/*::]*/:{/* n:"EndBlock", */},/*::[*/0x0854/*::]*/:{/* n:"StartObject", */},/*::[*/0x0855/*::]*/:{/* n:"EndObject", */},/*::[*/0x0856/*::]*/:{/* n:"CatLab", */},/*::[*/0x0857/*::]*/:{/* n:"YMult", */},/*::[*/0x0858/*::]*/:{/* n:"SXViewLink", */},/*::[*/0x0859/*::]*/:{/* n:"PivotChartBits", */},/*::[*/0x085a/*::]*/:{/* n:"FrtFontList", */},/*::[*/0x0862/*::]*/:{/* n:"SheetExt", */},/*::[*/0x0863/*::]*/:{/* n:"BookExt", */r:12},/*::[*/0x0864/*::]*/:{/* n:"SXAddl", */},/*::[*/0x0865/*::]*/:{/* n:"CrErr", */},/*::[*/0x0866/*::]*/:{/* n:"HFPicture", */},/*::[*/0x0867/*::]*/:{/* n:"FeatHdr", */f:parsenoop2},/*::[*/0x0868/*::]*/:{/* n:"Feat", */},/*::[*/0x086a/*::]*/:{/* n:"DataLabExt", */},/*::[*/0x086b/*::]*/:{/* n:"DataLabExtContents", */},/*::[*/0x086c/*::]*/:{/* n:"CellWatch", */},/*::[*/0x0871/*::]*/:{/* n:"FeatHdr11", */},/*::[*/0x0872/*::]*/:{/* n:"Feature11", */},/*::[*/0x0874/*::]*/:{/* n:"DropDownObjIds", */},/*::[*/0x0875/*::]*/:{/* n:"ContinueFrt11", */},/*::[*/0x0876/*::]*/:{/* n:"DConn", */},/*::[*/0x0877/*::]*/:{/* n:"List12", */},/*::[*/0x0878/*::]*/:{/* n:"Feature12", */},/*::[*/0x0879/*::]*/:{/* n:"CondFmt12", */},/*::[*/0x087a/*::]*/:{/* n:"CF12", */},/*::[*/0x087b/*::]*/:{/* n:"CFEx", */},/*::[*/0x087c/*::]*/:{/* n:"XFCRC", */f:parse_XFCRC,r:12},/*::[*/0x087d/*::]*/:{/* n:"XFExt", */f:parse_XFExt,r:12},/*::[*/0x087e/*::]*/:{/* n:"AutoFilter12", */},/*::[*/0x087f/*::]*/:{/* n:"ContinueFrt12", */},/*::[*/0x0884/*::]*/:{/* n:"MDTInfo", */},/*::[*/0x0885/*::]*/:{/* n:"MDXStr", */},/*::[*/0x0886/*::]*/:{/* n:"MDXTuple", */},/*::[*/0x0887/*::]*/:{/* n:"MDXSet", */},/*::[*/0x0888/*::]*/:{/* n:"MDXProp", */},/*::[*/0x0889/*::]*/:{/* n:"MDXKPI", */},/*::[*/0x088a/*::]*/:{/* n:"MDB", */},/*::[*/0x088b/*::]*/:{/* n:"PLV", */},/*::[*/0x088c/*::]*/:{/* n:"Compat12", */f:parsebool,r:12},/*::[*/0x088d/*::]*/:{/* n:"DXF", */},/*::[*/0x088e/*::]*/:{/* n:"TableStyles", */r:12},/*::[*/0x088f/*::]*/:{/* n:"TableStyle", */},/*::[*/0x0890/*::]*/:{/* n:"TableStyleElement", */},/*::[*/0x0892/*::]*/:{/* n:"StyleExt", */},/*::[*/0x0893/*::]*/:{/* n:"NamePublish", */},/*::[*/0x0894/*::]*/:{/* n:"NameCmt", */f:parse_NameCmt,r:12},/*::[*/0x0895/*::]*/:{/* n:"SortData", */},/*::[*/0x0896/*::]*/:{/* n:"Theme", */f:parse_Theme,r:12},/*::[*/0x0897/*::]*/:{/* n:"GUIDTypeLib", */},/*::[*/0x0898/*::]*/:{/* n:"FnGrp12", */},/*::[*/0x0899/*::]*/:{/* n:"NameFnGrp12", */},/*::[*/0x089a/*::]*/:{/* n:"MTRSettings", */f:parse_MTRSettings,r:12},/*::[*/0x089b/*::]*/:{/* n:"CompressPictures", */f:parsenoop2},/*::[*/0x089c/*::]*/:{/* n:"HeaderFooter", */},/*::[*/0x089d/*::]*/:{/* n:"CrtLayout12", */},/*::[*/0x089e/*::]*/:{/* n:"CrtMlFrt", */},/*::[*/0x089f/*::]*/:{/* n:"CrtMlFrtContinue", */},/*::[*/0x08a3/*::]*/:{/* n:"ForceFullCalculation", */f:parse_ForceFullCalculation},/*::[*/0x08a4/*::]*/:{/* n:"ShapePropsStream", */},/*::[*/0x08a5/*::]*/:{/* n:"TextPropsStream", */},/*::[*/0x08a6/*::]*/:{/* n:"RichTextStream", */},/*::[*/0x08a7/*::]*/:{/* n:"CrtLayout12A", */},/*::[*/0x1001/*::]*/:{/* n:"Units", */},/*::[*/0x1002/*::]*/:{/* n:"Chart", */},/*::[*/0x1003/*::]*/:{/* n:"Series", */},/*::[*/0x1006/*::]*/:{/* n:"DataFormat", */},/*::[*/0x1007/*::]*/:{/* n:"LineFormat", */},/*::[*/0x1009/*::]*/:{/* n:"MarkerFormat", */},/*::[*/0x100a/*::]*/:{/* n:"AreaFormat", */},/*::[*/0x100b/*::]*/:{/* n:"PieFormat", */},/*::[*/0x100c/*::]*/:{/* n:"AttachedLabel", */},/*::[*/0x100d/*::]*/:{/* n:"SeriesText", */},/*::[*/0x1014/*::]*/:{/* n:"ChartFormat", */},/*::[*/0x1015/*::]*/:{/* n:"Legend", */},/*::[*/0x1016/*::]*/:{/* n:"SeriesList", */},/*::[*/0x1017/*::]*/:{/* n:"Bar", */},/*::[*/0x1018/*::]*/:{/* n:"Line", */},/*::[*/0x1019/*::]*/:{/* n:"Pie", */},/*::[*/0x101a/*::]*/:{/* n:"Area", */},/*::[*/0x101b/*::]*/:{/* n:"Scatter", */},/*::[*/0x101c/*::]*/:{/* n:"CrtLine", */},/*::[*/0x101d/*::]*/:{/* n:"Axis", */},/*::[*/0x101e/*::]*/:{/* n:"Tick", */},/*::[*/0x101f/*::]*/:{/* n:"ValueRange", */},/*::[*/0x1020/*::]*/:{/* n:"CatSerRange", */},/*::[*/0x1021/*::]*/:{/* n:"AxisLine", */},/*::[*/0x1022/*::]*/:{/* n:"CrtLink", */},/*::[*/0x1024/*::]*/:{/* n:"DefaultText", */},/*::[*/0x1025/*::]*/:{/* n:"Text", */},/*::[*/0x1026/*::]*/:{/* n:"FontX", */f:parseuint16},/*::[*/0x1027/*::]*/:{/* n:"ObjectLink", */},/*::[*/0x1032/*::]*/:{/* n:"Frame", */},/*::[*/0x1033/*::]*/:{/* n:"Begin", */},/*::[*/0x1034/*::]*/:{/* n:"End", */},/*::[*/0x1035/*::]*/:{/* n:"PlotArea", */},/*::[*/0x103a/*::]*/:{/* n:"Chart3d", */},/*::[*/0x103c/*::]*/:{/* n:"PicF", */},/*::[*/0x103d/*::]*/:{/* n:"DropBar", */},/*::[*/0x103e/*::]*/:{/* n:"Radar", */},/*::[*/0x103f/*::]*/:{/* n:"Surf", */},/*::[*/0x1040/*::]*/:{/* n:"RadarArea", */},/*::[*/0x1041/*::]*/:{/* n:"AxisParent", */},/*::[*/0x1043/*::]*/:{/* n:"LegendException", */},/*::[*/0x1044/*::]*/:{/* n:"ShtProps", */f:parse_ShtProps},/*::[*/0x1045/*::]*/:{/* n:"SerToCrt", */},/*::[*/0x1046/*::]*/:{/* n:"AxesUsed", */},/*::[*/0x1048/*::]*/:{/* n:"SBaseRef", */},/*::[*/0x104a/*::]*/:{/* n:"SerParent", */},/*::[*/0x104b/*::]*/:{/* n:"SerAuxTrend", */},/*::[*/0x104e/*::]*/:{/* n:"IFmtRecord", */},/*::[*/0x104f/*::]*/:{/* n:"Pos", */},/*::[*/0x1050/*::]*/:{/* n:"AlRuns", */},/*::[*/0x1051/*::]*/:{/* n:"BRAI", */},/*::[*/0x105b/*::]*/:{/* n:"SerAuxErrBar", */},/*::[*/0x105c/*::]*/:{/* n:"ClrtClient", */f:parse_ClrtClient},/*::[*/0x105d/*::]*/:{/* n:"SerFmt", */},/*::[*/0x105f/*::]*/:{/* n:"Chart3DBarShape", */},/*::[*/0x1060/*::]*/:{/* n:"Fbi", */},/*::[*/0x1061/*::]*/:{/* n:"BopPop", */},/*::[*/0x1062/*::]*/:{/* n:"AxcExt", */},/*::[*/0x1063/*::]*/:{/* n:"Dat", */},/*::[*/0x1064/*::]*/:{/* n:"PlotGrowth", */},/*::[*/0x1065/*::]*/:{/* n:"SIIndex", */},/*::[*/0x1066/*::]*/:{/* n:"GelFrame", */},/*::[*/0x1067/*::]*/:{/* n:"BopPopCustom", */},/*::[*/0x1068/*::]*/:{/* n:"Fbi2", */},/*::[*/0x0000/*::]*/:{/* n:"Dimensions", */f:parse_Dimensions},/*::[*/0x0001/*::]*/:{/* n:"BIFF2BLANK", */},/*::[*/0x0002/*::]*/:{/* n:"BIFF2INT", */f:parse_BIFF2INT},/*::[*/0x0003/*::]*/:{/* n:"BIFF2NUM", */f:parse_BIFF2NUM},/*::[*/0x0004/*::]*/:{/* n:"BIFF2STR", */f:parse_BIFF2STR},/*::[*/0x0005/*::]*/:{/* n:"BoolErr", */f:parse_BoolErr},/*::[*/0x0007/*::]*/:{/* n:"String", */f:parse_BIFF2STRING},/*::[*/0x0008/*::]*/:{/* n:"BIFF2ROW", */},/*::[*/0x0009/*::]*/:{/* n:"BOF", */f:parse_BOF},/*::[*/0x000b/*::]*/:{/* n:"Index", */},/*::[*/0x0016/*::]*/:{/* n:"ExternCount", */f:parseuint16},/*::[*/0x001e/*::]*/:{/* n:"BIFF2FORMAT", */f:parse_BIFF2Format},/*::[*/0x001f/*::]*/:{/* n:"BIFF2FMTCNT", */},/* 16-bit cnt of BIFF2FORMAT records */ /*::[*/0x0020/*::]*/:{/* n:"BIFF2COLINFO", */},/*::[*/0x0021/*::]*/:{/* n:"Array", */f:parse_Array},/*::[*/0x0024/*::]*/:{/* n:"COLWIDTH", */},/*::[*/0x0025/*::]*/:{/* n:"DefaultRowHeight", */f:parse_DefaultRowHeight},// 0x2c ??
	// 0x2d ??
	// 0x2e ??
	// 0x30 FONTCOUNT: number of fonts
	/*::[*/0x0032/*::]*/:{/* n:"BIFF2FONTXTRA", */f:parse_BIFF2FONTXTRA},// 0x35: INFOOPTS
	// 0x36: TABLE (BIFF2 only)
	// 0x37: TABLE2 (BIFF2 only)
	// 0x38: WNDESK
	// 0x39 ??
	// 0x3a: BEGINPREF
	// 0x3b: ENDPREF
	/*::[*/0x003e/*::]*/:{/* n:"BIFF2WINDOW2", */},// 0x3f ??
	// 0x46: SHOWSCROLL
	// 0x47: SHOWFORMULA
	// 0x48: STATUSBAR
	// 0x49: SHORTMENUS
	// 0x4A:
	// 0x4B:
	// 0x4C:
	// 0x4E:
	// 0x4F:
	// 0x58: TOOLBAR (BIFF3)
	/* - - - */ /*::[*/0x0034/*::]*/:{/* n:"DDEObjName", */},/*::[*/0x0043/*::]*/:{/* n:"BIFF2XF", */},/*::[*/0x0044/*::]*/:{/* n:"BIFF2XFINDEX", */f:parseuint16},/*::[*/0x0045/*::]*/:{/* n:"BIFF2FONTCLR", */},/*::[*/0x0056/*::]*/:{/* n:"BIFF4FMTCNT", */},/* 16-bit cnt, similar to BIFF2 */ /*::[*/0x007e/*::]*/:{/* n:"RK", */},/* Not necessarily same as 0x027e */ /*::[*/0x007f/*::]*/:{/* n:"ImData", */f:parse_ImData},/*::[*/0x0087/*::]*/:{/* n:"Addin", */},/*::[*/0x0088/*::]*/:{/* n:"Edg", */},/*::[*/0x0089/*::]*/:{/* n:"Pub", */},// 0x8A
	// 0x8B LH: alternate menu key flag (BIFF3/4)
	// 0x8E
	// 0x8F
	/*::[*/0x0091/*::]*/:{/* n:"Sub", */},// 0x93 STYLE
	/*::[*/0x0094/*::]*/:{/* n:"LHRecord", */},/*::[*/0x0095/*::]*/:{/* n:"LHNGraph", */},/*::[*/0x0096/*::]*/:{/* n:"Sound", */},// 0xA2 FNPROTO: function prototypes (BIFF4)
	// 0xA3
	// 0xA8
	/*::[*/0x00a9/*::]*/:{/* n:"CoordList", */},/*::[*/0x00ab/*::]*/:{/* n:"GCW", */},/*::[*/0x00bc/*::]*/:{/* n:"ShrFmla", */},/* Not necessarily same as 0x04bc */ /*::[*/0x00bf/*::]*/:{/* n:"ToolbarHdr", */},/*::[*/0x00c0/*::]*/:{/* n:"ToolbarEnd", */},/*::[*/0x00c2/*::]*/:{/* n:"AddMenu", */},/*::[*/0x00c3/*::]*/:{/* n:"DelMenu", */},/*::[*/0x00d6/*::]*/:{/* n:"RString", */f:parse_RString},/*::[*/0x00df/*::]*/:{/* n:"UDDesc", */},/*::[*/0x00ea/*::]*/:{/* n:"TabIdConf", */},/*::[*/0x0162/*::]*/:{/* n:"XL5Modify", */},/*::[*/0x01a5/*::]*/:{/* n:"FileSharing2", */},/*::[*/0x0206/*::]*/:{/* n:"Formula", */f:parse_Formula},/*::[*/0x0209/*::]*/:{/* n:"BOF", */f:parse_BOF},/*::[*/0x0218/*::]*/:{/* n:"Lbl", */f:parse_Lbl},/*::[*/0x0223/*::]*/:{/* n:"ExternName", */f:parse_ExternName},/*::[*/0x0231/*::]*/:{/* n:"Font", */},/*::[*/0x0243/*::]*/:{/* n:"BIFF3XF", */},/*::[*/0x0406/*::]*/:{/* n:"Formula", */f:parse_Formula},/*::[*/0x0409/*::]*/:{/* n:"BOF", */f:parse_BOF},/*::[*/0x0443/*::]*/:{/* n:"BIFF4XF", */},/*::[*/0x086d/*::]*/:{/* n:"FeatInfo", */},/*::[*/0x0873/*::]*/:{/* n:"FeatInfo11", */},/*::[*/0x0881/*::]*/:{/* n:"SXAddl12", */},/*::[*/0x08c0/*::]*/:{/* n:"AutoWebPub", */},/*::[*/0x08c1/*::]*/:{/* n:"ListObj", */},/*::[*/0x08c2/*::]*/:{/* n:"ListField", */},/*::[*/0x08c3/*::]*/:{/* n:"ListDV", */},/*::[*/0x08c4/*::]*/:{/* n:"ListCondFmt", */},/*::[*/0x08c5/*::]*/:{/* n:"ListCF", */},/*::[*/0x08c6/*::]*/:{/* n:"FMQry", */},/*::[*/0x08c7/*::]*/:{/* n:"FMSQry", */},/*::[*/0x08c8/*::]*/:{/* n:"PLV", */},/*::[*/0x08c9/*::]*/:{/* n:"LnExt", */},/*::[*/0x08ca/*::]*/:{/* n:"MkrExt", */},/*::[*/0x08cb/*::]*/:{/* n:"CrtCoopt", */},/*::[*/0x08d6/*::]*/:{/* n:"FRTArchId$", */r:12},/*::[*/0x7262/*::]*/:{}};function write_biff_rec(ba/*:BufArray*/,type/*:number*/,payload,length/*:?number*/)/*:void*/{var t/*:number*/=type;if(isNaN(t))return;var len=length||(payload||[]).length||0;var o=ba.next(4);o.write_shift(2,t);o.write_shift(2,len);if(/*:: len != null &&*/len>0&&is_buf(payload))ba.push(payload);}/* note: browser DOM element cannot see mso- style attrs, must parse */function html_to_sheet(str/*:string*/,_opts)/*:Workbook*/{var opts=_opts||{};var ws/*:Worksheet*/=opts.dense?[]/*:any*/:{}/*:any*/;str=str.replace(/<!--.*?-->/g,"");var mtch/*:any*/=str.match(/<table/i);if(!mtch)throw new Error("Invalid HTML: could not find <table>");var mtch2/*:any*/=str.match(/<\/table/i);var i/*:number*/=mtch.index,j/*:number*/=mtch2&&mtch2.index||str.length;var rows=split_regex(str.slice(i,j),/(:?<tr[^>]*>)/i,"<tr>");var R=-1,C=0,RS=0,CS=0;var range/*:Range*/={s:{r:10000000,c:10000000},e:{r:0,c:0}};var merges/*:Array<Range>*/=[];for(i=0;i<rows.length;++i){var row=rows[i].trim();var hd=row.slice(0,3).toLowerCase();if(hd=="<tr"){++R;if(opts.sheetRows&&opts.sheetRows<=R){--R;break;}C=0;continue;}if(hd!="<td"&&hd!="<th")continue;var cells=row.split(/<\/t[dh]>/i);for(j=0;j<cells.length;++j){var cell=cells[j].trim();if(!cell.match(/<t[dh]/i))continue;var m=cell,cc=0;/* TODO: parse styles etc */while(m.charAt(0)=="<"&&(cc=m.indexOf(">"))>-1)m=m.slice(cc+1);for(var midx=0;midx<merges.length;++midx){var _merge/*:Range*/=merges[midx];if(_merge.s.c==C&&_merge.s.r<R&&R<=_merge.e.r){C=_merge.e.c+1;midx=-1;}}var tag=parsexmltag(cell.slice(0,cell.indexOf(">")));CS=tag.colspan?+tag.colspan:1;if((RS=+tag.rowspan)>1||CS>1)merges.push({s:{r:R,c:C},e:{r:R+(RS||1)-1,c:C+CS-1}});var _t/*:string*/=tag.t||tag["data-t"]||"";/* TODO: generate stub cells */if(!m.length){C+=CS;continue;}m=htmldecode(m);if(range.s.r>R)range.s.r=R;if(range.e.r<R)range.e.r=R;if(range.s.c>C)range.s.c=C;if(range.e.c<C)range.e.c=C;if(!m.length){C+=CS;continue;}var o/*:Cell*/={t:'s',v:m};if(opts.raw||!m.trim().length||_t=='s');else if(m==='TRUE')o={t:'b',v:true};else if(m==='FALSE')o={t:'b',v:false};else if(!isNaN(fuzzynum(m)))o={t:'n',v:fuzzynum(m)};else if(!isNaN(fuzzydate(m).getDate())){o={t:'d',v:parseDate(m)}/*:any*/;if(!opts.cellDates)o={t:'n',v:datenum(o.v)}/*:any*/;o.z=opts.dateNF||table_fmt[14];}if(opts.dense){if(!ws[R])ws[R]=[];ws[R][C]=o;}else ws[encode_cell({r:R,c:C})]=o;C+=CS;}}ws['!ref']=encode_range(range);if(merges.length)ws["!merges"]=merges;return ws;}function make_html_row(ws/*:Worksheet*/,r/*:Range*/,R/*:number*/,o/*:Sheet2HTMLOpts*/)/*:string*/{var M/*:Array<Range>*/=ws['!merges']||[];var oo/*:Array<string>*/=[];for(var C=r.s.c;C<=r.e.c;++C){var RS=0,CS=0;for(var j=0;j<M.length;++j){if(M[j].s.r>R||M[j].s.c>C)continue;if(M[j].e.r<R||M[j].e.c<C)continue;if(M[j].s.r<R||M[j].s.c<C){RS=-1;break;}RS=M[j].e.r-M[j].s.r+1;CS=M[j].e.c-M[j].s.c+1;break;}if(RS<0)continue;var coord=encode_cell({r:R,c:C});var cell=o.dense?(ws[R]||[])[C]:ws[coord];/* TODO: html entities */var w=cell&&cell.v!=null&&(cell.h||escapehtml(cell.w||(format_cell(cell),cell.w)||""))||"";var sp={}/*:any*/;if(RS>1)sp.rowspan=RS;if(CS>1)sp.colspan=CS;if(o.editable)w='<span contenteditable="true">'+w+'</span>';else if(cell){sp["data-t"]=cell&&cell.t||'z';if(cell.v!=null)sp["data-v"]=cell.v;if(cell.z!=null)sp["data-z"]=cell.z;if(cell.l&&(cell.l.Target||"#").charAt(0)!="#")w='<a href="'+cell.l.Target+'">'+w+'</a>';}sp.id=(o.id||"sjs")+"-"+coord;oo.push(writextag('td',w,sp));}var preamble="<tr>";return preamble+oo.join("")+"</tr>";}var HTML_BEGIN='<html><head><meta charset="utf-8"/><title>SheetJS Table Export</title></head><body>';var HTML_END='</body></html>';function html_to_workbook(str/*:string*/,opts)/*:Workbook*/{var mtch=str.match(/<table[\s\S]*?>[\s\S]*?<\/table>/gi);if(!mtch||mtch.length==0)throw new Error("Invalid HTML: could not find <table>");if(mtch.length==1)return sheet_to_workbook(html_to_sheet(mtch[0],opts),opts);var wb=book_new();mtch.forEach(function(s,idx){book_append_sheet(wb,html_to_sheet(s,opts),"Sheet"+(idx+1));});return wb;}function make_html_preamble(ws/*:Worksheet*/,R/*:Range*/,o/*:Sheet2HTMLOpts*/)/*:string*/{var out/*:Array<string>*/=[];return out.join("")+'<table'+(o&&o.id?' id="'+o.id+'"':"")+'>';}function sheet_to_html(ws/*:Worksheet*/,opts/*:?Sheet2HTMLOpts*/ /*, wb:?Workbook*/)/*:string*/{var o=opts||{};var header=o.header!=null?o.header:HTML_BEGIN;var footer=o.footer!=null?o.footer:HTML_END;var out/*:Array<string>*/=[header];var r=decode_range(ws['!ref']);o.dense=Array.isArray(ws);out.push(make_html_preamble(ws,r,o));for(var R=r.s.r;R<=r.e.r;++R)out.push(make_html_row(ws,r,R,o));out.push("</table>"+footer);return out.join("");}function sheet_add_dom(ws/*:Worksheet*/,table/*:HTMLElement*/,_opts/*:?any*/)/*:Worksheet*/{var opts=_opts||{};var or_R=0,or_C=0;if(opts.origin!=null){if(typeof opts.origin=='number')or_R=opts.origin;else {var _origin/*:CellAddress*/=typeof opts.origin=="string"?decode_cell(opts.origin):opts.origin;or_R=_origin.r;or_C=_origin.c;}}var rows/*:HTMLCollection<HTMLTableRowElement>*/=table.getElementsByTagName('tr');var sheetRows=Math.min(opts.sheetRows||10000000,rows.length);var range/*:Range*/={s:{r:0,c:0},e:{r:or_R,c:or_C}};if(ws["!ref"]){var _range/*:Range*/=decode_range(ws["!ref"]);range.s.r=Math.min(range.s.r,_range.s.r);range.s.c=Math.min(range.s.c,_range.s.c);range.e.r=Math.max(range.e.r,_range.e.r);range.e.c=Math.max(range.e.c,_range.e.c);if(or_R==-1)range.e.r=or_R=_range.e.r+1;}var merges/*:Array<Range>*/=[],midx=0;var rowinfo/*:Array<RowInfo>*/=ws["!rows"]||(ws["!rows"]=[]);var _R=0,R=0,_C=0,C=0,RS=0,CS=0;if(!ws["!cols"])ws['!cols']=[];for(;_R<rows.length&&R<sheetRows;++_R){var row/*:HTMLTableRowElement*/=rows[_R];if(is_dom_element_hidden(row)){if(opts.display)continue;rowinfo[R]={hidden:true};}var elts/*:HTMLCollection<HTMLTableCellElement>*/=row.children/*:any*/;for(_C=C=0;_C<elts.length;++_C){var elt/*:HTMLTableCellElement*/=elts[_C];if(opts.display&&is_dom_element_hidden(elt))continue;var v/*:?string*/=elt.hasAttribute('data-v')?elt.getAttribute('data-v'):elt.hasAttribute('v')?elt.getAttribute('v'):htmldecode(elt.innerHTML);var z/*:?string*/=elt.getAttribute('data-z')||elt.getAttribute('z');for(midx=0;midx<merges.length;++midx){var m/*:Range*/=merges[midx];if(m.s.c==C+or_C&&m.s.r<R+or_R&&R+or_R<=m.e.r){C=m.e.c+1-or_C;midx=-1;}}/* TODO: figure out how to extract nonstandard mso- style */CS=+elt.getAttribute("colspan")||1;if((RS=+elt.getAttribute("rowspan")||1)>1||CS>1)merges.push({s:{r:R+or_R,c:C+or_C},e:{r:R+or_R+(RS||1)-1,c:C+or_C+(CS||1)-1}});var o/*:Cell*/={t:'s',v:v};var _t/*:string*/=elt.getAttribute("data-t")||elt.getAttribute("t")||"";if(v!=null){if(v.length==0)o.t=_t||'z';else if(opts.raw||v.trim().length==0||_t=="s");else if(v==='TRUE')o={t:'b',v:true};else if(v==='FALSE')o={t:'b',v:false};else if(!isNaN(fuzzynum(v)))o={t:'n',v:fuzzynum(v)};else if(!isNaN(fuzzydate(v).getDate())){o={t:'d',v:parseDate(v)}/*:any*/;if(!opts.cellDates)o={t:'n',v:datenum(o.v)}/*:any*/;o.z=opts.dateNF||table_fmt[14];}}if(o.z===undefined&&z!=null)o.z=z;/* The first link is used.  Links are assumed to be fully specified.
				 * TODO: The right way to process relative links is to make a new <a> */var l="",Aelts=elt.getElementsByTagName("A");if(Aelts&&Aelts.length)for(var Aelti=0;Aelti<Aelts.length;++Aelti)if(Aelts[Aelti].hasAttribute("href")){l=Aelts[Aelti].getAttribute("href");if(l.charAt(0)!="#")break;}if(l&&l.charAt(0)!="#")o.l={Target:l};if(opts.dense){if(!ws[R+or_R])ws[R+or_R]=[];ws[R+or_R][C+or_C]=o;}else ws[encode_cell({c:C+or_C,r:R+or_R})]=o;if(range.e.c<C+or_C)range.e.c=C+or_C;C+=CS;}++R;}if(merges.length)ws['!merges']=(ws["!merges"]||[]).concat(merges);range.e.r=Math.max(range.e.r,R-1+or_R);ws['!ref']=encode_range(range);if(R>=sheetRows)ws['!fullref']=encode_range((range.e.r=rows.length-_R+R-1+or_R,range));// We can count the real number of rows to parse but we don't to improve the performance
	return ws;}function parse_dom_table(table/*:HTMLElement*/,_opts/*:?any*/)/*:Worksheet*/{var opts=_opts||{};var ws/*:Worksheet*/=opts.dense?[]/*:any*/:{}/*:any*/;return sheet_add_dom(ws,table,_opts);}function table_to_book(table/*:HTMLElement*/,opts/*:?any*/)/*:Workbook*/{return sheet_to_workbook(parse_dom_table(table,opts),opts);}function is_dom_element_hidden(element/*:HTMLElement*/)/*:boolean*/{var display/*:string*/='';var get_computed_style/*:?function*/=get_get_computed_style_function(element);if(get_computed_style)display=get_computed_style(element).getPropertyValue('display');if(!display)display=element.style&&element.style.display;return display==='none';}/* global getComputedStyle */function get_get_computed_style_function(element/*:HTMLElement*/)/*:?function*/{// The proper getComputedStyle implementation is the one defined in the element window
	if(element.ownerDocument.defaultView&&typeof element.ownerDocument.defaultView.getComputedStyle==='function')return element.ownerDocument.defaultView.getComputedStyle;// If it is not available, try to get one from the global namespace
	if(typeof getComputedStyle==='function')return getComputedStyle;return null;}/* OpenDocument */function parse_text_p(text/*:string*/ /*::, tag*/)/*:Array<any>*/{/* 6.1.2 White Space Characters */var fixed=text.replace(/[\t\r\n]/g," ").trim().replace(/ +/g," ").replace(/<text:s\/>/g," ").replace(/<text:s text:c="(\d+)"\/>/g,function($$,$1){return Array(parseInt($1,10)+1).join(" ");}).replace(/<text:tab[^>]*\/>/g,"\t").replace(/<text:line-break\/>/g,"\n");var v=unescapexml(fixed.replace(/<[^>]*>/g,""));return [v];}var number_formats_ods={/* ods name: [short ssf fmt, long ssf fmt] */day:["d","dd"],month:["m","mm"],year:["y","yy"],hours:["h","hh"],minutes:["m","mm"],seconds:["s","ss"],"am-pm":["A/P","AM/PM"],"day-of-week":["ddd","dddd"],era:["e","ee"],/* there is no native representation of LO "Q" format */quarter:["\\Qm","m\\\"th quarter\""]};function parse_content_xml(d/*:string*/,_opts)/*:Workbook*/{var opts=_opts||{};var str=xlml_normalize(d);var state/*:Array<any>*/=[],tmp;var tag/*:: = {}*/;var NFtag={name:""},NF="",pidx=0;var sheetag/*:: = {name:"", '名称':""}*/;var rowtag/*:: = {'行号':""}*/;var Sheets={},SheetNames/*:Array<string>*/=[];var ws=opts.dense?[]/*:any*/:{}/*:any*/;var Rn,q/*:: :any = ({t:"", v:null, z:null, w:"",c:[],}:any)*/;var ctag={value:""}/*:any*/;var textp="",textpidx=0;var textR=[];var R=-1,C=-1,range={s:{r:1000000,c:10000000},e:{r:0,c:0}};var row_ol=0;var number_format_map={};var merges/*:Array<Range>*/=[],mrange={},mR=0,mC=0;var rowinfo/*:Array<RowInfo>*/=[],rowpeat=1,colpeat=1;var arrayf/*:Array<[Range, string]>*/=[];var WB={Names:[]};var atag={}/*:any*/;var _Ref/*:[string, string]*/=["",""];var comments/*:Array<Comment>*/=[],comment/*:Comment*/={}/*:any*/;var creator="",creatoridx=0;var isstub=false,intable=false;var i=0;xlmlregex.lastIndex=0;str=str.replace(/<!--([\s\S]*?)-->/mg,"").replace(/<!DOCTYPE[^\[]*\[[^\]]*\]>/gm,"");while(Rn=xlmlregex.exec(str))switch(Rn[3]=Rn[3].replace(/_.*$/,"")){case'table':case'工作表':// 9.1.2 <table:table>
	if(Rn[1]==='/'){if(range.e.c>=range.s.c&&range.e.r>=range.s.r)ws['!ref']=encode_range(range);else ws['!ref']="A1:A1";if(opts.sheetRows>0&&opts.sheetRows<=range.e.r){ws['!fullref']=ws['!ref'];range.e.r=opts.sheetRows-1;ws['!ref']=encode_range(range);}if(merges.length)ws['!merges']=merges;if(rowinfo.length)ws["!rows"]=rowinfo;sheetag.name=sheetag['名称']||sheetag.name;if(typeof JSON!=='undefined')JSON.stringify(sheetag);SheetNames.push(sheetag.name);Sheets[sheetag.name]=ws;intable=false;}else if(Rn[0].charAt(Rn[0].length-2)!=='/'){sheetag=parsexmltag(Rn[0],false);R=C=-1;range.s.r=range.s.c=10000000;range.e.r=range.e.c=0;ws=opts.dense?[]/*:any*/:{}/*:any*/;merges=[];rowinfo=[];intable=true;}break;case'table-row-group':// 9.1.9 <table:table-row-group>
	if(Rn[1]==="/")--row_ol;else ++row_ol;break;case'table-row':case'行':// 9.1.3 <table:table-row>
	if(Rn[1]==='/'){R+=rowpeat;rowpeat=1;break;}rowtag=parsexmltag(Rn[0],false);if(rowtag['行号'])R=rowtag['行号']-1;else if(R==-1)R=0;rowpeat=+rowtag['number-rows-repeated']||1;/* TODO: remove magic */if(rowpeat<10)for(i=0;i<rowpeat;++i)if(row_ol>0)rowinfo[R+i]={level:row_ol};C=-1;break;case'covered-table-cell':// 9.1.5 <table:covered-table-cell>
	if(Rn[1]!=='/')++C;if(opts.sheetStubs){if(opts.dense){if(!ws[R])ws[R]=[];ws[R][C]={t:'z'};}else ws[encode_cell({r:R,c:C})]={t:'z'};}textp="";textR=[];break;/* stub */case'table-cell':case'数据':if(Rn[0].charAt(Rn[0].length-2)==='/'){++C;ctag=parsexmltag(Rn[0],false);colpeat=parseInt(ctag['number-columns-repeated']||"1",10);q={t:'z',v:null/*:: , z:null, w:"",c:[]*/}/*:any*/;if(ctag.formula&&opts.cellFormula!=false)q.f=ods_to_csf_formula(unescapexml(ctag.formula));if((ctag['数据类型']||ctag['value-type'])=="string"){q.t="s";q.v=unescapexml(ctag['string-value']||"");if(opts.dense){if(!ws[R])ws[R]=[];ws[R][C]=q;}else {ws[encode_cell({r:R,c:C})]=q;}}C+=colpeat-1;}else if(Rn[1]!=='/'){++C;textp="";textpidx=0;textR=[];colpeat=1;var rptR=rowpeat?R+rowpeat-1:R;if(C>range.e.c)range.e.c=C;if(C<range.s.c)range.s.c=C;if(R<range.s.r)range.s.r=R;if(rptR>range.e.r)range.e.r=rptR;ctag=parsexmltag(Rn[0],false);comments=[];comment={}/*:any*/;q={t:ctag['数据类型']||ctag['value-type'],v:null/*:: , z:null, w:"",c:[]*/}/*:any*/;if(opts.cellFormula){if(ctag.formula)ctag.formula=unescapexml(ctag.formula);if(ctag['number-matrix-columns-spanned']&&ctag['number-matrix-rows-spanned']){mR=parseInt(ctag['number-matrix-rows-spanned'],10)||0;mC=parseInt(ctag['number-matrix-columns-spanned'],10)||0;mrange={s:{r:R,c:C},e:{r:R+mR-1,c:C+mC-1}};q.F=encode_range(mrange);arrayf.push([mrange,q.F]);}if(ctag.formula)q.f=ods_to_csf_formula(ctag.formula);else for(i=0;i<arrayf.length;++i)if(R>=arrayf[i][0].s.r&&R<=arrayf[i][0].e.r)if(C>=arrayf[i][0].s.c&&C<=arrayf[i][0].e.c)q.F=arrayf[i][1];}if(ctag['number-columns-spanned']||ctag['number-rows-spanned']){mR=parseInt(ctag['number-rows-spanned'],10)||0;mC=parseInt(ctag['number-columns-spanned'],10)||0;mrange={s:{r:R,c:C},e:{r:R+mR-1,c:C+mC-1}};merges.push(mrange);}/* 19.675.2 table:number-columns-repeated */if(ctag['number-columns-repeated'])colpeat=parseInt(ctag['number-columns-repeated'],10);/* 19.385 office:value-type */switch(q.t){case'boolean':q.t='b';q.v=parsexmlbool(ctag['boolean-value']);break;case'float':q.t='n';q.v=parseFloat(ctag.value);break;case'percentage':q.t='n';q.v=parseFloat(ctag.value);break;case'currency':q.t='n';q.v=parseFloat(ctag.value);break;case'date':q.t='d';q.v=parseDate(ctag['date-value']);if(!opts.cellDates){q.t='n';q.v=datenum(q.v);}q.z='m/d/yy';break;case'time':q.t='n';q.v=parse_isodur(ctag['time-value'])/86400;if(opts.cellDates){q.t='d';q.v=numdate(q.v);}q.z='HH:MM:SS';break;case'number':q.t='n';q.v=parseFloat(ctag['数据数值']);break;default:if(q.t==='string'||q.t==='text'||!q.t){q.t='s';if(ctag['string-value']!=null){textp=unescapexml(ctag['string-value']);textR=[];}}else throw new Error('Unsupported value type '+q.t);}}else {isstub=false;if(q.t==='s'){q.v=textp||'';if(textR.length)q.R=textR;isstub=textpidx==0;}if(atag.Target)q.l=atag;if(comments.length>0){q.c=comments;comments=[];}if(textp&&opts.cellText!==false)q.w=textp;if(isstub){q.t="z";delete q.v;}if(!isstub||opts.sheetStubs){if(!(opts.sheetRows&&opts.sheetRows<=R)){for(var rpt=0;rpt<rowpeat;++rpt){colpeat=parseInt(ctag['number-columns-repeated']||"1",10);if(opts.dense){if(!ws[R+rpt])ws[R+rpt]=[];ws[R+rpt][C]=rpt==0?q:dup(q);while(--colpeat>0)ws[R+rpt][C+colpeat]=dup(q);}else {ws[encode_cell({r:R+rpt,c:C})]=q;while(--colpeat>0)ws[encode_cell({r:R+rpt,c:C+colpeat})]=dup(q);}if(range.e.c<=C)range.e.c=C;}}}colpeat=parseInt(ctag['number-columns-repeated']||"1",10);C+=colpeat-1;colpeat=0;q={/*:: t:"", v:null, z:null, w:"",c:[]*/};textp="";textR=[];}atag={}/*:any*/;break;// 9.1.4 <table:table-cell>
	/* pure state */case'document':// TODO: <office:document> is the root for FODS
	case'document-content':case'电子表格文档':// 3.1.3.2 <office:document-content>
	case'spreadsheet':case'主体':// 3.7 <office:spreadsheet>
	case'scripts':// 3.12 <office:scripts>
	case'styles':// TODO <office:styles>
	case'font-face-decls':// 3.14 <office:font-face-decls>
	case'master-styles':// 3.15.4 <office:master-styles> -- relevant for FODS
	if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw "Bad state: "+tmp;}else if(Rn[0].charAt(Rn[0].length-2)!=='/')state.push([Rn[3],true]);break;case'annotation':// 14.1 <office:annotation>
	if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw "Bad state: "+tmp;comment.t=textp;if(textR.length)/*::(*/comment/*:: :any)*/.R=textR;comment.a=creator;comments.push(comment);}else if(Rn[0].charAt(Rn[0].length-2)!=='/'){state.push([Rn[3],false]);}creator="";creatoridx=0;textp="";textpidx=0;textR=[];break;case'creator':// 4.3.2.7 <dc:creator>
	if(Rn[1]==='/'){creator=str.slice(creatoridx,Rn.index);}else creatoridx=Rn.index+Rn[0].length;break;/* ignore state */case'meta':case'元数据':// TODO: <office:meta> <uof:元数据> FODS/UOF
	case'settings':// TODO: <office:settings>
	case'config-item-set':// TODO: <office:config-item-set>
	case'config-item-map-indexed':// TODO: <office:config-item-map-indexed>
	case'config-item-map-entry':// TODO: <office:config-item-map-entry>
	case'config-item-map-named':// TODO: <office:config-item-map-entry>
	case'shapes':// 9.2.8 <table:shapes>
	case'frame':// 10.4.2 <draw:frame>
	case'text-box':// 10.4.3 <draw:text-box>
	case'image':// 10.4.4 <draw:image>
	case'data-pilot-tables':// 9.6.2 <table:data-pilot-tables>
	case'list-style':// 16.30 <text:list-style>
	case'form':// 13.13 <form:form>
	case'dde-links':// 9.8 <table:dde-links>
	case'event-listeners':// TODO
	case'chart':// TODO
	if(Rn[1]==='/'){if((tmp=state.pop())[0]!==Rn[3])throw "Bad state: "+tmp;}else if(Rn[0].charAt(Rn[0].length-2)!=='/')state.push([Rn[3],false]);textp="";textpidx=0;textR=[];break;case'scientific-number':// TODO: <number:scientific-number>
	break;case'currency-symbol':// TODO: <number:currency-symbol>
	break;case'currency-style':// TODO: <number:currency-style>
	break;case'number-style':// 16.27.2 <number:number-style>
	case'percentage-style':// 16.27.9 <number:percentage-style>
	case'date-style':// 16.27.10 <number:date-style>
	case'time-style':// 16.27.18 <number:time-style>
	if(Rn[1]==='/'){number_format_map[NFtag.name]=NF;if((tmp=state.pop())[0]!==Rn[3])throw "Bad state: "+tmp;}else if(Rn[0].charAt(Rn[0].length-2)!=='/'){NF="";NFtag=parsexmltag(Rn[0],false);state.push([Rn[3],true]);}break;case'script':break;// 3.13 <office:script>
	case'libraries':break;// TODO: <ooo:libraries>
	case'automatic-styles':break;// 3.15.3 <office:automatic-styles>
	case'default-style':// TODO: <style:default-style>
	case'page-layout':break;// TODO: <style:page-layout>
	case'style':// 16.2 <style:style>
	break;case'map':break;// 16.3 <style:map>
	case'font-face':break;// 16.21 <style:font-face>
	case'paragraph-properties':break;// 17.6 <style:paragraph-properties>
	case'table-properties':break;// 17.15 <style:table-properties>
	case'table-column-properties':break;// 17.16 <style:table-column-properties>
	case'table-row-properties':break;// 17.17 <style:table-row-properties>
	case'table-cell-properties':break;// 17.18 <style:table-cell-properties>
	case'number':// 16.27.3 <number:number>
	switch(state[state.length-1][0]){case'time-style':case'date-style':tag=parsexmltag(Rn[0],false);NF+=number_formats_ods[Rn[3]][tag.style==='long'?1:0];break;}break;case'fraction':break;// TODO 16.27.6 <number:fraction>
	case'day':// 16.27.11 <number:day>
	case'month':// 16.27.12 <number:month>
	case'year':// 16.27.13 <number:year>
	case'era':// 16.27.14 <number:era>
	case'day-of-week':// 16.27.15 <number:day-of-week>
	case'week-of-year':// 16.27.16 <number:week-of-year>
	case'quarter':// 16.27.17 <number:quarter>
	case'hours':// 16.27.19 <number:hours>
	case'minutes':// 16.27.20 <number:minutes>
	case'seconds':// 16.27.21 <number:seconds>
	case'am-pm':// 16.27.22 <number:am-pm>
	switch(state[state.length-1][0]){case'time-style':case'date-style':tag=parsexmltag(Rn[0],false);NF+=number_formats_ods[Rn[3]][tag.style==='long'?1:0];break;}break;case'boolean-style':break;// 16.27.23 <number:boolean-style>
	case'boolean':break;// 16.27.24 <number:boolean>
	case'text-style':break;// 16.27.25 <number:text-style>
	case'text':// 16.27.26 <number:text>
	if(Rn[0].slice(-2)==="/>")break;else if(Rn[1]==="/")switch(state[state.length-1][0]){case'number-style':case'date-style':case'time-style':NF+=str.slice(pidx,Rn.index);break;}else pidx=Rn.index+Rn[0].length;break;case'named-range':// 9.4.12 <table:named-range>
	tag=parsexmltag(Rn[0],false);_Ref=ods_to_csf_3D(tag['cell-range-address']);var nrange={Name:tag.name,Ref:_Ref[0]+'!'+_Ref[1]}/*:any*/;if(intable)nrange.Sheet=SheetNames.length;WB.Names.push(nrange);break;case'text-content':break;// 16.27.27 <number:text-content>
	case'text-properties':break;// 16.27.27 <style:text-properties>
	case'embedded-text':break;// 16.27.4 <number:embedded-text>
	case'body':case'电子表格':break;// 3.3 16.9.6 19.726.3
	case'forms':break;// 12.25.2 13.2
	case'table-column':break;// 9.1.6 <table:table-column>
	case'table-header-rows':break;// 9.1.7 <table:table-header-rows>
	case'table-rows':break;// 9.1.12 <table:table-rows>
	/* TODO: outline levels */case'table-column-group':break;// 9.1.10 <table:table-column-group>
	case'table-header-columns':break;// 9.1.11 <table:table-header-columns>
	case'table-columns':break;// 9.1.12 <table:table-columns>
	case'null-date':break;// 9.4.2 <table:null-date> TODO: date1904
	case'graphic-properties':break;// 17.21 <style:graphic-properties>
	case'calculation-settings':break;// 9.4.1 <table:calculation-settings>
	case'named-expressions':break;// 9.4.11 <table:named-expressions>
	case'label-range':break;// 9.4.9 <table:label-range>
	case'label-ranges':break;// 9.4.10 <table:label-ranges>
	case'named-expression':break;// 9.4.13 <table:named-expression>
	case'sort':break;// 9.4.19 <table:sort>
	case'sort-by':break;// 9.4.20 <table:sort-by>
	case'sort-groups':break;// 9.4.22 <table:sort-groups>
	case'tab':break;// 6.1.4 <text:tab>
	case'line-break':break;// 6.1.5 <text:line-break>
	case'span':break;// 6.1.7 <text:span>
	case'p':case'文本串':// 5.1.3 <text:p>
	if(['master-styles'].indexOf(state[state.length-1][0])>-1)break;if(Rn[1]==='/'&&(!ctag||!ctag['string-value'])){var ptp=parse_text_p(str.slice(textpidx,Rn.index));textp=(textp.length>0?textp+"\n":"")+ptp[0];}else {parsexmltag(Rn[0],false);textpidx=Rn.index+Rn[0].length;}break;// <text:p>
	case's':break;// <text:s>
	case'database-range':// 9.4.15 <table:database-range>
	if(Rn[1]==='/')break;try{_Ref=ods_to_csf_3D(parsexmltag(Rn[0])['target-range-address']);Sheets[_Ref[0]]['!autofilter']={ref:_Ref[1]};}catch(e){/* empty */}break;case'date':break;// <*:date>
	case'object':break;// 10.4.6.2 <draw:object>
	case'title':case'标题':break;// <*:title> OR <uof:标题>
	case'desc':break;// <*:desc>
	case'binary-data':break;// 10.4.5 TODO: b64 blob
	/* 9.2 Advanced Tables */case'table-source':break;// 9.2.6
	case'scenario':break;// 9.2.6
	case'iteration':break;// 9.4.3 <table:iteration>
	case'content-validations':break;// 9.4.4 <table:
	case'content-validation':break;// 9.4.5 <table:
	case'help-message':break;// 9.4.6 <table:
	case'error-message':break;// 9.4.7 <table:
	case'database-ranges':break;// 9.4.14 <table:database-ranges>
	case'filter':break;// 9.5.2 <table:filter>
	case'filter-and':break;// 9.5.3 <table:filter-and>
	case'filter-or':break;// 9.5.4 <table:filter-or>
	case'filter-condition':break;// 9.5.5 <table:filter-condition>
	case'list-level-style-bullet':break;// 16.31 <text:
	case'list-level-style-number':break;// 16.32 <text:
	case'list-level-properties':break;// 17.19 <style:
	/* 7.3 Document Fields */case'sender-firstname':// 7.3.6.2
	case'sender-lastname':// 7.3.6.3
	case'sender-initials':// 7.3.6.4
	case'sender-title':// 7.3.6.5
	case'sender-position':// 7.3.6.6
	case'sender-email':// 7.3.6.7
	case'sender-phone-private':// 7.3.6.8
	case'sender-fax':// 7.3.6.9
	case'sender-company':// 7.3.6.10
	case'sender-phone-work':// 7.3.6.11
	case'sender-street':// 7.3.6.12
	case'sender-city':// 7.3.6.13
	case'sender-postal-code':// 7.3.6.14
	case'sender-country':// 7.3.6.15
	case'sender-state-or-province':// 7.3.6.16
	case'author-name':// 7.3.7.1
	case'author-initials':// 7.3.7.2
	case'chapter':// 7.3.8
	case'file-name':// 7.3.9
	case'template-name':// 7.3.9
	case'sheet-name':// 7.3.9
	break;case'event-listener':break;/* TODO: FODS Properties */case'initial-creator':case'creation-date':case'print-date':case'generator':case'document-statistic':case'user-defined':case'editing-duration':case'editing-cycles':break;/* TODO: FODS Config */case'config-item':break;/* TODO: s