/*
 * jena-utilities v1.0.0
 * Deal with NMR forms
 * https://github.com/zakodium/jena-utilities#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.JenaUtilities = {}));
})(this, (function (exports) { 'use strict';

    const instruments = {
      instrument1: {
        solventMapping: {
          CDCl3: 'CDCl3',
          'DMSO-d6': 'DMSO-d6'
        },
        defaultExperiments: {
          '1h': 'N proton',
          '13c': 'N carbon',
          cosy: 'N COSY'
        },
        experiments: [{
          experiment: 'N proton',
          enabled: true,
          parameters: [{
            label: 'd1',
            value: ''
          }, {
            label: 'd9',
            value: ''
          }]
        }, {
          experiment: 'N carbon',
          parameters: [{
            label: 'ds',
            value: ''
          }, {
            label: 'ns',
            value: ''
          }, {
            label: 'd1',
            value: ''
          }]
        }, {
          experiment: 'N COSY',
          parameters: []
        }]
      },
      instrument2: {
        solventMapping: {
          CDCl3: 'Chloroform-d',
          'DMSO-d6': 'Dimethyl sulfoxide-d6'
        },
        defaultExperiments: {
          '1h': 'H1 NMR',
          '13c': 'C13 NMR',
          hsqc: 'HSQC'
        },
        experiments: [{
          experiment: 'H1 NMR',
          parameters: [{
            label: 'd1',
            value: ''
          }, {
            label: 'number of scans',
            value: ''
          }]
        }, {
          experiment: 'C13 NMR',
          parameters: [{
            label: 'ds',
            value: ''
          }, {
            label: 'ns',
            value: ''
          }, {
            label: 'd1',
            value: ''
          }]
        }, {
          experiment: 'HSQC',
          parameters: []
        }]
      }
    };

    function collectFormData(form) {
      const formData = {};
      const experimentsDiv = form.document.querySelector(`#${form.id}`);
      const activeTab = experimentsDiv.querySelector('.tab-content.active');
      const instrumentId = activeTab?.dataset.instrument;
      if (!instrumentId) return formData;
      formData.instrument = instrumentId;
      formData.experiments = [];
      const experiments = activeTab.querySelectorAll('.experiment-item');
      for (const exp of experiments) {
        const checkbox = exp.querySelector('.experiment-toggle');
        const experimentData = {
          name: checkbox.dataset.experiment,
          enabled: checkbox.checked,
          parameters: {}
        };
        for (const input of exp.querySelectorAll('.parameter-input')) {
          experimentData.parameters[input.dataset.label] = input.value;
        }
        formData.experiments.push(experimentData);
      }
      return formData;
    }
    function createExperimentElement(document, experiment) {
      const div = document.createElement('div');
      div.className = 'experiment-item';
      const header = document.createElement('div');
      header.className = 'experiment-header';
      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.className = 'experiment-toggle';
      checkbox.dataset.experiment = experiment.experiment;
      checkbox.checked = experiment.enabled || false;
      const label = document.createElement('label');
      label.textContent = experiment.experiment;
      label.style.cursor = 'pointer';
      header.append(checkbox);
      header.append(label);
      const paramsDiv = document.createElement('div');
      paramsDiv.className = 'experiment-parameters';
      paramsDiv.style.display = experiment.enabled || false ? 'block' : 'none';
      for (const param of experiment.parameters) {
        const paramDiv = document.createElement('div');
        paramDiv.className = 'parameter-item';
        const paramLabel = document.createElement('label');
        paramLabel.textContent = param.label;
        const input = document.createElement('input');
        input.type = 'text';
        input.className = 'parameter-input';
        input.dataset.label = param.label;
        input.value = param.value;
        paramDiv.append(paramLabel);
        paramDiv.append(input);
        paramsDiv.append(paramDiv);
      }
      div.append(header);
      div.append(paramsDiv);
      return div;
    }
    function createTabContent(form, instrumentId, instrumentData) {
      const content = form.document.createElement('div');
      content.className = 'tab-content';
      content.dataset.instrument = instrumentId;
      for (const experiment of instrumentData.experiments) {
        content.append(createExperimentElement(form.document, experiment));
      }
      return content;
    }
    function switchTab(form, instrumentId) {
      const experimentsDiv = form.document.querySelector(`#${form.id}`);
      for (const tab of experimentsDiv.querySelectorAll('.tab')) {
        tab.classList.toggle('active', tab.dataset.instrument === instrumentId);
      }
      for (const content of experimentsDiv.querySelectorAll('.tab-content')) {
        content.classList.toggle('active', content.dataset.instrument === instrumentId);
      }
      form.onFormChange(collectFormData(form));
    }
    function fillForm(form) {
      const experimentsDiv = form.document.querySelector(`#${form.id}`);
      const tabsContainer = experimentsDiv.querySelector('#tabs');
      const formContainer = experimentsDiv.querySelector('#form-container');
      const instrumentIds = Object.keys(instruments);
      for (const [index, instrumentId] of instrumentIds.entries()) {
        const tab = form.document.createElement('button');
        tab.className = 'tab';
        tab.dataset.instrument = instrumentId;
        tab.textContent = instrumentId;
        if (index === 0) tab.classList.add('active');
        tabsContainer.append(tab);
        const content = createTabContent(form, instrumentId, instruments[instrumentId]);
        if (index === 0) content.classList.add('active');
        formContainer.append(content);
      }
      // Event delegation at experiments div level
      experimentsDiv.addEventListener('click', e => {
        if (e.target.classList.contains('tab')) {
          switchTab(form, e.target.dataset.instrument);
        } else if (e.target.classList.contains('experiment-header') || e.target.closest('.experiment-header label')) {
          const checkbox = e.currentTarget.querySelector('.experiment-item')?.querySelector('.experiment-toggle');
          if (checkbox && (e.target.tagName === 'LABEL' || e.target.closest('.experiment-header'))) {
            const experimentItem = e.target.closest('.experiment-item');
            if (experimentItem) {
              const checkboxInItem = experimentItem.querySelector('.experiment-toggle');
              if (checkboxInItem) {
                checkboxInItem.click();
              }
            }
          }
        }
      });
      experimentsDiv.addEventListener('change', e => {
        if (e.target.classList.contains('experiment-toggle')) {
          const experimentItem = e.target.closest('.experiment-item');
          const params = experimentItem.querySelector('.experiment-parameters');
          params.style.display = e.target.checked ? 'block' : 'none';
          form.onFormChange(collectFormData(form));
        }
      });
      experimentsDiv.addEventListener('input', e => {
        if (e.target.classList.contains('parameter-input')) {
          form.onFormChange(collectFormData(form));
        }
      });
    }

    function getStyle(id) {
      return `#${id} {
  max-width: 800px;
  margin: 0 auto;
  background-color: white;
  padding: 20px;
  border-radius: 8px;
  box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
  box-sizing: border-box;
}

#${id} .tabs {
  display: flex;
  gap: 5px;
  border-bottom: 2px solid #ddd;
  margin-bottom: 20px;
}

#${id} .tab {
  padding: 10px 20px;
  background-color: #f0f0f0;
  border: none;
  border-radius: 5px 5px 0 0;
  cursor: pointer;
  font-size: 14px;
  transition: background-color 0.3s;
}

#${id} .tab:hover {
  background-color: #e0e0e0;
}

#${id} .tab.active {
  background-color: #007bff;
  color: white;
}

#${id} .tab-content {
  display: none;
}

#${id} .tab-content.active {
  display: block;
}

#${id} .experiment-item {
  border: 1px solid #ddd;
  border-radius: 5px;
  padding: 15px;
  margin-bottom: 10px;
  background-color: #fafafa;
}

#${id} .experiment-header {
  display: flex;
  align-items: center;
  gap: 10px;
}

#${id} .experiment-toggle {
  width: 18px;
  height: 18px;
  cursor: pointer;
}

#${id} .experiment-header label {
  font-weight: bold;
  color: #333;
  user-select: none;
}

#${id} .experiment-parameters {
  margin-top: 15px;
  padding-left: 28px;
}

#${id} .parameter-item {
  display: flex;
  align-items: center;
  gap: 10px;
  margin-bottom: 10px;
}

#${id} .parameter-item label {
  min-width: 150px;
  color: #555;
}

#${id} .parameter-input {
  flex: 1;
  padding: 8px;
  border: 1px solid #ccc;
  border-radius: 4px;
  font-size: 14px;
}

#${id} .parameter-input:focus {
  outline: none;
  border-color: #007bff;
}
`;
    }

    function getForm(options = {}) {
      const {
        id = `instrument-${crypto.randomUUID()}`
      } = options;
      return {
        id,
        html: `<style>${getStyle(id)}</style>
<div id="${id}">
  <div id="tabs" class="tabs"></div>
  <div id="form-container" class="form-container"></div>
</div>`
      };
    }

    /**
     * Generate iconNMR text file from requests
     * @param requests - array of PlateRequest objects
     * @param options - options for generating the file
     * @returns iconNMR text file
     */
    function generateIconNMRFile(requests, options = {}) {
      const {
        eol = '\r\n',
        deleteExistingHolder = false,
        autoSubmit = false
      } = options;
      const textFile = [];
      for (const request of requests) {
        let experimentNumberCounter = 1;
        const {
          holder
        } = request;
        if (deleteExistingHolder) {
          textFile.push(`USER ${request.user}`, `HOLDER ${holder}`, 'DELETE'); // this is required to delete already existing entries
        }
        textFile.push(`USER ${request.user}`, `HOLDER ${holder}`);
        if (!autoSubmit) textFile.push('NO_SUBMIT');
        textFile.push(`NAME ${request.name}`, `TITLE ${request.title}`);
        for (const experiment of request.experiments) {
          let experimentNumber = experiment.experimentNumber || experimentNumberCounter++;
          textFile.push(`EXPNO ${experimentNumber++}`, `SOLVENT ${experiment.solvent}`, `EXPERIMENT ${experiment.experiment}`);
          if (experiment.parameters && experiment.parameters.length > 0) {
            const parameters = [];
            for (const parameter of experiment.parameters) {
              parameters.push(parameter.label, parameter.value);
            }
            textFile.push(`PARAMETERS ${parameters.join(',')}`);
          }
        }
        textFile.push('');
      }
      return textFile.join(eol);
    }

    function updateIconNMR(form) {
      const {
        experiments,
        iconNMROptions
      } = form;
      const {
        user,
        name,
        title,
        holder,
        solvent,
        deleteExistingHolder,
        autoSubmit
      } = iconNMROptions;
      const data = {
        user,
        name,
        title,
        holder,
        experiments: []
      };
      for (const experiment of experiments) {
        data.experiments.push({
          experiment: experiment.name,
          solvent,
          parameters: experiment.parameters || []
        });
      }
      form.iconNMR = generateIconNMRFile([data], {
        deleteExistingHolder,
        autoSubmit
      });
    }

    /* eslint-disable unicorn/prefer-structured-clone */
    class ExperimentForm {
      document;
      id;
      html;
      instrument = null;
      experiments = [];
      iconNMR;
      iconNMROptions;
      onFormChange;
      constructor(document, options = {}) {
        this.document = document;
        const form = getForm(options);
        this.id = options.id ?? form.id;
        this.html = form.html;
        this.iconNMR = '';
        this.iconNMROptions = {
          user: 'defaultUser',
          name: 'defaultName',
          title: 'defaultTitle',
          holder: 0,
          solvent: 'defaultSolvent',
          deleteExistingHolder: false,
          autoSubmit: false
        };
        updateIconNMR(this);
        this.onFormChange = formData => {
          this.instrument = formData.instrument;
          this.experiments = JSON.parse(JSON.stringify(formData.experiments)).filter(experiment => experiment.enabled);
          updateIconNMR(this);
          if (options.onFormChange) {
            options.onFormChange(formData);
          } else {
            console.log('Form changed:', formData);
          }
        };
      }
      fillForm() {
        fillForm(this);
      }
    }

    exports.ExperimentForm = ExperimentForm;

}));
//# sourceMappingURL=jena-utilities.umd.js.map
