/**
 * xy-parser - Parse a text-file and convert it to an array of XY points
 * @version v5.0.5
 * @link https://github.com/cheminfo/xy-parser#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.XYParser = {}));
})(this, (function (exports) { 'use strict';

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */
    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }
      var i = 0;
      var len = buf.length;
      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        }
        // UTF8-2 = %xC2-DF UTF8-tail
        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        }
        // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail
        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )
        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )
        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }
        return false;
      }
      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */
    function ensureString(blob, options = {}) {
      if (typeof blob === 'string') {
        return blob;
      }
      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }
      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }
    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }
        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      }
      //@ts-expect-error an ArrayBuffer is also ok
      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    /**
     * Checks if input is of type array.
     *
     * @param input - input
     * @param options
     */
    function xCheck(input, options = {}) {
      const {
        minLength
      } = options;
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      if (typeof input[0] !== 'number') {
        throw new TypeError('input must contain numbers');
      }
      if (minLength && input.length < minLength) {
        throw new Error(`input must have a length of at least ${minLength}`);
      }
    }

    /**
     * Returns the closest index of a `target`
     *
     * @param array - array of numbers
     * @param target - target
     * @param options
     * @returns - closest index
     */
    function xFindClosestIndex(array, target, options = {}) {
      const {
        sorted = true
      } = options;
      if (sorted) {
        let low = 0;
        let high = array.length - 1;
        let middle = 0;
        while (high - low > 1) {
          middle = low + (high - low >> 1);
          if (array[middle] < target) {
            low = middle;
          } else if (array[middle] > target) {
            high = middle;
          } else {
            return middle;
          }
        }
        if (low < array.length - 1) {
          if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
            return low;
          } else {
            return low + 1;
          }
        } else {
          return low;
        }
      } else {
        let index = 0;
        let diff = Number.POSITIVE_INFINITY;
        for (let i = 0; i < array.length; i++) {
          const currentDiff = Math.abs(array[i] - target);
          if (currentDiff < diff) {
            diff = currentDiff;
            index = i;
          }
        }
        return index;
      }
    }

    /**
     * Returns an object with {fromIndex, toIndex} for a specific from / to
     *
     * @param x - array of numbers
     * @param options - Options
     */
    function xGetFromToIndex(x, options = {}) {
      let {
        fromIndex,
        toIndex
      } = options;
      const {
        from,
        to
      } = options;
      if (typeof fromIndex === 'undefined') {
        if (typeof from !== 'undefined') {
          fromIndex = xFindClosestIndex(x, from);
        } else {
          fromIndex = 0;
        }
      }
      if (typeof toIndex === 'undefined') {
        if (typeof to !== 'undefined') {
          toIndex = xFindClosestIndex(x, to);
        } else {
          toIndex = x.length - 1;
        }
      }
      if (fromIndex < 0) fromIndex = 0;
      if (toIndex < 0) toIndex = 0;
      if (fromIndex >= x.length) fromIndex = x.length - 1;
      if (toIndex >= x.length) toIndex = x.length - 1;
      if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
      return {
        fromIndex,
        toIndex
      };
    }

    /**
     * Computes the maximal value of an array of values
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMaxValue(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let maxValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] > maxValue) {
          maxValue = array[i];
        }
      }
      return maxValue;
    }

    /**
     * Returns true if x is monotonic.
     *
     * @param array - array of numbers.
     * @returns 1 if monotonic increasing, -1 if monotonic decreasing, 0 if not monotonic.
     */
    function xIsMonotonic(array) {
      if (array.length <= 2) {
        return 1;
      }
      if (array[0] === array[1]) {
        // maybe a constant series
        for (let i = 1; i < array.length - 1; i++) {
          if (array[i] !== array[i + 1]) return 0;
        }
        return 1;
      }
      if (array[0] < array[array.length - 1]) {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] >= array[i + 1]) return 0;
        }
        return 1;
      } else {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] <= array[i + 1]) return 0;
        }
        return -1;
      }
    }

    /**
     * Verify that `data` is an object of x,y arrays.
     * Throws an error if it's not.
     *
     * @param data
     * @param options
     */
    function xyCheck(data, options = {}) {
      const {
        minLength
      } = options;
      if (data === null || typeof data !== 'object' ||
      // @ts-expect-error Typechecking
      !isAnyArray(data.x) ||
      // @ts-expect-error Typechecking
      !isAnyArray(data.y)) {
        throw new Error('data must be an object of x and y arrays');
      }
      // @ts-expect-error Typechecking
      if (data.x.length !== data.y.length) {
        throw new Error('the x and y arrays must have the same length');
      }
      // @ts-expect-error Typechecking
      if (minLength && data.x.length < minLength) {
        throw new Error(`data.x must have a length of at least ${minLength}`);
      }
    }

    /**
     * This function performs a quick sort of the x array while transforming the y array to preserve the coordinates.
     *
     * @param data - Object that contains property x (Array) and y (Array)
     */
    function xySortX(data) {
      const {
        x,
        y
      } = data;
      if (xIsMonotonic(x) && x.length > 1) {
        if (x[0] < x[1]) {
          return {
            x: Float64Array.from(x),
            y: Float64Array.from(y)
          };
        } else {
          return {
            x: Float64Array.from(x).reverse(),
            y: Float64Array.from(y).reverse()
          };
        }
      }
      const xyObject = x.map((val, index) => ({
        x: val,
        y: y[index]
      })).sort((a, b) => a.x - b.x);
      const response = {
        x: new Float64Array(x.length),
        y: new Float64Array(y.length)
      };
      for (let i = 0; i < x.length; i++) {
        response.x[i] = xyObject[i].x;
        response.y[i] = xyObject[i].y;
      }
      return response;
    }

    /**
     * Ensure x values are unique
     *
     * @param data - Object that contains property x (Array) and y (Array)
     * @param options - Object containing a property algorithm (can be 'sum' or 'average', the latter being the default value), and a property isSorted (boolean indicating if the x-array is sorted).
     */
    function xyUniqueX(data, options = {}) {
      xyCheck(data);
      const {
        algorithm = 'average',
        isSorted = true
      } = options;
      if (!isSorted) {
        data = xySortX(data);
      }
      switch (algorithm) {
        case 'average':
          return average(data);
        case 'sum':
          return sum(data);
        default:
          throw new Error(`unknown algorithm: ${String(algorithm)}`);
      }
    }
    /**
     * Average.
     *
     * @param data - Input.
     * @returns Result.
     */
    function average(data) {
      const x = [];
      const y = [];
      let cumulativeY = data.y[0];
      let divider = 1;
      for (let i = 1; i < data.x.length; i++) {
        if (!(data.x[i] === data.x[i - 1])) {
          x.push(data.x[i - 1]);
          y.push(cumulativeY / divider);
          cumulativeY = 0;
          divider = 0;
        }
        cumulativeY += data.y[i];
        divider++;
      }
      x.push(data.x[data.x.length - 1]);
      y.push(cumulativeY / divider);
      return {
        x,
        y
      };
    }
    /**
     * Sum.
     *
     * @param data - Input.
     * @returns Result.
     */
    function sum(data) {
      const x = [];
      const y = [];
      let cumulativeY = data.y[0];
      for (let i = 1; i < data.x.length; i++) {
        if (!(data.x[i] === data.x[i - 1])) {
          x.push(data.x[i - 1]);
          y.push(cumulativeY);
          cumulativeY = 0;
        }
        cumulativeY += data.y[i];
      }
      x.push(data.x[data.x.length - 1]);
      y.push(cumulativeY);
      return {
        x,
        y
      };
    }

    /**
     * General internal parsing function
     * @param text - Csv or tsv strings.
     * @param options - Parsing options
     * @returns parsed text file with column information
     */
    function parse(text, options = {}) {
      const {
        rescale = false,
        uniqueX = false,
        bestGuess = false,
        //@ts-expect-error old library used this property and we want to throw an error so that people are forced to migrate
        keepInfo
      } = options;
      let {
        xColumn = 0,
        yColumn = 1,
        numberColumns = Number.MAX_SAFE_INTEGER,
        maxNumberColumns = Number.MAX_SAFE_INTEGER,
        minNumberColumns = 2
      } = options;
      if (keepInfo !== undefined) {
        throw new Error('keepInfo has been deprecated, pelase use the new method parseXYAndKeepInfo');
      }
      text = ensureString(text);
      maxNumberColumns = Math.max(maxNumberColumns, xColumn + 1, yColumn + 1);
      minNumberColumns = Math.max(xColumn + 1, yColumn + 1, minNumberColumns);
      const lines = text.split(/[\r\n]+/);
      let matrix = [];
      const info = [];
      let position = 0;
      lines.forEach(line => {
        line = line.trim();
        // we will consider only lines that contains only numbers
        if (/[0-9]+/.test(line) && /^[0-9eE,;. \t+-]+$/.test(line)) {
          let fields = line.split(/,[; \t]+|[; \t]+/);
          if (fields.length === 1) {
            fields = line.split(/[,; \t]+/);
          }
          if (fields && fields.length >= minNumberColumns &&
          // we filter lines that have not enough or too many columns
          fields.length <= maxNumberColumns) {
            matrix.push(fields.map(value => parseFloat(value.replace(',', '.'))));
            position++;
          }
        } else if (line) {
          info.push({
            position,
            value: line
          });
        }
      });
      if (bestGuess) {
        if (matrix[0] && matrix[0].length === 3 && options.xColumn === undefined && options.yColumn === undefined) {
          // is the first column a seuqnetial number ?
          let skipFirstColumn = true;
          for (let i = 0; i < matrix.length - 1; i++) {
            if (Math.abs(matrix[i][0] - matrix[i + 1][0]) !== 1) {
              skipFirstColumn = false;
            }
          }
          if (skipFirstColumn) {
            xColumn = 1;
            yColumn = 2;
          }
        }
        if (matrix[0] && matrix[0].length > 3) {
          const xs = [];
          for (const row of matrix) {
            for (let i = xColumn; i < row.length; i += 2) {
              xs.push(row[i]);
            }
          }
          if (xIsMonotonic(xs)) {
            numberColumns = 2;
          }
        }
      }
      if (numberColumns) {
        const newMatrix = [];
        for (const row of matrix) {
          for (let i = 0; i < row.length; i += numberColumns) {
            newMatrix.push(row.slice(i, i + numberColumns));
          }
        }
        matrix = newMatrix;
      }
      let result = {
        x: matrix.map(row => row[xColumn]),
        y: matrix.map(row => row[yColumn])
      };
      if (uniqueX) {
        result = xyUniqueX(result, {
          algorithm: 'sum'
        });
      }
      if (rescale) {
        const maxY = xMaxValue(result.y);
        for (let i = 0; i < result.y.length; i++) {
          result.y[i] /= maxY;
        }
      }
      return {
        info,
        data: result
      };
    }

    /**
     * Parse a text-file and convert it to an object {x:[], y:[]}
     * @param text - Csv or tsv strings.
     * @param options - Parsing options
     * @returns - The parsed data
     */
    function parseXY(text, options = {}) {
      return parse(text, options).data;
    }
    /**
     * Parse a text-file and returns the parsed data and information about the columns
     * @param text - Csv or tsv strings.
     * @param options - Parsing options
     * @returns - The parsed data with information about the columns
     */
    function parseXYAndKeepInfo(text, options = {}) {
      return parse(text, options);
    }

    exports.parseXY = parseXY;
    exports.parseXYAndKeepInfo = parseXYAndKeepInfo;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=xy-parser.js.map
