/**
 * nmr-parser - Read and convert any NMR file
 * @version v2.0.0
 * @link https://github.com/cheminfo/nmr-parser#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.NMRparser = {}));
})(this, (function (exports) { 'use strict';

    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array).
     *
     * @param {any} value - Object to check.
     * @returns {boolean} True if the object is an array.
     */

    function isAnyArray(value) {
      return toString.call(value).endsWith('Array]');
    }

    /**
     * a number that correspond to a type of numeric
     * @typedef {number} numericType
     * @const
     */
    const numericTypeTable = {
      0: 'uint8',
      1: 'uint16',
      2: 'uint32',
      3: 'uint64',
      4: 'int8',
      5: 'int16',
      6: 'int32',
      7: 'int64',
      8: 'float32',
      9: 'float64',
      10: 'complex64',
      11: 'complex128'
    };
    /**
     * a number that corresponds to a type of quantity
     * @typedef {number} quantityType
     * @const
     */

    const quantityTypeTable = {
      0: 'scalar',
      1: 'vector',
      2: 'matrix',
      3: 'symetricMatrix',
      4: 'pixel'
    };

    /**
     * a class for dependent variable
     * @param {object || array} data - the dependent variable
     * @param {numericType} numericType - a number that correspond to a type of numeric used to store the components
     * @param {object} [options] - an object with options (name, unit, quantityName, componentLabels, sparseSampling, application, description)
     * @param {string} [options.name] - a name of the dependent variable
     * @param {string} [options.unit] - the unit of the dependent variable
     * @param {string} [options.quantityName] - a name of the quantity
     * @param {array} [options.componentLabels] - an array of labels for each component of the dependent variable
     * @return {object} - an dependent variable
     */

    function formatDependentVariable(data, numericType) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      let {
        quantityType = 0,
        encoding = 'none',
        name = '',
        unit = '',
        quantityName = '',
        componentLabels = [],
        sparseSampling = {},
        from = 0,
        to = -1
      } = options;
      let components;

      if (isAnyArray(data)) {
        throw new Error('not yet implemented');
      } else if (Object.keys(data).length === 2) {
        components = fromReIm(data, from, to);
      }

      if (componentLabels.length === 0) {
        componentLabels = components.componentLabels;
      }

      return {
        type: 'internal',
        quantityType: quantityTypeTable[quantityType],
        numericType: numericTypeTable[numericType],
        encoding,
        name,
        unit,
        quantityName,
        componentLabels,
        sparseSampling,
        description: options.description || '',
        application: options.application || '',
        components: components.components,
        dataLength: components.dataLength
      };
    }
    /**
     * import object {re:[], im:[]} to component
     * @param {object} reIm - a reIm object to import
     * @param {number} from - lower limit
     * @param {number} to - upper limit
     * @return {array} - components
     */

    function fromReIm(reIm, from, to) {
      let dataLength = [];
      let componentLabels = [];
      let components = [];

      if (isAnyArray(reIm.re) & isAnyArray(reIm.im)) {
        if (typeof reIm.re[0] === 'number') {
          // if 1D
          dataLength[0] = setLengthComplex(from[0], to[0], reIm.re.length);
          let component = new Float64Array(dataLength[0]);

          for (let i = 0; i < dataLength[0]; i += 2) {
            let idx = i + from[0] * 2;
            component[i] = reIm.re[idx / 2];
            component[i + 1] = reIm.im[idx / 2];
          }

          components.push(component);
          componentLabels.push('complex');
        } else if (isAnyArray(reIm.re[0])) {
          // if 2D
          dataLength[0] = setLength(from[1], to[1], reIm.re.length);
          dataLength[1] = setLengthComplex(from[0], to[0], reIm.re[0].length);

          for (let j = 0; j < dataLength[0]; j++) {
            let component = new Float64Array(dataLength[1]);

            for (let i = 0; i < dataLength[1]; i += 2) {
              let idx = i + from[0] * 2;
              component[i] = reIm.re[j][idx / 2];
              component[i + 1] = reIm.im[j][idx / 2];
            }

            components.push(component);
          }
        } else {
          throw new Error('check your object');
        }
      } else if (isAnyArray(reIm.re.re)) {
        dataLength[0] = reIm.re.re.length * 2;
        let re = fromReIm(reIm.re, from, to).components;
        let im = fromReIm(reIm.im, from, to).components;

        for (let j = 0; j < dataLength[0] / 2; j++) {
          components.push(re[j]);
          components.push(im[j]);
        }
      } else {
        throw new Error('check the dimension or the type of data in your array');
      }

      return {
        dataLength,
        componentLabels,
        components
      };
    }

    function setLength(from, to, length) {
      if (to - from + 1 < length) {
        return to - from + 1;
      } else {
        return length;
      }
    }

    function setLengthComplex(from, to, length) {
      if (to - from + 1 < length) {
        return (to - from + 1) * 2;
      } else {
        return length * 2;
      }
    } // /**
    //  * add component to components from 1D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  * @return {Float64Array} - component
    //  */
    // function add1DArray(array) {
    //   let component;
    //   component = new Float64Array(array.length);
    //   for (let i = 0; i < array.length; i++) {
    //     component[i] = array[i];
    //   }
    //   return component;
    // }
    // /**
    //  * import component to InternalDEPENDENTVAR class object from 1D or 2D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  */
    // function fromArray(array) {
    //   this.dataLength[0] = array.length;
    //   if (typeof array[0] === 'number') {
    //     this.components = [this.add1DArray(array)];
    //   } else if (Array.isArray(array[0])) {
    //     this.dataLength[1] = array[0].length;
    //     for (let j = 0; j < this.dataLength[1]; j++) {
    //       this.components.push(this.add1DArray(array[j]));
    //     }
    //   } else {
    //     throw new Error('check the dimension or the type of data in your array');
    //   }
    //   return this;
    // }

    /**
     *
     * @param {*} label
     * @param {*} count
     * @param {*} increment
     * @param {*} options
     */
    function formatLinearDimension(label, count, increment) {
      let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
      return {
        label: String(label),
        count: Number(count),
        increment,
        type: 'linear',
        description: String(options.description) || '',
        application: options.application || {},
        coordinatesOffset: options.coordinatesOffset || 0,
        originOffset: options.originOffset || 0,
        quantityName: String(options.quantityName) || '',
        reciprocal: options.reciprocal || {},
        period: options.period || 0,
        complexFFT: options.complexFFT || false
      };
    }

    // eslint-disable-next-line import/no-unassigned-import
    const decoder = new TextDecoder('utf-8');
    function decode(bytes) {
      return decoder.decode(bytes);
    }
    const encoder = new TextEncoder();
    function encode(str) {
      return encoder.encode(str);
    }

    const defaultByteLength = 1024 * 8;
    class IOBuffer {
      /**
       * @param data - The data to construct the IOBuffer with.
       * If data is a number, it will be the new buffer's length<br>
       * If data is `undefined`, the buffer will be initialized with a default length of 8Kb<br>
       * If data is an ArrayBuffer, SharedArrayBuffer, an ArrayBufferView (Typed Array), an IOBuffer instance,
       * or a Node.js Buffer, a view will be created over the underlying ArrayBuffer.
       * @param options
       */
      constructor() {
        let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : defaultByteLength;
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let dataIsGiven = false;

        if (typeof data === 'number') {
          data = new ArrayBuffer(data);
        } else {
          dataIsGiven = true;
          this.lastWrittenByte = data.byteLength;
        }

        const offset = options.offset ? options.offset >>> 0 : 0;
        const byteLength = data.byteLength - offset;
        let dvOffset = offset;

        if (ArrayBuffer.isView(data) || data instanceof IOBuffer) {
          if (data.byteLength !== data.buffer.byteLength) {
            dvOffset = data.byteOffset + offset;
          }

          data = data.buffer;
        }

        if (dataIsGiven) {
          this.lastWrittenByte = byteLength;
        } else {
          this.lastWrittenByte = 0;
        }

        this.buffer = data;
        this.length = byteLength;
        this.byteLength = byteLength;
        this.byteOffset = dvOffset;
        this.offset = 0;
        this.littleEndian = true;
        this._data = new DataView(this.buffer, dvOffset, byteLength);
        this._mark = 0;
        this._marks = [];
      }
      /**
       * Checks if the memory allocated to the buffer is sufficient to store more
       * bytes after the offset.
       * @param byteLength - The needed memory in bytes.
       * @returns `true` if there is sufficient space and `false` otherwise.
       */


      available() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return this.offset + byteLength <= this.length;
      }
      /**
       * Check if little-endian mode is used for reading and writing multi-byte
       * values.
       * @returns `true` if little-endian mode is used, `false` otherwise.
       */


      isLittleEndian() {
        return this.littleEndian;
      }
      /**
       * Set little-endian mode for reading and writing multi-byte values.
       */


      setLittleEndian() {
        this.littleEndian = true;
        return this;
      }
      /**
       * Check if big-endian mode is used for reading and writing multi-byte values.
       * @returns `true` if big-endian mode is used, `false` otherwise.
       */


      isBigEndian() {
        return !this.littleEndian;
      }
      /**
       * Switches to big-endian mode for reading and writing multi-byte values.
       */


      setBigEndian() {
        this.littleEndian = false;
        return this;
      }
      /**
       * Move the pointer n bytes forward.
       * @param n - Number of bytes to skip.
       */


      skip() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        this.offset += n;
        return this;
      }
      /**
       * Move the pointer to the given offset.
       * @param offset
       */


      seek(offset) {
        this.offset = offset;
        return this;
      }
      /**
       * Store the current pointer offset.
       * @see {@link IOBuffer#reset}
       */


      mark() {
        this._mark = this.offset;
        return this;
      }
      /**
       * Move the pointer back to the last pointer offset set by mark.
       * @see {@link IOBuffer#mark}
       */


      reset() {
        this.offset = this._mark;
        return this;
      }
      /**
       * Push the current pointer offset to the mark stack.
       * @see {@link IOBuffer#popMark}
       */


      pushMark() {
        this._marks.push(this.offset);

        return this;
      }
      /**
       * Pop the last pointer offset from the mark stack, and set the current
       * pointer offset to the popped value.
       * @see {@link IOBuffer#pushMark}
       */


      popMark() {
        const offset = this._marks.pop();

        if (offset === undefined) {
          throw new Error('Mark stack empty');
        }

        this.seek(offset);
        return this;
      }
      /**
       * Move the pointer offset back to 0.
       */


      rewind() {
        this.offset = 0;
        return this;
      }
      /**
       * Make sure the buffer has sufficient memory to write a given byteLength at
       * the current pointer offset.
       * If the buffer's memory is insufficient, this method will create a new
       * buffer (a copy) with a length that is twice (byteLength + current offset).
       * @param byteLength
       */


      ensureAvailable() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;

        if (!this.available(byteLength)) {
          const lengthNeeded = this.offset + byteLength;
          const newLength = lengthNeeded * 2;
          const newArray = new Uint8Array(newLength);
          newArray.set(new Uint8Array(this.buffer));
          this.buffer = newArray.buffer;
          this.length = this.byteLength = newLength;
          this._data = new DataView(this.buffer);
        }

        return this;
      }
      /**
       * Read a byte and return false if the byte's value is 0, or true otherwise.
       * Moves pointer forward by one byte.
       */


      readBoolean() {
        return this.readUint8() !== 0;
      }
      /**
       * Read a signed 8-bit integer and move pointer forward by 1 byte.
       */


      readInt8() {
        return this._data.getInt8(this.offset++);
      }
      /**
       * Read an unsigned 8-bit integer and move pointer forward by 1 byte.
       */


      readUint8() {
        return this._data.getUint8(this.offset++);
      }
      /**
       * Alias for {@link IOBuffer#readUint8}.
       */


      readByte() {
        return this.readUint8();
      }
      /**
       * Read `n` bytes and move pointer forward by `n` bytes.
       */


      readBytes() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        const bytes = new Uint8Array(n);

        for (let i = 0; i < n; i++) {
          bytes[i] = this.readByte();
        }

        return bytes;
      }
      /**
       * Read a 16-bit signed integer and move pointer forward by 2 bytes.
       */


      readInt16() {
        const value = this._data.getInt16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 16-bit unsigned integer and move pointer forward by 2 bytes.
       */


      readUint16() {
        const value = this._data.getUint16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 32-bit signed integer and move pointer forward by 4 bytes.
       */


      readInt32() {
        const value = this._data.getInt32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit unsigned integer and move pointer forward by 4 bytes.
       */


      readUint32() {
        const value = this._data.getUint32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit floating number and move pointer forward by 4 bytes.
       */


      readFloat32() {
        const value = this._data.getFloat32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 64-bit floating number and move pointer forward by 8 bytes.
       */


      readFloat64() {
        const value = this._data.getFloat64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit signed integer number and move pointer forward by 8 bytes.
       */


      readBigInt64() {
        const value = this._data.getBigInt64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit unsigned integer number and move pointer forward by 8 bytes.
       */


      readBigUint64() {
        const value = this._data.getBigUint64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 1-byte ASCII character and move pointer forward by 1 byte.
       */


      readChar() {
        return String.fromCharCode(this.readInt8());
      }
      /**
       * Read `n` 1-byte ASCII characters and move pointer forward by `n` bytes.
       */


      readChars() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        let result = '';

        for (let i = 0; i < n; i++) {
          result += this.readChar();
        }

        return result;
      }
      /**
       * Read the next `n` bytes, return a UTF-8 decoded string and move pointer
       * forward by `n` bytes.
       */


      readUtf8() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return decode(this.readBytes(n));
      }
      /**
       * Write 0xff if the passed value is truthy, 0x00 otherwise and move pointer
       * forward by 1 byte.
       */


      writeBoolean(value) {
        this.writeUint8(value ? 0xff : 0x00);
        return this;
      }
      /**
       * Write `value` as an 8-bit signed integer and move pointer forward by 1 byte.
       */


      writeInt8(value) {
        this.ensureAvailable(1);

        this._data.setInt8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as an 8-bit unsigned integer and move pointer forward by 1
       * byte.
       */


      writeUint8(value) {
        this.ensureAvailable(1);

        this._data.setUint8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * An alias for {@link IOBuffer#writeUint8}.
       */


      writeByte(value) {
        return this.writeUint8(value);
      }
      /**
       * Write all elements of `bytes` as uint8 values and move pointer forward by
       * `bytes.length` bytes.
       */


      writeBytes(bytes) {
        this.ensureAvailable(bytes.length);

        for (let i = 0; i < bytes.length; i++) {
          this._data.setUint8(this.offset++, bytes[i]);
        }

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit signed integer and move pointer forward by 2
       * bytes.
       */


      writeInt16(value) {
        this.ensureAvailable(2);

        this._data.setInt16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit unsigned integer and move pointer forward by 2
       * bytes.
       */


      writeUint16(value) {
        this.ensureAvailable(2);

        this._data.setUint16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit signed integer and move pointer forward by 4
       * bytes.
       */


      writeInt32(value) {
        this.ensureAvailable(4);

        this._data.setInt32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit unsigned integer and move pointer forward by 4
       * bytes.
       */


      writeUint32(value) {
        this.ensureAvailable(4);

        this._data.setUint32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit floating number and move pointer forward by 4
       * bytes.
       */


      writeFloat32(value) {
        this.ensureAvailable(4);

        this._data.setFloat32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit floating number and move pointer forward by 8
       * bytes.
       */


      writeFloat64(value) {
        this.ensureAvailable(8);

        this._data.setFloat64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit signed bigint and move pointer forward by 8
       * bytes.
       */


      writeBigInt64(value) {
        this.ensureAvailable(8);

        this._data.setBigInt64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit unsigned bigint and move pointer forward by 8
       * bytes.
       */


      writeBigUint64(value) {
        this.ensureAvailable(8);

        this._data.setBigUint64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write the charCode of `str`'s first character as an 8-bit unsigned integer
       * and move pointer forward by 1 byte.
       */


      writeChar(str) {
        return this.writeUint8(str.charCodeAt(0));
      }
      /**
       * Write the charCodes of all `str`'s characters as 8-bit unsigned integers
       * and move pointer forward by `str.length` bytes.
       */


      writeChars(str) {
        for (let i = 0; i < str.length; i++) {
          this.writeUint8(str.charCodeAt(i));
        }

        return this;
      }
      /**
       * UTF-8 encode and write `str` to the current pointer offset and move pointer
       * forward according to the encoded length.
       */


      writeUtf8(str) {
        return this.writeBytes(encode(str));
      }
      /**
       * Export a Uint8Array view of the internal buffer.
       * The view starts at the byte offset and its length
       * is calculated to stop at the last written byte or the original length.
       */


      toArray() {
        return new Uint8Array(this.buffer, this.byteOffset, this.lastWrittenByte);
      }
      /**
       * Update the last written byte offset
       * @private
       */


      _updateLastWrittenByte() {
        if (this.offset > this.lastWrittenByte) {
          this.lastWrittenByte = this.offset;
        }
      }

    }

    const endianness = {
      0: 'bigEndian',
      1: 'littleEndian'
    };
    const instrumentTable = {
      0: 'NONE',
      1: 'GSX',
      2: 'ALPHA',
      3: 'ECLIPSE',
      4: 'MASS_SPEC',
      5: 'COMPILER',
      6: 'OTHER_NMR',
      7: 'UNKNOWN',
      8: 'GEMINI',
      9: 'UNITY',
      10: 'ASPECT',
      11: 'UX',
      12: 'FELIX',
      13: 'LAMBDA',
      14: 'GE_1280',
      15: 'GE_OMEGA',
      16: 'CHEMAGNETICS',
      17: 'CDFF',
      18: 'GALACTIC',
      19: 'TRIAD',
      20: 'GENERIC_NMR',
      21: 'GAMMA',
      22: 'JCAMP_DX',
      23: 'AMX',
      24: 'DMX',
      25: 'ECA',
      26: 'ALICE',
      27: 'NMR_PIPE',
      28: 'SIMPSON'
    };
    const dataTypeTable = {
      0: '64Bit Float',
      1: '32Bit Float',
      2: 'Reserved',
      3: 'Reserved'
    };
    const dataFormatTable = {
      1: 'One_D',
      2: 'Two_D',
      3: 'Three_D',
      4: 'Four_D',
      5: 'Five_D',
      6: 'Six_D',
      7: 'Seven_D',
      8: 'Eight_D',
      9: 'not for NMR data formats',
      10: 'not for NMR data formats',
      11: 'not for NMR data formats',
      12: 'Small_Two_D',
      13: 'Small_Three_D',
      14: 'Small_Four_D'
    };
    const dataAxisTypeTable = {
      0: 'None',
      //Axis is not used.
      1: 'Real',
      //Axis has real data only, no imaginary.
      2: 'TPPI',
      3: 'Complex',
      4: 'Real_Complex',

      /* Axis should be accessed as complex when it is the major axis,
                accessed as real otherwise.  This is only valid when all axes in
                use have this setting.*/
      5: 'Envelope'
      /* Behaves the same way as a Real_Complex dimension but the data
            has different meaning.  Instead of being treated as real and
            imaginary parts of a complex number, the data should be treated as minimum and maximum parts of a projection.  This is used
            for the data that results from an envelope projection.*/

    };
    const prefixTable = {
      '-8': 'Yotta',
      '-6': 'Exa',
      '-7': 'Zetta',
      '-5': 'Pecta',
      '-4': 'Tera',
      '-3': 'Giga',
      '-2': 'Mega',
      '-1': 'Kilo',
      0: 'None',
      1: 'Milli',
      2: 'Micro',
      3: 'Nano',
      4: 'Pico',
      5: 'Femto',
      6: 'Atto',
      7: 'Zepto'
    };
    const unitPrefixTable = {
      Yotta: 24,
      Exa: 21,
      Zetta: 18,
      Pecta: 15,
      Tera: 12,
      Giga: 9,
      Mega: 6,
      Kilo: 3,
      None: 0,
      Milli: -3,
      Micro: -6,
      Nano: -9,
      Pico: -12,
      Femto: -15,
      Atto: -18,
      Zepto: -21
    };
    const baseTable = {
      0: 'None',
      1: 'Abundance',
      2: 'Ampere',
      3: 'Candela',
      4: 'Celsius',
      5: 'Coulomb',
      6: 'Degree',
      7: 'Electronvolt',
      8: 'Farad',
      9: 'Sievert',
      10: 'Gram',
      11: 'Gray',
      12: 'Henry',
      13: 'Hertz',
      14: 'Kelvin',
      15: 'Joule',
      16: 'Liter',
      17: 'Lumen',
      18: 'Lux',
      19: 'Meter',
      20: 'Mole',
      21: 'Newton',
      22: 'Ohm',
      23: 'Pascal',
      24: 'Percent',
      25: 'Point',
      26: 'Ppm',
      27: 'Radian',
      28: 'Second',
      29: 'Siemens',
      30: 'Steradian',
      31: 'Tesla',
      32: 'Volt',
      33: 'Watt',
      34: 'Weber',
      35: 'Decibel',
      36: 'Dalton',
      37: 'Thompson',
      38: 'Ugeneric',
      // Treated as None, but never displayed',
      39: 'LPercent ',
      // Treated as percent for display, but different for comparison',
      40: 'PPT',
      // Parts per trillion (Private, do not use)',
      41: 'PPB ',
      // Parts per billion (Private, do not use)',
      42: 'Index'
    };
    const dataAxisRangedTable = {
      0: 'Ranged',

      /* The ruler for the axis ranges from Data_Axis_Start[n] to
            Data_Axis_Stop[n] with a step function of
                (Data_Axis_Stop[n] - Data_Axis_Start[n]) /
                (Data_Offset_Stop[n] - Data_Offset_Start[n]) */
      1: 'Listed',
      // (deprecated)

      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the ruler may be anything.*/
      2: 'Sparse',

      /*The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers must be strictly monotonically
            increasing or decreasing.*/
      3: 'Listed'
      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers do not fit definition of Sparse.*/

    };
    const valueTypeTable = {
      0: 'String',
      1: 'Integer',
      2: 'Float',
      3: 'Complex',
      4: 'Infinity'
    };

    function getPar(param, searchStr) {
      return param.paramArray.find(o => o.name === searchStr) || '';
    }
    function getDigitalFilter(param) {
      const orders = param.paramArray.find(e => e.name === 'orders');
      const factors = param.paramArray.find(e => e.name === 'factors');
      const sweep = param.paramArray.find(e => e.name === 'x_sweep');
      const acqTime = param.paramArray.find(e => e.name === 'x_acq_time');
      const nbPoints = param.paramArray.find(e => e.name === 'x_points');
      const shouldStop = [orders, factors, sweep, acqTime, nbPoints].some(e => e === undefined);

      if (shouldStop) {
        return;
      }

      const s = parseInt(orders.value.slice(0, 1), 10);
      const jump = orders.value.slice(1).length / s;
      let arg = 0;
      let factorNumber = new Int8Array(s);
      let offsetO = 1;
      let offsetF = 0;

      for (let i = 0; i < s; i++) {
        factorNumber[i] = parseInt(factors.value.slice(offsetF, offsetF + 1), 10);
        offsetF += 1;
      }

      for (let i = 0; i < s; i++) {
        let productorial = 1;

        for (let j = i; j < s; j++) {
          productorial *= factorNumber[j];
        }

        arg += (parseInt(orders.value.slice(offsetO, offsetO + jump), 10) - 1) / productorial;
        offsetO += jump;
      }

      arg /= 2;
      const delaySec = arg / sweep.value;
      return delaySec / acqTime.value * (nbPoints.value - 1);
    }
    function getMagnitude(param, searchStr) {
      let par = getPar(param, searchStr) || 'NA';

      if (par === 'NA') {
        return {
          magnitude: 'NA',
          unit: 'NA'
        };
      }

      let unit = par.unit[0].base;
      let unitMult = unitPrefixTable[par.unit[0].prefix];
      let magnitude = par.value * 10 ** unitMult;
      return {
        magnitude,
        unit
      };
    }
    function getUnit(buffer, size) {
      let unit = [];

      for (let i = 0; i < size; i++) {
        let byte = buffer.readByte();
        let prefix = prefixTable[byte >> 4];
        let power = byte & 0b00001111;
        let base = baseTable[buffer.readInt8()];
        unit.push({
          prefix,
          power,
          base
        });
      }

      return unit;
    }
    function getString(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== '\u0000') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getParamName(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== ' ') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getArray(buffer, size, format) {
      let double = [];

      for (let i = 0; i < size; i++) {
        switch (format) {
          case 'readUint32':
            double.push(buffer.readUint32());
            break;

          case 'readFloat64':
            double.push(buffer.readFloat64());
            break;

          case 'readFloat32':
            double.push(buffer.readFloat32());
            break;

          case 'readUint8':
            double.push(buffer.readUint8());
            break;

          case 'readBoolean':
            double.push(buffer.readBoolean());
            break;
        }
      }

      return double;
    }

    /**
     * A parser for 1D and 2D JDL NMR Files
     * @param {ArrayBuffer} buffer - a buffer object containing the JDL file
     * @return {Object} - an Object with converted data
     */

    function parseJEOL(buffer) {
      let ioBuffer = new IOBuffer(buffer);
      ioBuffer.setBigEndian(); // read header section

      let byte;
      let header = {};
      let byteArray = [];
      header.fileIdentifier = ioBuffer.readChars(8);
      header.endian = endianness[ioBuffer.readInt8()];
      header.majorVersion = ioBuffer.readUint8();
      header.minorVersion = ioBuffer.readUint16();
      header.dataDimensionNumber = ioBuffer.readUint8();
      header.dataDimensionExist = ioBuffer.readByte().toString(2).split('').map(x => Boolean(Number(x)));
      byte = ioBuffer.readByte();
      header.dataType = dataTypeTable[byte >> 6];
      header.dataFormat = dataFormatTable[byte & 0b00111111];
      header.dataInstrument = instrumentTable[ioBuffer.readInt8()];
      header.translate = getArray(ioBuffer, 8, 'readUint8');
      header.dataAxisType = getArray(ioBuffer, 8, 'readUint8').map(x => dataAxisTypeTable[x]);
      header.dataUnits = getUnit(ioBuffer, 8);
      header.title = getString(ioBuffer, 124);

      for (byte in getArray(ioBuffer, 4, 'readUint8')) {
        byteArray.push(dataAxisRangedTable[byte >> 4]);
        byteArray.push(dataAxisRangedTable[byte & 0b00001111]);
      }

      header.dataAxisRanged = byteArray;
      header.dataPoints = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStart = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStop = getArray(ioBuffer, 8, 'readUint32');
      header.dataAxisStart = getArray(ioBuffer, 8, 'readFloat64');
      header.dataAxisStop = getArray(ioBuffer, 8, 'readFloat64');
      byteArray = new Uint8Array(4);

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      let year = 1990 + (byteArray[0] >> 1);
      let month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      let day = byteArray[2] & 0b00011111;
      header.creationTime = {
        year,
        month,
        day
      };

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      year = 1990 + (byteArray[0] >> 1);
      month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      day = byteArray[2] & 0b00011111;
      header.revisionTime = {
        year,
        month,
        day
      };
      header.nodeName = getString(ioBuffer, 16);
      header.site = getString(ioBuffer, 128);
      header.author = getString(ioBuffer, 128);
      header.comment = getString(ioBuffer, 128);
      let dataAxisTitles = [];

      for (let i = 0; i < 8; i++) {
        dataAxisTitles.push(getString(ioBuffer, 32));
      }

      header.dataAxisTitles = dataAxisTitles;
      header.baseFreq = getArray(ioBuffer, 8, 'readFloat64');
      header.zeroPoint = getArray(ioBuffer, 8, 'readFloat64');
      header.reversed = getArray(ioBuffer, 8, 'readBoolean');
      ioBuffer.skip(3);
      header.annotationOK = Boolean(ioBuffer.readByte() >> 7);
      header.historyUsed = ioBuffer.readUint32();
      header.historyLength = ioBuffer.readUint32();
      header.paramStart = ioBuffer.readUint32();
      header.paramLength = ioBuffer.readUint32();
      header.ListStart = getArray(ioBuffer, 8, 'readUint32');
      header.ListLength = getArray(ioBuffer, 8, 'readUint32');
      header.dataStart = ioBuffer.readUint32();
      header.dataLength = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextLength = ioBuffer.readUint32();
      header.annoteStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.annoteLength = ioBuffer.readUint32();
      header.totalSize = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.unitLocation = getArray(ioBuffer, 8, 'readUint8');
      let compoundUnit = [];

      for (let i = 0; i < 2; i++) {
        let unit = [];
        let scaler = ioBuffer.readInt16();

        for (let j = 0; j < 5; j++) {
          byte = ioBuffer.readInt16();
          unit.push(byte);
        }

        compoundUnit.push({
          scaler,
          unit
        });
      }

      header.compoundUnit = compoundUnit; // section parameters (param header and array)

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      ioBuffer.seek(header.paramStart);
      let parameters = {
        parameterSize: ioBuffer.readUint32(),
        lowIndex: ioBuffer.readUint32(),
        highIndex: ioBuffer.readUint32(),
        totalSize: ioBuffer.readUint32()
      };
      let paramArray = [];

      for (let p = 0; p < parameters.highIndex + 1; p++) {
        ioBuffer.skip(4);
        let scaler = ioBuffer.readInt16();
        let unit = getUnit(ioBuffer, 5);
        ioBuffer.skip(16);
        let valueType = valueTypeTable[ioBuffer.readInt32()];
        ioBuffer.seek(ioBuffer.offset - 20);
        let value;

        switch (valueType) {
          case 'String':
            value = getParamName(ioBuffer, 16);
            break;

          case 'Integer':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          case 'Float':
            value = ioBuffer.readFloat64();
            ioBuffer.skip(8);
            break;

          case 'Complex':
            value.Real = ioBuffer.readFloat64();
            value.Imag = ioBuffer.readFloat64();
            break;

          case 'Infinity':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          default:
            ioBuffer.skip(16);
            break;
        }

        ioBuffer.skip(4);
        let name = getParamName(ioBuffer, 28);
        paramArray.push({
          name: name.toLowerCase(),
          scaler,
          unit,
          value,
          valueType
        });
      }

      parameters.paramArray = paramArray; // data section

      ioBuffer.seek(header.dataStart);

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      let data = {};
      let dataSectionCount = 1;
      let realComplex = 0;

      for (let type of header.dataAxisType) {
        if (type === 'Real_Complex' && realComplex === 0) {
          dataSectionCount += 1;
          realComplex += 1;
        }

        if (type === 'Complex') {
          dataSectionCount *= 2;
        }
      }

      if (header.dataFormat !== 'One_D' && header.dataFormat !== 'Two_D') {
        throw new Error('Only One_D and two_D data formats are implemented yet');
      }

      if (header.dataFormat === 'One_D') {
        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          if (header.dataType === '32Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat32');
          } else if (header.dataType === '64Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat64');
          }

          if (s === 0) data.re = section;
          if (s === 1) data.im = section;
        }
      }

      if (header.dataFormat === 'Two_D') {
        let me = 32;
        let dim1 = header.dataPoints[0];
        let dim2 = header.dataPoints[1]; // console.log(
        // `dim1: ${dim1},
        // dim2: ${dim2},
        // total: ${dim1 * dim2},
        // total(byte): ${dim1 * dim2 * 8},
        // total(length): ${dim1 * dim2 * 8 * dataSectionCount}
        // m size: ${dim1 / me} / ${dim2 / me}`,
        // );

        let I = dim2 / me;
        let J = dim1 / me;

        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          for (let i = 0; i < I; i++) {
            let row = [];

            for (let j = 0; j < J; j++) {
              for (let k = 0; k < me; k++) {
                if (j === 0) {
                  if (header.dataType === '32Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat32');
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat64');
                  }
                } else {
                  if (header.dataType === '32Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat32'));
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat64'));
                  }
                }
              }
            }

            if (i === 0) {
              section = row;
            } else {
              section = section.concat(row);
            }
          }

          if (dataSectionCount === 2) {
            if (s === 0) data.re = section;
            if (s === 1) data.im = section;
          }

          if (dataSectionCount === 4) {
            if (s === 0) {
              data.re = {};
              data.re.re = section;
            }

            if (s === 1) data.re.im = section;

            if (s === 2) {
              data.im = {};
              data.im.re = section;
            }

            if (s === 3) data.im.im = section;
          }
        }
      } // format output


      let nucleus = [];
      let acquisitionTime = [];
      let spectralWidth = [];
      let spectralWidthClipped = [];
      let resolution = [];
      let originFrequency = [];
      let frequencyOffset = [];
      let dataUnits = [];

      if (header.dataFormat === 'One_D' || header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'x_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'x_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'x_sweep'));
        spectralWidthClipped.push(getMagnitude(parameters, 'x_sweep_clipped'));
        resolution.push(getMagnitude(parameters, 'x_resolution'));
        originFrequency.push(getMagnitude(parameters, 'x_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'x_offset'));
        dataUnits.push(header.dataUnits[0].base);
      }

      if (header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'y_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'y_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'y_sweep'));
        resolution.push(getMagnitude(parameters, 'y_resolution'));
        originFrequency.push(getMagnitude(parameters, 'y_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'y_offset'));
        dataUnits.push(header.dataUnits[1].base);
      }

      let digest = {
        info: {
          sampleName: getPar(parameters, 'sample_id').value,
          creationTime: header.creationTime,
          revisionTime: header.revisionTime,
          author: header.author,
          comment: header.comment,
          solvent: getPar(parameters, 'solvent').value,
          temperature: getMagnitude(parameters, 'temp_get'),
          probeName: getPar(parameters, 'probe_id').value,
          fieldStrength: getMagnitude(parameters, 'field_strength'),
          experiment: getPar(parameters, 'experiment').value,
          dimension: header.dataDimensionNumber,
          nucleus,
          pulseStrength90: getMagnitude(parameters, 'x90'),
          numberOfScans: getPar(parameters, 'scans').value,
          relaxationTime: getMagnitude(parameters, 'relaxation_delay'),
          dataPoints: header.dataPoints.slice(0, header.dataDimensionNumber),
          dataOffsetStart: header.dataOffsetStart,
          dataOffsetStop: header.dataOffsetStop,
          dataUnits: dataUnits,
          dataSections: Object.keys(data),
          originFrequency,
          frequencyOffset,
          acquisitionTime,
          spectralWidth,
          spectralWidthClipped,
          dataAxisStart: header.dataAxisStart,
          dataAxisStop: header.dataAxisStop,
          resolution: resolution,
          decimationRate: getPar(parameters, 'decimation_rate').value,
          paramList: JSON.stringify(parameters.paramArray.map(par => par.name))
        },
        headers: header,
        parameters: parameters,
        data: data
      };
      digest.info.digitalFilter = getDigitalFilter(parameters);
      return digest;
    }

    const gyromagneticRatio = {
      '1H': 267.52218744e6,
      '2H': 41.065e6,
      '3H': 285.3508e6,
      '3He': -203.789e6,
      '7Li': 103.962e6,
      '13C': 67.28284e6,
      '14N': 19.331e6,
      '15N': -27.116e6,
      '17O': -36.264e6,
      '19F': 251.662e6,
      '23Na': 70.761e6,
      '27Al': 69.763e6,
      '29Si': -53.19e6,
      '31P': 108.291e6,
      '57Fe': 8.681e6,
      '63Cu': 71.118e6,
      '67Zn': 16.767e6,
      '129Xe': -73.997e6
    };

    var medianQuickselect_min = {exports: {}};

    (function (module) {
      (function () {
        function a(d) {
          for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
            if (f <= e) return d[j];
            if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

            for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
              do h++; while (d[e] > d[h]);

              do i--; while (d[i] > d[e]);

              if (i < h) break;
              b(d, h, i);
            }

            b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
          }
        }

        var b = function b(d, e, f) {
          var _ref;

          return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
        },
            c = function c(d, e) {
          return ~~((d + e) / 2);
        };

        module.exports ? module.exports = a : window.median = a;
      })();
    })(medianQuickselect_min);

    var quickSelectMedian = medianQuickselect_min.exports;

    function matrixCheck(data) {
      if (data.length === 0 || data[0].length === 0) {
        throw new RangeError('matrix should contain data');
      }

      const firstLength = data[0].length;

      for (let i = 1; i < data.length; i++) {
        if (data[i].length !== firstLength) {
          throw new RangeError('All rows should has the same length');
        }
      }
    }

    /**
     * Get min and max Z
     *
     * @param matrix - matrix [rows][cols].
     */

    function matrixMinMaxZ(matrix) {
      matrixCheck(matrix);
      const nbRows = matrix.length;
      const nbColumns = matrix[0].length;
      let min = matrix[0][0];
      let max = matrix[0][0];

      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          if (matrix[row][column] < min) min = matrix[row][column];
          if (matrix[row][column] > max) max = matrix[row][column];
        }
      }

      return {
        min,
        max
      };
    }

    /**
     * Create an array with numbers starting from "from" with step "step" of length "length"
     *
     * @param options - options
     * @return - array of distributed numbers with step "step" from "from"
     */
    function createStepArray() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        from = 0,
        step = 1,
        length = 1000
      } = options;
      const array = new Float64Array(length);
      let index = 0;

      while (index < length) {
        array[index] = from + step * index;
        index++;
      }

      return array;
    }

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */

    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }

      var i = 0;
      var len = buf.length;

      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        } // UTF8-2 = %xC2-DF UTF8-tail


        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        } // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail


        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        } // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )


        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        } // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )


        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }

        return false;
      }

      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */

    function ensureString(blob) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof blob === 'string') {
        return blob;
      }

      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }

      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }

    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }

        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      } //@ts-expect-error an ArrayBuffer is also ok


      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    var name = "nmr-parser";
    var version = "2.0.0";
    var description = "Read and convert any NMR file";
    var main = "lib/index.js";
    var module = "src/index.js";
    var files = [
    	"lib",
    	"src"
    ];
    var scripts = {
    	eslint: "eslint src",
    	"eslint-fix": "npm run eslint -- --fix",
    	compile: "rollup -c",
    	prepack: "npm run compile",
    	prettier: "prettier --check src",
    	"prettier-write": "prettier --write src",
    	test: "npm run test-only && npm run eslint && npm run prettier",
    	"test-only": "jest --coverage",
    	build: "cheminfo-build --entry src/index.js --root NMRparser"
    };
    var repository = {
    	type: "git",
    	url: "git+https://github.com/cheminfo/nmr-parser.git"
    };
    var keywords = [
    	"nmr",
    	"magnetic resonance",
    	"parser",
    	"bruker",
    	"JEOL",
    	"CSDM",
    	"data analysis"
    ];
    var author = "Julien Wist";
    var license = "MIT";
    var bugs = {
    	url: "https://github.com/cheminfo/nmr-parser/issues"
    };
    var homepage = "https://github.com/cheminfo/nmr-parser#readme";
    var jest = {
    	testEnvironment: "node"
    };
    var prettier = {
    	arrowParens: "always",
    	semi: true,
    	singleQuote: true,
    	tabWidth: 2,
    	trailingComma: "all"
    };
    var devDependencies = {
    	"@babel/plugin-transform-modules-commonjs": "^7.18.6",
    	"@rollup/plugin-json": "^4.1.0",
    	"@types/jest": "^28.1.6",
    	"bruker-data-test": "^0.2.1",
    	"cheminfo-build": "^1.1.11",
    	eslint: "^8.22.0",
    	"eslint-config-cheminfo": "^8.0.2",
    	"filelist-utils": "^0.5.0",
    	"jcamp-data-test": "^0.2.1",
    	"jeol-data-test": "^0.3.0",
    	jest: "^28.1.3",
    	prettier: "^2.7.1",
    	rollup: "^2.78.0"
    };
    var dependencies = {
    	brukerconverter: "^4.1.5",
    	"is-any-array": "^2.0.0",
    	jcampconverter: "^9.0.2",
    	jeolconverter: "^1.0.1",
    	"nmr-processing": "^9.0.3"
    };
    var packageJson = {
    	name: name,
    	version: version,
    	description: description,
    	main: main,
    	module: module,
    	files: files,
    	scripts: scripts,
    	repository: repository,
    	keywords: keywords,
    	author: author,
    	license: license,
    	bugs: bugs,
    	homepage: homepage,
    	jest: jest,
    	prettier: prettier,
    	devDependencies: devDependencies,
    	dependencies: dependencies
    };

    function toKeyValue(object) {
      let newObject = {};

      for (let key in object) {
        if (typeof object[key] !== 'string') {
          newObject[key] = JSON.stringify(object[key]);
        } else {
          newObject[key] = object[key];
        }
      }

      return newObject;
    }

    function fromJEOL(buffer) {
      let parsedData = parseJEOL(buffer);
      let info = parsedData.info;
      let headers = parsedData.headers;
      let parameters = parsedData.parameters;
      let paramArray = { ...parameters.paramArray
      };
      delete parameters.paramArray;
      let data = parsedData.data; // curation of parameters

      let newInfo = {};
      newInfo.title = `title: ${headers.title} / comment: ${headers.comment} / author:${headers.author} / site: ${headers.site}`;
      newInfo.nucleus = info.nucleus.map(x => {
        if (x === 'Proton') {
          x = '1H';
        }

        if (x === 'Carbon13') {
          x = '13C';
        }

        if (x === 'Nitrogen15') {
          x = '15N';
        }

        return x;
      });
      newInfo.sampleName = info.sampleName;
      newInfo.date = JSON.stringify(info.creationTime);
      newInfo.author = info.author; //newInfo.comment = info.comment;

      newInfo.solvent = info.solvent;
      newInfo.temperature = info.temperature.magnitude;
      newInfo.probeName = info.probeName || '';
      newInfo.fieldStrength = info.fieldStrength.magnitude;
      let gyromagneticRatioConstants = newInfo.nucleus.map(nucleus => gyromagneticRatio[nucleus]);
      newInfo.baseFrequency = gyromagneticRatioConstants.map(gmr => info.fieldStrength.magnitude * gmr / (2 * Math.PI * 1e6));
      newInfo.pulseSequence = info.experiment;
      newInfo.temperature = info.temperature.unit.toLowerCase() === 'celsius' ? 273.15 + info.temperature.magnitude : info.temperature.magnitude;
      newInfo.digitalFilter = info.digitalFilter;
      newInfo.pulseStrength90 = 1 / (4 * info.pulseStrength90.magnitude);
      newInfo.numberOfScans = info.numberOfScans;
      newInfo.relaxationTime = info.relaxationTime.magnitude;
      newInfo.isComplex = info.dataSections.includes('im');
      newInfo.isFid = info.dataUnits[0] === 'Second';
      newInfo.isFt = info.dataUnits[0] === 'Ppm';
      newInfo.dimension = info.dimension;
      const dimension = newInfo.dimension;
      newInfo.originFrequency = info.originFrequency.map(d => d.magnitude / 1e6).slice(0, dimension);
      newInfo.numberOfPoints = info.dataPoints.slice(0, 1);
      newInfo.frequencyOffset = info.frequencyOffset.map((f, i) => f.magnitude * newInfo.baseFrequency[i]).slice(0, dimension);
      newInfo.acquisitionTime = info.acquisitionTime.map(a => a.magnitude).slice(0, dimension);
      newInfo.spectralWidth = info.spectralWidth.map((sw, i) => sw.magnitude / info.originFrequency[i].magnitude * 1e6).slice(0, dimension); // set options for dimensions

      let dimensions = [];
      let options = {};
      let increment;

      for (let d = 0; d < info.dimension; d++) {
        increment = {
          magnitude: info.acquisitionTime[d].magnitude / (info.dataPoints[d] - 1),
          unit: 's'
        };

        if (info.dataUnits[d] === 'Second') {
          options.quantityName = 'time';
          options.originOffset = {
            magnitude: 0,
            unit: 's'
          };

          if (d === 0) {
            options.coordinatesOffset = {
              magnitude: info.digitalFilter * increment.magnitude,
              unit: 's'
            };
          } else {
            options.coordinatesOffset = {
              magnitude: 0,
              unit: 's'
            };
          }

          options.reciprocal = {
            originOffset: {
              magnitude: info.originFrequency[d].magnitude,
              unit: 'Hz'
            },
            quantityName: 'frequency',
            coordinatesOffset: {
              magnitude: info.frequencyOffset[d].magnitude * info.originFrequency[d].magnitude / 1000000,
              unit: 'Hz'
            }
          };
        } else if (info.dataUnits[d] === 'Ppm') {
          options.quantityName = 'frequency';
          let origin = info.originFrequency[d].magnitude;
          options.originOffset = {
            magnitude: origin,
            unit: 'Hz'
          };
          let firstPoint = info.dataOffsetStart[0];
          let lastPoint = info.dataOffsetStop[0];
          let dataLength = lastPoint - firstPoint + 1;
          let spectralWidth = info.spectralWidth[d].magnitude;
          let incr = spectralWidth / info.dataPoints[d];
          increment = {
            magnitude: incr,
            unit: 'Hz'
          };
          let offset = info.dataAxisStop[0] * origin / 1000000;
          options.coordinatesOffset = {
            magnitude: offset,
            unit: 'Hz'
          }; // after increment is computed with whole frequency
          // and original number of points, we recast the
          // number of point for export

          if (dataLength < info.dataPoints[d]) {
            info.dataPoints[d] = dataLength;
          }
        }

        if (d === 0) {
          options.description = 'direct dimension';
        } else {
          options.description = 'indirect dimension';
        }

        dimensions.push(formatLinearDimension(headers.dataAxisTitles[d], info.dataPoints[d], increment, options));
      } // set options for dependentVariable


      options = {
        unit: 'none',
        quantityName: 'relative intensity',
        from: info.dataOffsetStart,
        to: info.dataOffsetStop
      };
      let dependentVariables = [];
      dependentVariables.push(formatDependentVariable(data, 11, options));
      let description = { ...newInfo
      };
      delete description.paramList;
      description.metadata = { ...toKeyValue(headers),
        ...toKeyValue(parameters),
        ...toKeyValue(paramArray)
      };
      let dataStructure = {
        timeStamp: Date.now(),
        version: packageJson.version,
        description,
        tags: ['magnetic resonance'].concat(newInfo.nucleus),
        application: {
          spectralWidthClipped: info.spectralWidthClipped[0].magnitude / newInfo.baseFrequency[0]
        },
        dimensions,
        dependentVariables
      };
      return [dataStructure];
    }

    /**
     * Retrieve the list of files for further process
     * @param {File[]} fileList
     * @param {object} [options={}]
     * @param {number|number[]} [options.processingNumber] - processing number to select, default the smallest number
     * @param {number|number[]} [options.experimentNumber] - experiment number to select, default all
     * @param {boolean} [options.ignoreFID=false] - should we ignore fid
     * @param {boolean} [options.ignoreFT=false] - should we ignore the ft transformed fid
     * @param {boolean} [options.ignore1D=false] - should we ignore 1D spectra
     * @param {boolean} [options.ignore2D=false] - should we ignore 2D spectra
     * @param {boolean} [options.onlyFirstProcessedData=true] - should we take only the first processed data (procno).
     */
    function groupByExperiments(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        processingNumber,
        experimentNumber,
        ignoreFID = false,
        ignoreFT = false,
        ignore1D = false,
        ignore2D = false,
        onlyFirstProcessedData = true
      } = options;

      if (typeof processingNumber === 'number') {
        processingNumber = [processingNumber];
      }

      if (typeof experimentNumber === 'number') {
        experimentNumber = [experimentNumber];
      }

      const experiments = {};

      for (let file of fileList) {
        let currentProcessingNo;
        let currentExperimentNo;
        let experiment;
        let name;
        let id;
        const parts = file.webkitRelativePath.split('/');

        if (file.webkitRelativePath.match(/\/pdata\/[0-9]+\/.*$/)) {
          currentProcessingNo = Number(parts[parts.length - 2]);
          currentExperimentNo = Number(parts[parts.length - 4]);
          name = parts[parts.length - 5];
          id = parts.slice(0, -3).join('/');
        } else if (file.webkitRelativePath.match(/[0-9]+\/.*$/)) {
          currentExperimentNo = Number(parts[parts.length - 2]);
          name = parts[parts.length - 3] || parts[parts.length - 2];
          id = parts.slice(0, -1).join('/');
        } else {
          continue;
        }

        if (experimentNumber && !experimentNumber.includes(currentExperimentNo)) {
          continue;
        }

        if (!experiments[id]) {
          experiments[id] = {
            name,
            expno: currentExperimentNo,
            processedData: {},
            fileList: []
          };
        }

        experiment = experiments[id];

        if (currentProcessingNo) {
          if (!experiment.processedData[currentProcessingNo]) {
            experiment.processedData[currentProcessingNo] = {
              fileList: [],
              name,
              expno: currentExperimentNo
            };
          }

          const processedData = experiment.processedData[currentProcessingNo];
          processedData.fileList.push(file);

          if (file.name.match(/^(1r|1i|2rr|procs|proc2s)$/)) {
            processedData[file.name] = file;
          }

          if (file.name === '2rr') {
            processedData.is2D = true;
            processedData.isFT = true;
          }

          if (file.name === '1r') {
            processedData.is1D = true;
            processedData.isFT = true;
          }
        } else {
          experiment.fileList.push(file);

          if (file.name.match(/^(ser|fid|acqus|acqu2s)$/)) {
            experiment[file.name] = file;
          }

          if (file.name === 'ser') {
            experiment.is2D = true;
            experiment.isFID = true;
          }

          if (file.name === 'fid') {
            experiment.is1D = true;
            experiment.isFID = true;
          }
        }
      }

      if (processingNumber) {
        // we need to deal with the list of processingNumber that is specified
        for (const key in experiments) {
          const processedData = experiments[key].processedData;
          const newProcessedData = {};

          for (let key in processedData) {
            if (!processingNumber.includes(parseInt(key, 10))) continue;
            newProcessedData[key] = processedData[key];
          }

          experiments[key].processedData = newProcessedData;
        }
      } else if (onlyFirstProcessedData) {
        // take the smallest processing and remove all the other
        for (const key in experiments) {
          const processedData = experiments[key].processedData;
          const firstProcessedNumber = Object.keys(processedData).sort((a, b) => Number(a) - Number(b))[0];

          if (firstProcessedNumber !== undefined) {
            experiments[key].processedData = {
              firstProcessedNumber: processedData[firstProcessedNumber]
            };
          }
        }
      } // console.log('experiments', experiments)
      // We convert the object to an array
      // and before filtering we will move all the processedData


      let experimentsArray = [];

      for (let key in experiments) {
        const experiment = { ...experiments[key]
        };
        const processed = experiment.processedData;
        delete experiment.processedData;

        if (experiment.ser || experiment.fid) {
          if (Object.keys(processed).length > 0) {
            const firstProcessed = processed[Object.keys(processed)[0]];

            if (firstProcessed.procs) {
              experiment.fileList.push(firstProcessed.procs);
              experiment.procs = firstProcessed.procs;
            }

            if (firstProcessed.proc2s) {
              experiment.fileList.push(firstProcessed.proc2s);
              experiment.proc2s = firstProcessed.proc2s;
            }
          }

          experimentsArray.push(experiment);
        }

        for (let processedKey in processed) {
          const oneProcessed = processed[processedKey];

          if (oneProcessed['1r'] || oneProcessed['2rr']) {
            if (experiment.acqus) {
              oneProcessed.fileList.push(experiment.acqus);
            }

            if (experiment.acqu2s) {
              oneProcessed.fileList.push(experiment.acqu2s);
            }

            experimentsArray.push({
              acqus: experiment.acqus,
              acqu2s: experiment.acqu2s,
              ...oneProcessed
            });
          }
        }
      } // we can not easily filter


      if (ignoreFID) {
        experimentsArray = experimentsArray.filter(item => !item.isFID);
      }

      if (ignoreFT) {
        experimentsArray = experimentsArray.filter(item => !item.isFT);
      }

      if (ignore1D) {
        experimentsArray = experimentsArray.filter(item => !item.is1D);
      }

      if (ignore2D) {
        experimentsArray = experimentsArray.filter(item => !item.is2D);
      } // console.log(experimentsArray)


      return experimentsArray;
    }

    function joinInfoMeta(target, toAppend) {
      for (let key in toAppend.meta) {
        if (target.meta[key] === undefined) {
          target.meta[key] = toAppend.meta[key];
        }
      }

      for (let key in toAppend.info) {
        if (target.info[key] === undefined) {
          target.info[key] = toAppend.info[key];
        }
      }

      for (let key in target.meta) {
        if (!Array.isArray(target.meta[key])) {
          target.meta[key] = [target.meta[key]];
        }
      }
    }

    /**
     * Dynamically type a string
     * @param {string} value String to dynamically type
     * @returns {boolean|string|number}
     */
    function parseString(value) {
      if (value.length === 4 || value.length === 5) {
        let lowercase = value.toLowerCase();
        if (lowercase === 'true') return true;
        if (lowercase === 'false') return false;
      }

      let number = Number(value);

      if (number === 0 && !value.includes('0')) {
        return value;
      }

      if (!Number.isNaN(number)) return number;
      return value;
    }

    const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
    function complexChromatogram(result) {
      let spectra = result.spectra;
      let length = spectra.length;
      let chromatogram = {
        times: new Array(length),
        series: {
          ms: {
            dimension: 2,
            data: new Array(length)
          }
        }
      };
      let existingGCMSFields = [];

      for (let i = 0; i < GC_MS_FIELDS.length; i++) {
        let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

        if (spectra[0][label]) {
          existingGCMSFields.push(label);
          chromatogram.series[label] = {
            dimension: 1,
            data: new Array(length)
          };
        }
      }

      for (let i = 0; i < length; i++) {
        let spectrum = spectra[i];
        chromatogram.times[i] = spectrum.pageValue;

        for (let j = 0; j < existingGCMSFields.length; j++) {
          chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
        }

        if (spectrum.data) {
          chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
        }
      }

      result.chromatogram = chromatogram;
    }
    function isMSField(canonicDataLabel) {
      return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
    }
    function convertMSFieldToLabel(value) {
      return value.toLowerCase().replace(/[^a-z0-9]/g, '');
    }

    function convertToFloatArray$1(stringArray) {
      let floatArray = [];

      for (let i = 0; i < stringArray.length; i++) {
        floatArray.push(Number(stringArray[i]));
      }

      return floatArray;
    }

    function fastParseXYData(spectrum, value) {
      // TODO need to deal with result
      //  console.log(value);
      // we check if deltaX is defined otherwise we calculate it
      let yFactor = spectrum.yFactor;
      let deltaX = spectrum.deltaX;
      spectrum.isXYdata = true;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let currentX = spectrum.firstX;
      let currentY = spectrum.firstY; // we skip the first line
      //

      let endLine = false;
      let ascii;
      let i = 0;

      for (; i < value.length; i++) {
        ascii = value.charCodeAt(i);

        if (ascii === 13 || ascii === 10) {
          endLine = true;
        } else if (endLine) {
          break;
        }
      } // we proceed taking the i after the first line


      let newLine = true;
      let isDifference = false;
      let isLastDifference = false;
      let lastDifference = 0;
      let isDuplicate = false;
      let inComment = false;
      let currentValue = 0; // can be a difference or a duplicate

      let lastValue = 0; // must be the real last value

      let isNegative = false;
      let inValue = false;
      let skipFirstValue = false;
      let decimalPosition = 0;

      for (; i <= value.length; i++) {
        if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

        if (inComment) {
          // we should ignore the text if we are after $$
          if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          }
        } else {
          // when is it a new value ?
          // when it is not a digit, . or comma
          // it is a number that is either new or we continue
          // eslint-disable-next-line no-lonely-if
          if (ascii <= 57 && ascii >= 48) {
            // a number
            inValue = true;

            if (decimalPosition > 0) {
              currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
            } else {
              currentValue *= 10;
              currentValue += ascii - 48;
            }
          } else if (ascii === 44 || ascii === 46) {
            // a "," or "."
            inValue = true;
            decimalPosition++;
          } else {
            if (inValue) {
              // need to process the previous value
              if (newLine) {
                newLine = false; // we don't check the X value
                // console.log("NEW LINE",isDifference, lastDifference);
                // if new line and lastDifference, the first value is just a check !
                // that we don't check ...

                if (isLastDifference) skipFirstValue = true;
              } else {
                // need to deal with duplicate and differences
                // eslint-disable-next-line no-lonely-if
                if (skipFirstValue) {
                  skipFirstValue = false;
                } else {
                  if (isDifference) {
                    lastDifference = isNegative ? 0 - currentValue : currentValue;
                    isLastDifference = true;
                    isDifference = false;
                  } else if (!isDuplicate) {
                    lastValue = isNegative ? 0 - currentValue : currentValue;
                  }

                  let duplicate = isDuplicate ? currentValue - 1 : 1;

                  for (let j = 0; j < duplicate; j++) {
                    if (isLastDifference) {
                      currentY += lastDifference;
                    } else {
                      currentY = lastValue;
                    }

                    currentData.x.push(currentX);
                    currentData.y.push(currentY * yFactor);
                    currentX += deltaX;
                  }
                }
              }

              isNegative = false;
              currentValue = 0;
              decimalPosition = 0;
              inValue = false;
              isDuplicate = false;
            } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


            if (ascii < 74 && ascii > 63) {
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 64;
            } else if (ascii > 96 && ascii < 106) {
              // negative SQZ digits a b c d e f g h i (ascii 97-105)
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 96;
              isNegative = true;
            } else if (ascii === 115) {
              // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
              inValue = true;
              isDuplicate = true;
              currentValue = 9;
            } else if (ascii > 82 && ascii < 91) {
              inValue = true;
              isDuplicate = true;
              currentValue = ascii - 82;
            } else if (ascii > 73 && ascii < 83) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 73;
            } else if (ascii > 105 && ascii < 115) {
              // negative DIF digits j k l m n o p q r (ascii 106-114)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 105;
              isNegative = true;
            } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
              // $ sign, we need to check the next one
              inValue = true;
              inComment = true;
            } else if (ascii === 37) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = 0;
              isNegative = false;
            } else if (ascii === 45) {
              // a "-"
              // check if after there is a number, decimal or comma
              let ascii2 = value.charCodeAt(i + 1);

              if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
                inValue = true;
                if (!newLine) isLastDifference = false;
                isNegative = true;
              }
            } else if (ascii === 13 || ascii === 10) {
              newLine = true;
              inComment = false;
            } // and now analyse the details ... space or tabulation
            // if "+" we just don't care

          }
        }
      }
    }

    const removeCommentRegExp = /\$\$.*/;
    const peakTableSplitRegExp = /[,\t ]+/;
    function parsePeakTable(spectrum, value, result) {
      spectrum.isPeaktable = true;

      if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
        parseXY(spectrum, value, result);
      } else {
        parseXYZ(spectrum, value, result);
      } // we will add the data in the variables


      if (spectrum.variables) {
        for (let key in spectrum.variables) {
          spectrum.variables[key].data = spectrum.data[key];
        }
      }
    }

    function parseXY(spectrum, value, result) {
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % 2 === 0) {
          for (let j = 0; j < values.length; j = j + 2) {
            // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
            currentData.x.push(Number(values[j]) * spectrum.xFactor);
            currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYZ(spectrum, value, result) {
      let currentData = {};
      let variables = Object.keys(spectrum.variables);
      let numberOfVariables = variables.length;
      variables.forEach(variable => currentData[variable] = []);
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % numberOfVariables === 0) {
          for (let j = 0; j < values.length; j++) {
            // todo should try to find a xFactor (y, ...)
            currentData[variables[j % numberOfVariables]].push(Number(values[j]));
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYA(spectrum, value) {
      let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
      spectrum.isXYAdata = true;
      let values;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
        currentData.x.push(Number(values[0]));
        currentData.y.push(Number(values[1]));
      }
    }

    function median(input) {
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }

      return quickSelectMedian(input.slice());
    }

    function convertTo3DZ$1(spectra) {
      let minZ = spectra[0].data.y[0];
      let maxZ = minZ;
      let ySize = spectra.length;
      let xSize = spectra[0].data.x.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = spectra[i].data.y;

        for (let j = 0; j < xSize; j++) {
          let value = z[i][j];
          if (value < minZ) minZ = value;
          if (value > maxZ) maxZ = value;
        }
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      const medians = [];

      for (let i = 0; i < z.length; i++) {
        const row = Float64Array.from(z[i]);

        for (let i = 0; i < row.length; i++) {
          if (row[i] < 0) row[i] = -row[i];
        }

        medians.push(median(row));
      }

      const median$1 = median(medians);
      return {
        z,
        minX: Math.min(firstX, lastX),
        maxX: Math.max(firstX, lastX),
        minY: Math.min(firstY, lastY),
        maxY: Math.max(firstY, lastY),
        minZ,
        maxZ,
        noise: median$1
      };
    }

    function generateContourLines(zData, options) {
      let noise = zData.noise;
      let z = zData.z;
      let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
      let isOver0, isOver1, isOver2, isOver3;
      let nbSubSpectra = z.length;
      let nbPovars = z[0].length;
      let pAx, pAy, pBx, pBy;
      let x0 = zData.minX;
      let xN = zData.maxX;
      let dx = (xN - x0) / (nbPovars - 1);
      let y0 = zData.minY;
      let yN = zData.maxY;
      let dy = (yN - y0) / (nbSubSpectra - 1);
      let minZ = zData.minZ;
      let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
      // -------------------------
      // Povars attribution
      //
      // 0----1
      // |  / |
      // | /  |
      // 2----3
      //
      // ---------------------d------

      let iter = options.nbContourLevels * 2;
      let contourLevels = new Array(iter);
      let lineZValue;

      for (let level = 0; level < iter; level++) {
        // multiply by 2 for positif and negatif
        let contourLevel = {};
        contourLevels[level] = contourLevel;
        let side = level % 2;
        let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

        if (side === 0) {
          lineZValue = factor + options.noiseMultiplier * noise;
        } else {
          lineZValue = 0 - factor - options.noiseMultiplier * noise;
        }

        let lines = [];
        contourLevel.zValue = lineZValue;
        contourLevel.lines = lines;
        if (lineZValue <= minZ || lineZValue >= maxZ) continue;

        for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
          let subSpectra = z[iSubSpectra];
          let subSpectraAfter = z[iSubSpectra + 1];

          for (let povar = 0; povar < nbPovars - 1; povar++) {
            povarHeight0 = subSpectra[povar];
            povarHeight1 = subSpectra[povar + 1];
            povarHeight2 = subSpectraAfter[povar];
            povarHeight3 = subSpectraAfter[povar + 1];
            isOver0 = povarHeight0 > lineZValue;
            isOver1 = povarHeight1 > lineZValue;
            isOver2 = povarHeight2 > lineZValue;
            isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
            // povar2 are below, we find the varersections and add
            // the segment

            if (isOver0 !== isOver1 && isOver0 !== isOver2) {
              pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
              pAy = iSubSpectra;
              pBx = povar;
              pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // remove push does not help !!!!


            if (isOver3 !== isOver1 && isOver3 !== isOver2) {
              pAx = povar + 1;
              pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
              pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
              pBy = iSubSpectra + 1;
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // test around the diagonal


            if (isOver1 !== isOver2) {
              pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
              pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

              if (isOver1 !== isOver0) {
                pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
                pBy = iSubSpectra;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver0) {
                pBx = povar;
                pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver1 !== isOver3) {
                pBx = povar + 1;
                pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver3) {
                pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
                pBy = iSubSpectra + 1;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }
            }
          }
        }
      }

      return {
        minX: zData.minX,
        maxX: zData.maxX,
        minY: zData.minY,
        maxY: zData.maxY,
        segments: contourLevels
      };
    }

    function add2D(result, options) {
      let zData = convertTo3DZ$1(result.spectra);

      if (!options.noContour) {
        result.contourLines = generateContourLines(zData, options);
        delete zData.z;
      }

      result.minMax = zData;
    }

    function postProcessingNMR(entriesFlat) {
      // specific NMR functions
      for (let entry of entriesFlat) {
        let observeFrequency = 0;
        let shiftOffsetVal = 0;

        for (let spectrum of entry.spectra) {
          if (entry.ntuples && entry.ntuples.symbol) {
            if (!observeFrequency && spectrum.observeFrequency) {
              observeFrequency = spectrum.observeFrequency;
            }

            if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
              shiftOffsetVal = spectrum.shiftOffsetVal;
            }
          } else {
            observeFrequency = spectrum.observeFrequency;
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }

          if (observeFrequency) {
            if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
              spectrum.xUnits = 'PPM';
              spectrum.xFactor = spectrum.xFactor / observeFrequency;
              spectrum.firstX = spectrum.firstX / observeFrequency;
              spectrum.lastX = spectrum.lastX / observeFrequency;
              spectrum.deltaX = spectrum.deltaX / observeFrequency;

              for (let i = 0; i < spectrum.data.x.length; i++) {
                spectrum.data.x[i] /= observeFrequency;
              }
            }
          }

          if (shiftOffsetVal) {
            let shift = spectrum.firstX - shiftOffsetVal;
            spectrum.firstX = spectrum.firstX - shift;
            spectrum.lastX = spectrum.lastX - shift;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] -= shift;
            }
          } // we will check if some nucleus are missing ...


          if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
            for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
              let symbol = entry.ntuples.symbol[i];
              let nucleus = entry.ntuples.nucleus[i];

              if (symbol.startsWith('F') && !nucleus) {
                if (symbol === 'F1') {
                  // if F1 is defined we will use F2
                  if (entry.tmp.$NUC2) {
                    entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                  } else {
                    let f2index = entry.ntuples.symbol.indexOf('F2');

                    if (f2index && entry.ntuples.nucleus[f2index]) {
                      entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                    }
                  }
                }

                if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
              }

              if (symbol === 'F2') {
                entry.yType = entry.ntuples.nucleus[0];
              }
            }
          }

          if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
            let unit = '';
            let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

            if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
              unit = entry.ntuples.units[pageSymbolIndex];
            }

            if (unit !== 'PPM') {
              if (pageSymbolIndex !== 0) {
                throw Error('Not sure about this ntuples format');
              }

              let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
              let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

              if (!ratio0 || !ratio1) {
                throw Error('Problem with determination of gyromagnetic ratio');
              }

              let ratio = ratio0 / ratio1 * observeFrequency;
              spectrum.pageValue /= ratio;
            }
          }
        }
      }
    }

    function profiling(result, action, options) {
      if (result.profiling) {
        result.profiling.push({
          action,
          time: Date.now() - options.start
        });
      }
    }

    function simpleChromatogram(result) {
      let data = result.spectra[0].data;
      result.chromatogram = {
        times: data.x.slice(),
        series: {
          intensity: {
            dimension: 1,
            data: data.y.slice()
          }
        }
      };
    }

    function postProcessing(entriesFlat, result, options) {
      // converting Hz to ppm
      postProcessingNMR(entriesFlat);

      for (let entry of entriesFlat) {
        if (Object.keys(entry.ntuples).length > 0) {
          let newNtuples = [];
          let keys = Object.keys(entry.ntuples);

          for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let values = entry.ntuples[key];

            for (let j = 0; j < values.length; j++) {
              if (!newNtuples[j]) newNtuples[j] = {};
              newNtuples[j][key] = values[j];
            }
          }

          entry.ntuples = newNtuples;
        }

        if (entry.twoD && options.wantXY) {
          add2D(entry, options);
          profiling(result, 'Finished countour plot calculation', options);

          if (!options.keepSpectra) {
            delete entry.spectra;
          }
        } // maybe it is a GC (HPLC) / MS. In this case we add a new format


        if (options.chromatogram) {
          if (entry.spectra.length > 1) {
            complexChromatogram(entry);
          } else {
            simpleChromatogram(entry);
          }

          profiling(result, 'Finished chromatogram calculation', options);
        }

        delete entry.tmp;
      }
    }

    function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
      let xIndex = -1;
      let yIndex = -1;
      let firstVariable = '';
      let secondVariable = '';

      if (kind.indexOf('++') > 0) {
        firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
        secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
      } else {
        kind = kind.replace(/[^a-zA-Z]/g, '');
        firstVariable = kind.charAt(0);
        secondVariable = kind.charAt(1);
        spectrum.variables = {};

        for (let symbol of kind) {
          let lowerCaseSymbol = symbol.toLowerCase();
          let index = currentEntry.ntuples.symbol.indexOf(symbol);
          if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
          spectrum.variables[lowerCaseSymbol] = {};

          for (let key in currentEntry.ntuples) {
            if (currentEntry.ntuples[key][index]) {
              spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
            }
          }
        }
      }

      xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
      yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
      if (xIndex === -1) xIndex = 0;
      if (yIndex === -1) yIndex = 0;

      if (currentEntry.ntuples.first) {
        if (currentEntry.ntuples.first.length > xIndex) {
          spectrum.firstX = currentEntry.ntuples.first[xIndex];
        }

        if (currentEntry.ntuples.first.length > yIndex) {
          spectrum.firstY = currentEntry.ntuples.first[yIndex];
        }
      }

      if (currentEntry.ntuples.last) {
        if (currentEntry.ntuples.last.length > xIndex) {
          spectrum.lastX = currentEntry.ntuples.last[xIndex];
        }

        if (currentEntry.ntuples.last.length > yIndex) {
          spectrum.lastY = currentEntry.ntuples.last[yIndex];
        }
      }

      if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
        spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
      }

      if (currentEntry.ntuples.factor) {
        if (currentEntry.ntuples.factor.length > xIndex) {
          spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
        }

        if (currentEntry.ntuples.factor.length > yIndex) {
          spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
        }
      }

      if (currentEntry.ntuples.units) {
        if (currentEntry.ntuples.units.length > xIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
            spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
          } else {
            spectrum.xUnits = currentEntry.ntuples.units[xIndex];
          }
        }

        if (currentEntry.ntuples.units.length > yIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
            spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
          } else {
            spectrum.yUnits = currentEntry.ntuples.units[yIndex];
          }
        }
      }
    }

    function prepareSpectrum(spectrum) {
      if (!spectrum.xFactor) spectrum.xFactor = 1;
      if (!spectrum.yFactor) spectrum.yFactor = 1;
    }

    const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
    const defaultOptions$1 = {
      keepRecordsRegExp: /^$/,
      canonicDataLabels: true,
      canonicMetadataLabels: false,
      dynamicTyping: true,
      withoutXY: false,
      chromatogram: false,
      keepSpectra: false,
      noContour: false,
      nbContourLevels: 7,
      noiseMultiplier: 5,
      profiling: false
    };
    /**
     *
     * @typedef {object} ConvertOptions
     * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
     * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
     * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
     * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
     * @property {boolean} [options.withoutXY=false] - Remove the XY data.
     * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
     * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
     * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
     * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
     * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
     * @property {boolean} [options.profiling=false] - Add profiling information.
     */

    /**
     *
     * @typedef {object} Ntuples
     * @property {string[]} [varname]
     * @property {string[]} [symbol]
     * @property {string[]} [vartype]
     * @property {string[]} [varform]
     * @property {number[]} [vardim]
     * @property {string[]} [units]
     * @property {number[]} [factor]
     * @property {number[]} [first]
     * @property {number[]} [last]
     * @property {number[]} [min]
     * @property {number[]} [max]
     * @property {string[]} [nucleus]
     */

    /**
     * @typedef { Record<string, any> } Spectrum
     * @property {Record<string, number[]>} [data]
     * @property {number} [firstX]
     * @property {number} [lastX]
     * @property {number} [deltaX]
     * @property {number} [yFactor]
     * @property {number} [xFactor]
     * @property {number} [nbPoints]
     */

    /**
     *
     * @typedef {object} Entry
     * @property {Spectrum[]} spectra
     * @property {Ntuples} ntuples
     * @property {object} meta
     * @property {object} info
     * @property {object} tmp
     * @property {string} [title]
     * @property {string} [dataType]
     * @property {string} [dataClass]
     * @property {boolean} [twoD]
     */

    /**
     *
     * @typedef { object } ConvertResult
     * @property { object[] | boolean } profiling
     * @property { string[] } logs
     * @property { object[] } entries
     * @property { Entry[] } flatten
     */

    /**
     * Parse a jcamp.
     *
     * @param {string|ArrayBuffer|Uint8Array} jcamp
     * @param {ConvertOptions} [options]
     * @returns {ConvertResult}
     */

    function convert(jcamp) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      jcamp = ensureString(jcamp);
      options = { ...defaultOptions$1,
        ...options
      };
      options.wantXY = !options.withoutXY;
      options.start = Date.now();
      let entriesFlat = [];
      let result = {
        profiling: options.profiling ? [] : false,
        logs: [],
        entries: []
      };
      let tmpResult = {
        children: []
      };
      let currentEntry = tmpResult;
      let parentsStack = [];
      let spectrum = {};

      if (typeof jcamp !== 'string') {
        throw new TypeError('the JCAMP should be a string');
      }

      profiling(result, 'Before split to LDRS', options);
      let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
      profiling(result, 'Split to LDRS', options);
      if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

      for (let ldr of ldrs) {
        // This is a new LDR
        let position = ldr.indexOf('=');
        let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
        let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
        let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

        if (canonicDataLabel === 'DATATABLE') {
          let endLine = dataValue.indexOf('\n');
          if (endLine === -1) endLine = dataValue.indexOf('\r');

          if (endLine > 0) {
            // ##DATA TABLE= (X++(I..I)), XYDATA
            // We need to find the variables
            let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
            prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
            spectrum.datatable = infos[0];

            if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
              canonicDataLabel = 'PEAKTABLE';
            } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
              canonicDataLabel = 'XYDATA';

              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }
            }
          }
        }

        if (canonicDataLabel === 'XYDATA') {
          if (options.wantXY) {
            prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

            if (dataValue.match(/.*\+\+.*/)) {
              // ex: (X++(Y..Y))
              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }

              fastParseXYData(spectrum, dataValue);
            } else {
              parsePeakTable(spectrum, dataValue, result);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        } else if (canonicDataLabel === 'PEAKTABLE') {
          if (options.wantXY) {
            prepareSpectrum(spectrum);
            parsePeakTable(spectrum, dataValue, result);
            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'PEAKASSIGNMENTS') {
          if (options.wantXY) {
            if (dataValue.match(/.*(XYA).*/)) {
              // ex: (XYA)
              parseXYA(spectrum, dataValue);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'TITLE') {
          let parentEntry = currentEntry;

          if (!parentEntry.children) {
            parentEntry.children = [];
          }

          currentEntry = {
            spectra: [],
            ntuples: {},
            info: {},
            meta: {},
            tmp: {} // tmp information we need to keep for postprocessing

          };
          parentEntry.children.push(currentEntry);
          parentsStack.push(parentEntry);
          entriesFlat.push(currentEntry);
          currentEntry.title = dataValue;
        } else if (canonicDataLabel === 'DATATYPE') {
          currentEntry.dataType = dataValue;

          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'NTUPLES') {
          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'DATACLASS') {
          currentEntry.dataClass = dataValue;
        } else if (canonicDataLabel === 'XUNITS') {
          spectrum.xUnits = dataValue;
        } else if (canonicDataLabel === 'YUNITS') {
          spectrum.yUnits = dataValue;
        } else if (canonicDataLabel === 'FIRSTX') {
          spectrum.firstX = Number(dataValue);
        } else if (canonicDataLabel === 'LASTX') {
          spectrum.lastX = Number(dataValue);
        } else if (canonicDataLabel === 'FIRSTY') {
          spectrum.firstY = Number(dataValue);
        } else if (canonicDataLabel === 'LASTY') {
          spectrum.lastY = Number(dataValue);
        } else if (canonicDataLabel === 'NPOINTS') {
          spectrum.nbPoints = Number(dataValue);
        } else if (canonicDataLabel === 'XFACTOR') {
          spectrum.xFactor = Number(dataValue);
        } else if (canonicDataLabel === 'YFACTOR') {
          spectrum.yFactor = Number(dataValue);
        } else if (canonicDataLabel === 'MAXX') {
          spectrum.maxX = Number(dataValue);
        } else if (canonicDataLabel === 'MINX') {
          spectrum.minX = Number(dataValue);
        } else if (canonicDataLabel === 'MAXY') {
          spectrum.maxY = Number(dataValue);
        } else if (canonicDataLabel === 'MINY') {
          spectrum.minY = Number(dataValue);
        } else if (canonicDataLabel === 'DELTAX') {
          spectrum.deltaX = Number(dataValue);
        } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
          if (!spectrum.observeFrequency) {
            spectrum.observeFrequency = Number(dataValue);
          }
        } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
          if (!spectrum.xType) {
            currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
          }
        } else if (canonicDataLabel === '$OFFSET') {
          // OFFSET for Bruker spectra
          currentEntry.shiftOffsetNum = 0;

          if (!spectrum.shiftOffsetVal) {
            spectrum.shiftOffsetVal = Number(dataValue);
          }
        } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
          currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'SYMBOL') {
          currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARTYPE') {
          currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARFORM') {
          currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARDIM') {
          currentEntry.ntuples.vardim = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'UNITS') {
          currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'FACTOR') {
          currentEntry.ntuples.factor = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'FIRST') {
          currentEntry.ntuples.first = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'LAST') {
          currentEntry.ntuples.last = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MIN') {
          currentEntry.ntuples.min = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MAX') {
          currentEntry.ntuples.max = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === '.NUCLEUS') {
          if (currentEntry.ntuples) {
            currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
          }
        } else if (canonicDataLabel === 'PAGE') {
          spectrum.page = dataValue.trim();
          spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
          spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
        } else if (canonicDataLabel === 'RETENTIONTIME') {
          spectrum.pageValue = Number(dataValue);
        } else if (isMSField(canonicDataLabel)) {
          spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
        } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
          spectrum.sampleDescription = dataValue;
        } else if (canonicDataLabel.startsWith('$NUC')) {
          if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
            currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
          }
        } else if (canonicDataLabel === 'END') {
          currentEntry = parentsStack.pop();
        }

        if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
          let value = dataValue.trim();
          let target, label;

          if (dataLabel.startsWith('$')) {
            label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
            target = currentEntry.meta;
          } else {
            label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
            target = currentEntry.info;
          }

          if (options.dynamicTyping) {
            value = parseString(value);
          }

          if (target[label]) {
            if (!Array.isArray(target[label])) {
              target[label] = [target[label]];
            }

            target[label].push(value);
          } else {
            target[label] = value;
          }
        }
      }

      profiling(result, 'Finished parsing', options);
      postProcessing(entriesFlat, result, options);
      profiling(result, 'Total time', options);
      /*
      if (result.children && result.children.length>0) {
        result = { ...result, ...result.children[0] };
      }
      */

      result.entries = tmpResult.children;
      result.flatten = entriesFlat;
      return result;
    }

    /**
     *
     * @param {File} file
     * @param {object} [options={}]
     * @param {RegExp} [options.keepRecordsRegExp={}]
     * @returns
     */

    async function parseData(file, options) {
      if (!file) return {
        info: {},
        meta: {}
      };
      const data = await file.text();
      let {
        keepRecordsRegExp = /.*/
      } = options;
      let result = convert(data, {
        keepRecordsRegExp
      });
      const firstSpectrum = result.flatten.length === 0 ? {} : result.flatten[0];
      return firstSpectrum;
    }

    async function setFIDSpectrumData(file, spectra) {
      let td = parseInt(spectra.meta.TD[0], 10);
      let ioBuffer = new IOBuffer(await file.arrayBuffer());

      if (td * spectra.meta.nbSubSpectra * 4 > ioBuffer.length) {
        throw new RangeError(`td: ${td} and nbSubSpectra: ${nbSubSpectra} exceed the buffer size`);
      }

      let SW_H = Number(spectra.meta.SW_h[0]);
      let SF = Number(spectra.meta.SFO1[0]);
      spectra.meta.DATATYPE = 'NMR FID';
      let DW = 1 / (2 * SW_H);
      let AQ = td * DW;
      let endian = parseInt(spectra.meta.BYTORDA, 10);
      endian = endian ? 0 : 1;

      if (endian) {
        ioBuffer.setLittleEndian();
      } else {
        ioBuffer.setBigEndian();
      }

      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra / 2 : 1;
      spectra.spectra = new Array(nbSubSpectra);
      const stopReading = td / 2;

      for (let j = 0; j < nbSubSpectra; j++) {
        let toSave = {
          dataType: 'NMR FID',
          dataTable: '(X++(R..R))',
          nbPoints: td,
          firstX: 0,
          lastX: AQ,
          nucleus: spectra.meta.NUC1,
          xUnit: 'Sec',
          yUnit: 'Arbitrary',
          data: {
            x: createStepArray({
              length: td,
              step: DW
            }),
            re: new Float64Array(td),
            im: new Float64Array(td)
          },
          isXYdata: true,
          observeFrequency: SF,
          title: spectra.meta.TITLE,
          deltaX: DW
        };
        spectra.spectra[j] = toSave;

        for (let i = 0; i < stopReading; i++) {
          spectra.spectra[j].data.re[i] = ioBuffer.readInt32();
          spectra.spectra[j].data.im[i] = ioBuffer.readInt32();
        }
      }
    }

    async function setXYSpectrumData(file, spectra) {
      let ioBufferReal = file.re ? new IOBuffer(await file.re.arrayBuffer()) : null;
      let ioBufferImaginary = file.im ? new IOBuffer(await file.im.arrayBuffer()) : null;
      let td = getDirectParameter(spectra.meta.SI);
      let swP = getDirectParameter(spectra.meta.SW_p);
      let sf = getDirectParameter(spectra.meta.SF);
      let bf = sf;
      let offset = getDirectParameter(spectra.shiftOffsetVal || spectra.meta.OFFSET);
      spectra.meta.observeFrequency = sf;
      spectra.meta.brukerReference = bf;
      spectra.meta.DATATYPE = 'NMR Spectrum';
      let endian = parseInt(spectra.meta.BYTORDP, 10);
      endian = endian ? 0 : 1;
      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra : 1;

      if (endian) {
        if (file.re) ioBufferReal.setLittleEndian();
        if (file.im) ioBufferImaginary.setLittleEndian();
      } else {
        if (file.re) ioBufferReal.setBigEndian();
        if (file.im) ioBufferImaginary.setBigEndian();
      }

      for (let i = 0; i < nbSubSpectra; i++) {
        let toSave = {
          dataType: 'NMR Spectrum',
          nbPoints: td,
          firstX: offset,
          lastX: offset - swP / sf,
          xUnit: 'PPM',
          yUnit: 'Arbitrary',
          isXYdata: true,
          nucleus: spectra.meta.NUC1,
          observeFrequency: sf,
          title: spectra.meta.TITLE,
          deltaX: -(swP / sf) / (td - 1)
        };
        let deltaX = toSave.deltaX;
        let x = new Float64Array(td);
        let re = new Float64Array(td);
        let im = file.im ? new Float64Array(td) : null;

        if (im) {
          for (let k = 0; k < td; ++k) {
            x[k] = offset + k * deltaX;
            re[k] = ioBufferReal.readInt32();
            im[k] = ioBufferImaginary.readInt32();
          }
        } else {
          for (let k = 0; k < td; ++k) {
            x[k] = offset + k * deltaX;
            re[k] = ioBufferReal.readInt32();
          }
        }

        toSave.data = im ? {
          x,
          re,
          im
        } : {
          x,
          re
        };
        spectra.spectra.push(toSave);
      }
    }

    function getDirectParameter(meta) {
      return Number(isAnyArray(meta) ? meta[0] : meta);
    }

    async function convert1D(files, options) {
      const result = await parseData(files.procs, options);
      const acqus = await parseData(files.acqus, options);
      joinInfoMeta(result, acqus);

      if (files['1r'] || files['1i']) {
        await setXYSpectrumData({
          re: files['1r'],
          im: files['1i']
        }, result);
      } else if (files.fid) {
        await setFIDSpectrumData(files.fid, result);
      }

      return result;
    }

    function mergeMetadata(main, complement) {
      for (let key in complement.meta) {
        if (main.meta[key]) {
          if (!Array.isArray(main.meta[key])) {
            main.meta[key] = [main.meta[key]];
          }

          main.meta[key].push(complement.meta[key]);
        } else if (main.meta[key] === undefined) {
          main.meta[key] = [complement.meta[key]];
        }
      }

      return main;
    }

    async function convert2D(files, options) {
      const result = mergeMetadata(await parseData(files.procs, options), await parseData(files.proc2s, options));
      const acqus = mergeMetadata(await parseData(files.acqus, options), await parseData(files.acqu2s, options));
      joinInfoMeta(result, acqus);
      result.meta.nbSubSpectra = files['2rr'] ? parseInt(result.meta.SI[1], 10) : parseInt(result.meta.TD[1], 10);

      if (!result.meta.SW_p) {
        // eslint-disable-next-line camelcase
        result.meta.SW_p = result.meta.SW_h;
      }

      if (!result.meta.SF) {
        result.meta.SF = result.meta.SFO1;
      }

      let firstY, lastY, xOffset, yOffset;

      if (files['2rr']) {
        let sf = Number(result.meta.SF[1]);
        let swP = Number(result.meta.SW_p[1] || result.meta.SW[1]);
        yOffset = Number(result.meta.OFFSET[1]);
        xOffset = Number(result.meta.OFFSET[0]);
        firstY = yOffset;
        lastY = yOffset - swP / sf;
        result.meta.firstY = firstY;
        result.meta.lastY = lastY;
        await setXYSpectrumData({
          re: files['2rr']
        }, result);
      } else if (files.ser) {
        firstY = 0;
        lastY = result.meta.nbSubSpectra;
        let xWindowSize = Number(result.meta.SW[0]);
        let yWindowSize = Number(result.meta.SW[1]);
        let xTransmitterFrequency = Number(result.meta.SFO1[0]);
        let yTransmitterFrequency = Number(result.meta.SFO1[1]);
        let xTransmitterFrequencyOffset = Number(result.meta.O1[0]);
        let yTransmitterFrequencyOffset = Number(result.meta.O1[1]);
        xOffset = xTransmitterFrequencyOffset / xTransmitterFrequency + xWindowSize / 2;
        yOffset = yTransmitterFrequencyOffset / yTransmitterFrequency + yWindowSize / 2;
        await setFIDSpectrumData(files.ser, result);
      }

      let pageValue = firstY;
      let nbSubSpectra = result.meta.nbSubSpectra / (files['2rr'] ? 1 : 2);
      let deltaY = (lastY - firstY) / (nbSubSpectra - 1);

      for (let i = 0; i < nbSubSpectra; i++) {
        pageValue += deltaY;
        result.spectra[i].pageValue = pageValue;
      }

      let {
        NUC1: nuc1,
        AXNUC: axnuc,
        SF: sf
      } = result.meta;
      const nucleus = axnuc ? axnuc : nuc1 ? nuc1 : [];
      result.info['2D_Y_NUCLEUS'] = nucleus[1];
      result.info['2D_X_NUCLEUS'] = nucleus[0];
      result.info['2D_Y_FRECUENCY'] = sf[1];
      result.info['2D_X_FRECUENCY'] = sf[0];
      result.info['2D_Y_OFFSET'] = yOffset;
      result.info['2D_X_OFFSET'] = xOffset;
      result.info.twoD = true;
      result.twoD = true;
      return result;
    }

    /**
     * Those functions should disappear if add2D becomes accessible in jcampconvert
     * @param spectra
     * @returns {{z: Array, minX: *, maxX: *, minY: *, maxY: *, minZ: *, maxZ: *, noise: number}}
     */

    function convertTo3DZ(spectra) {
      let ySize = spectra.length;
      let xSize = spectra[0].data.re.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = new Float64Array(spectra[i].data.re);
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[xSize - 1];
      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      let minMaxX = firstX < lastX ? {
        minX: firstX,
        maxX: lastX
      } : {
        minX: lastX,
        maxX: firstX
      };
      let minMaxY = firstY < lastY ? {
        minY: firstY,
        maxY: lastY
      } : {
        minY: lastY,
        maxY: firstY
      };
      const {
        min: minZ,
        max: maxZ
      } = matrixMinMaxZ(z);
      return {
        z,
        minZ,
        maxZ,
        ...minMaxX,
        ...minMaxY
      };
    }

    /**
     * Extract information and data from bruker files.
     * @param {File[]} brukerFiles - Needed bruker files to parse raw data.
     * @param {object} [options = {}] - options.
     * @param {boolean} [options.xy] - if true, spectra data is a object with x and y
     * @param {boolean} [options.keepSpectra=false] - for 2D should we keep the spectra or just the matrix ?
     * @param {RegExp} [options.keepRecordsRegExp='\/.*\/'] - regular expresion to parse the metadata of the spectrum.
     * @returns
     */

    async function convertOneExperiment(brukerFiles) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let result;

      if (brukerFiles.ser || brukerFiles['2rr']) {
        result = await convert2D(brukerFiles, options);
      } else if (brukerFiles['1r'] || brukerFiles['1i'] || brukerFiles.fid) {
        result = await convert1D(brukerFiles, options);
      } else {
        throw new RangeError('The current files are invalid');
      }

      result.source = {};

      for (let key in brukerFiles) {
        if ((typeof brukerFiles[key]).match(/number|string|boolean/)) {
          result.source[key] = brukerFiles[key];
        } // todo we could as well keep the FileList at this level if
        // we want to keep the original data

      } //normalizing info


      result.meta.DATE = Number(result.meta.DATE);

      if (result.meta.GRPDLY) {
        result.meta.GRPDLY = Number(result.meta.GRPDLY);
        result.meta.DSPFVS = Number(result.meta.DSPFVS);
        result.meta.DECIM = Number(result.meta.DECIM);
      }

      for (let key in result.meta) {
        if (!Array.isArray(result.meta[key])) {
          continue;
        }

        if (result.meta[key].length === 1) {
          result.meta[key] = result.meta[key][0];
        } else if (typeof result.meta[key][0] === 'string' && result.meta[key][0].indexOf('(0..') > -1) {
          result.meta[key] = result.meta[key][0];
        }
      }

      if (result.twoD) {
        result.minMax = convertTo3DZ(result.spectra);

        if (!options.keepSpectra) {
          delete result.spectra;
        }
      }

      return result;
    }

    /**
     *
     * @param {*} fileList
     * @param {object} [options={}]
     * @param {object} [options.filter]
     * @param {object} [options.converter]
     * @returns
     */

    async function convertFileList(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        filter,
        converter
      } = options;
      const results = [];
      const experiments = groupByExperiments(fileList, filter);

      for (let experiment of experiments) {
        const result = await convertOneExperiment(experiment, converter);

        if (result) {
          results.push(result);
        }
      }

      return results;
    }

    function convertToFloatArray(data) {
      if (isAnyArray(data[0])) {
        return data.map(e => Float64Array.from(e));
      } else if (isAnyArray(data)) {
        return Float64Array.from(data);
      } else if (typeof data === 'object') {
        let keys = Object.keys(data);

        for (let key of keys) {
          data[key] = convertToFloatArray(data[key]);
        }

        return data;
      }

      return data;
    }

    /* eslint-disable no-loss-of-precision */

    /**
     * Returns the group delay for old Bruker NMR spectra
     * @param {number} gspfvs
     * @param {number} decim
     * @return {number}
     */
    function getDigitalFilterParameters(grpdly, dspfvs, decim) {
      let value;

      if (grpdly > 0) {
        value = Number(grpdly);
      } else if (dspfvs > 14) {
        value = 0;
      } else if (!brukerDspTable[dspfvs]) {
        throw new Error('dspfvs not in lookup table');
      } else {
        const dspfvsList = brukerDspTable[dspfvs];
        if (!dspfvsList[decim]) throw new Error('decim not in lookup table');
        value = dspfvsList[decim];
      }

      return value;
    }
    const brukerDspTable = {
      10: {
        2: 44.75,
        3: 33.5,
        4: 66.625,
        6: 59.083333333333333,
        8: 68.5625,
        12: 60.375,
        16: 69.53125,
        24: 61.020833333333333,
        32: 70.015625,
        48: 61.34375,
        64: 70.2578125,
        96: 61.505208333333333,
        128: 70.37890625,
        192: 61.5859375,
        256: 70.439453125,
        384: 61.626302083333333,
        512: 70.4697265625,
        768: 61.646484375,
        1024: 70.48486328125,
        1536: 61.656575520833333,
        2048: 70.492431640625
      },
      11: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 72.25,
        24: 70.166666666666667,
        32: 72.75,
        48: 70.5,
        64: 73,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      12: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 71.625,
        24: 70.166666666666667,
        32: 72.125,
        48: 70.5,
        64: 72.375,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      13: {
        2: 2.75,
        3: 2.8333333333333333,
        4: 2.875,
        6: 2.9166666666666667,
        8: 2.9375,
        12: 2.9583333333333333,
        16: 2.96875,
        24: 2.9791666666666667,
        32: 2.984375,
        48: 2.9895833333333333,
        64: 2.9921875,
        96: 2.9947916666666667
      }
    };

    function getNucleusFromMetadata(metaData, info, subfix) {
      let nucleus = [];

      if (metaData[`${subfix}AXNUC`]) {
        nucleus = metaData[`${subfix}AXNUC`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length < 1 && metaData[`${subfix}NUC1`]) {
        nucleus = metaData[`${subfix}NUC1`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length === 0) {
        if (metaData['.NUCLEUS']) {
          nucleus = metaData['.NUCLEUS'].split(',').map(nuc => nuc.trim());
        } else if (metaData['.OBSERVENUCLEUS']) {
          nucleus = [metaData['.OBSERVENUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
        } else {
          nucleus = getNucleusFrom2DExperiment(info.experiment);
        }

        nucleus = checkForNucleus(nucleus);
      }

      if (metaData['2D_X_NUCLEUS'] && metaData['2D_Y_NUCLEUS']) {
        nucleus = [metaData['2D_X_NUCLEUS'].replace(/[^A-Za-z0-9]/g, ''), metaData['2D_Y_NUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
      }

      return nucleus;
    }
    /**
     * Returns a list of likely nuclei based on an experiment string
     * This is really an hypothesis and should not be used
     * @param {string} experiment
     * @return {string[]}
     */

    function getNucleusFrom2DExperiment(experiment) {
      if (typeof experiment !== 'string') {
        return [];
      }

      experiment = experiment.toLowerCase();

      if (experiment.includes('jres')) {
        return ['1H', 'Hz'];
      }

      if (experiment.includes('hmbc') || experiment.includes('hsqc')) {
        return ['1H', '13C'];
      }

      if (experiment.includes('cosy') || experiment.includes('tocsy')) {
        return ['1H', '1H'];
      }

      return [];
    }

    function checkForNucleus(nucleus) {
      nucleus = nucleus.map(value => value.replace(/[^A-Za-z0-9]/g, '').replace('NA', '').replace('off', ''));
      let beforeLength = nucleus.length;
      nucleus = nucleus.filter(value => value);
      return nucleus.length !== beforeLength ? [] : nucleus;
    }

    /**
     * Returns an experiment string based on a pulse sequence
     * @param {string} pulse
     * @return {string}
     */
    function getSpectrumType() {
      let meta = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let info = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      const {
        subfix = ''
      } = options;
      if (meta === null) meta = {};
      if (typeof meta === 'string') meta = {
        pulseSequence: meta
      };
      let spectyp = info[`${subfix}SPECTYP`];
      spectyp = (Array.isArray(spectyp) ? spectyp[0] : spectyp || '').replace(/^<(.*)>$/, '$1') // eslint-disable-line prefer-named-capture-group
      .toLowerCase();
      if (spectyp.length > 0 && spectyp !== 'undefined') return spectyp;
      let pulse = Array.isArray(meta.pulseSequence) ? meta.pulseSequence[0] : meta.pulseSequence || '';

      if (typeof pulse !== 'string') {
        return meta.dimension ? `${meta.dimension}d` : '';
      }

      pulse = pulse.toLowerCase();

      if (pulse.includes('zg') || pulse.includes('single_pulse_dec') || pulse.includes('udeft')) {
        return '1d';
      }

      if (pulse.includes('hsqct') || pulse.includes('invi') && (pulse.includes('ml') || pulse.includes('di'))) {
        return 'hsqctocsy';
      }

      if (pulse.includes('hsqc') || pulse.includes('invi')) {
        return 'hsqc';
      }

      if (pulse.includes('hmbc') || pulse.includes('inv4') && pulse.includes('lp')) {
        return 'hmbc';
      }

      if (pulse.includes('hmqc')) {
        return 'hmqc';
      }

      if (pulse.includes('cosy')) {
        return 'cosy';
      }

      if (pulse.includes('jres')) {
        return 'jres';
      }

      if (pulse.includes('tocsy') || pulse.includes('mlev') || pulse.includes('dipsi')) {
        return 'tocsy';
      }

      if (pulse.includes('noesy')) {
        return 'noesy';
      }

      if (pulse.includes('roesy')) {
        return 'roesy';
      }

      if (pulse.includes('dept')) {
        return 'dept';
      }

      if (pulse.includes('jmod') || pulse.includes('apt')) {
        return 'aptjmod';
      }

      if (pulse.includes('inad')) {
        return 'inadequate';
      }

      if (pulse.includes('adeq')) {
        return 'adequate';
      }

      return meta.dimension ? `${meta.dimension}d` : '';
    }

    function getInfoFromJCAMP(metaData) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        subfix = ''
      } = options;
      const info = {
        dimension: 0,
        nucleus: [],
        isFid: false,
        isFt: false,
        isComplex: false
      };
      let metadataString = JSON.stringify(metaData);
      const separator = metadataString.match('\r\n') ? '\r\n' : '\n';
      let {
        JCAMPDX: jcampdx = '',
        ORIGIN: origin = ''
      } = metaData;
      let creator = String(jcampdx).toLowerCase() + origin.toLowerCase();

      if (creator.includes('mestre') || creator.includes('nova')) {
        creator = 'mnova';
      }

      if (creator === 'mnova') {
        if (metaData.LONGDATE) {
          info.date = metaData.LONGDATE;
        }
      }

      info.nucleus = getNucleusFromMetadata(metaData, info, subfix);
      info.dimension = info.nucleus.length;
      maybeAdd(info, 'title', metaData.TITLE);
      maybeAdd(info, 'solvent', metaData['.SOLVENTNAME']);
      maybeAdd(info, 'temperature', metaData[`${subfix}TE`] || metaData['.TE']);
      maybeAdd(info, 'type', metaData.DATATYPE);

      if (info.type) {
        let typeLowerCase = info.type[0].toUpperCase();

        if (typeLowerCase.indexOf('FID') >= 0) {
          info.isFid = true;
          info.isComplex = true;
        } else if (typeLowerCase.indexOf('SPECTRUM') >= 0) {
          info.isFt = true;
          info.isComplex = true;
        }
      }

      maybeAdd(info, 'pulseSequence', metaData['.PULSESEQUENCE'] || metaData['.PULPROG'] || metaData[`${subfix}PULPROG`]);
      maybeAdd(info, 'experiment', getSpectrumType(info, metaData, {
        subfix
      }));
      maybeAddNumber(info, 'originFrequency', metaData['.OBSERVEFREQUENCY']);

      if (creator !== 'mnova' && creator !== 'mestre') {
        const gyromagneticRatioConst = gyromagneticRatio[info.nucleus[0]];
        maybeAdd(info, 'probeName', metaData[`${subfix}PROBHD`]);
        maybeAdd(info, 'originFrequency', metaData[`${subfix}SFO1`]);
        maybeAdd(info, 'baseFrequency', metaData[`${subfix}BF1`]);

        if (!('baseFrequency' in info) && 'originFrequency' in info) {
          maybeAdd(info, 'baseFrequency', info.originFrequency);
        }

        if (!['baseFrequency', 'originFrequency'].some(e => !info[e])) {
          const {
            baseFrequency,
            originFrequency
          } = info;
          let fieldStrength = 2 * Math.PI * (baseFrequency[0] / gyromagneticRatioConst) * 1e6;
          let frequencyOffset = baseFrequency.map((bf, i) => (originFrequency[i] - bf) * 1e6);
          maybeAdd(info, 'fieldStrength', fieldStrength);
          maybeAdd(info, 'frequencyOffset', frequencyOffset);
        }

        maybeAddNumber(info, 'spectralWidth', metaData[`${subfix}SW`]);
        maybeAddNumber(info, 'spectralWidth', metaData[`${subfix}QM_SPECTRAL_WIDTH`]);
        maybeAdd(info, 'numberOfPoints', metaData[`${subfix}TD`]);
        const numberOfPoints = info.numberOfPoints;
        maybeAdd(info, 'sampleName', metaData[`${subfix}NAME`]);

        if (metaData[`${subfix}FNTYPE`] !== undefined) {
          maybeAdd(info, 'acquisitionMode', parseInt(metaData[`${subfix}FNTYPE`], 10));
        }

        let varName = metaData[`${subfix}VARNAME`] ? metaData[`${subfix}VARNAME`].split(',')[0] : '';

        if (varName === 'TIME') {
          let value = typeof metaData.LAST === 'string' || metaData.LAST instanceof String ? metaData.LAST.replace(' ', '').split(',')[0] : metaData.LAST;
          maybeAddNumber(info, 'acquisitionTime', value);
        }

        if (!info.acquisitionTime) {
          if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
            const {
              spectralWidth,
              originFrequency
            } = info;
            maybeAdd(info, 'acquisitionTime', Number((numberOfPoints[0] - 1) / (2 * spectralWidth[0] * originFrequency[0])));
          }
        }

        if (metaData[`${subfix}P`]) {
          let pulseStrength = 1e6 / (metaData[`${subfix}P`].split(separator)[1].split(' ')[1] * 4);
          maybeAdd(info, 'pulseStrength90', pulseStrength);
        }

        if (metaData[`${subfix}D`]) {
          let relaxationTime = metaData[`${subfix}D`].split(separator)[1].split(' ')[1];
          maybeAddNumber(info, 'relaxationTime', relaxationTime);
        }

        maybeAddNumber(info, 'numberOfScans', metaData[`${subfix}NS`]);
        maybeAddNumber(info, 'numberOfScans', metaData[`${subfix}QM_NSCANS`]);
        let increment;

        if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
          const {
            spectralWidth,
            numberOfPoints
          } = info;

          if (info.isFid) {
            maybeAdd(info, 'groupDelay', metaData[`${subfix}GRPDLY`] || 0);
            maybeAdd(info, 'DSPFVS', metaData[`${subfix}DSPFVS`]);
            maybeAdd(info, 'DECIM', metaData[`${subfix}DECIM`]);

            if (!['groupDelay', 'DSPFVS', 'DECIM'].some(e => !info[e])) {
              let {
                groupDelay,
                DSPFVS,
                DECIM
              } = info;
              let digitalFilterParameters = getDigitalFilterParameters(groupDelay[0], DSPFVS[0], DECIM[0]);
              maybeAdd(info, 'digitalFilter', digitalFilterParameters);
            }

            increment = numberOfPoints.map(nb => {
              return info.acquisitionTime[0] / (nb - 1);
            });
          } else {
            increment = numberOfPoints.map((nb, i) => {
              return spectralWidth[i] / (nb - 1);
            });
          }
        }

        maybeAdd(info, 'increment', increment);

        if (metaData[`${subfix}DATE`]) {
          info.date = new Date(parseInt(metaData[`${subfix}DATE`], 10) * 1000).toISOString();
        }

        if (!info.solvent) {
          maybeAdd(info, 'solvent', Array.isArray(metaData[`${subfix}SOLVENT`]) ? metaData[`${subfix}SOLVENT`][0] : metaData[`${subfix}SOLVENT`]);
        }
      }

      if (metaData.SYMBOL) {
        let symbols = metaData.SYMBOL.split(/[, ]+/);

        if (symbols.includes('R') && symbols.includes('I')) {
          info.isComplex = true;
        }
      }

      for (let key in info) {
        if (info[key].length === 1) info[key] = info[key][0];
      }

      if (!Array.isArray(info.nucleus)) info.nucleus = [info.nucleus];
      return info;
    }

    function maybeAddNumber(obj, name, value) {
      if (value === undefined) return;

      if (typeof value === 'string') {
        value = Number(value.replace(/\$.*/, ''));
      }

      maybeAdd(obj, name, value);
    }

    function maybeAdd(obj, name, value) {
      if (value === undefined) return;

      if (Array.isArray(value)) {
        obj[name] = value.map(cleanValue);
      } else {
        obj[name] = [cleanValue(value)];
      }
    }

    function cleanValue(value) {
      if (typeof value === 'string') {
        if (value.startsWith('<') && value.endsWith('>')) {
          value = value.substring(1, value.length - 1);
        }

        value = value.trim();
      }

      return value;
    }

    /**
     * convert/export all the file with a Bruker structure like. It is a wrapper of brukerconverter
     * the same options you can pass.
     */

    const defaultOptions = {
      converter: {
        xy: true,
        noContour: true,
        keepRecordsRegExp: /.*/,
        profiling: true
      }
    };
    async function fromBruker(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let parseData = await convertFileList(fileList, { ...defaultOptions,
        ...options
      });
      let dataStructure = [];

      for (let entry of parseData) {
        let metadata = { ...entry.info,
          ...entry.meta
        };
        let info = getInfoFromJCAMP(metadata);
        let dimensions = [];
        let dependentVariables = [];
        let dependentVariable = {};

        if (info.dimension === 1) {
          dependentVariable.components = entry.spectra;
        } else if (info.dimension === 2) {
          entry.minMax.z = convertToFloatArray(entry.minMax.z);
          dependentVariable.components = entry.minMax;
        }

        let dimension = {
          increment: info.increment,
          numberOfPoints: info.numberOfPoints
        };

        if (info.fid) {
          dimension.coordinatesOffset = {
            magnitude: -info.digitalFilter * info.increment,
            units: 'second'
          };
        } else {
          dimension.coordinatesOffset = {
            magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
            units: 'ppm'
          };
        }

        dimensions.push(dimension);
        dependentVariables.push(dependentVariable);
        const {
          source
        } = entry;
        dataStructure.push({
          dimensions,
          dependentVariables,
          source,
          info,
          meta: metadata,
          timeStamp: new Date().valueOf(),
          version: packageJson.version
        });
      }

      return dataStructure;
    }

    const expectedTypes = ['ndnmrspectrum', 'ndnmrfid', 'nmrspectrum', 'nmrfid'];
    function fromJCAMP(buffer) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        noContour = true,
        xy = true,
        keepRecordsRegExp = /.*/,
        profiling = true
      } = options;
      let parsedData = convert(buffer, {
        noContour,
        xy,
        keepRecordsRegExp,
        profiling
      });
      let dataStructure = [];
      let entries = parsedData.flatten;

      for (let entry of entries) {
        if (!isSpectraData(entry)) continue;

        if (entry.spectra && entry.spectra.length > 0 || entry.minMax) {
          let metadata = { ...entry.info,
            ...entry.meta
          };
          let info = getInfoFromJCAMP(metadata);
          if (info.experiment === 'wobble_curve') continue;
          let dimensions = [];
          let dependentVariables = [];
          let dependentVariable = {};

          if (info.dimension === 1) {
            for (let i = 0; i < entry.spectra.length; i++) {
              let data = entry.spectra[i].data;
              data = convertToFloatArray(data);
            }

            dependentVariable.components = entry.spectra;
          } else if (info.dimension === 2) {
            entry.minMax.z = convertToFloatArray(entry.minMax.z);
            dependentVariable.components = entry.minMax;
          }

          let dimension = {
            increment: info.increment,
            numberOfPoints: info.numberOfPoints
          };

          if (info.fid) {
            dimension.coordinatesOffset = {
              magnitude: -info.digitalFilter * info.increment,
              units: 'second'
            };
          } else {
            dimension.coordinatesOffset = {
              magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
              units: 'ppm'
            };
          }

          dimensions.push(dimension);
          dependentVariables.push(dependentVariable);
          const data = {
            dimensions,
            dependentVariables,
            info,
            meta: metadata,
            timeStamp: new Date().valueOf(),
            version: packageJson.version
          };
          dataStructure.push(data);
        }
      }

      return dataStructure;
    }

    function isSpectraData(entry) {
      const {
        dataType = '',
        dataClass = ''
      } = entry;
      const inputDataType = dataType.replace(/\s/g, '').toLowerCase();
      const inputDataClass = dataClass.replace(/\s/g, '').toLowerCase();
      return expectedTypes.some(type => type === inputDataType) && inputDataClass !== 'peak table';
    }

    async function read(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        bruker: BrukerOptions = {},
        jcamp: jcampOptions = {},
        jeol: jeolOptions = {}
      } = options;
      const result = await fromBruker(fileList, BrukerOptions);

      for (const file of fileList) {
        const extension = getFileExtension(file.name);
        let processed = [];

        if (extension === 'jdf') {
          processed = fromJEOL(await file.arrayBuffer());
        } else if (extension.match(/dx/) || extension === 'jcamp') {
          processed = fromJCAMP(await file.arrayBuffer(), jcampOptions);
        }

        if (processed.length > 0) result.push(...processed);
      }

      return result;
    }

    function getFileExtension(name) {
      return name.replace(/^.*\./, '').toLowerCase();
    }

    exports.formatDependentVariable = formatDependentVariable;
    exports.formatLinearDimension = formatLinearDimension;
    exports.fromBruker = fromBruker;
    exports.fromJCAMP = fromJCAMP;
    exports.fromJEOL = fromJEOL;
    exports.read = read;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=nmr-parser.js.map
