/**
 * fifo-logger - Simple event logger for the browser and node.js
 * @version v2.0.1
 * @link https://github.com/cheminfo/fifo-logger#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.FifoLogger = {}));
})(this, (function (exports) { 'use strict';

    class LogEvent extends CustomEvent {
      constructor(data) {
        super('log', {
          detail: data
        });
      }
    }
    class ChangeEvent extends CustomEvent {
      constructor(data) {
        super('change', {
          detail: data
        });
      }
    }

    var e = class extends EventTarget {
      dispatchTypedEvent(s, t) {
        return super.dispatchEvent(t);
      }
    };

    const levels = {
      values: {
        fatal: 60,
        error: 50,
        warn: 40,
        info: 30,
        debug: 20,
        trace: 10,
        silent: 0
      },
      labels: {
        0: 'silent',
        10: 'trace',
        20: 'debug',
        30: 'info',
        40: 'warn',
        50: 'error',
        60: 'fatal'
      }
    };

    /**
     * A FIFO logger that stores the most recent events in an array.
     */
    class FifoLogger extends e {
      lastID;
      initialOptions;
      events;
      uuids;
      levelAsNumber;
      limit;
      bindings;
      level;
      constructor(options = {}) {
        super();
        this.lastID = {
          id: 0
        };
        this.initialOptions = options;
        this.uuids = [crypto.randomUUID()];
        this.events = [];
        this.level = options.level || 'info';
        this.levelAsNumber = levels.values[this.level];
        this.limit = options.limit ?? 1000;
        this.bindings = options.bindings ?? {};
      }
      setLevel(level) {
        this.level = level;
        this.levelAsNumber = levels.values[level];
      }
      setLimit(limit) {
        this.limit = limit;
        this.checkSize();
      }
      checkSize() {
        if (this.events.length > this.limit) {
          this.events.splice(0, this.events.length - this.limit);
        }
      }
      /**
       * Remove events from the current logger and its children.
       */
      clear() {
        for (let i = this.events.length - 1; i >= 0; i--) {
          if (this.events[i].uuids.includes(this.uuids[0])) {
            this.events.splice(i, 1);
          }
        }
        this.dispatchTypedEvent('change', new ChangeEvent({
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
      }
      /**
       * Get a filtered list of all log events.
       * @param options - Options.
       * @returns Array of filtered logs.
       */
      getLogs(options = {}) {
        const {
          level,
          minLevel,
          includeChildren
        } = options;
        let logs = this.events.slice();
        if (includeChildren) {
          logs = logs.filter(log => log.uuids.includes(this.uuids[0]));
        } else {
          logs = logs.filter(log => log.uuids[0] === this.uuids[0]);
        }
        if (level) {
          const levelNumber = levels.values[level];
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level === levelNumber);
        }
        if (minLevel) {
          const levelNumber = levels.values[minLevel];
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level >= levelNumber);
        }
        return logs;
      }
      /**
       * Create a child logger with additional bindings.
       * @param bindings - an object of key-value pairs to include in log lines as properties.
       * @returns The new child logger.
       */
      child(bindings) {
        const newFifoLogger = new FifoLogger(this.initialOptions);
        newFifoLogger.events = this.events;
        newFifoLogger.uuids = [crypto.randomUUID(), ...this.uuids];
        newFifoLogger.lastID = this.lastID;
        newFifoLogger.bindings = {
          ...this.bindings,
          ...bindings
        };
        newFifoLogger.addEventListener('change', event => {
          this.dispatchTypedEvent('change', new ChangeEvent(event.detail));
        });
        return newFifoLogger;
      }
      trace(value, message) {
        this.#addEvent(levels.values.trace, value, message);
      }
      debug(value, message) {
        this.#addEvent(levels.values.debug, value, message);
      }
      info(value, message) {
        this.#addEvent(levels.values.info, value, message);
      }
      warn(value, message) {
        this.#addEvent(levels.values.warn, value, message);
      }
      error(value, message) {
        this.#addEvent(levels.values.error, value, message);
      }
      fatal(value, message) {
        this.#addEvent(levels.values.fatal, value, message);
      }
      #addEvent(level, value, message) {
        if (level < this.levelAsNumber) return;
        const event = {
          id: ++this.lastID.id,
          level,
          levelLabel: levels.labels[level],
          time: Date.now(),
          uuids: this.uuids,
          message: ''
        };
        if (value instanceof Error) {
          event.message = value.toString();
          event.error = value;
          event.meta = {
            ...this.bindings
          };
        } else if (message !== undefined && typeof value === 'object') {
          event.message = message;
          event.meta = {
            ...this.bindings,
            ...value
          };
        } else if (message === undefined && typeof value === 'string') {
          event.message = value;
          event.meta = {
            ...this.bindings
          };
        } else {
          throw new Error('Invalid arguments');
        }
        this.events.push(event);
        this.checkSize();
        this.dispatchTypedEvent('log', new LogEvent({
          log: event,
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
        this.dispatchTypedEvent('change', new ChangeEvent({
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
      }
    }

    exports.ChangeEvent = ChangeEvent;
    exports.FifoLogger = FifoLogger;
    exports.LogEvent = LogEvent;

}));
//# sourceMappingURL=fifo-logger.js.map
