/*
 * spectra-processor v5.0.0
 * https://github.com/cheminfo/spectra-processor#readme
 *
 * Licensed under the MIT license.
 */
// eslint-disable-next-line @typescript-eslint/unbound-method
const toString = Object.prototype.toString;
/**
 * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
 *
 * @param value - Object to check.
 * @returns True if the object is an array or a typed array.
 */
function isAnyArray$1(value) {
  const tag = toString.call(value);
  return tag.endsWith('Array]') && !tag.includes('Big');
}

var libEsm = /*#__PURE__*/Object.freeze({
    __proto__: null,
    isAnyArray: isAnyArray$1
});

/**
 * This function
 * @param output - undefined or a new array
 * @param length - length of the output array
 * @returns
 */
function getOutputArray(output, length) {
  if (output !== undefined) {
    if (!isAnyArray$1(output)) {
      throw new TypeError('output option must be an array if specified');
    }
    if (output.length !== length) {
      throw new TypeError('the output array does not have the correct length');
    }
    return output;
  } else {
    return new Float64Array(length);
  }
}

/**
 * Calculates the median of an array.
 * @param input - Array containing values
 * @param options
 * @returns - median
 */
function xMedian(input, options = {}) {
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  const {
    exact = false
  } = options || {};
  const array = input.slice();
  const middleIndex = calcMiddle(0, array.length - 1);
  const median = quickSelect(array, middleIndex);
  if (array.length % 2 === 1 || !exact) {
    return median;
  }
  const medianNext = quickSelect(array, middleIndex + 1);
  return (median + medianNext) / 2;
}
function quickSelect(array, middleIndex) {
  let low = 0;
  let high = array.length - 1;
  let middle = 0;
  let currentLow = 0;
  let currentHigh = 0;
  while (true) {
    if (high <= low) {
      return array[middleIndex];
    }
    if (high === low + 1) {
      if (array[low] > array[high]) {
        swap(array, low, high);
      }
      return array[middleIndex];
    }
    // Find median of low, middle and high items; swap into position low
    middle = calcMiddle(low, high);
    if (array[middle] > array[high]) swap(array, middle, high);
    if (array[low] > array[high]) swap(array, low, high);
    if (array[middle] > array[low]) swap(array, middle, low);
    // Swap low item (now in position middle) into position (low+1)
    swap(array, middle, low + 1);
    // Nibble from each end towards middle, swapping items when stuck
    currentLow = low + 1;
    currentHigh = high;
    while (true) {
      do currentLow++; while (array[low] > array[currentLow]);
      do currentHigh--; while (array[currentHigh] > array[low]);
      if (currentHigh < currentLow) {
        break;
      }
      swap(array, currentLow, currentHigh);
    }
    // Swap middle item (in position low) back into correct position
    swap(array, low, currentHigh);
    // Re-set active partition
    if (currentHigh <= middleIndex) {
      low = currentLow;
    }
    if (currentHigh >= middleIndex) {
      high = currentHigh - 1;
    }
  }
}
function swap(array, i, j) {
  const temp = array[j];
  array[j] = array[i];
  array[i] = temp;
}
function calcMiddle(i, j) {
  return Math.floor((i + j) / 2);
}

/**
 * Checks if the input is a non-empty array of numbers.
 * Only checks the first element.
 * @param input - Array to check.
 * @param options - Additional checks.
 */
function xCheck(input, options = {}) {
  const {
    minLength
  } = options;
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  if (typeof input[0] !== 'number') {
    throw new TypeError('input must contain numbers');
  }
  if (minLength && input.length < minLength) {
    throw new Error(`input must have a length of at least ${minLength}`);
  }
}

/**
 * Returns the closest index of a `target`
 * @param array - array of numbers
 * @param target - target
 * @param options
 * @returns - closest index
 */
function xFindClosestIndex(array, target, options = {}) {
  const {
    sorted = true
  } = options;
  if (sorted) {
    let low = 0;
    let high = array.length - 1;
    let middle = 0;
    while (high - low > 1) {
      middle = low + (high - low >> 1);
      if (array[middle] < target) {
        low = middle;
      } else if (array[middle] > target) {
        high = middle;
      } else {
        return middle;
      }
    }
    if (low < array.length - 1) {
      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
        return low;
      } else {
        return low + 1;
      }
    } else {
      return low;
    }
  } else {
    let index = 0;
    let diff = Number.POSITIVE_INFINITY;
    for (let i = 0; i < array.length; i++) {
      const currentDiff = Math.abs(array[i] - target);
      if (currentDiff < diff) {
        diff = currentDiff;
        index = i;
      }
    }
    return index;
  }
}

/**
 * Returns an object with {fromIndex, toIndex} for a specific from / to
 * @param x - array of numbers
 * @param options - Options
 */
function xGetFromToIndex(x, options = {}) {
  let {
    fromIndex,
    toIndex
  } = options;
  const {
    from,
    to
  } = options;
  if (fromIndex === undefined) {
    if (from !== undefined) {
      fromIndex = xFindClosestIndex(x, from);
    } else {
      fromIndex = 0;
    }
  }
  if (toIndex === undefined) {
    if (to !== undefined) {
      toIndex = xFindClosestIndex(x, to);
    } else {
      toIndex = x.length - 1;
    }
  }
  if (fromIndex < 0) fromIndex = 0;
  if (toIndex < 0) toIndex = 0;
  if (fromIndex >= x.length) fromIndex = x.length - 1;
  if (toIndex >= x.length) toIndex = x.length - 1;
  if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
  return {
    fromIndex,
    toIndex
  };
}

function xAbsoluteSum(array, options = {}) {
  xCheck(array);
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(array, options);
  let sum = 0;
  for (let i = fromIndex; i <= toIndex; i++) {
    sum += Math.abs(array[i]);
  }
  return sum;
}

/**
 * This function xAdd the first array by the second array or a constant value to each element of the first array
 * @param array1 - the first array
 * @param array2 - the second array or number
 * @returns the result of the addition
 */
function xAdd(array1, array2) {
  let isConstant = false;
  let constant = 0;
  if (isAnyArray$1(array2)) {
    if (array1.length !== array2.length) {
      throw new Error('size of array1 and array2 must be identical');
    }
  } else {
    isConstant = true;
    constant = array2;
  }
  const array3 = new Float64Array(array1.length);
  if (isConstant) {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] + constant;
    }
  } else {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] + array2[i];
    }
  }
  return array3;
}

/**
 * Will apply a function on each element of the array described as a string
 * By default we will use as variable 'x'
 * In front of sequence of lowercase we will add 'Math.'. This allows to write
 * `sin(x) + cos(x)` and it will be replace internally by (x) => (Math.sin(x) + Math.cos(x))
 * @param array
 * @param options
 * @returns
 */
function xApplyFunctionStr(array, options = {}) {
  const {
    variableLabel = 'x',
    fctString = variableLabel
  } = options;
  const fct = new Function(variableLabel, `return Number(${fctString.replaceAll(/(?<before>^|\W)(?<after>[\da-z]{2,}\()/g, '$<before>Math.$<after>').replaceAll('Math.Math', 'Math')})`);
  const toReturn = Float64Array.from(array);
  for (let i = 0; i < array.length; i++) {
    toReturn[i] = fct(array[i]);
    if (Number.isNaN(toReturn[i])) {
      throw new Error(`The callback ${fctString} does not return a number: ${array[i]}`);
    }
  }
  return toReturn;
}

/**
 * This function xMultiply the first array by the second array or a constant value to each element of the first array
 * @param array1 - first array
 * @param array2 - second array
 * @param options - options
 */
function xMultiply(array1, array2, options = {}) {
  let isConstant = false;
  let constant = 0;
  if (isAnyArray$1(array2)) {
    if (array1.length !== array2.length) {
      throw new Error('size of array1 and array2 must be identical');
    }
  } else {
    isConstant = true;
    constant = array2;
  }
  const array3 = getOutputArray(options.output, array1.length);
  if (isConstant) {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] * constant;
    }
  } else {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] * array2[i];
    }
  }
  return array3;
}

/**
 * Calculates the correlation between 2 vectors
 * https://en.wikipedia.org/wiki/Correlation_and_dependence
 * @param A - first array
 * @param B - sencond array
 */
function xCorrelation(A, B) {
  const n = A.length;
  let sumA = 0;
  let sumA2 = 0;
  let sumB = 0;
  let sumB2 = 0;
  let sumAB = 0;
  for (let i = 0; i < n; i++) {
    const a = A[i];
    const b = B[i];
    sumA += a;
    sumA2 += a ** 2;
    sumB += b;
    sumB2 += b ** 2;
    sumAB += a * b;
  }
  return (n * sumAB - sumA * sumB) / (Math.sqrt(n * sumA2 - sumA ** 2) * Math.sqrt(n * sumB2 - sumB ** 2));
}

function getDefaultExportFromCjs (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
}

function getAugmentedNamespace(n) {
  if (Object.prototype.hasOwnProperty.call(n, '__esModule')) return n;
  var f = n.default;
	if (typeof f == "function") {
		var a = function a () {
			var isInstance = false;
      try {
        isInstance = this instanceof a;
      } catch {}
			if (isInstance) {
        return Reflect.construct(f, arguments, this.constructor);
			}
			return f.apply(this, arguments);
		};
		a.prototype = f.prototype;
  } else a = {};
  Object.defineProperty(a, '__esModule', {value: true});
	Object.keys(n).forEach(function (k) {
		var d = Object.getOwnPropertyDescriptor(n, k);
		Object.defineProperty(a, k, d.get ? d : {
			enumerable: true,
			get: function () {
				return n[k];
			}
		});
	});
	return a;
}

var matrix = {};

var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

function max$6(input) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  var _options$fromIndex = options.fromIndex,
    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
    _options$toIndex = options.toIndex,
    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
    throw new Error('fromIndex must be a positive integer smaller than length');
  }
  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
  }
  var maxValue = input[fromIndex];
  for (var i = fromIndex + 1; i < toIndex; i++) {
    if (input[i] > maxValue) maxValue = input[i];
  }
  return maxValue;
}

function min$6(input) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  var _options$fromIndex = options.fromIndex,
    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
    _options$toIndex = options.toIndex,
    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
    throw new Error('fromIndex must be a positive integer smaller than length');
  }
  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
  }
  var minValue = input[fromIndex];
  for (var i = fromIndex + 1; i < toIndex; i++) {
    if (input[i] < minValue) minValue = input[i];
  }
  return minValue;
}

function rescale$2(input) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  } else if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  var output;
  if (options.output !== undefined) {
    if (!isAnyArray$1(options.output)) {
      throw new TypeError('output option must be an array if specified');
    }
    output = options.output;
  } else {
    output = new Array(input.length);
  }
  var currentMin = min$6(input);
  var currentMax = max$6(input);
  if (currentMin === currentMax) {
    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
  }
  var _options$min = options.min,
    minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
    _options$max = options.max,
    maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
  if (minValue >= maxValue) {
    throw new RangeError('min option must be smaller than max option');
  }
  var factor = (maxValue - minValue) / (currentMax - currentMin);
  for (var i = 0; i < input.length; i++) {
    output[i] = (input[i] - currentMin) * factor + minValue;
  }
  return output;
}

var libEs6 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    default: rescale$2
});

var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

Object.defineProperty(matrix, '__esModule', {
  value: true
});
var isAnyArray = require$$0;
var rescale$1 = require$$1;
const indent = ' '.repeat(2);
const indentData = ' '.repeat(4);

/**
 * @this {Matrix}
 * @returns {string}
 */
function inspectMatrix() {
  return inspectMatrixWithOptions(this);
}
function inspectMatrixWithOptions(matrix, options = {}) {
  const {
    maxRows = 15,
    maxColumns = 10,
    maxNumSize = 8,
    padMinus = 'auto'
  } = options;
  return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
}
function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
  const {
    rows,
    columns
  } = matrix;
  const maxI = Math.min(rows, maxRows);
  const maxJ = Math.min(columns, maxColumns);
  const result = [];
  if (padMinus === 'auto') {
    padMinus = false;
    loop: for (let i = 0; i < maxI; i++) {
      for (let j = 0; j < maxJ; j++) {
        if (matrix.get(i, j) < 0) {
          padMinus = true;
          break loop;
        }
      }
    }
  }
  for (let i = 0; i < maxI; i++) {
    let line = [];
    for (let j = 0; j < maxJ; j++) {
      line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
    }
    result.push(`${line.join(' ')}`);
  }
  if (maxJ !== columns) {
    result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
  }
  if (maxI !== rows) {
    result.push(`... ${rows - maxRows} more rows`);
  }
  return result.join(`\n${indentData}`);
}
function formatNumber(num, maxNumSize, padMinus) {
  return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
}
function formatNumber2(num, len) {
  // small.length numbers should be as is
  let str = num.toString();
  if (str.length <= len) return str;

  // (7)'0.00123' is better then (7)'1.23e-2'
  // (8)'0.000123' is worse then (7)'1.23e-3',
  let fix = num.toFixed(len);
  if (fix.length > len) {
    fix = num.toFixed(Math.max(0, len - (fix.length - len)));
  }
  if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
    return fix;
  }

  // well, if it's still too long the user should've used longer numbers
  let exp = num.toExponential(len);
  if (exp.length > len) {
    exp = num.toExponential(Math.max(0, len - (exp.length - len)));
  }
  return exp.slice(0);
}
function installMathOperations(AbstractMatrix, Matrix) {
  AbstractMatrix.prototype.add = function add(value) {
    if (typeof value === 'number') return this.addS(value);
    return this.addM(value);
  };
  AbstractMatrix.prototype.addS = function addS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) + value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.addM = function addM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) + matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.add = function add(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.add(value);
  };
  AbstractMatrix.prototype.sub = function sub(value) {
    if (typeof value === 'number') return this.subS(value);
    return this.subM(value);
  };
  AbstractMatrix.prototype.subS = function subS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) - value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.subM = function subM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) - matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.sub = function sub(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.sub(value);
  };
  AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
  AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
  AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
  AbstractMatrix.subtract = AbstractMatrix.sub;
  AbstractMatrix.prototype.mul = function mul(value) {
    if (typeof value === 'number') return this.mulS(value);
    return this.mulM(value);
  };
  AbstractMatrix.prototype.mulS = function mulS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) * value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.mulM = function mulM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) * matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.mul = function mul(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.mul(value);
  };
  AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
  AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
  AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
  AbstractMatrix.multiply = AbstractMatrix.mul;
  AbstractMatrix.prototype.div = function div(value) {
    if (typeof value === 'number') return this.divS(value);
    return this.divM(value);
  };
  AbstractMatrix.prototype.divS = function divS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) / value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.divM = function divM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) / matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.div = function div(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.div(value);
  };
  AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
  AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
  AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
  AbstractMatrix.divide = AbstractMatrix.div;
  AbstractMatrix.prototype.mod = function mod(value) {
    if (typeof value === 'number') return this.modS(value);
    return this.modM(value);
  };
  AbstractMatrix.prototype.modS = function modS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) % value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.modM = function modM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) % matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.mod = function mod(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.mod(value);
  };
  AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
  AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
  AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
  AbstractMatrix.modulus = AbstractMatrix.mod;
  AbstractMatrix.prototype.and = function and(value) {
    if (typeof value === 'number') return this.andS(value);
    return this.andM(value);
  };
  AbstractMatrix.prototype.andS = function andS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) & value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.andM = function andM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) & matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.and = function and(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.and(value);
  };
  AbstractMatrix.prototype.or = function or(value) {
    if (typeof value === 'number') return this.orS(value);
    return this.orM(value);
  };
  AbstractMatrix.prototype.orS = function orS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) | value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.orM = function orM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) | matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.or = function or(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.or(value);
  };
  AbstractMatrix.prototype.xor = function xor(value) {
    if (typeof value === 'number') return this.xorS(value);
    return this.xorM(value);
  };
  AbstractMatrix.prototype.xorS = function xorS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) ^ value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.xorM = function xorM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.xor = function xor(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.xor(value);
  };
  AbstractMatrix.prototype.leftShift = function leftShift(value) {
    if (typeof value === 'number') return this.leftShiftS(value);
    return this.leftShiftM(value);
  };
  AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) << value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) << matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.leftShift = function leftShift(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.leftShift(value);
  };
  AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
    if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
    return this.signPropagatingRightShiftM(value);
  };
  AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) >> value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) >> matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.signPropagatingRightShift(value);
  };
  AbstractMatrix.prototype.rightShift = function rightShift(value) {
    if (typeof value === 'number') return this.rightShiftS(value);
    return this.rightShiftM(value);
  };
  AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) >>> value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.rightShift = function rightShift(matrix, value) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.rightShift(value);
  };
  AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
  AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
  AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
  AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
  AbstractMatrix.prototype.not = function not() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, ~this.get(i, j));
      }
    }
    return this;
  };
  AbstractMatrix.not = function not(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.not();
  };
  AbstractMatrix.prototype.abs = function abs() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.abs(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.abs = function abs(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.abs();
  };
  AbstractMatrix.prototype.acos = function acos() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.acos(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.acos = function acos(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.acos();
  };
  AbstractMatrix.prototype.acosh = function acosh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.acosh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.acosh = function acosh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.acosh();
  };
  AbstractMatrix.prototype.asin = function asin() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.asin(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.asin = function asin(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.asin();
  };
  AbstractMatrix.prototype.asinh = function asinh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.asinh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.asinh = function asinh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.asinh();
  };
  AbstractMatrix.prototype.atan = function atan() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.atan(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.atan = function atan(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.atan();
  };
  AbstractMatrix.prototype.atanh = function atanh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.atanh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.atanh = function atanh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.atanh();
  };
  AbstractMatrix.prototype.cbrt = function cbrt() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.cbrt(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.cbrt = function cbrt(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.cbrt();
  };
  AbstractMatrix.prototype.ceil = function ceil() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.ceil(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.ceil = function ceil(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.ceil();
  };
  AbstractMatrix.prototype.clz32 = function clz32() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.clz32(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.clz32 = function clz32(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.clz32();
  };
  AbstractMatrix.prototype.cos = function cos() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.cos(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.cos = function cos(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.cos();
  };
  AbstractMatrix.prototype.cosh = function cosh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.cosh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.cosh = function cosh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.cosh();
  };
  AbstractMatrix.prototype.exp = function exp() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.exp(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.exp = function exp(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.exp();
  };
  AbstractMatrix.prototype.expm1 = function expm1() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.expm1(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.expm1 = function expm1(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.expm1();
  };
  AbstractMatrix.prototype.floor = function floor() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.floor(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.floor = function floor(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.floor();
  };
  AbstractMatrix.prototype.fround = function fround() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.fround(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.fround = function fround(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.fround();
  };
  AbstractMatrix.prototype.log = function log() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.log(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.log = function log(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.log();
  };
  AbstractMatrix.prototype.log1p = function log1p() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.log1p(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.log1p = function log1p(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.log1p();
  };
  AbstractMatrix.prototype.log10 = function log10() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.log10(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.log10 = function log10(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.log10();
  };
  AbstractMatrix.prototype.log2 = function log2() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.log2(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.log2 = function log2(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.log2();
  };
  AbstractMatrix.prototype.round = function round() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.round(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.round = function round(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.round();
  };
  AbstractMatrix.prototype.sign = function sign() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.sign(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.sign = function sign(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.sign();
  };
  AbstractMatrix.prototype.sin = function sin() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.sin(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.sin = function sin(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.sin();
  };
  AbstractMatrix.prototype.sinh = function sinh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.sinh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.sinh = function sinh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.sinh();
  };
  AbstractMatrix.prototype.sqrt = function sqrt() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.sqrt(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.sqrt = function sqrt(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.sqrt();
  };
  AbstractMatrix.prototype.tan = function tan() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.tan(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.tan = function tan(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.tan();
  };
  AbstractMatrix.prototype.tanh = function tanh() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.tanh(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.tanh = function tanh(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.tanh();
  };
  AbstractMatrix.prototype.trunc = function trunc() {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, Math.trunc(this.get(i, j)));
      }
    }
    return this;
  };
  AbstractMatrix.trunc = function trunc(matrix) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.trunc();
  };
  AbstractMatrix.pow = function pow(matrix, arg0) {
    const newMatrix = new Matrix(matrix);
    return newMatrix.pow(arg0);
  };
  AbstractMatrix.prototype.pow = function pow(value) {
    if (typeof value === 'number') return this.powS(value);
    return this.powM(value);
  };
  AbstractMatrix.prototype.powS = function powS(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) ** value);
      }
    }
    return this;
  };
  AbstractMatrix.prototype.powM = function powM(matrix) {
    matrix = Matrix.checkMatrix(matrix);
    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
      throw new RangeError('Matrices dimensions must be equal');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) ** matrix.get(i, j));
      }
    }
    return this;
  };
}

/**
 * @private
 * Check that a row index is not out of bounds
 * @param {Matrix} matrix
 * @param {number} index
 * @param {boolean} [outer]
 */
function checkRowIndex(matrix, index, outer) {
  let max = outer ? matrix.rows : matrix.rows - 1;
  if (index < 0 || index > max) {
    throw new RangeError('Row index out of range');
  }
}

/**
 * @private
 * Check that a column index is not out of bounds
 * @param {Matrix} matrix
 * @param {number} index
 * @param {boolean} [outer]
 */
function checkColumnIndex(matrix, index, outer) {
  let max = outer ? matrix.columns : matrix.columns - 1;
  if (index < 0 || index > max) {
    throw new RangeError('Column index out of range');
  }
}

/**
 * @private
 * Check that the provided vector is an array with the right length
 * @param {Matrix} matrix
 * @param {Array|Matrix} vector
 * @return {Array}
 * @throws {RangeError}
 */
function checkRowVector(matrix, vector) {
  if (vector.to1DArray) {
    vector = vector.to1DArray();
  }
  if (vector.length !== matrix.columns) {
    throw new RangeError('vector size must be the same as the number of columns');
  }
  return vector;
}

/**
 * @private
 * Check that the provided vector is an array with the right length
 * @param {Matrix} matrix
 * @param {Array|Matrix} vector
 * @return {Array}
 * @throws {RangeError}
 */
function checkColumnVector(matrix, vector) {
  if (vector.to1DArray) {
    vector = vector.to1DArray();
  }
  if (vector.length !== matrix.rows) {
    throw new RangeError('vector size must be the same as the number of rows');
  }
  return vector;
}
function checkRowIndices(matrix, rowIndices) {
  if (!isAnyArray.isAnyArray(rowIndices)) {
    throw new TypeError('row indices must be an array');
  }
  for (let i = 0; i < rowIndices.length; i++) {
    if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
      throw new RangeError('row indices are out of range');
    }
  }
}
function checkColumnIndices(matrix, columnIndices) {
  if (!isAnyArray.isAnyArray(columnIndices)) {
    throw new TypeError('column indices must be an array');
  }
  for (let i = 0; i < columnIndices.length; i++) {
    if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
      throw new RangeError('column indices are out of range');
    }
  }
}
function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
  if (arguments.length !== 5) {
    throw new RangeError('expected 4 arguments');
  }
  checkNumber('startRow', startRow);
  checkNumber('endRow', endRow);
  checkNumber('startColumn', startColumn);
  checkNumber('endColumn', endColumn);
  if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
    throw new RangeError('Submatrix indices are out of range');
  }
}
function newArray(length, value = 0) {
  let array = [];
  for (let i = 0; i < length; i++) {
    array.push(value);
  }
  return array;
}
function checkNumber(name, value) {
  if (typeof value !== 'number') {
    throw new TypeError(`${name} must be a number`);
  }
}
function checkNonEmpty(matrix) {
  if (matrix.isEmpty()) {
    throw new Error('Empty matrix has no elements to index');
  }
}
function sumByRow(matrix) {
  let sum = newArray(matrix.rows);
  for (let i = 0; i < matrix.rows; ++i) {
    for (let j = 0; j < matrix.columns; ++j) {
      sum[i] += matrix.get(i, j);
    }
  }
  return sum;
}
function sumByColumn(matrix) {
  let sum = newArray(matrix.columns);
  for (let i = 0; i < matrix.rows; ++i) {
    for (let j = 0; j < matrix.columns; ++j) {
      sum[j] += matrix.get(i, j);
    }
  }
  return sum;
}
function sumAll(matrix) {
  let v = 0;
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      v += matrix.get(i, j);
    }
  }
  return v;
}
function productByRow(matrix) {
  let sum = newArray(matrix.rows, 1);
  for (let i = 0; i < matrix.rows; ++i) {
    for (let j = 0; j < matrix.columns; ++j) {
      sum[i] *= matrix.get(i, j);
    }
  }
  return sum;
}
function productByColumn(matrix) {
  let sum = newArray(matrix.columns, 1);
  for (let i = 0; i < matrix.rows; ++i) {
    for (let j = 0; j < matrix.columns; ++j) {
      sum[j] *= matrix.get(i, j);
    }
  }
  return sum;
}
function productAll(matrix) {
  let v = 1;
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      v *= matrix.get(i, j);
    }
  }
  return v;
}
function varianceByRow(matrix, unbiased, mean) {
  const rows = matrix.rows;
  const cols = matrix.columns;
  const variance = [];
  for (let i = 0; i < rows; i++) {
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;
    for (let j = 0; j < cols; j++) {
      x = matrix.get(i, j) - mean[i];
      sum1 += x;
      sum2 += x * x;
    }
    if (unbiased) {
      variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
    } else {
      variance.push((sum2 - sum1 * sum1 / cols) / cols);
    }
  }
  return variance;
}
function varianceByColumn(matrix, unbiased, mean) {
  const rows = matrix.rows;
  const cols = matrix.columns;
  const variance = [];
  for (let j = 0; j < cols; j++) {
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;
    for (let i = 0; i < rows; i++) {
      x = matrix.get(i, j) - mean[j];
      sum1 += x;
      sum2 += x * x;
    }
    if (unbiased) {
      variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
    } else {
      variance.push((sum2 - sum1 * sum1 / rows) / rows);
    }
  }
  return variance;
}
function varianceAll(matrix, unbiased, mean) {
  const rows = matrix.rows;
  const cols = matrix.columns;
  const size = rows * cols;
  let sum1 = 0;
  let sum2 = 0;
  let x = 0;
  for (let i = 0; i < rows; i++) {
    for (let j = 0; j < cols; j++) {
      x = matrix.get(i, j) - mean;
      sum1 += x;
      sum2 += x * x;
    }
  }
  if (unbiased) {
    return (sum2 - sum1 * sum1 / size) / (size - 1);
  } else {
    return (sum2 - sum1 * sum1 / size) / size;
  }
}
function centerByRow(matrix, mean) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) - mean[i]);
    }
  }
}
function centerByColumn(matrix, mean) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) - mean[j]);
    }
  }
}
function centerAll(matrix, mean) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) - mean);
    }
  }
}
function getScaleByRow(matrix) {
  const scale = [];
  for (let i = 0; i < matrix.rows; i++) {
    let sum = 0;
    for (let j = 0; j < matrix.columns; j++) {
      sum += matrix.get(i, j) ** 2 / (matrix.columns - 1);
    }
    scale.push(Math.sqrt(sum));
  }
  return scale;
}
function scaleByRow(matrix, scale) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) / scale[i]);
    }
  }
}
function getScaleByColumn(matrix) {
  const scale = [];
  for (let j = 0; j < matrix.columns; j++) {
    let sum = 0;
    for (let i = 0; i < matrix.rows; i++) {
      sum += matrix.get(i, j) ** 2 / (matrix.rows - 1);
    }
    scale.push(Math.sqrt(sum));
  }
  return scale;
}
function scaleByColumn(matrix, scale) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) / scale[j]);
    }
  }
}
function getScaleAll(matrix) {
  const divider = matrix.size - 1;
  let sum = 0;
  for (let j = 0; j < matrix.columns; j++) {
    for (let i = 0; i < matrix.rows; i++) {
      sum += matrix.get(i, j) ** 2 / divider;
    }
  }
  return Math.sqrt(sum);
}
function scaleAll(matrix, scale) {
  for (let i = 0; i < matrix.rows; i++) {
    for (let j = 0; j < matrix.columns; j++) {
      matrix.set(i, j, matrix.get(i, j) / scale);
    }
  }
}
class AbstractMatrix {
  static from1DArray(newRows, newColumns, newData) {
    let length = newRows * newColumns;
    if (length !== newData.length) {
      throw new RangeError('data length does not match given dimensions');
    }
    let newMatrix = new Matrix$1(newRows, newColumns);
    for (let row = 0; row < newRows; row++) {
      for (let column = 0; column < newColumns; column++) {
        newMatrix.set(row, column, newData[row * newColumns + column]);
      }
    }
    return newMatrix;
  }
  static rowVector(newData) {
    let vector = new Matrix$1(1, newData.length);
    for (let i = 0; i < newData.length; i++) {
      vector.set(0, i, newData[i]);
    }
    return vector;
  }
  static columnVector(newData) {
    let vector = new Matrix$1(newData.length, 1);
    for (let i = 0; i < newData.length; i++) {
      vector.set(i, 0, newData[i]);
    }
    return vector;
  }
  static zeros(rows, columns) {
    return new Matrix$1(rows, columns);
  }
  static ones(rows, columns) {
    return new Matrix$1(rows, columns).fill(1);
  }
  static rand(rows, columns, options = {}) {
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      random = Math.random
    } = options;
    let matrix = new Matrix$1(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        matrix.set(i, j, random());
      }
    }
    return matrix;
  }
  static randInt(rows, columns, options = {}) {
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      min = 0,
      max = 1000,
      random = Math.random
    } = options;
    if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
    if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
    if (min >= max) throw new RangeError('min must be smaller than max');
    let interval = max - min;
    let matrix = new Matrix$1(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        let value = min + Math.round(random() * interval);
        matrix.set(i, j, value);
      }
    }
    return matrix;
  }
  static eye(rows, columns, value) {
    if (columns === undefined) columns = rows;
    if (value === undefined) value = 1;
    let min = Math.min(rows, columns);
    let matrix = this.zeros(rows, columns);
    for (let i = 0; i < min; i++) {
      matrix.set(i, i, value);
    }
    return matrix;
  }
  static diag(data, rows, columns) {
    let l = data.length;
    if (rows === undefined) rows = l;
    if (columns === undefined) columns = rows;
    let min = Math.min(l, rows, columns);
    let matrix = this.zeros(rows, columns);
    for (let i = 0; i < min; i++) {
      matrix.set(i, i, data[i]);
    }
    return matrix;
  }
  static min(matrix1, matrix2) {
    matrix1 = this.checkMatrix(matrix1);
    matrix2 = this.checkMatrix(matrix2);
    let rows = matrix1.rows;
    let columns = matrix1.columns;
    let result = new Matrix$1(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
      }
    }
    return result;
  }
  static max(matrix1, matrix2) {
    matrix1 = this.checkMatrix(matrix1);
    matrix2 = this.checkMatrix(matrix2);
    let rows = matrix1.rows;
    let columns = matrix1.columns;
    let result = new this(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
      }
    }
    return result;
  }
  static checkMatrix(value) {
    return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
  }
  static isMatrix(value) {
    return value != null && value.klass === 'Matrix';
  }
  get size() {
    return this.rows * this.columns;
  }
  apply(callback) {
    if (typeof callback !== 'function') {
      throw new TypeError('callback must be a function');
    }
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        callback.call(this, i, j);
      }
    }
    return this;
  }
  to1DArray() {
    let array = [];
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        array.push(this.get(i, j));
      }
    }
    return array;
  }
  to2DArray() {
    let copy = [];
    for (let i = 0; i < this.rows; i++) {
      copy.push([]);
      for (let j = 0; j < this.columns; j++) {
        copy[i].push(this.get(i, j));
      }
    }
    return copy;
  }
  toJSON() {
    return this.to2DArray();
  }
  isRowVector() {
    return this.rows === 1;
  }
  isColumnVector() {
    return this.columns === 1;
  }
  isVector() {
    return this.rows === 1 || this.columns === 1;
  }
  isSquare() {
    return this.rows === this.columns;
  }
  isEmpty() {
    return this.rows === 0 || this.columns === 0;
  }
  isSymmetric() {
    if (this.isSquare()) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j <= i; j++) {
          if (this.get(i, j) !== this.get(j, i)) {
            return false;
          }
        }
      }
      return true;
    }
    return false;
  }
  isDistance() {
    if (!this.isSymmetric()) return false;
    for (let i = 0; i < this.rows; i++) {
      if (this.get(i, i) !== 0) return false;
    }
    return true;
  }
  isEchelonForm() {
    let i = 0;
    let j = 0;
    let previousColumn = -1;
    let isEchelonForm = true;
    let checked = false;
    while (i < this.rows && isEchelonForm) {
      j = 0;
      checked = false;
      while (j < this.columns && checked === false) {
        if (this.get(i, j) === 0) {
          j++;
        } else if (this.get(i, j) === 1 && j > previousColumn) {
          checked = true;
          previousColumn = j;
        } else {
          isEchelonForm = false;
          checked = true;
        }
      }
      i++;
    }
    return isEchelonForm;
  }
  isReducedEchelonForm() {
    let i = 0;
    let j = 0;
    let previousColumn = -1;
    let isReducedEchelonForm = true;
    let checked = false;
    while (i < this.rows && isReducedEchelonForm) {
      j = 0;
      checked = false;
      while (j < this.columns && checked === false) {
        if (this.get(i, j) === 0) {
          j++;
        } else if (this.get(i, j) === 1 && j > previousColumn) {
          checked = true;
          previousColumn = j;
        } else {
          isReducedEchelonForm = false;
          checked = true;
        }
      }
      for (let k = j + 1; k < this.rows; k++) {
        if (this.get(i, k) !== 0) {
          isReducedEchelonForm = false;
        }
      }
      i++;
    }
    return isReducedEchelonForm;
  }
  echelonForm() {
    let result = this.clone();
    let h = 0;
    let k = 0;
    while (h < result.rows && k < result.columns) {
      let iMax = h;
      for (let i = h; i < result.rows; i++) {
        if (result.get(i, k) > result.get(iMax, k)) {
          iMax = i;
        }
      }
      if (result.get(iMax, k) === 0) {
        k++;
      } else {
        result.swapRows(h, iMax);
        let tmp = result.get(h, k);
        for (let j = k; j < result.columns; j++) {
          result.set(h, j, result.get(h, j) / tmp);
        }
        for (let i = h + 1; i < result.rows; i++) {
          let factor = result.get(i, k) / result.get(h, k);
          result.set(i, k, 0);
          for (let j = k + 1; j < result.columns; j++) {
            result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
          }
        }
        h++;
        k++;
      }
    }
    return result;
  }
  reducedEchelonForm() {
    let result = this.echelonForm();
    let m = result.columns;
    let n = result.rows;
    let h = n - 1;
    while (h >= 0) {
      if (result.maxRow(h) === 0) {
        h--;
      } else {
        let p = 0;
        let pivot = false;
        while (p < n && pivot === false) {
          if (result.get(h, p) === 1) {
            pivot = true;
          } else {
            p++;
          }
        }
        for (let i = 0; i < h; i++) {
          let factor = result.get(i, p);
          for (let j = p; j < m; j++) {
            let tmp = result.get(i, j) - factor * result.get(h, j);
            result.set(i, j, tmp);
          }
        }
        h--;
      }
    }
    return result;
  }
  set() {
    throw new Error('set method is unimplemented');
  }
  get() {
    throw new Error('get method is unimplemented');
  }
  repeat(options = {}) {
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      rows = 1,
      columns = 1
    } = options;
    if (!Number.isInteger(rows) || rows <= 0) {
      throw new TypeError('rows must be a positive integer');
    }
    if (!Number.isInteger(columns) || columns <= 0) {
      throw new TypeError('columns must be a positive integer');
    }
    let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        matrix.setSubMatrix(this, this.rows * i, this.columns * j);
      }
    }
    return matrix;
  }
  fill(value) {
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, value);
      }
    }
    return this;
  }
  neg() {
    return this.mulS(-1);
  }
  getRow(index) {
    checkRowIndex(this, index);
    let row = [];
    for (let i = 0; i < this.columns; i++) {
      row.push(this.get(index, i));
    }
    return row;
  }
  getRowVector(index) {
    return Matrix$1.rowVector(this.getRow(index));
  }
  setRow(index, array) {
    checkRowIndex(this, index);
    array = checkRowVector(this, array);
    for (let i = 0; i < this.columns; i++) {
      this.set(index, i, array[i]);
    }
    return this;
  }
  swapRows(row1, row2) {
    checkRowIndex(this, row1);
    checkRowIndex(this, row2);
    for (let i = 0; i < this.columns; i++) {
      let temp = this.get(row1, i);
      this.set(row1, i, this.get(row2, i));
      this.set(row2, i, temp);
    }
    return this;
  }
  getColumn(index) {
    checkColumnIndex(this, index);
    let column = [];
    for (let i = 0; i < this.rows; i++) {
      column.push(this.get(i, index));
    }
    return column;
  }
  getColumnVector(index) {
    return Matrix$1.columnVector(this.getColumn(index));
  }
  setColumn(index, array) {
    checkColumnIndex(this, index);
    array = checkColumnVector(this, array);
    for (let i = 0; i < this.rows; i++) {
      this.set(i, index, array[i]);
    }
    return this;
  }
  swapColumns(column1, column2) {
    checkColumnIndex(this, column1);
    checkColumnIndex(this, column2);
    for (let i = 0; i < this.rows; i++) {
      let temp = this.get(i, column1);
      this.set(i, column1, this.get(i, column2));
      this.set(i, column2, temp);
    }
    return this;
  }
  addRowVector(vector) {
    vector = checkRowVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) + vector[j]);
      }
    }
    return this;
  }
  subRowVector(vector) {
    vector = checkRowVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) - vector[j]);
      }
    }
    return this;
  }
  mulRowVector(vector) {
    vector = checkRowVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) * vector[j]);
      }
    }
    return this;
  }
  divRowVector(vector) {
    vector = checkRowVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) / vector[j]);
      }
    }
    return this;
  }
  addColumnVector(vector) {
    vector = checkColumnVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) + vector[i]);
      }
    }
    return this;
  }
  subColumnVector(vector) {
    vector = checkColumnVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) - vector[i]);
      }
    }
    return this;
  }
  mulColumnVector(vector) {
    vector = checkColumnVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) * vector[i]);
      }
    }
    return this;
  }
  divColumnVector(vector) {
    vector = checkColumnVector(this, vector);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        this.set(i, j, this.get(i, j) / vector[i]);
      }
    }
    return this;
  }
  mulRow(index, value) {
    checkRowIndex(this, index);
    for (let i = 0; i < this.columns; i++) {
      this.set(index, i, this.get(index, i) * value);
    }
    return this;
  }
  mulColumn(index, value) {
    checkColumnIndex(this, index);
    for (let i = 0; i < this.rows; i++) {
      this.set(i, index, this.get(i, index) * value);
    }
    return this;
  }
  max(by) {
    if (this.isEmpty()) {
      return NaN;
    }
    switch (by) {
      case 'row':
        {
          const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) > max[row]) {
                max[row] = this.get(row, column);
              }
            }
          }
          return max;
        }
      case 'column':
        {
          const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) > max[column]) {
                max[column] = this.get(row, column);
              }
            }
          }
          return max;
        }
      case undefined:
        {
          let max = this.get(0, 0);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) > max) {
                max = this.get(row, column);
              }
            }
          }
          return max;
        }
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  maxIndex() {
    checkNonEmpty(this);
    let v = this.get(0, 0);
    let idx = [0, 0];
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        if (this.get(i, j) > v) {
          v = this.get(i, j);
          idx[0] = i;
          idx[1] = j;
        }
      }
    }
    return idx;
  }
  min(by) {
    if (this.isEmpty()) {
      return NaN;
    }
    switch (by) {
      case 'row':
        {
          const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) < min[row]) {
                min[row] = this.get(row, column);
              }
            }
          }
          return min;
        }
      case 'column':
        {
          const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) < min[column]) {
                min[column] = this.get(row, column);
              }
            }
          }
          return min;
        }
      case undefined:
        {
          let min = this.get(0, 0);
          for (let row = 0; row < this.rows; row++) {
            for (let column = 0; column < this.columns; column++) {
              if (this.get(row, column) < min) {
                min = this.get(row, column);
              }
            }
          }
          return min;
        }
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  minIndex() {
    checkNonEmpty(this);
    let v = this.get(0, 0);
    let idx = [0, 0];
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        if (this.get(i, j) < v) {
          v = this.get(i, j);
          idx[0] = i;
          idx[1] = j;
        }
      }
    }
    return idx;
  }
  maxRow(row) {
    checkRowIndex(this, row);
    if (this.isEmpty()) {
      return NaN;
    }
    let v = this.get(row, 0);
    for (let i = 1; i < this.columns; i++) {
      if (this.get(row, i) > v) {
        v = this.get(row, i);
      }
    }
    return v;
  }
  maxRowIndex(row) {
    checkRowIndex(this, row);
    checkNonEmpty(this);
    let v = this.get(row, 0);
    let idx = [row, 0];
    for (let i = 1; i < this.columns; i++) {
      if (this.get(row, i) > v) {
        v = this.get(row, i);
        idx[1] = i;
      }
    }
    return idx;
  }
  minRow(row) {
    checkRowIndex(this, row);
    if (this.isEmpty()) {
      return NaN;
    }
    let v = this.get(row, 0);
    for (let i = 1; i < this.columns; i++) {
      if (this.get(row, i) < v) {
        v = this.get(row, i);
      }
    }
    return v;
  }
  minRowIndex(row) {
    checkRowIndex(this, row);
    checkNonEmpty(this);
    let v = this.get(row, 0);
    let idx = [row, 0];
    for (let i = 1; i < this.columns; i++) {
      if (this.get(row, i) < v) {
        v = this.get(row, i);
        idx[1] = i;
      }
    }
    return idx;
  }
  maxColumn(column) {
    checkColumnIndex(this, column);
    if (this.isEmpty()) {
      return NaN;
    }
    let v = this.get(0, column);
    for (let i = 1; i < this.rows; i++) {
      if (this.get(i, column) > v) {
        v = this.get(i, column);
      }
    }
    return v;
  }
  maxColumnIndex(column) {
    checkColumnIndex(this, column);
    checkNonEmpty(this);
    let v = this.get(0, column);
    let idx = [0, column];
    for (let i = 1; i < this.rows; i++) {
      if (this.get(i, column) > v) {
        v = this.get(i, column);
        idx[0] = i;
      }
    }
    return idx;
  }
  minColumn(column) {
    checkColumnIndex(this, column);
    if (this.isEmpty()) {
      return NaN;
    }
    let v = this.get(0, column);
    for (let i = 1; i < this.rows; i++) {
      if (this.get(i, column) < v) {
        v = this.get(i, column);
      }
    }
    return v;
  }
  minColumnIndex(column) {
    checkColumnIndex(this, column);
    checkNonEmpty(this);
    let v = this.get(0, column);
    let idx = [0, column];
    for (let i = 1; i < this.rows; i++) {
      if (this.get(i, column) < v) {
        v = this.get(i, column);
        idx[0] = i;
      }
    }
    return idx;
  }
  diag() {
    let min = Math.min(this.rows, this.columns);
    let diag = [];
    for (let i = 0; i < min; i++) {
      diag.push(this.get(i, i));
    }
    return diag;
  }
  norm(type = 'frobenius') {
    switch (type) {
      case 'max':
        return this.max();
      case 'frobenius':
        return Math.sqrt(this.dot(this));
      default:
        throw new RangeError(`unknown norm type: ${type}`);
    }
  }
  cumulativeSum() {
    let sum = 0;
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        sum += this.get(i, j);
        this.set(i, j, sum);
      }
    }
    return this;
  }
  dot(vector2) {
    if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
    let vector1 = this.to1DArray();
    if (vector1.length !== vector2.length) {
      throw new RangeError('vectors do not have the same size');
    }
    let dot = 0;
    for (let i = 0; i < vector1.length; i++) {
      dot += vector1[i] * vector2[i];
    }
    return dot;
  }
  mmul(other) {
    other = Matrix$1.checkMatrix(other);
    let m = this.rows;
    let n = this.columns;
    let p = other.columns;
    let result = new Matrix$1(m, p);
    let Bcolj = new Float64Array(n);
    for (let j = 0; j < p; j++) {
      for (let k = 0; k < n; k++) {
        Bcolj[k] = other.get(k, j);
      }
      for (let i = 0; i < m; i++) {
        let s = 0;
        for (let k = 0; k < n; k++) {
          s += this.get(i, k) * Bcolj[k];
        }
        result.set(i, j, s);
      }
    }
    return result;
  }
  mpow(scalar) {
    if (!this.isSquare()) {
      throw new RangeError('Matrix must be square');
    }
    if (!Number.isInteger(scalar) || scalar < 0) {
      throw new RangeError('Exponent must be a non-negative integer');
    }
    // Russian Peasant exponentiation, i.e. exponentiation by squaring
    let result = Matrix$1.eye(this.rows);
    let bb = this;
    // Note: Don't bit shift. In JS, that would truncate at 32 bits
    for (let e = scalar; e >= 1; e /= 2) {
      if ((e & 1) !== 0) {
        result = result.mmul(bb);
      }
      bb = bb.mmul(bb);
    }
    return result;
  }
  strassen2x2(other) {
    other = Matrix$1.checkMatrix(other);
    let result = new Matrix$1(2, 2);
    const a11 = this.get(0, 0);
    const b11 = other.get(0, 0);
    const a12 = this.get(0, 1);
    const b12 = other.get(0, 1);
    const a21 = this.get(1, 0);
    const b21 = other.get(1, 0);
    const a22 = this.get(1, 1);
    const b22 = other.get(1, 1);

    // Compute intermediate values.
    const m1 = (a11 + a22) * (b11 + b22);
    const m2 = (a21 + a22) * b11;
    const m3 = a11 * (b12 - b22);
    const m4 = a22 * (b21 - b11);
    const m5 = (a11 + a12) * b22;
    const m6 = (a21 - a11) * (b11 + b12);
    const m7 = (a12 - a22) * (b21 + b22);

    // Combine intermediate values into the output.
    const c00 = m1 + m4 - m5 + m7;
    const c01 = m3 + m5;
    const c10 = m2 + m4;
    const c11 = m1 - m2 + m3 + m6;
    result.set(0, 0, c00);
    result.set(0, 1, c01);
    result.set(1, 0, c10);
    result.set(1, 1, c11);
    return result;
  }
  strassen3x3(other) {
    other = Matrix$1.checkMatrix(other);
    let result = new Matrix$1(3, 3);
    const a00 = this.get(0, 0);
    const a01 = this.get(0, 1);
    const a02 = this.get(0, 2);
    const a10 = this.get(1, 0);
    const a11 = this.get(1, 1);
    const a12 = this.get(1, 2);
    const a20 = this.get(2, 0);
    const a21 = this.get(2, 1);
    const a22 = this.get(2, 2);
    const b00 = other.get(0, 0);
    const b01 = other.get(0, 1);
    const b02 = other.get(0, 2);
    const b10 = other.get(1, 0);
    const b11 = other.get(1, 1);
    const b12 = other.get(1, 2);
    const b20 = other.get(2, 0);
    const b21 = other.get(2, 1);
    const b22 = other.get(2, 2);
    const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
    const m2 = (a00 - a10) * (-b01 + b11);
    const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
    const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
    const m5 = (a10 + a11) * (-b00 + b01);
    const m6 = a00 * b00;
    const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
    const m8 = (-a00 + a20) * (b02 - b12);
    const m9 = (a20 + a21) * (-b00 + b02);
    const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
    const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
    const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
    const m13 = (a02 - a22) * (b11 - b21);
    const m14 = a02 * b20;
    const m15 = (a21 + a22) * (-b20 + b21);
    const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
    const m17 = (a02 - a12) * (b12 - b22);
    const m18 = (a11 + a12) * (-b20 + b22);
    const m19 = a01 * b10;
    const m20 = a12 * b21;
    const m21 = a10 * b02;
    const m22 = a20 * b01;
    const m23 = a22 * b22;
    const c00 = m6 + m14 + m19;
    const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
    const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
    const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
    const c11 = m2 + m4 + m5 + m6 + m20;
    const c12 = m14 + m16 + m17 + m18 + m21;
    const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
    const c21 = m12 + m13 + m14 + m15 + m22;
    const c22 = m6 + m7 + m8 + m9 + m23;
    result.set(0, 0, c00);
    result.set(0, 1, c01);
    result.set(0, 2, c02);
    result.set(1, 0, c10);
    result.set(1, 1, c11);
    result.set(1, 2, c12);
    result.set(2, 0, c20);
    result.set(2, 1, c21);
    result.set(2, 2, c22);
    return result;
  }
  mmulStrassen(y) {
    y = Matrix$1.checkMatrix(y);
    let x = this.clone();
    let r1 = x.rows;
    let c1 = x.columns;
    let r2 = y.rows;
    let c2 = y.columns;
    if (c1 !== r2) {
      // eslint-disable-next-line no-console
      console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
    }

    // Put a matrix into the top left of a matrix of zeros.
    // `rows` and `cols` are the dimensions of the output matrix.
    function embed(mat, rows, cols) {
      let r = mat.rows;
      let c = mat.columns;
      if (r === rows && c === cols) {
        return mat;
      } else {
        let resultat = AbstractMatrix.zeros(rows, cols);
        resultat = resultat.setSubMatrix(mat, 0, 0);
        return resultat;
      }
    }

    // Make sure both matrices are the same size.
    // This is exclusively for simplicity:
    // this algorithm can be implemented with matrices of different sizes.

    let r = Math.max(r1, r2);
    let c = Math.max(c1, c2);
    x = embed(x, r, c);
    y = embed(y, r, c);

    // Our recursive multiplication function.
    function blockMult(a, b, rows, cols) {
      // For small matrices, resort to naive multiplication.
      if (rows <= 512 || cols <= 512) {
        return a.mmul(b); // a is equivalent to this
      }

      // Apply dynamic padding.
      if (rows % 2 === 1 && cols % 2 === 1) {
        a = embed(a, rows + 1, cols + 1);
        b = embed(b, rows + 1, cols + 1);
      } else if (rows % 2 === 1) {
        a = embed(a, rows + 1, cols);
        b = embed(b, rows + 1, cols);
      } else if (cols % 2 === 1) {
        a = embed(a, rows, cols + 1);
        b = embed(b, rows, cols + 1);
      }
      let halfRows = parseInt(a.rows / 2, 10);
      let halfCols = parseInt(a.columns / 2, 10);
      // Subdivide input matrices.
      let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
      let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
      let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
      let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
      let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
      let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
      let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
      let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

      // Compute intermediate values.
      let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
      let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
      let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
      let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
      let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
      let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
      let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

      // Combine intermediate values into the output.
      let c11 = AbstractMatrix.add(m1, m4);
      c11.sub(m5);
      c11.add(m7);
      let c12 = AbstractMatrix.add(m3, m5);
      let c21 = AbstractMatrix.add(m2, m4);
      let c22 = AbstractMatrix.sub(m1, m2);
      c22.add(m3);
      c22.add(m6);

      // Crop output to the desired size (undo dynamic padding).
      let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
      result = result.setSubMatrix(c11, 0, 0);
      result = result.setSubMatrix(c12, c11.rows, 0);
      result = result.setSubMatrix(c21, 0, c11.columns);
      result = result.setSubMatrix(c22, c11.rows, c11.columns);
      return result.subMatrix(0, rows - 1, 0, cols - 1);
    }
    return blockMult(x, y, r, c);
  }
  scaleRows(options = {}) {
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      min = 0,
      max = 1
    } = options;
    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
    if (min >= max) throw new RangeError('min must be smaller than max');
    let newMatrix = new Matrix$1(this.rows, this.columns);
    for (let i = 0; i < this.rows; i++) {
      const row = this.getRow(i);
      if (row.length > 0) {
        rescale$1(row, {
          min,
          max,
          output: row
        });
      }
      newMatrix.setRow(i, row);
    }
    return newMatrix;
  }
  scaleColumns(options = {}) {
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      min = 0,
      max = 1
    } = options;
    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
    if (min >= max) throw new RangeError('min must be smaller than max');
    let newMatrix = new Matrix$1(this.rows, this.columns);
    for (let i = 0; i < this.columns; i++) {
      const column = this.getColumn(i);
      if (column.length) {
        rescale$1(column, {
          min,
          max,
          output: column
        });
      }
      newMatrix.setColumn(i, column);
    }
    return newMatrix;
  }
  flipRows() {
    const middle = Math.ceil(this.columns / 2);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < middle; j++) {
        let first = this.get(i, j);
        let last = this.get(i, this.columns - 1 - j);
        this.set(i, j, last);
        this.set(i, this.columns - 1 - j, first);
      }
    }
    return this;
  }
  flipColumns() {
    const middle = Math.ceil(this.rows / 2);
    for (let j = 0; j < this.columns; j++) {
      for (let i = 0; i < middle; i++) {
        let first = this.get(i, j);
        let last = this.get(this.rows - 1 - i, j);
        this.set(i, j, last);
        this.set(this.rows - 1 - i, j, first);
      }
    }
    return this;
  }
  kroneckerProduct(other) {
    other = Matrix$1.checkMatrix(other);
    let m = this.rows;
    let n = this.columns;
    let p = other.rows;
    let q = other.columns;
    let result = new Matrix$1(m * p, n * q);
    for (let i = 0; i < m; i++) {
      for (let j = 0; j < n; j++) {
        for (let k = 0; k < p; k++) {
          for (let l = 0; l < q; l++) {
            result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
          }
        }
      }
    }
    return result;
  }
  kroneckerSum(other) {
    other = Matrix$1.checkMatrix(other);
    if (!this.isSquare() || !other.isSquare()) {
      throw new Error('Kronecker Sum needs two Square Matrices');
    }
    let m = this.rows;
    let n = other.rows;
    let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
    let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
    return AxI.add(IxB);
  }
  transpose() {
    let result = new Matrix$1(this.columns, this.rows);
    for (let i = 0; i < this.rows; i++) {
      for (let j = 0; j < this.columns; j++) {
        result.set(j, i, this.get(i, j));
      }
    }
    return result;
  }
  sortRows(compareFunction = compareNumbers) {
    for (let i = 0; i < this.rows; i++) {
      this.setRow(i, this.getRow(i).sort(compareFunction));
    }
    return this;
  }
  sortColumns(compareFunction = compareNumbers) {
    for (let i = 0; i < this.columns; i++) {
      this.setColumn(i, this.getColumn(i).sort(compareFunction));
    }
    return this;
  }
  subMatrix(startRow, endRow, startColumn, endColumn) {
    checkRange(this, startRow, endRow, startColumn, endColumn);
    let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
    for (let i = startRow; i <= endRow; i++) {
      for (let j = startColumn; j <= endColumn; j++) {
        newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
      }
    }
    return newMatrix;
  }
  subMatrixRow(indices, startColumn, endColumn) {
    if (startColumn === undefined) startColumn = 0;
    if (endColumn === undefined) endColumn = this.columns - 1;
    if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
      throw new RangeError('Argument out of range');
    }
    let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
    for (let i = 0; i < indices.length; i++) {
      for (let j = startColumn; j <= endColumn; j++) {
        if (indices[i] < 0 || indices[i] >= this.rows) {
          throw new RangeError(`Row index out of range: ${indices[i]}`);
        }
        newMatrix.set(i, j - startColumn, this.get(indices[i], j));
      }
    }
    return newMatrix;
  }
  subMatrixColumn(indices, startRow, endRow) {
    if (startRow === undefined) startRow = 0;
    if (endRow === undefined) endRow = this.rows - 1;
    if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
      throw new RangeError('Argument out of range');
    }
    let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
    for (let i = 0; i < indices.length; i++) {
      for (let j = startRow; j <= endRow; j++) {
        if (indices[i] < 0 || indices[i] >= this.columns) {
          throw new RangeError(`Column index out of range: ${indices[i]}`);
        }
        newMatrix.set(j - startRow, i, this.get(j, indices[i]));
      }
    }
    return newMatrix;
  }
  setSubMatrix(matrix, startRow, startColumn) {
    matrix = Matrix$1.checkMatrix(matrix);
    if (matrix.isEmpty()) {
      return this;
    }
    let endRow = startRow + matrix.rows - 1;
    let endColumn = startColumn + matrix.columns - 1;
    checkRange(this, startRow, endRow, startColumn, endColumn);
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        this.set(startRow + i, startColumn + j, matrix.get(i, j));
      }
    }
    return this;
  }
  selection(rowIndices, columnIndices) {
    checkRowIndices(this, rowIndices);
    checkColumnIndices(this, columnIndices);
    let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
    for (let i = 0; i < rowIndices.length; i++) {
      let rowIndex = rowIndices[i];
      for (let j = 0; j < columnIndices.length; j++) {
        let columnIndex = columnIndices[j];
        newMatrix.set(i, j, this.get(rowIndex, columnIndex));
      }
    }
    return newMatrix;
  }
  trace() {
    let min = Math.min(this.rows, this.columns);
    let trace = 0;
    for (let i = 0; i < min; i++) {
      trace += this.get(i, i);
    }
    return trace;
  }
  clone() {
    return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
  }

  /**
   * @template {AbstractMatrix} M
   * @param {AbstractMatrix} from
   * @param {M} to
   * @return {M}
   */
  static copy(from, to) {
    for (const [row, column, value] of from.entries()) {
      to.set(row, column, value);
    }
    return to;
  }
  sum(by) {
    switch (by) {
      case 'row':
        return sumByRow(this);
      case 'column':
        return sumByColumn(this);
      case undefined:
        return sumAll(this);
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  product(by) {
    switch (by) {
      case 'row':
        return productByRow(this);
      case 'column':
        return productByColumn(this);
      case undefined:
        return productAll(this);
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  mean(by) {
    const sum = this.sum(by);
    switch (by) {
      case 'row':
        {
          for (let i = 0; i < this.rows; i++) {
            sum[i] /= this.columns;
          }
          return sum;
        }
      case 'column':
        {
          for (let i = 0; i < this.columns; i++) {
            sum[i] /= this.rows;
          }
          return sum;
        }
      case undefined:
        return sum / this.size;
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  variance(by, options = {}) {
    if (typeof by === 'object') {
      options = by;
      by = undefined;
    }
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      unbiased = true,
      mean = this.mean(by)
    } = options;
    if (typeof unbiased !== 'boolean') {
      throw new TypeError('unbiased must be a boolean');
    }
    switch (by) {
      case 'row':
        {
          if (!isAnyArray.isAnyArray(mean)) {
            throw new TypeError('mean must be an array');
          }
          return varianceByRow(this, unbiased, mean);
        }
      case 'column':
        {
          if (!isAnyArray.isAnyArray(mean)) {
            throw new TypeError('mean must be an array');
          }
          return varianceByColumn(this, unbiased, mean);
        }
      case undefined:
        {
          if (typeof mean !== 'number') {
            throw new TypeError('mean must be a number');
          }
          return varianceAll(this, unbiased, mean);
        }
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  standardDeviation(by, options) {
    if (typeof by === 'object') {
      options = by;
      by = undefined;
    }
    const variance = this.variance(by, options);
    if (by === undefined) {
      return Math.sqrt(variance);
    } else {
      for (let i = 0; i < variance.length; i++) {
        variance[i] = Math.sqrt(variance[i]);
      }
      return variance;
    }
  }
  center(by, options = {}) {
    if (typeof by === 'object') {
      options = by;
      by = undefined;
    }
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    const {
      center = this.mean(by)
    } = options;
    switch (by) {
      case 'row':
        {
          if (!isAnyArray.isAnyArray(center)) {
            throw new TypeError('center must be an array');
          }
          centerByRow(this, center);
          return this;
        }
      case 'column':
        {
          if (!isAnyArray.isAnyArray(center)) {
            throw new TypeError('center must be an array');
          }
          centerByColumn(this, center);
          return this;
        }
      case undefined:
        {
          if (typeof center !== 'number') {
            throw new TypeError('center must be a number');
          }
          centerAll(this, center);
          return this;
        }
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  scale(by, options = {}) {
    if (typeof by === 'object') {
      options = by;
      by = undefined;
    }
    if (typeof options !== 'object') {
      throw new TypeError('options must be an object');
    }
    let scale = options.scale;
    switch (by) {
      case 'row':
        {
          if (scale === undefined) {
            scale = getScaleByRow(this);
          } else if (!isAnyArray.isAnyArray(scale)) {
            throw new TypeError('scale must be an array');
          }
          scaleByRow(this, scale);
          return this;
        }
      case 'column':
        {
          if (scale === undefined) {
            scale = getScaleByColumn(this);
          } else if (!isAnyArray.isAnyArray(scale)) {
            throw new TypeError('scale must be an array');
          }
          scaleByColumn(this, scale);
          return this;
        }
      case undefined:
        {
          if (scale === undefined) {
            scale = getScaleAll(this);
          } else if (typeof scale !== 'number') {
            throw new TypeError('scale must be a number');
          }
          scaleAll(this, scale);
          return this;
        }
      default:
        throw new Error(`invalid option: ${by}`);
    }
  }
  toString(options) {
    return inspectMatrixWithOptions(this, options);
  }
  [Symbol.iterator]() {
    return this.entries();
  }

  /**
   * iterator from left to right, from top to bottom
   * yield [row, column, value]
   * @returns {Generator<[number, number, number], void, void>}
   */
  *entries() {
    for (let row = 0; row < this.rows; row++) {
      for (let col = 0; col < this.columns; col++) {
        yield [row, col, this.get(row, col)];
      }
    }
  }

  /**
   * iterator from left to right, from top to bottom
   * yield value
   * @returns {Generator<number, void, void>}
   */
  *values() {
    for (let row = 0; row < this.rows; row++) {
      for (let col = 0; col < this.columns; col++) {
        yield this.get(row, col);
      }
    }
  }
}
AbstractMatrix.prototype.klass = 'Matrix';
if (typeof Symbol !== 'undefined') {
  AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
}
function compareNumbers(a, b) {
  return a - b;
}
function isArrayOfNumbers(array) {
  return array.every(element => {
    return typeof element === 'number';
  });
}

// Synonyms
AbstractMatrix.random = AbstractMatrix.rand;
AbstractMatrix.randomInt = AbstractMatrix.randInt;
AbstractMatrix.diagonal = AbstractMatrix.diag;
AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
AbstractMatrix.identity = AbstractMatrix.eye;
AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
let Matrix$1 = class Matrix extends AbstractMatrix {
  /**
   * @type {Float64Array[]}
   */
  data;

  /**
   * Init an empty matrix
   * @param {number} nRows
   * @param {number} nColumns
   */
  #initData(nRows, nColumns) {
    this.data = [];
    if (Number.isInteger(nColumns) && nColumns >= 0) {
      for (let i = 0; i < nRows; i++) {
        this.data.push(new Float64Array(nColumns));
      }
    } else {
      throw new TypeError('nColumns must be a positive integer');
    }
    this.rows = nRows;
    this.columns = nColumns;
  }
  constructor(nRows, nColumns) {
    super();
    if (Matrix.isMatrix(nRows)) {
      this.#initData(nRows.rows, nRows.columns);
      Matrix.copy(nRows, this);
    } else if (Number.isInteger(nRows) && nRows >= 0) {
      this.#initData(nRows, nColumns);
    } else if (isAnyArray.isAnyArray(nRows)) {
      // Copy the values from the 2D array
      const arrayData = nRows;
      nRows = arrayData.length;
      nColumns = nRows ? arrayData[0].length : 0;
      if (typeof nColumns !== 'number') {
        throw new TypeError('Data must be a 2D array with at least one element');
      }
      this.data = [];
      for (let i = 0; i < nRows; i++) {
        if (arrayData[i].length !== nColumns) {
          throw new RangeError('Inconsistent array dimensions');
        }
        if (!isArrayOfNumbers(arrayData[i])) {
          throw new TypeError('Input data contains non-numeric values');
        }
        this.data.push(Float64Array.from(arrayData[i]));
      }
      this.rows = nRows;
      this.columns = nColumns;
    } else {
      throw new TypeError('First argument must be a positive number or an array');
    }
  }
  set(rowIndex, columnIndex, value) {
    this.data[rowIndex][columnIndex] = value;
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.data[rowIndex][columnIndex];
  }
  removeRow(index) {
    checkRowIndex(this, index);
    this.data.splice(index, 1);
    this.rows -= 1;
    return this;
  }
  addRow(index, array) {
    if (array === undefined) {
      array = index;
      index = this.rows;
    }
    checkRowIndex(this, index, true);
    array = Float64Array.from(checkRowVector(this, array));
    this.data.splice(index, 0, array);
    this.rows += 1;
    return this;
  }
  removeColumn(index) {
    checkColumnIndex(this, index);
    for (let i = 0; i < this.rows; i++) {
      const newRow = new Float64Array(this.columns - 1);
      for (let j = 0; j < index; j++) {
        newRow[j] = this.data[i][j];
      }
      for (let j = index + 1; j < this.columns; j++) {
        newRow[j - 1] = this.data[i][j];
      }
      this.data[i] = newRow;
    }
    this.columns -= 1;
    return this;
  }
  addColumn(index, array) {
    if (typeof array === 'undefined') {
      array = index;
      index = this.columns;
    }
    checkColumnIndex(this, index, true);
    array = checkColumnVector(this, array);
    for (let i = 0; i < this.rows; i++) {
      const newRow = new Float64Array(this.columns + 1);
      let j = 0;
      for (; j < index; j++) {
        newRow[j] = this.data[i][j];
      }
      newRow[j++] = array[i];
      for (; j < this.columns + 1; j++) {
        newRow[j] = this.data[i][j - 1];
      }
      this.data[i] = newRow;
    }
    this.columns += 1;
    return this;
  }
};
installMathOperations(AbstractMatrix, Matrix$1);

/**
 * @typedef {0 | 1 | number | boolean} Mask
 */

class SymmetricMatrix extends AbstractMatrix {
  /** @type {Matrix} */
  #matrix;
  get size() {
    return this.#matrix.size;
  }
  get rows() {
    return this.#matrix.rows;
  }
  get columns() {
    return this.#matrix.columns;
  }
  get diagonalSize() {
    return this.rows;
  }

  /**
   * not the same as matrix.isSymmetric()
   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
   *
   * @param value
   * @returns {boolean}
   */
  static isSymmetricMatrix(value) {
    return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
  }

  /**
   * @param diagonalSize
   * @return {SymmetricMatrix}
   */
  static zeros(diagonalSize) {
    return new this(diagonalSize);
  }

  /**
   * @param diagonalSize
   * @return {SymmetricMatrix}
   */
  static ones(diagonalSize) {
    return new this(diagonalSize).fill(1);
  }

  /**
   * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
   * @return {this}
   */
  constructor(diagonalSize) {
    super();
    if (Matrix$1.isMatrix(diagonalSize)) {
      if (!diagonalSize.isSymmetric()) {
        throw new TypeError('not symmetric data');
      }
      this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
    } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
      this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
    } else {
      this.#matrix = new Matrix$1(diagonalSize);
      if (!this.isSymmetric()) {
        throw new TypeError('not symmetric data');
      }
    }
  }
  clone() {
    const matrix = new SymmetricMatrix(this.diagonalSize);
    for (const [row, col, value] of this.upperRightEntries()) {
      matrix.set(row, col, value);
    }
    return matrix;
  }
  toMatrix() {
    return new Matrix$1(this);
  }
  get(rowIndex, columnIndex) {
    return this.#matrix.get(rowIndex, columnIndex);
  }
  set(rowIndex, columnIndex, value) {
    // symmetric set
    this.#matrix.set(rowIndex, columnIndex, value);
    this.#matrix.set(columnIndex, rowIndex, value);
    return this;
  }
  removeCross(index) {
    // symmetric remove side
    this.#matrix.removeRow(index);
    this.#matrix.removeColumn(index);
    return this;
  }
  addCross(index, array) {
    if (array === undefined) {
      array = index;
      index = this.diagonalSize;
    }
    const row = array.slice();
    row.splice(index, 1);
    this.#matrix.addRow(index, row);
    this.#matrix.addColumn(index, array);
    return this;
  }

  /**
   * @param {Mask[]} mask
   */
  applyMask(mask) {
    if (mask.length !== this.diagonalSize) {
      throw new RangeError('Mask size do not match with matrix size');
    }

    // prepare sides to remove from matrix from mask
    /** @type {number[]} */
    const sidesToRemove = [];
    for (const [index, passthroughs] of mask.entries()) {
      if (passthroughs) continue;
      sidesToRemove.push(index);
    }
    // to remove from highest to lowest for no mutation shifting
    sidesToRemove.reverse();

    // remove sides
    for (const sideIndex of sidesToRemove) {
      this.removeCross(sideIndex);
    }
    return this;
  }

  /**
   * Compact format upper-right corner of matrix
   * iterate from left to right, from top to bottom.
   *
   * ```
   *   A B C D
   * A 1 2 3 4
   * B 2 5 6 7
   * C 3 6 8 9
   * D 4 7 9 10
   * ```
   *
   * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
   *
   * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
   *
   * @returns {number[]}
   */
  toCompact() {
    const {
      diagonalSize
    } = this;

    /** @type {number[]} */
    const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
    for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
      compact[index] = this.get(row, col);
      if (++col >= diagonalSize) col = ++row;
    }
    return compact;
  }

  /**
   * @param {number[]} compact
   * @return {SymmetricMatrix}
   */
  static fromCompact(compact) {
    const compactSize = compact.length;
    // compactSize = (sideSize * (sideSize + 1)) / 2
    // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
    // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
    const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
    if (!Number.isInteger(diagonalSize)) {
      throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
    }
    const matrix = new SymmetricMatrix(diagonalSize);
    for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
      matrix.set(col, row, compact[index]);
      if (++col >= diagonalSize) col = ++row;
    }
    return matrix;
  }

  /**
   * half iterator upper-right-corner from left to right, from top to bottom
   * yield [row, column, value]
   *
   * @returns {Generator<[number, number, number], void, void>}
   */
  *upperRightEntries() {
    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
      const value = this.get(row, col);
      yield [row, col, value];

      // at the end of row, move cursor to next row at diagonal position
      if (++col >= this.diagonalSize) col = ++row;
    }
  }

  /**
   * half iterator upper-right-corner from left to right, from top to bottom
   * yield value
   *
   * @returns {Generator<[number, number, number], void, void>}
   */
  *upperRightValues() {
    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
      const value = this.get(row, col);
      yield value;

      // at the end of row, move cursor to next row at diagonal position
      if (++col >= this.diagonalSize) col = ++row;
    }
  }
}
SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
class DistanceMatrix extends SymmetricMatrix {
  /**
   * not the same as matrix.isSymmetric()
   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
   *
   * @param value
   * @returns {boolean}
   */
  static isDistanceMatrix(value) {
    return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
  }
  constructor(sideSize) {
    super(sideSize);
    if (!this.isDistance()) {
      throw new TypeError('Provided arguments do no produce a distance matrix');
    }
  }
  set(rowIndex, columnIndex, value) {
    // distance matrix diagonal is 0
    if (rowIndex === columnIndex) value = 0;
    return super.set(rowIndex, columnIndex, value);
  }
  addCross(index, array) {
    if (array === undefined) {
      array = index;
      index = this.diagonalSize;
    }

    // ensure distance
    array = array.slice();
    array[index] = 0;
    return super.addCross(index, array);
  }
  toSymmetricMatrix() {
    return new SymmetricMatrix(this);
  }
  clone() {
    const matrix = new DistanceMatrix(this.diagonalSize);
    for (const [row, col, value] of this.upperRightEntries()) {
      if (row === col) continue;
      matrix.set(row, col, value);
    }
    return matrix;
  }

  /**
   * Compact format upper-right corner of matrix
   * no diagonal (only zeros)
   * iterable from left to right, from top to bottom.
   *
   * ```
   *   A B C D
   * A 0 1 2 3
   * B 1 0 4 5
   * C 2 4 0 6
   * D 3 5 6 0
   * ```
   *
   * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
   *
   * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
   *
   * @returns {number[]}
   */
  toCompact() {
    const {
      diagonalSize
    } = this;
    const compactLength = (diagonalSize - 1) * diagonalSize / 2;

    /** @type {number[]} */
    const compact = new Array(compactLength);
    for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
      compact[index] = this.get(row, col);
      if (++col >= diagonalSize) col = ++row + 1;
    }
    return compact;
  }

  /**
   * @param {number[]} compact
   */
  static fromCompact(compact) {
    const compactSize = compact.length;
    if (compactSize === 0) {
      return new this(0);
    }

    // compactSize in Natural integer range ]0;∞]
    // compactSize = (sideSize * (sideSize - 1)) / 2
    // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
    const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
    if (!Number.isInteger(diagonalSize)) {
      throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
    }
    const matrix = new this(diagonalSize);
    for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
      matrix.set(col, row, compact[index]);
      if (++col >= diagonalSize) col = ++row + 1;
    }
    return matrix;
  }
}
DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
class BaseView extends AbstractMatrix {
  constructor(matrix, rows, columns) {
    super();
    this.matrix = matrix;
    this.rows = rows;
    this.columns = columns;
  }
}
class MatrixColumnView extends BaseView {
  constructor(matrix, column) {
    checkColumnIndex(matrix, column);
    super(matrix, matrix.rows, 1);
    this.column = column;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(rowIndex, this.column, value);
    return this;
  }
  get(rowIndex) {
    return this.matrix.get(rowIndex, this.column);
  }
}
class MatrixColumnSelectionView extends BaseView {
  constructor(matrix, columnIndices) {
    checkColumnIndices(matrix, columnIndices);
    super(matrix, matrix.rows, columnIndices.length);
    this.columnIndices = columnIndices;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
  }
}
class MatrixFlipColumnView extends BaseView {
  constructor(matrix) {
    super(matrix, matrix.rows, matrix.columns);
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
  }
}
class MatrixFlipRowView extends BaseView {
  constructor(matrix) {
    super(matrix, matrix.rows, matrix.columns);
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
  }
}
class MatrixRowView extends BaseView {
  constructor(matrix, row) {
    checkRowIndex(matrix, row);
    super(matrix, 1, matrix.columns);
    this.row = row;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(this.row, columnIndex, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(this.row, columnIndex);
  }
}
class MatrixRowSelectionView extends BaseView {
  constructor(matrix, rowIndices) {
    checkRowIndices(matrix, rowIndices);
    super(matrix, rowIndices.length, matrix.columns);
    this.rowIndices = rowIndices;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
  }
}
class MatrixSelectionView extends BaseView {
  constructor(matrix, rowIndices, columnIndices) {
    checkRowIndices(matrix, rowIndices);
    checkColumnIndices(matrix, columnIndices);
    super(matrix, rowIndices.length, columnIndices.length);
    this.rowIndices = rowIndices;
    this.columnIndices = columnIndices;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
  }
}
class MatrixSubView extends BaseView {
  constructor(matrix, startRow, endRow, startColumn, endColumn) {
    checkRange(matrix, startRow, endRow, startColumn, endColumn);
    super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
    this.startRow = startRow;
    this.startColumn = startColumn;
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
  }
}
let MatrixTransposeView$1 = class MatrixTransposeView extends BaseView {
  constructor(matrix) {
    super(matrix, matrix.columns, matrix.rows);
  }
  set(rowIndex, columnIndex, value) {
    this.matrix.set(columnIndex, rowIndex, value);
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.matrix.get(columnIndex, rowIndex);
  }
};
class WrapperMatrix1D extends AbstractMatrix {
  constructor(data, options = {}) {
    const {
      rows = 1
    } = options;
    if (data.length % rows !== 0) {
      throw new Error('the data length is not divisible by the number of rows');
    }
    super();
    this.rows = rows;
    this.columns = data.length / rows;
    this.data = data;
  }
  set(rowIndex, columnIndex, value) {
    let index = this._calculateIndex(rowIndex, columnIndex);
    this.data[index] = value;
    return this;
  }
  get(rowIndex, columnIndex) {
    let index = this._calculateIndex(rowIndex, columnIndex);
    return this.data[index];
  }
  _calculateIndex(row, column) {
    return row * this.columns + column;
  }
}
class WrapperMatrix2D extends AbstractMatrix {
  constructor(data) {
    super();
    this.data = data;
    this.rows = data.length;
    this.columns = data[0].length;
  }
  set(rowIndex, columnIndex, value) {
    this.data[rowIndex][columnIndex] = value;
    return this;
  }
  get(rowIndex, columnIndex) {
    return this.data[rowIndex][columnIndex];
  }
}
function wrap(array, options) {
  if (isAnyArray.isAnyArray(array)) {
    if (array[0] && isAnyArray.isAnyArray(array[0])) {
      return new WrapperMatrix2D(array);
    } else {
      return new WrapperMatrix1D(array, options);
    }
  } else {
    throw new Error('the argument is not an array');
  }
}
class LuDecomposition {
  constructor(matrix) {
    matrix = WrapperMatrix2D.checkMatrix(matrix);
    let lu = matrix.clone();
    let rows = lu.rows;
    let columns = lu.columns;
    let pivotVector = new Float64Array(rows);
    let pivotSign = 1;
    let i, j, k, p, s, t, v;
    let LUcolj, kmax;
    for (i = 0; i < rows; i++) {
      pivotVector[i] = i;
    }
    LUcolj = new Float64Array(rows);
    for (j = 0; j < columns; j++) {
      for (i = 0; i < rows; i++) {
        LUcolj[i] = lu.get(i, j);
      }
      for (i = 0; i < rows; i++) {
        kmax = Math.min(i, j);
        s = 0;
        for (k = 0; k < kmax; k++) {
          s += lu.get(i, k) * LUcolj[k];
        }
        LUcolj[i] -= s;
        lu.set(i, j, LUcolj[i]);
      }
      p = j;
      for (i = j + 1; i < rows; i++) {
        if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
          p = i;
        }
      }
      if (p !== j) {
        for (k = 0; k < columns; k++) {
          t = lu.get(p, k);
          lu.set(p, k, lu.get(j, k));
          lu.set(j, k, t);
        }
        v = pivotVector[p];
        pivotVector[p] = pivotVector[j];
        pivotVector[j] = v;
        pivotSign = -pivotSign;
      }
      if (j < rows && lu.get(j, j) !== 0) {
        for (i = j + 1; i < rows; i++) {
          lu.set(i, j, lu.get(i, j) / lu.get(j, j));
        }
      }
    }
    this.LU = lu;
    this.pivotVector = pivotVector;
    this.pivotSign = pivotSign;
  }
  isSingular() {
    let data = this.LU;
    let col = data.columns;
    for (let j = 0; j < col; j++) {
      if (data.get(j, j) === 0) {
        return true;
      }
    }
    return false;
  }
  solve(value) {
    value = Matrix$1.checkMatrix(value);
    let lu = this.LU;
    let rows = lu.rows;
    if (rows !== value.rows) {
      throw new Error('Invalid matrix dimensions');
    }
    if (this.isSingular()) {
      throw new Error('LU matrix is singular');
    }
    let count = value.columns;
    let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
    let columns = lu.columns;
    let i, j, k;
    for (k = 0; k < columns; k++) {
      for (i = k + 1; i < columns; i++) {
        for (j = 0; j < count; j++) {
          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
        }
      }
    }
    for (k = columns - 1; k >= 0; k--) {
      for (j = 0; j < count; j++) {
        X.set(k, j, X.get(k, j) / lu.get(k, k));
      }
      for (i = 0; i < k; i++) {
        for (j = 0; j < count; j++) {
          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
        }
      }
    }
    return X;
  }
  get determinant() {
    let data = this.LU;
    if (!data.isSquare()) {
      throw new Error('Matrix must be square');
    }
    let determinant = this.pivotSign;
    let col = data.columns;
    for (let j = 0; j < col; j++) {
      determinant *= data.get(j, j);
    }
    return determinant;
  }
  get lowerTriangularMatrix() {
    let data = this.LU;
    let rows = data.rows;
    let columns = data.columns;
    let X = new Matrix$1(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        if (i > j) {
          X.set(i, j, data.get(i, j));
        } else if (i === j) {
          X.set(i, j, 1);
        } else {
          X.set(i, j, 0);
        }
      }
    }
    return X;
  }
  get upperTriangularMatrix() {
    let data = this.LU;
    let rows = data.rows;
    let columns = data.columns;
    let X = new Matrix$1(rows, columns);
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < columns; j++) {
        if (i <= j) {
          X.set(i, j, data.get(i, j));
        } else {
          X.set(i, j, 0);
        }
      }
    }
    return X;
  }
  get pivotPermutationVector() {
    return Array.from(this.pivotVector);
  }
}
function hypotenuse(a, b) {
  let r = 0;
  if (Math.abs(a) > Math.abs(b)) {
    r = b / a;
    return Math.abs(a) * Math.sqrt(1 + r * r);
  }
  if (b !== 0) {
    r = a / b;
    return Math.abs(b) * Math.sqrt(1 + r * r);
  }
  return 0;
}
class QrDecomposition {
  constructor(value) {
    value = WrapperMatrix2D.checkMatrix(value);
    let qr = value.clone();
    let m = value.rows;
    let n = value.columns;
    let rdiag = new Float64Array(n);
    let i, j, k, s;
    for (k = 0; k < n; k++) {
      let nrm = 0;
      for (i = k; i < m; i++) {
        nrm = hypotenuse(nrm, qr.get(i, k));
      }
      if (nrm !== 0) {
        if (qr.get(k, k) < 0) {
          nrm = -nrm;
        }
        for (i = k; i < m; i++) {
          qr.set(i, k, qr.get(i, k) / nrm);
        }
        qr.set(k, k, qr.get(k, k) + 1);
        for (j = k + 1; j < n; j++) {
          s = 0;
          for (i = k; i < m; i++) {
            s += qr.get(i, k) * qr.get(i, j);
          }
          s = -s / qr.get(k, k);
          for (i = k; i < m; i++) {
            qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
          }
        }
      }
      rdiag[k] = -nrm;
    }
    this.QR = qr;
    this.Rdiag = rdiag;
  }
  solve(value) {
    value = Matrix$1.checkMatrix(value);
    let qr = this.QR;
    let m = qr.rows;
    if (value.rows !== m) {
      throw new Error('Matrix row dimensions must agree');
    }
    if (!this.isFullRank()) {
      throw new Error('Matrix is rank deficient');
    }
    let count = value.columns;
    let X = value.clone();
    let n = qr.columns;
    let i, j, k, s;
    for (k = 0; k < n; k++) {
      for (j = 0; j < count; j++) {
        s = 0;
        for (i = k; i < m; i++) {
          s += qr.get(i, k) * X.get(i, j);
        }
        s = -s / qr.get(k, k);
        for (i = k; i < m; i++) {
          X.set(i, j, X.get(i, j) + s * qr.get(i, k));
        }
      }
    }
    for (k = n - 1; k >= 0; k--) {
      for (j = 0; j < count; j++) {
        X.set(k, j, X.get(k, j) / this.Rdiag[k]);
      }
      for (i = 0; i < k; i++) {
        for (j = 0; j < count; j++) {
          X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
        }
      }
    }
    return X.subMatrix(0, n - 1, 0, count - 1);
  }
  isFullRank() {
    let columns = this.QR.columns;
    for (let i = 0; i < columns; i++) {
      if (this.Rdiag[i] === 0) {
        return false;
      }
    }
    return true;
  }
  get upperTriangularMatrix() {
    let qr = this.QR;
    let n = qr.columns;
    let X = new Matrix$1(n, n);
    let i, j;
    for (i = 0; i < n; i++) {
      for (j = 0; j < n; j++) {
        if (i < j) {
          X.set(i, j, qr.get(i, j));
        } else if (i === j) {
          X.set(i, j, this.Rdiag[i]);
        } else {
          X.set(i, j, 0);
        }
      }
    }
    return X;
  }
  get orthogonalMatrix() {
    let qr = this.QR;
    let rows = qr.rows;
    let columns = qr.columns;
    let X = new Matrix$1(rows, columns);
    let i, j, k, s;
    for (k = columns - 1; k >= 0; k--) {
      for (i = 0; i < rows; i++) {
        X.set(i, k, 0);
      }
      X.set(k, k, 1);
      for (j = k; j < columns; j++) {
        if (qr.get(k, k) !== 0) {
          s = 0;
          for (i = k; i < rows; i++) {
            s += qr.get(i, k) * X.get(i, j);
          }
          s = -s / qr.get(k, k);
          for (i = k; i < rows; i++) {
            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
          }
        }
      }
    }
    return X;
  }
}
class SingularValueDecomposition {
  constructor(value, options = {}) {
    value = WrapperMatrix2D.checkMatrix(value);
    if (value.isEmpty()) {
      throw new Error('Matrix must be non-empty');
    }
    let m = value.rows;
    let n = value.columns;
    const {
      computeLeftSingularVectors = true,
      computeRightSingularVectors = true,
      autoTranspose = false
    } = options;
    let wantu = Boolean(computeLeftSingularVectors);
    let wantv = Boolean(computeRightSingularVectors);
    let swapped = false;
    let a;
    if (m < n) {
      if (!autoTranspose) {
        a = value.clone();
        // eslint-disable-next-line no-console
        console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
      } else {
        a = value.transpose();
        m = a.rows;
        n = a.columns;
        swapped = true;
        let aux = wantu;
        wantu = wantv;
        wantv = aux;
      }
    } else {
      a = value.clone();
    }
    let nu = Math.min(m, n);
    let ni = Math.min(m + 1, n);
    let s = new Float64Array(ni);
    let U = new Matrix$1(m, nu);
    let V = new Matrix$1(n, n);
    let e = new Float64Array(n);
    let work = new Float64Array(m);
    let si = new Float64Array(ni);
    for (let i = 0; i < ni; i++) si[i] = i;
    let nct = Math.min(m - 1, n);
    let nrt = Math.max(0, Math.min(n - 2, m));
    let mrc = Math.max(nct, nrt);
    for (let k = 0; k < mrc; k++) {
      if (k < nct) {
        s[k] = 0;
        for (let i = k; i < m; i++) {
          s[k] = hypotenuse(s[k], a.get(i, k));
        }
        if (s[k] !== 0) {
          if (a.get(k, k) < 0) {
            s[k] = -s[k];
          }
          for (let i = k; i < m; i++) {
            a.set(i, k, a.get(i, k) / s[k]);
          }
          a.set(k, k, a.get(k, k) + 1);
        }
        s[k] = -s[k];
      }
      for (let j = k + 1; j < n; j++) {
        if (k < nct && s[k] !== 0) {
          let t = 0;
          for (let i = k; i < m; i++) {
            t += a.get(i, k) * a.get(i, j);
          }
          t = -t / a.get(k, k);
          for (let i = k; i < m; i++) {
            a.set(i, j, a.get(i, j) + t * a.get(i, k));
          }
        }
        e[j] = a.get(k, j);
      }
      if (wantu && k < nct) {
        for (let i = k; i < m; i++) {
          U.set(i, k, a.get(i, k));
        }
      }
      if (k < nrt) {
        e[k] = 0;
        for (let i = k + 1; i < n; i++) {
          e[k] = hypotenuse(e[k], e[i]);
        }
        if (e[k] !== 0) {
          if (e[k + 1] < 0) {
            e[k] = 0 - e[k];
          }
          for (let i = k + 1; i < n; i++) {
            e[i] /= e[k];
          }
          e[k + 1] += 1;
        }
        e[k] = -e[k];
        if (k + 1 < m && e[k] !== 0) {
          for (let i = k + 1; i < m; i++) {
            work[i] = 0;
          }
          for (let i = k + 1; i < m; i++) {
            for (let j = k + 1; j < n; j++) {
              work[i] += e[j] * a.get(i, j);
            }
          }
          for (let j = k + 1; j < n; j++) {
            let t = -e[j] / e[k + 1];
            for (let i = k + 1; i < m; i++) {
              a.set(i, j, a.get(i, j) + t * work[i]);
            }
          }
        }
        if (wantv) {
          for (let i = k + 1; i < n; i++) {
            V.set(i, k, e[i]);
          }
        }
      }
    }
    let p = Math.min(n, m + 1);
    if (nct < n) {
      s[nct] = a.get(nct, nct);
    }
    if (m < p) {
      s[p - 1] = 0;
    }
    if (nrt + 1 < p) {
      e[nrt] = a.get(nrt, p - 1);
    }
    e[p - 1] = 0;
    if (wantu) {
      for (let j = nct; j < nu; j++) {
        for (let i = 0; i < m; i++) {
          U.set(i, j, 0);
        }
        U.set(j, j, 1);
      }
      for (let k = nct - 1; k >= 0; k--) {
        if (s[k] !== 0) {
          for (let j = k + 1; j < nu; j++) {
            let t = 0;
            for (let i = k; i < m; i++) {
              t += U.get(i, k) * U.get(i, j);
            }
            t = -t / U.get(k, k);
            for (let i = k; i < m; i++) {
              U.set(i, j, U.get(i, j) + t * U.get(i, k));
            }
          }
          for (let i = k; i < m; i++) {
            U.set(i, k, -U.get(i, k));
          }
          U.set(k, k, 1 + U.get(k, k));
          for (let i = 0; i < k - 1; i++) {
            U.set(i, k, 0);
          }
        } else {
          for (let i = 0; i < m; i++) {
            U.set(i, k, 0);
          }
          U.set(k, k, 1);
        }
      }
    }
    if (wantv) {
      for (let k = n - 1; k >= 0; k--) {
        if (k < nrt && e[k] !== 0) {
          for (let j = k + 1; j < n; j++) {
            let t = 0;
            for (let i = k + 1; i < n; i++) {
              t += V.get(i, k) * V.get(i, j);
            }
            t = -t / V.get(k + 1, k);
            for (let i = k + 1; i < n; i++) {
              V.set(i, j, V.get(i, j) + t * V.get(i, k));
            }
          }
        }
        for (let i = 0; i < n; i++) {
          V.set(i, k, 0);
        }
        V.set(k, k, 1);
      }
    }
    let pp = p - 1;
    let eps = Number.EPSILON;
    while (p > 0) {
      let k, kase;
      for (k = p - 2; k >= -1; k--) {
        if (k === -1) {
          break;
        }
        const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
        if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
          e[k] = 0;
          break;
        }
      }
      if (k === p - 2) {
        kase = 4;
      } else {
        let ks;
        for (ks = p - 1; ks >= k; ks--) {
          if (ks === k) {
            break;
          }
          let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
          if (Math.abs(s[ks]) <= eps * t) {
            s[ks] = 0;
            break;
          }
        }
        if (ks === k) {
          kase = 3;
        } else if (ks === p - 1) {
          kase = 1;
        } else {
          kase = 2;
          k = ks;
        }
      }
      k++;
      switch (kase) {
        case 1:
          {
            let f = e[p - 2];
            e[p - 2] = 0;
            for (let j = p - 2; j >= k; j--) {
              let t = hypotenuse(s[j], f);
              let cs = s[j] / t;
              let sn = f / t;
              s[j] = t;
              if (j !== k) {
                f = -sn * e[j - 1];
                e[j - 1] = cs * e[j - 1];
              }
              if (wantv) {
                for (let i = 0; i < n; i++) {
                  t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                  V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                  V.set(i, j, t);
                }
              }
            }
            break;
          }
        case 2:
          {
            let f = e[k - 1];
            e[k - 1] = 0;
            for (let j = k; j < p; j++) {
              let t = hypotenuse(s[j], f);
              let cs = s[j] / t;
              let sn = f / t;
              s[j] = t;
              f = -sn * e[j];
              e[j] = cs * e[j];
              if (wantu) {
                for (let i = 0; i < m; i++) {
                  t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                  U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                  U.set(i, j, t);
                }
              }
            }
            break;
          }
        case 3:
          {
            const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
            const sp = s[p - 1] / scale;
            const spm1 = s[p - 2] / scale;
            const epm1 = e[p - 2] / scale;
            const sk = s[k] / scale;
            const ek = e[k] / scale;
            const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
            const c = sp * epm1 * (sp * epm1);
            let shift = 0;
            if (b !== 0 || c !== 0) {
              if (b < 0) {
                shift = 0 - Math.sqrt(b * b + c);
              } else {
                shift = Math.sqrt(b * b + c);
              }
              shift = c / (b + shift);
            }
            let f = (sk + sp) * (sk - sp) + shift;
            let g = sk * ek;
            for (let j = k; j < p - 1; j++) {
              let t = hypotenuse(f, g);
              if (t === 0) t = Number.MIN_VALUE;
              let cs = f / t;
              let sn = g / t;
              if (j !== k) {
                e[j - 1] = t;
              }
              f = cs * s[j] + sn * e[j];
              e[j] = cs * e[j] - sn * s[j];
              g = sn * s[j + 1];
              s[j + 1] = cs * s[j + 1];
              if (wantv) {
                for (let i = 0; i < n; i++) {
                  t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                  V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                  V.set(i, j, t);
                }
              }
              t = hypotenuse(f, g);
              if (t === 0) t = Number.MIN_VALUE;
              cs = f / t;
              sn = g / t;
              s[j] = t;
              f = cs * e[j] + sn * s[j + 1];
              s[j + 1] = -sn * e[j] + cs * s[j + 1];
              g = sn * e[j + 1];
              e[j + 1] = cs * e[j + 1];
              if (wantu && j < m - 1) {
                for (let i = 0; i < m; i++) {
                  t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                  U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                  U.set(i, j, t);
                }
              }
            }
            e[p - 2] = f;
            break;
          }
        case 4:
          {
            if (s[k] <= 0) {
              s[k] = s[k] < 0 ? -s[k] : 0;
              if (wantv) {
                for (let i = 0; i <= pp; i++) {
                  V.set(i, k, -V.get(i, k));
                }
              }
            }
            while (k < pp) {
              if (s[k] >= s[k + 1]) {
                break;
              }
              let t = s[k];
              s[k] = s[k + 1];
              s[k + 1] = t;
              if (wantv && k < n - 1) {
                for (let i = 0; i < n; i++) {
                  t = V.get(i, k + 1);
                  V.set(i, k + 1, V.get(i, k));
                  V.set(i, k, t);
                }
              }
              if (wantu && k < m - 1) {
                for (let i = 0; i < m; i++) {
                  t = U.get(i, k + 1);
                  U.set(i, k + 1, U.get(i, k));
                  U.set(i, k, t);
                }
              }
              k++;
            }
            p--;
            break;
          }
        // no default
      }
    }
    if (swapped) {
      let tmp = V;
      V = U;
      U = tmp;
    }
    this.m = m;
    this.n = n;
    this.s = s;
    this.U = U;
    this.V = V;
  }
  solve(value) {
    let Y = value;
    let e = this.threshold;
    let scols = this.s.length;
    let Ls = Matrix$1.zeros(scols, scols);
    for (let i = 0; i < scols; i++) {
      if (Math.abs(this.s[i]) <= e) {
        Ls.set(i, i, 0);
      } else {
        Ls.set(i, i, 1 / this.s[i]);
      }
    }
    let U = this.U;
    let V = this.rightSingularVectors;
    let VL = V.mmul(Ls);
    let vrows = V.rows;
    let urows = U.rows;
    let VLU = Matrix$1.zeros(vrows, urows);
    for (let i = 0; i < vrows; i++) {
      for (let j = 0; j < urows; j++) {
        let sum = 0;
        for (let k = 0; k < scols; k++) {
          sum += VL.get(i, k) * U.get(j, k);
        }
        VLU.set(i, j, sum);
      }
    }
    return VLU.mmul(Y);
  }
  solveForDiagonal(value) {
    return this.solve(Matrix$1.diag(value));
  }
  inverse() {
    let V = this.V;
    let e = this.threshold;
    let vrows = V.rows;
    let vcols = V.columns;
    let X = new Matrix$1(vrows, this.s.length);
    for (let i = 0; i < vrows; i++) {
      for (let j = 0; j < vcols; j++) {
        if (Math.abs(this.s[j]) > e) {
          X.set(i, j, V.get(i, j) / this.s[j]);
        }
      }
    }
    let U = this.U;
    let urows = U.rows;
    let ucols = U.columns;
    let Y = new Matrix$1(vrows, urows);
    for (let i = 0; i < vrows; i++) {
      for (let j = 0; j < urows; j++) {
        let sum = 0;
        for (let k = 0; k < ucols; k++) {
          sum += X.get(i, k) * U.get(j, k);
        }
        Y.set(i, j, sum);
      }
    }
    return Y;
  }
  get condition() {
    return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
  }
  get norm2() {
    return this.s[0];
  }
  get rank() {
    let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
    let r = 0;
    let s = this.s;
    for (let i = 0, ii = s.length; i < ii; i++) {
      if (s[i] > tol) {
        r++;
      }
    }
    return r;
  }
  get diagonal() {
    return Array.from(this.s);
  }
  get threshold() {
    return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
  }
  get leftSingularVectors() {
    return this.U;
  }
  get rightSingularVectors() {
    return this.V;
  }
  get diagonalMatrix() {
    return Matrix$1.diag(this.s);
  }
}
function inverse(matrix, useSVD = false) {
  matrix = WrapperMatrix2D.checkMatrix(matrix);
  if (useSVD) {
    return new SingularValueDecomposition(matrix).inverse();
  } else {
    return solve$1(matrix, Matrix$1.eye(matrix.rows));
  }
}
function solve$1(leftHandSide, rightHandSide, useSVD = false) {
  leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
  rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
  if (useSVD) {
    return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
  } else {
    return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
  }
}
function determinant(matrix) {
  matrix = Matrix$1.checkMatrix(matrix);
  if (matrix.isSquare()) {
    if (matrix.columns === 0) {
      return 1;
    }
    let a, b, c, d;
    if (matrix.columns === 2) {
      // 2 x 2 matrix
      a = matrix.get(0, 0);
      b = matrix.get(0, 1);
      c = matrix.get(1, 0);
      d = matrix.get(1, 1);
      return a * d - b * c;
    } else if (matrix.columns === 3) {
      // 3 x 3 matrix
      let subMatrix0, subMatrix1, subMatrix2;
      subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
      subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
      subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
      a = matrix.get(0, 0);
      b = matrix.get(0, 1);
      c = matrix.get(0, 2);
      return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
    } else {
      // general purpose determinant using the LU decomposition
      return new LuDecomposition(matrix).determinant;
    }
  } else {
    throw Error('determinant can only be calculated for a square matrix');
  }
}
function xrange(n, exception) {
  let range = [];
  for (let i = 0; i < n; i++) {
    if (i !== exception) {
      range.push(i);
    }
  }
  return range;
}
function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
  if (error > thresholdError) {
    return new Array(matrix.rows + 1).fill(0);
  } else {
    let returnArray = matrix.addRow(index, [0]);
    for (let i = 0; i < returnArray.rows; i++) {
      if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
        returnArray.set(i, 0, 0);
      }
    }
    return returnArray.to1DArray();
  }
}
function linearDependencies(matrix, options = {}) {
  const {
    thresholdValue = 10e-10,
    thresholdError = 10e-10
  } = options;
  matrix = Matrix$1.checkMatrix(matrix);
  let n = matrix.rows;
  let results = new Matrix$1(n, n);
  for (let i = 0; i < n; i++) {
    let b = Matrix$1.columnVector(matrix.getRow(i));
    let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
    let svd = new SingularValueDecomposition(Abis);
    let x = svd.solve(b);
    let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
    results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
  }
  return results;
}
function pseudoInverse(matrix, threshold = Number.EPSILON) {
  matrix = Matrix$1.checkMatrix(matrix);
  if (matrix.isEmpty()) {
    // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
    // (0xn)*(nx0)*(0xn) = 0xn
    // (nx0)*(0xn)*(nx0) = nx0
    return matrix.transpose();
  }
  let svdSolution = new SingularValueDecomposition(matrix, {
    autoTranspose: true
  });
  let U = svdSolution.leftSingularVectors;
  let V = svdSolution.rightSingularVectors;
  let s = svdSolution.diagonal;
  for (let i = 0; i < s.length; i++) {
    if (Math.abs(s[i]) > threshold) {
      s[i] = 1.0 / s[i];
    } else {
      s[i] = 0.0;
    }
  }
  return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
}
function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
  xMatrix = new Matrix$1(xMatrix);
  let yIsSame = false;
  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
    options = yMatrix;
    yMatrix = xMatrix;
    yIsSame = true;
  } else {
    yMatrix = new Matrix$1(yMatrix);
  }
  if (xMatrix.rows !== yMatrix.rows) {
    throw new TypeError('Both matrices must have the same number of rows');
  }
  const {
    center = true
  } = options;
  if (center) {
    xMatrix = xMatrix.center('column');
    if (!yIsSame) {
      yMatrix = yMatrix.center('column');
    }
  }
  const cov = xMatrix.transpose().mmul(yMatrix);
  for (let i = 0; i < cov.rows; i++) {
    for (let j = 0; j < cov.columns; j++) {
      cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
    }
  }
  return cov;
}
function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
  xMatrix = new Matrix$1(xMatrix);
  let yIsSame = false;
  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
    options = yMatrix;
    yMatrix = xMatrix;
    yIsSame = true;
  } else {
    yMatrix = new Matrix$1(yMatrix);
  }
  if (xMatrix.rows !== yMatrix.rows) {
    throw new TypeError('Both matrices must have the same number of rows');
  }
  const {
    center = true,
    scale = true
  } = options;
  if (center) {
    xMatrix.center('column');
    if (!yIsSame) {
      yMatrix.center('column');
    }
  }
  if (scale) {
    xMatrix.scale('column');
    if (!yIsSame) {
      yMatrix.scale('column');
    }
  }
  const sdx = xMatrix.standardDeviation('column', {
    unbiased: true
  });
  const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
    unbiased: true
  });
  const corr = xMatrix.transpose().mmul(yMatrix);
  for (let i = 0; i < corr.rows; i++) {
    for (let j = 0; j < corr.columns; j++) {
      corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
    }
  }
  return corr;
}
class EigenvalueDecomposition {
  constructor(matrix, options = {}) {
    const {
      assumeSymmetric = false
    } = options;
    matrix = WrapperMatrix2D.checkMatrix(matrix);
    if (!matrix.isSquare()) {
      throw new Error('Matrix is not a square matrix');
    }
    if (matrix.isEmpty()) {
      throw new Error('Matrix must be non-empty');
    }
    let n = matrix.columns;
    let V = new Matrix$1(n, n);
    let d = new Float64Array(n);
    let e = new Float64Array(n);
    let value = matrix;
    let i, j;
    let isSymmetric = false;
    if (assumeSymmetric) {
      isSymmetric = true;
    } else {
      isSymmetric = matrix.isSymmetric();
    }
    if (isSymmetric) {
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, value.get(i, j));
        }
      }
      tred2(n, e, d, V);
      tql2(n, e, d, V);
    } else {
      let H = new Matrix$1(n, n);
      let ort = new Float64Array(n);
      for (j = 0; j < n; j++) {
        for (i = 0; i < n; i++) {
          H.set(i, j, value.get(i, j));
        }
      }
      orthes(n, H, ort, V);
      hqr2(n, e, d, V, H);
    }
    this.n = n;
    this.e = e;
    this.d = d;
    this.V = V;
  }
  get realEigenvalues() {
    return Array.from(this.d);
  }
  get imaginaryEigenvalues() {
    return Array.from(this.e);
  }
  get eigenvectorMatrix() {
    return this.V;
  }
  get diagonalMatrix() {
    let n = this.n;
    let e = this.e;
    let d = this.d;
    let X = new Matrix$1(n, n);
    let i, j;
    for (i = 0; i < n; i++) {
      for (j = 0; j < n; j++) {
        X.set(i, j, 0);
      }
      X.set(i, i, d[i]);
      if (e[i] > 0) {
        X.set(i, i + 1, e[i]);
      } else if (e[i] < 0) {
        X.set(i, i - 1, e[i]);
      }
    }
    return X;
  }
}
function tred2(n, e, d, V) {
  let f, g, h, i, j, k, hh, scale;
  for (j = 0; j < n; j++) {
    d[j] = V.get(n - 1, j);
  }
  for (i = n - 1; i > 0; i--) {
    scale = 0;
    h = 0;
    for (k = 0; k < i; k++) {
      scale = scale + Math.abs(d[k]);
    }
    if (scale === 0) {
      e[i] = d[i - 1];
      for (j = 0; j < i; j++) {
        d[j] = V.get(i - 1, j);
        V.set(i, j, 0);
        V.set(j, i, 0);
      }
    } else {
      for (k = 0; k < i; k++) {
        d[k] /= scale;
        h += d[k] * d[k];
      }
      f = d[i - 1];
      g = Math.sqrt(h);
      if (f > 0) {
        g = -g;
      }
      e[i] = scale * g;
      h = h - f * g;
      d[i - 1] = f - g;
      for (j = 0; j < i; j++) {
        e[j] = 0;
      }
      for (j = 0; j < i; j++) {
        f = d[j];
        V.set(j, i, f);
        g = e[j] + V.get(j, j) * f;
        for (k = j + 1; k <= i - 1; k++) {
          g += V.get(k, j) * d[k];
          e[k] += V.get(k, j) * f;
        }
        e[j] = g;
      }
      f = 0;
      for (j = 0; j < i; j++) {
        e[j] /= h;
        f += e[j] * d[j];
      }
      hh = f / (h + h);
      for (j = 0; j < i; j++) {
        e[j] -= hh * d[j];
      }
      for (j = 0; j < i; j++) {
        f = d[j];
        g = e[j];
        for (k = j; k <= i - 1; k++) {
          V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
        }
        d[j] = V.get(i - 1, j);
        V.set(i, j, 0);
      }
    }
    d[i] = h;
  }
  for (i = 0; i < n - 1; i++) {
    V.set(n - 1, i, V.get(i, i));
    V.set(i, i, 1);
    h = d[i + 1];
    if (h !== 0) {
      for (k = 0; k <= i; k++) {
        d[k] = V.get(k, i + 1) / h;
      }
      for (j = 0; j <= i; j++) {
        g = 0;
        for (k = 0; k <= i; k++) {
          g += V.get(k, i + 1) * V.get(k, j);
        }
        for (k = 0; k <= i; k++) {
          V.set(k, j, V.get(k, j) - g * d[k]);
        }
      }
    }
    for (k = 0; k <= i; k++) {
      V.set(k, i + 1, 0);
    }
  }
  for (j = 0; j < n; j++) {
    d[j] = V.get(n - 1, j);
    V.set(n - 1, j, 0);
  }
  V.set(n - 1, n - 1, 1);
  e[0] = 0;
}
function tql2(n, e, d, V) {
  let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
  for (i = 1; i < n; i++) {
    e[i - 1] = e[i];
  }
  e[n - 1] = 0;
  let f = 0;
  let tst1 = 0;
  let eps = Number.EPSILON;
  for (l = 0; l < n; l++) {
    tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
    m = l;
    while (m < n) {
      if (Math.abs(e[m]) <= eps * tst1) {
        break;
      }
      m++;
    }
    if (m > l) {
      do {
        g = d[l];
        p = (d[l + 1] - g) / (2 * e[l]);
        r = hypotenuse(p, 1);
        if (p < 0) {
          r = -r;
        }
        d[l] = e[l] / (p + r);
        d[l + 1] = e[l] * (p + r);
        dl1 = d[l + 1];
        h = g - d[l];
        for (i = l + 2; i < n; i++) {
          d[i] -= h;
        }
        f = f + h;
        p = d[m];
        c = 1;
        c2 = c;
        c3 = c;
        el1 = e[l + 1];
        s = 0;
        s2 = 0;
        for (i = m - 1; i >= l; i--) {
          c3 = c2;
          c2 = c;
          s2 = s;
          g = c * e[i];
          h = c * p;
          r = hypotenuse(p, e[i]);
          e[i + 1] = s * r;
          s = e[i] / r;
          c = p / r;
          p = c * d[i] - s * g;
          d[i + 1] = h + s * (c * g + s * d[i]);
          for (k = 0; k < n; k++) {
            h = V.get(k, i + 1);
            V.set(k, i + 1, s * V.get(k, i) + c * h);
            V.set(k, i, c * V.get(k, i) - s * h);
          }
        }
        p = -s * s2 * c3 * el1 * e[l] / dl1;
        e[l] = s * p;
        d[l] = c * p;
      } while (Math.abs(e[l]) > eps * tst1);
    }
    d[l] = d[l] + f;
    e[l] = 0;
  }
  for (i = 0; i < n - 1; i++) {
    k = i;
    p = d[i];
    for (j = i + 1; j < n; j++) {
      if (d[j] < p) {
        k = j;
        p = d[j];
      }
    }
    if (k !== i) {
      d[k] = d[i];
      d[i] = p;
      for (j = 0; j < n; j++) {
        p = V.get(j, i);
        V.set(j, i, V.get(j, k));
        V.set(j, k, p);
      }
    }
  }
}
function orthes(n, H, ort, V) {
  let low = 0;
  let high = n - 1;
  let f, g, h, i, j, m;
  let scale;
  for (m = low + 1; m <= high - 1; m++) {
    scale = 0;
    for (i = m; i <= high; i++) {
      scale = scale + Math.abs(H.get(i, m - 1));
    }
    if (scale !== 0) {
      h = 0;
      for (i = high; i >= m; i--) {
        ort[i] = H.get(i, m - 1) / scale;
        h += ort[i] * ort[i];
      }
      g = Math.sqrt(h);
      if (ort[m] > 0) {
        g = -g;
      }
      h = h - ort[m] * g;
      ort[m] = ort[m] - g;
      for (j = m; j < n; j++) {
        f = 0;
        for (i = high; i >= m; i--) {
          f += ort[i] * H.get(i, j);
        }
        f = f / h;
        for (i = m; i <= high; i++) {
          H.set(i, j, H.get(i, j) - f * ort[i]);
        }
      }
      for (i = 0; i <= high; i++) {
        f = 0;
        for (j = high; j >= m; j--) {
          f += ort[j] * H.get(i, j);
        }
        f = f / h;
        for (j = m; j <= high; j++) {
          H.set(i, j, H.get(i, j) - f * ort[j]);
        }
      }
      ort[m] = scale * ort[m];
      H.set(m, m - 1, scale * g);
    }
  }
  for (i = 0; i < n; i++) {
    for (j = 0; j < n; j++) {
      V.set(i, j, i === j ? 1 : 0);
    }
  }
  for (m = high - 1; m >= low + 1; m--) {
    if (H.get(m, m - 1) !== 0) {
      for (i = m + 1; i <= high; i++) {
        ort[i] = H.get(i, m - 1);
      }
      for (j = m; j <= high; j++) {
        g = 0;
        for (i = m; i <= high; i++) {
          g += ort[i] * V.get(i, j);
        }
        g = g / ort[m] / H.get(m, m - 1);
        for (i = m; i <= high; i++) {
          V.set(i, j, V.get(i, j) + g * ort[i]);
        }
      }
    }
  }
}
function hqr2(nn, e, d, V, H) {
  let n = nn - 1;
  let low = 0;
  let high = nn - 1;
  let eps = Number.EPSILON;
  let exshift = 0;
  let norm = 0;
  let p = 0;
  let q = 0;
  let r = 0;
  let s = 0;
  let z = 0;
  let iter = 0;
  let i, j, k, l, m, t, w, x, y;
  let ra, sa, vr, vi;
  let notlast, cdivres;
  for (i = 0; i < nn; i++) {
    if (i < low || i > high) {
      d[i] = H.get(i, i);
      e[i] = 0;
    }
    for (j = Math.max(i - 1, 0); j < nn; j++) {
      norm = norm + Math.abs(H.get(i, j));
    }
  }
  while (n >= low) {
    l = n;
    while (l > low) {
      s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
      if (s === 0) {
        s = norm;
      }
      if (Math.abs(H.get(l, l - 1)) < eps * s) {
        break;
      }
      l--;
    }
    if (l === n) {
      H.set(n, n, H.get(n, n) + exshift);
      d[n] = H.get(n, n);
      e[n] = 0;
      n--;
      iter = 0;
    } else if (l === n - 1) {
      w = H.get(n, n - 1) * H.get(n - 1, n);
      p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
      q = p * p + w;
      z = Math.sqrt(Math.abs(q));
      H.set(n, n, H.get(n, n) + exshift);
      H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
      x = H.get(n, n);
      if (q >= 0) {
        z = p >= 0 ? p + z : p - z;
        d[n - 1] = x + z;
        d[n] = d[n - 1];
        if (z !== 0) {
          d[n] = x - w / z;
        }
        e[n - 1] = 0;
        e[n] = 0;
        x = H.get(n, n - 1);
        s = Math.abs(x) + Math.abs(z);
        p = x / s;
        q = z / s;
        r = Math.sqrt(p * p + q * q);
        p = p / r;
        q = q / r;
        for (j = n - 1; j < nn; j++) {
          z = H.get(n - 1, j);
          H.set(n - 1, j, q * z + p * H.get(n, j));
          H.set(n, j, q * H.get(n, j) - p * z);
        }
        for (i = 0; i <= n; i++) {
          z = H.get(i, n - 1);
          H.set(i, n - 1, q * z + p * H.get(i, n));
          H.set(i, n, q * H.get(i, n) - p * z);
        }
        for (i = low; i <= high; i++) {
          z = V.get(i, n - 1);
          V.set(i, n - 1, q * z + p * V.get(i, n));
          V.set(i, n, q * V.get(i, n) - p * z);
        }
      } else {
        d[n - 1] = x + p;
        d[n] = x + p;
        e[n - 1] = z;
        e[n] = -z;
      }
      n = n - 2;
      iter = 0;
    } else {
      x = H.get(n, n);
      y = 0;
      w = 0;
      if (l < n) {
        y = H.get(n - 1, n - 1);
        w = H.get(n, n - 1) * H.get(n - 1, n);
      }
      if (iter === 10) {
        exshift += x;
        for (i = low; i <= n; i++) {
          H.set(i, i, H.get(i, i) - x);
        }
        s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
        // eslint-disable-next-line no-multi-assign
        x = y = 0.75 * s;
        w = -0.4375 * s * s;
      }
      if (iter === 30) {
        s = (y - x) / 2;
        s = s * s + w;
        if (s > 0) {
          s = Math.sqrt(s);
          if (y < x) {
            s = -s;
          }
          s = x - w / ((y - x) / 2 + s);
          for (i = low; i <= n; i++) {
            H.set(i, i, H.get(i, i) - s);
          }
          exshift += s;
          // eslint-disable-next-line no-multi-assign
          x = y = w = 0.964;
        }
      }
      iter = iter + 1;
      m = n - 2;
      while (m >= l) {
        z = H.get(m, m);
        r = x - z;
        s = y - z;
        p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
        q = H.get(m + 1, m + 1) - z - r - s;
        r = H.get(m + 2, m + 1);
        s = Math.abs(p) + Math.abs(q) + Math.abs(r);
        p = p / s;
        q = q / s;
        r = r / s;
        if (m === l) {
          break;
        }
        if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
          break;
        }
        m--;
      }
      for (i = m + 2; i <= n; i++) {
        H.set(i, i - 2, 0);
        if (i > m + 2) {
          H.set(i, i - 3, 0);
        }
      }
      for (k = m; k <= n - 1; k++) {
        notlast = k !== n - 1;
        if (k !== m) {
          p = H.get(k, k - 1);
          q = H.get(k + 1, k - 1);
          r = notlast ? H.get(k + 2, k - 1) : 0;
          x = Math.abs(p) + Math.abs(q) + Math.abs(r);
          if (x !== 0) {
            p = p / x;
            q = q / x;
            r = r / x;
          }
        }
        if (x === 0) {
          break;
        }
        s = Math.sqrt(p * p + q * q + r * r);
        if (p < 0) {
          s = -s;
        }
        if (s !== 0) {
          if (k !== m) {
            H.set(k, k - 1, -s * x);
          } else if (l !== m) {
            H.set(k, k - 1, -H.get(k, k - 1));
          }
          p = p + s;
          x = p / s;
          y = q / s;
          z = r / s;
          q = q / p;
          r = r / p;
          for (j = k; j < nn; j++) {
            p = H.get(k, j) + q * H.get(k + 1, j);
            if (notlast) {
              p = p + r * H.get(k + 2, j);
              H.set(k + 2, j, H.get(k + 2, j) - p * z);
            }
            H.set(k, j, H.get(k, j) - p * x);
            H.set(k + 1, j, H.get(k + 1, j) - p * y);
          }
          for (i = 0; i <= Math.min(n, k + 3); i++) {
            p = x * H.get(i, k) + y * H.get(i, k + 1);
            if (notlast) {
              p = p + z * H.get(i, k + 2);
              H.set(i, k + 2, H.get(i, k + 2) - p * r);
            }
            H.set(i, k, H.get(i, k) - p);
            H.set(i, k + 1, H.get(i, k + 1) - p * q);
          }
          for (i = low; i <= high; i++) {
            p = x * V.get(i, k) + y * V.get(i, k + 1);
            if (notlast) {
              p = p + z * V.get(i, k + 2);
              V.set(i, k + 2, V.get(i, k + 2) - p * r);
            }
            V.set(i, k, V.get(i, k) - p);
            V.set(i, k + 1, V.get(i, k + 1) - p * q);
          }
        }
      }
    }
  }
  if (norm === 0) {
    return;
  }
  for (n = nn - 1; n >= 0; n--) {
    p = d[n];
    q = e[n];
    if (q === 0) {
      l = n;
      H.set(n, n, 1);
      for (i = n - 1; i >= 0; i--) {
        w = H.get(i, i) - p;
        r = 0;
        for (j = l; j <= n; j++) {
          r = r + H.get(i, j) * H.get(j, n);
        }
        if (e[i] < 0) {
          z = w;
          s = r;
        } else {
          l = i;
          if (e[i] === 0) {
            H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
          } else {
            x = H.get(i, i + 1);
            y = H.get(i + 1, i);
            q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
            t = (x * s - z * r) / q;
            H.set(i, n, t);
            H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
          }
          t = Math.abs(H.get(i, n));
          if (eps * t * t > 1) {
            for (j = i; j <= n; j++) {
              H.set(j, n, H.get(j, n) / t);
            }
          }
        }
      }
    } else if (q < 0) {
      l = n - 1;
      if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
        H.set(n - 1, n - 1, q / H.get(n, n - 1));
        H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
      } else {
        cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
        H.set(n - 1, n - 1, cdivres[0]);
        H.set(n - 1, n, cdivres[1]);
      }
      H.set(n, n - 1, 0);
      H.set(n, n, 1);
      for (i = n - 2; i >= 0; i--) {
        ra = 0;
        sa = 0;
        for (j = l; j <= n; j++) {
          ra = ra + H.get(i, j) * H.get(j, n - 1);
          sa = sa + H.get(i, j) * H.get(j, n);
        }
        w = H.get(i, i) - p;
        if (e[i] < 0) {
          z = w;
          r = ra;
          s = sa;
        } else {
          l = i;
          if (e[i] === 0) {
            cdivres = cdiv(-ra, -sa, w, q);
            H.set(i, n - 1, cdivres[0]);
            H.set(i, n, cdivres[1]);
          } else {
            x = H.get(i, i + 1);
            y = H.get(i + 1, i);
            vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
            vi = (d[i] - p) * 2 * q;
            if (vr === 0 && vi === 0) {
              vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
            }
            cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
            H.set(i, n - 1, cdivres[0]);
            H.set(i, n, cdivres[1]);
            if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
              H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
              H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
            } else {
              cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
              H.set(i + 1, n - 1, cdivres[0]);
              H.set(i + 1, n, cdivres[1]);
            }
          }
          t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
          if (eps * t * t > 1) {
            for (j = i; j <= n; j++) {
              H.set(j, n - 1, H.get(j, n - 1) / t);
              H.set(j, n, H.get(j, n) / t);
            }
          }
        }
      }
    }
  }
  for (i = 0; i < nn; i++) {
    if (i < low || i > high) {
      for (j = i; j < nn; j++) {
        V.set(i, j, H.get(i, j));
      }
    }
  }
  for (j = nn - 1; j >= low; j--) {
    for (i = low; i <= high; i++) {
      z = 0;
      for (k = low; k <= Math.min(j, high); k++) {
        z = z + V.get(i, k) * H.get(k, j);
      }
      V.set(i, j, z);
    }
  }
}
function cdiv(xr, xi, yr, yi) {
  let r, d;
  if (Math.abs(yr) > Math.abs(yi)) {
    r = yi / yr;
    d = yr + r * yi;
    return [(xr + r * xi) / d, (xi - r * xr) / d];
  } else {
    r = yr / yi;
    d = yi + r * yr;
    return [(r * xr + xi) / d, (r * xi - xr) / d];
  }
}
class CholeskyDecomposition {
  constructor(value) {
    value = WrapperMatrix2D.checkMatrix(value);
    if (!value.isSymmetric()) {
      throw new Error('Matrix is not symmetric');
    }
    let a = value;
    let dimension = a.rows;
    let l = new Matrix$1(dimension, dimension);
    let positiveDefinite = true;
    let i, j, k;
    for (j = 0; j < dimension; j++) {
      let d = 0;
      for (k = 0; k < j; k++) {
        let s = 0;
        for (i = 0; i < k; i++) {
          s += l.get(k, i) * l.get(j, i);
        }
        s = (a.get(j, k) - s) / l.get(k, k);
        l.set(j, k, s);
        d = d + s * s;
      }
      d = a.get(j, j) - d;
      positiveDefinite &&= d > 0;
      l.set(j, j, Math.sqrt(Math.max(d, 0)));
      for (k = j + 1; k < dimension; k++) {
        l.set(j, k, 0);
      }
    }
    this.L = l;
    this.positiveDefinite = positiveDefinite;
  }
  isPositiveDefinite() {
    return this.positiveDefinite;
  }
  solve(value) {
    value = WrapperMatrix2D.checkMatrix(value);
    let l = this.L;
    let dimension = l.rows;
    if (value.rows !== dimension) {
      throw new Error('Matrix dimensions do not match');
    }
    if (this.isPositiveDefinite() === false) {
      throw new Error('Matrix is not positive definite');
    }
    let count = value.columns;
    let B = value.clone();
    let i, j, k;
    for (k = 0; k < dimension; k++) {
      for (j = 0; j < count; j++) {
        for (i = 0; i < k; i++) {
          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
        }
        B.set(k, j, B.get(k, j) / l.get(k, k));
      }
    }
    for (k = dimension - 1; k >= 0; k--) {
      for (j = 0; j < count; j++) {
        for (i = k + 1; i < dimension; i++) {
          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
        }
        B.set(k, j, B.get(k, j) / l.get(k, k));
      }
    }
    return B;
  }
  get lowerTriangularMatrix() {
    return this.L;
  }
}
class nipals {
  constructor(X, options = {}) {
    X = WrapperMatrix2D.checkMatrix(X);
    let {
      Y
    } = options;
    const {
      scaleScores = false,
      maxIterations = 1000,
      terminationCriteria = 1e-10
    } = options;
    let u;
    if (Y) {
      if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
        Y = Matrix$1.columnVector(Y);
      } else {
        Y = WrapperMatrix2D.checkMatrix(Y);
      }
      if (Y.rows !== X.rows) {
        throw new Error('Y should have the same number of rows as X');
      }
      u = Y.getColumnVector(0);
    } else {
      u = X.getColumnVector(0);
    }
    let diff = 1;
    let t, q, w, tOld;
    for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
      w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
      w = w.div(w.norm());
      t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
      if (counter > 0) {
        diff = t.clone().sub(tOld).pow(2).sum();
      }
      tOld = t.clone();
      if (Y) {
        q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
        q = q.div(q.norm());
        u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
      } else {
        u = t;
      }
    }
    if (Y) {
      let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
      p = p.div(p.norm());
      let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
      let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
      let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
      this.t = t;
      this.p = p.transpose();
      this.w = w.transpose();
      this.q = q;
      this.u = u;
      this.s = t.transpose().mmul(t);
      this.xResidual = xResidual;
      this.yResidual = yResidual;
      this.betas = residual;
    } else {
      this.w = w.transpose();
      this.s = t.transpose().mmul(t).sqrt();
      if (scaleScores) {
        this.t = t.clone().div(this.s.get(0, 0));
      } else {
        this.t = t;
      }
      this.xResidual = X.sub(t.mmul(w.transpose()));
    }
  }
}
matrix.AbstractMatrix = AbstractMatrix;
matrix.CHO = CholeskyDecomposition;
matrix.CholeskyDecomposition = CholeskyDecomposition;
matrix.DistanceMatrix = DistanceMatrix;
matrix.EVD = EigenvalueDecomposition;
matrix.EigenvalueDecomposition = EigenvalueDecomposition;
matrix.LU = LuDecomposition;
matrix.LuDecomposition = LuDecomposition;
var Matrix_1 = matrix.Matrix = Matrix$1;
matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
matrix.MatrixColumnView = MatrixColumnView;
matrix.MatrixFlipColumnView = MatrixFlipColumnView;
matrix.MatrixFlipRowView = MatrixFlipRowView;
matrix.MatrixRowSelectionView = MatrixRowSelectionView;
matrix.MatrixRowView = MatrixRowView;
matrix.MatrixSelectionView = MatrixSelectionView;
matrix.MatrixSubView = MatrixSubView;
var MatrixTransposeView_1 = matrix.MatrixTransposeView = MatrixTransposeView$1;
matrix.NIPALS = nipals;
matrix.Nipals = nipals;
matrix.QR = QrDecomposition;
matrix.QrDecomposition = QrDecomposition;
matrix.SVD = SingularValueDecomposition;
matrix.SingularValueDecomposition = SingularValueDecomposition;
matrix.SymmetricMatrix = SymmetricMatrix;
matrix.WrapperMatrix1D = WrapperMatrix1D;
matrix.WrapperMatrix2D = WrapperMatrix2D;
matrix.correlation = correlation;
matrix.covariance = covariance;
var _default = matrix.default = Matrix$1;
matrix.determinant = determinant;
matrix.inverse = inverse;
matrix.linearDependencies = linearDependencies;
matrix.pseudoInverse = pseudoInverse;
var solve_1 = matrix.solve = solve$1;
matrix.wrap = wrap;

const Matrix = Matrix_1;
const MatrixTransposeView = MatrixTransposeView_1;
_default.Matrix ? _default.Matrix : Matrix_1;
const solve = solve_1;

/**
 * Computes the mean value of an array of values.
 * @param array - array of numbers
 * @param options - options
 */
function xMean(array, options = {}) {
  xCheck(array);
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(array, options);
  let sumValue = array[fromIndex];
  for (let i = fromIndex + 1; i <= toIndex; i++) {
    sumValue += array[i];
  }
  return sumValue / (toIndex - fromIndex + 1);
}

/**
 * Finds the variance of the data
 * @param values - the values of the array
 * @param options - options
 * @returns variance
 */
function xVariance(values, options = {}) {
  if (!isAnyArray$1(values)) {
    throw new TypeError('input must be an array');
  }
  const {
    unbiased = true,
    mean = xMean(values)
  } = options;
  let sqrError = 0;
  for (const value of values) {
    const x = value - mean;
    sqrError += x * x;
  }
  if (unbiased) {
    return sqrError / (values.length - 1);
  } else {
    return sqrError / values.length;
  }
}

/**
 * Finds the standard deviation for the data at hand
 * @param values - values in the data
 * @param options - options
 * @returns standard deviation
 */
function xStandardDeviation(values, options = {}) {
  return Math.sqrt(xVariance(values, options));
}

/**
 * Returns a copy of the data as a Float64Array.
 * @param array - array of numbers
 */
function xEnsureFloat64(array) {
  xCheck(array);
  if (array instanceof Float64Array) {
    return array.slice(0);
  }
  return Float64Array.from(array);
}

function FFT(size) {
  this.size = size | 0;
  if (this.size <= 1 || (this.size & this.size - 1) !== 0) throw new Error('FFT size must be a power of two and bigger than 1');
  this._csize = size << 1;

  // NOTE: Use of `var` is intentional for old V8 versions
  var table = new Array(this.size * 2);
  for (var i = 0; i < table.length; i += 2) {
    const angle = Math.PI * i / this.size;
    table[i] = Math.cos(angle);
    table[i + 1] = -Math.sin(angle);
  }
  this.table = table;

  // Find size's power of two
  var power = 0;
  for (var t = 1; this.size > t; t <<= 1) power++;

  // Calculate initial step's width:
  //   * If we are full radix-4 - it is 2x smaller to give inital len=8
  //   * Otherwise it is the same as `power` to give len=4
  this._width = power % 2 === 0 ? power - 1 : power;

  // Pre-compute bit-reversal patterns
  this._bitrev = new Array(1 << this._width);
  for (var j = 0; j < this._bitrev.length; j++) {
    this._bitrev[j] = 0;
    for (var shift = 0; shift < this._width; shift += 2) {
      var revShift = this._width - shift - 2;
      this._bitrev[j] |= (j >>> shift & 3) << revShift;
    }
  }
  this._out = null;
  this._data = null;
  this._inv = 0;
}
FFT.prototype.fromComplexArray = function fromComplexArray(complex, storage) {
  var res = storage || new Array(complex.length >>> 1);
  for (var i = 0; i < complex.length; i += 2) res[i >>> 1] = complex[i];
  return res;
};
FFT.prototype.createComplexArray = function createComplexArray() {
  const res = new Array(this._csize);
  for (var i = 0; i < res.length; i++) res[i] = 0;
  return res;
};
FFT.prototype.toComplexArray = function toComplexArray(input, storage) {
  var res = storage || this.createComplexArray();
  for (var i = 0; i < res.length; i += 2) {
    res[i] = input[i >>> 1];
    res[i + 1] = 0;
  }
  return res;
};
FFT.prototype.completeSpectrum = function completeSpectrum(spectrum) {
  var size = this._csize;
  var half = size >>> 1;
  for (var i = 2; i < half; i += 2) {
    spectrum[size - i] = spectrum[i];
    spectrum[size - i + 1] = -spectrum[i + 1];
  }
};
FFT.prototype.transform = function transform(out, data) {
  if (out === data) throw new Error('Input and output buffers must be different');
  this._out = out;
  this._data = data;
  this._inv = 0;
  this._transform4();
  this._out = null;
  this._data = null;
};
FFT.prototype.realTransform = function realTransform(out, data) {
  if (out === data) throw new Error('Input and output buffers must be different');
  this._out = out;
  this._data = data;
  this._inv = 0;
  this._realTransform4();
  this._out = null;
  this._data = null;
};
FFT.prototype.inverseTransform = function inverseTransform(out, data) {
  if (out === data) throw new Error('Input and output buffers must be different');
  this._out = out;
  this._data = data;
  this._inv = 1;
  this._transform4();
  for (var i = 0; i < out.length; i++) out[i] /= this.size;
  this._out = null;
  this._data = null;
};

// radix-4 implementation
//
// NOTE: Uses of `var` are intentional for older V8 version that do not
// support both `let compound assignments` and `const phi`
FFT.prototype._transform4 = function _transform4() {
  var out = this._out;
  var size = this._csize;

  // Initial step (permute and transform)
  var width = this._width;
  var step = 1 << width;
  var len = size / step << 1;
  var outOff;
  var t;
  var bitrev = this._bitrev;
  if (len === 4) {
    for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
      const off = bitrev[t];
      this._singleTransform2(outOff, off, step);
    }
  } else {
    // len === 8
    for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
      const off = bitrev[t];
      this._singleTransform4(outOff, off, step);
    }
  }

  // Loop through steps in decreasing order
  var inv = this._inv ? -1 : 1;
  var table = this.table;
  for (step >>= 2; step >= 2; step >>= 2) {
    len = size / step << 1;
    var quarterLen = len >>> 2;

    // Loop through offsets in the data
    for (outOff = 0; outOff < size; outOff += len) {
      // Full case
      var limit = outOff + quarterLen;
      for (var i = outOff, k = 0; i < limit; i += 2, k += step) {
        const A = i;
        const B = A + quarterLen;
        const C = B + quarterLen;
        const D = C + quarterLen;

        // Original values
        const Ar = out[A];
        const Ai = out[A + 1];
        const Br = out[B];
        const Bi = out[B + 1];
        const Cr = out[C];
        const Ci = out[C + 1];
        const Dr = out[D];
        const Di = out[D + 1];

        // Middle values
        const MAr = Ar;
        const MAi = Ai;
        const tableBr = table[k];
        const tableBi = inv * table[k + 1];
        const MBr = Br * tableBr - Bi * tableBi;
        const MBi = Br * tableBi + Bi * tableBr;
        const tableCr = table[2 * k];
        const tableCi = inv * table[2 * k + 1];
        const MCr = Cr * tableCr - Ci * tableCi;
        const MCi = Cr * tableCi + Ci * tableCr;
        const tableDr = table[3 * k];
        const tableDi = inv * table[3 * k + 1];
        const MDr = Dr * tableDr - Di * tableDi;
        const MDi = Dr * tableDi + Di * tableDr;

        // Pre-Final values
        const T0r = MAr + MCr;
        const T0i = MAi + MCi;
        const T1r = MAr - MCr;
        const T1i = MAi - MCi;
        const T2r = MBr + MDr;
        const T2i = MBi + MDi;
        const T3r = inv * (MBr - MDr);
        const T3i = inv * (MBi - MDi);

        // Final values
        const FAr = T0r + T2r;
        const FAi = T0i + T2i;
        const FCr = T0r - T2r;
        const FCi = T0i - T2i;
        const FBr = T1r + T3i;
        const FBi = T1i - T3r;
        const FDr = T1r - T3i;
        const FDi = T1i + T3r;
        out[A] = FAr;
        out[A + 1] = FAi;
        out[B] = FBr;
        out[B + 1] = FBi;
        out[C] = FCr;
        out[C + 1] = FCi;
        out[D] = FDr;
        out[D + 1] = FDi;
      }
    }
  }
};

// radix-2 implementation
//
// NOTE: Only called for len=4
FFT.prototype._singleTransform2 = function _singleTransform2(outOff, off, step) {
  const out = this._out;
  const data = this._data;
  const evenR = data[off];
  const evenI = data[off + 1];
  const oddR = data[off + step];
  const oddI = data[off + step + 1];
  const leftR = evenR + oddR;
  const leftI = evenI + oddI;
  const rightR = evenR - oddR;
  const rightI = evenI - oddI;
  out[outOff] = leftR;
  out[outOff + 1] = leftI;
  out[outOff + 2] = rightR;
  out[outOff + 3] = rightI;
};

// radix-4
//
// NOTE: Only called for len=8
FFT.prototype._singleTransform4 = function _singleTransform4(outOff, off, step) {
  const out = this._out;
  const data = this._data;
  const inv = this._inv ? -1 : 1;
  const step2 = step * 2;
  const step3 = step * 3;

  // Original values
  const Ar = data[off];
  const Ai = data[off + 1];
  const Br = data[off + step];
  const Bi = data[off + step + 1];
  const Cr = data[off + step2];
  const Ci = data[off + step2 + 1];
  const Dr = data[off + step3];
  const Di = data[off + step3 + 1];

  // Pre-Final values
  const T0r = Ar + Cr;
  const T0i = Ai + Ci;
  const T1r = Ar - Cr;
  const T1i = Ai - Ci;
  const T2r = Br + Dr;
  const T2i = Bi + Di;
  const T3r = inv * (Br - Dr);
  const T3i = inv * (Bi - Di);

  // Final values
  const FAr = T0r + T2r;
  const FAi = T0i + T2i;
  const FBr = T1r + T3i;
  const FBi = T1i - T3r;
  const FCr = T0r - T2r;
  const FCi = T0i - T2i;
  const FDr = T1r - T3i;
  const FDi = T1i + T3r;
  out[outOff] = FAr;
  out[outOff + 1] = FAi;
  out[outOff + 2] = FBr;
  out[outOff + 3] = FBi;
  out[outOff + 4] = FCr;
  out[outOff + 5] = FCi;
  out[outOff + 6] = FDr;
  out[outOff + 7] = FDi;
};

// Real input radix-4 implementation
FFT.prototype._realTransform4 = function _realTransform4() {
  var out = this._out;
  var size = this._csize;

  // Initial step (permute and transform)
  var width = this._width;
  var step = 1 << width;
  var len = size / step << 1;
  var outOff;
  var t;
  var bitrev = this._bitrev;
  if (len === 4) {
    for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
      const off = bitrev[t];
      this._singleRealTransform2(outOff, off >>> 1, step >>> 1);
    }
  } else {
    // len === 8
    for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
      const off = bitrev[t];
      this._singleRealTransform4(outOff, off >>> 1, step >>> 1);
    }
  }

  // Loop through steps in decreasing order
  var inv = this._inv ? -1 : 1;
  var table = this.table;
  for (step >>= 2; step >= 2; step >>= 2) {
    len = size / step << 1;
    var halfLen = len >>> 1;
    var quarterLen = halfLen >>> 1;
    var hquarterLen = quarterLen >>> 1;

    // Loop through offsets in the data
    for (outOff = 0; outOff < size; outOff += len) {
      for (var i = 0, k = 0; i <= hquarterLen; i += 2, k += step) {
        var A = outOff + i;
        var B = A + quarterLen;
        var C = B + quarterLen;
        var D = C + quarterLen;

        // Original values
        var Ar = out[A];
        var Ai = out[A + 1];
        var Br = out[B];
        var Bi = out[B + 1];
        var Cr = out[C];
        var Ci = out[C + 1];
        var Dr = out[D];
        var Di = out[D + 1];

        // Middle values
        var MAr = Ar;
        var MAi = Ai;
        var tableBr = table[k];
        var tableBi = inv * table[k + 1];
        var MBr = Br * tableBr - Bi * tableBi;
        var MBi = Br * tableBi + Bi * tableBr;
        var tableCr = table[2 * k];
        var tableCi = inv * table[2 * k + 1];
        var MCr = Cr * tableCr - Ci * tableCi;
        var MCi = Cr * tableCi + Ci * tableCr;
        var tableDr = table[3 * k];
        var tableDi = inv * table[3 * k + 1];
        var MDr = Dr * tableDr - Di * tableDi;
        var MDi = Dr * tableDi + Di * tableDr;

        // Pre-Final values
        var T0r = MAr + MCr;
        var T0i = MAi + MCi;
        var T1r = MAr - MCr;
        var T1i = MAi - MCi;
        var T2r = MBr + MDr;
        var T2i = MBi + MDi;
        var T3r = inv * (MBr - MDr);
        var T3i = inv * (MBi - MDi);

        // Final values
        var FAr = T0r + T2r;
        var FAi = T0i + T2i;
        var FBr = T1r + T3i;
        var FBi = T1i - T3r;
        out[A] = FAr;
        out[A + 1] = FAi;
        out[B] = FBr;
        out[B + 1] = FBi;

        // Output final middle point
        if (i === 0) {
          var FCr = T0r - T2r;
          var FCi = T0i - T2i;
          out[C] = FCr;
          out[C + 1] = FCi;
          continue;
        }

        // Do not overwrite ourselves
        if (i === hquarterLen) continue;

        // In the flipped case:
        // MAi = -MAi
        // MBr=-MBi, MBi=-MBr
        // MCr=-MCr
        // MDr=MDi, MDi=MDr
        var ST0r = T1r;
        var ST0i = -T1i;
        var ST1r = T0r;
        var ST1i = -T0i;
        var ST2r = -inv * T3i;
        var ST2i = -inv * T3r;
        var ST3r = -inv * T2i;
        var ST3i = -inv * T2r;
        var SFAr = ST0r + ST2r;
        var SFAi = ST0i + ST2i;
        var SFBr = ST1r + ST3i;
        var SFBi = ST1i - ST3r;
        var SA = outOff + quarterLen - i;
        var SB = outOff + halfLen - i;
        out[SA] = SFAr;
        out[SA + 1] = SFAi;
        out[SB] = SFBr;
        out[SB + 1] = SFBi;
      }
    }
  }
};

// radix-2 implementation
//
// NOTE: Only called for len=4
FFT.prototype._singleRealTransform2 = function _singleRealTransform2(outOff, off, step) {
  const out = this._out;
  const data = this._data;
  const evenR = data[off];
  const oddR = data[off + step];
  const leftR = evenR + oddR;
  const rightR = evenR - oddR;
  out[outOff] = leftR;
  out[outOff + 1] = 0;
  out[outOff + 2] = rightR;
  out[outOff + 3] = 0;
};

// radix-4
//
// NOTE: Only called for len=8
FFT.prototype._singleRealTransform4 = function _singleRealTransform4(outOff, off, step) {
  const out = this._out;
  const data = this._data;
  const inv = this._inv ? -1 : 1;
  const step2 = step * 2;
  const step3 = step * 3;

  // Original values
  const Ar = data[off];
  const Br = data[off + step];
  const Cr = data[off + step2];
  const Dr = data[off + step3];

  // Pre-Final values
  const T0r = Ar + Cr;
  const T1r = Ar - Cr;
  const T2r = Br + Dr;
  const T3r = inv * (Br - Dr);

  // Final values
  const FAr = T0r + T2r;
  const FBr = T1r;
  const FBi = -T3r;
  const FCr = T0r - T2r;
  const FDr = T1r;
  const FDi = T3r;
  out[outOff] = FAr;
  out[outOff + 1] = 0;
  out[outOff + 2] = FBr;
  out[outOff + 3] = FBi;
  out[outOff + 4] = FCr;
  out[outOff + 5] = 0;
  out[outOff + 6] = FDr;
  out[outOff + 7] = FDi;
};

function createNumberArray(ArrayConstructor, length) {
  if (ArrayConstructor === Array) {
    return new ArrayConstructor(length).fill(0);
  } else {
    return new ArrayConstructor(length);
  }
}

/**
 * Create an array with numbers between "from" and "to" of length "length"
 * @param options - options
 * @returns - array of distributed numbers between "from" and "to"
 */
function createFromToArray(options = {}) {
  const {
    from = 0,
    to = 1,
    length = 1000,
    includeFrom = true,
    includeTo = true,
    distribution = 'uniform'
  } = options;
  const array = new Float64Array(length);
  let div = length;
  if (includeFrom && includeTo) {
    div = length - 1;
  } else if (!includeFrom && includeTo || includeFrom && !includeTo) {
    div = length;
  } else if (!includeFrom && !includeTo) {
    div = length + 1;
  }
  const delta = (to - from) / div;
  if (distribution === 'uniform') {
    if (includeFrom) {
      let index = 0;
      while (index < length) {
        array[index] = from + delta * index;
        index++;
      }
    } else {
      let index = 0;
      while (index < length) {
        array[index] = from + delta * (index + 1);
        index++;
      }
    }
  } else if (distribution === 'log') {
    const base = (to / from) ** (1 / div);
    const firstExponent = Math.log(from) / Math.log(base);
    if (includeFrom) {
      let index = 0;
      while (index < length) {
        array[index] = base ** (firstExponent + index);
        index++;
      }
    } else {
      let index = 0;
      while (index < length) {
        array[index] = base ** (firstExponent + index + 1);
        index++;
      }
    }
  } else {
    throw new Error('distribution must be uniform or log');
  }
  return array;
}

/**
 * This function xSubtract the first array by the second array or a constant value from each element of the first array
 * @param array1 - the array that will be rotated
 * @param array2 - second array or number
 * @returns array after subtraction
 */
function xSubtract(array1, array2) {
  let isConstant = false;
  let constant = 0;
  if (isAnyArray$1(array2)) {
    if (array1.length !== array2.length) {
      throw new Error('size of array1 and array2 must be identical');
    }
  } else {
    isConstant = true;
    constant = array2;
  }
  const array3 = new Float64Array(array1.length);
  if (isConstant) {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] - constant;
    }
  } else {
    for (let i = 0; i < array1.length; i++) {
      array3[i] = array1[i] - array2[i];
    }
  }
  return array3;
}

/**
 * Computes the maximal value of an array of values
 * @param array - array of numbers
 * @param options - options
 */
function xMaxValue(array, options = {}) {
  xCheck(array);
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(array, options);
  let maxValue = array[fromIndex];
  for (let i = fromIndex + 1; i <= toIndex; i++) {
    if (array[i] > maxValue) {
      maxValue = array[i];
    }
  }
  return maxValue;
}

/**
 * Computes the minimal value of an array of values.
 * @param array - array of numbers
 * @param options - options
 */
function xMinValue(array, options = {}) {
  xCheck(array);
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(array, options);
  let minValue = array[fromIndex];
  for (let i = fromIndex + 1; i <= toIndex; i++) {
    if (array[i] < minValue) {
      minValue = array[i];
    }
  }
  return minValue;
}

/**
 * Check if the values are separated always by the same difference
 * @param array - Monotone growing array of number
 * @param options
 */
function xIsEquallySpaced(array, options = {}) {
  if (array.length < 3) return true;
  const {
    tolerance = 0.05
  } = options;
  let maxDx = 0;
  let minDx = Number.MAX_SAFE_INTEGER;
  for (let i = 0; i < array.length - 1; ++i) {
    const absoluteDifference = array[i + 1] - array[i];
    if (absoluteDifference < minDx) {
      minDx = absoluteDifference;
    }
    if (absoluteDifference > maxDx) {
      maxDx = absoluteDifference;
    }
  }
  return (maxDx - minDx) / maxDx < tolerance;
}

/**
 * Returns true if x is monotonic.
 * @param array - array of numbers.
 * @returns 1 if monotonic increasing, -1 if monotonic decreasing, 0 if not monotonic.
 */
function xIsMonotonic(array) {
  if (array.length <= 2) {
    return 1;
  }
  if (array[0] === array[1]) {
    // maybe a constant series
    for (let i = 1; i < array.length - 1; i++) {
      if (array[i] !== array[i + 1]) return 0;
    }
    return 1;
  }
  if (array[0] < array.at(-1)) {
    for (let i = 0; i < array.length - 1; i++) {
      if (array[i] >= array[i + 1]) return 0;
    }
    return 1;
  } else {
    for (let i = 0; i < array.length - 1; i++) {
      if (array[i] <= array[i + 1]) return 0;
    }
    return -1;
  }
}

/**
 * This function calculates the median absolute deviation (MAD).
 * https://en.wikipedia.org/wiki/Median_absolute_deviation
 * @param array
 */
function xMedianAbsoluteDeviation(array) {
  const median = xMedian(array);
  const averageDeviations = new Float64Array(array.length);
  for (let i = 0; i < array.length; i++) {
    averageDeviations[i] = Math.abs(array[i] - median);
  }
  return {
    median,
    mad: xMedian(averageDeviations)
  };
}

/**
 * Return min and max values of an array.
 * @param array - array of number
 * @returns - Object with 2 properties, min and max.
 */
function xMinMaxValues(array) {
  xCheck(array);
  let min = array[0];
  let max = array[0];
  for (const value of array) {
    if (value < min) min = value;
    if (value > max) max = value;
  }
  return {
    min,
    max
  };
}

/*
Adapted from: https://github.com/compute-io/erfcinv/blob/aa116e23883839359e310ad41a7c42f72815fc1e/lib/number.js

The MIT License (MIT)

Copyright (c) 2014-2015 The Compute.io Authors. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.


Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization obtaining a copy of the software and accompanying documentation covered by this license (the "Software") to use, reproduce, display, distribute, execute, and transmit the Software, and to prepare derivative works of the Software, and to permit third-parties to whom the Software is furnished to do so, all subject to the following:

The copyright notices in the Software and this entire statement, including the above license grant, this restriction and the following disclaimer, must be included in all copies of the Software, in whole or in part, and all derivative works of the Software, unless such copies or derivative works are solely in the form of machine-executable object code generated by a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
/* eslint-disable no-loss-of-precision */
// Coefficients for erfcinv on [0, 0.5]:
const Y1 = 8.91314744949340820313e-2;
const P1 = [-0.005387729650712429, 8.22687874676915743155e-3, 2.19878681111168899165e-2, -0.03656379714117627, -0.012692614766297404, 3.34806625409744615033e-2, -0.008368748197417368, -5087819496582806e-19];
const Q1 = [8.86216390456424707504e-4, -0.0023339375937419, 7.95283687341571680018e-2, -0.05273963823400997, -0.7122890234154284, 6.62328840472002992063e-1, 1.56221558398423026363, -1.5657455823417585, -0.9700050433032906, 1];
// Coefficients for erfcinv for 0.5 > 1-x >= 0:
const Y2 = 2.249481201171875;
const P2 = [-3.6719225470772936, 2.11294655448340526258e1, 1.7445385985570866523e1, -44.6382324441787, -18.851064805871424, 1.76447298408374015486e1, 8.37050328343119927838, 1.05264680699391713268e-1, -0.20243350835593876];
const Q2 = [1.72114765761200282724, -22.643693341313973, 1.08268667355460159008e1, 4.85609213108739935468e1, -20.14326346804852, -28.66081804998, 3.9713437953343869095, 6.24264124854247537712, 1];
// Coefficients for erfcinv for sqrt( -log(1-x)):
const Y3 = 8.07220458984375e-1;
const P3 = [-681149956853777e-24, 2.85225331782217055858e-8, -6.794655751811263e-7, 2.14558995388805277169e-3, 2.90157910005329060432e-2, 1.42869534408157156766e-1, 3.37785538912035898924e-1, 3.87079738972604337464e-1, 1.17030156341995252019e-1, -0.16379404719331705, -0.1311027816799519];
const Q3 = [1.105924229346489121e-2, 1.52264338295331783612e-1, 8.48854343457902036425e-1, 2.59301921623620271374, 4.77846592945843778382, 5.38168345707006855425, 3.46625407242567245975, 1];
const Y4 = 9.3995571136474609375e-1;
const P4 = [2.66339227425782031962e-12, -2304047769118826e-25, 4.60469890584317994083e-6, 1.57544617424960554631e-4, 1.87123492819559223345e-3, 9.50804701325919603619e-3, 1.85573306514231072324e-2, -0.0022242652921344794, -0.0350353787183178];
const Q4 = [7.64675292302794483503e-5, 2.63861676657015992959e-3, 3.41589143670947727934e-2, 2.20091105764131249824e-1, 7.62059164553623404043e-1, 1.3653349817554063097, 1];
const Y5 = 9.8362827301025390625e-1;
const P5 = [9.9055709973310326855e-17, -2811287356288318e-29, 4.62596163522878599135e-9, 4.49696789927706453732e-7, 1.49624783758342370182e-5, 2.09386317487588078668e-4, 1.05628862152492910091e-3, -0.0011295143874558028, -0.016743100507663373];
const Q5 = [2.82243172016108031869e-7, 2.75335474764726041141e-5, 9.64011807005165528527e-4, 1.60746087093676504695e-2, 1.38151865749083321638e-1, 5.91429344886417493481e-1, 1];
/**
 * Polyval.
 * @param c - Array of Number.
 * @param x - Number.
 * @returns Number.
 */
function polyval(c, x) {
  let p = 0;
  for (const coef of c) {
    p = p * x + coef;
  }
  return p;
}
/**
 * Calculates a rational approximation.
 * @private
 * @param x - Number.
 * @param v - Number.
 * @param P - Array of polynomial coefficients.
 * @param Q - Array of polynomial coefficients.
 * @param Y - Number.
 * @returns Rational approximation.
 */
function calc(x, v, P, Q, Y) {
  const s = x - v;
  const r = polyval(P, s) / polyval(Q, s);
  return Y * x + r * x;
}
/**
 * Evaluates the complementary inverse error function for an input value.
 * @private
 * @param x - Input value.
 * @returns Evaluated complementary inverse error function.
 */
function erfcinv(x) {
  let sign = false;
  let val;
  let q;
  let g;
  let r;
  // [1] Special cases...
  // NaN:
  if (Number.isNaN(x)) {
    return Number.NaN;
  }
  // x not on the interval: [0,2]
  if (x < 0 || x > 2) {
    throw new RangeError(`erfcinv()::invalid input argument. Value must be on the interval [0,2]. Value: \`${x}\`.`);
  }
  if (x === 0) {
    return Number.POSITIVE_INFINITY;
  }
  if (x === 2) {
    return Number.NEGATIVE_INFINITY;
  }
  if (x === 1) {
    return 0;
  }
  // [2] Get the sign and make use of `erfc` reflection formula: `erfc(-z)=2 - erfc(z)`...
  if (x > 1) {
    q = 2 - x;
    x = 1 - q;
    sign = true;
  } else {
    q = x;
    x = 1 - x;
  }
  // [3] |x| <= 0.5
  if (x <= 0.5) {
    g = x * (x + 10);
    r = polyval(P1, x) / polyval(Q1, x);
    val = g * Y1 + g * r;
    return sign ? -val : val;
  }
  // [4] 1-|x| >= 0.25
  if (q >= 0.25) {
    g = Math.sqrt(-2 * Math.log(q));
    q = q - 0.25;
    r = polyval(P2, q) / polyval(Q2, q);
    val = g / (Y2 + r);
    return sign ? -val : val;
  }
  q = Math.sqrt(-Math.log(q));
  // [5] q < 3
  if (q < 3) {
    return calc(q, 1.125, P3, Q3, Y3);
  }
  // [6] q < 6
  if (q < 6) {
    return calc(q, 3, P4, Q4, Y4);
  }
  // Note that the smallest number in JavaScript is 5e-324. Math.sqrt( -Math.log( 5e-324 ) ) ~27.2844
  return calc(q, 6, P5, Q5, Y5);
  // Note that in the boost library, they are able to go to much smaller values, as 128 bit long doubles support ~1e-5000; something which JavaScript does not natively support.
}

/**
 * Applies a simple normalization inverse transformation to the input data.
 * @param data - The input array of numbers to be transformed.
 * @param options - Optional parameters for the transformation.
 * @returns A new Float64Array containing the transformed data.
 */
function simpleNormInv(data, options = {}) {
  const {
    magnitudeMode = false
  } = options;
  const result = new Float64Array(data.length);
  if (magnitudeMode) {
    for (let i = 0; i < result.length; i++) {
      result[i] = -Math.sqrt(-2 * Math.log(1 - data[i]));
    }
  } else {
    for (let i = 0; i < result.length; i++) {
      result[i] = -1 * Math.SQRT2 * erfcinv(2 * data[i]);
    }
  }
  return result;
}
/**
 * Convenience wrapper for single number processing by simpleNormInv function.
 * @param data - The number to be normalized.
 * @param options - The options for the normalization process.
 * @returns The normalized number.
 */
function simpleNormInvNumber(data, options = {}) {
  return simpleNormInv([data], options)[0];
}

/**
 * Determine noise level by san plot methodology (https://doi.org/10.1002/mrc.4882)
 * @param array - real or magnitude spectra data.
 * @param options - options
 * @returns noise level
 */
function xNoiseSanPlot(array, options = {}) {
  const {
    mask,
    cutOff,
    refine = true,
    magnitudeMode = false,
    scaleFactor = 1,
    factorStd = 5,
    fixOffset = true
  } = options;
  const input = prepareData(array, {
    scaleFactor,
    mask
  });
  if (fixOffset && !magnitudeMode) {
    const medianIndex = Math.floor(input.length / 2);
    const median = input.length % 2 === 0 ? 0.5 * (input[medianIndex - 1] + input[medianIndex]) : input[medianIndex];
    for (let i = 0; i < input.length; i++) {
      input[i] -= median;
    }
  }
  const firstNegativeValueIndex = input.at(-1) >= 0 ? input.length : input.findIndex(e => e < 0);
  let lastPositiveValueIndex = firstNegativeValueIndex - 1;
  for (let i = lastPositiveValueIndex; i >= 0; i--) {
    if (input[i] > 0) {
      lastPositiveValueIndex = i;
      break;
    }
  }
  const signPositive = input.slice(0, lastPositiveValueIndex + 1);
  const signNegative = input.slice(firstNegativeValueIndex);
  const cutOffDist = cutOff || determineCutOff(signPositive, {
    magnitudeMode
  });
  const pIndex = Math.floor(signPositive.length * cutOffDist);
  const initialNoiseLevelPositive = signPositive[pIndex];
  const skyPoint = signPositive[0];
  let initialNoiseLevelNegative;
  if (signNegative.length > 0) {
    const nIndex = Math.floor(signNegative.length * (1 - cutOffDist));
    initialNoiseLevelNegative = -1 * signNegative[nIndex];
  } else {
    initialNoiseLevelNegative = 0;
  }
  let noiseLevelPositive = initialNoiseLevelPositive;
  let noiseLevelNegative = initialNoiseLevelNegative;
  let cloneSignPositive = signPositive.slice();
  let cloneSignNegative = signNegative.slice();
  let cutOffSignalsIndexPlus = 0;
  let cutOffSignalsIndexNeg = 2;
  if (refine) {
    let cutOffSignals = noiseLevelPositive * factorStd;
    cutOffSignalsIndexPlus = signPositive.findIndex(e => e < cutOffSignals);
    if (cutOffSignalsIndexPlus > -1) {
      cloneSignPositive = signPositive.slice(cutOffSignalsIndexPlus);
      noiseLevelPositive = cloneSignPositive[Math.floor(cloneSignPositive.length * cutOffDist)];
    }
    cutOffSignals = noiseLevelNegative * factorStd;
    cutOffSignalsIndexNeg = signNegative.findIndex(e => e < cutOffSignals);
    if (cutOffSignalsIndexNeg > -1) {
      cloneSignNegative = signNegative.slice(cutOffSignalsIndexNeg);
      noiseLevelNegative = cloneSignPositive[Math.floor(cloneSignNegative.length * (1 - cutOffDist))];
    }
  }
  const correctionFactor = -simpleNormInvNumber(cutOffDist / 2, {
    magnitudeMode
  });
  let effectiveCutOffDist, refinedCorrectionFactor;
  if (refine && cutOffSignalsIndexPlus > -1) {
    effectiveCutOffDist = (cutOffDist * cloneSignPositive.length + cutOffSignalsIndexPlus) / (cloneSignPositive.length + cutOffSignalsIndexPlus);
    refinedCorrectionFactor = -1 * simpleNormInvNumber(effectiveCutOffDist / 2, {
      magnitudeMode
    });
    noiseLevelPositive /= refinedCorrectionFactor;
    if (cutOffSignalsIndexNeg > -1) {
      effectiveCutOffDist = (cutOffDist * cloneSignNegative.length + cutOffSignalsIndexNeg) / (cloneSignNegative.length + cutOffSignalsIndexNeg);
      refinedCorrectionFactor = -1 * simpleNormInvNumber(effectiveCutOffDist / 2, {
        magnitudeMode
      });
      if (noiseLevelNegative !== 0) {
        noiseLevelNegative /= refinedCorrectionFactor;
      }
    }
  } else {
    noiseLevelPositive /= correctionFactor;
    noiseLevelNegative /= correctionFactor;
  }
  return {
    positive: noiseLevelPositive,
    negative: noiseLevelNegative,
    snr: skyPoint / noiseLevelPositive,
    sanplot: generateSanPlot(input, {
      fromTo: {
        positive: {
          from: 0,
          to: lastPositiveValueIndex
        },
        negative: {
          from: firstNegativeValueIndex,
          to: input.length
        }
      }
    })
  };
}
/**
 * Determines the optimal cut-off point for a given array of positive numbers.
 * @param signPositive - An array of positive numbers.
 * @param options - Optional parameters to configure the cut-off determination.
 * @param options.magnitudeMode - If true, uses magnitude mode for normalization. Default is false.
 * @param options.considerList - An object specifying the range and step for consideration.
 * @param options.considerList.from - The starting point of the range. Default is 0.5.
 * @param options.considerList.step - The step size for the range. Default is 0.1.
 * @param options.considerList.to - The ending point of the range. Default is 0.9.
 * @returns The optimal cut-off point as a number.
 */
function determineCutOff(signPositive, options = {}) {
  const {
    magnitudeMode = false,
    considerList = {
      from: 0.5,
      step: 0.1,
      to: 0.9
    }
  } = options;
  //generate a list of values for
  const cutOff = [];
  const indexMax = signPositive.length - 1;
  for (let i = 0.01; i <= 0.99; i += 0.01) {
    const index = Math.round(indexMax * i);
    const value = -signPositive[index] / simpleNormInvNumber(i / 2, {
      magnitudeMode
    });
    cutOff.push([i, value]);
  }
  let minKi = Number.MAX_SAFE_INTEGER;
  const {
    from,
    to,
    step
  } = considerList;
  const delta = step / 2;
  let whereToCutStat = 0.5;
  for (let i = from; i <= to; i += step) {
    const floor = i - delta;
    const top = i + delta;
    const elementsOfCutOff = cutOff.filter(e => e[0] < top && e[0] > floor);
    let averageValue = 0;
    for (const element of elementsOfCutOff) {
      averageValue += Math.abs(element[1]);
    }
    let kiSqrt = 0;
    for (const element of elementsOfCutOff) {
      kiSqrt += (element[1] - averageValue) ** 2;
    }
    if (kiSqrt < minKi) {
      minKi = kiSqrt;
      whereToCutStat = i;
    }
  }
  return whereToCutStat;
}
/**
 * Generates a SAN plot from the given array based on the specified options.
 * @param array - The input array of numbers to be processed.
 * @param options - An optional object containing configuration options.
 * @param options.logBaseY - The logarithmic base for the Y-axis. Defaults to 2.
 * @param options.fromTo - An object specifying the range for each key. Each key maps to an object with `from` and `to` properties.
 * @returns An object where each key maps to a DataXY object containing the processed data.
 */
function generateSanPlot(array, options = {}) {
  const {
    fromTo,
    logBaseY = 2
  } = options;
  const sanplot = {};
  for (const key in fromTo) {
    const {
      from,
      to
    } = fromTo[key];
    sanplot[key] = from !== to ? scale$1(array.slice(from, to), {
      logBaseY
    }) : {
      x: [],
      y: []
    };
    if (key === 'negative') {
      sanplot[key].y.reverse();
    }
  }
  return sanplot;
}
/**
 * Scales the input array based on the provided options.
 * @param array - The input array to be scaled.
 * @param options - An optional object containing scaling options.
 * @param options.logBaseY - If provided, the array values will be scaled using the logarithm of this base.
 * @returns An object containing the scaled x and y arrays.
 */
function scale$1(array, options = {}) {
  const {
    log10,
    abs
  } = Math;
  const {
    logBaseY
  } = options;
  if (logBaseY) {
    array = array.slice(0);
    const logOfBase = log10(logBaseY);
    for (let i = 0; i < array.length; i++) {
      array[i] = log10(abs(array[i])) / logOfBase;
    }
  }
  const xAxis = createFromToArray({
    from: 0,
    to: array.length - 1,
    length: array.length
  });
  return {
    x: xAxis,
    y: array
  };
}
/**
 * Prepares and processes the input data array based on the provided options.
 * @param array - The input array of numbers to be processed.
 * @param options - An object containing the following properties:
 *   - scaleFactor: A number by which to scale each element of the array.
 *   - mask: An optional array of the same length as the input array, where
 *           elements corresponding to `true` values will be excluded from processing.
 * @param options.scaleFactor
 * @param options.mask
 * @returns A new Float64Array containing the processed data, scaled by the
 *          scaleFactor and sorted in descending order.
 */
function prepareData(array, options) {
  const {
    scaleFactor,
    mask
  } = options;
  const input = xEnsureFloat64(isAnyArray$1(mask) && mask.length === array.length ? array.filter((_e, i) => !mask[i]) : array);
  if (scaleFactor > 1) {
    for (let i = 0; i < input.length; i++) {
      input[i] *= scaleFactor;
    }
  }
  input.sort();
  input.reverse();
  return input;
}

/**
 * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
 * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
 * This assumes a gaussian distribution of the noise
 * @param array
 * @returns noise level corresponding to one standard deviation
 */
function xNoiseStandardDeviation(array) {
  const {
    mad,
    median
  } = xMedianAbsoluteDeviation(array);
  return {
    sd: mad / 0.6744897501960817,
    mad,
    median
  };
}

/**
 * Calculate the sum of the values
 * @param array - Object that contains property x (an ordered increasing array) and y (an array).
 * @param options - Options.
 * @returns XSum value on the specified range.
 */
function xSum(array, options = {}) {
  xCheck(array);
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(array, options);
  let sumValue = array[fromIndex];
  for (let i = fromIndex + 1; i <= toIndex; i++) {
    sumValue += array[i];
  }
  return sumValue;
}

/**
 * Divides the data with either the sum, the absolute sum or the maximum of the data
 * @param input - Array containing values
 * @param options - options
 * @returns - normalized data
 */
function xNormed(input, options = {}) {
  const {
    algorithm = 'absolute',
    value = 1
  } = options;
  xCheck(input);
  const output = getOutputArray(options.output, input.length);
  if (input.length === 0) {
    throw new Error('input must not be empty');
  }
  switch (algorithm) {
    case 'absolute':
      {
        const absoluteSumValue = absoluteSum(input) / value;
        if (absoluteSumValue === 0) {
          throw new Error('trying to divide by 0');
        }
        for (let i = 0; i < input.length; i++) {
          output[i] = input[i] / absoluteSumValue;
        }
        return output;
      }
    case 'max':
      {
        const currentMaxValue = xMaxValue(input);
        if (currentMaxValue === 0) {
          throw new Error('trying to divide by 0');
        }
        const factor = value / currentMaxValue;
        for (let i = 0; i < input.length; i++) {
          output[i] = input[i] * factor;
        }
        return output;
      }
    case 'sum':
      {
        const sumFactor = xSum(input) / value;
        if (sumFactor === 0) {
          throw new Error('trying to divide by 0');
        }
        for (let i = 0; i < input.length; i++) {
          output[i] = input[i] / sumFactor;
        }
        return output;
      }
    default:
      throw new Error(`unknown algorithm: ${String(algorithm)}`);
  }
}
function absoluteSum(input) {
  let sumValue = 0;
  for (const value of input) {
    sumValue += Math.abs(value);
  }
  return sumValue;
}

/**
 * This function pads an array
 *s
 * @param array - the array that will be padded
 * @param options - options
 */
function xPadding(array, options = {}) {
  const {
    size = 0,
    value = 0,
    algorithm
  } = options;
  xCheck(array);
  if (!algorithm) {
    return xEnsureFloat64(array);
  }
  const result = new Float64Array(array.length + size * 2);
  for (let i = 0; i < array.length; i++) {
    result[i + size] = array[i];
  }
  const fromEnd = size + array.length;
  const toEnd = 2 * size + array.length;
  switch (algorithm) {
    case 'value':
      for (let i = 0; i < size; i++) {
        result[i] = value;
      }
      for (let i = fromEnd; i < toEnd; i++) {
        result[i] = value;
      }
      break;
    case 'duplicate':
      for (let i = 0; i < size; i++) {
        result[i] = array[0];
      }
      for (let i = fromEnd; i < toEnd; i++) {
        result[i] = array.at(-1);
      }
      break;
    case 'circular':
      for (let i = 0; i < size; i++) {
        result[i] = array[(array.length - size % array.length + i) % array.length];
      }
      for (let i = 0; i < size; i++) {
        result[i + fromEnd] = array[i % array.length];
      }
      break;
    default:
      throw new Error(`unknown algorithm ${String(algorithm)}`);
  }
  return result;
}

/**
 * Pareto scaling, which uses the square root of standard deviation as the scaling factor, circumvents the amplification of noise by retaining a small portion of magnitude information.
 * Noda, I. (2008). Scaling techniques to enhance two-dimensional correlation spectra. Journal of Molecular Structure, 883, 216-227.
 * DOI: 10.1016/j.molstruc.2007.12.026
 * @param array - array of number
 */
function xParetoNormalization(array) {
  xCheck(array);
  const result = new Float64Array(array.length);
  const sqrtSD = Math.sqrt(xStandardDeviation(array));
  for (let i = 0; i < array.length; i++) {
    result[i] = array[i] / sqrtSD;
  }
  return result;
}

/**
 * Function used to rescale data
 * @param input - input for the rescale
 * @param options - options
 * @returns rescaled data
 */
function xRescale(input, options = {}) {
  xCheck(input);
  const output = getOutputArray(options.output, input.length);
  const currentMin = xMinValue(input);
  const currentMax = xMaxValue(input);
  if (currentMin === currentMax) {
    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
  }
  const {
    min = 0,
    max = 1
  } = options;
  if (min >= max) {
    throw new RangeError('min option must be smaller than max option');
  }
  const factor = (max - min) / (currentMax - currentMin);
  for (let i = 0; i < input.length; i++) {
    output[i] = (input[i] - currentMin) * factor + min;
  }
  return output;
}

/**
 * This function calculates a rolling average
 * @param array - array
 * @param fct - callback function that from an array returns a value
 * @param options - options
 */
function xRolling(array, fct, options = {}) {
  xCheck(array);
  if (typeof fct !== 'function') {
    throw new TypeError('fct must be a function');
  }
  const {
    window = 5,
    padding = {}
  } = options;
  const {
    size = window - 1,
    algorithm,
    value
  } = padding;
  const padded = xPadding(array, {
    size,
    algorithm,
    value
  }); // ensure we get a copy and it is float64
  const newArray = [];
  for (let i = 0; i < padded.length - window + 1; i++) {
    // we will send a view to the original buffer
    newArray.push(fct(padded.subarray(i, i + window)));
  }
  return newArray;
}

/**
 * This function calculates a rolling average
 * @param array - array
 * @param options - option
 */
function xRollingAverage(array, options = {}) {
  return xRolling(array, xMean, options);
}

/**
 * This function calculates a rolling average
 * @param array - array
 * @param options - options
 */
function xRollingMedian(array, options = {}) {
  return xRolling(array, xMedian, options);
}

/**
 * Create a new matrix based on the size of the current one or by using specific dimensions.
 * @param options
 */
function matrixCreateEmpty(options) {
  const {
    matrix,
    nbRows = matrix?.length || 1,
    nbColumns = matrix?.[0].length || 1,
    ArrayConstructor = Float64Array
  } = options;
  const newMatrix = [];
  for (let row = 0; row < nbRows; row++) {
    newMatrix.push(createNumberArray(ArrayConstructor, nbColumns));
  }
  return newMatrix;
}

function matrixAutoCorrelation(matrix, index = 0) {
  const nbRows = matrix.length;
  const nbColumns = matrix[0].length;
  if (nbRows < 2) {
    throw new Error('can not calculate info if matrix contains less than 2 rows');
  }
  const targetArray = new Float64Array(nbRows);
  const sourceArray = new Float64Array(nbRows);
  const result = new Float64Array(nbColumns);
  for (let j = 0; j < nbRows; j++) {
    targetArray[j] = matrix[j][index];
  }
  for (let i = 0; i < nbColumns; i++) {
    for (let j = 0; j < nbRows; j++) {
      sourceArray[j] = matrix[j][i];
    }
    result[i] = xCorrelation(targetArray, sourceArray);
  }
  return result;
}

function matrixBoxPlot(matrix) {
  const nbRows = matrix.length;
  const nbColumns = matrix[0].length;
  if (nbRows < 5) {
    throw new Error('can not calculate info if matrix contains less than 5 rows');
  }
  const info = {
    q1: new Float64Array(nbColumns),
    median: new Float64Array(nbColumns),
    q3: new Float64Array(nbColumns),
    min: Float64Array.from(matrix[0]),
    max: Float64Array.from(matrix.at(-1))
  };
  const columnArray = new Float64Array(matrix.length);
  for (let column = 0; column < nbColumns; column++) {
    for (let row = 0; row < nbRows; row++) {
      columnArray[row] = matrix[row][column];
    }
    let q1max = 0;
    let q3min = 0;
    if (nbRows % 2 === 1) {
      // odd
      const middle = (nbRows - 1) / 2;
      info.median[column] = columnArray[middle];
      q1max = middle - 1;
      q3min = middle + 1;
    } else {
      // even
      q3min = nbRows / 2;
      q1max = q3min - 1;
      info.median[column] = (columnArray[q1max] + columnArray[q3min]) / 2;
    }
    if (q1max % 2 === 0) {
      info.q1[column] = columnArray[q1max / 2];
      info.q3[column] = columnArray[(nbRows + q3min - 1) / 2];
    } else {
      info.q1[column] = (columnArray[(q1max + 1) / 2] + columnArray[(q1max - 1) / 2]) / 2;
      const middleOver = (columnArray.length + q3min) / 2;
      info.q3[column] = (columnArray[middleOver] + columnArray[middleOver - 1]) / 2;
    }
  }
  return info;
}

/**
 * Center mean of matrix columns.
 * @param matrix - matrix [rows][cols]
 */
function matrixCenterZMean(matrix) {
  const nbColumns = matrix[0].length;
  const nbRows = matrix.length;
  const newMatrix = matrixCreateEmpty({
    nbColumns,
    nbRows
  });
  for (let column = 0; column < nbColumns; column++) {
    let mean = 0;
    for (let row = 0; row < nbRows; row++) {
      mean += matrix[row][column];
    }
    mean /= nbRows;
    for (let row = 0; row < nbRows; row++) {
      newMatrix[row][column] = matrix[row][column] - mean;
    }
  }
  return newMatrix;
}

function matrixCheck(data) {
  if (data.length === 0 || data[0].length === 0) {
    throw new RangeError('matrix must contain data');
  }
  const firstLength = data[0].length;
  for (let i = 1; i < data.length; i++) {
    if (data[i].length !== firstLength) {
      throw new RangeError('all rows must has the same length');
    }
  }
}

/**
 * Get min and max Z.
 * @param matrix - matrix [rows][cols].
 */
function matrixMinMaxZ(matrix) {
  matrixCheck(matrix);
  const nbRows = matrix.length;
  const nbColumns = matrix[0].length;
  let min = matrix[0][0];
  let max = matrix[0][0];
  for (let column = 0; column < nbColumns; column++) {
    for (let row = 0; row < nbRows; row++) {
      if (matrix[row][column] < min) min = matrix[row][column];
      if (matrix[row][column] > max) max = matrix[row][column];
    }
  }
  return {
    min,
    max
  };
}

/**
 * Performs a Probabilistic quotient normalization (PQN) over the dataset to account dilution based in median spectrum.
 * Dieterle, F., Ross, A., Schlotterbeck, G., & Senn, H. (2006). Probabilistic quotient normalization as robust method to account for dilution of complex biological mixtures. Application in 1H NMR metabonomics. Analytical chemistry, 78(13), 4281-4290.
 * DOI: 10.1021/ac051632c
 * @param matrix - matrix [rows][cols].
 * @param options - options
 * @returns - {data: Normalized dataset, medianOfQuotients: The median of quotients of each variables}
 */
function matrixPQN(matrix, options = {}) {
  const {
    max = 100
  } = options;
  const matrixB = new Matrix(matrix);
  for (let i = 0; i < matrixB.rows; i++) {
    const normalizationFactor = matrixB.getRowVector(i).norm('frobenius') / max;
    const row = matrixB.getRowVector(i).div(normalizationFactor);
    matrixB.setRow(i, row);
  }
  const referenceSpectrum = [];
  for (let i = 0; i < matrixB.columns; i++) {
    const currentVariable = matrixB.getColumn(i);
    referenceSpectrum.push(xMedian(currentVariable));
  }
  const medianOfQuotients = [];
  for (let i = 0; i < matrixB.columns; i++) {
    const quotients = matrixB.getColumnVector(i).div(referenceSpectrum[i]);
    medianOfQuotients.push(xMedian(quotients.getColumn(0)));
  }
  for (let i = 0; i < matrixB.rows; i++) {
    matrixB.mulRow(i, 1 / medianOfQuotients[i]);
  }
  return {
    data: matrixB.to2DArray(),
    medianOfQuotients
  };
}

/**
 * Rescale a matrix between min and max values.
 * @param matrix - matrix [rows][cols].
 * @param options - Options.
 */
function matrixZRescale(matrix, options = {}) {
  const {
    min = 0,
    max = 1,
    ArrayConstructor
  } = options;
  const nbColumns = matrix[0].length;
  const nbRows = matrix.length;
  const newMatrix = matrixCreateEmpty({
    nbColumns,
    nbRows,
    ArrayConstructor
  });
  const {
    min: currentMin,
    max: currentMax
  } = matrixMinMaxZ(matrix);
  const factor = (max - min) / (currentMax - currentMin);
  for (let column = 0; column < nbColumns; column++) {
    for (let row = 0; row < nbRows; row++) {
      newMatrix[row][column] = (matrix[row][column] - currentMin) * factor + min;
    }
  }
  return newMatrix;
}

/**
 * Verify that `data` is an object of x,y arrays.
 * Throws an error if it's not.
 * @param data
 * @param options
 * @param options.minLength
 */
function xyCheck(data, options = {}) {
  const {
    minLength
  } = options;
  if (data === null || typeof data !== 'object' ||
  // @ts-expect-error Typechecking
  !isAnyArray$1(data.x) ||
  // @ts-expect-error Typechecking
  !isAnyArray$1(data.y)) {
    throw new Error('data must be an object of x and y arrays');
  }
  // @ts-expect-error Typechecking
  if (data.x.length !== data.y.length) {
    throw new Error('the x and y arrays must have the same length');
  }
  // @ts-expect-error Typechecking
  if (minLength && data.x.length < minLength) {
    throw new Error(`data.x must have a length of at least ${minLength}`);
  }
}

/**
 * Finds the max y value in a range and return a {x,y} point
 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
 * @param options - Options
 */
function xyMaxYPoint(data, options = {}) {
  xyCheck(data, {
    minLength: 1
  });
  const {
    x,
    y
  } = data;
  if (x.length === 1) {
    return {
      x: x[0],
      y: y[0],
      index: 0
    };
  }
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(x, options);
  let current = {
    x: x[fromIndex],
    y: y[fromIndex],
    index: fromIndex
  };
  for (let i = fromIndex; i <= toIndex; i++) {
    if (y[i] > current.y) current = {
      x: x[i],
      y: y[i],
      index: i
    };
  }
  return current;
}

/**
 * Filters x,y values to allow strictly growing values in x-axis.
 * @param data - Object that contains property x (an ordered increasing array) and y (an array).
 */
function xyEnsureGrowingX(data) {
  xyCheck(data);
  if (xIsMonotonic(data.x) === 1) return data;
  const x = Array.from(data.x);
  const y = Array.from(data.y);
  let prevX = Number.NEGATIVE_INFINITY;
  let currentIndex = 0;
  for (let index = 0; index < x.length; index++) {
    if (prevX < x[index]) {
      if (currentIndex < index) {
        x[currentIndex] = x[index];
        y[currentIndex] = y[index];
      }
      currentIndex++;
      prevX = x[index];
    }
  }
  x.length = currentIndex;
  y.length = currentIndex;
  return {
    x,
    y
  };
}

/**
 * Normalize an array of zones:
 * - ensure than from < to
 * - merge overlapping zones
 * - deal with exclusions zones
 * - if no zones is specified add one between -Infinity and +Infinity
 * @param zones - array of zones
 * @param options - options
 * @returns array of zones
 */
function zonesNormalize(zones = [], options = {}) {
  const {
    exclusions = []
  } = options;
  let {
    from = Number.NEGATIVE_INFINITY,
    to = Number.POSITIVE_INFINITY
  } = options;
  if (from > to) [from, to] = [to, from];
  zones = zones.map(zone => zone.from > zone.to ? {
    from: zone.to,
    to: zone.from
  } : {
    ...zone
  });
  zones.sort((a, b) => {
    if (a.from !== b.from) return a.from - b.from;
    return a.to - b.to;
  });
  if (zones.length === 0) {
    zones.push({
      from,
      to
    });
  }
  for (const zone of zones) {
    if (from > zone.from) zone.from = from;
    if (to < zone.to) zone.to = to;
  }
  zones = zones.filter(zone => zone.from <= zone.to);
  if (zones.length === 0) return [];
  let currentZone = zones[0];
  const beforeExclusionsZones = [currentZone];
  for (let i = 1; i < zones.length; i++) {
    const zone = zones[i];
    if (zone.from <= currentZone.to) {
      if (currentZone.to < zone.to) {
        currentZone.to = zone.to;
      }
    } else {
      currentZone = zone;
      beforeExclusionsZones.push(currentZone);
    }
  }
  if (exclusions.length === 0) return beforeExclusionsZones;
  const normalizedExclusions = zonesNormalize(exclusions);
  let currentExclusionIndex = 0;
  const results = [];
  for (let zoneIndex = 0; zoneIndex < beforeExclusionsZones.length; zoneIndex++) {
    const zone = beforeExclusionsZones[zoneIndex];
    if (currentExclusionIndex === normalizedExclusions.length) {
      // we analysed all the exclusion zones
      results.push(zone);
      continue;
    }
    while (currentExclusionIndex < normalizedExclusions.length && normalizedExclusions[currentExclusionIndex].to <= zone.from) {
      currentExclusionIndex++;
    }
    if (currentExclusionIndex === normalizedExclusions.length) {
      // we analysed all the exclusion zones
      results.push(zone);
      continue;
    }
    if (zone.to < normalizedExclusions[currentExclusionIndex].from) {
      // no problems, not yet in exclusion
      results.push(zone);
      continue;
    }
    if (normalizedExclusions[currentExclusionIndex].to >= zone.to) {
      // could be totally excluded
      if (normalizedExclusions[currentExclusionIndex].from <= zone.from) {
        continue;
      }
      results.push({
        from: normalizedExclusions[currentExclusionIndex].to,
        to: zone.to
      });
    }
    // we cut in the middle, we need to create more zones, annoying !
    if (normalizedExclusions[currentExclusionIndex].from > zone.from) {
      results.push({
        from: zone.from,
        to: normalizedExclusions[currentExclusionIndex].from
      });
    }
    zone.from = normalizedExclusions[currentExclusionIndex].to;
    zoneIndex--;
  }
  return results;
}

/**
 * Add the number of points per zone to reach a specified total
 * @param zones - array of zones
 * @param numberOfPoints - total number of points to distribute between zones
 * @param options - options
 * @returns array of zones with points
 */
function zonesWithPoints(zones = [],
/**
 * total number of points to distribute between zones
 * @default 10
 */
numberOfPoints = 10, options = {}) {
  if (zones.length === 0) return [];
  const normalizedZones = zonesNormalize(zones, options);
  const zonesWithNumberOfPoints = [];
  const totalSize = normalizedZones.reduce((previous, current) => {
    return previous + (current.to - current.from);
  }, 0);
  const unitsPerPoint = totalSize / numberOfPoints;
  let currentTotal = 0;
  for (let i = 0; i < normalizedZones.length - 1; i++) {
    const tempZone = normalizedZones[i];
    const tempZoneNumberOfPoints = Math.min(Math.round((tempZone.to - tempZone.from) / unitsPerPoint), numberOfPoints - currentTotal);
    zonesWithNumberOfPoints.push({
      ...tempZone,
      numberOfPoints: tempZoneNumberOfPoints
    });
    currentTotal += tempZoneNumberOfPoints;
  }
  zonesWithNumberOfPoints.push({
    ...normalizedZones.at(-1),
    numberOfPoints: numberOfPoints - currentTotal
  });
  return zonesWithNumberOfPoints;
}

/**
 * Function that retrieves the getEquallySpacedData with the variant "slot".
 * @param x
 * @param y
 * @param from
 * @param to
 * @param numberOfPoints
 * @returns Array of y's equally spaced with the variant "slot"
 */
function equallySpacedSlot(/** x coordinates */
x, /** y coordinates */
y, /** from value */
from, /** to value */
to, /** number of points */
numberOfPoints) {
  const xLength = x.length;
  if (xLength < 2) {
    return Float64Array.from(x);
  }
  const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
  const halfStep = step / 2;
  const lastStep = x.at(-1) - x.at(-2);
  const start = from - halfStep;
  // Changed Array to Float64Array
  const output = new Float64Array(numberOfPoints);
  // Init main variables
  let min = start;
  let max = start + step;
  let previousX = -Number.MAX_VALUE;
  let previousY = 0;
  let nextX = x[0];
  let nextY = y[0];
  let frontOutsideSpectra = 0;
  let backOutsideSpectra = true;
  let currentValue = 0;
  // for slot algorithm
  let currentPoints = 0;
  let i = 1; // index of input
  let j = 0; // index of output
  main: while (true) {
    if (previousX >= nextX) throw new Error('x must be a growing series');
    while (previousX - max > 0) {
      // no overlap with original point, just consume current value
      if (backOutsideSpectra) {
        currentPoints++;
        backOutsideSpectra = false;
      }
      output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
      j++;
      if (j === numberOfPoints) {
        break main;
      }
      min = max;
      max += step;
      currentValue = 0;
      currentPoints = 0;
    }
    if (previousX > min) {
      currentValue += previousY;
      currentPoints++;
    }
    if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
      currentPoints--;
    }
    previousX = nextX;
    previousY = nextY;
    if (i < xLength) {
      nextX = x[i];
      nextY = y[i];
      i++;
    } else {
      nextX += lastStep;
      nextY = 0;
      frontOutsideSpectra++;
    }
  }
  return output;
}

/**
 * Function that calculates the integral of the line between two
 * x-coordinates, given the slope and intercept of the line.
 * @param x0
 * @param x1
 * @param slope
 * @param intercept
 * @returns integral value.
 */
function integral(/** first coordinate of point */
x0, /** second coordinate of point */
x1, /** slope of the line */
slope, /** intercept of the line on the y axis */
intercept) {
  return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
}

/**
 * Function that retrieves the getEquallySpacedData with the variant "smooth".
 * @param x
 * @param y
 * @param from
 * @param to
 * @param numberOfPoints
 * @returns - Array of y's equally spaced with the variant "smooth"
 */
function equallySpacedSmooth(/** x coordinates */
x, /** y coordinates */
y, /** from value */
from, /** to value */
to, /** number of points */
numberOfPoints) {
  const xLength = x.length;
  const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
  const halfStep = step / 2;
  // Changed Array to Float64Array
  const output = new Float64Array(numberOfPoints);
  const initialOriginalStep = x[1] - x[0];
  const lastOriginalStep = x[xLength - 1] - x[xLength - 2];
  // Init main variables
  let min = from - halfStep;
  let max = from + halfStep;
  let previousX = Number.MIN_SAFE_INTEGER;
  let previousY = 0;
  let nextX = x[0] - initialOriginalStep;
  let nextY = 0;
  let currentValue = 0;
  let slope = 0;
  let intercept = 0;
  let sumAtMin = 0;
  let sumAtMax = 0;
  let i = 0; // index of input
  let j = 0; // index of output
  let add = 0;
  main: while (true) {
    if (previousX >= nextX) throw new Error('x must be a growing series');
    if (previousX <= min && min <= nextX) {
      add = integral(0, min - previousX, slope, previousY);
      sumAtMin = currentValue + add;
    }
    while (nextX - max >= 0) {
      // no overlap with original point, just consume current value
      add = integral(0, max - previousX, slope, previousY);
      sumAtMax = currentValue + add;
      output[j++] = (sumAtMax - sumAtMin) / step;
      if (j === numberOfPoints) {
        break main;
      }
      min = max;
      max += step;
      sumAtMin = sumAtMax;
    }
    currentValue += integral(previousX, nextX, slope, intercept);
    previousX = nextX;
    previousY = nextY;
    if (i < xLength) {
      nextX = x[i];
      nextY = y[i];
      i++;
    } else if (i === xLength) {
      nextX += lastOriginalStep;
      nextY = 0;
    }
    slope = getSlope(previousX, previousY, nextX, nextY);
    intercept = -slope * previousX + previousY;
  }
  return output;
}
function getSlope(x0, y0, x1, y1) {
  return (y1 - y0) / (x1 - x0);
}

/**
 * Function that returns a Number array of equally spaced numberOfPoints
 * containing a representation of intensities of the spectra arguments x
 * and y.
 *
 * The options parameter contains an object in the following form:
 * from: starting point
 * to: last point
 * numberOfPoints: number of points between from and to
 * variant: "slot" or "smooth" - smooth is the default option
 *
 * The slot variant consist that each point in an array is calculated
 * averaging the existing points between the slot that belongs to the current
 * value. The smooth variant is the same but takes the integral of the range
 * of the slot and divide by the step size between two points in an array.
 *
 * If exclusions zone are present, zones are ignored !
 * @param data - object containing 2 properties x and y
 * @param options - options
 * @returns new object with x / y array with the equally spaced data.
 */
function xyEquallySpaced(data, options = {}) {
  const {
    x,
    y
  } = data;
  const xLength = x.length;
  const {
    from = x[0],
    to = x[xLength - 1],
    variant = 'smooth',
    numberOfPoints = 100,
    exclusions = [],
    zones = [{
      from,
      to
    }]
  } = options;
  if (from > to) {
    throw new RangeError('from should be larger than to');
  }
  xyCheck(data);
  if (numberOfPoints < 2) {
    throw new RangeError("'numberOfPoints' option must be greater than 1");
  }
  const normalizedZones = zonesNormalize(zones, {
    from,
    to,
    exclusions
  });
  const zonesWithPointsRes = zonesWithPoints(normalizedZones, numberOfPoints, {
    from,
    to
  }).filter(zone => zone.numberOfPoints);
  let xResult = [];
  let yResult = [];
  for (const zone of zonesWithPointsRes) {
    if (!zone.numberOfPoints) {
      zone.numberOfPoints = 0;
    }
    const zoneResult = processZone(Array.from(x), Array.from(y), zone.from, zone.to, zone.numberOfPoints, variant);
    xResult = xResult.concat(zoneResult.x);
    yResult = yResult.concat(zoneResult.y);
  }
  return {
    x: xResult,
    y: yResult
  };
}
function processZone(x, y, from, to, numberOfPoints, variant) {
  if (numberOfPoints < 1) {
    throw new RangeError('the number of points must be at least 1');
  }
  const output = variant === 'slot' ? Array.from(equallySpacedSlot(x, y, from, to, numberOfPoints)) : Array.from(equallySpacedSmooth(x, y, from, to, numberOfPoints));
  return {
    x: Array.from(createFromToArray({
      from,
      to,
      length: numberOfPoints
    })),
    y: output
  };
}

/**
 * Filter an array x/y based on various criteria x points are expected to be sorted
 * @param data - object containing 2 properties x and y
 * @param options - options
 * @returns filtered array
 */
function xyFilterX(data, options = {}) {
  const {
    x,
    y
  } = data;
  if (x.length === 0) {
    return {
      x: Array.from(x),
      y: Array.from(y)
    };
  }
  const {
    from = x[0],
    to = x.at(-1),
    zones = [{
      from,
      to
    }],
    exclusions = []
  } = options;
  const normalizedZones = zonesNormalize(zones, {
    from,
    to,
    exclusions
  });
  let currentZoneIndex = 0;
  const newX = [];
  const newY = [];
  let position = 0;
  while (position < x.length) {
    if (x[position] <= normalizedZones[currentZoneIndex].to && x[position] >= normalizedZones[currentZoneIndex].from) {
      newX.push(x[position]);
      newY.push(y[position]);
    } else if (x[position] > normalizedZones[currentZoneIndex].to) {
      currentZoneIndex++;
      if (!normalizedZones[currentZoneIndex]) break;
    }
    position++;
  }
  return {
    x: newX,
    y: newY
  };
}

/**
 * Order object of array, x has to be monotone. Ensure x is growing
 * @param data - Object of kind {x:[], y:[]}.
 */
function xyGrowingX(data) {
  const {
    x,
    y
  } = data;
  if (x.length !== y.length) {
    throw new TypeError('length of x and y must be identical');
  }
  if (x.length < 2 || x[0] < x.at(-1)) return data;
  return {
    x: x.toReversed(),
    y: y.toReversed()
  };
}

/**
 * Calculate integration
 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
 * @param options - Options
 * @returns - xyIntegration value on the specified range
 */
function xyIntegration(data, options = {}) {
  xyCheck(data, {
    minLength: 1
  });
  const {
    x,
    y
  } = data;
  if (x.length === 1) return 0;
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(x, options);
  let currentxyIntegration = 0;
  for (let i = fromIndex; i < toIndex; i++) {
    currentxyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
  }
  return currentxyIntegration;
}

/**
 * Finds the min y value in a range and return a {x,y} point
 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
 * @param options - Options
 */
function xyMinYPoint(data, options = {}) {
  xyCheck(data, {
    minLength: 1
  });
  const {
    x,
    y
  } = data;
  if (x.length === 1) return {
    x: x[0],
    y: y[0],
    index: 0
  };
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(x, options);
  let current = {
    x: x[fromIndex],
    y: y[fromIndex],
    index: fromIndex
  };
  for (let i = fromIndex; i <= toIndex; i++) {
    if (y[i] < current.y) current = {
      x: x[i],
      y: y[i],
      index: i
    };
  }
  return current;
}

/**
 * This function performs a quick sort of the x array while transforming the y array to preserve the coordinates.
 * @param data - Object that contains property x (Array) and y (Array)
 */
function xySortX(data) {
  const {
    x,
    y
  } = data;
  if (xIsMonotonic(x) && x.length > 1) {
    const floatX = Float64Array.from(x);
    const floatY = Float64Array.from(y);
    if (floatX.at(-1) < floatX[0]) {
      floatX.reverse();
      floatY.reverse();
    }
    return {
      x: floatX,
      y: floatY
    };
  }
  const xyObject = Array.from(x, (val, index) => ({
    x: val,
    y: y[index]
  }));
  xyObject.sort((a, b) => a.x - b.x);
  const response = {
    x: new Float64Array(x.length),
    y: new Float64Array(y.length)
  };
  for (let i = 0; i < x.length; i++) {
    response.x[i] = xyObject[i].x;
    response.y[i] = xyObject[i].y;
  }
  return response;
}

/**
 * Ensure x values are unique
 * @param data - Object that contains property x (Array) and y (Array)
 * @param options - Object containing a property algorithm (can be 'sum' or 'average', the latter being the default value), and a property isSorted (boolean indicating if the x-array is sorted).
 */
function xyUniqueX(data, options = {}) {
  xyCheck(data);
  if (data.x.length === 0) {
    return {
      x: [],
      y: []
    };
  }
  const {
    algorithm = 'average',
    isSorted = true
  } = options;
  if (!isSorted) {
    data = xySortX(data);
  }
  switch (algorithm) {
    case 'average':
      return average$1(data);
    case 'sum':
      return sum(data);
    default:
      throw new Error(`unknown algorithm: ${String(algorithm)}`);
  }
}
/**
 * Average.
 * @param data - Input.
 * @returns Result.
 */
function average$1(data) {
  const x = [];
  const y = [];
  let cumulativeY = data.y[0];
  let divider = 1;
  for (let i = 1; i < data.x.length; i++) {
    if (!(data.x[i] === data.x[i - 1])) {
      x.push(data.x[i - 1]);
      y.push(cumulativeY / divider);
      cumulativeY = 0;
      divider = 0;
    }
    cumulativeY += data.y[i];
    divider++;
  }
  x.push(data.x.at(-1));
  y.push(cumulativeY / divider);
  return {
    x,
    y
  };
}
/**
 * Sum.
 * @param data - Input.
 * @returns Result.
 */
function sum(data) {
  const x = [];
  const y = [];
  let cumulativeY = data.y[0];
  for (let i = 1; i < data.x.length; i++) {
    if (!(data.x[i] === data.x[i - 1])) {
      x.push(data.x[i - 1]);
      y.push(cumulativeY);
      cumulativeY = 0;
    }
    cumulativeY += data.y[i];
  }
  x.push(data.x.at(-1));
  y.push(cumulativeY);
  return {
    x,
    y
  };
}

const {
  min: min$5,
  max: max$5
} = Math;
var limit = (x, low = 0, high = 1) => {
  return min$5(max$5(low, x), high);
};

var clip_rgb = rgb => {
  rgb._clipped = false;
  rgb._unclipped = rgb.slice(0);
  for (let i = 0; i <= 3; i++) {
    if (i < 3) {
      if (rgb[i] < 0 || rgb[i] > 255) rgb._clipped = true;
      rgb[i] = limit(rgb[i], 0, 255);
    } else if (i === 3) {
      rgb[i] = limit(rgb[i], 0, 1);
    }
  }
  return rgb;
};

// ported from jQuery's $.type
const classToType = {};
for (let name of ['Boolean', 'Number', 'String', 'Function', 'Array', 'Date', 'RegExp', 'Undefined', 'Null']) {
  classToType[`[object ${name}]`] = name.toLowerCase();
}
function type (obj) {
  return classToType[Object.prototype.toString.call(obj)] || 'object';
}

var unpack = (args, keyOrder = null) => {
  // if called with more than 3 arguments, we return the arguments
  if (args.length >= 3) return Array.prototype.slice.call(args);
  // with less than 3 args we check if first arg is object
  // and use the keyOrder string to extract and sort properties
  if (type(args[0]) == 'object' && keyOrder) {
    return keyOrder.split('').filter(k => args[0][k] !== undefined).map(k => args[0][k]);
  }
  // otherwise we just return the first argument
  // (which we suppose is an array of args)
  return args[0].slice(0);
};

var last = args => {
  if (args.length < 2) return null;
  const l = args.length - 1;
  if (type(args[l]) == 'string') return args[l].toLowerCase();
  return null;
};

const {
  PI: PI$2,
  min: min$4,
  max: max$4
} = Math;
const rnd2 = a => Math.round(a * 100) / 100;
const rnd3 = a => Math.round(a * 100) / 100;
const TWOPI = PI$2 * 2;
const PITHIRD = PI$2 / 3;
const DEG2RAD = PI$2 / 180;
const RAD2DEG = 180 / PI$2;

/**
 * Reverse the first three elements of an array
 *
 * @param {any[]} arr
 * @returns {any[]}
 */
function reverse3(arr) {
  return [...arr.slice(0, 3).reverse(), ...arr.slice(3)];
}

var input = {
  format: {},
  autodetect: []
};

class Color {
  constructor(...args) {
    const me = this;
    if (type(args[0]) === 'object' && args[0].constructor && args[0].constructor === this.constructor) {
      // the argument is already a Color instance
      return args[0];
    }
    // last argument could be the mode
    let mode = last(args);
    let autodetect = false;
    if (!mode) {
      autodetect = true;
      if (!input.sorted) {
        input.autodetect = input.autodetect.sort((a, b) => b.p - a.p);
        input.sorted = true;
      }

      // auto-detect format
      for (let chk of input.autodetect) {
        mode = chk.test(...args);
        if (mode) break;
      }
    }
    if (input.format[mode]) {
      const rgb = input.format[mode].apply(null, autodetect ? args : args.slice(0, -1));
      me._rgb = clip_rgb(rgb);
    } else {
      throw new Error('unknown format: ' + args);
    }
    // add alpha channel
    if (me._rgb.length === 3) me._rgb.push(1);
  }
  toString() {
    if (type(this.hex) == 'function') return this.hex();
    return `[${this._rgb.join(',')}]`;
  }
}

// this gets updated automatically
const version = '3.2.0';

const chroma = (...args) => {
  return new Color(...args);
};
chroma.version = version;

/**
	X11 color names

	http://www.w3.org/TR/css3-color/#svg-color
*/

const w3cx11 = {
  aliceblue: '#f0f8ff',
  antiquewhite: '#faebd7',
  aqua: '#00ffff',
  aquamarine: '#7fffd4',
  azure: '#f0ffff',
  beige: '#f5f5dc',
  bisque: '#ffe4c4',
  black: '#000000',
  blanchedalmond: '#ffebcd',
  blue: '#0000ff',
  blueviolet: '#8a2be2',
  brown: '#a52a2a',
  burlywood: '#deb887',
  cadetblue: '#5f9ea0',
  chartreuse: '#7fff00',
  chocolate: '#d2691e',
  coral: '#ff7f50',
  cornflowerblue: '#6495ed',
  cornsilk: '#fff8dc',
  crimson: '#dc143c',
  cyan: '#00ffff',
  darkblue: '#00008b',
  darkcyan: '#008b8b',
  darkgoldenrod: '#b8860b',
  darkgray: '#a9a9a9',
  darkgreen: '#006400',
  darkgrey: '#a9a9a9',
  darkkhaki: '#bdb76b',
  darkmagenta: '#8b008b',
  darkolivegreen: '#556b2f',
  darkorange: '#ff8c00',
  darkorchid: '#9932cc',
  darkred: '#8b0000',
  darksalmon: '#e9967a',
  darkseagreen: '#8fbc8f',
  darkslateblue: '#483d8b',
  darkslategray: '#2f4f4f',
  darkslategrey: '#2f4f4f',
  darkturquoise: '#00ced1',
  darkviolet: '#9400d3',
  deeppink: '#ff1493',
  deepskyblue: '#00bfff',
  dimgray: '#696969',
  dimgrey: '#696969',
  dodgerblue: '#1e90ff',
  firebrick: '#b22222',
  floralwhite: '#fffaf0',
  forestgreen: '#228b22',
  fuchsia: '#ff00ff',
  gainsboro: '#dcdcdc',
  ghostwhite: '#f8f8ff',
  gold: '#ffd700',
  goldenrod: '#daa520',
  gray: '#808080',
  green: '#008000',
  greenyellow: '#adff2f',
  grey: '#808080',
  honeydew: '#f0fff0',
  hotpink: '#ff69b4',
  indianred: '#cd5c5c',
  indigo: '#4b0082',
  ivory: '#fffff0',
  khaki: '#f0e68c',
  laserlemon: '#ffff54',
  lavender: '#e6e6fa',
  lavenderblush: '#fff0f5',
  lawngreen: '#7cfc00',
  lemonchiffon: '#fffacd',
  lightblue: '#add8e6',
  lightcoral: '#f08080',
  lightcyan: '#e0ffff',
  lightgoldenrod: '#fafad2',
  lightgoldenrodyellow: '#fafad2',
  lightgray: '#d3d3d3',
  lightgreen: '#90ee90',
  lightgrey: '#d3d3d3',
  lightpink: '#ffb6c1',
  lightsalmon: '#ffa07a',
  lightseagreen: '#20b2aa',
  lightskyblue: '#87cefa',
  lightslategray: '#778899',
  lightslategrey: '#778899',
  lightsteelblue: '#b0c4de',
  lightyellow: '#ffffe0',
  lime: '#00ff00',
  limegreen: '#32cd32',
  linen: '#faf0e6',
  magenta: '#ff00ff',
  maroon: '#800000',
  maroon2: '#7f0000',
  maroon3: '#b03060',
  mediumaquamarine: '#66cdaa',
  mediumblue: '#0000cd',
  mediumorchid: '#ba55d3',
  mediumpurple: '#9370db',
  mediumseagreen: '#3cb371',
  mediumslateblue: '#7b68ee',
  mediumspringgreen: '#00fa9a',
  mediumturquoise: '#48d1cc',
  mediumvioletred: '#c71585',
  midnightblue: '#191970',
  mintcream: '#f5fffa',
  mistyrose: '#ffe4e1',
  moccasin: '#ffe4b5',
  navajowhite: '#ffdead',
  navy: '#000080',
  oldlace: '#fdf5e6',
  olive: '#808000',
  olivedrab: '#6b8e23',
  orange: '#ffa500',
  orangered: '#ff4500',
  orchid: '#da70d6',
  palegoldenrod: '#eee8aa',
  palegreen: '#98fb98',
  paleturquoise: '#afeeee',
  palevioletred: '#db7093',
  papayawhip: '#ffefd5',
  peachpuff: '#ffdab9',
  peru: '#cd853f',
  pink: '#ffc0cb',
  plum: '#dda0dd',
  powderblue: '#b0e0e6',
  purple: '#800080',
  purple2: '#7f007f',
  purple3: '#a020f0',
  rebeccapurple: '#663399',
  red: '#ff0000',
  rosybrown: '#bc8f8f',
  royalblue: '#4169e1',
  saddlebrown: '#8b4513',
  salmon: '#fa8072',
  sandybrown: '#f4a460',
  seagreen: '#2e8b57',
  seashell: '#fff5ee',
  sienna: '#a0522d',
  silver: '#c0c0c0',
  skyblue: '#87ceeb',
  slateblue: '#6a5acd',
  slategray: '#708090',
  slategrey: '#708090',
  snow: '#fffafa',
  springgreen: '#00ff7f',
  steelblue: '#4682b4',
  tan: '#d2b48c',
  teal: '#008080',
  thistle: '#d8bfd8',
  tomato: '#ff6347',
  turquoise: '#40e0d0',
  violet: '#ee82ee',
  wheat: '#f5deb3',
  white: '#ffffff',
  whitesmoke: '#f5f5f5',
  yellow: '#ffff00',
  yellowgreen: '#9acd32'
};

const RE_HEX = /^#?([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/;
const RE_HEXA = /^#?([A-Fa-f0-9]{8}|[A-Fa-f0-9]{4})$/;
const hex2rgb = hex => {
  if (hex.match(RE_HEX)) {
    // remove optional leading #
    if (hex.length === 4 || hex.length === 7) {
      hex = hex.substr(1);
    }
    // expand short-notation to full six-digit
    if (hex.length === 3) {
      hex = hex.split('');
      hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
    }
    const u = parseInt(hex, 16);
    const r = u >> 16;
    const g = u >> 8 & 0xff;
    const b = u & 0xff;
    return [r, g, b, 1];
  }

  // match rgba hex format, eg #FF000077
  if (hex.match(RE_HEXA)) {
    if (hex.length === 5 || hex.length === 9) {
      // remove optional leading #
      hex = hex.substr(1);
    }
    // expand short-notation to full eight-digit
    if (hex.length === 4) {
      hex = hex.split('');
      hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2] + hex[3] + hex[3];
    }
    const u = parseInt(hex, 16);
    const r = u >> 24 & 0xff;
    const g = u >> 16 & 0xff;
    const b = u >> 8 & 0xff;
    const a = Math.round((u & 0xff) / 0xff * 100) / 100;
    return [r, g, b, a];
  }

  // we used to check for css colors here
  // if _input.css? and rgb = _input.css hex
  //     return rgb

  throw new Error(`unknown hex color: ${hex}`);
};

const {
  round: round$5
} = Math;
const rgb2hex = (...args) => {
  let [r, g, b, a] = unpack(args, 'rgba');
  let mode = last(args) || 'auto';
  if (a === undefined) a = 1;
  if (mode === 'auto') {
    mode = a < 1 ? 'rgba' : 'rgb';
  }
  r = round$5(r);
  g = round$5(g);
  b = round$5(b);
  const u = r << 16 | g << 8 | b;
  let str = '000000' + u.toString(16); //#.toUpperCase();
  str = str.substr(str.length - 6);
  let hxa = '0' + round$5(a * 255).toString(16);
  hxa = hxa.substr(hxa.length - 2);
  switch (mode.toLowerCase()) {
    case 'rgba':
      return `#${str}${hxa}`;
    case 'argb':
      return `#${hxa}${str}`;
    default:
      return `#${str}`;
  }
};

Color.prototype.name = function () {
  const hex = rgb2hex(this._rgb, 'rgb');
  for (let n of Object.keys(w3cx11)) {
    if (w3cx11[n] === hex) return n.toLowerCase();
  }
  return hex;
};
input.format.named = name => {
  name = name.toLowerCase();
  if (w3cx11[name]) return hex2rgb(w3cx11[name]);
  throw new Error('unknown color name: ' + name);
};
input.autodetect.push({
  p: 5,
  test: (h, ...rest) => {
    if (!rest.length && type(h) === 'string' && w3cx11[h.toLowerCase()]) {
      return 'named';
    }
  }
});

Color.prototype.alpha = function (a, mutate = false) {
  if (a !== undefined && type(a) === 'number') {
    if (mutate) {
      this._rgb[3] = a;
      return this;
    }
    return new Color([this._rgb[0], this._rgb[1], this._rgb[2], a], 'rgb');
  }
  return this._rgb[3];
};

Color.prototype.clipped = function () {
  return this._rgb._clipped || false;
};

const labConstants = {
  // Corresponds roughly to RGB brighter/darker
  Kn: 18,
  // D65 standard referent
  labWhitePoint: 'd65',
  Xn: 0.95047,
  Yn: 1,
  Zn: 1.08883,
  // t1 * t1 * t1,

  kE: 216.0 / 24389.0,
  kKE: 8.0,
  kK: 24389.0 / 27.0,
  RefWhiteRGB: {
    // sRGB
    X: 0.95047,
    Y: 1,
    Z: 1.08883
  },
  MtxRGB2XYZ: {
    m00: 0.4124564390896922,
    m01: 0.21267285140562253,
    m02: 0.0193338955823293,
    m10: 0.357576077643909,
    m11: 0.715152155287818,
    m12: 0.11919202588130297,
    m20: 0.18043748326639894,
    m21: 0.07217499330655958,
    m22: 0.9503040785363679
  },
  MtxXYZ2RGB: {
    m00: 3.2404541621141045,
    m01: -0.9692660305051868,
    m02: 0.055643430959114726,
    m10: -1.5371385127977166,
    m11: 1.8760108454466942,
    m12: -0.2040259135167538,
    m20: -0.498531409556016,
    m21: 0.041556017530349834,
    m22: 1.0572251882231791
  },
  // used in rgb2xyz
  As: 0.9414285350000001,
  Bs: 1.040417467,
  Cs: 1.089532651,
  MtxAdaptMa: {
    m00: 0.8951,
    m01: -0.7502,
    m02: 0.0389,
    m10: 0.2664,
    m11: 1.7135,
    m12: -0.0685,
    m20: -0.1614,
    m21: 0.0367,
    m22: 1.0296
  },
  MtxAdaptMaI: {
    m00: 0.9869929054667123,
    m01: 0.43230526972339456,
    m02: -0.008528664575177328,
    m10: -0.14705425642099013,
    m11: 0.5183602715367776,
    m12: 0.04004282165408487,
    m20: 0.15996265166373125,
    m21: 0.0492912282128556,
    m22: 0.9684866957875502
  }
};

// taken from https://de.mathworks.com/help/images/ref/whitepoint.html
const ILLUMINANTS = new Map([
// ASTM E308-01
['a', [1.0985, 0.35585]],
// Wyszecki & Stiles, p. 769
['b', [1.0985, 0.35585]],
// C ASTM E308-01
['c', [0.98074, 1.18232]],
// D50 (ASTM E308-01)
['d50', [0.96422, 0.82521]],
// D55 (ASTM E308-01)
['d55', [0.95682, 0.92149]],
// D65 (ASTM E308-01)
['d65', [0.95047, 1.08883]],
// E (ASTM E308-01)
['e', [1, 1, 1]],
// F2 (ASTM E308-01)
['f2', [0.99186, 0.67393]],
// F7 (ASTM E308-01)
['f7', [0.95041, 1.08747]],
// F11 (ASTM E308-01)
['f11', [1.00962, 0.6435]], ['icc', [0.96422, 0.82521]]]);
function setLabWhitePoint(name) {
  const ill = ILLUMINANTS.get(String(name).toLowerCase());
  if (!ill) {
    throw new Error('unknown Lab illuminant ' + name);
  }
  labConstants.labWhitePoint = name;
  labConstants.Xn = ill[0];
  labConstants.Zn = ill[1];
}
function getLabWhitePoint() {
  return labConstants.labWhitePoint;
}

/*
 * L* [0..100]
 * a [-100..100]
 * b [-100..100]
 */
const lab2rgb = (...args) => {
  args = unpack(args, 'lab');
  const [L, a, b] = args;
  const [x, y, z] = lab2xyz(L, a, b);
  const [r, g, b_] = xyz2rgb(x, y, z);
  return [r, g, b_, args.length > 3 ? args[3] : 1];
};
const lab2xyz = (L, a, b) => {
  const {
    kE,
    kK,
    kKE,
    Xn,
    Yn,
    Zn
  } = labConstants;
  const fy = (L + 16.0) / 116.0;
  const fx = 0.002 * a + fy;
  const fz = fy - 0.005 * b;
  const fx3 = fx * fx * fx;
  const fz3 = fz * fz * fz;
  const xr = fx3 > kE ? fx3 : (116.0 * fx - 16.0) / kK;
  const yr = L > kKE ? Math.pow((L + 16.0) / 116.0, 3.0) : L / kK;
  const zr = fz3 > kE ? fz3 : (116.0 * fz - 16.0) / kK;
  const x = xr * Xn;
  const y = yr * Yn;
  const z = zr * Zn;
  return [x, y, z];
};
const compand = linear => {
  /* sRGB */
  const sign = Math.sign(linear);
  linear = Math.abs(linear);
  return (linear <= 0.0031308 ? linear * 12.92 : 1.055 * Math.pow(linear, 1.0 / 2.4) - 0.055) * sign;
};
const xyz2rgb = (x, y, z) => {
  const {
    MtxAdaptMa,
    MtxAdaptMaI,
    MtxXYZ2RGB,
    RefWhiteRGB,
    Xn,
    Yn,
    Zn
  } = labConstants;
  const As = Xn * MtxAdaptMa.m00 + Yn * MtxAdaptMa.m10 + Zn * MtxAdaptMa.m20;
  const Bs = Xn * MtxAdaptMa.m01 + Yn * MtxAdaptMa.m11 + Zn * MtxAdaptMa.m21;
  const Cs = Xn * MtxAdaptMa.m02 + Yn * MtxAdaptMa.m12 + Zn * MtxAdaptMa.m22;
  const Ad = RefWhiteRGB.X * MtxAdaptMa.m00 + RefWhiteRGB.Y * MtxAdaptMa.m10 + RefWhiteRGB.Z * MtxAdaptMa.m20;
  const Bd = RefWhiteRGB.X * MtxAdaptMa.m01 + RefWhiteRGB.Y * MtxAdaptMa.m11 + RefWhiteRGB.Z * MtxAdaptMa.m21;
  const Cd = RefWhiteRGB.X * MtxAdaptMa.m02 + RefWhiteRGB.Y * MtxAdaptMa.m12 + RefWhiteRGB.Z * MtxAdaptMa.m22;
  const X1 = (x * MtxAdaptMa.m00 + y * MtxAdaptMa.m10 + z * MtxAdaptMa.m20) * (Ad / As);
  const Y1 = (x * MtxAdaptMa.m01 + y * MtxAdaptMa.m11 + z * MtxAdaptMa.m21) * (Bd / Bs);
  const Z1 = (x * MtxAdaptMa.m02 + y * MtxAdaptMa.m12 + z * MtxAdaptMa.m22) * (Cd / Cs);
  const X2 = X1 * MtxAdaptMaI.m00 + Y1 * MtxAdaptMaI.m10 + Z1 * MtxAdaptMaI.m20;
  const Y2 = X1 * MtxAdaptMaI.m01 + Y1 * MtxAdaptMaI.m11 + Z1 * MtxAdaptMaI.m21;
  const Z2 = X1 * MtxAdaptMaI.m02 + Y1 * MtxAdaptMaI.m12 + Z1 * MtxAdaptMaI.m22;
  const r = compand(X2 * MtxXYZ2RGB.m00 + Y2 * MtxXYZ2RGB.m10 + Z2 * MtxXYZ2RGB.m20);
  const g = compand(X2 * MtxXYZ2RGB.m01 + Y2 * MtxXYZ2RGB.m11 + Z2 * MtxXYZ2RGB.m21);
  const b = compand(X2 * MtxXYZ2RGB.m02 + Y2 * MtxXYZ2RGB.m12 + Z2 * MtxXYZ2RGB.m22);
  return [r * 255, g * 255, b * 255];
};

const rgb2lab = (...args) => {
  const [r, g, b, ...rest] = unpack(args, 'rgb');
  const [x, y, z] = rgb2xyz(r, g, b);
  const [L, a, b_] = xyz2lab(x, y, z);
  return [L, a, b_, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};
function xyz2lab(x, y, z) {
  const {
    Xn,
    Yn,
    Zn,
    kE,
    kK
  } = labConstants;
  const xr = x / Xn;
  const yr = y / Yn;
  const zr = z / Zn;
  const fx = xr > kE ? Math.pow(xr, 1.0 / 3.0) : (kK * xr + 16.0) / 116.0;
  const fy = yr > kE ? Math.pow(yr, 1.0 / 3.0) : (kK * yr + 16.0) / 116.0;
  const fz = zr > kE ? Math.pow(zr, 1.0 / 3.0) : (kK * zr + 16.0) / 116.0;
  return [116.0 * fy - 16.0, 500.0 * (fx - fy), 200.0 * (fy - fz)];
}
function gammaAdjustSRGB(companded) {
  const sign = Math.sign(companded);
  companded = Math.abs(companded);
  const linear = companded <= 0.04045 ? companded / 12.92 : Math.pow((companded + 0.055) / 1.055, 2.4);
  return linear * sign;
}
const rgb2xyz = (r, g, b) => {
  // normalize and gamma adjust
  r = gammaAdjustSRGB(r / 255);
  g = gammaAdjustSRGB(g / 255);
  b = gammaAdjustSRGB(b / 255);
  const {
    MtxRGB2XYZ,
    MtxAdaptMa,
    MtxAdaptMaI,
    Xn,
    Yn,
    Zn,
    As,
    Bs,
    Cs
  } = labConstants;
  let x = r * MtxRGB2XYZ.m00 + g * MtxRGB2XYZ.m10 + b * MtxRGB2XYZ.m20;
  let y = r * MtxRGB2XYZ.m01 + g * MtxRGB2XYZ.m11 + b * MtxRGB2XYZ.m21;
  let z = r * MtxRGB2XYZ.m02 + g * MtxRGB2XYZ.m12 + b * MtxRGB2XYZ.m22;
  const Ad = Xn * MtxAdaptMa.m00 + Yn * MtxAdaptMa.m10 + Zn * MtxAdaptMa.m20;
  const Bd = Xn * MtxAdaptMa.m01 + Yn * MtxAdaptMa.m11 + Zn * MtxAdaptMa.m21;
  const Cd = Xn * MtxAdaptMa.m02 + Yn * MtxAdaptMa.m12 + Zn * MtxAdaptMa.m22;
  let X = x * MtxAdaptMa.m00 + y * MtxAdaptMa.m10 + z * MtxAdaptMa.m20;
  let Y = x * MtxAdaptMa.m01 + y * MtxAdaptMa.m11 + z * MtxAdaptMa.m21;
  let Z = x * MtxAdaptMa.m02 + y * MtxAdaptMa.m12 + z * MtxAdaptMa.m22;
  X *= Ad / As;
  Y *= Bd / Bs;
  Z *= Cd / Cs;
  x = X * MtxAdaptMaI.m00 + Y * MtxAdaptMaI.m10 + Z * MtxAdaptMaI.m20;
  y = X * MtxAdaptMaI.m01 + Y * MtxAdaptMaI.m11 + Z * MtxAdaptMaI.m21;
  z = X * MtxAdaptMaI.m02 + Y * MtxAdaptMaI.m12 + Z * MtxAdaptMaI.m22;
  return [x, y, z];
};

Color.prototype.lab = function () {
  return rgb2lab(this._rgb);
};
const lab$1 = (...args) => new Color(...args, 'lab');
Object.assign(chroma, {
  lab: lab$1,
  getLabWhitePoint,
  setLabWhitePoint
});
input.format.lab = lab2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'lab');
    if (type(args) === 'array' && args.length === 3) {
      return 'lab';
    }
  }
});

Color.prototype.darken = function (amount = 1) {
  const me = this;
  const lab = me.lab();
  lab[0] -= labConstants.Kn * amount;
  return new Color(lab, 'lab').alpha(me.alpha(), true);
};
Color.prototype.brighten = function (amount = 1) {
  return this.darken(-amount);
};
Color.prototype.darker = Color.prototype.darken;
Color.prototype.brighter = Color.prototype.brighten;

Color.prototype.get = function (mc) {
  const [mode, channel] = mc.split('.');
  const src = this[mode]();
  if (channel) {
    const i = mode.indexOf(channel) - (mode.substr(0, 2) === 'ok' ? 2 : 0);
    if (i > -1) return src[i];
    throw new Error(`unknown channel ${channel} in mode ${mode}`);
  } else {
    return src;
  }
};

const {
  pow: pow$6
} = Math;
const EPS = 1e-7;
const MAX_ITER = 20;
Color.prototype.luminance = function (lum, mode = 'rgb') {
  if (lum !== undefined && type(lum) === 'number') {
    if (lum === 0) {
      // return pure black
      return new Color([0, 0, 0, this._rgb[3]], 'rgb');
    }
    if (lum === 1) {
      // return pure white
      return new Color([255, 255, 255, this._rgb[3]], 'rgb');
    }
    // compute new color using...
    let cur_lum = this.luminance();
    let max_iter = MAX_ITER;
    const test = (low, high) => {
      const mid = low.interpolate(high, 0.5, mode);
      const lm = mid.luminance();
      if (Math.abs(lum - lm) < EPS || !max_iter--) {
        // close enough
        return mid;
      }
      return lm > lum ? test(low, mid) : test(mid, high);
    };
    const rgb = (cur_lum > lum ? test(new Color([0, 0, 0]), this) : test(this, new Color([255, 255, 255]))).rgb();
    return new Color([...rgb, this._rgb[3]]);
  }
  return rgb2luminance(...this._rgb.slice(0, 3));
};
const rgb2luminance = (r, g, b) => {
  // relative luminance
  // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#relativeluminancedef
  r = luminance_x(r);
  g = luminance_x(g);
  b = luminance_x(b);
  return 0.2126 * r + 0.7152 * g + 0.0722 * b;
};
const luminance_x = x => {
  x /= 255;
  return x <= 0.03928 ? x / 12.92 : pow$6((x + 0.055) / 1.055, 2.4);
};

var index = {};

var mix = (col1, col2, f = 0.5, ...rest) => {
  let mode = rest[0] || 'lrgb';
  if (!index[mode] && !rest.length) {
    // fall back to the first supported mode
    mode = Object.keys(index)[0];
  }
  if (!index[mode]) {
    throw new Error(`interpolation mode ${mode} is not defined`);
  }
  if (type(col1) !== 'object') col1 = new Color(col1);
  if (type(col2) !== 'object') col2 = new Color(col2);
  return index[mode](col1, col2, f).alpha(col1.alpha() + f * (col2.alpha() - col1.alpha()));
};

Color.prototype.mix = Color.prototype.interpolate = function (col2, f = 0.5, ...rest) {
  return mix(this, col2, f, ...rest);
};

Color.prototype.premultiply = function (mutate = false) {
  const rgb = this._rgb;
  const a = rgb[3];
  if (mutate) {
    this._rgb = [rgb[0] * a, rgb[1] * a, rgb[2] * a, a];
    return this;
  } else {
    return new Color([rgb[0] * a, rgb[1] * a, rgb[2] * a, a], 'rgb');
  }
};

const {
  sin: sin$3,
  cos: cos$4
} = Math;
const lch2lab = (...args) => {
  /*
  Convert from a qualitative parameter h and a quantitative parameter l to a 24-bit pixel.
  These formulas were invented by David Dalrymple to obtain maximum contrast without going
  out of gamut if the parameters are in the range 0-1.
   A saturation multiplier was added by Gregor Aisch
  */
  let [l, c, h] = unpack(args, 'lch');
  if (isNaN(h)) h = 0;
  h = h * DEG2RAD;
  return [l, cos$4(h) * c, sin$3(h) * c];
};

const lch2rgb = (...args) => {
  args = unpack(args, 'lch');
  const [l, c, h] = args;
  const [L, a, b_] = lch2lab(l, c, h);
  const [r, g, b] = lab2rgb(L, a, b_);
  return [r, g, b, args.length > 3 ? args[3] : 1];
};

const hcl2rgb = (...args) => {
  const hcl = reverse3(unpack(args, 'hcl'));
  return lch2rgb(...hcl);
};

const {
  sqrt: sqrt$4,
  atan2: atan2$2,
  round: round$4
} = Math;
const lab2lch = (...args) => {
  const [l, a, b] = unpack(args, 'lab');
  const c = sqrt$4(a * a + b * b);
  let h = (atan2$2(b, a) * RAD2DEG + 360) % 360;
  if (round$4(c * 10000) === 0) h = Number.NaN;
  return [l, c, h];
};

const rgb2lch = (...args) => {
  const [r, g, b, ...rest] = unpack(args, 'rgb');
  const [l, a, b_] = rgb2lab(r, g, b);
  const [L, c, h] = lab2lch(l, a, b_);
  return [L, c, h, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};

Color.prototype.lch = function () {
  return rgb2lch(this._rgb);
};
Color.prototype.hcl = function () {
  return reverse3(rgb2lch(this._rgb));
};
const lch$1 = (...args) => new Color(...args, 'lch');
const hcl = (...args) => new Color(...args, 'hcl');
Object.assign(chroma, {
  lch: lch$1,
  hcl
});
input.format.lch = lch2rgb;
input.format.hcl = hcl2rgb;
['lch', 'hcl'].forEach(m => input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, m);
    if (type(args) === 'array' && args.length === 3) {
      return m;
    }
  }
}));

Color.prototype.saturate = function (amount = 1) {
  const me = this;
  const lch = me.lch();
  lch[1] += labConstants.Kn * amount;
  if (lch[1] < 0) lch[1] = 0;
  return new Color(lch, 'lch').alpha(me.alpha(), true);
};
Color.prototype.desaturate = function (amount = 1) {
  return this.saturate(-amount);
};

Color.prototype.set = function (mc, value, mutate = false) {
  const [mode, channel] = mc.split('.');
  const src = this[mode]();
  if (channel) {
    const i = mode.indexOf(channel) - (mode.substr(0, 2) === 'ok' ? 2 : 0);
    if (i > -1) {
      if (type(value) == 'string') {
        switch (value.charAt(0)) {
          case '+':
            src[i] += +value;
            break;
          case '-':
            src[i] += +value;
            break;
          case '*':
            src[i] *= +value.substr(1);
            break;
          case '/':
            src[i] /= +value.substr(1);
            break;
          default:
            src[i] = +value;
        }
      } else if (type(value) === 'number') {
        src[i] = value;
      } else {
        throw new Error(`unsupported value for Color.set`);
      }
      const out = new Color(src, mode);
      if (mutate) {
        this._rgb = out._rgb;
        return this;
      }
      return out;
    }
    throw new Error(`unknown channel ${channel} in mode ${mode}`);
  } else {
    return src;
  }
};

Color.prototype.tint = function (f = 0.5, ...rest) {
  return mix(this, 'white', f, ...rest);
};
Color.prototype.shade = function (f = 0.5, ...rest) {
  return mix(this, 'black', f, ...rest);
};

const rgb$1 = (col1, col2, f) => {
  const xyz0 = col1._rgb;
  const xyz1 = col2._rgb;
  return new Color(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'rgb');
};

// register interpolator
index.rgb = rgb$1;

const {
  sqrt: sqrt$3,
  pow: pow$5
} = Math;
const lrgb = (col1, col2, f) => {
  const [x1, y1, z1] = col1._rgb;
  const [x2, y2, z2] = col2._rgb;
  return new Color(sqrt$3(pow$5(x1, 2) * (1 - f) + pow$5(x2, 2) * f), sqrt$3(pow$5(y1, 2) * (1 - f) + pow$5(y2, 2) * f), sqrt$3(pow$5(z1, 2) * (1 - f) + pow$5(z2, 2) * f), 'rgb');
};

// register interpolator
index.lrgb = lrgb;

const lab = (col1, col2, f) => {
  const xyz0 = col1.lab();
  const xyz1 = col2.lab();
  return new Color(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'lab');
};

// register interpolator
index.lab = lab;

var interpolate_hsx = (col1, col2, f, m) => {
  let xyz0, xyz1;
  if (m === 'hsl') {
    xyz0 = col1.hsl();
    xyz1 = col2.hsl();
  } else if (m === 'hsv') {
    xyz0 = col1.hsv();
    xyz1 = col2.hsv();
  } else if (m === 'hcg') {
    xyz0 = col1.hcg();
    xyz1 = col2.hcg();
  } else if (m === 'hsi') {
    xyz0 = col1.hsi();
    xyz1 = col2.hsi();
  } else if (m === 'lch' || m === 'hcl') {
    m = 'hcl';
    xyz0 = col1.hcl();
    xyz1 = col2.hcl();
  } else if (m === 'oklch') {
    xyz0 = col1.oklch().reverse();
    xyz1 = col2.oklch().reverse();
  }
  let hue0, hue1, sat0, sat1, lbv0, lbv1;
  if (m.substr(0, 1) === 'h' || m === 'oklch') {
    [hue0, sat0, lbv0] = xyz0;
    [hue1, sat1, lbv1] = xyz1;
  }
  let sat, hue, lbv, dh;
  if (!isNaN(hue0) && !isNaN(hue1)) {
    // both colors have hue
    if (hue1 > hue0 && hue1 - hue0 > 180) {
      dh = hue1 - (hue0 + 360);
    } else if (hue1 < hue0 && hue0 - hue1 > 180) {
      dh = hue1 + 360 - hue0;
    } else {
      dh = hue1 - hue0;
    }
    hue = hue0 + f * dh;
  } else if (!isNaN(hue0)) {
    hue = hue0;
    if ((lbv1 == 1 || lbv1 == 0) && m != 'hsv') sat = sat0;
  } else if (!isNaN(hue1)) {
    hue = hue1;
    if ((lbv0 == 1 || lbv0 == 0) && m != 'hsv') sat = sat1;
  } else {
    hue = Number.NaN;
  }
  if (sat === undefined) sat = sat0 + f * (sat1 - sat0);
  lbv = lbv0 + f * (lbv1 - lbv0);
  return m === 'oklch' ? new Color([lbv, sat, hue], m) : new Color([hue, sat, lbv], m);
};

const lch = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'lch');
};

// register interpolator
index.lch = lch;
index.hcl = lch;

const num2rgb = num => {
  if (type(num) == 'number' && num >= 0 && num <= 0xffffff) {
    const r = num >> 16;
    const g = num >> 8 & 0xff;
    const b = num & 0xff;
    return [r, g, b, 1];
  }
  throw new Error('unknown num color: ' + num);
};

const rgb2num = (...args) => {
  const [r, g, b] = unpack(args, 'rgb');
  return (r << 16) + (g << 8) + b;
};

Color.prototype.num = function () {
  return rgb2num(this._rgb);
};
const num$1 = (...args) => new Color(...args, 'num');
Object.assign(chroma, {
  num: num$1
});
input.format.num = num2rgb;
input.autodetect.push({
  p: 5,
  test: (...args) => {
    if (args.length === 1 && type(args[0]) === 'number' && args[0] >= 0 && args[0] <= 0xffffff) {
      return 'num';
    }
  }
});

const num = (col1, col2, f) => {
  const c1 = col1.num();
  const c2 = col2.num();
  return new Color(c1 + f * (c2 - c1), 'num');
};

// register interpolator
index.num = num;

const {
  floor: floor$3
} = Math;

/*
 * this is basically just HSV with some minor tweaks
 *
 * hue.. [0..360]
 * chroma .. [0..1]
 * grayness .. [0..1]
 */

const hcg2rgb = (...args) => {
  args = unpack(args, 'hcg');
  let [h, c, _g] = args;
  let r, g, b;
  _g = _g * 255;
  const _c = c * 255;
  if (c === 0) {
    r = g = b = _g;
  } else {
    if (h === 360) h = 0;
    if (h > 360) h -= 360;
    if (h < 0) h += 360;
    h /= 60;
    const i = floor$3(h);
    const f = h - i;
    const p = _g * (1 - c);
    const q = p + _c * (1 - f);
    const t = p + _c * f;
    const v = p + _c;
    switch (i) {
      case 0:
        [r, g, b] = [v, t, p];
        break;
      case 1:
        [r, g, b] = [q, v, p];
        break;
      case 2:
        [r, g, b] = [p, v, t];
        break;
      case 3:
        [r, g, b] = [p, q, v];
        break;
      case 4:
        [r, g, b] = [t, p, v];
        break;
      case 5:
        [r, g, b] = [v, p, q];
        break;
    }
  }
  return [r, g, b, args.length > 3 ? args[3] : 1];
};

const rgb2hcg = (...args) => {
  const [r, g, b] = unpack(args, 'rgb');
  const minRgb = min$4(r, g, b);
  const maxRgb = max$4(r, g, b);
  const delta = maxRgb - minRgb;
  const c = delta * 100 / 255;
  const _g = minRgb / (255 - delta) * 100;
  let h;
  if (delta === 0) {
    h = Number.NaN;
  } else {
    if (r === maxRgb) h = (g - b) / delta;
    if (g === maxRgb) h = 2 + (b - r) / delta;
    if (b === maxRgb) h = 4 + (r - g) / delta;
    h *= 60;
    if (h < 0) h += 360;
  }
  return [h, c, _g];
};

Color.prototype.hcg = function () {
  return rgb2hcg(this._rgb);
};
const hcg$1 = (...args) => new Color(...args, 'hcg');
chroma.hcg = hcg$1;
input.format.hcg = hcg2rgb;
input.autodetect.push({
  p: 1,
  test: (...args) => {
    args = unpack(args, 'hcg');
    if (type(args) === 'array' && args.length === 3) {
      return 'hcg';
    }
  }
});

const hcg = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'hcg');
};

// register interpolator
index.hcg = hcg;

const {
  cos: cos$3
} = Math;

/*
 * hue [0..360]
 * saturation [0..1]
 * intensity [0..1]
 */
const hsi2rgb = (...args) => {
  /*
  borrowed from here:
  http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/hsi2rgb.cpp
  */
  args = unpack(args, 'hsi');
  let [h, s, i] = args;
  let r, g, b;
  if (isNaN(h)) h = 0;
  if (isNaN(s)) s = 0;
  // normalize hue
  if (h > 360) h -= 360;
  if (h < 0) h += 360;
  h /= 360;
  if (h < 1 / 3) {
    b = (1 - s) / 3;
    r = (1 + s * cos$3(TWOPI * h) / cos$3(PITHIRD - TWOPI * h)) / 3;
    g = 1 - (b + r);
  } else if (h < 2 / 3) {
    h -= 1 / 3;
    r = (1 - s) / 3;
    g = (1 + s * cos$3(TWOPI * h) / cos$3(PITHIRD - TWOPI * h)) / 3;
    b = 1 - (r + g);
  } else {
    h -= 2 / 3;
    g = (1 - s) / 3;
    b = (1 + s * cos$3(TWOPI * h) / cos$3(PITHIRD - TWOPI * h)) / 3;
    r = 1 - (g + b);
  }
  r = limit(i * r * 3);
  g = limit(i * g * 3);
  b = limit(i * b * 3);
  return [r * 255, g * 255, b * 255, args.length > 3 ? args[3] : 1];
};

const {
  min: min$3,
  sqrt: sqrt$2,
  acos
} = Math;
const rgb2hsi = (...args) => {
  /*
  borrowed from here:
  http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/rgb2hsi.cpp
  */
  let [r, g, b] = unpack(args, 'rgb');
  r /= 255;
  g /= 255;
  b /= 255;
  let h;
  const min_ = min$3(r, g, b);
  const i = (r + g + b) / 3;
  const s = i > 0 ? 1 - min_ / i : 0;
  if (s === 0) {
    h = NaN;
  } else {
    h = (r - g + (r - b)) / 2;
    h /= sqrt$2((r - g) * (r - g) + (r - b) * (g - b));
    h = acos(h);
    if (b > g) {
      h = TWOPI - h;
    }
    h /= TWOPI;
  }
  return [h * 360, s, i];
};

Color.prototype.hsi = function () {
  return rgb2hsi(this._rgb);
};
const hsi$1 = (...args) => new Color(...args, 'hsi');
chroma.hsi = hsi$1;
input.format.hsi = hsi2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'hsi');
    if (type(args) === 'array' && args.length === 3) {
      return 'hsi';
    }
  }
});

const hsi = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'hsi');
};

// register interpolator
index.hsi = hsi;

const hsl2rgb = (...args) => {
  args = unpack(args, 'hsl');
  const [h, s, l] = args;
  let r, g, b;
  if (s === 0) {
    r = g = b = l * 255;
  } else {
    const t3 = [0, 0, 0];
    const c = [0, 0, 0];
    const t2 = l < 0.5 ? l * (1 + s) : l + s - l * s;
    const t1 = 2 * l - t2;
    const h_ = h / 360;
    t3[0] = h_ + 1 / 3;
    t3[1] = h_;
    t3[2] = h_ - 1 / 3;
    for (let i = 0; i < 3; i++) {
      if (t3[i] < 0) t3[i] += 1;
      if (t3[i] > 1) t3[i] -= 1;
      if (6 * t3[i] < 1) c[i] = t1 + (t2 - t1) * 6 * t3[i];else if (2 * t3[i] < 1) c[i] = t2;else if (3 * t3[i] < 2) c[i] = t1 + (t2 - t1) * (2 / 3 - t3[i]) * 6;else c[i] = t1;
    }
    [r, g, b] = [c[0] * 255, c[1] * 255, c[2] * 255];
  }
  if (args.length > 3) {
    // keep alpha channel
    return [r, g, b, args[3]];
  }
  return [r, g, b, 1];
};

/*
 * supported arguments:
 * - rgb2hsl(r,g,b)
 * - rgb2hsl(r,g,b,a)
 * - rgb2hsl([r,g,b])
 * - rgb2hsl([r,g,b,a])
 * - rgb2hsl({r,g,b,a})
 */
const rgb2hsl$1 = (...args) => {
  args = unpack(args, 'rgba');
  let [r, g, b] = args;
  r /= 255;
  g /= 255;
  b /= 255;
  const minRgb = min$4(r, g, b);
  const maxRgb = max$4(r, g, b);
  const l = (maxRgb + minRgb) / 2;
  let s, h;
  if (maxRgb === minRgb) {
    s = 0;
    h = Number.NaN;
  } else {
    s = l < 0.5 ? (maxRgb - minRgb) / (maxRgb + minRgb) : (maxRgb - minRgb) / (2 - maxRgb - minRgb);
  }
  if (r == maxRgb) h = (g - b) / (maxRgb - minRgb);else if (g == maxRgb) h = 2 + (b - r) / (maxRgb - minRgb);else if (b == maxRgb) h = 4 + (r - g) / (maxRgb - minRgb);
  h *= 60;
  if (h < 0) h += 360;
  if (args.length > 3 && args[3] !== undefined) return [h, s, l, args[3]];
  return [h, s, l];
};

Color.prototype.hsl = function () {
  return rgb2hsl$1(this._rgb);
};
const hsl$1 = (...args) => new Color(...args, 'hsl');
chroma.hsl = hsl$1;
input.format.hsl = hsl2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'hsl');
    if (type(args) === 'array' && args.length === 3) {
      return 'hsl';
    }
  }
});

const hsl = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'hsl');
};

// register interpolator
index.hsl = hsl;

const {
  floor: floor$2
} = Math;
const hsv2rgb = (...args) => {
  args = unpack(args, 'hsv');
  let [h, s, v] = args;
  let r, g, b;
  v *= 255;
  if (s === 0) {
    r = g = b = v;
  } else {
    if (h === 360) h = 0;
    if (h > 360) h -= 360;
    if (h < 0) h += 360;
    h /= 60;
    const i = floor$2(h);
    const f = h - i;
    const p = v * (1 - s);
    const q = v * (1 - s * f);
    const t = v * (1 - s * (1 - f));
    switch (i) {
      case 0:
        [r, g, b] = [v, t, p];
        break;
      case 1:
        [r, g, b] = [q, v, p];
        break;
      case 2:
        [r, g, b] = [p, v, t];
        break;
      case 3:
        [r, g, b] = [p, q, v];
        break;
      case 4:
        [r, g, b] = [t, p, v];
        break;
      case 5:
        [r, g, b] = [v, p, q];
        break;
    }
  }
  return [r, g, b, args.length > 3 ? args[3] : 1];
};

const {
  min: min$2,
  max: max$3
} = Math;

/*
 * supported arguments:
 * - rgb2hsv(r,g,b)
 * - rgb2hsv([r,g,b])
 * - rgb2hsv({r,g,b})
 */
const rgb2hsl = (...args) => {
  args = unpack(args, 'rgb');
  let [r, g, b] = args;
  const min_ = min$2(r, g, b);
  const max_ = max$3(r, g, b);
  const delta = max_ - min_;
  let h, s, v;
  v = max_ / 255.0;
  if (max_ === 0) {
    h = Number.NaN;
    s = 0;
  } else {
    s = delta / max_;
    if (r === max_) h = (g - b) / delta;
    if (g === max_) h = 2 + (b - r) / delta;
    if (b === max_) h = 4 + (r - g) / delta;
    h *= 60;
    if (h < 0) h += 360;
  }
  return [h, s, v];
};

Color.prototype.hsv = function () {
  return rgb2hsl(this._rgb);
};
const hsv$1 = (...args) => new Color(...args, 'hsv');
chroma.hsv = hsv$1;
input.format.hsv = hsv2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'hsv');
    if (type(args) === 'array' && args.length === 3) {
      return 'hsv';
    }
  }
});

const hsv = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'hsv');
};

// register interpolator
index.hsv = hsv;

// from https://www.w3.org/TR/css-color-4/multiply-matrices.js
function multiplyMatrices(A, B) {
  let m = A.length;
  if (!Array.isArray(A[0])) {
    // A is vector, convert to [[a, b, c, ...]]
    A = [A];
  }
  if (!Array.isArray(B[0])) {
    // B is vector, convert to [[a], [b], [c], ...]]
    B = B.map(x => [x]);
  }
  let p = B[0].length;
  let B_cols = B[0].map((_, i) => B.map(x => x[i])); // transpose B
  let product = A.map(row => B_cols.map(col => {
    if (!Array.isArray(row)) {
      return col.reduce((a, c) => a + c * row, 0);
    }
    return row.reduce((a, c, i) => a + c * (col[i] || 0), 0);
  }));
  if (m === 1) {
    product = product[0]; // Avoid [[a, b, c, ...]]
  }
  if (p === 1) {
    return product.map(x => x[0]); // Avoid [[a], [b], [c], ...]]
  }
  return product;
}

const oklab2rgb = (...args) => {
  args = unpack(args, 'lab');
  const [L, a, b, ...rest] = args;
  const [X, Y, Z] = OKLab_to_XYZ([L, a, b]);
  const [r, g, b_] = xyz2rgb(X, Y, Z);
  return [r, g, b_, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};

// from https://www.w3.org/TR/css-color-4/#color-conversion-code
function OKLab_to_XYZ(OKLab) {
  // Given OKLab, convert to XYZ relative to D65
  var LMStoXYZ = [[1.2268798758459243, -0.5578149944602171, 0.2813910456659647], [-0.0405757452148008, 1.112286803280317, -0.0717110580655164], [-0.0763729366746601, -0.4214933324022432, 1.5869240198367816]];
  var OKLabtoLMS = [[1.0, 0.3963377773761749, 0.2158037573099136], [1.0, -0.1055613458156586, -0.0638541728258133], [1.0, -0.0894841775298119, -1.2914855480194092]];
  var LMSnl = multiplyMatrices(OKLabtoLMS, OKLab);
  return multiplyMatrices(LMStoXYZ, LMSnl.map(c => c ** 3));
}

const rgb2oklab = (...args) => {
  const [r, g, b, ...rest] = unpack(args, 'rgb');
  const xyz = rgb2xyz(r, g, b);
  const oklab = XYZ_to_OKLab(xyz);
  return [...oklab, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};

// from https://www.w3.org/TR/css-color-4/#color-conversion-code
function XYZ_to_OKLab(XYZ) {
  // Given XYZ relative to D65, convert to OKLab
  const XYZtoLMS = [[0.819022437996703, 0.3619062600528904, -0.1288737815209879], [0.0329836539323885, 0.9292868615863434, 0.0361446663506424], [0.0481771893596242, 0.2642395317527308, 0.6335478284694309]];
  const LMStoOKLab = [[0.210454268309314, 0.7936177747023054, -0.0040720430116193], [1.9779985324311684, -2.42859224204858, 0.450593709617411], [0.0259040424655478, 0.7827717124575296, -0.8086757549230774]];
  const LMS = multiplyMatrices(XYZtoLMS, XYZ);
  // JavaScript Math.cbrt returns a sign-matched cube root
  // beware if porting to other languages
  // especially if tempted to use a general power function
  return multiplyMatrices(LMStoOKLab, LMS.map(c => Math.cbrt(c)));
  // L in range [0,1]. For use in CSS, multiply by 100 and add a percent
}

Color.prototype.oklab = function () {
  return rgb2oklab(this._rgb);
};
const oklab$1 = (...args) => new Color(...args, 'oklab');
Object.assign(chroma, {
  oklab: oklab$1
});
input.format.oklab = oklab2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'oklab');
    if (type(args) === 'array' && args.length === 3) {
      return 'oklab';
    }
  }
});

const oklab = (col1, col2, f) => {
  const xyz0 = col1.oklab();
  const xyz1 = col2.oklab();
  return new Color(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'oklab');
};

// register interpolator
index.oklab = oklab;

const oklch$1 = (col1, col2, f) => {
  return interpolate_hsx(col1, col2, f, 'oklch');
};

// register interpolator
index.oklch = oklch$1;

const {
  pow: pow$4,
  sqrt: sqrt$1,
  PI: PI$1,
  cos: cos$2,
  sin: sin$2,
  atan2: atan2$1
} = Math;
var average = (colors, mode = 'lrgb', weights = null) => {
  const l = colors.length;
  if (!weights) weights = Array.from(new Array(l)).map(() => 1);
  // normalize weights
  const k = l / weights.reduce(function (a, b) {
    return a + b;
  });
  weights.forEach((w, i) => {
    weights[i] *= k;
  });
  // convert colors to Color objects
  colors = colors.map(c => new Color(c));
  if (mode === 'lrgb') {
    return _average_lrgb(colors, weights);
  }
  const first = colors.shift();
  const xyz = first.get(mode);
  const cnt = [];
  let dx = 0;
  let dy = 0;
  // initial color
  for (let i = 0; i < xyz.length; i++) {
    xyz[i] = (xyz[i] || 0) * weights[0];
    cnt.push(isNaN(xyz[i]) ? 0 : weights[0]);
    if (mode.charAt(i) === 'h' && !isNaN(xyz[i])) {
      const A = xyz[i] / 180 * PI$1;
      dx += cos$2(A) * weights[0];
      dy += sin$2(A) * weights[0];
    }
  }
  let alpha = first.alpha() * weights[0];
  colors.forEach((c, ci) => {
    const xyz2 = c.get(mode);
    alpha += c.alpha() * weights[ci + 1];
    for (let i = 0; i < xyz.length; i++) {
      if (!isNaN(xyz2[i])) {
        cnt[i] += weights[ci + 1];
        if (mode.charAt(i) === 'h') {
          const A = xyz2[i] / 180 * PI$1;
          dx += cos$2(A) * weights[ci + 1];
          dy += sin$2(A) * weights[ci + 1];
        } else {
          xyz[i] += xyz2[i] * weights[ci + 1];
        }
      }
    }
  });
  for (let i = 0; i < xyz.length; i++) {
    if (mode.charAt(i) === 'h') {
      let A = atan2$1(dy / cnt[i], dx / cnt[i]) / PI$1 * 180;
      while (A < 0) A += 360;
      while (A >= 360) A -= 360;
      xyz[i] = A;
    } else {
      xyz[i] = xyz[i] / cnt[i];
    }
  }
  alpha /= l;
  return new Color(xyz, mode).alpha(alpha > 0.99999 ? 1 : alpha, true);
};
const _average_lrgb = (colors, weights) => {
  const l = colors.length;
  const xyz = [0, 0, 0, 0];
  for (let i = 0; i < colors.length; i++) {
    const col = colors[i];
    const f = weights[i] / l;
    const rgb = col._rgb;
    xyz[0] += pow$4(rgb[0], 2) * f;
    xyz[1] += pow$4(rgb[1], 2) * f;
    xyz[2] += pow$4(rgb[2], 2) * f;
    xyz[3] += rgb[3] * f;
  }
  xyz[0] = sqrt$1(xyz[0]);
  xyz[1] = sqrt$1(xyz[1]);
  xyz[2] = sqrt$1(xyz[2]);
  if (xyz[3] > 0.9999999) xyz[3] = 1;
  return new Color(clip_rgb(xyz));
};

// minimal multi-purpose interface

const {
  pow: pow$3
} = Math;
function scale (colors) {
  // constructor
  let _mode = 'rgb';
  let _nacol = chroma('#ccc');
  let _spread = 0;
  // const _fixed = false;
  let _positions = [0, 1];
  let _domain = [0, 1];
  let _pos = [];
  let _padding = [0, 0];
  let _classes = false;
  let _colors = [];
  let _out = false;
  let _min = 0;
  let _max = 1;
  let _correctLightness = false;
  let _colorCache = {};
  let _useCache = true;
  let _gamma = 1;

  // private methods

  const setColors = function (colors) {
    colors = colors || ['#fff', '#000'];
    if (colors && type(colors) === 'string' && chroma.brewer && chroma.brewer[colors.toLowerCase()]) {
      colors = chroma.brewer[colors.toLowerCase()];
    }
    if (type(colors) === 'array') {
      // handle single color
      if (colors.length === 1) {
        colors = [colors[0], colors[0]];
      }
      // make a copy of the colors
      colors = colors.slice(0);
      // convert to chroma classes
      for (let c = 0; c < colors.length; c++) {
        colors[c] = chroma(colors[c]);
      }
      // auto-fill color position
      _pos.length = 0;
      for (let c = 0; c < colors.length; c++) {
        _pos.push(c / (colors.length - 1));
      }
    }
    resetCache();
    return _colors = colors;
  };
  const getClass = function (value) {
    if (_classes != null) {
      const n = _classes.length - 1;
      let i = 0;
      while (i < n && value >= _classes[i]) {
        i++;
      }
      return i - 1;
    }
    return 0;
  };
  let tMapLightness = t => t;
  let tMapDomain = t => t;

  // const classifyValue = function(value) {
  //     let val = value;
  //     if (_classes.length > 2) {
  //         const n = _classes.length-1;
  //         const i = getClass(value);
  //         const minc = _classes[0] + ((_classes[1]-_classes[0]) * (0 + (_spread * 0.5)));  // center of 1st class
  //         const maxc = _classes[n-1] + ((_classes[n]-_classes[n-1]) * (1 - (_spread * 0.5)));  // center of last class
  //         val = _min + ((((_classes[i] + ((_classes[i+1] - _classes[i]) * 0.5)) - minc) / (maxc-minc)) * (_max - _min));
  //     }
  //     return val;
  // };

  const getColor = function (val, bypassMap) {
    let col, t;
    if (bypassMap == null) {
      bypassMap = false;
    }
    if (isNaN(val) || val === null) {
      return _nacol;
    }
    if (!bypassMap) {
      if (_classes && _classes.length > 2) {
        // find the class
        const c = getClass(val);
        t = c / (_classes.length - 2);
      } else if (_max !== _min) {
        // just interpolate between min/max
        t = (val - _min) / (_max - _min);
      } else {
        t = 1;
      }
    } else {
      t = val;
    }

    // domain map
    t = tMapDomain(t);
    if (!bypassMap) {
      t = tMapLightness(t); // lightness correction
    }
    if (_gamma !== 1) {
      t = pow$3(t, _gamma);
    }
    t = _padding[0] + t * (1 - _padding[0] - _padding[1]);
    t = limit(t, 0, 1);
    const k = Math.floor(t * 10000);
    if (_useCache && _colorCache[k]) {
      col = _colorCache[k];
    } else {
      if (type(_colors) === 'array') {
        //for i in [0.._pos.length-1]
        for (let i = 0; i < _pos.length; i++) {
          const p = _pos[i];
          if (t <= p) {
            col = _colors[i];
            break;
          }
          if (t >= p && i === _pos.length - 1) {
            col = _colors[i];
            break;
          }
          if (t > p && t < _pos[i + 1]) {
            t = (t - p) / (_pos[i + 1] - p);
            col = chroma.interpolate(_colors[i], _colors[i + 1], t, _mode);
            break;
          }
        }
      } else if (type(_colors) === 'function') {
        col = _colors(t);
      }
      if (_useCache) {
        _colorCache[k] = col;
      }
    }
    return col;
  };
  var resetCache = () => _colorCache = {};
  setColors(colors);

  // public interface

  const f = function (v) {
    const c = chroma(getColor(v));
    if (_out && c[_out]) {
      return c[_out]();
    } else {
      return c;
    }
  };
  f.classes = function (classes) {
    if (classes != null) {
      if (type(classes) === 'array') {
        _classes = classes;
        _positions = [classes[0], classes[classes.length - 1]];
      } else {
        const d = chroma.analyze(_positions);
        if (classes === 0) {
          _classes = [d.min, d.max];
        } else {
          _classes = chroma.limits(d, 'e', classes);
        }
      }
      return f;
    }
    return _classes;
  };
  f.domain = function (domain) {
    if (!arguments.length) {
      // return original domain
      return _domain;
    }
    // store original domain so we can return it later
    _domain = domain.slice(0);
    _min = domain[0];
    _max = domain[domain.length - 1];
    _pos = [];
    const k = _colors.length;
    if (domain.length === k && _min !== _max) {
      // update positions
      for (let d of Array.from(domain)) {
        _pos.push((d - _min) / (_max - _min));
      }
    } else {
      for (let c = 0; c < k; c++) {
        _pos.push(c / (k - 1));
      }
      if (domain.length > 2) {
        // set domain map
        const tOut = domain.map((d, i) => i / (domain.length - 1));
        const tBreaks = domain.map(d => (d - _min) / (_max - _min));
        if (!tBreaks.every((val, i) => tOut[i] === val)) {
          tMapDomain = t => {
            if (t <= 0 || t >= 1) return t;
            let i = 0;
            while (t >= tBreaks[i + 1]) i++;
            const f = (t - tBreaks[i]) / (tBreaks[i + 1] - tBreaks[i]);
            const out = tOut[i] + f * (tOut[i + 1] - tOut[i]);
            return out;
          };
        }
      }
    }
    _positions = [_min, _max];
    return f;
  };
  f.mode = function (_m) {
    if (!arguments.length) {
      return _mode;
    }
    _mode = _m;
    resetCache();
    return f;
  };
  f.range = function (colors, _pos) {
    setColors(colors);
    return f;
  };
  f.out = function (_o) {
    _out = _o;
    return f;
  };
  f.spread = function (val) {
    if (!arguments.length) {
      return _spread;
    }
    _spread = val;
    return f;
  };
  f.correctLightness = function (v) {
    if (v == null) {
      v = true;
    }
    _correctLightness = v;
    resetCache();
    if (_correctLightness) {
      tMapLightness = function (t) {
        const L0 = getColor(0, true).lab()[0];
        const L1 = getColor(1, true).lab()[0];
        const pol = L0 > L1;
        let L_actual = getColor(t, true).lab()[0];
        const L_ideal = L0 + (L1 - L0) * t;
        let L_diff = L_actual - L_ideal;
        let t0 = 0;
        let t1 = 1;
        let max_iter = 20;
        while (Math.abs(L_diff) > 1e-2 && max_iter-- > 0) {
          (function () {
            if (pol) {
              L_diff *= -1;
            }
            if (L_diff < 0) {
              t0 = t;
              t += (t1 - t) * 0.5;
            } else {
              t1 = t;
              t += (t0 - t) * 0.5;
            }
            L_actual = getColor(t, true).lab()[0];
            return L_diff = L_actual - L_ideal;
          })();
        }
        return t;
      };
    } else {
      tMapLightness = t => t;
    }
    return f;
  };
  f.padding = function (p) {
    if (p != null) {
      if (type(p) === 'number') {
        p = [p, p];
      }
      _padding = p;
      return f;
    } else {
      return _padding;
    }
  };
  f.colors = function (numColors, out) {
    // If no arguments are given, return the original colors that were provided
    if (arguments.length < 2) {
      out = 'hex';
    }
    let result = [];
    if (arguments.length === 0) {
      result = _colors.slice(0);
    } else if (numColors === 1) {
      result = [f(0.5)];
    } else if (numColors > 1) {
      const dm = _positions[0];
      const dd = _positions[1] - dm;
      result = __range__(0, numColors).map(i => f(dm + i / (numColors - 1) * dd));
    } else {
      // returns all colors based on the defined classes
      colors = [];
      let samples = [];
      if (_classes && _classes.length > 2) {
        for (let i = 1, end = _classes.length, asc = 1 <= end; asc ? i < end : i > end; asc ? i++ : i--) {
          samples.push((_classes[i - 1] + _classes[i]) * 0.5);
        }
      } else {
        samples = _positions;
      }
      result = samples.map(v => f(v));
    }
    if (chroma[out]) {
      result = result.map(c => c[out]());
    }
    return result;
  };
  f.cache = function (c) {
    if (c != null) {
      _useCache = c;
      return f;
    } else {
      return _useCache;
    }
  };
  f.gamma = function (g) {
    if (g != null) {
      _gamma = g;
      return f;
    } else {
      return _gamma;
    }
  };
  f.nodata = function (d) {
    if (d != null) {
      _nacol = chroma(d);
      return f;
    } else {
      return _nacol;
    }
  };
  return f;
}
function __range__(left, right, inclusive) {
  let range = [];
  let ascending = left < right;
  let end = right ;
  for (let i = left; ascending ? i < end : i > end; ascending ? i++ : i--) {
    range.push(i);
  }
  return range;
}

//
// interpolates between a set of colors uzing a bezier spline
//


// nth row of the pascal triangle
const binom_row = function (n) {
  let row = [1, 1];
  for (let i = 1; i < n; i++) {
    let newrow = [1];
    for (let j = 1; j <= row.length; j++) {
      newrow[j] = (row[j] || 0) + row[j - 1];
    }
    row = newrow;
  }
  return row;
};
const bezier$1 = function (colors) {
  let I, lab0, lab1, lab2;
  colors = colors.map(c => new Color(c));
  if (colors.length === 2) {
    // linear interpolation
    [lab0, lab1] = colors.map(c => c.lab());
    I = function (t) {
      const lab = [0, 1, 2].map(i => lab0[i] + t * (lab1[i] - lab0[i]));
      return new Color(lab, 'lab');
    };
  } else if (colors.length === 3) {
    // quadratic bezier interpolation
    [lab0, lab1, lab2] = colors.map(c => c.lab());
    I = function (t) {
      const lab = [0, 1, 2].map(i => (1 - t) * (1 - t) * lab0[i] + 2 * (1 - t) * t * lab1[i] + t * t * lab2[i]);
      return new Color(lab, 'lab');
    };
  } else if (colors.length === 4) {
    // cubic bezier interpolation
    let lab3;
    [lab0, lab1, lab2, lab3] = colors.map(c => c.lab());
    I = function (t) {
      const lab = [0, 1, 2].map(i => (1 - t) * (1 - t) * (1 - t) * lab0[i] + 3 * (1 - t) * (1 - t) * t * lab1[i] + 3 * (1 - t) * t * t * lab2[i] + t * t * t * lab3[i]);
      return new Color(lab, 'lab');
    };
  } else if (colors.length >= 5) {
    // general case (degree n bezier)
    let labs, row, n;
    labs = colors.map(c => c.lab());
    n = colors.length - 1;
    row = binom_row(n);
    I = function (t) {
      const u = 1 - t;
      const lab = [0, 1, 2].map(i => labs.reduce((sum, el, j) => sum + row[j] * u ** (n - j) * t ** j * el[i], 0));
      return new Color(lab, 'lab');
    };
  } else {
    throw new RangeError('No point in running bezier with only one color.');
  }
  return I;
};
var bezier = colors => {
  const f = bezier$1(colors);
  f.scale = () => scale(f);
  return f;
};

const {
  round: round$3
} = Math;
Color.prototype.rgb = function (rnd = true) {
  if (rnd === false) return this._rgb.slice(0, 3);
  return this._rgb.slice(0, 3).map(round$3);
};
Color.prototype.rgba = function (rnd = true) {
  return this._rgb.slice(0, 4).map((v, i) => {
    return i < 3 ? rnd === false ? v : round$3(v) : v;
  });
};
const rgb = (...args) => new Color(...args, 'rgb');
Object.assign(chroma, {
  rgb
});
input.format.rgb = (...args) => {
  const rgba = unpack(args, 'rgba');
  if (rgba[3] === undefined) rgba[3] = 1;
  return rgba;
};
input.autodetect.push({
  p: 3,
  test: (...args) => {
    args = unpack(args, 'rgba');
    if (type(args) === 'array' && (args.length === 3 || args.length === 4 && type(args[3]) == 'number' && args[3] >= 0 && args[3] <= 1)) {
      return 'rgb';
    }
  }
});

/*
 * interpolates between a set of colors uzing a bezier spline
 * blend mode formulas taken from https://web.archive.org/web/20180110014946/http://www.venture-ware.com/kevin/coding/lets-learn-math-photoshop-blend-modes/
 */

const blend = (bottom, top, mode) => {
  if (!blend[mode]) {
    throw new Error('unknown blend mode ' + mode);
  }
  return blend[mode](bottom, top);
};
const blend_f = f => (bottom, top) => {
  const c0 = chroma(top).rgb();
  const c1 = chroma(bottom).rgb();
  return chroma.rgb(f(c0, c1));
};
const each = f => (c0, c1) => {
  const out = [];
  out[0] = f(c0[0], c1[0]);
  out[1] = f(c0[1], c1[1]);
  out[2] = f(c0[2], c1[2]);
  return out;
};
const normal = a => a;
const multiply = (a, b) => a * b / 255;
const darken = (a, b) => a > b ? b : a;
const lighten = (a, b) => a > b ? a : b;
const screen = (a, b) => 255 * (1 - (1 - a / 255) * (1 - b / 255));
const overlay = (a, b) => b < 128 ? 2 * a * b / 255 : 255 * (1 - 2 * (1 - a / 255) * (1 - b / 255));
const burn = (a, b) => 255 * (1 - (1 - b / 255) / (a / 255));
const dodge = (a, b) => {
  if (a === 255) return 255;
  a = 255 * (b / 255) / (1 - a / 255);
  return a > 255 ? 255 : a;
};

// # add = (a,b) ->
// #     if (a + b > 255) then 255 else a + b

blend.normal = blend_f(each(normal));
blend.multiply = blend_f(each(multiply));
blend.screen = blend_f(each(screen));
blend.overlay = blend_f(each(overlay));
blend.darken = blend_f(each(darken));
blend.lighten = blend_f(each(lighten));
blend.dodge = blend_f(each(dodge));
blend.burn = blend_f(each(burn));

// cubehelix interpolation
// based on D.A. Green "A colour scheme for the display of astronomical intensity images"
// http://astron-soc.in/bulletin/11June/289392011.pdf
const {
  pow: pow$2,
  sin: sin$1,
  cos: cos$1
} = Math;
function cubehelix (start = 300, rotations = -1.5, hue = 1, gamma = 1, lightness = [0, 1]) {
  let dh = 0,
    dl;
  if (type(lightness) === 'array') {
    dl = lightness[1] - lightness[0];
  } else {
    dl = 0;
    lightness = [lightness, lightness];
  }
  const f = function (fract) {
    const a = TWOPI * ((start + 120) / 360 + rotations * fract);
    const l = pow$2(lightness[0] + dl * fract, gamma);
    const h = dh !== 0 ? hue[0] + fract * dh : hue;
    const amp = h * l * (1 - l) / 2;
    const cos_a = cos$1(a);
    const sin_a = sin$1(a);
    const r = l + amp * (-0.14861 * cos_a + 1.78277 * sin_a);
    const g = l + amp * (-0.29227 * cos_a - 0.90649 * sin_a);
    const b = l + amp * (1.97294 * cos_a);
    return chroma(clip_rgb([r * 255, g * 255, b * 255, 1]));
  };
  f.start = function (s) {
    if (s == null) {
      return start;
    }
    start = s;
    return f;
  };
  f.rotations = function (r) {
    if (r == null) {
      return rotations;
    }
    rotations = r;
    return f;
  };
  f.gamma = function (g) {
    if (g == null) {
      return gamma;
    }
    gamma = g;
    return f;
  };
  f.hue = function (h) {
    if (h == null) {
      return hue;
    }
    hue = h;
    if (type(hue) === 'array') {
      dh = hue[1] - hue[0];
      if (dh === 0) {
        hue = hue[1];
      }
    } else {
      dh = 0;
    }
    return f;
  };
  f.lightness = function (h) {
    if (h == null) {
      return lightness;
    }
    if (type(h) === 'array') {
      lightness = h;
      dl = h[1] - h[0];
    } else {
      lightness = [h, h];
      dl = 0;
    }
    return f;
  };
  f.scale = () => chroma.scale(f);
  f.hue(hue);
  return f;
}

const digits = '0123456789abcdef';
const {
  floor: floor$1,
  random: random$1
} = Math;

/**
 * Generates a random color.
 * @param {() => number} rng - A random number generator function.
 */
var random = (rng = random$1) => {
  let code = '#';
  for (let i = 0; i < 6; i++) {
    code += digits.charAt(floor$1(rng() * 16));
  }
  return new Color(code, 'hex');
};

const {
  log: log$1,
  pow: pow$1,
  floor,
  abs: abs$1
} = Math;
function analyze(data, key = null) {
  const r = {
    min: Number.MAX_VALUE,
    max: Number.MAX_VALUE * -1,
    sum: 0,
    values: [],
    count: 0
  };
  if (type(data) === 'object') {
    data = Object.values(data);
  }
  data.forEach(val => {
    if (key && type(val) === 'object') val = val[key];
    if (val !== undefined && val !== null && !isNaN(val)) {
      r.values.push(val);
      r.sum += val;
      if (val < r.min) r.min = val;
      if (val > r.max) r.max = val;
      r.count += 1;
    }
  });
  r.domain = [r.min, r.max];
  r.limits = (mode, num) => limits(r, mode, num);
  return r;
}
function limits(data, mode = 'equal', num = 7) {
  if (type(data) == 'array') {
    data = analyze(data);
  }
  const {
    min,
    max
  } = data;
  const values = data.values.sort((a, b) => a - b);
  if (num === 1) {
    return [min, max];
  }
  const limits = [];
  if (mode.substr(0, 1) === 'c') {
    // continuous
    limits.push(min);
    limits.push(max);
  }
  if (mode.substr(0, 1) === 'e') {
    // equal interval
    limits.push(min);
    for (let i = 1; i < num; i++) {
      limits.push(min + i / num * (max - min));
    }
    limits.push(max);
  } else if (mode.substr(0, 1) === 'l') {
    // log scale
    if (min <= 0) {
      throw new Error('Logarithmic scales are only possible for values > 0');
    }
    const min_log = Math.LOG10E * log$1(min);
    const max_log = Math.LOG10E * log$1(max);
    limits.push(min);
    for (let i = 1; i < num; i++) {
      limits.push(pow$1(10, min_log + i / num * (max_log - min_log)));
    }
    limits.push(max);
  } else if (mode.substr(0, 1) === 'q') {
    // quantile scale
    limits.push(min);
    for (let i = 1; i < num; i++) {
      const p = (values.length - 1) * i / num;
      const pb = floor(p);
      if (pb === p) {
        limits.push(values[pb]);
      } else {
        // p > pb
        const pr = p - pb;
        limits.push(values[pb] * (1 - pr) + values[pb + 1] * pr);
      }
    }
    limits.push(max);
  } else if (mode.substr(0, 1) === 'k') {
    // k-means clustering
    /*
    implementation based on
    http://code.google.com/p/figue/source/browse/trunk/figue.js#336
    simplified for 1-d input values
    */
    let cluster;
    const n = values.length;
    const assignments = new Array(n);
    const clusterSizes = new Array(num);
    let repeat = true;
    let nb_iters = 0;
    let centroids = null;

    // get seed values
    centroids = [];
    centroids.push(min);
    for (let i = 1; i < num; i++) {
      centroids.push(min + i / num * (max - min));
    }
    centroids.push(max);
    while (repeat) {
      // assignment step
      for (let j = 0; j < num; j++) {
        clusterSizes[j] = 0;
      }
      for (let i = 0; i < n; i++) {
        const value = values[i];
        let mindist = Number.MAX_VALUE;
        let best;
        for (let j = 0; j < num; j++) {
          const dist = abs$1(centroids[j] - value);
          if (dist < mindist) {
            mindist = dist;
            best = j;
          }
          clusterSizes[best]++;
          assignments[i] = best;
        }
      }

      // update centroids step
      const newCentroids = new Array(num);
      for (let j = 0; j < num; j++) {
        newCentroids[j] = null;
      }
      for (let i = 0; i < n; i++) {
        cluster = assignments[i];
        if (newCentroids[cluster] === null) {
          newCentroids[cluster] = values[i];
        } else {
          newCentroids[cluster] += values[i];
        }
      }
      for (let j = 0; j < num; j++) {
        newCentroids[j] *= 1 / clusterSizes[j];
      }

      // check convergence
      repeat = false;
      for (let j = 0; j < num; j++) {
        if (newCentroids[j] !== centroids[j]) {
          repeat = true;
          break;
        }
      }
      centroids = newCentroids;
      nb_iters++;
      if (nb_iters > 200) {
        repeat = false;
      }
    }

    // finished k-means clustering
    // the next part is borrowed from gabrielflor.it
    const kClusters = {};
    for (let j = 0; j < num; j++) {
      kClusters[j] = [];
    }
    for (let i = 0; i < n; i++) {
      cluster = assignments[i];
      kClusters[cluster].push(values[i]);
    }
    let tmpKMeansBreaks = [];
    for (let j = 0; j < num; j++) {
      tmpKMeansBreaks.push(kClusters[j][0]);
      tmpKMeansBreaks.push(kClusters[j][kClusters[j].length - 1]);
    }
    tmpKMeansBreaks = tmpKMeansBreaks.sort((a, b) => a - b);
    limits.push(tmpKMeansBreaks[0]);
    for (let i = 1; i < tmpKMeansBreaks.length; i += 2) {
      const v = tmpKMeansBreaks[i];
      if (!isNaN(v) && limits.indexOf(v) === -1) {
        limits.push(v);
      }
    }
  }
  return limits;
}

var contrast = (a, b) => {
  // WCAG contrast ratio
  // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef
  a = new Color(a);
  b = new Color(b);
  const l1 = a.luminance();
  const l2 = b.luminance();
  return l1 > l2 ? (l1 + 0.05) / (l2 + 0.05) : (l2 + 0.05) / (l1 + 0.05);
};

/**
 * @license
 *
 * The APCA contrast prediction algorithm is based of the formulas published
 * in the APCA-1.0.98G specification by Myndex. The specification is available at:
 * https://raw.githubusercontent.com/Myndex/apca-w3/master/images/APCAw3_0.1.17_APCA0.0.98G.svg
 *
 * Note that the APCA implementation is still beta, so please update to
 * future versions of chroma.js when they become available.
 *
 * You can read more about the APCA Readability Criterion at
 * https://readtech.org/ARC/
 */

// constants
const W_offset = 0.027;
const P_in = 0.0005;
const P_out = 0.1;
const R_scale = 1.14;
const B_threshold = 0.022;
const B_exp = 1.414;
var contrastAPCA = (text, bg) => {
  // parse input colors
  text = new Color(text);
  bg = new Color(bg);
  // if text color has alpha, blend against background
  if (text.alpha() < 1) {
    text = mix(bg, text, text.alpha(), 'rgb');
  }
  const l_text = lum(...text.rgb());
  const l_bg = lum(...bg.rgb());

  // soft clamp black levels
  const Y_text = l_text >= B_threshold ? l_text : l_text + Math.pow(B_threshold - l_text, B_exp);
  const Y_bg = l_bg >= B_threshold ? l_bg : l_bg + Math.pow(B_threshold - l_bg, B_exp);

  // normal polarity (dark text on light background)
  const S_norm = Math.pow(Y_bg, 0.56) - Math.pow(Y_text, 0.57);
  // reverse polarity (light text on dark background)
  const S_rev = Math.pow(Y_bg, 0.65) - Math.pow(Y_text, 0.62);
  // clamp noise then scale
  const C = Math.abs(Y_bg - Y_text) < P_in ? 0 : Y_text < Y_bg ? S_norm * R_scale : S_rev * R_scale;
  // clamp minimum contrast then offset
  const S_apc = Math.abs(C) < P_out ? 0 : C > 0 ? C - W_offset : C + W_offset;
  // scale to 100
  return S_apc * 100;
};
function lum(r, g, b) {
  return 0.2126729 * Math.pow(r / 255, 2.4) + 0.7151522 * Math.pow(g / 255, 2.4) + 0.072175 * Math.pow(b / 255, 2.4);
}

const {
  sqrt,
  pow,
  min: min$1,
  max: max$2,
  atan2,
  abs,
  cos,
  sin,
  exp,
  PI
} = Math;
function deltaE (a, b, Kl = 1, Kc = 1, Kh = 1) {
  // Delta E (CIE 2000)
  // see http://www.brucelindbloom.com/index.html?Eqn_DeltaE_CIE2000.html
  var rad2deg = function (rad) {
    return 360 * rad / (2 * PI);
  };
  var deg2rad = function (deg) {
    return 2 * PI * deg / 360;
  };
  a = new Color(a);
  b = new Color(b);
  const [L1, a1, b1] = Array.from(a.lab());
  const [L2, a2, b2] = Array.from(b.lab());
  const avgL = (L1 + L2) / 2;
  const C1 = sqrt(pow(a1, 2) + pow(b1, 2));
  const C2 = sqrt(pow(a2, 2) + pow(b2, 2));
  const avgC = (C1 + C2) / 2;
  const G = 0.5 * (1 - sqrt(pow(avgC, 7) / (pow(avgC, 7) + pow(25, 7))));
  const a1p = a1 * (1 + G);
  const a2p = a2 * (1 + G);
  const C1p = sqrt(pow(a1p, 2) + pow(b1, 2));
  const C2p = sqrt(pow(a2p, 2) + pow(b2, 2));
  const avgCp = (C1p + C2p) / 2;
  const arctan1 = rad2deg(atan2(b1, a1p));
  const arctan2 = rad2deg(atan2(b2, a2p));
  const h1p = arctan1 >= 0 ? arctan1 : arctan1 + 360;
  const h2p = arctan2 >= 0 ? arctan2 : arctan2 + 360;
  const avgHp = abs(h1p - h2p) > 180 ? (h1p + h2p + 360) / 2 : (h1p + h2p) / 2;
  const T = 1 - 0.17 * cos(deg2rad(avgHp - 30)) + 0.24 * cos(deg2rad(2 * avgHp)) + 0.32 * cos(deg2rad(3 * avgHp + 6)) - 0.2 * cos(deg2rad(4 * avgHp - 63));
  let deltaHp = h2p - h1p;
  deltaHp = abs(deltaHp) <= 180 ? deltaHp : h2p <= h1p ? deltaHp + 360 : deltaHp - 360;
  deltaHp = 2 * sqrt(C1p * C2p) * sin(deg2rad(deltaHp) / 2);
  const deltaL = L2 - L1;
  const deltaCp = C2p - C1p;
  const sl = 1 + 0.015 * pow(avgL - 50, 2) / sqrt(20 + pow(avgL - 50, 2));
  const sc = 1 + 0.045 * avgCp;
  const sh = 1 + 0.015 * avgCp * T;
  const deltaTheta = 30 * exp(-pow((avgHp - 275) / 25, 2));
  const Rc = 2 * sqrt(pow(avgCp, 7) / (pow(avgCp, 7) + pow(25, 7)));
  const Rt = -Rc * sin(2 * deg2rad(deltaTheta));
  const result = sqrt(pow(deltaL / (Kl * sl), 2) + pow(deltaCp / (Kc * sc), 2) + pow(deltaHp / (Kh * sh), 2) + Rt * (deltaCp / (Kc * sc)) * (deltaHp / (Kh * sh)));
  return max$2(0, min$1(100, result));
}

// simple Euclidean distance
function distance (a, b, mode = 'lab') {
  // Delta E (CIE 1976)
  // see http://www.brucelindbloom.com/index.html?Equations.html
  a = new Color(a);
  b = new Color(b);
  const l1 = a.get(mode);
  const l2 = b.get(mode);
  let sum_sq = 0;
  for (let i in l1) {
    const d = (l1[i] || 0) - (l2[i] || 0);
    sum_sq += d * d;
  }
  return Math.sqrt(sum_sq);
}

var valid = (...args) => {
  try {
    new Color(...args);
    return true;
    // eslint-disable-next-line
  } catch (e) {
    return false;
  }
};

// some pre-defined color scales:
var scales = {
  cool() {
    return scale([chroma.hsl(180, 1, 0.9), chroma.hsl(250, 0.7, 0.4)]);
  },
  hot() {
    return scale(['#000', '#f00', '#ff0', '#fff']).mode('rgb');
  }
};

/**
    ColorBrewer colors for chroma.js

    Copyright (c) 2002 Cynthia Brewer, Mark Harrower, and The
    Pennsylvania State University.

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software distributed
    under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
    CONDITIONS OF ANY KIND, either express or implied. See the License for the
    specific language governing permissions and limitations under the License.
*/

const colorbrewer = {
  // sequential
  OrRd: ['#fff7ec', '#fee8c8', '#fdd49e', '#fdbb84', '#fc8d59', '#ef6548', '#d7301f', '#b30000', '#7f0000'],
  PuBu: ['#fff7fb', '#ece7f2', '#d0d1e6', '#a6bddb', '#74a9cf', '#3690c0', '#0570b0', '#045a8d', '#023858'],
  BuPu: ['#f7fcfd', '#e0ecf4', '#bfd3e6', '#9ebcda', '#8c96c6', '#8c6bb1', '#88419d', '#810f7c', '#4d004b'],
  Oranges: ['#fff5eb', '#fee6ce', '#fdd0a2', '#fdae6b', '#fd8d3c', '#f16913', '#d94801', '#a63603', '#7f2704'],
  BuGn: ['#f7fcfd', '#e5f5f9', '#ccece6', '#99d8c9', '#66c2a4', '#41ae76', '#238b45', '#006d2c', '#00441b'],
  YlOrBr: ['#ffffe5', '#fff7bc', '#fee391', '#fec44f', '#fe9929', '#ec7014', '#cc4c02', '#993404', '#662506'],
  YlGn: ['#ffffe5', '#f7fcb9', '#d9f0a3', '#addd8e', '#78c679', '#41ab5d', '#238443', '#006837', '#004529'],
  Reds: ['#fff5f0', '#fee0d2', '#fcbba1', '#fc9272', '#fb6a4a', '#ef3b2c', '#cb181d', '#a50f15', '#67000d'],
  RdPu: ['#fff7f3', '#fde0dd', '#fcc5c0', '#fa9fb5', '#f768a1', '#dd3497', '#ae017e', '#7a0177', '#49006a'],
  Greens: ['#f7fcf5', '#e5f5e0', '#c7e9c0', '#a1d99b', '#74c476', '#41ab5d', '#238b45', '#006d2c', '#00441b'],
  YlGnBu: ['#ffffd9', '#edf8b1', '#c7e9b4', '#7fcdbb', '#41b6c4', '#1d91c0', '#225ea8', '#253494', '#081d58'],
  Purples: ['#fcfbfd', '#efedf5', '#dadaeb', '#bcbddc', '#9e9ac8', '#807dba', '#6a51a3', '#54278f', '#3f007d'],
  GnBu: ['#f7fcf0', '#e0f3db', '#ccebc5', '#a8ddb5', '#7bccc4', '#4eb3d3', '#2b8cbe', '#0868ac', '#084081'],
  Greys: ['#ffffff', '#f0f0f0', '#d9d9d9', '#bdbdbd', '#969696', '#737373', '#525252', '#252525', '#000000'],
  YlOrRd: ['#ffffcc', '#ffeda0', '#fed976', '#feb24c', '#fd8d3c', '#fc4e2a', '#e31a1c', '#bd0026', '#800026'],
  PuRd: ['#f7f4f9', '#e7e1ef', '#d4b9da', '#c994c7', '#df65b0', '#e7298a', '#ce1256', '#980043', '#67001f'],
  Blues: ['#f7fbff', '#deebf7', '#c6dbef', '#9ecae1', '#6baed6', '#4292c6', '#2171b5', '#08519c', '#08306b'],
  PuBuGn: ['#fff7fb', '#ece2f0', '#d0d1e6', '#a6bddb', '#67a9cf', '#3690c0', '#02818a', '#016c59', '#014636'],
  Viridis: ['#440154', '#482777', '#3f4a8a', '#31678e', '#26838f', '#1f9d8a', '#6cce5a', '#b6de2b', '#fee825'],
  // diverging
  Spectral: ['#9e0142', '#d53e4f', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#e6f598', '#abdda4', '#66c2a5', '#3288bd', '#5e4fa2'],
  RdYlGn: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#d9ef8b', '#a6d96a', '#66bd63', '#1a9850', '#006837'],
  RdBu: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#f7f7f7', '#d1e5f0', '#92c5de', '#4393c3', '#2166ac', '#053061'],
  PiYG: ['#8e0152', '#c51b7d', '#de77ae', '#f1b6da', '#fde0ef', '#f7f7f7', '#e6f5d0', '#b8e186', '#7fbc41', '#4d9221', '#276419'],
  PRGn: ['#40004b', '#762a83', '#9970ab', '#c2a5cf', '#e7d4e8', '#f7f7f7', '#d9f0d3', '#a6dba0', '#5aae61', '#1b7837', '#00441b'],
  RdYlBu: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee090', '#ffffbf', '#e0f3f8', '#abd9e9', '#74add1', '#4575b4', '#313695'],
  BrBG: ['#543005', '#8c510a', '#bf812d', '#dfc27d', '#f6e8c3', '#f5f5f5', '#c7eae5', '#80cdc1', '#35978f', '#01665e', '#003c30'],
  RdGy: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#ffffff', '#e0e0e0', '#bababa', '#878787', '#4d4d4d', '#1a1a1a'],
  PuOr: ['#7f3b08', '#b35806', '#e08214', '#fdb863', '#fee0b6', '#f7f7f7', '#d8daeb', '#b2abd2', '#8073ac', '#542788', '#2d004b'],
  // qualitative
  Set2: ['#66c2a5', '#fc8d62', '#8da0cb', '#e78ac3', '#a6d854', '#ffd92f', '#e5c494', '#b3b3b3'],
  Accent: ['#7fc97f', '#beaed4', '#fdc086', '#ffff99', '#386cb0', '#f0027f', '#bf5b17', '#666666'],
  Set1: ['#e41a1c', '#377eb8', '#4daf4a', '#984ea3', '#ff7f00', '#ffff33', '#a65628', '#f781bf', '#999999'],
  Set3: ['#8dd3c7', '#ffffb3', '#bebada', '#fb8072', '#80b1d3', '#fdb462', '#b3de69', '#fccde5', '#d9d9d9', '#bc80bd', '#ccebc5', '#ffed6f'],
  Dark2: ['#1b9e77', '#d95f02', '#7570b3', '#e7298a', '#66a61e', '#e6ab02', '#a6761d', '#666666'],
  Paired: ['#a6cee3', '#1f78b4', '#b2df8a', '#33a02c', '#fb9a99', '#e31a1c', '#fdbf6f', '#ff7f00', '#cab2d6', '#6a3d9a', '#ffff99', '#b15928'],
  Pastel2: ['#b3e2cd', '#fdcdac', '#cbd5e8', '#f4cae4', '#e6f5c9', '#fff2ae', '#f1e2cc', '#cccccc'],
  Pastel1: ['#fbb4ae', '#b3cde3', '#ccebc5', '#decbe4', '#fed9a6', '#ffffcc', '#e5d8bd', '#fddaec', '#f2f2f2']
};
const colorbrewerTypes = Object.keys(colorbrewer);
const typeMap = new Map(colorbrewerTypes.map(key => [key.toLowerCase(), key]));

// use Proxy to allow case-insensitive access to palettes
const colorbrewerProxy = typeof Proxy === 'function' ? new Proxy(colorbrewer, {
  get(target, prop) {
    const lower = prop.toLowerCase();
    if (typeMap.has(lower)) {
      return target[typeMap.get(lower)];
    }
  },
  getOwnPropertyNames() {
    return Object.getOwnPropertyNames(colorbrewerTypes);
  }
}) : colorbrewer;

const cmyk2rgb = (...args) => {
  args = unpack(args, 'cmyk');
  const [c, m, y, k] = args;
  const alpha = args.length > 4 ? args[4] : 1;
  if (k === 1) return [0, 0, 0, alpha];
  return [c >= 1 ? 0 : 255 * (1 - c) * (1 - k),
  // r
  m >= 1 ? 0 : 255 * (1 - m) * (1 - k),
  // g
  y >= 1 ? 0 : 255 * (1 - y) * (1 - k),
  // b
  alpha];
};

const {
  max: max$1
} = Math;
const rgb2cmyk = (...args) => {
  let [r, g, b] = unpack(args, 'rgb');
  r = r / 255;
  g = g / 255;
  b = b / 255;
  const k = 1 - max$1(r, max$1(g, b));
  const f = k < 1 ? 1 / (1 - k) : 0;
  const c = (1 - r - k) * f;
  const m = (1 - g - k) * f;
  const y = (1 - b - k) * f;
  return [c, m, y, k];
};

Color.prototype.cmyk = function () {
  return rgb2cmyk(this._rgb);
};
const cmyk = (...args) => new Color(...args, 'cmyk');
Object.assign(chroma, {
  cmyk
});
input.format.cmyk = cmyk2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'cmyk');
    if (type(args) === 'array' && args.length === 4) {
      return 'cmyk';
    }
  }
});

/*
 * supported arguments:
 * - hsl2css(h,s,l)
 * - hsl2css(h,s,l,a)
 * - hsl2css([h,s,l], mode)
 * - hsl2css([h,s,l,a], mode)
 * - hsl2css({h,s,l,a}, mode)
 */
const hsl2css = (...args) => {
  const hsla = unpack(args, 'hsla');
  let mode = last(args) || 'lsa';
  hsla[0] = rnd2(hsla[0] || 0) + 'deg';
  hsla[1] = rnd2(hsla[1] * 100) + '%';
  hsla[2] = rnd2(hsla[2] * 100) + '%';
  if (mode === 'hsla' || hsla.length > 3 && hsla[3] < 1) {
    hsla[3] = '/ ' + (hsla.length > 3 ? hsla[3] : 1);
    mode = 'hsla';
  } else {
    hsla.length = 3;
  }
  return `${mode.substr(0, 3)}(${hsla.join(' ')})`;
};

/*
 * supported arguments:
 * - lab2css(l,a,b)
 * - lab2css(l,a,b,alpha)
 * - lab2css([l,a,b], mode)
 * - lab2css([l,a,b,alpha], mode)
 */
const lab2css = (...args) => {
  const laba = unpack(args, 'lab');
  let mode = last(args) || 'lab';
  laba[0] = rnd2(laba[0]) + '%';
  laba[1] = rnd2(laba[1]);
  laba[2] = rnd2(laba[2]);
  if (mode === 'laba' || laba.length > 3 && laba[3] < 1) {
    laba[3] = '/ ' + (laba.length > 3 ? laba[3] : 1);
  } else {
    laba.length = 3;
  }
  return `lab(${laba.join(' ')})`;
};

/*
 * supported arguments:
 * - lab2css(l,a,b)
 * - lab2css(l,a,b,alpha)
 * - lab2css([l,a,b], mode)
 * - lab2css([l,a,b,alpha], mode)
 */
const lch2css = (...args) => {
  const lcha = unpack(args, 'lch');
  let mode = last(args) || 'lab';
  lcha[0] = rnd2(lcha[0]) + '%';
  lcha[1] = rnd2(lcha[1]);
  lcha[2] = isNaN(lcha[2]) ? 'none' : rnd2(lcha[2]) + 'deg'; // add deg unit to hue
  if (mode === 'lcha' || lcha.length > 3 && lcha[3] < 1) {
    lcha[3] = '/ ' + (lcha.length > 3 ? lcha[3] : 1);
  } else {
    lcha.length = 3;
  }
  return `lch(${lcha.join(' ')})`;
};

const oklab2css = (...args) => {
  const laba = unpack(args, 'lab');
  laba[0] = rnd2(laba[0] * 100) + '%';
  laba[1] = rnd3(laba[1]);
  laba[2] = rnd3(laba[2]);
  if (laba.length > 3 && laba[3] < 1) {
    laba[3] = '/ ' + (laba.length > 3 ? laba[3] : 1);
  } else {
    laba.length = 3;
  }
  return `oklab(${laba.join(' ')})`;
};

const rgb2oklch = (...args) => {
  const [r, g, b, ...rest] = unpack(args, 'rgb');
  const [l, a, b_] = rgb2oklab(r, g, b);
  const [L, c, h] = lab2lch(l, a, b_);
  return [L, c, h, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};

const oklch2css = (...args) => {
  const lcha = unpack(args, 'lch');
  lcha[0] = rnd2(lcha[0] * 100) + '%';
  lcha[1] = rnd3(lcha[1]);
  lcha[2] = isNaN(lcha[2]) ? 'none' : rnd2(lcha[2]) + 'deg'; // add deg unit to hue
  if (lcha.length > 3 && lcha[3] < 1) {
    lcha[3] = '/ ' + (lcha.length > 3 ? lcha[3] : 1);
  } else {
    lcha.length = 3;
  }
  return `oklch(${lcha.join(' ')})`;
};

const {
  round: round$2
} = Math;

/*
 * supported arguments:
 * - rgb2css(r,g,b)
 * - rgb2css(r,g,b,a)
 * - rgb2css([r,g,b], mode)
 * - rgb2css([r,g,b,a], mode)
 * - rgb2css({r,g,b,a}, mode)
 */
const rgb2css = (...args) => {
  const rgba = unpack(args, 'rgba');
  let mode = last(args) || 'rgb';
  if (mode.substr(0, 3) === 'hsl') {
    return hsl2css(rgb2hsl$1(rgba), mode);
  }
  if (mode.substr(0, 3) === 'lab') {
    // change to D50 lab whitepoint since this is what W3C is using for CSS Lab colors
    const prevWhitePoint = getLabWhitePoint();
    setLabWhitePoint('d50');
    const cssColor = lab2css(rgb2lab(rgba), mode);
    setLabWhitePoint(prevWhitePoint);
    return cssColor;
  }
  if (mode.substr(0, 3) === 'lch') {
    // change to D50 lab whitepoint since this is what W3C is using for CSS Lab colors
    const prevWhitePoint = getLabWhitePoint();
    setLabWhitePoint('d50');
    const cssColor = lch2css(rgb2lch(rgba), mode);
    setLabWhitePoint(prevWhitePoint);
    return cssColor;
  }
  if (mode.substr(0, 5) === 'oklab') {
    return oklab2css(rgb2oklab(rgba));
  }
  if (mode.substr(0, 5) === 'oklch') {
    return oklch2css(rgb2oklch(rgba));
  }
  rgba[0] = round$2(rgba[0]);
  rgba[1] = round$2(rgba[1]);
  rgba[2] = round$2(rgba[2]);
  if (mode === 'rgba' || rgba.length > 3 && rgba[3] < 1) {
    rgba[3] = '/ ' + (rgba.length > 3 ? rgba[3] : 1);
    mode = 'rgba';
  }
  return `${mode.substr(0, 3)}(${rgba.slice(0, mode === 'rgb' ? 3 : 4).join(' ')})`;
};

const oklch2rgb = (...args) => {
  args = unpack(args, 'lch');
  const [l, c, h, ...rest] = args;
  const [L, a, b_] = lch2lab(l, c, h);
  const [r, g, b] = oklab2rgb(L, a, b_);
  return [r, g, b, ...(rest.length > 0 && rest[0] < 1 ? [rest[0]] : [])];
};

const INT_OR_PCT = /((?:-?\d+)|(?:-?\d+(?:\.\d+)?)%|none)/.source;
const FLOAT_OR_PCT = /((?:-?(?:\d+(?:\.\d*)?|\.\d+)%?)|none)/.source;
const PCT = /((?:-?(?:\d+(?:\.\d*)?|\.\d+)%)|none)/.source;
const RE_S = /\s*/.source;
const SEP = /\s+/.source;
const COMMA = /\s*,\s*/.source;
const ANLGE = /((?:-?(?:\d+(?:\.\d*)?|\.\d+)(?:deg)?)|none)/.source;
const ALPHA = /\s*(?:\/\s*((?:[01]|[01]?\.\d+)|\d+(?:\.\d+)?%))?/.source;

// e.g. rgb(250 20 0), rgb(100% 50% 20%), rgb(100% 50% 20% / 0.5)
const RE_RGB = new RegExp('^rgba?\\(' + RE_S + [INT_OR_PCT, INT_OR_PCT, INT_OR_PCT].join(SEP) + ALPHA + '\\)$');
const RE_RGB_LEGACY = new RegExp('^rgb\\(' + RE_S + [INT_OR_PCT, INT_OR_PCT, INT_OR_PCT].join(COMMA) + RE_S + '\\)$');
const RE_RGBA_LEGACY = new RegExp('^rgba\\(' + RE_S + [INT_OR_PCT, INT_OR_PCT, INT_OR_PCT, FLOAT_OR_PCT].join(COMMA) + RE_S + '\\)$');
const RE_HSL = new RegExp('^hsla?\\(' + RE_S + [ANLGE, PCT, PCT].join(SEP) + ALPHA + '\\)$');
const RE_HSL_LEGACY = new RegExp('^hsl?\\(' + RE_S + [ANLGE, PCT, PCT].join(COMMA) + RE_S + '\\)$');
const RE_HSLA_LEGACY = /^hsla\(\s*(-?\d+(?:\.\d+)?),\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*,\s*([01]|[01]?\.\d+)\)$/;
const RE_LAB = new RegExp('^lab\\(' + RE_S + [FLOAT_OR_PCT, FLOAT_OR_PCT, FLOAT_OR_PCT].join(SEP) + ALPHA + '\\)$');
const RE_LCH = new RegExp('^lch\\(' + RE_S + [FLOAT_OR_PCT, FLOAT_OR_PCT, ANLGE].join(SEP) + ALPHA + '\\)$');
const RE_OKLAB = new RegExp('^oklab\\(' + RE_S + [FLOAT_OR_PCT, FLOAT_OR_PCT, FLOAT_OR_PCT].join(SEP) + ALPHA + '\\)$');
const RE_OKLCH = new RegExp('^oklch\\(' + RE_S + [FLOAT_OR_PCT, FLOAT_OR_PCT, ANLGE].join(SEP) + ALPHA + '\\)$');
const {
  round: round$1
} = Math;
const roundRGB = rgb => {
  return rgb.map((v, i) => i <= 2 ? limit(round$1(v), 0, 255) : v);
};
const percentToAbsolute = (pct, min = 0, max = 100, signed = false) => {
  if (typeof pct === 'string' && pct.endsWith('%')) {
    pct = parseFloat(pct.substring(0, pct.length - 1)) / 100;
    if (signed) {
      // signed percentages are in the range -100% to 100%
      pct = min + (pct + 1) * 0.5 * (max - min);
    } else {
      pct = min + pct * (max - min);
    }
  }
  return +pct;
};
const noneToValue = (v, noneValue) => {
  return v === 'none' ? noneValue : v;
};
const css2rgb = css => {
  css = css.toLowerCase().trim();
  if (css === 'transparent') {
    return [0, 0, 0, 0];
  }
  let m;
  if (input.format.named) {
    try {
      return input.format.named(css);
      // eslint-disable-next-line
    } catch (e) {}
  }

  // rgb(250 20 0) or rgb(250,20,0)
  if ((m = css.match(RE_RGB)) || (m = css.match(RE_RGB_LEGACY))) {
    let rgb = m.slice(1, 4);
    for (let i = 0; i < 3; i++) {
      rgb[i] = +percentToAbsolute(noneToValue(rgb[i], 0), 0, 255);
    }
    rgb = roundRGB(rgb);
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha; // default alpha
    return rgb;
  }

  // rgba(250,20,0,0.4)
  if (m = css.match(RE_RGBA_LEGACY)) {
    const rgb = m.slice(1, 5);
    for (let i = 0; i < 4; i++) {
      rgb[i] = +percentToAbsolute(rgb[i], 0, 255);
    }
    return rgb;
  }

  // hsl(0,100%,50%)
  if ((m = css.match(RE_HSL)) || (m = css.match(RE_HSL_LEGACY))) {
    const hsl = m.slice(1, 4);
    hsl[0] = +noneToValue(hsl[0].replace('deg', ''), 0);
    hsl[1] = +percentToAbsolute(noneToValue(hsl[1], 0), 0, 100) * 0.01;
    hsl[2] = +percentToAbsolute(noneToValue(hsl[2], 0), 0, 100) * 0.01;
    const rgb = roundRGB(hsl2rgb(hsl));
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha;
    return rgb;
  }

  // hsla(0,100%,50%,0.5)
  if (m = css.match(RE_HSLA_LEGACY)) {
    const hsl = m.slice(1, 4);
    hsl[1] *= 0.01;
    hsl[2] *= 0.01;
    const rgb = hsl2rgb(hsl);
    for (let i = 0; i < 3; i++) {
      rgb[i] = round$1(rgb[i]);
    }
    rgb[3] = +m[4]; // default alpha = 1
    return rgb;
  }
  if (m = css.match(RE_LAB)) {
    const lab = m.slice(1, 4);
    lab[0] = percentToAbsolute(noneToValue(lab[0], 0), 0, 100);
    lab[1] = percentToAbsolute(noneToValue(lab[1], 0), -125, 125, true);
    lab[2] = percentToAbsolute(noneToValue(lab[2], 0), -125, 125, true);
    // convert to D50 Lab whitepoint
    const wp = getLabWhitePoint();
    setLabWhitePoint('d50');
    const rgb = roundRGB(lab2rgb(lab));
    // convert back to original Lab whitepoint
    setLabWhitePoint(wp);
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha;
    return rgb;
  }
  if (m = css.match(RE_LCH)) {
    const lch = m.slice(1, 4);
    lch[0] = percentToAbsolute(lch[0], 0, 100);
    lch[1] = percentToAbsolute(noneToValue(lch[1], 0), 0, 150, false);
    lch[2] = +noneToValue(lch[2].replace('deg', ''), 0);
    // convert to D50 Lab whitepoint
    const wp = getLabWhitePoint();
    setLabWhitePoint('d50');
    const rgb = roundRGB(lch2rgb(lch));
    // convert back to original Lab whitepoint
    setLabWhitePoint(wp);
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha;
    return rgb;
  }
  if (m = css.match(RE_OKLAB)) {
    const oklab = m.slice(1, 4);
    oklab[0] = percentToAbsolute(noneToValue(oklab[0], 0), 0, 1);
    oklab[1] = percentToAbsolute(noneToValue(oklab[1], 0), -0.4, 0.4, true);
    oklab[2] = percentToAbsolute(noneToValue(oklab[2], 0), -0.4, 0.4, true);
    const rgb = roundRGB(oklab2rgb(oklab));
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha;
    return rgb;
  }
  if (m = css.match(RE_OKLCH)) {
    const oklch = m.slice(1, 4);
    oklch[0] = percentToAbsolute(noneToValue(oklch[0], 0), 0, 1);
    oklch[1] = percentToAbsolute(noneToValue(oklch[1], 0), 0, 0.4, false);
    oklch[2] = +noneToValue(oklch[2].replace('deg', ''), 0);
    const rgb = roundRGB(oklch2rgb(oklch));
    const alpha = m[4] !== undefined ? +percentToAbsolute(m[4], 0, 1) : 1;
    rgb[3] = alpha;
    return rgb;
  }
};
css2rgb.test = s => {
  return (
    // modern
    RE_RGB.test(s) || RE_HSL.test(s) || RE_LAB.test(s) || RE_LCH.test(s) || RE_OKLAB.test(s) || RE_OKLCH.test(s) ||
    // legacy
    RE_RGB_LEGACY.test(s) || RE_RGBA_LEGACY.test(s) || RE_HSL_LEGACY.test(s) || RE_HSLA_LEGACY.test(s) || s === 'transparent'
  );
};

Color.prototype.css = function (mode) {
  return rgb2css(this._rgb, mode);
};
const css = (...args) => new Color(...args, 'css');
chroma.css = css;
input.format.css = css2rgb;
input.autodetect.push({
  p: 5,
  test: (h, ...rest) => {
    if (!rest.length && type(h) === 'string' && css2rgb.test(h)) {
      return 'css';
    }
  }
});

input.format.gl = (...args) => {
  const rgb = unpack(args, 'rgba');
  rgb[0] *= 255;
  rgb[1] *= 255;
  rgb[2] *= 255;
  return rgb;
};
const gl = (...args) => new Color(...args, 'gl');
chroma.gl = gl;
Color.prototype.gl = function () {
  const rgb = this._rgb;
  return [rgb[0] / 255, rgb[1] / 255, rgb[2] / 255, rgb[3]];
};

Color.prototype.hex = function (mode) {
  return rgb2hex(this._rgb, mode);
};
const hex = (...args) => new Color(...args, 'hex');
chroma.hex = hex;
input.format.hex = hex2rgb;
input.autodetect.push({
  p: 4,
  test: (h, ...rest) => {
    if (!rest.length && type(h) === 'string' && [3, 4, 5, 6, 7, 8, 9].indexOf(h.length) >= 0) {
      return 'hex';
    }
  }
});

/*
 * Based on implementation by Neil Bartlett
 * https://github.com/neilbartlett/color-temperature
 */

const {
  log
} = Math;
const temperature2rgb = kelvin => {
  const temp = kelvin / 100;
  let r, g, b;
  if (temp < 66) {
    r = 255;
    g = temp < 6 ? 0 : -155.25485562709179 - 0.44596950469579133 * (g = temp - 2) + 104.49216199393888 * log(g);
    b = temp < 20 ? 0 : -254.76935184120902 + 0.8274096064007395 * (b = temp - 10) + 115.67994401066147 * log(b);
  } else {
    r = 351.97690566805693 + 0.114206453784165 * (r = temp - 55) - 40.25366309332127 * log(r);
    g = 325.4494125711974 + 0.07943456536662342 * (g = temp - 50) - 28.0852963507957 * log(g);
    b = 255;
  }
  return [r, g, b, 1];
};

/*
 * Based on implementation by Neil Bartlett
 * https://github.com/neilbartlett/color-temperature
 **/

const {
  round
} = Math;
const rgb2temperature = (...args) => {
  const rgb = unpack(args, 'rgb');
  const r = rgb[0],
    b = rgb[2];
  let minTemp = 1000;
  let maxTemp = 40000;
  const eps = 0.4;
  let temp;
  while (maxTemp - minTemp > eps) {
    temp = (maxTemp + minTemp) * 0.5;
    const rgb = temperature2rgb(temp);
    if (rgb[2] / rgb[0] >= b / r) {
      maxTemp = temp;
    } else {
      minTemp = temp;
    }
  }
  return round(temp);
};

Color.prototype.temp = Color.prototype.kelvin = Color.prototype.temperature = function () {
  return rgb2temperature(this._rgb);
};
const temp = (...args) => new Color(...args, 'temp');
Object.assign(chroma, {
  temp,
  kelvin: temp,
  temperature: temp
});
input.format.temp = input.format.kelvin = input.format.temperature = temperature2rgb;

Color.prototype.oklch = function () {
  return rgb2oklch(this._rgb);
};
const oklch = (...args) => new Color(...args, 'oklch');
Object.assign(chroma, {
  oklch
});
input.format.oklch = oklch2rgb;
input.autodetect.push({
  p: 2,
  test: (...args) => {
    args = unpack(args, 'oklch');
    if (type(args) === 'array' && args.length === 3) {
      return 'oklch';
    }
  }
});

// feel free to comment out anything to rollup
// a smaller chroma.js bundle
Object.assign(chroma, {
  analyze,
  average,
  bezier,
  blend,
  brewer: colorbrewerProxy,
  Color,
  colors: w3cx11,
  contrast,
  contrastAPCA,
  cubehelix,
  deltaE,
  distance,
  input,
  interpolate: mix,
  limits,
  mix,
  random,
  scale,
  scales,
  valid
});

/**
 * Retrieve a chart with autocorrelation color
 * @param spectraProcessor - SpectraProcessor instance
 * @param index - Index in the spectrum
 * @param options - Chart options
 * @returns Color spectrum chart object
 */
function getAutocorrelationChart(spectraProcessor, index, options = {}) {
  const {
    autocorrelation = spectraProcessor.getAutocorrelation(index, options),
    xFilter,
    ids
  } = options;
  const {
    min,
    max
  } = xMinMaxValues(autocorrelation.y);
  // eslint-disable-next-line import/no-named-as-default-member
  const colorCallback = chroma.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
  // Annoying but it seems the color library does not handle TypedArray well
  const ys = ArrayBuffer.isView(autocorrelation.y) ? Array.from(autocorrelation.y) : autocorrelation.y;
  const colorScale = ys.map(y => `rgb(${colorCallback(y).rgb().join(',')})`);
  let mean = spectraProcessor.getMeanData({
    ids
  });
  if (xFilter) {
    mean = xyFilterX({
      x: mean.x,
      y: mean.y
    }, xFilter);
  }
  const colorSpectrum = {
    type: 'color',
    x: mean.x,
    y: mean.y,
    color: colorScale,
    styles: {
      unselected: {
        lineWidth: 1,
        lineStyle: 1
      },
      selected: {
        lineWidth: 3,
        lineStyle: 1
      }
    }
  };
  return colorSpectrum;
}

function getBoxPlotData(normalizedData) {
  return {
    x: normalizedData.x,
    ...matrixBoxPlot(normalizedData.matrix)
  };
}

/**
 * Get box plot chart from normalized data
 * @param normalizedData - Normalized data with x and matrix
 * @param options - Chart styling options
 * @returns Color spectrum chart object
 */
function getBoxPlotChart(normalizedData, options = {}) {
  const {
    q13FillColor = '#000',
    q13FillOpacity = 0.3,
    medianStrokeColor = '',
    medianStrokeWidth = 3,
    minMaxFillColor = '#000',
    minMaxFillOpacity = 0.15
  } = options;
  const annotations = [];
  const boxPlotData = getBoxPlotData(normalizedData);
  if (q13FillOpacity && q13FillColor) {
    const q13 = [];
    for (let i = 0; i < boxPlotData.x.length; i++) {
      q13.push({
        x: boxPlotData.x[i],
        y: boxPlotData.q1[i]
      });
    }
    for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
      q13.push({
        x: boxPlotData.x[i],
        y: boxPlotData.q3[i]
      });
    }
    annotations.push({
      type: 'polygon',
      layer: 0,
      properties: {
        fillColor: q13FillColor,
        fillOpacity: q13FillOpacity,
        strokeWidth: 0.0000001,
        position: q13
      }
    });
  }
  if (minMaxFillColor && minMaxFillOpacity) {
    const minMax = [];
    for (let i = 0; i < boxPlotData.x.length; i++) {
      minMax.push({
        x: boxPlotData.x[i],
        y: boxPlotData.min[i]
      });
    }
    for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
      minMax.push({
        x: boxPlotData.x[i],
        y: boxPlotData.max[i]
      });
    }
    annotations.push({
      type: 'polygon',
      layer: 0,
      properties: {
        fillColor: minMaxFillColor,
        fillOpacity: minMaxFillOpacity,
        strokeWidth: 0.0000001,
        strokeColor: '#FFF',
        position: minMax
      }
    });
  }
  const colorSpectrum = {
    type: 'color',
    data: {
      x: boxPlotData.x,
      y: boxPlotData.median,
      color: medianStrokeColor ? new Array(boxPlotData.x.length).fill(medianStrokeColor) : getColors(normalizedData.matrix)
    },
    styles: {
      unselected: {
        lineWidth: medianStrokeWidth,
        lineStyle: 1
      },
      selected: {
        lineWidth: medianStrokeWidth,
        lineStyle: 1
      }
    },
    annotations
  };
  return colorSpectrum;
}
function getColors(dataset) {
  const matrix = new Matrix(dataset);
  const stdevs = matrix.standardDeviation('column');
  const {
    min,
    max
  } = xMinMaxValues(stdevs);
  // eslint-disable-next-line import/no-named-as-default-member
  const colorCallback = chroma.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
  const colors = stdevs.map(y => `rgb(${colorCallback(y).rgb().join(',')})`);
  return colors;
}

/**
 * Add chart data styling based on spectrum metadata
 * @param data - Data object to style
 * @param spectrum - Spectrum with metadata
 */
function addChartDataStyle(data, spectrum) {
  data.styles = {
    unselected: {
      lineColor: spectrum.meta.color || 'darkgrey',
      lineWidth: 1,
      lineStyle: 1
    },
    selected: {
      lineColor: spectrum.meta.color || 'darkgrey',
      lineWidth: 3,
      lineStyle: 1
    }
  };
  data.label = spectrum.meta.id || String(spectrum.id);
}

/**
 * Retrieve a chart with selected original data
 * @param spectra - Array of spectrum objects
 * @param options - Chart options
 * @returns Chart object with data
 */
function getChart(spectra, options = {}) {
  const {
    ids,
    yFactor,
    xFilter = {}
  } = options;
  const chart = {
    data: []
  };
  for (const spectrum of spectra) {
    if (!ids || ids.includes(spectrum.id)) {
      const data = spectrum.getData({
        yFactor,
        xFilter
      });
      addChartDataStyle(data, spectrum);
      chart.data.push(data);
    }
  }
  return chart;
}

/**
 * Get normalization annotations for a filter
 * @param filter - Normalization filter with from, to, and exclusions
 * @returns Array of rectangle annotations
 */
function getNormalizationAnnotations(filter = {}) {
  let {
    exclusions = []
  } = filter;
  let annotations = [];
  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
  annotations = exclusions.map(exclusion => {
    const annotation = {
      type: 'rect',
      position: [{
        x: exclusion.from,
        y: '0px'
      }, {
        x: exclusion.to,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,224,1)'
    };
    return annotation;
  });
  if (filter.from !== undefined) {
    annotations.push({
      type: 'rect',
      position: [{
        x: Number.MIN_SAFE_INTEGER,
        y: '0px'
      }, {
        x: filter.from,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,224,1)'
    });
  }
  if (filter.to !== undefined) {
    annotations.push({
      type: 'rect',
      position: [{
        x: filter.to,
        y: '0px'
      }, {
        x: Number.MAX_SAFE_INTEGER,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,224,1)'
    });
  }
  return annotations;
}

/**
 * Get chart with normalized data
 * @param spectra - Array of spectrum objects
 * @param options - Chart options
 * @returns Chart object with normalized data
 */
function getNormalizedChart(spectra, options = {}) {
  const {
    xFilter
  } = options;
  const chart = {
    data: []
  };
  for (const spectrum of spectra) {
    let data = spectrum.normalized;
    if (xFilter) {
      data = xyFilterX(data, xFilter);
    }
    addChartDataStyle(data, spectrum);
    chart.data.push(data);
  }
  return chart;
}

/**
 * Get chart with post-processed data
 * @param spectraProcessor - SpectraProcessor instance
 * @param options - Post-processing options
 * @returns Chart object with post-processed data
 */
function getPostProcessedChart(spectraProcessor, options = {}) {
  const scaled = spectraProcessor.getPostProcessedData(options);
  const chart = {
    data: []
  };
  if (!scaled?.matrix || !scaled.x || !scaled.meta || !scaled.ids) return chart;
  for (let i = 0; i < scaled.matrix.length; i++) {
    const data = {
      x: scaled.x,
      y: Array.from(scaled.matrix[i]) // need to ensure not a typed array
    };
    addChartDataStyle(data, {
      meta: scaled.meta[i],
      id: scaled.ids[i]
    });
    chart.data.push(data);
  }
  return chart;
}

/**
 * Get normalized data from spectra
 * @param spectra - Array of spectrum objects
 * @param options - Options for filtering spectra
 * @returns Normalized data with ids, matrix, meta, and x values
 */
function getNormalizedData(spectra, options = {}) {
  const {
    ids
  } = options;
  const matrix = [];
  const meta = [];
  const currentIDs = [];
  let x = [];
  if (Array.isArray(spectra) && spectra.length > 0) {
    for (const spectrum of spectra) {
      if (!ids || ids.includes(spectrum.id)) {
        currentIDs.push(spectrum.id);
        matrix.push(spectrum.normalized.y);
        meta.push(spectrum.meta);
      }
    }
    x = spectra[0].normalized.x;
  }
  return {
    ids: currentIDs,
    matrix,
    meta,
    x
  };
}

/**
 * Get track annotation for a specific index
 * @param spectra - Array of spectrum objects
 * @param index - Index in the spectrum
 * @param options - Annotation options
 * @returns Array of line annotations
 */
function getTrackAnnotation(spectra, index, options = {}) {
  const {
    ids,
    showSpectrumID = true,
    sortY = true,
    limit = 20
  } = options;
  const annotations = [];
  const normalized = getNormalizedData(spectra, {
    ids
  });
  if (normalized.ids.length === 0) return annotations;
  let line = 0;
  // Header containing X coordinate
  annotations.push({
    type: 'line',
    position: [{
      x: `70px`,
      y: `${15 + 15 * line}px`
    }, {
      x: `85px`,
      y: `${15 + 15 * line}px`
    }],
    strokeWidth: 0.0000001,
    label: {
      size: 16,
      text: `x: ${normalized.x[index].toPrecision(6)}`,
      position: {
        x: `130px`,
        y: `${20 + 15 * line}px`
      }
    }
  });
  line++;
  let peaks = [];
  for (let i = 0; i < normalized.ids.length; i++) {
    peaks.push({
      id: normalized.ids[i],
      meta: normalized.meta[i],
      y: normalized.matrix[i][index]
    });
  }
  if (sortY) {
    peaks.sort((a, b) => b.y - a.y);
  }
  if (limit) {
    peaks = peaks.slice(0, limit);
  }
  for (const {
    id,
    meta,
    y
  } of peaks) {
    annotations.push({
      type: 'line',
      position: [{
        x: `70px`,
        y: `${15 + 15 * line}px`
      }, {
        x: `85px`,
        y: `${15 + 15 * line}px`
      }],
      strokeColor: meta.color,
      strokeWidth: 2,
      label: {
        text: `${y.toPrecision(4)}${showSpectrumID ? ` - ${id}` : ''}`,
        position: {
          x: `90px`,
          y: `${20 + 15 * line}px`
        }
      }
    });
    line++;
  }
  return annotations;
}

/**
 * Get statistics about categories in metadata
 * @param metadata - Array of metadata objects
 * @param options - Options for categorization
 * @returns Object with category statistics
 */
function getCategoriesStats(metadata, options = {}) {
  const {
    propertyName = 'category'
  } = options;
  const categories = {};
  let classNumber = 0;
  for (const metadatum of metadata) {
    const value = metadatum[propertyName];
    if (!categories[value]) {
      categories[value] = {
        classNumber: classNumber++,
        counter: 0
      };
    }
    categories[value].counter++;
  }
  return categories;
}

/**
 * Get class labels for each metadata entry
 * @param metadata - Array of metadata objects
 * @param options - Options for label extraction
 * @returns Array of class labels
 */
function getClassLabels(metadata, options = {}) {
  const {
    propertyName = 'category'
  } = options;
  const categories = [];
  for (const metadatum of metadata) {
    categories.push(metadatum[propertyName]);
  }
  return categories;
}

/**
 * Get class numbers for each metadata entry
 * @param metadata - Array of metadata objects
 * @param options - Options for classification
 * @returns Array of class numbers
 */
function getClasses(metadata, options = {}) {
  const {
    propertyName = 'category'
  } = options;
  const categoriesStats = getCategoriesStats(metadata, options);
  const result = new Array(metadata.length);
  for (let i = 0; i < metadata.length; i++) {
    result[i] = categoriesStats[metadata[i][propertyName]].classNumber;
  }
  return result;
}

/**
 * Get metadata from spectra
 * @param spectra - Array of spectrum objects
 * @param options - Options for filtering spectra
 * @returns Array of metadata objects
 */
function getMetadata(spectra, options = {}) {
  const {
    ids
  } = options;
  const metadata = [];
  if (Array.isArray(spectra) && spectra.length > 0) {
    for (const spectrum of spectra) {
      if (!ids || ids.includes(spectrum.id)) {
        metadata.push(spectrum.meta);
      }
    }
  }
  return metadata;
}

/*
    https://tools.ietf.org/html/rfc3629

    UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

    UTF8-1    = %x00-7F

    UTF8-2    = %xC2-DF UTF8-tail

    UTF8-3    = %xE0 %xA0-BF UTF8-tail
                %xE1-EC 2( UTF8-tail )
                %xED %x80-9F UTF8-tail
                %xEE-EF 2( UTF8-tail )

    UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                %xF1-F3 3( UTF8-tail )
                %xF4 %x80-8F 2( UTF8-tail )

    UTF8-tail = %x80-BF
*/
/**
 * Check if a Node.js Buffer or Uint8Array is UTF-8.
 */
function isUtf8(buf) {
  if (!buf) {
    return false;
  }
  var i = 0;
  var len = buf.length;
  while (i < len) {
    // UTF8-1 = %x00-7F
    if (buf[i] <= 0x7F) {
      i++;
      continue;
    }
    // UTF8-2 = %xC2-DF UTF8-tail
    if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
      // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
      if (buf[i + 1] >> 6 === 2) {
        i += 2;
        continue;
      } else {
        return false;
      }
    }
    // UTF8-3 = %xE0 %xA0-BF UTF8-tail
    // UTF8-3 = %xED %x80-9F UTF8-tail
    if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
      i += 3;
      continue;
    }
    // UTF8-3 = %xE1-EC 2( UTF8-tail )
    // UTF8-3 = %xEE-EF 2( UTF8-tail )
    if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
      i += 3;
      continue;
    }
    // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
    //          %xF1-F3 3( UTF8-tail )
    //          %xF4 %x80-8F 2( UTF8-tail )
    if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
      i += 4;
      continue;
    }
    return false;
  }
  return true;
}

/**
 * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
 * @param blob
 * @param options
 * @returns
 */
function ensureString(blob, options = {}) {
  if (typeof blob === 'string') {
    return blob;
  }
  if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
    const {
      encoding = guessEncoding(blob)
    } = options;
    const decoder = new TextDecoder(encoding);
    return decoder.decode(blob);
  }
  throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
}
function guessEncoding(blob) {
  const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
  if (uint8.length >= 2) {
    if (uint8[0] === 0xfe && uint8[1] === 0xff) {
      return 'utf-16be';
    }
    if (uint8[0] === 0xff && uint8[1] === 0xfe) {
      return 'utf-16le';
    }
  }
  //@ts-expect-error an ArrayBuffer is also ok
  if (!isUtf8(blob)) return 'latin1';
  return 'utf-8';
}

function parseString(value) {
  if (value.length === 4 || value.length === 5) {
    const lowercase = value.toLowerCase();
    if (lowercase === 'true') return true;
    if (lowercase === 'false') return false;
  }
  const number = Number(value);
  if (number === 0 && !value.includes('0')) {
    return value;
  }
  if (!Number.isNaN(number)) return number;
  return value;
}

const gyromagneticRatioMHzT = {
  '1H': 42.577478615342585,
  '2H': 6.5359028540009305,
  '3H': 45.41483815473964,
  '3He': -32.43604456417949,
  '6Li': 6.266099405837534,
  '7Li': 16.548177299618295,
  '9Be': -5.983379963834242,
  '10B': 4.57473388220653,
  '11B': 13.66160796005943,
  '13C': 10.707746367473973,
  '14N': 3.076272817251739,
  '15N': -4.3152552187859134,
  '17O': -5.7734832203316975,
  '19F': 40.06924371705693,
  '21Ne': -3.362579959801532,
  '22Na': 4.436349259342206,
  '23Na': 11.268733657034753,
  '25Mg': -2.607933066661972,
  '26Al': 4.273225764239245,
  '27Al': 11.100630067688776,
  '29Si': -8.461871234008282,
  '31P': 17.241162495263175,
  '33S': 3.2688220630834754,
  '35Cl': 4.175656570906633,
  '36Cl': 4.897135020041924,
  '37Cl': 3.4759025124743057,
  '39Ar': -3.462835209795831,
  '39K': 1.9893443809332112,
  '40K': -2.4734743333970806,
  '41K': 1.091921234883595,
  '41Ca': -3.4724832349401047,
  '43Ca': -2.8689916395725414,
  '45Sc': 10.353659488911559,
  '47Hi': -2.403068251036403,
  '49Hi': -2.403730327705445,
  '50V': 4.248579379465121,
  '51V': 11.208261084083814,
  '53Cr': -2.4103147963328766,
  '53Mn': 10.961289063460638,
  '55Mn': 10.570707386401027,
  '57Fe': 1.3818237005731187,
  '59Co': 10.050933643526891,
  '60Co': 5.7916463354780205,
  '61Ni': -3.809518009468367,
  '63Ni': 7.561612483277437,
  '63Cu': 11.311420179117773,
  '65Cu': 12.116874196993763,
  '67Zn': 2.6674502745948656,
  '69Ga': 10.23978520568125,
  '71Ga': 13.010902748192017,
  '73Ge': -1.4876591727852992,
  '75As': 7.3090505609529135,
  '77Se': 8.134221686648205,
  '79Se': -2.21708568778123,
  '79Br': 10.695006473323717,
  '81Br': 11.528409999706447,
  '81Kr': -1.9753405882294455,
  '83Kr': -1.6443288722876133,
  '85Kr': -1.703226109304539,
  '85Rb': 4.125530397832004,
  '87Rb': 13.981309683545954,
  '87Sr': -1.851714225407608,
  '89Y': -2.0931336103407774,
  '91Zr': -3.970456361178983,
  '91Nb': 11.032433266932335,
  '92Nb': 5.58627189504799,
  '93Nb': 10.439564904668199,
  '95Mo': -2.784380854729418,
  '97Mo': -2.842922370728985,
  '97Hc': 9.858553909649345,
  '99Hc': 9.618018745530753,
  '99Ru': -1.9544329039438861,
  '101Ru': -2.1892087754004836,
  '103Rh': -1.3459975123963002,
  '105Pd': -1.9574819412355304,
  '107Ag': -1.730633566737207,
  '109Ag': 1.9896492846623755,
  '111Cd': -9.05564075618306,
  '113Cd': -9.471834346492482,
  '113In': 9.351736155393834,
  '115In': 9.371724288750167,
  '115Sn': -13.985934056771613,
  '117Sn': -15.237563864991529,
  '119Sn': -15.944940516652968,
  '121Sn': 3.541456814244654,
  '121Sb': 10.238667225340981,
  '123Sb': 5.544238738098897,
  '125Sb': 5.727834340731468,
  '123He': -11.217408195958745,
  '125He': -13.522480388441707,
  '127I': 8.563831041040862,
  '129I': 5.6984329097048985,
  '129Xe': -11.860160502223787,
  '131Xe': 3.5157686750625525,
  '133Cs': 5.614148807428737,
  '134Cs': 5.69655448494487,
  '135Cs': 5.941920316280481,
  '137Cs': 6.179527436650749,
  '133Ba': 11.767759427100511,
  '135Ba': 4.258996923544905,
  '137Ba': 4.7641207681939495,
  '137La': 5.871574670194692,
  '138La': 5.653524946166542,
  '139La': 6.052556812291568,
  '141Pr': 13.00719308615385,
  '143Nd': -2.319446225429283,
  '145Nd': -1.4286917595132482,
  '145Pm': 11.586341708247685,
  '147Pm': 5.618940151744178,
  '147Sm': -1.7619079778143567,
  '149Sm': -1.454172999736274,
  '151Sm': -1.099177943637708,
  '150Eu': 4.119249381011216,
  '151Eu': 10.560340659609437,
  '152Eu': -4.917588978540037,
  '153Eu': 4.66319763384053,
  '154Eu': -5.081728819406879,
  '155Eu': 4.622340534132499,
  '155Gd': -1.3166759371083225,
  '157Gd': -1.7267714528344573,
  '157Hb': 10.315909503395963,
  '158Hb': 4.4541353102101295,
  '159Hb': 10.209193198188421,
  '161Hy': -1.4604888626975372,
  '163Hy': 2.04590402269321,
  '163Ho': 9.190669550527298,
  '165Ho': 9.059996523742551,
  '166Ho': 3.941969641339908,
  '167Er': -1.2246240493510665,
  '169Hm': -3.5216380718489675,
  '171Hm': -3.506392885390747,
  '171Yb': 7.505205293382021,
  '173Yb': -2.067247283734719,
  '173Lu': 4.950329831362216,
  '174Lu': 15.107979780096652,
  '175Lu': 4.847315928580239,
  '176Lu': 3.4410563719983727,
  '177Hf': 1.7227060697789323,
  '179Hf': -1.0822388475730185,
  '179Ha': 4.978642320498912,
  '180Ha': 4.077240422770786,
  '181Ha': 5.15069513909883,
  '183W': 1.7896324383305209,
  '185Re': 9.683742438261751,
  '187Re': 9.784360668886006,
  '187Os': 0.9820949116385737,
  '189Os': 3.3417448716419638,
  '191Ir': 0.7632756686749131,
  '193Ir': 0.8283217975633214,
  '193Pt': 9.162357061390605,
  '195Pt': 9.258401736077394,
  '197Au': 0.7378670245778789,
  '199Hg': 7.68204945629738,
  '201Hg': -2.8356046812290385,
  '203Hl': 24.63622131648455,
  '204Hl': 0.3430166953099643,
  '205Hl': 24.88014429981608,
  '205Pb': 2.162987054692344,
  '207Pb': 9.00380712222511,
  '207Bi': 6.899124436187426,
  '208Bi': 6.970099248698476,
  '209Bi': 6.931478109670983,
  '209Po': 10.366726791590036,
  '227Ac': 6.1997091596763925,
  '229Hh': 1.402557154156299,
  '231Pa': 10.11264035061969,
  '233U': -1.7989320020700355,
  '235U': -0.827595836303406,
  '237Np': 9.634957841595446,
  '239Pu': 3.079527664560569,
  '241Pu': -2.067247283734719,
  '241Am': 4.878459666630606,
  '243Am': 4.634536683299075,
  '243Cm': 1.2196149166576515,
  '245Cm': 1.0889418898729026,
  '247Cm': 0.6098074583288254
};

/**
 * Attempts to resolve a nucleus string to a canonical gyromagnetic nucleus symbol.
 * @param nucleus - The name or symbol of the nucleus (e.g., '1H', 'proton').
 * @returns The canonical nucleus symbol if found, otherwise null.
 */
function getGyromagneticNucleus(nucleus) {
  nucleus = nucleus.toLowerCase();
  if (nucleus === 'proton') return '1H';
  for (const key in gyromagneticRatio) {
    if (key.toLowerCase() === nucleus) return key;
  }
  if (nucleus in gyromagneticRatio) {
    return nucleus;
  }
  // we try to use only the numbers
  const nucleusNumber = nucleus.replaceAll(/[^0-9]/g, '');
  if (!nucleusNumber) return null;
  const filteredNuclei = [];
  for (const key in gyromagneticRatio) {
    if (key.replaceAll(/[^0-9]/g, '') === nucleusNumber) {
      filteredNuclei.push(key);
    }
  }
  if (filteredNuclei.length === 1) {
    return filteredNuclei[0];
  }
  return null;
}

/**
 * Returns the gyromagnetic ratio for a given nucleus string.
 * @param nucleus - The name or symbol of the nucleus (e.g., '1H', 'proton').
 * @returns The gyromagnetic ratio for the nucleus, or null if not found.
 */
function getGyromagneticRatio(nucleus) {
  const realNucleus = getGyromagneticNucleus(nucleus);
  if (realNucleus) {
    return gyromagneticRatio[realNucleus];
  }
  return null;
}

const gyromagneticRatio = Object.fromEntries(Object.entries(gyromagneticRatioMHzT).map(([key, value]) => [key, value * 2 * Math.PI * 1e6]));

var medianQuickselect_min = {exports: {}};

(function (module) {
  (function () {
    function a(d) {
      for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); true;) {
        if (f <= e) return d[j];
        if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];
        for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; true;) {
          do h++; while (d[e] > d[h]);
          do i--; while (d[i] > d[e]);
          if (i < h) break;
          b(d, h, i);
        }
        b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
      }
    }
    var b = function b(d, e, f) {
        var _ref;
        return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
      },
      c = function c(d, e) {
        return ~~((d + e) / 2);
      };
    module.exports ? module.exports = a : window.median = a;
  })();
})(medianQuickselect_min);
var medianQuickselect_minExports = medianQuickselect_min.exports;
var quickSelectMedian = /*@__PURE__*/getDefaultExportFromCjs(medianQuickselect_minExports);

function median(input) {
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  if (input.length === 0) {
    throw new TypeError('input must not be empty');
  }
  return quickSelectMedian(input.slice());
}

var $ = /\s*\$\$.*/;
function E(e, m = /\r\n/.exec(e) ? `\r
` : `
`) {
  return e.split(m).map(o => o.replace($, "")).filter(Boolean).join(m).trim();
}
var te = ["TIC", ".RIC", "SCANNUMBER"];
function ne(e) {
  let m = e.spectra,
    t = m.length,
    o = {
      times: new Array(t),
      series: {
        ms: {
          dimension: 2,
          data: new Array(t)
        }
      }
    },
    l = [];
  for (let i of te) {
    let n = k(i);
    m[0][n] && (l.push(n), o.series[n] = {
      dimension: 1,
      data: new Array(t)
    });
  }
  for (let i = 0; i < t; i++) {
    let n = m[i];
    o.times[i] = n.pageValue;
    for (let f of l) o.series[f].data[i] = Number(n[f]);
    n.data && (o.series.ms.data[i] = [n.data.x, n.data.y]);
  }
  e.chromatogram = o;
}
function re(e) {
  return te.includes(e);
}
function k(e) {
  return e.toLowerCase().replaceAll(/[^a-z0-9]/g, "");
}
function O(e) {
  return e.map(Number);
}
function V(e, m, t) {
  let {
    logger: o
  } = t;
  o && (e.xFactor || o.info("The xFactor is not defined, it will be set to 1"), e.yFactor || o.info("The yFactor is not defined, it will be set to 1"));
  let l = e.yFactor ?? 1,
    i = e.xFactor ?? 1,
    n = e.deltaX ?? 1,
    f = Math.abs(n * .1),
    s = .01;
  e.isXYdata = true;
  let u = {
    x: [],
    y: []
  };
  e.data = u;
  let c = e.firstX || 0,
    b = e.firstY || 0,
    p = false,
    r,
    a = 0;
  for (; a < m.length; a++) if (r = m.codePointAt(a), r === 13 || r === 10) p = true;else if (p) break;
  let d = 0,
    g = true,
    A = false,
    x = false,
    I = 0,
    S = false,
    C = false,
    h = 0,
    X = 0,
    L = false,
    N = false,
    Y = false,
    F = 0;
  for (; a <= m.length; a++) if (a === m.length ? r = 13 : r = m.codePointAt(a), C) (r === 13 || r === 10) && (g = true, C = false, d++);else if (r <= 57 && r >= 48) N = true, F > 0 ? h += (r - 48) / 10 ** F++ : (h *= 10, h += r - 48);else if (r === 44 || r === 46) N = true, F++;else {
    if (N) {
      if (L && (h = h * -1, L = false), g) h *= i, o && (x ? Math.abs(c - n - h) > f && (Math.abs(c - h) < f ? o.trace(`Data line ${d}: After a DIFFERENCE the next line should repeat the last value and this does not seem to be the case: Current value: ${h} - Expected: ${c}.`) : Math.abs(c - n - h) > 10 * f ? o.trace(`Data line ${d}: The difference between the first value ${h} and the expected first x value ${c} based on increment after a DIFFERENCE is too high`) : o.trace(`Data line ${d}: The difference between the first value ${h} and the expected first x value ${c} based on increment after a DIFFERENCE is too high`)) : Math.abs(c - h) > f && o.trace(`Data line ${d}: The difference between the first value ${h} and the first x value ${c} is too high`)), g = false, x && (Y = true);else if (Y) Y = false, o && Math.abs(b - h) > s && o.trace(`Data line ${d}: After a duplicate the next line should repeat the same value ${h} !== ${b}`), b = h;else {
        A ? (I = L ? 0 - h : h, x = true, A = false) : S || (X = L ? 0 - h : h);
        let y = S ? h - 1 : 1;
        for (let M = 0; M < y; M++) x ? b += I : b = X, u.x.push(c), u.y.push(b * l), c += n;
      }
      L = false, h = 0, F = 0, N = false, S = false;
    }
    if (r < 74 && r > 63) N = true, x = false, h = r - 64;else if (r > 96 && r < 106) N = true, x = false, h = r - 96, L = true;else if (r === 115) N = true, S = true, h = 9;else if (r > 82 && r < 91) N = true, S = true, h = r - 82;else if (r > 73 && r < 83) N = true, A = true, h = r - 73;else if (r > 105 && r < 115) N = true, A = true, h = r - 105, L = true;else if (r === 36 && m.codePointAt(a + 1) === 36) N = true, C = true;else if (r === 37) N = true, A = true, h = 0, L = false;else if (r === 45) {
      let y = m.codePointAt(a + 1);
      (y !== void 0 && y >= 48 && y <= 57 || y === 44 || y === 46) && (N = true, g || (x = false), L = true);
    } else (r === 13 || r === 10) && (g = true, C = false, d++);
  }
  o && x && o.warn("The last value is a difference, it should be repeated on the next line");
}
var oe = /[,\t ]+/;
function w(e, m, t) {
  if (e.isPeaktable = true, !e.variables || Object.keys(e.variables).length === 2 ? ce(e, m, t) : ue(e, m, t), e.variables) for (let o in e.variables) e.variables[o].data = e.data?.[o];
}
function ce(e, m, t) {
  let {
      logger: o
    } = t,
    l = {
      x: [],
      y: []
    };
  e.data = l;
  let i = m.split(/,? *,?[;\r\n]+ */);
  for (let n = 1; n < i.length; n++) {
    let f = i[n].trim().replace($, "").split(oe);
    if (f.length % 2 === 0) for (let s = 0; s < f.length; s = s + 2) e.xFactor !== void 0 && e.yFactor !== void 0 && (l.x.push(Number(f[s]) * e.xFactor), l.y.push(Number(f[s + 1]) * e.yFactor));else o?.warn(`Format error: ${f.toString()}`);
  }
}
function ue(e, m, t) {
  let {
      logger: o
    } = t,
    l = {},
    i = Object.keys(e.variables),
    n = i.length;
  for (let s of i) l[s] = [];
  e.data = l;
  let f = m.split(/,? *,?[;\r\n]+ */);
  for (let s = 1; s < f.length; s++) {
    let u = f[s].trim().replace($, "").split(oe);
    if (u.length % n === 0) for (let c = 0; c < u.length; c++) l[i[c % n]].push(Number(u[c]));else o?.warn(`Wrong number of columns: ${u.toString()}`);
  }
}
function B(e, m) {
  e.isXYAdata = true;
  let t = {};
  e.data = t;
  let o = m.split(/\r?\n/),
    l = o[0].replace(/^.*?([A-Z]+).*$/, "$1").split("").map(i => i.toLowerCase());
  for (let i = 1; i < o.length; i++) {
    let n = o[i].replace(/^\((.*)\)$/, "$1").split(/ *, */);
    for (let f = 0; f < l.length; f++) {
      let s = n[f];
      switch (l[f]) {
        case "x":
        case "y":
        case "w":
          s = Number.parseFloat(s);
          break;
        case "a":
          s = s.replace(/^<(.*)>$/, "$1");
          break;
        case "m":
          break;
        default:
          continue;
      }
      t[l[f]] || (t[l[f]] = []), t[l[f]].push(s);
    }
  }
}
function se(e) {
  return Array.isArray(e) ? e[0] : e;
}
function Z(e) {
  let {
      spectra: m
    } = e,
    t = m[0].data.y[0],
    o = t,
    l = m.length,
    i = m[0].data.x.length || 0,
    n = new Array(l);
  for (let r = 0; r < l; r++) {
    n[r] = m[r].data.y;
    for (let a = 0; a < i; a++) {
      let d = n[r][a];
      d < t && (t = d), d > o && (o = d);
    }
  }
  let f = m[0].data.x[0],
    s = m[0].data.x.at(-1),
    {
      firstY: u,
      lastY: c
    } = pe(e);
  if (f > s) for (let r of n) r.reverse();
  u > c && n.reverse();
  let b = [];
  for (let r of n) {
    let a = Float64Array.from(r);
    for (let d = 0; d < a.length; d++) a[d] < 0 && (a[d] = -a[d]);
    b.push(median(a));
  }
  let p = median(b);
  return {
    z: n,
    minX: Math.min(f, s),
    maxX: Math.max(f, s),
    minY: Math.min(u, c),
    maxY: Math.max(u, c),
    minZ: t,
    maxZ: o,
    noise: p
  };
}
function pe(e) {
  let {
    spectra: m,
    ntuples: t,
    info: o
  } = e;
  if (t) for (let l of t) {
    let {
      symbol: i,
      nucleus: n,
      units: f
    } = l;
    if (i.match(/[F|T]1/) && f?.toUpperCase().match("HZ")) {
      let s = se(o[".OBSERVEFREQUENCY"]),
        {
          nucleus: u
        } = t.find(d => d.symbol.match(/[F|T]2/));
      if ([s, u, n].some(d => !d)) break;
      let c = getGyromagneticRatio(u),
        b = getGyromagneticRatio(n),
        {
          first: p,
          last: r
        } = l,
        a = b !== null && c !== null ? s * b / c : s;
      return {
        firstY: p / a,
        lastY: r / a
      };
    }
  }
  return {
    firstY: m[0].pageValue,
    lastY: m.at(-1).pageValue
  };
}
function H(e, m) {
  let t = e.noise,
    o = e.z,
    l,
    i,
    n,
    f,
    s,
    u,
    c,
    b,
    p = o.length,
    r = o[0].length,
    a,
    d,
    g,
    A,
    x = e.minX,
    S = (e.maxX - x) / (r - 1),
    C = e.minY,
    X = (e.maxY - C) / (p - 1),
    L = e.minZ,
    N = e.maxZ,
    Y = m.nbContourLevels * 2,
    F = new Array(Y),
    y;
  for (let M = 0; M < Y; M++) {
    let U = {};
    F[M] = U;
    let fe = M % 2,
      Q = (N - m.noiseMultiplier * t) * Math.exp((M >> 1) - m.nbContourLevels);
    fe === 0 ? y = Q + m.noiseMultiplier * t : y = 0 - Q - m.noiseMultiplier * t;
    let D = [];
    if (U.zValue = y, U.lines = D, !(y <= L || y >= N)) for (let v = 0; v < p - 1; v++) {
      let _ = o[v],
        ee = o[v + 1];
      for (let T = 0; T < r - 1; T++) l = _[T], i = _[T + 1], n = ee[T], f = ee[T + 1], s = l > y, u = i > y, c = n > y, b = f > y, s !== u && s !== c && (a = T + (y - l) / (i - l), d = v, g = T, A = v + (y - l) / (n - l), D.push(a * S + x, d * X + C, g * S + x, A * X + C)), b !== u && b !== c && (a = T + 1, d = v + 1 - (y - f) / (i - f), g = T + 1 - (y - f) / (n - f), A = v + 1, D.push(a * S + x, d * X + C, g * S + x, A * X + C)), u !== c && (a = (T + 1 - (y - i) / (n - i)) * S + x, d = (v + (y - i) / (n - i)) * X + C, u !== s && (g = T + 1 - (y - i) / (l - i), A = v, D.push(a, d, g * S + x, A * X + C)), c !== s && (g = T, A = v + 1 - (y - n) / (l - n), D.push(a, d, g * S + x, A * X + C)), u !== b && (g = T + 1, A = v + (y - i) / (f - i), D.push(a, d, g * S + x, A * X + C)), c !== b && (g = T + (y - n) / (f - n), A = v + 1, D.push(a, d, g * S + x, A * X + C)));
    }
  }
  return {
    minX: e.minX,
    maxX: e.maxX,
    minY: e.minY,
    maxY: e.maxY,
    segments: F
  };
}
function G(e, m) {
  let t = Z(e);
  m.noContour || (e.contourLines = H(t, m), delete t.z), e.minMax = t;
}
function z(e, m) {
  for (let t of e) {
    let o = 0,
      l = 0;
    for (let i of t.spectra) {
      if (t.ntuples?.symbol ? (!o && i.observeFrequency && (o = i.observeFrequency), !l && i.shiftOffsetVal && (l = i.shiftOffsetVal)) : (o = i.observeFrequency, l = i.shiftOffsetVal), o && i.xUnits?.toUpperCase().includes("HZ") && (i.xUnits = "PPM", i.xFactor = i.xFactor !== void 0 ? i.xFactor / o : void 0, i.firstX = i.firstX !== void 0 ? i.firstX / o : void 0, i.lastX = i.lastX !== void 0 ? i.lastX / o : void 0, i.deltaX = i.deltaX !== void 0 ? i.deltaX / o : void 0, i.data)) for (let n = 0; n < i.data.x.length; n++) i.data.x[n] /= o;
      if (l && i.xUnits.toLowerCase().includes("ppm") && i.firstX !== void 0 && i.lastX !== void 0) {
        let n = i.firstX - l;
        if (i.firstX = i.firstX - n, i.lastX = i.lastX - n, i.data) for (let f = 0; f < i.data.x.length; f++) i.data.x[f] -= n;
      }
      if (t.ntuples?.nucleus && t.ntuples.symbol) for (let n = 0; n < t.ntuples.nucleus.length; n++) {
        let f = t.ntuples.symbol[n],
          s = t.ntuples.nucleus[n];
        if (f.match(/^[F|T]/) && !s) {
          if (f.match(/[F|T]1/)) if (t.tmp.$NUC2 && typeof t.tmp.$NUC2 == "string") t.ntuples.nucleus[n] = t.tmp.$NUC2;else {
            let u = t.ntuples.symbol.indexOf(f.replace(/^([F|T]).*/, "$12"));
            u && t.ntuples.nucleus[u] && (t.ntuples.nucleus[n] = t.ntuples.nucleus[u]);
          }
          f.match(/[F|T]2/) && typeof t.tmp.$NUC1 == "string" && (t.ntuples.nucleus[n] = t.tmp.$NUC1);
        }
        f.match(/[F|T]2/) && (t.yType = t.ntuples.nucleus[0], t.xType && !getGyromagneticRatio(t.xType) && (t.xType = t.ntuples.nucleus[1]));
      }
      if (o && t.ntuples?.symbol && t.ntuples.nucleus) {
        let n = "",
          f = t.ntuples.symbol.indexOf(i.pageSymbol);
        t.ntuples.units?.[f] && (n = t.ntuples.units[f]);
        let {
          nucleus: s
        } = t.ntuples;
        if (!n.toLowerCase().match(/(ppm|seconds)/) && s.length > 1) {
          if (f !== 0) {
            let p = "Not sure about this ntuples format";
            if (m) {
              m.warn(p);
              continue;
            } else throw new Error(p);
          }
          let {
              nucleus: u
            } = t.ntuples,
            c = getGyromagneticRatio(u[0]),
            b = getGyromagneticRatio(u[1]);
          if (!c || !b) {
            let p = `Problem with determination of gyromagnetic ratio for ${u.join("-")}`;
            if (m) m.error(p);else throw new Error(p);
          } else {
            let p = c / b * o;
            i.pageValue /= p;
          }
        }
      }
    }
  }
}
function J(e) {
  let m = e.spectra[0].data;
  e.chromatogram = {
    times: m.x.slice(),
    series: {
      intensity: {
        dimension: 1,
        data: m.y.slice()
      }
    }
  };
}
function q(e, m, t) {
  z(e, t.logger), de(e, t);
  for (let o of e) {
    if (Object.keys(o.ntuples).length > 0) {
      let l = [],
        i = Object.keys(o.ntuples);
      for (let n of i) {
        let f = o.ntuples[n];
        if (f) for (let s = 0; s < f.length; s++) l[s] || (l[s] = {}), l[s][n] = f[s];
      }
      o.ntuples = l;
    }
    o.twoD && t.wantXY && (G(o, t), t.logger?.trace({
      profiling: true
    }, "Finished countour plot calculation"), t.keepSpectra || delete o.spectra), t.chromatogram && (o.spectra.length > 1 ? ne(o) : J(o), t.logger?.trace({
      profiling: true
    }, "Finished chromatogram calculation")), delete o.tmp;
  }
}
function de(e, m) {
  for (let t of e) for (let o in t.meta) {
    let l = t.meta[o];
    if (typeof l == "string") {
      if (l.startsWith("{")) {
        if (!l.includes(":") && l.endsWith("}")) {
          let i = l.slice(1, -1).split(/[,; ]+/).filter(Boolean);
          for (let n = 0; n < i.length; n++) t.meta[o + String(n)] = m.dynamicTyping ? parseString(i[n]) : i[n];
        }
      } else if (l.startsWith("(")) {
        let i = l.split(/\r?\n/),
          n = /^\((?<from>\d+)\.\.(?<to>\d+)\).*$/;
        if (n.test(i[0])) {
          let [f, s] = i[0].match(n)?.slice(1).map(Number) ?? [],
            u = i.slice(1).join(" ").split(/[,; ]+/).filter(Boolean);
          for (let c = f; c <= s; c++) m.dynamicTyping && typeof u[c - f] == "string" ? t.meta[o + String(c)] = parseString(u[c - f]) : t.meta[o + String(c)] = u[c - f];
        }
      }
    }
  }
}
function K(e, m, t) {
  let o = -1,
    l = -1,
    i = "",
    n = "";
  if (t.indexOf("++") > 0) i = t.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, "$1"), n = t.replace(/.*\.\.([a-zA-Z0-9]+).*/, "$1");else {
    t = t.replaceAll(/[^a-zA-Z%]/g, ""), i = t.charAt(0), n = t.charAt(1), m.variables = {};
    for (let f of t) {
      let s = f.toLowerCase(),
        u = e.ntuples.symbol?.indexOf(f) || 0;
      if (u === -1) throw new Error(`Symbol undefined: ${f}`);
      m.variables[s] = {};
      for (let c in e.ntuples) e.ntuples[c]?.[u] && (m.variables[s][c.replace(/^var/, "")] = e.ntuples[c]?.[u]);
    }
  }
  o = e.ntuples.symbol?.indexOf(i) ?? -1, l = e.ntuples.symbol?.indexOf(n) ?? -1, o === -1 && (o = 0), l === -1 && (l = 0), e.ntuples.first && (e.ntuples.first.length > o && (m.firstX = e.ntuples.first[o]), e.ntuples.first.length > l && (m.firstY = e.ntuples.first[l])), e.ntuples.last && (e.ntuples.last.length > o && (m.lastX = e.ntuples.last[o]), e.ntuples.last.length > l && (m.lastY = e.ntuples.last[l])), e.ntuples.vardim && e.ntuples.vardim.length > o && (m.nbPoints = e.ntuples.vardim[o]), e.ntuples.factor && (e.ntuples.factor.length > o && (m.xFactor = e.ntuples.factor[o]), e.ntuples.factor.length > l && (m.yFactor = e.ntuples.factor[l])), e.ntuples.units && (e.ntuples.units.length > o && (e.ntuples.varname?.[o] ? m.xUnits = `${e.ntuples.varname[o]} [${e.ntuples.units[o]}]` : m.xUnits = e.ntuples.units[o]), e.ntuples.units.length > l && (e.ntuples.varname?.[l] ? m.yUnits = `${e.ntuples.varname[l]} [${e.ntuples.units[l]}]` : m.yUnits = e.ntuples.units[l]));
}
function j(e) {
  e.xFactor || (e.xFactor = 1), e.yFactor || (e.yFactor = 1);
}
var R = /[ \t]*,[ \t]*/,
  he = {
    removeComments: false,
    keepRecordsRegExp: /^$/,
    canonicDataLabels: true,
    canonicMetadataLabels: false,
    dynamicTyping: true,
    withoutXY: false,
    noTrimRegExp: /^$/,
    chromatogram: false,
    keepSpectra: false,
    noContour: false,
    nbContourLevels: 7,
    noiseMultiplier: 5
  };
function pt(e, m = {}) {
  e = ensureString(e);
  let t = {
    ...he,
    ...m
  };
  t.logger?.debug("Starting jcamp conversion"), t.wantXY = !t.withoutXY, t.start = Date.now();
  let o = [],
    l = {
      entries: [],
      flatten: []
    },
    i = {
      children: [],
      spectra: [],
      ntuples: {},
      info: {},
      meta: {},
      tmp: {}
    },
    n = i,
    f = [],
    s = {
      data: {}
    };
  t.logger?.trace({
    profiling: true
  }, "Before split to LDRS");
  let u = e.replaceAll(/[\r\n]+##/g, `
##`).split(`
##`);
  t.logger?.trace({
    profiling: true
  }, "Split to LDRS"), u[0] && (u[0] = u[0].replace(/^[\r\n ]*##/, ""));
  for (let c of u) {
    let b = c.indexOf("="),
      p = b > 0 ? c.slice(0, Math.max(0, b)) : c,
      r = b > 0 ? p.match(t.noTrimRegExp) ? c.slice(Math.max(0, b + 1)) : c.slice(Math.max(0, b + 1)).trim() : "",
      a = p.replaceAll(/[_ -]/g, "").toUpperCase();
    if (a === "DATATABLE") {
      let d = r.indexOf(`
`);
      if (d === -1 && (d = r.indexOf("\r")), d > 0) {
        let g = r.slice(0, Math.max(0, d)).split(/[ ,;\t]+/);
        K(n, s, g[0]), s.datatable = g[0], g[1]?.includes("PEAKS") ? a = "PEAKTABLE" : g[1] && (g[1].indexOf("XYDATA") || g[0].indexOf("++") > 0) && (a = "XYDATA", s.nbPoints && s.lastX !== void 0 && s.firstX !== void 0 && (s.deltaX = (s.lastX - s.firstX) / (s.nbPoints - 1)));
      }
    }
    if (a === "XYDATA") {
      t.wantXY && (j(s), r.match(/.*\+\+.*/) ? (s.nbPoints && s.lastX !== void 0 && s.firstX !== void 0 && (s.deltaX = (s.lastX - s.firstX) / (s.nbPoints - 1)), V(s, r, t)) : w(s, r, t), n?.spectra.push(s), s = {
        data: {}
      });
      continue;
    } else if (a === "PEAKTABLE") {
      t.wantXY && (j(s), w(s, r, t), n?.spectra.push(s), s = {
        data: {}
      });
      continue;
    }
    if (a === "PEAKASSIGNMENTS") {
      t.wantXY && (r.match(/.*[^A-Z]*.*/) && B(s, r), n?.spectra.push(s), s = {
        data: {}
      });
      continue;
    }
    if (t.removeComments && (r = r.split(/\r?\n/).map(d => d.replace(/ *\$\$.*$/, "")).join(`
`)), a === "TITLE") {
      let d = n;
      d.children || (d.children = []), n = {
        spectra: [],
        ntuples: {},
        info: {},
        meta: {},
        tmp: {}
      }, d.children.push(n), f.push(d), o.push(n), n.title = r;
    } else a === "DATATYPE" ? (n.dataType = r, r.match(/^nd|\snd\s/i) && (n.twoD = true)) : a === "NTUPLES" ? r.match(/^nd|\snd\s/i) && (n.twoD = true) : a === "DATACLASS" ? n.dataClass = r : a === "JCAMPDX" ? n.jcampDX = E(r) : a === "JCAMPCS" ? n.jcampCS = E(r) : a === "XUNITS" ? s.xUnits = r : a === "YUNITS" ? s.yUnits = r : a === "FIRSTX" ? s.firstX = Number(r) : a === "LASTX" ? s.lastX = Number(r) : a === "FIRSTY" ? s.firstY = Number(r) : a === "LASTY" ? s.lastY = Number(r) : a === "NPOINTS" ? s.nbPoints = Number(r) : a === "XFACTOR" ? s.xFactor = Number(r) : a === "YFACTOR" ? s.yFactor = Number(r) : a === "MAXX" ? s.maxX = Number(r) : a === "MINX" ? s.minX = Number(r) : a === "MAXY" ? s.maxY = Number(r) : a === "MINY" ? s.minY = Number(r) : a === "DELTAX" ? s.deltaX = Number(r) : a === ".OBSERVEFREQUENCY" || a === "$SFO1" ? s.observeFrequency || (s.observeFrequency = Number(r)) : a === ".OBSERVENUCLEUS" ? s.xType || (n.xType = r.replaceAll(/[^a-zA-Z0-9]/g, "")) : a === "$OFFSET" ? (n.shiftOffsetNum = 0, s.shiftOffsetVal || (s.shiftOffsetVal = Number(r))) : a === "$REFERENCEPOINT" || (a === "VARNAME" ? n.ntuples.varname = r.split(R) : a === "SYMBOL" ? n.ntuples.symbol = r.split(R) : a === "VARTYPE" ? n.ntuples.vartype = r.split(R) : a === "VARFORM" ? n.ntuples.varform = r.split(R) : a === "VARDIM" ? n.ntuples.vardim = O(r.split(R)) : a === "UNITS" ? n.ntuples.units = r.split(R) : a === "FACTOR" ? n.ntuples.factor = O(r.split(R)) : a === "FIRST" ? n.ntuples.first = O(r.split(R)) : a === "LAST" ? n.ntuples.last = O(r.split(R)) : a === "MIN" ? n.ntuples.min = O(r.split(R)) : a === "MAX" ? n.ntuples.max = O(r.split(R)) : a === ".NUCLEUS" ? n.ntuples && (n.ntuples.nucleus = r.split(R).map(d => d.replaceAll(/[^a-zA-Z0-9]/g, ""))) : a === "PAGE" ? (s.page = r.trim(), s.pageValue = Number(r.replace(/^.*=/, "")), s.pageSymbol = s.page.replace(/[=].*/, "")) : a === "RETENTIONTIME" ? s.pageValue = Number(r) : re(a) ? s[k(a)] = r : a === "SAMPLEDESCRIPTION" ? s.sampleDescription = r : a.startsWith("$NUC") ? !n.tmp[a] && !r.includes("off") && (n.tmp[a] = r.replaceAll(/[<>]/g, "")) : a === "END" && (n = f.pop()));
    if (n?.info && n.meta && a.match(t.keepRecordsRegExp)) {
      let d, g;
      p.startsWith("$") ? (g = t.canonicMetadataLabels ? a.slice(1) : p.slice(1), d = n.meta) : (g = t.canonicDataLabels ? a : p, d = n.info), t.dynamicTyping && (r = parseString(r)), d[g] ? (Array.isArray(d[g]) || (d[g] = [d[g]]), d[g].push(r)) : d[g] = r;
    }
  }
  if (t.logger?.trace({
    profiling: true
  }, "Finished parsing"), q(o, l, t), t.logger?.trace({
    profiling: true
  }, "Total time"), l.entries = i.children || [], l.flatten = o, t.logger) {
    t.logger.debug("Finished jcamp conversion");
    for (let c of l.flatten) t.logger.debug(`${c.dataType} - ${c.title}`);
  }
  return l;
}

/**
 * Determines the appropriate Kind based on JCAMP data
 * @param data - JCAMP entry data
 * @returns The appropriate Kind or undefined
 */
function getJcampKind(data) {
  if (!data.dataType || !data.spectra?.[0]?.yUnits) {
    return undefined;
  }
  const dataType = data.dataType.toLowerCase();
  const yUnits = data.spectra[0].yUnits.toLowerCase();
  if (dataType.match(/infrared/)) {
    if (yUnits.match(/absorbance/)) {
      return IR_ABSORBANCE;
    } else {
      return IR_TRANSMITTANCE;
    }
  }
  return undefined;
}
const IR_TRANSMITTANCE = {
  normalization: {},
  importation: {
    converter: transmittance => -Math.log10(transmittance)
  },
  kind: 'Infrared',
  display: {
    xLabel: 'wavelength [cm-1]',
    xInverted: true,
    yLabel: 'Absorbance'
  }
};
const IR_ABSORBANCE = {
  normalization: {},
  kind: 'Infrared',
  display: {
    xLabel: 'wavelength [cm-1]',
    xInverted: true,
    yLabel: 'Absorbance'
  }
};

/**
 * Create a spectrum from a JCAMP file
 * @param jcampText - String containing the JCAMP data
 * @returns Parsed spectrum data with kind, meta, and info
 */
function parseJcamp(jcampText) {
  const parsed = pt(jcampText, {
    keepRecordsRegExp: /.*/
  }).flatten[0];
  const kind = getJcampKind(parsed);
  const data = parsed.spectra[0].data;
  const {
    meta,
    info
  } = parsed;
  // Convert the data if needed
  if (kind?.importation?.converter) {
    data.y = data.y.map(kind.importation.converter);
  }
  return {
    data,
    kind,
    meta,
    info
  };
}

/**
 * Parse a matrix from text format
 * @param text - String containing the text data
 * @param options - Parsing options
 * @returns Parsed matrix data with x, meta, matrix, and ids
 */
function parseMatrix(text, options = {}) {
  const lines = text.split(/[\n\r]+/).filter(Boolean);
  const {
    fs = '\t'
  } = options;
  const currentMatrix = [];
  const ids = [];
  const meta = [];
  let x = [];
  const headers = lines[0].split(fs);
  const labels = [];
  for (let i = 0; i < headers.length; i++) {
    const header = headers[i];
    if (Number.isNaN(Number(header))) {
      labels[i] = header;
    } else {
      x = headers.slice(i).map(Number);
      break;
    }
  }
  for (let i = 1; i < lines.length; i++) {
    const line = lines[i];
    const parts = line.split('\t');
    ids.push(parts[0]);
    const oneMeta = {};
    meta.push(oneMeta);
    for (let j = 1; j < parts.length; j++) {
      if (j < labels.length) {
        oneMeta[labels[j]] = parts[j];
      } else {
        currentMatrix.push(parts.slice(labels.length).map(Number));
        break;
      }
    }
  }
  return {
    x,
    meta,
    matrix: currentMatrix,
    ids
  };
}

/**
 * General internal parsing function
 * @param text - Csv or tsv strings.
 * @param options - Parsing options
 * @returns parsed text file with column information
 */
function parse(text, options = {}) {
  const {
    rescale = false,
    uniqueX = false,
    bestGuess = false,
    //@ts-expect-error old library used this property and we want to throw an error so that people are forced to migrate
    keepInfo
  } = options;
  let {
    xColumn = 0,
    yColumn = 1,
    numberColumns = Number.MAX_SAFE_INTEGER,
    maxNumberColumns = Number.MAX_SAFE_INTEGER,
    minNumberColumns = 2
  } = options;
  if (keepInfo !== undefined) {
    throw new Error('keepInfo has been deprecated, pelase use the new method parseXYAndKeepInfo');
  }
  text = ensureString(text);
  maxNumberColumns = Math.max(maxNumberColumns, xColumn + 1, yColumn + 1);
  minNumberColumns = Math.max(xColumn + 1, yColumn + 1, minNumberColumns);
  const lines = text.split(/[\r\n]+/);
  let matrix = [];
  const info = [];
  let position = 0;
  lines.forEach(line => {
    line = line.trim();
    // we will consider only lines that contains only numbers
    if (/[0-9]+/.test(line) && /^[0-9eE,;. \t+-]+$/.test(line)) {
      let fields = line.split(/,[; \t]+|[; \t]+/);
      if (fields.length === 1) {
        fields = line.split(/[,; \t]+/);
      }
      if (fields && fields.length >= minNumberColumns &&
      // we filter lines that have not enough or too many columns
      fields.length <= maxNumberColumns) {
        matrix.push(fields.map(value => parseFloat(value.replace(',', '.'))));
        position++;
      }
    } else if (line) {
      info.push({
        position,
        value: line
      });
    }
  });
  if (bestGuess) {
    if (matrix[0] && matrix[0].length === 3 && options.xColumn === undefined && options.yColumn === undefined) {
      // is the first column a seuqnetial number ?
      let skipFirstColumn = true;
      for (let i = 0; i < matrix.length - 1; i++) {
        if (Math.abs(matrix[i][0] - matrix[i + 1][0]) !== 1) {
          skipFirstColumn = false;
        }
      }
      if (skipFirstColumn) {
        xColumn = 1;
        yColumn = 2;
      }
    }
    if (matrix[0] && matrix[0].length > 3) {
      const xs = [];
      for (const row of matrix) {
        for (let i = xColumn; i < row.length; i += 2) {
          xs.push(row[i]);
        }
      }
      if (xIsMonotonic(xs)) {
        numberColumns = 2;
      }
    }
  }
  if (numberColumns) {
    const newMatrix = [];
    for (const row of matrix) {
      for (let i = 0; i < row.length; i += numberColumns) {
        newMatrix.push(row.slice(i, i + numberColumns));
      }
    }
    matrix = newMatrix;
  }
  let result = {
    x: matrix.map(row => row[xColumn]),
    y: matrix.map(row => row[yColumn])
  };
  if (uniqueX) {
    result = xyUniqueX(result, {
      algorithm: 'sum'
    });
  }
  if (rescale) {
    const maxY = xMaxValue(result.y);
    for (let i = 0; i < result.y.length; i++) {
      result.y[i] /= maxY;
    }
  }
  return {
    info,
    data: result
  };
}

/**
 * Parse a text-file and convert it to an object {x:[], y:[]}
 * @param text - Csv or tsv strings.
 * @param options - Parsing options
 * @returns - The parsed data
 */
function parseXY(text, options = {}) {
  return parse(text, options).data;
}

/**
 * Create a spectrum from a text file
 * @param value - String containing the text data
 * @param options - Parsing options
 * @returns Parsed data object
 */
function parseText(value, options = {}) {
  const {
    kind,
    parserOptions = {}
  } = options;
  const data = parseXY(value, parserOptions);
  // Convert the data if needed
  if (kind?.importation?.converter) {
    data.y = data.y.map(kind.importation.converter);
  }
  return {
    data
  };
}

function getAutocorrelation(normalized, index) {
  return {
    x: normalized.x,
    y: matrixAutoCorrelation(normalized.matrix, index)
  };
}

/**
 * Calculate mean of normalized data
 * @param normalized - Normalized data with x and matrix
 * @returns Mean data with x and y values
 */
function getMeanData(normalized) {
  const matrix = new Matrix(normalized.matrix);
  return {
    x: normalized.x,
    y: matrix.mean('column')
  };
}

/**
 * Convert data to text format
 * @param data - Data including matrix, meta, ids, and x values
 * @param options - Conversion options
 * @returns Formatted text string
 */
function convertToText(data, options = {}) {
  const {
    fs = '\t',
    rs = '\n'
  } = options;
  const {
    matrix,
    meta,
    ids,
    x
  } = data;
  const allKeysObject = {};
  for (const metum of meta) {
    if (metum) {
      for (const key of Object.keys(metum)) {
        const type = typeof metum[key];
        if (type === 'number' || type === 'string' || type === 'boolean') {
          allKeysObject[key] = true;
        }
      }
    }
  }
  const allKeys = Object.keys(allKeysObject);
  const lines = [];
  let line = ['id', ...allKeys, ...x];
  lines.push(line.join(fs));
  for (let i = 0; i < ids.length; i++) {
    line = [ids[i]];
    for (const key of allKeys) {
      line.push(removeSpecialCharacters(meta[i][key]));
    }
    line.push(...matrix[i]);
    lines.push(line.join(fs));
  }
  return lines.join(rs);
}
function removeSpecialCharacters(value) {
  if (typeof value !== 'string') {
    return value;
  }
  return value.replaceAll(/[\t\n\r]+/g, ' ');
}

/**
 * Get normalized data as text
 * @param spectra - Array of spectrum objects
 * @param options - Options for filtering and formatting
 * @returns Text representation of normalized data
 */
function getNormalizedText(spectra, options = {}) {
  const {
    fs = '\t',
    rs = '\n'
  } = options;
  return convertToText(getNormalizedData(spectra, options), {
    rs,
    fs
  });
}

function commonjsRequire(path) {
	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
}

var object_hash = {exports: {}};

(function (module, exports$1) {
  !function (e) {
    module.exports = e() ;
  }(function () {
    return function r(o, i, u) {
      function s(n, e) {
        if (!i[n]) {
          if (!o[n]) {
            var t = "function" == typeof commonjsRequire && commonjsRequire;
            if (!e && t) return t(n, true);
            if (a) return a(n, true);
            throw new Error("Cannot find module '" + n + "'");
          }
          e = i[n] = {
            exports: {}
          };
          o[n][0].call(e.exports, function (e) {
            var t = o[n][1][e];
            return s(t || e);
          }, e, e.exports, r, o, i, u);
        }
        return i[n].exports;
      }
      for (var a = "function" == typeof commonjsRequire && commonjsRequire, e = 0; e < u.length; e++) s(u[e]);
      return s;
    }({
      1: [function (w, b, m) {
        !function (e, n, s, c, d, h, p, g, y) {

          var r = w("crypto");
          function t(e, t) {
            t = u(e, t);
            var n;
            return void 0 === (n = "passthrough" !== t.algorithm ? r.createHash(t.algorithm) : new l()).write && (n.write = n.update, n.end = n.update), f(t, n).dispatch(e), n.update || n.end(""), n.digest ? n.digest("buffer" === t.encoding ? void 0 : t.encoding) : (e = n.read(), "buffer" !== t.encoding ? e.toString(t.encoding) : e);
          }
          (m = b.exports = t).sha1 = function (e) {
            return t(e);
          }, m.keys = function (e) {
            return t(e, {
              excludeValues: true,
              algorithm: "sha1",
              encoding: "hex"
            });
          }, m.MD5 = function (e) {
            return t(e, {
              algorithm: "md5",
              encoding: "hex"
            });
          }, m.keysMD5 = function (e) {
            return t(e, {
              algorithm: "md5",
              encoding: "hex",
              excludeValues: true
            });
          };
          var o = r.getHashes ? r.getHashes().slice() : ["sha1", "md5"],
            i = (o.push("passthrough"), ["buffer", "hex", "binary", "base64"]);
          function u(e, t) {
            var n = {};
            if (n.algorithm = (t = t || {}).algorithm || "sha1", n.encoding = t.encoding || "hex", n.excludeValues = !!t.excludeValues, n.algorithm = n.algorithm.toLowerCase(), n.encoding = n.encoding.toLowerCase(), n.ignoreUnknown = true === t.ignoreUnknown, n.respectType = false !== t.respectType, n.respectFunctionNames = false !== t.respectFunctionNames, n.respectFunctionProperties = false !== t.respectFunctionProperties, n.unorderedArrays = true === t.unorderedArrays, n.unorderedSets = false !== t.unorderedSets, n.unorderedObjects = false !== t.unorderedObjects, n.replacer = t.replacer || void 0, n.excludeKeys = t.excludeKeys || void 0, void 0 === e) throw new Error("Object argument required.");
            for (var r = 0; r < o.length; ++r) o[r].toLowerCase() === n.algorithm.toLowerCase() && (n.algorithm = o[r]);
            if (-1 === o.indexOf(n.algorithm)) throw new Error('Algorithm "' + n.algorithm + '"  not supported. supported values: ' + o.join(", "));
            if (-1 === i.indexOf(n.encoding) && "passthrough" !== n.algorithm) throw new Error('Encoding "' + n.encoding + '"  not supported. supported values: ' + i.join(", "));
            return n;
          }
          function a(e) {
            if ("function" == typeof e) return null != /^function\s+\w*\s*\(\s*\)\s*{\s+\[native code\]\s+}$/i.exec(Function.prototype.toString.call(e));
          }
          function f(o, t, i) {
            i = i || [];
            function u(e) {
              return t.update ? t.update(e, "utf8") : t.write(e, "utf8");
            }
            return {
              dispatch: function (e) {
                return this["_" + (null === (e = o.replacer ? o.replacer(e) : e) ? "null" : typeof e)](e);
              },
              _object: function (t) {
                var n,
                  e = Object.prototype.toString.call(t),
                  r = /\[object (.*)\]/i.exec(e);
                r = (r = r ? r[1] : "unknown:[" + e + "]").toLowerCase();
                if (0 <= (e = i.indexOf(t))) return this.dispatch("[CIRCULAR:" + e + "]");
                if (i.push(t), void 0 !== s && s.isBuffer && s.isBuffer(t)) return u("buffer:"), u(t);
                if ("object" === r || "function" === r || "asyncfunction" === r) return e = Object.keys(t), o.unorderedObjects && (e = e.sort()), false === o.respectType || a(t) || e.splice(0, 0, "prototype", "__proto__", "constructor"), o.excludeKeys && (e = e.filter(function (e) {
                  return !o.excludeKeys(e);
                })), u("object:" + e.length + ":"), n = this, e.forEach(function (e) {
                  n.dispatch(e), u(":"), o.excludeValues || n.dispatch(t[e]), u(",");
                });
                if (!this["_" + r]) {
                  if (o.ignoreUnknown) return u("[" + r + "]");
                  throw new Error('Unknown object type "' + r + '"');
                }
                this["_" + r](t);
              },
              _array: function (e, t) {
                t = void 0 !== t ? t : false !== o.unorderedArrays;
                var n = this;
                if (u("array:" + e.length + ":"), !t || e.length <= 1) return e.forEach(function (e) {
                  return n.dispatch(e);
                });
                var r = [],
                  t = e.map(function (e) {
                    var t = new l(),
                      n = i.slice();
                    return f(o, t, n).dispatch(e), r = r.concat(n.slice(i.length)), t.read().toString();
                  });
                return i = i.concat(r), t.sort(), this._array(t, false);
              },
              _date: function (e) {
                return u("date:" + e.toJSON());
              },
              _symbol: function (e) {
                return u("symbol:" + e.toString());
              },
              _error: function (e) {
                return u("error:" + e.toString());
              },
              _boolean: function (e) {
                return u("bool:" + e.toString());
              },
              _string: function (e) {
                u("string:" + e.length + ":"), u(e.toString());
              },
              _function: function (e) {
                u("fn:"), a(e) ? this.dispatch("[native]") : this.dispatch(e.toString()), false !== o.respectFunctionNames && this.dispatch("function-name:" + String(e.name)), o.respectFunctionProperties && this._object(e);
              },
              _number: function (e) {
                return u("number:" + e.toString());
              },
              _xml: function (e) {
                return u("xml:" + e.toString());
              },
              _null: function () {
                return u("Null");
              },
              _undefined: function () {
                return u("Undefined");
              },
              _regexp: function (e) {
                return u("regex:" + e.toString());
              },
              _uint8array: function (e) {
                return u("uint8array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _uint8clampedarray: function (e) {
                return u("uint8clampedarray:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _int8array: function (e) {
                return u("int8array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _uint16array: function (e) {
                return u("uint16array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _int16array: function (e) {
                return u("int16array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _uint32array: function (e) {
                return u("uint32array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _int32array: function (e) {
                return u("int32array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _float32array: function (e) {
                return u("float32array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _float64array: function (e) {
                return u("float64array:"), this.dispatch(Array.prototype.slice.call(e));
              },
              _arraybuffer: function (e) {
                return u("arraybuffer:"), this.dispatch(new Uint8Array(e));
              },
              _url: function (e) {
                return u("url:" + e.toString());
              },
              _map: function (e) {
                u("map:");
                e = Array.from(e);
                return this._array(e, false !== o.unorderedSets);
              },
              _set: function (e) {
                u("set:");
                e = Array.from(e);
                return this._array(e, false !== o.unorderedSets);
              },
              _file: function (e) {
                return u("file:"), this.dispatch([e.name, e.size, e.type, e.lastModfied]);
              },
              _blob: function () {
                if (o.ignoreUnknown) return u("[blob]");
                throw Error('Hashing Blob objects is currently not supported\n(see https://github.com/puleos/object-hash/issues/26)\nUse "options.replacer" or "options.ignoreUnknown"\n');
              },
              _domwindow: function () {
                return u("domwindow");
              },
              _bigint: function (e) {
                return u("bigint:" + e.toString());
              },
              _process: function () {
                return u("process");
              },
              _timer: function () {
                return u("timer");
              },
              _pipe: function () {
                return u("pipe");
              },
              _tcp: function () {
                return u("tcp");
              },
              _udp: function () {
                return u("udp");
              },
              _tty: function () {
                return u("tty");
              },
              _statwatcher: function () {
                return u("statwatcher");
              },
              _securecontext: function () {
                return u("securecontext");
              },
              _connection: function () {
                return u("connection");
              },
              _zlib: function () {
                return u("zlib");
              },
              _context: function () {
                return u("context");
              },
              _nodescript: function () {
                return u("nodescript");
              },
              _httpparser: function () {
                return u("httpparser");
              },
              _dataview: function () {
                return u("dataview");
              },
              _signal: function () {
                return u("signal");
              },
              _fsevent: function () {
                return u("fsevent");
              },
              _tlswrap: function () {
                return u("tlswrap");
              }
            };
          }
          function l() {
            return {
              buf: "",
              write: function (e) {
                this.buf += e;
              },
              end: function (e) {
                this.buf += e;
              },
              read: function () {
                return this.buf;
              }
            };
          }
          m.writeToStream = function (e, t, n) {
            return void 0 === n && (n = t, t = {}), f(t = u(e, t), n).dispatch(e);
          };
        }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/fake_9a5aa49d.js", "/");
      }, {
        buffer: 3,
        crypto: 5,
        lYpoI2: 11
      }],
      2: [function (e, t, f) {
        !function (e, t, n, r, o, i, u, s, a) {
          !function (e) {

            var a = "undefined" != typeof Uint8Array ? Uint8Array : Array,
              t = "+".charCodeAt(0),
              n = "/".charCodeAt(0),
              r = "0".charCodeAt(0),
              o = "a".charCodeAt(0),
              i = "A".charCodeAt(0),
              u = "-".charCodeAt(0),
              s = "_".charCodeAt(0);
            function f(e) {
              e = e.charCodeAt(0);
              return e === t || e === u ? 62 : e === n || e === s ? 63 : e < r ? -1 : e < r + 10 ? e - r + 26 + 26 : e < i + 26 ? e - i : e < o + 26 ? e - o + 26 : void 0;
            }
            e.toByteArray = function (e) {
              var t, n;
              if (0 < e.length % 4) throw new Error("Invalid string. Length must be a multiple of 4");
              var r = e.length,
                r = "=" === e.charAt(r - 2) ? 2 : "=" === e.charAt(r - 1) ? 1 : 0,
                o = new a(3 * e.length / 4 - r),
                i = 0 < r ? e.length - 4 : e.length,
                u = 0;
              function s(e) {
                o[u++] = e;
              }
              for (t = 0; t < i; t += 4, 0) s((16711680 & (n = f(e.charAt(t)) << 18 | f(e.charAt(t + 1)) << 12 | f(e.charAt(t + 2)) << 6 | f(e.charAt(t + 3)))) >> 16), s((65280 & n) >> 8), s(255 & n);
              return 2 == r ? s(255 & (n = f(e.charAt(t)) << 2 | f(e.charAt(t + 1)) >> 4)) : 1 == r && (s((n = f(e.charAt(t)) << 10 | f(e.charAt(t + 1)) << 4 | f(e.charAt(t + 2)) >> 2) >> 8 & 255), s(255 & n)), o;
            }, e.fromByteArray = function (e) {
              var t,
                n,
                r,
                o,
                i = e.length % 3,
                u = "";
              function s(e) {
                return "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".charAt(e);
              }
              for (t = 0, r = e.length - i; t < r; t += 3) n = (e[t] << 16) + (e[t + 1] << 8) + e[t + 2], u += s((o = n) >> 18 & 63) + s(o >> 12 & 63) + s(o >> 6 & 63) + s(63 & o);
              switch (i) {
                case 1:
                  u = (u += s((n = e[e.length - 1]) >> 2)) + s(n << 4 & 63) + "==";
                  break;
                case 2:
                  u = (u = (u += s((n = (e[e.length - 2] << 8) + e[e.length - 1]) >> 10)) + s(n >> 4 & 63)) + s(n << 2 & 63) + "=";
              }
              return u;
            };
          }(void 0 === f ? this.base64js = {} : f);
        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/base64-js/lib/b64.js", "/node_modules/gulp-browserify/node_modules/base64-js/lib");
      }, {
        buffer: 3,
        lYpoI2: 11
      }],
      3: [function (O, e, H) {
        !function (e, n, f, r, h, p, g, y, w) {
          var a = O("base64-js"),
            i = O("ieee754");
          function f(e, t, n) {
            if (!(this instanceof f)) return new f(e, t, n);
            var r,
              o,
              i,
              u,
              s = typeof e;
            if ("base64" === t && "string" == s) for (e = (u = e).trim ? u.trim() : u.replace(/^\s+|\s+$/g, ""); e.length % 4 != 0;) e += "=";
            if ("number" == s) r = j(e);else if ("string" == s) r = f.byteLength(e, t);else {
              if ("object" != s) throw new Error("First argument needs to be a number, array or string.");
              r = j(e.length);
            }
            if (f._useTypedArrays ? o = f._augment(new Uint8Array(r)) : ((o = this).length = r, o._isBuffer = true), f._useTypedArrays && "number" == typeof e.byteLength) o._set(e);else if (C(u = e) || f.isBuffer(u) || u && "object" == typeof u && "number" == typeof u.length) for (i = 0; i < r; i++) f.isBuffer(e) ? o[i] = e.readUInt8(i) : o[i] = e[i];else if ("string" == s) o.write(e, 0, t);else if ("number" == s && !f._useTypedArrays && !n) for (i = 0; i < r; i++) o[i] = 0;
            return o;
          }
          function b(e, t, n, r) {
            return f._charsWritten = c(function (e) {
              for (var t = [], n = 0; n < e.length; n++) t.push(255 & e.charCodeAt(n));
              return t;
            }(t), e, n, r);
          }
          function m(e, t, n, r) {
            return f._charsWritten = c(function (e) {
              for (var t, n, r = [], o = 0; o < e.length; o++) n = e.charCodeAt(o), t = n >> 8, n = n % 256, r.push(n), r.push(t);
              return r;
            }(t), e, n, r);
          }
          function v(e, t, n) {
            var r = "";
            n = Math.min(e.length, n);
            for (var o = t; o < n; o++) r += String.fromCharCode(e[o]);
            return r;
          }
          function o(e, t, n, r) {
            r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length"));
            var o,
              r = e.length;
            if (!(r <= t)) return n ? (o = e[t], t + 1 < r && (o |= e[t + 1] << 8)) : (o = e[t] << 8, t + 1 < r && (o |= e[t + 1])), o;
          }
          function u(e, t, n, r) {
            r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length"));
            var o,
              r = e.length;
            if (!(r <= t)) return n ? (t + 2 < r && (o = e[t + 2] << 16), t + 1 < r && (o |= e[t + 1] << 8), o |= e[t], t + 3 < r && (o += e[t + 3] << 24 >>> 0)) : (t + 1 < r && (o = e[t + 1] << 16), t + 2 < r && (o |= e[t + 2] << 8), t + 3 < r && (o |= e[t + 3]), o += e[t] << 24 >>> 0), o;
          }
          function _(e, t, n, r) {
            if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = o(e, t, n, true), 32768 & r ? -1 * (65535 - r + 1) : r;
          }
          function E(e, t, n, r) {
            if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = u(e, t, n, true), 2147483648 & r ? -1 * (4294967295 - r + 1) : r;
          }
          function I(e, t, n, r) {
            return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 3 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 23, 4);
          }
          function A(e, t, n, r) {
            return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 7 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 52, 8);
          }
          function s(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "trying to write beyond buffer length"), Y(t, 65535));
            o = e.length;
            if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 2); i < u; i++) e[n + i] = (t & 255 << 8 * (r ? i : 1 - i)) >>> 8 * (r ? i : 1 - i);
          }
          function l(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "trying to write beyond buffer length"), Y(t, 4294967295));
            o = e.length;
            if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 4); i < u; i++) e[n + i] = t >>> 8 * (r ? i : 3 - i) & 255;
          }
          function B(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "Trying to write beyond buffer length"), F(t, 32767, -32768)), e.length <= n || s(e, 0 <= t ? t : 65535 + t + 1, n, r, o);
          }
          function L(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), F(t, 2147483647, -2147483648)), e.length <= n || l(e, 0 <= t ? t : 4294967295 + t + 1, n, r, o);
          }
          function U(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), D(t, 34028234663852886e22, -34028234663852886e22)), e.length <= n || i.write(e, t, n, r, 23, 4);
          }
          function x(e, t, n, r, o) {
            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 7 < e.length, "Trying to write beyond buffer length"), D(t, 17976931348623157e292, -17976931348623157e292)), e.length <= n || i.write(e, t, n, r, 52, 8);
          }
          H.Buffer = f, H.SlowBuffer = f, H.INSPECT_MAX_BYTES = 50, f.poolSize = 8192, f._useTypedArrays = function () {
            try {
              var e = new ArrayBuffer(0),
                t = new Uint8Array(e);
              return t.foo = function () {
                return 42;
              }, 42 === t.foo() && "function" == typeof t.subarray;
            } catch (e) {
              return false;
            }
          }(), f.isEncoding = function (e) {
            switch (String(e).toLowerCase()) {
              case "hex":
              case "utf8":
              case "utf-8":
              case "ascii":
              case "binary":
              case "base64":
              case "raw":
              case "ucs2":
              case "ucs-2":
              case "utf16le":
              case "utf-16le":
                return true;
              default:
                return false;
            }
          }, f.isBuffer = function (e) {
            return !(null == e || !e._isBuffer);
          }, f.byteLength = function (e, t) {
            var n;
            switch (e += "", t || "utf8") {
              case "hex":
                n = e.length / 2;
                break;
              case "utf8":
              case "utf-8":
                n = T(e).length;
                break;
              case "ascii":
              case "binary":
              case "raw":
                n = e.length;
                break;
              case "base64":
                n = M(e).length;
                break;
              case "ucs2":
              case "ucs-2":
              case "utf16le":
              case "utf-16le":
                n = 2 * e.length;
                break;
              default:
                throw new Error("Unknown encoding");
            }
            return n;
          }, f.concat = function (e, t) {
            if (d(C(e), "Usage: Buffer.concat(list, [totalLength])\nlist should be an Array."), 0 === e.length) return new f(0);
            if (1 === e.length) return e[0];
            if ("number" != typeof t) for (o = t = 0; o < e.length; o++) t += e[o].length;
            for (var n = new f(t), r = 0, o = 0; o < e.length; o++) {
              var i = e[o];
              i.copy(n, r), r += i.length;
            }
            return n;
          }, f.prototype.write = function (e, t, n, r) {
            isFinite(t) ? isFinite(n) || (r = n, n = void 0) : (a = r, r = t, t = n, n = a), t = Number(t) || 0;
            var o,
              i,
              u,
              s,
              a = this.length - t;
            switch ((!n || a < (n = Number(n))) && (n = a), r = String(r || "utf8").toLowerCase()) {
              case "hex":
                o = function (e, t, n, r) {
                  n = Number(n) || 0;
                  var o = e.length - n;
                  (!r || o < (r = Number(r))) && (r = o), d((o = t.length) % 2 == 0, "Invalid hex string"), o / 2 < r && (r = o / 2);
                  for (var i = 0; i < r; i++) {
                    var u = parseInt(t.substr(2 * i, 2), 16);
                    d(!isNaN(u), "Invalid hex string"), e[n + i] = u;
                  }
                  return f._charsWritten = 2 * i, i;
                }(this, e, t, n);
                break;
              case "utf8":
              case "utf-8":
                i = this, u = t, s = n, o = f._charsWritten = c(T(e), i, u, s);
                break;
              case "ascii":
              case "binary":
                o = b(this, e, t, n);
                break;
              case "base64":
                i = this, u = t, s = n, o = f._charsWritten = c(M(e), i, u, s);
                break;
              case "ucs2":
              case "ucs-2":
              case "utf16le":
              case "utf-16le":
                o = m(this, e, t, n);
                break;
              default:
                throw new Error("Unknown encoding");
            }
            return o;
          }, f.prototype.toString = function (e, t, n) {
            var r,
              o,
              i,
              u,
              s = this;
            if (e = String(e || "utf8").toLowerCase(), t = Number(t) || 0, (n = void 0 !== n ? Number(n) : s.length) === t) return "";
            switch (e) {
              case "hex":
                r = function (e, t, n) {
                  var r = e.length;
                  (!t || t < 0) && (t = 0);
                  (!n || n < 0 || r < n) && (n = r);
                  for (var o = "", i = t; i < n; i++) o += k(e[i]);
                  return o;
                }(s, t, n);
                break;
              case "utf8":
              case "utf-8":
                r = function (e, t, n) {
                  var r = "",
                    o = "";
                  n = Math.min(e.length, n);
                  for (var i = t; i < n; i++) e[i] <= 127 ? (r += N(o) + String.fromCharCode(e[i]), o = "") : o += "%" + e[i].toString(16);
                  return r + N(o);
                }(s, t, n);
                break;
              case "ascii":
              case "binary":
                r = v(s, t, n);
                break;
              case "base64":
                o = s, u = n, r = 0 === (i = t) && u === o.length ? a.fromByteArray(o) : a.fromByteArray(o.slice(i, u));
                break;
              case "ucs2":
              case "ucs-2":
              case "utf16le":
              case "utf-16le":
                r = function (e, t, n) {
                  for (var r = e.slice(t, n), o = "", i = 0; i < r.length; i += 2) o += String.fromCharCode(r[i] + 256 * r[i + 1]);
                  return o;
                }(s, t, n);
                break;
              default:
                throw new Error("Unknown encoding");
            }
            return r;
          }, f.prototype.toJSON = function () {
            return {
              type: "Buffer",
              data: Array.prototype.slice.call(this._arr || this, 0)
            };
          }, f.prototype.copy = function (e, t, n, r) {
            if (t = t || 0, (r = r || 0 === r ? r : this.length) !== (n = n || 0) && 0 !== e.length && 0 !== this.length) {
              d(n <= r, "sourceEnd < sourceStart"), d(0 <= t && t < e.length, "targetStart out of bounds"), d(0 <= n && n < this.length, "sourceStart out of bounds"), d(0 <= r && r <= this.length, "sourceEnd out of bounds"), r > this.length && (r = this.length);
              var o = (r = e.length - t < r - n ? e.length - t + n : r) - n;
              if (o < 100 || !f._useTypedArrays) for (var i = 0; i < o; i++) e[i + t] = this[i + n];else e._set(this.subarray(n, n + o), t);
            }
          }, f.prototype.slice = function (e, t) {
            var n = this.length;
            if (e = S(e, n, 0), t = S(t, n, n), f._useTypedArrays) return f._augment(this.subarray(e, t));
            for (var r = t - e, o = new f(r, void 0, true), i = 0; i < r; i++) o[i] = this[i + e];
            return o;
          }, f.prototype.get = function (e) {
            return console.log(".get() is deprecated. Access using array indexes instead."), this.readUInt8(e);
          }, f.prototype.set = function (e, t) {
            return console.log(".set() is deprecated. Access using array indexes instead."), this.writeUInt8(e, t);
          }, f.prototype.readUInt8 = function (e, t) {
            if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return this[e];
          }, f.prototype.readUInt16LE = function (e, t) {
            return o(this, e, true, t);
          }, f.prototype.readUInt16BE = function (e, t) {
            return o(this, e, false, t);
          }, f.prototype.readUInt32LE = function (e, t) {
            return u(this, e, true, t);
          }, f.prototype.readUInt32BE = function (e, t) {
            return u(this, e, false, t);
          }, f.prototype.readInt8 = function (e, t) {
            if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return 128 & this[e] ? -1 * (255 - this[e] + 1) : this[e];
          }, f.prototype.readInt16LE = function (e, t) {
            return _(this, e, true, t);
          }, f.prototype.readInt16BE = function (e, t) {
            return _(this, e, false, t);
          }, f.prototype.readInt32LE = function (e, t) {
            return E(this, e, true, t);
          }, f.prototype.readInt32BE = function (e, t) {
            return E(this, e, false, t);
          }, f.prototype.readFloatLE = function (e, t) {
            return I(this, e, true, t);
          }, f.prototype.readFloatBE = function (e, t) {
            return I(this, e, false, t);
          }, f.prototype.readDoubleLE = function (e, t) {
            return A(this, e, true, t);
          }, f.prototype.readDoubleBE = function (e, t) {
            return A(this, e, false, t);
          }, f.prototype.writeUInt8 = function (e, t, n) {
            n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "trying to write beyond buffer length"), Y(e, 255)), t >= this.length || (this[t] = e);
          }, f.prototype.writeUInt16LE = function (e, t, n) {
            s(this, e, t, true, n);
          }, f.prototype.writeUInt16BE = function (e, t, n) {
            s(this, e, t, false, n);
          }, f.prototype.writeUInt32LE = function (e, t, n) {
            l(this, e, t, true, n);
          }, f.prototype.writeUInt32BE = function (e, t, n) {
            l(this, e, t, false, n);
          }, f.prototype.writeInt8 = function (e, t, n) {
            n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "Trying to write beyond buffer length"), F(e, 127, -128)), t >= this.length || (0 <= e ? this.writeUInt8(e, t, n) : this.writeUInt8(255 + e + 1, t, n));
          }, f.prototype.writeInt16LE = function (e, t, n) {
            B(this, e, t, true, n);
          }, f.prototype.writeInt16BE = function (e, t, n) {
            B(this, e, t, false, n);
          }, f.prototype.writeInt32LE = function (e, t, n) {
            L(this, e, t, true, n);
          }, f.prototype.writeInt32BE = function (e, t, n) {
            L(this, e, t, false, n);
          }, f.prototype.writeFloatLE = function (e, t, n) {
            U(this, e, t, true, n);
          }, f.prototype.writeFloatBE = function (e, t, n) {
            U(this, e, t, false, n);
          }, f.prototype.writeDoubleLE = function (e, t, n) {
            x(this, e, t, true, n);
          }, f.prototype.writeDoubleBE = function (e, t, n) {
            x(this, e, t, false, n);
          }, f.prototype.fill = function (e, t, n) {
            if (t = t || 0, n = n || this.length, d("number" == typeof (e = "string" == typeof (e = e || 0) ? e.charCodeAt(0) : e) && !isNaN(e), "value is not a number"), d(t <= n, "end < start"), n !== t && 0 !== this.length) {
              d(0 <= t && t < this.length, "start out of bounds"), d(0 <= n && n <= this.length, "end out of bounds");
              for (var r = t; r < n; r++) this[r] = e;
            }
          }, f.prototype.inspect = function () {
            for (var e = [], t = this.length, n = 0; n < t; n++) if (e[n] = k(this[n]), n === H.INSPECT_MAX_BYTES) {
              e[n + 1] = "...";
              break;
            }
            return "<Buffer " + e.join(" ") + ">";
          }, f.prototype.toArrayBuffer = function () {
            if ("undefined" == typeof Uint8Array) throw new Error("Buffer.toArrayBuffer not supported in this browser");
            if (f._useTypedArrays) return new f(this).buffer;
            for (var e = new Uint8Array(this.length), t = 0, n = e.length; t < n; t += 1) e[t] = this[t];
            return e.buffer;
          };
          var t = f.prototype;
          function S(e, t, n) {
            return "number" != typeof e ? n : t <= (e = ~~e) ? t : 0 <= e || 0 <= (e += t) ? e : 0;
          }
          function j(e) {
            return (e = ~~Math.ceil(+e)) < 0 ? 0 : e;
          }
          function C(e) {
            return (Array.isArray || function (e) {
              return "[object Array]" === Object.prototype.toString.call(e);
            })(e);
          }
          function k(e) {
            return e < 16 ? "0" + e.toString(16) : e.toString(16);
          }
          function T(e) {
            for (var t = [], n = 0; n < e.length; n++) {
              var r = e.charCodeAt(n);
              if (r <= 127) t.push(e.charCodeAt(n));else for (var o = n, i = (55296 <= r && r <= 57343 && n++, encodeURIComponent(e.slice(o, n + 1)).substr(1).split("%")), u = 0; u < i.length; u++) t.push(parseInt(i[u], 16));
            }
            return t;
          }
          function M(e) {
            return a.toByteArray(e);
          }
          function c(e, t, n, r) {
            for (var o = 0; o < r && !(o + n >= t.length || o >= e.length); o++) t[o + n] = e[o];
            return o;
          }
          function N(e) {
            try {
              return decodeURIComponent(e);
            } catch (e) {
              return String.fromCharCode(65533);
            }
          }
          function Y(e, t) {
            d("number" == typeof e, "cannot write a non-number as a number"), d(0 <= e, "specified a negative value for writing an unsigned value"), d(e <= t, "value is larger than maximum value for type"), d(Math.floor(e) === e, "value has a fractional component");
          }
          function F(e, t, n) {
            d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value"), d(Math.floor(e) === e, "value has a fractional component");
          }
          function D(e, t, n) {
            d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value");
          }
          function d(e, t) {
            if (!e) throw new Error(t || "Failed assertion");
          }
          f._augment = function (e) {
            return e._isBuffer = true, e._get = e.get, e._set = e.set, e.get = t.get, e.set = t.set, e.write = t.write, e.toString = t.toString, e.toLocaleString = t.toString, e.toJSON = t.toJSON, e.copy = t.copy, e.slice = t.slice, e.readUInt8 = t.readUInt8, e.readUInt16LE = t.readUInt16LE, e.readUInt16BE = t.readUInt16BE, e.readUInt32LE = t.readUInt32LE, e.readUInt32BE = t.readUInt32BE, e.readInt8 = t.readInt8, e.readInt16LE = t.readInt16LE, e.readInt16BE = t.readInt16BE, e.readInt32LE = t.readInt32LE, e.readInt32BE = t.readInt32BE, e.readFloatLE = t.readFloatLE, e.readFloatBE = t.readFloatBE, e.readDoubleLE = t.readDoubleLE, e.readDoubleBE = t.readDoubleBE, e.writeUInt8 = t.writeUInt8, e.writeUInt16LE = t.writeUInt16LE, e.writeUInt16BE = t.writeUInt16BE, e.writeUInt32LE = t.writeUInt32LE, e.writeUInt32BE = t.writeUInt32BE, e.writeInt8 = t.writeInt8, e.writeInt16LE = t.writeInt16LE, e.writeInt16BE = t.writeInt16BE, e.writeInt32LE = t.writeInt32LE, e.writeInt32BE = t.writeInt32BE, e.writeFloatLE = t.writeFloatLE, e.writeFloatBE = t.writeFloatBE, e.writeDoubleLE = t.writeDoubleLE, e.writeDoubleBE = t.writeDoubleBE, e.fill = t.fill, e.inspect = t.inspect, e.toArrayBuffer = t.toArrayBuffer, e;
          };
        }.call(this, O("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, O("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/buffer/index.js", "/node_modules/gulp-browserify/node_modules/buffer");
      }, {
        "base64-js": 2,
        buffer: 3,
        ieee754: 10,
        lYpoI2: 11
      }],
      4: [function (c, d, e) {
        !function (e, t, a, n, r, o, i, u, s) {
          var a = c("buffer").Buffer,
            f = 4,
            l = new a(f);
          l.fill(0);
          d.exports = {
            hash: function (e, t, n, r) {
              for (var o = t(function (e, t) {
                  e.length % f != 0 && (n = e.length + (f - e.length % f), e = a.concat([e, l], n));
                  for (var n, r = [], o = t ? e.readInt32BE : e.readInt32LE, i = 0; i < e.length; i += f) r.push(o.call(e, i));
                  return r;
                }(e = a.isBuffer(e) ? e : new a(e), r), 8 * e.length), t = r, i = new a(n), u = t ? i.writeInt32BE : i.writeInt32LE, s = 0; s < o.length; s++) u.call(i, o[s], 4 * s, true);
              return i;
            }
          };
        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/helpers.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        buffer: 3,
        lYpoI2: 11
      }],
      5: [function (v, e, _) {
        !function (l, c, u, d, h, p, g, y, w) {
          var u = v("buffer").Buffer,
            e = v("./sha"),
            t = v("./sha256"),
            n = v("./rng"),
            b = {
              sha1: e,
              sha256: t,
              md5: v("./md5")
            },
            s = 64,
            a = new u(s);
          function r(e, n) {
            var r = b[e = e || "sha1"],
              o = [];
            return r || i("algorithm:", e, "is not yet supported"), {
              update: function (e) {
                return u.isBuffer(e) || (e = new u(e)), o.push(e), e.length, this;
              },
              digest: function (e) {
                var t = u.concat(o),
                  t = n ? function (e, t, n) {
                    u.isBuffer(t) || (t = new u(t)), u.isBuffer(n) || (n = new u(n)), t.length > s ? t = e(t) : t.length < s && (t = u.concat([t, a], s));
                    for (var r = new u(s), o = new u(s), i = 0; i < s; i++) r[i] = 54 ^ t[i], o[i] = 92 ^ t[i];
                    return n = e(u.concat([r, n])), e(u.concat([o, n]));
                  }(r, n, t) : r(t);
                return o = null, e ? t.toString(e) : t;
              }
            };
          }
          function i() {
            var e = [].slice.call(arguments).join(" ");
            throw new Error([e, "we accept pull requests", "http://github.com/dominictarr/crypto-browserify"].join("\n"));
          }
          a.fill(0), _.createHash = function (e) {
            return r(e);
          }, _.createHmac = r, _.randomBytes = function (e, t) {
            if (!t || !t.call) return new u(n(e));
            try {
              t.call(this, void 0, new u(n(e)));
            } catch (e) {
              t(e);
            }
          };
          var o,
            f = ["createCredentials", "createCipher", "createCipheriv", "createDecipher", "createDecipheriv", "createSign", "createVerify", "createDiffieHellman", "pbkdf2"],
            m = function (e) {
              _[e] = function () {
                i("sorry,", e, "is not implemented yet");
              };
            };
          for (o in f) m(f[o]);
        }.call(this, v("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, v("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/index.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        "./md5": 6,
        "./rng": 7,
        "./sha": 8,
        "./sha256": 9,
        buffer: 3,
        lYpoI2: 11
      }],
      6: [function (w, b, e) {
        !function (e, r, o, i, u, a, f, l, y) {
          var t = w("./helpers");
          function n(e, t) {
            e[t >> 5] |= 128 << t % 32, e[14 + (t + 64 >>> 9 << 4)] = t;
            for (var n = 1732584193, r = -271733879, o = -1732584194, i = 271733878, u = 0; u < e.length; u += 16) {
              var s = n,
                a = r,
                f = o,
                l = i,
                n = c(n, r, o, i, e[u + 0], 7, -680876936),
                i = c(i, n, r, o, e[u + 1], 12, -389564586),
                o = c(o, i, n, r, e[u + 2], 17, 606105819),
                r = c(r, o, i, n, e[u + 3], 22, -1044525330);
              n = c(n, r, o, i, e[u + 4], 7, -176418897), i = c(i, n, r, o, e[u + 5], 12, 1200080426), o = c(o, i, n, r, e[u + 6], 17, -1473231341), r = c(r, o, i, n, e[u + 7], 22, -45705983), n = c(n, r, o, i, e[u + 8], 7, 1770035416), i = c(i, n, r, o, e[u + 9], 12, -1958414417), o = c(o, i, n, r, e[u + 10], 17, -42063), r = c(r, o, i, n, e[u + 11], 22, -1990404162), n = c(n, r, o, i, e[u + 12], 7, 1804603682), i = c(i, n, r, o, e[u + 13], 12, -40341101), o = c(o, i, n, r, e[u + 14], 17, -1502002290), n = d(n, r = c(r, o, i, n, e[u + 15], 22, 1236535329), o, i, e[u + 1], 5, -165796510), i = d(i, n, r, o, e[u + 6], 9, -1069501632), o = d(o, i, n, r, e[u + 11], 14, 643717713), r = d(r, o, i, n, e[u + 0], 20, -373897302), n = d(n, r, o, i, e[u + 5], 5, -701558691), i = d(i, n, r, o, e[u + 10], 9, 38016083), o = d(o, i, n, r, e[u + 15], 14, -660478335), r = d(r, o, i, n, e[u + 4], 20, -405537848), n = d(n, r, o, i, e[u + 9], 5, 568446438), i = d(i, n, r, o, e[u + 14], 9, -1019803690), o = d(o, i, n, r, e[u + 3], 14, -187363961), r = d(r, o, i, n, e[u + 8], 20, 1163531501), n = d(n, r, o, i, e[u + 13], 5, -1444681467), i = d(i, n, r, o, e[u + 2], 9, -51403784), o = d(o, i, n, r, e[u + 7], 14, 1735328473), n = h(n, r = d(r, o, i, n, e[u + 12], 20, -1926607734), o, i, e[u + 5], 4, -378558), i = h(i, n, r, o, e[u + 8], 11, -2022574463), o = h(o, i, n, r, e[u + 11], 16, 1839030562), r = h(r, o, i, n, e[u + 14], 23, -35309556), n = h(n, r, o, i, e[u + 1], 4, -1530992060), i = h(i, n, r, o, e[u + 4], 11, 1272893353), o = h(o, i, n, r, e[u + 7], 16, -155497632), r = h(r, o, i, n, e[u + 10], 23, -1094730640), n = h(n, r, o, i, e[u + 13], 4, 681279174), i = h(i, n, r, o, e[u + 0], 11, -358537222), o = h(o, i, n, r, e[u + 3], 16, -722521979), r = h(r, o, i, n, e[u + 6], 23, 76029189), n = h(n, r, o, i, e[u + 9], 4, -640364487), i = h(i, n, r, o, e[u + 12], 11, -421815835), o = h(o, i, n, r, e[u + 15], 16, 530742520), n = p(n, r = h(r, o, i, n, e[u + 2], 23, -995338651), o, i, e[u + 0], 6, -198630844), i = p(i, n, r, o, e[u + 7], 10, 1126891415), o = p(o, i, n, r, e[u + 14], 15, -1416354905), r = p(r, o, i, n, e[u + 5], 21, -57434055), n = p(n, r, o, i, e[u + 12], 6, 1700485571), i = p(i, n, r, o, e[u + 3], 10, -1894986606), o = p(o, i, n, r, e[u + 10], 15, -1051523), r = p(r, o, i, n, e[u + 1], 21, -2054922799), n = p(n, r, o, i, e[u + 8], 6, 1873313359), i = p(i, n, r, o, e[u + 15], 10, -30611744), o = p(o, i, n, r, e[u + 6], 15, -1560198380), r = p(r, o, i, n, e[u + 13], 21, 1309151649), n = p(n, r, o, i, e[u + 4], 6, -145523070), i = p(i, n, r, o, e[u + 11], 10, -1120210379), o = p(o, i, n, r, e[u + 2], 15, 718787259), r = p(r, o, i, n, e[u + 9], 21, -343485551), n = g(n, s), r = g(r, a), o = g(o, f), i = g(i, l);
            }
            return Array(n, r, o, i);
          }
          function s(e, t, n, r, o, i) {
            return g((t = g(g(t, e), g(r, i))) << o | t >>> 32 - o, n);
          }
          function c(e, t, n, r, o, i, u) {
            return s(t & n | ~t & r, e, t, o, i, u);
          }
          function d(e, t, n, r, o, i, u) {
            return s(t & r | n & ~r, e, t, o, i, u);
          }
          function h(e, t, n, r, o, i, u) {
            return s(t ^ n ^ r, e, t, o, i, u);
          }
          function p(e, t, n, r, o, i, u) {
            return s(n ^ (t | ~r), e, t, o, i, u);
          }
          function g(e, t) {
            var n = (65535 & e) + (65535 & t);
            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
          }
          b.exports = function (e) {
            return t.hash(e, n, 16);
          };
        }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/md5.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        "./helpers": 4,
        buffer: 3,
        lYpoI2: 11
      }],
      7: [function (e, l, t) {
        !function (e, t, n, r, o, i, u, s, f) {
          l.exports = function (e) {
            for (var t, n = new Array(e), r = 0; r < e; r++) 0 == (3 & r) && (t = 4294967296 * Math.random()), n[r] = t >>> ((3 & r) << 3) & 255;
            return n;
          };
        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/rng.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        buffer: 3,
        lYpoI2: 11
      }],
      8: [function (c, d, e) {
        !function (e, t, n, r, o, s, a, f, l) {
          var i = c("./helpers");
          function u(l, c) {
            l[c >> 5] |= 128 << 24 - c % 32, l[15 + (c + 64 >> 9 << 4)] = c;
            for (var e, t, n, r = Array(80), o = 1732584193, i = -271733879, u = -1732584194, s = 271733878, d = -1009589776, h = 0; h < l.length; h += 16) {
              for (var p = o, g = i, y = u, w = s, b = d, a = 0; a < 80; a++) {
                r[a] = a < 16 ? l[h + a] : v(r[a - 3] ^ r[a - 8] ^ r[a - 14] ^ r[a - 16], 1);
                var f = m(m(v(o, 5), (f = i, t = u, n = s, (e = a) < 20 ? f & t | ~f & n : !(e < 40) && e < 60 ? f & t | f & n | t & n : f ^ t ^ n)), m(m(d, r[a]), (e = a) < 20 ? 1518500249 : e < 40 ? 1859775393 : e < 60 ? -1894007588 : -899497514)),
                  d = s,
                  s = u,
                  u = v(i, 30),
                  i = o,
                  o = f;
              }
              o = m(o, p), i = m(i, g), u = m(u, y), s = m(s, w), d = m(d, b);
            }
            return Array(o, i, u, s, d);
          }
          function m(e, t) {
            var n = (65535 & e) + (65535 & t);
            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
          }
          function v(e, t) {
            return e << t | e >>> 32 - t;
          }
          d.exports = function (e) {
            return i.hash(e, u, 20, true);
          };
        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        "./helpers": 4,
        buffer: 3,
        lYpoI2: 11
      }],
      9: [function (c, d, e) {
        !function (e, t, n, r, u, s, a, f, l) {
          function b(e, t) {
            var n = (65535 & e) + (65535 & t);
            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
          }
          function o(e, l) {
            var c,
              d = new Array(1116352408, 1899447441, 3049323471, 3921009573, 961987163, 1508970993, 2453635748, 2870763221, 3624381080, 310598401, 607225278, 1426881987, 1925078388, 2162078206, 2614888103, 3248222580, 3835390401, 4022224774, 264347078, 604807628, 770255983, 1249150122, 1555081692, 1996064986, 2554220882, 2821834349, 2952996808, 3210313671, 3336571891, 3584528711, 113926993, 338241895, 666307205, 773529912, 1294757372, 1396182291, 1695183700, 1986661051, 2177026350, 2456956037, 2730485921, 2820302411, 3259730800, 3345764771, 3516065817, 3600352804, 4094571909, 275423344, 430227734, 506948616, 659060556, 883997877, 958139571, 1322822218, 1537002063, 1747873779, 1955562222, 2024104815, 2227730452, 2361852424, 2428436474, 2756734187, 3204031479, 3329325298),
              t = new Array(1779033703, 3144134277, 1013904242, 2773480762, 1359893119, 2600822924, 528734635, 1541459225),
              n = new Array(64);
            e[l >> 5] |= 128 << 24 - l % 32, e[15 + (l + 64 >> 9 << 4)] = l;
            for (var r, o, h = 0; h < e.length; h += 16) {
              for (var i = t[0], u = t[1], s = t[2], p = t[3], a = t[4], g = t[5], y = t[6], w = t[7], f = 0; f < 64; f++) n[f] = f < 16 ? e[f + h] : b(b(b((o = n[f - 2], m(o, 17) ^ m(o, 19) ^ v(o, 10)), n[f - 7]), (o = n[f - 15], m(o, 7) ^ m(o, 18) ^ v(o, 3))), n[f - 16]), c = b(b(b(b(w, m(o = a, 6) ^ m(o, 11) ^ m(o, 25)), a & g ^ ~a & y), d[f]), n[f]), r = b(m(r = i, 2) ^ m(r, 13) ^ m(r, 22), i & u ^ i & s ^ u & s), w = y, y = g, g = a, a = b(p, c), p = s, s = u, u = i, i = b(c, r);
              t[0] = b(i, t[0]), t[1] = b(u, t[1]), t[2] = b(s, t[2]), t[3] = b(p, t[3]), t[4] = b(a, t[4]), t[5] = b(g, t[5]), t[6] = b(y, t[6]), t[7] = b(w, t[7]);
            }
            return t;
          }
          var i = c("./helpers"),
            m = function (e, t) {
              return e >>> t | e << 32 - t;
            },
            v = function (e, t) {
              return e >>> t;
            };
          d.exports = function (e) {
            return i.hash(e, o, 32, true);
          };
        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha256.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
      }, {
        "./helpers": 4,
        buffer: 3,
        lYpoI2: 11
      }],
      10: [function (e, t, f) {
        !function (e, t, n, r, o, i, u, s, a) {
          f.read = function (e, t, n, r, o) {
            var i,
              u,
              l = 8 * o - r - 1,
              c = (1 << l) - 1,
              d = c >> 1,
              s = -7,
              a = n ? o - 1 : 0,
              f = n ? -1 : 1,
              o = e[t + a];
            for (a += f, i = o & (1 << -s) - 1, o >>= -s, s += l; 0 < s; i = 256 * i + e[t + a], a += f, s -= 8);
            for (u = i & (1 << -s) - 1, i >>= -s, s += r; 0 < s; u = 256 * u + e[t + a], a += f, s -= 8);
            if (0 === i) i = 1 - d;else {
              if (i === c) return u ? NaN : 1 / 0 * (o ? -1 : 1);
              u += Math.pow(2, r), i -= d;
            }
            return (o ? -1 : 1) * u * Math.pow(2, i - r);
          }, f.write = function (e, t, l, n, r, c) {
            var o,
              i,
              u = 8 * c - r - 1,
              s = (1 << u) - 1,
              a = s >> 1,
              d = 23 === r ? Math.pow(2, -24) - Math.pow(2, -77) : 0,
              f = n ? 0 : c - 1,
              h = n ? 1 : -1,
              c = t < 0 || 0 === t && 1 / t < 0 ? 1 : 0;
            for (t = Math.abs(t), isNaN(t) || t === 1 / 0 ? (i = isNaN(t) ? 1 : 0, o = s) : (o = Math.floor(Math.log(t) / Math.LN2), t * (n = Math.pow(2, -o)) < 1 && (o--, n *= 2), 2 <= (t += 1 <= o + a ? d / n : d * Math.pow(2, 1 - a)) * n && (o++, n /= 2), s <= o + a ? (i = 0, o = s) : 1 <= o + a ? (i = (t * n - 1) * Math.pow(2, r), o += a) : (i = t * Math.pow(2, a - 1) * Math.pow(2, r), o = 0)); 8 <= r; e[l + f] = 255 & i, f += h, i /= 256, r -= 8);
            for (o = o << r | i, u += r; 0 < u; e[l + f] = 255 & o, f += h, o /= 256, u -= 8);
            e[l + f - h] |= 128 * c;
          };
        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/ieee754/index.js", "/node_modules/gulp-browserify/node_modules/ieee754");
      }, {
        buffer: 3,
        lYpoI2: 11
      }],
      11: [function (e, h, t) {
        !function (e, t, n, r, o, f, l, c, d) {
          var i, u, s;
          function a() {}
          (e = h.exports = {}).nextTick = (u = "undefined" != typeof window && window.setImmediate, s = "undefined" != typeof window && window.postMessage && window.addEventListener, u ? function (e) {
            return window.setImmediate(e);
          } : s ? (i = [], window.addEventListener("message", function (e) {
            var t = e.source;
            t !== window && null !== t || "process-tick" !== e.data || (e.stopPropagation(), 0 < i.length && i.shift()());
          }, true), function (e) {
            i.push(e), window.postMessage("process-tick", "*");
          }) : function (e) {
            setTimeout(e, 0);
          }), e.title = "browser", e.browser = true, e.env = {}, e.argv = [], e.on = a, e.addListener = a, e.once = a, e.off = a, e.removeListener = a, e.removeAllListeners = a, e.emit = a, e.binding = function (e) {
            throw new Error("process.binding is not supported");
          }, e.cwd = function () {
            return "/";
          }, e.chdir = function (e) {
            throw new Error("process.chdir is not supported");
          };
        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/process/browser.js", "/node_modules/gulp-browserify/node_modules/process");
      }, {
        buffer: 3,
        lYpoI2: 11
      }]
    }, {}, [1])(1);
  });
})(object_hash);
var object_hashExports = object_hash.exports;
var hash = /*@__PURE__*/getDefaultExportFromCjs(object_hashExports);

/**
 * Scale matrix rows to match integration (sum) of target
 * @param matrix - Matrix to scale (modified in place)
 * @param normalizedTarget - Target data with x and y arrays
 * @param range - Optional range to consider
 */
function integration(matrix, normalizedTarget, range = {}) {
  const fromToIndex = xGetFromToIndex(normalizedTarget.x, range);
  const targetValue = xSum(normalizedTarget.y, fromToIndex);
  const values = matrix.map(row => xSum(row, fromToIndex));
  for (let i = 0; i < matrix.length; i++) {
    const factor = targetValue / values[i];
    matrix[i] = xMultiply(matrix[i], factor);
  }
}

/**
 * Scale matrix rows to match maximum value of target
 * @param matrix - Matrix to scale (modified in place)
 * @param normalizedTarget - Target data with x and y arrays
 * @param range - Optional range to consider
 */
function max(matrix, normalizedTarget, range = {}) {
  const fromToIndex = xGetFromToIndex(normalizedTarget.x, range);
  const targetValue = xMaxValue(normalizedTarget.y, fromToIndex);
  const values = matrix.map(row => xMaxValue(row, fromToIndex));
  for (let i = 0; i < matrix.length; i++) {
    const factor = targetValue / values[i];
    matrix[i] = xMultiply(matrix[i], factor);
  }
}

/**
 * Scale matrix rows to match minimum value of target
 * @param matrix - Matrix to scale (modified in place)
 * @param normalizedTarget - Target data with x and y arrays
 * @param range - Optional range to consider
 */
function min(matrix, normalizedTarget, range = {}) {
  const fromToIndex = xGetFromToIndex(normalizedTarget.x, range);
  const targetValue = xMinValue(normalizedTarget.y, fromToIndex);
  const values = matrix.map(row => xMinValue(row, fromToIndex));
  for (let i = 0; i < matrix.length; i++) {
    const factor = targetValue / values[i];
    matrix[i] = xMultiply(matrix[i], factor);
  }
}

/**
 * Scale matrix rows to match min-max range of target
 * @param matrix - Matrix to scale (modified in place)
 * @param normalizedTarget - Target data with x and y arrays
 * @param range - Optional range to consider
 */
function minMax(matrix, normalizedTarget, range = {}) {
  const fromToIndex = xGetFromToIndex(normalizedTarget.x, range);
  const targetValue = {
    min: xMinValue(normalizedTarget.y, fromToIndex),
    max: xMaxValue(normalizedTarget.y, fromToIndex)
  };
  const deltaTarget = targetValue.max - targetValue.min;
  const minTarget = targetValue.min;
  const values = matrix.map(row => {
    return {
      min: xMinValue(row, fromToIndex),
      max: xMaxValue(row, fromToIndex)
    };
  });
  for (let i = 0; i < matrix.length; i++) {
    const deltaSource = values[i].max - values[i].min;
    const minSource = values[i].min;
    const newData = [];
    for (let j = 0; j < normalizedTarget.y.length; j++) {
      newData.push((matrix[i][j] - minSource) / deltaSource * deltaTarget + minTarget);
    }
    matrix[i] = newData;
  }
}

let cache = {};
/**
 * Calculate post-processed data with various transformations and calculations
 * @param spectraProcessor - SpectraProcessor instance
 * @param options - Processing options
 * @returns Post-processed data
 */
function getPostProcessedData(spectraProcessor, options = {}) {
  const optionsHash = hash(options);
  if (!spectraProcessor.spectra || spectraProcessor.spectra.length === 0) {
    return {};
  }
  const {
    scale = {},
    ids,
    ranges,
    calculations,
    filters = []
  } = options;
  const {
    range,
    targetID,
    relative,
    method = ''
  } = scale;
  const spectra = spectraProcessor.getSpectra(ids);
  // Check if we can reuse the cache
  if (cache.optionsHash === optionsHash) {
    let validCache = true;
    for (const spectrum of spectra) {
      if (!cache.weakMap?.get(spectrum.normalized)) validCache = false;
    }
    if (validCache) return cache;
  }
  const weakMap = new WeakMap();
  for (const spectrum of spectra) {
    weakMap.set(spectrum.normalized, true);
  }
  const normalizedData = getNormalizedData(spectra);
  for (const filter of filters) {
    switch (filter.name) {
      case 'pqn':
        {
          normalizedData.matrix = matrixPQN(normalizedData.matrix, filter.options).data;
          break;
        }
      case 'centerMean':
        {
          normalizedData.matrix = matrixCenterZMean(normalizedData.matrix);
          break;
        }
      case 'rescale':
        {
          normalizedData.matrix = matrixZRescale(normalizedData.matrix, filter.options);
          break;
        }
      case '':
      case undefined:
        break;
      default:
        throw new Error(`Unknown matrix filter name: ${filter.name}`);
    }
  }
  const normalizedTarget = targetID ? spectraProcessor.getSpectrum(targetID)?.normalized : spectraProcessor.spectra[0].normalized;
  if (!normalizedTarget) {
    throw new Error('No normalized target found');
  }
  if (method) {
    switch (method.toLowerCase()) {
      case 'min':
        min(normalizedData.matrix, normalizedTarget, range);
        break;
      case 'max':
        max(normalizedData.matrix, normalizedTarget, range);
        break;
      case 'minmax':
        minMax(normalizedData.matrix, normalizedTarget, range);
        break;
      case 'integration':
        integration(normalizedData.matrix, normalizedTarget, range);
        break;
      default:
        throw new Error(`getPostProcessedData: unknown method: ${method}`);
    }
  }
  if (relative) {
    for (let i = 0; i < normalizedData.matrix.length; i++) {
      normalizedData.matrix[i] = xSubtract(normalizedData.matrix[i], normalizedTarget.y);
    }
  }
  const result = normalizedData;
  if (ranges) {
    result.ranges = [];
    for (const spectrum of normalizedData.matrix) {
      const rangesCopy = structuredClone(ranges);
      const yNormalized = spectrum;
      const resultRanges = {};
      result.ranges.push(resultRanges);
      for (const currentRange of rangesCopy) {
        if (currentRange.label) {
          const fromToIndex = xGetFromToIndex(normalizedTarget.x, currentRange);
          const deltaX = normalizedTarget.x[1] - normalizedTarget.x[0];
          currentRange.integration = xSum(yNormalized, fromToIndex) * deltaX;
          currentRange.maxPoint = xyMaxYPoint({
            x: normalizedData.x,
            y: yNormalized
          }, fromToIndex);
          resultRanges[currentRange.label] = currentRange;
        }
      }
    }
  }
  if (calculations && result.ranges) {
    result.calculations = result.ranges.map(() => {
      return {};
    });
    const parameters = Object.keys(result.ranges[0]);
    for (const calculation of calculations) {
      // eslint-disable-next-line @typescript-eslint/no-implied-eval, no-new-func
      const callback = new Function(...parameters, `return ${calculation.formula}`);
      for (let i = 0; i < result.ranges.length; i++) {
        const oneRanges = result.ranges[i];
        const values = parameters.map(key => oneRanges[key].integration);
        result.calculations[i][calculation.label] = callback(...values);
      }
    }
  }
  cache = {
    ...result,
    optionsHash,
    weakMap
  };
  return cache;
}

/**
 * Get post-processed data as text
 * @param spectraProcessor - SpectraProcessor instance
 * @param options - Options for post-processing and formatting
 * @returns Text representation of post-processed data
 */
function getPostProcessedText(spectraProcessor, options = {}) {
  const {
    fs = '\t',
    rs = '\n',
    postProcessing: postProcessingOptions = {}
  } = options;
  const data = getPostProcessedData(spectraProcessor, postProcessingOptions);
  // Only convert if we have valid data
  if (!data.matrix || !data.x || !data.ids || !data.meta) {
    return '';
  }
  return convertToText({
    matrix: data.matrix,
    x: data.x,
    ids: data.ids,
    meta: data.meta
  }, {
    rs,
    fs
  });
}

/**
 * Get spectrum data with optional filtering and scaling
 * @param spectrum - The spectrum object containing x and y arrays
 * @param options - Configuration options
 * @returns Data object with filtered/scaled x and y arrays
 */
function getData(spectrum, options = {}) {
  const {
    xFilter = {},
    yFactor = 1
  } = options;
  let data = {
    x: spectrum.x,
    y: spectrum.y
  };
  if (xFilter) {
    data = xyFilterX(spectrum, xFilter);
  }
  if (yFactor && yFactor !== 1) {
    data.y = data.y.map(y => y * yFactor);
  }
  return data;
}

/**
 * Center the mean
 * @param data
 */
function centerMean(data) {
  const {
    y
  } = data;
  const mean = xMean(y);
  for (let i = 0; i < y.length; i++) {
    y[i] -= mean;
  }
  return {
    data
  };
}

/**
 * Center the median
 * @param data
 */
function centerMedian(data) {
  const {
    y
  } = data;
  const median = xMedian(y);
  for (let i = 0; i < y.length; i++) {
    y[i] -= median;
  }
  return {
    data
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function fromTo(data, options = {}) {
  const {
    fromIndex,
    toIndex
  } = xGetFromToIndex(data.x, options);
  return {
    data: {
      x: data.x.subarray(fromIndex, toIndex + 1),
      y: data.y.subarray(fromIndex, toIndex + 1)
    }
  };
}

/**
 * Norm the Y values
 * @param data
 * @param options
 */
function normed(data, options = {}) {
  xNormed(data.y, {
    ...options,
    output: data.y
  });
  return {
    data
  };
}

/**
 * Center the mean
 * @param data
 */
function divideBySD(data) {
  const {
    y
  } = data;
  const sd = xStandardDeviation(y);
  for (let i = 0; i < y.length; i++) {
    y[i] /= sd;
  }
  return {
    data
  };
}

/**
 * Center the mean
 * @param data
 * @param options
 */
function rescale(data, options = {}) {
  xRescale(data.y, {
    ...options,
    output: data.y
  });
  return {
    data
  };
}

/**
 * Filter that allows to
 * @param data
 */
function paretoNormalization(data) {
  return {
    data: {
      x: data.x,
      y: xParetoNormalization(data.y)
    }
  };
}

// Based on https://github.com/scijs/cholesky-solve

/*
The MIT License (MIT)

Copyright (c) 2013 Eric Arnebäck

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

function ldlSymbolic(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n + 1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Lp /* output of size n + 1, not defined on input */, Parent /* output of size n, not defined on input */, Lnz /* output of size n, not defined on input */, Flag /* workspace of size n, not defn. on input or output */) {
  let i, k, p, kk, p2;
  for (k = 0; k < n; k++) {
    /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
    Parent[k] = -1; /* parent of k is not yet known */
    Flag[k] = k; /* mark node k as visited */
    Lnz[k] = 0; /* count of nonzeros in column k of L */
    kk = k; /* kth original, or permuted, column */
    p2 = Ap[kk + 1];
    for (p = Ap[kk]; p < p2; p++) {
      /* A (i,k) is nonzero (original or permuted A) */
      i = Ai[p];
      if (i < k) {
        /* follow path from i to root of etree, stop at flagged node */
        for (; Flag[i] !== k; i = Parent[i]) {
          /* find parent of i if not yet determined */
          if (Parent[i] === -1) Parent[i] = k;
          Lnz[i]++; /* L (k,i) is nonzero */
          Flag[i] = k; /* mark i as visited */
        }
      }
    }
  }
  /* construct Lp index array from Lnz column counts */
  Lp[0] = 0;
  for (k = 0; k < n; k++) {
    Lp[k + 1] = Lp[k] + Lnz[k];
  }
}
function ldlNumeric(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n+1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Ax /* input of size nz=Ap[n], not modified */, Lp /* input of size n+1, not modified */, Parent /* input of size n, not modified */, Lnz /* output of size n, not defn. on input */, Li /* output of size lnz=Lp[n], not defined on input */, Lx /* output of size lnz=Lp[n], not defined on input */, D /* output of size n, not defined on input */, Y /* workspace of size n, not defn. on input or output */, Pattern /* workspace of size n, not defn. on input or output */, Flag /* workspace of size n, not defn. on input or output */) {
  let yi, lKi;
  let i, k, p, kk, p2, len, top;
  for (k = 0; k < n; k++) {
    /* compute nonzero Pattern of kth row of L, in topological order */
    Y[k] = 0.0; /* Y(0:k) is now all zero */
    top = n; /* stack for pattern is empty */
    Flag[k] = k; /* mark node k as visited */
    Lnz[k] = 0; /* count of nonzeros in column k of L */
    kk = k; /* kth original, or permuted, column */
    p2 = Ap[kk + 1];
    for (p = Ap[kk]; p < p2; p++) {
      i = Ai[p]; /* get A(i,k) */
      if (i <= k) {
        Y[i] += Ax[p]; /* scatter A(i,k) into Y (sum duplicates) */
        for (len = 0; Flag[i] !== k; i = Parent[i]) {
          Pattern[len++] = i; /* L(k,i) is nonzero */
          Flag[i] = k; /* mark i as visited */
        }
        while (len > 0) Pattern[--top] = Pattern[--len];
      }
    }
    /* compute numerical values kth row of L (a sparse triangular solve) */
    D[k] = Y[k]; /* get D(k,k) and clear Y(k) */
    Y[k] = 0.0;
    for (; top < n; top++) {
      i = Pattern[top]; /* Pattern[top:n-1] is pattern of L(:,k) */
      yi = Y[i]; /* get and clear Y(i) */
      Y[i] = 0.0;
      p2 = Lp[i] + Lnz[i];
      for (p = Lp[i]; p < p2; p++) {
        Y[Li[p]] -= Lx[p] * yi;
      }
      lKi = yi / D[i]; /* the nonzero entry L(k,i) */
      D[k] -= lKi * yi;
      Li[p] = k; /* store L(k,i) in column form of L */
      Lx[p] = lKi;
      Lnz[i]++; /* increment count of nonzeros in col i */
    }
    if (D[k] === 0.0) return k; /* failure, D(k,k) is zero */
  }
  return n; /* success, diagonal of D is all nonzero */
}
function ldlLsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
  let j, p, p2;
  for (j = 0; j < n; j++) {
    p2 = Lp[j + 1];
    for (p = Lp[j]; p < p2; p++) {
      X[Li[p]] -= Lx[p] * X[j];
    }
  }
}
function ldlDsolve(n /* D is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, D /* input of size n, not modified */) {
  let j;
  for (j = 0; j < n; j++) {
    X[j] /= D[j];
  }
}
function ldlLTsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
  let j, p, p2;
  for (j = n - 1; j >= 0; j--) {
    p2 = Lp[j + 1];
    for (p = Lp[j]; p < p2; p++) {
      X[j] -= Lx[p] * X[Li[p]];
    }
  }
}
function ldlPerm(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
  let j;
  for (j = 0; j < n; j++) {
    X[j] = B[P[j]];
  }
}
function ldlPermt(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
  let j;
  for (j = 0; j < n; j++) {
    X[P[j]] = B[j];
  }
}
function prepare(M, n, P) {
  // if a permutation was specified, apply it.
  if (P) {
    let Pinv = new Array(n);
    for (let k = 0; k < n; k++) {
      Pinv[P[k]] = k;
    }
    let Mt = []; // scratch memory
    // Apply permutation. We make M into P*M*P^T
    for (let a = 0; a < M.length; ++a) {
      let ar = Pinv[M[a][0]];
      let ac = Pinv[M[a][1]];

      // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
      // if permuted element is below diagonal, we simply transpose it.
      if (ac < ar) {
        let t = ac;
        ac = ar;
        ar = t;
      }
      Mt[a] = [];
      Mt[a][0] = ar;
      Mt[a][1] = ac;
      Mt[a][2] = M[a][2];
    }
    M = Mt; // copy scratch memory.
  } else {
    // if P argument is null, we just use an identity permutation.
    P = [];
    for (let i = 0; i < n; ++i) {
      P[i] = i;
    }
  }

  // The sparse matrix we are decomposing is A.
  // Now we shall create A from M.
  let Ap = new Array(n + 1);
  let Ai = new Array(M.length);
  let Ax = new Array(M.length);

  // count number of non-zero elements in columns.
  let LNZ = [];
  for (let i = 0; i < n; ++i) {
    LNZ[i] = 0;
  }
  for (let a = 0; a < M.length; ++a) {
    LNZ[M[a][1]]++;
  }
  Ap[0] = 0;
  for (let i = 0; i < n; ++i) {
    Ap[i + 1] = Ap[i] + LNZ[i];
  }
  let coloffset = [];
  for (let a = 0; a < n; ++a) {
    coloffset[a] = 0;
  }

  // go through all elements in M, and add them to sparse matrix A.
  for (let i = 0; i < M.length; ++i) {
    let e = M[i];
    let col = e[1];
    let adr = Ap[col] + coloffset[col];
    Ai[adr] = e[0];
    Ax[adr] = e[2];
    coloffset[col]++;
  }
  let D = new Array(n);
  let Y = new Array(n);
  let Lp = new Array(n + 1);
  let Parent = new Array(n);
  let Lnz = new Array(n);
  let Flag = new Array(n);
  let Pattern = new Array(n);
  let bp1 = new Array(n);
  let x = new Array(n);
  let d;
  ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
  let Lx = new Array(Lp[n]);
  let Li = new Array(Lp[n]);
  d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);
  if (d === n) {
    return b => {
      ldlPerm(n, bp1, b, P);
      ldlLsolve(n, bp1, Lp, Li, Lx);
      ldlDsolve(n, bp1, D);
      ldlLTsolve(n, bp1, Lp, Li, Lx);
      ldlPermt(n, x, bp1, P);
      return x;
    };
  } else {
    return null;
  }
}

var cuthillMckee_1 = cuthillMckee$1;
function compareNum(a, b) {
  return a - b;
}
function cuthillMckee$1(list, n) {
  var adj = new Array(n);
  var visited = new Array(n);
  for (var i = 0; i < n; ++i) {
    adj[i] = [];
    visited[i] = false;
  }
  for (var i = 0; i < list.length; ++i) {
    var l = list[i];
    adj[l[0]].push(l[1]);
  }
  var toVisit = new Array(n);
  var eol = 0;
  var ptr = 0;
  for (var i = 0; i < n; ++i) {
    if (visited[i]) {
      continue;
    }
    toVisit[eol++] = i;
    visited[i] = true;
    while (ptr < eol) {
      var v = toVisit[ptr++];
      var nbhd = adj[v];
      nbhd.sort(compareNum);
      for (var j = 0; j < nbhd.length; ++j) {
        var u = nbhd[j];
        if (visited[u]) {
          continue;
        }
        visited[u] = true;
        toVisit[eol++] = u;
      }
    }
  }
  var result = new Array(n);
  for (var i = 0; i < n; ++i) {
    result[toVisit[i]] = i;
  }
  return result;
}
var cuthillMckee = /*@__PURE__*/getDefaultExportFromCjs(cuthillMckee_1);

const getClosestNumber = (array = [], goal = 0) => {
  const closest = array.reduce((prev, curr) => {
    return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
  });
  return closest;
};
const getCloseIndex = (array = [], goal = 0) => {
  const closest = getClosestNumber(array, goal);
  return array.indexOf(closest);
};
const updateSystem = (matrix, y, weights) => {
  let nbPoints = y.length;
  let l = nbPoints - 1;
  let newMatrix = new Array(matrix.length);
  let newVector = new Float64Array(nbPoints);
  for (let i = 0; i < l; i++) {
    let w = weights[i];
    let diag = i * 2;
    let next = diag + 1;
    newMatrix[diag] = matrix[diag].slice();
    newMatrix[next] = matrix[next].slice();
    if (w === 0) {
      newVector[i] = 0;
    } else {
      newVector[i] = y[i] * w;
      newMatrix[diag][2] += w;
    }
  }
  newVector[l] = y[l] * weights[l];
  newMatrix[l * 2] = matrix[l * 2].slice();
  newMatrix[l * 2][2] += weights[l];
  return [newMatrix, newVector];
};
const getDeltaMatrix = (nbPoints, lambda) => {
  let matrix = [];
  let last = nbPoints - 1;
  for (let i = 0; i < last; i++) {
    matrix.push([i, i, lambda * 2]);
    matrix.push([i + 1, i, -1 * lambda]);
  }
  matrix[0][2] = lambda;
  matrix.push([last, last, lambda]);
  return {
    lowerTriangularNonZeros: matrix,
    permutationEncodedArray: cuthillMckee(matrix, nbPoints)
  };
};

function getControlPoints(x, y, options = {}) {
  const {
    length
  } = x;
  let {
    controlPoints = Int8Array.from({
      length
    }).fill(0)
  } = options;
  const {
    zones = [],
    weights = Float64Array.from({
      length
    }).fill(1)
  } = options;
  if (x.length !== y.length) {
    throw new RangeError('Y should match the length with X');
  } else if (controlPoints.length !== x.length) {
    throw new RangeError('controlPoints should match the length with X');
  } else if (weights.length !== x.length) {
    throw new RangeError('weights should match the length with X');
  }
  zones.forEach(range => {
    let indexFrom = getCloseIndex(x, range.from);
    let indexTo = getCloseIndex(x, range.to);
    if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];
    for (let i = indexFrom; i < indexTo; i++) {
      controlPoints[i] = 1;
    }
  });
  return {
    weights: 'controlPoints' in options || zones.length > 0 ? xMultiply(weights, controlPoints) : weights,
    controlPoints
  };
}

/**
 * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
 * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/main/airPLS_manuscript.pdf)
 * @param {Array<number>} x - x axis data useful when control points or zones are submitted
 * @param {Array<number>} y - Original data
 * @param {object} [options={}] - Options object
 * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
 * @param {number} [options.tolerance = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
 * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
 * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
 * @param {Array<number>} [options.controlPoints = []] - Array of 0|1 to force the baseline cross those points.
 * @param {Array<number>} [options.zones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
 * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
 */

function airPLS(x, y, options = {}) {
  const {
    weights,
    controlPoints
  } = getControlPoints(x, y, options);
  let {
    maxIterations = 100,
    lambda = 10,
    tolerance = 0.001
  } = options;
  let baseline, iteration;
  let sumNegDifferences = Number.MAX_SAFE_INTEGER;
  const corrected = Float64Array.from(y);
  let stopCriterion = getStopCriterion(y, tolerance);
  const {
    length
  } = y;
  let {
    lowerTriangularNonZeros,
    permutationEncodedArray
  } = getDeltaMatrix(length, lambda);
  let threshold = 1;
  const l = length - 1;
  let prevNegSum = Number.MAX_SAFE_INTEGER;
  for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
    let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
    let cho = prepare(leftHandSide, length, permutationEncodedArray);
    baseline = cho(rightHandSide);
    sumNegDifferences = applyCorrection(y, baseline, corrected);
    if (iteration === 1) {
      const {
        positive
      } = xNoiseSanPlot(corrected);
      threshold = positive;
    } else {
      const absChange = Math.abs(prevNegSum / sumNegDifferences);
      if (absChange < 1.01 && absChange > 0.99) {
        break;
      }
    }
    prevNegSum = sumNegDifferences + 0;
    for (let i = 1; i < l; i++) {
      const diff = corrected[i];
      if (controlPoints[i] < 1 && Math.abs(diff) > threshold) {
        weights[i] = 0;
      } else {
        const factor = diff > 0 ? -1 : 1;
        weights[i] = Math.exp(factor * (iteration * diff) / Math.abs(sumNegDifferences));
      }
    }
    weights[0] = 1;
    weights[l] = 1;
  }
  return {
    corrected,
    baseline,
    iteration,
    error: sumNegDifferences
  };
  function applyCorrection(y, baseline, corrected) {
    let sumNegDifferences = 0;
    for (let i = 0; i < y.length; i++) {
      let diff = y[i] - baseline[i];
      if (diff < 0) sumNegDifferences += diff;
      corrected[i] = diff;
    }
    return sumNegDifferences;
  }
}
function getStopCriterion(y, tolerance) {
  let sum = xAbsoluteSum(y);
  return tolerance * sum;
}

function _typeof(obj) {
  "@babel/helpers - typeof";

  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof = function (obj) {
      return typeof obj;
    };
  } else {
    _typeof = function (obj) {
      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
    };
  }
  return _typeof(obj);
}

/**
 * Fill an array with sequential numbers
 * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
 * @param {object} [options={}]
 * @param {number} [options.from=0] - first value in the array
 * @param {number} [options.to=10] - last value in the array
 * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
 * @param {number} [options.step] - if not provided calculated from size
 * @return {Array<number>}
 */

function sequentialFill() {
  var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  if (_typeof(input) === 'object' && !isAnyArray$1(input)) {
    options = input;
    input = [];
  }
  if (!isAnyArray$1(input)) {
    throw new TypeError('input must be an array');
  }
  var _options = options,
    _options$from = _options.from,
    from = _options$from === void 0 ? 0 : _options$from,
    _options$to = _options.to,
    to = _options$to === void 0 ? 10 : _options$to,
    _options$size = _options.size,
    size = _options$size === void 0 ? input.length : _options$size,
    step = _options.step;
  if (size !== 0 && step) {
    throw new Error('step is defined by the array size');
  }
  if (!size) {
    if (step) {
      size = Math.floor((to - from) / step) + 1;
    } else {
      size = to - from + 1;
    }
  }
  if (!step && size) {
    step = (to - from) / (size - 1);
  }
  if (Array.isArray(input)) {
    // only works with normal array
    input.length = 0;
    for (var i = 0; i < size; i++) {
      input.push(from);
      from += step;
    }
  } else {
    if (input.length !== size) {
      throw new Error('sequentialFill typed array must have the correct length');
    }
    for (var _i = 0; _i < size; _i++) {
      input[_i] = from;
      from += step;
    }
  }
  return input;
}

/**
 * Adaptive iteratively reweighted penalized least squares [1]
 *
 * This function calls ml-airpls
 *
 * References:
 * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
 * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
 * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
 * @export
 * @param {Array<number>} ys
 * @param {object} [options] - Options object
 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
 * @param {object} [options.regression] - Options for the regression
 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
 * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
 * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
 * @returns {BaselineOutput}
 */
function airPLSBaseline$1(ys, options = {}) {
  const numberPoints = ys.length;
  let {
    x,
    regressionOptions
  } = options;
  if (!x) {
    x = sequentialFill({
      from: 0,
      to: numberPoints - 1,
      size: numberPoints
    });
  }
  let output = airPLS(x, ys, regressionOptions);
  return {
    baseline: output.baseline,
    correctedSpectrum: output.corrected
  };
}

/**
 * Check that x and y are arrays with the same length.
 * @param x - first array
 * @param y - second array
 * @throws if x or y are not the same length, or if they are not arrays
 */
function checkArrayLength(x, y) {
  if (!isAnyArray$1(x) || !isAnyArray$1(y)) {
    throw new TypeError('x and y must be arrays');
  }
  if (x.length !== y.length) {
    throw new RangeError('x and y arrays must have the same length');
  }
}

class BaseRegression {
  constructor() {
    if (new.target === BaseRegression) {
      throw new Error('BaseRegression must be subclassed');
    }
  }
  predict(x) {
    if (typeof x === 'number') {
      return this._predict(x);
    } else if (isAnyArray$1(x)) {
      const y = [];
      for (const xVal of x) {
        y.push(this._predict(xVal));
      }
      return y;
    } else {
      throw new TypeError('x must be a number or array');
    }
  }
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  _predict(x) {
    throw new Error('_predict must be implemented');
  }
  train() {
    // Do nothing for this package
  }
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  toString(precision) {
    return '';
  }
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  toLaTeX(precision) {
    return '';
  }
  /**
   * Return the correlation coefficient of determination (r) and chi-square.
   * @param x - explanatory variable
   * @param y - response variable
   * @return - Object with further statistics.
   */
  score(x, y) {
    checkArrayLength(x, y);
    const n = x.length;
    const y2 = new Array(n);
    for (let i = 0; i < n; i++) {
      y2[i] = this._predict(x[i]);
    }
    let xSum = 0;
    let ySum = 0;
    let chi2 = 0;
    let rmsd = 0;
    let xSquared = 0;
    let ySquared = 0;
    let xY = 0;
    for (let i = 0; i < n; i++) {
      xSum += y2[i];
      ySum += y[i];
      xSquared += y2[i] * y2[i];
      ySquared += y[i] * y[i];
      xY += y2[i] * y[i];
      if (y[i] !== 0) {
        chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
      }
      rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
    }
    const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
    return {
      r,
      r2: r * r,
      chi2,
      rmsd: Math.sqrt(rmsd / n)
    };
  }
}

/**
 * Cast `number` to string. Optionally `digits` specifies significant figures.
 * @param number
 * @param figures
 * @returns - A string representation of `number`.
 */
function maybeToPrecision(number, figures) {
  if (number < 0) {
    number = 0 - number;
    if (typeof figures === 'number') {
      return `- ${number.toPrecision(figures)}`;
    } else {
      return `- ${number.toString()}`;
    }
  } else if (typeof figures === 'number') {
    return number.toPrecision(figures);
  } else {
    return number.toString();
  }
}

class PolynomialRegression extends BaseRegression {
  /**
   * @param x - independent or explanatory variable
   * @param y - dependent or response variable
   * @param degree - degree of the polynomial regression, or array of powers to be used. When degree is an array, intercept at zero is forced to false/ignored.
   * @example `new PolynomialRegression(x, y, 2)`, in this case, you can pass the option `interceptAtZero`, if you need it.
   * @param options
   * @example `new PolynomialRegression(x, y, [1, 3, 5])`
   * Each of the degrees corresponds to a column, so if you have them switched, just do:
   * @example `new PolynomialRegression(x, y, [3, 1, 5])`
   * @param options.interceptAtZero - force the polynomial regression so that f(0) = 0
   */
  constructor(x, y, degree, options = {}) {
    super();
    // @ts-expect-error internal use only
    if (x === true) {
      // @ts-expect-error internal use only
      this.degree = y.degree;
      // @ts-expect-error internal use only
      this.powers = y.powers;
      // @ts-expect-error internal use only
      this.coefficients = y.coefficients;
    } else {
      checkArrayLength(x, y);
      const result = regress(x, y, degree, options);
      this.degree = result.degree;
      this.powers = result.powers;
      this.coefficients = result.coefficients;
    }
  }
  _predict(x) {
    let y = 0;
    for (let k = 0; k < this.powers.length; k++) {
      y += this.coefficients[k] * x ** this.powers[k];
    }
    return y;
  }
  toJSON() {
    return {
      name: 'polynomialRegression',
      degree: this.degree,
      powers: this.powers,
      coefficients: this.coefficients
    };
  }
  toString(precision) {
    return this._toFormula(precision, false);
  }
  toLaTeX(precision) {
    return this._toFormula(precision, true);
  }
  _toFormula(precision, isLaTeX) {
    let sup = '^';
    let closeSup = '';
    let times = ' * ';
    if (isLaTeX) {
      sup = '^{';
      closeSup = '}';
      times = '';
    }
    let fn = '';
    let str = '';
    for (let k = 0; k < this.coefficients.length; k++) {
      str = '';
      if (this.coefficients[k] !== 0) {
        if (this.powers[k] === 0) {
          str = maybeToPrecision(this.coefficients[k], precision);
        } else if (this.powers[k] === 1) {
          str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
        } else {
          str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
        }
        if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
          str = ` + ${str}`;
        } else if (k !== this.coefficients.length - 1) {
          str = ` ${str}`;
        }
      }
      fn = str + fn;
    }
    if (fn.startsWith('+')) {
      fn = fn.slice(1);
    }
    return `f(x) = ${fn}`;
  }
  static load(json) {
    if (json.name !== 'polynomialRegression') {
      throw new TypeError('not a polynomial regression model');
    }
    // @ts-expect-error internal use only
    return new PolynomialRegression(true, json);
  }
}
/**
 * Perform a polynomial regression on the given data set.
 * This is an internal function.
 * @param x - independent or explanatory variable
 * @param y - dependent or response variable
 * @param degree - degree of the polynomial regression
 * @param options.interceptAtZero - force the polynomial regression so that $f(0) = 0$
 * @param options
 */
function regress(x, y, degree, options = {}) {
  const n = x.length;
  let {
    interceptAtZero = false
  } = options;
  let powers = [];
  if (Array.isArray(degree)) {
    powers = degree;
    interceptAtZero = false; //must be false in this case
  } else if (typeof degree === 'number') {
    if (interceptAtZero) {
      powers = new Array(degree);
      for (let k = 0; k < degree; k++) {
        powers[k] = k + 1;
      }
    } else {
      powers = new Array(degree + 1);
      for (let k = 0; k <= degree; k++) {
        powers[k] = k;
      }
    }
  }
  const nCoefficients = powers.length; //1 per power, in any case.
  const F = new Matrix(n, nCoefficients);
  const Y = new Matrix([y]);
  for (let k = 0; k < nCoefficients; k++) {
    for (let i = 0; i < n; i++) {
      if (powers[k] === 0) {
        F.set(i, k, 1);
      } else {
        F.set(i, k, x[i] ** powers[k]);
      }
    }
  }
  const FT = new MatrixTransposeView(F);
  const A = FT.mmul(F);
  const B = FT.mmul(new MatrixTransposeView(Y));
  return {
    coefficients: solve(A, B).to1DArray(),
    degree: Math.max(...powers),
    powers
  };
}

/**
 * Iterative regression-based baseline correction
 * @param {Array<number>} x - Independent axis variable
 * @param {Array<number>} y - Dependent axis variable
 * @param {object} [options] - Options object
 * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
 * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
 * @param {*} [options.regressionOptions] - Options for regressionFunction
 * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
 * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
 */
function baselineCorrectionRegression(x, y, options = {}) {
  let {
    maxIterations = 100,
    Regression = PolynomialRegression,
    regressionOptions,
    tolerance = 0.001
  } = options;
  if (!regressionOptions && Regression === PolynomialRegression) {
    regressionOptions = 3;
  }
  let baseline = y.slice();
  let fitting = y.slice();
  let oldFitting = y;
  let iteration = 0;
  let delta;
  let regression;
  while (iteration < maxIterations) {
    // Calculate the fitting result
    regression = new Regression(x, baseline, regressionOptions);
    delta = 0;
    for (let i = 0; i < baseline.length; i++) {
      fitting[i] = regression.predict(x[i]);
      if (baseline[i] > fitting[i]) {
        baseline[i] = fitting[i];
      }
      delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
    }

    // Stop criterion
    if (delta < tolerance) {
      break;
    } else {
      oldFitting = fitting.slice();
      iteration++;
    }
  }

  // removes baseline
  let corrected = new Array(baseline.length);
  for (let j = 0; j < baseline.length; j++) {
    corrected[j] = y[j] - baseline[j];
  }
  return {
    corrected,
    delta,
    iteration,
    baseline,
    regression
  };
}

/**
 * Iterative polynomial fitting [1]
 *
 * Implementation based on ml-baseline-correction-regression
 *
 * References:
 * [1] Gan, F.; Ruan, G.; Mo, J.
 * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
 *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
 * https://doi.org/10.1016/j.chemolab.2005.08.009.
 * @export
 * @param {Array<number>} ys
 * @param {object} [options] - Options object
 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
 * @param {Object} [options.regression]
 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
 * @param {Object} [options.regression]
 * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
 * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
 * @returns {BaselineOutput}
 */
function iterativePolynomialBaseline$1(ys, options = {}) {
  const numberPoints = ys.length;
  let {
    x,
    regressionOptions
  } = options;
  if (!x) {
    x = sequentialFill({
      from: 0,
      to: numberPoints - 1,
      size: numberPoints
    });
  }
  let output = baselineCorrectionRegression(x, ys, regressionOptions);
  return {
    baseline: output.baseline,
    correctedSpectrum: output.corrected
  };
}

/**

 *
 * @export
 * @param {Array<number>} ys
 * @param {Object} [options={}]
 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
 * @returns {BaselineOutput}
 */
function rollingAverageBaseline$1(ys, options = {}) {
  let window = Math.max(Math.round(ys.length * 0.1), 2);
  let defaults = {
    window,
    padding: {
      size: window - 1,
      algorithm: 'duplicate',
      value: 0
    }
  };
  let actualOptions = {
    ...defaults,
    ...options
  };
  let baseline = xRollingAverage(ys, actualOptions);
  let corrected = new Float64Array(ys.length);
  for (let i = 0; i < corrected.length; i++) {
    corrected[i] = ys[i] - baseline[i];
  }
  return {
    baseline,
    correctedSpectrum: corrected
  };
}

function rollingBall(spectrum, options = {}) {
  if (!isAnyArray$1(spectrum)) {
    throw new Error('Spectrum must be an array');
  }
  if (spectrum.length === 0) {
    throw new TypeError('Spectrum must not be empty');
  }
  const numberPoints = spectrum.length;
  const maxima = new Float64Array(numberPoints);
  const minima = new Float64Array(numberPoints);
  const baseline = new Float64Array(numberPoints);
  // windowM 4 percent of spectrum length
  // windowS 8 percent of spectrum length
  const {
    windowM = Math.round(numberPoints * 0.04),
    windowS = Math.round(numberPoints * 0.08)
  } = options;
  // fi(1) in original paper
  for (let i = 0; i < spectrum.length; i++) {
    const windowLeft = Math.max(0, i - windowM);
    const windowRight = Math.min(i + windowM + 1, spectrum.length);
    minima[i] = xMinValue(spectrum, {
      fromIndex: windowLeft,
      toIndex: windowRight
    });
  }
  // fi in original paper
  for (let i = 0; i < minima.length; i++) {
    const windowLeft = Math.max(0, i - windowM);
    const windowRight = Math.min(i + windowM + 1, minima.length);
    maxima[i] = xMaxValue(minima, {
      fromIndex: windowLeft,
      toIndex: windowRight
    });
  }
  for (let i = 0; i < minima.length; i++) {
    const windowLeft = Math.max(0, i - windowS);
    const windowRight = Math.min(i + windowS + 1, maxima.length);
    baseline[i] = xMean(maxima.subarray(windowLeft, windowRight));
  }
  return baseline;
}

/**
 * Rolling ball baseline correction algorithm.
 * From the abstract of (1):
 * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
 * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
 *
 * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
 * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
 *
 * Algorithm described in (1), but in the implementation here the window width does not change.
 *
 * Reference:
 * (1) Kneen, M. A.; Annegarn, H. J.
 *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
 *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
 *     https://doi.org/10.1016/0168-583X(95)00908-6.
 * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
 *     https://cran.r-project.org/web/packages/baseline/index.html
 *
 * @export
 * @param {Array<number>} ys
 * @param {Object} [options={}]
 * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
 * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
 * @returns {BaselineOutput}
 */
function rollingBallBaseline$1(ys, options = {}) {
  const baseline = rollingBall(ys, options);
  let corrected = new Float64Array(ys.length);
  for (let i = 0; i < corrected.length; i++) {
    corrected[i] = ys[i] - baseline[i];
  }
  return {
    baseline,
    correctedSpectrum: corrected
  };
}

/**

 *
 * @export
 * @param {Array<number>} ys
 * @param {Object} [options={}]
 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
 * @returns {BaselineOutput}
 */
function rollingMedianBaseline$1(ys, options = {}) {
  let window = Math.max(Math.round(ys.length * 0.1), 2);
  let defaults = {
    window,
    padding: {
      size: window - 1,
      algorithm: 'duplicate',
      value: 0
    }
  };
  let actualOptions = {
    ...defaults,
    ...options
  };
  let baseline = xRollingMedian(ys, actualOptions);
  let corrected = new Float64Array(ys.length);
  for (let i = 0; i < corrected.length; i++) {
    corrected[i] = ys[i] - baseline[i];
  }
  return {
    baseline,
    correctedSpectrum: corrected
  };
}

//@ts-expect-error no type definition for baselines
/**
 * @param data
 */
function airPLSBaseline(data) {
  data.y = airPLSBaseline$1(data.y).correctedSpectrum;
  return {
    data
  };
}

//@ts-expect-error no type definition for baselines
/**
 * @param data
 */
function iterativePolynomialBaseline(data) {
  data.y = iterativePolynomialBaseline$1(data.y).correctedSpectrum;
  return {
    data
  };
}

//@ts-expect-error no type definition for baselines
/**
 * @param data
 */
function rollingAverageBaseline(data) {
  data.y = rollingAverageBaseline$1(data.y).correctedSpectrum;
  return {
    data
  };
}

//@ts-expect-error no type definition for baselines
/**
 * @param data
 */
function rollingBallBaseline(data) {
  data.y = rollingBallBaseline$1(data.y).correctedSpectrum;
  return {
    data
  };
}

//@ts-expect-error no type definition for baselines
/**
 * @param data
 */
function rollingMedianBaseline(data) {
  data.y = rollingMedianBaseline$1(data.y).correctedSpectrum;
  return {
    data
  };
}

/**
 * Apply Savitzky Golay algorithm
 * @param [ys] Array of y values
 * @param [xs] Array of X or deltaX
 * @return  Array containing the new ys (same length)
 */
function sgg(ys, xs, options = {}) {
  const {
    windowSize = 9,
    derivative = 0,
    polynomial = 3
  } = options;
  if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
    throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
  }
  if (!isAnyArray$1(ys)) {
    throw new TypeError('Y values must be an array');
  }
  if (typeof xs === 'undefined') {
    throw new TypeError('X must be defined');
  }
  if (windowSize > ys.length) {
    throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
  }
  if (derivative < 0 || !Number.isInteger(derivative)) {
    throw new RangeError('Derivative should be a positive integer');
  }
  if (polynomial < 1 || !Number.isInteger(polynomial)) {
    throw new RangeError('Polynomial should be a positive integer');
  }
  if (polynomial >= 6) {
    // eslint-disable-next-line no-console
    console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
  }
  const half = Math.floor(windowSize / 2);
  const np = ys.length;
  const ans = new Float64Array(np);
  const weights = fullWeights(windowSize, polynomial, derivative);
  let hs = 0;
  let constantH = true;
  if (isAnyArray$1(xs)) {
    constantH = false;
  } else {
    hs = xs ** derivative;
  }
  //For the borders
  for (let i = 0; i < half; i++) {
    const wg1 = weights[half - i - 1];
    const wg2 = weights[half + i + 1];
    let d1 = 0;
    let d2 = 0;
    for (let l = 0; l < windowSize; l++) {
      d1 += wg1[l] * ys[l];
      d2 += wg2[l] * ys[np - windowSize + l];
    }
    if (constantH) {
      ans[half - i - 1] = d1 / hs;
      ans[np - half + i] = d2 / hs;
    } else {
      hs = getHs(xs, half - i - 1, half, derivative);
      ans[half - i - 1] = d1 / hs;
      hs = getHs(xs, np - half + i, half, derivative);
      ans[np - half + i] = d2 / hs;
    }
  }
  //For the internal points
  const wg = weights[half];
  for (let i = windowSize; i <= np; i++) {
    let d = 0;
    for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];
    if (!constantH) {
      hs = getHs(xs, i - half - 1, half, derivative);
    }
    ans[i - half - 1] = d / hs;
  }
  return ans;
}
function getHs(h, center, half, derivative) {
  let hs = 0;
  let count = 0;
  for (let i = center - half; i < center + half; i++) {
    if (i >= 0 && i < h.length - 1) {
      hs += h[i + 1] - h[i];
      count++;
    }
  }
  return (hs / count) ** derivative;
}
function gramPoly(i, m, k, s) {
  let Grampoly = 0;
  if (k > 0) {
    Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
  } else if (k === 0 && s === 0) {
    Grampoly = 1;
  } else {
    Grampoly = 0;
  }
  return Grampoly;
}
function genFact(a, b) {
  let gf = 1;
  if (a >= b) {
    for (let j = a - b + 1; j <= a; j++) {
      gf *= j;
    }
  }
  return gf;
}
function weight(i, t, m, n, s) {
  let sum = 0;
  for (let k = 0; k <= n; k++) {
    sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
  }
  return sum;
}
/**
 * @private
 * @param m  Number of points
 * @param n  Polynomial grade
 * @param s  Derivative
 */
function fullWeights(m, n, s) {
  const weights = new Array(m);
  const np = Math.floor(m / 2);
  for (let t = -np; t <= np; t++) {
    weights[t + np] = new Float64Array(m);
    for (let j = -np; j <= np; j++) {
      weights[t + np][j + np] = weight(j, t, np, n, s);
    }
  }
  return weights;
}

/**
 * Calculate the first derivative using Savitzky–Golay filter.
 * @param data
 * @param options
 */
function firstDerivative$1(data, options = {}) {
  const {
    x,
    y
  } = data;
  return {
    data: {
      x,
      y: sgg(y, x, {
        ...options,
        derivative: 1
      })
    }
  };
}

/**
 * Calculate the second derivative using Savitzky–Golay filter.
 * @param data
 * @param options
 */
function secondDerivative$1(data, options = {}) {
  const {
    x,
    y
  } = data;
  return {
    data: {
      x,
      y: sgg(y, x, {
        ...options,
        derivative: 2
      })
    }
  };
}

/**
 * Calculate the third derivative using Savitzky–Golay filter.
 * @param data
 * @param options
 */
function thirdDerivative(data, options = {}) {
  const {
    x,
    y
  } = data;
  return {
    data: {
      x,
      y: sgg(y, x, {
        ...options,
        derivative: 3
      })
    }
  };
}

/**
 * Apply the Savitzky Golay Generalized Filter
 * @param data
 * @param options
 */
function savitzkyGolay(data, options = {}) {
  const {
    x,
    y
  } = data;
  return {
    data: {
      x,
      y: sgg(y, x, options)
    }
  };
}

/**
 * Ensure X values are strictly monotonic increasing
 * http://www-groups.mcs.st-andrews.ac.uk/~john/analysis/Lectures/L8.html
 * @param data
 */
function ensureGrowing(data) {
  return {
    data: xyEnsureGrowingX(data)
  };
}

/**
 * Ensure X values are strictly monotonic increasing
 * http://www-groups.mcs.st-andrews.ac.uk/~john/analysis/Lectures/L8.html
 * @param data
 */
function reverseIfNeeded(data) {
  return {
    data: xyGrowingX(data)
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function equallySpaced(data, options = {}) {
  return {
    data: xyEquallySpaced(data, options)
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function filterX(data, options = {}) {
  return {
    data: xyFilterX(data, options)
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function setMaxX(data, options = {}) {
  const {
    max = 1
  } = options;
  const existingMax = xMaxValue(data.x);
  if (existingMax === max) {
    return {
      data
    };
  }
  return {
    data: {
      x: xAdd(data.x, max - existingMax),
      y: data.y
    }
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function setMinX(data, options = {}) {
  const {
    min = 0
  } = options;
  const existingMin = xMinValue(data.x);
  if (existingMin === min) {
    return {
      data
    };
  }
  return {
    data: {
      x: xAdd(data.x, min - existingMin),
      y: data.y
    }
  };
}

function getMinMaxIntervalsDy(y, x, dY, dX) {
  let lastMax = null;
  let lastMin = null;
  const intervalL = [];
  const intervalR = [];
  for (let i = 1; i < y.length - 1; ++i) {
    if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
      lastMin = {
        x: x[i],
        index: i
      };
      if (dX > 0 && lastMax !== null) {
        intervalL.push(lastMax);
        intervalR.push(lastMin);
      }
    }
    // Maximum in first derivative
    if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
      lastMax = {
        x: x[i],
        index: i
      };
      if (dX < 0 && lastMin !== null) {
        intervalL.push(lastMax);
        intervalR.push(lastMin);
      }
    }
  }
  return {
    intervalL,
    intervalR
  };
}

function tryMatchOneIntervalWithMinData(options) {
  const {
    x,
    lastK,
    minData,
    yThreshold,
    intervalWidth,
    intervalCenter,
    yData
  } = options;
  let minDistance = Number.POSITIVE_INFINITY;
  let possible = -1;
  let newLastIndex = lastK;
  for (let k = newLastIndex + 1; k < minData.length; k++) {
    const centerIndex = minData[k];
    if (yData[centerIndex] <= yThreshold) {
      continue;
    }
    const deltaX = x[centerIndex];
    const currentDistance = Math.abs(deltaX - intervalCenter);
    if (currentDistance < intervalWidth) {
      if (currentDistance < minDistance) {
        possible = k;
      }
      newLastIndex = k;
    }
    if (currentDistance >= minDistance) break;
    minDistance = currentDistance;
  }
  return {
    lastIndex: newLastIndex,
    possible
  };
}

function autoAlgorithm(input) {
  const {
    x,
    y,
    yData,
    dY,
    ddY,
    dX,
    yThreshold
  } = input;
  const minddY = [];
  const crossDy = [];
  const {
    intervalL,
    intervalR
  } = getMinMaxIntervalsDy(y, x, dY, dX);
  for (let i = 1; i < y.length - 1; ++i) {
    if (dY[i] < 0 && dY[i + 1] > 0 || dY[i] > 0 && dY[i + 1] < 0) {
      // push the index of the element closer to zero
      crossDy.push(Math.abs(dY[i]) < Math.abs(dY[i + 1]) ? i : i + 1);
    }
    // Handle exact zero
    if (dY[i] === 0 && dY[i] < Math.abs(dY[i + 1]) && dY[i] < Math.abs(dY[i - 1])) {
      crossDy.push(i);
    }
    // Minimum in second derivative
    if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
      minddY.push(i);
    }
  }
  const peaks = [];
  let [lastK, lastJ] = [-1, -1];
  for (let i = 0; i < intervalL.length; i++) {
    const intervalWidth = (intervalR[i].x - intervalL[i].x) / 2;
    const intervalCenter = (intervalR[i].x + intervalL[i].x) / 2;
    let yIndex = -1;
    let match = tryMatchOneIntervalWithMinData({
      x,
      yData,
      lastK,
      yThreshold,
      intervalWidth,
      intervalCenter,
      minData: crossDy
    });
    lastK = match.lastIndex;
    if (match.possible !== -1) {
      yIndex = crossDy[match.possible];
    } else {
      match = tryMatchOneIntervalWithMinData({
        x,
        yData,
        yThreshold,
        lastK: lastJ,
        intervalWidth,
        intervalCenter,
        minData: minddY
      });
      if (match.possible !== -1) {
        yIndex = minddY[match.possible];
      }
      lastJ = match.lastIndex;
    }
    if (yIndex !== -1) {
      const width = Math.abs(intervalR[i].x - intervalL[i].x);
      peaks.push({
        id: crypto.randomUUID(),
        x: x[yIndex],
        y: y[yIndex],
        width,
        index: yIndex,
        ddY: ddY[yIndex],
        inflectionPoints: {
          from: intervalL[i],
          to: intervalR[i]
        }
      });
    }
  }
  return peaks;
}

function getPeakFromIntervals(options) {
  let lastK = -1;
  const peaks = [];
  const {
    x,
    ddY,
    yData,
    yThreshold,
    intervalR,
    intervalL,
    minData
  } = options;
  for (let i = 0; i < intervalL.length; i++) {
    const intervalWidth = (intervalR[i].x - intervalL[i].x) / 2;
    const intervalCenter = (intervalR[i].x + intervalL[i].x) / 2;
    const {
      possible = -1,
      lastIndex
    } = tryMatchOneIntervalWithMinData({
      x,
      lastK,
      minData,
      yThreshold,
      intervalWidth,
      intervalCenter,
      yData
    });
    if (possible !== -1) {
      const centerIndex = minData[possible];
      const width = Math.abs(intervalR[i].x - intervalL[i].x);
      peaks.push({
        id: crypto.randomUUID(),
        x: x[centerIndex],
        y: yData[centerIndex],
        width,
        index: centerIndex,
        ddY: ddY[centerIndex],
        inflectionPoints: {
          from: intervalL[i],
          to: intervalR[i]
        }
      });
    }
    lastK = lastIndex;
  }
  return peaks;
}

/**
 * Finds the indices where the first derivative crosses zero (sign change),
 * which are potential peak positions. This function does not detect zero-crossings
 * in regions with consecutive zero values in the derivative (flat regions).
 *
 * @param input - Object containing the y values and their first derivative (dY).
 * @returns Array of indices where the first derivative crosses zero (excluding consecutive zeros).
 */
function xGetCrossZeroPoints(input) {
  const {
    y,
    dY
  } = input;
  const crossDy = [];
  for (let i = 1; i < y.length - 1; ++i) {
    if (isLessAndGreaterThanZero(dY[i], dY[i + 1])) {
      // push the index of the element closer to zero
      crossDy.push(Math.abs(dY[i]) < Math.abs(dY[i + 1]) ? i : i + 1);
    } else if (
    // Handle exact zero
    dY[i] === 0 && isLessAndGreaterThanZero(dY[i - 1], dY[i + 1])) {
      crossDy.push(i);
    }
  }
  return crossDy;
}
function isLessAndGreaterThanZero(back, next) {
  return back < 0 && next > 0 || back > 0 && next < 0;
}

function firstDerivative(input) {
  const {
    y,
    x,
    dY,
    dX,
    yData,
    yThreshold,
    ddY
  } = input;
  const crossDy = xGetCrossZeroPoints(input);
  const {
    intervalL,
    intervalR
  } = getMinMaxIntervalsDy(y, x, dY, dX);
  return getPeakFromIntervals({
    minData: crossDy,
    intervalL,
    intervalR,
    x,
    yData,
    yThreshold,
    ddY
  });
}

function secondDerivative(input) {
  const {
    x,
    y,
    yData,
    dY,
    ddY,
    dX,
    yThreshold
  } = input;
  const minddY = [];
  const {
    intervalL,
    intervalR
  } = getMinMaxIntervalsDy(y, x, dY, dX);
  // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum
  for (let i = 1; i < y.length - 1; ++i) {
    // Minimum in second derivative
    if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
      minddY.push(i);
    }
  }
  return getPeakFromIntervals({
    minData: minddY,
    intervalL,
    intervalR,
    x,
    yData,
    yThreshold,
    ddY
  });
}

/**
 * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
 * This process is done in place and is very fast.
 * @param data
 * @param peaks
 */
function optimizeTop(data, peaks) {
  const {
    x,
    y
  } = data;
  for (const peak of peaks) {
    let currentIndex = peak.index;
    // The detected peak could be moved 1 or 2 units to left or right.
    if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
      currentIndex--;
    } else if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
      currentIndex++;
    } else if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
      currentIndex -= 2;
    } else if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
      currentIndex += 2;
    }
    // interpolation to a sin() function
    if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
      const alpha = 20 * Math.log10(y[currentIndex - 1]);
      const beta = 20 * Math.log10(y[currentIndex]);
      const gamma = 20 * Math.log10(y[currentIndex + 1]);
      const p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
      const xCurrent = x[currentIndex];
      const xPrevious = x[currentIndex - 1];
      peak.x = xCurrent + (xCurrent - xPrevious) * p;
      peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
    }
  }
}

/**
 * Global spectra deconvolution
 * @param  data - Object data with x and y arrays. Values in x has to be growing
 * @param options
 * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
 * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.
 */
function gsd(data, options = {}) {
  let {
    noiseLevel
  } = options;
  const {
    sgOptions = {
      windowSize: 9,
      polynomial: 3
    },
    smoothY = false,
    maxCriteria = true,
    minMaxRatio = 0.00025,
    realTopDetection = false,
    peakDetectionAlgorithm = 'second'
  } = options;
  const {
    x
  } = data;
  let {
    y
  } = data;
  if (xIsMonotonic(x) !== 1) {
    throw new Error('GSD only accepts monotone increasing x values');
  }
  //rescale;
  y = y.slice();
  // If the max difference between delta x is less than 5%, then,
  // we can assume it to be equally spaced variable
  const isEquallySpaced = xIsEquallySpaced(x);
  if (noiseLevel === undefined) {
    if (isEquallySpaced) {
      const noiseInfo = xNoiseStandardDeviation(y);
      if (maxCriteria) {
        noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
      } else {
        noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
      }
    } else {
      noiseLevel = 0;
    }
  } else if (!maxCriteria) {
    noiseLevel *= -1;
  }
  if (!maxCriteria) {
    for (let i = 0; i < y.length; i++) {
      y[i] *= -1;
    }
  }
  if (noiseLevel !== undefined) {
    for (let i = 0; i < y.length; i++) {
      if (y[i] < noiseLevel) {
        y[i] = noiseLevel;
      }
    }
  }
  const xValue = isEquallySpaced ? x[1] - x[0] : x;
  const yData = smoothY ? sgg(y, xValue, {
    ...sgOptions,
    derivative: 0
  }) : y;
  const {
    min: minY,
    max: maxY
  } = xMinMaxValues(yData);
  if (minY > maxY || minY === maxY) return [];
  const dY = sgg(y, xValue, {
    ...sgOptions,
    derivative: 1
  });
  const ddY = sgg(y, xValue, {
    ...sgOptions,
    derivative: 2
  });
  const yThreshold = Math.max(noiseLevel, minY + (maxY - minY) * minMaxRatio);
  const dX = x[1] - x[0];
  const peakData = {
    x,
    y,
    yData,
    dY,
    ddY,
    dX,
    yThreshold
  };
  let peaks = [];
  if (peakDetectionAlgorithm === 'first') {
    peaks = firstDerivative(peakData);
  } else if (peakDetectionAlgorithm === 'second') {
    peaks = secondDerivative(peakData);
  } else {
    peaks = autoAlgorithm(peakData);
  }
  if (realTopDetection) {
    optimizeTop({
      x,
      y: yData
    }, peaks);
  }
  peaks.forEach(peak => {
    if (!maxCriteria) {
      peak.y *= -1;
      peak.ddY = peak.ddY * -1;
    }
  });
  peaks.sort((a, b) => {
    return a.x - b.x;
  });
  return peaks;
}

/**
 * Filter that allows to calibrateX the x axis based on the presence of peaks
 * @param data
 * @param options
 */
function calibrateX(data, options = {}) {
  const {
    targetX = 0,
    nbPeaks = 1,
    from = data.x[0],
    to = data.x.at(-1),
    gsd: gsdOptions = {
      minMaxRatio: 0.1,
      realTopDetection: true,
      smoothY: true,
      sgOptions: {
        windowSize: 7,
        polynomial: 3
      }
    }
  } = options;
  const fromIndex = xFindClosestIndex(data.x, from);
  const toIndex = xFindClosestIndex(data.x, to);
  let peaks = gsd({
    x: data.x.subarray(fromIndex, toIndex),
    y: data.y.subarray(fromIndex, toIndex)
  }, gsdOptions);
  peaks.sort((a, b) => b.y - a.y);
  peaks = peaks.slice(0, nbPeaks);
  if (peaks.length < nbPeaks) return {
    data
  };
  const middle = xMean(peaks.map(peak => peak.x));
  return {
    data: {
      x: xAdd(data.x, targetX - middle),
      y: data.y
    }
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function xFunction(data, options = {}) {
  return {
    data: {
      x: xApplyFunctionStr(data.x, {
        variableLabel: 'x',
        fctString: options.function
      }),
      y: data.y
    }
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function yFunction(data, options = {}) {
  return {
    data: {
      x: data.x,
      y: xApplyFunctionStr(data.y, {
        variableLabel: 'y',
        fctString: options.function
      })
    }
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function setMaxY(data, options = {}) {
  const {
    max = 1
  } = options;
  const existingMax = xMaxValue(data.y);
  if (existingMax === max) {
    return {
      data
    };
  }
  return {
    data: {
      x: data.x,
      y: xAdd(data.y, max - existingMax)
    }
  };
}

/**
 * Filter that allows to
 * @param data
 * @param options
 */
function setMinY(data, options = {}) {
  const {
    min = 0
  } = options;
  const existingMin = xMinValue(data.y);
  if (existingMin === min) {
    return {
      data
    };
  }
  return {
    data: {
      x: data.x,
      y: xAdd(data.y, min - existingMin)
    }
  };
}

var Filters = /*#__PURE__*/Object.freeze({
    __proto__: null,
    airPLSBaseline: airPLSBaseline,
    calibrateX: calibrateX,
    centerMean: centerMean,
    centerMedian: centerMedian,
    divideBySD: divideBySD,
    ensureGrowing: ensureGrowing,
    equallySpaced: equallySpaced,
    filterX: filterX,
    firstDerivative: firstDerivative$1,
    fromTo: fromTo,
    iterativePolynomialBaseline: iterativePolynomialBaseline,
    normed: normed,
    paretoNormalization: paretoNormalization,
    rescale: rescale,
    reverseIfNeeded: reverseIfNeeded,
    rollingAverageBaseline: rollingAverageBaseline,
    rollingBallBaseline: rollingBallBaseline,
    rollingMedianBaseline: rollingMedianBaseline,
    savitzkyGolay: savitzkyGolay,
    secondDerivative: secondDerivative$1,
    setMaxX: setMaxX,
    setMaxY: setMaxY,
    setMinX: setMinX,
    setMinY: setMinY,
    thirdDerivative: thirdDerivative,
    xFunction: xFunction,
    yFunction: yFunction
});

/**
 * Apply filters on {x:[], y:[]}
 * @param data
 * @param filters
 * @returns
 */
function filterXY(data, filters) {
  let result = {
    data: xyGrowingX({
      x: xEnsureFloat64(data.x),
      y: xEnsureFloat64(data.y)
    })
  };
  const logs = [];
  for (const filter of filters) {
    const start = Date.now();
    if (!filter.name) {
      // we ignore empty filter names
      continue;
    }
    const filterFct = Filters[filter.name];
    if (!filterFct) {
      throw new Error(`Unknown filter: ${filter.name}`);
    }
    // @ts-expect-error some method have options and some other ones don't have any options
    result = filterFct(result.data, filter.options);
    result.data = xyGrowingX(result.data);
    logs.push({
      name: filter.name,
      time: Date.now() - start
    });
  }
  return {
    logs,
    data: result.data
  };
}

/**
 *
 * @private
 * @param input
 * @param [options={}]
 */
function getNormalized(input, options = {}) {
  xyCheck(input);
  let {
    filters = []
  } = options;
  const {
    from = input.x[0],
    to = input.x.at(-1),
    numberOfPoints = 1024,
    applyRangeSelectionFirst = false,
    exclusions = []
  } = options;
  filters = structuredClone(filters);
  const equallySpacedFilter = {
    name: 'equallySpaced',
    options: {
      from,
      to,
      numberOfPoints,
      exclusions
    }
  };
  if (applyRangeSelectionFirst) {
    filters.unshift(equallySpacedFilter);
  } else {
    filters.push(equallySpacedFilter);
  }
  const output = filterXY(input, filters).data;
  const allowedBoundary = {
    x: {
      min: output.x[0],
      max: output.x.at(-1)
    },
    y: xMinMaxValues(output.y)
  };
  return {
    data: output,
    allowedBoundary
  };
}

/**
 * Updates range information for a spectrum
 * @private
 * @param spectrum - The spectrum object to update
 * @param ranges - Array of range objects to process
 */
function updateRangesInfo(spectrum, ranges = []) {
  spectrum.ranges = {};
  for (let range of ranges) {
    range = structuredClone(range);
    spectrum.ranges[range.label] = range;
    const fromToIndex = {
      fromIndex: xFindClosestIndex(spectrum.normalized.x, range.from),
      toIndex: xFindClosestIndex(spectrum.normalized.x, range.to)
    };
    range.deltaX = spectrum.normalized.x[fromToIndex.toIndex] - spectrum.normalized.x[fromToIndex.fromIndex];
    range.baseline = (spectrum.normalized.y[fromToIndex.toIndex] + spectrum.normalized.y[fromToIndex.fromIndex]) * range.deltaX / 2;
    range.integration = xyIntegration(spectrum.normalized, fromToIndex);
    range.correctedIntegration = range.integration - range.baseline;
    range.maxPoint = xyMaxYPoint(spectrum.normalized, fromToIndex);
    range.minPoint = xyMinYPoint(spectrum.normalized, fromToIndex);
    range.x = spectrum.normalized.x.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1) || [];
    range.y = spectrum.normalized.y.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1) || [];
  }
}

/**
 * Class allowing manipulate one IR spectrum
 * @class spectrum
 * @param x - x values array
 * @param y - y values array
 * @param id - spectrum identifier
 * @param options - initialization options
 */
class Spectrum {
  x;
  y;
  id;
  meta;
  normalizedBoundary;
  normalized;
  normalizedAllowedBoundary;
  memory;
  ranges;
  /**
   * Constructor for Spectrum
   * @param x - x values array
   * @param y - y values array
   * @param id - spectrum identifier
   * @param options - initialization options with meta, normalization, and normalized properties
   */
  constructor(x, y, id, options = {}) {
    const {
      meta = {},
      normalization = {},
      normalized
    } = options;
    if (x && x.length > 1 && x[0] > x.at(-1)) {
      this.x = x.toReversed();
      this.y = y.toReversed();
    } else {
      this.x = x || [];
      this.y = y || [];
    }
    this.id = id;
    this.meta = meta;
    this.normalizedBoundary = {
      x: {
        min: 0,
        max: 0
      },
      y: {
        min: 0,
        max: 0
      }
    };
    if (normalized) {
      this.normalized = normalized;
      this.updateNormalizedBoundary();
      this.normalizedAllowedBoundary = this.normalizedBoundary;
    } else {
      this.normalized = this.updateNormalization(normalization);
    }
    this.updateMemory();
  }
  /**
   * Get the minimum x value
   */
  get minX() {
    return this.x[0];
  }
  /**
   * Get the maximum x value
   */
  get maxX() {
    return this.x.at(-1);
  }
  /**
   * Update memory usage statistics
   */
  updateMemory() {
    const original = (this.x?.length || 0) * 16;
    const normalizedMem = (this.normalized?.x?.length || 0) * 16;
    this.memory = {
      original,
      normalized: normalizedMem,
      total: original + normalizedMem
    };
  }
  /**
   * Remove original x and y data
   */
  removeOriginal() {
    this.x = undefined;
    this.y = undefined;
    this.updateMemory();
  }
  /**
   * Get x and y arrays
   * @returns Object containing x and y arrays
   * @throws Error if x or y is not an array
   */
  getXY() {
    if (!Array.isArray(this.x) || !Array.isArray(this.y)) {
      throw new Error('Can not get normalized data');
    }
    return {
      x: this.x,
      y: this.y
    };
  }
  /**
   * Update ranges information
   * @param ranges - ranges configuration
   */
  updateRangesInfo(ranges) {
    updateRangesInfo(this, ranges);
  }
  /**
   * Update normalization
   * @param normalization - normalization configuration
   */
  updateNormalization(normalization) {
    const result = getNormalized(this, normalization);
    this.normalized = result.data;
    this.normalizedAllowedBoundary = result.allowedBoundary;
    this.ranges = {};
    this.updateMemory();
    this.updateNormalizedBoundary();
    return result.data;
  }
  /**
   * Get spectrum data with optional filtering and scaling
   * @param options - options for data retrieval
   * @returns Data object with x and y arrays
   */
  getData(options) {
    return getData(this, options);
  }
  /**
   * Update normalized boundary values
   */
  updateNormalizedBoundary() {
    this.normalizedBoundary.x = {
      min: this.normalized.x[0],
      max: this.normalized.x.at(-1)
    };
    this.normalizedBoundary.y = xMinMaxValues(this.normalized.y);
  }
}

/**
 * Manager for a large number of spectra with the possibility to normalize the data
 * and skip the original data.
 */
class SpectraProcessor {
  normalization;
  maxMemory;
  keepOriginal;
  spectra;
  /**
   * Create a SpectraProcessor
   * @param options - Configuration options
   */
  constructor(options = {}) {
    this.normalization = options.normalization;
    this.maxMemory = options.maxMemory || 256 * 1024 * 1024;
    this.keepOriginal = true;
    this.spectra = [];
  }
  getNormalizationAnnotations() {
    return getNormalizationAnnotations(this.normalization);
  }
  /**
   * Recalculate the normalized data using the stored original data if available
   * This will throw an error if the original data is not present
   * @param normalization - Normalization options
   */
  setNormalization(normalization = {}) {
    if (JSON.stringify(this.normalization) === JSON.stringify(normalization)) {
      return;
    }
    this.normalization = normalization;
    for (const spectrum of this.spectra) {
      spectrum.updateNormalization(this.normalization);
    }
  }
  getNormalization() {
    return this.normalization;
  }
  /**
   * Returns an object {x:[], y:[]} containing the autocorrelation for the
   * specified index
   * @param index - X index of the spectrum to autocorrelate
   * @param options - Options for autocorrelation
   * @returns Autocorrelation result
   */
  getAutocorrelation(index, options = {}) {
    const {
      x
    } = options;
    const normalizedData = this.getNormalizedData(options);
    let actualIndex = index;
    if (actualIndex === undefined && x !== undefined) {
      actualIndex = xFindClosestIndex(normalizedData.x, x);
    } else {
      actualIndex = 0;
    }
    return getAutocorrelation(normalizedData, actualIndex);
  }
  /**
   * Returns a {x:[], y:[]} containing the average of specified spectra
   * @param options - Options for mean calculation
   * @returns Mean data
   */
  getMeanData(options) {
    return getMeanData(this.getNormalizedData(options));
  }
  /**
   * Returns an object containing 4 parameters with the normalized data
   * @param options - Options for normalization
   * @returns
   */
  getNormalizedData(options = {}) {
    const {
      ids
    } = options;
    const spectra = this.getSpectra(ids);
    return getNormalizedData(spectra);
  }
  /**
   * Returns a tab separated value containing the normalized data
   * @param options - Options for export
   * @returns Text string
   */
  getNormalizedText(options = {}) {
    const {
      ids
    } = options;
    const spectra = this.getSpectra(ids);
    return getNormalizedText(spectra, options);
  }
  /**
   * Returns a tab separated value containing the post processed data
   * @param options - Options for export
   * @returns Text string
   */
  getPostProcessedText(options = {}) {
    return getPostProcessedText(this, options);
  }
  getMinMaxX() {
    let min = Number.MAX_VALUE;
    let max = Number.MIN_VALUE;
    for (const spectrum of this.spectra) {
      if (spectrum.minX < min) min = spectrum.minX;
      if (spectrum.maxX > max) max = spectrum.maxX;
    }
    return {
      min,
      max
    };
  }
  /**
   * Returns an object containing 4 parameters with the scaled data
   * @param options - Options for post-processing
   * @returns
   */
  getPostProcessedData(options) {
    return getPostProcessedData(this, options);
  }
  /**
   * Add from text
   * By default TITLE from the jcamp will be in the meta information
   * @param text - Text data
   * @param options - Options for parsing
   */
  addFromText(text, options = {}) {
    if (options.force !== true && options.id && this.contains(options.id)) {
      return;
    }
    const parsed = parseText(text, options);
    const meta = {
      ...options.meta
    };
    this.addFromData(parsed.data, {
      meta,
      id: options.id
    });
  }
  /**
   * Add jcamp
   * By default TITLE from the jcamp will be in the meta information
   * @param jcamp - JCAMP data
   * @param options - Options for parsing
   */
  addFromJcamp(jcamp, options = {}) {
    if (options.force !== true && options.id && this.contains(options.id)) {
      return;
    }
    const parsed = parseJcamp(jcamp);
    const meta = {
      ...parsed.info,
      ...parsed.meta,
      ...options.meta
    };
    this.addFromData(parsed.data, {
      meta,
      id: options.id
    });
  }
  updateRangesInfo(options) {
    for (const spectrum of this.spectra) {
      spectrum.updateRangesInfo(options);
    }
  }
  /**
   * Returns the metadata for a set of spectra
   * @param options - Options for metadata extraction
   * @returns Metadata array
   */
  getMetadata(options = {}) {
    const {
      ids
    } = options;
    return getMetadata(this.getSpectra(ids));
  }
  /**
   * Get classes from metadata
   * @param options - Options for classification
   * @returns Array of class numbers
   */
  getClasses(options) {
    return getClasses(this.getMetadata(options), options);
  }
  /**
   * Get class labels from metadata
   * @param options - Options for label extraction
   * @returns Array of class labels
   */
  getClassLabels(options) {
    return getClassLabels(this.getMetadata(options), options);
  }
  /**
   * Get categories statistics
   * @param options - Options for statistics
   * @returns Category statistics
   */
  getCategoriesStats(options) {
    return getCategoriesStats(this.getMetadata(), options);
  }
  /**
   * Add a spectrum based on the data
   * @param data - {x, y}
   * @param options - Options for adding spectrum
   * @returns Spectrum
   */
  addFromData(data, options = {}) {
    if (this.spectra.length === 0) this.keepOriginal = true;
    const id = options.id || Math.random().toString(36).slice(2, 10);
    let index = this.getSpectrumIndex(id);
    if (index === undefined) index = this.spectra.length;
    const spectrum = new Spectrum(data.x, data.y, id, {
      meta: options.meta,
      normalized: options.normalized,
      normalization: this.normalization
    });
    this.spectra[index] = spectrum;
    if (!this.keepOriginal) {
      spectrum.removeOriginal();
    } else {
      const memoryInfo = this.getMemoryInfo();
      if (memoryInfo.total > this.maxMemory) {
        this.keepOriginal = false;
        this.removeOriginals();
      }
    }
  }
  removeOriginals() {
    for (const spectrum of this.spectra) {
      spectrum.removeOriginal();
    }
  }
  /**
   * Remove the spectrum from the SpectraProcessor for the specified id
   * @param id - Spectrum identifier
   * @returns Removed spectrum array
   */
  removeSpectrum(id) {
    const index = this.getSpectrumIndex(id);
    if (index === undefined) return undefined;
    return this.spectra.splice(index, 1);
  }
  /**
   * Remove all the spectra not present in the list
   * @param ids - Array of ids of the spectra to keep
   */
  removeSpectraNotIn(ids) {
    const currentIDs = this.spectra.map(spectrum => spectrum.id);
    for (const id of currentIDs) {
      if (!ids.includes(id)) {
        this.removeSpectrum(id);
      }
    }
  }
  /**
   * Checks if the ID of a spectrum exists in the SpectraProcessor
   * @param id - Spectrum identifier
   * @returns True if spectrum exists
   */
  contains(id) {
    return this.getSpectrumIndex(id) !== undefined;
  }
  /**
   * Returns the index of the spectrum in the spectra array
   * @param id - Spectrum identifier
   * @returns Index or undefined
   */
  getSpectrumIndex(id) {
    if (!id) return undefined;
    for (let i = 0; i < this.spectra.length; i++) {
      const spectrum = this.spectra[i];
      if (spectrum.id === id) return i;
    }
    return undefined;
  }
  /**
   * Returns an array of all the ids
   * @returns Array of IDs
   */
  getIDs() {
    return this.spectra.map(spectrum => spectrum.id);
  }
  /**
   * Returns an array of spectrum from their ids
   * @param ids - Array of spectrum IDs
   * @returns Array of Spectrum
   */
  getSpectra(ids) {
    if (!ids || !Array.isArray(ids)) return this.spectra;
    const spectra = [];
    for (const id of ids) {
      const index = this.getSpectrumIndex(id);
      if (index !== undefined) {
        spectra.push(this.spectra[index]);
      }
    }
    return spectra;
  }
  /**
   * Returns the spectrum for the given id
   * @param id - Spectrum identifier
   * @returns Spectrum or undefined
   */
  getSpectrum(id) {
    const index = this.getSpectrumIndex(id);
    if (index === undefined) return undefined;
    return this.spectra[index];
  }
  /**
   * Returns a JSGraph chart object for all the spectra
   * @param options - Chart options
   * @returns Chart object
   */
  getChart(options) {
    return getChart(this.spectra, options);
  }
  /**
   * Returns a JSGraph chart object for autocorrelation
   * @param index - Index in spectrum
   * @param options - Chart options
   * @returns Chart object
   */
  getAutocorrelationChart(index, options) {
    return getAutocorrelationChart(this, index, options);
  }
  /**
   * Returns a JSGraph annotation object for the specified index
   * @param index - Index in spectrum
   * @param options - Annotation options
   * @returns Annotation object
   */
  getTrackAnnotation(index, options) {
    return getTrackAnnotation(this.spectra, index, options);
  }
  /**
   * Returns a JSGraph annotation object for box plot
   * @param options - Chart options
   * @returns Chart object
   */
  getBoxPlotChart(options = {}) {
    const normalizedData = this.getNormalizedData(options);
    return getBoxPlotChart(normalizedData, options);
  }
  /**
   * Returns boxplot information
   * @param options - Options for box plot
   * @returns Box plot data
   */
  getBoxPlotData(options = {}) {
    const normalizedData = this.getNormalizedData(options);
    return getBoxPlotData(normalizedData);
  }
  /**
   * Returns a JSGraph chart object for all the normalized spectra
   * @param options - Chart options
   * @returns Chart object
   */
  getNormalizedChart(options = {}) {
    const {
      ids,
      ...chartOptions
    } = options;
    const spectra = this.getSpectra(ids);
    return getNormalizedChart(spectra, chartOptions);
  }
  /**
   * Returns a JSGraph chart object for all the scaled normalized spectra
   * @param options - Options for post-processing
   * @returns Chart object
   */
  getPostProcessedChart(options) {
    return getPostProcessedChart(this, options);
  }
  getMemoryInfo() {
    const memoryInfo = {
      original: 0,
      normalized: 0,
      total: 0
    };
    for (const spectrum of this.spectra) {
      const memory = spectrum.memory;
      if (memory) {
        memoryInfo.original += memory.original;
        memoryInfo.normalized += memory.normalized;
        memoryInfo.total += memory.total;
      }
    }
    return {
      ...memoryInfo,
      keepOriginal: this.keepOriginal,
      maxMemory: this.maxMemory
    };
  }
  getNormalizedBoundary() {
    const boundary = {
      x: {
        min: Number.MAX_VALUE,
        max: Number.MIN_VALUE
      },
      y: {
        min: Number.MAX_VALUE,
        max: Number.MIN_VALUE
      }
    };
    for (const spectrum of this.spectra) {
      if (spectrum.normalizedBoundary.x.min < boundary.x.min) {
        boundary.x.min = spectrum.normalizedBoundary.x.min;
      }
      if (spectrum.normalizedBoundary.x.max > boundary.x.max) {
        boundary.x.max = spectrum.normalizedBoundary.x.max;
      }
      if (spectrum.normalizedBoundary.y.min < boundary.y.min) {
        boundary.y.min = spectrum.normalizedBoundary.y.min;
      }
      if (spectrum.normalizedBoundary.y.max > boundary.y.max) {
        boundary.y.max = spectrum.normalizedBoundary.y.max;
      }
    }
    return boundary;
  }
  /**
   * We provide the allowed from / to after normalization
   * For the X axis we return the smallest common values
   * For the Y axis we return the largest min / max
   */
  getNormalizedCommonBoundary() {
    const boundary = {
      x: {
        min: Number.NEGATIVE_INFINITY,
        max: Number.POSITIVE_INFINITY
      },
      y: {
        min: Number.POSITIVE_INFINITY,
        max: Number.NEGATIVE_INFINITY
      }
    };
    for (const spectrum of this.spectra) {
      if (spectrum.normalizedAllowedBoundary) {
        if (spectrum.normalizedAllowedBoundary.x.min > boundary.x.min) {
          boundary.x.min = spectrum.normalizedAllowedBoundary.x.min;
        }
        if (spectrum.normalizedAllowedBoundary.x.max < boundary.x.max) {
          boundary.x.max = spectrum.normalizedAllowedBoundary.x.max;
        }
        if (spectrum.normalizedAllowedBoundary.y.min < boundary.y.min) {
          boundary.y.min = spectrum.normalizedAllowedBoundary.y.min;
        }
        if (spectrum.normalizedAllowedBoundary.y.max > boundary.y.max) {
          boundary.y.max = spectrum.normalizedAllowedBoundary.y.max;
        }
      }
    }
    return boundary;
  }
  /**
   * Create SpectraProcessor from normalized TSV
   * @param text - TSV text
   * @param options - Parsing options
   * @param options.fs
   * @returns SpectraProcessor instance
   */
  static fromNormalizedMatrix(text, options = {}) {
    const parsed = parseMatrix(text, options);
    if (!parsed) {
      throw new Error('Can not parse TSV file');
    }
    const spectraProcessor = new SpectraProcessor();
    spectraProcessor.setNormalization({
      from: parsed.x[0],
      to: parsed.x.at(-1),
      numberOfPoints: parsed.x.length
    });
    for (let i = 0; i < parsed.ids.length; i++) {
      spectraProcessor.addFromData({
        x: [],
        y: []
      }, {
        normalized: {
          x: parsed.x,
          y: parsed.matrix[i]
        },
        id: parsed.ids[i],
        meta: parsed.meta[i]
      });
    }
    spectraProcessor.keepOriginal = false;
    return spectraProcessor;
  }
}

/**
 * Get filter annotations
 * @param filter - Filter with from, to, and exclusions
 * @returns Array of rectangle annotations
 */
function getFilterAnnotations(filter = {}) {
  let {
    exclusions = []
  } = filter;
  let annotations = [];
  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
  annotations = exclusions.map(exclusion => {
    const annotation = {
      type: 'rect',
      position: [{
        x: exclusion.from,
        y: '0px'
      }, {
        x: exclusion.to,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,128,1)'
    };
    return annotation;
  });
  if (filter.from !== undefined) {
    annotations.push({
      type: 'rect',
      position: [{
        x: Number.MIN_SAFE_INTEGER,
        y: '0px'
      }, {
        x: filter.from,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,224,1)'
    });
  }
  if (filter.to !== undefined) {
    annotations.push({
      type: 'rect',
      position: [{
        x: filter.to,
        y: '0px'
      }, {
        x: Number.MAX_SAFE_INTEGER,
        y: '2000px'
      }],
      strokeWidth: 0,
      fillColor: 'rgba(255,255,224,1)'
    });
  }
  return annotations;
}

export { SpectraProcessor, getFilterAnnotations, xFindClosestIndex };
//# sourceMappingURL=spectra-processor.esm.js.map
