/**
 * spc-parser - Thermo Galactic GRAMS SPC files parser
 * @version v0.2.0
 * @link https://github.com/cheminfo/spc-parser#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.SPCParser = {}));
}(this, (function (exports) { 'use strict';

  /**
   * Gets the parameter in each bit of the flag
   * @param {number} flag First byte of the main header
   * @returns {object} The parameters
   */
  function getFlagParameters(flag) {
    const parameters = {}; //Z is time

    parameters.y16BitPrecision = (flag & 1) !== 0; //Y values are 16 bits instead of 32

    parameters.useExperimentExtension = (flag & 2) !== 0; //Enable experiment mode

    parameters.multiFile = (flag & 4) !== 0; //Multiple spectra

    parameters.zValuesRandom = (flag & 8) !== 0; //Z values in random order if multiFile

    parameters.zValuesUneven = (flag & 16) !== 0; //Z values ordered but unevenly spaced if multi

    parameters.customAxisLabels = (flag & 32) !== 0; //Custom labels

    parameters.xyxy = (flag & 64) !== 0; //One X array per subfile, for discontinuous curves

    parameters.xy = (flag & 128) !== 0; // Non-evenly spaced X, X before Y

    return parameters;
  }
  /**
   *
   * Gets the Subfile flags
   * @param {number} flag First byte of the subheader
   * @return {object} The parameters
   */

  function getSubFlagParameters(flag) {
    const parameters = {};
    parameters.changed = (flag & 1) !== 0;
    parameters.noPeakTable = (flag & 8) !== 0;
    parameters.modifiedArithmetic = (flag & 128) !== 0;
    return parameters;
  }
  /**
   * Generates an array of evenly spaced numbers
   * @param {number} minimum Lower bound
   * @param {number} maximum Upper bound
   * @param {number} numberPoints Number of points
   * @return {array} Evenly spaced numbers
   */

  function equidistantArray(minimum, maximum, numberPoints) {
    const equidistantArray = new Float64Array(numberPoints);
    const step = (maximum - minimum) / (numberPoints - 1);

    for (let i = 0; i < numberPoints; i++) {
      equidistantArray[i] = minimum + i * step;
    }

    return equidistantArray;
  }
  /**
   * Gets the date encoded in binary in a long number
   * @param {number} long Binary date
   * @return {string} Date formatted to ISO 8601:2019 convention
   */

  function longToDate(long) {
    if (long === 0) {
      return '0000-00-00T00:00:00.00Z';
    }

    const date = new Date();
    date.setUTCFullYear(long >> 20);
    date.setUTCMonth((long >> 16 & 0x0f) - 1);
    date.setUTCDate(long >> 11 & 0x1f);
    date.setUTCHours(long >> 6 & 0x1f);
    date.setUTCMinutes(long & 0x3f);
    date.setUTCSeconds(0);
    date.setUTCMilliseconds(0);
    return date.toISOString();
  }

  /**
   * Parses the subheader of the current subfile
   *
   * @export
   * @param {object} buffer SPC buffer
   * @return {object} Current subfile's subheader
   */

  function subHeader(buffer) {
    const subHeader = {};
    subHeader.parameters = getSubFlagParameters(buffer.readUint8());
    subHeader.exponentY = buffer.readInt8();
    subHeader.indexNumber = buffer.readUint16();
    subHeader.startingZ = buffer.readFloat32();
    subHeader.endingZ = buffer.readFloat32();
    subHeader.noiseValue = buffer.readFloat32();
    subHeader.numberPoints = buffer.readUint32();
    subHeader.numberCoAddedScans = buffer.readUint32();
    subHeader.wAxisValue = buffer.readFloat32();
    subHeader.reserved = buffer.readChars(4).trim();
    return subHeader;
  }
  /**
   * Reads the data block of the SPC file
   *
   * @export
   * @param {object} buffer spc buffer
   * @param {object} mainHeader main header
   * @return {array} Array containing the spectra
   */

  function readDataBlock(buffer, mainHeader) {
    let x;
    let y;
    let spectra = [];

    if (!mainHeader.parameters.xyxy && mainHeader.xy) {
      x = new Float32Array(mainHeader.numberPoints);

      for (let i = 0; i < mainHeader.numberPoints; i++) {
        x[i] = buffer.readFloat32();
      }
    } else if (!mainHeader.parameters.xy) {
      x = equidistantArray(mainHeader.startingX, mainHeader.endingX, mainHeader.numberPoints);
    }

    let spectrum;

    for (let i = 0; i < mainHeader.spectra || mainHeader.fileVersion === 0x4d && buffer.offset + mainHeader.numberPoints < buffer.length; i++) {
      spectrum = {};
      spectrum.meta = subHeader(buffer);

      if (mainHeader.parameters.xyxy) {
        x = new Float32Array(spectrum.meta.numberPoints);

        for (let j = 0; j < spectrum.meta.numberPoints; j++) {
          x[j] = buffer.readFloat32();
        }
      }

      const yFactor = Math.pow(2, spectrum.meta.exponentY - (mainHeader.parameters.y16BitPrecision && spectrum.meta.exponentY !== 0x80 ? 16 : 32));
      const nbPoints = spectrum.meta.numberPoints ? spectrum.meta.numberPoints : mainHeader.numberPoints;

      if (mainHeader.parameters.y16BitPrecision) {
        y = new Float32Array(nbPoints);

        for (let j = 0; j < nbPoints; j++) {
          y[j] = buffer.readInt16() * yFactor;
        }
      } else {
        y = new Float32Array(nbPoints);

        for (let j = 0; j < nbPoints; j++) {
          if (mainHeader.fileVersion === 0x4d) {
            y[j] = ((buffer.readUint8() << 16) + (buffer.readInt8() << 24) + (buffer.readUint8() << 0) + (buffer.readUint8() << 8)) * yFactor;
          } else {
            y[j] = buffer.readInt32() * yFactor;
          }
        }
      }

      spectrum.x = x;
      spectrum.y = y;
      spectra.push(spectrum);
    }

    return spectra;
  }

  /**
   *
   * @param {object} buffer SPC buffer
   * @param {number} logOffset Offset of the log (from mainHeader)
   * @return {object} Object containing log meta, data and text
   */
  function readLogBlock(buffer, logOffset) {
    const logHeader = {};
    logHeader.size = buffer.readUint32(); //Size of the block in bytes

    logHeader.memorySize = buffer.readUint32(); //Size of the memory rounded up to nearest multiple of 4096

    logHeader.textOffset = buffer.readUint32(); //Offset to Text section

    logHeader.binarySize = buffer.readUint32(); //Size of binary log block

    logHeader.diskArea = buffer.readUint32(); //Size of the disk area

    logHeader.reserved = buffer.readChars(44).trim(); //Reserved space

    const logData = buffer.readChars(logHeader.binarySize);
    buffer.offset = logOffset + logHeader.textOffset;
    const logASCII = buffer.readChars(logHeader.size - logHeader.textOffset).trim();
    return {
      meta: logHeader,
      data: logData,
      text: logASCII
    };
  }

  /**
   * Gives meaning to type codes
   * @param {number} xzwType x, z or w type code
   * @return {string} String corresponding to the code
   */
  function xzwTypes(xzwType) {
    switch (xzwType) {
      case 1:
        return 'Wavenumber (cm-1)';

      case 2:
        return 'Micrometers (um)';

      case 3:
        return 'Nanometers (nm)';

      case 4:
        return 'Seconds';

      case 5:
        return 'Minutes';

      case 6:
        return 'Hertz (Hz)';

      case 7:
        return 'Kilohertz (KHz)';

      case 8:
        return 'Megahertz (MHz)';

      case 9:
        return 'Mass (M/z)';

      case 10:
        return 'Parts per million (PPM)';

      case 11:
        return 'Days';

      case 12:
        return 'Years';

      case 13:
        return 'Raman Shift (cm-1)';

      case 14:
        return 'eV';

      case 15:
        return 'XYZ text labels in fcatxt (old 0x4D version only)';

      case 16:
        return 'Diode Number';

      case 17:
        return 'Channel ';

      case 18:
        return 'Degrees';

      case 19:
        return 'Temperature (F)';

      case 20:
        return 'Temperature (C)';

      case 21:
        return 'Temperature (K)';

      case 22:
        return 'Data Points';

      case 23:
        return 'Milliseconds (mSec)';

      case 24:
        return 'Microseconds (uSec)';

      case 25:
        return 'Nanoseconds (nSec)';

      case 26:
        return 'Gigahertz (GHz)';

      case 27:
        return 'Centimeters (cm)';

      case 28:
        return 'Meters (m)';

      case 29:
        return 'Millimeters (mm)';

      case 30:
        return 'Hours';

      case 255:
        return 'Double interferogram (no display labels)';

      default:
        return 'Arbitrary';
    }
  }
  /**
   * Gives meaning to y type codes
   * @param {number} yType y type code
   * @return {string} String corresponding to the code
   */

  function yTypes(yType) {
    switch (yType) {
      case 0:
        return 'Arbitrary Intensity';

      case 1:
        return 'Interferogram';

      case 2:
        return 'Absorbance';

      case 3:
        return 'Kubelka-Monk';

      case 4:
        return 'Counts';

      case 5:
        return 'Volts';

      case 6:
        return 'Degrees';

      case 7:
        return 'Milliamps';

      case 8:
        return 'Millimeters';

      case 9:
        return 'Millivolts';

      case 10:
        return 'Log(1/R)';

      case 11:
        return 'Percent';

      case 12:
        return 'Intensity';

      case 13:
        return 'Relative Intensity';

      case 14:
        return 'Energy';

      case 16:
        return 'Decibel';

      case 19:
        return 'Temperature (F)';

      case 20:
        return 'Temperature (C)';

      case 21:
        return 'Temperature (K)';

      case 22:
        return 'Index of Refraction [N]';

      case 23:
        return 'Extinction Coeff. [K]';

      case 24:
        return 'Real';

      case 25:
        return 'Imaginary';

      case 26:
        return 'Complex';

      case 128:
        return 'Transmission (ALL HIGHER MUST HAVE VALLEYS!)';

      case 129:
        return 'Reflectance';

      case 130:
        return 'Arbitrary or Single Beam with Valley Peaks';

      case 131:
        return 'Emission';

      default:
        return 'Reference Arbitrary Energy';
    }
  }
  /**
   * Experiment settings code converter
   * @param {number} code
   * @return {string}
   */

  function experimentSettings(code) {
    switch (code) {
      case 1:
        return 'Gas Chromatogram';

      case 2:
        return 'General Chromatogram (same as SPCGEN with TCGRAM)';

      case 3:
        return 'HPLC Chromatogram';

      case 4:
        return 'FT-IR, FT-NIR, FT-Raman Spectrum or Igram (Can also be used for scanning IR.)';

      case 5:
        return 'NIR Spectrum (Usually multi-spectral data sets for calibration.)';

      case 7:
        return 'UV-VIS Spectrum (Can be used for single scanning UV-VIS-NIR.)';

      case 8:
        return 'X-ray Diffraction Spectrum';

      case 9:
        return 'Mass Spectrum  (Can be single, GC-MS, Continuum, Centroid or TOF.)';

      case 10:
        return 'NMR Spectrum or FID';

      case 11:
        return 'Raman Spectrum (Usually Diode Array, CCD, etc. use SPCFTIR for FT-Raman.)';

      case 12:
        return 'Fluorescence Spectrum';

      case 13:
        return 'Atomic Spectrum';

      case 14:
        return 'Chromatography Diode Array Spectra';

      default:
        return 'General SPC (could be anything)';
    }
  }

  /**
   * Main header parsing - First 512/256 bytes (new/old format)
   * @param {object} buffer SPC buffer
   * @return {object} Main header
   */

  function mainHeader(buffer) {
    const header = {};
    header.parameters = getFlagParameters(buffer.readUint8()); //Each bit contains a parameter

    header.fileVersion = buffer.readUint8(); //4B => New format; 4D => LabCalc format

    switch (header.fileVersion) {
      case 0x4b:
        break;

      case 0x4c:
        buffer.setBigEndian();
        break;

      case 0x4d:
        return oldHeader(buffer, header);

      default:
        throw new Error('Unrecognized file format: byte 01 must be either 4B, 4C or 4D');
    }

    header.experimentType = experimentSettings(buffer.readUint8()); //Experiment type code (See SPC.h)

    header.exponentY = buffer.readInt8(); //Exponent for Y values (80h = floating point): FloatY = (2^Exp)*IntY/(2^32) 32-bit; FloatY = (2^Exp)*IntY/(2^16) 32-bit

    header.numberPoints = buffer.readUint32(); //Number of points (if not XYXY)

    header.startingX = buffer.readFloat64(); //First X coordinate

    header.endingX = buffer.readFloat64(); //Last X coordinate

    header.spectra = buffer.readUint32(); //Number of spectrums

    header.xUnitsType = xzwTypes(buffer.readUint8()); //X Units type code (See types.js)

    header.yUnitsType = yTypes(buffer.readUint8()); //Y ""

    header.zUnitsType = xzwTypes(buffer.readUint8()); //Z ""

    header.postingDisposition = buffer.readUint8(); //Posting disposition (See GRAMSDDE.H)

    header.date = longToDate(buffer.readUint32()); //Date: minutes = first 6 bits, hours = 5 next bits, days = 5 next, months = 4 next, years = 12 last

    header.resolutionDescription = buffer.readChars(9).trim(); //Resolution description text

    header.sourceInstrumentDescription = buffer.readChars(9).trim(); // Source Instrument description text

    header.peakPointNumber = buffer.readUint16(); //Peak point number for interferograms

    header.spare = [];

    for (let i = 0; i < 8; i++) {
      header.spare.push(buffer.readFloat32());
    }

    if (header.fileVersion === 0x4c) {
      //Untested case because no test files
      header.spare.reverse();
    }

    header.memo = buffer.readChars(130).trim();
    header.xyzLabels = buffer.readChars(30).trim();
    header.logOffset = buffer.readUint32(); //Byte offset to Log Block

    header.modifiedFlag = buffer.readUint32(); //File modification flag (See values in SPC.H)

    header.processingCode = buffer.readUint8(); //Processing code (See GRAMSDDE.H)

    header.calibrationLevel = buffer.readUint8(); //Calibration level + 1

    header.subMethodSampleInjectionNumber = buffer.readUint16(); //Sub-method sample injection number

    header.concentrationFactor = buffer.readFloat32(); //Floating data multiplier concentration factor

    header.methodFile = buffer.readChars(48).trim(); //Method file

    header.zSubIncrement = buffer.readFloat32(); //Z subfile increment for even Z Multifiles

    header.wPlanes = buffer.readUint32();
    header.wPlaneIncrement = buffer.readFloat32();
    header.wAxisUnits = xzwTypes(buffer.readUint8()); //W axis units code

    header.reserved = buffer.readChars(187).trim(); //Reserved space (Must be zero)

    return header;
  }
  /**
   *Old version files header parsing
   *
   * @export
   * @param {object} buffer SPC buffer
   * @param {object} header Header from the previous function
   * @return {object} Object containing the metadata of the old file
   */

  function oldHeader(buffer, header) {
    header.exponentY = buffer.readInt16(); //Word (16 bits) instead of byte

    header.numberPoints = buffer.readFloat32();
    header.startingX = buffer.readFloat32();
    header.endingX = buffer.readFloat32();
    header.xUnitsType = xzwTypes(buffer.readUint8());
    header.yUnitsType = yTypes(buffer.readUint8());
    const date = new Date();
    const zTypeYear = buffer.readUint16(); //Unrelated to Z axis

    date.setUTCFullYear(zTypeYear % 4096); // todo might be wrong

    date.setUTCMonth(Math.max(buffer.readUint8() - 1, 0));
    date.setUTCDate(buffer.readUint8());
    date.setUTCHours(buffer.readUint8());
    date.setUTCMinutes(buffer.readUint8());
    header.date = date.toISOString();
    header.resolutionDescription = buffer.readChars(8).trim();
    header.peakPointNumber = buffer.readUint16();
    header.scans = buffer.readUint16();
    header.spare = [];

    for (let i = 0; i < 7; i++) {
      header.spare.push(buffer.readFloat32());
    }

    header.memo = buffer.readChars(130).trim();
    header.xyzLabels = buffer.readChars(30).trim();
    return header;
  }

  /**
   * Parses an SPC file
   *
   * @param {object} buffer SPC file buffer
   * @return {object} Object containing every information contained in the SPC file
   */

  function parse(buffer) {
    const meta = mainHeader(buffer);
    const spectra = readDataBlock(buffer, meta);

    if (meta.logOffset && meta.logOffset !== 0) {
      return {
        meta,
        spectra,
        logs: readLogBlock(buffer, meta.logOffset)
      };
    }

    return {
      meta,
      spectra
    };
  }

  exports.parse = parse;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=spc-parser.js.map
