/**
 * spectrum-generator - generate a spectrum from discrete peaks
 * @version v5.4.0
 * @link https://github.com/cheminfo/spectrum-generator#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.spectrumGenerator = {}));
})(this, (function (exports) { 'use strict';

    const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
    const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
    const ROOT_THREE = Math.sqrt(3);
    const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
    const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

    // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    // This code yields to a good approximation
    // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    function erfinv(x) {
      let a = 0.147;
      if (x === 0) return 0;
      let ln1MinusXSqrd = Math.log(1 - x * x);
      let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
      let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
      let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
      return secondSqrt * (x > 0 ? 1 : -1);
    }

    class Shape1D {}

    class Gaussian extends Shape1D {
      constructor(options = {}) {
        super();
        const {
          fwhm = 500,
          sd,
          height
        } = options;
        this.fwhm = sd ? widthToFWHM$2(2 * sd) : fwhm;
        this.height = height === undefined ? Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) / this.fwhm : height;
      }

      fwhmToWidth(fwhm = this.fwhm) {
        return fwhmToWidth$2(fwhm);
      }

      widthToFWHM(width) {
        return widthToFWHM$2(width);
      }

      fct(x) {
        return fct$3(x, this.fwhm);
      }

      getArea() {
        return getArea$2({
          fwhm: this.fwhm,
          height: this.height
        });
      }

      getFactor(area) {
        return getFactor$3(area);
      }

      getData(options = {}) {
        const {
          length,
          factor
        } = options;
        return getData$3({
          fwhm: this.fwhm,
          height: this.height,
          factor,
          length
        });
      }

    }
    /**
     * Return a parameterized function of a gaussian shape (see README for equation).
     * @returns - the y value of gaussian with the current parameters.
     */

    function fct$3(x, fwhm) {
      return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * for more information check the [mathworld page](https://mathworld.wolfram.com/GaussianFunction.html)
     * @returns fwhm
     */

    function widthToFWHM$2(width) {
      return width * ROOT_2LN2;
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * for more information check the [mathworld page](https://mathworld.wolfram.com/GaussianFunction.html)
     * @param fwhm - Full Width at Half Maximum.
     * @returns width
     */

    function fwhmToWidth$2(fwhm) {
      return fwhm / ROOT_2LN2;
    }
    /**
     * Calculate the area of a specific shape.
     * @returns returns the area of the specific shape and parameters.
     */

    function getArea$2(options) {
      let {
        fwhm,
        sd,
        height = 1
      } = options;
      if (sd) fwhm = widthToFWHM$2(2 * sd);

      if (fwhm === undefined) {
        throw new Error('should pass fwhm or sd parameters');
      }

      return height * ROOT_PI_OVER_LN2 * fwhm / 2;
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage.
     * @param [area=0.9999] Expected area to be covered.
     * @returns
     */

    function getFactor$3(area = 0.9999) {
      return Math.sqrt(2) * erfinv(area);
    }
    /**
     * Calculate intensity array of a gaussian shape.
     * @returns {Float64Array} Intensity values.
     */

    function getData$3(options = {}) {
      let {
        length,
        factor = getFactor$3(),
        fwhm = 500,
        sd,
        height
      } = options;
      if (sd) fwhm = widthToFWHM$2(2 * sd);

      if (!height) {
        height = Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) / fwhm;
      }

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = fct$3(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }

    class Lorentzian extends Shape1D {
      constructor(options = {}) {
        super();
        const {
          fwhm = 500,
          height
        } = options;
        this.fwhm = fwhm;
        this.height = height === undefined ? 2 / Math.PI / fwhm : height;
      }

      fwhmToWidth(fwhm = this.fwhm) {
        return fwhmToWidth$1(fwhm);
      }

      widthToFWHM(width) {
        return widthToFWHM$1(width);
      }

      fct(x) {
        return fct$2(x, this.fwhm);
      }

      getArea() {
        return getArea$1({
          fwhm: this.fwhm,
          height: this.height
        });
      }

      getFactor(area) {
        return getFactor$2(area);
      }

      getData(options = {}) {
        const {
          length,
          factor
        } = options;
        return getData$2({
          fwhm: this.fwhm,
          height: this.height,
          factor,
          length
        });
      }

    }
    /**
     * Return a parameterized function of a lorentzian shape (see README for equation).
     * @param x - x value to calculate.
     * @param fwhm - full width half maximum
     * @returns - the y value of lorentzian with the current parameters.
     */

    function fct$2(x, fwhm) {
      return Math.pow(fwhm, 2) / (4 * Math.pow(x, 2) + Math.pow(fwhm, 2));
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * for more information check the [mathworld page](https://mathworld.wolfram.com/LorentzianFunction.html)
     * @param width - Width between the inflection points
     * @returns fwhm
     */

    function widthToFWHM$1(width) {
      return width * ROOT_THREE;
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * for more information check the [mathworld page](https://mathworld.wolfram.com/LorentzianFunction.html)
     * @param fwhm - Full Width at Half Maximum.
     * @returns width
     */

    function fwhmToWidth$1(fwhm) {
      return fwhm / ROOT_THREE;
    }
    /**
     * Calculate the area of a specific shape.
     * @returns returns the area of the specific shape and parameters.
     */

    function getArea$1(options) {
      const {
        fwhm,
        height = 1
      } = options;

      if (fwhm === undefined) {
        throw new Error('should pass fwhm or sd parameters');
      }

      return height * Math.PI * fwhm / 2;
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage.
     * @param [area=0.9999] Expected area to be covered.
     * @returns
     */

    function getFactor$2(area = 0.9999) {
      return 2 * Math.tan(Math.PI * (area - 0.5));
    }
    /**
     * Calculate intensity array of a lorentzian shape.
     * @returns {Float64Array} y values
     */

    function getData$2(options = {}) {
      let {
        length,
        factor = getFactor$2(),
        fwhm = 500,
        height
      } = options;

      if (!height) {
        height = 2 / Math.PI / fwhm;
      }

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = fct$2(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }

    class PseudoVoigt extends Shape1D {
      constructor(options = {}) {
        super();
        const {
          fwhm = 500,
          height,
          mu = 0.5
        } = options;
        this.mu = mu;
        this.fwhm = fwhm;
        this.height = height === undefined ? 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2) : height;
      }

      fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
        return fwhmToWidth(fwhm, mu);
      }

      widthToFWHM(width, mu = this.mu) {
        return widthToFWHM(width, mu);
      }

      fct(x) {
        return fct$1(x, this.fwhm, this.mu);
      }

      getArea() {
        return getArea({
          fwhm: this.fwhm,
          height: this.height,
          mu: this.mu
        });
      }

      getFactor(area) {
        return getFactor$1(area);
      }

      getData(options = {}) {
        const {
          length,
          factor
        } = options;
        return getData$1({
          fwhm: this.fwhm,
          height: this.height,
          mu: this.mu,
          factor,
          length
        });
      }

    }
    /**
     * Return a parameterized function of a pseudo voigt shape (see README for equation).
     * @param x - x value to calculate.
     * @param fwhm - full width half maximum
     * @returns - the y value of pseudo voigt with the current parameters.
     */

    function fct$1(x, fwhm, mu) {
      return (1 - mu) * fct$2(x, fwhm) + mu * fct$3(x, fwhm);
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * @param width - Width between the inflection points
     * @param [mu=0.5] Ratio of gaussian contribution in the shape
     * @returns fwhm
     */

    function widthToFWHM(width, mu = 0.5) {
      return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * @param fwhm - Full Width at Half Maximum.
     * @param [mu=0.5] Ratio of gaussian contribution in the shape
     * @returns width
     */

    function fwhmToWidth(fwhm, mu = 0.5) {
      return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
    }
    /**
     * Calculate the area of a specific shape.
     * @returns returns the area of the specific shape and parameters.
     */

    function getArea(options) {
      const {
        fwhm,
        height = 1,
        mu = 0.5
      } = options;

      if (fwhm === undefined) {
        throw new Error('should pass fwhm or sd parameters');
      }

      return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage.
     * @param [area=0.9999] Expected area to be covered.
     * @returns
     */

    function getFactor$1(area = 0.9999, mu = 0.5) {
      return mu < 1 ? getFactor$2(area) : getFactor$3(area);
    }
    /**
     * Calculate intensity array of a pseudo voigt shape.
     * @returns {Float64Array} y values
     */

    function getData$1(options = {}) {
      let {
        length,
        factor = getFactor$1(),
        fwhm = 500,
        height,
        mu = 0.5
      } = options;

      if (!height) {
        height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
      }

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = fct$1(i - center, fwhm, mu) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }

    class Shape2D {}

    class Gaussian2D extends Shape2D {
      constructor(options = {}) {
        super();
        let {
          fwhm = 50,
          sd,
          height
        } = options;
        fwhm = ensureFWHM2D(fwhm, sd);
        this.fwhmX = fwhm.x;
        this.fwhmY = fwhm.y;
        this.height = height === undefined ? -GAUSSIAN_EXP_FACTOR / Math.PI / this.fwhmY / this.fwhmX : height;
      }

      fct(x, y) {
        return fct(x, y, this.fwhmX, this.fwhmY);
      }

      getData(options = {}) {
        const {
          factor,
          length
        } = options;
        return getData({
          fwhm: {
            x: this.fwhmY,
            y: this.fwhmY
          },
          height: this.height,
          factor,
          length
        });
      }

      getFactor(surface) {
        return getFactor(surface);
      }

      getSurface() {
        return getSurface({
          fwhm: {
            x: this.fwhmY,
            y: this.fwhmY
          },
          height: this.height
        });
      }

      widthToFWHM(width) {
        return widthToFWHM$2(width);
      }

      fwhmToWidth(fwhm) {
        return fwhmToWidth$2(fwhm);
      }

      set fwhm(fwhm) {
        fwhm = ensureXYNumber$1(fwhm);
        this.fwhmX = fwhm.x;
        this.fwhmY = fwhm.y;
      }

    }
    /**
     * Return a parameterized function of a Gaussian2D shape (see README for equation).
     * @param x - x value to calculate.
     * @param y - y value to calculate.
     * @param fwhmX - full width half maximum in the x axis.
     * @param fwhmY - full width half maximum in the y axis.
     * @returns - the z value of bi-dimensional gaussian with the current parameters.
     */

    function fct(x, y, xFWHM, yFWHM) {
      return Math.exp(GAUSSIAN_EXP_FACTOR * (Math.pow(x / xFWHM, 2) + Math.pow(y / yFWHM, 2)));
    }
    /**
     * Calculate the intensity matrix of a gaussian shape.
     * @returns z values.
     */

    function getData(options = {}) {
      let {
        fwhm = 50,
        factor = getFactor(),
        height,
        sd,
        length = {
          x: 0,
          y: 0
        }
      } = options;
      fwhm = ensureFWHM2D(fwhm, sd);
      factor = ensureXYNumber$1(factor);
      length = ensureXYNumber$1(length);

      if (!height) {
        height = -GAUSSIAN_EXP_FACTOR / Math.PI / fwhm.y / fwhm.x;
      }

      for (const axis of ['x', 'y']) {
        if (!length[axis]) {
          length[axis] = Math.min(Math.ceil(fwhm[axis] * factor[axis]), Math.pow(2, 25) - 1);
          if (length[axis] % 2 === 0) length[axis]++;
        }
      }

      const xCenter = (length.x - 1) / 2;
      const yCenter = (length.y - 1) / 2;
      const data = new Array(length.x);

      for (let i = 0; i < length.x; i++) {
        data[i] = new Float64Array(length.y);
      }

      for (let i = 0; i < length.x; i++) {
        for (let j = 0; j < length.y; j++) {
          data[i][j] = fct(i - xCenter, j - yCenter, fwhm.x, fwhm.y) * height;
        }
      }

      return data;
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific surface coverage.
     * @param [surface=0.9999] Expected volume to be covered.
     * @returns
     */

    function getFactor(surface = 0.9999) {
      return Math.sqrt(2) * erfinv(surface);
    }
    /**
     * Calculate the surface of gaussian shape.
     * @returns The surface of the specific shape and parameters.
     */

    function getSurface(options = {}) {
      let {
        fwhm = 50,
        height = 1
      } = options;
      if (typeof fwhm !== 'object') fwhm = {
        x: fwhm,
        y: fwhm
      };
      return height * Math.PI * fwhm.y * fwhm.x / Math.LN2 / 4;
    }

    function ensureXYNumber$1(input) {
      return typeof input !== 'object' ? {
        x: input,
        y: input
      } : { ...input
      };
    }

    function ensureFWHM2D(fwhm, sd) {
      if (sd !== undefined) {
        let sdObject = ensureXYNumber$1(sd);
        return {
          x: widthToFWHM$2(2 * sdObject.x),
          y: widthToFWHM$2(2 * sdObject.y)
        };
      } else if (fwhm !== undefined) {
        return ensureXYNumber$1(fwhm);
      } else {
        throw new Error('ensureFWHM2D must have either fwhm or sd defined');
      }
    }

    /**
     * Generate a instance of a specific kind of shape.
     */

    function getShape1D(kind, shapeOptions = {}) {
      switch (kind) {
        case 'gaussian':
          return new Gaussian(shapeOptions);

        case 'lorentzian':
          return new Lorentzian(shapeOptions);

        case 'pseudoVoigt':
          return new PseudoVoigt(shapeOptions);

        default:
          {
            const unHandled = kind; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

            throw Error(`Unknown distribution ${unHandled}`);
          }
      }
    }

    /**
     * Generate a instance of a specific kind of shape.
     */

    function getShape2D(kind, shapeOptions = {}) {
      switch (kind) {
        case 'gaussian':
          return new Gaussian2D(shapeOptions);

        default:
          {
            const unHandled = kind; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

            throw Error(`Unknown distribution ${unHandled}`);
          }
      }
    }

    function addBaseline(data, baselineFct) {
      if (!baselineFct) return data;
      let xs = data.x;
      let ys = data.y;

      for (let i = 0; i < xs.length; i++) {
        ys[i] += baselineFct(xs[i]);
      }

      return data;
    }

    var defaultSource = Math.random;

    var randomUniform = (function sourceRandomUniform(source) {
      function randomUniform(min, max) {
        min = min == null ? 0 : +min;
        max = max == null ? 1 : +max;
        if (arguments.length === 1) max = min, min = 0;else max -= min;
        return function () {
          return source() * max + min;
        };
      }

      randomUniform.source = sourceRandomUniform;
      return randomUniform;
    })(defaultSource);

    var randomNormal = (function sourceRandomNormal(source) {
      function randomNormal(mu, sigma) {
        var x, r;
        mu = mu == null ? 0 : +mu;
        sigma = sigma == null ? 1 : +sigma;
        return function () {
          var y; // If available, use the second previously-generated uniform random.

          if (x != null) y = x, x = null; // Otherwise, generate a new x and y.
          else do {
            x = source() * 2 - 1;
            y = source() * 2 - 1;
            r = x * x + y * y;
          } while (!r || r > 1);
          return mu + sigma * y * Math.sqrt(-2 * Math.log(r) / r);
        };
      }

      randomNormal.source = sourceRandomNormal;
      return randomNormal;
    })(defaultSource);

    const LOOP = 8;
    const FLOAT_MUL = 1 / 16777216;
    const sh1 = 15;
    const sh2 = 18;
    const sh3 = 11;

    function multiply_uint32(n, m) {
      n >>>= 0;
      m >>>= 0;
      const nlo = n & 0xffff;
      const nhi = n - nlo;
      return (nhi * m >>> 0) + nlo * m >>> 0;
    }

    class XSadd {
      constructor(seed = Date.now()) {
        this.state = new Uint32Array(4);
        this.init(seed);
        this.random = this.getFloat.bind(this);
      }
      /**
       * Returns a 32-bit integer r (0 <= r < 2^32)
       */


      getUint32() {
        this.nextState();
        return this.state[3] + this.state[2] >>> 0;
      }
      /**
       * Returns a floating point number r (0.0 <= r < 1.0)
       */


      getFloat() {
        return (this.getUint32() >>> 8) * FLOAT_MUL;
      }

      init(seed) {
        if (!Number.isInteger(seed)) {
          throw new TypeError('seed must be an integer');
        }

        this.state[0] = seed;
        this.state[1] = 0;
        this.state[2] = 0;
        this.state[3] = 0;

        for (let i = 1; i < LOOP; i++) {
          this.state[i & 3] ^= i + multiply_uint32(1812433253, this.state[i - 1 & 3] ^ this.state[i - 1 & 3] >>> 30 >>> 0) >>> 0;
        }

        this.periodCertification();

        for (let i = 0; i < LOOP; i++) {
          this.nextState();
        }
      }

      periodCertification() {
        if (this.state[0] === 0 && this.state[1] === 0 && this.state[2] === 0 && this.state[3] === 0) {
          this.state[0] = 88; // X

          this.state[1] = 83; // S

          this.state[2] = 65; // A

          this.state[3] = 68; // D
        }
      }

      nextState() {
        let t = this.state[0];
        t ^= t << sh1;
        t ^= t >>> sh2;
        t ^= this.state[3] << sh3;
        this.state[0] = this.state[1];
        this.state[1] = this.state[2];
        this.state[2] = this.state[3];
        this.state[3] = t;
      }

    }

    function addNoise(data, percent = 0, options = {}) {
      const {
        seed
      } = options;
      const distribution = options.distribution || 'uniform';
      let generateRandomNumber;

      switch (distribution) {
        case 'uniform':
          {
            generateRandomNumber = getRandom(randomUniform, seed, -0.5, 0.5);
            break;
          }

        case 'normal':
          {
            generateRandomNumber = getRandom(randomNormal, seed);
            break;
          }

        default:
          {
            const unHandled = distribution; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

            throw Error(`Unknown distribution ${unHandled}`);
          }
      }

      if (!percent) return data;
      let ys = data.y;
      let factor = percent * findMax(ys) / 100;

      for (let i = 0; i < ys.length; i++) {
        ys[i] += generateRandomNumber() * factor;
      }

      return data;
    }

    function getRandom(func, seed, ...args) {
      return typeof seed === 'number' ? func.source(new XSadd(seed).random)(...args) : func(...args);
    }

    function findMax(array) {
      let max = Number.MIN_VALUE;

      for (let item of array) {
        if (item > max) max = item;
      }

      return max;
    }

    class SpectrumGenerator {
      constructor(options = {}) {
        const {
          from = 0,
          to = 1000,
          nbPoints = 10001,
          peakWidthFct = () => 5,
          shape = {
            kind: 'gaussian'
          }
        } = options;
        this.from = from;
        this.to = to;
        this.nbPoints = nbPoints;
        this.interval = (this.to - this.from) / (this.nbPoints - 1);
        this.peakWidthFct = peakWidthFct;
        this.maxPeakHeight = Number.MIN_SAFE_INTEGER;
        this.data = {
          x: new Float64Array(this.nbPoints),
          y: new Float64Array(this.nbPoints)
        };
        const kind = shape.kind;
        const {
          options: shapeOptions = {}
        } = shape;
        let shapeGenerator = getShape1D(kind, shapeOptions);
        this.shape = shapeGenerator;
        assertNumber$1(this.from, 'from');
        assertNumber$1(this.to, 'to');
        assertInteger$1(this.nbPoints, 'nbPoints');

        if (this.to <= this.from) {
          throw new RangeError('to option must be larger than from');
        }

        if (typeof this.peakWidthFct !== 'function') {
          throw new TypeError('peakWidthFct option must be a function');
        }

        this.reset();
      }
      /**
       * Add a series of peaks to the spectrum.
       * @param peaks - Peaks to add.
       */


      addPeaks(peaks, options) {
        if (!Array.isArray(peaks) && (typeof peaks !== 'object' || peaks.x === undefined || peaks.y === undefined || !Array.isArray(peaks.x) || !Array.isArray(peaks.y) || peaks.x.length !== peaks.y.length)) {
          throw new TypeError('peaks must be an array or an object containing x[] and y[]');
        }

        if (Array.isArray(peaks)) {
          for (const peak of peaks) {
            this.addPeak(peak, options);
          }
        } else {
          for (let i = 0; i < peaks.x.length; i++) {
            this.addPeak([peaks.x[i], peaks.y[i]], options);
          }
        }

        return this;
      }
      /**
       * Add a single peak to the spectrum.
       * @param peak
       * @param options
       */


      addPeak(peak, options = {}) {
        if (Array.isArray(peak) && peak.length < 2) {
          throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
        }

        if (!Array.isArray(peak) && (peak.x === undefined || peak.y === undefined)) {
          throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
        }

        let xPosition;
        let intensity;
        let peakWidth;
        let peakShapeOptions;

        if (Array.isArray(peak)) {
          [xPosition, intensity, peakWidth, peakShapeOptions] = peak;
        } else {
          xPosition = peak.x;
          intensity = peak.y;
          peakWidth = peak.width;
          peakShapeOptions = peak.shape;
        }

        if (intensity > this.maxPeakHeight) this.maxPeakHeight = intensity;
        let {
          width = peakWidth === undefined ? this.peakWidthFct(xPosition) : peakWidth,
          widthLeft,
          widthRight,
          shape: shapeOptions
        } = options;

        if (peakShapeOptions) {
          shapeOptions = shapeOptions ? { ...shapeOptions,
            ...peakShapeOptions
          } : peakShapeOptions;
        }

        if (shapeOptions) {
          const kind = shapeOptions.kind;
          const {
            options: shapeParameters = {}
          } = shapeOptions;
          this.shape = getShape1D(kind, shapeParameters);
        }

        if (!widthLeft) widthLeft = width;
        if (!widthRight) widthRight = width;
        let factor = options.factor === undefined ? this.shape.getFactor() : options.factor;
        const firstValue = xPosition - widthLeft / 2 * factor;
        const lastValue = xPosition + widthRight / 2 * factor;
        const firstPoint = Math.max(0, Math.floor((firstValue - this.from) / this.interval));
        const lastPoint = Math.min(this.nbPoints - 1, Math.ceil((lastValue - this.from) / this.interval));
        const middlePoint = Math.round((xPosition - this.from) / this.interval); // PEAK SHAPE MAY BE ASYMMETRC (widthLeft and widthRight) !
        // we calculate the left part of the shape

        this.shape.fwhm = widthLeft;

        for (let index = firstPoint; index < Math.max(middlePoint, 0); index++) {
          this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
        } // we calculate the right part of the gaussian


        this.shape.fwhm = widthRight;

        for (let index = Math.min(middlePoint, lastPoint); index <= lastPoint; index++) {
          this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
        }

        return this;
      }
      /**
       * Add a baseline to the spectrum.
       * @param baselineFct - Mathematical function producing the baseline you want.
       */


      addBaseline(baselineFct) {
        addBaseline(this.data, baselineFct);
        return this;
      }
      /**
       * Add noise to the spectrum.
       * @param percent - Noise's amplitude in percents of the spectrum max value. Default: 0.
       */


      addNoise(percent, options) {
        addNoise(this.data, percent, options);
        return this;
      }
      /**
       * Get the generated spectrum.
       */


      getSpectrum(options = {}) {
        if (typeof options === 'boolean') {
          options = {
            copy: options
          };
        }

        const {
          copy = true,
          threshold = 0
        } = options;

        if (threshold) {
          let minPeakHeight = this.maxPeakHeight * threshold;
          let x = [];
          let y = [];

          for (let i = 0; i < this.data.x.length; i++) {
            if (this.data.y[i] >= minPeakHeight) {
              x.push(this.data.x[i]);
              y.push(this.data.y[i]);
            }
          }

          return {
            x,
            y
          };
        }

        if (copy) {
          return {
            x: this.data.x.slice(),
            y: this.data.y.slice()
          };
        } else {
          return this.data;
        }
      }
      /**
       * Resets the generator with an empty spectrum.
       */


      reset() {
        const spectrum = this.data;

        for (let i = 0; i < this.nbPoints; i++) {
          spectrum.x[i] = this.from + i * this.interval;
        }

        return this;
      }

    }

    function assertInteger$1(value, name) {
      if (!Number.isInteger(value)) {
        throw new TypeError(`${name} option must be an integer`);
      }
    }

    function assertNumber$1(value, name) {
      if (!Number.isFinite(value)) {
        throw new TypeError(`${name} option must be a number`);
      }
    }
    /**
     * Generates a spectrum and returns it.
     * @param peaks - List of peaks to put in the spectrum.
     * @param options
     */


    function generateSpectrum(peaks, options = {}) {
      const {
        generator: generatorOptions,
        noise,
        baseline,
        threshold,
        peaks: addPeaksOptions
      } = options;
      const generator = new SpectrumGenerator(generatorOptions);
      generator.addPeaks(peaks, addPeaksOptions);
      if (baseline) generator.addBaseline(baseline);

      if (noise) {
        const {
          percent,
          options: addNoiseOptions
        } = noise;
        generator.addNoise(percent, addNoiseOptions);
      }

      return generator.getSpectrum({
        threshold
      });
    }

    const toString = Object.prototype.toString;
    function isAnyArray(object) {
      return toString.call(object).endsWith('Array]');
    }

    function max(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }

      var _options$fromIndex = options.fromIndex,
          fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
          _options$toIndex = options.toIndex,
          toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }

      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }

      var maxValue = input[fromIndex];

      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }

      return maxValue;
    }

    function min(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }

      var _options$fromIndex = options.fromIndex,
          fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
          _options$toIndex = options.toIndex,
          toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }

      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }

      var minValue = input[fromIndex];

      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }

      return minValue;
    }

    function getMinMax(data) {
      let min$1 = Number.MAX_SAFE_INTEGER;
      let max$1 = Number.MIN_SAFE_INTEGER;

      for (let row of data) {
        let rowMin = min(row);
        let rowMax = max(row);
        if (min$1 > rowMin) min$1 = rowMin;
        if (max$1 < rowMax) max$1 = rowMax;
      }

      return {
        min: min$1,
        max: max$1
      };
    }

    const axis2D = ['x', 'y'];
    const peakCoordinates = ['x', 'y', 'z'];
    class Spectrum2DGenerator {
      constructor(options = {}) {
        let {
          from = 0,
          to = 100,
          nbPoints = 1001,
          peakWidthFct = () => 5,
          shape = {
            kind: 'gaussian'
          }
        } = options;
        from = ensureXYNumber(from);
        to = ensureXYNumber(to);
        nbPoints = ensureXYNumber(nbPoints);

        for (const axis of axis2D) {
          assertNumber(from[axis], `from-${axis}`);
          assertNumber(to[axis], `to-${axis}`);
          assertInteger(nbPoints[axis], `nbPoints-${axis}`);
        }

        this.from = from;
        this.to = to;
        this.nbPoints = nbPoints;
        this.interval = calculeIntervals(from, to, nbPoints);
        this.peakWidthFct = peakWidthFct;
        this.maxPeakHeight = Number.MIN_SAFE_INTEGER;
        const kind = shape.kind;
        const {
          options: shapeOptions = {}
        } = shape;
        let shapeGenerator = getShape2D(kind, shapeOptions);
        this.shape = shapeGenerator;
        this.data = {
          x: new Float64Array(nbPoints.x),
          y: new Float64Array(nbPoints.y),
          z: createMatrix(this.nbPoints)
        };

        for (const axis of axis2D) {
          if (this.to[axis] <= this.from[axis]) {
            throw new RangeError('to option must be larger than from');
          }
        }

        if (typeof this.peakWidthFct !== 'function') {
          throw new TypeError('peakWidthFct option must be a function');
        }

        this.reset();
      }

      addPeaks(peaks, options) {
        if (!Array.isArray(peaks) && (typeof peaks !== 'object' || peaks.x === undefined || peaks.y === undefined || !Array.isArray(peaks.x) || !Array.isArray(peaks.y) || peaks.x.length !== peaks.y.length)) {
          throw new TypeError('peaks must be an array or an object containing x[] and y[]');
        }

        if (Array.isArray(peaks)) {
          for (const peak of peaks) {
            this.addPeak(peak, options);
          }
        } else {
          let nbPeaks = peaks.x.length;

          for (const c of peakCoordinates) {
            if (peaks[c] && Array.isArray(peaks[c])) {
              if (nbPeaks !== peaks[c].length) {
                throw new Error('x, y, z should have the same length');
              }
            }
          }

          for (let i = 0; i < peaks.x.length; i++) {
            this.addPeak([peaks.x[i], peaks.y[i], peaks.z[i]], options);
          }
        }

        return this;
      }

      addPeak(peak, options = {}) {
        if (Array.isArray(peak) && peak.length < 3) {
          throw new Error('peak must be an array with three (or four) values or an object with {x,y,z,width?}');
        }

        if (!Array.isArray(peak) && peakCoordinates.some(e => peak[e] === undefined)) {
          throw new Error('peak must be an array with three (or four) values or an object with {x,y,z,width?}');
        }

        let xPosition;
        let yPosition;
        let intensity;
        let peakWidth;
        let peakShapeOptions;

        if (Array.isArray(peak)) {
          [xPosition, yPosition, intensity, peakWidth, peakShapeOptions] = peak;
        } else {
          xPosition = peak.x;
          yPosition = peak.y;
          intensity = peak.z;
          peakWidth = peak.width;
          peakShapeOptions = peak.shape;
        }

        const position = {
          x: xPosition,
          y: yPosition
        };
        if (intensity > this.maxPeakHeight) this.maxPeakHeight = intensity;
        let {
          width = peakWidth === undefined ? this.peakWidthFct(xPosition, yPosition) : peakWidth,
          shape: shapeOptions
        } = options;

        if (peakShapeOptions) {
          shapeOptions = shapeOptions ? { ...shapeOptions,
            ...peakShapeOptions
          } : peakShapeOptions;
        }

        if (shapeOptions) {
          const kind = shapeOptions.kind;
          const {
            options: shapeParameters = {}
          } = shapeOptions;
          this.shape = getShape2D(kind, shapeParameters);
        }

        width = ensureXYNumber(width);
        let factor = options.factor === undefined ? this.shape.getFactor() : options.factor;
        factor = ensureXYNumber(factor);
        const firstPoint = {
          x: 0,
          y: 0
        };
        const lastPoint = {
          x: 0,
          y: 0
        };

        for (const axis of axis2D) {
          const first = position[axis] - width[axis] / 2 * factor[axis];
          const last = position[axis] + width[axis] / 2 * factor[axis];
          firstPoint[axis] = Math.max(0, Math.floor((first - this.from[axis]) / this.interval[axis]));
          lastPoint[axis] = Math.min(this.nbPoints[axis], Math.ceil((last - this.from[axis]) / this.interval[axis]));
        }

        this.shape.fwhmX = width.x;
        this.shape.fwhmY = width.y;

        for (let xIndex = firstPoint.x; xIndex < lastPoint.x; xIndex++) {
          for (let yIndex = firstPoint.y; yIndex < lastPoint.y; yIndex++) {
            this.data.z[yIndex][xIndex] += intensity * this.shape.fct(this.data.x[xIndex] - position.x, this.data.y[yIndex] - position.y);
          }
        }

        return this;
      }

      getSpectrum(options = {}) {
        if (typeof options === 'boolean') {
          options = {
            copy: options
          };
        }

        const {
          copy = true
        } = options;
        let minMaxZ = getMinMax(this.data.z);
        return {
          minX: this.from.x,
          maxX: this.to.x,
          maxY: this.to.y,
          minY: this.from.y,
          minZ: minMaxZ.min,
          maxZ: minMaxZ.max,
          z: copy ? this.data.z.slice() : this.data.z
        };
      }

      reset() {
        const spectrum = this.data;

        for (const axis of axis2D) {
          for (let i = 0; i < this.nbPoints[axis]; i++) {
            spectrum[axis][i] = this.from[axis] + i * this.interval[axis];
          }
        }

        for (let row of spectrum.z) {
          for (let j = 0; j < row.length; j++) {
            row[j] = 0;
          }
        }

        return this;
      }

    }
    function generateSpectrum2D(peaks, options = {}) {
      const {
        generator: generatorOptions,
        peaks: addPeaksOptions
      } = options;
      const generator = new Spectrum2DGenerator(generatorOptions);
      generator.addPeaks(peaks, addPeaksOptions);
      return generator.getSpectrum();
    }

    function ensureXYNumber(input) {
      let result = typeof input !== 'object' ? {
        x: input,
        y: input
      } : input;
      return result;
    }

    function calculeIntervals(from, to, nbPoints) {
      return {
        x: (to.x - from.x) / (nbPoints.x - 1),
        y: (to.y - from.y) / (nbPoints.y - 1)
      };
    }

    function assertInteger(value, name) {
      if (!Number.isInteger(value)) {
        throw new TypeError(`${name} option must be an integer`);
      }
    }

    function assertNumber(value, name) {
      if (!Number.isFinite(value)) {
        throw new TypeError(`${name} option must be a number`);
      }
    }

    function createMatrix(nbPoints) {
      const zMatrix = new Array(nbPoints.y);

      for (let i = 0; i < nbPoints.y; i++) {
        zMatrix[i] = new Float64Array(nbPoints.x);
      }

      return zMatrix;
    }

    exports.Spectrum2DGenerator = Spectrum2DGenerator;
    exports.SpectrumGenerator = SpectrumGenerator;
    exports.generateSpectrum = generateSpectrum;
    exports.generateSpectrum2D = generateSpectrum2D;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=spectrum-generator.js.map
