/**
 * oan-data - Project that allows to deal with Oil Absorption Number (OAN)
 * @version v0.1.0
 * @link https://github.com/cheminfo/oan-data#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OANData = {}));
}(this, (function (exports) { 'use strict';

  const toString = Object.prototype.toString;
  function isAnyArray(object) {
    return toString.call(object).endsWith('Array]');
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function getAugmentedNamespace(n) {
  	if (n.__esModule) return n;
  	var a = Object.defineProperty({}, '__esModule', {value: true});
  	Object.keys(n).forEach(function (k) {
  		var d = Object.getOwnPropertyDescriptor(n, k);
  		Object.defineProperty(a, k, d.get ? d : {
  			enumerable: true,
  			get: function () {
  				return n[k];
  			}
  		});
  	});
  	return a;
  }

  function createCommonjsModule(fn) {
    var module = { exports: {} };
  	return fn(module, module.exports), module.exports;
  }

  function commonjsRequire (target) {
  	throw new Error('Could not dynamically require "' + target + '". Please configure the dynamicRequireTargets option of @rollup/plugin-commonjs appropriately for this require call to behave properly.');
  }

  var medianQuickselect_min = createCommonjsModule(function (module) {
    (function () {
      function a(d) {
        for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
          if (f <= e) return d[j];
          if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

          for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
            do h++; while (d[e] > d[h]);

            do i--; while (d[i] > d[e]);

            if (i < h) break;
            b(d, h, i);
          }

          b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
        }
      }

      var b = function b(d, e, f) {
        var _ref;

        return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
      },
          c = function c(d, e) {
        return ~~((d + e) / 2);
      };

      module.exports ? module.exports = a : window.median = a;
    })();
  });

  function median(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    return medianQuickselect_min(input.slice());
  }

  /**
   * This function xAdd the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xAdd(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + array2[i];
      }
    }

    return array3;
  }

  /**
   * This function xMultiply the first array by the second array or a constant value to each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Float64Array}
   */

  function xMultiply(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Float64Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * array2[i];
      }
    }

    return array3;
  }

  /**
   * This function divide the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array<Number>|Number} array2
   * @return {Array}
   */

  function xDivide(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / array2[i];
      }
    }

    return array3;
  }

  function _typeof(obj) {
    "@babel/helpers - typeof";

    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
      _typeof = function (obj) {
        return typeof obj;
      };
    } else {
      _typeof = function (obj) {
        return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
      };
    }

    return _typeof(obj);
  }
  /**
   * Fill an array with sequential numbers
   * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
   * @param {object} [options={}]
   * @param {number} [options.from=0] - first value in the array
   * @param {number} [options.to=10] - last value in the array
   * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
   * @param {number} [options.step] - if not provided calculated from size
   * @return {Array<number>}
   */


  function sequentialFill() {
    var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (_typeof(input) === 'object' && !isAnyArray(input)) {
      options = input;
      input = [];
    }

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    var _options = options,
        _options$from = _options.from,
        from = _options$from === void 0 ? 0 : _options$from,
        _options$to = _options.to,
        to = _options$to === void 0 ? 10 : _options$to,
        _options$size = _options.size,
        size = _options$size === void 0 ? input.length : _options$size,
        step = _options.step;

    if (size !== 0 && step) {
      throw new Error('step is defined by the array size');
    }

    if (!size) {
      if (step) {
        size = Math.floor((to - from) / step) + 1;
      } else {
        size = to - from + 1;
      }
    }

    if (!step && size) {
      step = (to - from) / (size - 1);
    }

    if (Array.isArray(input)) {
      // only works with normal array
      input.length = 0;

      for (var i = 0; i < size; i++) {
        input.push(from);
        from += step;
      }
    } else {
      if (input.length !== size) {
        throw new Error('sequentialFill typed array must have the correct length');
      }

      for (var _i = 0; _i < size; _i++) {
        input[_i] = from;
        from += step;
      }
    }

    return input;
  }

  /**
   * Returns true if x is monotone
   * @param {Array} array
   * @return {boolean}
   */
  function xIsMonotone(array) {
    if (array.length <= 2) {
      return true;
    }

    if (array[0] === array[1]) {
      // maybe a constant series
      for (let i = 1; i < array.length - 1; i++) {
        if (array[i] !== array[i + 1]) return false;
      }

      return true;
    }

    if (array[0] < array[array.length - 1]) {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] >= array[i + 1]) return false;
      }
    } else {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] <= array[i + 1]) return false;
      }
    }

    return true;
  }

  function sum(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += input[i];
    }

    return sumValue;
  }

  function mean(input) {
    return sum(input) / input.length;
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  /**
   * This function xSubtract the first array by the second array or a constant value from each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xSubtract(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - array2[i];
      }
    }

    return array3;
  }

  /**
   * Throw an error in no an object of x,y arrays
   * @param {DataXY} [data={}]
   */

  function xyCheck(data = {}) {
    if (!isAnyArray(data.x) || !isAnyArray(data.y)) {
      throw new Error('Data must be an object of x and y arrays');
    }

    if (data.x.length !== data.y.length) {
      throw new Error('The x and y arrays mush have the same length');
    }
  }

  /**
   * Filters x,y values to allow strictly growing values in x axis.
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {DataXY}
   */

  function xyEnsureGrowingX(data = {}) {
    xyCheck(data);
    const x = Array.from(data.x);
    const y = Array.from(data.y);
    let prevX = -Infinity;
    let ansX = [];
    let ansY = [];

    for (let index = 0; index < x.length; index++) {
      if (prevX < x[index]) {
        ansX.push(x[index]);
        ansY.push(y[index]);
        prevX = x[index];
      }
    }

    return {
      x: ansX,
      y: ansY
    };
  }

  /**
   * Filter out all the points for which x <= 0. Useful to display log scale data
   * @param {DataXY} [data={}]
   * @return {{x:[],y:[]}} An object with the filtered data
   */

  function xyFilterXPositive(data = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    const newX = [];
    const newY = [];

    for (let i = 0; i < x.length; i++) {
      if (x[i] > 0) {
        newX.push(x[i]);
        newY.push(y[i]);
      }
    }

    return {
      x: newX,
      y: newY
    };
  }

  function rescale(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    var currentMin = min(input);
    var currentMax = max(input);

    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }

    var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }

    var factor = (maxValue - minValue) / (currentMax - currentMin);

    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }

    return output;
  }

  const indent = ' '.repeat(2);
  const indentData = ' '.repeat(4);
  function inspectMatrix() {
    return inspectMatrixWithOptions(this);
  }
  function inspectMatrixWithOptions(matrix, options = {}) {
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8
    } = options;
    return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
  }

  function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];

    for (let i = 0; i < maxI; i++) {
      let line = [];

      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber(matrix.get(i, j), maxNumSize));
      }

      result.push(`${line.join(' ')}`);
    }

    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }

    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }

    return result.join(`\n${indentData}`);
  }

  function formatNumber(num, maxNumSize) {
    const numStr = String(num);

    if (numStr.length <= maxNumSize) {
      return numStr.padEnd(maxNumSize, ' ');
    }

    const precise = num.toPrecision(maxNumSize - 2);

    if (precise.length <= maxNumSize) {
      return precise;
    }

    const exponential = num.toExponential(maxNumSize - 2);
    const eIndex = exponential.indexOf('e');
    const e = exponential.slice(eIndex);
    return exponential.slice(0, maxNumSize - e.length) + e;
  }

  function installMathOperations(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };

    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };

    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };

    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };

    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;

    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };

    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };

    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;

    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };

    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };

    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;

    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };

    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };

    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;

    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };

    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };

    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };

    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };

    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };

    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };

    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };

    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };

    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };

    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };

    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };

    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };

    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };

    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };

    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };

    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };

    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };

    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };

    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };

    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };

    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };

    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };

    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };

    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };

    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };

    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };

    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };

    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };

    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };

    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };

    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };

    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };

    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };

    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };

    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };

    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };

    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };

    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };

    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };

    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };

    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };

    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }

      return this;
    };

    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }

      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }
  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */

  function checkColumnIndex(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkRowVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }

    return vector;
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkColumnVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }

    return vector;
  }
  function checkIndices(matrix, rowIndices, columnIndices) {
    return {
      row: checkRowIndices(matrix, rowIndices),
      column: checkColumnIndices(matrix, columnIndices)
    };
  }
  function checkRowIndices(matrix, rowIndices) {
    if (typeof rowIndices !== 'object') {
      throw new TypeError('unexpected type for row indices');
    }

    let rowOut = rowIndices.some(r => {
      return r < 0 || r >= matrix.rows;
    });

    if (rowOut) {
      throw new RangeError('row indices are out of range');
    }

    if (!Array.isArray(rowIndices)) rowIndices = Array.from(rowIndices);
    return rowIndices;
  }
  function checkColumnIndices(matrix, columnIndices) {
    if (typeof columnIndices !== 'object') {
      throw new TypeError('unexpected type for column indices');
    }

    let columnOut = columnIndices.some(c => {
      return c < 0 || c >= matrix.columns;
    });

    if (columnOut) {
      throw new RangeError('column indices are out of range');
    }

    if (!Array.isArray(columnIndices)) columnIndices = Array.from(columnIndices);
    return columnIndices;
  }
  function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }

    checkNumber('startRow', startRow);
    checkNumber('endRow', endRow);
    checkNumber('startColumn', startColumn);
    checkNumber('endColumn', endColumn);

    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray(length, value = 0) {
    let array = [];

    for (let i = 0; i < length; i++) {
      array.push(value);
    }

    return array;
  }

  function checkNumber(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }

  function checkNonEmpty(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow(matrix) {
    let sum = newArray(matrix.rows);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumByColumn(matrix) {
    let sum = newArray(matrix.columns);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumAll(matrix) {
    let v = 0;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }

    return v;
  }
  function productByRow(matrix) {
    let sum = newArray(matrix.rows, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productByColumn(matrix) {
    let sum = newArray(matrix.columns, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productAll(matrix) {
    let v = 1;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }

    return v;
  }
  function varianceByRow(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }

    return variance;
  }
  function varianceByColumn(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }

    return variance;
  }
  function varianceAll(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;

    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }

    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow(matrix) {
    const scale = [];

    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;

      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByRow(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn(matrix) {
    const scale = [];

    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;

      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByColumn(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;

    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }

    return Math.sqrt(sum);
  }
  function scaleAll(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;

      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }

      let newMatrix = new Matrix(newRows, newColumns);

      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }

      return newMatrix;
    }

    static rowVector(newData) {
      let vector = new Matrix(1, newData.length);

      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }

      return vector;
    }

    static columnVector(newData) {
      let vector = new Matrix(newData.length, 1);

      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }

      return vector;
    }

    static zeros(rows, columns) {
      return new Matrix(rows, columns);
    }

    static ones(rows, columns) {
      return new Matrix(rows, columns).fill(1);
    }

    static rand(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        random = Math.random
      } = options;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }

      return matrix;
    }

    static randInt(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }

      return matrix;
    }

    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }

      return matrix;
    }

    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }

      return matrix;
    }

    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static checkMatrix(value) {
      return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
    }

    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }

    get size() {
      return this.rows * this.columns;
    }

    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }

      return this;
    }

    to1DArray() {
      let array = [];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }

      return array;
    }

    to2DArray() {
      let copy = [];

      for (let i = 0; i < this.rows; i++) {
        copy.push([]);

        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }

      return copy;
    }

    toJSON() {
      return this.to2DArray();
    }

    isRowVector() {
      return this.rows === 1;
    }

    isColumnVector() {
      return this.columns === 1;
    }

    isVector() {
      return this.rows === 1 || this.columns === 1;
    }

    isSquare() {
      return this.rows === this.columns;
    }

    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }

    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }

        return true;
      }

      return false;
    }

    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;

      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }

        i++;
      }

      return isEchelonForm;
    }

    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;

      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }

        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }

        i++;
      }

      return isReducedEchelonForm;
    }

    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;

      while (h < result.rows && k < result.columns) {
        let iMax = h;

        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }

        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);

          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }

          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);

            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }

          h++;
          k++;
        }
      }

      return result;
    }

    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;

      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;

          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }

          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);

            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }

          h--;
        }
      }

      return result;
    }

    set() {
      throw new Error('set method is unimplemented');
    }

    get() {
      throw new Error('get method is unimplemented');
    }

    repeat(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        rows = 1,
        columns = 1
      } = options;

      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }

      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }

      let matrix = new Matrix(this.rows * rows, this.columns * columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }

      return matrix;
    }

    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }

      return this;
    }

    neg() {
      return this.mulS(-1);
    }

    getRow(index) {
      checkRowIndex(this, index);
      let row = [];

      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }

      return row;
    }

    getRowVector(index) {
      return Matrix.rowVector(this.getRow(index));
    }

    setRow(index, array) {
      checkRowIndex(this, index);
      array = checkRowVector(this, array);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }

      return this;
    }

    swapRows(row1, row2) {
      checkRowIndex(this, row1);
      checkRowIndex(this, row2);

      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }

      return this;
    }

    getColumn(index) {
      checkColumnIndex(this, index);
      let column = [];

      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }

      return column;
    }

    getColumnVector(index) {
      return Matrix.columnVector(this.getColumn(index));
    }

    setColumn(index, array) {
      checkColumnIndex(this, index);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }

      return this;
    }

    swapColumns(column1, column2) {
      checkColumnIndex(this, column1);
      checkColumnIndex(this, column2);

      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }

      return this;
    }

    addRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }

      return this;
    }

    subRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }

      return this;
    }

    mulRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }

      return this;
    }

    divRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }

      return this;
    }

    addColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }

      return this;
    }

    subColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }

      return this;
    }

    mulColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }

      return this;
    }

    divColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }

      return this;
    }

    mulRow(index, value) {
      checkRowIndex(this, index);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }

      return this;
    }

    mulColumn(index, value) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }

      return this;
    }

    max() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    maxIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    min() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    minIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    maxRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    maxRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    minRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    minRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    maxColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    maxColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    minColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    minColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];

      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }

      return diag;
    }

    norm(type = 'frobenius') {
      let result = 0;

      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }

        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }

    cumulativeSum() {
      let sum = 0;

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }

      return this;
    }

    dot(vector2) {
      if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();

      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }

      let dot = 0;

      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }

      return dot;
    }

    mmul(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix(m, p);
      let Bcolj = new Float64Array(n);

      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }

        for (let i = 0; i < m; i++) {
          let s = 0;

          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }

          result.set(i, j, s);
        }
      }

      return result;
    }

    strassen2x2(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1); // Compute intermediate values.

      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }

    strassen3x3(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }

    mmulStrassen(y) {
      y = Matrix.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;

      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      } // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.


      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;

        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      } // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.


      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c); // Our recursive multiplication function.

      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        } // Apply dynamic padding.


        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }

        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

        let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

        let c11 = AbstractMatrix.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix.add(m3, m5);
        let c21 = AbstractMatrix.add(m2, m4);
        let c22 = AbstractMatrix.sub(m1, m2);
        c22.add(m3);
        c22.add(m6); // Crop output to the desired size (undo dynamic padding).

        let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }

      return blockMult(x, y, r, c);
    }

    scaleRows(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);

        if (row.length > 0) {
          rescale(row, {
            min,
            max,
            output: row
          });
        }

        newMatrix.setRow(i, row);
      }

      return newMatrix;
    }

    scaleColumns(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);

        if (column.length) {
          rescale(column, {
            min: min,
            max: max,
            output: column
          });
        }

        newMatrix.setColumn(i, column);
      }

      return newMatrix;
    }

    flipRows() {
      const middle = Math.ceil(this.columns / 2);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }

      return this;
    }

    flipColumns() {
      const middle = Math.ceil(this.rows / 2);

      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }

      return this;
    }

    kroneckerProduct(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix(m * p, n * q);

      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }

      return result;
    }

    kroneckerSum(other) {
      other = Matrix.checkMatrix(other);

      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }

      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix.eye(n, n));
      let IxB = Matrix.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }

    transpose() {
      let result = new Matrix(this.columns, this.rows);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }

      return result;
    }

    sortRows(compareFunction = compareNumbers) {
      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }

      return this;
    }

    sortColumns(compareFunction = compareNumbers) {
      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }

      return this;
    }

    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }

      return newMatrix;
    }

    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;

      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }

          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }

      return newMatrix;
    }

    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;

      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }

          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }

      return newMatrix;
    }

    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix.checkMatrix(matrix);

      if (matrix.isEmpty()) {
        return this;
      }

      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange(this, startRow, endRow, startColumn, endColumn);

      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }

      return this;
    }

    selection(rowIndices, columnIndices) {
      let indices = checkIndices(this, rowIndices, columnIndices);
      let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

      for (let i = 0; i < indices.row.length; i++) {
        let rowIndex = indices.row[i];

        for (let j = 0; j < indices.column.length; j++) {
          let columnIndex = indices.column[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }

      return newMatrix;
    }

    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;

      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }

      return trace;
    }

    clone() {
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }

      return newMatrix;
    }

    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow(this);

        case 'column':
          return sumByColumn(this);

        case undefined:
          return sumAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    product(by) {
      switch (by) {
        case 'row':
          return productByRow(this);

        case 'column':
          return productByColumn(this);

        case undefined:
          return productAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    mean(by) {
      const sum = this.sum(by);

      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }

            return sum;
          }

        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }

            return sum;
          }

        case undefined:
          return sum / this.size;

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    variance(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;

      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByRow(this, unbiased, mean);
          }

        case 'column':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByColumn(this, unbiased, mean);
          }

        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }

            return varianceAll(this, unbiased, mean);
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      const variance = this.variance(by, options);

      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }

        return variance;
      }
    }

    center(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        center = this.mean(by)
      } = options;

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByRow(this, center);
            return this;
          }

        case 'column':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByColumn(this, center);
            return this;
          }

        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }

            centerAll(this, center);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    scale(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      let scale = options.scale;

      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByRow(this, scale);
            return this;
          }

        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByColumn(this, scale);
            return this;
          }

        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }

            scaleAll(this, scale);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    toString(options) {
      return inspectMatrixWithOptions(this, options);
    }

  }
  AbstractMatrix.prototype.klass = 'Matrix';

  if (typeof Symbol !== 'undefined') {
    AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
  }

  function compareNumbers(a, b) {
    return a - b;
  } // Synonyms


  AbstractMatrix.random = AbstractMatrix.rand;
  AbstractMatrix.randomInt = AbstractMatrix.randInt;
  AbstractMatrix.diagonal = AbstractMatrix.diag;
  AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
  AbstractMatrix.identity = AbstractMatrix.eye;
  AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
  AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
  class Matrix extends AbstractMatrix {
    constructor(nRows, nColumns) {
      super();

      if (Matrix.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];

        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (Array.isArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;

        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }

        this.data = [];

        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }

          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }

      this.rows = nRows;
      this.columns = nColumns;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

    removeRow(index) {
      checkRowIndex(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }

    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }

      checkRowIndex(this, index, true);
      array = Float64Array.from(checkRowVector(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }

    removeColumn(index) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);

        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }

        this.data[i] = newRow;
      }

      this.columns -= 1;
      return this;
    }

    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }

      checkColumnIndex(this, index, true);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;

        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        newRow[j++] = array[i];

        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }

        this.data[i] = newRow;
      }

      this.columns += 1;
      return this;
    }

  }
  installMathOperations(AbstractMatrix, Matrix);

  class BaseView extends AbstractMatrix {
    constructor(matrix, rows, columns) {
      super();
      this.matrix = matrix;
      this.rows = rows;
      this.columns = columns;
    }

  }

  class MatrixTransposeView extends BaseView {
    constructor(matrix) {
      super(matrix, matrix.columns, matrix.rows);
    }

    set(rowIndex, columnIndex, value) {
      this.matrix.set(columnIndex, rowIndex, value);
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.matrix.get(columnIndex, rowIndex);
    }

  }

  class WrapperMatrix2D extends AbstractMatrix {
    constructor(data) {
      super();
      this.data = data;
      this.rows = data.length;
      this.columns = data[0].length;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

  }

  class LuDecomposition {
    constructor(matrix) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      let lu = matrix.clone();
      let rows = lu.rows;
      let columns = lu.columns;
      let pivotVector = new Float64Array(rows);
      let pivotSign = 1;
      let i, j, k, p, s, t, v;
      let LUcolj, kmax;

      for (i = 0; i < rows; i++) {
        pivotVector[i] = i;
      }

      LUcolj = new Float64Array(rows);

      for (j = 0; j < columns; j++) {
        for (i = 0; i < rows; i++) {
          LUcolj[i] = lu.get(i, j);
        }

        for (i = 0; i < rows; i++) {
          kmax = Math.min(i, j);
          s = 0;

          for (k = 0; k < kmax; k++) {
            s += lu.get(i, k) * LUcolj[k];
          }

          LUcolj[i] -= s;
          lu.set(i, j, LUcolj[i]);
        }

        p = j;

        for (i = j + 1; i < rows; i++) {
          if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
            p = i;
          }
        }

        if (p !== j) {
          for (k = 0; k < columns; k++) {
            t = lu.get(p, k);
            lu.set(p, k, lu.get(j, k));
            lu.set(j, k, t);
          }

          v = pivotVector[p];
          pivotVector[p] = pivotVector[j];
          pivotVector[j] = v;
          pivotSign = -pivotSign;
        }

        if (j < rows && lu.get(j, j) !== 0) {
          for (i = j + 1; i < rows; i++) {
            lu.set(i, j, lu.get(i, j) / lu.get(j, j));
          }
        }
      }

      this.LU = lu;
      this.pivotVector = pivotVector;
      this.pivotSign = pivotSign;
    }

    isSingular() {
      let data = this.LU;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        if (data.get(j, j) === 0) {
          return true;
        }
      }

      return false;
    }

    solve(value) {
      value = Matrix.checkMatrix(value);
      let lu = this.LU;
      let rows = lu.rows;

      if (rows !== value.rows) {
        throw new Error('Invalid matrix dimensions');
      }

      if (this.isSingular()) {
        throw new Error('LU matrix is singular');
      }

      let count = value.columns;
      let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
      let columns = lu.columns;
      let i, j, k;

      for (k = 0; k < columns; k++) {
        for (i = k + 1; i < columns; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      for (k = columns - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / lu.get(k, k));
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      return X;
    }

    get determinant() {
      let data = this.LU;

      if (!data.isSquare()) {
        throw new Error('Matrix must be square');
      }

      let determinant = this.pivotSign;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        determinant *= data.get(j, j);
      }

      return determinant;
    }

    get lowerTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i > j) {
            X.set(i, j, data.get(i, j));
          } else if (i === j) {
            X.set(i, j, 1);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get upperTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i <= j) {
            X.set(i, j, data.get(i, j));
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get pivotPermutationVector() {
      return Array.from(this.pivotVector);
    }

  }

  function hypotenuse(a, b) {
    let r = 0;

    if (Math.abs(a) > Math.abs(b)) {
      r = b / a;
      return Math.abs(a) * Math.sqrt(1 + r * r);
    }

    if (b !== 0) {
      r = a / b;
      return Math.abs(b) * Math.sqrt(1 + r * r);
    }

    return 0;
  }

  class QrDecomposition {
    constructor(value) {
      value = WrapperMatrix2D.checkMatrix(value);
      let qr = value.clone();
      let m = value.rows;
      let n = value.columns;
      let rdiag = new Float64Array(n);
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        let nrm = 0;

        for (i = k; i < m; i++) {
          nrm = hypotenuse(nrm, qr.get(i, k));
        }

        if (nrm !== 0) {
          if (qr.get(k, k) < 0) {
            nrm = -nrm;
          }

          for (i = k; i < m; i++) {
            qr.set(i, k, qr.get(i, k) / nrm);
          }

          qr.set(k, k, qr.get(k, k) + 1);

          for (j = k + 1; j < n; j++) {
            s = 0;

            for (i = k; i < m; i++) {
              s += qr.get(i, k) * qr.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < m; i++) {
              qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
            }
          }
        }

        rdiag[k] = -nrm;
      }

      this.QR = qr;
      this.Rdiag = rdiag;
    }

    solve(value) {
      value = Matrix.checkMatrix(value);
      let qr = this.QR;
      let m = qr.rows;

      if (value.rows !== m) {
        throw new Error('Matrix row dimensions must agree');
      }

      if (!this.isFullRank()) {
        throw new Error('Matrix is rank deficient');
      }

      let count = value.columns;
      let X = value.clone();
      let n = qr.columns;
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        for (j = 0; j < count; j++) {
          s = 0;

          for (i = k; i < m; i++) {
            s += qr.get(i, k) * X.get(i, j);
          }

          s = -s / qr.get(k, k);

          for (i = k; i < m; i++) {
            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
          }
        }
      }

      for (k = n - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / this.Rdiag[k]);
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
          }
        }
      }

      return X.subMatrix(0, n - 1, 0, count - 1);
    }

    isFullRank() {
      let columns = this.QR.columns;

      for (let i = 0; i < columns; i++) {
        if (this.Rdiag[i] === 0) {
          return false;
        }
      }

      return true;
    }

    get upperTriangularMatrix() {
      let qr = this.QR;
      let n = qr.columns;
      let X = new Matrix(n, n);
      let i, j;

      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          if (i < j) {
            X.set(i, j, qr.get(i, j));
          } else if (i === j) {
            X.set(i, j, this.Rdiag[i]);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get orthogonalMatrix() {
      let qr = this.QR;
      let rows = qr.rows;
      let columns = qr.columns;
      let X = new Matrix(rows, columns);
      let i, j, k, s;

      for (k = columns - 1; k >= 0; k--) {
        for (i = 0; i < rows; i++) {
          X.set(i, k, 0);
        }

        X.set(k, k, 1);

        for (j = k; j < columns; j++) {
          if (qr.get(k, k) !== 0) {
            s = 0;

            for (i = k; i < rows; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < rows; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
      }

      return X;
    }

  }

  class SingularValueDecomposition {
    constructor(value, options = {}) {
      value = WrapperMatrix2D.checkMatrix(value);

      if (value.isEmpty()) {
        throw new Error('Matrix must be non-empty');
      }

      let m = value.rows;
      let n = value.columns;
      const {
        computeLeftSingularVectors = true,
        computeRightSingularVectors = true,
        autoTranspose = false
      } = options;
      let wantu = Boolean(computeLeftSingularVectors);
      let wantv = Boolean(computeRightSingularVectors);
      let swapped = false;
      let a;

      if (m < n) {
        if (!autoTranspose) {
          a = value.clone(); // eslint-disable-next-line no-console

          console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
        } else {
          a = value.transpose();
          m = a.rows;
          n = a.columns;
          swapped = true;
          let aux = wantu;
          wantu = wantv;
          wantv = aux;
        }
      } else {
        a = value.clone();
      }

      let nu = Math.min(m, n);
      let ni = Math.min(m + 1, n);
      let s = new Float64Array(ni);
      let U = new Matrix(m, nu);
      let V = new Matrix(n, n);
      let e = new Float64Array(n);
      let work = new Float64Array(m);
      let si = new Float64Array(ni);

      for (let i = 0; i < ni; i++) si[i] = i;

      let nct = Math.min(m - 1, n);
      let nrt = Math.max(0, Math.min(n - 2, m));
      let mrc = Math.max(nct, nrt);

      for (let k = 0; k < mrc; k++) {
        if (k < nct) {
          s[k] = 0;

          for (let i = k; i < m; i++) {
            s[k] = hypotenuse(s[k], a.get(i, k));
          }

          if (s[k] !== 0) {
            if (a.get(k, k) < 0) {
              s[k] = -s[k];
            }

            for (let i = k; i < m; i++) {
              a.set(i, k, a.get(i, k) / s[k]);
            }

            a.set(k, k, a.get(k, k) + 1);
          }

          s[k] = -s[k];
        }

        for (let j = k + 1; j < n; j++) {
          if (k < nct && s[k] !== 0) {
            let t = 0;

            for (let i = k; i < m; i++) {
              t += a.get(i, k) * a.get(i, j);
            }

            t = -t / a.get(k, k);

            for (let i = k; i < m; i++) {
              a.set(i, j, a.get(i, j) + t * a.get(i, k));
            }
          }

          e[j] = a.get(k, j);
        }

        if (wantu && k < nct) {
          for (let i = k; i < m; i++) {
            U.set(i, k, a.get(i, k));
          }
        }

        if (k < nrt) {
          e[k] = 0;

          for (let i = k + 1; i < n; i++) {
            e[k] = hypotenuse(e[k], e[i]);
          }

          if (e[k] !== 0) {
            if (e[k + 1] < 0) {
              e[k] = 0 - e[k];
            }

            for (let i = k + 1; i < n; i++) {
              e[i] /= e[k];
            }

            e[k + 1] += 1;
          }

          e[k] = -e[k];

          if (k + 1 < m && e[k] !== 0) {
            for (let i = k + 1; i < m; i++) {
              work[i] = 0;
            }

            for (let i = k + 1; i < m; i++) {
              for (let j = k + 1; j < n; j++) {
                work[i] += e[j] * a.get(i, j);
              }
            }

            for (let j = k + 1; j < n; j++) {
              let t = -e[j] / e[k + 1];

              for (let i = k + 1; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * work[i]);
              }
            }
          }

          if (wantv) {
            for (let i = k + 1; i < n; i++) {
              V.set(i, k, e[i]);
            }
          }
        }
      }

      let p = Math.min(n, m + 1);

      if (nct < n) {
        s[nct] = a.get(nct, nct);
      }

      if (m < p) {
        s[p - 1] = 0;
      }

      if (nrt + 1 < p) {
        e[nrt] = a.get(nrt, p - 1);
      }

      e[p - 1] = 0;

      if (wantu) {
        for (let j = nct; j < nu; j++) {
          for (let i = 0; i < m; i++) {
            U.set(i, j, 0);
          }

          U.set(j, j, 1);
        }

        for (let k = nct - 1; k >= 0; k--) {
          if (s[k] !== 0) {
            for (let j = k + 1; j < nu; j++) {
              let t = 0;

              for (let i = k; i < m; i++) {
                t += U.get(i, k) * U.get(i, j);
              }

              t = -t / U.get(k, k);

              for (let i = k; i < m; i++) {
                U.set(i, j, U.get(i, j) + t * U.get(i, k));
              }
            }

            for (let i = k; i < m; i++) {
              U.set(i, k, -U.get(i, k));
            }

            U.set(k, k, 1 + U.get(k, k));

            for (let i = 0; i < k - 1; i++) {
              U.set(i, k, 0);
            }
          } else {
            for (let i = 0; i < m; i++) {
              U.set(i, k, 0);
            }

            U.set(k, k, 1);
          }
        }
      }

      if (wantv) {
        for (let k = n - 1; k >= 0; k--) {
          if (k < nrt && e[k] !== 0) {
            for (let j = k + 1; j < n; j++) {
              let t = 0;

              for (let i = k + 1; i < n; i++) {
                t += V.get(i, k) * V.get(i, j);
              }

              t = -t / V.get(k + 1, k);

              for (let i = k + 1; i < n; i++) {
                V.set(i, j, V.get(i, j) + t * V.get(i, k));
              }
            }
          }

          for (let i = 0; i < n; i++) {
            V.set(i, k, 0);
          }

          V.set(k, k, 1);
        }
      }

      let pp = p - 1;
      let eps = Number.EPSILON;

      while (p > 0) {
        let k, kase;

        for (k = p - 2; k >= -1; k--) {
          if (k === -1) {
            break;
          }

          const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));

          if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
            e[k] = 0;
            break;
          }
        }

        if (k === p - 2) {
          kase = 4;
        } else {
          let ks;

          for (ks = p - 1; ks >= k; ks--) {
            if (ks === k) {
              break;
            }

            let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);

            if (Math.abs(s[ks]) <= eps * t) {
              s[ks] = 0;
              break;
            }
          }

          if (ks === k) {
            kase = 3;
          } else if (ks === p - 1) {
            kase = 1;
          } else {
            kase = 2;
            k = ks;
          }
        }

        k++;

        switch (kase) {
          case 1:
            {
              let f = e[p - 2];
              e[p - 2] = 0;

              for (let j = p - 2; j >= k; j--) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;

                if (j !== k) {
                  f = -sn * e[j - 1];
                  e[j - 1] = cs * e[j - 1];
                }

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                    V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                    V.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 2:
            {
              let f = e[k - 1];
              e[k - 1] = 0;

              for (let j = k; j < p; j++) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;
                f = -sn * e[j];
                e[j] = cs * e[j];

                if (wantu) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                    U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                    U.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 3:
            {
              const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
              const sp = s[p - 1] / scale;
              const spm1 = s[p - 2] / scale;
              const epm1 = e[p - 2] / scale;
              const sk = s[k] / scale;
              const ek = e[k] / scale;
              const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
              const c = sp * epm1 * (sp * epm1);
              let shift = 0;

              if (b !== 0 || c !== 0) {
                if (b < 0) {
                  shift = 0 - Math.sqrt(b * b + c);
                } else {
                  shift = Math.sqrt(b * b + c);
                }

                shift = c / (b + shift);
              }

              let f = (sk + sp) * (sk - sp) + shift;
              let g = sk * ek;

              for (let j = k; j < p - 1; j++) {
                let t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                let cs = f / t;
                let sn = g / t;

                if (j !== k) {
                  e[j - 1] = t;
                }

                f = cs * s[j] + sn * e[j];
                e[j] = cs * e[j] - sn * s[j];
                g = sn * s[j + 1];
                s[j + 1] = cs * s[j + 1];

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                    V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                    V.set(i, j, t);
                  }
                }

                t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                cs = f / t;
                sn = g / t;
                s[j] = t;
                f = cs * e[j] + sn * s[j + 1];
                s[j + 1] = -sn * e[j] + cs * s[j + 1];
                g = sn * e[j + 1];
                e[j + 1] = cs * e[j + 1];

                if (wantu && j < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                    U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                    U.set(i, j, t);
                  }
                }
              }

              e[p - 2] = f;
              break;
            }

          case 4:
            {
              if (s[k] <= 0) {
                s[k] = s[k] < 0 ? -s[k] : 0;

                if (wantv) {
                  for (let i = 0; i <= pp; i++) {
                    V.set(i, k, -V.get(i, k));
                  }
                }
              }

              while (k < pp) {
                if (s[k] >= s[k + 1]) {
                  break;
                }

                let t = s[k];
                s[k] = s[k + 1];
                s[k + 1] = t;

                if (wantv && k < n - 1) {
                  for (let i = 0; i < n; i++) {
                    t = V.get(i, k + 1);
                    V.set(i, k + 1, V.get(i, k));
                    V.set(i, k, t);
                  }
                }

                if (wantu && k < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = U.get(i, k + 1);
                    U.set(i, k + 1, U.get(i, k));
                    U.set(i, k, t);
                  }
                }

                k++;
              }
              p--;
              break;
            }
          // no default
        }
      }

      if (swapped) {
        let tmp = V;
        V = U;
        U = tmp;
      }

      this.m = m;
      this.n = n;
      this.s = s;
      this.U = U;
      this.V = V;
    }

    solve(value) {
      let Y = value;
      let e = this.threshold;
      let scols = this.s.length;
      let Ls = Matrix.zeros(scols, scols);

      for (let i = 0; i < scols; i++) {
        if (Math.abs(this.s[i]) <= e) {
          Ls.set(i, i, 0);
        } else {
          Ls.set(i, i, 1 / this.s[i]);
        }
      }

      let U = this.U;
      let V = this.rightSingularVectors;
      let VL = V.mmul(Ls);
      let vrows = V.rows;
      let urows = U.rows;
      let VLU = Matrix.zeros(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < scols; k++) {
            sum += VL.get(i, k) * U.get(j, k);
          }

          VLU.set(i, j, sum);
        }
      }

      return VLU.mmul(Y);
    }

    solveForDiagonal(value) {
      return this.solve(Matrix.diag(value));
    }

    inverse() {
      let V = this.V;
      let e = this.threshold;
      let vrows = V.rows;
      let vcols = V.columns;
      let X = new Matrix(vrows, this.s.length);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < vcols; j++) {
          if (Math.abs(this.s[j]) > e) {
            X.set(i, j, V.get(i, j) / this.s[j]);
          }
        }
      }

      let U = this.U;
      let urows = U.rows;
      let ucols = U.columns;
      let Y = new Matrix(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < ucols; k++) {
            sum += X.get(i, k) * U.get(j, k);
          }

          Y.set(i, j, sum);
        }
      }

      return Y;
    }

    get condition() {
      return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
    }

    get norm2() {
      return this.s[0];
    }

    get rank() {
      let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
      let r = 0;
      let s = this.s;

      for (let i = 0, ii = s.length; i < ii; i++) {
        if (s[i] > tol) {
          r++;
        }
      }

      return r;
    }

    get diagonal() {
      return Array.from(this.s);
    }

    get threshold() {
      return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
    }

    get leftSingularVectors() {
      return this.U;
    }

    get rightSingularVectors() {
      return this.V;
    }

    get diagonalMatrix() {
      return Matrix.diag(this.s);
    }

  }

  function inverse(matrix, useSVD = false) {
    matrix = WrapperMatrix2D.checkMatrix(matrix);

    if (useSVD) {
      return new SingularValueDecomposition(matrix).inverse();
    } else {
      return solve(matrix, Matrix.eye(matrix.rows));
    }
  }
  function solve(leftHandSide, rightHandSide, useSVD = false) {
    leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
    rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);

    if (useSVD) {
      return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
    } else {
      return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
    }
  }

  function addStyle(serie, spectrum, options = {}) {
    let {
      color = '#A9A9A9',
      opacity = 1,
      lineWidth = 1
    } = options; // eslint-disable-next-line @typescript-eslint/prefer-regexp-exec

    if (color.match(/#[0-9A-F]{6}$/i)) {
      color = (color + (opacity * 255 >> 0).toString(16)).toUpperCase();
    } else {
      color = color.replace(/rgb ?\((.*)\)/, `rgba($1,${opacity})`);
    }

    serie.style = [{
      name: 'unselected',
      style: {
        line: {
          color,
          width: lineWidth,
          dash: 1
        }
      }
    }, {
      name: 'selected',
      style: {
        line: {
          color,
          width: lineWidth + 2,
          dash: 1
        }
      }
    }];
    serie.name = spectrum.label || spectrum.id;
  }

  const COLORS = ['#FFB300', '#803E75', '#FF6800', '#A6BDD7', '#C10020', '#CEA262', '#817066', '#007D34', '#F6768E', '#00538A', '#FF7A5C', '#53377A', '#FF8E00', '#B32851', '#F4C800', '#7F180D', '#93AA00', '#593315', '#F13A13', '#232C16'];

  /**
   * Generate a jsgraph chart format from an array of Analysis
   */

  function getJSGraph(analyses, options = {}) {
    const {
      colors = COLORS,
      opacities = [1],
      linesWidth = [1],
      selector = {},
      normalization,
      xAxis = {},
      yAxis = {}
    } = options;
    let series = [];
    let xLabel = '';
    let yLabel = '';

    for (let i = 0; i < analyses.length; i++) {
      const analysis = analyses[i];
      let serie = {};
      let currentData = analysis.getNormalizedSpectrum({
        selector,
        normalization
      });
      if (!currentData) continue;
      if (!xLabel) xLabel = currentData.variables.x.label;
      if (!yLabel) yLabel = currentData.variables.y.label;
      addStyle(serie, analysis, {
        color: colors[i % colors.length],
        opacity: opacities[i % opacities.length],
        lineWidth: linesWidth[i % linesWidth.length]
      });
      serie.data = {
        x: currentData.variables.x.data,
        y: currentData.variables.y.data
      };

      if (xAxis.logScale) {
        serie.data = xyFilterXPositive(serie.data);
      }

      series.push(serie);
    }

    return {
      axes: {
        x: {
          label: xLabel,
          unit: '',
          flipped: false,
          display: true,
          ...xAxis
        },
        y: {
          label: yLabel,
          unit: '',
          flipped: false,
          display: true,
          ...yAxis
        }
      },
      series
    };
  }

  function getNormalizationAnnotations(filter = {}, boundary = {
    y: {
      min: '0px',
      max: '2000px'
    }
  }) {
    let {
      exclusions = []
    } = filter;
    let annotations = [];
    exclusions = exclusions.filter(exclusion => !exclusion.ignore);
    annotations = exclusions.map(exclusion => {
      let annotation = {
        type: 'rect',
        position: [{
          x: exclusion.from,
          y: boundary.y.min
        }, {
          x: exclusion.to,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      };
      return annotation;
    });

    if (filter.from !== undefined) {
      annotations.push({
        type: 'rect',
        position: [{
          x: Number.MIN_SAFE_INTEGER,
          y: boundary.y.min
        }, {
          x: filter.from,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      });
    }

    if (filter.to !== undefined) {
      annotations.push({
        type: 'rect',
        position: [{
          x: filter.to,
          y: boundary.y.min
        }, {
          x: Number.MAX_SAFE_INTEGER,
          y: boundary.y.max
        }],
        strokeWidth: 0,
        fillColor: 'rgba(255,255,224,1)'
      });
    }

    return annotations;
  }

  function appendDistinctParameter(values, key, value) {
    if (!values[key]) {
      values[key] = {
        key,
        values: [],
        count: 0
      };
    }

    if (!values[key].values.includes(value)) {
      values[key].values.push(value);
    }

    values[key].count++;
  }

  function appendDistinctValue(values, key) {
    if (!values[key]) {
      values[key] = {
        key,
        count: 0
      };
    }

    values[key].count++;
  }

  class AnalysesManager {
    constructor() {
      this.analyses = [];
    }

    addAnalysis(analysis) {
      let index = this.getAnalysisIndex(analysis.id);

      if (index === undefined) {
        this.analyses.push(analysis);
      } else {
        this.analyses[index] = analysis;
      }
    }

    getAnalyses(options = {}) {
      const {
        ids
      } = options;
      let analyses = [];

      for (const analysis of this.analyses) {
        if (!ids || ids.includes(analysis.id)) {
          analyses.push(analysis);
        }
      }

      return analyses;
    }

    getSpectra() {
      const spectra = [];

      for (const analysis of this.analyses) {
        spectra.push(...analysis.spectra);
      }

      return spectra;
    }
    /**
     * Get an array of objects (key + count) of all the titles
     */


    getDistinctTitles() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.title) {
          appendDistinctValue(values, spectrum.title);
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the units
     */


    getDistinctUnits() {
      var _a;

      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.variables) {
          for (let key in spectrum.variables) {
            const units = (_a = spectrum.variables[key].units) === null || _a === void 0 ? void 0 : _a.replace(/\s+\[.*/, '');

            if (units) {
              appendDistinctValue(values, units);
            }
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the labels
     */


    getDistinctLabels() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.variables) {
          for (let key in spectrum.variables) {
            appendDistinctValue(values, spectrum.variables[key].label.replace(/\s+\[.*/, ''));
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the dataTypes
     */


    getDistinctDataTypes() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.dataType) {
          appendDistinctValue(values, spectrum.dataType);
        }
      }

      return Object.keys(values).map(key => values[key]);
    }
    /**
     * Get an array of objects (key + count) of all the meta
     */


    getDistinctMeta() {
      let values = {};

      for (let spectrum of this.getSpectra()) {
        if (spectrum.meta) {
          for (let key in spectrum.meta) {
            appendDistinctParameter(values, key, spectrum.meta[key]);
          }
        }
      }

      return Object.keys(values).map(key => values[key]);
    }

    removeAllAnalyses() {
      this.analyses.splice(0);
    }
    /**
     * Remove the analysis from the AnalysesManager for the specified id
     */


    removeAnalysis(id) {
      let index = this.getAnalysisIndex(id);
      if (index === undefined) return undefined;
      return this.analyses.splice(index, 1);
    }
    /**
     * Returns the index of the analysis in the analyses array
     */


    getAnalysisIndex(id) {
      if (!id) return undefined;

      for (let i = 0; i < this.analyses.length; i++) {
        let analysis = this.analyses[i];
        if (analysis.id === id) return i;
      }

      return undefined;
    }
    /**
     * Checks if the ID of an analysis exists in the AnalysesManager
     */


    includes(id) {
      const index = this.getAnalysisIndex(id);
      return index === undefined ? false : !isNaN(index);
    }

  }

  // Based on https://github.com/scijs/cholesky-solve

  /*
  The MIT License (MIT)

  Copyright (c) 2013 Eric Arnebäck

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
  */
  function ldlSymbolic(n
  /* A and L are n-by-n, where n >= 0 */
  , Ap
  /* input of size n + 1, not modified */
  , Ai
  /* input of size nz=Ap[n], not modified */
  , Lp
  /* output of size n + 1, not defined on input */
  , Parent
  /* output of size n, not defined on input */
  , Lnz
  /* output of size n, not defined on input */
  , Flag
  /* workspace of size n, not defn. on input or output */
  ) {
    let i, k, p, kk, p2;

    for (k = 0; k < n; k++) {
      /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
      Parent[k] = -1;
      /* parent of k is not yet known */

      Flag[k] = k;
      /* mark node k as visited */

      Lnz[k] = 0;
      /* count of nonzeros in column k of L */

      kk = k;
      /* kth original, or permuted, column */

      p2 = Ap[kk + 1];

      for (p = Ap[kk]; p < p2; p++) {
        /* A (i,k) is nonzero (original or permuted A) */
        i = Ai[p];

        if (i < k) {
          /* follow path from i to root of etree, stop at flagged node */
          for (; Flag[i] !== k; i = Parent[i]) {
            /* find parent of i if not yet determined */
            if (Parent[i] === -1) Parent[i] = k;
            Lnz[i]++;
            /* L (k,i) is nonzero */

            Flag[i] = k;
            /* mark i as visited */
          }
        }
      }
    }
    /* construct Lp index array from Lnz column counts */


    Lp[0] = 0;

    for (k = 0; k < n; k++) {
      Lp[k + 1] = Lp[k] + Lnz[k];
    }
  }

  function ldlNumeric(n
  /* A and L are n-by-n, where n >= 0 */
  , Ap
  /* input of size n+1, not modified */
  , Ai
  /* input of size nz=Ap[n], not modified */
  , Ax
  /* input of size nz=Ap[n], not modified */
  , Lp
  /* input of size n+1, not modified */
  , Parent
  /* input of size n, not modified */
  , Lnz
  /* output of size n, not defn. on input */
  , Li
  /* output of size lnz=Lp[n], not defined on input */
  , Lx
  /* output of size lnz=Lp[n], not defined on input */
  , D
  /* output of size n, not defined on input */
  , Y
  /* workspace of size n, not defn. on input or output */
  , Pattern
  /* workspace of size n, not defn. on input or output */
  , Flag
  /* workspace of size n, not defn. on input or output */
  ) {
    let yi, lKi;
    let i, k, p, kk, p2, len, top;

    for (k = 0; k < n; k++) {
      /* compute nonzero Pattern of kth row of L, in topological order */
      Y[k] = 0.0;
      /* Y(0:k) is now all zero */

      top = n;
      /* stack for pattern is empty */

      Flag[k] = k;
      /* mark node k as visited */

      Lnz[k] = 0;
      /* count of nonzeros in column k of L */

      kk = k;
      /* kth original, or permuted, column */

      p2 = Ap[kk + 1];

      for (p = Ap[kk]; p < p2; p++) {
        i = Ai[p];
        /* get A(i,k) */

        if (i <= k) {
          Y[i] += Ax[p];
          /* scatter A(i,k) into Y (sum duplicates) */

          for (len = 0; Flag[i] !== k; i = Parent[i]) {
            Pattern[len++] = i;
            /* L(k,i) is nonzero */

            Flag[i] = k;
            /* mark i as visited */
          }

          while (len > 0) Pattern[--top] = Pattern[--len];
        }
      }
      /* compute numerical values kth row of L (a sparse triangular solve) */


      D[k] = Y[k];
      /* get D(k,k) and clear Y(k) */

      Y[k] = 0.0;

      for (; top < n; top++) {
        i = Pattern[top];
        /* Pattern[top:n-1] is pattern of L(:,k) */

        yi = Y[i];
        /* get and clear Y(i) */

        Y[i] = 0.0;
        p2 = Lp[i] + Lnz[i];

        for (p = Lp[i]; p < p2; p++) {
          Y[Li[p]] -= Lx[p] * yi;
        }

        lKi = yi / D[i];
        /* the nonzero entry L(k,i) */

        D[k] -= lKi * yi;
        Li[p] = k;
        /* store L(k,i) in column form of L */

        Lx[p] = lKi;
        Lnz[i]++;
        /* increment count of nonzeros in col i */
      }

      if (D[k] === 0.0) return k;
      /* failure, D(k,k) is zero */
    }

    return n;
    /* success, diagonal of D is all nonzero */
  }

  function ldlLsolve(n
  /* L is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , Lp
  /* input of size n+1, not modified */
  , Li
  /* input of size lnz=Lp[n], not modified */
  , Lx
  /* input of size lnz=Lp[n], not modified */
  ) {
    let j, p, p2;

    for (j = 0; j < n; j++) {
      p2 = Lp[j + 1];

      for (p = Lp[j]; p < p2; p++) {
        X[Li[p]] -= Lx[p] * X[j];
      }
    }
  }

  function ldlDsolve(n
  /* D is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , D
  /* input of size n, not modified */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[j] /= D[j];
    }
  }

  function ldlLTsolve(n
  /* L is n-by-n, where n >= 0 */
  , X
  /* size n. right-hand-side on input, soln. on output */
  , Lp
  /* input of size n+1, not modified */
  , Li
  /* input of size lnz=Lp[n], not modified */
  , Lx
  /* input of size lnz=Lp[n], not modified */
  ) {
    let j, p, p2;

    for (j = n - 1; j >= 0; j--) {
      p2 = Lp[j + 1];

      for (p = Lp[j]; p < p2; p++) {
        X[j] -= Lx[p] * X[Li[p]];
      }
    }
  }

  function ldlPerm(n
  /* size of X, B, and P */
  , X
  /* output of size n. */
  , B
  /* input of size n. */
  , P
  /* input permutation array of size n. */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[j] = B[P[j]];
    }
  }

  function ldlPermt(n
  /* size of X, B, and P */
  , X
  /* output of size n. */
  , B
  /* input of size n. */
  , P
  /* input permutation array of size n. */
  ) {
    let j;

    for (j = 0; j < n; j++) {
      X[P[j]] = B[j];
    }
  }

  function prepare(M, n, P) {
    // if a permutation was specified, apply it.
    if (P) {
      let Pinv = new Array(n);

      for (let k = 0; k < n; k++) {
        Pinv[P[k]] = k;
      }

      let Mt = []; // scratch memory
      // Apply permutation. We make M into P*M*P^T

      for (let a = 0; a < M.length; ++a) {
        let ar = Pinv[M[a][0]];
        let ac = Pinv[M[a][1]]; // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
        // if permuted element is below diagonal, we simply transpose it.

        if (ac < ar) {
          let t = ac;
          ac = ar;
          ar = t;
        }

        Mt[a] = [];
        Mt[a][0] = ar;
        Mt[a][1] = ac;
        Mt[a][2] = M[a][2];
      }

      M = Mt; // copy scratch memory.
    } else {
      // if P argument is null, we just use an identity permutation.
      P = [];

      for (let i = 0; i < n; ++i) {
        P[i] = i;
      }
    } // The sparse matrix we are decomposing is A.
    // Now we shall create A from M.


    let Ap = new Array(n + 1);
    let Ai = new Array(M.length);
    let Ax = new Array(M.length); // count number of non-zero elements in columns.

    let LNZ = [];

    for (let i = 0; i < n; ++i) {
      LNZ[i] = 0;
    }

    for (let a = 0; a < M.length; ++a) {
      LNZ[M[a][1]]++;
    }

    Ap[0] = 0;

    for (let i = 0; i < n; ++i) {
      Ap[i + 1] = Ap[i] + LNZ[i];
    }

    let coloffset = [];

    for (let a = 0; a < n; ++a) {
      coloffset[a] = 0;
    } // go through all elements in M, and add them to sparse matrix A.


    for (let i = 0; i < M.length; ++i) {
      let e = M[i];
      let col = e[1];
      let adr = Ap[col] + coloffset[col];
      Ai[adr] = e[0];
      Ax[adr] = e[2];
      coloffset[col]++;
    }

    let D = new Array(n);
    let Y = new Array(n);
    let Lp = new Array(n + 1);
    let Parent = new Array(n);
    let Lnz = new Array(n);
    let Flag = new Array(n);
    let Pattern = new Array(n);
    let bp1 = new Array(n);
    let x = new Array(n);
    let d;
    ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
    let Lx = new Array(Lp[n]);
    let Li = new Array(Lp[n]);
    d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);

    if (d === n) {
      return function (b) {
        ldlPerm(n, bp1, b, P);
        ldlLsolve(n, bp1, Lp, Li, Lx);
        ldlDsolve(n, bp1, D);
        ldlLTsolve(n, bp1, Lp, Li, Lx);
        ldlPermt(n, x, bp1, P);
        return x;
      };
    } else {
      return null;
    }
  }

  var cuthillMckee_1 = cuthillMckee;

  function compareNum(a, b) {
    return a - b;
  }

  function cuthillMckee(list, n) {
    var adj = new Array(n);
    var visited = new Array(n);

    for (var i = 0; i < n; ++i) {
      adj[i] = [];
      visited[i] = false;
    }

    for (var i = 0; i < list.length; ++i) {
      var l = list[i];
      adj[l[0]].push(l[1]);
    }

    var toVisit = new Array(n);
    var eol = 0;
    var ptr = 0;

    for (var i = 0; i < n; ++i) {
      if (visited[i]) {
        continue;
      }

      toVisit[eol++] = i;
      visited[i] = true;

      while (ptr < eol) {
        var v = toVisit[ptr++];
        var nbhd = adj[v];
        nbhd.sort(compareNum);

        for (var j = 0; j < nbhd.length; ++j) {
          var u = nbhd[j];

          if (visited[u]) {
            continue;
          }

          visited[u] = true;
          toVisit[eol++] = u;
        }
      }
    }

    var result = new Array(n);

    for (var i = 0; i < n; ++i) {
      result[toVisit[i]] = i;
    }

    return result;
  }

  const getClosestNumber = (array = [], goal = 0) => {
    const closest = array.reduce((prev, curr) => {
      return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
    });
    return closest;
  };

  const getCloseIndex = (array = [], goal = 0) => {
    const closest = getClosestNumber(array, goal);
    return array.indexOf(closest);
  };

  const updateSystem = (matrix, y, weights) => {
    let nbPoints = y.length;
    let l = nbPoints - 1;
    let newMatrix = new Array(matrix.length);
    let newVector = new Float64Array(nbPoints);

    for (let i = 0; i < l; i++) {
      let w = weights[i];
      let diag = i * 2;
      let next = diag + 1;
      newMatrix[diag] = matrix[diag].slice();
      newMatrix[next] = matrix[next].slice();

      if (w === 0) {
        newVector[i] = 0;
      } else {
        newVector[i] = y[i] * w;
        newMatrix[diag][2] += w;
      }
    }

    newVector[l] = y[l] * weights[l];
    newMatrix[l * 2] = matrix[l * 2].slice();
    newMatrix[l * 2][2] += weights[l];
    return [newMatrix, newVector];
  };

  const getDeltaMatrix = (nbPoints, lambda) => {
    let matrix = [];
    let last = nbPoints - 1;

    for (let i = 0; i < last; i++) {
      matrix.push([i, i, lambda * 2]);
      matrix.push([i + 1, i, -1 * lambda]);
    }

    matrix[0][2] = lambda;
    matrix.push([last, last, lambda]);
    return {
      lowerTriangularNonZeros: matrix,
      permutationEncodedArray: cuthillMckee_1(matrix, nbPoints)
    };
  };

  /**
   * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
   * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/master/airPLS_manuscript.pdf)
   * @param {Array<number>} x - x axis data useful when control points or zones are submitted
   * @param {Array<number>} y - Original data
   * @param {object} [options={}] - Options object
   * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
   * @param {number} [options.factorCriterion = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
   * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
   * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
   * @param {Array<number>} [options.controlPoints = []] - Array of x axis values to force that baseline cross those points.
   * @param {Array<number>} [options.baseLineZones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
   * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
   */

  function airPLS(x, y, options = {}) {
    let {
      maxIterations = 100,
      lambda = 100,
      factorCriterion = 0.001,
      weights = new Array(y.length).fill(1),
      controlPoints = [],
      baseLineZones = []
    } = options;

    if (controlPoints.length > 0) {
      controlPoints.forEach((e, i, arr) => arr[i] = getCloseIndex(x, e));
    }

    if (baseLineZones.length > 0) {
      baseLineZones.forEach(range => {
        let indexFrom = getCloseIndex(x, range.from);
        let indexTo = getCloseIndex(x, range.to);
        if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];

        for (let i = indexFrom; i < indexTo; i++) {
          controlPoints.push(i);
        }
      });
    }

    let baseline, iteration;
    let nbPoints = y.length;
    let l = nbPoints - 1;
    let sumNegDifferences = Number.MAX_SAFE_INTEGER;
    let stopCriterion = factorCriterion * y.reduce((sum, e) => Math.abs(e) + sum, 0);
    let {
      lowerTriangularNonZeros,
      permutationEncodedArray
    } = getDeltaMatrix(nbPoints, lambda);

    for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
      let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
      let cho = prepare(leftHandSide, nbPoints, permutationEncodedArray);
      baseline = cho(rightHandSide);
      sumNegDifferences = 0;
      let difference = y.map(calculateError);
      let maxNegativeDiff = -1 * Number.MAX_SAFE_INTEGER;

      for (let i = 1; i < l; i++) {
        let diff = difference[i];

        if (diff >= 0) {
          weights[i] = 0;
        } else {
          weights[i] = Math.exp(iteration * diff / sumNegDifferences);
          if (maxNegativeDiff < diff) maxNegativeDiff = diff;
        }
      }

      let value = Math.exp(iteration * maxNegativeDiff / sumNegDifferences);
      weights[0] = value;
      weights[l] = value;
      controlPoints.forEach(i => weights[i] = value);
    }

    return {
      corrected: y.map((e, i) => e - baseline[i]),
      baseline,
      iteration,
      error: sumNegDifferences
    };

    function calculateError(e, i) {
      let diff = e - baseline[i];
      if (diff < 0) sumNegDifferences += diff;
      return diff;
    }
  }

  /**
   * Adaptive iteratively reweighted penalized least squares [1]
   *
   * This function calls ml-airpls
   *
   * References:
   * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
   * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
   * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
   * @export
   * @param {Array<number>} ys
   * @param {object} [options] - Options object
   * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
   * @param {object} [options.regression] - Options for the regression
   * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
   * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
   * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
   * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
   * @returns {BaselineOutput}
   */

  function airPLSBaseline(ys, options = {}) {
    const numberPoints = ys.length;
    let {
      x,
      regressionOptions
    } = options;

    if (!x) {
      x = sequentialFill({
        from: 0,
        to: numberPoints - 1,
        size: numberPoints
      });
    }

    let output = airPLS(x, ys, regressionOptions);
    return {
      baseline: output.baseline,
      correctedSpectrum: output.corrected
    };
  }

  function maybeToPrecision(value, digits) {
    if (value < 0) {
      value = 0 - value;

      if (typeof digits === 'number') {
        return `- ${value.toPrecision(digits)}`;
      } else {
        return `- ${value.toString()}`;
      }
    } else {
      if (typeof digits === 'number') {
        return value.toPrecision(digits);
      } else {
        return value.toString();
      }
    }
  }

  function checkArraySize(x, y) {
    if (!isAnyArray(x) || !isAnyArray(y)) {
      throw new TypeError('x and y must be arrays');
    }

    if (x.length !== y.length) {
      throw new RangeError('x and y arrays must have the same length');
    }
  }

  class BaseRegression {
    constructor() {
      if (new.target === BaseRegression) {
        throw new Error('BaseRegression must be subclassed');
      }
    }

    predict(x) {
      if (typeof x === 'number') {
        return this._predict(x);
      } else if (isAnyArray(x)) {
        const y = [];

        for (let i = 0; i < x.length; i++) {
          y.push(this._predict(x[i]));
        }

        return y;
      } else {
        throw new TypeError('x must be a number or array');
      }
    }

    _predict() {
      throw new Error('_predict must be implemented');
    }

    train() {// Do nothing for this package
    }

    toString() {
      return '';
    }

    toLaTeX() {
      return '';
    }
    /**
     * Return the correlation coefficient of determination (r) and chi-square.
     * @param {Array<number>} x
     * @param {Array<number>} y
     * @return {object}
     */


    score(x, y) {
      if (!isAnyArray(x) || !isAnyArray(y) || x.length !== y.length) {
        throw new Error('x and y must be arrays of the same length');
      }

      const n = x.length;
      const y2 = new Array(n);

      for (let i = 0; i < n; i++) {
        y2[i] = this._predict(x[i]);
      }

      let xSum = 0;
      let ySum = 0;
      let chi2 = 0;
      let rmsd = 0;
      let xSquared = 0;
      let ySquared = 0;
      let xY = 0;

      for (let i = 0; i < n; i++) {
        xSum += y2[i];
        ySum += y[i];
        xSquared += y2[i] * y2[i];
        ySquared += y[i] * y[i];
        xY += y2[i] * y[i];

        if (y[i] !== 0) {
          chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
        }

        rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
      }

      const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
      return {
        r: r,
        r2: r * r,
        chi2: chi2,
        rmsd: Math.sqrt(rmsd / n)
      };
    }

  }

  class PolynomialRegression extends BaseRegression {
    constructor(x, y, degree) {
      super();

      if (x === true) {
        this.degree = y.degree;
        this.powers = y.powers;
        this.coefficients = y.coefficients;
      } else {
        checkArraySize(x, y);
        regress(this, x, y, degree);
      }
    }

    _predict(x) {
      let y = 0;

      for (let k = 0; k < this.powers.length; k++) {
        y += this.coefficients[k] * Math.pow(x, this.powers[k]);
      }

      return y;
    }

    toJSON() {
      return {
        name: 'polynomialRegression',
        degree: this.degree,
        powers: this.powers,
        coefficients: this.coefficients
      };
    }

    toString(precision) {
      return this._toFormula(precision, false);
    }

    toLaTeX(precision) {
      return this._toFormula(precision, true);
    }

    _toFormula(precision, isLaTeX) {
      let sup = '^';
      let closeSup = '';
      let times = ' * ';

      if (isLaTeX) {
        sup = '^{';
        closeSup = '}';
        times = '';
      }

      let fn = '';
      let str = '';

      for (let k = 0; k < this.coefficients.length; k++) {
        str = '';

        if (this.coefficients[k] !== 0) {
          if (this.powers[k] === 0) {
            str = maybeToPrecision(this.coefficients[k], precision);
          } else {
            if (this.powers[k] === 1) {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
            } else {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
            }
          }

          if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
            str = ` + ${str}`;
          } else if (k !== this.coefficients.length - 1) {
            str = ` ${str}`;
          }
        }

        fn = str + fn;
      }

      if (fn.charAt(0) === '+') {
        fn = fn.slice(1);
      }

      return `f(x) = ${fn}`;
    }

    static load(json) {
      if (json.name !== 'polynomialRegression') {
        throw new TypeError('not a polynomial regression model');
      }

      return new PolynomialRegression(true, json);
    }

  }

  function regress(pr, x, y, degree) {
    const n = x.length;
    let powers;

    if (Array.isArray(degree)) {
      powers = degree;
      degree = powers.length;
    } else {
      degree++;
      powers = new Array(degree);

      for (let k = 0; k < degree; k++) {
        powers[k] = k;
      }
    }

    const F = new Matrix(n, degree);
    const Y = new Matrix([y]);

    for (let k = 0; k < degree; k++) {
      for (let i = 0; i < n; i++) {
        if (powers[k] === 0) {
          F.set(i, k, 1);
        } else {
          F.set(i, k, Math.pow(x[i], powers[k]));
        }
      }
    }

    const FT = new MatrixTransposeView(F);
    const A = FT.mmul(F);
    const B = FT.mmul(new MatrixTransposeView(Y));
    pr.degree = degree - 1;
    pr.powers = powers;
    pr.coefficients = solve(A, B).to1DArray();
  }

  /**
   * Iterative regression-based baseline correction
   * @param {Array<number>} x - Independent axis variable
   * @param {Array<number>} y - Dependent axis variable
   * @param {object} [options] - Options object
   * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
   * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
   * @param {*} [options.regressionOptions] - Options for regressionFunction
   * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
   * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
   */

  function baselineCorrectionRegression(x, y, options = {}) {
    let {
      maxIterations = 100,
      Regression = PolynomialRegression,
      regressionOptions,
      tolerance = 0.001
    } = options;

    if (!regressionOptions && Regression === PolynomialRegression) {
      regressionOptions = 3;
    }

    let baseline = y.slice();
    let fitting = y.slice();
    let oldFitting = y;
    let iteration = 0;
    let delta;
    let regression;

    while (iteration < maxIterations) {
      // Calculate the fitting result
      regression = new Regression(x, baseline, regressionOptions);
      delta = 0;

      for (let i = 0; i < baseline.length; i++) {
        fitting[i] = regression.predict(x[i]);

        if (baseline[i] > fitting[i]) {
          baseline[i] = fitting[i];
        }

        delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
      } // Stop criterion


      if (delta < tolerance) {
        break;
      } else {
        oldFitting = fitting.slice();
        iteration++;
      }
    } // removes baseline


    let corrected = new Array(baseline.length);

    for (let j = 0; j < baseline.length; j++) {
      corrected[j] = y[j] - baseline[j];
    }

    return {
      corrected,
      delta,
      iteration,
      baseline,
      regression: regression
    };
  }

  /**
   * Iterative polynomial fitting [1]
   *
   * Implementation based on ml-baseline-correction-regression
   *
   * References:
   * [1] Gan, F.; Ruan, G.; Mo, J.
   * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
   *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
   * https://doi.org/10.1016/j.chemolab.2005.08.009.
   * @export
   * @param {Array<number>} ys
   * @param {object} [options] - Options object
   * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
   * @param {Object} [options.regression]
   * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
   * @param {Object} [options.regression]
   * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
   * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
   * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
   * @returns {BaselineOutput}
   */

  function iterativePolynomialBaseline(ys, options = {}) {
    const numberPoints = ys.length;
    let {
      x,
      regressionOptions
    } = options;

    if (!x) {
      x = sequentialFill({
        from: 0,
        to: numberPoints - 1,
        size: numberPoints
      });
    }

    let output = baselineCorrectionRegression(x, ys, regressionOptions);
    return {
      baseline: output.baseline,
      correctedSpectrum: output.corrected
    };
  }

  /**
   * Checks if input is valdi
   * @param {Array<number>} input

   */

  function xCheck(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
  }

  /**
   * This function pads an array
   * @param {Array} array - the array that will be padded
   * @param {object} [options={}]
   * @param {string} [options.algorithm=''] '', value, circular, duplicate
   * @param {number} [options.size=0] padding size before first element and after last element
   * @param {number} [options.value=0] value to use for padding (if algorithm='value')
   * @return {Array}
   */

  function xPadding(array, options = {}) {
    const {
      size = 0,
      value = 0,
      algorithm = ''
    } = options;
    xCheck(array);

    if (!algorithm) {
      if (array instanceof Float64Array) {
        return array.slice();
      } else {
        return Float64Array.from(array);
      }
    }

    let result = new Float64Array(array.length + size * 2);

    for (let i = 0; i < array.length; i++) {
      result[i + size] = array[i];
    }

    let fromEnd = size + array.length;
    let toEnd = 2 * size + array.length;

    switch (algorithm.toLowerCase()) {
      case 'value':
        for (let i = 0; i < size; i++) {
          result[i] = value;
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = value;
        }

        break;

      case 'duplicate':
        for (let i = 0; i < size; i++) {
          result[i] = array[0];
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = array[array.length - 1];
        }

        break;

      case 'circular':
        for (let i = 0; i < size; i++) {
          result[i] = array[(array.length - size % array.length + i) % array.length];
        }

        for (let i = 0; i < size; i++) {
          result[i + fromEnd] = array[i % array.length];
        }

        break;

      default:
        throw Error('xPadding: unknown algorithm');
    }

    return result;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {function} fct callback function that from an array returns a value.
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=0] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='value'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRolling(array, fct, options = {}) {
    xCheck(array);
    if (typeof fct !== 'function') throw Error('fct has to be a function');
    const {
      window = 5,
      padding = {}
    } = options;
    const {
      size = window - 1,
      algorithm,
      value
    } = padding;
    array = xPadding(array, {
      size,
      algorithm,
      value
    }); // ensure we get a copy and it is float64

    const newArray = [];

    for (let i = 0; i < array.length - window + 1; i++) {
      let subArray = new Float64Array(array.buffer, i * 8, window); // we will send a view to the original buffer

      newArray.push(fct(subArray));
    }

    return newArray;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingAverage(array, options = {}) {
    return xRolling(array, mean, options);
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingMedian(array, options = {}) {
    return xRolling(array, median, options);
  }

  /**

   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @returns {BaselineOutput}
   */

  function rollingAverageBaseline(ys, options = {}) {
    let window = Math.max(Math.round(ys.length * 0.1), 2);
    let defaults = {
      window: window,
      padding: {
        size: window - 1,
        algorithm: 'duplicate',
        value: 0
      }
    };
    let actualOptions = Object.assign({}, defaults, options);
    let baseline = xRollingAverage(ys, actualOptions);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  /**
   * Rolling ball baseline correction algorithm.
   * From the abstract of (1):
   * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
   * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
   *
   * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
   * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
   *
   * Reference:
   * (1) Kneen, M. A.; Annegarn, H. J.
   *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
   *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
   *     https://doi.org/10.1016/0168-583X(95)00908-6.
   * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
   *     https://cran.r-project.org/web/packages/baseline/index.html
   * @export
   * @param {Array} spectrum
   * @param {Object} [options={}]
   * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
   * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
   */

  function rollingBall(spectrum, options = {}) {
    if (!isAnyArray(spectrum)) {
      throw new Error('Spectrum must be an array');
    }

    if (spectrum.length === 0) {
      throw new TypeError('Spectrum must not be empty');
    }

    const numberPoints = spectrum.length;
    const maxima = new Float64Array(numberPoints);
    const minima = new Float64Array(numberPoints);
    const baseline = new Float64Array(numberPoints); // windowM 4 percent of spectrum length
    // windowS 8 percent of spectrum length

    const {
      windowM = Math.round(numberPoints * 0.04),
      windowS = Math.round(numberPoints * 0.08)
    } = options; // fi(1) in original paper

    for (let i = 0; i < spectrum.length; i++) {
      let windowLeft = max([0, i - windowM]);
      let windowRight = min([i + windowM + 1, spectrum.length]);
      minima[i] = min(spectrum.slice(windowLeft, windowRight));
    } // fi in original paper


    for (let i = 0; i < minima.length; i++) {
      let windowLeft = max([0, i - windowM]);
      let windowRight = min([i + windowM + 1, minima.length]);
      maxima[i] = max(minima.slice(windowLeft, windowRight));
    }

    for (let i = 0; i < minima.length; i++) {
      let windowLeft = max([0, i - windowS]);
      let windowRight = min([i + windowS + 1, maxima.length]);
      baseline[i] = mean(maxima.slice(windowLeft, windowRight));
    }

    return baseline;
  }

  /**
   * Rolling ball baseline correction algorithm.
   * From the abstract of (1):
   * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
   * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
   *
   * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
   * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
   *
   * Algorithm described in (1), but in the implementation here the window width does not change.
   *
   * Reference:
   * (1) Kneen, M. A.; Annegarn, H. J.
   *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
   *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
   *     https://doi.org/10.1016/0168-583X(95)00908-6.
   * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
   *     https://cran.r-project.org/web/packages/baseline/index.html
   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
   * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
   * @returns {BaselineOutput}
   */

  function rollingBallBaseline(ys, options = {}) {
    const baseline = rollingBall(ys, options);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  /**

   *
   * @export
   * @param {Array<number>} ys
   * @param {Object} [options={}]
   * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @returns {BaselineOutput}
   */

  function rollingMedianBaseline(ys, options = {}) {
    let window = Math.max(Math.round(ys.length * 0.1), 2);
    let defaults = {
      window: window,
      padding: {
        size: window - 1,
        algorithm: 'duplicate',
        value: 0
      }
    };
    let actualOptions = Object.assign({}, defaults, options);
    let baseline = xRollingMedian(ys, actualOptions);
    let corrected = new Float64Array(ys.length);

    for (let i = 0; i < corrected.length; i++) {
      corrected[i] = ys[i] - baseline[i];
    }

    return {
      baseline: baseline,
      correctedSpectrum: corrected
    };
  }

  function norm(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var _options$algorithm = options.algorithm,
        algorithm = _options$algorithm === void 0 ? 'absolute' : _options$algorithm,
        _options$sumValue = options.sumValue,
        sumValue = _options$sumValue === void 0 ? 1 : _options$sumValue,
        _options$maxValue = options.maxValue,
        maxValue = _options$maxValue === void 0 ? 1 : _options$maxValue;

    if (!isAnyArray(input)) {
      throw new Error('input must be an array');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    if (input.length === 0) {
      throw new Error('input must not be empty');
    }

    switch (algorithm.toLowerCase()) {
      case 'absolute':
        {
          var absoluteSumValue = absoluteSum(input) / sumValue;
          if (absoluteSumValue === 0) return input.slice(0);

          for (var i = 0; i < input.length; i++) {
            output[i] = input[i] / absoluteSumValue;
          }

          return output;
        }

      case 'max':
        {
          var currentMaxValue = max(input);
          if (currentMaxValue === 0) return input.slice(0);
          var factor = maxValue / currentMaxValue;

          for (var _i = 0; _i < input.length; _i++) {
            output[_i] = input[_i] * factor;
          }

          return output;
        }

      case 'sum':
        {
          var sumFactor = sum(input) / sumValue;
          if (sumFactor === 0) return input.slice(0);

          for (var _i2 = 0; _i2 < input.length; _i2++) {
            output[_i2] = input[_i2] / sumFactor;
          }

          return output;
        }

      default:
        throw new Error("norm: unknown algorithm: ".concat(algorithm));
    }
  }

  function absoluteSum(input) {
    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += Math.abs(input[i]);
    }

    return sumValue;
  }

  /**
   * Normalize an array of zones:
   * - ensure than from < to
   * - merge overlapping zones
   *
   * The method will always check if from if lower than to and will swap if required.
   * @param {Array} [zones=[]]
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
   */
  function normalize$1(zones = [], options = {}) {
    if (zones.length === 0) return [];
    let {
      from = Number.NEGATIVE_INFINITY,
      to = Number.POSITIVE_INFINITY
    } = options;
    if (from > to) [from, to] = [to, from];
    zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
      from: zone.to,
      to: zone.from
    } : zone);
    zones = zones.sort((a, b) => {
      if (a.from !== b.from) return a.from - b.from;
      return a.to - b.to;
    });
    zones.forEach(zone => {
      if (from > zone.from) zone.from = from;
      if (to < zone.to) zone.to = to;
    });
    zones = zones.filter(zone => zone.from <= zone.to);
    if (zones.length === 0) return [];
    let currentZone = zones[0];
    let result = [currentZone];

    for (let i = 1; i < zones.length; i++) {
      let zone = zones[i];

      if (zone.from <= currentZone.to) {
        currentZone.to = zone.to;
      } else {
        currentZone = zone;
        result.push(currentZone);
      }
    }

    return result;
  }

  /**
   * Convert an array of exclusions and keep only from / to
   *
   * The method will always check if from if lower than to and will swap if required.
   * @param {Array} [exclusions=[]]
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of zones (after inversion)
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of zones (after inversion)
   */

  function invert(exclusions = [], options = {}) {
    let {
      from = Number.NEGATIVE_INFINITY,
      to = Number.POSITIVE_INFINITY
    } = options;
    if (from > to) [from, to] = [to, from];
    exclusions = normalize$1(exclusions, {
      from,
      to
    });
    if (exclusions.length === 0) return [{
      from,
      to
    }];
    let zones = [];

    for (let i = 0; i < exclusions.length; i++) {
      let exclusion = exclusions[i];
      let nextExclusion = exclusions[i + 1];

      if (i === 0) {
        if (exclusion.from > from) {
          zones.push({
            from,
            to: exclusion.from
          });
        }
      }

      if (i === exclusions.length - 1) {
        if (exclusion.to < to) {
          zones.push({
            from: exclusion.to,
            to
          });
        }
      } else {
        zones.push({
          from: exclusion.to,
          to: nextExclusion.from
        });
      }
    }

    return zones;
  }

  /**
   * Add the number of points per zone to reach a specified total
   * @param {Array} [zones=[]]
   * @param {number} [numberOfPoints] Total number of points to distribute between zones
   * @param {object} [options={}]
   * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
   * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
   */

  function zonesWithPoints(zones, numberOfPoints, options = {}) {
    if (zones.length === 0) return zones;
    zones = normalize$1(zones, options);
    const totalSize = zones.reduce((previous, current) => {
      return previous + (current.to - current.from);
    }, 0);
    let unitsPerPoint = totalSize / numberOfPoints;
    let currentTotal = 0;

    for (let i = 0; i < zones.length - 1; i++) {
      let zone = zones[i];
      zone.numberOfPoints = Math.min(Math.round((zone.to - zone.from) / unitsPerPoint), numberOfPoints - currentTotal);
      currentTotal += zone.numberOfPoints;
    }

    zones[zones.length - 1].numberOfPoints = numberOfPoints - currentTotal;
    return zones;
  }

  /**
   * function that retrieves the getEquallySpacedData with the variant "slot"
   *
   * @param {Array<number>} x
   * @param {Array<number>} y
   * @param {number} from - Initial point
   * @param {number} to - Final point
   * @param {number} numberOfPoints
   * @return {Array} - Array of y's equally spaced with the variant "slot"
   */
  function equallySpacedSlot(x, y, from, to, numberOfPoints) {
    let xLength = x.length;
    let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
    let halfStep = step / 2;
    let lastStep = x[x.length - 1] - x[x.length - 2];
    let start = from - halfStep;
    let output = new Array(numberOfPoints); // Init main variables

    let min = start;
    let max = start + step;
    let previousX = -Number.MAX_VALUE;
    let previousY = 0;
    let nextX = x[0];
    let nextY = y[0];
    let frontOutsideSpectra = 0;
    let backOutsideSpectra = true;
    let currentValue = 0; // for slot algorithm

    let currentPoints = 0;
    let i = 1; // index of input

    let j = 0; // index of output

    main: while (true) {
      if (previousX >= nextX) throw new Error('x must be an increasing serie');

      while (previousX - max > 0) {
        // no overlap with original point, just consume current value
        if (backOutsideSpectra) {
          currentPoints++;
          backOutsideSpectra = false;
        }

        output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
        j++;

        if (j === numberOfPoints) {
          break main;
        }

        min = max;
        max += step;
        currentValue = 0;
        currentPoints = 0;
      }

      if (previousX > min) {
        currentValue += previousY;
        currentPoints++;
      }

      if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
        currentPoints--;
      }

      previousX = nextX;
      previousY = nextY;

      if (i < xLength) {
        nextX = x[i];
        nextY = y[i];
        i++;
      } else {
        nextX += lastStep;
        nextY = 0;
        frontOutsideSpectra++;
      }
    }

    return output;
  }

  /**
   * Function that calculates the integral of the line between two
   * x-coordinates, given the slope and intercept of the line.
   * @param {number} x0
   * @param {number} x1
   * @param {number} slope
   * @param {number} intercept
   * @return {number} integral value.
   */
  function integral(x0, x1, slope, intercept) {
    return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
  }

  /**
   * function that retrieves the getEquallySpacedData with the variant "smooth"
   *
   * @param {Array<number>} x
   * @param {Array<number>} y
   * @param {number} from - Initial point
   * @param {number} to - Final point
   * @param {number} numberOfPoints
   * @return {Array} - Array of y's equally spaced with the variant "smooth"
   */

  function equallySpacedSmooth(x, y, from, to, numberOfPoints) {
    let xLength = x.length;
    let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
    let halfStep = step / 2;
    let output = new Array(numberOfPoints);
    let initialOriginalStep = x[1] - x[0];
    let lastOriginalStep = x[xLength - 1] - x[xLength - 2]; // Init main variables

    let min = from - halfStep;
    let max = from + halfStep;
    let previousX = Number.MIN_VALUE;
    let previousY = 0;
    let nextX = x[0] - initialOriginalStep;
    let nextY = 0;
    let currentValue = 0;
    let slope = 0;
    let intercept = 0;
    let sumAtMin = 0;
    let sumAtMax = 0;
    let i = 0; // index of input

    let j = 0; // index of output

    function getSlope(x0, y0, x1, y1) {
      return (y1 - y0) / (x1 - x0);
    }

    let add = 0;

    main: while (true) {
      if (previousX <= min && min <= nextX) {
        add = integral(0, min - previousX, slope, previousY);
        sumAtMin = currentValue + add;
      }

      while (nextX - max >= 0) {
        // no overlap with original point, just consume current value
        add = integral(0, max - previousX, slope, previousY);
        sumAtMax = currentValue + add;
        output[j++] = (sumAtMax - sumAtMin) / step;

        if (j === numberOfPoints) {
          break main;
        }

        min = max;
        max += step;
        sumAtMin = sumAtMax;
      }

      currentValue += integral(previousX, nextX, slope, intercept);
      previousX = nextX;
      previousY = nextY;

      if (i < xLength) {
        nextX = x[i];
        nextY = y[i];
        i++;
      } else if (i === xLength) {
        nextX += lastOriginalStep;
        nextY = 0;
      }

      slope = getSlope(previousX, previousY, nextX, nextY);
      intercept = -slope * previousX + previousY;
    }

    return output;
  }

  /**
   * Function that returns a Number array of equally spaced numberOfPoints
   * containing a representation of intensities of the spectra arguments x
   * and y.
   *
   * The options parameter contains an object in the following form:
   * from: starting point
   * to: last point
   * numberOfPoints: number of points between from and to
   * variant: "slot" or "smooth" - smooth is the default option
   *
   * The slot variant consist that each point in the new array is calculated
   * averaging the existing points between the slot that belongs to the current
   * value. The smooth variant is the same but takes the integral of the range
   * of the slot and divide by the step size between two points in the new array.
   *
   * If exclusions zone are present, zones are ignored !
   * @param {object} [arrayXY={}] - object containing 2 properties x and y (both an array)
   * @param {object} [options={}]
   * @param {number} [options.from=x[0]]
   * @param {number} [options.to=x[x.length-1]]
   * @param {string} [options.variant='smooth']
   * @param {number} [options.numberOfPoints=100]
   * @param {Array} [options.exclusions=[]] array of from / to that should be skipped for the generation of the points
   * @param {Array} [options.zones=[]] array of from / to that should be kept
   * @return {object<x: Array, y:Array>} new object with x / y array with the equally spaced data.
   */

  function equallySpaced(arrayXY = {}, options = {}) {
    let {
      x,
      y
    } = arrayXY;
    let xLength = x.length;
    let reverse = false;

    if (x.length > 1 && x[0] > x[1]) {
      x = x.slice().reverse();
      y = y.slice().reverse();
      reverse = true;
    }

    let {
      from = x[0],
      to = x[xLength - 1],
      variant = 'smooth',
      numberOfPoints = 100,
      exclusions = [],
      zones = []
    } = options;

    if (xLength !== y.length) {
      throw new RangeError("the x and y vector doesn't have the same size.");
    }

    if (typeof from !== 'number' || isNaN(from)) {
      throw new RangeError("'from' option must be a number");
    }

    if (typeof to !== 'number' || isNaN(to)) {
      throw new RangeError("'to' option must be a number");
    }

    if (typeof numberOfPoints !== 'number' || isNaN(numberOfPoints)) {
      throw new RangeError("'numberOfPoints' option must be a number");
    }

    if (numberOfPoints < 2) {
      throw new RangeError("'numberOfPoints' option must be greater than 1");
    }

    if (zones.length === 0) {
      zones = invert(exclusions, {
        from,
        to
      });
    }

    zones = zonesWithPoints(zones, numberOfPoints, {
      from,
      to
    });
    let xResult = [];
    let yResult = [];

    for (let zone of zones) {
      let zoneResult = processZone(x, y, zone.from, zone.to, zone.numberOfPoints, variant);
      xResult = xResult.concat(zoneResult.x);
      yResult = yResult.concat(zoneResult.y);
    }

    if (reverse) {
      if (from < to) {
        return {
          x: xResult.reverse(),
          y: yResult.reverse()
        };
      } else {
        return {
          x: xResult,
          y: yResult
        };
      }
    } else {
      if (from < to) {
        return {
          x: xResult,
          y: yResult
        };
      } else {
        return {
          x: xResult.reverse(),
          y: yResult.reverse()
        };
      }
    }
  }

  function processZone(x, y, from, to, numberOfPoints, variant) {
    if (numberOfPoints < 1) {
      throw new RangeError('the number of points must be at least 1');
    }

    let output = variant === 'slot' ? equallySpacedSlot(x, y, from, to, numberOfPoints) : equallySpacedSmooth(x, y, from, to, numberOfPoints);
    return {
      x: sequentialFill({
        from,
        to,
        size: numberOfPoints
      }),
      y: output
    };
  }

  function getZones(from, to, exclusions = []) {
    if (from > to) {
      [from, to] = [to, from];
    } // in exclusions from and to have to be defined


    exclusions = exclusions.filter(exclusion => exclusion.from !== undefined && exclusion.to !== undefined);
    exclusions = JSON.parse(JSON.stringify(exclusions)); // we ensure that from before to

    exclusions.forEach(exclusion => {
      if (exclusion.from > exclusion.to) {
        [exclusion.to, exclusion.from] = [exclusion.from, exclusion.to];
      }
    });
    exclusions.sort((a, b) => a.from - b.from); // we will rework the exclusions in order to remove overlap and outside range (from / to)

    exclusions.forEach(exclusion => {
      if (exclusion.from < from) exclusion.from = from;
      if (exclusion.to > to) exclusion.to = to;
    });

    for (let i = 0; i < exclusions.length - 1; i++) {
      if (exclusions[i].to > exclusions[i + 1].from) {
        exclusions[i].to = exclusions[i + 1].from;
      }
    }

    exclusions = exclusions.filter(exclusion => exclusion.from < exclusion.to);

    if (!exclusions || exclusions.length === 0) {
      return [{
        from,
        to
      }];
    }

    let zones = [];
    let currentFrom = from;

    for (let exclusion of exclusions) {
      if (currentFrom < exclusion.from) {
        zones.push({
          from: currentFrom,
          to: exclusion.from
        });
      }

      currentFrom = exclusion.to;
    }

    if (currentFrom < to) {
      zones.push({
        from: currentFrom,
        to: to
      });
    }

    return zones;
  }

  /**
   * Filter an array x/y based on various criteria
   * x points are expected to be sorted
   *
   * @param {object} points
   * @param {object} [options={}]
   * @param {array} [options.from]
   * @param {array} [options.to]
   * @param {array} [options.exclusions=[]]
   * @return {{x: Array<number>, y: Array<number>}}
   */

  function filterX(points, options = {}) {
    const {
      x,
      y
    } = points;
    const {
      from = x[0],
      to = x[x.length - 1],
      exclusions = []
    } = options;
    let zones = getZones(from, to, exclusions);
    let currentZoneIndex = 0;
    let newX = [];
    let newY = [];
    let position = 0;

    while (position < x.length) {
      if (x[position] <= zones[currentZoneIndex].to && x[position] >= zones[currentZoneIndex].from) {
        newX.push(x[position]);
        newY.push(y[position]);
      } else {
        if (x[position] > zones[currentZoneIndex].to) {
          currentZoneIndex++;
          if (!zones[currentZoneIndex]) break;
        }
      }

      position++;
    }

    return {
      x: newX,
      y: newY
    };
  }

  var defaultOptions$2 = {
    size: 1,
    value: 0
  };
  /**
   * Case when the entry is an array
   * @param data
   * @param options
   * @returns {Array}
   */

  function arrayCase(data, options) {
    var len = data.length;

    if (typeof options.size === 'number') {
      options.size = [options.size, options.size];
    }

    var cond = len + options.size[0] + options.size[1];
    var output;

    if (options.output) {
      if (options.output.length !== cond) {
        throw new RangeError('Wrong output size');
      }

      output = options.output;
    } else {
      output = new Array(cond);
    }

    var i;

    if (options.value === 'circular') {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) {
          output[i] = data[(len - options.size[0] % len + i) % len];
        } else if (i < options.size[0] + len) {
          output[i] = data[i - options.size[0]];
        } else {
          output[i] = data[(i - options.size[0]) % len];
        }
      }
    } else if (options.value === 'replicate') {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = data[0];else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = data[len - 1];
      }
    } else if (options.value === 'symmetric') {
      if (options.size[0] > len || options.size[1] > len) {
        throw new RangeError('expanded value should not be bigger than the data length');
      }

      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = data[options.size[0] - 1 - i];else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = data[2 * len + options.size[0] - i - 1];
      }
    } else {
      for (i = 0; i < cond; i++) {
        if (i < options.size[0]) output[i] = options.value;else if (i < options.size[0] + len) output[i] = data[i - options.size[0]];else output[i] = options.value;
      }
    }

    return output;
  }
  /**
   * Case when the entry is a matrix
   * @param data
   * @param options
   * @returns {Array}
   */


  function matrixCase(data, options) {
    // var row = data.length;
    // var col = data[0].length;
    if (options.size[0] === undefined) {
      options.size = [options.size, options.size, options.size, options.size];
    }

    throw new Error('matrix not supported yet, sorry');
  }
  /**
   * Pads and array
   * @param {Array <number>} data
   * @param {object} options
   */


  function padArray(data, options) {
    options = Object.assign({}, defaultOptions$2, options);

    if (Array.isArray(data)) {
      if (Array.isArray(data[0])) return matrixCase(data, options);else return arrayCase(data, options);
    } else {
      throw new TypeError('data should be an array');
    }
  }

  var src = padArray;

  /**
   * Factorial of a number
   * @ignore
   * @param n
   * @return {number}
   */

  function factorial(n) {
    let r = 1;

    while (n > 0) r *= n--;

    return r;
  }

  const defaultOptions$1 = {
    windowSize: 5,
    derivative: 1,
    polynomial: 2,
    pad: 'none',
    padValue: 'replicate'
  };
  /**
   * Savitzky-Golay filter
   * @param {Array <number>} data
   * @param {number} h
   * @param {Object} options
   * @returns {Array}
   */

  function savitzkyGolay(data, h, options) {
    options = Object.assign({}, defaultOptions$1, options);

    if (options.windowSize % 2 === 0 || options.windowSize < 5 || !Number.isInteger(options.windowSize)) {
      throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
    }

    if (options.derivative < 0 || !Number.isInteger(options.derivative)) {
      throw new RangeError('Derivative should be a positive integer');
    }

    if (options.polynomial < 1 || !Number.isInteger(options.polynomial)) {
      throw new RangeError('Polynomial should be a positive integer');
    }

    let C, norm;
    let step = Math.floor(options.windowSize / 2);

    if (options.pad === 'pre') {
      data = src(data, {
        size: step,
        value: options.padValue
      });
    }

    let ans = new Array(data.length - 2 * step);

    if (options.windowSize === 5 && options.polynomial === 2 && (options.derivative === 1 || options.derivative === 2)) {
      if (options.derivative === 1) {
        C = [-2, -1, 0, 1, 2];
        norm = 10;
      } else {
        C = [2, -1, -2, -1, 2];
        norm = 7;
      }
    } else {
      let J = Matrix.ones(options.windowSize, options.polynomial + 1);
      let inic = -(options.windowSize - 1) / 2;

      for (let i = 0; i < J.rows; i++) {
        for (let j = 0; j < J.columns; j++) {
          if (inic + 1 !== 0 || j !== 0) J.set(i, j, Math.pow(inic + i, j));
        }
      }

      let Jtranspose = new MatrixTransposeView(J);
      let Jinv = inverse(Jtranspose.mmul(J));
      C = Jinv.mmul(Jtranspose);
      C = C.getRow(options.derivative);
      norm = 1 / factorial(options.derivative);
    }

    let det = norm * Math.pow(h, options.derivative);

    for (let k = step; k < data.length - step; k++) {
      let d = 0;

      for (let l = 0; l < C.length; l++) d += C[l] * data[l + k - step] / det;

      ans[k - step] = d;
    }

    if (options.pad === 'post') {
      ans = src(ans, {
        size: step,
        value: options.padValue
      });
    }

    return ans;
  }

  var array = createCommonjsModule(function (module, exports) {

    function compareNumbers(a, b) {
      return a - b;
    }
    /**
     * Computes the sum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.sum = function sum(values) {
      var sum = 0;

      for (var i = 0; i < values.length; i++) {
        sum += values[i];
      }

      return sum;
    };
    /**
     * Computes the maximum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.max = function max(values) {
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] > max) max = values[i];
      }

      return max;
    };
    /**
     * Computes the minimum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.min = function min(values) {
      var min = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
      }

      return min;
    };
    /**
     * Computes the min and max of the given values
     * @param {Array} values
     * @returns {{min: number, max: number}}
     */


    exports.minMax = function minMax(values) {
      var min = values[0];
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
        if (values[i] > max) max = values[i];
      }

      return {
        min: min,
        max: max
      };
    };
    /**
     * Computes the arithmetic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.arithmeticMean = function arithmeticMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        sum += values[i];
      }

      return sum / l;
    };
    /**
     * {@link arithmeticMean}
     */


    exports.mean = exports.arithmeticMean;
    /**
     * Computes the geometric mean of the given values
     * @param {Array} values
     * @returns {number}
     */

    exports.geometricMean = function geometricMean(values) {
      var mul = 1;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        mul *= values[i];
      }

      return Math.pow(mul, 1 / l);
    };
    /**
     * Computes the mean of the log of the given values
     * If the return value is exponentiated, it gives the same result as the
     * geometric mean.
     * @param {Array} values
     * @returns {number}
     */


    exports.logMean = function logMean(values) {
      var lnsum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        lnsum += Math.log(values[i]);
      }

      return lnsum / l;
    };
    /**
     * Computes the weighted grand mean for a list of means and sample sizes
     * @param {Array} means - Mean values for each set of samples
     * @param {Array} samples - Number of original values for each set of samples
     * @returns {number}
     */


    exports.grandMean = function grandMean(means, samples) {
      var sum = 0;
      var n = 0;
      var l = means.length;

      for (var i = 0; i < l; i++) {
        sum += samples[i] * means[i];
        n += samples[i];
      }

      return sum / n;
    };
    /**
     * Computes the truncated mean of the given values using a given percentage
     * @param {Array} values
     * @param {number} percent - The percentage of values to keep (range: [0,1])
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.truncatedMean = function truncatedMean(values, percent, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var k = Math.floor(l * percent);
      var sum = 0;

      for (var i = k; i < l - k; i++) {
        sum += values[i];
      }

      return sum / (l - 2 * k);
    };
    /**
     * Computes the harmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.harmonicMean = function harmonicMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        if (values[i] === 0) {
          throw new RangeError('value at index ' + i + 'is zero');
        }

        sum += 1 / values[i];
      }

      return l / sum;
    };
    /**
     * Computes the contraharmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.contraHarmonicMean = function contraHarmonicMean(values) {
      var r1 = 0;
      var r2 = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        r1 += values[i] * values[i];
        r2 += values[i];
      }

      if (r2 < 0) {
        throw new RangeError('sum of values is negative');
      }

      return r1 / r2;
    };
    /**
     * Computes the median of the given values
     * @param {Array} values
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.median = function median(values, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var half = Math.floor(l / 2);

      if (l % 2 === 0) {
        return (values[half - 1] + values[half]) * 0.5;
      } else {
        return values[half];
      }
    };
    /**
     * Computes the variance of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.variance = function variance(values, unbiased) {
      if (unbiased === undefined) unbiased = true;
      var theMean = exports.mean(values);
      var theVariance = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        var x = values[i] - theMean;
        theVariance += x * x;
      }

      if (unbiased) {
        return theVariance / (l - 1);
      } else {
        return theVariance / l;
      }
    };
    /**
     * Computes the standard deviation of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.standardDeviation = function standardDeviation(values, unbiased) {
      return Math.sqrt(exports.variance(values, unbiased));
    };

    exports.standardError = function standardError(values) {
      return exports.standardDeviation(values) / Math.sqrt(values.length);
    };
    /**
     * IEEE Transactions on biomedical engineering, vol. 52, no. 1, january 2005, p. 76-
     * Calculate the standard deviation via the Median of the absolute deviation
     *  The formula for the standard deviation only holds for Gaussian random variables.
     * @returns {{mean: number, stdev: number}}
     */


    exports.robustMeanAndStdev = function robustMeanAndStdev(y) {
      var mean = 0,
          stdev = 0;
      var length = y.length,
          i = 0;

      for (i = 0; i < length; i++) {
        mean += y[i];
      }

      mean /= length;
      var averageDeviations = new Array(length);

      for (i = 0; i < length; i++) averageDeviations[i] = Math.abs(y[i] - mean);

      averageDeviations.sort(compareNumbers);

      if (length % 2 === 1) {
        stdev = averageDeviations[(length - 1) / 2] / 0.6745;
      } else {
        stdev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
      }

      return {
        mean: mean,
        stdev: stdev
      };
    };

    exports.quartiles = function quartiles(values, alreadySorted) {
      if (typeof alreadySorted === 'undefined') alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var quart = values.length / 4;
      var q1 = values[Math.ceil(quart) - 1];
      var q2 = exports.median(values, true);
      var q3 = values[Math.ceil(quart * 3) - 1];
      return {
        q1: q1,
        q2: q2,
        q3: q3
      };
    };

    exports.pooledStandardDeviation = function pooledStandardDeviation(samples, unbiased) {
      return Math.sqrt(exports.pooledVariance(samples, unbiased));
    };

    exports.pooledVariance = function pooledVariance(samples, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var sum = 0;
      var length = 0,
          l = samples.length;

      for (var i = 0; i < l; i++) {
        var values = samples[i];
        var vari = exports.variance(values);
        sum += (values.length - 1) * vari;
        if (unbiased) length += values.length - 1;else length += values.length;
      }

      return sum / length;
    };

    exports.mode = function mode(values) {
      var l = values.length,
          itemCount = new Array(l),
          i;

      for (i = 0; i < l; i++) {
        itemCount[i] = 0;
      }

      var itemArray = new Array(l);
      var count = 0;

      for (i = 0; i < l; i++) {
        var index = itemArray.indexOf(values[i]);
        if (index >= 0) itemCount[index]++;else {
          itemArray[count] = values[i];
          itemCount[count] = 1;
          count++;
        }
      }

      var maxValue = 0,
          maxIndex = 0;

      for (i = 0; i < count; i++) {
        if (itemCount[i] > maxValue) {
          maxValue = itemCount[i];
          maxIndex = i;
        }
      }

      return itemArray[maxIndex];
    };

    exports.covariance = function covariance(vector1, vector2, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var mean1 = exports.mean(vector1);
      var mean2 = exports.mean(vector2);
      if (vector1.length !== vector2.length) throw 'Vectors do not have the same dimensions';
      var cov = 0,
          l = vector1.length;

      for (var i = 0; i < l; i++) {
        var x = vector1[i] - mean1;
        var y = vector2[i] - mean2;
        cov += x * y;
      }

      if (unbiased) return cov / (l - 1);else return cov / l;
    };

    exports.skewness = function skewness(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var s2 = 0,
          s3 = 0,
          l = values.length;

      for (var i = 0; i < l; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s3 += dev * dev * dev;
      }

      var m2 = s2 / l;
      var m3 = s3 / l;
      var g = m3 / Math.pow(m2, 3 / 2.0);

      if (unbiased) {
        var a = Math.sqrt(l * (l - 1));
        var b = l - 2;
        return a / b * g;
      } else {
        return g;
      }
    };

    exports.kurtosis = function kurtosis(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var n = values.length,
          s2 = 0,
          s4 = 0;

      for (var i = 0; i < n; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s4 += dev * dev * dev * dev;
      }

      var m2 = s2 / n;
      var m4 = s4 / n;

      if (unbiased) {
        var v = s2 / (n - 1);
        var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
        var b = s4 / (v * v);
        var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
        return a * b - 3 * c;
      } else {
        return m4 / (m2 * m2) - 3;
      }
    };

    exports.entropy = function entropy(values, eps) {
      if (typeof eps === 'undefined') eps = 0;
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * Math.log(values[i] + eps);

      return -sum;
    };

    exports.weightedMean = function weightedMean(values, weights) {
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * weights[i];

      return sum;
    };

    exports.weightedStandardDeviation = function weightedStandardDeviation(values, weights) {
      return Math.sqrt(exports.weightedVariance(values, weights));
    };

    exports.weightedVariance = function weightedVariance(values, weights) {
      var theMean = exports.weightedMean(values, weights);
      var vari = 0,
          l = values.length;
      var a = 0,
          b = 0;

      for (var i = 0; i < l; i++) {
        var z = values[i] - theMean;
        var w = weights[i];
        vari += w * (z * z);
        b += w;
        a += w * w;
      }

      return vari * (b / (b * b - a));
    };

    exports.center = function center(values, inPlace) {
      if (typeof inPlace === 'undefined') inPlace = false;
      var result = values;
      if (!inPlace) result = [].concat(values);
      var theMean = exports.mean(result),
          l = result.length;

      for (var i = 0; i < l; i++) result[i] -= theMean;
    };

    exports.standardize = function standardize(values, standardDev, inPlace) {
      if (typeof standardDev === 'undefined') standardDev = exports.standardDeviation(values);
      if (typeof inPlace === 'undefined') inPlace = false;
      var l = values.length;
      var result = inPlace ? values : new Array(l);

      for (var i = 0; i < l; i++) result[i] = values[i] / standardDev;

      return result;
    };

    exports.cumulativeSum = function cumulativeSum(array) {
      var l = array.length;
      var result = new Array(l);
      result[0] = array[0];

      for (var i = 1; i < l; i++) result[i] = result[i - 1] + array[i];

      return result;
    };
  });

  function getNormalizedSpectrum(spectrum, options = {}) {
    var _a;

    let data = {
      x: spectrum.variables.x.data,
      y: spectrum.variables.y.data
    };
    let newSpectrum = {
      variables: {
        x: {
          data: spectrum.variables.x.data,
          units: spectrum.variables.x.units,
          label: spectrum.variables.x.label
        },
        y: {
          data: spectrum.variables.y.data,
          units: spectrum.variables.y.units,
          label: spectrum.variables.y.label
        }
      }
    };
    if (spectrum.title) newSpectrum.title = spectrum.title;
    if (spectrum.dataType) newSpectrum.dataType = spectrum.dataType;
    if (spectrum.meta) newSpectrum.meta = spectrum.meta;
    let {
      from = spectrum.variables.x.min,
      to = spectrum.variables.x.max,
      numberOfPoints,
      filters = [],
      exclusions = [],
      processing = '',
      keepYUnits = false
    } = options;
    let {
      x,
      y
    } = filterX(data, {
      from,
      to
    });

    switch (processing) {
      case 'firstDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `1st derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 1,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;

      case 'secondDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `2nd derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 2,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;

      case 'thirdDerivative':
        if (options.processing) {
          newSpectrum.variables.y.units = '';
          newSpectrum.variables.y.label = newSpectrum.variables.y.label && `3rd derivative of ${newSpectrum.variables.y.label.replace(/\s*\[.*\]/, '')}`;
          y = savitzkyGolay(y, 1, {
            derivative: 3,
            polynomial: 2,
            windowSize: 5
          });
          x = x.slice(2, x.length - 2);
        }

        break;
    }

    if (!keepYUnits && filters.length) {
      // filters change the y axis, we get rid of the units
      newSpectrum.variables.y.units = '';
      newSpectrum.variables.y.label = (_a = newSpectrum.variables.y.label) === null || _a === void 0 ? void 0 : _a.replace(/\s*\[.*\]/, '');
    }

    for (let filter of filters) {
      let filterOptions = filter.options || {};

      switch (filter.name.toLowerCase()) {
        case 'centermean':
          {
            let mean = array.mean(y);
            y = xSubtract(y, mean);
            break;
          }

        case 'dividebysd':
          {
            let std = array.standardDeviation(y);
            y = xDivide(y, std);
            break;
          }

        case 'normalize':
          {
            y = norm(y, {
              sumValue: filterOptions.value ? Number(filterOptions.value) : 1,
              algorithm: 'absolute'
            });
            break;
          }

        case 'rescale':
          {
            y = rescale(y, {
              min: filterOptions.min ? Number(filterOptions.min) : 0,
              max: filterOptions.max ? Number(filterOptions.max) : 1
            });
            break;
          }

        case 'dividebymax':
          {
            let maxValue = max(y);
            y = xDivide(y, maxValue);
            break;
          }

        case 'multiply':
          {
            y = xMultiply(y, filterOptions.value ? Number(filterOptions.value) : 1);
            break;
          }

        case 'add':
          {
            y = xAdd(y, filterOptions.value ? Number(filterOptions.value) : 0);
            break;
          }

        case 'airplsbaseline':
          {
            y = airPLSBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'rollingaveragebaseline':
          {
            y = rollingAverageBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'iterativepolynomialbaseline':
          {
            y = iterativePolynomialBaseline(y, undefined).correctedSpectrum;
            break;
          }

        case 'rollingballbaseline':
          {
            y = rollingBallBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'rollingmedianbaseline':
          {
            y = rollingMedianBaseline(y, filterOptions).correctedSpectrum;
            break;
          }

        case 'ensuregrowing':
          {
            const ans = xyEnsureGrowingX({
              x,
              y
            });
            x = ans.x;
            y = ans.y;
            break;
          }

        case '':
        case undefined:
          break;

        default:
          throw new Error(`Unknown process kind: ${filter.name}`);
      }
    }

    if (!numberOfPoints) {
      data = filterX({
        x,
        y
      }, {
        from,
        to,
        exclusions
      });
    } else {
      data = equallySpaced({
        x,
        y
      }, {
        from,
        to,
        numberOfPoints,
        exclusions
      });
    }

    newSpectrum.variables.x.data = x;
    newSpectrum.variables.x.min = min(x);
    newSpectrum.variables.x.max = max(x);
    newSpectrum.variables.x.isMonotone = xIsMonotone(x);
    newSpectrum.variables.y.data = y;
    newSpectrum.variables.y.min = min(y);
    newSpectrum.variables.y.max = max(y);
    newSpectrum.variables.y.isMonotone = xIsMonotone(y);
    return newSpectrum;
  }

  /*
  The MIT License (MIT)
  Copyright © 2006-2007 Kevin C. Olbrich
  Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

  Permission is hereby granted, free of charge, to any person obtaining a copy of
  this software and associated documentation files (the "Software"), to deal in
  the Software without restriction, including without limitation the rights to
  use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
  of the Software, and to permit persons to whom the Software is furnished to do
  so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
  */
  var quantities = createCommonjsModule(function (module, exports) {
    (function (global, factory) {
      module.exports = factory() ;
    })(commonjsGlobal, function () {
      /**
       * Tests if a value is a string
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a string, false otherwise
       */

      function isString(value) {
        return typeof value === "string" || value instanceof String;
      }
      /*
       * Prefer stricter Number.isFinite if currently supported.
       * To be dropped when ES6 is finalized. Obsolete browsers will
       * have to use ES6 polyfills.
       */


      var isFiniteImpl = Number.isFinite || window.isFinite;
      /**
       * Tests if a value is a number
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a number, false otherwise
       */

      function isNumber(value) {
        // Number.isFinite allows not to consider NaN or '1' as numbers
        return isFiniteImpl(value);
      }
      /*
       * Identity function
       */


      function identity(value) {
        return value;
      }
      /**
       * Returns unique strings from list
       *
       * @param {string[]} strings - array of strings
       *
       *
       * @returns {string[]} a new array of strings without duplicates
       */


      function uniq(strings) {
        var seen = {};
        return strings.filter(function (item) {
          return seen.hasOwnProperty(item) ? false : seen[item] = true;
        });
      }

      function compareArray(array1, array2) {
        if (array2.length !== array1.length) {
          return false;
        }

        for (var i = 0; i < array1.length; i++) {
          if (array2[i].compareArray) {
            if (!array2[i].compareArray(array1[i])) {
              return false;
            }
          }

          if (array2[i] !== array1[i]) {
            return false;
          }
        }

        return true;
      }

      function assign(target, properties) {
        Object.keys(properties).forEach(function (key) {
          target[key] = properties[key];
        });
      }
      /**
       * Safely multiplies numbers while avoiding floating errors
       * like 0.1 * 0.1 => 0.010000000000000002
       *
       * @param {...number} numbers - numbers to multiply
       *
       * @returns {number} result
       */


      function mulSafe() {
        var result = 1,
            decimals = 0;

        for (var i = 0; i < arguments.length; i++) {
          var arg = arguments[i];
          decimals = decimals + getFractional(arg);
          result *= arg;
        }

        return decimals !== 0 ? round(result, decimals) : result;
      }
      /**
       * Safely divides two numbers while avoiding floating errors
       * like 0.3 / 0.05 => 5.999999999999999
       *
       * @returns {number} result
       * @param {number} num Numerator
       * @param {number} den Denominator
       */


      function divSafe(num, den) {
        if (den === 0) {
          throw new Error("Divide by zero");
        }

        var factor = Math.pow(10, getFractional(den));
        var invDen = factor / (factor * den);
        return mulSafe(num, invDen);
      }
      /**
       * Rounds value at the specified number of decimals
       *
       * @param {number} val - value to round
       * @param {number} decimals - number of decimals
       *
       * @returns {number} rounded number
       */


      function round(val, decimals) {
        return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
      }

      function getFractional(num) {
        // Check for NaNs or Infinities
        if (!isFinite(num)) {
          return 0;
        } // Faster than parsing strings
        // http://jsperf.com/count-decimals/2


        var count = 0;

        while (num % 1 !== 0) {
          num *= 10;
          count++;
        }

        return count;
      }
      /**
       * Custom error type definition
       * @constructor
       */


      function QtyError() {
        var err;

        if (!this) {
          // Allows to instantiate QtyError without new()
          err = Object.create(QtyError.prototype);
          QtyError.apply(err, arguments);
          return err;
        }

        err = Error.apply(this, arguments);
        this.name = "QtyError";
        this.message = err.message;
        this.stack = err.stack;
      }

      QtyError.prototype = Object.create(Error.prototype, {
        constructor: {
          value: QtyError
        }
      });
      /*
       * Throws incompatible units error
       * @param {string} left - units
       * @param {string} right - units incompatible with first argument
       * @throws "Incompatible units" error
       */

      function throwIncompatibleUnits(left, right) {
        throw new QtyError("Incompatible units: " + left + " and " + right);
      }

      var UNITS = {
        /* prefixes */
        "<googol>": [["googol"], 1e100, "prefix"],
        "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
        "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
        "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
        "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
        "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
        "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
        "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
        "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
        "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
        "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
        "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
        "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
        "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
        "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
        "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
        "<kilo>": [["k", "kilo"], 1e3, "prefix"],
        "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
        "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
        "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
        "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
        "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
        "<micro>": [["u", "\u03BC"
        /*µ as greek letter*/
        , "\u00B5"
        /*µ as micro sign*/
        , "Micro", "mc", "micro"], 1e-6, "prefix"],
        "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
        "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
        "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
        "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
        "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
        "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
        "<1>": [["1", "<1>"], 1, ""],

        /* length units */
        "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
        "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
        "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
        "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
        "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
        "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
        "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
        "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
        "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
        "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
        "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
        "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
        "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
        "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
        "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
        "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
        "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
        "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
        "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
        "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
        "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],

        /* mass */
        "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
        "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
        "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
        "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
        "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
        "<metric-ton>": [["tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
        "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
        "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
        "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
        "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
        "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
        "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
        "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],

        /* area */
        "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
        "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
        "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],

        /* volume */
        "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
        "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],

        /* speed */
        "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
        "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
        "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
        "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],

        /* acceleration */
        "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
        "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],

        /* temperature_difference */
        "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
        "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
        "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
        "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
        "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
        "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
        "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
        "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],

        /* time */
        "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
        "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
        "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
        "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
        "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
        "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
        "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
        "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
        "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],

        /* pressure */
        "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
        "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],

        /* viscosity */
        "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
        "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],

        /* substance */
        "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],

        /* concentration */
        "<molar>": [["M", "molar"], 1000, "concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
        "<wtpercent>": [["wt%", "wtpercent"], 10, "concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],

        /* activity */
        "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
        "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],

        /* capacitance */
        "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],

        /* charge */
        "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
        "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],

        /* current */
        "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],

        /* conductance */
        "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],

        /* inductance */
        "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],

        /* potential */
        "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],

        /* resistance */
        "<ohm>": [["Ohm", "ohm", "\u03A9"
        /*Ω as greek letter*/
        , "\u2126"
        /*Ω as ohm sign*/
        ], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],

        /* magnetism */
        "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
        "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],

        /* energy */
        "<joule>": [["J", "joule", "Joule", "joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
        "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],

        /* force */
        "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
        "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
        "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],

        /* frequency */
        "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],

        /* angle */
        "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
        "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
        "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
        "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],

        /* rotation */
        "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
        "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],

        /* information */
        "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
        "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],

        /* information rate */
        "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
        "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],

        /* currency */
        "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
        "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],

        /* luminosity */
        "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
        "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
        "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],

        /* power */
        "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
        "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],

        /* radiation */
        "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
        "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
        "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],

        /* rate */
        "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
        "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
        "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],

        /* resolution / typography */
        "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
        "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
        "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
        "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],

        /* other */
        "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
        "<each>": [["each"], 1.0, "counting", ["<each>"]],
        "<count>": [["count"], 1.0, "counting", ["<each>"]],
        "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
        "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
        "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
        "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
        "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
        "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
        "<ppt>": [["ppt"], 1e-9, "prefix_only", ["<1>"]],
        "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
        "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
      };
      var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
      var UNITY = "<1>";
      var UNITY_ARRAY = [UNITY]; // Setup

      /**
       * Asserts unit definition is valid
       *
       * @param {string} unitDef - Name of unit to test
       * @param {Object} definition - Definition of unit to test
       *
       * @returns {void}
       * @throws {QtyError} if unit definition is not valid
       */

      function validateUnitDefinition(unitDef, definition) {
        var scalar = definition[1];
        var numerator = definition[3] || [];
        var denominator = definition[4] || [];

        if (!isNumber(scalar)) {
          throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
        }

        numerator.forEach(function (unit) {
          if (UNITS[unit] === undefined) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
          }
        });
        denominator.forEach(function (unit) {
          if (UNITS[unit] === undefined) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
          }
        });
      }

      var PREFIX_VALUES = {};
      var PREFIX_MAP = {};
      var UNIT_VALUES = {};
      var UNIT_MAP = {};
      var OUTPUT_MAP = {};

      for (var unitDef in UNITS) {
        if (UNITS.hasOwnProperty(unitDef)) {
          var definition = UNITS[unitDef];

          if (definition[2] === "prefix") {
            PREFIX_VALUES[unitDef] = definition[1];

            for (var i = 0; i < definition[0].length; i++) {
              PREFIX_MAP[definition[0][i]] = unitDef;
            }
          } else {
            validateUnitDefinition(unitDef, definition);
            UNIT_VALUES[unitDef] = {
              scalar: definition[1],
              numerator: definition[3],
              denominator: definition[4]
            };

            for (var j = 0; j < definition[0].length; j++) {
              UNIT_MAP[definition[0][j]] = unitDef;
            }
          }

          OUTPUT_MAP[unitDef] = definition[0][0];
        }
      }
      /**
       * Returns a list of available units of kind
       *
       * @param {string} [kind] - kind of units
       * @returns {array} names of units
       * @throws {QtyError} if kind is unknown
       */


      function getUnits(kind) {
        var i;
        var units = [];
        var unitKeys = Object.keys(UNITS);

        if (typeof kind === "undefined") {
          for (i = 0; i < unitKeys.length; i++) {
            if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
              units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
            }
          }
        } else if (this.getKinds().indexOf(kind) === -1) {
          throw new QtyError("Kind not recognized");
        } else {
          for (i = 0; i < unitKeys.length; i++) {
            if (UNITS[unitKeys[i]][2] === kind) {
              units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
            }
          }
        }

        return units.sort(function (a, b) {
          if (a.toLowerCase() < b.toLowerCase()) {
            return -1;
          }

          if (a.toLowerCase() > b.toLowerCase()) {
            return 1;
          }

          return 0;
        });
      }
      /**
       * Returns a list of alternative names for a unit
       *
       * @param {string} unitName - name of unit
       * @returns {string[]} aliases for unit
       * @throws {QtyError} if unit is unknown
       */


      function getAliases(unitName) {
        if (!UNIT_MAP[unitName]) {
          throw new QtyError("Unit not recognized");
        }

        return UNITS[UNIT_MAP[unitName]][0];
      }

      var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];
      /*
      calculates the unit signature id for use in comparing compatible units and simplification
      the signature is based on a simple classification of units and is based on the following publication
       Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
      21(8), Aug 1995, pp.651-661
      doi://10.1109/32.403789
      http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
      */

      function unitSignature() {
        if (this.signature) {
          return this.signature;
        }

        var vector = unitSignatureVector.call(this);

        for (var i = 0; i < vector.length; i++) {
          vector[i] *= Math.pow(20, i);
        }

        return vector.reduce(function (previous, current) {
          return previous + current;
        }, 0);
      } // calculates the unit signature vector used by unit_signature


      function unitSignatureVector() {
        if (!this.isBase()) {
          return unitSignatureVector.call(this.toBase());
        }

        var vector = new Array(SIGNATURE_VECTOR.length);

        for (var i = 0; i < vector.length; i++) {
          vector[i] = 0;
        }

        var r, n;

        for (var j = 0; j < this.numerator.length; j++) {
          if (r = UNITS[this.numerator[j]]) {
            n = SIGNATURE_VECTOR.indexOf(r[2]);

            if (n >= 0) {
              vector[n] = vector[n] + 1;
            }
          }
        }

        for (var k = 0; k < this.denominator.length; k++) {
          if (r = UNITS[this.denominator[k]]) {
            n = SIGNATURE_VECTOR.indexOf(r[2]);

            if (n >= 0) {
              vector[n] = vector[n] - 1;
            }
          }
        }

        return vector;
      }

      var SIGN = "[+-]";
      var INTEGER = "\\d+";
      var SIGNED_INTEGER = SIGN + "?" + INTEGER;
      var FRACTION = "\\." + INTEGER;
      var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
      var EXPONENT = "[Ee]" + SIGNED_INTEGER;
      var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
      var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
      var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
      var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
      var POWER_OP = "\\^|\\*{2}"; // Allow unit powers representing scalar, length, area, volume; 4 is for some
      // special case representations in SI base units.

      var SAFE_POWER = "[01234]";
      var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
      var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");
      /* parse a string into a unit object.
       * Typical formats like :
       * "5.6 kg*m/s^2"
       * "5.6 kg*m*s^-2"
       * "5.6 kilogram*meter*second^-2"
       * "2.2 kPa"
       * "37 degC"
       * "1"  -- creates a unitless constant with value 1
       * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
       * 6'4"  -- recognized as 6 feet + 4 inches
       * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
       */

      function parse(val) {
        if (!isString(val)) {
          val = val.toString();
        }

        val = val.trim();
        var result = QTY_STRING_REGEX.exec(val);

        if (!result) {
          throw new QtyError(val + ": Quantity not recognized");
        }

        var scalarMatch = result[1];

        if (scalarMatch) {
          // Allow whitespaces between sign and scalar for loose parsing
          scalarMatch = scalarMatch.replace(/\s/g, "");
          this.scalar = parseFloat(scalarMatch);
        } else {
          this.scalar = 1;
        }

        var top = result[2];
        var bottom = result[3];
        var n, x, nx; // TODO DRY me

        while (result = TOP_REGEX.exec(top)) {
          n = parseFloat(result[2]);

          if (isNaN(n)) {
            // Prevents infinite loops
            throw new QtyError("Unit exponent is not a number");
          } // Disallow unrecognized unit even if exponent is 0


          if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
            throw new QtyError("Unit not recognized");
          }

          x = result[1] + " ";
          nx = "";

          for (var i = 0; i < Math.abs(n); i++) {
            nx += x;
          }

          if (n >= 0) {
            top = top.replace(result[0], nx);
          } else {
            bottom = bottom ? bottom + nx : nx;
            top = top.replace(result[0], "");
          }
        }

        while (result = BOTTOM_REGEX.exec(bottom)) {
          n = parseFloat(result[2]);

          if (isNaN(n)) {
            // Prevents infinite loops
            throw new QtyError("Unit exponent is not a number");
          } // Disallow unrecognized unit even if exponent is 0


          if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
            throw new QtyError("Unit not recognized");
          }

          x = result[1] + " ";
          nx = "";

          for (var j = 0; j < n; j++) {
            nx += x;
          }

          bottom = bottom.replace(result[0], nx);
        }

        if (top) {
          this.numerator = parseUnits(top.trim());
        }

        if (bottom) {
          this.denominator = parseUnits(bottom.trim());
        }
      }

      var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
        return b.length - a.length;
      }).join("|");
      var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
        return b.length - a.length;
      }).join("|");
      /*
       * Minimal boundary regex to support units with Unicode characters
       * \b only works for ASCII
       */

      var BOUNDARY_REGEX = "\\b|$";
      var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
      var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
      var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences

      var parsedUnitsCache = {};
      /**
       * Parses and converts units string to normalized unit array.
       * Result is cached to speed up next calls.
       *
       * @param {string} units Units string
       * @returns {string[]} Array of normalized units
       *
       * @example
       * // Returns ["<second>", "<meter>", "<second>"]
       * parseUnits("s m s");
       *
       */

      function parseUnits(units) {
        var cached = parsedUnitsCache[units];

        if (cached) {
          return cached;
        }

        var unitMatch,
            normalizedUnits = []; // Scan

        if (!UNIT_TEST_REGEX.test(units)) {
          throw new QtyError("Unit not recognized");
        }

        while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
          normalizedUnits.push(unitMatch.slice(1));
        }

        normalizedUnits = normalizedUnits.map(function (item) {
          return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
        }); // Flatten and remove null elements

        normalizedUnits = normalizedUnits.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        normalizedUnits = normalizedUnits.filter(function (item) {
          return item;
        });
        parsedUnitsCache[units] = normalizedUnits;
        return normalizedUnits;
      }
      /**
       * Parses a string as a quantity
       * @param {string} value - quantity as text
       * @throws if value is not a string
       * @returns {Qty|null} Parsed quantity or null if unrecognized
       */


      function globalParse(value) {
        if (!isString(value)) {
          throw new QtyError("Argument should be a string");
        }

        try {
          return this(value);
        } catch (e) {
          return null;
        }
      }
      /**
       * Tests if a value is a Qty instance
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a Qty instance, false otherwise
       */


      function isQty(value) {
        return value instanceof Qty;
      }

      function Qty(initValue, initUnits) {
        assertValidConstructorArgs.apply(null, arguments);

        if (!isQty(this)) {
          return new Qty(initValue, initUnits);
        }

        this.scalar = null;
        this.baseScalar = null;
        this.signature = null;
        this._conversionCache = {};
        this.numerator = UNITY_ARRAY;
        this.denominator = UNITY_ARRAY;

        if (isDefinitionObject(initValue)) {
          this.scalar = initValue.scalar;
          this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
          this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
        } else if (initUnits) {
          parse.call(this, initUnits);
          this.scalar = initValue;
        } else {
          parse.call(this, initValue);
        } // math with temperatures is very limited


        if (this.denominator.join("*").indexOf("temp") >= 0) {
          throw new QtyError("Cannot divide with temperatures");
        }

        if (this.numerator.join("*").indexOf("temp") >= 0) {
          if (this.numerator.length > 1) {
            throw new QtyError("Cannot multiply by temperatures");
          }

          if (!compareArray(this.denominator, UNITY_ARRAY)) {
            throw new QtyError("Cannot divide with temperatures");
          }
        }

        this.initValue = initValue;
        updateBaseScalar.call(this);

        if (this.isTemperature() && this.baseScalar < 0) {
          throw new QtyError("Temperatures must not be less than absolute zero");
        }
      }

      Qty.prototype = {
        // Properly set up constructor
        constructor: Qty
      };
      /**
       * Asserts constructor arguments are valid
       *
       * @param {*} value - Value to test
       * @param {string} [units] - Optional units when value is passed as a number
       *
       * @returns {void}
       * @throws {QtyError} if constructor arguments are invalid
       */

      function assertValidConstructorArgs(value, units) {
        if (units) {
          if (!(isNumber(value) && isString(units))) {
            throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
          }
        } else {
          if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
            throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
          }
        }
      }
      /**
       * Tests if a value is a Qty definition object
       *
       * @param {*} value - Value to test
       *
       * @returns {boolean} true if value is a definition object, false otherwise
       */


      function isDefinitionObject(value) {
        return value && typeof value === "object" && value.hasOwnProperty("scalar");
      }

      function updateBaseScalar() {
        if (this.baseScalar) {
          return this.baseScalar;
        }

        if (this.isBase()) {
          this.baseScalar = this.scalar;
          this.signature = unitSignature.call(this);
        } else {
          var base = this.toBase();
          this.baseScalar = base.scalar;
          this.signature = base.signature;
        }
      }

      var KINDS = {
        "-312078": "elastance",
        "-312058": "resistance",
        "-312038": "inductance",
        "-152058": "potential",
        "-152040": "magnetism",
        "-152038": "magnetism",
        "-7997": "specific_volume",
        "-79": "snap",
        "-59": "jolt",
        "-39": "acceleration",
        "-38": "radiation",
        "-20": "frequency",
        "-19": "speed",
        "-18": "viscosity",
        "-17": "volumetric_flow",
        "-1": "wavenumber",
        "0": "unitless",
        "1": "length",
        "2": "area",
        "3": "volume",
        "20": "time",
        "400": "temperature",
        "7941": "yank",
        "7942": "power",
        "7959": "pressure",
        "7961": "force",
        "7962": "energy",
        "7979": "viscosity",
        "7981": "momentum",
        "7982": "angular_momentum",
        "7997": "density",
        "7998": "area_density",
        "8000": "mass",
        "152020": "radiation_exposure",
        "159999": "magnetism",
        "160000": "current",
        "160020": "charge",
        "312058": "conductance",
        "312078": "capacitance",
        "3199980": "activity",
        "3199997": "molar_concentration",
        "3200000": "substance",
        "63999998": "illuminance",
        "64000000": "luminous_power",
        "1280000000": "currency",
        "25599999980": "information_rate",
        "25600000000": "information",
        "511999999980": "angular_velocity",
        "512000000000": "angle"
      };
      /**
       * Returns the list of available well-known kinds of units, e.g.
       * "radiation" or "length".
       *
       * @returns {string[]} names of kinds of units
       */

      function getKinds() {
        return uniq(Object.keys(KINDS).map(function (knownSignature) {
          return KINDS[knownSignature];
        }));
      }

      Qty.prototype.kind = function () {
        return KINDS[this.signature.toString()];
      };

      assign(Qty.prototype, {
        isDegrees: function () {
          // signature may not have been calculated yet
          return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
        },
        isTemperature: function () {
          return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
        }
      });

      function subtractTemperatures(lhs, rhs) {
        var lhsUnits = lhs.units();
        var rhsConverted = rhs.to(lhsUnits);
        var dstDegrees = Qty(getDegreeUnits(lhsUnits));
        return Qty({
          "scalar": lhs.scalar - rhsConverted.scalar,
          "numerator": dstDegrees.numerator,
          "denominator": dstDegrees.denominator
        });
      }

      function subtractTempDegrees(temp, deg) {
        var tempDegrees = deg.to(getDegreeUnits(temp.units()));
        return Qty({
          "scalar": temp.scalar - tempDegrees.scalar,
          "numerator": temp.numerator,
          "denominator": temp.denominator
        });
      }

      function addTempDegrees(temp, deg) {
        var tempDegrees = deg.to(getDegreeUnits(temp.units()));
        return Qty({
          "scalar": temp.scalar + tempDegrees.scalar,
          "numerator": temp.numerator,
          "denominator": temp.denominator
        });
      }

      function getDegreeUnits(units) {
        if (units === "tempK") {
          return "degK";
        } else if (units === "tempC") {
          return "degC";
        } else if (units === "tempF") {
          return "degF";
        } else if (units === "tempR") {
          return "degR";
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }
      }

      function toDegrees(src, dst) {
        var srcDegK = toDegK(src);
        var dstUnits = dst.units();
        var dstScalar;

        if (dstUnits === "degK") {
          dstScalar = srcDegK.scalar;
        } else if (dstUnits === "degC") {
          dstScalar = srcDegK.scalar;
        } else if (dstUnits === "degF") {
          dstScalar = srcDegK.scalar * 9 / 5;
        } else if (dstUnits === "degR") {
          dstScalar = srcDegK.scalar * 9 / 5;
        } else {
          throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
        }

        return Qty({
          "scalar": dstScalar,
          "numerator": dst.numerator,
          "denominator": dst.denominator
        });
      }

      function toDegK(qty) {
        var units = qty.units();
        var q;

        if (units.match(/(deg)[CFRK]/)) {
          q = qty.baseScalar;
        } else if (units === "tempK") {
          q = qty.scalar;
        } else if (units === "tempC") {
          q = qty.scalar;
        } else if (units === "tempF") {
          q = qty.scalar * 5 / 9;
        } else if (units === "tempR") {
          q = qty.scalar * 5 / 9;
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }

        return Qty({
          "scalar": q,
          "numerator": ["<kelvin>"],
          "denominator": UNITY_ARRAY
        });
      }

      function toTemp(src, dst) {
        var dstUnits = dst.units();
        var dstScalar;

        if (dstUnits === "tempK") {
          dstScalar = src.baseScalar;
        } else if (dstUnits === "tempC") {
          dstScalar = src.baseScalar - 273.15;
        } else if (dstUnits === "tempF") {
          dstScalar = src.baseScalar * 9 / 5 - 459.67;
        } else if (dstUnits === "tempR") {
          dstScalar = src.baseScalar * 9 / 5;
        } else {
          throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
        }

        return Qty({
          "scalar": dstScalar,
          "numerator": dst.numerator,
          "denominator": dst.denominator
        });
      }

      function toTempK(qty) {
        var units = qty.units();
        var q;

        if (units.match(/(deg)[CFRK]/)) {
          q = qty.baseScalar;
        } else if (units === "tempK") {
          q = qty.scalar;
        } else if (units === "tempC") {
          q = qty.scalar + 273.15;
        } else if (units === "tempF") {
          q = (qty.scalar + 459.67) * 5 / 9;
        } else if (units === "tempR") {
          q = qty.scalar * 5 / 9;
        } else {
          throw new QtyError("Unknown type for temp conversion from: " + units);
        }

        return Qty({
          "scalar": q,
          "numerator": ["<temp-K>"],
          "denominator": UNITY_ARRAY
        });
      }

      assign(Qty.prototype, {
        /**
         * Converts to other compatible units.
         * Instance's converted quantities are cached for faster subsequent calls.
         *
         * @param {(string|Qty)} other - Target units as string or retrieved from
         *                               other Qty instance (scalar is ignored)
         *
         * @returns {Qty} New converted Qty instance with target units
         *
         * @throws {QtyError} if target units are incompatible
         *
         * @example
         * var weight = Qty("25 kg");
         * weight.to("lb"); // => Qty("55.11556554621939 lbs");
         * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
         */
        to: function (other) {
          var cached, target;

          if (other === undefined || other === null) {
            return this;
          }

          if (!isString(other)) {
            return this.to(other.units());
          }

          cached = this._conversionCache[other];

          if (cached) {
            return cached;
          } // Instantiating target to normalize units


          target = Qty(other);

          if (target.units() === this.units()) {
            return this;
          }

          if (!this.isCompatible(target)) {
            if (this.isInverse(target)) {
              target = this.inverse().to(other);
            } else {
              throwIncompatibleUnits(this.units(), target.units());
            }
          } else {
            if (target.isTemperature()) {
              target = toTemp(this, target);
            } else if (target.isDegrees()) {
              target = toDegrees(this, target);
            } else {
              var q = divSafe(this.baseScalar, target.baseScalar);
              target = Qty({
                "scalar": q,
                "numerator": target.numerator,
                "denominator": target.denominator
              });
            }
          }

          this._conversionCache[other] = target;
          return target;
        },
        // convert to base SI units
        // results of the conversion are cached so subsequent calls to this will be fast
        toBase: function () {
          if (this.isBase()) {
            return this;
          }

          if (this.isTemperature()) {
            return toTempK(this);
          }

          var cached = baseUnitCache[this.units()];

          if (!cached) {
            cached = toBaseUnits(this.numerator, this.denominator);
            baseUnitCache[this.units()] = cached;
          }

          return cached.mul(this.scalar);
        },
        // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
        toFloat: function () {
          if (this.isUnitless()) {
            return this.scalar;
          }

          throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
        },

        /**
         * Returns the nearest multiple of quantity passed as
         * precision
         *
         * @param {(Qty|string|number)} precQuantity - Quantity, string formated
         *   quantity or number as expected precision
         *
         * @returns {Qty} Nearest multiple of precQuantity
         *
         * @example
         * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
         * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
         * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
         * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
         *
         */
        toPrec: function (precQuantity) {
          if (isString(precQuantity)) {
            precQuantity = Qty(precQuantity);
          }

          if (isNumber(precQuantity)) {
            precQuantity = Qty(precQuantity + " " + this.units());
          }

          if (!this.isUnitless()) {
            precQuantity = precQuantity.to(this.units());
          } else if (!precQuantity.isUnitless()) {
            throwIncompatibleUnits(this.units(), precQuantity.units());
          }

          if (precQuantity.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
          return Qty(precRoundedResult + this.units());
        }
      });
      /**
       * Configures and returns a fast function to convert
       * Number values from units to others.
       * Useful to efficiently convert large array of values
       * with same units into others with iterative methods.
       * Does not take care of rounding issues.
       *
       * @param {string} srcUnits Units of values to convert
       * @param {string} dstUnits Units to convert to
       *
       * @returns {Function} Converting function accepting Number value
       *   and returning converted value
       *
       * @throws "Incompatible units" if units are incompatible
       *
       * @example
       * // Converting large array of numbers with the same units
       * // into other units
       * var converter = Qty.swiftConverter("m/h", "ft/s");
       * var convertedSerie = largeSerie.map(converter);
       *
       */

      function swiftConverter(srcUnits, dstUnits) {
        var srcQty = Qty(srcUnits);
        var dstQty = Qty(dstUnits);

        if (srcQty.eq(dstQty)) {
          return identity;
        }

        var convert;

        if (!srcQty.isTemperature()) {
          convert = function (value) {
            return value * srcQty.baseScalar / dstQty.baseScalar;
          };
        } else {
          convert = function (value) {
            // TODO Not optimized
            return srcQty.mul(value).to(dstQty).scalar;
          };
        }

        return function converter(value) {
          var i, length, result;

          if (!Array.isArray(value)) {
            return convert(value);
          } else {
            length = value.length;
            result = [];

            for (i = 0; i < length; i++) {
              result.push(convert(value[i]));
            }

            return result;
          }
        };
      }

      var baseUnitCache = {};

      function toBaseUnits(numerator, denominator) {
        var num = [];
        var den = [];
        var q = 1;
        var unit;

        for (var i = 0; i < numerator.length; i++) {
          unit = numerator[i];

          if (PREFIX_VALUES[unit]) {
            // workaround to fix
            // 0.1 * 0.1 => 0.010000000000000002
            q = mulSafe(q, PREFIX_VALUES[unit]);
          } else {
            if (UNIT_VALUES[unit]) {
              q *= UNIT_VALUES[unit].scalar;

              if (UNIT_VALUES[unit].numerator) {
                num.push(UNIT_VALUES[unit].numerator);
              }

              if (UNIT_VALUES[unit].denominator) {
                den.push(UNIT_VALUES[unit].denominator);
              }
            }
          }
        }

        for (var j = 0; j < denominator.length; j++) {
          unit = denominator[j];

          if (PREFIX_VALUES[unit]) {
            q /= PREFIX_VALUES[unit];
          } else {
            if (UNIT_VALUES[unit]) {
              q /= UNIT_VALUES[unit].scalar;

              if (UNIT_VALUES[unit].numerator) {
                den.push(UNIT_VALUES[unit].numerator);
              }

              if (UNIT_VALUES[unit].denominator) {
                num.push(UNIT_VALUES[unit].denominator);
              }
            }
          }
        } // Flatten


        num = num.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        den = den.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        return Qty({
          "scalar": q,
          "numerator": num,
          "denominator": den
        });
      }

      Qty.parse = globalParse;
      Qty.getUnits = getUnits;
      Qty.getAliases = getAliases;
      Qty.mulSafe = mulSafe;
      Qty.divSafe = divSafe;
      Qty.getKinds = getKinds;
      Qty.swiftConverter = swiftConverter;
      Qty.Error = QtyError;
      assign(Qty.prototype, {
        // Returns new instance with units of this
        add: function (other) {
          if (isString(other)) {
            other = Qty(other);
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.isTemperature() && other.isTemperature()) {
            throw new QtyError("Cannot add two temperatures");
          } else if (this.isTemperature()) {
            return addTempDegrees(this, other);
          } else if (other.isTemperature()) {
            return addTempDegrees(other, this);
          }

          return Qty({
            "scalar": this.scalar + other.to(this).scalar,
            "numerator": this.numerator,
            "denominator": this.denominator
          });
        },
        sub: function (other) {
          if (isString(other)) {
            other = Qty(other);
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.isTemperature() && other.isTemperature()) {
            return subtractTemperatures(this, other);
          } else if (this.isTemperature()) {
            return subtractTempDegrees(this, other);
          } else if (other.isTemperature()) {
            throw new QtyError("Cannot subtract a temperature from a differential degree unit");
          }

          return Qty({
            "scalar": this.scalar - other.to(this).scalar,
            "numerator": this.numerator,
            "denominator": this.denominator
          });
        },
        mul: function (other) {
          if (isNumber(other)) {
            return Qty({
              "scalar": mulSafe(this.scalar, other),
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          } else if (isString(other)) {
            other = Qty(other);
          }

          if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
            throw new QtyError("Cannot multiply by temperatures");
          } // Quantities should be multiplied with same units if compatible, with base units else


          var op1 = this;
          var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
          // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

          if (op1.isCompatible(op2) && op1.signature !== 400) {
            op2 = op2.to(op1);
          }

          var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
          return Qty({
            "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
            "numerator": numdenscale[0],
            "denominator": numdenscale[1]
          });
        },
        div: function (other) {
          if (isNumber(other)) {
            if (other === 0) {
              throw new QtyError("Divide by zero");
            }

            return Qty({
              "scalar": this.scalar / other,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          } else if (isString(other)) {
            other = Qty(other);
          }

          if (other.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          if (other.isTemperature()) {
            throw new QtyError("Cannot divide with temperatures");
          } else if (this.isTemperature() && !other.isUnitless()) {
            throw new QtyError("Cannot divide with temperatures");
          } // Quantities should be multiplied with same units if compatible, with base units else


          var op1 = this;
          var op2 = other; // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
          // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"

          if (op1.isCompatible(op2) && op1.signature !== 400) {
            op2 = op2.to(op1);
          }

          var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
          return Qty({
            "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
            "numerator": numdenscale[0],
            "denominator": numdenscale[1]
          });
        },
        // Returns a Qty that is the inverse of this Qty,
        inverse: function () {
          if (this.isTemperature()) {
            throw new QtyError("Cannot divide with temperatures");
          }

          if (this.scalar === 0) {
            throw new QtyError("Divide by zero");
          }

          return Qty({
            "scalar": 1 / this.scalar,
            "numerator": this.denominator,
            "denominator": this.numerator
          });
        }
      });

      function cleanTerms(num1, den1, num2, den2) {
        function notUnity(val) {
          return val !== UNITY;
        }

        num1 = num1.filter(notUnity);
        num2 = num2.filter(notUnity);
        den1 = den1.filter(notUnity);
        den2 = den2.filter(notUnity);
        var combined = {};

        function combineTerms(terms, direction) {
          var k;
          var prefix;
          var prefixValue;

          for (var i = 0; i < terms.length; i++) {
            if (PREFIX_VALUES[terms[i]]) {
              k = terms[i + 1];
              prefix = terms[i];
              prefixValue = PREFIX_VALUES[prefix];
              i++;
            } else {
              k = terms[i];
              prefix = null;
              prefixValue = 1;
            }

            if (k && k !== UNITY) {
              if (combined[k]) {
                combined[k][0] += direction;
                var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
                combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
              } else {
                combined[k] = [direction, k, prefix, 1, 1];
              }
            }
          }
        }

        combineTerms(num1, 1);
        combineTerms(den1, -1);
        combineTerms(num2, 1);
        combineTerms(den2, -1);
        var num = [];
        var den = [];
        var scale = 1;

        for (var prop in combined) {
          if (combined.hasOwnProperty(prop)) {
            var item = combined[prop];
            var n;

            if (item[0] > 0) {
              for (n = 0; n < item[0]; n++) {
                num.push(item[2] === null ? item[1] : [item[2], item[1]]);
              }
            } else if (item[0] < 0) {
              for (n = 0; n < -item[0]; n++) {
                den.push(item[2] === null ? item[1] : [item[2], item[1]]);
              }
            }

            scale *= divSafe(item[3], item[4]);
          }
        }

        if (num.length === 0) {
          num = UNITY_ARRAY;
        }

        if (den.length === 0) {
          den = UNITY_ARRAY;
        } // Flatten


        num = num.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        den = den.reduce(function (a, b) {
          return a.concat(b);
        }, []);
        return [num, den, scale];
      }

      assign(Qty.prototype, {
        eq: function (other) {
          return this.compareTo(other) === 0;
        },
        lt: function (other) {
          return this.compareTo(other) === -1;
        },
        lte: function (other) {
          return this.eq(other) || this.lt(other);
        },
        gt: function (other) {
          return this.compareTo(other) === 1;
        },
        gte: function (other) {
          return this.eq(other) || this.gt(other);
        },
        // Compare two Qty objects. Throws an exception if they are not of compatible types.
        // Comparisons are done based on the value of the quantity in base SI units.
        //
        // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
        //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
        //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
        //
        //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
        //     Qty("10S").inverse().compareTo("10ohm") == -1
        //     Qty("10ohm").inverse().compareTo("10S") == -1
        //
        //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
        compareTo: function (other) {
          if (isString(other)) {
            return this.compareTo(Qty(other));
          }

          if (!this.isCompatible(other)) {
            throwIncompatibleUnits(this.units(), other.units());
          }

          if (this.baseScalar < other.baseScalar) {
            return -1;
          } else if (this.baseScalar === other.baseScalar) {
            return 0;
          } else if (this.baseScalar > other.baseScalar) {
            return 1;
          }
        },
        // Return true if quantities and units match
        // Unit("100 cm").same(Unit("100 cm"))  # => true
        // Unit("100 cm").same(Unit("1 m"))     # => false
        same: function (other) {
          return this.scalar === other.scalar && this.units() === other.units();
        }
      });
      assign(Qty.prototype, {
        // returns true if no associated units
        // false, even if the units are "unitless" like 'radians, each, etc'
        isUnitless: function () {
          return [this.numerator, this.denominator].every(function (item) {
            return compareArray(item, UNITY_ARRAY);
          });
        },

        /*
        check to see if units are compatible, but not the scalar part
        this check is done by comparing signatures for performance reasons
        if passed a string, it will create a unit object with the string and then do the comparison
        this permits a syntax like:
        unit =~ "mm"
        if you want to do a regexp on the unit string do this ...
        unit.units =~ /regexp/
        */
        isCompatible: function (other) {
          if (isString(other)) {
            return this.isCompatible(Qty(other));
          }

          if (!isQty(other)) {
            return false;
          }

          if (other.signature !== undefined) {
            return this.signature === other.signature;
          } else {
            return false;
          }
        },

        /*
        check to see if units are inverse of each other, but not the scalar part
        this check is done by comparing signatures for performance reasons
        if passed a string, it will create a unit object with the string and then do the comparison
        this permits a syntax like:
        unit =~ "mm"
        if you want to do a regexp on the unit string do this ...
        unit.units =~ /regexp/
        */
        isInverse: function (other) {
          return this.inverse().isCompatible(other);
        },
        // Returns 'true' if the Unit is represented in base units
        isBase: function () {
          if (this._isBase !== undefined) {
            return this._isBase;
          }

          if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
            this._isBase = true;
            return this._isBase;
          }

          this.numerator.concat(this.denominator).forEach(function (item) {
            if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
              this._isBase = false;
            }
          }, this);

          if (this._isBase === false) {
            return this._isBase;
          }

          this._isBase = true;
          return this._isBase;
        }
      });

      function NestedMap() {}

      NestedMap.prototype.get = function (keys) {
        // Allows to pass key1, key2, ... instead of [key1, key2, ...]
        if (arguments.length > 1) {
          // Slower with Firefox but faster with Chrome than
          // Array.prototype.slice.call(arguments)
          // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
          keys = Array.apply(null, arguments);
        }

        return keys.reduce(function (map, key, index) {
          if (map) {
            var childMap = map[key];

            if (index === keys.length - 1) {
              return childMap ? childMap.data : undefined;
            } else {
              return childMap;
            }
          }
        }, this);
      };

      NestedMap.prototype.set = function (keys, value) {
        if (arguments.length > 2) {
          keys = Array.prototype.slice.call(arguments, 0, -1);
          value = arguments[arguments.length - 1];
        }

        return keys.reduce(function (map, key, index) {
          var childMap = map[key];

          if (childMap === undefined) {
            childMap = map[key] = {};
          }

          if (index === keys.length - 1) {
            childMap.data = value;
            return value;
          } else {
            return childMap;
          }
        }, this);
      };
      /**
       * Default formatter
       *
       * @param {number} scalar - scalar value
       * @param {string} units - units as string
       *
       * @returns {string} formatted result
       */


      function defaultFormatter(scalar, units) {
        return (scalar + " " + units).trim();
      }
      /**
       *
       * Configurable Qty default formatter
       *
       * @type {function}
       *
       * @param {number} scalar
       * @param {string} units
       *
       * @returns {string} formatted result
       */


      Qty.formatter = defaultFormatter;
      assign(Qty.prototype, {
        // returns the 'unit' part of the Unit object without the scalar
        units: function () {
          if (this._units !== undefined) {
            return this._units;
          }

          var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
          var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);

          if (numIsUnity && denIsUnity) {
            this._units = "";
            return this._units;
          }

          var numUnits = stringifyUnits(this.numerator);
          var denUnits = stringifyUnits(this.denominator);
          this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
          return this._units;
        },

        /**
         * Stringifies the quantity
         * Deprecation notice: only units parameter is supported.
         *
         * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
         *                              target units if string,
         *                              max number of decimals if number,
         *                              passed to #toPrec before converting if Qty
         *
         * @param {number=} maxDecimals - Maximum number of decimals of
         *                                formatted output
         *
         * @returns {string} reparseable quantity as string
         */
        toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
          var targetUnits;

          if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
            targetUnits = this.units();
            maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
          } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
            targetUnits = targetUnitsOrMaxDecimalsOrPrec;
          } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
            return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
          }

          var out = this.to(targetUnits);
          var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
          out = (outScalar + " " + out.units()).trim();
          return out;
        },

        /**
         * Format the quantity according to optional passed target units
         * and formatter
         *
         * @param {string} [targetUnits=current units] -
         *                 optional units to convert to before formatting
         *
         * @param {function} [formatter=Qty.formatter] -
         *                   delegates formatting to formatter callback.
         *                   formatter is called back with two parameters (scalar, units)
         *                   and should return formatted result.
         *                   If unspecified, formatting is delegated to default formatter
         *                   set to Qty.formatter
         *
         * @example
         * var roundingAndLocalizingFormatter = function(scalar, units) {
         *   // localize or limit scalar to n max decimals for instance
         *   // return formatted result
         * };
         * var qty = Qty('1.1234 m');
         * qty.format(); // same units, default formatter => "1.234 m"
         * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
         * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
         * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
         *
         * @returns {string} quantity as string
         */
        format: function (targetUnits, formatter) {
          if (arguments.length === 1) {
            if (typeof targetUnits === "function") {
              formatter = targetUnits;
              targetUnits = undefined;
            }
          }

          formatter = formatter || Qty.formatter;
          var targetQty = this.to(targetUnits);
          return formatter.call(this, targetQty.scalar, targetQty.units());
        }
      });
      var stringifiedUnitsCache = new NestedMap();
      /**
       * Returns a string representing a normalized unit array
       *
       * @param {string[]} units Normalized unit array
       * @returns {string} String representing passed normalized unit array and
       *   suitable for output
       *
       */

      function stringifyUnits(units) {
        var stringified = stringifiedUnitsCache.get(units);

        if (stringified) {
          return stringified;
        }

        var isUnity = compareArray(units, UNITY_ARRAY);

        if (isUnity) {
          stringified = "1";
        } else {
          stringified = simplify(getOutputNames(units)).join("*");
        } // Cache result


        stringifiedUnitsCache.set(units, stringified);
        return stringified;
      }

      function getOutputNames(units) {
        var unitNames = [],
            token,
            tokenNext;

        for (var i = 0; i < units.length; i++) {
          token = units[i];
          tokenNext = units[i + 1];

          if (PREFIX_VALUES[token]) {
            unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
            i++;
          } else {
            unitNames.push(OUTPUT_MAP[token]);
          }
        }

        return unitNames;
      }

      function simplify(units) {
        // this turns ['s','m','s'] into ['s2','m']
        var unitCounts = units.reduce(function (acc, unit) {
          var unitCounter = acc[unit];

          if (!unitCounter) {
            acc.push(unitCounter = acc[unit] = [unit, 0]);
          }

          unitCounter[1]++;
          return acc;
        }, []);
        return unitCounts.map(function (unitCount) {
          return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
        });
      }

      Qty.version = "1.7.6";
      return Qty;
    });
  });

  function convertUnit(array, fromUnit, toUnit) {
    fromUnit = normalize(fromUnit);
    toUnit = normalize(toUnit);
    if (fromUnit === toUnit) return array;

    try {
      const convert = quantities.swiftConverter(fromUnit, toUnit); // Configures converter

      return convert(array);
    } catch (e) {
      return undefined;
    }
  }

  function normalize(unit) {
    unit = unit.replace(/°C/g, 'tempC');
    unit = unit.replace(/°F/g, 'tempF');
    unit = unit.replace(/(^|\W)K(\W|$)/g, '$1tempK$2');
    return unit;
  }

  const testRegExp = /^\/((?:\\\/|[^/])+)\/([migyu]{0,5})?$/;
  function ensureRegexp(string) {
    const parts = testRegExp.exec(string);

    if (parts) {
      try {
        return new RegExp(parts[1], parts[2]);
      } catch (err) {
        return stringToRegexp(string);
      }
    } else {
      return stringToRegexp(string);
    }
  }

  function stringToRegexp(string, flags = 'i') {
    return new RegExp(string.replace(/[[\]\\{}()+*?.$^|]/g, function (match) {
      return `\\${match}`;
    }), flags);
  }

  function getConvertedVariable(variable, newUnits) {
    const data = variable.units !== undefined && variable.units !== newUnits // would be nice if convertUnit would allow typedArray
    ? convertUnit(Array.from(variable.data), variable.units, newUnits) : variable.data;
    return {
      units: newUnits,
      label: variable.label.replace(`[${variable.units || ''}]`, `[${newUnits}]`),
      data: data || [],
      min: data ? min(data) : undefined,
      max: data ? max(data) : undefined,
      isMonotone: xIsMonotone(data)
    };
  }

  /**
   * Retrieve the spectrum with only X/Y data that match all the selectors
   * If more than one variable match the selector the 'x' or 'y' variable will be
   * taken
   */

  function getXYSpectrum(spectra = [], selector = {}) {
    if (spectra.length < 1) return;

    for (let spectrum of spectra) {
      let variableNames = Object.keys(spectrum.variables);
      if (!(variableNames.length > 1)) continue;
      let {
        dataType,
        title,
        xUnits,
        yUnits,
        variables,
        xVariable = 'x',
        yVariable = 'y',
        units,
        labels,
        xLabel,
        yLabel,
        meta
      } = selector; // we filter on general spectrum information

      if (dataType) {
        dataType = ensureRegexp(dataType);
        if (!spectrum.dataType || !spectrum.dataType.match(dataType)) continue;
      }

      if (title) {
        title = ensureRegexp(title);
        if (!spectrum.title || !spectrum.title.match(title)) continue;
      }

      if (meta && typeof meta === 'object') {
        if (!spectrum.meta) continue;

        for (let key in spectrum.meta) {
          if (!spectrum.meta[key]) continue;
          let value = ensureRegexp(spectrum.meta[key]);
          if (!spectrum.meta[key].match(value)) continue;
        }
      }

      if (units && !xUnits && !yUnits) [yUnits, xUnits] = units.split(/\s*vs\s*/);

      if (labels && !xLabel && !yLabel) {
        [yLabel, xLabel] = labels.split(/\s*vs\s*/);
      }

      if (variables) [yVariable, xVariable] = variables.split(/\s*vs\s*/);
      if (xLabel) xLabel = ensureRegexp(xLabel);
      if (yLabel) yLabel = ensureRegexp(yLabel);
      let x = getPossibleVariable(spectrum.variables, {
        units: xUnits,
        label: xLabel,
        variableName: xVariable
      });
      let y = getPossibleVariable(spectrum.variables, {
        units: yUnits,
        label: yLabel,
        variableName: yVariable
      });

      if (x && y) {
        return {
          title: spectrum.title,
          dataType: spectrum.dataType,
          meta: spectrum.meta,
          variables: {
            x,
            y
          }
        };
      }
    }

    return;
  }

  function getPossibleVariable(variables, selector = {}) {
    const {
      units,
      label,
      variableName
    } = selector;
    let possible = { ...variables
    };

    if (units !== undefined) {
      for (let key in possible) {
        let converted = convertUnit(1, variables[key].units || '', units);

        if (converted) {
          possible[key] = getConvertedVariable(variables[key], units);
        } else {
          possible[key] = undefined;
        }
      }
    }

    if (label !== undefined) {
      for (let key in possible) {
        if (!variables[key].label.match(label)) {
          possible[key] = undefined;
        }
      }
    }

    if (variableName !== undefined) {
      if (possible[variableName]) return possible[variableName];

      if (possible[variableName.toUpperCase()]) {
        return possible[variableName.toUpperCase()];
      }

      if (possible[variableName.toLowerCase()]) {
        return possible[variableName.toLowerCase()];
      }
    }

    const possibleFiltered = Object.values(possible).filter(val => val !== undefined);

    if (possibleFiltered.length > 0) {
      return possibleFiltered[0];
    }
  }

  /**
   * Class allowing to store and manipulate an analysis.
   * An analysis may contain one or more spectra that can be selected
   * based on their units
   */

  class Analysis {
    constructor(options = {}) {
      this.id = options.id || Math.random().toString(36).substring(2, 10);
      this.label = options.label || this.id;
      this.spectrumCallback = options.spectrumCallback;
      this.spectra = [];
      this.cache = {};
    }
    /**
     * Add a spectrum in the internal spectra variable
     */


    pushSpectrum(variables, options = {}) {
      this.spectra.push(standardizeData(variables, options, {
        spectrumCallback: this.spectrumCallback
      }));
      this.cache = {};
    }
    /**
     * Retrieve a Spectrum based on x/y units
     * @param selector.units Units separated by vs like for example "g vs °C"
     * @param selector.xUnits if undefined takes the first variable
     * @param selector.yUnits if undefined takes the second variable
     */


    getXYSpectrum(selector = {}) {
      let id = JSON.stringify(selector);

      if (!this.cache[id]) {
        this.cache[id] = getXYSpectrum(this.spectra, selector);
      }

      return this.cache[id];
    }
    /**
     * Retrieve a xy object
     * @param selector.units Units separated by vs like for example "g vs °C"
     * @param selector.xUnits if undefined takes the first variable
     * @param selector.yUnits if undefined takes the second variable
     */


    getXY(selector = {}) {
      let spectrum = this.getXYSpectrum(selector);
      if (!spectrum) return undefined;
      return {
        x: spectrum.variables.x.data,
        y: spectrum.variables.y.data
      };
    }
    /**
     * Return the data object for specific x/y units with possibly some
     * normalization options
     * @param options.selector.xUnits // if undefined takes the first variable
     * @param options.selector.yUnits // if undefined takes the second variable
     */


    getNormalizedSpectrum(options = {}) {
      const {
        normalization,
        selector
      } = options;
      const spectrum = this.getXYSpectrum(selector);
      if (!spectrum) return undefined;
      return getNormalizedSpectrum(spectrum, normalization);
    }
    /**
     * Returns the first spectrum. This method could be improved in the future
     * @returns
     */


    getSpectrum() {
      return this.spectra[0];
    }
    /**
     * Returns the xLabel
     * @param selector.xUnits // if undefined takes the first variable
     * @param selector.yUnits // if undefined takes the second variable
     */


    getXLabel(selector) {
      var _a;

      return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.x.label;
    }
    /**
     * Returns the yLabel
     * @param selector.xUnits // if undefined takes the first variable
     * @param selector.yUnits // if undefined takes the second variable
     */


    getYLabel(selector) {
      var _a;

      return (_a = this.getXYSpectrum(selector)) === null || _a === void 0 ? void 0 : _a.variables.y.label;
    }

  }
  /**
   * Internal function that ensure the order of x / y array
   */

  function standardizeData(variables, options, analysisOptions) {
    let {
      meta = {},
      tmp = {},
      dataType = '',
      title = ''
    } = options;
    const {
      spectrumCallback
    } = analysisOptions;

    if (spectrumCallback) {
      spectrumCallback(variables);
    }

    let xVariable = variables.x;
    let yVariable = variables.y;

    if (!xVariable || !yVariable) {
      throw Error('A spectrum must contain at least x and y variables');
    }

    if (!isAnyArray(xVariable.data) || !isAnyArray(yVariable.data)) {
      throw Error('x and y variables must contain an array data');
    }

    let x = xVariable.data;
    let reverse = x && x.length > 1 && x[0] > x[x.length - 1];

    for (let key in variables) {
      let variable = variables[key];
      if (reverse) variable.data = variable.data.reverse();
      variable.label = variable.label || key;
      variable.units = variable.units || variable.label.replace(/^.*[([](?<units>.*)[)\]].*$/, '$<units>');
      variable.min = min(variable.data);
      variable.max = max(variable.data);
      variable.isMonotone = xIsMonotone(variable.data);
    }

    return {
      variables,
      title,
      dataType,
      meta,
      tmp
    };
  }

  /**
   * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
   * @param {string|ArrayBuffer} blob
   * @param {object} [options={}]
   * @param {string} [options.encoding='utf8']
   * @returns {string}
   */
  function ensureString(blob, options = {}) {
    const {
      encoding = 'utf8'
    } = options;

    if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
      const decoder = new TextDecoder(encoding);
      return decoder.decode(blob);
    }

    return blob;
  }

  const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
  function complexChromatogram(result) {
    let spectra = result.spectra;
    let length = spectra.length;
    let chromatogram = {
      times: new Array(length),
      series: {
        ms: {
          dimension: 2,
          data: new Array(length)
        }
      }
    };
    let existingGCMSFields = [];

    for (let i = 0; i < GC_MS_FIELDS.length; i++) {
      let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

      if (spectra[0][label]) {
        existingGCMSFields.push(label);
        chromatogram.series[label] = {
          dimension: 1,
          data: new Array(length)
        };
      }
    }

    for (let i = 0; i < length; i++) {
      let spectrum = spectra[i];
      chromatogram.times[i] = spectrum.pageValue;

      for (let j = 0; j < existingGCMSFields.length; j++) {
        chromatogram.series[existingGCMSFields[j]].data[i] = parseFloat(spectrum[existingGCMSFields[j]]);
      }

      if (spectrum.data) {
        chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
      }
    }

    result.chromatogram = chromatogram;
  }
  function isMSField(canonicDataLabel) {
    return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
  }
  function convertMSFieldToLabel(value) {
    return value.toLowerCase().replace(/[^a-z0-9]/g, '');
  }

  function convertToFloatArray(stringArray) {
    let floatArray = [];

    for (let i = 0; i < stringArray.length; i++) {
      floatArray.push(parseFloat(stringArray[i]));
    }

    return floatArray;
  }

  function fastParseXYData(spectrum, value) {
    // TODO need to deal with result
    //  console.log(value);
    // we check if deltaX is defined otherwise we calculate it
    let yFactor = spectrum.yFactor;
    let deltaX = spectrum.deltaX;
    spectrum.isXYdata = true;
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData;
    let currentX = spectrum.firstX;
    let currentY = spectrum.firstY; // we skip the first line
    //

    let endLine = false;
    let ascii;
    let i = 0;

    for (; i < value.length; i++) {
      ascii = value.charCodeAt(i);

      if (ascii === 13 || ascii === 10) {
        endLine = true;
      } else {
        if (endLine) break;
      }
    } // we proceed taking the i after the first line


    let newLine = true;
    let isDifference = false;
    let isLastDifference = false;
    let lastDifference = 0;
    let isDuplicate = false;
    let inComment = false;
    let currentValue = 0; // can be a difference or a duplicate

    let lastValue = 0; // must be the real last value

    let isNegative = false;
    let inValue = false;
    let skipFirstValue = false;
    let decimalPosition = 0;

    for (; i <= value.length; i++) {
      if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

      if (inComment) {
        // we should ignore the text if we are after $$
        if (ascii === 13 || ascii === 10) {
          newLine = true;
          inComment = false;
        }
      } else {
        // when is it a new value ?
        // when it is not a digit, . or comma
        // it is a number that is either new or we continue
        if (ascii <= 57 && ascii >= 48) {
          // a number
          inValue = true;

          if (decimalPosition > 0) {
            currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
          } else {
            currentValue *= 10;
            currentValue += ascii - 48;
          }
        } else if (ascii === 44 || ascii === 46) {
          // a "," or "."
          inValue = true;
          decimalPosition++;
        } else {
          if (inValue) {
            // need to process the previous value
            if (newLine) {
              newLine = false; // we don't check the X value
              // console.log("NEW LINE",isDifference, lastDifference);
              // if new line and lastDifference, the first value is just a check !
              // that we don't check ...

              if (isLastDifference) skipFirstValue = true;
            } else {
              // need to deal with duplicate and differences
              if (skipFirstValue) {
                skipFirstValue = false;
              } else {
                if (isDifference) {
                  lastDifference = isNegative ? 0 - currentValue : currentValue;
                  isLastDifference = true;
                  isDifference = false;
                } else if (!isDuplicate) {
                  lastValue = isNegative ? 0 - currentValue : currentValue;
                }

                let duplicate = isDuplicate ? currentValue - 1 : 1;

                for (let j = 0; j < duplicate; j++) {
                  if (isLastDifference) {
                    currentY += lastDifference;
                  } else {
                    currentY = lastValue;
                  }

                  currentData.x.push(currentX);
                  currentData.y.push(currentY * yFactor);
                  currentX += deltaX;
                }
              }
            }

            isNegative = false;
            currentValue = 0;
            decimalPosition = 0;
            inValue = false;
            isDuplicate = false;
          } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


          if (ascii < 74 && ascii > 63) {
            inValue = true;
            isLastDifference = false;
            currentValue = ascii - 64;
          } else if (ascii > 96 && ascii < 106) {
            // negative SQZ digits a b c d e f g h i (ascii 97-105)
            inValue = true;
            isLastDifference = false;
            currentValue = ascii - 96;
            isNegative = true;
          } else if (ascii === 115) {
            // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
            inValue = true;
            isDuplicate = true;
            currentValue = 9;
          } else if (ascii > 82 && ascii < 91) {
            inValue = true;
            isDuplicate = true;
            currentValue = ascii - 82;
          } else if (ascii > 73 && ascii < 83) {
            // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
            inValue = true;
            isDifference = true;
            currentValue = ascii - 73;
          } else if (ascii > 105 && ascii < 115) {
            // negative DIF digits j k l m n o p q r (ascii 106-114)
            inValue = true;
            isDifference = true;
            currentValue = ascii - 105;
            isNegative = true;
          } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
            // $ sign, we need to check the next one
            inValue = true;
            inComment = true;
          } else if (ascii === 37) {
            // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
            inValue = true;
            isDifference = true;
            currentValue = 0;
            isNegative = false;
          } else if (ascii === 45) {
            // a "-"
            // check if after there is a number, decimal or comma
            let ascii2 = value.charCodeAt(i + 1);

            if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
              inValue = true;
              if (!newLine) isLastDifference = false;
              isNegative = true;
            }
          } else if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          } // and now analyse the details ... space or tabulation
          // if "+" we just don't care

        }
      }
    }
  }

  const removeCommentRegExp = /\$\$.*/;
  const peakTableSplitRegExp = /[,\t ]+/;
  function parsePeakTable(spectrum, value, result) {
    spectrum.isPeaktable = true;

    if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
      parseXY(spectrum, value, result);
    } else {
      parseXYZ(spectrum, value, result);
    } // we will add the data in the variables


    if (spectrum.variables) {
      for (let key in spectrum.variables) {
        spectrum.variables[key].data = spectrum.data[key];
      }
    }
  }

  function parseXY(spectrum, value, result) {
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData; // counts for around 20% of the time

    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

      if (values.length % 2 === 0) {
        for (let j = 0; j < values.length; j = j + 2) {
          // takes around 40% of the time to add and parse the 2 values nearly exclusively because of parseFloat
          currentData.x.push(parseFloat(values[j]) * spectrum.xFactor);
          currentData.y.push(parseFloat(values[j + 1]) * spectrum.yFactor);
        }
      } else {
        result.logs.push(`Format error: ${values}`);
      }
    }
  }

  function parseXYZ(spectrum, value, result) {
    let currentData = {};
    let variables = Object.keys(spectrum.variables);
    let numberOfVariables = variables.length;
    variables.forEach(variable => currentData[variable] = []);
    spectrum.data = currentData; // counts for around 20% of the time

    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

      if (values.length % numberOfVariables === 0) {
        for (let j = 0; j < values.length; j++) {
          // todo should try to find a xFactor (y, ...)
          currentData[variables[j % numberOfVariables]].push(parseFloat(values[j]));
        }
      } else {
        result.logs.push(`Format error: ${values}`);
      }
    }
  }

  function parseXYA(spectrum, value) {
    let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
    spectrum.isXYAdata = true;
    let values;
    let currentData = {
      x: [],
      y: []
    };
    spectrum.data = currentData;
    let lines = value.split(/,? *,?[;\r\n]+ */);

    for (let i = 1; i < lines.length; i++) {
      values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
      currentData.x.push(parseFloat(values[0]));
      currentData.y.push(parseFloat(values[1]));
    }
  }

  function convertTo3DZ(spectra) {
    let minZ = spectra[0].data.y[0];
    let maxZ = minZ;
    let ySize = spectra.length;
    let xSize = spectra[0].data.x.length;
    let z = new Array(ySize);

    for (let i = 0; i < ySize; i++) {
      z[i] = spectra[i].data.y;

      for (let j = 0; j < xSize; j++) {
        let value = z[i][j];
        if (value < minZ) minZ = value;
        if (value > maxZ) maxZ = value;
      }
    }

    const firstX = spectra[0].data.x[0];
    const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

    const firstY = spectra[0].pageValue;
    const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
    // min and max we need to invert the array

    if (firstX > lastX) {
      for (let spectrum of z) {
        spectrum.reverse();
      }
    }

    if (firstY > lastY) {
      z.reverse();
    }

    const medians = [];

    for (let i = 0; i < z.length; i++) {
      const row = Float64Array.from(z[i]);

      for (let i = 0; i < row.length; i++) {
        if (row[i] < 0) row[i] = -row[i];
      }

      medians.push(median(row));
    }

    const median$1 = median(medians);
    return {
      z: z,
      minX: Math.min(firstX, lastX),
      maxX: Math.max(firstX, lastX),
      minY: Math.min(firstY, lastY),
      maxY: Math.max(firstY, lastY),
      minZ: minZ,
      maxZ: maxZ,
      noise: median$1
    };
  }

  function generateContourLines(zData, options) {
    let noise = zData.noise;
    let z = zData.z;
    let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
    let isOver0, isOver1, isOver2, isOver3;
    let nbSubSpectra = z.length;
    let nbPovars = z[0].length;
    let pAx, pAy, pBx, pBy;
    let x0 = zData.minX;
    let xN = zData.maxX;
    let dx = (xN - x0) / (nbPovars - 1);
    let y0 = zData.minY;
    let yN = zData.maxY;
    let dy = (yN - y0) / (nbSubSpectra - 1);
    let minZ = zData.minZ;
    let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
    // -------------------------
    // Povars attribution
    //
    // 0----1
    // |  / |
    // | /  |
    // 2----3
    //
    // ---------------------d------

    let iter = options.nbContourLevels * 2;
    let contourLevels = new Array(iter);
    let lineZValue;

    for (let level = 0; level < iter; level++) {
      // multiply by 2 for positif and negatif
      let contourLevel = {};
      contourLevels[level] = contourLevel;
      let side = level % 2;
      let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

      if (side === 0) {
        lineZValue = factor + options.noiseMultiplier * noise;
      } else {
        lineZValue = 0 - factor - options.noiseMultiplier * noise;
      }

      let lines = [];
      contourLevel.zValue = lineZValue;
      contourLevel.lines = lines;
      if (lineZValue <= minZ || lineZValue >= maxZ) continue;

      for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
        let subSpectra = z[iSubSpectra];
        let subSpectraAfter = z[iSubSpectra + 1];

        for (let povar = 0; povar < nbPovars - 1; povar++) {
          povarHeight0 = subSpectra[povar];
          povarHeight1 = subSpectra[povar + 1];
          povarHeight2 = subSpectraAfter[povar];
          povarHeight3 = subSpectraAfter[povar + 1];
          isOver0 = povarHeight0 > lineZValue;
          isOver1 = povarHeight1 > lineZValue;
          isOver2 = povarHeight2 > lineZValue;
          isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
          // povar2 are below, we find the varersections and add
          // the segment

          if (isOver0 !== isOver1 && isOver0 !== isOver2) {
            pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
            pAy = iSubSpectra;
            pBx = povar;
            pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
            lines.push(pAx * dx + x0);
            lines.push(pAy * dy + y0);
            lines.push(pBx * dx + x0);
            lines.push(pBy * dy + y0);
          } // remove push does not help !!!!


          if (isOver3 !== isOver1 && isOver3 !== isOver2) {
            pAx = povar + 1;
            pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
            pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
            pBy = iSubSpectra + 1;
            lines.push(pAx * dx + x0);
            lines.push(pAy * dy + y0);
            lines.push(pBx * dx + x0);
            lines.push(pBy * dy + y0);
          } // test around the diagonal


          if (isOver1 !== isOver2) {
            pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
            pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

            if (isOver1 !== isOver0) {
              pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
              pBy = iSubSpectra;
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver2 !== isOver0) {
              pBx = povar;
              pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver1 !== isOver3) {
              pBx = povar + 1;
              pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }

            if (isOver2 !== isOver3) {
              pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
              pBy = iSubSpectra + 1;
              lines.push(pAx);
              lines.push(pAy);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            }
          }
        }
      }
    }

    return {
      minX: zData.minX,
      maxX: zData.maxX,
      minY: zData.minY,
      maxY: zData.maxY,
      segments: contourLevels
    };
  }

  function add2D(result, options) {
    let zData = convertTo3DZ(result.spectra);

    if (!options.noContour) {
      result.contourLines = generateContourLines(zData, options);
      delete zData.z;
    }

    result.minMax = zData;
  }

  // sources:
  // https://en.wikipedia.org/wiki/Gyromagnetic_ratio
  // TODO: #13 can we have a better source and more digits ? @jwist
  const gyromagneticRatio = {
    '1H': 267.52218744e6,
    '2H': 41.065e6,
    '3H': 285.3508e6,
    '3He': -203.789e6,
    '7Li': 103.962e6,
    '13C': 67.28284e6,
    '14N': 19.331e6,
    '15N': -27.116e6,
    '17O': -36.264e6,
    '19F': 251.662e6,
    '23Na': 70.761e6,
    '27Al': 69.763e6,
    '29Si': -53.19e6,
    '31P': 108.291e6,
    '57Fe': 8.681e6,
    '63Cu': 71.118e6,
    '67Zn': 16.767e6,
    '129Xe': -73.997e6
  };

  function postProcessingNMR(entriesFlat) {
    // specific NMR functions
    for (let entry of entriesFlat) {
      let observeFrequency = 0;
      let shiftOffsetVal = 0;

      for (let spectrum of entry.spectra) {
        if (entry.ntuples && entry.ntuples.symbol) {
          if (!observeFrequency && spectrum.observeFrequency) {
            observeFrequency = spectrum.observeFrequency;
          }

          if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }
        } else {
          observeFrequency = spectrum.observeFrequency;
          shiftOffsetVal = spectrum.shiftOffsetVal;
        }

        if (observeFrequency) {
          if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
            spectrum.xUnits = 'PPM';
            spectrum.xFactor = spectrum.xFactor / observeFrequency;
            spectrum.firstX = spectrum.firstX / observeFrequency;
            spectrum.lastX = spectrum.lastX / observeFrequency;
            spectrum.deltaX = spectrum.deltaX / observeFrequency;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] /= observeFrequency;
            }
          }
        }

        if (shiftOffsetVal) {
          let shift = spectrum.firstX - shiftOffsetVal;
          spectrum.firstX = spectrum.firstX - shift;
          spectrum.lastX = spectrum.lastX - shift;

          for (let i = 0; i < spectrum.data.x.length; i++) {
            spectrum.data.x[i] -= shift;
          }
        } // we will check if some nucleus are missing ...


        if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
          for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
            let symbol = entry.ntuples.symbol[i];
            let nucleus = entry.ntuples.nucleus[i];

            if (symbol.startsWith('F') && !nucleus) {
              if (symbol === 'F1') {
                // if F1 is defined we will use F2
                if (entry.tmp.$NUC2) {
                  entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                } else {
                  let f2index = entry.ntuples.symbol.indexOf('F2');

                  if (f2index && entry.ntuples.nucleus[f2index]) {
                    entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                  }
                }
              }

              if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
            }

            if (symbol === 'F2') {
              entry.yType = entry.ntuples.nucleus[0];
            }
          }
        }

        if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
          let unit = '';
          let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

          if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
            unit = entry.ntuples.units[pageSymbolIndex];
          }

          if (unit !== 'PPM') {
            if (pageSymbolIndex !== 0) {
              throw Error('Not sure about this ntuples format');
            }

            let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
            let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

            if (!ratio0 || !ratio1) {
              throw Error('Problem with determination of gyromagnetic ratio');
            }

            let ratio = ratio0 / ratio1 * observeFrequency;
            spectrum.pageValue /= ratio;
          }
        }
      }
    }
  }

  function profiling(result, action, options) {
    if (result.profiling) {
      result.profiling.push({
        action,
        time: Date.now() - options.start
      });
    }
  }

  function simpleChromatogram(result) {
    let data = result.spectra[0].data;
    result.chromatogram = {
      times: data.x.slice(),
      series: {
        intensity: {
          dimension: 1,
          data: data.y.slice()
        }
      }
    };
  }

  function postProcessing(entriesFlat, result, options) {
    // converting Hz to ppm
    postProcessingNMR(entriesFlat);

    for (let entry of entriesFlat) {
      if (Object.keys(entry.ntuples).length > 0) {
        let newNtuples = [];
        let keys = Object.keys(entry.ntuples);

        for (let i = 0; i < keys.length; i++) {
          let key = keys[i];
          let values = entry.ntuples[key];

          for (let j = 0; j < values.length; j++) {
            if (!newNtuples[j]) newNtuples[j] = {};
            newNtuples[j][key] = values[j];
          }
        }

        entry.ntuples = newNtuples;
      }

      if (entry.twoD && options.wantXY) {
        add2D(entry, options);
        profiling(result, 'Finished countour plot calculation', options);

        if (!options.keepSpectra) {
          delete entry.spectra;
        }
      } // maybe it is a GC (HPLC) / MS. In this case we add a new format


      if (options.chromatogram) {
        if (entry.spectra.length > 1) {
          complexChromatogram(entry);
        } else {
          simpleChromatogram(entry);
        }

        profiling(result, 'Finished chromatogram calculation', options);
      }

      delete entry.tmp;
    }
  }

  function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
    let xIndex = -1;
    let yIndex = -1;
    let firstVariable = '';
    let secondVariable = '';

    if (kind.indexOf('++') > 0) {
      firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
      secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
    } else {
      kind = kind.replace(/[^a-zA-Z]/g, '');
      firstVariable = kind.charAt(0);
      secondVariable = kind.charAt(1);
      spectrum.variables = {};

      for (let symbol of kind) {
        let lowerCaseSymbol = symbol.toLowerCase();
        let index = currentEntry.ntuples.symbol.indexOf(symbol);
        if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
        spectrum.variables[lowerCaseSymbol] = {};

        for (let key in currentEntry.ntuples) {
          if (currentEntry.ntuples[key][index]) {
            spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
          }
        }
      }
    }

    xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
    yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
    if (xIndex === -1) xIndex = 0;
    if (yIndex === -1) yIndex = 0;

    if (currentEntry.ntuples.first) {
      if (currentEntry.ntuples.first.length > xIndex) {
        spectrum.firstX = currentEntry.ntuples.first[xIndex];
      }

      if (currentEntry.ntuples.first.length > yIndex) {
        spectrum.firstY = currentEntry.ntuples.first[yIndex];
      }
    }

    if (currentEntry.ntuples.last) {
      if (currentEntry.ntuples.last.length > xIndex) {
        spectrum.lastX = currentEntry.ntuples.last[xIndex];
      }

      if (currentEntry.ntuples.last.length > yIndex) {
        spectrum.lastY = currentEntry.ntuples.last[yIndex];
      }
    }

    if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
      spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
    }

    if (currentEntry.ntuples.factor) {
      if (currentEntry.ntuples.factor.length > xIndex) {
        spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
      }

      if (currentEntry.ntuples.factor.length > yIndex) {
        spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
      }
    }

    if (currentEntry.ntuples.units) {
      if (currentEntry.ntuples.units.length > xIndex) {
        if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
          spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
        } else {
          spectrum.xUnits = currentEntry.ntuples.units[xIndex];
        }
      }

      if (currentEntry.ntuples.units.length > yIndex) {
        if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
          spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
        } else {
          spectrum.yUnits = currentEntry.ntuples.units[yIndex];
        }
      }
    }
  }

  function prepareSpectrum(spectrum) {
    if (!spectrum.xFactor) spectrum.xFactor = 1;
    if (!spectrum.yFactor) spectrum.yFactor = 1;
  }

  const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
  const numberRegExp = /^[-+]?[0-9]*\.?[0-9]+(e[-+]?[0-9]+)?$/;

  class Spectrum {}

  const defaultOptions = {
    keepRecordsRegExp: /^$/,
    canonicDataLabels: true,
    canonicMetadataLabels: false,
    dynamicTyping: true,
    withoutXY: false,
    chromatogram: false,
    keepSpectra: false,
    noContour: false,
    nbContourLevels: 7,
    noiseMultiplier: 5,
    profiling: false
  };
  /**
   *
   * @param {string|ArrayBuffer} jcamp
   * @param {object} [options]
   * @param {number} [options.keepRecordsRegExp=/^$/] By default we don't keep meta information
   * @param {number} [options.canonicDataLabels=true] Canonize the Labels (uppercase without symbol)
   * @param {number} [options.canonicMetadataLabels=false] Canonize the metadata Labels (uppercase without symbol)
   * @param {number} [options.dynamicTyping=false] Convert numbers to Number
   * @param {number} [options.withoutXY=false] Remove the XY data
   * @param {number} [options.chromatogram=false] Special post-processing for GC / HPLC / MS
   * @param {number} [options.keepSpectra=false] Force to keep the spectra in case of 2D
   * @param {number} [options.noContour=false] Don't calculate countour in case of 2D
   * @param {number} [options.nbContourLevels=7] Number of positive / negative contour levels to calculate
   * @param {number} [options.noiseMultiplier=5] Define for 2D the level as 5 times the median as default
   * @param {number} [options.profiling=false] Add profiling information
   */

  function convert(jcamp, options = {}) {
    jcamp = ensureString(jcamp);
    options = { ...defaultOptions,
      ...options
    };
    options.wantXY = !options.withoutXY;
    options.start = Date.now();
    let entriesFlat = [];
    let result = {
      profiling: options.profiling ? [] : false,
      logs: [],
      entries: []
    };
    let tmpResult = {
      children: []
    };
    let currentEntry = tmpResult;
    let parentsStack = [];
    let spectrum = new Spectrum();

    if (typeof jcamp !== 'string') {
      throw new TypeError('the JCAMP should be a string');
    }

    profiling(result, 'Before split to LDRS', options);
    let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
    profiling(result, 'Split to LDRS', options);
    if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

    for (let ldr of ldrs) {
      // This is a new LDR
      let position = ldr.indexOf('=');
      let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
      let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
      let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

      if (canonicDataLabel === 'DATATABLE') {
        let endLine = dataValue.indexOf('\n');
        if (endLine === -1) endLine = dataValue.indexOf('\r');

        if (endLine > 0) {
          // ##DATA TABLE= (X++(I..I)), XYDATA
          // We need to find the variables
          let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
          prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
          spectrum.datatable = infos[0];

          if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
            canonicDataLabel = 'PEAKTABLE';
          } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
            canonicDataLabel = 'XYDATA';
            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
          }
        }
      }

      if (canonicDataLabel === 'XYDATA') {
        if (options.wantXY) {
          prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

          if (dataValue.match(/.*\+\+.*/)) {
            // ex: (X++(Y..Y))
            spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
            fastParseXYData(spectrum, dataValue);
          } else {
            parsePeakTable(spectrum, dataValue, result);
          }

          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      } else if (canonicDataLabel === 'PEAKTABLE') {
        if (options.wantXY) {
          prepareSpectrum(spectrum);
          parsePeakTable(spectrum, dataValue, result);
          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      }

      if (canonicDataLabel === 'PEAKASSIGNMENTS') {
        if (options.wantXY) {
          if (dataValue.match(/.*(XYA).*/)) {
            // ex: (XYA)
            parseXYA(spectrum, dataValue);
          }

          currentEntry.spectra.push(spectrum);
          spectrum = new Spectrum();
        }

        continue;
      }

      if (canonicDataLabel === 'TITLE') {
        let parentEntry = currentEntry;

        if (!parentEntry.children) {
          parentEntry.children = [];
        }

        currentEntry = {
          spectra: [],
          ntuples: {},
          info: {},
          meta: {},
          tmp: {} // tmp information we need to keep for postprocessing

        };
        parentEntry.children.push(currentEntry);
        parentsStack.push(parentEntry);
        entriesFlat.push(currentEntry);
        currentEntry.title = dataValue;
      } else if (canonicDataLabel === 'DATATYPE') {
        currentEntry.dataType = dataValue;

        if (dataValue.toLowerCase().indexOf('nd') > -1) {
          currentEntry.twoD = true;
        }
      } else if (canonicDataLabel === 'NTUPLES') {
        if (dataValue.toLowerCase().indexOf('nd') > -1) {
          currentEntry.twoD = true;
        }
      } else if (canonicDataLabel === 'DATACLASS') {
        currentEntry.dataClass = dataValue;
      } else if (canonicDataLabel === 'XUNITS') {
        spectrum.xUnits = dataValue;
      } else if (canonicDataLabel === 'YUNITS') {
        spectrum.yUnits = dataValue;
      } else if (canonicDataLabel === 'FIRSTX') {
        spectrum.firstX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'LASTX') {
        spectrum.lastX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'FIRSTY') {
        spectrum.firstY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'LASTY') {
        spectrum.lastY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'NPOINTS') {
        spectrum.nbPoints = parseFloat(dataValue);
      } else if (canonicDataLabel === 'XFACTOR') {
        spectrum.xFactor = parseFloat(dataValue);
      } else if (canonicDataLabel === 'YFACTOR') {
        spectrum.yFactor = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MAXX') {
        spectrum.maxX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MINX') {
        spectrum.minX = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MAXY') {
        spectrum.maxY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'MINY') {
        spectrum.minY = parseFloat(dataValue);
      } else if (canonicDataLabel === 'DELTAX') {
        spectrum.deltaX = parseFloat(dataValue);
      } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
        if (!spectrum.observeFrequency) {
          spectrum.observeFrequency = parseFloat(dataValue);
        }
      } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
        if (!spectrum.xType) {
          currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
        }
      } else if (canonicDataLabel === '$OFFSET') {
        // OFFSET for Bruker spectra
        currentEntry.shiftOffsetNum = 0;

        if (!spectrum.shiftOffsetVal) {
          spectrum.shiftOffsetVal = parseFloat(dataValue);
        }
      } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
        currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'SYMBOL') {
        currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARTYPE') {
        currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARFORM') {
        currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'VARDIM') {
        currentEntry.ntuples.vardim = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'UNITS') {
        currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
      } else if (canonicDataLabel === 'FACTOR') {
        currentEntry.ntuples.factor = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'FIRST') {
        currentEntry.ntuples.first = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'LAST') {
        currentEntry.ntuples.last = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'MIN') {
        currentEntry.ntuples.min = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === 'MAX') {
        currentEntry.ntuples.max = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
      } else if (canonicDataLabel === '.NUCLEUS') {
        if (currentEntry.ntuples) {
          currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
        }
      } else if (canonicDataLabel === 'PAGE') {
        spectrum.page = dataValue.trim();
        spectrum.pageValue = parseFloat(dataValue.replace(/^.*=/, ''));
        spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
      } else if (canonicDataLabel === 'RETENTIONTIME') {
        spectrum.pageValue = parseFloat(dataValue);
      } else if (isMSField(canonicDataLabel)) {
        spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
      } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
        spectrum.sampleDescription = dataValue;
      } else if (canonicDataLabel.startsWith('$NUC')) {
        if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
          currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
        }
      } else if (canonicDataLabel === 'END') {
        currentEntry = parentsStack.pop();
      }

      if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
        let value = dataValue.trim();
        let target, label;

        if (dataLabel.startsWith('$')) {
          label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
          target = currentEntry.meta;
        } else {
          label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
          target = currentEntry.info;
        }

        if (options.dynamicTyping) {
          if (value.match(numberRegExp)) {
            value = Number.parseFloat(value);
          }
        }

        if (target[label]) {
          if (!Array.isArray(target[label])) {
            target[label] = [target[label]];
          }

          target[label].push(value);
        } else {
          target[label] = value;
        }
      }
    }

    profiling(result, 'Finished parsing', options);
    postProcessing(entriesFlat, result, options);
    profiling(result, 'Total time', options);
    /*
    if (result.children && result.children.length>0) {
      result = { ...result, ...result.children[0] };
    }
    */

    result.entries = tmpResult.children;
    result.flatten = entriesFlat;
    return result;
  }

  /**
   * Creates a new Analysis from a JCAMP string
   * @param {string} jcamp - String containing the JCAMP data
   * @param {object} [options={}]
   * @param {object} [options.id=Math.random()]
   * @param {string} [options.label=options.id] human redeable label
   * @param {string} [options.spectrumCallback] a callback to apply on variables when creating spectrum
   * @return {Analysis} - New class element with the given data
   */

  function fromJcamp(jcamp, options = {}) {
    let analysis = new Analysis(options);
    addJcamp(analysis, jcamp);
    return analysis;
  }

  function addJcamp(analysis, jcamp) {
    let converted = convert(jcamp, {
      keepRecordsRegExp: /.*/
    });

    for (let entry of converted.flatten) {
      let currentSpectrum = entry.spectra[0]; // we ensure variables

      if (!currentSpectrum.variables) {
        const variables = {};
        currentSpectrum.variables = variables;
        variables.x = {
          label: currentSpectrum.xUnits,
          symbol: 'X',
          data: currentSpectrum.data.x || currentSpectrum.data.X
        };
        variables.y = {
          label: currentSpectrum.yUnits,
          symbol: 'Y',
          data: currentSpectrum.data.y || currentSpectrum.data.Y
        };
      } else {
        for (let key in currentSpectrum.variables) {
          const variable = currentSpectrum.variables[key];
          if (variable.label) continue;
          variable.label = variable.name || variable.symbol || key;

          if (variable.units && !variable.label.includes(variable.units)) {
            variable.label += ` [${variable.units}]`;
          }
        }
      }

      analysis.pushSpectrum(currentSpectrum.variables, {
        dataType: entry.dataType,
        title: entry.title,
        meta: entry.meta
      });
    }
  }

  /**
   * Parse from a xyxy data array
   * @param {Array<Array<number>>} variables
   * @param {object} [meta] - same metadata object format that the fromText
   * @return {string} JCAMP of the input
   */

  function creatorNtuples(variables, options) {
    const {
      meta = {},
      info = {}
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = ''
    } = info;
    const symbol = [];
    const varName = [];
    const varType = [];
    const varDim = [];
    const units = [];
    const first = [];
    const last = [];
    const min$1 = [];
    const max$1 = [];
    const keys = Object.keys(variables);

    for (let i = 0; i < keys.length; i++) {
      const key = keys[i];
      let variable = variables[key];
      let name = variable.label && variable.label.replace(/ *\[.*/, '');
      let unit = variable.label && variable.label.replace(/.*\[(.*)\].*/, '$1');
      symbol.push(variable.symbol || key);
      varName.push(variable.name || name || key);
      varDim.push(variables[key].data.length);
      varType.push(variable.type ? variable.type.toUpperCase() : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
      units.push(variable.units || unit || '');
      first.push(variables[key][0]);
      last.push(variables[key][variables[key].length - 1]);
      min$1.push(min(variables[key].data));
      max$1.push(max(variables[key].data));
    }

    let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}\n`;

    for (const key in meta) {
      header += typeof meta[key] === 'object' ? `##$${key}=${JSON.stringify(meta[key])}\n` : `##$${key}=${meta[key]}\n`;
    }

    header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##PAGE= N=1\n`;
    header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;

    for (let i = 0; i < variables[keys[0]].data.length; i++) {
      let point = [];

      for (let key of keys) {
        point.push(variables[key].data[i]);
      }

      header += `${point.join('\t')}\n`;
    }

    header += '##END';
    return header;
  }

  /**
   * Create a jcamp
   * @param {object} data - object of array
   * @param {object} [options={}] - metadata object
   * @param {string} [options.info={}] - metadata of the file
   * @param {string} [options.info.title = ''] - title of the file
   * @param {string} [options.info.owner = ''] - owner of the file
   * @param {string} [options.info.origin = ''] - origin of the file
   * @param {string} [options.info.dataType = ''] - type of data
   * @param {string} [options.info.xUnits = ''] - units for the x axis for variables===undefined
   * @param {string} [options.info.yUnits = ''] - units for the y axis for variables===undefined
   * @param {object} [options.meta = {}] - comments to add to the file
   * @return {string} JCAMP of the input
   */
  function fromJSON(data, options = {}) {
    const {
      meta = {},
      info = {}
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = '',
      xUnits = '',
      yUnits = ''
    } = info;
    let firstX = Number.POSITIVE_INFINITY;
    let lastX = Number.NEGATIVE_INFINITY;
    let firstY = Number.POSITIVE_INFINITY;
    let lastY = Number.NEGATIVE_INFINITY;
    let points = [];

    for (let i = 0; i < data.x.length; i++) {
      let x = data.x[i];
      let y = data.y[i];

      if (firstX > x) {
        firstX = x;
      }

      if (lastX < x) {
        lastX = x;
      }

      if (firstY > y) {
        firstY = y;
      }

      if (lastY < y) {
        lastY = y;
      }

      points.push(`${x} ${y}`);
    }

    let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}
##FIRSTX=${firstX}
##LASTX=${lastX}
##FIRSTY=${firstY}
##LASTY=${lastY}\n`;

    for (const key in meta) {
      header += key.toLowerCase() === 'cheminfo' ? `##$${key}=${JSON.stringify(meta[key])}\n` : `##$${key}=${meta[key]}\n`;
    } // we leave the header and utf8 fonts ${header.replace(/[^\t\r\n\x20-\x7F]/g, '')


    return `${header}##NPOINTS=${points.length}
##PEAK TABLE=(XY..XY)
${points.join('\n')}
##END`;
  }

  /**
   * Create a jcamp from variables
   * @param {Array<Variable} [variables={}] - object of variables
   * @param {string} [options.info={}] - metadata of the file
   * @param {string} [options.info.title = ''] - title of the file
   * @param {string} [options.info.owner = ''] - owner of the file
   * @param {string} [options.info.origin = ''] - origin of the file
   * @param {string} [options.info.dataType = ''] - type of data
   * @param {object} [options.meta = {}] - comments to add to the file
   * @param {object} [options.forceNtuples = false] - force the ntuples format even if there is only x and y variables
   */

  function fromVariables(variables = {}, options = {}) {
    const {
      info,
      meta,
      forceNtuples
    } = options;
    let jcampOptions = {
      info,
      meta
    };
    let keys = Object.keys(variables).map(key => key.toLowerCase());

    if (keys.length === 2 && keys.includes('x') && keys.includes('y') && !forceNtuples) {
      let x = variables.x;
      let xLabel = x.label || x.name || 'x';
      jcampOptions.info.xUnits = xLabel.includes(variables.x.units) ? xLabel : `${xLabel} [${variables.x.units}]`;
      let y = variables.y;
      let yLabel = y.label || y.name || 'y';
      jcampOptions.info.yUnits = yLabel.includes(variables.y.units) ? yLabel : `${yLabel} [${variables.y.units}]`;
      return fromJSON({
        x: variables.x.data,
        y: variables.y.data
      }, jcampOptions);
    } else {
      return creatorNtuples(variables, options);
    }
  }

  function toJcamps(analysis, options = {}) {
    let jcamps = [];

    for (let spectrum of analysis.spectra) {
      jcamps.push(getJcamp(spectrum, options));
    }

    return jcamps;
  }

  function getJcamp(spectrum, options) {
    const {
      info = {},
      meta = {}
    } = options;
    let jcampOptions = {
      options: {},
      info: {
        title: spectrum.title,
        dataType: spectrum.dataType,
        ...info
      },
      meta: { ...spectrum.meta,
        ...meta
      }
    };
    return fromVariables(spectrum.variables, jcampOptions);
  }

  function toJcamp(analysis, options = {}) {
    return toJcamps(analysis, options).join('\n');
  }

  const JSGraph = {
    getJSGraph,
    getNormalizationAnnotations
  };

  function parseRes(res) {
    res = ensureString(res);
    let meta = {};
    let currentMeta;

    for (let line of res.split(/\r?\n/).filter(line => line)) {
      if (line.startsWith('[')) {
        currentMeta = {};
        meta[line.replace(/\[(.*)\]/, '$1')] = currentMeta;
      } else {
        const {
          field,
          value
        } = line.match(/(?<field>.*?)=(?<value>.*)/).groups;
        currentMeta[field] = isNaN(value) ? value : parseFloat(value);
      }
    }

    return meta;
  }

  /* cpexcel.js (C) 2013-present SheetJS -- http://sheetjs.com */
  var cpexcel = createCommonjsModule(function (module) {
    /*jshint -W100 */
    var cptable = {
      version: "1.14.0"
    };

    cptable[437] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜ¢£¥₧ƒáíóúñÑªº¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[620] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàąçêëèïîćÄĄĘęłôöĆûùŚÖÜ¢Ł¥śƒŹŻóÓńŃźż¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[737] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩαβγδεζηθικλμνξοπρσςτυφχψ░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀ωάέήϊίόύϋώΆΈΉΊΌΎΏ±≥≤ΪΫ÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[850] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø×ƒáíóúñÑªº¿®¬½¼¡«»░▒▓│┤ÁÂÀ©╣║╗╝¢¥┐└┴┬├─┼ãÃ╚╔╩╦╠═╬¤ðÐÊËÈıÍÎÏ┘┌█▄¦Ì▀ÓßÔÒõÕµþÞÚÛÙýÝ¯´­±‗¾¶§÷¸°¨·¹³²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[852] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäůćçłëŐőîŹÄĆÉĹĺôöĽľŚśÖÜŤťŁ×čáíóúĄąŽžĘę¬źČş«»░▒▓│┤ÁÂĚŞ╣║╗╝Żż┐└┴┬├─┼Ăă╚╔╩╦╠═╬¤đĐĎËďŇÍÎě┘┌█▄ŢŮ▀ÓßÔŃńňŠšŔÚŕŰýÝţ´­˝˛ˇ˘§÷¸°¨˙űŘř■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[857] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàåçêëèïîıÄÅÉæÆôöòûùİÖÜø£ØŞşáíóúñÑĞğ¿®¬½¼¡«»░▒▓│┤ÁÂÀ©╣║╗╝¢¥┐└┴┬├─┼ãÃ╚╔╩╦╠═╬¤ºªÊËÈ�ÍÎÏ┘┌█▄¦Ì▀ÓßÔÒõÕµ�×ÚÛÙìÿ¯´­±�¾¶§÷¸°¨·¹³²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[861] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàåçêëèÐðÞÄÅÉæÆôöþûÝýÖÜø£Ø₧ƒáíóúÁÍÓÚ¿⌐¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[865] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø₧ƒáíóúñÑªº¿⌐¬½¼¡«¤░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[866] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмноп░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀рстуфхцчшщъыьэюяЁёЄєЇїЎў°∙·√№¤■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[874] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€����…�����������‘’“”•–—�������� กขฃคฅฆงจฉชซฌญฎฏฐฑฒณดตถทธนบปผฝพฟภมยรฤลฦวศษสหฬอฮฯะัาำิีึืฺุู����฿เแโใไๅๆ็่้๊๋์ํ๎๏๐๑๒๓๔๕๖๗๘๙๚๛����",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[895] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ČüéďäĎŤčěĚĹÍľǪÄÁÉžŽôöÓůÚýÖÜŠĽÝŘťáíóúňŇŮÔšřŕŔ¼§«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αßΓπΣσµτΦΘΩδ∞φε∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■ ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[932] = function () {
      var d = [],
          e = {},
          D = [],
          j;
      D[0] = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~���������������������������������｡｢｣､･ｦｧｨｩｪｫｬｭｮｯｰｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜﾝﾞﾟ��������������������������������".split("");

      for (j = 0; j != D[0].length; ++j) if (D[0][j].charCodeAt(0) !== 0xFFFD) {
        e[D[0][j]] = 0 + j;
        d[0 + j] = D[0][j];
      }

      D[129] = "����������������������������������������������������������������　、。，．・：；？！゛゜´｀¨＾￣＿ヽヾゝゞ〃仝々〆〇ー―‐／＼～∥｜…‥‘’“”（）〔〕［］｛｝〈〉《》「」『』【】＋－±×�÷＝≠＜＞≦≧∞∴♂♀°′″℃￥＄￠￡％＃＆＊＠§☆★○●◎◇◆□■△▲▽▼※〒→←↑↓〓�����������∈∋⊆⊇⊂⊃∪∩��������∧∨￢⇒⇔∀∃�����������∠⊥⌒∂∇≡≒≪≫√∽∝∵∫∬�������Å‰♯♭♪†‡¶����◯���".split("");

      for (j = 0; j != D[129].length; ++j) if (D[129][j].charCodeAt(0) !== 0xFFFD) {
        e[D[129][j]] = 33024 + j;
        d[33024 + j] = D[129][j];
      }

      D[130] = "�������������������������������������������������������������������������������０１２３４５６７８９�������ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ�������ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ����ぁあぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとどなにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをん��������������".split("");

      for (j = 0; j != D[130].length; ++j) if (D[130][j].charCodeAt(0) !== 0xFFFD) {
        e[D[130][j]] = 33280 + j;
        d[33280 + j] = D[130][j];
      }

      D[131] = "����������������������������������������������������������������ァアィイゥウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトドナニヌネノハバパヒビピフブプヘベペホボポマミ�ムメモャヤュユョヨラリルレロヮワヰヱヲンヴヵヶ��������ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ��������αβγδεζηθικλμνξοπρστυφχψω�����������������������������������������".split("");

      for (j = 0; j != D[131].length; ++j) if (D[131][j].charCodeAt(0) !== 0xFFFD) {
        e[D[131][j]] = 33536 + j;
        d[33536 + j] = D[131][j];
      }

      D[132] = "����������������������������������������������������������������АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ���������������абвгдеёжзийклмн�опрстуфхцчшщъыьэюя�������������─│┌┐┘└├┬┤┴┼━┃┏┓┛┗┣┳┫┻╋┠┯┨┷┿┝┰┥┸╂�����������������������������������������������������������������".split("");

      for (j = 0; j != D[132].length; ++j) if (D[132][j].charCodeAt(0) !== 0xFFFD) {
        e[D[132][j]] = 33792 + j;
        d[33792 + j] = D[132][j];
      }

      D[135] = "����������������������������������������������������������������①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ�㍉㌔㌢㍍㌘㌧㌃㌶㍑㍗㌍㌦㌣㌫㍊㌻㎜㎝㎞㎎㎏㏄㎡��������㍻�〝〟№㏍℡㊤㊥㊦㊧㊨㈱㈲㈹㍾㍽㍼≒≡∫∮∑√⊥∠∟⊿∵∩∪���������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[135].length; ++j) if (D[135][j].charCodeAt(0) !== 0xFFFD) {
        e[D[135][j]] = 34560 + j;
        d[34560 + j] = D[135][j];
      }

      D[136] = "���������������������������������������������������������������������������������������������������������������������������������������������������������������亜唖娃阿哀愛挨姶逢葵茜穐悪握渥旭葦芦鯵梓圧斡扱宛姐虻飴絢綾鮎或粟袷安庵按暗案闇鞍杏以伊位依偉囲夷委威尉惟意慰易椅為畏異移維緯胃萎衣謂違遺医井亥域育郁磯一壱溢逸稲茨芋鰯允印咽員因姻引飲淫胤蔭���".split("");

      for (j = 0; j != D[136].length; ++j) if (D[136][j].charCodeAt(0) !== 0xFFFD) {
        e[D[136][j]] = 34816 + j;
        d[34816 + j] = D[136][j];
      }

      D[137] = "����������������������������������������������������������������院陰隠韻吋右宇烏羽迂雨卯鵜窺丑碓臼渦嘘唄欝蔚鰻姥厩浦瓜閏噂云運雲荏餌叡営嬰影映曳栄永泳洩瑛盈穎頴英衛詠鋭液疫益駅悦謁越閲榎厭円�園堰奄宴延怨掩援沿演炎焔煙燕猿縁艶苑薗遠鉛鴛塩於汚甥凹央奥往応押旺横欧殴王翁襖鴬鴎黄岡沖荻億屋憶臆桶牡乙俺卸恩温穏音下化仮何伽価佳加可嘉夏嫁家寡科暇果架歌河火珂禍禾稼箇花苛茄荷華菓蝦課嘩貨迦過霞蚊俄峨我牙画臥芽蛾賀雅餓駕介会解回塊壊廻快怪悔恢懐戒拐改���".split("");

      for (j = 0; j != D[137].length; ++j) if (D[137][j].charCodeAt(0) !== 0xFFFD) {
        e[D[137][j]] = 35072 + j;
        d[35072 + j] = D[137][j];
      }

      D[138] = "����������������������������������������������������������������魁晦械海灰界皆絵芥蟹開階貝凱劾外咳害崖慨概涯碍蓋街該鎧骸浬馨蛙垣柿蛎鈎劃嚇各廓拡撹格核殻獲確穫覚角赫較郭閣隔革学岳楽額顎掛笠樫�橿梶鰍潟割喝恰括活渇滑葛褐轄且鰹叶椛樺鞄株兜竃蒲釜鎌噛鴨栢茅萱粥刈苅瓦乾侃冠寒刊勘勧巻喚堪姦完官寛干幹患感慣憾換敢柑桓棺款歓汗漢澗潅環甘監看竿管簡緩缶翰肝艦莞観諌貫還鑑間閑関陥韓館舘丸含岸巌玩癌眼岩翫贋雁頑顔願企伎危喜器基奇嬉寄岐希幾忌揮机旗既期棋棄���".split("");

      for (j = 0; j != D[138].length; ++j) if (D[138][j].charCodeAt(0) !== 0xFFFD) {
        e[D[138][j]] = 35328 + j;
        d[35328 + j] = D[138][j];
      }

      D[139] = "����������������������������������������������������������������機帰毅気汽畿祈季稀紀徽規記貴起軌輝飢騎鬼亀偽儀妓宜戯技擬欺犠疑祇義蟻誼議掬菊鞠吉吃喫桔橘詰砧杵黍却客脚虐逆丘久仇休及吸宮弓急救�朽求汲泣灸球究窮笈級糾給旧牛去居巨拒拠挙渠虚許距鋸漁禦魚亨享京供侠僑兇競共凶協匡卿叫喬境峡強彊怯恐恭挟教橋況狂狭矯胸脅興蕎郷鏡響饗驚仰凝尭暁業局曲極玉桐粁僅勤均巾錦斤欣欽琴禁禽筋緊芹菌衿襟謹近金吟銀九倶句区狗玖矩苦躯駆駈駒具愚虞喰空偶寓遇隅串櫛釧屑屈���".split("");

      for (j = 0; j != D[139].length; ++j) if (D[139][j].charCodeAt(0) !== 0xFFFD) {
        e[D[139][j]] = 35584 + j;
        d[35584 + j] = D[139][j];
      }

      D[140] = "����������������������������������������������������������������掘窟沓靴轡窪熊隈粂栗繰桑鍬勲君薫訓群軍郡卦袈祁係傾刑兄啓圭珪型契形径恵慶慧憩掲携敬景桂渓畦稽系経継繋罫茎荊蛍計詣警軽頚鶏芸迎鯨�劇戟撃激隙桁傑欠決潔穴結血訣月件倹倦健兼券剣喧圏堅嫌建憲懸拳捲検権牽犬献研硯絹県肩見謙賢軒遣鍵険顕験鹸元原厳幻弦減源玄現絃舷言諺限乎個古呼固姑孤己庫弧戸故枯湖狐糊袴股胡菰虎誇跨鈷雇顧鼓五互伍午呉吾娯後御悟梧檎瑚碁語誤護醐乞鯉交佼侯候倖光公功効勾厚口向���".split("");

      for (j = 0; j != D[140].length; ++j) if (D[140][j].charCodeAt(0) !== 0xFFFD) {
        e[D[140][j]] = 35840 + j;
        d[35840 + j] = D[140][j];
      }

      D[141] = "����������������������������������������������������������������后喉坑垢好孔孝宏工巧巷幸広庚康弘恒慌抗拘控攻昂晃更杭校梗構江洪浩港溝甲皇硬稿糠紅紘絞綱耕考肯肱腔膏航荒行衡講貢購郊酵鉱砿鋼閤降�項香高鴻剛劫号合壕拷濠豪轟麹克刻告国穀酷鵠黒獄漉腰甑忽惚骨狛込此頃今困坤墾婚恨懇昏昆根梱混痕紺艮魂些佐叉唆嵯左差査沙瑳砂詐鎖裟坐座挫債催再最哉塞妻宰彩才採栽歳済災采犀砕砦祭斎細菜裁載際剤在材罪財冴坂阪堺榊肴咲崎埼碕鷺作削咋搾昨朔柵窄策索錯桜鮭笹匙冊刷���".split("");

      for (j = 0; j != D[141].length; ++j) if (D[141][j].charCodeAt(0) !== 0xFFFD) {
        e[D[141][j]] = 36096 + j;
        d[36096 + j] = D[141][j];
      }

      D[142] = "����������������������������������������������������������������察拶撮擦札殺薩雑皐鯖捌錆鮫皿晒三傘参山惨撒散桟燦珊産算纂蚕讃賛酸餐斬暫残仕仔伺使刺司史嗣四士始姉姿子屍市師志思指支孜斯施旨枝止�死氏獅祉私糸紙紫肢脂至視詞詩試誌諮資賜雌飼歯事似侍児字寺慈持時次滋治爾璽痔磁示而耳自蒔辞汐鹿式識鴫竺軸宍雫七叱執失嫉室悉湿漆疾質実蔀篠偲柴芝屡蕊縞舎写射捨赦斜煮社紗者謝車遮蛇邪借勺尺杓灼爵酌釈錫若寂弱惹主取守手朱殊狩珠種腫趣酒首儒受呪寿授樹綬需囚収周���".split("");

      for (j = 0; j != D[142].length; ++j) if (D[142][j].charCodeAt(0) !== 0xFFFD) {
        e[D[142][j]] = 36352 + j;
        d[36352 + j] = D[142][j];
      }

      D[143] = "����������������������������������������������������������������宗就州修愁拾洲秀秋終繍習臭舟蒐衆襲讐蹴輯週酋酬集醜什住充十従戎柔汁渋獣縦重銃叔夙宿淑祝縮粛塾熟出術述俊峻春瞬竣舜駿准循旬楯殉淳�準潤盾純巡遵醇順処初所暑曙渚庶緒署書薯藷諸助叙女序徐恕鋤除傷償勝匠升召哨商唱嘗奨妾娼宵将小少尚庄床廠彰承抄招掌捷昇昌昭晶松梢樟樵沼消渉湘焼焦照症省硝礁祥称章笑粧紹肖菖蒋蕉衝裳訟証詔詳象賞醤鉦鍾鐘障鞘上丈丞乗冗剰城場壌嬢常情擾条杖浄状畳穣蒸譲醸錠嘱埴飾���".split("");

      for (j = 0; j != D[143].length; ++j) if (D[143][j].charCodeAt(0) !== 0xFFFD) {
        e[D[143][j]] = 36608 + j;
        d[36608 + j] = D[143][j];
      }

      D[144] = "����������������������������������������������������������������拭植殖燭織職色触食蝕辱尻伸信侵唇娠寝審心慎振新晋森榛浸深申疹真神秦紳臣芯薪親診身辛進針震人仁刃塵壬尋甚尽腎訊迅陣靭笥諏須酢図厨�逗吹垂帥推水炊睡粋翠衰遂酔錐錘随瑞髄崇嵩数枢趨雛据杉椙菅頗雀裾澄摺寸世瀬畝是凄制勢姓征性成政整星晴棲栖正清牲生盛精聖声製西誠誓請逝醒青静斉税脆隻席惜戚斥昔析石積籍績脊責赤跡蹟碩切拙接摂折設窃節説雪絶舌蝉仙先千占宣専尖川戦扇撰栓栴泉浅洗染潜煎煽旋穿箭線���".split("");

      for (j = 0; j != D[144].length; ++j) if (D[144][j].charCodeAt(0) !== 0xFFFD) {
        e[D[144][j]] = 36864 + j;
        d[36864 + j] = D[144][j];
      }

      D[145] = "����������������������������������������������������������������繊羨腺舛船薦詮賎践選遷銭銑閃鮮前善漸然全禅繕膳糎噌塑岨措曾曽楚狙疏疎礎祖租粗素組蘇訴阻遡鼠僧創双叢倉喪壮奏爽宋層匝惣想捜掃挿掻�操早曹巣槍槽漕燥争痩相窓糟総綜聡草荘葬蒼藻装走送遭鎗霜騒像増憎臓蔵贈造促側則即息捉束測足速俗属賊族続卒袖其揃存孫尊損村遜他多太汰詑唾堕妥惰打柁舵楕陀駄騨体堆対耐岱帯待怠態戴替泰滞胎腿苔袋貸退逮隊黛鯛代台大第醍題鷹滝瀧卓啄宅托択拓沢濯琢託鐸濁諾茸凧蛸只���".split("");

      for (j = 0; j != D[145].length; ++j) if (D[145][j].charCodeAt(0) !== 0xFFFD) {
        e[D[145][j]] = 37120 + j;
        d[37120 + j] = D[145][j];
      }

      D[146] = "����������������������������������������������������������������叩但達辰奪脱巽竪辿棚谷狸鱈樽誰丹単嘆坦担探旦歎淡湛炭短端箪綻耽胆蛋誕鍛団壇弾断暖檀段男談値知地弛恥智池痴稚置致蜘遅馳築畜竹筑蓄�逐秩窒茶嫡着中仲宙忠抽昼柱注虫衷註酎鋳駐樗瀦猪苧著貯丁兆凋喋寵帖帳庁弔張彫徴懲挑暢朝潮牒町眺聴脹腸蝶調諜超跳銚長頂鳥勅捗直朕沈珍賃鎮陳津墜椎槌追鎚痛通塚栂掴槻佃漬柘辻蔦綴鍔椿潰坪壷嬬紬爪吊釣鶴亭低停偵剃貞呈堤定帝底庭廷弟悌抵挺提梯汀碇禎程締艇訂諦蹄逓���".split("");

      for (j = 0; j != D[146].length; ++j) if (D[146][j].charCodeAt(0) !== 0xFFFD) {
        e[D[146][j]] = 37376 + j;
        d[37376 + j] = D[146][j];
      }

      D[147] = "����������������������������������������������������������������邸鄭釘鼎泥摘擢敵滴的笛適鏑溺哲徹撤轍迭鉄典填天展店添纏甜貼転顛点伝殿澱田電兎吐堵塗妬屠徒斗杜渡登菟賭途都鍍砥砺努度土奴怒倒党冬�凍刀唐塔塘套宕島嶋悼投搭東桃梼棟盗淘湯涛灯燈当痘祷等答筒糖統到董蕩藤討謄豆踏逃透鐙陶頭騰闘働動同堂導憧撞洞瞳童胴萄道銅峠鴇匿得徳涜特督禿篤毒独読栃橡凸突椴届鳶苫寅酉瀞噸屯惇敦沌豚遁頓呑曇鈍奈那内乍凪薙謎灘捺鍋楢馴縄畷南楠軟難汝二尼弐迩匂賑肉虹廿日乳入���".split("");

      for (j = 0; j != D[147].length; ++j) if (D[147][j].charCodeAt(0) !== 0xFFFD) {
        e[D[147][j]] = 37632 + j;
        d[37632 + j] = D[147][j];
      }

      D[148] = "����������������������������������������������������������������如尿韮任妊忍認濡禰祢寧葱猫熱年念捻撚燃粘乃廼之埜嚢悩濃納能脳膿農覗蚤巴把播覇杷波派琶破婆罵芭馬俳廃拝排敗杯盃牌背肺輩配倍培媒梅�楳煤狽買売賠陪這蝿秤矧萩伯剥博拍柏泊白箔粕舶薄迫曝漠爆縛莫駁麦函箱硲箸肇筈櫨幡肌畑畠八鉢溌発醗髪伐罰抜筏閥鳩噺塙蛤隼伴判半反叛帆搬斑板氾汎版犯班畔繁般藩販範釆煩頒飯挽晩番盤磐蕃蛮匪卑否妃庇彼悲扉批披斐比泌疲皮碑秘緋罷肥被誹費避非飛樋簸備尾微枇毘琵眉美���".split("");

      for (j = 0; j != D[148].length; ++j) if (D[148][j].charCodeAt(0) !== 0xFFFD) {
        e[D[148][j]] = 37888 + j;
        d[37888 + j] = D[148][j];
      }

      D[149] = "����������������������������������������������������������������鼻柊稗匹疋髭彦膝菱肘弼必畢筆逼桧姫媛紐百謬俵彪標氷漂瓢票表評豹廟描病秒苗錨鋲蒜蛭鰭品彬斌浜瀕貧賓頻敏瓶不付埠夫婦富冨布府怖扶敷�斧普浮父符腐膚芙譜負賦赴阜附侮撫武舞葡蕪部封楓風葺蕗伏副復幅服福腹複覆淵弗払沸仏物鮒分吻噴墳憤扮焚奮粉糞紛雰文聞丙併兵塀幣平弊柄並蔽閉陛米頁僻壁癖碧別瞥蔑箆偏変片篇編辺返遍便勉娩弁鞭保舗鋪圃捕歩甫補輔穂募墓慕戊暮母簿菩倣俸包呆報奉宝峰峯崩庖抱捧放方朋���".split("");

      for (j = 0; j != D[149].length; ++j) if (D[149][j].charCodeAt(0) !== 0xFFFD) {
        e[D[149][j]] = 38144 + j;
        d[38144 + j] = D[149][j];
      }

      D[150] = "����������������������������������������������������������������法泡烹砲縫胞芳萌蓬蜂褒訪豊邦鋒飽鳳鵬乏亡傍剖坊妨帽忘忙房暴望某棒冒紡肪膨謀貌貿鉾防吠頬北僕卜墨撲朴牧睦穆釦勃没殆堀幌奔本翻凡盆�摩磨魔麻埋妹昧枚毎哩槙幕膜枕鮪柾鱒桝亦俣又抹末沫迄侭繭麿万慢満漫蔓味未魅巳箕岬密蜜湊蓑稔脈妙粍民眠務夢無牟矛霧鵡椋婿娘冥名命明盟迷銘鳴姪牝滅免棉綿緬面麺摸模茂妄孟毛猛盲網耗蒙儲木黙目杢勿餅尤戻籾貰問悶紋門匁也冶夜爺耶野弥矢厄役約薬訳躍靖柳薮鑓愉愈油癒���".split("");

      for (j = 0; j != D[150].length; ++j) if (D[150][j].charCodeAt(0) !== 0xFFFD) {
        e[D[150][j]] = 38400 + j;
        d[38400 + j] = D[150][j];
      }

      D[151] = "����������������������������������������������������������������諭輸唯佑優勇友宥幽悠憂揖有柚湧涌猶猷由祐裕誘遊邑郵雄融夕予余与誉輿預傭幼妖容庸揚揺擁曜楊様洋溶熔用窯羊耀葉蓉要謡踊遥陽養慾抑欲�沃浴翌翼淀羅螺裸来莱頼雷洛絡落酪乱卵嵐欄濫藍蘭覧利吏履李梨理璃痢裏裡里離陸律率立葎掠略劉流溜琉留硫粒隆竜龍侶慮旅虜了亮僚両凌寮料梁涼猟療瞭稜糧良諒遼量陵領力緑倫厘林淋燐琳臨輪隣鱗麟瑠塁涙累類令伶例冷励嶺怜玲礼苓鈴隷零霊麗齢暦歴列劣烈裂廉恋憐漣煉簾練聯���".split("");

      for (j = 0; j != D[151].length; ++j) if (D[151][j].charCodeAt(0) !== 0xFFFD) {
        e[D[151][j]] = 38656 + j;
        d[38656 + j] = D[151][j];
      }

      D[152] = "����������������������������������������������������������������蓮連錬呂魯櫓炉賂路露労婁廊弄朗楼榔浪漏牢狼篭老聾蝋郎六麓禄肋録論倭和話歪賄脇惑枠鷲亙亘鰐詫藁蕨椀湾碗腕��������������������������������������������弌丐丕个丱丶丼丿乂乖乘亂亅豫亊舒弍于亞亟亠亢亰亳亶从仍仄仆仂仗仞仭仟价伉佚估佛佝佗佇佶侈侏侘佻佩佰侑佯來侖儘俔俟俎俘俛俑俚俐俤俥倚倨倔倪倥倅伜俶倡倩倬俾俯們倆偃假會偕偐偈做偖偬偸傀傚傅傴傲���".split("");

      for (j = 0; j != D[152].length; ++j) if (D[152][j].charCodeAt(0) !== 0xFFFD) {
        e[D[152][j]] = 38912 + j;
        d[38912 + j] = D[152][j];
      }

      D[153] = "����������������������������������������������������������������僉僊傳僂僖僞僥僭僣僮價僵儉儁儂儖儕儔儚儡儺儷儼儻儿兀兒兌兔兢竸兩兪兮冀冂囘册冉冏冑冓冕冖冤冦冢冩冪冫决冱冲冰况冽凅凉凛几處凩凭�凰凵凾刄刋刔刎刧刪刮刳刹剏剄剋剌剞剔剪剴剩剳剿剽劍劔劒剱劈劑辨辧劬劭劼劵勁勍勗勞勣勦飭勠勳勵勸勹匆匈甸匍匐匏匕匚匣匯匱匳匸區卆卅丗卉卍凖卞卩卮夘卻卷厂厖厠厦厥厮厰厶參簒雙叟曼燮叮叨叭叺吁吽呀听吭吼吮吶吩吝呎咏呵咎呟呱呷呰咒呻咀呶咄咐咆哇咢咸咥咬哄哈咨���".split("");

      for (j = 0; j != D[153].length; ++j) if (D[153][j].charCodeAt(0) !== 0xFFFD) {
        e[D[153][j]] = 39168 + j;
        d[39168 + j] = D[153][j];
      }

      D[154] = "����������������������������������������������������������������咫哂咤咾咼哘哥哦唏唔哽哮哭哺哢唹啀啣啌售啜啅啖啗唸唳啝喙喀咯喊喟啻啾喘喞單啼喃喩喇喨嗚嗅嗟嗄嗜嗤嗔嘔嗷嘖嗾嗽嘛嗹噎噐營嘴嘶嘲嘸�噫噤嘯噬噪嚆嚀嚊嚠嚔嚏嚥嚮嚶嚴囂嚼囁囃囀囈囎囑囓囗囮囹圀囿圄圉圈國圍圓團圖嗇圜圦圷圸坎圻址坏坩埀垈坡坿垉垓垠垳垤垪垰埃埆埔埒埓堊埖埣堋堙堝塲堡塢塋塰毀塒堽塹墅墹墟墫墺壞墻墸墮壅壓壑壗壙壘壥壜壤壟壯壺壹壻壼壽夂夊夐夛梦夥夬夭夲夸夾竒奕奐奎奚奘奢奠奧奬奩���".split("");

      for (j = 0; j != D[154].length; ++j) if (D[154][j].charCodeAt(0) !== 0xFFFD) {
        e[D[154][j]] = 39424 + j;
        d[39424 + j] = D[154][j];
      }

      D[155] = "����������������������������������������������������������������奸妁妝佞侫妣妲姆姨姜妍姙姚娥娟娑娜娉娚婀婬婉娵娶婢婪媚媼媾嫋嫂媽嫣嫗嫦嫩嫖嫺嫻嬌嬋嬖嬲嫐嬪嬶嬾孃孅孀孑孕孚孛孥孩孰孳孵學斈孺宀�它宦宸寃寇寉寔寐寤實寢寞寥寫寰寶寳尅將專對尓尠尢尨尸尹屁屆屎屓屐屏孱屬屮乢屶屹岌岑岔妛岫岻岶岼岷峅岾峇峙峩峽峺峭嶌峪崋崕崗嵜崟崛崑崔崢崚崙崘嵌嵒嵎嵋嵬嵳嵶嶇嶄嶂嶢嶝嶬嶮嶽嶐嶷嶼巉巍巓巒巖巛巫已巵帋帚帙帑帛帶帷幄幃幀幎幗幔幟幢幤幇幵并幺麼广庠廁廂廈廐廏���".split("");

      for (j = 0; j != D[155].length; ++j) if (D[155][j].charCodeAt(0) !== 0xFFFD) {
        e[D[155][j]] = 39680 + j;
        d[39680 + j] = D[155][j];
      }

      D[156] = "����������������������������������������������������������������廖廣廝廚廛廢廡廨廩廬廱廳廰廴廸廾弃弉彝彜弋弑弖弩弭弸彁彈彌彎弯彑彖彗彙彡彭彳彷徃徂彿徊很徑徇從徙徘徠徨徭徼忖忻忤忸忱忝悳忿怡恠�怙怐怩怎怱怛怕怫怦怏怺恚恁恪恷恟恊恆恍恣恃恤恂恬恫恙悁悍惧悃悚悄悛悖悗悒悧悋惡悸惠惓悴忰悽惆悵惘慍愕愆惶惷愀惴惺愃愡惻惱愍愎慇愾愨愧慊愿愼愬愴愽慂慄慳慷慘慙慚慫慴慯慥慱慟慝慓慵憙憖憇憬憔憚憊憑憫憮懌懊應懷懈懃懆憺懋罹懍懦懣懶懺懴懿懽懼懾戀戈戉戍戌戔戛���".split("");

      for (j = 0; j != D[156].length; ++j) if (D[156][j].charCodeAt(0) !== 0xFFFD) {
        e[D[156][j]] = 39936 + j;
        d[39936 + j] = D[156][j];
      }

      D[157] = "����������������������������������������������������������������戞戡截戮戰戲戳扁扎扞扣扛扠扨扼抂抉找抒抓抖拔抃抔拗拑抻拏拿拆擔拈拜拌拊拂拇抛拉挌拮拱挧挂挈拯拵捐挾捍搜捏掖掎掀掫捶掣掏掉掟掵捫�捩掾揩揀揆揣揉插揶揄搖搴搆搓搦搶攝搗搨搏摧摯摶摎攪撕撓撥撩撈撼據擒擅擇撻擘擂擱擧舉擠擡抬擣擯攬擶擴擲擺攀擽攘攜攅攤攣攫攴攵攷收攸畋效敖敕敍敘敞敝敲數斂斃變斛斟斫斷旃旆旁旄旌旒旛旙无旡旱杲昊昃旻杳昵昶昴昜晏晄晉晁晞晝晤晧晨晟晢晰暃暈暎暉暄暘暝曁暹曉暾暼���".split("");

      for (j = 0; j != D[157].length; ++j) if (D[157][j].charCodeAt(0) !== 0xFFFD) {
        e[D[157][j]] = 40192 + j;
        d[40192 + j] = D[157][j];
      }

      D[158] = "����������������������������������������������������������������曄暸曖曚曠昿曦曩曰曵曷朏朖朞朦朧霸朮朿朶杁朸朷杆杞杠杙杣杤枉杰枩杼杪枌枋枦枡枅枷柯枴柬枳柩枸柤柞柝柢柮枹柎柆柧檜栞框栩桀桍栲桎�梳栫桙档桷桿梟梏梭梔條梛梃檮梹桴梵梠梺椏梍桾椁棊椈棘椢椦棡椌棍棔棧棕椶椒椄棗棣椥棹棠棯椨椪椚椣椡棆楹楷楜楸楫楔楾楮椹楴椽楙椰楡楞楝榁楪榲榮槐榿槁槓榾槎寨槊槝榻槃榧樮榑榠榜榕榴槞槨樂樛槿權槹槲槧樅榱樞槭樔槫樊樒櫁樣樓橄樌橲樶橸橇橢橙橦橈樸樢檐檍檠檄檢檣���".split("");

      for (j = 0; j != D[158].length; ++j) if (D[158][j].charCodeAt(0) !== 0xFFFD) {
        e[D[158][j]] = 40448 + j;
        d[40448 + j] = D[158][j];
      }

      D[159] = "����������������������������������������������������������������檗蘗檻櫃櫂檸檳檬櫞櫑櫟檪櫚櫪櫻欅蘖櫺欒欖鬱欟欸欷盜欹飮歇歃歉歐歙歔歛歟歡歸歹歿殀殄殃殍殘殕殞殤殪殫殯殲殱殳殷殼毆毋毓毟毬毫毳毯�麾氈氓气氛氤氣汞汕汢汪沂沍沚沁沛汾汨汳沒沐泄泱泓沽泗泅泝沮沱沾沺泛泯泙泪洟衍洶洫洽洸洙洵洳洒洌浣涓浤浚浹浙涎涕濤涅淹渕渊涵淇淦涸淆淬淞淌淨淒淅淺淙淤淕淪淮渭湮渮渙湲湟渾渣湫渫湶湍渟湃渺湎渤滿渝游溂溪溘滉溷滓溽溯滄溲滔滕溏溥滂溟潁漑灌滬滸滾漿滲漱滯漲滌���".split("");

      for (j = 0; j != D[159].length; ++j) if (D[159][j].charCodeAt(0) !== 0xFFFD) {
        e[D[159][j]] = 40704 + j;
        d[40704 + j] = D[159][j];
      }

      D[224] = "����������������������������������������������������������������漾漓滷澆潺潸澁澀潯潛濳潭澂潼潘澎澑濂潦澳澣澡澤澹濆澪濟濕濬濔濘濱濮濛瀉瀋濺瀑瀁瀏濾瀛瀚潴瀝瀘瀟瀰瀾瀲灑灣炙炒炯烱炬炸炳炮烟烋烝�烙焉烽焜焙煥煕熈煦煢煌煖煬熏燻熄熕熨熬燗熹熾燒燉燔燎燠燬燧燵燼燹燿爍爐爛爨爭爬爰爲爻爼爿牀牆牋牘牴牾犂犁犇犒犖犢犧犹犲狃狆狄狎狒狢狠狡狹狷倏猗猊猜猖猝猴猯猩猥猾獎獏默獗獪獨獰獸獵獻獺珈玳珎玻珀珥珮珞璢琅瑯琥珸琲琺瑕琿瑟瑙瑁瑜瑩瑰瑣瑪瑶瑾璋璞璧瓊瓏瓔珱���".split("");

      for (j = 0; j != D[224].length; ++j) if (D[224][j].charCodeAt(0) !== 0xFFFD) {
        e[D[224][j]] = 57344 + j;
        d[57344 + j] = D[224][j];
      }

      D[225] = "����������������������������������������������������������������瓠瓣瓧瓩瓮瓲瓰瓱瓸瓷甄甃甅甌甎甍甕甓甞甦甬甼畄畍畊畉畛畆畚畩畤畧畫畭畸當疆疇畴疊疉疂疔疚疝疥疣痂疳痃疵疽疸疼疱痍痊痒痙痣痞痾痿�痼瘁痰痺痲痳瘋瘍瘉瘟瘧瘠瘡瘢瘤瘴瘰瘻癇癈癆癜癘癡癢癨癩癪癧癬癰癲癶癸發皀皃皈皋皎皖皓皙皚皰皴皸皹皺盂盍盖盒盞盡盥盧盪蘯盻眈眇眄眩眤眞眥眦眛眷眸睇睚睨睫睛睥睿睾睹瞎瞋瞑瞠瞞瞰瞶瞹瞿瞼瞽瞻矇矍矗矚矜矣矮矼砌砒礦砠礪硅碎硴碆硼碚碌碣碵碪碯磑磆磋磔碾碼磅磊磬���".split("");

      for (j = 0; j != D[225].length; ++j) if (D[225][j].charCodeAt(0) !== 0xFFFD) {
        e[D[225][j]] = 57600 + j;
        d[57600 + j] = D[225][j];
      }

      D[226] = "����������������������������������������������������������������磧磚磽磴礇礒礑礙礬礫祀祠祗祟祚祕祓祺祿禊禝禧齋禪禮禳禹禺秉秕秧秬秡秣稈稍稘稙稠稟禀稱稻稾稷穃穗穉穡穢穩龝穰穹穽窈窗窕窘窖窩竈窰�窶竅竄窿邃竇竊竍竏竕竓站竚竝竡竢竦竭竰笂笏笊笆笳笘笙笞笵笨笶筐筺笄筍笋筌筅筵筥筴筧筰筱筬筮箝箘箟箍箜箚箋箒箏筝箙篋篁篌篏箴篆篝篩簑簔篦篥籠簀簇簓篳篷簗簍篶簣簧簪簟簷簫簽籌籃籔籏籀籐籘籟籤籖籥籬籵粃粐粤粭粢粫粡粨粳粲粱粮粹粽糀糅糂糘糒糜糢鬻糯糲糴糶糺紆���".split("");

      for (j = 0; j != D[226].length; ++j) if (D[226][j].charCodeAt(0) !== 0xFFFD) {
        e[D[226][j]] = 57856 + j;
        d[57856 + j] = D[226][j];
      }

      D[227] = "����������������������������������������������������������������紂紜紕紊絅絋紮紲紿紵絆絳絖絎絲絨絮絏絣經綉絛綏絽綛綺綮綣綵緇綽綫總綢綯緜綸綟綰緘緝緤緞緻緲緡縅縊縣縡縒縱縟縉縋縢繆繦縻縵縹繃縷�縲縺繧繝繖繞繙繚繹繪繩繼繻纃緕繽辮繿纈纉續纒纐纓纔纖纎纛纜缸缺罅罌罍罎罐网罕罔罘罟罠罨罩罧罸羂羆羃羈羇羌羔羞羝羚羣羯羲羹羮羶羸譱翅翆翊翕翔翡翦翩翳翹飜耆耄耋耒耘耙耜耡耨耿耻聊聆聒聘聚聟聢聨聳聲聰聶聹聽聿肄肆肅肛肓肚肭冐肬胛胥胙胝胄胚胖脉胯胱脛脩脣脯腋���".split("");

      for (j = 0; j != D[227].length; ++j) if (D[227][j].charCodeAt(0) !== 0xFFFD) {
        e[D[227][j]] = 58112 + j;
        d[58112 + j] = D[227][j];
      }

      D[228] = "����������������������������������������������������������������隋腆脾腓腑胼腱腮腥腦腴膃膈膊膀膂膠膕膤膣腟膓膩膰膵膾膸膽臀臂膺臉臍臑臙臘臈臚臟臠臧臺臻臾舁舂舅與舊舍舐舖舩舫舸舳艀艙艘艝艚艟艤�艢艨艪艫舮艱艷艸艾芍芒芫芟芻芬苡苣苟苒苴苳苺莓范苻苹苞茆苜茉苙茵茴茖茲茱荀茹荐荅茯茫茗茘莅莚莪莟莢莖茣莎莇莊荼莵荳荵莠莉莨菴萓菫菎菽萃菘萋菁菷萇菠菲萍萢萠莽萸蔆菻葭萪萼蕚蒄葷葫蒭葮蒂葩葆萬葯葹萵蓊葢蒹蒿蒟蓙蓍蒻蓚蓐蓁蓆蓖蒡蔡蓿蓴蔗蔘蔬蔟蔕蔔蓼蕀蕣蕘蕈���".split("");

      for (j = 0; j != D[228].length; ++j) if (D[228][j].charCodeAt(0) !== 0xFFFD) {
        e[D[228][j]] = 58368 + j;
        d[58368 + j] = D[228][j];
      }

      D[229] = "����������������������������������������������������������������蕁蘂蕋蕕薀薤薈薑薊薨蕭薔薛藪薇薜蕷蕾薐藉薺藏薹藐藕藝藥藜藹蘊蘓蘋藾藺蘆蘢蘚蘰蘿虍乕虔號虧虱蚓蚣蚩蚪蚋蚌蚶蚯蛄蛆蚰蛉蠣蚫蛔蛞蛩蛬�蛟蛛蛯蜒蜆蜈蜀蜃蛻蜑蜉蜍蛹蜊蜴蜿蜷蜻蜥蜩蜚蝠蝟蝸蝌蝎蝴蝗蝨蝮蝙蝓蝣蝪蠅螢螟螂螯蟋螽蟀蟐雖螫蟄螳蟇蟆螻蟯蟲蟠蠏蠍蟾蟶蟷蠎蟒蠑蠖蠕蠢蠡蠱蠶蠹蠧蠻衄衂衒衙衞衢衫袁衾袞衵衽袵衲袂袗袒袮袙袢袍袤袰袿袱裃裄裔裘裙裝裹褂裼裴裨裲褄褌褊褓襃褞褥褪褫襁襄褻褶褸襌褝襠襞���".split("");

      for (j = 0; j != D[229].length; ++j) if (D[229][j].charCodeAt(0) !== 0xFFFD) {
        e[D[229][j]] = 58624 + j;
        d[58624 + j] = D[229][j];
      }

      D[230] = "����������������������������������������������������������������襦襤襭襪襯襴襷襾覃覈覊覓覘覡覩覦覬覯覲覺覽覿觀觚觜觝觧觴觸訃訖訐訌訛訝訥訶詁詛詒詆詈詼詭詬詢誅誂誄誨誡誑誥誦誚誣諄諍諂諚諫諳諧�諤諱謔諠諢諷諞諛謌謇謚諡謖謐謗謠謳鞫謦謫謾謨譁譌譏譎證譖譛譚譫譟譬譯譴譽讀讌讎讒讓讖讙讚谺豁谿豈豌豎豐豕豢豬豸豺貂貉貅貊貍貎貔豼貘戝貭貪貽貲貳貮貶賈賁賤賣賚賽賺賻贄贅贊贇贏贍贐齎贓賍贔贖赧赭赱赳趁趙跂趾趺跏跚跖跌跛跋跪跫跟跣跼踈踉跿踝踞踐踟蹂踵踰踴蹊���".split("");

      for (j = 0; j != D[230].length; ++j) if (D[230][j].charCodeAt(0) !== 0xFFFD) {
        e[D[230][j]] = 58880 + j;
        d[58880 + j] = D[230][j];
      }

      D[231] = "����������������������������������������������������������������蹇蹉蹌蹐蹈蹙蹤蹠踪蹣蹕蹶蹲蹼躁躇躅躄躋躊躓躑躔躙躪躡躬躰軆躱躾軅軈軋軛軣軼軻軫軾輊輅輕輒輙輓輜輟輛輌輦輳輻輹轅轂輾轌轉轆轎轗轜�轢轣轤辜辟辣辭辯辷迚迥迢迪迯邇迴逅迹迺逑逕逡逍逞逖逋逧逶逵逹迸遏遐遑遒逎遉逾遖遘遞遨遯遶隨遲邂遽邁邀邊邉邏邨邯邱邵郢郤扈郛鄂鄒鄙鄲鄰酊酖酘酣酥酩酳酲醋醉醂醢醫醯醪醵醴醺釀釁釉釋釐釖釟釡釛釼釵釶鈞釿鈔鈬鈕鈑鉞鉗鉅鉉鉤鉈銕鈿鉋鉐銜銖銓銛鉚鋏銹銷鋩錏鋺鍄錮���".split("");

      for (j = 0; j != D[231].length; ++j) if (D[231][j].charCodeAt(0) !== 0xFFFD) {
        e[D[231][j]] = 59136 + j;
        d[59136 + j] = D[231][j];
      }

      D[232] = "����������������������������������������������������������������錙錢錚錣錺錵錻鍜鍠鍼鍮鍖鎰鎬鎭鎔鎹鏖鏗鏨鏥鏘鏃鏝鏐鏈鏤鐚鐔鐓鐃鐇鐐鐶鐫鐵鐡鐺鑁鑒鑄鑛鑠鑢鑞鑪鈩鑰鑵鑷鑽鑚鑼鑾钁鑿閂閇閊閔閖閘閙�閠閨閧閭閼閻閹閾闊濶闃闍闌闕闔闖關闡闥闢阡阨阮阯陂陌陏陋陷陜陞陝陟陦陲陬隍隘隕隗險隧隱隲隰隴隶隸隹雎雋雉雍襍雜霍雕雹霄霆霈霓霎霑霏霖霙霤霪霰霹霽霾靄靆靈靂靉靜靠靤靦靨勒靫靱靹鞅靼鞁靺鞆鞋鞏鞐鞜鞨鞦鞣鞳鞴韃韆韈韋韜韭齏韲竟韶韵頏頌頸頤頡頷頽顆顏顋顫顯顰���".split("");

      for (j = 0; j != D[232].length; ++j) if (D[232][j].charCodeAt(0) !== 0xFFFD) {
        e[D[232][j]] = 59392 + j;
        d[59392 + j] = D[232][j];
      }

      D[233] = "����������������������������������������������������������������顱顴顳颪颯颱颶飄飃飆飩飫餃餉餒餔餘餡餝餞餤餠餬餮餽餾饂饉饅饐饋饑饒饌饕馗馘馥馭馮馼駟駛駝駘駑駭駮駱駲駻駸騁騏騅駢騙騫騷驅驂驀驃�騾驕驍驛驗驟驢驥驤驩驫驪骭骰骼髀髏髑髓體髞髟髢髣髦髯髫髮髴髱髷髻鬆鬘鬚鬟鬢鬣鬥鬧鬨鬩鬪鬮鬯鬲魄魃魏魍魎魑魘魴鮓鮃鮑鮖鮗鮟鮠鮨鮴鯀鯊鮹鯆鯏鯑鯒鯣鯢鯤鯔鯡鰺鯲鯱鯰鰕鰔鰉鰓鰌鰆鰈鰒鰊鰄鰮鰛鰥鰤鰡鰰鱇鰲鱆鰾鱚鱠鱧鱶鱸鳧鳬鳰鴉鴈鳫鴃鴆鴪鴦鶯鴣鴟鵄鴕鴒鵁鴿鴾鵆鵈���".split("");

      for (j = 0; j != D[233].length; ++j) if (D[233][j].charCodeAt(0) !== 0xFFFD) {
        e[D[233][j]] = 59648 + j;
        d[59648 + j] = D[233][j];
      }

      D[234] = "����������������������������������������������������������������鵝鵞鵤鵑鵐鵙鵲鶉鶇鶫鵯鵺鶚鶤鶩鶲鷄鷁鶻鶸鶺鷆鷏鷂鷙鷓鷸鷦鷭鷯鷽鸚鸛鸞鹵鹹鹽麁麈麋麌麒麕麑麝麥麩麸麪麭靡黌黎黏黐黔黜點黝黠黥黨黯�黴黶黷黹黻黼黽鼇鼈皷鼕鼡鼬鼾齊齒齔齣齟齠齡齦齧齬齪齷齲齶龕龜龠堯槇遙瑤凜熙�������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[234].length; ++j) if (D[234][j].charCodeAt(0) !== 0xFFFD) {
        e[D[234][j]] = 59904 + j;
        d[59904 + j] = D[234][j];
      }

      D[237] = "����������������������������������������������������������������纊褜鍈銈蓜俉炻昱棈鋹曻彅丨仡仼伀伃伹佖侒侊侚侔俍偀倢俿倞偆偰偂傔僴僘兊兤冝冾凬刕劜劦勀勛匀匇匤卲厓厲叝﨎咜咊咩哿喆坙坥垬埈埇﨏�塚增墲夋奓奛奝奣妤妺孖寀甯寘寬尞岦岺峵崧嵓﨑嵂嵭嶸嶹巐弡弴彧德忞恝悅悊惞惕愠惲愑愷愰憘戓抦揵摠撝擎敎昀昕昻昉昮昞昤晥晗晙晴晳暙暠暲暿曺朎朗杦枻桒柀栁桄棏﨓楨﨔榘槢樰橫橆橳橾櫢櫤毖氿汜沆汯泚洄涇浯涖涬淏淸淲淼渹湜渧渼溿澈澵濵瀅瀇瀨炅炫焏焄煜煆煇凞燁燾犱���".split("");

      for (j = 0; j != D[237].length; ++j) if (D[237][j].charCodeAt(0) !== 0xFFFD) {
        e[D[237][j]] = 60672 + j;
        d[60672 + j] = D[237][j];
      }

      D[238] = "����������������������������������������������������������������犾猤猪獷玽珉珖珣珒琇珵琦琪琩琮瑢璉璟甁畯皂皜皞皛皦益睆劯砡硎硤硺礰礼神祥禔福禛竑竧靖竫箞精絈絜綷綠緖繒罇羡羽茁荢荿菇菶葈蒴蕓蕙�蕫﨟薰蘒﨡蠇裵訒訷詹誧誾諟諸諶譓譿賰賴贒赶﨣軏﨤逸遧郞都鄕鄧釚釗釞釭釮釤釥鈆鈐鈊鈺鉀鈼鉎鉙鉑鈹鉧銧鉷鉸鋧鋗鋙鋐﨧鋕鋠鋓錥錡鋻﨨錞鋿錝錂鍰鍗鎤鏆鏞鏸鐱鑅鑈閒隆﨩隝隯霳霻靃靍靏靑靕顗顥飯飼餧館馞驎髙髜魵魲鮏鮱鮻鰀鵰鵫鶴鸙黑��ⅰⅱⅲⅳⅴⅵⅶⅷⅸⅹ￢￤＇＂���".split("");

      for (j = 0; j != D[238].length; ++j) if (D[238][j].charCodeAt(0) !== 0xFFFD) {
        e[D[238][j]] = 60928 + j;
        d[60928 + j] = D[238][j];
      }

      D[250] = "����������������������������������������������������������������ⅰⅱⅲⅳⅴⅵⅶⅷⅸⅹⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ￢￤＇＂㈱№℡∵纊褜鍈銈蓜俉炻昱棈鋹曻彅丨仡仼伀伃伹佖侒侊侚侔俍偀倢俿倞偆偰偂傔僴僘兊�兤冝冾凬刕劜劦勀勛匀匇匤卲厓厲叝﨎咜咊咩哿喆坙坥垬埈埇﨏塚增墲夋奓奛奝奣妤妺孖寀甯寘寬尞岦岺峵崧嵓﨑嵂嵭嶸嶹巐弡弴彧德忞恝悅悊惞惕愠惲愑愷愰憘戓抦揵摠撝擎敎昀昕昻昉昮昞昤晥晗晙晴晳暙暠暲暿曺朎朗杦枻桒柀栁桄棏﨓楨﨔榘槢樰橫橆橳橾櫢櫤毖氿汜沆汯泚洄涇浯���".split("");

      for (j = 0; j != D[250].length; ++j) if (D[250][j].charCodeAt(0) !== 0xFFFD) {
        e[D[250][j]] = 64000 + j;
        d[64000 + j] = D[250][j];
      }

      D[251] = "����������������������������������������������������������������涖涬淏淸淲淼渹湜渧渼溿澈澵濵瀅瀇瀨炅炫焏焄煜煆煇凞燁燾犱犾猤猪獷玽珉珖珣珒琇珵琦琪琩琮瑢璉璟甁畯皂皜皞皛皦益睆劯砡硎硤硺礰礼神�祥禔福禛竑竧靖竫箞精絈絜綷綠緖繒罇羡羽茁荢荿菇菶葈蒴蕓蕙蕫﨟薰蘒﨡蠇裵訒訷詹誧誾諟諸諶譓譿賰賴贒赶﨣軏﨤逸遧郞都鄕鄧釚釗釞釭釮釤釥鈆鈐鈊鈺鉀鈼鉎鉙鉑鈹鉧銧鉷鉸鋧鋗鋙鋐﨧鋕鋠鋓錥錡鋻﨨錞鋿錝錂鍰鍗鎤鏆鏞鏸鐱鑅鑈閒隆﨩隝隯霳霻靃靍靏靑靕顗顥飯飼餧館馞驎髙���".split("");

      for (j = 0; j != D[251].length; ++j) if (D[251][j].charCodeAt(0) !== 0xFFFD) {
        e[D[251][j]] = 64256 + j;
        d[64256 + j] = D[251][j];
      }

      D[252] = "����������������������������������������������������������������髜魵魲鮏鮱鮻鰀鵰鵫鶴鸙黑������������������������������������������������������������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[252].length; ++j) if (D[252][j].charCodeAt(0) !== 0xFFFD) {
        e[D[252][j]] = 64512 + j;
        d[64512 + j] = D[252][j];
      }

      return {
        "enc": e,
        "dec": d
      };
    }();

    cptable[936] = function () {
      var d = [],
          e = {},
          D = [],
          j;
      D[0] = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[0].length; ++j) if (D[0][j].charCodeAt(0) !== 0xFFFD) {
        e[D[0][j]] = 0 + j;
        d[0 + j] = D[0][j];
      }

      D[129] = "����������������������������������������������������������������丂丄丅丆丏丒丗丟丠両丣並丩丮丯丱丳丵丷丼乀乁乂乄乆乊乑乕乗乚乛乢乣乤乥乧乨乪乫乬乭乮乯乲乴乵乶乷乸乹乺乻乼乽乿亀亁亂亃亄亅亇亊�亐亖亗亙亜亝亞亣亪亯亰亱亴亶亷亸亹亼亽亾仈仌仏仐仒仚仛仜仠仢仦仧仩仭仮仯仱仴仸仹仺仼仾伀伂伃伄伅伆伇伈伋伌伒伓伔伕伖伜伝伡伣伨伩伬伭伮伱伳伵伷伹伻伾伿佀佁佂佄佅佇佈佉佊佋佌佒佔佖佡佢佦佨佪佫佭佮佱佲併佷佸佹佺佽侀侁侂侅來侇侊侌侎侐侒侓侕侖侘侙侚侜侞侟価侢�".split("");

      for (j = 0; j != D[129].length; ++j) if (D[129][j].charCodeAt(0) !== 0xFFFD) {
        e[D[129][j]] = 33024 + j;
        d[33024 + j] = D[129][j];
      }

      D[130] = "����������������������������������������������������������������侤侫侭侰侱侲侳侴侶侷侸侹侺侻侼侽侾俀俁係俆俇俈俉俋俌俍俒俓俔俕俖俙俛俠俢俤俥俧俫俬俰俲俴俵俶俷俹俻俼俽俿倀倁倂倃倄倅倆倇倈倉倊�個倎倐們倓倕倖倗倛倝倞倠倢倣値倧倫倯倰倱倲倳倴倵倶倷倸倹倻倽倿偀偁偂偄偅偆偉偊偋偍偐偑偒偓偔偖偗偘偙偛偝偞偟偠偡偢偣偤偦偧偨偩偪偫偭偮偯偰偱偲偳側偵偸偹偺偼偽傁傂傃傄傆傇傉傊傋傌傎傏傐傑傒傓傔傕傖傗傘備傚傛傜傝傞傟傠傡傢傤傦傪傫傭傮傯傰傱傳傴債傶傷傸傹傼�".split("");

      for (j = 0; j != D[130].length; ++j) if (D[130][j].charCodeAt(0) !== 0xFFFD) {
        e[D[130][j]] = 33280 + j;
        d[33280 + j] = D[130][j];
      }

      D[131] = "����������������������������������������������������������������傽傾傿僀僁僂僃僄僅僆僇僈僉僊僋僌働僎僐僑僒僓僔僕僗僘僙僛僜僝僞僟僠僡僢僣僤僥僨僩僪僫僯僰僱僲僴僶僷僸價僺僼僽僾僿儀儁儂儃億儅儈�儉儊儌儍儎儏儐儑儓儔儕儖儗儘儙儚儛儜儝儞償儠儢儣儤儥儦儧儨儩優儫儬儭儮儯儰儱儲儳儴儵儶儷儸儹儺儻儼儽儾兂兇兊兌兎兏児兒兓兗兘兙兛兝兞兟兠兡兣兤兦內兩兪兯兲兺兾兿冃冄円冇冊冋冎冏冐冑冓冔冘冚冝冞冟冡冣冦冧冨冩冪冭冮冴冸冹冺冾冿凁凂凃凅凈凊凍凎凐凒凓凔凕凖凗�".split("");

      for (j = 0; j != D[131].length; ++j) if (D[131][j].charCodeAt(0) !== 0xFFFD) {
        e[D[131][j]] = 33536 + j;
        d[33536 + j] = D[131][j];
      }

      D[132] = "����������������������������������������������������������������凘凙凚凜凞凟凢凣凥処凧凨凩凪凬凮凱凲凴凷凾刄刅刉刋刌刏刐刓刔刕刜刞刟刡刢刣別刦刧刪刬刯刱刲刴刵刼刾剄剅剆則剈剉剋剎剏剒剓剕剗剘�剙剚剛剝剟剠剢剣剤剦剨剫剬剭剮剰剱剳剴創剶剷剸剹剺剻剼剾劀劃劄劅劆劇劉劊劋劌劍劎劏劑劒劔劕劖劗劘劙劚劜劤劥劦劧劮劯劰労劵劶劷劸効劺劻劼劽勀勁勂勄勅勆勈勊勌勍勎勏勑勓勔動勗務勚勛勜勝勞勠勡勢勣勥勦勧勨勩勪勫勬勭勮勯勱勲勳勴勵勶勷勸勻勼勽匁匂匃匄匇匉匊匋匌匎�".split("");

      for (j = 0; j != D[132].length; ++j) if (D[132][j].charCodeAt(0) !== 0xFFFD) {
        e[D[132][j]] = 33792 + j;
        d[33792 + j] = D[132][j];
      }

      D[133] = "����������������������������������������������������������������匑匒匓匔匘匛匜匞匟匢匤匥匧匨匩匫匬匭匯匰匱匲匳匴匵匶匷匸匼匽區卂卄卆卋卌卍卐協単卙卛卝卥卨卪卬卭卲卶卹卻卼卽卾厀厁厃厇厈厊厎厏�厐厑厒厓厔厖厗厙厛厜厞厠厡厤厧厪厫厬厭厯厰厱厲厳厴厵厷厸厹厺厼厽厾叀參叄叅叆叇収叏叐叒叓叕叚叜叝叞叡叢叧叴叺叾叿吀吂吅吇吋吔吘吙吚吜吢吤吥吪吰吳吶吷吺吽吿呁呂呄呅呇呉呌呍呎呏呑呚呝呞呟呠呡呣呥呧呩呪呫呬呭呮呯呰呴呹呺呾呿咁咃咅咇咈咉咊咍咑咓咗咘咜咞咟咠咡�".split("");

      for (j = 0; j != D[133].length; ++j) if (D[133][j].charCodeAt(0) !== 0xFFFD) {
        e[D[133][j]] = 34048 + j;
        d[34048 + j] = D[133][j];
      }

      D[134] = "����������������������������������������������������������������咢咥咮咰咲咵咶咷咹咺咼咾哃哅哊哋哖哘哛哠員哢哣哤哫哬哯哰哱哴哵哶哷哸哹哻哾唀唂唃唄唅唈唊唋唌唍唎唒唓唕唖唗唘唙唚唜唝唞唟唡唥唦�唨唩唫唭唲唴唵唶唸唹唺唻唽啀啂啅啇啈啋啌啍啎問啑啒啓啔啗啘啙啚啛啝啞啟啠啢啣啨啩啫啯啰啱啲啳啴啹啺啽啿喅喆喌喍喎喐喒喓喕喖喗喚喛喞喠喡喢喣喤喥喦喨喩喪喫喬喭單喯喰喲喴営喸喺喼喿嗀嗁嗂嗃嗆嗇嗈嗊嗋嗎嗏嗐嗕嗗嗘嗙嗚嗛嗞嗠嗢嗧嗩嗭嗮嗰嗱嗴嗶嗸嗹嗺嗻嗼嗿嘂嘃嘄嘅�".split("");

      for (j = 0; j != D[134].length; ++j) if (D[134][j].charCodeAt(0) !== 0xFFFD) {
        e[D[134][j]] = 34304 + j;
        d[34304 + j] = D[134][j];
      }

      D[135] = "����������������������������������������������������������������嘆嘇嘊嘋嘍嘐嘑嘒嘓嘔嘕嘖嘗嘙嘚嘜嘝嘠嘡嘢嘥嘦嘨嘩嘪嘫嘮嘯嘰嘳嘵嘷嘸嘺嘼嘽嘾噀噁噂噃噄噅噆噇噈噉噊噋噏噐噑噒噓噕噖噚噛噝噞噟噠噡�噣噥噦噧噭噮噯噰噲噳噴噵噷噸噹噺噽噾噿嚀嚁嚂嚃嚄嚇嚈嚉嚊嚋嚌嚍嚐嚑嚒嚔嚕嚖嚗嚘嚙嚚嚛嚜嚝嚞嚟嚠嚡嚢嚤嚥嚦嚧嚨嚩嚪嚫嚬嚭嚮嚰嚱嚲嚳嚴嚵嚶嚸嚹嚺嚻嚽嚾嚿囀囁囂囃囄囅囆囇囈囉囋囌囍囎囏囐囑囒囓囕囖囘囙囜団囥囦囧囨囩囪囬囮囯囲図囶囷囸囻囼圀圁圂圅圇國圌圍圎圏圐圑�".split("");

      for (j = 0; j != D[135].length; ++j) if (D[135][j].charCodeAt(0) !== 0xFFFD) {
        e[D[135][j]] = 34560 + j;
        d[34560 + j] = D[135][j];
      }

      D[136] = "����������������������������������������������������������������園圓圔圕圖圗團圙圚圛圝圞圠圡圢圤圥圦圧圫圱圲圴圵圶圷圸圼圽圿坁坃坄坅坆坈坉坋坒坓坔坕坖坘坙坢坣坥坧坬坮坰坱坲坴坵坸坹坺坽坾坿垀�垁垇垈垉垊垍垎垏垐垑垔垕垖垗垘垙垚垜垝垞垟垥垨垪垬垯垰垱垳垵垶垷垹垺垻垼垽垾垿埀埁埄埅埆埇埈埉埊埌埍埐埑埓埖埗埛埜埞埡埢埣埥埦埧埨埩埪埫埬埮埰埱埲埳埵埶執埻埼埾埿堁堃堄堅堈堉堊堌堎堏堐堒堓堔堖堗堘堚堛堜堝堟堢堣堥堦堧堨堩堫堬堭堮堯報堲堳場堶堷堸堹堺堻堼堽�".split("");

      for (j = 0; j != D[136].length; ++j) if (D[136][j].charCodeAt(0) !== 0xFFFD) {
        e[D[136][j]] = 34816 + j;
        d[34816 + j] = D[136][j];
      }

      D[137] = "����������������������������������������������������������������堾堿塀塁塂塃塅塆塇塈塉塊塋塎塏塐塒塓塕塖塗塙塚塛塜塝塟塠塡塢塣塤塦塧塨塩塪塭塮塯塰塱塲塳塴塵塶塷塸塹塺塻塼塽塿墂墄墆墇墈墊墋墌�墍墎墏墐墑墔墕墖増墘墛墜墝墠墡墢墣墤墥墦墧墪墫墬墭墮墯墰墱墲墳墴墵墶墷墸墹墺墻墽墾墿壀壂壃壄壆壇壈壉壊壋壌壍壎壏壐壒壓壔壖壗壘壙壚壛壜壝壞壟壠壡壢壣壥壦壧壨壩壪壭壯壱売壴壵壷壸壺壻壼壽壾壿夀夁夃夅夆夈変夊夋夌夎夐夑夒夓夗夘夛夝夞夠夡夢夣夦夨夬夰夲夳夵夶夻�".split("");

      for (j = 0; j != D[137].length; ++j) if (D[137][j].charCodeAt(0) !== 0xFFFD) {
        e[D[137][j]] = 35072 + j;
        d[35072 + j] = D[137][j];
      }

      D[138] = "����������������������������������������������������������������夽夾夿奀奃奅奆奊奌奍奐奒奓奙奛奜奝奞奟奡奣奤奦奧奨奩奪奫奬奭奮奯奰奱奲奵奷奺奻奼奾奿妀妅妉妋妌妎妏妐妑妔妕妘妚妛妜妝妟妠妡妢妦�妧妬妭妰妱妳妴妵妶妷妸妺妼妽妿姀姁姂姃姄姅姇姈姉姌姍姎姏姕姖姙姛姞姟姠姡姢姤姦姧姩姪姫姭姮姯姰姱姲姳姴姵姶姷姸姺姼姽姾娀娂娊娋娍娎娏娐娒娔娕娖娗娙娚娛娝娞娡娢娤娦娧娨娪娫娬娭娮娯娰娳娵娷娸娹娺娻娽娾娿婁婂婃婄婅婇婈婋婌婍婎婏婐婑婒婓婔婖婗婘婙婛婜婝婞婟婠�".split("");

      for (j = 0; j != D[138].length; ++j) if (D[138][j].charCodeAt(0) !== 0xFFFD) {
        e[D[138][j]] = 35328 + j;
        d[35328 + j] = D[138][j];
      }

      D[139] = "����������������������������������������������������������������婡婣婤婥婦婨婩婫婬婭婮婯婰婱婲婳婸婹婻婼婽婾媀媁媂媃媄媅媆媇媈媉媊媋媌媍媎媏媐媑媓媔媕媖媗媘媙媜媝媞媟媠媡媢媣媤媥媦媧媨媩媫媬�媭媮媯媰媱媴媶媷媹媺媻媼媽媿嫀嫃嫄嫅嫆嫇嫈嫊嫋嫍嫎嫏嫐嫑嫓嫕嫗嫙嫚嫛嫝嫞嫟嫢嫤嫥嫧嫨嫪嫬嫭嫮嫯嫰嫲嫳嫴嫵嫶嫷嫸嫹嫺嫻嫼嫽嫾嫿嬀嬁嬂嬃嬄嬅嬆嬇嬈嬊嬋嬌嬍嬎嬏嬐嬑嬒嬓嬔嬕嬘嬙嬚嬛嬜嬝嬞嬟嬠嬡嬢嬣嬤嬥嬦嬧嬨嬩嬪嬫嬬嬭嬮嬯嬰嬱嬳嬵嬶嬸嬹嬺嬻嬼嬽嬾嬿孁孂孃孄孅孆孇�".split("");

      for (j = 0; j != D[139].length; ++j) if (D[139][j].charCodeAt(0) !== 0xFFFD) {
        e[D[139][j]] = 35584 + j;
        d[35584 + j] = D[139][j];
      }

      D[140] = "����������������������������������������������������������������孈孉孊孋孌孍孎孏孒孖孞孠孡孧孨孫孭孮孯孲孴孶孷學孹孻孼孾孿宂宆宊宍宎宐宑宒宔宖実宧宨宩宬宭宮宯宱宲宷宺宻宼寀寁寃寈寉寊寋寍寎寏�寑寔寕寖寗寘寙寚寛寜寠寢寣實寧審寪寫寬寭寯寱寲寳寴寵寶寷寽対尀専尃尅將專尋尌對導尐尒尓尗尙尛尞尟尠尡尣尦尨尩尪尫尭尮尯尰尲尳尵尶尷屃屄屆屇屌屍屒屓屔屖屗屘屚屛屜屝屟屢層屧屨屩屪屫屬屭屰屲屳屴屵屶屷屸屻屼屽屾岀岃岄岅岆岇岉岊岋岎岏岒岓岕岝岞岟岠岡岤岥岦岧岨�".split("");

      for (j = 0; j != D[140].length; ++j) if (D[140][j].charCodeAt(0) !== 0xFFFD) {
        e[D[140][j]] = 35840 + j;
        d[35840 + j] = D[140][j];
      }

      D[141] = "����������������������������������������������������������������岪岮岯岰岲岴岶岹岺岻岼岾峀峂峃峅峆峇峈峉峊峌峍峎峏峐峑峓峔峕峖峗峘峚峛峜峝峞峟峠峢峣峧峩峫峬峮峯峱峲峳峴峵島峷峸峹峺峼峽峾峿崀�崁崄崅崈崉崊崋崌崍崏崐崑崒崓崕崗崘崙崚崜崝崟崠崡崢崣崥崨崪崫崬崯崰崱崲崳崵崶崷崸崹崺崻崼崿嵀嵁嵂嵃嵄嵅嵆嵈嵉嵍嵎嵏嵐嵑嵒嵓嵔嵕嵖嵗嵙嵚嵜嵞嵟嵠嵡嵢嵣嵤嵥嵦嵧嵨嵪嵭嵮嵰嵱嵲嵳嵵嵶嵷嵸嵹嵺嵻嵼嵽嵾嵿嶀嶁嶃嶄嶅嶆嶇嶈嶉嶊嶋嶌嶍嶎嶏嶐嶑嶒嶓嶔嶕嶖嶗嶘嶚嶛嶜嶞嶟嶠�".split("");

      for (j = 0; j != D[141].length; ++j) if (D[141][j].charCodeAt(0) !== 0xFFFD) {
        e[D[141][j]] = 36096 + j;
        d[36096 + j] = D[141][j];
      }

      D[142] = "����������������������������������������������������������������嶡嶢嶣嶤嶥嶦嶧嶨嶩嶪嶫嶬嶭嶮嶯嶰嶱嶲嶳嶴嶵嶶嶸嶹嶺嶻嶼嶽嶾嶿巀巁巂巃巄巆巇巈巉巊巋巌巎巏巐巑巒巓巔巕巖巗巘巙巚巜巟巠巣巤巪巬巭�巰巵巶巸巹巺巻巼巿帀帄帇帉帊帋帍帎帒帓帗帞帟帠帡帢帣帤帥帨帩帪師帬帯帰帲帳帴帵帶帹帺帾帿幀幁幃幆幇幈幉幊幋幍幎幏幐幑幒幓幖幗幘幙幚幜幝幟幠幣幤幥幦幧幨幩幪幫幬幭幮幯幰幱幵幷幹幾庁庂広庅庈庉庌庍庎庒庘庛庝庡庢庣庤庨庩庪庫庬庮庯庰庱庲庴庺庻庼庽庿廀廁廂廃廄廅�".split("");

      for (j = 0; j != D[142].length; ++j) if (D[142][j].charCodeAt(0) !== 0xFFFD) {
        e[D[142][j]] = 36352 + j;
        d[36352 + j] = D[142][j];
      }

      D[143] = "����������������������������������������������������������������廆廇廈廋廌廍廎廏廐廔廕廗廘廙廚廜廝廞廟廠廡廢廣廤廥廦廧廩廫廬廭廮廯廰廱廲廳廵廸廹廻廼廽弅弆弇弉弌弍弎弐弒弔弖弙弚弜弝弞弡弢弣弤�弨弫弬弮弰弲弳弴張弶強弸弻弽弾弿彁彂彃彄彅彆彇彈彉彊彋彌彍彎彏彑彔彙彚彛彜彞彟彠彣彥彧彨彫彮彯彲彴彵彶彸彺彽彾彿徃徆徍徎徏徑従徔徖徚徛徝從徟徠徢徣徤徥徦徧復徫徬徯徰徱徲徳徴徶徸徹徺徻徾徿忀忁忂忇忈忊忋忎忓忔忕忚忛応忞忟忢忣忥忦忨忩忬忯忰忲忳忴忶忷忹忺忼怇�".split("");

      for (j = 0; j != D[143].length; ++j) if (D[143][j].charCodeAt(0) !== 0xFFFD) {
        e[D[143][j]] = 36608 + j;
        d[36608 + j] = D[143][j];
      }

      D[144] = "����������������������������������������������������������������怈怉怋怌怐怑怓怗怘怚怞怟怢怣怤怬怭怮怰怱怲怳怴怶怷怸怹怺怽怾恀恄恅恆恇恈恉恊恌恎恏恑恓恔恖恗恘恛恜恞恟恠恡恥恦恮恱恲恴恵恷恾悀�悁悂悅悆悇悈悊悋悎悏悐悑悓悕悗悘悙悜悞悡悢悤悥悧悩悪悮悰悳悵悶悷悹悺悽悾悿惀惁惂惃惄惇惈惉惌惍惎惏惐惒惓惔惖惗惙惛惞惡惢惣惤惥惪惱惲惵惷惸惻惼惽惾惿愂愃愄愅愇愊愋愌愐愑愒愓愔愖愗愘愙愛愜愝愞愡愢愥愨愩愪愬愭愮愯愰愱愲愳愴愵愶愷愸愹愺愻愼愽愾慀慁慂慃慄慅慆�".split("");

      for (j = 0; j != D[144].length; ++j) if (D[144][j].charCodeAt(0) !== 0xFFFD) {
        e[D[144][j]] = 36864 + j;
        d[36864 + j] = D[144][j];
      }

      D[145] = "����������������������������������������������������������������慇慉態慍慏慐慒慓慔慖慗慘慙慚慛慜慞慟慠慡慣慤慥慦慩慪慫慬慭慮慯慱慲慳慴慶慸慹慺慻慼慽慾慿憀憁憂憃憄憅憆憇憈憉憊憌憍憏憐憑憒憓憕�憖憗憘憙憚憛憜憞憟憠憡憢憣憤憥憦憪憫憭憮憯憰憱憲憳憴憵憶憸憹憺憻憼憽憿懀懁懃懄懅懆懇應懌懍懎懏懐懓懕懖懗懘懙懚懛懜懝懞懟懠懡懢懣懤懥懧懨懩懪懫懬懭懮懯懰懱懲懳懴懶懷懸懹懺懻懼懽懾戀戁戂戃戄戅戇戉戓戔戙戜戝戞戠戣戦戧戨戩戫戭戯戰戱戲戵戶戸戹戺戻戼扂扄扅扆扊�".split("");

      for (j = 0; j != D[145].length; ++j) if (D[145][j].charCodeAt(0) !== 0xFFFD) {
        e[D[145][j]] = 37120 + j;
        d[37120 + j] = D[145][j];
      }

      D[146] = "����������������������������������������������������������������扏扐払扖扗扙扚扜扝扞扟扠扡扢扤扥扨扱扲扴扵扷扸扺扻扽抁抂抃抅抆抇抈抋抌抍抎抏抐抔抙抜抝択抣抦抧抩抪抭抮抯抰抲抳抴抶抷抸抺抾拀拁�拃拋拏拑拕拝拞拠拡拤拪拫拰拲拵拸拹拺拻挀挃挄挅挆挊挋挌挍挏挐挒挓挔挕挗挘挙挜挦挧挩挬挭挮挰挱挳挴挵挶挷挸挻挼挾挿捀捁捄捇捈捊捑捒捓捔捖捗捘捙捚捛捜捝捠捤捥捦捨捪捫捬捯捰捲捳捴捵捸捹捼捽捾捿掁掃掄掅掆掋掍掑掓掔掕掗掙掚掛掜掝掞掟採掤掦掫掯掱掲掵掶掹掻掽掿揀�".split("");

      for (j = 0; j != D[146].length; ++j) if (D[146][j].charCodeAt(0) !== 0xFFFD) {
        e[D[146][j]] = 37376 + j;
        d[37376 + j] = D[146][j];
      }

      D[147] = "����������������������������������������������������������������揁揂揃揅揇揈揊揋揌揑揓揔揕揗揘揙揚換揜揝揟揢揤揥揦揧揨揫揬揮揯揰揱揳揵揷揹揺揻揼揾搃搄搆搇搈搉搊損搎搑搒搕搖搗搘搙搚搝搟搢搣搤�搥搧搨搩搫搮搯搰搱搲搳搵搶搷搸搹搻搼搾摀摂摃摉摋摌摍摎摏摐摑摓摕摖摗摙摚摛摜摝摟摠摡摢摣摤摥摦摨摪摫摬摮摯摰摱摲摳摴摵摶摷摻摼摽摾摿撀撁撃撆撈撉撊撋撌撍撎撏撐撓撔撗撘撚撛撜撝撟撠撡撢撣撥撦撧撨撪撫撯撱撲撳撴撶撹撻撽撾撿擁擃擄擆擇擈擉擊擋擌擏擑擓擔擕擖擙據�".split("");

      for (j = 0; j != D[147].length; ++j) if (D[147][j].charCodeAt(0) !== 0xFFFD) {
        e[D[147][j]] = 37632 + j;
        d[37632 + j] = D[147][j];
      }

      D[148] = "����������������������������������������������������������������擛擜擝擟擠擡擣擥擧擨擩擪擫擬擭擮擯擰擱擲擳擴擵擶擷擸擹擺擻擼擽擾擿攁攂攃攄攅攆攇攈攊攋攌攍攎攏攐攑攓攔攕攖攗攙攚攛攜攝攞攟攠攡�攢攣攤攦攧攨攩攪攬攭攰攱攲攳攷攺攼攽敀敁敂敃敄敆敇敊敋敍敎敐敒敓敔敗敘敚敜敟敠敡敤敥敧敨敩敪敭敮敯敱敳敵敶數敹敺敻敼敽敾敿斀斁斂斃斄斅斆斈斉斊斍斎斏斒斔斕斖斘斚斝斞斠斢斣斦斨斪斬斮斱斲斳斴斵斶斷斸斺斻斾斿旀旂旇旈旉旊旍旐旑旓旔旕旘旙旚旛旜旝旞旟旡旣旤旪旫�".split("");

      for (j = 0; j != D[148].length; ++j) if (D[148][j].charCodeAt(0) !== 0xFFFD) {
        e[D[148][j]] = 37888 + j;
        d[37888 + j] = D[148][j];
      }

      D[149] = "����������������������������������������������������������������旲旳旴旵旸旹旻旼旽旾旿昁昄昅昇昈昉昋昍昐昑昒昖昗昘昚昛昜昞昡昢昣昤昦昩昪昫昬昮昰昲昳昷昸昹昺昻昽昿晀時晄晅晆晇晈晉晊晍晎晐晑晘�晙晛晜晝晞晠晢晣晥晧晩晪晫晬晭晱晲晳晵晸晹晻晼晽晿暀暁暃暅暆暈暉暊暋暍暎暏暐暒暓暔暕暘暙暚暛暜暞暟暠暡暢暣暤暥暦暩暪暫暬暭暯暰暱暲暳暵暶暷暸暺暻暼暽暿曀曁曂曃曄曅曆曇曈曉曊曋曌曍曎曏曐曑曒曓曔曕曖曗曘曚曞曟曠曡曢曣曤曥曧曨曪曫曬曭曮曯曱曵曶書曺曻曽朁朂會�".split("");

      for (j = 0; j != D[149].length; ++j) if (D[149][j].charCodeAt(0) !== 0xFFFD) {
        e[D[149][j]] = 38144 + j;
        d[38144 + j] = D[149][j];
      }

      D[150] = "����������������������������������������������������������������朄朅朆朇朌朎朏朑朒朓朖朘朙朚朜朞朠朡朢朣朤朥朧朩朮朰朲朳朶朷朸朹朻朼朾朿杁杄杅杇杊杋杍杒杔杕杗杘杙杚杛杝杢杣杤杦杧杫杬杮東杴杶�杸杹杺杻杽枀枂枃枅枆枈枊枌枍枎枏枑枒枓枔枖枙枛枟枠枡枤枦枩枬枮枱枲枴枹枺枻枼枽枾枿柀柂柅柆柇柈柉柊柋柌柍柎柕柖柗柛柟柡柣柤柦柧柨柪柫柭柮柲柵柶柷柸柹柺査柼柾栁栂栃栄栆栍栐栒栔栕栘栙栚栛栜栞栟栠栢栣栤栥栦栧栨栫栬栭栮栯栰栱栴栵栶栺栻栿桇桋桍桏桒桖桗桘桙桚桛�".split("");

      for (j = 0; j != D[150].length; ++j) if (D[150][j].charCodeAt(0) !== 0xFFFD) {
        e[D[150][j]] = 38400 + j;
        d[38400 + j] = D[150][j];
      }

      D[151] = "����������������������������������������������������������������桜桝桞桟桪桬桭桮桯桰桱桲桳桵桸桹桺桻桼桽桾桿梀梂梄梇梈梉梊梋梌梍梎梐梑梒梔梕梖梘梙梚梛梜條梞梟梠梡梣梤梥梩梪梫梬梮梱梲梴梶梷梸�梹梺梻梼梽梾梿棁棃棄棅棆棇棈棊棌棎棏棐棑棓棔棖棗棙棛棜棝棞棟棡棢棤棥棦棧棨棩棪棫棬棭棯棲棳棴棶棷棸棻棽棾棿椀椂椃椄椆椇椈椉椊椌椏椑椓椔椕椖椗椘椙椚椛検椝椞椡椢椣椥椦椧椨椩椪椫椬椮椯椱椲椳椵椶椷椸椺椻椼椾楀楁楃楄楅楆楇楈楉楊楋楌楍楎楏楐楑楒楓楕楖楘楙楛楜楟�".split("");

      for (j = 0; j != D[151].length; ++j) if (D[151][j].charCodeAt(0) !== 0xFFFD) {
        e[D[151][j]] = 38656 + j;
        d[38656 + j] = D[151][j];
      }

      D[152] = "����������������������������������������������������������������楡楢楤楥楧楨楩楪楬業楯楰楲楳楴極楶楺楻楽楾楿榁榃榅榊榋榌榎榏榐榑榒榓榖榗榙榚榝榞榟榠榡榢榣榤榥榦榩榪榬榮榯榰榲榳榵榶榸榹榺榼榽�榾榿槀槂槃槄槅槆槇槈槉構槍槏槑槒槓槕槖槗様槙槚槜槝槞槡槢槣槤槥槦槧槨槩槪槫槬槮槯槰槱槳槴槵槶槷槸槹槺槻槼槾樀樁樂樃樄樅樆樇樈樉樋樌樍樎樏樐樑樒樓樔樕樖標樚樛樜樝樞樠樢樣樤樥樦樧権樫樬樭樮樰樲樳樴樶樷樸樹樺樻樼樿橀橁橂橃橅橆橈橉橊橋橌橍橎橏橑橒橓橔橕橖橗橚�".split("");

      for (j = 0; j != D[152].length; ++j) if (D[152][j].charCodeAt(0) !== 0xFFFD) {
        e[D[152][j]] = 38912 + j;
        d[38912 + j] = D[152][j];
      }

      D[153] = "����������������������������������������������������������������橜橝橞機橠橢橣橤橦橧橨橩橪橫橬橭橮橯橰橲橳橴橵橶橷橸橺橻橽橾橿檁檂檃檅檆檇檈檉檊檋檌檍檏檒檓檔檕檖檘檙檚檛檜檝檞檟檡檢檣檤檥檦�檧檨檪檭檮檯檰檱檲檳檴檵檶檷檸檹檺檻檼檽檾檿櫀櫁櫂櫃櫄櫅櫆櫇櫈櫉櫊櫋櫌櫍櫎櫏櫐櫑櫒櫓櫔櫕櫖櫗櫘櫙櫚櫛櫜櫝櫞櫟櫠櫡櫢櫣櫤櫥櫦櫧櫨櫩櫪櫫櫬櫭櫮櫯櫰櫱櫲櫳櫴櫵櫶櫷櫸櫹櫺櫻櫼櫽櫾櫿欀欁欂欃欄欅欆欇欈欉權欋欌欍欎欏欐欑欒欓欔欕欖欗欘欙欚欛欜欝欞欟欥欦欨欩欪欫欬欭欮�".split("");

      for (j = 0; j != D[153].length; ++j) if (D[153][j].charCodeAt(0) !== 0xFFFD) {
        e[D[153][j]] = 39168 + j;
        d[39168 + j] = D[153][j];
      }

      D[154] = "����������������������������������������������������������������欯欰欱欳欴欵欶欸欻欼欽欿歀歁歂歄歅歈歊歋歍歎歏歐歑歒歓歔歕歖歗歘歚歛歜歝歞歟歠歡歨歩歫歬歭歮歯歰歱歲歳歴歵歶歷歸歺歽歾歿殀殅殈�殌殎殏殐殑殔殕殗殘殙殜殝殞殟殠殢殣殤殥殦殧殨殩殫殬殭殮殯殰殱殲殶殸殹殺殻殼殽殾毀毃毄毆毇毈毉毊毌毎毐毑毘毚毜毝毞毟毠毢毣毤毥毦毧毨毩毬毭毮毰毱毲毴毶毷毸毺毻毼毾毿氀氁氂氃氄氈氉氊氋氌氎氒気氜氝氞氠氣氥氫氬氭氱氳氶氷氹氺氻氼氾氿汃汄汅汈汋汌汍汎汏汑汒汓汖汘�".split("");

      for (j = 0; j != D[154].length; ++j) if (D[154][j].charCodeAt(0) !== 0xFFFD) {
        e[D[154][j]] = 39424 + j;
        d[39424 + j] = D[154][j];
      }

      D[155] = "����������������������������������������������������������������汙汚汢汣汥汦汧汫汬汭汮汯汱汳汵汷汸決汻汼汿沀沄沇沊沋沍沎沑沒沕沖沗沘沚沜沝沞沠沢沨沬沯沰沴沵沶沷沺泀況泂泃泆泇泈泋泍泎泏泑泒泘�泙泚泜泝泟泤泦泧泩泬泭泲泴泹泿洀洂洃洅洆洈洉洊洍洏洐洑洓洔洕洖洘洜洝洟洠洡洢洣洤洦洨洩洬洭洯洰洴洶洷洸洺洿浀浂浄浉浌浐浕浖浗浘浛浝浟浡浢浤浥浧浨浫浬浭浰浱浲浳浵浶浹浺浻浽浾浿涀涁涃涄涆涇涊涋涍涏涐涒涖涗涘涙涚涜涢涥涬涭涰涱涳涴涶涷涹涺涻涼涽涾淁淂淃淈淉淊�".split("");

      for (j = 0; j != D[155].length; ++j) if (D[155][j].charCodeAt(0) !== 0xFFFD) {
        e[D[155][j]] = 39680 + j;
        d[39680 + j] = D[155][j];
      }

      D[156] = "����������������������������������������������������������������淍淎淏淐淒淓淔淕淗淚淛淜淟淢淣淥淧淨淩淪淭淯淰淲淴淵淶淸淺淽淾淿渀渁渂渃渄渆渇済渉渋渏渒渓渕渘渙減渜渞渟渢渦渧渨渪測渮渰渱渳渵�渶渷渹渻渼渽渾渿湀湁湂湅湆湇湈湉湊湋湌湏湐湑湒湕湗湙湚湜湝湞湠湡湢湣湤湥湦湧湨湩湪湬湭湯湰湱湲湳湴湵湶湷湸湹湺湻湼湽満溁溂溄溇溈溊溋溌溍溎溑溒溓溔溕準溗溙溚溛溝溞溠溡溣溤溦溨溩溫溬溭溮溰溳溵溸溹溼溾溿滀滃滄滅滆滈滉滊滌滍滎滐滒滖滘滙滛滜滝滣滧滪滫滬滭滮滯�".split("");

      for (j = 0; j != D[156].length; ++j) if (D[156][j].charCodeAt(0) !== 0xFFFD) {
        e[D[156][j]] = 39936 + j;
        d[39936 + j] = D[156][j];
      }

      D[157] = "����������������������������������������������������������������滰滱滲滳滵滶滷滸滺滻滼滽滾滿漀漁漃漄漅漇漈漊漋漌漍漎漐漑漒漖漗漘漙漚漛漜漝漞漟漡漢漣漥漦漧漨漬漮漰漲漴漵漷漸漹漺漻漼漽漿潀潁潂�潃潄潅潈潉潊潌潎潏潐潑潒潓潔潕潖潗潙潚潛潝潟潠潡潣潤潥潧潨潩潪潫潬潯潰潱潳潵潶潷潹潻潽潾潿澀澁澂澃澅澆澇澊澋澏澐澑澒澓澔澕澖澗澘澙澚澛澝澞澟澠澢澣澤澥澦澨澩澪澫澬澭澮澯澰澱澲澴澵澷澸澺澻澼澽澾澿濁濃濄濅濆濇濈濊濋濌濍濎濏濐濓濔濕濖濗濘濙濚濛濜濝濟濢濣濤濥�".split("");

      for (j = 0; j != D[157].length; ++j) if (D[157][j].charCodeAt(0) !== 0xFFFD) {
        e[D[157][j]] = 40192 + j;
        d[40192 + j] = D[157][j];
      }

      D[158] = "����������������������������������������������������������������濦濧濨濩濪濫濬濭濰濱濲濳濴濵濶濷濸濹濺濻濼濽濾濿瀀瀁瀂瀃瀄瀅瀆瀇瀈瀉瀊瀋瀌瀍瀎瀏瀐瀒瀓瀔瀕瀖瀗瀘瀙瀜瀝瀞瀟瀠瀡瀢瀤瀥瀦瀧瀨瀩瀪�瀫瀬瀭瀮瀯瀰瀱瀲瀳瀴瀶瀷瀸瀺瀻瀼瀽瀾瀿灀灁灂灃灄灅灆灇灈灉灊灋灍灎灐灑灒灓灔灕灖灗灘灙灚灛灜灝灟灠灡灢灣灤灥灦灧灨灩灪灮灱灲灳灴灷灹灺灻災炁炂炃炄炆炇炈炋炌炍炏炐炑炓炗炘炚炛炞炟炠炡炢炣炤炥炦炧炨炩炪炰炲炴炵炶為炾炿烄烅烆烇烉烋烌烍烎烏烐烑烒烓烔烕烖烗烚�".split("");

      for (j = 0; j != D[158].length; ++j) if (D[158][j].charCodeAt(0) !== 0xFFFD) {
        e[D[158][j]] = 40448 + j;
        d[40448 + j] = D[158][j];
      }

      D[159] = "����������������������������������������������������������������烜烝烞烠烡烢烣烥烪烮烰烱烲烳烴烵烶烸烺烻烼烾烿焀焁焂焃焄焅焆焇焈焋焌焍焎焏焑焒焔焗焛焜焝焞焟焠無焢焣焤焥焧焨焩焪焫焬焭焮焲焳焴�焵焷焸焹焺焻焼焽焾焿煀煁煂煃煄煆煇煈煉煋煍煏煐煑煒煓煔煕煖煗煘煙煚煛煝煟煠煡煢煣煥煩煪煫煬煭煯煰煱煴煵煶煷煹煻煼煾煿熀熁熂熃熅熆熇熈熉熋熌熍熎熐熑熒熓熕熖熗熚熛熜熝熞熡熢熣熤熥熦熧熩熪熫熭熮熯熰熱熲熴熶熷熸熺熻熼熽熾熿燀燁燂燄燅燆燇燈燉燊燋燌燍燏燐燑燒燓�".split("");

      for (j = 0; j != D[159].length; ++j) if (D[159][j].charCodeAt(0) !== 0xFFFD) {
        e[D[159][j]] = 40704 + j;
        d[40704 + j] = D[159][j];
      }

      D[160] = "����������������������������������������������������������������燖燗燘燙燚燛燜燝燞營燡燢燣燤燦燨燩燪燫燬燭燯燰燱燲燳燴燵燶燷燸燺燻燼燽燾燿爀爁爂爃爄爅爇爈爉爊爋爌爍爎爏爐爑爒爓爔爕爖爗爘爙爚�爛爜爞爟爠爡爢爣爤爥爦爧爩爫爭爮爯爲爳爴爺爼爾牀牁牂牃牄牅牆牉牊牋牎牏牐牑牓牔牕牗牘牚牜牞牠牣牤牥牨牪牫牬牭牰牱牳牴牶牷牸牻牼牽犂犃犅犆犇犈犉犌犎犐犑犓犔犕犖犗犘犙犚犛犜犝犞犠犡犢犣犤犥犦犧犨犩犪犫犮犱犲犳犵犺犻犼犽犾犿狀狅狆狇狉狊狋狌狏狑狓狔狕狖狘狚狛�".split("");

      for (j = 0; j != D[160].length; ++j) if (D[160][j].charCodeAt(0) !== 0xFFFD) {
        e[D[160][j]] = 40960 + j;
        d[40960 + j] = D[160][j];
      }

      D[161] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������　、。·ˉˇ¨〃々—～‖…‘’“”〔〕〈〉《》「」『』〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓�".split("");

      for (j = 0; j != D[161].length; ++j) if (D[161][j].charCodeAt(0) !== 0xFFFD) {
        e[D[161][j]] = 41216 + j;
        d[41216 + j] = D[161][j];
      }

      D[162] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ⅰⅱⅲⅳⅴⅵⅶⅷⅸⅹ������⒈⒉⒊⒋⒌⒍⒎⒏⒐⒑⒒⒓⒔⒕⒖⒗⒘⒙⒚⒛⑴⑵⑶⑷⑸⑹⑺⑻⑼⑽⑾⑿⒀⒁⒂⒃⒄⒅⒆⒇①②③④⑤⑥⑦⑧⑨⑩��㈠㈡㈢㈣㈤㈥㈦㈧㈨㈩��ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩⅪⅫ���".split("");

      for (j = 0; j != D[162].length; ++j) if (D[162][j].charCodeAt(0) !== 0xFFFD) {
        e[D[162][j]] = 41472 + j;
        d[41472 + j] = D[162][j];
      }

      D[163] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������！＂＃￥％＆＇（）＊＋，－．／０１２３４５６７８９：；＜＝＞？＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ［＼］＾＿｀ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ｛｜｝￣�".split("");

      for (j = 0; j != D[163].length; ++j) if (D[163][j].charCodeAt(0) !== 0xFFFD) {
        e[D[163][j]] = 41728 + j;
        d[41728 + j] = D[163][j];
      }

      D[164] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ぁあぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとどなにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをん������������".split("");

      for (j = 0; j != D[164].length; ++j) if (D[164][j].charCodeAt(0) !== 0xFFFD) {
        e[D[164][j]] = 41984 + j;
        d[41984 + j] = D[164][j];
      }

      D[165] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ァアィイゥウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトドナニヌネノハバパヒビピフブプヘベペホボポマミムメモャヤュユョヨラリルレロヮワヰヱヲンヴヵヶ���������".split("");

      for (j = 0; j != D[165].length; ++j) if (D[165][j].charCodeAt(0) !== 0xFFFD) {
        e[D[165][j]] = 42240 + j;
        d[42240 + j] = D[165][j];
      }

      D[166] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ��������αβγδεζηθικλμνξοπρστυφχψω�������︵︶︹︺︿﹀︽︾﹁﹂﹃﹄��︻︼︷︸︱�︳︴����������".split("");

      for (j = 0; j != D[166].length; ++j) if (D[166][j].charCodeAt(0) !== 0xFFFD) {
        e[D[166][j]] = 42496 + j;
        d[42496 + j] = D[166][j];
      }

      D[167] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ���������������абвгдеёжзийклмнопрстуфхцчшщъыьэюя��������������".split("");

      for (j = 0; j != D[167].length; ++j) if (D[167][j].charCodeAt(0) !== 0xFFFD) {
        e[D[167][j]] = 42752 + j;
        d[42752 + j] = D[167][j];
      }

      D[168] = "����������������������������������������������������������������ˊˋ˙–―‥‵℅℉↖↗↘↙∕∟∣≒≦≧⊿═║╒╓╔╕╖╗╘╙╚╛╜╝╞╟╠╡╢╣╤╥╦╧╨╩╪╫╬╭╮╯╰╱╲╳▁▂▃▄▅▆▇�█▉▊▋▌▍▎▏▓▔▕▼▽◢◣◤◥☉⊕〒〝〞�����������āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüêɑ�ńň�ɡ����ㄅㄆㄇㄈㄉㄊㄋㄌㄍㄎㄏㄐㄑㄒㄓㄔㄕㄖㄗㄘㄙㄚㄛㄜㄝㄞㄟㄠㄡㄢㄣㄤㄥㄦㄧㄨㄩ����������������������".split("");

      for (j = 0; j != D[168].length; ++j) if (D[168][j].charCodeAt(0) !== 0xFFFD) {
        e[D[168][j]] = 43008 + j;
        d[43008 + j] = D[168][j];
      }

      D[169] = "����������������������������������������������������������������〡〢〣〤〥〦〧〨〩㊣㎎㎏㎜㎝㎞㎡㏄㏎㏑㏒㏕︰￢￤�℡㈱�‐���ー゛゜ヽヾ〆ゝゞ﹉﹊﹋﹌﹍﹎﹏﹐﹑﹒﹔﹕﹖﹗﹙﹚﹛﹜﹝﹞﹟﹠﹡�﹢﹣﹤﹥﹦﹨﹩﹪﹫�������������〇�������������─━│┃┄┅┆┇┈┉┊┋┌┍┎┏┐┑┒┓└┕┖┗┘┙┚┛├┝┞┟┠┡┢┣┤┥┦┧┨┩┪┫┬┭┮┯┰┱┲┳┴┵┶┷┸┹┺┻┼┽┾┿╀╁╂╃╄╅╆╇╈╉╊╋����������������".split("");

      for (j = 0; j != D[169].length; ++j) if (D[169][j].charCodeAt(0) !== 0xFFFD) {
        e[D[169][j]] = 43264 + j;
        d[43264 + j] = D[169][j];
      }

      D[170] = "����������������������������������������������������������������狜狝狟狢狣狤狥狦狧狪狫狵狶狹狽狾狿猀猂猄猅猆猇猈猉猋猌猍猏猐猑猒猔猘猙猚猟猠猣猤猦猧猨猭猯猰猲猳猵猶猺猻猼猽獀獁獂獃獄獅獆獇獈�獉獊獋獌獎獏獑獓獔獕獖獘獙獚獛獜獝獞獟獡獢獣獤獥獦獧獨獩獪獫獮獰獱�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[170].length; ++j) if (D[170][j].charCodeAt(0) !== 0xFFFD) {
        e[D[170][j]] = 43520 + j;
        d[43520 + j] = D[170][j];
      }

      D[171] = "����������������������������������������������������������������獲獳獴獵獶獷獸獹獺獻獼獽獿玀玁玂玃玅玆玈玊玌玍玏玐玒玓玔玕玗玘玙玚玜玝玞玠玡玣玤玥玦玧玨玪玬玭玱玴玵玶玸玹玼玽玾玿珁珃珄珅珆珇�珋珌珎珒珓珔珕珖珗珘珚珛珜珝珟珡珢珣珤珦珨珪珫珬珮珯珰珱珳珴珵珶珷�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[171].length; ++j) if (D[171][j].charCodeAt(0) !== 0xFFFD) {
        e[D[171][j]] = 43776 + j;
        d[43776 + j] = D[171][j];
      }

      D[172] = "����������������������������������������������������������������珸珹珺珻珼珽現珿琀琁琂琄琇琈琋琌琍琎琑琒琓琔琕琖琗琘琙琜琝琞琟琠琡琣琤琧琩琫琭琯琱琲琷琸琹琺琻琽琾琿瑀瑂瑃瑄瑅瑆瑇瑈瑉瑊瑋瑌瑍�瑎瑏瑐瑑瑒瑓瑔瑖瑘瑝瑠瑡瑢瑣瑤瑥瑦瑧瑨瑩瑪瑫瑬瑮瑯瑱瑲瑳瑴瑵瑸瑹瑺�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[172].length; ++j) if (D[172][j].charCodeAt(0) !== 0xFFFD) {
        e[D[172][j]] = 44032 + j;
        d[44032 + j] = D[172][j];
      }

      D[173] = "����������������������������������������������������������������瑻瑼瑽瑿璂璄璅璆璈璉璊璌璍璏璑璒璓璔璕璖璗璘璙璚璛璝璟璠璡璢璣璤璥璦璪璫璬璭璮璯環璱璲璳璴璵璶璷璸璹璻璼璽璾璿瓀瓁瓂瓃瓄瓅瓆瓇�瓈瓉瓊瓋瓌瓍瓎瓏瓐瓑瓓瓔瓕瓖瓗瓘瓙瓚瓛瓝瓟瓡瓥瓧瓨瓩瓪瓫瓬瓭瓰瓱瓲�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[173].length; ++j) if (D[173][j].charCodeAt(0) !== 0xFFFD) {
        e[D[173][j]] = 44288 + j;
        d[44288 + j] = D[173][j];
      }

      D[174] = "����������������������������������������������������������������瓳瓵瓸瓹瓺瓻瓼瓽瓾甀甁甂甃甅甆甇甈甉甊甋甌甎甐甒甔甕甖甗甛甝甞甠甡產産甤甦甧甪甮甴甶甹甼甽甿畁畂畃畄畆畇畉畊畍畐畑畒畓畕畖畗畘�畝畞畟畠畡畢畣畤畧畨畩畫畬畭畮畯異畱畳畵當畷畺畻畼畽畾疀疁疂疄疅疇�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[174].length; ++j) if (D[174][j].charCodeAt(0) !== 0xFFFD) {
        e[D[174][j]] = 44544 + j;
        d[44544 + j] = D[174][j];
      }

      D[175] = "����������������������������������������������������������������疈疉疊疌疍疎疐疓疕疘疛疜疞疢疦疧疨疩疪疭疶疷疺疻疿痀痁痆痋痌痎痏痐痑痓痗痙痚痜痝痟痠痡痥痩痬痭痮痯痲痳痵痶痷痸痺痻痽痾瘂瘄瘆瘇�瘈瘉瘋瘍瘎瘏瘑瘒瘓瘔瘖瘚瘜瘝瘞瘡瘣瘧瘨瘬瘮瘯瘱瘲瘶瘷瘹瘺瘻瘽癁療癄�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[175].length; ++j) if (D[175][j].charCodeAt(0) !== 0xFFFD) {
        e[D[175][j]] = 44800 + j;
        d[44800 + j] = D[175][j];
      }

      D[176] = "����������������������������������������������������������������癅癆癇癈癉癊癋癎癏癐癑癒癓癕癗癘癙癚癛癝癟癠癡癢癤癥癦癧癨癩癪癬癭癮癰癱癲癳癴癵癶癷癹発發癿皀皁皃皅皉皊皌皍皏皐皒皔皕皗皘皚皛�皜皝皞皟皠皡皢皣皥皦皧皨皩皪皫皬皭皯皰皳皵皶皷皸皹皺皻皼皽皾盀盁盃啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥�".split("");

      for (j = 0; j != D[176].length; ++j) if (D[176][j].charCodeAt(0) !== 0xFFFD) {
        e[D[176][j]] = 45056 + j;
        d[45056 + j] = D[176][j];
      }

      D[177] = "����������������������������������������������������������������盄盇盉盋盌盓盕盙盚盜盝盞盠盡盢監盤盦盧盨盩盪盫盬盭盰盳盵盶盷盺盻盽盿眀眂眃眅眆眊県眎眏眐眑眒眓眔眕眖眗眘眛眜眝眞眡眣眤眥眧眪眫�眬眮眰眱眲眳眴眹眻眽眾眿睂睄睅睆睈睉睊睋睌睍睎睏睒睓睔睕睖睗睘睙睜薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳�".split("");

      for (j = 0; j != D[177].length; ++j) if (D[177][j].charCodeAt(0) !== 0xFFFD) {
        e[D[177][j]] = 45312 + j;
        d[45312 + j] = D[177][j];
      }

      D[178] = "����������������������������������������������������������������睝睞睟睠睤睧睩睪睭睮睯睰睱睲睳睴睵睶睷睸睺睻睼瞁瞂瞃瞆瞇瞈瞉瞊瞋瞏瞐瞓瞔瞕瞖瞗瞘瞙瞚瞛瞜瞝瞞瞡瞣瞤瞦瞨瞫瞭瞮瞯瞱瞲瞴瞶瞷瞸瞹瞺�瞼瞾矀矁矂矃矄矅矆矇矈矉矊矋矌矎矏矐矑矒矓矔矕矖矘矙矚矝矞矟矠矡矤病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖�".split("");

      for (j = 0; j != D[178].length; ++j) if (D[178][j].charCodeAt(0) !== 0xFFFD) {
        e[D[178][j]] = 45568 + j;
        d[45568 + j] = D[178][j];
      }

      D[179] = "����������������������������������������������������������������矦矨矪矯矰矱矲矴矵矷矹矺矻矼砃砄砅砆砇砈砊砋砎砏砐砓砕砙砛砞砠砡砢砤砨砪砫砮砯砱砲砳砵砶砽砿硁硂硃硄硆硈硉硊硋硍硏硑硓硔硘硙硚�硛硜硞硟硠硡硢硣硤硥硦硧硨硩硯硰硱硲硳硴硵硶硸硹硺硻硽硾硿碀碁碂碃场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚�".split("");

      for (j = 0; j != D[179].length; ++j) if (D[179][j].charCodeAt(0) !== 0xFFFD) {
        e[D[179][j]] = 45824 + j;
        d[45824 + j] = D[179][j];
      }

      D[180] = "����������������������������������������������������������������碄碅碆碈碊碋碏碐碒碔碕碖碙碝碞碠碢碤碦碨碩碪碫碬碭碮碯碵碶碷碸確碻碼碽碿磀磂磃磄磆磇磈磌磍磎磏磑磒磓磖磗磘磚磛磜磝磞磟磠磡磢磣�磤磥磦磧磩磪磫磭磮磯磰磱磳磵磶磸磹磻磼磽磾磿礀礂礃礄礆礇礈礉礊礋礌础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮�".split("");

      for (j = 0; j != D[180].length; ++j) if (D[180][j].charCodeAt(0) !== 0xFFFD) {
        e[D[180][j]] = 46080 + j;
        d[46080 + j] = D[180][j];
      }

      D[181] = "����������������������������������������������������������������礍礎礏礐礑礒礔礕礖礗礘礙礚礛礜礝礟礠礡礢礣礥礦礧礨礩礪礫礬礭礮礯礰礱礲礳礵礶礷礸礹礽礿祂祃祄祅祇祊祋祌祍祎祏祐祑祒祔祕祘祙祡祣�祤祦祩祪祫祬祮祰祱祲祳祴祵祶祹祻祼祽祾祿禂禃禆禇禈禉禋禌禍禎禐禑禒怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠�".split("");

      for (j = 0; j != D[181].length; ++j) if (D[181][j].charCodeAt(0) !== 0xFFFD) {
        e[D[181][j]] = 46336 + j;
        d[46336 + j] = D[181][j];
      }

      D[182] = "����������������������������������������������������������������禓禔禕禖禗禘禙禛禜禝禞禟禠禡禢禣禤禥禦禨禩禪禫禬禭禮禯禰禱禲禴禵禶禷禸禼禿秂秄秅秇秈秊秌秎秏秐秓秔秖秗秙秚秛秜秝秞秠秡秢秥秨秪�秬秮秱秲秳秴秵秶秷秹秺秼秾秿稁稄稅稇稈稉稊稌稏稐稑稒稓稕稖稘稙稛稜丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二�".split("");

      for (j = 0; j != D[182].length; ++j) if (D[182][j].charCodeAt(0) !== 0xFFFD) {
        e[D[182][j]] = 46592 + j;
        d[46592 + j] = D[182][j];
      }

      D[183] = "����������������������������������������������������������������稝稟稡稢稤稥稦稧稨稩稪稫稬稭種稯稰稱稲稴稵稶稸稺稾穀穁穂穃穄穅穇穈穉穊穋穌積穎穏穐穒穓穔穕穖穘穙穚穛穜穝穞穟穠穡穢穣穤穥穦穧穨�穩穪穫穬穭穮穯穱穲穳穵穻穼穽穾窂窅窇窉窊窋窌窎窏窐窓窔窙窚窛窞窡窢贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服�".split("");

      for (j = 0; j != D[183].length; ++j) if (D[183][j].charCodeAt(0) !== 0xFFFD) {
        e[D[183][j]] = 46848 + j;
        d[46848 + j] = D[183][j];
      }

      D[184] = "����������������������������������������������������������������窣窤窧窩窪窫窮窯窰窱窲窴窵窶窷窸窹窺窻窼窽窾竀竁竂竃竄竅竆竇竈竉竊竌竍竎竏竐竑竒竓竔竕竗竘竚竛竜竝竡竢竤竧竨竩竪竫竬竮竰竱竲竳�竴竵競竷竸竻竼竾笀笁笂笅笇笉笌笍笎笐笒笓笖笗笘笚笜笝笟笡笢笣笧笩笭浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹�".split("");

      for (j = 0; j != D[184].length; ++j) if (D[184][j].charCodeAt(0) !== 0xFFFD) {
        e[D[184][j]] = 47104 + j;
        d[47104 + j] = D[184][j];
      }

      D[185] = "����������������������������������������������������������������笯笰笲笴笵笶笷笹笻笽笿筀筁筂筃筄筆筈筊筍筎筓筕筗筙筜筞筟筡筣筤筥筦筧筨筩筪筫筬筭筯筰筳筴筶筸筺筼筽筿箁箂箃箄箆箇箈箉箊箋箌箎箏�箑箒箓箖箘箙箚箛箞箟箠箣箤箥箮箯箰箲箳箵箶箷箹箺箻箼箽箾箿節篂篃範埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈�".split("");

      for (j = 0; j != D[185].length; ++j) if (D[185][j].charCodeAt(0) !== 0xFFFD) {
        e[D[185][j]] = 47360 + j;
        d[47360 + j] = D[185][j];
      }

      D[186] = "����������������������������������������������������������������篅篈築篊篋篍篎篏篐篒篔篕篖篗篘篛篜篞篟篠篢篣篤篧篨篩篫篬篭篯篰篲篳篴篵篶篸篹篺篻篽篿簀簁簂簃簄簅簆簈簉簊簍簎簐簑簒簓簔簕簗簘簙�簚簛簜簝簞簠簡簢簣簤簥簨簩簫簬簭簮簯簰簱簲簳簴簵簶簷簹簺簻簼簽簾籂骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖�".split("");

      for (j = 0; j != D[186].length; ++j) if (D[186][j].charCodeAt(0) !== 0xFFFD) {
        e[D[186][j]] = 47616 + j;
        d[47616 + j] = D[186][j];
      }

      D[187] = "����������������������������������������������������������������籃籄籅籆籇籈籉籊籋籌籎籏籐籑籒籓籔籕籖籗籘籙籚籛籜籝籞籟籠籡籢籣籤籥籦籧籨籩籪籫籬籭籮籯籰籱籲籵籶籷籸籹籺籾籿粀粁粂粃粄粅粆粇�粈粊粋粌粍粎粏粐粓粔粖粙粚粛粠粡粣粦粧粨粩粫粬粭粯粰粴粵粶粷粸粺粻弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕�".split("");

      for (j = 0; j != D[187].length; ++j) if (D[187][j].charCodeAt(0) !== 0xFFFD) {
        e[D[187][j]] = 47872 + j;
        d[47872 + j] = D[187][j];
      }

      D[188] = "����������������������������������������������������������������粿糀糂糃糄糆糉糋糎糏糐糑糒糓糔糘糚糛糝糞糡糢糣糤糥糦糧糩糪糫糬糭糮糰糱糲糳糴糵糶糷糹糺糼糽糾糿紀紁紂紃約紅紆紇紈紉紋紌納紎紏紐�紑紒紓純紕紖紗紘紙級紛紜紝紞紟紡紣紤紥紦紨紩紪紬紭紮細紱紲紳紴紵紶肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件�".split("");

      for (j = 0; j != D[188].length; ++j) if (D[188][j].charCodeAt(0) !== 0xFFFD) {
        e[D[188][j]] = 48128 + j;
        d[48128 + j] = D[188][j];
      }

      D[189] = "����������������������������������������������������������������紷紸紹紺紻紼紽紾紿絀絁終絃組絅絆絇絈絉絊絋経絍絎絏結絑絒絓絔絕絖絗絘絙絚絛絜絝絞絟絠絡絢絣絤絥給絧絨絩絪絫絬絭絯絰統絲絳絴絵絶�絸絹絺絻絼絽絾絿綀綁綂綃綄綅綆綇綈綉綊綋綌綍綎綏綐綑綒經綔綕綖綗綘健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸�".split("");

      for (j = 0; j != D[189].length; ++j) if (D[189][j].charCodeAt(0) !== 0xFFFD) {
        e[D[189][j]] = 48384 + j;
        d[48384 + j] = D[189][j];
      }

      D[190] = "����������������������������������������������������������������継続綛綜綝綞綟綠綡綢綣綤綥綧綨綩綪綫綬維綯綰綱網綳綴綵綶綷綸綹綺綻綼綽綾綿緀緁緂緃緄緅緆緇緈緉緊緋緌緍緎総緐緑緒緓緔緕緖緗緘緙�線緛緜緝緞緟締緡緢緣緤緥緦緧編緩緪緫緬緭緮緯緰緱緲緳練緵緶緷緸緹緺尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻�".split("");

      for (j = 0; j != D[190].length; ++j) if (D[190][j].charCodeAt(0) !== 0xFFFD) {
        e[D[190][j]] = 48640 + j;
        d[48640 + j] = D[190][j];
      }

      D[191] = "����������������������������������������������������������������緻緼緽緾緿縀縁縂縃縄縅縆縇縈縉縊縋縌縍縎縏縐縑縒縓縔縕縖縗縘縙縚縛縜縝縞縟縠縡縢縣縤縥縦縧縨縩縪縫縬縭縮縯縰縱縲縳縴縵縶縷縸縹�縺縼總績縿繀繂繃繄繅繆繈繉繊繋繌繍繎繏繐繑繒繓織繕繖繗繘繙繚繛繜繝俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀�".split("");

      for (j = 0; j != D[191].length; ++j) if (D[191][j].charCodeAt(0) !== 0xFFFD) {
        e[D[191][j]] = 48896 + j;
        d[48896 + j] = D[191][j];
      }

      D[192] = "����������������������������������������������������������������繞繟繠繡繢繣繤繥繦繧繨繩繪繫繬繭繮繯繰繱繲繳繴繵繶繷繸繹繺繻繼繽繾繿纀纁纃纄纅纆纇纈纉纊纋續纍纎纏纐纑纒纓纔纕纖纗纘纙纚纜纝纞�纮纴纻纼绖绤绬绹缊缐缞缷缹缻缼缽缾缿罀罁罃罆罇罈罉罊罋罌罍罎罏罒罓馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐�".split("");

      for (j = 0; j != D[192].length; ++j) if (D[192][j].charCodeAt(0) !== 0xFFFD) {
        e[D[192][j]] = 49152 + j;
        d[49152 + j] = D[192][j];
      }

      D[193] = "����������������������������������������������������������������罖罙罛罜罝罞罠罣罤罥罦罧罫罬罭罯罰罳罵罶罷罸罺罻罼罽罿羀羂羃羄羅羆羇羈羉羋羍羏羐羑羒羓羕羖羗羘羙羛羜羠羢羣羥羦羨義羪羫羬羭羮羱�羳羴羵羶羷羺羻羾翀翂翃翄翆翇翈翉翋翍翏翐翑習翓翖翗翙翚翛翜翝翞翢翣痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿�".split("");

      for (j = 0; j != D[193].length; ++j) if (D[193][j].charCodeAt(0) !== 0xFFFD) {
        e[D[193][j]] = 49408 + j;
        d[49408 + j] = D[193][j];
      }

      D[194] = "����������������������������������������������������������������翤翧翨翪翫翬翭翯翲翴翵翶翷翸翹翺翽翾翿耂耇耈耉耊耎耏耑耓耚耛耝耞耟耡耣耤耫耬耭耮耯耰耲耴耹耺耼耾聀聁聄聅聇聈聉聎聏聐聑聓聕聖聗�聙聛聜聝聞聟聠聡聢聣聤聥聦聧聨聫聬聭聮聯聰聲聳聴聵聶職聸聹聺聻聼聽隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫�".split("");

      for (j = 0; j != D[194].length; ++j) if (D[194][j].charCodeAt(0) !== 0xFFFD) {
        e[D[194][j]] = 49664 + j;
        d[49664 + j] = D[194][j];
      }

      D[195] = "����������������������������������������������������������������聾肁肂肅肈肊肍肎肏肐肑肒肔肕肗肙肞肣肦肧肨肬肰肳肵肶肸肹肻胅胇胈胉胊胋胏胐胑胒胓胔胕胘胟胠胢胣胦胮胵胷胹胻胾胿脀脁脃脄脅脇脈脋�脌脕脗脙脛脜脝脟脠脡脢脣脤脥脦脧脨脩脪脫脭脮脰脳脴脵脷脹脺脻脼脽脿谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸�".split("");

      for (j = 0; j != D[195].length; ++j) if (D[195][j].charCodeAt(0) !== 0xFFFD) {
        e[D[195][j]] = 49920 + j;
        d[49920 + j] = D[195][j];
      }

      D[196] = "����������������������������������������������������������������腀腁腂腃腄腅腇腉腍腎腏腒腖腗腘腛腜腝腞腟腡腢腣腤腦腨腪腫腬腯腲腳腵腶腷腸膁膃膄膅膆膇膉膋膌膍膎膐膒膓膔膕膖膗膙膚膞膟膠膡膢膤膥�膧膩膫膬膭膮膯膰膱膲膴膵膶膷膸膹膼膽膾膿臄臅臇臈臉臋臍臎臏臐臑臒臓摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁�".split("");

      for (j = 0; j != D[196].length; ++j) if (D[196][j].charCodeAt(0) !== 0xFFFD) {
        e[D[196][j]] = 50176 + j;
        d[50176 + j] = D[196][j];
      }

      D[197] = "����������������������������������������������������������������臔臕臖臗臘臙臚臛臜臝臞臟臠臡臢臤臥臦臨臩臫臮臯臰臱臲臵臶臷臸臹臺臽臿舃與興舉舊舋舎舏舑舓舕舖舗舘舙舚舝舠舤舥舦舧舩舮舲舺舼舽舿�艀艁艂艃艅艆艈艊艌艍艎艐艑艒艓艔艕艖艗艙艛艜艝艞艠艡艢艣艤艥艦艧艩拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗�".split("");

      for (j = 0; j != D[197].length; ++j) if (D[197][j].charCodeAt(0) !== 0xFFFD) {
        e[D[197][j]] = 50432 + j;
        d[50432 + j] = D[197][j];
      }

      D[198] = "����������������������������������������������������������������艪艫艬艭艱艵艶艷艸艻艼芀芁芃芅芆芇芉芌芐芓芔芕芖芚芛芞芠芢芣芧芲芵芶芺芻芼芿苀苂苃苅苆苉苐苖苙苚苝苢苧苨苩苪苬苭苮苰苲苳苵苶苸�苺苼苽苾苿茀茊茋茍茐茒茓茖茘茙茝茞茟茠茡茢茣茤茥茦茩茪茮茰茲茷茻茽啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐�".split("");

      for (j = 0; j != D[198].length; ++j) if (D[198][j].charCodeAt(0) !== 0xFFFD) {
        e[D[198][j]] = 50688 + j;
        d[50688 + j] = D[198][j];
      }

      D[199] = "����������������������������������������������������������������茾茿荁荂荄荅荈荊荋荌荍荎荓荕荖荗荘荙荝荢荰荱荲荳荴荵荶荹荺荾荿莀莁莂莃莄莇莈莊莋莌莍莏莐莑莔莕莖莗莙莚莝莟莡莢莣莤莥莦莧莬莭莮�莯莵莻莾莿菂菃菄菆菈菉菋菍菎菐菑菒菓菕菗菙菚菛菞菢菣菤菦菧菨菫菬菭恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠�".split("");

      for (j = 0; j != D[199].length; ++j) if (D[199][j].charCodeAt(0) !== 0xFFFD) {
        e[D[199][j]] = 50944 + j;
        d[50944 + j] = D[199][j];
      }

      D[200] = "����������������������������������������������������������������菮華菳菴菵菶菷菺菻菼菾菿萀萂萅萇萈萉萊萐萒萓萔萕萖萗萙萚萛萞萟萠萡萢萣萩萪萫萬萭萮萯萰萲萳萴萵萶萷萹萺萻萾萿葀葁葂葃葄葅葇葈葉�葊葋葌葍葎葏葐葒葓葔葕葖葘葝葞葟葠葢葤葥葦葧葨葪葮葯葰葲葴葷葹葻葼取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁�".split("");

      for (j = 0; j != D[200].length; ++j) if (D[200][j].charCodeAt(0) !== 0xFFFD) {
        e[D[200][j]] = 51200 + j;
        d[51200 + j] = D[200][j];
      }

      D[201] = "����������������������������������������������������������������葽葾葿蒀蒁蒃蒄蒅蒆蒊蒍蒏蒐蒑蒒蒓蒔蒕蒖蒘蒚蒛蒝蒞蒟蒠蒢蒣蒤蒥蒦蒧蒨蒩蒪蒫蒬蒭蒮蒰蒱蒳蒵蒶蒷蒻蒼蒾蓀蓂蓃蓅蓆蓇蓈蓋蓌蓎蓏蓒蓔蓕蓗�蓘蓙蓚蓛蓜蓞蓡蓢蓤蓧蓨蓩蓪蓫蓭蓮蓯蓱蓲蓳蓴蓵蓶蓷蓸蓹蓺蓻蓽蓾蔀蔁蔂伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳�".split("");

      for (j = 0; j != D[201].length; ++j) if (D[201][j].charCodeAt(0) !== 0xFFFD) {
        e[D[201][j]] = 51456 + j;
        d[51456 + j] = D[201][j];
      }

      D[202] = "����������������������������������������������������������������蔃蔄蔅蔆蔇蔈蔉蔊蔋蔍蔎蔏蔐蔒蔔蔕蔖蔘蔙蔛蔜蔝蔞蔠蔢蔣蔤蔥蔦蔧蔨蔩蔪蔭蔮蔯蔰蔱蔲蔳蔴蔵蔶蔾蔿蕀蕁蕂蕄蕅蕆蕇蕋蕌蕍蕎蕏蕐蕑蕒蕓蕔蕕�蕗蕘蕚蕛蕜蕝蕟蕠蕡蕢蕣蕥蕦蕧蕩蕪蕫蕬蕭蕮蕯蕰蕱蕳蕵蕶蕷蕸蕼蕽蕿薀薁省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱�".split("");

      for (j = 0; j != D[202].length; ++j) if (D[202][j].charCodeAt(0) !== 0xFFFD) {
        e[D[202][j]] = 51712 + j;
        d[51712 + j] = D[202][j];
      }

      D[203] = "����������������������������������������������������������������薂薃薆薈薉薊薋薌薍薎薐薑薒薓薔薕薖薗薘薙薚薝薞薟薠薡薢薣薥薦薧薩薫薬薭薱薲薳薴薵薶薸薺薻薼薽薾薿藀藂藃藄藅藆藇藈藊藋藌藍藎藑藒�藔藖藗藘藙藚藛藝藞藟藠藡藢藣藥藦藧藨藪藫藬藭藮藯藰藱藲藳藴藵藶藷藸恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔�".split("");

      for (j = 0; j != D[203].length; ++j) if (D[203][j].charCodeAt(0) !== 0xFFFD) {
        e[D[203][j]] = 51968 + j;
        d[51968 + j] = D[203][j];
      }

      D[204] = "����������������������������������������������������������������藹藺藼藽藾蘀蘁蘂蘃蘄蘆蘇蘈蘉蘊蘋蘌蘍蘎蘏蘐蘒蘓蘔蘕蘗蘘蘙蘚蘛蘜蘝蘞蘟蘠蘡蘢蘣蘤蘥蘦蘨蘪蘫蘬蘭蘮蘯蘰蘱蘲蘳蘴蘵蘶蘷蘹蘺蘻蘽蘾蘿虀�虁虂虃虄虅虆虇虈虉虊虋虌虒虓處虖虗虘虙虛虜虝號虠虡虣虤虥虦虧虨虩虪獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃�".split("");

      for (j = 0; j != D[204].length; ++j) if (D[204][j].charCodeAt(0) !== 0xFFFD) {
        e[D[204][j]] = 52224 + j;
        d[52224 + j] = D[204][j];
      }

      D[205] = "����������������������������������������������������������������虭虯虰虲虳虴虵虶虷虸蚃蚄蚅蚆蚇蚈蚉蚎蚏蚐蚑蚒蚔蚖蚗蚘蚙蚚蚛蚞蚟蚠蚡蚢蚥蚦蚫蚭蚮蚲蚳蚷蚸蚹蚻蚼蚽蚾蚿蛁蛂蛃蛅蛈蛌蛍蛒蛓蛕蛖蛗蛚蛜�蛝蛠蛡蛢蛣蛥蛦蛧蛨蛪蛫蛬蛯蛵蛶蛷蛺蛻蛼蛽蛿蜁蜄蜅蜆蜋蜌蜎蜏蜐蜑蜔蜖汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威�".split("");

      for (j = 0; j != D[205].length; ++j) if (D[205][j].charCodeAt(0) !== 0xFFFD) {
        e[D[205][j]] = 52480 + j;
        d[52480 + j] = D[205][j];
      }

      D[206] = "����������������������������������������������������������������蜙蜛蜝蜟蜠蜤蜦蜧蜨蜪蜫蜬蜭蜯蜰蜲蜳蜵蜶蜸蜹蜺蜼蜽蝀蝁蝂蝃蝄蝅蝆蝊蝋蝍蝏蝐蝑蝒蝔蝕蝖蝘蝚蝛蝜蝝蝞蝟蝡蝢蝦蝧蝨蝩蝪蝫蝬蝭蝯蝱蝲蝳蝵�蝷蝸蝹蝺蝿螀螁螄螆螇螉螊螌螎螏螐螑螒螔螕螖螘螙螚螛螜螝螞螠螡螢螣螤巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺�".split("");

      for (j = 0; j != D[206].length; ++j) if (D[206][j].charCodeAt(0) !== 0xFFFD) {
        e[D[206][j]] = 52736 + j;
        d[52736 + j] = D[206][j];
      }

      D[207] = "����������������������������������������������������������������螥螦螧螩螪螮螰螱螲螴螶螷螸螹螻螼螾螿蟁蟂蟃蟄蟅蟇蟈蟉蟌蟍蟎蟏蟐蟔蟕蟖蟗蟘蟙蟚蟜蟝蟞蟟蟡蟢蟣蟤蟦蟧蟨蟩蟫蟬蟭蟯蟰蟱蟲蟳蟴蟵蟶蟷蟸�蟺蟻蟼蟽蟿蠀蠁蠂蠄蠅蠆蠇蠈蠉蠋蠌蠍蠎蠏蠐蠑蠒蠔蠗蠘蠙蠚蠜蠝蠞蠟蠠蠣稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓�".split("");

      for (j = 0; j != D[207].length; ++j) if (D[207][j].charCodeAt(0) !== 0xFFFD) {
        e[D[207][j]] = 52992 + j;
        d[52992 + j] = D[207][j];
      }

      D[208] = "����������������������������������������������������������������蠤蠥蠦蠧蠨蠩蠪蠫蠬蠭蠮蠯蠰蠱蠳蠴蠵蠶蠷蠸蠺蠻蠽蠾蠿衁衂衃衆衇衈衉衊衋衎衏衐衑衒術衕衖衘衚衛衜衝衞衟衠衦衧衪衭衯衱衳衴衵衶衸衹衺�衻衼袀袃袆袇袉袊袌袎袏袐袑袓袔袕袗袘袙袚袛袝袞袟袠袡袣袥袦袧袨袩袪小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄�".split("");

      for (j = 0; j != D[208].length; ++j) if (D[208][j].charCodeAt(0) !== 0xFFFD) {
        e[D[208][j]] = 53248 + j;
        d[53248 + j] = D[208][j];
      }

      D[209] = "����������������������������������������������������������������袬袮袯袰袲袳袴袵袶袸袹袺袻袽袾袿裀裃裄裇裈裊裋裌裍裏裐裑裓裖裗裚裛補裝裞裠裡裦裧裩裪裫裬裭裮裯裲裵裶裷裺裻製裿褀褁褃褄褅褆複褈�褉褋褌褍褎褏褑褔褕褖褗褘褜褝褞褟褠褢褣褤褦褧褨褩褬褭褮褯褱褲褳褵褷选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶�".split("");

      for (j = 0; j != D[209].length; ++j) if (D[209][j].charCodeAt(0) !== 0xFFFD) {
        e[D[209][j]] = 53504 + j;
        d[53504 + j] = D[209][j];
      }

      D[210] = "����������������������������������������������������������������褸褹褺褻褼褽褾褿襀襂襃襅襆襇襈襉襊襋襌襍襎襏襐襑襒襓襔襕襖襗襘襙襚襛襜襝襠襡襢襣襤襥襧襨襩襪襫襬襭襮襯襰襱襲襳襴襵襶襷襸襹襺襼�襽襾覀覂覄覅覇覈覉覊見覌覍覎規覐覑覒覓覔覕視覗覘覙覚覛覜覝覞覟覠覡摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐�".split("");

      for (j = 0; j != D[210].length; ++j) if (D[210][j].charCodeAt(0) !== 0xFFFD) {
        e[D[210][j]] = 53760 + j;
        d[53760 + j] = D[210][j];
      }

      D[211] = "����������������������������������������������������������������覢覣覤覥覦覧覨覩親覫覬覭覮覯覰覱覲観覴覵覶覷覸覹覺覻覼覽覾覿觀觃觍觓觔觕觗觘觙觛觝觟觠觡觢觤觧觨觩觪觬觭觮觰觱觲觴觵觶觷觸觹觺�觻觼觽觾觿訁訂訃訄訅訆計訉訊訋訌訍討訏訐訑訒訓訔訕訖託記訙訚訛訜訝印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉�".split("");

      for (j = 0; j != D[211].length; ++j) if (D[211][j].charCodeAt(0) !== 0xFFFD) {
        e[D[211][j]] = 54016 + j;
        d[54016 + j] = D[211][j];
      }

      D[212] = "����������������������������������������������������������������訞訟訠訡訢訣訤訥訦訧訨訩訪訫訬設訮訯訰許訲訳訴訵訶訷訸訹診註証訽訿詀詁詂詃詄詅詆詇詉詊詋詌詍詎詏詐詑詒詓詔評詖詗詘詙詚詛詜詝詞�詟詠詡詢詣詤詥試詧詨詩詪詫詬詭詮詯詰話該詳詴詵詶詷詸詺詻詼詽詾詿誀浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧�".split("");

      for (j = 0; j != D[212].length; ++j) if (D[212][j].charCodeAt(0) !== 0xFFFD) {
        e[D[212][j]] = 54272 + j;
        d[54272 + j] = D[212][j];
      }

      D[213] = "����������������������������������������������������������������誁誂誃誄誅誆誇誈誋誌認誎誏誐誑誒誔誕誖誗誘誙誚誛誜誝語誟誠誡誢誣誤誥誦誧誨誩說誫説読誮誯誰誱課誳誴誵誶誷誸誹誺誻誼誽誾調諀諁諂�諃諄諅諆談諈諉諊請諌諍諎諏諐諑諒諓諔諕論諗諘諙諚諛諜諝諞諟諠諡諢諣铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政�".split("");

      for (j = 0; j != D[213].length; ++j) if (D[213][j].charCodeAt(0) !== 0xFFFD) {
        e[D[213][j]] = 54528 + j;
        d[54528 + j] = D[213][j];
      }

      D[214] = "����������������������������������������������������������������諤諥諦諧諨諩諪諫諬諭諮諯諰諱諲諳諴諵諶諷諸諹諺諻諼諽諾諿謀謁謂謃謄謅謆謈謉謊謋謌謍謎謏謐謑謒謓謔謕謖謗謘謙謚講謜謝謞謟謠謡謢謣�謤謥謧謨謩謪謫謬謭謮謯謰謱謲謳謴謵謶謷謸謹謺謻謼謽謾謿譀譁譂譃譄譅帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑�".split("");

      for (j = 0; j != D[214].length; ++j) if (D[214][j].charCodeAt(0) !== 0xFFFD) {
        e[D[214][j]] = 54784 + j;
        d[54784 + j] = D[214][j];
      }

      D[215] = "����������������������������������������������������������������譆譇譈證譊譋譌譍譎譏譐譑譒譓譔譕譖譗識譙譚譛譜譝譞譟譠譡譢譣譤譥譧譨譩譪譫譭譮譯議譱譲譳譴譵譶護譸譹譺譻譼譽譾譿讀讁讂讃讄讅讆�讇讈讉變讋讌讍讎讏讐讑讒讓讔讕讖讗讘讙讚讛讜讝讞讟讬讱讻诇诐诪谉谞住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座������".split("");

      for (j = 0; j != D[215].length; ++j) if (D[215][j].charCodeAt(0) !== 0xFFFD) {
        e[D[215][j]] = 55040 + j;
        d[55040 + j] = D[215][j];
      }

      D[216] = "����������������������������������������������������������������谸谹谺谻谼谽谾谿豀豂豃豄豅豈豊豋豍豎豏豐豑豒豓豔豖豗豘豙豛豜豝豞豟豠豣豤豥豦豧豨豩豬豭豮豯豰豱豲豴豵豶豷豻豼豽豾豿貀貁貃貄貆貇�貈貋貍貎貏貐貑貒貓貕貖貗貙貚貛貜貝貞貟負財貢貣貤貥貦貧貨販貪貫責貭亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝�".split("");

      for (j = 0; j != D[216].length; ++j) if (D[216][j].charCodeAt(0) !== 0xFFFD) {
        e[D[216][j]] = 55296 + j;
        d[55296 + j] = D[216][j];
      }

      D[217] = "����������������������������������������������������������������貮貯貰貱貲貳貴貵貶買貸貹貺費貼貽貾貿賀賁賂賃賄賅賆資賈賉賊賋賌賍賎賏賐賑賒賓賔賕賖賗賘賙賚賛賜賝賞賟賠賡賢賣賤賥賦賧賨賩質賫賬�賭賮賯賰賱賲賳賴賵賶賷賸賹賺賻購賽賾賿贀贁贂贃贄贅贆贇贈贉贊贋贌贍佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼�".split("");

      for (j = 0; j != D[217].length; ++j) if (D[217][j].charCodeAt(0) !== 0xFFFD) {
        e[D[217][j]] = 55552 + j;
        d[55552 + j] = D[217][j];
      }

      D[218] = "����������������������������������������������������������������贎贏贐贑贒贓贔贕贖贗贘贙贚贛贜贠赑赒赗赟赥赨赩赪赬赮赯赱赲赸赹赺赻赼赽赾赿趀趂趃趆趇趈趉趌趍趎趏趐趒趓趕趖趗趘趙趚趛趜趝趞趠趡�趢趤趥趦趧趨趩趪趫趬趭趮趯趰趲趶趷趹趻趽跀跁跂跅跇跈跉跊跍跐跒跓跔凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺�".split("");

      for (j = 0; j != D[218].length; ++j) if (D[218][j].charCodeAt(0) !== 0xFFFD) {
        e[D[218][j]] = 55808 + j;
        d[55808 + j] = D[218][j];
      }

      D[219] = "����������������������������������������������������������������跕跘跙跜跠跡跢跥跦跧跩跭跮跰跱跲跴跶跼跾跿踀踁踂踃踄踆踇踈踋踍踎踐踑踒踓踕踖踗踘踙踚踛踜踠踡踤踥踦踧踨踫踭踰踲踳踴踶踷踸踻踼踾�踿蹃蹅蹆蹌蹍蹎蹏蹐蹓蹔蹕蹖蹗蹘蹚蹛蹜蹝蹞蹟蹠蹡蹢蹣蹤蹥蹧蹨蹪蹫蹮蹱邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝�".split("");

      for (j = 0; j != D[219].length; ++j) if (D[219][j].charCodeAt(0) !== 0xFFFD) {
        e[D[219][j]] = 56064 + j;
        d[56064 + j] = D[219][j];
      }

      D[220] = "����������������������������������������������������������������蹳蹵蹷蹸蹹蹺蹻蹽蹾躀躂躃躄躆躈躉躊躋躌躍躎躑躒躓躕躖躗躘躙躚躛躝躟躠躡躢躣躤躥躦躧躨躩躪躭躮躰躱躳躴躵躶躷躸躹躻躼躽躾躿軀軁軂�軃軄軅軆軇軈軉車軋軌軍軏軐軑軒軓軔軕軖軗軘軙軚軛軜軝軞軟軠軡転軣軤堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥�".split("");

      for (j = 0; j != D[220].length; ++j) if (D[220][j].charCodeAt(0) !== 0xFFFD) {
        e[D[220][j]] = 56320 + j;
        d[56320 + j] = D[220][j];
      }

      D[221] = "����������������������������������������������������������������軥軦軧軨軩軪軫軬軭軮軯軰軱軲軳軴軵軶軷軸軹軺軻軼軽軾軿輀輁輂較輄輅輆輇輈載輊輋輌輍輎輏輐輑輒輓輔輕輖輗輘輙輚輛輜輝輞輟輠輡輢輣�輤輥輦輧輨輩輪輫輬輭輮輯輰輱輲輳輴輵輶輷輸輹輺輻輼輽輾輿轀轁轂轃轄荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺�".split("");

      for (j = 0; j != D[221].length; ++j) if (D[221][j].charCodeAt(0) !== 0xFFFD) {
        e[D[221][j]] = 56576 + j;
        d[56576 + j] = D[221][j];
      }

      D[222] = "����������������������������������������������������������������轅轆轇轈轉轊轋轌轍轎轏轐轑轒轓轔轕轖轗轘轙轚轛轜轝轞轟轠轡轢轣轤轥轪辀辌辒辝辠辡辢辤辥辦辧辪辬辭辮辯農辳辴辵辷辸辺辻込辿迀迃迆�迉迊迋迌迍迏迒迖迗迚迠迡迣迧迬迯迱迲迴迵迶迺迻迼迾迿逇逈逌逎逓逕逘蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖�".split("");

      for (j = 0; j != D[222].length; ++j) if (D[222][j].charCodeAt(0) !== 0xFFFD) {
        e[D[222][j]] = 56832 + j;
        d[56832 + j] = D[222][j];
      }

      D[223] = "����������������������������������������������������������������這逜連逤逥逧逨逩逪逫逬逰週進逳逴逷逹逺逽逿遀遃遅遆遈遉遊運遌過達違遖遙遚遜遝遞遟遠遡遤遦遧適遪遫遬遯遰遱遲遳遶遷選遹遺遻遼遾邁�還邅邆邇邉邊邌邍邎邏邐邒邔邖邘邚邜邞邟邠邤邥邧邨邩邫邭邲邷邼邽邿郀摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼�".split("");

      for (j = 0; j != D[223].length; ++j) if (D[223][j].charCodeAt(0) !== 0xFFFD) {
        e[D[223][j]] = 57088 + j;
        d[57088 + j] = D[223][j];
      }

      D[224] = "����������������������������������������������������������������郂郃郆郈郉郋郌郍郒郔郕郖郘郙郚郞郟郠郣郤郥郩郪郬郮郰郱郲郳郵郶郷郹郺郻郼郿鄀鄁鄃鄅鄆鄇鄈鄉鄊鄋鄌鄍鄎鄏鄐鄑鄒鄓鄔鄕鄖鄗鄘鄚鄛鄜�鄝鄟鄠鄡鄤鄥鄦鄧鄨鄩鄪鄫鄬鄭鄮鄰鄲鄳鄴鄵鄶鄷鄸鄺鄻鄼鄽鄾鄿酀酁酂酄唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼�".split("");

      for (j = 0; j != D[224].length; ++j) if (D[224][j].charCodeAt(0) !== 0xFFFD) {
        e[D[224][j]] = 57344 + j;
        d[57344 + j] = D[224][j];
      }

      D[225] = "����������������������������������������������������������������酅酇酈酑酓酔酕酖酘酙酛酜酟酠酦酧酨酫酭酳酺酻酼醀醁醂醃醄醆醈醊醎醏醓醔醕醖醗醘醙醜醝醞醟醠醡醤醥醦醧醨醩醫醬醰醱醲醳醶醷醸醹醻�醼醽醾醿釀釁釂釃釄釅釆釈釋釐釒釓釔釕釖釗釘釙釚釛針釞釟釠釡釢釣釤釥帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺�".split("");

      for (j = 0; j != D[225].length; ++j) if (D[225][j].charCodeAt(0) !== 0xFFFD) {
        e[D[225][j]] = 57600 + j;
        d[57600 + j] = D[225][j];
      }

      D[226] = "����������������������������������������������������������������釦釧釨釩釪釫釬釭釮釯釰釱釲釳釴釵釶釷釸釹釺釻釼釽釾釿鈀鈁鈂鈃鈄鈅鈆鈇鈈鈉鈊鈋鈌鈍鈎鈏鈐鈑鈒鈓鈔鈕鈖鈗鈘鈙鈚鈛鈜鈝鈞鈟鈠鈡鈢鈣鈤�鈥鈦鈧鈨鈩鈪鈫鈬鈭鈮鈯鈰鈱鈲鈳鈴鈵鈶鈷鈸鈹鈺鈻鈼鈽鈾鈿鉀鉁鉂鉃鉄鉅狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧饨饩饪饫饬饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂�".split("");

      for (j = 0; j != D[226].length; ++j) if (D[226][j].charCodeAt(0) !== 0xFFFD) {
        e[D[226][j]] = 57856 + j;
        d[57856 + j] = D[226][j];
      }

      D[227] = "����������������������������������������������������������������鉆鉇鉈鉉鉊鉋鉌鉍鉎鉏鉐鉑鉒鉓鉔鉕鉖鉗鉘鉙鉚鉛鉜鉝鉞鉟鉠鉡鉢鉣鉤鉥鉦鉧鉨鉩鉪鉫鉬鉭鉮鉯鉰鉱鉲鉳鉵鉶鉷鉸鉹鉺鉻鉼鉽鉾鉿銀銁銂銃銄銅�銆銇銈銉銊銋銌銍銏銐銑銒銓銔銕銖銗銘銙銚銛銜銝銞銟銠銡銢銣銤銥銦銧恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾�".split("");

      for (j = 0; j != D[227].length; ++j) if (D[227][j].charCodeAt(0) !== 0xFFFD) {
        e[D[227][j]] = 58112 + j;
        d[58112 + j] = D[227][j];
      }

      D[228] = "����������������������������������������������������������������銨銩銪銫銬銭銯銰銱銲銳銴銵銶銷銸銹銺銻銼銽銾銿鋀鋁鋂鋃鋄鋅鋆鋇鋉鋊鋋鋌鋍鋎鋏鋐鋑鋒鋓鋔鋕鋖鋗鋘鋙鋚鋛鋜鋝鋞鋟鋠鋡鋢鋣鋤鋥鋦鋧鋨�鋩鋪鋫鋬鋭鋮鋯鋰鋱鋲鋳鋴鋵鋶鋷鋸鋹鋺鋻鋼鋽鋾鋿錀錁錂錃錄錅錆錇錈錉洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑�".split("");

      for (j = 0; j != D[228].length; ++j) if (D[228][j].charCodeAt(0) !== 0xFFFD) {
        e[D[228][j]] = 58368 + j;
        d[58368 + j] = D[228][j];
      }

      D[229] = "����������������������������������������������������������������錊錋錌錍錎錏錐錑錒錓錔錕錖錗錘錙錚錛錜錝錞錟錠錡錢錣錤錥錦錧錨錩錪錫錬錭錮錯錰錱録錳錴錵錶錷錸錹錺錻錼錽錿鍀鍁鍂鍃鍄鍅鍆鍇鍈鍉�鍊鍋鍌鍍鍎鍏鍐鍑鍒鍓鍔鍕鍖鍗鍘鍙鍚鍛鍜鍝鍞鍟鍠鍡鍢鍣鍤鍥鍦鍧鍨鍩鍫濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣�".split("");

      for (j = 0; j != D[229].length; ++j) if (D[229][j].charCodeAt(0) !== 0xFFFD) {
        e[D[229][j]] = 58624 + j;
        d[58624 + j] = D[229][j];
      }

      D[230] = "����������������������������������������������������������������鍬鍭鍮鍯鍰鍱鍲鍳鍴鍵鍶鍷鍸鍹鍺鍻鍼鍽鍾鍿鎀鎁鎂鎃鎄鎅鎆鎇鎈鎉鎊鎋鎌鎍鎎鎐鎑鎒鎓鎔鎕鎖鎗鎘鎙鎚鎛鎜鎝鎞鎟鎠鎡鎢鎣鎤鎥鎦鎧鎨鎩鎪鎫�鎬鎭鎮鎯鎰鎱鎲鎳鎴鎵鎶鎷鎸鎹鎺鎻鎼鎽鎾鎿鏀鏁鏂鏃鏄鏅鏆鏇鏈鏉鏋鏌鏍妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩�".split("");

      for (j = 0; j != D[230].length; ++j) if (D[230][j].charCodeAt(0) !== 0xFFFD) {
        e[D[230][j]] = 58880 + j;
        d[58880 + j] = D[230][j];
      }

      D[231] = "����������������������������������������������������������������鏎鏏鏐鏑鏒鏓鏔鏕鏗鏘鏙鏚鏛鏜鏝鏞鏟鏠鏡鏢鏣鏤鏥鏦鏧鏨鏩鏪鏫鏬鏭鏮鏯鏰鏱鏲鏳鏴鏵鏶鏷鏸鏹鏺鏻鏼鏽鏾鏿鐀鐁鐂鐃鐄鐅鐆鐇鐈鐉鐊鐋鐌鐍�鐎鐏鐐鐑鐒鐓鐔鐕鐖鐗鐘鐙鐚鐛鐜鐝鐞鐟鐠鐡鐢鐣鐤鐥鐦鐧鐨鐩鐪鐫鐬鐭鐮纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡缢缣缤缥缦缧缪缫缬缭缯缰缱缲缳缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬�".split("");

      for (j = 0; j != D[231].length; ++j) if (D[231][j].charCodeAt(0) !== 0xFFFD) {
        e[D[231][j]] = 59136 + j;
        d[59136 + j] = D[231][j];
      }

      D[232] = "����������������������������������������������������������������鐯鐰鐱鐲鐳鐴鐵鐶鐷鐸鐹鐺鐻鐼鐽鐿鑀鑁鑂鑃鑄鑅鑆鑇鑈鑉鑊鑋鑌鑍鑎鑏鑐鑑鑒鑓鑔鑕鑖鑗鑘鑙鑚鑛鑜鑝鑞鑟鑠鑡鑢鑣鑤鑥鑦鑧鑨鑩鑪鑬鑭鑮鑯�鑰鑱鑲鑳鑴鑵鑶鑷鑸鑹鑺鑻鑼鑽鑾鑿钀钁钂钃钄钑钖钘铇铏铓铔铚铦铻锜锠琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹�".split("");

      for (j = 0; j != D[232].length; ++j) if (D[232][j].charCodeAt(0) !== 0xFFFD) {
        e[D[232][j]] = 59392 + j;
        d[59392 + j] = D[232][j];
      }

      D[233] = "����������������������������������������������������������������锧锳锽镃镈镋镕镚镠镮镴镵長镸镹镺镻镼镽镾門閁閂閃閄閅閆閇閈閉閊開閌閍閎閏閐閑閒間閔閕閖閗閘閙閚閛閜閝閞閟閠閡関閣閤閥閦閧閨閩閪�閫閬閭閮閯閰閱閲閳閴閵閶閷閸閹閺閻閼閽閾閿闀闁闂闃闄闅闆闇闈闉闊闋椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋�".split("");

      for (j = 0; j != D[233].length; ++j) if (D[233][j].charCodeAt(0) !== 0xFFFD) {
        e[D[233][j]] = 59648 + j;
        d[59648 + j] = D[233][j];
      }

      D[234] = "����������������������������������������������������������������闌闍闎闏闐闑闒闓闔闕闖闗闘闙闚闛關闝闞闟闠闡闢闣闤闥闦闧闬闿阇阓阘阛阞阠阣阤阥阦阧阨阩阫阬阭阯阰阷阸阹阺阾陁陃陊陎陏陑陒陓陖陗�陘陙陚陜陝陞陠陣陥陦陫陭陮陯陰陱陳陸陹険陻陼陽陾陿隀隁隂隃隄隇隉隊辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰�".split("");

      for (j = 0; j != D[234].length; ++j) if (D[234][j].charCodeAt(0) !== 0xFFFD) {
        e[D[234][j]] = 59904 + j;
        d[59904 + j] = D[234][j];
      }

      D[235] = "����������������������������������������������������������������隌階隑隒隓隕隖隚際隝隞隟隠隡隢隣隤隥隦隨隩險隫隬隭隮隯隱隲隴隵隷隸隺隻隿雂雃雈雊雋雐雑雓雔雖雗雘雙雚雛雜雝雞雟雡離難雤雥雦雧雫�雬雭雮雰雱雲雴雵雸雺電雼雽雿霂霃霅霊霋霌霐霑霒霔霕霗霘霙霚霛霝霟霠搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻�".split("");

      for (j = 0; j != D[235].length; ++j) if (D[235][j].charCodeAt(0) !== 0xFFFD) {
        e[D[235][j]] = 60160 + j;
        d[60160 + j] = D[235][j];
      }

      D[236] = "����������������������������������������������������������������霡霢霣霤霥霦霧霨霩霫霬霮霯霱霳霴霵霶霷霺霻霼霽霿靀靁靂靃靄靅靆靇靈靉靊靋靌靍靎靏靐靑靔靕靗靘靚靜靝靟靣靤靦靧靨靪靫靬靭靮靯靰靱�靲靵靷靸靹靺靻靽靾靿鞀鞁鞂鞃鞄鞆鞇鞈鞉鞊鞌鞎鞏鞐鞓鞕鞖鞗鞙鞚鞛鞜鞝臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐�".split("");

      for (j = 0; j != D[236].length; ++j) if (D[236][j].charCodeAt(0) !== 0xFFFD) {
        e[D[236][j]] = 60416 + j;
        d[60416 + j] = D[236][j];
      }

      D[237] = "����������������������������������������������������������������鞞鞟鞡鞢鞤鞥鞦鞧鞨鞩鞪鞬鞮鞰鞱鞳鞵鞶鞷鞸鞹鞺鞻鞼鞽鞾鞿韀韁韂韃韄韅韆韇韈韉韊韋韌韍韎韏韐韑韒韓韔韕韖韗韘韙韚韛韜韝韞韟韠韡韢韣�韤韥韨韮韯韰韱韲韴韷韸韹韺韻韼韽韾響頀頁頂頃頄項順頇須頉頊頋頌頍頎怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨�".split("");

      for (j = 0; j != D[237].length; ++j) if (D[237][j].charCodeAt(0) !== 0xFFFD) {
        e[D[237][j]] = 60672 + j;
        d[60672 + j] = D[237][j];
      }

      D[238] = "����������������������������������������������������������������頏預頑頒頓頔頕頖頗領頙頚頛頜頝頞頟頠頡頢頣頤頥頦頧頨頩頪頫頬頭頮頯頰頱頲頳頴頵頶頷頸頹頺頻頼頽頾頿顀顁顂顃顄顅顆顇顈顉顊顋題額�顎顏顐顑顒顓顔顕顖顗願顙顚顛顜顝類顟顠顡顢顣顤顥顦顧顨顩顪顫顬顭顮睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶钷钸钹钺钼钽钿铄铈铉铊铋铌铍铎铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪�".split("");

      for (j = 0; j != D[238].length; ++j) if (D[238][j].charCodeAt(0) !== 0xFFFD) {
        e[D[238][j]] = 60928 + j;
        d[60928 + j] = D[238][j];
      }

      D[239] = "����������������������������������������������������������������顯顰顱顲顳顴颋颎颒颕颙颣風颩颪颫颬颭颮颯颰颱颲颳颴颵颶颷颸颹颺颻颼颽颾颿飀飁飂飃飄飅飆飇飈飉飊飋飌飍飏飐飔飖飗飛飜飝飠飡飢飣飤�飥飦飩飪飫飬飭飮飯飰飱飲飳飴飵飶飷飸飹飺飻飼飽飾飿餀餁餂餃餄餅餆餇铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒锓锔锕锖锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤镥镦镧镨镩镪镫镬镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔�".split("");

      for (j = 0; j != D[239].length; ++j) if (D[239][j].charCodeAt(0) !== 0xFFFD) {
        e[D[239][j]] = 61184 + j;
        d[61184 + j] = D[239][j];
      }

      D[240] = "����������������������������������������������������������������餈餉養餋餌餎餏餑餒餓餔餕餖餗餘餙餚餛餜餝餞餟餠餡餢餣餤餥餦餧館餩餪餫餬餭餯餰餱餲餳餴餵餶餷餸餹餺餻餼餽餾餿饀饁饂饃饄饅饆饇饈饉�饊饋饌饍饎饏饐饑饒饓饖饗饘饙饚饛饜饝饞饟饠饡饢饤饦饳饸饹饻饾馂馃馉稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨鸩鸪鸫鸬鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦鹧鹨鹩鹪鹫鹬鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙�".split("");

      for (j = 0; j != D[240].length; ++j) if (D[240][j].charCodeAt(0) !== 0xFFFD) {
        e[D[240][j]] = 61440 + j;
        d[61440 + j] = D[240][j];
      }

      D[241] = "����������������������������������������������������������������馌馎馚馛馜馝馞馟馠馡馢馣馤馦馧馩馪馫馬馭馮馯馰馱馲馳馴馵馶馷馸馹馺馻馼馽馾馿駀駁駂駃駄駅駆駇駈駉駊駋駌駍駎駏駐駑駒駓駔駕駖駗駘�駙駚駛駜駝駞駟駠駡駢駣駤駥駦駧駨駩駪駫駬駭駮駯駰駱駲駳駴駵駶駷駸駹瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃�".split("");

      for (j = 0; j != D[241].length; ++j) if (D[241][j].charCodeAt(0) !== 0xFFFD) {
        e[D[241][j]] = 61696 + j;
        d[61696 + j] = D[241][j];
      }

      D[242] = "����������������������������������������������������������������駺駻駼駽駾駿騀騁騂騃騄騅騆騇騈騉騊騋騌騍騎騏騐騑騒験騔騕騖騗騘騙騚騛騜騝騞騟騠騡騢騣騤騥騦騧騨騩騪騫騬騭騮騯騰騱騲騳騴騵騶騷騸�騹騺騻騼騽騾騿驀驁驂驃驄驅驆驇驈驉驊驋驌驍驎驏驐驑驒驓驔驕驖驗驘驙颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒�".split("");

      for (j = 0; j != D[242].length; ++j) if (D[242][j].charCodeAt(0) !== 0xFFFD) {
        e[D[242][j]] = 61952 + j;
        d[61952 + j] = D[242][j];
      }

      D[243] = "����������������������������������������������������������������驚驛驜驝驞驟驠驡驢驣驤驥驦驧驨驩驪驫驲骃骉骍骎骔骕骙骦骩骪骫骬骭骮骯骲骳骴骵骹骻骽骾骿髃髄髆髇髈髉髊髍髎髏髐髒體髕髖髗髙髚髛髜�髝髞髠髢髣髤髥髧髨髩髪髬髮髰髱髲髳髴髵髶髷髸髺髼髽髾髿鬀鬁鬂鬄鬅鬆蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋�".split("");

      for (j = 0; j != D[243].length; ++j) if (D[243][j].charCodeAt(0) !== 0xFFFD) {
        e[D[243][j]] = 62208 + j;
        d[62208 + j] = D[243][j];
      }

      D[244] = "����������������������������������������������������������������鬇鬉鬊鬋鬌鬍鬎鬐鬑鬒鬔鬕鬖鬗鬘鬙鬚鬛鬜鬝鬞鬠鬡鬢鬤鬥鬦鬧鬨鬩鬪鬫鬬鬭鬮鬰鬱鬳鬴鬵鬶鬷鬸鬹鬺鬽鬾鬿魀魆魊魋魌魎魐魒魓魕魖魗魘魙魚�魛魜魝魞魟魠魡魢魣魤魥魦魧魨魩魪魫魬魭魮魯魰魱魲魳魴魵魶魷魸魹魺魻簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤�".split("");

      for (j = 0; j != D[244].length; ++j) if (D[244][j].charCodeAt(0) !== 0xFFFD) {
        e[D[244][j]] = 62464 + j;
        d[62464 + j] = D[244][j];
      }

      D[245] = "����������������������������������������������������������������魼魽魾魿鮀鮁鮂鮃鮄鮅鮆鮇鮈鮉鮊鮋鮌鮍鮎鮏鮐鮑鮒鮓鮔鮕鮖鮗鮘鮙鮚鮛鮜鮝鮞鮟鮠鮡鮢鮣鮤鮥鮦鮧鮨鮩鮪鮫鮬鮭鮮鮯鮰鮱鮲鮳鮴鮵鮶鮷鮸鮹鮺�鮻鮼鮽鮾鮿鯀鯁鯂鯃鯄鯅鯆鯇鯈鯉鯊鯋鯌鯍鯎鯏鯐鯑鯒鯓鯔鯕鯖鯗鯘鯙鯚鯛酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜�".split("");

      for (j = 0; j != D[245].length; ++j) if (D[245][j].charCodeAt(0) !== 0xFFFD) {
        e[D[245][j]] = 62720 + j;
        d[62720 + j] = D[245][j];
      }

      D[246] = "����������������������������������������������������������������鯜鯝鯞鯟鯠鯡鯢鯣鯤鯥鯦鯧鯨鯩鯪鯫鯬鯭鯮鯯鯰鯱鯲鯳鯴鯵鯶鯷鯸鯹鯺鯻鯼鯽鯾鯿鰀鰁鰂鰃鰄鰅鰆鰇鰈鰉鰊鰋鰌鰍鰎鰏鰐鰑鰒鰓鰔鰕鰖鰗鰘鰙鰚�鰛鰜鰝鰞鰟鰠鰡鰢鰣鰤鰥鰦鰧鰨鰩鰪鰫鰬鰭鰮鰯鰰鰱鰲鰳鰴鰵鰶鰷鰸鰹鰺鰻觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅龆龇龈龉龊龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞鲟鲠鲡鲢鲣鲥鲦鲧鲨鲩鲫鲭鲮鲰鲱鲲鲳鲴鲵鲶鲷鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋�".split("");

      for (j = 0; j != D[246].length; ++j) if (D[246][j].charCodeAt(0) !== 0xFFFD) {
        e[D[246][j]] = 62976 + j;
        d[62976 + j] = D[246][j];
      }

      D[247] = "����������������������������������������������������������������鰼鰽鰾鰿鱀鱁鱂鱃鱄鱅鱆鱇鱈鱉鱊鱋鱌鱍鱎鱏鱐鱑鱒鱓鱔鱕鱖鱗鱘鱙鱚鱛鱜鱝鱞鱟鱠鱡鱢鱣鱤鱥鱦鱧鱨鱩鱪鱫鱬鱭鱮鱯鱰鱱鱲鱳鱴鱵鱶鱷鱸鱹鱺�鱻鱽鱾鲀鲃鲄鲉鲊鲌鲏鲓鲖鲗鲘鲙鲝鲪鲬鲯鲹鲾鲿鳀鳁鳂鳈鳉鳑鳒鳚鳛鳠鳡鳌鳍鳎鳏鳐鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄�".split("");

      for (j = 0; j != D[247].length; ++j) if (D[247][j].charCodeAt(0) !== 0xFFFD) {
        e[D[247][j]] = 63232 + j;
        d[63232 + j] = D[247][j];
      }

      D[248] = "����������������������������������������������������������������鳣鳤鳥鳦鳧鳨鳩鳪鳫鳬鳭鳮鳯鳰鳱鳲鳳鳴鳵鳶鳷鳸鳹鳺鳻鳼鳽鳾鳿鴀鴁鴂鴃鴄鴅鴆鴇鴈鴉鴊鴋鴌鴍鴎鴏鴐鴑鴒鴓鴔鴕鴖鴗鴘鴙鴚鴛鴜鴝鴞鴟鴠鴡�鴢鴣鴤鴥鴦鴧鴨鴩鴪鴫鴬鴭鴮鴯鴰鴱鴲鴳鴴鴵鴶鴷鴸鴹鴺鴻鴼鴽鴾鴿鵀鵁鵂�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[248].length; ++j) if (D[248][j].charCodeAt(0) !== 0xFFFD) {
        e[D[248][j]] = 63488 + j;
        d[63488 + j] = D[248][j];
      }

      D[249] = "����������������������������������������������������������������鵃鵄鵅鵆鵇鵈鵉鵊鵋鵌鵍鵎鵏鵐鵑鵒鵓鵔鵕鵖鵗鵘鵙鵚鵛鵜鵝鵞鵟鵠鵡鵢鵣鵤鵥鵦鵧鵨鵩鵪鵫鵬鵭鵮鵯鵰鵱鵲鵳鵴鵵鵶鵷鵸鵹鵺鵻鵼鵽鵾鵿鶀鶁�鶂鶃鶄鶅鶆鶇鶈鶉鶊鶋鶌鶍鶎鶏鶐鶑鶒鶓鶔鶕鶖鶗鶘鶙鶚鶛鶜鶝鶞鶟鶠鶡鶢�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[249].length; ++j) if (D[249][j].charCodeAt(0) !== 0xFFFD) {
        e[D[249][j]] = 63744 + j;
        d[63744 + j] = D[249][j];
      }

      D[250] = "����������������������������������������������������������������鶣鶤鶥鶦鶧鶨鶩鶪鶫鶬鶭鶮鶯鶰鶱鶲鶳鶴鶵鶶鶷鶸鶹鶺鶻鶼鶽鶾鶿鷀鷁鷂鷃鷄鷅鷆鷇鷈鷉鷊鷋鷌鷍鷎鷏鷐鷑鷒鷓鷔鷕鷖鷗鷘鷙鷚鷛鷜鷝鷞鷟鷠鷡�鷢鷣鷤鷥鷦鷧鷨鷩鷪鷫鷬鷭鷮鷯鷰鷱鷲鷳鷴鷵鷶鷷鷸鷹鷺鷻鷼鷽鷾鷿鸀鸁鸂�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[250].length; ++j) if (D[250][j].charCodeAt(0) !== 0xFFFD) {
        e[D[250][j]] = 64000 + j;
        d[64000 + j] = D[250][j];
      }

      D[251] = "����������������������������������������������������������������鸃鸄鸅鸆鸇鸈鸉鸊鸋鸌鸍鸎鸏鸐鸑鸒鸓鸔鸕鸖鸗鸘鸙鸚鸛鸜鸝鸞鸤鸧鸮鸰鸴鸻鸼鹀鹍鹐鹒鹓鹔鹖鹙鹝鹟鹠鹡鹢鹥鹮鹯鹲鹴鹵鹶鹷鹸鹹鹺鹻鹼鹽麀�麁麃麄麅麆麉麊麌麍麎麏麐麑麔麕麖麗麘麙麚麛麜麞麠麡麢麣麤麥麧麨麩麪�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[251].length; ++j) if (D[251][j].charCodeAt(0) !== 0xFFFD) {
        e[D[251][j]] = 64256 + j;
        d[64256 + j] = D[251][j];
      }

      D[252] = "����������������������������������������������������������������麫麬麭麮麯麰麱麲麳麵麶麷麹麺麼麿黀黁黂黃黅黆黇黈黊黋黌黐黒黓黕黖黗黙黚點黡黣黤黦黨黫黬黭黮黰黱黲黳黴黵黶黷黸黺黽黿鼀鼁鼂鼃鼄鼅�鼆鼇鼈鼉鼊鼌鼏鼑鼒鼔鼕鼖鼘鼚鼛鼜鼝鼞鼟鼡鼣鼤鼥鼦鼧鼨鼩鼪鼫鼭鼮鼰鼱�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[252].length; ++j) if (D[252][j].charCodeAt(0) !== 0xFFFD) {
        e[D[252][j]] = 64512 + j;
        d[64512 + j] = D[252][j];
      }

      D[253] = "����������������������������������������������������������������鼲鼳鼴鼵鼶鼸鼺鼼鼿齀齁齂齃齅齆齇齈齉齊齋齌齍齎齏齒齓齔齕齖齗齘齙齚齛齜齝齞齟齠齡齢齣齤齥齦齧齨齩齪齫齬齭齮齯齰齱齲齳齴齵齶齷齸�齹齺齻齼齽齾龁龂龍龎龏龐龑龒龓龔龕龖龗龘龜龝龞龡龢龣龤龥郎凉秊裏隣�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[253].length; ++j) if (D[253][j].charCodeAt(0) !== 0xFFFD) {
        e[D[253][j]] = 64768 + j;
        d[64768 + j] = D[253][j];
      }

      D[254] = "����������������������������������������������������������������兀嗀﨎﨏﨑﨓﨔礼﨟蘒﨡﨣﨤﨧﨨﨩��������������������������������������������������������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[254].length; ++j) if (D[254][j].charCodeAt(0) !== 0xFFFD) {
        e[D[254][j]] = 65024 + j;
        d[65024 + j] = D[254][j];
      }

      return {
        "enc": e,
        "dec": d
      };
    }();

    cptable[949] = function () {
      var d = [],
          e = {},
          D = [],
          j;
      D[0] = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~��������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[0].length; ++j) if (D[0][j].charCodeAt(0) !== 0xFFFD) {
        e[D[0][j]] = 0 + j;
        d[0 + j] = D[0][j];
      }

      D[129] = "�����������������������������������������������������������������갂갃갅갆갋갌갍갎갏갘갞갟갡갢갣갥갦갧갨갩갪갫갮갲갳갴������갵갶갷갺갻갽갾갿걁걂걃걄걅걆걇걈걉걊걌걎걏걐걑걒걓걕������걖걗걙걚걛걝걞걟걠걡걢걣걤걥걦걧걨걩걪걫걬걭걮걯걲걳걵걶걹걻걼걽걾걿겂겇겈겍겎겏겑겒겓겕겖겗겘겙겚겛겞겢겣겤겥겦겧겫겭겮겱겲겳겴겵겶겷겺겾겿곀곂곃곅곆곇곉곊곋곍곎곏곐곑곒곓곔곖곘곙곚곛곜곝곞곟곢곣곥곦곩곫곭곮곲곴곷곸곹곺곻곾곿괁괂괃괅괇괈괉괊괋괎괐괒괓�".split("");

      for (j = 0; j != D[129].length; ++j) if (D[129][j].charCodeAt(0) !== 0xFFFD) {
        e[D[129][j]] = 33024 + j;
        d[33024 + j] = D[129][j];
      }

      D[130] = "�����������������������������������������������������������������괔괕괖괗괙괚괛괝괞괟괡괢괣괤괥괦괧괨괪괫괮괯괰괱괲괳������괶괷괹괺괻괽괾괿굀굁굂굃굆굈굊굋굌굍굎굏굑굒굓굕굖굗������굙굚굛굜굝굞굟굠굢굤굥굦굧굨굩굪굫굮굯굱굲굷굸굹굺굾궀궃궄궅궆궇궊궋궍궎궏궑궒궓궔궕궖궗궘궙궚궛궞궟궠궡궢궣궥궦궧궨궩궪궫궬궭궮궯궰궱궲궳궴궵궶궸궹궺궻궼궽궾궿귂귃귅귆귇귉귊귋귌귍귎귏귒귔귕귖귗귘귙귚귛귝귞귟귡귢귣귥귦귧귨귩귪귫귬귭귮귯귰귱귲귳귴귵귶귷�".split("");

      for (j = 0; j != D[130].length; ++j) if (D[130][j].charCodeAt(0) !== 0xFFFD) {
        e[D[130][j]] = 33280 + j;
        d[33280 + j] = D[130][j];
      }

      D[131] = "�����������������������������������������������������������������귺귻귽귾긂긃긄긅긆긇긊긌긎긏긐긑긒긓긕긖긗긘긙긚긛긜������긝긞긟긠긡긢긣긤긥긦긧긨긩긪긫긬긭긮긯긲긳긵긶긹긻긼������긽긾긿깂깄깇깈깉깋깏깑깒깓깕깗깘깙깚깛깞깢깣깤깦깧깪깫깭깮깯깱깲깳깴깵깶깷깺깾깿꺀꺁꺂꺃꺆꺇꺈꺉꺊꺋꺍꺎꺏꺐꺑꺒꺓꺔꺕꺖꺗꺘꺙꺚꺛꺜꺝꺞꺟꺠꺡꺢꺣꺤꺥꺦꺧꺨꺩꺪꺫꺬꺭꺮꺯꺰꺱꺲꺳꺴꺵꺶꺷꺸꺹꺺꺻꺿껁껂껃껅껆껇껈껉껊껋껎껒껓껔껕껖껗껚껛껝껞껟껠껡껢껣껤껥�".split("");

      for (j = 0; j != D[131].length; ++j) if (D[131][j].charCodeAt(0) !== 0xFFFD) {
        e[D[131][j]] = 33536 + j;
        d[33536 + j] = D[131][j];
      }

      D[132] = "�����������������������������������������������������������������껦껧껩껪껬껮껯껰껱껲껳껵껶껷껹껺껻껽껾껿꼀꼁꼂꼃꼄꼅������꼆꼉꼊꼋꼌꼎꼏꼑꼒꼓꼔꼕꼖꼗꼘꼙꼚꼛꼜꼝꼞꼟꼠꼡꼢꼣������꼤꼥꼦꼧꼨꼩꼪꼫꼮꼯꼱꼳꼵꼶꼷꼸꼹꼺꼻꼾꽀꽄꽅꽆꽇꽊꽋꽌꽍꽎꽏꽑꽒꽓꽔꽕꽖꽗꽘꽙꽚꽛꽞꽟꽠꽡꽢꽣꽦꽧꽨꽩꽪꽫꽬꽭꽮꽯꽰꽱꽲꽳꽴꽵꽶꽷꽸꽺꽻꽼꽽꽾꽿꾁꾂꾃꾅꾆꾇꾉꾊꾋꾌꾍꾎꾏꾒꾓꾔꾖꾗꾘꾙꾚꾛꾝꾞꾟꾠꾡꾢꾣꾤꾥꾦꾧꾨꾩꾪꾫꾬꾭꾮꾯꾰꾱꾲꾳꾴꾵꾶꾷꾺꾻꾽꾾�".split("");

      for (j = 0; j != D[132].length; ++j) if (D[132][j].charCodeAt(0) !== 0xFFFD) {
        e[D[132][j]] = 33792 + j;
        d[33792 + j] = D[132][j];
      }

      D[133] = "�����������������������������������������������������������������꾿꿁꿂꿃꿄꿅꿆꿊꿌꿏꿐꿑꿒꿓꿕꿖꿗꿘꿙꿚꿛꿝꿞꿟꿠꿡������꿢꿣꿤꿥꿦꿧꿪꿫꿬꿭꿮꿯꿲꿳꿵꿶꿷꿹꿺꿻꿼꿽꿾꿿뀂뀃������뀅뀆뀇뀈뀉뀊뀋뀍뀎뀏뀑뀒뀓뀕뀖뀗뀘뀙뀚뀛뀞뀟뀠뀡뀢뀣뀤뀥뀦뀧뀩뀪뀫뀬뀭뀮뀯뀰뀱뀲뀳뀴뀵뀶뀷뀸뀹뀺뀻뀼뀽뀾뀿끀끁끂끃끆끇끉끋끍끏끐끑끒끖끘끚끛끜끞끟끠끡끢끣끤끥끦끧끨끩끪끫끬끭끮끯끰끱끲끳끴끵끶끷끸끹끺끻끾끿낁낂낃낅낆낇낈낉낊낋낎낐낒낓낔낕낖낗낛낝낞낣낤�".split("");

      for (j = 0; j != D[133].length; ++j) if (D[133][j].charCodeAt(0) !== 0xFFFD) {
        e[D[133][j]] = 34048 + j;
        d[34048 + j] = D[133][j];
      }

      D[134] = "�����������������������������������������������������������������낥낦낧낪낰낲낶낷낹낺낻낽낾낿냀냁냂냃냆냊냋냌냍냎냏냒������냓냕냖냗냙냚냛냜냝냞냟냡냢냣냤냦냧냨냩냪냫냬냭냮냯냰������냱냲냳냴냵냶냷냸냹냺냻냼냽냾냿넀넁넂넃넄넅넆넇넊넍넎넏넑넔넕넖넗넚넞넟넠넡넢넦넧넩넪넫넭넮넯넰넱넲넳넶넺넻넼넽넾넿녂녃녅녆녇녉녊녋녌녍녎녏녒녓녖녗녙녚녛녝녞녟녡녢녣녤녥녦녧녨녩녪녫녬녭녮녯녰녱녲녳녴녵녶녷녺녻녽녾녿놁놃놄놅놆놇놊놌놎놏놐놑놕놖놗놙놚놛놝�".split("");

      for (j = 0; j != D[134].length; ++j) if (D[134][j].charCodeAt(0) !== 0xFFFD) {
        e[D[134][j]] = 34304 + j;
        d[34304 + j] = D[134][j];
      }

      D[135] = "�����������������������������������������������������������������놞놟놠놡놢놣놤놥놦놧놩놪놫놬놭놮놯놰놱놲놳놴놵놶놷놸������놹놺놻놼놽놾놿뇀뇁뇂뇃뇄뇅뇆뇇뇈뇉뇊뇋뇍뇎뇏뇑뇒뇓뇕������뇖뇗뇘뇙뇚뇛뇞뇠뇡뇢뇣뇤뇥뇦뇧뇪뇫뇭뇮뇯뇱뇲뇳뇴뇵뇶뇷뇸뇺뇼뇾뇿눀눁눂눃눆눇눉눊눍눎눏눐눑눒눓눖눘눚눛눜눝눞눟눡눢눣눤눥눦눧눨눩눪눫눬눭눮눯눰눱눲눳눵눶눷눸눹눺눻눽눾눿뉀뉁뉂뉃뉄뉅뉆뉇뉈뉉뉊뉋뉌뉍뉎뉏뉐뉑뉒뉓뉔뉕뉖뉗뉙뉚뉛뉝뉞뉟뉡뉢뉣뉤뉥뉦뉧뉪뉫뉬뉭뉮�".split("");

      for (j = 0; j != D[135].length; ++j) if (D[135][j].charCodeAt(0) !== 0xFFFD) {
        e[D[135][j]] = 34560 + j;
        d[34560 + j] = D[135][j];
      }

      D[136] = "�����������������������������������������������������������������뉯뉰뉱뉲뉳뉶뉷뉸뉹뉺뉻뉽뉾뉿늀늁늂늃늆늇늈늊늋늌늍늎������늏늒늓늕늖늗늛늜늝늞늟늢늤늧늨늩늫늭늮늯늱늲늳늵늶늷������늸늹늺늻늼늽늾늿닀닁닂닃닄닅닆닇닊닋닍닎닏닑닓닔닕닖닗닚닜닞닟닠닡닣닧닩닪닰닱닲닶닼닽닾댂댃댅댆댇댉댊댋댌댍댎댏댒댖댗댘댙댚댛댝댞댟댠댡댢댣댤댥댦댧댨댩댪댫댬댭댮댯댰댱댲댳댴댵댶댷댸댹댺댻댼댽댾댿덀덁덂덃덄덅덆덇덈덉덊덋덌덍덎덏덐덑덒덓덗덙덚덝덠덡덢덣�".split("");

      for (j = 0; j != D[136].length; ++j) if (D[136][j].charCodeAt(0) !== 0xFFFD) {
        e[D[136][j]] = 34816 + j;
        d[34816 + j] = D[136][j];
      }

      D[137] = "�����������������������������������������������������������������덦덨덪덬덭덯덲덳덵덶덷덹덺덻덼덽덾덿뎂뎆뎇뎈뎉뎊뎋뎍������뎎뎏뎑뎒뎓뎕뎖뎗뎘뎙뎚뎛뎜뎝뎞뎟뎢뎣뎤뎥뎦뎧뎩뎪뎫뎭������뎮뎯뎰뎱뎲뎳뎴뎵뎶뎷뎸뎹뎺뎻뎼뎽뎾뎿돀돁돂돃돆돇돉돊돍돏돑돒돓돖돘돚돜돞돟돡돢돣돥돦돧돩돪돫돬돭돮돯돰돱돲돳돴돵돶돷돸돹돺돻돽돾돿됀됁됂됃됄됅됆됇됈됉됊됋됌됍됎됏됑됒됓됔됕됖됗됙됚됛됝됞됟됡됢됣됤됥됦됧됪됬됭됮됯됰됱됲됳됵됶됷됸됹됺됻됼됽됾됿둀둁둂둃둄�".split("");

      for (j = 0; j != D[137].length; ++j) if (D[137][j].charCodeAt(0) !== 0xFFFD) {
        e[D[137][j]] = 35072 + j;
        d[35072 + j] = D[137][j];
      }

      D[138] = "�����������������������������������������������������������������둅둆둇둈둉둊둋둌둍둎둏둒둓둕둖둗둙둚둛둜둝둞둟둢둤둦������둧둨둩둪둫둭둮둯둰둱둲둳둴둵둶둷둸둹둺둻둼둽둾둿뒁뒂������뒃뒄뒅뒆뒇뒉뒊뒋뒌뒍뒎뒏뒐뒑뒒뒓뒔뒕뒖뒗뒘뒙뒚뒛뒜뒞뒟뒠뒡뒢뒣뒥뒦뒧뒩뒪뒫뒭뒮뒯뒰뒱뒲뒳뒴뒶뒸뒺뒻뒼뒽뒾뒿듁듂듃듅듆듇듉듊듋듌듍듎듏듑듒듓듔듖듗듘듙듚듛듞듟듡듢듥듧듨듩듪듫듮듰듲듳듴듵듶듷듹듺듻듼듽듾듿딀딁딂딃딄딅딆딇딈딉딊딋딌딍딎딏딐딑딒딓딖딗딙딚딝�".split("");

      for (j = 0; j != D[138].length; ++j) if (D[138][j].charCodeAt(0) !== 0xFFFD) {
        e[D[138][j]] = 35328 + j;
        d[35328 + j] = D[138][j];
      }

      D[139] = "�����������������������������������������������������������������딞딟딠딡딢딣딦딫딬딭딮딯딲딳딵딶딷딹딺딻딼딽딾딿땂땆������땇땈땉땊땎땏땑땒땓땕땖땗땘땙땚땛땞땢땣땤땥땦땧땨땩땪������땫땬땭땮땯땰땱땲땳땴땵땶땷땸땹땺땻땼땽땾땿떀떁떂떃떄떅떆떇떈떉떊떋떌떍떎떏떐떑떒떓떔떕떖떗떘떙떚떛떜떝떞떟떢떣떥떦떧떩떬떭떮떯떲떶떷떸떹떺떾떿뗁뗂뗃뗅뗆뗇뗈뗉뗊뗋뗎뗒뗓뗔뗕뗖뗗뗙뗚뗛뗜뗝뗞뗟뗠뗡뗢뗣뗤뗥뗦뗧뗨뗩뗪뗫뗭뗮뗯뗰뗱뗲뗳뗴뗵뗶뗷뗸뗹뗺뗻뗼뗽뗾뗿�".split("");

      for (j = 0; j != D[139].length; ++j) if (D[139][j].charCodeAt(0) !== 0xFFFD) {
        e[D[139][j]] = 35584 + j;
        d[35584 + j] = D[139][j];
      }

      D[140] = "�����������������������������������������������������������������똀똁똂똃똄똅똆똇똈똉똊똋똌똍똎똏똒똓똕똖똗똙똚똛똜똝������똞똟똠똡똢똣똤똦똧똨똩똪똫똭똮똯똰똱똲똳똵똶똷똸똹똺������똻똼똽똾똿뙀뙁뙂뙃뙄뙅뙆뙇뙉뙊뙋뙌뙍뙎뙏뙐뙑뙒뙓뙔뙕뙖뙗뙘뙙뙚뙛뙜뙝뙞뙟뙠뙡뙢뙣뙥뙦뙧뙩뙪뙫뙬뙭뙮뙯뙰뙱뙲뙳뙴뙵뙶뙷뙸뙹뙺뙻뙼뙽뙾뙿뚀뚁뚂뚃뚄뚅뚆뚇뚈뚉뚊뚋뚌뚍뚎뚏뚐뚑뚒뚓뚔뚕뚖뚗뚘뚙뚚뚛뚞뚟뚡뚢뚣뚥뚦뚧뚨뚩뚪뚭뚮뚯뚰뚲뚳뚴뚵뚶뚷뚸뚹뚺뚻뚼뚽뚾뚿뛀뛁뛂�".split("");

      for (j = 0; j != D[140].length; ++j) if (D[140][j].charCodeAt(0) !== 0xFFFD) {
        e[D[140][j]] = 35840 + j;
        d[35840 + j] = D[140][j];
      }

      D[141] = "�����������������������������������������������������������������뛃뛄뛅뛆뛇뛈뛉뛊뛋뛌뛍뛎뛏뛐뛑뛒뛓뛕뛖뛗뛘뛙뛚뛛뛜뛝������뛞뛟뛠뛡뛢뛣뛤뛥뛦뛧뛨뛩뛪뛫뛬뛭뛮뛯뛱뛲뛳뛵뛶뛷뛹뛺������뛻뛼뛽뛾뛿뜂뜃뜄뜆뜇뜈뜉뜊뜋뜌뜍뜎뜏뜐뜑뜒뜓뜔뜕뜖뜗뜘뜙뜚뜛뜜뜝뜞뜟뜠뜡뜢뜣뜤뜥뜦뜧뜪뜫뜭뜮뜱뜲뜳뜴뜵뜶뜷뜺뜼뜽뜾뜿띀띁띂띃띅띆띇띉띊띋띍띎띏띐띑띒띓띖띗띘띙띚띛띜띝띞띟띡띢띣띥띦띧띩띪띫띬띭띮띯띲띴띶띷띸띹띺띻띾띿랁랂랃랅랆랇랈랉랊랋랎랓랔랕랚랛랝랞�".split("");

      for (j = 0; j != D[141].length; ++j) if (D[141][j].charCodeAt(0) !== 0xFFFD) {
        e[D[141][j]] = 36096 + j;
        d[36096 + j] = D[141][j];
      }

      D[142] = "�����������������������������������������������������������������랟랡랢랣랤랥랦랧랪랮랯랰랱랲랳랶랷랹랺랻랼랽랾랿럀럁������럂럃럄럅럆럈럊럋럌럍럎럏럐럑럒럓럔럕럖럗럘럙럚럛럜럝������럞럟럠럡럢럣럤럥럦럧럨럩럪럫럮럯럱럲럳럵럶럷럸럹럺럻럾렂렃렄렅렆렊렋렍렎렏렑렒렓렔렕렖렗렚렜렞렟렠렡렢렣렦렧렩렪렫렭렮렯렰렱렲렳렶렺렻렼렽렾렿롁롂롃롅롆롇롈롉롊롋롌롍롎롏롐롒롔롕롖롗롘롙롚롛롞롟롡롢롣롥롦롧롨롩롪롫롮롰롲롳롴롵롶롷롹롺롻롽롾롿뢀뢁뢂뢃뢄�".split("");

      for (j = 0; j != D[142].length; ++j) if (D[142][j].charCodeAt(0) !== 0xFFFD) {
        e[D[142][j]] = 36352 + j;
        d[36352 + j] = D[142][j];
      }

      D[143] = "�����������������������������������������������������������������뢅뢆뢇뢈뢉뢊뢋뢌뢎뢏뢐뢑뢒뢓뢔뢕뢖뢗뢘뢙뢚뢛뢜뢝뢞뢟������뢠뢡뢢뢣뢤뢥뢦뢧뢩뢪뢫뢬뢭뢮뢯뢱뢲뢳뢵뢶뢷뢹뢺뢻뢼뢽������뢾뢿룂룄룆룇룈룉룊룋룍룎룏룑룒룓룕룖룗룘룙룚룛룜룞룠룢룣룤룥룦룧룪룫룭룮룯룱룲룳룴룵룶룷룺룼룾룿뤀뤁뤂뤃뤅뤆뤇뤈뤉뤊뤋뤌뤍뤎뤏뤐뤑뤒뤓뤔뤕뤖뤗뤙뤚뤛뤜뤝뤞뤟뤡뤢뤣뤤뤥뤦뤧뤨뤩뤪뤫뤬뤭뤮뤯뤰뤱뤲뤳뤴뤵뤶뤷뤸뤹뤺뤻뤾뤿륁륂륃륅륆륇륈륉륊륋륍륎륐륒륓륔륕륖륗�".split("");

      for (j = 0; j != D[143].length; ++j) if (D[143][j].charCodeAt(0) !== 0xFFFD) {
        e[D[143][j]] = 36608 + j;
        d[36608 + j] = D[143][j];
      }

      D[144] = "�����������������������������������������������������������������륚륛륝륞륟륡륢륣륤륥륦륧륪륬륮륯륰륱륲륳륶륷륹륺륻륽������륾륿릀릁릂릃릆릈릋릌릏릐릑릒릓릔릕릖릗릘릙릚릛릜릝릞������릟릠릡릢릣릤릥릦릧릨릩릪릫릮릯릱릲릳릵릶릷릸릹릺릻릾맀맂맃맄맅맆맇맊맋맍맓맔맕맖맗맚맜맟맠맢맦맧맩맪맫맭맮맯맰맱맲맳맶맻맼맽맾맿먂먃먄먅먆먇먉먊먋먌먍먎먏먐먑먒먓먔먖먗먘먙먚먛먜먝먞먟먠먡먢먣먤먥먦먧먨먩먪먫먬먭먮먯먰먱먲먳먴먵먶먷먺먻먽먾먿멁멃멄멅멆�".split("");

      for (j = 0; j != D[144].length; ++j) if (D[144][j].charCodeAt(0) !== 0xFFFD) {
        e[D[144][j]] = 36864 + j;
        d[36864 + j] = D[144][j];
      }

      D[145] = "�����������������������������������������������������������������멇멊멌멏멐멑멒멖멗멙멚멛멝멞멟멠멡멢멣멦멪멫멬멭멮멯������멲멳멵멶멷멹멺멻멼멽멾멿몀몁몂몆몈몉몊몋몍몎몏몐몑몒������몓몔몕몖몗몘몙몚몛몜몝몞몟몠몡몢몣몤몥몦몧몪몭몮몯몱몳몴몵몶몷몺몼몾몿뫀뫁뫂뫃뫅뫆뫇뫉뫊뫋뫌뫍뫎뫏뫐뫑뫒뫓뫔뫕뫖뫗뫚뫛뫜뫝뫞뫟뫠뫡뫢뫣뫤뫥뫦뫧뫨뫩뫪뫫뫬뫭뫮뫯뫰뫱뫲뫳뫴뫵뫶뫷뫸뫹뫺뫻뫽뫾뫿묁묂묃묅묆묇묈묉묊묋묌묎묐묒묓묔묕묖묗묙묚묛묝묞묟묡묢묣묤묥묦묧�".split("");

      for (j = 0; j != D[145].length; ++j) if (D[145][j].charCodeAt(0) !== 0xFFFD) {
        e[D[145][j]] = 37120 + j;
        d[37120 + j] = D[145][j];
      }

      D[146] = "�����������������������������������������������������������������묨묪묬묭묮묯묰묱묲묳묷묹묺묿뭀뭁뭂뭃뭆뭈뭊뭋뭌뭎뭑뭒������뭓뭕뭖뭗뭙뭚뭛뭜뭝뭞뭟뭠뭢뭤뭥뭦뭧뭨뭩뭪뭫뭭뭮뭯뭰뭱������뭲뭳뭴뭵뭶뭷뭸뭹뭺뭻뭼뭽뭾뭿뮀뮁뮂뮃뮄뮅뮆뮇뮉뮊뮋뮍뮎뮏뮑뮒뮓뮔뮕뮖뮗뮘뮙뮚뮛뮜뮝뮞뮟뮠뮡뮢뮣뮥뮦뮧뮩뮪뮫뮭뮮뮯뮰뮱뮲뮳뮵뮶뮸뮹뮺뮻뮼뮽뮾뮿믁믂믃믅믆믇믉믊믋믌믍믎믏믑믒믔믕믖믗믘믙믚믛믜믝믞믟믠믡믢믣믤믥믦믧믨믩믪믫믬믭믮믯믰믱믲믳믴믵믶믷믺믻믽믾밁�".split("");

      for (j = 0; j != D[146].length; ++j) if (D[146][j].charCodeAt(0) !== 0xFFFD) {
        e[D[146][j]] = 37376 + j;
        d[37376 + j] = D[146][j];
      }

      D[147] = "�����������������������������������������������������������������밃밄밅밆밇밊밎밐밒밓밙밚밠밡밢밣밦밨밪밫밬밮밯밲밳밵������밶밷밹밺밻밼밽밾밿뱂뱆뱇뱈뱊뱋뱎뱏뱑뱒뱓뱔뱕뱖뱗뱘뱙������뱚뱛뱜뱞뱟뱠뱡뱢뱣뱤뱥뱦뱧뱨뱩뱪뱫뱬뱭뱮뱯뱰뱱뱲뱳뱴뱵뱶뱷뱸뱹뱺뱻뱼뱽뱾뱿벀벁벂벃벆벇벉벊벍벏벐벑벒벓벖벘벛벜벝벞벟벢벣벥벦벩벪벫벬벭벮벯벲벶벷벸벹벺벻벾벿볁볂볃볅볆볇볈볉볊볋볌볎볒볓볔볖볗볙볚볛볝볞볟볠볡볢볣볤볥볦볧볨볩볪볫볬볭볮볯볰볱볲볳볷볹볺볻볽�".split("");

      for (j = 0; j != D[147].length; ++j) if (D[147][j].charCodeAt(0) !== 0xFFFD) {
        e[D[147][j]] = 37632 + j;
        d[37632 + j] = D[147][j];
      }

      D[148] = "�����������������������������������������������������������������볾볿봀봁봂봃봆봈봊봋봌봍봎봏봑봒봓봕봖봗봘봙봚봛봜봝������봞봟봠봡봢봣봥봦봧봨봩봪봫봭봮봯봰봱봲봳봴봵봶봷봸봹������봺봻봼봽봾봿뵁뵂뵃뵄뵅뵆뵇뵊뵋뵍뵎뵏뵑뵒뵓뵔뵕뵖뵗뵚뵛뵜뵝뵞뵟뵠뵡뵢뵣뵥뵦뵧뵩뵪뵫뵬뵭뵮뵯뵰뵱뵲뵳뵴뵵뵶뵷뵸뵹뵺뵻뵼뵽뵾뵿붂붃붅붆붋붌붍붎붏붒붔붖붗붘붛붝붞붟붠붡붢붣붥붦붧붨붩붪붫붬붭붮붯붱붲붳붴붵붶붷붹붺붻붼붽붾붿뷀뷁뷂뷃뷄뷅뷆뷇뷈뷉뷊뷋뷌뷍뷎뷏뷐뷑�".split("");

      for (j = 0; j != D[148].length; ++j) if (D[148][j].charCodeAt(0) !== 0xFFFD) {
        e[D[148][j]] = 37888 + j;
        d[37888 + j] = D[148][j];
      }

      D[149] = "�����������������������������������������������������������������뷒뷓뷖뷗뷙뷚뷛뷝뷞뷟뷠뷡뷢뷣뷤뷥뷦뷧뷨뷪뷫뷬뷭뷮뷯뷱������뷲뷳뷵뷶뷷뷹뷺뷻뷼뷽뷾뷿븁븂븄븆븇븈븉븊븋븎븏븑븒븓������븕븖븗븘븙븚븛븞븠븡븢븣븤븥븦븧븨븩븪븫븬븭븮븯븰븱븲븳븴븵븶븷븸븹븺븻븼븽븾븿빀빁빂빃빆빇빉빊빋빍빏빐빑빒빓빖빘빜빝빞빟빢빣빥빦빧빩빫빬빭빮빯빲빶빷빸빹빺빾빿뺁뺂뺃뺅뺆뺇뺈뺉뺊뺋뺎뺒뺓뺔뺕뺖뺗뺚뺛뺜뺝뺞뺟뺠뺡뺢뺣뺤뺥뺦뺧뺩뺪뺫뺬뺭뺮뺯뺰뺱뺲뺳뺴뺵뺶뺷�".split("");

      for (j = 0; j != D[149].length; ++j) if (D[149][j].charCodeAt(0) !== 0xFFFD) {
        e[D[149][j]] = 38144 + j;
        d[38144 + j] = D[149][j];
      }

      D[150] = "�����������������������������������������������������������������뺸뺹뺺뺻뺼뺽뺾뺿뻀뻁뻂뻃뻄뻅뻆뻇뻈뻉뻊뻋뻌뻍뻎뻏뻒뻓������뻕뻖뻙뻚뻛뻜뻝뻞뻟뻡뻢뻦뻧뻨뻩뻪뻫뻭뻮뻯뻰뻱뻲뻳뻴뻵������뻶뻷뻸뻹뻺뻻뻼뻽뻾뻿뼀뼂뼃뼄뼅뼆뼇뼊뼋뼌뼍뼎뼏뼐뼑뼒뼓뼔뼕뼖뼗뼚뼞뼟뼠뼡뼢뼣뼤뼥뼦뼧뼨뼩뼪뼫뼬뼭뼮뼯뼰뼱뼲뼳뼴뼵뼶뼷뼸뼹뼺뼻뼼뼽뼾뼿뽂뽃뽅뽆뽇뽉뽊뽋뽌뽍뽎뽏뽒뽓뽔뽖뽗뽘뽙뽚뽛뽜뽝뽞뽟뽠뽡뽢뽣뽤뽥뽦뽧뽨뽩뽪뽫뽬뽭뽮뽯뽰뽱뽲뽳뽴뽵뽶뽷뽸뽹뽺뽻뽼뽽뽾뽿뾀뾁뾂�".split("");

      for (j = 0; j != D[150].length; ++j) if (D[150][j].charCodeAt(0) !== 0xFFFD) {
        e[D[150][j]] = 38400 + j;
        d[38400 + j] = D[150][j];
      }

      D[151] = "�����������������������������������������������������������������뾃뾄뾅뾆뾇뾈뾉뾊뾋뾌뾍뾎뾏뾐뾑뾒뾓뾕뾖뾗뾘뾙뾚뾛뾜뾝������뾞뾟뾠뾡뾢뾣뾤뾥뾦뾧뾨뾩뾪뾫뾬뾭뾮뾯뾱뾲뾳뾴뾵뾶뾷뾸������뾹뾺뾻뾼뾽뾾뾿뿀뿁뿂뿃뿄뿆뿇뿈뿉뿊뿋뿎뿏뿑뿒뿓뿕뿖뿗뿘뿙뿚뿛뿝뿞뿠뿢뿣뿤뿥뿦뿧뿨뿩뿪뿫뿬뿭뿮뿯뿰뿱뿲뿳뿴뿵뿶뿷뿸뿹뿺뿻뿼뿽뿾뿿쀀쀁쀂쀃쀄쀅쀆쀇쀈쀉쀊쀋쀌쀍쀎쀏쀐쀑쀒쀓쀔쀕쀖쀗쀘쀙쀚쀛쀜쀝쀞쀟쀠쀡쀢쀣쀤쀥쀦쀧쀨쀩쀪쀫쀬쀭쀮쀯쀰쀱쀲쀳쀴쀵쀶쀷쀸쀹쀺쀻쀽쀾쀿�".split("");

      for (j = 0; j != D[151].length; ++j) if (D[151][j].charCodeAt(0) !== 0xFFFD) {
        e[D[151][j]] = 38656 + j;
        d[38656 + j] = D[151][j];
      }

      D[152] = "�����������������������������������������������������������������쁀쁁쁂쁃쁄쁅쁆쁇쁈쁉쁊쁋쁌쁍쁎쁏쁐쁒쁓쁔쁕쁖쁗쁙쁚쁛������쁝쁞쁟쁡쁢쁣쁤쁥쁦쁧쁪쁫쁬쁭쁮쁯쁰쁱쁲쁳쁴쁵쁶쁷쁸쁹������쁺쁻쁼쁽쁾쁿삀삁삂삃삄삅삆삇삈삉삊삋삌삍삎삏삒삓삕삖삗삙삚삛삜삝삞삟삢삤삦삧삨삩삪삫삮삱삲삷삸삹삺삻삾샂샃샄샆샇샊샋샍샎샏샑샒샓샔샕샖샗샚샞샟샠샡샢샣샦샧샩샪샫샭샮샯샰샱샲샳샶샸샺샻샼샽샾샿섁섂섃섅섆섇섉섊섋섌섍섎섏섑섒섓섔섖섗섘섙섚섛섡섢섥섨섩섪섫섮�".split("");

      for (j = 0; j != D[152].length; ++j) if (D[152][j].charCodeAt(0) !== 0xFFFD) {
        e[D[152][j]] = 38912 + j;
        d[38912 + j] = D[152][j];
      }

      D[153] = "�����������������������������������������������������������������섲섳섴섵섷섺섻섽섾섿셁셂셃셄셅셆셇셊셎셏셐셑셒셓셖셗������셙셚셛셝셞셟셠셡셢셣셦셪셫셬셭셮셯셱셲셳셵셶셷셹셺셻������셼셽셾셿솀솁솂솃솄솆솇솈솉솊솋솏솑솒솓솕솗솘솙솚솛솞솠솢솣솤솦솧솪솫솭솮솯솱솲솳솴솵솶솷솸솹솺솻솼솾솿쇀쇁쇂쇃쇅쇆쇇쇉쇊쇋쇍쇎쇏쇐쇑쇒쇓쇕쇖쇙쇚쇛쇜쇝쇞쇟쇡쇢쇣쇥쇦쇧쇩쇪쇫쇬쇭쇮쇯쇲쇴쇵쇶쇷쇸쇹쇺쇻쇾쇿숁숂숃숅숆숇숈숉숊숋숎숐숒숓숔숕숖숗숚숛숝숞숡숢숣�".split("");

      for (j = 0; j != D[153].length; ++j) if (D[153][j].charCodeAt(0) !== 0xFFFD) {
        e[D[153][j]] = 39168 + j;
        d[39168 + j] = D[153][j];
      }

      D[154] = "�����������������������������������������������������������������숤숥숦숧숪숬숮숰숳숵숶숷숸숹숺숻숼숽숾숿쉀쉁쉂쉃쉄쉅������쉆쉇쉉쉊쉋쉌쉍쉎쉏쉒쉓쉕쉖쉗쉙쉚쉛쉜쉝쉞쉟쉡쉢쉣쉤쉦������쉧쉨쉩쉪쉫쉮쉯쉱쉲쉳쉵쉶쉷쉸쉹쉺쉻쉾슀슂슃슄슅슆슇슊슋슌슍슎슏슑슒슓슔슕슖슗슙슚슜슞슟슠슡슢슣슦슧슩슪슫슮슯슰슱슲슳슶슸슺슻슼슽슾슿싀싁싂싃싄싅싆싇싈싉싊싋싌싍싎싏싐싑싒싓싔싕싖싗싘싙싚싛싞싟싡싢싥싦싧싨싩싪싮싰싲싳싴싵싷싺싽싾싿쌁쌂쌃쌄쌅쌆쌇쌊쌋쌎쌏�".split("");

      for (j = 0; j != D[154].length; ++j) if (D[154][j].charCodeAt(0) !== 0xFFFD) {
        e[D[154][j]] = 39424 + j;
        d[39424 + j] = D[154][j];
      }

      D[155] = "�����������������������������������������������������������������쌐쌑쌒쌖쌗쌙쌚쌛쌝쌞쌟쌠쌡쌢쌣쌦쌧쌪쌫쌬쌭쌮쌯쌰쌱쌲������쌳쌴쌵쌶쌷쌸쌹쌺쌻쌼쌽쌾쌿썀썁썂썃썄썆썇썈썉썊썋썌썍������썎썏썐썑썒썓썔썕썖썗썘썙썚썛썜썝썞썟썠썡썢썣썤썥썦썧썪썫썭썮썯썱썳썴썵썶썷썺썻썾썿쎀쎁쎂쎃쎅쎆쎇쎉쎊쎋쎍쎎쎏쎐쎑쎒쎓쎔쎕쎖쎗쎘쎙쎚쎛쎜쎝쎞쎟쎠쎡쎢쎣쎤쎥쎦쎧쎨쎩쎪쎫쎬쎭쎮쎯쎰쎱쎲쎳쎴쎵쎶쎷쎸쎹쎺쎻쎼쎽쎾쎿쏁쏂쏃쏄쏅쏆쏇쏈쏉쏊쏋쏌쏍쏎쏏쏐쏑쏒쏓쏔쏕쏖쏗쏚�".split("");

      for (j = 0; j != D[155].length; ++j) if (D[155][j].charCodeAt(0) !== 0xFFFD) {
        e[D[155][j]] = 39680 + j;
        d[39680 + j] = D[155][j];
      }

      D[156] = "�����������������������������������������������������������������쏛쏝쏞쏡쏣쏤쏥쏦쏧쏪쏫쏬쏮쏯쏰쏱쏲쏳쏶쏷쏹쏺쏻쏼쏽쏾������쏿쐀쐁쐂쐃쐄쐅쐆쐇쐉쐊쐋쐌쐍쐎쐏쐑쐒쐓쐔쐕쐖쐗쐘쐙쐚������쐛쐜쐝쐞쐟쐠쐡쐢쐣쐥쐦쐧쐨쐩쐪쐫쐭쐮쐯쐱쐲쐳쐵쐶쐷쐸쐹쐺쐻쐾쐿쑀쑁쑂쑃쑄쑅쑆쑇쑉쑊쑋쑌쑍쑎쑏쑐쑑쑒쑓쑔쑕쑖쑗쑘쑙쑚쑛쑜쑝쑞쑟쑠쑡쑢쑣쑦쑧쑩쑪쑫쑭쑮쑯쑰쑱쑲쑳쑶쑷쑸쑺쑻쑼쑽쑾쑿쒁쒂쒃쒄쒅쒆쒇쒈쒉쒊쒋쒌쒍쒎쒏쒐쒑쒒쒓쒕쒖쒗쒘쒙쒚쒛쒝쒞쒟쒠쒡쒢쒣쒤쒥쒦쒧쒨쒩�".split("");

      for (j = 0; j != D[156].length; ++j) if (D[156][j].charCodeAt(0) !== 0xFFFD) {
        e[D[156][j]] = 39936 + j;
        d[39936 + j] = D[156][j];
      }

      D[157] = "�����������������������������������������������������������������쒪쒫쒬쒭쒮쒯쒰쒱쒲쒳쒴쒵쒶쒷쒹쒺쒻쒽쒾쒿쓀쓁쓂쓃쓄쓅������쓆쓇쓈쓉쓊쓋쓌쓍쓎쓏쓐쓑쓒쓓쓔쓕쓖쓗쓘쓙쓚쓛쓜쓝쓞쓟������쓠쓡쓢쓣쓤쓥쓦쓧쓨쓪쓫쓬쓭쓮쓯쓲쓳쓵쓶쓷쓹쓻쓼쓽쓾씂씃씄씅씆씇씈씉씊씋씍씎씏씑씒씓씕씖씗씘씙씚씛씝씞씟씠씡씢씣씤씥씦씧씪씫씭씮씯씱씲씳씴씵씶씷씺씼씾씿앀앁앂앃앆앇앋앏앐앑앒앖앚앛앜앟앢앣앥앦앧앩앪앫앬앭앮앯앲앶앷앸앹앺앻앾앿얁얂얃얅얆얈얉얊얋얎얐얒얓얔�".split("");

      for (j = 0; j != D[157].length; ++j) if (D[157][j].charCodeAt(0) !== 0xFFFD) {
        e[D[157][j]] = 40192 + j;
        d[40192 + j] = D[157][j];
      }

      D[158] = "�����������������������������������������������������������������얖얙얚얛얝얞얟얡얢얣얤얥얦얧얨얪얫얬얭얮얯얰얱얲얳얶������얷얺얿엀엁엂엃엋엍엏엒엓엕엖엗엙엚엛엜엝엞엟엢엤엦엧������엨엩엪엫엯엱엲엳엵엸엹엺엻옂옃옄옉옊옋옍옎옏옑옒옓옔옕옖옗옚옝옞옟옠옡옢옣옦옧옩옪옫옯옱옲옶옸옺옼옽옾옿왂왃왅왆왇왉왊왋왌왍왎왏왒왖왗왘왙왚왛왞왟왡왢왣왤왥왦왧왨왩왪왫왭왮왰왲왳왴왵왶왷왺왻왽왾왿욁욂욃욄욅욆욇욊욌욎욏욐욑욒욓욖욗욙욚욛욝욞욟욠욡욢욣욦�".split("");

      for (j = 0; j != D[158].length; ++j) if (D[158][j].charCodeAt(0) !== 0xFFFD) {
        e[D[158][j]] = 40448 + j;
        d[40448 + j] = D[158][j];
      }

      D[159] = "�����������������������������������������������������������������욨욪욫욬욭욮욯욲욳욵욶욷욻욼욽욾욿웂웄웆웇웈웉웊웋웎������웏웑웒웓웕웖웗웘웙웚웛웞웟웢웣웤웥웦웧웪웫웭웮웯웱웲������웳웴웵웶웷웺웻웼웾웿윀윁윂윃윆윇윉윊윋윍윎윏윐윑윒윓윖윘윚윛윜윝윞윟윢윣윥윦윧윩윪윫윬윭윮윯윲윴윶윸윹윺윻윾윿읁읂읃읅읆읇읈읉읋읎읐읙읚읛읝읞읟읡읢읣읤읥읦읧읩읪읬읭읮읯읰읱읲읳읶읷읹읺읻읿잀잁잂잆잋잌잍잏잒잓잕잙잛잜잝잞잟잢잧잨잩잪잫잮잯잱잲잳잵잶잷�".split("");

      for (j = 0; j != D[159].length; ++j) if (D[159][j].charCodeAt(0) !== 0xFFFD) {
        e[D[159][j]] = 40704 + j;
        d[40704 + j] = D[159][j];
      }

      D[160] = "�����������������������������������������������������������������잸잹잺잻잾쟂쟃쟄쟅쟆쟇쟊쟋쟍쟏쟑쟒쟓쟔쟕쟖쟗쟙쟚쟛쟜������쟞쟟쟠쟡쟢쟣쟥쟦쟧쟩쟪쟫쟭쟮쟯쟰쟱쟲쟳쟴쟵쟶쟷쟸쟹쟺������쟻쟼쟽쟾쟿젂젃젅젆젇젉젋젌젍젎젏젒젔젗젘젙젚젛젞젟젡젢젣젥젦젧젨젩젪젫젮젰젲젳젴젵젶젷젹젺젻젽젾젿졁졂졃졄졅졆졇졊졋졎졏졐졑졒졓졕졖졗졘졙졚졛졜졝졞졟졠졡졢졣졤졥졦졧졨졩졪졫졬졭졮졯졲졳졵졶졷졹졻졼졽졾졿좂좄좈좉좊좎좏좐좑좒좓좕좖좗좘좙좚좛좜좞좠좢좣좤�".split("");

      for (j = 0; j != D[160].length; ++j) if (D[160][j].charCodeAt(0) !== 0xFFFD) {
        e[D[160][j]] = 40960 + j;
        d[40960 + j] = D[160][j];
      }

      D[161] = "�����������������������������������������������������������������좥좦좧좩좪좫좬좭좮좯좰좱좲좳좴좵좶좷좸좹좺좻좾좿죀죁������죂죃죅죆죇죉죊죋죍죎죏죐죑죒죓죖죘죚죛죜죝죞죟죢죣죥������죦죧죨죩죪죫죬죭죮죯죰죱죲죳죴죶죷죸죹죺죻죾죿줁줂줃줇줈줉줊줋줎　、。·‥…¨〃­―∥＼∼‘’“”〔〕〈〉《》「」『』【】±×÷≠≤≥∞∴°′″℃Å￠￡￥♂♀∠⊥⌒∂∇≡≒§※☆★○●◎◇◆□■△▲▽▼→←↑↓↔〓≪≫√∽∝∵∫∬∈∋⊆⊇⊂⊃∪∩∧∨￢�".split("");

      for (j = 0; j != D[161].length; ++j) if (D[161][j].charCodeAt(0) !== 0xFFFD) {
        e[D[161][j]] = 41216 + j;
        d[41216 + j] = D[161][j];
      }

      D[162] = "�����������������������������������������������������������������줐줒줓줔줕줖줗줙줚줛줜줝줞줟줠줡줢줣줤줥줦줧줨줩줪줫������줭줮줯줰줱줲줳줵줶줷줸줹줺줻줼줽줾줿쥀쥁쥂쥃쥄쥅쥆쥇������쥈쥉쥊쥋쥌쥍쥎쥏쥒쥓쥕쥖쥗쥙쥚쥛쥜쥝쥞쥟쥢쥤쥥쥦쥧쥨쥩쥪쥫쥭쥮쥯⇒⇔∀∃´～ˇ˘˝˚˙¸˛¡¿ː∮∑∏¤℉‰◁◀▷▶♤♠♡♥♧♣⊙◈▣◐◑▒▤▥▨▧▦▩♨☏☎☜☞¶†‡↕↗↙↖↘♭♩♪♬㉿㈜№㏇™㏂㏘℡€®������������������������".split("");

      for (j = 0; j != D[162].length; ++j) if (D[162][j].charCodeAt(0) !== 0xFFFD) {
        e[D[162][j]] = 41472 + j;
        d[41472 + j] = D[162][j];
      }

      D[163] = "�����������������������������������������������������������������쥱쥲쥳쥵쥶쥷쥸쥹쥺쥻쥽쥾쥿즀즁즂즃즄즅즆즇즊즋즍즎즏������즑즒즓즔즕즖즗즚즜즞즟즠즡즢즣즤즥즦즧즨즩즪즫즬즭즮������즯즰즱즲즳즴즵즶즷즸즹즺즻즼즽즾즿짂짃짅짆짉짋짌짍짎짏짒짔짗짘짛！＂＃＄％＆＇（）＊＋，－．／０１２３４５６７８９：；＜＝＞？＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ［￦］＾＿｀ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ｛｜｝￣�".split("");

      for (j = 0; j != D[163].length; ++j) if (D[163][j].charCodeAt(0) !== 0xFFFD) {
        e[D[163][j]] = 41728 + j;
        d[41728 + j] = D[163][j];
      }

      D[164] = "�����������������������������������������������������������������짞짟짡짣짥짦짨짩짪짫짮짲짳짴짵짶짷짺짻짽짾짿쨁쨂쨃쨄������쨅쨆쨇쨊쨎쨏쨐쨑쨒쨓쨕쨖쨗쨙쨚쨛쨜쨝쨞쨟쨠쨡쨢쨣쨤쨥������쨦쨧쨨쨪쨫쨬쨭쨮쨯쨰쨱쨲쨳쨴쨵쨶쨷쨸쨹쨺쨻쨼쨽쨾쨿쩀쩁쩂쩃쩄쩅쩆ㄱㄲㄳㄴㄵㄶㄷㄸㄹㄺㄻㄼㄽㄾㄿㅀㅁㅂㅃㅄㅅㅆㅇㅈㅉㅊㅋㅌㅍㅎㅏㅐㅑㅒㅓㅔㅕㅖㅗㅘㅙㅚㅛㅜㅝㅞㅟㅠㅡㅢㅣㅤㅥㅦㅧㅨㅩㅪㅫㅬㅭㅮㅯㅰㅱㅲㅳㅴㅵㅶㅷㅸㅹㅺㅻㅼㅽㅾㅿㆀㆁㆂㆃㆄㆅㆆㆇㆈㆉㆊㆋㆌㆍㆎ�".split("");

      for (j = 0; j != D[164].length; ++j) if (D[164][j].charCodeAt(0) !== 0xFFFD) {
        e[D[164][j]] = 41984 + j;
        d[41984 + j] = D[164][j];
      }

      D[165] = "�����������������������������������������������������������������쩇쩈쩉쩊쩋쩎쩏쩑쩒쩓쩕쩖쩗쩘쩙쩚쩛쩞쩢쩣쩤쩥쩦쩧쩩쩪������쩫쩬쩭쩮쩯쩰쩱쩲쩳쩴쩵쩶쩷쩸쩹쩺쩻쩼쩾쩿쪀쪁쪂쪃쪅쪆������쪇쪈쪉쪊쪋쪌쪍쪎쪏쪐쪑쪒쪓쪔쪕쪖쪗쪙쪚쪛쪜쪝쪞쪟쪠쪡쪢쪣쪤쪥쪦쪧ⅰⅱⅲⅳⅴⅵⅶⅷⅸⅹ�����ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ�������ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ��������αβγδεζηθικλμνξοπρστυφχψω�������".split("");

      for (j = 0; j != D[165].length; ++j) if (D[165][j].charCodeAt(0) !== 0xFFFD) {
        e[D[165][j]] = 42240 + j;
        d[42240 + j] = D[165][j];
      }

      D[166] = "�����������������������������������������������������������������쪨쪩쪪쪫쪬쪭쪮쪯쪰쪱쪲쪳쪴쪵쪶쪷쪸쪹쪺쪻쪾쪿쫁쫂쫃쫅������쫆쫇쫈쫉쫊쫋쫎쫐쫒쫔쫕쫖쫗쫚쫛쫜쫝쫞쫟쫡쫢쫣쫤쫥쫦쫧������쫨쫩쫪쫫쫭쫮쫯쫰쫱쫲쫳쫵쫶쫷쫸쫹쫺쫻쫼쫽쫾쫿쬀쬁쬂쬃쬄쬅쬆쬇쬉쬊─│┌┐┘└├┬┤┴┼━┃┏┓┛┗┣┳┫┻╋┠┯┨┷┿┝┰┥┸╂┒┑┚┙┖┕┎┍┞┟┡┢┦┧┩┪┭┮┱┲┵┶┹┺┽┾╀╁╃╄╅╆╇╈╉╊���������������������������".split("");

      for (j = 0; j != D[166].length; ++j) if (D[166][j].charCodeAt(0) !== 0xFFFD) {
        e[D[166][j]] = 42496 + j;
        d[42496 + j] = D[166][j];
      }

      D[167] = "�����������������������������������������������������������������쬋쬌쬍쬎쬏쬑쬒쬓쬕쬖쬗쬙쬚쬛쬜쬝쬞쬟쬢쬣쬤쬥쬦쬧쬨쬩������쬪쬫쬬쬭쬮쬯쬰쬱쬲쬳쬴쬵쬶쬷쬸쬹쬺쬻쬼쬽쬾쬿쭀쭂쭃쭄������쭅쭆쭇쭊쭋쭍쭎쭏쭑쭒쭓쭔쭕쭖쭗쭚쭛쭜쭞쭟쭠쭡쭢쭣쭥쭦쭧쭨쭩쭪쭫쭬㎕㎖㎗ℓ㎘㏄㎣㎤㎥㎦㎙㎚㎛㎜㎝㎞㎟㎠㎡㎢㏊㎍㎎㎏㏏㎈㎉㏈㎧㎨㎰㎱㎲㎳㎴㎵㎶㎷㎸㎹㎀㎁㎂㎃㎄㎺㎻㎼㎽㎾㎿㎐㎑㎒㎓㎔Ω㏀㏁㎊㎋㎌㏖㏅㎭㎮㎯㏛㎩㎪㎫㎬㏝㏐㏓㏃㏉㏜㏆����������������".split("");

      for (j = 0; j != D[167].length; ++j) if (D[167][j].charCodeAt(0) !== 0xFFFD) {
        e[D[167][j]] = 42752 + j;
        d[42752 + j] = D[167][j];
      }

      D[168] = "�����������������������������������������������������������������쭭쭮쭯쭰쭱쭲쭳쭴쭵쭶쭷쭺쭻쭼쭽쭾쭿쮀쮁쮂쮃쮄쮅쮆쮇쮈������쮉쮊쮋쮌쮍쮎쮏쮐쮑쮒쮓쮔쮕쮖쮗쮘쮙쮚쮛쮝쮞쮟쮠쮡쮢쮣������쮤쮥쮦쮧쮨쮩쮪쮫쮬쮭쮮쮯쮰쮱쮲쮳쮴쮵쮶쮷쮹쮺쮻쮼쮽쮾쮿쯀쯁쯂쯃쯄ÆÐªĦ�Ĳ�ĿŁØŒºÞŦŊ�㉠㉡㉢㉣㉤㉥㉦㉧㉨㉩㉪㉫㉬㉭㉮㉯㉰㉱㉲㉳㉴㉵㉶㉷㉸㉹㉺㉻ⓐⓑⓒⓓⓔⓕⓖⓗⓘⓙⓚⓛⓜⓝⓞⓟⓠⓡⓢⓣⓤⓥⓦⓧⓨⓩ①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮½⅓⅔¼¾⅛⅜⅝⅞�".split("");

      for (j = 0; j != D[168].length; ++j) if (D[168][j].charCodeAt(0) !== 0xFFFD) {
        e[D[168][j]] = 43008 + j;
        d[43008 + j] = D[168][j];
      }

      D[169] = "�����������������������������������������������������������������쯅쯆쯇쯈쯉쯊쯋쯌쯍쯎쯏쯐쯑쯒쯓쯕쯖쯗쯘쯙쯚쯛쯜쯝쯞쯟������쯠쯡쯢쯣쯥쯦쯨쯪쯫쯬쯭쯮쯯쯰쯱쯲쯳쯴쯵쯶쯷쯸쯹쯺쯻쯼������쯽쯾쯿찀찁찂찃찄찅찆찇찈찉찊찋찎찏찑찒찓찕찖찗찘찙찚찛찞찟찠찣찤æđðħıĳĸŀłøœßþŧŋŉ㈀㈁㈂㈃㈄㈅㈆㈇㈈㈉㈊㈋㈌㈍㈎㈏㈐㈑㈒㈓㈔㈕㈖㈗㈘㈙㈚㈛⒜⒝⒞⒟⒠⒡⒢⒣⒤⒥⒦⒧⒨⒩⒪⒫⒬⒭⒮⒯⒰⒱⒲⒳⒴⒵⑴⑵⑶⑷⑸⑹⑺⑻⑼⑽⑾⑿⒀⒁⒂¹²³⁴ⁿ₁₂₃₄�".split("");

      for (j = 0; j != D[169].length; ++j) if (D[169][j].charCodeAt(0) !== 0xFFFD) {
        e[D[169][j]] = 43264 + j;
        d[43264 + j] = D[169][j];
      }

      D[170] = "�����������������������������������������������������������������찥찦찪찫찭찯찱찲찳찴찵찶찷찺찿챀챁챂챃챆챇챉챊챋챍챎������챏챐챑챒챓챖챚챛챜챝챞챟챡챢챣챥챧챩챪챫챬챭챮챯챱챲������챳챴챶챷챸챹챺챻챼챽챾챿첀첁첂첃첄첅첆첇첈첉첊첋첌첍첎첏첐첑첒첓ぁあぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとどなにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをん������������".split("");

      for (j = 0; j != D[170].length; ++j) if (D[170][j].charCodeAt(0) !== 0xFFFD) {
        e[D[170][j]] = 43520 + j;
        d[43520 + j] = D[170][j];
      }

      D[171] = "�����������������������������������������������������������������첔첕첖첗첚첛첝첞첟첡첢첣첤첥첦첧첪첮첯첰첱첲첳첶첷첹������첺첻첽첾첿쳀쳁쳂쳃쳆쳈쳊쳋쳌쳍쳎쳏쳑쳒쳓쳕쳖쳗쳘쳙쳚������쳛쳜쳝쳞쳟쳠쳡쳢쳣쳥쳦쳧쳨쳩쳪쳫쳭쳮쳯쳱쳲쳳쳴쳵쳶쳷쳸쳹쳺쳻쳼쳽ァアィイゥウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトドナニヌネノハバパヒビピフブプヘベペホボポマミムメモャヤュユョヨラリルレロヮワヰヱヲンヴヵヶ���������".split("");

      for (j = 0; j != D[171].length; ++j) if (D[171][j].charCodeAt(0) !== 0xFFFD) {
        e[D[171][j]] = 43776 + j;
        d[43776 + j] = D[171][j];
      }

      D[172] = "�����������������������������������������������������������������쳾쳿촀촂촃촄촅촆촇촊촋촍촎촏촑촒촓촔촕촖촗촚촜촞촟촠������촡촢촣촥촦촧촩촪촫촭촮촯촰촱촲촳촴촵촶촷촸촺촻촼촽촾������촿쵀쵁쵂쵃쵄쵅쵆쵇쵈쵉쵊쵋쵌쵍쵎쵏쵐쵑쵒쵓쵔쵕쵖쵗쵘쵙쵚쵛쵝쵞쵟АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ���������������абвгдеёжзийклмнопрстуфхцчшщъыьэюя��������������".split("");

      for (j = 0; j != D[172].length; ++j) if (D[172][j].charCodeAt(0) !== 0xFFFD) {
        e[D[172][j]] = 44032 + j;
        d[44032 + j] = D[172][j];
      }

      D[173] = "�����������������������������������������������������������������쵡쵢쵣쵥쵦쵧쵨쵩쵪쵫쵮쵰쵲쵳쵴쵵쵶쵷쵹쵺쵻쵼쵽쵾쵿춀������춁춂춃춄춅춆춇춉춊춋춌춍춎춏춐춑춒춓춖춗춙춚춛춝춞춟������춠춡춢춣춦춨춪춫춬춭춮춯춱춲춳춴춵춶춷춸춹춺춻춼춽춾춿췀췁췂췃췅�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[173].length; ++j) if (D[173][j].charCodeAt(0) !== 0xFFFD) {
        e[D[173][j]] = 44288 + j;
        d[44288 + j] = D[173][j];
      }

      D[174] = "�����������������������������������������������������������������췆췇췈췉췊췋췍췎췏췑췒췓췔췕췖췗췘췙췚췛췜췝췞췟췠췡������췢췣췤췥췦췧췩췪췫췭췮췯췱췲췳췴췵췶췷췺췼췾췿츀츁츂������츃츅츆츇츉츊츋츍츎츏츐츑츒츓츕츖츗츘츚츛츜츝츞츟츢츣츥츦츧츩츪츫�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[174].length; ++j) if (D[174][j].charCodeAt(0) !== 0xFFFD) {
        e[D[174][j]] = 44544 + j;
        d[44544 + j] = D[174][j];
      }

      D[175] = "�����������������������������������������������������������������츬츭츮츯츲츴츶츷츸츹츺츻츼츽츾츿칀칁칂칃칄칅칆칇칈칉������칊칋칌칍칎칏칐칑칒칓칔칕칖칗칚칛칝칞칢칣칤칥칦칧칪칬������칮칯칰칱칲칳칶칷칹칺칻칽칾칿캀캁캂캃캆캈캊캋캌캍캎캏캒캓캕캖캗캙�����������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[175].length; ++j) if (D[175][j].charCodeAt(0) !== 0xFFFD) {
        e[D[175][j]] = 44800 + j;
        d[44800 + j] = D[175][j];
      }

      D[176] = "�����������������������������������������������������������������캚캛캜캝캞캟캢캦캧캨캩캪캫캮캯캰캱캲캳캴캵캶캷캸캹캺������캻캼캽캾캿컀컂컃컄컅컆컇컈컉컊컋컌컍컎컏컐컑컒컓컔컕������컖컗컘컙컚컛컜컝컞컟컠컡컢컣컦컧컩컪컭컮컯컰컱컲컳컶컺컻컼컽컾컿가각간갇갈갉갊감갑값갓갔강갖갗같갚갛개객갠갤갬갭갯갰갱갸갹갼걀걋걍걔걘걜거걱건걷걸걺검겁것겄겅겆겉겊겋게겐겔겜겝겟겠겡겨격겪견겯결겸겹겻겼경곁계곈곌곕곗고곡곤곧골곪곬곯곰곱곳공곶과곽관괄괆�".split("");

      for (j = 0; j != D[176].length; ++j) if (D[176][j].charCodeAt(0) !== 0xFFFD) {
        e[D[176][j]] = 45056 + j;
        d[45056 + j] = D[176][j];
      }

      D[177] = "�����������������������������������������������������������������켂켃켅켆켇켉켊켋켌켍켎켏켒켔켖켗켘켙켚켛켝켞켟켡켢켣������켥켦켧켨켩켪켫켮켲켳켴켵켶켷켹켺켻켼켽켾켿콀콁콂콃콄������콅콆콇콈콉콊콋콌콍콎콏콐콑콒콓콖콗콙콚콛콝콞콟콠콡콢콣콦콨콪콫콬괌괍괏광괘괜괠괩괬괭괴괵괸괼굄굅굇굉교굔굘굡굣구국군굳굴굵굶굻굼굽굿궁궂궈궉권궐궜궝궤궷귀귁귄귈귐귑귓규균귤그극근귿글긁금급긋긍긔기긱긴긷길긺김깁깃깅깆깊까깍깎깐깔깖깜깝깟깠깡깥깨깩깬깰깸�".split("");

      for (j = 0; j != D[177].length; ++j) if (D[177][j].charCodeAt(0) !== 0xFFFD) {
        e[D[177][j]] = 45312 + j;
        d[45312 + j] = D[177][j];
      }

      D[178] = "�����������������������������������������������������������������콭콮콯콲콳콵콶콷콹콺콻콼콽콾콿쾁쾂쾃쾄쾆쾇쾈쾉쾊쾋쾍������쾎쾏쾐쾑쾒쾓쾔쾕쾖쾗쾘쾙쾚쾛쾜쾝쾞쾟쾠쾢쾣쾤쾥쾦쾧쾩������쾪쾫쾬쾭쾮쾯쾱쾲쾳쾴쾵쾶쾷쾸쾹쾺쾻쾼쾽쾾쾿쿀쿁쿂쿃쿅쿆쿇쿈쿉쿊쿋깹깻깼깽꺄꺅꺌꺼꺽꺾껀껄껌껍껏껐껑께껙껜껨껫껭껴껸껼꼇꼈꼍꼐꼬꼭꼰꼲꼴꼼꼽꼿꽁꽂꽃꽈꽉꽐꽜꽝꽤꽥꽹꾀꾄꾈꾐꾑꾕꾜꾸꾹꾼꿀꿇꿈꿉꿋꿍꿎꿔꿜꿨꿩꿰꿱꿴꿸뀀뀁뀄뀌뀐뀔뀜뀝뀨끄끅끈끊끌끎끓끔끕끗끙�".split("");

      for (j = 0; j != D[178].length; ++j) if (D[178][j].charCodeAt(0) !== 0xFFFD) {
        e[D[178][j]] = 45568 + j;
        d[45568 + j] = D[178][j];
      }

      D[179] = "�����������������������������������������������������������������쿌쿍쿎쿏쿐쿑쿒쿓쿔쿕쿖쿗쿘쿙쿚쿛쿜쿝쿞쿟쿢쿣쿥쿦쿧쿩������쿪쿫쿬쿭쿮쿯쿲쿴쿶쿷쿸쿹쿺쿻쿽쿾쿿퀁퀂퀃퀅퀆퀇퀈퀉퀊������퀋퀌퀍퀎퀏퀐퀒퀓퀔퀕퀖퀗퀙퀚퀛퀜퀝퀞퀟퀠퀡퀢퀣퀤퀥퀦퀧퀨퀩퀪퀫퀬끝끼끽낀낄낌낍낏낑나낙낚난낟날낡낢남납낫났낭낮낯낱낳내낵낸낼냄냅냇냈냉냐냑냔냘냠냥너넉넋넌널넒넓넘넙넛넜넝넣네넥넨넬넴넵넷넸넹녀녁년녈념녑녔녕녘녜녠노녹논놀놂놈놉놋농높놓놔놘놜놨뇌뇐뇔뇜뇝�".split("");

      for (j = 0; j != D[179].length; ++j) if (D[179][j].charCodeAt(0) !== 0xFFFD) {
        e[D[179][j]] = 45824 + j;
        d[45824 + j] = D[179][j];
      }

      D[180] = "�����������������������������������������������������������������퀮퀯퀰퀱퀲퀳퀶퀷퀹퀺퀻퀽퀾퀿큀큁큂큃큆큈큊큋큌큍큎큏������큑큒큓큕큖큗큙큚큛큜큝큞큟큡큢큣큤큥큦큧큨큩큪큫큮큯������큱큲큳큵큶큷큸큹큺큻큾큿킀킂킃킄킅킆킇킈킉킊킋킌킍킎킏킐킑킒킓킔뇟뇨뇩뇬뇰뇹뇻뇽누눅눈눋눌눔눕눗눙눠눴눼뉘뉜뉠뉨뉩뉴뉵뉼늄늅늉느늑는늘늙늚늠늡늣능늦늪늬늰늴니닉닌닐닒님닙닛닝닢다닥닦단닫달닭닮닯닳담답닷닸당닺닻닿대댁댄댈댐댑댓댔댕댜더덕덖던덛덜덞덟덤덥�".split("");

      for (j = 0; j != D[180].length; ++j) if (D[180][j].charCodeAt(0) !== 0xFFFD) {
        e[D[180][j]] = 46080 + j;
        d[46080 + j] = D[180][j];
      }

      D[181] = "�����������������������������������������������������������������킕킖킗킘킙킚킛킜킝킞킟킠킡킢킣킦킧킩킪킫킭킮킯킰킱킲������킳킶킸킺킻킼킽킾킿탂탃탅탆탇탊탋탌탍탎탏탒탖탗탘탙탚������탛탞탟탡탢탣탥탦탧탨탩탪탫탮탲탳탴탵탶탷탹탺탻탼탽탾탿턀턁턂턃턄덧덩덫덮데덱덴델뎀뎁뎃뎄뎅뎌뎐뎔뎠뎡뎨뎬도독돈돋돌돎돐돔돕돗동돛돝돠돤돨돼됐되된될됨됩됫됴두둑둔둘둠둡둣둥둬뒀뒈뒝뒤뒨뒬뒵뒷뒹듀듄듈듐듕드득든듣들듦듬듭듯등듸디딕딘딛딜딤딥딧딨딩딪따딱딴딸�".split("");

      for (j = 0; j != D[181].length; ++j) if (D[181][j].charCodeAt(0) !== 0xFFFD) {
        e[D[181][j]] = 46336 + j;
        d[46336 + j] = D[181][j];
      }

      D[182] = "�����������������������������������������������������������������턅턆턇턈턉턊턋턌턎턏턐턑턒턓턔턕턖턗턘턙턚턛턜턝턞턟������턠턡턢턣턤턥턦턧턨턩턪턫턬턭턮턯턲턳턵턶턷턹턻턼턽턾������턿텂텆텇텈텉텊텋텎텏텑텒텓텕텖텗텘텙텚텛텞텠텢텣텤텥텦텧텩텪텫텭땀땁땃땄땅땋때땍땐땔땜땝땟땠땡떠떡떤떨떪떫떰떱떳떴떵떻떼떽뗀뗄뗌뗍뗏뗐뗑뗘뗬또똑똔똘똥똬똴뙈뙤뙨뚜뚝뚠뚤뚫뚬뚱뛔뛰뛴뛸뜀뜁뜅뜨뜩뜬뜯뜰뜸뜹뜻띄띈띌띔띕띠띤띨띰띱띳띵라락란랄람랍랏랐랑랒랖랗�".split("");

      for (j = 0; j != D[182].length; ++j) if (D[182][j].charCodeAt(0) !== 0xFFFD) {
        e[D[182][j]] = 46592 + j;
        d[46592 + j] = D[182][j];
      }

      D[183] = "�����������������������������������������������������������������텮텯텰텱텲텳텴텵텶텷텸텹텺텻텽텾텿톀톁톂톃톅톆톇톉톊������톋톌톍톎톏톐톑톒톓톔톕톖톗톘톙톚톛톜톝톞톟톢톣톥톦톧������톩톪톫톬톭톮톯톲톴톶톷톸톹톻톽톾톿퇁퇂퇃퇄퇅퇆퇇퇈퇉퇊퇋퇌퇍퇎퇏래랙랜랠램랩랫랬랭랴략랸럇량러럭런럴럼럽럿렀렁렇레렉렌렐렘렙렛렝려력련렬렴렵렷렸령례롄롑롓로록론롤롬롭롯롱롸롼뢍뢨뢰뢴뢸룀룁룃룅료룐룔룝룟룡루룩룬룰룸룹룻룽뤄뤘뤠뤼뤽륀륄륌륏륑류륙륜률륨륩�".split("");

      for (j = 0; j != D[183].length; ++j) if (D[183][j].charCodeAt(0) !== 0xFFFD) {
        e[D[183][j]] = 46848 + j;
        d[46848 + j] = D[183][j];
      }

      D[184] = "�����������������������������������������������������������������퇐퇑퇒퇓퇔퇕퇖퇗퇙퇚퇛퇜퇝퇞퇟퇠퇡퇢퇣퇤퇥퇦퇧퇨퇩퇪������퇫퇬퇭퇮퇯퇰퇱퇲퇳퇵퇶퇷퇹퇺퇻퇼퇽퇾퇿툀툁툂툃툄툅툆������툈툊툋툌툍툎툏툑툒툓툔툕툖툗툘툙툚툛툜툝툞툟툠툡툢툣툤툥툦툧툨툩륫륭르륵른를름릅릇릉릊릍릎리릭린릴림립릿링마막만많맏말맑맒맘맙맛망맞맡맣매맥맨맬맴맵맷맸맹맺먀먁먈먕머먹먼멀멂멈멉멋멍멎멓메멕멘멜멤멥멧멨멩며멱면멸몃몄명몇몌모목몫몬몰몲몸몹못몽뫄뫈뫘뫙뫼�".split("");

      for (j = 0; j != D[184].length; ++j) if (D[184][j].charCodeAt(0) !== 0xFFFD) {
        e[D[184][j]] = 47104 + j;
        d[47104 + j] = D[184][j];
      }

      D[185] = "�����������������������������������������������������������������툪툫툮툯툱툲툳툵툶툷툸툹툺툻툾퉀퉂퉃퉄퉅퉆퉇퉉퉊퉋퉌������퉍퉎퉏퉐퉑퉒퉓퉔퉕퉖퉗퉘퉙퉚퉛퉝퉞퉟퉠퉡퉢퉣퉥퉦퉧퉨������퉩퉪퉫퉬퉭퉮퉯퉰퉱퉲퉳퉴퉵퉶퉷퉸퉹퉺퉻퉼퉽퉾퉿튂튃튅튆튇튉튊튋튌묀묄묍묏묑묘묜묠묩묫무묵묶문묻물묽묾뭄뭅뭇뭉뭍뭏뭐뭔뭘뭡뭣뭬뮈뮌뮐뮤뮨뮬뮴뮷므믄믈믐믓미믹민믿밀밂밈밉밋밌밍및밑바박밖밗반받발밝밞밟밤밥밧방밭배백밴밸뱀뱁뱃뱄뱅뱉뱌뱍뱐뱝버벅번벋벌벎범법벗�".split("");

      for (j = 0; j != D[185].length; ++j) if (D[185][j].charCodeAt(0) !== 0xFFFD) {
        e[D[185][j]] = 47360 + j;
        d[47360 + j] = D[185][j];
      }

      D[186] = "�����������������������������������������������������������������튍튎튏튒튓튔튖튗튘튙튚튛튝튞튟튡튢튣튥튦튧튨튩튪튫튭������튮튯튰튲튳튴튵튶튷튺튻튽튾틁틃틄틅틆틇틊틌틍틎틏틐틑������틒틓틕틖틗틙틚틛틝틞틟틠틡틢틣틦틧틨틩틪틫틬틭틮틯틲틳틵틶틷틹틺벙벚베벡벤벧벨벰벱벳벴벵벼벽변별볍볏볐병볕볘볜보복볶본볼봄봅봇봉봐봔봤봬뵀뵈뵉뵌뵐뵘뵙뵤뵨부북분붇불붉붊붐붑붓붕붙붚붜붤붰붸뷔뷕뷘뷜뷩뷰뷴뷸븀븃븅브븍븐블븜븝븟비빅빈빌빎빔빕빗빙빚빛빠빡빤�".split("");

      for (j = 0; j != D[186].length; ++j) if (D[186][j].charCodeAt(0) !== 0xFFFD) {
        e[D[186][j]] = 47616 + j;
        d[47616 + j] = D[186][j];
      }

      D[187] = "�����������������������������������������������������������������틻틼틽틾틿팂팄팆팇팈팉팊팋팏팑팒팓팕팗팘팙팚팛팞팢팣������팤팦팧팪팫팭팮팯팱팲팳팴팵팶팷팺팾팿퍀퍁퍂퍃퍆퍇퍈퍉������퍊퍋퍌퍍퍎퍏퍐퍑퍒퍓퍔퍕퍖퍗퍘퍙퍚퍛퍜퍝퍞퍟퍠퍡퍢퍣퍤퍥퍦퍧퍨퍩빨빪빰빱빳빴빵빻빼빽뺀뺄뺌뺍뺏뺐뺑뺘뺙뺨뻐뻑뻔뻗뻘뻠뻣뻤뻥뻬뼁뼈뼉뼘뼙뼛뼜뼝뽀뽁뽄뽈뽐뽑뽕뾔뾰뿅뿌뿍뿐뿔뿜뿟뿡쀼쁑쁘쁜쁠쁨쁩삐삑삔삘삠삡삣삥사삭삯산삳살삵삶삼삽삿샀상샅새색샌샐샘샙샛샜생샤�".split("");

      for (j = 0; j != D[187].length; ++j) if (D[187][j].charCodeAt(0) !== 0xFFFD) {
        e[D[187][j]] = 47872 + j;
        d[47872 + j] = D[187][j];
      }

      D[188] = "�����������������������������������������������������������������퍪퍫퍬퍭퍮퍯퍰퍱퍲퍳퍴퍵퍶퍷퍸퍹퍺퍻퍾퍿펁펂펃펅펆펇������펈펉펊펋펎펒펓펔펕펖펗펚펛펝펞펟펡펢펣펤펥펦펧펪펬펮������펯펰펱펲펳펵펶펷펹펺펻펽펾펿폀폁폂폃폆폇폊폋폌폍폎폏폑폒폓폔폕폖샥샨샬샴샵샷샹섀섄섈섐섕서석섞섟선섣설섦섧섬섭섯섰성섶세섹센셀셈셉셋셌셍셔셕션셜셤셥셧셨셩셰셴셸솅소속솎손솔솖솜솝솟송솥솨솩솬솰솽쇄쇈쇌쇔쇗쇘쇠쇤쇨쇰쇱쇳쇼쇽숀숄숌숍숏숑수숙순숟술숨숩숫숭�".split("");

      for (j = 0; j != D[188].length; ++j) if (D[188][j].charCodeAt(0) !== 0xFFFD) {
        e[D[188][j]] = 48128 + j;
        d[48128 + j] = D[188][j];
      }

      D[189] = "�����������������������������������������������������������������폗폙폚폛폜폝폞폟폠폢폤폥폦폧폨폩폪폫폮폯폱폲폳폵폶폷������폸폹폺폻폾퐀퐂퐃퐄퐅퐆퐇퐉퐊퐋퐌퐍퐎퐏퐐퐑퐒퐓퐔퐕퐖������퐗퐘퐙퐚퐛퐜퐞퐟퐠퐡퐢퐣퐤퐥퐦퐧퐨퐩퐪퐫퐬퐭퐮퐯퐰퐱퐲퐳퐴퐵퐶퐷숯숱숲숴쉈쉐쉑쉔쉘쉠쉥쉬쉭쉰쉴쉼쉽쉿슁슈슉슐슘슛슝스슥슨슬슭슴습슷승시식신싣실싫심십싯싱싶싸싹싻싼쌀쌈쌉쌌쌍쌓쌔쌕쌘쌜쌤쌥쌨쌩썅써썩썬썰썲썸썹썼썽쎄쎈쎌쏀쏘쏙쏜쏟쏠쏢쏨쏩쏭쏴쏵쏸쐈쐐쐤쐬쐰�".split("");

      for (j = 0; j != D[189].length; ++j) if (D[189][j].charCodeAt(0) !== 0xFFFD) {
        e[D[189][j]] = 48384 + j;
        d[48384 + j] = D[189][j];
      }

      D[190] = "�����������������������������������������������������������������퐸퐹퐺퐻퐼퐽퐾퐿푁푂푃푅푆푇푈푉푊푋푌푍푎푏푐푑푒푓������푔푕푖푗푘푙푚푛푝푞푟푡푢푣푥푦푧푨푩푪푫푬푮푰푱푲������푳푴푵푶푷푺푻푽푾풁풃풄풅풆풇풊풌풎풏풐풑풒풓풕풖풗풘풙풚풛풜풝쐴쐼쐽쑈쑤쑥쑨쑬쑴쑵쑹쒀쒔쒜쒸쒼쓩쓰쓱쓴쓸쓺쓿씀씁씌씐씔씜씨씩씬씰씸씹씻씽아악안앉않알앍앎앓암압앗았앙앝앞애액앤앨앰앱앳앴앵야약얀얄얇얌얍얏양얕얗얘얜얠얩어억언얹얻얼얽얾엄업없엇었엉엊엌엎�".split("");

      for (j = 0; j != D[190].length; ++j) if (D[190][j].charCodeAt(0) !== 0xFFFD) {
        e[D[190][j]] = 48640 + j;
        d[48640 + j] = D[190][j];
      }

      D[191] = "�����������������������������������������������������������������풞풟풠풡풢풣풤풥풦풧풨풪풫풬풭풮풯풰풱풲풳풴풵풶풷풸������풹풺풻풼풽풾풿퓀퓁퓂퓃퓄퓅퓆퓇퓈퓉퓊퓋퓍퓎퓏퓑퓒퓓퓕������퓖퓗퓘퓙퓚퓛퓝퓞퓠퓡퓢퓣퓤퓥퓦퓧퓩퓪퓫퓭퓮퓯퓱퓲퓳퓴퓵퓶퓷퓹퓺퓼에엑엔엘엠엡엣엥여역엮연열엶엷염엽엾엿였영옅옆옇예옌옐옘옙옛옜오옥온올옭옮옰옳옴옵옷옹옻와왁완왈왐왑왓왔왕왜왝왠왬왯왱외왹왼욀욈욉욋욍요욕욘욜욤욥욧용우욱운울욹욺움웁웃웅워웍원월웜웝웠웡웨�".split("");

      for (j = 0; j != D[191].length; ++j) if (D[191][j].charCodeAt(0) !== 0xFFFD) {
        e[D[191][j]] = 48896 + j;
        d[48896 + j] = D[191][j];
      }

      D[192] = "�����������������������������������������������������������������퓾퓿픀픁픂픃픅픆픇픉픊픋픍픎픏픐픑픒픓픖픘픙픚픛픜픝������픞픟픠픡픢픣픤픥픦픧픨픩픪픫픬픭픮픯픰픱픲픳픴픵픶픷������픸픹픺픻픾픿핁핂핃핅핆핇핈핉핊핋핎핐핒핓핔핕핖핗핚핛핝핞핟핡핢핣웩웬웰웸웹웽위윅윈윌윔윕윗윙유육윤율윰윱윳융윷으윽은을읊음읍읏응읒읓읔읕읖읗의읜읠읨읫이익인일읽읾잃임입잇있잉잊잎자작잔잖잗잘잚잠잡잣잤장잦재잭잰잴잼잽잿쟀쟁쟈쟉쟌쟎쟐쟘쟝쟤쟨쟬저적전절젊�".split("");

      for (j = 0; j != D[192].length; ++j) if (D[192][j].charCodeAt(0) !== 0xFFFD) {
        e[D[192][j]] = 49152 + j;
        d[49152 + j] = D[192][j];
      }

      D[193] = "�����������������������������������������������������������������핤핦핧핪핬핮핯핰핱핲핳핶핷핹핺핻핽핾핿햀햁햂햃햆햊햋������햌햍햎햏햑햒햓햔햕햖햗햘햙햚햛햜햝햞햟햠햡햢햣햤햦햧������햨햩햪햫햬햭햮햯햰햱햲햳햴햵햶햷햸햹햺햻햼햽햾햿헀헁헂헃헄헅헆헇점접젓정젖제젝젠젤젬젭젯젱져젼졀졈졉졌졍졔조족존졸졺좀좁좃종좆좇좋좌좍좔좝좟좡좨좼좽죄죈죌죔죕죗죙죠죡죤죵주죽준줄줅줆줌줍줏중줘줬줴쥐쥑쥔쥘쥠쥡쥣쥬쥰쥴쥼즈즉즌즐즘즙즛증지직진짇질짊짐집짓�".split("");

      for (j = 0; j != D[193].length; ++j) if (D[193][j].charCodeAt(0) !== 0xFFFD) {
        e[D[193][j]] = 49408 + j;
        d[49408 + j] = D[193][j];
      }

      D[194] = "�����������������������������������������������������������������헊헋헍헎헏헑헓헔헕헖헗헚헜헞헟헠헡헢헣헦헧헩헪헫헭헮������헯헰헱헲헳헶헸헺헻헼헽헾헿혂혃혅혆혇혉혊혋혌혍혎혏혒������혖혗혘혙혚혛혝혞혟혡혢혣혥혦혧혨혩혪혫혬혮혯혰혱혲혳혴혵혶혷혺혻징짖짙짚짜짝짠짢짤짧짬짭짯짰짱째짹짼쨀쨈쨉쨋쨌쨍쨔쨘쨩쩌쩍쩐쩔쩜쩝쩟쩠쩡쩨쩽쪄쪘쪼쪽쫀쫄쫌쫍쫏쫑쫓쫘쫙쫠쫬쫴쬈쬐쬔쬘쬠쬡쭁쭈쭉쭌쭐쭘쭙쭝쭤쭸쭹쮜쮸쯔쯤쯧쯩찌찍찐찔찜찝찡찢찧차착찬찮찰참찹찻�".split("");

      for (j = 0; j != D[194].length; ++j) if (D[194][j].charCodeAt(0) !== 0xFFFD) {
        e[D[194][j]] = 49664 + j;
        d[49664 + j] = D[194][j];
      }

      D[195] = "�����������������������������������������������������������������혽혾혿홁홂홃홄홆홇홊홌홎홏홐홒홓홖홗홙홚홛홝홞홟홠홡������홢홣홤홥홦홨홪홫홬홭홮홯홲홳홵홶홷홸홹홺홻홼홽홾홿횀������횁횂횄횆횇횈횉횊횋횎횏횑횒횓횕횖횗횘횙횚횛횜횞횠횢횣횤횥횦횧횩횪찼창찾채책챈챌챔챕챗챘챙챠챤챦챨챰챵처척천철첨첩첫첬청체첵첸첼쳄쳅쳇쳉쳐쳔쳤쳬쳰촁초촉촌촐촘촙촛총촤촨촬촹최쵠쵤쵬쵭쵯쵱쵸춈추축춘출춤춥춧충춰췄췌췐취췬췰췸췹췻췽츄츈츌츔츙츠측츤츨츰츱츳층�".split("");

      for (j = 0; j != D[195].length; ++j) if (D[195][j].charCodeAt(0) !== 0xFFFD) {
        e[D[195][j]] = 49920 + j;
        d[49920 + j] = D[195][j];
      }

      D[196] = "�����������������������������������������������������������������횫횭횮횯횱횲횳횴횵횶횷횸횺횼횽횾횿훀훁훂훃훆훇훉훊훋������훍훎훏훐훒훓훕훖훘훚훛훜훝훞훟훡훢훣훥훦훧훩훪훫훬훭������훮훯훱훲훳훴훶훷훸훹훺훻훾훿휁휂휃휅휆휇휈휉휊휋휌휍휎휏휐휒휓휔치칙친칟칠칡침칩칫칭카칵칸칼캄캅캇캉캐캑캔캘캠캡캣캤캥캬캭컁커컥컨컫컬컴컵컷컸컹케켁켄켈켐켑켓켕켜켠켤켬켭켯켰켱켸코콕콘콜콤콥콧콩콰콱콴콸쾀쾅쾌쾡쾨쾰쿄쿠쿡쿤쿨쿰쿱쿳쿵쿼퀀퀄퀑퀘퀭퀴퀵퀸퀼�".split("");

      for (j = 0; j != D[196].length; ++j) if (D[196][j].charCodeAt(0) !== 0xFFFD) {
        e[D[196][j]] = 50176 + j;
        d[50176 + j] = D[196][j];
      }

      D[197] = "�����������������������������������������������������������������휕휖휗휚휛휝휞휟휡휢휣휤휥휦휧휪휬휮휯휰휱휲휳휶휷휹������휺휻휽휾휿흀흁흂흃흅흆흈흊흋흌흍흎흏흒흓흕흚흛흜흝흞������흟흢흤흦흧흨흪흫흭흮흯흱흲흳흵흶흷흸흹흺흻흾흿힀힂힃힄힅힆힇힊힋큄큅큇큉큐큔큘큠크큭큰클큼큽킁키킥킨킬킴킵킷킹타탁탄탈탉탐탑탓탔탕태택탠탤탬탭탯탰탱탸턍터턱턴털턺텀텁텃텄텅테텍텐텔템텝텟텡텨텬텼톄톈토톡톤톨톰톱톳통톺톼퇀퇘퇴퇸툇툉툐투툭툰툴툼툽툿퉁퉈퉜�".split("");

      for (j = 0; j != D[197].length; ++j) if (D[197][j].charCodeAt(0) !== 0xFFFD) {
        e[D[197][j]] = 50432 + j;
        d[50432 + j] = D[197][j];
      }

      D[198] = "�����������������������������������������������������������������힍힎힏힑힒힓힔힕힖힗힚힜힞힟힠힡힢힣������������������������������������������������������������������������������퉤튀튁튄튈튐튑튕튜튠튤튬튱트특튼튿틀틂틈틉틋틔틘틜틤틥티틱틴틸팀팁팃팅파팍팎판팔팖팜팝팟팠팡팥패팩팬팰팸팹팻팼팽퍄퍅퍼퍽펀펄펌펍펏펐펑페펙펜펠펨펩펫펭펴편펼폄폅폈평폐폘폡폣포폭폰폴폼폽폿퐁�".split("");

      for (j = 0; j != D[198].length; ++j) if (D[198][j].charCodeAt(0) !== 0xFFFD) {
        e[D[198][j]] = 50688 + j;
        d[50688 + j] = D[198][j];
      }

      D[199] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������퐈퐝푀푄표푠푤푭푯푸푹푼푿풀풂품풉풋풍풔풩퓌퓐퓔퓜퓟퓨퓬퓰퓸퓻퓽프픈플픔픕픗피픽핀필핌핍핏핑하학한할핥함합핫항해핵핸핼햄햅햇했행햐향허헉헌헐헒험헙헛헝헤헥헨헬헴헵헷헹혀혁현혈혐협혓혔형혜혠�".split("");

      for (j = 0; j != D[199].length; ++j) if (D[199][j].charCodeAt(0) !== 0xFFFD) {
        e[D[199][j]] = 50944 + j;
        d[50944 + j] = D[199][j];
      }

      D[200] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������혤혭호혹혼홀홅홈홉홋홍홑화확환활홧황홰홱홴횃횅회획횐횔횝횟횡효횬횰횹횻후훅훈훌훑훔훗훙훠훤훨훰훵훼훽휀휄휑휘휙휜휠휨휩휫휭휴휵휸휼흄흇흉흐흑흔흖흗흘흙흠흡흣흥흩희흰흴흼흽힁히힉힌힐힘힙힛힝�".split("");

      for (j = 0; j != D[200].length; ++j) if (D[200][j].charCodeAt(0) !== 0xFFFD) {
        e[D[200][j]] = 51200 + j;
        d[51200 + j] = D[200][j];
      }

      D[202] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������伽佳假價加可呵哥嘉嫁家暇架枷柯歌珂痂稼苛茄街袈訶賈跏軻迦駕刻却各恪慤殼珏脚覺角閣侃刊墾奸姦干幹懇揀杆柬桿澗癎看磵稈竿簡肝艮艱諫間乫喝曷渴碣竭葛褐蝎鞨勘坎堪嵌感憾戡敢柑橄減甘疳監瞰紺邯鑑鑒龕�".split("");

      for (j = 0; j != D[202].length; ++j) if (D[202][j].charCodeAt(0) !== 0xFFFD) {
        e[D[202][j]] = 51712 + j;
        d[51712 + j] = D[202][j];
      }

      D[203] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������匣岬甲胛鉀閘剛堈姜岡崗康强彊慷江畺疆糠絳綱羌腔舡薑襁講鋼降鱇介价個凱塏愷愾慨改槪漑疥皆盖箇芥蓋豈鎧開喀客坑更粳羹醵倨去居巨拒据據擧渠炬祛距踞車遽鉅鋸乾件健巾建愆楗腱虔蹇鍵騫乞傑杰桀儉劍劒檢�".split("");

      for (j = 0; j != D[203].length; ++j) if (D[203][j].charCodeAt(0) !== 0xFFFD) {
        e[D[203][j]] = 51968 + j;
        d[51968 + j] = D[203][j];
      }

      D[204] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������瞼鈐黔劫怯迲偈憩揭擊格檄激膈覡隔堅牽犬甄絹繭肩見譴遣鵑抉決潔結缺訣兼慊箝謙鉗鎌京俓倞傾儆勁勍卿坰境庚徑慶憬擎敬景暻更梗涇炅烱璟璥瓊痙硬磬竟競絅經耕耿脛莖警輕逕鏡頃頸驚鯨係啓堺契季屆悸戒桂械�".split("");

      for (j = 0; j != D[204].length; ++j) if (D[204][j].charCodeAt(0) !== 0xFFFD) {
        e[D[204][j]] = 52224 + j;
        d[52224 + j] = D[204][j];
      }

      D[205] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������棨溪界癸磎稽系繫繼計誡谿階鷄古叩告呱固姑孤尻庫拷攷故敲暠枯槁沽痼皐睾稿羔考股膏苦苽菰藁蠱袴誥賈辜錮雇顧高鼓哭斛曲梏穀谷鵠困坤崑昆梱棍滾琨袞鯤汨滑骨供公共功孔工恐恭拱控攻珙空蚣貢鞏串寡戈果瓜�".split("");

      for (j = 0; j != D[205].length; ++j) if (D[205][j].charCodeAt(0) !== 0xFFFD) {
        e[D[205][j]] = 52480 + j;
        d[52480 + j] = D[205][j];
      }

      D[206] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������科菓誇課跨過鍋顆廓槨藿郭串冠官寬慣棺款灌琯瓘管罐菅觀貫關館刮恝括适侊光匡壙廣曠洸炚狂珖筐胱鑛卦掛罫乖傀塊壞怪愧拐槐魁宏紘肱轟交僑咬喬嬌嶠巧攪敎校橋狡皎矯絞翹膠蕎蛟較轎郊餃驕鮫丘久九仇俱具勾�".split("");

      for (j = 0; j != D[206].length; ++j) if (D[206][j].charCodeAt(0) !== 0xFFFD) {
        e[D[206][j]] = 52736 + j;
        d[52736 + j] = D[206][j];
      }

      D[207] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������區口句咎嘔坵垢寇嶇廐懼拘救枸柩構歐毆毬求溝灸狗玖球瞿矩究絿耉臼舅舊苟衢謳購軀逑邱鉤銶駒驅鳩鷗龜國局菊鞠鞫麴君窘群裙軍郡堀屈掘窟宮弓穹窮芎躬倦券勸卷圈拳捲權淃眷厥獗蕨蹶闕机櫃潰詭軌饋句晷歸貴�".split("");

      for (j = 0; j != D[207].length; ++j) if (D[207][j].charCodeAt(0) !== 0xFFFD) {
        e[D[207][j]] = 52992 + j;
        d[52992 + j] = D[207][j];
      }

      D[208] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������鬼龜叫圭奎揆槻珪硅窺竅糾葵規赳逵閨勻均畇筠菌鈞龜橘克剋劇戟棘極隙僅劤勤懃斤根槿瑾筋芹菫覲謹近饉契今妗擒昑檎琴禁禽芩衾衿襟金錦伋及急扱汲級給亘兢矜肯企伎其冀嗜器圻基埼夔奇妓寄岐崎己幾忌技旗旣�".split("");

      for (j = 0; j != D[208].length; ++j) if (D[208][j].charCodeAt(0) !== 0xFFFD) {
        e[D[208][j]] = 53248 + j;
        d[53248 + j] = D[208][j];
      }

      D[209] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������朞期杞棋棄機欺氣汽沂淇玘琦琪璂璣畸畿碁磯祁祇祈祺箕紀綺羈耆耭肌記譏豈起錡錤飢饑騎騏驥麒緊佶吉拮桔金喫儺喇奈娜懦懶拏拿癩羅蘿螺裸邏那樂洛烙珞落諾酪駱亂卵暖欄煖爛蘭難鸞捏捺南嵐枏楠湳濫男藍襤拉�".split("");

      for (j = 0; j != D[209].length; ++j) if (D[209][j].charCodeAt(0) !== 0xFFFD) {
        e[D[209][j]] = 53504 + j;
        d[53504 + j] = D[209][j];
      }

      D[210] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������納臘蠟衲囊娘廊朗浪狼郎乃來內奈柰耐冷女年撚秊念恬拈捻寧寗努勞奴弩怒擄櫓爐瑙盧老蘆虜路露駑魯鷺碌祿綠菉錄鹿論壟弄濃籠聾膿農惱牢磊腦賂雷尿壘屢樓淚漏累縷陋嫩訥杻紐勒肋凜凌稜綾能菱陵尼泥匿溺多茶�".split("");

      for (j = 0; j != D[210].length; ++j) if (D[210][j].charCodeAt(0) !== 0xFFFD) {
        e[D[210][j]] = 53760 + j;
        d[53760 + j] = D[210][j];
      }

      D[211] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������丹亶但單團壇彖斷旦檀段湍短端簞緞蛋袒鄲鍛撻澾獺疸達啖坍憺擔曇淡湛潭澹痰聃膽蕁覃談譚錟沓畓答踏遝唐堂塘幢戇撞棠當糖螳黨代垈坮大對岱帶待戴擡玳臺袋貸隊黛宅德悳倒刀到圖堵塗導屠島嶋度徒悼挑掉搗桃�".split("");

      for (j = 0; j != D[211].length; ++j) if (D[211][j].charCodeAt(0) !== 0xFFFD) {
        e[D[211][j]] = 54016 + j;
        d[54016 + j] = D[211][j];
      }

      D[212] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������棹櫂淘渡滔濤燾盜睹禱稻萄覩賭跳蹈逃途道都鍍陶韜毒瀆牘犢獨督禿篤纛讀墩惇敦旽暾沌焞燉豚頓乭突仝冬凍動同憧東桐棟洞潼疼瞳童胴董銅兜斗杜枓痘竇荳讀豆逗頭屯臀芚遁遯鈍得嶝橙燈登等藤謄鄧騰喇懶拏癩羅�".split("");

      for (j = 0; j != D[212].length; ++j) if (D[212][j].charCodeAt(0) !== 0xFFFD) {
        e[D[212][j]] = 54272 + j;
        d[54272 + j] = D[212][j];
      }

      D[213] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������蘿螺裸邏樂洛烙珞絡落諾酪駱丹亂卵欄欒瀾爛蘭鸞剌辣嵐擥攬欖濫籃纜藍襤覽拉臘蠟廊朗浪狼琅瑯螂郞來崍徠萊冷掠略亮倆兩凉梁樑粮粱糧良諒輛量侶儷勵呂廬慮戾旅櫚濾礪藜蠣閭驢驪麗黎力曆歷瀝礫轢靂憐戀攣漣�".split("");

      for (j = 0; j != D[213].length; ++j) if (D[213][j].charCodeAt(0) !== 0xFFFD) {
        e[D[213][j]] = 54528 + j;
        d[54528 + j] = D[213][j];
      }

      D[214] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������煉璉練聯蓮輦連鍊冽列劣洌烈裂廉斂殮濂簾獵令伶囹寧岺嶺怜玲笭羚翎聆逞鈴零靈領齡例澧禮醴隷勞怒撈擄櫓潞瀘爐盧老蘆虜路輅露魯鷺鹵碌祿綠菉錄鹿麓論壟弄朧瀧瓏籠聾儡瀨牢磊賂賚賴雷了僚寮廖料燎療瞭聊蓼�".split("");

      for (j = 0; j != D[214].length; ++j) if (D[214][j].charCodeAt(0) !== 0xFFFD) {
        e[D[214][j]] = 54784 + j;
        d[54784 + j] = D[214][j];
      }

      D[215] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������遼鬧龍壘婁屢樓淚漏瘻累縷蔞褸鏤陋劉旒柳榴流溜瀏琉瑠留瘤硫謬類六戮陸侖倫崙淪綸輪律慄栗率隆勒肋凜凌楞稜綾菱陵俚利厘吏唎履悧李梨浬犁狸理璃異痢籬罹羸莉裏裡里釐離鯉吝潾燐璘藺躪隣鱗麟林淋琳臨霖砬�".split("");

      for (j = 0; j != D[215].length; ++j) if (D[215][j].charCodeAt(0) !== 0xFFFD) {
        e[D[215][j]] = 55040 + j;
        d[55040 + j] = D[215][j];
      }

      D[216] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������立笠粒摩瑪痲碼磨馬魔麻寞幕漠膜莫邈万卍娩巒彎慢挽晩曼滿漫灣瞞萬蔓蠻輓饅鰻唜抹末沫茉襪靺亡妄忘忙望網罔芒茫莽輞邙埋妹媒寐昧枚梅每煤罵買賣邁魅脈貊陌驀麥孟氓猛盲盟萌冪覓免冕勉棉沔眄眠綿緬面麵滅�".split("");

      for (j = 0; j != D[216].length; ++j) if (D[216][j].charCodeAt(0) !== 0xFFFD) {
        e[D[216][j]] = 55296 + j;
        d[55296 + j] = D[216][j];
      }

      D[217] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������蔑冥名命明暝椧溟皿瞑茗蓂螟酩銘鳴袂侮冒募姆帽慕摸摹暮某模母毛牟牡瑁眸矛耗芼茅謀謨貌木沐牧目睦穆鶩歿沒夢朦蒙卯墓妙廟描昴杳渺猫竗苗錨務巫憮懋戊拇撫无楙武毋無珷畝繆舞茂蕪誣貿霧鵡墨默們刎吻問文�".split("");

      for (j = 0; j != D[217].length; ++j) if (D[217][j].charCodeAt(0) !== 0xFFFD) {
        e[D[217][j]] = 55552 + j;
        d[55552 + j] = D[217][j];
      }

      D[218] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������汶紊紋聞蚊門雯勿沕物味媚尾嵋彌微未梶楣渼湄眉米美薇謎迷靡黴岷悶愍憫敏旻旼民泯玟珉緡閔密蜜謐剝博拍搏撲朴樸泊珀璞箔粕縛膊舶薄迫雹駁伴半反叛拌搬攀斑槃泮潘班畔瘢盤盼磐磻礬絆般蟠返頒飯勃拔撥渤潑�".split("");

      for (j = 0; j != D[218].length; ++j) if (D[218][j].charCodeAt(0) !== 0xFFFD) {
        e[D[218][j]] = 55808 + j;
        d[55808 + j] = D[218][j];
      }

      D[219] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������發跋醱鉢髮魃倣傍坊妨尨幇彷房放方旁昉枋榜滂磅紡肪膀舫芳蒡蚌訪謗邦防龐倍俳北培徘拜排杯湃焙盃背胚裴裵褙賠輩配陪伯佰帛柏栢白百魄幡樊煩燔番磻繁蕃藩飜伐筏罰閥凡帆梵氾汎泛犯範范法琺僻劈壁擘檗璧癖�".split("");

      for (j = 0; j != D[219].length; ++j) if (D[219][j].charCodeAt(0) !== 0xFFFD) {
        e[D[219][j]] = 56064 + j;
        d[56064 + j] = D[219][j];
      }

      D[220] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������碧蘗闢霹便卞弁變辨辯邊別瞥鱉鼈丙倂兵屛幷昞昺柄棅炳甁病秉竝輧餠騈保堡報寶普步洑湺潽珤甫菩補褓譜輔伏僕匐卜宓復服福腹茯蔔複覆輹輻馥鰒本乶俸奉封峯峰捧棒烽熢琫縫蓬蜂逢鋒鳳不付俯傅剖副否咐埠夫婦�".split("");

      for (j = 0; j != D[220].length; ++j) if (D[220][j].charCodeAt(0) !== 0xFFFD) {
        e[D[220][j]] = 56320 + j;
        d[56320 + j] = D[220][j];
      }

      D[221] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������孚孵富府復扶敷斧浮溥父符簿缶腐腑膚艀芙莩訃負賦賻赴趺部釜阜附駙鳧北分吩噴墳奔奮忿憤扮昐汾焚盆粉糞紛芬賁雰不佛弗彿拂崩朋棚硼繃鵬丕備匕匪卑妃婢庇悲憊扉批斐枇榧比毖毗毘沸泌琵痺砒碑秕秘粃緋翡肥�".split("");

      for (j = 0; j != D[221].length; ++j) if (D[221][j].charCodeAt(0) !== 0xFFFD) {
        e[D[221][j]] = 56576 + j;
        d[56576 + j] = D[221][j];
      }

      D[222] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������脾臂菲蜚裨誹譬費鄙非飛鼻嚬嬪彬斌檳殯浜濱瀕牝玭貧賓頻憑氷聘騁乍事些仕伺似使俟僿史司唆嗣四士奢娑寫寺射巳師徙思捨斜斯柶査梭死沙泗渣瀉獅砂社祀祠私篩紗絲肆舍莎蓑蛇裟詐詞謝賜赦辭邪飼駟麝削數朔索�".split("");

      for (j = 0; j != D[222].length; ++j) if (D[222][j].charCodeAt(0) !== 0xFFFD) {
        e[D[222][j]] = 56832 + j;
        d[56832 + j] = D[222][j];
      }

      D[223] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������傘刪山散汕珊産疝算蒜酸霰乷撒殺煞薩三參杉森渗芟蔘衫揷澁鈒颯上傷像償商喪嘗孀尙峠常床庠廂想桑橡湘爽牀狀相祥箱翔裳觴詳象賞霜塞璽賽嗇塞穡索色牲生甥省笙墅壻嶼序庶徐恕抒捿敍暑曙書栖棲犀瑞筮絮緖署�".split("");

      for (j = 0; j != D[223].length; ++j) if (D[223][j].charCodeAt(0) !== 0xFFFD) {
        e[D[223][j]] = 57088 + j;
        d[57088 + j] = D[223][j];
      }

      D[224] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������胥舒薯西誓逝鋤黍鼠夕奭席惜昔晳析汐淅潟石碩蓆釋錫仙僊先善嬋宣扇敾旋渲煽琁瑄璇璿癬禪線繕羨腺膳船蘚蟬詵跣選銑鐥饍鮮卨屑楔泄洩渫舌薛褻設說雪齧剡暹殲纖蟾贍閃陝攝涉燮葉城姓宬性惺成星晟猩珹盛省筬�".split("");

      for (j = 0; j != D[224].length; ++j) if (D[224][j].charCodeAt(0) !== 0xFFFD) {
        e[D[224][j]] = 57344 + j;
        d[57344 + j] = D[224][j];
      }

      D[225] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������聖聲腥誠醒世勢歲洗稅笹細說貰召嘯塑宵小少巢所掃搔昭梳沼消溯瀟炤燒甦疏疎瘙笑篠簫素紹蔬蕭蘇訴逍遡邵銷韶騷俗屬束涑粟續謖贖速孫巽損蓀遜飡率宋悚松淞訟誦送頌刷殺灑碎鎖衰釗修受嗽囚垂壽嫂守岫峀帥愁�".split("");

      for (j = 0; j != D[225].length; ++j) if (D[225][j].charCodeAt(0) !== 0xFFFD) {
        e[D[225][j]] = 57600 + j;
        d[57600 + j] = D[225][j];
      }

      D[226] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������戍手授搜收數樹殊水洙漱燧狩獸琇璲瘦睡秀穗竪粹綏綬繡羞脩茱蒐蓚藪袖誰讐輸遂邃酬銖銹隋隧隨雖需須首髓鬚叔塾夙孰宿淑潚熟琡璹肅菽巡徇循恂旬栒楯橓殉洵淳珣盾瞬筍純脣舜荀蓴蕣詢諄醇錞順馴戌術述鉥崇崧�".split("");

      for (j = 0; j != D[226].length; ++j) if (D[226][j].charCodeAt(0) !== 0xFFFD) {
        e[D[226][j]] = 57856 + j;
        d[57856 + j] = D[226][j];
      }

      D[227] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������嵩瑟膝蝨濕拾習褶襲丞乘僧勝升承昇繩蠅陞侍匙嘶始媤尸屎屍市弑恃施是時枾柴猜矢示翅蒔蓍視試詩諡豕豺埴寔式息拭植殖湜熄篒蝕識軾食飾伸侁信呻娠宸愼新晨燼申神紳腎臣莘薪藎蜃訊身辛辰迅失室實悉審尋心沁�".split("");

      for (j = 0; j != D[227].length; ++j) if (D[227][j].charCodeAt(0) !== 0xFFFD) {
        e[D[227][j]] = 58112 + j;
        d[58112 + j] = D[227][j];
      }

      D[228] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������沈深瀋甚芯諶什十拾雙氏亞俄兒啞娥峨我牙芽莪蛾衙訝阿雅餓鴉鵝堊岳嶽幄惡愕握樂渥鄂鍔顎鰐齷安岸按晏案眼雁鞍顔鮟斡謁軋閼唵岩巖庵暗癌菴闇壓押狎鴨仰央怏昻殃秧鴦厓哀埃崖愛曖涯碍艾隘靄厄扼掖液縊腋額�".split("");

      for (j = 0; j != D[228].length; ++j) if (D[228][j].charCodeAt(0) !== 0xFFFD) {
        e[D[228][j]] = 58368 + j;
        d[58368 + j] = D[228][j];
      }

      D[229] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������櫻罌鶯鸚也倻冶夜惹揶椰爺耶若野弱掠略約若葯蒻藥躍亮佯兩凉壤孃恙揚攘敭暘梁楊樣洋瀁煬痒瘍禳穰糧羊良襄諒讓釀陽量養圄御於漁瘀禦語馭魚齬億憶抑檍臆偃堰彦焉言諺孼蘖俺儼嚴奄掩淹嶪業円予余勵呂女如廬�".split("");

      for (j = 0; j != D[229].length; ++j) if (D[229][j].charCodeAt(0) !== 0xFFFD) {
        e[D[229][j]] = 58624 + j;
        d[58624 + j] = D[229][j];
      }

      D[230] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������旅歟汝濾璵礖礪與艅茹輿轝閭餘驪麗黎亦力域役易曆歷疫繹譯轢逆驛嚥堧姸娟宴年延憐戀捐挻撚椽沇沿涎涓淵演漣烟然煙煉燃燕璉硏硯秊筵緣練縯聯衍軟輦蓮連鉛鍊鳶列劣咽悅涅烈熱裂說閱厭廉念捻染殮炎焰琰艶苒�".split("");

      for (j = 0; j != D[230].length; ++j) if (D[230][j].charCodeAt(0) !== 0xFFFD) {
        e[D[230][j]] = 58880 + j;
        d[58880 + j] = D[230][j];
      }

      D[231] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������簾閻髥鹽曄獵燁葉令囹塋寧嶺嶸影怜映暎楹榮永泳渶潁濚瀛瀯煐營獰玲瑛瑩瓔盈穎纓羚聆英詠迎鈴鍈零霙靈領乂倪例刈叡曳汭濊猊睿穢芮藝蘂禮裔詣譽豫醴銳隸霓預五伍俉傲午吾吳嗚塢墺奧娛寤悟惡懊敖旿晤梧汚澳�".split("");

      for (j = 0; j != D[231].length; ++j) if (D[231][j].charCodeAt(0) !== 0xFFFD) {
        e[D[231][j]] = 59136 + j;
        d[59136 + j] = D[231][j];
      }

      D[232] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������烏熬獒筽蜈誤鰲鼇屋沃獄玉鈺溫瑥瘟穩縕蘊兀壅擁瓮甕癰翁邕雍饔渦瓦窩窪臥蛙蝸訛婉完宛梡椀浣玩琓琬碗緩翫脘腕莞豌阮頑曰往旺枉汪王倭娃歪矮外嵬巍猥畏了僚僥凹堯夭妖姚寥寮尿嶢拗搖撓擾料曜樂橈燎燿瑤療�".split("");

      for (j = 0; j != D[232].length; ++j) if (D[232][j].charCodeAt(0) !== 0xFFFD) {
        e[D[232][j]] = 59392 + j;
        d[59392 + j] = D[232][j];
      }

      D[233] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������窈窯繇繞耀腰蓼蟯要謠遙遼邀饒慾欲浴縟褥辱俑傭冗勇埇墉容庸慂榕涌湧溶熔瑢用甬聳茸蓉踊鎔鏞龍于佑偶優又友右宇寓尤愚憂旴牛玗瑀盂祐禑禹紆羽芋藕虞迂遇郵釪隅雨雩勖彧旭昱栯煜稶郁頊云暈橒殞澐熉耘芸蕓�".split("");

      for (j = 0; j != D[233].length; ++j) if (D[233][j].charCodeAt(0) !== 0xFFFD) {
        e[D[233][j]] = 59648 + j;
        d[59648 + j] = D[233][j];
      }

      D[234] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������運隕雲韻蔚鬱亐熊雄元原員圓園垣媛嫄寃怨愿援沅洹湲源爰猿瑗苑袁轅遠阮院願鴛月越鉞位偉僞危圍委威尉慰暐渭爲瑋緯胃萎葦蔿蝟衛褘謂違韋魏乳侑儒兪劉唯喩孺宥幼幽庾悠惟愈愉揄攸有杻柔柚柳楡楢油洧流游溜�".split("");

      for (j = 0; j != D[234].length; ++j) if (D[234][j].charCodeAt(0) !== 0xFFFD) {
        e[D[234][j]] = 59904 + j;
        d[59904 + j] = D[234][j];
      }

      D[235] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������濡猶猷琉瑜由留癒硫紐維臾萸裕誘諛諭踰蹂遊逾遺酉釉鍮類六堉戮毓肉育陸倫允奫尹崙淪潤玧胤贇輪鈗閏律慄栗率聿戎瀜絨融隆垠恩慇殷誾銀隱乙吟淫蔭陰音飮揖泣邑凝應膺鷹依倚儀宜意懿擬椅毅疑矣義艤薏蟻衣誼�".split("");

      for (j = 0; j != D[235].length; ++j) if (D[235][j].charCodeAt(0) !== 0xFFFD) {
        e[D[235][j]] = 60160 + j;
        d[60160 + j] = D[235][j];
      }

      D[236] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������議醫二以伊利吏夷姨履已弛彛怡易李梨泥爾珥理異痍痢移罹而耳肄苡荑裏裡貽貳邇里離飴餌匿溺瀷益翊翌翼謚人仁刃印吝咽因姻寅引忍湮燐璘絪茵藺蚓認隣靭靷鱗麟一佚佾壹日溢逸鎰馹任壬妊姙恁林淋稔臨荏賃入卄�".split("");

      for (j = 0; j != D[236].length; ++j) if (D[236][j].charCodeAt(0) !== 0xFFFD) {
        e[D[236][j]] = 60416 + j;
        d[60416 + j] = D[236][j];
      }

      D[237] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������立笠粒仍剩孕芿仔刺咨姉姿子字孜恣慈滋炙煮玆瓷疵磁紫者自茨蔗藉諮資雌作勺嚼斫昨灼炸爵綽芍酌雀鵲孱棧殘潺盞岑暫潛箴簪蠶雜丈仗匠場墻壯奬將帳庄張掌暲杖樟檣欌漿牆狀獐璋章粧腸臟臧莊葬蔣薔藏裝贓醬長�".split("");

      for (j = 0; j != D[237].length; ++j) if (D[237][j].charCodeAt(0) !== 0xFFFD) {
        e[D[237][j]] = 60672 + j;
        d[60672 + j] = D[237][j];
      }

      D[238] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������障再哉在宰才材栽梓渽滓災縡裁財載齋齎爭箏諍錚佇低儲咀姐底抵杵楮樗沮渚狙猪疽箸紵苧菹著藷詛貯躇這邸雎齟勣吊嫡寂摘敵滴狄炙的積笛籍績翟荻謫賊赤跡蹟迪迹適鏑佃佺傳全典前剪塡塼奠專展廛悛戰栓殿氈澱�".split("");

      for (j = 0; j != D[238].length; ++j) if (D[238][j].charCodeAt(0) !== 0xFFFD) {
        e[D[238][j]] = 60928 + j;
        d[60928 + j] = D[238][j];
      }

      D[239] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������煎琠田甸畑癲筌箋箭篆纏詮輾轉鈿銓錢鐫電顚顫餞切截折浙癤竊節絶占岾店漸点粘霑鮎點接摺蝶丁井亭停偵呈姃定幀庭廷征情挺政整旌晶晸柾楨檉正汀淀淨渟湞瀞炡玎珽町睛碇禎程穽精綎艇訂諪貞鄭酊釘鉦鋌錠霆靖�".split("");

      for (j = 0; j != D[239].length; ++j) if (D[239][j].charCodeAt(0) !== 0xFFFD) {
        e[D[239][j]] = 61184 + j;
        d[61184 + j] = D[239][j];
      }

      D[240] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������靜頂鼎制劑啼堤帝弟悌提梯濟祭第臍薺製諸蹄醍除際霽題齊俎兆凋助嘲弔彫措操早晁曺曹朝條棗槽漕潮照燥爪璪眺祖祚租稠窕粗糟組繰肇藻蚤詔調趙躁造遭釣阻雕鳥族簇足鏃存尊卒拙猝倧宗從悰慫棕淙琮種終綜縱腫�".split("");

      for (j = 0; j != D[240].length; ++j) if (D[240][j].charCodeAt(0) !== 0xFFFD) {
        e[D[240][j]] = 61440 + j;
        d[61440 + j] = D[240][j];
      }

      D[241] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������踪踵鍾鐘佐坐左座挫罪主住侏做姝胄呪周嗾奏宙州廚晝朱柱株注洲湊澍炷珠疇籌紂紬綢舟蛛註誅走躊輳週酎酒鑄駐竹粥俊儁准埈寯峻晙樽浚準濬焌畯竣蠢逡遵雋駿茁中仲衆重卽櫛楫汁葺增憎曾拯烝甑症繒蒸證贈之只�".split("");

      for (j = 0; j != D[241].length; ++j) if (D[241][j].charCodeAt(0) !== 0xFFFD) {
        e[D[241][j]] = 61696 + j;
        d[61696 + j] = D[241][j];
      }

      D[242] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������咫地址志持指摯支旨智枝枳止池沚漬知砥祉祗紙肢脂至芝芷蜘誌識贄趾遲直稙稷織職唇嗔塵振搢晉晋桭榛殄津溱珍瑨璡畛疹盡眞瞋秦縉縝臻蔯袗診賑軫辰進鎭陣陳震侄叱姪嫉帙桎瓆疾秩窒膣蛭質跌迭斟朕什執潗緝輯�".split("");

      for (j = 0; j != D[242].length; ++j) if (D[242][j].charCodeAt(0) !== 0xFFFD) {
        e[D[242][j]] = 61952 + j;
        d[61952 + j] = D[242][j];
      }

      D[243] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������鏶集徵懲澄且侘借叉嗟嵯差次此磋箚茶蹉車遮捉搾着窄錯鑿齪撰澯燦璨瓚竄簒纂粲纘讚贊鑽餐饌刹察擦札紮僭參塹慘慙懺斬站讒讖倉倡創唱娼廠彰愴敞昌昶暢槍滄漲猖瘡窓脹艙菖蒼債埰寀寨彩採砦綵菜蔡采釵冊柵策�".split("");

      for (j = 0; j != D[243].length; ++j) if (D[243][j].charCodeAt(0) !== 0xFFFD) {
        e[D[243][j]] = 62208 + j;
        d[62208 + j] = D[243][j];
      }

      D[244] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������責凄妻悽處倜刺剔尺慽戚拓擲斥滌瘠脊蹠陟隻仟千喘天川擅泉淺玔穿舛薦賤踐遷釧闡阡韆凸哲喆徹撤澈綴輟轍鐵僉尖沾添甛瞻簽籤詹諂堞妾帖捷牒疊睫諜貼輒廳晴淸聽菁請靑鯖切剃替涕滯締諦逮遞體初剿哨憔抄招梢�".split("");

      for (j = 0; j != D[244].length; ++j) if (D[244][j].charCodeAt(0) !== 0xFFFD) {
        e[D[244][j]] = 62464 + j;
        d[62464 + j] = D[244][j];
      }

      D[245] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������椒楚樵炒焦硝礁礎秒稍肖艸苕草蕉貂超酢醋醮促囑燭矗蜀觸寸忖村邨叢塚寵悤憁摠總聰蔥銃撮催崔最墜抽推椎楸樞湫皺秋芻萩諏趨追鄒酋醜錐錘鎚雛騶鰍丑畜祝竺筑築縮蓄蹙蹴軸逐春椿瑃出朮黜充忠沖蟲衝衷悴膵萃�".split("");

      for (j = 0; j != D[245].length; ++j) if (D[245][j].charCodeAt(0) !== 0xFFFD) {
        e[D[245][j]] = 62720 + j;
        d[62720 + j] = D[245][j];
      }

      D[246] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������贅取吹嘴娶就炊翠聚脆臭趣醉驟鷲側仄厠惻測層侈値嗤峙幟恥梔治淄熾痔痴癡稚穉緇緻置致蚩輜雉馳齒則勅飭親七柒漆侵寢枕沈浸琛砧針鍼蟄秤稱快他咤唾墮妥惰打拖朶楕舵陀馱駝倬卓啄坼度托拓擢晫柝濁濯琢琸託�".split("");

      for (j = 0; j != D[246].length; ++j) if (D[246][j].charCodeAt(0) !== 0xFFFD) {
        e[D[246][j]] = 62976 + j;
        d[62976 + j] = D[246][j];
      }

      D[247] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������鐸呑嘆坦彈憚歎灘炭綻誕奪脫探眈耽貪塔搭榻宕帑湯糖蕩兌台太怠態殆汰泰笞胎苔跆邰颱宅擇澤撑攄兎吐土討慟桶洞痛筒統通堆槌腿褪退頹偸套妬投透鬪慝特闖坡婆巴把播擺杷波派爬琶破罷芭跛頗判坂板版瓣販辦鈑�".split("");

      for (j = 0; j != D[247].length; ++j) if (D[247][j].charCodeAt(0) !== 0xFFFD) {
        e[D[247][j]] = 63232 + j;
        d[63232 + j] = D[247][j];
      }

      D[248] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������阪八叭捌佩唄悖敗沛浿牌狽稗覇貝彭澎烹膨愎便偏扁片篇編翩遍鞭騙貶坪平枰萍評吠嬖幣廢弊斃肺蔽閉陛佈包匍匏咆哺圃布怖抛抱捕暴泡浦疱砲胞脯苞葡蒲袍褒逋鋪飽鮑幅暴曝瀑爆輻俵剽彪慓杓標漂瓢票表豹飇飄驃�".split("");

      for (j = 0; j != D[248].length; ++j) if (D[248][j].charCodeAt(0) !== 0xFFFD) {
        e[D[248][j]] = 63488 + j;
        d[63488 + j] = D[248][j];
      }

      D[249] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������品稟楓諷豊風馮彼披疲皮被避陂匹弼必泌珌畢疋筆苾馝乏逼下何厦夏廈昰河瑕荷蝦賀遐霞鰕壑學虐謔鶴寒恨悍旱汗漢澣瀚罕翰閑閒限韓割轄函含咸啣喊檻涵緘艦銜陷鹹合哈盒蛤閤闔陜亢伉姮嫦巷恒抗杭桁沆港缸肛航�".split("");

      for (j = 0; j != D[249].length; ++j) if (D[249][j].charCodeAt(0) !== 0xFFFD) {
        e[D[249][j]] = 63744 + j;
        d[63744 + j] = D[249][j];
      }

      D[250] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������行降項亥偕咳垓奚孩害懈楷海瀣蟹解該諧邂駭骸劾核倖幸杏荇行享向嚮珦鄕響餉饗香噓墟虛許憲櫶獻軒歇險驗奕爀赫革俔峴弦懸晛泫炫玄玹現眩睍絃絢縣舷衒見賢鉉顯孑穴血頁嫌俠協夾峽挾浹狹脅脇莢鋏頰亨兄刑型�".split("");

      for (j = 0; j != D[250].length; ++j) if (D[250][j].charCodeAt(0) !== 0xFFFD) {
        e[D[250][j]] = 64000 + j;
        d[64000 + j] = D[250][j];
      }

      D[251] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������形泂滎瀅灐炯熒珩瑩荊螢衡逈邢鎣馨兮彗惠慧暳蕙蹊醯鞋乎互呼壕壺好岵弧戶扈昊晧毫浩淏湖滸澔濠濩灝狐琥瑚瓠皓祜糊縞胡芦葫蒿虎號蝴護豪鎬頀顥惑或酷婚昏混渾琿魂忽惚笏哄弘汞泓洪烘紅虹訌鴻化和嬅樺火畵�".split("");

      for (j = 0; j != D[251].length; ++j) if (D[251][j].charCodeAt(0) !== 0xFFFD) {
        e[D[251][j]] = 64256 + j;
        d[64256 + j] = D[251][j];
      }

      D[252] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������禍禾花華話譁貨靴廓擴攫確碻穫丸喚奐宦幻患換歡晥桓渙煥環紈還驩鰥活滑猾豁闊凰幌徨恍惶愰慌晃晄榥況湟滉潢煌璜皇篁簧荒蝗遑隍黃匯回廻徊恢悔懷晦會檜淮澮灰獪繪膾茴蛔誨賄劃獲宖橫鐄哮嚆孝效斅曉梟涍淆�".split("");

      for (j = 0; j != D[252].length; ++j) if (D[252][j].charCodeAt(0) !== 0xFFFD) {
        e[D[252][j]] = 64512 + j;
        d[64512 + j] = D[252][j];
      }

      D[253] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������爻肴酵驍侯候厚后吼喉嗅帿後朽煦珝逅勛勳塤壎焄熏燻薰訓暈薨喧暄煊萱卉喙毁彙徽揮暉煇諱輝麾休携烋畦虧恤譎鷸兇凶匈洶胸黑昕欣炘痕吃屹紇訖欠欽歆吸恰洽翕興僖凞喜噫囍姬嬉希憙憘戱晞曦熙熹熺犧禧稀羲詰�".split("");

      for (j = 0; j != D[253].length; ++j) if (D[253][j].charCodeAt(0) !== 0xFFFD) {
        e[D[253][j]] = 64768 + j;
        d[64768 + j] = D[253][j];
      }

      return {
        "enc": e,
        "dec": d
      };
    }();

    cptable[950] = function () {
      var d = [],
          e = {},
          D = [],
          j;
      D[0] = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~��������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[0].length; ++j) if (D[0][j].charCodeAt(0) !== 0xFFFD) {
        e[D[0][j]] = 0 + j;
        d[0 + j] = D[0][j];
      }

      D[161] = "����������������������������������������������������������������　，、。．‧；：？！︰…‥﹐﹑﹒·﹔﹕﹖﹗｜–︱—︳╴︴﹏（）︵︶｛｝︷︸〔〕︹︺【】︻︼《》︽︾〈〉︿﹀「」﹁﹂『』﹃﹄﹙﹚����������������������������������﹛﹜﹝﹞‘’“”〝〞‵′＃＆＊※§〃○●△▲◎☆★◇◆□■▽▼㊣℅¯￣＿ˍ﹉﹊﹍﹎﹋﹌﹟﹠﹡＋－×÷±√＜＞＝≦≧≠∞≒≡﹢﹣﹤﹥﹦～∩∪⊥∠∟⊿㏒㏑∫∮∵∴♀♂⊕⊙↑↓←→↖↗↙↘∥∣／�".split("");

      for (j = 0; j != D[161].length; ++j) if (D[161][j].charCodeAt(0) !== 0xFFFD) {
        e[D[161][j]] = 41216 + j;
        d[41216 + j] = D[161][j];
      }

      D[162] = "����������������������������������������������������������������＼∕﹨＄￥〒￠￡％＠℃℉﹩﹪﹫㏕㎜㎝㎞㏎㎡㎎㎏㏄°兙兛兞兝兡兣嗧瓩糎▁▂▃▄▅▆▇█▏▎▍▌▋▊▉┼┴┬┤├▔─│▕┌┐└┘╭����������������������������������╮╰╯═╞╪╡◢◣◥◤╱╲╳０１２３４５６７８９ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ〡〢〣〤〥〦〧〨〩十卄卅ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖ�".split("");

      for (j = 0; j != D[162].length; ++j) if (D[162][j].charCodeAt(0) !== 0xFFFD) {
        e[D[162][j]] = 41472 + j;
        d[41472 + j] = D[162][j];
      }

      D[163] = "����������������������������������������������������������������ｗｘｙｚΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩαβγδεζηθικλμνξοπρστυφχψωㄅㄆㄇㄈㄉㄊㄋㄌㄍㄎㄏ����������������������������������ㄐㄑㄒㄓㄔㄕㄖㄗㄘㄙㄚㄛㄜㄝㄞㄟㄠㄡㄢㄣㄤㄥㄦㄧㄨㄩ˙ˉˊˇˋ���������������������������������€������������������������������".split("");

      for (j = 0; j != D[163].length; ++j) if (D[163][j].charCodeAt(0) !== 0xFFFD) {
        e[D[163][j]] = 41728 + j;
        d[41728 + j] = D[163][j];
      }

      D[164] = "����������������������������������������������������������������一乙丁七乃九了二人儿入八几刀刁力匕十卜又三下丈上丫丸凡久么也乞于亡兀刃勺千叉口土士夕大女子孑孓寸小尢尸山川工己已巳巾干廾弋弓才����������������������������������丑丐不中丰丹之尹予云井互五亢仁什仃仆仇仍今介仄元允內六兮公冗凶分切刈勻勾勿化匹午升卅卞厄友及反壬天夫太夭孔少尤尺屯巴幻廿弔引心戈戶手扎支文斗斤方日曰月木欠止歹毋比毛氏水火爪父爻片牙牛犬王丙�".split("");

      for (j = 0; j != D[164].length; ++j) if (D[164][j].charCodeAt(0) !== 0xFFFD) {
        e[D[164][j]] = 41984 + j;
        d[41984 + j] = D[164][j];
      }

      D[165] = "����������������������������������������������������������������世丕且丘主乍乏乎以付仔仕他仗代令仙仞充兄冉冊冬凹出凸刊加功包匆北匝仟半卉卡占卯卮去可古右召叮叩叨叼司叵叫另只史叱台句叭叻四囚外����������������������������������央失奴奶孕它尼巨巧左市布平幼弁弘弗必戊打扔扒扑斥旦朮本未末札正母民氐永汁汀氾犯玄玉瓜瓦甘生用甩田由甲申疋白皮皿目矛矢石示禾穴立丞丟乒乓乩亙交亦亥仿伉伙伊伕伍伐休伏仲件任仰仳份企伋光兇兆先全�".split("");

      for (j = 0; j != D[165].length; ++j) if (D[165][j].charCodeAt(0) !== 0xFFFD) {
        e[D[165][j]] = 42240 + j;
        d[42240 + j] = D[165][j];
      }

      D[166] = "����������������������������������������������������������������共再冰列刑划刎刖劣匈匡匠印危吉吏同吊吐吁吋各向名合吃后吆吒因回囝圳地在圭圬圯圩夙多夷夸妄奸妃好她如妁字存宇守宅安寺尖屹州帆并年����������������������������������式弛忙忖戎戌戍成扣扛托收早旨旬旭曲曳有朽朴朱朵次此死氖汝汗汙江池汐汕污汛汍汎灰牟牝百竹米糸缶羊羽老考而耒耳聿肉肋肌臣自至臼舌舛舟艮色艾虫血行衣西阡串亨位住佇佗佞伴佛何估佐佑伽伺伸佃佔似但佣�".split("");

      for (j = 0; j != D[166].length; ++j) if (D[166][j].charCodeAt(0) !== 0xFFFD) {
        e[D[166][j]] = 42496 + j;
        d[42496 + j] = D[166][j];
      }

      D[167] = "����������������������������������������������������������������作你伯低伶余佝佈佚兌克免兵冶冷別判利刪刨劫助努劬匣即卵吝吭吞吾否呎吧呆呃吳呈呂君吩告吹吻吸吮吵吶吠吼呀吱含吟听囪困囤囫坊坑址坍����������������������������������均坎圾坐坏圻壯夾妝妒妨妞妣妙妖妍妤妓妊妥孝孜孚孛完宋宏尬局屁尿尾岐岑岔岌巫希序庇床廷弄弟彤形彷役忘忌志忍忱快忸忪戒我抄抗抖技扶抉扭把扼找批扳抒扯折扮投抓抑抆改攻攸旱更束李杏材村杜杖杞杉杆杠�".split("");

      for (j = 0; j != D[167].length; ++j) if (D[167][j].charCodeAt(0) !== 0xFFFD) {
        e[D[167][j]] = 42752 + j;
        d[42752 + j] = D[167][j];
      }

      D[168] = "����������������������������������������������������������������杓杗步每求汞沙沁沈沉沅沛汪決沐汰沌汨沖沒汽沃汲汾汴沆汶沍沔沘沂灶灼災灸牢牡牠狄狂玖甬甫男甸皂盯矣私秀禿究系罕肖肓肝肘肛肚育良芒����������������������������������芋芍見角言谷豆豕貝赤走足身車辛辰迂迆迅迄巡邑邢邪邦那酉釆里防阮阱阪阬並乖乳事些亞享京佯依侍佳使佬供例來侃佰併侈佩佻侖佾侏侑佺兔兒兕兩具其典冽函刻券刷刺到刮制剁劾劻卒協卓卑卦卷卸卹取叔受味呵�".split("");

      for (j = 0; j != D[168].length; ++j) if (D[168][j].charCodeAt(0) !== 0xFFFD) {
        e[D[168][j]] = 43008 + j;
        d[43008 + j] = D[168][j];
      }

      D[169] = "����������������������������������������������������������������咖呸咕咀呻呷咄咒咆呼咐呱呶和咚呢周咋命咎固垃坷坪坩坡坦坤坼夜奉奇奈奄奔妾妻委妹妮姑姆姐姍始姓姊妯妳姒姅孟孤季宗定官宜宙宛尚屈居����������������������������������屆岷岡岸岩岫岱岳帘帚帖帕帛帑幸庚店府底庖延弦弧弩往征彿彼忝忠忽念忿怏怔怯怵怖怪怕怡性怩怫怛或戕房戾所承拉拌拄抿拂抹拒招披拓拔拋拈抨抽押拐拙拇拍抵拚抱拘拖拗拆抬拎放斧於旺昔易昌昆昂明昀昏昕昊�".split("");

      for (j = 0; j != D[169].length; ++j) if (D[169][j].charCodeAt(0) !== 0xFFFD) {
        e[D[169][j]] = 43264 + j;
        d[43264 + j] = D[169][j];
      }

      D[170] = "����������������������������������������������������������������昇服朋杭枋枕東果杳杷枇枝林杯杰板枉松析杵枚枓杼杪杲欣武歧歿氓氛泣注泳沱泌泥河沽沾沼波沫法泓沸泄油況沮泗泅泱沿治泡泛泊沬泯泜泖泠����������������������������������炕炎炒炊炙爬爭爸版牧物狀狎狙狗狐玩玨玟玫玥甽疝疙疚的盂盲直知矽社祀祁秉秈空穹竺糾罔羌羋者肺肥肢肱股肫肩肴肪肯臥臾舍芳芝芙芭芽芟芹花芬芥芯芸芣芰芾芷虎虱初表軋迎返近邵邸邱邶采金長門阜陀阿阻附�".split("");

      for (j = 0; j != D[170].length; ++j) if (D[170][j].charCodeAt(0) !== 0xFFFD) {
        e[D[170][j]] = 43520 + j;
        d[43520 + j] = D[170][j];
      }

      D[171] = "����������������������������������������������������������������陂隹雨青非亟亭亮信侵侯便俠俑俏保促侶俘俟俊俗侮俐俄係俚俎俞侷兗冒冑冠剎剃削前剌剋則勇勉勃勁匍南卻厚叛咬哀咨哎哉咸咦咳哇哂咽咪品����������������������������������哄哈咯咫咱咻咩咧咿囿垂型垠垣垢城垮垓奕契奏奎奐姜姘姿姣姨娃姥姪姚姦威姻孩宣宦室客宥封屎屏屍屋峙峒巷帝帥帟幽庠度建弈弭彥很待徊律徇後徉怒思怠急怎怨恍恰恨恢恆恃恬恫恪恤扁拜挖按拼拭持拮拽指拱拷�".split("");

      for (j = 0; j != D[171].length; ++j) if (D[171][j].charCodeAt(0) !== 0xFFFD) {
        e[D[171][j]] = 43776 + j;
        d[43776 + j] = D[171][j];
      }

      D[172] = "����������������������������������������������������������������拯括拾拴挑挂政故斫施既春昭映昧是星昨昱昤曷柿染柱柔某柬架枯柵柩柯柄柑枴柚查枸柏柞柳枰柙柢柝柒歪殃殆段毒毗氟泉洋洲洪流津洌洱洞洗����������������������������������活洽派洶洛泵洹洧洸洩洮洵洎洫炫為炳炬炯炭炸炮炤爰牲牯牴狩狠狡玷珊玻玲珍珀玳甚甭畏界畎畋疫疤疥疢疣癸皆皇皈盈盆盃盅省盹相眉看盾盼眇矜砂研砌砍祆祉祈祇禹禺科秒秋穿突竿竽籽紂紅紀紉紇約紆缸美羿耄�".split("");

      for (j = 0; j != D[172].length; ++j) if (D[172][j].charCodeAt(0) !== 0xFFFD) {
        e[D[172][j]] = 44032 + j;
        d[44032 + j] = D[172][j];
      }

      D[173] = "����������������������������������������������������������������耐耍耑耶胖胥胚胃胄背胡胛胎胞胤胝致舢苧范茅苣苛苦茄若茂茉苒苗英茁苜苔苑苞苓苟苯茆虐虹虻虺衍衫要觔計訂訃貞負赴赳趴軍軌述迦迢迪迥����������������������������������迭迫迤迨郊郎郁郃酋酊重閂限陋陌降面革韋韭音頁風飛食首香乘亳倌倍倣俯倦倥俸倩倖倆值借倚倒們俺倀倔倨俱倡個候倘俳修倭倪俾倫倉兼冤冥冢凍凌准凋剖剜剔剛剝匪卿原厝叟哨唐唁唷哼哥哲唆哺唔哩哭員唉哮哪�".split("");

      for (j = 0; j != D[173].length; ++j) if (D[173][j].charCodeAt(0) !== 0xFFFD) {
        e[D[173][j]] = 44288 + j;
        d[44288 + j] = D[173][j];
      }

      D[174] = "����������������������������������������������������������������哦唧唇哽唏圃圄埂埔埋埃堉夏套奘奚娑娘娜娟娛娓姬娠娣娩娥娌娉孫屘宰害家宴宮宵容宸射屑展屐峭峽峻峪峨峰島崁峴差席師庫庭座弱徒徑徐恙����������������������������������恣恥恐恕恭恩息悄悟悚悍悔悌悅悖扇拳挈拿捎挾振捕捂捆捏捉挺捐挽挪挫挨捍捌效敉料旁旅時晉晏晃晒晌晅晁書朔朕朗校核案框桓根桂桔栩梳栗桌桑栽柴桐桀格桃株桅栓栘桁殊殉殷氣氧氨氦氤泰浪涕消涇浦浸海浙涓�".split("");

      for (j = 0; j != D[174].length; ++j) if (D[174][j].charCodeAt(0) !== 0xFFFD) {
        e[D[174][j]] = 44544 + j;
        d[44544 + j] = D[174][j];
      }

      D[175] = "����������������������������������������������������������������浬涉浮浚浴浩涌涊浹涅浥涔烊烘烤烙烈烏爹特狼狹狽狸狷玆班琉珮珠珪珞畔畝畜畚留疾病症疲疳疽疼疹痂疸皋皰益盍盎眩真眠眨矩砰砧砸砝破砷����������������������������������砥砭砠砟砲祕祐祠祟祖神祝祗祚秤秣秧租秦秩秘窄窈站笆笑粉紡紗紋紊素索純紐紕級紜納紙紛缺罟羔翅翁耆耘耕耙耗耽耿胱脂胰脅胭胴脆胸胳脈能脊胼胯臭臬舀舐航舫舨般芻茫荒荔荊茸荐草茵茴荏茲茹茶茗荀茱茨荃�".split("");

      for (j = 0; j != D[175].length; ++j) if (D[175][j].charCodeAt(0) !== 0xFFFD) {
        e[D[175][j]] = 44800 + j;
        d[44800 + j] = D[175][j];
      }

      D[176] = "����������������������������������������������������������������虔蚊蚪蚓蚤蚩蚌蚣蚜衰衷袁袂衽衹記訐討訌訕訊託訓訖訏訑豈豺豹財貢起躬軒軔軏辱送逆迷退迺迴逃追逅迸邕郡郝郢酒配酌釘針釗釜釙閃院陣陡����������������������������������陛陝除陘陞隻飢馬骨高鬥鬲鬼乾偺偽停假偃偌做偉健偶偎偕偵側偷偏倏偯偭兜冕凰剪副勒務勘動匐匏匙匿區匾參曼商啪啦啄啞啡啃啊唱啖問啕唯啤唸售啜唬啣唳啁啗圈國圉域堅堊堆埠埤基堂堵執培夠奢娶婁婉婦婪婀�".split("");

      for (j = 0; j != D[176].length; ++j) if (D[176][j].charCodeAt(0) !== 0xFFFD) {
        e[D[176][j]] = 45056 + j;
        d[45056 + j] = D[176][j];
      }

      D[177] = "����������������������������������������������������������������娼婢婚婆婊孰寇寅寄寂宿密尉專將屠屜屝崇崆崎崛崖崢崑崩崔崙崤崧崗巢常帶帳帷康庸庶庵庾張強彗彬彩彫得徙從徘御徠徜恿患悉悠您惋悴惦悽����������������������������������情悻悵惜悼惘惕惆惟悸惚惇戚戛扈掠控捲掖探接捷捧掘措捱掩掉掃掛捫推掄授掙採掬排掏掀捻捩捨捺敝敖救教敗啟敏敘敕敔斜斛斬族旋旌旎晝晚晤晨晦晞曹勗望梁梯梢梓梵桿桶梱梧梗械梃棄梭梆梅梔條梨梟梡梂欲殺�".split("");

      for (j = 0; j != D[177].length; ++j) if (D[177][j].charCodeAt(0) !== 0xFFFD) {
        e[D[177][j]] = 45312 + j;
        d[45312 + j] = D[177][j];
      }

      D[178] = "����������������������������������������������������������������毫毬氫涎涼淳淙液淡淌淤添淺清淇淋涯淑涮淞淹涸混淵淅淒渚涵淚淫淘淪深淮淨淆淄涪淬涿淦烹焉焊烽烯爽牽犁猜猛猖猓猙率琅琊球理現琍瓠瓶����������������������������������瓷甜產略畦畢異疏痔痕疵痊痍皎盔盒盛眷眾眼眶眸眺硫硃硎祥票祭移窒窕笠笨笛第符笙笞笮粒粗粕絆絃統紮紹紼絀細紳組累終紲紱缽羞羚翌翎習耜聊聆脯脖脣脫脩脰脤舂舵舷舶船莎莞莘荸莢莖莽莫莒莊莓莉莠荷荻荼�".split("");

      for (j = 0; j != D[178].length; ++j) if (D[178][j].charCodeAt(0) !== 0xFFFD) {
        e[D[178][j]] = 45568 + j;
        d[45568 + j] = D[178][j];
      }

      D[179] = "����������������������������������������������������������������莆莧處彪蛇蛀蚶蛄蚵蛆蛋蚱蚯蛉術袞袈被袒袖袍袋覓規訪訝訣訥許設訟訛訢豉豚販責貫貨貪貧赧赦趾趺軛軟這逍通逗連速逝逐逕逞造透逢逖逛途����������������������������������部郭都酗野釵釦釣釧釭釩閉陪陵陳陸陰陴陶陷陬雀雪雩章竟頂頃魚鳥鹵鹿麥麻傢傍傅備傑傀傖傘傚最凱割剴創剩勞勝勛博厥啻喀喧啼喊喝喘喂喜喪喔喇喋喃喳單喟唾喲喚喻喬喱啾喉喫喙圍堯堪場堤堰報堡堝堠壹壺奠�".split("");

      for (j = 0; j != D[179].length; ++j) if (D[179][j].charCodeAt(0) !== 0xFFFD) {
        e[D[179][j]] = 45824 + j;
        d[45824 + j] = D[179][j];
      }

      D[180] = "����������������������������������������������������������������婷媚婿媒媛媧孳孱寒富寓寐尊尋就嵌嵐崴嵇巽幅帽幀幃幾廊廁廂廄弼彭復循徨惑惡悲悶惠愜愣惺愕惰惻惴慨惱愎惶愉愀愒戟扉掣掌描揀揩揉揆揍����������������������������������插揣提握揖揭揮捶援揪換摒揚揹敞敦敢散斑斐斯普晰晴晶景暑智晾晷曾替期朝棺棕棠棘棗椅棟棵森棧棹棒棲棣棋棍植椒椎棉棚楮棻款欺欽殘殖殼毯氮氯氬港游湔渡渲湧湊渠渥渣減湛湘渤湖湮渭渦湯渴湍渺測湃渝渾滋�".split("");

      for (j = 0; j != D[180].length; ++j) if (D[180][j].charCodeAt(0) !== 0xFFFD) {
        e[D[180][j]] = 46080 + j;
        d[46080 + j] = D[180][j];
      }

      D[181] = "����������������������������������������������������������������溉渙湎湣湄湲湩湟焙焚焦焰無然煮焜牌犄犀猶猥猴猩琺琪琳琢琥琵琶琴琯琛琦琨甥甦畫番痢痛痣痙痘痞痠登發皖皓皴盜睏短硝硬硯稍稈程稅稀窘����������������������������������窗窖童竣等策筆筐筒答筍筋筏筑粟粥絞結絨絕紫絮絲絡給絢絰絳善翔翕耋聒肅腕腔腋腑腎脹腆脾腌腓腴舒舜菩萃菸萍菠菅萋菁華菱菴著萊菰萌菌菽菲菊萸萎萄菜萇菔菟虛蛟蛙蛭蛔蛛蛤蛐蛞街裁裂袱覃視註詠評詞証詁�".split("");

      for (j = 0; j != D[181].length; ++j) if (D[181][j].charCodeAt(0) !== 0xFFFD) {
        e[D[181][j]] = 46336 + j;
        d[46336 + j] = D[181][j];
      }

      D[182] = "����������������������������������������������������������������詔詛詐詆訴診訶詖象貂貯貼貳貽賁費賀貴買貶貿貸越超趁跎距跋跚跑跌跛跆軻軸軼辜逮逵週逸進逶鄂郵鄉郾酣酥量鈔鈕鈣鈉鈞鈍鈐鈇鈑閔閏開閑����������������������������������間閒閎隊階隋陽隅隆隍陲隄雁雅雄集雇雯雲韌項順須飧飪飯飩飲飭馮馭黃黍黑亂傭債傲傳僅傾催傷傻傯僇剿剷剽募勦勤勢勣匯嗟嗨嗓嗦嗎嗜嗇嗑嗣嗤嗯嗚嗡嗅嗆嗥嗉園圓塞塑塘塗塚塔填塌塭塊塢塒塋奧嫁嫉嫌媾媽媼�".split("");

      for (j = 0; j != D[182].length; ++j) if (D[182][j].charCodeAt(0) !== 0xFFFD) {
        e[D[182][j]] = 46592 + j;
        d[46592 + j] = D[182][j];
      }

      D[183] = "����������������������������������������������������������������媳嫂媲嵩嵯幌幹廉廈弒彙徬微愚意慈感想愛惹愁愈慎慌慄慍愾愴愧愍愆愷戡戢搓搾搞搪搭搽搬搏搜搔損搶搖搗搆敬斟新暗暉暇暈暖暄暘暍會榔業����������������������������������楚楷楠楔極椰概楊楨楫楞楓楹榆楝楣楛歇歲毀殿毓毽溢溯滓溶滂源溝滇滅溥溘溼溺溫滑準溜滄滔溪溧溴煎煙煩煤煉照煜煬煦煌煥煞煆煨煖爺牒猷獅猿猾瑯瑚瑕瑟瑞瑁琿瑙瑛瑜當畸瘀痰瘁痲痱痺痿痴痳盞盟睛睫睦睞督�".split("");

      for (j = 0; j != D[183].length; ++j) if (D[183][j].charCodeAt(0) !== 0xFFFD) {
        e[D[183][j]] = 46848 + j;
        d[46848 + j] = D[183][j];
      }

      D[184] = "����������������������������������������������������������������睹睪睬睜睥睨睢矮碎碰碗碘碌碉硼碑碓硿祺祿禁萬禽稜稚稠稔稟稞窟窠筷節筠筮筧粱粳粵經絹綑綁綏絛置罩罪署義羨群聖聘肆肄腱腰腸腥腮腳腫����������������������������������腹腺腦舅艇蒂葷落萱葵葦葫葉葬葛萼萵葡董葩葭葆虞虜號蛹蜓蜈蜇蜀蛾蛻蜂蜃蜆蜊衙裟裔裙補裘裝裡裊裕裒覜解詫該詳試詩詰誇詼詣誠話誅詭詢詮詬詹詻訾詨豢貊貉賊資賈賄貲賃賂賅跡跟跨路跳跺跪跤跦躲較載軾輊�".split("");

      for (j = 0; j != D[184].length; ++j) if (D[184][j].charCodeAt(0) !== 0xFFFD) {
        e[D[184][j]] = 47104 + j;
        d[47104 + j] = D[184][j];
      }

      D[185] = "����������������������������������������������������������������辟農運遊道遂達逼違遐遇遏過遍遑逾遁鄒鄗酬酪酩釉鈷鉗鈸鈽鉀鈾鉛鉋鉤鉑鈴鉉鉍鉅鈹鈿鉚閘隘隔隕雍雋雉雊雷電雹零靖靴靶預頑頓頊頒頌飼飴����������������������������������飽飾馳馱馴髡鳩麂鼎鼓鼠僧僮僥僖僭僚僕像僑僱僎僩兢凳劃劂匱厭嗾嘀嘛嘗嗽嘔嘆嘉嘍嘎嗷嘖嘟嘈嘐嗶團圖塵塾境墓墊塹墅塽壽夥夢夤奪奩嫡嫦嫩嫗嫖嫘嫣孵寞寧寡寥實寨寢寤察對屢嶄嶇幛幣幕幗幔廓廖弊彆彰徹慇�".split("");

      for (j = 0; j != D[185].length; ++j) if (D[185][j].charCodeAt(0) !== 0xFFFD) {
        e[D[185][j]] = 47360 + j;
        d[47360 + j] = D[185][j];
      }

      D[186] = "����������������������������������������������������������������愿態慷慢慣慟慚慘慵截撇摘摔撤摸摟摺摑摧搴摭摻敲斡旗旖暢暨暝榜榨榕槁榮槓構榛榷榻榫榴槐槍榭槌榦槃榣歉歌氳漳演滾漓滴漩漾漠漬漏漂漢����������������������������������滿滯漆漱漸漲漣漕漫漯澈漪滬漁滲滌滷熔熙煽熊熄熒爾犒犖獄獐瑤瑣瑪瑰瑭甄疑瘧瘍瘋瘉瘓盡監瞄睽睿睡磁碟碧碳碩碣禎福禍種稱窪窩竭端管箕箋筵算箝箔箏箸箇箄粹粽精綻綰綜綽綾綠緊綴網綱綺綢綿綵綸維緒緇綬�".split("");

      for (j = 0; j != D[186].length; ++j) if (D[186][j].charCodeAt(0) !== 0xFFFD) {
        e[D[186][j]] = 47616 + j;
        d[47616 + j] = D[186][j];
      }

      D[187] = "����������������������������������������������������������������罰翠翡翟聞聚肇腐膀膏膈膊腿膂臧臺與舔舞艋蓉蒿蓆蓄蒙蒞蒲蒜蓋蒸蓀蓓蒐蒼蓑蓊蜿蜜蜻蜢蜥蜴蜘蝕蜷蜩裳褂裴裹裸製裨褚裯誦誌語誣認誡誓誤����������������������������������說誥誨誘誑誚誧豪貍貌賓賑賒赫趙趕跼輔輒輕輓辣遠遘遜遣遙遞遢遝遛鄙鄘鄞酵酸酷酴鉸銀銅銘銖鉻銓銜銨鉼銑閡閨閩閣閥閤隙障際雌雒需靼鞅韶頗領颯颱餃餅餌餉駁骯骰髦魁魂鳴鳶鳳麼鼻齊億儀僻僵價儂儈儉儅凜�".split("");

      for (j = 0; j != D[187].length; ++j) if (D[187][j].charCodeAt(0) !== 0xFFFD) {
        e[D[187][j]] = 47872 + j;
        d[47872 + j] = D[187][j];
      }

      D[188] = "����������������������������������������������������������������劇劈劉劍劊勰厲嘮嘻嘹嘲嘿嘴嘩噓噎噗噴嘶嘯嘰墀墟增墳墜墮墩墦奭嬉嫻嬋嫵嬌嬈寮寬審寫層履嶝嶔幢幟幡廢廚廟廝廣廠彈影德徵慶慧慮慝慕憂����������������������������������慼慰慫慾憧憐憫憎憬憚憤憔憮戮摩摯摹撞撲撈撐撰撥撓撕撩撒撮播撫撚撬撙撢撳敵敷數暮暫暴暱樣樟槨樁樞標槽模樓樊槳樂樅槭樑歐歎殤毅毆漿潼澄潑潦潔澆潭潛潸潮澎潺潰潤澗潘滕潯潠潟熟熬熱熨牖犛獎獗瑩璋璃�".split("");

      for (j = 0; j != D[188].length; ++j) if (D[188][j].charCodeAt(0) !== 0xFFFD) {
        e[D[188][j]] = 48128 + j;
        d[48128 + j] = D[188][j];
      }

      D[189] = "����������������������������������������������������������������瑾璀畿瘠瘩瘟瘤瘦瘡瘢皚皺盤瞎瞇瞌瞑瞋磋磅確磊碾磕碼磐稿稼穀稽稷稻窯窮箭箱範箴篆篇篁箠篌糊締練緯緻緘緬緝編緣線緞緩綞緙緲緹罵罷羯����������������������������������翩耦膛膜膝膠膚膘蔗蔽蔚蓮蔬蔭蔓蔑蔣蔡蔔蓬蔥蓿蔆螂蝴蝶蝠蝦蝸蝨蝙蝗蝌蝓衛衝褐複褒褓褕褊誼諒談諄誕請諸課諉諂調誰論諍誶誹諛豌豎豬賠賞賦賤賬賭賢賣賜質賡赭趟趣踫踐踝踢踏踩踟踡踞躺輝輛輟輩輦輪輜輞�".split("");

      for (j = 0; j != D[189].length; ++j) if (D[189][j].charCodeAt(0) !== 0xFFFD) {
        e[D[189][j]] = 48384 + j;
        d[48384 + j] = D[189][j];
      }

      D[190] = "����������������������������������������������������������������輥適遮遨遭遷鄰鄭鄧鄱醇醉醋醃鋅銻銷鋪銬鋤鋁銳銼鋒鋇鋰銲閭閱霄霆震霉靠鞍鞋鞏頡頫頜颳養餓餒餘駝駐駟駛駑駕駒駙骷髮髯鬧魅魄魷魯鴆鴉����������������������������������鴃麩麾黎墨齒儒儘儔儐儕冀冪凝劑劓勳噙噫噹噩噤噸噪器噥噱噯噬噢噶壁墾壇壅奮嬝嬴學寰導彊憲憑憩憊懍憶憾懊懈戰擅擁擋撻撼據擄擇擂操撿擒擔撾整曆曉暹曄曇暸樽樸樺橙橫橘樹橄橢橡橋橇樵機橈歙歷氅濂澱澡�".split("");

      for (j = 0; j != D[190].length; ++j) if (D[190][j].charCodeAt(0) !== 0xFFFD) {
        e[D[190][j]] = 48640 + j;
        d[48640 + j] = D[190][j];
      }

      D[191] = "����������������������������������������������������������������濃澤濁澧澳激澹澶澦澠澴熾燉燐燒燈燕熹燎燙燜燃燄獨璜璣璘璟璞瓢甌甍瘴瘸瘺盧盥瞠瞞瞟瞥磨磚磬磧禦積穎穆穌穋窺篙簑築篤篛篡篩篦糕糖縊����������������������������������縑縈縛縣縞縝縉縐罹羲翰翱翮耨膳膩膨臻興艘艙蕊蕙蕈蕨蕩蕃蕉蕭蕪蕞螃螟螞螢融衡褪褲褥褫褡親覦諦諺諫諱謀諜諧諮諾謁謂諷諭諳諶諼豫豭貓賴蹄踱踴蹂踹踵輻輯輸輳辨辦遵遴選遲遼遺鄴醒錠錶鋸錳錯錢鋼錫錄錚�".split("");

      for (j = 0; j != D[191].length; ++j) if (D[191][j].charCodeAt(0) !== 0xFFFD) {
        e[D[191][j]] = 48896 + j;
        d[48896 + j] = D[191][j];
      }

      D[192] = "����������������������������������������������������������������錐錦錡錕錮錙閻隧隨險雕霎霑霖霍霓霏靛靜靦鞘頰頸頻頷頭頹頤餐館餞餛餡餚駭駢駱骸骼髻髭鬨鮑鴕鴣鴦鴨鴒鴛默黔龍龜優償儡儲勵嚎嚀嚐嚅嚇����������������������������������嚏壕壓壑壎嬰嬪嬤孺尷屨嶼嶺嶽嶸幫彌徽應懂懇懦懋戲戴擎擊擘擠擰擦擬擱擢擭斂斃曙曖檀檔檄檢檜櫛檣橾檗檐檠歜殮毚氈濘濱濟濠濛濤濫濯澀濬濡濩濕濮濰燧營燮燦燥燭燬燴燠爵牆獰獲璩環璦璨癆療癌盪瞳瞪瞰瞬�".split("");

      for (j = 0; j != D[192].length; ++j) if (D[192][j].charCodeAt(0) !== 0xFFFD) {
        e[D[192][j]] = 49152 + j;
        d[49152 + j] = D[192][j];
      }

      D[193] = "����������������������������������������������������������������瞧瞭矯磷磺磴磯礁禧禪穗窿簇簍篾篷簌篠糠糜糞糢糟糙糝縮績繆縷縲繃縫總縱繅繁縴縹繈縵縿縯罄翳翼聱聲聰聯聳臆臃膺臂臀膿膽臉膾臨舉艱薪����������������������������������薄蕾薜薑薔薯薛薇薨薊虧蟀蟑螳蟒蟆螫螻螺蟈蟋褻褶襄褸褽覬謎謗謙講謊謠謝謄謐豁谿豳賺賽購賸賻趨蹉蹋蹈蹊轄輾轂轅輿避遽還邁邂邀鄹醣醞醜鍍鎂錨鍵鍊鍥鍋錘鍾鍬鍛鍰鍚鍔闊闋闌闈闆隱隸雖霜霞鞠韓顆颶餵騁�".split("");

      for (j = 0; j != D[193].length; ++j) if (D[193][j].charCodeAt(0) !== 0xFFFD) {
        e[D[193][j]] = 49408 + j;
        d[49408 + j] = D[193][j];
      }

      D[194] = "����������������������������������������������������������������駿鮮鮫鮪鮭鴻鴿麋黏點黜黝黛鼾齋叢嚕嚮壙壘嬸彝懣戳擴擲擾攆擺擻擷斷曜朦檳檬櫃檻檸櫂檮檯歟歸殯瀉瀋濾瀆濺瀑瀏燻燼燾燸獷獵璧璿甕癖癘����������������������������������癒瞽瞿瞻瞼礎禮穡穢穠竄竅簫簧簪簞簣簡糧織繕繞繚繡繒繙罈翹翻職聶臍臏舊藏薩藍藐藉薰薺薹薦蟯蟬蟲蟠覆覲觴謨謹謬謫豐贅蹙蹣蹦蹤蹟蹕軀轉轍邇邃邈醫醬釐鎔鎊鎖鎢鎳鎮鎬鎰鎘鎚鎗闔闖闐闕離雜雙雛雞霤鞣鞦�".split("");

      for (j = 0; j != D[194].length; ++j) if (D[194][j].charCodeAt(0) !== 0xFFFD) {
        e[D[194][j]] = 49664 + j;
        d[49664 + j] = D[194][j];
      }

      D[195] = "����������������������������������������������������������������鞭韹額顏題顎顓颺餾餿餽餮馥騎髁鬃鬆魏魎魍鯊鯉鯽鯈鯀鵑鵝鵠黠鼕鼬儳嚥壞壟壢寵龐廬懲懷懶懵攀攏曠曝櫥櫝櫚櫓瀛瀟瀨瀚瀝瀕瀘爆爍牘犢獸����������������������������������獺璽瓊瓣疇疆癟癡矇礙禱穫穩簾簿簸簽簷籀繫繭繹繩繪羅繳羶羹羸臘藩藝藪藕藤藥藷蟻蠅蠍蟹蟾襠襟襖襞譁譜識證譚譎譏譆譙贈贊蹼蹲躇蹶蹬蹺蹴轔轎辭邊邋醱醮鏡鏑鏟鏃鏈鏜鏝鏖鏢鏍鏘鏤鏗鏨關隴難霪霧靡韜韻類�".split("");

      for (j = 0; j != D[195].length; ++j) if (D[195][j].charCodeAt(0) !== 0xFFFD) {
        e[D[195][j]] = 49920 + j;
        d[49920 + j] = D[195][j];
      }

      D[196] = "����������������������������������������������������������������願顛颼饅饉騖騙鬍鯨鯧鯖鯛鶉鵡鵲鵪鵬麒麗麓麴勸嚨嚷嚶嚴嚼壤孀孃孽寶巉懸懺攘攔攙曦朧櫬瀾瀰瀲爐獻瓏癢癥礦礪礬礫竇競籌籃籍糯糰辮繽繼����������������������������������纂罌耀臚艦藻藹蘑藺蘆蘋蘇蘊蠔蠕襤覺觸議譬警譯譟譫贏贍躉躁躅躂醴釋鐘鐃鏽闡霰飄饒饑馨騫騰騷騵鰓鰍鹹麵黨鼯齟齣齡儷儸囁囀囂夔屬巍懼懾攝攜斕曩櫻欄櫺殲灌爛犧瓖瓔癩矓籐纏續羼蘗蘭蘚蠣蠢蠡蠟襪襬覽譴�".split("");

      for (j = 0; j != D[196].length; ++j) if (D[196][j].charCodeAt(0) !== 0xFFFD) {
        e[D[196][j]] = 50176 + j;
        d[50176 + j] = D[196][j];
      }

      D[197] = "����������������������������������������������������������������護譽贓躊躍躋轟辯醺鐮鐳鐵鐺鐸鐲鐫闢霸霹露響顧顥饗驅驃驀騾髏魔魑鰭鰥鶯鶴鷂鶸麝黯鼙齜齦齧儼儻囈囊囉孿巔巒彎懿攤權歡灑灘玀瓤疊癮癬����������������������������������禳籠籟聾聽臟襲襯觼讀贖贗躑躓轡酈鑄鑑鑒霽霾韃韁顫饕驕驍髒鬚鱉鰱鰾鰻鷓鷗鼴齬齪龔囌巖戀攣攫攪曬欐瓚竊籤籣籥纓纖纔臢蘸蘿蠱變邐邏鑣鑠鑤靨顯饜驚驛驗髓體髑鱔鱗鱖鷥麟黴囑壩攬灞癱癲矗罐羈蠶蠹衢讓讒�".split("");

      for (j = 0; j != D[197].length; ++j) if (D[197][j].charCodeAt(0) !== 0xFFFD) {
        e[D[197][j]] = 50432 + j;
        d[50432 + j] = D[197][j];
      }

      D[198] = "����������������������������������������������������������������讖艷贛釀鑪靂靈靄韆顰驟鬢魘鱟鷹鷺鹼鹽鼇齷齲廳欖灣籬籮蠻觀躡釁鑲鑰顱饞髖鬣黌灤矚讚鑷韉驢驥纜讜躪釅鑽鑾鑼鱷鱸黷豔鑿鸚爨驪鬱鸛鸞籲���������������������������������������������������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[198].length; ++j) if (D[198][j].charCodeAt(0) !== 0xFFFD) {
        e[D[198][j]] = 50688 + j;
        d[50688 + j] = D[198][j];
      }

      D[201] = "����������������������������������������������������������������乂乜凵匚厂万丌乇亍囗兀屮彳丏冇与丮亓仂仉仈冘勼卬厹圠夃夬尐巿旡殳毌气爿丱丼仨仜仩仡仝仚刌匜卌圢圣夗夯宁宄尒尻屴屳帄庀庂忉戉扐氕����������������������������������氶汃氿氻犮犰玊禸肊阞伎优伬仵伔仱伀价伈伝伂伅伢伓伄仴伒冱刓刉刐劦匢匟卍厊吇囡囟圮圪圴夼妀奼妅奻奾奷奿孖尕尥屼屺屻屾巟幵庄异弚彴忕忔忏扜扞扤扡扦扢扙扠扚扥旯旮朾朹朸朻机朿朼朳氘汆汒汜汏汊汔汋�".split("");

      for (j = 0; j != D[201].length; ++j) if (D[201][j].charCodeAt(0) !== 0xFFFD) {
        e[D[201][j]] = 51456 + j;
        d[51456 + j] = D[201][j];
      }

      D[202] = "����������������������������������������������������������������汌灱牞犴犵玎甪癿穵网艸艼芀艽艿虍襾邙邗邘邛邔阢阤阠阣佖伻佢佉体佤伾佧佒佟佁佘伭伳伿佡冏冹刜刞刡劭劮匉卣卲厎厏吰吷吪呔呅吙吜吥吘����������������������������������吽呏呁吨吤呇囮囧囥坁坅坌坉坋坒夆奀妦妘妠妗妎妢妐妏妧妡宎宒尨尪岍岏岈岋岉岒岊岆岓岕巠帊帎庋庉庌庈庍弅弝彸彶忒忑忐忭忨忮忳忡忤忣忺忯忷忻怀忴戺抃抌抎抏抔抇扱扻扺扰抁抈扷扽扲扴攷旰旴旳旲旵杅杇�".split("");

      for (j = 0; j != D[202].length; ++j) if (D[202][j].charCodeAt(0) !== 0xFFFD) {
        e[D[202][j]] = 51712 + j;
        d[51712 + j] = D[202][j];
      }

      D[203] = "����������������������������������������������������������������杙杕杌杈杝杍杚杋毐氙氚汸汧汫沄沋沏汱汯汩沚汭沇沕沜汦汳汥汻沎灴灺牣犿犽狃狆狁犺狅玕玗玓玔玒町甹疔疕皁礽耴肕肙肐肒肜芐芏芅芎芑芓����������������������������������芊芃芄豸迉辿邟邡邥邞邧邠阰阨阯阭丳侘佼侅佽侀侇佶佴侉侄佷佌侗佪侚佹侁佸侐侜侔侞侒侂侕佫佮冞冼冾刵刲刳剆刱劼匊匋匼厒厔咇呿咁咑咂咈呫呺呾呥呬呴呦咍呯呡呠咘呣呧呤囷囹坯坲坭坫坱坰坶垀坵坻坳坴坢�".split("");

      for (j = 0; j != D[203].length; ++j) if (D[203][j].charCodeAt(0) !== 0xFFFD) {
        e[D[203][j]] = 51968 + j;
        d[51968 + j] = D[203][j];
      }

      D[204] = "����������������������������������������������������������������坨坽夌奅妵妺姏姎妲姌姁妶妼姃姖妱妽姀姈妴姇孢孥宓宕屄屇岮岤岠岵岯岨岬岟岣岭岢岪岧岝岥岶岰岦帗帔帙弨弢弣弤彔徂彾彽忞忥怭怦怙怲怋����������������������������������怴怊怗怳怚怞怬怢怍怐怮怓怑怌怉怜戔戽抭抴拑抾抪抶拊抮抳抯抻抩抰抸攽斨斻昉旼昄昒昈旻昃昋昍昅旽昑昐曶朊枅杬枎枒杶杻枘枆构杴枍枌杺枟枑枙枃杽极杸杹枔欥殀歾毞氝沓泬泫泮泙沶泔沭泧沷泐泂沺泃泆泭泲�".split("");

      for (j = 0; j != D[204].length; ++j) if (D[204][j].charCodeAt(0) !== 0xFFFD) {
        e[D[204][j]] = 52224 + j;
        d[52224 + j] = D[204][j];
      }

      D[205] = "����������������������������������������������������������������泒泝沴沊沝沀泞泀洰泍泇沰泹泏泩泑炔炘炅炓炆炄炑炖炂炚炃牪狖狋狘狉狜狒狔狚狌狑玤玡玭玦玢玠玬玝瓝瓨甿畀甾疌疘皯盳盱盰盵矸矼矹矻矺����������������������������������矷祂礿秅穸穻竻籵糽耵肏肮肣肸肵肭舠芠苀芫芚芘芛芵芧芮芼芞芺芴芨芡芩苂芤苃芶芢虰虯虭虮豖迒迋迓迍迖迕迗邲邴邯邳邰阹阽阼阺陃俍俅俓侲俉俋俁俔俜俙侻侳俛俇俖侺俀侹俬剄剉勀勂匽卼厗厖厙厘咺咡咭咥哏�".split("");

      for (j = 0; j != D[205].length; ++j) if (D[205][j].charCodeAt(0) !== 0xFFFD) {
        e[D[205][j]] = 52480 + j;
        d[52480 + j] = D[205][j];
      }

      D[206] = "����������������������������������������������������������������哃茍咷咮哖咶哅哆咠呰咼咢咾呲哞咰垵垞垟垤垌垗垝垛垔垘垏垙垥垚垕壴复奓姡姞姮娀姱姝姺姽姼姶姤姲姷姛姩姳姵姠姾姴姭宨屌峐峘峌峗峋峛����������������������������������峞峚峉峇峊峖峓峔峏峈峆峎峟峸巹帡帢帣帠帤庰庤庢庛庣庥弇弮彖徆怷怹恔恲恞恅恓恇恉恛恌恀恂恟怤恄恘恦恮扂扃拏挍挋拵挎挃拫拹挏挌拸拶挀挓挔拺挕拻拰敁敃斪斿昶昡昲昵昜昦昢昳昫昺昝昴昹昮朏朐柁柲柈枺�".split("");

      for (j = 0; j != D[206].length; ++j) if (D[206][j].charCodeAt(0) !== 0xFFFD) {
        e[D[206][j]] = 52736 + j;
        d[52736 + j] = D[206][j];
      }

      D[207] = "����������������������������������������������������������������柜枻柸柘柀枷柅柫柤柟枵柍枳柷柶柮柣柂枹柎柧柰枲柼柆柭柌枮柦柛柺柉柊柃柪柋欨殂殄殶毖毘毠氠氡洨洴洭洟洼洿洒洊泚洳洄洙洺洚洑洀洝浂����������������������������������洁洘洷洃洏浀洇洠洬洈洢洉洐炷炟炾炱炰炡炴炵炩牁牉牊牬牰牳牮狊狤狨狫狟狪狦狣玅珌珂珈珅玹玶玵玴珫玿珇玾珃珆玸珋瓬瓮甮畇畈疧疪癹盄眈眃眄眅眊盷盻盺矧矨砆砑砒砅砐砏砎砉砃砓祊祌祋祅祄秕种秏秖秎窀�".split("");

      for (j = 0; j != D[207].length; ++j) if (D[207][j].charCodeAt(0) !== 0xFFFD) {
        e[D[207][j]] = 52992 + j;
        d[52992 + j] = D[207][j];
      }

      D[208] = "����������������������������������������������������������������穾竑笀笁籺籸籹籿粀粁紃紈紁罘羑羍羾耇耎耏耔耷胘胇胠胑胈胂胐胅胣胙胜胊胕胉胏胗胦胍臿舡芔苙苾苹茇苨茀苕茺苫苖苴苬苡苲苵茌苻苶苰苪����������������������������������苤苠苺苳苭虷虴虼虳衁衎衧衪衩觓訄訇赲迣迡迮迠郱邽邿郕郅邾郇郋郈釔釓陔陏陑陓陊陎倞倅倇倓倢倰倛俵俴倳倷倬俶俷倗倜倠倧倵倯倱倎党冔冓凊凄凅凈凎剡剚剒剞剟剕剢勍匎厞唦哢唗唒哧哳哤唚哿唄唈哫唑唅哱�".split("");

      for (j = 0; j != D[208].length; ++j) if (D[208][j].charCodeAt(0) !== 0xFFFD) {
        e[D[208][j]] = 53248 + j;
        d[53248 + j] = D[208][j];
      }

      D[209] = "����������������������������������������������������������������唊哻哷哸哠唎唃唋圁圂埌堲埕埒垺埆垽垼垸垶垿埇埐垹埁夎奊娙娖娭娮娕娏娗娊娞娳孬宧宭宬尃屖屔峬峿峮峱峷崀峹帩帨庨庮庪庬弳弰彧恝恚恧����������������������������������恁悢悈悀悒悁悝悃悕悛悗悇悜悎戙扆拲挐捖挬捄捅挶捃揤挹捋捊挼挩捁挴捘捔捙挭捇挳捚捑挸捗捀捈敊敆旆旃旄旂晊晟晇晑朒朓栟栚桉栲栳栻桋桏栖栱栜栵栫栭栯桎桄栴栝栒栔栦栨栮桍栺栥栠欬欯欭欱欴歭肂殈毦毤�".split("");

      for (j = 0; j != D[209].length; ++j) if (D[209][j].charCodeAt(0) !== 0xFFFD) {
        e[D[209][j]] = 53504 + j;
        d[53504 + j] = D[209][j];
      }

      D[210] = "����������������������������������������������������������������毨毣毢毧氥浺浣浤浶洍浡涒浘浢浭浯涑涍淯浿涆浞浧浠涗浰浼浟涂涘洯浨涋浾涀涄洖涃浻浽浵涐烜烓烑烝烋缹烢烗烒烞烠烔烍烅烆烇烚烎烡牂牸����������������������������������牷牶猀狺狴狾狶狳狻猁珓珙珥珖玼珧珣珩珜珒珛珔珝珚珗珘珨瓞瓟瓴瓵甡畛畟疰痁疻痄痀疿疶疺皊盉眝眛眐眓眒眣眑眕眙眚眢眧砣砬砢砵砯砨砮砫砡砩砳砪砱祔祛祏祜祓祒祑秫秬秠秮秭秪秜秞秝窆窉窅窋窌窊窇竘笐�".split("");

      for (j = 0; j != D[210].length; ++j) if (D[210][j].charCodeAt(0) !== 0xFFFD) {
        e[D[210][j]] = 53760 + j;
        d[53760 + j] = D[210][j];
      }

      D[211] = "����������������������������������������������������������������笄笓笅笏笈笊笎笉笒粄粑粊粌粈粍粅紞紝紑紎紘紖紓紟紒紏紌罜罡罞罠罝罛羖羒翃翂翀耖耾耹胺胲胹胵脁胻脀舁舯舥茳茭荄茙荑茥荖茿荁茦茜茢����������������������������������荂荎茛茪茈茼荍茖茤茠茷茯茩荇荅荌荓茞茬荋茧荈虓虒蚢蚨蚖蚍蚑蚞蚇蚗蚆蚋蚚蚅蚥蚙蚡蚧蚕蚘蚎蚝蚐蚔衃衄衭衵衶衲袀衱衿衯袃衾衴衼訒豇豗豻貤貣赶赸趵趷趶軑軓迾迵适迿迻逄迼迶郖郠郙郚郣郟郥郘郛郗郜郤酐�".split("");

      for (j = 0; j != D[211].length; ++j) if (D[211][j].charCodeAt(0) !== 0xFFFD) {
        e[D[211][j]] = 54016 + j;
        d[54016 + j] = D[211][j];
      }

      D[212] = "����������������������������������������������������������������酎酏釕釢釚陜陟隼飣髟鬯乿偰偪偡偞偠偓偋偝偲偈偍偁偛偊偢倕偅偟偩偫偣偤偆偀偮偳偗偑凐剫剭剬剮勖勓匭厜啵啶唼啍啐唴唪啑啢唶唵唰啒啅����������������������������������唌唲啥啎唹啈唭唻啀啋圊圇埻堔埢埶埜埴堀埭埽堈埸堋埳埏堇埮埣埲埥埬埡堎埼堐埧堁堌埱埩埰堍堄奜婠婘婕婧婞娸娵婭婐婟婥婬婓婤婗婃婝婒婄婛婈媎娾婍娹婌婰婩婇婑婖婂婜孲孮寁寀屙崞崋崝崚崠崌崨崍崦崥崏�".split("");

      for (j = 0; j != D[212].length; ++j) if (D[212][j].charCodeAt(0) !== 0xFFFD) {
        e[D[212][j]] = 54272 + j;
        d[54272 + j] = D[212][j];
      }

      D[213] = "����������������������������������������������������������������崰崒崣崟崮帾帴庱庴庹庲庳弶弸徛徖徟悊悐悆悾悰悺惓惔惏惤惙惝惈悱惛悷惊悿惃惍惀挲捥掊掂捽掽掞掭掝掗掫掎捯掇掐据掯捵掜捭掮捼掤挻掟����������������������������������捸掅掁掑掍捰敓旍晥晡晛晙晜晢朘桹梇梐梜桭桮梮梫楖桯梣梬梩桵桴梲梏桷梒桼桫桲梪梀桱桾梛梖梋梠梉梤桸桻梑梌梊桽欶欳欷欸殑殏殍殎殌氪淀涫涴涳湴涬淩淢涷淶淔渀淈淠淟淖涾淥淜淝淛淴淊涽淭淰涺淕淂淏淉�".split("");

      for (j = 0; j != D[213].length; ++j) if (D[213][j].charCodeAt(0) !== 0xFFFD) {
        e[D[213][j]] = 54528 + j;
        d[54528 + j] = D[213][j];
      }

      D[214] = "����������������������������������������������������������������淐淲淓淽淗淍淣涻烺焍烷焗烴焌烰焄烳焐烼烿焆焓焀烸烶焋焂焎牾牻牼牿猝猗猇猑猘猊猈狿猏猞玈珶珸珵琄琁珽琇琀珺珼珿琌琋珴琈畤畣痎痒痏����������������������������������痋痌痑痐皏皉盓眹眯眭眱眲眴眳眽眥眻眵硈硒硉硍硊硌砦硅硐祤祧祩祪祣祫祡离秺秸秶秷窏窔窐笵筇笴笥笰笢笤笳笘笪笝笱笫笭笯笲笸笚笣粔粘粖粣紵紽紸紶紺絅紬紩絁絇紾紿絊紻紨罣羕羜羝羛翊翋翍翐翑翇翏翉耟�".split("");

      for (j = 0; j != D[214].length; ++j) if (D[214][j].charCodeAt(0) !== 0xFFFD) {
        e[D[214][j]] = 54784 + j;
        d[54784 + j] = D[214][j];
      }

      D[215] = "����������������������������������������������������������������耞耛聇聃聈脘脥脙脛脭脟脬脞脡脕脧脝脢舑舸舳舺舴舲艴莐莣莨莍荺荳莤荴莏莁莕莙荵莔莩荽莃莌莝莛莪莋荾莥莯莈莗莰荿莦莇莮荶莚虙虖蚿蚷����������������������������������蛂蛁蛅蚺蚰蛈蚹蚳蚸蛌蚴蚻蚼蛃蚽蚾衒袉袕袨袢袪袚袑袡袟袘袧袙袛袗袤袬袌袓袎覂觖觙觕訰訧訬訞谹谻豜豝豽貥赽赻赹趼跂趹趿跁軘軞軝軜軗軠軡逤逋逑逜逌逡郯郪郰郴郲郳郔郫郬郩酖酘酚酓酕釬釴釱釳釸釤釹釪�".split("");

      for (j = 0; j != D[215].length; ++j) if (D[215][j].charCodeAt(0) !== 0xFFFD) {
        e[D[215][j]] = 55040 + j;
        d[55040 + j] = D[215][j];
      }

      D[216] = "����������������������������������������������������������������釫釷釨釮镺閆閈陼陭陫陱陯隿靪頄飥馗傛傕傔傞傋傣傃傌傎傝偨傜傒傂傇兟凔匒匑厤厧喑喨喥喭啷噅喢喓喈喏喵喁喣喒喤啽喌喦啿喕喡喎圌堩堷����������������������������������堙堞堧堣堨埵塈堥堜堛堳堿堶堮堹堸堭堬堻奡媯媔媟婺媢媞婸媦婼媥媬媕媮娷媄媊媗媃媋媩婻婽媌媜媏媓媝寪寍寋寔寑寊寎尌尰崷嵃嵫嵁嵋崿崵嵑嵎嵕崳崺嵒崽崱嵙嵂崹嵉崸崼崲崶嵀嵅幄幁彘徦徥徫惉悹惌惢惎惄愔�".split("");

      for (j = 0; j != D[216].length; ++j) if (D[216][j].charCodeAt(0) !== 0xFFFD) {
        e[D[216][j]] = 55296 + j;
        d[55296 + j] = D[216][j];
      }

      D[217] = "����������������������������������������������������������������惲愊愖愅惵愓惸惼惾惁愃愘愝愐惿愄愋扊掔掱掰揎揥揨揯揃撝揳揊揠揶揕揲揵摡揟掾揝揜揄揘揓揂揇揌揋揈揰揗揙攲敧敪敤敜敨敥斌斝斞斮旐旒����������������������������������晼晬晻暀晱晹晪晲朁椌棓椄棜椪棬棪棱椏棖棷棫棤棶椓椐棳棡椇棌椈楰梴椑棯棆椔棸棐棽棼棨椋椊椗棎棈棝棞棦棴棑椆棔棩椕椥棇欹欻欿欼殔殗殙殕殽毰毲毳氰淼湆湇渟湉溈渼渽湅湢渫渿湁湝湳渜渳湋湀湑渻渃渮湞�".split("");

      for (j = 0; j != D[217].length; ++j) if (D[217][j].charCodeAt(0) !== 0xFFFD) {
        e[D[217][j]] = 55552 + j;
        d[55552 + j] = D[217][j];
      }

      D[218] = "����������������������������������������������������������������湨湜湡渱渨湠湱湫渹渢渰湓湥渧湸湤湷湕湹湒湦渵渶湚焠焞焯烻焮焱焣焥焢焲焟焨焺焛牋牚犈犉犆犅犋猒猋猰猢猱猳猧猲猭猦猣猵猌琮琬琰琫琖����������������������������������琚琡琭琱琤琣琝琩琠琲瓻甯畯畬痧痚痡痦痝痟痤痗皕皒盚睆睇睄睍睅睊睎睋睌矞矬硠硤硥硜硭硱硪确硰硩硨硞硢祴祳祲祰稂稊稃稌稄窙竦竤筊笻筄筈筌筎筀筘筅粢粞粨粡絘絯絣絓絖絧絪絏絭絜絫絒絔絩絑絟絎缾缿罥�".split("");

      for (j = 0; j != D[218].length; ++j) if (D[218][j].charCodeAt(0) !== 0xFFFD) {
        e[D[218][j]] = 55808 + j;
        d[55808 + j] = D[218][j];
      }

      D[219] = "����������������������������������������������������������������罦羢羠羡翗聑聏聐胾胔腃腊腒腏腇脽腍脺臦臮臷臸臹舄舼舽舿艵茻菏菹萣菀菨萒菧菤菼菶萐菆菈菫菣莿萁菝菥菘菿菡菋菎菖菵菉萉萏菞萑萆菂菳����������������������������������菕菺菇菑菪萓菃菬菮菄菻菗菢萛菛菾蛘蛢蛦蛓蛣蛚蛪蛝蛫蛜蛬蛩蛗蛨蛑衈衖衕袺裗袹袸裀袾袶袼袷袽袲褁裉覕覘覗觝觚觛詎詍訹詙詀詗詘詄詅詒詈詑詊詌詏豟貁貀貺貾貰貹貵趄趀趉跘跓跍跇跖跜跏跕跙跈跗跅軯軷軺�".split("");

      for (j = 0; j != D[219].length; ++j) if (D[219][j].charCodeAt(0) !== 0xFFFD) {
        e[D[219][j]] = 56064 + j;
        d[56064 + j] = D[219][j];
      }

      D[220] = "����������������������������������������������������������������軹軦軮軥軵軧軨軶軫軱軬軴軩逭逴逯鄆鄬鄄郿郼鄈郹郻鄁鄀鄇鄅鄃酡酤酟酢酠鈁鈊鈥鈃鈚鈦鈏鈌鈀鈒釿釽鈆鈄鈧鈂鈜鈤鈙鈗鈅鈖镻閍閌閐隇陾隈����������������������������������隉隃隀雂雈雃雱雰靬靰靮頇颩飫鳦黹亃亄亶傽傿僆傮僄僊傴僈僂傰僁傺傱僋僉傶傸凗剺剸剻剼嗃嗛嗌嗐嗋嗊嗝嗀嗔嗄嗩喿嗒喍嗏嗕嗢嗖嗈嗲嗍嗙嗂圔塓塨塤塏塍塉塯塕塎塝塙塥塛堽塣塱壼嫇嫄嫋媺媸媱媵媰媿嫈媻嫆�".split("");

      for (j = 0; j != D[220].length; ++j) if (D[220][j].charCodeAt(0) !== 0xFFFD) {
        e[D[220][j]] = 56320 + j;
        d[56320 + j] = D[220][j];
      }

      D[221] = "����������������������������������������������������������������媷嫀嫊媴媶嫍媹媐寖寘寙尟尳嵱嵣嵊嵥嵲嵬嵞嵨嵧嵢巰幏幎幊幍幋廅廌廆廋廇彀徯徭惷慉慊愫慅愶愲愮慆愯慏愩慀戠酨戣戥戤揅揱揫搐搒搉搠搤����������������������������������搳摃搟搕搘搹搷搢搣搌搦搰搨摁搵搯搊搚摀搥搧搋揧搛搮搡搎敯斒旓暆暌暕暐暋暊暙暔晸朠楦楟椸楎楢楱椿楅楪椹楂楗楙楺楈楉椵楬椳椽楥棰楸椴楩楀楯楄楶楘楁楴楌椻楋椷楜楏楑椲楒椯楻椼歆歅歃歂歈歁殛嗀毻毼�".split("");

      for (j = 0; j != D[221].length; ++j) if (D[221][j].charCodeAt(0) !== 0xFFFD) {
        e[D[221][j]] = 56576 + j;
        d[56576 + j] = D[221][j];
      }

      D[222] = "����������������������������������������������������������������毹毷毸溛滖滈溏滀溟溓溔溠溱溹滆滒溽滁溞滉溷溰滍溦滏溲溾滃滜滘溙溒溎溍溤溡溿溳滐滊溗溮溣煇煔煒煣煠煁煝煢煲煸煪煡煂煘煃煋煰煟煐煓����������������������������������煄煍煚牏犍犌犑犐犎猼獂猻猺獀獊獉瑄瑊瑋瑒瑑瑗瑀瑏瑐瑎瑂瑆瑍瑔瓡瓿瓾瓽甝畹畷榃痯瘏瘃痷痾痼痹痸瘐痻痶痭痵痽皙皵盝睕睟睠睒睖睚睩睧睔睙睭矠碇碚碔碏碄碕碅碆碡碃硹碙碀碖硻祼禂祽祹稑稘稙稒稗稕稢稓�".split("");

      for (j = 0; j != D[222].length; ++j) if (D[222][j].charCodeAt(0) !== 0xFFFD) {
        e[D[222][j]] = 56832 + j;
        d[56832 + j] = D[222][j];
      }

      D[223] = "����������������������������������������������������������������稛稐窣窢窞竫筦筤筭筴筩筲筥筳筱筰筡筸筶筣粲粴粯綈綆綀綍絿綅絺綎絻綃絼綌綔綄絽綒罭罫罧罨罬羦羥羧翛翜耡腤腠腷腜腩腛腢腲朡腞腶腧腯����������������������������������腄腡舝艉艄艀艂艅蓱萿葖葶葹蒏蒍葥葑葀蒆葧萰葍葽葚葙葴葳葝蔇葞萷萺萴葺葃葸萲葅萩菙葋萯葂萭葟葰萹葎葌葒葯蓅蒎萻葇萶萳葨葾葄萫葠葔葮葐蜋蜄蛷蜌蛺蛖蛵蝍蛸蜎蜉蜁蛶蜍蜅裖裋裍裎裞裛裚裌裐覅覛觟觥觤�".split("");

      for (j = 0; j != D[223].length; ++j) if (D[223][j].charCodeAt(0) !== 0xFFFD) {
        e[D[223][j]] = 57088 + j;
        d[57088 + j] = D[223][j];
      }

      D[224] = "����������������������������������������������������������������觡觠觢觜触詶誆詿詡訿詷誂誄詵誃誁詴詺谼豋豊豥豤豦貆貄貅賌赨赩趑趌趎趏趍趓趔趐趒跰跠跬跱跮跐跩跣跢跧跲跫跴輆軿輁輀輅輇輈輂輋遒逿����������������������������������遄遉逽鄐鄍鄏鄑鄖鄔鄋鄎酮酯鉈鉒鈰鈺鉦鈳鉥鉞銃鈮鉊鉆鉭鉬鉏鉠鉧鉯鈶鉡鉰鈱鉔鉣鉐鉲鉎鉓鉌鉖鈲閟閜閞閛隒隓隑隗雎雺雽雸雵靳靷靸靲頏頍頎颬飶飹馯馲馰馵骭骫魛鳪鳭鳧麀黽僦僔僗僨僳僛僪僝僤僓僬僰僯僣僠�".split("");

      for (j = 0; j != D[224].length; ++j) if (D[224][j].charCodeAt(0) !== 0xFFFD) {
        e[D[224][j]] = 57344 + j;
        d[57344 + j] = D[224][j];
      }

      D[225] = "����������������������������������������������������������������凘劀劁勩勫匰厬嘧嘕嘌嘒嗼嘏嘜嘁嘓嘂嗺嘝嘄嗿嗹墉塼墐墘墆墁塿塴墋塺墇墑墎塶墂墈塻墔墏壾奫嫜嫮嫥嫕嫪嫚嫭嫫嫳嫢嫠嫛嫬嫞嫝嫙嫨嫟孷寠����������������������������������寣屣嶂嶀嵽嶆嵺嶁嵷嶊嶉嶈嵾嵼嶍嵹嵿幘幙幓廘廑廗廎廜廕廙廒廔彄彃彯徶愬愨慁慞慱慳慒慓慲慬憀慴慔慺慛慥愻慪慡慖戩戧戫搫摍摛摝摴摶摲摳摽摵摦撦摎撂摞摜摋摓摠摐摿搿摬摫摙摥摷敳斠暡暠暟朅朄朢榱榶槉�".split("");

      for (j = 0; j != D[225].length; ++j) if (D[225][j].charCodeAt(0) !== 0xFFFD) {
        e[D[225][j]] = 57600 + j;
        d[57600 + j] = D[225][j];
      }

      D[226] = "����������������������������������������������������������������榠槎榖榰榬榼榑榙榎榧榍榩榾榯榿槄榽榤槔榹槊榚槏榳榓榪榡榞槙榗榐槂榵榥槆歊歍歋殞殟殠毃毄毾滎滵滱漃漥滸漷滻漮漉潎漙漚漧漘漻漒滭漊����������������������������������漶潳滹滮漭潀漰漼漵滫漇漎潃漅滽滶漹漜滼漺漟漍漞漈漡熇熐熉熀熅熂熏煻熆熁熗牄牓犗犕犓獃獍獑獌瑢瑳瑱瑵瑲瑧瑮甀甂甃畽疐瘖瘈瘌瘕瘑瘊瘔皸瞁睼瞅瞂睮瞀睯睾瞃碲碪碴碭碨硾碫碞碥碠碬碢碤禘禊禋禖禕禔禓�".split("");

      for (j = 0; j != D[226].length; ++j) if (D[226][j].charCodeAt(0) !== 0xFFFD) {
        e[D[226][j]] = 57856 + j;
        d[57856 + j] = D[226][j];
      }

      D[227] = "����������������������������������������������������������������禗禈禒禐稫穊稰稯稨稦窨窫窬竮箈箜箊箑箐箖箍箌箛箎箅箘劄箙箤箂粻粿粼粺綧綷緂綣綪緁緀緅綝緎緄緆緋緌綯綹綖綼綟綦綮綩綡緉罳翢翣翥翞����������������������������������耤聝聜膉膆膃膇膍膌膋舕蒗蒤蒡蒟蒺蓎蓂蒬蒮蒫蒹蒴蓁蓍蒪蒚蒱蓐蒝蒧蒻蒢蒔蓇蓌蒛蒩蒯蒨蓖蒘蒶蓏蒠蓗蓔蓒蓛蒰蒑虡蜳蜣蜨蝫蝀蜮蜞蜡蜙蜛蝃蜬蝁蜾蝆蜠蜲蜪蜭蜼蜒蜺蜱蜵蝂蜦蜧蜸蜤蜚蜰蜑裷裧裱裲裺裾裮裼裶裻�".split("");

      for (j = 0; j != D[227].length; ++j) if (D[227][j].charCodeAt(0) !== 0xFFFD) {
        e[D[227][j]] = 58112 + j;
        d[58112 + j] = D[227][j];
      }

      D[228] = "����������������������������������������������������������������裰裬裫覝覡覟覞觩觫觨誫誙誋誒誏誖谽豨豩賕賏賗趖踉踂跿踍跽踊踃踇踆踅跾踀踄輐輑輎輍鄣鄜鄠鄢鄟鄝鄚鄤鄡鄛酺酲酹酳銥銤鉶銛鉺銠銔銪銍����������������������������������銦銚銫鉹銗鉿銣鋮銎銂銕銢鉽銈銡銊銆銌銙銧鉾銇銩銝銋鈭隞隡雿靘靽靺靾鞃鞀鞂靻鞄鞁靿韎韍頖颭颮餂餀餇馝馜駃馹馻馺駂馽駇骱髣髧鬾鬿魠魡魟鳱鳲鳵麧僿儃儰僸儆儇僶僾儋儌僽儊劋劌勱勯噈噂噌嘵噁噊噉噆噘�".split("");

      for (j = 0; j != D[228].length; ++j) if (D[228][j].charCodeAt(0) !== 0xFFFD) {
        e[D[228][j]] = 58368 + j;
        d[58368 + j] = D[228][j];
      }

      D[229] = "����������������������������������������������������������������噚噀嘳嘽嘬嘾嘸嘪嘺圚墫墝墱墠墣墯墬墥墡壿嫿嫴嫽嫷嫶嬃嫸嬂嫹嬁嬇嬅嬏屧嶙嶗嶟嶒嶢嶓嶕嶠嶜嶡嶚嶞幩幝幠幜緳廛廞廡彉徲憋憃慹憱憰憢憉����������������������������������憛憓憯憭憟憒憪憡憍慦憳戭摮摰撖撠撅撗撜撏撋撊撌撣撟摨撱撘敶敺敹敻斲斳暵暰暩暲暷暪暯樀樆樗槥槸樕槱槤樠槿槬槢樛樝槾樧槲槮樔槷槧橀樈槦槻樍槼槫樉樄樘樥樏槶樦樇槴樖歑殥殣殢殦氁氀毿氂潁漦潾澇濆澒�".split("");

      for (j = 0; j != D[229].length; ++j) if (D[229][j].charCodeAt(0) !== 0xFFFD) {
        e[D[229][j]] = 58624 + j;
        d[58624 + j] = D[229][j];
      }

      D[230] = "����������������������������������������������������������������澍澉澌潢潏澅潚澖潶潬澂潕潲潒潐潗澔澓潝漀潡潫潽潧澐潓澋潩潿澕潣潷潪潻熲熯熛熰熠熚熩熵熝熥熞熤熡熪熜熧熳犘犚獘獒獞獟獠獝獛獡獚獙����������������������������������獢璇璉璊璆璁瑽璅璈瑼瑹甈甇畾瘥瘞瘙瘝瘜瘣瘚瘨瘛皜皝皞皛瞍瞏瞉瞈磍碻磏磌磑磎磔磈磃磄磉禚禡禠禜禢禛歶稹窲窴窳箷篋箾箬篎箯箹篊箵糅糈糌糋緷緛緪緧緗緡縃緺緦緶緱緰緮緟罶羬羰羭翭翫翪翬翦翨聤聧膣膟�".split("");

      for (j = 0; j != D[230].length; ++j) if (D[230][j].charCodeAt(0) !== 0xFFFD) {
        e[D[230][j]] = 58880 + j;
        d[58880 + j] = D[230][j];
      }

      D[231] = "����������������������������������������������������������������膞膕膢膙膗舖艏艓艒艐艎艑蔤蔻蔏蔀蔩蔎蔉蔍蔟蔊蔧蔜蓻蔫蓺蔈蔌蓴蔪蓲蔕蓷蓫蓳蓼蔒蓪蓩蔖蓾蔨蔝蔮蔂蓽蔞蓶蔱蔦蓧蓨蓰蓯蓹蔘蔠蔰蔋蔙蔯虢����������������������������������蝖蝣蝤蝷蟡蝳蝘蝔蝛蝒蝡蝚蝑蝞蝭蝪蝐蝎蝟蝝蝯蝬蝺蝮蝜蝥蝏蝻蝵蝢蝧蝩衚褅褌褔褋褗褘褙褆褖褑褎褉覢覤覣觭觰觬諏諆誸諓諑諔諕誻諗誾諀諅諘諃誺誽諙谾豍貏賥賟賙賨賚賝賧趠趜趡趛踠踣踥踤踮踕踛踖踑踙踦踧�".split("");

      for (j = 0; j != D[231].length; ++j) if (D[231][j].charCodeAt(0) !== 0xFFFD) {
        e[D[231][j]] = 59136 + j;
        d[59136 + j] = D[231][j];
      }

      D[232] = "����������������������������������������������������������������踔踒踘踓踜踗踚輬輤輘輚輠輣輖輗遳遰遯遧遫鄯鄫鄩鄪鄲鄦鄮醅醆醊醁醂醄醀鋐鋃鋄鋀鋙銶鋏鋱鋟鋘鋩鋗鋝鋌鋯鋂鋨鋊鋈鋎鋦鋍鋕鋉鋠鋞鋧鋑鋓����������������������������������銵鋡鋆銴镼閬閫閮閰隤隢雓霅霈霂靚鞊鞎鞈韐韏頞頝頦頩頨頠頛頧颲餈飺餑餔餖餗餕駜駍駏駓駔駎駉駖駘駋駗駌骳髬髫髳髲髱魆魃魧魴魱魦魶魵魰魨魤魬鳼鳺鳽鳿鳷鴇鴀鳹鳻鴈鴅鴄麃黓鼏鼐儜儓儗儚儑凞匴叡噰噠噮�".split("");

      for (j = 0; j != D[232].length; ++j) if (D[232][j].charCodeAt(0) !== 0xFFFD) {
        e[D[232][j]] = 59392 + j;
        d[59392 + j] = D[232][j];
      }

      D[233] = "����������������������������������������������������������������噳噦噣噭噲噞噷圜圛壈墽壉墿墺壂墼壆嬗嬙嬛嬡嬔嬓嬐嬖嬨嬚嬠嬞寯嶬嶱嶩嶧嶵嶰嶮嶪嶨嶲嶭嶯嶴幧幨幦幯廩廧廦廨廥彋徼憝憨憖懅憴懆懁懌憺����������������������������������憿憸憌擗擖擐擏擉撽撉擃擛擳擙攳敿敼斢曈暾曀曊曋曏暽暻暺曌朣樴橦橉橧樲橨樾橝橭橶橛橑樨橚樻樿橁橪橤橐橏橔橯橩橠樼橞橖橕橍橎橆歕歔歖殧殪殫毈毇氄氃氆澭濋澣濇澼濎濈潞濄澽澞濊澨瀄澥澮澺澬澪濏澿澸�".split("");

      for (j = 0; j != D[233].length; ++j) if (D[233][j].charCodeAt(0) !== 0xFFFD) {
        e[D[233][j]] = 59648 + j;
        d[59648 + j] = D[233][j];
      }

      D[234] = "����������������������������������������������������������������澢濉澫濍澯澲澰燅燂熿熸燖燀燁燋燔燊燇燏熽燘熼燆燚燛犝犞獩獦獧獬獥獫獪瑿璚璠璔璒璕璡甋疀瘯瘭瘱瘽瘳瘼瘵瘲瘰皻盦瞚瞝瞡瞜瞛瞢瞣瞕瞙����������������������������������瞗磝磩磥磪磞磣磛磡磢磭磟磠禤穄穈穇窶窸窵窱窷篞篣篧篝篕篥篚篨篹篔篪篢篜篫篘篟糒糔糗糐糑縒縡縗縌縟縠縓縎縜縕縚縢縋縏縖縍縔縥縤罃罻罼罺羱翯耪耩聬膱膦膮膹膵膫膰膬膴膲膷膧臲艕艖艗蕖蕅蕫蕍蕓蕡蕘�".split("");

      for (j = 0; j != D[234].length; ++j) if (D[234][j].charCodeAt(0) !== 0xFFFD) {
        e[D[234][j]] = 59904 + j;
        d[59904 + j] = D[234][j];
      }

      D[235] = "����������������������������������������������������������������蕀蕆蕤蕁蕢蕄蕑蕇蕣蔾蕛蕱蕎蕮蕵蕕蕧蕠薌蕦蕝蕔蕥蕬虣虥虤螛螏螗螓螒螈螁螖螘蝹螇螣螅螐螑螝螄螔螜螚螉褞褦褰褭褮褧褱褢褩褣褯褬褟觱諠����������������������������������諢諲諴諵諝謔諤諟諰諈諞諡諨諿諯諻貑貒貐賵賮賱賰賳赬赮趥趧踳踾踸蹀蹅踶踼踽蹁踰踿躽輶輮輵輲輹輷輴遶遹遻邆郺鄳鄵鄶醓醐醑醍醏錧錞錈錟錆錏鍺錸錼錛錣錒錁鍆錭錎錍鋋錝鋺錥錓鋹鋷錴錂錤鋿錩錹錵錪錔錌�".split("");

      for (j = 0; j != D[235].length; ++j) if (D[235][j].charCodeAt(0) !== 0xFFFD) {
        e[D[235][j]] = 60160 + j;
        d[60160 + j] = D[235][j];
      }

      D[236] = "����������������������������������������������������������������錋鋾錉錀鋻錖閼闍閾閹閺閶閿閵閽隩雔霋霒霐鞙鞗鞔韰韸頵頯頲餤餟餧餩馞駮駬駥駤駰駣駪駩駧骹骿骴骻髶髺髹髷鬳鮀鮅鮇魼魾魻鮂鮓鮒鮐魺鮕����������������������������������魽鮈鴥鴗鴠鴞鴔鴩鴝鴘鴢鴐鴙鴟麈麆麇麮麭黕黖黺鼒鼽儦儥儢儤儠儩勴嚓嚌嚍嚆嚄嚃噾嚂噿嚁壖壔壏壒嬭嬥嬲嬣嬬嬧嬦嬯嬮孻寱寲嶷幬幪徾徻懃憵憼懧懠懥懤懨懞擯擩擣擫擤擨斁斀斶旚曒檍檖檁檥檉檟檛檡檞檇檓檎�".split("");

      for (j = 0; j != D[236].length; ++j) if (D[236][j].charCodeAt(0) !== 0xFFFD) {
        e[D[236][j]] = 60416 + j;
        d[60416 + j] = D[236][j];
      }

      D[237] = "����������������������������������������������������������������檕檃檨檤檑橿檦檚檅檌檒歛殭氉濌澩濴濔濣濜濭濧濦濞濲濝濢濨燡燱燨燲燤燰燢獳獮獯璗璲璫璐璪璭璱璥璯甐甑甒甏疄癃癈癉癇皤盩瞵瞫瞲瞷瞶����������������������������������瞴瞱瞨矰磳磽礂磻磼磲礅磹磾礄禫禨穜穛穖穘穔穚窾竀竁簅簏篲簀篿篻簎篴簋篳簂簉簃簁篸篽簆篰篱簐簊糨縭縼繂縳顈縸縪繉繀繇縩繌縰縻縶繄縺罅罿罾罽翴翲耬膻臄臌臊臅臇膼臩艛艚艜薃薀薏薧薕薠薋薣蕻薤薚薞�".split("");

      for (j = 0; j != D[237].length; ++j) if (D[237][j].charCodeAt(0) !== 0xFFFD) {
        e[D[237][j]] = 60672 + j;
        d[60672 + j] = D[237][j];
      }

      D[238] = "����������������������������������������������������������������蕷蕼薉薡蕺蕸蕗薎薖薆薍薙薝薁薢薂薈薅蕹蕶薘薐薟虨螾螪螭蟅螰螬螹螵螼螮蟉蟃蟂蟌螷螯蟄蟊螴螶螿螸螽蟞螲褵褳褼褾襁襒褷襂覭覯覮觲觳謞����������������������������������謘謖謑謅謋謢謏謒謕謇謍謈謆謜謓謚豏豰豲豱豯貕貔賹赯蹎蹍蹓蹐蹌蹇轃轀邅遾鄸醚醢醛醙醟醡醝醠鎡鎃鎯鍤鍖鍇鍼鍘鍜鍶鍉鍐鍑鍠鍭鎏鍌鍪鍹鍗鍕鍒鍏鍱鍷鍻鍡鍞鍣鍧鎀鍎鍙闇闀闉闃闅閷隮隰隬霠霟霘霝霙鞚鞡鞜�".split("");

      for (j = 0; j != D[238].length; ++j) if (D[238][j].charCodeAt(0) !== 0xFFFD) {
        e[D[238][j]] = 60928 + j;
        d[60928 + j] = D[238][j];
      }

      D[239] = "����������������������������������������������������������������鞞鞝韕韔韱顁顄顊顉顅顃餥餫餬餪餳餲餯餭餱餰馘馣馡騂駺駴駷駹駸駶駻駽駾駼騃骾髾髽鬁髼魈鮚鮨鮞鮛鮦鮡鮥鮤鮆鮢鮠鮯鴳鵁鵧鴶鴮鴯鴱鴸鴰����������������������������������鵅鵂鵃鴾鴷鵀鴽翵鴭麊麉麍麰黈黚黻黿鼤鼣鼢齔龠儱儭儮嚘嚜嚗嚚嚝嚙奰嬼屩屪巀幭幮懘懟懭懮懱懪懰懫懖懩擿攄擽擸攁攃擼斔旛曚曛曘櫅檹檽櫡櫆檺檶檷櫇檴檭歞毉氋瀇瀌瀍瀁瀅瀔瀎濿瀀濻瀦濼濷瀊爁燿燹爃燽獶�".split("");

      for (j = 0; j != D[239].length; ++j) if (D[239][j].charCodeAt(0) !== 0xFFFD) {
        e[D[239][j]] = 61184 + j;
        d[61184 + j] = D[239][j];
      }

      D[240] = "����������������������������������������������������������������璸瓀璵瓁璾璶璻瓂甔甓癜癤癙癐癓癗癚皦皽盬矂瞺磿礌礓礔礉礐礒礑禭禬穟簜簩簙簠簟簭簝簦簨簢簥簰繜繐繖繣繘繢繟繑繠繗繓羵羳翷翸聵臑臒����������������������������������臐艟艞薴藆藀藃藂薳薵薽藇藄薿藋藎藈藅薱薶藒蘤薸薷薾虩蟧蟦蟢蟛蟫蟪蟥蟟蟳蟤蟔蟜蟓蟭蟘蟣螤蟗蟙蠁蟴蟨蟝襓襋襏襌襆襐襑襉謪謧謣謳謰謵譇謯謼謾謱謥謷謦謶謮謤謻謽謺豂豵貙貘貗賾贄贂贀蹜蹢蹠蹗蹖蹞蹥蹧�".split("");

      for (j = 0; j != D[240].length; ++j) if (D[240][j].charCodeAt(0) !== 0xFFFD) {
        e[D[240][j]] = 61440 + j;
        d[61440 + j] = D[240][j];
      }

      D[241] = "����������������������������������������������������������������蹛蹚蹡蹝蹩蹔轆轇轈轋鄨鄺鄻鄾醨醥醧醯醪鎵鎌鎒鎷鎛鎝鎉鎧鎎鎪鎞鎦鎕鎈鎙鎟鎍鎱鎑鎲鎤鎨鎴鎣鎥闒闓闑隳雗雚巂雟雘雝霣霢霥鞬鞮鞨鞫鞤鞪����������������������������������鞢鞥韗韙韖韘韺顐顑顒颸饁餼餺騏騋騉騍騄騑騊騅騇騆髀髜鬈鬄鬅鬩鬵魊魌魋鯇鯆鯃鮿鯁鮵鮸鯓鮶鯄鮹鮽鵜鵓鵏鵊鵛鵋鵙鵖鵌鵗鵒鵔鵟鵘鵚麎麌黟鼁鼀鼖鼥鼫鼪鼩鼨齌齕儴儵劖勷厴嚫嚭嚦嚧嚪嚬壚壝壛夒嬽嬾嬿巃幰�".split("");

      for (j = 0; j != D[241].length; ++j) if (D[241][j].charCodeAt(0) !== 0xFFFD) {
        e[D[241][j]] = 61696 + j;
        d[61696 + j] = D[241][j];
      }

      D[242] = "����������������������������������������������������������������徿懻攇攐攍攉攌攎斄旞旝曞櫧櫠櫌櫑櫙櫋櫟櫜櫐櫫櫏櫍櫞歠殰氌瀙瀧瀠瀖瀫瀡瀢瀣瀩瀗瀤瀜瀪爌爊爇爂爅犥犦犤犣犡瓋瓅璷瓃甖癠矉矊矄矱礝礛����������������������������������礡礜礗礞禰穧穨簳簼簹簬簻糬糪繶繵繸繰繷繯繺繲繴繨罋罊羃羆羷翽翾聸臗臕艤艡艣藫藱藭藙藡藨藚藗藬藲藸藘藟藣藜藑藰藦藯藞藢蠀蟺蠃蟶蟷蠉蠌蠋蠆蟼蠈蟿蠊蠂襢襚襛襗襡襜襘襝襙覈覷覶觶譐譈譊譀譓譖譔譋譕�".split("");

      for (j = 0; j != D[242].length; ++j) if (D[242][j].charCodeAt(0) !== 0xFFFD) {
        e[D[242][j]] = 61952 + j;
        d[61952 + j] = D[242][j];
      }

      D[243] = "����������������������������������������������������������������譑譂譒譗豃豷豶貚贆贇贉趬趪趭趫蹭蹸蹳蹪蹯蹻軂轒轑轏轐轓辴酀鄿醰醭鏞鏇鏏鏂鏚鏐鏹鏬鏌鏙鎩鏦鏊鏔鏮鏣鏕鏄鏎鏀鏒鏧镽闚闛雡霩霫霬霨霦����������������������������������鞳鞷鞶韝韞韟顜顙顝顗颿颽颻颾饈饇饃馦馧騚騕騥騝騤騛騢騠騧騣騞騜騔髂鬋鬊鬎鬌鬷鯪鯫鯠鯞鯤鯦鯢鯰鯔鯗鯬鯜鯙鯥鯕鯡鯚鵷鶁鶊鶄鶈鵱鶀鵸鶆鶋鶌鵽鵫鵴鵵鵰鵩鶅鵳鵻鶂鵯鵹鵿鶇鵨麔麑黀黼鼭齀齁齍齖齗齘匷嚲�".split("");

      for (j = 0; j != D[243].length; ++j) if (D[243][j].charCodeAt(0) !== 0xFFFD) {
        e[D[243][j]] = 62208 + j;
        d[62208 + j] = D[243][j];
      }

      D[244] = "����������������������������������������������������������������嚵嚳壣孅巆巇廮廯忀忁懹攗攖攕攓旟曨曣曤櫳櫰櫪櫨櫹櫱櫮櫯瀼瀵瀯瀷瀴瀱灂瀸瀿瀺瀹灀瀻瀳灁爓爔犨獽獼璺皫皪皾盭矌矎矏矍矲礥礣礧礨礤礩����������������������������������禲穮穬穭竷籉籈籊籇籅糮繻繾纁纀羺翿聹臛臙舋艨艩蘢藿蘁藾蘛蘀藶蘄蘉蘅蘌藽蠙蠐蠑蠗蠓蠖襣襦覹觷譠譪譝譨譣譥譧譭趮躆躈躄轙轖轗轕轘轚邍酃酁醷醵醲醳鐋鐓鏻鐠鐏鐔鏾鐕鐐鐨鐙鐍鏵鐀鏷鐇鐎鐖鐒鏺鐉鏸鐊鏿�".split("");

      for (j = 0; j != D[244].length; ++j) if (D[244][j].charCodeAt(0) !== 0xFFFD) {
        e[D[244][j]] = 62464 + j;
        d[62464 + j] = D[244][j];
      }

      D[245] = "����������������������������������������������������������������鏼鐌鏶鐑鐆闞闠闟霮霯鞹鞻韽韾顠顢顣顟飁飂饐饎饙饌饋饓騲騴騱騬騪騶騩騮騸騭髇髊髆鬐鬒鬑鰋鰈鯷鰅鰒鯸鱀鰇鰎鰆鰗鰔鰉鶟鶙鶤鶝鶒鶘鶐鶛����������������������������������鶠鶔鶜鶪鶗鶡鶚鶢鶨鶞鶣鶿鶩鶖鶦鶧麙麛麚黥黤黧黦鼰鼮齛齠齞齝齙龑儺儹劘劗囃嚽嚾孈孇巋巏廱懽攛欂櫼欃櫸欀灃灄灊灈灉灅灆爝爚爙獾甗癪矐礭礱礯籔籓糲纊纇纈纋纆纍罍羻耰臝蘘蘪蘦蘟蘣蘜蘙蘧蘮蘡蘠蘩蘞蘥�".split("");

      for (j = 0; j != D[245].length; ++j) if (D[245][j].charCodeAt(0) !== 0xFFFD) {
        e[D[245][j]] = 62720 + j;
        d[62720 + j] = D[245][j];
      }

      D[246] = "����������������������������������������������������������������蠩蠝蠛蠠蠤蠜蠫衊襭襩襮襫觺譹譸譅譺譻贐贔趯躎躌轞轛轝酆酄酅醹鐿鐻鐶鐩鐽鐼鐰鐹鐪鐷鐬鑀鐱闥闤闣霵霺鞿韡顤飉飆飀饘饖騹騽驆驄驂驁騺����������������������������������騿髍鬕鬗鬘鬖鬺魒鰫鰝鰜鰬鰣鰨鰩鰤鰡鶷鶶鶼鷁鷇鷊鷏鶾鷅鷃鶻鶵鷎鶹鶺鶬鷈鶱鶭鷌鶳鷍鶲鹺麜黫黮黭鼛鼘鼚鼱齎齥齤龒亹囆囅囋奱孋孌巕巑廲攡攠攦攢欋欈欉氍灕灖灗灒爞爟犩獿瓘瓕瓙瓗癭皭礵禴穰穱籗籜籙籛籚�".split("");

      for (j = 0; j != D[246].length; ++j) if (D[246][j].charCodeAt(0) !== 0xFFFD) {
        e[D[246][j]] = 62976 + j;
        d[62976 + j] = D[246][j];
      }

      D[247] = "����������������������������������������������������������������糴糱纑罏羇臞艫蘴蘵蘳蘬蘲蘶蠬蠨蠦蠪蠥襱覿覾觻譾讄讂讆讅譿贕躕躔躚躒躐躖躗轠轢酇鑌鑐鑊鑋鑏鑇鑅鑈鑉鑆霿韣顪顩飋饔饛驎驓驔驌驏驈驊����������������������������������驉驒驐髐鬙鬫鬻魖魕鱆鱈鰿鱄鰹鰳鱁鰼鰷鰴鰲鰽鰶鷛鷒鷞鷚鷋鷐鷜鷑鷟鷩鷙鷘鷖鷵鷕鷝麶黰鼵鼳鼲齂齫龕龢儽劙壨壧奲孍巘蠯彏戁戃戄攩攥斖曫欑欒欏毊灛灚爢玂玁玃癰矔籧籦纕艬蘺虀蘹蘼蘱蘻蘾蠰蠲蠮蠳襶襴襳觾�".split("");

      for (j = 0; j != D[247].length; ++j) if (D[247][j].charCodeAt(0) !== 0xFFFD) {
        e[D[247][j]] = 63232 + j;
        d[63232 + j] = D[247][j];
      }

      D[248] = "����������������������������������������������������������������讌讎讋讈豅贙躘轤轣醼鑢鑕鑝鑗鑞韄韅頀驖驙鬞鬟鬠鱒鱘鱐鱊鱍鱋鱕鱙鱌鱎鷻鷷鷯鷣鷫鷸鷤鷶鷡鷮鷦鷲鷰鷢鷬鷴鷳鷨鷭黂黐黲黳鼆鼜鼸鼷鼶齃齏����������������������������������齱齰齮齯囓囍孎屭攭曭曮欓灟灡灝灠爣瓛瓥矕礸禷禶籪纗羉艭虃蠸蠷蠵衋讔讕躞躟躠躝醾醽釂鑫鑨鑩雥靆靃靇韇韥驞髕魙鱣鱧鱦鱢鱞鱠鸂鷾鸇鸃鸆鸅鸀鸁鸉鷿鷽鸄麠鼞齆齴齵齶囔攮斸欘欙欗欚灢爦犪矘矙礹籩籫糶纚�".split("");

      for (j = 0; j != D[248].length; ++j) if (D[248][j].charCodeAt(0) !== 0xFFFD) {
        e[D[248][j]] = 63488 + j;
        d[63488 + j] = D[248][j];
      }

      D[249] = "����������������������������������������������������������������纘纛纙臠臡虆虇虈襹襺襼襻觿讘讙躥躤躣鑮鑭鑯鑱鑳靉顲饟鱨鱮鱭鸋鸍鸐鸏鸒鸑麡黵鼉齇齸齻齺齹圞灦籯蠼趲躦釃鑴鑸鑶鑵驠鱴鱳鱱鱵鸔鸓黶鼊����������������������������������龤灨灥糷虪蠾蠽蠿讞貜躩軉靋顳顴飌饡馫驤驦驧鬤鸕鸗齈戇欞爧虌躨钂钀钁驩驨鬮鸙爩虋讟钃鱹麷癵驫鱺鸝灩灪麤齾齉龘碁銹裏墻恒粧嫺╔╦╗╠╬╣╚╩╝╒╤╕╞╪╡╘╧╛╓╥╖╟╫╢╙╨╜║═╭╮╰╯▓�".split("");

      for (j = 0; j != D[249].length; ++j) if (D[249][j].charCodeAt(0) !== 0xFFFD) {
        e[D[249][j]] = 63744 + j;
        d[63744 + j] = D[249][j];
      }

      return {
        "enc": e,
        "dec": d
      };
    }();

    cptable[1250] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚�„…†‡�‰Š‹ŚŤŽŹ�‘’“”•–—�™š›śťžź ˇ˘Ł¤Ą¦§¨©Ş«¬­®Ż°±˛ł´µ¶·¸ąş»Ľ˝ľżŔÁÂĂÄĹĆÇČÉĘËĚÍÎĎĐŃŇÓÔŐÖ×ŘŮÚŰÜÝŢßŕáâăäĺćçčéęëěíîďđńňóôőö÷řůúűüýţ˙",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1251] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ЂЃ‚ѓ„…†‡€‰Љ‹ЊЌЋЏђ‘’“”•–—�™љ›њќћџ ЎўЈ¤Ґ¦§Ё©Є«¬­®Ї°±Ііґµ¶·ё№є»јЅѕїАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюя",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1252] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚ƒ„…†‡ˆ‰Š‹Œ�Ž��‘’“”•–—˜™š›œ�žŸ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1253] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚ƒ„…†‡�‰�‹�����‘’“”•–—�™�›���� ΅Ά£¤¥¦§¨©�«¬­®―°±²³΄µ¶·ΈΉΊ»Ό½ΎΏΐΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡ�ΣΤΥΦΧΨΩΪΫάέήίΰαβγδεζηθικλμνξοπρςστυφχψωϊϋόύώ�",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1254] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚ƒ„…†‡ˆ‰Š‹Œ����‘’“”•–—˜™š›œ��Ÿ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏĞÑÒÓÔÕÖ×ØÙÚÛÜİŞßàáâãäåæçèéêëìíîïğñòóôõö÷øùúûüışÿ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1255] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚ƒ„…†‡ˆ‰�‹�����‘’“”•–—˜™�›���� ¡¢£₪¥¦§¨©×«¬­®¯°±²³´µ¶·¸¹÷»¼½¾¿ְֱֲֳִֵֶַָֹ�ֻּֽ־ֿ׀ׁׂ׃װױײ׳״�������אבגדהוזחטיךכלםמןנסעףפץצקרשת��‎‏�",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1256] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€پ‚ƒ„…†‡ˆ‰ٹ‹Œچژڈگ‘’“”•–—ک™ڑ›œ‌‍ں ،¢£¤¥¦§¨©ھ«¬­®¯°±²³´µ¶·¸¹؛»¼½¾؟ہءآأؤإئابةتثجحخدذرزسشصض×طظعغـفقكàلâمنهوçèéêëىيîïًٌٍَôُِ÷ّùْûü‎‏ے",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1257] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚�„…†‡�‰�‹�¨ˇ¸�‘’“”•–—�™�›�¯˛� �¢£¤�¦§Ø©Ŗ«¬­®Æ°±²³´µ¶·ø¹ŗ»¼½¾æĄĮĀĆÄÅĘĒČÉŹĖĢĶĪĻŠŃŅÓŌÕÖ×ŲŁŚŪÜŻŽßąįāćäåęēčéźėģķīļšńņóōõö÷ųłśūüżž˙",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[1258] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~€�‚ƒ„…†‡ˆ‰�‹Œ����‘’“”•–—˜™�›œ��Ÿ ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂĂÄÅÆÇÈÉÊË̀ÍÎÏĐÑ̉ÓÔƠÖ×ØÙÚÛÜỮßàáâăäåæçèéêë́íîïđṇ̃óôơö÷øùúûüư₫ÿ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10000] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤‹›ﬁﬂ‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10006] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~Ä¹²É³ÖÜ΅àâä΄¨çéèêë£™îï•½‰ôö¦­ùûü†ΓΔΘΛΞΠß®©ΣΪ§≠°·Α±≤≥¥ΒΕΖΗΙΚΜΦΫΨΩάΝ¬ΟΡ≈Τ«»… ΥΧΆΈœ–―“”‘’÷ΉΊΌΎέήίόΏύαβψδεφγηιξκλμνοπώρστθωςχυζϊϋΐΰ�",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10007] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~АБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ†°¢£§•¶І®©™Ђђ≠Ѓѓ∞±≤≥іµ∂ЈЄєЇїЉљЊњјЅ¬√ƒ≈∆«»… ЋћЌќѕ–—“”‘’÷„ЎўЏџ№Ёёяабвгдежзийклмнопрстуфхцчшщъыьэю¤",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10008] = function () {
      var d = [],
          e = {},
          D = [],
          j;
      D[0] = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~���������������������������������������������������������������������������������������".split("");

      for (j = 0; j != D[0].length; ++j) if (D[0][j].charCodeAt(0) !== 0xFFFD) {
        e[D[0][j]] = 0 + j;
        d[0 + j] = D[0][j];
      }

      D[161] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������　、。・ˉˇ¨〃々―～�…‘’“”〔〕〈〉《》「」『』〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓�".split("");

      for (j = 0; j != D[161].length; ++j) if (D[161][j].charCodeAt(0) !== 0xFFFD) {
        e[D[161][j]] = 41216 + j;
        d[41216 + j] = D[161][j];
      }

      D[162] = "���������������������������������������������������������������������������������������������������������������������������������������������������������������������������������⒈⒉⒊⒋⒌⒍⒎⒏⒐⒑⒒⒓⒔⒕⒖⒗⒘⒙⒚⒛⑴⑵⑶⑷⑸⑹⑺⑻⑼⑽⑾⑿⒀⒁⒂⒃⒄⒅⒆⒇①②③④⑤⑥⑦⑧⑨⑩��㈠㈡㈢㈣㈤㈥㈦㈧㈨㈩��ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩⅪⅫ���".split("");

      for (j = 0; j != D[162].length; ++j) if (D[162][j].charCodeAt(0) !== 0xFFFD) {
        e[D[162][j]] = 41472 + j;
        d[41472 + j] = D[162][j];
      }

      D[163] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������！＂＃￥％＆＇（）＊＋，－．／０１２３４５６７８９：；＜＝＞？＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ［＼］＾＿｀ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ｛｜｝￣�".split("");

      for (j = 0; j != D[163].length; ++j) if (D[163][j].charCodeAt(0) !== 0xFFFD) {
        e[D[163][j]] = 41728 + j;
        d[41728 + j] = D[163][j];
      }

      D[164] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ぁあぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとどなにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをん������������".split("");

      for (j = 0; j != D[164].length; ++j) if (D[164][j].charCodeAt(0) !== 0xFFFD) {
        e[D[164][j]] = 41984 + j;
        d[41984 + j] = D[164][j];
      }

      D[165] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ァアィイゥウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトドナニヌネノハバパヒビピフブプヘベペホボポマミムメモャヤュユョヨラリルレロヮワヰヱヲンヴヵヶ���������".split("");

      for (j = 0; j != D[165].length; ++j) if (D[165][j].charCodeAt(0) !== 0xFFFD) {
        e[D[165][j]] = 42240 + j;
        d[42240 + j] = D[165][j];
      }

      D[166] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ��������αβγδεζηθικλμνξοπρστυφχψω���������������������������������������".split("");

      for (j = 0; j != D[166].length; ++j) if (D[166][j].charCodeAt(0) !== 0xFFFD) {
        e[D[166][j]] = 42496 + j;
        d[42496 + j] = D[166][j];
      }

      D[167] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ���������������абвгдеёжзийклмнопрстуфхцчшщъыьэюя��������������".split("");

      for (j = 0; j != D[167].length; ++j) if (D[167][j].charCodeAt(0) !== 0xFFFD) {
        e[D[167][j]] = 42752 + j;
        d[42752 + j] = D[167][j];
      }

      D[168] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüê����������ㄅㄆㄇㄈㄉㄊㄋㄌㄍㄎㄏㄐㄑㄒㄓㄔㄕㄖㄗㄘㄙㄚㄛㄜㄝㄞㄟㄠㄡㄢㄣㄤㄥㄦㄧㄨㄩ����������������������".split("");

      for (j = 0; j != D[168].length; ++j) if (D[168][j].charCodeAt(0) !== 0xFFFD) {
        e[D[168][j]] = 43008 + j;
        d[43008 + j] = D[168][j];
      }

      D[169] = "��������������������������������������������������������������������������������������������������������������������������������������������������������������������─━│┃┄┅┆┇┈┉┊┋┌┍┎┏┐┑┒┓└┕┖┗┘┙┚┛├┝┞┟┠┡┢┣┤┥┦┧┨┩┪┫┬┭┮┯┰┱┲┳┴┵┶┷┸┹┺┻┼┽┾┿╀╁╂╃╄╅╆╇╈╉╊╋����������������".split("");

      for (j = 0; j != D[169].length; ++j) if (D[169][j].charCodeAt(0) !== 0xFFFD) {
        e[D[169][j]] = 43264 + j;
        d[43264 + j] = D[169][j];
      }

      D[176] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥�".split("");

      for (j = 0; j != D[176].length; ++j) if (D[176][j].charCodeAt(0) !== 0xFFFD) {
        e[D[176][j]] = 45056 + j;
        d[45056 + j] = D[176][j];
      }

      D[177] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳�".split("");

      for (j = 0; j != D[177].length; ++j) if (D[177][j].charCodeAt(0) !== 0xFFFD) {
        e[D[177][j]] = 45312 + j;
        d[45312 + j] = D[177][j];
      }

      D[178] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖�".split("");

      for (j = 0; j != D[178].length; ++j) if (D[178][j].charCodeAt(0) !== 0xFFFD) {
        e[D[178][j]] = 45568 + j;
        d[45568 + j] = D[178][j];
      }

      D[179] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚�".split("");

      for (j = 0; j != D[179].length; ++j) if (D[179][j].charCodeAt(0) !== 0xFFFD) {
        e[D[179][j]] = 45824 + j;
        d[45824 + j] = D[179][j];
      }

      D[180] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮�".split("");

      for (j = 0; j != D[180].length; ++j) if (D[180][j].charCodeAt(0) !== 0xFFFD) {
        e[D[180][j]] = 46080 + j;
        d[46080 + j] = D[180][j];
      }

      D[181] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠�".split("");

      for (j = 0; j != D[181].length; ++j) if (D[181][j].charCodeAt(0) !== 0xFFFD) {
        e[D[181][j]] = 46336 + j;
        d[46336 + j] = D[181][j];
      }

      D[182] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二�".split("");

      for (j = 0; j != D[182].length; ++j) if (D[182][j].charCodeAt(0) !== 0xFFFD) {
        e[D[182][j]] = 46592 + j;
        d[46592 + j] = D[182][j];
      }

      D[183] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服�".split("");

      for (j = 0; j != D[183].length; ++j) if (D[183][j].charCodeAt(0) !== 0xFFFD) {
        e[D[183][j]] = 46848 + j;
        d[46848 + j] = D[183][j];
      }

      D[184] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹�".split("");

      for (j = 0; j != D[184].length; ++j) if (D[184][j].charCodeAt(0) !== 0xFFFD) {
        e[D[184][j]] = 47104 + j;
        d[47104 + j] = D[184][j];
      }

      D[185] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈�".split("");

      for (j = 0; j != D[185].length; ++j) if (D[185][j].charCodeAt(0) !== 0xFFFD) {
        e[D[185][j]] = 47360 + j;
        d[47360 + j] = D[185][j];
      }

      D[186] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖�".split("");

      for (j = 0; j != D[186].length; ++j) if (D[186][j].charCodeAt(0) !== 0xFFFD) {
        e[D[186][j]] = 47616 + j;
        d[47616 + j] = D[186][j];
      }

      D[187] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕�".split("");

      for (j = 0; j != D[187].length; ++j) if (D[187][j].charCodeAt(0) !== 0xFFFD) {
        e[D[187][j]] = 47872 + j;
        d[47872 + j] = D[187][j];
      }

      D[188] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件�".split("");

      for (j = 0; j != D[188].length; ++j) if (D[188][j].charCodeAt(0) !== 0xFFFD) {
        e[D[188][j]] = 48128 + j;
        d[48128 + j] = D[188][j];
      }

      D[189] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸�".split("");

      for (j = 0; j != D[189].length; ++j) if (D[189][j].charCodeAt(0) !== 0xFFFD) {
        e[D[189][j]] = 48384 + j;
        d[48384 + j] = D[189][j];
      }

      D[190] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻�".split("");

      for (j = 0; j != D[190].length; ++j) if (D[190][j].charCodeAt(0) !== 0xFFFD) {
        e[D[190][j]] = 48640 + j;
        d[48640 + j] = D[190][j];
      }

      D[191] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀�".split("");

      for (j = 0; j != D[191].length; ++j) if (D[191][j].charCodeAt(0) !== 0xFFFD) {
        e[D[191][j]] = 48896 + j;
        d[48896 + j] = D[191][j];
      }

      D[192] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐�".split("");

      for (j = 0; j != D[192].length; ++j) if (D[192][j].charCodeAt(0) !== 0xFFFD) {
        e[D[192][j]] = 49152 + j;
        d[49152 + j] = D[192][j];
      }

      D[193] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿�".split("");

      for (j = 0; j != D[193].length; ++j) if (D[193][j].charCodeAt(0) !== 0xFFFD) {
        e[D[193][j]] = 49408 + j;
        d[49408 + j] = D[193][j];
      }

      D[194] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫�".split("");

      for (j = 0; j != D[194].length; ++j) if (D[194][j].charCodeAt(0) !== 0xFFFD) {
        e[D[194][j]] = 49664 + j;
        d[49664 + j] = D[194][j];
      }

      D[195] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸�".split("");

      for (j = 0; j != D[195].length; ++j) if (D[195][j].charCodeAt(0) !== 0xFFFD) {
        e[D[195][j]] = 49920 + j;
        d[49920 + j] = D[195][j];
      }

      D[196] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁�".split("");

      for (j = 0; j != D[196].length; ++j) if (D[196][j].charCodeAt(0) !== 0xFFFD) {
        e[D[196][j]] = 50176 + j;
        d[50176 + j] = D[196][j];
      }

      D[197] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗�".split("");

      for (j = 0; j != D[197].length; ++j) if (D[197][j].charCodeAt(0) !== 0xFFFD) {
        e[D[197][j]] = 50432 + j;
        d[50432 + j] = D[197][j];
      }

      D[198] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐�".split("");

      for (j = 0; j != D[198].length; ++j) if (D[198][j].charCodeAt(0) !== 0xFFFD) {
        e[D[198][j]] = 50688 + j;
        d[50688 + j] = D[198][j];
      }

      D[199] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠�".split("");

      for (j = 0; j != D[199].length; ++j) if (D[199][j].charCodeAt(0) !== 0xFFFD) {
        e[D[199][j]] = 50944 + j;
        d[50944 + j] = D[199][j];
      }

      D[200] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁�".split("");

      for (j = 0; j != D[200].length; ++j) if (D[200][j].charCodeAt(0) !== 0xFFFD) {
        e[D[200][j]] = 51200 + j;
        d[51200 + j] = D[200][j];
      }

      D[201] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳�".split("");

      for (j = 0; j != D[201].length; ++j) if (D[201][j].charCodeAt(0) !== 0xFFFD) {
        e[D[201][j]] = 51456 + j;
        d[51456 + j] = D[201][j];
      }

      D[202] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱�".split("");

      for (j = 0; j != D[202].length; ++j) if (D[202][j].charCodeAt(0) !== 0xFFFD) {
        e[D[202][j]] = 51712 + j;
        d[51712 + j] = D[202][j];
      }

      D[203] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔�".split("");

      for (j = 0; j != D[203].length; ++j) if (D[203][j].charCodeAt(0) !== 0xFFFD) {
        e[D[203][j]] = 51968 + j;
        d[51968 + j] = D[203][j];
      }

      D[204] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃�".split("");

      for (j = 0; j != D[204].length; ++j) if (D[204][j].charCodeAt(0) !== 0xFFFD) {
        e[D[204][j]] = 52224 + j;
        d[52224 + j] = D[204][j];
      }

      D[205] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威�".split("");

      for (j = 0; j != D[205].length; ++j) if (D[205][j].charCodeAt(0) !== 0xFFFD) {
        e[D[205][j]] = 52480 + j;
        d[52480 + j] = D[205][j];
      }

      D[206] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺�".split("");

      for (j = 0; j != D[206].length; ++j) if (D[206][j].charCodeAt(0) !== 0xFFFD) {
        e[D[206][j]] = 52736 + j;
        d[52736 + j] = D[206][j];
      }

      D[207] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓�".split("");

      for (j = 0; j != D[207].length; ++j) if (D[207][j].charCodeAt(0) !== 0xFFFD) {
        e[D[207][j]] = 52992 + j;
        d[52992 + j] = D[207][j];
      }

      D[208] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄�".split("");

      for (j = 0; j != D[208].length; ++j) if (D[208][j].charCodeAt(0) !== 0xFFFD) {
        e[D[208][j]] = 53248 + j;
        d[53248 + j] = D[208][j];
      }

      D[209] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶�".split("");

      for (j = 0; j != D[209].length; ++j) if (D[209][j].charCodeAt(0) !== 0xFFFD) {
        e[D[209][j]] = 53504 + j;
        d[53504 + j] = D[209][j];
      }

      D[210] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐�".split("");

      for (j = 0; j != D[210].length; ++j) if (D[210][j].charCodeAt(0) !== 0xFFFD) {
        e[D[210][j]] = 53760 + j;
        d[53760 + j] = D[210][j];
      }

      D[211] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉�".split("");

      for (j = 0; j != D[211].length; ++j) if (D[211][j].charCodeAt(0) !== 0xFFFD) {
        e[D[211][j]] = 54016 + j;
        d[54016 + j] = D[211][j];
      }

      D[212] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧�".split("");

      for (j = 0; j != D[212].length; ++j) if (D[212][j].charCodeAt(0) !== 0xFFFD) {
        e[D[212][j]] = 54272 + j;
        d[54272 + j] = D[212][j];
      }

      D[213] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政�".split("");

      for (j = 0; j != D[213].length; ++j) if (D[213][j].charCodeAt(0) !== 0xFFFD) {
        e[D[213][j]] = 54528 + j;
        d[54528 + j] = D[213][j];
      }

      D[214] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑�".split("");

      for (j = 0; j != D[214].length; ++j) if (D[214][j].charCodeAt(0) !== 0xFFFD) {
        e[D[214][j]] = 54784 + j;
        d[54784 + j] = D[214][j];
      }

      D[215] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座������".split("");

      for (j = 0; j != D[215].length; ++j) if (D[215][j].charCodeAt(0) !== 0xFFFD) {
        e[D[215][j]] = 55040 + j;
        d[55040 + j] = D[215][j];
      }

      D[216] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝�".split("");

      for (j = 0; j != D[216].length; ++j) if (D[216][j].charCodeAt(0) !== 0xFFFD) {
        e[D[216][j]] = 55296 + j;
        d[55296 + j] = D[216][j];
      }

      D[217] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼�".split("");

      for (j = 0; j != D[217].length; ++j) if (D[217][j].charCodeAt(0) !== 0xFFFD) {
        e[D[217][j]] = 55552 + j;
        d[55552 + j] = D[217][j];
      }

      D[218] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺�".split("");

      for (j = 0; j != D[218].length; ++j) if (D[218][j].charCodeAt(0) !== 0xFFFD) {
        e[D[218][j]] = 55808 + j;
        d[55808 + j] = D[218][j];
      }

      D[219] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝�".split("");

      for (j = 0; j != D[219].length; ++j) if (D[219][j].charCodeAt(0) !== 0xFFFD) {
        e[D[219][j]] = 56064 + j;
        d[56064 + j] = D[219][j];
      }

      D[220] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥�".split("");

      for (j = 0; j != D[220].length; ++j) if (D[220][j].charCodeAt(0) !== 0xFFFD) {
        e[D[220][j]] = 56320 + j;
        d[56320 + j] = D[220][j];
      }

      D[221] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺�".split("");

      for (j = 0; j != D[221].length; ++j) if (D[221][j].charCodeAt(0) !== 0xFFFD) {
        e[D[221][j]] = 56576 + j;
        d[56576 + j] = D[221][j];
      }

      D[222] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖�".split("");

      for (j = 0; j != D[222].length; ++j) if (D[222][j].charCodeAt(0) !== 0xFFFD) {
        e[D[222][j]] = 56832 + j;
        d[56832 + j] = D[222][j];
      }

      D[223] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼�".split("");

      for (j = 0; j != D[223].length; ++j) if (D[223][j].charCodeAt(0) !== 0xFFFD) {
        e[D[223][j]] = 57088 + j;
        d[57088 + j] = D[223][j];
      }

      D[224] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼�".split("");

      for (j = 0; j != D[224].length; ++j) if (D[224][j].charCodeAt(0) !== 0xFFFD) {
        e[D[224][j]] = 57344 + j;
        d[57344 + j] = D[224][j];
      }

      D[225] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺�".split("");

      for (j = 0; j != D[225].length; ++j) if (D[225][j].charCodeAt(0) !== 0xFFFD) {
        e[D[225][j]] = 57600 + j;
        d[57600 + j] = D[225][j];
      }

      D[226] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧饨饩饪饫饬饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂�".split("");

      for (j = 0; j != D[226].length; ++j) if (D[226][j].charCodeAt(0) !== 0xFFFD) {
        e[D[226][j]] = 57856 + j;
        d[57856 + j] = D[226][j];
      }

      D[227] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾�".split("");

      for (j = 0; j != D[227].length; ++j) if (D[227][j].charCodeAt(0) !== 0xFFFD) {
        e[D[227][j]] = 58112 + j;
        d[58112 + j] = D[227][j];
      }

      D[228] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑�".split("");

      for (j = 0; j != D[228].length; ++j) if (D[228][j].charCodeAt(0) !== 0xFFFD) {
        e[D[228][j]] = 58368 + j;
        d[58368 + j] = D[228][j];
      }

      D[229] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣�".split("");

      for (j = 0; j != D[229].length; ++j) if (D[229][j].charCodeAt(0) !== 0xFFFD) {
        e[D[229][j]] = 58624 + j;
        d[58624 + j] = D[229][j];
      }

      D[230] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩�".split("");

      for (j = 0; j != D[230].length; ++j) if (D[230][j].charCodeAt(0) !== 0xFFFD) {
        e[D[230][j]] = 58880 + j;
        d[58880 + j] = D[230][j];
      }

      D[231] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡缢缣缤缥缦缧缪缫缬缭缯缰缱缲缳缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬�".split("");

      for (j = 0; j != D[231].length; ++j) if (D[231][j].charCodeAt(0) !== 0xFFFD) {
        e[D[231][j]] = 59136 + j;
        d[59136 + j] = D[231][j];
      }

      D[232] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹�".split("");

      for (j = 0; j != D[232].length; ++j) if (D[232][j].charCodeAt(0) !== 0xFFFD) {
        e[D[232][j]] = 59392 + j;
        d[59392 + j] = D[232][j];
      }

      D[233] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋�".split("");

      for (j = 0; j != D[233].length; ++j) if (D[233][j].charCodeAt(0) !== 0xFFFD) {
        e[D[233][j]] = 59648 + j;
        d[59648 + j] = D[233][j];
      }

      D[234] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰�".split("");

      for (j = 0; j != D[234].length; ++j) if (D[234][j].charCodeAt(0) !== 0xFFFD) {
        e[D[234][j]] = 59904 + j;
        d[59904 + j] = D[234][j];
      }

      D[235] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻�".split("");

      for (j = 0; j != D[235].length; ++j) if (D[235][j].charCodeAt(0) !== 0xFFFD) {
        e[D[235][j]] = 60160 + j;
        d[60160 + j] = D[235][j];
      }

      D[236] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐�".split("");

      for (j = 0; j != D[236].length; ++j) if (D[236][j].charCodeAt(0) !== 0xFFFD) {
        e[D[236][j]] = 60416 + j;
        d[60416 + j] = D[236][j];
      }

      D[237] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨�".split("");

      for (j = 0; j != D[237].length; ++j) if (D[237][j].charCodeAt(0) !== 0xFFFD) {
        e[D[237][j]] = 60672 + j;
        d[60672 + j] = D[237][j];
      }

      D[238] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶钷钸钹钺钼钽钿铄铈铉铊铋铌铍铎铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪�".split("");

      for (j = 0; j != D[238].length; ++j) if (D[238][j].charCodeAt(0) !== 0xFFFD) {
        e[D[238][j]] = 60928 + j;
        d[60928 + j] = D[238][j];
      }

      D[239] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒锓锔锕锖锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤镥镦镧镨镩镪镫镬镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔�".split("");

      for (j = 0; j != D[239].length; ++j) if (D[239][j].charCodeAt(0) !== 0xFFFD) {
        e[D[239][j]] = 61184 + j;
        d[61184 + j] = D[239][j];
      }

      D[240] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨鸩鸪鸫鸬鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦鹧鹨鹩鹪鹫鹬鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙�".split("");

      for (j = 0; j != D[240].length; ++j) if (D[240][j].charCodeAt(0) !== 0xFFFD) {
        e[D[240][j]] = 61440 + j;
        d[61440 + j] = D[240][j];
      }

      D[241] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃�".split("");

      for (j = 0; j != D[241].length; ++j) if (D[241][j].charCodeAt(0) !== 0xFFFD) {
        e[D[241][j]] = 61696 + j;
        d[61696 + j] = D[241][j];
      }

      D[242] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒�".split("");

      for (j = 0; j != D[242].length; ++j) if (D[242][j].charCodeAt(0) !== 0xFFFD) {
        e[D[242][j]] = 61952 + j;
        d[61952 + j] = D[242][j];
      }

      D[243] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋�".split("");

      for (j = 0; j != D[243].length; ++j) if (D[243][j].charCodeAt(0) !== 0xFFFD) {
        e[D[243][j]] = 62208 + j;
        d[62208 + j] = D[243][j];
      }

      D[244] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤�".split("");

      for (j = 0; j != D[244].length; ++j) if (D[244][j].charCodeAt(0) !== 0xFFFD) {
        e[D[244][j]] = 62464 + j;
        d[62464 + j] = D[244][j];
      }

      D[245] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜�".split("");

      for (j = 0; j != D[245].length; ++j) if (D[245][j].charCodeAt(0) !== 0xFFFD) {
        e[D[245][j]] = 62720 + j;
        d[62720 + j] = D[245][j];
      }

      D[246] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅龆龇龈龉龊龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞鲟鲠鲡鲢鲣鲥鲦鲧鲨鲩鲫鲭鲮鲰鲱鲲鲳鲴鲵鲶鲷鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋�".split("");

      for (j = 0; j != D[246].length; ++j) if (D[246][j].charCodeAt(0) !== 0xFFFD) {
        e[D[246][j]] = 62976 + j;
        d[62976 + j] = D[246][j];
      }

      D[247] = "�����������������������������������������������������������������������������������������������������������������������������������������������������������������鳌鳍鳎鳏鳐鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄�".split("");

      for (j = 0; j != D[247].length; ++j) if (D[247][j].charCodeAt(0) !== 0xFFFD) {
        e[D[247][j]] = 63232 + j;
        d[63232 + j] = D[247][j];
      }

      return {
        "enc": e,
        "dec": d
      };
    }();

    cptable[10029] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÄĀāÉĄÖÜáąČäčĆćéŹźĎíďĒēĖóėôöõúĚěü†°Ę£§•¶ß®©™ę¨≠ģĮįĪ≤≥īĶ∂∑łĻļĽľĹĺŅņŃ¬√ńŇ∆«»… ňŐÕőŌ–—“”‘’÷◊ōŔŕŘ‹›řŖŗŠ‚„šŚśÁŤťÍŽžŪÓÔūŮÚůŰűŲųÝýķŻŁżĢˇ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10079] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûüÝ°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸ⁄¤ÐðÞþý·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙıˆ˜¯˘˙˚¸˝˛ˇ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }();

    cptable[10081] = function () {
      var d = "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000b\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~ÄÅÇÉÑÖÜáàâäãåçéèêëíìîïñóòôöõúùûü†°¢£§•¶ß®©™´¨≠ÆØ∞±≤≥¥µ∂∑∏π∫ªºΩæø¿¡¬√ƒ≈∆«»… ÀÃÕŒœ–—“”‘’÷◊ÿŸĞğİıŞş‡·‚„‰ÂÊÁËÈÍÎÏÌÓÔ�ÒÚÛÙ�ˆ˜¯˘˙˚¸˝˛ˇ",
          D = [],
          e = {};

      for (var i = 0; i != d.length; ++i) {
        if (d.charCodeAt(i) !== 0xFFFD) e[d.charAt(i)] = i;
        D[i] = d.charAt(i);
      }

      return {
        "enc": e,
        "dec": D
      };
    }(); // eslint-disable-next-line no-undef


    if (module.exports && typeof DO_NOT_EXPORT_CODEPAGE === 'undefined') module.exports = cptable;
    /* cputils.js (C) 2013-present SheetJS -- http://sheetjs.com */

    /* vim: set ft=javascript: */

    /*jshint newcap: false */

    (function (root, factory) {

      if (typeof cptable === "undefined") {
        if (typeof commonjsRequire !== "undefined") {
          var cpt = cptable;
          if (module.exports && typeof DO_NOT_EXPORT_CODEPAGE === 'undefined') module.exports = factory(cpt);else root.cptable = factory(cpt);
        } else throw new Error("cptable not found");
      } else cptable = factory(cptable);
      /*eslint-enable */

      /*jshint ignore:end */

    })(commonjsGlobal, function (cpt) {
      /*global module, Buffer */

      var magic = {
        "1200": "utf16le",
        "1201": "utf16be",
        "12000": "utf32le",
        "12001": "utf32be",
        "16969": "utf64le",
        "20127": "ascii",
        "65000": "utf7",
        "65001": "utf8"
      };
      var sbcs_cache = [874, 1250, 1251, 1252, 1253, 1254, 1255, 1256, 10000];
      var dbcs_cache = [932, 936, 949, 950];
      var magic_cache = [65001];
      var magic_decode = {};
      var magic_encode = {};
      var cpdcache = {};
      var cpecache = {};

      var sfcc = function sfcc(x) {
        return String.fromCharCode(x);
      };

      var cca = function cca(x) {
        return x.charCodeAt(0);
      };

      var has_buf = typeof Buffer !== 'undefined';

      var Buffer_from = function () {};

      if (has_buf) {
        var nbfs = !Buffer.from;
        if (!nbfs) try {
          Buffer.from("foo", "utf8");
        } catch (e) {
          nbfs = true;
        }
        Buffer_from = nbfs ? function (buf, enc) {
          return enc ? new Buffer(buf, enc) : new Buffer(buf);
        } : Buffer.from.bind(Buffer); // $FlowIgnore

        if (!Buffer.allocUnsafe) Buffer.allocUnsafe = function (n) {
          return new Buffer(n);
        };
        var mdl = 1024,
            mdb = Buffer.allocUnsafe(mdl);

        var make_EE = function make_EE(E) {
          var EE = Buffer.allocUnsafe(65536);

          for (var i = 0; i < 65536; ++i) EE[i] = 0;

          var keys = Object.keys(E),
              len = keys.length;

          for (var ee = 0, e = keys[ee]; ee < len; ++ee) {
            if (!(e = keys[ee])) continue;
            EE[e.charCodeAt(0)] = E[e];
          }

          return EE;
        };

        var sbcs_encode = function make_sbcs_encode(cp) {
          var EE = make_EE(cpt[cp].enc);
          return function sbcs_e(data, ofmt) {
            var len = data.length;
            var out,
                i = 0,
                j = 0,
                D = 0,
                w = 0;

            if (typeof data === 'string') {
              out = Buffer.allocUnsafe(len);

              for (i = 0; i < len; ++i) out[i] = EE[data.charCodeAt(i)];
            } else if (Buffer.isBuffer(data)) {
              out = Buffer.allocUnsafe(2 * len);
              j = 0;

              for (i = 0; i < len; ++i) {
                D = data[i];
                if (D < 128) out[j++] = EE[D];else if (D < 224) {
                  out[j++] = EE[((D & 31) << 6) + (data[i + 1] & 63)];
                  ++i;
                } else if (D < 240) {
                  out[j++] = EE[((D & 15) << 12) + ((data[i + 1] & 63) << 6) + (data[i + 2] & 63)];
                  i += 2;
                } else {
                  w = ((D & 7) << 18) + ((data[i + 1] & 63) << 12) + ((data[i + 2] & 63) << 6) + (data[i + 3] & 63);
                  i += 3;
                  if (w < 65536) out[j++] = EE[w];else {
                    w -= 65536;
                    out[j++] = EE[0xD800 + (w >> 10 & 1023)];
                    out[j++] = EE[0xDC00 + (w & 1023)];
                  }
                }
              }

              out = out.slice(0, j);
            } else {
              out = Buffer.allocUnsafe(len);

              for (i = 0; i < len; ++i) out[i] = EE[data[i].charCodeAt(0)];
            }

            if (!ofmt || ofmt === 'buf') return out;
            if (ofmt !== 'arr') return out.toString('binary');
            return [].slice.call(out);
          };
        };

        var sbcs_decode = function make_sbcs_decode(cp) {
          var D = cpt[cp].dec;
          var DD = Buffer.allocUnsafe(131072),
              d = 0,
              c = "";

          for (d = 0; d < D.length; ++d) {
            if (!(c = D[d])) continue;
            var w = c.charCodeAt(0);
            DD[2 * d] = w & 255;
            DD[2 * d + 1] = w >> 8;
          }

          return function sbcs_d(data) {
            var len = data.length,
                i = 0,
                j = 0;

            if (2 * len > mdl) {
              mdl = 2 * len;
              mdb = Buffer.allocUnsafe(mdl);
            }

            if (Buffer.isBuffer(data)) {
              for (i = 0; i < len; i++) {
                j = 2 * data[i];
                mdb[2 * i] = DD[j];
                mdb[2 * i + 1] = DD[j + 1];
              }
            } else if (typeof data === "string") {
              for (i = 0; i < len; i++) {
                j = 2 * data.charCodeAt(i);
                mdb[2 * i] = DD[j];
                mdb[2 * i + 1] = DD[j + 1];
              }
            } else {
              for (i = 0; i < len; i++) {
                j = 2 * data[i];
                mdb[2 * i] = DD[j];
                mdb[2 * i + 1] = DD[j + 1];
              }
            }

            return mdb.slice(0, 2 * len).toString('ucs2');
          };
        };

        var dbcs_encode = function make_dbcs_encode(cp) {
          var E = cpt[cp].enc;
          var EE = Buffer.allocUnsafe(131072);

          for (var i = 0; i < 131072; ++i) EE[i] = 0;

          var keys = Object.keys(E);

          for (var ee = 0, e = keys[ee]; ee < keys.length; ++ee) {
            if (!(e = keys[ee])) continue;
            var f = e.charCodeAt(0);
            EE[2 * f] = E[e] & 255;
            EE[2 * f + 1] = E[e] >> 8;
          }

          return function dbcs_e(data, ofmt) {
            var len = data.length,
                out = Buffer.allocUnsafe(2 * len),
                i = 0,
                j = 0,
                jj = 0,
                k = 0,
                D = 0;

            if (typeof data === 'string') {
              for (i = k = 0; i < len; ++i) {
                j = data.charCodeAt(i) * 2;
                out[k++] = EE[j + 1] || EE[j];
                if (EE[j + 1] > 0) out[k++] = EE[j];
              }

              out = out.slice(0, k);
            } else if (Buffer.isBuffer(data)) {
              for (i = k = 0; i < len; ++i) {
                D = data[i];
                if (D < 128) j = D;else if (D < 224) {
                  j = ((D & 31) << 6) + (data[i + 1] & 63);
                  ++i;
                } else if (D < 240) {
                  j = ((D & 15) << 12) + ((data[i + 1] & 63) << 6) + (data[i + 2] & 63);
                  i += 2;
                } else {
                  j = ((D & 7) << 18) + ((data[i + 1] & 63) << 12) + ((data[i + 2] & 63) << 6) + (data[i + 3] & 63);
                  i += 3;
                }

                if (j < 65536) {
                  j *= 2;
                  out[k++] = EE[j + 1] || EE[j];
                  if (EE[j + 1] > 0) out[k++] = EE[j];
                } else {
                  jj = j - 65536;
                  j = 2 * (0xD800 + (jj >> 10 & 1023));
                  out[k++] = EE[j + 1] || EE[j];
                  if (EE[j + 1] > 0) out[k++] = EE[j];
                  j = 2 * (0xDC00 + (jj & 1023));
                  out[k++] = EE[j + 1] || EE[j];
                  if (EE[j + 1] > 0) out[k++] = EE[j];
                }
              }

              out = out.slice(0, k);
            } else {
              for (i = k = 0; i < len; i++) {
                j = data[i].charCodeAt(0) * 2;
                out[k++] = EE[j + 1] || EE[j];
                if (EE[j + 1] > 0) out[k++] = EE[j];
              }
            }

            if (!ofmt || ofmt === 'buf') return out;
            if (ofmt !== 'arr') return out.toString('binary');
            return [].slice.call(out);
          };
        };

        var dbcs_decode = function make_dbcs_decode(cp) {
          var D = cpt[cp].dec;
          var DD = Buffer.allocUnsafe(131072),
              d = 0,
              c,
              w = 0,
              j = 0,
              i = 0;

          for (i = 0; i < 65536; ++i) {
            DD[2 * i] = 0xFF;
            DD[2 * i + 1] = 0xFD;
          }

          for (d = 0; d < D.length; ++d) {
            if (!(c = D[d])) continue;
            w = c.charCodeAt(0);
            j = 2 * d;
            DD[j] = w & 255;
            DD[j + 1] = w >> 8;
          }

          return function dbcs_d(data) {
            var len = data.length,
                out = Buffer.allocUnsafe(2 * len),
                i = 0,
                j = 0,
                k = 0;

            if (Buffer.isBuffer(data)) {
              for (i = 0; i < len; i++) {
                j = 2 * data[i];

                if (DD[j] === 0xFF && DD[j + 1] === 0xFD) {
                  j = 2 * ((data[i] << 8) + data[i + 1]);
                  ++i;
                }

                out[k++] = DD[j];
                out[k++] = DD[j + 1];
              }
            } else if (typeof data === "string") {
              for (i = 0; i < len; i++) {
                j = 2 * data.charCodeAt(i);

                if (DD[j] === 0xFF && DD[j + 1] === 0xFD) {
                  j = 2 * ((data.charCodeAt(i) << 8) + data.charCodeAt(i + 1));
                  ++i;
                }

                out[k++] = DD[j];
                out[k++] = DD[j + 1];
              }
            } else {
              for (i = 0; i < len; i++) {
                j = 2 * data[i];

                if (DD[j] === 0xFF && DD[j + 1] === 0xFD) {
                  j = 2 * ((data[i] << 8) + data[i + 1]);
                  ++i;
                }

                out[k++] = DD[j];
                out[k++] = DD[j + 1];
              }
            }

            return out.slice(0, k).toString('ucs2');
          };
        };

        magic_decode[65001] = function utf8_d(data) {
          if (typeof data === "string") return utf8_d(data.split("").map(cca));
          var len = data.length,
              w = 0,
              ww = 0;

          if (4 * len > mdl) {
            mdl = 4 * len;
            mdb = Buffer.allocUnsafe(mdl);
          }

          var i = 0;
          if (len >= 3 && data[0] == 0xEF) if (data[1] == 0xBB && data[2] == 0xBF) i = 3;

          for (var j = 1, k = 0, D = 0; i < len; i += j) {
            j = 1;
            D = data[i];
            if (D < 128) w = D;else if (D < 224) {
              w = (D & 31) * 64 + (data[i + 1] & 63);
              j = 2;
            } else if (D < 240) {
              w = ((D & 15) << 12) + (data[i + 1] & 63) * 64 + (data[i + 2] & 63);
              j = 3;
            } else {
              w = (D & 7) * 262144 + ((data[i + 1] & 63) << 12) + (data[i + 2] & 63) * 64 + (data[i + 3] & 63);
              j = 4;
            }

            if (w < 65536) {
              mdb[k++] = w & 255;
              mdb[k++] = w >> 8;
            } else {
              w -= 65536;
              ww = 0xD800 + (w >> 10 & 1023);
              w = 0xDC00 + (w & 1023);
              mdb[k++] = ww & 255;
              mdb[k++] = ww >>> 8;
              mdb[k++] = w & 255;
              mdb[k++] = w >>> 8 & 255;
            }
          }

          return mdb.slice(0, k).toString('ucs2');
        };

        magic_encode[65001] = function utf8_e(data, ofmt) {
          if (has_buf && Buffer.isBuffer(data)) {
            if (!ofmt || ofmt === 'buf') return data;
            if (ofmt !== 'arr') return data.toString('binary');
            return [].slice.call(data);
          }

          var len = data.length,
              w = 0,
              ww = 0,
              j = 0;
          var direct = typeof data === "string";

          if (4 * len > mdl) {
            mdl = 4 * len;
            mdb = Buffer.allocUnsafe(mdl);
          }

          for (var i = 0; i < len; ++i) {
            w = direct ? data.charCodeAt(i) : data[i].charCodeAt(0);
            if (w <= 0x007F) mdb[j++] = w;else if (w <= 0x07FF) {
              mdb[j++] = 192 + (w >> 6);
              mdb[j++] = 128 + (w & 63);
            } else if (w >= 0xD800 && w <= 0xDFFF) {
              w -= 0xD800;
              ++i;
              ww = (direct ? data.charCodeAt(i) : data[i].charCodeAt(0)) - 0xDC00 + (w << 10);
              mdb[j++] = 240 + (ww >>> 18 & 0x07);
              mdb[j++] = 144 + (ww >>> 12 & 0x3F);
              mdb[j++] = 128 + (ww >>> 6 & 0x3F);
              mdb[j++] = 128 + (ww & 0x3F);
            } else {
              mdb[j++] = 224 + (w >> 12);
              mdb[j++] = 128 + (w >> 6 & 63);
              mdb[j++] = 128 + (w & 63);
            }
          }

          if (!ofmt || ofmt === 'buf') return mdb.slice(0, j);
          if (ofmt !== 'arr') return mdb.slice(0, j).toString('binary');
          return [].slice.call(mdb, 0, j);
        };
      }

      var encache = function encache() {
        if (has_buf) {
          if (cpdcache[sbcs_cache[0]]) return;
          var i = 0,
              s = 0;

          for (i = 0; i < sbcs_cache.length; ++i) {
            s = sbcs_cache[i];

            if (cpt[s]) {
              cpdcache[s] = sbcs_decode(s);
              cpecache[s] = sbcs_encode(s);
            }
          }

          for (i = 0; i < dbcs_cache.length; ++i) {
            s = dbcs_cache[i];

            if (cpt[s]) {
              cpdcache[s] = dbcs_decode(s);
              cpecache[s] = dbcs_encode(s);
            }
          }

          for (i = 0; i < magic_cache.length; ++i) {
            s = magic_cache[i];
            if (magic_decode[s]) cpdcache[s] = magic_decode[s];
            if (magic_encode[s]) cpecache[s] = magic_encode[s];
          }
        }
      };

      var null_enc = function (data, ofmt) {
        return "";
      };

      var cp_decache = function cp_decache(cp) {
        delete cpdcache[cp];
        delete cpecache[cp];
      };

      var decache = function decache() {
        if (has_buf) {
          if (!cpdcache[sbcs_cache[0]]) return;
          sbcs_cache.forEach(cp_decache);
          dbcs_cache.forEach(cp_decache);
          magic_cache.forEach(cp_decache);
        }

        last_enc = null_enc;
        last_cp = 0;
      };

      var cache = {
        encache: encache,
        decache: decache,
        sbcs: sbcs_cache,
        dbcs: dbcs_cache
      };
      encache();
      var BM = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
      var SetD = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789'(),-./:?";
      var last_enc = null_enc,
          last_cp = 0;

      var encode = function encode(cp, data, ofmt) {
        if (cp === last_cp && last_enc) {
          return last_enc(data, ofmt);
        }

        if (cpecache[cp]) {
          last_enc = cpecache[last_cp = cp];
          return last_enc(data, ofmt);
        }

        if (has_buf && Buffer.isBuffer(data)) data = data.toString('utf8');
        var len = data.length;
        var out = has_buf ? Buffer.allocUnsafe(4 * len) : [],
            w = 0,
            i = 0,
            j = 0,
            ww = 0;
        var C = cpt[cp],
            E,
            M = "";
        var isstr = typeof data === 'string';
        if (C && (E = C.enc)) for (i = 0; i < len; ++i, ++j) {
          w = E[isstr ? data.charAt(i) : data[i]];

          if (w > 255) {
            out[j] = w >> 8;
            out[++j] = w & 255;
          } else out[j] = w & 255;
        } else if (M = magic[cp]) switch (M) {
          case "utf8":
            if (has_buf && isstr) {
              out = Buffer_from(data, M);
              j = out.length;
              break;
            }

            for (i = 0; i < len; ++i, ++j) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              if (w <= 0x007F) out[j] = w;else if (w <= 0x07FF) {
                out[j] = 192 + (w >> 6);
                out[++j] = 128 + (w & 63);
              } else if (w >= 0xD800 && w <= 0xDFFF) {
                w -= 0xD800;
                ww = (isstr ? data.charCodeAt(++i) : data[++i].charCodeAt(0)) - 0xDC00 + (w << 10);
                out[j] = 240 + (ww >>> 18 & 0x07);
                out[++j] = 144 + (ww >>> 12 & 0x3F);
                out[++j] = 128 + (ww >>> 6 & 0x3F);
                out[++j] = 128 + (ww & 0x3F);
              } else {
                out[j] = 224 + (w >> 12);
                out[++j] = 128 + (w >> 6 & 63);
                out[++j] = 128 + (w & 63);
              }
            }

            break;

          case "ascii":
            if (has_buf && typeof data === "string") {
              out = Buffer_from(data, M);
              j = out.length;
              break;
            }

            for (i = 0; i < len; ++i, ++j) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              if (w <= 0x007F) out[j] = w;else throw new Error("bad ascii " + w);
            }

            break;

          case "utf16le":
            if (has_buf && typeof data === "string") {
              out = Buffer_from(data, M);
              j = out.length;
              break;
            }

            for (i = 0; i < len; ++i) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              out[j++] = w & 255;
              out[j++] = w >> 8;
            }

            break;

          case "utf16be":
            for (i = 0; i < len; ++i) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              out[j++] = w >> 8;
              out[j++] = w & 255;
            }

            break;

          case "utf32le":
            for (i = 0; i < len; ++i) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              if (w >= 0xD800 && w <= 0xDFFF) w = 0x10000 + (w - 0xD800 << 10) + (data[++i].charCodeAt(0) - 0xDC00);
              out[j++] = w & 255;
              w >>= 8;
              out[j++] = w & 255;
              w >>= 8;
              out[j++] = w & 255;
              w >>= 8;
              out[j++] = w & 255;
            }

            break;

          case "utf32be":
            for (i = 0; i < len; ++i) {
              w = isstr ? data.charCodeAt(i) : data[i].charCodeAt(0);
              if (w >= 0xD800 && w <= 0xDFFF) w = 0x10000 + (w - 0xD800 << 10) + (data[++i].charCodeAt(0) - 0xDC00);
              out[j + 3] = w & 255;
              w >>= 8;
              out[j + 2] = w & 255;
              w >>= 8;
              out[j + 1] = w & 255;
              w >>= 8;
              out[j] = w & 255;
              j += 4;
            }

            break;

          case "utf7":
            for (i = 0; i < len; i++) {
              var c = isstr ? data.charAt(i) : data[i].charAt(0);

              if (c === "+") {
                out[j++] = 0x2b;
                out[j++] = 0x2d;
                continue;
              }

              if (SetD.indexOf(c) > -1) {
                out[j++] = c.charCodeAt(0);
                continue;
              }

              var tt = encode(1201, c);
              out[j++] = 0x2b;
              out[j++] = BM.charCodeAt(tt[0] >> 2);
              out[j++] = BM.charCodeAt(((tt[0] & 0x03) << 4) + ((tt[1] || 0) >> 4));
              out[j++] = BM.charCodeAt(((tt[1] & 0x0F) << 2) + ((tt[2] || 0) >> 6));
              out[j++] = 0x2d;
            }

            break;

          default:
            throw new Error("Unsupported magic: " + cp + " " + magic[cp]);
        } else throw new Error("Unrecognized CP: " + cp);
        out = out.slice(0, j);
        if (!has_buf) return ofmt == 'str' ? out.map(sfcc).join("") : out;
        if (!ofmt || ofmt === 'buf') return out;
        if (ofmt !== 'arr') return out.toString('binary');
        return [].slice.call(out);
      };

      var decode = function decode(cp, data) {
        var F;
        if (F = cpdcache[cp]) return F(data);
        if (typeof data === "string") return decode(cp, data.split("").map(cca));
        var len = data.length,
            out = new Array(len),
            s = "",
            w = 0,
            i = 0,
            j = 1,
            k = 0,
            ww = 0;
        var C = cpt[cp],
            D,
            M = "";

        if (C && (D = C.dec)) {
          for (i = 0; i < len; i += j) {
            j = 2;
            s = D[(data[i] << 8) + data[i + 1]];

            if (!s) {
              j = 1;
              s = D[data[i]];
            }

            if (!s) throw new Error('Unrecognized code: ' + data[i] + ' ' + data[i + j - 1] + ' ' + i + ' ' + j + ' ' + D[data[i]]);
            out[k++] = s;
          }
        } else if (M = magic[cp]) switch (M) {
          case "utf8":
            if (len >= 3 && data[0] == 0xEF) if (data[1] == 0xBB && data[2] == 0xBF) i = 3;

            for (; i < len; i += j) {
              j = 1;
              if (data[i] < 128) w = data[i];else if (data[i] < 224) {
                w = (data[i] & 31) * 64 + (data[i + 1] & 63);
                j = 2;
              } else if (data[i] < 240) {
                w = ((data[i] & 15) << 12) + (data[i + 1] & 63) * 64 + (data[i + 2] & 63);
                j = 3;
              } else {
                w = (data[i] & 7) * 262144 + ((data[i + 1] & 63) << 12) + (data[i + 2] & 63) * 64 + (data[i + 3] & 63);
                j = 4;
              }

              if (w < 65536) {
                out[k++] = String.fromCharCode(w);
              } else {
                w -= 65536;
                ww = 0xD800 + (w >> 10 & 1023);
                w = 0xDC00 + (w & 1023);
                out[k++] = String.fromCharCode(ww);
                out[k++] = String.fromCharCode(w);
              }
            }

            break;

          case "ascii":
            if (has_buf && Buffer.isBuffer(data)) return data.toString(M);

            for (i = 0; i < len; i++) out[i] = String.fromCharCode(data[i]);

            k = len;
            break;

          case "utf16le":
            if (len >= 2 && data[0] == 0xFF) if (data[1] == 0xFE) i = 2;
            if (has_buf && Buffer.isBuffer(data)) return data.toString(M);
            j = 2;

            for (; i + 1 < len; i += j) {
              out[k++] = String.fromCharCode((data[i + 1] << 8) + data[i]);
            }

            break;

          case "utf16be":
            if (len >= 2 && data[0] == 0xFE) if (data[1] == 0xFF) i = 2;
            j = 2;

            for (; i + 1 < len; i += j) {
              out[k++] = String.fromCharCode((data[i] << 8) + data[i + 1]);
            }

            break;

          case "utf32le":
            if (len >= 4 && data[0] == 0xFF) if (data[1] == 0xFE && data[2] === 0 && data[3] === 0) i = 4;
            j = 4;

            for (; i < len; i += j) {
              w = (data[i + 3] << 24) + (data[i + 2] << 16) + (data[i + 1] << 8) + data[i];

              if (w > 0xFFFF) {
                w -= 0x10000;
                out[k++] = String.fromCharCode(0xD800 + (w >> 10 & 0x3FF));
                out[k++] = String.fromCharCode(0xDC00 + (w & 0x3FF));
              } else out[k++] = String.fromCharCode(w);
            }

            break;

          case "utf32be":
            if (len >= 4 && data[3] == 0xFF) if (data[2] == 0xFE && data[1] === 0 && data[0] === 0) i = 4;
            j = 4;

            for (; i < len; i += j) {
              w = (data[i] << 24) + (data[i + 1] << 16) + (data[i + 2] << 8) + data[i + 3];

              if (w > 0xFFFF) {
                w -= 0x10000;
                out[k++] = String.fromCharCode(0xD800 + (w >> 10 & 0x3FF));
                out[k++] = String.fromCharCode(0xDC00 + (w & 0x3FF));
              } else out[k++] = String.fromCharCode(w);
            }

            break;

          case "utf7":
            if (len >= 4 && data[0] == 0x2B && data[1] == 0x2F && data[2] == 0x76) {
              if (len >= 5 && data[3] == 0x38 && data[4] == 0x2D) i = 5;else if (data[3] == 0x38 || data[3] == 0x39 || data[3] == 0x2B || data[3] == 0x2F) i = 4;
            }

            for (; i < len; i += j) {
              if (data[i] !== 0x2b) {
                j = 1;
                out[k++] = String.fromCharCode(data[i]);
                continue;
              }

              j = 1;

              if (data[i + 1] === 0x2d) {
                j = 2;
                out[k++] = "+";
                continue;
              } // eslint-disable-next-line no-useless-escape


              while (String.fromCharCode(data[i + j]).match(/[A-Za-z0-9+\/]/)) j++;

              var dash = 0;

              if (data[i + j] === 0x2d) {
                ++j;
                dash = 1;
              }

              var tt = [];
              var o64 = "";
              var c1 = 0,
                  c2 = 0,
                  c3 = 0;
              var e1 = 0,
                  e2 = 0,
                  e3 = 0,
                  e4 = 0;

              for (var l = 1; l < j - dash;) {
                e1 = BM.indexOf(String.fromCharCode(data[i + l++]));
                e2 = BM.indexOf(String.fromCharCode(data[i + l++]));
                c1 = e1 << 2 | e2 >> 4;
                tt.push(c1);
                e3 = BM.indexOf(String.fromCharCode(data[i + l++]));
                if (e3 === -1) break;
                c2 = (e2 & 15) << 4 | e3 >> 2;
                tt.push(c2);
                e4 = BM.indexOf(String.fromCharCode(data[i + l++]));
                if (e4 === -1) break;
                c3 = (e3 & 3) << 6 | e4;
                if (e4 < 64) tt.push(c3);
              }

              o64 = decode(1201, tt);

              for (l = 0; l < o64.length; ++l) out[k++] = o64.charAt(l);
            }

            break;

          default:
            throw new Error("Unsupported magic: " + cp + " " + magic[cp]);
        } else throw new Error("Unrecognized CP: " + cp);

        return out.slice(0, k).join("");
      };

      var hascp = function hascp(cp) {
        return !!(cpt[cp] || magic[cp]);
      };

      cpt.utils = {
        decode: decode,
        encode: encode,
        hascp: hascp,
        magic: magic,
        cache: cache
      };
      return cpt;
    });
  });

  var _nodeResolve_empty = {};

  var _nodeResolve_empty$1 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': _nodeResolve_empty
  });

  /*

  JSZip - A Javascript class for generating and reading zip files
  <http://stuartk.com/jszip>

  (c) 2009-2014 Stuart Knightley <stuart [at] stuartk.com>
  Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

  JSZip uses the library pako released under the MIT license :
  https://github.com/nodeca/pako/blob/master/LICENSE

  Note: since JSZip 3 removed critical functionality, this version assigns to the
  `JSZipSync` variable.  Another JSZip version can be loaded in parallel.
  */
  var jszip = createCommonjsModule(function (module, exports) {
    (function (e) {
      if ("undefined" == typeof DO_NOT_EXPORT_JSZIP) module.exports = e();else {
        var f;
        "undefined" != typeof globalThis ? f = globalThis : "undefined" != typeof window ? f = window : "undefined" != typeof commonjsGlobal ? f = commonjsGlobal : "undefined" != typeof $ && $.global ? f = $.global : "undefined" != typeof self && (f = self), f.JSZipSync = e();
      }
    })(function () {
      return function e(t, n, r) {
        function s(o, u) {
          if (!n[o]) {
            if (!t[o]) {
              var a = typeof commonjsRequire == "function" && commonjsRequire;
              if (!u && a) return a(o, !0);
              if (i) return i(o, !0);
              throw new Error("Cannot find module '" + o + "'");
            }

            var f = n[o] = {
              exports: {}
            };
            t[o][0].call(f.exports, function (e) {
              var n = t[o][1][e];
              return s(n ? n : e);
            }, f, f.exports, e, t, n, r);
          }

          return n[o].exports;
        }

        var i = typeof commonjsRequire == "function" && commonjsRequire;

        for (var o = 0; o < r.length; o++) s(r[o]);

        return s;
      }({
        1: [function (_dereq_, module, exports) {

          var _keyStr = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="; // public method for encoding

          exports.encode = function (input, utf8) {
            var output = "";
            var chr1, chr2, chr3, enc1, enc2, enc3, enc4;
            var i = 0;

            while (i < input.length) {
              chr1 = input.charCodeAt(i++);
              chr2 = input.charCodeAt(i++);
              chr3 = input.charCodeAt(i++);
              enc1 = chr1 >> 2;
              enc2 = (chr1 & 3) << 4 | chr2 >> 4;
              enc3 = (chr2 & 15) << 2 | chr3 >> 6;
              enc4 = chr3 & 63;

              if (isNaN(chr2)) {
                enc3 = enc4 = 64;
              } else if (isNaN(chr3)) {
                enc4 = 64;
              }

              output = output + _keyStr.charAt(enc1) + _keyStr.charAt(enc2) + _keyStr.charAt(enc3) + _keyStr.charAt(enc4);
            }

            return output;
          }; // public method for decoding


          exports.decode = function (input, utf8) {
            var output = "";
            var chr1, chr2, chr3;
            var enc1, enc2, enc3, enc4;
            var i = 0;
            input = input.replace(/[^A-Za-z0-9\+\/\=]/g, "");

            while (i < input.length) {
              enc1 = _keyStr.indexOf(input.charAt(i++));
              enc2 = _keyStr.indexOf(input.charAt(i++));
              enc3 = _keyStr.indexOf(input.charAt(i++));
              enc4 = _keyStr.indexOf(input.charAt(i++));
              chr1 = enc1 << 2 | enc2 >> 4;
              chr2 = (enc2 & 15) << 4 | enc3 >> 2;
              chr3 = (enc3 & 3) << 6 | enc4;
              output = output + String.fromCharCode(chr1);

              if (enc3 != 64) {
                output = output + String.fromCharCode(chr2);
              }

              if (enc4 != 64) {
                output = output + String.fromCharCode(chr3);
              }
            }

            return output;
          };
        }, {}],
        2: [function (_dereq_, module, exports) {

          function CompressedObject() {
            this.compressedSize = 0;
            this.uncompressedSize = 0;
            this.crc32 = 0;
            this.compressionMethod = null;
            this.compressedContent = null;
          }

          CompressedObject.prototype = {
            /**
             * Return the decompressed content in an unspecified format.
             * The format will depend on the decompressor.
             * @return {Object} the decompressed content.
             */
            getContent: function () {
              return null; // see implementation
            },

            /**
             * Return the compressed content in an unspecified format.
             * The format will depend on the compressed conten source.
             * @return {Object} the compressed content.
             */
            getCompressedContent: function () {
              return null; // see implementation
            }
          };
          module.exports = CompressedObject;
        }, {}],
        3: [function (_dereq_, module, exports) {

          exports.STORE = {
            magic: "\x00\x00",
            compress: function (content) {
              return content; // no compression
            },
            uncompress: function (content) {
              return content; // no compression
            },
            compressInputType: null,
            uncompressInputType: null
          };
          exports.DEFLATE = _dereq_('./flate');
        }, {
          "./flate": 8
        }],
        4: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');

          var table = [0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3, 0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91, 0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7, 0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5, 0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B, 0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59, 0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F, 0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D, 0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433, 0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01, 0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457, 0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65, 0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB, 0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9, 0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F, 0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD, 0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683, 0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1, 0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7, 0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5, 0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B, 0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79, 0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F, 0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D, 0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713, 0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21, 0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777, 0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45, 0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB, 0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9, 0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF, 0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D];
          /**
           *
           *  Javascript crc32
           *  http://www.webtoolkit.info/
           *
           */

          module.exports = function crc32(input, crc) {
            if (typeof input === "undefined" || !input.length) {
              return 0;
            }

            var isArray = utils.getTypeOf(input) !== "string";

            if (typeof crc == "undefined") {
              crc = 0;
            }

            var x = 0;
            var y = 0;
            var b = 0;
            crc = crc ^ -1;

            for (var i = 0, iTop = input.length; i < iTop; i++) {
              b = isArray ? input[i] : input.charCodeAt(i);
              y = (crc ^ b) & 0xFF;
              x = table[y];
              crc = crc >>> 8 ^ x;
            }

            return crc ^ -1;
          }; // vim: set shiftwidth=4 softtabstop=4:

        }, {
          "./utils": 21
        }],
        5: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');

          function DataReader(data) {
            this.data = null; // type : see implementation

            this.length = 0;
            this.index = 0;
          }

          DataReader.prototype = {
            /**
             * Check that the offset will not go too far.
             * @param {string} offset the additional offset to check.
             * @throws {Error} an Error if the offset is out of bounds.
             */
            checkOffset: function (offset) {
              this.checkIndex(this.index + offset);
            },

            /**
             * Check that the specifed index will not be too far.
             * @param {string} newIndex the index to check.
             * @throws {Error} an Error if the index is out of bounds.
             */
            checkIndex: function (newIndex) {
              if (this.length < newIndex || newIndex < 0) {
                throw new Error("End of data reached (data length = " + this.length + ", asked index = " + newIndex + "). Corrupted zip ?");
              }
            },

            /**
             * Change the index.
             * @param {number} newIndex The new index.
             * @throws {Error} if the new index is out of the data.
             */
            setIndex: function (newIndex) {
              this.checkIndex(newIndex);
              this.index = newIndex;
            },

            /**
             * Skip the next n bytes.
             * @param {number} n the number of bytes to skip.
             * @throws {Error} if the new index is out of the data.
             */
            skip: function (n) {
              this.setIndex(this.index + n);
            },

            /**
             * Get the byte at the specified index.
             * @param {number} i the index to use.
             * @return {number} a byte.
             */
            byteAt: function (i) {// see implementations
            },

            /**
             * Get the next number with a given byte size.
             * @param {number} size the number of bytes to read.
             * @return {number} the corresponding number.
             */
            readInt: function (size) {
              var result = 0,
                  i;
              this.checkOffset(size);

              for (i = this.index + size - 1; i >= this.index; i--) {
                result = (result << 8) + this.byteAt(i);
              }

              this.index += size;
              return result;
            },

            /**
             * Get the next string with a given byte size.
             * @param {number} size the number of bytes to read.
             * @return {string} the corresponding string.
             */
            readString: function (size) {
              return utils.transformTo("string", this.readData(size));
            },

            /**
             * Get raw data without conversion, <size> bytes.
             * @param {number} size the number of bytes to read.
             * @return {Object} the raw data, implementation specific.
             */
            readData: function (size) {// see implementations
            },

            /**
             * Find the last occurence of a zip signature (4 bytes).
             * @param {string} sig the signature to find.
             * @return {number} the index of the last occurence, -1 if not found.
             */
            lastIndexOfSignature: function (sig) {// see implementations
            },

            /**
             * Get the next date.
             * @return {Date} the date.
             */
            readDate: function () {
              var dostime = this.readInt(4);
              return new Date((dostime >> 25 & 0x7f) + 1980, // year
              (dostime >> 21 & 0x0f) - 1, // month
              dostime >> 16 & 0x1f, // day
              dostime >> 11 & 0x1f, // hour
              dostime >> 5 & 0x3f, // minute
              (dostime & 0x1f) << 1); // second
            }
          };
          module.exports = DataReader;
        }, {
          "./utils": 21
        }],
        6: [function (_dereq_, module, exports) {

          exports.base64 = false;
          exports.binary = false;
          exports.dir = false;
          exports.createFolders = false;
          exports.date = null;
          exports.compression = null;
          exports.comment = null;
        }, {}],
        7: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.string2binary = function (str) {
            return utils.string2binary(str);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.string2Uint8Array = function (str) {
            return utils.transformTo("uint8array", str);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.uint8Array2String = function (array) {
            return utils.transformTo("string", array);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.string2Blob = function (str) {
            var buffer = utils.transformTo("arraybuffer", str);
            return utils.arrayBuffer2Blob(buffer);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.arrayBuffer2Blob = function (buffer) {
            return utils.arrayBuffer2Blob(buffer);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.transformTo = function (outputType, input) {
            return utils.transformTo(outputType, input);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.getTypeOf = function (input) {
            return utils.getTypeOf(input);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.checkSupport = function (type) {
            return utils.checkSupport(type);
          };
          /**
           * @deprecated
           * This value will be removed in a future version without replacement.
           */


          exports.MAX_VALUE_16BITS = utils.MAX_VALUE_16BITS;
          /**
           * @deprecated
           * This value will be removed in a future version without replacement.
           */

          exports.MAX_VALUE_32BITS = utils.MAX_VALUE_32BITS;
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */

          exports.pretty = function (str) {
            return utils.pretty(str);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.findCompression = function (compressionMethod) {
            return utils.findCompression(compressionMethod);
          };
          /**
           * @deprecated
           * This function will be removed in a future version without replacement.
           */


          exports.isRegExp = function (object) {
            return utils.isRegExp(object);
          };
        }, {
          "./utils": 21
        }],
        8: [function (_dereq_, module, exports) {

          var USE_TYPEDARRAY = typeof Uint8Array !== 'undefined' && typeof Uint16Array !== 'undefined' && typeof Uint32Array !== 'undefined';

          var pako = _dereq_("pako");

          exports.uncompressInputType = USE_TYPEDARRAY ? "uint8array" : "array";
          exports.compressInputType = USE_TYPEDARRAY ? "uint8array" : "array";
          exports.magic = "\x08\x00";

          exports.compress = function (input) {
            return pako.deflateRaw(input);
          };

          exports.uncompress = function (input) {
            return pako.inflateRaw(input);
          };
        }, {
          "pako": 24
        }],
        9: [function (_dereq_, module, exports) {

          var base64 = _dereq_('./base64');
          /**
          Usage:
             zip = new JSZip();
             zip.file("hello.txt", "Hello, World!").file("tempfile", "nothing");
             zip.folder("images").file("smile.gif", base64Data, {base64: true});
             zip.file("Xmas.txt", "Ho ho ho !", {date : new Date("December 25, 2007 00:00:01")});
             zip.remove("tempfile");
          
             base64zip = zip.generate();
          
          **/

          /**
           * Representation a of zip file in js
           * @constructor
           * @param {String=|ArrayBuffer=|Uint8Array=} data the data to load, if any (optional).
           * @param {Object=} options the options for creating this objects (optional).
           */


          function JSZipSync(data, options) {
            // if this constructor is used without `new`, it adds `new` before itself:
            if (!(this instanceof JSZipSync)) return new JSZipSync(data, options); // object containing the files :
            // {
            //   "folder/" : {...},
            //   "folder/data.txt" : {...}
            // }

            this.files = {};
            this.comment = null; // Where we are in the hierarchy

            this.root = "";

            if (data) {
              this.load(data, options);
            }

            this.clone = function () {
              var newObj = new JSZipSync();

              for (var i in this) {
                if (typeof this[i] !== "function") {
                  newObj[i] = this[i];
                }
              }

              return newObj;
            };
          }

          JSZipSync.prototype = _dereq_('./object');
          JSZipSync.prototype.load = _dereq_('./load');
          JSZipSync.support = _dereq_('./support');
          JSZipSync.defaults = _dereq_('./defaults');
          /**
           * @deprecated
           * This namespace will be removed in a future version without replacement.
           */

          JSZipSync.utils = _dereq_('./deprecatedPublicUtils');
          JSZipSync.base64 = {
            /**
             * @deprecated
             * This method will be removed in a future version without replacement.
             */
            encode: function (input) {
              return base64.encode(input);
            },

            /**
             * @deprecated
             * This method will be removed in a future version without replacement.
             */
            decode: function (input) {
              return base64.decode(input);
            }
          };
          JSZipSync.compressions = _dereq_('./compressions');
          module.exports = JSZipSync;
        }, {
          "./base64": 1,
          "./compressions": 3,
          "./defaults": 6,
          "./deprecatedPublicUtils": 7,
          "./load": 10,
          "./object": 13,
          "./support": 17
        }],
        10: [function (_dereq_, module, exports) {

          var base64 = _dereq_('./base64');

          var ZipEntries = _dereq_('./zipEntries');

          module.exports = function (data, options) {
            var files, zipEntries, i, input;
            options = options || {};

            if (options.base64) {
              data = base64.decode(data);
            }

            zipEntries = new ZipEntries(data, options);
            files = zipEntries.files;

            for (i = 0; i < files.length; i++) {
              input = files[i];
              this.file(input.fileName, input.decompressed, {
                binary: true,
                optimizedBinaryString: true,
                date: input.date,
                dir: input.dir,
                comment: input.fileComment.length ? input.fileComment : null,
                createFolders: options.createFolders
              });
            }

            if (zipEntries.zipComment.length) {
              this.comment = zipEntries.zipComment;
            }

            return this;
          };
        }, {
          "./base64": 1,
          "./zipEntries": 22
        }],
        11: [function (_dereq_, module, exports) {
          (function (Buffer) {

            var Buffer_from =
            /*::(*/
            function () {}
            /*:: :any)*/
            ;

            if (typeof Buffer !== 'undefined') {
              var nbfs = !Buffer.from;
              if (!nbfs) try {
                Buffer.from("foo", "utf8");
              } catch (e) {
                nbfs = true;
              }
              Buffer_from = nbfs ? function (buf, enc) {
                return enc ? new Buffer(buf, enc) : new Buffer(buf);
              } : Buffer.from.bind(Buffer); // $FlowIgnore

              if (!Buffer.alloc) Buffer.alloc = function (n) {
                return new Buffer(n);
              };
            }

            module.exports = function (data, encoding) {
              return typeof data == 'number' ? Buffer.alloc(data) : Buffer_from(data, encoding);
            };

            module.exports.test = function (b) {
              return Buffer.isBuffer(b);
            };
          }).call(this, typeof Buffer !== "undefined" ? Buffer : undefined);
        }, {}],
        12: [function (_dereq_, module, exports) {

          var Uint8ArrayReader = _dereq_('./uint8ArrayReader');

          function NodeBufferReader(data) {
            this.data = data;
            this.length = this.data.length;
            this.index = 0;
          }

          NodeBufferReader.prototype = new Uint8ArrayReader();
          /**
           * @see DataReader.readData
           */

          NodeBufferReader.prototype.readData = function (size) {
            this.checkOffset(size);
            var result = this.data.slice(this.index, this.index + size);
            this.index += size;
            return result;
          };

          module.exports = NodeBufferReader;
        }, {
          "./uint8ArrayReader": 18
        }],
        13: [function (_dereq_, module, exports) {

          var support = _dereq_('./support');

          var utils = _dereq_('./utils');

          var crc32 = _dereq_('./crc32');

          var signature = _dereq_('./signature');

          var defaults = _dereq_('./defaults');

          var base64 = _dereq_('./base64');

          var compressions = _dereq_('./compressions');

          var CompressedObject = _dereq_('./compressedObject');

          var nodeBuffer = _dereq_('./nodeBuffer');

          var utf8 = _dereq_('./utf8');

          var StringWriter = _dereq_('./stringWriter');

          var Uint8ArrayWriter = _dereq_('./uint8ArrayWriter');
          /**
           * Returns the raw data of a ZipObject, decompress the content if necessary.
           * @param {ZipObject} file the file to use.
           * @return {String|ArrayBuffer|Uint8Array|Buffer} the data.
           */


          var getRawData = function (file) {
            if (file._data instanceof CompressedObject) {
              file._data = file._data.getContent();
              file.options.binary = true;
              file.options.base64 = false;

              if (utils.getTypeOf(file._data) === "uint8array") {
                var copy = file._data; // when reading an arraybuffer, the CompressedObject mechanism will keep it and subarray() a Uint8Array.
                // if we request a file in the same format, we might get the same Uint8Array or its ArrayBuffer (the original zip file).

                file._data = new Uint8Array(copy.length); // with an empty Uint8Array, Opera fails with a "Offset larger than array size"

                if (copy.length !== 0) {
                  file._data.set(copy, 0);
                }
              }
            }

            return file._data;
          };
          /**
           * Returns the data of a ZipObject in a binary form. If the content is an unicode string, encode it.
           * @param {ZipObject} file the file to use.
           * @return {String|ArrayBuffer|Uint8Array|Buffer} the data.
           */


          var getBinaryData = function (file) {
            var result = getRawData(file),
                type = utils.getTypeOf(result);

            if (type === "string") {
              if (!file.options.binary) {
                // unicode text !
                // unicode string => binary string is a painful process, check if we can avoid it.
                if (support.nodebuffer) {
                  return nodeBuffer(result, "utf-8");
                }
              }

              return file.asBinary();
            }

            return result;
          };
          /**
           * Transform this._data into a string.
           * @param {function} filter a function String -> String, applied if not null on the result.
           * @return {String} the string representing this._data.
           */


          var dataToString = function (asUTF8) {
            var result = getRawData(this);

            if (result === null || typeof result === "undefined") {
              return "";
            } // if the data is a base64 string, we decode it before checking the encoding !


            if (this.options.base64) {
              result = base64.decode(result);
            }

            if (asUTF8 && this.options.binary) {
              // JSZip.prototype.utf8decode supports arrays as input
              // skip to array => string step, utf8decode will do it.
              result = out.utf8decode(result);
            } else {
              // no utf8 transformation, do the array => string step.
              result = utils.transformTo("string", result);
            }

            if (!asUTF8 && !this.options.binary) {
              result = utils.transformTo("string", out.utf8encode(result));
            }

            return result;
          };
          /**
           * A simple object representing a file in the zip file.
           * @constructor
           * @param {string} name the name of the file
           * @param {String|ArrayBuffer|Uint8Array|Buffer} data the data
           * @param {Object} options the options of the file
           */


          var ZipObject = function (name, data, options) {
            this.name = name;
            this.dir = options.dir;
            this.date = options.date;
            this.comment = options.comment;
            this._data = data;
            this.options = options;
            /*
             * This object contains initial values for dir and date.
             * With them, we can check if the user changed the deprecated metadata in
             * `ZipObject#options` or not.
             */

            this._initialMetadata = {
              dir: options.dir,
              date: options.date
            };
          };

          ZipObject.prototype = {
            /**
             * Return the content as UTF8 string.
             * @return {string} the UTF8 string.
             */
            asText: function () {
              return dataToString.call(this, true);
            },

            /**
             * Returns the binary content.
             * @return {string} the content as binary.
             */
            asBinary: function () {
              return dataToString.call(this, false);
            },

            /**
             * Returns the content as a nodejs Buffer.
             * @return {Buffer} the content as a Buffer.
             */
            asNodeBuffer: function () {
              var result = getBinaryData(this);
              return utils.transformTo("nodebuffer", result);
            },

            /**
             * Returns the content as an Uint8Array.
             * @return {Uint8Array} the content as an Uint8Array.
             */
            asUint8Array: function () {
              var result = getBinaryData(this);
              return utils.transformTo("uint8array", result);
            },

            /**
             * Returns the content as an ArrayBuffer.
             * @return {ArrayBuffer} the content as an ArrayBufer.
             */
            asArrayBuffer: function () {
              return this.asUint8Array().buffer;
            }
          };
          /**
           * Transform an integer into a string in hexadecimal.
           * @private
           * @param {number} dec the number to convert.
           * @param {number} bytes the number of bytes to generate.
           * @returns {string} the result.
           */

          var decToHex = function (dec, bytes) {
            var hex = "",
                i;

            for (i = 0; i < bytes; i++) {
              hex += String.fromCharCode(dec & 0xff);
              dec = dec >>> 8;
            }

            return hex;
          };
          /**
           * Merge the objects passed as parameters into a new one.
           * @private
           * @param {...Object} var_args All objects to merge.
           * @return {Object} a new object with the data of the others.
           */


          var extend = function () {
            var result = {},
                i,
                attr;

            for (i = 0; i < arguments.length; i++) {
              // arguments is not enumerable in some browsers
              for (attr in arguments[i]) {
                if (arguments[i].hasOwnProperty(attr) && typeof result[attr] === "undefined") {
                  result[attr] = arguments[i][attr];
                }
              }
            }

            return result;
          };
          /**
           * Transforms the (incomplete) options from the user into the complete
           * set of options to create a file.
           * @private
           * @param {Object} o the options from the user.
           * @return {Object} the complete set of options.
           */


          var prepareFileAttrs = function (o) {
            o = o || {};

            if (o.base64 === true && (o.binary === null || o.binary === undefined)) {
              o.binary = true;
            }

            o = extend(o, defaults);
            o.date = o.date || new Date();
            if (o.compression !== null) o.compression = o.compression.toUpperCase();
            return o;
          };
          /**
           * Add a file in the current folder.
           * @private
           * @param {string} name the name of the file
           * @param {String|ArrayBuffer|Uint8Array|Buffer} data the data of the file
           * @param {Object} o the options of the file
           * @return {Object} the new file.
           */


          var fileAdd = function (name, data, o) {
            // be sure sub folders exist
            var dataType = utils.getTypeOf(data),
                parent;
            o = prepareFileAttrs(o);

            if (o.createFolders && (parent = parentFolder(name))) {
              folderAdd.call(this, parent, true);
            }

            if (o.dir || data === null || typeof data === "undefined") {
              o.base64 = false;
              o.binary = false;
              data = null;
            } else if (dataType === "string") {
              if (o.binary && !o.base64) {
                // optimizedBinaryString == true means that the file has already been filtered with a 0xFF mask
                if (o.optimizedBinaryString !== true) {
                  // this is a string, not in a base64 format.
                  // Be sure that this is a correct "binary string"
                  data = utils.string2binary(data);
                }
              }
            } else {
              // arraybuffer, uint8array, ...
              o.base64 = false;
              o.binary = true;

              if (!dataType && !(data instanceof CompressedObject)) {
                throw new Error("The data of '" + name + "' is in an unsupported format !");
              } // special case : it's way easier to work with Uint8Array than with ArrayBuffer


              if (dataType === "arraybuffer") {
                data = utils.transformTo("uint8array", data);
              }
            }

            var object = new ZipObject(name, data, o);
            this.files[name] = object;
            return object;
          };
          /**
           * Find the parent folder of the path.
           * @private
           * @param {string} path the path to use
           * @return {string} the parent folder, or ""
           */


          var parentFolder = function (path) {
            if (path.slice(-1) == '/') {
              path = path.substring(0, path.length - 1);
            }

            var lastSlash = path.lastIndexOf('/');
            return lastSlash > 0 ? path.substring(0, lastSlash) : "";
          };
          /**
           * Add a (sub) folder in the current folder.
           * @private
           * @param {string} name the folder's name
           * @param {boolean=} [createFolders] If true, automatically create sub
           *  folders. Defaults to false.
           * @return {Object} the new folder.
           */


          var folderAdd = function (name, createFolders) {
            // Check the name ends with a /
            if (name.slice(-1) != "/") {
              name += "/"; // IE doesn't like substr(-1)
            }

            createFolders = typeof createFolders !== 'undefined' ? createFolders : false; // Does this folder already exist?

            if (!this.files[name]) {
              fileAdd.call(this, name, null, {
                dir: true,
                createFolders: createFolders
              });
            }

            return this.files[name];
          };
          /**
           * Generate a JSZip.CompressedObject for a given zipOject.
           * @param {ZipObject} file the object to read.
           * @param {JSZip.compression} compression the compression to use.
           * @return {JSZip.CompressedObject} the compressed result.
           */


          var generateCompressedObjectFrom = function (file, compression) {
            var result = new CompressedObject(),
                content; // the data has not been decompressed, we might reuse things !

            if (file._data instanceof CompressedObject) {
              result.uncompressedSize = file._data.uncompressedSize;
              result.crc32 = file._data.crc32;

              if (result.uncompressedSize === 0 || file.dir) {
                compression = compressions['STORE'];
                result.compressedContent = "";
                result.crc32 = 0;
              } else if (file._data.compressionMethod === compression.magic) {
                result.compressedContent = file._data.getCompressedContent();
              } else {
                content = file._data.getContent(); // need to decompress / recompress

                result.compressedContent = compression.compress(utils.transformTo(compression.compressInputType, content));
              }
            } else {
              // have uncompressed data
              content = getBinaryData(file);

              if (!content || content.length === 0 || file.dir) {
                compression = compressions['STORE'];
                content = "";
              }

              result.uncompressedSize = content.length;
              result.crc32 = crc32(content);
              result.compressedContent = compression.compress(utils.transformTo(compression.compressInputType, content));
            }

            result.compressedSize = result.compressedContent.length;
            result.compressionMethod = compression.magic;
            return result;
          };
          /**
           * Generate the various parts used in the construction of the final zip file.
           * @param {string} name the file name.
           * @param {ZipObject} file the file content.
           * @param {JSZip.CompressedObject} compressedObject the compressed object.
           * @param {number} offset the current offset from the start of the zip file.
           * @return {object} the zip parts.
           */


          var generateZipParts = function (name, file, compressedObject, offset) {
            compressedObject.compressedContent;
                var utfEncodedFileName = utils.transformTo("string", utf8.utf8encode(file.name)),
                comment = file.comment || "",
                utfEncodedComment = utils.transformTo("string", utf8.utf8encode(comment)),
                useUTF8ForFileName = utfEncodedFileName.length !== file.name.length,
                useUTF8ForComment = utfEncodedComment.length !== comment.length,
                o = file.options,
                dosTime,
                dosDate,
                extraFields = "",
                unicodePathExtraField = "",
                unicodeCommentExtraField = "",
                dir,
                date; // handle the deprecated options.dir

            if (file._initialMetadata.dir !== file.dir) {
              dir = file.dir;
            } else {
              dir = o.dir;
            } // handle the deprecated options.date


            if (file._initialMetadata.date !== file.date) {
              date = file.date;
            } else {
              date = o.date;
            }

            dosTime = date.getHours();
            dosTime = dosTime << 6;
            dosTime = dosTime | date.getMinutes();
            dosTime = dosTime << 5;
            dosTime = dosTime | date.getSeconds() / 2;
            dosDate = date.getFullYear() - 1980;
            dosDate = dosDate << 4;
            dosDate = dosDate | date.getMonth() + 1;
            dosDate = dosDate << 5;
            dosDate = dosDate | date.getDate();

            if (useUTF8ForFileName) {
              // set the unicode path extra field. unzip needs at least one extra
              // field to correctly handle unicode path, so using the path is as good
              // as any other information. This could improve the situation with
              // other archive managers too.
              // This field is usually used without the utf8 flag, with a non
              // unicode path in the header (winrar, winzip). This helps (a bit)
              // with the messy Windows' default compressed folders feature but
              // breaks on p7zip which doesn't seek the unicode path extra field.
              // So for now, UTF-8 everywhere !
              unicodePathExtraField = // Version
              decToHex(1, 1) + // NameCRC32
              decToHex(crc32(utfEncodedFileName), 4) + // UnicodeName
              utfEncodedFileName;
              extraFields += // Info-ZIP Unicode Path Extra Field
              "\x75\x70" + // size
              decToHex(unicodePathExtraField.length, 2) + // content
              unicodePathExtraField;
            }

            if (useUTF8ForComment) {
              unicodeCommentExtraField = // Version
              decToHex(1, 1) + // CommentCRC32
              decToHex(this.crc32(utfEncodedComment), 4) + // UnicodeName
              utfEncodedComment;
              extraFields += // Info-ZIP Unicode Path Extra Field
              "\x75\x63" + // size
              decToHex(unicodeCommentExtraField.length, 2) + // content
              unicodeCommentExtraField;
            }

            var header = ""; // version needed to extract

            header += "\x0A\x00"; // general purpose bit flag
            // set bit 11 if utf8

            header += useUTF8ForFileName || useUTF8ForComment ? "\x00\x08" : "\x00\x00"; // compression method

            header += compressedObject.compressionMethod; // last mod file time

            header += decToHex(dosTime, 2); // last mod file date

            header += decToHex(dosDate, 2); // crc-32

            header += decToHex(compressedObject.crc32, 4); // compressed size

            header += decToHex(compressedObject.compressedSize, 4); // uncompressed size

            header += decToHex(compressedObject.uncompressedSize, 4); // file name length

            header += decToHex(utfEncodedFileName.length, 2); // extra field length

            header += decToHex(extraFields.length, 2);
            var fileRecord = signature.LOCAL_FILE_HEADER + header + utfEncodedFileName + extraFields;
            var dirRecord = signature.CENTRAL_FILE_HEADER + // version made by (00: DOS)
            "\x14\x00" + // file header (common to file and central directory)
            header + // file comment length
            decToHex(utfEncodedComment.length, 2) + // disk number start
            "\x00\x00" + // internal file attributes TODO
            "\x00\x00" + ( // external file attributes
            dir === true ? "\x10\x00\x00\x00" : "\x00\x00\x00\x00") + // relative offset of local header
            decToHex(offset, 4) + // file name
            utfEncodedFileName + // extra field
            extraFields + // file comment
            utfEncodedComment;
            return {
              fileRecord: fileRecord,
              dirRecord: dirRecord,
              compressedObject: compressedObject
            };
          }; // return the actual prototype of JSZip


          var out = {
            /**
             * Read an existing zip and merge the data in the current JSZip object.
             * The implementation is in jszip-load.js, don't forget to include it.
             * @param {String|ArrayBuffer|Uint8Array|Buffer} stream  The stream to load
             * @param {Object} options Options for loading the stream.
             *  options.base64 : is the stream in base64 ? default : false
             * @return {JSZip} the current JSZip object
             */
            load: function (stream, options) {
              throw new Error("Load method is not defined. Is the file jszip-load.js included ?");
            },

            /**
             * Filter nested files/folders with the specified function.
             * @param {Function} search the predicate to use :
             * function (relativePath, file) {...}
             * It takes 2 arguments : the relative path and the file.
             * @return {Array} An array of matching elements.
             */
            filter: function (search) {
              var result = [],
                  filename,
                  relativePath,
                  file,
                  fileClone;

              for (filename in this.files) {
                if (!this.files.hasOwnProperty(filename)) {
                  continue;
                }

                file = this.files[filename]; // return a new object, don't let the user mess with our internal objects :)

                fileClone = new ZipObject(file.name, file._data, extend(file.options));
                relativePath = filename.slice(this.root.length, filename.length);

                if (filename.slice(0, this.root.length) === this.root && // the file is in the current root
                search(relativePath, fileClone)) {
                  // and the file matches the function
                  result.push(fileClone);
                }
              }

              return result;
            },

            /**
             * Add a file to the zip file, or search a file.
             * @param   {string|RegExp} name The name of the file to add (if data is defined),
             * the name of the file to find (if no data) or a regex to match files.
             * @param   {String|ArrayBuffer|Uint8Array|Buffer} data  The file data, either raw or base64 encoded
             * @param   {Object} o     File options
             * @return  {JSZip|Object|Array} this JSZip object (when adding a file),
             * a file (when searching by string) or an array of files (when searching by regex).
             */
            file: function (name, data, o) {
              if (arguments.length === 1) {
                if (utils.isRegExp(name)) {
                  var regexp = name;
                  return this.filter(function (relativePath, file) {
                    return !file.dir && regexp.test(relativePath);
                  });
                } else {
                  // text
                  return this.filter(function (relativePath, file) {
                    return !file.dir && relativePath === name;
                  })[0] || null;
                }
              } else {
                // more than one argument : we have data !
                name = this.root + name;
                fileAdd.call(this, name, data, o);
              }

              return this;
            },

            /**
             * Add a directory to the zip file, or search.
             * @param   {String|RegExp} arg The name of the directory to add, or a regex to search folders.
             * @return  {JSZip} an object with the new directory as the root, or an array containing matching folders.
             */
            folder: function (arg) {
              if (!arg) {
                return this;
              }

              if (utils.isRegExp(arg)) {
                return this.filter(function (relativePath, file) {
                  return file.dir && arg.test(relativePath);
                });
              } // else, name is a new folder


              var name = this.root + arg;
              var newFolder = folderAdd.call(this, name); // Allow chaining by returning a new object with this folder as the root

              var ret = this.clone();
              ret.root = newFolder.name;
              return ret;
            },

            /**
             * Delete a file, or a directory and all sub-files, from the zip
             * @param {string} name the name of the file to delete
             * @return {JSZip} this JSZip object
             */
            remove: function (name) {
              name = this.root + name;
              var file = this.files[name];

              if (!file) {
                // Look for any folders
                if (name.slice(-1) != "/") {
                  name += "/";
                }

                file = this.files[name];
              }

              if (file && !file.dir) {
                // file
                delete this.files[name];
              } else {
                // maybe a folder, delete recursively
                var kids = this.filter(function (relativePath, file) {
                  return file.name.slice(0, name.length) === name;
                });

                for (var i = 0; i < kids.length; i++) {
                  delete this.files[kids[i].name];
                }
              }

              return this;
            },

            /**
             * Generate the complete zip file
             * @param {Object} options the options to generate the zip file :
             * - base64, (deprecated, use type instead) true to generate base64.
             * - compression, "STORE" by default.
             * - type, "base64" by default. Values are : string, base64, uint8array, arraybuffer, blob.
             * @return {String|Uint8Array|ArrayBuffer|Buffer|Blob} the zip file
             */
            generate: function (options) {
              options = extend(options || {}, {
                base64: true,
                compression: "STORE",
                type: "base64",
                comment: null
              });
              utils.checkSupport(options.type);
              var zipData = [],
                  localDirLength = 0,
                  centralDirLength = 0,
                  writer,
                  i,
                  utfEncodedComment = utils.transformTo("string", this.utf8encode(options.comment || this.comment || "")); // first, generate all the zip parts.

              for (var name in this.files) {
                if (!this.files.hasOwnProperty(name)) {
                  continue;
                }

                var file = this.files[name];
                var compressionName = file.options.compression || options.compression.toUpperCase();
                var compression = compressions[compressionName];

                if (!compression) {
                  throw new Error(compressionName + " is not a valid compression method !");
                }

                var compressedObject = generateCompressedObjectFrom.call(this, file, compression);
                var zipPart = generateZipParts.call(this, name, file, compressedObject, localDirLength);
                localDirLength += zipPart.fileRecord.length + compressedObject.compressedSize;
                centralDirLength += zipPart.dirRecord.length;
                zipData.push(zipPart);
              }

              var dirEnd = ""; // end of central dir signature

              dirEnd = signature.CENTRAL_DIRECTORY_END + // number of this disk
              "\x00\x00" + // number of the disk with the start of the central directory
              "\x00\x00" + // total number of entries in the central directory on this disk
              decToHex(zipData.length, 2) + // total number of entries in the central directory
              decToHex(zipData.length, 2) + // size of the central directory   4 bytes
              decToHex(centralDirLength, 4) + // offset of start of central directory with respect to the starting disk number
              decToHex(localDirLength, 4) + // .ZIP file comment length
              decToHex(utfEncodedComment.length, 2) + // .ZIP file comment
              utfEncodedComment; // we have all the parts (and the total length)
              // time to create a writer !

              var typeName = options.type.toLowerCase();

              if (typeName === "uint8array" || typeName === "arraybuffer" || typeName === "blob" || typeName === "nodebuffer") {
                writer = new Uint8ArrayWriter(localDirLength + centralDirLength + dirEnd.length);
              } else {
                writer = new StringWriter(localDirLength + centralDirLength + dirEnd.length);
              }

              for (i = 0; i < zipData.length; i++) {
                writer.append(zipData[i].fileRecord);
                writer.append(zipData[i].compressedObject.compressedContent);
              }

              for (i = 0; i < zipData.length; i++) {
                writer.append(zipData[i].dirRecord);
              }

              writer.append(dirEnd);
              var zip = writer.finalize();

              switch (options.type.toLowerCase()) {
                // case "zip is an Uint8Array"
                case "uint8array":
                case "arraybuffer":
                case "nodebuffer":
                  return utils.transformTo(options.type.toLowerCase(), zip);

                case "blob":
                  return utils.arrayBuffer2Blob(utils.transformTo("arraybuffer", zip));
                // case "zip is a string"

                case "base64":
                  return options.base64 ? base64.encode(zip) : zip;

                default:
                  // case "string" :
                  return zip;
              }
            },

            /**
             * @deprecated
             * This method will be removed in a future version without replacement.
             */
            crc32: function (input, crc) {
              return crc32(input, crc);
            },

            /**
             * @deprecated
             * This method will be removed in a future version without replacement.
             */
            utf8encode: function (string) {
              return utils.transformTo("string", utf8.utf8encode(string));
            },

            /**
             * @deprecated
             * This method will be removed in a future version without replacement.
             */
            utf8decode: function (input) {
              return utf8.utf8decode(input);
            }
          };
          module.exports = out;
        }, {
          "./base64": 1,
          "./compressedObject": 2,
          "./compressions": 3,
          "./crc32": 4,
          "./defaults": 6,
          "./nodeBuffer": 11,
          "./signature": 14,
          "./stringWriter": 16,
          "./support": 17,
          "./uint8ArrayWriter": 19,
          "./utf8": 20,
          "./utils": 21
        }],
        14: [function (_dereq_, module, exports) {

          exports.LOCAL_FILE_HEADER = "PK\x03\x04";
          exports.CENTRAL_FILE_HEADER = "PK\x01\x02";
          exports.CENTRAL_DIRECTORY_END = "PK\x05\x06";
          exports.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK\x06\x07";
          exports.ZIP64_CENTRAL_DIRECTORY_END = "PK\x06\x06";
          exports.DATA_DESCRIPTOR = "PK\x07\x08";
        }, {}],
        15: [function (_dereq_, module, exports) {

          var DataReader = _dereq_('./dataReader');

          var utils = _dereq_('./utils');

          function StringReader(data, optimizedBinaryString) {
            this.data = data;

            if (!optimizedBinaryString) {
              this.data = utils.string2binary(this.data);
            }

            this.length = this.data.length;
            this.index = 0;
          }

          StringReader.prototype = new DataReader();
          /**
           * @see DataReader.byteAt
           */

          StringReader.prototype.byteAt = function (i) {
            return this.data.charCodeAt(i);
          };
          /**
           * @see DataReader.lastIndexOfSignature
           */


          StringReader.prototype.lastIndexOfSignature = function (sig) {
            return this.data.lastIndexOf(sig);
          };
          /**
           * @see DataReader.readData
           */


          StringReader.prototype.readData = function (size) {
            this.checkOffset(size); // this will work because the constructor applied the "& 0xff" mask.

            var result = this.data.slice(this.index, this.index + size);
            this.index += size;
            return result;
          };

          module.exports = StringReader;
        }, {
          "./dataReader": 5,
          "./utils": 21
        }],
        16: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');
          /**
           * An object to write any content to a string.
           * @constructor
           */


          var StringWriter = function () {
            this.data = [];
          };

          StringWriter.prototype = {
            /**
             * Append any content to the current string.
             * @param {Object} input the content to add.
             */
            append: function (input) {
              input = utils.transformTo("string", input);
              this.data.push(input);
            },

            /**
             * Finalize the construction an return the result.
             * @return {string} the generated string.
             */
            finalize: function () {
              return this.data.join("");
            }
          };
          module.exports = StringWriter;
        }, {
          "./utils": 21
        }],
        17: [function (_dereq_, module, exports) {
          (function (Buffer) {

            exports.base64 = true;
            exports.array = true;
            exports.string = true;
            exports.arraybuffer = typeof ArrayBuffer !== "undefined" && typeof Uint8Array !== "undefined"; // contains true if JSZip can read/generate nodejs Buffer, false otherwise.
            // Browserify will provide a Buffer implementation for browsers, which is
            // an augmented Uint8Array (i.e., can be used as either Buffer or U8).

            exports.nodebuffer = typeof Buffer !== "undefined"; // contains true if JSZip can read/generate Uint8Array, false otherwise.

            exports.uint8array = typeof Uint8Array !== "undefined";

            if (typeof ArrayBuffer === "undefined") {
              exports.blob = false;
            } else {
              var buffer = new ArrayBuffer(0);

              try {
                exports.blob = new Blob([buffer], {
                  type: "application/zip"
                }).size === 0;
              } catch (e) {
                try {
                  var Builder = window.BlobBuilder || window.WebKitBlobBuilder || window.MozBlobBuilder || window.MSBlobBuilder;
                  var builder = new Builder();
                  builder.append(buffer);
                  exports.blob = builder.getBlob('application/zip').size === 0;
                } catch (e) {
                  exports.blob = false;
                }
              }
            }
          }).call(this, typeof Buffer !== "undefined" ? Buffer : undefined);
        }, {}],
        18: [function (_dereq_, module, exports) {

          var DataReader = _dereq_('./dataReader');

          function Uint8ArrayReader(data) {
            if (data) {
              this.data = data;
              this.length = this.data.length;
              this.index = 0;
            }
          }

          Uint8ArrayReader.prototype = new DataReader();
          /**
           * @see DataReader.byteAt
           */

          Uint8ArrayReader.prototype.byteAt = function (i) {
            return this.data[i];
          };
          /**
           * @see DataReader.lastIndexOfSignature
           */


          Uint8ArrayReader.prototype.lastIndexOfSignature = function (sig) {
            var sig0 = sig.charCodeAt(0),
                sig1 = sig.charCodeAt(1),
                sig2 = sig.charCodeAt(2),
                sig3 = sig.charCodeAt(3);

            for (var i = this.length - 4; i >= 0; --i) {
              if (this.data[i] === sig0 && this.data[i + 1] === sig1 && this.data[i + 2] === sig2 && this.data[i + 3] === sig3) {
                return i;
              }
            }

            return -1;
          };
          /**
           * @see DataReader.readData
           */


          Uint8ArrayReader.prototype.readData = function (size) {
            this.checkOffset(size);

            if (size === 0) {
              // in IE10, when using subarray(idx, idx), we get the array [0x00] instead of [].
              return new Uint8Array(0);
            }

            var result = this.data.subarray(this.index, this.index + size);
            this.index += size;
            return result;
          };

          module.exports = Uint8ArrayReader;
        }, {
          "./dataReader": 5
        }],
        19: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');
          /**
           * An object to write any content to an Uint8Array.
           * @constructor
           * @param {number} length The length of the array.
           */


          var Uint8ArrayWriter = function (length) {
            this.data = new Uint8Array(length);
            this.index = 0;
          };

          Uint8ArrayWriter.prototype = {
            /**
             * Append any content to the current array.
             * @param {Object} input the content to add.
             */
            append: function (input) {
              if (input.length !== 0) {
                // with an empty Uint8Array, Opera fails with a "Offset larger than array size"
                input = utils.transformTo("uint8array", input);
                this.data.set(input, this.index);
                this.index += input.length;
              }
            },

            /**
             * Finalize the construction an return the result.
             * @return {Uint8Array} the generated array.
             */
            finalize: function () {
              return this.data;
            }
          };
          module.exports = Uint8ArrayWriter;
        }, {
          "./utils": 21
        }],
        20: [function (_dereq_, module, exports) {

          var utils = _dereq_('./utils');

          var support = _dereq_('./support');

          var nodeBuffer = _dereq_('./nodeBuffer');
          /**
           * The following functions come from pako, from pako/lib/utils/strings
           * released under the MIT license, see pako https://github.com/nodeca/pako/
           */
          // Table with utf8 lengths (calculated by first byte of sequence)
          // Note, that 5 & 6-byte values and some 4-byte values can not be represented in JS,
          // because max possible codepoint is 0x10ffff


          var _utf8len = new Array(256);

          for (var i = 0; i < 256; i++) {
            _utf8len[i] = i >= 252 ? 6 : i >= 248 ? 5 : i >= 240 ? 4 : i >= 224 ? 3 : i >= 192 ? 2 : 1;
          }

          _utf8len[254] = _utf8len[254] = 1; // Invalid sequence start
          // convert string to array (typed, when possible)

          var string2buf = function (str) {
            var buf,
                c,
                c2,
                m_pos,
                i,
                str_len = str.length,
                buf_len = 0; // count binary size

            for (m_pos = 0; m_pos < str_len; m_pos++) {
              c = str.charCodeAt(m_pos);

              if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
                c2 = str.charCodeAt(m_pos + 1);

                if ((c2 & 0xfc00) === 0xdc00) {
                  c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
                  m_pos++;
                }
              }

              buf_len += c < 0x80 ? 1 : c < 0x800 ? 2 : c < 0x10000 ? 3 : 4;
            } // allocate buffer


            if (support.uint8array) {
              buf = new Uint8Array(buf_len);
            } else {
              buf = new Array(buf_len);
            } // convert


            for (i = 0, m_pos = 0; i < buf_len; m_pos++) {
              c = str.charCodeAt(m_pos);

              if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
                c2 = str.charCodeAt(m_pos + 1);

                if ((c2 & 0xfc00) === 0xdc00) {
                  c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
                  m_pos++;
                }
              }

              if (c < 0x80) {
                /* one byte */
                buf[i++] = c;
              } else if (c < 0x800) {
                /* two bytes */
                buf[i++] = 0xC0 | c >>> 6;
                buf[i++] = 0x80 | c & 0x3f;
              } else if (c < 0x10000) {
                /* three bytes */
                buf[i++] = 0xE0 | c >>> 12;
                buf[i++] = 0x80 | c >>> 6 & 0x3f;
                buf[i++] = 0x80 | c & 0x3f;
              } else {
                /* four bytes */
                buf[i++] = 0xf0 | c >>> 18;
                buf[i++] = 0x80 | c >>> 12 & 0x3f;
                buf[i++] = 0x80 | c >>> 6 & 0x3f;
                buf[i++] = 0x80 | c & 0x3f;
              }
            }

            return buf;
          }; // Calculate max possible position in utf8 buffer,
          // that will not break sequence. If that's not possible
          // - (very small limits) return max size as is.
          //
          // buf[] - utf8 bytes array
          // max   - length limit (mandatory);


          var utf8border = function (buf, max) {
            var pos;
            max = max || buf.length;

            if (max > buf.length) {
              max = buf.length;
            } // go back from last position, until start of sequence found


            pos = max - 1;

            while (pos >= 0 && (buf[pos] & 0xC0) === 0x80) {
              pos--;
            } // Fuckup - very small and broken sequence,
            // return max, because we should return something anyway.


            if (pos < 0) {
              return max;
            } // If we came to start of buffer - that means vuffer is too small,
            // return max too.


            if (pos === 0) {
              return max;
            }

            return pos + _utf8len[buf[pos]] > max ? pos : max;
          }; // convert array to string


          var buf2string = function (buf) {
            var i, out, c, c_len;
            var len = buf.length; // Reserve max possible length (2 words per char)
            // NB: by unknown reasons, Array is significantly faster for
            //     String.fromCharCode.apply than Uint16Array.

            var utf16buf = new Array(len * 2);

            for (out = 0, i = 0; i < len;) {
              c = buf[i++]; // quick process ascii

              if (c < 0x80) {
                utf16buf[out++] = c;
                continue;
              }

              c_len = _utf8len[c]; // skip 5 & 6 byte codes

              if (c_len > 4) {
                utf16buf[out++] = 0xfffd;
                i += c_len - 1;
                continue;
              } // apply mask on first byte


              c &= c_len === 2 ? 0x1f : c_len === 3 ? 0x0f : 0x07; // join the rest

              while (c_len > 1 && i < len) {
                c = c << 6 | buf[i++] & 0x3f;
                c_len--;
              } // terminated by end of string?


              if (c_len > 1) {
                utf16buf[out++] = 0xfffd;
                continue;
              }

              if (c < 0x10000) {
                utf16buf[out++] = c;
              } else {
                c -= 0x10000;
                utf16buf[out++] = 0xd800 | c >> 10 & 0x3ff;
                utf16buf[out++] = 0xdc00 | c & 0x3ff;
              }
            } // shrinkBuf(utf16buf, out)


            if (utf16buf.length !== out) {
              if (utf16buf.subarray) {
                utf16buf = utf16buf.subarray(0, out);
              } else {
                utf16buf.length = out;
              }
            } // return String.fromCharCode.apply(null, utf16buf);


            return utils.applyFromCharCode(utf16buf);
          }; // That's all for the pako functions.

          /**
           * Transform a javascript string into an array (typed if possible) of bytes,
           * UTF-8 encoded.
           * @param {String} str the string to encode
           * @return {Array|Uint8Array|Buffer} the UTF-8 encoded string.
           */


          exports.utf8encode = function utf8encode(str) {
            if (support.nodebuffer) {
              return nodeBuffer(str, "utf-8");
            }

            return string2buf(str);
          };
          /**
           * Transform a bytes array (or a representation) representing an UTF-8 encoded
           * string into a javascript string.
           * @param {Array|Uint8Array|Buffer} buf the data de decode
           * @return {String} the decoded string.
           */


          exports.utf8decode = function utf8decode(buf) {
            if (support.nodebuffer) {
              return utils.transformTo("nodebuffer", buf).toString("utf-8");
            }

            buf = utils.transformTo(support.uint8array ? "uint8array" : "array", buf); // return buf2string(buf);
            // Chrome prefers to work with "small" chunks of data
            // for the method buf2string.
            // Firefox and Chrome has their own shortcut, IE doesn't seem to really care.

            var result = [],
                k = 0,
                len = buf.length,
                chunk = 65536;

            while (k < len) {
              var nextBoundary = utf8border(buf, Math.min(k + chunk, len));

              if (support.uint8array) {
                result.push(buf2string(buf.subarray(k, nextBoundary)));
              } else {
                result.push(buf2string(buf.slice(k, nextBoundary)));
              }

              k = nextBoundary;
            }

            return result.join("");
          }; // vim: set shiftwidth=4 softtabstop=4:

        }, {
          "./nodeBuffer": 11,
          "./support": 17,
          "./utils": 21
        }],
        21: [function (_dereq_, module, exports) {

          var support = _dereq_('./support');

          var compressions = _dereq_('./compressions');

          var nodeBuffer = _dereq_('./nodeBuffer');
          /**
           * Convert a string to a "binary string" : a string containing only char codes between 0 and 255.
           * @param {string} str the string to transform.
           * @return {String} the binary string.
           */


          exports.string2binary = function (str) {
            var result = "";

            for (var i = 0; i < str.length; i++) {
              result += String.fromCharCode(str.charCodeAt(i) & 0xff);
            }

            return result;
          };

          exports.arrayBuffer2Blob = function (buffer) {
            exports.checkSupport("blob");

            try {
              // Blob constructor
              return new Blob([buffer], {
                type: "application/zip"
              });
            } catch (e) {
              try {
                // deprecated, browser only, old way
                var Builder = window.BlobBuilder || window.WebKitBlobBuilder || window.MozBlobBuilder || window.MSBlobBuilder;
                var builder = new Builder();
                builder.append(buffer);
                return builder.getBlob('application/zip');
              } catch (e) {
                // well, fuck ?!
                throw new Error("Bug : can't construct the Blob.");
              }
            }
          };
          /**
           * The identity function.
           * @param {Object} input the input.
           * @return {Object} the same input.
           */


          function identity(input) {
            return input;
          }
          /**
           * Fill in an array with a string.
           * @param {String} str the string to use.
           * @param {Array|ArrayBuffer|Uint8Array|Buffer} array the array to fill in (will be mutated).
           * @return {Array|ArrayBuffer|Uint8Array|Buffer} the updated array.
           */


          function stringToArrayLike(str, array) {
            for (var i = 0; i < str.length; ++i) {
              array[i] = str.charCodeAt(i) & 0xFF;
            }

            return array;
          }
          /**
           * Transform an array-like object to a string.
           * @param {Array|ArrayBuffer|Uint8Array|Buffer} array the array to transform.
           * @return {String} the result.
           */


          function arrayLikeToString(array) {
            // Performances notes :
            // --------------------
            // String.fromCharCode.apply(null, array) is the fastest, see
            // see http://jsperf.com/converting-a-uint8array-to-a-string/2
            // but the stack is limited (and we can get huge arrays !).
            //
            // result += String.fromCharCode(array[i]); generate too many strings !
            //
            // This code is inspired by http://jsperf.com/arraybuffer-to-string-apply-performance/2
            var chunk = 65536;
            var result = [],
                len = array.length,
                type = exports.getTypeOf(array),
                k = 0,
                canUseApply = true;

            try {
              switch (type) {
                case "uint8array":
                  String.fromCharCode.apply(null, new Uint8Array(0));
                  break;

                case "nodebuffer":
                  String.fromCharCode.apply(null, nodeBuffer(0));
                  break;
              }
            } catch (e) {
              canUseApply = false;
            } // no apply : slow and painful algorithm
            // default browser on android 4.*


            if (!canUseApply) {
              var resultStr = "";

              for (var i = 0; i < array.length; i++) {
                resultStr += String.fromCharCode(array[i]);
              }

              return resultStr;
            }

            while (k < len && chunk > 1) {
              try {
                if (type === "array" || type === "nodebuffer") {
                  result.push(String.fromCharCode.apply(null, array.slice(k, Math.min(k + chunk, len))));
                } else {
                  result.push(String.fromCharCode.apply(null, array.subarray(k, Math.min(k + chunk, len))));
                }

                k += chunk;
              } catch (e) {
                chunk = Math.floor(chunk / 2);
              }
            }

            return result.join("");
          }

          exports.applyFromCharCode = arrayLikeToString;
          /**
           * Copy the data from an array-like to an other array-like.
           * @param {Array|ArrayBuffer|Uint8Array|Buffer} arrayFrom the origin array.
           * @param {Array|ArrayBuffer|Uint8Array|Buffer} arrayTo the destination array which will be mutated.
           * @return {Array|ArrayBuffer|Uint8Array|Buffer} the updated destination array.
           */

          function arrayLikeToArrayLike(arrayFrom, arrayTo) {
            for (var i = 0; i < arrayFrom.length; i++) {
              arrayTo[i] = arrayFrom[i];
            }

            return arrayTo;
          } // a matrix containing functions to transform everything into everything.


          var transform = {}; // string to ?

          transform["string"] = {
            "string": identity,
            "array": function (input) {
              return stringToArrayLike(input, new Array(input.length));
            },
            "arraybuffer": function (input) {
              return transform["string"]["uint8array"](input).buffer;
            },
            "uint8array": function (input) {
              return stringToArrayLike(input, new Uint8Array(input.length));
            },
            "nodebuffer": function (input) {
              return stringToArrayLike(input, nodeBuffer(input.length));
            }
          }; // array to ?

          transform["array"] = {
            "string": arrayLikeToString,
            "array": identity,
            "arraybuffer": function (input) {
              return new Uint8Array(input).buffer;
            },
            "uint8array": function (input) {
              return new Uint8Array(input);
            },
            "nodebuffer": function (input) {
              return nodeBuffer(input);
            }
          }; // arraybuffer to ?

          transform["arraybuffer"] = {
            "string": function (input) {
              return arrayLikeToString(new Uint8Array(input));
            },
            "array": function (input) {
              return arrayLikeToArrayLike(new Uint8Array(input), new Array(input.byteLength));
            },
            "arraybuffer": identity,
            "uint8array": function (input) {
              return new Uint8Array(input);
            },
            "nodebuffer": function (input) {
              return nodeBuffer(new Uint8Array(input));
            }
          }; // uint8array to ?

          transform["uint8array"] = {
            "string": arrayLikeToString,
            "array": function (input) {
              return arrayLikeToArrayLike(input, new Array(input.length));
            },
            "arraybuffer": function (input) {
              return input.buffer;
            },
            "uint8array": identity,
            "nodebuffer": function (input) {
              return nodeBuffer(input);
            }
          }; // nodebuffer to ?

          transform["nodebuffer"] = {
            "string": arrayLikeToString,
            "array": function (input) {
              return arrayLikeToArrayLike(input, new Array(input.length));
            },
            "arraybuffer": function (input) {
              return transform["nodebuffer"]["uint8array"](input).buffer;
            },
            "uint8array": function (input) {
              return arrayLikeToArrayLike(input, new Uint8Array(input.length));
            },
            "nodebuffer": identity
          };
          /**
           * Transform an input into any type.
           * The supported output type are : string, array, uint8array, arraybuffer, nodebuffer.
           * If no output type is specified, the unmodified input will be returned.
           * @param {String} outputType the output type.
           * @param {String|Array|ArrayBuffer|Uint8Array|Buffer} input the input to convert.
           * @throws {Error} an Error if the browser doesn't support the requested output type.
           */

          exports.transformTo = function (outputType, input) {
            if (!input) {
              // undefined, null, etc
              // an empty string won't harm.
              input = "";
            }

            if (!outputType) {
              return input;
            }

            exports.checkSupport(outputType);
            var inputType = exports.getTypeOf(input);
            var result = transform[inputType][outputType](input);
            return result;
          };
          /**
           * Return the type of the input.
           * The type will be in a format valid for JSZip.utils.transformTo : string, array, uint8array, arraybuffer.
           * @param {Object} input the input to identify.
           * @return {String} the (lowercase) type of the input.
           */


          exports.getTypeOf = function (input) {
            if (typeof input === "string") {
              return "string";
            }

            if (Object.prototype.toString.call(input) === "[object Array]") {
              return "array";
            }

            if (support.nodebuffer && nodeBuffer.test(input)) {
              return "nodebuffer";
            }

            if (support.uint8array && input instanceof Uint8Array) {
              return "uint8array";
            }

            if (support.arraybuffer && input instanceof ArrayBuffer) {
              return "arraybuffer";
            }
          };
          /**
           * Throw an exception if the type is not supported.
           * @param {String} type the type to check.
           * @throws {Error} an Error if the browser doesn't support the requested type.
           */


          exports.checkSupport = function (type) {
            var supported = support[type.toLowerCase()];

            if (!supported) {
              throw new Error(type + " is not supported by this browser");
            }
          };

          exports.MAX_VALUE_16BITS = 65535;
          exports.MAX_VALUE_32BITS = -1; // well, "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF" is parsed as -1

          /**
           * Prettify a string read as binary.
           * @param {string} str the string to prettify.
           * @return {string} a pretty string.
           */

          exports.pretty = function (str) {
            var res = '',
                code,
                i;

            for (i = 0; i < (str || "").length; i++) {
              code = str.charCodeAt(i);
              res += '\\x' + (code < 16 ? "0" : "") + code.toString(16).toUpperCase();
            }

            return res;
          };
          /**
           * Find a compression registered in JSZip.
           * @param {string} compressionMethod the method magic to find.
           * @return {Object|null} the JSZip compression object, null if none found.
           */


          exports.findCompression = function (compressionMethod) {
            for (var method in compressions) {
              if (!compressions.hasOwnProperty(method)) {
                continue;
              }

              if (compressions[method].magic === compressionMethod) {
                return compressions[method];
              }
            }

            return null;
          };
          /**
          * Cross-window, cross-Node-context regular expression detection
          * @param  {Object}  object Anything
          * @return {Boolean}        true if the object is a regular expression,
          * false otherwise
          */


          exports.isRegExp = function (object) {
            return Object.prototype.toString.call(object) === "[object RegExp]";
          };
        }, {
          "./compressions": 3,
          "./nodeBuffer": 11,
          "./support": 17
        }],
        22: [function (_dereq_, module, exports) {

          var StringReader = _dereq_('./stringReader');

          var NodeBufferReader = _dereq_('./nodeBufferReader');

          var Uint8ArrayReader = _dereq_('./uint8ArrayReader');

          var utils = _dereq_('./utils');

          var sig = _dereq_('./signature');

          var ZipEntry = _dereq_('./zipEntry');

          var support = _dereq_('./support');

          var jszipProto = _dereq_('./object'); //  class ZipEntries {{{

          /**
           * All the entries in the zip file.
           * @constructor
           * @param {String|ArrayBuffer|Uint8Array} data the binary stream to load.
           * @param {Object} loadOptions Options for loading the stream.
           */


          function ZipEntries(data, loadOptions) {
            this.files = [];
            this.loadOptions = loadOptions;

            if (data) {
              this.load(data);
            }
          }

          ZipEntries.prototype = {
            /**
             * Check that the reader is on the speficied signature.
             * @param {string} expectedSignature the expected signature.
             * @throws {Error} if it is an other signature.
             */
            checkSignature: function (expectedSignature) {
              var signature = this.reader.readString(4);

              if (signature !== expectedSignature) {
                throw new Error("Corrupted zip or bug : unexpected signature " + "(" + utils.pretty(signature) + ", expected " + utils.pretty(expectedSignature) + ")");
              }
            },

            /**
             * Read the end of the central directory.
             */
            readBlockEndOfCentral: function () {
              this.diskNumber = this.reader.readInt(2);
              this.diskWithCentralDirStart = this.reader.readInt(2);
              this.centralDirRecordsOnThisDisk = this.reader.readInt(2);
              this.centralDirRecords = this.reader.readInt(2);
              this.centralDirSize = this.reader.readInt(4);
              this.centralDirOffset = this.reader.readInt(4);
              this.zipCommentLength = this.reader.readInt(2); // warning : the encoding depends of the system locale
              // On a linux machine with LANG=en_US.utf8, this field is utf8 encoded.
              // On a windows machine, this field is encoded with the localized windows code page.

              this.zipComment = this.reader.readString(this.zipCommentLength); // To get consistent behavior with the generation part, we will assume that
              // this is utf8 encoded.

              this.zipComment = jszipProto.utf8decode(this.zipComment);
            },

            /**
             * Read the end of the Zip 64 central directory.
             * Not merged with the method readEndOfCentral :
             * The end of central can coexist with its Zip64 brother,
             * I don't want to read the wrong number of bytes !
             */
            readBlockZip64EndOfCentral: function () {
              this.zip64EndOfCentralSize = this.reader.readInt(8);
              this.versionMadeBy = this.reader.readString(2);
              this.versionNeeded = this.reader.readInt(2);
              this.diskNumber = this.reader.readInt(4);
              this.diskWithCentralDirStart = this.reader.readInt(4);
              this.centralDirRecordsOnThisDisk = this.reader.readInt(8);
              this.centralDirRecords = this.reader.readInt(8);
              this.centralDirSize = this.reader.readInt(8);
              this.centralDirOffset = this.reader.readInt(8);
              this.zip64ExtensibleData = {};
              var extraDataSize = this.zip64EndOfCentralSize - 44,
                  index = 0,
                  extraFieldId,
                  extraFieldLength,
                  extraFieldValue;

              while (index < extraDataSize) {
                extraFieldId = this.reader.readInt(2);
                extraFieldLength = this.reader.readInt(4);
                extraFieldValue = this.reader.readString(extraFieldLength);
                this.zip64ExtensibleData[extraFieldId] = {
                  id: extraFieldId,
                  length: extraFieldLength,
                  value: extraFieldValue
                };
              }
            },

            /**
             * Read the end of the Zip 64 central directory locator.
             */
            readBlockZip64EndOfCentralLocator: function () {
              this.diskWithZip64CentralDirStart = this.reader.readInt(4);
              this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8);
              this.disksCount = this.reader.readInt(4);

              if (this.disksCount > 1) {
                throw new Error("Multi-volumes zip are not supported");
              }
            },

            /**
             * Read the local files, based on the offset read in the central part.
             */
            readLocalFiles: function () {
              var i, file;

              for (i = 0; i < this.files.length; i++) {
                file = this.files[i];
                this.reader.setIndex(file.localHeaderOffset);
                this.checkSignature(sig.LOCAL_FILE_HEADER);
                file.readLocalPart(this.reader);
                file.handleUTF8();
              }
            },

            /**
             * Read the central directory.
             */
            readCentralDir: function () {
              var file;
              this.reader.setIndex(this.centralDirOffset);

              while (this.reader.readString(4) === sig.CENTRAL_FILE_HEADER) {
                file = new ZipEntry({
                  zip64: this.zip64
                }, this.loadOptions);
                file.readCentralPart(this.reader);
                this.files.push(file);
              }
            },

            /**
             * Read the end of central directory.
             */
            readEndOfCentral: function () {
              var offset = this.reader.lastIndexOfSignature(sig.CENTRAL_DIRECTORY_END);

              if (offset === -1) {
                throw new Error("Corrupted zip : can't find end of central directory");
              }

              this.reader.setIndex(offset);
              this.checkSignature(sig.CENTRAL_DIRECTORY_END);
              this.readBlockEndOfCentral();
              /* extract from the zip spec :
                  4)  If one of the fields in the end of central directory
                      record is too small to hold required data, the field
                      should be set to -1 (0xFFFF or 0xFFFFFFFF) and the
                      ZIP64 format record should be created.
                  5)  The end of central directory record and the
                      Zip64 end of central directory locator record must
                      reside on the same disk when splitting or spanning
                      an archive.
               */

              if (this.diskNumber === utils.MAX_VALUE_16BITS || this.diskWithCentralDirStart === utils.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === utils.MAX_VALUE_16BITS || this.centralDirRecords === utils.MAX_VALUE_16BITS || this.centralDirSize === utils.MAX_VALUE_32BITS || this.centralDirOffset === utils.MAX_VALUE_32BITS) {
                this.zip64 = true;
                /*
                Warning : the zip64 extension is supported, but ONLY if the 64bits integer read from
                the zip file can fit into a 32bits integer. This cannot be solved : Javascript represents
                all numbers as 64-bit double precision IEEE 754 floating point numbers.
                So, we have 53bits for integers and bitwise operations treat everything as 32bits.
                see https://developer.mozilla.org/en-US/docs/JavaScript/Reference/Operators/Bitwise_Operators
                and http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-262.pdf section 8.5
                */
                // should look for a zip64 EOCD locator

                offset = this.reader.lastIndexOfSignature(sig.ZIP64_CENTRAL_DIRECTORY_LOCATOR);

                if (offset === -1) {
                  throw new Error("Corrupted zip : can't find the ZIP64 end of central directory locator");
                }

                this.reader.setIndex(offset);
                this.checkSignature(sig.ZIP64_CENTRAL_DIRECTORY_LOCATOR);
                this.readBlockZip64EndOfCentralLocator(); // now the zip64 EOCD record

                this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir);
                this.checkSignature(sig.ZIP64_CENTRAL_DIRECTORY_END);
                this.readBlockZip64EndOfCentral();
              }
            },
            prepareReader: function (data) {
              var type = utils.getTypeOf(data);

              if (type === "string" && !support.uint8array) {
                this.reader = new StringReader(data, this.loadOptions.optimizedBinaryString);
              } else if (type === "nodebuffer") {
                this.reader = new NodeBufferReader(data);
              } else {
                this.reader = new Uint8ArrayReader(utils.transformTo("uint8array", data));
              }
            },

            /**
             * Read a zip file and create ZipEntries.
             * @param {String|ArrayBuffer|Uint8Array|Buffer} data the binary string representing a zip file.
             */
            load: function (data) {
              this.prepareReader(data);
              this.readEndOfCentral();
              this.readCentralDir();
              this.readLocalFiles();
            }
          }; // }}} end of ZipEntries

          module.exports = ZipEntries;
        }, {
          "./nodeBufferReader": 12,
          "./object": 13,
          "./signature": 14,
          "./stringReader": 15,
          "./support": 17,
          "./uint8ArrayReader": 18,
          "./utils": 21,
          "./zipEntry": 23
        }],
        23: [function (_dereq_, module, exports) {

          var StringReader = _dereq_('./stringReader');

          var utils = _dereq_('./utils');

          var CompressedObject = _dereq_('./compressedObject');

          var jszipProto = _dereq_('./object'); // class ZipEntry {{{

          /**
           * An entry in the zip file.
           * @constructor
           * @param {Object} options Options of the current file.
           * @param {Object} loadOptions Options for loading the stream.
           */


          function ZipEntry(options, loadOptions) {
            this.options = options;
            this.loadOptions = loadOptions;
          }

          ZipEntry.prototype = {
            /**
             * say if the file is encrypted.
             * @return {boolean} true if the file is encrypted, false otherwise.
             */
            isEncrypted: function () {
              // bit 1 is set
              return (this.bitFlag & 0x0001) === 0x0001;
            },

            /**
             * say if the file has utf-8 filename/comment.
             * @return {boolean} true if the filename/comment is in utf-8, false otherwise.
             */
            useUTF8: function () {
              // bit 11 is set
              return (this.bitFlag & 0x0800) === 0x0800;
            },

            /**
             * Prepare the function used to generate the compressed content from this ZipFile.
             * @param {DataReader} reader the reader to use.
             * @param {number} from the offset from where we should read the data.
             * @param {number} length the length of the data to read.
             * @return {Function} the callback to get the compressed content (the type depends of the DataReader class).
             */
            prepareCompressedContent: function (reader, from, length) {
              return function () {
                var previousIndex = reader.index;
                reader.setIndex(from);
                var compressedFileData = reader.readData(length);
                reader.setIndex(previousIndex);
                return compressedFileData;
              };
            },

            /**
             * Prepare the function used to generate the uncompressed content from this ZipFile.
             * @param {DataReader} reader the reader to use.
             * @param {number} from the offset from where we should read the data.
             * @param {number} length the length of the data to read.
             * @param {JSZip.compression} compression the compression used on this file.
             * @param {number} uncompressedSize the uncompressed size to expect.
             * @return {Function} the callback to get the uncompressed content (the type depends of the DataReader class).
             */
            prepareContent: function (reader, from, length, compression, uncompressedSize) {
              return function () {
                var compressedFileData = utils.transformTo(compression.uncompressInputType, this.getCompressedContent());
                var uncompressedFileData = compression.uncompress(compressedFileData);

                if (uncompressedFileData.length !== uncompressedSize) {
                  throw new Error("Bug : uncompressed data size mismatch");
                }

                return uncompressedFileData;
              };
            },

            /**
             * Read the local part of a zip file and add the info in this object.
             * @param {DataReader} reader the reader to use.
             */
            readLocalPart: function (reader) {
              var compression, localExtraFieldsLength; // we already know everything from the central dir !
              // If the central dir data are false, we are doomed.
              // On the bright side, the local part is scary  : zip64, data descriptors, both, etc.
              // The less data we get here, the more reliable this should be.
              // Let's skip the whole header and dash to the data !

              reader.skip(22); // in some zip created on windows, the filename stored in the central dir contains \ instead of /.
              // Strangely, the filename here is OK.
              // I would love to treat these zip files as corrupted (see http://www.info-zip.org/FAQ.html#backslashes
              // or APPNOTE#4.4.17.1, "All slashes MUST be forward slashes '/'") but there are a lot of bad zip generators...
              // Search "unzip mismatching "local" filename continuing with "central" filename version" on
              // the internet.
              //
              // I think I see the logic here : the central directory is used to display
              // content and the local directory is used to extract the files. Mixing / and \
              // may be used to display \ to windows users and use / when extracting the files.
              // Unfortunately, this lead also to some issues : http://seclists.org/fulldisclosure/2009/Sep/394

              this.fileNameLength = reader.readInt(2);
              localExtraFieldsLength = reader.readInt(2); // can't be sure this will be the same as the central dir

              this.fileName = reader.readString(this.fileNameLength);
              reader.skip(localExtraFieldsLength);

              if (this.compressedSize == -1 || this.uncompressedSize == -1) {
                throw new Error("Bug or corrupted zip : didn't get enough informations from the central directory " + "(compressedSize == -1 || uncompressedSize == -1)");
              }

              compression = utils.findCompression(this.compressionMethod);

              if (compression === null) {
                // no compression found
                throw new Error("Corrupted zip : compression " + utils.pretty(this.compressionMethod) + " unknown (inner file : " + this.fileName + ")");
              }

              this.decompressed = new CompressedObject();
              this.decompressed.compressedSize = this.compressedSize;
              this.decompressed.uncompressedSize = this.uncompressedSize;
              this.decompressed.crc32 = this.crc32;
              this.decompressed.compressionMethod = this.compressionMethod;
              this.decompressed.getCompressedContent = this.prepareCompressedContent(reader, reader.index, this.compressedSize, compression);
              this.decompressed.getContent = this.prepareContent(reader, reader.index, this.compressedSize, compression, this.uncompressedSize); // we need to compute the crc32...

              if (this.loadOptions.checkCRC32) {
                this.decompressed = utils.transformTo("string", this.decompressed.getContent());

                if (jszipProto.crc32(this.decompressed) !== this.crc32) {
                  throw new Error("Corrupted zip : CRC32 mismatch");
                }
              }
            },

            /**
             * Read the central part of a zip file and add the info in this object.
             * @param {DataReader} reader the reader to use.
             */
            readCentralPart: function (reader) {
              this.versionMadeBy = reader.readString(2);
              this.versionNeeded = reader.readInt(2);
              this.bitFlag = reader.readInt(2);
              this.compressionMethod = reader.readString(2);
              this.date = reader.readDate();
              this.crc32 = reader.readInt(4);
              this.compressedSize = reader.readInt(4);
              this.uncompressedSize = reader.readInt(4);
              this.fileNameLength = reader.readInt(2);
              this.extraFieldsLength = reader.readInt(2);
              this.fileCommentLength = reader.readInt(2);
              this.diskNumberStart = reader.readInt(2);
              this.internalFileAttributes = reader.readInt(2);
              this.externalFileAttributes = reader.readInt(4);
              this.localHeaderOffset = reader.readInt(4);

              if (this.isEncrypted()) {
                throw new Error("Encrypted zip are not supported");
              }

              this.fileName = reader.readString(this.fileNameLength);
              this.readExtraFields(reader);
              this.parseZIP64ExtraField(reader);
              this.fileComment = reader.readString(this.fileCommentLength); // warning, this is true only for zip with madeBy == DOS (plateform dependent feature)

              this.dir = this.externalFileAttributes & 0x00000010 ? true : false;
            },

            /**
             * Parse the ZIP64 extra field and merge the info in the current ZipEntry.
             * @param {DataReader} reader the reader to use.
             */
            parseZIP64ExtraField: function (reader) {
              if (!this.extraFields[0x0001]) {
                return;
              } // should be something, preparing the extra reader


              var extraReader = new StringReader(this.extraFields[0x0001].value); // I really hope that these 64bits integer can fit in 32 bits integer, because js
              // won't let us have more.

              if (this.uncompressedSize === utils.MAX_VALUE_32BITS) {
                this.uncompressedSize = extraReader.readInt(8);
              }

              if (this.compressedSize === utils.MAX_VALUE_32BITS) {
                this.compressedSize = extraReader.readInt(8);
              }

              if (this.localHeaderOffset === utils.MAX_VALUE_32BITS) {
                this.localHeaderOffset = extraReader.readInt(8);
              }

              if (this.diskNumberStart === utils.MAX_VALUE_32BITS) {
                this.diskNumberStart = extraReader.readInt(4);
              }
            },

            /**
             * Read the central part of a zip file and add the info in this object.
             * @param {DataReader} reader the reader to use.
             */
            readExtraFields: function (reader) {
              var start = reader.index,
                  extraFieldId,
                  extraFieldLength,
                  extraFieldValue;
              this.extraFields = this.extraFields || {};

              while (reader.index < start + this.extraFieldsLength) {
                extraFieldId = reader.readInt(2);
                extraFieldLength = reader.readInt(2);
                extraFieldValue = reader.readString(extraFieldLength);
                this.extraFields[extraFieldId] = {
                  id: extraFieldId,
                  length: extraFieldLength,
                  value: extraFieldValue
                };
              }
            },

            /**
             * Apply an UTF8 transformation if needed.
             */
            handleUTF8: function () {
              if (this.useUTF8()) {
                this.fileName = jszipProto.utf8decode(this.fileName);
                this.fileComment = jszipProto.utf8decode(this.fileComment);
              } else {
                var upath = this.findExtraFieldUnicodePath();

                if (upath !== null) {
                  this.fileName = upath;
                }

                var ucomment = this.findExtraFieldUnicodeComment();

                if (ucomment !== null) {
                  this.fileComment = ucomment;
                }
              }
            },

            /**
             * Find the unicode path declared in the extra field, if any.
             * @return {String} the unicode path, null otherwise.
             */
            findExtraFieldUnicodePath: function () {
              var upathField = this.extraFields[0x7075];

              if (upathField) {
                var extraReader = new StringReader(upathField.value); // wrong version

                if (extraReader.readInt(1) !== 1) {
                  return null;
                } // the crc of the filename changed, this field is out of date.


                if (jszipProto.crc32(this.fileName) !== extraReader.readInt(4)) {
                  return null;
                }

                return jszipProto.utf8decode(extraReader.readString(upathField.length - 5));
              }

              return null;
            },

            /**
             * Find the unicode comment declared in the extra field, if any.
             * @return {String} the unicode comment, null otherwise.
             */
            findExtraFieldUnicodeComment: function () {
              var ucommentField = this.extraFields[0x6375];

              if (ucommentField) {
                var extraReader = new StringReader(ucommentField.value); // wrong version

                if (extraReader.readInt(1) !== 1) {
                  return null;
                } // the crc of the comment changed, this field is out of date.


                if (jszipProto.crc32(this.fileComment) !== extraReader.readInt(4)) {
                  return null;
                }

                return jszipProto.utf8decode(extraReader.readString(ucommentField.length - 5));
              }

              return null;
            }
          };
          module.exports = ZipEntry;
        }, {
          "./compressedObject": 2,
          "./object": 13,
          "./stringReader": 15,
          "./utils": 21
        }],
        24: [function (_dereq_, module, exports) {

          var assign = _dereq_('./lib/utils/common').assign;

          var deflate = _dereq_('./lib/deflate');

          var inflate = _dereq_('./lib/inflate');

          var constants = _dereq_('./lib/zlib/constants');

          var pako = {};
          assign(pako, deflate, inflate, constants);
          module.exports = pako;
        }, {
          "./lib/deflate": 25,
          "./lib/inflate": 26,
          "./lib/utils/common": 27,
          "./lib/zlib/constants": 30
        }],
        25: [function (_dereq_, module, exports) {

          var zlib_deflate = _dereq_('./zlib/deflate.js');

          var utils = _dereq_('./utils/common');

          var strings = _dereq_('./utils/strings');

          var msg = _dereq_('./zlib/messages');

          var zstream = _dereq_('./zlib/zstream');
          /* Public constants ==========================================================*/

          /* ===========================================================================*/


          var Z_NO_FLUSH = 0;
          var Z_FINISH = 4;
          var Z_OK = 0;
          var Z_STREAM_END = 1;
          var Z_DEFAULT_COMPRESSION = -1;
          var Z_DEFAULT_STRATEGY = 0;
          var Z_DEFLATED = 8;
          /* ===========================================================================*/

          /**
           * class Deflate
           *
           * Generic JS-style wrapper for zlib calls. If you don't need
           * streaming behaviour - use more simple functions: [[deflate]],
           * [[deflateRaw]] and [[gzip]].
           **/

          /* internal
           * Deflate.chunks -> Array
           *
           * Chunks of output data, if [[Deflate#onData]] not overriden.
           **/

          /**
           * Deflate.result -> Uint8Array|Array
           *
           * Compressed result, generated by default [[Deflate#onData]]
           * and [[Deflate#onEnd]] handlers. Filled after you push last chunk
           * (call [[Deflate#push]] with `Z_FINISH` / `true` param).
           **/

          /**
           * Deflate.err -> Number
           *
           * Error code after deflate finished. 0 (Z_OK) on success.
           * You will not need it in real life, because deflate errors
           * are possible only on wrong options or bad `onData` / `onEnd`
           * custom handlers.
           **/

          /**
           * Deflate.msg -> String
           *
           * Error message, if [[Deflate.err]] != 0
           **/

          /**
           * new Deflate(options)
           * - options (Object): zlib deflate options.
           *
           * Creates new deflator instance with specified params. Throws exception
           * on bad params. Supported options:
           *
           * - `level`
           * - `windowBits`
           * - `memLevel`
           * - `strategy`
           *
           * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
           * for more information on these.
           *
           * Additional options, for internal needs:
           *
           * - `chunkSize` - size of generated data chunks (16K by default)
           * - `raw` (Boolean) - do raw deflate
           * - `gzip` (Boolean) - create gzip wrapper
           * - `to` (String) - if equal to 'string', then result will be "binary string"
           *    (each char code [0..255])
           * - `header` (Object) - custom header for gzip
           *   - `text` (Boolean) - true if compressed data believed to be text
           *   - `time` (Number) - modification time, unix timestamp
           *   - `os` (Number) - operation system code
           *   - `extra` (Array) - array of bytes with extra data (max 65536)
           *   - `name` (String) - file name (binary string)
           *   - `comment` (String) - comment (binary string)
           *   - `hcrc` (Boolean) - true if header crc should be added
           *
           * ##### Example:
           *
           * ```javascript
           * var pako = require('pako')
           *   , chunk1 = Uint8Array([1,2,3,4,5,6,7,8,9])
           *   , chunk2 = Uint8Array([10,11,12,13,14,15,16,17,18,19]);
           *
           * var deflate = new pako.Deflate({ level: 3});
           *
           * deflate.push(chunk1, false);
           * deflate.push(chunk2, true);  // true -> last chunk
           *
           * if (deflate.err) { throw new Error(deflate.err); }
           *
           * console.log(deflate.result);
           * ```
           **/

          var Deflate = function (options) {
            this.options = utils.assign({
              level: Z_DEFAULT_COMPRESSION,
              method: Z_DEFLATED,
              chunkSize: 16384,
              windowBits: 15,
              memLevel: 8,
              strategy: Z_DEFAULT_STRATEGY,
              to: ''
            }, options || {});
            var opt = this.options;

            if (opt.raw && opt.windowBits > 0) {
              opt.windowBits = -opt.windowBits;
            } else if (opt.gzip && opt.windowBits > 0 && opt.windowBits < 16) {
              opt.windowBits += 16;
            }

            this.err = 0; // error code, if happens (0 = Z_OK)

            this.msg = ''; // error message

            this.ended = false; // used to avoid multiple onEnd() calls

            this.chunks = []; // chunks of compressed data

            this.strm = new zstream();
            this.strm.avail_out = 0;
            var status = zlib_deflate.deflateInit2(this.strm, opt.level, opt.method, opt.windowBits, opt.memLevel, opt.strategy);

            if (status !== Z_OK) {
              throw new Error(msg[status]);
            }

            if (opt.header) {
              zlib_deflate.deflateSetHeader(this.strm, opt.header);
            }
          };
          /**
           * Deflate#push(data[, mode]) -> Boolean
           * - data (Uint8Array|Array|String): input data. Strings will be converted to
           *   utf8 byte sequence.
           * - mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
           *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` meansh Z_FINISH.
           *
           * Sends input data to deflate pipe, generating [[Deflate#onData]] calls with
           * new compressed chunks. Returns `true` on success. The last data block must have
           * mode Z_FINISH (or `true`). That flush internal pending buffers and call
           * [[Deflate#onEnd]].
           *
           * On fail call [[Deflate#onEnd]] with error code and return false.
           *
           * We strongly recommend to use `Uint8Array` on input for best speed (output
           * array format is detected automatically). Also, don't skip last param and always
           * use the same type in your code (boolean or number). That will improve JS speed.
           *
           * For regular `Array`-s make sure all elements are [0..255].
           *
           * ##### Example
           *
           * ```javascript
           * push(chunk, false); // push one of data chunks
           * ...
           * push(chunk, true);  // push last chunk
           * ```
           **/


          Deflate.prototype.push = function (data, mode) {
            var strm = this.strm;
            var chunkSize = this.options.chunkSize;

            var status, _mode;

            if (this.ended) {
              return false;
            }

            _mode = mode === ~~mode ? mode : mode === true ? Z_FINISH : Z_NO_FLUSH; // Convert data if needed

            if (typeof data === 'string') {
              // If we need to compress text, change encoding to utf8.
              strm.input = strings.string2buf(data);
            } else {
              strm.input = data;
            }

            strm.next_in = 0;
            strm.avail_in = strm.input.length;

            do {
              if (strm.avail_out === 0) {
                strm.output = new utils.Buf8(chunkSize);
                strm.next_out = 0;
                strm.avail_out = chunkSize;
              }

              status = zlib_deflate.deflate(strm, _mode);
              /* no bad return value */

              if (status !== Z_STREAM_END && status !== Z_OK) {
                this.onEnd(status);
                this.ended = true;
                return false;
              }

              if (strm.avail_out === 0 || strm.avail_in === 0 && _mode === Z_FINISH) {
                if (this.options.to === 'string') {
                  this.onData(strings.buf2binstring(utils.shrinkBuf(strm.output, strm.next_out)));
                } else {
                  this.onData(utils.shrinkBuf(strm.output, strm.next_out));
                }
              }
            } while ((strm.avail_in > 0 || strm.avail_out === 0) && status !== Z_STREAM_END); // Finalize on the last chunk.


            if (_mode === Z_FINISH) {
              status = zlib_deflate.deflateEnd(this.strm);
              this.onEnd(status);
              this.ended = true;
              return status === Z_OK;
            }

            return true;
          };
          /**
           * Deflate#onData(chunk) -> Void
           * - chunk (Uint8Array|Array|String): ouput data. Type of array depends
           *   on js engine support. When string output requested, each chunk
           *   will be string.
           *
           * By default, stores data blocks in `chunks[]` property and glue
           * those in `onEnd`. Override this handler, if you need another behaviour.
           **/


          Deflate.prototype.onData = function (chunk) {
            this.chunks.push(chunk);
          };
          /**
           * Deflate#onEnd(status) -> Void
           * - status (Number): deflate status. 0 (Z_OK) on success,
           *   other if not.
           *
           * Called once after you tell deflate that input stream complete
           * or error happenned. By default - join collected chunks,
           * free memory and fill `results` / `err` properties.
           **/


          Deflate.prototype.onEnd = function (status) {
            // On success - join
            if (status === Z_OK) {
              if (this.options.to === 'string') {
                this.result = this.chunks.join('');
              } else {
                this.result = utils.flattenChunks(this.chunks);
              }
            }

            this.chunks = [];
            this.err = status;
            this.msg = this.strm.msg;
          };
          /**
           * deflate(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to compress.
           * - options (Object): zlib deflate options.
           *
           * Compress `data` with deflate alrorythm and `options`.
           *
           * Supported options are:
           *
           * - level
           * - windowBits
           * - memLevel
           * - strategy
           *
           * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
           * for more information on these.
           *
           * Sugar (options):
           *
           * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
           *   negative windowBits implicitly.
           * - `to` (String) - if equal to 'string', then result will be "binary string"
           *    (each char code [0..255])
           *
           * ##### Example:
           *
           * ```javascript
           * var pako = require('pako')
           *   , data = Uint8Array([1,2,3,4,5,6,7,8,9]);
           *
           * console.log(pako.deflate(data));
           * ```
           **/


          function deflate(input, options) {
            var deflator = new Deflate(options);
            deflator.push(input, true); // That will never happens, if you don't cheat with options :)

            if (deflator.err) {
              throw deflator.msg;
            }

            return deflator.result;
          }
          /**
           * deflateRaw(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to compress.
           * - options (Object): zlib deflate options.
           *
           * The same as [[deflate]], but creates raw data, without wrapper
           * (header and adler32 crc).
           **/


          function deflateRaw(input, options) {
            options = options || {};
            options.raw = true;
            return deflate(input, options);
          }
          /**
           * gzip(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to compress.
           * - options (Object): zlib deflate options.
           *
           * The same as [[deflate]], but create gzip wrapper instead of
           * deflate one.
           **/


          function gzip(input, options) {
            options = options || {};
            options.gzip = true;
            return deflate(input, options);
          }

          exports.Deflate = Deflate;
          exports.deflate = deflate;
          exports.deflateRaw = deflateRaw;
          exports.gzip = gzip;
        }, {
          "./utils/common": 27,
          "./utils/strings": 28,
          "./zlib/deflate.js": 32,
          "./zlib/messages": 37,
          "./zlib/zstream": 39
        }],
        26: [function (_dereq_, module, exports) {

          var zlib_inflate = _dereq_('./zlib/inflate.js');

          var utils = _dereq_('./utils/common');

          var strings = _dereq_('./utils/strings');

          var c = _dereq_('./zlib/constants');

          var msg = _dereq_('./zlib/messages');

          var zstream = _dereq_('./zlib/zstream');

          var gzheader = _dereq_('./zlib/gzheader');
          /**
           * class Inflate
           *
           * Generic JS-style wrapper for zlib calls. If you don't need
           * streaming behaviour - use more simple functions: [[inflate]]
           * and [[inflateRaw]].
           **/

          /* internal
           * inflate.chunks -> Array
           *
           * Chunks of output data, if [[Inflate#onData]] not overriden.
           **/

          /**
           * Inflate.result -> Uint8Array|Array|String
           *
           * Uncompressed result, generated by default [[Inflate#onData]]
           * and [[Inflate#onEnd]] handlers. Filled after you push last chunk
           * (call [[Inflate#push]] with `Z_FINISH` / `true` param).
           **/

          /**
           * Inflate.err -> Number
           *
           * Error code after inflate finished. 0 (Z_OK) on success.
           * Should be checked if broken data possible.
           **/

          /**
           * Inflate.msg -> String
           *
           * Error message, if [[Inflate.err]] != 0
           **/

          /**
           * new Inflate(options)
           * - options (Object): zlib inflate options.
           *
           * Creates new inflator instance with specified params. Throws exception
           * on bad params. Supported options:
           *
           * - `windowBits`
           *
           * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
           * for more information on these.
           *
           * Additional options, for internal needs:
           *
           * - `chunkSize` - size of generated data chunks (16K by default)
           * - `raw` (Boolean) - do raw inflate
           * - `to` (String) - if equal to 'string', then result will be converted
           *   from utf8 to utf16 (javascript) string. When string output requested,
           *   chunk length can differ from `chunkSize`, depending on content.
           *
           * By default, when no options set, autodetect deflate/gzip data format via
           * wrapper header.
           *
           * ##### Example:
           *
           * ```javascript
           * var pako = require('pako')
           *   , chunk1 = Uint8Array([1,2,3,4,5,6,7,8,9])
           *   , chunk2 = Uint8Array([10,11,12,13,14,15,16,17,18,19]);
           *
           * var inflate = new pako.Inflate({ level: 3});
           *
           * inflate.push(chunk1, false);
           * inflate.push(chunk2, true);  // true -> last chunk
           *
           * if (inflate.err) { throw new Error(inflate.err); }
           *
           * console.log(inflate.result);
           * ```
           **/


          var Inflate = function (options) {
            this.options = utils.assign({
              chunkSize: 16384,
              windowBits: 0,
              to: ''
            }, options || {});
            var opt = this.options; // Force window size for `raw` data, if not set directly,
            // because we have no header for autodetect.

            if (opt.raw && opt.windowBits >= 0 && opt.windowBits < 16) {
              opt.windowBits = -opt.windowBits;

              if (opt.windowBits === 0) {
                opt.windowBits = -15;
              }
            } // If `windowBits` not defined (and mode not raw) - set autodetect flag for gzip/deflate


            if (opt.windowBits >= 0 && opt.windowBits < 16 && !(options && options.windowBits)) {
              opt.windowBits += 32;
            } // Gzip header has no info about windows size, we can do autodetect only
            // for deflate. So, if window size not set, force it to max when gzip possible


            if (opt.windowBits > 15 && opt.windowBits < 48) {
              // bit 3 (16) -> gzipped data
              // bit 4 (32) -> autodetect gzip/deflate
              if ((opt.windowBits & 15) === 0) {
                opt.windowBits |= 15;
              }
            }

            this.err = 0; // error code, if happens (0 = Z_OK)

            this.msg = ''; // error message

            this.ended = false; // used to avoid multiple onEnd() calls

            this.chunks = []; // chunks of compressed data

            this.strm = new zstream();
            this.strm.avail_out = 0;
            var status = zlib_inflate.inflateInit2(this.strm, opt.windowBits);

            if (status !== c.Z_OK) {
              throw new Error(msg[status]);
            }

            this.header = new gzheader();
            zlib_inflate.inflateGetHeader(this.strm, this.header);
          };
          /**
           * Inflate#push(data[, mode]) -> Boolean
           * - data (Uint8Array|Array|String): input data
           * - mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE modes.
           *   See constants. Skipped or `false` means Z_NO_FLUSH, `true` meansh Z_FINISH.
           *
           * Sends input data to inflate pipe, generating [[Inflate#onData]] calls with
           * new output chunks. Returns `true` on success. The last data block must have
           * mode Z_FINISH (or `true`). That flush internal pending buffers and call
           * [[Inflate#onEnd]].
           *
           * On fail call [[Inflate#onEnd]] with error code and return false.
           *
           * We strongly recommend to use `Uint8Array` on input for best speed (output
           * format is detected automatically). Also, don't skip last param and always
           * use the same type in your code (boolean or number). That will improve JS speed.
           *
           * For regular `Array`-s make sure all elements are [0..255].
           *
           * ##### Example
           *
           * ```javascript
           * push(chunk, false); // push one of data chunks
           * ...
           * push(chunk, true);  // push last chunk
           * ```
           **/


          Inflate.prototype.push = function (data, mode) {
            var strm = this.strm;
            var chunkSize = this.options.chunkSize;

            var status, _mode;

            var next_out_utf8, tail, utf8str;

            if (this.ended) {
              return false;
            }

            _mode = mode === ~~mode ? mode : mode === true ? c.Z_FINISH : c.Z_NO_FLUSH; // Convert data if needed

            if (typeof data === 'string') {
              // Only binary strings can be decompressed on practice
              strm.input = strings.binstring2buf(data);
            } else {
              strm.input = data;
            }

            strm.next_in = 0;
            strm.avail_in = strm.input.length;

            do {
              if (strm.avail_out === 0) {
                strm.output = new utils.Buf8(chunkSize);
                strm.next_out = 0;
                strm.avail_out = chunkSize;
              }

              status = zlib_inflate.inflate(strm, c.Z_NO_FLUSH);
              /* no bad return value */

              if (status !== c.Z_STREAM_END && status !== c.Z_OK) {
                this.onEnd(status);
                this.ended = true;
                return false;
              }

              if (strm.next_out) {
                if (strm.avail_out === 0 || status === c.Z_STREAM_END || strm.avail_in === 0 && _mode === c.Z_FINISH) {
                  if (this.options.to === 'string') {
                    next_out_utf8 = strings.utf8border(strm.output, strm.next_out);
                    tail = strm.next_out - next_out_utf8;
                    utf8str = strings.buf2string(strm.output, next_out_utf8); // move tail

                    strm.next_out = tail;
                    strm.avail_out = chunkSize - tail;

                    if (tail) {
                      utils.arraySet(strm.output, strm.output, next_out_utf8, tail, 0);
                    }

                    this.onData(utf8str);
                  } else {
                    this.onData(utils.shrinkBuf(strm.output, strm.next_out));
                  }
                }
              }
            } while (strm.avail_in > 0 && status !== c.Z_STREAM_END);

            if (status === c.Z_STREAM_END) {
              _mode = c.Z_FINISH;
            } // Finalize on the last chunk.


            if (_mode === c.Z_FINISH) {
              status = zlib_inflate.inflateEnd(this.strm);
              this.onEnd(status);
              this.ended = true;
              return status === c.Z_OK;
            }

            return true;
          };
          /**
           * Inflate#onData(chunk) -> Void
           * - chunk (Uint8Array|Array|String): ouput data. Type of array depends
           *   on js engine support. When string output requested, each chunk
           *   will be string.
           *
           * By default, stores data blocks in `chunks[]` property and glue
           * those in `onEnd`. Override this handler, if you need another behaviour.
           **/


          Inflate.prototype.onData = function (chunk) {
            this.chunks.push(chunk);
          };
          /**
           * Inflate#onEnd(status) -> Void
           * - status (Number): inflate status. 0 (Z_OK) on success,
           *   other if not.
           *
           * Called once after you tell inflate that input stream complete
           * or error happenned. By default - join collected chunks,
           * free memory and fill `results` / `err` properties.
           **/


          Inflate.prototype.onEnd = function (status) {
            // On success - join
            if (status === c.Z_OK) {
              if (this.options.to === 'string') {
                // Glue & convert here, until we teach pako to send
                // utf8 alligned strings to onData
                this.result = this.chunks.join('');
              } else {
                this.result = utils.flattenChunks(this.chunks);
              }
            }

            this.chunks = [];
            this.err = status;
            this.msg = this.strm.msg;
          };
          /**
           * inflate(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to decompress.
           * - options (Object): zlib inflate options.
           *
           * Decompress `data` with inflate/ungzip and `options`. Autodetect
           * format via wrapper header by default. That's why we don't provide
           * separate `ungzip` method.
           *
           * Supported options are:
           *
           * - windowBits
           *
           * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
           * for more information.
           *
           * Sugar (options):
           *
           * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
           *   negative windowBits implicitly.
           * - `to` (String) - if equal to 'string', then result will be converted
           *   from utf8 to utf16 (javascript) string. When string output requested,
           *   chunk length can differ from `chunkSize`, depending on content.
           *
           *
           * ##### Example:
           *
           * ```javascript
           * var pako = require('pako')
           *   , input = pako.deflate([1,2,3,4,5,6,7,8,9])
           *   , output;
           *
           * try {
           *   output = pako.inflate(input);
           * } catch (err)
           *   console.log(err);
           * }
           * ```
           **/


          function inflate(input, options) {
            var inflator = new Inflate(options);
            inflator.push(input, true); // That will never happens, if you don't cheat with options :)

            if (inflator.err) {
              throw inflator.msg;
            }

            return inflator.result;
          }
          /**
           * inflateRaw(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to decompress.
           * - options (Object): zlib inflate options.
           *
           * The same as [[inflate]], but creates raw data, without wrapper
           * (header and adler32 crc).
           **/


          function inflateRaw(input, options) {
            options = options || {};
            options.raw = true;
            return inflate(input, options);
          }
          /**
           * ungzip(data[, options]) -> Uint8Array|Array|String
           * - data (Uint8Array|Array|String): input data to decompress.
           * - options (Object): zlib inflate options.
           *
           * Just shortcut to [[inflate]], because it autodetects format
           * by header.content. Done for convenience.
           **/


          exports.Inflate = Inflate;
          exports.inflate = inflate;
          exports.inflateRaw = inflateRaw;
          exports.ungzip = inflate;
        }, {
          "./utils/common": 27,
          "./utils/strings": 28,
          "./zlib/constants": 30,
          "./zlib/gzheader": 33,
          "./zlib/inflate.js": 35,
          "./zlib/messages": 37,
          "./zlib/zstream": 39
        }],
        27: [function (_dereq_, module, exports) {

          var TYPED_OK = typeof Uint8Array !== 'undefined' && typeof Uint16Array !== 'undefined' && typeof Int32Array !== 'undefined';

          exports.assign = function (obj
          /*from1, from2, from3, ...*/
          ) {
            var sources = Array.prototype.slice.call(arguments, 1);

            while (sources.length) {
              var source = sources.shift();

              if (!source) {
                continue;
              }

              if (typeof source !== 'object') {
                throw new TypeError(source + 'must be non-object');
              }

              for (var p in source) {
                if (source.hasOwnProperty(p)) {
                  obj[p] = source[p];
                }
              }
            }

            return obj;
          }; // reduce buffer size, avoiding mem copy


          exports.shrinkBuf = function (buf, size) {
            if (buf.length === size) {
              return buf;
            }

            if (buf.subarray) {
              return buf.subarray(0, size);
            }

            buf.length = size;
            return buf;
          };

          var fnTyped = {
            arraySet: function (dest, src, src_offs, len, dest_offs) {
              if (src.subarray && dest.subarray) {
                dest.set(src.subarray(src_offs, src_offs + len), dest_offs);
                return;
              } // Fallback to ordinary array


              for (var i = 0; i < len; i++) {
                dest[dest_offs + i] = src[src_offs + i];
              }
            },
            // Join array of chunks to single array.
            flattenChunks: function (chunks) {
              var i, l, len, pos, chunk, result; // calculate data length

              len = 0;

              for (i = 0, l = chunks.length; i < l; i++) {
                len += chunks[i].length;
              } // join chunks


              result = new Uint8Array(len);
              pos = 0;

              for (i = 0, l = chunks.length; i < l; i++) {
                chunk = chunks[i];
                result.set(chunk, pos);
                pos += chunk.length;
              }

              return result;
            }
          };
          var fnUntyped = {
            arraySet: function (dest, src, src_offs, len, dest_offs) {
              for (var i = 0; i < len; i++) {
                dest[dest_offs + i] = src[src_offs + i];
              }
            },
            // Join array of chunks to single array.
            flattenChunks: function (chunks) {
              return [].concat.apply([], chunks);
            }
          }; // Enable/Disable typed arrays use, for testing
          //

          exports.setTyped = function (on) {
            if (on) {
              exports.Buf8 = Uint8Array;
              exports.Buf16 = Uint16Array;
              exports.Buf32 = Int32Array;
              exports.assign(exports, fnTyped);
            } else {
              exports.Buf8 = Array;
              exports.Buf16 = Array;
              exports.Buf32 = Array;
              exports.assign(exports, fnUntyped);
            }
          };

          exports.setTyped(TYPED_OK);
        }, {}],
        28: [function (_dereq_, module, exports) {

          var utils = _dereq_('./common'); // Quick check if we can use fast array to bin string conversion
          //
          // - apply(Array) can fail on Android 2.2
          // - apply(Uint8Array) can fail on iOS 5.1 Safary
          //


          var STR_APPLY_OK = true;
          var STR_APPLY_UIA_OK = true;

          try {
            String.fromCharCode.apply(null, [0]);
          } catch (__) {
            STR_APPLY_OK = false;
          }

          try {
            String.fromCharCode.apply(null, new Uint8Array(1));
          } catch (__) {
            STR_APPLY_UIA_OK = false;
          } // Table with utf8 lengths (calculated by first byte of sequence)
          // Note, that 5 & 6-byte values and some 4-byte values can not be represented in JS,
          // because max possible codepoint is 0x10ffff


          var _utf8len = new utils.Buf8(256);

          for (var i = 0; i < 256; i++) {
            _utf8len[i] = i >= 252 ? 6 : i >= 248 ? 5 : i >= 240 ? 4 : i >= 224 ? 3 : i >= 192 ? 2 : 1;
          }

          _utf8len[254] = _utf8len[254] = 1; // Invalid sequence start
          // convert string to array (typed, when possible)

          exports.string2buf = function (str) {
            var buf,
                c,
                c2,
                m_pos,
                i,
                str_len = str.length,
                buf_len = 0; // count binary size

            for (m_pos = 0; m_pos < str_len; m_pos++) {
              c = str.charCodeAt(m_pos);

              if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
                c2 = str.charCodeAt(m_pos + 1);

                if ((c2 & 0xfc00) === 0xdc00) {
                  c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
                  m_pos++;
                }
              }

              buf_len += c < 0x80 ? 1 : c < 0x800 ? 2 : c < 0x10000 ? 3 : 4;
            } // allocate buffer


            buf = new utils.Buf8(buf_len); // convert

            for (i = 0, m_pos = 0; i < buf_len; m_pos++) {
              c = str.charCodeAt(m_pos);

              if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
                c2 = str.charCodeAt(m_pos + 1);

                if ((c2 & 0xfc00) === 0xdc00) {
                  c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
                  m_pos++;
                }
              }

              if (c < 0x80) {
                /* one byte */
                buf[i++] = c;
              } else if (c < 0x800) {
                /* two bytes */
                buf[i++] = 0xC0 | c >>> 6;
                buf[i++] = 0x80 | c & 0x3f;
              } else if (c < 0x10000) {
                /* three bytes */
                buf[i++] = 0xE0 | c >>> 12;
                buf[i++] = 0x80 | c >>> 6 & 0x3f;
                buf[i++] = 0x80 | c & 0x3f;
              } else {
                /* four bytes */
                buf[i++] = 0xf0 | c >>> 18;
                buf[i++] = 0x80 | c >>> 12 & 0x3f;
                buf[i++] = 0x80 | c >>> 6 & 0x3f;
                buf[i++] = 0x80 | c & 0x3f;
              }
            }

            return buf;
          }; // Helper (used in 2 places)


          function buf2binstring(buf, len) {
            // use fallback for big arrays to avoid stack overflow
            if (len < 65537) {
              if (buf.subarray && STR_APPLY_UIA_OK || !buf.subarray && STR_APPLY_OK) {
                return String.fromCharCode.apply(null, utils.shrinkBuf(buf, len));
              }
            }

            var result = '';

            for (var i = 0; i < len; i++) {
              result += String.fromCharCode(buf[i]);
            }

            return result;
          } // Convert byte array to binary string


          exports.buf2binstring = function (buf) {
            return buf2binstring(buf, buf.length);
          }; // Convert binary string (typed, when possible)


          exports.binstring2buf = function (str) {
            var buf = new utils.Buf8(str.length);

            for (var i = 0, len = buf.length; i < len; i++) {
              buf[i] = str.charCodeAt(i);
            }

            return buf;
          }; // convert array to string


          exports.buf2string = function (buf, max) {
            var i, out, c, c_len;
            var len = max || buf.length; // Reserve max possible length (2 words per char)
            // NB: by unknown reasons, Array is significantly faster for
            //     String.fromCharCode.apply than Uint16Array.

            var utf16buf = new Array(len * 2);

            for (out = 0, i = 0; i < len;) {
              c = buf[i++]; // quick process ascii

              if (c < 0x80) {
                utf16buf[out++] = c;
                continue;
              }

              c_len = _utf8len[c]; // skip 5 & 6 byte codes

              if (c_len > 4) {
                utf16buf[out++] = 0xfffd;
                i += c_len - 1;
                continue;
              } // apply mask on first byte


              c &= c_len === 2 ? 0x1f : c_len === 3 ? 0x0f : 0x07; // join the rest

              while (c_len > 1 && i < len) {
                c = c << 6 | buf[i++] & 0x3f;
                c_len--;
              } // terminated by end of string?


              if (c_len > 1) {
                utf16buf[out++] = 0xfffd;
                continue;
              }

              if (c < 0x10000) {
                utf16buf[out++] = c;
              } else {
                c -= 0x10000;
                utf16buf[out++] = 0xd800 | c >> 10 & 0x3ff;
                utf16buf[out++] = 0xdc00 | c & 0x3ff;
              }
            }

            return buf2binstring(utf16buf, out);
          }; // Calculate max possible position in utf8 buffer,
          // that will not break sequence. If that's not possible
          // - (very small limits) return max size as is.
          //
          // buf[] - utf8 bytes array
          // max   - length limit (mandatory);


          exports.utf8border = function (buf, max) {
            var pos;
            max = max || buf.length;

            if (max > buf.length) {
              max = buf.length;
            } // go back from last position, until start of sequence found


            pos = max - 1;

            while (pos >= 0 && (buf[pos] & 0xC0) === 0x80) {
              pos--;
            } // Fuckup - very small and broken sequence,
            // return max, because we should return something anyway.


            if (pos < 0) {
              return max;
            } // If we came to start of buffer - that means vuffer is too small,
            // return max too.


            if (pos === 0) {
              return max;
            }

            return pos + _utf8len[buf[pos]] > max ? pos : max;
          };
        }, {
          "./common": 27
        }],
        29: [function (_dereq_, module, exports) {
          // It doesn't worth to make additional optimizationa as in original.
          // Small size is preferable.

          function adler32(adler, buf, len, pos) {
            var s1 = adler & 0xffff | 0,
                s2 = adler >>> 16 & 0xffff | 0,
                n = 0;

            while (len !== 0) {
              // Set limit ~ twice less than 5552, to keep
              // s2 in 31-bits, because we force signed ints.
              // in other case %= will fail.
              n = len > 2000 ? 2000 : len;
              len -= n;

              do {
                s1 = s1 + buf[pos++] | 0;
                s2 = s2 + s1 | 0;
              } while (--n);

              s1 %= 65521;
              s2 %= 65521;
            }

            return s1 | s2 << 16 | 0;
          }

          module.exports = adler32;
        }, {}],
        30: [function (_dereq_, module, exports) {
          module.exports = {
            /* Allowed flush values; see deflate() and inflate() below for details */
            Z_NO_FLUSH: 0,
            Z_PARTIAL_FLUSH: 1,
            Z_SYNC_FLUSH: 2,
            Z_FULL_FLUSH: 3,
            Z_FINISH: 4,
            Z_BLOCK: 5,
            Z_TREES: 6,

            /* Return codes for the compression/decompression functions. Negative values
            * are errors, positive values are used for special but normal events.
            */
            Z_OK: 0,
            Z_STREAM_END: 1,
            Z_NEED_DICT: 2,
            Z_ERRNO: -1,
            Z_STREAM_ERROR: -2,
            Z_DATA_ERROR: -3,
            //Z_MEM_ERROR:     -4,
            Z_BUF_ERROR: -5,
            //Z_VERSION_ERROR: -6,

            /* compression levels */
            Z_NO_COMPRESSION: 0,
            Z_BEST_SPEED: 1,
            Z_BEST_COMPRESSION: 9,
            Z_DEFAULT_COMPRESSION: -1,
            Z_FILTERED: 1,
            Z_HUFFMAN_ONLY: 2,
            Z_RLE: 3,
            Z_FIXED: 4,
            Z_DEFAULT_STRATEGY: 0,

            /* Possible values of the data_type field (though see inflate()) */
            Z_BINARY: 0,
            Z_TEXT: 1,
            //Z_ASCII:                1, // = Z_TEXT (deprecated)
            Z_UNKNOWN: 2,

            /* The deflate compression method */
            Z_DEFLATED: 8 //Z_NULL:                 null // Use -1 or null inline, depending on var type

          };
        }, {}],
        31: [function (_dereq_, module, exports) {
          // So write code to minimize size - no pregenerated tables
          // and array tools dependencies.
          // Use ordinary array, since untyped makes no boost here

          function makeTable() {
            var c,
                table = [];

            for (var n = 0; n < 256; n++) {
              c = n;

              for (var k = 0; k < 8; k++) {
                c = c & 1 ? 0xEDB88320 ^ c >>> 1 : c >>> 1;
              }

              table[n] = c;
            }

            return table;
          } // Create table on load. Just 255 signed longs. Not a problem.


          var crcTable = makeTable();

          function crc32(crc, buf, len, pos) {
            var t = crcTable,
                end = pos + len;
            crc = crc ^ -1;

            for (var i = pos; i < end; i++) {
              crc = crc >>> 8 ^ t[(crc ^ buf[i]) & 0xFF];
            }

            return crc ^ -1; // >>> 0;
          }

          module.exports = crc32;
        }, {}],
        32: [function (_dereq_, module, exports) {

          var utils = _dereq_('../utils/common');

          var trees = _dereq_('./trees');

          var adler32 = _dereq_('./adler32');

          var crc32 = _dereq_('./crc32');

          var msg = _dereq_('./messages');
          /* Public constants ==========================================================*/

          /* ===========================================================================*/

          /* Allowed flush values; see deflate() and inflate() below for details */


          var Z_NO_FLUSH = 0;
          var Z_PARTIAL_FLUSH = 1; //var Z_SYNC_FLUSH    = 2;

          var Z_FULL_FLUSH = 3;
          var Z_FINISH = 4;
          var Z_BLOCK = 5; //var Z_TREES         = 6;

          /* Return codes for the compression/decompression functions. Negative values
           * are errors, positive values are used for special but normal events.
           */

          var Z_OK = 0;
          var Z_STREAM_END = 1; //var Z_NEED_DICT     = 2;
          //var Z_ERRNO         = -1;

          var Z_STREAM_ERROR = -2;
          var Z_DATA_ERROR = -3; //var Z_MEM_ERROR     = -4;

          var Z_BUF_ERROR = -5; //var Z_VERSION_ERROR = -6;

          /* compression levels */
          //var Z_NO_COMPRESSION      = 0;
          //var Z_BEST_SPEED          = 1;
          //var Z_BEST_COMPRESSION    = 9;

          var Z_DEFAULT_COMPRESSION = -1;
          var Z_FILTERED = 1;
          var Z_HUFFMAN_ONLY = 2;
          var Z_RLE = 3;
          var Z_FIXED = 4;
          var Z_DEFAULT_STRATEGY = 0;
          /* Possible values of the data_type field (though see inflate()) */
          //var Z_BINARY              = 0;
          //var Z_TEXT                = 1;
          //var Z_ASCII               = 1; // = Z_TEXT

          var Z_UNKNOWN = 2;
          /* The deflate compression method */

          var Z_DEFLATED = 8;
          /*============================================================================*/

          var MAX_MEM_LEVEL = 9;
          /* Maximum value for memLevel in deflateInit2 */

          var MAX_WBITS = 15;
          /* 32K LZ77 window */

          var DEF_MEM_LEVEL = 8;
          var LENGTH_CODES = 29;
          /* number of length codes, not counting the special END_BLOCK code */

          var LITERALS = 256;
          /* number of literal bytes 0..255 */

          var L_CODES = LITERALS + 1 + LENGTH_CODES;
          /* number of Literal or Length codes, including the END_BLOCK code */

          var D_CODES = 30;
          /* number of distance codes */

          var BL_CODES = 19;
          /* number of codes used to transfer the bit lengths */

          var HEAP_SIZE = 2 * L_CODES + 1;
          /* maximum heap size */

          var MAX_BITS = 15;
          /* All codes must not exceed MAX_BITS bits */

          var MIN_MATCH = 3;
          var MAX_MATCH = 258;
          var MIN_LOOKAHEAD = MAX_MATCH + MIN_MATCH + 1;
          var PRESET_DICT = 0x20;
          var INIT_STATE = 42;
          var EXTRA_STATE = 69;
          var NAME_STATE = 73;
          var COMMENT_STATE = 91;
          var HCRC_STATE = 103;
          var BUSY_STATE = 113;
          var FINISH_STATE = 666;
          var BS_NEED_MORE = 1;
          /* block not completed, need more input or more output */

          var BS_BLOCK_DONE = 2;
          /* block flush performed */

          var BS_FINISH_STARTED = 3;
          /* finish started, need only more output at next deflate */

          var BS_FINISH_DONE = 4;
          /* finish done, accept no more input or output */

          var OS_CODE = 0x03; // Unix :) . Don't detect, use this default.

          function err(strm, errorCode) {
            strm.msg = msg[errorCode];
            return errorCode;
          }

          function rank(f) {
            return (f << 1) - (f > 4 ? 9 : 0);
          }

          function zero(buf) {
            var len = buf.length;

            while (--len >= 0) {
              buf[len] = 0;
            }
          }
          /* =========================================================================
           * Flush as much pending output as possible. All deflate() output goes
           * through this function so some applications may wish to modify it
           * to avoid allocating a large strm->output buffer and copying into it.
           * (See also read_buf()).
           */


          function flush_pending(strm) {
            var s = strm.state; //_tr_flush_bits(s);

            var len = s.pending;

            if (len > strm.avail_out) {
              len = strm.avail_out;
            }

            if (len === 0) {
              return;
            }

            utils.arraySet(strm.output, s.pending_buf, s.pending_out, len, strm.next_out);
            strm.next_out += len;
            s.pending_out += len;
            strm.total_out += len;
            strm.avail_out -= len;
            s.pending -= len;

            if (s.pending === 0) {
              s.pending_out = 0;
            }
          }

          function flush_block_only(s, last) {
            trees._tr_flush_block(s, s.block_start >= 0 ? s.block_start : -1, s.strstart - s.block_start, last);

            s.block_start = s.strstart;
            flush_pending(s.strm);
          }

          function put_byte(s, b) {
            s.pending_buf[s.pending++] = b;
          }
          /* =========================================================================
           * Put a short in the pending buffer. The 16-bit value is put in MSB order.
           * IN assertion: the stream state is correct and there is enough room in
           * pending_buf.
           */


          function putShortMSB(s, b) {
            //  put_byte(s, (Byte)(b >> 8));
            //  put_byte(s, (Byte)(b & 0xff));
            s.pending_buf[s.pending++] = b >>> 8 & 0xff;
            s.pending_buf[s.pending++] = b & 0xff;
          }
          /* ===========================================================================
           * Read a new buffer from the current input stream, update the adler32
           * and total number of bytes read.  All deflate() input goes through
           * this function so some applications may wish to modify it to avoid
           * allocating a large strm->input buffer and copying from it.
           * (See also flush_pending()).
           */


          function read_buf(strm, buf, start, size) {
            var len = strm.avail_in;

            if (len > size) {
              len = size;
            }

            if (len === 0) {
              return 0;
            }

            strm.avail_in -= len;
            utils.arraySet(buf, strm.input, strm.next_in, len, start);

            if (strm.state.wrap === 1) {
              strm.adler = adler32(strm.adler, buf, len, start);
            } else if (strm.state.wrap === 2) {
              strm.adler = crc32(strm.adler, buf, len, start);
            }

            strm.next_in += len;
            strm.total_in += len;
            return len;
          }
          /* ===========================================================================
           * Set match_start to the longest match starting at the given string and
           * return its length. Matches shorter or equal to prev_length are discarded,
           * in which case the result is equal to prev_length and match_start is
           * garbage.
           * IN assertions: cur_match is the head of the hash chain for the current
           *   string (strstart) and its distance is <= MAX_DIST, and prev_length >= 1
           * OUT assertion: the match length is not greater than s->lookahead.
           */


          function longest_match(s, cur_match) {
            var chain_length = s.max_chain_length;
            /* max hash chain length */

            var scan = s.strstart;
            /* current string */

            var match;
            /* matched string */

            var len;
            /* length of current match */

            var best_len = s.prev_length;
            /* best match length so far */

            var nice_match = s.nice_match;
            /* stop if match long enough */

            var limit = s.strstart > s.w_size - MIN_LOOKAHEAD ? s.strstart - (s.w_size - MIN_LOOKAHEAD) : 0
            /*NIL*/
            ;
            var _win = s.window; // shortcut

            var wmask = s.w_mask;
            var prev = s.prev;
            /* Stop when cur_match becomes <= limit. To simplify the code,
             * we prevent matches with the string of window index 0.
             */

            var strend = s.strstart + MAX_MATCH;
            var scan_end1 = _win[scan + best_len - 1];
            var scan_end = _win[scan + best_len];
            /* The code is optimized for HASH_BITS >= 8 and MAX_MATCH-2 multiple of 16.
             * It is easy to get rid of this optimization if necessary.
             */
            // Assert(s->hash_bits >= 8 && MAX_MATCH == 258, "Code too clever");

            /* Do not waste too much time if we already have a good match: */

            if (s.prev_length >= s.good_match) {
              chain_length >>= 2;
            }
            /* Do not look for matches beyond the end of the input. This is necessary
             * to make deflate deterministic.
             */


            if (nice_match > s.lookahead) {
              nice_match = s.lookahead;
            } // Assert((ulg)s->strstart <= s->window_size-MIN_LOOKAHEAD, "need lookahead");


            do {
              // Assert(cur_match < s->strstart, "no future");
              match = cur_match;
              /* Skip to next match if the match length cannot increase
               * or if the match length is less than 2.  Note that the checks below
               * for insufficient lookahead only occur occasionally for performance
               * reasons.  Therefore uninitialized memory will be accessed, and
               * conditional jumps will be made that depend on those values.
               * However the length of the match is limited to the lookahead, so
               * the output of deflate is not affected by the uninitialized values.
               */

              if (_win[match + best_len] !== scan_end || _win[match + best_len - 1] !== scan_end1 || _win[match] !== _win[scan] || _win[++match] !== _win[scan + 1]) {
                continue;
              }
              /* The check at best_len-1 can be removed because it will be made
               * again later. (This heuristic is not always a win.)
               * It is not necessary to compare scan[2] and match[2] since they
               * are always equal when the other bytes match, given that
               * the hash keys are equal and that HASH_BITS >= 8.
               */


              scan += 2;
              match++; // Assert(*scan == *match, "match[2]?");

              /* We check for insufficient lookahead only every 8th comparison;
               * the 256th check will be made at strstart+258.
               */

              do {
                /*jshint noempty:false*/
              } while (_win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && _win[++scan] === _win[++match] && scan < strend); // Assert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");


              len = MAX_MATCH - (strend - scan);
              scan = strend - MAX_MATCH;

              if (len > best_len) {
                s.match_start = cur_match;
                best_len = len;

                if (len >= nice_match) {
                  break;
                }

                scan_end1 = _win[scan + best_len - 1];
                scan_end = _win[scan + best_len];
              }
            } while ((cur_match = prev[cur_match & wmask]) > limit && --chain_length !== 0);

            if (best_len <= s.lookahead) {
              return best_len;
            }

            return s.lookahead;
          }
          /* ===========================================================================
           * Fill the window when the lookahead becomes insufficient.
           * Updates strstart and lookahead.
           *
           * IN assertion: lookahead < MIN_LOOKAHEAD
           * OUT assertions: strstart <= window_size-MIN_LOOKAHEAD
           *    At least one byte has been read, or avail_in == 0; reads are
           *    performed for at least two bytes (required for the zip translate_eol
           *    option -- not supported here).
           */


          function fill_window(s) {
            var _w_size = s.w_size;
            var p, n, m, more, str; //Assert(s->lookahead < MIN_LOOKAHEAD, "already enough lookahead");

            do {
              more = s.window_size - s.lookahead - s.strstart; // JS ints have 32 bit, block below not needed

              /* Deal with !@#$% 64K limit: */
              //if (sizeof(int) <= 2) {
              //    if (more == 0 && s->strstart == 0 && s->lookahead == 0) {
              //        more = wsize;
              //
              //  } else if (more == (unsigned)(-1)) {
              //        /* Very unlikely, but possible on 16 bit machine if
              //         * strstart == 0 && lookahead == 1 (input done a byte at time)
              //         */
              //        more--;
              //    }
              //}

              /* If the window is almost full and there is insufficient lookahead,
               * move the upper half to the lower one to make room in the upper half.
               */

              if (s.strstart >= _w_size + (_w_size - MIN_LOOKAHEAD)) {
                utils.arraySet(s.window, s.window, _w_size, _w_size, 0);
                s.match_start -= _w_size;
                s.strstart -= _w_size;
                /* we now have strstart >= MAX_DIST */

                s.block_start -= _w_size;
                /* Slide the hash table (could be avoided with 32 bit values
                 at the expense of memory usage). We slide even when level == 0
                 to keep the hash table consistent if we switch back to level > 0
                 later. (Using level 0 permanently is not an optimal usage of
                 zlib, so we don't care about this pathological case.)
                 */

                n = s.hash_size;
                p = n;

                do {
                  m = s.head[--p];
                  s.head[p] = m >= _w_size ? m - _w_size : 0;
                } while (--n);

                n = _w_size;
                p = n;

                do {
                  m = s.prev[--p];
                  s.prev[p] = m >= _w_size ? m - _w_size : 0;
                  /* If n is not on any hash chain, prev[n] is garbage but
                   * its value will never be used.
                   */
                } while (--n);

                more += _w_size;
              }

              if (s.strm.avail_in === 0) {
                break;
              }
              /* If there was no sliding:
               *    strstart <= WSIZE+MAX_DIST-1 && lookahead <= MIN_LOOKAHEAD - 1 &&
               *    more == window_size - lookahead - strstart
               * => more >= window_size - (MIN_LOOKAHEAD-1 + WSIZE + MAX_DIST-1)
               * => more >= window_size - 2*WSIZE + 2
               * In the BIG_MEM or MMAP case (not yet supported),
               *   window_size == input_size + MIN_LOOKAHEAD  &&
               *   strstart + s->lookahead <= input_size => more >= MIN_LOOKAHEAD.
               * Otherwise, window_size == 2*WSIZE so more >= 2.
               * If there was sliding, more >= WSIZE. So in all cases, more >= 2.
               */
              //Assert(more >= 2, "more < 2");


              n = read_buf(s.strm, s.window, s.strstart + s.lookahead, more);
              s.lookahead += n;
              /* Initialize the hash value now that we have some input: */

              if (s.lookahead + s.insert >= MIN_MATCH) {
                str = s.strstart - s.insert;
                s.ins_h = s.window[str];
                /* UPDATE_HASH(s, s->ins_h, s->window[str + 1]); */

                s.ins_h = (s.ins_h << s.hash_shift ^ s.window[str + 1]) & s.hash_mask; //#if MIN_MATCH != 3
                //        Call update_hash() MIN_MATCH-3 more times
                //#endif

                while (s.insert) {
                  /* UPDATE_HASH(s, s->ins_h, s->window[str + MIN_MATCH-1]); */
                  s.ins_h = (s.ins_h << s.hash_shift ^ s.window[str + MIN_MATCH - 1]) & s.hash_mask;
                  s.prev[str & s.w_mask] = s.head[s.ins_h];
                  s.head[s.ins_h] = str;
                  str++;
                  s.insert--;

                  if (s.lookahead + s.insert < MIN_MATCH) {
                    break;
                  }
                }
              }
              /* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,
               * but this is not important since only literal bytes will be emitted.
               */

            } while (s.lookahead < MIN_LOOKAHEAD && s.strm.avail_in !== 0);
            /* If the WIN_INIT bytes after the end of the current data have never been
             * written, then zero those bytes in order to avoid memory check reports of
             * the use of uninitialized (or uninitialised as Julian writes) bytes by
             * the longest match routines.  Update the high water mark for the next
             * time through here.  WIN_INIT is set to MAX_MATCH since the longest match
             * routines allow scanning to strstart + MAX_MATCH, ignoring lookahead.
             */
            //  if (s.high_water < s.window_size) {
            //    var curr = s.strstart + s.lookahead;
            //    var init = 0;
            //
            //    if (s.high_water < curr) {
            //      /* Previous high water mark below current data -- zero WIN_INIT
            //       * bytes or up to end of window, whichever is less.
            //       */
            //      init = s.window_size - curr;
            //      if (init > WIN_INIT)
            //        init = WIN_INIT;
            //      zmemzero(s->window + curr, (unsigned)init);
            //      s->high_water = curr + init;
            //    }
            //    else if (s->high_water < (ulg)curr + WIN_INIT) {
            //      /* High water mark at or above current data, but below current data
            //       * plus WIN_INIT -- zero out to current data plus WIN_INIT, or up
            //       * to end of window, whichever is less.
            //       */
            //      init = (ulg)curr + WIN_INIT - s->high_water;
            //      if (init > s->window_size - s->high_water)
            //        init = s->window_size - s->high_water;
            //      zmemzero(s->window + s->high_water, (unsigned)init);
            //      s->high_water += init;
            //    }
            //  }
            //
            //  Assert((ulg)s->strstart <= s->window_size - MIN_LOOKAHEAD,
            //    "not enough room for search");

          }
          /* ===========================================================================
           * Copy without compression as much as possible from the input stream, return
           * the current block state.
           * This function does not insert new strings in the dictionary since
           * uncompressible data is probably not useful. This function is used
           * only for the level=0 compression option.
           * NOTE: this function should be optimized to avoid extra copying from
           * window to pending_buf.
           */


          function deflate_stored(s, flush) {
            /* Stored blocks are limited to 0xffff bytes, pending_buf is limited
             * to pending_buf_size, and each stored block has a 5 byte header:
             */
            var max_block_size = 0xffff;

            if (max_block_size > s.pending_buf_size - 5) {
              max_block_size = s.pending_buf_size - 5;
            }
            /* Copy as much as possible from input to output: */


            for (;;) {
              /* Fill the window as much as possible: */
              if (s.lookahead <= 1) {
                //Assert(s->strstart < s->w_size+MAX_DIST(s) ||
                //  s->block_start >= (long)s->w_size, "slide too late");
                //      if (!(s.strstart < s.w_size + (s.w_size - MIN_LOOKAHEAD) ||
                //        s.block_start >= s.w_size)) {
                //        throw  new Error("slide too late");
                //      }
                fill_window(s);

                if (s.lookahead === 0 && flush === Z_NO_FLUSH) {
                  return BS_NEED_MORE;
                }

                if (s.lookahead === 0) {
                  break;
                }
                /* flush the current block */

              } //Assert(s->block_start >= 0L, "block gone");
              //    if (s.block_start < 0) throw new Error("block gone");


              s.strstart += s.lookahead;
              s.lookahead = 0;
              /* Emit a stored block if pending_buf will be full: */

              var max_start = s.block_start + max_block_size;

              if (s.strstart === 0 || s.strstart >= max_start) {
                /* strstart == 0 is possible when wraparound on 16-bit machine */
                s.lookahead = s.strstart - max_start;
                s.strstart = max_start;
                /*** FLUSH_BLOCK(s, 0); ***/

                flush_block_only(s, false);

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
                /***/

              }
              /* Flush if we may have to slide, otherwise block_start may become
               * negative and the data will be gone:
               */


              if (s.strstart - s.block_start >= s.w_size - MIN_LOOKAHEAD) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
                /***/

              }
            }

            s.insert = 0;

            if (flush === Z_FINISH) {
              /*** FLUSH_BLOCK(s, 1); ***/
              flush_block_only(s, true);

              if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
              }
              /***/


              return BS_FINISH_DONE;
            }

            if (s.strstart > s.block_start) {
              /*** FLUSH_BLOCK(s, 0); ***/
              flush_block_only(s, false);

              if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
              }
              /***/

            }

            return BS_NEED_MORE;
          }
          /* ===========================================================================
           * Compress as much as possible from the input stream, return the current
           * block state.
           * This function does not perform lazy evaluation of matches and inserts
           * new strings in the dictionary only for unmatched strings or for short
           * matches. It is used only for the fast compression options.
           */


          function deflate_fast(s, flush) {
            var hash_head;
            /* head of the hash chain */

            var bflush;
            /* set if current block must be flushed */

            for (;;) {
              /* Make sure that we always have enough lookahead, except
               * at the end of the input file. We need MAX_MATCH bytes
               * for the next match, plus MIN_MATCH bytes to insert the
               * string following the next match.
               */
              if (s.lookahead < MIN_LOOKAHEAD) {
                fill_window(s);

                if (s.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH) {
                  return BS_NEED_MORE;
                }

                if (s.lookahead === 0) {
                  break;
                  /* flush the current block */
                }
              }
              /* Insert the string window[strstart .. strstart+2] in the
               * dictionary, and set hash_head to the head of the hash chain:
               */


              hash_head = 0
              /*NIL*/
              ;

              if (s.lookahead >= MIN_MATCH) {
                /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                s.ins_h = (s.ins_h << s.hash_shift ^ s.window[s.strstart + MIN_MATCH - 1]) & s.hash_mask;
                hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                s.head[s.ins_h] = s.strstart;
                /***/
              }
              /* Find the longest match, discarding those <= prev_length.
               * At this point we have always match_length < MIN_MATCH
               */


              if (hash_head !== 0
              /*NIL*/
              && s.strstart - hash_head <= s.w_size - MIN_LOOKAHEAD) {
                /* To simplify the code, we prevent matches with the string
                 * of window index 0 (in particular we have to avoid a match
                 * of the string with itself at the start of the input file).
                 */
                s.match_length = longest_match(s, hash_head);
                /* longest_match() sets match_start */
              }

              if (s.match_length >= MIN_MATCH) {
                // check_match(s, s.strstart, s.match_start, s.match_length); // for debug only

                /*** _tr_tally_dist(s, s.strstart - s.match_start,
                               s.match_length - MIN_MATCH, bflush); ***/
                bflush = trees._tr_tally(s, s.strstart - s.match_start, s.match_length - MIN_MATCH);
                s.lookahead -= s.match_length;
                /* Insert new strings in the hash table only if the match length
                 * is not too large. This saves time but degrades compression.
                 */

                if (s.match_length <= s.max_lazy_match
                /*max_insert_length*/
                && s.lookahead >= MIN_MATCH) {
                  s.match_length--;
                  /* string at strstart already in table */

                  do {
                    s.strstart++;
                    /*** INSERT_STRING(s, s.strstart, hash_head); ***/

                    s.ins_h = (s.ins_h << s.hash_shift ^ s.window[s.strstart + MIN_MATCH - 1]) & s.hash_mask;
                    hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                    s.head[s.ins_h] = s.strstart;
                    /***/

                    /* strstart never exceeds WSIZE-MAX_MATCH, so there are
                     * always MIN_MATCH bytes ahead.
                     */
                  } while (--s.match_length !== 0);

                  s.strstart++;
                } else {
                  s.strstart += s.match_length;
                  s.match_length = 0;
                  s.ins_h = s.window[s.strstart];
                  /* UPDATE_HASH(s, s.ins_h, s.window[s.strstart+1]); */

                  s.ins_h = (s.ins_h << s.hash_shift ^ s.window[s.strstart + 1]) & s.hash_mask; //#if MIN_MATCH != 3
                  //                Call UPDATE_HASH() MIN_MATCH-3 more times
                  //#endif

                  /* If lookahead < MIN_MATCH, ins_h is garbage, but it does not
                   * matter since it will be recomputed at next deflate call.
                   */
                }
              } else {
                /* No match, output a literal byte */
                //Tracevv((stderr,"%c", s.window[s.strstart]));

                /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
                bflush = trees._tr_tally(s, 0, s.window[s.strstart]);
                s.lookahead--;
                s.strstart++;
              }

              if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
                /***/

              }
            }

            s.insert = s.strstart < MIN_MATCH - 1 ? s.strstart : MIN_MATCH - 1;

            if (flush === Z_FINISH) {
              /*** FLUSH_BLOCK(s, 1); ***/
              flush_block_only(s, true);

              if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
              }
              /***/


              return BS_FINISH_DONE;
            }

            if (s.last_lit) {
              /*** FLUSH_BLOCK(s, 0); ***/
              flush_block_only(s, false);

              if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
              }
              /***/

            }

            return BS_BLOCK_DONE;
          }
          /* ===========================================================================
           * Same as above, but achieves better compression. We use a lazy
           * evaluation for matches: a match is finally adopted only if there is
           * no better match at the next window position.
           */


          function deflate_slow(s, flush) {
            var hash_head;
            /* head of hash chain */

            var bflush;
            /* set if current block must be flushed */

            var max_insert;
            /* Process the input block. */

            for (;;) {
              /* Make sure that we always have enough lookahead, except
               * at the end of the input file. We need MAX_MATCH bytes
               * for the next match, plus MIN_MATCH bytes to insert the
               * string following the next match.
               */
              if (s.lookahead < MIN_LOOKAHEAD) {
                fill_window(s);

                if (s.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH) {
                  return BS_NEED_MORE;
                }

                if (s.lookahead === 0) {
                  break;
                }
                /* flush the current block */

              }
              /* Insert the string window[strstart .. strstart+2] in the
               * dictionary, and set hash_head to the head of the hash chain:
               */


              hash_head = 0
              /*NIL*/
              ;

              if (s.lookahead >= MIN_MATCH) {
                /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                s.ins_h = (s.ins_h << s.hash_shift ^ s.window[s.strstart + MIN_MATCH - 1]) & s.hash_mask;
                hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                s.head[s.ins_h] = s.strstart;
                /***/
              }
              /* Find the longest match, discarding those <= prev_length.
               */


              s.prev_length = s.match_length;
              s.prev_match = s.match_start;
              s.match_length = MIN_MATCH - 1;

              if (hash_head !== 0
              /*NIL*/
              && s.prev_length < s.max_lazy_match && s.strstart - hash_head <= s.w_size - MIN_LOOKAHEAD
              /*MAX_DIST(s)*/
              ) {
                  /* To simplify the code, we prevent matches with the string
                   * of window index 0 (in particular we have to avoid a match
                   * of the string with itself at the start of the input file).
                   */
                  s.match_length = longest_match(s, hash_head);
                  /* longest_match() sets match_start */

                  if (s.match_length <= 5 && (s.strategy === Z_FILTERED || s.match_length === MIN_MATCH && s.strstart - s.match_start > 4096
                  /*TOO_FAR*/
                  )) {
                    /* If prev_match is also MIN_MATCH, match_start is garbage
                     * but we will ignore the current match anyway.
                     */
                    s.match_length = MIN_MATCH - 1;
                  }
                }
              /* If there was a match at the previous step and the current
               * match is not better, output the previous match:
               */


              if (s.prev_length >= MIN_MATCH && s.match_length <= s.prev_length) {
                max_insert = s.strstart + s.lookahead - MIN_MATCH;
                /* Do not insert strings in hash table beyond this. */
                //check_match(s, s.strstart-1, s.prev_match, s.prev_length);

                /***_tr_tally_dist(s, s.strstart - 1 - s.prev_match,
                               s.prev_length - MIN_MATCH, bflush);***/

                bflush = trees._tr_tally(s, s.strstart - 1 - s.prev_match, s.prev_length - MIN_MATCH);
                /* Insert in hash table all strings up to the end of the match.
                 * strstart-1 and strstart are already inserted. If there is not
                 * enough lookahead, the last two strings are not inserted in
                 * the hash table.
                 */

                s.lookahead -= s.prev_length - 1;
                s.prev_length -= 2;

                do {
                  if (++s.strstart <= max_insert) {
                    /*** INSERT_STRING(s, s.strstart, hash_head); ***/
                    s.ins_h = (s.ins_h << s.hash_shift ^ s.window[s.strstart + MIN_MATCH - 1]) & s.hash_mask;
                    hash_head = s.prev[s.strstart & s.w_mask] = s.head[s.ins_h];
                    s.head[s.ins_h] = s.strstart;
                    /***/
                  }
                } while (--s.prev_length !== 0);

                s.match_available = 0;
                s.match_length = MIN_MATCH - 1;
                s.strstart++;

                if (bflush) {
                  /*** FLUSH_BLOCK(s, 0); ***/
                  flush_block_only(s, false);

                  if (s.strm.avail_out === 0) {
                    return BS_NEED_MORE;
                  }
                  /***/

                }
              } else if (s.match_available) {
                /* If there was no match at the previous position, output a
                 * single literal. If there was a match but the current match
                 * is longer, truncate the previous match to a single literal.
                 */
                //Tracevv((stderr,"%c", s->window[s->strstart-1]));

                /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
                bflush = trees._tr_tally(s, 0, s.window[s.strstart - 1]);

                if (bflush) {
                  /*** FLUSH_BLOCK_ONLY(s, 0) ***/
                  flush_block_only(s, false);
                  /***/
                }

                s.strstart++;
                s.lookahead--;

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
              } else {
                /* There is no previous match to compare with, wait for
                 * the next step to decide.
                 */
                s.match_available = 1;
                s.strstart++;
                s.lookahead--;
              }
            } //Assert (flush != Z_NO_FLUSH, "no flush?");


            if (s.match_available) {
              //Tracevv((stderr,"%c", s->window[s->strstart-1]));

              /*** _tr_tally_lit(s, s.window[s.strstart-1], bflush); ***/
              bflush = trees._tr_tally(s, 0, s.window[s.strstart - 1]);
              s.match_available = 0;
            }

            s.insert = s.strstart < MIN_MATCH - 1 ? s.strstart : MIN_MATCH - 1;

            if (flush === Z_FINISH) {
              /*** FLUSH_BLOCK(s, 1); ***/
              flush_block_only(s, true);

              if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
              }
              /***/


              return BS_FINISH_DONE;
            }

            if (s.last_lit) {
              /*** FLUSH_BLOCK(s, 0); ***/
              flush_block_only(s, false);

              if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
              }
              /***/

            }

            return BS_BLOCK_DONE;
          }
          /* ===========================================================================
           * For Z_RLE, simply look for runs of bytes, generate matches only of distance
           * one.  Do not maintain a hash table.  (It will be regenerated if this run of
           * deflate switches away from Z_RLE.)
           */


          function deflate_rle(s, flush) {
            var bflush;
            /* set if current block must be flushed */

            var prev;
            /* byte at distance one to match */

            var scan, strend;
            /* scan goes up to strend for length of run */

            var _win = s.window;

            for (;;) {
              /* Make sure that we always have enough lookahead, except
               * at the end of the input file. We need MAX_MATCH bytes
               * for the longest run, plus one for the unrolled loop.
               */
              if (s.lookahead <= MAX_MATCH) {
                fill_window(s);

                if (s.lookahead <= MAX_MATCH && flush === Z_NO_FLUSH) {
                  return BS_NEED_MORE;
                }

                if (s.lookahead === 0) {
                  break;
                }
                /* flush the current block */

              }
              /* See how many times the previous byte repeats */


              s.match_length = 0;

              if (s.lookahead >= MIN_MATCH && s.strstart > 0) {
                scan = s.strstart - 1;
                prev = _win[scan];

                if (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan]) {
                  strend = s.strstart + MAX_MATCH;

                  do {
                    /*jshint noempty:false*/
                  } while (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && scan < strend);

                  s.match_length = MAX_MATCH - (strend - scan);

                  if (s.match_length > s.lookahead) {
                    s.match_length = s.lookahead;
                  }
                } //Assert(scan <= s->window+(uInt)(s->window_size-1), "wild scan");

              }
              /* Emit match if have run of MIN_MATCH or longer, else emit literal */


              if (s.match_length >= MIN_MATCH) {
                //check_match(s, s.strstart, s.strstart - 1, s.match_length);

                /*** _tr_tally_dist(s, 1, s.match_length - MIN_MATCH, bflush); ***/
                bflush = trees._tr_tally(s, 1, s.match_length - MIN_MATCH);
                s.lookahead -= s.match_length;
                s.strstart += s.match_length;
                s.match_length = 0;
              } else {
                /* No match, output a literal byte */
                //Tracevv((stderr,"%c", s->window[s->strstart]));

                /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/
                bflush = trees._tr_tally(s, 0, s.window[s.strstart]);
                s.lookahead--;
                s.strstart++;
              }

              if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
                /***/

              }
            }

            s.insert = 0;

            if (flush === Z_FINISH) {
              /*** FLUSH_BLOCK(s, 1); ***/
              flush_block_only(s, true);

              if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
              }
              /***/


              return BS_FINISH_DONE;
            }

            if (s.last_lit) {
              /*** FLUSH_BLOCK(s, 0); ***/
              flush_block_only(s, false);

              if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
              }
              /***/

            }

            return BS_BLOCK_DONE;
          }
          /* ===========================================================================
           * For Z_HUFFMAN_ONLY, do not look for matches.  Do not maintain a hash table.
           * (It will be regenerated if this run of deflate switches away from Huffman.)
           */


          function deflate_huff(s, flush) {
            var bflush;
            /* set if current block must be flushed */

            for (;;) {
              /* Make sure that we have a literal to write. */
              if (s.lookahead === 0) {
                fill_window(s);

                if (s.lookahead === 0) {
                  if (flush === Z_NO_FLUSH) {
                    return BS_NEED_MORE;
                  }

                  break;
                  /* flush the current block */
                }
              }
              /* Output a literal byte */


              s.match_length = 0; //Tracevv((stderr,"%c", s->window[s->strstart]));

              /*** _tr_tally_lit(s, s.window[s.strstart], bflush); ***/

              bflush = trees._tr_tally(s, 0, s.window[s.strstart]);
              s.lookahead--;
              s.strstart++;

              if (bflush) {
                /*** FLUSH_BLOCK(s, 0); ***/
                flush_block_only(s, false);

                if (s.strm.avail_out === 0) {
                  return BS_NEED_MORE;
                }
                /***/

              }
            }

            s.insert = 0;

            if (flush === Z_FINISH) {
              /*** FLUSH_BLOCK(s, 1); ***/
              flush_block_only(s, true);

              if (s.strm.avail_out === 0) {
                return BS_FINISH_STARTED;
              }
              /***/


              return BS_FINISH_DONE;
            }

            if (s.last_lit) {
              /*** FLUSH_BLOCK(s, 0); ***/
              flush_block_only(s, false);

              if (s.strm.avail_out === 0) {
                return BS_NEED_MORE;
              }
              /***/

            }

            return BS_BLOCK_DONE;
          }
          /* Values for max_lazy_match, good_match and max_chain_length, depending on
           * the desired pack level (0..9). The values given below have been tuned to
           * exclude worst case performance for pathological files. Better values may be
           * found for specific files.
           */


          var Config = function (good_length, max_lazy, nice_length, max_chain, func) {
            this.good_length = good_length;
            this.max_lazy = max_lazy;
            this.nice_length = nice_length;
            this.max_chain = max_chain;
            this.func = func;
          };

          var configuration_table;
          configuration_table = [
          /*      good lazy nice chain */
          new Config(0, 0, 0, 0, deflate_stored),
          /* 0 store only */
          new Config(4, 4, 8, 4, deflate_fast),
          /* 1 max speed, no lazy matches */
          new Config(4, 5, 16, 8, deflate_fast),
          /* 2 */
          new Config(4, 6, 32, 32, deflate_fast),
          /* 3 */
          new Config(4, 4, 16, 16, deflate_slow),
          /* 4 lazy matches */
          new Config(8, 16, 32, 32, deflate_slow),
          /* 5 */
          new Config(8, 16, 128, 128, deflate_slow),
          /* 6 */
          new Config(8, 32, 128, 256, deflate_slow),
          /* 7 */
          new Config(32, 128, 258, 1024, deflate_slow),
          /* 8 */
          new Config(32, 258, 258, 4096, deflate_slow)
          /* 9 max compression */
          ];
          /* ===========================================================================
           * Initialize the "longest match" routines for a new zlib stream
           */

          function lm_init(s) {
            s.window_size = 2 * s.w_size;
            /*** CLEAR_HASH(s); ***/

            zero(s.head); // Fill with NIL (= 0);

            /* Set the default configuration parameters:
             */

            s.max_lazy_match = configuration_table[s.level].max_lazy;
            s.good_match = configuration_table[s.level].good_length;
            s.nice_match = configuration_table[s.level].nice_length;
            s.max_chain_length = configuration_table[s.level].max_chain;
            s.strstart = 0;
            s.block_start = 0;
            s.lookahead = 0;
            s.insert = 0;
            s.match_length = s.prev_length = MIN_MATCH - 1;
            s.match_available = 0;
            s.ins_h = 0;
          }

          function DeflateState() {
            this.strm = null;
            /* pointer back to this zlib stream */

            this.status = 0;
            /* as the name implies */

            this.pending_buf = null;
            /* output still pending */

            this.pending_buf_size = 0;
            /* size of pending_buf */

            this.pending_out = 0;
            /* next pending byte to output to the stream */

            this.pending = 0;
            /* nb of bytes in the pending buffer */

            this.wrap = 0;
            /* bit 0 true for zlib, bit 1 true for gzip */

            this.gzhead = null;
            /* gzip header information to write */

            this.gzindex = 0;
            /* where in extra, name, or comment */

            this.method = Z_DEFLATED;
            /* can only be DEFLATED */

            this.last_flush = -1;
            /* value of flush param for previous deflate call */

            this.w_size = 0;
            /* LZ77 window size (32K by default) */

            this.w_bits = 0;
            /* log2(w_size)  (8..16) */

            this.w_mask = 0;
            /* w_size - 1 */

            this.window = null;
            /* Sliding window. Input bytes are read into the second half of the window,
             * and move to the first half later to keep a dictionary of at least wSize
             * bytes. With this organization, matches are limited to a distance of
             * wSize-MAX_MATCH bytes, but this ensures that IO is always
             * performed with a length multiple of the block size.
             */

            this.window_size = 0;
            /* Actual size of window: 2*wSize, except when the user input buffer
             * is directly used as sliding window.
             */

            this.prev = null;
            /* Link to older string with same hash index. To limit the size of this
             * array to 64K, this link is maintained only for the last 32K strings.
             * An index in this array is thus a window index modulo 32K.
             */

            this.head = null;
            /* Heads of the hash chains or NIL. */

            this.ins_h = 0;
            /* hash index of string to be inserted */

            this.hash_size = 0;
            /* number of elements in hash table */

            this.hash_bits = 0;
            /* log2(hash_size) */

            this.hash_mask = 0;
            /* hash_size-1 */

            this.hash_shift = 0;
            /* Number of bits by which ins_h must be shifted at each input
             * step. It must be such that after MIN_MATCH steps, the oldest
             * byte no longer takes part in the hash key, that is:
             *   hash_shift * MIN_MATCH >= hash_bits
             */

            this.block_start = 0;
            /* Window position at the beginning of the current output block. Gets
             * negative when the window is moved backwards.
             */

            this.match_length = 0;
            /* length of best match */

            this.prev_match = 0;
            /* previous match */

            this.match_available = 0;
            /* set if previous match exists */

            this.strstart = 0;
            /* start of string to insert */

            this.match_start = 0;
            /* start of matching string */

            this.lookahead = 0;
            /* number of valid bytes ahead in window */

            this.prev_length = 0;
            /* Length of the best match at previous step. Matches not greater than this
             * are discarded. This is used in the lazy match evaluation.
             */

            this.max_chain_length = 0;
            /* To speed up deflation, hash chains are never searched beyond this
             * length.  A higher limit improves compression ratio but degrades the
             * speed.
             */

            this.max_lazy_match = 0;
            /* Attempt to find a better match only when the current match is strictly
             * smaller than this value. This mechanism is used only for compression
             * levels >= 4.
             */
            // That's alias to max_lazy_match, don't use directly
            //this.max_insert_length = 0;

            /* Insert new strings in the hash table only if the match length is not
             * greater than this length. This saves time but degrades compression.
             * max_insert_length is used only for compression levels <= 3.
             */

            this.level = 0;
            /* compression level (1..9) */

            this.strategy = 0;
            /* favor or force Huffman coding*/

            this.good_match = 0;
            /* Use a faster search when the previous match is longer than this */

            this.nice_match = 0;
            /* Stop searching when current match exceeds this */

            /* used by trees.c: */

            /* Didn't use ct_data typedef below to suppress compiler warning */
            // struct ct_data_s dyn_ltree[HEAP_SIZE];   /* literal and length tree */
            // struct ct_data_s dyn_dtree[2*D_CODES+1]; /* distance tree */
            // struct ct_data_s bl_tree[2*BL_CODES+1];  /* Huffman tree for bit lengths */
            // Use flat array of DOUBLE size, with interleaved fata,
            // because JS does not support effective

            this.dyn_ltree = new utils.Buf16(HEAP_SIZE * 2);
            this.dyn_dtree = new utils.Buf16((2 * D_CODES + 1) * 2);
            this.bl_tree = new utils.Buf16((2 * BL_CODES + 1) * 2);
            zero(this.dyn_ltree);
            zero(this.dyn_dtree);
            zero(this.bl_tree);
            this.l_desc = null;
            /* desc. for literal tree */

            this.d_desc = null;
            /* desc. for distance tree */

            this.bl_desc = null;
            /* desc. for bit length tree */
            //ush bl_count[MAX_BITS+1];

            this.bl_count = new utils.Buf16(MAX_BITS + 1);
            /* number of codes at each bit length for an optimal tree */
            //int heap[2*L_CODES+1];      /* heap used to build the Huffman trees */

            this.heap = new utils.Buf16(2 * L_CODES + 1);
            /* heap used to build the Huffman trees */

            zero(this.heap);
            this.heap_len = 0;
            /* number of elements in the heap */

            this.heap_max = 0;
            /* element of largest frequency */

            /* The sons of heap[n] are heap[2*n] and heap[2*n+1]. heap[0] is not used.
             * The same heap array is used to build all trees.
             */

            this.depth = new utils.Buf16(2 * L_CODES + 1); //uch depth[2*L_CODES+1];

            zero(this.depth);
            /* Depth of each subtree used as tie breaker for trees of equal frequency
             */

            this.l_buf = 0;
            /* buffer index for literals or lengths */

            this.lit_bufsize = 0;
            /* Size of match buffer for literals/lengths.  There are 4 reasons for
             * limiting lit_bufsize to 64K:
             *   - frequencies can be kept in 16 bit counters
             *   - if compression is not successful for the first block, all input
             *     data is still in the window so we can still emit a stored block even
             *     when input comes from standard input.  (This can also be done for
             *     all blocks if lit_bufsize is not greater than 32K.)
             *   - if compression is not successful for a file smaller than 64K, we can
             *     even emit a stored file instead of a stored block (saving 5 bytes).
             *     This is applicable only for zip (not gzip or zlib).
             *   - creating new Huffman trees less frequently may not provide fast
             *     adaptation to changes in the input data statistics. (Take for
             *     example a binary file with poorly compressible code followed by
             *     a highly compressible string table.) Smaller buffer sizes give
             *     fast adaptation but have of course the overhead of transmitting
             *     trees more frequently.
             *   - I can't count above 4
             */

            this.last_lit = 0;
            /* running index in l_buf */

            this.d_buf = 0;
            /* Buffer index for distances. To simplify the code, d_buf and l_buf have
             * the same number of elements. To use different lengths, an extra flag
             * array would be necessary.
             */

            this.opt_len = 0;
            /* bit length of current block with optimal trees */

            this.static_len = 0;
            /* bit length of current block with static trees */

            this.matches = 0;
            /* number of string matches in current block */

            this.insert = 0;
            /* bytes at end of window left to insert */

            this.bi_buf = 0;
            /* Output buffer. bits are inserted starting at the bottom (least
             * significant bits).
             */

            this.bi_valid = 0;
            /* Number of valid bits in bi_buf.  All bits above the last valid bit
             * are always zero.
             */
            // Used for window memory init. We safely ignore it for JS. That makes
            // sense only for pointers and memory check tools.
            //this.high_water = 0;

            /* High water mark offset in window for initialized bytes -- bytes above
             * this are set to zero in order to avoid memory check warnings when
             * longest match routines access bytes past the input.  This is then
             * updated to the new high water mark.
             */
          }

          function deflateResetKeep(strm) {
            var s;

            if (!strm || !strm.state) {
              return err(strm, Z_STREAM_ERROR);
            }

            strm.total_in = strm.total_out = 0;
            strm.data_type = Z_UNKNOWN;
            s = strm.state;
            s.pending = 0;
            s.pending_out = 0;

            if (s.wrap < 0) {
              s.wrap = -s.wrap;
              /* was made negative by deflate(..., Z_FINISH); */
            }

            s.status = s.wrap ? INIT_STATE : BUSY_STATE;
            strm.adler = s.wrap === 2 ? 0 // crc32(0, Z_NULL, 0)
            : 1; // adler32(0, Z_NULL, 0)

            s.last_flush = Z_NO_FLUSH;

            trees._tr_init(s);

            return Z_OK;
          }

          function deflateReset(strm) {
            var ret = deflateResetKeep(strm);

            if (ret === Z_OK) {
              lm_init(strm.state);
            }

            return ret;
          }

          function deflateSetHeader(strm, head) {
            if (!strm || !strm.state) {
              return Z_STREAM_ERROR;
            }

            if (strm.state.wrap !== 2) {
              return Z_STREAM_ERROR;
            }

            strm.state.gzhead = head;
            return Z_OK;
          }

          function deflateInit2(strm, level, method, windowBits, memLevel, strategy) {
            if (!strm) {
              // === Z_NULL
              return Z_STREAM_ERROR;
            }

            var wrap = 1;

            if (level === Z_DEFAULT_COMPRESSION) {
              level = 6;
            }

            if (windowBits < 0) {
              /* suppress zlib wrapper */
              wrap = 0;
              windowBits = -windowBits;
            } else if (windowBits > 15) {
              wrap = 2;
              /* write gzip wrapper instead */

              windowBits -= 16;
            }

            if (memLevel < 1 || memLevel > MAX_MEM_LEVEL || method !== Z_DEFLATED || windowBits < 8 || windowBits > 15 || level < 0 || level > 9 || strategy < 0 || strategy > Z_FIXED) {
              return err(strm, Z_STREAM_ERROR);
            }

            if (windowBits === 8) {
              windowBits = 9;
            }
            /* until 256-byte window bug fixed */


            var s = new DeflateState();
            strm.state = s;
            s.strm = strm;
            s.wrap = wrap;
            s.gzhead = null;
            s.w_bits = windowBits;
            s.w_size = 1 << s.w_bits;
            s.w_mask = s.w_size - 1;
            s.hash_bits = memLevel + 7;
            s.hash_size = 1 << s.hash_bits;
            s.hash_mask = s.hash_size - 1;
            s.hash_shift = ~~((s.hash_bits + MIN_MATCH - 1) / MIN_MATCH);
            s.window = new utils.Buf8(s.w_size * 2);
            s.head = new utils.Buf16(s.hash_size);
            s.prev = new utils.Buf16(s.w_size); // Don't need mem init magic for JS.
            //s.high_water = 0;  /* nothing written to s->window yet */

            s.lit_bufsize = 1 << memLevel + 6;
            /* 16K elements by default */

            s.pending_buf_size = s.lit_bufsize * 4;
            s.pending_buf = new utils.Buf8(s.pending_buf_size);
            s.d_buf = s.lit_bufsize >> 1;
            s.l_buf = (1 + 2) * s.lit_bufsize;
            s.level = level;
            s.strategy = strategy;
            s.method = method;
            return deflateReset(strm);
          }

          function deflateInit(strm, level) {
            return deflateInit2(strm, level, Z_DEFLATED, MAX_WBITS, DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY);
          }

          function deflate(strm, flush) {
            var old_flush, s;
            var beg, val; // for gzip header write only

            if (!strm || !strm.state || flush > Z_BLOCK || flush < 0) {
              return strm ? err(strm, Z_STREAM_ERROR) : Z_STREAM_ERROR;
            }

            s = strm.state;

            if (!strm.output || !strm.input && strm.avail_in !== 0 || s.status === FINISH_STATE && flush !== Z_FINISH) {
              return err(strm, strm.avail_out === 0 ? Z_BUF_ERROR : Z_STREAM_ERROR);
            }

            s.strm = strm;
            /* just in case */

            old_flush = s.last_flush;
            s.last_flush = flush;
            /* Write the header */

            if (s.status === INIT_STATE) {
              if (s.wrap === 2) {
                // GZIP header
                strm.adler = 0; //crc32(0L, Z_NULL, 0);

                put_byte(s, 31);
                put_byte(s, 139);
                put_byte(s, 8);

                if (!s.gzhead) {
                  // s->gzhead == Z_NULL
                  put_byte(s, 0);
                  put_byte(s, 0);
                  put_byte(s, 0);
                  put_byte(s, 0);
                  put_byte(s, 0);
                  put_byte(s, s.level === 9 ? 2 : s.strategy >= Z_HUFFMAN_ONLY || s.level < 2 ? 4 : 0);
                  put_byte(s, OS_CODE);
                  s.status = BUSY_STATE;
                } else {
                  put_byte(s, (s.gzhead.text ? 1 : 0) + (s.gzhead.hcrc ? 2 : 0) + (!s.gzhead.extra ? 0 : 4) + (!s.gzhead.name ? 0 : 8) + (!s.gzhead.comment ? 0 : 16));
                  put_byte(s, s.gzhead.time & 0xff);
                  put_byte(s, s.gzhead.time >> 8 & 0xff);
                  put_byte(s, s.gzhead.time >> 16 & 0xff);
                  put_byte(s, s.gzhead.time >> 24 & 0xff);
                  put_byte(s, s.level === 9 ? 2 : s.strategy >= Z_HUFFMAN_ONLY || s.level < 2 ? 4 : 0);
                  put_byte(s, s.gzhead.os & 0xff);

                  if (s.gzhead.extra && s.gzhead.extra.length) {
                    put_byte(s, s.gzhead.extra.length & 0xff);
                    put_byte(s, s.gzhead.extra.length >> 8 & 0xff);
                  }

                  if (s.gzhead.hcrc) {
                    strm.adler = crc32(strm.adler, s.pending_buf, s.pending, 0);
                  }

                  s.gzindex = 0;
                  s.status = EXTRA_STATE;
                }
              } else // DEFLATE header
                {
                  var header = Z_DEFLATED + (s.w_bits - 8 << 4) << 8;
                  var level_flags = -1;

                  if (s.strategy >= Z_HUFFMAN_ONLY || s.level < 2) {
                    level_flags = 0;
                  } else if (s.level < 6) {
                    level_flags = 1;
                  } else if (s.level === 6) {
                    level_flags = 2;
                  } else {
                    level_flags = 3;
                  }

                  header |= level_flags << 6;

                  if (s.strstart !== 0) {
                    header |= PRESET_DICT;
                  }

                  header += 31 - header % 31;
                  s.status = BUSY_STATE;
                  putShortMSB(s, header);
                  /* Save the adler32 of the preset dictionary: */

                  if (s.strstart !== 0) {
                    putShortMSB(s, strm.adler >>> 16);
                    putShortMSB(s, strm.adler & 0xffff);
                  }

                  strm.adler = 1; // adler32(0L, Z_NULL, 0);
                }
            } //#ifdef GZIP


            if (s.status === EXTRA_STATE) {
              if (s.gzhead.extra
              /* != Z_NULL*/
              ) {
                  beg = s.pending;
                  /* start of bytes to update crc */

                  while (s.gzindex < (s.gzhead.extra.length & 0xffff)) {
                    if (s.pending === s.pending_buf_size) {
                      if (s.gzhead.hcrc && s.pending > beg) {
                        strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                      }

                      flush_pending(strm);
                      beg = s.pending;

                      if (s.pending === s.pending_buf_size) {
                        break;
                      }
                    }

                    put_byte(s, s.gzhead.extra[s.gzindex] & 0xff);
                    s.gzindex++;
                  }

                  if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                  }

                  if (s.gzindex === s.gzhead.extra.length) {
                    s.gzindex = 0;
                    s.status = NAME_STATE;
                  }
                } else {
                s.status = NAME_STATE;
              }
            }

            if (s.status === NAME_STATE) {
              if (s.gzhead.name
              /* != Z_NULL*/
              ) {
                  beg = s.pending;
                  /* start of bytes to update crc */
                  //int val;

                  do {
                    if (s.pending === s.pending_buf_size) {
                      if (s.gzhead.hcrc && s.pending > beg) {
                        strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                      }

                      flush_pending(strm);
                      beg = s.pending;

                      if (s.pending === s.pending_buf_size) {
                        val = 1;
                        break;
                      }
                    } // JS specific: little magic to add zero terminator to end of string


                    if (s.gzindex < s.gzhead.name.length) {
                      val = s.gzhead.name.charCodeAt(s.gzindex++) & 0xff;
                    } else {
                      val = 0;
                    }

                    put_byte(s, val);
                  } while (val !== 0);

                  if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                  }

                  if (val === 0) {
                    s.gzindex = 0;
                    s.status = COMMENT_STATE;
                  }
                } else {
                s.status = COMMENT_STATE;
              }
            }

            if (s.status === COMMENT_STATE) {
              if (s.gzhead.comment
              /* != Z_NULL*/
              ) {
                  beg = s.pending;
                  /* start of bytes to update crc */
                  //int val;

                  do {
                    if (s.pending === s.pending_buf_size) {
                      if (s.gzhead.hcrc && s.pending > beg) {
                        strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                      }

                      flush_pending(strm);
                      beg = s.pending;

                      if (s.pending === s.pending_buf_size) {
                        val = 1;
                        break;
                      }
                    } // JS specific: little magic to add zero terminator to end of string


                    if (s.gzindex < s.gzhead.comment.length) {
                      val = s.gzhead.comment.charCodeAt(s.gzindex++) & 0xff;
                    } else {
                      val = 0;
                    }

                    put_byte(s, val);
                  } while (val !== 0);

                  if (s.gzhead.hcrc && s.pending > beg) {
                    strm.adler = crc32(strm.adler, s.pending_buf, s.pending - beg, beg);
                  }

                  if (val === 0) {
                    s.status = HCRC_STATE;
                  }
                } else {
                s.status = HCRC_STATE;
              }
            }

            if (s.status === HCRC_STATE) {
              if (s.gzhead.hcrc) {
                if (s.pending + 2 > s.pending_buf_size) {
                  flush_pending(strm);
                }

                if (s.pending + 2 <= s.pending_buf_size) {
                  put_byte(s, strm.adler & 0xff);
                  put_byte(s, strm.adler >> 8 & 0xff);
                  strm.adler = 0; //crc32(0L, Z_NULL, 0);

                  s.status = BUSY_STATE;
                }
              } else {
                s.status = BUSY_STATE;
              }
            } //#endif

            /* Flush as much pending output as possible */


            if (s.pending !== 0) {
              flush_pending(strm);

              if (strm.avail_out === 0) {
                /* Since avail_out is 0, deflate will be called again with
                 * more output space, but possibly with both pending and
                 * avail_in equal to zero. There won't be anything to do,
                 * but this is not an error situation so make sure we
                 * return OK instead of BUF_ERROR at next call of deflate:
                 */
                s.last_flush = -1;
                return Z_OK;
              }
              /* Make sure there is something to do and avoid duplicate consecutive
               * flushes. For repeated and useless calls with Z_FINISH, we keep
               * returning Z_STREAM_END instead of Z_BUF_ERROR.
               */

            } else if (strm.avail_in === 0 && rank(flush) <= rank(old_flush) && flush !== Z_FINISH) {
              return err(strm, Z_BUF_ERROR);
            }
            /* User must not provide more input after the first FINISH: */


            if (s.status === FINISH_STATE && strm.avail_in !== 0) {
              return err(strm, Z_BUF_ERROR);
            }
            /* Start a new block or continue the current one.
             */


            if (strm.avail_in !== 0 || s.lookahead !== 0 || flush !== Z_NO_FLUSH && s.status !== FINISH_STATE) {
              var bstate = s.strategy === Z_HUFFMAN_ONLY ? deflate_huff(s, flush) : s.strategy === Z_RLE ? deflate_rle(s, flush) : configuration_table[s.level].func(s, flush);

              if (bstate === BS_FINISH_STARTED || bstate === BS_FINISH_DONE) {
                s.status = FINISH_STATE;
              }

              if (bstate === BS_NEED_MORE || bstate === BS_FINISH_STARTED) {
                if (strm.avail_out === 0) {
                  s.last_flush = -1;
                  /* avoid BUF_ERROR next call, see above */
                }

                return Z_OK;
                /* If flush != Z_NO_FLUSH && avail_out == 0, the next call
                 * of deflate should use the same flush parameter to make sure
                 * that the flush is complete. So we don't have to output an
                 * empty block here, this will be done at next call. This also
                 * ensures that for a very small output buffer, we emit at most
                 * one empty block.
                 */
              }

              if (bstate === BS_BLOCK_DONE) {
                if (flush === Z_PARTIAL_FLUSH) {
                  trees._tr_align(s);
                } else if (flush !== Z_BLOCK) {
                  /* FULL_FLUSH or SYNC_FLUSH */
                  trees._tr_stored_block(s, 0, 0, false);
                  /* For a full flush, this empty block will be recognized
                   * as a special marker by inflate_sync().
                   */


                  if (flush === Z_FULL_FLUSH) {
                    /*** CLEAR_HASH(s); ***/

                    /* forget history */
                    zero(s.head); // Fill with NIL (= 0);

                    if (s.lookahead === 0) {
                      s.strstart = 0;
                      s.block_start = 0;
                      s.insert = 0;
                    }
                  }
                }

                flush_pending(strm);

                if (strm.avail_out === 0) {
                  s.last_flush = -1;
                  /* avoid BUF_ERROR at next call, see above */

                  return Z_OK;
                }
              }
            } //Assert(strm->avail_out > 0, "bug2");
            //if (strm.avail_out <= 0) { throw new Error("bug2");}


            if (flush !== Z_FINISH) {
              return Z_OK;
            }

            if (s.wrap <= 0) {
              return Z_STREAM_END;
            }
            /* Write the trailer */


            if (s.wrap === 2) {
              put_byte(s, strm.adler & 0xff);
              put_byte(s, strm.adler >> 8 & 0xff);
              put_byte(s, strm.adler >> 16 & 0xff);
              put_byte(s, strm.adler >> 24 & 0xff);
              put_byte(s, strm.total_in & 0xff);
              put_byte(s, strm.total_in >> 8 & 0xff);
              put_byte(s, strm.total_in >> 16 & 0xff);
              put_byte(s, strm.total_in >> 24 & 0xff);
            } else {
              putShortMSB(s, strm.adler >>> 16);
              putShortMSB(s, strm.adler & 0xffff);
            }

            flush_pending(strm);
            /* If avail_out is zero, the application will call deflate again
             * to flush the rest.
             */

            if (s.wrap > 0) {
              s.wrap = -s.wrap;
            }
            /* write the trailer only once! */


            return s.pending !== 0 ? Z_OK : Z_STREAM_END;
          }

          function deflateEnd(strm) {
            var status;

            if (!strm
            /*== Z_NULL*/
            || !strm.state
            /*== Z_NULL*/
            ) {
                return Z_STREAM_ERROR;
              }

            status = strm.state.status;

            if (status !== INIT_STATE && status !== EXTRA_STATE && status !== NAME_STATE && status !== COMMENT_STATE && status !== HCRC_STATE && status !== BUSY_STATE && status !== FINISH_STATE) {
              return err(strm, Z_STREAM_ERROR);
            }

            strm.state = null;
            return status === BUSY_STATE ? err(strm, Z_DATA_ERROR) : Z_OK;
          }
          /* =========================================================================
           * Copy the source state to the destination state
           */
          //function deflateCopy(dest, source) {
          //
          //}


          exports.deflateInit = deflateInit;
          exports.deflateInit2 = deflateInit2;
          exports.deflateReset = deflateReset;
          exports.deflateResetKeep = deflateResetKeep;
          exports.deflateSetHeader = deflateSetHeader;
          exports.deflate = deflate;
          exports.deflateEnd = deflateEnd;
          exports.deflateInfo = 'pako deflate (from Nodeca project)';
          /* Not implemented
          exports.deflateBound = deflateBound;
          exports.deflateCopy = deflateCopy;
          exports.deflateSetDictionary = deflateSetDictionary;
          exports.deflateParams = deflateParams;
          exports.deflatePending = deflatePending;
          exports.deflatePrime = deflatePrime;
          exports.deflateTune = deflateTune;
          */
        }, {
          "../utils/common": 27,
          "./adler32": 29,
          "./crc32": 31,
          "./messages": 37,
          "./trees": 38
        }],
        33: [function (_dereq_, module, exports) {

          function GZheader() {
            /* true if compressed data believed to be text */
            this.text = 0;
            /* modification time */

            this.time = 0;
            /* extra flags (not used when writing a gzip file) */

            this.xflags = 0;
            /* operating system */

            this.os = 0;
            /* pointer to extra field or Z_NULL if none */

            this.extra = null;
            /* extra field length (valid if extra != Z_NULL) */

            this.extra_len = 0; // Actually, we don't need it in JS,
            // but leave for few code modifications
            //
            // Setup limits is not necessary because in js we should not preallocate memory
            // for inflate use constant limit in 65536 bytes
            //

            /* space at extra (only when reading header) */
            // this.extra_max  = 0;

            /* pointer to zero-terminated file name or Z_NULL */

            this.name = '';
            /* space at name (only when reading header) */
            // this.name_max   = 0;

            /* pointer to zero-terminated comment or Z_NULL */

            this.comment = '';
            /* space at comment (only when reading header) */
            // this.comm_max   = 0;

            /* true if there was or will be a header crc */

            this.hcrc = 0;
            /* true when done reading gzip header (not used when writing a gzip file) */

            this.done = false;
          }

          module.exports = GZheader;
        }, {}],
        34: [function (_dereq_, module, exports) {

          var BAD = 30;
          /* got a data error -- remain here until reset */

          var TYPE = 12;
          /* i: waiting for type bits, including last-flag bit */

          /*
             Decode literal, length, and distance codes and write out the resulting
             literal and match bytes until either not enough input or output is
             available, an end-of-block is encountered, or a data error is encountered.
             When large enough input and output buffers are supplied to inflate(), for
             example, a 16K input buffer and a 64K output buffer, more than 95% of the
             inflate execution time is spent in this routine.
          
             Entry assumptions:
          
                  state.mode === LEN
                  strm.avail_in >= 6
                  strm.avail_out >= 258
                  start >= strm.avail_out
                  state.bits < 8
          
             On return, state.mode is one of:
          
                  LEN -- ran out of enough output space or enough available input
                  TYPE -- reached end of block code, inflate() to interpret next block
                  BAD -- error in block data
          
             Notes:
          
              - The maximum input bits used by a length/distance pair is 15 bits for the
                length code, 5 bits for the length extra, 15 bits for the distance code,
                and 13 bits for the distance extra.  This totals 48 bits, or six bytes.
                Therefore if strm.avail_in >= 6, then there is enough input to avoid
                checking for available input while decoding.
          
              - The maximum bytes that a single length/distance pair can output is 258
                bytes, which is the maximum length that can be coded.  inflate_fast()
                requires strm.avail_out >= 258 for each loop to avoid checking for
                output space.
           */

          module.exports = function inflate_fast(strm, start) {
            var state;

            var _in;
            /* local strm.input */


            var last;
            /* have enough input while in < last */

            var _out;
            /* local strm.output */


            var beg;
            /* inflate()'s initial strm.output */

            var end;
            /* while out < end, enough space available */
            //#ifdef INFLATE_STRICT

            var dmax;
            /* maximum distance from zlib header */
            //#endif

            var wsize;
            /* window size or zero if not using window */

            var whave;
            /* valid bytes in the window */

            var wnext;
            /* window write index */

            var window;
            /* allocated sliding window, if wsize != 0 */

            var hold;
            /* local strm.hold */

            var bits;
            /* local strm.bits */

            var lcode;
            /* local strm.lencode */

            var dcode;
            /* local strm.distcode */

            var lmask;
            /* mask for first level of length codes */

            var dmask;
            /* mask for first level of distance codes */

            var here;
            /* retrieved table entry */

            var op;
            /* code bits, operation, extra bits, or */

            /*  window position, window bytes to copy */

            var len;
            /* match length, unused bytes */

            var dist;
            /* match distance */

            var from;
            /* where to copy match from */

            var from_source;
            var input, output; // JS specific, because we have no pointers

            /* copy state to local variables */

            state = strm.state; //here = state.here;

            _in = strm.next_in;
            input = strm.input;
            last = _in + (strm.avail_in - 5);
            _out = strm.next_out;
            output = strm.output;
            beg = _out - (start - strm.avail_out);
            end = _out + (strm.avail_out - 257); //#ifdef INFLATE_STRICT

            dmax = state.dmax; //#endif

            wsize = state.wsize;
            whave = state.whave;
            wnext = state.wnext;
            window = state.window;
            hold = state.hold;
            bits = state.bits;
            lcode = state.lencode;
            dcode = state.distcode;
            lmask = (1 << state.lenbits) - 1;
            dmask = (1 << state.distbits) - 1;
            /* decode literals and length/distances until end-of-block or not enough
               input data or output space */

            top: do {
              if (bits < 15) {
                hold += input[_in++] << bits;
                bits += 8;
                hold += input[_in++] << bits;
                bits += 8;
              }

              here = lcode[hold & lmask];

              dolen: for (;;) {
                // Goto emulation
                op = here >>> 24
                /*here.bits*/
                ;
                hold >>>= op;
                bits -= op;
                op = here >>> 16 & 0xff
                /*here.op*/
                ;

                if (op === 0) {
                  /* literal */
                  //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
                  //        "inflate:         literal '%c'\n" :
                  //        "inflate:         literal 0x%02x\n", here.val));
                  output[_out++] = here & 0xffff
                  /*here.val*/
                  ;
                } else if (op & 16) {
                  /* length base */
                  len = here & 0xffff
                  /*here.val*/
                  ;
                  op &= 15;
                  /* number of extra bits */

                  if (op) {
                    if (bits < op) {
                      hold += input[_in++] << bits;
                      bits += 8;
                    }

                    len += hold & (1 << op) - 1;
                    hold >>>= op;
                    bits -= op;
                  } //Tracevv((stderr, "inflate:         length %u\n", len));


                  if (bits < 15) {
                    hold += input[_in++] << bits;
                    bits += 8;
                    hold += input[_in++] << bits;
                    bits += 8;
                  }

                  here = dcode[hold & dmask];

                  dodist: for (;;) {
                    // goto emulation
                    op = here >>> 24
                    /*here.bits*/
                    ;
                    hold >>>= op;
                    bits -= op;
                    op = here >>> 16 & 0xff
                    /*here.op*/
                    ;

                    if (op & 16) {
                      /* distance base */
                      dist = here & 0xffff
                      /*here.val*/
                      ;
                      op &= 15;
                      /* number of extra bits */

                      if (bits < op) {
                        hold += input[_in++] << bits;
                        bits += 8;

                        if (bits < op) {
                          hold += input[_in++] << bits;
                          bits += 8;
                        }
                      }

                      dist += hold & (1 << op) - 1; //#ifdef INFLATE_STRICT

                      if (dist > dmax) {
                        strm.msg = 'invalid distance too far back';
                        state.mode = BAD;
                        break top;
                      } //#endif


                      hold >>>= op;
                      bits -= op; //Tracevv((stderr, "inflate:         distance %u\n", dist));

                      op = _out - beg;
                      /* max distance in output */

                      if (dist > op) {
                        /* see if copy from window */
                        op = dist - op;
                        /* distance back in window */

                        if (op > whave) {
                          if (state.sane) {
                            strm.msg = 'invalid distance too far back';
                            state.mode = BAD;
                            break top;
                          } // (!) This block is disabled in zlib defailts,
                          // don't enable it for binary compatibility
                          //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                          //                if (len <= op - whave) {
                          //                  do {
                          //                    output[_out++] = 0;
                          //                  } while (--len);
                          //                  continue top;
                          //                }
                          //                len -= op - whave;
                          //                do {
                          //                  output[_out++] = 0;
                          //                } while (--op > whave);
                          //                if (op === 0) {
                          //                  from = _out - dist;
                          //                  do {
                          //                    output[_out++] = output[from++];
                          //                  } while (--len);
                          //                  continue top;
                          //                }
                          //#endif

                        }

                        from = 0; // window index

                        from_source = window;

                        if (wnext === 0) {
                          /* very common case */
                          from += wsize - op;

                          if (op < len) {
                            /* some from window */
                            len -= op;

                            do {
                              output[_out++] = window[from++];
                            } while (--op);

                            from = _out - dist;
                            /* rest from output */

                            from_source = output;
                          }
                        } else if (wnext < op) {
                          /* wrap around window */
                          from += wsize + wnext - op;
                          op -= wnext;

                          if (op < len) {
                            /* some from end of window */
                            len -= op;

                            do {
                              output[_out++] = window[from++];
                            } while (--op);

                            from = 0;

                            if (wnext < len) {
                              /* some from start of window */
                              op = wnext;
                              len -= op;

                              do {
                                output[_out++] = window[from++];
                              } while (--op);

                              from = _out - dist;
                              /* rest from output */

                              from_source = output;
                            }
                          }
                        } else {
                          /* contiguous in window */
                          from += wnext - op;

                          if (op < len) {
                            /* some from window */
                            len -= op;

                            do {
                              output[_out++] = window[from++];
                            } while (--op);

                            from = _out - dist;
                            /* rest from output */

                            from_source = output;
                          }
                        }

                        while (len > 2) {
                          output[_out++] = from_source[from++];
                          output[_out++] = from_source[from++];
                          output[_out++] = from_source[from++];
                          len -= 3;
                        }

                        if (len) {
                          output[_out++] = from_source[from++];

                          if (len > 1) {
                            output[_out++] = from_source[from++];
                          }
                        }
                      } else {
                        from = _out - dist;
                        /* copy direct from output */

                        do {
                          /* minimum length is three */
                          output[_out++] = output[from++];
                          output[_out++] = output[from++];
                          output[_out++] = output[from++];
                          len -= 3;
                        } while (len > 2);

                        if (len) {
                          output[_out++] = output[from++];

                          if (len > 1) {
                            output[_out++] = output[from++];
                          }
                        }
                      }
                    } else if ((op & 64) === 0) {
                      /* 2nd level distance code */
                      here = dcode[(here & 0xffff) + (
                      /*here.val*/
                      hold & (1 << op) - 1)];
                      continue dodist;
                    } else {
                      strm.msg = 'invalid distance code';
                      state.mode = BAD;
                      break top;
                    }

                    break; // need to emulate goto via "continue"
                  }
                } else if ((op & 64) === 0) {
                  /* 2nd level length code */
                  here = lcode[(here & 0xffff) + (
                  /*here.val*/
                  hold & (1 << op) - 1)];
                  continue dolen;
                } else if (op & 32) {
                  /* end-of-block */
                  //Tracevv((stderr, "inflate:         end of block\n"));
                  state.mode = TYPE;
                  break top;
                } else {
                  strm.msg = 'invalid literal/length code';
                  state.mode = BAD;
                  break top;
                }

                break; // need to emulate goto via "continue"
              }
            } while (_in < last && _out < end);
            /* return unused bytes (on entry, bits < 8, so in won't go too far back) */


            len = bits >> 3;
            _in -= len;
            bits -= len << 3;
            hold &= (1 << bits) - 1;
            /* update state and return */

            strm.next_in = _in;
            strm.next_out = _out;
            strm.avail_in = _in < last ? 5 + (last - _in) : 5 - (_in - last);
            strm.avail_out = _out < end ? 257 + (end - _out) : 257 - (_out - end);
            state.hold = hold;
            state.bits = bits;
            return;
          };
        }, {}],
        35: [function (_dereq_, module, exports) {

          var utils = _dereq_('../utils/common');

          var adler32 = _dereq_('./adler32');

          var crc32 = _dereq_('./crc32');

          var inflate_fast = _dereq_('./inffast');

          var inflate_table = _dereq_('./inftrees');

          var CODES = 0;
          var LENS = 1;
          var DISTS = 2;
          /* Public constants ==========================================================*/

          /* ===========================================================================*/

          /* Allowed flush values; see deflate() and inflate() below for details */
          //var Z_NO_FLUSH      = 0;
          //var Z_PARTIAL_FLUSH = 1;
          //var Z_SYNC_FLUSH    = 2;
          //var Z_FULL_FLUSH    = 3;

          var Z_FINISH = 4;
          var Z_BLOCK = 5;
          var Z_TREES = 6;
          /* Return codes for the compression/decompression functions. Negative values
           * are errors, positive values are used for special but normal events.
           */

          var Z_OK = 0;
          var Z_STREAM_END = 1;
          var Z_NEED_DICT = 2; //var Z_ERRNO         = -1;

          var Z_STREAM_ERROR = -2;
          var Z_DATA_ERROR = -3;
          var Z_MEM_ERROR = -4;
          var Z_BUF_ERROR = -5; //var Z_VERSION_ERROR = -6;

          /* The deflate compression method */

          var Z_DEFLATED = 8;
          /* STATES ====================================================================*/

          /* ===========================================================================*/

          var HEAD = 1;
          /* i: waiting for magic header */

          var FLAGS = 2;
          /* i: waiting for method and flags (gzip) */

          var TIME = 3;
          /* i: waiting for modification time (gzip) */

          var OS = 4;
          /* i: waiting for extra flags and operating system (gzip) */

          var EXLEN = 5;
          /* i: waiting for extra length (gzip) */

          var EXTRA = 6;
          /* i: waiting for extra bytes (gzip) */

          var NAME = 7;
          /* i: waiting for end of file name (gzip) */

          var COMMENT = 8;
          /* i: waiting for end of comment (gzip) */

          var HCRC = 9;
          /* i: waiting for header crc (gzip) */

          var DICTID = 10;
          /* i: waiting for dictionary check value */

          var DICT = 11;
          /* waiting for inflateSetDictionary() call */

          var TYPE = 12;
          /* i: waiting for type bits, including last-flag bit */

          var TYPEDO = 13;
          /* i: same, but skip check to exit inflate on new block */

          var STORED = 14;
          /* i: waiting for stored size (length and complement) */

          var COPY_ = 15;
          /* i/o: same as COPY below, but only first time in */

          var COPY = 16;
          /* i/o: waiting for input or output to copy stored block */

          var TABLE = 17;
          /* i: waiting for dynamic block table lengths */

          var LENLENS = 18;
          /* i: waiting for code length code lengths */

          var CODELENS = 19;
          /* i: waiting for length/lit and distance code lengths */

          var LEN_ = 20;
          /* i: same as LEN below, but only first time in */

          var LEN = 21;
          /* i: waiting for length/lit/eob code */

          var LENEXT = 22;
          /* i: waiting for length extra bits */

          var DIST = 23;
          /* i: waiting for distance code */

          var DISTEXT = 24;
          /* i: waiting for distance extra bits */

          var MATCH = 25;
          /* o: waiting for output space to copy string */

          var LIT = 26;
          /* o: waiting for output space to write literal */

          var CHECK = 27;
          /* i: waiting for 32-bit check value */

          var LENGTH = 28;
          /* i: waiting for 32-bit length (gzip) */

          var DONE = 29;
          /* finished check, done -- remain here until reset */

          var BAD = 30;
          /* got a data error -- remain here until reset */

          var MEM = 31;
          /* got an inflate() memory error -- remain here until reset */

          var SYNC = 32;
          /* looking for synchronization bytes to restart inflate() */

          /* ===========================================================================*/

          var ENOUGH_LENS = 852;
          var ENOUGH_DISTS = 592; //var ENOUGH =  (ENOUGH_LENS+ENOUGH_DISTS);

          var MAX_WBITS = 15;
          /* 32K LZ77 window */

          var DEF_WBITS = MAX_WBITS;

          function ZSWAP32(q) {
            return (q >>> 24 & 0xff) + (q >>> 8 & 0xff00) + ((q & 0xff00) << 8) + ((q & 0xff) << 24);
          }

          function InflateState() {
            this.mode = 0;
            /* current inflate mode */

            this.last = false;
            /* true if processing last block */

            this.wrap = 0;
            /* bit 0 true for zlib, bit 1 true for gzip */

            this.havedict = false;
            /* true if dictionary provided */

            this.flags = 0;
            /* gzip header method and flags (0 if zlib) */

            this.dmax = 0;
            /* zlib header max distance (INFLATE_STRICT) */

            this.check = 0;
            /* protected copy of check value */

            this.total = 0;
            /* protected copy of output count */
            // TODO: may be {}

            this.head = null;
            /* where to save gzip header information */

            /* sliding window */

            this.wbits = 0;
            /* log base 2 of requested window size */

            this.wsize = 0;
            /* window size or zero if not using window */

            this.whave = 0;
            /* valid bytes in the window */

            this.wnext = 0;
            /* window write index */

            this.window = null;
            /* allocated sliding window, if needed */

            /* bit accumulator */

            this.hold = 0;
            /* input bit accumulator */

            this.bits = 0;
            /* number of bits in "in" */

            /* for string and stored block copying */

            this.length = 0;
            /* literal or length of data to copy */

            this.offset = 0;
            /* distance back to copy string from */

            /* for table and code decoding */

            this.extra = 0;
            /* extra bits needed */

            /* fixed and dynamic code tables */

            this.lencode = null;
            /* starting table for length/literal codes */

            this.distcode = null;
            /* starting table for distance codes */

            this.lenbits = 0;
            /* index bits for lencode */

            this.distbits = 0;
            /* index bits for distcode */

            /* dynamic table building */

            this.ncode = 0;
            /* number of code length code lengths */

            this.nlen = 0;
            /* number of length code lengths */

            this.ndist = 0;
            /* number of distance code lengths */

            this.have = 0;
            /* number of code lengths in lens[] */

            this.next = null;
            /* next available space in codes[] */

            this.lens = new utils.Buf16(320);
            /* temporary storage for code lengths */

            this.work = new utils.Buf16(288);
            /* work area for code table building */

            /*
             because we don't have pointers in js, we use lencode and distcode directly
             as buffers so we don't need codes
            */
            //this.codes = new utils.Buf32(ENOUGH);       /* space for code tables */

            this.lendyn = null;
            /* dynamic table for length/literal codes (JS specific) */

            this.distdyn = null;
            /* dynamic table for distance codes (JS specific) */

            this.sane = 0;
            /* if false, allow invalid distance too far */

            this.back = 0;
            /* bits back of last unprocessed length/lit */

            this.was = 0;
            /* initial length of match */
          }

          function inflateResetKeep(strm) {
            var state;

            if (!strm || !strm.state) {
              return Z_STREAM_ERROR;
            }

            state = strm.state;
            strm.total_in = strm.total_out = state.total = 0;
            strm.msg = '';
            /*Z_NULL*/

            if (state.wrap) {
              /* to support ill-conceived Java test suite */
              strm.adler = state.wrap & 1;
            }

            state.mode = HEAD;
            state.last = 0;
            state.havedict = 0;
            state.dmax = 32768;
            state.head = null
            /*Z_NULL*/
            ;
            state.hold = 0;
            state.bits = 0; //state.lencode = state.distcode = state.next = state.codes;

            state.lencode = state.lendyn = new utils.Buf32(ENOUGH_LENS);
            state.distcode = state.distdyn = new utils.Buf32(ENOUGH_DISTS);
            state.sane = 1;
            state.back = -1; //Tracev((stderr, "inflate: reset\n"));

            return Z_OK;
          }

          function inflateReset(strm) {
            var state;

            if (!strm || !strm.state) {
              return Z_STREAM_ERROR;
            }

            state = strm.state;
            state.wsize = 0;
            state.whave = 0;
            state.wnext = 0;
            return inflateResetKeep(strm);
          }

          function inflateReset2(strm, windowBits) {
            var wrap;
            var state;
            /* get the state */

            if (!strm || !strm.state) {
              return Z_STREAM_ERROR;
            }

            state = strm.state;
            /* extract wrap request from windowBits parameter */

            if (windowBits < 0) {
              wrap = 0;
              windowBits = -windowBits;
            } else {
              wrap = (windowBits >> 4) + 1;

              if (windowBits < 48) {
                windowBits &= 15;
              }
            }
            /* set number of window bits, free window if different */


            if (windowBits && (windowBits < 8 || windowBits > 15)) {
              return Z_STREAM_ERROR;
            }

            if (state.window !== null && state.wbits !== windowBits) {
              state.window = null;
            }
            /* update state and reset the rest of it */


            state.wrap = wrap;
            state.wbits = windowBits;
            return inflateReset(strm);
          }

          function inflateInit2(strm, windowBits) {
            var ret;
            var state;

            if (!strm) {
              return Z_STREAM_ERROR;
            } //strm.msg = Z_NULL;                 /* in case we return an error */


            state = new InflateState(); //if (state === Z_NULL) return Z_MEM_ERROR;
            //Tracev((stderr, "inflate: allocated\n"));

            strm.state = state;
            state.window = null
            /*Z_NULL*/
            ;
            ret = inflateReset2(strm, windowBits);

            if (ret !== Z_OK) {
              strm.state = null
              /*Z_NULL*/
              ;
            }

            return ret;
          }

          function inflateInit(strm) {
            return inflateInit2(strm, DEF_WBITS);
          }
          /*
           Return state with length and distance decoding tables and index sizes set to
           fixed code decoding.  Normally this returns fixed tables from inffixed.h.
           If BUILDFIXED is defined, then instead this routine builds the tables the
           first time it's called, and returns those tables the first time and
           thereafter.  This reduces the size of the code by about 2K bytes, in
           exchange for a little execution time.  However, BUILDFIXED should not be
           used for threaded applications, since the rewriting of the tables and virgin
           may not be thread-safe.
           */


          var virgin = true;
          var lenfix, distfix; // We have no pointers in JS, so keep tables separate

          function fixedtables(state) {
            /* build fixed huffman tables if first call (may not be thread safe) */
            if (virgin) {
              var sym;
              lenfix = new utils.Buf32(512);
              distfix = new utils.Buf32(32);
              /* literal/length table */

              sym = 0;

              while (sym < 144) {
                state.lens[sym++] = 8;
              }

              while (sym < 256) {
                state.lens[sym++] = 9;
              }

              while (sym < 280) {
                state.lens[sym++] = 7;
              }

              while (sym < 288) {
                state.lens[sym++] = 8;
              }

              inflate_table(LENS, state.lens, 0, 288, lenfix, 0, state.work, {
                bits: 9
              });
              /* distance table */

              sym = 0;

              while (sym < 32) {
                state.lens[sym++] = 5;
              }

              inflate_table(DISTS, state.lens, 0, 32, distfix, 0, state.work, {
                bits: 5
              });
              /* do this just once */

              virgin = false;
            }

            state.lencode = lenfix;
            state.lenbits = 9;
            state.distcode = distfix;
            state.distbits = 5;
          }
          /*
           Update the window with the last wsize (normally 32K) bytes written before
           returning.  If window does not exist yet, create it.  This is only called
           when a window is already in use, or when output has been written during this
           inflate call, but the end of the deflate stream has not been reached yet.
           It is also called to create a window for dictionary data when a dictionary
           is loaded.
          
           Providing output buffers larger than 32K to inflate() should provide a speed
           advantage, since only the last 32K of output is copied to the sliding window
           upon return from inflate(), and since all distances after the first 32K of
           output will fall in the output data, making match copies simpler and faster.
           The advantage may be dependent on the size of the processor's data caches.
           */


          function updatewindow(strm, src, end, copy) {
            var dist;
            var state = strm.state;
            /* if it hasn't been done already, allocate space for the window */

            if (state.window === null) {
              state.wsize = 1 << state.wbits;
              state.wnext = 0;
              state.whave = 0;
              state.window = new utils.Buf8(state.wsize);
            }
            /* copy state->wsize or less output bytes into the circular window */


            if (copy >= state.wsize) {
              utils.arraySet(state.window, src, end - state.wsize, state.wsize, 0);
              state.wnext = 0;
              state.whave = state.wsize;
            } else {
              dist = state.wsize - state.wnext;

              if (dist > copy) {
                dist = copy;
              } //zmemcpy(state->window + state->wnext, end - copy, dist);


              utils.arraySet(state.window, src, end - copy, dist, state.wnext);
              copy -= dist;

              if (copy) {
                //zmemcpy(state->window, end - copy, copy);
                utils.arraySet(state.window, src, end - copy, copy, 0);
                state.wnext = copy;
                state.whave = state.wsize;
              } else {
                state.wnext += dist;

                if (state.wnext === state.wsize) {
                  state.wnext = 0;
                }

                if (state.whave < state.wsize) {
                  state.whave += dist;
                }
              }
            }

            return 0;
          }

          function inflate(strm, flush) {
            var state;
            var input, output; // input/output buffers

            var next;
            /* next input INDEX */

            var put;
            /* next output INDEX */

            var have, left;
            /* available input and output */

            var hold;
            /* bit buffer */

            var bits;
            /* bits in bit buffer */

            var _in, _out;
            /* save starting available input and output */


            var copy;
            /* number of stored or match bytes to copy */

            var from;
            /* where to copy match bytes from */

            var from_source;
            var here = 0;
            /* current decoding table entry */

            var here_bits, here_op, here_val; // paked "here" denormalized (JS specific)
            //var last;                   /* parent table entry */

            var last_bits, last_op, last_val; // paked "last" denormalized (JS specific)

            var len;
            /* length to copy for repeats, bits to drop */

            var ret;
            /* return code */

            var hbuf = new utils.Buf8(4);
            /* buffer for gzip header crc calculation */

            var opts;
            var n; // temporary var for NEED_BITS

            var order =
            /* permutation of code lengths */
            [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];

            if (!strm || !strm.state || !strm.output || !strm.input && strm.avail_in !== 0) {
              return Z_STREAM_ERROR;
            }

            state = strm.state;

            if (state.mode === TYPE) {
              state.mode = TYPEDO;
            }
            /* skip check */
            //--- LOAD() ---


            put = strm.next_out;
            output = strm.output;
            left = strm.avail_out;
            next = strm.next_in;
            input = strm.input;
            have = strm.avail_in;
            hold = state.hold;
            bits = state.bits; //---

            _in = have;
            _out = left;
            ret = Z_OK;

            inf_leave: // goto emulation
            for (;;) {
              switch (state.mode) {
                case HEAD:
                  if (state.wrap === 0) {
                    state.mode = TYPEDO;
                    break;
                  } //=== NEEDBITS(16);


                  while (bits < 16) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  if (state.wrap & 2 && hold === 0x8b1f) {
                    /* gzip header */
                    state.check = 0
                    /*crc32(0L, Z_NULL, 0)*/
                    ; //=== CRC2(state.check, hold);

                    hbuf[0] = hold & 0xff;
                    hbuf[1] = hold >>> 8 & 0xff;
                    state.check = crc32(state.check, hbuf, 2, 0); //===//
                    //=== INITBITS();

                    hold = 0;
                    bits = 0; //===//

                    state.mode = FLAGS;
                    break;
                  }

                  state.flags = 0;
                  /* expect zlib header */

                  if (state.head) {
                    state.head.done = false;
                  }

                  if (!(state.wrap & 1) ||
                  /* check if zlib header allowed */
                  (((hold & 0xff) <<
                  /*BITS(8)*/
                  8) + (hold >> 8)) % 31) {
                    strm.msg = 'incorrect header check';
                    state.mode = BAD;
                    break;
                  }

                  if ((hold & 0x0f) !==
                  /*BITS(4)*/
                  Z_DEFLATED) {
                    strm.msg = 'unknown compression method';
                    state.mode = BAD;
                    break;
                  } //--- DROPBITS(4) ---//


                  hold >>>= 4;
                  bits -= 4; //---//

                  len = (hold & 0x0f) +
                  /*BITS(4)*/
                  8;

                  if (state.wbits === 0) {
                    state.wbits = len;
                  } else if (len > state.wbits) {
                    strm.msg = 'invalid window size';
                    state.mode = BAD;
                    break;
                  }

                  state.dmax = 1 << len; //Tracev((stderr, "inflate:   zlib header ok\n"));

                  strm.adler = state.check = 1
                  /*adler32(0L, Z_NULL, 0)*/
                  ;
                  state.mode = hold & 0x200 ? DICTID : TYPE; //=== INITBITS();

                  hold = 0;
                  bits = 0; //===//

                  break;

                case FLAGS:
                  //=== NEEDBITS(16); */
                  while (bits < 16) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  state.flags = hold;

                  if ((state.flags & 0xff) !== Z_DEFLATED) {
                    strm.msg = 'unknown compression method';
                    state.mode = BAD;
                    break;
                  }

                  if (state.flags & 0xe000) {
                    strm.msg = 'unknown header flags set';
                    state.mode = BAD;
                    break;
                  }

                  if (state.head) {
                    state.head.text = hold >> 8 & 1;
                  }

                  if (state.flags & 0x0200) {
                    //=== CRC2(state.check, hold);
                    hbuf[0] = hold & 0xff;
                    hbuf[1] = hold >>> 8 & 0xff;
                    state.check = crc32(state.check, hbuf, 2, 0); //===//
                  } //=== INITBITS();


                  hold = 0;
                  bits = 0; //===//

                  state.mode = TIME;

                /* falls through */

                case TIME:
                  //=== NEEDBITS(32); */
                  while (bits < 32) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  if (state.head) {
                    state.head.time = hold;
                  }

                  if (state.flags & 0x0200) {
                    //=== CRC4(state.check, hold)
                    hbuf[0] = hold & 0xff;
                    hbuf[1] = hold >>> 8 & 0xff;
                    hbuf[2] = hold >>> 16 & 0xff;
                    hbuf[3] = hold >>> 24 & 0xff;
                    state.check = crc32(state.check, hbuf, 4, 0); //===
                  } //=== INITBITS();


                  hold = 0;
                  bits = 0; //===//

                  state.mode = OS;

                /* falls through */

                case OS:
                  //=== NEEDBITS(16); */
                  while (bits < 16) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  if (state.head) {
                    state.head.xflags = hold & 0xff;
                    state.head.os = hold >> 8;
                  }

                  if (state.flags & 0x0200) {
                    //=== CRC2(state.check, hold);
                    hbuf[0] = hold & 0xff;
                    hbuf[1] = hold >>> 8 & 0xff;
                    state.check = crc32(state.check, hbuf, 2, 0); //===//
                  } //=== INITBITS();


                  hold = 0;
                  bits = 0; //===//

                  state.mode = EXLEN;

                /* falls through */

                case EXLEN:
                  if (state.flags & 0x0400) {
                    //=== NEEDBITS(16); */
                    while (bits < 16) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    state.length = hold;

                    if (state.head) {
                      state.head.extra_len = hold;
                    }

                    if (state.flags & 0x0200) {
                      //=== CRC2(state.check, hold);
                      hbuf[0] = hold & 0xff;
                      hbuf[1] = hold >>> 8 & 0xff;
                      state.check = crc32(state.check, hbuf, 2, 0); //===//
                    } //=== INITBITS();


                    hold = 0;
                    bits = 0; //===//
                  } else if (state.head) {
                    state.head.extra = null
                    /*Z_NULL*/
                    ;
                  }

                  state.mode = EXTRA;

                /* falls through */

                case EXTRA:
                  if (state.flags & 0x0400) {
                    copy = state.length;

                    if (copy > have) {
                      copy = have;
                    }

                    if (copy) {
                      if (state.head) {
                        len = state.head.extra_len - state.length;

                        if (!state.head.extra) {
                          // Use untyped array for more conveniend processing later
                          state.head.extra = new Array(state.head.extra_len);
                        }

                        utils.arraySet(state.head.extra, input, next, // extra field is limited to 65536 bytes
                        // - no need for additional size check
                        copy,
                        /*len + copy > state.head.extra_max - len ? state.head.extra_max : copy,*/
                        len); //zmemcpy(state.head.extra + len, next,
                        //        len + copy > state.head.extra_max ?
                        //        state.head.extra_max - len : copy);
                      }

                      if (state.flags & 0x0200) {
                        state.check = crc32(state.check, input, copy, next);
                      }

                      have -= copy;
                      next += copy;
                      state.length -= copy;
                    }

                    if (state.length) {
                      break inf_leave;
                    }
                  }

                  state.length = 0;
                  state.mode = NAME;

                /* falls through */

                case NAME:
                  if (state.flags & 0x0800) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    copy = 0;

                    do {
                      // TODO: 2 or 1 bytes?
                      len = input[next + copy++];
                      /* use constant limit because in js we should not preallocate memory */

                      if (state.head && len && state.length < 65536
                      /*state.head.name_max*/
                      ) {
                        state.head.name += String.fromCharCode(len);
                      }
                    } while (len && copy < have);

                    if (state.flags & 0x0200) {
                      state.check = crc32(state.check, input, copy, next);
                    }

                    have -= copy;
                    next += copy;

                    if (len) {
                      break inf_leave;
                    }
                  } else if (state.head) {
                    state.head.name = null;
                  }

                  state.length = 0;
                  state.mode = COMMENT;

                /* falls through */

                case COMMENT:
                  if (state.flags & 0x1000) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    copy = 0;

                    do {
                      len = input[next + copy++];
                      /* use constant limit because in js we should not preallocate memory */

                      if (state.head && len && state.length < 65536
                      /*state.head.comm_max*/
                      ) {
                        state.head.comment += String.fromCharCode(len);
                      }
                    } while (len && copy < have);

                    if (state.flags & 0x0200) {
                      state.check = crc32(state.check, input, copy, next);
                    }

                    have -= copy;
                    next += copy;

                    if (len) {
                      break inf_leave;
                    }
                  } else if (state.head) {
                    state.head.comment = null;
                  }

                  state.mode = HCRC;

                /* falls through */

                case HCRC:
                  if (state.flags & 0x0200) {
                    //=== NEEDBITS(16); */
                    while (bits < 16) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    if (hold !== (state.check & 0xffff)) {
                      strm.msg = 'header crc mismatch';
                      state.mode = BAD;
                      break;
                    } //=== INITBITS();


                    hold = 0;
                    bits = 0; //===//
                  }

                  if (state.head) {
                    state.head.hcrc = state.flags >> 9 & 1;
                    state.head.done = true;
                  }

                  strm.adler = state.check = 0
                  /*crc32(0L, Z_NULL, 0)*/
                  ;
                  state.mode = TYPE;
                  break;

                case DICTID:
                  //=== NEEDBITS(32); */
                  while (bits < 32) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  strm.adler = state.check = ZSWAP32(hold); //=== INITBITS();

                  hold = 0;
                  bits = 0; //===//

                  state.mode = DICT;

                /* falls through */

                case DICT:
                  if (state.havedict === 0) {
                    //--- RESTORE() ---
                    strm.next_out = put;
                    strm.avail_out = left;
                    strm.next_in = next;
                    strm.avail_in = have;
                    state.hold = hold;
                    state.bits = bits; //---

                    return Z_NEED_DICT;
                  }

                  strm.adler = state.check = 1
                  /*adler32(0L, Z_NULL, 0)*/
                  ;
                  state.mode = TYPE;

                /* falls through */

                case TYPE:
                  if (flush === Z_BLOCK || flush === Z_TREES) {
                    break inf_leave;
                  }

                /* falls through */

                case TYPEDO:
                  if (state.last) {
                    //--- BYTEBITS() ---//
                    hold >>>= bits & 7;
                    bits -= bits & 7; //---//

                    state.mode = CHECK;
                    break;
                  } //=== NEEDBITS(3); */


                  while (bits < 3) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  state.last = hold & 0x01
                  /*BITS(1)*/
                  ; //--- DROPBITS(1) ---//

                  hold >>>= 1;
                  bits -= 1; //---//

                  switch (hold & 0x03) {
                    /*BITS(2)*/
                    case 0:
                      /* stored block */
                      //Tracev((stderr, "inflate:     stored block%s\n",
                      //        state.last ? " (last)" : ""));
                      state.mode = STORED;
                      break;

                    case 1:
                      /* fixed block */
                      fixedtables(state); //Tracev((stderr, "inflate:     fixed codes block%s\n",
                      //        state.last ? " (last)" : ""));

                      state.mode = LEN_;
                      /* decode codes */

                      if (flush === Z_TREES) {
                        //--- DROPBITS(2) ---//
                        hold >>>= 2;
                        bits -= 2; //---//

                        break inf_leave;
                      }

                      break;

                    case 2:
                      /* dynamic block */
                      //Tracev((stderr, "inflate:     dynamic codes block%s\n",
                      //        state.last ? " (last)" : ""));
                      state.mode = TABLE;
                      break;

                    case 3:
                      strm.msg = 'invalid block type';
                      state.mode = BAD;
                  } //--- DROPBITS(2) ---//


                  hold >>>= 2;
                  bits -= 2; //---//

                  break;

                case STORED:
                  //--- BYTEBITS() ---// /* go to byte boundary */
                  hold >>>= bits & 7;
                  bits -= bits & 7; //---//
                  //=== NEEDBITS(32); */

                  while (bits < 32) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  if ((hold & 0xffff) !== (hold >>> 16 ^ 0xffff)) {
                    strm.msg = 'invalid stored block lengths';
                    state.mode = BAD;
                    break;
                  }

                  state.length = hold & 0xffff; //Tracev((stderr, "inflate:       stored length %u\n",
                  //        state.length));
                  //=== INITBITS();

                  hold = 0;
                  bits = 0; //===//

                  state.mode = COPY_;

                  if (flush === Z_TREES) {
                    break inf_leave;
                  }

                /* falls through */

                case COPY_:
                  state.mode = COPY;

                /* falls through */

                case COPY:
                  copy = state.length;

                  if (copy) {
                    if (copy > have) {
                      copy = have;
                    }

                    if (copy > left) {
                      copy = left;
                    }

                    if (copy === 0) {
                      break inf_leave;
                    } //--- zmemcpy(put, next, copy); ---


                    utils.arraySet(output, input, next, copy, put); //---//

                    have -= copy;
                    next += copy;
                    left -= copy;
                    put += copy;
                    state.length -= copy;
                    break;
                  } //Tracev((stderr, "inflate:       stored end\n"));


                  state.mode = TYPE;
                  break;

                case TABLE:
                  //=== NEEDBITS(14); */
                  while (bits < 14) {
                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  } //===//


                  state.nlen = (hold & 0x1f) +
                  /*BITS(5)*/
                  257; //--- DROPBITS(5) ---//

                  hold >>>= 5;
                  bits -= 5; //---//

                  state.ndist = (hold & 0x1f) +
                  /*BITS(5)*/
                  1; //--- DROPBITS(5) ---//

                  hold >>>= 5;
                  bits -= 5; //---//

                  state.ncode = (hold & 0x0f) +
                  /*BITS(4)*/
                  4; //--- DROPBITS(4) ---//

                  hold >>>= 4;
                  bits -= 4; //---//
                  //#ifndef PKZIP_BUG_WORKAROUND

                  if (state.nlen > 286 || state.ndist > 30) {
                    strm.msg = 'too many length or distance symbols';
                    state.mode = BAD;
                    break;
                  } //#endif
                  //Tracev((stderr, "inflate:       table sizes ok\n"));


                  state.have = 0;
                  state.mode = LENLENS;

                /* falls through */

                case LENLENS:
                  while (state.have < state.ncode) {
                    //=== NEEDBITS(3);
                    while (bits < 3) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    state.lens[order[state.have++]] = hold & 0x07; //BITS(3);
                    //--- DROPBITS(3) ---//

                    hold >>>= 3;
                    bits -= 3; //---//
                  }

                  while (state.have < 19) {
                    state.lens[order[state.have++]] = 0;
                  } // We have separate tables & no pointers. 2 commented lines below not needed.
                  //state.next = state.codes;
                  //state.lencode = state.next;
                  // Switch to use dynamic table


                  state.lencode = state.lendyn;
                  state.lenbits = 7;
                  opts = {
                    bits: state.lenbits
                  };
                  ret = inflate_table(CODES, state.lens, 0, 19, state.lencode, 0, state.work, opts);
                  state.lenbits = opts.bits;

                  if (ret) {
                    strm.msg = 'invalid code lengths set';
                    state.mode = BAD;
                    break;
                  } //Tracev((stderr, "inflate:       code lengths ok\n"));


                  state.have = 0;
                  state.mode = CODELENS;

                /* falls through */

                case CODELENS:
                  while (state.have < state.nlen + state.ndist) {
                    for (;;) {
                      here = state.lencode[hold & (1 << state.lenbits) - 1];
                      /*BITS(state.lenbits)*/

                      here_bits = here >>> 24;
                      here_op = here >>> 16 & 0xff;
                      here_val = here & 0xffff;

                      if (here_bits <= bits) {
                        break;
                      } //--- PULLBYTE() ---//


                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8; //---//
                    }

                    if (here_val < 16) {
                      //--- DROPBITS(here.bits) ---//
                      hold >>>= here_bits;
                      bits -= here_bits; //---//

                      state.lens[state.have++] = here_val;
                    } else {
                      if (here_val === 16) {
                        //=== NEEDBITS(here.bits + 2);
                        n = here_bits + 2;

                        while (bits < n) {
                          if (have === 0) {
                            break inf_leave;
                          }

                          have--;
                          hold += input[next++] << bits;
                          bits += 8;
                        } //===//
                        //--- DROPBITS(here.bits) ---//


                        hold >>>= here_bits;
                        bits -= here_bits; //---//

                        if (state.have === 0) {
                          strm.msg = 'invalid bit length repeat';
                          state.mode = BAD;
                          break;
                        }

                        len = state.lens[state.have - 1];
                        copy = 3 + (hold & 0x03); //BITS(2);
                        //--- DROPBITS(2) ---//

                        hold >>>= 2;
                        bits -= 2; //---//
                      } else if (here_val === 17) {
                        //=== NEEDBITS(here.bits + 3);
                        n = here_bits + 3;

                        while (bits < n) {
                          if (have === 0) {
                            break inf_leave;
                          }

                          have--;
                          hold += input[next++] << bits;
                          bits += 8;
                        } //===//
                        //--- DROPBITS(here.bits) ---//


                        hold >>>= here_bits;
                        bits -= here_bits; //---//

                        len = 0;
                        copy = 3 + (hold & 0x07); //BITS(3);
                        //--- DROPBITS(3) ---//

                        hold >>>= 3;
                        bits -= 3; //---//
                      } else {
                        //=== NEEDBITS(here.bits + 7);
                        n = here_bits + 7;

                        while (bits < n) {
                          if (have === 0) {
                            break inf_leave;
                          }

                          have--;
                          hold += input[next++] << bits;
                          bits += 8;
                        } //===//
                        //--- DROPBITS(here.bits) ---//


                        hold >>>= here_bits;
                        bits -= here_bits; //---//

                        len = 0;
                        copy = 11 + (hold & 0x7f); //BITS(7);
                        //--- DROPBITS(7) ---//

                        hold >>>= 7;
                        bits -= 7; //---//
                      }

                      if (state.have + copy > state.nlen + state.ndist) {
                        strm.msg = 'invalid bit length repeat';
                        state.mode = BAD;
                        break;
                      }

                      while (copy--) {
                        state.lens[state.have++] = len;
                      }
                    }
                  }
                  /* handle error breaks in while */


                  if (state.mode === BAD) {
                    break;
                  }
                  /* check for end-of-block code (better have one) */


                  if (state.lens[256] === 0) {
                    strm.msg = 'invalid code -- missing end-of-block';
                    state.mode = BAD;
                    break;
                  }
                  /* build code tables -- note: do not change the lenbits or distbits
                     values here (9 and 6) without reading the comments in inftrees.h
                     concerning the ENOUGH constants, which depend on those values */


                  state.lenbits = 9;
                  opts = {
                    bits: state.lenbits
                  };
                  ret = inflate_table(LENS, state.lens, 0, state.nlen, state.lencode, 0, state.work, opts); // We have separate tables & no pointers. 2 commented lines below not needed.
                  // state.next_index = opts.table_index;

                  state.lenbits = opts.bits; // state.lencode = state.next;

                  if (ret) {
                    strm.msg = 'invalid literal/lengths set';
                    state.mode = BAD;
                    break;
                  }

                  state.distbits = 6; //state.distcode.copy(state.codes);
                  // Switch to use dynamic table

                  state.distcode = state.distdyn;
                  opts = {
                    bits: state.distbits
                  };
                  ret = inflate_table(DISTS, state.lens, state.nlen, state.ndist, state.distcode, 0, state.work, opts); // We have separate tables & no pointers. 2 commented lines below not needed.
                  // state.next_index = opts.table_index;

                  state.distbits = opts.bits; // state.distcode = state.next;

                  if (ret) {
                    strm.msg = 'invalid distances set';
                    state.mode = BAD;
                    break;
                  } //Tracev((stderr, 'inflate:       codes ok\n'));


                  state.mode = LEN_;

                  if (flush === Z_TREES) {
                    break inf_leave;
                  }

                /* falls through */

                case LEN_:
                  state.mode = LEN;

                /* falls through */

                case LEN:
                  if (have >= 6 && left >= 258) {
                    //--- RESTORE() ---
                    strm.next_out = put;
                    strm.avail_out = left;
                    strm.next_in = next;
                    strm.avail_in = have;
                    state.hold = hold;
                    state.bits = bits; //---

                    inflate_fast(strm, _out); //--- LOAD() ---

                    put = strm.next_out;
                    output = strm.output;
                    left = strm.avail_out;
                    next = strm.next_in;
                    input = strm.input;
                    have = strm.avail_in;
                    hold = state.hold;
                    bits = state.bits; //---

                    if (state.mode === TYPE) {
                      state.back = -1;
                    }

                    break;
                  }

                  state.back = 0;

                  for (;;) {
                    here = state.lencode[hold & (1 << state.lenbits) - 1];
                    /*BITS(state.lenbits)*/

                    here_bits = here >>> 24;
                    here_op = here >>> 16 & 0xff;
                    here_val = here & 0xffff;

                    if (here_bits <= bits) {
                      break;
                    } //--- PULLBYTE() ---//


                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8; //---//
                  }

                  if (here_op && (here_op & 0xf0) === 0) {
                    last_bits = here_bits;
                    last_op = here_op;
                    last_val = here_val;

                    for (;;) {
                      here = state.lencode[last_val + ((hold & (1 << last_bits + last_op) - 1) >>
                      /*BITS(last.bits + last.op)*/
                      last_bits)];
                      here_bits = here >>> 24;
                      here_op = here >>> 16 & 0xff;
                      here_val = here & 0xffff;

                      if (last_bits + here_bits <= bits) {
                        break;
                      } //--- PULLBYTE() ---//


                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8; //---//
                    } //--- DROPBITS(last.bits) ---//


                    hold >>>= last_bits;
                    bits -= last_bits; //---//

                    state.back += last_bits;
                  } //--- DROPBITS(here.bits) ---//


                  hold >>>= here_bits;
                  bits -= here_bits; //---//

                  state.back += here_bits;
                  state.length = here_val;

                  if (here_op === 0) {
                    //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
                    //        "inflate:         literal '%c'\n" :
                    //        "inflate:         literal 0x%02x\n", here.val));
                    state.mode = LIT;
                    break;
                  }

                  if (here_op & 32) {
                    //Tracevv((stderr, "inflate:         end of block\n"));
                    state.back = -1;
                    state.mode = TYPE;
                    break;
                  }

                  if (here_op & 64) {
                    strm.msg = 'invalid literal/length code';
                    state.mode = BAD;
                    break;
                  }

                  state.extra = here_op & 15;
                  state.mode = LENEXT;

                /* falls through */

                case LENEXT:
                  if (state.extra) {
                    //=== NEEDBITS(state.extra);
                    n = state.extra;

                    while (bits < n) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    state.length += hold & (1 << state.extra) - 1
                    /*BITS(state.extra)*/
                    ; //--- DROPBITS(state.extra) ---//

                    hold >>>= state.extra;
                    bits -= state.extra; //---//

                    state.back += state.extra;
                  } //Tracevv((stderr, "inflate:         length %u\n", state.length));


                  state.was = state.length;
                  state.mode = DIST;

                /* falls through */

                case DIST:
                  for (;;) {
                    here = state.distcode[hold & (1 << state.distbits) - 1];
                    /*BITS(state.distbits)*/

                    here_bits = here >>> 24;
                    here_op = here >>> 16 & 0xff;
                    here_val = here & 0xffff;

                    if (here_bits <= bits) {
                      break;
                    } //--- PULLBYTE() ---//


                    if (have === 0) {
                      break inf_leave;
                    }

                    have--;
                    hold += input[next++] << bits;
                    bits += 8; //---//
                  }

                  if ((here_op & 0xf0) === 0) {
                    last_bits = here_bits;
                    last_op = here_op;
                    last_val = here_val;

                    for (;;) {
                      here = state.distcode[last_val + ((hold & (1 << last_bits + last_op) - 1) >>
                      /*BITS(last.bits + last.op)*/
                      last_bits)];
                      here_bits = here >>> 24;
                      here_op = here >>> 16 & 0xff;
                      here_val = here & 0xffff;

                      if (last_bits + here_bits <= bits) {
                        break;
                      } //--- PULLBYTE() ---//


                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8; //---//
                    } //--- DROPBITS(last.bits) ---//


                    hold >>>= last_bits;
                    bits -= last_bits; //---//

                    state.back += last_bits;
                  } //--- DROPBITS(here.bits) ---//


                  hold >>>= here_bits;
                  bits -= here_bits; //---//

                  state.back += here_bits;

                  if (here_op & 64) {
                    strm.msg = 'invalid distance code';
                    state.mode = BAD;
                    break;
                  }

                  state.offset = here_val;
                  state.extra = here_op & 15;
                  state.mode = DISTEXT;

                /* falls through */

                case DISTEXT:
                  if (state.extra) {
                    //=== NEEDBITS(state.extra);
                    n = state.extra;

                    while (bits < n) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    state.offset += hold & (1 << state.extra) - 1
                    /*BITS(state.extra)*/
                    ; //--- DROPBITS(state.extra) ---//

                    hold >>>= state.extra;
                    bits -= state.extra; //---//

                    state.back += state.extra;
                  } //#ifdef INFLATE_STRICT


                  if (state.offset > state.dmax) {
                    strm.msg = 'invalid distance too far back';
                    state.mode = BAD;
                    break;
                  } //#endif
                  //Tracevv((stderr, "inflate:         distance %u\n", state.offset));


                  state.mode = MATCH;

                /* falls through */

                case MATCH:
                  if (left === 0) {
                    break inf_leave;
                  }

                  copy = _out - left;

                  if (state.offset > copy) {
                    /* copy from window */
                    copy = state.offset - copy;

                    if (copy > state.whave) {
                      if (state.sane) {
                        strm.msg = 'invalid distance too far back';
                        state.mode = BAD;
                        break;
                      } // (!) This block is disabled in zlib defailts,
                      // don't enable it for binary compatibility
                      //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                      //          Trace((stderr, "inflate.c too far\n"));
                      //          copy -= state.whave;
                      //          if (copy > state.length) { copy = state.length; }
                      //          if (copy > left) { copy = left; }
                      //          left -= copy;
                      //          state.length -= copy;
                      //          do {
                      //            output[put++] = 0;
                      //          } while (--copy);
                      //          if (state.length === 0) { state.mode = LEN; }
                      //          break;
                      //#endif

                    }

                    if (copy > state.wnext) {
                      copy -= state.wnext;
                      from = state.wsize - copy;
                    } else {
                      from = state.wnext - copy;
                    }

                    if (copy > state.length) {
                      copy = state.length;
                    }

                    from_source = state.window;
                  } else {
                    /* copy from output */
                    from_source = output;
                    from = put - state.offset;
                    copy = state.length;
                  }

                  if (copy > left) {
                    copy = left;
                  }

                  left -= copy;
                  state.length -= copy;

                  do {
                    output[put++] = from_source[from++];
                  } while (--copy);

                  if (state.length === 0) {
                    state.mode = LEN;
                  }

                  break;

                case LIT:
                  if (left === 0) {
                    break inf_leave;
                  }

                  output[put++] = state.length;
                  left--;
                  state.mode = LEN;
                  break;

                case CHECK:
                  if (state.wrap) {
                    //=== NEEDBITS(32);
                    while (bits < 32) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--; // Use '|' insdead of '+' to make sure that result is signed

                      hold |= input[next++] << bits;
                      bits += 8;
                    } //===//


                    _out -= left;
                    strm.total_out += _out;
                    state.total += _out;

                    if (_out) {
                      strm.adler = state.check =
                      /*UPDATE(state.check, put - _out, _out);*/
                      state.flags ? crc32(state.check, output, _out, put - _out) : adler32(state.check, output, _out, put - _out);
                    }

                    _out = left; // NB: crc32 stored as signed 32-bit int, ZSWAP32 returns signed too

                    if ((state.flags ? hold : ZSWAP32(hold)) !== state.check) {
                      strm.msg = 'incorrect data check';
                      state.mode = BAD;
                      break;
                    } //=== INITBITS();


                    hold = 0;
                    bits = 0; //===//
                    //Tracev((stderr, "inflate:   check matches trailer\n"));
                  }

                  state.mode = LENGTH;

                /* falls through */

                case LENGTH:
                  if (state.wrap && state.flags) {
                    //=== NEEDBITS(32);
                    while (bits < 32) {
                      if (have === 0) {
                        break inf_leave;
                      }

                      have--;
                      hold += input[next++] << bits;
                      bits += 8;
                    } //===//


                    if (hold !== (state.total & 0xffffffff)) {
                      strm.msg = 'incorrect length check';
                      state.mode = BAD;
                      break;
                    } //=== INITBITS();


                    hold = 0;
                    bits = 0; //===//
                    //Tracev((stderr, "inflate:   length matches trailer\n"));
                  }

                  state.mode = DONE;

                /* falls through */

                case DONE:
                  ret = Z_STREAM_END;
                  break inf_leave;

                case BAD:
                  ret = Z_DATA_ERROR;
                  break inf_leave;

                case MEM:
                  return Z_MEM_ERROR;

                case SYNC:
                /* falls through */

                default:
                  return Z_STREAM_ERROR;
              }
            } // inf_leave <- here is real place for "goto inf_leave", emulated via "break inf_leave"

            /*
               Return from inflate(), updating the total counts and the check value.
               If there was no progress during the inflate() call, return a buffer
               error.  Call updatewindow() to create and/or update the window state.
               Note: a memory error from inflate() is non-recoverable.
             */
            //--- RESTORE() ---


            strm.next_out = put;
            strm.avail_out = left;
            strm.next_in = next;
            strm.avail_in = have;
            state.hold = hold;
            state.bits = bits; //---

            if (state.wsize || _out !== strm.avail_out && state.mode < BAD && (state.mode < CHECK || flush !== Z_FINISH)) {
              if (updatewindow(strm, strm.output, strm.next_out, _out - strm.avail_out)) ;
            }

            _in -= strm.avail_in;
            _out -= strm.avail_out;
            strm.total_in += _in;
            strm.total_out += _out;
            state.total += _out;

            if (state.wrap && _out) {
              strm.adler = state.check =
              /*UPDATE(state.check, strm.next_out - _out, _out);*/
              state.flags ? crc32(state.check, output, _out, strm.next_out - _out) : adler32(state.check, output, _out, strm.next_out - _out);
            }

            strm.data_type = state.bits + (state.last ? 64 : 0) + (state.mode === TYPE ? 128 : 0) + (state.mode === LEN_ || state.mode === COPY_ ? 256 : 0);

            if ((_in === 0 && _out === 0 || flush === Z_FINISH) && ret === Z_OK) {
              ret = Z_BUF_ERROR;
            }

            return ret;
          }

          function inflateEnd(strm) {
            if (!strm || !strm.state
            /*|| strm->zfree == (free_func)0*/
            ) {
                return Z_STREAM_ERROR;
              }

            var state = strm.state;

            if (state.window) {
              state.window = null;
            }

            strm.state = null;
            return Z_OK;
          }

          function inflateGetHeader(strm, head) {
            var state;
            /* check state */

            if (!strm || !strm.state) {
              return Z_STREAM_ERROR;
            }

            state = strm.state;

            if ((state.wrap & 2) === 0) {
              return Z_STREAM_ERROR;
            }
            /* save header structure */


            state.head = head;
            head.done = false;
            return Z_OK;
          }

          exports.inflateReset = inflateReset;
          exports.inflateReset2 = inflateReset2;
          exports.inflateResetKeep = inflateResetKeep;
          exports.inflateInit = inflateInit;
          exports.inflateInit2 = inflateInit2;
          exports.inflate = inflate;
          exports.inflateEnd = inflateEnd;
          exports.inflateGetHeader = inflateGetHeader;
          exports.inflateInfo = 'pako inflate (from Nodeca project)';
          /* Not implemented
          exports.inflateCopy = inflateCopy;
          exports.inflateGetDictionary = inflateGetDictionary;
          exports.inflateMark = inflateMark;
          exports.inflatePrime = inflatePrime;
          exports.inflateSetDictionary = inflateSetDictionary;
          exports.inflateSync = inflateSync;
          exports.inflateSyncPoint = inflateSyncPoint;
          exports.inflateUndermine = inflateUndermine;
          */
        }, {
          "../utils/common": 27,
          "./adler32": 29,
          "./crc32": 31,
          "./inffast": 34,
          "./inftrees": 36
        }],
        36: [function (_dereq_, module, exports) {

          var utils = _dereq_('../utils/common');

          var MAXBITS = 15;
          var ENOUGH_LENS = 852;
          var ENOUGH_DISTS = 592; //var ENOUGH = (ENOUGH_LENS+ENOUGH_DISTS);

          var CODES = 0;
          var LENS = 1;
          var DISTS = 2;
          var lbase = [
          /* Length codes 257..285 base */
          3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0];
          var lext = [
          /* Length codes 257..285 extra */
          16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78];
          var dbase = [
          /* Distance codes 0..29 base */
          1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0];
          var dext = [
          /* Distance codes 0..29 extra */
          16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];

          module.exports = function inflate_table(type, lens, lens_index, codes, table, table_index, work, opts) {
            var bits = opts.bits; //here = opts.here; /* table entry for duplication */

            var len = 0;
            /* a code's length in bits */

            var sym = 0;
            /* index of code symbols */

            var min = 0,
                max = 0;
            /* minimum and maximum code lengths */

            var root = 0;
            /* number of index bits for root table */

            var curr = 0;
            /* number of index bits for current table */

            var drop = 0;
            /* code bits to drop for sub-table */

            var left = 0;
            /* number of prefix codes available */

            var used = 0;
            /* code entries in table used */

            var huff = 0;
            /* Huffman code */

            var incr;
            /* for incrementing code, index */

            var fill;
            /* index for replicating entries */

            var low;
            /* low bits for current root entry */

            var mask;
            /* mask for low root bits */

            var next;
            /* next available space in table */

            var base = null;
            /* base value table to use */

            var base_index = 0; //  var shoextra;    /* extra bits table to use */

            var end;
            /* use base and extra for symbol > end */

            var count = new utils.Buf16(MAXBITS + 1); //[MAXBITS+1];    /* number of codes of each length */

            var offs = new utils.Buf16(MAXBITS + 1); //[MAXBITS+1];     /* offsets in table for each length */

            var extra = null;
            var extra_index = 0;
            var here_bits, here_op, here_val;
            /*
             Process a set of code lengths to create a canonical Huffman code.  The
             code lengths are lens[0..codes-1].  Each length corresponds to the
             symbols 0..codes-1.  The Huffman code is generated by first sorting the
             symbols by length from short to long, and retaining the symbol order
             for codes with equal lengths.  Then the code starts with all zero bits
             for the first code of the shortest length, and the codes are integer
             increments for the same length, and zeros are appended as the length
             increases.  For the deflate format, these bits are stored backwards
             from their more natural integer increment ordering, and so when the
             decoding tables are built in the large loop below, the integer codes
             are incremented backwards.
              This routine assumes, but does not check, that all of the entries in
             lens[] are in the range 0..MAXBITS.  The caller must assure this.
             1..MAXBITS is interpreted as that code length.  zero means that that
             symbol does not occur in this code.
              The codes are sorted by computing a count of codes for each length,
             creating from that a table of starting indices for each length in the
             sorted table, and then entering the symbols in order in the sorted
             table.  The sorted table is work[], with that space being provided by
             the caller.
              The length counts are used for other purposes as well, i.e. finding
             the minimum and maximum length codes, determining if there are any
             codes at all, checking for a valid set of lengths, and looking ahead
             at length counts to determine sub-table sizes when building the
             decoding tables.
             */

            /* accumulate lengths for codes (assumes lens[] all in 0..MAXBITS) */

            for (len = 0; len <= MAXBITS; len++) {
              count[len] = 0;
            }

            for (sym = 0; sym < codes; sym++) {
              count[lens[lens_index + sym]]++;
            }
            /* bound code lengths, force root to be within code lengths */


            root = bits;

            for (max = MAXBITS; max >= 1; max--) {
              if (count[max] !== 0) {
                break;
              }
            }

            if (root > max) {
              root = max;
            }

            if (max === 0) {
              /* no symbols to code at all */
              //table.op[opts.table_index] = 64;  //here.op = (var char)64;    /* invalid code marker */
              //table.bits[opts.table_index] = 1;   //here.bits = (var char)1;
              //table.val[opts.table_index++] = 0;   //here.val = (var short)0;
              table[table_index++] = 1 << 24 | 64 << 16 | 0; //table.op[opts.table_index] = 64;
              //table.bits[opts.table_index] = 1;
              //table.val[opts.table_index++] = 0;

              table[table_index++] = 1 << 24 | 64 << 16 | 0;
              opts.bits = 1;
              return 0;
              /* no symbols, but wait for decoding to report error */
            }

            for (min = 1; min < max; min++) {
              if (count[min] !== 0) {
                break;
              }
            }

            if (root < min) {
              root = min;
            }
            /* check for an over-subscribed or incomplete set of lengths */


            left = 1;

            for (len = 1; len <= MAXBITS; len++) {
              left <<= 1;
              left -= count[len];

              if (left < 0) {
                return -1;
              }
              /* over-subscribed */

            }

            if (left > 0 && (type === CODES || max !== 1)) {
              return -1;
              /* incomplete set */
            }
            /* generate offsets into symbol table for each length for sorting */


            offs[1] = 0;

            for (len = 1; len < MAXBITS; len++) {
              offs[len + 1] = offs[len] + count[len];
            }
            /* sort symbols by length, by symbol order within each length */


            for (sym = 0; sym < codes; sym++) {
              if (lens[lens_index + sym] !== 0) {
                work[offs[lens[lens_index + sym]]++] = sym;
              }
            }
            /*
             Create and fill in decoding tables.  In this loop, the table being
             filled is at next and has curr index bits.  The code being used is huff
             with length len.  That code is converted to an index by dropping drop
             bits off of the bottom.  For codes where len is less than drop + curr,
             those top drop + curr - len bits are incremented through all values to
             fill the table with replicated entries.
              root is the number of index bits for the root table.  When len exceeds
             root, sub-tables are created pointed to by the root entry with an index
             of the low root bits of huff.  This is saved in low to check for when a
             new sub-table should be started.  drop is zero when the root table is
             being filled, and drop is root when sub-tables are being filled.
              When a new sub-table is needed, it is necessary to look ahead in the
             code lengths to determine what size sub-table is needed.  The length
             counts are used for this, and so count[] is decremented as codes are
             entered in the tables.
              used keeps track of how many table entries have been allocated from the
             provided *table space.  It is checked for LENS and DIST tables against
             the constants ENOUGH_LENS and ENOUGH_DISTS to guard against changes in
             the initial root table size constants.  See the comments in inftrees.h
             for more information.
              sym increments through all symbols, and the loop terminates when
             all codes of length max, i.e. all codes, have been processed.  This
             routine permits incomplete codes, so another loop after this one fills
             in the rest of the decoding tables with invalid code markers.
             */

            /* set up for code type */
            // poor man optimization - use if-else instead of switch,
            // to avoid deopts in old v8


            if (type === CODES) {
              base = extra = work;
              /* dummy value--not used */

              end = 19;
            } else if (type === LENS) {
              base = lbase;
              base_index -= 257;
              extra = lext;
              extra_index -= 257;
              end = 256;
            } else {
              /* DISTS */
              base = dbase;
              extra = dext;
              end = -1;
            }
            /* initialize opts for loop */


            huff = 0;
            /* starting code */

            sym = 0;
            /* starting code symbol */

            len = min;
            /* starting code length */

            next = table_index;
            /* current table to fill in */

            curr = root;
            /* current table index bits */

            drop = 0;
            /* current bits to drop from code for index */

            low = -1;
            /* trigger new sub-table when len > root */

            used = 1 << root;
            /* use root table entries */

            mask = used - 1;
            /* mask for comparing low */

            /* check available table space */

            if (type === LENS && used > ENOUGH_LENS || type === DISTS && used > ENOUGH_DISTS) {
              return 1;
            }
            /* process all codes and make table entries */

            for (;;) {
              /* create table entry */

              here_bits = len - drop;

              if (work[sym] < end) {
                here_op = 0;
                here_val = work[sym];
              } else if (work[sym] > end) {
                here_op = extra[extra_index + work[sym]];
                here_val = base[base_index + work[sym]];
              } else {
                here_op = 32 + 64;
                /* end of block */

                here_val = 0;
              }
              /* replicate for those indices with low len bits equal to huff */


              incr = 1 << len - drop;
              fill = 1 << curr;
              min = fill;
              /* save offset to next table */

              do {
                fill -= incr;
                table[next + (huff >> drop) + fill] = here_bits << 24 | here_op << 16 | here_val | 0;
              } while (fill !== 0);
              /* backwards increment the len-bit code huff */


              incr = 1 << len - 1;

              while (huff & incr) {
                incr >>= 1;
              }

              if (incr !== 0) {
                huff &= incr - 1;
                huff += incr;
              } else {
                huff = 0;
              }
              /* go to next symbol, update count, len */


              sym++;

              if (--count[len] === 0) {
                if (len === max) {
                  break;
                }

                len = lens[lens_index + work[sym]];
              }
              /* create new sub-table if needed */


              if (len > root && (huff & mask) !== low) {
                /* if first time, transition to sub-tables */
                if (drop === 0) {
                  drop = root;
                }
                /* increment past last table */


                next += min;
                /* here min is 1 << curr */

                /* determine length of next table */

                curr = len - drop;
                left = 1 << curr;

                while (curr + drop < max) {
                  left -= count[curr + drop];

                  if (left <= 0) {
                    break;
                  }

                  curr++;
                  left <<= 1;
                }
                /* check for enough space */


                used += 1 << curr;

                if (type === LENS && used > ENOUGH_LENS || type === DISTS && used > ENOUGH_DISTS) {
                  return 1;
                }
                /* point entry in root table to sub-table */


                low = huff & mask;
                /*table.op[low] = curr;
                table.bits[low] = root;
                table.val[low] = next - opts.table_index;*/

                table[low] = root << 24 | curr << 16 | next - table_index | 0;
              }
            }
            /* fill in remaining table entry if code is incomplete (guaranteed to have
             at most one remaining entry, since if the code is incomplete, the
             maximum code length that was allowed to get this far is one bit) */


            if (huff !== 0) {
              //table.op[next + huff] = 64;            /* invalid code marker */
              //table.bits[next + huff] = len - drop;
              //table.val[next + huff] = 0;
              table[next + huff] = len - drop << 24 | 64 << 16 | 0;
            }
            /* set return parameters */
            //opts.table_index += used;


            opts.bits = root;
            return 0;
          };
        }, {
          "../utils/common": 27
        }],
        37: [function (_dereq_, module, exports) {

          module.exports = {
            '2': 'need dictionary',

            /* Z_NEED_DICT       2  */
            '1': 'stream end',

            /* Z_STREAM_END      1  */
            '0': '',

            /* Z_OK              0  */
            '-1': 'file error',

            /* Z_ERRNO         (-1) */
            '-2': 'stream error',

            /* Z_STREAM_ERROR  (-2) */
            '-3': 'data error',

            /* Z_DATA_ERROR    (-3) */
            '-4': 'insufficient memory',

            /* Z_MEM_ERROR     (-4) */
            '-5': 'buffer error',

            /* Z_BUF_ERROR     (-5) */
            '-6': 'incompatible version'
            /* Z_VERSION_ERROR (-6) */

          };
        }, {}],
        38: [function (_dereq_, module, exports) {

          var utils = _dereq_('../utils/common');
          /* Public constants ==========================================================*/

          /* ===========================================================================*/
          //var Z_FILTERED          = 1;
          //var Z_HUFFMAN_ONLY      = 2;
          //var Z_RLE               = 3;


          var Z_FIXED = 4; //var Z_DEFAULT_STRATEGY  = 0;

          /* Possible values of the data_type field (though see inflate()) */

          var Z_BINARY = 0;
          var Z_TEXT = 1; //var Z_ASCII             = 1; // = Z_TEXT

          var Z_UNKNOWN = 2;
          /*============================================================================*/

          function zero(buf) {
            var len = buf.length;

            while (--len >= 0) {
              buf[len] = 0;
            }
          } // From zutil.h


          var STORED_BLOCK = 0;
          var STATIC_TREES = 1;
          var DYN_TREES = 2;
          /* The three kinds of block type */

          var MIN_MATCH = 3;
          var MAX_MATCH = 258;
          /* The minimum and maximum match lengths */
          // From deflate.h

          /* ===========================================================================
           * Internal compression state.
           */

          var LENGTH_CODES = 29;
          /* number of length codes, not counting the special END_BLOCK code */

          var LITERALS = 256;
          /* number of literal bytes 0..255 */

          var L_CODES = LITERALS + 1 + LENGTH_CODES;
          /* number of Literal or Length codes, including the END_BLOCK code */

          var D_CODES = 30;
          /* number of distance codes */

          var BL_CODES = 19;
          /* number of codes used to transfer the bit lengths */

          var HEAP_SIZE = 2 * L_CODES + 1;
          /* maximum heap size */

          var MAX_BITS = 15;
          /* All codes must not exceed MAX_BITS bits */

          var Buf_size = 16;
          /* size of bit buffer in bi_buf */

          /* ===========================================================================
           * Constants
           */

          var MAX_BL_BITS = 7;
          /* Bit length codes must not exceed MAX_BL_BITS bits */

          var END_BLOCK = 256;
          /* end of block literal code */

          var REP_3_6 = 16;
          /* repeat previous bit length 3-6 times (2 bits of repeat count) */

          var REPZ_3_10 = 17;
          /* repeat a zero length 3-10 times  (3 bits of repeat count) */

          var REPZ_11_138 = 18;
          /* repeat a zero length 11-138 times  (7 bits of repeat count) */

          var extra_lbits =
          /* extra bits for each length code */
          [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0];
          var extra_dbits =
          /* extra bits for each distance code */
          [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13];
          var extra_blbits =
          /* extra bits for each bit length code */
          [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7];
          var bl_order = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];
          /* The lengths of the bit length codes are sent in order of decreasing
           * probability, to avoid transmitting the lengths for unused bit length codes.
           */

          /* ===========================================================================
           * Local data. These are initialized only once.
           */
          // We pre-fill arrays with 0 to avoid uninitialized gaps

          var DIST_CODE_LEN = 512;
          /* see definition of array dist_code below */
          // !!!! Use flat array insdead of structure, Freq = i*2, Len = i*2+1

          var static_ltree = new Array((L_CODES + 2) * 2);
          zero(static_ltree);
          /* The static literal tree. Since the bit lengths are imposed, there is no
           * need for the L_CODES extra codes used during heap construction. However
           * The codes 286 and 287 are needed to build a canonical tree (see _tr_init
           * below).
           */

          var static_dtree = new Array(D_CODES * 2);
          zero(static_dtree);
          /* The static distance tree. (Actually a trivial tree since all codes use
           * 5 bits.)
           */

          var _dist_code = new Array(DIST_CODE_LEN);

          zero(_dist_code);
          /* Distance codes. The first 256 values correspond to the distances
           * 3 .. 258, the last 256 values correspond to the top 8 bits of
           * the 15 bit distances.
           */

          var _length_code = new Array(MAX_MATCH - MIN_MATCH + 1);

          zero(_length_code);
          /* length code for each normalized match length (0 == MIN_MATCH) */

          var base_length = new Array(LENGTH_CODES);
          zero(base_length);
          /* First normalized length for each code (0 = MIN_MATCH) */

          var base_dist = new Array(D_CODES);
          zero(base_dist);
          /* First normalized distance for each code (0 = distance of 1) */

          var StaticTreeDesc = function (static_tree, extra_bits, extra_base, elems, max_length) {
            this.static_tree = static_tree;
            /* static tree or NULL */

            this.extra_bits = extra_bits;
            /* extra bits for each code or NULL */

            this.extra_base = extra_base;
            /* base index for extra_bits */

            this.elems = elems;
            /* max number of elements in the tree */

            this.max_length = max_length;
            /* max bit length for the codes */
            // show if `static_tree` has data or dummy - needed for monomorphic objects

            this.has_stree = static_tree && static_tree.length;
          };

          var static_l_desc;
          var static_d_desc;
          var static_bl_desc;

          var TreeDesc = function (dyn_tree, stat_desc) {
            this.dyn_tree = dyn_tree;
            /* the dynamic tree */

            this.max_code = 0;
            /* largest code with non zero frequency */

            this.stat_desc = stat_desc;
            /* the corresponding static tree */
          };

          function d_code(dist) {
            return dist < 256 ? _dist_code[dist] : _dist_code[256 + (dist >>> 7)];
          }
          /* ===========================================================================
           * Output a short LSB first on the stream.
           * IN assertion: there is enough room in pendingBuf.
           */


          function put_short(s, w) {
            //    put_byte(s, (uch)((w) & 0xff));
            //    put_byte(s, (uch)((ush)(w) >> 8));
            s.pending_buf[s.pending++] = w & 0xff;
            s.pending_buf[s.pending++] = w >>> 8 & 0xff;
          }
          /* ===========================================================================
           * Send a value on a given number of bits.
           * IN assertion: length <= 16 and value fits in length bits.
           */


          function send_bits(s, value, length) {
            if (s.bi_valid > Buf_size - length) {
              s.bi_buf |= value << s.bi_valid & 0xffff;
              put_short(s, s.bi_buf);
              s.bi_buf = value >> Buf_size - s.bi_valid;
              s.bi_valid += length - Buf_size;
            } else {
              s.bi_buf |= value << s.bi_valid & 0xffff;
              s.bi_valid += length;
            }
          }

          function send_code(s, c, tree) {
            send_bits(s, tree[c * 2]
            /*.Code*/
            , tree[c * 2 + 1]
            /*.Len*/
            );
          }
          /* ===========================================================================
           * Reverse the first len bits of a code, using straightforward code (a faster
           * method would use a table)
           * IN assertion: 1 <= len <= 15
           */


          function bi_reverse(code, len) {
            var res = 0;

            do {
              res |= code & 1;
              code >>>= 1;
              res <<= 1;
            } while (--len > 0);

            return res >>> 1;
          }
          /* ===========================================================================
           * Flush the bit buffer, keeping at most 7 bits in it.
           */


          function bi_flush(s) {
            if (s.bi_valid === 16) {
              put_short(s, s.bi_buf);
              s.bi_buf = 0;
              s.bi_valid = 0;
            } else if (s.bi_valid >= 8) {
              s.pending_buf[s.pending++] = s.bi_buf & 0xff;
              s.bi_buf >>= 8;
              s.bi_valid -= 8;
            }
          }
          /* ===========================================================================
           * Compute the optimal bit lengths for a tree and update the total bit length
           * for the current block.
           * IN assertion: the fields freq and dad are set, heap[heap_max] and
           *    above are the tree nodes sorted by increasing frequency.
           * OUT assertions: the field len is set to the optimal bit length, the
           *     array bl_count contains the frequencies for each bit length.
           *     The length opt_len is updated; static_len is also updated if stree is
           *     not null.
           */


          function gen_bitlen(s, desc) //    deflate_state *s;
          //    tree_desc *desc;    /* the tree descriptor */
          {
            var tree = desc.dyn_tree;
            var max_code = desc.max_code;
            var stree = desc.stat_desc.static_tree;
            var has_stree = desc.stat_desc.has_stree;
            var extra = desc.stat_desc.extra_bits;
            var base = desc.stat_desc.extra_base;
            var max_length = desc.stat_desc.max_length;
            var h;
            /* heap index */

            var n, m;
            /* iterate over the tree elements */

            var bits;
            /* bit length */

            var xbits;
            /* extra bits */

            var f;
            /* frequency */

            var overflow = 0;
            /* number of elements with bit length too large */

            for (bits = 0; bits <= MAX_BITS; bits++) {
              s.bl_count[bits] = 0;
            }
            /* In a first pass, compute the optimal bit lengths (which may
             * overflow in the case of the bit length tree).
             */


            tree[s.heap[s.heap_max] * 2 + 1]
            /*.Len*/
            = 0;
            /* root of the heap */

            for (h = s.heap_max + 1; h < HEAP_SIZE; h++) {
              n = s.heap[h];
              bits = tree[tree[n * 2 + 1]
              /*.Dad*/
              * 2 + 1]
              /*.Len*/
              + 1;

              if (bits > max_length) {
                bits = max_length;
                overflow++;
              }

              tree[n * 2 + 1]
              /*.Len*/
              = bits;
              /* We overwrite tree[n].Dad which is no longer needed */

              if (n > max_code) {
                continue;
              }
              /* not a leaf node */


              s.bl_count[bits]++;
              xbits = 0;

              if (n >= base) {
                xbits = extra[n - base];
              }

              f = tree[n * 2]
              /*.Freq*/
              ;
              s.opt_len += f * (bits + xbits);

              if (has_stree) {
                s.static_len += f * (stree[n * 2 + 1]
                /*.Len*/
                + xbits);
              }
            }

            if (overflow === 0) {
              return;
            } // Trace((stderr,"\nbit length overflow\n"));

            /* This happens for example on obj2 and pic of the Calgary corpus */

            /* Find the first bit length which could increase: */


            do {
              bits = max_length - 1;

              while (s.bl_count[bits] === 0) {
                bits--;
              }

              s.bl_count[bits]--;
              /* move one leaf down the tree */

              s.bl_count[bits + 1] += 2;
              /* move one overflow item as its brother */

              s.bl_count[max_length]--;
              /* The brother of the overflow item also moves one step up,
               * but this does not affect bl_count[max_length]
               */

              overflow -= 2;
            } while (overflow > 0);
            /* Now recompute all bit lengths, scanning in increasing frequency.
             * h is still equal to HEAP_SIZE. (It is simpler to reconstruct all
             * lengths instead of fixing only the wrong ones. This idea is taken
             * from 'ar' written by Haruhiko Okumura.)
             */


            for (bits = max_length; bits !== 0; bits--) {
              n = s.bl_count[bits];

              while (n !== 0) {
                m = s.heap[--h];

                if (m > max_code) {
                  continue;
                }

                if (tree[m * 2 + 1]
                /*.Len*/
                !== bits) {
                  // Trace((stderr,"code %d bits %d->%d\n", m, tree[m].Len, bits));
                  s.opt_len += (bits - tree[m * 2 + 1]
                  /*.Len*/
                  ) * tree[m * 2]
                  /*.Freq*/
                  ;
                  tree[m * 2 + 1]
                  /*.Len*/
                  = bits;
                }

                n--;
              }
            }
          }
          /* ===========================================================================
           * Generate the codes for a given tree and bit counts (which need not be
           * optimal).
           * IN assertion: the array bl_count contains the bit length statistics for
           * the given tree and the field len is set for all tree elements.
           * OUT assertion: the field code is set for all tree elements of non
           *     zero code length.
           */


          function gen_codes(tree, max_code, bl_count) //    ct_data *tree;             /* the tree to decorate */
          //    int max_code;              /* largest code with non zero frequency */
          //    ushf *bl_count;            /* number of codes at each bit length */
          {
            var next_code = new Array(MAX_BITS + 1);
            /* next code value for each bit length */

            var code = 0;
            /* running code value */

            var bits;
            /* bit index */

            var n;
            /* code index */

            /* The distribution counts are first used to generate the code values
             * without bit reversal.
             */

            for (bits = 1; bits <= MAX_BITS; bits++) {
              next_code[bits] = code = code + bl_count[bits - 1] << 1;
            }
            /* Check that the bit counts in bl_count are consistent. The last code
             * must be all ones.
             */
            //Assert (code + bl_count[MAX_BITS]-1 == (1<<MAX_BITS)-1,
            //        "inconsistent bit counts");
            //Tracev((stderr,"\ngen_codes: max_code %d ", max_code));


            for (n = 0; n <= max_code; n++) {
              var len = tree[n * 2 + 1]
              /*.Len*/
              ;

              if (len === 0) {
                continue;
              }
              /* Now reverse the bits */


              tree[n * 2]
              /*.Code*/
              = bi_reverse(next_code[len]++, len); //Tracecv(tree != static_ltree, (stderr,"\nn %3d %c l %2d c %4x (%x) ",
              //     n, (isgraph(n) ? n : ' '), len, tree[n].Code, next_code[len]-1));
            }
          }
          /* ===========================================================================
           * Initialize the various 'constant' tables.
           */


          function tr_static_init() {
            var n;
            /* iterates over tree elements */

            var bits;
            /* bit counter */

            var length;
            /* length value */

            var code;
            /* code value */

            var dist;
            /* distance index */

            var bl_count = new Array(MAX_BITS + 1);
            /* number of codes at each bit length for an optimal tree */
            // do check in _tr_init()
            //if (static_init_done) return;

            /* For some embedded targets, global variables are not initialized: */

            /*#ifdef NO_INIT_GLOBAL_POINTERS
              static_l_desc.static_tree = static_ltree;
              static_l_desc.extra_bits = extra_lbits;
              static_d_desc.static_tree = static_dtree;
              static_d_desc.extra_bits = extra_dbits;
              static_bl_desc.extra_bits = extra_blbits;
            #endif*/

            /* Initialize the mapping length (0..255) -> length code (0..28) */

            length = 0;

            for (code = 0; code < LENGTH_CODES - 1; code++) {
              base_length[code] = length;

              for (n = 0; n < 1 << extra_lbits[code]; n++) {
                _length_code[length++] = code;
              }
            } //Assert (length == 256, "tr_static_init: length != 256");

            /* Note that the length 255 (match length 258) can be represented
             * in two different ways: code 284 + 5 bits or code 285, so we
             * overwrite length_code[255] to use the best encoding:
             */


            _length_code[length - 1] = code;
            /* Initialize the mapping dist (0..32K) -> dist code (0..29) */

            dist = 0;

            for (code = 0; code < 16; code++) {
              base_dist[code] = dist;

              for (n = 0; n < 1 << extra_dbits[code]; n++) {
                _dist_code[dist++] = code;
              }
            } //Assert (dist == 256, "tr_static_init: dist != 256");


            dist >>= 7;
            /* from now on, all distances are divided by 128 */

            for (; code < D_CODES; code++) {
              base_dist[code] = dist << 7;

              for (n = 0; n < 1 << extra_dbits[code] - 7; n++) {
                _dist_code[256 + dist++] = code;
              }
            } //Assert (dist == 256, "tr_static_init: 256+dist != 512");

            /* Construct the codes of the static literal tree */


            for (bits = 0; bits <= MAX_BITS; bits++) {
              bl_count[bits] = 0;
            }

            n = 0;

            while (n <= 143) {
              static_ltree[n * 2 + 1]
              /*.Len*/
              = 8;
              n++;
              bl_count[8]++;
            }

            while (n <= 255) {
              static_ltree[n * 2 + 1]
              /*.Len*/
              = 9;
              n++;
              bl_count[9]++;
            }

            while (n <= 279) {
              static_ltree[n * 2 + 1]
              /*.Len*/
              = 7;
              n++;
              bl_count[7]++;
            }

            while (n <= 287) {
              static_ltree[n * 2 + 1]
              /*.Len*/
              = 8;
              n++;
              bl_count[8]++;
            }
            /* Codes 286 and 287 do not exist, but we must include them in the
             * tree construction to get a canonical Huffman tree (longest code
             * all ones)
             */


            gen_codes(static_ltree, L_CODES + 1, bl_count);
            /* The static distance tree is trivial: */

            for (n = 0; n < D_CODES; n++) {
              static_dtree[n * 2 + 1]
              /*.Len*/
              = 5;
              static_dtree[n * 2]
              /*.Code*/
              = bi_reverse(n, 5);
            } // Now data ready and we can init static trees


            static_l_desc = new StaticTreeDesc(static_ltree, extra_lbits, LITERALS + 1, L_CODES, MAX_BITS);
            static_d_desc = new StaticTreeDesc(static_dtree, extra_dbits, 0, D_CODES, MAX_BITS);
            static_bl_desc = new StaticTreeDesc(new Array(0), extra_blbits, 0, BL_CODES, MAX_BL_BITS); //static_init_done = true;
          }
          /* ===========================================================================
           * Initialize a new block.
           */


          function init_block(s) {
            var n;
            /* iterates over tree elements */

            /* Initialize the trees. */

            for (n = 0; n < L_CODES; n++) {
              s.dyn_ltree[n * 2]
              /*.Freq*/
              = 0;
            }

            for (n = 0; n < D_CODES; n++) {
              s.dyn_dtree[n * 2]
              /*.Freq*/
              = 0;
            }

            for (n = 0; n < BL_CODES; n++) {
              s.bl_tree[n * 2]
              /*.Freq*/
              = 0;
            }

            s.dyn_ltree[END_BLOCK * 2]
            /*.Freq*/
            = 1;
            s.opt_len = s.static_len = 0;
            s.last_lit = s.matches = 0;
          }
          /* ===========================================================================
           * Flush the bit buffer and align the output on a byte boundary
           */


          function bi_windup(s) {
            if (s.bi_valid > 8) {
              put_short(s, s.bi_buf);
            } else if (s.bi_valid > 0) {
              //put_byte(s, (Byte)s->bi_buf);
              s.pending_buf[s.pending++] = s.bi_buf;
            }

            s.bi_buf = 0;
            s.bi_valid = 0;
          }
          /* ===========================================================================
           * Copy a stored block, storing first the length and its
           * one's complement if requested.
           */


          function copy_block(s, buf, len, header) //DeflateState *s;
          //charf    *buf;    /* the input data */
          //unsigned len;     /* its length */
          //int      header;  /* true if block header must be written */
          {
            bi_windup(s);
            /* align on byte boundary */

            if (header) {
              put_short(s, len);
              put_short(s, ~len);
            } //  while (len--) {
            //    put_byte(s, *buf++);
            //  }


            utils.arraySet(s.pending_buf, s.window, buf, len, s.pending);
            s.pending += len;
          }
          /* ===========================================================================
           * Compares to subtrees, using the tree depth as tie breaker when
           * the subtrees have equal frequency. This minimizes the worst case length.
           */


          function smaller(tree, n, m, depth) {
            var _n2 = n * 2;

            var _m2 = m * 2;

            return tree[_n2]
            /*.Freq*/
            < tree[_m2]
            /*.Freq*/
            || tree[_n2]
            /*.Freq*/
            === tree[_m2]
            /*.Freq*/
            && depth[n] <= depth[m];
          }
          /* ===========================================================================
           * Restore the heap property by moving down the tree starting at node k,
           * exchanging a node with the smallest of its two sons if necessary, stopping
           * when the heap property is re-established (each father smaller than its
           * two sons).
           */


          function pqdownheap(s, tree, k) //    deflate_state *s;
          //    ct_data *tree;  /* the tree to restore */
          //    int k;               /* node to move down */
          {
            var v = s.heap[k];
            var j = k << 1;
            /* left son of k */

            while (j <= s.heap_len) {
              /* Set j to the smallest of the two sons: */
              if (j < s.heap_len && smaller(tree, s.heap[j + 1], s.heap[j], s.depth)) {
                j++;
              }
              /* Exit if v is smaller than both sons */


              if (smaller(tree, v, s.heap[j], s.depth)) {
                break;
              }
              /* Exchange v with the smallest son */


              s.heap[k] = s.heap[j];
              k = j;
              /* And continue down the tree, setting j to the left son of k */

              j <<= 1;
            }

            s.heap[k] = v;
          } // inlined manually
          // var SMALLEST = 1;

          /* ===========================================================================
           * Send the block data compressed using the given Huffman trees
           */


          function compress_block(s, ltree, dtree) //    deflate_state *s;
          //    const ct_data *ltree; /* literal tree */
          //    const ct_data *dtree; /* distance tree */
          {
            var dist;
            /* distance of matched string */

            var lc;
            /* match length or unmatched char (if dist == 0) */

            var lx = 0;
            /* running index in l_buf */

            var code;
            /* the code to send */

            var extra;
            /* number of extra bits to send */

            if (s.last_lit !== 0) {
              do {
                dist = s.pending_buf[s.d_buf + lx * 2] << 8 | s.pending_buf[s.d_buf + lx * 2 + 1];
                lc = s.pending_buf[s.l_buf + lx];
                lx++;

                if (dist === 0) {
                  send_code(s, lc, ltree);
                  /* send a literal byte */
                  //Tracecv(isgraph(lc), (stderr," '%c' ", lc));
                } else {
                  /* Here, lc is the match length - MIN_MATCH */
                  code = _length_code[lc];
                  send_code(s, code + LITERALS + 1, ltree);
                  /* send the length code */

                  extra = extra_lbits[code];

                  if (extra !== 0) {
                    lc -= base_length[code];
                    send_bits(s, lc, extra);
                    /* send the extra length bits */
                  }

                  dist--;
                  /* dist is now the match distance - 1 */

                  code = d_code(dist); //Assert (code < D_CODES, "bad d_code");

                  send_code(s, code, dtree);
                  /* send the distance code */

                  extra = extra_dbits[code];

                  if (extra !== 0) {
                    dist -= base_dist[code];
                    send_bits(s, dist, extra);
                    /* send the extra distance bits */
                  }
                }
                /* literal or match pair ? */

                /* Check that the overlay between pending_buf and d_buf+l_buf is ok: */
                //Assert((uInt)(s->pending) < s->lit_bufsize + 2*lx,
                //       "pendingBuf overflow");

              } while (lx < s.last_lit);
            }

            send_code(s, END_BLOCK, ltree);
          }
          /* ===========================================================================
           * Construct one Huffman tree and assigns the code bit strings and lengths.
           * Update the total bit length for the current block.
           * IN assertion: the field freq is set for all tree elements.
           * OUT assertions: the fields len and code are set to the optimal bit length
           *     and corresponding code. The length opt_len is updated; static_len is
           *     also updated if stree is not null. The field max_code is set.
           */


          function build_tree(s, desc) //    deflate_state *s;
          //    tree_desc *desc; /* the tree descriptor */
          {
            var tree = desc.dyn_tree;
            var stree = desc.stat_desc.static_tree;
            var has_stree = desc.stat_desc.has_stree;
            var elems = desc.stat_desc.elems;
            var n, m;
            /* iterate over heap elements */

            var max_code = -1;
            /* largest code with non zero frequency */

            var node;
            /* new node being created */

            /* Construct the initial heap, with least frequent element in
             * heap[SMALLEST]. The sons of heap[n] are heap[2*n] and heap[2*n+1].
             * heap[0] is not used.
             */

            s.heap_len = 0;
            s.heap_max = HEAP_SIZE;

            for (n = 0; n < elems; n++) {
              if (tree[n * 2]
              /*.Freq*/
              !== 0) {
                s.heap[++s.heap_len] = max_code = n;
                s.depth[n] = 0;
              } else {
                tree[n * 2 + 1]
                /*.Len*/
                = 0;
              }
            }
            /* The pkzip format requires that at least one distance code exists,
             * and that at least one bit should be sent even if there is only one
             * possible code. So to avoid special checks later on we force at least
             * two codes of non zero frequency.
             */


            while (s.heap_len < 2) {
              node = s.heap[++s.heap_len] = max_code < 2 ? ++max_code : 0;
              tree[node * 2]
              /*.Freq*/
              = 1;
              s.depth[node] = 0;
              s.opt_len--;

              if (has_stree) {
                s.static_len -= stree[node * 2 + 1]
                /*.Len*/
                ;
              }
              /* node is 0 or 1 so it does not have extra bits */

            }

            desc.max_code = max_code;
            /* The elements heap[heap_len/2+1 .. heap_len] are leaves of the tree,
             * establish sub-heaps of increasing lengths:
             */

            for (n = s.heap_len >> 1
            /*int /2*/
            ; n >= 1; n--) {
              pqdownheap(s, tree, n);
            }
            /* Construct the Huffman tree by repeatedly combining the least two
             * frequent nodes.
             */


            node = elems;
            /* next internal node of the tree */

            do {
              //pqremove(s, tree, n);  /* n = node of least frequency */

              /*** pqremove ***/
              n = s.heap[1
              /*SMALLEST*/
              ];
              s.heap[1
              /*SMALLEST*/
              ] = s.heap[s.heap_len--];
              pqdownheap(s, tree, 1
              /*SMALLEST*/
              );
              /***/

              m = s.heap[1
              /*SMALLEST*/
              ];
              /* m = node of next least frequency */

              s.heap[--s.heap_max] = n;
              /* keep the nodes sorted by frequency */

              s.heap[--s.heap_max] = m;
              /* Create a new node father of n and m */

              tree[node * 2]
              /*.Freq*/
              = tree[n * 2]
              /*.Freq*/
              + tree[m * 2]
              /*.Freq*/
              ;
              s.depth[node] = (s.depth[n] >= s.depth[m] ? s.depth[n] : s.depth[m]) + 1;
              tree[n * 2 + 1]
              /*.Dad*/
              = tree[m * 2 + 1]
              /*.Dad*/
              = node;
              /* and insert the new node in the heap */

              s.heap[1
              /*SMALLEST*/
              ] = node++;
              pqdownheap(s, tree, 1
              /*SMALLEST*/
              );
            } while (s.heap_len >= 2);

            s.heap[--s.heap_max] = s.heap[1
            /*SMALLEST*/
            ];
            /* At this point, the fields freq and dad are set. We can now
             * generate the bit lengths.
             */

            gen_bitlen(s, desc);
            /* The field len is now set, we can generate the bit codes */

            gen_codes(tree, max_code, s.bl_count);
          }
          /* ===========================================================================
           * Scan a literal or distance tree to determine the frequencies of the codes
           * in the bit length tree.
           */


          function scan_tree(s, tree, max_code) //    deflate_state *s;
          //    ct_data *tree;   /* the tree to be scanned */
          //    int max_code;    /* and its largest code of non zero frequency */
          {
            var n;
            /* iterates over all tree elements */

            var prevlen = -1;
            /* last emitted length */

            var curlen;
            /* length of current code */

            var nextlen = tree[0 * 2 + 1]
            /*.Len*/
            ;
            /* length of next code */

            var count = 0;
            /* repeat count of the current code */

            var max_count = 7;
            /* max repeat count */

            var min_count = 4;
            /* min repeat count */

            if (nextlen === 0) {
              max_count = 138;
              min_count = 3;
            }

            tree[(max_code + 1) * 2 + 1]
            /*.Len*/
            = 0xffff;
            /* guard */

            for (n = 0; n <= max_code; n++) {
              curlen = nextlen;
              nextlen = tree[(n + 1) * 2 + 1]
              /*.Len*/
              ;

              if (++count < max_count && curlen === nextlen) {
                continue;
              } else if (count < min_count) {
                s.bl_tree[curlen * 2]
                /*.Freq*/
                += count;
              } else if (curlen !== 0) {
                if (curlen !== prevlen) {
                  s.bl_tree[curlen * 2] /*.Freq*/++;
                }

                s.bl_tree[REP_3_6 * 2] /*.Freq*/++;
              } else if (count <= 10) {
                s.bl_tree[REPZ_3_10 * 2] /*.Freq*/++;
              } else {
                s.bl_tree[REPZ_11_138 * 2] /*.Freq*/++;
              }

              count = 0;
              prevlen = curlen;

              if (nextlen === 0) {
                max_count = 138;
                min_count = 3;
              } else if (curlen === nextlen) {
                max_count = 6;
                min_count = 3;
              } else {
                max_count = 7;
                min_count = 4;
              }
            }
          }
          /* ===========================================================================
           * Send a literal or distance tree in compressed form, using the codes in
           * bl_tree.
           */


          function send_tree(s, tree, max_code) //    deflate_state *s;
          //    ct_data *tree; /* the tree to be scanned */
          //    int max_code;       /* and its largest code of non zero frequency */
          {
            var n;
            /* iterates over all tree elements */

            var prevlen = -1;
            /* last emitted length */

            var curlen;
            /* length of current code */

            var nextlen = tree[0 * 2 + 1]
            /*.Len*/
            ;
            /* length of next code */

            var count = 0;
            /* repeat count of the current code */

            var max_count = 7;
            /* max repeat count */

            var min_count = 4;
            /* min repeat count */

            /* tree[max_code+1].Len = -1; */

            /* guard already set */

            if (nextlen === 0) {
              max_count = 138;
              min_count = 3;
            }

            for (n = 0; n <= max_code; n++) {
              curlen = nextlen;
              nextlen = tree[(n + 1) * 2 + 1]
              /*.Len*/
              ;

              if (++count < max_count && curlen === nextlen) {
                continue;
              } else if (count < min_count) {
                do {
                  send_code(s, curlen, s.bl_tree);
                } while (--count !== 0);
              } else if (curlen !== 0) {
                if (curlen !== prevlen) {
                  send_code(s, curlen, s.bl_tree);
                  count--;
                } //Assert(count >= 3 && count <= 6, " 3_6?");


                send_code(s, REP_3_6, s.bl_tree);
                send_bits(s, count - 3, 2);
              } else if (count <= 10) {
                send_code(s, REPZ_3_10, s.bl_tree);
                send_bits(s, count - 3, 3);
              } else {
                send_code(s, REPZ_11_138, s.bl_tree);
                send_bits(s, count - 11, 7);
              }

              count = 0;
              prevlen = curlen;

              if (nextlen === 0) {
                max_count = 138;
                min_count = 3;
              } else if (curlen === nextlen) {
                max_count = 6;
                min_count = 3;
              } else {
                max_count = 7;
                min_count = 4;
              }
            }
          }
          /* ===========================================================================
           * Construct the Huffman tree for the bit lengths and return the index in
           * bl_order of the last bit length code to send.
           */


          function build_bl_tree(s) {
            var max_blindex;
            /* index of last bit length code of non zero freq */

            /* Determine the bit length frequencies for literal and distance trees */

            scan_tree(s, s.dyn_ltree, s.l_desc.max_code);
            scan_tree(s, s.dyn_dtree, s.d_desc.max_code);
            /* Build the bit length tree: */

            build_tree(s, s.bl_desc);
            /* opt_len now includes the length of the tree representations, except
             * the lengths of the bit lengths codes and the 5+5+4 bits for the counts.
             */

            /* Determine the number of bit length codes to send. The pkzip format
             * requires that at least 4 bit length codes be sent. (appnote.txt says
             * 3 but the actual value used is 4.)
             */

            for (max_blindex = BL_CODES - 1; max_blindex >= 3; max_blindex--) {
              if (s.bl_tree[bl_order[max_blindex] * 2 + 1]
              /*.Len*/
              !== 0) {
                break;
              }
            }
            /* Update opt_len to include the bit length tree and counts */


            s.opt_len += 3 * (max_blindex + 1) + 5 + 5 + 4; //Tracev((stderr, "\ndyn trees: dyn %ld, stat %ld",
            //        s->opt_len, s->static_len));

            return max_blindex;
          }
          /* ===========================================================================
           * Send the header for a block using dynamic Huffman trees: the counts, the
           * lengths of the bit length codes, the literal tree and the distance tree.
           * IN assertion: lcodes >= 257, dcodes >= 1, blcodes >= 4.
           */


          function send_all_trees(s, lcodes, dcodes, blcodes) //    deflate_state *s;
          //    int lcodes, dcodes, blcodes; /* number of codes for each tree */
          {
            var rank;
            /* index in bl_order */
            //Assert (lcodes >= 257 && dcodes >= 1 && blcodes >= 4, "not enough codes");
            //Assert (lcodes <= L_CODES && dcodes <= D_CODES && blcodes <= BL_CODES,
            //        "too many codes");
            //Tracev((stderr, "\nbl counts: "));

            send_bits(s, lcodes - 257, 5);
            /* not +255 as stated in appnote.txt */

            send_bits(s, dcodes - 1, 5);
            send_bits(s, blcodes - 4, 4);
            /* not -3 as stated in appnote.txt */

            for (rank = 0; rank < blcodes; rank++) {
              //Tracev((stderr, "\nbl code %2d ", bl_order[rank]));
              send_bits(s, s.bl_tree[bl_order[rank] * 2 + 1]
              /*.Len*/
              , 3);
            } //Tracev((stderr, "\nbl tree: sent %ld", s->bits_sent));


            send_tree(s, s.dyn_ltree, lcodes - 1);
            /* literal tree */
            //Tracev((stderr, "\nlit tree: sent %ld", s->bits_sent));

            send_tree(s, s.dyn_dtree, dcodes - 1);
            /* distance tree */
            //Tracev((stderr, "\ndist tree: sent %ld", s->bits_sent));
          }
          /* ===========================================================================
           * Check if the data type is TEXT or BINARY, using the following algorithm:
           * - TEXT if the two conditions below are satisfied:
           *    a) There are no non-portable control characters belonging to the
           *       "black list" (0..6, 14..25, 28..31).
           *    b) There is at least one printable character belonging to the
           *       "white list" (9 {TAB}, 10 {LF}, 13 {CR}, 32..255).
           * - BINARY otherwise.
           * - The following partially-portable control characters form a
           *   "gray list" that is ignored in this detection algorithm:
           *   (7 {BEL}, 8 {BS}, 11 {VT}, 12 {FF}, 26 {SUB}, 27 {ESC}).
           * IN assertion: the fields Freq of dyn_ltree are set.
           */


          function detect_data_type(s) {
            /* black_mask is the bit mask of black-listed bytes
             * set bits 0..6, 14..25, and 28..31
             * 0xf3ffc07f = binary 11110011111111111100000001111111
             */
            var black_mask = 0xf3ffc07f;
            var n;
            /* Check for non-textual ("black-listed") bytes. */

            for (n = 0; n <= 31; n++, black_mask >>>= 1) {
              if (black_mask & 1 && s.dyn_ltree[n * 2]
              /*.Freq*/
              !== 0) {
                return Z_BINARY;
              }
            }
            /* Check for textual ("white-listed") bytes. */


            if (s.dyn_ltree[9 * 2]
            /*.Freq*/
            !== 0 || s.dyn_ltree[10 * 2]
            /*.Freq*/
            !== 0 || s.dyn_ltree[13 * 2]
            /*.Freq*/
            !== 0) {
              return Z_TEXT;
            }

            for (n = 32; n < LITERALS; n++) {
              if (s.dyn_ltree[n * 2]
              /*.Freq*/
              !== 0) {
                return Z_TEXT;
              }
            }
            /* There are no "black-listed" or "white-listed" bytes:
             * this stream either is empty or has tolerated ("gray-listed") bytes only.
             */


            return Z_BINARY;
          }

          var static_init_done = false;
          /* ===========================================================================
           * Initialize the tree data structures for a new zlib stream.
           */

          function _tr_init(s) {
            if (!static_init_done) {
              tr_static_init();
              static_init_done = true;
            }

            s.l_desc = new TreeDesc(s.dyn_ltree, static_l_desc);
            s.d_desc = new TreeDesc(s.dyn_dtree, static_d_desc);
            s.bl_desc = new TreeDesc(s.bl_tree, static_bl_desc);
            s.bi_buf = 0;
            s.bi_valid = 0;
            /* Initialize the first block of the first file: */

            init_block(s);
          }
          /* ===========================================================================
           * Send a stored block
           */


          function _tr_stored_block(s, buf, stored_len, last) //DeflateState *s;
          //charf *buf;       /* input block */
          //ulg stored_len;   /* length of input block */
          //int last;         /* one if this is the last block for a file */
          {
            send_bits(s, (STORED_BLOCK << 1) + (last ? 1 : 0), 3);
            /* send block type */

            copy_block(s, buf, stored_len, true);
            /* with header */
          }
          /* ===========================================================================
           * Send one empty static block to give enough lookahead for inflate.
           * This takes 10 bits, of which 7 may remain in the bit buffer.
           */


          function _tr_align(s) {
            send_bits(s, STATIC_TREES << 1, 3);
            send_code(s, END_BLOCK, static_ltree);
            bi_flush(s);
          }
          /* ===========================================================================
           * Determine the best encoding for the current block: dynamic trees, static
           * trees or store, and output the encoded block to the zip file.
           */


          function _tr_flush_block(s, buf, stored_len, last) //DeflateState *s;
          //charf *buf;       /* input block, or NULL if too old */
          //ulg stored_len;   /* length of input block */
          //int last;         /* one if this is the last block for a file */
          {
            var opt_lenb, static_lenb;
            /* opt_len and static_len in bytes */

            var max_blindex = 0;
            /* index of last bit length code of non zero freq */

            /* Build the Huffman trees unless a stored block is forced */

            if (s.level > 0) {
              /* Check if the file is binary or text */
              if (s.strm.data_type === Z_UNKNOWN) {
                s.strm.data_type = detect_data_type(s);
              }
              /* Construct the literal and distance trees */


              build_tree(s, s.l_desc); // Tracev((stderr, "\nlit data: dyn %ld, stat %ld", s->opt_len,
              //        s->static_len));

              build_tree(s, s.d_desc); // Tracev((stderr, "\ndist data: dyn %ld, stat %ld", s->opt_len,
              //        s->static_len));

              /* At this point, opt_len and static_len are the total bit lengths of
               * the compressed block data, excluding the tree representations.
               */

              /* Build the bit length tree for the above two trees, and get the index
               * in bl_order of the last bit length code to send.
               */

              max_blindex = build_bl_tree(s);
              /* Determine the best encoding. Compute the block lengths in bytes. */

              opt_lenb = s.opt_len + 3 + 7 >>> 3;
              static_lenb = s.static_len + 3 + 7 >>> 3; // Tracev((stderr, "\nopt %lu(%lu) stat %lu(%lu) stored %lu lit %u ",
              //        opt_lenb, s->opt_len, static_lenb, s->static_len, stored_len,
              //        s->last_lit));

              if (static_lenb <= opt_lenb) {
                opt_lenb = static_lenb;
              }
            } else {
              // Assert(buf != (char*)0, "lost buf");
              opt_lenb = static_lenb = stored_len + 5;
              /* force a stored block */
            }

            if (stored_len + 4 <= opt_lenb && buf !== -1) {
              /* 4: two words for the lengths */

              /* The test buf != NULL is only necessary if LIT_BUFSIZE > WSIZE.
               * Otherwise we can't have processed more than WSIZE input bytes since
               * the last block flush, because compression would have been
               * successful. If LIT_BUFSIZE <= WSIZE, it is never too late to
               * transform a block into a stored block.
               */
              _tr_stored_block(s, buf, stored_len, last);
            } else if (s.strategy === Z_FIXED || static_lenb === opt_lenb) {
              send_bits(s, (STATIC_TREES << 1) + (last ? 1 : 0), 3);
              compress_block(s, static_ltree, static_dtree);
            } else {
              send_bits(s, (DYN_TREES << 1) + (last ? 1 : 0), 3);
              send_all_trees(s, s.l_desc.max_code + 1, s.d_desc.max_code + 1, max_blindex + 1);
              compress_block(s, s.dyn_ltree, s.dyn_dtree);
            } // Assert (s->compressed_len == s->bits_sent, "bad compressed size");

            /* The above check is made mod 2^32, for files larger than 512 MB
             * and uLong implemented on 32 bits.
             */


            init_block(s);

            if (last) {
              bi_windup(s);
            } // Tracev((stderr,"\ncomprlen %lu(%lu) ", s->compressed_len>>3,
            //       s->compressed_len-7*last));

          }
          /* ===========================================================================
           * Save the match info and tally the frequency counts. Return true if
           * the current block must be flushed.
           */


          function _tr_tally(s, dist, lc) //    deflate_state *s;
          //    unsigned dist;  /* distance of matched string */
          //    unsigned lc;    /* match length-MIN_MATCH or unmatched char (if dist==0) */
          {
            //var out_length, in_length, dcode;
            s.pending_buf[s.d_buf + s.last_lit * 2] = dist >>> 8 & 0xff;
            s.pending_buf[s.d_buf + s.last_lit * 2 + 1] = dist & 0xff;
            s.pending_buf[s.l_buf + s.last_lit] = lc & 0xff;
            s.last_lit++;

            if (dist === 0) {
              /* lc is the unmatched char */
              s.dyn_ltree[lc * 2] /*.Freq*/++;
            } else {
              s.matches++;
              /* Here, lc is the match length - MIN_MATCH */

              dist--;
              /* dist = match distance - 1 */
              //Assert((ush)dist < (ush)MAX_DIST(s) &&
              //       (ush)lc <= (ush)(MAX_MATCH-MIN_MATCH) &&
              //       (ush)d_code(dist) < (ush)D_CODES,  "_tr_tally: bad match");

              s.dyn_ltree[(_length_code[lc] + LITERALS + 1) * 2] /*.Freq*/++;
              s.dyn_dtree[d_code(dist) * 2] /*.Freq*/++;
            } // (!) This block is disabled in zlib defailts,
            // don't enable it for binary compatibility
            //#ifdef TRUNCATE_BLOCK
            //  /* Try to guess if it is profitable to stop the current block here */
            //  if ((s.last_lit & 0x1fff) === 0 && s.level > 2) {
            //    /* Compute an upper bound for the compressed length */
            //    out_length = s.last_lit*8;
            //    in_length = s.strstart - s.block_start;
            //
            //    for (dcode = 0; dcode < D_CODES; dcode++) {
            //      out_length += s.dyn_dtree[dcode*2]/*.Freq*/ * (5 + extra_dbits[dcode]);
            //    }
            //    out_length >>>= 3;
            //    //Tracev((stderr,"\nlast_lit %u, in %ld, out ~%ld(%ld%%) ",
            //    //       s->last_lit, in_length, out_length,
            //    //       100L - out_length*100L/in_length));
            //    if (s.matches < (s.last_lit>>1)/*int /2*/ && out_length < (in_length>>1)/*int /2*/) {
            //      return true;
            //    }
            //  }
            //#endif


            return s.last_lit === s.lit_bufsize - 1;
            /* We avoid equality with lit_bufsize because of wraparound at 64K
             * on 16 bit machines and because stored blocks are restricted to
             * 64K-1 bytes.
             */
          }

          exports._tr_init = _tr_init;
          exports._tr_stored_block = _tr_stored_block;
          exports._tr_flush_block = _tr_flush_block;
          exports._tr_tally = _tr_tally;
          exports._tr_align = _tr_align;
        }, {
          "../utils/common": 27
        }],
        39: [function (_dereq_, module, exports) {

          function ZStream() {
            /* next input byte */
            this.input = null; // JS specific, because we have no pointers

            this.next_in = 0;
            /* number of bytes available at input */

            this.avail_in = 0;
            /* total number of input bytes read so far */

            this.total_in = 0;
            /* next output byte should be put there */

            this.output = null; // JS specific, because we have no pointers

            this.next_out = 0;
            /* remaining free space at output */

            this.avail_out = 0;
            /* total number of bytes output so far */

            this.total_out = 0;
            /* last error message, NULL if no error */

            this.msg = ''
            /*Z_NULL*/
            ;
            /* not visible by applications */

            this.state = null;
            /* best guess about the data type: binary or text */

            this.data_type = 2
            /*Z_UNKNOWN*/
            ;
            /* adler32 value of the uncompressed data */

            this.adler = 0;
          }

          module.exports = ZStream;
        }, {}]
      }, {}, [9])(9);
    });
  });

  var require$$4 = /*@__PURE__*/getAugmentedNamespace(_nodeResolve_empty$1);

  var xlsx=createCommonjsModule(function(module,exports){function make_xlsx_lib(XLSX){XLSX.version='0.16.9';var current_codepage=1200,current_ansi=1252;/*global cptable:true, window */if(typeof commonjsRequire!=='undefined'){if(typeof cptable==='undefined'){if(typeof commonjsGlobal!=='undefined')commonjsGlobal.cptable=cpexcel;else if(typeof window!=='undefined')window.cptable=cpexcel;}}var VALID_ANSI=[874,932,936,949,950];for(var i=0;i<=8;++i)VALID_ANSI.push(1250+i);/* ECMA-376 Part I 18.4.1 charset to codepage mapping */var CS2CP={0:1252,/* ANSI */1:65001,/* DEFAULT */2:65001,/* SYMBOL */77:10000,/* MAC */128:932,/* SHIFTJIS */129:949,/* HANGUL */130:1361,/* JOHAB */134:936,/* GB2312 */136:950,/* CHINESEBIG5 */161:1253,/* GREEK */162:1254,/* TURKISH */163:1258,/* VIETNAMESE */177:1255,/* HEBREW */178:1256,/* ARABIC */186:1257,/* BALTIC */204:1251,/* RUSSIAN */222:874,/* THAI */238:1250,/* EASTEUROPE */255:1252,/* OEM */69:6969/* MISC */};var set_ansi=function(cp){if(VALID_ANSI.indexOf(cp)==-1)return;current_ansi=CS2CP[0]=cp;};function reset_ansi(){set_ansi(1252);}var set_cp=function(cp){current_codepage=cp;set_ansi(cp);};function reset_cp(){set_cp(1200);reset_ansi();}function char_codes(data){var o=[];for(var i=0,len=data.length;i<len;++i)o[i]=data.charCodeAt(i);return o;}function utf16leread(data){var o=[];for(var i=0;i<data.length>>1;++i)o[i]=String.fromCharCode(data.charCodeAt(2*i)+(data.charCodeAt(2*i+1)<<8));return o.join("");}function utf16beread(data){var o=[];for(var i=0;i<data.length>>1;++i)o[i]=String.fromCharCode(data.charCodeAt(2*i+1)+(data.charCodeAt(2*i)<<8));return o.join("");}var debom=function(data){var c1=data.charCodeAt(0),c2=data.charCodeAt(1);if(c1==0xFF&&c2==0xFE)return utf16leread(data.slice(2));if(c1==0xFE&&c2==0xFF)return utf16beread(data.slice(2));if(c1==0xFEFF)return data.slice(1);return data;};var _getchar=function _gc1(x){return String.fromCharCode(x);};var _getansi=function _ga1(x){return String.fromCharCode(x);};if(typeof cptable!=='undefined'){set_cp=function(cp){current_codepage=cp;set_ansi(cp);};debom=function(data){if(data.charCodeAt(0)===0xFF&&data.charCodeAt(1)===0xFE){return cptable.utils.decode(1200,char_codes(data.slice(2)));}return data;};_getchar=function _gc2(x){if(current_codepage===1200)return String.fromCharCode(x);return cptable.utils.decode(current_codepage,[x&255,x>>8])[0];};_getansi=function _ga2(x){return cptable.utils.decode(current_ansi,[x])[0];};}var Base64=function make_b64(){var map="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";return {encode:function(input){var o="";var c1=0,c2=0,c3=0,e1=0,e2=0,e3=0,e4=0;for(var i=0;i<input.length;){c1=input.charCodeAt(i++);e1=c1>>2;c2=input.charCodeAt(i++);e2=(c1&3)<<4|c2>>4;c3=input.charCodeAt(i++);e3=(c2&15)<<2|c3>>6;e4=c3&63;if(isNaN(c2)){e3=e4=64;}else if(isNaN(c3)){e4=64;}o+=map.charAt(e1)+map.charAt(e2)+map.charAt(e3)+map.charAt(e4);}return o;},decode:function b64_decode(input){var o="";var c1=0,c2=0,c3=0,e1=0,e2=0,e3=0,e4=0;input=input.replace(/[^\w\+\/\=]/g,"");for(var i=0;i<input.length;){e1=map.indexOf(input.charAt(i++));e2=map.indexOf(input.charAt(i++));c1=e1<<2|e2>>4;o+=String.fromCharCode(c1);e3=map.indexOf(input.charAt(i++));c2=(e2&15)<<4|e3>>2;if(e3!==64){o+=String.fromCharCode(c2);}e4=map.indexOf(input.charAt(i++));c3=(e3&3)<<6|e4;if(e4!==64){o+=String.fromCharCode(c3);}}return o;}};}();var has_buf=typeof Buffer!=='undefined'&&typeof process!=='undefined'&&typeof process.versions!=='undefined'&&!!process.versions.node;var Buffer_from=function(){};if(typeof Buffer!=='undefined'){var nbfs=!Buffer.from;if(!nbfs)try{Buffer.from("foo","utf8");}catch(e){nbfs=true;}Buffer_from=nbfs?function(buf,enc){return enc?new Buffer(buf,enc):new Buffer(buf);}:Buffer.from.bind(Buffer);// $FlowIgnore
  if(!Buffer.alloc)Buffer.alloc=function(n){return new Buffer(n);};// $FlowIgnore
  if(!Buffer.allocUnsafe)Buffer.allocUnsafe=function(n){return new Buffer(n);};}function new_raw_buf(len){/* jshint -W056 */return has_buf?Buffer.alloc(len):new Array(len);/* jshint +W056 */}function new_unsafe_buf(len){/* jshint -W056 */return has_buf?Buffer.allocUnsafe(len):new Array(len);/* jshint +W056 */}var s2a=function s2a(s){if(has_buf)return Buffer_from(s,"binary");return s.split("").map(function(x){return x.charCodeAt(0)&0xff;});};function s2ab(s){if(typeof ArrayBuffer==='undefined')return s2a(s);var buf=new ArrayBuffer(s.length),view=new Uint8Array(buf);for(var i=0;i!=s.length;++i)view[i]=s.charCodeAt(i)&0xFF;return buf;}function a2s(data){if(Array.isArray(data))return data.map(function(c){return String.fromCharCode(c);}).join("");var o=[];for(var i=0;i<data.length;++i)o[i]=String.fromCharCode(data[i]);return o.join("");}function a2u(data){if(typeof Uint8Array==='undefined')throw new Error("Unsupported");return new Uint8Array(data);}function ab2a(data){if(typeof ArrayBuffer=='undefined')throw new Error("Unsupported");if(data instanceof ArrayBuffer)return ab2a(new Uint8Array(data));var o=new Array(data.length);for(var i=0;i<data.length;++i)o[i]=data[i];return o;}var bconcat=function(bufs){return [].concat.apply([],bufs);};var chr0=/\u0000/g,chr1=/[\u0001-\u0006]/g;/* ssf.js (C) 2013-present SheetJS -- http://sheetjs.com */ /*jshint -W041 */var SSF={};var make_ssf=function make_ssf(SSF){SSF.version='0.11.2';function _strrev(x){var o="",i=x.length-1;while(i>=0)o+=x.charAt(i--);return o;}function fill(c,l){var o="";while(o.length<l)o+=c;return o;}function pad0(v,d){var t=""+v;return t.length>=d?t:fill('0',d-t.length)+t;}function pad_(v,d){var t=""+v;return t.length>=d?t:fill(' ',d-t.length)+t;}function rpad_(v,d){var t=""+v;return t.length>=d?t:t+fill(' ',d-t.length);}function pad0r1(v,d){var t=""+Math.round(v);return t.length>=d?t:fill('0',d-t.length)+t;}function pad0r2(v,d){var t=""+v;return t.length>=d?t:fill('0',d-t.length)+t;}var p2_32=Math.pow(2,32);function pad0r(v,d){if(v>p2_32||v<-p2_32)return pad0r1(v,d);var i=Math.round(v);return pad0r2(i,d);}function isgeneral(s,i){i=i||0;return s.length>=7+i&&(s.charCodeAt(i)|32)===103&&(s.charCodeAt(i+1)|32)===101&&(s.charCodeAt(i+2)|32)===110&&(s.charCodeAt(i+3)|32)===101&&(s.charCodeAt(i+4)|32)===114&&(s.charCodeAt(i+5)|32)===97&&(s.charCodeAt(i+6)|32)===108;}var days=[['Sun','Sunday'],['Mon','Monday'],['Tue','Tuesday'],['Wed','Wednesday'],['Thu','Thursday'],['Fri','Friday'],['Sat','Saturday']];var months=[['J','Jan','January'],['F','Feb','February'],['M','Mar','March'],['A','Apr','April'],['M','May','May'],['J','Jun','June'],['J','Jul','July'],['A','Aug','August'],['S','Sep','September'],['O','Oct','October'],['N','Nov','November'],['D','Dec','December']];function init_table(t){t[0]='General';t[1]='0';t[2]='0.00';t[3]='#,##0';t[4]='#,##0.00';t[9]='0%';t[10]='0.00%';t[11]='0.00E+00';t[12]='# ?/?';t[13]='# ??/??';t[14]='m/d/yy';t[15]='d-mmm-yy';t[16]='d-mmm';t[17]='mmm-yy';t[18]='h:mm AM/PM';t[19]='h:mm:ss AM/PM';t[20]='h:mm';t[21]='h:mm:ss';t[22]='m/d/yy h:mm';t[37]='#,##0 ;(#,##0)';t[38]='#,##0 ;[Red](#,##0)';t[39]='#,##0.00;(#,##0.00)';t[40]='#,##0.00;[Red](#,##0.00)';t[45]='mm:ss';t[46]='[h]:mm:ss';t[47]='mmss.0';t[48]='##0.0E+0';t[49]='@';t[56]='"上午/下午 "hh"時"mm"分"ss"秒 "';}var table_fmt={};init_table(table_fmt);/* Defaults determined by systematically testing in Excel 2019 */ /* These formats appear to default to other formats in the table */var default_map=[];var defi=0;//  5 -> 37 ...  8 -> 40
  for(defi=5;defi<=8;++defi)default_map[defi]=32+defi;// 23 ->  0 ... 26 ->  0
  for(defi=23;defi<=26;++defi)default_map[defi]=0;// 27 -> 14 ... 31 -> 14
  for(defi=27;defi<=31;++defi)default_map[defi]=14;// 50 -> 14 ... 58 -> 14
  for(defi=50;defi<=58;++defi)default_map[defi]=14;// 59 ->  1 ... 62 ->  4
  for(defi=59;defi<=62;++defi)default_map[defi]=defi-58;// 67 ->  9 ... 68 -> 10
  for(defi=67;defi<=68;++defi)default_map[defi]=defi-58;// 72 -> 14 ... 75 -> 17
  for(defi=72;defi<=75;++defi)default_map[defi]=defi-58;// 69 -> 12 ... 71 -> 14
  for(defi=67;defi<=68;++defi)default_map[defi]=defi-57;// 76 -> 20 ... 78 -> 22
  for(defi=76;defi<=78;++defi)default_map[defi]=defi-56;// 79 -> 45 ... 81 -> 47
  for(defi=79;defi<=81;++defi)default_map[defi]=defi-34;// 82 ->  0 ... 65536 -> 0 (omitted)
  /* These formats technically refer to Accounting formats with no equivalent */var default_str=[];//  5 -- Currency,   0 decimal, black negative
  default_str[5]=default_str[63]='"$"#,##0_);\\("$"#,##0\\)';//  6 -- Currency,   0 decimal, red   negative
  default_str[6]=default_str[64]='"$"#,##0_);[Red]\\("$"#,##0\\)';//  7 -- Currency,   2 decimal, black negative
  default_str[7]=default_str[65]='"$"#,##0.00_);\\("$"#,##0.00\\)';//  8 -- Currency,   2 decimal, red   negative
  default_str[8]=default_str[66]='"$"#,##0.00_);[Red]\\("$"#,##0.00\\)';// 41 -- Accounting, 0 decimal, No Symbol
  default_str[41]='_(* #,##0_);_(* \\(#,##0\\);_(* "-"_);_(@_)';// 42 -- Accounting, 0 decimal, $  Symbol
  default_str[42]='_("$"* #,##0_);_("$"* \\(#,##0\\);_("$"* "-"_);_(@_)';// 43 -- Accounting, 2 decimal, No Symbol
  default_str[43]='_(* #,##0.00_);_(* \\(#,##0.00\\);_(* "-"??_);_(@_)';// 44 -- Accounting, 2 decimal, $  Symbol
  default_str[44]='_("$"* #,##0.00_);_("$"* \\(#,##0.00\\);_("$"* "-"??_);_(@_)';function frac(x,D,mixed){var sgn=x<0?-1:1;var B=x*sgn;var P_2=0,P_1=1,P=0;var Q_2=1,Q_1=0,Q=0;var A=Math.floor(B);while(Q_1<D){A=Math.floor(B);P=A*P_1+P_2;Q=A*Q_1+Q_2;if(B-A<0.00000005)break;B=1/(B-A);P_2=P_1;P_1=P;Q_2=Q_1;Q_1=Q;}if(Q>D){if(Q_1>D){Q=Q_2;P=P_2;}else {Q=Q_1;P=P_1;}}if(!mixed)return [0,sgn*P,Q];var q=Math.floor(sgn*P/Q);return [q,sgn*P-q*Q,Q];}function parse_date_code(v,opts,b2){if(v>2958465||v<0)return null;var date=v|0,time=Math.floor(86400*(v-date)),dow=0;var dout=[];var out={D:date,T:time,u:86400*(v-date)-time,y:0,m:0,d:0,H:0,M:0,S:0,q:0};if(Math.abs(out.u)<1e-6)out.u=0;if(opts&&opts.date1904)date+=1462;if(out.u>0.9999){out.u=0;if(++time==86400){out.T=time=0;++date;++out.D;}}if(date===60){dout=b2?[1317,10,29]:[1900,2,29];dow=3;}else if(date===0){dout=b2?[1317,8,29]:[1900,1,0];dow=6;}else {if(date>60)--date;/* 1 = Jan 1 1900 in Gregorian */var d=new Date(1900,0,1);d.setDate(d.getDate()+date-1);dout=[d.getFullYear(),d.getMonth()+1,d.getDate()];dow=d.getDay();if(date<60)dow=(dow+6)%7;if(b2)dow=fix_hijri(d,dout);}out.y=dout[0];out.m=dout[1];out.d=dout[2];out.S=time%60;time=Math.floor(time/60);out.M=time%60;time=Math.floor(time/60);out.H=time;out.q=dow;return out;}SSF.parse_date_code=parse_date_code;var basedate=new Date(1899,11,31,0,0,0);var dnthresh=basedate.getTime();var base1904=new Date(1900,2,1,0,0,0);function datenum_local(v,date1904){var epoch=v.getTime();if(date1904)epoch-=1461*24*60*60*1000;else if(v>=base1904)epoch+=24*60*60*1000;return (epoch-(dnthresh+(v.getTimezoneOffset()-basedate.getTimezoneOffset())*60000))/(24*60*60*1000);}/* The longest 32-bit integer text is "-4294967296", exactly 11 chars */function general_fmt_int(v){return v.toString(10);}SSF._general_int=general_fmt_int;/* ECMA-376 18.8.30 numFmt*/ /* Note: `toPrecision` uses standard form when prec > E and E >= -6 */var general_fmt_num=function make_general_fmt_num(){var trailing_zeroes_and_decimal=/(?:\.0*|(\.\d*[1-9])0+)$/;function strip_decimal(o){return o.indexOf(".")==-1?o:o.replace(trailing_zeroes_and_decimal,"$1");}/* General Exponential always shows 2 digits exp and trims the mantissa */var mantissa_zeroes_and_decimal=/(?:\.0*|(\.\d*[1-9])0+)[Ee]/;var exp_with_single_digit=/(E[+-])(\d)$/;function normalize_exp(o){if(o.indexOf("E")==-1)return o;return o.replace(mantissa_zeroes_and_decimal,"$1E").replace(exp_with_single_digit,"$10$2");}/* exponent >= -9 and <= 9 */function small_exp(v){var w=v<0?12:11;var o=strip_decimal(v.toFixed(12));if(o.length<=w)return o;o=v.toPrecision(10);if(o.length<=w)return o;return v.toExponential(5);}/* exponent >= 11 or <= -10 likely exponential */function large_exp(v){var o=strip_decimal(v.toFixed(11));return o.length>(v<0?12:11)||o==="0"||o==="-0"?v.toPrecision(6):o;}function general_fmt_num_base(v){var V=Math.floor(Math.log(Math.abs(v))*Math.LOG10E),o;if(V>=-4&&V<=-1)o=v.toPrecision(10+V);else if(Math.abs(V)<=9)o=small_exp(v);else if(V===10)o=v.toFixed(10).substr(0,12);else o=large_exp(v);return strip_decimal(normalize_exp(o.toUpperCase()));}return general_fmt_num_base;}();SSF._general_num=general_fmt_num;/*
  	"General" rules:
  	- text is passed through ("@")
  	- booleans are rendered as TRUE/FALSE
  	- "up to 11 characters" displayed for numbers
  	- Default date format (code 14) used for Dates

  	TODO: technically the display depends on the width of the cell
  */function general_fmt(v,opts){switch(typeof v){case'string':return v;case'boolean':return v?"TRUE":"FALSE";case'number':return (v|0)===v?v.toString(10):general_fmt_num(v);case'undefined':return "";case'object':if(v==null)return "";if(v instanceof Date)return format(14,datenum_local(v,opts&&opts.date1904),opts);}throw new Error("unsupported value in General format: "+v);}SSF._general=general_fmt;function fix_hijri(date,o){/* TODO: properly adjust y/m/d and  */o[0]-=581;var dow=date.getDay();if(date<60)dow=(dow+6)%7;return dow;}//var THAI_DIGITS = "\u0E50\u0E51\u0E52\u0E53\u0E54\u0E55\u0E56\u0E57\u0E58\u0E59".split("");
  /*jshint -W086 */function write_date(type,fmt,val,ss0){var o="",ss=0,tt=0,y=val.y,out,outl=0;switch(type){case 98:/* 'b' buddhist year */y=val.y+543;/* falls through */case 121:/* 'y' year */switch(fmt.length){case 1:case 2:out=y%100;outl=2;break;default:out=y%10000;outl=4;break;}break;case 109:/* 'm' month */switch(fmt.length){case 1:case 2:out=val.m;outl=fmt.length;break;case 3:return months[val.m-1][1];case 5:return months[val.m-1][0];default:return months[val.m-1][2];}break;case 100:/* 'd' day */switch(fmt.length){case 1:case 2:out=val.d;outl=fmt.length;break;case 3:return days[val.q][0];default:return days[val.q][1];}break;case 104:/* 'h' 12-hour */switch(fmt.length){case 1:case 2:out=1+(val.H+11)%12;outl=fmt.length;break;default:throw 'bad hour format: '+fmt;}break;case 72:/* 'H' 24-hour */switch(fmt.length){case 1:case 2:out=val.H;outl=fmt.length;break;default:throw 'bad hour format: '+fmt;}break;case 77:/* 'M' minutes */switch(fmt.length){case 1:case 2:out=val.M;outl=fmt.length;break;default:throw 'bad minute format: '+fmt;}break;case 115:/* 's' seconds */if(fmt!='s'&&fmt!='ss'&&fmt!='.0'&&fmt!='.00'&&fmt!='.000')throw 'bad second format: '+fmt;if(val.u===0&&(fmt=="s"||fmt=="ss"))return pad0(val.S,fmt.length);if(ss0>=2)tt=ss0===3?1000:100;else tt=ss0===1?10:1;ss=Math.round(tt*(val.S+val.u));if(ss>=60*tt)ss=0;if(fmt==='s')return ss===0?"0":""+ss/tt;o=pad0(ss,2+ss0);if(fmt==='ss')return o.substr(0,2);return "."+o.substr(2,fmt.length-1);case 90:/* 'Z' absolute time */switch(fmt){case'[h]':case'[hh]':out=val.D*24+val.H;break;case'[m]':case'[mm]':out=(val.D*24+val.H)*60+val.M;break;case'[s]':case'[ss]':out=((val.D*24+val.H)*60+val.M)*60+Math.round(val.S+val.u);break;default:throw 'bad abstime format: '+fmt;}outl=fmt.length===3?1:2;break;case 101:/* 'e' era */out=y;outl=1;break;}var outstr=outl>0?pad0(out,outl):"";return outstr;}/*jshint +W086 */function commaify(s){var w=3;if(s.length<=w)return s;var j=s.length%w,o=s.substr(0,j);for(;j!=s.length;j+=w)o+=(o.length>0?",":"")+s.substr(j,w);return o;}var write_num=function make_write_num(){var pct1=/%/g;function write_num_pct(type,fmt,val){var sfmt=fmt.replace(pct1,""),mul=fmt.length-sfmt.length;return write_num(type,sfmt,val*Math.pow(10,2*mul))+fill("%",mul);}function write_num_cm(type,fmt,val){var idx=fmt.length-1;while(fmt.charCodeAt(idx-1)===44)--idx;return write_num(type,fmt.substr(0,idx),val/Math.pow(10,3*(fmt.length-idx)));}function write_num_exp(fmt,val){var o;var idx=fmt.indexOf("E")-fmt.indexOf(".")-1;if(fmt.match(/^#+0.0E\+0$/)){if(val==0)return "0.0E+0";else if(val<0)return "-"+write_num_exp(fmt,-val);var period=fmt.indexOf(".");if(period===-1)period=fmt.indexOf('E');var ee=Math.floor(Math.log(val)*Math.LOG10E)%period;if(ee<0)ee+=period;o=(val/Math.pow(10,ee)).toPrecision(idx+1+(period+ee)%period);if(o.indexOf("e")===-1){var fakee=Math.floor(Math.log(val)*Math.LOG10E);if(o.indexOf(".")===-1)o=o.charAt(0)+"."+o.substr(1)+"E+"+(fakee-o.length+ee);else o+="E+"+(fakee-ee);while(o.substr(0,2)==="0."){o=o.charAt(0)+o.substr(2,period)+"."+o.substr(2+period);o=o.replace(/^0+([1-9])/,"$1").replace(/^0+\./,"0.");}o=o.replace(/\+-/,"-");}o=o.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/,function($$,$1,$2,$3){return $1+$2+$3.substr(0,(period+ee)%period)+"."+$3.substr(ee)+"E";});}else o=val.toExponential(idx);if(fmt.match(/E\+00$/)&&o.match(/e[+-]\d$/))o=o.substr(0,o.length-1)+"0"+o.charAt(o.length-1);if(fmt.match(/E\-/)&&o.match(/e\+/))o=o.replace(/e\+/,"e");return o.replace("e","E");}var frac1=/# (\?+)( ?)\/( ?)(\d+)/;function write_num_f1(r,aval,sign){var den=parseInt(r[4],10),rr=Math.round(aval*den),base=Math.floor(rr/den);var myn=rr-base*den,myd=den;return sign+(base===0?"":""+base)+" "+(myn===0?fill(" ",r[1].length+1+r[4].length):pad_(myn,r[1].length)+r[2]+"/"+r[3]+pad0(myd,r[4].length));}function write_num_f2(r,aval,sign){return sign+(aval===0?"":""+aval)+fill(" ",r[1].length+2+r[4].length);}var dec1=/^#*0*\.([0#]+)/;var closeparen=/\).*[0#]/;var phone=/\(###\) ###\\?-####/;function hashq(str){var o="",cc;for(var i=0;i!=str.length;++i)switch(cc=str.charCodeAt(i)){case 35:break;case 63:o+=" ";break;case 48:o+="0";break;default:o+=String.fromCharCode(cc);}return o;}function rnd(val,d){var dd=Math.pow(10,d);return ""+Math.round(val*dd)/dd;}function dec(val,d){var _frac=val-Math.floor(val),dd=Math.pow(10,d);if(d<(''+Math.round(_frac*dd)).length)return 0;return Math.round(_frac*dd);}function carry(val,d){if(d<(''+Math.round((val-Math.floor(val))*Math.pow(10,d))).length){return 1;}return 0;}function flr(val){if(val<2147483647&&val>-2147483648)return ""+(val>=0?val|0:val-1|0);return ""+Math.floor(val);}function write_num_flt(type,fmt,val){if(type.charCodeAt(0)===40&&!fmt.match(closeparen)){var ffmt=fmt.replace(/\( */,"").replace(/ \)/,"").replace(/\)/,"");if(val>=0)return write_num_flt('n',ffmt,val);return '('+write_num_flt('n',ffmt,-val)+')';}if(fmt.charCodeAt(fmt.length-1)===44)return write_num_cm(type,fmt,val);if(fmt.indexOf('%')!==-1)return write_num_pct(type,fmt,val);if(fmt.indexOf('E')!==-1)return write_num_exp(fmt,val);if(fmt.charCodeAt(0)===36)return "$"+write_num_flt(type,fmt.substr(fmt.charAt(1)==' '?2:1),val);var o;var r,ri,ff,aval=Math.abs(val),sign=val<0?"-":"";if(fmt.match(/^00+$/))return sign+pad0r(aval,fmt.length);if(fmt.match(/^[#?]+$/)){o=pad0r(val,0);if(o==="0")o="";return o.length>fmt.length?o:hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(frac1))return write_num_f1(r,aval,sign);if(fmt.match(/^#+0+$/))return sign+pad0r(aval,fmt.length-fmt.indexOf("0"));if(r=fmt.match(dec1)){o=rnd(val,r[1].length).replace(/^([^\.]+)$/,"$1."+hashq(r[1])).replace(/\.$/,"."+hashq(r[1])).replace(/\.(\d*)$/,function($$,$1){return "."+$1+fill("0",hashq(r[1]).length-$1.length);});return fmt.indexOf("0.")!==-1?o:o.replace(/^0\./,".");}fmt=fmt.replace(/^#+([0.])/,"$1");if(r=fmt.match(/^(0*)\.(#*)$/)){return sign+rnd(aval,r[2].length).replace(/\.(\d*[1-9])0*$/,".$1").replace(/^(-?\d*)$/,"$1.").replace(/^0\./,r[1].length?"0.":".");}if(r=fmt.match(/^#{1,3},##0(\.?)$/))return sign+commaify(pad0r(aval,0));if(r=fmt.match(/^#,##0\.([#0]*0)$/)){return val<0?"-"+write_num_flt(type,fmt,-val):commaify(""+(Math.floor(val)+carry(val,r[1].length)))+"."+pad0(dec(val,r[1].length),r[1].length);}if(r=fmt.match(/^#,#*,#0/))return write_num_flt(type,fmt.replace(/^#,#*,/,""),val);if(r=fmt.match(/^([0#]+)(\\?-([0#]+))+$/)){o=_strrev(write_num_flt(type,fmt.replace(/[\\-]/g,""),val));ri=0;return _strrev(_strrev(fmt.replace(/\\/g,"")).replace(/[0#]/g,function(x){return ri<o.length?o.charAt(ri++):x==='0'?'0':"";}));}if(fmt.match(phone)){o=write_num_flt(type,"##########",val);return "("+o.substr(0,3)+") "+o.substr(3,3)+"-"+o.substr(6);}var oa="";if(r=fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(r[4].length,7);ff=frac(aval,Math.pow(10,ri)-1,false);o=""+sign;oa=write_num("n",r[1],ff[1]);if(oa.charAt(oa.length-1)==" ")oa=oa.substr(0,oa.length-1)+"0";o+=oa+r[2]+"/"+r[3];oa=rpad_(ff[2],ri);if(oa.length<r[4].length)oa=hashq(r[4].substr(r[4].length-oa.length))+oa;o+=oa;return o;}if(r=fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(Math.max(r[1].length,r[4].length),7);ff=frac(aval,Math.pow(10,ri)-1,true);return sign+(ff[0]||(ff[1]?"":"0"))+" "+(ff[1]?pad_(ff[1],ri)+r[2]+"/"+r[3]+rpad_(ff[2],ri):fill(" ",2*ri+1+r[2].length+r[3].length));}if(r=fmt.match(/^[#0?]+$/)){o=pad0r(val,0);if(fmt.length<=o.length)return o;return hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(/^([#0?]+)\.([#0]+)$/)){o=""+val.toFixed(Math.min(r[2].length,10)).replace(/([^0])0+$/,"$1");ri=o.indexOf(".");var lres=fmt.indexOf(".")-ri,rres=fmt.length-o.length-lres;return hashq(fmt.substr(0,lres)+o+fmt.substr(fmt.length-rres));}if(r=fmt.match(/^00,000\.([#0]*0)$/)){ri=dec(val,r[1].length);return val<0?"-"+write_num_flt(type,fmt,-val):commaify(flr(val)).replace(/^\d,\d{3}$/,"0$&").replace(/^\d*$/,function($$){return "00,"+($$.length<3?pad0(0,3-$$.length):"")+$$;})+"."+pad0(ri,r[1].length);}switch(fmt){case"###,##0.00":return write_num_flt(type,"#,##0.00",val);case"###,###":case"##,###":case"#,###":var x=commaify(pad0r(aval,0));return x!=="0"?sign+x:"";case"###,###.00":return write_num_flt(type,"###,##0.00",val).replace(/^0\./,".");case"#,###.00":return write_num_flt(type,"#,##0.00",val).replace(/^0\./,".");}throw new Error("unsupported format |"+fmt+"|");}function write_num_cm2(type,fmt,val){var idx=fmt.length-1;while(fmt.charCodeAt(idx-1)===44)--idx;return write_num(type,fmt.substr(0,idx),val/Math.pow(10,3*(fmt.length-idx)));}function write_num_pct2(type,fmt,val){var sfmt=fmt.replace(pct1,""),mul=fmt.length-sfmt.length;return write_num(type,sfmt,val*Math.pow(10,2*mul))+fill("%",mul);}function write_num_exp2(fmt,val){var o;var idx=fmt.indexOf("E")-fmt.indexOf(".")-1;if(fmt.match(/^#+0.0E\+0$/)){if(val==0)return "0.0E+0";else if(val<0)return "-"+write_num_exp2(fmt,-val);var period=fmt.indexOf(".");if(period===-1)period=fmt.indexOf('E');var ee=Math.floor(Math.log(val)*Math.LOG10E)%period;if(ee<0)ee+=period;o=(val/Math.pow(10,ee)).toPrecision(idx+1+(period+ee)%period);if(!o.match(/[Ee]/)){var fakee=Math.floor(Math.log(val)*Math.LOG10E);if(o.indexOf(".")===-1)o=o.charAt(0)+"."+o.substr(1)+"E+"+(fakee-o.length+ee);else o+="E+"+(fakee-ee);o=o.replace(/\+-/,"-");}o=o.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/,function($$,$1,$2,$3){return $1+$2+$3.substr(0,(period+ee)%period)+"."+$3.substr(ee)+"E";});}else o=val.toExponential(idx);if(fmt.match(/E\+00$/)&&o.match(/e[+-]\d$/))o=o.substr(0,o.length-1)+"0"+o.charAt(o.length-1);if(fmt.match(/E\-/)&&o.match(/e\+/))o=o.replace(/e\+/,"e");return o.replace("e","E");}function write_num_int(type,fmt,val){if(type.charCodeAt(0)===40&&!fmt.match(closeparen)){var ffmt=fmt.replace(/\( */,"").replace(/ \)/,"").replace(/\)/,"");if(val>=0)return write_num_int('n',ffmt,val);return '('+write_num_int('n',ffmt,-val)+')';}if(fmt.charCodeAt(fmt.length-1)===44)return write_num_cm2(type,fmt,val);if(fmt.indexOf('%')!==-1)return write_num_pct2(type,fmt,val);if(fmt.indexOf('E')!==-1)return write_num_exp2(fmt,val);if(fmt.charCodeAt(0)===36)return "$"+write_num_int(type,fmt.substr(fmt.charAt(1)==' '?2:1),val);var o;var r,ri,ff,aval=Math.abs(val),sign=val<0?"-":"";if(fmt.match(/^00+$/))return sign+pad0(aval,fmt.length);if(fmt.match(/^[#?]+$/)){o=""+val;if(val===0)o="";return o.length>fmt.length?o:hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(frac1))return write_num_f2(r,aval,sign);if(fmt.match(/^#+0+$/))return sign+pad0(aval,fmt.length-fmt.indexOf("0"));if(r=fmt.match(dec1)){o=(""+val).replace(/^([^\.]+)$/,"$1."+hashq(r[1])).replace(/\.$/,"."+hashq(r[1]));o=o.replace(/\.(\d*)$/,function($$,$1){return "."+$1+fill("0",hashq(r[1]).length-$1.length);});return fmt.indexOf("0.")!==-1?o:o.replace(/^0\./,".");}fmt=fmt.replace(/^#+([0.])/,"$1");if(r=fmt.match(/^(0*)\.(#*)$/)){return sign+(""+aval).replace(/\.(\d*[1-9])0*$/,".$1").replace(/^(-?\d*)$/,"$1.").replace(/^0\./,r[1].length?"0.":".");}if(r=fmt.match(/^#{1,3},##0(\.?)$/))return sign+commaify(""+aval);if(r=fmt.match(/^#,##0\.([#0]*0)$/)){return val<0?"-"+write_num_int(type,fmt,-val):commaify(""+val)+"."+fill('0',r[1].length);}if(r=fmt.match(/^#,#*,#0/))return write_num_int(type,fmt.replace(/^#,#*,/,""),val);if(r=fmt.match(/^([0#]+)(\\?-([0#]+))+$/)){o=_strrev(write_num_int(type,fmt.replace(/[\\-]/g,""),val));ri=0;return _strrev(_strrev(fmt.replace(/\\/g,"")).replace(/[0#]/g,function(x){return ri<o.length?o.charAt(ri++):x==='0'?'0':"";}));}if(fmt.match(phone)){o=write_num_int(type,"##########",val);return "("+o.substr(0,3)+") "+o.substr(3,3)+"-"+o.substr(6);}var oa="";if(r=fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(r[4].length,7);ff=frac(aval,Math.pow(10,ri)-1,false);o=""+sign;oa=write_num("n",r[1],ff[1]);if(oa.charAt(oa.length-1)==" ")oa=oa.substr(0,oa.length-1)+"0";o+=oa+r[2]+"/"+r[3];oa=rpad_(ff[2],ri);if(oa.length<r[4].length)oa=hashq(r[4].substr(r[4].length-oa.length))+oa;o+=oa;return o;}if(r=fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)){ri=Math.min(Math.max(r[1].length,r[4].length),7);ff=frac(aval,Math.pow(10,ri)-1,true);return sign+(ff[0]||(ff[1]?"":"0"))+" "+(ff[1]?pad_(ff[1],ri)+r[2]+"/"+r[3]+rpad_(ff[2],ri):fill(" ",2*ri+1+r[2].length+r[3].length));}if(r=fmt.match(/^[#0?]+$/)){o=""+val;if(fmt.length<=o.length)return o;return hashq(fmt.substr(0,fmt.length-o.length))+o;}if(r=fmt.match(/^([#0]+)\.([#0]+)$/)){o=""+val.toFixed(Math.min(r[2].length,10)).replace(/([^0])0+$/,"$1");ri=o.indexOf(".");var lres=fmt.indexOf(".")-ri,rres=fmt.length-o.length-lres;return hashq(fmt.substr(0,lres)+o+fmt.substr(fmt.length-rres));}if(r=fmt.match(/^00,000\.([#0]*0)$/)){return val<0?"-"+write_num_int(type,fmt,-val):commaify(""+val).replace(/^\d,\d{3}$/,"0$&").replace(/^\d*$/,function($$){return "00,"+($$.length<3?pad0(0,3-$$.length):"")+$$;})+"."+pad0(0,r[1].length);}switch(fmt){case"###,###":case"##,###":case"#,###":var x=commaify(""+aval);return x!=="0"?sign+x:"";default:if(fmt.match(/\.[0#?]*$/))return write_num_int(type,fmt.slice(0,fmt.lastIndexOf(".")),val)+hashq(fmt.slice(fmt.lastIndexOf(".")));}throw new Error("unsupported format |"+fmt+"|");}return function write_num(type,fmt,val){return (val|0)===val?write_num_int(type,fmt,val):write_num_flt(type,fmt,val);};}();function split_fmt(fmt){var out=[];var in_str=false/*, cc*/;for(var i=0,j=0;i<fmt.length;++i)switch(/*cc=*/fmt.charCodeAt(i)){case 34:/* '"' */in_str=!in_str;break;case 95:case 42:case 92:/* '_' '*' '\\' */++i;break;case 59:/* ';' */out[out.length]=fmt.substr(j,i-j);j=i+1;}out[out.length]=fmt.substr(j);if(in_str===true)throw new Error("Format |"+fmt+"| unterminated string ");return out;}SSF._split=split_fmt;var abstime=/\[[HhMmSs\u0E0A\u0E19\u0E17]*\]/;function fmt_is_date(fmt){var i=0,/*cc = 0,*/c="",o="";while(i<fmt.length){switch(c=fmt.charAt(i)){case'G':if(isgeneral(fmt,i))i+=6;i++;break;case'"':for(;/*cc=*/fmt.charCodeAt(++i)!==34&&i<fmt.length;){/*empty*/}++i;break;case'\\':i+=2;break;case'_':i+=2;break;case'@':++i;break;case'B':case'b':if(fmt.charAt(i+1)==="1"||fmt.charAt(i+1)==="2")return true;/* falls through */case'M':case'D':case'Y':case'H':case'S':case'E':/* falls through */case'm':case'd':case'y':case'h':case's':case'e':case'g':return true;case'A':case'a':case'上':if(fmt.substr(i,3).toUpperCase()==="A/P")return true;if(fmt.substr(i,5).toUpperCase()==="AM/PM")return true;if(fmt.substr(i,5).toUpperCase()==="上午/下午")return true;++i;break;case'[':o=c;while(fmt.charAt(i++)!==']'&&i<fmt.length)o+=fmt.charAt(i);if(o.match(abstime))return true;break;case'.':/* falls through */case'0':case'#':while(i<fmt.length&&("0#?.,E+-%".indexOf(c=fmt.charAt(++i))>-1||c=='\\'&&fmt.charAt(i+1)=="-"&&"0#".indexOf(fmt.charAt(i+2))>-1)){/* empty */}break;case'?':while(fmt.charAt(++i)===c){/* empty */}break;case'*':++i;if(fmt.charAt(i)==' '||fmt.charAt(i)=='*')++i;break;case'(':case')':++i;break;case'1':case'2':case'3':case'4':case'5':case'6':case'7':case'8':case'9':while(i<fmt.length&&"0123456789".indexOf(fmt.charAt(++i))>-1){/* empty */}break;case' ':++i;break;default:++i;break;}}return false;}SSF.is_date=fmt_is_date;function eval_fmt(fmt,v,opts,flen){var out=[],o="",i=0,c="",lst='t',dt,j,cc;var hr='H';/* Tokenize */while(i<fmt.length){switch(c=fmt.charAt(i)){case'G':/* General */if(!isgeneral(fmt,i))throw new Error('unrecognized character '+c+' in '+fmt);out[out.length]={t:'G',v:'General'};i+=7;break;case'"':/* Literal text */for(o="";(cc=fmt.charCodeAt(++i))!==34&&i<fmt.length;)o+=String.fromCharCode(cc);out[out.length]={t:'t',v:o};++i;break;case'\\':var w=fmt.charAt(++i),t=w==="("||w===")"?w:'t';out[out.length]={t:t,v:w};++i;break;case'_':out[out.length]={t:'t',v:" "};i+=2;break;case'@':/* Text Placeholder */out[out.length]={t:'T',v:v};++i;break;case'B':case'b':if(fmt.charAt(i+1)==="1"||fmt.charAt(i+1)==="2"){if(dt==null){dt=parse_date_code(v,opts,fmt.charAt(i+1)==="2");if(dt==null)return "";}out[out.length]={t:'X',v:fmt.substr(i,2)};lst=c;i+=2;break;}/* falls through */case'M':case'D':case'Y':case'H':case'S':case'E':c=c.toLowerCase();/* falls through */case'm':case'd':case'y':case'h':case's':case'e':case'g':if(v<0)return "";if(dt==null){dt=parse_date_code(v,opts);if(dt==null)return "";}o=c;while(++i<fmt.length&&fmt.charAt(i).toLowerCase()===c)o+=c;if(c==='m'&&lst.toLowerCase()==='h')c='M';if(c==='h')c=hr;out[out.length]={t:c,v:o};lst=c;break;case'A':case'a':case'上':var q={t:c,v:c};if(dt==null)dt=parse_date_code(v,opts);if(fmt.substr(i,3).toUpperCase()==="A/P"){if(dt!=null)q.v=dt.H>=12?"P":"A";q.t='T';hr='h';i+=3;}else if(fmt.substr(i,5).toUpperCase()==="AM/PM"){if(dt!=null)q.v=dt.H>=12?"PM":"AM";q.t='T';i+=5;hr='h';}else if(fmt.substr(i,5).toUpperCase()==="上午/下午"){if(dt!=null)q.v=dt.H>=12?"下午":"上午";q.t='T';i+=5;hr='h';}else {q.t="t";++i;}if(dt==null&&q.t==='T')return "";out[out.length]=q;lst=c;break;case'[':o=c;while(fmt.charAt(i++)!==']'&&i<fmt.length)o+=fmt.charAt(i);if(o.slice(-1)!==']')throw 'unterminated "[" block: |'+o+'|';if(o.match(abstime)){if(dt==null){dt=parse_date_code(v,opts);if(dt==null)return "";}out[out.length]={t:'Z',v:o.toLowerCase()};lst=o.charAt(1);}else if(o.indexOf("$")>-1){o=(o.match(/\$([^-\[\]]*)/)||[])[1]||"$";if(!fmt_is_date(fmt))out[out.length]={t:'t',v:o};}break;/* Numbers */case'.':if(dt!=null){o=c;while(++i<fmt.length&&(c=fmt.charAt(i))==="0")o+=c;out[out.length]={t:'s',v:o};break;}/* falls through */case'0':case'#':o=c;while(++i<fmt.length&&"0#?.,E+-%".indexOf(c=fmt.charAt(i))>-1)o+=c;out[out.length]={t:'n',v:o};break;case'?':o=c;while(fmt.charAt(++i)===c)o+=c;out[out.length]={t:c,v:o};lst=c;break;case'*':++i;if(fmt.charAt(i)==' '||fmt.charAt(i)=='*')++i;break;// **
  case'(':case')':out[out.length]={t:flen===1?'t':c,v:c};++i;break;case'1':case'2':case'3':case'4':case'5':case'6':case'7':case'8':case'9':o=c;while(i<fmt.length&&"0123456789".indexOf(fmt.charAt(++i))>-1)o+=fmt.charAt(i);out[out.length]={t:'D',v:o};break;case' ':out[out.length]={t:c,v:c};++i;break;case'$':out[out.length]={t:'t',v:'$'};++i;break;default:if(",$-+/():!^&'~{}<>=€acfijklopqrtuvwxzP".indexOf(c)===-1)throw new Error('unrecognized character '+c+' in '+fmt);out[out.length]={t:'t',v:c};++i;break;}}/* Scan for date/time parts */var bt=0,ss0=0,ssm;for(i=out.length-1,lst='t';i>=0;--i){switch(out[i].t){case'h':case'H':out[i].t=hr;lst='h';if(bt<1)bt=1;break;case's':if(ssm=out[i].v.match(/\.0+$/))ss0=Math.max(ss0,ssm[0].length-1);if(bt<3)bt=3;/* falls through */case'd':case'y':case'M':case'e':lst=out[i].t;break;case'm':if(lst==='s'){out[i].t='M';if(bt<2)bt=2;}break;case'X':/*if(out[i].v === "B2");*/break;case'Z':if(bt<1&&out[i].v.match(/[Hh]/))bt=1;if(bt<2&&out[i].v.match(/[Mm]/))bt=2;if(bt<3&&out[i].v.match(/[Ss]/))bt=3;}}/* time rounding depends on presence of minute / second / usec fields */switch(bt){case 0:break;case 1:if(dt.u>=0.5){dt.u=0;++dt.S;}if(dt.S>=60){dt.S=0;++dt.M;}if(dt.M>=60){dt.M=0;++dt.H;}break;case 2:if(dt.u>=0.5){dt.u=0;++dt.S;}if(dt.S>=60){dt.S=0;++dt.M;}break;}/* replace fields */var nstr="",jj;for(i=0;i<out.length;++i){switch(out[i].t){case't':case'T':case' ':case'D':break;case'X':out[i].v="";out[i].t=";";break;case'd':case'm':case'y':case'h':case'H':case'M':case's':case'e':case'b':case'Z':out[i].v=write_date(out[i].t.charCodeAt(0),out[i].v,dt,ss0);out[i].t='t';break;case'n':case'?':jj=i+1;while(out[jj]!=null&&((c=out[jj].t)==="?"||c==="D"||(c===" "||c==="t")&&out[jj+1]!=null&&(out[jj+1].t==='?'||out[jj+1].t==="t"&&out[jj+1].v==='/')||out[i].t==='('&&(c===' '||c==='n'||c===')')||c==='t'&&(out[jj].v==='/'||out[jj].v===' '&&out[jj+1]!=null&&out[jj+1].t=='?'))){out[i].v+=out[jj].v;out[jj]={v:"",t:";"};++jj;}nstr+=out[i].v;i=jj-1;break;case'G':out[i].t='t';out[i].v=general_fmt(v,opts);break;}}var vv="",myv,ostr;if(nstr.length>0){if(nstr.charCodeAt(0)==40)/* '(' */{myv=v<0&&nstr.charCodeAt(0)===45?-v:v;ostr=write_num('n',nstr,myv);}else {myv=v<0&&flen>1?-v:v;ostr=write_num('n',nstr,myv);if(myv<0&&out[0]&&out[0].t=='t'){ostr=ostr.substr(1);out[0].v="-"+out[0].v;}}jj=ostr.length-1;var decpt=out.length;for(i=0;i<out.length;++i)if(out[i]!=null&&out[i].t!='t'&&out[i].v.indexOf(".")>-1){decpt=i;break;}var lasti=out.length;if(decpt===out.length&&ostr.indexOf("E")===-1){for(i=out.length-1;i>=0;--i){if(out[i]==null||'n?'.indexOf(out[i].t)===-1)continue;if(jj>=out[i].v.length-1){jj-=out[i].v.length;out[i].v=ostr.substr(jj+1,out[i].v.length);}else if(jj<0)out[i].v="";else {out[i].v=ostr.substr(0,jj+1);jj=-1;}out[i].t='t';lasti=i;}if(jj>=0&&lasti<out.length)out[lasti].v=ostr.substr(0,jj+1)+out[lasti].v;}else if(decpt!==out.length&&ostr.indexOf("E")===-1){jj=ostr.indexOf(".")-1;for(i=decpt;i>=0;--i){if(out[i]==null||'n?'.indexOf(out[i].t)===-1)continue;j=out[i].v.indexOf(".")>-1&&i===decpt?out[i].v.indexOf(".")-1:out[i].v.length-1;vv=out[i].v.substr(j+1);for(;j>=0;--j){if(jj>=0&&(out[i].v.charAt(j)==="0"||out[i].v.charAt(j)==="#"))vv=ostr.charAt(jj--)+vv;}out[i].v=vv;out[i].t='t';lasti=i;}if(jj>=0&&lasti<out.length)out[lasti].v=ostr.substr(0,jj+1)+out[lasti].v;jj=ostr.indexOf(".")+1;for(i=decpt;i<out.length;++i){if(out[i]==null||'n?('.indexOf(out[i].t)===-1&&i!==decpt)continue;j=out[i].v.indexOf(".")>-1&&i===decpt?out[i].v.indexOf(".")+1:0;vv=out[i].v.substr(0,j);for(;j<out[i].v.length;++j){if(jj<ostr.length)vv+=ostr.charAt(jj++);}out[i].v=vv;out[i].t='t';lasti=i;}}}for(i=0;i<out.length;++i)if(out[i]!=null&&'n?'.indexOf(out[i].t)>-1){myv=flen>1&&v<0&&i>0&&out[i-1].v==="-"?-v:v;out[i].v=write_num(out[i].t,out[i].v,myv);out[i].t='t';}var retval="";for(i=0;i!==out.length;++i)if(out[i]!=null)retval+=out[i].v;return retval;}SSF._eval=eval_fmt;var cfregex=/\[[=<>]/;var cfregex2=/\[(=|>[=]?|<[>=]?)(-?\d+(?:\.\d*)?)\]/;function chkcond(v,rr){if(rr==null)return false;var thresh=parseFloat(rr[2]);switch(rr[1]){case"=":if(v==thresh)return true;break;case">":if(v>thresh)return true;break;case"<":if(v<thresh)return true;break;case"<>":if(v!=thresh)return true;break;case">=":if(v>=thresh)return true;break;case"<=":if(v<=thresh)return true;break;}return false;}function choose_fmt(f,v){var fmt=split_fmt(f);var l=fmt.length,lat=fmt[l-1].indexOf("@");if(l<4&&lat>-1)--l;if(fmt.length>4)throw new Error("cannot find right format for |"+fmt.join("|")+"|");if(typeof v!=="number")return [4,fmt.length===4||lat>-1?fmt[fmt.length-1]:"@"];switch(fmt.length){case 1:fmt=lat>-1?["General","General","General",fmt[0]]:[fmt[0],fmt[0],fmt[0],"@"];break;case 2:fmt=lat>-1?[fmt[0],fmt[0],fmt[0],fmt[1]]:[fmt[0],fmt[1],fmt[0],"@"];break;case 3:fmt=lat>-1?[fmt[0],fmt[1],fmt[0],fmt[2]]:[fmt[0],fmt[1],fmt[2],"@"];break;}var ff=v>0?fmt[0]:v<0?fmt[1]:fmt[2];if(fmt[0].indexOf("[")===-1&&fmt[1].indexOf("[")===-1)return [l,ff];if(fmt[0].match(cfregex)!=null||fmt[1].match(cfregex)!=null){var m1=fmt[0].match(cfregex2);var m2=fmt[1].match(cfregex2);return chkcond(v,m1)?[l,fmt[0]]:chkcond(v,m2)?[l,fmt[1]]:[l,fmt[m1!=null&&m2!=null?2:1]];}return [l,ff];}function format(fmt,v,o){if(o==null)o={};var sfmt="";switch(typeof fmt){case"string":if(fmt=="m/d/yy"&&o.dateNF)sfmt=o.dateNF;else sfmt=fmt;break;case"number":if(fmt==14&&o.dateNF)sfmt=o.dateNF;else sfmt=(o.table!=null?o.table:table_fmt)[fmt];if(sfmt==null)sfmt=o.table&&o.table[default_map[fmt]]||table_fmt[default_map[fmt]];if(sfmt==null)sfmt=default_str[fmt]||"General";break;}if(isgeneral(sfmt,0))return general_fmt(v,o);if(v instanceof Date)v=datenum_local(v,o.date1904);var f=choose_fmt(sfmt,v);if(isgeneral(f[1]))return general_fmt(v,o);if(v===true)v="TRUE";else if(v===false)v="FALSE";else if(v===""||v==null)return "";return eval_fmt(f[1],v,o,f[0]);}function load_entry(fmt,idx){if(typeof idx!='number'){idx=+idx||-1;for(var i=0;i<0x0188;++i){if(table_fmt[i]==undefined){if(idx<0)idx=i;continue;}if(table_fmt[i]==fmt){idx=i;break;}}if(idx<0)idx=0x187;}table_fmt[idx]=fmt;return idx;}SSF.load=load_entry;SSF._table=table_fmt;SSF.get_table=function get_table(){return table_fmt;};SSF.load_table=function load_table(tbl){for(var i=0;i!=0x0188;++i)if(tbl[i]!==undefined)load_entry(tbl[i],i);};SSF.init_table=init_table;SSF.format=format;};make_ssf(SSF);/* map from xlml named formats to SSF TODO: localize */var XLMLFormatMap/*{[string]:string}*/={"General Number":"General","General Date":SSF._table[22],"Long Date":"dddd, mmmm dd, yyyy","Medium Date":SSF._table[15],"Short Date":SSF._table[14],"Long Time":SSF._table[19],"Medium Time":SSF._table[18],"Short Time":SSF._table[20],"Currency":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"Fixed":SSF._table[2],"Standard":SSF._table[4],"Percent":SSF._table[10],"Scientific":SSF._table[11],"Yes/No":'"Yes";"Yes";"No";@',"True/False":'"True";"True";"False";@',"On/Off":'"Yes";"Yes";"No";@'};var SSFImplicit/*{[number]:string}*/={"5":'"$"#,##0_);\\("$"#,##0\\)',"6":'"$"#,##0_);[Red]\\("$"#,##0\\)',"7":'"$"#,##0.00_);\\("$"#,##0.00\\)',"8":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"23":'General',"24":'General',"25":'General',"26":'General',"27":'m/d/yy',"28":'m/d/yy',"29":'m/d/yy',"30":'m/d/yy',"31":'m/d/yy',"32":'h:mm:ss',"33":'h:mm:ss',"34":'h:mm:ss',"35":'h:mm:ss',"36":'m/d/yy',"41":'_(* #,##0_);_(* \(#,##0\);_(* "-"_);_(@_)',"42":'_("$"* #,##0_);_("$"* \(#,##0\);_("$"* "-"_);_(@_)',"43":'_(* #,##0.00_);_(* \(#,##0.00\);_(* "-"??_);_(@_)',"44":'_("$"* #,##0.00_);_("$"* \(#,##0.00\);_("$"* "-"??_);_(@_)',"50":'m/d/yy',"51":'m/d/yy',"52":'m/d/yy',"53":'m/d/yy',"54":'m/d/yy',"55":'m/d/yy',"56":'m/d/yy',"57":'m/d/yy',"58":'m/d/yy',"59":'0',"60":'0.00',"61":'#,##0',"62":'#,##0.00',"63":'"$"#,##0_);\\("$"#,##0\\)',"64":'"$"#,##0_);[Red]\\("$"#,##0\\)',"65":'"$"#,##0.00_);\\("$"#,##0.00\\)',"66":'"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',"67":'0%',"68":'0.00%',"69":'# ?/?',"70":'# ??/??',"71":'m/d/yy',"72":'m/d/yy',"73":'d-mmm-yy',"74":'d-mmm',"75":'mmm-yy',"76":'h:mm',"77":'h:mm:ss',"78":'m/d/yy h:mm',"79":'mm:ss',"80":'[h]:mm:ss',"81":'mmss.0'};/* dateNF parse TODO: move to SSF */var dateNFregex=/[dD]+|[mM]+|[yYeE]+|[Hh]+|[Ss]+/g;function dateNF_regex(dateNF){var fmt=typeof dateNF=="number"?SSF._table[dateNF]:dateNF;fmt=fmt.replace(dateNFregex,"(\\d+)");return new RegExp("^"+fmt+"$");}function dateNF_fix(str,dateNF,match){var Y=-1,m=-1,d=-1,H=-1,M=-1,S=-1;(dateNF.match(dateNFregex)||[]).forEach(function(n,i){var v=parseInt(match[i+1],10);switch(n.toLowerCase().charAt(0)){case'y':Y=v;break;case'd':d=v;break;case'h':H=v;break;case's':S=v;break;case'm':if(H>=0)M=v;else m=v;break;}});if(S>=0&&M==-1&&m>=0){M=m;m=-1;}var datestr=(""+(Y>=0?Y:new Date().getFullYear())).slice(-4)+"-"+("00"+(m>=1?m:1)).slice(-2)+"-"+("00"+(d>=1?d:1)).slice(-2);if(datestr.length==7)datestr="0"+datestr;if(datestr.length==8)datestr="20"+datestr;var timestr=("00"+(H>=0?H:0)).slice(-2)+":"+("00"+(M>=0?M:0)).slice(-2)+":"+("00"+(S>=0?S:0)).slice(-2);if(H==-1&&M==-1&&S==-1)return datestr;if(Y==-1&&m==-1&&d==-1)return timestr;return datestr