/**
 * spectra-processor
 * @version v3.1.0
 * @link https://github.com/cheminfo/spectra-processor#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.spectraProcessor = {}));
})(this, (function (exports) { 'use strict';

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray$1(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    var libEsm = /*#__PURE__*/Object.freeze({
        __proto__: null,
        isAnyArray: isAnyArray$1
    });

    /**
     * This function
     * @param output - undefined or a new array
     * @param length - length of the output array
     * @returns
     */
    function getOutputArray(output, length) {
      if (typeof output !== 'undefined') {
        if (!isAnyArray$1(output)) {
          throw new TypeError('output option must be an array if specified');
        }
        if (output.length !== length) {
          throw new TypeError('the output array does not have the correct length');
        }
        return output;
      } else {
        return new Float64Array(length);
      }
    }

    /**
     * Calculates the median of an array.
     *
     * @param input - Array containing values
     * @returns - median
     */
    function xMedian(input) {
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      const array = input.slice();
      let low = 0;
      let high = array.length - 1;
      let middle = 0;
      let currentLow = 0;
      let currentHigh = 0;
      const median = calcMiddle(low, high);
      while (true) {
        if (high <= low) {
          return array[median];
        }
        if (high === low + 1) {
          if (array[low] > array[high]) {
            swap(array, low, high);
          }
          return array[median];
        }
        // Find median of low, middle and high items; swap into position low
        middle = calcMiddle(low, high);
        if (array[middle] > array[high]) swap(array, middle, high);
        if (array[low] > array[high]) swap(array, low, high);
        if (array[middle] > array[low]) swap(array, middle, low);
        // Swap low item (now in position middle) into position (low+1)
        swap(array, middle, low + 1);
        // Nibble from each end towards middle, swapping items when stuck
        currentLow = low + 1;
        currentHigh = high;
        while (true) {
          do currentLow++; while (array[low] > array[currentLow]);
          do currentHigh--; while (array[currentHigh] > array[low]);
          if (currentHigh < currentLow) {
            break;
          }
          swap(array, currentLow, currentHigh);
        }
        // Swap middle item (in position low) back into correct position
        swap(array, low, currentHigh);
        // Re-set active partition
        if (currentHigh <= median) {
          low = currentLow;
        }
        if (currentHigh >= median) {
          high = currentHigh - 1;
        }
      }
    }
    function swap(array, i, j) {
      const temp = array[j];
      array[j] = array[i];
      array[i] = temp;
    }
    function calcMiddle(i, j) {
      return Math.floor((i + j) / 2);
    }

    /**
     * Checks if input is of type array.
     *
     * @param input - input
     * @param options
     */
    function xCheck(input, options = {}) {
      const {
        minLength
      } = options;
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      if (typeof input[0] !== 'number') {
        throw new TypeError('input must contain numbers');
      }
      if (minLength && input.length < minLength) {
        throw new Error(`input must have a length of at least ${minLength}`);
      }
    }

    /**
     * Returns the closest index of a `target`
     *
     * @param array - array of numbers
     * @param target - target
     * @param options
     * @returns - closest index
     */
    function xFindClosestIndex(array, target, options = {}) {
      const {
        sorted = true
      } = options;
      if (sorted) {
        let low = 0;
        let high = array.length - 1;
        let middle = 0;
        while (high - low > 1) {
          middle = low + (high - low >> 1);
          if (array[middle] < target) {
            low = middle;
          } else if (array[middle] > target) {
            high = middle;
          } else {
            return middle;
          }
        }
        if (low < array.length - 1) {
          if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
            return low;
          } else {
            return low + 1;
          }
        } else {
          return low;
        }
      } else {
        let index = 0;
        let diff = Number.POSITIVE_INFINITY;
        for (let i = 0; i < array.length; i++) {
          const currentDiff = Math.abs(array[i] - target);
          if (currentDiff < diff) {
            diff = currentDiff;
            index = i;
          }
        }
        return index;
      }
    }

    /**
     * Returns an object with {fromIndex, toIndex} for a specific from / to
     *
     * @param x - array of numbers
     * @param options - Options
     */
    function xGetFromToIndex(x, options = {}) {
      let {
        fromIndex,
        toIndex
      } = options;
      const {
        from,
        to
      } = options;
      if (typeof fromIndex === 'undefined') {
        if (typeof from !== 'undefined') {
          fromIndex = xFindClosestIndex(x, from);
        } else {
          fromIndex = 0;
        }
      }
      if (typeof toIndex === 'undefined') {
        if (typeof to !== 'undefined') {
          toIndex = xFindClosestIndex(x, to);
        } else {
          toIndex = x.length - 1;
        }
      }
      if (fromIndex < 0) fromIndex = 0;
      if (toIndex < 0) toIndex = 0;
      if (fromIndex >= x.length) fromIndex = x.length - 1;
      if (toIndex >= x.length) toIndex = x.length - 1;
      if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
      return {
        fromIndex,
        toIndex
      };
    }

    function xAbsoluteSum(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let sum = 0;
      for (let i = fromIndex; i <= toIndex; i++) {
        if (array[i] < 0) {
          sum -= array[i];
        } else {
          sum += array[i];
        }
      }
      return sum;
    }

    /**
     * This function xAdd the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - the first array
     * @param array2 - the second array or number
     */
    function xAdd(array1, array2) {
      let isConstant = false;
      let constant = 0;
      if (isAnyArray$1(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = array2;
      }
      const array3 = new Float64Array(array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + array2[i];
        }
      }
      return array3;
    }

    /**
     * Will apply a function on each element of the array described as a string
     * By default we will use as variable 'x'
     * In front of sequence of lowercase we will add 'Math.'. This allows to write
     * `sin(x) + cos(x)` and it will be replace internally by (x) => (Math.sin(x) + Math.cos(x))
     * @param array
     * @param options
     * @returns
     */
    function xApplyFunctionStr(array, options = {}) {
      const {
        variableLabel = 'x',
        fctString = variableLabel
      } = options;
      const fct = new Function(variableLabel, `return Number(${fctString.replace(/(?<before>^|\W)(?<after>[\da-z]{2,}\()/g, '$<before>Math.$<after>').replace(/Math\.Math/g, 'Math')})`);
      const toReturn = Float64Array.from(array);
      for (let i = 0; i < array.length; i++) {
        toReturn[i] = fct(array[i]);
        if (Number.isNaN(toReturn[i])) {
          throw new Error(`The callback ${fctString} does not return a number: ${array[i]}`);
        }
      }
      return toReturn;
    }

    /**
     * This function xMultiply the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - first array
     * @param array2 - second array
     * @param options - options
     */
    function xMultiply(array1, array2, options = {}) {
      let isConstant = false;
      let constant = 0;
      if (isAnyArray$1(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = Number(array2);
      }
      const array3 = getOutputArray(options.output, array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] * constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] * array2[i];
        }
      }
      return array3;
    }

    /**
     * Calculating the box plot of the array
     *
     * @param array - data
     * @param options
     */
    function xBoxPlot(array, options = {}) {
      const {
        allowSmallArray = false
      } = options;
      if (array.length < 5) {
        if (allowSmallArray) {
          if (array.length < 1) {
            throw new Error('can not calculate info if array is empty');
          }
        } else {
          throw new Error('can not calculate info if array contains less than 5 elements');
        }
      }
      array = Float64Array.from(array).sort();
      const info = {
        q1: 0,
        median: 0,
        q3: 0,
        min: array[0],
        max: array[array.length - 1]
      };
      let q1max, q3min;
      if (array.length % 2 === 1) {
        // odd
        const middle = (array.length - 1) / 2;
        info.median = array[middle];
        q1max = Math.max(middle - 1, 0);
        q3min = Math.min(middle + 1, array.length - 1);
      } else {
        // even
        q3min = array.length / 2;
        q1max = q3min - 1;
        info.median = (array[q1max] + array[q3min]) / 2;
      }
      if (q1max % 2 === 0) {
        info.q1 = array[q1max / 2];
        info.q3 = array[(array.length + q3min - 1) / 2];
      } else {
        info.q1 = (array[(q1max + 1) / 2] + array[(q1max - 1) / 2]) / 2;
        const middleOver = (array.length + q3min) / 2;
        info.q3 = (array[middleOver] + array[middleOver - 1]) / 2;
      }
      return info;
    }

    /**
     * Calculates the correlation between 2 vectors
     * https://en.wikipedia.org/wiki/Correlation_and_dependence
     *
     * @param A - first array
     * @param B - sencond array
     */
    function xCorrelation(A, B) {
      const n = A.length;
      let sumA = 0;
      let sumA2 = 0;
      let sumB = 0;
      let sumB2 = 0;
      let sumAB = 0;
      for (let i = 0; i < n; i++) {
        const a = A[i];
        const b = B[i];
        sumA += a;
        sumA2 += a ** 2;
        sumB += b;
        sumB2 += b ** 2;
        sumAB += a * b;
      }
      return (n * sumAB - sumA * sumB) / (Math.sqrt(n * sumA2 - sumA ** 2) * Math.sqrt(n * sumB2 - sumB ** 2));
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getAugmentedNamespace(n) {
    	if (n.__esModule) return n;
    	var a = Object.defineProperty({}, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    function commonjsRequire (path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    var matrix = {};

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

    function max$3(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var maxValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }
      return maxValue;
    }

    function min$3(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var minValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }
      return minValue;
    }

    function rescale$2(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      } else if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var output;
      if (options.output !== undefined) {
        if (!isAnyArray$1(options.output)) {
          throw new TypeError('output option must be an array if specified');
        }
        output = options.output;
      } else {
        output = new Array(input.length);
      }
      var currentMin = min$3(input);
      var currentMax = max$3(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
      if (minValue >= maxValue) {
        throw new RangeError('min option must be smaller than max option');
      }
      var factor = (maxValue - minValue) / (currentMax - currentMin);
      for (var i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + minValue;
      }
      return output;
    }

    var libEs6 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        'default': rescale$2
    });

    var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

    Object.defineProperty(matrix, '__esModule', {
      value: true
    });
    var isAnyArray = require$$0;
    var rescale$1 = require$$1;
    const indent = ' '.repeat(2);
    const indentData = ' '.repeat(4);

    /**
     * @this {Matrix}
     * @returns {string}
     */
    function inspectMatrix() {
      return inspectMatrixWithOptions(this);
    }
    function inspectMatrixWithOptions(matrix, options = {}) {
      const {
        maxRows = 15,
        maxColumns = 10,
        maxNumSize = 8,
        padMinus = 'auto'
      } = options;
      return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
    }
    function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
      const {
        rows,
        columns
      } = matrix;
      const maxI = Math.min(rows, maxRows);
      const maxJ = Math.min(columns, maxColumns);
      const result = [];
      if (padMinus === 'auto') {
        padMinus = false;
        loop: for (let i = 0; i < maxI; i++) {
          for (let j = 0; j < maxJ; j++) {
            if (matrix.get(i, j) < 0) {
              padMinus = true;
              break loop;
            }
          }
        }
      }
      for (let i = 0; i < maxI; i++) {
        let line = [];
        for (let j = 0; j < maxJ; j++) {
          line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
        }
        result.push(`${line.join(' ')}`);
      }
      if (maxJ !== columns) {
        result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
      }
      if (maxI !== rows) {
        result.push(`... ${rows - maxRows} more rows`);
      }
      return result.join(`\n${indentData}`);
    }
    function formatNumber(num, maxNumSize, padMinus) {
      return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
    }
    function formatNumber2(num, len) {
      // small.length numbers should be as is
      let str = num.toString();
      if (str.length <= len) return str;

      // (7)'0.00123' is better then (7)'1.23e-2'
      // (8)'0.000123' is worse then (7)'1.23e-3',
      let fix = num.toFixed(len);
      if (fix.length > len) {
        fix = num.toFixed(Math.max(0, len - (fix.length - len)));
      }
      if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
        return fix;
      }

      // well, if it's still too long the user should've used longer numbers
      let exp = num.toExponential(len);
      if (exp.length > len) {
        exp = num.toExponential(Math.max(0, len - (exp.length - len)));
      }
      return exp.slice(0);
    }
    function installMathOperations(AbstractMatrix, Matrix) {
      AbstractMatrix.prototype.add = function add(value) {
        if (typeof value === 'number') return this.addS(value);
        return this.addM(value);
      };
      AbstractMatrix.prototype.addS = function addS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.addM = function addM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.add = function add(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.add(value);
      };
      AbstractMatrix.prototype.sub = function sub(value) {
        if (typeof value === 'number') return this.subS(value);
        return this.subM(value);
      };
      AbstractMatrix.prototype.subS = function subS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.subM = function subM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.sub = function sub(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sub(value);
      };
      AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
      AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
      AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
      AbstractMatrix.subtract = AbstractMatrix.sub;
      AbstractMatrix.prototype.mul = function mul(value) {
        if (typeof value === 'number') return this.mulS(value);
        return this.mulM(value);
      };
      AbstractMatrix.prototype.mulS = function mulS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.mulM = function mulM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mul = function mul(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mul(value);
      };
      AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
      AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
      AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
      AbstractMatrix.multiply = AbstractMatrix.mul;
      AbstractMatrix.prototype.div = function div(value) {
        if (typeof value === 'number') return this.divS(value);
        return this.divM(value);
      };
      AbstractMatrix.prototype.divS = function divS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.divM = function divM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.div = function div(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.div(value);
      };
      AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
      AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
      AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
      AbstractMatrix.divide = AbstractMatrix.div;
      AbstractMatrix.prototype.mod = function mod(value) {
        if (typeof value === 'number') return this.modS(value);
        return this.modM(value);
      };
      AbstractMatrix.prototype.modS = function modS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.modM = function modM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mod = function mod(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mod(value);
      };
      AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
      AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
      AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
      AbstractMatrix.modulus = AbstractMatrix.mod;
      AbstractMatrix.prototype.and = function and(value) {
        if (typeof value === 'number') return this.andS(value);
        return this.andM(value);
      };
      AbstractMatrix.prototype.andS = function andS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.andM = function andM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.and = function and(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.and(value);
      };
      AbstractMatrix.prototype.or = function or(value) {
        if (typeof value === 'number') return this.orS(value);
        return this.orM(value);
      };
      AbstractMatrix.prototype.orS = function orS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.orM = function orM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.or = function or(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.or(value);
      };
      AbstractMatrix.prototype.xor = function xor(value) {
        if (typeof value === 'number') return this.xorS(value);
        return this.xorM(value);
      };
      AbstractMatrix.prototype.xorS = function xorS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.xorM = function xorM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.xor = function xor(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.xor(value);
      };
      AbstractMatrix.prototype.leftShift = function leftShift(value) {
        if (typeof value === 'number') return this.leftShiftS(value);
        return this.leftShiftM(value);
      };
      AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.leftShift = function leftShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.leftShift(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
        if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
        return this.signPropagatingRightShiftM(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.signPropagatingRightShift(value);
      };
      AbstractMatrix.prototype.rightShift = function rightShift(value) {
        if (typeof value === 'number') return this.rightShiftS(value);
        return this.rightShiftM(value);
      };
      AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.rightShift = function rightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.rightShift(value);
      };
      AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
      AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
      AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
      AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
      AbstractMatrix.prototype.not = function not() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, ~this.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.not = function not(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.not();
      };
      AbstractMatrix.prototype.abs = function abs() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.abs(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.abs = function abs(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.abs();
      };
      AbstractMatrix.prototype.acos = function acos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acos = function acos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acos();
      };
      AbstractMatrix.prototype.acosh = function acosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acosh = function acosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acosh();
      };
      AbstractMatrix.prototype.asin = function asin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asin = function asin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asin();
      };
      AbstractMatrix.prototype.asinh = function asinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asinh = function asinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asinh();
      };
      AbstractMatrix.prototype.atan = function atan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atan = function atan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atan();
      };
      AbstractMatrix.prototype.atanh = function atanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atanh = function atanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atanh();
      };
      AbstractMatrix.prototype.cbrt = function cbrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cbrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cbrt = function cbrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cbrt();
      };
      AbstractMatrix.prototype.ceil = function ceil() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.ceil(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.ceil = function ceil(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.ceil();
      };
      AbstractMatrix.prototype.clz32 = function clz32() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.clz32(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.clz32 = function clz32(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.clz32();
      };
      AbstractMatrix.prototype.cos = function cos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cos = function cos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cos();
      };
      AbstractMatrix.prototype.cosh = function cosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cosh = function cosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cosh();
      };
      AbstractMatrix.prototype.exp = function exp() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.exp(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.exp = function exp(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.exp();
      };
      AbstractMatrix.prototype.expm1 = function expm1() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.expm1(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.expm1 = function expm1(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.expm1();
      };
      AbstractMatrix.prototype.floor = function floor() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.floor(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.floor = function floor(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.floor();
      };
      AbstractMatrix.prototype.fround = function fround() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.fround(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.fround = function fround(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.fround();
      };
      AbstractMatrix.prototype.log = function log() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log = function log(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log();
      };
      AbstractMatrix.prototype.log1p = function log1p() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log1p(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log1p = function log1p(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log1p();
      };
      AbstractMatrix.prototype.log10 = function log10() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log10(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log10 = function log10(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log10();
      };
      AbstractMatrix.prototype.log2 = function log2() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log2(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log2 = function log2(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log2();
      };
      AbstractMatrix.prototype.round = function round() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.round(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.round = function round(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.round();
      };
      AbstractMatrix.prototype.sign = function sign() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sign(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sign = function sign(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sign();
      };
      AbstractMatrix.prototype.sin = function sin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sin = function sin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sin();
      };
      AbstractMatrix.prototype.sinh = function sinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sinh = function sinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sinh();
      };
      AbstractMatrix.prototype.sqrt = function sqrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sqrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sqrt = function sqrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sqrt();
      };
      AbstractMatrix.prototype.tan = function tan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tan = function tan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tan();
      };
      AbstractMatrix.prototype.tanh = function tanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tanh = function tanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tanh();
      };
      AbstractMatrix.prototype.trunc = function trunc() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.trunc(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.trunc = function trunc(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.trunc();
      };
      AbstractMatrix.pow = function pow(matrix, arg0) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.pow(arg0);
      };
      AbstractMatrix.prototype.pow = function pow(value) {
        if (typeof value === 'number') return this.powS(value);
        return this.powM(value);
      };
      AbstractMatrix.prototype.powS = function powS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.powM = function powM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** matrix.get(i, j));
          }
        }
        return this;
      };
    }

    /**
     * @private
     * Check that a row index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkRowIndex(matrix, index, outer) {
      let max = outer ? matrix.rows : matrix.rows - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Row index out of range');
      }
    }

    /**
     * @private
     * Check that a column index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkColumnIndex(matrix, index, outer) {
      let max = outer ? matrix.columns : matrix.columns - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Column index out of range');
      }
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkRowVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.columns) {
        throw new RangeError('vector size must be the same as the number of columns');
      }
      return vector;
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkColumnVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.rows) {
        throw new RangeError('vector size must be the same as the number of rows');
      }
      return vector;
    }
    function checkRowIndices(matrix, rowIndices) {
      if (!isAnyArray.isAnyArray(rowIndices)) {
        throw new TypeError('row indices must be an array');
      }
      for (let i = 0; i < rowIndices.length; i++) {
        if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
          throw new RangeError('row indices are out of range');
        }
      }
    }
    function checkColumnIndices(matrix, columnIndices) {
      if (!isAnyArray.isAnyArray(columnIndices)) {
        throw new TypeError('column indices must be an array');
      }
      for (let i = 0; i < columnIndices.length; i++) {
        if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
          throw new RangeError('column indices are out of range');
        }
      }
    }
    function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
      if (arguments.length !== 5) {
        throw new RangeError('expected 4 arguments');
      }
      checkNumber('startRow', startRow);
      checkNumber('endRow', endRow);
      checkNumber('startColumn', startColumn);
      checkNumber('endColumn', endColumn);
      if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
        throw new RangeError('Submatrix indices are out of range');
      }
    }
    function newArray(length, value = 0) {
      let array = [];
      for (let i = 0; i < length; i++) {
        array.push(value);
      }
      return array;
    }
    function checkNumber(name, value) {
      if (typeof value !== 'number') {
        throw new TypeError(`${name} must be a number`);
      }
    }
    function checkNonEmpty(matrix) {
      if (matrix.isEmpty()) {
        throw new Error('Empty matrix has no elements to index');
      }
    }
    function sumByRow(matrix) {
      let sum = newArray(matrix.rows);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumByColumn(matrix) {
      let sum = newArray(matrix.columns);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumAll(matrix) {
      let v = 0;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v += matrix.get(i, j);
        }
      }
      return v;
    }
    function productByRow(matrix) {
      let sum = newArray(matrix.rows, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productByColumn(matrix) {
      let sum = newArray(matrix.columns, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productAll(matrix) {
      let v = 1;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v *= matrix.get(i, j);
        }
      }
      return v;
    }
    function varianceByRow(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let i = 0; i < rows; i++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean[i];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / cols) / cols);
        }
      }
      return variance;
    }
    function varianceByColumn(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let j = 0; j < cols; j++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let i = 0; i < rows; i++) {
          x = matrix.get(i, j) - mean[j];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / rows) / rows);
        }
      }
      return variance;
    }
    function varianceAll(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const size = rows * cols;
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean;
          sum1 += x;
          sum2 += x * x;
        }
      }
      if (unbiased) {
        return (sum2 - sum1 * sum1 / size) / (size - 1);
      } else {
        return (sum2 - sum1 * sum1 / size) / size;
      }
    }
    function centerByRow(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[i]);
        }
      }
    }
    function centerByColumn(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[j]);
        }
      }
    }
    function centerAll(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean);
        }
      }
    }
    function getScaleByRow(matrix) {
      const scale = [];
      for (let i = 0; i < matrix.rows; i++) {
        let sum = 0;
        for (let j = 0; j < matrix.columns; j++) {
          sum += matrix.get(i, j) ** 2 / (matrix.columns - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByRow(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[i]);
        }
      }
    }
    function getScaleByColumn(matrix) {
      const scale = [];
      for (let j = 0; j < matrix.columns; j++) {
        let sum = 0;
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / (matrix.rows - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByColumn(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[j]);
        }
      }
    }
    function getScaleAll(matrix) {
      const divider = matrix.size - 1;
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / divider;
        }
      }
      return Math.sqrt(sum);
    }
    function scaleAll(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale);
        }
      }
    }
    class AbstractMatrix {
      static from1DArray(newRows, newColumns, newData) {
        let length = newRows * newColumns;
        if (length !== newData.length) {
          throw new RangeError('data length does not match given dimensions');
        }
        let newMatrix = new Matrix$1(newRows, newColumns);
        for (let row = 0; row < newRows; row++) {
          for (let column = 0; column < newColumns; column++) {
            newMatrix.set(row, column, newData[row * newColumns + column]);
          }
        }
        return newMatrix;
      }
      static rowVector(newData) {
        let vector = new Matrix$1(1, newData.length);
        for (let i = 0; i < newData.length; i++) {
          vector.set(0, i, newData[i]);
        }
        return vector;
      }
      static columnVector(newData) {
        let vector = new Matrix$1(newData.length, 1);
        for (let i = 0; i < newData.length; i++) {
          vector.set(i, 0, newData[i]);
        }
        return vector;
      }
      static zeros(rows, columns) {
        return new Matrix$1(rows, columns);
      }
      static ones(rows, columns) {
        return new Matrix$1(rows, columns).fill(1);
      }
      static rand(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          random = Math.random
        } = options;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.set(i, j, random());
          }
        }
        return matrix;
      }
      static randInt(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1000,
          random = Math.random
        } = options;
        if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
        if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let interval = max - min;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            let value = min + Math.round(random() * interval);
            matrix.set(i, j, value);
          }
        }
        return matrix;
      }
      static eye(rows, columns, value) {
        if (columns === undefined) columns = rows;
        if (value === undefined) value = 1;
        let min = Math.min(rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, value);
        }
        return matrix;
      }
      static diag(data, rows, columns) {
        let l = data.length;
        if (rows === undefined) rows = l;
        if (columns === undefined) columns = rows;
        let min = Math.min(l, rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, data[i]);
        }
        return matrix;
      }
      static min(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static max(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new this(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static checkMatrix(value) {
        return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
      }
      static isMatrix(value) {
        return value != null && value.klass === 'Matrix';
      }
      get size() {
        return this.rows * this.columns;
      }
      apply(callback) {
        if (typeof callback !== 'function') {
          throw new TypeError('callback must be a function');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            callback.call(this, i, j);
          }
        }
        return this;
      }
      to1DArray() {
        let array = [];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            array.push(this.get(i, j));
          }
        }
        return array;
      }
      to2DArray() {
        let copy = [];
        for (let i = 0; i < this.rows; i++) {
          copy.push([]);
          for (let j = 0; j < this.columns; j++) {
            copy[i].push(this.get(i, j));
          }
        }
        return copy;
      }
      toJSON() {
        return this.to2DArray();
      }
      isRowVector() {
        return this.rows === 1;
      }
      isColumnVector() {
        return this.columns === 1;
      }
      isVector() {
        return this.rows === 1 || this.columns === 1;
      }
      isSquare() {
        return this.rows === this.columns;
      }
      isEmpty() {
        return this.rows === 0 || this.columns === 0;
      }
      isSymmetric() {
        if (this.isSquare()) {
          for (let i = 0; i < this.rows; i++) {
            for (let j = 0; j <= i; j++) {
              if (this.get(i, j) !== this.get(j, i)) {
                return false;
              }
            }
          }
          return true;
        }
        return false;
      }
      isDistance() {
        if (!this.isSymmetric()) return false;
        for (let i = 0; i < this.rows; i++) {
          if (this.get(i, i) !== 0) return false;
        }
        return true;
      }
      isEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isEchelonForm = true;
        let checked = false;
        while (i < this.rows && isEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isEchelonForm = false;
              checked = true;
            }
          }
          i++;
        }
        return isEchelonForm;
      }
      isReducedEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isReducedEchelonForm = true;
        let checked = false;
        while (i < this.rows && isReducedEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isReducedEchelonForm = false;
              checked = true;
            }
          }
          for (let k = j + 1; k < this.rows; k++) {
            if (this.get(i, k) !== 0) {
              isReducedEchelonForm = false;
            }
          }
          i++;
        }
        return isReducedEchelonForm;
      }
      echelonForm() {
        let result = this.clone();
        let h = 0;
        let k = 0;
        while (h < result.rows && k < result.columns) {
          let iMax = h;
          for (let i = h; i < result.rows; i++) {
            if (result.get(i, k) > result.get(iMax, k)) {
              iMax = i;
            }
          }
          if (result.get(iMax, k) === 0) {
            k++;
          } else {
            result.swapRows(h, iMax);
            let tmp = result.get(h, k);
            for (let j = k; j < result.columns; j++) {
              result.set(h, j, result.get(h, j) / tmp);
            }
            for (let i = h + 1; i < result.rows; i++) {
              let factor = result.get(i, k) / result.get(h, k);
              result.set(i, k, 0);
              for (let j = k + 1; j < result.columns; j++) {
                result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
              }
            }
            h++;
            k++;
          }
        }
        return result;
      }
      reducedEchelonForm() {
        let result = this.echelonForm();
        let m = result.columns;
        let n = result.rows;
        let h = n - 1;
        while (h >= 0) {
          if (result.maxRow(h) === 0) {
            h--;
          } else {
            let p = 0;
            let pivot = false;
            while (p < n && pivot === false) {
              if (result.get(h, p) === 1) {
                pivot = true;
              } else {
                p++;
              }
            }
            for (let i = 0; i < h; i++) {
              let factor = result.get(i, p);
              for (let j = p; j < m; j++) {
                let tmp = result.get(i, j) - factor * result.get(h, j);
                result.set(i, j, tmp);
              }
            }
            h--;
          }
        }
        return result;
      }
      set() {
        throw new Error('set method is unimplemented');
      }
      get() {
        throw new Error('get method is unimplemented');
      }
      repeat(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          rows = 1,
          columns = 1
        } = options;
        if (!Number.isInteger(rows) || rows <= 0) {
          throw new TypeError('rows must be a positive integer');
        }
        if (!Number.isInteger(columns) || columns <= 0) {
          throw new TypeError('columns must be a positive integer');
        }
        let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.setSubMatrix(this, this.rows * i, this.columns * j);
          }
        }
        return matrix;
      }
      fill(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, value);
          }
        }
        return this;
      }
      neg() {
        return this.mulS(-1);
      }
      getRow(index) {
        checkRowIndex(this, index);
        let row = [];
        for (let i = 0; i < this.columns; i++) {
          row.push(this.get(index, i));
        }
        return row;
      }
      getRowVector(index) {
        return Matrix$1.rowVector(this.getRow(index));
      }
      setRow(index, array) {
        checkRowIndex(this, index);
        array = checkRowVector(this, array);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, array[i]);
        }
        return this;
      }
      swapRows(row1, row2) {
        checkRowIndex(this, row1);
        checkRowIndex(this, row2);
        for (let i = 0; i < this.columns; i++) {
          let temp = this.get(row1, i);
          this.set(row1, i, this.get(row2, i));
          this.set(row2, i, temp);
        }
        return this;
      }
      getColumn(index) {
        checkColumnIndex(this, index);
        let column = [];
        for (let i = 0; i < this.rows; i++) {
          column.push(this.get(i, index));
        }
        return column;
      }
      getColumnVector(index) {
        return Matrix$1.columnVector(this.getColumn(index));
      }
      setColumn(index, array) {
        checkColumnIndex(this, index);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, array[i]);
        }
        return this;
      }
      swapColumns(column1, column2) {
        checkColumnIndex(this, column1);
        checkColumnIndex(this, column2);
        for (let i = 0; i < this.rows; i++) {
          let temp = this.get(i, column1);
          this.set(i, column1, this.get(i, column2));
          this.set(i, column2, temp);
        }
        return this;
      }
      addRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[j]);
          }
        }
        return this;
      }
      subRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[j]);
          }
        }
        return this;
      }
      mulRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[j]);
          }
        }
        return this;
      }
      divRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[j]);
          }
        }
        return this;
      }
      addColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[i]);
          }
        }
        return this;
      }
      subColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[i]);
          }
        }
        return this;
      }
      mulColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[i]);
          }
        }
        return this;
      }
      divColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[i]);
          }
        }
        return this;
      }
      mulRow(index, value) {
        checkRowIndex(this, index);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, this.get(index, i) * value);
        }
        return this;
      }
      mulColumn(index, value) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, this.get(i, index) * value);
        }
        return this;
      }
      max(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[row]) {
                    max[row] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case 'column':
            {
              const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[column]) {
                    max[column] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case undefined:
            {
              let max = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max) {
                    max = this.get(row, column);
                  }
                }
              }
              return max;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      maxIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) > v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      min(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[row]) {
                    min[row] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case 'column':
            {
              const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[column]) {
                    min[column] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case undefined:
            {
              let min = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min) {
                    min = this.get(row, column);
                  }
                }
              }
              return min;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      minIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) < v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      maxRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      maxRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      minRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      minRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      maxColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      maxColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      minColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      minColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      diag() {
        let min = Math.min(this.rows, this.columns);
        let diag = [];
        for (let i = 0; i < min; i++) {
          diag.push(this.get(i, i));
        }
        return diag;
      }
      norm(type = 'frobenius') {
        switch (type) {
          case 'max':
            return this.max();
          case 'frobenius':
            return Math.sqrt(this.dot(this));
          default:
            throw new RangeError(`unknown norm type: ${type}`);
        }
      }
      cumulativeSum() {
        let sum = 0;
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            sum += this.get(i, j);
            this.set(i, j, sum);
          }
        }
        return this;
      }
      dot(vector2) {
        if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
        let vector1 = this.to1DArray();
        if (vector1.length !== vector2.length) {
          throw new RangeError('vectors do not have the same size');
        }
        let dot = 0;
        for (let i = 0; i < vector1.length; i++) {
          dot += vector1[i] * vector2[i];
        }
        return dot;
      }
      mmul(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.columns;
        let result = new Matrix$1(m, p);
        let Bcolj = new Float64Array(n);
        for (let j = 0; j < p; j++) {
          for (let k = 0; k < n; k++) {
            Bcolj[k] = other.get(k, j);
          }
          for (let i = 0; i < m; i++) {
            let s = 0;
            for (let k = 0; k < n; k++) {
              s += this.get(i, k) * Bcolj[k];
            }
            result.set(i, j, s);
          }
        }
        return result;
      }
      strassen2x2(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(2, 2);
        const a11 = this.get(0, 0);
        const b11 = other.get(0, 0);
        const a12 = this.get(0, 1);
        const b12 = other.get(0, 1);
        const a21 = this.get(1, 0);
        const b21 = other.get(1, 0);
        const a22 = this.get(1, 1);
        const b22 = other.get(1, 1);

        // Compute intermediate values.
        const m1 = (a11 + a22) * (b11 + b22);
        const m2 = (a21 + a22) * b11;
        const m3 = a11 * (b12 - b22);
        const m4 = a22 * (b21 - b11);
        const m5 = (a11 + a12) * b22;
        const m6 = (a21 - a11) * (b11 + b12);
        const m7 = (a12 - a22) * (b21 + b22);

        // Combine intermediate values into the output.
        const c00 = m1 + m4 - m5 + m7;
        const c01 = m3 + m5;
        const c10 = m2 + m4;
        const c11 = m1 - m2 + m3 + m6;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        return result;
      }
      strassen3x3(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(3, 3);
        const a00 = this.get(0, 0);
        const a01 = this.get(0, 1);
        const a02 = this.get(0, 2);
        const a10 = this.get(1, 0);
        const a11 = this.get(1, 1);
        const a12 = this.get(1, 2);
        const a20 = this.get(2, 0);
        const a21 = this.get(2, 1);
        const a22 = this.get(2, 2);
        const b00 = other.get(0, 0);
        const b01 = other.get(0, 1);
        const b02 = other.get(0, 2);
        const b10 = other.get(1, 0);
        const b11 = other.get(1, 1);
        const b12 = other.get(1, 2);
        const b20 = other.get(2, 0);
        const b21 = other.get(2, 1);
        const b22 = other.get(2, 2);
        const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
        const m2 = (a00 - a10) * (-b01 + b11);
        const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
        const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
        const m5 = (a10 + a11) * (-b00 + b01);
        const m6 = a00 * b00;
        const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
        const m8 = (-a00 + a20) * (b02 - b12);
        const m9 = (a20 + a21) * (-b00 + b02);
        const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
        const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
        const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
        const m13 = (a02 - a22) * (b11 - b21);
        const m14 = a02 * b20;
        const m15 = (a21 + a22) * (-b20 + b21);
        const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
        const m17 = (a02 - a12) * (b12 - b22);
        const m18 = (a11 + a12) * (-b20 + b22);
        const m19 = a01 * b10;
        const m20 = a12 * b21;
        const m21 = a10 * b02;
        const m22 = a20 * b01;
        const m23 = a22 * b22;
        const c00 = m6 + m14 + m19;
        const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
        const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
        const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
        const c11 = m2 + m4 + m5 + m6 + m20;
        const c12 = m14 + m16 + m17 + m18 + m21;
        const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
        const c21 = m12 + m13 + m14 + m15 + m22;
        const c22 = m6 + m7 + m8 + m9 + m23;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(0, 2, c02);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        result.set(1, 2, c12);
        result.set(2, 0, c20);
        result.set(2, 1, c21);
        result.set(2, 2, c22);
        return result;
      }
      mmulStrassen(y) {
        y = Matrix$1.checkMatrix(y);
        let x = this.clone();
        let r1 = x.rows;
        let c1 = x.columns;
        let r2 = y.rows;
        let c2 = y.columns;
        if (c1 !== r2) {
          // eslint-disable-next-line no-console
          console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
        }

        // Put a matrix into the top left of a matrix of zeros.
        // `rows` and `cols` are the dimensions of the output matrix.
        function embed(mat, rows, cols) {
          let r = mat.rows;
          let c = mat.columns;
          if (r === rows && c === cols) {
            return mat;
          } else {
            let resultat = AbstractMatrix.zeros(rows, cols);
            resultat = resultat.setSubMatrix(mat, 0, 0);
            return resultat;
          }
        }

        // Make sure both matrices are the same size.
        // This is exclusively for simplicity:
        // this algorithm can be implemented with matrices of different sizes.

        let r = Math.max(r1, r2);
        let c = Math.max(c1, c2);
        x = embed(x, r, c);
        y = embed(y, r, c);

        // Our recursive multiplication function.
        function blockMult(a, b, rows, cols) {
          // For small matrices, resort to naive multiplication.
          if (rows <= 512 || cols <= 512) {
            return a.mmul(b); // a is equivalent to this
          }

          // Apply dynamic padding.
          if (rows % 2 === 1 && cols % 2 === 1) {
            a = embed(a, rows + 1, cols + 1);
            b = embed(b, rows + 1, cols + 1);
          } else if (rows % 2 === 1) {
            a = embed(a, rows + 1, cols);
            b = embed(b, rows + 1, cols);
          } else if (cols % 2 === 1) {
            a = embed(a, rows, cols + 1);
            b = embed(b, rows, cols + 1);
          }
          let halfRows = parseInt(a.rows / 2, 10);
          let halfCols = parseInt(a.columns / 2, 10);
          // Subdivide input matrices.
          let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
          let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
          let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
          let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
          let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
          let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

          // Compute intermediate values.
          let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
          let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
          let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
          let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
          let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
          let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
          let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

          // Combine intermediate values into the output.
          let c11 = AbstractMatrix.add(m1, m4);
          c11.sub(m5);
          c11.add(m7);
          let c12 = AbstractMatrix.add(m3, m5);
          let c21 = AbstractMatrix.add(m2, m4);
          let c22 = AbstractMatrix.sub(m1, m2);
          c22.add(m3);
          c22.add(m6);

          // Crop output to the desired size (undo dynamic padding).
          let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
          result = result.setSubMatrix(c11, 0, 0);
          result = result.setSubMatrix(c12, c11.rows, 0);
          result = result.setSubMatrix(c21, 0, c11.columns);
          result = result.setSubMatrix(c22, c11.rows, c11.columns);
          return result.subMatrix(0, rows - 1, 0, cols - 1);
        }
        return blockMult(x, y, r, c);
      }
      scaleRows(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.rows; i++) {
          const row = this.getRow(i);
          if (row.length > 0) {
            rescale$1(row, {
              min,
              max,
              output: row
            });
          }
          newMatrix.setRow(i, row);
        }
        return newMatrix;
      }
      scaleColumns(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.columns; i++) {
          const column = this.getColumn(i);
          if (column.length) {
            rescale$1(column, {
              min,
              max,
              output: column
            });
          }
          newMatrix.setColumn(i, column);
        }
        return newMatrix;
      }
      flipRows() {
        const middle = Math.ceil(this.columns / 2);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < middle; j++) {
            let first = this.get(i, j);
            let last = this.get(i, this.columns - 1 - j);
            this.set(i, j, last);
            this.set(i, this.columns - 1 - j, first);
          }
        }
        return this;
      }
      flipColumns() {
        const middle = Math.ceil(this.rows / 2);
        for (let j = 0; j < this.columns; j++) {
          for (let i = 0; i < middle; i++) {
            let first = this.get(i, j);
            let last = this.get(this.rows - 1 - i, j);
            this.set(i, j, last);
            this.set(this.rows - 1 - i, j, first);
          }
        }
        return this;
      }
      kroneckerProduct(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.rows;
        let q = other.columns;
        let result = new Matrix$1(m * p, n * q);
        for (let i = 0; i < m; i++) {
          for (let j = 0; j < n; j++) {
            for (let k = 0; k < p; k++) {
              for (let l = 0; l < q; l++) {
                result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
              }
            }
          }
        }
        return result;
      }
      kroneckerSum(other) {
        other = Matrix$1.checkMatrix(other);
        if (!this.isSquare() || !other.isSquare()) {
          throw new Error('Kronecker Sum needs two Square Matrices');
        }
        let m = this.rows;
        let n = other.rows;
        let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
        let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
        return AxI.add(IxB);
      }
      transpose() {
        let result = new Matrix$1(this.columns, this.rows);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result.set(j, i, this.get(i, j));
          }
        }
        return result;
      }
      sortRows(compareFunction = compareNumbers) {
        for (let i = 0; i < this.rows; i++) {
          this.setRow(i, this.getRow(i).sort(compareFunction));
        }
        return this;
      }
      sortColumns(compareFunction = compareNumbers) {
        for (let i = 0; i < this.columns; i++) {
          this.setColumn(i, this.getColumn(i).sort(compareFunction));
        }
        return this;
      }
      subMatrix(startRow, endRow, startColumn, endColumn) {
        checkRange(this, startRow, endRow, startColumn, endColumn);
        let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
        for (let i = startRow; i <= endRow; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
          }
        }
        return newMatrix;
      }
      subMatrixRow(indices, startColumn, endColumn) {
        if (startColumn === undefined) startColumn = 0;
        if (endColumn === undefined) endColumn = this.columns - 1;
        if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            if (indices[i] < 0 || indices[i] >= this.rows) {
              throw new RangeError(`Row index out of range: ${indices[i]}`);
            }
            newMatrix.set(i, j - startColumn, this.get(indices[i], j));
          }
        }
        return newMatrix;
      }
      subMatrixColumn(indices, startRow, endRow) {
        if (startRow === undefined) startRow = 0;
        if (endRow === undefined) endRow = this.rows - 1;
        if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startRow; j <= endRow; j++) {
            if (indices[i] < 0 || indices[i] >= this.columns) {
              throw new RangeError(`Column index out of range: ${indices[i]}`);
            }
            newMatrix.set(j - startRow, i, this.get(j, indices[i]));
          }
        }
        return newMatrix;
      }
      setSubMatrix(matrix, startRow, startColumn) {
        matrix = Matrix$1.checkMatrix(matrix);
        if (matrix.isEmpty()) {
          return this;
        }
        let endRow = startRow + matrix.rows - 1;
        let endColumn = startColumn + matrix.columns - 1;
        checkRange(this, startRow, endRow, startColumn, endColumn);
        for (let i = 0; i < matrix.rows; i++) {
          for (let j = 0; j < matrix.columns; j++) {
            this.set(startRow + i, startColumn + j, matrix.get(i, j));
          }
        }
        return this;
      }
      selection(rowIndices, columnIndices) {
        checkRowIndices(this, rowIndices);
        checkColumnIndices(this, columnIndices);
        let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
        for (let i = 0; i < rowIndices.length; i++) {
          let rowIndex = rowIndices[i];
          for (let j = 0; j < columnIndices.length; j++) {
            let columnIndex = columnIndices[j];
            newMatrix.set(i, j, this.get(rowIndex, columnIndex));
          }
        }
        return newMatrix;
      }
      trace() {
        let min = Math.min(this.rows, this.columns);
        let trace = 0;
        for (let i = 0; i < min; i++) {
          trace += this.get(i, i);
        }
        return trace;
      }
      clone() {
        return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
      }

      /**
       * @template {AbstractMatrix} M
       * @param {AbstractMatrix} from
       * @param {M} to
       * @return {M}
       */
      static copy(from, to) {
        for (const [row, column, value] of from.entries()) {
          to.set(row, column, value);
        }
        return to;
      }
      sum(by) {
        switch (by) {
          case 'row':
            return sumByRow(this);
          case 'column':
            return sumByColumn(this);
          case undefined:
            return sumAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      product(by) {
        switch (by) {
          case 'row':
            return productByRow(this);
          case 'column':
            return productByColumn(this);
          case undefined:
            return productAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      mean(by) {
        const sum = this.sum(by);
        switch (by) {
          case 'row':
            {
              for (let i = 0; i < this.rows; i++) {
                sum[i] /= this.columns;
              }
              return sum;
            }
          case 'column':
            {
              for (let i = 0; i < this.columns; i++) {
                sum[i] /= this.rows;
              }
              return sum;
            }
          case undefined:
            return sum / this.size;
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      variance(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          unbiased = true,
          mean = this.mean(by)
        } = options;
        if (typeof unbiased !== 'boolean') {
          throw new TypeError('unbiased must be a boolean');
        }
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByRow(this, unbiased, mean);
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByColumn(this, unbiased, mean);
            }
          case undefined:
            {
              if (typeof mean !== 'number') {
                throw new TypeError('mean must be a number');
              }
              return varianceAll(this, unbiased, mean);
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      standardDeviation(by, options) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        const variance = this.variance(by, options);
        if (by === undefined) {
          return Math.sqrt(variance);
        } else {
          for (let i = 0; i < variance.length; i++) {
            variance[i] = Math.sqrt(variance[i]);
          }
          return variance;
        }
      }
      center(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          center = this.mean(by)
        } = options;
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByRow(this, center);
              return this;
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByColumn(this, center);
              return this;
            }
          case undefined:
            {
              if (typeof center !== 'number') {
                throw new TypeError('center must be a number');
              }
              centerAll(this, center);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      scale(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        let scale = options.scale;
        switch (by) {
          case 'row':
            {
              if (scale === undefined) {
                scale = getScaleByRow(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByRow(this, scale);
              return this;
            }
          case 'column':
            {
              if (scale === undefined) {
                scale = getScaleByColumn(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByColumn(this, scale);
              return this;
            }
          case undefined:
            {
              if (scale === undefined) {
                scale = getScaleAll(this);
              } else if (typeof scale !== 'number') {
                throw new TypeError('scale must be a number');
              }
              scaleAll(this, scale);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      toString(options) {
        return inspectMatrixWithOptions(this, options);
      }
      [Symbol.iterator]() {
        return this.entries();
      }

      /**
       * iterator from left to right, from top to bottom
       * yield [row, column, value]
       * @returns {Generator<[number, number, number], void, *>}
       */
      *entries() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield [row, col, this.get(row, col)];
          }
        }
      }

      /**
       * iterator from left to right, from top to bottom
       * yield value
       * @returns {Generator<number, void, *>}
       */
      *values() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield this.get(row, col);
          }
        }
      }
    }
    AbstractMatrix.prototype.klass = 'Matrix';
    if (typeof Symbol !== 'undefined') {
      AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
    }
    function compareNumbers(a, b) {
      return a - b;
    }
    function isArrayOfNumbers(array) {
      return array.every(element => {
        return typeof element === 'number';
      });
    }

    // Synonyms
    AbstractMatrix.random = AbstractMatrix.rand;
    AbstractMatrix.randomInt = AbstractMatrix.randInt;
    AbstractMatrix.diagonal = AbstractMatrix.diag;
    AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
    AbstractMatrix.identity = AbstractMatrix.eye;
    AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
    AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
    class Matrix$1 extends AbstractMatrix {
      /**
       * @type {Float64Array[]}
       */
      data;

      /**
       * Init an empty matrix
       * @param {number} nRows
       * @param {number} nColumns
       */
      #initData(nRows, nColumns) {
        this.data = [];
        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
        this.rows = nRows;
        this.columns = nColumns;
      }
      constructor(nRows, nColumns) {
        super();
        if (Matrix$1.isMatrix(nRows)) {
          this.#initData(nRows.rows, nRows.columns);
          Matrix$1.copy(nRows, this);
        } else if (Number.isInteger(nRows) && nRows >= 0) {
          this.#initData(nRows, nColumns);
        } else if (isAnyArray.isAnyArray(nRows)) {
          // Copy the values from the 2D array
          const arrayData = nRows;
          nRows = arrayData.length;
          nColumns = nRows ? arrayData[0].length : 0;
          if (typeof nColumns !== 'number') {
            throw new TypeError('Data must be a 2D array with at least one element');
          }
          this.data = [];
          for (let i = 0; i < nRows; i++) {
            if (arrayData[i].length !== nColumns) {
              throw new RangeError('Inconsistent array dimensions');
            }
            if (!isArrayOfNumbers(arrayData[i])) {
              throw new TypeError('Input data contains non-numeric values');
            }
            this.data.push(Float64Array.from(arrayData[i]));
          }
          this.rows = nRows;
          this.columns = nColumns;
        } else {
          throw new TypeError('First argument must be a positive number or an array');
        }
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
      removeRow(index) {
        checkRowIndex(this, index);
        this.data.splice(index, 1);
        this.rows -= 1;
        return this;
      }
      addRow(index, array) {
        if (array === undefined) {
          array = index;
          index = this.rows;
        }
        checkRowIndex(this, index, true);
        array = Float64Array.from(checkRowVector(this, array));
        this.data.splice(index, 0, array);
        this.rows += 1;
        return this;
      }
      removeColumn(index) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns - 1);
          for (let j = 0; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          for (let j = index + 1; j < this.columns; j++) {
            newRow[j - 1] = this.data[i][j];
          }
          this.data[i] = newRow;
        }
        this.columns -= 1;
        return this;
      }
      addColumn(index, array) {
        if (typeof array === 'undefined') {
          array = index;
          index = this.columns;
        }
        checkColumnIndex(this, index, true);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns + 1);
          let j = 0;
          for (; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          newRow[j++] = array[i];
          for (; j < this.columns + 1; j++) {
            newRow[j] = this.data[i][j - 1];
          }
          this.data[i] = newRow;
        }
        this.columns += 1;
        return this;
      }
    }
    installMathOperations(AbstractMatrix, Matrix$1);

    /**
     * @typedef {0 | 1 | number | boolean} Mask
     */

    class SymmetricMatrix extends AbstractMatrix {
      /** @type {Matrix} */
      #matrix;
      get size() {
        return this.#matrix.size;
      }
      get rows() {
        return this.#matrix.rows;
      }
      get columns() {
        return this.#matrix.columns;
      }
      get diagonalSize() {
        return this.rows;
      }

      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isSymmetricMatrix(value) {
        return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static zeros(diagonalSize) {
        return new this(diagonalSize);
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static ones(diagonalSize) {
        return new this(diagonalSize).fill(1);
      }

      /**
       * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
       * @return {this}
       */
      constructor(diagonalSize) {
        super();
        if (Matrix$1.isMatrix(diagonalSize)) {
          if (!diagonalSize.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
          this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
        } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
          this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
        } else {
          this.#matrix = new Matrix$1(diagonalSize);
          if (!this.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
        }
      }
      clone() {
        const matrix = new SymmetricMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          matrix.set(row, col, value);
        }
        return matrix;
      }
      toMatrix() {
        return new Matrix$1(this);
      }
      get(rowIndex, columnIndex) {
        return this.#matrix.get(rowIndex, columnIndex);
      }
      set(rowIndex, columnIndex, value) {
        // symmetric set
        this.#matrix.set(rowIndex, columnIndex, value);
        this.#matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      removeCross(index) {
        // symmetric remove side
        this.#matrix.removeRow(index);
        this.#matrix.removeColumn(index);
        return this;
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }
        const row = array.slice();
        row.splice(index, 1);
        this.#matrix.addRow(index, row);
        this.#matrix.addColumn(index, array);
        return this;
      }

      /**
       * @param {Mask[]} mask
       */
      applyMask(mask) {
        if (mask.length !== this.diagonalSize) {
          throw new RangeError('Mask size do not match with matrix size');
        }

        // prepare sides to remove from matrix from mask
        /** @type {number[]} */
        const sidesToRemove = [];
        for (const [index, passthroughs] of mask.entries()) {
          if (passthroughs) continue;
          sidesToRemove.push(index);
        }
        // to remove from highest to lowest for no mutation shifting
        sidesToRemove.reverse();

        // remove sides
        for (const sideIndex of sidesToRemove) {
          this.removeCross(sideIndex);
        }
        return this;
      }

      /**
       * Compact format upper-right corner of matrix
       * iterate from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 1 2 3 4
       * B 2 5 6 7
       * C 3 6 8 9
       * D 4 7 9 10
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
       *
       * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;

        /** @type {number[]} */
        const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
        for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       * @return {SymmetricMatrix}
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        // compactSize = (sideSize * (sideSize + 1)) / 2
        // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
        // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new SymmetricMatrix(diagonalSize);
        for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row;
        }
        return matrix;
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield [row, column, value]
       *
       * @returns {Generator<[number, number, number], void, *>}
       */
      *upperRightEntries() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield [row, col, value];

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield value
       *
       * @returns {Generator<[number, number, number], void, *>}
       */
      *upperRightValues() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield value;

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }
    }
    SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
    class DistanceMatrix extends SymmetricMatrix {
      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isDistanceMatrix(value) {
        return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
      }
      constructor(sideSize) {
        super(sideSize);
        if (!this.isDistance()) {
          throw new TypeError('Provided arguments do no produce a distance matrix');
        }
      }
      set(rowIndex, columnIndex, value) {
        // distance matrix diagonal is 0
        if (rowIndex === columnIndex) value = 0;
        return super.set(rowIndex, columnIndex, value);
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }

        // ensure distance
        array = array.slice();
        array[index] = 0;
        return super.addCross(index, array);
      }
      toSymmetricMatrix() {
        return new SymmetricMatrix(this);
      }
      clone() {
        const matrix = new DistanceMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          if (row === col) continue;
          matrix.set(row, col, value);
        }
        return matrix;
      }

      /**
       * Compact format upper-right corner of matrix
       * no diagonal (only zeros)
       * iterable from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 0 1 2 3
       * B 1 0 4 5
       * C 2 4 0 6
       * D 3 5 6 0
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
       *
       * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;
        const compactLength = (diagonalSize - 1) * diagonalSize / 2;

        /** @type {number[]} */
        const compact = new Array(compactLength);
        for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        if (compactSize === 0) {
          return new this(0);
        }

        // compactSize in Natural integer range ]0;∞]
        // compactSize = (sideSize * (sideSize - 1)) / 2
        // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new this(diagonalSize);
        for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return matrix;
      }
    }
    DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
    class BaseView extends AbstractMatrix {
      constructor(matrix, rows, columns) {
        super();
        this.matrix = matrix;
        this.rows = rows;
        this.columns = columns;
      }
    }
    class MatrixColumnView extends BaseView {
      constructor(matrix, column) {
        checkColumnIndex(matrix, column);
        super(matrix, matrix.rows, 1);
        this.column = column;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.column, value);
        return this;
      }
      get(rowIndex) {
        return this.matrix.get(rowIndex, this.column);
      }
    }
    class MatrixColumnSelectionView extends BaseView {
      constructor(matrix, columnIndices) {
        checkColumnIndices(matrix, columnIndices);
        super(matrix, matrix.rows, columnIndices.length);
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
      }
    }
    class MatrixFlipColumnView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
      }
    }
    class MatrixFlipRowView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
      }
    }
    class MatrixRowView extends BaseView {
      constructor(matrix, row) {
        checkRowIndex(matrix, row);
        super(matrix, 1, matrix.columns);
        this.row = row;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.row, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.row, columnIndex);
      }
    }
    class MatrixRowSelectionView extends BaseView {
      constructor(matrix, rowIndices) {
        checkRowIndices(matrix, rowIndices);
        super(matrix, rowIndices.length, matrix.columns);
        this.rowIndices = rowIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
      }
    }
    class MatrixSelectionView extends BaseView {
      constructor(matrix, rowIndices, columnIndices) {
        checkRowIndices(matrix, rowIndices);
        checkColumnIndices(matrix, columnIndices);
        super(matrix, rowIndices.length, columnIndices.length);
        this.rowIndices = rowIndices;
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
      }
    }
    class MatrixSubView extends BaseView {
      constructor(matrix, startRow, endRow, startColumn, endColumn) {
        checkRange(matrix, startRow, endRow, startColumn, endColumn);
        super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
        this.startRow = startRow;
        this.startColumn = startColumn;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
      }
    }
    class MatrixTransposeView$1 extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.columns, matrix.rows);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(columnIndex, rowIndex);
      }
    }
    class WrapperMatrix1D extends AbstractMatrix {
      constructor(data, options = {}) {
        const {
          rows = 1
        } = options;
        if (data.length % rows !== 0) {
          throw new Error('the data length is not divisible by the number of rows');
        }
        super();
        this.rows = rows;
        this.columns = data.length / rows;
        this.data = data;
      }
      set(rowIndex, columnIndex, value) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        this.data[index] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        return this.data[index];
      }
      _calculateIndex(row, column) {
        return row * this.columns + column;
      }
    }
    class WrapperMatrix2D extends AbstractMatrix {
      constructor(data) {
        super();
        this.data = data;
        this.rows = data.length;
        this.columns = data[0].length;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
    }
    function wrap(array, options) {
      if (isAnyArray.isAnyArray(array)) {
        if (array[0] && isAnyArray.isAnyArray(array[0])) {
          return new WrapperMatrix2D(array);
        } else {
          return new WrapperMatrix1D(array, options);
        }
      } else {
        throw new Error('the argument is not an array');
      }
    }
    class LuDecomposition {
      constructor(matrix) {
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        let lu = matrix.clone();
        let rows = lu.rows;
        let columns = lu.columns;
        let pivotVector = new Float64Array(rows);
        let pivotSign = 1;
        let i, j, k, p, s, t, v;
        let LUcolj, kmax;
        for (i = 0; i < rows; i++) {
          pivotVector[i] = i;
        }
        LUcolj = new Float64Array(rows);
        for (j = 0; j < columns; j++) {
          for (i = 0; i < rows; i++) {
            LUcolj[i] = lu.get(i, j);
          }
          for (i = 0; i < rows; i++) {
            kmax = Math.min(i, j);
            s = 0;
            for (k = 0; k < kmax; k++) {
              s += lu.get(i, k) * LUcolj[k];
            }
            LUcolj[i] -= s;
            lu.set(i, j, LUcolj[i]);
          }
          p = j;
          for (i = j + 1; i < rows; i++) {
            if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
              p = i;
            }
          }
          if (p !== j) {
            for (k = 0; k < columns; k++) {
              t = lu.get(p, k);
              lu.set(p, k, lu.get(j, k));
              lu.set(j, k, t);
            }
            v = pivotVector[p];
            pivotVector[p] = pivotVector[j];
            pivotVector[j] = v;
            pivotSign = -pivotSign;
          }
          if (j < rows && lu.get(j, j) !== 0) {
            for (i = j + 1; i < rows; i++) {
              lu.set(i, j, lu.get(i, j) / lu.get(j, j));
            }
          }
        }
        this.LU = lu;
        this.pivotVector = pivotVector;
        this.pivotSign = pivotSign;
      }
      isSingular() {
        let data = this.LU;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          if (data.get(j, j) === 0) {
            return true;
          }
        }
        return false;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let lu = this.LU;
        let rows = lu.rows;
        if (rows !== value.rows) {
          throw new Error('Invalid matrix dimensions');
        }
        if (this.isSingular()) {
          throw new Error('LU matrix is singular');
        }
        let count = value.columns;
        let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
        let columns = lu.columns;
        let i, j, k;
        for (k = 0; k < columns; k++) {
          for (i = k + 1; i < columns; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        for (k = columns - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / lu.get(k, k));
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        return X;
      }
      get determinant() {
        let data = this.LU;
        if (!data.isSquare()) {
          throw new Error('Matrix must be square');
        }
        let determinant = this.pivotSign;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          determinant *= data.get(j, j);
        }
        return determinant;
      }
      get lowerTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i > j) {
              X.set(i, j, data.get(i, j));
            } else if (i === j) {
              X.set(i, j, 1);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get upperTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i <= j) {
              X.set(i, j, data.get(i, j));
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get pivotPermutationVector() {
        return Array.from(this.pivotVector);
      }
    }
    function hypotenuse(a, b) {
      let r = 0;
      if (Math.abs(a) > Math.abs(b)) {
        r = b / a;
        return Math.abs(a) * Math.sqrt(1 + r * r);
      }
      if (b !== 0) {
        r = a / b;
        return Math.abs(b) * Math.sqrt(1 + r * r);
      }
      return 0;
    }
    class QrDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let qr = value.clone();
        let m = value.rows;
        let n = value.columns;
        let rdiag = new Float64Array(n);
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          let nrm = 0;
          for (i = k; i < m; i++) {
            nrm = hypotenuse(nrm, qr.get(i, k));
          }
          if (nrm !== 0) {
            if (qr.get(k, k) < 0) {
              nrm = -nrm;
            }
            for (i = k; i < m; i++) {
              qr.set(i, k, qr.get(i, k) / nrm);
            }
            qr.set(k, k, qr.get(k, k) + 1);
            for (j = k + 1; j < n; j++) {
              s = 0;
              for (i = k; i < m; i++) {
                s += qr.get(i, k) * qr.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < m; i++) {
                qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
              }
            }
          }
          rdiag[k] = -nrm;
        }
        this.QR = qr;
        this.Rdiag = rdiag;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let qr = this.QR;
        let m = qr.rows;
        if (value.rows !== m) {
          throw new Error('Matrix row dimensions must agree');
        }
        if (!this.isFullRank()) {
          throw new Error('Matrix is rank deficient');
        }
        let count = value.columns;
        let X = value.clone();
        let n = qr.columns;
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          for (j = 0; j < count; j++) {
            s = 0;
            for (i = k; i < m; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }
            s = -s / qr.get(k, k);
            for (i = k; i < m; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
        for (k = n - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / this.Rdiag[k]);
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
            }
          }
        }
        return X.subMatrix(0, n - 1, 0, count - 1);
      }
      isFullRank() {
        let columns = this.QR.columns;
        for (let i = 0; i < columns; i++) {
          if (this.Rdiag[i] === 0) {
            return false;
          }
        }
        return true;
      }
      get upperTriangularMatrix() {
        let qr = this.QR;
        let n = qr.columns;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            if (i < j) {
              X.set(i, j, qr.get(i, j));
            } else if (i === j) {
              X.set(i, j, this.Rdiag[i]);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get orthogonalMatrix() {
        let qr = this.QR;
        let rows = qr.rows;
        let columns = qr.columns;
        let X = new Matrix$1(rows, columns);
        let i, j, k, s;
        for (k = columns - 1; k >= 0; k--) {
          for (i = 0; i < rows; i++) {
            X.set(i, k, 0);
          }
          X.set(k, k, 1);
          for (j = k; j < columns; j++) {
            if (qr.get(k, k) !== 0) {
              s = 0;
              for (i = k; i < rows; i++) {
                s += qr.get(i, k) * X.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < rows; i++) {
                X.set(i, j, X.get(i, j) + s * qr.get(i, k));
              }
            }
          }
        }
        return X;
      }
    }
    class SingularValueDecomposition {
      constructor(value, options = {}) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (value.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let m = value.rows;
        let n = value.columns;
        const {
          computeLeftSingularVectors = true,
          computeRightSingularVectors = true,
          autoTranspose = false
        } = options;
        let wantu = Boolean(computeLeftSingularVectors);
        let wantv = Boolean(computeRightSingularVectors);
        let swapped = false;
        let a;
        if (m < n) {
          if (!autoTranspose) {
            a = value.clone();
            // eslint-disable-next-line no-console
            console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
          } else {
            a = value.transpose();
            m = a.rows;
            n = a.columns;
            swapped = true;
            let aux = wantu;
            wantu = wantv;
            wantv = aux;
          }
        } else {
          a = value.clone();
        }
        let nu = Math.min(m, n);
        let ni = Math.min(m + 1, n);
        let s = new Float64Array(ni);
        let U = new Matrix$1(m, nu);
        let V = new Matrix$1(n, n);
        let e = new Float64Array(n);
        let work = new Float64Array(m);
        let si = new Float64Array(ni);
        for (let i = 0; i < ni; i++) si[i] = i;
        let nct = Math.min(m - 1, n);
        let nrt = Math.max(0, Math.min(n - 2, m));
        let mrc = Math.max(nct, nrt);
        for (let k = 0; k < mrc; k++) {
          if (k < nct) {
            s[k] = 0;
            for (let i = k; i < m; i++) {
              s[k] = hypotenuse(s[k], a.get(i, k));
            }
            if (s[k] !== 0) {
              if (a.get(k, k) < 0) {
                s[k] = -s[k];
              }
              for (let i = k; i < m; i++) {
                a.set(i, k, a.get(i, k) / s[k]);
              }
              a.set(k, k, a.get(k, k) + 1);
            }
            s[k] = -s[k];
          }
          for (let j = k + 1; j < n; j++) {
            if (k < nct && s[k] !== 0) {
              let t = 0;
              for (let i = k; i < m; i++) {
                t += a.get(i, k) * a.get(i, j);
              }
              t = -t / a.get(k, k);
              for (let i = k; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * a.get(i, k));
              }
            }
            e[j] = a.get(k, j);
          }
          if (wantu && k < nct) {
            for (let i = k; i < m; i++) {
              U.set(i, k, a.get(i, k));
            }
          }
          if (k < nrt) {
            e[k] = 0;
            for (let i = k + 1; i < n; i++) {
              e[k] = hypotenuse(e[k], e[i]);
            }
            if (e[k] !== 0) {
              if (e[k + 1] < 0) {
                e[k] = 0 - e[k];
              }
              for (let i = k + 1; i < n; i++) {
                e[i] /= e[k];
              }
              e[k + 1] += 1;
            }
            e[k] = -e[k];
            if (k + 1 < m && e[k] !== 0) {
              for (let i = k + 1; i < m; i++) {
                work[i] = 0;
              }
              for (let i = k + 1; i < m; i++) {
                for (let j = k + 1; j < n; j++) {
                  work[i] += e[j] * a.get(i, j);
                }
              }
              for (let j = k + 1; j < n; j++) {
                let t = -e[j] / e[k + 1];
                for (let i = k + 1; i < m; i++) {
                  a.set(i, j, a.get(i, j) + t * work[i]);
                }
              }
            }
            if (wantv) {
              for (let i = k + 1; i < n; i++) {
                V.set(i, k, e[i]);
              }
            }
          }
        }
        let p = Math.min(n, m + 1);
        if (nct < n) {
          s[nct] = a.get(nct, nct);
        }
        if (m < p) {
          s[p - 1] = 0;
        }
        if (nrt + 1 < p) {
          e[nrt] = a.get(nrt, p - 1);
        }
        e[p - 1] = 0;
        if (wantu) {
          for (let j = nct; j < nu; j++) {
            for (let i = 0; i < m; i++) {
              U.set(i, j, 0);
            }
            U.set(j, j, 1);
          }
          for (let k = nct - 1; k >= 0; k--) {
            if (s[k] !== 0) {
              for (let j = k + 1; j < nu; j++) {
                let t = 0;
                for (let i = k; i < m; i++) {
                  t += U.get(i, k) * U.get(i, j);
                }
                t = -t / U.get(k, k);
                for (let i = k; i < m; i++) {
                  U.set(i, j, U.get(i, j) + t * U.get(i, k));
                }
              }
              for (let i = k; i < m; i++) {
                U.set(i, k, -U.get(i, k));
              }
              U.set(k, k, 1 + U.get(k, k));
              for (let i = 0; i < k - 1; i++) {
                U.set(i, k, 0);
              }
            } else {
              for (let i = 0; i < m; i++) {
                U.set(i, k, 0);
              }
              U.set(k, k, 1);
            }
          }
        }
        if (wantv) {
          for (let k = n - 1; k >= 0; k--) {
            if (k < nrt && e[k] !== 0) {
              for (let j = k + 1; j < n; j++) {
                let t = 0;
                for (let i = k + 1; i < n; i++) {
                  t += V.get(i, k) * V.get(i, j);
                }
                t = -t / V.get(k + 1, k);
                for (let i = k + 1; i < n; i++) {
                  V.set(i, j, V.get(i, j) + t * V.get(i, k));
                }
              }
            }
            for (let i = 0; i < n; i++) {
              V.set(i, k, 0);
            }
            V.set(k, k, 1);
          }
        }
        let pp = p - 1;
        let eps = Number.EPSILON;
        while (p > 0) {
          let k, kase;
          for (k = p - 2; k >= -1; k--) {
            if (k === -1) {
              break;
            }
            const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
            if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
              e[k] = 0;
              break;
            }
          }
          if (k === p - 2) {
            kase = 4;
          } else {
            let ks;
            for (ks = p - 1; ks >= k; ks--) {
              if (ks === k) {
                break;
              }
              let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
              if (Math.abs(s[ks]) <= eps * t) {
                s[ks] = 0;
                break;
              }
            }
            if (ks === k) {
              kase = 3;
            } else if (ks === p - 1) {
              kase = 1;
            } else {
              kase = 2;
              k = ks;
            }
          }
          k++;
          switch (kase) {
            case 1:
              {
                let f = e[p - 2];
                e[p - 2] = 0;
                for (let j = p - 2; j >= k; j--) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  if (j !== k) {
                    f = -sn * e[j - 1];
                    e[j - 1] = cs * e[j - 1];
                  }
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                      V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                      V.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 2:
              {
                let f = e[k - 1];
                e[k - 1] = 0;
                for (let j = k; j < p; j++) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  f = -sn * e[j];
                  e[j] = cs * e[j];
                  if (wantu) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                      U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                      U.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 3:
              {
                const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
                const sp = s[p - 1] / scale;
                const spm1 = s[p - 2] / scale;
                const epm1 = e[p - 2] / scale;
                const sk = s[k] / scale;
                const ek = e[k] / scale;
                const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
                const c = sp * epm1 * (sp * epm1);
                let shift = 0;
                if (b !== 0 || c !== 0) {
                  if (b < 0) {
                    shift = 0 - Math.sqrt(b * b + c);
                  } else {
                    shift = Math.sqrt(b * b + c);
                  }
                  shift = c / (b + shift);
                }
                let f = (sk + sp) * (sk - sp) + shift;
                let g = sk * ek;
                for (let j = k; j < p - 1; j++) {
                  let t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  let cs = f / t;
                  let sn = g / t;
                  if (j !== k) {
                    e[j - 1] = t;
                  }
                  f = cs * s[j] + sn * e[j];
                  e[j] = cs * e[j] - sn * s[j];
                  g = sn * s[j + 1];
                  s[j + 1] = cs * s[j + 1];
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                      V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                      V.set(i, j, t);
                    }
                  }
                  t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  cs = f / t;
                  sn = g / t;
                  s[j] = t;
                  f = cs * e[j] + sn * s[j + 1];
                  s[j + 1] = -sn * e[j] + cs * s[j + 1];
                  g = sn * e[j + 1];
                  e[j + 1] = cs * e[j + 1];
                  if (wantu && j < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                      U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                      U.set(i, j, t);
                    }
                  }
                }
                e[p - 2] = f;
                break;
              }
            case 4:
              {
                if (s[k] <= 0) {
                  s[k] = s[k] < 0 ? -s[k] : 0;
                  if (wantv) {
                    for (let i = 0; i <= pp; i++) {
                      V.set(i, k, -V.get(i, k));
                    }
                  }
                }
                while (k < pp) {
                  if (s[k] >= s[k + 1]) {
                    break;
                  }
                  let t = s[k];
                  s[k] = s[k + 1];
                  s[k + 1] = t;
                  if (wantv && k < n - 1) {
                    for (let i = 0; i < n; i++) {
                      t = V.get(i, k + 1);
                      V.set(i, k + 1, V.get(i, k));
                      V.set(i, k, t);
                    }
                  }
                  if (wantu && k < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = U.get(i, k + 1);
                      U.set(i, k + 1, U.get(i, k));
                      U.set(i, k, t);
                    }
                  }
                  k++;
                }
                p--;
                break;
              }
            // no default
          }
        }
        if (swapped) {
          let tmp = V;
          V = U;
          U = tmp;
        }
        this.m = m;
        this.n = n;
        this.s = s;
        this.U = U;
        this.V = V;
      }
      solve(value) {
        let Y = value;
        let e = this.threshold;
        let scols = this.s.length;
        let Ls = Matrix$1.zeros(scols, scols);
        for (let i = 0; i < scols; i++) {
          if (Math.abs(this.s[i]) <= e) {
            Ls.set(i, i, 0);
          } else {
            Ls.set(i, i, 1 / this.s[i]);
          }
        }
        let U = this.U;
        let V = this.rightSingularVectors;
        let VL = V.mmul(Ls);
        let vrows = V.rows;
        let urows = U.rows;
        let VLU = Matrix$1.zeros(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < scols; k++) {
              sum += VL.get(i, k) * U.get(j, k);
            }
            VLU.set(i, j, sum);
          }
        }
        return VLU.mmul(Y);
      }
      solveForDiagonal(value) {
        return this.solve(Matrix$1.diag(value));
      }
      inverse() {
        let V = this.V;
        let e = this.threshold;
        let vrows = V.rows;
        let vcols = V.columns;
        let X = new Matrix$1(vrows, this.s.length);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < vcols; j++) {
            if (Math.abs(this.s[j]) > e) {
              X.set(i, j, V.get(i, j) / this.s[j]);
            }
          }
        }
        let U = this.U;
        let urows = U.rows;
        let ucols = U.columns;
        let Y = new Matrix$1(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < ucols; k++) {
              sum += X.get(i, k) * U.get(j, k);
            }
            Y.set(i, j, sum);
          }
        }
        return Y;
      }
      get condition() {
        return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
      }
      get norm2() {
        return this.s[0];
      }
      get rank() {
        let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
        let r = 0;
        let s = this.s;
        for (let i = 0, ii = s.length; i < ii; i++) {
          if (s[i] > tol) {
            r++;
          }
        }
        return r;
      }
      get diagonal() {
        return Array.from(this.s);
      }
      get threshold() {
        return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
      }
      get leftSingularVectors() {
        return this.U;
      }
      get rightSingularVectors() {
        return this.V;
      }
      get diagonalMatrix() {
        return Matrix$1.diag(this.s);
      }
    }
    function inverse(matrix, useSVD = false) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      if (useSVD) {
        return new SingularValueDecomposition(matrix).inverse();
      } else {
        return solve$1(matrix, Matrix$1.eye(matrix.rows));
      }
    }
    function solve$1(leftHandSide, rightHandSide, useSVD = false) {
      leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
      rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
      if (useSVD) {
        return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
      } else {
        return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
      }
    }
    function determinant(matrix) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isSquare()) {
        if (matrix.columns === 0) {
          return 1;
        }
        let a, b, c, d;
        if (matrix.columns === 2) {
          // 2 x 2 matrix
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(1, 0);
          d = matrix.get(1, 1);
          return a * d - b * c;
        } else if (matrix.columns === 3) {
          // 3 x 3 matrix
          let subMatrix0, subMatrix1, subMatrix2;
          subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
          subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
          subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(0, 2);
          return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
        } else {
          // general purpose determinant using the LU decomposition
          return new LuDecomposition(matrix).determinant;
        }
      } else {
        throw Error('determinant can only be calculated for a square matrix');
      }
    }
    function xrange(n, exception) {
      let range = [];
      for (let i = 0; i < n; i++) {
        if (i !== exception) {
          range.push(i);
        }
      }
      return range;
    }
    function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
      if (error > thresholdError) {
        return new Array(matrix.rows + 1).fill(0);
      } else {
        let returnArray = matrix.addRow(index, [0]);
        for (let i = 0; i < returnArray.rows; i++) {
          if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
            returnArray.set(i, 0, 0);
          }
        }
        return returnArray.to1DArray();
      }
    }
    function linearDependencies(matrix, options = {}) {
      const {
        thresholdValue = 10e-10,
        thresholdError = 10e-10
      } = options;
      matrix = Matrix$1.checkMatrix(matrix);
      let n = matrix.rows;
      let results = new Matrix$1(n, n);
      for (let i = 0; i < n; i++) {
        let b = Matrix$1.columnVector(matrix.getRow(i));
        let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
        let svd = new SingularValueDecomposition(Abis);
        let x = svd.solve(b);
        let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
        results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
      }
      return results;
    }
    function pseudoInverse(matrix, threshold = Number.EPSILON) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
        // (0xn)*(nx0)*(0xn) = 0xn
        // (nx0)*(0xn)*(nx0) = nx0
        return matrix.transpose();
      }
      let svdSolution = new SingularValueDecomposition(matrix, {
        autoTranspose: true
      });
      let U = svdSolution.leftSingularVectors;
      let V = svdSolution.rightSingularVectors;
      let s = svdSolution.diagonal;
      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > threshold) {
          s[i] = 1.0 / s[i];
        } else {
          s[i] = 0.0;
        }
      }
      return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
    }
    function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true
      } = options;
      if (center) {
        xMatrix = xMatrix.center('column');
        if (!yIsSame) {
          yMatrix = yMatrix.center('column');
        }
      }
      const cov = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < cov.rows; i++) {
        for (let j = 0; j < cov.columns; j++) {
          cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
        }
      }
      return cov;
    }
    function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true,
        scale = true
      } = options;
      if (center) {
        xMatrix.center('column');
        if (!yIsSame) {
          yMatrix.center('column');
        }
      }
      if (scale) {
        xMatrix.scale('column');
        if (!yIsSame) {
          yMatrix.scale('column');
        }
      }
      const sdx = xMatrix.standardDeviation('column', {
        unbiased: true
      });
      const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
        unbiased: true
      });
      const corr = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < corr.rows; i++) {
        for (let j = 0; j < corr.columns; j++) {
          corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
        }
      }
      return corr;
    }
    class EigenvalueDecomposition {
      constructor(matrix, options = {}) {
        const {
          assumeSymmetric = false
        } = options;
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        if (!matrix.isSquare()) {
          throw new Error('Matrix is not a square matrix');
        }
        if (matrix.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let n = matrix.columns;
        let V = new Matrix$1(n, n);
        let d = new Float64Array(n);
        let e = new Float64Array(n);
        let value = matrix;
        let i, j;
        let isSymmetric = false;
        if (assumeSymmetric) {
          isSymmetric = true;
        } else {
          isSymmetric = matrix.isSymmetric();
        }
        if (isSymmetric) {
          for (i = 0; i < n; i++) {
            for (j = 0; j < n; j++) {
              V.set(i, j, value.get(i, j));
            }
          }
          tred2(n, e, d, V);
          tql2(n, e, d, V);
        } else {
          let H = new Matrix$1(n, n);
          let ort = new Float64Array(n);
          for (j = 0; j < n; j++) {
            for (i = 0; i < n; i++) {
              H.set(i, j, value.get(i, j));
            }
          }
          orthes(n, H, ort, V);
          hqr2(n, e, d, V, H);
        }
        this.n = n;
        this.e = e;
        this.d = d;
        this.V = V;
      }
      get realEigenvalues() {
        return Array.from(this.d);
      }
      get imaginaryEigenvalues() {
        return Array.from(this.e);
      }
      get eigenvectorMatrix() {
        return this.V;
      }
      get diagonalMatrix() {
        let n = this.n;
        let e = this.e;
        let d = this.d;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            X.set(i, j, 0);
          }
          X.set(i, i, d[i]);
          if (e[i] > 0) {
            X.set(i, i + 1, e[i]);
          } else if (e[i] < 0) {
            X.set(i, i - 1, e[i]);
          }
        }
        return X;
      }
    }
    function tred2(n, e, d, V) {
      let f, g, h, i, j, k, hh, scale;
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
      }
      for (i = n - 1; i > 0; i--) {
        scale = 0;
        h = 0;
        for (k = 0; k < i; k++) {
          scale = scale + Math.abs(d[k]);
        }
        if (scale === 0) {
          e[i] = d[i - 1];
          for (j = 0; j < i; j++) {
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
            V.set(j, i, 0);
          }
        } else {
          for (k = 0; k < i; k++) {
            d[k] /= scale;
            h += d[k] * d[k];
          }
          f = d[i - 1];
          g = Math.sqrt(h);
          if (f > 0) {
            g = -g;
          }
          e[i] = scale * g;
          h = h - f * g;
          d[i - 1] = f - g;
          for (j = 0; j < i; j++) {
            e[j] = 0;
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            V.set(j, i, f);
            g = e[j] + V.get(j, j) * f;
            for (k = j + 1; k <= i - 1; k++) {
              g += V.get(k, j) * d[k];
              e[k] += V.get(k, j) * f;
            }
            e[j] = g;
          }
          f = 0;
          for (j = 0; j < i; j++) {
            e[j] /= h;
            f += e[j] * d[j];
          }
          hh = f / (h + h);
          for (j = 0; j < i; j++) {
            e[j] -= hh * d[j];
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            g = e[j];
            for (k = j; k <= i - 1; k++) {
              V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
            }
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
          }
        }
        d[i] = h;
      }
      for (i = 0; i < n - 1; i++) {
        V.set(n - 1, i, V.get(i, i));
        V.set(i, i, 1);
        h = d[i + 1];
        if (h !== 0) {
          for (k = 0; k <= i; k++) {
            d[k] = V.get(k, i + 1) / h;
          }
          for (j = 0; j <= i; j++) {
            g = 0;
            for (k = 0; k <= i; k++) {
              g += V.get(k, i + 1) * V.get(k, j);
            }
            for (k = 0; k <= i; k++) {
              V.set(k, j, V.get(k, j) - g * d[k]);
            }
          }
        }
        for (k = 0; k <= i; k++) {
          V.set(k, i + 1, 0);
        }
      }
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
        V.set(n - 1, j, 0);
      }
      V.set(n - 1, n - 1, 1);
      e[0] = 0;
    }
    function tql2(n, e, d, V) {
      let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
      for (i = 1; i < n; i++) {
        e[i - 1] = e[i];
      }
      e[n - 1] = 0;
      let f = 0;
      let tst1 = 0;
      let eps = Number.EPSILON;
      for (l = 0; l < n; l++) {
        tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
        m = l;
        while (m < n) {
          if (Math.abs(e[m]) <= eps * tst1) {
            break;
          }
          m++;
        }
        if (m > l) {
          do {
            g = d[l];
            p = (d[l + 1] - g) / (2 * e[l]);
            r = hypotenuse(p, 1);
            if (p < 0) {
              r = -r;
            }
            d[l] = e[l] / (p + r);
            d[l + 1] = e[l] * (p + r);
            dl1 = d[l + 1];
            h = g - d[l];
            for (i = l + 2; i < n; i++) {
              d[i] -= h;
            }
            f = f + h;
            p = d[m];
            c = 1;
            c2 = c;
            c3 = c;
            el1 = e[l + 1];
            s = 0;
            s2 = 0;
            for (i = m - 1; i >= l; i--) {
              c3 = c2;
              c2 = c;
              s2 = s;
              g = c * e[i];
              h = c * p;
              r = hypotenuse(p, e[i]);
              e[i + 1] = s * r;
              s = e[i] / r;
              c = p / r;
              p = c * d[i] - s * g;
              d[i + 1] = h + s * (c * g + s * d[i]);
              for (k = 0; k < n; k++) {
                h = V.get(k, i + 1);
                V.set(k, i + 1, s * V.get(k, i) + c * h);
                V.set(k, i, c * V.get(k, i) - s * h);
              }
            }
            p = -s * s2 * c3 * el1 * e[l] / dl1;
            e[l] = s * p;
            d[l] = c * p;
          } while (Math.abs(e[l]) > eps * tst1);
        }
        d[l] = d[l] + f;
        e[l] = 0;
      }
      for (i = 0; i < n - 1; i++) {
        k = i;
        p = d[i];
        for (j = i + 1; j < n; j++) {
          if (d[j] < p) {
            k = j;
            p = d[j];
          }
        }
        if (k !== i) {
          d[k] = d[i];
          d[i] = p;
          for (j = 0; j < n; j++) {
            p = V.get(j, i);
            V.set(j, i, V.get(j, k));
            V.set(j, k, p);
          }
        }
      }
    }
    function orthes(n, H, ort, V) {
      let low = 0;
      let high = n - 1;
      let f, g, h, i, j, m;
      let scale;
      for (m = low + 1; m <= high - 1; m++) {
        scale = 0;
        for (i = m; i <= high; i++) {
          scale = scale + Math.abs(H.get(i, m - 1));
        }
        if (scale !== 0) {
          h = 0;
          for (i = high; i >= m; i--) {
            ort[i] = H.get(i, m - 1) / scale;
            h += ort[i] * ort[i];
          }
          g = Math.sqrt(h);
          if (ort[m] > 0) {
            g = -g;
          }
          h = h - ort[m] * g;
          ort[m] = ort[m] - g;
          for (j = m; j < n; j++) {
            f = 0;
            for (i = high; i >= m; i--) {
              f += ort[i] * H.get(i, j);
            }
            f = f / h;
            for (i = m; i <= high; i++) {
              H.set(i, j, H.get(i, j) - f * ort[i]);
            }
          }
          for (i = 0; i <= high; i++) {
            f = 0;
            for (j = high; j >= m; j--) {
              f += ort[j] * H.get(i, j);
            }
            f = f / h;
            for (j = m; j <= high; j++) {
              H.set(i, j, H.get(i, j) - f * ort[j]);
            }
          }
          ort[m] = scale * ort[m];
          H.set(m, m - 1, scale * g);
        }
      }
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, i === j ? 1 : 0);
        }
      }
      for (m = high - 1; m >= low + 1; m--) {
        if (H.get(m, m - 1) !== 0) {
          for (i = m + 1; i <= high; i++) {
            ort[i] = H.get(i, m - 1);
          }
          for (j = m; j <= high; j++) {
            g = 0;
            for (i = m; i <= high; i++) {
              g += ort[i] * V.get(i, j);
            }
            g = g / ort[m] / H.get(m, m - 1);
            for (i = m; i <= high; i++) {
              V.set(i, j, V.get(i, j) + g * ort[i]);
            }
          }
        }
      }
    }
    function hqr2(nn, e, d, V, H) {
      let n = nn - 1;
      let low = 0;
      let high = nn - 1;
      let eps = Number.EPSILON;
      let exshift = 0;
      let norm = 0;
      let p = 0;
      let q = 0;
      let r = 0;
      let s = 0;
      let z = 0;
      let iter = 0;
      let i, j, k, l, m, t, w, x, y;
      let ra, sa, vr, vi;
      let notlast, cdivres;
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          d[i] = H.get(i, i);
          e[i] = 0;
        }
        for (j = Math.max(i - 1, 0); j < nn; j++) {
          norm = norm + Math.abs(H.get(i, j));
        }
      }
      while (n >= low) {
        l = n;
        while (l > low) {
          s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
          if (s === 0) {
            s = norm;
          }
          if (Math.abs(H.get(l, l - 1)) < eps * s) {
            break;
          }
          l--;
        }
        if (l === n) {
          H.set(n, n, H.get(n, n) + exshift);
          d[n] = H.get(n, n);
          e[n] = 0;
          n--;
          iter = 0;
        } else if (l === n - 1) {
          w = H.get(n, n - 1) * H.get(n - 1, n);
          p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
          q = p * p + w;
          z = Math.sqrt(Math.abs(q));
          H.set(n, n, H.get(n, n) + exshift);
          H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
          x = H.get(n, n);
          if (q >= 0) {
            z = p >= 0 ? p + z : p - z;
            d[n - 1] = x + z;
            d[n] = d[n - 1];
            if (z !== 0) {
              d[n] = x - w / z;
            }
            e[n - 1] = 0;
            e[n] = 0;
            x = H.get(n, n - 1);
            s = Math.abs(x) + Math.abs(z);
            p = x / s;
            q = z / s;
            r = Math.sqrt(p * p + q * q);
            p = p / r;
            q = q / r;
            for (j = n - 1; j < nn; j++) {
              z = H.get(n - 1, j);
              H.set(n - 1, j, q * z + p * H.get(n, j));
              H.set(n, j, q * H.get(n, j) - p * z);
            }
            for (i = 0; i <= n; i++) {
              z = H.get(i, n - 1);
              H.set(i, n - 1, q * z + p * H.get(i, n));
              H.set(i, n, q * H.get(i, n) - p * z);
            }
            for (i = low; i <= high; i++) {
              z = V.get(i, n - 1);
              V.set(i, n - 1, q * z + p * V.get(i, n));
              V.set(i, n, q * V.get(i, n) - p * z);
            }
          } else {
            d[n - 1] = x + p;
            d[n] = x + p;
            e[n - 1] = z;
            e[n] = -z;
          }
          n = n - 2;
          iter = 0;
        } else {
          x = H.get(n, n);
          y = 0;
          w = 0;
          if (l < n) {
            y = H.get(n - 1, n - 1);
            w = H.get(n, n - 1) * H.get(n - 1, n);
          }
          if (iter === 10) {
            exshift += x;
            for (i = low; i <= n; i++) {
              H.set(i, i, H.get(i, i) - x);
            }
            s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
            // eslint-disable-next-line no-multi-assign
            x = y = 0.75 * s;
            w = -0.4375 * s * s;
          }
          if (iter === 30) {
            s = (y - x) / 2;
            s = s * s + w;
            if (s > 0) {
              s = Math.sqrt(s);
              if (y < x) {
                s = -s;
              }
              s = x - w / ((y - x) / 2 + s);
              for (i = low; i <= n; i++) {
                H.set(i, i, H.get(i, i) - s);
              }
              exshift += s;
              // eslint-disable-next-line no-multi-assign
              x = y = w = 0.964;
            }
          }
          iter = iter + 1;
          m = n - 2;
          while (m >= l) {
            z = H.get(m, m);
            r = x - z;
            s = y - z;
            p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
            q = H.get(m + 1, m + 1) - z - r - s;
            r = H.get(m + 2, m + 1);
            s = Math.abs(p) + Math.abs(q) + Math.abs(r);
            p = p / s;
            q = q / s;
            r = r / s;
            if (m === l) {
              break;
            }
            if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
              break;
            }
            m--;
          }
          for (i = m + 2; i <= n; i++) {
            H.set(i, i - 2, 0);
            if (i > m + 2) {
              H.set(i, i - 3, 0);
            }
          }
          for (k = m; k <= n - 1; k++) {
            notlast = k !== n - 1;
            if (k !== m) {
              p = H.get(k, k - 1);
              q = H.get(k + 1, k - 1);
              r = notlast ? H.get(k + 2, k - 1) : 0;
              x = Math.abs(p) + Math.abs(q) + Math.abs(r);
              if (x !== 0) {
                p = p / x;
                q = q / x;
                r = r / x;
              }
            }
            if (x === 0) {
              break;
            }
            s = Math.sqrt(p * p + q * q + r * r);
            if (p < 0) {
              s = -s;
            }
            if (s !== 0) {
              if (k !== m) {
                H.set(k, k - 1, -s * x);
              } else if (l !== m) {
                H.set(k, k - 1, -H.get(k, k - 1));
              }
              p = p + s;
              x = p / s;
              y = q / s;
              z = r / s;
              q = q / p;
              r = r / p;
              for (j = k; j < nn; j++) {
                p = H.get(k, j) + q * H.get(k + 1, j);
                if (notlast) {
                  p = p + r * H.get(k + 2, j);
                  H.set(k + 2, j, H.get(k + 2, j) - p * z);
                }
                H.set(k, j, H.get(k, j) - p * x);
                H.set(k + 1, j, H.get(k + 1, j) - p * y);
              }
              for (i = 0; i <= Math.min(n, k + 3); i++) {
                p = x * H.get(i, k) + y * H.get(i, k + 1);
                if (notlast) {
                  p = p + z * H.get(i, k + 2);
                  H.set(i, k + 2, H.get(i, k + 2) - p * r);
                }
                H.set(i, k, H.get(i, k) - p);
                H.set(i, k + 1, H.get(i, k + 1) - p * q);
              }
              for (i = low; i <= high; i++) {
                p = x * V.get(i, k) + y * V.get(i, k + 1);
                if (notlast) {
                  p = p + z * V.get(i, k + 2);
                  V.set(i, k + 2, V.get(i, k + 2) - p * r);
                }
                V.set(i, k, V.get(i, k) - p);
                V.set(i, k + 1, V.get(i, k + 1) - p * q);
              }
            }
          }
        }
      }
      if (norm === 0) {
        return;
      }
      for (n = nn - 1; n >= 0; n--) {
        p = d[n];
        q = e[n];
        if (q === 0) {
          l = n;
          H.set(n, n, 1);
          for (i = n - 1; i >= 0; i--) {
            w = H.get(i, i) - p;
            r = 0;
            for (j = l; j <= n; j++) {
              r = r + H.get(i, j) * H.get(j, n);
            }
            if (e[i] < 0) {
              z = w;
              s = r;
            } else {
              l = i;
              if (e[i] === 0) {
                H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
                t = (x * s - z * r) / q;
                H.set(i, n, t);
                H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
              }
              t = Math.abs(H.get(i, n));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        } else if (q < 0) {
          l = n - 1;
          if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
            H.set(n - 1, n - 1, q / H.get(n, n - 1));
            H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
          } else {
            cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
            H.set(n - 1, n - 1, cdivres[0]);
            H.set(n - 1, n, cdivres[1]);
          }
          H.set(n, n - 1, 0);
          H.set(n, n, 1);
          for (i = n - 2; i >= 0; i--) {
            ra = 0;
            sa = 0;
            for (j = l; j <= n; j++) {
              ra = ra + H.get(i, j) * H.get(j, n - 1);
              sa = sa + H.get(i, j) * H.get(j, n);
            }
            w = H.get(i, i) - p;
            if (e[i] < 0) {
              z = w;
              r = ra;
              s = sa;
            } else {
              l = i;
              if (e[i] === 0) {
                cdivres = cdiv(-ra, -sa, w, q);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
                vi = (d[i] - p) * 2 * q;
                if (vr === 0 && vi === 0) {
                  vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                }
                cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
                if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                  H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
                  H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
                } else {
                  cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
                  H.set(i + 1, n - 1, cdivres[0]);
                  H.set(i + 1, n, cdivres[1]);
                }
              }
              t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n - 1, H.get(j, n - 1) / t);
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        }
      }
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          for (j = i; j < nn; j++) {
            V.set(i, j, H.get(i, j));
          }
        }
      }
      for (j = nn - 1; j >= low; j--) {
        for (i = low; i <= high; i++) {
          z = 0;
          for (k = low; k <= Math.min(j, high); k++) {
            z = z + V.get(i, k) * H.get(k, j);
          }
          V.set(i, j, z);
        }
      }
    }
    function cdiv(xr, xi, yr, yi) {
      let r, d;
      if (Math.abs(yr) > Math.abs(yi)) {
        r = yi / yr;
        d = yr + r * yi;
        return [(xr + r * xi) / d, (xi - r * xr) / d];
      } else {
        r = yr / yi;
        d = yi + r * yr;
        return [(r * xr + xi) / d, (r * xi - xr) / d];
      }
    }
    class CholeskyDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (!value.isSymmetric()) {
          throw new Error('Matrix is not symmetric');
        }
        let a = value;
        let dimension = a.rows;
        let l = new Matrix$1(dimension, dimension);
        let positiveDefinite = true;
        let i, j, k;
        for (j = 0; j < dimension; j++) {
          let d = 0;
          for (k = 0; k < j; k++) {
            let s = 0;
            for (i = 0; i < k; i++) {
              s += l.get(k, i) * l.get(j, i);
            }
            s = (a.get(j, k) - s) / l.get(k, k);
            l.set(j, k, s);
            d = d + s * s;
          }
          d = a.get(j, j) - d;
          positiveDefinite &= d > 0;
          l.set(j, j, Math.sqrt(Math.max(d, 0)));
          for (k = j + 1; k < dimension; k++) {
            l.set(j, k, 0);
          }
        }
        this.L = l;
        this.positiveDefinite = Boolean(positiveDefinite);
      }
      isPositiveDefinite() {
        return this.positiveDefinite;
      }
      solve(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let l = this.L;
        let dimension = l.rows;
        if (value.rows !== dimension) {
          throw new Error('Matrix dimensions do not match');
        }
        if (this.isPositiveDefinite() === false) {
          throw new Error('Matrix is not positive definite');
        }
        let count = value.columns;
        let B = value.clone();
        let i, j, k;
        for (k = 0; k < dimension; k++) {
          for (j = 0; j < count; j++) {
            for (i = 0; i < k; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        for (k = dimension - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            for (i = k + 1; i < dimension; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        return B;
      }
      get lowerTriangularMatrix() {
        return this.L;
      }
    }
    class nipals {
      constructor(X, options = {}) {
        X = WrapperMatrix2D.checkMatrix(X);
        let {
          Y
        } = options;
        const {
          scaleScores = false,
          maxIterations = 1000,
          terminationCriteria = 1e-10
        } = options;
        let u;
        if (Y) {
          if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
            Y = Matrix$1.columnVector(Y);
          } else {
            Y = WrapperMatrix2D.checkMatrix(Y);
          }
          if (Y.rows !== X.rows) {
            throw new Error('Y should have the same number of rows as X');
          }
          u = Y.getColumnVector(0);
        } else {
          u = X.getColumnVector(0);
        }
        let diff = 1;
        let t, q, w, tOld;
        for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
          w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
          w = w.div(w.norm());
          t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
          if (counter > 0) {
            diff = t.clone().sub(tOld).pow(2).sum();
          }
          tOld = t.clone();
          if (Y) {
            q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
            q = q.div(q.norm());
            u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
          } else {
            u = t;
          }
        }
        if (Y) {
          let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          p = p.div(p.norm());
          let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
          let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
          this.t = t;
          this.p = p.transpose();
          this.w = w.transpose();
          this.q = q;
          this.u = u;
          this.s = t.transpose().mmul(t);
          this.xResidual = xResidual;
          this.yResidual = yResidual;
          this.betas = residual;
        } else {
          this.w = w.transpose();
          this.s = t.transpose().mmul(t).sqrt();
          if (scaleScores) {
            this.t = t.clone().div(this.s.get(0, 0));
          } else {
            this.t = t;
          }
          this.xResidual = X.sub(t.mmul(w.transpose()));
        }
      }
    }
    matrix.AbstractMatrix = AbstractMatrix;
    matrix.CHO = CholeskyDecomposition;
    matrix.CholeskyDecomposition = CholeskyDecomposition;
    matrix.DistanceMatrix = DistanceMatrix;
    matrix.EVD = EigenvalueDecomposition;
    matrix.EigenvalueDecomposition = EigenvalueDecomposition;
    matrix.LU = LuDecomposition;
    matrix.LuDecomposition = LuDecomposition;
    var Matrix_1 = matrix.Matrix = Matrix$1;
    matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
    matrix.MatrixColumnView = MatrixColumnView;
    matrix.MatrixFlipColumnView = MatrixFlipColumnView;
    matrix.MatrixFlipRowView = MatrixFlipRowView;
    matrix.MatrixRowSelectionView = MatrixRowSelectionView;
    matrix.MatrixRowView = MatrixRowView;
    matrix.MatrixSelectionView = MatrixSelectionView;
    matrix.MatrixSubView = MatrixSubView;
    var MatrixTransposeView_1 = matrix.MatrixTransposeView = MatrixTransposeView$1;
    matrix.NIPALS = nipals;
    matrix.Nipals = nipals;
    matrix.QR = QrDecomposition;
    matrix.QrDecomposition = QrDecomposition;
    matrix.SVD = SingularValueDecomposition;
    matrix.SingularValueDecomposition = SingularValueDecomposition;
    matrix.SymmetricMatrix = SymmetricMatrix;
    matrix.WrapperMatrix1D = WrapperMatrix1D;
    matrix.WrapperMatrix2D = WrapperMatrix2D;
    matrix.correlation = correlation;
    matrix.covariance = covariance;
    var _default = matrix.default = Matrix$1;
    matrix.determinant = determinant;
    matrix.inverse = inverse;
    matrix.linearDependencies = linearDependencies;
    matrix.pseudoInverse = pseudoInverse;
    var solve_1 = matrix.solve = solve$1;
    matrix.wrap = wrap;

    const Matrix = Matrix_1;
    const MatrixTransposeView = MatrixTransposeView_1;
    _default.Matrix ? _default.Matrix : Matrix_1;
    const solve = solve_1;

    /**
     * Returns a copy of the data as a Float64Array.
     *
     * @param array - array of numbers
     */
    function xEnsureFloat64(array) {
      xCheck(array);
      if (array instanceof Float64Array) {
        return array.slice(0);
      }
      return Float64Array.from(array);
    }

    function createNumberArray(ArrayConstructor, length) {
      if (ArrayConstructor === Array) {
        return new ArrayConstructor(length).fill(0);
      } else {
        return new ArrayConstructor(length);
      }
    }

    /**
     * Create an array with numbers between "from" and "to" of length "length"
     *
     * @param options - options
     * @return - array of distributed numbers between "from" and "to"
     */
    function createFromToArray(options = {}) {
      const {
        from = 0,
        to = 1,
        length = 1000,
        includeFrom = true,
        includeTo = true,
        distribution = 'uniform'
      } = options;
      const array = new Float64Array(length);
      let div = length;
      if (includeFrom && includeTo) {
        div = length - 1;
      } else if (!includeFrom && includeTo || includeFrom && !includeTo) {
        div = length;
      } else if (!includeFrom && !includeTo) {
        div = length + 1;
      }
      const delta = (to - from) / div;
      if (distribution === 'uniform') {
        if (includeFrom) {
          let index = 0;
          while (index < length) {
            array[index] = from + delta * index;
            index++;
          }
        } else {
          let index = 0;
          while (index < length) {
            array[index] = from + delta * (index + 1);
            index++;
          }
        }
      } else if (distribution === 'log') {
        const base = (to / from) ** (1 / div);
        const firstExponent = Math.log(from) / Math.log(base);
        if (includeFrom) {
          let index = 0;
          while (index < length) {
            array[index] = base ** (firstExponent + index);
            index++;
          }
        } else {
          let index = 0;
          while (index < length) {
            array[index] = base ** (firstExponent + index + 1);
            index++;
          }
        }
      } else {
        throw new Error('distribution must be uniform or log');
      }
      return array;
    }

    /**
     * Computes the maximal value of an array of values
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMaxValue(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let maxValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] > maxValue) {
          maxValue = array[i];
        }
      }
      return maxValue;
    }

    /**
     * Computes the minimal value of an array of values.
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMinValue(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let minValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] < minValue) {
          minValue = array[i];
        }
      }
      return minValue;
    }

    /**
     * Check if the values are separated always by the same difference
     *
     * @param array - Monotone growing array of number
     * @param options
     */
    function xIsEquallySpaced(array, options = {}) {
      if (array.length < 3) return true;
      const {
        tolerance = 0.05
      } = options;
      let maxDx = 0;
      let minDx = Number.MAX_SAFE_INTEGER;
      for (let i = 0; i < array.length - 1; ++i) {
        const absoluteDifference = array[i + 1] - array[i];
        if (absoluteDifference < minDx) {
          minDx = absoluteDifference;
        }
        if (absoluteDifference > maxDx) {
          maxDx = absoluteDifference;
        }
      }
      return (maxDx - minDx) / maxDx < tolerance;
    }

    /**
     * Returns true if x is monotonic.
     *
     * @param array - array of numbers.
     * @returns 1 if monotonic increasing, -1 if monotonic decreasing, 0 if not monotonic.
     */
    function xIsMonotonic(array) {
      if (array.length <= 2) {
        return 1;
      }
      if (array[0] === array[1]) {
        // maybe a constant series
        for (let i = 1; i < array.length - 1; i++) {
          if (array[i] !== array[i + 1]) return 0;
        }
        return 1;
      }
      if (array[0] < array[array.length - 1]) {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] >= array[i + 1]) return 0;
        }
        return 1;
      } else {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] <= array[i + 1]) return 0;
        }
        return -1;
      }
    }

    /**
     * Computes the mean value of an array of values.
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMean(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let sumValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        sumValue += array[i];
      }
      return sumValue / (toIndex - fromIndex + 1);
    }

    /**
     * This function calculates the median absolute deviation (MAD).
     * https://en.wikipedia.org/wiki/Median_absolute_deviation
     * @param array
     */
    function xMedianAbsoluteDeviation(array) {
      const median = xMedian(array);
      const averageDeviations = new Float64Array(array.length);
      for (let i = 0; i < array.length; i++) {
        averageDeviations[i] = Math.abs(array[i] - median);
      }
      return {
        median,
        mad: xMedian(averageDeviations)
      };
    }

    /**
     * Return min and max values of an array.
     *
     * @param array - array of number
     * @returns - Object with 2 properties, min and max
     */
    function xMinMaxValues(array) {
      xCheck(array);
      let min = array[0];
      let max = array[0];
      for (const value of array) {
        if (value < min) min = value;
        if (value > max) max = value;
      }
      return {
        min,
        max
      };
    }

    var d3Array = {exports: {}};

    (function (module, exports) {
      (function (global, factory) {
        factory(exports) ;
      })(commonjsGlobal, function (exports) {

        function ascending(a, b) {
          return a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN;
        }
        function bisector(compare) {
          if (compare.length === 1) compare = ascendingComparator(compare);
          return {
            left: function (a, x, lo, hi) {
              if (lo == null) lo = 0;
              if (hi == null) hi = a.length;
              while (lo < hi) {
                var mid = lo + hi >>> 1;
                if (compare(a[mid], x) < 0) lo = mid + 1;else hi = mid;
              }
              return lo;
            },
            right: function (a, x, lo, hi) {
              if (lo == null) lo = 0;
              if (hi == null) hi = a.length;
              while (lo < hi) {
                var mid = lo + hi >>> 1;
                if (compare(a[mid], x) > 0) hi = mid;else lo = mid + 1;
              }
              return lo;
            }
          };
        }
        function ascendingComparator(f) {
          return function (d, x) {
            return ascending(f(d), x);
          };
        }
        var ascendingBisect = bisector(ascending);
        var bisectRight = ascendingBisect.right;
        var bisectLeft = ascendingBisect.left;
        function descending(a, b) {
          return b < a ? -1 : b > a ? 1 : b >= a ? 0 : NaN;
        }
        function number$1(x) {
          return x === null ? NaN : +x;
        }
        function variance(array, f) {
          var n = array.length,
            m = 0,
            a,
            d,
            s = 0,
            i = -1,
            j = 0;
          if (f == null) {
            while (++i < n) {
              if (!isNaN(a = number$1(array[i]))) {
                d = a - m;
                m += d / ++j;
                s += d * (a - m);
              }
            }
          } else {
            while (++i < n) {
              if (!isNaN(a = number$1(f(array[i], i, array)))) {
                d = a - m;
                m += d / ++j;
                s += d * (a - m);
              }
            }
          }
          if (j > 1) return s / (j - 1);
        }
        function deviation(array, f) {
          var v = variance(array, f);
          return v ? Math.sqrt(v) : v;
        }
        function extent(array, f) {
          var i = -1,
            n = array.length,
            a,
            b,
            c;
          if (f == null) {
            while (++i < n) if ((b = array[i]) != null && b >= b) {
              a = c = b;
              break;
            }
            while (++i < n) if ((b = array[i]) != null) {
              if (a > b) a = b;
              if (c < b) c = b;
            }
          } else {
            while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
              a = c = b;
              break;
            }
            while (++i < n) if ((b = f(array[i], i, array)) != null) {
              if (a > b) a = b;
              if (c < b) c = b;
            }
          }
          return [a, c];
        }
        function constant(x) {
          return function () {
            return x;
          };
        }
        function identity(x) {
          return x;
        }
        function range(start, stop, step) {
          start = +start, stop = +stop, step = (n = arguments.length) < 2 ? (stop = start, start = 0, 1) : n < 3 ? 1 : +step;
          var i = -1,
            n = Math.max(0, Math.ceil((stop - start) / step)) | 0,
            range = new Array(n);
          while (++i < n) {
            range[i] = start + i * step;
          }
          return range;
        }
        var e10 = Math.sqrt(50);
        var e5 = Math.sqrt(10);
        var e2 = Math.sqrt(2);
        function ticks(start, stop, count) {
          var step = tickStep(start, stop, count);
          return range(Math.ceil(start / step) * step, Math.floor(stop / step) * step + step / 2,
          // inclusive
          step);
        }
        function tickStep(start, stop, count) {
          var step0 = Math.abs(stop - start) / Math.max(0, count),
            step1 = Math.pow(10, Math.floor(Math.log(step0) / Math.LN10)),
            error = step0 / step1;
          if (error >= e10) step1 *= 10;else if (error >= e5) step1 *= 5;else if (error >= e2) step1 *= 2;
          return stop < start ? -step1 : step1;
        }
        function sturges(values) {
          return Math.ceil(Math.log(values.length) / Math.LN2) + 1;
        }
        function number(x) {
          return +x;
        }
        function histogram() {
          var value = identity,
            domain = extent,
            threshold = sturges;
          function histogram(data) {
            var i,
              n = data.length,
              x,
              values = new Array(n);

            // Coerce values to numbers.
            for (i = 0; i < n; ++i) {
              values[i] = +value(data[i], i, data);
            }
            var xz = domain(values),
              x0 = +xz[0],
              x1 = +xz[1],
              tz = threshold(values, x0, x1);

            // Convert number of thresholds into uniform thresholds.
            if (!Array.isArray(tz)) tz = ticks(x0, x1, +tz);

            // Coerce thresholds to numbers, ignoring any outside the domain.
            var m = tz.length;
            for (i = 0; i < m; ++i) tz[i] = +tz[i];
            while (tz[0] <= x0) tz.shift(), --m;
            while (tz[m - 1] >= x1) tz.pop(), --m;
            var bins = new Array(m + 1),
              bin;

            // Initialize bins.
            for (i = 0; i <= m; ++i) {
              bin = bins[i] = [];
              bin.x0 = i > 0 ? tz[i - 1] : x0;
              bin.x1 = i < m ? tz[i] : x1;
            }

            // Assign data to bins by value, ignoring any outside the domain.
            for (i = 0; i < n; ++i) {
              x = values[i];
              if (x0 <= x && x <= x1) {
                bins[bisectRight(tz, x, 0, m)].push(data[i]);
              }
            }
            return bins;
          }
          histogram.value = function (_) {
            return arguments.length ? (value = typeof _ === "function" ? _ : constant(+_), histogram) : value;
          };
          histogram.domain = function (_) {
            return arguments.length ? (domain = typeof _ === "function" ? _ : constant([+_[0], +_[1]]), histogram) : domain;
          };
          histogram.thresholds = function (_) {
            if (!arguments.length) return threshold;
            threshold = typeof _ === "function" ? _ : Array.isArray(_) ? constant(Array.prototype.map.call(_, number)) : constant(+_);
            return histogram;
          };
          return histogram;
        }
        function quantile(array, p, f) {
          if (f == null) f = number$1;
          if (!(n = array.length)) return;
          if ((p = +p) <= 0 || n < 2) return +f(array[0], 0, array);
          if (p >= 1) return +f(array[n - 1], n - 1, array);
          var n,
            h = (n - 1) * p,
            i = Math.floor(h),
            a = +f(array[i], i, array),
            b = +f(array[i + 1], i + 1, array);
          return a + (b - a) * (h - i);
        }
        function freedmanDiaconis(values, min, max) {
          values.sort(ascending);
          return Math.ceil((max - min) / (2 * (quantile(values, 0.75) - quantile(values, 0.25)) * Math.pow(values.length, -1 / 3)));
        }
        function scott(values, min, max) {
          return Math.ceil((max - min) / (3.5 * deviation(values) * Math.pow(values.length, -1 / 3)));
        }
        function max(array, f) {
          var i = -1,
            n = array.length,
            a,
            b;
          if (f == null) {
            while (++i < n) if ((b = array[i]) != null && b >= b) {
              a = b;
              break;
            }
            while (++i < n) if ((b = array[i]) != null && b > a) a = b;
          } else {
            while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
              a = b;
              break;
            }
            while (++i < n) if ((b = f(array[i], i, array)) != null && b > a) a = b;
          }
          return a;
        }
        function mean(array, f) {
          var s = 0,
            n = array.length,
            a,
            i = -1,
            j = n;
          if (f == null) {
            while (++i < n) if (!isNaN(a = number$1(array[i]))) s += a;else --j;
          } else {
            while (++i < n) if (!isNaN(a = number$1(f(array[i], i, array)))) s += a;else --j;
          }
          if (j) return s / j;
        }
        function median(array, f) {
          var numbers = [],
            n = array.length,
            a,
            i = -1;
          if (f == null) {
            while (++i < n) if (!isNaN(a = number$1(array[i]))) numbers.push(a);
          } else {
            while (++i < n) if (!isNaN(a = number$1(f(array[i], i, array)))) numbers.push(a);
          }
          return quantile(numbers.sort(ascending), 0.5);
        }
        function merge(arrays) {
          var n = arrays.length,
            m,
            i = -1,
            j = 0,
            merged,
            array;
          while (++i < n) j += arrays[i].length;
          merged = new Array(j);
          while (--n >= 0) {
            array = arrays[n];
            m = array.length;
            while (--m >= 0) {
              merged[--j] = array[m];
            }
          }
          return merged;
        }
        function min(array, f) {
          var i = -1,
            n = array.length,
            a,
            b;
          if (f == null) {
            while (++i < n) if ((b = array[i]) != null && b >= b) {
              a = b;
              break;
            }
            while (++i < n) if ((b = array[i]) != null && a > b) a = b;
          } else {
            while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
              a = b;
              break;
            }
            while (++i < n) if ((b = f(array[i], i, array)) != null && a > b) a = b;
          }
          return a;
        }
        function pairs(array) {
          var i = 0,
            n = array.length - 1,
            p = array[0],
            pairs = new Array(n < 0 ? 0 : n);
          while (i < n) pairs[i] = [p, p = array[++i]];
          return pairs;
        }
        function permute(array, indexes) {
          var i = indexes.length,
            permutes = new Array(i);
          while (i--) permutes[i] = array[indexes[i]];
          return permutes;
        }
        function scan(array, compare) {
          if (!(n = array.length)) return;
          var i = 0,
            n,
            j = 0,
            xi,
            xj = array[j];
          if (!compare) compare = ascending;
          while (++i < n) if (compare(xi = array[i], xj) < 0 || compare(xj, xj) !== 0) xj = xi, j = i;
          if (compare(xj, xj) === 0) return j;
        }
        function shuffle(array, i0, i1) {
          var m = (i1 == null ? array.length : i1) - (i0 = i0 == null ? 0 : +i0),
            t,
            i;
          while (m) {
            i = Math.random() * m-- | 0;
            t = array[m + i0];
            array[m + i0] = array[i + i0];
            array[i + i0] = t;
          }
          return array;
        }
        function sum(array, f) {
          var s = 0,
            n = array.length,
            a,
            i = -1;
          if (f == null) {
            while (++i < n) if (a = +array[i]) s += a; // Note: zero and null are equivalent.
          } else {
            while (++i < n) if (a = +f(array[i], i, array)) s += a;
          }
          return s;
        }
        function transpose(matrix) {
          if (!(n = matrix.length)) return [];
          for (var i = -1, m = min(matrix, length), transpose = new Array(m); ++i < m;) {
            for (var j = -1, n, row = transpose[i] = new Array(n); ++j < n;) {
              row[j] = matrix[j][i];
            }
          }
          return transpose;
        }
        function length(d) {
          return d.length;
        }
        function zip() {
          return transpose(arguments);
        }
        var version = "0.7.1";
        exports.version = version;
        exports.bisect = bisectRight;
        exports.bisectRight = bisectRight;
        exports.bisectLeft = bisectLeft;
        exports.ascending = ascending;
        exports.bisector = bisector;
        exports.descending = descending;
        exports.deviation = deviation;
        exports.extent = extent;
        exports.histogram = histogram;
        exports.thresholdFreedmanDiaconis = freedmanDiaconis;
        exports.thresholdScott = scott;
        exports.thresholdSturges = sturges;
        exports.max = max;
        exports.mean = mean;
        exports.median = median;
        exports.merge = merge;
        exports.min = min;
        exports.pairs = pairs;
        exports.permute = permute;
        exports.quantile = quantile;
        exports.range = range;
        exports.scan = scan;
        exports.shuffle = shuffle;
        exports.sum = sum;
        exports.ticks = ticks;
        exports.tickStep = tickStep;
        exports.transpose = transpose;
        exports.variance = variance;
        exports.zip = zip;
      });
    })(d3Array, d3Array.exports);

    const {
      bisectRight
    } = d3Array.exports;
    const quincunx = (u, v, w, q) => {
      const n = u.length - 1;
      u[0] = 0;
      v[0] = 0;
      w[0] = 0;
      v[1] = v[1] / u[1];
      w[1] = w[1] / u[1];
      for (let i = 2; i < n; ++i) {
        u[i] = u[i] - u[i - 2] * w[i - 2] * w[i - 2] - u[i - 1] * v[i - 1] * v[i - 1];
        v[i] = (v[i] - u[i - 1] * v[i - 1] * w[i - 1]) / u[i];
        w[i] = w[i] / u[i];
      }
      for (let i = 2; i < n; ++i) {
        q[i] = q[i] - v[i - 1] * q[i - 1] - w[i - 2] * q[i - 2];
      }
      for (let i = 1; i < n; ++i) {
        q[i] = q[i] / u[i];
      }
      q[n - 2] = q[n - 2] - v[n - 2] * q[n - 1];
      for (let i = n - 3; i > 0; --i) {
        q[i] = q[i] - v[i] * q[i + 1] - w[i] * q[i + 2];
      }
    };
    const smoothingSpline = (x, y, sigma, lambda) => {
      const n = x.length - 1;
      const h = new Array(n + 1);
      const r = new Array(n + 1);
      const f = new Array(n + 1);
      const p = new Array(n + 1);
      const q = new Array(n + 1);
      const u = new Array(n + 1);
      const v = new Array(n + 1);
      const w = new Array(n + 1);
      const params = x.map(() => [0, 0, 0, 0]);
      params.pop();
      const mu = 2 * (1 - lambda) / (3 * lambda);
      for (let i = 0; i < n; ++i) {
        h[i] = x[i + 1] - x[i];
        r[i] = 3 / h[i];
      }
      q[0] = 0;
      for (let i = 1; i < n; ++i) {
        f[i] = -(r[i - 1] + r[i]);
        p[i] = 2 * (x[i + 1] - x[i - 1]);
        q[i] = 3 * (y[i + 1] - y[i]) / h[i] - 3 * (y[i] - y[i - 1]) / h[i - 1];
      }
      q[n] = 0;
      for (let i = 1; i < n; ++i) {
        u[i] = r[i - 1] * r[i - 1] * sigma[i - 1] + f[i] * f[i] * sigma[i] + r[i] * r[i] * sigma[i + 1];
        u[i] = mu * u[i] + p[i];
      }
      for (let i = 1; i < n - 1; ++i) {
        v[i] = f[i] * r[i] * sigma[i] + r[i] * f[i + 1] * sigma[i + 1];
        v[i] = mu * v[i] + h[i];
      }
      for (let i = 1; i < n - 2; ++i) {
        w[i] = mu * r[i] * r[i + 1] * sigma[i + 1];
      }
      quincunx(u, v, w, q);
      params[0][3] = y[0] - mu * r[0] * q[1] * sigma[0];
      params[1][3] = y[1] - mu * (f[1] * q[1] + r[1] * q[2]) * sigma[0];
      params[0][0] = q[1] / (3 * h[0]);
      params[0][1] = 0;
      params[0][2] = (params[1][3] - params[0][3]) / h[0] - q[1] * h[0] / 3;
      r[0] = 0;
      for (let i = 1; i < n; ++i) {
        params[i][0] = (q[i + 1] - q[i]) / (3 * h[i]);
        params[i][1] = q[i];
        params[i][2] = (q[i] + q[i - 1]) * h[i - 1] + params[i - 1][2];
        params[i][3] = r[i - 1] * q[i - 1] + f[i] * q[i] + r[i] * q[i + 1];
        params[i][3] = y[i] - mu * params[i][3] * sigma[i];
      }
      return params;
    };
    class SplineInterpolator {
      constructor(xIn, yIn, lambda = 1) {
        const indices = xIn.map((_, i) => i);
        indices.sort((i, j) => xIn[i] - xIn[j]);
        const x = indices.map(i => xIn[i]);
        const y = indices.map(i => yIn[i]);
        const n = indices.length;
        const sigma = indices.map(() => 1);
        this.n = n;
        this.x = x;
        this.y = y;
        this.params = smoothingSpline(x, y, sigma, lambda);
      }
      interpolate(v) {
        if (v === this.x[this.n - 1]) {
          return this.y[this.n - 1];
        }
        const i = Math.min(Math.max(0, bisectRight(this.x, v) - 1), this.n - 2);
        const [a, b, c, d] = this.params[i];
        v = v - this.x[i];
        return a * v * v * v + b * v * v + c * v + d;
      }
      max(step = 100) {
        const xStart = this.x[0];
        const xStop = this.x[this.n - 1];
        const delta = (xStop - xStart) / step;
        let maxValue = -Infinity;
        for (let i = 0, x = xStart; i < step; ++i, x += delta) {
          const y = this.interpolate(x);
          if (y > maxValue) {
            maxValue = y;
          }
        }
        return maxValue;
      }
      min(step = 100) {
        const xStart = this.x[0];
        const xStop = this.x[this.n - 1];
        const delta = (xStop - xStart) / step;
        let minValue = Infinity;
        for (let i = 0, x = xStart; i < step; ++i, x += delta) {
          const y = this.interpolate(x);
          if (y < minValue) {
            minValue = y;
          }
        }
        return minValue;
      }
      domain() {
        return [this.x[0], this.x[this.x.length - 1]];
      }
      range() {
        return [this.min(), this.max()];
      }
      curve(nInterval, domain = null) {
        domain = domain || this.domain();
        const delta = (domain[1] - domain[0]) / (nInterval - 1);
        const vals = new Array(nInterval);
        for (let i = 0; i < nInterval; ++i) {
          const x = delta * i + domain[0];
          vals[i] = [x, this.interpolate(x)];
        }
        return vals;
      }
    }
    var splineInterpolator = SplineInterpolator;
    var SplineInterpolator$1 = splineInterpolator;

    /* eslint-disable @typescript-eslint/no-loss-of-precision */
    /*
    Adapted from: https://github.com/compute-io/erfcinv/blob/aa116e23883839359e310ad41a7c42f72815fc1e/lib/number.js

    The MIT License (MIT)

    Copyright (c) 2014-2015 The Compute.io Authors. All rights reserved.

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.


    Boost Software License - Version 1.0 - August 17th, 2003

    Permission is hereby granted, free of charge, to any person or organization obtaining a copy of the software and accompanying documentation covered by this license (the "Software") to use, reproduce, display, distribute, execute, and transmit the Software, and to prepare derivative works of the Software, and to permit third-parties to whom the Software is furnished to do so, all subject to the following:

    The copyright notices in the Software and this entire statement, including the above license grant, this restriction and the following disclaimer, must be included in all copies of the Software, in whole or in part, and all derivative works of the Software, unless such copies or derivative works are solely in the form of machine-executable object code generated by a source language processor.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
    */
    // Coefficients for erfcinv on [0, 0.5]:
    const Y1 = 8.91314744949340820313e-2;
    const P1 = [-5.38772965071242932965e-3, 8.22687874676915743155e-3, 2.19878681111168899165e-2, -3.65637971411762664006e-2, -1.26926147662974029034e-2, 3.34806625409744615033e-2, -8.36874819741736770379e-3, -5.08781949658280665617e-4];
    const Q1 = [8.86216390456424707504e-4, -2.33393759374190016776e-3, 7.95283687341571680018e-2, -5.27396382340099713954e-2, -7.1228902341542847553e-1, 6.62328840472002992063e-1, 1.56221558398423026363, -1.56574558234175846809, -9.70005043303290640362e-1, 1];
    // Coefficients for erfcinv for 0.5 > 1-x >= 0:
    const Y2 = 2.249481201171875;
    const P2 = [-3.67192254707729348546, 2.11294655448340526258e1, 1.7445385985570866523e1, -4.46382324441786960818e1, -1.88510648058714251895e1, 1.76447298408374015486e1, 8.37050328343119927838, 1.05264680699391713268e-1, -2.02433508355938759655e-1];
    const Q2 = [1.72114765761200282724, -2.26436933413139721736e1, 1.08268667355460159008e1, 4.85609213108739935468e1, -2.01432634680485188801e1, -2.86608180499800029974e1, 3.9713437953343869095, 6.24264124854247537712, 1];
    // Coefficients for erfcinv for sqrt( -log(1-x)):
    const Y3 = 8.07220458984375e-1;
    const P3 = [-6.81149956853776992068e-10, 2.85225331782217055858e-8, -6.79465575181126350155e-7, 2.14558995388805277169e-3, 2.90157910005329060432e-2, 1.42869534408157156766e-1, 3.37785538912035898924e-1, 3.87079738972604337464e-1, 1.17030156341995252019e-1, -1.63794047193317060787e-1, -1.31102781679951906451e-1];
    const Q3 = [1.105924229346489121e-2, 1.52264338295331783612e-1, 8.48854343457902036425e-1, 2.59301921623620271374, 4.77846592945843778382, 5.38168345707006855425, 3.46625407242567245975, 1];
    const Y4 = 9.3995571136474609375e-1;
    const P4 = [2.66339227425782031962e-12, -2.30404776911882601748e-10, 4.60469890584317994083e-6, 1.57544617424960554631e-4, 1.87123492819559223345e-3, 9.50804701325919603619e-3, 1.85573306514231072324e-2, -2.22426529213447927281e-3, -3.50353787183177984712e-2];
    const Q4 = [7.64675292302794483503e-5, 2.63861676657015992959e-3, 3.41589143670947727934e-2, 2.20091105764131249824e-1, 7.62059164553623404043e-1, 1.3653349817554063097, 1];
    const Y5 = 9.8362827301025390625e-1;
    const P5 = [9.9055709973310326855e-17, -2.81128735628831791805e-14, 4.62596163522878599135e-9, 4.49696789927706453732e-7, 1.49624783758342370182e-5, 2.09386317487588078668e-4, 1.05628862152492910091e-3, -1.12951438745580278863e-3, -1.67431005076633737133e-2];
    const Q5 = [2.82243172016108031869e-7, 2.75335474764726041141e-5, 9.64011807005165528527e-4, 1.60746087093676504695e-2, 1.38151865749083321638e-1, 5.91429344886417493481e-1, 1];
    /**
     * Polyval.
     *
     * @param c - Array of Number.
     * @param x - Number.
     * @returns Number.
     */
    function polyval(c, x) {
      let p = 0;
      for (const coef of c) {
        p = p * x + coef;
      }
      return p;
    }
    /**
     * Calculates a rational approximation.
     *
     * @private
     * @param x - Number.
     * @param v - Number.
     * @param P - Array of polynomial coefficients.
     * @param Q - Array of polynomial coefficients.
     * @param Y - Number.
     * @returns Rational approximation.
     */
    function calc(x, v, P, Q, Y) {
      const s = x - v;
      const r = polyval(P, s) / polyval(Q, s);
      return Y * x + r * x;
    }
    /**
     * Evaluates the complementary inverse error function for an input value.
     *
     * @private
     * @param x - Input value.
     * @returns Evaluated complementary inverse error function.
     */
    function erfcinv(x) {
      let sign = false;
      let val;
      let q;
      let g;
      let r;
      // [1] Special cases...
      // NaN:
      if (Number.isNaN(x)) {
        return Number.NaN;
      }
      // x not on the interval: [0,2]
      if (x < 0 || x > 2) {
        throw new RangeError(`erfcinv()::invalid input argument. Value must be on the interval [0,2]. Value: \`${x}\`.`);
      }
      if (x === 0) {
        return Number.POSITIVE_INFINITY;
      }
      if (x === 2) {
        return Number.NEGATIVE_INFINITY;
      }
      if (x === 1) {
        return 0;
      }
      // [2] Get the sign and make use of `erfc` reflection formula: `erfc(-z)=2 - erfc(z)`...
      if (x > 1) {
        q = 2 - x;
        x = 1 - q;
        sign = true;
      } else {
        q = x;
        x = 1 - x;
      }
      // [3] |x| <= 0.5
      if (x <= 0.5) {
        g = x * (x + 10);
        r = polyval(P1, x) / polyval(Q1, x);
        val = g * Y1 + g * r;
        return sign ? -val : val;
      }
      // [4] 1-|x| >= 0.25
      if (q >= 0.25) {
        g = Math.sqrt(-2 * Math.log(q));
        q = q - 0.25;
        r = polyval(P2, q) / polyval(Q2, q);
        val = g / (Y2 + r);
        return sign ? -val : val;
      }
      q = Math.sqrt(-Math.log(q));
      // [5] q < 3
      if (q < 3) {
        return calc(q, 1.125, P3, Q3, Y3);
      }
      // [6] q < 6
      if (q < 6) {
        return calc(q, 3, P4, Q4, Y4);
      }
      // Note that the smallest number in JavaScript is 5e-324. Math.sqrt( -Math.log( 5e-324 ) ) ~27.2844
      return calc(q, 6, P5, Q5, Y5);
      // Note that in the boost library, they are able to go to much smaller values, as 128 bit long doubles support ~1e-5000; something which JavaScript does not natively support.
    }

    /**
     * RayleighCdf.
     *
     * @param x - data
     * @param sigma - standard deviation
     * @returns - rayleigh cdf
     */
    function rayleighCdf(x, sigma = 1) {
      if (x < 0) {
        return 0;
      }
      return -Math.expm1(-(x ** 2) / (2 * sigma ** 2));
    }

    /* eslint-disable max-lines-per-function */
    /**
     * Determine noise level by san plot methodology (https://doi.org/10.1002/mrc.4882)
     *
     * @param array - real or magnitude spectra data.
     * @param options - options
     * @returns noise level
     */
    function xNoiseSanPlot(array, options = {}) {
      const {
        mask,
        cutOff,
        refine = true,
        magnitudeMode = false,
        scaleFactor = 1,
        factorStd = 5,
        fixOffset = true
      } = options;
      let input;
      if (Array.isArray(mask) && mask.length === array.length) {
        input = new Float64Array(array.filter((_e, i) => !mask[i]));
      } else {
        input = new Float64Array(array);
      }
      if (scaleFactor > 1) {
        for (let i = 0; i < input.length; i++) {
          input[i] *= scaleFactor;
        }
      }
      input = input.sort().reverse();
      if (fixOffset && !magnitudeMode) {
        const medianIndex = Math.floor(input.length / 2);
        const median = 0.5 * (input[medianIndex] + input[medianIndex + 1]);
        for (let i = 0; i < input.length; i++) {
          input[i] -= median;
        }
      }
      const firstNegativeValueIndex = input[input.length - 1] >= 0 ? input.length : input.findIndex(e => e < 0);
      let lastPositiveValueIndex = firstNegativeValueIndex - 1;
      for (let i = lastPositiveValueIndex; i >= 0; i--) {
        if (input[i] > 0) {
          lastPositiveValueIndex = i;
          break;
        }
      }
      const signPositive = input.slice(0, lastPositiveValueIndex + 1);
      const signNegative = input.slice(firstNegativeValueIndex);
      const cutOffDist = cutOff || determineCutOff(signPositive, {
        magnitudeMode
      });
      const pIndex = Math.floor(signPositive.length * cutOffDist);
      const initialNoiseLevelPositive = signPositive[pIndex];
      const skyPoint = signPositive[0];
      let initialNoiseLevelNegative;
      if (signNegative.length > 0) {
        const nIndex = Math.floor(signNegative.length * (1 - cutOffDist));
        initialNoiseLevelNegative = -1 * signNegative[nIndex];
      } else {
        initialNoiseLevelNegative = 0;
      }
      let noiseLevelPositive = initialNoiseLevelPositive;
      let noiseLevelNegative = initialNoiseLevelNegative;
      let cloneSignPositive = signPositive.slice();
      let cloneSignNegative = signNegative.slice();
      let cutOffSignalsIndexPlus = 0;
      let cutOffSignalsIndexNeg = 2;
      if (refine) {
        let cutOffSignals = noiseLevelPositive * factorStd;
        cutOffSignalsIndexPlus = signPositive.findIndex(e => e < cutOffSignals);
        if (cutOffSignalsIndexPlus > -1) {
          cloneSignPositive = signPositive.slice(cutOffSignalsIndexPlus);
          noiseLevelPositive = cloneSignPositive[Math.floor(cloneSignPositive.length * cutOffDist)];
        }
        cutOffSignals = noiseLevelNegative * factorStd;
        cutOffSignalsIndexNeg = signNegative.findIndex(e => e < cutOffSignals);
        if (cutOffSignalsIndexNeg > -1) {
          cloneSignNegative = signNegative.slice(cutOffSignalsIndexNeg);
          noiseLevelNegative = cloneSignPositive[Math.floor(cloneSignNegative.length * (1 - cutOffDist))];
        }
      }
      const correctionFactor = -simpleNormInvNumber(cutOffDist / 2, {
        magnitudeMode
      });
      let effectiveCutOffDist, refinedCorrectionFactor;
      if (refine && cutOffSignalsIndexPlus > -1) {
        effectiveCutOffDist = (cutOffDist * cloneSignPositive.length + cutOffSignalsIndexPlus) / (cloneSignPositive.length + cutOffSignalsIndexPlus);
        refinedCorrectionFactor = -1 * simpleNormInvNumber(effectiveCutOffDist / 2, {
          magnitudeMode
        });
        noiseLevelPositive /= refinedCorrectionFactor;
        if (cutOffSignalsIndexNeg > -1) {
          effectiveCutOffDist = (cutOffDist * cloneSignNegative.length + cutOffSignalsIndexNeg) / (cloneSignNegative.length + cutOffSignalsIndexNeg);
          refinedCorrectionFactor = -1 * simpleNormInvNumber(effectiveCutOffDist / 2, {
            magnitudeMode
          });
          if (noiseLevelNegative !== 0) {
            noiseLevelNegative /= refinedCorrectionFactor;
          }
        }
      } else {
        noiseLevelPositive /= correctionFactor;
        noiseLevelNegative /= correctionFactor;
      }
      return {
        positive: noiseLevelPositive,
        negative: noiseLevelNegative,
        snr: skyPoint / noiseLevelPositive,
        sanplot: generateSanPlot(input, {
          fromTo: {
            positive: {
              from: 0,
              to: lastPositiveValueIndex
            },
            negative: {
              from: firstNegativeValueIndex,
              to: input.length
            }
          }
        })
      };
    }
    /**
     * DetermineCutOff.
     *
     * @param signPositive - Array of numbers.
     * @param [options = {}] - Options.
     * @param [options.mask] - Boolean array to filter data, if the i-th element is true then the i-th element of the distribution will be ignored.
     * @param [options.scaleFactor=1] - Factor to scale the data input[i]*=scaleFactor.
     * @param [options.cutOff] - Percent of positive signal distribution where the noise level will be determined, if it is not defined the program calculate it.
     * @param [options.factorStd=5] - Factor times std to determine what will be marked as signals.
     * @param [options.refine=true] - If true the noise level will be recalculated get out the signals using factorStd.
     * @param [options.fixOffset=true] - If the baseline is correct, the midpoint of distribution should be zero. If true, the distribution will be centered.
     * @param [options.logBaseY=2] - Log scale to apply in the intensity axis in order to avoid big numbers.
     * @param options.magnitudeMode -
     * @param options.considerList -
     * @param options.considerList.from -
     * @param options.considerList.step -
     * @param options.considerList.to -
     * @param options.fromTo -
     * @returns Result.
     */
    function determineCutOff(signPositive, options = {}) {
      const {
        magnitudeMode = false,
        considerList = {
          from: 0.5,
          step: 0.1,
          to: 0.9
        }
      } = options;
      //generate a list of values for
      const cutOff = [];
      const indexMax = signPositive.length - 1;
      for (let i = 0.01; i <= 0.99; i += 0.01) {
        const index = Math.round(indexMax * i);
        const value = -signPositive[index] / simpleNormInvNumber(i / 2, {
          magnitudeMode
        });
        cutOff.push([i, value]);
      }
      let minKi = Number.MAX_SAFE_INTEGER;
      const {
        from,
        to,
        step
      } = considerList;
      const delta = step / 2;
      let whereToCutStat = 0.5;
      for (let i = from; i <= to; i += step) {
        const floor = i - delta;
        const top = i + delta;
        const elementsOfCutOff = cutOff.filter(e => e[0] < top && e[0] > floor);
        const averageValue = elementsOfCutOff.reduce((a, b) => a + Math.abs(b[1]), 0);
        let kiSqrt = 0;
        for (const element of elementsOfCutOff) {
          kiSqrt += (element[1] - averageValue) ** 2;
        }
        if (kiSqrt < minKi) {
          minKi = kiSqrt;
          whereToCutStat = i;
        }
      }
      return whereToCutStat;
    }
    function simpleNormInvNumber(data, options) {
      return simpleNormInv([data], options)[0];
    }
    /**
     * SimpleNormInvs.
     *
     * @param data - Data array.
     * @param options
     */
    function simpleNormInv(data, options = {}) {
      const {
        magnitudeMode = false
      } = options;
      const from = 0;
      const to = 2;
      const step = 0.01;
      const xTraining = createArray(from, to, step);
      const result = new Float64Array(data.length);
      const yTraining = new Float64Array(xTraining.length);
      if (magnitudeMode) {
        const factor = 1;
        for (let i = 0; i < yTraining.length; i++) {
          const finalInput = xTraining[i] * factor;
          yTraining[i] = 1 - rayleighCdf(finalInput);
        }
        const interp = new SplineInterpolator$1(xTraining, yTraining);
        for (let i = 0; i < result.length; i++) {
          const yValue = 2 * data[i];
          result[i] = -1 * interp.interpolate(yValue);
        }
      } else {
        for (let i = 0; i < result.length; i++) {
          result[i] = -1 * Math.SQRT2 * erfcinv(2 * data[i]);
        }
      }
      return result;
    }
    /**
     * CreateArray.
     *
     * @param from - From.
     * @param to - To.
     * @param step - Step.
     * @returns Array of results.
     */
    function createArray(from, to, step) {
      const length = Math.abs((from - to) / step + 1);
      const result = [];
      for (let i = 0; i < length; i++) {
        result.push(from + i * step);
      }
      return result;
    }
    /**
     * GenerateSanPlot.
     *
     * @param array - Array.
     * @param [options = {}] - Options.
     * @param [options.mask] - Boolean array to filter data, if the i-th element is true then the i-th element of the distribution will be ignored.
     * @param [options.scaleFactor=1] - Factor to scale the data input[i]*=scaleFactor.
     * @param [options.cutOff] - Percent of positive signal distribution where the noise level will be determined, if it is not defined the program calculate it.
     * @param [options.factorStd=5] - Factor times std to determine what will be marked as signals.
     * @param [options.refine=true] - If true the noise level will be recalculated get out the signals using factorStd.
     * @param [options.fixOffset=true] - If the baseline is correct, the midpoint of distribution should be zero. If true, the distribution will be centered.
     * @param [options.logBaseY=2] - Log scale to apply in the intensity axis in order to avoid big numbers.
     * @param options.magnitudeMode -
     * @param options.considerList -
     * @param options.considerList.from -
     * @param options.considerList.step -
     * @param options.considerList.to -
     * @param options.fromTo -
     * @returns Results.
     */
    function generateSanPlot(array, options = {}) {
      const {
        fromTo,
        logBaseY = 2
      } = options;
      const sanplot = {};
      for (const key in fromTo) {
        const {
          from,
          to
        } = fromTo[key];
        sanplot[key] = from !== to ? scale$1(array.slice(from, to), {
          logBaseY
        }) : {
          x: [],
          y: []
        };
        if (key === 'negative') {
          sanplot[key].y.reverse();
        }
      }
      return sanplot;
    }
    /**
     * Scale.
     *
     * @param array - Array.
     * @param [options = {}] - Options.
     * @param [options.mask] - Boolean array to filter data, if the i-th element is true then the i-th element of the distribution will be ignored.
     * @param [options.scaleFactor=1] - Factor to scale the data input[i]*=scaleFactor.
     * @param [options.cutOff] - Percent of positive signal distribution where the noise level will be determined, if it is not defined the program calculate it.
     * @param [options.factorStd=5] - Factor times std to determine what will be marked as signals.
     * @param [options.refine=true] - If true the noise level will be recalculated get out the signals using factorStd.
     * @param [options.fixOffset=true] - If the baseline is correct, the midpoint of distribution should be zero. If true, the distribution will be centered.
     * @param [options.logBaseY=2] - Log scale to apply in the intensity axis in order to avoid big numbers.
     * @param options.magnitudeMode -
     * @param options.considerList -
     * @param options.considerList.from -
     * @param options.considerList.step -
     * @param options.considerList.to -
     * @param options.fromTo -
     * @returns Results.
     */
    function scale$1(array, options = {}) {
      const {
        log10,
        abs
      } = Math;
      const {
        logBaseY
      } = options;
      if (logBaseY) {
        array = array.slice();
        const logOfBase = log10(logBaseY);
        for (let i = 0; i < array.length; i++) {
          array[i] = log10(abs(array[i])) / logOfBase;
        }
      }
      const xAxis = createFromToArray({
        from: 0,
        to: array.length - 1,
        length: array.length
      });
      return {
        x: xAxis,
        y: array
      };
    }

    /**
     * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
     * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
     * This assumes a gaussian distribution of the noise
     * @param array
     * @returns noise level corresponding to one standard deviation
     */
    function xNoiseStandardDeviation(array) {
      const {
        mad,
        median
      } = xMedianAbsoluteDeviation(array);
      return {
        sd: mad / 0.6744897501960817,
        mad,
        median
      };
    }

    /**
     * Calculate the sum of the values
     *
     * @param array - Object that contains property x (an ordered increasing array) and y (an array).
     * @param options - Options.
     * @returns XSum value on the specified range.
     */
    function xSum(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let sumValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        sumValue += array[i];
      }
      return sumValue;
    }

    /**
     * Divides the data with either the sum, the absolute sum or the maximum of the data
     * @param input - Array containing values
     * @param options - options
     * @returns - normalized data
     */
    function xNormed(input, options = {}) {
      const {
        algorithm = 'absolute',
        value = 1
      } = options;
      xCheck(input);
      const output = getOutputArray(options.output, input.length);
      if (input.length === 0) {
        throw new Error('input must not be empty');
      }
      switch (algorithm) {
        case 'absolute':
          {
            const absoluteSumValue = absoluteSum(input) / value;
            if (absoluteSumValue === 0) {
              throw new Error('trying to divide by 0');
            }
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] / absoluteSumValue;
            }
            return output;
          }
        case 'max':
          {
            const currentMaxValue = xMaxValue(input);
            if (currentMaxValue === 0) {
              throw new Error('trying to divide by 0');
            }
            const factor = value / currentMaxValue;
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] * factor;
            }
            return output;
          }
        case 'sum':
          {
            const sumFactor = xSum(input) / value;
            if (sumFactor === 0) {
              throw new Error('trying to divide by 0');
            }
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] / sumFactor;
            }
            return output;
          }
        default:
          throw new Error(`unknown algorithm: ${String(algorithm)}`);
      }
    }
    function absoluteSum(input) {
      let sumValue = 0;
      for (let i = 0; i < input.length; i++) {
        sumValue += Math.abs(input[i]);
      }
      return sumValue;
    }

    /**
     * This function pads an array
     *s
     * @param array - the array that will be padded
     * @param options - options
     */
    function xPadding(array, options = {}) {
      const {
        size = 0,
        value = 0,
        algorithm
      } = options;
      xCheck(array);
      if (!algorithm) {
        return xEnsureFloat64(array);
      }
      const result = new Float64Array(array.length + size * 2);
      for (let i = 0; i < array.length; i++) {
        result[i + size] = array[i];
      }
      const fromEnd = size + array.length;
      const toEnd = 2 * size + array.length;
      switch (algorithm) {
        case 'value':
          for (let i = 0; i < size; i++) {
            result[i] = value;
          }
          for (let i = fromEnd; i < toEnd; i++) {
            result[i] = value;
          }
          break;
        case 'duplicate':
          for (let i = 0; i < size; i++) {
            result[i] = array[0];
          }
          for (let i = fromEnd; i < toEnd; i++) {
            result[i] = array[array.length - 1];
          }
          break;
        case 'circular':
          for (let i = 0; i < size; i++) {
            result[i] = array[(array.length - size % array.length + i) % array.length];
          }
          for (let i = 0; i < size; i++) {
            result[i + fromEnd] = array[i % array.length];
          }
          break;
        default:
          throw new Error(`unknown algorithm ${String(algorithm)}`);
      }
      return result;
    }

    /** Finds the variance of the data
     *
     * @param values - the values of the array
     * @param options - options
     * @returns variance
     */
    function xVariance(values, options = {}) {
      if (!isAnyArray$1(values)) {
        throw new TypeError('input must be an array');
      }
      const {
        unbiased = true,
        mean = xMean(values)
      } = options;
      let sqrError = 0;
      for (let i = 0; i < values.length; i++) {
        const x = values[i] - mean;
        sqrError += x * x;
      }
      if (unbiased) {
        return sqrError / (values.length - 1);
      } else {
        return sqrError / values.length;
      }
    }

    /** Finds the standard deviation for the data at hand
     *
     * @param values - values in the data
     * @param options - options
     * @returns standard deviation
     */
    function xStandardDeviation(values, options = {}) {
      return Math.sqrt(xVariance(values, options));
    }

    /**
     * Pareto scaling, which uses the square root of standard deviation as the scaling factor, circumvents the amplification of noise by retaining a small portion of magnitude information.
     * Noda, I. (2008). Scaling techniques to enhance two-dimensional correlation spectra. Journal of Molecular Structure, 883, 216-227.
     * DOI: 10.1016/j.molstruc.2007.12.026
     *
     * @param array - array of number
     */
    function xParetoNormalization(array) {
      xCheck(array);
      const result = new Float64Array(array.length);
      const sqrtSD = Math.sqrt(xStandardDeviation(array));
      for (let i = 0; i < array.length; i++) {
        result[i] = array[i] / sqrtSD;
      }
      return result;
    }

    /**
     * Function used to rescale data
     *
     * @param input - input for the rescale
     * @param options - options
     * @returns rescaled data
     */
    function xRescale(input, options = {}) {
      xCheck(input);
      const output = getOutputArray(options.output, input.length);
      const currentMin = xMinValue(input);
      const currentMax = xMaxValue(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      const {
        min = 0,
        max = 1
      } = options;
      if (min >= max) {
        throw new RangeError('min option must be smaller than max option');
      }
      const factor = (max - min) / (currentMax - currentMin);
      for (let i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + min;
      }
      return output;
    }

    /**
     * This function calculates a rolling average
     *
     * @param array - array
     * @param fct - callback function that from an array returns a value
     * @param options - options
     */
    function xRolling(array, fct, options = {}) {
      xCheck(array);
      if (typeof fct !== 'function') {
        throw new TypeError('fct must be a function');
      }
      const {
        window = 5,
        padding = {}
      } = options;
      const {
        size = window - 1,
        algorithm,
        value
      } = padding;
      const padded = xPadding(array, {
        size,
        algorithm,
        value
      }); // ensure we get a copy and it is float64
      const newArray = [];
      for (let i = 0; i < padded.length - window + 1; i++) {
        // we will send a view to the original buffer
        newArray.push(fct(padded.subarray(i, i + window)));
      }
      return newArray;
    }

    /**
     * This function calculates a rolling average
     *
     * @param array - array
     * @param options - option
     */
    function xRollingAverage(array, options = {}) {
      return xRolling(array, xMean, options);
    }

    /**
     * This function calculates a rolling average
     *
     * @param array - array
     * @param options - options
     */
    function xRollingMedian(array, options = {}) {
      return xRolling(array, xMedian, options);
    }

    /**
     * This function xSubtract the first array by the second array or a constant value from each element of the first array
     *
     * @param array1 - the array that will be rotated
     * @param array2 - second array or number
     * @returns array after subtraction
     */
    function xSubtract(array1, array2) {
      let isConstant = false;
      let constant = 0;
      if (isAnyArray$1(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = Number(array2);
      }
      const array3 = new Float64Array(array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] - constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] - array2[i];
        }
      }
      return array3;
    }

    /**
     * Verify that `data` is an object of x,y arrays.
     * Throws an error if it's not.
     *
     * @param data
     * @param options
     */
    function xyCheck(data, options = {}) {
      const {
        minLength
      } = options;
      if (data === null || typeof data !== 'object' ||
      // @ts-expect-error Typechecking
      !isAnyArray$1(data.x) ||
      // @ts-expect-error Typechecking
      !isAnyArray$1(data.y)) {
        throw new Error('data must be an object of x and y arrays');
      }
      // @ts-expect-error Typechecking
      if (data.x.length !== data.y.length) {
        throw new Error('the x and y arrays must have the same length');
      }
      // @ts-expect-error Typechecking
      if (minLength && data.x.length < minLength) {
        throw new Error(`data.x must have a length of at least ${minLength}`);
      }
    }

    /**
     * Finds the max y value in a range and return a {x,y} point
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array)
     * @param options Options
     */
    function xyMaxYPoint(data, options = {}) {
      xyCheck(data, {
        minLength: 1
      });
      const {
        x,
        y
      } = data;
      if (x.length === 1) {
        return {
          x: x[0],
          y: y[0],
          index: 0
        };
      }
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(x, options);
      let current = {
        x: x[fromIndex],
        y: y[fromIndex],
        index: fromIndex
      };
      for (let i = fromIndex; i <= toIndex; i++) {
        if (y[i] > current.y) current = {
          x: x[i],
          y: y[i],
          index: i
        };
      }
      return current;
    }

    /**
     * Filters x,y values to allow strictly growing values in x-axis.
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array).
     */
    function xyEnsureGrowingX(data) {
      xyCheck(data);
      if (xIsMonotonic(data.x) === 1) return data;
      const x = Array.from(data.x);
      const y = Array.from(data.y);
      let prevX = Number.NEGATIVE_INFINITY;
      let currentIndex = 0;
      for (let index = 0; index < x.length; index++) {
        if (prevX < x[index]) {
          if (currentIndex < index) {
            x[currentIndex] = x[index];
            y[currentIndex] = y[index];
          }
          currentIndex++;
          prevX = x[index];
        }
      }
      x.length = currentIndex;
      y.length = currentIndex;
      return {
        x,
        y
      };
    }

    /* eslint-disable max-lines-per-function */
    /**
     * Normalize an array of zones:
     * - ensure than from < to
     * - merge overlapping zones
     * - deal with exclusions zones
     * - if no zones is specified add one between -Infinity and +Infinity
     * @param zones - array of zones
     * @param options - options
     * @returns array of zones
     */
    function zonesNormalize(zones = [], options = {}) {
      const {
        exclusions = []
      } = options;
      let {
        from = Number.NEGATIVE_INFINITY,
        to = Number.POSITIVE_INFINITY
      } = options;
      if (from > to) [from, to] = [to, from];
      zones = zones.map(zone => zone.from > zone.to ? {
        from: zone.to,
        to: zone.from
      } : {
        ...zone
      }).sort((a, b) => {
        if (a.from !== b.from) return a.from - b.from;
        return a.to - b.to;
      });
      if (zones.length === 0) {
        zones.push({
          from,
          to
        });
      }
      for (const zone of zones) {
        if (from > zone.from) zone.from = from;
        if (to < zone.to) zone.to = to;
      }
      zones = zones.filter(zone => zone.from <= zone.to);
      if (zones.length === 0) return [];
      let currentZone = zones[0];
      const beforeExclusionsZones = [currentZone];
      for (let i = 1; i < zones.length; i++) {
        const zone = zones[i];
        if (zone.from <= currentZone.to) {
          if (currentZone.to < zone.to) {
            currentZone.to = zone.to;
          }
        } else {
          currentZone = zone;
          beforeExclusionsZones.push(currentZone);
        }
      }
      if (exclusions.length === 0) return beforeExclusionsZones;
      const normalizedExclusions = zonesNormalize(exclusions);
      let currentExclusionIndex = 0;
      const results = [];
      for (let zoneIndex = 0; zoneIndex < beforeExclusionsZones.length; zoneIndex++) {
        const zone = beforeExclusionsZones[zoneIndex];
        if (currentExclusionIndex === normalizedExclusions.length) {
          // we analysed all the exclusion zones
          results.push(zone);
          continue;
        }
        while (currentExclusionIndex < normalizedExclusions.length && normalizedExclusions[currentExclusionIndex].to <= zone.from) {
          currentExclusionIndex++;
        }
        if (currentExclusionIndex === normalizedExclusions.length) {
          // we analysed all the exclusion zones
          results.push(zone);
          continue;
        }
        if (zone.to < normalizedExclusions[currentExclusionIndex].from) {
          // no problems, not yet in exclusion
          results.push(zone);
          continue;
        }
        if (normalizedExclusions[currentExclusionIndex].to >= zone.to) {
          // could be totally excluded
          if (normalizedExclusions[currentExclusionIndex].from <= zone.from) {
            continue;
          }
          results.push({
            from: normalizedExclusions[currentExclusionIndex].to,
            to: zone.to
          });
        }
        // we cut in the middle, we need to create more zones, annoying !
        if (normalizedExclusions[currentExclusionIndex].from > zone.from) {
          results.push({
            from: zone.from,
            to: normalizedExclusions[currentExclusionIndex].from
          });
        }
        zone.from = normalizedExclusions[currentExclusionIndex].to;
        zoneIndex--;
      }
      return results;
    }

    /**
     * Add the number of points per zone to reach a specified total
     *
     * @param zones - array of zones
     * @param numberOfPoints - total number of points to distribute between zones
     * @param options - options
     * @returns array of zones with points
     */
    function zonesWithPoints(zones = [],
    /**
     * total number of points to distribute between zones
     * @default 10
     */
    numberOfPoints = 10, options = {}) {
      if (zones.length === 0) return [];
      const normalizedZones = zonesNormalize(zones, options);
      const zonesWithNumberOfPoints = [];
      const totalSize = normalizedZones.reduce((previous, current) => {
        return previous + (current.to - current.from);
      }, 0);
      const unitsPerPoint = totalSize / numberOfPoints;
      let currentTotal = 0;
      for (let i = 0; i < normalizedZones.length - 1; i++) {
        const tempZone = normalizedZones[i];
        const tempZoneNumberOfPoints = Math.min(Math.round((tempZone.to - tempZone.from) / unitsPerPoint), numberOfPoints - currentTotal);
        zonesWithNumberOfPoints.push({
          ...tempZone,
          numberOfPoints: tempZoneNumberOfPoints
        });
        currentTotal += tempZoneNumberOfPoints;
      }
      zonesWithNumberOfPoints.push({
        ...normalizedZones[normalizedZones.length - 1],
        numberOfPoints: numberOfPoints - currentTotal
      });
      return zonesWithNumberOfPoints;
    }

    /* eslint-disable max-lines-per-function */
    /**
     * Function that retrieves the getEquallySpacedData with the variant "slot".
     *
     * @param x
     * @param y
     * @param from
     * @param to
     * @param numberOfPoints
     * @return Array of y's equally spaced with the variant "slot"
     */
    function equallySpacedSlot(/** x coordinates */
    x, /** y coordinates */
    y, /** from value */
    from, /** to value */
    to, /** number of points */
    numberOfPoints) {
      const xLength = x.length;
      const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
      const halfStep = step / 2;
      const lastStep = x[x.length - 1] - x[x.length - 2];
      const start = from - halfStep;
      // Changed Array to Float64Array
      const output = new Float64Array(numberOfPoints);
      // Init main variables
      let min = start;
      let max = start + step;
      let previousX = -Number.MAX_VALUE;
      let previousY = 0;
      let nextX = x[0];
      let nextY = y[0];
      let frontOutsideSpectra = 0;
      let backOutsideSpectra = true;
      let currentValue = 0;
      // for slot algorithm
      let currentPoints = 0;
      let i = 1; // index of input
      let j = 0; // index of output
      main: while (true) {
        if (previousX >= nextX) throw new Error('x must be a growing series');
        while (previousX - max > 0) {
          // no overlap with original point, just consume current value
          if (backOutsideSpectra) {
            currentPoints++;
            backOutsideSpectra = false;
          }
          output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
          j++;
          if (j === numberOfPoints) {
            break main;
          }
          min = max;
          max += step;
          currentValue = 0;
          currentPoints = 0;
        }
        if (previousX > min) {
          currentValue += previousY;
          currentPoints++;
        }
        if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
          currentPoints--;
        }
        previousX = nextX;
        previousY = nextY;
        if (i < xLength) {
          nextX = x[i];
          nextY = y[i];
          i++;
        } else {
          nextX += lastStep;
          nextY = 0;
          frontOutsideSpectra++;
        }
      }
      return output;
    }

    /**
     * Function that calculates the integral of the line between two
     * x-coordinates, given the slope and intercept of the line.
     * @param x0
     * @param x1
     * @param slope
     * @param intercept
     * @return integral value.
     */
    function integral(/** first coordinate of point */
    x0, /** second coordinate of point */
    x1, /** slope of the line */
    slope, /** intercept of the line on the y axis */
    intercept) {
      return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
    }

    /**
     * Function that retrieves the getEquallySpacedData with the variant "smooth".
     *
     * @param x
     * @param y
     * @param from
     * @param to
     * @param numberOfPoints
     * @return - Array of y's equally spaced with the variant "smooth"
     */
    function equallySpacedSmooth(/** x coordinates */
    x, /** y coordinates */
    y, /** from value */
    from, /** to value */
    to, /** number of points */
    numberOfPoints) {
      const xLength = x.length;
      const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
      const halfStep = step / 2;
      // Changed Array to Float64Array
      const output = new Float64Array(numberOfPoints);
      const initialOriginalStep = x[1] - x[0];
      const lastOriginalStep = x[xLength - 1] - x[xLength - 2];
      // Init main variables
      let min = from - halfStep;
      let max = from + halfStep;
      let previousX = Number.MIN_SAFE_INTEGER;
      let previousY = 0;
      let nextX = x[0] - initialOriginalStep;
      let nextY = 0;
      let currentValue = 0;
      let slope = 0;
      let intercept = 0;
      let sumAtMin = 0;
      let sumAtMax = 0;
      let i = 0; // index of input
      let j = 0; // index of output
      let add = 0;
      main: while (true) {
        if (previousX >= nextX) throw new Error('x must be a growing series');
        if (previousX <= min && min <= nextX) {
          add = integral(0, min - previousX, slope, previousY);
          sumAtMin = currentValue + add;
        }
        while (nextX - max >= 0) {
          // no overlap with original point, just consume current value
          add = integral(0, max - previousX, slope, previousY);
          sumAtMax = currentValue + add;
          output[j++] = (sumAtMax - sumAtMin) / step;
          if (j === numberOfPoints) {
            break main;
          }
          min = max;
          max += step;
          sumAtMin = sumAtMax;
        }
        currentValue += integral(previousX, nextX, slope, intercept);
        previousX = nextX;
        previousY = nextY;
        if (i < xLength) {
          nextX = x[i];
          nextY = y[i];
          i++;
        } else if (i === xLength) {
          nextX += lastOriginalStep;
          nextY = 0;
        }
        slope = getSlope(previousX, previousY, nextX, nextY);
        intercept = -slope * previousX + previousY;
      }
      return output;
    }
    function getSlope(x0, y0, x1, y1) {
      return (y1 - y0) / (x1 - x0);
    }

    /**
     * Function that returns a Number array of equally spaced numberOfPoints
     * containing a representation of intensities of the spectra arguments x
     * and y.
     *
     * The options parameter contains an object in the following form:
     * from: starting point
     * to: last point
     * numberOfPoints: number of points between from and to
     * variant: "slot" or "smooth" - smooth is the default option
     *
     * The slot variant consist that each point in an array is calculated
     * averaging the existing points between the slot that belongs to the current
     * value. The smooth variant is the same but takes the integral of the range
     * of the slot and divide by the step size between two points in an array.
     *
     * If exclusions zone are present, zones are ignored !
     *
     * @param data - object containing 2 properties x and y
     * @param options - options
     * @return new object with x / y array with the equally spaced data.
     */
    function xyEquallySpaced(data, options = {}) {
      const {
        x,
        y
      } = data;
      const xLength = x.length;
      const {
        from = x[0],
        to = x[xLength - 1],
        variant = 'smooth',
        numberOfPoints = 100,
        exclusions = [],
        zones = [{
          from,
          to
        }]
      } = options;
      if (from > to) {
        throw new RangeError('from should be larger than to');
      }
      xyCheck(data);
      if (numberOfPoints < 2) {
        throw new RangeError("'numberOfPoints' option must be greater than 1");
      }
      const normalizedZones = zonesNormalize(zones, {
        from,
        to,
        exclusions
      });
      const zonesWithPointsRes = zonesWithPoints(normalizedZones, numberOfPoints, {
        from,
        to
      }).filter(zone => zone.numberOfPoints);
      let xResult = [];
      let yResult = [];
      for (const zone of zonesWithPointsRes) {
        if (!zone.numberOfPoints) {
          zone.numberOfPoints = 0;
        }
        const zoneResult = processZone(Array.from(x), Array.from(y), zone.from, zone.to, zone.numberOfPoints, variant);
        xResult = xResult.concat(zoneResult.x);
        yResult = yResult.concat(zoneResult.y);
      }
      return {
        x: xResult,
        y: yResult
      };
    }
    function processZone(x, y, from, to, numberOfPoints, variant) {
      if (numberOfPoints < 1) {
        throw new RangeError('the number of points must be at least 1');
      }
      const output = variant === 'slot' ? Array.from(equallySpacedSlot(x, y, from, to, numberOfPoints)) : Array.from(equallySpacedSmooth(x, y, from, to, numberOfPoints));
      return {
        x: Array.from(createFromToArray({
          from,
          to,
          length: numberOfPoints
        })),
        y: output
      };
    }

    /** Filter an array x/y based on various criteria x points are expected to be sorted
     *
     * @param data - object containing 2 properties x and y
     * @param options - options
     * @return filtered array
     */
    function xyFilterX(data, options = {}) {
      const {
        x,
        y
      } = data;
      const {
        from = x[0],
        to = x[x.length - 1],
        zones = [{
          from,
          to
        }],
        exclusions = []
      } = options;
      const normalizedZones = zonesNormalize(zones, {
        from,
        to,
        exclusions
      });
      let currentZoneIndex = 0;
      const newX = [];
      const newY = [];
      let position = 0;
      while (position < x.length) {
        if (x[position] <= normalizedZones[currentZoneIndex].to && x[position] >= normalizedZones[currentZoneIndex].from) {
          newX.push(x[position]);
          newY.push(y[position]);
        } else if (x[position] > normalizedZones[currentZoneIndex].to) {
          currentZoneIndex++;
          if (!normalizedZones[currentZoneIndex]) break;
        }
        position++;
      }
      return {
        x: newX,
        y: newY
      };
    }

    /**
     * Calculate integration
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array)
     * @param options - Options
     * @returns - xyIntegration value on the specified range
     */
    function xyIntegration(data, options = {}) {
      xyCheck(data, {
        minLength: 1
      });
      const {
        x,
        y
      } = data;
      if (x.length === 1) return 0;
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(x, options);
      let currentxyIntegration = 0;
      for (let i = fromIndex; i < toIndex; i++) {
        currentxyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
      }
      return currentxyIntegration;
    }

    /**
     * Finds the min y value in a range and return a {x,y} point
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array)
     * @param options - Options
     */
    function xyMinYPoint(data, options = {}) {
      xyCheck(data, {
        minLength: 1
      });
      const {
        x,
        y
      } = data;
      if (x.length === 1) return {
        x: x[0],
        y: y[0],
        index: 0
      };
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(x, options);
      let current = {
        x: x[fromIndex],
        y: y[fromIndex],
        index: fromIndex
      };
      for (let i = fromIndex; i <= toIndex; i++) {
        if (y[i] < current.y) current = {
          x: x[i],
          y: y[i],
          index: i
        };
      }
      return current;
    }

    /**
     * This function performs a quick sort of the x array while transforming the y array to preserve the coordinates.
     *
     * @param data - Object that contains property x (Array) and y (Array)
     */
    function xySortX(data) {
      const {
        x,
        y
      } = data;
      if (xIsMonotonic(x) && x.length > 1) {
        if (x[0] < x[1]) {
          return {
            x: Float64Array.from(x),
            y: Float64Array.from(y)
          };
        } else {
          return {
            x: Float64Array.from(x).reverse(),
            y: Float64Array.from(y).reverse()
          };
        }
      }
      const xyObject = Array.from(x, (val, index) => ({
        x: val,
        y: y[index]
      })).sort((a, b) => a.x - b.x);
      const response = {
        x: new Float64Array(x.length),
        y: new Float64Array(y.length)
      };
      for (let i = 0; i < x.length; i++) {
        response.x[i] = xyObject[i].x;
        response.y[i] = xyObject[i].y;
      }
      return response;
    }

    /**
     * Ensure x values are unique
     *
     * @param data - Object that contains property x (Array) and y (Array)
     * @param options - Object containing a property algorithm (can be 'sum' or 'average', the latter being the default value), and a property isSorted (boolean indicating if the x-array is sorted).
     */
    function xyUniqueX(data, options = {}) {
      xyCheck(data);
      const {
        algorithm = 'average',
        isSorted = true
      } = options;
      if (!isSorted) {
        data = xySortX(data);
      }
      switch (algorithm) {
        case 'average':
          return average$1(data);
        case 'sum':
          return sum(data);
        default:
          throw new Error(`unknown algorithm: ${String(algorithm)}`);
      }
    }
    /**
     * Average.
     *
     * @param data - Input.
     * @returns Result.
     */
    function average$1(data) {
      const x = [];
      const y = [];
      let cumulativeY = data.y[0];
      let divider = 1;
      for (let i = 1; i < data.x.length; i++) {
        if (!(data.x[i] === data.x[i - 1])) {
          x.push(data.x[i - 1]);
          y.push(cumulativeY / divider);
          cumulativeY = 0;
          divider = 0;
        }
        cumulativeY += data.y[i];
        divider++;
      }
      x.push(data.x[data.x.length - 1]);
      y.push(cumulativeY / divider);
      return {
        x,
        y
      };
    }
    /**
     * Sum.
     *
     * @param data - Input.
     * @returns Result.
     */
    function sum(data) {
      const x = [];
      const y = [];
      let cumulativeY = data.y[0];
      for (let i = 1; i < data.x.length; i++) {
        if (!(data.x[i] === data.x[i - 1])) {
          x.push(data.x[i - 1]);
          y.push(cumulativeY);
          cumulativeY = 0;
        }
        cumulativeY += data.y[i];
      }
      x.push(data.x[data.x.length - 1]);
      y.push(cumulativeY);
      return {
        x,
        y
      };
    }

    /**
     * Create a new matrix based on the size of the current one or by using specific dimensions.
     */
    function matrixCreateEmpty(options) {
      const {
        matrix,
        nbRows = matrix?.length || 1,
        nbColumns = matrix?.[0].length || 1,
        ArrayConstructor = Float64Array
      } = options;
      const newMatrix = [];
      for (let row = 0; row < nbRows; row++) {
        newMatrix.push(createNumberArray(ArrayConstructor, nbColumns));
      }
      return newMatrix;
    }

    /**
     * Center mean of matrix columns.
     *
     * @param matrix - matrix [rows][cols]
     */
    function matrixCenterZMean(matrix) {
      const nbColumns = matrix[0].length;
      const nbRows = matrix.length;
      const newMatrix = matrixCreateEmpty({
        nbColumns,
        nbRows
      });
      for (let column = 0; column < nbColumns; column++) {
        let mean = 0;
        for (let row = 0; row < nbRows; row++) {
          mean += matrix[row][column];
        }
        mean /= nbRows;
        for (let row = 0; row < nbRows; row++) {
          newMatrix[row][column] = matrix[row][column] - mean;
        }
      }
      return newMatrix;
    }

    function matrixCheck(data) {
      if (data.length === 0 || data[0].length === 0) {
        throw new RangeError('matrix must contain data');
      }
      const firstLength = data[0].length;
      for (let i = 1; i < data.length; i++) {
        if (data[i].length !== firstLength) {
          throw new RangeError('all rows must has the same length');
        }
      }
    }

    /**
     * Get min and max Z.
     *
     * @param matrix - matrix [rows][cols].
     */
    function matrixMinMaxZ(matrix) {
      matrixCheck(matrix);
      const nbRows = matrix.length;
      const nbColumns = matrix[0].length;
      let min = matrix[0][0];
      let max = matrix[0][0];
      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          if (matrix[row][column] < min) min = matrix[row][column];
          if (matrix[row][column] > max) max = matrix[row][column];
        }
      }
      return {
        min,
        max
      };
    }

    /**
     * Performs a Probabilistic quotient normalization (PQN) over the dataset to account dilution based in median spectrum.
     * Dieterle, F., Ross, A., Schlotterbeck, G., & Senn, H. (2006). Probabilistic quotient normalization as robust method to account for dilution of complex biological mixtures. Application in 1H NMR metabonomics. Analytical chemistry, 78(13), 4281-4290.
     * DOI: 10.1021/ac051632c
     *
     * @param matrix - matrix [rows][cols].
     * @param options - options
     * @returns - {data: Normalized dataset, medianOfQuotients: The median of quotients of each variables}
     */
    function matrixPQN(matrix, options = {}) {
      const {
        max = 100
      } = options;
      const matrixB = new Matrix(matrix);
      for (let i = 0; i < matrixB.rows; i++) {
        const normalizationFactor = matrixB.getRowVector(i).norm('frobenius') / max;
        const row = matrixB.getRowVector(i).div(normalizationFactor);
        matrixB.setRow(i, row);
      }
      const referenceSpectrum = [];
      for (let i = 0; i < matrixB.columns; i++) {
        const currentVariable = matrixB.getColumn(i);
        referenceSpectrum.push(xMedian(currentVariable));
      }
      const medianOfQuotients = [];
      for (let i = 0; i < matrixB.columns; i++) {
        const quotients = matrixB.getColumnVector(i).div(referenceSpectrum[i]);
        medianOfQuotients.push(xMedian(quotients.getColumn(0)));
      }
      for (let i = 0; i < matrixB.rows; i++) {
        matrixB.mulRow(i, 1 / medianOfQuotients[i]);
      }
      return {
        data: matrixB.to2DArray(),
        medianOfQuotients
      };
    }

    /**
     * Rescale a matrix between min and max values.
     *
     * @param matrix - matrix [rows][cols].
     * @param options - Options.
     */
    function matrixZRescale(matrix, options = {}) {
      const {
        min = 0,
        max = 1,
        ArrayConstructor
      } = options;
      const nbColumns = matrix[0].length;
      const nbRows = matrix.length;
      const newMatrix = matrixCreateEmpty({
        nbColumns,
        nbRows,
        ArrayConstructor
      });
      const {
        min: currentMin,
        max: currentMax
      } = matrixMinMaxZ(matrix);
      const factor = (max - min) / (currentMax - currentMin);
      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          newMatrix[row][column] = (matrix[row][column] - currentMin) * factor + min;
        }
      }
      return newMatrix;
    }

    var limit = (x, low = 0, high = 1) => {
      return min$2(max$2(low, x), high);
    };

    var clip_rgb = rgb => {
      rgb._clipped = false;
      rgb._unclipped = rgb.slice(0);
      for (let i = 0; i <= 3; i++) {
        if (i < 3) {
          if (rgb[i] < 0 || rgb[i] > 255) rgb._clipped = true;
          rgb[i] = limit(rgb[i], 0, 255);
        } else if (i === 3) {
          rgb[i] = limit(rgb[i], 0, 1);
        }
      }
      return rgb;
    };

    // ported from jQuery's $.type
    const classToType = {};
    for (let name of ['Boolean', 'Number', 'String', 'Function', 'Array', 'Date', 'RegExp', 'Undefined', 'Null']) {
      classToType[`[object ${name}]`] = name.toLowerCase();
    }
    function type (obj) {
      return classToType[Object.prototype.toString.call(obj)] || 'object';
    }

    var last = args => {
      if (args.length < 2) return null;
      const l = args.length - 1;
      if (type(args[l]) == 'string') return args[l].toLowerCase();
      return null;
    };

    const {
      PI: PI$2,
      min: min$2,
      max: max$2
    } = Math;
    const TWOPI = PI$2 * 2;

    var input = {
      format: {},
      autodetect: []
    };

    class Color {
      constructor(...args) {
        const me = this;
        if (type(args[0]) === 'object' && args[0].constructor && args[0].constructor === this.constructor) {
          // the argument is already a Color instance
          return args[0];
        }
        // last argument could be the mode
        let mode = last(args);
        let autodetect = false;
        if (!mode) {
          autodetect = true;
          if (!input.sorted) {
            input.autodetect = input.autodetect.sort((a, b) => b.p - a.p);
            input.sorted = true;
          }
          // auto-detect format
          for (let chk of input.autodetect) {
            mode = chk.test(...args);
            if (mode) break;
          }
        }
        if (input.format[mode]) {
          const rgb = input.format[mode].apply(null, autodetect ? args : args.slice(0, -1));
          me._rgb = clip_rgb(rgb);
        } else {
          throw new Error('unknown format: ' + args);
        }
        // add alpha channel
        if (me._rgb.length === 3) me._rgb.push(1);
      }
      toString() {
        if (type(this.hex) == 'function') return this.hex();
        return `[${this._rgb.join(',')}]`;
      }
    }
    var Color$1 = Color;

    // this gets updated automatically
    const version = '2.6.0';

    const chroma = (...args) => {
      return new chroma.Color(...args);
    };
    chroma.Color = Color$1;
    chroma.version = version;
    var chroma$1 = chroma;

    /**
    	X11 color names

    	http://www.w3.org/TR/css3-color/#svg-color
    */

    const w3cx11 = {
      aliceblue: '#f0f8ff',
      antiquewhite: '#faebd7',
      aqua: '#00ffff',
      aquamarine: '#7fffd4',
      azure: '#f0ffff',
      beige: '#f5f5dc',
      bisque: '#ffe4c4',
      black: '#000000',
      blanchedalmond: '#ffebcd',
      blue: '#0000ff',
      blueviolet: '#8a2be2',
      brown: '#a52a2a',
      burlywood: '#deb887',
      cadetblue: '#5f9ea0',
      chartreuse: '#7fff00',
      chocolate: '#d2691e',
      coral: '#ff7f50',
      cornflowerblue: '#6495ed',
      cornsilk: '#fff8dc',
      crimson: '#dc143c',
      cyan: '#00ffff',
      darkblue: '#00008b',
      darkcyan: '#008b8b',
      darkgoldenrod: '#b8860b',
      darkgray: '#a9a9a9',
      darkgreen: '#006400',
      darkgrey: '#a9a9a9',
      darkkhaki: '#bdb76b',
      darkmagenta: '#8b008b',
      darkolivegreen: '#556b2f',
      darkorange: '#ff8c00',
      darkorchid: '#9932cc',
      darkred: '#8b0000',
      darksalmon: '#e9967a',
      darkseagreen: '#8fbc8f',
      darkslateblue: '#483d8b',
      darkslategray: '#2f4f4f',
      darkslategrey: '#2f4f4f',
      darkturquoise: '#00ced1',
      darkviolet: '#9400d3',
      deeppink: '#ff1493',
      deepskyblue: '#00bfff',
      dimgray: '#696969',
      dimgrey: '#696969',
      dodgerblue: '#1e90ff',
      firebrick: '#b22222',
      floralwhite: '#fffaf0',
      forestgreen: '#228b22',
      fuchsia: '#ff00ff',
      gainsboro: '#dcdcdc',
      ghostwhite: '#f8f8ff',
      gold: '#ffd700',
      goldenrod: '#daa520',
      gray: '#808080',
      green: '#008000',
      greenyellow: '#adff2f',
      grey: '#808080',
      honeydew: '#f0fff0',
      hotpink: '#ff69b4',
      indianred: '#cd5c5c',
      indigo: '#4b0082',
      ivory: '#fffff0',
      khaki: '#f0e68c',
      laserlemon: '#ffff54',
      lavender: '#e6e6fa',
      lavenderblush: '#fff0f5',
      lawngreen: '#7cfc00',
      lemonchiffon: '#fffacd',
      lightblue: '#add8e6',
      lightcoral: '#f08080',
      lightcyan: '#e0ffff',
      lightgoldenrod: '#fafad2',
      lightgoldenrodyellow: '#fafad2',
      lightgray: '#d3d3d3',
      lightgreen: '#90ee90',
      lightgrey: '#d3d3d3',
      lightpink: '#ffb6c1',
      lightsalmon: '#ffa07a',
      lightseagreen: '#20b2aa',
      lightskyblue: '#87cefa',
      lightslategray: '#778899',
      lightslategrey: '#778899',
      lightsteelblue: '#b0c4de',
      lightyellow: '#ffffe0',
      lime: '#00ff00',
      limegreen: '#32cd32',
      linen: '#faf0e6',
      magenta: '#ff00ff',
      maroon: '#800000',
      maroon2: '#7f0000',
      maroon3: '#b03060',
      mediumaquamarine: '#66cdaa',
      mediumblue: '#0000cd',
      mediumorchid: '#ba55d3',
      mediumpurple: '#9370db',
      mediumseagreen: '#3cb371',
      mediumslateblue: '#7b68ee',
      mediumspringgreen: '#00fa9a',
      mediumturquoise: '#48d1cc',
      mediumvioletred: '#c71585',
      midnightblue: '#191970',
      mintcream: '#f5fffa',
      mistyrose: '#ffe4e1',
      moccasin: '#ffe4b5',
      navajowhite: '#ffdead',
      navy: '#000080',
      oldlace: '#fdf5e6',
      olive: '#808000',
      olivedrab: '#6b8e23',
      orange: '#ffa500',
      orangered: '#ff4500',
      orchid: '#da70d6',
      palegoldenrod: '#eee8aa',
      palegreen: '#98fb98',
      paleturquoise: '#afeeee',
      palevioletred: '#db7093',
      papayawhip: '#ffefd5',
      peachpuff: '#ffdab9',
      peru: '#cd853f',
      pink: '#ffc0cb',
      plum: '#dda0dd',
      powderblue: '#b0e0e6',
      purple: '#800080',
      purple2: '#7f007f',
      purple3: '#a020f0',
      rebeccapurple: '#663399',
      red: '#ff0000',
      rosybrown: '#bc8f8f',
      royalblue: '#4169e1',
      saddlebrown: '#8b4513',
      salmon: '#fa8072',
      sandybrown: '#f4a460',
      seagreen: '#2e8b57',
      seashell: '#fff5ee',
      sienna: '#a0522d',
      silver: '#c0c0c0',
      skyblue: '#87ceeb',
      slateblue: '#6a5acd',
      slategray: '#708090',
      slategrey: '#708090',
      snow: '#fffafa',
      springgreen: '#00ff7f',
      steelblue: '#4682b4',
      tan: '#d2b48c',
      teal: '#008080',
      thistle: '#d8bfd8',
      tomato: '#ff6347',
      turquoise: '#40e0d0',
      violet: '#ee82ee',
      wheat: '#f5deb3',
      white: '#ffffff',
      whitesmoke: '#f5f5f5',
      yellow: '#ffff00',
      yellowgreen: '#9acd32'
    };
    var colors = w3cx11;

    var index = {};

    var mix = (col1, col2, f = 0.5, ...rest) => {
      let mode = rest[0] || 'lrgb';
      if (!index[mode] && !rest.length) {
        // fall back to the first supported mode
        mode = Object.keys(index)[0];
      }
      if (!index[mode]) {
        throw new Error(`interpolation mode ${mode} is not defined`);
      }
      if (type(col1) !== 'object') col1 = new Color$1(col1);
      if (type(col2) !== 'object') col2 = new Color$1(col2);
      return index[mode](col1, col2, f).alpha(col1.alpha() + f * (col2.alpha() - col1.alpha()));
    };

    const {
      pow: pow$4,
      sqrt: sqrt$1,
      PI: PI$1,
      cos: cos$2,
      sin: sin$2,
      atan2: atan2$1
    } = Math;
    var average = (colors, mode = 'lrgb', weights = null) => {
      const l = colors.length;
      if (!weights) weights = Array.from(new Array(l)).map(() => 1);
      // normalize weights
      const k = l / weights.reduce(function (a, b) {
        return a + b;
      });
      weights.forEach((w, i) => {
        weights[i] *= k;
      });
      // convert colors to Color objects
      colors = colors.map(c => new Color$1(c));
      if (mode === 'lrgb') {
        return _average_lrgb(colors, weights);
      }
      const first = colors.shift();
      const xyz = first.get(mode);
      const cnt = [];
      let dx = 0;
      let dy = 0;
      // initial color
      for (let i = 0; i < xyz.length; i++) {
        xyz[i] = (xyz[i] || 0) * weights[0];
        cnt.push(isNaN(xyz[i]) ? 0 : weights[0]);
        if (mode.charAt(i) === 'h' && !isNaN(xyz[i])) {
          const A = xyz[i] / 180 * PI$1;
          dx += cos$2(A) * weights[0];
          dy += sin$2(A) * weights[0];
        }
      }
      let alpha = first.alpha() * weights[0];
      colors.forEach((c, ci) => {
        const xyz2 = c.get(mode);
        alpha += c.alpha() * weights[ci + 1];
        for (let i = 0; i < xyz.length; i++) {
          if (!isNaN(xyz2[i])) {
            cnt[i] += weights[ci + 1];
            if (mode.charAt(i) === 'h') {
              const A = xyz2[i] / 180 * PI$1;
              dx += cos$2(A) * weights[ci + 1];
              dy += sin$2(A) * weights[ci + 1];
            } else {
              xyz[i] += xyz2[i] * weights[ci + 1];
            }
          }
        }
      });
      for (let i = 0; i < xyz.length; i++) {
        if (mode.charAt(i) === 'h') {
          let A = atan2$1(dy / cnt[i], dx / cnt[i]) / PI$1 * 180;
          while (A < 0) A += 360;
          while (A >= 360) A -= 360;
          xyz[i] = A;
        } else {
          xyz[i] = xyz[i] / cnt[i];
        }
      }
      alpha /= l;
      return new Color$1(xyz, mode).alpha(alpha > 0.99999 ? 1 : alpha, true);
    };
    const _average_lrgb = (colors, weights) => {
      const l = colors.length;
      const xyz = [0, 0, 0, 0];
      for (let i = 0; i < colors.length; i++) {
        const col = colors[i];
        const f = weights[i] / l;
        const rgb = col._rgb;
        xyz[0] += pow$4(rgb[0], 2) * f;
        xyz[1] += pow$4(rgb[1], 2) * f;
        xyz[2] += pow$4(rgb[2], 2) * f;
        xyz[3] += rgb[3] * f;
      }
      xyz[0] = sqrt$1(xyz[0]);
      xyz[1] = sqrt$1(xyz[1]);
      xyz[2] = sqrt$1(xyz[2]);
      if (xyz[3] > 0.9999999) xyz[3] = 1;
      return new Color$1(clip_rgb(xyz));
    };

    // minimal multi-purpose interface
    const {
      pow: pow$3
    } = Math;
    function scale (colors) {
      // constructor
      let _mode = 'rgb';
      let _nacol = chroma$1('#ccc');
      let _spread = 0;
      // const _fixed = false;
      let _domain = [0, 1];
      let _pos = [];
      let _padding = [0, 0];
      let _classes = false;
      let _colors = [];
      let _out = false;
      let _min = 0;
      let _max = 1;
      let _correctLightness = false;
      let _colorCache = {};
      let _useCache = true;
      let _gamma = 1;

      // private methods

      const setColors = function (colors) {
        colors = colors || ['#fff', '#000'];
        if (colors && type(colors) === 'string' && chroma$1.brewer && chroma$1.brewer[colors.toLowerCase()]) {
          colors = chroma$1.brewer[colors.toLowerCase()];
        }
        if (type(colors) === 'array') {
          // handle single color
          if (colors.length === 1) {
            colors = [colors[0], colors[0]];
          }
          // make a copy of the colors
          colors = colors.slice(0);
          // convert to chroma classes
          for (let c = 0; c < colors.length; c++) {
            colors[c] = chroma$1(colors[c]);
          }
          // auto-fill color position
          _pos.length = 0;
          for (let c = 0; c < colors.length; c++) {
            _pos.push(c / (colors.length - 1));
          }
        }
        resetCache();
        return _colors = colors;
      };
      const getClass = function (value) {
        if (_classes != null) {
          const n = _classes.length - 1;
          let i = 0;
          while (i < n && value >= _classes[i]) {
            i++;
          }
          return i - 1;
        }
        return 0;
      };
      let tMapLightness = t => t;
      let tMapDomain = t => t;

      // const classifyValue = function(value) {
      //     let val = value;
      //     if (_classes.length > 2) {
      //         const n = _classes.length-1;
      //         const i = getClass(value);
      //         const minc = _classes[0] + ((_classes[1]-_classes[0]) * (0 + (_spread * 0.5)));  // center of 1st class
      //         const maxc = _classes[n-1] + ((_classes[n]-_classes[n-1]) * (1 - (_spread * 0.5)));  // center of last class
      //         val = _min + ((((_classes[i] + ((_classes[i+1] - _classes[i]) * 0.5)) - minc) / (maxc-minc)) * (_max - _min));
      //     }
      //     return val;
      // };

      const getColor = function (val, bypassMap) {
        let col, t;
        if (bypassMap == null) {
          bypassMap = false;
        }
        if (isNaN(val) || val === null) {
          return _nacol;
        }
        if (!bypassMap) {
          if (_classes && _classes.length > 2) {
            // find the class
            const c = getClass(val);
            t = c / (_classes.length - 2);
          } else if (_max !== _min) {
            // just interpolate between min/max
            t = (val - _min) / (_max - _min);
          } else {
            t = 1;
          }
        } else {
          t = val;
        }

        // domain map
        t = tMapDomain(t);
        if (!bypassMap) {
          t = tMapLightness(t); // lightness correction
        }
        if (_gamma !== 1) {
          t = pow$3(t, _gamma);
        }
        t = _padding[0] + t * (1 - _padding[0] - _padding[1]);
        t = limit(t, 0, 1);
        const k = Math.floor(t * 10000);
        if (_useCache && _colorCache[k]) {
          col = _colorCache[k];
        } else {
          if (type(_colors) === 'array') {
            //for i in [0.._pos.length-1]
            for (let i = 0; i < _pos.length; i++) {
              const p = _pos[i];
              if (t <= p) {
                col = _colors[i];
                break;
              }
              if (t >= p && i === _pos.length - 1) {
                col = _colors[i];
                break;
              }
              if (t > p && t < _pos[i + 1]) {
                t = (t - p) / (_pos[i + 1] - p);
                col = chroma$1.interpolate(_colors[i], _colors[i + 1], t, _mode);
                break;
              }
            }
          } else if (type(_colors) === 'function') {
            col = _colors(t);
          }
          if (_useCache) {
            _colorCache[k] = col;
          }
        }
        return col;
      };
      var resetCache = () => _colorCache = {};
      setColors(colors);

      // public interface

      const f = function (v) {
        const c = chroma$1(getColor(v));
        if (_out && c[_out]) {
          return c[_out]();
        } else {
          return c;
        }
      };
      f.classes = function (classes) {
        if (classes != null) {
          if (type(classes) === 'array') {
            _classes = classes;
            _domain = [classes[0], classes[classes.length - 1]];
          } else {
            const d = chroma$1.analyze(_domain);
            if (classes === 0) {
              _classes = [d.min, d.max];
            } else {
              _classes = chroma$1.limits(d, 'e', classes);
            }
          }
          return f;
        }
        return _classes;
      };
      f.domain = function (domain) {
        if (!arguments.length) {
          return _domain;
        }
        _min = domain[0];
        _max = domain[domain.length - 1];
        _pos = [];
        const k = _colors.length;
        if (domain.length === k && _min !== _max) {
          // update positions
          for (let d of Array.from(domain)) {
            _pos.push((d - _min) / (_max - _min));
          }
        } else {
          for (let c = 0; c < k; c++) {
            _pos.push(c / (k - 1));
          }
          if (domain.length > 2) {
            // set domain map
            const tOut = domain.map((d, i) => i / (domain.length - 1));
            const tBreaks = domain.map(d => (d - _min) / (_max - _min));
            if (!tBreaks.every((val, i) => tOut[i] === val)) {
              tMapDomain = t => {
                if (t <= 0 || t >= 1) return t;
                let i = 0;
                while (t >= tBreaks[i + 1]) i++;
                const f = (t - tBreaks[i]) / (tBreaks[i + 1] - tBreaks[i]);
                const out = tOut[i] + f * (tOut[i + 1] - tOut[i]);
                return out;
              };
            }
          }
        }
        _domain = [_min, _max];
        return f;
      };
      f.mode = function (_m) {
        if (!arguments.length) {
          return _mode;
        }
        _mode = _m;
        resetCache();
        return f;
      };
      f.range = function (colors, _pos) {
        setColors(colors);
        return f;
      };
      f.out = function (_o) {
        _out = _o;
        return f;
      };
      f.spread = function (val) {
        if (!arguments.length) {
          return _spread;
        }
        _spread = val;
        return f;
      };
      f.correctLightness = function (v) {
        if (v == null) {
          v = true;
        }
        _correctLightness = v;
        resetCache();
        if (_correctLightness) {
          tMapLightness = function (t) {
            const L0 = getColor(0, true).lab()[0];
            const L1 = getColor(1, true).lab()[0];
            const pol = L0 > L1;
            let L_actual = getColor(t, true).lab()[0];
            const L_ideal = L0 + (L1 - L0) * t;
            let L_diff = L_actual - L_ideal;
            let t0 = 0;
            let t1 = 1;
            let max_iter = 20;
            while (Math.abs(L_diff) > 1e-2 && max_iter-- > 0) {
              (function () {
                if (pol) {
                  L_diff *= -1;
                }
                if (L_diff < 0) {
                  t0 = t;
                  t += (t1 - t) * 0.5;
                } else {
                  t1 = t;
                  t += (t0 - t) * 0.5;
                }
                L_actual = getColor(t, true).lab()[0];
                return L_diff = L_actual - L_ideal;
              })();
            }
            return t;
          };
        } else {
          tMapLightness = t => t;
        }
        return f;
      };
      f.padding = function (p) {
        if (p != null) {
          if (type(p) === 'number') {
            p = [p, p];
          }
          _padding = p;
          return f;
        } else {
          return _padding;
        }
      };
      f.colors = function (numColors, out) {
        // If no arguments are given, return the original colors that were provided
        if (arguments.length < 2) {
          out = 'hex';
        }
        let result = [];
        if (arguments.length === 0) {
          result = _colors.slice(0);
        } else if (numColors === 1) {
          result = [f(0.5)];
        } else if (numColors > 1) {
          const dm = _domain[0];
          const dd = _domain[1] - dm;
          result = __range__(0, numColors, false).map(i => f(dm + i / (numColors - 1) * dd));
        } else {
          // returns all colors based on the defined classes
          colors = [];
          let samples = [];
          if (_classes && _classes.length > 2) {
            for (let i = 1, end = _classes.length, asc = 1 <= end; asc ? i < end : i > end; asc ? i++ : i--) {
              samples.push((_classes[i - 1] + _classes[i]) * 0.5);
            }
          } else {
            samples = _domain;
          }
          result = samples.map(v => f(v));
        }
        if (chroma$1[out]) {
          result = result.map(c => c[out]());
        }
        return result;
      };
      f.cache = function (c) {
        if (c != null) {
          _useCache = c;
          return f;
        } else {
          return _useCache;
        }
      };
      f.gamma = function (g) {
        if (g != null) {
          _gamma = g;
          return f;
        } else {
          return _gamma;
        }
      };
      f.nodata = function (d) {
        if (d != null) {
          _nacol = chroma$1(d);
          return f;
        } else {
          return _nacol;
        }
      };
      return f;
    }
    function __range__(left, right, inclusive) {
      let range = [];
      let ascending = left < right;
      let end = !inclusive ? right : ascending ? right + 1 : right - 1;
      for (let i = left; ascending ? i < end : i > end; ascending ? i++ : i--) {
        range.push(i);
      }
      return range;
    }

    //

    // nth row of the pascal triangle
    const binom_row = function (n) {
      let row = [1, 1];
      for (let i = 1; i < n; i++) {
        let newrow = [1];
        for (let j = 1; j <= row.length; j++) {
          newrow[j] = (row[j] || 0) + row[j - 1];
        }
        row = newrow;
      }
      return row;
    };
    const bezier = function (colors) {
      let I, lab0, lab1, lab2;
      colors = colors.map(c => new Color$1(c));
      if (colors.length === 2) {
        // linear interpolation
        [lab0, lab1] = colors.map(c => c.lab());
        I = function (t) {
          const lab = [0, 1, 2].map(i => lab0[i] + t * (lab1[i] - lab0[i]));
          return new Color$1(lab, 'lab');
        };
      } else if (colors.length === 3) {
        // quadratic bezier interpolation
        [lab0, lab1, lab2] = colors.map(c => c.lab());
        I = function (t) {
          const lab = [0, 1, 2].map(i => (1 - t) * (1 - t) * lab0[i] + 2 * (1 - t) * t * lab1[i] + t * t * lab2[i]);
          return new Color$1(lab, 'lab');
        };
      } else if (colors.length === 4) {
        // cubic bezier interpolation
        let lab3;
        [lab0, lab1, lab2, lab3] = colors.map(c => c.lab());
        I = function (t) {
          const lab = [0, 1, 2].map(i => (1 - t) * (1 - t) * (1 - t) * lab0[i] + 3 * (1 - t) * (1 - t) * t * lab1[i] + 3 * (1 - t) * t * t * lab2[i] + t * t * t * lab3[i]);
          return new Color$1(lab, 'lab');
        };
      } else if (colors.length >= 5) {
        // general case (degree n bezier)
        let labs, row, n;
        labs = colors.map(c => c.lab());
        n = colors.length - 1;
        row = binom_row(n);
        I = function (t) {
          const u = 1 - t;
          const lab = [0, 1, 2].map(i => labs.reduce((sum, el, j) => sum + row[j] * u ** (n - j) * t ** j * el[i], 0));
          return new Color$1(lab, 'lab');
        };
      } else {
        throw new RangeError('No point in running bezier with only one color.');
      }
      return I;
    };
    var bezier$1 = colors => {
      const f = bezier(colors);
      f.scale = () => scale(f);
      return f;
    };

    /*
     * interpolates between a set of colors uzing a bezier spline
     * blend mode formulas taken from https://web.archive.org/web/20180110014946/http://www.venture-ware.com/kevin/coding/lets-learn-math-photoshop-blend-modes/
     */
    const blend = (bottom, top, mode) => {
      if (!blend[mode]) {
        throw new Error('unknown blend mode ' + mode);
      }
      return blend[mode](bottom, top);
    };
    const blend_f = f => (bottom, top) => {
      const c0 = chroma$1(top).rgb();
      const c1 = chroma$1(bottom).rgb();
      return chroma$1.rgb(f(c0, c1));
    };
    const each = f => (c0, c1) => {
      const out = [];
      out[0] = f(c0[0], c1[0]);
      out[1] = f(c0[1], c1[1]);
      out[2] = f(c0[2], c1[2]);
      return out;
    };
    const normal = a => a;
    const multiply = (a, b) => a * b / 255;
    const darken = (a, b) => a > b ? b : a;
    const lighten = (a, b) => a > b ? a : b;
    const screen = (a, b) => 255 * (1 - (1 - a / 255) * (1 - b / 255));
    const overlay = (a, b) => b < 128 ? 2 * a * b / 255 : 255 * (1 - 2 * (1 - a / 255) * (1 - b / 255));
    const burn = (a, b) => 255 * (1 - (1 - b / 255) / (a / 255));
    const dodge = (a, b) => {
      if (a === 255) return 255;
      a = 255 * (b / 255) / (1 - a / 255);
      return a > 255 ? 255 : a;
    };

    // # add = (a,b) ->
    // #     if (a + b > 255) then 255 else a + b

    blend.normal = blend_f(each(normal));
    blend.multiply = blend_f(each(multiply));
    blend.screen = blend_f(each(screen));
    blend.overlay = blend_f(each(overlay));
    blend.darken = blend_f(each(darken));
    blend.lighten = blend_f(each(lighten));
    blend.dodge = blend_f(each(dodge));
    blend.burn = blend_f(each(burn));
    // blend.add = blend_f(each(add));

    var blend$1 = blend;

    // cubehelix interpolation
    const {
      pow: pow$2,
      sin: sin$1,
      cos: cos$1
    } = Math;
    function cubehelix (start = 300, rotations = -1.5, hue = 1, gamma = 1, lightness = [0, 1]) {
      let dh = 0,
        dl;
      if (type(lightness) === 'array') {
        dl = lightness[1] - lightness[0];
      } else {
        dl = 0;
        lightness = [lightness, lightness];
      }
      const f = function (fract) {
        const a = TWOPI * ((start + 120) / 360 + rotations * fract);
        const l = pow$2(lightness[0] + dl * fract, gamma);
        const h = dh !== 0 ? hue[0] + fract * dh : hue;
        const amp = h * l * (1 - l) / 2;
        const cos_a = cos$1(a);
        const sin_a = sin$1(a);
        const r = l + amp * (-0.14861 * cos_a + 1.78277 * sin_a);
        const g = l + amp * (-0.29227 * cos_a - 0.90649 * sin_a);
        const b = l + amp * (+1.97294 * cos_a);
        return chroma$1(clip_rgb([r * 255, g * 255, b * 255, 1]));
      };
      f.start = function (s) {
        if (s == null) {
          return start;
        }
        start = s;
        return f;
      };
      f.rotations = function (r) {
        if (r == null) {
          return rotations;
        }
        rotations = r;
        return f;
      };
      f.gamma = function (g) {
        if (g == null) {
          return gamma;
        }
        gamma = g;
        return f;
      };
      f.hue = function (h) {
        if (h == null) {
          return hue;
        }
        hue = h;
        if (type(hue) === 'array') {
          dh = hue[1] - hue[0];
          if (dh === 0) {
            hue = hue[1];
          }
        } else {
          dh = 0;
        }
        return f;
      };
      f.lightness = function (h) {
        if (h == null) {
          return lightness;
        }
        if (type(h) === 'array') {
          lightness = h;
          dl = h[1] - h[0];
        } else {
          lightness = [h, h];
          dl = 0;
        }
        return f;
      };
      f.scale = () => chroma$1.scale(f);
      f.hue(hue);
      return f;
    }

    const digits = '0123456789abcdef';
    const {
      floor: floor$1,
      random
    } = Math;
    var random$1 = () => {
      let code = '#';
      for (let i = 0; i < 6; i++) {
        code += digits.charAt(floor$1(random() * 16));
      }
      return new Color$1(code, 'hex');
    };

    const {
      log,
      pow: pow$1,
      floor,
      abs: abs$1
    } = Math;
    function analyze(data, key = null) {
      const r = {
        min: Number.MAX_VALUE,
        max: Number.MAX_VALUE * -1,
        sum: 0,
        values: [],
        count: 0
      };
      if (type(data) === 'object') {
        data = Object.values(data);
      }
      data.forEach(val => {
        if (key && type(val) === 'object') val = val[key];
        if (val !== undefined && val !== null && !isNaN(val)) {
          r.values.push(val);
          r.sum += val;
          if (val < r.min) r.min = val;
          if (val > r.max) r.max = val;
          r.count += 1;
        }
      });
      r.domain = [r.min, r.max];
      r.limits = (mode, num) => limits(r, mode, num);
      return r;
    }
    function limits(data, mode = 'equal', num = 7) {
      if (type(data) == 'array') {
        data = analyze(data);
      }
      const {
        min,
        max
      } = data;
      const values = data.values.sort((a, b) => a - b);
      if (num === 1) {
        return [min, max];
      }
      const limits = [];
      if (mode.substr(0, 1) === 'c') {
        // continuous
        limits.push(min);
        limits.push(max);
      }
      if (mode.substr(0, 1) === 'e') {
        // equal interval
        limits.push(min);
        for (let i = 1; i < num; i++) {
          limits.push(min + i / num * (max - min));
        }
        limits.push(max);
      } else if (mode.substr(0, 1) === 'l') {
        // log scale
        if (min <= 0) {
          throw new Error('Logarithmic scales are only possible for values > 0');
        }
        const min_log = Math.LOG10E * log(min);
        const max_log = Math.LOG10E * log(max);
        limits.push(min);
        for (let i = 1; i < num; i++) {
          limits.push(pow$1(10, min_log + i / num * (max_log - min_log)));
        }
        limits.push(max);
      } else if (mode.substr(0, 1) === 'q') {
        // quantile scale
        limits.push(min);
        for (let i = 1; i < num; i++) {
          const p = (values.length - 1) * i / num;
          const pb = floor(p);
          if (pb === p) {
            limits.push(values[pb]);
          } else {
            // p > pb
            const pr = p - pb;
            limits.push(values[pb] * (1 - pr) + values[pb + 1] * pr);
          }
        }
        limits.push(max);
      } else if (mode.substr(0, 1) === 'k') {
        // k-means clustering
        /*
        implementation based on
        http://code.google.com/p/figue/source/browse/trunk/figue.js#336
        simplified for 1-d input values
        */
        let cluster;
        const n = values.length;
        const assignments = new Array(n);
        const clusterSizes = new Array(num);
        let repeat = true;
        let nb_iters = 0;
        let centroids = null;

        // get seed values
        centroids = [];
        centroids.push(min);
        for (let i = 1; i < num; i++) {
          centroids.push(min + i / num * (max - min));
        }
        centroids.push(max);
        while (repeat) {
          // assignment step
          for (let j = 0; j < num; j++) {
            clusterSizes[j] = 0;
          }
          for (let i = 0; i < n; i++) {
            const value = values[i];
            let mindist = Number.MAX_VALUE;
            let best;
            for (let j = 0; j < num; j++) {
              const dist = abs$1(centroids[j] - value);
              if (dist < mindist) {
                mindist = dist;
                best = j;
              }
              clusterSizes[best]++;
              assignments[i] = best;
            }
          }

          // update centroids step
          const newCentroids = new Array(num);
          for (let j = 0; j < num; j++) {
            newCentroids[j] = null;
          }
          for (let i = 0; i < n; i++) {
            cluster = assignments[i];
            if (newCentroids[cluster] === null) {
              newCentroids[cluster] = values[i];
            } else {
              newCentroids[cluster] += values[i];
            }
          }
          for (let j = 0; j < num; j++) {
            newCentroids[j] *= 1 / clusterSizes[j];
          }

          // check convergence
          repeat = false;
          for (let j = 0; j < num; j++) {
            if (newCentroids[j] !== centroids[j]) {
              repeat = true;
              break;
            }
          }
          centroids = newCentroids;
          nb_iters++;
          if (nb_iters > 200) {
            repeat = false;
          }
        }

        // finished k-means clustering
        // the next part is borrowed from gabrielflor.it
        const kClusters = {};
        for (let j = 0; j < num; j++) {
          kClusters[j] = [];
        }
        for (let i = 0; i < n; i++) {
          cluster = assignments[i];
          kClusters[cluster].push(values[i]);
        }
        let tmpKMeansBreaks = [];
        for (let j = 0; j < num; j++) {
          tmpKMeansBreaks.push(kClusters[j][0]);
          tmpKMeansBreaks.push(kClusters[j][kClusters[j].length - 1]);
        }
        tmpKMeansBreaks = tmpKMeansBreaks.sort((a, b) => a - b);
        limits.push(tmpKMeansBreaks[0]);
        for (let i = 1; i < tmpKMeansBreaks.length; i += 2) {
          const v = tmpKMeansBreaks[i];
          if (!isNaN(v) && limits.indexOf(v) === -1) {
            limits.push(v);
          }
        }
      }
      return limits;
    }

    var contrast = (a, b) => {
      // WCAG contrast ratio
      // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef
      a = new Color$1(a);
      b = new Color$1(b);
      const l1 = a.luminance();
      const l2 = b.luminance();
      return l1 > l2 ? (l1 + 0.05) / (l2 + 0.05) : (l2 + 0.05) / (l1 + 0.05);
    };

    const {
      sqrt,
      pow,
      min: min$1,
      max: max$1,
      atan2,
      abs,
      cos,
      sin,
      exp,
      PI
    } = Math;
    function deltaE (a, b, Kl = 1, Kc = 1, Kh = 1) {
      // Delta E (CIE 2000)
      // see http://www.brucelindbloom.com/index.html?Eqn_DeltaE_CIE2000.html
      var rad2deg = function (rad) {
        return 360 * rad / (2 * PI);
      };
      var deg2rad = function (deg) {
        return 2 * PI * deg / 360;
      };
      a = new Color$1(a);
      b = new Color$1(b);
      const [L1, a1, b1] = Array.from(a.lab());
      const [L2, a2, b2] = Array.from(b.lab());
      const avgL = (L1 + L2) / 2;
      const C1 = sqrt(pow(a1, 2) + pow(b1, 2));
      const C2 = sqrt(pow(a2, 2) + pow(b2, 2));
      const avgC = (C1 + C2) / 2;
      const G = 0.5 * (1 - sqrt(pow(avgC, 7) / (pow(avgC, 7) + pow(25, 7))));
      const a1p = a1 * (1 + G);
      const a2p = a2 * (1 + G);
      const C1p = sqrt(pow(a1p, 2) + pow(b1, 2));
      const C2p = sqrt(pow(a2p, 2) + pow(b2, 2));
      const avgCp = (C1p + C2p) / 2;
      const arctan1 = rad2deg(atan2(b1, a1p));
      const arctan2 = rad2deg(atan2(b2, a2p));
      const h1p = arctan1 >= 0 ? arctan1 : arctan1 + 360;
      const h2p = arctan2 >= 0 ? arctan2 : arctan2 + 360;
      const avgHp = abs(h1p - h2p) > 180 ? (h1p + h2p + 360) / 2 : (h1p + h2p) / 2;
      const T = 1 - 0.17 * cos(deg2rad(avgHp - 30)) + 0.24 * cos(deg2rad(2 * avgHp)) + 0.32 * cos(deg2rad(3 * avgHp + 6)) - 0.2 * cos(deg2rad(4 * avgHp - 63));
      let deltaHp = h2p - h1p;
      deltaHp = abs(deltaHp) <= 180 ? deltaHp : h2p <= h1p ? deltaHp + 360 : deltaHp - 360;
      deltaHp = 2 * sqrt(C1p * C2p) * sin(deg2rad(deltaHp) / 2);
      const deltaL = L2 - L1;
      const deltaCp = C2p - C1p;
      const sl = 1 + 0.015 * pow(avgL - 50, 2) / sqrt(20 + pow(avgL - 50, 2));
      const sc = 1 + 0.045 * avgCp;
      const sh = 1 + 0.015 * avgCp * T;
      const deltaTheta = 30 * exp(-pow((avgHp - 275) / 25, 2));
      const Rc = 2 * sqrt(pow(avgCp, 7) / (pow(avgCp, 7) + pow(25, 7)));
      const Rt = -Rc * sin(2 * deg2rad(deltaTheta));
      const result = sqrt(pow(deltaL / (Kl * sl), 2) + pow(deltaCp / (Kc * sc), 2) + pow(deltaHp / (Kh * sh), 2) + Rt * (deltaCp / (Kc * sc)) * (deltaHp / (Kh * sh)));
      return max$1(0, min$1(100, result));
    }

    // simple Euclidean distance
    function distance (a, b, mode = 'lab') {
      // Delta E (CIE 1976)
      // see http://www.brucelindbloom.com/index.html?Equations.html
      a = new Color$1(a);
      b = new Color$1(b);
      const l1 = a.get(mode);
      const l2 = b.get(mode);
      let sum_sq = 0;
      for (let i in l1) {
        const d = (l1[i] || 0) - (l2[i] || 0);
        sum_sq += d * d;
      }
      return Math.sqrt(sum_sq);
    }

    var valid = (...args) => {
      try {
        new Color$1(...args);
        return true;
        // eslint-disable-next-line
      } catch (e) {
        return false;
      }
    };

    // some pre-defined color scales:
    var scales = {
      cool() {
        return scale([chroma$1.hsl(180, 1, 0.9), chroma$1.hsl(250, 0.7, 0.4)]);
      },
      hot() {
        return scale(['#000', '#f00', '#ff0', '#fff']).mode('rgb');
      }
    };

    /**
        ColorBrewer colors for chroma.js

        Copyright (c) 2002 Cynthia Brewer, Mark Harrower, and The
        Pennsylvania State University.

        Licensed under the Apache License, Version 2.0 (the "License");
        you may not use this file except in compliance with the License.
        You may obtain a copy of the License at
        http://www.apache.org/licenses/LICENSE-2.0

        Unless required by applicable law or agreed to in writing, software distributed
        under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
        CONDITIONS OF ANY KIND, either express or implied. See the License for the
        specific language governing permissions and limitations under the License.
    */

    const colorbrewer = {
      // sequential
      OrRd: ['#fff7ec', '#fee8c8', '#fdd49e', '#fdbb84', '#fc8d59', '#ef6548', '#d7301f', '#b30000', '#7f0000'],
      PuBu: ['#fff7fb', '#ece7f2', '#d0d1e6', '#a6bddb', '#74a9cf', '#3690c0', '#0570b0', '#045a8d', '#023858'],
      BuPu: ['#f7fcfd', '#e0ecf4', '#bfd3e6', '#9ebcda', '#8c96c6', '#8c6bb1', '#88419d', '#810f7c', '#4d004b'],
      Oranges: ['#fff5eb', '#fee6ce', '#fdd0a2', '#fdae6b', '#fd8d3c', '#f16913', '#d94801', '#a63603', '#7f2704'],
      BuGn: ['#f7fcfd', '#e5f5f9', '#ccece6', '#99d8c9', '#66c2a4', '#41ae76', '#238b45', '#006d2c', '#00441b'],
      YlOrBr: ['#ffffe5', '#fff7bc', '#fee391', '#fec44f', '#fe9929', '#ec7014', '#cc4c02', '#993404', '#662506'],
      YlGn: ['#ffffe5', '#f7fcb9', '#d9f0a3', '#addd8e', '#78c679', '#41ab5d', '#238443', '#006837', '#004529'],
      Reds: ['#fff5f0', '#fee0d2', '#fcbba1', '#fc9272', '#fb6a4a', '#ef3b2c', '#cb181d', '#a50f15', '#67000d'],
      RdPu: ['#fff7f3', '#fde0dd', '#fcc5c0', '#fa9fb5', '#f768a1', '#dd3497', '#ae017e', '#7a0177', '#49006a'],
      Greens: ['#f7fcf5', '#e5f5e0', '#c7e9c0', '#a1d99b', '#74c476', '#41ab5d', '#238b45', '#006d2c', '#00441b'],
      YlGnBu: ['#ffffd9', '#edf8b1', '#c7e9b4', '#7fcdbb', '#41b6c4', '#1d91c0', '#225ea8', '#253494', '#081d58'],
      Purples: ['#fcfbfd', '#efedf5', '#dadaeb', '#bcbddc', '#9e9ac8', '#807dba', '#6a51a3', '#54278f', '#3f007d'],
      GnBu: ['#f7fcf0', '#e0f3db', '#ccebc5', '#a8ddb5', '#7bccc4', '#4eb3d3', '#2b8cbe', '#0868ac', '#084081'],
      Greys: ['#ffffff', '#f0f0f0', '#d9d9d9', '#bdbdbd', '#969696', '#737373', '#525252', '#252525', '#000000'],
      YlOrRd: ['#ffffcc', '#ffeda0', '#fed976', '#feb24c', '#fd8d3c', '#fc4e2a', '#e31a1c', '#bd0026', '#800026'],
      PuRd: ['#f7f4f9', '#e7e1ef', '#d4b9da', '#c994c7', '#df65b0', '#e7298a', '#ce1256', '#980043', '#67001f'],
      Blues: ['#f7fbff', '#deebf7', '#c6dbef', '#9ecae1', '#6baed6', '#4292c6', '#2171b5', '#08519c', '#08306b'],
      PuBuGn: ['#fff7fb', '#ece2f0', '#d0d1e6', '#a6bddb', '#67a9cf', '#3690c0', '#02818a', '#016c59', '#014636'],
      Viridis: ['#440154', '#482777', '#3f4a8a', '#31678e', '#26838f', '#1f9d8a', '#6cce5a', '#b6de2b', '#fee825'],
      // diverging
      Spectral: ['#9e0142', '#d53e4f', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#e6f598', '#abdda4', '#66c2a5', '#3288bd', '#5e4fa2'],
      RdYlGn: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#d9ef8b', '#a6d96a', '#66bd63', '#1a9850', '#006837'],
      RdBu: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#f7f7f7', '#d1e5f0', '#92c5de', '#4393c3', '#2166ac', '#053061'],
      PiYG: ['#8e0152', '#c51b7d', '#de77ae', '#f1b6da', '#fde0ef', '#f7f7f7', '#e6f5d0', '#b8e186', '#7fbc41', '#4d9221', '#276419'],
      PRGn: ['#40004b', '#762a83', '#9970ab', '#c2a5cf', '#e7d4e8', '#f7f7f7', '#d9f0d3', '#a6dba0', '#5aae61', '#1b7837', '#00441b'],
      RdYlBu: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee090', '#ffffbf', '#e0f3f8', '#abd9e9', '#74add1', '#4575b4', '#313695'],
      BrBG: ['#543005', '#8c510a', '#bf812d', '#dfc27d', '#f6e8c3', '#f5f5f5', '#c7eae5', '#80cdc1', '#35978f', '#01665e', '#003c30'],
      RdGy: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#ffffff', '#e0e0e0', '#bababa', '#878787', '#4d4d4d', '#1a1a1a'],
      PuOr: ['#7f3b08', '#b35806', '#e08214', '#fdb863', '#fee0b6', '#f7f7f7', '#d8daeb', '#b2abd2', '#8073ac', '#542788', '#2d004b'],
      // qualitative
      Set2: ['#66c2a5', '#fc8d62', '#8da0cb', '#e78ac3', '#a6d854', '#ffd92f', '#e5c494', '#b3b3b3'],
      Accent: ['#7fc97f', '#beaed4', '#fdc086', '#ffff99', '#386cb0', '#f0027f', '#bf5b17', '#666666'],
      Set1: ['#e41a1c', '#377eb8', '#4daf4a', '#984ea3', '#ff7f00', '#ffff33', '#a65628', '#f781bf', '#999999'],
      Set3: ['#8dd3c7', '#ffffb3', '#bebada', '#fb8072', '#80b1d3', '#fdb462', '#b3de69', '#fccde5', '#d9d9d9', '#bc80bd', '#ccebc5', '#ffed6f'],
      Dark2: ['#1b9e77', '#d95f02', '#7570b3', '#e7298a', '#66a61e', '#e6ab02', '#a6761d', '#666666'],
      Paired: ['#a6cee3', '#1f78b4', '#b2df8a', '#33a02c', '#fb9a99', '#e31a1c', '#fdbf6f', '#ff7f00', '#cab2d6', '#6a3d9a', '#ffff99', '#b15928'],
      Pastel2: ['#b3e2cd', '#fdcdac', '#cbd5e8', '#f4cae4', '#e6f5c9', '#fff2ae', '#f1e2cc', '#cccccc'],
      Pastel1: ['#fbb4ae', '#b3cde3', '#ccebc5', '#decbe4', '#fed9a6', '#ffffcc', '#e5d8bd', '#fddaec', '#f2f2f2']
    };

    // add lowercase aliases for case-insensitive matches
    for (let key of Object.keys(colorbrewer)) {
      colorbrewer[key.toLowerCase()] = colorbrewer[key];
    }
    var brewer = colorbrewer;

    Object.assign(chroma$1, {
      average,
      bezier: bezier$1,
      blend: blend$1,
      cubehelix,
      mix,
      interpolate: mix,
      random: random$1,
      scale,
      analyze,
      contrast,
      deltaE,
      distance,
      limits,
      valid,
      scales,
      input,
      colors,
      brewer
    });

    /**
     * Retrieve a chart with autocorrelation color
     * @param {SpectraProcessor} [spectraProcessor]
     * @param {number} [index]
     * @param {object} [options={}]
     * @param {Array} [options.autocorrelation] - precalculated autocorrelation {x,y}
     * @param {Array} - - [options.maxDataPoints=]
     * @param {Array} [options.ids] - ids of the spectra to select, by default all
     * @param {object} [options.xFilter={}]
     * @param {number} [options.xFilter.from]
     * @param {number} [options.xFilter.to]
     * @param {Array} [options.xFilter.exclusions=[]]
     */
    function getAutocorrelationChart(spectraProcessor, index, options = {}) {
      const {
        autocorrelation = spectraProcessor.getAutocorrelation(index, options),
        xFilter,
        ids
      } = options;
      const {
        min,
        max
      } = xMinMaxValues(autocorrelation.y);
      let colorCallback = chroma$1.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
      let colorScale = autocorrelation.y.map(y => `rgb(${colorCallback(y).rgb().join(',')})`);
      let mean = spectraProcessor.getMeanData({
        ids
      });
      if (xFilter) {
        mean = xyFilterX({
          x: mean.x,
          y: mean.y
        }, xFilter);
      }
      let colorSpectrum = {
        type: 'color',
        x: mean.x,
        y: mean.y,
        color: colorScale,
        styles: {
          unselected: {
            lineWidth: 1,
            lineStyle: 1
          },
          selected: {
            lineWidth: 3,
            lineStyle: 1
          }
        }
      };
      return colorSpectrum;
    }

    /**
     * @private
     * @param {object} normalizedData
     * @returns {x:[],q1:[],q3:[],media:[],min:[],max:[]}
     */

    function getBoxPlotData(normalizedData) {
      const {
        x,
        matrix
      } = normalizedData;
      let nbRow = matrix.length;
      let nbColumn = matrix[0].length;
      let result;
      let aColumn = new Float64Array(nbRow);
      for (let column = 0; column < nbColumn; column++) {
        for (let row = 0; row < nbRow; row++) {
          aColumn[row] = matrix[row][column];
        }
        let info = xBoxPlot(aColumn);
        if (!result) {
          result = {
            x
          };
          for (let key in info) {
            result[key] = [];
          }
        }
        for (let key in info) {
          result[key].push(info[key]);
        }
      }
      return result;
    }

    function getBoxPlotChart(normalizedData, options = {}) {
      const {
        q13FillColor = '#000',
        q13FillOpacity = 0.3,
        medianStrokeColor = '',
        medianStrokeWidth = 3,
        minMaxFillColor = '#000',
        minMaxFillOpacity = 0.15
      } = options;
      let annotations = [];
      const boxPlotData = getBoxPlotData(normalizedData);
      if (q13FillOpacity && q13FillColor) {
        let q13 = [];
        for (let i = 0; i < boxPlotData.x.length; i++) {
          q13.push({
            x: boxPlotData.x[i],
            y: boxPlotData.q1[i]
          });
        }
        for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
          q13.push({
            x: boxPlotData.x[i],
            y: boxPlotData.q3[i]
          });
        }
        annotations.push({
          type: 'polygon',
          layer: 0,
          properties: {
            fillColor: q13FillColor,
            fillOpacity: q13FillOpacity,
            strokeWidth: 0.0000001,
            position: q13
          }
        });
      }
      if (minMaxFillColor && minMaxFillOpacity) {
        let minMax = [];
        for (let i = 0; i < boxPlotData.x.length; i++) {
          minMax.push({
            x: boxPlotData.x[i],
            y: boxPlotData.min[i]
          });
        }
        for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
          minMax.push({
            x: boxPlotData.x[i],
            y: boxPlotData.max[i]
          });
        }
        annotations.push({
          type: 'polygon',
          layer: 0,
          properties: {
            fillColor: minMaxFillColor,
            fillOpacity: minMaxFillOpacity,
            strokeWidth: 0.0000001,
            strokeColor: '#FFF',
            position: minMax
          }
        });
      }
      let colorSpectrum = {
        type: 'color',
        data: {
          x: boxPlotData.x,
          y: boxPlotData.median,
          color: medianStrokeColor ? new Array(boxPlotData.x.length).fill(medianStrokeColor) : getColors(normalizedData.matrix)
        },
        styles: {
          unselected: {
            lineWidth: medianStrokeWidth,
            lineStyle: 1
          },
          selected: {
            lineWidth: medianStrokeWidth,
            lineStyle: 1
          }
        },
        annotations
      };
      return colorSpectrum;
    }
    function getColors(dataset) {
      const matrix = new Matrix(dataset);
      const stdevs = matrix.standardDeviation('column');
      const {
        min,
        max
      } = xMinMaxValues(stdevs);
      const colorCallback = chroma$1.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
      const colors = stdevs.map(y => `rgb(${colorCallback(y).rgb().join(',')})`);
      return colors;
    }

    function addChartDataStyle(data, spectrum) {
      data.styles = {
        unselected: {
          lineColor: spectrum.meta.color || 'darkgrey',
          lineWidth: 1,
          lineStyle: 1
        },
        selected: {
          lineColor: spectrum.meta.color || 'darkgrey',
          lineWidth: 3,
          lineStyle: 1
        }
      };
      data.label = spectrum.meta.id || spectrum.id;
    }

    /**
     * Retrieve a chart with selected original data
     * @param spectra
     * @param {object} [options={}]
     * @param {Array} [options.ids] - List of spectra ids, by default all
     * @param {Array} [options.yFactor=1]
     */
    function getChart(spectra, options = {}) {
      const {
        ids,
        yFactor,
        xFilter = {}
      } = options;
      let chart = {
        data: []
      };
      for (let spectrum of spectra) {
        if (!ids || ids.includes(spectrum.id)) {
          let data = spectrum.getData({
            yFactor,
            xFilter
          });
          addChartDataStyle(data, spectrum);
          chart.data.push(data);
        }
      }
      return chart;
    }

    function getNormalizationAnnotations(filter = {}) {
      let {
        exclusions = []
      } = filter;
      let annotations = [];
      exclusions = exclusions.filter(exclusion => !exclusion.ignore);
      annotations = exclusions.map(exclusion => {
        let annotation = {
          type: 'rect',
          position: [{
            x: exclusion.from,
            y: '0px'
          }, {
            x: exclusion.to,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,224,1)'
        };
        return annotation;
      });
      if (filter.from !== undefined) {
        annotations.push({
          type: 'rect',
          position: [{
            x: Number.MIN_SAFE_INTEGER,
            y: '0px'
          }, {
            x: filter.from,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,224,1)'
        });
      }
      if (filter.to !== undefined) {
        annotations.push({
          type: 'rect',
          position: [{
            x: filter.to,
            y: '0px'
          }, {
            x: Number.MAX_SAFE_INTEGER,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,224,1)'
        });
      }
      return annotations;
    }

    /**
     * @param {Array<Spectrum>} spectra
     * @param {object} [options={}]
     * @param {object} [options.xFilter={}]
     * @param {number} [options.xFilter.from]
     * @param {number} [options.xFilter.to]
     * @param {Array} [options.xFilter.exclusions=[]]
     */

    function getNormalizedChart(spectra, options = {}) {
      const {
        xFilter
      } = options;
      let chart = {
        data: []
      };
      for (let spectrum of spectra) {
        let data = spectrum.normalized;
        if (xFilter) {
          data = xyFilterX(data, xFilter);
        }
        addChartDataStyle(data, spectrum);
        chart.data.push(data);
      }
      return chart;
    }

    /**
     *
     * @param {SpectraProcessor} spectraProcessor
     * @param {object}   [options={}] - scale spectra based on various parameters
     * @param {Array}    [options.ids] - ids of selected spectra
     * @param {Array}    [options.filters=[]] - Array of object containing {name:'', options:''}
     * @param {object}   [options.scale={}] - object containing the options for rescaling
     * @param {string}   [options.scale.targetID=spectra[0].id]
     * @param {string}   [options.scale.method='max'] - min, max, integration, minMax
     * @param {Array}    [options.scale.range] - from - to to apply the method and rescale
     * @param {boolean}  [options.scale.relative=false]
     */
    function getPostProcessedChart(spectraProcessor, options = {}) {
      let scaled = spectraProcessor.getPostProcessedData(options);
      let chart = {
        data: []
      };
      if (!scaled || !scaled.matrix) return chart;
      for (let i = 0; i < scaled.matrix.length; i++) {
        let data = {
          x: scaled.x,
          y: Array.from(scaled.matrix[i]) // need to ensure not a typed array
        };
        addChartDataStyle(data, {
          meta: scaled.meta[i],
          id: scaled.ids[i]
        });
        chart.data.push(data);
      }
      return chart;
    }

    /**
     * @private
     * @param {*} spectra
     * @param {object} [options={}]
     * @param {Array} [options.ids] - ids of selected spectra
     */

    function getNormalizedData(spectra, options = {}) {
      const {
        ids
      } = options;
      let matrix = [];
      let meta = [];
      let currentIDs = [];
      let x = [];
      if (Array.isArray(spectra) && spectra.length > 0) {
        for (let spectrum of spectra) {
          if (!ids || ids.includes(spectrum.id)) {
            currentIDs.push(spectrum.id);
            matrix.push(spectrum.normalized.y);
            meta.push(spectrum.meta);
          }
        }
        x = spectra[0].normalized.x;
      }
      return {
        ids: currentIDs,
        matrix,
        meta,
        x
      };
    }

    function getTrackAnnotation(spectra, index, options = {}) {
      const {
        ids,
        showSpectrumID = true,
        sortY = true,
        limit = 20
      } = options;
      let annotations = [];
      let normalized = getNormalizedData(spectra, {
        ids
      });
      if (normalized.ids.length === 0) return annotations;
      let line = 0;

      // header containing X coordinate
      annotations.push({
        type: 'line',
        position: [{
          x: `${70}px`,
          y: `${15 + 15 * line}px`
        }, {
          x: `${85}px`,
          y: `${15 + 15 * line}px`
        }],
        strokeWidth: 0.0000001,
        label: {
          size: 16,
          text: `x: ${normalized.x[index].toPrecision(6)}`,
          position: {
            x: `${130}px`,
            y: `${20 + 15 * line}px`
          }
        }
      });
      line++;
      let peaks = [];
      for (let i = 0; i < normalized.ids.length; i++) {
        peaks.push({
          id: normalized.ids[i],
          meta: normalized.meta[i],
          y: normalized.matrix[i][index]
        });
      }
      if (sortY) {
        peaks = peaks.sort((a, b) => b.y - a.y);
      }
      if (limit) {
        peaks = peaks.slice(0, limit);
      }
      for (let {
        id,
        meta,
        y
      } of peaks) {
        annotations.push({
          type: 'line',
          position: [{
            x: `${70}px`,
            y: `${15 + 15 * line}px`
          }, {
            x: `${85}px`,
            y: `${15 + 15 * line}px`
          }],
          strokeColor: meta.color,
          strokeWidth: 2,
          label: {
            text: `${y.toPrecision(4)}${showSpectrumID ? ` - ${id}` : ''}`,
            position: {
              x: `${90}px`,
              y: `${20 + 15 * line}px`
            }
          }
        });
        line++;
      }
      return annotations;
    }

    /**
     * @private
     * @param {Array<object>} [metadata]
     * @param {object} [options={}] - scale spectra based on various parameters
     * @param {string} [options.propertyName="category"]
     */

    function getCategoriesStats(metadata, options = {}) {
      const {
        propertyName = 'category'
      } = options;
      let categories = {};
      let classNumber = 0;
      for (let metadatum of metadata) {
        let value = metadatum[propertyName];
        if (!categories[value]) {
          categories[value] = {
            classNumber: classNumber++,
            counter: 0
          };
        }
        categories[value].counter++;
      }
      return categories;
    }

    /**
     * @private
     * @param {Array<object>} [metadata]
     * @param {object} [options={}] - scale spectra based on various parameters
     * @param {string} [options.propertyName="category"]
     */

    function getClassLabels(metadata, options = {}) {
      const {
        propertyName = 'category'
      } = options;
      const categories = [];
      for (const metadatum of metadata) {
        categories.push(metadatum[propertyName]);
      }
      return categories;
    }

    /**
     * @private
     * @param {Array<object>} [metadata]
     * @param {object} [options={}] - scale spectra based on various parameters
     * @param {string} [options.propertyName="category"]
     */

    function getClasses(metadata, options = {}) {
      const {
        propertyName = 'category'
      } = options;
      const categoriesStats = getCategoriesStats(metadata, options);
      const result = new Array(metadata.length);
      for (let i = 0; i < metadata.length; i++) {
        result[i] = categoriesStats[metadata[i][propertyName]].classNumber;
      }
      return result;
    }

    /**
     * @private
     * @param {*} spectra
     * @param {object} options
     */

    function getMetadata(spectra, options = {}) {
      const {
        ids
      } = options;
      let metadata = [];
      if (Array.isArray(spectra) && spectra.length > 0) {
        for (let spectrum of spectra) {
          if (!ids || ids.includes(spectrum.id)) {
            metadata.push(spectrum.meta);
          }
        }
      }
      return metadata;
    }

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */
    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }
      var i = 0;
      var len = buf.length;
      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        }
        // UTF8-2 = %xC2-DF UTF8-tail
        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        }
        // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail
        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )
        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )
        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }
        return false;
      }
      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */
    function ensureString(blob, options = {}) {
      if (typeof blob === 'string') {
        return blob;
      }
      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }
      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }
    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }
        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      }
      //@ts-expect-error an ArrayBuffer is also ok
      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    function parseString(value) {
      if (value.length === 4 || value.length === 5) {
        const lowercase = value.toLowerCase();
        if (lowercase === 'true') return true;
        if (lowercase === 'false') return false;
      }
      const number = Number(value);
      if (number === 0 && !value.includes('0')) {
        return value;
      }
      if (!Number.isNaN(number)) return number;
      return value;
    }

    const gyromagneticRatio = {
      '1H': 267.52218744e6,
      '2H': 41.065e6,
      '3H': 285.3508e6,
      '3He': -203.789e6,
      '7Li': 103.962e6,
      '13C': 67.28284e6,
      '14N': 19.331e6,
      '15N': -27.116e6,
      '17O': -36.264e6,
      '19F': 251.662e6,
      '23Na': 70.761e6,
      '27Al': 69.763e6,
      '29Si': -53.19e6,
      '31P': 108.291e6,
      '57Fe': 8.681e6,
      '63Cu': 71.118e6,
      '67Zn': 16.767e6,
      '129Xe': -73.997e6,
      '195Pt': 58.385e6,
      '199Hg': 48.457916e6,
      '187Os': 6.192895e6,
      '183W': 11.282403e6,
      '125Te': -85.108404e6,
      '123Te': -70.59098e6,
      '119Sn': -100.317e6,
      '117Sn': -95.8879e6,
      '115Sn': -88.013e6,
      '113Cd': -59.609155e6,
      '111Cd': -56.983131e6,
      '109Ag': -12.518634e6,
      '77Se': 51.253857e6,
      '89Y': -13.162791e6,
      '103Rh': -8.468e6,
      '107Ag': -10.889181e6,
      '203Tl': 155.393338e6,
      '205Tl': 156.921808e6,
      '207Pb': 55.8046e6
    };
    function getGyromagneticRatio(nucleus) {
      if (gyromagneticRatio[nucleus]) {
        return gyromagneticRatio[nucleus];
      }
      nucleus = nucleus.toLowerCase();
      if (nucleus === 'proton') return gyromagneticRatio['1H'];
      // we try to use only the numbers
      const nucleusNumber = nucleus.replace(/[^0-9]/g, '');
      if (!nucleusNumber) return null;
      for (const key in gyromagneticRatio) {
        if (key.replace(/[^0-9]/g, '') === nucleusNumber) {
          return gyromagneticRatio[key];
        }
      }
      return null;
    }

    var medianQuickselect_min = {exports: {}};

    (function (module) {
      (function () {
        function a(d) {
          for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
            if (f <= e) return d[j];
            if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];
            for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
              do h++; while (d[e] > d[h]);
              do i--; while (d[i] > d[e]);
              if (i < h) break;
              b(d, h, i);
            }
            b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
          }
        }
        var b = function b(d, e, f) {
            var _ref;
            return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
          },
          c = function c(d, e) {
            return ~~((d + e) / 2);
          };
        module.exports ? module.exports = a : window.median = a;
      })();
    })(medianQuickselect_min);
    var quickSelectMedian = medianQuickselect_min.exports;

    function median(input) {
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      return quickSelectMedian(input.slice());
    }

    var P = /\s*\$\$.*/;
    function M(e) {
      return e.replace(P, "");
    }
    var E = ["TIC", ".RIC", "SCANNUMBER"];
    function te(e) {
      let a = e.spectra,
        s = a.length,
        i = {
          times: new Array(s),
          series: {
            ms: {
              dimension: 2,
              data: new Array(s)
            }
          }
        },
        o = [];
      for (let t = 0; t < E.length; t++) {
        let r = I(E[t]);
        a[0][r] && (o.push(r), i.series[r] = {
          dimension: 1,
          data: new Array(s)
        });
      }
      for (let t = 0; t < s; t++) {
        let r = a[t];
        i.times[t] = r.pageValue;
        for (let l = 0; l < o.length; l++) i.series[o[l]].data[t] = Number(r[o[l]]);
        r.data && (i.series.ms.data[t] = [r.data.x, r.data.y]);
      }
      e.chromatogram = i;
    }
    function le(e) {
      return E.indexOf(e) !== -1;
    }
    function I(e) {
      return e.toLowerCase().replace(/[^a-z0-9]/g, "");
    }
    function Y(e) {
      let a = [];
      for (let s = 0; s < e.length; s++) a.push(Number(e[s]));
      return a;
    }
    function U(e, a) {
      let s = e.yFactor,
        i = e.deltaX;
      e.isXYdata = !0;
      let o = {
        x: [],
        y: []
      };
      e.data = o;
      let t = e.firstX,
        r = e.firstY,
        l = !1,
        n,
        p = 0;
      for (; p < a.length; p++) if (n = a.charCodeAt(p), n === 13 || n === 10) l = !0;else if (l) break;
      let c = !0,
        d = !1,
        u = !1,
        f = 0,
        m = !1,
        h = !1,
        g = 0,
        A = 0,
        N = !1,
        y = !1,
        C = !1,
        T = 0;
      for (; p <= a.length; p++) if (p === a.length ? n = 13 : n = a.charCodeAt(p), h) (n === 13 || n === 10) && (c = !0, h = !1);else if (n <= 57 && n >= 48) y = !0, T > 0 ? g += (n - 48) / 10 ** T++ : (g *= 10, g += n - 48);else if (n === 44 || n === 46) y = !0, T++;else {
        if (y) {
          if (c) c = !1, u && (C = !0);else if (C) C = !1;else {
            d ? (f = N ? 0 - g : g, u = !0, d = !1) : m || (A = N ? 0 - g : g);
            let O = m ? g - 1 : 1;
            for (let v = 0; v < O; v++) u ? r += f : r = A, o.x.push(t), o.y.push(r * s), t += i;
          }
          N = !1, g = 0, T = 0, y = !1, m = !1;
        }
        if (n < 74 && n > 63) y = !0, u = !1, g = n - 64;else if (n > 96 && n < 106) y = !0, u = !1, g = n - 96, N = !0;else if (n === 115) y = !0, m = !0, g = 9;else if (n > 82 && n < 91) y = !0, m = !0, g = n - 82;else if (n > 73 && n < 83) y = !0, d = !0, g = n - 73;else if (n > 105 && n < 115) y = !0, d = !0, g = n - 105, N = !0;else if (n === 36 && a.charCodeAt(p + 1) === 36) y = !0, h = !0;else if (n === 37) y = !0, d = !0, g = 0, N = !1;else if (n === 45) {
          let O = a.charCodeAt(p + 1);
          (O >= 48 && O <= 57 || O === 44 || O === 46) && (y = !0, c || (u = !1), N = !0);
        } else (n === 13 || n === 10) && (c = !0, h = !1);
      }
    }
    var se = /[,\t ]+/;
    function D(e, a, s) {
      if (e.isPeaktable = !0, !e.variables || Object.keys(e.variables) === 2 ? me(e, a, s) : ce(e, a, s), e.variables) for (let i in e.variables) e.variables[i].data = e.data[i];
    }
    function me(e, a, s) {
      let i = {
        x: [],
        y: []
      };
      e.data = i;
      let o = a.split(/,? *,?[;\r\n]+ */);
      for (let t = 1; t < o.length; t++) {
        let r = o[t].trim().replace(P, "").split(se);
        if (r.length % 2 === 0) for (let l = 0; l < r.length; l = l + 2) i.x.push(Number(r[l]) * e.xFactor), i.y.push(Number(r[l + 1]) * e.yFactor);else s.logs.push(`Format error: ${r}`);
      }
    }
    function ce(e, a, s) {
      let i = {},
        o = Object.keys(e.variables),
        t = o.length;
      o.forEach(l => i[l] = []), e.data = i;
      let r = a.split(/,? *,?[;\r\n]+ */);
      for (let l = 1; l < r.length; l++) {
        let n = r[l].trim().replace(P, "").split(se);
        if (n.length % t === 0) for (let p = 0; p < n.length; p++) i[o[p % t]].push(Number(n[p]));else s.logs.push(`Wrong number of columns: ${n}`);
      }
    }
    function V(e, a) {
      e.isXYAdata = !0;
      let s = {};
      e.data = s;
      let i = a.split(/\r?\n/),
        o = i[0].replace(/^.*?([A-Z]+).*$/, "$1").split("").map(t => t.toLowerCase());
      for (let t = 1; t < i.length; t++) {
        let r = i[t].replace(/^\((.*)\)$/, "$1").split(/ *, */);
        for (let l = 0; l < o.length; l++) {
          let n = r[l];
          switch (o[l]) {
            case "x":
            case "y":
            case "w":
              n = Number.parseFloat(n);
              break;
            case "a":
              n = n.replace(/^<(.*)>$/, "$1");
              break;
            case "m":
              break;
            default:
              continue;
          }
          s[o[l]] || (s[o[l]] = []), s[o[l]].push(n);
        }
      }
    }
    function ie(e) {
      return Array.isArray(e) ? e[0] : e;
    }
    function B(e) {
      let {
          spectra: a
        } = e,
        s = a[0].data.y[0],
        i = s,
        o = a.length,
        t = a[0].data.x.length,
        r = new Array(o);
      for (let f = 0; f < o; f++) {
        r[f] = a[f].data.y;
        for (let m = 0; m < t; m++) {
          let h = r[f][m];
          h < s && (s = h), h > i && (i = h);
        }
      }
      let l = a[0].data.x[0],
        n = a[0].data.x[a[0].data.x.length - 1],
        {
          firstY: p,
          lastY: c
        } = he(e);
      if (l > n) for (let f of r) f.reverse();
      p > c && r.reverse();
      let d = [];
      for (let f = 0; f < r.length; f++) {
        let m = Float64Array.from(r[f]);
        for (let h = 0; h < m.length; h++) m[h] < 0 && (m[h] = -m[h]);
        d.push(median(m));
      }
      let u = median(d);
      return {
        z: r,
        minX: Math.min(l, n),
        maxX: Math.max(l, n),
        minY: Math.min(p, c),
        maxY: Math.max(p, c),
        minZ: s,
        maxZ: i,
        noise: u
      };
    }
    function he(e) {
      let {
        spectra: a,
        ntuples: s,
        info: i
      } = e;
      if (s) for (let o = 0; o < s.length; o++) {
        let {
          symbol: t,
          nucleus: r,
          units: l
        } = s[o];
        if (t.match(/[F|T]1/) && l?.toUpperCase().match("HZ")) {
          let n = ie(i[".OBSERVEFREQUENCY"]),
            {
              nucleus: p
            } = s.find(h => h.symbol.match(/[F|T]2/));
          if ([n, p, r].some(h => !h)) break;
          let c = getGyromagneticRatio(p),
            d = getGyromagneticRatio(r),
            {
              first: u,
              last: f
            } = s[o],
            m = n * d / c;
          return {
            firstY: u / m,
            lastY: f / m
          };
        }
      }
      return {
        firstY: a[0].pageValue,
        lastY: a[a.length - 1].pageValue
      };
    }
    function k(e, a) {
      let s = e.noise,
        i = e.z,
        o,
        t,
        r,
        l,
        n,
        p,
        c,
        d,
        u = i.length,
        f = i[0].length,
        m,
        h,
        g,
        A,
        N = e.minX,
        C = (e.maxX - N) / (f - 1),
        T = e.minY,
        v = (e.maxY - T) / (u - 1),
        oe = e.minZ,
        K = e.maxZ,
        Q = a.nbContourLevels * 2,
        W = new Array(Q),
        x;
      for (let R = 0; R < Q; R++) {
        let $ = {};
        W[R] = $;
        let fe = R % 2,
          _ = (K - a.noiseMultiplier * s) * Math.exp((R >> 1) - a.nbContourLevels);
        fe === 0 ? x = _ + a.noiseMultiplier * s : x = 0 - _ - a.noiseMultiplier * s;
        let b = [];
        if ($.zValue = x, $.lines = b, !(x <= oe || x >= K)) for (let X = 0; X < u - 1; X++) {
          let z = i[X],
            ee = i[X + 1];
          for (let S = 0; S < f - 1; S++) o = z[S], t = z[S + 1], r = ee[S], l = ee[S + 1], n = o > x, p = t > x, c = r > x, d = l > x, n !== p && n !== c && (m = S + (x - o) / (t - o), h = X, g = S, A = X + (x - o) / (r - o), b.push(m * C + N), b.push(h * v + T), b.push(g * C + N), b.push(A * v + T)), d !== p && d !== c && (m = S + 1, h = X + 1 - (x - l) / (t - l), g = S + 1 - (x - l) / (r - l), A = X + 1, b.push(m * C + N), b.push(h * v + T), b.push(g * C + N), b.push(A * v + T)), p !== c && (m = (S + 1 - (x - t) / (r - t)) * C + N, h = (X + (x - t) / (r - t)) * v + T, p !== n && (g = S + 1 - (x - t) / (o - t), A = X, b.push(m), b.push(h), b.push(g * C + N), b.push(A * v + T)), c !== n && (g = S, A = X + 1 - (x - r) / (o - r), b.push(m), b.push(h), b.push(g * C + N), b.push(A * v + T)), p !== d && (g = S + 1, A = X + (x - t) / (l - t), b.push(m), b.push(h), b.push(g * C + N), b.push(A * v + T)), c !== d && (g = S + (x - r) / (l - r), A = X + 1, b.push(m), b.push(h), b.push(g * C + N), b.push(A * v + T)));
        }
      }
      return {
        minX: e.minX,
        maxX: e.maxX,
        minY: e.minY,
        maxY: e.maxY,
        segments: W
      };
    }
    function Z(e, a) {
      let s = B(e);
      a.noContour || (e.contourLines = k(s, a), delete s.z), e.minMax = s;
    }
    function J(e, a) {
      for (let s of e) {
        let i = 0,
          o = 0;
        for (let t of s.spectra) {
          if (s.ntuples && s.ntuples.symbol ? (!i && t.observeFrequency && (i = t.observeFrequency), !o && t.shiftOffsetVal && (o = t.shiftOffsetVal)) : (i = t.observeFrequency, o = t.shiftOffsetVal), i && t.xUnits && t.xUnits.toUpperCase().includes("HZ")) {
            t.xUnits = "PPM", t.xFactor = t.xFactor / i, t.firstX = t.firstX / i, t.lastX = t.lastX / i, t.deltaX = t.deltaX / i;
            for (let r = 0; r < t.data.x.length; r++) t.data.x[r] /= i;
          }
          if (o && t.xUnits.toLowerCase().includes("ppm")) {
            let r = t.firstX - o;
            t.firstX = t.firstX - r, t.lastX = t.lastX - r;
            for (let l = 0; l < t.data.x.length; l++) t.data.x[l] -= r;
          }
          if (s.ntuples && s.ntuples.nucleus && s.ntuples.symbol) for (let r = 0; r < s.ntuples.nucleus.length; r++) {
            let l = s.ntuples.symbol[r],
              n = s.ntuples.nucleus[r];
            if (l.match(/^[F|T]/) && !n) {
              if (l.match(/[F|T]1/)) if (s.tmp.$NUC2) s.ntuples.nucleus[r] = s.tmp.$NUC2;else {
                let p = s.ntuples.symbol.indexOf(l.replace(/^([F|T]).*/, "$12"));
                p && s.ntuples.nucleus[p] && (s.ntuples.nucleus[r] = s.ntuples.nucleus[p]);
              }
              l.match(/[F|T]2/) && (s.ntuples.nucleus[r] = s.tmp.$NUC1);
            }
            l.match(/[F|T]2/) && (s.yType = s.ntuples.nucleus[0], getGyromagneticRatio(s.xType) || (s.xType = s.ntuples.nucleus[1]));
          }
          if (i && s.ntuples && s.ntuples.symbol && s.ntuples.nucleus) {
            let r = "",
              l = s.ntuples.symbol.indexOf(t.pageSymbol);
            if (s.ntuples.units && s.ntuples.units[l] && (r = s.ntuples.units[l]), !r.toLowerCase().match(/(ppm|seconds)/)) {
              if (l !== 0) {
                let u = "Not sure about this ntuples format";
                if (a) a.warn(u);else throw Error(u);
              }
              let {
                  nucleus: n
                } = s.ntuples,
                p = getGyromagneticRatio(n[0]),
                c = getGyromagneticRatio(n[1]);
              if (!p || !c) {
                let u = `Problem with determination of gyromagnetic ratio for ${n.join("-")}`;
                if (a) a.error(u);else throw Error(u);
              }
              let d = p / c * i;
              t.pageValue /= d;
            }
          }
        }
      }
    }
    function L(e, a, s = {}) {
      s.logger?.trace(a), e.profiling && e.profiling.push({
        action: a,
        time: Date.now() - s.start
      });
    }
    function G(e) {
      let a = e.spectra[0].data;
      e.chromatogram = {
        times: a.x.slice(),
        series: {
          intensity: {
            dimension: 1,
            data: a.y.slice()
          }
        }
      };
    }
    function H(e, a, s) {
      J(e, s.logger), de(e, s);
      for (let i of e) {
        if (Object.keys(i.ntuples).length > 0) {
          let o = [],
            t = Object.keys(i.ntuples);
          for (let r = 0; r < t.length; r++) {
            let l = t[r],
              n = i.ntuples[l];
            for (let p = 0; p < n.length; p++) o[p] || (o[p] = {}), o[p][l] = n[p];
          }
          i.ntuples = o;
        }
        i.twoD && s.wantXY && (Z(i, s), L(a, "Finished countour plot calculation", s), s.keepSpectra || delete i.spectra), s.chromatogram && (i.spectra.length > 1 ? te(i) : G(i), L(a, "Finished chromatogram calculation", s)), delete i.tmp;
      }
    }
    function de(e, a) {
      for (let s of e) for (let i in s.meta) {
        let o = s.meta[i];
        if (typeof o == "string") {
          if (o[0] === "{") {
            if (!o.includes(":") && o[o.length - 1] === "}") {
              let t = o.slice(1, -1).split(/[,; ]+/).filter(r => r);
              for (let r = 0; r < t.length; r++) s.meta[i + String(r)] = a.dynamicTyping ? parseString(t[r]) : t[r];
            }
          } else if (o[0] === "(") {
            let t = o.split(/\r?\n/),
              r = /^\((?<from>\d+)\.\.(?<to>\d+)\).*$/;
            if (r.test(t[0])) {
              let [l, n] = t[0].match(r).slice(1).map(Number),
                p = t.slice(1).join(" ").split(/[,; ]+/).filter(c => c);
              for (let c = l; c <= n; c++) a.dynamicTyping && typeof p[c - l] == "string" ? s.meta[i + String(c)] = parseString(p[c - l]) : s.meta[i + String(c)] = p[c - l];
            }
          }
        }
      }
    }
    function q(e, a, s) {
      let i = -1,
        o = -1,
        t = "",
        r = "";
      if (s.indexOf("++") > 0) t = s.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, "$1"), r = s.replace(/.*\.\.([a-zA-Z0-9]+).*/, "$1");else {
        s = s.replace(/[^a-zA-Z%]/g, ""), t = s.charAt(0), r = s.charAt(1), a.variables = {};
        for (let l of s) {
          let n = l.toLowerCase(),
            p = e.ntuples.symbol.indexOf(l);
          if (p === -1) throw Error(`Symbol undefined: ${l}`);
          a.variables[n] = {};
          for (let c in e.ntuples) e.ntuples[c][p] && (a.variables[n][c.replace(/^var/, "")] = e.ntuples[c][p]);
        }
      }
      i = e.ntuples.symbol.indexOf(t), o = e.ntuples.symbol.indexOf(r), i === -1 && (i = 0), o === -1 && (o = 0), e.ntuples.first && (e.ntuples.first.length > i && (a.firstX = e.ntuples.first[i]), e.ntuples.first.length > o && (a.firstY = e.ntuples.first[o])), e.ntuples.last && (e.ntuples.last.length > i && (a.lastX = e.ntuples.last[i]), e.ntuples.last.length > o && (a.lastY = e.ntuples.last[o])), e.ntuples.vardim && e.ntuples.vardim.length > i && (a.nbPoints = e.ntuples.vardim[i]), e.ntuples.factor && (e.ntuples.factor.length > i && (a.xFactor = e.ntuples.factor[i]), e.ntuples.factor.length > o && (a.yFactor = e.ntuples.factor[o])), e.ntuples.units && (e.ntuples.units.length > i && (e.ntuples.varname && e.ntuples.varname[i] ? a.xUnits = `${e.ntuples.varname[i]} [${e.ntuples.units[i]}]` : a.xUnits = e.ntuples.units[i]), e.ntuples.units.length > o && (e.ntuples.varname && e.ntuples.varname[o] ? a.yUnits = `${e.ntuples.varname[o]} [${e.ntuples.units[o]}]` : a.yUnits = e.ntuples.units[o]));
    }
    function w(e) {
      e.xFactor || (e.xFactor = 1), e.yFactor || (e.yFactor = 1);
    }
    var F = /[ \t]*,[ \t]*/,
      xe = {
        keepRecordsRegExp: /^$/,
        canonicDataLabels: !0,
        canonicMetadataLabels: !1,
        dynamicTyping: !0,
        withoutXY: !1,
        noTrimRegExp: /^$/,
        chromatogram: !1,
        keepSpectra: !1,
        noContour: !1,
        nbContourLevels: 7,
        noiseMultiplier: 5,
        profiling: !1
      };
    function Ne(e, a = {}) {
      e = ensureString(e), a = {
        ...xe,
        ...a
      }, a.logger?.debug("Starting jcamp conversion"), a.wantXY = !a.withoutXY, a.start = Date.now();
      let s = [],
        i = {
          profiling: a.profiling ? [] : !1,
          logs: [],
          entries: []
        },
        o = {
          children: []
        },
        t = o,
        r = [],
        l = {};
      if (typeof e != "string") throw new TypeError("the JCAMP should be a string");
      L(i, "Before split to LDRS", a);
      let n = e.replace(/[\r\n]+##/g, `
##`).split(`
##`);
      L(i, "Split to LDRS", a), n[0] && (n[0] = n[0].replace(/^[\r\n ]*##/, ""));
      for (let p of n) {
        let c = p.indexOf("="),
          d = c > 0 ? p.substring(0, c) : p,
          u = c > 0 ? d.match(a.noTrimRegExp) ? p.substring(c + 1) : p.substring(c + 1).trim() : "",
          f = d.replace(/[_ -]/g, "").toUpperCase();
        if (f === "DATATABLE") {
          let m = u.indexOf(`
`);
          if (m === -1 && (m = u.indexOf("\r")), m > 0) {
            let h = u.substring(0, m).split(/[ ,;\t]+/);
            q(t, l, h[0]), l.datatable = h[0], h[1] && h[1].indexOf("PEAKS") > -1 ? f = "PEAKTABLE" : h[1] && (h[1].indexOf("XYDATA") || h[0].indexOf("++") > 0) && (f = "XYDATA", l.nbPoints && (l.deltaX = (l.lastX - l.firstX) / (l.nbPoints - 1)));
          }
        }
        if (f === "XYDATA") {
          a.wantXY && (w(l), u.match(/.*\+\+.*/) ? (l.nbPoints && (l.deltaX = (l.lastX - l.firstX) / (l.nbPoints - 1)), U(l, u)) : D(l, u, i), t.spectra.push(l), l = {});
          continue;
        } else if (f === "PEAKTABLE") {
          a.wantXY && (w(l), D(l, u, i), t.spectra.push(l), l = {});
          continue;
        }
        if (f === "PEAKASSIGNMENTS") {
          a.wantXY && (u.match(/.*([^A-Z]*).*/) && V(l, u), t.spectra.push(l), l = {});
          continue;
        }
        if (f === "TITLE") {
          let m = t;
          m.children || (m.children = []), t = {
            spectra: [],
            ntuples: {},
            info: {},
            meta: {},
            tmp: {}
          }, m.children.push(t), r.push(m), s.push(t), t.title = u;
        } else f === "DATATYPE" ? (t.dataType = u, u.match(/(^nd|\snd\s)/i) && (t.twoD = !0)) : f === "NTUPLES" ? u.match(/(^nd|\snd\s)/i) && (t.twoD = !0) : f === "DATACLASS" ? t.dataClass = u : f === "JCAMPDX" ? t.jcampDX = M(u) : f === "JCAMPCS" ? t.jcampCS = M(u) : f === "XUNITS" ? l.xUnits = u : f === "YUNITS" ? l.yUnits = u : f === "FIRSTX" ? l.firstX = Number(u) : f === "LASTX" ? l.lastX = Number(u) : f === "FIRSTY" ? l.firstY = Number(u) : f === "LASTY" ? l.lastY = Number(u) : f === "NPOINTS" ? l.nbPoints = Number(u) : f === "XFACTOR" ? l.xFactor = Number(u) : f === "YFACTOR" ? l.yFactor = Number(u) : f === "MAXX" ? l.maxX = Number(u) : f === "MINX" ? l.minX = Number(u) : f === "MAXY" ? l.maxY = Number(u) : f === "MINY" ? l.minY = Number(u) : f === "DELTAX" ? l.deltaX = Number(u) : f === ".OBSERVEFREQUENCY" || f === "$SFO1" ? l.observeFrequency || (l.observeFrequency = Number(u)) : f === ".OBSERVENUCLEUS" ? l.xType || (t.xType = u.replace(/[^a-zA-Z0-9]/g, "")) : f === "$OFFSET" ? (t.shiftOffsetNum = 0, l.shiftOffsetVal || (l.shiftOffsetVal = Number(u))) : f === "$REFERENCEPOINT" || (f === "VARNAME" ? t.ntuples.varname = u.split(F) : f === "SYMBOL" ? t.ntuples.symbol = u.split(F) : f === "VARTYPE" ? t.ntuples.vartype = u.split(F) : f === "VARFORM" ? t.ntuples.varform = u.split(F) : f === "VARDIM" ? t.ntuples.vardim = Y(u.split(F)) : f === "UNITS" ? t.ntuples.units = u.split(F) : f === "FACTOR" ? t.ntuples.factor = Y(u.split(F)) : f === "FIRST" ? t.ntuples.first = Y(u.split(F)) : f === "LAST" ? t.ntuples.last = Y(u.split(F)) : f === "MIN" ? t.ntuples.min = Y(u.split(F)) : f === "MAX" ? t.ntuples.max = Y(u.split(F)) : f === ".NUCLEUS" ? t.ntuples && (t.ntuples.nucleus = u.split(F).map(m => m.replace(/[^a-zA-Z0-9]/g, ""))) : f === "PAGE" ? (l.page = u.trim(), l.pageValue = Number(u.replace(/^.*=/, "")), l.pageSymbol = l.page.replace(/[=].*/, "")) : f === "RETENTIONTIME" ? l.pageValue = Number(u) : le(f) ? l[I(f)] = u : f === "SAMPLEDESCRIPTION" ? l.sampleDescription = u : f.startsWith("$NUC") ? !t.tmp[f] && !u.includes("off") && (t.tmp[f] = u.replace(/[<>]/g, "")) : f === "END" && (t = r.pop()));
        if (t && t.info && t.meta && f.match(a.keepRecordsRegExp)) {
          let m, h;
          d.startsWith("$") ? (h = a.canonicMetadataLabels ? f.substring(1) : d.substring(1), m = t.meta) : (h = a.canonicDataLabels ? f : d, m = t.info), a.dynamicTyping && (u = parseString(u)), m[h] ? (Array.isArray(m[h]) || (m[h] = [m[h]]), m[h].push(u)) : m[h] = u;
        }
      }
      if (L(i, "Finished parsing", a), H(s, i, a), L(i, "Total time", a), i.entries = o.children, i.flatten = s, a.logger) {
        a.logger.debug("Finished jcamp conversion");
        for (let p of i.flatten) a.logger.debug(`${p.dataType} - ${p.title}`);
      }
      return i;
    }

    function getJcampKind(data) {
      let dataType = data.dataType.toLowerCase();
      let yUnits = data.spectra[0].yUnits.toLowerCase();
      if (dataType.match(/infrared/)) {
        if (yUnits.match(/absorbance/)) {
          return IR_ABSORBANCE;
        } else {
          return IR_TRANSMITTANCE;
        }
      }
      return undefined;
    }
    const IR_TRANSMITTANCE = {
      normalization: {},
      importation: {
        converter: transmittance => -Math.log10(transmittance)
      },
      kind: 'Infrared',
      display: {
        xLabel: 'wavelength [cm-1]',
        xInverted: true,
        yLabel: 'Absorbance'
      }
    };
    const IR_ABSORBANCE = {
      normalization: {},
      kind: 'Infrared',
      display: {
        xLabel: 'wavelength [cm-1]',
        xInverted: true,
        yLabel: 'Absorbance'
      }
    };

    /**
     * Create a spectrum from a jcamp
     * @param {string} jcampText - String containing the JCAMP data
     * @returns {Spectrum} - new instance of Spectrum with the provided data
     */
    function parseJcamp(jcampText) {
      const parsed = Ne(jcampText, {
        keepRecordsRegExp: /.*/
      }).flatten[0];
      const kind = getJcampKind(parsed);
      const data = parsed.spectra[0].data;
      const {
        meta,
        info
      } = parsed;
      // we convert the data
      if (kind && kind.importation && kind.importation.converter) {
        data.y = data.y.map(kind.importation.converter);
      }
      return {
        data,
        kind,
        meta,
        info
      };
    }

    /**
     * Creates a g
     * @param {string} text - String containing the text data
     * @param {object} [options={}]
     * @param {string} [options.fs='\t'] - Field separator
     * @returns {object} - {matrix, data, x, ids}
     */
    function parseMatrix(text, options = {}) {
      const lines = text.split(/[\n\r]+/).filter(Boolean);
      const {
        fs = '\t'
      } = options;
      let currentMatrix = [];
      let ids = [];
      let meta = [];
      let x = [];
      let headers = lines[0].split(fs);
      let labels = [];
      for (let i = 0; i < headers.length; i++) {
        let header = headers[i];
        if (Number.isNaN(Number(header))) {
          labels[i] = header;
        } else {
          x = headers.slice(i).map(Number);
          break;
        }
      }
      for (let i = 1; i < lines.length; i++) {
        let line = lines[i];
        let parts = line.split('\t');
        ids.push(parts[0]);
        let oneMeta = {};
        meta.push(oneMeta);
        for (let j = 1; j < parts.length; j++) {
          if (j < labels.length) {
            oneMeta[labels[j]] = parts[j];
          } else {
            currentMatrix.push(parts.slice(labels.length).map(Number));
            break;
          }
        }
      }
      return {
        x,
        meta,
        matrix: currentMatrix,
        ids
      };
    }

    /**
     * General internal parsing function
     * @param text - Csv or tsv strings.
     * @param options - Parsing options
     * @returns parsed text file with column information
     */
    function parse(text, options = {}) {
      const {
        rescale = false,
        uniqueX = false,
        bestGuess = false,
        //@ts-expect-error old library used this property and we want to throw an error so that people are forced to migrate
        keepInfo
      } = options;
      let {
        xColumn = 0,
        yColumn = 1,
        numberColumns = Number.MAX_SAFE_INTEGER,
        maxNumberColumns = Number.MAX_SAFE_INTEGER,
        minNumberColumns = 2
      } = options;
      if (keepInfo !== undefined) {
        throw new Error('keepInfo has been deprecated, pelase use the new method parseXYAndKeepInfo');
      }
      text = ensureString(text);
      maxNumberColumns = Math.max(maxNumberColumns, xColumn + 1, yColumn + 1);
      minNumberColumns = Math.max(xColumn + 1, yColumn + 1, minNumberColumns);
      const lines = text.split(/[\r\n]+/);
      let matrix = [];
      const info = [];
      let position = 0;
      lines.forEach(line => {
        line = line.trim();
        // we will consider only lines that contains only numbers
        if (/[0-9]+/.test(line) && /^[0-9eE,;. \t+-]+$/.test(line)) {
          let fields = line.split(/,[; \t]+|[; \t]+/);
          if (fields.length === 1) {
            fields = line.split(/[,; \t]+/);
          }
          if (fields && fields.length >= minNumberColumns &&
          // we filter lines that have not enough or too many columns
          fields.length <= maxNumberColumns) {
            matrix.push(fields.map(value => parseFloat(value.replace(',', '.'))));
            position++;
          }
        } else if (line) {
          info.push({
            position,
            value: line
          });
        }
      });
      if (bestGuess) {
        if (matrix[0] && matrix[0].length === 3 && options.xColumn === undefined && options.yColumn === undefined) {
          // is the first column a seuqnetial number ?
          let skipFirstColumn = true;
          for (let i = 0; i < matrix.length - 1; i++) {
            if (Math.abs(matrix[i][0] - matrix[i + 1][0]) !== 1) {
              skipFirstColumn = false;
            }
          }
          if (skipFirstColumn) {
            xColumn = 1;
            yColumn = 2;
          }
        }
        if (matrix[0] && matrix[0].length > 3) {
          const xs = [];
          for (const row of matrix) {
            for (let i = xColumn; i < row.length; i += 2) {
              xs.push(row[i]);
            }
          }
          if (xIsMonotonic(xs)) {
            numberColumns = 2;
          }
        }
      }
      if (numberColumns) {
        const newMatrix = [];
        for (const row of matrix) {
          for (let i = 0; i < row.length; i += numberColumns) {
            newMatrix.push(row.slice(i, i + numberColumns));
          }
        }
        matrix = newMatrix;
      }
      let result = {
        x: matrix.map(row => row[xColumn]),
        y: matrix.map(row => row[yColumn])
      };
      if (uniqueX) {
        result = xyUniqueX(result, {
          algorithm: 'sum'
        });
      }
      if (rescale) {
        const maxY = xMaxValue(result.y);
        for (let i = 0; i < result.y.length; i++) {
          result.y[i] /= maxY;
        }
      }
      return {
        info,
        data: result
      };
    }

    /**
     * Parse a text-file and convert it to an object {x:[], y:[]}
     * @param text - Csv or tsv strings.
     * @param options - Parsing options
     * @returns - The parsed data
     */
    function parseXY(text, options = {}) {
      return parse(text, options).data;
    }

    /**
     * Create a spectrum from a text file
     * @param {string} value - String containing the JCAMP data
     * @param options
     * @returns {Spectrum} - new instance of Spectrum with the provided data
     */
    function parseText(value, options = {}) {
      const {
        kind,
        parserOptions = {}
      } = options;
      const data = parseXY(value, parserOptions);

      // we convert the data
      if (kind && kind.importation && kind.importation.converter) {
        data.y = data.y.map(kind.importation.converter);
      }
      return {
        data
      };
    }

    /**
     * @param normalized
     * @param index
     * @private
     */

    function getAutocorrelation(normalized, index) {
      let matrix = normalized.matrix;
      let nbRow = matrix.length;
      let nbColumn = matrix[0].length;
      let array1 = new Float64Array(nbRow);
      let array2 = new Float64Array(nbRow);
      let result = new Array(nbColumn);
      for (let j = 0; j < nbRow; j++) {
        array1[j] = matrix[j][index];
      }
      for (let i = 0; i < nbColumn; i++) {
        for (let j = 0; j < nbRow; j++) {
          array2[j] = matrix[j][i];
        }
        result[i] = xCorrelation(array1, array2);
      }
      return {
        x: normalized.x,
        y: result
      };
    }

    /**
     * @param normalized
     * @private
     */

    function getMeanData(normalized) {
      let matrix = new Matrix(normalized.matrix);
      return {
        x: normalized.x,
        y: matrix.mean('column')
      };
    }

    function convertToText(data, options = {}) {
      let {
        fs = '\t',
        rs = '\n'
      } = options;
      let {
        matrix,
        meta,
        ids,
        x
      } = data;
      let allKeysObject = {};
      for (let metum of meta) {
        if (metum) {
          for (let key of Object.keys(metum)) {
            let type = typeof metum[key];
            if (type === 'number' || type === 'string' || type === 'boolean') {
              allKeysObject[key] = true;
            }
          }
        }
      }
      let allKeys = Object.keys(allKeysObject);
      let lines = [];
      let line = [];
      line.push('id', ...allKeys, ...x);
      lines.push(line.join(fs));
      for (let i = 0; i < ids.length; i++) {
        line = [];
        line.push(ids[i]);
        for (let key of allKeys) {
          line.push(removeSpecialCharacters(meta[i][key]));
        }
        line.push(...matrix[i]);
        lines.push(line.join(fs));
      }
      return lines.join(rs);
    }
    function removeSpecialCharacters(string) {
      if (typeof string !== 'string') {
        return string;
      }
      return string.replaceAll(/[\t\n\r]+/g, ' ');
    }

    /**
     * @private
     * @param {*} spectra
     * @param {object} [options={}]
     * @param {string} [options.fs='\t'] - field separator
     * @param {string} [options.rs='\n'] - record (line) separator
     */

    function getNormalizedText(spectra, options = {}) {
      let {
        fs = '\t',
        rs = '\n'
      } = options;
      return convertToText(getNormalizedData(spectra), {
        rs,
        fs
      });
    }

    var object_hash = {exports: {}};

    (function (module, exports) {
      !function (e) {
        module.exports = e() ;
      }(function () {
        return function r(o, i, u) {
          function s(n, e) {
            if (!i[n]) {
              if (!o[n]) {
                var t = "function" == typeof commonjsRequire && commonjsRequire;
                if (!e && t) return t(n, !0);
                if (a) return a(n, !0);
                throw new Error("Cannot find module '" + n + "'");
              }
              e = i[n] = {
                exports: {}
              };
              o[n][0].call(e.exports, function (e) {
                var t = o[n][1][e];
                return s(t || e);
              }, e, e.exports, r, o, i, u);
            }
            return i[n].exports;
          }
          for (var a = "function" == typeof commonjsRequire && commonjsRequire, e = 0; e < u.length; e++) s(u[e]);
          return s;
        }({
          1: [function (w, b, m) {
            !function (e, n, s, c, d, h, p, g, y) {

              var r = w("crypto");
              function t(e, t) {
                t = u(e, t);
                var n;
                return void 0 === (n = "passthrough" !== t.algorithm ? r.createHash(t.algorithm) : new l()).write && (n.write = n.update, n.end = n.update), f(t, n).dispatch(e), n.update || n.end(""), n.digest ? n.digest("buffer" === t.encoding ? void 0 : t.encoding) : (e = n.read(), "buffer" !== t.encoding ? e.toString(t.encoding) : e);
              }
              (m = b.exports = t).sha1 = function (e) {
                return t(e);
              }, m.keys = function (e) {
                return t(e, {
                  excludeValues: !0,
                  algorithm: "sha1",
                  encoding: "hex"
                });
              }, m.MD5 = function (e) {
                return t(e, {
                  algorithm: "md5",
                  encoding: "hex"
                });
              }, m.keysMD5 = function (e) {
                return t(e, {
                  algorithm: "md5",
                  encoding: "hex",
                  excludeValues: !0
                });
              };
              var o = r.getHashes ? r.getHashes().slice() : ["sha1", "md5"],
                i = (o.push("passthrough"), ["buffer", "hex", "binary", "base64"]);
              function u(e, t) {
                var n = {};
                if (n.algorithm = (t = t || {}).algorithm || "sha1", n.encoding = t.encoding || "hex", n.excludeValues = !!t.excludeValues, n.algorithm = n.algorithm.toLowerCase(), n.encoding = n.encoding.toLowerCase(), n.ignoreUnknown = !0 === t.ignoreUnknown, n.respectType = !1 !== t.respectType, n.respectFunctionNames = !1 !== t.respectFunctionNames, n.respectFunctionProperties = !1 !== t.respectFunctionProperties, n.unorderedArrays = !0 === t.unorderedArrays, n.unorderedSets = !1 !== t.unorderedSets, n.unorderedObjects = !1 !== t.unorderedObjects, n.replacer = t.replacer || void 0, n.excludeKeys = t.excludeKeys || void 0, void 0 === e) throw new Error("Object argument required.");
                for (var r = 0; r < o.length; ++r) o[r].toLowerCase() === n.algorithm.toLowerCase() && (n.algorithm = o[r]);
                if (-1 === o.indexOf(n.algorithm)) throw new Error('Algorithm "' + n.algorithm + '"  not supported. supported values: ' + o.join(", "));
                if (-1 === i.indexOf(n.encoding) && "passthrough" !== n.algorithm) throw new Error('Encoding "' + n.encoding + '"  not supported. supported values: ' + i.join(", "));
                return n;
              }
              function a(e) {
                if ("function" == typeof e) return null != /^function\s+\w*\s*\(\s*\)\s*{\s+\[native code\]\s+}$/i.exec(Function.prototype.toString.call(e));
              }
              function f(o, t, i) {
                i = i || [];
                function u(e) {
                  return t.update ? t.update(e, "utf8") : t.write(e, "utf8");
                }
                return {
                  dispatch: function (e) {
                    return this["_" + (null === (e = o.replacer ? o.replacer(e) : e) ? "null" : typeof e)](e);
                  },
                  _object: function (t) {
                    var n,
                      e = Object.prototype.toString.call(t),
                      r = /\[object (.*)\]/i.exec(e);
                    r = (r = r ? r[1] : "unknown:[" + e + "]").toLowerCase();
                    if (0 <= (e = i.indexOf(t))) return this.dispatch("[CIRCULAR:" + e + "]");
                    if (i.push(t), void 0 !== s && s.isBuffer && s.isBuffer(t)) return u("buffer:"), u(t);
                    if ("object" === r || "function" === r || "asyncfunction" === r) return e = Object.keys(t), o.unorderedObjects && (e = e.sort()), !1 === o.respectType || a(t) || e.splice(0, 0, "prototype", "__proto__", "constructor"), o.excludeKeys && (e = e.filter(function (e) {
                      return !o.excludeKeys(e);
                    })), u("object:" + e.length + ":"), n = this, e.forEach(function (e) {
                      n.dispatch(e), u(":"), o.excludeValues || n.dispatch(t[e]), u(",");
                    });
                    if (!this["_" + r]) {
                      if (o.ignoreUnknown) return u("[" + r + "]");
                      throw new Error('Unknown object type "' + r + '"');
                    }
                    this["_" + r](t);
                  },
                  _array: function (e, t) {
                    t = void 0 !== t ? t : !1 !== o.unorderedArrays;
                    var n = this;
                    if (u("array:" + e.length + ":"), !t || e.length <= 1) return e.forEach(function (e) {
                      return n.dispatch(e);
                    });
                    var r = [],
                      t = e.map(function (e) {
                        var t = new l(),
                          n = i.slice();
                        return f(o, t, n).dispatch(e), r = r.concat(n.slice(i.length)), t.read().toString();
                      });
                    return i = i.concat(r), t.sort(), this._array(t, !1);
                  },
                  _date: function (e) {
                    return u("date:" + e.toJSON());
                  },
                  _symbol: function (e) {
                    return u("symbol:" + e.toString());
                  },
                  _error: function (e) {
                    return u("error:" + e.toString());
                  },
                  _boolean: function (e) {
                    return u("bool:" + e.toString());
                  },
                  _string: function (e) {
                    u("string:" + e.length + ":"), u(e.toString());
                  },
                  _function: function (e) {
                    u("fn:"), a(e) ? this.dispatch("[native]") : this.dispatch(e.toString()), !1 !== o.respectFunctionNames && this.dispatch("function-name:" + String(e.name)), o.respectFunctionProperties && this._object(e);
                  },
                  _number: function (e) {
                    return u("number:" + e.toString());
                  },
                  _xml: function (e) {
                    return u("xml:" + e.toString());
                  },
                  _null: function () {
                    return u("Null");
                  },
                  _undefined: function () {
                    return u("Undefined");
                  },
                  _regexp: function (e) {
                    return u("regex:" + e.toString());
                  },
                  _uint8array: function (e) {
                    return u("uint8array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _uint8clampedarray: function (e) {
                    return u("uint8clampedarray:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _int8array: function (e) {
                    return u("int8array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _uint16array: function (e) {
                    return u("uint16array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _int16array: function (e) {
                    return u("int16array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _uint32array: function (e) {
                    return u("uint32array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _int32array: function (e) {
                    return u("int32array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _float32array: function (e) {
                    return u("float32array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _float64array: function (e) {
                    return u("float64array:"), this.dispatch(Array.prototype.slice.call(e));
                  },
                  _arraybuffer: function (e) {
                    return u("arraybuffer:"), this.dispatch(new Uint8Array(e));
                  },
                  _url: function (e) {
                    return u("url:" + e.toString());
                  },
                  _map: function (e) {
                    u("map:");
                    e = Array.from(e);
                    return this._array(e, !1 !== o.unorderedSets);
                  },
                  _set: function (e) {
                    u("set:");
                    e = Array.from(e);
                    return this._array(e, !1 !== o.unorderedSets);
                  },
                  _file: function (e) {
                    return u("file:"), this.dispatch([e.name, e.size, e.type, e.lastModfied]);
                  },
                  _blob: function () {
                    if (o.ignoreUnknown) return u("[blob]");
                    throw Error('Hashing Blob objects is currently not supported\n(see https://github.com/puleos/object-hash/issues/26)\nUse "options.replacer" or "options.ignoreUnknown"\n');
                  },
                  _domwindow: function () {
                    return u("domwindow");
                  },
                  _bigint: function (e) {
                    return u("bigint:" + e.toString());
                  },
                  _process: function () {
                    return u("process");
                  },
                  _timer: function () {
                    return u("timer");
                  },
                  _pipe: function () {
                    return u("pipe");
                  },
                  _tcp: function () {
                    return u("tcp");
                  },
                  _udp: function () {
                    return u("udp");
                  },
                  _tty: function () {
                    return u("tty");
                  },
                  _statwatcher: function () {
                    return u("statwatcher");
                  },
                  _securecontext: function () {
                    return u("securecontext");
                  },
                  _connection: function () {
                    return u("connection");
                  },
                  _zlib: function () {
                    return u("zlib");
                  },
                  _context: function () {
                    return u("context");
                  },
                  _nodescript: function () {
                    return u("nodescript");
                  },
                  _httpparser: function () {
                    return u("httpparser");
                  },
                  _dataview: function () {
                    return u("dataview");
                  },
                  _signal: function () {
                    return u("signal");
                  },
                  _fsevent: function () {
                    return u("fsevent");
                  },
                  _tlswrap: function () {
                    return u("tlswrap");
                  }
                };
              }
              function l() {
                return {
                  buf: "",
                  write: function (e) {
                    this.buf += e;
                  },
                  end: function (e) {
                    this.buf += e;
                  },
                  read: function () {
                    return this.buf;
                  }
                };
              }
              m.writeToStream = function (e, t, n) {
                return void 0 === n && (n = t, t = {}), f(t = u(e, t), n).dispatch(e);
              };
            }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/fake_9a5aa49d.js", "/");
          }, {
            buffer: 3,
            crypto: 5,
            lYpoI2: 11
          }],
          2: [function (e, t, f) {
            !function (e, t, n, r, o, i, u, s, a) {
              !function (e) {

                var a = "undefined" != typeof Uint8Array ? Uint8Array : Array,
                  t = "+".charCodeAt(0),
                  n = "/".charCodeAt(0),
                  r = "0".charCodeAt(0),
                  o = "a".charCodeAt(0),
                  i = "A".charCodeAt(0),
                  u = "-".charCodeAt(0),
                  s = "_".charCodeAt(0);
                function f(e) {
                  e = e.charCodeAt(0);
                  return e === t || e === u ? 62 : e === n || e === s ? 63 : e < r ? -1 : e < r + 10 ? e - r + 26 + 26 : e < i + 26 ? e - i : e < o + 26 ? e - o + 26 : void 0;
                }
                e.toByteArray = function (e) {
                  var t, n;
                  if (0 < e.length % 4) throw new Error("Invalid string. Length must be a multiple of 4");
                  var r = e.length,
                    r = "=" === e.charAt(r - 2) ? 2 : "=" === e.charAt(r - 1) ? 1 : 0,
                    o = new a(3 * e.length / 4 - r),
                    i = 0 < r ? e.length - 4 : e.length,
                    u = 0;
                  function s(e) {
                    o[u++] = e;
                  }
                  for (t = 0; t < i; t += 4, 0) s((16711680 & (n = f(e.charAt(t)) << 18 | f(e.charAt(t + 1)) << 12 | f(e.charAt(t + 2)) << 6 | f(e.charAt(t + 3)))) >> 16), s((65280 & n) >> 8), s(255 & n);
                  return 2 == r ? s(255 & (n = f(e.charAt(t)) << 2 | f(e.charAt(t + 1)) >> 4)) : 1 == r && (s((n = f(e.charAt(t)) << 10 | f(e.charAt(t + 1)) << 4 | f(e.charAt(t + 2)) >> 2) >> 8 & 255), s(255 & n)), o;
                }, e.fromByteArray = function (e) {
                  var t,
                    n,
                    r,
                    o,
                    i = e.length % 3,
                    u = "";
                  function s(e) {
                    return "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".charAt(e);
                  }
                  for (t = 0, r = e.length - i; t < r; t += 3) n = (e[t] << 16) + (e[t + 1] << 8) + e[t + 2], u += s((o = n) >> 18 & 63) + s(o >> 12 & 63) + s(o >> 6 & 63) + s(63 & o);
                  switch (i) {
                    case 1:
                      u = (u += s((n = e[e.length - 1]) >> 2)) + s(n << 4 & 63) + "==";
                      break;
                    case 2:
                      u = (u = (u += s((n = (e[e.length - 2] << 8) + e[e.length - 1]) >> 10)) + s(n >> 4 & 63)) + s(n << 2 & 63) + "=";
                  }
                  return u;
                };
              }(void 0 === f ? this.base64js = {} : f);
            }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/base64-js/lib/b64.js", "/node_modules/gulp-browserify/node_modules/base64-js/lib");
          }, {
            buffer: 3,
            lYpoI2: 11
          }],
          3: [function (O, e, H) {
            !function (e, n, f, r, h, p, g, y, w) {
              var a = O("base64-js"),
                i = O("ieee754");
              function f(e, t, n) {
                if (!(this instanceof f)) return new f(e, t, n);
                var r,
                  o,
                  i,
                  u,
                  s = typeof e;
                if ("base64" === t && "string" == s) for (e = (u = e).trim ? u.trim() : u.replace(/^\s+|\s+$/g, ""); e.length % 4 != 0;) e += "=";
                if ("number" == s) r = j(e);else if ("string" == s) r = f.byteLength(e, t);else {
                  if ("object" != s) throw new Error("First argument needs to be a number, array or string.");
                  r = j(e.length);
                }
                if (f._useTypedArrays ? o = f._augment(new Uint8Array(r)) : ((o = this).length = r, o._isBuffer = !0), f._useTypedArrays && "number" == typeof e.byteLength) o._set(e);else if (C(u = e) || f.isBuffer(u) || u && "object" == typeof u && "number" == typeof u.length) for (i = 0; i < r; i++) f.isBuffer(e) ? o[i] = e.readUInt8(i) : o[i] = e[i];else if ("string" == s) o.write(e, 0, t);else if ("number" == s && !f._useTypedArrays && !n) for (i = 0; i < r; i++) o[i] = 0;
                return o;
              }
              function b(e, t, n, r) {
                return f._charsWritten = c(function (e) {
                  for (var t = [], n = 0; n < e.length; n++) t.push(255 & e.charCodeAt(n));
                  return t;
                }(t), e, n, r);
              }
              function m(e, t, n, r) {
                return f._charsWritten = c(function (e) {
                  for (var t, n, r = [], o = 0; o < e.length; o++) n = e.charCodeAt(o), t = n >> 8, n = n % 256, r.push(n), r.push(t);
                  return r;
                }(t), e, n, r);
              }
              function v(e, t, n) {
                var r = "";
                n = Math.min(e.length, n);
                for (var o = t; o < n; o++) r += String.fromCharCode(e[o]);
                return r;
              }
              function o(e, t, n, r) {
                r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length"));
                var o,
                  r = e.length;
                if (!(r <= t)) return n ? (o = e[t], t + 1 < r && (o |= e[t + 1] << 8)) : (o = e[t] << 8, t + 1 < r && (o |= e[t + 1])), o;
              }
              function u(e, t, n, r) {
                r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length"));
                var o,
                  r = e.length;
                if (!(r <= t)) return n ? (t + 2 < r && (o = e[t + 2] << 16), t + 1 < r && (o |= e[t + 1] << 8), o |= e[t], t + 3 < r && (o += e[t + 3] << 24 >>> 0)) : (t + 1 < r && (o = e[t + 1] << 16), t + 2 < r && (o |= e[t + 2] << 8), t + 3 < r && (o |= e[t + 3]), o += e[t] << 24 >>> 0), o;
              }
              function _(e, t, n, r) {
                if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = o(e, t, n, !0), 32768 & r ? -1 * (65535 - r + 1) : r;
              }
              function E(e, t, n, r) {
                if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = u(e, t, n, !0), 2147483648 & r ? -1 * (4294967295 - r + 1) : r;
              }
              function I(e, t, n, r) {
                return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 3 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 23, 4);
              }
              function A(e, t, n, r) {
                return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 7 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 52, 8);
              }
              function s(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "trying to write beyond buffer length"), Y(t, 65535));
                o = e.length;
                if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 2); i < u; i++) e[n + i] = (t & 255 << 8 * (r ? i : 1 - i)) >>> 8 * (r ? i : 1 - i);
              }
              function l(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "trying to write beyond buffer length"), Y(t, 4294967295));
                o = e.length;
                if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 4); i < u; i++) e[n + i] = t >>> 8 * (r ? i : 3 - i) & 255;
              }
              function B(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "Trying to write beyond buffer length"), F(t, 32767, -32768)), e.length <= n || s(e, 0 <= t ? t : 65535 + t + 1, n, r, o);
              }
              function L(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), F(t, 2147483647, -2147483648)), e.length <= n || l(e, 0 <= t ? t : 4294967295 + t + 1, n, r, o);
              }
              function U(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), D(t, 34028234663852886e22, -34028234663852886e22)), e.length <= n || i.write(e, t, n, r, 23, 4);
              }
              function x(e, t, n, r, o) {
                o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 7 < e.length, "Trying to write beyond buffer length"), D(t, 17976931348623157e292, -17976931348623157e292)), e.length <= n || i.write(e, t, n, r, 52, 8);
              }
              H.Buffer = f, H.SlowBuffer = f, H.INSPECT_MAX_BYTES = 50, f.poolSize = 8192, f._useTypedArrays = function () {
                try {
                  var e = new ArrayBuffer(0),
                    t = new Uint8Array(e);
                  return t.foo = function () {
                    return 42;
                  }, 42 === t.foo() && "function" == typeof t.subarray;
                } catch (e) {
                  return !1;
                }
              }(), f.isEncoding = function (e) {
                switch (String(e).toLowerCase()) {
                  case "hex":
                  case "utf8":
                  case "utf-8":
                  case "ascii":
                  case "binary":
                  case "base64":
                  case "raw":
                  case "ucs2":
                  case "ucs-2":
                  case "utf16le":
                  case "utf-16le":
                    return !0;
                  default:
                    return !1;
                }
              }, f.isBuffer = function (e) {
                return !(null == e || !e._isBuffer);
              }, f.byteLength = function (e, t) {
                var n;
                switch (e += "", t || "utf8") {
                  case "hex":
                    n = e.length / 2;
                    break;
                  case "utf8":
                  case "utf-8":
                    n = T(e).length;
                    break;
                  case "ascii":
                  case "binary":
                  case "raw":
                    n = e.length;
                    break;
                  case "base64":
                    n = M(e).length;
                    break;
                  case "ucs2":
                  case "ucs-2":
                  case "utf16le":
                  case "utf-16le":
                    n = 2 * e.length;
                    break;
                  default:
                    throw new Error("Unknown encoding");
                }
                return n;
              }, f.concat = function (e, t) {
                if (d(C(e), "Usage: Buffer.concat(list, [totalLength])\nlist should be an Array."), 0 === e.length) return new f(0);
                if (1 === e.length) return e[0];
                if ("number" != typeof t) for (o = t = 0; o < e.length; o++) t += e[o].length;
                for (var n = new f(t), r = 0, o = 0; o < e.length; o++) {
                  var i = e[o];
                  i.copy(n, r), r += i.length;
                }
                return n;
              }, f.prototype.write = function (e, t, n, r) {
                isFinite(t) ? isFinite(n) || (r = n, n = void 0) : (a = r, r = t, t = n, n = a), t = Number(t) || 0;
                var o,
                  i,
                  u,
                  s,
                  a = this.length - t;
                switch ((!n || a < (n = Number(n))) && (n = a), r = String(r || "utf8").toLowerCase()) {
                  case "hex":
                    o = function (e, t, n, r) {
                      n = Number(n) || 0;
                      var o = e.length - n;
                      (!r || o < (r = Number(r))) && (r = o), d((o = t.length) % 2 == 0, "Invalid hex string"), o / 2 < r && (r = o / 2);
                      for (var i = 0; i < r; i++) {
                        var u = parseInt(t.substr(2 * i, 2), 16);
                        d(!isNaN(u), "Invalid hex string"), e[n + i] = u;
                      }
                      return f._charsWritten = 2 * i, i;
                    }(this, e, t, n);
                    break;
                  case "utf8":
                  case "utf-8":
                    i = this, u = t, s = n, o = f._charsWritten = c(T(e), i, u, s);
                    break;
                  case "ascii":
                  case "binary":
                    o = b(this, e, t, n);
                    break;
                  case "base64":
                    i = this, u = t, s = n, o = f._charsWritten = c(M(e), i, u, s);
                    break;
                  case "ucs2":
                  case "ucs-2":
                  case "utf16le":
                  case "utf-16le":
                    o = m(this, e, t, n);
                    break;
                  default:
                    throw new Error("Unknown encoding");
                }
                return o;
              }, f.prototype.toString = function (e, t, n) {
                var r,
                  o,
                  i,
                  u,
                  s = this;
                if (e = String(e || "utf8").toLowerCase(), t = Number(t) || 0, (n = void 0 !== n ? Number(n) : s.length) === t) return "";
                switch (e) {
                  case "hex":
                    r = function (e, t, n) {
                      var r = e.length;
                      (!t || t < 0) && (t = 0);
                      (!n || n < 0 || r < n) && (n = r);
                      for (var o = "", i = t; i < n; i++) o += k(e[i]);
                      return o;
                    }(s, t, n);
                    break;
                  case "utf8":
                  case "utf-8":
                    r = function (e, t, n) {
                      var r = "",
                        o = "";
                      n = Math.min(e.length, n);
                      for (var i = t; i < n; i++) e[i] <= 127 ? (r += N(o) + String.fromCharCode(e[i]), o = "") : o += "%" + e[i].toString(16);
                      return r + N(o);
                    }(s, t, n);
                    break;
                  case "ascii":
                  case "binary":
                    r = v(s, t, n);
                    break;
                  case "base64":
                    o = s, u = n, r = 0 === (i = t) && u === o.length ? a.fromByteArray(o) : a.fromByteArray(o.slice(i, u));
                    break;
                  case "ucs2":
                  case "ucs-2":
                  case "utf16le":
                  case "utf-16le":
                    r = function (e, t, n) {
                      for (var r = e.slice(t, n), o = "", i = 0; i < r.length; i += 2) o += String.fromCharCode(r[i] + 256 * r[i + 1]);
                      return o;
                    }(s, t, n);
                    break;
                  default:
                    throw new Error("Unknown encoding");
                }
                return r;
              }, f.prototype.toJSON = function () {
                return {
                  type: "Buffer",
                  data: Array.prototype.slice.call(this._arr || this, 0)
                };
              }, f.prototype.copy = function (e, t, n, r) {
                if (t = t || 0, (r = r || 0 === r ? r : this.length) !== (n = n || 0) && 0 !== e.length && 0 !== this.length) {
                  d(n <= r, "sourceEnd < sourceStart"), d(0 <= t && t < e.length, "targetStart out of bounds"), d(0 <= n && n < this.length, "sourceStart out of bounds"), d(0 <= r && r <= this.length, "sourceEnd out of bounds"), r > this.length && (r = this.length);
                  var o = (r = e.length - t < r - n ? e.length - t + n : r) - n;
                  if (o < 100 || !f._useTypedArrays) for (var i = 0; i < o; i++) e[i + t] = this[i + n];else e._set(this.subarray(n, n + o), t);
                }
              }, f.prototype.slice = function (e, t) {
                var n = this.length;
                if (e = S(e, n, 0), t = S(t, n, n), f._useTypedArrays) return f._augment(this.subarray(e, t));
                for (var r = t - e, o = new f(r, void 0, !0), i = 0; i < r; i++) o[i] = this[i + e];
                return o;
              }, f.prototype.get = function (e) {
                return console.log(".get() is deprecated. Access using array indexes instead."), this.readUInt8(e);
              }, f.prototype.set = function (e, t) {
                return console.log(".set() is deprecated. Access using array indexes instead."), this.writeUInt8(e, t);
              }, f.prototype.readUInt8 = function (e, t) {
                if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return this[e];
              }, f.prototype.readUInt16LE = function (e, t) {
                return o(this, e, !0, t);
              }, f.prototype.readUInt16BE = function (e, t) {
                return o(this, e, !1, t);
              }, f.prototype.readUInt32LE = function (e, t) {
                return u(this, e, !0, t);
              }, f.prototype.readUInt32BE = function (e, t) {
                return u(this, e, !1, t);
              }, f.prototype.readInt8 = function (e, t) {
                if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return 128 & this[e] ? -1 * (255 - this[e] + 1) : this[e];
              }, f.prototype.readInt16LE = function (e, t) {
                return _(this, e, !0, t);
              }, f.prototype.readInt16BE = function (e, t) {
                return _(this, e, !1, t);
              }, f.prototype.readInt32LE = function (e, t) {
                return E(this, e, !0, t);
              }, f.prototype.readInt32BE = function (e, t) {
                return E(this, e, !1, t);
              }, f.prototype.readFloatLE = function (e, t) {
                return I(this, e, !0, t);
              }, f.prototype.readFloatBE = function (e, t) {
                return I(this, e, !1, t);
              }, f.prototype.readDoubleLE = function (e, t) {
                return A(this, e, !0, t);
              }, f.prototype.readDoubleBE = function (e, t) {
                return A(this, e, !1, t);
              }, f.prototype.writeUInt8 = function (e, t, n) {
                n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "trying to write beyond buffer length"), Y(e, 255)), t >= this.length || (this[t] = e);
              }, f.prototype.writeUInt16LE = function (e, t, n) {
                s(this, e, t, !0, n);
              }, f.prototype.writeUInt16BE = function (e, t, n) {
                s(this, e, t, !1, n);
              }, f.prototype.writeUInt32LE = function (e, t, n) {
                l(this, e, t, !0, n);
              }, f.prototype.writeUInt32BE = function (e, t, n) {
                l(this, e, t, !1, n);
              }, f.prototype.writeInt8 = function (e, t, n) {
                n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "Trying to write beyond buffer length"), F(e, 127, -128)), t >= this.length || (0 <= e ? this.writeUInt8(e, t, n) : this.writeUInt8(255 + e + 1, t, n));
              }, f.prototype.writeInt16LE = function (e, t, n) {
                B(this, e, t, !0, n);
              }, f.prototype.writeInt16BE = function (e, t, n) {
                B(this, e, t, !1, n);
              }, f.prototype.writeInt32LE = function (e, t, n) {
                L(this, e, t, !0, n);
              }, f.prototype.writeInt32BE = function (e, t, n) {
                L(this, e, t, !1, n);
              }, f.prototype.writeFloatLE = function (e, t, n) {
                U(this, e, t, !0, n);
              }, f.prototype.writeFloatBE = function (e, t, n) {
                U(this, e, t, !1, n);
              }, f.prototype.writeDoubleLE = function (e, t, n) {
                x(this, e, t, !0, n);
              }, f.prototype.writeDoubleBE = function (e, t, n) {
                x(this, e, t, !1, n);
              }, f.prototype.fill = function (e, t, n) {
                if (t = t || 0, n = n || this.length, d("number" == typeof (e = "string" == typeof (e = e || 0) ? e.charCodeAt(0) : e) && !isNaN(e), "value is not a number"), d(t <= n, "end < start"), n !== t && 0 !== this.length) {
                  d(0 <= t && t < this.length, "start out of bounds"), d(0 <= n && n <= this.length, "end out of bounds");
                  for (var r = t; r < n; r++) this[r] = e;
                }
              }, f.prototype.inspect = function () {
                for (var e = [], t = this.length, n = 0; n < t; n++) if (e[n] = k(this[n]), n === H.INSPECT_MAX_BYTES) {
                  e[n + 1] = "...";
                  break;
                }
                return "<Buffer " + e.join(" ") + ">";
              }, f.prototype.toArrayBuffer = function () {
                if ("undefined" == typeof Uint8Array) throw new Error("Buffer.toArrayBuffer not supported in this browser");
                if (f._useTypedArrays) return new f(this).buffer;
                for (var e = new Uint8Array(this.length), t = 0, n = e.length; t < n; t += 1) e[t] = this[t];
                return e.buffer;
              };
              var t = f.prototype;
              function S(e, t, n) {
                return "number" != typeof e ? n : t <= (e = ~~e) ? t : 0 <= e || 0 <= (e += t) ? e : 0;
              }
              function j(e) {
                return (e = ~~Math.ceil(+e)) < 0 ? 0 : e;
              }
              function C(e) {
                return (Array.isArray || function (e) {
                  return "[object Array]" === Object.prototype.toString.call(e);
                })(e);
              }
              function k(e) {
                return e < 16 ? "0" + e.toString(16) : e.toString(16);
              }
              function T(e) {
                for (var t = [], n = 0; n < e.length; n++) {
                  var r = e.charCodeAt(n);
                  if (r <= 127) t.push(e.charCodeAt(n));else for (var o = n, i = (55296 <= r && r <= 57343 && n++, encodeURIComponent(e.slice(o, n + 1)).substr(1).split("%")), u = 0; u < i.length; u++) t.push(parseInt(i[u], 16));
                }
                return t;
              }
              function M(e) {
                return a.toByteArray(e);
              }
              function c(e, t, n, r) {
                for (var o = 0; o < r && !(o + n >= t.length || o >= e.length); o++) t[o + n] = e[o];
                return o;
              }
              function N(e) {
                try {
                  return decodeURIComponent(e);
                } catch (e) {
                  return String.fromCharCode(65533);
                }
              }
              function Y(e, t) {
                d("number" == typeof e, "cannot write a non-number as a number"), d(0 <= e, "specified a negative value for writing an unsigned value"), d(e <= t, "value is larger than maximum value for type"), d(Math.floor(e) === e, "value has a fractional component");
              }
              function F(e, t, n) {
                d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value"), d(Math.floor(e) === e, "value has a fractional component");
              }
              function D(e, t, n) {
                d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value");
              }
              function d(e, t) {
                if (!e) throw new Error(t || "Failed assertion");
              }
              f._augment = function (e) {
                return e._isBuffer = !0, e._get = e.get, e._set = e.set, e.get = t.get, e.set = t.set, e.write = t.write, e.toString = t.toString, e.toLocaleString = t.toString, e.toJSON = t.toJSON, e.copy = t.copy, e.slice = t.slice, e.readUInt8 = t.readUInt8, e.readUInt16LE = t.readUInt16LE, e.readUInt16BE = t.readUInt16BE, e.readUInt32LE = t.readUInt32LE, e.readUInt32BE = t.readUInt32BE, e.readInt8 = t.readInt8, e.readInt16LE = t.readInt16LE, e.readInt16BE = t.readInt16BE, e.readInt32LE = t.readInt32LE, e.readInt32BE = t.readInt32BE, e.readFloatLE = t.readFloatLE, e.readFloatBE = t.readFloatBE, e.readDoubleLE = t.readDoubleLE, e.readDoubleBE = t.readDoubleBE, e.writeUInt8 = t.writeUInt8, e.writeUInt16LE = t.writeUInt16LE, e.writeUInt16BE = t.writeUInt16BE, e.writeUInt32LE = t.writeUInt32LE, e.writeUInt32BE = t.writeUInt32BE, e.writeInt8 = t.writeInt8, e.writeInt16LE = t.writeInt16LE, e.writeInt16BE = t.writeInt16BE, e.writeInt32LE = t.writeInt32LE, e.writeInt32BE = t.writeInt32BE, e.writeFloatLE = t.writeFloatLE, e.writeFloatBE = t.writeFloatBE, e.writeDoubleLE = t.writeDoubleLE, e.writeDoubleBE = t.writeDoubleBE, e.fill = t.fill, e.inspect = t.inspect, e.toArrayBuffer = t.toArrayBuffer, e;
              };
            }.call(this, O("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, O("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/buffer/index.js", "/node_modules/gulp-browserify/node_modules/buffer");
          }, {
            "base64-js": 2,
            buffer: 3,
            ieee754: 10,
            lYpoI2: 11
          }],
          4: [function (c, d, e) {
            !function (e, t, a, n, r, o, i, u, s) {
              var a = c("buffer").Buffer,
                f = 4,
                l = new a(f);
              l.fill(0);
              d.exports = {
                hash: function (e, t, n, r) {
                  for (var o = t(function (e, t) {
                      e.length % f != 0 && (n = e.length + (f - e.length % f), e = a.concat([e, l], n));
                      for (var n, r = [], o = t ? e.readInt32BE : e.readInt32LE, i = 0; i < e.length; i += f) r.push(o.call(e, i));
                      return r;
                    }(e = a.isBuffer(e) ? e : new a(e), r), 8 * e.length), t = r, i = new a(n), u = t ? i.writeInt32BE : i.writeInt32LE, s = 0; s < o.length; s++) u.call(i, o[s], 4 * s, !0);
                  return i;
                }
              };
            }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/helpers.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            buffer: 3,
            lYpoI2: 11
          }],
          5: [function (v, e, _) {
            !function (l, c, u, d, h, p, g, y, w) {
              var u = v("buffer").Buffer,
                e = v("./sha"),
                t = v("./sha256"),
                n = v("./rng"),
                b = {
                  sha1: e,
                  sha256: t,
                  md5: v("./md5")
                },
                s = 64,
                a = new u(s);
              function r(e, n) {
                var r = b[e = e || "sha1"],
                  o = [];
                return r || i("algorithm:", e, "is not yet supported"), {
                  update: function (e) {
                    return u.isBuffer(e) || (e = new u(e)), o.push(e), e.length, this;
                  },
                  digest: function (e) {
                    var t = u.concat(o),
                      t = n ? function (e, t, n) {
                        u.isBuffer(t) || (t = new u(t)), u.isBuffer(n) || (n = new u(n)), t.length > s ? t = e(t) : t.length < s && (t = u.concat([t, a], s));
                        for (var r = new u(s), o = new u(s), i = 0; i < s; i++) r[i] = 54 ^ t[i], o[i] = 92 ^ t[i];
                        return n = e(u.concat([r, n])), e(u.concat([o, n]));
                      }(r, n, t) : r(t);
                    return o = null, e ? t.toString(e) : t;
                  }
                };
              }
              function i() {
                var e = [].slice.call(arguments).join(" ");
                throw new Error([e, "we accept pull requests", "http://github.com/dominictarr/crypto-browserify"].join("\n"));
              }
              a.fill(0), _.createHash = function (e) {
                return r(e);
              }, _.createHmac = r, _.randomBytes = function (e, t) {
                if (!t || !t.call) return new u(n(e));
                try {
                  t.call(this, void 0, new u(n(e)));
                } catch (e) {
                  t(e);
                }
              };
              var o,
                f = ["createCredentials", "createCipher", "createCipheriv", "createDecipher", "createDecipheriv", "createSign", "createVerify", "createDiffieHellman", "pbkdf2"],
                m = function (e) {
                  _[e] = function () {
                    i("sorry,", e, "is not implemented yet");
                  };
                };
              for (o in f) m(f[o]);
            }.call(this, v("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, v("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/index.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            "./md5": 6,
            "./rng": 7,
            "./sha": 8,
            "./sha256": 9,
            buffer: 3,
            lYpoI2: 11
          }],
          6: [function (w, b, e) {
            !function (e, r, o, i, u, a, f, l, y) {
              var t = w("./helpers");
              function n(e, t) {
                e[t >> 5] |= 128 << t % 32, e[14 + (t + 64 >>> 9 << 4)] = t;
                for (var n = 1732584193, r = -271733879, o = -1732584194, i = 271733878, u = 0; u < e.length; u += 16) {
                  var s = n,
                    a = r,
                    f = o,
                    l = i,
                    n = c(n, r, o, i, e[u + 0], 7, -680876936),
                    i = c(i, n, r, o, e[u + 1], 12, -389564586),
                    o = c(o, i, n, r, e[u + 2], 17, 606105819),
                    r = c(r, o, i, n, e[u + 3], 22, -1044525330);
                  n = c(n, r, o, i, e[u + 4], 7, -176418897), i = c(i, n, r, o, e[u + 5], 12, 1200080426), o = c(o, i, n, r, e[u + 6], 17, -1473231341), r = c(r, o, i, n, e[u + 7], 22, -45705983), n = c(n, r, o, i, e[u + 8], 7, 1770035416), i = c(i, n, r, o, e[u + 9], 12, -1958414417), o = c(o, i, n, r, e[u + 10], 17, -42063), r = c(r, o, i, n, e[u + 11], 22, -1990404162), n = c(n, r, o, i, e[u + 12], 7, 1804603682), i = c(i, n, r, o, e[u + 13], 12, -40341101), o = c(o, i, n, r, e[u + 14], 17, -1502002290), n = d(n, r = c(r, o, i, n, e[u + 15], 22, 1236535329), o, i, e[u + 1], 5, -165796510), i = d(i, n, r, o, e[u + 6], 9, -1069501632), o = d(o, i, n, r, e[u + 11], 14, 643717713), r = d(r, o, i, n, e[u + 0], 20, -373897302), n = d(n, r, o, i, e[u + 5], 5, -701558691), i = d(i, n, r, o, e[u + 10], 9, 38016083), o = d(o, i, n, r, e[u + 15], 14, -660478335), r = d(r, o, i, n, e[u + 4], 20, -405537848), n = d(n, r, o, i, e[u + 9], 5, 568446438), i = d(i, n, r, o, e[u + 14], 9, -1019803690), o = d(o, i, n, r, e[u + 3], 14, -187363961), r = d(r, o, i, n, e[u + 8], 20, 1163531501), n = d(n, r, o, i, e[u + 13], 5, -1444681467), i = d(i, n, r, o, e[u + 2], 9, -51403784), o = d(o, i, n, r, e[u + 7], 14, 1735328473), n = h(n, r = d(r, o, i, n, e[u + 12], 20, -1926607734), o, i, e[u + 5], 4, -378558), i = h(i, n, r, o, e[u + 8], 11, -2022574463), o = h(o, i, n, r, e[u + 11], 16, 1839030562), r = h(r, o, i, n, e[u + 14], 23, -35309556), n = h(n, r, o, i, e[u + 1], 4, -1530992060), i = h(i, n, r, o, e[u + 4], 11, 1272893353), o = h(o, i, n, r, e[u + 7], 16, -155497632), r = h(r, o, i, n, e[u + 10], 23, -1094730640), n = h(n, r, o, i, e[u + 13], 4, 681279174), i = h(i, n, r, o, e[u + 0], 11, -358537222), o = h(o, i, n, r, e[u + 3], 16, -722521979), r = h(r, o, i, n, e[u + 6], 23, 76029189), n = h(n, r, o, i, e[u + 9], 4, -640364487), i = h(i, n, r, o, e[u + 12], 11, -421815835), o = h(o, i, n, r, e[u + 15], 16, 530742520), n = p(n, r = h(r, o, i, n, e[u + 2], 23, -995338651), o, i, e[u + 0], 6, -198630844), i = p(i, n, r, o, e[u + 7], 10, 1126891415), o = p(o, i, n, r, e[u + 14], 15, -1416354905), r = p(r, o, i, n, e[u + 5], 21, -57434055), n = p(n, r, o, i, e[u + 12], 6, 1700485571), i = p(i, n, r, o, e[u + 3], 10, -1894986606), o = p(o, i, n, r, e[u + 10], 15, -1051523), r = p(r, o, i, n, e[u + 1], 21, -2054922799), n = p(n, r, o, i, e[u + 8], 6, 1873313359), i = p(i, n, r, o, e[u + 15], 10, -30611744), o = p(o, i, n, r, e[u + 6], 15, -1560198380), r = p(r, o, i, n, e[u + 13], 21, 1309151649), n = p(n, r, o, i, e[u + 4], 6, -145523070), i = p(i, n, r, o, e[u + 11], 10, -1120210379), o = p(o, i, n, r, e[u + 2], 15, 718787259), r = p(r, o, i, n, e[u + 9], 21, -343485551), n = g(n, s), r = g(r, a), o = g(o, f), i = g(i, l);
                }
                return Array(n, r, o, i);
              }
              function s(e, t, n, r, o, i) {
                return g((t = g(g(t, e), g(r, i))) << o | t >>> 32 - o, n);
              }
              function c(e, t, n, r, o, i, u) {
                return s(t & n | ~t & r, e, t, o, i, u);
              }
              function d(e, t, n, r, o, i, u) {
                return s(t & r | n & ~r, e, t, o, i, u);
              }
              function h(e, t, n, r, o, i, u) {
                return s(t ^ n ^ r, e, t, o, i, u);
              }
              function p(e, t, n, r, o, i, u) {
                return s(n ^ (t | ~r), e, t, o, i, u);
              }
              function g(e, t) {
                var n = (65535 & e) + (65535 & t);
                return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
              }
              b.exports = function (e) {
                return t.hash(e, n, 16);
              };
            }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/md5.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            "./helpers": 4,
            buffer: 3,
            lYpoI2: 11
          }],
          7: [function (e, l, t) {
            !function (e, t, n, r, o, i, u, s, f) {
              l.exports = function (e) {
                for (var t, n = new Array(e), r = 0; r < e; r++) 0 == (3 & r) && (t = 4294967296 * Math.random()), n[r] = t >>> ((3 & r) << 3) & 255;
                return n;
              };
            }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/rng.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            buffer: 3,
            lYpoI2: 11
          }],
          8: [function (c, d, e) {
            !function (e, t, n, r, o, s, a, f, l) {
              var i = c("./helpers");
              function u(l, c) {
                l[c >> 5] |= 128 << 24 - c % 32, l[15 + (c + 64 >> 9 << 4)] = c;
                for (var e, t, n, r = Array(80), o = 1732584193, i = -271733879, u = -1732584194, s = 271733878, d = -1009589776, h = 0; h < l.length; h += 16) {
                  for (var p = o, g = i, y = u, w = s, b = d, a = 0; a < 80; a++) {
                    r[a] = a < 16 ? l[h + a] : v(r[a - 3] ^ r[a - 8] ^ r[a - 14] ^ r[a - 16], 1);
                    var f = m(m(v(o, 5), (f = i, t = u, n = s, (e = a) < 20 ? f & t | ~f & n : !(e < 40) && e < 60 ? f & t | f & n | t & n : f ^ t ^ n)), m(m(d, r[a]), (e = a) < 20 ? 1518500249 : e < 40 ? 1859775393 : e < 60 ? -1894007588 : -899497514)),
                      d = s,
                      s = u,
                      u = v(i, 30),
                      i = o,
                      o = f;
                  }
                  o = m(o, p), i = m(i, g), u = m(u, y), s = m(s, w), d = m(d, b);
                }
                return Array(o, i, u, s, d);
              }
              function m(e, t) {
                var n = (65535 & e) + (65535 & t);
                return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
              }
              function v(e, t) {
                return e << t | e >>> 32 - t;
              }
              d.exports = function (e) {
                return i.hash(e, u, 20, !0);
              };
            }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            "./helpers": 4,
            buffer: 3,
            lYpoI2: 11
          }],
          9: [function (c, d, e) {
            !function (e, t, n, r, u, s, a, f, l) {
              function b(e, t) {
                var n = (65535 & e) + (65535 & t);
                return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
              }
              function o(e, l) {
                var c,
                  d = new Array(1116352408, 1899447441, 3049323471, 3921009573, 961987163, 1508970993, 2453635748, 2870763221, 3624381080, 310598401, 607225278, 1426881987, 1925078388, 2162078206, 2614888103, 3248222580, 3835390401, 4022224774, 264347078, 604807628, 770255983, 1249150122, 1555081692, 1996064986, 2554220882, 2821834349, 2952996808, 3210313671, 3336571891, 3584528711, 113926993, 338241895, 666307205, 773529912, 1294757372, 1396182291, 1695183700, 1986661051, 2177026350, 2456956037, 2730485921, 2820302411, 3259730800, 3345764771, 3516065817, 3600352804, 4094571909, 275423344, 430227734, 506948616, 659060556, 883997877, 958139571, 1322822218, 1537002063, 1747873779, 1955562222, 2024104815, 2227730452, 2361852424, 2428436474, 2756734187, 3204031479, 3329325298),
                  t = new Array(1779033703, 3144134277, 1013904242, 2773480762, 1359893119, 2600822924, 528734635, 1541459225),
                  n = new Array(64);
                e[l >> 5] |= 128 << 24 - l % 32, e[15 + (l + 64 >> 9 << 4)] = l;
                for (var r, o, h = 0; h < e.length; h += 16) {
                  for (var i = t[0], u = t[1], s = t[2], p = t[3], a = t[4], g = t[5], y = t[6], w = t[7], f = 0; f < 64; f++) n[f] = f < 16 ? e[f + h] : b(b(b((o = n[f - 2], m(o, 17) ^ m(o, 19) ^ v(o, 10)), n[f - 7]), (o = n[f - 15], m(o, 7) ^ m(o, 18) ^ v(o, 3))), n[f - 16]), c = b(b(b(b(w, m(o = a, 6) ^ m(o, 11) ^ m(o, 25)), a & g ^ ~a & y), d[f]), n[f]), r = b(m(r = i, 2) ^ m(r, 13) ^ m(r, 22), i & u ^ i & s ^ u & s), w = y, y = g, g = a, a = b(p, c), p = s, s = u, u = i, i = b(c, r);
                  t[0] = b(i, t[0]), t[1] = b(u, t[1]), t[2] = b(s, t[2]), t[3] = b(p, t[3]), t[4] = b(a, t[4]), t[5] = b(g, t[5]), t[6] = b(y, t[6]), t[7] = b(w, t[7]);
                }
                return t;
              }
              var i = c("./helpers"),
                m = function (e, t) {
                  return e >>> t | e << 32 - t;
                },
                v = function (e, t) {
                  return e >>> t;
                };
              d.exports = function (e) {
                return i.hash(e, o, 32, !0);
              };
            }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha256.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
          }, {
            "./helpers": 4,
            buffer: 3,
            lYpoI2: 11
          }],
          10: [function (e, t, f) {
            !function (e, t, n, r, o, i, u, s, a) {
              f.read = function (e, t, n, r, o) {
                var i,
                  u,
                  l = 8 * o - r - 1,
                  c = (1 << l) - 1,
                  d = c >> 1,
                  s = -7,
                  a = n ? o - 1 : 0,
                  f = n ? -1 : 1,
                  o = e[t + a];
                for (a += f, i = o & (1 << -s) - 1, o >>= -s, s += l; 0 < s; i = 256 * i + e[t + a], a += f, s -= 8);
                for (u = i & (1 << -s) - 1, i >>= -s, s += r; 0 < s; u = 256 * u + e[t + a], a += f, s -= 8);
                if (0 === i) i = 1 - d;else {
                  if (i === c) return u ? NaN : 1 / 0 * (o ? -1 : 1);
                  u += Math.pow(2, r), i -= d;
                }
                return (o ? -1 : 1) * u * Math.pow(2, i - r);
              }, f.write = function (e, t, l, n, r, c) {
                var o,
                  i,
                  u = 8 * c - r - 1,
                  s = (1 << u) - 1,
                  a = s >> 1,
                  d = 23 === r ? Math.pow(2, -24) - Math.pow(2, -77) : 0,
                  f = n ? 0 : c - 1,
                  h = n ? 1 : -1,
                  c = t < 0 || 0 === t && 1 / t < 0 ? 1 : 0;
                for (t = Math.abs(t), isNaN(t) || t === 1 / 0 ? (i = isNaN(t) ? 1 : 0, o = s) : (o = Math.floor(Math.log(t) / Math.LN2), t * (n = Math.pow(2, -o)) < 1 && (o--, n *= 2), 2 <= (t += 1 <= o + a ? d / n : d * Math.pow(2, 1 - a)) * n && (o++, n /= 2), s <= o + a ? (i = 0, o = s) : 1 <= o + a ? (i = (t * n - 1) * Math.pow(2, r), o += a) : (i = t * Math.pow(2, a - 1) * Math.pow(2, r), o = 0)); 8 <= r; e[l + f] = 255 & i, f += h, i /= 256, r -= 8);
                for (o = o << r | i, u += r; 0 < u; e[l + f] = 255 & o, f += h, o /= 256, u -= 8);
                e[l + f - h] |= 128 * c;
              };
            }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/ieee754/index.js", "/node_modules/gulp-browserify/node_modules/ieee754");
          }, {
            buffer: 3,
            lYpoI2: 11
          }],
          11: [function (e, h, t) {
            !function (e, t, n, r, o, f, l, c, d) {
              var i, u, s;
              function a() {}
              (e = h.exports = {}).nextTick = (u = "undefined" != typeof window && window.setImmediate, s = "undefined" != typeof window && window.postMessage && window.addEventListener, u ? function (e) {
                return window.setImmediate(e);
              } : s ? (i = [], window.addEventListener("message", function (e) {
                var t = e.source;
                t !== window && null !== t || "process-tick" !== e.data || (e.stopPropagation(), 0 < i.length && i.shift()());
              }, !0), function (e) {
                i.push(e), window.postMessage("process-tick", "*");
              }) : function (e) {
                setTimeout(e, 0);
              }), e.title = "browser", e.browser = !0, e.env = {}, e.argv = [], e.on = a, e.addListener = a, e.once = a, e.off = a, e.removeListener = a, e.removeAllListeners = a, e.emit = a, e.binding = function (e) {
                throw new Error("process.binding is not supported");
              }, e.cwd = function () {
                return "/";
              }, e.chdir = function (e) {
                throw new Error("process.chdir is not supported");
              };
            }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/process/browser.js", "/node_modules/gulp-browserify/node_modules/process");
          }, {
            buffer: 3,
            lYpoI2: 11
          }]
        }, {}, [1])(1);
      });
    })(object_hash);
    var hash = object_hash.exports;

    function getFromToIndex(xs, range) {
      let {
        from,
        to
      } = range;
      if (from === undefined) {
        from = xs[0];
      }
      if (to === undefined) {
        to = xs.at(-1);
      }
      return {
        fromIndex: xFindClosestIndex(xs, from),
        toIndex: xFindClosestIndex(xs, to)
      };
    }

    function integration(matrix, normalizedTarget, range = {}) {
      let fromToIndex = getFromToIndex(normalizedTarget.x, range);
      let targetValue = xSum(normalizedTarget.y, fromToIndex);
      let values = matrix.map(row => xSum(row, fromToIndex));
      for (let i = 0; i < matrix.length; i++) {
        let factor = targetValue / values[i];
        matrix[i] = xMultiply(matrix[i], factor);
      }
    }

    function max(matrix, normalizedTarget, range = {}) {
      let fromToIndex = getFromToIndex(normalizedTarget.x, range);
      let targetValue = xMaxValue(normalizedTarget.y, fromToIndex);
      let values = matrix.map(row => xMaxValue(row, fromToIndex));
      for (let i = 0; i < matrix.length; i++) {
        let factor = targetValue / values[i];
        matrix[i] = xMultiply(matrix[i], factor);
      }
    }

    function min(matrix, normalizedTarget, range = {}) {
      let fromToIndex = getFromToIndex(normalizedTarget.x, range);
      let targetValue = xMinValue(normalizedTarget.y, fromToIndex);
      let values = matrix.map(row => xMinValue(row, fromToIndex));
      for (let i = 0; i < matrix.length; i++) {
        let factor = targetValue / values[i];
        matrix[i] = xMultiply(matrix[i], factor);
      }
    }

    function minMax(matrix, normalizedTarget, range = {}) {
      let fromToIndex = getFromToIndex(normalizedTarget.x, range);
      let targetValue = {
        min: xMinValue(normalizedTarget.y, fromToIndex),
        max: xMaxValue(normalizedTarget.y, fromToIndex)
      };
      let deltaTarget = targetValue.max - targetValue.min;
      let minTarget = targetValue.min;
      let values = matrix.map(row => {
        return {
          min: xMinValue(row, fromToIndex),
          max: xMaxValue(row, fromToIndex)
        };
      });
      for (let i = 0; i < matrix.length; i++) {
        let deltaSource = values[i].max - values[i].min;
        let minSource = values[i].min;
        let newData = [];
        for (let j = 0; j < normalizedTarget.y.length; j++) {
          newData.push((matrix[i][j] - minSource) / deltaSource * deltaTarget + minTarget);
        }
        matrix[i] = newData;
      }
    }

    /**
     * Allows to calculate relative intensity between normalized spectra
     * @param {SpectraProcessor} spectraProcessor
     * @param {object}   [options={}] - scale spectra based on various parameters
     * @param {Array}    [options.ids] - ids of selected spectra
     * @param {Array}    [options.filters=[]] - Array of object containing {name:'', options:''}
     * @param {object}   [options.scale={}] - object containing the options for rescaling
     * @param {string}   [options.scale.targetID=spectra[0].id]
     * @param {string}   [options.scale.method='max'] - min, max, integration, minMax
     * @param {Array}    [options.scale.range] - from - to to apply the method and rescale
     * @param {boolean}  [options.scale.relative=false]
     * @param {Array}    [options.ranges] - Array of object containing {from:'', to:'', label:''}
     * @param {Array}    [options.calculations] - Array of object containing {label:'', formula:''}
     * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[], ranges:[object] }
     */

    let cache = {};
    function getPostProcessedData(spectraProcessor, options = {}) {
      /**
       * could implement a cache if all the options are identical and the normalized data is identical as well
       * in order ot check if the normalized data are identical we should check if the normalized array of all the spectra are identical
       * Because we don't make in-place modification when creating normalized array we can check if the 'pointer' to the object
       * is identical
       */

      const optionsHash = hash(options);
      if (!spectraProcessor.spectra || !spectraProcessor.spectra[0]) return {};
      const {
        scale = {},
        ids,
        ranges,
        calculations,
        filters = []
      } = options;
      const {
        range,
        targetID,
        relative,
        method = ''
      } = scale;
      let spectra = spectraProcessor.getSpectra(ids);

      // are we able to reuse the cache ?
      // we can if the normalized information didn't change and optionsHash is the same
      if (cache.optionsHash === optionsHash) {
        let validCache = true;
        for (let spectrum of spectra) {
          if (!cache.weakMap.get(spectrum.normalized)) validCache = false;
        }
        if (validCache) return cache;
      }
      const weakMap = new WeakMap();
      for (let spectrum of spectra) {
        weakMap.set(spectrum.normalized, true);
      }
      let normalizedData = getNormalizedData(spectra);
      for (let filter of filters) {
        switch (filter.name) {
          case 'pqn':
            {
              normalizedData.matrix = matrixPQN(normalizedData.matrix, filter.options).data;
              break;
            }
          case 'centerMean':
            {
              normalizedData.matrix = matrixCenterZMean(normalizedData.matrix);
              break;
            }
          case 'rescale':
            {
              normalizedData.matrix = matrixZRescale(normalizedData.matrix, filter.options);
              break;
            }
          case '':
          case undefined:
            break;
          default:
            throw new Error(`Unknown matrix filter name: ${filter.name}`);
        }
      }
      let normalizedTarget = targetID ? spectraProcessor.getSpectrum(targetID).normalized : spectraProcessor.spectra[0].normalized;
      if (method) {
        switch (method.toLowerCase()) {
          case 'min':
            min(normalizedData.matrix, normalizedTarget, range);
            break;
          case 'max':
            max(normalizedData.matrix, normalizedTarget, range);
            break;
          case 'minmax':
            minMax(normalizedData.matrix, normalizedTarget, range);
            break;
          case 'integration':
            integration(normalizedData.matrix, normalizedTarget, range);
            break;
          default:
            throw new Error(`getPostProcessedData: unknown method: ${method}`);
        }
      }
      if (relative) {
        for (let i = 0; i < normalizedData.matrix.length; i++) {
          normalizedData.matrix[i] = xSubtract(normalizedData.matrix[i], normalizedTarget.y);
        }
      }
      if (ranges) {
        normalizedData.ranges = [];
        for (let i = 0; i < normalizedData.matrix.length; i++) {
          let rangesCopy = structuredClone(ranges);
          let yNormalized = normalizedData.matrix[i];
          let resultRanges = {};
          normalizedData.ranges.push(resultRanges);
          for (let currentRange of rangesCopy) {
            if (currentRange.label) {
              let fromToIndex = getFromToIndex(normalizedTarget.x, currentRange);
              let deltaX = normalizedTarget.x[1] - normalizedTarget.x[0];
              currentRange.integration = xSum(yNormalized, fromToIndex) * deltaX;
              currentRange.maxPoint = xyMaxYPoint({
                x: normalizedData.x,
                y: yNormalized
              }, fromToIndex);
              resultRanges[currentRange.label] = currentRange;
            }
          }
        }
      }
      if (calculations && normalizedData.ranges) {
        normalizedData.calculations = normalizedData.ranges.map(() => {
          return {};
        });
        const parameters = Object.keys(normalizedData.ranges[0]);
        for (let calculation of calculations) {
          // eslint-disable-next-line no-new-func
          const callback = new Function(...parameters, `return ${calculation.formula}`);
          for (let i = 0; i < normalizedData.ranges.length; i++) {
            let oneRanges = normalizedData.ranges[i];
            let values = parameters.map(key => oneRanges[key].integration);
            normalizedData.calculations[i][calculation.label] = callback(...values);
          }
        }
      }
      cache = {
        ...normalizedData,
        optionsHash,
        weakMap
      };
      return cache;
    }

    /**
     * @private
     * @param {SpectraProcessor} spectraProcessor
     * @param {object} [options={}]
     * @param {string} [options.fs='\t'] - field separator
     * @param {string} [options.rs='\n'] - record (line) separator
     * @param {object} [options.postProcessing={}] - post processing options
     */

    function getPostProcessedText(spectraProcessor, options = {}) {
      let {
        fs = '\t',
        rs = '\n',
        postProcessing: postProcessingOptions = {}
      } = options;
      return convertToText(getPostProcessedData(spectraProcessor, postProcessingOptions), {
        rs,
        fs
      });
    }

    /**
     *
     * @private
     * @param {Spectrum} spectrum
     * @param {object} [options={}]
     * @param {object} [options.xFilter={}]
     * @param {number} [options.xFilter.from]
     * @param {number} [options.xFilter.to]
     * @param {Array} [options.xFilter.exclusions=[]]
     * @param {number} [options.yFactor=1]
     */

    function getData(spectrum, options = {}) {
      const {
        xFilter = {},
        yFactor = 1
      } = options;
      let data = {
        x: spectrum.x,
        y: spectrum.y
      };
      if (xFilter) {
        data = xyFilterX(spectrum, xFilter);
      }
      if (yFactor && yFactor !== 1) {
        data.y = data.y.map(y => y * yFactor);
      }
      return data;
    }

    /**
     * Center the mean
     * @param data
     */
    function centerMean(data) {
      const {
        y
      } = data;
      const mean = xMean(y);
      for (let i = 0; i < y.length; i++) {
        y[i] -= mean;
      }
      return {
        data
      };
    }

    /**
     * Center the median
     * @param data
     */
    function centerMedian(data) {
      const {
        y
      } = data;
      const median = xMedian(y);
      for (let i = 0; i < y.length; i++) {
        y[i] -= median;
      }
      return {
        data
      };
    }

    /**
     * Filter that allows to
     * @param data
     * @param options
     */
    function fromTo(data, options = {}) {
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(data.x, options);
      return {
        data: {
          x: data.x.subarray(fromIndex, toIndex + 1),
          y: data.y.subarray(fromIndex, toIndex + 1)
        }
      };
    }

    /**
     * Norm the Y values
     * @param data
     */
    function normed(data, options = {}) {
      xNormed(data.y, {
        ...options,
        output: data.y
      });
      return {
        data
      };
    }

    /**
     * Center the mean
     * @param data
     */
    function divideBySD(data) {
      const {
        y
      } = data;
      const sd = xStandardDeviation(y);
      for (let i = 0; i < y.length; i++) {
        y[i] /= sd;
      }
      return {
        data
      };
    }

    /**
     * Center the mean
     * @param data
     */
    function rescale(data, options = {}) {
      xRescale(data.y, {
        ...options,
        output: data.y
      });
      return {
        data
      };
    }

    /**
     * Filter that allows to
     * @param data
     */
    function paretoNormalization(data) {
      return {
        data: {
          x: data.x,
          y: xParetoNormalization(data.y)
        }
      };
    }

    // Based on https://github.com/scijs/cholesky-solve

    /*
    The MIT License (MIT)

    Copyright (c) 2013 Eric Arnebäck

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
    */

    function ldlSymbolic(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n + 1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Lp /* output of size n + 1, not defined on input */, Parent /* output of size n, not defined on input */, Lnz /* output of size n, not defined on input */, Flag /* workspace of size n, not defn. on input or output */) {
      let i, k, p, kk, p2;
      for (k = 0; k < n; k++) {
        /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
        Parent[k] = -1; /* parent of k is not yet known */
        Flag[k] = k; /* mark node k as visited */
        Lnz[k] = 0; /* count of nonzeros in column k of L */
        kk = k; /* kth original, or permuted, column */
        p2 = Ap[kk + 1];
        for (p = Ap[kk]; p < p2; p++) {
          /* A (i,k) is nonzero (original or permuted A) */
          i = Ai[p];
          if (i < k) {
            /* follow path from i to root of etree, stop at flagged node */
            for (; Flag[i] !== k; i = Parent[i]) {
              /* find parent of i if not yet determined */
              if (Parent[i] === -1) Parent[i] = k;
              Lnz[i]++; /* L (k,i) is nonzero */
              Flag[i] = k; /* mark i as visited */
            }
          }
        }
      }
      /* construct Lp index array from Lnz column counts */
      Lp[0] = 0;
      for (k = 0; k < n; k++) {
        Lp[k + 1] = Lp[k] + Lnz[k];
      }
    }
    function ldlNumeric(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n+1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Ax /* input of size nz=Ap[n], not modified */, Lp /* input of size n+1, not modified */, Parent /* input of size n, not modified */, Lnz /* output of size n, not defn. on input */, Li /* output of size lnz=Lp[n], not defined on input */, Lx /* output of size lnz=Lp[n], not defined on input */, D /* output of size n, not defined on input */, Y /* workspace of size n, not defn. on input or output */, Pattern /* workspace of size n, not defn. on input or output */, Flag /* workspace of size n, not defn. on input or output */) {
      let yi, lKi;
      let i, k, p, kk, p2, len, top;
      for (k = 0; k < n; k++) {
        /* compute nonzero Pattern of kth row of L, in topological order */
        Y[k] = 0.0; /* Y(0:k) is now all zero */
        top = n; /* stack for pattern is empty */
        Flag[k] = k; /* mark node k as visited */
        Lnz[k] = 0; /* count of nonzeros in column k of L */
        kk = k; /* kth original, or permuted, column */
        p2 = Ap[kk + 1];
        for (p = Ap[kk]; p < p2; p++) {
          i = Ai[p]; /* get A(i,k) */
          if (i <= k) {
            Y[i] += Ax[p]; /* scatter A(i,k) into Y (sum duplicates) */
            for (len = 0; Flag[i] !== k; i = Parent[i]) {
              Pattern[len++] = i; /* L(k,i) is nonzero */
              Flag[i] = k; /* mark i as visited */
            }
            while (len > 0) Pattern[--top] = Pattern[--len];
          }
        }
        /* compute numerical values kth row of L (a sparse triangular solve) */
        D[k] = Y[k]; /* get D(k,k) and clear Y(k) */
        Y[k] = 0.0;
        for (; top < n; top++) {
          i = Pattern[top]; /* Pattern[top:n-1] is pattern of L(:,k) */
          yi = Y[i]; /* get and clear Y(i) */
          Y[i] = 0.0;
          p2 = Lp[i] + Lnz[i];
          for (p = Lp[i]; p < p2; p++) {
            Y[Li[p]] -= Lx[p] * yi;
          }
          lKi = yi / D[i]; /* the nonzero entry L(k,i) */
          D[k] -= lKi * yi;
          Li[p] = k; /* store L(k,i) in column form of L */
          Lx[p] = lKi;
          Lnz[i]++; /* increment count of nonzeros in col i */
        }
        if (D[k] === 0.0) return k; /* failure, D(k,k) is zero */
      }
      return n; /* success, diagonal of D is all nonzero */
    }
    function ldlLsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
      let j, p, p2;
      for (j = 0; j < n; j++) {
        p2 = Lp[j + 1];
        for (p = Lp[j]; p < p2; p++) {
          X[Li[p]] -= Lx[p] * X[j];
        }
      }
    }
    function ldlDsolve(n /* D is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, D /* input of size n, not modified */) {
      let j;
      for (j = 0; j < n; j++) {
        X[j] /= D[j];
      }
    }
    function ldlLTsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
      let j, p, p2;
      for (j = n - 1; j >= 0; j--) {
        p2 = Lp[j + 1];
        for (p = Lp[j]; p < p2; p++) {
          X[j] -= Lx[p] * X[Li[p]];
        }
      }
    }
    function ldlPerm(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
      let j;
      for (j = 0; j < n; j++) {
        X[j] = B[P[j]];
      }
    }
    function ldlPermt(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
      let j;
      for (j = 0; j < n; j++) {
        X[P[j]] = B[j];
      }
    }
    function prepare(M, n, P) {
      // if a permutation was specified, apply it.
      if (P) {
        let Pinv = new Array(n);
        for (let k = 0; k < n; k++) {
          Pinv[P[k]] = k;
        }
        let Mt = []; // scratch memory
        // Apply permutation. We make M into P*M*P^T
        for (let a = 0; a < M.length; ++a) {
          let ar = Pinv[M[a][0]];
          let ac = Pinv[M[a][1]];

          // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
          // if permuted element is below diagonal, we simply transpose it.
          if (ac < ar) {
            let t = ac;
            ac = ar;
            ar = t;
          }
          Mt[a] = [];
          Mt[a][0] = ar;
          Mt[a][1] = ac;
          Mt[a][2] = M[a][2];
        }
        M = Mt; // copy scratch memory.
      } else {
        // if P argument is null, we just use an identity permutation.
        P = [];
        for (let i = 0; i < n; ++i) {
          P[i] = i;
        }
      }

      // The sparse matrix we are decomposing is A.
      // Now we shall create A from M.
      let Ap = new Array(n + 1);
      let Ai = new Array(M.length);
      let Ax = new Array(M.length);

      // count number of non-zero elements in columns.
      let LNZ = [];
      for (let i = 0; i < n; ++i) {
        LNZ[i] = 0;
      }
      for (let a = 0; a < M.length; ++a) {
        LNZ[M[a][1]]++;
      }
      Ap[0] = 0;
      for (let i = 0; i < n; ++i) {
        Ap[i + 1] = Ap[i] + LNZ[i];
      }
      let coloffset = [];
      for (let a = 0; a < n; ++a) {
        coloffset[a] = 0;
      }

      // go through all elements in M, and add them to sparse matrix A.
      for (let i = 0; i < M.length; ++i) {
        let e = M[i];
        let col = e[1];
        let adr = Ap[col] + coloffset[col];
        Ai[adr] = e[0];
        Ax[adr] = e[2];
        coloffset[col]++;
      }
      let D = new Array(n);
      let Y = new Array(n);
      let Lp = new Array(n + 1);
      let Parent = new Array(n);
      let Lnz = new Array(n);
      let Flag = new Array(n);
      let Pattern = new Array(n);
      let bp1 = new Array(n);
      let x = new Array(n);
      let d;
      ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
      let Lx = new Array(Lp[n]);
      let Li = new Array(Lp[n]);
      d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);
      if (d === n) {
        return b => {
          ldlPerm(n, bp1, b, P);
          ldlLsolve(n, bp1, Lp, Li, Lx);
          ldlDsolve(n, bp1, D);
          ldlLTsolve(n, bp1, Lp, Li, Lx);
          ldlPermt(n, x, bp1, P);
          return x;
        };
      } else {
        return null;
      }
    }

    var cuthillMckee_1 = cuthillMckee;
    function compareNum(a, b) {
      return a - b;
    }
    function cuthillMckee(list, n) {
      var adj = new Array(n);
      var visited = new Array(n);
      for (var i = 0; i < n; ++i) {
        adj[i] = [];
        visited[i] = false;
      }
      for (var i = 0; i < list.length; ++i) {
        var l = list[i];
        adj[l[0]].push(l[1]);
      }
      var toVisit = new Array(n);
      var eol = 0;
      var ptr = 0;
      for (var i = 0; i < n; ++i) {
        if (visited[i]) {
          continue;
        }
        toVisit[eol++] = i;
        visited[i] = true;
        while (ptr < eol) {
          var v = toVisit[ptr++];
          var nbhd = adj[v];
          nbhd.sort(compareNum);
          for (var j = 0; j < nbhd.length; ++j) {
            var u = nbhd[j];
            if (visited[u]) {
              continue;
            }
            visited[u] = true;
            toVisit[eol++] = u;
          }
        }
      }
      var result = new Array(n);
      for (var i = 0; i < n; ++i) {
        result[toVisit[i]] = i;
      }
      return result;
    }
    var cuthillMckee$1 = cuthillMckee_1;

    const getClosestNumber = (array = [], goal = 0) => {
      const closest = array.reduce((prev, curr) => {
        return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
      });
      return closest;
    };
    const getCloseIndex = (array = [], goal = 0) => {
      const closest = getClosestNumber(array, goal);
      return array.indexOf(closest);
    };
    const updateSystem = (matrix, y, weights) => {
      let nbPoints = y.length;
      let l = nbPoints - 1;
      let newMatrix = new Array(matrix.length);
      let newVector = new Float64Array(nbPoints);
      for (let i = 0; i < l; i++) {
        let w = weights[i];
        let diag = i * 2;
        let next = diag + 1;
        newMatrix[diag] = matrix[diag].slice();
        newMatrix[next] = matrix[next].slice();
        if (w === 0) {
          newVector[i] = 0;
        } else {
          newVector[i] = y[i] * w;
          newMatrix[diag][2] += w;
        }
      }
      newVector[l] = y[l] * weights[l];
      newMatrix[l * 2] = matrix[l * 2].slice();
      newMatrix[l * 2][2] += weights[l];
      return [newMatrix, newVector];
    };
    const getDeltaMatrix = (nbPoints, lambda) => {
      let matrix = [];
      let last = nbPoints - 1;
      for (let i = 0; i < last; i++) {
        matrix.push([i, i, lambda * 2]);
        matrix.push([i + 1, i, -1 * lambda]);
      }
      matrix[0][2] = lambda;
      matrix.push([last, last, lambda]);
      return {
        lowerTriangularNonZeros: matrix,
        permutationEncodedArray: cuthillMckee$1(matrix, nbPoints)
      };
    };

    function getControlPoints(x, y, options = {}) {
      const {
        length
      } = x;
      let {
        controlPoints = Int8Array.from({
          length
        }).fill(0)
      } = options;
      const {
        zones = [],
        weights = Float64Array.from({
          length
        }).fill(1)
      } = options;
      if (x.length !== y.length) {
        throw new RangeError('Y should match the length with X');
      } else if (controlPoints.length !== x.length) {
        throw new RangeError('controlPoints should match the length with X');
      } else if (weights.length !== x.length) {
        throw new RangeError('weights should match the length with X');
      }
      zones.forEach(range => {
        let indexFrom = getCloseIndex(x, range.from);
        let indexTo = getCloseIndex(x, range.to);
        if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];
        for (let i = indexFrom; i < indexTo; i++) {
          controlPoints[i] = 1;
        }
      });
      return {
        weights: 'controlPoints' in options || zones.length > 0 ? xMultiply(weights, controlPoints) : weights,
        controlPoints
      };
    }

    /**
     * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
     * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/main/airPLS_manuscript.pdf)
     * @param {Array<number>} x - x axis data useful when control points or zones are submitted
     * @param {Array<number>} y - Original data
     * @param {object} [options={}] - Options object
     * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
     * @param {number} [options.tolerance = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
     * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
     * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
     * @param {Array<number>} [options.controlPoints = []] - Array of 0|1 to force the baseline cross those points.
     * @param {Array<number>} [options.zones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
     * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
     */

    function airPLS(x, y, options = {}) {
      const {
        weights,
        controlPoints
      } = getControlPoints(x, y, options);
      let {
        maxIterations = 100,
        lambda = 10,
        tolerance = 0.001
      } = options;
      let baseline, iteration;
      let sumNegDifferences = Number.MAX_SAFE_INTEGER;
      const corrected = Float64Array.from(y);
      let stopCriterion = getStopCriterion(y, tolerance);
      const {
        length
      } = y;
      let {
        lowerTriangularNonZeros,
        permutationEncodedArray
      } = getDeltaMatrix(length, lambda);
      let threshold = 1;
      const l = length - 1;
      let prevNegSum = Number.MAX_SAFE_INTEGER;
      for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
        let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
        let cho = prepare(leftHandSide, length, permutationEncodedArray);
        baseline = cho(rightHandSide);
        sumNegDifferences = applyCorrection(y, baseline, corrected);
        if (iteration === 1) {
          const {
            positive
          } = xNoiseSanPlot(corrected);
          threshold = positive;
        } else {
          const absChange = Math.abs(prevNegSum / sumNegDifferences);
          if (absChange < 1.01 && absChange > 0.99) {
            break;
          }
        }
        prevNegSum = sumNegDifferences + 0;
        for (let i = 1; i < l; i++) {
          const diff = corrected[i];
          if (controlPoints[i] < 1 && Math.abs(diff) > threshold) {
            weights[i] = 0;
          } else {
            const factor = diff > 0 ? -1 : 1;
            weights[i] = Math.exp(factor * (iteration * diff) / Math.abs(sumNegDifferences));
          }
        }
        weights[0] = 1;
        weights[l] = 1;
      }
      return {
        corrected,
        baseline,
        iteration,
        error: sumNegDifferences
      };
      function applyCorrection(y, baseline, corrected) {
        let sumNegDifferences = 0;
        for (let i = 0; i < y.length; i++) {
          let diff = y[i] - baseline[i];
          if (diff < 0) sumNegDifferences += diff;
          corrected[i] = diff;
        }
        return sumNegDifferences;
      }
    }
    function getStopCriterion(y, tolerance) {
      let sum = xAbsoluteSum(y);
      return tolerance * sum;
    }

    function _typeof(obj) {
      "@babel/helpers - typeof";

      if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function (obj) {
          return typeof obj;
        };
      } else {
        _typeof = function (obj) {
          return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
        };
      }
      return _typeof(obj);
    }

    /**
     * Fill an array with sequential numbers
     * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
     * @param {object} [options={}]
     * @param {number} [options.from=0] - first value in the array
     * @param {number} [options.to=10] - last value in the array
     * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
     * @param {number} [options.step] - if not provided calculated from size
     * @return {Array<number>}
     */

    function sequentialFill() {
      var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (_typeof(input) === 'object' && !isAnyArray$1(input)) {
        options = input;
        input = [];
      }
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      var _options = options,
        _options$from = _options.from,
        from = _options$from === void 0 ? 0 : _options$from,
        _options$to = _options.to,
        to = _options$to === void 0 ? 10 : _options$to,
        _options$size = _options.size,
        size = _options$size === void 0 ? input.length : _options$size,
        step = _options.step;
      if (size !== 0 && step) {
        throw new Error('step is defined by the array size');
      }
      if (!size) {
        if (step) {
          size = Math.floor((to - from) / step) + 1;
        } else {
          size = to - from + 1;
        }
      }
      if (!step && size) {
        step = (to - from) / (size - 1);
      }
      if (Array.isArray(input)) {
        // only works with normal array
        input.length = 0;
        for (var i = 0; i < size; i++) {
          input.push(from);
          from += step;
        }
      } else {
        if (input.length !== size) {
          throw new Error('sequentialFill typed array must have the correct length');
        }
        for (var _i = 0; _i < size; _i++) {
          input[_i] = from;
          from += step;
        }
      }
      return input;
    }

    /**
     * Adaptive iteratively reweighted penalized least squares [1]
     *
     * This function calls ml-airpls
     *
     * References:
     * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
     * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
     * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
     * @export
     * @param {Array<number>} ys
     * @param {object} [options] - Options object
     * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
     * @param {object} [options.regression] - Options for the regression
     * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
     * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
     * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
     * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
     * @returns {BaselineOutput}
     */
    function airPLSBaseline$1(ys, options = {}) {
      const numberPoints = ys.length;
      let {
        x,
        regressionOptions
      } = options;
      if (!x) {
        x = sequentialFill({
          from: 0,
          to: numberPoints - 1,
          size: numberPoints
        });
      }
      let output = airPLS(x, ys, regressionOptions);
      return {
        baseline: output.baseline,
        correctedSpectrum: output.corrected
      };
    }

    /**
     * Check that x and y are arrays with the same length.
     * @param x - first array
     * @param y - second array
     * @throws if x or y are not the same length, or if they are not arrays
     */
    function checkArrayLength(x, y) {
      if (!isAnyArray$1(x) || !isAnyArray$1(y)) {
        throw new TypeError('x and y must be arrays');
      }
      if (x.length !== y.length) {
        throw new RangeError('x and y arrays must have the same length');
      }
    }

    class BaseRegression {
      constructor() {
        if (new.target === BaseRegression) {
          throw new Error('BaseRegression must be subclassed');
        }
      }
      predict(x) {
        if (typeof x === 'number') {
          return this._predict(x);
        } else if (isAnyArray$1(x)) {
          const y = [];
          for (const xVal of x) {
            y.push(this._predict(xVal));
          }
          return y;
        } else {
          throw new TypeError('x must be a number or array');
        }
      }
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      _predict(x) {
        throw new Error('_predict must be implemented');
      }
      train() {
        // Do nothing for this package
      }
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      toString(precision) {
        return '';
      }
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      toLaTeX(precision) {
        return '';
      }
      /**
       * Return the correlation coefficient of determination (r) and chi-square.
       * @param x - explanatory variable
       * @param y - response variable
       * @return - Object with further statistics.
       */
      score(x, y) {
        checkArrayLength(x, y);
        const n = x.length;
        const y2 = new Array(n);
        for (let i = 0; i < n; i++) {
          y2[i] = this._predict(x[i]);
        }
        let xSum = 0;
        let ySum = 0;
        let chi2 = 0;
        let rmsd = 0;
        let xSquared = 0;
        let ySquared = 0;
        let xY = 0;
        for (let i = 0; i < n; i++) {
          xSum += y2[i];
          ySum += y[i];
          xSquared += y2[i] * y2[i];
          ySquared += y[i] * y[i];
          xY += y2[i] * y[i];
          if (y[i] !== 0) {
            chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
          }
          rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
        }
        const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
        return {
          r,
          r2: r * r,
          chi2,
          rmsd: Math.sqrt(rmsd / n)
        };
      }
    }

    /**
     * Cast `number` to string. Optionally `digits` specifies significant figures.
     * @param number
     * @param figures
     * @returns - A string representation of `number`.
     */
    function maybeToPrecision(number, figures) {
      if (number < 0) {
        number = 0 - number;
        if (typeof figures === 'number') {
          return `- ${number.toPrecision(figures)}`;
        } else {
          return `- ${number.toString()}`;
        }
      } else if (typeof figures === 'number') {
        return number.toPrecision(figures);
      } else {
        return number.toString();
      }
    }

    class PolynomialRegression extends BaseRegression {
      /**
       * @param x - independent or explanatory variable
       * @param y - dependent or response variable
       * @param degree - degree of the polynomial regression, or array of powers to be used. When degree is an array, intercept at zero is forced to false/ignored.
       * @example `new PolynomialRegression(x, y, 2)`, in this case, you can pass the option `interceptAtZero`, if you need it.
       * @example `new PolynomialRegression(x, y, [1, 3, 5])`
       * Each of the degrees corresponds to a column, so if you have them switched, just do:
       * @example `new PolynomialRegression(x, y, [3, 1, 5])`
       *
       * @param options.interceptAtZero - force the polynomial regression so that f(0) = 0
       */
      constructor(x, y, degree, options = {}) {
        super();
        // @ts-expect-error internal use only
        if (x === true) {
          // @ts-expect-error internal use only
          this.degree = y.degree;
          // @ts-expect-error internal use only
          this.powers = y.powers;
          // @ts-expect-error internal use only
          this.coefficients = y.coefficients;
        } else {
          checkArrayLength(x, y);
          const result = regress(x, y, degree, options);
          this.degree = result.degree;
          this.powers = result.powers;
          this.coefficients = result.coefficients;
        }
      }
      _predict(x) {
        let y = 0;
        for (let k = 0; k < this.powers.length; k++) {
          y += this.coefficients[k] * x ** this.powers[k];
        }
        return y;
      }
      toJSON() {
        return {
          name: 'polynomialRegression',
          degree: this.degree,
          powers: this.powers,
          coefficients: this.coefficients
        };
      }
      toString(precision) {
        return this._toFormula(precision, false);
      }
      toLaTeX(precision) {
        return this._toFormula(precision, true);
      }
      _toFormula(precision, isLaTeX) {
        let sup = '^';
        let closeSup = '';
        let times = ' * ';
        if (isLaTeX) {
          sup = '^{';
          closeSup = '}';
          times = '';
        }
        let fn = '';
        let str = '';
        for (let k = 0; k < this.coefficients.length; k++) {
          str = '';
          if (this.coefficients[k] !== 0) {
            if (this.powers[k] === 0) {
              str = maybeToPrecision(this.coefficients[k], precision);
            } else if (this.powers[k] === 1) {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
            } else {
              str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
            }
            if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
              str = ` + ${str}`;
            } else if (k !== this.coefficients.length - 1) {
              str = ` ${str}`;
            }
          }
          fn = str + fn;
        }
        if (fn.startsWith('+')) {
          fn = fn.slice(1);
        }
        return `f(x) = ${fn}`;
      }
      static load(json) {
        if (json.name !== 'polynomialRegression') {
          throw new TypeError('not a polynomial regression model');
        }
        // @ts-expect-error internal use only
        return new PolynomialRegression(true, json);
      }
    }
    /**
     * Perform a polynomial regression on the given data set.
     * This is an internal function.
     * @param x - independent or explanatory variable
     * @param y - dependent or response variable
     * @param degree - degree of the polynomial regression
     * @param options.interceptAtZero - force the polynomial regression so that $f(0) = 0$
     */
    function regress(x, y, degree, options = {}) {
      const n = x.length;
      let {
        interceptAtZero = false
      } = options;
      let powers = [];
      if (Array.isArray(degree)) {
        powers = degree;
        interceptAtZero = false; //must be false in this case
      } else if (typeof degree === 'number') {
        if (interceptAtZero) {
          powers = new Array(degree);
          for (let k = 0; k < degree; k++) {
            powers[k] = k + 1;
          }
        } else {
          powers = new Array(degree + 1);
          for (let k = 0; k <= degree; k++) {
            powers[k] = k;
          }
        }
      }
      const nCoefficients = powers.length; //1 per power, in any case.
      const F = new Matrix(n, nCoefficients);
      const Y = new Matrix([y]);
      for (let k = 0; k < nCoefficients; k++) {
        for (let i = 0; i < n; i++) {
          if (powers[k] === 0) {
            F.set(i, k, 1);
          } else {
            F.set(i, k, x[i] ** powers[k]);
          }
        }
      }
      const FT = new MatrixTransposeView(F);
      const A = FT.mmul(F);
      const B = FT.mmul(new MatrixTransposeView(Y));
      return {
        coefficients: solve(A, B).to1DArray(),
        degree: Math.max(...powers),
        powers
      };
    }

    /**
     * Iterative regression-based baseline correction
     * @param {Array<number>} x - Independent axis variable
     * @param {Array<number>} y - Dependent axis variable
     * @param {object} [options] - Options object
     * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
     * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
     * @param {*} [options.regressionOptions] - Options for regressionFunction
     * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
     * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
     */
    function baselineCorrectionRegression(x, y, options = {}) {
      let {
        maxIterations = 100,
        Regression = PolynomialRegression,
        regressionOptions,
        tolerance = 0.001
      } = options;
      if (!regressionOptions && Regression === PolynomialRegression) {
        regressionOptions = 3;
      }
      let baseline = y.slice();
      let fitting = y.slice();
      let oldFitting = y;
      let iteration = 0;
      let delta;
      let regression;
      while (iteration < maxIterations) {
        // Calculate the fitting result
        regression = new Regression(x, baseline, regressionOptions);
        delta = 0;
        for (let i = 0; i < baseline.length; i++) {
          fitting[i] = regression.predict(x[i]);
          if (baseline[i] > fitting[i]) {
            baseline[i] = fitting[i];
          }
          delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
        }

        // Stop criterion
        if (delta < tolerance) {
          break;
        } else {
          oldFitting = fitting.slice();
          iteration++;
        }
      }

      // removes baseline
      let corrected = new Array(baseline.length);
      for (let j = 0; j < baseline.length; j++) {
        corrected[j] = y[j] - baseline[j];
      }
      return {
        corrected,
        delta,
        iteration,
        baseline,
        regression
      };
    }

    /**
     * Iterative polynomial fitting [1]
     *
     * Implementation based on ml-baseline-correction-regression
     *
     * References:
     * [1] Gan, F.; Ruan, G.; Mo, J.
     * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
     *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
     * https://doi.org/10.1016/j.chemolab.2005.08.009.
     * @export
     * @param {Array<number>} ys
     * @param {object} [options] - Options object
     * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
     * @param {Object} [options.regression]
     * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
     * @param {Object} [options.regression]
     * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
     * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
     * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
     * @returns {BaselineOutput}
     */
    function iterativePolynomialBaseline$1(ys, options = {}) {
      const numberPoints = ys.length;
      let {
        x,
        regressionOptions
      } = options;
      if (!x) {
        x = sequentialFill({
          from: 0,
          to: numberPoints - 1,
          size: numberPoints
        });
      }
      let output = baselineCorrectionRegression(x, ys, regressionOptions);
      return {
        baseline: output.baseline,
        correctedSpectrum: output.corrected
      };
    }

    /**

     *
     * @export
     * @param {Array<number>} ys
     * @param {Object} [options={}]
     * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
     * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
     * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
     * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
     * @returns {BaselineOutput}
     */
    function rollingAverageBaseline$1(ys, options = {}) {
      let window = Math.max(Math.round(ys.length * 0.1), 2);
      let defaults = {
        window,
        padding: {
          size: window - 1,
          algorithm: 'duplicate',
          value: 0
        }
      };
      let actualOptions = {
        ...defaults,
        ...options
      };
      let baseline = xRollingAverage(ys, actualOptions);
      let corrected = new Float64Array(ys.length);
      for (let i = 0; i < corrected.length; i++) {
        corrected[i] = ys[i] - baseline[i];
      }
      return {
        baseline,
        correctedSpectrum: corrected
      };
    }

    function rollingBall(spectrum, options = {}) {
      if (!isAnyArray$1(spectrum)) {
        throw new Error('Spectrum must be an array');
      }
      if (spectrum.length === 0) {
        throw new TypeError('Spectrum must not be empty');
      }
      const numberPoints = spectrum.length;
      const maxima = new Float64Array(numberPoints);
      const minima = new Float64Array(numberPoints);
      const baseline = new Float64Array(numberPoints);
      // windowM 4 percent of spectrum length
      // windowS 8 percent of spectrum length
      const {
        windowM = Math.round(numberPoints * 0.04),
        windowS = Math.round(numberPoints * 0.08)
      } = options;
      // fi(1) in original paper
      for (let i = 0; i < spectrum.length; i++) {
        const windowLeft = Math.max(0, i - windowM);
        const windowRight = Math.min(i + windowM + 1, spectrum.length);
        minima[i] = xMinValue(spectrum, {
          fromIndex: windowLeft,
          toIndex: windowRight
        });
      }
      // fi in original paper
      for (let i = 0; i < minima.length; i++) {
        const windowLeft = Math.max(0, i - windowM);
        const windowRight = Math.min(i + windowM + 1, minima.length);
        maxima[i] = xMaxValue(minima, {
          fromIndex: windowLeft,
          toIndex: windowRight
        });
      }
      for (let i = 0; i < minima.length; i++) {
        const windowLeft = Math.max(0, i - windowS);
        const windowRight = Math.min(i + windowS + 1, maxima.length);
        baseline[i] = xMean(maxima.subarray(windowLeft, windowRight));
      }
      return baseline;
    }

    /**
     * Rolling ball baseline correction algorithm.
     * From the abstract of (1):
     * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
     * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
     *
     * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
     * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
     *
     * Algorithm described in (1), but in the implementation here the window width does not change.
     *
     * Reference:
     * (1) Kneen, M. A.; Annegarn, H. J.
     *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
     *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
     *     https://doi.org/10.1016/0168-583X(95)00908-6.
     * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
     *     https://cran.r-project.org/web/packages/baseline/index.html
     *
     * @export
     * @param {Array<number>} ys
     * @param {Object} [options={}]
     * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
     * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
     * @returns {BaselineOutput}
     */
    function rollingBallBaseline$1(ys, options = {}) {
      const baseline = rollingBall(ys, options);
      let corrected = new Float64Array(ys.length);
      for (let i = 0; i < corrected.length; i++) {
        corrected[i] = ys[i] - baseline[i];
      }
      return {
        baseline,
        correctedSpectrum: corrected
      };
    }

    /**

     *
     * @export
     * @param {Array<number>} ys
     * @param {Object} [options={}]
     * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
     * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
     * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
     * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
     * @returns {BaselineOutput}
     */
    function rollingMedianBaseline$1(ys, options = {}) {
      let window = Math.max(Math.round(ys.length * 0.1), 2);
      let defaults = {
        window,
        padding: {
          size: window - 1,
          algorithm: 'duplicate',
          value: 0
        }
      };
      let actualOptions = {
        ...defaults,
        ...options
      };
      let baseline = xRollingMedian(ys, actualOptions);
      let corrected = new Float64Array(ys.length);
      for (let i = 0; i < corrected.length; i++) {
        corrected[i] = ys[i] - baseline[i];
      }
      return {
        baseline,
        correctedSpectrum: corrected
      };
    }

    //@ts-expect-error no type definition for baselines
    /**
     * @param data
     */
    function airPLSBaseline(data) {
      data.y = airPLSBaseline$1(data.y).correctedSpectrum;
      return {
        data
      };
    }

    //@ts-expect-error no type definition for baselines
    /**
     * @param data
     */
    function iterativePolynomialBaseline(data) {
      data.y = iterativePolynomialBaseline$1(data.y).correctedSpectrum;
      return {
        data
      };
    }

    //@ts-expect-error no type definition for baselines
    /**
     * @param data
     */
    function rollingAverageBaseline(data) {
      data.y = rollingAverageBaseline$1(data.y).correctedSpectrum;
      return {
        data
      };
    }

    //@ts-expect-error no type definition for baselines
    /**
     * @param data
     */
    function rollingBallBaseline(data) {
      data.y = rollingBallBaseline$1(data.y).correctedSpectrum;
      return {
        data
      };
    }

    //@ts-expect-error no type definition for baselines
    /**
     * @param data
     */
    function rollingMedianBaseline(data) {
      data.y = rollingMedianBaseline$1(data.y).correctedSpectrum;
      return {
        data
      };
    }

    /**
     * Apply Savitzky Golay algorithm
     * @param [ys] Array of y values
     * @param [xs] Array of X or deltaX
     * @return  Array containing the new ys (same length)
     */
    function sgg(ys, xs, options = {}) {
      const {
        windowSize = 9,
        derivative = 0,
        polynomial = 3
      } = options;
      if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
        throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
      }
      if (!isAnyArray$1(ys)) {
        throw new TypeError('Y values must be an array');
      }
      if (typeof xs === 'undefined') {
        throw new TypeError('X must be defined');
      }
      if (windowSize > ys.length) {
        throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
      }
      if (derivative < 0 || !Number.isInteger(derivative)) {
        throw new RangeError('Derivative should be a positive integer');
      }
      if (polynomial < 1 || !Number.isInteger(polynomial)) {
        throw new RangeError('Polynomial should be a positive integer');
      }
      if (polynomial >= 6) {
        // eslint-disable-next-line no-console
        console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
      }
      const half = Math.floor(windowSize / 2);
      const np = ys.length;
      const ans = new Float64Array(np);
      const weights = fullWeights(windowSize, polynomial, derivative);
      let hs = 0;
      let constantH = true;
      if (isAnyArray$1(xs)) {
        constantH = false;
      } else {
        hs = xs ** derivative;
      }
      //For the borders
      for (let i = 0; i < half; i++) {
        const wg1 = weights[half - i - 1];
        const wg2 = weights[half + i + 1];
        let d1 = 0;
        let d2 = 0;
        for (let l = 0; l < windowSize; l++) {
          d1 += wg1[l] * ys[l];
          d2 += wg2[l] * ys[np - windowSize + l];
        }
        if (constantH) {
          ans[half - i - 1] = d1 / hs;
          ans[np - half + i] = d2 / hs;
        } else {
          hs = getHs(xs, half - i - 1, half, derivative);
          ans[half - i - 1] = d1 / hs;
          hs = getHs(xs, np - half + i, half, derivative);
          ans[np - half + i] = d2 / hs;
        }
      }
      //For the internal points
      const wg = weights[half];
      for (let i = windowSize; i <= np; i++) {
        let d = 0;
        for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];
        if (!constantH) {
          hs = getHs(xs, i - half - 1, half, derivative);
        }
        ans[i - half - 1] = d / hs;
      }
      return ans;
    }
    function getHs(h, center, half, derivative) {
      let hs = 0;
      let count = 0;
      for (let i = center - half; i < center + half; i++) {
        if (i >= 0 && i < h.length - 1) {
          hs += h[i + 1] - h[i];
          count++;
        }
      }
      return (hs / count) ** derivative;
    }
    function gramPoly(i, m, k, s) {
      let Grampoly = 0;
      if (k > 0) {
        Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
      } else if (k === 0 && s === 0) {
        Grampoly = 1;
      } else {
        Grampoly = 0;
      }
      return Grampoly;
    }
    function genFact(a, b) {
      let gf = 1;
      if (a >= b) {
        for (let j = a - b + 1; j <= a; j++) {
          gf *= j;
        }
      }
      return gf;
    }
    function weight(i, t, m, n, s) {
      let sum = 0;
      for (let k = 0; k <= n; k++) {
        sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
      }
      return sum;
    }
    /**
     * @private
     * @param m  Number of points
     * @param n  Polynomial grade
     * @param s  Derivative
     */
    function fullWeights(m, n, s) {
      const weights = new Array(m);
      const np = Math.floor(m / 2);
      for (let t = -np; t <= np; t++) {
        weights[t + np] = new Float64Array(m);
        for (let j = -np; j <= np; j++) {
          weights[t + np][j + np] = weight(j, t, np, n, s);
        }
      }
      return weights;
    }

    /**
     * Calculate the first derivative using Savitzky–Golay filter.
     * @param data
     */
    function firstDerivative(data, options = {}) {
      const {
        x,
        y
      } = data;
      return {
        data: {
          x,
          y: sgg(y, x, {
            ...options,
            derivative: 1
          })
        }
      };
    }

    /**
     * Calculate the second derivative using Savitzky–Golay filter.
     * @param data
     */
    function secondDerivative(data, options = {}) {
      const {
        x,
        y
      } = data;
      return {
        data: {
          x,
          y: sgg(y, x, {
            ...options,
            derivative: 2
          })
        }
      };
    }

    /**
     * Calculate the third derivative using Savitzky–Golay filter.
     * @param data
     */
    function thirdDerivative(data, options = {}) {
      const {
        x,
        y
      } = data;
      return {
        data: {
          x,
          y: sgg(y, x, {
            ...options,
            derivative: 3
          })
        }
      };
    }

    /**
     * Apply the Savitzky Golay Generalized Filter
     * @param data
     */
    function savitzkyGolay(data, options = {}) {
      const {
        x,
        y
      } = data;
      return {
        data: {
          x,
          y: sgg(y, x, options)
        }
      };
    }

    /**
     * Ensure X values are strictly monotonic increasing
     * http://www-groups.mcs.st-andrews.ac.uk/~john/analysis/Lectures/L8.html
     * @param data
     */
    function ensureGrowing(data) {
      return {
        data: xyEnsureGrowingX(data)
      };
    }

    /**
     * Filter that allows to
     * @param data
     * @param options
     */
    function equallySpaced(data, options = {}) {
      return {
        data: xyEquallySpaced(data, options)
      };
    }

    /**
     * Filter that allows to
     * @param data
     * @param options
     */
    function filterX(data, options = {}) {
      return {
        data: xyFilterX(data, options)
      };
    }

    var IDX = 256,
      HEX = [],
      BUFFER;
    while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
    function v4() {
      var i = 0,
        num,
        out = '';
      if (!BUFFER || IDX + 16 > 256) {
        BUFFER = Array(i = 256);
        while (i--) BUFFER[i] = 256 * Math.random() | 0;
        i = IDX = 0;
      }
      for (; i < 16; i++) {
        num = BUFFER[IDX + i];
        if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
        if (i & 1 && i > 1 && i < 11) out += '-';
      }
      IDX++;
      return out;
    }

    /**
     * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
     * This process is done in place and is very fast.
     * @param data
     * @param peaks
     */
    function optimizeTop(data, peaks) {
      const {
        x,
        y
      } = data;
      for (const peak of peaks) {
        let currentIndex = peak.index;
        // The detected peak could be moved 1 or 2 units to left or right.
        if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
          currentIndex--;
        } else if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
          currentIndex++;
        } else if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
          currentIndex -= 2;
        } else if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
          currentIndex += 2;
        }
        // interpolation to a sin() function
        if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
          const alpha = 20 * Math.log10(y[currentIndex - 1]);
          const beta = 20 * Math.log10(y[currentIndex]);
          const gamma = 20 * Math.log10(y[currentIndex + 1]);
          const p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
          peak.x = x[currentIndex] + (x[currentIndex] - x[currentIndex - 1]) * p;
          peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
        }
      }
    }

    /**
     * Global spectra deconvolution
     * @param  data - Object data with x and y arrays. Values in x has to be growing
     * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
     * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.

     */
    function gsd(data, options = {}) {
      let {
        noiseLevel
      } = options;
      const {
        sgOptions = {
          windowSize: 9,
          polynomial: 3
        },
        smoothY = false,
        maxCriteria = true,
        minMaxRatio = 0.00025,
        realTopDetection = false
      } = options;
      const {
        x
      } = data;
      let {
        y
      } = data;
      if (xIsMonotonic(x) !== 1) {
        throw new Error('GSD only accepts monotone increasing x values');
      }
      //rescale;
      y = y.slice();
      // If the max difference between delta x is less than 5%, then,
      // we can assume it to be equally spaced variable
      const equallySpaced = xIsEquallySpaced(x);
      if (noiseLevel === undefined) {
        if (equallySpaced) {
          const noiseInfo = xNoiseStandardDeviation(y);
          if (maxCriteria) {
            noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
          } else {
            noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
          }
        } else {
          noiseLevel = 0;
        }
      } else if (!maxCriteria) {
        noiseLevel *= -1;
      }
      if (!maxCriteria) {
        for (let i = 0; i < y.length; i++) {
          y[i] *= -1;
        }
      }
      if (noiseLevel !== undefined) {
        for (let i = 0; i < y.length; i++) {
          if (y[i] < noiseLevel) {
            y[i] = noiseLevel;
          }
        }
      }
      const xValue = equallySpaced ? x[1] - x[0] : x;
      const yData = smoothY ? sgg(y, xValue, {
        ...sgOptions,
        derivative: 0
      }) : y;
      const dY = sgg(y, xValue, {
        ...sgOptions,
        derivative: 1
      });
      const ddY = sgg(y, xValue, {
        ...sgOptions,
        derivative: 2
      });
      const {
        min: minY,
        max: maxY
      } = xMinMaxValues(yData);
      if (minY > maxY || minY === maxY) return [];
      const yThreshold = minY + (maxY - minY) * minMaxRatio;
      const dX = x[1] - x[0];
      let lastMax = null;
      let lastMin = null;
      const minddY = [];
      const intervalL = [];
      const intervalR = [];
      // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum
      for (let i = 1; i < yData.length - 1; ++i) {
        if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
          lastMin = {
            x: x[i],
            index: i
          };
          if (dX > 0 && lastMax !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
        // Maximum in first derivative
        if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
          lastMax = {
            x: x[i],
            index: i
          };
          if (dX < 0 && lastMin !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
        // Minimum in second derivative
        if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
          minddY.push(i);
        }
      }
      let lastK = -1;
      const peaks = [];
      for (const minddYIndex of minddY) {
        const deltaX = x[minddYIndex];
        let possible = -1;
        let k = lastK + 1;
        let minDistance = Number.POSITIVE_INFINITY;
        let currentDistance = 0;
        while (possible === -1 && k < intervalL.length) {
          currentDistance = Math.abs(deltaX - (intervalL[k].x + intervalR[k].x) / 2);
          if (currentDistance < (intervalR[k].x - intervalL[k].x) / 2) {
            possible = k;
            lastK = k;
          }
          ++k;
          // Not getting closer?
          if (currentDistance >= minDistance) {
            break;
          }
          minDistance = currentDistance;
        }
        if (possible !== -1) {
          if (yData[minddYIndex] > yThreshold) {
            const width = Math.abs(intervalR[possible].x - intervalL[possible].x);
            peaks.push({
              id: v4(),
              x: deltaX,
              y: yData[minddYIndex],
              width,
              index: minddYIndex,
              ddY: ddY[minddYIndex],
              inflectionPoints: {
                from: intervalL[possible],
                to: intervalR[possible]
              }
            });
          }
        }
      }
      if (realTopDetection) {
        optimizeTop({
          x,
          y: yData
        }, peaks);
      }
      peaks.forEach(peak => {
        if (!maxCriteria) {
          peak.y *= -1;
          peak.ddY = peak.ddY * -1;
        }
      });
      peaks.sort((a, b) => {
        return a.x - b.x;
      });
      return peaks;
    }

    /**
     * Filter that allows to calibrateX the x axis based on the presence of peaks
     */
    function calibrateX(data, options = {}) {
      const {
        targetX = 0,
        nbPeaks = 1,
        from = data.x[0],
        to = data.x[data.x.length - 1],
        gsd: gsdOptions = {
          minMaxRatio: 0.1,
          realTopDetection: true,
          smoothY: true,
          sgOptions: {
            windowSize: 7,
            polynomial: 3
          }
        }
      } = options;
      const fromIndex = xFindClosestIndex(data.x, from);
      const toIndex = xFindClosestIndex(data.x, to);
      const peaks = gsd({
        x: data.x.subarray(fromIndex, toIndex),
        y: data.y.subarray(fromIndex, toIndex)
      }, gsdOptions).sort((a, b) => b.y - a.y).slice(0, nbPeaks);
      if (peaks.length < nbPeaks) return {
        data
      };
      const middle = xMean(peaks.map(peak => peak.x));
      return {
        data: {
          x: xAdd(data.x, targetX - middle),
          y: data.y
        }
      };
    }

    /**
     * Filter that allows to
     * @param data
     * @param options
     */
    function xFunction(data, options = {}) {
      return {
        data: {
          x: xApplyFunctionStr(data.x, {
            variableLabel: 'x',
            fctString: options.function
          }),
          y: data.y
        }
      };
    }

    /**
     * Filter that allows to
     * @param data
     * @param options
     */
    function yFunction(data, options = {}) {
      return {
        data: {
          x: data.x,
          y: xApplyFunctionStr(data.y, {
            variableLabel: 'y',
            fctString: options.function
          })
        }
      };
    }

    var Filters = /*#__PURE__*/Object.freeze({
        __proto__: null,
        centerMean: centerMean,
        centerMedian: centerMedian,
        fromTo: fromTo,
        normed: normed,
        divideBySD: divideBySD,
        rescale: rescale,
        paretoNormalization: paretoNormalization,
        airPLSBaseline: airPLSBaseline,
        iterativePolynomialBaseline: iterativePolynomialBaseline,
        rollingAverageBaseline: rollingAverageBaseline,
        rollingBallBaseline: rollingBallBaseline,
        rollingMedianBaseline: rollingMedianBaseline,
        firstDerivative: firstDerivative,
        secondDerivative: secondDerivative,
        thirdDerivative: thirdDerivative,
        savitzkyGolay: savitzkyGolay,
        ensureGrowing: ensureGrowing,
        equallySpaced: equallySpaced,
        filterX: filterX,
        calibrateX: calibrateX,
        xFunction: xFunction,
        yFunction: yFunction
    });

    /**
     * Apply filters on {x:[], y:[]}
     * @returns A very important number
     */
    function filterXY(data, filters) {
      let result = {
        data: {
          x: xEnsureFloat64(data.x),
          y: xEnsureFloat64(data.y)
        }
      };
      const logs = [];
      for (const filter of filters) {
        const start = Date.now();
        // eslint-disable-next-line import/namespace
        const filterFct = Filters[filter.name];
        if (!filterFct) {
          throw new Error(`Unknown filter: ${filter.name}`);
        }
        // @ts-expect-error some method have options and some other ones don't have any options
        result = filterFct(result.data, filter.options);
        logs.push({
          name: filter.name,
          time: Date.now() - start
        });
      }
      return {
        logs,
        data: result.data
      };
    }

    /**
     *
     * @private
     * @param {DataXY} input
     * @param {object} [options={}]
     * @param {number} [options.from=x[0]]
     * @param {number} [options.to=x[x.length-1]]
     * @param {number} [options.numberOfPoints=1024]
     * @param {Array} [options.filters=[]]
     * @param {Array} [options.exclusions=[]]
     */
    function getNormalized(input, options = {}) {
      xyCheck(input);
      let {
        from = input.x[0],
        to = input.x.at(-1),
        numberOfPoints = 1024,
        filters = [],
        exclusions = []
      } = options;

      // we will add a get
      const equallySpacedFilter = {
        name: 'equallySpaced',
        options: {
          from,
          to,
          numberOfPoints,
          exclusions
        }
      };
      const output = filterXY(input, filters.concat(equallySpacedFilter)).data;
      const allowedBoundary = {
        x: {
          min: output.x[0],
          max: output.x.at(-1)
        },
        y: xMinMaxValues(output.y)
      };
      return {
        data: output,
        allowedBoundary
      };
    }

    /**
     * @private
     * @param {*} spectrum
     * @param {*} ranges
     */
    function updateRangesInfo(spectrum, ranges = []) {
      spectrum.ranges = {};
      for (let range of ranges) {
        range = structuredClone(range);
        spectrum.ranges[range.label] = range;
        let fromToIndex = {
          fromIndex: xFindClosestIndex(spectrum.normalized.x, range.from),
          toIndex: xFindClosestIndex(spectrum.normalized.x, range.to)
        };
        range.integration = xyIntegration(spectrum.normalized, fromToIndex);
        range.maxPoint = xyMaxYPoint(spectrum.normalized, fromToIndex);
        range.minPoint = xyMinYPoint(spectrum.normalized, fromToIndex);
        range.x = spectrum.normalized.x.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1);
        range.y = spectrum.normalized.y.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1);
      }
    }

    /**
     * Class allowing manipulate one IR spectrum
     * @class spectrum
     * @param {object} [json={}] - object containing a spectrum
     * @param {Array} [json.x=[]] - x values
     * @param {Array} [json.y=[]] - y values
     */
    class Spectrum {
      /**
       *
       * @param {Array} x
       * @param {Array} y
       * @param {string} id
       * @param {object} [options={}]
       * @param {object} [options.meta={}]
       * @param {object} [options.normalization]
       * @param {object} [options.normalized]
       */
      constructor(x, y, id, options = {}) {
        const {
          meta = {},
          normalization = {},
          normalized
        } = options;
        if (x && x.length > 1 && x[0] > x.at(-1)) {
          this.x = x.slice().reverse();
          this.y = y.slice().reverse();
        } else {
          this.x = x || [];
          this.y = y || [];
        }
        this.id = id;
        this.meta = meta;
        this.normalizedBoundary = {
          x: {
            min: 0,
            max: 0
          },
          y: {
            min: 0,
            max: 0
          }
        };
        if (normalized) {
          this.normalized = normalized;
          this.updateNormalizedBoundary();
          this.normalizedAllowedBoundary = this.normalizedBoundary;
        } else {
          this.updateNormalization(normalization);
        }
        this.updateMemory();
      }
      get minX() {
        return this.x[0];
      }
      get maxX() {
        return this.x.at(-1);
      }
      updateMemory() {
        let original = (this.x && this.x.length > 0 || 0) * 16;
        let normalized = this.normalized.x.length * 16;
        this.memory = {
          original,
          normalized,
          total: original + normalized
        };
      }
      removeOriginal() {
        this.x = undefined;
        this.y = undefined;
        this.updateMemory();
      }
      getXY() {
        if (!Array.isArray(this.x) || !Array.isArray(this.y)) {
          throw new Error('Can not get normalized data');
        }
        return {
          x: this.x,
          y: this.y
        };
      }
      updateRangesInfo(ranges) {
        updateRangesInfo(this, ranges);
      }
      updateNormalization(normalization) {
        const result = getNormalized(this, normalization);
        this.normalized = result.data;
        this.normalizedAllowedBoundary = result.allowedBoundary;
        this.ranges = {};
        this.updateMemory();
        this.updateNormalizedBoundary();
      }
      getData(options) {
        return getData(this, options);
      }
      updateNormalizedBoundary() {
        this.normalizedBoundary.x = {
          min: this.normalized.x[0],
          max: this.normalized.x.at(-1)
        };
        this.normalizedBoundary.y = xMinMaxValues(this.normalized.y);
      }
    }

    class SpectraProcessor {
      /**
       * Manager a large number of spectra with the possibility to normalize the data
       * and skip the original data.
       * @param {object} [options={}]
       * @param {boolean} [options.maxMemory=64M]
       * @param {object} [options.normalization={}] - options to normalize the spectra before comparison
       * @param {number} [options.normalization.from]
       * @param {number} [options.normalization.to]
       * @param {number} [options.normalization.numberOfPoints]
       * @param {Array<object>} [options.normalization.filters]
       * @param {string} [options.normalization.filters.X.name]
       * @param {object} [options.normalization.filters.X.options]
       * @param {Array<object>} [options.normalization.exclusions]
       * @param {string} [options.normalization.exclusions.X.from]
       * @param {object} [options.normalization.exclusions.X.to]
       */
      constructor(options = {}) {
        this.normalization = options.normalization;
        this.maxMemory = options.maxMemory || 256 * 1024 * 1024;
        this.keepOriginal = true;
        this.spectra = [];
      }
      getNormalizationAnnotations() {
        return getNormalizationAnnotations(this.normalization, this.getNormalizedBoundary());
      }

      /**
       * Recalculate the normalized data using the stored original data if available
       * This will throw an error in the original data is not present
       * @param {number} [normalization.from]
       * @param {number} [normalization.to]
       * @param {number} [normalization.numberOfPoints]
       * @param {Array<object>} [normalization.filters]
       * @param {string} [normalization.filters.X.name]
       * @param {object} [normalization.filters.X.options]
       * @param {Array<object>} [normalization.exclusions]
       * @param {string} [normalization.exclusions.X.from]
       * @param {object} [normalization.exclusions.X.to]
       * @param normalization
       */
      setNormalization(normalization = {}) {
        if (JSON.stringify(this.normalization) === JSON.stringify(normalization)) {
          return;
        }
        this.normalization = normalization;
        for (let spectrum of this.spectra) {
          spectrum.updateNormalization(this.normalization);
        }
      }
      getNormalization() {
        return this.normalization;
      }

      /**
       * Returns an object {x:[], y:[]} containing the autocorrelation for the
       * specified index
       * @param {integer} [index|undefined] - x index of the spectrum to autocorrelate
       * @param index
       * @param {object} [options={}]
       * @param {Array} [options.ids=[]] - list of ids, by default all spectra
       * @param {Array} [options.x] - x value if index is undefined
       */
      getAutocorrelation(index, options = {}) {
        const {
          x
        } = options;
        const normalizedData = this.getNormalizedData(options);
        if (index === undefined && x !== undefined) {
          index = xFindClosestIndex(normalizedData.x, x);
        }
        return getAutocorrelation(normalizedData, index);
      }

      /**
       * Returns a {x:[], y:[]} containg the average of specified spectra
       * @param {object} [options={}]
       * @param {Array} [options.ids=[]] - list of ids, by default all spectra
       */
      getMeanData(options) {
        return getMeanData(this.getNormalizedData(options));
      }

      /**
       * Returns an object contains 4 parameters with the normalized data
       * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[] }
       * @param {object} [options={}]
       * @param {Array} [options.ids] - List of spectra ids to export, by default all
       */
      getNormalizedData(options = {}) {
        const {
          ids
        } = options;
        let spectra = this.getSpectra(ids);
        return getNormalizedData(spectra);
      }

      /**
       * Returns a tab separated value containing the normalized data
       * @param {object} [options={}]
       * @param {Array} [options.ids] - List of spectra ids to export, by default all
       * @param {string} [options.fs='\t'] - field separator
       * @param {string} [options.rs='\n'] - record (line) separator
       * @returns {string}
       */
      getNormalizedText(options = {}) {
        const {
          ids
        } = options;
        let spectra = this.getSpectra(ids);
        return getNormalizedText(spectra, options);
      }

      /**
       * Returns a tab separated value containing the post processed data
       * @param {object} [options={}]
       * @param {Array} [options.ids] - List of spectra ids to export, by default all
       * @param {string} [options.fs='\t'] - field separator
       * @param {string} [options.rs='\n'] - record (line) separator
       * @param {object} [postProcessing={}]
       * @returns {string}
       */
      getPostProcessedText(options = {}) {
        return getPostProcessedText(this, options);
      }
      getMinMaxX() {
        let min = Number.MAX_VALUE;
        let max = Number.MIN_VALUE;
        for (let spectrum of this.spectra) {
          if (spectrum.minX < min) min = spectrum.minX;
          if (spectrum.maxX > max) max = spectrum.maxX;
        }
        return {
          min,
          max
        };
      }

      /**
       *
       * Returns an object contains 4 parameters with the scaled data
       * @param {object} [options={}] - scale spectra based on various parameters
       * @param {object} [options.range] - from - to
       * @param {Array} [options.ids] - ids of selected spectra, by default all
       * @param {string} [options.targetID=spectra[0].id]
       * @param {string} [options.method='max'] - min, max, range, minMax
       * @param {boolean} [options.relative=false]
       * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[] }
       */
      getPostProcessedData(options) {
        return getPostProcessedData(this, options);
      }

      /**
       * Add from text
       * By default TITLE from the jcamp will be in the meta information
       * @param {string} text
       * @param {object} [options={}]
       * @param {object} [options.parserOptions={}] - XY parser options
       * @param {object} [options.meta={}]
       * @param {string} [options.meta.color]
       * @param {object} [options.id={}]
       * @param {object} [options.kind]
       * @param {boolean} [options.force=false] - replace existing spectrum (same ID)
       */

      addFromText(text, options = {}) {
        if (options.force !== true && options.id && this.contains(options.id)) {
          return;
        }
        let parsed = parseText(text, options);
        let meta = {
          ...parsed.meta,
          ...options.meta
        };
        this.addFromData(parsed.data, {
          meta,
          id: options.id
        });
      }

      /**
       * Add jcamp
       * By default TITLE from the jcamp will be in the meta information
       * @param {string} jcamp
       * @param {object} [options={}]
       * @param {object} [options.meta={}]
       * @param {string} [options.meta.color]
       * @param {object} [options.id={}]
       * @param {boolean} [options.force=false] - replace existing spectrum (same ID)
       */

      addFromJcamp(jcamp, options = {}) {
        if (options.force !== true && options.id && this.contains(options.id)) {
          return;
        }
        let parsed = parseJcamp(jcamp);
        let meta = {
          ...parsed.info,
          ...parsed.meta,
          ...options.meta
        };
        this.addFromData(parsed.data, {
          meta,
          id: options.id
        });
      }
      updateRangesInfo(options) {
        for (let spectrum of this.spectra) {
          spectrum.updateRangesInfo(options);
        }
      }

      /**
       * Returns the metadata for a set of spectra
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @returns {object}
       */
      getMetadata(options = {}) {
        const {
          ids
        } = options;
        return getMetadata(this.getSpectra(ids));
      }

      /**
       *
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @param {string} [options.propertyName="category"]
       */
      getClasses(options) {
        return getClasses(this.getMetadata(options), options);
      }

      /**
       *
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @param {string} [options.propertyName="category"]
       */
      getClassLabels(options) {
        return getClassLabels(this.getMetadata(options), options);
      }

      /**
       *
       * @param {object} [options={}]
       * @param {string} [options.propertyName="category"]
       */
      getCategoriesStats(options) {
        return getCategoriesStats(this.getMetadata(), options);
      }

      /**
       * Add a spectrum based on the data
       * @param {object} data - {x, y}}
       * @param {object} [options={}]
       * @param {object} [options.meta={}]
       * @param {object} [options.id]
       * @param {object} [options.normalization={}]
       * @param {object} [options.normalized]
       * @returns {Spectrum}
       */

      addFromData(data, options = {}) {
        if (this.spectra.length === 0) this.keepOriginal = true;
        const id = options.id || Math.random().toString(36).slice(2, 10);
        let index = this.getSpectrumIndex(id);
        if (index === undefined) index = this.spectra.length;
        let spectrum = new Spectrum(data.x, data.y, id, {
          meta: options.meta,
          normalized: options.normalized,
          normalization: this.normalization
        });
        this.spectra[index] = spectrum;
        if (!this.keepOriginal) {
          spectrum.removeOriginal();
        } else {
          let memoryInfo = this.getMemoryInfo();
          if (memoryInfo.total > this.maxMemory) {
            this.keepOriginal = false;
            this.removeOriginals();
          }
        }
      }
      removeOriginals() {
        for (let spectrum of this.spectra) {
          spectrum.removeOriginal();
        }
      }

      /**
       * Remove the spectrum from the SpectraProcessor for the specified id
       * @param {string} id
       */
      removeSpectrum(id) {
        let index = this.getSpectrumIndex(id);
        if (index === undefined) return undefined;
        return this.spectra.splice(index, 1);
      }

      /**
       * Remove all the spectra not present in the list
       * @param {Array} [ids] - Array of ids of the spectra to keep
       */
      removeSpectraNotIn(ids) {
        let currentIDs = this.spectra.map(spectrum => spectrum.id);
        for (let id of currentIDs) {
          if (!ids.includes(id)) {
            this.removeSpectrum(id);
          }
        }
      }

      /**
       * Checks if the ID of a spectrum exists in the SpectraProcessor
       * @param {string} id
       */
      contains(id) {
        return this.getSpectrumIndex(id) !== undefined;
      }

      /**
       * Returns the index of the spectrum in the spectra array
       * @param {string} id
       * @returns {number}
       */
      getSpectrumIndex(id) {
        if (!id) return undefined;
        for (let i = 0; i < this.spectra.length; i++) {
          let spectrum = this.spectra[i];
          if (spectrum.id === id) return i;
        }
        return undefined;
      }

      /**
       * Returns an array of all the ids
       * @returns {Array}
       */
      getIDs() {
        return this.spectra.map(spectrum => spectrum.id);
      }

      /**
       * Returns an array of spectrum from their ids
       * @param {Array} ids
       * @returns {Array<Spectrum}
       */
      getSpectra(ids) {
        if (!ids || !Array.isArray(ids)) return this.spectra;
        let spectra = [];
        for (let id of ids) {
          let index = this.getSpectrumIndex(id);
          if (index !== undefined) {
            spectra.push(this.spectra[index]);
          }
        }
        return spectra;
      }

      /**
       * Returns the index of the spectrum in the spectra array
       * @param {string} id
       * @returns {number}
       */
      getSpectrum(id) {
        let index = this.getSpectrumIndex(id);
        if (index === undefined) return undefined;
        return this.spectra[index];
      }

      /**
       * Returns a JSGraph chart object for all the spectra
       * @param options
       * @returns {object}
       */
      getChart(options) {
        return getChart(this.spectra, options);
      }

      /**
       * Returns a JSGraph chart object for all the spectra
       * @param index
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @returns {object}
       */
      getAutocorrelationChart(index, options) {
        return getAutocorrelationChart(this, index, options);
      }

      /**
       * Returns a JSGraph annotation object for the specified index
       * @param index
       * @param {object} [options={}]
       * @param {Array} [options.ids=[]]
       * @param {Array} [options.showSpectrumID=true]
       * @param {Array} [options.sortY=true]
       * @param {object} [options.limit=20]- -
       * ids, showSpectrumID = true, sort = true, limit = 20
       * @returns {object}
       */
      getTrackAnnotation(index, options) {
        return getTrackAnnotation(this.spectra, index, options);
      }

      /**
       * Returns a JSGraph annotation object for box plot
       * @param {object} [options={}]
       * @param {Array} [options.ids=[]] - list of ids, by default all spectra
       * @returns {object}
       */
      getBoxPlotChart(options = {}) {
        const normalizedData = this.getNormalizedData(options);
        return getBoxPlotChart(normalizedData, options);
      }

      /**
       * Returns boxplot information
       * @param {object} [options={}]
       * @param {Array} [options.ids=[]] - list of ids, by default all spectra
       * @returns {object}
       */
      getBoxPlotData(options = {}) {
        const normalizedData = this.getNormalizedData(options);
        return getBoxPlotData(normalizedData);
      }

      /**
       * Returns a JSGraph chart object for all the normalized spectra
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @param {object} [options.xFilter={}]
       * @param {Array} [options.xFilter.from]
       * @param {Array} [options.xFilter.to]
       * @param {Array} [options.xFilter.exclusions=[]]
       * @returns {object}
       */
      getNormalizedChart(options = {}) {
        const {
          ids
        } = options;
        let spectra = this.getSpectra(ids);
        return getNormalizedChart(spectra, options);
      }

      /**
       * Returns a JSGraph chart object for all the scaled normalized spectra
       * @param {object} [options={}]
       * @param {Array} [options.ids] - ids of the spectra to select, by default all
       * @param {object} [options.range] - from - to
       * @param {string} [options.targetID=spectra[0].id]
       * @param {string} [options.method='max'] - min, max, range, minMax
       * @param {boolean} [options.relative=false]
       * @returns {object}
       */
      getPostProcessedChart(options) {
        return getPostProcessedChart(this, options);
      }
      getMemoryInfo() {
        let memoryInfo = {
          original: 0,
          normalized: 0,
          total: 0
        };
        for (const spectrum of this.spectra) {
          let memory = spectrum.memory;
          memoryInfo.original += memory.original;
          memoryInfo.normalized += memory.normalized;
          memoryInfo.total += memory.total;
        }
        memoryInfo.keepOriginal = this.keepOriginal;
        memoryInfo.maxMemory = this.maxMemory;
        return memoryInfo;
      }
      getNormalizedBoundary() {
        let boundary = {
          x: {
            min: Number.MAX_VALUE,
            max: Number.MIN_VALUE
          },
          y: {
            min: Number.MAX_VALUE,
            max: Number.MIN_VALUE
          }
        };
        for (let spectrum of this.spectra) {
          if (spectrum.normalizedBoundary.x.min < boundary.x.min) {
            boundary.x.min = spectrum.normalizedBoundary.x.min;
          }
          if (spectrum.normalizedBoundary.x.max > boundary.x.max) {
            boundary.x.max = spectrum.normalizedBoundary.x.max;
          }
          if (spectrum.normalizedBoundary.y.min < boundary.y.min) {
            boundary.y.min = spectrum.normalizedBoundary.y.min;
          }
          if (spectrum.normalizedBoundary.y.max > boundary.y.max) {
            boundary.y.max = spectrum.normalizedBoundary.y.max;
          }
        }
        return boundary;
      }

      /**
       * We provide the allowed from / to after normalization
       * For the X axis we return the smallest common values
       * For the Y axis we return the largest min / max
       */
      getNormalizedCommonBoundary() {
        let boundary = {
          x: {
            min: Number.NEGATIVE_INFINITY,
            max: Number.POSITIVE_INFINITY
          },
          y: {
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY
          }
        };
        for (let spectrum of this.spectra) {
          if (spectrum.normalizedAllowedBoundary.x.min > boundary.x.min) {
            boundary.x.min = spectrum.normalizedAllowedBoundary.x.min;
          }
          if (spectrum.normalizedAllowedBoundary.x.max < boundary.x.max) {
            boundary.x.max = spectrum.normalizedAllowedBoundary.x.max;
          }
          if (spectrum.normalizedAllowedBoundary.y.min < boundary.y.min) {
            boundary.y.min = spectrum.normalizedAllowedBoundary.y.min;
          }
          if (spectrum.normalizedAllowedBoundary.y.max > boundary.y.max) {
            boundary.y.max = spectrum.normalizedAllowedBoundary.y.max;
          }
        }
        return boundary;
      }

      /**
       * Create SpectraProcessor from normalized TSV
       * @param {string} text
       * @param {object} [options={}]
       * @param {object} [options.separator='\t']
       */
      static fromNormalizedMatrix(text, options = {}) {
        let parsed = parseMatrix(text, options);
        if (!parsed) {
          throw new Error('Can not parse TSV file');
        }
        let spectraProcessor = new SpectraProcessor();
        spectraProcessor.setNormalization({
          from: parsed.x[0],
          to: parsed.x.at(-1),
          numberOfPoints: parsed.x.length
        });
        for (let i = 0; i < parsed.ids.length; i++) {
          spectraProcessor.addFromData({
            x: [],
            y: []
          }, {
            normalized: {
              x: parsed.x,
              y: parsed.matrix[i]
            },
            id: parsed.ids[i],
            meta: parsed.meta[i]
          });
        }
        spectraProcessor.keepOriginal = false;
        return spectraProcessor;
      }
    }

    function getFilterAnnotations(filter = {}) {
      let {
        exclusions = []
      } = filter;
      let annotations = [];
      exclusions = exclusions.filter(exclusion => !exclusion.ignore);
      annotations = exclusions.map(exclusion => {
        let annotation = {
          type: 'rect',
          position: [{
            x: exclusion.from,
            y: '0px'
          }, {
            x: exclusion.to,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,128,1)'
        };
        return annotation;
      });
      if (filter.from !== undefined) {
        annotations.push({
          type: 'rect',
          position: [{
            x: Number.MIN_SAFE_INTEGER,
            y: '0px'
          }, {
            x: filter.from,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,224,1)'
        });
      }
      if (filter.to !== undefined) {
        annotations.push({
          type: 'rect',
          position: [{
            x: filter.to,
            y: '0px'
          }, {
            x: Number.MAX_SAFE_INTEGER,
            y: '2000px'
          }],
          strokeWidth: 0,
          fillColor: 'rgba(255,255,224,1)'
        });
      }
      return annotations;
    }

    exports.SpectraProcessor = SpectraProcessor;
    exports.getFilterAnnotations = getFilterAnnotations;
    exports.xFindClosestIndex = xFindClosestIndex;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=spectra-processor.js.map
