/**
 * netcdfjs - Read and explore NetCDF files
 * @version v2.0.2
 * @link https://github.com/cheminfo/netcdfjs
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.NetCDF = {}));
})(this, (function (exports) { 'use strict';

    // eslint-disable-next-line import/no-unassigned-import
    function decode(bytes) {
      let encoding = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'utf8';
      const decoder = new TextDecoder(encoding);
      return decoder.decode(bytes);
    }
    const encoder = new TextEncoder();
    function encode(str) {
      return encoder.encode(str);
    }

    const defaultByteLength = 1024 * 8;
    class IOBuffer {
      /**
       * @param data - The data to construct the IOBuffer with.
       * If data is a number, it will be the new buffer's length<br>
       * If data is `undefined`, the buffer will be initialized with a default length of 8Kb<br>
       * If data is an ArrayBuffer, SharedArrayBuffer, an ArrayBufferView (Typed Array), an IOBuffer instance,
       * or a Node.js Buffer, a view will be created over the underlying ArrayBuffer.
       * @param options
       */
      constructor() {
        let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : defaultByteLength;
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let dataIsGiven = false;
        if (typeof data === 'number') {
          data = new ArrayBuffer(data);
        } else {
          dataIsGiven = true;
          this.lastWrittenByte = data.byteLength;
        }
        const offset = options.offset ? options.offset >>> 0 : 0;
        const byteLength = data.byteLength - offset;
        let dvOffset = offset;
        if (ArrayBuffer.isView(data) || data instanceof IOBuffer) {
          if (data.byteLength !== data.buffer.byteLength) {
            dvOffset = data.byteOffset + offset;
          }
          data = data.buffer;
        }
        if (dataIsGiven) {
          this.lastWrittenByte = byteLength;
        } else {
          this.lastWrittenByte = 0;
        }
        this.buffer = data;
        this.length = byteLength;
        this.byteLength = byteLength;
        this.byteOffset = dvOffset;
        this.offset = 0;
        this.littleEndian = true;
        this._data = new DataView(this.buffer, dvOffset, byteLength);
        this._mark = 0;
        this._marks = [];
      }
      /**
       * Checks if the memory allocated to the buffer is sufficient to store more
       * bytes after the offset.
       * @param byteLength - The needed memory in bytes.
       * @returns `true` if there is sufficient space and `false` otherwise.
       */
      available() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return this.offset + byteLength <= this.length;
      }
      /**
       * Check if little-endian mode is used for reading and writing multi-byte
       * values.
       * @returns `true` if little-endian mode is used, `false` otherwise.
       */
      isLittleEndian() {
        return this.littleEndian;
      }
      /**
       * Set little-endian mode for reading and writing multi-byte values.
       */
      setLittleEndian() {
        this.littleEndian = true;
        return this;
      }
      /**
       * Check if big-endian mode is used for reading and writing multi-byte values.
       * @returns `true` if big-endian mode is used, `false` otherwise.
       */
      isBigEndian() {
        return !this.littleEndian;
      }
      /**
       * Switches to big-endian mode for reading and writing multi-byte values.
       */
      setBigEndian() {
        this.littleEndian = false;
        return this;
      }
      /**
       * Move the pointer n bytes forward.
       * @param n - Number of bytes to skip.
       */
      skip() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        this.offset += n;
        return this;
      }
      /**
       * Move the pointer n bytes backward.
       * @param n - Number of bytes to move back.
       */
      back() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        this.offset -= n;
        return this;
      }
      /**
       * Move the pointer to the given offset.
       * @param offset
       */
      seek(offset) {
        this.offset = offset;
        return this;
      }
      /**
       * Store the current pointer offset.
       * @see {@link IOBuffer#reset}
       */
      mark() {
        this._mark = this.offset;
        return this;
      }
      /**
       * Move the pointer back to the last pointer offset set by mark.
       * @see {@link IOBuffer#mark}
       */
      reset() {
        this.offset = this._mark;
        return this;
      }
      /**
       * Push the current pointer offset to the mark stack.
       * @see {@link IOBuffer#popMark}
       */
      pushMark() {
        this._marks.push(this.offset);
        return this;
      }
      /**
       * Pop the last pointer offset from the mark stack, and set the current
       * pointer offset to the popped value.
       * @see {@link IOBuffer#pushMark}
       */
      popMark() {
        const offset = this._marks.pop();
        if (offset === undefined) {
          throw new Error('Mark stack empty');
        }
        this.seek(offset);
        return this;
      }
      /**
       * Move the pointer offset back to 0.
       */
      rewind() {
        this.offset = 0;
        return this;
      }
      /**
       * Make sure the buffer has sufficient memory to write a given byteLength at
       * the current pointer offset.
       * If the buffer's memory is insufficient, this method will create a new
       * buffer (a copy) with a length that is twice (byteLength + current offset).
       * @param byteLength
       */
      ensureAvailable() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        if (!this.available(byteLength)) {
          const lengthNeeded = this.offset + byteLength;
          const newLength = lengthNeeded * 2;
          const newArray = new Uint8Array(newLength);
          newArray.set(new Uint8Array(this.buffer));
          this.buffer = newArray.buffer;
          this.length = this.byteLength = newLength;
          this._data = new DataView(this.buffer);
        }
        return this;
      }
      /**
       * Read a byte and return false if the byte's value is 0, or true otherwise.
       * Moves pointer forward by one byte.
       */
      readBoolean() {
        return this.readUint8() !== 0;
      }
      /**
       * Read a signed 8-bit integer and move pointer forward by 1 byte.
       */
      readInt8() {
        return this._data.getInt8(this.offset++);
      }
      /**
       * Read an unsigned 8-bit integer and move pointer forward by 1 byte.
       */
      readUint8() {
        return this._data.getUint8(this.offset++);
      }
      /**
       * Alias for {@link IOBuffer#readUint8}.
       */
      readByte() {
        return this.readUint8();
      }
      /**
       * Read `n` bytes and move pointer forward by `n` bytes.
       */
      readBytes() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        const bytes = new Uint8Array(n);
        for (let i = 0; i < n; i++) {
          bytes[i] = this.readByte();
        }
        return bytes;
      }
      /**
       * Read a 16-bit signed integer and move pointer forward by 2 bytes.
       */
      readInt16() {
        const value = this._data.getInt16(this.offset, this.littleEndian);
        this.offset += 2;
        return value;
      }
      /**
       * Read a 16-bit unsigned integer and move pointer forward by 2 bytes.
       */
      readUint16() {
        const value = this._data.getUint16(this.offset, this.littleEndian);
        this.offset += 2;
        return value;
      }
      /**
       * Read a 32-bit signed integer and move pointer forward by 4 bytes.
       */
      readInt32() {
        const value = this._data.getInt32(this.offset, this.littleEndian);
        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit unsigned integer and move pointer forward by 4 bytes.
       */
      readUint32() {
        const value = this._data.getUint32(this.offset, this.littleEndian);
        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit floating number and move pointer forward by 4 bytes.
       */
      readFloat32() {
        const value = this._data.getFloat32(this.offset, this.littleEndian);
        this.offset += 4;
        return value;
      }
      /**
       * Read a 64-bit floating number and move pointer forward by 8 bytes.
       */
      readFloat64() {
        const value = this._data.getFloat64(this.offset, this.littleEndian);
        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit signed integer number and move pointer forward by 8 bytes.
       */
      readBigInt64() {
        const value = this._data.getBigInt64(this.offset, this.littleEndian);
        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit unsigned integer number and move pointer forward by 8 bytes.
       */
      readBigUint64() {
        const value = this._data.getBigUint64(this.offset, this.littleEndian);
        this.offset += 8;
        return value;
      }
      /**
       * Read a 1-byte ASCII character and move pointer forward by 1 byte.
       */
      readChar() {
        return String.fromCharCode(this.readInt8());
      }
      /**
       * Read `n` 1-byte ASCII characters and move pointer forward by `n` bytes.
       */
      readChars() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        let result = '';
        for (let i = 0; i < n; i++) {
          result += this.readChar();
        }
        return result;
      }
      /**
       * Read the next `n` bytes, return a UTF-8 decoded string and move pointer
       * forward by `n` bytes.
       */
      readUtf8() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return decode(this.readBytes(n));
      }
      /**
       * Read the next `n` bytes, return a string decoded with `encoding` and move pointer
       * forward by `n` bytes.
       * If no encoding is passed, the function is equivalent to @see {@link IOBuffer#readUtf8}
       */
      decodeText() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        let encoding = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'utf-8';
        return decode(this.readBytes(n), encoding);
      }
      /**
       * Write 0xff if the passed value is truthy, 0x00 otherwise and move pointer
       * forward by 1 byte.
       */
      writeBoolean(value) {
        this.writeUint8(value ? 0xff : 0x00);
        return this;
      }
      /**
       * Write `value` as an 8-bit signed integer and move pointer forward by 1 byte.
       */
      writeInt8(value) {
        this.ensureAvailable(1);
        this._data.setInt8(this.offset++, value);
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as an 8-bit unsigned integer and move pointer forward by 1
       * byte.
       */
      writeUint8(value) {
        this.ensureAvailable(1);
        this._data.setUint8(this.offset++, value);
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * An alias for {@link IOBuffer#writeUint8}.
       */
      writeByte(value) {
        return this.writeUint8(value);
      }
      /**
       * Write all elements of `bytes` as uint8 values and move pointer forward by
       * `bytes.length` bytes.
       */
      writeBytes(bytes) {
        this.ensureAvailable(bytes.length);
        for (let i = 0; i < bytes.length; i++) {
          this._data.setUint8(this.offset++, bytes[i]);
        }
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 16-bit signed integer and move pointer forward by 2
       * bytes.
       */
      writeInt16(value) {
        this.ensureAvailable(2);
        this._data.setInt16(this.offset, value, this.littleEndian);
        this.offset += 2;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 16-bit unsigned integer and move pointer forward by 2
       * bytes.
       */
      writeUint16(value) {
        this.ensureAvailable(2);
        this._data.setUint16(this.offset, value, this.littleEndian);
        this.offset += 2;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 32-bit signed integer and move pointer forward by 4
       * bytes.
       */
      writeInt32(value) {
        this.ensureAvailable(4);
        this._data.setInt32(this.offset, value, this.littleEndian);
        this.offset += 4;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 32-bit unsigned integer and move pointer forward by 4
       * bytes.
       */
      writeUint32(value) {
        this.ensureAvailable(4);
        this._data.setUint32(this.offset, value, this.littleEndian);
        this.offset += 4;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 32-bit floating number and move pointer forward by 4
       * bytes.
       */
      writeFloat32(value) {
        this.ensureAvailable(4);
        this._data.setFloat32(this.offset, value, this.littleEndian);
        this.offset += 4;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 64-bit floating number and move pointer forward by 8
       * bytes.
       */
      writeFloat64(value) {
        this.ensureAvailable(8);
        this._data.setFloat64(this.offset, value, this.littleEndian);
        this.offset += 8;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 64-bit signed bigint and move pointer forward by 8
       * bytes.
       */
      writeBigInt64(value) {
        this.ensureAvailable(8);
        this._data.setBigInt64(this.offset, value, this.littleEndian);
        this.offset += 8;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write `value` as a 64-bit unsigned bigint and move pointer forward by 8
       * bytes.
       */
      writeBigUint64(value) {
        this.ensureAvailable(8);
        this._data.setBigUint64(this.offset, value, this.littleEndian);
        this.offset += 8;
        this._updateLastWrittenByte();
        return this;
      }
      /**
       * Write the charCode of `str`'s first character as an 8-bit unsigned integer
       * and move pointer forward by 1 byte.
       */
      writeChar(str) {
        return this.writeUint8(str.charCodeAt(0));
      }
      /**
       * Write the charCodes of all `str`'s characters as 8-bit unsigned integers
       * and move pointer forward by `str.length` bytes.
       */
      writeChars(str) {
        for (let i = 0; i < str.length; i++) {
          this.writeUint8(str.charCodeAt(i));
        }
        return this;
      }
      /**
       * UTF-8 encode and write `str` to the current pointer offset and move pointer
       * forward according to the encoded length.
       */
      writeUtf8(str) {
        return this.writeBytes(encode(str));
      }
      /**
       * Export a Uint8Array view of the internal buffer.
       * The view starts at the byte offset and its length
       * is calculated to stop at the last written byte or the original length.
       */
      toArray() {
        return new Uint8Array(this.buffer, this.byteOffset, this.lastWrittenByte);
      }
      /**
       * Update the last written byte offset
       * @private
       */
      _updateLastWrittenByte() {
        if (this.offset > this.lastWrittenByte) {
          this.lastWrittenByte = this.offset;
        }
      }
    }

    /**
     * Throws a non-valid NetCDF exception if the statement it's true
     * @ignore
     * @param {boolean} statement - Throws if true
     * @param {string} reason - Reason to throw
     */
    function notNetcdf(statement, reason) {
      if (statement) {
        throw new TypeError(`Not a valid NetCDF v3.x file: ${reason}`);
      }
    }

    /**
     * Moves 1, 2, or 3 bytes to next 4-byte boundary
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     */
    function padding(buffer) {
      if (buffer.offset % 4 !== 0) {
        buffer.skip(4 - buffer.offset % 4);
      }
    }

    /**
     * Reads the name
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @return {string} - Name
     */
    function readName(buffer) {
      // Read name
      let nameLength = buffer.readUint32();
      let name = buffer.readChars(nameLength);

      // validate name
      // TODO

      // Apply padding
      padding(buffer);
      return name;
    }

    const types = {
      BYTE: 1,
      CHAR: 2,
      SHORT: 3,
      INT: 4,
      FLOAT: 5,
      DOUBLE: 6
    };

    /**
     * Parse a number into their respective type
     * @ignore
     * @param {number} type - integer that represents the type
     * @return {string} - parsed value of the type
     */
    function num2str(type) {
      switch (Number(type)) {
        case types.BYTE:
          return "byte";
        case types.CHAR:
          return "char";
        case types.SHORT:
          return "short";
        case types.INT:
          return "int";
        case types.FLOAT:
          return "float";
        case types.DOUBLE:
          return "double";
        /* istanbul ignore next */
        default:
          return "undefined";
      }
    }

    /**
     * Parse a number type identifier to his size in bytes
     * @ignore
     * @param {number} type - integer that represents the type
     * @return {number} -size of the type
     */
    function num2bytes(type) {
      switch (Number(type)) {
        case types.BYTE:
          return 1;
        case types.CHAR:
          return 1;
        case types.SHORT:
          return 2;
        case types.INT:
          return 4;
        case types.FLOAT:
          return 4;
        case types.DOUBLE:
          return 8;
        /* istanbul ignore next */
        default:
          return -1;
      }
    }

    /**
     * Reverse search of num2str
     * @ignore
     * @param {string} type - string that represents the type
     * @return {number} - parsed value of the type
     */
    function str2num(type) {
      switch (String(type)) {
        case "byte":
          return types.BYTE;
        case "char":
          return types.CHAR;
        case "short":
          return types.SHORT;
        case "int":
          return types.INT;
        case "float":
          return types.FLOAT;
        case "double":
          return types.DOUBLE;
        /* istanbul ignore next */
        default:
          return -1;
      }
    }

    /**
     * Auxiliary function to read numeric data
     * @ignore
     * @param {number} size - Size of the element to read
     * @param {function} bufferReader - Function to read next value
     * @return {Array<number>|number}
     */
    function readNumber(size, bufferReader) {
      if (size !== 1) {
        let numbers = new Array(size);
        for (let i = 0; i < size; i++) {
          numbers[i] = bufferReader();
        }
        return numbers;
      } else {
        return bufferReader();
      }
    }

    /**
     * Given a type and a size reads the next element
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @param {number} type - Type of the data to read
     * @param {number} size - Size of the element to read
     * @return {string|Array<number>|number}
     */
    function readType(buffer, type, size) {
      switch (type) {
        case types.BYTE:
          return buffer.readBytes(size);
        case types.CHAR:
          return trimNull(buffer.readChars(size));
        case types.SHORT:
          return readNumber(size, buffer.readInt16.bind(buffer));
        case types.INT:
          return readNumber(size, buffer.readInt32.bind(buffer));
        case types.FLOAT:
          return readNumber(size, buffer.readFloat32.bind(buffer));
        case types.DOUBLE:
          return readNumber(size, buffer.readFloat64.bind(buffer));
        /* istanbul ignore next */
        default:
          notNetcdf(true, `non valid type ${type}`);
          return undefined;
      }
    }

    /**
     * Removes null terminate value
     * @ignore
     * @param {string} value - String to trim
     * @return {string} - Trimmed string
     */
    function trimNull(value) {
      if (value.charCodeAt(value.length - 1) === 0) {
        return value.substring(0, value.length - 1);
      }
      return value;
    }

    // const STREAMING = 4294967295;

    /**
     * Read data for the given non-record variable
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @param {object} variable - Variable metadata
     * @return {Array} - Data of the element
     */
    function nonRecord(buffer, variable) {
      // variable type
      const type = str2num(variable.type);

      // size of the data
      let size = variable.size / num2bytes(type);

      // iterates over the data
      let data = new Array(size);
      for (let i = 0; i < size; i++) {
        data[i] = readType(buffer, type, 1);
      }
      return data;
    }

    /**
     * Read data for the given record variable
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @param {object} variable - Variable metadata
     * @param {object} recordDimension - Record dimension metadata
     * @return {Array} - Data of the element
     */
    function record(buffer, variable, recordDimension) {
      // variable type
      const type = str2num(variable.type);
      const width = variable.size ? variable.size / num2bytes(type) : 1;

      // size of the data
      // TODO streaming data
      let size = recordDimension.length;

      // iterates over the data
      let data = new Array(size);
      const step = recordDimension.recordStep;
      for (let i = 0; i < size; i++) {
        let currentOffset = buffer.offset;
        data[i] = readType(buffer, type, width);
        buffer.seek(currentOffset + step);
      }
      return data;
    }

    // Grammar constants
    const ZERO = 0;
    const NC_DIMENSION = 10;
    const NC_VARIABLE = 11;
    const NC_ATTRIBUTE = 12;

    /**
     * Read the header of the file
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @param {number} version - Version of the file
     * @return {object} - Object with the fields:
     *  * `recordDimension`: Number with the length of record dimension
     *  * `dimensions`: List of dimensions
     *  * `globalAttributes`: List of global attributes
     *  * `variables`: List of variables
     */
    function header(buffer, version) {
      // Length of record dimension
      // sum of the varSize's of all the record variables.
      let header = {
        recordDimension: {
          length: buffer.readUint32()
        }
      };

      // Version
      header.version = version;

      // List of dimensions
      let dimList = dimensionsList(buffer);
      header.recordDimension.id = dimList.recordId; // id of the unlimited dimension
      header.recordDimension.name = dimList.recordName; // name of the unlimited dimension
      header.dimensions = dimList.dimensions;

      // List of global attributes
      header.globalAttributes = attributesList(buffer);

      // List of variables
      let variables = variablesList(buffer, dimList.recordId, version);
      header.variables = variables.variables;
      header.recordDimension.recordStep = variables.recordStep;
      return header;
    }
    const NC_UNLIMITED = 0;

    /**
     * List of dimensions
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @return {object} - Ojbect containing the following properties:
     *  * `dimensions` that is an array of dimension object:
     *  * `name`: String with the name of the dimension
     *  * `size`: Number with the size of the dimension dimensions: dimensions
     *  * `recordId`: the id of the dimension that has unlimited size or undefined,
     *  * `recordName`: name of the dimension that has unlimited size
     */
    function dimensionsList(buffer) {
      let recordId, recordName;
      const dimList = buffer.readUint32();
      let dimensions;
      if (dimList === ZERO) {
        notNetcdf(buffer.readUint32() !== ZERO, "wrong empty tag for list of dimensions");
        return [];
      } else {
        notNetcdf(dimList !== NC_DIMENSION, "wrong tag for list of dimensions");

        // Length of dimensions
        const dimensionSize = buffer.readUint32();
        dimensions = new Array(dimensionSize);
        for (let dim = 0; dim < dimensionSize; dim++) {
          // Read name
          let name = readName(buffer);

          // Read dimension size
          const size = buffer.readUint32();
          if (size === NC_UNLIMITED) {
            // in netcdf 3 one field can be of size unlimmited
            recordId = dim;
            recordName = name;
          }
          dimensions[dim] = {
            name,
            size
          };
        }
      }
      return {
        dimensions,
        recordId,
        recordName
      };
    }

    /**
     * List of attributes
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @return {Array<object>} - List of attributes with:
     *  * `name`: String with the name of the attribute
     *  * `type`: String with the type of the attribute
     *  * `value`: A number or string with the value of the attribute
     */
    function attributesList(buffer) {
      const gAttList = buffer.readUint32();
      let attributes;
      if (gAttList === ZERO) {
        notNetcdf(buffer.readUint32() !== ZERO, "wrong empty tag for list of attributes");
        return [];
      } else {
        notNetcdf(gAttList !== NC_ATTRIBUTE, "wrong tag for list of attributes");

        // Length of attributes
        const attributeSize = buffer.readUint32();
        attributes = new Array(attributeSize);
        for (let gAtt = 0; gAtt < attributeSize; gAtt++) {
          // Read name
          let name = readName(buffer);

          // Read type
          let type = buffer.readUint32();
          notNetcdf(type < 1 || type > 6, `non valid type ${type}`);

          // Read attribute
          let size = buffer.readUint32();
          let value = readType(buffer, type, size);

          // Apply padding
          padding(buffer);
          attributes[gAtt] = {
            name,
            type: num2str(type),
            value
          };
        }
      }
      return attributes;
    }

    /**
     * List of variables
     * @ignore
     * @param {IOBuffer} buffer - Buffer for the file data
     * @param {number} recordId - Id of the unlimited dimension (also called record dimension)
     *                            This value may be undefined if there is no unlimited dimension
     * @param {number} version - Version of the file
     * @return {object} - Number of recordStep and list of variables with:
     *  * `name`: String with the name of the variable
     *  * `dimensions`: Array with the dimension IDs of the variable
     *  * `attributes`: Array with the attributes of the variable
     *  * `type`: String with the type of the variable
     *  * `size`: Number with the size of the variable
     *  * `offset`: Number with the offset where of the variable begins
     *  * `record`: True if is a record variable, false otherwise (unlimited size)
     */

    function variablesList(buffer, recordId, version) {
      const varList = buffer.readUint32();
      let recordStep = 0;
      let variables;
      if (varList === ZERO) {
        notNetcdf(buffer.readUint32() !== ZERO, "wrong empty tag for list of variables");
        return [];
      } else {
        notNetcdf(varList !== NC_VARIABLE, "wrong tag for list of variables");

        // Length of variables
        const variableSize = buffer.readUint32();
        variables = new Array(variableSize);
        for (let v = 0; v < variableSize; v++) {
          // Read name
          let name = readName(buffer);

          // Read dimensionality of the variable
          const dimensionality = buffer.readUint32();

          // Index into the list of dimensions
          let dimensionsIds = new Array(dimensionality);
          for (let dim = 0; dim < dimensionality; dim++) {
            dimensionsIds[dim] = buffer.readUint32();
          }

          // Read variables size
          let attributes = attributesList(buffer);

          // Read type
          let type = buffer.readUint32();
          notNetcdf(type < 1 && type > 6, `non valid type ${type}`);

          // Read variable size
          // The 32-bit varSize field is not large enough to contain the size of variables that require
          // more than 2^32 - 4 bytes, so 2^32 - 1 is used in the varSize field for such variables.
          const varSize = buffer.readUint32();

          // Read offset
          let offset = buffer.readUint32();
          if (version === 2) {
            notNetcdf(offset > 0, "offsets larger than 4GB not supported");
            offset = buffer.readUint32();
          }
          let record = false;
          // Count amount of record variables
          if (typeof recordId !== "undefined" && dimensionsIds[0] === recordId) {
            recordStep += varSize;
            record = true;
          }
          variables[v] = {
            name,
            dimensions: dimensionsIds,
            attributes,
            type: num2str(type),
            size: varSize,
            offset,
            record
          };
        }
      }
      return {
        variables,
        recordStep
      };
    }

    function toString() {
      let result = [];
      result.push("DIMENSIONS");
      for (let dimension of this.dimensions) {
        result.push(`  ${dimension.name.padEnd(30)} = size: ${dimension.size}`);
      }
      result.push("");
      result.push("GLOBAL ATTRIBUTES");
      for (let attribute of this.globalAttributes) {
        result.push(`  ${attribute.name.padEnd(30)} = ${attribute.value}`);
      }
      let variables = JSON.parse(JSON.stringify(this.variables));
      result.push("");
      result.push("VARIABLES:");
      for (let variable of variables) {
        variable.value = this.getDataVariable(variable);
        let stringify = JSON.stringify(variable.value);
        if (stringify.length > 50) stringify = stringify.substring(0, 50);
        if (!isNaN(variable.value.length)) {
          stringify += ` (length: ${variable.value.length})`;
        }
        result.push(`  ${variable.name.padEnd(30)} = ${stringify}`);
      }
      return result.join("\n");
    }

    /**
     * Reads a NetCDF v3.x file
     * https://www.unidata.ucar.edu/software/netcdf/docs/file_format_specifications.html
     * @param {ArrayBuffer} data - ArrayBuffer or any Typed Array (including Node.js' Buffer from v4) with the data
     * @constructor
     */
    class NetCDFReader {
      constructor(data) {
        const buffer = new IOBuffer(data);
        buffer.setBigEndian();

        // Validate that it's a NetCDF file
        notNetcdf(buffer.readChars(3) !== "CDF", "should start with CDF");

        // Check the NetCDF format
        const version = buffer.readByte();
        notNetcdf(version > 2, "unknown version");

        // Read the header
        this.header = header(buffer, version);
        this.buffer = buffer;
      }

      /**
       * @return {string} - Version for the NetCDF format
       */
      get version() {
        if (this.header.version === 1) {
          return "classic format";
        } else {
          return "64-bit offset format";
        }
      }

      /**
       * @return {object} - Metadata for the record dimension
       *  * `length`: Number of elements in the record dimension
       *  * `id`: Id number in the list of dimensions for the record dimension
       *  * `name`: String with the name of the record dimension
       *  * `recordStep`: Number with the record variables step size
       */
      get recordDimension() {
        return this.header.recordDimension;
      }

      /**
       * @return {Array<object>} - List of dimensions with:
       *  * `name`: String with the name of the dimension
       *  * `size`: Number with the size of the dimension
       */
      get dimensions() {
        return this.header.dimensions;
      }

      /**
       * @return {Array<object>} - List of global attributes with:
       *  * `name`: String with the name of the attribute
       *  * `type`: String with the type of the attribute
       *  * `value`: A number or string with the value of the attribute
       */
      get globalAttributes() {
        return this.header.globalAttributes;
      }

      /**
       * Returns the value of an attribute
       * @param {string} attributeName
       * @return {string} Value of the attributeName or null
       */
      getAttribute(attributeName) {
        const attribute = this.globalAttributes.find(val => val.name === attributeName);
        if (attribute) return attribute.value;
        return null;
      }

      /**
       * Returns the value of a variable as a string
       * @param {string} variableName
       * @return {string} Value of the variable as a string or null
       */
      getDataVariableAsString(variableName) {
        const variable = this.getDataVariable(variableName);
        if (variable) return variable.join("");
        return null;
      }

      /**
       * @return {Array<object>} - List of variables with:
       *  * `name`: String with the name of the variable
       *  * `dimensions`: Array with the dimension IDs of the variable
       *  * `attributes`: Array with the attributes of the variable
       *  * `type`: String with the type of the variable
       *  * `size`: Number with the size of the variable
       *  * `offset`: Number with the offset where of the variable begins
       *  * `record`: True if is a record variable, false otherwise
       */
      get variables() {
        return this.header.variables;
      }
      toString() {
        return toString.call(this);
      }

      /**
       * Retrieves the data for a given variable
       * @param {string|object} variableName - Name of the variable to search or variable object
       * @return {Array} - List with the variable values
       */
      getDataVariable(variableName) {
        let variable;
        if (typeof variableName === "string") {
          // search the variable
          variable = this.header.variables.find(val => {
            return val.name === variableName;
          });
        } else {
          variable = variableName;
        }

        // throws if variable not found
        notNetcdf(variable === undefined, `variable not found: ${variableName}`);

        // go to the offset position
        this.buffer.seek(variable.offset);
        if (variable.record) {
          // record variable case
          return record(this.buffer, variable, this.header.recordDimension);
        } else {
          // non-record variable case
          return nonRecord(this.buffer, variable);
        }
      }

      /**
       * Check if a dataVariable exists
       * @param {string} variableName - Name of the variable to find
       * @return {boolean}
       */
      dataVariableExists(variableName) {
        const variable = this.header.variables.find(val => {
          return val.name === variableName;
        });
        return variable !== undefined;
      }

      /**
       * Check if an attribute exists
       * @param {string} attributeName - Name of the attribute to find
       * @return {boolean}
       */
      attributeExists(attributeName) {
        const attribute = this.globalAttributes.find(val => val.name === attributeName);
        return attribute !== undefined;
      }
    }

    exports.NetCDFReader = NetCDFReader;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=netcdfjs.js.map
